//
// RSAKey.h
//
// $Id: //poco/1.4/Crypto/include/Poco/Crypto/RSAKey.h#2 $
//
// Library: Crypto
// Package: RSA
// Module:  RSAKey
//
// Definition of the RSAKey class.
//
// Copyright (c) 2008, Applied Informatics Software Engineering GmbH.
// and Contributors.
//
// Permission is hereby granted, free of charge, to any person or organization
// obtaining a copy of the software and accompanying documentation covered by
// this license (the "Software") to use, reproduce, display, distribute,
// execute, and transmit the Software, and to prepare derivative works of the
// Software, and to permit third-parties to whom the Software is furnished to
// do so, all subject to the following:
// 
// The copyright notices in the Software and this entire statement, including
// the above license grant, this restriction and the following disclaimer,
// must be included in all copies of the Software, in whole or in part, and
// all derivative works of the Software, unless such copies or derivative
// works are solely in the form of machine-executable object code generated by
// a source language processor.
// 
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO EVENT
// SHALL THE COPYRIGHT HOLDERS OR ANYONE DISTRIBUTING THE SOFTWARE BE LIABLE
// FOR ANY DAMAGES OR OTHER LIABILITY, WHETHER IN CONTRACT, TORT OR OTHERWISE,
// ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
// DEALINGS IN THE SOFTWARE.
//


#ifndef Crypto_RSAKey_INCLUDED
#define Crypto_RSAKey_INCLUDED


#include "Poco/Crypto/Crypto.h"
#include "Poco/Crypto/RSAKeyImpl.h"


namespace Poco {
namespace Crypto {


class X509Certificate;


class Crypto_API RSAKey
	/// This class stores an RSA key pair, consisting
	/// of private and public key. Storage of the private
	/// key is optional.
	///
	/// If a private key is available, the RSAKey can be
	/// used for decrypting data (encrypted with the public key)
	/// or computing secure digital signatures.
{
public:
	enum KeyLength
	{
		KL_512  = 512,
		KL_1024 = 1024,
		KL_2048 = 2048,
		KL_4096 = 4096
	};

	enum Exponent
	{
		EXP_SMALL = 0,
		EXP_LARGE
	};

	explicit RSAKey(const X509Certificate& cert);
		/// Extracts the RSA public key from the given certificate.

	RSAKey(KeyLength keyLength, Exponent exp);
		/// Creates the RSAKey. Creates a new public/private keypair using the given parameters.
		/// Can be used to sign data and verify signatures.

	RSAKey(const std::string& publicKeyFile, const std::string& privateKeyFile = "", const std::string& privateKeyPassphrase = "");
		/// Creates the RSAKey, by reading public and private key from the given files and
		/// using the given passphrase for the private key. Can only by used for signing if 
		/// a private key is available. 

	RSAKey(std::istream* pPublicKeyStream, std::istream* pPrivateKeyStream = 0, const std::string& privateKeyPassphrase = "");
		/// Creates the RSAKey. Can only by used for signing if pPrivKey
		/// is not null. If a private key file is specified, you don't need to
		/// specify a public key file. OpenSSL will auto-create it from the private key.

	~RSAKey();
		/// Destroys the RSAKey.

	int size() const;
		/// Returns the RSA modulus size.

	RSAKeyImpl::ByteVec modulus() const;
		/// Returns the RSA modulus.

	RSAKeyImpl::ByteVec encryptionExponent() const;
		/// Returns the RSA encryption exponent.

	RSAKeyImpl::ByteVec decryptionExponent() const;
		/// Returns the RSA decryption exponent.

	void save(const std::string& publicKeyFile, const std::string& privateKeyFile = "", const std::string& privateKeyPassphrase = "");
		/// Exports the public and private keys to the given files. 
		///
		/// If an empty filename is specified, the corresponding key
		/// is not exported.

	void save(std::ostream* pPublicKeyStream, std::ostream* pPrivateKeyStream = 0, const std::string& privateKeyPassphrase = "");
		/// Exports the public and private key to the given streams.
		///
		/// If a null pointer is passed for a stream, the corresponding
		/// key is not exported.

	RSAKeyImpl::Ptr impl() const;
		/// Returns the impl object.

	const std::string& name() const;
		/// Returns "rsa"
	
private:
	RSAKeyImpl::Ptr _pImpl;
};


//
// inlines
//
inline RSAKeyImpl::Ptr RSAKey::impl() const
{
	return _pImpl;
}


} } // namespace Poco::Crypto


#endif // Crypto_RSAKey_INCLUDED
