// Copyright (c) 2022-2023 Advanced Micro Devices, Inc. All rights reserved.
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
// THE SOFTWARE.

#include <cstring>
#include <fstream>
#include <iomanip>
#include <iostream>
#include <sstream>

/// Number of bits in the state space of MT19937.
constexpr unsigned int mexp = 19937;
/// Number of bits in an unsigned int.
constexpr unsigned int w_size = 32;
/// Minimum number of unsigned ints to represent mexp bits.
constexpr unsigned int p_size = (mexp + w_size - 1) / w_size;
/// Number of radixes in the jump table.
constexpr unsigned int jumps_radixes = 2;
/// Size of each radix (number of polynomials) in the jump table.
constexpr unsigned int jumps_radix = 256;
/// Number of polynomials required to create jumps_radix ^ jumps_radixes independent generators
/// of 2 ^ 1000 values
constexpr unsigned int jumps_count = (jumps_radix - 1) * jumps_radixes;

/// Set the coefficients of polynomial \p pf based on <tt>p</tt>.
void set_coef(const char p[mexp + 1], unsigned int pf[p_size])
{
    for(size_t i = 0; i < mexp; i++)
    {
        size_t j = mexp - i - 1;
        if(p[i] == '1')
        {
            // Set coefficient j of the polynomial pf.
            constexpr unsigned int log_w_size  = 5U;
            constexpr unsigned int w_size_mask = (1U << log_w_size) - 1;
            pf[j >> log_w_size] ^= 1U << (j & w_size_mask);
        }
    }
}

int main(int argc, char const* argv[])
{
    if(argc != 3 || std::string(argv[1]) == "--help")
    {
        std::cout << "Usage:" << std::endl;
        std::cout << "  ./mt19937_precomputed_generator "
                     "clist_mt19937.txt "
                     "../../library/src/rocrand_mt19937_precomputed.cpp"
                  << std::endl;
        return argc != 3 ? -1 : 0;
    }

    const std::string input_file_path(argv[1]);
    std::ifstream     fin(input_file_path);
    // This file contains (jumps_radix - 1) polynomials to skip <tt>i * 2 ^ 1000</tt> states and
    // (jumps_radix - 1) polynomials to skip <tt>jumps_radix * i * 2 ^ 1000</tt> states.
    // Values are produced using minipoly_mt19937.c as distributed in:
    // http://www.math.sci.hiroshima-u.ac.jp/~m-mat/MT/JUMP/jump_ahead_1.02.tar.gz
    // Size of the table is <tt>jumps_count * mexp</tt>.
    // Use clist_mt19937.txt from mt19937_precomputed_source.zip
    std::stringstream stream;
    stream << fin.rdbuf();
    const std::string jump = stream.str();

    const std::string output_file_path(argv[2]);
    std::ofstream     fout(output_file_path, std::ios_base::out | std::ios_base::trunc);
    fout << R"(// Copyright (c) 2022-2023 Advanced Micro Devices, Inc. All rights reserved.
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
// THE SOFTWARE.

// Auto-generated file. Do not edit!
// Generated by tools/mt19937_precomputed_generator

#include "rocrand/rocrand_mt19937_precomputed.h"
#include "rocrand/rocrandapi.h"
)";

    fout << "\n"
            "// clang-format off\n"
            "extern ROCRANDAPI const unsigned int rocrand_h_mt19937_jump[mt19937_jumps_count * "
            "mt19937_p_size] = {\n";

    for(unsigned int j = 0; j < jumps_count; j++)
    {
        unsigned int pf[p_size] = {};

        set_coef(jump.c_str() + j * mexp, pf);

        for(size_t i = 0; i < p_size; i++)
        {
            if(i % 8 == 0)
            {
                fout << "    ";
            }
            fout << std::setw(10);
            fout << pf[i] << "U";
            if(!(i == p_size - 1 && j == jumps_count - 1))
            {
                fout << ",";
                fout << ((i + 1) % 8 == 0 ? "\n" : " ");
            }
        }
    }

    fout << "\n};\n"
            "// clang-format on\n";
}
