# SPDX-License-Identifier: Apache-2.0
# SPDX-FileCopyrightText: Zygmunt Krynicki
QEMU_IMG ?= $(or $(shell command -v qemu-img),$(error program qemu-img is required))

# Try to give the native arch some more cores as those are usually cheaper than tcg.
QEMU_SMP_$(shell uname -m | tr a-z- A-Z_) = 4

QEMU_IMG_SIZE = 64G
QEMU_MEM_SIZE = 2048
QEMU_ENV_QUIRKS ?=

QEMU_DEFAULT_MEM_OPTION ?= -m $(QEMU_MEM_SIZE)
QEMU_DEFAULT_DISPLAY_OPTION ?= -nographic
ifneq ($(wildcard $(value SNAP_COMMON)/apt-cacher-ng.sock),)
QEMU_DEFAULT_NETDEV_SYS_EXTRA =,guestfwd=tcp:10.0.2.100:3142-cmd:image-garden-proxy
else
QEMU_DEFAULT_NETDEV_SYS_EXTRA ?=
endif
QEMU_DEFAULT_RNG_OPTION ?= -device virtio-rng-pci

define BASE_CLOUD_INIT_META_DATA_TEMPLATE
#cloud-config
instance-iid: iid-$1
local-hostname: $(subst .,-,$1)
endef

CLOUD_INIT_META_DATA_TEMPLATE=$(BASE_CLOUD_INIT_META_DATA_TEMPLATE)

ifneq ($(value SNAP),)
define INJECTED_CLOUD_INIT_FILES
- path: /etc/hosts
  permissions: '0644'
  append: true
  content: |
    # image-garden: there MAY be an apt-cacher-ng proxy on port 3142 there.
    10.0.2.100 apt-proxy
- path: /etc/apt/apt.conf.d/auto-apt-proxy.conf
  encoding: base64
  permissions: '0644'
  content: |
    $(shell base64 --wrap=0 $(SNAP)/usr/share/image-garden/auto-apt-proxy/etc/apt/apt.conf.d/auto-apt-proxy.conf)
- path: /usr/bin/auto-apt-proxy
  encoding: base64
  permissions: '0755'
  content: |
    $(shell base64 --wrap=0 $(SNAP)/usr/share/image-garden/auto-apt-proxy/usr/bin/auto-apt-proxy)
endef
else
INJECTED_CLOUD_INIT_FILES=
endif

define BASE_CLOUD_INIT_USER_DATA_TEMPLATE
#cloud-config
timezone: Europe/Warsaw
power_state:
  delay: now
  mode: poweroff
  message: Powering off
  timeout: 0
  condition: true
hostname: $(subst .,-,$1)
ssh_pwauth: true
users:
- name: root
  hashed_passwd: '$(shell echo -n root | $(MKPASSWD) --method=SHA-512 --rounds=4096 -s)'
  lock_passwd: false
  shell: /bin/bash
- name: $2
  groups: users
  gecos: $2 User
  # Both the username and the password is "$2"
  hashed_passwd: '$(shell echo -n $2 | $(MKPASSWD) --method=SHA-512 --rounds=4096 -s)'
  sudo: ["ALL=(ALL) NOPASSWD:ALL"]
  lock_passwd: false
  shell: /bin/bash
write_files:
# Use /root/foo.APPEND in case /etc is a read-only fs.
- path: /root/sshd_config.APPEND
  permissions: '0600'
  content: |
    # image-garden: allow interactive login with password authentication.
    UsePam yes
    # image-garden: allow root to log in over ssh.
    PermitRootLogin yes
- path: /etc/cloud/cloud.cfg.d/99-image-garden.cfg
  permissions: '0600'
  content: |
    # image-garden: allow interactive login with password authentication.
    # Use only none and no-cloud data sources to avoid probing cloud-specific
    # network services.
    ssh_pwauth: true
    # Instruct cloud-init not to look for cloud APIs.
    datasource_list:
    - NoCloud
- path: /root/hosts.APPEND
  permissions: '0644'
  content: |
    # image-garden: ensure hostname resolves to localhost
    127.0.0.1 $(subst .,-,$1)
$(INJECTED_CLOUD_INIT_FILES)
runcmd:
- cat /root/hosts.APPEND >> /etc/hosts
- rm -f /root/hosts.APPEND
- cat /root/sshd_config.APPEND >> /etc/ssh/sshd_config
- rm /root/sshd_config.APPEND
- sed -i -e 's/^PermitRootLogin prohibit-password/#PermitRootLogin prohibit-password/' /etc/ssh/sshd_config || true
- if [ -d /etc/apt ]; then find /etc/apt -type f -name '*.list' -exec sed -i -e 's/https/http/g' {} \; ; fi
- touch /etc/cloud/cloud-init.disabled
endef

CLOUD_INIT_USER_DATA_TEMPLATE=$(BASE_CLOUD_INIT_USER_DATA_TEMPLATE)

# Compatibility symbolic links for users that use conventions from image-garden
# 0.3 to image-garden 0.4. When CI/CD logic hard-codes creation of those files
# upgrades would cause a regression.
%.seed.iso: %.$(GARDEN_ARCH).seed.iso
	ln -sf $< $@

%.user-data: %.$(GARDEN_ARCH).user-data
	ln -sf $< $@

%.meta-data: %.$(GARDEN_ARCH).meta-data
	ln -sf $< $@

# Seed ISOs with cloud-init files.
# The switch -graft-points allows us to have different (fixed) name in the ISO
# and different (variable) names in the local file system. See the manual page
# of xorriso for details. The volume label is "cidata", lowercase, which works
# across both 2014-era and 2025-era systems.
%.seed.iso: %.user-data %.meta-data
	$(XORRISO) \
		-as genisoimage \
		-input-charset utf-8 \
		-output $@ \
		-volid cidata \
		-joliet \
		-rock \
		-graft-points \
		user-data=$(word 1,$^) \
		meta-data=$(word 2,$^)

# Pick one of the three templates, in order of priority:
# - $(DISTRO)_$(VERSION)_CLOUD_INIT_USER_DATA_TEMPLATE
# - $(DISTRO)_CLOUD_INIT_USER_DATA_TEMPLATE
# - CLOUD_INIT_USER_DATA_TEMPLATE
# This evaluates the name of the template. Use $(call $(call ...)...) to
# evaluate the resulting template.
PICK_CLOUD_INIT_USER_DATA_TEMPLATE_FOR = $(or $(if $(value $1_$2_CLOUD_INIT_USER_DATA_TEMPLATE),$1_$2_CLOUD_INIT_USER_DATA_TEMPLATE),$(if $(value $1_CLOUD_INIT_USER_DATA_TEMPLATE),$1_CLOUD_INIT_USER_DATA_TEMPLATE),CLOUD_INIT_USER_DATA_TEMPLATE)

.PHONY: clean
clean::
	rm -f *.seed.iso *.meta-data *.user-data

%.img: %.img.xz
	unxz --keep $<

%.qcow2: %.img
	qemu-img convert -f raw -O qcow2 $< $@
