/* Clzip - LZMA lossless data compressor
   Copyright (C) 2010-2025 Antonio Diaz Diaz.

   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#ifndef INT64_MAX
#define INT64_MAX 0x7FFFFFFFFFFFFFFFLL
#endif


typedef struct Block
  {
  long long pos, size;		/* pos >= 0, size >= 0, pos + size <= INT64_MAX */
  } Block;

static inline void init_block( Block * const b,
                               const long long p, const long long s )
  { b->pos = p; b->size = s; }

static inline long long block_end( const Block b ) { return b.pos + b.size; }


typedef struct Member
  {
  Block dblock, mblock;		/* data block, member block */
  unsigned dictionary_size;
  } Member;

static inline void init_member( Member * const m, const long long dpos,
                              const long long dsize, const long long mpos,
                              const long long msize, const unsigned dict_size )
  { init_block( &m->dblock, dpos, dsize );
    init_block( &m->mblock, mpos, msize ); m->dictionary_size = dict_size; }

typedef struct Lzip_index
  {
  Member * member_vector;
  char * error;
  long long insize;
  long members;
  int error_size;
  int retval;
  unsigned dictionary_size;	/* largest dictionary size in the file */
  } Lzip_index;

bool Li_init( Lzip_index * const li, const int infd,
              const Cl_options * const cl_opts );

void Li_free( Lzip_index * const li );

/* multimember file with empty member(s) */
static inline bool Li_multi_empty( Lzip_index * const li )
  {
  long i;
  if( li->members > 1 )
    for( i = 0; i < li->members; ++i )
      if( li->member_vector[i].dblock.size == 0 ) return true;
   return false;
  }

static inline long long Li_udata_size( const Lzip_index * const li )
  {
  if( li->members <= 0 ) return 0;
  return block_end( li->member_vector[li->members-1].dblock );
  }

static inline long long Li_cdata_size( const Lzip_index * const li )
  {
  if( li->members <= 0 ) return 0;
  return block_end( li->member_vector[li->members-1].mblock );
  }

  /* total size including trailing data (if any) */
static inline long long Li_file_size( const Lzip_index * const li )
  { if( li->insize >= 0 ) return li->insize; else return 0; }

static inline const Block * Li_dblock( const Lzip_index * const li,
                                       const long i )
  { return &li->member_vector[i].dblock; }

static inline const Block * Li_mblock( const Lzip_index * const li,
                                       const long i )
  { return &li->member_vector[i].mblock; }

static inline unsigned Li_dictionary_size( const Lzip_index * const li,
                                           const long i )
  { return li->member_vector[i].dictionary_size; }
