#ifndef _RHEOLEF_FIELD_COMPONENT_H
#define _RHEOLEF_FIELD_COMPONENT_H
///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================

/* TODO: when slip B.C. or Raviart-Thomas RT H(div) approx :
   uh[0] values are computed on the fly in a temporary
 
struct field_component {	
// data:
   field_basic<T,M>::iterator _start, _last;
   const space_constit&       _constit;         // a ce niveau
   field*                     _tmp_comp_ptr;    // si necesite un temporaire
// actions:
   field_component (field& uh, i_comp) {
	constit = uh.get_space().get_constitution();
	_constit = constit[i_comp]);
	if (! constit.has_slip_bc() && ! constit.has_hdiv()) {
          _tmp_comp_ptr = 0;
	  _star & _last : pointe dans le field uh
	} else {
	  space Vi (_constit);
          _tmp_comp_ptr = new field (Vi);
	  uh.compute_slip_comp (i_comp, *_tmp_comp_ptr);
	  _star & _last : pointe dans *_tmp_comp_ptr
	}
   }
   ~field_component () {
      if (_tmp_comp_ptr) delete _tmp_comp_ptr;
   }
};
*/

#include "rheolef/field.h"
#include "rheolef/field_indirect.h"

namespace rheolef {

// =========================================================================
// field_component
// =========================================================================
template<class T, class M>
class field_component {
public:

// typename:

  typedef typename field_basic<T,M>::size_type      size_type;
  typedef T                                         value_type;
  typedef M                                         memory_type;
  typedef typename scalar_traits<T>::type           scalar_type;
  typedef typename float_traits<T>::type            float_type;
  typedef typename field_basic<T,M>::iterator       iterator;
  typedef typename field_basic<T,M>::const_iterator const_iterator;

// allocators:

  field_component ();
  field_component (field_basic<T,M>&     uh, size_type i_comp);
  field_component (field_component<T,M>& uh, size_type i_comp);

  field_component<T,M>& operator= (const T& alpha);

  template <class Expr,
            class Sfinae
                  = typename std::enable_if<
                           details::is_field_expr_affine_homogeneous<Expr>::value
                      && ! details::is_field_expr_v2_constant       <Expr>::value
                    >::type>
  field_component<T,M>& operator=  (const Expr&);

  // explicit copy cstor (avoid simple copy of the proxy; see nfem/ptst/field_comp_assign_tst.cc )
  field_component<T,M>& operator= (const field_component<T,M>& expr) {
	return this -> template  operator=<field_component<T,M> > (expr);
  }

// high-level accessors & modifiers:

  field_indirect<T,M> operator[] (const geo_basic<T,M>& dom);
  field_indirect<T,M> operator[] (const std::string&    dom_name);

// recursive call:

  field_component<T,M> operator[] (size_t i_comp);

// low-level accessors & modifiers:

  const space_constitution<T,M>& get_constitution() const { return _constit; }
  std::string name() const { return _constit.name(); }
  geo_basic<T,M>   get_geo() const { return _constit.get_geo(); }
  space_basic<T,M> get_space() const { return space_basic<T,M>(_constit); }
  bool have_homogeneous_space (space_basic<T,M>& Xh) const { Xh = get_space(); return true; }
  const distributor& ownership() const { return _constit.ownership(); }	
  const communicator& comm() const { return ownership().comm(); }
  size_type     ndof() const { return ownership().size(); }
  size_type dis_ndof() const { return ownership().dis_size(); }
        T& dof (size_type idof)       { return _start [idof]; }
  const T& dof (size_type idof) const { return _start [idof]; }
  iterator begin_dof() { return _start; }
  iterator end_dof()   { return _last; }
  const_iterator begin_dof() const { return const_iterator(_start); }
  const_iterator end_dof() const   { return const_iterator(_last); }

// advanced usage:
  field_component<T,M>& proxy_assign (field_component<T,M>&& uh_comp);
  friend class field_component_const<T,M>;
  template<class Iterator>
  static void initialize (
      const space_constitution<T,M>& sup_constit,
      size_type                      i_comp,
      space_constitution<T,M>&       constit,
      Iterator&                      start,
      Iterator&                      last);
protected:
  static const size_type _unset = std::numeric_limits<size_type>::max();
// data:
  space_constitution<T,M>        _constit;     // at this level
  iterator                       _start;
  iterator                       _last;
};
// =========================================================================
// field_component_const
// =========================================================================
template<class T, class M>
class field_component_const {
public:

// typename:

  typedef typename field_basic<T,M>::size_type      size_type;
  typedef T                                         value_type;
  typedef M                                         memory_type;
  typedef typename scalar_traits<T>::type           scalar_type;
  typedef typename float_traits<T>::type            float_type;
  typedef typename field_basic<T,M>::const_iterator const_iterator;

// allocators:

  field_component_const ();
  field_component_const (const field_basic<T,M>& uh, size_type i_comp);
  field_component_const (const field_component<T,M>& uh_comp);
  const space_constitution<T,M>& get_constitution() const { return _constit; }
  std::string name() const { return _constit.name(); }
  const distributor& ownership() const { return _constit.ownership(); }	
  const communicator& comm() const { return ownership().comm(); }
  geo_basic<T,M>   get_geo() const { return _constit.get_geo(); }
  space_basic<T,M> get_space() const { return space_basic<T,M>(_constit); }
  bool have_homogeneous_space (space_basic<T,M>& Xh) const { Xh = get_space(); return true; }
  size_type     ndof() const { return ownership().size(); }
  size_type dis_ndof() const { return ownership().dis_size(); }
  const T& dof (size_type idof) const { return _start [idof]; }
  const_iterator begin_dof() const { return const_iterator(_start); }
  const_iterator end_dof() const   { return const_iterator(_last); }

// advanced usage:
  field_component_const<T,M>& proxy_assign (const field_component_const<T,M>& uh_comp);
protected:
  static const size_type _unset = std::numeric_limits<size_type>::max();
// data:
  space_constitution<T,M>        _constit;     // at this level
  const_iterator                 _start;
  const_iterator                 _last;
private:
  field_component_const<T,M>& operator= (const field_component_const<T,M>& uh_comp);
};
// =========================================================================
// inlined
// =========================================================================
template<class T, class M>
inline
field_component<T,M>::field_component()
 : _constit(),
   _start(),
   _last()
{
}
template<class T, class M>
inline
field_component_const<T,M>::field_component_const()
 : _constit(),
   _start(),
   _last()
{
}
template<class T, class M>
template<class Iterator>
void
field_component<T,M>::initialize (
  const space_constitution<T,M>& sup_constit,
  size_type                      i_comp,
  space_constitution<T,M>&       constit,
  Iterator&                      start,
  Iterator&                      last)
{
  if (sup_constit.is_hierarchical()) {
    size_type n_comp = sup_constit.size();
    check_macro (i_comp < n_comp,
      "field component index "<<i_comp<<" is out of range [0:"<<n_comp<<"[");
    size_type shift = 0;
    for (size_type j_comp = 0; j_comp < i_comp; j_comp++) {
      shift += sup_constit [j_comp].ndof();
    }
    constit = sup_constit [i_comp];
    size_type sz = constit.ndof();
    start += shift;	
    last  += shift + sz;
  } else {
    size_type n_comp = sup_constit.get_basis().size();
    check_macro (i_comp < n_comp,
      "field component index "<<i_comp<<" is out of range [0:"<<n_comp<<"[");
    constit = space_constitution<T,M>(sup_constit.get_geo(), sup_constit.get_basis()[i_comp].name());
    start += i_comp;
     last += i_comp + n_comp*constit.ndof();
    start.set_increment (n_comp);
     last.set_increment (n_comp);
  }
}
template<class T, class M>
field_component<T,M>::field_component (field_basic<T,M>& uh, size_type i_comp)
 : _constit(),
   _start(uh.begin_dof()),
   _last(uh.begin_dof())
{
  initialize (uh.get_space().get_constitution(), i_comp, _constit, _start, _last);
}
template<class T, class M>
field_component_const<T,M>::field_component_const (const field_basic<T,M>& uh, size_type i_comp)
 : _constit(),
   _start(uh.begin_dof()),
   _last(uh.begin_dof())
{
  field_component<T,M>::initialize (uh.get_space().get_constitution(), i_comp, _constit, _start, _last);
}
template<class T, class M>
field_component<T,M>::field_component (field_component<T,M>& uh, size_type i_comp)
 : _constit(),
   _start(uh._start),
   _last(uh._last)
{
  initialize (uh._constit, i_comp, _constit, _start, _last);
}
template<class T, class M>
inline
field_component_const<T,M>::field_component_const (const field_component<T,M>& uh_comp)
 : _constit (uh_comp.get_constitution()),
   _start(uh_comp.begin_dof()),
   _last(uh_comp.end_dof())
{
}
template<class T, class M>
inline
field_component_const<T,M>&
field_component_const<T,M>::operator= (const field_component_const<T,M>& uh_comp)
{
  // avoided at compile time: is private
  fatal_macro ("try to assign const field in field[i_comp] = field[j_comp]");
  return *this;
}
template<class T, class M>
inline
field_component_const<T,M>&
field_component_const<T,M>::proxy_assign (const field_component_const<T,M>& uh_comp)
{
   _constit = uh_comp.get_constitution();
   _start   = uh_comp.begin_dof();
   _last    = uh_comp.end_dof();
  return *this;
}
template<class T, class M>
inline
field_component<T,M>&
field_component<T,M>::proxy_assign (field_component<T,M>&& uh_comp)
{
   _constit = uh_comp.get_constitution();
   _start   = uh_comp.begin_dof();
   _last    = uh_comp.end_dof();
  return *this;
}
template<class T, class M>
inline
field_component<T,M>
field_basic<T,M>::operator[] (size_type i_comp) 
{
  dis_dof_indexes_requires_update();
  return field_component<T,M> (*this, i_comp);
}
template<class T, class M>
inline
field_component<T,M>
field_component<T,M>::operator[] (size_t i_comp)
{
  return field_component<T,M> (*this, i_comp);
}
template<class T, class M>
inline
field_component_const<T,M>
field_basic<T,M>::operator[] (size_type i_comp) const
{
  return field_component_const<T,M> (*this, i_comp);
}
template<class T, class M>
inline
field_component<T,M>&
field_component<T,M>::operator= (const T& alpha)
{
  std::fill (begin_dof(), end_dof(), alpha);
  return *this;
}
#ifdef TODO
// TODO: as: field vh; din >> vh ; uh[i_comp] = vh;
template <class T, class M>
inline
idiststream& operator >> (odiststream& ids, field_basic<T,M>& u);
#endif // TODO

template <class T, class M>
odiststream& operator << (odiststream& ods, const field_component<T,M>& uh_comp)
{
  return ods << field_basic<T,M>(uh_comp);
}
template <class T, class M>
inline
odiststream& operator << (odiststream& ods, const field_component_const<T,M>& uh_comp)
{
  return ods << field_basic<T,M>(uh_comp);
}
// =========================================================================
// uh[0]["boundary"] = alpha;
// => field_component & field_indirect together
// =========================================================================
template <class T, class M>
inline
field_indirect<T,M>::field_indirect (field_component<T,M>& uh_comp, const geo_basic<T,M>& dom)
  : _V(uh_comp.get_space()),
    _W(dom, _V.get_basis().name()),
    _dom(dom),
    _indirect(_V.build_indirect_array (_W, _dom.name())),
    _val(uh_comp.begin_dof())
{
}
template <class T, class M>
inline
field_indirect<T,M>
field_component<T,M>::operator[] (const geo_basic<T,M>& dom)
{
  return field_indirect<T,M> (*this, dom);
}
template <class T, class M>
inline
field_indirect<T,M>
field_component<T,M>::operator[] (const std::string& dom_name)
{
  return field_indirect<T,M> (*this, get_geo().operator[](dom_name));
}

} // namespace rheolef
#endif // _RHEOLEF_FIELD_COMPONENT_H
