/*
 * Copyright 2009-2019 The VOTCA Development Team (http://www.votca.org)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */
#define BOOST_TEST_MAIN

#define BOOST_TEST_MODULE aomatrix_test
#include "votca/xtp/orbitals.h"
#include <boost/test/unit_test.hpp>
#include <fstream>
#include <votca/xtp/sphere_lebedev_rule.h>
using namespace votca::xtp;
using namespace votca;

BOOST_AUTO_TEST_SUITE(sphere_lebedev_rule_test)

Eigen::VectorXd ReadVectorFromString(const std::string& matrix) {
  votca::tools::Tokenizer lines(matrix, "\n");

  std::vector<double> entries;
  for (auto line : lines) {
    if (line[0] == '#') {
      continue;
    }
    votca::tools::Tokenizer entries_tok(line, ", ");
    std::vector<std::string> temp = entries_tok.ToVector();

    for (const auto& s : temp) {
      entries.push_back(std::stod(s));
    }
  }

  return Eigen::Map<Eigen::VectorXd>(entries.data(), entries.size());
}

BOOST_AUTO_TEST_CASE(fine_test) {

  std::ofstream xyzfile("molecule.xyz");
  xyzfile << " 5" << std::endl;
  xyzfile << " methane" << std::endl;
  xyzfile << " Ge            .000000     .000000     .000000" << std::endl;
  xyzfile << " H            .629118     .629118     .629118" << std::endl;
  xyzfile << " H           -.629118    -.629118     .629118" << std::endl;
  xyzfile << " H            .629118    -.629118    -.629118" << std::endl;
  xyzfile << " H           -.629118     .629118    -.629118" << std::endl;
  xyzfile.close();

  QMMolecule mol("noname", 0);
  mol.LoadFromFile("molecule.xyz");

  LebedevGrid spheregrid;

  auto grid = spheregrid.CalculateSphericalGrids(mol, "fine");

  auto Hgrid = grid.at("H");
  auto Gegrid = grid.at("Ge");
  std::string Ge_phi_ref_string =
      "1.5708,1.5708,1.5708,1.5708,0,3.14159,0.955317,0.955317,"
      "0.955317,0.955317,2.18628,2.18628,2.18628,2.18628,0.060749,"
      "0.060749,0.060749,0.060749,3.08084,3.08084,3.08084,3.08084,1.52785,"
      "1.52785,1.52785,1.52785,1.61374,1.61374,1.61374,1.61374,1.52785,"
      "1.52785,1.52785,1.52785,1.61374,1.61374,1.61374,1.61374,0.149248,"
      "0.149248,0.149248,0.149248,2.99234,2.99234,2.99234,2.99234,1.46546,"
      "1.46546,1.46546,1.46546,1.67613,1.67613,1.67613,1.67613,1.46546,"
      "1.46546,1.46546,1.46546,1.67613,1.67613,1.67613,1.67613,0.25009,"
      "0.25009,0.25009,0.25009,2.8915,2.8915,2.8915,2.8915,1.39489,"
      "1.39489,1.39489,1.39489,1.7467,1.7467,1.7467,1.7467,1.39489,"
      "1.39489,1.39489,1.39489,1.7467,1.7467,1.7467,1.7467,0.357991,"
      "0.357991,0.357991,0.357991,2.7836,2.7836,2.7836,2.7836,1.32042,"
      "1.32042,1.32042,1.32042,1.82117,1.82117,1.82117,1.82117,1.32042,"
      "1.32042,1.32042,1.32042,1.82117,1.82117,1.82117,1.82117,0.470663,"
      "0.470663,0.470663,0.470663,2.67093,2.67093,2.67093,2.67093,1.24437,"
      "1.24437,1.24437,1.24437,1.89722,1.89722,1.89722,1.89722,1.24437,"
      "1.24437,1.24437,1.24437,1.89722,1.89722,1.89722,1.89722,0.587017,"
      "0.587017,0.587017,0.587017,2.55458,2.55458,2.55458,2.55458,1.16837,"
      "1.16837,1.16837,1.16837,1.97322,1.97322,1.97322,1.97322,1.16837,"
      "1.16837,1.16837,1.16837,1.97322,1.97322,1.97322,1.97322,0.706586,"
      "0.706586,0.706586,0.706586,2.43501,2.43501,2.43501,2.43501,1.09383,"
      "1.09383,1.09383,1.09383,2.04776,2.04776,2.04776,2.04776,1.09383,"
      "1.09383,1.09383,1.09383,2.04776,2.04776,2.04776,2.04776,0.829288,"
      "0.829288,0.829288,0.829288,2.3123,2.3123,2.3123,2.3123,1.02224,"
      "1.02224,1.02224,1.02224,2.11935,2.11935,2.11935,2.11935,1.02224,"
      "1.02224,1.02224,1.02224,2.11935,2.11935,2.11935,2.11935,1.08506,"
      "1.08506,1.08506,1.08506,2.05653,2.05653,2.05653,2.05653,0.895259,"
      "0.895259,0.895259,0.895259,2.24633,2.24633,2.24633,2.24633,"
      "0.895259,0.895259,0.895259,0.895259,2.24633,2.24633,2.24633,"
      "2.24633,1.21897,1.21897,1.21897,1.21897,1.92262,1.92262,1.92262,"
      "1.92262,0.844918,0.844918,0.844918,0.844918,2.29667,2.29667,"
      "2.29667,2.29667,0.844918,0.844918,0.844918,0.844918,2.29667,"
      "2.29667,2.29667,2.29667,1.35722,1.35722,1.35722,1.35722,1.78437,"
      "1.78437,1.78437,1.78437,0.807868,0.807868,0.807868,0.807868,"
      "2.33372,2.33372,2.33372,2.33372,0.807868,0.807868,0.807868,"
      "0.807868,2.33372,2.33372,2.33372,2.33372,1.49911,1.49911,1.49911,"
      "1.49911,1.64248,1.64248,1.64248,1.64248,0.787963,0.787963,0.787963,"
      "0.787963,2.35363,2.35363,2.35363,2.35363,0.787963,0.787963,"
      "0.787963,0.787963,2.35363,2.35363,2.35363,2.35363,1.5708,1.5708,"
      "1.5708,1.5708,1.5708,1.5708,1.5708,1.5708,0.123986,0.123986,"
      "3.01761,3.01761,1.44681,1.44681,1.69478,1.69478,0.123986,0.123986,"
      "3.01761,3.01761,1.44681,1.44681,1.69478,1.69478,1.5708,1.5708,"
      "1.5708,1.5708,1.5708,1.5708,1.5708,1.5708,0.29849,0.29849,2.8431,"
      "2.8431,1.27231,1.27231,1.86929,1.86929,0.29849,0.29849,2.8431,"
      "2.8431,1.27231,1.27231,1.86929,1.86929,1.5708,1.5708,1.5708,"
      "1.5708,1.5708,1.5708,1.5708,1.5708,0.489036,0.489036,2.65256,"
      "2.65256,1.08176,1.08176,2.05983,2.05983,0.489036,0.489036,2.65256,"
      "2.65256,1.08176,1.08176,2.05983,2.05983,1.5708,1.5708,1.5708,"
      "1.5708,1.5708,1.5708,1.5708,1.5708,0.686012,0.686012,2.45558,"
      "2.45558,0.884784,0.884784,2.25681,2.25681,0.686012,0.686012,"
      "2.45558,2.45558,0.884784,0.884784,2.25681,2.25681,0.213134,"
      "0.213134,0.213134,0.213134,2.92846,2.92846,2.92846,2.92846,1.36646,"
      "1.36646,1.36646,1.36646,1.77513,1.77513,1.77513,1.77513,0.213134,"
      "0.213134,0.213134,0.213134,2.92846,2.92846,2.92846,2.92846,1.51102,"
      "1.51102,1.51102,1.51102,1.63057,1.63057,1.63057,1.63057,1.36646,"
      "1.36646,1.36646,1.36646,1.77513,1.77513,1.77513,1.77513,1.51102,"
      "1.51102,1.51102,1.51102,1.63057,1.63057,1.63057,1.63057,0.501092,"
      "0.501092,0.501092,0.501092,2.6405,2.6405,2.6405,2.6405,1.09251,"
      "1.09251,1.09251,1.09251,2.04909,2.04909,2.04909,2.04909,0.501092,"
      "0.501092,0.501092,0.501092,2.6405,2.6405,2.6405,2.6405,1.43278,"
      "1.43278,1.43278,1.43278,1.70881,1.70881,1.70881,1.70881,1.09251,"
      "1.09251,1.09251,1.09251,2.04909,2.04909,2.04909,2.04909,1.43278,"
      "1.43278,1.43278,1.43278,1.70881,1.70881,1.70881,1.70881,0.651884,"
      "0.651884,0.651884,0.651884,2.48971,2.48971,2.48971,2.48971,1.04365,"
      "1.04365,1.04365,1.04365,2.09794,2.09794,2.09794,2.09794,0.651884,"
      "0.651884,0.651884,0.651884,2.48971,2.48971,2.48971,2.48971,1.22483,"
      "1.22483,1.22483,1.22483,1.91676,1.91676,1.91676,1.91676,1.04365,"
      "1.04365,1.04365,1.04365,2.09794,2.09794,2.09794,2.09794,1.22483,"
      "1.22483,1.22483,1.22483,1.91676,1.91676,1.91676,1.91676,0.314277,"
      "0.314277,0.314277,0.314277,2.82732,2.82732,2.82732,2.82732,1.28517,"
      "1.28517,1.28517,1.28517,1.85642,1.85642,1.85642,1.85642,0.314277,"
      "0.314277,0.314277,0.314277,2.82732,2.82732,2.82732,2.82732,1.44328,"
      "1.44328,1.44328,1.44328,1.69831,1.69831,1.69831,1.69831,1.28517,"
      "1.28517,1.28517,1.28517,1.85642,1.85642,1.85642,1.85642,1.44328,"
      "1.44328,1.44328,1.44328,1.69831,1.69831,1.69831,1.69831,0.535245,"
      "0.535245,0.535245,0.535245,2.60635,2.60635,2.60635,2.60635,1.1228,"
      "1.1228,1.1228,1.1228,2.01879,2.01879,2.01879,2.01879,0.535245,"
      "0.535245,0.535245,0.535245,2.60635,2.60635,2.60635,2.60635,1.29812,"
      "1.29812,1.29812,1.29812,1.84347,1.84347,1.84347,1.84347,1.1228,"
      "1.1228,1.1228,1.1228,2.01879,2.01879,2.01879,2.01879,1.29812,"
      "1.29812,1.29812,1.29812,1.84347,1.84347,1.84347,1.84347,0.696485,"
      "0.696485,0.696485,0.696485,2.44511,2.44511,2.44511,2.44511,"
      "0.894874,0.894874,0.894874,0.894874,2.24672,2.24672,2.24672,"
      "2.24672,0.696485,0.696485,0.696485,0.696485,2.44511,2.44511,"
      "2.44511,2.44511,1.42834,1.42834,1.42834,1.42834,1.71326,1.71326,"
      "1.71326,1.71326,0.894874,0.894874,0.894874,0.894874,2.24672,"
      "2.24672,2.24672,2.24672,1.42834,1.42834,1.42834,1.42834,1.71326,"
      "1.71326,1.71326,1.71326,0.395988,0.395988,0.395988,0.395988,"
      "2.74561,2.74561,2.74561,2.74561,1.18117,1.18117,1.18117,1.18117,"
      "1.96042,1.96042,1.96042,1.96042,0.395988,0.395988,0.395988,"
      "0.395988,2.74561,2.74561,2.74561,2.74561,1.50365,1.50365,1.50365,"
      "1.50365,1.63794,1.63794,1.63794,1.63794,1.18117,1.18117,1.18117,"
      "1.18117,1.96042,1.96042,1.96042,1.96042,1.50365,1.50365,1.50365,"
      "1.50365,1.63794,1.63794,1.63794,1.63794,0.589862,0.589862,0.589862,"
      "0.589862,2.55173,2.55173,2.55173,2.55173,0.986335,0.986335,"
      "0.986335,0.986335,2.15526,2.15526,2.15526,2.15526,0.589862,"
      "0.589862,0.589862,0.589862,2.55173,2.55173,2.55173,2.55173,1.50016,"
      "1.50016,1.50016,1.50016,1.64143,1.64143,1.64143,1.64143,0.986335,"
      "0.986335,0.986335,0.986335,2.15526,2.15526,2.15526,2.15526,1.50016,"
      "1.50016,1.50016,1.50016,1.64143,1.64143,1.64143,1.64143,0.726324,"
      "0.726324,0.726324,0.726324,2.41527,2.41527,2.41527,2.41527,"
      "0.923588,0.923588,0.923588,0.923588,2.218,2.218,2.218,2.218,"
      "0.726324,0.726324,0.726324,0.726324,2.41527,2.41527,2.41527,"
      "2.41527,1.28868,1.28868,1.28868,1.28868,1.85291,1.85291,1.85291,"
      "1.85291,0.923588,0.923588,0.923588,0.923588,2.218,2.218,2.218,"
      "2.218,1.28868,1.28868,1.28868,1.28868,1.85291,1.85291,1.85291,"
      "1.85291,0.422374,0.422374,0.422374,0.422374,2.71922,2.71922,"
      "2.71922,2.71922,1.20364,1.20364,1.20364,1.20364,1.93795,1.93795,"
      "1.93795,1.93795,0.422374,0.422374,0.422374,0.422374,2.71922,"
      "2.71922,2.71922,2.71922,1.37152,1.37152,1.37152,1.37152,1.77007,"
      "1.77007,1.77007,1.77007,1.20364,1.20364,1.20364,1.20364,1.93795,"
      "1.93795,1.93795,1.93795,1.37152,1.37152,1.37152,1.37152,1.77007,"
      "1.77007,1.77007,1.77007,0.611568,0.611568,0.611568,0.611568,"
      "2.53002,2.53002,2.53002,2.53002,1.00645,1.00645,1.00645,1.00645,"
      "2.13515,2.13515,2.13515,2.13515,0.611568,0.611568,0.611568,"
      "0.611568,2.53002,2.53002,2.53002,2.53002,1.36052,1.36052,1.36052,"
      "1.36052,1.78107,1.78107,1.78107,1.78107,1.00645,1.00645,1.00645,"
      "1.00645,2.13515,2.13515,2.13515,2.13515,1.36052,1.36052,1.36052,"
      "1.36052,1.78107,1.78107,1.78107,1.78107,0.771893,0.771893,0.771893,"
      "0.771893,2.3697,2.3697,2.3697,2.3697,0.96733,0.96733,0.96733,"
      "0.96733,2.17426,2.17426,2.17426,2.17426,0.771893,0.771893,0.771893,"
      "0.771893,2.3697,2.3697,2.3697,2.3697,1.15326,1.15326,1.15326,"
      "1.15326,1.98834,1.98834,1.98834,1.98834,0.96733,0.96733,0.96733,"
      "0.96733,2.17426,2.17426,2.17426,2.17426,1.15326,1.15326,1.15326,"
      "1.15326,1.98834,1.98834,1.98834,1.98834";
  Eigen::VectorXd Ge_phi_ref = ReadVectorFromString(Ge_phi_ref_string);

  std::string Ge_theta_ref_string =
      "0,3.14159,1.5708,-1.5708,0,0,0.785398,2.35619,"
      "-0.785398,-2.35619,0.785398,2.35619,-0.785398,-2.35619,0.785398,"
      "2.35619,-0.785398,-2.35619,0.785398,2.35619,-0.785398,-2.35619,"
      "1.52781,1.61378,-1.52781,-1.61378,1.52781,1.61378,-1.52781,"
      "-1.61378,0.0429825,3.09861,-0.0429825,-3.09861,0.0429825,3.09861,"
      "-0.0429825,-3.09861,0.785398,2.35619,-0.785398,-2.35619,0.785398,"
      "2.35619,-0.785398,-2.35619,1.46487,1.67672,-1.46487,-1.67672,"
      "1.46487,1.67672,-1.46487,-1.67672,0.105927,3.03567,-0.105927,"
      "-3.03567,0.105927,3.03567,-0.105927,-3.03567,0.785398,2.35619,"
      "-0.785398,-2.35619,0.785398,2.35619,-0.785398,-2.35619,1.3921,"
      "1.74949,-1.3921,-1.74949,1.3921,1.74949,-1.3921,-1.74949,0.178695,"
      "2.9629,-0.178695,-2.9629,0.178695,2.9629,-0.178695,-2.9629,"
      "0.785398,2.35619,-0.785398,-2.35619,0.785398,2.35619,-0.785398,"
      "-2.35619,1.31218,1.82941,-1.31218,-1.82941,1.31218,1.82941,"
      "-1.31218,-1.82941,0.258612,2.88298,-0.258612,-2.88298,0.258612,"
      "2.88298,-0.258612,-2.88298,0.785398,2.35619,-0.785398,-2.35619,"
      "0.785398,2.35619,-0.785398,-2.35619,1.22544,1.91615,-1.22544,"
      "-1.91615,1.22544,1.91615,-1.22544,-1.91615,0.345357,2.79624,"
      "-0.345357,-2.79624,0.345357,2.79624,-0.345357,-2.79624,0.785398,"
      "2.35619,-0.785398,-2.35619,0.785398,2.35619,-0.785398,-2.35619,"
      "1.13111,2.01048,-1.13111,-2.01048,1.13111,2.01048,-1.13111,"
      "-2.01048,0.439687,2.70191,-0.439687,-2.70191,0.439687,2.70191,"
      "-0.439687,-2.70191,0.785398,2.35619,-0.785398,-2.35619,0.785398,"
      "2.35619,-0.785398,-2.35619,1.02774,2.11385,-1.02774,-2.11385,"
      "1.02774,2.11385,-1.02774,-2.11385,0.543057,2.59854,-0.543057,"
      "-2.59854,0.543057,2.59854,-0.543057,-2.59854,0.785398,2.35619,"
      "-0.785398,-2.35619,0.785398,2.35619,-0.785398,-2.35619,0.91332,"
      "2.22827,-0.91332,-2.22827,0.91332,2.22827,-0.91332,-2.22827,"
      "0.657477,2.48412,-0.657477,-2.48412,0.657477,2.48412,-0.657477,"
      "-2.48412,0.785398,2.35619,-0.785398,-2.35619,0.785398,2.35619,"
      "-0.785398,-2.35619,0.641319,2.50027,-0.641319,-2.50027,0.641319,"
      "2.50027,-0.641319,-2.50027,0.929478,2.21212,-0.929478,-2.21212,"
      "0.929478,2.21212,-0.929478,-2.21212,0.785398,2.35619,-0.785398,"
      "-2.35619,0.785398,2.35619,-0.785398,-2.35619,0.478857,2.66274,"
      "-0.478857,-2.66274,0.478857,2.66274,-0.478857,-2.66274,1.09194,"
      "2.04965,-1.09194,-2.04965,1.09194,2.04965,-1.09194,-2.04965,"
      "0.785398,2.35619,-0.785398,-2.35619,0.785398,2.35619,-0.785398,"
      "-2.35619,0.297608,2.84398,-0.297608,-2.84398,0.297608,2.84398,"
      "-0.297608,-2.84398,1.27319,1.8684,-1.27319,-1.8684,1.27319,1.8684,"
      "-1.27319,-1.8684,0.785398,2.35619,-0.785398,-2.35619,0.785398,"
      "2.35619,-0.785398,-2.35619,0.101206,3.04039,-0.101206,-3.04039,"
      "0.101206,3.04039,-0.101206,-3.04039,1.46959,1.672,-1.46959,-1.672,"
      "1.46959,1.672,-1.46959,-1.672,1.44681,1.69478,-1.44681,-1.69478,"
      "0.123986,3.01761,-0.123986,-3.01761,0,3.14159,0,3.14159,0,"
      "3.14159,0,3.14159,1.5708,-1.5708,1.5708,-1.5708,1.5708,-1.5708,"
      "1.5708,-1.5708,1.27231,1.86929,-1.27231,-1.86929,0.29849,2.8431,"
      "-0.29849,-2.8431,0,3.14159,0,3.14159,0,3.14159,0,3.14159,1.5708,"
      "-1.5708,1.5708,-1.5708,1.5708,-1.5708,1.5708,-1.5708,1.08176,"
      "2.05983,-1.08176,-2.05983,0.489036,2.65256,-0.489036,-2.65256,0,"
      "3.14159,0,3.14159,0,3.14159,0,3.14159,1.5708,-1.5708,1.5708,"
      "-1.5708,1.5708,-1.5708,1.5708,-1.5708,0.884784,2.25681,-0.884784,"
      "-2.25681,0.686012,2.45558,-0.686012,-2.45558,0,3.14159,0,3.14159,"
      "0,3.14159,0,3.14159,1.5708,-1.5708,1.5708,-1.5708,1.5708,-1.5708,"
      "1.5708,-1.5708,1.28447,1.85712,-1.28447,-1.85712,1.28447,1.85712,"
      "-1.28447,-1.85712,1.50975,1.63184,-1.50975,-1.63184,1.50975,"
      "1.63184,-1.50975,-1.63184,0.286325,2.85527,-0.286325,-2.85527,"
      "0.286325,2.85527,-0.286325,-2.85527,1.36609,1.7755,-1.36609,"
      "-1.7755,1.36609,1.7755,-1.36609,-1.7755,0.0610476,3.08055,"
      "-0.0610476,-3.08055,0.0610476,3.08055,-0.0610476,-3.08055,0.204703,"
      "2.93689,-0.204703,-2.93689,0.204703,2.93689,-0.204703,-2.93689,"
      "1.28034,1.86125,-1.28034,-1.86125,1.28034,1.86125,-1.28034,"
      "-1.86125,1.4152,1.72639,-1.4152,-1.72639,1.4152,1.72639,-1.4152,"
      "-1.72639,0.290455,2.85114,-0.290455,-2.85114,0.290455,2.85114,"
      "-0.290455,-2.85114,1.08752,2.05407,-1.08752,-2.05407,1.08752,"
      "2.05407,-1.08752,-2.05407,0.155593,2.986,-0.155593,-2.986,0.155593,"
      "2.986,-0.155593,-2.986,0.483274,2.65832,-0.483274,-2.65832,"
      "0.483274,2.65832,-0.483274,-2.65832,0.977687,2.16391,-0.977687,"
      "-2.16391,0.977687,2.16391,-0.977687,-2.16391,1.16759,1.974,"
      "-1.16759,-1.974,1.16759,1.974,-1.16759,-1.974,0.593109,2.54848,"
      "-0.593109,-2.54848,0.593109,2.54848,-0.593109,-2.54848,1.00658,"
      "2.13501,-1.00658,-2.13501,1.00658,2.13501,-1.00658,-2.13501,"
      "0.403203,2.73839,-0.403203,-2.73839,0.403203,2.73839,-0.403203,"
      "-2.73839,0.564214,2.57738,-0.564214,-2.57738,0.564214,2.57738,"
      "-0.564214,-2.57738,1.14684,1.99476,-1.14684,-1.99476,1.14684,"
      "1.99476,-1.14684,-1.99476,1.43787,1.70372,-1.43787,-1.70372,"
      "1.43787,1.70372,-1.43787,-1.70372,0.423961,2.71763,-0.423961,"
      "-2.71763,0.423961,2.71763,-0.423961,-2.71763,1.28276,1.85883,"
      "-1.28276,-1.85883,1.28276,1.85883,-1.28276,-1.85883,0.132928,"
      "3.00866,-0.132928,-3.00866,0.132928,3.00866,-0.132928,-3.00866,"
      "0.288033,2.85356,-0.288033,-2.85356,0.288033,2.85356,-0.288033,"
      "-2.85356,1.01454,2.12705,-1.01454,-2.12705,1.01454,2.12705,"
      "-1.01454,-2.12705,1.26736,1.87423,-1.26736,-1.87423,1.26736,"
      "1.87423,-1.26736,-1.87423,0.556254,2.58534,-0.556254,-2.58534,"
      "0.556254,2.58534,-0.556254,-2.58534,1.10428,2.03731,-1.10428,"
      "-2.03731,1.10428,2.03731,-1.10428,-2.03731,0.303433,2.83816,"
      "-0.303433,-2.83816,0.303433,2.83816,-0.303433,-2.83816,0.466512,"
      "2.67508,-0.466512,-2.67508,0.466512,2.67508,-0.466512,-2.67508,"
      "1.34763,1.79396,-1.34763,-1.79396,1.34763,1.79396,-1.34763,"
      "-1.79396,1.38778,1.75381,-1.38778,-1.75381,1.38778,1.75381,"
      "-1.38778,-1.75381,0.223162,2.91843,-0.223162,-2.91843,0.223162,"
      "2.91843,-0.223162,-2.91843,0.88664,2.25495,-0.88664,-2.25495,"
      "0.88664,2.25495,-0.88664,-2.25495,0.183013,2.95858,-0.183013,"
      "-2.95858,0.183013,2.95858,-0.183013,-2.95858,0.684156,2.45744,"
      "-0.684156,-2.45744,0.684156,2.45744,-0.684156,-2.45744,1.39596,"
      "1.74563,-1.39596,-1.74563,1.39596,1.74563,-1.39596,-1.74563,1.4982,"
      "1.64339,-1.4982,-1.64339,1.4982,1.64339,-1.4982,-1.64339,0.174831,"
      "2.96676,-0.174831,-2.96676,0.174831,2.96676,-0.174831,-2.96676,"
      "1.18025,1.96135,-1.18025,-1.96135,1.18025,1.96135,-1.18025,"
      "-1.96135,0.0725924,3.069,-0.0725924,-3.069,0.0725924,3.069,"
      "-0.0725924,-3.069,0.39055,2.75104,-0.39055,-2.75104,0.39055,"
      "2.75104,-0.39055,-2.75104,1.44357,1.69802,-1.44357,-1.69802,"
      "1.44357,1.69802,-1.44357,-1.69802,1.48607,1.65552,-1.48607,"
      "-1.65552,1.48607,1.65552,-1.48607,-1.65552,0.127224,3.01437,"
      "-0.127224,-3.01437,0.127224,3.01437,-0.127224,-3.01437,0.98468,"
      "2.15691,-0.98468,-2.15691,0.98468,2.15691,-0.98468,-2.15691,"
      "0.0847256,3.05687,-0.0847256,-3.05687,0.0847256,3.05687,-0.0847256,"
      "-3.05687,0.586117,2.55548,-0.586117,-2.55548,0.586117,2.55548,"
      "-0.586117,-2.55548,1.13825,2.00334,-1.13825,-2.00334,1.13825,"
      "2.00334,-1.13825,-2.00334,1.21434,1.92726,-1.21434,-1.92726,"
      "1.21434,1.92726,-1.21434,-1.92726,0.432543,2.70905,-0.432543,"
      "-2.70905,0.432543,2.70905,-0.432543,-2.70905,0.8921,2.24949,"
      "-0.8921,-2.24949,0.8921,2.24949,-0.8921,-2.24949,0.35646,2.78513,"
      "-0.35646,-2.78513,0.35646,2.78513,-0.35646,-2.78513,0.678697,"
      "2.4629,-0.678697,-2.4629,0.678697,2.4629,-0.678697,-2.4629,1.06682,"
      "2.07477,-1.06682,-2.07477,1.06682,2.07477,-1.06682,-2.07477,"
      "1.35708,1.78451,-1.35708,-1.78451,1.35708,1.78451,-1.35708,"
      "-1.78451,0.503975,2.63762,-0.503975,-2.63762,0.503975,2.63762,"
      "-0.503975,-2.63762,1.19587,1.94573,-1.19587,-1.94573,1.19587,"
      "1.94573,-1.19587,-1.94573,0.213717,2.92788,-0.213717,-2.92788,"
      "0.213717,2.92788,-0.213717,-2.92788,0.37493,2.76666,-0.37493,"
      "-2.76666,0.37493,2.76666,-0.37493,-2.76666,1.19872,1.94287,"
      "-1.19872,-1.94287,1.19872,1.94287,-1.19872,-1.94287,1.32117,"
      "1.82042,-1.32117,-1.82042,1.32117,1.82042,-1.32117,-1.82042,"
      "0.372071,2.76952,-0.372071,-2.76952,0.372071,2.76952,-0.372071,"
      "-2.76952,0.992123,2.14947,-0.992123,-2.14947,0.992123,2.14947,"
      "-0.992123,-2.14947,0.249621,2.89197,-0.249621,-2.89197,0.249621,"
      "2.89197,-0.249621,-2.89197,0.578673,2.56292,-0.578673,-2.56292,"
      "0.578673,2.56292,-0.578673,-2.56292,0.950366,2.19123,-0.950366,"
      "-2.19123,0.950366,2.19123,-0.950366,-2.19123,1.05584,2.08576,"
      "-1.05584,-2.08576,1.05584,2.08576,-1.05584,-2.08576,0.620431,"
      "2.52116,-0.620431,-2.52116,0.620431,2.52116,-0.620431,-2.52116,"
      "0.900988,2.2406,-0.900988,-2.2406,0.900988,2.2406,-0.900988,"
      "-2.2406,0.514961,2.62663,-0.514961,-2.62663,0.514961,2.62663,"
      "-0.514961,-2.62663,0.669809,2.47178,-0.669809,-2.47178,0.669809,"
      "2.47178,-0.669809,-2.47178";
  Eigen::VectorXd Ge_theta_ref = ReadVectorFromString(Ge_theta_ref_string);

  std::string Ge_weight_ref_string =
      "0.00180741,0.00180741,0.00180741,0.00180741,0.00180741,"
      "0.00180741,0.0141469,0.0141469,0.0141469,0.0141469,0.0141469,"
      "0.0141469,0.0141469,0.0141469,0.00621788,0.00621788,0.00621788,"
      "0.00621788,0.00621788,0.00621788,0.00621788,0.00621788,0.00621788,"
      "0.00621788,0.00621788,0.00621788,0.00621788,0.00621788,0.00621788,"
      "0.00621788,0.00621788,0.00621788,0.00621788,0.00621788,0.00621788,"
      "0.00621788,0.00621788,0.00621788,0.00924632,0.00924632,0.00924632,"
      "0.00924632,0.00924632,0.00924632,0.00924632,0.00924632,0.00924632,"
      "0.00924632,0.00924632,0.00924632,0.00924632,0.00924632,0.00924632,"
      "0.00924632,0.00924632,0.00924632,0.00924632,0.00924632,0.00924632,"
      "0.00924632,0.00924632,0.00924632,0.0111704,0.0111704,0.0111704,"
      "0.0111704,0.0111704,0.0111704,0.0111704,0.0111704,0.0111704,"
      "0.0111704,0.0111704,0.0111704,0.0111704,0.0111704,0.0111704,"
      "0.0111704,0.0111704,0.0111704,0.0111704,0.0111704,0.0111704,"
      "0.0111704,0.0111704,0.0111704,0.0124261,0.0124261,0.0124261,"
      "0.0124261,0.0124261,0.0124261,0.0124261,0.0124261,0.0124261,"
      "0.0124261,0.0124261,0.0124261,0.0124261,0.0124261,0.0124261,"
      "0.0124261,0.0124261,0.0124261,0.0124261,0.0124261,0.0124261,"
      "0.0124261,0.0124261,0.0124261,0.0132362,0.0132362,0.0132362,"
      "0.0132362,0.0132362,0.0132362,0.0132362,0.0132362,0.0132362,"
      "0.0132362,0.0132362,0.0132362,0.0132362,0.0132362,0.0132362,"
      "0.0132362,0.0132362,0.0132362,0.0132362,0.0132362,0.0132362,"
      "0.0132362,0.0132362,0.0132362,0.0137323,0.0137323,0.0137323,"
      "0.0137323,0.0137323,0.0137323,0.0137323,0.0137323,0.0137323,"
      "0.0137323,0.0137323,0.0137323,0.0137323,0.0137323,0.0137323,"
      "0.0137323,0.0137323,0.0137323,0.0137323,0.0137323,0.0137323,"
      "0.0137323,0.0137323,0.0137323,0.0140038,0.0140038,0.0140038,"
      "0.0140038,0.0140038,0.0140038,0.0140038,0.0140038,0.0140038,"
      "0.0140038,0.0140038,0.0140038,0.0140038,0.0140038,0.0140038,"
      "0.0140038,0.0140038,0.0140038,0.0140038,0.0140038,0.0140038,"
      "0.0140038,0.0140038,0.0140038,0.0141211,0.0141211,0.0141211,"
      "0.0141211,0.0141211,0.0141211,0.0141211,0.0141211,0.0141211,"
      "0.0141211,0.0141211,0.0141211,0.0141211,0.0141211,0.0141211,"
      "0.0141211,0.0141211,0.0141211,0.0141211,0.0141211,0.0141211,"
      "0.0141211,0.0141211,0.0141211,0.0141402,0.0141402,0.0141402,"
      "0.0141402,0.0141402,0.0141402,0.0141402,0.0141402,0.0141402,"
      "0.0141402,0.0141402,0.0141402,0.0141402,0.0141402,0.0141402,"
      "0.0141402,0.0141402,0.0141402,0.0141402,0.0141402,0.0141402,"
      "0.0141402,0.0141402,0.0141402,0.0141517,0.0141517,0.0141517,"
      "0.0141517,0.0141517,0.0141517,0.0141517,0.0141517,0.0141517,"
      "0.0141517,0.0141517,0.0141517,0.0141517,0.0141517,0.0141517,"
      "0.0141517,0.0141517,0.0141517,0.0141517,0.0141517,0.0141517,"
      "0.0141517,0.0141517,0.0141517,0.0142036,0.0142036,0.0142036,"
      "0.0142036,0.0142036,0.0142036,0.0142036,0.0142036,0.0142036,"
      "0.0142036,0.0142036,0.0142036,0.0142036,0.0142036,0.0142036,"
      "0.0142036,0.0142036,0.0142036,0.0142036,0.0142036,0.0142036,"
      "0.0142036,0.0142036,0.0142036,0.0142627,0.0142627,0.0142627,"
      "0.0142627,0.0142627,0.0142627,0.0142627,0.0142627,0.0142627,"
      "0.0142627,0.0142627,0.0142627,0.0142627,0.0142627,0.0142627,"
      "0.0142627,0.0142627,0.0142627,0.0142627,0.0142627,0.0142627,"
      "0.0142627,0.0142627,0.0142627,0.0085745,0.0085745,0.0085745,"
      "0.0085745,0.0085745,0.0085745,0.0085745,0.0085745,0.0085745,"
      "0.0085745,0.0085745,0.0085745,0.0085745,0.0085745,0.0085745,"
      "0.0085745,0.0085745,0.0085745,0.0085745,0.0085745,0.0085745,"
      "0.0085745,0.0085745,0.0085745,0.0118804,0.0118804,0.0118804,"
      "0.0118804,0.0118804,0.0118804,0.0118804,0.0118804,0.0118804,"
      "0.0118804,0.0118804,0.0118804,0.0118804,0.0118804,0.0118804,"
      "0.0118804,0.0118804,0.0118804,0.0118804,0.0118804,0.0118804,"
      "0.0118804,0.0118804,0.0118804,0.0135017,0.0135017,0.0135017,"
      "0.0135017,0.0135017,0.0135017,0.0135017,0.0135017,0.0135017,"
      "0.0135017,0.0135017,0.0135017,0.0135017,0.0135017,0.0135017,"
      "0.0135017,0.0135017,0.0135017,0.0135017,0.0135017,0.0135017,"
      "0.0135017,0.0135017,0.0135017,0.0141912,0.0141912,0.0141912,"
      "0.0141912,0.0141912,0.0141912,0.0141912,0.0141912,0.0141912,"
      "0.0141912,0.0141912,0.0141912,0.0141912,0.0141912,0.0141912,"
      "0.0141912,0.0141912,0.0141912,0.0141912,0.0141912,0.0141912,"
      "0.0141912,0.0141912,0.0141912,0.0106021,0.0106021,0.0106021,"
      "0.0106021,0.0106021,0.0106021,0.0106021,0.0106021,0.0106021,"
      "0.0106021,0.0106021,0.0106021,0.0106021,0.0106021,0.0106021,"
      "0.0106021,0.0106021,0.0106021,0.0106021,0.0106021,0.0106021,"
      "0.0106021,0.0106021,0.0106021,0.0106021,0.0106021,0.0106021,"
      "0.0106021,0.0106021,0.0106021,0.0106021,0.0106021,0.0106021,"
      "0.0106021,0.0106021,0.0106021,0.0106021,0.0106021,0.0106021,"
      "0.0106021,0.0106021,0.0106021,0.0106021,0.0106021,0.0106021,"
      "0.0106021,0.0106021,0.0106021,0.0135121,0.0135121,0.0135121,"
      "0.0135121,0.0135121,0.0135121,0.0135121,0.0135121,0.0135121,"
      "0.0135121,0.0135121,0.0135121,0.0135121,0.0135121,0.0135121,"
      "0.0135121,0.0135121,0.0135121,0.0135121,0.0135121,0.0135121,"
      "0.0135121,0.0135121,0.0135121,0.0135121,0.0135121,0.0135121,"
      "0.0135121,0.0135121,0.0135121,0.0135121,0.0135121,0.0135121,"
      "0.0135121,0.0135121,0.0135121,0.0135121,0.0135121,0.0135121,"
      "0.0135121,0.0135121,0.0135121,0.0135121,0.0135121,0.0135121,"
      "0.0135121,0.0135121,0.0135121,0.0139308,0.0139308,0.0139308,"
      "0.0139308,0.0139308,0.0139308,0.0139308,0.0139308,0.0139308,"
      "0.0139308,0.0139308,0.0139308,0.0139308,0.0139308,0.0139308,"
      "0.0139308,0.0139308,0.0139308,0.0139308,0.0139308,0.0139308,"
      "0.0139308,0.0139308,0.0139308,0.0139308,0.0139308,0.0139308,"
      "0.0139308,0.0139308,0.0139308,0.0139308,0.0139308,0.0139308,"
      "0.0139308,0.0139308,0.0139308,0.0139308,0.0139308,0.0139308,"
      "0.0139308,0.0139308,0.0139308,0.0139308,0.0139308,0.0139308,"
      "0.0139308,0.0139308,0.0139308,0.0120216,0.0120216,0.0120216,"
      "0.0120216,0.0120216,0.0120216,0.0120216,0.0120216,0.0120216,"
      "0.0120216,0.0120216,0.0120216,0.0120216,0.0120216,0.0120216,"
      "0.0120216,0.0120216,0.0120216,0.0120216,0.0120216,0.0120216,"
      "0.0120216,0.0120216,0.0120216,0.0120216,0.0120216,0.0120216,"
      "0.0120216,0.0120216,0.0120216,0.0120216,0.0120216,0.0120216,"
      "0.0120216,0.0120216,0.0120216,0.0120216,0.0120216,0.0120216,"
      "0.0120216,0.0120216,0.0120216,0.0120216,0.0120216,0.0120216,"
      "0.0120216,0.0120216,0.0120216,0.01358,0.01358,0.01358,0.01358,"
      "0.01358,0.01358,0.01358,0.01358,0.01358,0.01358,0.01358,0.01358,"
      "0.01358,0.01358,0.01358,0.01358,0.01358,0.01358,0.01358,0.01358,"
      "0.01358,0.01358,0.01358,0.01358,0.01358,0.01358,0.01358,0.01358,"
      "0.01358,0.01358,0.01358,0.01358,0.01358,0.01358,0.01358,0.01358,"
      "0.01358,0.01358,0.01358,0.01358,0.01358,0.01358,0.01358,0.01358,"
      "0.01358,0.01358,0.01358,0.01358,0.0141598,0.0141598,0.0141598,"
      "0.0141598,0.0141598,0.0141598,0.0141598,0.0141598,0.0141598,"
      "0.0141598,0.0141598,0.0141598,0.0141598,0.0141598,0.0141598,"
      "0.0141598,0.0141598,0.0141598,0.0141598,0.0141598,0.0141598,"
      "0.0141598,0.0141598,0.0141598,0.0141598,0.0141598,0.0141598,"
      "0.0141598,0.0141598,0.0141598,0.0141598,0.0141598,0.0141598,"
      "0.0141598,0.0141598,0.0141598,0.0141598,0.0141598,0.0141598,"
      "0.0141598,0.0141598,0.0141598,0.0141598,0.0141598,0.0141598,"
      "0.0141598,0.0141598,0.0141598,0.01285,0.01285,0.01285,0.01285,"
      "0.01285,0.01285,0.01285,0.01285,0.01285,0.01285,0.01285,0.01285,"
      "0.01285,0.01285,0.01285,0.01285,0.01285,0.01285,0.01285,0.01285,"
      "0.01285,0.01285,0.01285,0.01285,0.01285,0.01285,0.01285,0.01285,"
      "0.01285,0.01285,0.01285,0.01285,0.01285,0.01285,0.01285,0.01285,"
      "0.01285,0.01285,0.01285,0.01285,0.01285,0.01285,0.01285,0.01285,"
      "0.01285,0.01285,0.01285,0.01285,0.0139356,0.0139356,0.0139356,"
      "0.0139356,0.0139356,0.0139356,0.0139356,0.0139356,0.0139356,"
      "0.0139356,0.0139356,0.0139356,0.0139356,0.0139356,0.0139356,"
      "0.0139356,0.0139356,0.0139356,0.0139356,0.0139356,0.0139356,"
      "0.0139356,0.0139356,0.0139356,0.0139356,0.0139356,0.0139356,"
      "0.0139356,0.0139356,0.0139356,0.0139356,0.0139356,0.0139356,"
      "0.0139356,0.0139356,0.0139356,0.0139356,0.0139356,0.0139356,"
      "0.0139356,0.0139356,0.0139356,0.0139356,0.0139356,0.0139356,"
      "0.0139356,0.0139356,0.0139356,0.0141094,0.0141094,0.0141094,"
      "0.0141094,0.0141094,0.0141094,0.0141094,0.0141094,0.0141094,"
      "0.0141094,0.0141094,0.0141094,0.0141094,0.0141094,0.0141094,"
      "0.0141094,0.0141094,0.0141094,0.0141094,0.0141094,0.0141094,"
      "0.0141094,0.0141094,0.0141094,0.0141094,0.0141094,0.0141094,"
      "0.0141094,0.0141094,0.0141094,0.0141094,0.0141094,0.0141094,"
      "0.0141094,0.0141094,0.0141094,0.0141094,0.0141094,0.0141094,"
      "0.0141094,0.0141094,0.0141094,0.0141094,0.0141094,0.0141094,"
      "0.0141094,0.0141094,0.0141094,0.0129735,0.0129735,0.0129735,"
      "0.0129735,0.0129735,0.0129735,0.0129735,0.0129735,0.0129735,"
      "0.0129735,0.0129735,0.0129735,0.0129735,0.0129735,0.0129735,"
      "0.0129735,0.0129735,0.0129735,0.0129735,0.0129735,0.0129735,"
      "0.0129735,0.0129735,0.0129735,0.0129735,0.0129735,0.0129735,"
      "0.0129735,0.0129735,0.0129735,0.0129735,0.0129735,0.0129735,"
      "0.0129735,0.0129735,0.0129735,0.0129735,0.0129735,0.0129735,"
      "0.0129735,0.0129735,0.0129735,0.0129735,0.0129735,0.0129735,"
      "0.0129735,0.0129735,0.0129735,0.0139141,0.0139141,0.0139141,"
      "0.0139141,0.0139141,0.0139141,0.0139141,0.0139141,0.0139141,"
      "0.0139141,0.0139141,0.0139141,0.0139141,0.0139141,0.0139141,"
      "0.0139141,0.0139141,0.0139141,0.0139141,0.0139141,0.0139141,"
      "0.0139141,0.0139141,0.0139141,0.0139141,0.0139141,0.0139141,"
      "0.0139141,0.0139141,0.0139141,0.0139141,0.0139141,0.0139141,"
      "0.0139141,0.0139141,0.0139141,0.0139141,0.0139141,0.0139141,"
      "0.0139141,0.0139141,0.0139141,0.0139141,0.0139141,0.0139141,"
      "0.0139141,0.0139141,0.0139141,0.0140967,0.0140967,0.0140967,"
      "0.0140967,0.0140967,0.0140967,0.0140967,0.0140967,0.0140967,"
      "0.0140967,0.0140967,0.0140967,0.0140967,0.0140967,0.0140967,"
      "0.0140967,0.0140967,0.0140967,0.0140967,0.0140967,0.0140967,"
      "0.0140967,0.0140967,0.0140967,0.0140967,0.0140967,0.0140967,"
      "0.0140967,0.0140967,0.0140967,0.0140967,0.0140967,0.0140967,"
      "0.0140967,0.0140967,0.0140967,0.0140967,0.0140967,0.0140967,"
      "0.0140967,0.0140967,0.0140967,0.0140967,0.0140967,0.0140967,"
      "0.0140967,0.0140967,0.0140967";
  Eigen::VectorXd Ge_weight_ref = ReadVectorFromString(Ge_weight_ref_string);

  std::string H_phi_ref_string =
      "1.5708,1.5708,1.5708,1.5708,0,3.14159,0.955317,0.955317,"
      "0.955317,0.955317,2.18628,2.18628,2.18628,2.18628,1.47847,1.47847,"
      "1.47847,1.47847,1.66312,1.66312,1.66312,1.66312,0.789648,0.789648,"
      "0.789648,0.789648,2.35194,2.35194,2.35194,2.35194,0.789648,"
      "0.789648,0.789648,0.789648,2.35194,2.35194,2.35194,2.35194,1.29703,"
      "1.29703,1.29703,1.29703,1.84456,1.84456,1.84456,1.84456,0.821977,"
      "0.821977,0.821977,0.821977,2.31962,2.31962,2.31962,2.31962,"
      "0.821977,0.821977,0.821977,0.821977,2.31962,2.31962,2.31962,"
      "2.31962,1.12256,1.12256,1.12256,1.12256,2.01903,2.01903,2.01903,"
      "2.01903,0.879866,0.879866,0.879866,0.879866,2.26173,2.26173,"
      "2.26173,2.26173,0.879866,0.879866,0.879866,0.879866,2.26173,"
      "2.26173,2.26173,2.26173,0.794322,0.794322,0.794322,0.794322,"
      "2.34727,2.34727,2.34727,2.34727,1.04206,1.04206,1.04206,1.04206,"
      "2.09953,2.09953,2.09953,2.09953,1.04206,1.04206,1.04206,1.04206,"
      "2.09953,2.09953,2.09953,2.09953,0.638758,0.638758,0.638758,"
      "0.638758,2.50283,2.50283,2.50283,2.50283,1.13561,1.13561,1.13561,"
      "1.13561,2.00598,2.00598,2.00598,2.00598,1.13561,1.13561,1.13561,"
      "1.13561,2.00598,2.00598,2.00598,2.00598,0.488413,0.488413,0.488413,"
      "0.488413,2.65318,2.65318,2.65318,2.65318,1.23259,1.23259,1.23259,"
      "1.23259,1.909,1.909,1.909,1.909,1.23259,1.23259,1.23259,1.23259,"
      "1.909,1.909,1.909,1.909,0.343997,0.343997,0.343997,0.343997,"
      "2.7976,2.7976,2.7976,2.7976,1.33,1.33,1.33,1.33,1.81159,1.81159,"
      "1.81159,1.81159,1.33,1.33,1.33,1.33,1.81159,1.81159,1.81159,"
      "1.81159,0.207832,0.207832,0.207832,0.207832,2.93376,2.93376,"
      "2.93376,2.93376,1.42437,1.42437,1.42437,1.42437,1.71722,1.71722,"
      "1.71722,1.71722,1.42437,1.42437,1.42437,1.42437,1.71722,1.71722,"
      "1.71722,1.71722,0.0863039,0.0863039,0.0863039,0.0863039,3.05529,"
      "3.05529,3.05529,3.05529,1.50981,1.50981,1.50981,1.50981,1.63178,"
      "1.63178,1.63178,1.63178,1.50981,1.50981,1.50981,1.50981,1.63178,"
      "1.63178,1.63178,1.63178,1.5708,1.5708,1.5708,1.5708,1.5708,1.5708,"
      "1.5708,1.5708,0.658188,0.658188,2.4834,2.4834,0.912608,0.912608,"
      "2.22898,2.22898,0.658188,0.658188,2.4834,2.4834,0.912608,0.912608,"
      "2.22898,2.22898,1.5708,1.5708,1.5708,1.5708,1.5708,1.5708,1.5708,"
      "1.5708,0.407675,0.407675,2.73392,2.73392,1.16312,1.16312,1.97847,"
      "1.97847,0.407675,0.407675,2.73392,2.73392,1.16312,1.16312,1.97847,"
      "1.97847,1.5708,1.5708,1.5708,1.5708,1.5708,1.5708,1.5708,1.5708,"
      "0.173345,0.173345,2.96825,2.96825,1.39745,1.39745,1.74414,1.74414,"
      "0.173345,0.173345,2.96825,2.96825,1.39745,1.39745,1.74414,1.74414,"
      "0.723776,0.723776,0.723776,0.723776,2.41782,2.41782,2.41782,"
      "2.41782,1.21127,1.21127,1.21127,1.21127,1.93032,1.93032,1.93032,"
      "1.93032,0.723776,0.723776,0.723776,0.723776,2.41782,2.41782,"
      "2.41782,2.41782,0.975171,0.975171,0.975171,0.975171,2.16642,"
      "2.16642,2.16642,2.16642,1.21127,1.21127,1.21127,1.21127,1.93032,"
      "1.93032,1.93032,1.93032,0.975171,0.975171,0.975171,0.975171,"
      "2.16642,2.16642,2.16642,2.16642,0.573426,0.573426,0.573426,"
      "0.573426,2.56817,2.56817,2.56817,2.56817,1.30418,1.30418,1.30418,"
      "1.30418,1.83742,1.83742,1.83742,1.83742,0.573426,0.573426,0.573426,"
      "0.573426,2.56817,2.56817,2.56817,2.56817,1.0767,1.0767,1.0767,"
      "1.0767,2.0649,2.0649,2.0649,2.0649,1.30418,1.30418,1.30418,"
      "1.30418,1.83742,1.83742,1.83742,1.83742,1.0767,1.0767,1.0767,"
      "1.0767,2.0649,2.0649,2.0649,2.0649,0.675618,0.675618,0.675618,"
      "0.675618,2.46597,2.46597,2.46597,2.46597,1.38812,1.38812,1.38812,"
      "1.38812,1.75347,1.75347,1.75347,1.75347,0.675618,0.675618,0.675618,"
      "0.675618,2.46597,2.46597,2.46597,2.46597,0.929278,0.929278,"
      "0.929278,0.929278,2.21231,2.21231,2.21231,2.21231,1.38812,1.38812,"
      "1.38812,1.38812,1.75347,1.75347,1.75347,1.75347,0.929278,0.929278,"
      "0.929278,0.929278,2.21231,2.21231,2.21231,2.21231,0.429405,"
      "0.429405,0.429405,0.429405,2.71219,2.71219,2.71219,2.71219,1.39786,"
      "1.39786,1.39786,1.39786,1.74374,1.74374,1.74374,1.74374,0.429405,"
      "0.429405,0.429405,0.429405,2.71219,2.71219,2.71219,2.71219,1.18197,"
      "1.18197,1.18197,1.18197,1.95962,1.95962,1.95962,1.95962,1.39786,"
      "1.39786,1.39786,1.39786,1.74374,1.74374,1.74374,1.74374,1.18197,"
      "1.18197,1.18197,1.18197,1.95962,1.95962,1.95962,1.95962,0.293967,"
      "0.293967,0.293967,0.293967,2.84763,2.84763,2.84763,2.84763,1.48857,"
      "1.48857,1.48857,1.48857,1.65302,1.65302,1.65302,1.65302,0.293967,"
      "0.293967,0.293967,0.293967,2.84763,2.84763,2.84763,2.84763,1.28922,"
      "1.28922,1.28922,1.28922,1.85237,1.85237,1.85237,1.85237,1.48857,"
      "1.48857,1.48857,1.48857,1.65302,1.65302,1.65302,1.65302,1.28922,"
      "1.28922,1.28922,1.28922,1.85237,1.85237,1.85237,1.85237,0.536741,"
      "0.536741,0.536741,0.536741,2.60485,2.60485,2.60485,2.60485,1.48068,"
      "1.48068,1.48068,1.48068,1.66091,1.66091,1.66091,1.66091,0.536741,"
      "0.536741,0.536741,0.536741,2.60485,2.60485,2.60485,2.60485,1.04332,"
      "1.04332,1.04332,1.04332,2.09828,2.09828,2.09828,2.09828,1.48068,"
      "1.48068,1.48068,1.48068,1.66091,1.66091,1.66091,1.66091,1.04332,"
      "1.04332,1.04332,1.04332,2.09828,2.09828,2.09828,2.09828";
  Eigen::VectorXd H_phi_ref = ReadVectorFromString(H_phi_ref_string);
  std::string H_theta_ref_string =
      "0,3.14159,1.5708,-1.5708,0,0,0.785398,2.35619,"
      "-0.785398,-2.35619,0.785398,2.35619,-0.785398,-2.35619,0.785398,"
      "2.35619,-0.785398,-2.35619,0.785398,2.35619,-0.785398,-2.35619,"
      "0.130194,3.0114,-0.130194,-3.0114,0.130194,3.0114,-0.130194,"
      "-3.0114,1.4406,1.70099,-1.4406,-1.70099,1.4406,1.70099,-1.4406,"
      "-1.70099,0.785398,2.35619,-0.785398,-2.35619,0.785398,2.35619,"
      "-0.785398,-2.35619,0.37803,2.76356,-0.37803,-2.76356,0.37803,"
      "2.76356,-0.37803,-2.76356,1.19277,1.94883,-1.19277,-1.94883,"
      "1.19277,1.94883,-1.19277,-1.94883,0.785398,2.35619,-0.785398,"
      "-2.35619,0.785398,2.35619,-0.785398,-2.35619,0.59722,2.54437,"
      "-0.59722,-2.54437,0.59722,2.54437,-0.59722,-2.54437,0.973576,"
      "2.16802,-0.973576,-2.16802,0.973576,2.16802,-0.973576,-2.16802,"
      "0.785398,2.35619,-0.785398,-2.35619,0.785398,2.35619,-0.785398,"
      "-2.35619,0.946878,2.19471,-0.946878,-2.19471,0.946878,2.19471,"
      "-0.946878,-2.19471,0.623918,2.51767,-0.623918,-2.51767,0.623918,"
      "2.51767,-0.623918,-2.51767,0.785398,2.35619,-0.785398,-2.35619,"
      "0.785398,2.35619,-0.785398,-2.35619,1.08726,2.05433,-1.08726,"
      "-2.05433,1.08726,2.05433,-1.08726,-2.05433,0.483532,2.65806,"
      "-0.483532,-2.65806,0.483532,2.65806,-0.483532,-2.65806,0.785398,"
      "2.35619,-0.785398,-2.35619,0.785398,2.35619,-0.785398,-2.35619,"
      "1.21139,1.9302,-1.21139,-1.9302,1.21139,1.9302,-1.21139,-1.9302,"
      "0.359403,2.78219,-0.359403,-2.78219,0.359403,2.78219,-0.359403,"
      "-2.78219,0.785398,2.35619,-0.785398,-2.35619,0.785398,2.35619,"
      "-0.785398,-2.35619,1.3227,1.81889,-1.3227,-1.81889,1.3227,1.81889,"
      "-1.3227,-1.81889,0.248096,2.8935,-0.248096,-2.8935,0.248096,2.8935,"
      "-0.248096,-2.8935,0.785398,2.35619,-0.785398,-2.35619,0.785398,"
      "2.35619,-0.785398,-2.35619,1.42277,1.71882,-1.42277,-1.71882,"
      "1.42277,1.71882,-1.42277,-1.71882,0.148022,2.99357,-0.148022,"
      "-2.99357,0.148022,2.99357,-0.148022,-2.99357,0.785398,2.35619,"
      "-0.785398,-2.35619,0.785398,2.35619,-0.785398,-2.35619,1.50969,"
      "1.6319,-1.50969,-1.6319,1.50969,1.6319,-1.50969,-1.6319,0.0611019,"
      "3.08049,-0.0611019,-3.08049,0.0611019,3.08049,-0.0611019,-3.08049,"
      "0.912608,2.22898,-0.912608,-2.22898,0.658188,2.4834,-0.658188,"
      "-2.4834,0,3.14159,0,3.14159,0,3.14159,0,3.14159,1.5708,-1.5708,"
      "1.5708,-1.5708,1.5708,-1.5708,1.5708,-1.5708,1.16312,1.97847,"
      "-1.16312,-1.97847,0.407675,2.73392,-0.407675,-2.73392,0,3.14159,0,"
      "3.14159,0,3.14159,0,3.14159,1.5708,-1.5708,1.5708,-1.5708,1.5708,"
      "-1.5708,1.5708,-1.5708,1.39745,1.74414,-1.39745,-1.74414,0.173345,"
      "2.96825,-0.173345,-2.96825,0,3.14159,0,3.14159,0,3.14159,0,"
      "3.14159,1.5708,-1.5708,1.5708,-1.5708,1.5708,-1.5708,1.5708,"
      "-1.5708,0.560119,2.58147,-0.560119,-2.58147,0.560119,2.58147,"
      "-0.560119,-2.58147,0.928113,2.21348,-0.928113,-2.21348,0.928113,"
      "2.21348,-0.928113,-2.21348,1.01068,2.13092,-1.01068,-2.13092,"
      "1.01068,2.13092,-1.01068,-2.13092,1.13182,2.00978,-1.13182,"
      "-2.00978,1.13182,2.00978,-1.13182,-2.00978,0.642684,2.49891,"
      "-0.642684,-2.49891,0.642684,2.49891,-0.642684,-2.49891,0.438981,"
      "2.70261,-0.438981,-2.70261,0.438981,2.70261,-0.438981,-2.70261,"
      "0.507107,2.63449,-0.507107,-2.63449,0.507107,2.63449,-0.507107,"
      "-2.63449,1.05686,2.08473,-1.05686,-2.08473,1.05686,2.08473,"
      "-1.05686,-2.08473,1.06369,2.0779,-1.06369,-2.0779,1.06369,2.0779,"
      "-1.06369,-2.0779,1.26687,1.87472,-1.26687,-1.87472,1.26687,1.87472,"
      "-1.26687,-1.87472,0.513937,2.62766,-0.513937,-2.62766,0.513937,"
      "2.62766,-0.513937,-2.62766,0.303922,2.83767,-0.303922,-2.83767,"
      "0.303922,2.83767,-0.303922,-2.83767,0.294735,2.84686,-0.294735,"
      "-2.84686,0.294735,2.84686,-0.294735,-2.84686,0.916579,2.22501,"
      "-0.916579,-2.22501,0.916579,2.22501,-0.916579,-2.22501,1.27606,"
      "1.86553,-1.27606,-1.86553,1.27606,1.86553,-1.27606,-1.86553,"
      "1.34206,1.79953,-1.34206,-1.79953,1.34206,1.79953,-1.34206,"
      "-1.79953,0.654217,2.48738,-0.654217,-2.48738,0.654217,2.48738,"
      "-0.654217,-2.48738,0.228733,2.91286,-0.228733,-2.91286,0.228733,"
      "2.91286,-0.228733,-2.91286,0.426102,2.71549,-0.426102,-2.71549,"
      "0.426102,2.71549,-0.426102,-2.71549,1.17576,1.96584,-1.17576,"
      "-1.96584,1.17576,1.96584,-1.17576,-1.96584,1.14469,1.9969,-1.14469,"
      "-1.9969,1.14469,1.9969,-1.14469,-1.9969,1.38375,1.75785,-1.38375,"
      "-1.75785,1.38375,1.75785,-1.38375,-1.75785,0.395039,2.74655,"
      "-0.395039,-2.74655,0.395039,2.74655,-0.395039,-2.74655,0.18705,"
      "2.95454,-0.18705,-2.95454,0.18705,2.95454,-0.18705,-2.95454,"
      "0.287391,2.8542,-0.287391,-2.8542,0.287391,2.8542,-0.287391,"
      "-2.8542,1.28824,1.85335,-1.28824,-1.85335,1.28824,1.85335,-1.28824,"
      "-1.85335,1.28341,1.85819,-1.28341,-1.85819,1.28341,1.85819,"
      "-1.28341,-1.85819,1.48519,1.6564,-1.48519,-1.6564,1.48519,1.6564,"
      "-1.48519,-1.6564,0.282554,2.85904,-0.282554,-2.85904,0.282554,"
      "2.85904,-0.282554,-2.85904,0.0856016,3.05599,-0.0856016,-3.05599,"
      "0.0856016,3.05599,-0.0856016,-3.05599,0.176915,2.96468,-0.176915,"
      "-2.96468,0.176915,2.96468,-0.176915,-2.96468,1.04094,2.10065,"
      "-1.04094,-2.10065,1.04094,2.10065,-1.04094,-2.10065,1.39388,"
      "1.74771,-1.39388,-1.74771,1.39388,1.74771,-1.39388,-1.74771,"
      "1.46646,1.67513,-1.46646,-1.67513,1.46646,1.67513,-1.46646,"
      "-1.67513,0.529854,2.61174,-0.529854,-2.61174,0.529854,2.61174,"
      "-0.529854,-2.61174,0.104337,3.03726,-0.104337,-3.03726,0.104337,"
      "3.03726,-0.104337,-3.03726";
  Eigen::VectorXd H_theta_ref = ReadVectorFromString(H_theta_ref_string);
  std::string H_weight_ref_string =
      "0.00388944,0.00388944,0.00388944,0.00388944,0.00388944,"
      "0.00388944,0.0232777,0.0232777,0.0232777,0.0232777,0.0232777,"
      "0.0232777,0.0232777,0.0232777,0.0235216,0.0235216,0.0235216,"
      "0.0235216,0.0235216,0.0235216,0.0235216,0.0235216,0.0235216,"
      "0.0235216,0.0235216,0.0235216,0.0235216,0.0235216,0.0235216,"
      "0.0235216,0.0235216,0.0235216,0.0235216,0.0235216,0.0235216,"
      "0.0235216,0.0235216,0.0235216,0.0233585,0.0233585,0.0233585,"
      "0.0233585,0.0233585,0.0233585,0.0233585,0.0233585,0.0233585,"
      "0.0233585,0.0233585,0.0233585,0.0233585,0.0233585,0.0233585,"
      "0.0233585,0.0233585,0.0233585,0.0233585,0.0233585,0.0233585,"
      "0.0233585,0.0233585,0.0233585,0.0232733,0.0232733,0.0232733,"
      "0.0232733,0.0232733,0.0232733,0.0232733,0.0232733,0.0232733,"
      "0.0232733,0.0232733,0.0232733,0.0232733,0.0232733,0.0232733,"
      "0.0232733,0.0232733,0.0232733,0.0232733,0.0232733,0.0232733,"
      "0.0232733,0.0232733,0.0232733,0.0232065,0.0232065,0.0232065,"
      "0.0232065,0.0232065,0.0232065,0.0232065,0.0232065,0.0232065,"
      "0.0232065,0.0232065,0.0232065,0.0232065,0.0232065,0.0232065,"
      "0.0232065,0.0232065,0.0232065,0.0232065,0.0232065,0.0232065,"
      "0.0232065,0.0232065,0.0232065,0.0228516,0.0228516,0.0228516,"
      "0.0228516,0.0228516,0.0228516,0.0228516,0.0228516,0.0228516,"
      "0.0228516,0.0228516,0.0228516,0.0228516,0.0228516,0.0228516,"
      "0.0228516,0.0228516,0.0228516,0.0228516,0.0228516,0.0228516,"
      "0.0228516,0.0228516,0.0228516,0.0219857,0.0219857,0.0219857,"
      "0.0219857,0.0219857,0.0219857,0.0219857,0.0219857,0.0219857,"
      "0.0219857,0.0219857,0.0219857,0.0219857,0.0219857,0.0219857,"
      "0.0219857,0.0219857,0.0219857,0.0219857,0.0219857,0.0219857,"
      "0.0219857,0.0219857,0.0219857,0.0203225,0.0203225,0.0203225,"
      "0.0203225,0.0203225,0.0203225,0.0203225,0.0203225,0.0203225,"
      "0.0203225,0.0203225,0.0203225,0.0203225,0.0203225,0.0203225,"
      "0.0203225,0.0203225,0.0203225,0.0203225,0.0203225,0.0203225,"
      "0.0203225,0.0203225,0.0203225,0.0174011,0.0174011,0.0174011,"
      "0.0174011,0.0174011,0.0174011,0.0174011,0.0174011,0.0174011,"
      "0.0174011,0.0174011,0.0174011,0.0174011,0.0174011,0.0174011,"
      "0.0174011,0.0174011,0.0174011,0.0174011,0.0174011,0.0174011,"
      "0.0174011,0.0174011,0.0174011,0.0122702,0.0122702,0.0122702,"
      "0.0122702,0.0122702,0.0122702,0.0122702,0.0122702,0.0122702,"
      "0.0122702,0.0122702,0.0122702,0.0122702,0.0122702,0.0122702,"
      "0.0122702,0.0122702,0.0122702,0.0122702,0.0122702,0.0122702,"
      "0.0122702,0.0122702,0.0122702,0.0233378,0.0233378,0.0233378,"
      "0.0233378,0.0233378,0.0233378,0.0233378,0.0233378,0.0233378,"
      "0.0233378,0.0233378,0.0233378,0.0233378,0.0233378,0.0233378,"
      "0.0233378,0.0233378,0.0233378,0.0233378,0.0233378,0.0233378,"
      "0.0233378,0.0233378,0.0233378,0.0214276,0.0214276,0.0214276,"
      "0.0214276,0.0214276,0.0214276,0.0214276,0.0214276,0.0214276,"
      "0.0214276,0.0214276,0.0214276,0.0214276,0.0214276,0.0214276,"
      "0.0214276,0.0214276,0.0214276,0.0214276,0.0214276,0.0214276,"
      "0.0214276,0.0214276,0.0214276,0.0163403,0.0163403,0.0163403,"
      "0.0163403,0.0163403,0.0163403,0.0163403,0.0163403,0.0163403,"
      "0.0163403,0.0163403,0.0163403,0.0163403,0.0163403,0.0163403,"
      "0.0163403,0.0163403,0.0163403,0.0163403,0.0163403,0.0163403,"
      "0.0163403,0.0163403,0.0163403,0.0231581,0.0231581,0.0231581,"
      "0.0231581,0.0231581,0.0231581,0.0231581,0.0231581,0.0231581,"
      "0.0231581,0.0231581,0.0231581,0.0231581,0.0231581,0.0231581,"
      "0.0231581,0.0231581,0.0231581,0.0231581,0.0231581,0.0231581,"
      "0.0231581,0.0231581,0.0231581,0.0231581,0.0231581,0.0231581,"
      "0.0231581,0.0231581,0.0231581,0.0231581,0.0231581,0.0231581,"
      "0.0231581,0.0231581,0.0231581,0.0231581,0.0231581,0.0231581,"
      "0.0231581,0.0231581,0.0231581,0.0231581,0.0231581,0.0231581,"
      "0.0231581,0.0231581,0.0231581,0.0226529,0.0226529,0.0226529,"
      "0.0226529,0.0226529,0.0226529,0.0226529,0.0226529,0.0226529,"
      "0.0226529,0.0226529,0.0226529,0.0226529,0.0226529,0.0226529,"
      "0.0226529,0.0226529,0.0226529,0.0226529,0.0226529,0.0226529,"
      "0.0226529,0.0226529,0.0226529,0.0226529,0.0226529,0.0226529,"
      "0.0226529,0.0226529,0.0226529,0.0226529,0.0226529,0.0226529,"
      "0.0226529,0.0226529,0.0226529,0.0226529,0.0226529,0.0226529,"
      "0.0226529,0.0226529,0.0226529,0.0226529,0.0226529,0.0226529,"
      "0.0226529,0.0226529,0.0226529,0.0232457,0.0232457,0.0232457,"
      "0.0232457,0.0232457,0.0232457,0.0232457,0.0232457,0.0232457,"
      "0.0232457,0.0232457,0.0232457,0.0232457,0.0232457,0.0232457,"
      "0.0232457,0.0232457,0.0232457,0.0232457,0.0232457,0.0232457,"
      "0.0232457,0.0232457,0.0232457,0.0232457,0.0232457,0.0232457,"
      "0.0232457,0.0232457,0.0232457,0.0232457,0.0232457,0.0232457,"
      "0.0232457,0.0232457,0.0232457,0.0232457,0.0232457,0.0232457,"
      "0.0232457,0.0232457,0.0232457,0.0232457,0.0232457,0.0232457,"
      "0.0232457,0.0232457,0.0232457,0.0215376,0.0215376,0.0215376,"
      "0.0215376,0.0215376,0.0215376,0.0215376,0.0215376,0.0215376,"
      "0.0215376,0.0215376,0.0215376,0.0215376,0.0215376,0.0215376,"
      "0.0215376,0.0215376,0.0215376,0.0215376,0.0215376,0.0215376,"
      "0.0215376,0.0215376,0.0215376,0.0215376,0.0215376,0.0215376,"
      "0.0215376,0.0215376,0.0215376,0.0215376,0.0215376,0.0215376,"
      "0.0215376,0.0215376,0.0215376,0.0215376,0.0215376,0.0215376,"
      "0.0215376,0.0215376,0.0215376,0.0215376,0.0215376,0.0215376,"
      "0.0215376,0.0215376,0.0215376,0.0195434,0.0195434,0.0195434,"
      "0.0195434,0.0195434,0.0195434,0.0195434,0.0195434,0.0195434,"
      "0.0195434,0.0195434,0.0195434,0.0195434,0.0195434,0.0195434,"
      "0.0195434,0.0195434,0.0195434,0.0195434,0.0195434,0.0195434,"
      "0.0195434,0.0195434,0.0195434,0.0195434,0.0195434,0.0195434,"
      "0.0195434,0.0195434,0.0195434,0.0195434,0.0195434,0.0195434,"
      "0.0195434,0.0195434,0.0195434,0.0195434,0.0195434,0.0195434,"
      "0.0195434,0.0195434,0.0195434,0.0195434,0.0195434,0.0195434,"
      "0.0195434,0.0195434,0.0195434,0.0226476,0.0226476,0.0226476,"
      "0.0226476,0.0226476,0.0226476,0.0226476,0.0226476,0.0226476,"
      "0.0226476,0.0226476,0.0226476,0.0226476,0.0226476,0.0226476,"
      "0.0226476,0.0226476,0.0226476,0.0226476,0.0226476,0.0226476,"
      "0.0226476,0.0226476,0.0226476,0.0226476,0.0226476,0.0226476,"
      "0.0226476,0.0226476,0.0226476,0.0226476,0.0226476,0.0226476,"
      "0.0226476,0.0226476,0.0226476,0.0226476,0.0226476,0.0226476,"
      "0.0226476,0.0226476,0.0226476,0.0226476,0.0226476,0.0226476,"
      "0.0226476,0.0226476,0.0226476";
  Eigen::VectorXd H_weight_ref = ReadVectorFromString(H_weight_ref_string);

  BOOST_CHECK_EQUAL(Gegrid.phi.size(), Ge_phi_ref.size());
  BOOST_CHECK_EQUAL(Gegrid.theta.size(), Ge_theta_ref.size());
  BOOST_CHECK_EQUAL(Gegrid.weight.size(), Ge_weight_ref.size());
  BOOST_CHECK_EQUAL(Hgrid.phi.size(), H_phi_ref.size());
  BOOST_CHECK_EQUAL(Hgrid.theta.size(), H_theta_ref.size());
  BOOST_CHECK_EQUAL(Hgrid.weight.size(), H_weight_ref.size());

  bool Gephi = Ge_phi_ref.isApprox(Gegrid.phi, 0.001);
  bool Getheta = Ge_theta_ref.isApprox(Gegrid.theta, 0.001);
  if (!Gephi || !Getheta) {
    std::cout << "phi_ref : Phi_comp | theta_ref : theta_comp" << std::endl;
    for (Index i = 0; i < Ge_phi_ref.size(); i++) {
      std::cout << Gegrid.phi[i] << ":" << Ge_phi_ref[i] << " | "
                << Gegrid.theta[i] << ":" << Ge_theta_ref[i] << std::endl;
    }
  }

  bool Geweight = Ge_weight_ref.isApprox(Gegrid.weight, 0.0001);
  BOOST_CHECK_EQUAL(Gephi, true);
  BOOST_CHECK_EQUAL(Getheta, true);
  BOOST_CHECK_EQUAL(Geweight, true);

  bool Hphi = H_phi_ref.isApprox(Hgrid.phi, 0.001);
  bool Htheta = H_theta_ref.isApprox(Hgrid.theta, 0.001);
  bool Hweight = H_weight_ref.isApprox(Hgrid.weight, 0.0001);
  BOOST_CHECK_EQUAL(Hphi, true);
  BOOST_CHECK_EQUAL(Htheta, true);
  BOOST_CHECK_EQUAL(Hweight, true);
}

BOOST_AUTO_TEST_SUITE_END()
