/*
 * Decompiled with CFR 0.152.
 */
package org.openhab.core.automation.module.script.rulesupport.loader;

import java.io.File;
import java.io.IOException;
import java.io.InputStreamReader;
import java.nio.charset.StandardCharsets;
import java.nio.file.FileVisitOption;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.attribute.FileAttribute;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.eclipse.jdt.annotation.NonNullByDefault;
import org.eclipse.jdt.annotation.Nullable;
import org.openhab.core.automation.module.script.ScriptDependencyTracker;
import org.openhab.core.automation.module.script.ScriptEngineContainer;
import org.openhab.core.automation.module.script.ScriptEngineManager;
import org.openhab.core.automation.module.script.rulesupport.internal.loader.ScriptFileReference;
import org.openhab.core.automation.module.script.rulesupport.loader.ScriptFileWatcher;
import org.openhab.core.common.NamedThreadFactory;
import org.openhab.core.service.ReadyMarker;
import org.openhab.core.service.ReadyMarkerFilter;
import org.openhab.core.service.ReadyService;
import org.openhab.core.service.StartLevelService;
import org.openhab.core.service.WatchService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@NonNullByDefault
public abstract class AbstractScriptFileWatcher
implements WatchService.WatchEventListener,
ReadyService.ReadyTracker,
ScriptDependencyTracker.Listener,
ScriptEngineManager.FactoryChangeListener,
ScriptFileWatcher {
    private static final Set<String> EXCLUDED_FILE_EXTENSIONS = Set.of("txt", "old", "example", "backup", "md", "swp", "tmp", "bak");
    private static final String REGEX_SEPARATOR = "\\".equals(File.separator) ? "\\\\" : File.separator;
    private static final List<Pattern> START_LEVEL_PATTERNS = List.of(Pattern.compile(".*" + REGEX_SEPARATOR + "sl(\\d{2})" + REGEX_SEPARATOR + "[^" + REGEX_SEPARATOR + "]+"), Pattern.compile(".*" + REGEX_SEPARATOR + "[^" + REGEX_SEPARATOR + "]+\\.sl(\\d{2})\\.[^" + REGEX_SEPARATOR + ".]+"));
    private final Logger logger = LoggerFactory.getLogger(AbstractScriptFileWatcher.class);
    private final ScriptEngineManager manager;
    private final ReadyService readyService;
    private final WatchService watchService;
    private final Path watchPath;
    private final boolean watchSubDirectories;
    protected ScheduledExecutorService scheduler;
    private final Map<String, ScriptFileReference> scriptMap = new ConcurrentHashMap<String, ScriptFileReference>();
    private final Map<String, Lock> scriptLockMap = new ConcurrentHashMap<String, Lock>();
    private final CompletableFuture<@Nullable Void> initialized = new CompletableFuture();
    private volatile int currentStartLevel;

    protected AbstractScriptFileWatcher(WatchService watchService, ScriptEngineManager manager, ReadyService readyService, StartLevelService startLevelService, String fileDirectory, boolean watchSubDirectories) {
        this.watchService = watchService;
        this.manager = manager;
        this.readyService = readyService;
        this.watchSubDirectories = watchSubDirectories;
        this.watchPath = watchService.getWatchPath().resolve(fileDirectory);
        this.scheduler = this.getScheduler();
        this.currentStartLevel = 10;
    }

    protected Path getWatchPath() {
        return this.watchPath;
    }

    protected ScheduledExecutorService getScheduler() {
        return Executors.newSingleThreadScheduledExecutor((ThreadFactory)new NamedThreadFactory("scriptwatcher"));
    }

    public void activate() {
        if (!Files.exists(this.watchPath, new LinkOption[0])) {
            try {
                Files.createDirectories(this.watchPath, new FileAttribute[0]);
            }
            catch (IOException e) {
                this.logger.warn("Failed to create watched directory: {}", (Object)this.watchPath);
            }
        } else if (!Files.isDirectory(this.watchPath, new LinkOption[0])) {
            this.logger.warn("Trying to watch directory {}, however it is a file", (Object)this.watchPath);
        }
        this.manager.addFactoryChangeListener((ScriptEngineManager.FactoryChangeListener)this);
        this.readyService.registerTracker((ReadyService.ReadyTracker)this, new ReadyMarkerFilter().withType("startlevel"));
        this.watchService.registerListener((WatchService.WatchEventListener)this, this.watchPath, this.watchSubDirectories);
    }

    public void deactivate() {
        this.watchService.unregisterListener((WatchService.WatchEventListener)this);
        this.manager.removeFactoryChangeListener((ScriptEngineManager.FactoryChangeListener)this);
        this.readyService.unregisterTracker((ReadyService.ReadyTracker)this);
        CompletableFuture.allOf((CompletableFuture[])Set.copyOf(this.scriptMap.keySet()).stream().map(this::removeFile).toArray(CompletableFuture[]::new)).thenRun(this.scheduler::shutdownNow);
    }

    @Override
    public CompletableFuture<@Nullable Void> ifInitialized() {
        return this.initialized;
    }

    protected Optional<String> getScriptType(Path scriptFilePath) {
        String fileName = scriptFilePath.toString();
        int index = fileName.lastIndexOf(".");
        if (index == -1) {
            return Optional.empty();
        }
        String fileExtension = fileName.substring(index + 1);
        if (EXCLUDED_FILE_EXTENSIONS.contains(fileExtension) || fileExtension.endsWith("~")) {
            return Optional.empty();
        }
        return Optional.of(fileExtension);
    }

    protected int getStartLevel(Path scriptFilePath) {
        for (Pattern p : START_LEVEL_PATTERNS) {
            Matcher m = p.matcher(scriptFilePath.toString());
            if (!m.find() || m.groupCount() <= 0) continue;
            try {
                return Integer.parseInt(m.group(1));
            }
            catch (NumberFormatException nfe) {
                this.logger.warn("Extracted start level {} from {}, but it's not an integer. Ignoring.", (Object)m.group(1), (Object)scriptFilePath);
            }
        }
        return 50;
    }

    private List<Path> listFiles(Path path, boolean includeSubDirectory) {
        try {
            Throwable throwable = null;
            Object var4_6 = null;
            try (Stream<Path> stream = Files.walk(path, includeSubDirectory ? Integer.MAX_VALUE : 1, new FileVisitOption[0]);){
                return stream.filter(file -> !Files.isDirectory(file, new LinkOption[0])).toList();
            }
            catch (Throwable throwable2) {
                if (throwable == null) {
                    throwable = throwable2;
                } else if (throwable != throwable2) {
                    throwable.addSuppressed(throwable2);
                }
                throw throwable;
            }
        }
        catch (IOException iOException) {
            return List.of();
        }
    }

    public void processWatchEvent(WatchService.Kind kind, Path fullPath) {
        if (!this.initialized.isDone()) {
            return;
        }
        File file = fullPath.toFile();
        if (kind == WatchService.Kind.DELETE) {
            String scriptIdentifier = ScriptFileReference.getScriptIdentifier(fullPath);
            if (this.scriptMap.containsKey(scriptIdentifier)) {
                this.removeFile(scriptIdentifier);
            }
        } else if (!file.isHidden() && file.canRead() && (kind == WatchService.Kind.CREATE || kind == WatchService.Kind.MODIFY)) {
            this.addFiles(this.listFiles(fullPath, this.watchSubDirectories));
        }
    }

    private CompletableFuture<Void> addFiles(Collection<Path> files) {
        return CompletableFuture.allOf((CompletableFuture[])files.stream().map(this::getScriptFileReference).flatMap(Optional::stream).sorted().map(this::addScriptFileReference).toArray(CompletableFuture[]::new));
    }

    private CompletableFuture<Void> addScriptFileReference(ScriptFileReference newRef) {
        ScriptFileReference ref = this.scriptMap.computeIfAbsent(newRef.getScriptIdentifier(), k -> newRef);
        if (this.currentStartLevel >= ref.getStartLevel() && !ref.getQueueStatus().getAndSet(true)) {
            return this.importFileWhenReady(ref.getScriptIdentifier());
        }
        return CompletableFuture.completedFuture(null);
    }

    private Optional<ScriptFileReference> getScriptFileReference(Path path) {
        return this.getScriptType(path).map(scriptType -> new ScriptFileReference(path, (String)scriptType, this.getStartLevel(path)));
    }

    private CompletableFuture<Void> removeFile(String scriptIdentifier) {
        return CompletableFuture.runAsync(() -> {
            Lock lock = this.getLockForScript(scriptIdentifier);
            try {
                this.scriptMap.computeIfPresent(scriptIdentifier, (id, ref) -> {
                    if (ref.getLoadedStatus().get()) {
                        this.manager.removeEngine(scriptIdentifier);
                        this.logger.debug("Unloaded script '{}'", (Object)scriptIdentifier);
                    } else {
                        this.logger.debug("Dequeued script '{}'", (Object)scriptIdentifier);
                    }
                    return null;
                });
            }
            finally {
                if (this.scriptMap.containsKey(scriptIdentifier)) {
                    this.logger.warn("Failed to unload script '{}'", (Object)scriptIdentifier);
                }
                this.scriptLockMap.remove(scriptIdentifier);
                lock.unlock();
            }
        }, this.scheduler);
    }

    private synchronized Lock getLockForScript(String scriptIdentifier) {
        Lock lock = this.scriptLockMap.computeIfAbsent(scriptIdentifier, k -> new ReentrantLock());
        lock.lock();
        return lock;
    }

    private CompletableFuture<Void> importFileWhenReady(String scriptIdentifier) {
        return CompletableFuture.runAsync(() -> {
            ScriptFileReference ref = this.scriptMap.get(scriptIdentifier);
            if (ref == null) {
                this.logger.warn("Failed to import script '{}': script reference not found", (Object)scriptIdentifier);
                return;
            }
            ref.getQueueStatus().set(false);
            Lock lock = this.getLockForScript(scriptIdentifier);
            try {
                if (ref.getLoadedStatus().get()) {
                    this.manager.removeEngine(scriptIdentifier);
                    this.logger.debug("Unloaded script '{}'", (Object)scriptIdentifier);
                }
                if (this.manager.isSupported(ref.getScriptType()) && ref.getStartLevel() <= this.currentStartLevel) {
                    this.logger.info("(Re-)Loading script '{}'", (Object)scriptIdentifier);
                    if (this.createAndLoad(ref)) {
                        ref.getLoadedStatus().set(true);
                    } else {
                        this.manager.removeEngine(ref.getScriptIdentifier());
                        ref.getLoadedStatus().set(false);
                    }
                } else {
                    ref.getLoadedStatus().set(false);
                    this.logger.debug("Enqueued script '{}'", (Object)ref.getScriptIdentifier());
                }
            }
            finally {
                lock.unlock();
            }
        }, this.scheduler);
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private boolean createAndLoad(ScriptFileReference ref) {
        String scriptIdentifier = ref.getScriptIdentifier();
        try {
            Throwable throwable = null;
            Object var4_6 = null;
            try (InputStreamReader reader = new InputStreamReader(Files.newInputStream(ref.getScriptFilePath(), new OpenOption[0]), StandardCharsets.UTF_8);){
                ScriptEngineContainer container = this.manager.createScriptEngine(ref.getScriptType(), ref.getScriptIdentifier());
                if (container != null) {
                    container.getScriptEngine().put("javax.script.filename", scriptIdentifier);
                    if (this.manager.loadScript(container.getIdentifier(), reader)) {
                        return true;
                    }
                }
                this.logger.warn("Script loading error, ignoring file '{}'", (Object)scriptIdentifier);
                return false;
            }
            catch (Throwable throwable2) {
                if (throwable == null) {
                    throwable = throwable2;
                    throw throwable;
                }
                if (throwable == throwable2) throw throwable;
                throwable.addSuppressed(throwable2);
                throw throwable;
            }
        }
        catch (IOException e) {
            this.logger.warn("Failed to load file '{}': {}", (Object)ref.getScriptFilePath(), (Object)e.getMessage());
        }
        return false;
    }

    public void onDependencyChange(String scriptIdentifier) {
        this.logger.debug("Reimporting {}...", (Object)scriptIdentifier);
        ScriptFileReference ref = this.scriptMap.get(scriptIdentifier);
        if (ref != null && !ref.getQueueStatus().getAndSet(true)) {
            this.importFileWhenReady(scriptIdentifier);
        }
    }

    public synchronized void onReadyMarkerAdded(ReadyMarker readyMarker) {
        int previousLevel = this.currentStartLevel;
        this.currentStartLevel = Integer.parseInt(readyMarker.getIdentifier());
        this.logger.trace("Added ready marker {}: start level changed from {} to {}. watchPath: {}", new Object[]{readyMarker, previousLevel, this.currentStartLevel, this.watchPath});
        if (this.currentStartLevel < 30) {
            return;
        }
        if (!this.initialized.isDone()) {
            this.addFiles(this.listFiles(this.watchPath, this.watchSubDirectories)).thenRun(() -> {
                boolean bl = this.initialized.complete(null);
            });
        } else {
            this.scriptMap.values().stream().sorted().filter(ref -> this.needsStartLevelProcessing((ScriptFileReference)ref, previousLevel, this.currentStartLevel)).forEach(ref -> {
                CompletableFuture<Void> completableFuture = this.importFileWhenReady(ref.getScriptIdentifier());
            });
        }
    }

    private boolean needsStartLevelProcessing(ScriptFileReference ref, int previousLevel, int newLevel) {
        int refStartLevel = ref.getStartLevel();
        return !ref.getLoadedStatus().get() && newLevel >= refStartLevel && previousLevel < refStartLevel && !ref.getQueueStatus().getAndSet(true);
    }

    public void onReadyMarkerRemoved(ReadyMarker readyMarker) {
    }

    public void factoryAdded(@Nullable String scriptType) {
        this.scriptMap.forEach((scriptIdentifier, ref) -> {
            if (ref.getScriptType().equals(scriptType) && !ref.getQueueStatus().getAndSet(true)) {
                this.importFileWhenReady((String)scriptIdentifier);
            }
        });
    }

    public void factoryRemoved(@Nullable String scriptType) {
        if (scriptType == null) {
            return;
        }
        Set<String> toRemove = this.scriptMap.values().stream().filter(ref -> ref.getLoadedStatus().get() && scriptType.equals(ref.getScriptType())).map(ScriptFileReference::getScriptIdentifier).collect(Collectors.toSet());
        toRemove.forEach(this::removeFile);
    }
}

