/**
 * @class Array
 * #core
 */
Ext.apply(Array.prototype, {
	/**
	 * Creates a shallow clone of the array.
	 * @return {Array} a clone of the array.
	 */
	clone: function()
	{
		return this.slice(0);
	},

	/**
	 * Equality comparison for the Array. Only when both
	 * arrays contain the exact same elements the Arrays
	 * are equal.
	 */
	equals: function(arr)
	{
		// We are comparing the exact same references.
		if (this === arr) {
			return true;
		}

		// If the lengths are not equal, then the arrays
		// cannot be equal.
		if (this.length !== arr.length) {
			return false;
		}

		// Compare each element for equality.
		return this.every(function(element, index) {
			return element === arr[index];
		});
	}
});

/* istanbul ignore if */
if (!Array.prototype.find) {
	/**
	 * Returns the value of the first element in the array that satisfies
	 * the provided testing function. Otherwise undefined is returned.
	 * @param {Function} predicate Function to execute on each value in the array
	 * @return {Mixed} A value in the array if an element passes the test; otherwise, undefined.
	 */
	Object.defineProperty(Array.prototype, 'find', {
		value: function(predicate) {
			// 1. Let O be ? ToObject(this value).
			if (this === null) {
				throw new TypeError('"this" is null or not defined');
			}

			var o = Object(this);

			// 2. Let len be ? ToLength(? Get(O, "length")).
			var len = o.length >>> 0;

			// 3. If IsCallable(predicate) is false, throw a TypeError exception.
			if (typeof predicate !== 'function') {
				throw new TypeError('predicate must be a function');
			}

			// 4. If thisArg was supplied, let T be thisArg; else let T be undefined.
			var thisArg = arguments[1];

			// 5. Let k be 0.
			var k = 0;

			// 6. Repeat, while k < len
			while (k < len) {
				// b. Let kValue be ? Get(O, Pk).
				// a. Let Pk be ! ToString(k).
				// c. Let testResult be ToBoolean(? Call(predicate, T, « kValue, k, O »)).
				// d. If testResult is true, return kValue.
				var kValue = o[k];
				if (predicate.call(thisArg, kValue, k, o)) {
					return kValue;
				}
				// e. Increase k by 1.
				k++;
			}

			// 7. Return undefined.
			return undefined;
		}
	});
}
/**
 * @class Date
 *
 * The date parsing and formatting syntax contains a subset of
 * <a href="http://www.php.net/date">PHP's date() function</a>, and the formats that are
 * supported will provide results equivalent to their PHP versions.
 *
 * The following is a list of all currently supported formats:
 * <pre>
Format  Description                                                               Example returned values
------  -----------------------------------------------------------------------   -----------------------
  d     Day of the month, 2 digits with leading zeros                             01 to 31
  D     A short textual representation of the day of the week                     Mon to Sun
  j     Day of the month without leading zeros                                    1 to 31
  l     A full textual representation of the day of the week                      Sunday to Saturday
  N     ISO-8601 numeric representation of the day of the week                    1 (for Monday) through 7 (for Sunday)
  S     English ordinal suffix for the day of the month, 2 characters             st, nd, rd or th. Works well with j
  w     Numeric representation of the day of the week                             0 (for Sunday) to 6 (for Saturday)
  z     The day of the year (starting from 0)                                     0 to 364 (365 in leap years)
  W     ISO-8601 week number of year, weeks starting on Monday                    01 to 53
  F     A full textual representation of a month, such as January or March        January to December
  m     Numeric representation of a month, with leading zeros                     01 to 12
  M     A short textual representation of a month                                 Jan to Dec
  n     Numeric representation of a month, without leading zeros                  1 to 12
  t     Number of days in the given month                                         28 to 31
  L     Whether it's a leap year                                                  1 if it is a leap year, 0 otherwise.
  o     ISO-8601 year number (identical to (Y), but if the ISO week number (W)    Examples: 1998 or 2004
  belongs to the previous or next year, that year is used instead)
  Y     A full numeric representation of a year, 4 digits                         Examples: 1999 or 2003
  y     A two digit representation of a year                                      Examples: 99 or 03
  a     Lowercase Ante meridiem and Post meridiem                                 am or pm
  A     Uppercase Ante meridiem and Post meridiem                                 AM or PM
  g     12-hour format of an hour without leading zeros                           1 to 12
  G     24-hour format of an hour without leading zeros                           0 to 23
  h     12-hour format of an hour with leading zeros                              01 to 12
  H     24-hour format of an hour with leading zeros                              00 to 23
  i     Minutes, with leading zeros                                               00 to 59
  s     Seconds, with leading zeros                                               00 to 59
  u     Decimal fraction of a second                                              Examples:
        (minimum 1 digit, arbitrary number of digits allowed)                     001 (i.e. 0.001s) or
                                                                                  100 (i.e. 0.100s) or
                                                                                  999 (i.e. 0.999s) or
                                                                                  999876543210 (i.e. 0.999876543210s)
  O     Difference to Greenwich time (GMT) in hours and minutes                   Example: +1030
  P     Difference to Greenwich time (GMT) with colon between hours and minutes   Example: -08:00
  T     Timezone abbreviation of the machine running the code                     Examples: EST, MDT, PDT ...
  Z     Timezone offset in seconds (negative if west of UTC, positive if east)    -43200 to 50400
  c     ISO 8601 date
        Notes:                                                                    Examples:
        1) If unspecified, the month / day defaults to the current month / day,   1991 or
           the time defaults to midnight, while the timezone defaults to the      1992-10 or
           browser's timezone. If a time is specified, it must include both hours 1993-09-20 or
           and minutes. The "T" delimiter, seconds, milliseconds and timezone     1994-08-19T16:20+01:00 or
           are optional.                                                          1995-07-18T17:21:28-02:00 or
        2) The decimal fraction of a second, if specified, must contain at        1996-06-17T18:22:29.98765+03:00 or
           least 1 digit (there is no limit to the maximum number                 1997-05-16T19:23:30,12345-0400 or
           of digits allowed), and may be delimited by either a '.' or a ','      1998-04-15T20:24:31.2468Z or
        Refer to the examples on the right for the various levels of              1999-03-14T20:24:32Z or
        date-time granularity which are supported, or see                         2000-02-13T21:25:33
        http://www.w3.org/TR/NOTE-datetime for more info.                         2001-01-12 22:26:34
  U     Seconds since the Unix Epoch (January 1 1970 00:00:00 GMT)                1193432466 or -2138434463
  M$    Microsoft AJAX serialized dates                                           \/Date(1238606590509)\/ (i.e. UTC milliseconds since epoch) or
                                                                                  \/Date(1238606590509+0800)\/
</pre>
 *
 * Example usage (note that you must escape format specifiers with '\\' to render them as character literals):
 * <pre><code>
// Sample date:
// 'Wed Jan 10 2007 15:05:01 GMT-0600 (Central Standard Time)'

var dt = new Date('1/10/2007 03:05:01 PM GMT-0600');
document.write(dt.format('Y-m-d'));                           // 2007-01-10
document.write(dt.format('F j, Y, g:i a'));                   // January 10, 2007, 3:05 pm
document.write(dt.format('l, \\t\\he jS \\of F Y h:i:s A'));  // Wednesday, the 10th of January 2007 03:05:01 PM
</code></pre>
 *
 * Here are some standard date/time patterns that you might find helpful.  They
 * are not part of the source of Date.js, but to use them you can simply copy this
 * block of code into any script that is included after Date.js and they will also become
 * globally available on the Date object.  Feel free to add or remove patterns as needed in your code.
 * <pre><code>
Date.patterns = {
    ISO8601Long:"Y-m-d H:i:s",
    ISO8601Short:"Y-m-d",
    ShortDate: "d/m/Y",
    LongDate: "l jS F Y",
    FullDateTime: "l jS F Y G:i:s",
    MonthDay: "jS F",
    ShortTime: "G:i",
    LongTime: "G:i:s",
    SortableDateTime: "Y-m-d\\TH:i:s",
    UniversalSortableDateTime: "Y-m-d H:i:sO",
    YearMonth: "F, Y"
};
</code></pre>
 *
 * Example usage:
 * <pre><code>
var dt = new Date();
document.write(dt.format(Date.patterns.ShortDate));
</code></pre>
 * <p>Developer-written, custom formats may be used by supplying both a formatting and a parsing function
 * which perform to specialized requirements. The functions are stored in {@link #parseFunctions} and {@link #formatFunctions}.</p>
 * #core
 */
Ext.apply(Date.prototype, {
	/**
	 * Provides a convenient method for performing basic date arithmetic. This method
	 * does not modify the Date instance being called - it creates and returns
	 * a new Date instance containing the resulting date value.
	 *
	 * Examples:
	 * <pre><code>
		// Basic usage:
		var dt = new Date('10/29/2006').add(Date.DAY, 5);
		document.write(dt); //returns 'Fri Nov 03 2006 00:00:00'

		// Negative values will be subtracted:
		var dt2 = new Date('10/1/2006').add(Date.DAY, -5);
		document.write(dt2); //returns 'Tue Sep 26 2006 00:00:00'

		// You can even chain several calls together in one line:
		var dt3 = new Date('10/1/2006').add(Date.DAY, 5).add(Date.HOUR, 8).add(Date.MINUTE, -30);
		document.write(dt3); //returns 'Fri Oct 06 2006 07:30:00'
	 * </code></pre>
	 *
	 * Furthermore, changes to {@link Date#HOUR hours}, {@link Date#MINUTE minutes},
	 * {@link Date#SECOND seconds} and {@link Date#MILLI milliseconds} are treated more accurately
	 * regarding DST changes then the {@link Date#DAY days}, {@link Date#MONTH months} and {@link Date#YEAR years}
	 * changes. When changing the time the standard is applied, which means that if the DST kicks in at 2AM,
	 * and the time becomes 3AM. Doing new Date('Mar 25 2012 01:00').add(Date.HOUR, 1) will be 'Mar 25 2012 03:00'.
	 * However when changing the date, we will use the JS behavior, which means that
	 * new Date('Mar 24 2012 02:00').add(Date.DAY, 1) could become 'Mar 25 2012 01:00' as JS will not correctly
	 * move the time correctly passed the DST switch.
	 *
	 * @param {String} interval A valid date interval enum value.
	 * @param {Number} value The amount to add to the current date.
	 * @return {Date} The new Date instance.
	 */
	add: function(interval, value)
	{
		var d = this.clone();
		if (!interval || value === 0) {
			return d;
		}

		switch(interval.toLowerCase()) {
			// Changing the time is done more accuretely then
			// changing the date. This is because we have to work
			// around DST issues (which we don't care for when
			// changing the day). In JS, we have the following
			// scenario at the following date: Mar 25 2012.
			// At 2:00:00 the DST kicks in and the time will be
			//     Mar 25 2012 03:00:00
			// However, when using setMilliseconds, setSeconds,
			// setMinutes or setHours, JS decides to wrap back
			// to:
			// 	Mar 25 2012 01:00:00
			// How can this go wrong, take the following date:
			//      a = new Date('Mar 25 2012 01:45:00')
			// add 30 minutes to it
			//      a.setMinutes(a.getMinutes() + 30)
			// we expect the time to be 03:15:00 however JS
			// decides to fall back to 01:15:00.
			// To fix this correctly, we have to work using timestamps
			// as JS is able to correctly step over the DST switch.
			case Date.HOUR:
				// Convert value to minutes
				value *= 60;
				/* falls through */
			case Date.MINUTE:
				// Convert value to seconds
				value *= 60;
				/* falls through */
			case Date.SECOND:
				// Convert value to milliseconds
				value *= 1000;
				/* falls through */
			case Date.MILLI:
				d = new Date(d.getTime() + value);
				break;
			// Changing the date is done with less accuracy,
			// basically we don't care if we come at exactly
			// the same time as before. If the JS decides to
			// perform weird tricks, then so be it.
			case Date.DAY:
				d.setDate(this.getDate() + value);
				break;
			case Date.MONTH:
				var day = this.getDate();
				if (day > 28) {
					day = Math.min(day, this.getFirstDateOfMonth().add(Date.MONTH, value).getLastDateOfMonth().getDate());
				}
				d.setDate(day);
				d.setMonth(this.getMonth() + value);
				break;
			case Date.YEAR:
				d.setFullYear(this.getFullYear() + value);
				break;
		}
		return d;
	},

	/**
	 * This should be called if the Date() object represents a UTC date, and we want to obtain
	 * the time it represents in UTC shown as if it was the localtime. This implies that:
	 *
	 * 00:00 UTC (01:00 GMT+0100) will be converted to 00:00 GMT+0100 (23:00 UTC)
	 *
	 * @return {Date} The UTC date
	 */
	toUTC: function()
	{
		var utc = new Date(this.getTime() + (this.getTimezoneOffset() * 60000));

		// Obtain the DST difference which might have occurred during conversion,
		// if there was a difference it must be applied to the utc date accordingly.
		utc.setMilliseconds(utc.getMilliseconds() + Date.getDSTDiff(utc, this));

		return utc;
	},

	/**
	 * This should be called if the Date() was obtained using {@link #toUTC}, and we want
	 * to convert the date back to the local representation.
	 *
	 * 00:00 GMT+0100 (23:00 UTC) with be converted to 00:00 UTC (01:00 GMT+0100)
	 *
	 * @return {Date} The local-time date
	 */
	fromUTC: function()
	{
		return new Date(this.getTime() - (this.getTimezoneOffset() * 60000));
	},

	/**
	 * Get the next given weekday starting from this date. If the current date is this weekday,
	 * then the current day will be returned.
	 * @param {Number} weekday The day in the week to skip to (0: Sunday -  6: Saturday). If
	 * not given, tomorrow will be returned.
	 * @return {Date} this or the clone
	 */
	getNextWeekDay: function(weekday)
	{
		var currentday = this.getDay();

		if (!Ext.isDefined(weekday)) {
			return this.add(Date.DAY, 1);
		} else if (weekday < currentday) {
			return this.add(Date.DAY, 7 - (currentday - weekday));
		} else {
			return this.add(Date.DAY, weekday - currentday);
		}
	},

	/**
	 * Get the previous given weekday starting from this date. If the current date is this weekday,
	 * then the current day will be returned.
	 * @param {Number} weekday The day in the week to skip to (0: Sunday -  6: Saturday). If
	 * not given, yesterday will be returned.
	 * @return {Date} this or the clone
	 */
	getPreviousWeekDay: function(weekday)
	{
		var currentday = this.getDay();

		if (!Ext.isDefined(weekday)) {
			return this.add(Date.DAY, -1);
		} else if (weekday <= currentday) {
			return this.add(Date.DAY, weekday - currentday);
		} else {
			return this.add(Date.DAY, -7 + (weekday - currentday));
		}
	},

	/**
	 * Get the next given working weekday starting from the date passed as argument or
	 * current date in case no argument was provided.
	 * @param {Date} currentDate (Optional) The date for which next working day should be returned
	 * @return {Date} Date fall on the next working day or false if no working days are defined
	 */
	getNextWorkWeekDay: function(currentDate)
	{
		currentDate = currentDate || new Date();
		var nextDate = currentDate.getNextWeekDay();
		var workingDaysList = container.getSettingsModel().get('zarafa/v1/main/working_days');
		if ( Ext.isEmpty(workingDaysList) ){
			return false;
		}

		if (workingDaysList.indexOf(nextDate.getDay()) !== -1 ) {
			return nextDate;
		} else {
			return this.getNextWorkWeekDay(nextDate);
		}
	},

	/**
	 * Attempts to clear all Second and millisecond time information from this Date by rounding the time down
	 * to the current minute.
	 * @param {Boolean} clone true to create a clone of this date, clear the time and return it (defaults to false).
	 * @return {Date} this or the clone.
	 */
	clearSeconds: function(clone)
	{
		if (clone) {
			return this.clone().clearSeconds();
		}

		// clear seconds
		this.setSeconds(0);
		this.setMilliseconds(0);

		return this;
	},

	/**
	 * Sets the time of the date to 12:00 am.
	 * @return {Date} this
	 */
	setToNoon: function()
	{
		this.clearTime().setHours(12);

		return this;
	},

	/**
	 * Checks if the given date is in the same week.
	 * @param {Date} date The date to compare
	 * @return {Boolean} true if the date is in the same week as this date, false otherwise.
	 */
	inSameWeekAs: function(date)
	{
		var clone = this.clone().setToNoon();
		clone = clone.add(Date.DAY, -1*clone.getDay());
		date.setToNoon();
		date = date.add(Date.DAY, -1*date.getDay());

		return clone.getTime() === date.getTime();
	},

	/**
	 * Checks if the given date is in the next week.
	 * @param {Date} date The date to compare
	 * @return {Boolean} true if the date is in the next week from this date, false otherwise.
	 */
	inNextWeek: function(date)
	{
		return this.add(Date.DAY, 7).inSameWeekAs(date);
	},

	/**
	 * Round the number of milliseconds/seconds/minutes or hours depending on the given value.
	 * Note that days, months and years cannot be rounded.
	 *
	 * Example of rounding:
	 *  9:12   round to 30	-> 9:00
	 *  9:17   round to 30	-> 9:30
	 *
	 * @param {String} field The field to round (e.g. {@link Date.MINUTE}, {@link Date.SECOND}, etc.)
	 * @param {Number} roundTimeValue The number of minutes to round the time to.
	 * @return {Date} this date
	 */
	round: function(field, roundTimeValue)
	{
		// For each field we have a slightly different approach.
		// In all cases, if the field-value is already rounded,
		// then we don't need to do anything.
		// The calculation for the rounded value looks a bit weird, but
		// it is a bit more optimal then calling this.floor() or this.ceiling().
		// For seconds or higher units, we set all smaller units to 0,
		// to correctly round of the entire time.
		var value;
		switch (field) {
			case Date.MILLI:
				value = this.getMilliseconds();
				if (value % roundTimeValue > 0) {
					this.setMilliseconds(value - (value % roundTimeValue) + (((value % roundTimeValue) >= (roundTimeValue / 2)) * roundTimeValue));
				}
				break;
			case Date.SECOND:
				value = this.getSeconds();
				if (value % roundTimeValue > 0) {
					this.setSeconds(value - (value % roundTimeValue) + (((value % roundTimeValue) >= (roundTimeValue / 2)) * roundTimeValue));
				}
				this.setMilliseconds(0);
				break;
			case Date.MINUTE:
				value = this.getMinutes();
				if (value % roundTimeValue > 0) {
					this.setMinutes(value - (value % roundTimeValue) + (((value % roundTimeValue) >= (roundTimeValue / 2)) * roundTimeValue));
				}
				this.setSeconds(0);
				this.setMilliseconds(0);
				break;
			case Date.HOUR:
				value = this.getHours();
				if (value % roundTimeValue > 0) {
					this.setHours(value - (value % roundTimeValue) + (((value % roundTimeValue) >= (roundTimeValue / 2)) * roundTimeValue));
				}
				this.setMinutes(0);
				this.setSeconds(0);
				this.setMilliseconds(0);
				break;
		}

		return this;
	},

	/**
	 * Function to ceil timings according to the passed ceil milliseconds, seconds, minutes or hours.
	 * Note that days, months and years cannot be ceiled.
	 * @param {String} field The field to ceil (e.g. {@link Date.MINUTE}, {@link Date.SECOND}, etc.)
	 * @param date ceilTimeValue date time which needs to be ceil (5/10/15/30/60 or so on)
	 * @return number Time number which is unixtimestamp of time.
	 *
	 * Example to understand what the code is actually suppose to do.
	 *	9:12	5min		ceil-9:15
	 *			10min		ceil-9.20
	 *			15min		ceil-9.15
	 *			30min		ceil-9.30
	 *			1hr/60min	ceil-10.00
	 *
	 */
	ceil: function(field, ceilTimeValue)
	{
		// For each field we have a slightly different approach.
		// In all cases, if the field-value is already rounded to the
		// given ceiling then we don't need to do anything.
		// For seconds or higher units, we set all smaller units to 0,
		// to correctly round of the entire time.
		var value;
		switch (field) {
			case Date.MILLI:
				value = this.getMilliseconds();
				if (value % ceilTimeValue > 0) {
					this.setMilliseconds(value - (value % ceilTimeValue) + ceilTimeValue);
				}
				break;
			case Date.SECOND:
				value = this.getSeconds();
				if (value % ceilTimeValue > 0) {
					this.setSeconds(value - (value % ceilTimeValue) + ceilTimeValue);
				}
				this.setMilliseconds(0);
				break;
			case Date.MINUTE:
				value = this.getMinutes();
				if (value % ceilTimeValue > 0) {
					this.setMinutes(value - (value % ceilTimeValue) + ceilTimeValue);
				}
				this.setSeconds(0);
				this.setMilliseconds(0);
				break;
			case Date.HOUR:
				value = this.getHours();
				if (value % ceilTimeValue > 0) {
					this.setHours(value - (value % ceilTimeValue) + ceilTimeValue);
				}
				this.setMinutes(0);
				this.setSeconds(0);
				this.setMilliseconds(0);
				break;
		}

		return this;
	},

	/**
	 * Function to floor timings according to the passed floor milliseconds, seconds, minutes or hours.
	 * Note that days, months and years cannot be floored.
	 * @param {String} field The field to floor (e.g. {@link Date.MINUTE}, {@link Date.SECOND}, etc.)
	 * @param {Number} floorTimeValue date time which needs to be floor (5/10/15/30/60 or so on)
	 * @return {Date} This Date object
	 *
	 * Example to understand what the code is actually suppose to do.
	 *	9:12	5min		floor-9.10
	 *			10min		floor-9.10
	 *			15min		floor-9.00
	 *			30min		floor-9.00
	 *			1hr/60min	floor-9.00
	 *
	 */
	floor: function(field, floorTimeValue)
	{
		// For each field we have a slightly different approach.
		// In all cases, if the field-value is already rounded to the
		// given floor then we don't need to do anything.
		// For seconds or higher units, we set all smaller units to 0,
		// to correctly round of the entire time.
		var value;
		switch (field) {
			case Date.MILLI:
				value = this.getMilliseconds();
				if (value % floorTimeValue > 0) {
					this.setMilliseconds(value - (value % floorTimeValue));
				}
				break;
			case Date.SECOND:
				value = this.getSeconds();
				if (value % floorTimeValue > 0) {
					this.setSeconds(value - (value % floorTimeValue));
				}
				this.setMilliseconds(0);
				break;
			case Date.MINUTE:
				value = this.getMinutes();
				if (value % floorTimeValue > 0) {
					this.setMinutes(value - (value % floorTimeValue));
				}
				this.setSeconds(0);
				this.setMilliseconds(0);
				break;
			case Date.HOUR:
				value = this.getHours();
				if (value % floorTimeValue > 0) {
					this.setHours(value - (value % floorTimeValue));
				}
				this.setMinutes(0);
				this.setSeconds(0);
				this.setMilliseconds(0);
				break;
		}

		return this;
	},

	/**
	 * Get the week number of the month (1 to 5)
	*/
	getWeekOfMonth: function()
	{
		// get current week number in year
		var currentWeek = this.getWeekOfYear();

		// get month's first week number in year
		var monthStartDate = this.add(Date.DAY, -(this.getDate() - 1));
		var monthStartWeek = monthStartDate.getWeekOfYear();

		return currentWeek - monthStartWeek + 1;
	},

	/**
	 * Returns the date in a 'nicely' formatted string.
	 * @param {Boolean} includeTime The time will be added to the nicely
	 * formatted string when true, or omitted otherwise. Default is true.
	 *
	 * @return {String} The nicely formatted date string.
	 */
	getNiceFormat: function(includeTime)
	{
		includeTime = includeTime !== false;

		var d = this.clearTime(true);
		var now = new Date().clearTime();

		// Check for today. We'll only use time then.
		if ( d.getTime() === now.getTime() ){
			if ( includeTime ){
				return this.formatDefaultTime();
			} else {
				return _('Today');
			}
		}

		// Check for tomorrow.
		if ( d.add(Date.DAY, -1).getTime() === now.getTime() ){
			if ( includeTime ){
				return _('Tomorrow') + ' ' + this.formatDefaultTime();
			} else {
				return _('Tomorrow');
			}
		}

		// Check for future dates. We'll only show the date then.
		if ( d > now ){
			return this.format(_('d-m-Y'));
		}

		// Check for past week. We'll use day (name) + time then.
		if ( d.add(Date.DAY, 6) >= now ){
			if ( includeTime ){
				return this.formatDefaultTime(_('D {0}'));
			} else {
				return this.format(_('D d-m'));
			}
		}

		// Check for two weeks ago. We'll use the day (name) + date (without year)
		if ( d.add(Date.DAY, 14) >= now ){
			return this.format(_('D d-m'));
		}

		// For anything older than two weeks ago, we'll show the date
		return this.format(_('d-m-Y'));
	},

	/**
	* Function returns a formatted date string with the specified time format.
	*
	* @param {String} formatString The time format string
	* @return {String} The formatted date/time string
	*/
	formatDefaultTime: function(formatString)
	{
		var timeFormat = container.settingsModel.get('zarafa/v1/main/datetime_time_format');
		var newFormat;

		if (Ext.isDefined(formatString)) {
			newFormat = String.format(formatString, timeFormat);
		} else {
			newFormat = timeFormat;
		}
		return this.format(newFormat);
	}
});

Ext.apply(Date, {
    /**
     * Get the short day name for the given day number.
     * Overridden to make the short days translatable.
     * @param {Number} day A zero-based javascript day number.
     * @return {String} The short day name.
     * @static
     */
    getShortDayName: function(day) {
		// Let's first check if the short days have been translated.
		// If not, we'll fall back to using the first 3 letters of
		// the full day.
		if (
			Date.shortDayNames[0] === 'Sun' &&
			Date.shortDayNames[1] === 'Mon' &&
			Date.shortDayNames[2] === 'Tue' &&
			Date.shortDayNames[3] === 'Wed' &&
			Date.shortDayNames[4] === 'Thu' &&
			Date.shortDayNames[5] === 'Fri' &&
			Date.shortDayNames[6] === 'Sat'
		) {
			return Date.dayNames[day].substring(0, 3);
		}

        return Date.shortDayNames[day];
    },

	/**
	 * The number milliseconds per day
	 *
	 * @property
	 * @type Number
	 */
	dayInMillis : 24 * 60 * 60 * 1000,

	/**
	 * Calculate the DST difference between the 2 given dates.
	 * The first date serves as base, so when 'date' is not DST, but
	 * the second date is DST then a negative offset is returned. A
	 * positive value is returned when it is the other way around.
	 * When both dates have the same DST offset then this returns 0.
	 * @param {Date} date The base date from where the DST is calculated.
	 * @return {Number} milliseconds The DST difference in milliseconds
	 */
	getDSTDiff: function(a, b)
	{
		return (a.getTimezoneOffset() - b.getTimezoneOffset()) * 60 * 1000;
	},

	/**
	 * Calculates the difference between the 2 given dates.
	 * This applies the {@link #getDSTDiff} if needed to ensure that
	 * it always calculates the correct difference regardless of the DST changes
	 * which might have been made.
	 *
	 * In its absolute basic this function is equal to 'a.getTime() - b.getTime()'.
	 *
	 * @param {String} field The field which indicates the accuracy of the diff (e.g. {@link Date.MINUTE}, {@link Date.SECOND}, etc.)
	 * @param {Date} a The date object
	 * @param {Date} b The date object
	 * @return {Number} The difference between the 2 given dates
	 */
	diff: function(field, a, b)
	{
		var ta = a.getTime();
		var tb = b.getTime();
		var difference = ta-tb;

		switch (field) {
			case Date.DAY:
				// For calculating days we apply the same
				// inaccuracy as Date::add() we are not 100%
				// sure a day lasts 24 hour when DST is in play.
				difference -= Date.getDSTDiff(a, b);
				difference /= 24;
				/* falls through */
			case Date.HOUR:
				difference /= 60;
				/* falls through */
			case Date.MINUTE:
				difference /= 60;
				/* falls through */
			case Date.SECOND:
				difference /= 1000;
				/* falls through */
			case Date.MILLI:
				/* falls through */
			default:
				break;
		}

		return difference;
	},

	/**
	 * Function to getTimezone and all dst props
	 * This is a hard one. To create a recurring appointment, we need to save
	 * the start and end time of the appointment in local time. So if I'm in
	 * GMT+8, and I want the appointment at 9:00, I will simply save 9*60 = 540
	 * in the startDate. To make this usable for other users in other timezones,
	 * we have to tell the server in which timezone this is. The timezone is normally
	 * defined as a startdate and enddate for DST, the offset in minutes (so GMT+2 is 120)
	 * plus the extra DST offset when DST is in effect.
	 *
	 * We can't retrieve this directly from the browser, so we assume that the DST change
	 * will occur on a Sunday at 2:00 or 3:00 AM, and simply scan all the sundays in a
	 * year, looking for changes. We then have to guess which bit is DST and which is 'normal'
	 * by assuming that the DST offset will be less than the normal offset. From this we
	 * calculate the start and end dates of DST and the actual offset in minutes.
	 *
	 * Unfortunately we can't detect the difference between 'the last week of october' and
	 * 'the fourth week of october'. This can cause subtle problems, so we assume 'last week'
	 * because this is most prevalent.
	 *
	 * Note that this doesn't work for many strange DST changes, see
	 * http://webexhibits.org/daylightsaving/g.html
	 * @static
	 */
	getTimezoneStruct: function()
	{
		var tzswitch = [],
			switchCount = 0,
			testDate = new Date(),
			tzStruct = {};

		// Clear the time
		testDate.setMonth(0);
		testDate.setDate(1);
		testDate.setMinutes(0);
		testDate.setSeconds(0);
		testDate.setMilliseconds(0);

		// Move to the next sunday
		testDate = testDate.getNextWeekDay(0);

		// Use 5:00 am because any change should have happened by then
		testDate.setHours(5);

		var lastoffset = testDate.getTimezoneOffset();

		for(var weekNr = 0; weekNr < 52; weekNr++) {
			if(testDate.getTimezoneOffset() != lastoffset) {
				// Found a switch
				tzswitch[switchCount] = {
					switchweek : testDate.getWeekOfMonth(),
					switchmonth : testDate.getMonth(),
					offset : testDate.getTimezoneOffset()
				};

				switchCount++;

				// We assume DST is only set or removed once per year
				if(switchCount == 2) {
					break;
				}

				lastoffset = testDate.getTimezoneOffset();
			}

			// advance one week
			testDate = testDate.add(Date.DAY, 7);
		}

		if(switchCount === 0) {
			// No DST in this timezone
			tzStruct = {
				timezone : testDate.getTimezoneOffset(),
				timezonedst : 0,
				dststartday : 0,
				dststartweek : 0,
				dststartmonth : 0,
				dststarthour : 0,
				dstendday : 0,
				dstendweek : 0,
				dstendmonth : 0,
				dstendhour : 0
			};

			return tzStruct;
		} else if(switchCount == 1) {
			// This should be impossible unless DST started somewhere in the year 2000
			// and ended more than a year later. This is an error.
			return tzStruct;
		} else if(switchCount == 2) {
			if(tzswitch[0].offset < tzswitch[1].offset) {
				// Northern hemisphere, eg DST is during Mar-Oct
				tzStruct = {
					timezone : tzswitch[1].offset,
					timezonedst : tzswitch[0].offset - tzswitch[1].offset,
					dststartday : 0, // assume sunday
					dststartweek : tzswitch[0].switchweek == 4 ? 5 : tzswitch[0].switchweek, // assume 'last' week if week = 4
					dststartmonth : tzswitch[0].switchmonth + 1, // javascript months are zero index based
					dststarthour : 2, // Start at 02:00 AM
					dstendday : 0,
					dstendweek : tzswitch[1].switchweek == 4 ? 5 : tzswitch[1].switchweek,
					dstendmonth : tzswitch[1].switchmonth + 1,
					dstendhour : 3
				};

				return tzStruct;

			} else {
				// Southern hemisphere, eg DST is during Oct-Mar
				tzStruct = {
					timezone : tzswitch[0].offset,
					timezonedst : tzswitch[1].offset - tzswitch[0].offset,
					dststartday : 0, // assume sunday
					dststartweek : tzswitch[1].switchweek == 4 ? 5 : tzswitch[1].switchweek, // assume 'last' week if week = 4
					dststartmonth : tzswitch[1].switchmonth + 1,
					dststarthour : 2, // Start at 02:00 AM
					dstendday : 0,
					dstendweek : tzswitch[0].switchweek == 4 ? 5 : tzswitch[0].switchweek,
					dstendmonth : tzswitch[0].switchmonth + 1,
					dstendhour : 3
				};

				return tzStruct;
			}
		} else {
			// Multi-DST timezone ? This is also an error.
			return tzStruct;
		}
	}
});
/**
 * @class Ext.EventObjectImpl
 * #core
 */
Ext.apply(Ext.EventObjectImpl.prototype, {
	/**
	 * Function will return Character code for the key event
	 * Number keys of numpad keys have keycode of 96 to 105, this function will
	 * convert it to proper ASCII character and return it
	 *
	 * @return {String} The trimmed string
	 */
	getKeyCharCode: function()
	{
		var key = this.getCharCode();
		// Handle numpad keys here
		if (key >= this.NUM_ZERO && key <= this.NUM_NINE) {
			// These are numkeys and it will have different keyCode than number's
			// ASCII value so we are mapping these keys to it's original character values.
			key = key - 48;
		}
		return String.fromCharCode(key);
	}
});
/**
 * @class String
 * #core
 */

// https://github.com/uxitten/polyfill/blob/master/string.polyfill.js
// https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/String/repeat
if (!String.prototype.padEnd) {
  /* jshint freeze: false */
  String.prototype.padEnd = function padEnd(targetLength,padString) {
    targetLength = targetLength>>0; //floor if number or convert non-number to 0;
    padString = String(padString || ' ');
    if (this.length > targetLength) {
      return String(this);
    }
    else {
      targetLength = targetLength-this.length;
      if (targetLength > padString.length) {
        padString += padString.repeat(targetLength/padString.length); //append to original to ensure we are longer than needed
      }
      return String(this) + padString.slice(0,targetLength);
    }
  };
}

if (!String.prototype.startsWith) {
	/**
	 * Determines whether a string begins with the characters of a specified string,
	 * returning true or false as appropriate.
	 * @param {String} searchString The characters to be searched for at the start of this string.
	 * @param {Number} position The position in this string at which to begin searching for searchString; defaults to 0.
	 * @return {Boolean} true if the given characters are found at the beginning of the string; otherwise, false.
	 */
	Object.defineProperty(String.prototype, 'startsWith', {
		value: function(searchString, position) {
			return this.substr(position || 0, searchString.length) === searchString;
		}
	});
}

// https://github.com/uxitten/polyfill/blob/master/string.polyfill.js
// https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/String/padStart
if (!String.prototype.padStart) {
  /* jshint freeze: false */
  String.prototype.padStart = function padStart(targetLength,padString) {
    targetLength = targetLength>>0; //floor if number or convert non-number to 0;
    padString = String(padString || ' ');
    if (this.length > targetLength) {
      return String(this);
    }
    else {
      targetLength = targetLength-this.length;
      if (targetLength > padString.length) {
        padString += padString.repeat(targetLength/padString.length); //append to original to ensure we are longer than needed
      }
      return padString.slice(0,targetLength) + String(this);
    }
  };
}

if (!String.prototype.repeat) {

	/**
	 * IE does not supported repeat function so, add a function into string to available repeat the string function.
	 * @param {Number} times an integer indicating the number of times to repeat the string in the newly-created string.
	 * @returns {string} A new string containing the specified number of copies of the given string.
	 */
	String.prototype.repeat = function repeat(times) {
		var repeatedString = "";

		while (times > 0) {
			repeatedString += this;
			times--;
		}

		return repeatedString;
	};
}Ext.namespace('Zarafa');

/**
 * @class Zarafa
 * Global convenience methods.
 * @singleton
 * #core
 */
Ext.apply(Zarafa, {
	/**
	 * The string we use to create the regular expressions for email address validation
	 *
	 * @property
	 * @type {String}
	 * @private
	 */
	emailAddressRegExpString: '(?:[a-zA-Z0-9.!#$%&\'*+\\-/=?^_`{|}~])+@[a-zA-Z0-9-]+(?:\\.[a-zA-Z0-9-]+)*\\.(?:[a-zA-Z0-9]{2,15})',

	/**
	 * A regular expression to test the validity of an email address
	 *
	 * @property
	 * @type {RegExp}
	 */
	reSingleEmailAddress: undefined,

	/**
	 * A regular expression to find email addresses in a string
	 *
	 * @property
	 * @type {RegExp}
	 */
	reMultipleEmailAddresses: undefined,

	/**
	 * Ready flag which indicates that Webapp has been loaded.
	 * (See {@link #onReady}).
	 * @property
	 * @type Boolean
	 */
	isReady: false,

	/**
	 * Registration object for {@link #onReady} onto which all event
	 * handlers are being registered which want to be notified when
	 * grommunio Web has been initialized and ready for plugin interaction.
	 *
	 * @property
	 * @type Ext.util.Event
	 * @private
	 */
	readyEvent: new Ext.util.Event(),

	/**
	 * Ready flag which indicates that Webapp UI has been loaded.
	 * (See {@link #onUIReady}).
	 * @property
	 * @type Boolean
	 */
	uiReady: false,

	/**
	 * Registration object for {@link #uiReady} onto which all event
	 * handlers are being registered which want to be notified when
	 * grommunio Web has drawn the main UI and has loaded the hierarchy panel.
	 *
	 * @property
	 * @type Ext.util.Event
	 * @private
	 */
	uiReadyEvent: new Ext.util.Event(),

	/**
	 * The time that the user has not done any action
	 * (like mousemove, click, or keypress) in grommunio Web.
	 *
	 * @property
	 * @type Integer
	 * @private
	 */
	idleTime: 0,

	/**
	 * True if the user is running DeskApp to view grommunio Web, false otherwise.
	 *
	 * @property
	 * @type {Boolean}
	 */
	isDeskApp: Ext.isDefined(window.nw),

	/**
	 * Adds a listener to be notified when grommunio Web is ready. This will be somewhere during {@link Ext.onReady}, when
	 * grommunio Web has initialized the bare essentials. When the event is fired, the {@link Zarafa.core.Container} will
	 * be available, and plugins are allowed to register.
	 *
	 * @param {Function} fn The method the event invokes.
	 * @param {Object} scope (optional) The scope (<code>this</code> reference) in which the handler function executes. Defaults to the browser window.
	 * @param {Boolean} options (optional) Options object as passed to {@link Ext.Element#addListener}. It is recommended that the options
	 * <code>{single: true}</code> be used so that the handler is removed on first invocation.
	 */
	onReady: function(fn, scope, options)
	{
		this.readyEvent.addListener(fn, scope, options);

		// If the environment is already ready, can
		// should call fireReady again to fire the
		// just registered event.
		if (this.isReady) {
			this.fireReady();
		}
	},

	/**
	 * Called when {@link Ext.onReady} has been invoked, and grommunio Web has been initialized.
	 * All handlers registered through {@link #onReady} will now be fired and {@link #isReady}
	 * will be set.
	 *
	 * @private
	 */
	fireReady: function()
	{
		this.isReady = true;
		this.readyEvent.fire();
		this.readyEvent.clearListeners();
	},

	/**
	 * Adds a listener to be notified when grommunio Web UI is drawn and the hierarchy is loaded.
	 * @param {Function} fn The method the event invokes.
	 * @param {Object} scope (optional) The scope (<code>this</code> reference) in which the handler function executes. Defaults to the browser window.
	 * @param {Boolean} options (optional) Options object as passed to {@link Ext.Element#addListener}. It is recommended that the options
	 * <code>{single: true}</code> be used so that the handler is removed on first invocation.
	 */
	onUIReady: function(fn, scope, options)
	{
		// Force single is true for events.
		options = options || {};
		options.single = true;

		this.uiReadyEvent.addListener(fn, scope, options);

		// If the environment is already ready, call fireUIReady again
		// to fire the just registered event.
		if (this.uiReady) {
			this.fireUIReady();
		}
	},

	/**
	 * Called when grommunio Web's UI has been loaded and the hierarchy is loaded.
	 * All handlers registered through {@link #onUIReady} will now be fired and {@link #uiReady}
	 * will be set.
	 *
	 * @private
	 */
	fireUIReady: function()
	{
		this.uiReady = true;
		this.uiReadyEvent.fire();
		this.uiReadyEvent.clearListeners();
	},

	/**
	 * Initialize all Global variables as used by grommunio Web.
	 *
	 * This will utilize some global objects as received by the PHP
	 * side, and apply them into the proper classes, after which the
	 * global objects will be destroyed.
	 *
	 * This will instantiate {@link Zarafa.core.Container container}.
	 * @private
	 */
	initializeGlobals: function()
	{
		// Use native json handling of browser for performance benefit
		Ext.USE_NATIVE_JSON = true;

		//show confirm dialog before user leave the page.
		Zarafa.core.Util.enableLeaveRequester();

		// When the browser is unloading, all active requests will be aborted and
		// If more than one browser windows are open then close all browser windows.
		const handlePageHide = function () {
			if (Zarafa.core.BrowserWindowMgr.browserWindows.length > 1) {
				Zarafa.core.BrowserWindowMgr.closeAllBrowserWindow();
			}
			container.getRequest().paralyze(Zarafa.core.data.ParalyzeReason.BROWSER_RELOADING);
		};
		window.addEventListener('pagehide', handlePageHide);

		// Initialize the regular expressions that can be used to validate email addresses
		this.reSingleEmailAddress = new RegExp('^' + this.emailAddressRegExpString + '$');
		this.reMultipleEmailAddresses = new RegExp('([^,;\\n\\r]*?<{0,1}' + Zarafa.emailAddressRegExpString + ')>{0,1}(?=(?:$|[,;\\n\\r\\s]))', 'g');

		// Create global container object
		container = new Zarafa.core.Container();

		// Set the server object
		/*global user version languages serverconfig settings:true*/
		container.setServerConfig(serverconfig);
		delete serverconfig;

		// Load all settings
		container.getSettingsModel().initialize(settings);
		delete settings;

		// Set Onlyoffice theme
		const activeTheme = container.getSettingsModel().get('zarafa/v1/main/active_theme');
		window.localStorage.setItem('ui-theme',
			activeTheme === 'dark' ? '{"id":"theme-dark","type":"dark"}' : '{"id":"theme-light","type":"light"}');
		window.localStorage.setItem('ui-theme-id',
			activeTheme === 'dark' ? 'theme-dark' : 'theme-light');

		// Load all persistent settings (i.e. settings that will not be deleted when the user resets his settings)
		// Persistent settings are not added to the welcome screen, so check if they exist first.
		if ( Ext.isDefined(window.persistentsettings) ){
			container.getPersistentSettingsModel().initialize(window.persistentsettings);
			delete window.persistentsettings;
		}

		// Set the user object
		container.setUser(user);
		delete user;

		// Set the version object
		container.setVersion(version);
		delete version;

		// Set the language object
		container.setLanguages(languages);
		delete languages;

		// Set up DOMPurify
		DOMPurify.setConfig({
			FORBID_TAGS: ['iframe', 'webview', 'meta', 'html', 'head', 'link'],
			WHOLE_DOCUMENT: false,
			// Default regEx of DOMPurify for uri does not allow some protocols like file, smb, etc.
			// So we need to whitelist them by this new regEx.
			ALLOWED_URI_REGEXP: Object.seal(/^(?:(?:(?:f|ht)tps?|mailto|elodms|msteams|webexteams|zoommtg|zoomus|gotomeeting|tel|callto|cid|xmpp|smb|file):|[^a-z]|[a-z]:|[a-z+.\-]+(?:[^a-z+.\-:]|$))/i),
			ALLOW_DATA_ATTR: true,
			ADD_DATA_URI_TAGS: [ 'a', 'img', 'image' ],
			ADD_TAGS: ['svg', 'use', 'symbol'],
			ADD_ATTRIBUTES: ['xlink', 'xlink:href', 'href'],
		});

		DOMPurify.addHook('afterSanitizeAttributes', function(node) {
			// Set all elements owning target to target=_blank.
			if ('target' in node) {
				node.setAttribute('target', '_blank');
				// prevent https://www.owasp.org/index.php/Reverse_Tabnabbing
				node.setAttribute('rel', 'noopener noreferrer external');

				if (node.nodeName === "A" && !Ext.isEmpty(node.href)){
					node.setAttribute('title', `${node.href} \n ${_("Click the link to open the URL in a new window.")}`);
				}
			} else if (!node.hasAttribute('target') && (node.hasAttribute('xlink:href') || node.hasAttribute('href'))) {
				// set non-HTML/MathML links to xlink:show=new
				node.setAttribute('xlink:show', 'new');
			}
		});

	},

	/**
	 * Initialize the ExtJs/Web environment, register generic event listeners,
	 * This will listen to the 'contextmenu' event on the {@link Ext#getBody body element}
	 * as well as the exception events on the {@link Zarafa.core.data.IPMStoreMgr} and
	 * {@link Zarafa.core.ResponseRouter}.
	 * @private
	 */
	initializeEnvironment: function()
	{
		// Register the State provider which uses the SettingsModel.
		Ext.state.Manager.setProvider(new Zarafa.core.data.SettingsStateProvider());

		// Disable contextmenu globally
		Ext.getBody().on('contextmenu', this.onBodyContextMenu, this);

		// Disable default file drop behavior
		Ext.EventManager.on(window, 'dragover', this.onWindowDragDrop, this);
		Ext.EventManager.on(window, 'drop', this.onWindowDragDrop, this);

		// Add main event handlers to listen for errors
		container.getRequest().on('connectionparalyzed', this.onConnectionParalyze, this);
		container.getRequest().on('connectioninterrupted', this.onConnectionLoss, this);
		container.getRequest().on('connectionrestored', this.onConnectionRestore, this);
		container.getResponseRouter().on('receiveexception', this.onReceiveException, this);
		// We listen on the Ext.data.DataProxy object to listen in on all exception events
		Ext.data.DataProxy.on('exception', this.onException, this);

		// Enable tooltips
		Ext.QuickTips.init();
	},

	/**
	 * Event handler which is fired when the {@link Ext#getBody &lt;body&gt;} elements fires
	 * the 'contextmenu' event. If the element which fired the event doesn't have the
	 * 'zarafa-contextmenu-enabled' class and isn't a regular text input then the
	 * Browser contextmenu will be disabled.
	 * @param {Ext.EventObject} event The event object
	 * @param {Ext.Element} el The element on which the contextmenu was requested
	 * @private
	 */
	onBodyContextMenu: function(event, el)
	{
		el = Ext.get(el);

		// Don't disable the browser contextmenu when the
		// 'zarafa-contextmenu-enabled' CSS class is applied
		// on the element.
		if ( el.hasClass('zarafa-contextmenu-enabled') || el.up('div.zarafa-contextmenu-enabled')){
			return;
		}

		// Don't disable the browser contextmenu for regular
		// text inputs.
		if (el.dom.tagName.toUpperCase() === 'INPUT')
			return;

		// Disable contextmenu.
		event.preventDefault();
	},

	/**
	 * Event handler which is fired when the 'window' element fires the 'dragover' or
	 * the 'drop' event. This happens when the user drops a file over the webpage. On
	 * some UI fields this will provide a special action, but the browsers default is
	 * to open the file in the current page, which is not what we want.
	 * @param {Ext.EventObject} event The event object
	 * @private
	 */
	onWindowDragDrop: function(event)
	{
		event.stopPropagation();
		event.preventDefault();
		return false;
	},

	/**
	 * Event handler called when the {@link Ext.data.DataProxy} fired the
	 * {@link Ext.data.DataProxy#storeexception storeexception} event.
	 * This will check what type of exception it was ('response' or 'remote') and
	 * handle the exception accordingly.
	 *
	 * @param {Misc} misc See {@link Ext.data.DataProxy}#{@link Ext.data.DataProxy#exception exception}
	 * for description.
	 * @private
	 */
	onException: function(proxy, type, action, options, response, args)
	{
		var message;
		var detailsMessage = '';
		var notificationType = 'error.proxy';

		var title = _('Error');
		if (type === 'response') {
			// The error message can be in args when it is an Error object. This happens when the
			// processing of the response throws an Javascript Exception.
			if (Ext.isDefined(args) && args.error instanceof Error) {
				message = args.error.toString();
			} else {
				// When the exception has to do with the response itself we delegate this behavior to
				// onReceiveException
				this.onReceiveException(options, response);
				return;
			}
		} else if (response && response.error) {
			var errorObj = response.error;
			switch (errorObj.type) {
				case Zarafa.core.ErrorType['MAPI']:
				case Zarafa.core.ErrorType['ZARAFA']:
				case Zarafa.core.ErrorType['GENERAL']:
					message = errorObj.info.display_message;
					detailsMessage = errorObj.info.details_message || '';
					break;
				default:
					message = _('The server reported an unknown error on your request.');
					break;
			}
			if (!Ext.isEmpty(errorObj.info) && !Ext.isEmpty(errorObj.info.title)) {
				title = errorObj.info.title;
			}
		} else {
			message = _('The server reported an unspecified error on your request.');
		}

		if (Ext.get('loading')) {
			this.setErrorLoadingMask(title, message);
		} else {
			container.getNotifier().notify('error.proxy', title, message, {
				details_message : detailsMessage
			});
		}
		// If the meeting request hasn't been sent yet, and there was an error sending it,
		// it should be possible to save it afterwards anyway.
		// See web issue #113
		if (Ext.isDefined(args.sendRecords) && Ext.isArray(args.sendRecords)) {
			args.sendRecords.forEach(record => {
				if (typeof record.isMeetingSent === 'function' &&
				    !record?.isMeetingSent() &&
				    record?.hasMessageAction('send')) {
					record.deleteMessageAction('send');
				}
			});
		}
	},

	/**
	 * Called when the connection is being paralyzed and no further requests can be made
	 * to the server. Check if we should show a notification to the user about this, and
	 * ask if the user wishes to return to the logon page.
	 * @param {Zarafa.core.Request} request The request object
	 * @param {Zarafa.core.data.ParalyzeReason} reason The reason to paralyze grommunio Web
	 * @private
	 */
	onConnectionParalyze: function(request, reason)
	{
		var message = '';
		var logoutFn = Ext.emptyFn;

		switch (reason) {
			case Zarafa.core.data.ParalyzeReason.BROWSER_RELOADING:
			/* falls through */
			default:
				// No message for the user needed.
				return;
			case Zarafa.core.data.ParalyzeReason.SESSION_EXPIRED:
				message = _('The session has expired, reauthentication is required.');
				// When logging out, we preserve the username for convenience.
				logoutFn = container.logout.createDelegate(container, [ true ], false);
				break;
			case Zarafa.core.data.ParalyzeReason.SESSION_INVALID:
				message = _('The session in the current browser window has been closed from another browser window or tab.');
				// When logging out, we preserve the username for convenience,
				// but we won't close the session which was created in the other tab.
				logoutFn = container.logout.createDelegate(container, [ true, true ], false);
				break;
		}

		if (Ext.get('loading')) {
			this.setErrorLoadingMask(_('Error'), message);
		} else {
			Ext.MessageBox.show({
				title: _('Session expired'),
				msg: message + '<br>' + _('Do you wish to be redirected to the logon page?'),
				cls: Ext.MessageBox.ERROR_CLS,
				buttons: Ext.MessageBox.YESNO,
				fn: this.onConnectionParalyzeConfirmation,
				scope: this,
				logoutFn: logoutFn
			});
		}
	},

	/**
	 * Event handler for the {@link Ext.MessageBox MessageBox} which was opened by {@link #onConnectionParalyze}.
	 * This determines what the user has pressed, and if the user wishes to {@link Zarafa.core.Container#logout}
	 * immediately or not. If not, then a {@link Zarafa.core.ui.notifier.Notifier notification} will be shown
	 * to remind the user about the session.
	 * @param {String} button The button which was pressed by the user
	 * @param {String} id The id of the button which was clicked
	 * @param {Object} opt The options which was used to create the MessageBox.
	 * @private
	 */
	onConnectionParalyzeConfirmation: function(button, value, opt)
	{
		if (button === 'yes') {
			opt.logoutFn.call(this);
		} else {
			container.getNotifier().notify('error.connection', _('Session expired'), _('Reauthentication required, click here to go to back to logon page.'), {
				persistent: true,
				listeners: {
					click: opt.logoutFn
				}
			});
		}
	},

	/**
	 * Periodic function which is used to update the {@link Zarafa.core.ui.notification.Notifier notification}
	 * on the screen with the message that there is a connection problem, and the requests will be retried
	 * after the given timeout. This timeout counter will be updated every second.
	 * @param Zarafa.core.PingService} service The ping service handling the connection loss
	 * @param {Object} object Containing the information related to the connection loss
	 * @param {Number} timeout The number of seconds until the next retry to connect to the server
	 * @private
	 */
	onConnectionTimeupdate: function(service, object, timeout)
	{
		var request = container.getRequest();

		// Since we use defers, it is possible that the
		// connection was already restored. In that case,
		// we don't need to update the Notification message.
		if (!request.isInterrupted() || request.isParalyzed()) {
			return;
		}

		// Create the notification, we store the reference in connEl,
		// if it already exists, this will be an update action, otherwise
		// a new notification will be created.
		this.connEl = container.getNotifier().notify('error.connection', _('Connection problem'),
								 String.format(_('Could not connect to server, retrying in {0} second(s)'), timeout / 1000) + '<br />' + _('Click to retry now'), {
			persistent: true,
			update: !!this.connEl,
			reference: this.connEl,
			listeners: {
				// If the user clicks on the notification,
				// immediately retry to connect to to the server.
				click: service.retry,
				scope: service
			}
		});

		// Defer the function by 1 second, so we can update the retry counter.
		if (timeout > 1000) {
			this.connElTask.delay(1000, this.onConnectionTimeupdate, this, [ service, object, timeout - 1000 ]);
		}
	},

	/**
	 * Event handler for the {@link Zarafa.core.PingService#retry} event. This will
	 * cancel the currently scheduled call to {@link #onConnectionTimeupdate} and
	 * invoke it manually with the updated information.
	 * @param Zarafa.core.PingService} service The ping service handling the connection loss
	 * @param {Object} object Containing the information related to the connection loss
	 * @param {Number} timeout The number of seconds until the next retry to connect to the server
	 * @private
	 */
	onConnectionRetry: function(service, object, timeout)
	{
		// In case there was still a pending
		// update task, we interrupt that one.
		this.connElTask.cancel();

		// No we can update the notification
		this.onConnectionTimeupdate(service, object, timeout);
	},

	/**
	 * Event handler for the {@link Zarafa.core.Request#connectionloss} event. This will register the
	 * event handlers to the provided {@link Zarafa.core.PingService} which will give us the information
	 * about the next reconnect attempt.
	 * @param {Zarafa.core.Request} request The request object
	 * @param {Zarafa.core.PingService} service The ping service which is going to handle the connection loss
	 * @private
	 */
	onConnectionLoss: function(request, service)
	{
		this.connElTask = new Ext.util.DelayedTask(this.onConnectionTimeupdate, this);
		service.on('retry', this.onConnectionRetry, this);
	},

	/**
	 * Event handler for the {@link Zarafa.core.Request#connectionrestore} event. This will remove
	 * the error.connection {@link Zarafa.core.ui.notification.Notifier notification}
	 * and will show a info.connection.restore {@link Zarafa.core.ui.notification.Notifier notification}.
	 * @param {Zarafa.core.Request} request The request object
	 * @private
	 */
	onConnectionRestore: function(request)
	{
		if (this.connElTask) {
			this.connElTask.cancel();
			delete this.connElTask;
		}

		if (this.connEl) {
			container.getNotifier().notify('error.connection', null, null, {
				destroy: true,
				reference: this.connEl
			});
			container.getNotifier().notify('info.connection.restore', _('Connection restored'), _('Connection with server has been restored'));
			delete this.connEl;
		}
	},

	/**
	 * Event handler called when the PHP server returned an error
	 * in the root of the response. This indicates that the communication
	 * with the PHP server failed and the user should login again.
	 *
	 * @param {Object} requestdata The request data which was send to the server.
	 * @param {Object} xmlHttpRequest The raw browser response object
	 * @private
	 */
	onReceiveException: function(requestdata, xmlHttpRequest)
	{
		var loading = Ext.get('loading');
		var errorTitle;
		var errorMsg;

		if (xmlHttpRequest.status !== 200) {
			// # TRANSLATORS: Example: HTTP 404
			errorTitle = String.format(_('HTTP {0}'), xmlHttpRequest.status);
			errorMsg = xmlHttpRequest.statusText;
		} else {
			errorTitle = _('Error');
			errorMsg = _('Invalid data received from the server');
		}

		if (loading) {
			this.setErrorLoadingMask(errorTitle, errorMsg);
		} else {
			container.getNotifier().notify('error.json', errorTitle, errorMsg);
		}
	},

	/**
	 * Hide the loading mask which is shown before the {@link Ext.Viewport} is being rendered.
	 * The loadmask is marked with the element classname 'loading' and the background 'loading-mask'.
	 * @param {Function} callback An optional callback function that will be called when the the
	 * loading mask is completely hidden.
	 * @private
	 */
	hideLoadingMask: function(callback)
	{
		var loadingMask = Ext.get('loading-mask');

		if ( loadingMask ) {
			// Hide loading mask
			loadingMask.remove();
			if (Ext.isFunction(callback)) {
				callback();
			}
		}
	},

	/**
	 * Set an error text in the loading screen.
	 * @param {String} newError The title for the loading screen
	 * @param {String} newMessage The message for the loading screen
	 * @private
	 */
	setErrorLoadingMask: function(newTitle, newMessage)
	{
		var template = new Ext.Template('<div><b>{title}</b><br />{msg}</div>', { compiled: true, disableFormats: true });
		var message = Ext.get('loading-message');
		if (message) {
			message.dom.className = 'loading-error';
			template.overwrite(message, { title: newTitle, msg: newMessage });
		}
	},

	/**
	 * Validate the {@link Zarafa.hierarchy.data.HierarchyStore HierarchyStore} to determine
	 * if the {@link Zarafa.hierarchy.data.HierarchyStore#getDefaultStore Default Store} is present
	 * along with all the {@link Zarafa.hierarchy.data.HierarchyStore#getDefaultFolder Default Folders}.
	 * If there is a problem, this will show a {@link Zarafa.core.ui.notifier.Notifier Notification}
	 * indicating which store or folders are missing, and warning the user that not all functionality
	 * might be working as expected.
	 * @param {Zarafa.hierarchy.data.HierarchyStore} store The Hierarchy Store to validate
	 * @private
	 */
	validateHierarchy: function(store)
	{
		if (!store.getDefaultStore()) {
			container.getNotifier().notify('error.hierarchy.defaultfolder',
				_('Missing store'),
				_('The default store is missing from the hierarchy.') +
					'<br>' +
					_('Not all functionality of grommunio Web might be working properly because of this.'),
				{
					persistent: true,
					listeners: {
						'click': this.onHierarchyNotifierClick,
						'scope': this
					}
				}
			);
			return;
		}

		// The following default folders are required to be present
		// to be able to properly work with grommunio Web.
		var defaultFolders = [{
			type: 'inbox',
			name: pgettext('hierarchy.foldername', 'Inbox')
		},{
			type: 'outbox',
			name: pgettext('hierarchy.foldername', 'Outbox')
		},{
			type: 'sent',
			name: pgettext('hierarchy.foldername', 'Sent Items')
		},{
			type: 'wastebasket',
			name: pgettext('hierarchy.foldername', 'Deleted items')
		},{
			type: 'calendar',
			name: pgettext('hierarchy.foldername', 'Calendar')
		},{
			type: 'contact',
			name: pgettext('hierarchy.foldername', 'Contacts')
		},{
			type: 'drafts',
			name: pgettext('hierarchy.foldername', 'Drafts')
		},{
			type: 'journal',
			name: pgettext('hierarchy.foldername', 'Journal')
		},{
			type: 'note',
			name: pgettext('hierarchy.foldername', 'Notes')
		},{
			type: 'task',
			name: pgettext('hierarchy.foldername', 'Tasks')
		},{
			type: 'junk',
			name: pgettext('hierarchy.foldername', 'Junk Email')
		}];

		var missing = [];

		// Go over all default folders which we expect to be present,
		// if any of them is missing we update the 'missing' array
		// and will show them in a notification box.
		for (var i = 0, len = defaultFolders.length; i < len; i++) {
			var folder = defaultFolders[i];

			if (!store.getDefaultFolder(folder.type)) {
				missing.push(folder.name);
			}
		}

		// If any of the folders is missing, show a notification to the
		// user to inform him of the missing folders.
		if (!Ext.isEmpty(missing)) {
			// Construct an HTML list of the missing folders
			var list = '<ul><li>' + missing.join('</li><li>') + '</li></ul>';

			container.getNotifier().notify('error.hierarchy.defaultfolder',
				_('Missing folders'),
				String.format(
					ngettext('The following required folder is missing in the hierarchy: {0}',
						 'The following required folders are missing in the hierarchy: {0}', missing.length), list) +
					_('Not all functionality of grommunio Web might be working properly because of this.'),
				{
					persistent: true,
					listeners: {
						'click': this.onHierarchyNotifierClick,
						'scope': this
					}
				}
			);
		}
	},

	/**
	 * Event handler which is fired when the user clicked on the {@link Zarafa.core.ui.notifier.Notifier notification}.
	 * This will remove the notification.
	 * @param {Ext.Element} element The notification element
	 * @param {Ext.EventObject} event The event object
	 * @private
	 */
	onHierarchyNotifierClick: function(element, event)
	{
		container.getNotifier().notify('error.hierarchy.defaultfolder', null, null, {
			reference: element,
			destroy: true
		});
	},

	/**
	 * Event handler called when load is received in hierarchy. This will {@link #hideLoadingMask hide the loadmask}.
	 * @param {Zarafa.hierarchy.data.HierarchyStore} store The store which was loaded
	 * @param {Ext.data.Record[]} records The records which were loaded by the store
	 * @param {Object} options The options which were originally passed to {@link Ext.data.Store#load}.
	 * @private
	 */
	onHierarchyLoad: function(store, records, options)
	{
		if (!Ext.isEmpty(records)) {
			// We have the hierarchy, load the entire UI
			container.getMainPanel();

			// validate the hierarchy, if the hierarchy is not valid a warning
			// notification box will be shown to the user informing him of incompatibilities.
			this.validateHierarchy(store);

			// Remove loading mask, we might still be busy loading the data for the
			// store of the user, but the context will have its own loadmask for that.
			// The user is at least allowed to see the Hierarchy and press buttons.
			this.hideLoadingMask(function(){
				container.fireEvent('webapploaded');

				// Notify that grommunio Web UI is loaded.
				Zarafa.fireUIReady();
			});

			// Remove resize event listener of loading page
			window.removeEventListener('resize', resizeLoginBox);

			// Register webapp to handle mailto urls
			this.registerMailto();

			// Process data that was passed as URL data
			Zarafa.core.URLActionMgr.execute(urlActionData);
			delete urlActionData;

			// Start the keepalive to make sure we stay logged into the zarafa-server,
			// the keepalive is also used to get notifications back to the client.
			store.startKeepAlive();
		} else {
			this.setErrorLoadingMask(_('Error'), _('Loading model from server failed'));
		}
	},

	/**
	 * Function will register webapp as default client to handle mailto urls
	 * Ideally we should only register this handler if already not registered
	 * but browsers support is not proper for that, so we are always registering it
	 * in chrome this is not a problem, in firefox it will show bar that handler is already registered
	 * but that is a bug and already fixed in trunk version https://bugzilla.mozilla.org/show_bug.cgi?id=912347
	 * Support for isProtocolHandlerRegistered is also limited https://bugzilla.mozilla.org/show_bug.cgi?id=440620
	 */
	registerMailto: function()
	{
		var navigator = window.navigator;

		if (Ext.isFunction(navigator.registerProtocolHandler)) {
			// Register webapp handler for 'mailto' protocol in browsers.
			var url = container.getBaseURL() + '?action=mailto&to=%s';

			// Check if we have already registered for this protocol
			var register = true;
			if (Ext.isFunction(navigator.isProtocolHandlerRegistered)) {
				register = !navigator.isProtocolHandlerRegistered('mailto', url);
			}

			// Register if required
			if (register) {
				navigator.registerProtocolHandler('mailto', url, 'grommunio Web');
			}
		}
	},

	/**
	 * Load the default Webclient into the browser. This will initialize the environment,
	 * load the {@link Zarafa.hierarchy.data.HierarchyStore} and open the
	 * {@link Zarafa.core.ui.MainViewport MainViewport}.
	 */
	loadWebclient: function()
	{
		// Initialize globals & environment
		Zarafa.initializeGlobals();
		Zarafa.initializeEnvironment();

		// Recolor the icons if necessary
		this.recolorIcons();

		// Start loading all plugins
		Zarafa.fireReady();

		// We need to register the event handler for outOfOfficeStore on load to check if user is out of office
		// and ask them if they want to switch it off.
		var oofStore = container.getOutOfOfficeStore();
		oofStore.on('load', this.onOofStoreLoad, this, { single: true });

		// Initialize context - check if there is one selected in settings
		this.initContext();

		// Start polling the server to get reminders data back to client
		container.getReminderStore().initializeReminderInterval();

		// Setup the hierarchy, this will complete the initialization
		// and allow the Main Viewport to be opened
		var hierarchyStore = container.getHierarchyStore();

		// For the hierarchy we only need to register the event handler once,
		// as only during the first time we have to remove the load mask.
		hierarchyStore.on('load', this.onHierarchyLoad, this, { single: true });

		// Load the folder hierarchy
		hierarchyStore.load();
		// When a client timeout has been defined, we will start keeping track
		// of idle time.
		var server = container.getServerConfig();
		var clientTimeout = server.getClientTimeout();
		if (clientTimeout){
			this.startIdleTimeChecker(clientTimeout);
		}

		// Starts shared stores unread email poller.
		this.startSharedStoresHierarchyChecker();
	},

	/**
	 * Starts the checking of idle time.
	 * This function uses original javascript events because we cannot set
	 * the useCapture property with ExtJS events.
	 * See https://developer.mozilla.org/en/docs/Web/API/EventTarget.addEventListener
	 *
	 * @param {Number} clientTimeout The timeout time in seconds.
	 * @private
	 */
	startIdleTimeChecker: function(clientTimeout)
	{
		if ( !document.addEventListener ) {
			// User is using a browser that does not support addEventListener.
			// Probably IE<9 which we don't support.
			// However there is no reason to create errors for IE<9
			// Client timeout will still be handled by the backend though,
			// but the message will only be shown to the user when he tries to
			// connect to the backend after the session has timed out.
			return;
		}
		var me = this;

		document.addEventListener('click', function(){
			me.idleTime = 0;
		}, true);
		document.addEventListener('mousemove', function(){
			me.idleTime = 0;
		}, true);
		document.addEventListener('keydown', function(){
			me.idleTime = 0;
		}, true);

		var hierarchyStore = container.getHierarchyStore();

		// Start an interval for increasing the idle time
		Ext.TaskMgr.start.createDelegate(this, [{
			run: function(){
				// Add 5 seconds to the idle time counter
				this.idleTime += 5;
				if ( this.idleTime > clientTimeout ){
					hierarchyStore.sendDestroySession();
				}
			},
			scope: this,
			interval: 5000 //Run every 5 seconds
		}]).defer(5000); // Start after 5 seconds

		// Start an interval for sending keepalive requests
		// We need this keepalive to keep the connection alive if the user has made an
		// action in the grommunio Web without connecting to the server. (like mousemove, click, keydown)
		// Subtracting 5 seconds to account for latency
		var interval = (clientTimeout-5)*1000;
		if ( interval < 5000 ){
			// This one is especially for Sean who was so smart to set a client timeout of 5 seconds
			// causing keepalive requests to be sent constantly and thereby ddos'ing his own server :-)
			// Let's never send keepalive requests with an interval lower than 5 seconds.
			// Anyone who sets a timeout this low deserves to be logged out! (and punished severely)
			interval = 5000;
		}

		Ext.TaskMgr.start.createDelegate(this, [{
			run: function(){
				hierarchyStore.sendKeepAlive();
			},
			scope: this,
			interval: interval
		}]).defer(interval); //Start sending keepalives after interval milliseconds.
	},

	/**
	 * Start the Shared Stores unread mail poller, fetches the hierarchy
	 * once to fill the server side cache.
	 * @private
	 */
	startSharedStoresHierarchyChecker: function()
	{
		const interval = container.getServerConfig().getSharedStorePollingInterval();

		if (!Ext.isNumber(interval)) {
			return;
		}

		// Fetch shared stores state on login.
		container.getHierarchyStore().sendSharedStoreHierarchyUpdate();

		setInterval(function() {
			container.getHierarchyStore().sendSharedStoreHierarchyUpdate();
		}, interval);
	},

	/**
	 * init UI by lazily constructing the main panel (implicit in container.getMainPanel) and
	 * setting the default context to visible. Note that during onHierarchyLoad we will also
	 * open the default folder for that specific context to ensure the user can see all from
	 * the folder.
	 * @private
	 */
	initContext: function()
	{
		var defaultContext = container.getSettingsModel().get('zarafa/v1/main/default_context');
		var plugin = container.getContextByName(defaultContext);
		if (!plugin) {
			// If the defaultContext has an invalid name,
			// we will default to the 'mail' context
			plugin = container.getContextByName('mail');
		}
		container.switchContext(plugin);
	},

	/**
	 * When an iconset has SVG icons and it has defined the primary-color and/or
	 * secondary-color properties and the active theme has defined the
	 * icons-primary-color and/or icons-secondary-color property this function
	 * will rewrite the css rules of the iconset to update the colors of the icons.
	 */
	recolorIcons: function()
	{
		// Get the properties defined by the active iconset
		var serverConfig = container.getServerConfig();
		var iconsets = serverConfig.getIconsets();
		var activeIconsetName = serverConfig.getActiveIconset();
		var activeIconset = iconsets[activeIconsetName];

		// If the iconset did not define a primary or secondary color, we cannot
		// redefine the colors, so return without doing anything
		if ( !activeIconset['primary-color'] && !activeIconset['secondary-color'] ) {
			return;
		}

		// Get the primary and secondary icon color defined in the active theme
		// (themes can override the colors of iconsets that allow recoloring)
		var themeIconsPrimaryColor = serverConfig.getPrimaryIconColor();
		var themeIconsSecondaryColor = serverConfig.getSecondaryIconColor();

		// If the active theme did not define a new primary or secondary color
		// for the icons there is nothing to do, so return
		if ( !themeIconsPrimaryColor && !themeIconsSecondaryColor ) {
			return;
		}

		// Get the stylesheet element that contains the icons as background images
		// and check all rules in it for SVG icons we can recolor
		var sheet = document.getElementById('grommunio-iconset-stylesheet');
		for ( var i=0; i<sheet.sheet.cssRules.length; i++ ) {
			var rule = sheet.sheet.cssRules[i];

			// Check the rule to see if it contains an SVG background image
			// (we can only recolor SVG icons)
			var matches = (/url\("data:image\/svg\+xml;base64,(.+?)"\)/).exec(rule.cssText);
			if ( matches !== null ) {
				// base64 decode the SVG image
				var svg = atob(matches[1]);

				// Simply replace the color codes
				var svgRecolored = svg.replace(new RegExp(activeIconset['primary-color'], 'gi'), themeIconsPrimaryColor).replace(new RegExp(activeIconset['secondary-color'], 'gi'), themeIconsSecondaryColor);

				// If we changed anything, replace the CSS rule to use the base64 encoded SVG
				// with the new color(s)
				if ( svg !== svgRecolored ) {
					var cssText = rule.cssText.replace(
						/url\("data:image\/svg\+xml;base64,(.+)"\)/,
						'url("data:image/svg+xml;base64,' + btoa(svgRecolored) + '")'
					);
					sheet.sheet.deleteRule(i);
					sheet.sheet.insertRule(cssText, i);
				}
			}
		}
	},

	/**
	 * Event handler called when load is received in outofoffice store.
	 * This will check if user is out of office If so, ask them if they want to switch OOF off.
	 *
	 * @param {Zarafa.common.outofoffice.data.OofStore} store The store which was loaded
	 * @param {Zarafa.common.outofoffice.data.OofRecord} records The records which were loaded by the store
	 * @param {Object} options The options which were originally passed to {@link Ext.data.Store#load}.
	 * @private
	 */
	onOofStoreLoad: function(store, records, options)
	{
		var oof = false;

		var loginUserEntryId = container.getUser().getEntryId();
		var oofUserSettings;

		// If logged in user is out of office then only this will give the user's out of office settings information.
		for (var i=0; i< records.length; i++) {
			if (Zarafa.core.EntryId.compareEntryIds(records[i].get('entryid'), loginUserEntryId)) {
				oofUserSettings = records[i];
				break;
			}
		}

		if (oofUserSettings) {
			var oofFrom = oofUserSettings.get('from');
			var oofUntil = oofUserSettings.get('until');
			var isOofSet = oofUserSettings.get('set');
			var date = new Date().getTime()/1000;

			if (isOofSet) {
				// Check if current date fall within the time span of OOF start-date and end-date, if configured.
				if (oofFrom <= date) {
					// Check if end-date is configured, no need to check otherwise
					if(oofUntil === 0 || oofUntil > date) {
						oof = true;
					} else {
						// Current date falls out of the configured time span, so disable the OOF
						oofUserSettings.set('set', false);
						store.save();
					}
				}
			}
		}

		if (oof) {
			Ext.MessageBox.show({
				title: _('Out of Office'),
				msg: _('Out of Office is currently activated. Would you like to turn it off?'),
				buttons: Ext.MessageBox.YESNO,
				fn: this.onOofConfirm,
				scope: oofUserSettings
			});
		}
	},

	/**
	 * Handler for the out of office confirmation box
	 * If the user pressed Yes/Ok, then disable out of office
	 * @param {String} id of the button that was clicked
	 * @private
	 */
	onOofConfirm: function(button)
	{
		if (button === 'yes') {
			this.set('set', false);
			this.save();
		}
	},

	/**
	 * Load the Welcome message for new users into the browser. This will initialize
	 * the environment and open the {@link Zarafa.core.ui.WelcomeViewport WelcomeViewport}.
	 */
	loadWelcome: function()
	{
		// Setup globals & environment
		this.initializeGlobals();
		this.initializeEnvironment();

		// Start loading all plugins
		this.fireReady();

		// Load the welcome view
		container.getWelcomePanel();
		this.hideLoadingMask();
	},

	//
	// Generic Utility functions, should probably be moved elsewhere
	//

	/**
	 * This will resize a canvas {@link Ext.Element element}.
	 * Canvas resizing is more tricky then one would expect. For canvas 2 settings are
	 * important: the CSS and attribute dimensions.
	 *
	 * As one would expect, the CSS dimensions, given through a CSS file, or
	 * 'style' attribute (controlled by the functions {@link Ext.Element#setWidth setWidth}
	 * and {@link Ext.Element#setHeight setHeight}) controls how big the element itself
	 * is. It however does not mean that the drawing area is of the same size. The drawing
	 * area is controlled by the element attributes 'width' and 'height'.
	 *
	 * Now for the fun part, if you use CSS to size of the element to 1000x1000px,
	 * thus the element looks like:
	 *   <canvas style="width=1000px; height=1000px;">
	 * and set the attributes to 50x50px, making the complete element look like:
	 *   <canvas style="width=1000px; height=1000px;" width="50" height="50">
	 * you can then draw anything you like on the canvas, but only the first
	 * 50x50px of the canvas will be resized to the entire element size. Making
	 * your cute little drawing of a giraffe completely stretched.
	 *
	 * @param {Ext.Element} canvas The canvas element which must be resized.
	 * @param {Number} width The desired width of the canvas element
	 * @param {Number} height The desired height of the canvas element
	 */
	resizeCanvas : function(canvas, width, height)
	{
		canvas.setWidth(canvas.dom.width = width);
		canvas.setHeight(canvas.dom.height = height);
	},

	/**
	 * Generate a random string.
	 * @param {Number} len Length of the string
	 * @return {String} Random string
	 */
	generateId: function(len)
	{
		var text = "";
		var possible = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789";

		for( var i = 0; i < len; i++ ) {
			text += possible.charAt(Math.floor(Math.random() * possible.length));
		}

		return text;
	},

	/**
	 * Determine if separate window popout is supported.
	 * Support is based on browser's ability to render any element into another browser window.
	 *
	 * @return {Boolean} True if popout is supported, false otherwise
	 */
	supportsPopOut: function()
	{
		// Currently, we do not support the popout in case of IE/Edge.
		return (!(Ext.isIE || Ext.isEdge || navigator.userAgent.includes('grommunio')));
	}
});
Ext.namespace('Zarafa.core');

/**
 * @class Zarafa.core.Actions
 * List of valid action types. Action types are used for identifying request and response types for modules.
 * For instance, to request a mail from the server the client performs an 'open' action on the 'previewreadmailitemmodule'.
 * The server then responds with an 'item' action, containing the email data.
 * @singleton
 */
Zarafa.core.Actions =
{
	/**
	 * The list action retrieves a list of items such as mails, tasks, etc. The server then responds with a list action containing
	 * a list of items.
	 * @property
	 * @type String
	 */
	list: "list",

	/**
	 * Retrieves a list of entries in the global address book
	 * @property
	 * @type String
	 */
	globaladdressbook: "globaladdressbook",

	/**
	 * Retrieves the hierarchy, a set of stores and folders within those stores. Used for updating the
	 * hierarchy tree.
	 * @property
	 * @type String
	 */
	hierarchy: "hierarchy",

	/**
	 * Open an item, usually a mail item. The server responds with an item action.
	 * @property
	 * @type String
	 */
	open: 'open',

	/**
	 * Can mean different things in different contexts. Can be used to request a single item from the server, and is returned by the
	 * server to return the contents on a single item.
	 * @property
	 * @type String
	 */
	item: 'item',

	/**
	 * Update item(s)
	 * @property
	 * @type String
	 */
	update: 'update',

	/**
	 * Save an item.
	 * @property
	 * @type String
	 */
	save: 'save',

	/**
	 * Copy an item.
	 * @property
	 * @type String
	 */
	copy: 'copy',

	/**
	 * Delete an item.
	 * @property
	 * @type String
	 */
	'delete': 'delete',

	/**
	 * Gets folder details.
	 * @property
	 * @type String
	 */
	folder: 'folder',

	/**
	 * Used for setting properties.
	 * @property
	 * @type String
	 */
	set: 'set',

	/**
	 * Used for getting properties.
	 * @property
	 * @type String
	 */
	get: 'get',

	/**
	 * Used for reset properties.
	 * @property
	 * @type String
	 */
	reset: 'reset',

	/**
	 * Used for deleting properties/items.
	 * @property
	 * @type String
	 */
	_delete: 'delete',

	/**
	 * Used for searching on a folder.
	 * @property
	 * @type String
	 */
	search: 'search',

	/**
	 * Used for incremental search on folder.
	 * @property
	 * @type String
	 */
	updatesearch: 'updatesearch',

	/**
	 * Used for live scroll.
	 * @property
	 * @type String
	 */
	updatelist: 'updatelist',

	/**
	 * Used for stopping search on folder.
	 * @property
	 * @type String
	 */
	stopsearch: 'stopsearch',

	/**
	 * Used for requesting contacts from addressbook
	 * @property
	 * @type String
	 */
	contacts: 'contacts',

	/**
	 * Used to send a keepalive to the server
	 * @property
	 * @type String
	 */
	keepalive: 'keepalive',

	/**
	 * Used to send a request to destroy the session to the server
	 * @property
	 * @type String
	 */
	destroysession: 'destroysession',

	/**
	 * Used when receiving update from server indicating there is new mail
	 * @property
	 * @type String
	 */
	newmail: 'newmail',

	/**
	 * Used for creating new folder
	 * @property
	 * @type String
	 */
	addFolder: 'add',

	/**
	 * Used for renaming folder in tree
	 * @property
	 * @type String
	 */
	modifyFolder: 'modify',

	/**
	 * Used for deleting folder from tree
	 * @property
	 * @type String
	 */
	deleteFolder: 'delete',

	/**
	 * Used on Deleted Items to empty the folder
	 * @property
	 * @type String
	 */
	emptyFolder: 'emptyfolder',

	/**
	 * Used on folders to mark all messages as 'read'
	 * @property
	 * @type String
	 */
	readAllMsgs: 'readflags',
	/**
	 * Used in {@link Zarafa.hierarchy.dialogs.FolderPropertiesContentPanel FolderPropertiesContentPanel} show/update folder props
	 * @property
	 * @type String
	 */
	folderProps: 'folderprops',
	/**
	 * Used in {@link Zarafa.core.data.IPMRecipientStoreCheckNamesProxy IPMRecipientStoreCheckNamesProxy} for resolve requests
	 * @property
	 * @type String
	 */
	checknames: 'checknames',

	/**
	 * Used in {@link Zarafa.core.data.IPMExpandDistlistProxy IPMExpandDistlistProxy} for expand requests
	 * @property
	 * @type String
	 */
	expand: 'expand',

	/**
	 * Used in {@link Zarafa.core.data.IPMAttachmentProxy IPMAttachmentProxy} for uploading attachments.
	 * @property
	 * @type String
	 */
	upload: 'upload',

	/**
	 * Used for importing attached item into folder
	 * @property
	 * @type String
	 */
	import: 'import',

	/**
	 * Used for shared stores unread mail updates
	 * @property
	 * @type String
	 */
	sharedstoreupdate: 'sharedstoreupdate',

	/**
	 * Used for ensuring license for supported products.
	 * @property
	 * @type String
	 */
	ensure: 'ensure'
};
Ext.namespace('Zarafa.core');

/**
 * @class Zarafa.core.BrowserWindowMgr
 * @extends Ext.util.Observable
 *
 * The global Browser-window manager that will keep track of all the available browser windows with
 * a unique key assigned to each of the browser window as its name.
 * @singleton
 */
Zarafa.core.BrowserWindowMgr = Ext.extend(Ext.util.Observable, {
	/**
	 * The list of registered browser window. It contains the list with the window DOM object bound to a
	 * unique key.
	 * @property
	 * @type Ext.util.MixedCollection
	 * @private
	 */
	browserWindows: undefined,

	/**
	 * The key of registered browser window which is active currently.
	 * @property
	 * @type String
	 * @private
	 */
	activeBrowserWindow: undefined,

	/**
	 * The list of registered browser window with the object which contain,
	 * component The constructor of the component which has to be created in the container layer and
	 * config which must be passed to the constructor when creating the component.
	 * @property
	 * @type Ext.util.MixedCollection
	 * @private
	 */
	browserWindowComponents: undefined,

	/**
	 * Denotes that the multiple popups are blocked by the browser or not.
	 * @property
	 * @type Boolean
	 * @private
	 */
	isPopupsBlocked: false,

	/**
	 * An array which contain, component The constructor of the component which has to be created in the container layer and
	 * config which must be passed to the constructor when creating the component,
	 * for the popup window which was blocked by the browser.
	 * @property
	 * @type Array
	 * @private
	 */
	blockedPopupsContent: [],

	/**
	 * @constructor
	 */
	constructor: function()
	{
		this.browserWindows = new Ext.util.MixedCollection();

		// It is required to manage multiple browser windows when first separate window is created by user.
		// So, we must have to register the main webapp-browser-window to the BrowserWindowMgr.
		window.name = 'mainBrowserWindow';
		this.register(window);
		window.addEventListener("focus", function(event) {
			Zarafa.core.BrowserWindowMgr.setActive('mainBrowserWindow');
		}, false);
		this.browserWindowComponents = new Ext.util.MixedCollection();

		this.addEvents('separatewindowresize');
	},

	/**
	 * Register component and config which will use to re create separate window ui on reload or refresh
	 * @param {String} uniqueWindowName unique name of browser window
	 * @param {Function} component The constructor of the component which has to be created in the container layer.
	 * @param {Object} config The configuration object which must be
	 */
	initComponent: function (uniqueWindowName, component, config)
	{
		this.browserWindowComponents.add(uniqueWindowName, {
			component: component,
			config: config
		});
	},


	/**
	 * Helper function which registers necessary events. This happens mostly when anew browser window
	 * gets loaded for the first time.
	 *
	 * @param {Object} browserWindowObject The newly created window object which must be registered.
	 * @param {Ext.Component} componentInstance component which is required to be rendered into the separate browser window.
	 * @param {Ext.Container} mainContainer The container which is the parent most container of separate window.
	 * @private
	 */
	initEvents: function(browserWindowObject, componentInstance, mainContainer)
	{
		// Initialize ext doc classes on separate window.
		this.initExtCss(browserWindowObject);

		componentInstance.on('close', this.onSeparateWindowClose.createDelegate(this, [ browserWindowObject ]));
		componentInstance.on('userupdaterecord', this.onComponentUserupdateRecord.createDelegate(this, [ componentInstance ], true));

		Ext.EventManager.on(
			browserWindowObject,
			"resize",
			this.onSeparateWindowResize.createDelegate(this, [ browserWindowObject, mainContainer ]),
			browserWindowObject
		);


		// Initialize events which are use to handle drag and drop in separate window.
		Ext.dd.DragDropMgr.initEvents(browserWindowObject);

		browserWindowObject.addEventListener("focus", this.onSeparateWindowFocus.createDelegate(this, [ browserWindowObject.name ]));
		browserWindowObject.addEventListener("unload", this.onSeparateWindowUnload.createDelegate(this, [ browserWindowObject, componentInstance, mainContainer ]));

		// Disable contextmenu globally in the separate browser window.
		Ext.getBody().on('contextmenu', this.onBodyContextMenu, this);

		browserWindowObject.onkeydown = this.preventWindowReload.createDelegate(this);

		// Check component instance has before unload event handler if yes then
		// Register event handler for separate window onbeforeunload event
		// which is use to show a confirmation dialog warning if window record has any unsaved changes
		if (Ext.isFunction(componentInstance.onBeforeUnload)) {
			browserWindowObject.onbeforeunload = componentInstance.onBeforeUnload.createDelegate(componentInstance);
		}
	},

	/**
	 * Function which used to prevent to reload popout window by F5 and Ctrl + R keys.
	 * @param {Object} event The event object
	 */
	preventWindowReload: function (event)
	{
		switch (event.keyCode) {
			case 116: //F5 button
				event.returnValue = false;
				event.keyCode = 0;
				return false;
			case 82: //R button
				if (event.ctrlKey) {
					event.returnValue = false;
					event.keyCode = 0;
					return false;
				}
		}
	},

	/**
	 * Event handler for the {@link Zarafa.core.plugins.RecordComponentPlugin#userupdaterecord userupdaterecord} event
	 * on the {@link #field}. This will relay the value of {@link Zarafa.core.plugins.RecordComponentPlugin#isChangedByUser} config option.
	 * to the newly created content panel into popout window.
	 * It is required to persist the user change state into RecordComponentPlugin for this particular field.
	 * @param {Ext.Component} field The component which fired the event
	 * @param {Zarafa.core.data.MAPIRecord} record The record which was updated
	 * @param {Boolean} isChangedByUser Indicates if the record has been changed by the user since it has been loaded.
	 * @param {Ext.Component} componentInstance component which gets rendered into the separate browser window.
	 * @private
	 */
	onComponentUserupdateRecord: function(field, record, isChangedByUser, componentInstance)
	{
		// Check the record has unsaved user changes before popout
		// If true then update the record component plugin value
		if (componentInstance.isRecordChangeByUser) {
			field.recordComponentPlugin.isChangedByUser = componentInstance.isRecordChangeByUser;
		}

		// If record get saved then add record id
		if (!record.phantom) {
			var windowName = this.getOwnerWindow(componentInstance).name;
			var windowObject = this.browserWindows.get(windowName);
			if (Ext.isEmpty(windowObject.entryid)) {
				windowObject.entryid = record.get("entryid");
			}
		}
	},

	/**
	 * Register a newly created browser-window object with the {@link Zarafa.core.BrowserWindowMgr BrowserWindowMgr}.
	 * @param {Object} browserWindowObject The newly created window object which must be registered.
	 * @param {Function} component The constructor of the component which has to be created in the container layer.
	 * @param {Object} config The configuration object which must be

	 */
	register: function(browserWindowObject, component, config)
	{
		var uniqueWindowName = browserWindowObject.name;
		if (uniqueWindowName != 'mainBrowserWindow') {
			this.initComponent(uniqueWindowName, component, config);

			// add entryid along with the window object to identify browser window of particular record.
			var entryid = this.getWindowId(config.record);
			if (!Ext.isEmpty(entryid)) {
				browserWindowObject.entryid = entryid;
			}
		}
		this.browserWindows.add(uniqueWindowName, browserWindowObject);
		this.setActive(uniqueWindowName);
	},

	/**
	 * De-register an already registered browser-window object from the {@link Zarafa.core.BrowserWindowMgr BrowserWindowMgr}.
	 * @param {String} uniqueWindowName The unique name of the browser window.
	 */
	unRegister: function(uniqueWindowName)
	{
		this.browserWindows.removeKey(uniqueWindowName);
		Ext.MessageBox.removeBrowserWindowMessageBox(uniqueWindowName);
		// As the window which was closed is also the active one, we are not able to decide which window should be considered
		// as an active one. So assign 'undefined' to the activeBrowserWindow which will later on replaced while other
		// available window receive focus.
		if(this.activeBrowserWindow === uniqueWindowName) {
			this.activeBrowserWindow = undefined;
		}
	},

	/**
	 * A helper function which creates main container into the separate window and load the required
	 * component within. Some necessary events needs to be registered as well.
	 * @param {Object} separateWindowInstance Browser window object.
	 * @protected
	 */
	createUI: function(separateWindowInstance)
	{
		// Enable tooltips
		Ext.QuickTips.init();

		var separateWindowId = separateWindowInstance.name;

		//The constructor of the component which has to be created in the container layer.
		var browserWindowComponent;

		if (this.browserWindowComponents.containsKey(separateWindowId)) {
			browserWindowComponent = this.browserWindowComponents.get(separateWindowId);
		} else if (!Ext.isEmpty(this.blockedPopupsContent)) {
			// No inner components found for this particular popup window.
			// This is the situation where some popups were blocked by browser and
			// user manually allows to load the same.
			var blockedPopup = this.blockedPopupsContent.pop();
			this.register(separateWindowInstance, blockedPopup.component, blockedPopup.config);
			browserWindowComponent = this.browserWindowComponents.get(separateWindowId);
			this.isPopupsBlocked = false;
		}

		var component = browserWindowComponent.component;

		//The configuration object
		var config = {
			plugins: [ 'zarafa.contentlayerplugin' ],
			confirmClose: false
		};
		config = Ext.applyIf(config,browserWindowComponent.config);

		// Create instance of the component which is required to be rendered into the separate browser window.
		var componentInstance = new component(config);

		var mainContainer = Ext.create({
			xtype: 'panel',
			height: separateWindowInstance.innerHeight,
			width: separateWindowInstance.innerWidth,
			renderTo: Ext.get(separateWindowInstance.document.body),
			layout: 'fit',
			items: [componentInstance]
		});

		// Check the record has unsaved user changes before popout

		// If true then update the record component plugin value
		if(config.isRecordChangeByUser) {
			componentInstance.recordComponentPlugin.isChangedByUser = config.isRecordChangeByUser;
		}
		this.initEvents(separateWindowInstance, componentInstance, mainContainer);

		// We need to use some delay due to the behavioral difference between various browsers.
		// Let say we are creating 5 popups, Chrome blocks the last popups, and FF blocks the first one.
		// When first popup gets rendered and following popups will be blocked than there is no way to
		// listen to any event of already-rendered popup and display warning message in already rendered popup.
		var task = new Ext.util.DelayedTask(Zarafa.core.BrowserWindowMgr.displayBlockedPopupWarning.createDelegate(this, [separateWindowInstance], this));
		task.delay(200);
	},

	/**

	/**
	 * Provides browser window object mapped with the unique name currently assigned in {@link #activeBrowserWindow}.
	 * @return {Object} Browser window object
	 */
	getActive: function()
	{
		// Some time while child window not available any more and main window focus not set
		// at that time activeBrowser window will not available.
		// So handle this type of situation by returning main browser window while active browser window not available

		return this.browserWindows.get(this.activeBrowserWindow) || this.browserWindows.get('mainBrowserWindow');
	},

	/**
	 * Assign the provided window name to {@link #activeBrowserWindow}.
	 * @param {String} uniqueWindowName The unique name of the browser window.
	 */
	setActive: function(uniqueWindowName)
	{
		this.activeBrowserWindow = uniqueWindowName;

		// Activate respective QuickTip which is associated with the active browser window.
		if(Ext.QuickTips && Ext.QuickTips.tip) {
			var activeQuickTip = Ext.QuickTips.browserQuickTips.get(uniqueWindowName);
			if(Ext.isDefined(activeQuickTip)) {
				Ext.QuickTips.tip = activeQuickTip;
			}
		}

		// Activate respective MessageBox which is associated with the active browser window.
		Ext.MessageBox.setActiveWindowMessageBox(uniqueWindowName);
	},

	/**
	 * Determines if the webapp-main-window is currently active or not by checking that the value currently
	 * assigned into {@link #activeBrowserWindow} is 'mainBrowserWindow'.
	 * @return {Boolean} True if the active window is webapp-main-window, false otherwise.
	 */
	isMainWindowActive: function()
	{
		return this.activeBrowserWindow === 'mainBrowserWindow';
	},

	/**
	 * Helper function which allows to know which browser window object is the owner of any particular component
	 * passed as argument.
	 * @param {Ext.Component} component of which we want to get the owning browser window.
	 *
	 * @return {Boolean} true if the owner is main webapp window, false otherwise
	 */
	isOwnedByMainWindow: function(component)
	{
		var ownerWindow = this.getOwnerWindow(component);
		return ownerWindow ? ownerWindow.name === 'mainBrowserWindow' : false;
	},

	/**
	 * Helper function which returned the owning window of the element passed as argument.
	 * @param {Ext.Component/Ext.Element} component/element of which we want to get the owning browser window.
	 *
	 * @return {Object} Browser window object
	 * @private
	 */
	getOwnerWindow: function(component)
	{
		if(!Ext.isDefined(component)) {
			return undefined;
		}

		// In case we receive Ext.Component as parameter then we have to get underlying Ext.Element
		if(Ext.isFunction(component.getEl)) {
			component = component.getEl();
		}

		var componentDom = component.dom ? component.dom : component;
		var ownerDocument = componentDom ? componentDom.ownerDocument : undefined;
		var defaultView = ownerDocument ? ownerDocument.defaultView : undefined;
		return defaultView ? defaultView : this.browserWindows.get('mainBrowserWindow');
	},

	/**
	 * Event handler which is raised when browser window receives user focus.
	 * This will make respective browser window active into {@link Zarafa.core.BrowserWindowMgr BrowserWindowMgr}.
	 *
	 * @param {Object} browserWindowId The newly created window object id which must be registered.
	 * @private
	 */
	onSeparateWindowFocus: function(browserWindowId)
	{
		this.setActive(browserWindowId);
	},

	/**
	 * Event handler which is raised when browser window gets closed.
	 * This will de-register the closed component instance from {@link Zarafa.core.data.ContentPanelMgr ContentPanelMgr}.
	 * This will remove all the child elements of the parent most container of separate window.
	 * This will de-register the closed window instance from {@link Zarafa.core.BrowserWindowMgr BrowserWindowMgr}.
	 *
	 * @param {Object} browserWindowObject The newly created window object which must be registered.
	 * @param {Ext.Component} componentInstance component which is required to be rendered into the separate browser window.
	 * @param {Ext.Container} mainContainer The container which is the parent most container of separate window.
	 * @private
	 */
	onSeparateWindowUnload: function(browserWindowObject, componentInstance, mainContainer)
	{
		Ext.defer(this.onWindowRefreshOrClose, 50, this, [browserWindowObject, componentInstance, browserWindowObject.name]);

		// Remove the panel from the ContentPanelMgr
		Zarafa.core.data.ContentPanelMgr.unregister(componentInstance);

		mainContainer.destroy();
	},

	/**
	 * Function which handles the situation where window gets refreshed or closed
	 * if it was refresh then create record instance based on original record and update the browser window component config.
	 * And if window was closed then de-register the closed window component from {@link #browserWindowComponents}.
	 * @param {Object} win The browser window which gets refresh or closed.
	 * @param {Ext.Component} componentInstance component of the separate browser window.
	 * @param {Object} browserWindowId The newly created window object id which must be registered.
	 */
	onWindowRefreshOrClose: function (win, componentInstance, browserWindowId)
	{
		var isClose = false,
			oldRecord = componentInstance.record;
		try {
			if (!win || !win.innerWidth) {
				isClose = true;
			} else {
				var config = this.browserWindowComponents.get(win.name).config,
					model = container.getCurrentContext().getModel(),
					newRecord;

				if (Ext.isDefined(config.recordComponentPluginConfig) && Ext.isDefined(config.recordComponentPluginConfig.useShadowStore)) {
					config.recordComponentPluginConfig.useShadowStore = true;
				}

				// on window refresh we have to discard all unsaved user changes
				// If record is not phantom then get new record from respective store using oldRecord id properties.
				if (!oldRecord.phantom) {
					var recordConfig = {
						store_entryid: oldRecord.get('store_entryid'),
						parent_entryid: oldRecord.get('parent_entryid'),
						entryid: oldRecord.get('entryid')
					};

					newRecord = Zarafa.core.data.RecordFactory.createRecordObjectByMessageClass('IPM.Note', recordConfig, oldRecord.get('entryid'));
				} else {
					// if record is phantom then we have to discard this record and
					// create new record using respective context model
					newRecord = model.createRecord();
				}
				if (Ext.isDefined(config.isRecordChangeByUser)) {
					config.isRecordChangeByUser = undefined;
				}
				win.onbeforeunload = undefined;

				// Update the new record in respective browser window config
				config.record = newRecord;
			}
		} catch (e) {
			// Catch an Exception. Firefox will throw one when we
			// try to access property of the window that does not
			// exist anymore.
			isClose = true;
		}
		container.getShadowStore().remove(oldRecord, true);
		if (isClose) {
			this.browserWindowComponents.removeKey(browserWindowId);
			this.unRegister(browserWindowId);
		}
	},

	/**

	/**
	 * Event handler which is raised after the {@link Zarafa.core.ui.ContentPanel contentPanel} has been
	 * closed. This will de-register the closed window instance from {@link Zarafa.core.BrowserWindowMgr BrowserWindowMgr}.
	 *
	 * @private
	 */
	onSeparateWindowClose: function(browserWindowObject)
	{
		browserWindowObject.close();
	},

	/**
	 * Event handler which is raised when the browser window gets resized.
	 * This will register a {@link Ext.util.DelayedTask DelayedTask} and set the delay of 100ms.
	 * Delay is required to avoid executing the handler frequently, if event fires multiple time within the specified time-frame
	 * then the handler will gets executed only once for the last event.
	 *
	 * @param {Object} browserWindowObject The browser window which gets resized.
	 * @param {Ext.Container} mainContainer The container which needs to be resized according to the size of browser window.
	 * @private
	 */
	onSeparateWindowResize: function(browserWindowObject, mainContainer)
	{
		var resizeTask = new Ext.util.DelayedTask(
			this.doSeparateWindowResize,
			this,
			[browserWindowObject, mainContainer]
		);

		resizeTask.delay(100);
	},

	/**
	 * A {@link Ext.util.DelayedTask DelayedTask} which will set size of main container according to the browser window
	 * when it gets resized.
	 *
	 * @param {Object} browserWindowObject The browser window which gets resized.
	 * @param {Ext.Container} mainContainer The container which needs to be resized according to the size of browser window.
	 * @private
	 */
	doSeparateWindowResize: function(browserWindowObject, mainContainer)
	{
		var width = browserWindowObject.innerWidth || Ext.lib.Dom.getViewWidth();
		var height = browserWindowObject.innerHeight || Ext.lib.Dom.getViewHeight();
		mainContainer.setSize(width, height);

		this.fireEvent('separatewindowresize', browserWindowObject, width, height);
	},

	/**
	 * Event handler which is fired when the {@link Ext#getBody &lt;body&gt;} elements fires
	 * the 'contextmenu' event. If the element which fired the event doesn't have the
	 * 'zarafa-contextmenu-enabled' class then the Browser contextmenu will be disabled.
	 * @param {Ext.EventObject} event The event object
	 * @param {Ext.Element} el The element on which the contextmenu was requested
	 * @private
	 */
	onBodyContextMenu: function(event, el)
	{
		el = Ext.get(el);

		// Don't disable the browser contextmenu when the
		// 'zarafa-contextmenu-enabled' CSS class is applied
		// on the element.
		if (el.hasClass('zarafa-contextmenu-enabled') || el.up('div.zarafa-contextmenu-enabled')) {
			return;
		}

		// Don't disable the browser contextmenu for regular
		// text inputs.
		if ( el.dom.tagName.toUpperCase() === 'INPUT' ) {
			var type = el.getAttribute('type') || '';
			var readonly = !Ext.isEmpty(el.dom.attributes.readonly);
			if ( type.toUpperCase() === 'TEXT' && !readonly ) {
				return;
			}
		}

		event.preventDefault();
	},

	/**
	 * Function which is used to close all browser windows except main window
	 */
	closeAllBrowserWindow: function()
	{
		var browserWindows = Zarafa.core.BrowserWindowMgr.browserWindows.items;
		Ext.each(browserWindows,function(windowObject) {
			if(windowObject.name !== 'mainBrowserWindow') {
				windowObject.close();
			}
		});
	},

	/**
	 * Function which is used to initialize ext doc classes on separate window
	 * Ext has some separate doc classes for all browsers,and
	 * must be initialize those classes for newly created window.
	 * @param browserWindowObject The newly created window object which must be registered.
	 * @returns {boolean} true if all doc classes successfully apply on new window, false otherwise
	 */
	initExtCss: function (browserWindowObject) {
		// find the body element
		var body = browserWindowObject.document.body || browserWindowObject.document.getElementsByTagName('body')[0];
		if (!body) {
			return false;
		}

		Ext.fly(body.parentElement).addClass('x-viewport');

		var cls = [];

		if (Ext.isIE) {
			// Only treat IE9 and less like IE in the css
			if (!Ext.isIE10p) {
				cls.push('ext-ie');
			}
			if (Ext.isIE6) {
				cls.push('ext-ie6');
			} else if (Ext.isIE7) {
				cls.push('ext-ie7', 'ext-ie7m');
			} else if (Ext.isIE8) {
				cls.push('ext-ie8', 'ext-ie8m');
			} else if (Ext.isIE9) {
				cls.push('ext-ie9', 'ext-ie9m');
			} else if (Ext.isIE10) {
				cls.push('ext-ie10');
			}
		}

		if (Ext.isGecko) {
			if (Ext.isGecko2) {
				cls.push('ext-gecko2');
			} else {
				cls.push('ext-gecko3');
			}
		}

		if (Ext.isOpera) {
			cls.push('ext-opera');
		}

		if (Ext.isWebKit) {
			cls.push('ext-webkit');
		}

		if (Ext.isSafari) {
			cls.push("ext-safari " + (Ext.isSafari2 ? 'ext-safari2' : (Ext.isSafari3 ? 'ext-safari3' : 'ext-safari4')));
		} else if (Ext.isChrome) {
			cls.push("ext-chrome");
		}

		if (Ext.isMac) {
			cls.push("ext-mac");
		}
		if (Ext.isLinux) {
			cls.push("ext-linux");
		}

		// add to the parent to allow for selectors like ".ext-strict .ext-ie"
		if (Ext.isStrict || Ext.isBorderBox) {
			var p = body.parentNode;
			if (p) {
				if (!Ext.isStrict) {
					Ext.fly(p, '_internal').addClass('x-quirks');
					if (Ext.isIE9m && !Ext.isStrict) {
						Ext.isIEQuirks = true;
					}
				}
				Ext.fly(p, '_internal').addClass(((Ext.isStrict && Ext.isIE ) || (!Ext.enableForcedBoxModel && !Ext.isIE)) ? ' ext-strict' : ' ext-border-box');
			}
		}
		// Forced border box model class applied to all elements. Bypassing javascript based box model adjustments
		// in favor of css. This is for non-IE browsers.
		if (Ext.enableForcedBoxModel && !Ext.isIE) {
			Ext.isForcedBorderBox = true;
			cls.push("ext-forced-border-box");
		}

		Ext.fly(body, '_internal').addClass(cls);
		return true;
	},

	/**
	 * Function which is used to display warning message while browser blocks popups
	 */
	displayBlockedPopupWarning: function() {
		if(Zarafa.core.BrowserWindowMgr.isPopupsBlocked) {
			// We completed with the internal UI of single-allowed popup, Inform user that popup is blocked
			Ext.MessageBox.show({
				title: _("Open in new browser window"),
				msg: _("Your browser seems to have blocked one or more pop-ups. Please change your browser's settings to always allow pop-ups from grommunio Web."),
				buttons: Ext.Msg.OK,
				cls: Ext.MessageBox.WARNING_CLS
			});

			Zarafa.core.BrowserWindowMgr.isPopupsBlocked = false;
		}
	},

	/**
	 * Helper function which will bring the main webapp window to front
	 * @param {Ext.Component/Ext.Element} component/element of which will use to get the owning browser window.
	 */
	switchFocusToMainWindow: function (component)
	{
		var activeWindow = this.getActive();

		// if component belongs to one of the currently opened popout windows then get owner window of component
		if (Ext.isDefined(component) && !this.isOwnedByMainWindow(component)) {
			activeWindow = this.getOwnerWindow(component);
		}

		activeWindow.setFocusOnMainWindow();
	},

	/**
	 * Function which is use to find browser window of given record.
	 * @param {Zarafa.core.data.MAPIRecord} record The mapi record
	 * @return {Object} Browser window object
	 */
	getOpenedWindow: function (record)
	{
		var entryid = this.getWindowId(record);

		return this.browserWindows.find(function (browserWindow) {
			return Zarafa.core.EntryId.compareEntryIds(browserWindow.entryid, entryid);
		}, this);
	},

	/**
	 * Helper function which return entryid of given record.
	 * If record is attachment record then take id instead of entryid because
	 * Attachment record id contains entryid with attach_num.
	 * @param {Zarafa.core.data.MAPIRecord} record The mapi record.
	 * @returns {String} entryid Return entryid of record.
	 */
	getWindowId: function (record)
	{
		// When we have more than one record (e.g. for the reminder dialog), we
		// will concatenate the entryids to generate an id
		if (Array.isArray(record)) {
			return record.map(function(r) {
				return r.get('entryid') || r.get('id');
			}).sort().join('-');
		}

		return Array.isArray(record.get('attach_num')) ? record.id : record.get('entryid');
	},

	/**
	 * Helper function which return main browser window body.
	 * @returns {Object} body of main browser window.
	 */
	getMainBrowserWindowBody: function ()
	{
		var activeBrowserWindow = this.browserWindows.get('mainBrowserWindow');
		var documentObject = Ext.isDefined(activeBrowserWindow) ? activeBrowserWindow.document : document;
		return Ext.get(documentObject.body || documentObject.documentElement);
	}
});

Zarafa.core.BrowserWindowMgr = new Zarafa.core.BrowserWindowMgr();
Ext.namespace('Zarafa.core');

/**
 * @class Zarafa.core.ColorSchemes
 * @singleton
 *
 * An object that can be used to handle color schemes.
 * Color schemes are used by {@link Zarafa.core.Context contexts}
 * that can display different folders in one view. Currently only
 * the {@link Zarafa.calendar.CalendarContext calendar context}
 * is such a context.
 *
 * It has methods to create color schemes based on a single color
 * and to add complete color schemes.
 */
Zarafa.core.ColorSchemes = {

	/**
	 * An array with the fields that represent a color available in
	 * a color scheme. The fields have a name that can be used a key
	 * in a color scheme, and a weight (percentage) that will be used
	 * to create these color based on the base color of a color scheme.
	 * Fields can be added using the function {@link #addField}. When
	 * a field is added, all existing color schemes will also get the
	 * new field.
	 * @property
	 * @type Object[]
	 * @private
	 */
	fields: [
		{
			name: 'base',
			weight: 1
		}
	],

	/**
	 * The list of color schemes. Contexts can add color schemes
	 * using the function {@link #createColorScheme} or {@link #addColorScheme}.
	 * @property
	 * @type Object[]
	 * @private
	 */
	colorSchemes: [],

	/**
	 * Adds a field to the {@link #fields colorScheme fields}
	 * @param {Object|Array} field An object with properties name (mandatory) and
	 * weight (the weight of the color as ratio of the base color of the color
	 * scheme) or color (an RGB hex value that will be used as color for this field
	 * (e.g. '#0067AC')), or an array with field objects.
	 */
	addField: function(field)
	{
		var i;

		if ( Array.isArray(field) ){
			for ( i=0; i<field.length; i++ ){
				this.addField(field[i]);
			}
			return;
		}

		this.fields.push(field);

		// Add the color to existing color schemes
		for ( i =0; i<this.colorSchemes.length; i++ ){
			if ( !Ext.isDefined(this.colorSchemes[i][field.name]) ){
				if ( Ext.isDefined(field.color) ){
					this.colorSchemes[i][field.name] = field.color;
				} else if ( Ext.isDefined(field.weight) ){
					this.colorSchemes[i][field.name] = this.createColor(this.colorSchemes[i].base, field.weight);
				}
			}
		}
	},

	/**
	 * Converts a hexadecimal RGB color value into an object with
	 * red, green, and blue fields
	 * @param {String} hexColor A hexadecimal RGB color value
	 * (e.g. '#0067AC for grommunio Blue)
	 * @return {Object} An object with decimal red, green, and blue values
	 * @private
	 */
	hexToRgb: function(hexColor)
	{
		return {
			red: parseInt(hexColor.substring(1,3), 16),
			green: parseInt(hexColor.substring(3,5), 16),
			blue: parseInt(hexColor.substring(5,7), 16)
		};
	},

	/**
	 * Converts an RGB object into a hexadecimal RGB color value
	 * @param {Object} rgbObj An decimal RGB color object
	 * @return {String} A hexadecimal RGB color value
	 */
	rgbToHex: function(rgbObj)
	{
		// function that will convert a number to a hexadecimal string (between 0 and 255)
		const _toHex = function(n) {
			n = parseInt(n,10);
			if (isNaN(n)) {
				return "00";
			}
			n = Math.max(0,Math.min(n,255));
			return "0123456789ABCDEF".charAt((n-n%16)/16) + "0123456789ABCDEF".charAt(n%16);
		};

		return '#' + _toHex(rgbObj.red)+_toHex(rgbObj.green)+_toHex(rgbObj.blue);
	},

	/**
	 * Creates a color for a color scheme based on the baseColor of
	 * that scheme and a weight factor.
	 * @param {String} baseColor A hexadecimal RGB color value
	 * @param {Number} colorWeight A value that defines the new color
	 * as related to the baseColor. Should be between 0 and 1.
	 * @return {String} A hexadecimal RGB color value
	 * @private
	 */
	createDarkColor: function(baseColor, colorWeight)
	{
		var rgbBaseColor = this.hexToRgb(baseColor);

		var rgbColor = {
			red: rgbBaseColor.red * colorWeight,
			green: rgbBaseColor.green * colorWeight,
			blue: rgbBaseColor.blue * colorWeight
		};

		return this.rgbToHex(rgbColor);
	},

	/**
	 * Creates a color for a color scheme based on the baseColor of
	 * that scheme and a weight factor.
	 * @param {String} baseColor A hexadecimal RGB color value
	 * @param {Number} colorWeight A value that defines the new color
	 * as related to the baseColor. Should be larger than 1.
	 * @return {String} A hexadecimal RGB color value
	 * @private
	 */
	createLightColor: function(baseColor, colorWeight)
	{
		var rgbBaseColor = this.hexToRgb(baseColor);

		var rgbColor = {
			red: 255 - (255-rgbBaseColor.red) * (255-128*colorWeight) / 127,
			green: 255 - (255-rgbBaseColor.green) * (255-128*colorWeight) / 127,
			blue: 255 - (255-rgbBaseColor.blue) * (255-128*colorWeight) / 127
		};

		return this.rgbToHex(rgbColor);
	},

	/**
	 * Creates a color for a color scheme based on the baseColor of
	 * that scheme and a weight factor.
	 * @param {String} baseColor A hexadecimal RGB color value
	 * @param {Number} colorWeight A value that defines the new color
	 * as related to the baseColor. Between 0 and 1 for a color that is darker
	 * than the baseColor. Larger then 1 for a color that is lighter
	 * than the baseColor.
	 * @return {String} A hexadecimal RGB color value
	 * @private
	 */
	createColor: function(baseColor, colorWeight)
	{
		var rgbBaseColor = this.hexToRgb(baseColor);

		if ( Math.max(rgbBaseColor.red, rgbBaseColor.green, rgbBaseColor.blue) > 127 ){
			return this.createLightColor(baseColor, colorWeight);
		} else {
			return this.createDarkColor(baseColor, colorWeight);
		}
	},

	/**
	 * Creates a color scheme based on a single base color
	 *
	 * @param {String} name The unique name for this color scheme
	 * (can be used to identify the color scheme)
	 * @param {String} displayName The name of the color scheme that
	 * will be used if a name for the color scheme must be shown
	 * to the user.
	 * @param {String} baseColor an RGB hexadecimal color value
	 * (e.g. '#0067AC for grommunio Blue)
	 */
	createColorScheme: function(name, displayName, baseColor)
	{
		var i;

		if ( Array.isArray(name) ){
			for ( i=0; i<name.length; i++ ){
				this.createColorScheme(name[i]);
			}

			return;
		}

		if ( Ext.isObject(name) ){
			displayName = name.displayName;
			baseColor = name.baseColor || name.base;
			name = name.name;
		}

		var colorScheme = {
			name: name,
			displayName: displayName,
			base: baseColor
		};

		// Loop through all the fields and create a color for it in this scheme
		for ( i=0; i<this.fields.length; i++ ){
			if ( this.fields[i].name !== 'base' ){
				if ( Ext.isDefined(this.fields[i].color) ){
					colorScheme[this.fields[i].name] = this.fields[i].color;
				} else {
					colorScheme[this.fields[i].name] = this.createColor(baseColor, this.fields[i].weight);
				}
			}
		}

		if ( !Ext.isDefined(this.getColorScheme(name)) ){
			this.colorSchemes.push(colorScheme);
		}
	},

	/**
	 * Adds a complete color scheme to the color scheme list
	 * @param {Object} colorScheme
	 */
	addColorScheme: function(colorScheme)
	{
		// Simple check
		if ( !Ext.isDefined(colorScheme.name) || !Ext.isDefined(colorScheme.displayName) || !Ext.isDefined(colorScheme.base) ){
			// Missing necessary properties for a valid color scheme
			// So don't add the color scheme.
			return;
		}

		// Create colors that are not available in the passed color scheme
		for ( var i=0; i<this.fields.length; i++ ){
			if ( !Ext.isDefined(colorScheme[this.fields[i].name]) ){
				if ( Ext.isDefined(this.fields[i].color) ){
					colorScheme[this.fields[i].name] = this.fields[i].color;
				} else {
					colorScheme[this.fields[i].name] = this.createColor(colorScheme.base, this.fields[i].weight);
				}
			}
		}

		this.colorSchemes.push(colorScheme);
	},

	/**
	 * Adds the color schemes and additional color schemes that are defined
	 * in the config.php/default.php
	 */
	addColorSchemesFromConfig: function()
	{
		var i;
		var colorSchemes = container.getServerConfig().getColorSchemes();
		if ( colorSchemes && Array.isArray(colorSchemes) ){
			for ( i=0; i<colorSchemes.length; i++ ){
				this.addColorScheme(colorSchemes[i]);
			}
		}
		var additionalColorSchemes = container.getServerConfig().getAdditionalColorSchemes();
		if ( additionalColorSchemes && Array.isArray(additionalColorSchemes) ){
			for ( i=0; i<additionalColorSchemes.length; i++ ){
				this.addColorScheme(additionalColorSchemes[i]);
			}
		}
	},

	/**
	 * Returns the color scheme with the passed name if found,
	 * or undefined otherwise
	 * @param {String} colorSchemeName The name of the color scheme
	 * @return {Object|undefined} A color scheme or undefined if not found
	 */
	getColorScheme: function(colorSchemeName)
	{
		for ( var i=0; i<this.colorSchemes.length; i++ ){
			if ( this.colorSchemes[i].name === colorSchemeName ){
				return this.colorSchemes[i];
			}
		}

		return undefined;
	},

	/**
	 * Returns the array with all defined color schemes
	 * @return {Object[]} An array with all defined color schemes
	 */
	getColorSchemes: function()
	{
		return this.colorSchemes;
	},

	/**
	 * Converts an RGB color value to HSL. Conversion formula
	 * adapted from http://en.wikipedia.org/wiki/HSL_color_space.
	 * Assumes r, g, and b are contained in the set [0, 255] and
	 * returns h, s, and l in the set [0, 1].
	 *
	 * @param {Number} r    The red color value
	 * @param {Number} g    The green color value
	 * @param {Number} b    The blue color value
	 * @return {Array}      The HSL representation
	 */
	rgbToHsl: function(r, g, b)
	{
		if ( arguments.length === 1 && r.substr(0,1) === '#' ){
			var rgb = this.hexToRgb(r);
			r = rgb.red;
			g = rgb.green;
			b = rgb.blue;
		}

	  r /= 255;
	  g /= 255;
	  b /= 255;
	  var max = Math.max(r, g, b), min = Math.min(r, g, b);
	  var h, s, l = (max + min) / 2;

	  if ( max === min ) {
	    h = s = 0; // achromatic
	  } else {
	    var d = max - min;
	    s = l > 0.5 ? d / (2 - max - min) : d / (max + min);
	    switch(max){
	      case r: h = (g - b) / d + (g < b ? 6: 0); break;
	      case g: h = (b - r) / d + 2; break;
	      case b: h = (r - g) / d + 4; break;
	    }
	    h /= 6;
	  }

	  return [h, s, l];
	},

	/**
	 * Returns the luma value (brightness perception) of a color.
	 * The color can be passed as an red, green & blue triplet or as a hex string.
	 * See http://stackoverflow.com/a/12043228
	 *
	 * @param {Number|String} r The red color value or the hex string of an rgb color (starting with #)
	 * @param {Number} g The green color value
	 * @param {Number} b The blue color value
	 * @return {Number} luma value
	 */
	getLuma: function(r, g, b)
	{
		if ( g === undefined && r.substr(0,1) === '#' ){
			var rgb = this.hexToRgb(r);
			r = rgb.red;
			g = rgb.green;
			b = rgb.blue;
		}

		var luma = 0.2126 * r + 0.7152 * g + 0.0722 * b; // per ITU-R BT.709

		return luma;
	},

	/**
	 * Checks if a color should be treated as dark (e.g. a dark background should have a light text color)
	 * The color can be passed as an red, green & blue triplet or as a hex string.
	 *
	 * @param {Number|String} r The red color value or the hex string of an rgb color (starting with #)
	 * @param {Number} g The green color value
	 * @param {Number} b The blue color value
	 * @return {Boolean} True is the given color should be considered as dark
	 */
	isDark: function(r, g, b)
	{
		// Use the (very subjective) value of 155 as the border between dark and light
		return Zarafa.core.ColorSchemes.getLuma(r, g, b) < 155;
	}
};

Zarafa.onReady(function(){
		// Load the color schemes that are defined in the config
		Zarafa.core.ColorSchemes.addColorSchemesFromConfig();
});
/**
 * @class Zarafa.core.Container
 * @extends Object
 *
 * Container class for lazy instantiation of globally used objects such as a Request instance, HierarchyStore instance, etc.
 * The Container is also used to register plugins and populate insertion points.
 * <p>
 * Don't instantiate the a new Container object yourself, just use the global <code>container</code> instance.
 */
Zarafa.core.Container = Ext.extend(Ext.util.Observable, {
	/**
	 * List of registered {@link Zarafa.core.Context context instances}.
	 * @property
	 * @private
	 * @type Array
	 */
	contexts: undefined,

	/**
	 * The Meta Data for all registered {@link #contexts}. This is an array
	 * of {@link Zarafa.core.ContextMetaData ContextMetaData instances}.
	 * @property
	 * @private
	 * @type Array
	 */
	contextsMetaData: undefined,

	/**
	 * List of registered {@link Zarafa.core.Plugin plugin instances}
	 * (also includes {@link #contexts}).
	 * @property
	 * @private
	 * @type Array
	 */
	plugins: undefined,

	/**
	 * The Meta Data for all registered {@link #plugins}. This is an array
	 * of {@link Zarafa.core.PluginMetaData PluginMetaData instances}.
	 * @property
	 * @private
	 * @type Array
	 */
	pluginsMetaData: undefined,

	/**
	 * The Meta Data for all registered {@link Zarafa.core.ui.widget.Widget widgets}. This is an array
	 * of {@link Zarafa.core.ui.widget.WidgetMetaData WidgetMetaData instances}.
	 * @property
	 * @private
	 * @type Array
	 */
	widgetsMetaData: undefined,

	/**
	 * @constructor
	 */
	constructor: function()
	{
		this.addEvents([
			/**
			 * @event contextswitch fired right before the context switch is made
			 * @param {Zarafa.hierarchy.data.MAPIFolderRecord} folder folder that is loaded for the new context
			 * @param {Zarafa.core.Context} oldContext context being switched out
			 * @param {Zarafa.core.Context} newContext new context being switched in
			 * @return {Boolean} False to prevent the context from being switched
			 */
			'beforecontextswitch',
			/**
			 * @event contextswitch fired after a context switch has been performed
			 * @param {Zarafa.hierarchy.data.MAPIFolderRecord} folder folder that is loaded for the new context
			 * @param {Zarafa.core.Context} oldContext context that was switched out
			 * @param {Zarafa.core.Context} newContext new context that was switched
			 */
			'contextswitch',
			/**
			 * @event contextswitch fired after a context switch has been performed and after the contextswitch has been fired.
			 * @param {Zarafa.hierarchy.data.MAPIFolderRecord} folder folder that is loaded for the new context
			 * @param {Zarafa.core.Context} oldContext context that was switched out
			 * @param {Zarafa.core.Context} newContext new context that was switched
			 */
			'aftercontextswitch',
			/**
			 * Fires when the user selects a folder from the hierarchy.
			 * @event folderselect
			 * @param {Zarafa.hierarchy.data.MAPIFolderRecord[]} folder MAPI folder object.
			 */
			'folderselect',
			/**
			 * Fires when the webapp has loaded and the loading mask has been removed.
			 * @event webapploaded
			 */
			'webapploaded',
			/**
			 * Fires when the webapp will reload. Return false from an event handler to cancel the reload.
			 * @event beforewebappreload
			 */
			'beforewebappreload',
			/**
			 * Fires before the user logs out from the webapp. Return false from an event handler to stop the show.
			 * @event beforelogout
			 */
			'beforelogout',
			/**
			 * Fires when the user logs out from the webapp.
			 * @event logout
			 */
			'logout',
			/**
			 * Fires when the {@link Zarafa.core.ui.MainContentTabPanel MainContentTabPanel} rendered successfully.
			 * @event afterrendercontentpanel
			 * @param {Zarafa.core.ui.MainContentTabPanel} MainContentTabPanel The main content tab panel.
			 */
			'afterrendercontentpanel'
		]);

		Zarafa.core.Container.superclass.constructor.call(this);

		// initialize properties
		this.plugins = [];
		this.pluginsMetaData = [];
		this.contexts = [];
		this.contextsMetaData = [];
		this.widgetsMetaData = [];
	},

	/**
	 * Logout from the Webapp. this will fire the {@link #beforelogout} and
	 * {@link #logout} events before calling {@link #doLogout}.
	 * @param {Boolean} preserveUser True to preserve the username when he
	 * is forwarded to the logon page
	 * @param {Boolean} preserveSession True to preserve the existing session
	 * on the server and only redirect the user to the logon page.
	 */
	logout: function(preserveUser, preserveSession, reauthenticate)
	{
		if (this.fireEvent('beforelogout') !== false) {
			this.fireEvent('logout');
			this.doLogout(preserveUser, preserveSession, reauthenticate);
		}
	},

	/**
	 * Logout from the Webapp (this function is called by {@link #logout}.
	 * Override this to change the logout method.
	 * @param {Boolean} preserveUser True to preserve the username when he
	 * is forwarded to the logon page
	 * @param {Boolean} preserveSession True to preserve the existing session
	 * on the server and only redirect the user to the logon page.
	 * @protected
	 */
	doLogout: function(preserveUser, preserveSession, reauthenticate)
	{
		var user = ((preserveUser === true) ? ('&user=' + this.getUser().getUserName()) : '');

		Zarafa.core.Util.disableLeaveRequester();
		// OIDC is enabled, signout via oidc-client.
		if (preserveSession !== true) {
			window.location = 'index.php?logout' + user;
		} else {
			window.location = 'index.php?load=logon' + user;
		}
	},

	/**
	 * Obtain the server configuration data
	 * @return {Zarafa.core.data.ServerConfig} The server configuration data
	 */
	getServerConfig: function()
	{
		return this.serverConfigRecord;
	},

	/**
	 * Set the server configuration data
	 * @param {Object} serverData The server configuration data.
	 */
	setServerConfig: function(serverData)
	{
		this.serverConfigRecord = new Zarafa.core.data.ServerConfig(serverData);
	},

	/**
	 * Obtain the user data for the currently logged in user.
	 * @return {Zarafa.core.data.User} The user data of the currently logged in user.
	 */
	getUser: function()
	{
		return this.userRecord;
	},

	/**
	 * Set the user data for the currently logged in user.
	 * @param {Object} userData The user data of the currently logged in user.
	 */
	setUser: function(userData)
	{
		this.userRecord = new Zarafa.core.data.User(userData);
	},

	/**
	 * Obtain the versioning data for grommunio Web environment
	 * @return {Zarafa.core.data.Version} The version data of the grommunio Web environment
	 */
	getVersion: function()
	{
		return this.versionRecord;
	},

	/**
	 * Set the version data for grommunio Web environment
	 * @param {Object} versionData The version data of the grommunio Web environment
	 */
	setVersion: function(versionData)
	{
		this.versionRecord = new Zarafa.core.data.Version(versionData);
	},

	/**
	 * Obtain the array of all available languages. Each item in the array
	 * contains 2 keys. The first key is 'lang' which is the language code
	 * (e.g. en_GB), and the second key is 'name' which is the display name.
	 * @return {Array} The array of available languages
	 */
	getLanguages: function()
	{
		return this.languages;
	},

	/**
	 * Set the languages which are available to the user
	 * @param {Array} languages The available languages
	 */
	setLanguages: function(languages)
	{
		this.languages = languages;
	},

	/**
	 * Returns the currently active {@link Zarafa.core.Context context}.
	 * @return {Zarafa.core.Context} the currently active context.
	 */
	getCurrentContext: function()
	{
		return this.currentContext || this.getContextByName('default');
	},

	/**
	 * Returns the global {@link Zarafa.core.Request request} instance.
	 * All server requests should be lodged through this instance.
	 *
	 * @return {Zarafa.core.Request} the global {@link Zarafa.core.Request Request} instance.
	 */
	getRequest: function()
	{
		return this.request || (this.request = new Zarafa.core.Request({ url:"grommunio.php" }));
	},

	/**
	 * Returns the global {@link Zarafa.core.ResponseRouter ResponseRouter} instance.
	 * All server responses are lodged through this instance.
	 *
	 * @return {Zarafa.core.ResponseRouter} the global {@link Zarafa.core.ResponseRouter ResponseRouter} instance.
	 */
	getResponseRouter: function()
	{
		return this.responseRouter || (this.responseRouter = new Zarafa.core.ResponseRouter());
	},

	/**
	 * Returns the global {@link Zarafa.core.data.NotificationResolver NotificationResolver} instance.
	 * All notifications are being resolved through this instance, the constructed
	 * {@link Zarafa.core.data.AbstractResponseHandler ResponseHandler} is then returned to the
	 * {@link Zarafa.core.ResponseRouter ResponseRouter} for further processing.
	 *
	 * @return {Zarafa.core.data.NotificationResolver} the global {@link Zarafa.core.data.NotificationResolver NotificationResolver} instance.
	 */
	getNotificationResolver: function()
	{
		return this.notificationResolver || (this.notificationResolver = new Zarafa.core.data.NotificationResolver());
	},

	/**
	 * Returns the global {@link Zarafa.hierarchy.data.HierarchyStore HierarchyStore} instance.
	 * @return {Zarafa.hierarchy.data.HierarchyStore} the global {@link Zarafa.hierarchy.data.HierarchyStore HierarchyStore} instance.
	 */
	getHierarchyStore: function()
	{
		return this.hierarchyStore || (this.hierarchyStore = new Zarafa.hierarchy.data.HierarchyStore());
	},

	/**
	 * Returns the global {@link Zarafa.common.outofoffice.data.OofStore OofStore} instance.
	 * @return {Zarafa.common.outofoffice.data.OofStore} the global {@link Zarafa.common.outofoffice.data.OofStore} instance.
	 */
	getOutOfOfficeStore: function()
	{
		return this.outOfOfficeStore || (this.outOfOfficeStore = new Zarafa.common.outofoffice.data.OofStore());
	},

	/**
	 * Returns the global {@link Zarafa.settings.SettingsModel SettingsModel} instance.
	 * @return {Zarafa.settings.SettingsModel} the global {@link Zarafa.settings.SettingsModel SettingsModel} instance.
	 */
	getSettingsModel: function()
	{
		return this.settingsModel || (this.settingsModel = new Zarafa.settings.SettingsModel());
	},

	/**
	 * Returns the global {@link Zarafa.settings.PersistentSettingsModel PersistentSettingsModel} instance.
	 * @return {Zarafa.settings.PersistentSettingsModel} the global
	 * {@link Zarafa.settings.PersistentSettingsModel PersistentSettingsModel} instance.
	 */
	getPersistentSettingsModel: function()
	{
		return this.persistentSettingsModel || (this.persistentSettingsModel = new Zarafa.settings.PersistentSettingsModel());
	},

	/**
	 * Returns the {@link Zarafa.core.data.ShadowStore ShadowStore} instance.
	 * @return {Zarafa.core.data.ShadowStore} the {@link Zarafa.core.data.ShadowStore ShadowStore} instance.
	 */
	getShadowStore: function()
	{
		return this.shadowStore || (this.shadowStore = new Zarafa.core.data.ShadowStore());
	},

	/**
	 * Returns the {@link Zarafa.core.ui.notifier.Notifier Notifier} instance which can be used
	 * for sending notifications to the user.
	 * @return {Zarafa.core.ui.notifier.Notifier} The notifier for User notifications
	 */
	getNotifier: function()
	{
		return this.notifier || (this.notifier = new Zarafa.core.ui.notifier.Notifier());
	},

	/**
	 * Returns the application main panel.
	 * @return {Zarafa.core.ui.MainViewport} the application main panel.
	 */
	getMainPanel: function()
	{
		return this.mainPanel || (this.mainPanel = new Zarafa.core.ui.MainViewport());
	},

	/**
	 * Returns the applications main toolbar
	 * @return {Zarafa.core.ui.MainToolbar} then application main tool bar
	 */
	getMainToolbar: function()
	{
		return this.getMainPanel().mainToolbar;
	},

	/**
	 * Returns the application welcome panel.
	 * @return {Zarafa.core.ui.WelcomeViewport} the application welcome panel.
	 */
	getWelcomePanel: function()
	{
		return this.welcomePanel || (this.welcomePanel = new Zarafa.core.ui.WelcomeViewport());
	},

	/**
	 * Returns the application tab panel
	 * @return {Zarafa.core.ui.ContextContainer} The application tab panel
	 */
	getTabPanel: function()
	{
		return this.getMainPanel().getContentPanel();
	},

	/**
	 * Returns the application content panel
	 * @return {Zarafa.common.ui.ContextMainPanel} the application content panel.
	 */
	getContentPanel: function()
	{
		return this.getTabPanel().get(0).getActiveItem();
	},

	/**
	 * Returns the application navigation sidebar.
	 * @return {Zarafa.core.ui.NavigationPanel} the navigation sidebar
	 */
	getNavigationBar: function()
	{
		return this.getMainPanel().getNavigationPanel();
	},

	/**
	 * Returns the application widget sidebar.
	 * @return {Zarafa.core.ui.widget.WidgetPanel} the application widget sidebar.
	 */
	getWidgetSideBar: function()
	{
		return this.getMainPanel().getWidgetPanel();
	},

	/**
	 * Returns an array of all registered {@link Zarafa.core.Plugin plugins}.
	 * @return {Array} plugins
	 */
	getPlugins: function()
	{
		return this.plugins;
	},

	/**
	 * Returns the Meta Data for {@link #pluginsMetaData all registered} {@link Zarafa.core.Plugin plugins}.
	 * @return {Array} The plugins meta data
	 */
	getPluginsMetaData: function()
	{
		return this.pluginsMetaData;
	},

	/**
	 * Returns an array of all registered {@link Zarafa.core.Context contexts}.
	 * @return {Array} Contexts
	 */
	getContexts: function()
	{
		return this.contexts;
	},

	/**
	 * Returns the Meta Data for {@link #contextsMetaData all registered} {@link Zarafa.core.Context contexts}.
	 * @return {Array} The contexts meta data
	 */
	getContextsMetaData: function()
	{
		return this.contextsMetaData;
	},

	/**
	 * Returns the Meta Data for {@link #widgetsMetaData all registered} {@link Zarafa.core.ui.widget.Widget widgets}.
	 * @return {Array} The widgets meta data
	 */
	getWidgetsMetaData: function()
	{
		return this.widgetsMetaData;
	},

	/**
	 * Returns the context that matches the supplied name.
	 * @param {String} name The name of the context which is requested
	 * @return {Zarafa.core.Context} matching context or <code>undefined</code> if not found.
	 */
	getContextByName: function(name)
	{
		var contexts = this.getContexts();

		for (var index = 0, len = contexts.length; index < len; index++) {
			if (contexts[index].getName() === name) {
				return contexts[index];
			}
		}
	},

	/**
	 * Returns the Context Meta Data that matches the supplied name.
	 * @param {String} name The name of the context for which the meta data is requested
	 * @return {Zarafa.core.ContextMetaData} The Meta Data for the context or <code>undefined</code> if not found.
	 */
	getContextMetaDataByName: function(name)
	{
		var contexts = this.getContextsMetaData();

		for (var index = 0, len = contexts.length; index < len; index++) {
			if (contexts[index].getName() === name) {
				return contexts[index];
			}
		}
	},

	/**
	 * Returns the plug-in that matches the supplied name.
	 * @param {String} name The name of the plugin which is requested
	 * @return {Zarafa.core.Plugin} matching plug-in or <code>undefined</code> if not found.
	 */
	getPluginByName: function(name)
	{
		var plugins = this.getPlugins();

		for (var index = 0, len = plugins.length; index < len; index++) {
			if (plugins[index].getName() === name) {
				return plugins[index];
			}
		}
	},

	/**
	 * Returns the Plugin Meta Data that matches the supplied name.
	 * @param {String} name The name of the plugin for which the meta data is requested
	 * @return {Zarafa.core.PluginMetaData} The Meta Data for the plugin or <code>undefined</code> if not found.
	 */
	getPluginMetaDataByName: function(name)
	{
		var plugins = this.getPluginsMetaData();

		for (var index = 0, len = plugins.length; index < len; index++) {
			if (plugins[index].getName() === name) {
				return plugins[index];
			}
		}
	},

	/**
	 * Returns the Widget Meta Data that matches the supplied name.
	 * @param {String} name The name of the widget for which the meta data is requested
	 * @return {Zarafa.core.ui.widget.WidgetMetaData} The Meta Data for the widget or <code>undefined</code> if not found.
	 */
	getWidgetMetaDataByName: function(name)
	{
		var widgets = this.getWidgetsMetaData();

		for (var index = 0, len = widgets.length; index < len; index++) {
			if (widgets[index].getName() === name) {
				return widgets[index];
			}
		}
	},

	/**
	 * Queries registered plug-ins in for components and returns the gathered results.
	 * @param {String} insertionPoint name of the insertion point
	 * @param {Object} args (optional) optional arguments such as scope
	 * @return {Ext.Component[]} an array of components
	 */
	populateInsertionPoint: function(insertionPoint)
	{
		var plugins = this.getPlugins();
		var items = [];

		// convert arguments object to a real array
		var args = Ext.toArray(arguments);

		for (var i = 0, len = plugins.length; i < len; i++) {
			var plugin = plugins[i];

			var components = plugin.getComponents.apply(plugin, args);

			// FIXME: Why do we need to assign the plugin to the component?
			Ext.each(components, function(component) {
				component.plugin = plugin;
				items.push(component);
			});
		}

		// every plugin will give items in their own array so we need to merge all arrays
		// this will not interfere with objects
		return Ext.flatten(items);
	},

	/**
	 * Registers a Context Meta Data instance with the container.
	 * @param {Zarafa.core.ContextMetaData} info context to register
	 */
	registerContext: function(info)
	{
		this.getContextsMetaData().push(info);
		if (info.isEnabled()) {
			this.getContexts().push(info.getInstance());
		}

		// A Context is also a plugin, so register it
		// as such as well.
		this.registerPlugin(info);
	},

	/**
	 * Registers a Plugin Meta Data instance with the container.
	 * @param {Zarafa.core.PluginMetaData} info plugin info to register
	 */
	registerPlugin: function(info)
	{
		// Get the list of plugins that are always enabled
		var alwaysEnabledPlugins = this.getServerConfig().getAlwaysEnabledPluginsList().split(';');
		if ( alwaysEnabledPlugins.indexOf(info.name)>=0 ){
			info.allowUserDisable = false;
			info.enable = true;
		}

		this.getPluginsMetaData().push(info);
		if (info.isEnabled()) {
			this.getPlugins().push(info.getInstance());
		}
	},

	/**
	 * Registers a Widget Meta Data instance with the container.
	 * @param {Zarafa.core.ui.widget.WidgetMetaData} info widget meta data to register
	 */
	registerWidget: function(info)
	{
		this.getWidgetsMetaData().push(info);
	},

	/**
	 * Performs a context switch by switching out the current context and switching in the new one.
	 * @param {Zarafa.core.Context} context context to switch to.
	 * @param {Zarafa.hierarchy.data.MAPIFolderRecord} folder folder that should be shown by the selected context.
	 * @param {Boolean} suspended True if the {@link Zarafa.core.ContextModel model} for the
	 * {@link Zarafa.core.Context context} should be enabled {@link Zarafa.core.ContextModel#suspendLoading suspended}.
	 * @private
	 */
	switchContext: function(context, folder, suspended)
	{
		var oldContext = this.getCurrentContext();

		if (oldContext !== context && this.fireEvent('beforecontextswitch', folder, oldContext, context) !== false) {
			if (oldContext) {
				oldContext.disable();

				var oldModel = oldContext.getModel();
				if (oldModel) {
					oldModel.un('folderchange', this.onContextFolderChange, this);
				}
			}

			context.enable(folder, suspended);
			var newModel = context.getModel();
			if (newModel) {
				newModel.on('folderchange', this.onContextFolderChange, this);
			}

			this.currentContext = context;

			this.fireEvent('folderselect', folder);
			this.fireEvent('contextswitch', folder, oldContext, context);

			// Nothing needs to be done between 'contextswitch' and 'aftercontextswitch',
			// the difference between the two events is that the first one can be used
			// internally for building up the UI, while the latter event is ideal for
			// plugins which want the UI components to be setup correctly.
			this.fireEvent('aftercontextswitch', folder, oldContext, context);
		}
	},

	/**
	 * The container will start a bidding round to determine which context should be chosen to
	 * display the given folder.
	 * @param {Zarafa.hierarchy.data.MAPIFolderRecord} folder folder for which we need to find corresponding context.
	 * @return {Zarafa.core.Context} context that should be used to load data from {@link Zarafa.hierarchy.data.MAPIFolderRecord MAPIFolder}.
	 */
	getContextByFolder: function(folder)
	{
		// walk over the context list and select the one that provides the highest bid
		var selectedContext;
		var highestBid;
		var contexts = this.getContexts();

		if (contexts) {
			for(var index = 0, len = contexts.length; index < len; index++) {
				var context = contexts[index];
				var bid = context.bid(folder);

				if (highestBid === undefined || bid > highestBid) {
					highestBid = bid;
					selectedContext = context;
				}
			}
		}

		return selectedContext;
	},

	/**
	 * Select a specific folder in the UI. The container will start a bidding round to determine which context should be chosen to
	 * display the given folder. The current context is then disabled and switched out, and the newly chosen context is enabled and
	 * switched in. Fires the 'folderselect' event.
	 * @param {Zarafa.hierarchy.data.MAPIFolderRecord[]} folder folder to select either as an array of
	 * {@link Zarafa.hierarchy.data.MAPIFolderRecord MAPIFolder} objects or a single object.
	 */
	selectFolder: function(folder)
	{
		var selectedContext = this.getContextByFolder(Ext.isArray(folder) ? folder[0] : folder);

		// Check if a new context has been selected, if we can
		// stay with the current context, then simply update
		// the folders.
		if (this.getCurrentContext() !== selectedContext) {
			this.switchContext(selectedContext, folder);
		} else {
			var model = selectedContext.getModel();
			if (model) {
				model.setFolders(folder);
			}
		}
	},

	/**
	 * Helps in reloading a context. It checks if given changed folder is holded by current context
	 * then re-enables context all of its folders.
	 * @param {Zarafa.hierarchy.data.MAPIFolderRecord} folder which is changed
	 */
	reloadContext: function(folder)
	{
		var currentContext = this.getCurrentContext();
		var contextModel = currentContext.getModel();

		if (!Ext.isDefined(contextModel)) {
			return;
		}

		folder = contextModel.getFolder(folder.get('entryid'));
		if (Ext.isDefined(folder)) {
			var allFolders = contextModel.getFolders();
			currentContext.disable();
			currentContext.enable(allFolders);
		}
	},

	/**
	 * Event handler which is fired when the {@link #getCurrentContext current context}
	 * {@link Zarafa.core.ContextModel model} fires the {@link Zarafa.core.ContextModel#folderchange} event.
	 * This will redirect the event and fire the {@link #folderselect} event.
	 * @param {Zarafa.core.ContextModel} model The model which fired the event
	 * @param {Zarafa.hierarchy.data.MAPIFolderRecord[]} folders The folders which are
	 * currently selected
	 * @private
	 */
	onContextFolderChange: function(model, folders)
	{
		this.fireEvent('folderselect', folders);
	},

	/**
	 * Starts a bidding round to determine what plug-in should be chosen to
	 * deliver the component that is requested. The component of the highest
	 * bidder is returned. If the supplied type is undefined the function will
	 * return undefined as well.
	 * @param {Zarafa.core.data.SharedComponentType} type Type of component a context can bid for.
	 * @param {Ext.data.Record} record (optional) Passed record.
	 * @return {Ext.Component} Component
	 *
	 */
	getSharedComponent: function(type, record)
	{
		// walk over the context list and select the one that provides the highest bid
		var selectedPlugin;
		var highestBid;
		var plugins = container.getPlugins();
		var component;

		if (type) {
			for (var i = 0, len = plugins.length; i < len; i++) {
				var plugin = plugins[i];
				var bid = plugin.bidSharedComponent(type, record);

				if (highestBid === undefined || bid > highestBid) {
					highestBid = bid;
					selectedPlugin = plugin;
				}
			}

			if (selectedPlugin && highestBid >= 0) {
				component = selectedPlugin.getSharedComponent(type, record);
			}
		}

		return component;
	},

	/**
	 * Returns the global {@link Zarafa.common.reminder.data.ReminderStore RemimderStore} instance.
	 * @return {Zarafa.common.reminder.data.ReminderStore} instance.
	 */
	getReminderStore: function()
	{
		return this.reminderStore || (this.reminderStore = new Zarafa.common.reminder.data.ReminderStore());
	},

	/**
	 * Returns base url of webapp without trailing slash, which can be used to request resources from server.
	 * @return {String} base url of webapp.
	 */
	getBaseURL: function()
	{
		var loc = window.location;
		var url = loc.protocol + '//' + loc.host + loc.pathname;

		// it's possible that webapp is loaded without index.php in url, so if it the case
		// then append index.php in url so that will make our base url complete
		if(url.indexOf('index.php') === -1) {
			url += 'index.php';
		}

		return url;
	},

	/**
	 * Returns base path of webapp with trailing slash, which can be used to request resources from server.
	 * @return {String} base path of webapp.
	 */
	getBasePath: function()
	{
		var baseURL = container.getBaseURL();

		return baseURL.substring(0, baseURL.lastIndexOf('index.php'));
	},

	/**
	 * Returns instance of factory registered with {@link Zarafa.common.data.AbstractRulesFactory} factory
	 * depending on factoryType given in the parameter.
	 *
	 * @param {Zarafa.common.data.RulesFactoryType} factoryType type of the required factory.
	 * @return {Object} factory instance based on factoryType.
	 */
	getRulesFactoryByType: function(factoryType)
	{
		return Zarafa.common.data.AbstractRulesFactory.getFactoryById(factoryType);
	},

	/**
	 * Helper function which used to check the conversation view is enabled or not.
	 * @return {Boolean} True if conversation view is enabled else false.
	 */
	isEnabledConversation: function ()
	{
		return this.getSettingsModel().get("zarafa/v1/contexts/mail/enable_conversation_view", false);
	}
});
Ext.namespace('Zarafa.core');

/**
 * @class Zarafa.core.DateRange
 * @extends Ext.util.Observable
 *
 * Represents a date range defined by a start and due date. The start date is inclusive, while the due date is exclusive. For example,
 * to denote an appointment that lasts all day on July 1st, 2010 one would write this as (00:00 July 1st 2010, 00:00 July 2nd 2010).
 * In sort, the range is defined as [startDate, dueDate>.
 * <p>
 * This class encapsulates such ranges because they are used often in especially the calendaring components.
 */
Zarafa.core.DateRange = Ext.extend(Ext.util.Observable, {
	/**
	 * @cfg {Date} startDate start date for this {@link Zarafa.core.DateRange DateRange}.
	 * use {@link #getStartDate} and {@link #setStartDate} to modify this value.
	 */
	startDate: null,

	/**
	 * @cfg {Date} dueDate due date for this {@link Zarafa.core.DateRange DateRange}.
	 * use {@link #getDueDate} and {@link #setDueDate} to modify this value.
	 */
	dueDate: null,

	/**
	 * @cfg {Boolean} allowBlank Specifies empty dates are accepted by this {@link Zarafa.core.DateRange DateRange},
	 * if {@link #allowBlank} is true then only we can use empty start/due dates in {@link Zarafa.core.DateRange DateRange}.
	 * otherwise {@link #startDate} and {@link #dueDate} will be initialized with current dates.
	 */
	allowBlank: false,

	/**
	 * @constructor
	 * @param {Date} startDate (optional) start date
	 * @param {Date} dueDate (optional) due date
	 */
	constructor: function(config)
	{
		config = config || {};

		Ext.applyIf(config, {
			allowBlank: this.allowBlank
		});

		if(config.allowBlank) {
			// if allowBlank is true then we should initialize start / due date with undefined
			// but this should only be done when start / due dates are not provided in the configs hence used Ext.applyIf
			// there is a restriction that start date can not exist without due date, so we will be initializing due date
			// same as start date if only start date is provided
			Ext.applyIf(config, {
				startDate: null,
				dueDate: config.startDate ? config.startDate.clone() : null
			});
		} else {
			// if allowBlank is false then we should initialize start / due date with current dates
			Ext.applyIf(config, {
				startDate: new Date(),
				dueDate: config.startDate ? config.startDate.clone() : new Date()
			});
		}

		Ext.apply(this, config);

		// Precondition
		if (Ext.isDate(this.startDate) && Ext.isDate(this.dueDate) && (this.getStartTime() > this.getDueTime())) {
			throw 'Invalid date range, start date is after due date';
		}

		this.addEvents(
			/**
			 * @event update
			 * Fires when the daterange is modified.
			 * @param {Zarafa.core.DateRange} newRange The changed daterange object
			 * @param {Zarafa.core.DateRange} oldRange The original daterange values (clone of the daterange object,
			 * prior of the change).
			 */
			'update'
		);

		Zarafa.core.DateRange.superclass.constructor.call(this);
	},

	/**
	 * @return {Date} the range's start date.
	 */
	getStartDate: function()
	{
		return this.startDate;
	},

	/**
	 * @return {Date} the range's due date.
	 */
	getDueDate: function()
	{
		return this.dueDate;
	},

	/**
	 * Sets the range start date.
	 * @param {Date} startDate the range's start date.
	 * @param {Boolean} silence When set to true it will not throw an start date after due date error.
	 * @param {Boolean} ignoreUpdate When set to true it will not fire the update event.
	 * @return {Zarafa.core.DateRange} this date range
	 */
	setStartDate: function(startDate, silence, ignoreUpdate)
	{
		var original = null;

		// make sure that you are saving null when no valid date is passed
		startDate = Ext.isDate(startDate) ? startDate : null;

		// Preconditions
		if (!this.allowBlank && !Ext.isDate(startDate)) {
			throw 'Cannot set DateRange start to undefined';
		} else if (Ext.isDate(this.dueDate) && Ext.isDate(startDate) && startDate.getTime() > this.getDueTime() && !silence) {
			throw 'Cannot set DateRange start date to after its due date';
		}

		// No update event when new value equals old value
		ignoreUpdate = ignoreUpdate || (Ext.isDate(this.startDate) && Ext.isDate(startDate) && this.getStartTime() == startDate.getTime());

		if (!ignoreUpdate) {
			original = this.clone();
		}

		this.startDate = startDate;

		if(!ignoreUpdate) {
			this.fireEvent("update", this, original);
		}

		return this;
	},

	/**
	 * Sets the range due date.
	 * @param {Date} dueDate the range's due date.
	 * @param {Boolean} silence When set to true it will not throw an start date after due date error.
	 * @param {Boolean} ignoreUpdate When set to true it will not fire the update event.
	 * @return {Zarafa.core.DateRange} this date range
	 */
	setDueDate: function(dueDate, silence, ignoreUpdate)
	{
		var original = null;

		// make sure that you are saving null when no valid date is passed
		dueDate = Ext.isDate(dueDate) ? dueDate : null;

		// Precondition
		if (!this.allowBlank && !Ext.isDate(dueDate)) {
			throw 'Cannot set DateRange due date to undefined';
		} else if (Ext.isDate(this.startDate) && Ext.isDate(dueDate) && dueDate.getTime() < this.getStartTime() && !silence) {
			throw 'Cannot set DateRange due date to before its start date';
		}

		// No update event when new value equals old value
		ignoreUpdate = ignoreUpdate || (Ext.isDate(this.dueDate) && Ext.isDate(dueDate) && this.getDueTime() == dueDate.getTime());

		if (!ignoreUpdate) {
			original = this.clone();
		}

		this.dueDate = dueDate;

		if(!ignoreUpdate) {
			this.fireEvent("update", this, original);
		}

		return this;
	},

	/**
	 * Sets the start and due dates.
	 * @param {Date} startDate the range's start date.
	 * @param {Date} dueDate the range's due date.
	 * @param {Boolean} silence When set to true it will not throw an start date after due date error.
	 * @param {Boolean} ignoreUpdate When set to true it will not fire the update event.
	 * @return {Zarafa.core.DateRange} this date range
	 */
	set: function(startDate, dueDate, silence, ignoreUpdate)
	{
		var original = null;

		// make sure that you are saving null when no valid date is passed
		startDate = Ext.isDate(startDate) ? startDate : null;
		dueDate = Ext.isDate(dueDate) ? dueDate : null;

		// Precondition
		if (!this.allowBlank && !Ext.isDate(startDate)) {
			throw 'Cannot set DateRange start to undefined';
		}

		if (!this.allowBlank && !Ext.isDate(dueDate)) {
			throw 'Cannot set DateRange due date to undefined';
		}

		if (Ext.isDate(startDate) && Ext.isDate(dueDate) && startDate.getTime() > dueDate.getTime() && !silence) {
			throw 'Invalid date range, start date is after due date';
		}

		ignoreUpdate = ignoreUpdate || (Ext.isDate(startDate) && Ext.isDate(dueDate) && (startDate.getTime() == this.getStartTime()) && (dueDate.getTime() == this.getDueTime()));

		if (!ignoreUpdate) {
			original = this.clone();
		}

		// don't fire update event from these functions
		this.setStartDate(startDate, true, true);
		this.setDueDate(dueDate, true, true);

		if (!ignoreUpdate) {
			this.fireEvent("update", this, original);
		}

		return this;
	},

	/**
	 * @return {Number} the range's start time in milliseconds since epoch, GMT.
	 */
	getStartTime: function()
	{
		return (Ext.isDate(this.startDate) && this.startDate.getTime()) || this.startDate;
	},

	/**
	 * @return {Number} the range's due time in milliseconds since epoch, GMT.
	 */
	getDueTime: function()
	{
		return (Ext.isDate(this.dueDate) && this.dueDate.getTime()) || this.dueDate;
	},

	/**
	 * Sets the range start time.
	 * @param {Number} startTime the range's start time.
	 * @param {Boolean} silence When set to true it will not throw an start date after due date error.
	 * @param {Boolean} ignoreUpdate When set to true it will not fire the update event.
	 * @return {Zarafa.core.DateRange} this date range
	 */
	setStartTime: function(startTime, silence, ignoreUpdate)
	{
		var original = null;

		// Preconditions
		if (!this.allowBlank && !startTime) {
			throw 'Cannot set DateRange start to undefined';
		} else if (Ext.isDate(this.dueDate) && startTime > this.dueDate.getTime() && !silence) {
			throw 'Cannot set DateRange start date to after its due date';
		}

		// No update event when new value equals old value
		ignoreUpdate = ignoreUpdate || (this.getStartTime() == startTime);

		if (!ignoreUpdate) {
			original = this.clone();
		}

		if (!Ext.isEmpty(startTime)) {
			this.startDate = new Date(startTime);
		} else {
			this.startDate = null;
		}

		if(!ignoreUpdate) {
			this.fireEvent("update", this, original);
		}

		return this;
	},

	/**
	 * Sets the range due time.
	 * @param {Number} dueTime the range's due time.
	 * @param {Boolean} silence When set to true it will not throw an start date after due date error.
	 * @param {Boolean} ignoreUpdate When set to true it will not fire the update event.
	 * @return {Zarafa.core.DateRange} this date range
	 */
	setDueTime: function(dueTime, silence, ignoreUpdate)
	{
		var original = null;

		// Precondition
		if (!this.allowBlank && !dueTime) {
			throw 'Cannot set DateRange due date to undefined';
		} else if (Ext.isDate(this.startDate) && dueTime < this.startDate.getTime() && !silence) {
			throw 'Cannot set DateRange due date to before its start date';
		}

		// No update event when new value equals old value
		ignoreUpdate = ignoreUpdate || (this.getDueTime() == dueTime);

		if (!ignoreUpdate) {
			original = this.clone();
		}

		if (!Ext.isEmpty(dueTime)) {
			this.dueDate = new Date(dueTime);
		} else {
			this.dueDate = null;
		}

		if(!ignoreUpdate) {
			this.fireEvent("update", this, original);
		}

		return this;
	},

	/**
	 * Sets the start and due times.
	 * @param {Number} startDate the range's start time.
	 * @param {Number} dueDate the range's due time.
	 * @param {Boolean} silence When set to true it will not throw an start date after due date error.
	 * @param {Boolean} ignoreUpdate When set to true it will not fire the update event.
	 * @return {Zarafa.core.DateRange} this date range
	 */
	setTime: function(startTime, dueTime, silence, ignoreUpdate)
	{
		var original = null;

		// Precondition
		if(!startTime || !dueTime) {
			if (!this.allowBlank && !startTime) {
				throw 'Cannot set DateRange start to undefined';
			}

			if (!this.allowBlank && !dueTime) {
				throw 'Cannot set DateRange due date to undefined';
			}
		} else if (startTime > dueTime && !silence) {
			throw 'Invalid date range, start date is after due date';
		}

		ignoreUpdate = ignoreUpdate || ((startTime == this.getStartTime()) && (dueTime == this.getDueTime()));

		if (!ignoreUpdate) {
			original = this.clone();
		}

		this.setStartTime(startTime, true, true);
		this.setDueTime(dueTime, true, true);

		if (!ignoreUpdate) {
			this.fireEvent("update", this, original);
		}

		return this;
	},

	/**
	 * @param {String} interval (optional) A valid date interval enum value.
	 * @return {Number} the range's duration in milliseconds
	 */
	getDuration: function(interval)
	{
		if (Ext.isDate(this.dueDate) && Ext.isDate(this.startDate)) {
			return Date.diff(interval || Date.MILLI, this.dueDate, this.startDate);
		}

		return 0;
	},

	/**
	 * @param {Number} duration the new duration of the range in milliseconds.
	 * @param {Boolean} ignoreUpdate When set to true it will not fire the update event.
	 * @return {Zarafa.core.DateRange} this date range
	 */
	setDuration: function(duration, ignoreUpdate)
	{
		var original = null;

		if(!Ext.isDate(this.startDate)) {
			// if no start date is provided then we can't set duration
			throw 'Cannot set duration when start date is not specified';
		}

		// No update event when new value equals old value
		ignoreUpdate = ignoreUpdate || (this.getDuration() == duration);

		if (!ignoreUpdate) {
			original = this.clone();
		}

		this.dueDate = new Date(this.getStartTime() + duration);

		if (!ignoreUpdate) {
			this.fireEvent("update", this, original);
		}

		return this;
	},

	/**
	 * Calculates the number of days spanned by this appointment, rounded to whole days. The function
	 * assumes that the range is an all day range. Even so, the rounding is still required to deal with
	 * date ranges that start and end in different time zones.
	 *
	 * @return {Number} the number of days spanned by this appointment, rounded to whole days.
	 */
	getNumDays: function()
	{
		var duration = this.getDuration(Date.DAY);
		if (Ext.isDefined(duration)) {
			return Math.round(duration);
		} else {
			return 0;
		}
	},

	/**
	 * Expands the range so that both the start and due times are multiples of the 'timeSlice' parameter. The start
	 * date is moved back, the due date is moved forward. Since this method does not care about time zones the value
	 * of 'timeSlice' is assumed to be <= 60 minutes.
	 *
	 * @param {Number} timeSlice the time slice to 'snap' to.
	 * @return {Zarafa.core.DateRange} this date range
	 */
	expand: function(timeSlice)
	{
		var original = this.clone();

		if(Ext.isDate(this.startDate)) {
			this.startDate = new Date(this.getStartTime() - this.getStartTime() % timeSlice);
		}

		if(Ext.isDate(this.dueDate)) {
			this.dueDate = new Date(this.getDueTime() + timeSlice - this.getDueTime() % timeSlice);
		}

		// only fire event if anything has changed
		if(Ext.isDate(this.startDate) || Ext.isDate(this.dueDate)) {
			this.fireEvent("update", this, original);
		}

		return this;
	},

	/**
	 * Deep-clones the date range.
	 * @return {Zarafa.core.DateRange} a clone of this date range.
	 */
	clone: function()
	{
		return new Zarafa.core.DateRange({
			startDate: Ext.isDate(this.startDate) ? new Date(this.getStartTime()) : undefined,
			dueDate: Ext.isDate(this.dueDate) ? new Date(this.getDueTime()) : undefined,
			allowBlank: this.allowBlank
		});
	},

	/**
	 * Test this date range for equality against another date range.
	 * @param {Zarafa.core.DateRange} otherRange a date range to compare with.
	 * @return {Boolean} true if this range equals the given other range.
	 */
	equals: function(otherRange)
	{
		if (!otherRange) {
			return false;
		}

		if(this.getStartTime() !== otherRange.getStartTime()) {
			// start dates don't match
			return false;
		}

		if(this.getDueTime() !== otherRange.getDueTime()) {
			// due dates don't match
			return false;
		}

		// both start/due dates matches
		return true;
	},

	/**
	 * Compares two date ranges for order. If range A comes before B, the function returns -1. If B comes before
	 * A, the function returns 1. If both are equal, this function returns 0. This functionality is equivalent
	 * to Java's Comparable interface.
	 *
	 * Comparison is based on the range start. If the start dates are equal, further distinction is made by
	 * due date, with earlier due dates coming first.
	 *
	 * Undefined dates will be considered as zero and compared.
	 *
	 * @param {Zarafa.core.DateRange} otherRange Date range to compare with.
	 * @return {Number} If this range 'comes before' otherRange, the function returns -1. If this range 'comes
	 * after' otherRange, return 1. Otherwise return 0.
	 *
	 */
	compare: function(otherRange)
	{
		// Compare start times.
		var aStartTime = this.getStartTime() || 0;
		var bStartTime = otherRange.getStartTime() || 0;

		if (aStartTime !== bStartTime) {
			return aStartTime > bStartTime ? 1: -1;
		}

		// If start times are equal, compare due times.
		var aDueTime = this.getDueTime() || 0;
		var bDueTime = otherRange.getDueTime() || 0;

		if (aDueTime !== bDueTime) {
			return aDueTime > bDueTime ? 1: -1;
		}

		// If ranges are equal, return 0.
		return 0;
	},

	/**
	 * Tests whether the date range is a so-called 'all day' range,
	 * meaning that start and due date duration time is more then one day(24 hours).
	 * @return {Boolean} true if the range is an all day range.
	 */
	isAllDay: function()
	{
		if(Ext.isDate(this.startDate) && Ext.isDate(this.dueDate) && !this.isZeroMinuteRange()) {
			return (this.startDate.clearTime(true).getTime() === this.getStartTime())
			&& (this.dueDate.clearTime(true).getTime() === this.getDueTime());
		}

		return false;
	},

	/**
	 * Tests whether the date range is a 0 minute.
	 * @return {Boolean} true if the range duration is 0 minute.
	 */
	isZeroMinuteRange: function()
	{
		return (this.dueDate.getTime() === this.startDate.getTime());
	},

	/**
	 * @return {Boolean} true if this date range overlaps with the other date range.
	 */
	overlaps: function(other)
	{
		var start1 = this.getStartTime();
		var due1 = this.getDueTime();
		var start2 = other.getStartTime();
		var due2 = other.getDueTime();

		return (start1 >= start2 && start1 < due2) || (start2 >= start1 && start2 < due1);
	},

	/**
	 * @param {Zarafa.core.DateRange} range date range to check against.
	 * @return {Boolean} true if this date range is inside the given date range.
	 */
	inside: function(range)
	{
		if(this.getStartTime() && range.getStartTime() && this.getDueTime() && range.getDueTime()) {
			return this.getStartTime() >= range.getStartTime() && this.getDueTime() <= range.getDueTime();
		}

		return false;
	},

	/**
	 * @param {Date} date the date to test.
	 * @return {Boolean} true if the give date is inside this date range.
	 */
	containsDate: function(date)
	{
		return this.getStartTime() <= date.getTime() && this.getDueTime() > date.getTime();
	},

	/**
	 * Formats the current visible date range as human-readable text. The formatter looks at which components the
	 * start and due dates have in common.
	 * For instance, the first and last days of a week range might lie in the same month (i.e. '13 - 19 July 2009'),
	 * or they might not (i.e. '28 September - 2 November 2009'). Finally a range may have the start and end dates
	 * in different years, i.e. '28 December 2009 - 1 January 2010'.
	 *
	 * @return {String} the current date range as text.
	 */
	format: function()
	{
		var startDate = this.startDate;
		var dueDate = this.dueDate;

		if(!Ext.isDate(startDate) || !Ext.isDate(dueDate)) {
			return '';
		}

		// If the due date is _exactly_ midnight, we must assume the last day of the period
		// is the previous day. So decrease the duedate with a day (making sure the duedate
		// does not move before the startDate).
		if (dueDate.getTime() === dueDate.clearTime(true).getTime()) {
			// Move to the previous day, use 12:00 as starting hour,
			// to prevent problems when the DST switches at 00:00 (e.g. in Brasil).
			// We don't need to restore to the original time, as the string
			// which we are going to ignore doesn't contain a time representation.
			dueDate = dueDate.clone();
			dueDate.setHours(12);
			dueDate = dueDate.add(Date.DAY, -1);

			if (dueDate.getTime() < startDate.getTime()) {
				dueDate = startDate;
			}
		}

		// The startDate and duedate are in completely different years.
		// Format the full date strings for both dates.
		if (startDate.getYear() != dueDate.getYear()) {
			// # TRANSLATORS: See http://docs.sencha.com/extjs/3.4.0/#!/api/Date for the meaning of these formatting instructions
			return String.format('{0} - {1}', startDate.format(_('jS F Y')), dueDate.format(_('jS F Y')));
		}

		// The startDate and dueDate are in different months.
		// Format the date strings with the year in common.
		if (startDate.getMonth() != dueDate.getMonth()) {
			// # TRANSLATORS: See http://docs.sencha.com/extjs/3.4.0/#!/api/Date for the meaning of these formatting instructions
			return String.format('{0} - {1} {2}', startDate.format(_('jS F')), dueDate.format(_('jS F')), startDate.format(_('Y')));
		}

		// The startDate and dueDate are on different days.
		// Format the date strings with the month and year in common.
		if (startDate.getDate() != dueDate.getDate()) {
			// # TRANSLATORS: See http://docs.sencha.com/extjs/3.4.0/#!/api/Date for the meaning of these formatting instructions
			return String.format('{0} - {1} {2}', startDate.format(_('jS')), dueDate.format(_('jS')), startDate.format(_('F Y')));
		}

		// The startDate and dueDate are on the same day.
		// Format the date string with everything in common.
		// # TRANSLATORS: See http://docs.sencha.com/extjs/3.4.0/#!/api/Date for the meaning of these formatting instructions
		return startDate.format(_('jS F Y'));
	}
});
Ext.namespace('Zarafa.core');

/**
 * @class Zarafa.core.EntryId
 *
 * Class for decoding entryids and for comparison between two entryids
 * we have basically two types of entryids object and store entryids.
 *
 * object entryids uses structure EID for entryids created
 * and for older entryids it uses structure EID_V0. Store entryids are generally wrapped
 * with some extra information (like guid for provider, dll name) which should be removed
 * before comparing two store entryids, after removing this wrapping the unwrapped entryid
 * uses the format same as object entryids (EID or EID_V0).
 *
 * version flag in EID and EID_V0 are specific flags and indicate which structure is used
 * to create that entryid, EID always contains version as '01000000' and EID_V0 always contains
 * '00000000' as version flag.
 *
 * server part of EID and EID_V0 indicates server name and it can be variable length, padding can be
 * upto 3 bytes so it can be anything between 0 to 3 bytes.
 *
 * in public store public root folder, ipm_subtree and favorites folder are custom folders
 * so they have static uniqueids.
 *
 * @singleton
 */
Zarafa.core.EntryId = (function()
{
	/* Bit definitions for abFlags[3] of ENTRYID */
	var ZARAFA_FAVORITE = '01';

	/* GUID of root public folder */
	var STATIC_GUID_PUBLICFOLDER = '00000000000000000000000000000003';
	/* GUID of root favorite folder */
	var STATIC_GUID_FAVORITE = '00000000000000000000000000000002';
	/* GUID of ipm_subtree of public store*/
	var STATIC_GUID_FAVSUBTREE = '00000000000000000000000000000001';
	/* GUID of Global Addressbook */
	var MUIDECSAB = 'AC21A95040D3EE48B319FBA753304425';
	/* GUID of Contact Provider */
	var MUIDZCSAB = '727F0430E3924FDAB86AE52A7FE46571';
	/* GUID for OneOff entryid */
	var MAPI_ONE_OFF_UID = '812B1FA4BEA310199D6E00DD010F5402';
	/* GUID for Address book recipient */
	var MUIDEMSAB = 'DCA740C8C042101AB4B908002B2FE182';

	/* Hardcoded ID used for generating entryid of addressbook container */
	var ZARAFA_UID_GLOBAL_ADDRESS_BOOK = '01000000';

	var BASE_EID = Ext.extend(Object, {

		// The entryid which this object represents
		entryId: '',

		// The length of the entryid
		length: 0,

		// Constructor
		// param: Entryid The entryid represented by this object
		constructor: function(entryId)
		{
			if(entryId) {
				// always make entryids in uppercase so comparison will be case insensitive
				this.entryId = entryId.toUpperCase();
				this.length = entryId.length;

				this.decomposeEntryId(this.entryId);
			}
		},

		// Detect padding (max 3 bytes) from the entryId
		getPadding: function(entryId)
		{
			var padding = '';
			var offset = 0;

			for (var iterations = 4; iterations > 0; iterations--) {
				if (entryId.substring(entryId.length - (offset + 2), entryId.length - offset) === '00') {
					padding += '00';
					offset += 2;
				} else {
					// if non-null character found then break the loop
					break;
				}
			}

			return padding;
		}

	});

	// Entryid from version 6
	var EID = Ext.extend(BASE_EID, {
		abFlags: '',      // BYTE[4],  4 bytes, 8 hex characters
		guid: '',       // GUID,   16 bytes, 32 hex characters
		version: '',      // ULONG,   4 bytes, 8 hex characters
		type: '',       // ULONG,   4 bytes, 8 hex characters
		uniqueId: '',     // GUID,   16 bytes, 32 hex characters
		server: '',      // CHAR,   variable length
		padding: '',      // TCHAR[3], 4 bytes, 8 hex characters (upto 4 bytes)

		MIN_LENGTH: 88,
		name: 'EID',

		// decompose the entryid and populate all flags of entryid
		decomposeEntryId: function(entryId)
		{
			var offset = 0;

			// First determine padding, and remove if from the entryId
			this.padding = this.getPadding(entryId);
			entryId = entryId.substring(0, entryId.length - this.padding.length);

			this.abFlags = entryId.substr(offset, 8);
			offset =+ 8;

			this.guid = entryId.substr(offset, 32);
			offset += 32;

			this.version = entryId.substr(offset, 8);
			offset += 8;

			this.type = entryId.substr(offset, 8);
			offset += 8;

			this.uniqueId = entryId.substr(offset, 32);
			offset += 32;

			this.server = entryId.substr(offset);
		}
	});

	// The entryid from the begin
	var EID_V0 = Ext.extend(BASE_EID, {
		abFlags: '',      // BYTE[4],  4 bytes, 8 hex characters
		guid: '',       // GUID,   16 bytes, 32 hex characters
		version: '',      // ULONG,   4 bytes, 8 hex characters
		type: '',       // ULONG,   4 bytes, 8 hex characters
		id: '',        // ULONG,   4 bytes, 8 hex characters
		server: '',      // CHAR,   variable length
		padding: '',      // TCHAR[3], 4 bytes, 8 hex characters (upto 4 bytes)

		MIN_LENGTH: 64,
		name: 'EID_V0',

		// decompose the entryid and populate all flags of entryid
		decomposeEntryId: function(entryId)
		{
			var offset = 0;

			// First determine padding, and remove if from the entryId
			this.padding = this.getPadding(entryId);
			entryId = entryId.substring(0, entryId.length - this.padding.length);

			this.abFlags = entryId.substr(offset, 8);
			offset =+ 8;

			this.guid = entryId.substr(offset, 32);
			offset += 32;

			this.version = entryId.substr(offset, 8);
			offset += 8;

			this.type = entryId.substr(offset, 8);
			offset += 8;

			this.id = entryId.substr(offset, 8);
			offset += 8;

			this.server = entryId.substr(offset);
		}
	});

	// wrapped store entryid
	var WrappedSEID = Ext.extend(BASE_EID, {
		flags: '',       // BYTE[4],   4 bytes, 8 hex characters
		providerUID: '',    // GUID,    16 bytes, 32 hex characters
		version: '',      // ULONG,    1 bytes, 2 hex characters	// zero
		type: '',       // ULONG,    1 bytes, 2 hex characters	// zero
		DLLFileName: '',    // BYTE,    variable length
		terminationChar: '',  // BYTE[1],   1 bytes, 2 hex characters	// zero
		unWrappedEntryId: '', // EID/EID_V0, variable length because it contains server name

		name: 'WrappedSEID',

		// decompose the entryid and populate all flags of entryid
		decomposeEntryId: function(storeEntryId)
		{
			var offset = 0;

			this.flags = storeEntryId.substr(offset, 8);
			offset += 8;

			this.providerUID = storeEntryId.substr(offset, 32);
			offset += 32;

			this.version = storeEntryId.substr(offset, 2);
			offset += 2;

			this.type = storeEntryId.substr(offset, 2);
			offset += 2;

			// find length of dll name, find null character which indicates end of dll name after the current offset
			var termCharIndex = storeEntryId.slice(offset).indexOf('00');
			this.DLLFileName = storeEntryId.substr(offset, termCharIndex);
			offset += termCharIndex;

			this.terminationChar = storeEntryId.substr(offset, 2);
			offset += 2;

			this.unWrappedEntryId = storeEntryId.substr(offset);

			// unwrapped entryid is actually an object entryid so decompose it
			this.unWrappedEntryId = Zarafa.core.EntryId.createEntryIdObj(this.unWrappedEntryId);
		}
	});

	// The entryid for addressbook items
	var ABEID = Ext.extend(BASE_EID, {
		abFlags: '',      // BYTE[4],  4 bytes, 8 hex characters
		guid: '',       // GUID,   16 bytes, 32 hex characters
		version: '',      // ULONG,   4 bytes, 8 hex characters
		type: '',       // ULONG,   4 bytes, 8 hex characters
		id: '',        // ULONG,   4 bytes, 8 hex characters
		extid: '',       // CHAR,   variable length
		padding: '',      // TCHAR[3], 4 bytes, 8 hex characters (upto 4 bytes)

		MIN_LENGTH: 64,
		name: 'ABEID',

		// decompose the entryid and populate all flags of entryid
		decomposeEntryId: function(entryId)
		{
			var offset = 0;

			// First determine padding, and remove if from the entryId
			this.padding = this.getPadding(entryId);
			entryId = entryId.substring(0, entryId.length - this.padding.length);

			this.abFlags = entryId.substr(offset, 8);
			offset =+ 8;

			this.guid = entryId.substr(offset, 32);
			offset += 32;

			this.version = entryId.substr(offset, 8);
			offset += 8;

			this.type = entryId.substr(offset, 8);
			offset += 8;

			this.id = entryId.substr(offset, 8);
			offset += 8;

			this.extid = entryId.substr(offset);
		}
	});

	// The entryid for local addressbook items
	var WrappedABEID = Ext.extend(BASE_EID, {
		ulVersion: '',     // ULONG,   4 bytes, 8 hex characters
		muid: '',       // MAPIUID, 16 bytes, 32 hex characters
		ulObjType: '',     // ULONG,   4 bytes, 8 hex characters
		ulOffset: '',     // ULONG,   4 bytes, 8 hex characters
		unWrappedEntryId: '', // EID/EID_V0, variable length because it contains server name

		name: 'WrappedABEID',

		// decompose the entryid and populate all flags of entryid
		decomposeEntryId: function(ABEntryId)
		{
			var offset = 0;

			this.ulVersion = ABEntryId.substr(offset, 8);
			offset += 8;

			this.muid = ABEntryId.substr(offset, 32);
			offset += 32;

			this.ulObjType = ABEntryId.substr(offset, 8);
			offset += 8;

			this.ulOffset = ABEntryId.substr(offset, 8);
			offset += 8;

			this.unWrappedEntryId = ABEntryId.substr(offset);

			// unwrapped entryid is actually an object entryid so decompose it
			this.unWrappedEntryId = Zarafa.core.EntryId.createEntryIdObj(this.unWrappedEntryId);
		}
	});

	// Wrap an entryid into a Contact Provider entryid
	// @static
	WrappedABEID.wrapABEID = function(entryId, objType)
	{
		objType = objType.toString(16);

		// add padding for the type, which is of 4 bytes (8 characters)
		objType = objType.padStart(2, '0');
		objType = objType.padEnd(8, '0');

		return '00000000' + MUIDZCSAB + objType + '00000000' + entryId;
	};

	// Unwrap an Contact Provider entryid
	// @static
	WrappedABEID.unwrapABEID = function(entryId)
	{
		// Remove ulVersion (8 char), muid (32 char), ulObjType (8 char) and ulOffset (8 char)
		return entryId.substring(56);
	};

	return {
		/**
		 * Compares two AB entryIds. It is possible to have two different entryIds that should match as they
		 * represent the same object (in multiserver environments).
		 * @param {String} entryId1 EntryID
		 * @param {String} entryId2 EntryID
		 * @return {Boolean} Result of the comparison
		 */
		compareABEntryIds: function(entryId1, entryId2)
		{
			if(!Ext.isString(entryId1) || !Ext.isString(entryId2)) {
				return false;
			}

			if(entryId1 === entryId2) {
				// if normal comparison succeeds then we can directly say that entryids are same
				return true;
			}
			return false;
		},

		/**
		 * Creates an object that has split up all the components of an entryID.
		 * @param {String} entryid Entryid
		 * @return {Object} EntryID object
		 */
		createEntryIdObj: function(entryid)
		{
			// check if we are dealing with old or new object entryids
			var versionString = entryid.substr(40, 8);
			var eidObj;

			if(versionString === '00000000') {
				// use EID_V0 struct
				eidObj = new EID_V0(entryid);
			} else {
				// use EID struct
				eidObj = new EID(entryid);
			}

			return eidObj;
		},

		/**
		 * Compares two entryIds. It is possible to have two different entryIds that should match as they
		 * represent the same object (in multiserver environments).
		 * @param {String} entryId1 EntryID
		 * @param {String} entryId2 EntryID
		 * @return {Boolean} Result of the comparison
		 */
		compareEntryIds: function(entryId1, entryId2)
		{
			if(!Ext.isString(entryId1) || !Ext.isString(entryId2)) {
				return false;
			}

			if(entryId1 === entryId2) {
				// if normal comparison succeeds then we can directly say that entryids are same
				return true;
			}
			return false;
		},

		/**
		 * Creates an object that has split up all the components of a store entryid.
		 * @param {String} storeEntryId unwrapped store entryid.
		 * @return {Object} store entryid object.
		 */
		createStoreEntryIdObj: function(storeEntryId)
		{
			return new WrappedSEID(storeEntryId);
		},

		/**
		 * Compares two entryIds. It is possible to have two different entryIds that should match as they
		 * represent the same object (in multiserver environments).
		 * @param {String} storeEntryId1 store entryid
		 * @param {String} storeEntryId2 store entryid
		 * @return {Boolean} Result of the comparison
		 */
		compareStoreEntryIds: function(storeEntryId1, storeEntryId2)
		{
			if(!Ext.isString(storeEntryId1) || !Ext.isString(storeEntryId2)) {
				return false;
			}

			if(storeEntryId1 === storeEntryId2) {
				// if normal comparison succeeds then we can directly say that entryids are same
				return true;
			}
			return false;
		},

		/**
		 * Unwrap an Entryid which is of the Contact Provider ({@link #hasContactProviderGUID}
		 * returned true for this entryid}.
		 * @param {String} entryId the Address Book entryid.
		 * @return {String} The unwrapped entryId
		 */
		unwrapContactProviderEntryId: function(entryId)
		{
			return WrappedABEID.unwrapABEID(entryId);
		},

		/**
		 * Wrap an EntryId which should be wrapped using the Contact Provider
		 * @param {String} entryId The entryid
		 * @return {String} The wrapped entryId
		 */
		wrapContactProviderEntryId: function(entryId, objType)
		{
			return WrappedABEID.wrapABEID(entryId, objType);
		},

		/**
		 * Create a one-off entryid from the applied parameters.
		 * @param {String} displayname displaye name as configured in record.
		 * @param {String} addrtype weather the record is of type SMTP.
		 * @param {String} emailaddress email address as configured in record.
		 * @return {String} The oneoff entryId
		 */
		createOneOffEntryId: function(displayname, addrtype, emailaddress)
		{
			return '00000000' + MAPI_ONE_OFF_UID + '00000080' + Zarafa.core.Util.encode_utf16(displayname) + '0000' + Zarafa.core.Util.encode_utf16(addrtype) + '0000' + Zarafa.core.Util.encode_utf16(emailaddress) + '0000';
		},

		/**
		 * Checks if the passed folder entryid is a folder in the favorites folder, favorites folder
		 * contains 0x01 in the abFlags[3] flag.
		 * @param {String} entryId folder entryid
		 * @return {Boolean} true of folder is a favorite folder else false
		 */
		isFavoriteFolder: function(entryId)
		{
			var entryIdObj = Zarafa.core.EntryId.createEntryIdObj(entryId);

			return (entryIdObj.abFlags.substr(6, 8) === ZARAFA_FAVORITE);
		},

		/**
		 * Checks if the given entryid is a oneoff entryid.
		 * @param {String} entryId The entryid
		 * @return {Boolean} true if the entryid is a oneoff
		 */
		isOneOffEntryId: function(entryId)
		{
			var entryIdObj = Zarafa.core.EntryId.createEntryIdObj(entryId);

			return entryIdObj.guid === MAPI_ONE_OFF_UID;
		},

		/**
		 * Checks if the GUID part of the entryid is of the Contact Provider.
		 * @param {String} entryId Address Book entryid
		 * @return {Boolean} true if guid matches the Contact Provider else false
		 */
		hasContactProviderGUID: function(entryId)
		{
			var entryIdObj = Zarafa.core.EntryId.createEntryIdObj(entryId);

			return entryIdObj.guid === MUIDZCSAB;
		},

		/**
		 * Checks if the GUID part of the entryid is of the AB Provider.
		 * @param {String} entryId Address Book entryid
		 * @return {Boolean} true if guid matches the AB Provider else false
		 */
		hasABProviderGUID: function(entryId)
		{
			var entryIdObj = Zarafa.core.EntryId.createEntryIdObj(entryId);

			return entryIdObj.guid === MUIDEMSAB;
		},

		/**
		 * Format an entryid into the Object ID text shown in the mail properties dialog.
		 * @param {String} entryId Entryid of the message record
		 * @return {String|undefined} Object ID representation or undefined when formatting fails
		 */
		formatObjectId: function(entryId)
		{
			if (!Ext.isString(entryId) || entryId.length < 136) {
				return undefined;
			}

			var provider = entryId.substr(8, 32);
			var folderGcvHex = entryId.substr(76, 12);
			var databaseGuid = entryId.substr(92, 32);
			var messageGcvHex = entryId.substr(124, 12);

			var folderGcv = parseInt(folderGcvHex, 16);
			var messageGcv = parseInt(messageGcvHex, 16);

			if (isNaN(folderGcv) || isNaN(messageGcv)) {
				return undefined;
			}

			return messageGcv + '/0x' + messageGcv.toString(16) +
				'; folder=' + folderGcv + '/0x' + folderGcv.toString(16) +
				'; dbguid=' + databaseGuid + '; store=' + provider;
		},
	};
})();
Ext.namespace('Zarafa.core');

/**
 * @class Zarafa.core.Enum
 * @extends Object
 *
 * Base class for Enumerations, which are defined as <String, Number> hash maps.
 * Provides a set of get methods for getting properties or property names by value.
 */
Zarafa.core.Enum = Ext.extend(Object, {

	/**
	 * Get a property by value.
	 * @param {Number} value
	 * @return the property value (equal to the input) iff a property with the given value exists. Returns undefined otherwise.
	 */
	get: function(value)
	{
		value = parseInt(value, 10);
		for (var key in this) {
			if (this[key]==value) {
				return this[key];
			}
		}
	},

	/**
	 * Gets a property name by value.
	 * @param {Number} value
	 * @return the property name iff a property with the given value exists. Returns undefined otherwise.
	 */
	getName: function(value)
	{
		value = parseInt(value, 10);
		for (var key in this) {
			if (this[key]==value) {
				return key;
			}
		}
	},

	/**
	 * Gets a property value by name.
	 * @param {String} key key of the <String, Number> hashmap
	 * @return {Number} the property value corresponding to string key
	 */
	getValue: function(key)
	{
		var value = this[key];

		if(Ext.isNumber(value)) {
			return value;
		}
	},

	/**
	 * Adds a new property and assigns it a unique value. If the property already exists it does not
	 * add the new property and returns the value of the existing one.
	 * @param {String} key key of the <String, Number> hashmap
	 * @return {Number} the property value corresponding to string key
	 */
	addProperty: function(propKey)
	{
		// Look for the highest value
		var highestValue = 0;
		for (var key in this) {
			if (typeof this[key] == 'number'){
				// Return the value if the property already exists
				if(key == propKey) {
					return this[key];
				}

				if(highestValue < this[key]){
					highestValue = this[key];
				}
			}
		}

		// Take the highest value, increase it by one and set it as value for the new property
		highestValue++;
		this[propKey] = highestValue;

		return highestValue;

	}
});

// Convenience method
Zarafa.core.Enum.create = function(object)
{
	var ExtendedObject = Ext.extend(Zarafa.core.Enum, object);
	return new ExtendedObject();
};
Ext.namespace('Zarafa.core');

/**
 * @class Zarafa.core.Events
 * Utility class for handling special events.
 * @singleton
 */
Zarafa.core.Events = {
	/**
	 * The list of fields and the corresponding {@link Ext.util.Observable}
	 * objects which are used to register the event handlers. Use
	 * {@link #registerListener} and {@link #unregisterListener} to add/remove
	 * {@Link Ext.util.Observable} instances. Use {@link #getListener}
	 * to obtain a {@link Ext.util.Observable} instance.
	 * @property
	 * @type Object
	 * @private
	 */
	listeners: {},

	/**
	 * Register a {@link Ext.util.Observable} instance for the given
	 * {@link Ext.form.Field}/{@link Ext.Element} pair to the {@link #listeners}
	 * object.
	 * @param {Ext.form.Field} field The register for which the observable is added
	 * @param {Ext.Element} el The element for which the the observable is added
	 * @param {Ext.util.Observable} observable The observable to register
	 * @private
	 */
	registerListener: function(field, el, observable)
	{
		var observables = Zarafa.core.Events.listeners[field.id];
		if (observables) {
			observables.push(observable);
		} else {
			Zarafa.core.Events.listeners[field.id] = [ observable ];
		}
	},

	/**
	 * Unregister a previously {@link #registerListener registered}
	 * {@link Ext.util.Observable} from the {@link #listeners}.
	 * @param {Ext.form.Field} field The register to which the observable belongs
	 * @param {Ext.Element} el The element to which the observable belongs
	 * @private
	 */
	unregisterListener: function(field, el)
	{
		var observables = Zarafa.core.Events.listeners[field.id];
		if (observables) {
			for (var i = 0, len = observables.length; i < len; i++) {
				var observable = observables[i];
				if (observable.el === el) {
					observables.splice(i, 1);
					if (observables.length === 0) {
						delete Zarafa.core.Events.listeners[field.id];
					}
				}
			}
		}
	},

	/**
	 * Obtain a previously {@link #registerListener registered}
	 * {@link Ext.util.Observable} from the {@link #listeners}.
	 * @param {Ext.form.Field} field The register to which the observable belongs
	 * @param {Ext.Element} el The element to which the observable belongs
	 * @private
	 */
	getListener: function(field, el)
	{
		var observables = Zarafa.core.Events.listeners[field.id];
		if (!observables) {
			return undefined;
		}

		for (var i = 0, len = observables.length; i < len; i++) {
			var observable = observables[i];
			if (observable.el === el) {
				return observable;
			}
		}
	},

	/**
	 * Add a special event handler to the given field to catch
	 * 'paste' events. There are multiple ways to paste text into
	 * a textfield.
	 * 1) contextmenu
	 * 2) Ctrl-V (shortcut depending on OS)
	 * 3) Drag & Drop text
	 *
	 * Support for catching these options depends severely on the browser,
	 * and thus this special event handler will serve as compatibility
	 * handler to handle the various cases correctly.
	 *
	 * @param {Ext.form.Field} field The field on which to listen for
	 * paste events.
	 * @param {Ext.Element} el The element on which the event should be registered
	 * @param {Function} fn The callback function to be called when text
	 * has been pasted. This function has no arguments (See {@link Ext.util.Observable#on}).
	 * @param {Object} scope The scope in which the function will be called (See {@link Ext.util.Observable#on}).
	 * @param {Object} obj (optional) Additional options (See {@link Ext.util.Observable#on}).
	 */
	addPasteEventHandler: function(field, el, fn, scope, obj)
	{
		// Check if this field has already been used to register
		// an event handler for pasting. If that is not the case
		// we need to construct a new Ext.util.Observable object
		// for the field and add the event handlers.
		var observable = Zarafa.core.Events.getListener(field, el);
		if (!Ext.isDefined(observable)) {
			var noOn = !Ext.isFunction(el.on);
			observable = new Ext.util.Observable();
			observable.field = field;
			observable.el = el;
			observable.hasFocus = field.hasFocus;
			observable.originalValue = field.getValue();
			observable.addEvents('paste');

			// Register the event handler for paste events. This will ensure
			// the input element will be resized when pasting.
			if (noOn) {
				el.addEventListener('paste', this.onPaste.createDelegate(observable));
			} else {
				field.mon(el, 'paste', this.onPaste, observable);
			}

			// A special kind of pasting is dragging & dropping text into
			// the boxfield. There really isn't a true event handler for that,
			// as it isn't pasting, but neither is it typing. So use the mouse
			// to detect such changes.
			//
			// For some reason the 'mouseup' event is not being triggered when
			// the user drops the text into the input field. So we must use
			// the event which is fired a bit later.
			if (noOn) {
				el.addEventListener('mouseover', this.onPasteMouseOver.createDelegate(observable));
			} else {
				field.mon(el, 'mouseover', this.onPasteMouseOver, observable);
			}
			field.on('blur', this.onPasteBlur, observable);

			Zarafa.core.Events.registerListener(field, el, observable);
		}

		observable.addListener('paste', fn, scope, obj);
	},

	/**
	 * Removes the event handler as registered by {@link #addPasteEventHandler}.
	 *
	 * @param {Ext.form.Field} field The field on which the event was registered
	 * @param {Ext.Element} el The element on which the event should be registered
	 * @param {Function} fn The function to unregister
	 * @param {Object} scope The scope for the function.
	 */
	removePasteEventHandler: function(field, el, fn, scope)
	{
		var observable = Zarafa.core.Events.getListener(field, el);
		if (Ext.isDefined(observable)) {
			observable.removeListener('paste', fn, scope);
			// If this was the last event handler, delete
			// the Observable instance.
			if (!observable.hasListener('paste')) {
				Zarafa.core.Events.unregisterListener(field, el);
			}
		}
	},

	/**
	 * As Opera doesn't have the 'paste' browser event we are mimicking the behavior
	 * here by having a global 'keyup' event handler. This event will simply check
	 * if 'Ctrl-V has been pressed and will fire the event handler.
	 *
	 * This function is called in the scope of an {@link Ext.util.Observable}.
	 *
	 * @param {Number} key The key which was pressed
	 * @param {Ext.EventObject} e The event object which fired the event
	 * @private
	 */
	onPasteKeyUp: function(key, e)
	{
		// There are references online which indicate that Opera doesn't detect
		// the Ctrl key properly and the keyCode is 0. But Opera 11 seems to
		// behave correctly...
		if (key.ctrlKey === true && key.keyCode === Ext.EventObject.V) {
			this.fireEvent('paste');
		}
	},

	/**
	 * Event handler for the 'paste' event on the {@link #el input element}.
	 * This will start the {@link #onPastePoll} function to wait for the contents
	 * to be pasted so it can fire the event handler.
	 *
	 * This function is called in the scope of an {@link Ext.util.Observable}.
	 *
	 * @private
	 */
	onPaste: function()
	{
		Zarafa.core.Events.onPastePoll.call(this, this.field, this.field.getValue(), 5);
	},

	/**
	 * The 'paste' event on an input element is fired before the text which must
	 * be pasted is added into the input field. There is no cross-browser solution
	 * to access the text before it is put into the input field, hence polling is used
	 * to wait for the input field to be updated with the new text, so it can fire
	 * the event handler.
	 *
	 * This function is called in the scope of an {@link Ext.util.Observable}.
	 *
	 * @param {Ext.form.Field} field The field to poll for the new value
	 * @param {String} oldValue The original value of the input element
	 * @param {Number} limit The number of polling tries left
	 * @private
	 */
	onPastePoll: function(field, oldValue, limit)
	{
		if (limit === 0) {
			return;
		} else if (field.getValue() === oldValue) {
			Zarafa.core.Events.onPastePoll.defer(1, this, [field, oldValue, --limit]);
			return;
		}
		this.fireEvent('paste');
		this.originalValue = this.field.getValue();
	},

	/**
	 * Event handler which is fired when the Mouse is being moved over the
	 * input field. This will check if the input field has been magically
	 * changed without the user pressing any button. If that happens, then
	 * the user has dragged text into the input element and we need to
	 * fire the event handler.
	 *
	 * This function is called in the scope of an {@link Ext.util.Observable}.
	 *
	 * @private
	 */
	onPasteMouseOver: function()
	{
		if (this.hasFocus === false) {
			if (this.originalValue !== this.field.getValue()) {
				this.field.focus();
				this.fireEvent('paste');
				this.originalValue = this.field.getValue();
			}

			this.hasFocus = true;
		}
	},

	/**
	 * Event handler which is fired when the registered field is being
	 * {@link Ext.form.Field#blur blurred}. This will store the value
	 * currently in the editor so during {@link #onPasteMouseOver} we can
	 * check if text has been dropped into the editor.
	 * @private
	 */
	onPasteBlur: function()
	{
		this.hasFocus = false;
		this.originalValue = this.field.getValue();
	}
};
Ext.namespace("Zarafa.core");

/**
 * @class Zarafa.core.HTMLParser
 *
 * Class for performing operations on HTML content.
 *
 * @singleton
 */
Zarafa.core.HTMLParser = (function() {
	// regular expression to strip all style tags with its content
	var stripStylesRe = /<style[^>]*>[\s\S]*?<\/style[^>]*>/gim;

	// regular expression to strip all style tags with its content
	var stripScriptsRe = /<script[^>]*>[\s\S]*?<\/script[^>]*>/gim;

	// regular expression to convert <br /> tags to newlines
	var br2nlRe = /<br\s*[^<>]*?\/*>/igm;

	// regular expression to convert \r\n, \n or \r tags to <br />
	var nl2brRe = /\r\n|\n|\r/gim;

	// regular expression to convert outlook style inline image urls to url which can request image using download_attachment.php
	var cidToUrlRe = /(src\s*=\s*[\"\']?)cid:([^ \"\']*)([\"\']?)/igm;

	// regular expression to convert url for inline images to outlook style url
	var urlToCidRe = /(src\s*=\s*[\"\']?)\S*attachCid=([^ &\"\']*)[^ \"\']*([\"\']?)/igm;

	return {
		/**
		 * Strips all style tags, and also remove its contents
		 * @param {Mixed} value The text from which to strip style tags
		 * @return {String} The stripped text
		 */
		stripStyles: function(v)
		{
			return !v ? v: String(v).replace(stripStylesRe, '');
		},

		/**
		 * Strips all script tags, and also remove its contents
		 * @param {Mixed} value The text from which to strip script tags
		 * @return {String} The stripped text
		 */
		stripScripts: function(v)
		{
			return !v ? v: String(v).replace(stripScriptsRe, '');
		},

		/**
		 * Converts HTML tag <br /> to newline characters
		 * @param {String} The string value to format.
		 * @return {String} The string with embedded \n tags in place of <br />.
		 */
		br2nl: function(v)
		{
			return Ext.isEmpty(v) ? '' : v.replace(br2nlRe, '\n');
		},

		/**
		 * Converts newline characters to HTML tag <br />
		 * @param {String} The string value to format.
		 * @return {String} The string with embedded <br /> tags in place of \n.
		 */
		nl2br: function(v)
		{
			return Ext.isEmpty(v) ? '' : v.replace(nl2brRe, '<br>');
		},

		/**
		 * Converts newline(\r\n, \r) characters to \n
		 * @param {String} The string value to format.
		 * @return {String} The string with embedded \n in place of \r\n or \r.
		 */
		rlnl2nl: function(v)
		{
			return Ext.isEmpty(v) ? '' : v.replace(/\r\n|\r/gim, '\n');
		},

		/**
		 * Format string with plain-text contents into a HTML formatted string.
		 *
		 * This will convert new-line characters into <br /> elements.
		 *
		 * @param {String} content The plain-text contents to be formatted
		 * @return {String} The HTML representation of the content
		 */
		convertPlainToHTML: function(content)
		{
			if(Ext.isEmpty(content)) {
				return content;
			}

			// convert all html entities to their html equivalents
			content = Zarafa.core.HTMLParser.entityEncode(content, 'ENT_NOQUOTES');

			// We should wrap the content in <pre> tag to maintain
			// text indentation/spacing when we convert it to HTML.
			content = '<div><pre wrap style=\"white-space: pre-wrap; word-wrap: break-word;\">' + content + '</pre></div>';

			// convert all breaklines
			content = Zarafa.core.HTMLParser.nl2br(content);

			return content;
		},

		/**
		 * Format string with HTML contents into a plain-text string.
		 *
		 * This will convert <br /> characters into \n elements.
		 *
		 * @param {String} content The HTML contents to be formatted
		 * @return {String} The plain-text representation of the content
		 */
		convertHTMLToPlain: function(content)
		{
			if(Ext.isEmpty(content)) {
				return content;
			}

			//----- remove tags but preserve the content ----

			// remove all select / options tags
			content = content.replace( /<[\/]?(?:select)[^>]*>/gim, '\n');
			content = content.replace(/<[\/]?(?:option)[^>]*>/gim, '\t');

			// replace all tags with their text equivalents
			content = content.replace(/<(?:hr)[^>]*>/gim, '\n-----------\n');
			content = content.replace(/<[\/]?(?:h[123456]|div|p|pre|title)[^>]*>/gim, '\n\n');
			content = content.replace(/<[\/]?(?:ul|ol|dl|dt|textarea|img)[^>]*>/gim, '\n');
			content = content.replace(/<[\/]?(?:dd|li)[^>]*>/gim, '\t');

			// tags related to table
			content = content.replace(/<[\/]?(?:table)[^>]*>/gim, '\n\n');
			content = content.replace(/<[\/]?(?:caption|tr)[^>]*>/gim, '\n');
			content = content.replace(/<[^\/]?(?:th|td)[^>]*>/gim, '<br />');

			// remove anchor tag by preserving the links, links will be added after content of anchor tag in between <> signs
			content = content.replace(/<a[^>]* href=[\'\"]?([^\s\'\">]*)[^>]*>([\s\S]*?)<\/a[^>]*>/gim, '$2 &lt;$1&gt;');

			//------ remove tags without preserving the contents -----

			// remove style tags
			content = Zarafa.core.HTMLParser.stripStyles(content);

			// remove script tags
			content = Zarafa.core.HTMLParser.stripScripts(content);

			// remove comments
			content = content.replace(/<!--[\s\S]*?-->/gim, '');

			// we have processed tags which are useful for plain text conversion so now remove all remaining tags
			content = Zarafa.core.HTMLParser.stripUnwantedTags(content, ['br']);

			// decode html entities
			content = Zarafa.core.HTMLParser.entityDecode(content);

			// remove unnecessary space
			content = content.replace(/^\s*$/gm, '');

			// add <br> in line which hasn't <br> at end of line
			content = content.replace(/(.*[^<>\n]$)/gm, '$1<br />');

			// remove extra line breaks
			content = content.replace(/\n/gm, '');

			// convert all breaklines
			content = Zarafa.core.HTMLParser.br2nl(content);

			// remove remaining html entities
			content = content.replace(/[&][#0-9a-z]*[;]/gim, '');

			// remove breaklines from the end of the lines
			content = content.replace(/\n(?!\n)$/gm, '');

			return content;
		},

		/**
		 * Function which strips unwanted tags, This function is whitelisting based so you need
		 * to pass tags that are allowed in the text, all other tags will be removed. This function will
		 * only remove tags and will not remove content in between tags.
		 * @param {String} content html content
		 * @param {Array} allowedTags tags that should not be removed from the content.
		 * @return {String} content after removing unwanted tags.
		 */
		stripUnwantedTags: function (content, allowedTags)
		{
			// Match all HTML tags
			var matches = content.match(/(<\/?[^>]+>)/gi);

			var html = '';
			var allowedTag = '';
			var i = -1;
			var allowed = false;

			if(!allowedTags) {
				allowedTags = [];
			}

			// Go through all HTML tags
			if(matches && matches.length > 0) {
				for(var index1 = 0, len1 = matches.length; index1 < len1; index1++) {
					// Save HTML tag
					html = matches[index1].toString();

					// flag to indicate that tag should be removed or not
					allowed = false;

					// Go through all allowed tags
					for(var index2 = 0, len2 = allowedTags.length; index2 < len2; index2++) {
						allowedTag = allowedTags[index2].toLowerCase();
						i = -1;

						if (i !== 0) {
							i = html.toLowerCase().indexOf('<'+allowedTag+'>');
						}

						if (i !== 0) {
							i = html.toLowerCase().indexOf('<'+allowedTag+' ');
						}

						if (i !== 0) {
							i = html.toLowerCase().indexOf('</'+allowedTag);
						}

						// Determine
						if (i === 0) {
							allowed = true;
							break;
						}
					}

					if (allowed !== true) {
						content = content.split(html).join('');		// Custom replace. No regexing
					}
				}
			}

			return content;
		},

		/**
		 * Function will check if data contains external contents in any html tag (img, audio, video),
		 * and will also check for external stylesheets.
		 * @param {String} data data that should be checked for external content.
		 * @return {Boolean} true if data contains external content else false.
		 */
		hasExternalContent: function(data)
		{
			if(Ext.isEmpty(data)) {
				return false;
			}

			var hasExternalImage = this.handleExternalImage(data, function (srcs) {
				var basePath = container.getBasePath();

				// It will return true if data contains some external image. if image src
				// is not start with basePath then we consider it to external image.
				return srcs.some(function (src) {
					return src.search("&attachCid=") === -1 && (src.startsWith("src=\"" + basePath) === false || src.startsWith("background=\"" + basePath) === false);
				});
			});

			if (hasExternalImage) {
				return true;
			}

			// check tags whose attributes contains style attribute with external url
			if(data.search(/(style)\s*=(\S*)(url)\(([\'\"]*?)\s*(https*:.*[^\'\"])([\'\"]*?)\)/igm) !== -1) {
				return true;
			}

			return false;
		},

		/**
		 * Function will replace external content links with blank strings, so external content would not be loaded.
		 * @param {String} data raw data.
		 * @return {String} filtered data.
		 */
		blockExternalContent: function(data)
		{
			if(Ext.isEmpty(data)) {
				return data;
			}

			data = this.handleExternalImage(data, function (srcs) {
				var basePath = container.getBasePath();
				for (var i = 0; i < srcs.length; i++) {
					// Replace src url to empty string if image is external image and not inline image.
					if (srcs[i].search("&attachCid=") === -1 && (srcs[i].startsWith("src=\"" + basePath) === false || srcs[i].startsWith("background=\"" + basePath) === false)) {
					data = data.replace(srcs[i], srcs[i].startsWith("b") ? "background=\"\"" : "src=\"\"");
					}
				}
				return data;
			});

			// @TODO more work needs for these regular expressions or else a dom based html parser
			data = data.replace(/(style)\s*=(\S*)(url)\(([\'\"]*?)\s*(https*:.*[^\'\"])([\'\"]*?)\)/igm, "$1=$2$3($4$6)");

			return data;
		},

		/**
		* Helper function which is used to figure out content has external image or not and performa
		* action based on action handler.
		*
		* @param {String} data data that should be checked for external content.
		* @param {Function} actionHandler actionHandler is callback function to perform an action.
		* @returns {Mixed} actionHandler return data.
		*/
		handleExternalImage: function (data, actionHandler) {
			var imgSrcs = [];
			// @TODO more work needs for these regular expressions or else a dom based html parser
			// check tags whose attributes are src or background

			if (data.search(/(src|background|srcset)\s*=\s*([\'\"])*?\s*(https*:[^ \'\"]*)([\'\"])*/igm) !== -1) {
				imgSrcs = data.match(/(src|background|srcset)\s*=\s*([\'\"])*?\s*(https*:[^ \'\"]*)([\'\"])*/igm);
			}
			return actionHandler(imgSrcs);
		},

		/**
		 * Function will return translation table for HTML entities which can be used to
		 * encode/decode HTML entities.
		 * @param {String} tableType type of table to get, options are HTML_SPECIALCHARS or HTML_ENTITIES.
		 * @param {String} quoteStyle options are ENT_COMPAT, ENT_NOQUOTES or ENT_QUOTES.
		 * @return {Object} table for translation of HTML entities
		 */
		getHTMLTranslationTable: function(tableType, quoteStyle)
		{
			if(!tableType) {
				tableType = 'HTML_SPECIALCHARS';
			}

			if(!quoteStyle) {
				quoteStyle = 'ENT_COMPAT';
			}

			var entities = {};
			entities['38'] = '&amp;';

			if (tableType === 'HTML_ENTITIES') {
				entities['94'] = '&circ;';
				entities['126'] = '&tilde;';

				entities['130'] = '&sbquo;';	// Single Low-9 Quotation Mark
				entities['131'] = '&fnof;';		// Latin Small Letter F With Hook
				entities['132'] = '&bdquo;';	// Double Low-9 Quotation Mark
				entities['133'] = '&hellip;';	// Horizontal Ellipsis
				entities['134'] = '&dagger;';	// Dagger
				entities['135'] = '&Dagger;';	// Double Dagger
				entities['136'] = '&circ;';		// Modifier Letter Circumflex Accent
				entities['137'] = '&permil;';	// Per Mille Sign
				entities['138'] = '&Scaron;';	// Latin Capital Letter S With Caron
				entities['139'] = '&lsaquo;';	// Single Left-Pointing Angle Quotation Mark
				entities['140'] = '&OElig;';	// Latin Capital Ligature OE
				entities['145'] = '&lsquo;';	// Left Single Quotation Mark
				entities['146'] = '&rsquo;';	// Right Single Quotation Mark
				entities['147'] = '&ldquo;';	// Left Double Quotation Mark
				entities['148'] = '&rdquo;';	// Right Double Quotation Mark
				entities['149'] = '&bull;';		// Bullet
				entities['150'] = '&ndash;';	// En Dash
				entities['151'] = '&mdash;';	// Em Dash
				entities['152'] = '&tilde;';	// Small Tilde
				entities['153'] = '&trade;';	// Trade Mark Sign
				entities['154'] = '&scaron;';	// Latin Small Letter S With Caron
				entities['155'] = '&rsaquo;';	// Single Right-Pointing Angle Quotation Mark
				entities['156'] = '&oelig;';	// Latin Small Ligature OE
				entities['159'] = '&Yuml;';		// Latin Capital Letter Y With Diaeresis

				entities['160'] = '&nbsp;';		// Non-breaking space
				entities['161'] = '&iexcl;';	// Inverted exclamation mark
				entities['162'] = '&cent;';		// Cent sign
				entities['163'] = '&pound;';	// Pound sign
				entities['164'] = '&curren;';	// Currency sign
				entities['165'] = '&yen;';		// Yen sign
				entities['166'] = '&brvbar;';	// Broken vertical bar
				entities['167'] = '&sect;';		// Section sign
				entities['168'] = '&uml;';		// Diaeresis
				entities['169'] = '&copy;';		// Copyright sign
				entities['170'] = '&ordf;';		// Feminine ordinal indicator
				entities['171'] = '&laquo;';	// Left-pointing double angle quotation mark
				entities['172'] = '&not;';		// Not sign
				entities['173'] = '&shy;';		// Soft hyphen
				entities['174'] = '&reg;';		// Registered sign
				entities['175'] = '&macr;';		// Macron
				entities['176'] = '&deg;';		// Degree sign
				entities['177'] = '&plusmn;';	// Plus-minus sign
				entities['178'] = '&sup2;';		// Superscript two
				entities['179'] = '&sup3;';		// Superscript three
				entities['180'] = '&acute;';	// Acute accent
				entities['181'] = '&micro;';	// Micro sign
				entities['182'] = '&para;';		// Pilcrow sign
				entities['183'] = '&middot;';	// Middle dot
				entities['184'] = '&cedil;';	// Cedilla
				entities['185'] = '&sup1;';		// Superscript one
				entities['186'] = '&ordm;';		// Masculine ordinal indicator
				entities['187'] = '&raquo;';	// Right-pointing double angle quotation mark
				entities['188'] = '&frac14;';	// Vulgar fraction one-quarter
				entities['189'] = '&frac12;';	// Vulgar fraction one-half
				entities['190'] = '&frac34;';	// Vulgar fraction three-quarters
				entities['191'] = '&iquest;';	// Inverted question mark
				entities['192'] = '&Agrave;';	// A with grave
				entities['193'] = '&Aacute;';	// A with acute
				entities['194'] = '&Acirc;';	// A with circumflex
				entities['195'] = '&Atilde;';	// A with tilde
				entities['196'] = '&Auml;';		// A with diaeresis
				entities['197'] = '&Aring;';	// A with ring above
				entities['198'] = '&AElig;';	// AE
				entities['199'] = '&Ccedil;';	// C with cedilla
				entities['200'] = '&Egrave;';	// E with grave
				entities['201'] = '&Eacute;';	// E with acute
				entities['202'] = '&Ecirc;';	// E with circumflex
				entities['203'] = '&Euml;';		// E with diaeresis
				entities['204'] = '&Igrave;';	// I with grave
				entities['205'] = '&Iacute;';	// I with acute
				entities['206'] = '&Icirc;';	// I with circumflex
				entities['207'] = '&Iuml;';		// I with diaeresis
				entities['208'] = '&ETH;';		// Eth
				entities['209'] = '&Ntilde;';	// N with tilde
				entities['210'] = '&Ograve;';	// O with grave
				entities['211'] = '&Oacute;';	// O with acute
				entities['212'] = '&Ocirc;';	// O with circumflex
				entities['213'] = '&Otilde;';	// O with tilde
				entities['214'] = '&Ouml;';		// O with diaeresis
				entities['215'] = '&times;';	// Multiplication sign
				entities['216'] = '&Oslash;';	// O with stroke
				entities['217'] = '&Ugrave;';	// U with grave
				entities['218'] = '&Uacute;';	// U with acute
				entities['219'] = '&Ucirc;';	// U with circumflex
				entities['220'] = '&Uuml;';		// U with diaeresis
				entities['221'] = '&Yacute;';	// Y with acute
				entities['222'] = '&THORN;';	// Thorn
				entities['223'] = '&szlig;';	// Sharp s. Also known as ess-zed
				entities['224'] = '&agrave;';	// a with grave
				entities['225'] = '&aacute;';	// a with acute
				entities['226'] = '&acirc;';	// a with circumflex
				entities['227'] = '&atilde;';	// a with tilde
				entities['228'] = '&auml;';		// a with diaeresis
				entities['229'] = '&aring;';	// a with ring above
				entities['230'] = '&aelig;';	// ae. Also known as ligature ae
				entities['231'] = '&ccedil;';	// c with cedilla
				entities['232'] = '&egrave;';	// e with grave
				entities['233'] = '&eacute;';	// e with acute
				entities['234'] = '&ecirc;';	// e with circumflex
				entities['235'] = '&euml;';		// e with diaeresis
				entities['236'] = '&igrave;';	// i with grave
				entities['237'] = '&iacute;';	// i with acute
				entities['238'] = '&icirc;';	// i with circumflex
				entities['239'] = '&iuml;';		// i with diaeresis
				entities['240'] = '&eth;';		// eth
				entities['241'] = '&ntilde;';	// n with tilde
				entities['242'] = '&ograve;';	// o with grave
				entities['243'] = '&oacute;';	// o with acute
				entities['244'] = '&ocirc;';	// o with circumflex
				entities['245'] = '&otilde;';	// o with tilde
				entities['246'] = '&ouml;';		// o with diaeresis
				entities['247'] = '&divide;';	// Division sign
				entities['248'] = '&oslash;';	// o with stroke. Also known as o with slash
				entities['249'] = '&ugrave;';	// u with grave
				entities['250'] = '&uacute;';	// u with acute
				entities['251'] = '&ucirc;';	// u with circumflex
				entities['252'] = '&uuml;';		// u with diaeresis
				entities['253'] = '&yacute;';	// y with acute
				entities['254'] = '&thorn;';	// thorn
				entities['255'] = '&yuml;';		// y with diaeresis
				entities['264'] = '&#264;';		// Latin capital letter C with circumflex
				entities['265'] = '&#265;';		// Latin small letter c with circumflex
				entities['338'] = '&OElig;';	// Latin capital ligature OE
				entities['339'] = '&oelig;';	// Latin small ligature oe
				entities['352'] = '&Scaron;';	// Latin capital letter S with caron
				entities['353'] = '&scaron;';	// Latin small letter s with caron
				entities['372'] = '&#372;';		// Latin capital letter W with circumflex
				entities['373'] = '&#373;';		// Latin small letter w with circumflex
				entities['374'] = '&#374;';		// Latin capital letter Y with circumflex
				entities['375'] = '&#375;';		// Latin small letter y with circumflex
				entities['376'] = '&Yuml;';		// Latin capital letter Y with diaeresis
				entities['402'] = '&fnof;';		// Latin small f with hook, function, florin
				entities['710'] = '&circ;';		// Modifier letter circumflex accent
				entities['732'] = '&tilde;';	// Small tilde
				entities['913'] = '&Alpha;';	// Alpha
				entities['914'] = '&Beta;';		// Beta
				entities['915'] = '&Gamma;';	// Gamma
				entities['916'] = '&Delta;';	// Delta
				entities['917'] = '&Epsilon;';	// Epsilon
				entities['918'] = '&Zeta;';		// Zeta
				entities['919'] = '&Eta;';		// Eta
				entities['920'] = '&Theta;';	// Theta
				entities['921'] = '&Iota;';		// Iota
				entities['922'] = '&Kappa;';	// Kappa
				entities['923'] = '&Lambda;';	// Lambda
				entities['924'] = '&Mu;';		// Mu
				entities['925'] = '&Nu;';		// Nu
				entities['926'] = '&Xi;';		// Xi
				entities['927'] = '&Omicron;';	// Omicron
				entities['928'] = '&Pi;';		// Pi
				entities['929'] = '&Rho;';		// Rho
				entities['931'] = '&Sigma;';	// Sigma
				entities['932'] = '&Tau;';		// Tau
				entities['933'] = '&Upsilon;';	// Upsilon
				entities['934'] = '&Phi;';		// Phi
				entities['935'] = '&Chi;';		// Chi
				entities['936'] = '&Psi;';		// Psi
				entities['937'] = '&Omega;';	// Omega
				entities['945'] = '&alpha;';	// alpha
				entities['946'] = '&beta;';		// beta
				entities['947'] = '&gamma;';	// gamma
				entities['948'] = '&delta;';	// delta
				entities['949'] = '&epsilon;';	// epsilon
				entities['950'] = '&zeta;';		// zeta
				entities['951'] = '&eta;';		// eta
				entities['952'] = '&theta;';	// theta
				entities['953'] = '&iota;';		// iota
				entities['954'] = '&kappa;';	// kappa
				entities['955'] = '&lambda;';	// lambda
				entities['956'] = '&mu;';		// mu
				entities['957'] = '&nu;';		// nu
				entities['958'] = '&xi;';		// xi
				entities['959'] = '&omicron;';	// omicron
				entities['960'] = '&pi;';		// pi
				entities['961'] = '&rho;';		// rho
				entities['962'] = '&sigmaf;';	// sigmaf
				entities['963'] = '&sigma;';	// sigma
				entities['964'] = '&tau;';		// tau
				entities['965'] = '&upsilon;';	// upsilon
				entities['966'] = '&phi;';		// phi
				entities['967'] = '&chi;';		// chi
				entities['968'] = '&psi;';		// psi
				entities['969'] = '&omega;';	// omega
				entities['977'] = '&thetasym;';	// Theta symbol
				entities['978'] = '&upsih;';	// Greek upsilon with hook symbol
				entities['982'] = '&piv;';		// Pi symbol
				entities['8194'] = '&ensp;';	// En space
				entities['8195'] = '&emsp;';	// Em space
				entities['8201'] = '&thinsp;';	// Thin space
				entities['8204'] = '&zwnj;';	// Zero width non-joiner
				entities['8205'] = '&zwj;';		// Zero width joiner
				entities['8206'] = '&lrm;';		// Left-to-right mark
				entities['8207'] = '&rlm;';		// Right-to-left mark
				entities['8211'] = '&ndash;';	// En dash
				entities['8212'] = '&mdash;';	// Em dash
				entities['8216'] = '&lsquo;';	// Left single quotation mark
				entities['8217'] = '&rsquo;';	// Right single quotation mark
				entities['8218'] = '&sbquo;';	// Single low-9 quotation mark
				entities['8220'] = '&ldquo;';	// Left double quotation mark
				entities['8221'] = '&rdquo;';	// Right double quotation mark
				entities['8222'] = '&bdquo;';	// Double low-9 quotation mark
				entities['8224'] = '&dagger;';	// Dagger
				entities['8225'] = '&Dagger;';	// Double dagger
				entities['8226'] = '&bull;';	// Bullet
				entities['8230'] = '&hellip;';	// Horizontal ellipsis
				entities['8240'] = '&permil;';	// Per mille sign
				entities['8242'] = '&prime;';	// Prime
				entities['8243'] = '&Prime;';	// Double Prime
				entities['8249'] = '&lsaquo;';	// Single left-pointing angle quotation
				entities['8250'] = '&rsaquo;';	// Single right-pointing angle quotation
				entities['8254'] = '&oline;';	// Overline
				entities['8260'] = '&frasl;';	// Fraction Slash
				entities['8364'] = '&euro;';	// Euro sign
				entities['8472'] = '&weierp;';	// Script capital
				entities['8465'] = '&image;';	// Blackletter capital I
				entities['8476'] = '&real;';	// Blackletter capital R
				entities['8482'] = '&trade;';	// Trade mark sign
				entities['8501'] = '&alefsym;';	// Alef symbol
				entities['8592'] = '&larr;';	// Leftward arrow
				entities['8593'] = '&uarr;';	// Upward arrow
				entities['8594'] = '&rarr;';	// Rightward arrow
				entities['8595'] = '&darr;';	// Downward arrow
				entities['8596'] = '&harr;';	// Left right arrow
				entities['8629'] = '&crarr;';	// Downward arrow with corner leftward. Also known as carriage return
				entities['8656'] = '&lArr;';	// Leftward double arrow. ISO 10646 does not say that lArr is the same as the 'is implied by' arrow but also does not have any other character for that function. So ? lArr can be used for 'is implied by' as ISOtech suggests
				entities['8657'] = '&uArr;';	// Upward double arrow
				entities['8658'] = '&rArr;';	// Rightward double arrow. ISO 10646 does not say this is the 'implies' character but does not have another character with this function so ? rArr can be used for 'implies' as ISOtech suggests
				entities['8659'] = '&dArr;';	// Downward double arrow
				entities['8660'] = '&hArr;';	// Left-right double arrow

				// Mathematical Operators
				entities['8704'] = '&forall;';	// For all
				entities['8706'] = '&part;';	// Partial differential
				entities['8707'] = '&exist;';	// There exists
				entities['8709'] = '&empty;';	// Empty set. Also known as null set and diameter
				entities['8711'] = '&nabla;';	// Nabla. Also known as backward difference
				entities['8712'] = '&isin;';	// Element of
				entities['8713'] = '&notin;';	// Not an element of
				entities['8715'] = '&ni;';		// Contains as member
				entities['8719'] = '&prod;';	// N-ary product. Also known as product sign. Prod is not the same character as U+03A0 'greek capital letter pi' though the same glyph might be used for both
				entities['8721'] = '&sum;';		// N-ary summation. Sum is not the same character as U+03A3 'greek capital letter sigma' though the same glyph might be used for both
				entities['8722'] = '&minus;';	// Minus sign
				entities['8727'] = '&lowast;';	// Asterisk operator
				entities['8729'] = '&#8729;';	// Bullet operator
				entities['8730'] = '&radic;';	// Square root. Also known as radical sign
				entities['8733'] = '&prop;';	// Proportional to
				entities['8734'] = '&infin;';	// Infinity
				entities['8736'] = '&ang;';		// Angle
				entities['8743'] = '&and;';		// Logical and. Also known as wedge
				entities['8744'] = '&or;';		// Logical or. Also known as vee
				entities['8745'] = '&cap;';		// Intersection. Also known as cap
				entities['8746'] = '&cup;';		// Union. Also known as cup
				entities['8747'] = '&int;';		// Integral
				entities['8756'] = '&there4;';	// Therefore
				entities['8764'] = '&sim;';		// tilde operator. Also known as varies with and similar to. The tilde operator is not the same character as the tilde, U+007E, although the same glyph might be used to represent both
				entities['8773'] = '&cong;';	// Approximately equal to
				entities['8776'] = '&asymp;';	// Almost equal to. Also known as asymptotic to
				entities['8800'] = '&ne;';		// Not equal to
				entities['8801'] = '&equiv;';	// Identical to
				entities['8804'] = '&le;';		// Less-than or equal to
				entities['8805'] = '&ge;';		// Greater-than or equal to
				entities['8834'] = '&sub;';		// Subset of
				entities['8835'] = '&sup;';		// Superset of. Note that nsup, 'not a superset of, U+2283' is not covered by the Symbol font encoding and is not included.
				entities['8836'] = '&nsub;';	// Not a subset of
				entities['8838'] = '&sube;';	// Subset of or equal to
				entities['8839'] = '&supe;';	// Superset of or equal to
				entities['8853'] = '&oplus;';	// Circled plus. Also known as direct sum
				entities['8855'] = '&otimes;';	// Circled times. Also known as vector product
				entities['8869'] = '&perp;';	// Up tack. Also known as orthogonal to and perpendicular
				entities['8901'] = '&sdot;';	// Dot operator. The dot operator is not the same character as U+00B7 middle dot

				// Miscellaneous Technical
				entities['8968'] = '&lceil;';	// Left ceiling. Also known as an APL upstile
				entities['8969'] = '&rceil;';	// Right ceiling
				entities['8970'] = '&lfloor;';	// left floor. Also known as APL downstile
				entities['8971'] = '&rfloor;';	// Right floor
				entities['9001'] = '&lang;';	// Left-pointing angle bracket. Also known as bra. Lang is not the same character as U+003C 'less than'or U+2039 'single left-pointing angle quotation mark'
				entities['9002'] = '&rang;';	// Right-pointing angle bracket. Also known as ket. Rang is not the same character as U+003E 'greater than' or U+203A 'single right-pointing angle quotation mark'

				// Geometric Shapes
				entities['9642'] = '&#9642;';	// Black small square
				entities['9643'] = '&#9643;';	// White small square
				entities['9674'] = '&loz;';		// Lozenge

				// Miscellaneous Symbols
				entities['9702'] = '&#9702;';	// White bullet
				entities['9824'] = '&spades;';	// Black (filled) spade suit
				entities['9827'] = '&clubs;';	// Black (filled) club suit. Also known as shamrock
				entities['9829'] = '&hearts;';	// Black (filled) heart suit. Also known as shamrock
				entities['9830'] = '&diams;';  // Black (filled) diamond suit
			}

			if (quoteStyle !== 'ENT_NOQUOTES') {
				entities['34'] = '&quot;';
			}

			if (quoteStyle === 'ENT_QUOTES') {
				entities['39'] = '&#39;';
			}

			entities['60'] = '&lt;';
			entities['62'] = '&gt;';

			// ascii decimals to real symbols
			var decimal;
			var hashMap = {};
			for (decimal in entities) {
				hashMap[String.fromCharCode(decimal)] = entities[decimal];
			}

			return hashMap;
		},

		/**
		 * Function will decode HTML entities in the string.
		 * @param {String} content string that should be decoded.
		 * @param {String} quoteStyle options are ENT_COMPAT, ENT_NOQUOTES or ENT_QUOTES.
		 * @return {String} decoded string.
		 */
		entityDecode: function(content, quoteStyle)
		{
			var hashMap = Zarafa.core.HTMLParser.getHTMLTranslationTable('HTML_ENTITIES', quoteStyle);
			var symbol = '';
			var entity = '';

			for (symbol in hashMap) {
				entity = hashMap[symbol];
				content = content.split(entity).join(symbol);
			}

			// Convert HTML entities like &#224;
			content = content.replace(/&#(\d+);/g, function(match, dec) {
				return String.fromCharCode(dec);
			});

			return content;
		},

		/**
		 * Function will encode HTML entities in the string.
		 * @param {String} content string that should be encoded.
		 * @param {String} quoteStyle options are ENT_COMPAT, ENT_NOQUOTES or ENT_QUOTES.
		 * @return {String} decoded string.
		 */
		entityEncode: function(content, quoteStyle)
		{
			var hashMap = Zarafa.core.HTMLParser.getHTMLTranslationTable('HTML_ENTITIES', quoteStyle);
			var symbol = '';
			var entity = '';

			for (symbol in hashMap) {
				entity = hashMap[symbol];
				content = content.split(symbol).join(entity);
			}
			content = content.split('\'').join('&#039;');

			return content;
		},

		/**
		 * Function to convert from Outlook inline attachment format (src="cid:...") to img tag
		 * that will display the image (by querying download_attachment.php)
		 * Builds up the query string for asking the image, using store and record entryid, and content id
		 * @param {String} body The message body to be modified
		 * @param {String} storeEntryId The store entryid
		 * @param {String} entryId The message entryid
		 * @param {Array} attachNum Attachment number of the attachment that should be downloaded.
		 * When accessing embedded messages this array can contain multiple elements indicating
		 * attachment numbers at each level, So value [0, 1] will indicate we want to download
		 * second attachment of first embedded message.
		 * @return {String} the modified HTML body
		 */
		inlineImgOutlookToZarafa: function(body, storeEntryId, entryId, attachNum)
		{
			var cidToUrl = function(match, srcStart, imgCid, srcEnd, offset, str) {
				if(imgCid) {
					var url = container.getBaseURL();
					url = Ext.urlAppend(url, 'load=download_attachment');
					url = Ext.urlAppend(url, 'attachCid=' + encodeURIComponent(imgCid));
					url = Ext.urlAppend(url, 'store=' + encodeURIComponent(storeEntryId));
					url = Ext.urlAppend(url, 'entryid=' + encodeURIComponent(entryId));
					url = Ext.urlAppend(url, 'contentDispositionType=inline');
					if(!Ext.isEmpty(attachNum)) {
						for (var i = 0; i< attachNum.length; i++)
						{
							url = Ext.urlAppend(url, 'attachNum[]=' + encodeURIComponent(attachNum[i]));
						}
					}

					// return our own url for getting inline attachments
					return srcStart + url + srcEnd;
				}

				// return match as it is but in a real world this is not going to happen
				return match;
			};

			// replace cid: with our own url to get inline attachment
			return body.replace(cidToUrlRe, cidToUrl);
		},

		/**
		 * Function to convert from Zarafa inline img format to Outlook format (src="cid:...")
		 * Grabs the cid from the img tag (src="...attachCid=...")
		 * @param {String} body the message body to be modified
		 * @return {String} the modified html body
		 */
		inlineImgZarafaToOutlook: function(body)
		{
			if(!Ext.isDefined(body)){
				return;
			}

			var urlToCid = function(match, srcStart, imgCid, srcEnd, offset, str) {
				if(imgCid) {
					// return img src with just cid: tag
					return srcStart + 'cid:' + imgCid + srcEnd;
				}

				// return match as it is but in a real world this is not going to happen
				return match;
			};

			// replace our own url with cid:
			return body.replace(urlToCidRe, urlToCid);
		}
	};
})();
Ext.namespace('Zarafa.core');

/**
 * @class Zarafa.core.KeyMap
 * @extends Ext.KeyMap
 *
 * This class extends {@link Ext.KeyMap KeyMap} to assign component with key events
 * so we can pass component as a parameter while calling callback function, to make
 * key event handling easier.
 */
Zarafa.core.KeyMap = Ext.extend(Ext.KeyMap, {
	/**
	 * @cfg {Ext.Component} component The component on which key event is registered.
	 */
	component: undefined,

	/**
	 * @constructor
	 * @param {Ext.Component} component The component which should be used to pass in handlers and additionally
	 * if element is not passed then get it from component.
	 * @param {Object} config The configuration object (see {@link #addBinding})
	 * @param {Ext.Element} element {optional} The element on which this keymap will be bound.
	 * @param {String} eventName (optional) The event to bind to (defaults to "keydown")
	 */
	constructor: function(component, config, element, eventName)
	{
		if(component) {
			// store reference of component in keymap that will be passed with call to handlers
			this.component = component;
		}

		// if no element is passed then get it from component
		if(!element) {
			element = component.getEl();
		}

		// if element is passed as dom node then find its corresponding Ext.Element object
		element = Ext.get(element);

		Zarafa.core.KeyMap.superclass.constructor.call(this, element, config, eventName);
	},

	/**
	 * Function overrides {@link Ext.KeyMap#addBinding addBinding} to <br>
	 * 1) add component in callback function parameter for which event is fired. <br>
	 * 2) to solve the problem of {@link Ext.KeyMap} that, When registering two different key bindings to
	 * same {@link Ext.KeyMap} object, we can't set 'stopEvent: false' in any of the bindings <br>
	 * for more info - http://www.sencha.com/forum/showthread.php?265199-Ext.KeyMap-problems-with-stopEvent-flag <br>
	 * 3) additionally this can accept some more config options which is used in {@link Zarafa.core.KeyMapMgr KeyMapMgr},
	 * <pre>
Property      Type    Description
----------     ---------  ----------------------------------------------------------------------
enableGlobally   Boolean   A flag to indicate the key binding should also be registered with the body of webapp so that key combination can globally be disabled
settingsCfg    Object   Object containing two keys:
                  1) description - description of the key combination to show in keyboard settings widget
                  2) category - name of the category in which this key combination will be added
</pre>
	 */
	addBinding: function(config)
	{
		if(Array.isArray(config)){
			Ext.each(config, function(c){
				this.addBinding(c);
			}, this);
			return;
		}
		var keyCode = config.key,
			fn = config.fn || config.handler,
			scope = config.scope,
			stopEvent = config.stopEvent;

		if(!Ext.isDefined(stopEvent)) {
			stopEvent = this.stopEvent;
		}

		if(typeof keyCode == "string"){
			var ks = [];
			var keyString = keyCode.toUpperCase();
			for(var j = 0, len = keyString.length; j < len; j++){
				ks.push(keyString.charCodeAt(j));
			}
			keyCode = ks;
		}
		var keyArray = Array.isArray(keyCode);

		var handler = function(e){
			if(this.checkModifiers(config, e)){
				var k = e.getKey();

				if(keyArray){
					for(var i = 0, len = keyCode.length; i < len; i++){
						if(keyCode[i] == k){
							if(stopEvent){
								e.stopEvent();
							}
							fn.call(scope || window, k, e, this.component);
							return;
						}
					}
				} else {
					if(k == keyCode){
						if(stopEvent){
							e.stopEvent();
						}
						fn.call(scope || window, k, e, this.component);
					}
				}
			}
		};
		this.bindings.push(handler);
	}
});Ext.namespace('Zarafa.core');

/**
 * @class Zarafa.core.KeyMapMgr
 * @extends Object
 *
 * The {@link Zarafa.core.KeyMap KeyMap} manager.
 * @singleton
 */
Zarafa.core.KeyMapMgr = Ext.extend(Object, {
	/**
	 * The list of registered {@link Zarafa.core.KeyMap keymaps}. It contains the list with the element the
	 * keymap is bound to as key.
	 * @property
	 * @type Object
	 * @private
	 */
	keyMaps: undefined,

	/**
	 * The list of key configurations (see {@link Zarafa.core.KeyMap#addBinding Zarafa.core.KeyMap.addBinding})
	 * registeredlisted to a mapId. It contains the list with the mapId as the key.
	 * @property
	 * @type Object
	 * @private
	 */
	keys: undefined,

	/**
	 * @constructor
	 */
	constructor: function()
	{
		this.keyMaps = new Ext.util.MixedCollection();
		this.keys = {};
	},

	/**
	 * Registers the key binding configuration to the specified mapId.
	 * @param {String} mapId The ID of the map keys are registered to.
	 * @param {Object|Array} keyConfig The config (see {@link Zarafa.core.KeyMap#addBinding addBinding}).
	 */
	register: function(mapId, keyConfig)
	{
		if(!this.keys[mapId]){
			this.keys[mapId] = [];
		}

		if(!Array.isArray(keyConfig)) {
			keyConfig = [keyConfig];
		}

		this.keys[mapId] = this.keys[mapId].concat(keyConfig);

		// register same bindings for global webapp as well
		this.preventBrowserDefaults(mapId, keyConfig);
	},

	/**
	 * Function will register all key bindings that are registered in {@link Zarafa.core.KeyMapMgr this}
	 * class to the 'globaldisable' mapid and assign stop event and empty handler so that we can disable
	 * default behaviour of browser when no component is focused and user tries to perform some
	 * key-combinations which isn't available globally.
	 * @param {String} mapId The ID of the map keys are registered to.
	 * @param {Object|Array} keyConfig The config (see {@link Zarafa.core.KeyMap#addBinding addBinding}).
	 */
	preventBrowserDefaults: function(mapId, keyConfig)
	{
		// Check that mapID isn't 'global' as they are already registered globally.
		// Check that mapID isn't 'globaldisable' to prevent recursive call.
		if(mapId !== 'global' && mapId !== 'globaldisable') {
			for(var i = 0, len = keyConfig.length; i < len; i++) {
				var config = keyConfig[i];

				if(config.enableGlobally !== true) {
					var disableKeyConfig = {
						handler: Ext.emptyFn,
						stopEvent: true
					};

					Ext.applyIf(disableKeyConfig, config);

					// Add all events in 'globaldisable' keymap.
					this.register('globaldisable', disableKeyConfig);
				} else if (config.ctrl === true) {
					var key = config.key;
					if(Ext.isString(key)) {
						key = key.toUpperCase().charCodeAt(0);
					}

					if (key === Ext.EventObject.A) {
						// special case for ctrl + a, as we want to allow action in text fields but not on body of webapp
						var disableKeyConfig = {
							handler: this.disableTextSelection,
							// don't blindly prevent default action instead leave that handling for the handler
							stopEvent: false
						};

						Ext.applyIf(disableKeyConfig, config);

						// Add event in 'globaldisable' keymap.
						this.register('globaldisable', disableKeyConfig);
					}
				}
			}
		}
	},

	/**
	 * Event handler for the keydown event of the {@link Ext.KeyMap KeyMap}
	 * when the user presses ctrl + a on a field which doesn't have text selection. So we will prevent
	 * default action of browser to select all text in the body.
	 * @param {Number} key Key code
	 * @param {Ext.EventObject} event The event
	 * @param {Ext.Component} component The component on which key event is fired.
	 */
	disableTextSelection: function(key, event, component)
	{
		var target = event.getTarget().nodeName.toLowerCase();

		if(target !== 'textarea' && target !== 'input') {
			// we don't want to select everything in the body
			event.stopEvent();
		}
	},

	/**
	 * Adds the keymap to the specified Element. It will take the keys registered in {@link #keys keys}
	 * under the specified mapId. If a keymap has already been registered under this element it will add new keys
	 * to the same {@link Zarafa.core.KeyMap KeyMap}.
	 *
	 * If basic shortcuts are enabled we filter the bindings obtained from the mapId to only
	 * enable the bindings which contain the basic key.
	 *
	 * @param {Ext.Component} component The component to which keymap should be bound and
	 * will listen keypress events on {@link Ext.Component#el}.
	 * @param {String} mapId The ID of the map keys are registered to.
	 * @param {Ext.Element} element {optional} if component is not present and we need to register key events
	 * on any element then we can pass it here (eg Ext.getBody()). Additionally if we want to register events on
	 * different element then {@link Ext.Component#el} then we should pass the element on which keymap
	 * will be registered (eg Zarafa.common.ui.HTMLEditor).
	 */
	activate: function(component, mapId, element)
	{
		// if element is not passed then get it from component
		if(!element) {
			element = component.getEl();
		}

		// if element is passed as dom node then find its corresponding Ext.Element object
		element = Ext.get(element);

		var elementId = element.id;
		if(Ext.isEmpty(elementId)) {
			// without element id its not possible to activate keymap
			return;
		}

		var bindings = this.getKeyBindings(mapId);
		var setting = container.getSettingsModel().get('zarafa/v1/main/keycontrols');

		// Filter basic shortcuts
		if (setting === Zarafa.settings.data.KeyboardSettings.BASIC_KEYBOARD_SHORTCUTS) {
			bindings = bindings.filter(function(binding) {
				return Ext.isDefined(binding.basic);
			});
		} else if (setting === Zarafa.settings.data.KeyboardSettings.NO_KEYBOARD_SHORTCUTS) {
			bindings = [];
		}

		if(Ext.isEmpty(bindings)) {
			// if no bindings are found then ignore
			return;
		}

		// check if we already have a registered keymap on this component
		// if we have then add binding with that keymap.
		var keymap = this.getActive(element);
		if(keymap) {
			keymap.addBinding(bindings);
		} else {
			// register event that will remove keymap from KeyMapMgr if component is destroyed
			if(component instanceof Ext.Component) {
				component.on('beforedestroy', this.onComponentDestroy, this);
			}

			// create a new keymap and register it on component
			this.keyMaps.add(elementId, new Zarafa.core.KeyMap(component, bindings, element));
		}
	},

	/**
	 * Function is used to get key bindings registered for a particular map id.
	 * It will recursively get bindings for every string separated using dot and then will
	 * combine and return the bindings.
	 * @param {String} mapId map id for which we need to get bindings
	 * @return {Array} array of key bindings that be added in keymap
	 */
	getKeyBindings: function(mapId)
	{
		// Get the bindings registered for the mapId. Also see if any other bindings have been
		// registered on mapIds that are hierarchically-speaking its parent. So
		// "contentpanel.record.message" will also get keys for "contentpanel.record" and
		// "contentpanel".
		var bindings = [];

		while(!Ext.isEmpty(mapId)) {
			// get binding for particular map id
			if(this.keys[mapId]) {
				bindings = bindings.concat(this.keys[mapId]);
			}

			// change map id to point to its parent
			mapId = mapId.substr(0, mapId.lastIndexOf('.'));
		}

		return bindings;
	},

	/**
	 * Handler function that will be called when any component registered with {@link Zarafa.core.KeyMapMgr KeyMapMgr}
	 * is going to be destroyed, so we can safely remove keymappings registered with that component.
	 * @param {Ext.Component} component component that is going to be destroyed
	 */
	onComponentDestroy: function(component)
 	{
		this.deactivate(component.getEl());
	},

	/**
	 * Disables the {@link Zarafa.core.KeyMap keymap} and removes it from list of {@link Zarafa.core.KeyMapMgr#keyMaps}.
	 * {@link Zarafa.core.KeyMap#disable} removes event listener which is used to check and fire handler events for bindings
	 * but it doesn't remove bindings registered with the {@link Zarafa.core.KeyMap keymap} that will be done only when
	 * element is destroyed from dom.
	 * @param {Ext.Element} element The element on which keymap is bound.
	 */
	deactivate: function(element)
	{
		element = Ext.get(element);
		var keymap = this.keyMaps.get(element.id);

		if(keymap) {
			keymap.disable();
			this.keyMaps.remove(keymap);
		}
	},

	/**
	 * Function returns {@link Zarafa.core.KeyMap keymap} bound to component or element.
	 * @param {Ext.Element} element The element which should be used to check for bound keymaps.
	 * @return {Zarafa.core.KeyMap} keymap object registered on component/element
	 */
	getActive: function(element)
	{
		// if element is passed as dom node then find its corresponding Ext.Element object
		element = Ext.get(element);

		return this.keyMaps.get(element.id);
	},

	/**
	 * Will enable the {@link Zarafa.core.KeyMap keymap} that is registered on the specified element.
	 * @param {Ext.Element} element The element on which keymap is bound.
	 */
	enableKeyMap: function(element)
	{
		// if element is passed as dom node then find its corresponding Ext.Element object
		element = Ext.get(element);

		var keymap = this.keyMaps.get(element.id);

		if(keymap) {
			if(this.isGloballyEnabled()) {
				// We need to check whether we are not in a state where the keymaps have been disabled globally.
				keymap.enable();
			} else {
				// Only set the originallyEnabled flag on the keymap when the keymaps have been disabled globally in the KeyMapMgr.
				keymap.originallyEnabled = true;
			}
		}
	},

	/**
	 * Will disable the {@link Zarafa.core.KeyMap keymap} that is registered on the specified element.
	 * @param {Ext.Element} element The element on which keymap is bound.
	 */
	disableKeyMap: function(element)
	{
		// if element is passed as dom node then find its corresponding Ext.Element object
		element = Ext.get(element);

		var keymap = this.keyMaps.get(element.id);

		if(keymap) {
			keymap.disable();

			// Only set the originallyEnabled flag on the keymap when the keymaps have been disabled globally in the KeyMapMgr.
			if(!this.isGloballyEnabled()) {
				keymap.originallyEnabled = false;
			}
		}
	},

	/**
	 * Can be used to enable the keymaps globally and works in conjucture with {@link #disable} that
	 * is able to disable all the keymaps globally.
	 * Will enable the all the registered keymaps that have an originallyEnabled flag set to true.
	 * After that each keymap, including the ones that have the originallyEnabled flag set to false,
	 * will have that flag cleared from the keymap object.
	 */
	enableAllKeymaps: function()
	{
		this.keyMaps.each(function(keymap) {
			if(keymap.originallyEnabled === true) {
				keymap.enable();
			}

			// Unset the originallyEnabled flag on the keymap object
			delete keymap.originallyEnabled;
		}, this);
	},

	/**
	 * Can be used to disable the keymaps globally and works in conjucture with {@link #enable} that
	 * is able to enable all the keymaps globally.
	 * Will enable the all the registered keymaps that have an originallyEnabled flag set to true.
	 * After that each keymap, including the ones that have the originallyEnabled flag set to false,
	 * will have that flag cleared from the keymap object.
	 */
	disableAllKeymaps: function()
	{
		this.keyMaps.each(function(keymap) {
			keymap.originallyEnabled = keymap.isEnabled();
			keymap.disable();
		}, this);
	},

	/**
	 * Checks if the {@link Zarafa.core.KeyMap keymap} is enabled on the specified element.
	 * @param {Ext.Element} element The element on which keymap is bound.
	 * @return {Boolean} True when enabled, false otherwise.
	 */
	isEnabled: function(element)
	{
		var keymap = this.getActive(element);

		return keymap && keymap.isEnabled();
	},

	/**
	 * Returns whether the keymaps are globally enabled or not.
	 * return {Boolean} True when enabled, false otherwise.
	 */
	isGloballyEnabled: function()
	{
		return container.getSettingsModel().get('zarafa/v1/main/keycontrols') !== Zarafa.settings.data.KeyboardSettings.NO_KEYBOARD_SHORTCUTS;
	}
});

Zarafa.core.KeyMapMgr = new Zarafa.core.KeyMapMgr();
Ext.namespace('Zarafa.core');

/**
 * @class Zarafa.core.MessageClass
 *
 * Special utility class for manipulation and comparisons
 * of Message Class Strings
 * @singleton
 */
Zarafa.core.MessageClass = {

	/**
	 * Check if the given messageclass property on this record matches the given 'expectedClass'.
	 * This comparison is done in a case-insensite way. This function can be used
	 * for comparing the value of 'message_class' safely.
	 *
	 * @param {String} className The className which should be checked,
	 * @param {String|Array} expectedClass The expected class against which the className
	 * must be compared
	 * @param {Boolean} baseOnly (optional) Only compare the start of the className,
	 * this can be used for partial comparison (e.g. record.isClass('message_class', 'IPM', true)
	 * will return true when the actual message_class is 'IPM.Note'). Defaults to false.
	 * @return {Boolean} True when the className matches.
	 */
	isClass: function(className, expectedClass, baseOnly)
	{
		if (Ext.isEmpty(className)) {
			return false;
		}

		// Convert the className to uppercase
		className = className.toUpperCase();

		// If the expectedClass is an array, we check if the className
		// is either one of the names in expectedClas
		if (Array.isArray(expectedClass)) {
			for (var i = 0, len = expectedClass.length; i < len; i++) {
				if (this.isClass(className, expectedClass[i], baseOnly)) {
					return true;
				}
			}
			return false;
		}

		// If the expectedClass length is larger then the className,
		// then it can never be equal.
		if (expectedClass.length > className.length) {
			return false;
		}

		// If baseOnly is true, we only want to match expectedClass against
		// the start of the className. Although we can use className.search()
		// it is faster to create a substring to ensure className and expectedClass
		// have the same length.
		if (baseOnly === true) {
			className = className.substring(0, expectedClass.length);
		}

		// Now the entire string must match
		return className == expectedClass.toUpperCase();
	},

	/**
	 * Test if the messageClass is compatible with the give containerClass.
	 * An IPM.Note is for example compatible with IPF.Note containerClass.
	 *
	 * @param {String} messageClass The Message Class to compare
	 * @param {String} containerClass The Container Class to compare
	 * @return {Boolean} True when the messageClass is compatible with the given containerClass
	 */
	isContainerClassCompatible: function(messageClass, containerClass)
	{
		messageClass = messageClass.toUpperCase();
		containerClass = containerClass.toUpperCase();

		if (Ext.isEmpty(containerClass)) {
			return true;
		}

		switch (messageClass) {
			case 'IPM.APPOINTMENT':
			case 'IPM.OLE.CLASS.{00061055-0000-0000-C000-000000000046}':
				return Zarafa.core.ContainerClass.isClass(containerClass, 'IPF.Appointment', true);
			case 'IPM.STICKYNOTE':
				return Zarafa.core.ContainerClass.isClass(containerClass, 'IPF.StickyNote', true);
			case 'IPM.CONTACT':
			case 'IPM.DISTLIST':
				return Zarafa.core.ContainerClass.isClass(containerClass, 'IPF.Contact', true);
			case 'IPM.TASK':
				return Zarafa.core.ContainerClass.isClass(containerClass, 'IPF.Task', true);
			case 'IPM.NOTE':
			case 'REPORT.IPM':
			case 'IPM.SCHEDULE':
			case "IPM.TASKREQUEST":
				// These are all normal messages, so those should be in IPF.Note containers,
				// however the IPF.Note.OutlookHomepage is special (as it is used for RSS feeds),
				// and should thus be excluded.
				return this.isClass(containerClass, 'IPF.Note', true) && !this.isClass(containerClass, 'IPF.Note.OutlookHomepage', true);
		}

		// Our fallthrough, the switch didn't catch any message class,
		// so perhaps we should cut of everything after the last dot (.)
		// and then we just retry by calling this function recursively.
		var index = messageClass.lastIndexOf('.');
		if (index > 0) {
			messageClass = messageClass.substr(0, index);
			return this.isContainerClassCompatible(messageClass, containerClass);
		}

		return false;
	},

	/**
	 * Function will return default foldertype from the hierarchy
	 * for the supplied message_class to the function.
	 *
	 * @param {String} messageClass The message_class of the mapi record.
	 * @return {String} The foldertype of the default folder for the supplied message_class
	 */
	getDefaultFolderTypeFromMessageClass: function(messageClass)
	{
		messageClass = messageClass.toUpperCase();

		switch (messageClass) {
			case 'IPM.APPOINTMENT':
			case 'IPM.OLE.CLASS.{00061055-0000-0000-C000-000000000046}':
				return 'calendar';
			case 'IPM.STICKYNOTE':
				return 'note';
			case 'IPM.CONTACT':
			case 'IPM.DISTLIST':
				return 'contact';
			case 'IPM.TASK':
				return 'task';
			case 'IPM.NOTE':
			case 'REPORT.IPM':
			case 'IPM.SCHEDULE':
			case "IPM.TASKREQUEST":
				return 'inbox';
		}

		// Our fallthrough, the switch didn't catch any message class,
		// so perhaps we should cut of everything after the last dot (.)
		// and then we just retry by calling this function recursively.
		var index = messageClass.lastIndexOf('.');
		if (index > 0) {
			messageClass = messageClass.substr(0, index);
			return this.getDefaultFolderTypeFromMessageClass(messageClass);
		}

		return '';
	}
};
Ext.namespace('Zarafa.core');

/**
 * @class Zarafa.core.ModuleNames
 * List of module names. Each module represents a server side component that can process actions. A list module
 * for instance will allow 'list' actions.
 * @singleton
 */
Zarafa.core.ModuleNames =
{
	/**
	 * Module information for the Address book.
	 * @property
	 * @type Mixed
	 */
	'ADDRESSBOOK': {
		list: 'addressbooklistmodule',
		item: 'addressbookitemmodule'
	},

	/**
	 * Module information for IPM.Appointment
	 * @property
	 * @type Mixed
	 */
	'IPM.APPOINTMENT': {
		list: 'appointmentlistmodule',
		item: 'appointmentitemmodule'
	},

	/**
	 * Module information for Appointment recurrence exceptions
	 * @property
	 * @type Mixed
	 */
	'IPM.OLE.CLASS.{00061055-0000-0000-C000-000000000046}': {
		list: 'appointmentlistmodule',
		item: 'appointmentitemmodule'
	},

	/**
	 * Module information for IPM.Task
	 * @property
	 * @type Mixed
	 */
	'IPM.TASK': {
		list: 'tasklistmodule',
		item: 'taskitemmodule'
	},

	/**
	 * Module information for IPM.TaskRequest
	 * @property
	 * @type Mixed
	 */
	'IPM.TASKREQUEST': {
		list: 'tasklistmodule',
		item: 'taskitemmodule'
	},

	/**
	 * contact list module.
	 * @property
	 * @type Mixed
	 */
	'IPM.CONTACT': {
		list: 'contactlistmodule',
		item: 'contactitemmodule'
	},

	/**
	 * Distribution list module.
	 * Here we use same module as contact module.
	 * @property
	 * @type Mixed
	 */
	'IPM.DISTLIST': {
		list: 'contactlistmodule',
		item: 'contactitemmodule'
	},

	/**
	 * Module information for hierarchy.
	 * The hierarchy is the set of stores a user can see, and the folders within those
	 * stores.
	 * @property
	 * @type Mixed
	 */
	'HIERARCHY': {
		list: 'hierarchymodule',
		item: 'hierarchymodule'
	},

	/**
	 * Module information for settings.
	 * stores.
	 * @property
	 * @type Mixed
	 */
	'SETTINGS': {
		list: 'settingsmodule'
	},

	/**
	 * Module information for IPM.Note
	 * @property
	 * @type Mixed
	 */
	'IPM.NOTE': {
		list: 'maillistmodule',
		item: 'createmailitemmodule'
	},

	/**
	 * Module information for IPM.StickyNote
	 * @property
	 * @type Mixed
	 */
	'IPM.STICKYNOTE': {
		list: 'stickynotelistmodule',
		item: 'stickynoteitemmodule'
	},

	/**
	 * Module information for freebusy
	 * @property
	 * @type Mixed
	 */
	'FREEBUSY': {
		list: 'freebusymodule'
	},

	/**
	 * Module information for busytime
	 * @property
	 * @type Mixed
	 */
	 'BUSYTIME': {
		 list: 'busytimelistmodule'
	 },

	/**
	 * Module information for freebusy
	 * @property
	 * @type Mixed
	 */
	'SUGGESTEMAILADDRESS': {
		list: 'suggestemailaddressmodule',
		item: 'suggestemailaddressmodule'
	},

	/**
	 * Module information for reminder
	 * @property
	 * @type Mixed
	 */
	'REMINDER': {
		list: 'reminderlistmodule',
		item: 'reminderitemmodule'
	},

	/**
	 * Module information for delegates
	 * @property
	 * @type Mixed
	 */
	'DELEGATES': {
		list: 'delegatesmodule',
		item: 'delegatesmodule'
	},

	/**
	 * Module information for Rules
	 * @property
	 * @type Mixed
	 */
	'RULES': {
		list: 'rulesmodule',
		item: 'rulesmodule'
	},

	/**
	 * Module information for Out of office settings
	 * @property
	 * @type Mixed
	 */
	'OUTOFOFFICESETTINGS': {
		list: 'outofofficesettingsmodule',
		item: 'outofofficesettingsmodule'
	},

	/**
	 * Module information for Restore Soft Deleted Items
	 * @property
	 * @type Mixed
	 */
	'RESTOREITEMS': {
		list: 'restoreitemslistmodule',
		item: 'restoreitemslistmodule'
	},

	/**
	 * Module information for advanced search.
	 * @property
	 * @type Mixed
	 */
	'IPM.SEARCH': {
		list: 'advancedsearchlistmodule',
		item: 'createmailitemmodule'
	},

	/**
	 * Obtain the moduleName for an appropriate key.
	 * The key could either be a component name, or a Message Class.
	 *
	 * @param {String} key The key for which the module must be found
	 * @param {Boolean} baseOnly (optional) When the the key is not exactly
	 * matched againsts a moduleName, remove everything after the last '.',
	 * defaults to false.
	 * @return {Object} An object containing a 'list' and 'item' key,
	 * for the moduleNames for the 'list' and 'item' module respectively.
	 */
	getModule: function(key, baseOnly)
	{
		key = key.toUpperCase();

		var module = this[key];
		if (!module && baseOnly === true) {
			var index = key.lastIndexOf('.');
			if (index > 0) {
				key = key.substr(0, index);
				module = this.getModule(key, baseOnly);
			}
		}

		return module;
	},

	/**
	 * Obtain the moduleName for the Item module for the appropriate key.
	 * This uses {@link #getModule} to obtain the module, and returns the
	 * itemModulename from the result.
	 *
	 * @param {String} key The key for which the module must be found
	 * @param {Boolean} baseOnly (optional) When the the key is not exactly
	 * matched againsts a moduleName, remove everything after the last '.',
	 * defaults to false.
	 * @return {String} The item moduleName for the requested module
	 */
	getItemName: function(key, baseOnly)
	{
		var module = this.getModule(key, baseOnly);
		if (module) {
			return module.item;
		}
	},

	/**
	 * Obtain the moduleName for the List module for the appropriate key.
	 * This uses {@link #getModule} to obtain the module, and returns the
	 * listModulename from the result.
	 *
	 * @param {String} key The key for which the module must be found
	 * @param {Boolean} baseOnly (optional) When the the key is not exactly
	 * matched againsts a moduleName, remove everything after the last '.',
	 * defaults to false.
	 * @return {String} The list moduleName for the requested module
	 */
	getListName: function(key, baseOnly)
	{
		var module = this.getModule(key, baseOnly);
		if (module) {
			return module.list;
		}
	}
};
Ext.namespace('Zarafa.core');

/**
 * @class Zarafa.core.PingService
 * @extends Ext.util.Observable
 *
 * Ping service which will periodically ping the server to determine
 * if the HTTP and Gromox server are both available again, so the user can
 * continue working with grommunio Web.
 */
Zarafa.core.PingService = Ext.extend(Ext.util.Observable, {

	/**
	 * @cfg {String} url
	 * The url used to send the requests to. defaults to grommunio.php.
	 */
	url: 'grommunio.php',

	/**
	 * @cfg {String} cmd
	 * The GET attribute to send to the server. defaults to 'ping'
	 */
	cmd: 'ping',

	/**
	 * @cfg {Object} headers
	 * The default headers to be applied to the request. defaults to
	 *   'Content-Type' => 'application/json; charset=utf-8;'
	 */
	headers: undefined,

	/**
	 * @cfg {Number} timeout The initial timeout value for the call
	 * to {@link #sendPing}. This will be incremented by {@link #getNextTimeout}.
	 */
	timeout: 1000,

	/**
	 * @cfg {Number} maxTimeout The maximum timeout value which can be used
	 * and returned by {@link #getNextTimeout}.
	 */
	maxTimeout: 300000,

	/**
	 * The DelayedTask which will be used to defer the {@link #sendPing}
	 * function call to periodically poll the server for availability.
	 * @property
	 * @type Ext.util.DelayedTask
	 * @private
	 */
	pingTask: undefined,

	/**
	 * The current timeout value for the {@link #pingTask}.
	 * This is obtained (and incremented) through {@link #getNextTimeout}.
	 * @property
	 * @type Number
	 * @private
	 */
	currentTimeout: undefined,

	/**
	 * True if the Ping has been send to the server, and the PingService
	 * is currently awaiting the response from the server.
	 * @property
	 * @type Boolean
	 * @private
	 */
	pingPending: false,

	/**
	 * @constructor
	 * @param {Object} config Configuration object
	 */
	constructor: function(config)
	{
		config = config || {};

		Ext.apply(config, {
			// Apply here instead of class prototype, to prevent
			// accidental sharing of object between all instances.
			headers: {
				'Content-Type': 'application/json; charset=utf-8;'
			}
		});

		Ext.apply(this, config);

		this.addEvents(
			/**
			 * @event start
			 * Fired during {@link #start} to indicate that the Ping Service will start
			 * polling the server for link connectivity.
			 * @param {Zarafa.core.PingService} service This object
			 * @param {Number} timeout The timeout for the next ping which will be send
			 * to the server
			 */
			'start',
			/**
			 * @event stop
			 * Fired during {@link #stop} to indicate that the Ping Service will stop
			 * polling the server for link connectivity. This doesn't imply that the
			 * Service has been stopped before or after a successful response was returned.
			 * @param {Zarafa.core.PingService} service This object
			 */
			'stop',
			/**
			 * @event send
			 * Fired during {@link #sendPing} to indicate that a new Ping request will
			 * be made to the server.
			 * @param {Zarafa.core.PingService} service This object
			 * @param {XMLHttpRequest} xhrObj The XMLHttpRequest which is send to the server
			 */
			'send',
			/**
			 * @event retry
			 * Fired when a ping request was completed, but the connectivity has failed,
			 * a new attempt will be made after a specific timeout.
			 * @param {Zarafa.core.PingService} service This object
			 * @param {Object} response The response, if any, as send by the server
			 * @param {Number} timeout The timeout after which the next ping will be send
			 */
			'retry',
			/**
			 * @event restored
			 * Fired when a ping request was completed, and the connectivity has been restored.
			 * This means connectivity is restored, but the user might no longer have an active
			 * session. No new attempts will be made after this.
			 * @param {Zarafa.core.PingService} service This object
			 * @param {Object} response The response as send by the server
			 */
			'restored'
		);

		Zarafa.core.PingService.superclass.constructor.call(this, config);

		// Instantiate the delayed task for the sendPing function
		this.pingTask = new Ext.util.DelayedTask(this.sendPing, this);
	},

	/**
	 * Start the Ping Service and schedule the first run of {@link #pingTask}.
	 */
	start: function()
	{
		// reset the current timeout
		delete this.currentTimeout;

		// Obtain a new timeout and start polling
		var timeout = this.getNextTimeout();
		this.fireEvent('start', this, timeout);
		this.pingTask.delay(timeout);
	},

	/**
	 * Stop the Ping Service and cancel the {@link #pingTask}.
	 */
	stop: function()
	{
		this.pingTask.cancel();
		this.fireEvent('stop', this);
	},

	/**
	 * Interrupt the current timeout for {@link #pingTask} and manually
	 * invoke {@link #sendPing} causing a new request to be send out right now.
	 */
	retry: function()
	{
		this.pingTask.cancel();
		this.sendPing();
	},

	/**
	 * Obtain the next timeout value for the {@link #pingTask}. If
	 * {@link #currentTimeout} is initialized this will double the value
	 * (restricted by {@link #maxTimeout}, or otherwise it will take {@link #timeout}.
	 * @return {Number} The timeout for the {@link #pingTask}
	 * @private
	 */
	getNextTimeout: function()
	{
		this.currentTimeout = this.currentTimeout ? (2 * this.currentTimeout) : this.timeout;
		this.currentTimeout = Math.min(this.maxTimeout, this.currentTimeout);
		return this.currentTimeout;
	},

	/**
	 * Send a Ping request to the configured {@link #url} with the given {@link #cmd GET action}.
	 * {@link #onStateChange} will handle the response as received from the server.
	 * @private
	 */
	sendPing: function()
	{
		// A Ping request was already send,
		// we will not send another one simultaneously.
		if (this.pingPending) {
			return;
		}

		var xmlHttpRequest = new XMLHttpRequest();

		// Open the HTTP request object
		var url = this.url;
		url = Ext.urlAppend(url, this.cmd);
		xmlHttpRequest.open('GET', url, true);

		// Apply the headers
		for (var key in this.headers) {
			xmlHttpRequest.setRequestHeader(key, this.headers[key]);
		}

		// Register statechange callback function
		xmlHttpRequest.onreadystatechange = this.onStateChange.createDelegate(this, [ xmlHttpRequest ]);

		// Mark that the Ping request is currently pending.
		this.pingPending = true;

		// Send the request
		xmlHttpRequest.send();

		this.fireEvent('send', this, xmlHttpRequest);
	},

	/**
	 * Called by {@link XMLHttpRequest} when a response from the server has been received.
	 * This will determine if the link connection to the server has been restored or not.
	 * @param {XMLHttpRequest} xmlHttpRequest The request object
	 * @private
	 */
	onStateChange: function(xmlHttpRequest)
	{
		var response;

		// The readyState can be 4 values:
		// 0 - Object is created, but not initialized
		// 1 - Request has been opened, but send() has not been called yet
		// 2 - send() has been called, no data available yet
		// 3 - Some data has been received, responseText nor responseBody are available
		// 4 - All data has been received
		//
		// readyState 0 - 3 can be completely ignored by us, as they are only updates
		// about the current progress. Only on readyState 4, should we continue and
		// start checking for the response status.
		if (xmlHttpRequest.readyState != 4) {
			return;
		}

		// Regardless of the state, the Ping request
		// is no longer pending.
		this.pingPending = false;

		// HTTP request must have succeeded
		if (xmlHttpRequest.status !== 200) {
			this.failure();
			return;
		}

		// Depending on the response type, convert it into a data Object.
		if (xmlHttpRequest.responseText) {
			// JSON response
			response = Ext.decode(xmlHttpRequest.responseText);
		} else {
			// XML response is not supported
			this.failure();
			return;
		}

		// Determine if the response indicates that a link connection
		// exists, and call the proper handler.
		if (response.success) {
			this.restored(response);
		} else {
			this.failure(response);
		}
	},

	/**
	 * Called when the link connection has been restored. This will fire
	 * the {@link #restored} event.
	 * @param {Object} response The response as received by the server
	 * @private
	 */
	restored: function(response)
	{
		this.fireEvent('restored', this, response);
	},

	/**
	 * Called when the link connection has not been restored and will be
	 * retried later. This will fire the {@link #retry} event.
	 * @param {Object} response The response, if any, as received by the server
	 * @private
	 */
	failure: function(response)
	{
		var timeout = this.getNextTimeout();
		this.fireEvent('retry', this, response, timeout);
		this.pingTask.delay(timeout);
	}
});
Ext.namespace('Zarafa.core');

/**
 * @class Zarafa.core.PluginMetaData
 * @extends Object
 *
 * The Meta Data object containing the registration details
 * of a {@link Zarafa.core.Plugin}. An instance of this object
 * must be passed to {@link Zarafa.core.Container#registerPlugin}.
 */
Zarafa.core.PluginMetaData = Ext.extend(Object, {
	/**
	 * @cfg {String} name (required) The unique name for this plugin.
	 * For a user-friendly name for UI components, see {@link #displayName}
	 */
	name: '',

	/**
	 * @cfg {String} displayName The display name for this plugin. This
	 * will be used in places where the plugin is referenced in UI components.
	 * If not provided, {@link #name} will be used.
	 */
	displayName: '',

	/**
	 * @cfg {String} settingsName Alternative name for the plugin as used
	 * in the {@link Zarafa.settings.SettingsModel settings} in which the settings
	 * for this {@link Zarafa.core.Plugin plugin} are being saved. If not provided,
	 * then {@link #name} will be used.
	 */
	settingsName: '',

	/**
	 * @cfg {String} iconCls The icon to be used in places where the plugin is referenced
	 * in UI components.
	 */
	iconCls: '',

	/**
	 * @cfg {String} about The about text. If provided, {@link Zarafa.core.Plugin#registerAboutText}
	 * will be automatically called during {@link Zarafa.core.Plugin#initPlugin initialization}.
	 */
	about: undefined,

	/**
	 * @cfg {Boolean} allowUserDisable True if the user is allowed to enable/disable
	 * the plugin through the settings. To obtain the enabled status, the function
	 * {@link #isEnabled} should always be referenced.
	 */
	allowUserDisable: true,

	/**
	 * @cfg {Boolean} allowUserVisible True if the user is allowed to see the plugin
	 * in the settings. To obtain the visibility status, the function
	 * {@link #isPrivate} should always be referenced.
	 */
	allowUserVisible: true,

	/**
	 * @cfg {Constructor} pluginConstructor (required) The constructor of the
	 * {@link Zarafa.core.Plugin} which is described by this PluginMetaData instance.
	 */
	pluginConstructor: undefined,

	/**
	 * The instance of the {@link Zarafa.core.Plugin} (instantiated using the
	 * {@link #pluginConstructor}). This is obtained using the {@link #getInstance}
	 * function.
	 * @property
	 * @type Zarafa.core.Plugin
	 * @private
	 */
	instance: undefined,

	/**
	 * @constructor
	 * @param {Object} config Configuration object
	 */
	constructor: function(config)
	{
		config = config || {};

		Ext.apply(this, config);

		Zarafa.core.PluginMetaData.superclass.constructor.call(this, config);

		// Initialize displayName if not initialized from config
		if (Ext.isEmpty(this.displayName)) {
			this.displayName = this.name;
		}

		// Initialize settingsName if not initialized from config
		if (Ext.isEmpty(this.settingsName)) {
			this.settingsName = this.name;
		}
	},

	/**
	 * Determine if the plugin is enabled.
	 * @return {Boolean} True if the plugin is enabled
	 */
	isEnabled: function()
	{
		if ( !this.allowUserDisable ){
			return true;
		} else {
			return container.getSettingsModel().get(this.getSettingsBase() + '/enable') === true;
		}
	},

	/**
	 * Determine if the plugin should be considered private.
	 * @return {Boolean} True if the plugin is private
	 */
	isPrivate: function()
	{
		return this.allowUserVisible === false;
	},

	/**
	 * Obtain the unique name for this plugin
	 * @return {String} The unique name for this plugin
	 */
	getName: function()
	{
		return this.name;
	},

	/**
	 * Obtain the display name for this plugin
	 * @return {String} The display name for this plugin
	 */
	getDisplayName: function()
	{
		return this.displayName;
	},

	/**
	 * Obtain the CSS classname for this plugin
	 * @return {String} The CSS classname for this plugin
	 */
	getIconCls: function()
	{
		return this.iconCls;
	},

	/**
	 * Obtain the About text containing the copyright and other disclaimers.
	 * @return {String} The about text for this plugin
	 */
	getAbout: function()
	{
		return this.about;
	},

	/**
	 * Obtain the base path for the {@link Zarafa.settings.SettingsModel settings} in which the settings
	 * for this plugin can be found. This uses the {@link #settingsName} within the special 'plugins' section
	 * of the settings.
	 * @return {String} The settings path
	 */
	getSettingsBase: function()
	{
		return 'zarafa/v1/plugins/' + this.settingsName;
	},

	/**
	 * Obtain the instance of the {@link Zarafa.core.Plugin} which is instantiated
	 * using the {@link #pluginConstructor}. This uses single-instancing using the {@link #instance}.
	 * Before calling this, the {@link #isEnabled} function should have been used to determine if the
	 * {@link Zarafa.core.Plugin} is allowed to be instantiated.
	 * property.
	 * @return {Zarafa.core.Plugin} The Plugin instance
	 */
	getInstance: function()
	{
		if (!this.instance) {
			this.instance = new this.pluginConstructor({ info: this });
		}
		return this.instance;
	}
});
Ext.namespace('Zarafa.core');

/**
 * @class Zarafa.core.PresenceManager
 * @extends Ext.util.Observable
 *
 * Has functions to handle presence of users
 *
 * @singleton
 */
Zarafa.core.PresenceManager = Ext.extend(Ext.util.Observable, {
	/**
	 * The stores that have been registered with the {@link Zarafa.core.PresenceManager PresenceManager}
	 * @property
	 * @private
	 */
	registeredStores: [],

	/**
	 * An array with userIdObjects for which a presence status
	 * will be requested from all {#presencePlugins}
	 * @property
	 * @private
	 */
	requestQueue: [],

	/**
	 * The polling interval in milliseconds. This is the interval with which
	 * the presence status of the users in the registered stores will be
	 * updated
	 * @property
	 * @private
	 */
	pollingInterval: 20000,

	/**
	 * The constructor
	 */
	constructor: function()
	{
		var me = this;

		Zarafa.onReady(function(){
			setInterval(function(){
				me.pollForUpdates();
			}, me.pollingInterval);
		});
	},

	/**
	 * Returns an array with all registered
	 * {@link Zarafa.core.PresencePlugin PresencePlugins}
	 * @return {Zarafa.core.PresencePlugin[]}
	 */
	getPresencePlugins: function()
	{
		return container.getPlugins().filter(function(plugin){
			return plugin instanceof Zarafa.core.PresencePlugin;
		});
	},

	/**
	 * Returns the presence status for the users for whom the status has been requested
	 * @param {Zarafa.core.data.UserIdObject[]} users An array of userIdObjects
	 * for whom a presence status is requested.
	 * @return {Object[]} An array of objects that contain a
	 * {@link Zarafa.core.data.PresenceStatus presence status} for
	 * each presence plugin
	 */
	getPresenceStatusForUsers: function(users)
	{
		users = [].concat(users);

		if ( users.length === 0 ){
			return [];
		}

		var presenceStatuses = users.map(function() {
			return {};
		});

		var presencePlugins = this.getPresencePlugins();
		if ( presencePlugins.length === 0 ){
			// No presence plugins available, so let's return an array with empty objects
			return presenceStatuses;
		}

		// Request the statuses from the presence plugins.
		Ext.each(presencePlugins, function(presencePlugin){
			var pluginName = presencePlugin.getName();
			var pluginPresenceStatuses = presencePlugin.getPresenceStatuses(users);

			// Fill the statuses in the array
			// XXX: required?
			Ext.each(users, function(user, userIndex){
				presenceStatuses[userIndex][pluginName] = pluginPresenceStatuses[userIndex];
			});
		});

		return presenceStatuses;
	},

	/**
	 * Returns the presence status for a user. If the user is not available in the
	 * cache, a status for the user will be requested from the presence plugins.
	 * @param {Zarafa.core.data.UserIdObject} user The user for whom a presence
	 * status is requested.
	 * @return {Zarafa.core.data.PresenceStatus} The squashed presence status
	 */
	getPresenceStatusForUser: function(user)
	{
		if ( Ext.isEmpty(user) ){
			return Zarafa.core.data.PresenceStatus.UNKNOWN;
		}

		var statusByPlugin = this.getPresenceStatusForUsers([user])[0];
		// Squash the status
		var presenceStatus = Zarafa.core.data.PresenceStatus.UNKNOWN;
		Ext.iterate(statusByPlugin, function(pluginName){
			if ( statusByPlugin[pluginName] > presenceStatus ){
				presenceStatus = statusByPlugin[pluginName];
			}
		});
		return presenceStatus;
	},

	/**
	 * Add a {@link Zarafa.core.data.UserIdObject user} to the user queue for which
	 * a presence status will be requested. The queue will be resolved in a
	 * delayedTask of 500ms.
	 * @param {Zarafa.core.data.UserIdObject} user An object that identifies the user for which
	 * a presence status will be requested.
	 */
	queuePresenceRequest: function(user)
	{
		// If the user is already in the request queue, we can simply return.
		var userAlreadyinRequestQueue = false;
		Ext.each(this.requestQueue, function(queuedUser){
			if ( user.equals(queuedUser) ){
				userAlreadyinRequestQueue = true;
				return false;
			}
		}, this);
		if ( userAlreadyinRequestQueue ){
			return;
		}

		this.requestQueue.push(user);

		// If the task has not yet been created, do it now
		this.task = this.task || new Ext.util.DelayedTask(this.doQueuedRequests, this);

		// Delay the task with 500ms to buffer the requests
		this.task.delay(500);
	},

	/**
	 * {@link Ext.util.DelayedTask} handler that requests a
	 * {@link Zarafa.core.data.PresenceStatus presence status} for all
	 * users in the {#requestQueue queue}.
	 */
	doQueuedRequests: function()
	{
		var statuses = this.getPresenceStatusForUsers(this.requestQueue);

		// Store the updates to the cache and update the ui
		var statusesByPlugin = this.rearangeStatuses(this.requestQueue, statuses);
		Ext.iterate(statusesByPlugin, function(pluginName){
			this.updateStatuses(pluginName, statusesByPlugin[pluginName]);
		}, this);

		// Empty the request queue
		this.requestQueue = [];
	},

	/**
	 * Rearanges an array with presence statuses that is originally orderded by user
	 * (e.g. the return value of {#link getPresenceStatusForUsers}) to an array
	 * that is ordered by pluginName)
	 * @param {Zarafa.core.data.UserIdObject[]} users An array with users
	 * @param {Object[]} statuses An array with objects that contain a presence
	 * status per presence plugin. The entries in the statuses array correspond
	 * to the entries in the given users array.
	 * @return {Object} An object that contains an array for each presence plugin.
	 * The arrays contain objects that contain a
	 * {@link Zarafa.core.data.UserIdObject user} and a status.
	 */
	rearangeStatuses: function(users, statuses)
	{
		var statusesByPlugin = {};
		Ext.each(statuses, function(statusByPlugin, index){
			Ext.iterate(statusByPlugin, function(pluginName){
				if ( !Ext.isDefined(statusesByPlugin[pluginName]) ){
					statusesByPlugin[pluginName] = [];
				}
				statusesByPlugin[pluginName].push({
					user: users[index],
					status: statusByPlugin[pluginName]
				});
			}, this);
		}, this);

		return statusesByPlugin;
	},

	/**
	 * Registers a store with the {@link Zarafa.core.PresenceManager}. The
	 * {@link Zarafa.core.PresenceManager} will then make sure that presence statuses
	 * are fetched when the store loads or when the {@link Zarafa.core.PresenceManager}
	 * polls for updates.
	 * @param {Zarafa.core.data.MAPIStore|Zarafa.core.data.MAPISubStore} store The store
	 * that will be registered.
	 * will be added to the records. (e.g.: 'sender')
	 */
	registerStore: function(store)
	{
		if (this.getPresencePlugins().length === 0) {
			return;
		}

		var storeRegistered = Ext.each(this.registeredStores, function(registeredStore){
			if ( registeredStore.store === store ){
				return false;
			}
		}, this);

		if ( Ext.isDefined(storeRegistered) ){
			// Store was already registered
			return;
		}

		this.registeredStores.push({
			store: store
		});

		// Add the current users to the cache by calling the onLoad listener
		this.onStoreLoad(store, store.getRange(), null);

		// Register an event handler for the load event of this store
		store.on('load', this.onStoreLoad, this);
	},

	/**
	 * Unregisters a store that was previously registered with the {#registerStore}.
	 * @param {Zarafa.core.data.MAPIStore|Zarafa.core.data.MAPISubStore} store The store
	 * that will be unregistered.
	 */
	unregisterStore: function(store)
	{
		for ( var i=0; i<this.registeredStores.length; i++) {
			if ( this.registeredStores[i].store === store ){
				this.registeredStores.splice(i, 1);
				return;
			}
		}
	},

	/**
	 * Event handler for the load event of a registered store. If records are loaded that contain
	 * users for which the {@link Zarafa.core.data.PresenceCache} does not have an
	 * entry yet, then a status for these users is requested from the presence plugins.
	 * @param {Zarafa.core.data.MAPIStore|Zarafa.core.data.MAPISubStore} store The store
	 * that contains records for which presence statuses are needed.
	 * @param {Zarafa.core.data.MAPIRecord[]} records The records that are being loaded into
	 * this store.
	 * @param {Object} options The loading options that were specified
	 */
	onStoreLoad: function(store, records, options)
	{
		// Create an array with user info objects to send to the PresenceManager
		var users = Zarafa.core.data.UserIdObjectFactory.createFromStore(store);

		// Check if we already have an entry for these users in the cache
		// because otherwise we must add it and make a request to the plugins
		Ext.each(users, function(user){
			if (!Ext.isDefined(Zarafa.core.data.PresenceCache.getUser(user))){
				// Ask the presence plugins about this user
				this.queuePresenceRequest(user);
			}
		}, this);
	},

	/**
	 * Processes status updates from presence plugins. The updates will be stored
	 * in the {@link Zarafa.core.data.PresenceCache} and the UI will be updated.
	 * @param {Object} pluginName The name of the plugin from which the updates come
	 * @param {Array} updates An array with objects that contain a
	 * {Zarafa.core.data.UserIdObject UserIdObject} and a
	 * {@link Zarafa.core.data.PresenceStatus} for that user
	 */
	updateStatuses: function(pluginName, updates) {
		// Make sure we have an array
		if ( !Array.isArray(updates) ){
			updates = [updates];
		}

		// First add the statuses to the cache, and check if the UI needs to be updated
		var realUpdates = [];
		Ext.each(updates, function(update){
			var statusBeforeUpdate = Zarafa.core.data.PresenceCache.getStatusForUser(update.user);
			Zarafa.core.data.PresenceCache.addStatusForUser(pluginName, update.user, update.status);
			var statusAfterUpdate = Zarafa.core.data.PresenceCache.getStatusForUser(update.user);
			if ( statusAfterUpdate !== statusBeforeUpdate ){
				realUpdates.push(update);
			}
		});

		// Now update the stores that have records for these users
		Ext.each(this.registeredStores, function(registeredStore){
			var records = registeredStore.store.getRange();
			var modified = [];
			Ext.each(records, function(record) {
				if ( modified.indexOf(record)>=0 ){
					return;
				}

				var recordAdded = Ext.each(realUpdates, function(update) {
					var user = Zarafa.core.data.UserIdObjectFactory.createFromRecord(record);
					if ( user && user.equals(update.user) ){
						modified.push(record);
						return false;
					}
				}, this);

				if ( Ext.isDefined(recordAdded) ){
					return false;
				}
			}, this);

			// Fire update events for all modified records, so components that display these records
			// update their UI.
			if ( modified.length ){
				Ext.each( modified, function(record) {
					registeredStore.store.fireEvent('update', registeredStore.store, record, Ext.data.Record.COMMIT);
				});

				// Recipient links don't get updated when we fire an update event, so we fire a datachanged event
				if ( registeredStore.store instanceof Zarafa.core.data.IPMRecipientStore ){
					registeredStore.store.fireEvent('datachanged', registeredStore.store);
				}
			}
		}, this);
	},

	/**
	 * Requests a status from all presence plugins for all users in the
	 * registered stores.
	 */
	pollForUpdates: function()
	{
		var users = Zarafa.core.data.PresenceCache.getUserInfoList();
		var statuses = this.getPresenceStatusForUsers(users);
		var statusesByPlugin = this.rearangeStatuses(users, statuses);
		Ext.iterate(statusesByPlugin, function(pluginName){
			this.updateStatuses(pluginName, statusesByPlugin[pluginName]);
		}, this);
	}
});

// Make a singleton of this class
Zarafa.core.PresenceManager = new Zarafa.core.PresenceManager();
Ext.namespace('Zarafa.core');

/**
 * @class Zarafa.core.Request
 * @extends Ext.util.Observable
 *
 * Request object for asynchronous communication with the server. The request object
 * automatically serialises single or multiple action requests into corresponding
 * JSON and provides a callback system to allow asynchronous handling of the
 * response from the server.
 * <p>
 * The request object supports global events surrounding the sending of the data
 * to the server. For handling the responses it communicates directly with the
 * {@link Zarafa.core.ResponseRouter ResponseRouter} on which events are provided
 * surrounding the receiving of the Responses (as well as handling of the Notifications).
 * <p>
 * You shouldn't have to instantiate this class yourself as a global instance
 * can be obtained from the globally available {@link Zarafa.core.Container container}
 * object.
 * <p>
 * The structure of data that will be used for communication will be like
 * <pre><code>
 * zarafa: {
 *	module_name: {
 *		module_id: {
 *			action_type: {
 *				// data
 *			}
 *		}
 *	}
 * }
 * </code></pre>
 */
Zarafa.core.Request = Ext.extend(Ext.util.Observable, (function() {
	/**
	 * True if the connection has been {@link #paralyze}. This will
	 * prevent any requests from being send to the server, or any responses
	 * from being processed.
	 * @property
	 * @type Boolean
	 * @private
	 */
	var paralyzed = false;

	/**
	 * True if the Webapp has been {@link #interrupt}. This will
	 * queue all requests which will be send to the server, until the connection
	 * is restored again.
	 * @property
	 * @type Boolean
	 * @private
	 */
	var interrupted = false;

	/**
	 * The current base number for the generating new requestsIds
	 * using the {@link #getRequestId} function.
	 * @property
	 * @type Number
	 * @private
	 */
	var requestIdBase = 0;

	/**
	 * The current JSON object containing all requests which
	 * should be send to the server on the next call to
	 * {@link #send}. Will be reset on {@link #reset}.
	 * @property
	 * @type Object
	 * @private
	 */
	var zarafaTag;

	/**
	 * The flag to indicate if zarafaTag holds JSON data.
	 * This influences if the contents of zarafaTag will be encoded or not.
	 * @property
	 * @type Boolean
	 * @private
	 */
	var hasJson = false;

	/**
	 * The flag to indicate if zarafaTag holds RAW data.
	 * This influences if the contents of zarafaTag will be encoded or not.
	 * @property
	 * @type Boolean
	 * @private
	 */
	var hasData = false;

	/**
	 * The list of request ids as generated by {@link #getRequestId} of all
	 * requests which are added after a {@link #reset} and before {@link #send}.
	 * These will be merged into {@link #activeRequests} on {@link #send}.
	 * @property
	 * @type Array
	 * @private
	 */
	var queuedRequests = [];

	/**
	 * Key-value list of request ids and the corresponding {@link XMLHttpRequest} objects
	 * which was used to send the request to the server. These requests are all pending
	 * on the PHP side. These will be cleared as soon as the response has been received.
	 * As long as a request is listed here, it can be canceled using {@link #cancelActiveRequest}.
	 * @property
	 * @type Array
	 * @private
	 */
	var activeRequests = {};

	/**
	 * The list of {@link #XMLHttpRequest} objects which are queued while the
	 * {@link #interrupted connection is interrupted}. These will be dequeued
	 * later when the connection is restored.
	 * @property
	 * @type Array
	 * @private
	 */
	var queuedInterruptedHttpRequests = [];

	/**
	 * The unique id for subsystem which will be used to differentiate between two sessions of same user
	 * using same browser but in different tabs.
	 * @property
	 * @type String
	 * @private
	 */
	var subSystemId;

	return {
		// public variables
		/**
		 * @cfg {String} defaultUrl
		 * The url used to send the requests to. defaults to grommunio.php.
		 */
		defaultUrl: 'grommunio.php',

		/**
		 * @cfg {Object} defaultHeaders
		 * The default headers to be applied to the request. defaults to
		 *	'Content-Type' => 'application/json; charset=utf-8;'
		 */
		defaultHeaders: undefined,

		/**
		 * @cfg {String} subSystemPrefix string that will be used to generate session filename which
		 * will be used to store session data in php. current timestamp will be appended to this string
		 * so different tabs opened in same browser will have different session data. So every request object
		 * corresponds to one unique subSystem string id (accuracy upto milliseconds).
		 */
		subSystemPrefix: 'webapp',

		/**
		 * @cfg {Object} requestHeaders
		 * Headers that should be sent with every request.
		 */
		requestHeaders: undefined,

		/**
		 * @constructor
		 * @param {Object} config Configuration object
		 */
		constructor: function(config)
		{
			config = config || {};

			Ext.apply(config, {
				// Apply here instead of class prototype, to prevent
				// accidental sharing of object between all instances.
				defaultHeaders: {
					'Content-Type': 'application/json; charset=utf-8;'
				},
				requestHeaders: {}
			});

			Ext.apply(this, config);

			this.addEvents(
				/**
				 * @event connectionparalyzed
				 * Fired when the window is about to be unloaded. At this moment
				 * grommunio Web will shutdown and start dropping all communication with the PHP
				 * @param {Zarafa.core.Request} request
				 * @param {Zarafa.core.data.ParalyzeReason} reason The reason to paralyze the grommunio Web
				 */
				'connectionparalyzed',
				/**
				 * @event connectioninterrupted
				 * Fired when the connection to the server has been lost, but is likely
				 * to be restored soon. grommunio Web will start pausing all communication with
				 * the PHP until the connection has been restored.
				 * @param {Zarafa.core.Request} request
				 * @param {Zarafa.core.PingService} service The ping service which will
				 * be used to ping for the connection restoration.
				 */
				'connectioninterrupted',
				/**
				 * @event connectionrestored
				 * Fired when the connection to the server has been restored. This will
				 * resend any requests to the PHP which were scheduled while the connection
				 * was lost.
				 * @param {Zarafa.core.Request} request
				 */
				'connectionrestored',
				/**
				 * @event beforesend
				 * Fires before the XmlHttpRequest sends a request to the server.
				 * Return false to not send request to server.
				 * @param {Zarafa.core.Request} request
				 * @param {window.XMLHttpRequest} xmlHttpRequest XmlHttpRequest object
				 */
				'beforesend',
				/**
				 * @event aftersend
				 * Fires after the XmlHttpRequest sent a request to the server.
				 * @param {Zarafa.core.Request} request
				 * @param {window.XMLHttpRequest} xmlHttpRequest XmlHttpRequest object
				 */
				'aftersend'
			);

			Zarafa.core.Request.superclass.constructor.call(this, config);

			// Initialize all private variables
			this.initialize();
		},

		/**
		 * Initialize all private variables of the Request object
		 * @private
		 */
		initialize: function()
		{
			paralyzed = false;
			interrupted = false;
			requestIdBase = 0;
			zarafaTag = undefined;
			hasJson = false;
			hasData = false;
			queuedRequests = [];
			activeRequests = {};
			queuedInterruptedHttpRequests = [];
			subSystemId = this.subSystemPrefix + '_' + new Date().getTime();
		},

		/**
		 * Set the {@link #paralyzed} property which will prevent any requests from
		 * being send out to the server. This will fire the {@link #connectionparalyzed} event.
		 * @param {Zarafa.core.data.ParalyzeReason} reason The reason to paralyze grommunio Web
		 */
		paralyze: function(reason)
		{
			if (this.isParalyzed()) {
				return;
			}

			paralyzed = true;
			this.fireEvent('connectionparalyzed', this, reason);
		},

		/**
		 * @return {Boolean} True if the connection is {@link #paralyzed}.
		 */
		isParalyzed: function()
		{
			return paralyzed;
		},

		/**
		 * Set the {@link #interrupted} property which will pause all requests to be send out
		 * to the server. These will be send when the connection is {@link #restore restored}.
		 * @private
		 */
		interrupt: function()
		{
			if (this.isInterrupted() || this.isParalyzed()) {
				return;
			}

			interrupted = true;

			// Instantiate the PingService for polling the
			// server for the recovery of the connection.
			var service = new Zarafa.core.PingService({
				url: this.defaultUrl,
				headers: this.defaultHeaders
			});

			// Add event handler which will restore interaction with
			// the PHP server when the link has been restored.
			service.on('restored', this.restore, this);

			// Add some event handlers which will stop the PingService
			// as soon as the connection is restored, or if grommunio Web
			// is being paralyzed. We defer the connectionrestored by 1 ms,
			// so we force the 'stop' event from the service to be fired after
			// all 'restored' event handlers from the service have been handled.
			this.on('connectionparalyzed', service.stop, service);
			this.on('connectionrestored', service.stop, service, { delay: 1 });

			// Fire the event, allow interested parties to register
			// for events on the service before we start it.
			this.fireEvent('connectioninterrupted', this, service);

			// start the service
			service.start();
		},

		/**
		 * Clear the {@link #interrupted} property which will restore all requests which were
		 * paused while the connection was interrupted.
		 * @param {Zarafa.core.PingService} service This object
		 * @param {Object} response The response as send by the server
		 * @private
		 */
		restore: function(service, response)
		{
			if (!this.isInterrupted()) {
				return;
			}

			// If the response object indicates that our session
			// is no longer active, we must paralyze grommunio Web
			// as no further requests can be send to the server.
			if (response && response.active === false) {
				this.paralyze(Zarafa.core.data.ParalyzeReason.SESSION_EXPIRED);
				return;
			} else {
				interrupted = false;
				this.fireEvent('connectionrestored', this);

				// If there was a HTTP request queued, we should
				// send the first one now. The next one will be
				// send when the first one has responded. (This prevents
				// a DDOS if the connection was lost for a long time,
				// and many clients are connected to the server).
				if (this.hasQueuedHttpRequests()) {
					this.dequeueHttpRequest();
				}
			}
		},

		/**
		 * @return {Boolean} True if the connection is {@link #interrupted}.
		 */
		isInterrupted: function()
		{
			return interrupted;
		},

		/**
		 * Sets request headers that will be sent with every request made through {@link Zarafa.core.Request} object.
		 * @param {String} key header key.
		 * @param {String} value header value.
		 */
		setRequestHeader: function(key, value)
		{
			this.requestHeaders[key] = value;
		},

		/**
		 * Generates a new unique Request ID, which can be used to match
		 * the request with a response.
		 * @param {String} prefix (optional) The prefix for the unique request id.
		 * @return {String} request ID
		 */
		getRequestId: function(prefix)
		{
			return (prefix || 'z-gen') + (++requestIdBase);
		},

		/**
		 * Function will be used to cancel a previously generated request which is waiting for the server request.
		 * When doing multiple requests we need to make sure that previous requests are cancelled otherwise data
		 * can be overwritten by previous requests. Special requirement was that we can't do {@link window.XMLHttpRequest#abort}
		 * as it will abort the whole request and discard data that was added to response by server side, which is not
		 * good when server has added any notification data to response (like new mail), so we will need to overwrite
		 * responseHandler of this particular request with {@link Zarafa.core.data.DummyResponseHandler}, which will
		 * have empty functions for discarding response came from server and notifications will be handled normally.
		 * @param {String} requestId The request id.
		 */
		cancelActiveRequest: function(requestId)
		{
			var responseRouter = container.getResponseRouter();

			// If the request is still queued, we have an opportunity to
			// prevent the entire request. However this is only possible
			// if no other request was queued, as then we can cancel all the
			// pending data. If that is not the case, we have to register the
			// DummyResponseHandler
			if (queuedRequests.indexOf(requestId) >= 0) {
				if (queuedRequests.length === 1) {
					// Reset environment
					this.reset();
					responseRouter.removeRequestResponseHandler(requestId);
				} else {
					responseRouter.addRequestResponseHandler(requestId, new Zarafa.core.data.DummyResponseHandler());
				}
				return;
			}

			// If there is a link interruption, then the request might still be queued.
			// This means we can interrupt the request before it is being send to the
			// server.
			if (this.hasQueuedHttpRequests()) {
				// Search through the list of pending HTTP requests
				// to find the one which holds our cancelled request id.
				for (var i = 0; i < queuedInterruptedHttpRequests.length; i++) {
					var xhrObj = queuedInterruptedHttpRequests[i];
					var index = xhrObj.queuedRequests.indexOf(requestId);

					if (index >= 0) {
						// We found the request, but now comes the problem.
						// We can only cancel the entire HTTPRequest if this
						// was the only request on the object. Because we don't
						// know which part of the JSON corresponds to this request.
						if (xhrObj.queuedRequests.length === 1) {
							queuedInterruptedHttpRequests.splice(i, 1);
							responseRouter.removeRequestResponseHandler(requestId);
							return;
						}

						// If we can't cancel the request, we can break
						// the loop and just change the response handler.
						break;
					}
				}

				// We didn't find the request to which the request corresponds,
				// or we couldn't cancel the entire request. In either case,
				// we have to register the DummyResponseHandler.
				responseRouter.addRequestResponseHandler(requestId, new Zarafa.core.data.DummyResponseHandler());
				return;
			}

			// If the request has been send out to the server, we cannot cancel
			// it and thus we will have to register the DummyResponseHandler to
			// prevent the response from being processed. However we can mark
			// the corresponding xmlHttpRequest so we will not retry it if all
			// requests have been cancelled.
			var xhrObj = activeRequests[requestId];
			if (Ext.isDefined(xhrObj)) {
				// Only prevent the retry if all requests attached
				// to the xmlHttpRequest have been cancelled.
				if (xhrObj.queuedRequests.length === 1) {
					xhrObj.preventRetry = true;
				}
				responseRouter.addRequestResponseHandler(requestId, new Zarafa.core.data.DummyResponseHandler());
				return;
			}
		},

		/**
		 * Queue a request to {@link #queuedRequests}. This can be dequeued
		 * in {@link #dequeueRequests}.
		 * @param {String} The request to be queued
		 * @private
		 */
		queueRequest: function(request)
		{
			queuedRequests.push(request);
		},

		/**
		 * Obtain and dequeue any requests from {@link #queuedRequests}. Use
		 * {@link #activateRequests} to add them the the {@link #activeRequests} array
		 * once the request has been send out.
		 * @return {Array} The array of previously queued requests
		 * @private
		 */
		dequeueRequests: function()
		{
			var requests = queuedRequests;
			queuedRequests = [];
			return requests;
		},

		/**
		 * Move all requests into {@link #activeRequests}. The provided requests
		 * should previously have been obtained through {@link #dequeueRequests}.
		 * @param {Array} requests The list of requests to activate.
		 * @private
		 */
		activateRequests: function(requests, xmlHttpRequest)
		{
			for (var i = 0; i < requests.length; i++) {
				activeRequests[requests[i]] = xmlHttpRequest;
			}
		},

		/**
		 * Remove all given requests from the {@link #activeRequests} queue.
		 * @param {Array} requests The list of requests to complete
		 * @private
		 */
		completeRequests: function(requests)
		{
			for (var i = 0, len = requests.length; i < len; i++) {
				delete activeRequests[requests[i]];
			}
		},

		/**
		 * Prepare a {@link XMLHttpRequest} object which will be used to transmit data to the server for processing.
		 * it also registers a callback function for onreadystatechange event that will be called when server sends response.
		 * @param {Mixed} requestData data that will be send using this request object, for XML request it
		 * will be XML document or else it will be JSON string.
		 * @param {Boolean} encoded True of the requestData is encoded, and must be decoded to be used.
		 * @param {String} url (optional) url to post to. Defaults to {@link #defaultUrl}
		 * @param {Object} headers (optional) headers to apply to the request. Defaults to {@link #defaultHeaders}
		 * @return {XMLHttpRequest} The initialized XMLHttpRequest
		 * @private
		 */
		prepareHttpRequest: function(requestData, encoded, url, headers)
		{
			var xmlHttpRequest = new XMLHttpRequest();

			if (!url) {
				url = this.defaultUrl;
				url = Ext.urlAppend(url, 'subsystem=' + subSystemId);
			}

			xmlHttpRequest.open('POST', url, true);
			xmlHttpRequest.requestUrl = url;

			// Apply header from argument, or apply the default headers
			headers = Ext.apply({}, this.requestHeaders, headers || this.defaultHeaders);
			for (var key in headers) {
				xmlHttpRequest.setRequestHeader(key, headers[key]);
			}
			xmlHttpRequest.requestHeaders = headers;

			// Store requestData into the xmlHttpRequest, when the request failed,
			// we can still determine report the failure back to the requestee.
			xmlHttpRequest.requestData = requestData;
			xmlHttpRequest.requestDataEncoded = encoded;
			xmlHttpRequest.queuedRequests = this.dequeueRequests();

			return xmlHttpRequest;
		},

		/**
		 * Clone a {@link XMLHttpRequest} instance which was prepared by {@link #prepareHttpRequest} and
		 * has previously been attempted to be {@link #sendHttpRequest send out} but failed due to a broken
		 * connection. This will clone the instance, so the request can be retried when the connection
		 * returns.
		 * @param {XMLHttpRequest} xmlHttpRequest The request object to clone
		 * @return {XMLHttpRequest} The cloned XMLhttpRequest object
		 * @private
		 */
		cloneRequest: function(xmlHttpRequest)
		{
			var cloneRequest = new XMLHttpRequest();
			cloneRequest.open('POST', xmlHttpRequest.requestUrl, true);
			cloneRequest.requestUrl = xmlHttpRequest.requestUrl;

			var headers = xmlHttpRequest.requestHeaders;
			for (var key in headers) {
				cloneRequest.setRequestHeader(key, headers[key]);
			}
			cloneRequest.requestHeaders = headers;

			cloneRequest.requestData = xmlHttpRequest.requestData;
			cloneRequest.requestDataEncoded = xmlHttpRequest.requestDataEncoded;
			cloneRequest.queuedRequests = xmlHttpRequest.queuedRequests;

			return cloneRequest;
		},

		/**
		 * Send out a {@link XMLHttpRequest} which was initialized by {@link #prepareHttpRequest}. This will
		 * perform the final step for transmitting the request, by firing the {@link #beforesend} event,
		 * and {@link #activeRequests activating the queued requests}.
		 * @param {XMLHttpRequest} xmlHttpRequest The request object
		 * @private
		 */
		sendHttpRequest: function(xmlHttpRequest)
		{
			var requestData = xmlHttpRequest.requestData;
			var requests = xmlHttpRequest.queuedRequests;

			// When the connection paralyzed, we cannot handle any incoming data anymore.
			// So all outgoing requests will be dropped.
			if (this.isParalyzed()) {
				return;
			}

			if (this.fireEvent('beforesend', this, xmlHttpRequest) === false) {
				return;
			}

			// Register the onready StateChange event handler
			xmlHttpRequest.onreadystatechange = this.stateChange.createDelegate(this, [xmlHttpRequest]);

			// Move the queued requests to the active list.
			this.activateRequests(requests, xmlHttpRequest);

			// send request
			xmlHttpRequest.send(requestData);

			this.fireEvent('aftersend', this, xmlHttpRequest);
		},

		/**
		 * {@link #queuedInterruptedHttpRequests Queue} a {@link XMLHttpRequest} which was initialized by
		 * {@link #prepareHttpRequest}. This will keep the request instance until it can be send
		 * to the server at a {@link #dequeueHttpRequest later time}.
		 * @param {XMLHttpRequest} xmlHttpRequest The request object
		 * @private
		 */
		queueHttpRequest: function(xmlHttpRequest)
		{
			queuedInterruptedHttpRequests.push(xmlHttpRequest);
		},

		/**
		 * Check if there are queued {@link #queuedInterruptedHttpRequests HTTP requests}
		 * @return {Boolean} True if there are queued HTTP requests
		 * @private
		 */
		hasQueuedHttpRequests: function()
		{
			return !Ext.isEmpty(queuedInterruptedHttpRequests);
		},

		/**
		 * Dequeue a {@link #queuedInterruptedHttpRequests HTTP request} and {@link #sendHttpRequest send it} to the server
		 * @private
		 */
		dequeueHttpRequest: function()
		{
			var xmlHttpRequest = queuedInterruptedHttpRequests.shift();
			if (xmlHttpRequest) {
				this.sendHttpRequest(xmlHttpRequest);
			}
		},

		/**
		 * Called by XMLHttpRequest when a response from the server is coming in. When the entire response has been
		 * completed it calls {@link Zarafa.core.ResponseRouter#receive receive}} which handles the rest of the process.
		 * @param {Object} xmlHttpRequest The raw HTTP request object that is used for communication.
		 * @private
		 */
		stateChange: function(xmlHttpRequest)
		{
			var requestIds = xmlHttpRequest.queuedRequests;
			var responseRouter = container.getResponseRouter();
			var response;

			// When the connection is paralyzed, we cannot handle any incoming data anymore.
			// All incoming responses from the server will be dropped.
			if (this.isParalyzed()) {
				return;
			}

			// The readyState can be 4 values:
			// 0 - Object is created, but not initialized
			// 1 - Request has been opened, but send() has not been called yet
			// 2 - send() has been called, no data available yet
			// 3 - Some data has been received, responseText nor responseBody are available
			// 4 - All data has been received
			//
			// readyState 0 - 3 can be completely ignored by us, as they are only updates
			// about the current progress. Only on readyState 4, should we continue and
			// start checking for the response status.
			if (xmlHttpRequest.readyState != 4) {
				return;
			}

			// The transaction is complete, all requests must now be cleared from the list.
			// Note that when we got a HTTP error, the requests are still removed, since
			// the HTTP request is no longer pending.
			this.completeRequests(requestIds);

			// HTTP request must have succeeded
			switch (xmlHttpRequest.status) {
				case 401: /* Unauthorized */
					// Indicate that the user is no longer logged in, and
					// he must re-authenticate. This must be done through the
					// normal logon page, so here we just paralyze the Request.
					// The exact reason for the paralyzation can be found in the
					// headers.
					var reason = xmlHttpRequest.getResponseHeader('X-grommunio-Hresult');
					if (reason === 'MAPI_E_INVALID_WORKSTATION_ACCOUNT') {
						this.paralyze(Zarafa.core.data.ParalyzeReason.SESSION_INVALID);
					} else {
						this.paralyze(Zarafa.core.data.ParalyzeReason.SESSION_EXPIRED);
					}
					return;
				case 500: /* Internal Server Error */
					// The connection is present, if there still are queued
					// HTTP requests, we can send the next one right now.
					if (this.hasQueuedHttpRequests()) {
						this.dequeueHttpRequest();
					}

					// Indicate that the request failed
					// inside the server and exit the function.
					this.receiveFailure(xmlHttpRequest);
					return;
				case 200: /* OK */
					// The connection is present, if there still are queued
					// HTTP requests, we can send the next one right now.
					if (this.hasQueuedHttpRequests()) {
						this.dequeueHttpRequest();
					}
					break;
				default: /* Connection errors */
					// Interrupt the connection
					this.interrupt();

					// Clone the XMLHttpRequest and queue it
					// for when the connection is restored.
					if (xmlHttpRequest.preventRetry !== true) {
						var clone = this.cloneRequest(xmlHttpRequest);
						this.queueHttpRequest(clone);
					}
					return;
			}

			// Depending on the response type, convert it into a data Object.
			if (xmlHttpRequest.responseText) {
				// JSON response
				response = Ext.decode(xmlHttpRequest.responseText);
			} else {
				// XML response is not supported
				this.receiveFailure(xmlHttpRequest);
				return;
			}

			// Check for empty response, sometimes the PHP server doesn't bother with
			// responding with any data.
			if (Ext.isEmpty(response) || Ext.isEmpty(response.zarafa)) {
				this.receiveFailure(xmlHttpRequest);
				return;
			}

			responseRouter.receive(response);
		},

		/**
		 * Called when the {@link XMLHttpRequest} object indicates a problem
		 * in the returned data from the server. This will call
		 * {@link Zarafa.core.ResponseRouter#receiveFailure receiveFailure} on
		 * the {@link Zarafa.core.ResponseRouter ResponseRouter}.
		 * @param {XMLHttpRequest} xmlHttpRequest The xmlHttpRequest which contains the problem
		 * @private
		 */
		receiveFailure: function(xmlHttpRequest)
		{
			var responseRouter = container.getResponseRouter();
			var requestData = xmlHttpRequest.requestData;
			if (xmlHttpRequest.requestDataEncoded) {
				requestData = Ext.decode(requestData);
			}
			responseRouter.receiveFailure(requestData, xmlHttpRequest);
		},

		/**
		 * Resets the {@link Zarafa.core.Request Request} object and prepares it for
		 * accepting new calls to {@link #addRequest} or {@link #singleRequest}.
		 */
		reset: function()
		{
			queuedRequests = [];

			zarafaTag = {
				'zarafa': {}
			};
			hasJson = false;
			hasData = false;
		},

		/**
		 * Adds a single request to the Request object. This is a combination of a module name, id and action.
		 * The parameters objected will be encoded and added into the action tag.
		 * The callbacks are used when the responds for this specific request has returned.
		 * @param {String} moduleName name of the module to communicate with (i.e. 'addressbooklistmodule')
		 * @param {String} actionType action to perform (i.e. 'list' or 'globaladdressbook')
		 * @param {Object} actionData data that will included in the action tag (i.e. { restriction: { name: 'piet' } })
		 * @param {Zarafa.core.data.AbstractResponseHandler} responseHandler The response handler which must be
		 * used for handling the responds for this specific request.
		 * @return {String} The unique Request ID which was assigned to this request.
		 */
		addRequest: function(moduleName, actionType, actionData, responseHandler)
		{
			if (Ext.isEmpty(zarafaTag)) {
				throw 'Request object not initialised. Call reset() first';
			}
			if (hasData) {
				throw 'Request object initialized with RAW data';
			}

			var requestId = this.getRequestId(moduleName);
			this.queueRequest(requestId);

			if (Ext.isDefined(responseHandler)) {
				container.getResponseRouter().addRequestResponseHandler(requestId, responseHandler);
			}

			actionData = Ext.value(actionData, {});

			// create new module tag if not present
			if (!Ext.isDefined(zarafaTag.zarafa[moduleName])) {
				zarafaTag.zarafa[moduleName] = {};
			}

			// create new module id tag if not present
			if (!Ext.isDefined(zarafaTag.zarafa[moduleName][requestId])) {
				zarafaTag.zarafa[moduleName][requestId] = {};
			}

			// add action data
			zarafaTag.zarafa[moduleName][requestId][actionType] = actionData;

			// JSON data was added
			hasJson = true;

			return requestId;
		},

		/**
		 * Adds a single request to the Request object. Opposed to {@link #addRequest} this will be raw data.
		 * The parameters object will be placed to the action tag.
		 * The callbacks are used when the responds for this specific request has returned.
		 * @param {String} moduleName name of the module to communicate with (i.e. 'addressbooklistmodule')
		 * @param {String} actionType action to perform (i.e. 'list' or 'globaladdressbook')
		 * @param {Object} actionData data that will included in the action tag
		 * @param {Zarafa.core.data.AbstractResponseHandler} responseHandler The response handler which must be
		 * used for handling the responds for this specific request.
		 * @return {String} The unique Request ID which was assigned to this request.
		 */
		addDataRequest: function(moduleName, actionType, actionData, responseHandler)
		{
			if (Ext.isEmpty(zarafaTag)) {
				throw 'Request object not initialised. Call reset() first';
			}
			if (hasJson) {
				throw 'Request object initialized with JSON data';
			}
			if (hasData) {
				throw 'Request object already contains RAW data';
			}

			var requestId = this.getRequestId(moduleName);
			this.queueRequest(requestId);

			if (Ext.isDefined(responseHandler)) {
				container.getResponseRouter().addRequestResponseHandler(requestId, responseHandler);
			}

			// add action data
			zarafaTag = actionData;

			// Raw data was added
			hasData = true;

			return requestId;
		},

		/**
		 * Calls {@link #prepareHttpRequest} to actually create the request using passed data,
		 * and calls {@link #sendHttpRequest} to send request.
		 * @param {String} url (optional) url to post to. Defaults to {@link #defaultUrl}
		 * @param {Object} headers (optional) headers to apply to the request. Defaults to {@link #defaultHeaders}
		 */
		send: function(url, headers)
		{
			if (Ext.isEmpty(zarafaTag)) {
				throw 'Request object not initialised. Call reset() first';
			}

			if (Ext.isEmpty(queuedRequests)) {
				throw 'No requests have been added. Use addRequest()';
			}

			if (!this.isParalyzed()) {
				var xmlHttpRequest = this.prepareHttpRequest(hasJson ? Ext.encode(zarafaTag) : zarafaTag, hasJson, url, headers);
				if (this.isInterrupted() || this.hasQueuedHttpRequests()) {
					this.queueHttpRequest(xmlHttpRequest);
				} else {
					this.sendHttpRequest(xmlHttpRequest);
				}
			}

			// All data has been send, reset the zarafaTag to
			// prevent sending the same request twice.
			zarafaTag = undefined;
			hasJson = false;
			hasData = false;
		},

		/**
		 * Convenience method for performing a single JSON request to the server.
		 * It calls {@link #reset reset()}, {@link #addRequest addRequest()}, and {@link #send send()} in turn.
		 * @param {String} moduleName name of the module to communicate with (i.e. 'addressbooklistmodule')
		 * @param {String} actionType action to perform (i.e. 'list' or 'globaladdressbook')
		 * @param {Object} actionData data that will included in the action tag (i.e. { restriction: { name: 'piet' } })
		 * @param {Zarafa.core.data.AbstractResponseHandler} responseHandler The response handler which must be
		 * used for handling the responds for this specific request.
		 * @return {String} The unique request ID which was assigned to this transaction object.
		 */
		singleRequest: function(moduleName, actionType, actionData, responseHandler)
		{
			var requestId;

			this.reset();
			requestId = this.addRequest(moduleName, actionType, actionData, responseHandler);
			this.send();

			return requestId;
		},

		/**
		 * Method to {@link window.XMLHttpRequest#abort} given request discarding whatever the data
		 * added to response by server.
		 * @param {XMLHttpRequest} xhrObj Object of the request made previously.
		 */
		abortRequest: function(xhrObj)
		{
			xhrObj.preventRetry = true;
			xhrObj.abort();
		},

		/**
		 * Method gives active {@link XMLHttpRequest request} object based on given requestId which
		 * was used to send the request to server.
		 * @param {String} requestId Unique identifier of the request made previously.
		 */
		getActiveRequest: function(requestId)
		{
			return activeRequests[requestId];
		}
	};
})());
Ext.namespace('Zarafa.core');

/**
 * @class Zarafa.core.ResponseRouter
 * @extends Ext.util.Observable
 *
 * The router for Responses to requests made by the {@link Zarafa.core.Request Request} object.
 * Each response is delivered to its destination {@link Ext.data.Store store} through the
 * {@link Zarafa.core.data.AbstractResponseHandler ResponseHandler}.
 * Upon receiving a response, the ResponseRouter will determine if it is a response to
 * a direct request from the {@link Zarafa.core.Request Request Object} or a notification
 * generated by the PHP-side.
 * If the response came from a request from the {@link Zarafa.core.Request Request Object} a
 * {@link Zarafa.core.data.AbstractResponseHandler ResponseHandler} will have been registered
 * by the {@link Ext.data.DataProxy Proxy} which made the request. If the
 * {@link Zarafa.core.data.AbstractResponseHandler ResponseHandler} is not available the
 * response is considered a Notification, in which case the
 * {@link Zarafa.core.data.NotificationResolver NotificationResolver} is used to generate
 * a special {@link Zarafa.core.data.AbstractNotificationResponseHandler ResponseHandler} which can
 * update the {@link Ext.data.Store stores} which contain the updated {@link Ext.data.Record records}.
 */
Zarafa.core.ResponseRouter = Ext.extend(Ext.util.Observable, {
	/**
	 * The collection of {@link Zarafa.core.data.AbstractResponseHandler ResponseHandlers}
	 * stored by using the moduleid of the outgoing request.
	 * @property
	 * @type Object
	 */
	responseHandlers: undefined,

	/**
	 * @constructor
	 * @param {Object} config Configuration object
	 */
	constructor: function(config)
	{
		config = config || {};

		Ext.applyIf(config, {
			responseHandlers: {}
		});

		this.addEvents(
			/**
			 * @event beforereceive
			 * Main event which is triggered when data has been received from
			 * the PHP server, and is about to be processed by the router.
			 * @param {Object} data The data which was received by the router.
			 */
			'beforereceive',
			/**
			 * @event afterreceive
			 * Main event which is triggered when the data which has been received from
			 * the PHP server has been processed.
			 * @param {Object} data The data which was received by the router.
			 */
			'afterreceive',
			/**
			 * @event receiveexception
			 * Main event which is triggered when a Request has failed, or the response
			 * doesn't contain sufficient data to handle it.
			 * @param {Object} requestdata The request data which was send to the server.
			 * @param {Object} xmlHttpRequest The raw browser response object.
			 */
			'receiveexception',
			/**
			 * @event response
			 * Main event which is triggered when a response has been received
			 * from the PHP server.
			 * @param {String} module The module name for this response
			 * @param {String} id The module id for this response
			 * @param {Object} response The response which was received.
			 * @param {Number} timestamp The {@link Date#getTime timestamp} on which the response was received
			 * @return {Boolean} False to cancel the handling of the response by ResponseHandlers
			 */
			'response'
		);

		Ext.apply(this, config);

		Zarafa.core.ResponseRouter.superclass.constructor.call(this, config);
	},

	/**
	 * Register a {@link Zarafa.core.data.AbstractResponseHandler ResponseHandler} to the
	 * Response Router. This handler will be used to handle the Response for the request
	 * with the given identifier.
	 * @param {String} id The unique request identifier on which the ResponseHandler
	 * must be registerdd.
	 * @param {Zarafa.core.data.AbstractResponseHandler} handler The ResponseHandler
	 * which must be registered for the given id.
	 */
	addRequestResponseHandler: function(id, handler)
	{
		this.responseHandlers[id] = handler;
	},

	/**
	 * Get a {@link Zarafa.core.data.AbstractResponseHandler ResponseHandler} from
	 * the {@link #responseHandlers} which has been registered for the given module identifier.
	 * This will automatically deregister the handler to prevent it being used twice.
	 * @param {String} id The unique request identifier on which the ResponseHandler
	 * could be registered.
	 * @return {Zarafa.core.data.AbstractResponseHandler} The registered response handler.
	 * @private
	 */
	getRequestResponseHandler: function(id)
	{
		var handler = this.responseHandlers[id];
		this.removeRequestResponseHandler(id);
		return handler;
	},

	/**
	 * Removes a {@link Zarafa.core.data.AbstractResponseHandler ResponseHandler} from the
	 * {@link Zarafa.core.ResponsRouter ResponsRouter} to prevent it being called twice.
	 * @param {String} id The unique request identifier of the ResponseHandler which will be
	 * deregistered from {@link Zarafa.core.ResponsRouter ResponsRouter}.
	 */
	removeRequestResponseHandler: function(id)
	{
		delete this.responseHandlers[id];
	},

	/**
	 * Get a {@link Zarafa.core.data.AbstractResponseHandler ResponseHandler} from
	 * the {@link Zarafa.core.data.NotificationResolver NotificationResolver}. This
	 * will construct a special {@link Zarafa.core.data.AbstractResponseHandler ResponseHandler}
	 * which is dedicated to handling this specific notification.
	 * @param {String} module The module from which the notification originated.
	 * @param {Object} data The response data which was send as notification, this is used
	 * to determine which {@link Ext.data.Store stores} are affected by this notification.
	 * @private
	 */
	getNotificationResponseHandler: function(module, data)
	{
		return container.getNotificationResolver().getHandlerForResponse(module, data);
	},

	/**
	 * This will have a Response from the PHP server and will process it
	 * by delivering the Response over the configured route to the destination.
	 *
	 * @param {Object} data The data object which was received and
	 * must be processed.
	 */
	receive: function(data)
	{
		this.fireEvent('beforereceive', data);

		this.processResponse(data);

		this.fireEvent('afterreceive', data);
	},

	/**
	 * This will report a Receive failure which can be triggered when
	 * the request failed with HTTP-error (e.g. 404) or when the response data
	 * was incomplete. This will go through all
	 * {@link Zarafa.core.data.AbstractResponseHandlers ResponseHandlers} which
	 * are affected by this error to report the error using:
	 * {@link Zarafa.core.data.AbstractResponseHandlers#responseFailure responseFailure}.
	 * @param {Object} requestdata The request data which was send to the server.
	 * @param {Object} xmlHttpRequest The raw browser response object.
	 */
	receiveFailure: function(requestData, xmlHttpRequest)
	{
		this.fireEvent('receiveexception', requestData, xmlHttpRequest);

		// Without requestData we cannot report the request failure
		// back to the requestee.
		if (!Ext.isObject(requestData)) {
			return;
		}

		// Find all registered ResponseHandlers which are affected by this
		// failure and propagate the responseFailure to them,
		Ext.iterate(requestData.zarafa, function(moduleName, modules) {
			Ext.iterate(modules, function(moduleId, moduleData) {
				var handler = this.getRequestResponseHandler(moduleId);
				if (!Ext.isEmpty(handler)) {
					handler.responseFailure(xmlHttpRequest);
				}
			}, this);
		}, this);
	},

	/**
	 * Resolve all response data into a collection of {@link Zarafa.core.data.AbstractResponseHandlers}
	 * which will be in charge of handling all responses. The responsehandlers will be returned into
	 * an array which is sorted on priority, meaning that the response handlers should be called in
	 * the order in which they are listed in the array.
	 * @param {Object} data The data from the response
	 * @return {Array} Array of objects containing the data
	 * @private
	 */
	resolveResponseHandlers: function(data)
	{
		var responses = [];
		var notifications = [];

		// Iterate over all modules and ids, and obtain the corresponding
		// ResponseHandlers. We separate the RequestResponses from the notifications.
		Ext.iterate(data, function(moduleName, modules) {
			// iterate over module ids
			Ext.iterate(modules, function(moduleId, moduleData) {
				var handler = {
					moduleName: moduleName,
					moduleId: moduleId,
					moduleData: moduleData
				};

				// Check if a RequestResponse Handler is registered for this moduleId
				handler.handler = this.getRequestResponseHandler(moduleId);
				if (!Ext.isEmpty(handler.handler)) {
					responses.push(handler);
					return;
				}

				// No RequestResponse was available, this is a notification
				handler.handler = this.getNotificationResponseHandler(moduleName, moduleData);
				if (!Ext.isEmpty(handler.handler)) {
					notifications.push(handler);
					return;
				}
			}, this);
		}, this);

		// Return the objects as a single array, the RequestResponses have highest priority,
		// followed by the notifications.
		return responses.concat(notifications);
	},

	/**
	 * Perform a transaction through the {@link Zarafa.core.data.AbstractResponseHandler ResponseHandler}.
	 *
	 * @param {Zarafa.core.data.AbstractResponseHandler} handler The handler which should be used
	 * for handling the response.
	 * @param {String} moduleName The name of the module from which the response was received
	 * @param {String} moduleId The unique id which is used to correlate the response to a request
	 * @param {Object} moduleData The data which was provided for the given moduleName/moduleId
	 * @param {Number} timestamp The {@link Date#getTime timestamp} on which the response was received
	 * @private
	 */
	handleResponse: function(handler, moduleName, moduleId, moduleData, timestamp)
	{
		var success = true;

		// Begin the Response transaction. When the transaction cannot
		// be started, we bail out immediately.
		if (handler.start(moduleName, moduleId, moduleData, timestamp) === false) {
			return;
		}

		if (Ext.isObject(moduleData)) {
			// Iterate over each action, and start handling them with
			// the corresponding actionData. If one of the handlers indicate
			// failure, we only change the 'success' status, but continue
			// with the other handlers. The 'success' status itself will
			// be used when the transaction is being completed.
			Ext.iterate(moduleData, function(actionType, actionData) {
				if (handler.handle(actionType, actionData) === false) {
					success = false;
				}
			}, this);
		}

		// Complete the transaction.
		handler.done(success);
	},

	/**
	 * Processes a response from the server. the data is examined for
	 * any error tags and call error listeners.
	 * @param {Object} jsonData A JSON object containing server response.
	 * @private
	 */
	processResponse: function(jsonData)
	{
		// check for errors, these are global errors which can be generated from grommunio.php
		// file, module level errors will be handled by module callback functions.
		if (!Ext.isEmpty(jsonData.zarafa.error)) {
			// Fire the exception event on the DataProxy like this, as the response cannot be matched to a specific proxy.
			Ext.data.DataProxy.fireEvent('exception', Ext.data.DataProxy, 'remote', null, null, jsonData.zarafa, null);
			return;
		}

		// Create the timestamp which is used as receive date for the current response
		var timestamp = new Date().getTime();

		// when all's fine, unpack the server response and obtain the responseHandlers
		var handlers = this.resolveResponseHandlers(jsonData.zarafa);

		for (var i = 0, len = handlers.length; i < len; i++) {
			var handler = handlers[i];
			var moduleName = handler.moduleName;
			var moduleId = handler.moduleId;
			var moduleData = handler.moduleData;
			var responseHandler = handler.handler;

			if (this.fireEvent('response', moduleName, moduleId, moduleData, timestamp) !== false) {
				this.handleResponse(responseHandler, moduleName, moduleId, moduleData, timestamp);
			}
		}
	}
});
Ext.namespace('Zarafa.core');
Ext.namespace('Zarafa.core.themes');

/**
 * Central theme registration for all available color themes.
 * This file registers all themes at once instead of requiring separate JS files per theme.
 *
 * @class Zarafa.core.Themes
 */
Zarafa.core.Themes = {
	/**
	 * List of all available unified themes (CSS-based, no separate theme files)
	 */
	themes: [
		{ name: 'purple', displayName: _('Purple') },
		{ name: 'orange', displayName: _('Orange') },
		{ name: 'lime', displayName: _('Lime') },
		{ name: 'magenta', displayName: _('Magenta') },
		{ name: 'highcontrast', displayName: _('High Contrast') },
		{ name: 'cyan', displayName: _('Cyan') },
		{ name: 'teal', displayName: _('Teal') },
		{ name: 'indigo', displayName: _('Indigo') },
		{ name: 'red', displayName: _('Red') },
		{ name: 'green', displayName: _('Green') },
		{ name: 'brown', displayName: _('Brown') },
		{ name: 'dark', displayName: _('Dark') }
	],

	/**
	 * Register all themes with the container
	 */
	registerAll: function()
	{
		Ext.each(this.themes, function(themeInfo) {
			// Create a proper class name (capitalize first letter of each word)
			var className = 'Theme' + themeInfo.name.charAt(0).toUpperCase() + themeInfo.name.slice(1);

			// Create a dynamic theme class for each theme
			Zarafa.core.themes[className] = Ext.extend(Zarafa.core.ThemePlugin, {});

			// Register the theme plugin
			container.registerPlugin(new Zarafa.core.PluginMetaData({
				name: themeInfo.name,
				displayName: themeInfo.displayName,
				allowUserDisable: false,
				allowUserVisible: false,
				pluginConstructor: Zarafa.core.themes[className]
			}));
		}, this);
	}
};

// Register all themes when ready
Zarafa.onReady(function() {
	Zarafa.core.Themes.registerAll();
});
Ext.namespace('Zarafa.core');

/**
 * @class Zarafa.core.URLActionMgr
 * @extends Object
 *
 * The {@link Zarafa.core.URLActionMgr URLActionMgr} singleton. This singleton will be
 * used to execute actions that are passed in URL, such as mailto action
 * @singleton
 */
Zarafa.core.URLActionMgr = Ext.extend(Ext.util.Observable, {
	/**
	 * The list of registered {@link Zarafa.core.URLAction action handlers}.
	 * Multiple action handlers can be registered for a single url action.
	 * @property
	 * @type Object
	 * @private
	 */
	mappings: undefined,

	/**
	 * @constructor
	 */
	constructor: function()
	{
		this.mappings = {};
	},

	/**
	 * Registers the url action configuration to the specified action.
	 * This will make sure if already a handler is registered for a particular action
	 * then another handler is appended, so both handlers will be executed.
	 *
	 * @param {String} action The name of the URL action to register on.
	 * @param {Object} handlerConfig The config containing handler info for action.
	 * In this object handler is a required key.
	 */
	register: function(action, handlerConfig)
	{
		if(!this.mappings[action]){
			this.mappings[action] = [];
		}

		// Register/Merge action configuration options to mappings array.
		this.mappings[action] = this.mappings[action].concat(handlerConfig);
	},

	/**
	 * Function will be called to execute any URL action, function will find proper handler
	 * based on action and will call handlers which are registered for the action.
	 *
	 * @param {Object} data The data containing action and action data.
	 */
	execute: function(data)
	{
		Ext.iterate(data, function(action, actionData) {
			// check if we have any handlers registered for this action
			var handlers = this.mappings[action];

			if(handlers) {
				Ext.each(handlers, function(handler) {
					// call handler to execute url action
					handler.handler.call(handler.scope || this, action, actionData, handler);
				});
			}
		}, this);
	}
});

Zarafa.core.URLActionMgr = new Zarafa.core.URLActionMgr();Ext.namespace('Zarafa.core');

/**
 * @class Zarafa.core.Util
 * Utility class
 * @singleton
 */
Zarafa.core.Util =
{
	/**
	 * @cfg {Boolean} skipRequester This flag specifies if confirm dialog will skip.
	 * If it is true then confirm dialog will not show otherwise it will show.
	 */
	skipRequester: false,

	/**
	 * Sort an array of objects
	 *
	 * @param {Array} list The array of objects which must be sorted
	 * @param {String} order The order in which the array must be sorted, can be either "ASC" or "DESC"
	 * @param {String/Function} sort (optional) Can either be a string, a function or nothing.
	 * A String argument will be used to sort the array on the given fieldname. A function will be used
	 * as comparison function for sorting. When not provided the default comparison function shall be
	 * used which compares items based on their value.
	 * @return {Array} The sorted array
	 * @method
	 */
	sortArray: function(list, order, sort)
	{
		var collection = new Ext.util.MixedCollection();
		var fn;
		collection.addAll(list);

		if (Ext.isFunction(sort)) {
			// Use given sort function
			fn = sort;
		} else if (Ext.isString(sort)) {
			// Sort on object attribute value, by default this is a
			// numeric sort. We need to wrap the function to
			// access the attribute value.
			fn = function(obj1, obj2) {
				return Zarafa.core.Util.numericComparison(obj1[sort], obj2[sort]);
			};
		} else {
			// Sort the object, by default this is a numeric sort
			fn = this.numericComparison;
		}

		collection.sort(order, fn);
		return collection.getRange();
	},

	/**
	 * Comparison function for comparing two numbers.
	 * This function can be used for sorting in for example the functions
	 * {@link Ext.util.MixedCollection#sort} and {@link #sortArray}.
	 *
	 * @param {Number} number1 The first number to compare
	 * @param {Number} number2 The second number to compare
	 * @return {Number} A positive value when number1 is greater then number2.
	 * A negative value when number2 is greater then number1. 0 when both objects are equal.
	 */
	numericComparison: function(number1, number2)
	{
		if(!Ext.isDefined(number2) || !Ext.isDefined(number1)) {
			return -1;
		}
		return number1 - number2;
	},

	/**
	 * Comparison function for comparing two strings using case sensitive comparison.
	 * This function can be used for sorting in for example the functions
	 * {@link Ext.util.MixedCollection#sort} and {@link #sortArray}.
	 *
	 * @param {String} string1 The first object to compare
	 * @param {String} string2 The second object to compare
	 * @return {Number} A positive value when string1 is greater then string2.
	 * A negative value when string2 is greater then string1. 0 when both objects are equal.
	 */
	caseSensitiveComparison: function(string1, string2)
	{
		return string1 > string2 ? 1 : (string1 < string2 ? -1 : 0);
	},

	/**
	 * Comparison function for comparing two strings using case insensitive comparison.
	 * This function can be used for sorting in for example the functions
	 * {@link Ext.util.MixedCollection#sort} and {@link #sortArray}.
	 *
	 * @param {String} string1 The first object to compare
	 * @param {String} string2 The second object to compare
	 * @return {Number} A positive value when string1 is greater then string2.
	 * A negative value when string2 is greater then string1. 0 when both objects are equal.
	 */
	caseInsensitiveComparison: function(string1, string2)
	{
		var v1 = String(string1).toUpperCase(), v2 = String(string2).toUpperCase();
		return Zarafa.core.Util.caseSensitiveComparison(v1, v2);
	},

	/**
	 * Remove all duplicate entries from an array of objects
	 *
	 * @param {Array} list The array of objects which must be filtered
	 * @param {String} attr (optional) The fieldname on which objects must be compared to detect duplicates
	 * If not provided the comparison is done on the object value.
	 * @return {Array} The array with only unique elements
	 * @method
	 */
	uniqueArray: function(list, attr)
	{
		var collection = new Ext.util.MixedCollection();

		Ext.each(list, function(item) {
			var value = attr ? item[attr] : item;

			if (!collection.containsKey(value)) {
				collection.add(value, item);
			}
		}, this);

		return collection.getRange();
	},

	/**
	 * This is a utility function to trim strings in a single or multi dimensional array
	 * this function should only be used with array which has only string values, otherwise
	 * this function will give unpredicated results without any error
	 * @param {Array} arrayToTrim array whose values should be trimmed
	 * @return {Array} trimmed array
	 */
	trimStringArray: function(arrayToTrim)
	{
		var tmpArray = arrayToTrim;
		arrayToTrim = [];				// reset array

		for(var index = 0, len = tmpArray.length; index < len; index++) {
			if(Array.isArray(tmpArray[index])) {
				// recursively call the same function
				arrayToTrim.push(Zarafa.core.Util.trimStringArray(tmpArray[index]));
			} else {
				if(Ext.isString(tmpArray[index]) && !Ext.isEmpty(tmpArray[index].trim())) {
					arrayToTrim.push(tmpArray[index].trim());
				}
			}
		}

		return arrayToTrim;
	},

	/**
	 * This is a utility function to check if any multi dimensional array contains any token
	 * @param {Array} multiDimArray single or multi-dimensional array
	 * @param {Mixed} tokenToSearch token to search in array
	 * @param {Boolean} caseInSensitive case sensitive match
	 * @param {Boolean} matchPartial comparison will also check for partial match
	 * @return {Boolean} true if token is found else false
	 */
	inArray: function(multiDimArray, tokenToSearch, caseInSensitive, matchPartial)
	{
		for(var index = 0, len = multiDimArray.length; index < len; index++) {
			if(Array.isArray(multiDimArray[index])) {
				// recursively call the same function
				if(Zarafa.core.Util.inArray(multiDimArray[index], tokenToSearch, caseInSensitive, matchPartial)) {
					return true;
				}
			} else {
				if(tokenToSearch) {
					if(matchPartial) {
						if(caseInSensitive) {
							if(multiDimArray[index].indexOf(tokenToSearch.toLowerCase()) != -1 || tokenToSearch.indexOf(multiDimArray[index].toLowerCase()) != -1) {
								return true;
							}
						} else {
							if(multiDimArray[index].indexOf(tokenToSearch) != -1) {
								return true;
							}
						}
					} else {
						if(caseInSensitive) {
							if(multiDimArray[index].toLowerCase() === tokenToSearch.toLowerCase()) {
								return true;
							}
						} else {
							if(multiDimArray[index] === tokenToSearch) {
								return true;
							}
						}
					}
				}
			}
		}

		return false;
	},

	/**
	 * This is a utility function to copy all the properties of config to obj recursively.
	 * Ext.apply() does the same thing but this extension preserves child object's previous values
	 * instead of overwriting it with new values.
	 * @param {Object} obj The receiver of the properties
	 * @param {Object} config The source of the properties
	 * @param {Object} defaults A different object that will also be applied for default values
	 * @return {Object} returns obj
	 */
	applyRecursive: function(obj, config, defaults)
	{
		if(defaults) {
			obj = Zarafa.core.Util.applyRecursive(obj, defaults);
		}

		if(obj && config && (Ext.isObject(config) || Array.isArray(config))) {
			for(var key in config) {
				if (Ext.isDefined(obj[key]) && Ext.isObject(obj[key])) {
					// object with child elements, so call this function recursively
					obj[key] = Zarafa.core.Util.applyRecursive(obj[key], config[key]);
				} else if (Ext.isObject(config[key])) {
					obj[key] = Zarafa.core.Util.applyRecursive({}, config[key]);
				} else {
					// normal copy
					obj[key] = config[key];
				}
			}
		}

		return obj;
	},

	/**
	 * This is a utility function to copy all the properties of config to obj recursively,
	 * if they don't already exist. Ext.appplyIf() does the same thing but this extension
	 * preserves child object's previous values instead of overwriting it with new values.
	 * @param {Object} obj The receiver of the properties
	 * @param {Object} config The source of the properties
	 * @param {Object} defaults A different object that will also be applied for default values
	 * @return {Object} returns obj
	 */
	applyIfRecursive: function(obj, config, defaults)
	{
		if(defaults) {
			obj = Zarafa.core.Util.applyIfRecursive(obj, defaults);
		}

		if(obj && config && (Ext.isObject(config) || Array.isArray(config))) {
			for(var key in config) {
				if(Ext.isDefined(obj[key]) && Ext.isObject(obj[key])) {
					// object with child elements, so call this function recursively
					obj[key] = Zarafa.core.Util.applyIfRecursive(obj[key], config[key]);
				} else if (Ext.isObject(config[key])) {
					obj[key] = Zarafa.core.Util.applyIfRecursive({}, config[key]);
				} else if(!Ext.isDefined(obj[key])) {
					// normal copy
					obj[key] = config[key];
				}
			}
		}

		return obj;
	},

	/**
	 * Recursively flattens a JSON object hierarchy into a flat list of key/value pairs.
	 *
	 * For example: The object:
	 *	{
	 *  	'zarafa': {
	 *  		'v1': {
	 *  			'main': {
	 *  				'settingA': 'value1',
	 *					'settingB': 'value2'
	 *  			}
	 *  		}
	 *  	}
	 *	}
	 *
	 * will be flattened to:
	 *	{
	 * 		'zarafa/v1/main/settingA': 'value1',
	 * 		'zarafa/v1/main/settingB': 'value2'
	 *	}
	 *
	 * @param {Object} obj The object to flatten.
	 * @param {String} sep The separator which must be applied between each path-key (e.g: '/')
	 * @param {String} path The basePath for the keys inside the object.
	 * @return {Object} The flattened object
	 */
	flattenObject: function(obj, sep, path)
	{
		var ret = {};

		var separator = '';
		if (Ext.isEmpty(path)) {
			path = '';
		} else {
			separator = sep;
		}

		if (Ext.isObject(obj)) {
			for (var key in obj) {
				Ext.apply(ret, Zarafa.core.Util.flattenObject(obj[key], sep, path + separator + key));
			}
		} else {
			ret[path] = obj;
		}

		return ret;
	},

	/**
	 * Function will return object which has all keys in lowercase
	 *
	 * @param {Object} obj The object.
	 * @return {Object} The object with all keys as lowercase
	 */
	objectKeysToLowerCase: function(object)
	{
		var key, keys = Object.keys(object);
		var newObject={};
		for (var i=0; i<keys.length; i++) {
			key = keys[i];
			newObject[key.toLowerCase()] = object[key];
		}
		return newObject;
	},

	/**
	 * Split a string in pieces based on whether each piece matches the passed
	 * pattern. It returns both the pieces that match and that do not match the
	 * pattern.
	 * @param {String} str The input string to be split up
	 * @param {RegExp} pattern The regex pattern used to be split the string
	 * @return {Array} The array of pieces
	 * @private
	 */
	splitStringByPattern: function(str, pattern)
	{
		var cutOffPoints = [0];
		var found;
		// Find the cutOffPoints in the str
		while((found = pattern.exec(str)) !== null){
			if(found.index!==0){
				cutOffPoints.push(found.index);
			}
			if(pattern.lastIndex < str.length){
				cutOffPoints.push(pattern.lastIndex);
			}
		}
		// Cut the string up into the pieces based on the cutOffPoints
		var parts = [];
		if(cutOffPoints.length > 1){
			for(var i=0;i<cutOffPoints.length;i++){
				// Use the current and the next cutOffPoint to calculate the number of character we need to extract.
				if(Ext.isDefined(cutOffPoints[i+1])){
					parts.push(str.slice(cutOffPoints[i], cutOffPoints[i+1]));
				} else {
					parts.push(str.slice(cutOffPoints[i]));
				}
			}
		} else {
			parts = [str];
		}
		return parts;
	},

	/**
	 * Convenience method to check if a given point (x,y) is inside a box (x,y,width,height)
	 * @param {Object} box a (x, y, with, height) tuple
	 * @param {Number} x point x component
	 * @param {Number} y point y component
	 * @return {Boolean} True if the given point is inside the box.
	 */
	inside: function(box, x, y)
	{
		return (x >= box.x && x < (box.x + box.width) && y >= box.y && y < (box.y + box.height));
	},

	/**
	 * Restrict a box containing 'x', 'y', 'width' and 'height' properties,
	 * to fall completely inside the given container box (which has the same properties).
	 * This ensures that the position of the box left-top corner will always be inside
	 * the container, and will attempt to move the x and y coordinates in such a way
	 * that the full width and height will fit inside the container box.
	 * @param {Object} container The container box
	 * @param {Object} box The box
	 * @return {Object} The updated box position
	 */
	restrictBox: function(container, box)
	{
		// Ensure we copy the box
		box = Ext.apply({}, box);

		// For all our calculations, we at least
		// want the top-left position to be inside
		// the container.
		box.x = Math.max(container.x, box.x);
		box.y = Math.max(container.y, box.y);

		// We can only correct the x-coordinate
		// if it doesn't equal are most-left position
		if (box.x > container.x) {
			var overflowX = Math.max(0, (box.x + box.width) - (container.x + container.width));
			box.x -= overflowX;
		}

		// We can only correct the x-coordinate
		// if it doesn't equal are most-upper position
		if (box.y > container.y) {
			var overflowY = Math.max(0, (box.y + box.height) - (container.y + container.height));
			box.y -= overflowY;
		}

		// Technically we could have moved the boxed
		// beyond our minimum top-left position. Fix
		// that here, and just accept that we will
		// overflow...
		box.x = Math.max(container.x, box.x);
		box.y = Math.max(container.y, box.y);

		return box;
	},

	/**
	 * Function will get the index of the start and end position of the current selection.
	 * @param {Ext.Element/HTMLElement} obj Reference to the textfield
	 * @return {Object} An object containing a 'start' and 'end' field which indicate
	 * the current position of the start and end of the selection. The fields will
	 * be -1 if there is no selection.
	 */
	getSelectionRange: function(obj)
	{
		obj = obj.dom || obj;

		if (obj.selectionStart || (obj.selectionStart == "0")) {
			return { start: obj.selectionStart, end: obj.selectionEnd };
		}
	},

	/**
	 * Function which will set the current caret position in the given textfield.
	 * @param {Ext.Element/HTMLElement} obj Reference to the textfield or the id of the textfield
	 * @param {Number} position The desired position for the caret
	 */
	setCaretPosition: function(obj, position)
	{
		Zarafa.core.Util.setSelectionRange(obj, position, position);
	},

	/**
	 * Function makes a selection in the textfield
	 * @param {Ext.Element/HTMLElement} obj Reference to the textfield or the id of the textfield
	 * @param {Number} selectionStart Index of the starting position of the selection
	 * @param {Number} selectionEnd Index of the ending position of the selection
	 */
	setSelectionRange: function(obj, selectionStart, selectionEnd)
	{
		obj = obj.dom || obj;

		if (obj && typeof obj == "object" && obj.setSelectionRange) {
			obj.focus();
			obj.setSelectionRange(selectionStart, selectionEnd);
		}
	},

	/**
	 * Checks whether a string is a valid email address using a regular expression. This check is
	 * not performed to the extend as the RFC 5322 specification describes the format.
	 * @param {String} str String to be validated
	 * @return {Boolean} Returns true when the email address is valid
	 */
	validateEmailAddress: function(str)
	{
		//TODO make a better check
		var filter = new RegExp(/^([^<]*<){0,1}(([a-z0-9\.\!\#\$\%\&\'\*\+\-\/\=\?\^\_\`\{\|\}\~])+\@(([a-z0-9\-])+\.)+([a-z0-9]{2,5})+)>{0,1}$|^\[[^\]]+\]$/i);
		if(Ext.isString(str) && str.length > 0 ){
			return filter.test(str);
		} else {
			return false;
		}
	},

	/**
	 * Merge 2 objects containing event handlers into a single object,
	 * while preserving scopes. This can be used when a {@link Ext.Component}
	 * receives a {@link Ext.Component#listeners} object while it also needs
	 * to add listeners in the same way (while it cannot use {@link Ext.Component#on}).
	 * By default the source listeners will {@link Function#createInterceptor intercept}
	 * the functions from the target.
	 *
	 * @param {Object} target The object with event handlers into which the new
	 * handlers will be merged.
	 * @param {Object} sourcec The object with event handlers which will be merged
	 * into the target
	 * @param {Boolean} intercept (optional) False to use {@link Function#createSequence}
	 * rather then {@link Function#createInterceptor}.
	 * @return {Object} The merged object
	 */
	mergeListeners: function(target, source, intercept)
	{
		// Make sure we have a target
		target = Ext.value(target, {});

		// Take the scope from our source, otherwise we default to the target
		var scope = source['scope'] || target['scope'];

		// Go over all listeners
		for (var key in source) {
			if (key === 'scope') {
				continue;
			}

			// Always create a delegate, the default scope inside the
			// target might might be equal to the scope of the source.
			var handler = source[key].createDelegate(scope);

			// Add the event handler
			if (Ext.isDefined(target[key])) {
				if (intercept !== false) {
					target[key] = target[key].createInterceptor(handler);
				} else {
					target[key] = target[key].createSequence(handler);
				}
			} else {
				target[key] = handler;
			}
		}

		return target;
	},

	/**
	 * Encode string in utf-16 hex format
	 * @param {String} str String to be converted into utf-16 encoded hex string
	 * @return {String} The utf-16 encoded string in hex format
	 */
	encode_utf16: function(str)
	{
		var num1, num2;
		var result = '';

		if(!Ext.isString(str)) {
		  str = String(str);
		}

		for (var i = 0, len = str.length; i < len; i++) {
		  num2 = (str.charCodeAt(i) >> 8).toString(16);
		  num1 = (str.charCodeAt(i) & 0xff).toString(16);

		  result += String.leftPad(String(num1), 2, '0');
		  result += String.leftPad(String(num2), 2, '0');
		}

		return result;
	},

	/**
	 * Function converts string in to hexadecimal string
	 * @param {String} str ASCII string to be converted into hexadecimal representation.
	 * @return {String} The hexadecimal representation as a string.
	 */
	stringToHex: function(string)
	{
		string = string.toUpperCase();
		var hexString = '';
		for(var i=0; i < string.length; i++) {
			hexString += '' + string.charCodeAt(i).toString(16);
		}

		return hexString;
	},

	/**
	 * Function converts hexadecimal string in to ASCII string
	 * @param {String} hexString The hexadecimal string
	 * @return {String} converted ASCII string
	 */
	hexToString: function(hexString)
	{
		hexString = hexString.toString();
		var string = '';
		for (var i = 0; i < hexString.length; i += 2) {
			string += String.fromCharCode(parseInt(hexString.substr(i, 2), 16));
		}
		return string.toLowerCase();
	},

	/**
	 * Function used to reload the webapp.
	 */
	reloadWebapp: function()
	{
		if ( container.fireEvent('beforewebappreload') !== false ){
			this.disableLeaveRequester();
			window.location.reload();
		}
	},

	/**
	 * Function is use to register onbeforeunload event to show confirm dialog
	 * when user trying to leave the page.
	 */
	enableLeaveRequester: function()
	{
		window.onbeforeunload = this.onBeforeUnload.createDelegate(this);
	},

	/**
	 * Function is use to deregistering onbeforeunload event.
	 */
	disableLeaveRequester: function()
	{
		window.onbeforeunload = null;
	},

	/**
	 * Function which is show confirm dialog when user trying to leave the page.
	 * It will also check the value of {#Zarafa.core.Util.skipRequester skipRequester}
	 * If it's true then it will not show the confirm dialog.
	 */
	onBeforeUnload: function()
	{
		if(!this.skipRequester) {
			return _('Your changes will be lost if you leave this page now.');
		} else {
			this.skipRequester = false;
			return;
		}
	},

	/**
	 * Helper function which used to show the {@link Ext.MessageBox}
	 * and {@link window.Zarafa.core.Util#disableLeaveRequester disable Leave Requester}
	 *
	 * @param {Object} options The options object which used to construct the {@link Ext.MessageBox}.
	 * @param {boolean} custom True to show {@link Zarafa.common.dialogs.MessageBox.addCustomButtons messageBox}.
	 */
	showMessageBox: function(options, custom)
	{
		Zarafa.core.Util.disableLeaveRequester();
		if (custom){
			Zarafa.common.dialogs.MessageBox.addCustomButtons(options);
		} else {
			Ext.MessageBox.show(options);
		}
	},

	/**
	 * Helper function which add the given text into
	 * browser clipboard data.
	 *
	 * @param {String} text The text which is going to
	 * add in clipboard data.
	 */
	copyToClipboard: function (text)
	{
		var doc = document;
		if (Zarafa.core.BrowserWindowMgr.isMainWindowActive() === false) {
			doc = Zarafa.core.BrowserWindowMgr.getActive().document;
		}

		var textArea = doc.createElement("textarea");
		textArea.setAttribute("id", "copyTextArea");
		doc.body.appendChild(textArea);
		textArea.innerText = text;
		textArea.select();
		doc.execCommand("copy");
		doc.body.removeChild(textArea);
	},

	/**
	 * Function which will convert given input into Bytes.
	 * @param {Number} input the number which needs to be converted in bytes.
	 * @param {String} sizeUnit string which indicates given input is in which unit.
	 * And value must be 'KB' or 'MB'.
	 * @return {Number} input in Bytes.
	 */
	convertToBytes: function (input, sizeUnit)
	{
		sizeUnit = sizeUnit.toUpperCase();
		if (sizeUnit === 'MB') {
			input = input * Math.pow(1024,2);
		} else if (sizeUnit === 'KB') {
			input = input * 1024;
		}
		return input;
	},

	/**
	 * Function which will convert given input into
	 * KB or MB size unit.
	 * @param {Number} input the number in bytes which needs to be converted.
	 * @param {String} sizeUnit string which should indicate
	 * in which unit input is needed to be converted.
	 * And value must be 'KB' or 'MB'.
	 * @return {Number} input in KB or MB.
	 */
	convertBytesToKBorMB: function (input, sizeUnit)
	{
		sizeUnit = sizeUnit.toUpperCase();
		if (sizeUnit === 'MB') {
			input = input / Math.pow(1024,2);
		} else if (sizeUnit === 'KB') {
			input = input / 1024;
		}
		return input;
	},

	/**
	 * Function gets the formatted date according to the given/selected language, as formats differ
	 * according to languages.
	 * @param {String} value The input date value
	 * @param {String} language The current language
	 * @param {String} format The current format of the date
	 * @return {Array} newDate The date array which is to be set
	 */
	getDateByLanguageFormat: function(value, language, format)
	{
		if (!Ext.isDefined(language)) {
			language = container.getSettingsModel().get('zarafa/v1/main/language');
		}
		// We extract the language code from the language since the language retrieved
		// from the settings is in the format 'en_GB.UTF-8'
		language = language.split('.')[0];

		var rawValue = value;
		var formattedDate = [];
		var newDate = new Date();
		var regExSpecialChar = /[^0-9a-zA-Z]/g;

		switch (language) {
			case 'ja_JP':
			case 'hu_HU':
				// For format 'Y/m/d'
				if (regExSpecialChar.test(rawValue)) {
					rawValue = rawValue.split(regExSpecialChar);
					// Since the format is 'Y/m/d', the first part might be the year
					// In that case, we change the value to maintain the 'd/m/Y' format.
					if (rawValue[0].length === 4) {
						[rawValue[0], rawValue[2]] = [rawValue[2], rawValue[0]];
					}
					rawValue = rawValue.join('/');
				}
				formattedDate = this.getFormattedDate(rawValue, '/');
				break;
			case 'de_DE':
			case 'nb_NO':
			case 'fi_FI':
			case 'cs_CZ':
				// For format 'd.m.Y' and 'd. m. Y.'
				formattedDate = this.getFormattedDate(rawValue, '.');
				break;
			case 'ca_ES':
			case 'es_ES':
			case 'pl_PL':
				// For format 'd/m/aa' and 'd/m/r'
				// We remove alphabets part if any, to maintain the format.
				rawValue = rawValue.replace(/[a-zA-Z]/g, "");
				formattedDate = this.getFormattedDate(rawValue, '/');
				break;
			case 'nl_NL':
				// For format 'd-m-Y'
				formattedDate = this.getFormattedDate(rawValue, '-');
				break;
			case 'en_US':
				// For format 'm/d/Y', since this is a different format, we need to specifically format
				// the date as the 'getFormattedDate' function supports formats similar to 'd/m/Y'.
				var result = [];
				// If the selected language is English(US) but grommunio Web is using the local
				// format 'd/m/Y', then no need to format according to 'm/d/Y' format
				if (format === 'd/m/Y') {
					formattedDate = this.getFormattedDate(rawValue, '/');
					break;
				}
				rawValue = rawValue.replace(/[a-zA-Z\s]/g, "");
				// If the date value has a separator and is not a single digit or two digit day
				if (regExSpecialChar.test(rawValue) && rawValue.length > 2) {
					rawValue = rawValue.split(regExSpecialChar);
					// Since the format is 'm/d/Y', we interchange the day with the month.
					[result[0], result[1]] = [rawValue[1], rawValue[0]];
					if (rawValue[0] > 12) {
						result = rawValue.slice();
					}
					var year = rawValue[2];
					// Formatting the year if incomplete or incorrect
					if (year && year.length !== 4) {
						rawValue[2] = ('20' + year).length === 4 ? '20' + year : newDate.getFullYear().toString();
					}
					result[2] = rawValue[2];
				} else {
					// If there is no separator, we format the date and then
					// swap the month with the day if needed (according to 'm/d/Y').
					result = this.getFormattedDate(rawValue, '/');
					if (result[0] <= 12 && result[1] <= 12 && rawValue.length > 2) {
						[result[0], result[1]] = [result[1], result[0]];
					}
				}
				formattedDate = result.slice();
				break;
			case 'en_GB':
			case 'ru_RU':
			case 'it_IT':
			case 'pt_BR':
			case 'tr_TR':
			case 'sl_SI':
			case 'zh_TW':
			case 'fr_FR':
			case 'da_DR':
			default:
				// For format 'd/m/Y'
				formattedDate = this.getFormattedDate(rawValue, '/');
				break;
		}
		return formattedDate;
	},

	/**
	 * Function corrects the date string if it has no separator or it is incomplete/incorrect.
	 * @param {String} rawValue The input date value
	 * @param {String} separator The separator of the date according to the format
	 * @return {Array} dateArray The date array which is to be set
	 */
	getFormattedDate: function(rawValue, separator)
	{
		var dateArray, hasNoSeparator = false;
		var currentFullDate = new Date();
		var currentYear = currentFullDate.getFullYear();
		var currentMonth = currentFullDate.getMonth() + 1;
		var currentDate = currentFullDate.getDate();

		var regExSeparator = /[^0-9a-zA-Z]/g;
		if (!regExSeparator.test(rawValue)) {
			hasNoSeparator = true;
		}

		if (hasNoSeparator) {
			rawValue = rawValue.replace(/[a-zA-Z]/g, "");
			switch (rawValue.length) {
				// Example: '06062020' -> '06/06/2020'
				case 8:
					rawValue = rawValue.slice(0, 2) + separator + rawValue.slice(2, 4) + separator + rawValue.slice(4);
					break;
				// Example: '662020' -> '06/06/2020'
				case 6:
					rawValue = rawValue.slice(0, 1) + separator + rawValue.slice(1, 2) + separator + rawValue.slice(2);
					break;
				// Example: '06' -> '06/02/2020'
				case 2:
					rawValue = rawValue + separator + currentMonth + separator + currentYear;
					break;
				// Example: '6' -> '06/02/2020'
				case 1:
					rawValue = '0' + rawValue + separator + currentMonth + separator + currentYear;
					break;
				default:
					rawValue = currentDate + separator + currentMonth + separator + currentYear;
					break;
			}
			dateArray = rawValue.split(separator);
		} else {
			rawValue = rawValue.replace(/[a-zA-Z\s]/g, "");
			dateArray = rawValue.split(regExSeparator);

			// Check for the year value
			if (dateArray.length > 1 && rawValue.length > 0) {
				var year = dateArray[2];
				if (year && year.length !== 4) {
					dateArray[2] = ('20' + year).length === 4 ? '20' + year : currentYear;
				}
			} else {
				dateArray = [currentDate, currentMonth, currentYear];
			}
		}
		// If value of month part is greater than 12, swap with day.
		if (dateArray[1] > 12) {
			[dateArray[0], dateArray[1]] = [dateArray[1], dateArray[0]];
		}
		return dateArray;
	},

	/**
	 * Function converts data into hexadecimal representation.
	 * @param {String} str string to be converted into hexadecimal representation.
	 * @return {String} An ASCII string containing the hexadecimal representation of string.
	 */
	bin2hex : function (str)
	{
		return str.split('').reduce((str, aChar) =>
			str += `0${aChar.charCodeAt().toString(16)}`.slice(-2), '');
	},

	/**
	 * Function decodes a hexadecimally encoded binary string.
	 * @param {String} str Hexadecimal representation of data.
	 * @return {String} Returns the binary representation of the given data.
	 */
	hex2bin : function (str)
	{
		return str.match(/.{1,2}/g).reduce((str, hex) =>
			str += String.fromCharCode(parseInt(hex, 16)), '');
	}
};
Ext.namespace('Zarafa.core.data');

/**
 * @class Zarafa.core.data.AbstractResponseHandler
 * @extends Object
 *
 * The main interface used by the {@link Zarafa.core.ResponseRouter ResponseRouter}
 * for handling responses received by the PHP-side.
 *
 * This class is used as 'scoped' object, containing all information for handling
 * a specific server-response or server-notification.
 *
 * The {@link #start} function will be called directly before the
 * {@link Zarafa.core.ResponseRouter ResponseRouter} starts processing
 * the Response from the server which is intended for this handler.
 *
 * After this function the {@link #handle} function will be invoked
 * for each object inside the responds. Subclasses should not overwrite
 * this function directly, instead they can implement the specific handles
 * they need. Each {@link Zarafa.core.Actions} action will invoke the handler
 * with the same name but with 'do' prefixed. For example, the action
 * {@link Zarafa.core.Actions#item} will cause the function 'doItem' to be
 * invoked. Likewise, the action {@link Zarafa.core.Actions#open} will cause
 * the function 'doOpen' to be invoked.
 *
 * Finally the {@link #done} function is called to indicate the entire
 * response has been processed. The handler can then complete the
 * transaction by updating the {@link Ext.data.Store}.
 */
Zarafa.core.data.AbstractResponseHandler = Ext.extend(Object, {
	/**
	 * @constructor
	 * @param {Object} config Configuration object
	 */
	constructor: function(config)
	{
		Ext.apply(this, config);
	},

	/**
	 * The handler which is invoked when no valid response was returned
	 * for the Request. This could be a HTTP-404 error, or the PHP-side returned
	 * an invalid object which could not be parsed by a {@link Ext.data.DataReader DataReader}.
	 * @param {Object} responseObject The raw browser response object (e.g.: XMLHttpRequest)
	 * @param {Object} args (optional) A Javascript error object if the response could not
	 * have been parsed by a {@link Ext.data.DataReader DataReader}.
	 */
	responseFailure: Ext.emptyFn,

	/**
	 * The main handler to begin a Response processing transaction.
	 * @param {String} moduleName The name of the PHP module from which this response originated.
	 * @param {String} moduleId The unique identifier for the PHP response.
	 * @param {Object} data The entire response object which will be processed during this transaction.
	 * @param {Number} timestamp The {@link Date#getTime timestamp} on which the response was received
	 * @return {Boolean} False when the given data object cannot be handled by this response handler,
	 * and the transaction must be canceled.
	 */
	start: Ext.emptyFn,

	/**
	 * The handler for handling the given command from a Response.
	 * @param {Zarafa.core.Actions} action The action which must be executed.
	 * @param {Object} data The response object belonging to the given command.
	 * @return {Boolean} False when action could not be handled successfully. This will
	 * not cancel the transaction itself, but rather causes the 'success' argument for the
	 * {@link #done} function to be false.
	 */
	handle: function(action, data)
	{
		var handler = this['do' + Ext.util.Format.capitalize(action)];
		if (Ext.isFunction(handler)) {
			return handler.call(this, data);
		}
	},

	/**
	 * The main handler to complete a Response processing transaction.
	 * @param {Boolean} success True if no errors were returned from the PHP-side.
	 */
	done: Ext.emptyFn
});
Ext.namespace('Zarafa.core.data');

/**
 * @class Zarafa.core.data.CallbackQueue
 * @extends Ext.util.Observable
 *
 * A Special queue containing callback functions which can be used
 * to serialize a series of actions/validations where it is possible
 * that each action/validation might use a {@link Ext.MessageBox MessageBox}
 * to request the user input. While the {@link Ext.MessageBox MessageBox}
 * is opened, the queue will be paused until the user has closed the message.
 */
Zarafa.core.data.CallbackQueue = Ext.extend(Ext.util.Observable, {
	/**
	 * The queue which contains all the tasks which should be run
	 * @property
	 * @type Array
	 * @private
	 */
	queue: undefined,

	/**
	 * Indicates that {@link #run} has been called, and the various callbacks
	 * in the {@link #queue} are being executed.
	 * @property
	 * @type Boolean
	 * @private
	 */
	running: false,

	/**
	 * Internal counter to keep track at which task is currently being executed,
	 * this will be reset when the {@link #run queue starts} and will be updated
	 * after the {@link #onCompleteTask completion of each task}.
	 * @property
	 * @type Number
	 * @private
	 */
	currentTask: 0,

	/**
	 * The function which is provided to {@link #run} which should be called as
	 * soon as the last task has been called. It will be called with the scope
	 * {@link #completionScope}.
	 * @property
	 * @type Function
	 * @private
	 */
	completionFn: undefined,

	/**
	 * The scope for the {@link #completionFn} which is provided to {@link #run}.
	 * @property
	 * @type Object
	 * @private
	 */
	completionScope: undefined,

	/**
	 * @constructor
	 * @param {Object} config Configuration object
	 */
	constructor: function(config)
	{
		config = config || {};

		// Initialize the queue
		config.queue = [];

		this.addEvents(
			/**
			 * @event startqueue
			 * Event which is fired when the queue is about to start
			 * @param {Zarafa.core.data.CallbackQueue} queue The queue which is started
			 */
			'startqueue',
			/**
			 * @event completequeue
			 * Event which is fired when the last callback from the queue has been completed.
			 * @param {Zarafa.core.data.CallbackQueue} queue The queue which has completed
			 * @param {Boolean} success True if all callbacks were executed successfully
			 */
			'completequeue',
			/**
			 * @event starttask
			 * Event which is fired when a task has started
			 * @param {Zarafa.core.data.CallbackQueue} queue The queue to which the task belongs
			 * @param {Function} fn The task function which has been started
			 * @param {Object} scope The scope of the function which has started
			 */
			'starttask',
			/**
			 * @event completetask
			 * Event which is fired when a task has completed
			 * @param {Function} fn The task function which has been completed
			 * @param {Object} scope The scope of the function which has completed
			 * @param {Boolean} success True if the callback was executed successfully
			 */
			'completetask'
		);

		Ext.apply(this, config);

		Zarafa.core.data.CallbackQueue.superclass.constructor.call(this, config);
	},

	/**
	 * Add a callback function to the end of the {@link #queue}. When {@link #run} is called,
	 * this function will be executed with the provided scope.
	 * @param {Function} fn The function which will be called
	 * @param {Object} scope The scope in which the function will be called
	 */
	add: function(fn, scope)
	{
		this.queue.push({ fn: fn, scope: scope });
	},

	/**
	 * Remove a callback function which was previously registered using {@link #add}.
	 * This will search for the task in the {@link #queue} which matches the given
	 * function and scope exactly, and removed it from the {@link #queue}.
	 * @param {Function} fn The function to remove
	 * @param {Object} scope The scope of the function
	 */
	remove: function(fn, scope)
	{
		var queue = this.queue;

		for (var i = 0; i < queue.length; i++) {
			var task = queue[i];

			// Check if this is the same function and scope,
			// if so remove it from the queue.
			if (task.fn === fn && task.scope === scope) {
				this.queue.splice(i, 1);
				return;
			}
		}
	},

	/**
	 * Run all Callback functions in the {@link #queue}. This will fire the {@link #start} event,
	 * and starts all tasks {@link #currentTask starting with} 0.
	 */
	run: function(fn, scope)
	{
		this.running = true;
		this.currentTask = 0;
		this.completionFn = fn;
		this.completionScope = scope;

		this.fireEvent('startqueue', this);

		this.doTask(this.currentTask);
	},

	/**
	 * @returns {Boolean} True if the queue is currently running
	 */
	isRunning: function()
	{
		return this.running;
	},

	/**
	 * Called to execute the task at the specified location in the {@link #queue}.
	 * This will execute the callback function, and pass the {@link #onCompleteTask} function
	 * as callback function.
	 * @param {Number} index The index in the queue of the callback to execute
	 * @private
	 */
	doTask: function(index)
	{
		var task = this.queue[index];
		this.fireEvent('starttask', this, task.fn, task.scope);
		task.fn.call(task.scope, this.onCompleteTask.createDelegate(this, [ task ], 1));
	},

	/**
	 * Callback function for the task which was executed using {@link #doTask}. This
	 * checks if the task was successfully completed and if so if this was the last task.
	 * If either the task has failed, or this was the last task, the queue will be stopped,
	 * and the {@link #complete} event will be fired. Otherwise {@link #doTask} will be
	 * called to execute the {@link #currentTask next task}.
	 * @param {Boolean} success True if the task completed successfully
	 * @param {Object} task The task which was completed successfully
	 * @private
	 */
	onCompleteTask: function(success, task)
	{
		// If not provided, then assume success
		success = success !== false;

		this.fireEvent('completetask', this, task.fn, task.scope, success);

		if (success && this.currentTask < (this.queue.length - 1)) {
			this.currentTask++;
			this.doTask(this.currentTask);
		} else {
			if (this.completionFn) {
				this.completionFn.call(this.completionScope, success);
				this.completionFn = undefined;
				this.completionScope = undefined;
			}
			this.fireEvent('completequeue', this, success);
			this.running = false;
		}
	}
});
Ext.namespace('Zarafa.core.data');

/**
 * @class Zarafa.core.data.ContentPanelMgr
 * @extends Ext.util.Observable
 * @singleton
 *
 * The {@link Zarafa.core.ui.ContentPanel ContentPanel} manager. Each
 * {@link Zarafa.core.ui.ContentPanel ContentPanel} which is created
 * must register itself to this manager.
 *
 * This manager can be used by Plugins to hook into
 * {@link Zarafa.core.ui.ContentPanel ContentPanel} from the moment they
 * are being displayed.
 */
Zarafa.core.data.ContentPanelMgr = Ext.extend(Ext.util.Observable, {
	/**
	 * The collection of {@link Zarafa.core.ui.ContentPanel contentPanels}
	 * which have been registered to this manager.
	 * @property
	 * @type Ext.util.MixedCollection
	 */
	contentPanels: undefined,
	/**
	 * @constructor
	 */
	constructor: function()
	{
		this.contentPanels = new Ext.util.MixedCollection();
		this.addEvents([
			/**
			 * @event createcontentpanel
			 * Fires when a {@link Zarafa.core.ui.ContentPanel contentPanel} is being created.
			 * @param {Zarafa.core.ui.ContentPanel} contentPanels The
			 * {@link Zarafa.core.ui.ContentPanel} which is being created.
			 */
			'createcontentpanel',
			/**
			 * @event destroycontentpanel
			 * Fires when a {@link Zarafa.core.ui.ContentPanel contentPanel} is being destroyed.
			 * @param {Zarafa.core.ui.ContentPanel} contentPanel The
			 * {@link Zarafa.core.ui.ContentPanel} which is being destroyed.
			 */
			'destroycontentpanel'
		]);

		Zarafa.core.data.ContentPanelMgr.superclass.constructor.call(this);
	},

	/**
	 * Register a {@link Zarafa.core.ui.ContentPanel contentPanel} with the {@link Zarafa.core.data.ContentPanelMgr ContentPanelMgr}.
	 * @param {Zarafa.core.ui.ContentPanel} contentPanel the {@link Zarafa.core.ui.ContentPanel contentPanel} which must be registered.
	 */
	register: function(contentPanel)
	{
		contentPanel.on('show', this.onContentPanelShow, this);
		contentPanel.on('close', this.onContentPanelHide, this);
		contentPanel.on('hide', this.onContentPanelHide, this);

		this.contentPanels.add(contentPanel);
	},

	/**
	 * UnRegister a {@link Zarafa.core.ui.ContentPanel contentPanel} from the {@link Zarafa.core.data.ContentPanelMgr ContentPanelMgr}.
	 * @param {Zarafa.core.ui.ContentPanel} contentPanel the {@link Zarafa.core.ui.ContentPanel contentPanel} which must be unregistered.
	 */
	unregister: function(contentPanel)
	{
		contentPanel.un('show', this.onContentPanelShow, this);
		contentPanel.un('close', this.onContentPanelHide, this);
		contentPanel.un('hide', this.onContentPanelHide, this);

		this.contentPanels.remove(contentPanel);
	},

	/**
	 * Event handler which is raised before the {@link Zarafa.core.ui.ContentPanel contentPanel} is
	 * being shown. This will raise the {@link #createcontentpanel} event to allow any
	 * listeners to hook into further events coming from the given
	 * {@link Zarafa.core.ui.ContentPanel contentPanel}.
	 *
	 * @param {Ext.Container} contentPanel The contentPanel which is being rendered
	 * @private
	 */
	onContentPanelShow: function(contentPanel)
	{
		this.fireEvent('createcontentpanel', contentPanel);
	},

	/**
	 * Event handler which is raised when the {@link Zarafa.core.ui.ContentPanel contentPanel} is
	 * being hidden. This will raise the {@link #destroycontentpanel} event to inform
	 * any listeners that their {@link Zarafa.core.ui.ContentPanel contentPanel} is going to disappear.
	 *
	 * @param {Ext.Container} contentPanel The contentPanel which is being destroyed
	 * @private
	 */
	onContentPanelHide: function(contentPanel)
	{
		this.fireEvent('destroycontentpanel', contentPanel);
	},

	/**
	 * Find instances of {@link Zarafa.core.ui.ContentPanel contentPanel} from the {@link Zarafa.core.data.ContentPanelMgr ContentPanelMgr}.
	 * @param {Ext.Component} component the class name of the contentPanel for which we should perform the search.
	 * @return {Ext.util.MixedCollection} {@link Ext.util.MixedCollection MixedCollection} of contentPanels for specified
	 * component.
	 */
	getContentPanelInstances: function(component)
	{
		return this.contentPanels.filterBy(function(contentPanel) {
			return contentPanel instanceof component;
		});
	}
});

Zarafa.core.data.ContentPanelMgr = new Zarafa.core.data.ContentPanelMgr();
Ext.namespace('Zarafa.core.data');

/**
 * @class Zarafa.core.data.IPFStoreMgr
 * @extends Ext.util.Observable
 *
 * The {@link Zarafa.core.data.IPFStore IPFStore} manager. Each
 * {@link Zarafa.core.data.IPFStore IPFStore} which is created
 * must register itself to this manager.
 *
 * {@link Zarafa.core.data.IPFStoreMgr} will then handle inter-store
 * communications. Listening to {@link Zarafa.core.data.IPFStoreMgr IPFStoreMgr}
 * events allow UI components and {@link Zarafa.core.data.IPFStore stores} to
 * detect {@link Zarafa.core.data.IPFRecord record} editing in dialogs.
 * @singleton
 */
Zarafa.core.data.IPFStoreMgr = Ext.extend(Ext.util.Observable, {
		/**
		 * The collection of {@link Zarafa.core.data.IPFStore stores}
		 * which have been registered to this manager.
		 * @property
		 * @type Ext.util.MixedCollection
		 */
		IPFStores: undefined,
		/**
		 * @constructor
		 */
		constructor: function()
		{
			this.IPFStores = new Ext.util.MixedCollection();
			this.addEvents([
				/**
				 * @event storeexception
				 * Fires when the {@link Zarafa.core.data.IPFStore IPFStore} fired an {@link Zarafa.core.data.IPFStore#exception exception}.
				 * @param {Zarafa.core.data.IPFStore} store The store on which the exception occurred
				 * @param {Ext.data.DataProxy} proxy The proxy from where the exception event originated
				 * @param {String} type The value of this parameter will be either 'response' or 'remote'.
				 * @param {String} action Name of the action.
				 * @param {Object} options The options for the action that were specified in the request.
				 * @param {Object} response If the 'type' argument was 'response' this will be the raw browser response object,
				 * otherwise it will be the decoded response object
				 * @param {Mixed} arg (optional) Additional arguments for the exception
				 */
				'storeexception',
				/**
				 * @event beforerecordsave
				 * Fires before a Message {@link Zarafa.core.data.IPFRecord record} is being saved.
				 * @param {Zarafa.core.data.IPFStore} store The {@link Zarafa.core.data.IPFStore store} in which the
				 * {@link Zarafa.core.data.IPFRecord record} is located while being saved.
				 * @param {Object} data An object containing the data that is to be saved.
				 * The object will contain a key for each appropriate action, with an array
				 * of updated data for each record.
				 */
				'beforerecordsave',
				/**
				 * @event afterrecordsave
				 * Fires when a Message {@link Zarafa.core.data.IPFRecord record} has been saved.
				 * @param {Zarafa.core.data.IPFStore} store The {@link Zarafa.core.data.IPFStore store} in which the
				 * {@link Zarafa.core.data.IPFRecord record} is located while being saved.
				 * @param {Mixed} obj The object containing {@link Zarafa.core.data.IPFRecord record} which has been saved. This
				 * {@link Zarafa.core.data.IPFRecord record} is the most recent version which came from the server.
				 */
				'afterrecordsave',
				/**
				 * @event afterrecordupdate
				 * Fires when a Message {@link Zarafa.core.data.IPFRecord record} has been updated.
				 * @param {Zarafa.core.data.IPFStore} store The {@link Zarafa.core.data.IPFStore store} to which the
				 * {@link Zarafa.core.data.IPFRecord record} belongs.
				 * @param {Zarafa.core.data.IPFrecord} record The most recent version which came from the server.
				 * @param {String} operation The update operation being performed.
				 * ({@link Ext.data.Record#EDIT}, {@link Ext.data.Record#REJECT}, {@link Ext.data.Record#COMMIT}).
				 */
				'afterrecordupdate',
				/**
				 * @event recordremove
				 * Fires when a Message {@link Zarafa.core.data.IPFRecord record} has been removed from the server
				 * @param {Zarafa.core.data.IPFStore} store The {@link Zarafa.core.data.IPFStore store} to which the
				 * {@link Zarafa.core.data.IPFRecord record} belongs.
				 * @param {Zarafa.core.data.IPFrecord} record The most recent version which came from the server.
				 */
				'recordremove',
				/**
				 * @event afterrecordwrite
				 * Fires when {@link Zarafa.core.IPFRecord IPFRecords} are modified (created, updated, destroyed, opened)
				 * on {@link Zarafa.core.data.IPFStore IPFStore}.
				 * @param {Zarafa.core.data.IPFStore} store The {@link Zarafa.core.data.IPFStore store} to which the
				 * {@link Zarafa.core.data.IPFRecord record} belongs.
				 * @param {String} action [Ext.data.Api.actions.create|update|destroy|open]
				 * @param {Object} result The 'data' picked-out out of the response for convenience.
				 * @param {Ext.Direct.Transaction} res
				 * @param {Zarafa.core.data.IPFrecord[]} records The most recent version of the records
				 * which came from the server.
				 */
				'afterrecordwrite'
			]);

			Zarafa.core.data.IPFStoreMgr.superclass.constructor.call(this);
		},

		/**
		 * Register a {@link Zarafa.core.data.IPFStore IPFStore} with the {@link Zarafa.core.data.IPFStoreMgr IPFStoreMgr}.
		 * @param {Zarafa.core.data.IPFStore} IPFStore the {@link Zarafa.core.data.IPFStore IPFStore} which must be registered.
		 * @param {Boolean} serverOnly True to register the {@link Zarafa.core.data.IPFStore IPFStore} only for
		 * events originating directly from the server.
		 */
		register: function(IPFStore, serverOnly)
		{
			if (!serverOnly) {
				IPFStore.on('beforesave', this.onBeforeSave, this);
				IPFStore.on('save', this.onSave, this);
				IPFStore.on('update', this.onUpdate, this);
				IPFStore.on('remove', this.onRemove, this);
			}
			IPFStore.on('write', this.onWrite, this);
			IPFStore.on('exception', this.onException, this);

			this.IPFStores.add(IPFStore);
		},

		/**
		 * Unregister a {@link Zarafa.core.data.IPFStore IPFStore} from the {@link Zarafa.core.data.IPFStoreMgr IPFStoreMgr}.
		 * @param {Zarafa.core.data.IPFStore} IPFStore the {@link Zarafa.core.data.IPFStore IPFStore} which must be unregistered.
		 * @param {Boolean} serverOnly True if the {@link Zarafa.core.data.IPFStore IPFStore} was {@link #register registered}
		 * with the serverOnly argument.
		 */
		unregister: function(IPFStore, serverOnly)
		{
			if (!serverOnly) {
				IPFStore.un('beforesave', this.onBeforeSave, this);
				IPFStore.un('save', this.onSave, this);
				IPFStore.un('update', this.onUpdate, this);
				IPFStore.un('remove', this.onRemove, this);
			}
			IPFStore.un('write', this.onWrite, this);
			IPFStore.un('exception', this.onException, this);

			this.IPFStores.remove(IPFStore);
		},

		/**
		 * Filter the list of {@link Zarafa.core.data.IPFRecord records} to contain
		 * only those which can be {@link Zarafa.core.data.IPFRecord#eventPropagation propagated}
		 * over a new event to other {@link Zarafa.core.data.IPFStore stores}.
		 *
		 * @param {Zarafa.core.data.IPFRecord/Array} records The record or records to filter
		 * out the non-propagatable records.
		 * @return {Zarafa.core.data.IPFRecord/Array} All propagatable records
		 * @private
		 */
		getPropagatableRecords: function(records)
		{
			var propagateRecords = [];

			if (!Array.isArray(records)) {
				records = [ records ];
			}

			for (var i = 0, len = records.length; i < len; i++) {
				var record = records[i];

				if (record instanceof Zarafa.core.data.IPFRecord && record.hasEventPropagation()) {
					propagateRecords.push(record);
				}
			}

			return propagateRecords;
		},

		/**
		 * Event handler which is triggered when a {@link Zarafa.core.data.IPFRecord record} is about to
		 * be saved in a {@link Zarafa.core.data.IPFStore store}. This function will inform the
		 * {@link Zarafa.core.data.IPFStore IPFStores} about the event through the
		 * {@link #beforerecordsave} event.
		 *
		 * @param {Zarafa.core.data.IPFStore} IPFStore
		 * @param {Object} data An object containing the data that is to be saved.
		 * The object will contain a key for each appropriate action, with an array
		 * of updated data for each record.
		 * @private
		 */
		onBeforeSave: function(IPFStore, data)
		{
			var propagateData = {};

			for (var key in data) {
				var propagateRecords = this.getPropagatableRecords(data[key]);
				if (!Ext.isEmpty(propagateData)) {
					propagateData[key] = propagateRecords;
				}
			}

			this.fireEvent('beforerecordsave', IPFStore, propagateData);
		},

		/**
		 * Event handler which is triggered when a {@link Zarafa.core.data.IPFRecord record} has been
		 * saved in a {@link Zarafa.core.data.IPFStore store}. This function will inform the
		 * {@link arafa.core.data.IPFStore IPFStores} about the event through the
		 * {@link #afterrecordsave} event.
		 *
		 * @param {Zarafa.core.data.IPFStore} IPFStore
		 * @param {Number} batch The identifier for the batch that was saved.
		 * @param {Object} data An object containing the data that is to be saved.
		 * The object will contain a key for each appropriate action, with an array
		 * of updated data for each record.
		 * @private
		 */
		onSave: function(IPFStore, batch, data)
		{
			this.fireEvent('afterrecordsave', IPFStore, data);
		},

		/**
		 * Event handler which is triggered when a {@link Zarafa.core.data.IPFRecord record} has been
		 * updated from the server. This function will inform the {@link Zarafa.core.data.IPFStore IPFStores}
		 * about the event through the {@link #afterrecordupdate} event.
		 *
		 * @param {Zarafa.core.data.IPFStore} IPFStore
		 * @param {Zarafa.core.data.IPFRecord} record The Record which has been updated
		 * @param {String} operation The update operation being performed.
		 * ({@link Ext.data.Record#EDIT}, {@link Ext.data.Record#REJECT}, {@link Ext.data.Record#COMMIT}).
		 * @private
		 */
		onUpdate: function(IPFStore, record, operation)
		{
			if (record.hasEventPropagation()) {
				this.fireEvent('afterrecordupdate', IPFStore, record, operation);
			}
		},

		/**
		 * Event handler which is triggered when a {@link Zarafa.core.data.IPFRecord record} has been
		 * removed from the server. This function will inform the {@link Zarafa.core.data.IPFStore IPFStores}
		 * about the event through the {@link #recordremove} event.
		 *
		 * @param {Zarafa.core.data.IPFStore} IPFStore
		 * @param {Zarafa.core.data.IPFRecord} record The Record which has been updated
		 * @param {String} The index at which the record was removed
		 * @private
		 */
		onRemove: function(IPFStore, record, index)
		{
			if (record.hasEventPropagation()) {
				this.fireEvent('recordremove', IPFStore, record);
			}
		},

		/**
		 * Event handler will be called on successful completion of any CRUD operation,
		 * the difference between this event and {@link #save} event is that {@link #save}
		 * event will pass only data set that is modified not the record that is modified.
		 * so this event removes burden of finding record from the record set.
		 *
		 * @param {Zarafa.core.data.IPFStore} store The {@link Zarafa.core.data.IPFStore store} to which the
		 * {@link Zarafa.core.data.IPFRecord record} belongs.
		 * @param {String} action [Ext.data.Api.actions.create|update|destroy|open]
		 * @param {Object} data The 'data' picked-out out of the response for convenience.
		 * @param {Ext.Direct.Transaction} res
		 * @param {Zarafa.core.data.IPFrecord[]} records The most recent version of the records
		 * which came from the server.
		 * @private
		 */
		onWrite: function(IPFStore, action, data, transaction, records)
		{
			var propagateRecords = this.getPropagatableRecords(records);
			if (!Ext.isEmpty(propagateRecords)) {
				this.fireEvent('afterrecordwrite', IPFStore, action, data, transaction, propagateRecords);
			}
		},

		/**
		 * Event handler which will be called when the store has fired the {@link Ext.data.Store#exception} event.
		 * This will look up which store has exactly fired the event, and will fire the {@link #exception} event.
		 * @param {Ext.data.DataProxy} proxy The proxy from where the exception event originated
		 * @param {String} type The value of this parameter will be either 'response' or 'remote'.
		 * @param {String} action Name of the action.
		 * @param {Object} options The options for the action that were specified in the request.
		 * @param {Object} response If the 'type' argument was 'response' this will be the raw browser response object,
		 * otherwise it will be the decoded response object
		 * @param {Mixed} arg (optional) Additional arguments for the exception
		 * @private
		 */
		onException: function(proxy, type, action, options, response, arg)
		{
			var proxies = Ext.pluck(this.IPFStores.items, 'proxy');
			if (!Ext.isEmpty(proxies)) {
				var storeIndex = proxies.indexOf(proxy);
				this.fireEvent('storeexception', this.IPFStores.get(storeIndex), proxy, type, action, options, response, arg);
			}
		},

		/**
		 * Executes the specified function once for every {@link Zarafa.core.data.IPFStore store} in the collection,
		 * passing the following arguments:
		 * <div class="mdetail-params"><ul>
		 * <li><b>item</b>: Zarafa.core.data.IPFStore<p class="sub-desc">The collection item</p></li>
		 * <li><b>index</b>: Number<p class="sub-desc">The item's index</p></li>
		 * <li><b>length</b>: Number<p class="sub-desc">The total number of items in the collection</p></li>
		 * </ul></div>
		 * The function should return a boolean value. Returning false from the function will stop the iteration.
		 * @param {Function} fn The function to execute for each item.
		 * @param {Object} scope (optional) The scope (<code>this</code> reference) in which the function is executed.
		 * Defaults to the current item in the iteration.
		 */
		each: function(fn, scope)
		{
			this.IPFStores.each.apply(this, arguments);
		},

		/**
		 * Obtain a list of {@link Zarafa.core.data.IPFStore stores} which have one or more of the
		 * requested {@link Zarafa.core.data.IPFRecord folders} currently
		 * {@link Zarafa.core.data.IPFStore#containsStoreInLastLoad loaded}.
		 * @param {Array} folders The list of {@link Zarafa.core.data.IPFRecord folders} or
		 * {@link Zarafa.hierarchy.data.MAPIFolderRecord#getId folder entryIds}
		 * for which the {@link Zarafa.core.data.IPFStore stores} are requested.
		 * @return {Array} The array of {@link Zarafa.core.data.IPFStore stores} which have the
		 * one or more of the requested stores loaded.
		 */
		getStoresForStores: function(folders)
		{
			var stores = [];

			if (!Array.isArray(folders)) {
				folders = [ folders ];
			}

			if (folders[0] instanceof Zarafa.core.data.IPFRecord) {
				folders = Ext.pluck(folders, 'id');
			}

			this.IPFStores.each(function(store) {
				if (store.containsStoreInLastLoad(folders)) {
					stores.push(store);
				}
			}, this);

			return stores;
		}
});

Zarafa.core.data.IPFStoreMgr = new Zarafa.core.data.IPFStoreMgr();
Ext.namespace('Zarafa.core.data');

/**
 * @class Zarafa.core.data.IPMRecipientResolveRecord
 * Contains a description of what the data that will be used to resolve an user looks like. It is 
 * used by the reader in the Zarafa.core.data.CheckNamesResponseHandler.
 */
Zarafa.core.data.IPMRecipientResolveRecordFields = [
	{name: 'user_name'},
	{name: 'display_name'},
	{name: 'address_type'},
	{name: 'smtp_address'},
	{name: 'email_address'},
	{name: 'entryid'},
	{name: 'search_key'},
	{name: 'object_type', type: 'int'},
	{name: 'display_type', type: 'int'},
	{name: 'display_type_ex', type: 'int'}

];
Zarafa.core.data.IPMRecipientResolveRecord = Ext.data.Record.create(Zarafa.core.data.IPMRecipientResolveRecordFields);
Ext.namespace('Zarafa.core.data');

/**
 * @class Zarafa.core.data.IPMStoreMgr
 * @extends Ext.util.Observable
 *
 * The {@link Zarafa.core.data.IPMStore IPMStore} manager. Each
 * {@link Zarafa.core.data.IPMStore IPMStore} which is created
 * must register itself to this manager.
 *
 * {@link Zarafa.core.data.IPMStoreMgr} will then handle inter-store
 * communications. Listening to {@link Zarafa.core.data.IPMStoreMgr IPMStoreMgr}
 * events allow UI components and {@link Zarafa.core.data.IPMStore stores} to
 * detect {@link Zarafa.core.data.IPMRecord record} editing in dialogs.
 * @singleton
 */
Zarafa.core.data.IPMStoreMgr = Ext.extend(Ext.util.Observable, {
		/**
		 * The collection of {@link Zarafa.core.data.IPMStore stores}
		 * which have been registered to this manager.
		 * @property
		 * @type Ext.util.MixedCollection
		 */
		IPMStores: undefined,
		/**
		 * @constructor
		 */
		constructor: function()
		{
			this.IPMStores = new Ext.util.MixedCollection();
			this.addEvents([
				/**
				 * @event storeexception
				 * Fires when the {@link Zarafa.core.data.IPMStore IPMStore} fired an {@link Zarafa.core.data.IPMStore#exception exception}.
				 * @param {Zarafa.core.data.IPMStore} store The store on which the exception occurred
				 * @param {Ext.data.DataProxy} proxy The proxy from where the exception event originated
				 * @param {String} type The value of this parameter will be either 'response' or 'remote'.
				 * @param {String} action Name of the action.
				 * @param {Object} options The options for the action that were specified in the request.
				 * @param {Object} response If the 'type' argument was 'response' this will be the raw browser response object,
				 * otherwise it will be the decoded response object
				 * @param {Mixed} arg (optional) Additional arguments for the exception
				 */
				'storeexception',
				/**
				 * @event beforerecordsave
				 * Fires before a Message {@link Zarafa.core.data.IPMRecord record} is being saved.
				 * @param {Zarafa.core.data.IPMStore} store The {@link Zarafa.core.data.IPMStore store} in which the
				 * {@link Zarafa.core.data.IPMRecord record} is located while being saved.
				 * @param {Object} data An object containing the data that is to be saved.
				 * The object will contain a key for each appropriate action, with an array
				 * of updated data for each record.
				 */
				'beforerecordsave',
				/**
				 * @event afterrecordsave
				 * Fires when a Message {@link Zarafa.core.data.IPMRecord record} has been saved.
				 * @param {Zarafa.core.data.IPMStore} store The {@link Zarafa.core.data.IPMStore store} in which the
				 * {@link Zarafa.core.data.IPMRecord record} is located while being saved.
				 * @param {Mixed} obj The object containing {@link Zarafa.core.data.IPMRecord record} which has been saved. This
				 * {@link Zarafa.core.data.IPMRecord record} is the most recent version which came from the server.
				 */
				'afterrecordsave',
				/**
				 * @event afterrecordupdate
				 * Fires when a Message {@link Zarafa.core.data.IPMRecord record} has been updated.
				 * @param {Zarafa.core.data.IPMStore} store The {@link Zarafa.core.data.IPMStore store} to which the
				 * {@link Zarafa.core.data.IPMRecord record} belongs.
				 * @param {Zarafa.core.data.IPMrecord} record The most recent version which came from the server.
				 * @param {String} operation The update operation being performed.
				 * ({@link Ext.data.Record#EDIT}, {@link Ext.data.Record#REJECT}, {@link Ext.data.Record#COMMIT}).
				 */
				'afterrecordupdate',
				/**
				 * @event recordremove
				 * Fires when a Message {@link Zarafa.core.data.IPMRecord record} has been removed from the server
				 * @param {Zarafa.core.data.IPMStore} store The {@link Zarafa.core.data.IPMStore store} to which the
				 * {@link Zarafa.core.data.IPMRecord record} belongs.
				 * @param {Zarafa.core.data.IPMrecord} record The most recent version which came from the server.
				 */
				'recordremove',
				/**
				 * @event afterrecordwrite
				 * Fires when {@link Zarafa.core.IPMRecord IPMRecords} are modified (created, updated, destroyed, opened)
				 * on {@link Zarafa.core.data.IPMStore IPMStore}.
				 * @param {Zarafa.core.data.IPMStore} store The {@link Zarafa.core.data.IPMStore store} to which the
				 * {@link Zarafa.core.data.IPMRecord record} belongs.
				 * @param {String} action [Ext.data.Api.actions.create|update|destroy|open]
				 * @param {Object} result The 'data' picked-out out of the response for convenience.
				 * @param {Ext.Direct.Transaction} res
				 * @param {Zarafa.core.data.IPMrecord[]} records The most recent version of the records
				 * which came from the server.
				 */
				'afterrecordwrite'
			]);

			Zarafa.core.data.IPMStoreMgr.superclass.constructor.call(this);
		},

		/**
		 * Register a {@link Zarafa.core.data.IPMStore IPMStore} with the {@link Zarafa.core.data.IPMStoreMgr IPMStoreMgr}.
		 * @param {Zarafa.core.data.IPMStore} IPMStore the {@link Zarafa.core.data.IPMStore IPMStore} which must be registered.
		 * @param {Boolean} serverOnly True to register the {@link Zarafa.core.data.IPMStore IPMStore} only for
		 * events originating directly from the server.
		 */
		register: function(IPMStore, serverOnly)
		{
			if (!serverOnly) {
				IPMStore.on('beforesave', this.onBeforeSave, this);
				IPMStore.on('save', this.onSave, this);
				IPMStore.on('update', this.onUpdate, this);
				IPMStore.on('remove', this.onRemove, this);
			}
			IPMStore.on('write', this.onWrite, this);
			IPMStore.on('exception', this.onException, this);

			this.IPMStores.add(IPMStore);
		},

		/**
		 * Unregister a {@link Zarafa.core.data.IPMStore IPMStore} from the {@link Zarafa.core.data.IPMStoreMgr IPMStoreMgr}.
		 * @param {Zarafa.core.data.IPMStore} IPMStore the {@link Zarafa.core.data.IPMStore IPMStore} which must be unregistered.
		 * @param {Boolean} serverOnly True if the {@link Zarafa.core.data.IPMStore IPMStore} was {@link #register registered}
		 * with the serverOnly argument.
		 */
		unregister: function(IPMStore, serverOnly)
		{
			if (!serverOnly) {
				IPMStore.un('beforesave', this.onBeforeSave, this);
				IPMStore.un('save', this.onSave, this);
				IPMStore.un('update', this.onUpdate, this);
				IPMStore.un('remove', this.onRemove, this);
			}
			IPMStore.un('write', this.onWrite, this);
			IPMStore.un('exception', this.onException, this);

			this.IPMStores.remove(IPMStore);
		},

		/**
		 * Filter the list of {@link Zarafa.core.data.IPMRecord records} to contain
		 * only those which can be {@link Zarafa.core.data.IPMRecord#eventPropagation propagated}
		 * over a new event to other {@link Zarafa.core.data.IPMStore stores}.
		 *
		 * @param {Zarafa.core.data.IPMRecord/Array} records The record or records to filter
		 * out the non-propagatable records.
		 * @return {Zarafa.core.data.IPMRecord/Array} All propagatable records
		 * @private
		 */
		getPropagatableRecords: function(records)
		{
			var propagateRecords = [];

			if (!Array.isArray(records)) {
				records = [ records ];
			}

			for (var i = 0, len = records.length; i < len; i++) {
				var record = records[i];

				if (record instanceof Zarafa.core.data.IPMRecord && record.hasEventPropagation()) {
					propagateRecords.push(record);
				}
			}

			return propagateRecords;
		},

		/**
		 * Event handler which is triggered when a {@link Zarafa.core.data.IPMRecord record} is about to
		 * be saved in a {@link Zarafa.core.data.IPMStore store}. This function will inform the
		 * {@link Zarafa.core.data.IPMStore IPMStores} about the event through the
		 * {@link #beforerecordsave} event.
		 *
		 * @param {Zarafa.core.data.IPMStore} IPMStore
		 * @param {Object} data An object containing the data that is to be saved.
		 * The object will contain a key for each appropriate action, with an array
		 * of updated data for each record.
		 * @private
		 */
		onBeforeSave: function(IPMStore, data)
		{
			var propagateData = {};

			for (var key in data) {
				var propagateRecords = this.getPropagatableRecords(data[key]);
				if (!Ext.isEmpty(propagateData)) {
					propagateData[key] = propagateRecords;
				}
			}

			this.fireEvent('beforerecordsave', IPMStore, propagateData);
		},

		/**
		 * Event handler which is triggered when a {@link Zarafa.core.data.IPMRecord record} has been
		 * saved in a {@link Zarafa.core.data.IPMStore store}. This function will inform the
		 * {@link arafa.core.data.IPMStore IPMStores} about the event through the
		 * {@link #afterrecordsave} event.
		 *
		 * @param {Zarafa.core.data.IPMStore} IPMStore
		 * @param {Number} batch The identifier for the batch that was saved.
		 * @param {Object} data An object containing the data that is to be saved.
		 * The object will contain a key for each appropriate action, with an array
		 * of updated data for each record.
		 * @private
		 */
		onSave: function(IPMStore, batch, data)
		{
			this.fireEvent('afterrecordsave', IPMStore, data);
		},

		/**
		 * Event handler which is triggered when a {@link Zarafa.core.data.IPMRecord record} has been
		 * updated from the server. This function will inform the {@link Zarafa.core.data.IPMStore IPMStores}
		 * about the event through the {@link #afterrecordupdate} event.
		 *
		 * @param {Zarafa.core.data.IPMStore} IPMStore
		 * @param {Zarafa.core.data.IPMRecord} record The Record which has been updated
		 * @param {String} operation The update operation being performed.
		 * ({@link Ext.data.Record#EDIT}, {@link Ext.data.Record#REJECT}, {@link Ext.data.Record#COMMIT}).
		 * @private
		 */
		onUpdate: function(IPMStore, record, operation)
		{
			if (record.hasEventPropagation()) {
				this.fireEvent('afterrecordupdate', IPMStore, record, operation);
			}
		},

		/**
		 * Event handler which is triggered when a {@link Zarafa.core.data.IPMRecord record} has been
		 * removed from the server. This function will inform the {@link Zarafa.core.data.IPMStore IPMStores}
		 * about the event through the {@link #recordremove} event.
		 *
		 * @param {Zarafa.core.data.IPMStore} IPMStore
		 * @param {Zarafa.core.data.IPMRecord} record The Record which has been updated
		 * @param {String} The index at which the record was removed
		 * @private
		 */
		onRemove: function(IPMStore, record, index)
		{
			if (record.hasEventPropagation()) {
				this.fireEvent('recordremove', IPMStore, record);
			}
		},

		/**
		 * Event handler will be called on successful completion of any CRUD operation,
		 * the difference between this event and {@link #save} event is that {@link #save}
		 * event will pass only data set that is modified not the record that is modified.
		 * so this event removes burden of finding record from the record set.
		 *
		 * @param {Zarafa.core.data.IPMStore} store The {@link Zarafa.core.data.IPMStore store} to which the
		 * {@link Zarafa.core.data.IPMRecord record} belongs.
		 * @param {String} action [Ext.data.Api.actions.create|update|destroy|open]
		 * @param {Object} data The 'data' picked-out out of the response for convenience.
		 * @param {Ext.Direct.Transaction} res
		 * @param {Zarafa.core.data.IPMrecord[]} records The most recent version of the records
		 * which came from the server.
		 * @private
		 */
		onWrite: function(IPMStore, action, data, transaction, records)
		{
			var propagateRecords = this.getPropagatableRecords(records);
			if (!Ext.isEmpty(propagateRecords)) {
				this.fireEvent('afterrecordwrite', IPMStore, action, data, transaction, propagateRecords);
			}
		},

		/**
		 * Event handler which will be called when the store has fired the {@link Ext.data.Store#exception} event.
		 * This will look up which store has exactly fired the event, and will fire the {@link #exception} event.
		 * @param {Ext.data.DataProxy} proxy The proxy from where the exception event originated
		 * @param {String} type The value of this parameter will be either 'response' or 'remote'.
		 * @param {String} action Name of the action.
		 * @param {Object} options The options for the action that were specified in the request.
		 * @param {Object} response If the 'type' argument was 'response' this will be the raw browser response object,
		 * otherwise it will be the decoded response object
		 * @param {Mixed} arg (optional) Additional arguments for the exception
		 * @private
		 */
		onException: function(proxy, type, action, options, response, arg)
		{
			var proxies = Ext.pluck(this.IPMStores.items, 'proxy');
			if (!Ext.isEmpty(proxies)) {
				var storeIndex = proxies.indexOf(proxy);
				this.fireEvent('storeexception', this.IPMStores.get(storeIndex), proxy, type, action, options, response, arg);
			}
		},

		/**
		 * Executes the specified function once for every {@link Zarafa.core.data.IPMStore store} in the collection,
		 * passing the following arguments:
		 * <div class="mdetail-params"><ul>
		 * <li><b>item</b>: Zarafa.core.data.IPMStore<p class="sub-desc">The collection item</p></li>
		 * <li><b>index</b>: Number<p class="sub-desc">The item's index</p></li>
		 * <li><b>length</b>: Number<p class="sub-desc">The total number of items in the collection</p></li>
		 * </ul></div>
		 * The function should return a boolean value. Returning false from the function will stop the iteration.
		 * @param {Function} fn The function to execute for each item.
		 * @param {Object} scope (optional) The scope (<code>this</code> reference) in which the function is executed.
		 * Defaults to the current item in the iteration.
		 */
		each: function(fn, scope)
		{
			this.IPMStores.each.apply(this, arguments);
		},

		/**
		 * Obtain a list of {@link Zarafa.core.data.IPMStore stores} which have one or more of the
		 * requested {@link Zarafa.core.date.IPFRecord folders} currently
		 * {@link Zarafa.core.data.IPMStore#containsFolderInLastLoad loaded}.
		 * @param {Array} folders The list of {@link Zarafa.core.data.IPFRecord folders} or
		 * {@link Zarafa.hierarchy.data.MAPIFolderRecord#getId folder entryIds}
		 * for which the {@link Zarafa.core.data.IPMStore stores} are requested.
		 * @return {Array} The array of {@link Zarafa.core.data.IPMStore stores} which have the
		 * one or more of the requested folders loaded.
		 */
		getStoresForFolders: function(folders)
		{
			var stores = [];

			if (!Array.isArray(folders)) {
				folders = [ folders ];
			}

			if (folders[0] instanceof Zarafa.core.data.IPFRecord) {
				folders = Ext.pluck(folders, 'id');
			}

			this.IPMStores.each(function(store) {
				if (store.containsFolderInLastLoad(folders)) {
					stores.push(store);
				}
			}, this);

			return stores;
		}
});

Zarafa.core.data.IPMStoreMgr = new Zarafa.core.data.IPMStoreMgr();
Ext.namespace('Zarafa.core.data');

/**
 * @class Zarafa.core.data.JsonReader
 * @extends Ext.data.JsonReader
 *
 * This extension of the {@link Ext.data.JsonReader} supports {@link Zarafa.core.data.IPMStore stores}
 * which can hold different type of {@link Zarafa.core.data.IPMRecord records}.
 *
 * If in the constructor no recordType is provided, dynamic {@link Zarafa.core.data.IPMRecord record} type
 * support is assumed. With dynamic types, the incoming response data is used to determine which
 * {@link Zarafa.core.data.MAPIRecord MAPIRecord} must be constructed for each individual root element.
 */
Zarafa.core.data.JsonReader = Ext.extend(Ext.data.JsonReader, {
	/**
	 * In {@link #getEfMapping} we generate a mapping of all Record Fields per ObjectType,
	 * all mappings are stored in a special cache to prevent them from being regenerated
	 * each time.
	 * @property
	 * @type Object
	 * @private
	 */
	efCache: undefined,

	/**
	 * @cfg {Boolean} dynamicRecord Enable dynamic detection of the records
	 * which are read by this JsonReader. When enabled this will prefer using
	 * the {@link Zarafa.core.data.RecordFactory} to detect the recordType rather
	 * then using the {@link #recordType} directly. (defaults to true)
	 */
	dynamicRecord: true,

	/**
	 * @constructor
	 * @param {Object} meta Metadata configuration options.
	 * @param {Object} recordType (optional) Optional Record type matches the type
	 * which must be read from response. If no type is given, {@link Zarafa.core.data.JsonReader}
	 * will dynamically detect the record type based on the response.
	 */
	constructor: function(meta, recordType)
	{
		meta = Ext.applyIf(meta || {}, {
			totalProperty: 'count',
			root: 'item',
			id: 'entryid',
			idProperty: 'entryid'
		});

		// Check if the meta object contained the successProperty.
		// Note that this must be called before the superclass constructor,
		// because the meta object will be altered there.
		var hasSuccessProperty = Ext.isDefined(meta.successProperty);

		// If no recordType is provided, enable the dynamic record handling
		// of the JsonReader.
		if (!Ext.isDefined(recordType)) {
			// FIXME: We shouldn't for IPM as base recordclass here, instead
			// this should be handled as configuration option, or something even smarter.
			recordType = Zarafa.core.data.RecordFactory.getRecordClassByMessageClass('IPM');
		}

		// Check we dynamic records are disabled.
		if (Ext.isDefined(meta.dynamicRecord)) {
			this.dynamicRecord = meta.dynamicRecord;
		}

		if (this.dynamicRecord !== false) {
			this.efCache = {};
		}

		Zarafa.core.data.JsonReader.superclass.constructor.call(this, meta, recordType);

		// This fixes a bug in the Ext.data.JsonReader. Even when no successProperty
		// is given, the getSuccess function will still be implemented after which
		// the function will often fail due to the lack of the success property within
		// the response data.
		if (!hasSuccessProperty) {
			this.getSuccess = function(o) { return true; };
		}
	},

	/**
	 * Build the extractors which are used when reading the Json data. This initialized
	 * functions like {@link #getTotal}, {@link #getSuccess}, {@link #getId}.
	 * @private
	 */
	buildExtractors: function()
	{
		var s = this.meta;

		Zarafa.core.data.JsonReader.superclass.buildExtractors.call(this);

		// Wrap the original getId function to check if the data is the raw
		// data which has wrapped the 'props' field, or if this is the unwrapped
		// data.
		if (s.id || s.idProperty) {
			var old = this.getId;
			this.getId = function(rec) {
				if (rec.props) {
					var id = old(rec.props);
					if (!Ext.isEmpty(id)) {
						return id;
					}
				}

				return old(rec);
			};
		}
	},

	/**
	 * Obtain the mapping between response objects and {@link Zarafa.core.data.IPMRecord record} fields for
	 * the given recordType. If no mapping yet exist for this recordType one will be constructed
	 * and added to the {@link Zarafa.core.data.IPMRecord.efCache cache}.
	 *
	 * @param {String} key The unique key for this record type (used for caching purposes).
	 * @param {Array} items The array of {@link Zarafa.core.data.IPMRecord record} items.
	 * @param {Number} len The length of the items array.
	 * @return {Array} The name/value list of response to {@link Zarafa.core.data.IPMRecord record} fields.
	 * @private
	 */
	getEfMapping: function(key, items, len)
	{
		if (Ext.isString(key)) {
			key = key.toUpperCase();
		}

		var ef = this.efCache[key];

		if (!Ext.isDefined(ef))
		{
			ef = [];
    	for(var i = 0; i < len; i++){
				var f = items[i];
				var map = (!Ext.isEmpty(f.mapping)) ? f.mapping : f.name;
				ef.push(this.createAccessor.call(this, map));
			}
			this.efCache[key] = ef;
		}

		return ef;
	},

	/**
	 * Type-casts a single row of raw-data from server
	 * @param {Object} data The data object which must be deserialized.
	 * @param {Array} items The {@link Ext.data.Field Field} used for deserialization.
	 * @param {Integer} len The length of the items array.
	 * @private
	 */
	extractValues: function(data, items, len)
	{
		var values = {};

		// If the data object is wrapped (it contains objects like 'props', 'attachments',
		// 'recipients', etc... Then we must call extractValues for each individual subobject.
		if (Ext.isDefined(data.props)) {
			values = Ext.apply({}, data);
			values.props = this.extractValues(data.props, items, len);
			return values;
		}

		if (this.dynamicRecord === true)
		{
			var recordType = Zarafa.core.data.RecordFactory.getRecordClassByRecordData(data);
			if (!Ext.isDefined(recordType)) {
				recordType = this.recordType;
			}

			items = recordType.prototype.fields.items;
			len = recordType.prototype.fields.length;

			// Normally the caller has initialized this.ef for us, but only at this time
			// do we know the real recordType. As such we have to override the previous
			// this.ef mapping.
			this.ef = this.getEfMapping(data.message_class || data.object_type, items, len);
		}

		// Extract the values per object which we want to deserialize.
		for (var j = 0; j < len; j++) {
			var f = items[j];
			var v = this.ef[j](data);
			if (Ext.isDefined(v)) {
				values[f.name] = f.convert(v, data);
			}
		}

		return values;
	},

	/**
	 * Returns extracted, type-cast rows of data. Iterates to call #extractValues for each row
	 *
	 * This function is exactly copied from {@link Ext.data.DataReader.extractData} with the only
	 * difference is using the RecordFactory for record allocation.
	 *
	 * @param {Object|Array} data-root from server response
	 * @param {Boolean} returnRecords [false] Set true to return instances of {@link Zarafa.core.data.MAPIRecord MAPIRecord}.
	 * @private
	 */
	extractData: function(root, returnRecords)
	{
		// A bit ugly this, too bad the Record's raw data couldn't be saved in a common property named "raw" or something.
		var rawName = (this instanceof Ext.data.JsonReader) ? 'json' : 'node';

		var rs = [];

		// Had to add Check for XmlReader, #isData returns true if root is an Xml-object. Want to check in order to re-factor
		// #extractData into DataReader base, since the implementations are almost identical for JsonReader, XmlReader
		if (this.isData(root) && !(this instanceof Ext.data.XmlReader)) {
			root = [root];
		}
		if (returnRecords === true) {
			for (var i = 0; i < root.length; i++) {
				var n = root[i];

				var record = undefined;
				var id = this.getId(n);
				var data = n.props || n;

				// Clear all data from the object which must be deserialized,
				// we only want the 'object_type' and 'message_class' properties.
				data = { message_class: data.message_class, object_type: data.object_type };

				if (this.dynamicRecord === true) {
					record = Zarafa.core.data.RecordFactory.createRecordObjectByRecordData(data, id);
				}

				if (!record) {
					record = new this.recordType({}, id);
				}

				// move primitive properties and identification properties at same level
				this.moveIdProperties(n, record.baseIdProperties);

				var f		= record.fields,
					fi		= f.items,
					fl		= f.length;
				this.update(record, this.extractValues(n, fi, fl));
				record[rawName] = n;  // <-- There's implementation of ugly bit, setting the raw record-data.
				rs.push(record);
			}
		} else {
			for (var i = 0; i < root.length; i++) {
				var n = root[i];

				var Record = undefined;
				if (this.dynamicRecord === true) {
					Record = Zarafa.core.data.RecordFactory.getRecordClassByRecordData(n.props || n);
				}

				// Fall back to specified record type if we can't get the type from the data
				if (!Record) {
					Record = this.recordType;
				}

				// move primitive properties and identification properties at same level
				this.moveIdProperties(n, Record.prototype.baseIdProperties);

				var f		= Record.prototype.fields,
					fi		= f.items,
					fl		= f.length;

				// here we can't do anything about complex structures so its ignored here
				var data = this.extractValues(n, fi, fl);
				data[this.meta.idProperty] = this.getId(n.props || n);
				rs.push(data);
			}
		}
		return rs;
	},

	/**
	 * Function will merge all identification properties and primitive properties
	 * to the props field and return the merged data. so {@link Zarafa.core.JsonReader JsonReader}
	 * can read the data and extract the properties.
	 * @param {Object} data data that is passed to {@link #extractData} to extract properties.
	 * @param {String|Array} idProperties The id properties which should be moved into the properties object.
	 * @return {Object} The updated data object.
	 */
	moveIdProperties: function(data, idProperties)
	{
		// If there is not data then return no data
		if (!data) {
			return data;
		}

		if (!data.props) {
			data.props = {};
		}

		// move the base identification property to <props> tag level
		var idProperty = this.meta.idProperty;
		if (idProperty) {
			var value = data[idProperty];
			if (Ext.isDefined(value)) {
				data.props[idProperty] = value;
				delete data[idProperty];
			}
		}

		// move all extra identification properties to <props> tag level
		if (Ext.isString(idProperties)) {
			var value = data[idProperties];
			if (Ext.isDefined(value)) {
				data.props[idProperties] = value;
				delete data[idProperties];
			}
		} else if (idProperties) {
			for (var i = 0, len = idProperties.length; i < len; i++) {
				var idProp = idProperties[i];
				var value = data[idProp];

				if(Ext.isDefined(value)) {
					data.props[idProp] = value;
					delete data[idProp];
				}
			}
		}

		return data;
	},

	/**
	 * Used for un-phantoming a record after a successful database insert.
	 * Sets the records pk along with new data from server.
	 * You must return at least the database pk using the idProperty defined in
	 * your DataReader configuration. The incoming data from server will be merged
	 * with the data in the local record. In addition, you must return record-data
	 * from the server in the same order received. Will perform a commit as well,
	 * un-marking dirty-fields. Store's "update" event will be suppressed.
	 *
	 * @param {Record/Record[]} record The phantom record to be realized.
	 * @param {Object/Object[]} data The new record data to apply. Must include the primary-key from database defined in idProperty field.
	 * @private
	 */
	realize: function(record, data)
	{
		// This function is copy & pasted from Ext.js Ext.data.JsonReader#realize.
		// Our only difference is the assignment of the record.data field.
		if (Array.isArray(record)) {
			for (var i = record.length - 1; i >= 0; i--) {
				// recurse
				if (Array.isArray(data)) {
					this.realize(record.splice(i,1).shift(), data.splice(i,1).shift());
				} else {
					// weird...record is an array but data isn't?? recurse but just send in the whole invalid data object.
					// the else clause below will detect !this.isData and throw exception.
					this.realize(record.splice(i,1).shift(), data);
				}
			}
		} else {
			// If records is NOT an array but data IS, see if data contains just 1 record. If so extract it and carry on.
			if (Array.isArray(data)) {
				data = data.shift();
			}
			if (!this.isData(data)) {
				// TODO: Let exception-handler choose to commit or not rather than blindly records.commit() here.
				// record.commit();
				throw new Ext.data.DataReader.Error('realize', record);
			}
			record.phantom = false; // <-- That's what it's all about
			record._phid = record.id; // <-- copy phantom-id -> _phid, so we can remap in Store#onCreateRecords
			record.id = this.getId(data);

			// And now the infamous line for which we copied this entire function.
			// Extjs expects that we transfer _all_ properties back from the server to the client after
			// a new item was created. Since this has a negative impact on the performance.
			//
			// This has been solved to let the server-side only return the properties which have
			// changed, or are new (like the entryid). We can then simply use Ext.apply to apply
			// the updated data over the already available data.
			//
			// But for those who have paid attention in the data flow of Extjs, know that this
			// sounds quite a lot like the function description of Ext.data.JsonReader#update.
			//
			// So to make everything even simpler, we don'e update the record.data object here,
			// but instead we simply continue to the Ext.data.JsonReader#update function to
			// handle the rest of the work.
			//
			//record.data = data;

			// Since we postpone the record.data update, there is no need to commit,
			// this too is done in update().
			//record.commit();

			// Time for the real work...
			this.update(record, data);

			// During realize the record might have received a new
			// id value. We have to reMap the store to update the keys.
			if (record.store) {
				record.store.reMap(record);
			}
		}
	},

	/**
	 * Used for updating a non-phantom or "real" record's data with fresh data from
	 * server after remote-save. If returning data from multiple-records after a batch-update,
	 * you must return record-data from the server in the same order received.
	 * Will perform a commit as well, un-marking dirty-fields. Store's "update" event
	 * will be suppressed as the record receives fresh new data-hash
	 *
	 * @param {Record/Record[]} record
	 * @param {Object/Object[]} data
	 * @private
	 */
	update: function(record, data)
	{
		// Recursively call into update to update each record individually.
		if (Array.isArray(record)) {
			for (var i = 0; i < record.length; i++) {
				if(Array.isArray(data)) {
					this.update(record[i], data[i]);
				} else {
					this.update(record[i], data);
				}
			}
			return;
		}

		// It can happen that the data is wrapped in a array of length 1.
		if (Array.isArray(data)) {
			data = data.shift();
		}

		if (this.isData(data)) {
			// move primitive properties and identification properties at same level
			data = this.moveIdProperties(data, record.baseIdProperties);

			// All preprocessing has been done. All remaining data
			// can be applied into the IPMRecord directly.
			record.data = Ext.apply(record.data, data.props || data);

			// scalar values from props are applied so remove it from data object
			delete data.props;

			// Put the action response from the server in the record
			if(data.action_response){
				record.action_response = data.action_response;
				delete data.action_response;
			}

			// If the record contains substores to store complex data then we have to first
			// serialize those data into its consecutive stores and then we can continue
			// with normal processing
			Ext.iterate(data, function(key, value) {
				if (Array.isArray(value) || Ext.isObject(value)) {
					var store;

					if (record.supportsSubStore(key)) {
						store = record.getSubStore(key);
						if (!store) {
							store = record.createSubStore(key);
						} else {
							store.removeAll(true);
						}

						// Load data into the SubStore, and remove
						// it from the data object.
						if (!Ext.isEmpty(value)) {
							store.loadData(value);
							delete data[key];
						}
					}
				}
			}, this);

			// Discard any additional data which was set on the data object,
			// this data has probably been set by plugins, but they have sufficient
			// alternatives to fit their custom data into the IPMRecord structure.
		}

		record.commit();
	}
});
Ext.namespace('Zarafa.core.data');

/**
 * @class Zarafa.core.data.JsonWriter
 * @extends Ext.data.JsonWriter
 */
Zarafa.core.data.JsonWriter = Ext.extend(Ext.data.JsonWriter, {
	/**
	 * @constructor
	 * @param {Object} config Configuration object
	 */
	constructor: function(config)
	{
		config = config || {};

		Ext.applyIf(config, {
			writeAllFields: false,
			// FIXME: Disable automatic encoding for now,
			// the MAPIProxy needs an individual encoded string
			// for each record in the request. We might want to
			// fix this in the future though.
			encode: false
		});

		Zarafa.core.data.JsonWriter.superclass.constructor.call(this, config);
	},

	/**
	 * Render the data in the data object which will be {@link Ext.encode encoded}
	 * and send over the protocol to the server after this function call. During
	 * rendering all {@link Date date} objects will be converted to UNIX timestamps.
	 * This will prevent ExtJs/JSON specific encoding functions to convert the
	 * date object into a "YYYY-MM-DDTHH:MM:SS" timestring.
	 * @param {Object/Array} data The object which musted be rendered
	 * @private
	 */
	renderData: function(data)
	{
		if (Array.isArray(data)) {
			for (var i = 0, len = data.length; i < len; i++) {
				this.renderData(data[i]);
			}
			return;
		}

		Ext.iterate(data, function(key, value) {
			if (Ext.isDate(value)) {
				data[key] = Math.floor(value.getTime() / 1000);
			}
			if (Ext.isObject(value)) {
				this.renderData(value);
			}
		}, this);
	},

	/**
	 * Final action of a write event. Apply the written data-object to params.
	 * This function is extended from {@link Ext.data.JsonWriter Extjs}, to use
	 * {@link #renderData} to add some extra data conversions before encoding
	 * the data by {@link Ext.encode Ext.encode}.
	 * @param {Object} http params-object to write-to.
	 * @param {Object} baseParams as defined by {@link Ext.data.Store#baseParams}.
	 * The baseParms must be encoded by the extending class, eg: {@link Ext.data.JsonWriter}, {@link Ext.data.XmlWriter}.
	 * @param {Object/Object[]} data Data-object representing compiled Store-recordset.
	 */
	render: function(params, baseParams, data)
	{
		// Apply the parameters into the data object, this allows
		// optional data to be send to the server.
		Ext.apply(data, baseParams, params);

		// Apply special rendering to convert all objects
		this.renderData(data);
		Zarafa.core.data.JsonWriter.superclass.render.call(this, params, baseParams, data);
	},

	/**
	 * Adds special function for serialization needed when opening
	 * a record. We can use the default {@link Zarafa.core.data.JsonWriter.toIdHash toIdHash}
	 * function.
	 *
	 * @param {Ext.data.Record} record
	 * @return {Object}
	 * @private
	 */
	openRecord: function(record)
	{
		return this.toIdHash(record);
	},

	/**
	 * Rather then using the regular {@link Ext.data.JsonWriter#toHash toHash}
	 * function, this will use the specialized {@link Zarafa.core.data.JsonWriter#toPropHash toPropHash}
	 * function.
	 *
	 * @param {Ext.data.Record} record
	 * @return {Object}
	 * @override
	 * @private
	 */
	createRecord: function(record)
	{
		return this.toPropHash(record);
	},

	/**
	 * Rather then using the regular {@link Ext.data.JsonWriter#toHash toHash}
	 * function, this will use the specialized {@link Zarafa.core.data.JsonWriter#toPropHash toPropHash}
	 * function.
	 *
	 * @param {Ext.data.Record} record
	 * @return {Object}
	 * @override
	 * @private
	 */
	updateRecord: function(record)
	{
		return this.toPropHash(record);
	},

	/**
	 * Use the {@link Zarafa.core.data.JsonWriter#toIdHash toIdHash} function for creating the hash.
	 *
	 * @param {Ext.data.Record} record
	 * @return {Object}
	 * @override
	 * @private
	 */
	destroyRecord: function(record)
	{
		return this.toIdHash(record);
	},

	/**
	 * Similar to {@link Ext.data.JsonWriter#toHash}
	 *
	 * This will limit the serialization to only the ID properties and message
	 * action commands for the given {@link Zarafa.core.data.IPMRecord record}.
	 *
	 * @param {Ext.data.Record} record The record to hash
	 * @param {Boolean} allowEmpty True to allow empty ID elements to be send
	 * @return {Object} The hashed object
	 * @private
	 */
	toIdHash: function(record, allowEmpty)
	{
		var hash = {};

		Ext.each(record.getIdProps(), function(idProp) {
			var id = record.get(idProp);
			if (allowEmpty || Ext.isDefined(id)) {
				hash[idProp] = id;
			}
		}, this);

		this.addMessageActionsHash(hash, record);
		hash.timezone_iana = Intl.DateTimeFormat().resolvedOptions().timeZone;

		return hash;
	},

	/**
	 * Similar to {@link Ext.data.JsonWriter#toHash}
	 *
	 * Besides serializing the data itself, it will insert
	 * the recipients, attachments and message action commands
	 * into the object data.
	 *
	 * @param {Ext.data.Record} record The record to hash
	 * @return {Object} The hashed object
	 * @private
	 */
	toPropHash: function(record)
	{
		var hash = this.toIdHash(record, false);

		// FIXME: How to pass on deleted properties?
		hash.props = this.toHash.call(this, record);

		// FIXME: remove identification entryids from props,
		// in the future Extjs will support the 'config'
		// argument to toHash which we can use the filter
		// out the ID properties...
		this.removeIdHashFromProps(hash, record);

		// Add additional information from the subStores into the hash
		for (var key in record.subStores) {
			if (record.supportsSubStore(key) === true) {
				var store = record.subStores[key];

				if (store && store.writer) {
					Ext.apply(hash, store.writer.toPropHash(record));
				}
			}
		}

		this.addMessageActionsHash(hash, record);

		return hash;
	},

	/**
	 * remove additional identification properties from the props using the
	 * {@link Zarafa.core.data.JsonWriter.idProperties idProperties}
	 * field.
	 *
	 * @param {Object} hash The hash into which the identification fields must be added
	 * @param {Zarafa.core.data.IPMrecord} record The record to serialize from
	 * @private
	 */
	removeIdHashFromProps: function(hash, record)
	{
		Ext.each(record.getIdProps(), function(idProp) {
			if (Ext.isDefined(hash.props) && Ext.isDefined(hash.props[idProp])) {
				delete hash.props[idProp];
			}
		}, this);
	},

	/**
	 * Add message actions into the hash. Message actions are not properties
	 * which come from the server, but are used to add an additional action
	 * instruction for the server to perform. As such the action needs to
	 * be serialized separately into the hash object.
	 *
	 * @param {Object} hash The hash into which the message actions must be added
	 * @param {Zarafa.core.data.IPMrecord} record The record to serialize from
	 * @private
	 */
	addMessageActionsHash: function(hash, record)
	{
		var actions = record.getMessageActions();
		var message_action = {};

		// No Message actions defined
		if (!Ext.isDefined(actions)) {
			return;
		}

		for (var key in actions) {
			if (Ext.isDefined(actions[key])) {
				message_action[key] = actions[key];
			}
		}

		hash.message_action = message_action;
	}
});
Ext.namespace('Zarafa.core.data');

/**
 * @class Zarafa.core.data.MAPIProxy
 * @extends Ext.data.DataProxy
 */
Zarafa.core.data.MAPIProxy = Ext.extend(Ext.data.DataProxy, {
	/**
	 * @cfg {String} listModuleName Name of the listModule on the server.
	 */
	listModuleName: undefined,
	/**
	 * @cfg {String} itemModuleName Name of the itemModule on the server.
	 */
	itemModuleName: undefined,

	/**
	 * Currently active requests for {@link Zarafa.core.data.MAPIProxy MAPIProxy} mapped by the
	 * action type used in requests, {@link Zarafa.core.data.MAPIProxy MAPIProxy} can use this
	 * active request data to abort any previous request and start a new request.
	 * @property
	 * @type Object
	 * @private
	 */
	activeRequestMapping: undefined,

	/**
	 * The {@link Date#getTime timestamps} for the last time a response was received for
	 * a given {@link Zarafa.core.Actions action}.
	 * @property
	 * @type Object
	 * @private
	 */
	lastResponseTime: undefined,

	/**
	 * @constructor
	 * @param {Object} config Configuration object
	 */
	constructor: function(config)
	{
		Ext.apply(this, config);
		Zarafa.core.data.MAPIProxy.superclass.constructor.call(this, config);

		this.activeRequestMapping = {};
		this.lastResponseTime = {};
	},

	/**
	 * Check if the given action has registered requestIds in the {@link #activeRequestMapping}.
	 * When this is the case, the action is considered to be active. When no action is passed,
	 * this function will check if there are any requestsIds pending for any action.
	 *
	 * @param {Zarafa.core.Action} action The action which is to be checked.
	 * @return {Boolean} True if the given action has registered requestIds.
	 */
	isExecuting: function(action)
	{
		if (Ext.isEmpty(action)) {
			return !Ext.isEmpty(Object.keys(this.activeRequestMapping));
		} else {
			return !Ext.isEmpty(this.activeRequestMapping[action]);
		}
	},

	/**
	 * Register a requestId to a particular action. This will update {@link activeRequestMapping}
	 * to contain the requestId for the given action. By this registration it is possible to
	 * track all current outstanding requests, and it is possible to cancel them using {@link #cancelRequests}.
	 * @param {Zarafa.core.Actions} action The action for which this request id was generated
	 * @param {String} requestId The unique id which was given to the request
	 */
	addRequestId: function(action, requestId)
	{
		if (!Ext.isDefined(this.activeRequestMapping[action])) {
			this.activeRequestMapping[action] = [ requestId ];
		} else {
			this.activeRequestMapping[action].push(requestId);
		}
	},

	/**
	 * Remove a requestId from a particular action. This will update {@link activeRequestMapping}
	 * to remove the requestId from the given action.
	 * @param {Zarafa.core.Actions} requestId The unique id which was given to the request.
	 */
	deleteRequestId: function(requestId)
	{
		for (var key in this.activeRequestMapping) {
			if (Array.isArray(this.activeRequestMapping[key])) {
				this.activeRequestMapping[key].remove(requestId);
				if (Ext.isEmpty(this.activeRequestMapping[key])) {
					delete this.activeRequestMapping[key];
				}
			}
		}
	},

	/**
	 * Cancel all requests made by this proxy for a particular action.
	 * This will call {@link Zarafa.core.Request#cancelActiveRequest} to cancel
	 * the response handling of all requests which were send out by this proxy for
	 * the given action.
	 * @param {Zarafa.core.Actions} action The action
	 * @protected
	 */
	cancelRequests: function(action)
	{
		if (this.activeRequestMapping[action]) {
			var requests = this.activeRequestMapping[action];

			for (var i = 0, len = requests.length; i < len; i++) {
				container.getRequest().cancelActiveRequest(requests[i]);
			}

			delete this.activeRequestMapping[action];
		}
	},

	/**
	 * Update the {@link #lastResponseTime} with the {@link Date#getTime timestamp}
	 * of when the response for the given action was received.
	 * @param {Zarafa.core.Actions} action The action
	 * @param {Number} timestamp The timestamp
	 */
	updateExecutionTimestamp: function(action, timestamp)
	{
		this.lastResponseTime[action] = timestamp;
	},

	/**
	 * Obtain the {@link Date#getTime timestamp} of the last time
	 * the given {@link #updateExecutionTimestamp action was executed}.
	 * It will mark the time of when the action was completed (because the
	 * response was returned from the server).
	 * @param {Zarafa.core.Actions} action The action
	 * @return {Number} The timestamp of the last action
	 */
	lastExecutionTime: function(action)
	{
		return this.lastResponseTime[action] || 0;
	},

	/**
	 * Obtain the name of the listModule for communication with PHP.
	 * @param {Zarafa.core.data.MAPIRecord} record the record for which the listModuleName is requested
	 * @return {String} The name of the listModule.
	 * @private
	 */
	getListModuleName: function(record)
	{
		return this.listModuleName;
	},

	/**
	 * Obtain the name of the itemModule for communication with PHP.
	 * @param {Zarafa.core.data.MAPIRecord} record the record for which the itemModuleName is requested
	 * @return {String} The name of the itemModule.
	 * @private
	 */
	getItemModuleName: function(record)
	{
		return this.itemModuleName;
	},

	/**
	 * This will create a {@link Zarafa.core.data.ProxyResponseHandler ProxyResponseHandler} object
	 * which will be used by the {@link Zarafa.core.data.ResponseRouter ResponseRouter} when the
	 * response for the given request has returned.
	 *
	 * @param {String} modulename The modulename which is being accessed with this request
	 * @param {Zarafa.core.Actions} serverAction The action to perform on the server.
	 * @param {Ext.data.Api.action} action name of the action to perform.
	 * @param {Ext.data.Record[]} records list of records to operate on.
	 * @param {Object} parameters object containing user parameters such as range (pagination) information, sorting information, etc.
	 * @param {Ext.data.DataReader} reader data reader. Converts raw JavaScript objects (in our case) to instances of {@link Ext.data.Record}
	 * @param {Function} callback call back function to call when the request has finished successfully.
	 * @param {Object} scope scope for the call back function.
	 * @param {Object} args arguments object. This will be passed to the call back function on successful read.
	 * @return {Object} An instance of the {@link Zarafa.core.data.ProxyResponseHandler ProxyResponseHandler}
	 * which should be used for this request.
	 * @private
	 */
	getResponseHandlerForRequest: Ext.emptyFn,

	/**
	 * Performs a request for a store. This single entry point carries out all CRUD requests.
	 * @param {Ext.data.Api.action} action name of the action to perform. One of 'create', 'destroy', 'update', and 'read'.
	 * @param {Ext.data.Record[]} records list of records to operate on. In case of 'read' this will be ignored.
	 * @param {Object} parameters object containing user parameters such as range (pagination) information, sorting information, etc.
	 * @param {Ext.data.DataReader} reader data reader. Converts raw JavaScript objects (in our case) to instances of {@link Ext.data.Record}
	 * @param {Function} callback call back function to call when the request has finished successfully.
	 * @param {Object} scope scope for the call back function.
	 * @param {Object} args arguments object. This will be passed to the call back function on successful read.
	 */
	request: function(action, records, parameters, reader, callback, scope, args)
	{
		switch (action)
		{
			case 'update':
			case 'create':
				this.createUpdateAction(action, records, parameters, reader, callback, scope, args);
				break;
			case 'destroy':
				this.destroyAction(action, records, parameters, reader, callback, scope, args);
				break;
			case 'read':
				this.readAction(action, records, parameters, reader, callback, scope, args);
				break;
			case 'open':
				this.openAction(action, records, parameters, reader, callback, scope, args);
				break;
			case 'import':
				this.importAction(action, records, parameters, reader, callback, scope, args);
				break;
		}
	},

	/**
	 * @param {Ext.data.Api.action} action name of the action to perform. Either 'create' or 'update'.
	 * @param {Ext.data.Record[]} records list of records to operate on.
	 * @param {Object} parameters object containing user parameters such as range (pagination) information, sorting information, etc.
	 * @param {Ext.data.DataReader} reader data reader. Converts raw JavaScript objects (in our case) to instances of {@link Ext.data.Record}
	 * @param {Function} callback call back function to call when the request has finished successfully.
	 * @param {Object} scope scope for the call back function.
	 * @param {Object} args arguments object. This will be passed to the call back function on successful read.
	 * @private
	 */
	createUpdateAction: function(action, records, parameters, reader, callback, scope, args)
	{
		this.doRequests(args.actionType || Zarafa.core.Actions['save'], action, records, parameters, reader, callback, scope, args);
	},

	/**
	 * Performs a destroy action on one or more records.
	 * @param {Ext.data.Api.action} action name of the action to perform. Always 'destroy'.
	 * @param {Ext.data.Record[]} records list of records to operate on.
	 * @param {Object} parameters object containing user parameters such as range (pagination) information, sorting information, etc.
	 * @param {Ext.data.DataReader} reader data reader. Converts raw JavaScript objects (in our case) to instances of {@link Ext.data.Record}
	 * @param {Function} callback call back function to call when the request has finished successfully.
	 * @param {Object} scope scope for the call back function.
	 * @param {Object} args arguments object. This will be passed to the call back function on successful read.
	 * @private
	 */
	destroyAction: function(action, records, parameters, reader, callback, scope, args)
	{
		this.doRequests(args.actionType || Zarafa.core.Actions['delete'], action, records, parameters, reader, callback, scope, args);
	},

	/**
	 * Performs a read action on one or more records.
	 * @param {Ext.data.Api.action} action name of the action to perform. Always 'open'.
	 * @param {Ext.data.Record[]} records list of records to operate on.
	 * @param {Object} parameters object containing user parameters such as range (pagination) information, sorting information, etc.
	 * @param {Ext.data.DataReader} reader data reader. Converts raw JavaScript objects (in our case) to instances of {@link Ext.data.Record}
	 * @param {Function} callback call back function to call when the request has finished successfully.
	 * @param {Object} scope scope for the call back function.
	 * @param {Object} args arguments object. This will be passed to the call back function on successful read.
	 * @private
	 */
	openAction: function(action, records, parameters, reader, callback, scope, args)
	{
		this.doRequests(args.actionType || Zarafa.core.Actions['open'], action, records, parameters, reader, callback, scope, args);
	},

	/**
	 * Performs an import action on one or more records.
	 * @param {Ext.data.Api.action} action name of the action to perform. Always 'import'.
	 * @param {Ext.data.Record[]} records list of records to operate on.
	 * @param {Object} parameters object containing user parameters such as range (pagination) information, sorting information, etc.
	 * @param {Ext.data.DataReader} reader data reader. Converts raw JavaScript objects (in our case) to instances of {@link Ext.data.Record}
	 * @param {Function} callback call back function to call when the request has finished successfully.
	 * @param {Object} scope scope for the call back function.
	 * @param {Object} args arguments object. This will be passed to the call back function on successful read.
	 * @private
	 */
	importAction: function(action, records, parameters, reader, callback, scope, args)
	{
		this.doRequests(args.actionType || Zarafa.core.Actions['import'], action, records, parameters, reader, callback, scope, args);
	},

	/**
	 * Performs a read action on a Folder/Store to load all records.
	 * @param {Ext.data.Api.action} action name of the action to perform. Always 'read'.
	 * @param {Ext.data.Record[]} records list of records to operate on. In case of 'read' this will be ignored.
	 * @param {Object} parameters object containing user parameters such as range (pagination) information, sorting information, etc.
	 * @param {Ext.data.DataReader} reader data reader. Converts raw JavaScript objects (in our case) to instances of{@link Ext.data.Record}
	 * @param {Function} callback call back function to call when the request has finished successfully.
	 * @param {Object} scope scope for the call back function.
	 * @param {Object} args arguments object. This will be passed to the call back function on successful read.
	 * @private
	 */
	readAction: function(action, records, parameters, reader, callback, scope, args)
	{
		this.doRequests(args.actionType || Zarafa.core.Actions['list'], action, records, parameters, reader, callback, scope, args);
	},

	/**
	 * Initialize the the {@link Zarafa.core.Request request} structure. The initial
	 * {@link Zarafa.core.Request request} will be reset and new requests will be added
	 * for each individual {@link Zarafa.core.date.MAPIRecord record} which was provided.
	 * @param {Zarafa.core.Actions} serverAction The action to perform on the server.
	 * @param {Ext.data.Api.action} action name of the action to perform.
	 * @param {Ext.data.Record[]} records list of records to operate on.
	 * @param {Object} parameters object containing user parameters such as range (pagination) information, sorting information, etc.
	 * @param {Ext.data.DataReader} reader data reader. Converts raw JavaScript objects (in our case) to instances of {@link Ext.data.Record}
	 * @param {Function} callback call back function to call when the request has finished successfully.
	 * @param {Object} scope scope for the call back function.
	 * @param {Object} args arguments object. This will be passed to the call back function on successful read.
	 * @private
	 */
	doRequests: function(serverAction, action, records, parameters, reader, callback, scope, args)
	{
		// Check if the previous request needs to be cancelled.
		if (args.cancelPreviousRequest === true) {
			this.cancelRequests(serverAction);
		}

		// reset the request object, starts a new composite request
		container.getRequest().reset();

		// If records are provided, we must perform a 'itemmodule' action on each of the given records
		if (records && args.listRequest !== true) {
			var items = parameters.jsonData[reader.meta.root];

			// Force the records object to be an array
			if (!Array.isArray(records)) {
				records = [ records ];
			}

			// Force the serialized data to be an array
			if (!Array.isArray(items)) {
				items = [ items ];
			}

			for (var i = 0; i < records.length; i++) {
				var record = records[i];
				var data = items[i];
				var module = this.getItemModuleName(record);
				var handler = this.getResponseHandlerForRequest(module, serverAction, action, record, parameters, reader, callback, scope, args);

				// Add the request
				var requestId = container.getRequest().addRequest(module, serverAction, data, handler);

				// store reference of transaction id to active request mapping
				this.addRequestId(serverAction, requestId);
			}
		} else {
			// No records were provided, we must perform a 'listmodule' action
			var module = this.getListModuleName(records);
			var handler = this.getResponseHandlerForRequest(module, serverAction, action, records, parameters, reader, callback, scope, args);

			// Add the request
			var requestId = container.getRequest().addRequest(module, serverAction, parameters, handler);

			// store reference of transaction id to active request mapping
			this.addRequestId(serverAction, requestId);
		}

		// send out the request
		container.getRequest().send();
	}
});
Ext.namespace('Zarafa.core.data');

// Extend Ext.data.Field documentation with a single config option.
/**
 * @class Ext.data.Field
 * @extends Object
 *
 * @cfg {Boolean} forceProtocol Used to determine if this field must
 * always be send to the server when {@link Zarafa.core.data.MAPIRecord#set record.set}
 * has been called. By default a field is only send to the server when it has been
 * modified, but with this option enabled, it will also be send when it has simply been set
 * with the same value as before.
 */

/**
 * @class Zarafa.core.data.MAPIRecord
 * @extends Ext.data.Record
 *
 * An extension to the {@link Ext.data.Record Record} that adds the open() method for loading
 * the 'full' contents of a MAPI item. The list modules on the server side only return partial records,
 * omitting, for example, the body of email messages. The open() method can be used to retrieve
 * these fields.
 *
 */
Zarafa.core.data.MAPIRecord = Ext.extend(Ext.data.Record, {
	/**
	 * The base array of ID properties which is copied to the {@link #idProperties}
	 * when the record is being created.
	 * @property
	 * @type Array
	 * @private
	 */
	baseIdProperties: [ 'entryid', 'store_entryid', 'parent_entryid' ],

	/**
	 * The array of properties which must be used to uniquely
	 * identify this record on the server (See {@link #baseIdProperties}.
	 * @property
	 * @type Array
	 * @private
	 */
	idProperties: undefined,

	/**
	 * The key-value object of {@link Zarafa.core.data.MAPISubStore MAPISubStore} which stores
	 * data of the complex properties like recipients, attachments etc. This container is mainly
	 * created to hold all {@link Zarafa.core.data.MAPISubStore MAPISubStore} at one place so
	 * {@link Zarafa.core.data.JsonWriter JsonWriter} can determine which {@link Zarafa.core.data.MAPISubStore MAPISubStore}
	 * it needs to serialize when serializing an {@link Zarafa.core.data.MAPIRecord MAPIRecord}.
	 * @property
	 * @type Object
	 * @private
	 */
	subStores: undefined,

	/**
	 * The key-value object of Booleans which stores the name of each SubStore which is supported
	 * by this Record. For each subStore name, the matching SubStore Type is provided which must be
	 * used to allocate the subStore. If no type is provided, then the given SubStore is not supported
	 * by this record.
	 * @property
	 * @type Object
	 * @private
	 */
	subStoresTypes: undefined,

	/**
	 * Used by {@link #setEventPropagation} and {@link #hasEventPropagation}, using this a record
	 * can be configured to prevent any events from the {@link Zarafa.core.data.IPMStore store} to
	 * be propagated through the {@link Zarafa.core.data.IPMStoreMgr IPMStoreMgr}
	 * @property
	 * @type Boolean
	 * @private
	 */
	eventPropagation: true,

	/**
	 * True to enable fine-grained modifications tracking between individual {@link Ext.data.Store#update} events.
	 * When trackUpdateModifications is true, {@link #updateModifications} will keep track of all properties
	 * which were updated since the last {@link Ext.data.Store#update} event.
	 * Must be updated through {@link #setUpdateModificationsTracking}.
	 * @property
	 * @type Boolean
	 * @private
	 */
	trackUpdateModifications: false,

	/**
	 * If {@link #trackUpdateModifications} is true, this field will contain all changes since the
	 * last {@link Ext.data.Store#update} event. When the next {@link Ext.data.Store#update} event
	 * has been fired, this object will be cleared again.
	 * @property
	 * @type Object
	 * @private
	 */
	updateModifications: undefined,

	/**
	 * If {@link #trackUpdateModifications} is true, this field will contains all substores which have
	 * fired the 'update' event since the last {@link Ext.data.Store#update} event. When the next
	 * {@link Ext.data.Store#update} event has been fired, this object will be cleared again.
	 * @property
	 * @type Object
	 * @private
	 */
	updateSubStoreModifications: undefined,

	/**
	 * The number of editors working on this records. The {@link #beginEdit} and {@link #endEdit}
	 * support nested editing blocks. This means that {@link #update} will not be fired until
	 * this counter drops to 0.
	 * @property
	 * @type Number
	 * @private
	 */
	editingCount: 0,

	/**
	 * The property will contain list of sub action types that will be sent to server when saving/deleting
	 * this record.
	 * @property
	 * @type Object
	 */
	actions: undefined,

	/**
	 * True if record is used for modal dialog. This is required because modal dialog will be the second dialog,
	 * with the same record (even though they are copies the entryid will be the same for both)
	 *
	 * @property
	 * @type Boolean
	 * @private
	 */
	isModalDialogRecord: false,

	/**
	 * @constructor
	 * @param {Object} data The data which must be applied to this record
	 * @param {Object} id The unique id for this record
	 * @param {Zarafa.core.data.RecordDefinition} definition The record definition used to
	 * construct this record
	 */
	constructor: function(data, id, definition)
	{
		Zarafa.core.data.MAPIRecord.superclass.constructor.call(this, data, id);

		this.idProperties = this.baseIdProperties.clone();

		this.actions = {};

		// initialize substore container
		this.subStores = {};
		this.subStoresTypes = {};

		if (definition) {
			this.subStoresTypes = definition.getSubStores();
		}
	},

	/**
	 * Copy the {@link Zarafa.core.data.MAPIRecord Record} to a new instance
	 * @param {String} newId (optional) A new Record id, defaults to the id of the record being copied. See id.
	 * @return {Zarafa.core.data.MAPIRecord} The copy of the record.
	 */
	copy: function(newId)
	{
		var copy = Zarafa.core.data.RecordFactory.createRecordObjectByRecordData(this.data, newId || this.id);

		copy.idProperties = this.idProperties.clone();
		copy.phantom = this.phantom;

		return copy.applyData(this, true);
	},

	/**
	 * Applies all data from an {@link Zarafa.core.data.MAPIRecord Record}
	 * to this instance. This will update all data, attachments, recipients, etc..
	 *
	 * @param {Zarafa.core.data.MAPIRecord} record The record to apply to this
	 * @param {Boolean} cheapCopy True to allow cheap assignment rather then the more
	 * expensive copying of all data.
	 * @return {Zarafa.core.data.MAPIRecord} this
	 */
	applyData: function(record, cheapCopy)
	{
		this.beginEdit();

		// This has to be synced before calling set(), as this field will
		// be used inside that function.
		this.trackUpdateModifications = record.trackUpdateModifications;

		// For each key in the remote data set, we are going to manually set
		// it inside our own dataset. This ensures that the 'modified' and
		// 'updateModifications' fields will automatically be updated to contain
		// the correct set of data.
		// if record is modal dialog record then instead of all merge only those properties which was changed
		var data = record.isModalDialogRecord ? record.modified : record.data;
		for (var key in data) {
			if(key === 'message_class') {
				// Don't update record's message_class while updating the record
				// As it is it's identity of record type and fields assigned to it.
				continue;
			}
			if (Ext.isDate(record.data[key])) {
				this.set(key, record.data[key].clone());
			} else {
				this.set(key, record.data[key]);
			}
		}

		// The actions are separate from the 'data', so we must copy it
		// separately. Note that we have no change or event mechanism for
		// this field, so bluntly copying the object is sufficient.
		this.actions = Ext.apply({}, record.actions);

		// Trigger a fake 'open' event, when the original
		// record is open and the id properties indicate that
		// both instances are referring to the exact same instance
		if (this.idProperties.equals(record.idProperties)) {
			// Create all substores & merge the contents into
			// the new substores. Perform this action before
			// calling afterOpen() as that function expects
			// the contents of the substores to be properly
			// initialized, and might add additional records
			// into the substores
			this.createSubStores();
			this.mergeSubStores(record.subStores, cheapCopy);

			// We are done with merging everything,
			// call afterOpen() if needed to ensure the
			// last initialization of the record occurs,
			// and if record which referred by "this"
			// is not opened.
			if (record.isOpened() && !this.isOpened()) {
				this.afterOpen();
			}
		}

		this.endEdit();

		return this;
	},

	/**
	 * Save all changes inside this record to the store.
	 * This will call {@link Zarafa.core.data.MAPIStore#save} with itself
	 * as argument.
	 */
	save: function()
	{
		this.getStore().save(this);
	},

	/**
	 * Compare this {@link Zarafa.core.data.MAPIRecord record} instance with another one to see
	 * if they are the same MAPI Item from the server (i.e. The entryid matches).
	 *
	 * @param {Zarafa.core.data.MAPIRecord} record The MAPIRecord to compare with
	 * @return {Boolean} True if the records are the same.
	 */
	equals: Ext.emptyFn,

	/**
	 * Enable/Disable event {@link #eventPropagation propagation} from the
	 * {@link Zarafa.core.data.IPMStore store} regarding this record by the
	 * {@link Zarafa.core.data.IPMStoreMgr IPMStoreMgr}.
	 * @param {Boolean} eventPropagation True if events can be propagated.
	 */
	setEventPropagation: function(eventPropagation)
	{
		this.eventPropagation = eventPropagation;
	},

	/**
	 * Checks if events can from the {@link Zarafa.core.data.IPMStore store} regarding
	 * this record can be {@link #eventPropagation propagated} further by the
	 * {@link Zarafa.core.data.IPMStoreMgr IPMStoreMgr}.
	 * @return {Boolean} True if events can be propagated.
	 */
	hasEventPropagation: function()
	{
		return this.eventPropagation;
	},

	/**
	 * Enable/disable UpdateModifications tracking. This will keep track of the
	 * exact modifications between two {@link Ext.data.Store#update} events. This allows
	 * finegrained tuning of UI components which constantly listen to
	 * {@link Ext.data.Store#update} events and only require the modifications since the
	 * last update call.
	 * @param {Boolean} enable Enable updatemodification tracking
	 */
	setUpdateModificationsTracking: function(enable)
	{
		this.trackUpdateModifications = enable;
	},

	/**
	 * Get the updateModifications tracking status
	 * @return {Boolean} True when update Modifications tracking has been enabled..
	 */
	getUpdateModificationsTracking: function()
	{
		return this.trackUpdateModifications;
	},

	/**
	 * @return {Ext.data.Store} data store this record belongs to.
	 */
	getStore: function()
	{
		return this.store;
	},

	/**
	 * Opens the record, loading all fields.
	 * @param {Object} options Extra options to be used for loading the record
	 */
	open: function(options)
	{
		if (this.isOpened() && (!options || options.forceLoad !== true)) {
			return;
		}
		this.store.open(this, options);
	},

	/**
	 * Called by the store after the record was opened successfully.
	 * @private
	 */
	afterOpen: function()
	{
		this.opened = true;
		this.createSubStores();
		delete this.updateModifications;
		delete this.updateSubStoreModifications;
	},

	/**
	 * @return {Boolean} true iff the record has been fully loaded.
	 */
	isOpened: function()
	{
		return this.opened === true;
	},

	/**
	 * Set a value for the given fieldname.
	 * @param {String} name The {@link Ext.data.Field#name name of the field} to set.
	 * @param {String/Object/Array} value The value to set the field to.
	 * @param {Boolean} force (optional) True to force the property to be changed in
	 * the modified array. Setting this argument will effectively override the
	 * {@link Ext.data.Field#forceProtocol forceProtocol} option for the property we are modifying here.
	 */
	set: function(name, value, force)
	{
		var forceProtocol = force;
		if (!Ext.isDefined(forceProtocol)) {
			var field = this.fields.get(name);
			forceProtocol = Ext.isDefined(field) ? field.forceProtocol : false;
		}

		if (this.trackUpdateModifications === true) {
			var encode = Ext.isPrimitive(value) ? String : Ext.encode;

			if (encode(this.data[name]) !== encode(value)) {
				// If the value is different, then we have to update
				// the 'updateModifications'. This is slightly different
				// behavior then the 'modified' array, as that array will
				// be updated even when the value is the same but
				// 'forceProtocol' was enabled.
				if (!this.updateModifications) {
					this.updateModifications = {};
				}

				if (this.updateModifications[name] === undefined) {
					this.updateModifications[name] = this.data[name];
				}
			} else if (forceProtocol !== true) {
				// No changes were made, nor is this property being
				// forced to be transmitted to the server...
				return;
			}
		}

		Zarafa.core.data.MAPIRecord.superclass.set.call(this, name, value);
		this.modified = this.modified || {};
		if (forceProtocol === true && this.modified[name] === undefined) {
			this.modified[name] = this.data[name];
		}
	},

	/**
	 * Called after a record has been edited
	 * @private
	 */
	afterEdit: function()
	{
		Zarafa.core.data.MAPIRecord.superclass.afterEdit.call(this);
		delete this.updateModifications;
		delete this.updateSubStoreModifications;
	},

	/**
	 * Called after a record modifications have been rejected
	 * @private
	 */
	afterReject: function()
	{
		Zarafa.core.data.MAPIRecord.superclass.afterReject.call(this);
		delete this.updateModifications;
		delete this.updateSubStoreModifications;
	},

	/**
	 * Begin an edit. While in edit mode, no events (e.g.. the <code>update</code> event)
	 * are relayed to the containing store.
	 * See also: <code>{@link #endEdit}</code> and <code>{@link #cancelEdit}</code>.
	 *
	 * This function is overridden from {@link Ext.data.Record#beginEdit} and adds
	 * support for nested editing blocks by using the {@link #editingCount}.
	 */
	beginEdit: function()
	{
		// Increase editing counter, if it is a negative value, it means that
		// it has been corrupted and we must force it to something valid.
		this.editingCount++;
		if (this.editingCount < 1) {
			this.editingCount = 1;
		}

		// If this is not a nested call, we can direct the call to the superclass.
		if (this.editingCount === 1) {
			Zarafa.core.data.MAPIRecord.superclass.beginEdit.call(this);
			this.updateModifications = this.updateModifications || {};
			this.updateSubStoreModifications = this.updateSubStoreModifications || {};
		}
	},

	/**
	 * Cancels all changes made in the current edit operation.
	 *
	 * This function is overridden from {@link Ext.data.Record#cancelEdit} and adds
	 * support for nested editing blocks by using the {@link #editingCount}.
	 */
	cancelEdit: function()
	{
		// Increase editing counter, if it is a negative value, it means that
		// it has been corrupted and we must force it to something valid.
		this.editingCount--;
		if (this.editingCount < 0) {
			this.editingCount = 0;
		}

		// If this is not a nested call, we can direct the call to the superclass.
		if (this.editingCount === 0) {
			Zarafa.core.data.MAPIRecord.superclass.cancelEdit.call(this);
			delete this.updateModifications;
			delete this.updateSubStoreModifications;
		}
	},

	/**
	 * Ends all editing made in the current edit operation and calls {@link #afterEdit}.
	 *
	 * This function is overridden from {@link Ext.data.Record#endEdit} and adds
	 * support for nested editing blocks by using the {@link #editingCount}.
	 */
	endEdit: function()
	{
		// Increase editing counter, if it is a negative value, it means that
		// it has been corrupted and we must force it to something valid.
		this.editingCount--;
		if (this.editingCount < 0) {
			this.editingCount = 0;
		}

		// If this is not a nested call, we can direct the call to the superclass.
		if (this.editingCount === 0) {
			Zarafa.core.data.MAPIRecord.superclass.endEdit.call(this);
		}
	},

	/**
	 * Usually called by the {@link Ext.data.Store} which owns the Record.
	 * Commits all changes made to the Record since either creation, or the last commit operation.
	 * <p>Developers should subscribe to the {@link Ext.data.Store#update} event
	 * to have their code notified of commit operations.</p>
	 * @param {Boolean} silent (optional) True to skip notification of the owning
	 * store of the change (defaults to false)
	 */
	commit: function(silent)
	{
		// Delete the local modification tracking
		delete this.updateModifications;
		delete this.updateSubStoreModifications;

		// Commit changes to substores
		for (var key in this.subStores) {
			this.subStores[key].commitChanges();
		}

		Zarafa.core.data.MAPIRecord.superclass.commit.call(this, silent);
	},

	/**
	 * When UpdateModifications tracking has been enabled, this function will return
	 * true if the given field has been modified since the last {@link Ext.data.Store#update}
	 * event. If UpdateModifications tracking has been disabled, this function will return
	 * the same value as {@link #isModified}.
	 * @param {String} fieldName The fieldname which has been modified
	 * @return {Boolean} True if the field has been modified
	 */
	isModifiedSinceLastUpdate: function(fieldName)
	{
		if (this.trackUpdateModifications === true) {
			return !!(this.updateModifications && this.updateModifications.hasOwnProperty(fieldName));
		} else {
			return this.isModified(fieldName);
		}
	},

	/**
	 * When updateModifications tracking has been enabled, this function will return
	 * true if the given substore has been modified since the last {@link Ext.data.Store#update}
	 * event. UpdateModifications tracking has been disabled, this function will return
	 * if the subStore has been modified since the subStore was created.
	 * @param {String} subStore Name of the subStore
	 * @return {Boolean} True if the field has been modified
	 */
	isSubStoreModifiedSincelastUpdate: function(subStore)
	{
		if (this.trackUpdateModifications === true) {
			return !!(this.updateSubStoreModifications && this.updateSubStoreModifications.hasOwnProperty(subStore));
		} else {
			subStore = this.getSubStore(subStore);
			return (!Ext.isEmpty(subStore.modified) || !Ext.isEmpty(subStore.removed));
		}
	},

	/**
	 * Returns the list of all added/modified/deleted records inside the subStore since the last
	 * {@link Ext.data.Store#update} event.
	 * @param {String} subStore Name of the subStore
	 * @return {Array} The array of the records which were changed since the last update.
	 */
	getSubStoreChangesSinceLastUpdate: function(subStore)
	{
		if (this.trackUpdateModifications === true) {
			if (this.updateSubStoreModifications && this.updateSubStoreModifications[subStore]) {
				return this.updateSubStoreModifications[subStore].changes;
			}
		} else {
			subStore = this.getSubStore(subStore);
			return [].concat(subStore.modified, subStore.removed);
		}
	},

	/**
	 * Get the Message Action list for the {@link Zarafa.core.data.MAPIRecord record}.
	 * @return {Mixed} The Message Action list.
	 */
	getMessageActions: function()
	{
		return this.actions;
	},

	/**
	 * Get Message Action for the {@link Zarafa.core.data.MAPIRecord record}.
	 * @param {String} actionName The name of action.
	 * @return {Mixed} The Message Action.
	 */
	getMessageAction: function(actionName)
	{
		if(this.actions[actionName]) {
			return this.actions[actionName];
		} else {
			return false;
		}
	},

	/**
	 * Add action to Message Action list
	 * @param {String} name The action name to add to the list.
	 * @param {String} value The value attached to the action name
	 */
	addMessageAction: function(name, value)
	{
		this.actions[name] = value;

		// @todo we don't want to send updates at this point since the record may not be complete. Needs a begin/end construct
		// to work properly

		// Notify modification change, but do not send a notification to the UI (since no UI has changed)
		if(Ext.isDefined(this.store) && this.store.modified.indexOf(this) == -1){
			this.store.modified.push(this);
		}
	},

	/**
	 * Delete action from the Message Action list
	 * @param {String} name The action name to delete from the list.
	 */
	deleteMessageAction: function(name)
	{
		delete this.actions[name];
	},

	/**
	 * @param {String} name name of message action.
	 * @return {Boolean} True if a {@link #actions message action} with the given name exists.
	 */
	hasMessageAction: function(name)
	{
		return Ext.isDefined(this.actions[name]);
	},

	/**
	 * Clear all Message Actions.
	 */
	clearMessageActions: function()
	{
		this.actions = {};
	},

	/**
	 * Clear Action Response.
	 */
	clearActionResponse: function()
	{
		delete this.action_response;
	},

	/**
	 * Get requested data from the Action Response.
	 * @param {String} key Requested action response property
	 * @return {Mixed} The corresponding data
	 */
	getActionResponse: function(key)
	{
		if(this.action_response){
			return this.action_response[key];
		}
	},

	/**
	 * Unset private the {@link Zarafa.core.data.MAPIRecord record} to a different
	 * {@link Zarafa.hierarchy.data.MAPIFolderRecord folder}.
	 * @param {Zarafa.hierarchy.data.MAPIFolderRecord} folder The folder to copy/move the record to
	 */
	unsetPrivate: function()
	{
		this.addMessageAction('unset_Private', true);
	},

	/**
	 * Copy the {@link Zarafa.core.data.MAPIRecord record} to a different
	 * {@link Zarafa.hierarchy.data.MAPIFolderRecord folder}.
	 * @param {Zarafa.hierarchy.data.MAPIFolderRecord} folder The folder to copy the record to
	 */
	copyTo: function(folder)
	{
		this.addMessageAction('action_type', 'copy');
		this.addMessageAction('destination_parent_entryid', folder.get('entryid'));
		this.addMessageAction('destination_store_entryid', folder.get('store_entryid'));
	},

	/**
	 * Move the {@link Zarafa.core.data.MAPIRecord record} to a different
	 * {@link Zarafa.hierarchy.data.MAPIFolderRecord folder}.
	 * @param {Zarafa.hierarchy.data.MAPIFolderRecord} folder The folder to copy the record to
	 */
	moveTo: function(folder)
	{
		this.addMessageAction('action_type', 'move');
		this.addMessageAction('destination_parent_entryid', folder.get('entryid'));
		this.addMessageAction('destination_store_entryid', folder.get('store_entryid'));
	},

	/**
	 * Adds message action 'suppress_exception' which suppresses an exception pop-up.
	 */
	suppressException: function()
	{
		this.addMessageAction('suppress_exception', true);
	},

	/**
	 * Checks if the SubStore with the given name is supported by this record.
	 * @param {String} name The name of the subStore to check
	 * @return {Boolean} True if the given SubStore is supported by this Record.
	 */
	supportsSubStore: function(name)
	{
		return Ext.isFunction(this.subStoresTypes[name]);
	},

	/**
	 * This will create a new {@link #subStores SubStore} for the given name
	 * (if this is {@link #supportsSubStore supported}). The new substore will
	 * automatically be {@link #setSubStore set} on this record.
	 *
	 * @param {String} name The name of the subStore to create
	 * @return {Zarafa.core.data.MAPISubStore} The new substore.
	 */
	createSubStore: function(name)
	{
		if (this.supportsSubStore(name)) {
			var store = this.getSubStore(name);
			if (!Ext.isDefined(store)) {
				store = this.setSubStore(name, new this.subStoresTypes[name]());
				store.on('update', this.onSubStoreUpdate, this);
				store.on('add', this.onSubStoreChange, this);
				store.on('remove', this.onSubStoreChange, this);
			}
			return store;
		}
	},

	/**
	 * Create all {@link #subStores} which are {@link #subStoresTypes supported}
	 * by this record.
	 */
	createSubStores: function()
	{
		for (var key in this.subStoresTypes) {
			if (!this.getSubStore(key)) {
				this.createSubStore(key);
			}
		}
	},

	/**
	 * Get the SubStore for a particular name. This will get the SubStore from the {@link #subStores} field.
	 * @param {String} name The name of the substore to get
	 * @return {Zarafa.core.data.MAPISubStore} The substore.
	 */
	getSubStore: function(name)
	{
		if (this.subStores !== null) {
			return this.subStores[name];
		}
		return undefined;
	},

	/**
	 * Set the SubStore for a particular name. This will set the SubStore on the {@link #subStores} field.
	 * @param {String} name The name of the subStore to set
	 * @param {Zarafa.core.data.MAPISubStore} store the subStore.
	 * @return {Zarafa.core.data.MAPISubStore} The substore.
	 */
	setSubStore: function(name, store)
	{
		if (this.subStores === null) {
			this.subStores = {name: store};
		} else {
			this.subStores[name] = store;
		}
		store.setParentRecord(this);
		return store;
	},

	/**
	 * Merge a substore into the substore inside this record.
	 * @param {String} name The name of the subStore to merge
	 * @param {Zarafa.core.data.MAPISubStore} remoteSubStore The store to merge
	 * @param {Boolean} cheapCopy Use the cheap assignment rather then the more expensive copying
	 * of all records
	 */
	mergeSubStore: function(name, remoteSubStore, cheapCopy)
	{
		var subStore = this.getSubStore(name);

		if (subStore && remoteSubStore) {
			if (cheapCopy !== true ) {
					// When we are not performing a cheap copy we wish to preserve
					// the "add", "modify" and "delete" changes in the subStore.

					var prop = name === 'attachments' ? 'attach_id' : 'entryid';
					var isPermissionSubStore = name === 'permissions';
					// Go over the current store, and start searching for the corresponding
					// record in the remote store.
					subStore.each(function(record) {
						var remoteRecordIndex = remoteSubStore.findBy(function (remoteRecord) {
							var isRecordEqual = this.idComparison(record.get(prop), remoteRecord.get(prop));
							// If substore is permissions substore then check both the copy of record 
							// having same rights if not it means substore has old copy of user permission record 
							// which needs to remove from substore.
							if (isPermissionSubStore && isRecordEqual) {
								return record.get("rights") === remoteRecord.get("rights");
							}
							return isRecordEqual;
						}, this);

						if (remoteRecordIndex < 0) {
							// The other store doesn't contain this record,
							// remove it from the current store.
							subStore.remove(record);
						}
					}, this);

					// Go over the remote store to search for any new records which were added
					remoteSubStore.each(function(record) {
						var origRecordIndex = subStore.findBy(function (storeRecord) {
							return this.idComparison(record.get(prop), storeRecord.get(prop));
						}, this);

						if (origRecordIndex < 0) {
							// New record, add it to the current store.
							subStore.add(record.copy());
						}
					}, this);
			} else {
				// A cheap copy is nothing more that destroy all
				// currently available data and move all records
				// from the remote store into the current store.
				// We fire the 'datachanged' event to inform the
				// UI of the bulk change which has been performed.
				subStore.removeAll(true);
				subStore.add(remoteSubStore.getRange(), true);
				subStore.fireEvent('datachanged', subStore);
			}
		}
	},

	/**
	 * Function which is used to compare two entry ids also
	 * it is take care of comparing local contact items.
	 *
	 * @param {String} entryIdOne The first id to compare
	 * @param {String} entryIdTwo The second id to compare
	 * @return {Boolean} return true if entryId is same else false.
	 * @protected
	 */
	idComparison: function(entryIdOne, entryIdTwo)
	{
		entryIdOne = Zarafa.core.EntryId.hasContactProviderGUID(entryIdOne) ?
			Zarafa.core.EntryId.unwrapContactProviderEntryId(entryIdOne): entryIdOne;

		entryIdTwo = Zarafa.core.EntryId.hasContactProviderGUID(entryIdTwo) ?
			Zarafa.core.EntryId.unwrapContactProviderEntryId(entryIdTwo): entryIdTwo;

		return Zarafa.core.EntryId.compareEntryIds(entryIdOne, entryIdTwo);
	},

	/**
	 * Merge all data from the object containing subStores into this record.
	 * This will call {@link #mergeSubStore} for each subStore found in the SubStores object,
	 * note that only stores which are {@link #supportsSubStore supported} will be merged.
	 * @param {Object} subStores The key-value array containing all subStores which must
	 * be applied to the record.
	 * @param {Boolean} cheapCopy Use the cheap assignment rather then the more expensive copying
	 * of all records
	 */
	mergeSubStores: function(subStores, cheapCopy)
	{
		for (var key in subStores) {
			if (this.supportsSubStore(key)) {
				this.mergeSubStore(key, subStores[key], cheapCopy);
			}
		}
	},

	/**
	 * Event handler which is fired when data in this subStore has changed. This will markt the subStore as
	 * changed and force the {@link Ext.data.Store#update} event of the store of this record.
	 * @param {Zarafa.core.data.MAPISubStore} store The store which was changed
	 * @param {Zarafa.core.data.MAPIRecord[]} records The records which were added/modified/removed
	 * @private
	 */
	onSubStoreChange: function(store, records)
	{
		for (var key in this.subStores) {
			if (this.subStores[key] === store) {
				if (this.trackUpdateModifications === true) {
					if (!this.updateSubStoreModifications) {
						this.updateSubStoreModifications = {};
					}

					if (!Ext.isDefined(records)) {
						records = store.getRange();
					}

					var changes = this.updateSubStoreModifications[key];
					if (!changes) {
						changes = {
							store: store,
							changes: [].concat(records)
						};
						this.updateSubStoreModifications[key] = changes;
					} else {
						changes.changes = changes.changes.concat(records);
					}
				}

				this.dirty = true;
				// Because we manually force the 'update' event to be fired,
				// we must create the local modified array. ExtJs demands
				// that afterEdit() must only be called when the modified
				// array exists.
				this.modified = this.modified || {};
				if (!this.editing) {
					this.afterEdit();
				}
				break;
			}
		}
	},

	/**
	 * Event handler which is fired when data in this subStore has been updated. This will check
	 * if the given action is {@link Ext.data.Record#COMMIT}, in that case no action is taken,
	 * in other cases {@link #onSubStoreChange} is called to act upon a change inside the substore
	 * contents.
	 * @param {Zarafa.core.data.MAPISubStore} store The store which fired the event
	 * @param {Zarafa.core.data.MAPIRecord[]} records The records which were modified
	 * @param {String} action The action which was performed (could be
	 * {@link Ext.data.Record#EDIT}, {@link Ext.data.Record#REJECT}, {@link Ext.data.Record#COMMIT}).
	 * @private
	 */
	onSubStoreUpdate: function(store, records, action)
	{
		if (action !== Ext.data.Record.COMMIT) {
			this.onSubStoreChange(store, records);
		}
	},

	/**
	 * Add extra Identification property to the {@link #idProperties} array.
	 * This will serialize the given property into the identification section
	 * when communication with the server.
	 *
	 * @param {String} prop The propertyname to be added
	 */
	addIdProp: function(prop)
	{
		if (!this.hasIdProp(prop)) {
			this.idProperties.push(prop);
		}
	},

	/**
	 * Check if the given property is an {@link #idProperties id prop}.
	 * @param {String} prop The name of the property
	 * @return {Boolean} True if the given prop is an ID property
	 */
	hasIdProp: function(prop)
	{
		return this.idProperties.indexOf(prop) > -1;
	},

	/**
	 * Remove extra Identification property from the {@link #idProperties} array.
	 * @param {String} prop The name of the property
	 */
	removeIdProp: function(prop)
	{
		if(this.hasIdProp(prop)) {
			this.idProperties.splice(this.idProperties.indexOf(prop), 1);
		}
	},

	/**
	 * Obtain the list of of identification properties ({@link #idProperties}).
	 * These properties will be placed inside the identification section of
	 * the protocol during the communication with the server.
	 *
	 * @return {array} the array of identification properties.
	 */
	getIdProps: function()
	{
		return this.idProperties;
	},

	/**
	 * Convenience method to get {@link Zarafa.core.mapi.DisplayType} or {@link Zarafa.core.mapi.DisplayTypeEx}
	 * property value from {@link Zarafa.core.data.IPMRecord}.
	 *
	 * @return {Zarafa.core.mapi.DisplayType|Zarafa.core.mapi.DisplayTypeEx} The display type value.
	 */
	getDisplayType: function()
	{
		var displayType = this.get('display_type');
		var displayTypeEx = this.get('display_type_ex');
		var returnValue;

		switch(displayType) {
			case Zarafa.core.mapi.DisplayType.DT_MAILUSER:
			case Zarafa.core.mapi.DisplayType.DT_DISTLIST:
				returnValue = displayTypeEx & ~Zarafa.core.mapi.DisplayTypeEx.DTE_FLAG_ACL_CAPABLE;
				break;
			default:
				returnValue = displayType;
				break;
		}

		return returnValue;
	},

	/**
	 * Destroy the record, this will destroy the record and the record data
	 * ensuring that all references are lost.
	 */
	destroy: function()
	{
		// Destroy all substores
		for (var key in this.subStores) {
			this.subStores[key].destroy();
		}
		this.subStores = null;
	},

	/**
	 * @returns {boolean} return true only if record has delete access else false.
	 */
	hasDeleteAccess: function ()
	{
		return (this.get('access') & Zarafa.core.mapi.Access.ACCESS_DELETE) > 0;
	},

	/**
	 * Event handler for the 'datachanged' event of {@link Ext.data.Store store}
	 * When we have modal dialog open and if we receive a new email then modified, store
	 * and sub store of the selected records are not accessible anymore,
	 * so we have to set the modified things from the record along with updated store in modal record.
	 *
	 * @param {Zarafa.core.data.ListModuleStore} store This store
	 * @private
	 */
	onDataChange: function (store)
	{
		var modalRecord = this.modalRecord;
		if (Ext.isEmpty(modalRecord.getStore()) && this.isModal()) {
			modalRecord.applyData(this.record);
			modalRecord.store = store;
		}
	}
});
Ext.namespace('Zarafa.core.data');

Ext.data.Api.actions.open = 'open';

/**
 * @class Zarafa.core.data.MAPIStore
 * @extends Ext.data.GroupingStore
 * @xtype zarafa.mapistore
 *
 * Extension of the Extjs store which adding support for the 'open' command,
 * which is used by MAPI to request additional data for a record.
 */
Zarafa.core.data.MAPIStore = Ext.extend(Ext.data.GroupingStore, {
	/**
	 * @cfg {Boolean} persistentFilter True when the {@link #filter} which
	 * has been applied on this store should be reapplied when the store
	 * has been {@link #load loaded}
	 */
	persistentFilter: true,

	/**
	 * The currently active function which was given to {@link #filterBy}.
	 * @property
	 * @type Function
	 * @private
	 */
	filterFn: undefined,

	/**
	 * The currently active {@link #filterFn function} scope which was given to {@link #filterBy}.
	 * @property
	 * @type Object
	 * @private
	 */
	filterScope: undefined,

	/**
	 * Set to true when the {Zarafa.core.data.MAPIStore} starts saving, set to false when done.
	 * @property
	 * @type Boolean
	 * @private
	 */
	isSaving: false,

	/**
	 * @constructor
	 * @param config Configuration structure
	 */
	constructor: function(config)
	{
		config = config || {};

		Ext.applyIf(config, {
			// Don't automatically update changes to records to the server.
			autoSave: false,

			// When autoSave is false, indicates that CRUD operations are batched into a single request.
			batch: true
		});

		this.addEvents(
			/**
			 * @event open
			 * Fires when the {@link Zarafa.core.data.MAPIStore MAPIStore} gets extra data for a specific
			 * {@link Zarafa.core.data.MAPIRecord MAPIRecord}.
			 * @param {Zarafa.core.data.MAPIStore} store The {@link Zarafa.core.data.MAPIStore MAPIStore} which issues
			 * open request to get extra data for specific record.
			 * @param {Zarafa.core.data.MAPIRecord} record Record which is being opened to get extra information.
			 */
			'open'
		);

		Zarafa.core.data.MAPIStore.superclass.constructor.call(this, config);

		// Update the getKey function inside the MixedCollection to match
		// the one provided getRecordKey.
		this.data.getKey = this.getRecordKey;

		this.initEvents();
	},

	/**
	 * Initialize all events which Zarafa.core.data.MAPIStore MAPIStore} will listen to.
	 * @protected
	 */
	initEvents: function()
	{
		this.on('beforeload', this.onBeforeLoad, this);
		this.on('add', this.onAdd, this);
		this.on('remove', this.onRemove, this);
		this.on('write', this.onWrite, this);

		this.on('beforesave', function(){ this.isSaving = true; }, this);
		this.on('save', function(){ this.isSaving = false; }, this);
	},

	/**
	 * The {@link Ext.util.MixedCollection#getKey} function which must be
	 * applied to the {@link #data}{@link Ext.util.MixedCollection#getKey #getKey}
	 * function. This is assigned by the constructor and allows subclasses to
	 * simply override this function rather then apply it manually to {@link #data}
	 * themselves.
	 * @param {Ext.data.Record} o The record for which the key is requested
	 * @return {String} The key by which the record must be saved into the {@link Ext.util.MixedCollection}.
	 * @protected
	 */
	getRecordKey: Ext.util.MixedCollection.prototype.getKey,

	/**
	 * Check if a particular {@link Zarafa.core.Action action} is being executed
	 * by the {@link #proxy} of this store. When no action is given, this function
	 * will check if the proxy is busy with any action.
	 *
	 * @param {Zarafa.core.Action} action The action which is being checked
	 * @return {Boolean} True if the given action is being executed by the proxy
	 */
	isExecuting: function(action)
	{
		return this.proxy.isExecuting(action);
	},

	/**
	 * Determine if a {@link #isExecuting 'list'} or {@link #isExecuting 'open'}
	 * request is still pending. And if so
	 * {@link Zarafa.core.data.MAPIProxy#cancelRequest cancel} those requests.
	 */
	cancelLoadRequests: function()
	{
		// If we are loading data, we want to cancel
		// the request as we don't want the data anymore.
		if (this.isExecuting('list')) {
			this.proxy.cancelRequests('list');
		}

		// If we are opening the record, we want to cancel
		// the request as we don't want the data anymore.
		if (this.isExecuting('open')) {
			this.proxy.cancelRequests('open');
		}

		// Saving is still interesting as the user might
		// not expect that action to be still pending.
	},

	/**
	 * Get the {@link Date#getTime timestamp} of the last time a response was given
	 * for the given action.
	 * @param {Zarafa.core.Action} action The action which is being checked
	 * @return {Number} The timestamp of the last action time
	 */
	lastExecutionTime: function(action)
	{
		return this.proxy.lastExecutionTime(action);
	},

	/**
	 * <p>Reloads the Record cache from the configured Proxy. See the superclass {@link Ext.data.Store#reload documentation}
	 * for more detaiils.
	 * During reload we add an extra option into the {@link #load} argument which marks the action as a reload
	 * action.
	 */
	reload: function(options)
	{
		options = Ext.applyIf(options || {}, { reload: true });
		Zarafa.core.data.MAPIStore.superclass.reload.call(this, options);
	},

	/**
	 * Saves all pending changes to the store. See the superclass {@link Ext.data.Store#save documentation}
	 * for more details. Where the superclass saves all {@link #removed} and {@link #modified} records,
	 * this function will only save the records which are passed as argument.
	 *
	 * @param {Zarafa.core.data.MAPIRecord/Zarafa.core.data.MAPIRecord[]} records The records which
	 * must be saved to the server.
	 * @return {Number} batch Returns a number to uniquely identify the "batch" of saves occurring. -1 will be returned
	 * if there are no items to save or the save was cancelled.
	 */
	save: function(records) {
		// When no records are provided, fall back to the default behavior of the superclass.
		if (!Ext.isDefined(records)) {
			return Zarafa.core.data.MAPIStore.superclass.save.call(this);
		}

		if (!Array.isArray(records)) {
			records = [ records ];
		}

		if (!this.writer) {
			throw new Ext.data.Store.Error('writer-undefined');
		}

		var destroyed = [],
			created = [],
			updated = [],
			queue = [],
			trans,
			batch,
			data = {};

		for (var i = 0, len = records.length; i < len; i++) {
			var record = records[i];

			if (this.removed.indexOf(record) >= 0) {
				// Check for removed records first, a record located in this.removed is
				// guaranteed to be a non-phantom. See store.remove().
				destroyed.push(record);
			} else if (this.modified.indexOf(record) >= 0) {
				// Only accept valid records.
				if (record.isValid()) {
					if (record.phantom) {
						created.push(record);
					} else {
						updated.push(record);
					}
				}
			}
		}

		if (destroyed.length > 0) {
			queue.push(['destroy', destroyed]);
		}
		if (created.length > 0) {
			queue.push(['create', created]);
		}
		if (updated.length > 0) {
			queue.push(['update', updated]);
		}

		var len = queue.length;
		if(len){
			batch = ++this.batchCounter;
			for(var i = 0; i < len; ++i){
				trans = queue[i];
				data[trans[0]] = trans[1];
			}
			if(this.fireEvent('beforesave', this, data) !== false){
				for(var i = 0; i < len; ++i){
					trans = queue[i];
					this.doTransaction(trans[0], trans[1], batch);
				}
				return batch;
			}
		}
		return -1;
	},

	/**
	 * Event handler which is fired when we are about to (re)load the store.
	 * When this happens we should cancel all pending {@link #open} requests,
	 * as they cannot be completed anymore (the record will have been deleted,
	 * so the opened record has become useless.
	 * @private
	 */
	onBeforeLoad: function()
	{
		if (this.isExecuting('open')) {
			this.proxy.cancelRequests('open');
		}
	},

	/**
	 * Event handler which is raised when a {@link Zarafa.core.data.MAPIRecord MAPIRecord} has been added
	 * to this {@link Zarafa.core.data.MAPIStore MAPIStore}.
	 *
	 * @param {Zarafa.core.data.MAPIStore} store The {@link Zarafa.core.data.MAPIStore MAPIStore} to which the store was added.
	 * @param {Zarafa.core.data.MAPIRecord[]} records The array of {@link Zarafa.core.data.MAPIRecord records} which have been added.
	 * @param {Number} index The index at which the record(s) were added
	 * @private
	 */
	onAdd: function(store, records, index)
	{
		this.setRecordsStore(store, records);
	},

	/**
	 * Event handler which is raised when a {@link Zarafa.core.data.MAPIRecord MAPIRecord} has been removed
	 * from this {@link Zarafa.core.data.MAPIStore MAPIStore}.
	 *
	 * @param {Zarafa.core.data.MAPIStore} store The {@link Zarafa.core.data.MAPIStore MAPIStore} from which the records were removed.
	 * @param {Zarafa.core.data.MAPIRecord[]} records The array of {@link Zarafa.core.data.MAPIRecord records} which have been removed.
	 * @param {Number} index The index at which the record(s) were removed.
	 * @private
	 */
	onRemove: function(store, records, index)
	{
		this.setRecordsStore(undefined, records);
	},

	/**
	 * Event handler which is raised when the {@link #write} event has been fired. This will clear
	 * all {@link Zarafa.core.data.MAPIRecord#actions Message Actions} from the given records.
	 *
	 * @param {Zarafa.core.data.MAPIStore} store The store which fired the event
	 * @param {String} action [Ext.data.Api.actions.create|update|destroy]
	 * @param {Object} result The 'data' picked-out out of the response for convenience
	 * @param {Ext.Direct.Transaction} res The transaction
	 * @param {Record/Record[]} records The records which were written to the server
	 * @private
	 */
	onWrite: function(store, action, result, res, records)
	{
		if (!Array.isArray(records)) {
			records = [ records ];
		}

		for (var i = 0, len = records.length; i < len; i++) {
			records[i].clearMessageActions();
			records[i].clearActionResponse();
		}
	},

	/**
	 * Iterates through all {@link Zarafa.core.data.MAPIRecord records} and sets the
	 * reference to the {@link Zarafa.core.data.MAPIStore MAPIStore} to which it belongs.
	 *
	 * @param {Zarafa.core.data.MAPIStore} store The {@link Zarafa.core.data.MAPIStore MAPIStore} to which the
	 * {@link Zarafa.core.data.MAPIRecord records} must be assigned.
	 * @param {Zarafa.core.data.MAPIRecord[]} records The array of
	 * {@link Zarafa.core.data.MAPIRecord records} which must be updated.
	 * @private
	 */
	setRecordsStore: function(store, records)
	{
		records = Array.isArray(records) ? records : [ records ];
		Ext.each(records, function(record) { record.join(store); }, this);
	},

	/**
	 * Function is used to get extra properties from the server, which are not received in
	 * 'list' action. function will call {@link #execute} event, which is entry point for every
	 * CRUD operation, {@link #execute} will internal call {@link #createCallback} to create a
	 * callback function based on operation type ('open' -> onOpenRecords).
	 * @param {Zarafa.core.data.MAPIRecord[]} records records for which we need extra properties.
	 * @param {Object} options Extra options which can be used for opening the records
	 * @return {Boolean|undefined} false when this.execute fails
	 */
	open: function(records, options)
	{
		try {
			return this.execute('open', records, options);
		} catch (e) {
			this.handleException(e);
			return false;
		}
	},

	/**
	 * Function will work as callback function for 'open' operation, and update the
	 * existing records with the new data that is received from server.
	 * @param {Boolean} success true if operation completed successfully else false.
	 * @param {Zarafa.core.data.MAPIRecord|Zarafa.core.dataMAPIRecord[]} records updated records.
	 * @param {Object|Array} data properties of records which is received from server (in key/value pair).
	 */
	onOpenRecords: function(success, records, data)
	{
		if (success !== true) {
			return;
		}

		if (!Array.isArray(records)) {
			records = [ records ];
		}

		records.forEach(function(record) {
			if (this.indexOf(record) === -1) {
				return;
			}

			// Opening items in batch passes multiple records but only one data object per onOpenRecords
			// call, therefore the entryid has to be compared to update the correct record.
			// Also opening meeting requests passes the task request as record and expects the actual task to
			// be applied onto the record, here the entryids don't match so just check if records length is not 1.
			if (records.length !== 1 && Array.isArray(data) && data.length !== 0 &&
				!Zarafa.core.EntryId.compareEntryIds(record.get('entryid'), data[0].entryid)) {
				return;
			}

			try {
				// call reader to update record data
				var oldRecord = record;

				this.reader.update(record, data);
				record.afterOpen();

				this.fireEvent('open', this, record, oldRecord);
			} catch (e) {
				this.handleException(e);
			}
		}, this);
	},

	/**
	 * Get the Record with the specified id.
	 * If the {@link #reader} has the {@link Ext.data.JsonReader#idProperty} set to 'entryid',
	 * then this function will also use {@link Zarafa.core.EntryId#compareEntryIds}. For
	 * 'store_entryid' then {@link Zarafa.core.EntryId#compareStoreEntryIds} is used.
	 * @param {String} id The id of the Record to find.
	 * @return {Ext.data.Record} The Record with the passed id. Returns undefined if not found.
	 */
	getById: function(id)
	{
		// First use the original implementation
		var item = Zarafa.core.data.MAPIStore.superclass.getById.call(this, id);

		// If no item was found, and the reader uses the 'entryid' property,
		// we should retry searching using the compareEntryIds function. If that
		// fails as well, then the item is really not present.
		if (!item) {
			var index = this.findBy(function(record) { return this.idComparison(id, record.id); }, this);
			if (index >= 0) {
				item = this.getAt(index);
			}
		}

		return item;
	},

	/**
	 * Compare a {@link Ext.data.Record#id ids} to determine if they are equal.
	 * @param {String} a The first id to compare
	 * @param {String} b The second id to compare
	 * @protected
	 */
	idComparison: function(a, b)
	{
		return a === b;
	},

	/**
	 * Filter by a function. Returns a <i>new</i> collection that has been filtered.
	 * The passed function will be called with each object in the collection.
	 * If the function returns true, the value is included otherwise it is filtered.
	 * @param {Function} fn The function to be called, it will receive the args o (the object), k (the key)
	 * @param {Object} scope (optional) The scope (<code>this</code> reference) in which the function is executed. Defaults to this MixedCollection.
	 * @return {MixedCollection} The new filtered collection
	 */
	filterBy: function(fn, scope)
	{
		// Save the function for later usage.
		this.filterFn = fn;
		this.filterScope = scope;

		Zarafa.core.data.MAPIStore.superclass.filterBy.apply(this, arguments);
	},

	/**
	 * Revert to a view of the Record cache with no filtering applied.
	 * @param {Boolean} suppressEvent If <tt>true</tt> the filter is cleared silently without firing the
	 * {@link #datachanged} event.
	 */
	clearFilter: function()
	{
		// Reset the filter
		delete this.filterFn;
		delete this.filterScope;

		Zarafa.core.data.MAPIStore.superclass.clearFilter.apply(this, arguments);
	},

	/**
	 * Callback function which will be called when 'read' action is executed
	 * and {@link Zarafa.core.data.JsonReader JsonReader} has deserialized data
	 * into {@link Zarafa.core.data.MAPIRecord MAPIRecord},
	 * so the records can be added to the {@link Zarafa.core.data.NoSyncStore NoSyncStore}.
	 * @param {Object} o response object containing array of {@link Zarafa.core.data.MAPIRecord MAPIRecord}
	 * and optionally a property indicating total number of records.
	 * @param {Object} options optionally can contain 'add' which will append {@link Zarafa.core.data.MAPIRecord MAPIRecord}
	 * to the existing set of cached {@link Zarafa.core.data.MAPIRecord MAPIRecord}.
	 * @private
	 */
	loadRecords: function(o, options, success)
	{
		Zarafa.core.data.MAPIStore.superclass.loadRecords.apply(this, arguments);

		if (this.persistentFilter === true && !this.isDestroyed && (!options || options.add !== true)) {
			if (this.filterFn) {
				this.filterBy(this.filterFn, this.filterScope);
			}
		}
	},

	/**
	 * Clear all data in the store
	 * @private
	 */
	clearData: function()
	{
		this.data.each(function(rec) {
			rec.destroy();
		});
		Zarafa.core.data.MAPIStore.superclass.clearData.apply(this, arguments);
	},

	/**
	 * Sort the data in the store using the given sort function.
	 * This will call {@link Ext.util.MixedCollection#sort sort} on the
	 * {@link #data} object.
	 * @param {String} direction (optional) 'ASC' or 'DESC'. Defaults to 'ASC'.
	 * @param {Function} fn (optional) Comparison function that defines the sort order. Defaults to sorting by numeric value.
	 */
	sortBy: function(direction, fn)
	{
		this.data.sort(direction, fn);
		if (this.snapshot && this.snapshot != this.data) {
			this.snapshot.sort(direction, fn);
		}
		this.fireEvent('datachanged', this);
	},

	/**
	 * Clears any existing grouping and refreshes the data using the default sort.
	 */
	clearGrouping: function()
	{
		// Only clear grouping when
		// grouping was previously applied
		if (this.groupField) {
			Zarafa.core.data.MAPIStore.superclass.clearGrouping.apply(this, arguments);
		}
	},

	/**
	 * Destroys the store
	 */
	destroy: function()
	{
		// Make sure we cancel all load requests
		// to the server as we are no longer
		// interested in the results.
		if (!this.isDestroyed) {
			this.cancelLoadRequests();
		}

		Zarafa.core.data.MAPIStore.superclass.destroy.apply(this, arguments);
	}
});

Ext.reg('zarafa.mapistore', Zarafa.core.data.MAPIStore);
Ext.namespace('Zarafa.core.data');

/**
 * @class Zarafa.core.data.NoSyncStore
 * @extends Ext.util.Observable
 *
 * The {@link Zarafa.core.data.NoSyncStore NoSyncStore} represents the collection of
 * {@link Ext.data.Record records}. It offers the same interface
 * as {@link Zarafa.core.data.Store Store} without any CRUD operations being
 * send to the server. This implies that the {@link Zarafa.core.data.NoSyncStore NoSyncStore}
 * will only work for working on {@link Ext.data.Record records} offline.
 */
Zarafa.core.data.NoSyncStore = Ext.extend(Ext.util.Observable, {
	/**
	 * The {@link Ext.data.Record Record} constructor as supplied to (or created by) the
	 * {@link Ext.data.DataReader Reader}. Read-only.
	 * <p>If the Reader was constructed by passing in an Array of {@link Ext.data.Field} definition objects,
	 * instead of a Record constructor, it will implicitly create a Record constructor from that Array (see
	 * {@link Ext.data.Record}.{@link Ext.data.Record#create create} for additional details).</p>
	 * <p>This property may be used to create new Records of the type held in this Store
	 * @property recordType
	 * @type Function
	 */
	recordType: undefined,

	/**
	 * A {@link Ext.util.MixedCollection MixedCollection} containing the defined {@link Ext.data.Field Field}s
	 * for the {@link Ext.data.Record Records} stored in this Store. Read-only.
	 * @property fields
	 * @type Ext.util.MixedCollection
	 */
	fields: undefined,

	/**
	 * True if this store is currently sorted by more than one field/direction combination.
	 * @property
	 * @type Boolean
	 */
	hasMultiSort: false,

	/**
	 * Object containing the current sorting information.
	 * @property
	 * @type Object
	 */
	sortToggle: undefined,

	/**
	 * @cfg {String} sortField
	 * (optional) Initial column on which to sort.
	 */
	sortField: undefined,

	/**
	 * @cfg {String} sortDir
	 * (Optional) Initial direction to sort (<code>"ASC"</code> or <code>"DESC"</code>). Defaults to
	 * <code>"ASC"</code>.
	 */
	sortDir: 'ASC',

	/**
	 * @cfg {Object} sortInfo A config object to specify the sort order in the request of a Store's load operation.
	 * Note that for local sorting, the direction property is case-sensitive.
	 */
	sortInfo: undefined,

	/**
	 * @constructor
	 * @param config Configuration structure
	 */
	constructor: function(config)
	{
		config = config || {};

		Ext.apply(this, config);

		// If the recordType is provided, we can obtain the fields.
		if (this.recordType) {
			this.fields = this.recordType.prototype.fields;
		}

		this.addEvents(
			/**
			 * @event add
			 * Fires when Records have been {@link #add}ed to the Store
			 * @param {Store} this
			 * @param {Ext.data.Record[]} records The array of Records added
			 * @param {Number} index The index at which the record(s) were added
			 */
			'add',
			/**
			 * @event remove
			 * Fires when a Record has been {@link #remove}d from the Store
			 * @param {Store} this
			 * @param {Ext.data.Record} record The Record that was removed
			 * @param {Number} index The index at which the record was removed
			 */
			'remove',
			/**
			 * @event update
			 * Fires when a Record has been updated
			 * @param {Store} this
			 * @param {Ext.data.Record} record The Record that was updated
			 * @param {String} operation The update operation being performed. Value may be one of:
			 * <pre><code>
	Ext.data.Record.EDIT
	Ext.data.Record.REJECT
	Ext.data.Record.COMMIT
			 * </code></pre>
			 */
			'update',
			/**
			 * @event clear
			 * Fires when the data cache has been cleared.
			 * @param {Store} this
			 * @param {Record[]} The records that were cleared.
			 */
			'clear'
		);

		this.sortToggle = {};
		if (this.sortField){
			this.setDefaultSort(this.sortField, this.sortDir);
		} else if(this.sortInfo) {
			this.setDefaultSort(this.sortInfo.field, this.sortInfo.direction);
		}

		Zarafa.core.data.NoSyncStore.superclass.constructor.call(this, config);

		this.initEvents();
		this.initData();
	},

	/**
	 * Initialize data structures in which the {@link Ext.data.Record records} are stored.
	 * @private
	 */
	initData: function()
	{
		this.data = new Ext.util.MixedCollection(false);
		this.data.getKey = function(o){
			return o.id;
		};

		this.removed = [];
		this.modified = [];
	},

	/**
	 * Initialize events which can be raised by the {@link Zarafa.core.data.NoSyncStore NoSyncStore}
	 * @private
	 */
	initEvents: function()
	{
		this.on({
			scope: this,
			add: this.createRecords,
			remove: this.destroyRecord,
			clear: this.onClear
		});
	},

	/**
	 * Destroys the store.
	 */
	destroy: function()
	{
		if (!this.isDestroyed) {
			this.clearData();
			this.data = null;
			this.purgeListeners();
			this.isDestroyed = true;
		}
	},

	/**
	 * Add Records to the Store and fires the {@link #add} event.
	 * See also <code>{@link #insert}</code>.
	 * @param {Ext.data.Record[]} records An Array of Ext.data.Record objects
	 * to add to the cache.
	 * @param {Boolean} silent [false] Defaults to <tt>false</tt>.
	 * Set <tt>true</tt> to not fire add event.
	 */
	add: function(records, silent)
	{
		records = [].concat(records);
		if(records.length < 1) {
			return;
		}

		for (var i = 0, len = records.length; i < len; i++) {
			records[i].join(this);
		}

		var index = this.data.length;
		this.data.addAll(records);

		if(this.snapshot){
			this.snapshot.addAll(records);
		}

		if (silent !== true) {
			this.fireEvent('add', this, records, index);
		}
	},

	/**
	 * Remove Records from the Store and fires the {@link #remove} event.
	 * @param {Ext.data.Record/Ext.data.Record[]} record The record object or array of records to remove from the cache.
	 * @param {Boolean} silent [false] Defaults to <tt>false</tt>. Set <tt>true</tt> to not fire remove event.
	 */
	remove: function(record, silent)
	{
		if (Array.isArray(record)) {
			Ext.each(record, function(r){
				this.remove(r, silent);
			}, this);
		}

		var index = this.data.indexOf(record);
		if(this.snapshot){
			this.snapshot.remove(record);
		}
		if (index > -1) {
			record.join(null);

			this.data.removeAt(index);

			if (silent !== true) {
				this.fireEvent('remove', this, record, index);
			}
		}
	},

	/**
	 * Remove a Record from the Store at the specified index. Fires the {@link #remove} event.
	 * @param {Number} index The index of the record to remove.
	 * @param {Boolean} silent [false] Defaults to <tt>false</tt>. Set <tt>true</tt> to not fire remove event.
	 */
	removeAt: function(index, silent)
	{
		this.remove(this.getAt(index), silent);
	},

	/**
	 * Remove all Records from the Store and fires the {@link #clear} event.
	 * @param {Boolean} silent [false] Defaults to <tt>false</tt>. Set <tt>true</tt> to not fire clear event.
	 */
	removeAll: function(silent)
	{
		var items = [];
		this.each(function(rec){
			items.push(rec);
		});

		this.clearData();
		if(this.snapshot){
			this.snapshot.clear();
		}
		this.modified = [];
		this.removed = [];

		if (silent !== true) {
			this.fireEvent('clear', this, items);
		}
	},

	/**
	 * Remove all Records for which the callback returns true
	 * from the Store and fires the {@link #remove} event.
	 *
	 * @param {Function} callback The callback function which is used to determine
	 * if a record must be removed. Function must accept a {@link Ext.data.Record}
	 * as argument.
	 * @param {Object} scope The scope which must be used for the callback function
	 */
	removeIf: function(callback, scope)
	{
		this.each(function(record) {
			if (callback.call(scope || this, record)) {
				this.remove(record);
			}
		}, this);
	},

	/**
	 * Inserts Records into the Store at the given index and fires the {@link #add} event.
	 * See also <code>{@link #add}</code>.
	 * @param {Number} index The start index at which to insert the passed Records.
	 * @param {Ext.data.Record[]} records An Array of Ext.data.Record objects to add to the cache.
	 */
	insert: function(index, records)
	{
		records = [].concat(records);
		for (var i = 0, len = records.length; i < len; i++) {
			this.data.insert(index, records[i]);
			records[i].join(this);
		}

		if(this.snapshot){
			this.snapshot.addAll(records);
		}

		this.fireEvent('add', this, records, index);
	},

	/**
	 * Get the index within the cache of the passed Record.
	 * @param {Ext.data.Record} record The Ext.data.Record object to find.
	 * @return {Number} The index of the passed Record. Returns -1 if not found.
	 */
	indexOf: function(record)
	{
		return this.data.indexOf(record);
	},

	/**
	 * Get the index within the cache of the Record with the passed id.
	 * @param {String} id The id of the Record to find.
	 * @return {Number} The index of the Record. Returns -1 if not found.
	 */
	indexOfId: function(id)
	{
		return this.data.indexOfKey(id);
	},

	/**
	 * Get the Record at the specified index.
	 * @param {Number} index The index of the Record to find.
	 * @return {Ext.data.Record} The Record at the passed index. Returns undefined if not found.
	 */
	getAt: function(index)
	{
		return this.data.itemAt(index);
	},

	/**
	 * Returns a range of Records between specified indices.
	 * @param {Number} startIndex (optional) The starting index (defaults to 0)
	 * @param {Number} endIndex (optional) The ending index (defaults to the last Record in the Store)
	 * @return {Ext.data.Record[]} An array of Records
	 */
	getRange: function(start, end)
	{
		return this.data.getRange(start, end);
	},

	/**
	 * Gets the number of cached records
	 * @return {Number} The number of records
	 */
	getCount: function()
	{
		return this.data.length;
	},

	/**
	 * Gets the total number of records in the dataset.
	 * @return {Number} The number of records
	 */
	getTotalCount: function()
	{
		return this.getCount();
	},

	/**
	 * Calls the specified function for each of the {@link Ext.data.Record Records} in the cache.
	 * @param {Function} fn The function to call. The {@link Ext.data.Record Record} is passed as the first parameter.
	 * Returning <tt>false</tt> aborts and exits the iteration.
	 * @param {Object} scope (optional) The scope (<code>this</code> reference) in which the function is executed.
	 * Defaults to the current {@link Ext.data.Record Record} in the iteration.
	 */
	each: function(fn, scope)
	{
		this.data.each(fn, scope);
	},

	/**
	 * Gets all {@link Ext.data.Record records} modified since the last commit.
	 * <b>Note</b>: deleted records are not included.
	 * See also {@link Ext.data.Record}<tt>{@link Ext.data.Record#markDirty markDirty}.</tt>.
	 * @return {Ext.data.Record[]} An array of {@link Ext.data.Record Records} containing outstanding
	 * modifications. To obtain modified fields within a modified record see
	 *{@link Ext.data.Record}<tt>{@link Ext.data.Record#modified modified}.</tt>.
	 */
	getModifiedRecords: function()
	{
		return this.modified;
	},

	/**
	 * Gets all {@link Ext.data.Record records} removed since the last commit.
	 * @return {Ext.data.Record[]} An array of {@link Ext.data.Record Records} which have
	 * been marked as removed.
	 */
	getRemovedRecords: function()
	{
		return this.removed;
	},

	/**
	 * Called by {@link Ext.data.Record}
	 * @param {Ext.data.Record} The record which has been edited.
	 * @private
	 */
	afterEdit: function(record)
	{
		if(this.modified.indexOf(record) === -1) {
			this.modified.push(record);
		}
		this.fireEvent('update', this, record, Ext.data.Record.EDIT);
	},

	/**
	 * Called by {@link Ext.data.Record}
	 * @param {Ext.data.Record} The record which has been rejected.
	 * @private
	 */
	afterReject: function(record)
	{
		this.modified.remove(record);
		this.fireEvent('update', this, record, Ext.data.Record.REJECT);
	},

	/**
	 * Called by {@link Ext.data.Record}
	 * @param {Ext.data.Record} The record which has been committed.
	 * @private
	 */
	afterCommit: function(record)
	{
		this.modified.remove(record);
		this.fireEvent('update', this, record, Ext.data.Record.COMMIT);
	},

	/**
	 * 'Commit' outstanding changes. Since {@link Zarafa.core.data.NoSyncStore NoSyncStore}
	 * has no commit capability, changes are not actually sent, but are only cleared.
	 */
	commitChanges: function()
	{
		var m = this.modified.slice(0);
		for(var i = 0, len = m.length; i < len; i++){
			var mi = m[i];
			// Committing means unphantoming.
			mi.phantom = false;
			mi.commit();
		}

		this.modified = [];
		this.removed = [];
	},

	/**
	 * Clear the data within this store
	 * @private
	 */
	clearData: function()
	{
		this.data.clear();
	},

	/**
	 * Should not be used directly. Store#add will call this automatically
	 * @param {Object} store
	 * @param {Object} rs
	 * @param {Object} index
	 * @private
	 */
	createRecords: function(store, rs, index)
	{
		for (var i = 0, len = rs.length; i < len; i++) {
			if (rs[i].phantom && rs[i].isValid()) {
				rs[i].markDirty(); // <-- Mark new records dirty
				this.modified.push(rs[i]); // <-- add to modified
			}
		}
	},

	/**
	 * Destroys a record or records. Should not be used directly. It's called by Store#remove automatically
	 * @param {Store} store
	 * @param {Ext.data.Record} record
	 * @param {Number} index
	 * @private
	 */
	destroyRecord: function(store, record, index)
	{
		var modifiedRecord = this.modified.find(function(item){
			return item.get("entryid") === record.get("entryid") && item.phantom === record.phantom;
		});
			 
		if (Ext.isDefined(modifiedRecord)) {
			this.modified.remove(modifiedRecord);
		}
 
		if (!record.phantom) {
			this.removed.push(record);
		}
	},

	/**
	 * Clears all records. Show not be used directly. It's called by Store#removeAll automatically
	 * @param {Store} store
	 * @param {Ext.data.Record/Ext.data/Record[]} records
	 * @private
	 */
	onClear: function(store, records)
	{
		Ext.each(records, function(rec, index) {
			this.destroyRecord(this, rec, index);
		}, this);
	},

	/**
	 * Returns an object describing the current sort state of this Store.
	 * @return {Object} The sort state of the Store. An object with two properties:
	 * field: String The name of the field by which the Records are sorted.
	 * direction: String The sort order, 'ASC' or 'DESC' (case-sensitive).
	 *
	 * Added for grid support with store, grid's store needs sortinfo.
	 *
	 * See <tt>{@link #sortInfo}</tt> for additional details.
	 */
	getSortState: Ext.data.Store.prototype.getSortState,

	/**
	 * Invokes sortData if we have sortInfo to sort on and are not sorting remotely
	 * @private
	 */
	applySort: Ext.data.Store.prototype.applySort,

	/**
	 * Performs the actual sorting of data. This checks to see if we currently have a multi sort or not. It applies
	 * each sorter field/direction pair in turn by building an OR'ed master sorting function and running it against
	 * the full dataset
	 * @private
	 */
	sortData: Ext.data.Store.prototype.sortData,

	/**
	 * Creates and returns a function which sorts an array by the given field and direction
	 * @param {String} field The field to create the sorter for
	 * @param {String} direction The direction to sort by (defaults to "ASC")
	 * @return {Function} A function which sorts by the field/direction combination provided
	 * @private
	 */
	createSortFunction: Ext.data.Store.prototype.createSortFunction,

	/**
	 * Sets the default sort column and order to be used by the next {@link #load} operation.
	 * @param {String} fieldName The name of the field to sort by.
	 * @param {String} dir (optional) The sort order, 'ASC' or 'DESC' (case-sensitive, defaults to <tt>'ASC'</tt>)
	 */
	setDefaultSort: Ext.data.Store.prototype.setDefaultSort,

	/**
	 * Sort the Records.
	 * If remote sorting is used, the sort is performed on the server, and the cache is reloaded. If local
	 * sorting is used, the cache is sorted internally. See also {@link #remoteSort} and {@link #paramNames}.
	 * This function accepts two call signatures - pass in a field name as the first argument to sort on a single
	 * field, or pass in an array of sort configuration objects to sort by multiple fields.
	 * Single sort example:
	 * store.sort('name', 'ASC');
	 * Multi sort example:
	 * store.sort([
	 *  {
	 *   field : 'name',
	 *   direction: 'ASC'
	 *  },
	 *  {
	 *   field : 'salary',
	 *   direction: 'DESC'
	 *  }
	 * ], 'ASC');
	 * In this second form, the sort configs are applied in order, with later sorters sorting within earlier sorters' results.
	 * For example, if two records with the same name are present they will also be sorted by salary if given the sort configs
	 * above. Any number of sort configs can be added.
	 * @param {String/Array} fieldName The name of the field to sort by, or an array of ordered sort configs
	 * @param {String} dir (optional) The sort order, 'ASC' or 'DESC' (case-sensitive, defaults to <tt>'ASC'</tt>)
	 */
	sort: Ext.data.Store.prototype.sort,

	/**
	 * Sorts the store contents by a single field and direction. This is called internally by {@link sort} and would
	 * not usually be called manually
	 * @param {String} fieldName The name of the field to sort by.
	 * @param {String} dir (optional) The sort order, 'ASC' or 'DESC' (case-sensitive, defaults to <tt>'ASC'</tt>)
	 */
	singleSort: Ext.data.Store.prototype.singleSort,

	/**
	 * Sorts the contents of this store by multiple field/direction sorters. This is called internally by {@link sort}
	 * and would not usually be called manually.
	 * Multi sorting only currently applies to local datasets - multiple sort data is not currently sent to a proxy
	 * if remoteSort is used.
	 * @param {Array} sorters Array of sorter objects (field and direction)
	 * @param {String} direction Overall direction to sort the ordered results by (defaults to "ASC")
	 */
	multiSort: Ext.data.Store.prototype.multiSort,

	/**
	 * Sums the value of property for each record between start and end and returns the result
	 * @param {String} property A field in each record
	 * @param {Number} start (optional) The record index to start at (defaults to 0)
	 * @param {Number} end (optional) The last record index to include (defaults to length - 1)
	 * @return {Number} The sum
	 */
	sum: Ext.data.Store.prototype.sum,

	/**
	 * Returns a filter function used to test a the given property's value. Defers most of the work to
	 * Ext.util.MixedCollection's createValueMatcher function
	 * @param {String} property The property to create the filter function for
	 * @param {String/RegExp} value The string/regex to compare the property value to
	 * @param {Boolean} anyMatch True if we don't care if the filter value is not the full value (defaults to false)
	 * @param {Boolean} caseSensitive True to create a case-sensitive regex (defaults to false)
	 * @param {Boolean} exactMatch True to force exact match (^ and $ characters added to the regex). Defaults to false. Ignored if anyMatch is true.
	 * @private
	 */
	createFilterFn: Ext.data.Store.prototype.createFilterFn,

	/**
	 * Filters an array of elements to only include matches of a simple selector (e.g. div.some-class or span:first-child)
	 * @param {Array} el An array of elements to filter
	 * @param {String} selector The simple selector to test
	 * @param {Boolean} nonMatches If true, it returns the elements that DON'T match
	 * the selector instead of the ones that match
	 * @return {Array} An Array of DOM elements which match the selector. If there are
	 * no matches, and empty Array is returned.
	 */
	filter: Ext.data.Store.prototype.filter,

	/**
	 * Filter by a function. Returns a <i>new</i> collection that has been filtered.
	 * The passed function will be called with each object in the collection.
	 * If the function returns true, the value is included otherwise it is filtered.
	 * @param {Function} fn The function to be called, it will receive the args o (the object), k (the key)
	 * @param {Object} scope (optional) The scope (<code>this</code> reference) in which the function is executed. Defaults to this MixedCollection.
	 * @return {MixedCollection} The new filtered collection
	 */
	filterBy: Ext.data.Store.prototype.filterBy,

	/**
	 * Query the records by a specified property.
	 * @param {String} field A field on your records
	 * @param {String/RegExp} value Either a string that the field
	 * should begin with, or a RegExp to test against the field.
	 * @param {Boolean} anyMatch (optional) True to match any part not just the beginning
	 * @param {Boolean} caseSensitive (optional) True for case sensitive comparison
	 * @return {MixedCollection} Returns an Ext.util.MixedCollection of the matched records
	 */
	query: Ext.data.Store.prototype.query,

	/**
	 * Query the cached records in this Store using a filtering function. The specified function
	 * will be called with each record in this Store. If the function returns <tt>true</tt> the record is
	 * included in the results.
	 * @param {Function} fn The function to be called. It will be passed the following parameters:<ul>
	 * <li><b>record</b>: Ext.data.Record<p class="sub-desc">The {@link Ext.data.Record record}
	 * to test for filtering. Access field values using {@link Ext.data.Record#get}.</p></li>
	 * <li><b>id</b>: Object<p class="sub-desc">The ID of the Record passed.</p></li>
	 * </ul>
	 * @param {Object} scope (optional) The scope (<code>this</code> reference) in which the function is executed. Defaults to this Store.
	 * @return {MixedCollection} Returns an Ext.util.MixedCollection of the matched records
	 **/
	queryBy: Ext.data.Store.prototype.queryBy,

	/**
	 * Finds the index of the first matching Record in this store by a specific field value.
	 * @param {String} fieldName The name of the Record field to test.
	 * @param {String/RegExp} value Either a string that the field value
	 * should begin with, or a RegExp to test against the field.
	 * @param {Number} startIndex (optional) The index to start searching at
	 * @param {Boolean} anyMatch (optional) True to match any part of the string, not just the beginning
	 * @param {Boolean} caseSensitive (optional) True for case sensitive comparison
	 * @return {Number} The matched index or -1
	 */
	find: Ext.data.Store.prototype.find,

	/**
	 * Finds the index of the first matching Record in this store by a specific field value.
	 * @param {String} fieldName The name of the Record field to test.
	 * @param {Mixed} value The value to match the field against.
	 * @param {Number} startIndex (optional) The index to start searching at
	 * @return {Number} The matched index or -1
	 */
	findExact: Ext.data.Store.prototype.findExact,

	/**
	 * Find the index of the first matching Record in this Store by a function.
	 * If the function returns <tt>true</tt> it is considered a match.
	 * @param {Function} fn The function to be called. It will be passed the following parameters:<ul>
	 * <li><b>record</b>: Ext.data.Record<p class="sub-desc">The {@link Ext.data.Record record}
	 * to test for filtering. Access field values using {@link Ext.data.Record#get}.</p></li>
	 * <li><b>id</b>: Object<p class="sub-desc">The ID of the Record passed.</p></li>
	 * </ul>
	 * @param {Object} scope (optional) The scope (<code>this</code> reference) in which the function is executed. Defaults to this Store.
	 * @param {Number} startIndex (optional) The index to start searching at
	 * @return {Number} The matched index or -1
	 */
	findBy: Ext.data.Store.prototype.findBy,

	/**
	 * Get the Record with the specified id.
	 * @param {String} id The id of the Record to find.
	 * @return {Ext.data.Record} The Record with the passed id. Returns undefined if not found.
	 */
	getById: Ext.data.Store.prototype.getById,

	/**
	 * Revert to a view of the Record cache with no filtering applied.
	 * @param {Boolean} suppressEvent If <tt>true</tt> the filter is cleared silently without firing the
	 * {@link #datachanged} event.
	 */
	clearFilter: Ext.data.Store.prototype.clearFilter,

	/**
	 * Returns true if this store is currently filtered
	 * @return {Boolean}
	 */
	isFiltered: Ext.data.Store.prototype.isFiltered,

	/**
	 * Collects unique values for a particular dataIndex from this store.
	 * @param {String} dataIndex The property to collect
	 * @param {Boolean} allowNull (optional) Pass true to allow null, undefined or empty string values
	 * @param {Boolean} bypassFilter (optional) Pass true to collect from all records, even ones which are filtered
	 * @return {Array} An array of the unique values
	 **/
	collect: Ext.data.Store.prototype.collect,

	/**
	 * When store's reader provides new metadata (fields) this function is called.
	 * @param {Object} meta The JSON metadata
	 * @private
	 */
	onMetaChange: Ext.data.Store.prototype.onMetaChange
});
Ext.namespace('Zarafa.core.data');

/**
 * @class Zarafa.core.data.NotificationResolver
 * @extends Ext.util.Observable
 *
 * The NotificationResolver is used when the {@link Zarafa.core.ResponseRouter ResponseRouter}
 * finds Responses which were not a response to a direct Request. This is usually the case,
 * when the PHP-side generated a Notification to update particular UI components based on
 * a server-side change.
 *
 * This resolver will read the response to determine which
 * {@link Zarafa.core.data.AbstractNotificationResponseHandler ResponseHandler} will
 * be needed to handle the response correctly.
 */
Zarafa.core.data.NotificationResolver = Ext.extend(Ext.util.Observable, {
	/**
	 * @cfg {Array} IPFNotificationModules The IPFNotificationModules contains the IPF notification
	 * module names.
	 */
	IPFNotificationModules: ['hierarchynotifier', 'newmailnotifier', 'addressbooknotifier', 'newtodotasknotifier'],

	/**
	 * @constructor
	 * @param {Object} config Configuration object.
	 */
	constructor: function(config)
	{
		config = config || {};

		Ext.apply(this, config);

		Zarafa.core.data.NotificationResolver.superclass.constructor.call(this, config);
	},

	/**
	 * Function which used to add the IPF notification module name to {@link #IPFNotificationModules}.
	 * function mainly used by the external as well as internal plugins when plugins required to
	 * implement server side notification.
	 *
	 * @param {String} module The module is name of IPF notification module.
	 */
	addIPFNotificationModule: function(module)
	{
		this.IPFNotificationModules.push(module);
	},

	/**
	 * Obtain the {@link Zarafa.core.data.AbstractNotificationResponseHandler ResponseHandler}
	 * which can be used for handling the given response in a correct way. This will look into
	 * the response data to determine what kind of Notifications are in there, and will look
	 * up the most suitable {@link Zarafa.core.data.AbstractNotificationResponseHandler ResponseHandler}.
	 *
	 * @param {String} moduleName The Module which generated the notification.
	 * @param {Object} response The Json response data for which the responseHandler is needed
	 * @return {Zarafa.core.data.AbstractResponseHandler} The response handler
	 * which is suitable for handling the given response object.
	 */
	getHandlerForResponse: function(moduleName, response)
	{
		var handlers;

		if (!Ext.isObject(response)) {
			return null;
		}

		if (this.IPFNotificationModules.indexOf(moduleName) !== -1) {
			handlers = this.getHandlersForIPFResponse(response);
		} else {
			handlers = this.getHandlersForIPMResponse(response);
		}

		if (Array.isArray(handlers)) {
			if (handlers.length > 1) {
				return new Zarafa.core.data.CompositeResponseHandler({
					handlers: handlers
				});
			} else {
				return handlers[0];
			}
		} else {
			return handlers;
		}
	},

	/**
	 * Helper function for {@link #getHandlerForResponse}, this will construct
	 * the {@link Zarafa.hierarchy.data.HierarchyNotificationResponseHandler Response Handlers}
	 * for the {@link Zarafa.core.data.IPFStore IPFStores}.
	 *
	 * @param {Object} response The Json response data for which the responseHandler is needed
	 * @return {Zarafa.core.data.AbstractResponseHandler[]} The response handlers
	 * which are suitable for handling the given response object.
	 */
	getHandlersForIPFResponse: function(response)
	{
		var folderParents = [];
		var folderStores = [];

		// get handlers for folder's notifications.
		if(response['folders'] || response['newmail']) {
			var folders = response['folders'];
			if (Ext.isDefined(folders) && !Ext.isEmpty(folders['item'])) {
				folderParents = folderParents.concat(Ext.pluck(folders['item'], 'store_entryid'));
			}

			folderStores = Zarafa.core.data.IPFStoreMgr.getStoresForStores(folderParents);

			var newMail = response['newmail'];
			if (Ext.isDefined(newMail)) {
				folderStores.push(container.getHierarchyStore());
			}

			if (Array.isArray(folderStores)) {
				var responseHandlers = [];
				for (var i = 0, len = folderStores.length; i < len; i++) {
					responseHandlers.push(new Zarafa.hierarchy.data.HierarchyNotificationResponseHandler({
						store: folderStores[i],
						reader: folderStores[i].reader,
						notifyObject: folderStores[i]
					}));
				}

				return responseHandlers;
			}
		} else if (response['stores']) {
			// get handlers for stores's notifications.
			var hierarchyStore = container.getHierarchyStore();
			return new Zarafa.hierarchy.data.HierarchyNotificationResponseHandler({
				store: hierarchyStore,
				reader: hierarchyStore.reader,
				notifyObject: hierarchyStore
			});
		} else if(response['addressbook']) {
			var addressBookStore = Zarafa.addressbook.AddressBookHierarchyStore;
			return new Zarafa.addressbook.AddressBookHierarchyNotificationResponseHandler({
				store: addressBookStore,
				notifyObject: addressBookStore
			});
		} else if(response['newtodotask']) {
			var taskStore = container.getContextByName('task').getModel().getStore();
			return new Zarafa.task.data.TodoTaskListNotificationResponseHandler({
				store: taskStore,
				notifyObject: taskStore
			});
		}
	},

	/**
	 * Helper function for {@link #getHandlerForResponse}, this will construct
	 * the {@link Zarafa.core.data.IPMNotificationResponseHandler Response Handlers}
	 * for the {@link Zarafa.core.data.IPMStore IPMStores}.
	 *
	 * @param {Object} response The Json response data for which the responseHandler is needed
	 * @return {Zarafa.core.data.AbstractResponseHandler[]} The response handlers
	 * which are suitable for handling the given response object.
	 */
	getHandlersForIPMResponse: function(response)
	{
		var messageParents = [];
		var messageStores;

		var creates = response['newobject'];
		if (Ext.isDefined(creates) && Array.isArray(creates.item)) {
			messageParents = messageParents.concat(Ext.pluck(creates.item, 'entryid'));
		}

		var updates = response['update'];
		if (Ext.isDefined(updates) && Array.isArray(updates.item)) {
			messageParents = messageParents.concat(Ext.pluck(updates.item, 'parent_entryid'));
		}

		var deletes = response['delete'];
		if (Ext.isDefined(deletes) && Array.isArray(deletes.item)) {
			messageParents = messageParents.concat(Ext.pluck(deletes.item, 'parent_entryid'));
		}

		messageStores = Zarafa.core.data.IPMStoreMgr.getStoresForFolders(messageParents);

		if (Array.isArray(messageStores)) {
			var responseHandlers = [];
			for (var i = 0, len = messageStores.length; i < len; i++) {
				responseHandlers.push(new Zarafa.core.data.IPMNotificationResponseHandler({
					store: messageStores[i],
					reader: messageStores[i].reader,
					notifyObject: messageStores[i]
				}));
			}

			return responseHandlers;
		} else {
			return new Zarafa.core.data.IPMNotificationResponseHandler({
				store: messageStores,
				reader: messageStores.reader,
				notifyObject: messageStores
			});
		}
	}
});
Ext.namespace('Zarafa.core.data');

/**
 * @class Zarafa.core.data.Notifications
 * List of valid notification types.
 * @singleton
 */
Zarafa.core.data.Notifications =
{
	/**
	 * The 'New Mail' notification to indicate a new Mail has
	 * been delivered to the user.
	 * @property
	 * @type String
	 */
	newMail: 'newMail',

	/**
	 * The 'Object Created' notification to indicate that a new
	 * object has been added to the {@link Zarafa.core.MAPIStore Store}
	 * or {@link Zarafa.hierarchy.data.MAPIFolderRecord Folder}.
	 * @property
	 * @type String
	 */
	objectCreated: 'objectCreated',

	/**
	 * The 'Object Deleted' notification to indicate that an
	 * object has been deleted from the {@link Zarafa.core.MAPIStore Store}
	 * or {@link Zarafa.hierarchy.data.MAPIFolderRecord Folder}.
	 * @property
	 * @type String
	 */
	objectDeleted: 'objectDeleted',

	/**
	 * The 'Object Modified' notification to indicate that an
	 * object has been modified in the {@link Zarafa.core.MAPIStore Store}
	 * or {@link Zarafa.hierarchy.data.MAPIFolderRecord Folder}.
	 * @property
	 * @type String
	 */
	objectModified: 'objectModified'
};
Ext.namespace('Zarafa.core.data');

/**
 * @class Zarafa.core.data.PresenceCache
 * @extends Object
 *
 * Has functions to handle presence of users
 *
 * @singleton
 */
Zarafa.core.data.PresenceCache = Ext.extend(Object, {
	/**
	 * Holds the cached statuses for users. It is an object that
	 * contain an entryid to Object mapping. The object contains the {@link
	 * Zarafa.core.data.UserIdObject user}, the statusByPlugin for the
	 * user, and a squashed {@link Zarafa.core.data.PresenceStatus status}.
	 * @property
	 * @private
	 */
	cache: {},

	/**
	 * Returns the entry of the user in the {#cache presence cache} for the passed user, or
	 * undefined if the user is not found.
	 * @param {Zarafa.core.data.UserIdObject} user The user for which the entry in the
	 * {#cache presence cache} will be returned.
	 * @param {Boolean} noSync If set to true then the given user will not be used to
	 * update the cache. Note: this is a very ugly way to prevent infinite recursion
	 * when this function is called from {#syncUsers}
	 * @return {Object|undefined}
	 * @private
	 */
	getUser: function(user, noSync) {
		if (!user.hasOwnProperty('entryid')) {
			return;
		}

		// Check if we can update the cached user data
		if (user.entryid in this.cache) {
			// (IPMRecipientStores might change recipients from SMTP to
			// ZARAFA after the names have been resolved)
			if ( noSync !== true ){
				this.syncUsers([user]);
			}
			return this.cache[user.entryid];
		}
	},

	/**
	 * Synchronizes the given {@link Zarafa.core.data.UserIdObject users} with the cache.
	 * @param {Zarafa.core.data.UserIdObject[]} users The users that need to be synced
	 * with the cached users.
	 * @return {Zarafa.core.data.UserIdObject[]}
	 */
	syncUsers: function(users) {
		Ext.each( users, function(user, index){
			var syncedUser = this.getUser(user, true);
			if (Ext.isDefined(syncedUser)) {
				// Check if we can update the cache
				this.cache[user.entryid].user.syncWithUser(user);
				syncedUser = this.cache[user.entryid].user;
			} else {
				// Add the new user to the cache, so it will be updated during polling
				this.cache[user.entryid] = {
					user: user,
					statusByPlugin: {},
					status: Zarafa.core.data.PresenceStatus.UNKNOWN
				};
				syncedUser = this.cache[user.entryid].user;
			}

			// Update the return value
			users[index] = syncedUser;
		}, this);

		return users;
	},

	/**
	 * Returns the cached presence status for the given user or
	 * undefined if the user was not found in the cache.
	 * @param {Zarafa.core.data.UserIdObject} userInfo The user for which
	 * a {@link Zarafa.core.data.PresenceStatus presence status} is
	 * requested.
	 * @return {Zarafa.core.data.PresenceStatus|undefined}
	 */
	getStatusForUser: function(userInfo) {
		// Distlists don't have userInfos but will send null
		if ( !userInfo ){
			return Zarafa.core.data.PresenceStatus.UNKNOWN;
		}

		var user = this.getUser(userInfo);
		return user && user.status;
	},

	/**
	 * Adds the status for the given user for the given plugin to the cache.
	 * @param {String} pluginName Name of the plugin for which a status is added/updated
	 * @param {Zarafa.core.data.UserIdObject} user The user for which the status is added
	 * @param {Zarafa.core.data.PresenceStatus} status The status for the given user according to the
	 * given plugin
	 */
	addStatusForUser: function(pluginName, user, status){
		// First see if we already have a cached status for this user
		var cacheUser = this.getUser(user);

		// Add the status of the user for this plugin
		if ( Ext.isDefined(cacheUser) ){
			cacheUser.statusByPlugin[pluginName] = status;
		} else {
			var statusByPlugin = {};
			statusByPlugin[pluginName] = status;
			this.cache[user.entryid] = {
				user: user,
				statusByPlugin: statusByPlugin
			};
		}

		// Add a squashed status
		var presenceUser = this.cache[user.entryid];
		presenceUser.status = Zarafa.core.data.PresenceStatus.UNKNOWN;
		Ext.iterate(presenceUser.statusByPlugin, function(pluginName){
			if ( presenceUser.statusByPlugin[pluginName] > presenceUser.status ){
				presenceUser.status = presenceUser.statusByPlugin[pluginName];
			}
		}, this);
	},

	/**
	 * Returns a list of {Zarafa.core.data.UserIdObject} objects in the cache.
	 * @return {Array} of {@link Zarafa.core.data.UserIdObject UserIdObjects}
	 */
	getUserInfoList: function() {
		return Object.keys(this).map(function(key) {
			return this.cache[key].user;
		}, this);
	}
});

// Make a singleton of this class
Zarafa.core.data.PresenceCache = new Zarafa.core.data.PresenceCache();
Ext.namespace('Zarafa.core.data');

/**
 * @class Zarafa.core.data.Record
 * @singleton
 */
Zarafa.core.data.Record = {
	/**
	 * Create subclass of a {@link Ext.data.Record record} using the
	 * default list of fields.
	 * @param {Object} fields The fields which must be added to the subclass.
	 * @param {Type} base The base type from which the subclass must be derived.
	 * @return {Object} The type of the subclass.
	 */
	create: function(fields, base)
	{
		var subclass = Ext.extend(base || Ext.data.Record, {});
		var proto = subclass.prototype;

		proto.fields = new Ext.util.MixedCollection(false, function(field) {
			return field.name;
		});

		this.addFields(proto, fields || []);

		subclass.getField = function(name)
		{
			return proto.fields.get(name);
		};

		return subclass;
	},

	/**
	 * Add fields to an object
	 * @param {Object} proto The object to update the fields to
	 * @param {Object} fields The array of fields which must be added
	 * @private
	 */
	addFields: function(proto, fields)
	{
		for(var i = 0, len = fields.length; i < len; i++) {
			if (Array.isArray(fields[i])) {
				this.addFields(proto, fields[i]);
			} else {
				proto.fields.add(new Ext.data.Field(fields[i]));
			}
		}
	}
};
Ext.namespace('Zarafa.core.data');

/**
 * @class Zarafa.core.data.RecordDefinition
 * @extends Ext.util.Observable
 * The definition for a {@link Ext.data.Record record}
 * for a particular message class. Each message class can have its
 * own set of fields and default values. Within this class, the
 * specific values are kept. By default all fields and default values
 * are inherited from the parent definition. So each definition only
 * contains the data which is new for this particular message class.
 *
 * NOTE: This class should never be used outside of the
 * {@link Zarafa.core.data.RecordFactory RecordFactory}.
 */
Zarafa.core.data.RecordDefinition = Ext.extend(Ext.util.Observable, {
	/**
	 * @cfg {Zarafa.core.data.RecordDefinition} parent The parent record definition for this definition.
	 */
	parent: undefined,

	/**
	 * The object class definition for the record.
	 * This type is always a subclass of the {@link Zarafa.core.data.RecordDefinition.base base},
	 * field within this {@link Zarafa.core.data.RecordDefinition definition}.
	 * @property
	 * @type Class
	 * @private
	 */
	type: undefined,

	/**
	 * The configuration object in where all options for the Record instances are configured.
	 *
	 * This is editable as long as no record has been created using this definition yet, after
	 * that this property can no longer be used.
	 * @property
	 * @type Object
	 * @private
	 */
	cfg: undefined,

	/**
	 * @cfg {Object} base The object class definition which must be used as base for the record.
	 * In most cases this base must be extended from {@link Ext.data.Record record}.
	 *
	 * This is applied on the {@link #cfg} object and can be changed as long as no record has been
	 * created using this definition yet, after that this property can no longer be used.
	 */
	base: undefined,

	/**
	 * @cfg {Object} subStores The key-value array where for each subStore type is provided with the type
	 * of the {@link Zarafa.core.data.SubStore SubStore} which must be used for the given subStore.
	 *
	 * This is applied on the {@link #cfg} object and can be changed as long as no record has been
	 * created using this definition yet, after that this property can no longer be used.
	 */
	subStores: undefined,

	/**
	 * @cfg {Ext.data.Field[]} fields The array of fields which belong to this {@link Ext.data.Record record}.
	 * This is mapped on the server to MAPI properties which are stored in the zarafa-server.
	 *
	 * This is applied on the {@link #cfg} object and can be changed as long as no record has been
	 * created using this definition yet, after that this property can no longer be used.
	 */
	fields: undefined,

	/**
	 * @cfg {Object} defaults The key-value array where the default values for particular fields is provided.
	 * When a new {@link Ext.data.Record record} is created (one that does
	 * not yet exist on the server) all default values will be applied to the new
	 * {@link Ext.data.Record record}.
	 *
	 * This is applied on the {@link #cfg} object and can be changed as long as no record has been
	 * created using this definition yet, after that this property can no longer be used.
	 */
	defaults: undefined,

	/**
	 * @cfg {Object} createDefaults The key-value array for default values that should be applied
	 * when a record is created (either a phantom or non-phantom).
	 *
	 * This is applied on the {@link #cfg} object and can be changed as long as no record has been
	 * created using this definition yet, after that this property can no longer be used.
	 */
	createDefaults: undefined,

	/**
	 * @constructor
	 * @param {Object} config Configuration object
	 */
	constructor: function(config)
	{
		this.addEvents([
			/**
			 * @event createphantom
			 * Fires when the {@link Zarafa.core.data.RecordFactory RecordFactory} has
			 * created a new phantom record.
			 * @param {Ext.data.Record} record The created phantom record
			 * @param {Object} data The data used to initialize the record
			 */
			'createphantom',
			/**
			 * @event createrecord
			 * Fires when the {@link Zarafa.core.data.RecordFactory RecordFactory} has
			 * created a new record.
			 * @param {Ext.data.Record} record The created record
			 * @param {Object} data The data used to initialize the record
			 */
			'createrecord'
		]);

		// Only the parent is applied on the current
		// class. the rest can be applied to the cfg object.
		this.parent = config.parent;
		delete config.parent;

		this.cfg = Ext.apply({}, config);

		Zarafa.core.data.RecordDefinition.superclass.constructor.call(this, config);
	},

	/**
	 * Create a new {@link Ext.data.Record record} object
	 * based on this {@link Zarafa.core.data.RecordDefinition definition}.
	 *
	 * The data object used to instantiate the new Record will be created using the
	 * default values from {@link Ext.data.Field field} definitions as obtained by
	 * {@link #getFieldDefaultValues}, the default values configured through
	 * the {@link Zarafa.core.data.RecordFactory Record Factory} which are obtained
	 * by the {@link #getDefaultValues} function. And finally the data passed as
	 * argument will be applied. This ordering implies that the {@link Ext.data.Field field}
	 * default values are most likely to be overridden by the other values.
	 *
	 * @param {Object} data (Optional) An object, the properties of which provide
	 * values for the new Record's fields. If not specified the {@link Ext.data.Field.defaultValue}
	 * for each field will be assigned.
	 * @param {Object} id (Optional) The id of the Record. The id is used by the
	 * {@link Ext.data.Store Store} object which owns the {@link Ext.data.Record}
	 * to index its collection of Records (therefore this id should be unique within
	 * each store). If an id is not specified a phantom Record will be created with an
	 * automatically generated id.
	 * @return {Ext.data.Record} the new record which was created
	 */
	createRecord: function(data, id)
	{
		var RecordClass = this.getType();
		var applyData = {};
		Ext.apply(applyData, this.getFieldDefaultValues());
		if (!Ext.isDefined(id)) {
			Ext.apply(applyData, this.getDefaultValues());
		}
		if (data) {
			Ext.apply(applyData, data);
		}

		// Apply the createDefaults object to initialize the record with.
		var record = new RecordClass(applyData, id, this);

		if (record.phantom) {
			this.fireEvent('createphantom', record, data);
		} else {
			// field default values will be applied by the jsonreader
			this.fireEvent('createrecord', record, data);
		}

		// All changes which were made should be committed,
		// as only after this function returns should the changes
		// for the record be kept.
		record.commit();

		return record;
	},

	/**
	 * Obtain the {@link #type object class} definition which can be used for constructing
	 * new {@link Ext.data.Record records}. The type will be a subclass of
	 * {@link Zarafa.core.data.RecordDefinition#base base}.
	 * @return {Class} The object class definition for this record definition
	 */
	getType: function()
	{
		if (!Ext.isDefined(this.type)) {
			var baseClass = this.getBaseClass();
			var fields = this.getFields();
			this.type = Zarafa.core.data.Record.create(fields, baseClass);
		}
		return this.type;
	},

	/**
	 * Set the {@link #base} class for the {@link Zarafa.core.data.RecordDefinition definition}.
	 *
	 * NOTE: If {@link #createRecord} has been called, then changes made by this function will
	 * no longer have effect.
	 *
	 * @param {Class} baseClass The base class to set.
	 */
	setBaseClass: function(baseClass)
	{
		this.cfg.base = baseClass;
	},

	/**
	 * Obtain the {@link #base} class which must be used when creating a new record
	 * from this {@link Zarafa.core.data.RecordDefinition definition}. If
	 * no base class is yet provided, this function will call recursively
	 * into its {@link #parent parents} until a valid base class has been found.
	 * @return {Class} The base class to be used for creating a new record.
	 * @private
	 */
	getBaseClass: function()
	{
		if (!Ext.isDefined(this.base)) {
			if (this.cfg.base) {
				this.base = this.cfg.base;
			} else if (Ext.isDefined(this.parent)) {
				this.base = this.parent.getBaseClass();
			} else {
				this.base = Ext.data.Record;
			}
		}
		return this.base;
	},

	/**
	 * Set the type which must be used for a particular SubStore.
	 * When a SubStore type has been set on a RecordType, then the
	 * subStore is considered 'supported' by the Record.
	 *
	 * NOTE: If {@link #createRecord} has been called, then changes made by this function will
	 * no longer have effect.
	 *
	 * @param {String} name The name of the SubStore (This matches the name how the
	 * data for this subStore is send through Json).
	 * @param {Type} type The Object type which must be used to allocate the subStore
	 */
	setSubStore: function(name, type)
	{
		this.cfg.subStores = Ext.value(this.cfg.subStores, {});
		this.cfg.subStores[name] = type;
	},

	/**
	 * Obtain the key-value array of subStore types which are active for
	 * this {@link Zarafa.core.data.RecordDefinition definition}. This object
	 * is used to {@link Zarafa.core.data.MAPIRecord#setSubStoreTypes set} on
	 * the {@link Zarafa.core.data.MAPIRecord MAPIRecord} which is created.
	 * @return {Object} key-value array of all subStore Types
	 * @private
	 */
	getSubStores: function()
	{
		if (!this.subStores) {
			this.subStores = {};
			if (Ext.isDefined(this.parent)) {
				Ext.apply(this.subStores, this.parent.getSubStores());
			}
			Ext.apply(this.subStores, this.cfg.subStores);
		}

		return Ext.apply({}, this.subStores);
	},

	/**
	 * Add a {@link Ext.data.Field field} to the {@link Zarafa.core.data.RecordDefinition definition}.
	 *
	 * NOTE: If {@link #createRecord} has been called, then changes made by this function will
	 * no longer have effect.
	 *
	 * @param {Ext.data.Field} field The field to add.
	 */
	addField: function(field)
	{
		this.cfg.fields = Ext.value(this.cfg.fields, []);

		if (!Array.isArray(field)) {
			this.cfg.fields.push(field);
		} else {
			this.cfg.fields = this.cfg.fields.concat(field);
		}
	},

	/**
	 * Obtain the list of {@link Ext.data.Field fields} which are active
	 * for this {@link Zarafa.core.data.RecordDefinition definition}.
	 * These fields must be used to create a record, to indicate which
	 * {@link Ext.data.Field fields} are allowed.
	 * This function will recursively call into its parents to obtain
	 * all fields from the parents.
	 *
	 * @return {Ext.data.Field[]} All fields which are active for
	 * this definition.
	 * @private
	 */
	getFields: function()
	{
		if (!this.fields) {
			this.fields = [];

			if (Ext.isDefined(this.parent)) {
				this.fields = this.fields.concat(this.parent.getFields());
			}
			if (this.cfg.fields) {
				this.fields = this.fields.concat(this.cfg.fields);
			}
		}

		return this.fields;
	},

	/**
	 * Add a default value for a {@link Ext.data.Field field} to the
	 * {@link Zarafa.core.data.RecordDefinition definition}.
	 *
	 * NOTE: If {@link #createRecord} has been called, then changes made by this function will
	 * no longer have effect.
	 *
	 * @param {Ext.data.Field} field The field for which the default value applies
	 * @param {Mixed} value The default value for the field
	 */
	addDefaultValue: function(field, value)
	{
		this.cfg.defaults = Ext.value(this.cfg.defaults, {});
		var name = Ext.isString(field) ? field : field.name;
		this.cfg.defaults[name] = value;
	},

	/**
	 * Apply all default values for the {@link Ext.data.Field field}
	 * which have been set for the {@link Zarafa.core.data.RecordDefinition definition}.
	 *
	 * This function will recursively call into its parents to apply
	 * all default values from the parents.
	 * @private
	 */
	getDefaultValues: function()
	{
		if (!this.defaults) {
			this.defaults = {};

			if (Ext.isDefined(this.parent)) {
				Ext.apply(this.defaults, this.parent.getDefaultValues());
			}

			if (this.cfg.defaults) {
				Ext.apply(this.defaults, this.cfg.defaults);
			}
		}

		return this.defaults;
	},

	/**
	 * Apply all default values from the {@link Ext.data.Field field}
	 * definitions. These are applied by the {@link Ext.data.DataReader DataReader}
	 * when reading {@link Ext.data.Record records} from the server, but not for
	 * {@link Ext.data.Record#phantom phantom} records.
	 * @private
	 */
	getFieldDefaultValues: function()
	{
		if (!this.fieldDefaults) {
			var fields = this.getFields();

			this.fieldDefaults = {};
			for (var i = 0, len = fields.length; i < len; i++) {
				var field = fields[i];
				var value = field.defaultValue;
				if (!Ext.isDefined(value)) {
					value = Ext.data.Field.prototype.defaultValue;
				}

				this.fieldDefaults[field.name] = value;
			}
			Ext.apply(this.fieldDefaults, this.cfg.createDefaults);
		}

		return this.fieldDefaults;
	},

	/**
	 * Fires the specified event with the passed parameters (minus the event name).
	 *
	 * This function will recursively call into its parents to fire the event
	 * from the parents.
	 * @param {String} eventName The name of the event to fire.
	 * @param {Object...} args The arguments for the event
	 * @return {Boolean} returns false if any of the handlers return false otherwise it returns true.
	 */
	fireEvent: function()
	{
		if (Ext.isDefined(this.parent)) {
			this.parent.fireEvent.apply(this.parent, arguments);
		}
		Zarafa.core.data.RecordDefinition.superclass.fireEvent.apply(this, arguments);
	}
});
Ext.namespace('Zarafa.core.data');

/**
 * @class Zarafa.core.data.RecordFactory
 * The factory class for creating objects of {@link Ext.data.Record record}
 * which are specialized for certain messageclasses (IPM.Note, IPM.Contact, etc).
 * The exact field definition of the different {@link Ext.data.Record records}
 * can be configured by {@link Zarafa.core.Context contexts} and {@link Zarafa.core.Plugin plugins}
 * to add new fields, but also to control the default values for particular fields.
 *
 * For fine-grained control over all classes, the factory also supports
 * {@link Ext.data.Record record} for sub messageclasses like (IPM.Note.NDR),
 * where the sub messageclass depends on its parent (IPM.Note.NDR inherits its fields, and
 * default values from IPM.Note).
 *
 * For plugins with custom types (no official MAPI ObjectType, or Message Class definition),
 * we have the {@link Zarafa.core.data.RecordCustomObjectType Custom Type} definitions to which
 * they can register their custom type, which can then be used to work together with
 * the record factory.
 *
 * @singleton
 */
Zarafa.core.data.RecordFactory = {
	/**
	 * Key-value map of {@link Zarafa.core.data.RecordDefinition} as registered by
	 * {@link #createRecordDefinition}.
	 *
	 * @property
	 * @type Object
	 */
	definitions: {},

	/**
	 * Obtain the parent messageclass for the given messageclass.
	 * The hierarchy of the messageclass is based on the dot ('.')
	 * character. Each dot indicates a step deeper into the hierarchy.
	 * Thus if the messageclass is IPM.Note, the parent is IPM.
	 *
	 * @param {String} messageClass The messageclass for which
	 * the parent is requested.
	 * @return {String} The parent messageclass (undefined, if
	 * the messageclass has no parent.
	 * @private
	 */
	getMessageClassParent: function(messageClass)
	{
		var lastDot = messageClass.lastIndexOf('.');
		if (lastDot > 0) {
			return messageClass.substr(0, lastDot);
		}
	},

	/**
	 * Obtain the parent {@link Zarafa.core.data.RecordDefinition RecordDefinition}
	 * for this messageclass.
	 * @param {String} The messageclass for the parent is searched for
	 * @return {Zarafa.core.data.RecordDefinition} The parent record definition.
	 * Undefined if the messageclass does not have a parent.
	 */
	getMessageClassParentDefinition: function(messageClass)
	{
		var messageClassParent = this.getMessageClassParent(messageClass);

		if (Ext.isDefined(messageClassParent)) {
			return this.getRecordDefinitionByMessageClass(messageClassParent);
		}
	},

	/**
	 * Create a new {@link Zarafa.core.data.RecordDefinition RecordDefinition}
	 * for the given key value.
	 *
	 * @param {Mixed} key The key value on how the definition is stored
	 * in the definitions table.
	 * @param {Zarafa.core.data.RecordDefinition} parent The parent record
	 * definition.
	 * @param {Object} defaults The default values which must be applied to
	 * the record definition.
	 * @return {Zarafa.core.data.RecordDefinition} The new record definition.
	 */
	createRecordDefinition: function(key, parent, defaults)
	{
		var definition = new Zarafa.core.data.RecordDefinition({
			base: undefined,
			parent: parent,
			createDefaults: defaults
		});

		this.definitions[key] = definition;
		return definition;
	},

	/**
	 * Obtain the {@link Zarafa.core.data.RecordDefinition RecordDefinition} object
	 * which belongs to the given custom type. If the custom type is previously unknown
	 * and the {@link Zarafa.core.data.RecordDefinition definition} does not exist,
	 * a new {@link Zarafa.core.data.RecordDefinition definition} will be created and
	 * returned.
	 *
	 * @param {Zarafa.core.data.RecordCustomObjectType} customType The custom type for which the definition is requested.
	 * @return {Zarafa.core.data.RecordDefinition} The definition object for the custom type
	 * @private
	 */
	getRecordDefinitionByCustomType: function(customType)
	{
		var definition = this.definitions[customType];
		if (Ext.isDefined(definition)) {
			return definition;
		}

		// Custom types don't have any hierarchy...
		return this.createRecordDefinition(customType, undefined);
	},

	/**
	 * Obtain the {@link Zarafa.core.data.RecordDefinition RecordDefinition} object
	 * which belongs to the given object class. If the object type is previously unknown
	 * and the {@link Zarafa.core.data.RecordDefinition definition} does not exist,
	 * a new {@link Zarafa.core.data.RecordDefinition definition} will be created and
	 * returned.
	 *
	 * @param {Zarafa.core.mapi.ObjectType} objectType The object type for which the definition is requested.
	 * @return {Zarafa.core.data.RecordDefinition} The definition object for the object type
	 * @private
	 */
	getRecordDefinitionByObjectType: function(objectType)
	{
		var definition = this.definitions[objectType];
		if (Ext.isDefined(definition)) {
			return definition;
		}

		// Object types don't have any hierarchy...
		return this.createRecordDefinition(objectType, undefined, {'object_type': objectType});
	},

	/**
	 * Obtain the {@link Zarafa.core.data.RecordDefinition RecordDefinition} object
	 * which belongs to the given messageclass. If the messageClass is previously unknown
	 * and the {@link Zarafa.core.data.RecordDefinition definition} does not exist,
	 * a new {@link Zarafa.core.data.RecordDefinition definition} will be created and
	 * returned.
	 *
	 * @param {String} messageClass The messageclass for which the definition is requested.
	 * @return {Zarafa.core.data.RecordDefinition} The definition object for the messageclass.
	 * @private
	 */
	getRecordDefinitionByMessageClass: function(messageClass)
	{
		var keyName = messageClass.toUpperCase();
		var definition = this.definitions[keyName];
		var parent;

		if (Ext.isDefined(definition)) {
			return definition;
		}

		parent = this.getMessageClassParentDefinition(keyName);
		if (!Ext.isDefined(parent)){
			/* - If record contains message_class which has no IPM as prefix or suffix followed by dot(.)
			 * like MEMO, REPORT etc, which are not supported by webapp and no fields are available as well (fields
			 * are not registered by any record in webapp) hence record is treated as faulty message in webapp.
			 *
			 * - Webapp can create such records as Ext.data.Record, without throwing any warning or error,
			 * this particular record not able to use Zarafa.core.data.IPMRecord functions.
			 *
			 * - To Fix this, If webapp found any message_class of record which has no IPM as prefix or
			 * suffix followed by dot(.), then by default we set the IPM record definition as parent of this record
			 * definition, further webapp will treat this message as faulty message.
			 */
			parent = this.definitions["IPM"];
		}
		return this.createRecordDefinition(keyName, parent, {'message_class': messageClass});
	},

	/**
	 * Create a new {@link Ext.data.Record record} object based on the
	 * definitions given for the custom type.
	 *
	 * @param {Zarafa.core.data.RecordCustomObjectType} customType The custom type for which
	 * the record is created.
	 * @param {Object} data (Optional) An object, the properties of which provide
	 * values for the new Record's fields. If not specified the {@link Ext.data.Field.defaultValue}
	 * for each field will be assigned.
	 * @param {Object} id (Optional) The id of the Record. The id is used by the
	 * {@link Ext.data.Store Store} object which owns the {@link Ext.data.Record}
	 * to index its collection of Records (therefore this id should be unique within
	 * each store). If an id is not specified a phantom Record will be created with an
	 * automatically generated id.
	 * @return {Ext.data.Record} The new record
	 */
	createRecordObjectByCustomType: function(customType, data, id)
	{
		var definition = this.getRecordDefinitionByCustomType(customType);
		return definition.createRecord(data, id);
	},

	/**
	 * Create a new {@link Ext.data.Record record} object based on the
	 * definitions given for the object type.
	 *
	 * @param {Zarafa.core.mapi.ObjectType} objectType The object type for which
	 * the record is created.
	 * @param {Object} data (Optional) An object, the properties of which provide
	 * values for the new Record's fields. If not specified the {@link Ext.data.Field.defaultValue}
	 * for each field will be assigned.
	 * @param {Object} id (Optional) The id of the Record. The id is used by the
	 * {@link Ext.data.Store Store} object which owns the {@link Ext.data.Record}
	 * to index its collection of Records (therefore this id should be unique within
	 * each store). If an id is not specified a phantom Record will be created with an
	 * automatically generated id.
	 * @return {Ext.data.Record} The new record
	 */
	createRecordObjectByObjectType: function(objectType, data, id)
	{
		var definition = this.getRecordDefinitionByObjectType(objectType);
		return definition.createRecord(data, id);
	},

	/**
	 * Create a new {@link Ext.data.Record record} object based on the
	 * definitions given for the messageclass.
	 *
	 * @param {String} messageClass The messageclass for which the record is created.
	 * @param {Object} data (Optional) An object, the properties of which provide
	 * values for the new Record's fields. If not specified the {@link Ext.data.Field.defaultValue}
	 * for each field will be assigned.
	 * @param {Object} id (Optional) The id of the Record. The id is used by the
	 * {@link Ext.data.Store Store} object which owns the {@link Ext.data.Record}
	 * to index its collection of Records (therefore this id should be unique within
	 * each store). If an id is not specified a phantom Record will be created with an
	 * automatically generated id.
	 * @return {Ext.data.Record} The new record
	 */
	createRecordObjectByMessageClass: function(messageClass, data, id)
	{
		var definition = this.getRecordDefinitionByMessageClass(messageClass);
		return definition.createRecord(data, id);
	},

	/**
	 * Create a new {@link Ext.data.Record record} object based on the
	 * definitions from the record data, this will determine if
	 * the messageclass of object type has been provided. If neither has been provided
	 * this function will return undefined.
	 *
	 * @param {Object} recordData The record data from which the record class must be detected
	 * it slaos contains the properties of which provide values for the new Record's fields.
	 * @param {Object} id (Optional) The id of the Record. The id is used by the
	 * {@link Ext.data.Store Store} object which owns the {@link Ext.data.Record}
	 * to index its collection of Records (therefore this id should be unique within
	 * each store). If an id is not specified a phantom Record will be created with an
	 * automatically generated id.
	 * @return {Class} The record class definition
	 */
	createRecordObjectByRecordData: function(recordData, id)
	{
		if (!Ext.isEmpty(recordData.message_class)) {
			return this.createRecordObjectByMessageClass(recordData.message_class, recordData, id);
		} else if (!Ext.isEmpty(recordData.object_type)) {
			return this.createRecordObjectByObjectType(recordData.object_type, recordData, id);
		}
	},

	/**
	 * Get the record class definition for this custom type
	 * This class definition can be used to construct new {@link Ext.data.Record record}
	 * objects which is completely tuned for the given object type.
	 *
	 * @param {Zarafa.core.data.RecordCustomObjectType} customType The object type for which the
	 * record class is requested
	 * @return {Class} The record class definition
	 */
	getRecordClassByCustomType: function(customType)
	{
		var definition = this.getRecordDefinitionByCustomType(customType);
		return definition.getType();
	},

	/**
	 * Get the record class definition for this object type
	 * This class definition can be used to construct new {@link Ext.data.Record record}
	 * objects which is completely tuned for the given object type.
	 *
	 * @param {Zarafa.core.mapi.ObjectType} objectType The object type for which the
	 * record class is requested
	 * @return {Class} The record class definition
	 */
	getRecordClassByObjectType: function(objectType)
	{
		var definition = this.getRecordDefinitionByObjectType(objectType);
		return definition.getType();
	},

	/**
	 * Get the record class definition for this messageclass
	 * This class definition can be used to construct new {@link Ext.data.Record record}
	 * objects which is completely tuned for the given messageclass.
	 *
	 * @param {String} messageClass The messageclass for which the record class is requested
	 * @return {Class} The record class definition
	 */
	getRecordClassByMessageClass: function(messageClass)
	{
		var definition = this.getRecordDefinitionByMessageClass(messageClass);
		return definition.getType();
	},

	/**
	 * Get the record class definition for this record data, this will determine if
	 * the messageclass of object type has been provided. If neither has been provided
	 * this function will return undefined.
	 * This class definition can be used to construct new {@link Ext.data.Record record}
	 * objects which is completely tuned for the given messageclass.
	 *
	 * @param {Object} recordData The record data from which the record class must be detected
	 * @return {Class} The record class definition
	 */
	getRecordClassByRecordData: function(recordData)
	{
		if (Ext.isDefined(recordData.message_class)) {
			return this.getRecordClassByMessageClass(recordData.message_class);
		} else if (Ext.isDefined(recordData.object_type)) {
			return this.getRecordClassByObjectType(recordData.object_type);
		}
	},

	/**
	 * This will set a new base class on the {@link Zarafa.core.data.RecordDefinition definition}.
	 * The base class is used when creating a new record, by default the ExtJS
	 * {@link Ext.data.Record record} will be used as base.
	 *
	 * @param {Zarafa.core.data.RecordCustomObjectType} customType The custom type for which
	 * the new base class must be set.
	 * @param {Class} baseClass The new base class to use
	 */
	setBaseClassToCustomType: function(customType, baseClass)
	{
		var definition = this.getRecordDefinitionByCustomType(customType);
		definition.setBaseClass(baseClass);
	},

	/**
	 * This will set a new base class on the {@link Zarafa.core.data.RecordDefinition definition}.
	 * The base class is used when creating a new record, by default the ExtJS
	 * {@link Ext.data.Record record} will be used as base.
	 *
	 * @param {Zarafa.core.mapi.ObjectType} objectType The object type for which
	 * the new base class must be set.
	 * @param {Class} baseClass The new base class to use
	 */
	setBaseClassToObjectType: function(objectType, baseClass)
	{
		var definition = this.getRecordDefinitionByObjectType(objectType);
		definition.setBaseClass(baseClass);
	},

	/**
	 * This will set a new base class on the {@link Zarafa.core.data.RecordDefinition definition}.
	 * The base class is used when creating a new record, by default the ExtJS
	 * {@link Ext.data.Record record} will be used as base.
	 *
	 * @param {String} messageClass The messageclass for which
	 * the new base class must be set.
	 * @param {Class} baseClass The new base class to use
	 */
	setBaseClassToMessageClass: function(messageClass, baseClass)
	{
		var definition = this.getRecordDefinitionByMessageClass(messageClass);
		definition.setBaseClass(baseClass);
	},

	/**
	 * This will set the SubStore type for a particular subStore name
	 * in the {@link Zarafa.core.data.RecordDefinition definition}. Inside
	 * a SubStore a Table can be loaded like Attachments, recipients, members, etc.
	 *
	 * @param {Zarafa.core.data.RecordCustomObjectType} customType The custom type for which the
	 * SubStore type must be set
	 * @param {String} name The name of the SubStore table
	 * @param {Constructor} type The SubStore type
	 */
	setSubStoreToCustomType: function(customType, name, type)
	{
		var definition = this.getRecordDefinitionByCustomType(customType);
		definition.setSubStore(name, type);
	},

	/**
	 * This will set the SubStore type for a particular subStore name
	 * in the {@link Zarafa.core.data.RecordDefinition definition}. Inside
	 * a SubStore a Table can be loaded like Attachments, recipients, members, etc.
	 *
	 * @param {Zarafa.core.mapi.ObjectType} objectType The object type for which the
	 * SubStore type must be set
	 * @param {String} name The name of the SubStore table
	 * @param {Constructor} type The SubStore type
	 */
	setSubStoreToObjectType: function(objectType, name, type)
	{
		var definition = this.getRecordDefinitionByObjectType(objectType);
		definition.setSubStore(name, type);
	},

	/**
	 * This will set the SubStore type for a particular subStore name
	 * in the {@link Zarafa.core.data.RecordDefinition definition}. Inside
	 * a SubStore a Table can be loaded like Attachments, recipients, members, etc.
	 *
	 * @param {String} messageClass The messageclass for which
	 * @param {String} name The name of the SubStore table
	 * @param {Constructor} type The SubStore type
	 */
	setSubStoreToMessageClass: function(messageClass, name, type)
	{
		var definition = this.getRecordDefinitionByMessageClass(messageClass);
		definition.setSubStore(name, type);
	},

	/**
	 * This will add a new {@link Ext.data.Field field} to the
	 * {@link Zarafa.core.data.RecordDefinition definition}.
	 *
	 * @param {Zarafa.core.data.RecordCustomObjectType} customType The custom type to which the field
	 * should be added.
	 * @param {String/Ext.data.Field} field The field which must be added.
	 */
	addFieldToCustomType: function(customType, field)
	{
		var definition = this.getRecordDefinitionByCustomType(customType);
		definition.addField(field);
	},

	/**
	 * This will add a new {@link Ext.data.Field field} to the
	 * {@link Zarafa.core.data.RecordDefinition definition}.
	 *
	 * @param {Zarafa.core.mapi.ObjectType} objectType The object type to which the field
	 * should be added.
	 * @param {String/Ext.data.Field} field The field which must be added.
	 */
	addFieldToObjectType: function(objectType, field)
	{
		var definition = this.getRecordDefinitionByObjectType(objectType);
		definition.addField(field);
	},

	/**
	 * This will add a new {@link Ext.data.Field field} to the
	 * {@link Zarafa.core.data.RecordDefinition definition}.
	 *
	 * @param {String} messageClass The messageclass to which the field
	 * should be added.
	 * @param {String/Ext.data.Field} field The field which must be added.
	 */
	addFieldToMessageClass: function(messageClass, field)
	{
		var definition = this.getRecordDefinitionByMessageClass(messageClass);
		definition.addField(field);
	},

	/**
	 * This will add a default value for a {@link Ext.data.Field field}
	 * to the {@link Zarafa.core.data.RecordDefinition definition}. When
	 * creating a new local item
	 * ({@link Zarafa.core.data.RecordFactory.createRecordObjectByCustomType createRecordObjectByCustomType}
	 * with the phantom argument to true) the default values which be applied
	 * to the newly created {@link Ext.data.Record record}.
	 *
	 * @param {Zarafa.core.data.RecordCustomObjectType} customType The custom type to which the field belongs
	 * @param {String/Ext.data.Field} field The field for which the default
	 * value should be applied
	 * @param {Mixed} defaultValue the default value for the given fieldname
	 */
	addDefaultValueToCustomType: function(customType, field, defaultValue)
	{
		var definition = this.getRecordDefinitionByCustomType(customType);
		definition.addDefaultValue(field, defaultValue);
	},

	/**
	 * This will add a default value for a {@link Ext.data.Field field}
	 * to the {@link Zarafa.core.data.RecordDefinition definition}. When
	 * creating a new local item
	 * ({@link Zarafa.core.data.RecordFactory.createRecordObjectByObjectType createRecordObjectByObjectType}
	 * with the phantom argument to true) the default values which be applied
	 * to the newly created {@link Ext.data.Record record}.
	 *
	 * @param {Zarafa.core.mapi.ObjectType} objectType The object type to which the field belongs
	 * @param {String/Ext.data.Field} field The field for which the default
	 * value should be applied
	 * @param {Mixed} defaultValue the default value for the given fieldname
	 */
	addDefaultValueToObjectType: function(objectType, field, defaultValue)
	{
		var definition = this.getRecordDefinitionByObjectType(objectType);
		definition.addDefaultValue(field, defaultValue);
	},

	/**
	 * This will add a default value for a {@link Ext.data.Field field}
	 * to the {@link Zarafa.core.data.RecordDefinition definition}. When
	 * creating a new local item
	 * ({@link Zarafa.core.data.RecordFactory.createRecordObjectByMessageClass createRecordObjectByMessageClass}
	 * with the phantom argument to true) the default values which be applied
	 * to the newly created {@link Ext.data.Record record}.
	 *
	 * @param {String} messageClass The messageclass to which the field belongs
	 * @param {String/Ext.data.Field} field The field for which the default
	 * value should be applied
	 * @param {Mixed} defaultValue the default value for the given fieldname
	 */
	addDefaultValueToMessageClass: function(messageClass, field, defaultValue)
	{
		var definition = this.getRecordDefinitionByMessageClass(messageClass);
		definition.addDefaultValue(field, defaultValue);
	},

	/**
	 * Appends an event handler to the {@link Zarafa.core.data.RecordDefinition RecordDefinition}
	 * for the given CustomType.
	 * @param {Zarafa.core.data.RecordCustomObjectType} customType The custom type to which the event must be added
	 * @param {String} eventName The name of the event to listen for
	 * @param {Function} handler The method the event invokes
	 * @param {Object} scope (optional) The scope (this reference) in which the handler function is executed.
	 * If omitted, defaults to the object which fired the event.
	 * @param {Object} options (optional) An object containing handler configuration properties.
	 */
	addListenerToCustomType: function(customType, eventName, handler, scope, options)
	{
		var definition = this.getRecordDefinitionByCustomType(customType);
		definition.on(eventName, handler, scope, options);
	},

	/**
	 * Appends an event handler to the {@link Zarafa.core.data.RecordDefinition RecordDefinition}
	 * for the given ObjectType.
	 * @param {Zarafa.core.mapi.ObjectType} objectType The object type to which the event must be added
	 * @param {String} eventName The name of the event to listen for
	 * @param {Function} handler The method the event invokes
	 * @param {Object} scope (optional) The scope (this reference) in which the handler function is executed.
	 * If omitted, defaults to the object which fired the event.
	 * @param {Object} options (optional) An object containing handler configuration properties.
	 */
	addListenerToObjectType: function(objectType, eventName, handler, scope, options)
	{
		var definition = this.getRecordDefinitionByObjectType(objectType);
		definition.on(eventName, handler, scope, options);
	},

	/**
	 * Appends an event handler to the {@link Zarafa.core.data.RecordDefinition RecordDefinition}
	 * for the given messageClass.
	 * @param {String} messageClass The messageclass to which the event must be added
	 * @param {String} eventName The name of the event to listen for
	 * @param {Function} handler The method the event invokes
	 * @param {Object} scope (optional) The scope (this reference) in which the handler function is executed.
	 * If omitted, defaults to the object which fired the event.
	 * @param {Object} options (optional) An object containing handler configuration properties.
	 */
	addListenerToMessageClass: function(messageClass, eventName, handler, scope, options)
	{
		var definition = this.getRecordDefinitionByMessageClass(messageClass);
		definition.on(eventName, handler, scope, options);
	}
};
Ext.namespace('Zarafa.core.data');

/**
 * @class Zarafa.core.data.ServerConfig
 * @extends Object
 *
 * An object which represents the server
 * configuration. To obtain the instance
 * of this object, use {@link Zarafa.core.Container#getServerConfig}.
 */
Zarafa.core.data.ServerConfig = Ext.extend(Object, {

	/**
	 * Object containing all meta data for this server configuration
	 * @property
	 * @type Object
	 */
	meta: undefined,

	/**
	 * @constructor
	 * @param {Object} config Configuration object
	 */
	constructor: function(config)
	{
		this.meta = config;
	},

	/**
	 * @return {String} the base url of grommunio Web
	 */
	getBaseUrl: function()
	{
		return this.meta.base_url;
	},

	/**
	 * @return {String} the title of grommunio Web
	 */
	getWebappTitle: function()
	{
		return this.meta.webapp_title;
	},

	/**
	 * @return {Boolean} True if the GAB list should only be enabled when searching
	 */
	isFullGabDisabled: function()
	{
		return this.meta.disable_full_gab === true;
	},

	/**
	 * @return {Boolean} True if it should be possible to set rules on the store
	 * of other users.
	 */
	isSharedRulesEnabled: function()
	{
		return this.meta.enable_shared_rules === true;
	},

	/**
	 * @return {Boolean} True if grommunio Web is using Single Sign-On to login
	 */
	usingSSO: function()
	{
		return this.meta.using_sso;
	},

	/**
	 * @return {Boolean} True if Plugins are enabled
	 */
	isPluginsEnabled: function()
	{
		return this.meta.enable_plugins;
	},

	/**
	 * @return {String} A semicolon separated list of plugins that cannot be
	 * disabled by the user.
	 */
	getAlwaysEnabledPluginsList: function()
	{
		return this.meta.always_enabled_plugins || '';
	},

	/**
	 * @return {Boolean} True if Advanced Settings are enabled
	 */
	isAdvancedSettingsEnabled: function()
	{
		return this.meta.enable_advanced_settings;
	},

	/**
	 * @return {Number} The maximum number of allowed attachments in a single message
	 */
	getMaxAttachments: function()
	{
		return this.meta.max_attachments;
	},

	/**
	 * @return {String} The base url for webapp help manual plugin.
	 */
	getWebappManualUrl: function()
	{
		return this.meta.plugin_webappmanual_url;
	},

	/**
	 * @return {Number} The maximum number of files that can be uploaded via a single request.
	 */
	getMaxFileUploads: function()
	{
		return this.meta.max_file_uploads;
	},

	/**
	 * @reutn {Number} The maximum attachment size allowed to attach in single request.
	 */
	getMaxPostRequestSize: function()
	{
		return this.meta.post_max_size;
	},

	/**
	 * @return {Number} The maximum size of a single attachment
	 */
	getMaxAttachmentSize: function()
	{
		return this.meta.max_attachment_size;
	},

	/**
	 * @return {Number} The maximum size of all attachments in a single message combined
	 */
	getMaxAttachmentTotalSize: function()
	{
		return this.meta.max_attachment_total_size;
	},

	/**
	 * @return {Number} The start offset to use when loading freebusy data
	 */
	getFreebusyLoadStartOffset: function()
	{
		return this.meta.freebusy_load_start_offset;
	},

	/**
	 * @return {Number} The end offset to use when loading freebusy data
	 */
	getFreebusyLoadEndOffset: function()
	{
		return this.meta.freebusy_load_end_offset;
	},

	/**
	 * @return {Number} The upper limit of the eml files allowed to be included in single ZIP archive
	 */
	getMaxEmlFilesInZIP: function()
	{
		return this.meta.maximum_eml_files_in_zip;
	},

	/**
	 * @return {Mixed} The client timeout time (in seconds) if set or false otherwise.
	 */
	getClientTimeout: function()
	{
		return this.meta.client_timeout;
	},

	/**
	 * @return {String} The active theme selected by admin or user.
	 */
	getActiveTheme: function()
	{
		return this.meta.active_theme;
	},

	/**
	 * @return {Array} The installed json themes
	 */
	getJsonThemes: function()
	{
		return this.meta.json_themes;
	},

	/**
	 * @return {String} The active iconset selected by admin or user.
	 */
	getActiveIconset: function()
	{
		return this.meta.active_iconset;
	},

	/**
	 * @return {Array} The installed iconsets
	 */
	getIconsets: function()
	{
		return this.meta.iconsets;
	},

	/**
	 * @return {String|Boolean} The primary color for SVG icons if defined by the active theme,
	 * or false otherwise
	 */
	getPrimaryIconColor: function()
	{
		return this.meta.icons_primary_color;
	},

	/**
	 * @return {String|Boolean} The secondary color for SVG icons if defined by the active theme,
	 * or false otherwise
	 */
	getSecondaryIconColor: function()
	{
		return this.meta.icons_secondary_color;
	},

	/**
	 * @return {Object} The about texts of iconsets
	 */
	getIconsetAbouts: function()
	{
		return this.meta.iconsets_about;
	},

	/**
 	 * @return {Array} returns the installed plugins version information array.
	 */
	getPluginsVersion: function()
	{
		return this.meta.version_info;
	},

	/**
	 * @return {Boolean} True if VCF import functionality is supported on backend, false otherwise.
	 */
	isVCfImportSupported: function()
	{
		return this.meta.is_vcfimport_supported;
	},

	/**
	 * @return {Boolean} True if ICS and VCS import functionality is supported on backend, false otherwise.
	 */
	isICSImportSupported: function()
	{
		return this.meta.is_icsimport_supported;
	},

	/**
	 * @return {Array} returns the color schemes defined in config.php/default.php.
	 */
	getColorSchemes: function()
	{
		return this.meta.color_schemes;
	},

	/**
	 * @return {Array} returns the additional color schemes defined in config.php/default.php.
	 */
	getAdditionalColorSchemes: function()
	{
		return this.meta.additional_color_schemes;
	},

	/**
	 * @return {Array} returns the categories defined in config.php/default.php.
	 */
	getDefaultCategories: function()
	{
		return this.meta.default_categories;
	},

	/**
	 * @return {Array} returns the additional categories defined in config.php/default.php.
	 */
	getAdditionalDefaultCategories: function()
	{
		return this.meta.additional_default_categories;
	},

	/**
	 * @return {Array} returns the contact prefix defined in config.php.
	 */
	getContactPrefix: function ()
	{
		return this.meta.contact_prefix;
	},

	/**
	 * @return {Array} returns the contact suffix defined in config.php.
	 */
	getContactSuffix: function ()
	{
		return this.meta.contact_suffix;
	},

	/**
	 * @return {Array} returns the powerpaste config defined in config.php.
	 */
	getPowerpasteConfig: function ()
	{
		return this.meta.powerpaste;
	},

	/**
	 * @return {Number} return the shared stores polling interval in microseconds
	 */
	getSharedStorePollingInterval: function()
	{
		return this.meta.shared_store_polling_interval * 60000;
	},

	/**
	 * @return {Boolean} True when the client should prefetch the bodies of the mails that are currently visible.
	 */
	isPrefetchEnabled: function()
	{
		if (Ext.isDefined(this.meta.prefetch_email_enabled)) {
			return this.meta.prefetch_email_enabled;
		}

		return this.getPrefetchTotalCount() > 0;
	},

	/**
	 * @return {Number} return the amount of emails to load in the background
	 */
	getPrefetchTotalCount: function()
	{
		return this.meta.prefetch_email_count;
	},

	/**
	 * @return {String} return the strategy used for mail prefetching.
	 */
	getPrefetchStrategy: function()
	{
		if (!Ext.isString(this.meta.prefetch_email_strategy)) {
			return 'VIEWPORT';
		}

		return this.meta.prefetch_email_strategy;
	},

	/**
	 * @return {Number} return the interval in microseconds to load new emails in the background.
	 */
	getPrefetchInterval: function()
	{
		return this.meta.prefetch_email_interval * 1000;
	},

	/**
	 * @returns {Boolean} True if DOMPurify is enabled by admin(from config.php) else false.
	 */
	getDOMPurifyEnabled: function ()
	{
		return this.meta.enable_dompurify;
	},

	/**
	 * @returns {Boolean} True if file previewer is enabled by admin(from config.php) else false.
	 */
	isFilePreviewerEnabled: function ()
	{
		return this.meta.enable_file_previewer;
	},

	/**
	 * @returns {Boolean} True if theming is enabled by admin(from config.php) else false.
	 */
	isThemingEnabled: function ()
	{
		return this.meta.enable_themes;
	},

	/**
	 * @returns {Boolean} True if iconsets are enabled by admin(from config.php) else false.
	 */
	isIconSetsEnabled: function ()
	{
		return this.meta.enable_iconsets;
	},

	/**
	 * @returns {Boolean} True if widgets are enabled by admin(from config.php) else false.
	 */
	isWidgetEnabled : function()
	{
		return this.meta.enable_widgets;
	}
});
Ext.namespace('Zarafa.core.data');

/**
 * @class Zarafa.core.data.SettingsStateProvider
 * @type Ext.state.Provider
 *
 * A special {@link Ext.state.Provider} for the {@link Ext.state.Manager}
 * which stores the state of {@link Zarafa.core.ui.ContentPanel content panels}, {@link Ext.Panel panels}
 * and {@link Ext.grid.GridPanel grids} into the {@link Zarafa.settings.SettingsModel settings}.
 */
Zarafa.core.data.SettingsStateProvider = Ext.extend(Ext.state.Provider, {

	/**
	 * @cfg {String} basePath The base path of the {@link Zarafa.settings.SettingsModel settings}
	 * in which all State settings must be saved. The complete name for a settings path is generated
	 * by {@link #getStatePath}.
	 */
	basePath: 'zarafa/v1/state',

	/**
	 * @constructor
	 * @param {Object} config Configuration object
	 */
	constructor: function(config)
	{
		Ext.apply(this, config);

		Zarafa.core.data.SettingsStateProvider.superclass.constructor.call(this, config);
	},

	/**
	 * Create the path of the {@link Zarafa.settings.SettingsModel settings}
	 * in which the {@link Ext.Component#getState state} for the given {@link Ext.Component component}
	 * must be saved. This adds the {@link Ext.Component#getStateName State name} to the
	 * {@link #basePath}.
	 * @return {String} The Settings path in which the state for the component must be saved.
	 */
	getStatePath: function(component)
	{
		return this.basePath + '/' + component.getStateName();
	},

	/**
	 * Encodes a value including type information. Decode with {@link #decodeValue}.
	 * @param {Mixed} value The value to encode
	 * @param {String} name The property name (if this function was called recursively) of the property being encoded
	 * @param {Ext.Component} component The component for which the settings are being encoded
	 * @return {String} The encoded value
	 */
	encodeValue: function(value, name, component)
	{
		if (Array.isArray(value)) {
			for (var i = 0, len = value.length; i < len; i++) {
				value[i] = this.encodeValue(value[i], name, component);
			}
		} else if (Ext.isObject(value)) {
			for (var key in value) {
				value[key] = this.encodeValue(value[key], key, component);
			}
		} else if ((name === 'height' || name === 'width') && component.statefulRelativeDimensions === true) {
			value = this.scaleSizeToPercentage(component, value, name);
		}

		return value;
	},

	/**
	 * Decodes a string previously encoded with {@link #encodeValue}.
	 * @param {Mixed} value The value to decode
	 * @param {String} name The property name (if this function was called recursively) of the property being decoded
	 * @param {Ext.Component} component The component for which the settings are being decoded
	 * @return {Mixed} The decoded value
	 */
	decodeValue: function(value, name, component)
	{
		if (Array.isArray(value)) {
			for (var i = 0, len = value.length; i < len; i++) {
				value[i] = this.decodeValue(value[i], name, component);
			}
		} else if (Ext.isObject(value)) {
			for (var key in value) {
				value[key] = this.decodeValue(value[key], key, component);
			}
		} else if ((name === 'height' || name === 'width') && value <= 1.0) {
			value = this.scaleSizeToBody(component, value, name);
		}

		return value;
	},

	/**
	 * Convert a 'width' or 'height' value into a percentage of the current size of the body.
	 * This will ensure that when grommunio Web is reloaded on a new display (with different resolution)
	 * the panel is scaled accordingly.
	 * @param {Ext.Component} component The component for which the percentages are calculated
	 * @param {Number} value The value to convert
	 * @param {String} type The type of the value (can be 'width' or 'height')
	 * @return {Number} The converted value
	 * @private
	 */
	scaleSizeToPercentage: function(component, value, type)
	{
		type = Ext.util.Format.capitalize(type);

		var body = window['inner' + type];
		value = parseFloat((value / body).toFixed(2));

		// Limit the value to 1, we don't accept it when the
		// component is resized to beyond the maximum of the
		// body element.
		return Math.min(value, 1);
	},

	/**
	 * Convert a 'width' or 'height' percentage into a real size depending on the size of the body.
	 * This will ensure that when grommunio Web is reloaded on a new display (with different resolution)
	 * the panel is scaled accordingly. It will check the {@link Ext.Component#minWidth}/{@link Ext.Panel#maxWidth}
	 * and {@link Ext.Panel#minHeight}/{@link Ext.Panel#maxHeight} properties in the component to prevent
	 * it from becoming too big.
	 * @param {Ext.Component} component The component for which the real dimensions are determined.
	 * @param {Number} value The value to convert
	 * @param {String} type The type of the value (can be 'width' or 'height')
	 * @return {Number} The converted value
	 * @private
	 */
	scaleSizeToBody: function(component, value, type)
	{
		type = Ext.util.Format.capitalize(type);

		// Convert a 'width' or 'height' percentage into a real size depending on the size of active browser window body.
		var browserWindow = Zarafa.core.BrowserWindowMgr.getActive();
		var body = browserWindow['inner' + type];
		var minSize = component['min' + type];
		var maxSize = component['max' + type];

		value = Math.round(value * body);

		// Collapsible components with a SplitBar have a minSize/maxSize properties,
		// while normal panels have the minWidth/maxWidth & minHeight/maxHeight properties.
		if (component.minSize || component.maxSize) {
			value = Math.min(Math.max(value, component.minSize || 0), component.maxSize || value);
		} else if (minSize || maxSize) {
			value = Math.min(Math.max(value, minSize || 0), maxSize || value);
		}

		return value;
	},

	/**
	 * Sets the value for a key
	 * @param {String} name The key name
	 * @param {Mixed} value The value to set
	 */
	set: function(name, value)
	{
		var component = Ext.state.Manager.getComponent(name);
		container.getSettingsModel().set(this.getStatePath(component), this.encodeValue(value, undefined, component));
		this.fireEvent('statechange', this, name, value);
	},

	/**
	 * Returns the current value for a key
	 * @param {String} name The key name
	 * @param {Mixed} defaultValue A default value to return if the key's value is not found
	 * @return {Mixed} The state data
	 */
	get: function(name, defaultValue)
	{
		var component = Ext.state.Manager.getComponent(name);
		var value = container.getSettingsModel().get(this.getStatePath(component), true);

		if (Ext.isObject(defaultValue)) {
			value = Ext.apply(defaultValue, value);
		}

		return this.decodeValue(value, undefined, component);
	},

	/**
	 * Clears a value from the state
	 * @param {String} name The key name
	 */
	clear: function(name)
	{
		var component = Ext.state.Manager.getComponent(name);
		// Call restore rather then remove, to ensure the default values
		// are applied again.
		container.getSettingsModel().restore(this.getStatePath(component));
		this.fireEvent('statechange', this, name, null);
	}
});
Ext.namespace('Zarafa.core.data');

/**
 * @class Zarafa.core.data.StatefulObservable
 * @extends Ext.util.Observable
 *
 * Extends the {@link Ext.util.Observable} and add {@link #stateful}
 * properties to the object allowing similar features as the
 * {@link Ext.Component}#{@link Ext.Component#stateful}.
 */
Zarafa.core.data.StatefulObservable = Ext.extend(Ext.util.Observable, {
	/**
	 * @cfg {Boolean} stateful A flag which causes the Component to attempt to restore the state of internal properties
	 * from a saved state on startup. The component must have a {@link #stateId} for state to be managed.
	 */
	stateful: false,

	/**
	 * @cfg {String} stateId The unique id for this component to use for state management purposes
	 * See {@link #stateful} for an explanation of saving and restoring Component state.
	 */
	stateId: undefined,

	/**
	 * @cfg {Array} stateEvents An array of events that, when fired, should trigger this component to save its state
	 * (defaults to {@link #datamodechange}). stateEvents may be any type of event supported by this component.
	 * See {@link #stateful} for an explanation of saving and restoring Component state.
	 */
	stateEvents: undefined,

	/**
	 * @cfg {String} statefulName The unique name for this component by which the {@link #getState state}
	 * must be saved into the {@link Zarafa.settings.SettingsModel settings}.
	 * This option is only used when the {@link Zarafa.core.data.SettingsStateProvider SettingsStateProvider} is
	 * used in the {@link Ext.state.Manager}.
	 */
	statefulName: undefined,

	/**
	 * @constructor
	 * @param {Object} config Configuration object
	 */
	constructor: function(config)
	{
		Ext.apply(this, config);

		Zarafa.core.data.StatefulObservable.superclass.constructor.call(this, config);

		// If stateful, generate a stateId if not provided manually,
		// and register the object to the Ext.state.Manager.
		if (this.stateful !== false) {
			if (!this.stateId) {
				this.stateId = Ext.id(null, 'state-');
			}

			Ext.state.Manager.register(this);
			this.initStateEvents();
		}
	},

	/**
	 * Obtain the path in which the {@link #getState state} must be saved.
	 * This option is only used when the {@link Zarafa.core.data.SettingsStateProvider SettingsStateProvider} is
	 * used in the {@link Ext.state.Manager}. This returns {@link #statefulName} if provided, or else generates
	 * a custom name.
	 * @return {String} The unique name for this component by which the {@link #getState state} must be saved.
	 */
	getStateName: function()
	{
		return this.statefulName;
	},

	/**
	 * Obtain the {@link #stateId} from the component
	 * @return {String} The state ID
	 * @private
	 */
	getStateId: function()
	{
		return this.stateId || this.id;
	},

	/**
	 * Register the {@link #stateEvents state events} to the {@link #saveState} callback function.
	 * @protected
	 */
	initStateEvents: function()
	{
		if (this.stateEvents) {
			for (var i = 0, e; e = this.stateEvents[i]; i++) {
				this.on(e, this.saveState, this, {delay:100});
			}
		}
	},

	/**
	 * Intialialize the component with {@link #applyState} using the state which has
	 * been {@link Ext.state.Manager#get obtained} from the {@link Ext.state.Manager State Manager}.
	 * @private
	 */
	initState: function()
	{
		if (Ext.state.Manager) {
			var id = this.getStateId();
			if (id) {
				var state = Ext.state.Manager.get(id);
				if (state) {
					if (this.fireEvent('beforestaterestore', this, state) !== false) {
						this.applyState(Ext.apply({}, state));
						this.fireEvent('staterestore', this, state);
					}
				}
			}
		}
	},

	/**
	 * Apply the given state to this object activating the properties which were previously
	 * saved in {@link Ext.state.Manager}.
	 * @param {Object} state The state object
	 * @protected
	 */
	applyState: function(state)
	{
		if (state) {
			Ext.apply(this, state);
		}
	},

	/**
	 * When {@link #stateful} the State object which should be saved into the
	 * {@link Ext.state.Manager}.
	 * @return {Object} The state object
	 * @protected
	 */
	getState: function()
	{
		return null;
	},

	/**
	 * Obtain the {@link #getState current state} and {@link Ext.state.Manager#set save} it
	 * to the {@link Ext.state.Manager State Manager}.
	 * @private
	 */
	saveState: function()
	{
		if (Ext.state.Manager && this.stateful !== false) {
			var id = this.getStateId();
			if (id) {
				var state = this.getState();
				if (this.fireEvent('beforestatesave', this, state) !== false) {
					Ext.state.Manager.set(id, state);
					this.fireEvent('statesave', this, state);
				}
			}
		}
	}
});
Ext.namespace('Zarafa.core.data');

/**
 * @class Zarafa.core.data.UIFactory
 * @extends Object
 * @singleton
 *
 *
 * The UI Factory is responsible for opening {@link Zarafa.core.ui.ContentPanel panels} in {@link Zarafa.core.data.UIFactoryLayer layers}.
 * The UI Factory decides the type of layer, based on cascading rules.
 * Types of layers are registered via {@link #registerLayer}.
 * First the type of layer is taken from settings, and then from a config object passed when creating the panel.
 */
Zarafa.core.data.UIFactory = {

	/**
	 * The array of {@link #registerLayer registered} {@link Zarafa.core.data.UIFactoryLayer layers}.
	 * @property
	 * @type Array
	 * @private
	 */
	layers: [],

	/**
	 * Register a layer a layer to the {@Link #layers} array.
	 * @param {Zarafa.core.data.UIFactoryLayer} layer The layer to register
	 */
	registerLayer: function(layer)
	{
		var index = layer.index || 0;

		if (this.layers.length > 0) {
			for (var i = 0, len = this.layers.length; i < len; i++) {
				if (this.layers[i].index > index) {
					this.layers.splice(i, 0, layer);
					return;
				}
			}

		}

		this.layers.push(layer);
	},

	/**
	 * Method to determine the {@link Zarafa.core.data.UIFactoryLayer layer} to use
	 *
	 * @param {Object} config Configuration object
	 * @param {Zarafa.core.data.MAPIRecord} record The record(s) loaded in the component
	 * @return {Zarafa.core.data.UIFactoryLayer} The Layer in which to place a component
	 */
	getPreferredLayer: function(config, record)
	{
		var layers = this.layers;

		// By default the bottom layout is the default,
		// this might be overridden from the settings.
		var layerIndex = 0;

		// First, we must check the special situations which are provided
		// by the configuration object.
		if (config) {
			// The first case checks if a manager was configured, if so the layer
			// which utilizes this manager get preference for the layer in which
			// the component is going to be installed.
			if (config.manager) {
				for (var i = 0, len = layers.length; i < len; i++) {
					if (config.manager === layers[i].manager) {
						layerIndex = i;
						break;
					}
				}
			}

			// The second case is if the new component has the 'modal' property,
			// not all layers accept this, so starting from the layerIndex,
			// we have to search upwards to find the layer which allows the
			// 'modal' property to be set.
			if (config.modal === true) {
				for (var i = layerIndex, len = layers.length; i < len; i++) {
					if (layers[i].allowModal === true) {
						layerIndex = i;
					}
				}
			}
		}

		// The settings might have a different idea on the layer in which
		// the component must be placed. Request the 'type' which should be the
		// layer in which components should be placed.
		var baseContentLayer;
		if(config && config.layerType) {
			baseContentLayer = config.layerType;
		} else {
			// Currently the support of popout feature is restricted to mail context only.
			// So, honor the configured value of base_content_layer settings for mail context only.
			if (Ext.isDefined(record) && record !== null && Ext.isFunction(record.isMessageClass) && record.isMessageClass(['IPM.Note', 'IPM.Schedule.Meeting', 'REPORT.IPM', 'REPORT.IPM.Note'], true)) {
				baseContentLayer = container.getSettingsModel().get('zarafa/v1/main/base_content_layer');
			}
		}

		// Go for popout only if supported.
		if (Zarafa.supportsPopOut() && !Ext.isEmpty(baseContentLayer) && layerIndex === 0 && !(config && config.searchText)) {
			for (var i = 0, len = layers.length; i < len; i++) {
				if (layers[i].type === baseContentLayer) {
					layerIndex = i;
					break;
				}
			}
		}

		return layers[layerIndex];
	},

	/**
	 * Open a {@link Ext.Component component} in a chosen layer.
	 *
	 * @param {Number} componentType A component type taken from the enumeration {@link Zarafa.core.data.SharedComponentType}
	 * @param {Zarafa.core.data.MAPIRecord} record The record(s) loaded in the component
	 * @param {Object} config Configuration object
	 */
	openLayerComponent: function(componentType, records, config)
	{
		var ComponentConstructor = container.getSharedComponent(componentType, records);
		if (ComponentConstructor) {
			if (ComponentConstructor.prototype instanceof Ext.Component) {
				var layer = this.getPreferredLayer(config, records);

				// FIXME: This shouldn't be here, the caller should have
				// applied the record and closable information.
				config = Ext.applyIf(config || {}, {
					record: records,
					closable: true,
					plugins: []
				});
				config.plugins = config.plugins.concat(layer.plugins);

				layer.create(ComponentConstructor, config);
			} else {
				ComponentConstructor.doOpen(records);
			}
		}
	},

	/**
	 * Will open the view action for the passed record.
	 * @param {Zarafa.core.data.MAPIRecord|Zarafa.core.data.MAPIRecord[]} records
	 * The record/records which will be loaded in dialog.
	 * @param {Object} config configuration object.
	 */
	openViewRecord: function(records, config)
	{
		var componentType = Zarafa.core.data.SharedComponentType['common.view'];
		this.openLayerComponent(componentType, records, config);
	},

	/**
	 * Will open the create action for the passed record.
	 * @param {Zarafa.core.data.MAPIRecord|Zarafa.core.data.MAPIRecord[]} records
	 * The record/records which will be loaded in dialog.
	 * @param {Object} config configuration object.
	 */
	openCreateRecord: function(records, config)
	{
		var componentType = Zarafa.core.data.SharedComponentType['common.create'];
		this.openLayerComponent(componentType, records, config);
	},

	/**
	 * Will open a contextmenu for the passed record/records
	 * This method uses the default componentType for context menus and calls {@link #openContextMenu} with it.
	 * @param {Zarafa.core.data.MAPIRecord|Zarafa.core.data.MAPIRecord[]} records
	 * The record/records for which the contextmenu should be shown
	 * @param {Object} position The X and Y coordinate where the contextmenu was requested
	 * @param {Object} configuration object which should be applied to the contextmenu.
	 */
	openDefaultContextMenu: function(records, config)
	{
		var componentType = Zarafa.core.data.SharedComponentType['common.contextmenu'];
		this.openContextMenu(componentType, records, config);
	},

	/**
	 * Open a context menu with supplied componentType
	 *
	 * @param {Number} componentType Shared component type taken from {@link Zarafa.core.data.SharedComponentType}
	 * @param {Zarafa.core.data.MAPIRecord|Zarafa.core.data.MAPIRecord[]} records The record(s) for which the contextmenu will be shown
	 * @param {Object} config Configuration object
	 */
	openContextMenu: function(componentType, records, config)
	{
		var ComponentConstructor = container.getSharedComponent(componentType, records);
		if (ComponentConstructor) {
			new ComponentConstructor(Ext.applyIf(config || {}, {records: records })).showAt(config.position);
		}
	},

	/**
	 * Open a hover card with supplied componentType
	 *
	 * @param {Zarafa.core.data.MAPIRecord|Zarafa.core.data.MAPIRecord[]} records The record(s) for which the hover card will be shown
	 * @param {Object} config Configuration object
	 */
	openHoverCard: function (records, config)
	{
		// Close all existing hover card
		Ext.WindowMgr.getBy(function (win) {
			if (win instanceof Zarafa.common.recipientfield.ui.RecipientHoverCardView) {
				win.hide();
			}
		}, this);
		var componentType = Zarafa.core.data.SharedComponentType['common.hovercard'];
		this.openContextMenu(componentType, records, config);
	}
};
Ext.namespace('Zarafa.core.data');

/**
 * @class Zarafa.core.data.UIFactoryLayer
 * @extends Object
 *
 * A Layer object which must be {@link Zarafa.core.data.UIFactory#registerLayer registered}}
 * to the {@link Zarafa.core.data.UIFactory UIFactory}. This object describes the
 * Layer on which a {@link Zarafa.core.ui.ContentPanel Content Panel} can be placed.
 */
Zarafa.core.data.UIFactoryLayer = Ext.extend(Object, {
	/**
	 * @cfg {String} type The name of the Content Layer
	 * which uniqely identifies this layer.
	 */
	type: '',

	/**
	 * @cfg {Number} index The index of the Content Layer. See
	 * {@link Zarafa.core.data.UIFactory#getPreferredLayer} for
	 * the ordering logic which is applied.
	 */
	index: undefined,

	/**
	 * @cfg {Array} plugins The plugins which must be installed
	 * on the Panel which is going to be installed on the Content Layer
	 */
	plugins: undefined,

	/**
	 * @cfg {Boolean} allowModal True if this Content Layer
	 * supports 'modal' components to be added.
	 */
	allowModal: false,

	/**
	 * @cfg {Object} manager The manager instance which is managing
	 * the Content Layer (e.g. {@link Ext.WindowGroup} or {@link Ext.TabPanel}.
	 */
	manager: undefined,

	/**
	 * @constructor
	 * @param {Object} config Configuration object
	 */
	constructor: function(config)
	{
		Ext.apply(this, config);
	},

	/**
	 * The create function which is invoked when a component needs to be added to the Container using
	 * this Layer. This must be overridden by subclasses.
	 * @param {Function} Constructor The constructor of the component which has to be created in the container layer.
	 * @param {Object} config The configuration object which must be
	 * passed to the constructor when creating the component
	 * @protected
	 */
	create: Ext.emptyFn
});
Ext.namespace('Zarafa.core.data');

/**
 * @class Zarafa.core.data.User
 * @extends Object
 *
 * An object which represents a logged
 * on user in grommunio Web environment.
 * To obtain the instance of this object
 * for the currently logged in user,
 * refer to {@link Zarafa.core.Container#getUser}
 */
Zarafa.core.data.User = Ext.extend(Object, {

	/**
	 * Object containing all meta data for
	 * this user.
	 * @property
	 * @type Object
	 */
	meta: undefined,

	/**
	 * @constructor
	 * @param {Object} config Configuration object
	 */
	constructor: function(config)
	{
		this.meta = config;
	},

	/**
	 * @return {String} The entryid for the user
	 */
	getEntryId: function()
	{
		return this.meta.entryid;
	},

	/**
	 * @return {String} The searchkey for the user
	 */
	getSearchKey: function()
	{
		return this.meta.search_key;
	},

	/**
	 * The display name for the user, this can be either
	 * the {@link #getFullName} or {@link #getUserName}
	 * depending which one is a non-empty string.
	 * @return {String} The displayname for the user
	 */
	getDisplayName: function()
	{
		return this.meta.fullname || this.meta.username;
	},
	
	/**
	 * @return {String} The thumbnail photho for this user
	 */
	getUserImage : function()
	{
		return this.meta.user_image || Ext.BLANK_IMAGE_URL;
	},
	
	setUserImage : function(image_data)
	{
		this.meta.user_image = image_data;
	},

	/**
	 * @return {String} The fullname for this user
	 */
	getFullName: function()
	{
		return this.meta.fullname;
	},

	/**
	 * @return {String} The username for this user
	 */
	getUserName: function()
	{
		return this.meta.username;
	},

	/**
	 * @return {String} The emailaddress for this user
	 */
	getEmailAddress: function()
	{
		return this.meta.email_address;
	},

	/**
	 * @return {String} The email address for this user
	 */
	getSMTPAddress: function()
	{
		return this.meta.smtp_address;
	},

	/**
	 * @deprecated 2.2.0 This function only exists for backward compatibility with
	 * 		 that want to send the session id as a GET parameter with requests that
	 * 		 they make to grommunio.php. Currently grommunio.php does not expect this
	 * 		 parameter anymore, but plugins that have not been updated might still
	 * 		 call this function.
	 * @return {String} Always empty
	 */
	getSessionId: function()
	{
		return '';
	},

	/**
	 * @return {String} The first name for this user
	 */
	getFirstName: function()
	{
		return this.meta.given_name;
	},

	/**
	 * @return {String} The initials for this user
	 */
	getInitials: function()
	{
		return this.meta.initials;
	},

	/**
	 * @return {String} The surname for this user
	 */
	getLastName: function()
	{
		return this.meta.surname;
	},

	/**
	 * @return {String} The street address for this user
	 */
	getAddress: function()
	{
		return this.meta.street_address;
	},

	/**
	 * @return {String} The city for this user
	 */
	getCity: function()
	{
		return this.meta.locality;
	},

	/**
	 * @return {String} The state or province for this user
	 */
	getState: function()
	{
		return this.meta.state_or_province;
	},

	/**
	 * @return {String} The zipcode for this user
	 */
	getZipCode: function()
	{
		return this.meta.postal_code;
	},

	/**
	 * @return {String} The country for this user
	 */
	getCountry: function()
	{
		return this.meta.country;
	},

	/**
	 * @return {String} The title for this user
	 */
	getTitle: function()
	{
		return this.meta.title;
	},

	/**
	 * @return {String} The company for this user
	 */
	getCompany: function()
	{
		return this.meta.company_name;
	},

	/**
	 * @return {String} The department for this user
	 */
	getDepartment: function()
	{
		return this.meta.department_name;
	},

	/**
	 * @return {String} The office location for this user
	 */
	getOffice: function()
	{
		return this.meta.office_location;
	},

	/**
	 * @return {String} The assistant for this user
	 */
	getAssistant: function()
	{
		return this.meta.assistant;
	},

	/**
	 * @return {String} The business phone number for this user
	 */
	getPhone: function()
	{
		return this.getPhoneBusiness();
	},

	/**
	 * @return {String} The business phone number for this user
	 */
	getPhoneBusiness: function()
	{
		return this.meta.business_telephone_number || this.meta.office_telephone_number;
	},

	/**
	 * @return {String} The second business phone number for this user
	 */
	getPhoneBusiness2: function()
	{
		return this.meta.business2_telephone_number;
	},

	/**
	 * @return {String} The fax number for this user
	 */
	getFax: function()
	{
		return this.meta.primary_fax_number;
	},

	/**
	 * @return {String} The phone number of the assistant for this user
	 */
	getPhoneAssistant: function()
	{
		return this.meta.assistant_telephone_number;
	},

	/**
	 * @return {String} The home phone number for this user
	 */
	getPhoneHome: function()
	{
		return this.meta.home_telephone_number;
	},

	/**
	 * @return {String} The second home phone number for this user
	 */
	getPhoneHome2: function()
	{
		return this.meta.home2_telephone_number;
	},

	/**
	 * @return {String} The mobile phone number for this user
	 */
	getPhoneMobile: function()
	{
		return this.meta.mobile_telephone_number;
	},

	/**
	 * @return {String} The pager phone number for this user
	 */
	getPhonePager: function()
	{
		return this.meta.pager_telephone_number;
	}
});
Ext.namespace('Zarafa.core.data');

/**
 * @class Zarafa.core.data.UserIdObject
 * @extends Object
 *
 * An object that identifies a user and has some functions to create and
 * compare.
 */
Zarafa.core.data.UserIdObject = Ext.extend(Object, {
	/**
	 * The type of this user. Can be ZARAFA or SMTP
	 * @property
	 */
	type: '',

	/**
	 * The display name of this user. Only added as a property to make
	 * easier for humans to recognize the user. Should not be used to
	 * identify the user!
	 * @property
	 */
	display_name: '',

	/**
	 * The email address of this user. Not always available.
	 * @property
	 */
	email_address: '',

	/**
	 * The username of this user. Only available for ZARAFA type users.
	 * @property
	 */
	username: '',

	/**
	 * The entryid of this user. Only available for ZARAFA type users.
	 * @property
	 */
	entryid: '',

	/**
	 * Constructor.
	 * @param {Object} idObject An object with properties
	 * that identify a user.
	 */
	constructor: function(idObject)
	{
		if ( idObject.type ) {
			this.type = idObject.type;
		}
		if ( idObject.display_name ) {
			this.display_name = idObject.display_name;
		}
		if ( idObject.email_address ) {
			this.email_address = idObject.email_address;
		}
		if ( idObject.username ) {
			this.username = idObject.username;
		}
		if ( idObject.entryid ) {
			this.entryid = idObject.entryid;
		}
	},

	/**
	 * Compares the {@link Zarafa.core.data.UserIdObject UserIdObject} to the passed
	 * {@link Zarafa.core.data.UserIdObject}. Returns true if both objects
	 * identify the same user.
	 * @param {Zarafa.core.data.UserIdObject} userIdObject A user of whom the id data
	 * will be compared to this users id data.
	 * @return {Boolean}
	 */
	equals: function(userIdObject)
	{
		// First we will compare the entryids if they are set
		if ( !Ext.isEmpty(this.entryid) && !Ext.isEmpty(userIdObject.entryid) ) {
			return Zarafa.core.EntryId.compareEntryIds(this.entryid, userIdObject.entryid);
		}

		// Second we will compare usernames. If a username is set, then an entryId should also be set,
		// so this comparison should not be necessary. However, this makes it possible to request
		// presence status with only a username, so let's include it.
		if ( !Ext.isEmpty(this.username) && this.username === userIdObject.username ) {
			return true;
		}

		// Third we will compare email addresses
		if ( !Ext.isEmpty(this.email_address) && this.email_address === userIdObject.email_address ) {
			return true;
		}

		//Last we will take care of when a username field is filled with the email address
		if (
			( !Ext.isEmpty(this.username) && this.username === userIdObject.email_address ) ||
			( !Ext.isEmpty(this.email_address) && this.email_address === userIdObject.username )
		){
			return true;
		}

		return false;
	},

	/**
	 * Synchronizes the data of this user with the given user.
	 * @param {Zarafa.core.data.UserIdObject} user A user whose id data will be used
	 * to update the id data of this user if possible.
	 */
	syncWithUser: function(user)
	{
		if ( user.type === 'EX' ){
			if ( this.type !== 'EX' ){
				this.type = user.type;
				this.display_name = user.display_name;
				this.username = user.username;
				this.entryid = user.entryid;
			}
			// Only store the email_address if we don't already have one. (Sometime we don't get an
			// email address for ZARAFA users, so we shouldn't overwrite if we have one)
			if ( Ext.isEmpty(this.email_address) ){
				this.email_address = user.username.indexOf('@')>=0 ? user.username : user.smtp_address;
			}
		}
	}
});
Ext.namespace('Zarafa.core.data');

/**
 * @class Zarafa.core.data.UserIdObjectFactory
 * @extends Object
 *
 * An factory object that can be used to create
 * {@link Zarafa.core.data.UserIdObject UserIdObjects}
 *
 * @singleton
 */
Zarafa.core.data.UserIdObjectFactory = {
	/**
	 * Creates a {@link Zarafa.core.data.UserIdObject} from the data in the record.
	 * @param {Zarafa.core.data.IPMRecord} record The record from which the data for the
	 * userIdObject is taken.
	 * @return {Zarafa.core.data.UserIdObject|null}
	 */
	createFromRecord: function(record)
	{
		// Return null for distlists
		if ( !Ext.isFunction(record.get) || !Ext.isEmpty(record.get('object_type')) && record.get('object_type') === Zarafa.core.mapi.ObjectType.MAPI_DISTLIST ) {
			return null;
		}

		// Add the general fields to the object
		var user = {};

		// Try to be smart to find a display_name
		user.display_name = record.get('display_name');

		user.type = record.get('address_type');

		if ( user.type === 'EX' ){

			// Add ZARAFA specific fields to the object
			var recordEntryId = record.get('entryid');
			var recordUsername = record.get('username');
			var recordEmailAddress = record.get('email_address');
			var recordSmtpAddress = record.get('smtp_address');

			user.entryid = recordEntryId;

			// Try to be smart to find the username and email address.
			// Sometimes the username is available in the email_address field,
			// sometimes the email address is available in the username field,
			// and sometimes it is available in the smtp_address field.
			if ( Ext.isString(recordUsername) && recordUsername.indexOf('@')==-1 ){
				user.username = recordUsername;
			} else if ( Ext.isString(recordEmailAddress) && recordEmailAddress.indexOf('@')==-1 ){
				user.username = recordEmailAddress;
			}
			if ( Ext.isString(recordEmailAddress) && recordEmailAddress.indexOf('@')>0 ){
				user.email_address = recordEmailAddress;
			} else if ( Ext.isString(recordSmtpAddress) && recordSmtpAddress.indexOf('@')>0 ){
				user.email_address = recordSmtpAddress;
			} else if ( Ext.isString(recordUsername) && recordUsername.indexOf('@')>0 ){
				user.email_address = recordUsername;
			}
		} else if ( !Ext.isEmpty(user.type) ) {

			// Add the field for non-ZARAFA users to the object
			// Try to be smart to find the email address
			user.email_address = record.get('email_address') || record.get('smtp_address');
			user.entryid = record.get('entryid');
		} else {
			return null;
		}

		var userIdObject = new Zarafa.core.data.UserIdObject(user);

		// If possible sync the userInfo with the cached one
		return Zarafa.core.data.PresenceCache.syncUsers([userIdObject])[0];
	},

	/**
	 * Returns an array with {@link Zarafa.core.data.UserIdObject UserIdObjects} for all
	 * users in the passed store.
	 * @param {Zarafa.core.data.MAPIStore|Zarafa.core.data.MAPISubStore} store The store
	 * from which {@link Zarafa.core.data.UserIdObject UserIdObjects} will be created.
	 * @return {Zarafa.core.data.UserIdObject[]} An array with userInfo
	 * objects.
	 */
	createFromStore: function(store)
	{
		if ( !store.data ) {
			return [];
		}

		var userIdObjects = [];
		var records = store.getRange();

		Ext.each(records, function(record){
			var userIdObject = Zarafa.core.data.UserIdObjectFactory.createFromRecord(record);
			if ( userIdObject ) {
				userIdObjects.push(userIdObject);
			}
		}, this);

		return userIdObjects;
	}
};
Ext.namespace('Zarafa.core.data');

/**
 * @class Zarafa.core.data.Version
 * @extends Object
 *
 * An object which represents the versioning
 * information of grommunio Web environment.
 * To obtain the instance of this object
 * refer to {@link Zarafa.core.Container#getVersion}
 */
Zarafa.core.data.Version = Ext.extend(Object, {

	/**
	 * Object containing all meta data for
	 * this user.
	 * @property
	 * @type Object
	 */
	meta: undefined,

	/**
	 * @constructor
	 * @param {Object} config Configuration object
	 */
	constructor: function(config)
	{
		this.meta = config;
	},

	/**
	 * @return {String} Return grommunio Web version number
	 */
	getWebApp: function()
	{
		return this.meta.webapp;
	},

	/**
	 * @return {String} Return the Gromox version number
	 */
	getZCP: function()
	{
		return this.meta.zcp;
	},

	/**
	 * @return {String} Return the current Git branch
	 */
	getGit: function()
	{
		return this.meta.git;
	},

	/**
	 * @return {String} Return the cache buster string
	 */
	getCachebuster: function()
	{
		return this.meta.cachebuster;
	},

	/**
	 * Compares two version strings.
	 * @param {String} version1 The first version string for the comparison
	 * @param {String} version2 The second version string for the comparison
	 * @return {Integer} -1 if version1 is lower than version2,
	 * 1 if version2 is lower, 0 if the versions are equal.
	 */
	versionCompare: function(version1, version2)
	{
		// First remove unnecessary parts
		if ( version1.indexOf('-') > -1 ){
			version1 = version1.split('-')[0];
		}
		if ( version2.indexOf('-') > -1 ){
			version2 = version2.split('-')[0];
		}

		version1 = version1.split('.');
		version2 = version2.split('.');

		// Only compare major, minor, patch version.
		if (version1.length > 3) {
			version1 = version1.splice(0,3);
		}

		if (version2.length > 3) {
			version2 = version2.splice(0,3);
		}

		for ( var i=0; i<version1.length; i++ ){
			if ( !Ext.isDefined(version2[i]) ){
				return 1;
			}

			var v1 = parseInt(version1[i], 10);
			var v2 = parseInt(version2[i], 10);

			if ( v1 < v2 ){
				return -1;
			} else if ( v1 > v2 ){
				return 1;
			}
		}

		if ( version2[version1.length] ){
			return -1;
		}

		// When we get here the versions are considered the same
		return 0;
	}
});
Ext.namespace('Zarafa.core.data');

/**
 * @class Zarafa.core.data.ZarafaCustomEventDispatcher
 * @extends Ext.util.Observable
 * @singleton
 *
 * This singleton can be used to register events that have to be available in the whole Webapp.
 */
Zarafa.core.data.ZarafaCustomEventDispatcher = Ext.extend(Ext.util.Observable, {
	/**
	 * @constructor
	 * @param config
	 */
	constructor: function(config) {
		// Call our superclass constructor to complete construction process.
		Zarafa.core.data.ZarafaCustomEventDispatcher.superclass.constructor.call(this, config);
	}
});

// make it a singleton
Zarafa.core.data.ZarafaCustomEventDispatcher = new Zarafa.core.data.ZarafaCustomEventDispatcher();
Ext.namespace('Zarafa.core.plugins');

/**
 * @class Zarafa.core.plugins.AutoSaveMessagePlugin
 * @extends Object
 * @ptype zarafa.autosavemessageplugin
 *
 * Special plugin which must be used in combination with the
 * {@link Zarafa.core.plugins.RecordComponentPlugin}. This will
 * save message periodically after certain time.
 */
Zarafa.core.plugins.AutoSaveMessagePlugin = Ext.extend(Object, {
	/**
	 * The component which has been {@link #init initialized} on
	 * this plugin.
	 * @property
	 * @type Ext.Component
	 */
	field: undefined,

	/**
	 * The record which has been {@link #onSetRecord set} on the
	 * {@link #field}.
	 * @property
	 * @type Zarafa.core.data.MAPIRecord
	 */
	record: undefined,

	/**
	 * Timer that is used to save message after specified minutes
	 * @property
	 * @type Number
	 */
	messageAutoSaveTimer: undefined,

	/**
	 * @constructor
	 * @param {Object} config Configuration object
	 */
	constructor: function(config)
	{
		Ext.apply(this, config);
	},

	/**
	 * Initializes the {@link Ext.Component Component} to which this plugin has been hooked.
	 * @param {Ext.Component} field The component on which the plugin is installed
	 */
	init: function(field)
	{
		this.field = field;
		field.messageAutoSavePlugin = this;

		this.field.on('render', this.onRender, this);
	},

	/**
	 * Event handler for the {@link Ext.Component#render render} event
	 * on the {@link #field}. This will register the event handlers which
	 * are needed to listen for record changes.
	 */
	onRender: function()
	{
		// Initialize necessary events based on ContentPanel and RecordComponentPlugin.
		this.field.on({
			'setrecord': this.onSetRecord,
			'userupdaterecord': this.onUserUpdateRecord,
			'saverecord': this.onSaveRecord,
			'close': this.onDestroy,
			'destroy': this.onDestroy,
			'scope': this
		});

		var fields = this.field.find('enableKeyEvents', true);

		// Register keypress event on every input field of the ContentPanel.
		Ext.each(fields, function(inputField) {
			this.field.mon(inputField, {
				'keypress': this.onFieldKeyPress,
				'scope': this
			});
		}, this);
	},

	/**
	 * Event handler for the {@link Zarafa.core.plugins.RecordComponentPlugin#setrecord setrecord} event
	 * @param {Ext.Component} field The component which fired the event
	 * @param {Zarafa.core.data.MAPIRecord} record The new record which is being set
	 * @param {Zarafa.core.data.MAPIRecord} oldrecord The old record which was previously set
	 * @private
	 */
	onSetRecord: function(field, record, oldrecord)
	{
		// Only IPMRecords will be handled by this plugin,
		// all other records will be discarded.
		if (record instanceof Zarafa.core.data.IPMRecord) {
			this.record = record;
		} else {
			this.record = undefined;
		}
	},

	/**
	 * Event handler for the {@link Zarafa.core.plugins.RecordComponentPlugin#userupdaterecord userupdaterecord} event
	 * on the {@link #field}. This will {@link #startMessageAutoSaveTimer start} the {@link #messageAutoSaveTimer}.
	 * @param {Ext.Component} field The component which fired the event
	 * @param {Zarafa.core.data.MAPIRecord} record The record which was updated
	 * @param {Boolean} isChangedByUser Indicates if the record has been changed by the user since it has been loaded.
	 * @private
	 */
	onUserUpdateRecord: function(field, record, isChangedByUser)
	{
		if (isChangedByUser) {
			this.startMessageAutoSaveTimer();
		}
	},

	/**
	 * Event handler for the input fields of the {@link #field}.
	 * This will {@link #startMessageAutoSaveTimer start} the {@link #messageAutoSaveTimer}.
	 * @param {HtmlElement} element The target of the event
	 * @param {Ext.EventObject} event The Ext.EventObject encapsulating the DOM event
	 * @param {Object} options The options configuration passed to the addListener call
	 * @private
	 */
	onFieldKeyPress: function(element, event, options)
	{
		this.startMessageAutoSaveTimer();
	},

	/**
	 * Start the {@link #messageAutoSaveTimer} if it was not yet started already.
	 * This will check the settings for the desired interval, and delay the call
	 * to {@link #messageAutoSave} as configured.
	 * @private
	 */
	startMessageAutoSaveTimer: function()
	{
		if (!this.messageAutoSaveTimer && this.record && this.record.isUnsent()) {
			var timeout = container.getSettingsModel().get('zarafa/v1/contexts/mail/autosave_time') * 1000;
			this.messageAutoSaveTimer = this.messageAutoSave.defer(timeout, this);
		}
	},

	/**
	 * Stop the {@link #messageAutoSaveTimer} if it is still pending.
	 * This will clear the interval and delete the {@link #messageAutoSaveTimer} making it
	 * available again for rescheduling.
	 * @private
	 */
	resetMessageAutoSaveTimer: function()
	{
		if (this.messageAutoSaveTimer) {
			window.clearTimeout(this.messageAutoSaveTimer);
			this.messageAutoSaveTimer = null;
		}
	},

	/**
	 * Event handler for the {@link #messageAutoSaveTimer}, this will
	 * {@link Zarafa.common.Actions.messageAutoSave autosave} the {@link #record}.
	 * @private
	 */
	messageAutoSave: function()
	{
		if (this.record) {

			// Check if the response received after resolve-attempt is ambiguous or not, if this is the case then "check names" dialog is still opened,
			// just halt the auto-saving mechanism until user select any record from suggestion.
			var recipientSubStore = this.record.getSubStore('recipients');
			var ambiguityDetected = false;
			recipientSubStore.each(function(record){
				if(record.resolveAttemptAmbiguous) {
					ambiguityDetected = true;
				}
			});

			if(!ambiguityDetected && this.field.autoSave){
				this.field.saveRecord();
			} else {
				this.resetMessageAutoSaveTimer();
				this.startMessageAutoSaveTimer();
			}
		}
	},

	/**
	 * When the record is about to be saved on the {@link Zarafa.core.ui.RecordContentPanel RecordContentPanel}
	 * we should {@link #resetMessageAutoSaveTimer cancel} {@link #messageAutoSaveTimer}
	 * @param {Zarafa.core.ui.RecordContentPanel} contentpanel The contentpanel which invoked the event
	 * @param {Zarafa.core.data.IPMRecord} record The record which is being saved.
	 * @private
	 */
	onSaveRecord: function(contentpanel, record)
	{
		this.resetMessageAutoSaveTimer();
	},

	/**
	 * Event handler for the {@link Ext.Container#destroy destroy} event.
	 * This will {@link #resetMessageAutoSaveTimer cancel} the {@link #messageAutoSaveTimer}
	 * @private
	 */
	onDestroy: function()
	{
		this.resetMessageAutoSaveTimer();
		this.record = undefined;
	}
});

Ext.preg('zarafa.autosavemessageplugin', Zarafa.core.plugins.AutoSaveMessagePlugin);
Ext.ns('Zarafa.core.plugins');

/**
 * @class Zarafa.core.plugins.ComboAutoWidth
 * @extends Ext.util.Observable
 * @ptype zarafa.comboautowidth
 *
 * this plugin will set the width of the combo box to the width of the maximum text in list
 */
Zarafa.core.plugins.ComboAutoWidth = Ext.extend(Ext.util.Observable, {
	/**
	 * Initialize the plugin for the given {@link Ext.form.Field field}
	 * @param {Ext.form.ComboBox} comboBox on which the plugin is placed
	 */
	init: function(combo)
	{
		this.container = combo;

		combo.mon(combo.store, 'load', this.resizeToFitContent, this);
		combo.on('render', this.resizeToFitContent, this);
	},

	/**
	 * Called in response to a load event from the addressbook comboBox store.
	 * resize the combobox innerlist according to the widest list item content
	 * @param {Zarafa.core.data.IPMStore} store store that fired the event.
	 * @private
	 */
	resizeToFitContent: function()
	{
		var combo = this.container;
		var store = combo.getStore();
		var listWidth = 0;
		var textMetrics = Ext.util.TextMetrics.createInstance(combo.el);

		store.each(function(record) {
			var curWidth = textMetrics.getWidth(record.get(combo.displayField));
			if (curWidth > listWidth) {
				listWidth = curWidth;
			}
		});

		if (listWidth > 0) {
			listWidth = Math.max(combo.minListWidth, listWidth);

			listWidth += combo.getTriggerWidth(); // to accommodate combo's down arrow.
			if (combo.getWidth() < listWidth) {
				if (!combo.list) {
					combo.listWidth = listWidth;
				} else {
					combo.list.setWidth(listWidth);
				}
			}
		}
	}
});

Ext.preg('zarafa.comboautowidth', Zarafa.core.plugins.ComboAutoWidth);
Ext.namespace('Zarafa.core.plugins');

/**
 * @class Zarafa.core.plugins.ComponentTooltipPlugin
 * @extends Object
 * @ptype zarafa.componenttooltipplugin
 *
 * This plugin is use to set the tooltip on {@link Ext.Component Component}.
 */
Zarafa.core.plugins.ComponentTooltipPlugin = Ext.extend(Object,{

	/**
	 * @constructor
	 * @param {Object} config Configuration object
	 */
	constructor: function(config)
	{
		config = config || {};
		Ext.apply(this, config);
	},

	/**
	 * Initializes the {@link Ext.Component Component} to which this plugin has been hooked.
	 * @param {Ext.menu.Item} field The field on which the plugin is installed.
	 */
	init: function(field)
	{
		this.field = field;
	}
});

Ext.preg('zarafa.componenttooltipplugin', Zarafa.core.plugins.ComponentTooltipPlugin);Ext.namespace('Zarafa.core.plugins');

/**
 * @class Zarafa.core.plugins.ContentLayerPlugin
 * @extends Object
 * @ptype zarafa.contentlayerplugin
 *
 * The Plugin which must be installed on the {@link Zarafa.core.ui.ContentPanel}
 * to glue it together with the Content Layer on which the component is being
 * installed. This ensures that events and actions performed on the
 * {@link Zarafa.core.ui.ContentPanel panel} are forwarded correctly to the content layer.
 */
Zarafa.core.plugins.ContentLayerPlugin = Ext.extend(Object, {
	/**
	 * The Content Panel on which this plugin is installed.
	 * @property
	 * @type Zarafa.core.ui.ContentPanel
	 * @protected
	 */
	field: undefined,

	/**
	 * @constructor
	 * @param {Object} config Configuration object
	 */
	constructor: function(config)
	{
		Ext.apply(this, config);
	},

	/**
	 * Initialize the plugin with the corresponding field.
	 * @param {Zarafa.core.ui.ContentPanel} field The panel on which the plugin is installed
	 */
	init: function(field)
	{
		this.field = field;

		// Listen for the moment the field was added to the parent container,
		// at that moment we can continue initializing.
		this.field.on('added', this.onAdded, this, { single: true });
	},

	/**
	 * This is called after {@link #init} when the {@link #field} has been fully initialized,
	 * and the owner of the component is known. This allows the plugin to access the parent.
	 *
	 * This will also {@link #focus} the component, to ensure it is visible for the user.
	 *
	 * @protected
	 */
	initPlugin: function()
	{
		this.focus();

		this.field.on('titlechange', this.onTitleChange, this);
		this.field.on('iconchange', this.onIconChange, this);
		this.field.on('hide', this.onHide, this);
		this.field.on('close', this.onClose, this);
	},

	/**
	 * Event handler for the {@link Ext.Component#added 'added'} event. At this moment
	 * we have access to the parent container in which the item was placed, but not
	 * all initialization might have been done yet. So this will register the
	 * {@link #onContainerAdd} event handler for the {@Link Ext.Component#add} event
	 * which is fired when the component is completely initialized.
	 * @param {Ext.Component} item The item which fired the event
	 * @param {Ext.Container} container The container on which the item was placed
	 * @private
	 */
	onAdded: function(item, container)
	{
		container.on('add', this.onContainerAdd, this);
	},

	/**
	 * Event handler for the {@link Ext.Container#add 'add'} event. At this moment
	 * the component must have been fully initialized so we can {@link #initPlugin finalize plugin initialization}.
	 * @param {Ext.Container} container The container which fired the event
	 * @param {Ext.Component} component The component which was added
	 * @private
	 */
	onContainerAdd: function(container, item)
	{
		if (this.field === item) {
			container.un('add', this.onContainerAdd, this);
			this.initPlugin();
		}
	},

	/**
	 * Event handler for the 'titlechange' event, fired by a {@link Ext.Component} that is managed by this plugin.
	 * @param {Zarafa.core.ui.ContentPanel} field The component that triggered the event
	 * @param {String} newTitle Title to set
	 * @param {String} oldTitle Previous title
	 * @private
	 */
	onTitleChange: function(field, newTitle, oldTitle)
	{
		this.setTitle(newTitle);
	},

	/**
	 * Event handler for the 'iconchange' event
	 * @param {Zarafa.core.ui.ContentPanel} field The component that triggered the event
	 * @param {String} newTitle Newly set title
	 * @param {String} oldTitle The previous title
	 * @private
	 */
	onIconChange: function(field, newIconCls, oldIconCls)
	{
		this.setIconClass(newIconCls);
	},

	/**
	 * Event handler for the 'hide' event
	 * @private
	 */
	onHide: function()
	{
		this.hide();
	},

	/**
	 * Event handler for the 'close' event
	 * @private
	 */
	onClose: function()
	{
		this.close();
	},

	/**
	 * This will apply the required title onto the {@link #field}
	 *
	 * Must be overridden by the subclasses
	 *
	 * @param {String} title The title to apply
	 * @protected
	 */
	setTitle: Ext.emptyFn,

	/**
	 * This will apply the required Icon Class onto the {@link #field}
	 *
	 * Must be overridden by the subclasses
	 *
	 * @param {String} iconCls The Icon Class to apply
	 * @protected
	 */
	setIconClass: Ext.emptyFn,

	/**
	 * This will hide panel containing the {@link #field}
	 *
	 * Must be overridden by the subclasses
	 *
	 * @protected
	 */
	hide: Ext.emptyFn,

	/**
	 * This will close panel containing the {@link #field}
	 *
	 * Must be overridden by the subclasses
	 *
	 * @protected
	 */
	close: Ext.emptyFn,

	/**
	 * This will bring focus to the Container by bringing it
	 * to the attention of the user.
	 *
	 * Must be overridden by subclasses
	 *
	 * @protected
	 */
	focus: Ext.emptyFn
});

Ext.preg('zarafa.contentlayerplugin', Zarafa.core.plugins.ContentLayerPlugin);
Ext.namespace('Zarafa.core.plugins');

/**
 * @class Zarafa.core.plugins.EnableFocusPlugin
 * @extends Object
 * @ptype zarafa.enablefocusplugin
 *
 * Normally div elements or span elements are not able to listen keyboard events because they
 * are non focusable elements,
 * So this plugin will add anchor element under the component's element, Which will get focus
 * when user clicks on the field, and anchor field will listen key events and will relay it to
 * parent component.
 * Adding this plugin to {@link Ext.Container} will intercept {@link Ext.Container#focus} event and
 * do focus handling itself.
 */
Zarafa.core.plugins.EnableFocusPlugin = Ext.extend(Object, {
	/**
	 * We are creating this element as anchor element, div elements can't get focus so we won't be able to
	 * handle keyboard in the panel, so when panel gets clicked we will set focus on this anchor element,
	 * to get keyboard events.
	 * @property
	 * @type Ext.Element
	 */
	focusEl: undefined,

	/**
	 * @constructor
	 * @param {Object} config Configuration object
	 */
	constructor: function(config)
	{
		Ext.apply(this, config);
	},

	/**
	 * Initializes the {@link Ext.Component Component} to which this plugin has been hooked.
	 * @param {Ext.Component} field The component on which the plugin is installed
	 */
	init: function(field)
	{
		this.field = field;
		field.enableFocusPlugin = this;

		// override focus method of field to our own implementation
		// so focusing field should also transfer focus to the focusEl
		// and we can just ignore handling of focus in field
		this.field.focus = this.field.focus.createInterceptor(this.focus, this);

		this.field.on('render', this.onRender, this);
	},

	/**
	 * Function will intercept focus of the {@link Ext.Component Component} and will set it to
	 * {@link #focusEl} so we can listen for the key events and do keyboard handling.
	 * This function will return boolean value to indicate that intercepted function should be called or not,
	 * If the plugin is able to focus then it will return false and field's focus method will not be called
	 * and if plugin is not able to do focus then it will return true so field's focus method will be called.
	 * This return value is only used when this function is called as interceptor function.
	 * @private
	 */
	focus: function()
	{
		if(this.field.rendered && !this.field.isDestroyed) {
			if(this.focusEl) {
				// Creating a selection will also be seen as a click (because it is), but
				// Firefox will drop the current selection when we set the focus to the focusEl.
				// Fortunately FireFox has implemented most of the Selection api. So we can get
				// the current selection and put it back after we focus the focusEl.
				var selection = this.field.getEl().dom.ownerDocument.getSelection();
				var ranges = [];
				// Get all selection ranges
				if ( selection && selection.isCollapsed===false && selection.getRangeAt && selection.addRange ){
					for ( var i=0; i<selection.rangeCount; i++ ){
						ranges.push(selection.getRangeAt(i));
					}
				}

				this.focusEl.focus();

				// Add all ranges to the selection again
				Ext.each(ranges, function(range){
					selection.addRange(range);
				});

				// we have handled focus here so don't allow intercepted function to do its handling
				return false;
			}
		}

		// intercepted function will do its default handling
		return true;
	},

	/**
	 * Event handler for the {@link Ext.Component#render render} event
	 * on the {@link #field}. This will add anchor element under the component's element.
	 * This will add a mouse-click listener on field when it is clicked it will set focus
	 * on {@link #focusEl}, so once focus is set on {@link #focusEl} keyboard events can be
	 * listened by the components which register this plugin.
	 */
	onRender: function()
	{
		if(this.field) {
			var element = this.field.getEl();

			// this element allows the panel to be focused for keyboard events
			this.focusEl = element.createChild({
				tag: 'a',
				href: '#',
				// Disable tab-index, and position it somewhere where it cannot be seen
				// by the user. This will make the element completely invisible for the
				// user while we can benefit from the focus capabilities.
				tabindex: -1,
				style: 'position: absolute; left:-10000px; top:-10000px;'
			});

			this.focusEl.swallowEvent('click', true);

			this.field.mon(element, {
				click: this.onFieldClick,
				scope: this
			});
		}
	},

	/**
	 * Event handler which is fired when user clicks anywhere on panel,
	 * this will set the focus on {@link #focusEl} so that keyboard events can be fired on the component.
	 * @param {Ext.EventObject} eventObj The {@link Ext.EventObject} encapsulating the DOM event.
	 * @param {HtmlElement} element The target of the event.
	 * @param {Object} opt The options configuration passed to the event call.
	 * @private
	 */
	onFieldClick: function(eventObj, element, opt)
	{
		var focusedElement = Ext.get(document.activeElement);
		var clickedElement = Ext.get(element);

		/*
		 * Here we need to handle Keyevents that are fired on this component so we
		 * need to set focus on this element, now if any of the sub-component
		 * already has focus then we don't need to set focus on this component,
		 * as Keyevents will be propagated to this component, So here we are checking
		 * that whether user has clicked on any input field e.g. textfield, or focus
		 * is set on any sub-component.
		 */
		if(focusedElement != clickedElement) {
			var fieldEl = this.field.getEl();

			if(!fieldEl.contains(focusedElement)) {
				this.focus();
			}
		}
	}
});

Ext.preg('zarafa.enablefocusplugin', Zarafa.core.plugins.EnableFocusPlugin);
Ext.namespace('Zarafa.core.plugins');

/**
 * @class Zarafa.core.plugins.InputAutoFocusPlugin
 * @extends Object
 * @ptype zarafa.inputautofocusplugin
 *
 * A special plugin which, when placed on a {@link Ext.Panel} will
 * automatically select the first available Input element it can find.
 */
Zarafa.core.plugins.InputAutoFocusPlugin = Ext.extend(Object, {
	/**
	 * @cfg {Ext.Component} autoFocus The component which should
	 * be focused when the {@link #field} has been {@link Ext.Container#afterlayout laid out}
	 * or {@link Ext.Container#activate activated}.
	 */
	autoFocus: undefined,

	/**
	 * The field on which this plugin has been installed
	 * @property
	 * @type Ext.Container
	 */
	field: undefined,

	/**
	 * The extra {@link Ext.Element} which is inserted at the start of the {@link #field container},
	 * this is used to apply a {@link #doCyclicFocus cyclic focus} to prevent the focus
	 * to leave the {@link #field container}.
	 * @property
	 * @type Ext.Element
	 */
	beginFocusEl: undefined,

	/**
	 * The extra {@link Ext.Element} which is inserted at the end of the {@link #field container},
	 * this is used to apply a {@link #doCyclicFocus cyclic focus} to prevent the focus
	 * to leave the {@link #field container}.
	 * @property
	 * @type Ext.Element
	 */
	endFocusEl: undefined,

	/**
	 * @constructor
	 * @param {Object} config Configuration object
	 */
	constructor: function(config)
	{
		Ext.apply(this, config);
	},

	/**
	 * Called by the {@link Ext.Container} when the plugin is being registered.
	 * @param {Ext.Container} field The field into which this plugin is installed
	 */
	init: function(field)
	{
		this.field = field;
		field.inputAutoFocusPlugin = this;

		this.field.on('afterrender', this.onAfterRender, this);
		this.field.on('afterlayout', this.onAfterFirstLayout, this, { single: true });
		this.field.on('destroy', this.onDestroy, this);
	},

	/**
	 * Event handler which is fired after the {@link #field} has been {@link Ext.Container#afterrender rendered}.
	 * This will create the {@link #beginFocusEl} and {@link #endFocusEl} elements to enable {@link #doCyclicFocus}
	 * cyclic focussing.
	 * @private
	 */
	onAfterRender: function()
	{
		this.beginFocusEl = Ext.DomHelper.insertBefore(this.field.el, {
			tag: 'a',
			href: '#',
			style: 'position: absolute; text-decoration: none; font-height: 1px; width: 1px; height: 1px; left:-10000px; top:-10000px;'
		}, true);
		this.beginFocusEl.dom.innerHTML = "&nbsp;";
		this.beginFocusEl.on('focus', this.onLimitFocussed, this);

		this.endFocusEl = Ext.DomHelper.insertAfter(this.field.el, {
			tag: 'a',
			href: '#',
			style: 'position: absolute; text-decoration: none; font-height: 1px; width: 1px; height: 1px; left:-10000px; top:-10000px;'
		}, true);
		this.endFocusEl.dom.innerHTML = "&nbsp;";
		this.endFocusEl.on('focus', this.onLimitFocussed, this);
		// Check if the field is a child of the Ext.Window, if a focus element exists on it,
		// then we need to intercept its focus and move it to the form
		var win = this.field.findParentByType('window');
		if (win && win.focusEl) {
			win.focusEl.on('focus', this.onDialogFocussed, this);
		}

	},

	/**
	 * Event handler which is fired after the {@link #field} has been {@link Ext.Container#afterlayout laid out}.
	 * This will check if the {@link #field} is a tab in a {@link Ext.TabPanel} or a normal {@link Ext.Container}.
	 * When it is a tab, it will wait for the {@link Ext.Container#activate tab-activation}, otherwise it will
	 * start the {@link #doAutoFocus}.
	 * @private
	 */
	onAfterFirstLayout: function()
	{
		if (Ext.isDefined(this.field.tabEl)) {
			this.field.on('activate', this.onActivate, this);
		} else {
			// This field isn't a tab, but perhaps one of the child elements is...
			var tabs = this.field.findBy(function(cmp) { return Ext.isDefined(cmp.tabEl); });
			if (!Ext.isEmpty(tabs)) {
				for (var i = 0, len = tabs.length; i < len; i++) {
					this.field.mon(tabs[i], 'activate', this.onActivate, this);
				}
			}
		}
	},

	/**
	 * Event handler which is fired after the {@link #field} has been {@link Ext.Container#activate activated}.
	 * This will start the {@link #doAutoFocus}.
	 * @private
	 */
	onActivate: function()
	{
		// Apply a 1ms delay, otherwise we cannot detect which input field
		// is hidden or not.
		this.doAutoFocus.defer(1, this);
	},

	/**
	 * Called when either the {@link #beginFocusEl} or {@link #endFocusEl} has been focussed.
	 * This will start the {@link #doCyclicFocus}.
	 * @param {Ext.EventObject} event The event object
	 * @param {HTMLElement} element The element which was focussed
	 * @private
	 */
	onLimitFocussed: function(event, element)
	{
		this.doCyclicFocus(this.beginFocusEl.dom === element);
	},

	/**
	 * Event handler which is called when the {@link #field} is being destroyed.
	 * This will automatically destroy the {@link #beginFocusEl} and {@link #endFocusEl}.
	 * @private
	 */
	onDestroy: function()
	{
		Ext.destroy(this.beginFocusEl);
		Ext.destroy(this.endFocusEl);
	},

	/**
	 * Change the {@link #autoFocus} property and automatically
	 * have the focus {@link #doAutoFocus} updated to this component.
	 * @param {Mixed} autoFocus The replacement for the {@link #autoFocus} object
	 */
	setAutoFocus: function(autoFocus)
	{
		this.autoFocus = autoFocus;
		this.doAutoFocus();
	},

	/**
	 * Called by {@link #onAfterFirstLayout} and {@link #onActivate}. If {@link #autoFocus} has been
	 * provided, this will apply the focus to that component, otherwise it will search for the first
	 * {@link #isFocusElement focussable element} inside the {@link #field container} (and the sub-containers).
	 * @private
	 */
	doAutoFocus: function()
	{
		var focusCmp;

		if (this.autoFocus) {
			if (Ext.isString(this.autoFocus)) {
				// Perhaps the string is a property on our field
				focusCmp = this.field[this.autoFocus];
				if (!focusCmp) {
					// Perhaps the string is a xtype of a child component
					var children = this.field.findByType(this.autoFocus);
					if (!Ext.isEmpty(children)) {
						focusCmp = children[0];
					}
				}
				if (!focusCmp) {
					// Perhaps the string is an ID of the HTML element
					focusCmp = Ext.get(this.autoFocus);
				}
			} else if (Ext.isElement(this.autoFocus) || Ext.isFunction(this.autoFocus.focus)) {
				focusCmp = this.autoFocus;
			}
		} else {
			focusCmp = this.findFocusElement(this.field);
			// If no focus element was found, then lets try
			// if we can focus on a button instead.
			if (!focusCmp) {
				focusCmp = this.findFocusElement(this.field, false, true);
			}
		}

		if (focusCmp) {
			focusCmp.focus.defer(1, focusCmp);
			if(focusCmp.events['setAutoFocusCursor']){
				focusCmp.fireEvent('setAutoFocusCursor', focusCmp.getEditor());
			}
		}
	},

	/**
	 * Called when the {@link #beginFocusEl} or {@link #endFocusEl} elements have been focussed.
	 * This will apply a cycle to the focus, making sure the focus will not leave the {@link #field}
	 * in which this plugin is installed.
	 * @param {Boolean} inverse Search for the last focussable item in the container rather then
	 * the first.
	 * @private
	 */
	doCyclicFocus: function(inverse)
	{
		var focusCmp;

		focusCmp = this.findFocusElement(this.field, inverse, true);
		if (focusCmp) {
			focusCmp.focus.defer(1, focusCmp);
		}
	},

	/**
	 * Search through the given container to search for the first {@link #isFocusElement focussable element}
	 * inside the given container (and the sub-containers).
	 * @param {Ext.Container} container The container through which we must search to find the first
	 * focussable element
	 * @param {Boolean} inverse True to search for the last Focus Element
	 * @param {Boolean} allowButton Allow a button to be considered an focussable input element
	 * @return {Ext.Component} The focussable component which was found in this container
	 * @private
	 */
	findFocusElement: function(container, inverse, allowButton)
	{
		var objects = [];

		if (container.topToolbar) {
			objects = objects.concat(container.topToolbar.items.items);
		}
		if (container.items) {
			objects = objects.concat(container.items.items);
		}
		if (container.buttons) {
			objects = objects.concat(container.buttons);
		}

		if (inverse) {
			objects.reverse();
		}

		for (var i = 0, len = objects.length; i < len; i++) {
			var item = objects[i];
			var focus;

			if (this.isContainer(item)) {
				if (item.isVisible()) {
					focus = this.findFocusElement(item, inverse, allowButton);
				}
			} else if (this.isFocusElement(item, allowButton)) {
				focus = item;
			}

			if (focus) {
				return focus;
			}
		}
	},

	/**
	 * Check if the given {@link Ext.Component} is a container which contains items through which
	 * we can search for a {@link #isFocusElement focussable element}.
	 * @param {Ext.Component} item The component we are checking if it is a container
	 * @return {Boolean} True if the item is a container, false otherwise
	 * @private
	 */
	isContainer: function(item)
	{
		return (item instanceof Ext.Container || item instanceof Ext.form.CompositeField);
	},

	/**
	 * Check if the given {@link Ext.Component} is a visible input-element which can be focussed.
	 * This includes that the size of the component is non-0, the element itself is visible and
	 * not readonly input element.
	 * @param {Ext.Component} item The component we are checking
	 * @param {Boolean} allowButton Allow a button to be considered an focussable input element
	 * @return {Boolean} True if the item is focussable, false otherwise
	 */
	isFocusElement: function(item, allowButton)
	{
		var el = item.btnEl || item.el;

		if (!el) {
			return false;
		}

		var disallowedType = (allowButton === true) ? /submit|reset|hidden/i : /button|submit|reset|hidden/i;
		var allowedTagname = (allowButton === true) ? /button|input|textarea|select/i : /input|textarea|select/i;

		return (el.dom.style.height !== '0px' &&
				el.dom.style.width !== '0px' &&
				el.dom.style.display !== 'none' &&
				el.dom.style.visibility !== 'hidden' &&
				!el.dom.disabled &&
				!el.dom.getAttribute('readonly') &&
				!disallowedType.test(el.dom.type) &&
				allowedTagname.test(el.dom.tagName));
	},

	/**
	 * Call this when Ext.JS moves the focus to its own focus element a#x-dlg-focus (see source of {@link Ext.Window#render})
	 * Call {@link #doAutoFocus} to move focus to the first form field
	 */
	onDialogFocussed: function()
	{
		this.doAutoFocus();
	}
});

Ext.preg('zarafa.inputautofocusplugin', Zarafa.core.plugins.InputAutoFocusPlugin);
Ext.namespace('Zarafa.core.plugins');

/**
 * @class Zarafa.core.plugins.MarkAsReadPlugin
 * @extends Object
 * @ptype zarafa.markasreadplugin
 *
 * Special plugin which must be used in combination with the
 * {@link Zarafa.core.plugins.RecordComponentPlugin}. This will
 * mark the message as read after a certain time.
 */
Zarafa.core.plugins.MarkAsReadPlugin = Ext.extend(Object, {
	/**
	 * The component which has been {@link #init initialized} on
	 * this plugin.
	 * @property
	 * @type Ext.Component
	 */
	field: undefined,

	/**
	 * The record which has been {@link #onSetRecord set} on the
	 * {@link #field}.
	 * @property
	 * @type Zarafa.core.data.MAPIRecord
	 */
	record: undefined,

	/**
	 * timer that is used to set message as read after specified seconds
	 * @property
	 * @type Number
	 */
	readFlagTimer: undefined,

	/**
	 * Indicate that setting/starting up the {@link #readFlagTimer} will be
	 * ignored and record will be marked as read directly.
	 * @property
	 * @type Boolean
	 */
	ignoreReadFlagTimer: undefined,

	/**
	 * @constructor
	 * @param {Object} config Configuration object
	 */
	constructor: function(config)
	{
		Ext.apply(this, config);
	},

	/**
	 * Initializes the {@link Ext.Component Component} to which this plugin has been hooked.
	 * @param {Ext.Component} field The component on which the plugin is installed
	 */
	init: function(field)
	{
		this.field = field;
		field.markAsReadPlugin = this;

		this.field.on('render', this.onRender, this);

		if (field.addInternalAction) {
			field.addInternalAction('send_read_receipt');
		}
	},

	/**
	 * Event handler for the {@link Ext.Component#render render} event
	 * on the {@link #field}. This will register the event handlers which
	 * are needed to listen for record changes.
	 */
	onRender: function()
	{
		this.field.on({
			'setrecord': this.onSetRecord,
			'beforesetrecord': this.onBeforeSetRecord,
			'loadrecord': this.onLoadRecord,
			'close': this.onDestroy,
			'destroy': this.onDestroy,
			'scope': this
		});
	},

	/**
	 * Event handler for the {@link Zarafa.core.plugins.RecordComponentPlugin#setrecord setrecord} event
	 * on the {@link #field}. This will {@link #resetReadFlagTimer stop} the {@link #readFlagTimer}.
	 * @param {Ext.Component} field The component which fired the event
	 * @param {Zarafa.core.data.MAPIRecord} record The new record which is being set
	 * @param {Zarafa.core.data.MAPIRecord} oldrecord The old record which was previously set
	 * @private
	 */
	onSetRecord: function(field, record, oldrecord)
	{
		// If the record is the same, the data of the record was just updated. No need to mark it as
		// read/unread then.
		if ( record && oldrecord && record.equals(oldrecord) ){
			return;
		}

		this.resetReadFlagTimer();

		// here, parameter named 'record' is the instance of shadowstore record.
		if(record && record.isOpened()) {
			// if shadowstore record is opened then fire load event on original record(record of list view)
			// as we are using that record to mark it as read.
			this.onLoadRecord(this.field, this.record);
		}
	},

	/**
	 * Event handler for the {@link Zarafa.core.plugins.RecordComponentPlugin#beforesetrecord beforesetrecord} event
	 * on the {@link #field}. This will get the original record because it is {@link Ext.data.Record#copy copied} within
	 * {@link Zarafa.core.plugins.RecordComponentPlugin#setRecord setRecord} method to start shadow editing.
	 * @param {Ext.Component} field The component which fired the event
	 * @param {Zarafa.core.data.MAPIRecord} record The new record which is being set
	 * @param {Zarafa.core.data.MAPIRecord} oldrecord The old record which was previously set
	 * @private
	 */
	onBeforeSetRecord: function(field, record, oldrecord)
	{

		this.resetReadFlagTimer();

		// Only IPMRecords will be handled by this plugin,
		// all other records will be discarded.
		if (record instanceof Zarafa.core.data.IPMRecord) {
			this.record = record;

			var store = this.record.getStore();

			if(Ext.isDefined(store)) {
				this.field.mon(store, 'load', this.onStoreLoad, this);
				this.field.mon(store, 'remove', this.onStoreRemove, this);
			}

		} else {
			this.record = undefined;
		}
	},

	/**
	 * Called when {@link Zarafa.mail.ui.MailGrid#store store} is loading new data.
	 * This will {@link Zarafa.core.plugins.MarkAsReadPlugin#resetReadFlagTimer reset}
	 * the {@link Zarafa.core.plugins.MarkAsReadPlugin#readFlagTimer}.
	 *
	 * @param {Zarafa.core.data.MAPIStore} store The store which being loaded.
	 * @param {Ext.data.Record[]} records The records which have been loaded from the store
	 * @param {Object} options The loading options that were specified (see {@link Ext.data.Store#load load} for details)
	 * @private
	 */
	onStoreLoad: function(store, records, options)
	{
		this.resetReadFlagTimer();
	},

	/**
	 * Event handler which is fired when the {@link Zarafa.mail.ui.MailGrid#store store} fires the
	 * remove event. This will {@link Zarafa.core.plugins.MarkAsReadPlugin#resetReadFlagTimer reset}
	 * the {@link Zarafa.core.plugins.MarkAsReadPlugin#readFlagTimer} if
	 * {@link Zarafa.core.plugins.MarkAsReadPlugin#record record} is removed
	 * from {@link Zarafa.mail.ui.MailGrid#store store}
	 *
	 * @param {Zarafa.hierarchy.data.HierarchyStore} store The store which fired the event
	 * @param {Zarafa.hierarchy.data.MAPIStoreRecord} record The record which was deleted
	 * @private
	 */
	onStoreRemove: function(store, record)
	{
		// Check if the removed record is the one which referred by the markAsRead plugin
		if (this.record === record) {
			this.resetReadFlagTimer();
		}
	},

	/**
	 * Event handler for the {@link Zarafa.core.plugins.RecordComponentPlugin#loadrecord loadrecord} event
	 * on the {@link #field}. This will {@link #startReadFlagTimer start} the {@link #readFlagTimer}.
	 * @param {Ext.Component} field The component which fired the event
	 * @param {Zarafa.core.data.MAPIRecord} record The record which was updated
	 * @private
	 */
	onLoadRecord: function(field, record)
	{
		if (record && !record.phantom && !record.isRead()) {
			// decide based on flag if we want to instantly mark as read or start the timer
			if (this.ignoreReadFlagTimer) {
				this.markAsRead();
			} else {
				this.startReadFlagTimer();
			}
		}
	},

	/**
	 * Start the {@link #readFlagTimer} if it was not yet fired already.
	 * This will check the settings for the desired timeout, and delay the call
	 * to {@link #markAsRead} as configured.
	 * @private
	 */
	startReadFlagTimer: function()
	{
		if (!this.readFlagTimer) {
			var timeout = container.getSettingsModel().get('zarafa/v1/contexts/mail/readflag_time') * 1000;
			this.readFlagTimer = this.markAsRead.defer(timeout, this);
		}
	},

	/**
	 * Stop the {@link #readFlagTimer} if it is still pending.
	 * This will clear the timeout and delete the {@link #readFlagTimer} making it
	 * available again for rescheduling.
	 * @private
	 */
	resetReadFlagTimer: function()
	{
		if (this.readFlagTimer) {
			window.clearTimeout(this.readFlagTimer);
			delete this.readFlagTimer;
		}
	},

	/**
	 * Event handler for the {@link #readFlagTimer}, this will mark the {@link #record}
	 * as {@link Zarafa.common.Actions.markAsRead read}.
	 * @private
	 */
	markAsRead: function()
	{
		if (this.record && !Ext.isEmpty(this.record.getStore())) {
			Zarafa.common.Actions.markAsRead(this.record);
		}
		delete this.readFlagTimer;
	},

	/**
	 * Event handler for the {@link Ext.Container#destroy destroy} event.
	 * This will {@link #resetReadFlagTimer cancel} the {@link #readFlagTimer} to
	 * prevent the record from being marked as read.
	 * @private
	 */
	onDestroy: function()
	{
		this.resetReadFlagTimer();
		this.record = undefined;
	}
});

Ext.preg('zarafa.markasreadplugin', Zarafa.core.plugins.MarkAsReadPlugin);
Ext.namespace('Zarafa.core.plugins');

/**
 * @class Zarafa.core.plugins.MenuItemTooltipPlugin
 * @extends Object
 * @ptype zarafa.menuitemtooltipplugin
 *
 * This plugin is use to set the tooltip on {@link Ext.menu.Item menuitem}
 * of {@link Ext.splitButton SplitButton}.
 */
Zarafa.core.plugins.MenuItemTooltipPlugin = Ext.extend(Zarafa.core.plugins.ComponentTooltipPlugin,{

	/**
	 * @constructor
	 * @param {Object} config Configuration object
	 */
	constructor: function(config)
	{
		config = config || {};
		Ext.apply(this, config);

		Zarafa.core.plugins.MenuItemTooltipPlugin.superclass.constructor.call(this, config);
	},

	/**
	 * Initializes the {@link Ext.Component Component} to which this plugin has been hooked.
	 * @param {Ext.menu.Item} field The field on which the plugin is installed.
	 */
	init: function(field)
	{
		Zarafa.core.plugins.MenuItemTooltipPlugin.superclass.init.apply(this, arguments);
		// Add event listener for the 'activate' event, if we are move the cursor on menu item then the
		// tooltip is display for particular menu item.
		this.field.on('activate', this.applyTooltip, this);
	},

	/**
	 * Used to apply the tooltip on {@link Ext.menu.Item menuitem} of {@link Ext.SplitBtton SplitBtton}
	 * @param {Ext.menu.Item} itemMenu The menu item of {@link Ext.SplitButton splitbutton}
	 */
	applyTooltip: function(itemMenu)
	{
		if(Ext.isDefined(itemMenu.tooltip)){
			Ext.QuickTips.unregister(itemMenu.getEl());
			if(Ext.isObject(itemMenu.tooltip)){
				Ext.QuickTips.register(Ext.apply({
					target: itemMenu.getEl().id
				}, itemMenu.tooltip));
			} else {
				itemMenu.getEl().dom.qtip = itemMenu.tooltip;
			}
		}
	}
});

Ext.preg('zarafa.menuitemtooltipplugin', Zarafa.core.plugins.MenuItemTooltipPlugin);Ext.namespace('Zarafa.core.plugins');

/**
 * @class Zarafa.core.plugins.RecordComponentPlugin
 * @extends Object
 * @ptype zarafa.recordcomponentplugin
 *
 * This plugin is used on a root {@link Ext.Container container} in which multiple children are
 * located which share a {@link Zarafa.core.data.MAPIRecord record} for displaying or editing.
 * This can for example occur in a {@link Ext.Window Dialog} which is used for editing a message,
 * or in the {@link Zarafa.core.ui.PreviewPanel PreviewPanel} for displaying a message.
 *
 * If the {@link #allowWrite} option is enabled, the {@link #record} will be editable, this will be done
 * by copying the {@link #record} into the {@link Zarafa.core.data.ShadowStore ShadowStore}
 * from where it can be saved to the server.
 *
 * The components located in the root {@link Ext.Container container} can use the
 * {@link Zarafa.core.ui.RecordComponentUpdaterPlugin RecordComponentUpdaterPlugin} for handling the notifications
 * from this plugin.
 *
 * When this plugin is being installed, it will set a reference to itself on the {@link #field}
 * with the name 'recordComponentPlugin'.
 */
Zarafa.core.plugins.RecordComponentPlugin = Ext.extend(Object, {
	/**
	 * The contained on which this plugin has been installed.
	 * @property
	 * @type Ext.Container
	 */
	field: undefined,

	/**
	 * @cfg {Boolean} ignoreUpdates True if the {@link #field} should not be listening to changes
	 * made to the {@link #record} from a {@link Zarafa.core.data.MAPIStore Store}. This will force
	 * the {@link #useShadowStore} option to be enabled.
	 */
	ignoreUpdates: false,

	/**
	 * @cfg {Boolean} allowWrite True if the {@link #field} supports
	 * the editing and saving of the {@link #record}.
	 */
	allowWrite: false,

	/**
	 * True when the {@link #field} has been laid out.
	 * @property
	 * @type Boolean
	 */
	isLayoutCalled: false,

	/**
	 * The record which is currently displayed in {@link #field}.
	 * @property
	 * @type Zarafa.core.data.MAPIRecord
	 */
	record: undefined,

	/**
	 * The cheapCopy argument of {@link #setRecord}, this is used when {@link #setRecord} was
	 * called when the panel has not been rendered yet and the call is deferred.
	 * @property
	 * @type Boolean
	 */
	cheapCopy: undefined,

	/**
	 * @cfg {Boolean} useShadowStore True, if the {@link #record} should be pushed into the
	 * shadowStore.
	 */
	useShadowStore: false,

	/**
	 * @cfg {Ext.data.Store} shadowStore is data store, it allows to use custom shadow store
	 * by record component plugin. This config mostly used by the external plugins like Files plugin.
	 */
	shadowStore: undefined,

	/**
	 * Indicates if the {@link #record} has been changed by the user since it has been loaded.
	 * If this is changed to true for the first time, the {@link #userupdaterecord} event will be
	 * fired on the {@link #field}.
	 * @property
	 * @type Boolean
	 * @private
	 */
	isChangedByUser: false,

	/**
	 * @cfg {Array} loadTasks An array of objects containing tasks which should be executed
	 * in order to properly load all data into the Component. The object should contain the
	 * 'fn' field which contains the function to be called, this will be called with the following
	 * arguments:
	 * - container {@link Ext.Container} The container on which this plugin is installed
	 * - record {@link Zarafa.core.data.MAPIRecord} The record which is being loaded
	 * - task {@link Object} The task object which was registered
	 * - callback {@link Function} The callback function which should be called when the task
	 *  has completed its work.
	 * The task should also contain the 'scope' property which refers to the scope in which the
	 * 'fn' should be called. If the task contains the 'defer' property, the call to 'fn' will
	 * be deferred for the given number of milliseconds.
	 */
	loadTasks: undefined,

	/**
	 * @cfg {Boolean} enableOpenLoadTask True to create a {@link #loadTask} which will
	 * {@link Zarafa.core.data.MAPIRecord#open open} the {@link #record} when it is set
	 * on the {@link #field}.
	 */
	enableOpenLoadTask: true,

	/**
	 * @cfg {Number} autoOpenLoadTaskDefer The 'defer' configuration option for the task
	 * which will be put in {@link #loadTasks} when {@link #enableOpenLoadTask} is true.
	 * This can be used to delay the opening of the record.
	 */
	autoOpenLoadTaskDefer: 0,

	/**
	 * List of {@link Ext.util.DelayedTask} instances which will be filled in when
	 * tasks in {@link #loadTasks} was configured with a 'defer' property.
	 * @property
	 * @type Array
	 * @private
	 */
	scheduledTasks: undefined,

	/**
	 * The list of Tasks from {@link #loadTasks} which have been started but are awaiting
	 * the call to the callback function.
	 * @property
	 * @type Array
	 * @private
	 */
	pendingTasks: undefined,

	/**
	 * @constructor
	 * @param {Object} config Configuration object
	 */
	constructor: function(config)
	{
		config = config || {};

		// Apply a default value for the useShadowStore
		// based on the other settings.
		if (!Ext.isDefined(config.useShadowStore)) {
			if (config.allowWrite === true) {
				config.useShadowStore = true;
			}
			if (config.ignoreUpdates === true) {
				config.useShadowStore = true;
			}
		}

		Ext.applyIf(config, {
			shadowStore: container.getShadowStore()
		});

		Ext.apply(this, config);
	},

	/**
	 * Initializes the {@link Ext.Component Component} to which this plugin has been hooked.
	 * @param {Ext.Component} The parent field to which this component is connected
	 */
	init: function(field)
	{
		this.field = field;
		field.recordComponentPlugin = this;

		this.field.addEvents(
			/**
			 * @event beforesetrecord
			 *
			 * Fired from {@link #field}
			 *
			 * Fires when a record is about to be added to the {@link #field}
			 * This will allow Subclasses modify the {@link Zarafa.core.data.MAPIRecord record} before
			 * it has been hooked into the {@link #field}.
			 * @param {Ext.Container} panel The panel to which the record was set
			 * @param {Zarafa.core.data.MAPIRecord} record The record which is going to be set
			 * @param {Zarafa.core.data.MAPIRecord} oldrecord The oldrecord which is currently set
			 * @return {Boolean} false if the record should not be set on this panel.
			 */
			'beforesetrecord',
			/**
			 * @event setrecord
			 *
			 * Fired from {@link #field}
			 *
			 * Fires when a record has been added to the {@link #field}.
			 * No event handler may modify any properties inside the provided record (if this
			 * is needed for the Panel initialization, use the {@link #beforesetrecord} event).
			 * @param {Ext.Container} panel The panel to which the record was set
			 * @param {Zarafa.core.data.MAPIRecord} record The record which was set
			 * @param {Zarafa.core.data.MAPIRecord} oldrecord The oldrecord which was previously set
			 */
			'setrecord',
			/**
			 * @event beforeloadrecord
			 *
			 * Fired from {@link #field}
			 *
			 * Fires when a record is going to be {@link Zarafa.core.data.MAPIRecord#isOpened opened},
			 * or if there are registered {@link #loadTasks} to be executed.
			 * No event handler may modify any properties inside the provided record.
			 *
			 * @param {Ext.Container} panel The panel to which the record was set
			 * @param {Zarafa.core.data.MAPIRecord} record The record which was updated
			 * @return {Boolean} false if the record should not be loaded.
			 */
			'beforeloadrecord',
			/**
			 * @event loadrecord
			 *
			 * Fired from {@link #field}
			 *
			 * Fires when a record has been {@link Zarafa.core.data.MAPIRecord#isOpened opened},
			 * and all registered {@link #loadTasks} have been executed.
			 * No event handler may modify any properties inside the provided record.
			 *
			 * @param {Ext.Container} panel The panel to which the record was set
			 * @param {Zarafa.core.data.MAPIRecord} record The record which was updated
			 */
			'loadrecord',
			/**
			 * @event updaterecord
			 *
			 * Fired from {@link #field}
			 *
			 * Fires when the record of this {@link #field} was updated,
			 * through a {@link Zarafa.core.data.MAPIStore store}.
			 * No event handler may modify any properties inside the provided record.
			 * @param {Ext.Container} panel The panel to which the record was set
			 * @param {String} action write Action that occurred. Can be one of
			 * {@link Ext.data.Record.EDIT EDIT}, {@link Ext.data.Record.REJECT REJECT} or
			 * {@link Ext.data.Record.COMMIT COMMIT}
			 * @param {Zarafa.core.data.MAPIRecord} record The record which was updated
			 */
			'updaterecord',
			/**
			 * @event userupdaterecord
			 *
			 * Fired from {@link #field}
			 *
			 * Fires when the record of this {@link #field} was changed by the user.
			 * This allows the UI to apply special indicators to show the user that
			 * the user might need to save his changes.
			 * @param {Ext.Container} panel The panel to which the record was set
			 * @param {Zarafa.core.data.MAPIRecord} record The record which is dirty
			 */
			'userupdaterecord',
			/**
			 * @event exceptionrecord
			 *
			 * Fired from {@link #field}
			 *
			 * Fires when an exception event was fired by the {@link Ext.data.DataProxy}.
			 * No event handler may modify any properties inside the provided record.
			 * @param {String} type See {@link Ext.data.DataProxy}.{@link Ext.data.DataProxy#exception exception}
			 * for description.
			 * @param {String} action See {@link Ext.data.DataProxy}.{@link Ext.data.DataProxy#exception exception}
			 * for description.
			 * @param {Object} options See {@link Ext.data.DataProxy}.{@link Ext.data.DataProxy#exception exception}
			 * for description.
			 * @param {Object} response See {@link Ext.data.DataProxy}.{@link Ext.data.DataProxy#exception exception}
			 * for description.
			 * @param {Zarafa.core.data.MAPIRecord} record The record which was subject of the request
			 * that encountered an exception.
			 * @param {String} error (Optional) Passed when a thrown JS Exception or JS Error is
			 * available.
			 */
			'exceptionrecord',
			/**
			 * @event writerecord
			 *
			 * Fired from {@link #field}
			 *
			 * Fires when write event was fired by the {@link Zarafa.core.data.MAPIStore store}.
			 * @param {Ext.data.Store} store
			 * @param {String} action [Ext.data.Api.actions.create|update|destroy]
			 * @param {Object} result The 'data' picked-out out of the response for convenience.
			 * @param {Ext.Direct.Transaction} res
			 * @param {Zarafa.core.data.IPMRecord[]} records Store's records, the subject(s) of the write-action
			 */
			'writerecord'
		);

		// Raise an event after all components have been layout, we can then call this.setRecord
		// to force all components to be aware of the Record which is bound to this component.
		// we can't do that in the constructor because then nobody is listening to the event
		// yet. Neither can we do it after rendering, since that only indicates that this container.
		// has been rendered and not the components.
		this.field.on('afterlayout', this.onAfterFirstLayout, this, {single: true});

		// Add event listener for the 'close' event, if we are editing the record, then the
		// record must be removed from the shadowStore when closing the dialog.
		this.field.on('close', this.onClose, this);
	},

	/**
	 * Start editing on a {@link Zarafa.core.data.MAPIRecord record} by hooking
	 * it into the {@link Zarafa.core.data.ShadowStore ShadowStore} and mark the
	 * {@link Zarafa.core.data.MAPIRecord record} as being edited. This function
	 * should only be called when {@link #useShadowStore} is set to true.
	 *
	 * @param {Zarafa.core.data.MAPIRecord} record The record to edit
	 * @private
	 */
	startShadowRecordEdit: function(record)
	{
		if(this.useShadowStore) {
			this.shadowStore.add(record);
		}
	},

	/**
	 * Stop editing on a {@link Zarafa.core.data.MAPIRecord record} by unhooking it
	 * from the {@link Zarafa.core.data.ShadowStore ShadowStore}. This function
	 * should only be called when {@link #useShadowStore} is set to true.
	 *
	 * @param {Zarafa.core.data.MAPIRecord} record The record to finish editing
	 * @private
	 */
	endShadowRecordEdit: function(record)
	{
		if(this.useShadowStore) {
			this.shadowStore.remove(record, true);
		}
	},

	/**
	 * Check if the given {@link Zarafa.core.data.MAPIRecord record} matches the {@link #record}
	 * which is used inside this container. When we have {@link #allowWrite} support, we only accept updates
	 * when the record is exactly the same as the record inside the container. But when we have
	 * read-only support, we accept updates from all records with the same entryid.
	 *
	 * @param {Zarafa.core.data.MAPIRecord} record The record to compare
	 * @return {Boolean} True if the given record matches the record inside this container
	 * @private
	 */
	isRecordInContainer: function(record)
	{
		if (this.allowWrite === true && this.useShadowStore === false) {
			return (this.record && this.record === record);
		} else {
			return (this.record && this.record.equals(record));
		}
	},

	/**
	 * Set the {@link Zarafa.core.data.MAPIRecord record} which must be shown inside the {@link #field}
	 * When the field has not yet {@link #isLayoutCalled laid out}, then the {@link #record} is
	 * set, but all work is deferred to the first {@link #doLayout layout} action on this container.
	 * Otherwise this function will call {@link #beforesetrecord} to check if the record should be
	 * set on this field. Depending on the {@link Zarafa.core.data.MAPIRecord#isOpened opened} status
	 * of the record, it will call to the server to open the record (and wait with displaying the
	 * record until the server has responded) or display the record directly.
	 * @param {Zarafa.core.data.MAPIRecord} record The record to set
	 * @param {Boolean} cheapCopy true to prevent the record from being copied. This is usually the case
	 * when {@link #allowWrite} is enabled, and the given record is not a phantom.
	 */
	setRecord: function(record, cheapCopy)
	{
		var oldrecord = this.record;

		if (this.record === record) {
			return;
		}

		// Cancel any scheduled tasks which
		// might have been registered for
		// the previous record.
		if (this.scheduledTasks) {
			Ext.each(this.scheduledTasks, function(task) {
				task.cancel();
			});
			this.scheduledTasks = [];
		}

		// Clear all pending tasks for the
		// previous registered record.
		if (this.pendingTasks) {
			this.pendingTasks = [];
		}

		// Layout has not yet been called, cache the record
		// until the onAfterFirstLayout has been called, which will
		// set the record again.
		if (this.isLayoutCalled === false) {
			this.record = record;
			this.cheapCopy = cheapCopy;
			return;
		}

		if (this.field.fireEvent('beforesetrecord', this, record, oldrecord) === false) {
			return;
		}

		// Unhook the old record from the modifications tracking
		if (oldrecord) {
			oldrecord.setUpdateModificationsTracking(false);
			if (this.ignoreUpdates !== true) {
				this.field.mun(oldrecord.getStore(), 'update', this.onUpdateRecord, this);
				this.field.mun(oldrecord.getStore(), 'write', this.onWrite, this);
				this.field.mun(oldrecord.getStore(), 'exception', this.onExceptionRecord, this);
			}
			if (this.useShadowStore === true) {
				this.endShadowRecordEdit(oldrecord);
			}
		}

		// If we are clearing the record, we will take a shortcut, simply fire the
		// 'setrecord' event.
		if (!record) {
			this.record = record;
			this.field.fireEvent('setrecord', this, record, oldrecord);
			return;
		}

		// Check if the record must be move into the ShadowStore. When the record
		// is a phantom we can safely move it, otherwise we need to create a copy.
		// If the cheapCopy argument was provided, then it means we have already
		// received a copy and we can move it without problems.
		if (this.useShadowStore) {
			if (record.phantom !== true && cheapCopy !== true) {
				record = record.copy();
			}
			this.startShadowRecordEdit(record);
		}

		// All preparations have been completed, it is now time to set
		// the Record correctly into the component.
		this.record = record;

		// remove modified from modal dialog record
		if(this.record.isModalDialogRecord) {
			delete this.record.modified;
		}

		this.record.setUpdateModificationsTracking(true);
		this.field.fireEvent('setrecord', this, record, oldrecord);

		// Register the exception event handler, as we want to catch exceptions for
		// opening the record as well.
		if (this.ignoreUpdates !== true) {
			this.field.mon(record.getStore(), 'exception', this.onExceptionRecord, this);
		}

		// If the record is not yet opened, we must open the record now,
		// Add the 'openTaskHandler' to the loadTasks so we can start loading it.
		var tasks = this.loadTasks ? this.loadTasks.clone() : [];
		if (!record.isOpened() && this.enableOpenLoadTask) {
			tasks.push({
				fn: this.openTaskHandler,
				scope: this,
				defer: this.autoOpenLoadTaskDefer
			});
		}

		// Start loading the record.
		this.doLoadRecord(record, tasks);
	},

	/**
	 * Task handler for {@link #loadTasks} in case the {@link #record} needs
	 * to be opened. This will open the record, and wait for the response from
	 * the server.
	 * @param {Ext.Component} component The component which contains the record
	 * @param {Zarafa.core.data.MAPIRecord} record The record to be opened
	 * @param {Object} task The task object
	 * @param {Function} callback The function to call when the task has been
	 * completed.
	 * @private
	 */
	openTaskHandler: function(component, record, task, callback)
	{
		var fn = function(store, record) {
			if (this.isRecordInContainer(record)) {
				this.field.mun(store, 'open', fn, this);

				if (this.record !== record) {
					this.record.applyData(record);
				}
				callback();
			}
		};
		this.field.mon(record.getStore(), 'open', fn, this);
		record.open();
	},

	/**
	 * Obtain the current {@link #record} which has been sent to all
	 * listeners of the {@link #setrecord} event.
	 *
	 * @return {Zarafa.core.data.MAPIRecord} The record which is currently active
	 */
	getActiveRecord: function()
	{
		// The setrecord event is called during setRecord, but always after
		// isLayoutCalled.
		if (this.isLayoutCalled === true) {
			return this.record;
		}
	},

	/**
	 * Event handler which is triggered after the {@link #field}
	 * is first time laid out. This will reset the current {@link #record}
	 * to trigger the {@link #setrecord} event for the initial Record.
	 *
	 * @param {Ext.Component} component This component
	 * @param {ContainerLayout} layout The layout
	 * @private
	 */
	onAfterFirstLayout: function(component, layout)
	{
		this.isLayoutCalled = true;

		if (this.record) {
			// Force record reload by clearing this.record first.
			var record = this.record;
			this.record = undefined;
			this.setRecord(record, this.cheapCopy);
		}
	},

	/**
	 * This will check if any {@link #loadTasks tasks} have been registered.
	 * If there are, this will fire the {@link #beforeloadrecord} event, and
	 * starts executing the tasks. When it is done, the {@link #loadrecord} event
	 * is fired.
	 * Finally in all cases, the function {@link #afterLoadRecord} is called
	 * to complete the loading of the record.
	 * @param {Zarafa.core.data.MAPIRecord} record The record to load
	 * @param {Array} Array of Task objects as copied from {@link this.loadTasks}.
	 * @private
	 */
	doLoadRecord: function(record, tasks)
	{
		this.pendingTasks = [];
		this.scheduledTasks = [];

		// Check if we have any loading tasks, if we do, then start executing
		// them and wait for all of them to complete before continuing
		// to afterLoadRecord.
		if (!Ext.isEmpty(tasks)) {
			if (this.field.fireEvent('beforeloadrecord', this.field, record) !== false) {
				Ext.each(tasks, function(task) {
					// Add it to the pendingTasks list, so we can
					// keep track of the pending completion events.
					this.pendingTasks.push(task);

					// Check if the task should be deferred, if so
					// then use a Ext.util.DelayedTask to schedule it.
					if (task.defer && task.defer > 0) {
						var delayTask = new Ext.util.DelayedTask(this.doTask, this, [ record, task ]);

						// Add it to the scheduledTasks, so we can
						// keep track of the tasks (and cancel them
						// if needed).
						this.scheduledTasks.push(delayTask);
						delayTask.delay(task.defer);
					} else {
						this.doTask(record, task);
					}
				}, this);
			}
		} else {
			this.afterLoadRecord(record);
		}

	},

	/**
	 * Execute a task as registered in the {@link #loadTasks registered tasks}.
	 * @param {Zarafa.core.data.MAPIRecord} record The record to perform the action on
	 * @param {Object} task The task object from {@link #loadTasks}
	 * @private
	 */
	doTask: function(record, task)
	{
		if (this.isRecordInContainer(record)) {
			task.fn.call(task.scope || task, this.field, record, task, this.doLoadRecordCallback.createDelegate(this, [ task ]));
		}
	},

	/**
	 * Callback function used when executing {@link #loadTasks tasks}. Each
	 * time this function is called, the task is removed from the {@link #loadTasks}
	 * list. If this was the last task, {@link #loadrecord} event is fired, and
	 * {@link #afterLoadRecord} will be called.
	 * @param {Object} task The task which was completed
	 * @private
	 */
	doLoadRecordCallback: function(task)
	{
		var record = this.record;

		// Unregister the task when it is completed.
		this.pendingTasks.remove(task);

		// Check if this was the last task...
		if (this.pendingTasks.length === 0) {
			// If all pending Tasks were done, all
			// scheduled tasks have been completed as well
			this.scheduledTasks = [];

			this.field.fireEvent('loadrecord', this.field, record);
			this.afterLoadRecord(record);
		}
	},

	/**
	 * Reset the {@link #isChangedByUser} property to false and fire the {@link #userupdaterecord} event
	 * to signal the value change. This will tell all components that the {@link #record} contains
	 * no modifications from the user.
	 * @private
	 */
	resetUserChangeTracker: function()
	{
		this.isChangedByUser = false;
		this.field.fireEvent('userupdaterecord', this.field, this.record, this.isChangedByUser);
	},

	/**
	 * Set the {@link #isChangedByUser} property to true and fire the {@link #userupdaterecord} event
	 * when this means the property was changed. This will tell all components that the {@link #record}
	 * contains modifications from the user.
	 * @private
	 */
	registerUserChange: function()
	{
		if (this.isChangedByUser !== true) {
			this.isChangedByUser = true;
			this.field.fireEvent('userupdaterecord', this.field, this.record, this.isChangedByUser);
		}
	},

	/**
	 * Check if the given record contains user changes. This will check the contents for
	 * {@link Zarafa.core.data.MAPIRecord#updateModifications updateModifications} and
	 * {@link Zarafa.core.data.MAPIRecord#updateSubStoreModifications updateSubStoreModifications}
	 * to see if they were modified.
	 * @param {Zarafa.core.data.MAPIRecord} record The record to validate
	 * @private
	 */
	checkForUserChanges: function(record)
	{
		var updateModifications = record.updateModifications;
		var updateSubStoreModifications = record.updateSubStoreModifications;

		if ((updateModifications && Object.keys(updateModifications).length > 0) ||
		  (updateSubStoreModifications && Object.keys(updateSubStoreModifications).length > 0)) {
			this.registerUserChange();
		}
	},

	/**
	 * Called by {@link #doLoadRecord} when all {@link #loadTasks} have been executed
	 * (or when no tasks were registered}.
	 *
	 * This will register additional event handlers on the {@link Zarafa.core.data.MAPIStore store}.
	 *
	 * @param {Zarafa.core.data.MAPIRecord} record The record which has been loaded
	 * @private
	 */
	afterLoadRecord: function(record)
	{
		if (this.ignoreUpdates !== true) {
			this.resetUserChangeTracker();

			var store = this.record.getStore();
			this.field.mon(store, {
				'update': this.onUpdateRecord,
				'write': this.onWrite,
				'scope': this
			});
		}
	},

	/**
	 * Event handler will be called when the {@link Zarafa.core.data.MAPIStore Store} has
	 * updated the {@link Zarafa.core.data.MAPIRecord record}. This function will fire
	 * {@link #updaterecord} event to notify that the record is updated.
	 *
	 * @param {Zarafa.core.data.IPMStore} store The store in which the record is located.
	 * @param {Zarafa.core.data.IPMRecord} record The record which has been updated
	 * @param {String} operation The update operation being performed. Value may be one of
	 * {@link Ext.data.Record.EDIT EDIT}, {@link Ext.data.Record.REJECT REJECT} or
	 * {@link Ext.data.Record.COMMIT COMMIT}
	 * @private
	 */
	onUpdateRecord: function(store, record, operation)
	{
		if (!this.isRecordInContainer(record)) {
			return;
		}

		// Check if this exact record was updated, or if this is an external notification.
		// If this was a notification, then we only accept committed changes, before applying
		// the update to our current active record.
		if (this.record !== record) {
			if (operation !== Ext.data.Record.COMMIT) {
				return;
			}

			var modelRecord = this.record.isModalDialogRecord ? this.field.modalRecord : this.record;
			if (modelRecord !== record) {
				this.record.applyData(record);
			}
		}

		if (this.ignoreUpdates !== true) {
			if (operation === Ext.data.Record.COMMIT) {
				this.resetUserChangeTracker();
			} else {
				this.checkForUserChanges(record);
			}
		}

		this.field.fireEvent('updaterecord', this.field, operation, this.record);
	},

	/**
	 * Event handler will be called when the {@link Zarafa.core.data.MAPIStore Store} has
	 * fired an exception event.
	 * @param {Ext.data.DataProxy} proxy The proxy which fired the event.
	 * No event handler may modify any properties inside the provided record.
	 * @param {String} type See {@link Ext.data.DataProxy}.{@link Ext.data.DataProxy#exception exception}
	 * for description.
	 * @param {String} action See {@link Ext.data.DataProxy}.{@link Ext.data.DataProxy#exception exception}
	 * for description.
	 * @param {Object} options See {@link Ext.data.DataProxy}.{@link Ext.data.DataProxy#exception exception}
	 * for description.
	 * @param {Object} response See {@link Ext.data.DataProxy}.{@link Ext.data.DataProxy#exception exception}
	 * for description.
	 * @param {Zarafa.core.data.MAPIRecord} record The record which was subject of the request
	 * that encountered an exception.
	 * @param {String} error (Optional) Passed when a thrown JS Exception or JS Error is
	 * available.
	 * @private
	 */
	onExceptionRecord: function(proxy, type, action, options, response, args)
	{
		/** Extract the record from the args, and pull it out of an array if that is the case.
		 * If the array length is bigger than 1, we can discard this exception as this
		 * RecordCompontentPlugin can only handle one at a time. In that case the record variable
		 * will remain an array and will not pass the next IF-statement when it is compared against
		 * the MAPIRecord class.
		 */
		var record = args.sendRecords;
		if(Array.isArray(record) && record.length == 1){
			record = record[0];
		}
		// If it is not a record or if the record is not the record in the Container we should skip it
		if(!(record instanceof Zarafa.core.data.MAPIRecord) || !this.isRecordInContainer(record)){
			return;
		}
		var error = args.error || undefined;

		this.field.fireEvent('exceptionrecord', type, action, options, response, record, error);
	},

	/**
	 * Event handler will be called when the {@link Zarafa.core.data.MAPIStore Store} has
	 * fired a write event.
	 * Store fires it if the server returns 200 after an Ext.data.Api.actions CRUD action.
	 *
	 * @param {Ext.data.Store} store
	 * @param {String} action [Ext.data.Api.actions.create|update|destroy]
	 * @param {Object} result The 'data' picked-out out of the response for convenience.
	 * @param {Ext.Direct.Transaction} res
	 * @param {Zarafa.core.data.IPMRecord[]} records Store's records, the subject(s) of the write-action
	 * @private
	 */
	onWrite: function(store, action, result, res, records)
	{
		// If records isn't array then make it.
		records = [].concat(records);

		for (var i = 0, len = records.length; i < len; i++) {
			if (this.record === records[i]) {
				if(action == Ext.data.Api.actions.destroy) {
					// the record has been destroyed and removed from store
					// so user made changes are not useful anymore
					this.resetUserChangeTracker();
				}
				this.field.fireEvent('writerecord', store, action, result, res, records[i]);
				return;
			}
		}
	},

	/**
	 * Event handler which is called when the container is being closed.
	 * This will remove the {@link Zarafa.core.data.MAPIRecord record} from the
	 * {@link Zarafa.core.data.ShadowStore ShadowStore} (when the record was being {@link #allowWrite edited}).
	 * @param {Ext.Component} container The parent container on which this plugin is installed.
	 * @private
	 */
	onClose: function(dialog)
	{
		if (this.record) {
			this.endShadowRecordEdit(this.record);
		}
	}
});

Ext.preg('zarafa.recordcomponentplugin', Zarafa.core.plugins.RecordComponentPlugin);
Ext.namespace('Zarafa.core.plugins');

/**
 * @class Zarafa.core.plugins.RecordComponentUpdaterPlugin
 * @extends Object
 * @ptype zarafa.recordcomponentupdaterplugin
 *
 * This plugin is used on {@link Ext.Container Container} elements which
 * are placed as child (or grandchild) to a {@link Ext.Container Container} which
 * contains the {@link Zarafa.core.plugins.RecordComponentPlugin RecordComponent} plugin.
 *
 * This plugin will hook into the {@link Zarafa.core.plugins.RecordComponentPlugin#setrecord setrecord}
 * {@link Zarafa.core.plugins.RecordComponentPlugin#updaterecord updaterecord} events and call
 * the {@link Ext.Container#update update} function with the {@link Ext.data.Record Record}
 * argument.
 */
Zarafa.core.plugins.RecordComponentUpdaterPlugin = Ext.extend(Object, {
	/**
	 * @cfg {Ext.Container} rootContainer (optional) The container on which
	 * the {@link Zarafa.core.plugins.RecordComponentPlugin RecordComponent} plugin has been
	 * installed to which the container on which this plugin has been installed must
	 * listen to.
	 */
	rootContainer: undefined,
	/**
	 * The contained on which this plugin has been installed.
	 * @property
	 * @type Ext.Container
	 */
	field: undefined,
	/**
	 * True when the {@link #field} is {@link #onReadyComponent ready}.
	 * @property
	 * @type Boolean
	 */
	isReady: false,
	/**
	 * The record which is currently displayed in {@link #field}.
	 * @property
	 * @type Zarafa.core.data.IPMRecord
	 */
	record: undefined,
	/**
	 * @constructor
	 * @param {Object} config Configuration object
	 */
	constructor: function(config)
	{
		Ext.apply(this, config);
	},

	/**
	 * Initializes the {@link Ext.Component Component} to which this plugin has been hooked.
	 * @param {Ext.Component} The parent field to which this component is connected
	 */
	init: function(field)
	{
		this.field = field;
		field.recordComponentUpdaterPlugin = this;

		// Check if we already know our rootContainer.
		// If not, we have to wait for the 'added' event
		// to walk up the hierarchy tree to find a parent
		// which is the root container.
		if (this.rootContainer) {
			this.hookToRootContainer(this.rootContainer);
		} else if (field.rootContainer) {
			this.hookToRootContainer(field.rootContainer);
		} else {
			this.field.on('beforerender', this.onFirstBeforeRender, this, { single: true });
		}

		// If a layout is present on the component, then we must wait until the component
		// has been laid out. Otherwise rendering will be sufficient.
		if (Ext.isDefined(this.field.layout)) {
			this.field.on('afterlayout', this.onReadyComponent, this, { single: true });
		} else if (this.field.rendered !== true) {
			this.field.on('afterrender', this.onReadyComponent, this, { single: true });
		} else {
			this.onReadyComponent(this.field);
		}
	},

	/**
	 * This will set the {@link #rootContainer} for this plugin. The rootContainer is a
	 * {@link Ext.Container container} which has the {@link Zarafa.core.plugins.RecordComponentPlugin RecordComponent}
	 * plugin installed.
	 *
	 * @param {Ext.Container} root The root container on which we must hook our events.
	 * @private
	 */
	hookToRootContainer: function(root)
	{
		this.rootContainer = root;

		// Obtain the record from the root container,
		// this prevents issues when the root container
		// has already been rendered and we have missed the
		// initial 'setrecord' event.
		this.record = root.recordComponentPlugin.getActiveRecord();

		this.field.mon(root, {
			'activate': this.onActivateRoot,
			'setrecord': this.onSetRecord,
			'loadrecord': this.onLoadRecord,
			'updaterecord': this.onUpdateRecord,
			'exceptionrecord': this.onExceptionRecord,
			'beforesaverecord': this.onBeforeSaveRecord,
			'beforesendrecord': this.onBeforeSendRecord,
			'beforeclose': this.onBeforeClose,
			'deactivate': this.onDeactivate,
			scope: this
		});
	},

	/**
	 * Event handler which is fired when the {@link #field} is being rendered
	 * When this happened, we start walking up the hierarchy tree to find the
	 * {@link #rootContainer} which contains the {@link Zarafa.core.plugins.RecordComponentPlugin RecordComponent}
	 * from which we must listen for updates.
	 *
	 * @private
	 */
	onFirstBeforeRender: function()
	{
		var root = this.field;

		while (!(root.recordComponentPlugin instanceof Zarafa.core.plugins.RecordComponentPlugin)) {
			root = root.ownerCt;
			if (!Ext.isDefined(root)) {
				break;
			}
		}

		if (root) {
			this.hookToRootContainer(root);
		}
	},

	/**
	 * Called after the {@link #field component} has is ready for initialization by the plugin,
	 * depending on the type this can be on {@link Ext.Container#afterlayout} or {@link Ext.Component#afterrender}.
	 *
	 * This will load the current {@link Zarafa.core.data.IPMRecord Record} using
	 * the {@link Ext.Container#update} function.
	 *
	 * @param {Ext.Component} component This component
	 * @private
	 */
	onReadyComponent: function(component)
	{
		if (this.record) {
			this.field.update(this.record, true);
		}
		this.isReady = true;
	},

	/**
	 * Event handler which is fired for the {@link #rootContainer}#{@link Ext.Panel#activate activate}
	 * event. This will call the {@link Ext.Container#update} function to update the
	 * {@link #field} to which this plugin is attached. This ensures that any changes
	 * in data or UI which have occurred while the tab was deactivated are being forced
	 * into the UI again.
	 * @param {Ext.Panel} panel The panel which was activated
	 * @private
	 */
	onActivateRoot: function(panel)
	{
		if (this.record && this.isReady === true && this.field.isDestroyed !== true) {
			this.field.update(this.record, true);
		}
	},

	/**
	 * Event handler for the {@link Zarafa.core.plugins.RecordComponentPlugin#setrecord setrecord} event.
	 * This will call the {@link Ext.Container#update update} function
	 * to update the {@link #field} to which this plugin is attached.
	 *
	 * @param {Ext.Container} panel The panel to which the record was set
	 * @param {Zarafa.core.data.IPMRecord} record The record which was set
	 * @param {Zarafa.core.data.IPMRecord} oldrecord The oldrecord which was previously set
	 * @private
	 */
	onSetRecord: function(panel, record, oldrecord)
	{
		this.record = record;
		if (this.isReady === true) {
			this.field.update(record, true);
		}
	},

	/**
	 * Event handler for the {@link Zarafa.core.plugins.RecordComponentPlugin#loadrecord loadrecord} event.
	 * This will call the {@link Ext.Container#update update} function
	 * to update the {@link #field} to which this plugin is attached
	 *
	 * @param {Ext.Container} panel The panel to which the record belongs
	 * @param {Zarafa.core.data.IPMRecord} record The record which was updated
	 * @private
	 */
	onLoadRecord: function(panel, record)
	{
		if (this.isReady === true && this.field.isDestroyed !== true) {
			this.field.update(record, true);
		}
	},

	/**
	 * Event handler for the {@link Zarafa.core.plugins.RecordComponentPlugin#updaterecord updaterecord} event.
	 * This will call the {@link Ext.Container#update update} function
	 * to update the {@link #field} to which this plugin is attached
	 *
	 * @param {Ext.Container} panel The panel to which the record belongs
	 * @param {String} action write Action that occurred. Can be one of
	 * {@link Ext.data.Record.EDIT EDIT}, {@link Ext.data.Record.REJECT REJECT} or
	 * {@link Ext.data.Record.COMMIT COMMIT}
	 * @param {Zarafa.core.data.IPMRecord} record The record which was updated
	 * @private
	 */
	onUpdateRecord: function(panel, action, record)
	{
		if (this.isReady === true && this.field.isDestroyed !== true) {
			this.field.update(record, false);
		}
	},

	/**
	 * Event handler for the {@link Zarafa.core.plugins.RecordComponentPlugin#exceptionrecord exceptionrecord} event.
	 * This will call the {@link Ext.Container#doException doException} function if it is available.
	 * to update the {@link #field} to which this plugin is attached
	 * @param {String} type See {@link Ext.data.DataProxy}.{@link Ext.data.DataProxy#exception exception}
	 * for description.
	 * @param {String} action See {@link Ext.data.DataProxy}.{@link Ext.data.DataProxy#exception exception}
	 * for description.
	 * @param {Object} options See {@link Ext.data.DataProxy}.{@link Ext.data.DataProxy#exception exception}
	 * for description.
	 * @param {Object} response See {@link Ext.data.DataProxy}.{@link Ext.data.DataProxy#exception exception}
	 * for description.
	 * @param {Zarafa.core.data.MAPIRecord} record The record which was subject of the request
	 * that encountered an exception.
	 * @param {String} error (Optional) Passed when a thrown JS Exception or JS Error is
	 * available.
	 * @private
	 */
	onExceptionRecord: function(type, action, options, response, record, error)
	{
		if (this.isReady === true && this.field.isDestroyed !== true) {
			if (Ext.isFunction(this.field.doException)){
				this.field.doException(type, action, options, response, record, error);
			}
		}
	},

	/**
	 * Event handler for the {@link Zarafa.core.plugins.RecordComponentPlugin#beforesave beforesave} event.
	 * This will call the 'updateRecord' function to update the {@link #record} based on the
	 * information from the {@link #field}.
	 *
	 * @param {Ext.Container} panel The panel to which the record belongs.
	 * @param {Zarafa.core.data.IPMRecord} record The record which is being saved.
	 * @return {Boolean} false if the operation failed, and the save action must be aborted.
	 * @private
	 */
	onBeforeSaveRecord: function(panel, record)
	{
		if (this.isReady === true && this.field.isDestroyed !== true) {
			if (Ext.isFunction(this.field.updateRecord)) {
				return this.field.updateRecord(record);
			}
		}
		return true;
	},

	/**
	 * Event handler for the {@link Zarafa.core.plugins.RecordComponentPlugin#beforesendrecord beforesendrecord} event.
	 * This will call the 'updateRecord' function to update the {@link #record} based on the
	 * information from the {@link #field}.
	 *
	 * @param {Ext.Container} panel The panel to which the record belongs.
	 * @param {Zarafa.core.data.IPMRecord} record The record which is being saved.
	 * @return {Boolean} false if the operation failed, and the save action must be aborted.
	 * @private
	 */
	onBeforeSendRecord: function(panel, record)
	{
		if (this.isReady === true && this.field.isDestroyed !== true) {
			if (Ext.isFunction(this.field.updateRecord)) {
				return this.field.updateRecord(record);
			}
		}
		return true;
	},

	/**
	 * Event handler for the {@link Ext.Container#beforeclose} event.
	 * This will call the 'updateRecord' function to update the {@link #record} based on the
	 * information from the {@link #field}.
	 *
	 * @param {Ext.Container} panel The panel to which the record belongs.
	 * @private
	 */
	onBeforeClose: function(panel)
	{
		if (this.isReady === true && this.field.isDestroyed !== true) {
			if (this.record && Ext.isFunction(this.field.updateRecord)) {
				return this.field.updateRecord(this.record);
			}
		}
		return true;
	},

	/**
	 * Event handler for the {@link Ext.Panel#deactivate} event.
	 * This will call the 'updateRecord' function to update the {@link #record} based on the
	 * information from the {@link #field}.
	 *
	 * @param {Ext.Container} panel The panel to which the record belongs.
	 * @private
	 */
	onDeactivate: function(panel)
	{
		if (this.isReady === true && this.field.isDestroyed !== true) {
			if (this.record && Ext.isFunction(this.field.updateRecord)) {
				return this.field.updateRecord(this.record);
			}
		}
		return true;
	}
});

Ext.preg('zarafa.recordcomponentupdaterplugin', Zarafa.core.plugins.RecordComponentUpdaterPlugin);
Ext.namespace('Zarafa.core.plugins');

/**
 * @class Zarafa.core.plugins.TabCloseMenuPlugin
 * @extends Ext.ux.TabCloseMenu
 * @ptype zarafa.tabclosemenuplugin
 *
 * Extended so that the context menu would call the tab's close() method instead of directly destroying it
 * This class is intended to be used with {@link Zarafa.core.ui.MainContentTabPanel}
 * TODO: handle closing multiple tabs better
 */
Zarafa.core.plugins.TabCloseMenuPlugin = Ext.extend(Ext.ux.TabCloseMenu, {
	/**
	 * Overridden to add menu items from an insertion point
	 * @override
	 * @private
	 */
	createMenu: function(){
		if(!this.menu){
			var items = [{
				itemId: 'close',
				text: this.closeTabText,
				scope: this,
				handler: this.onClose
			}];
			if(this.showCloseAll){
				items.push('-');
			}
			items.push({
				itemId: 'closeothers',
				text: this.closeOtherTabsText,
				scope: this,
				handler: this.onCloseOthers
			});
			if(this.showCloseAll){
				items.push({
					itemId: 'closeall',
					text: this.closeAllTabsText,
					scope: this,
					handler: this.onCloseAll
				});
			}
			var lazyItems = container.populateInsertionPoint('main.content.tabpanel.tabclosemenu');
			this.menu = new Ext.menu.Menu({
				items: items.concat(lazyItems)
			});
		}
		return this.menu;
	},

	/**
	 * Overridden, because the original function calls remove() directly
	 * item.close() is better because it fires an event that notifies other components that the tab wants to close
	 * @override
	 * @private
	 */
	onClose: function(){
		this.active.close();
	},

	/**
	 * Overridden, because the original function calls remove() directly
	 * item.close() is better because it fires an event that notifies other components that the tab wants to close
	 * @param {Boolean} excludeActive Flag for whether to close all tabs but preserve the currently active one
	 * @override
	 * @private
	 */
	doClose: function(excludeActive){
		this.tabs.items.each(function(item){
			if(item.closable){
				if(!excludeActive || item != this.active){
					item.close();
				}
			}
		}, this);
	}
});

Ext.preg('zarafa.tabclosemenuplugin', Zarafa.core.plugins.TabCloseMenuPlugin);
Ext.namespace('Zarafa.core.ui');

/**
 * @class Zarafa.core.ui.ButtonGroup
 * @extends Ext.ButtonGroup
 * @xtype zarafa.buttongroup
 */
Zarafa.core.ui.ButtonGroup = Ext.extend(Ext.ButtonGroup, {
	/**
	 * @constructor
	 * @param config
	 */
	constructor: function(config)
	{
		Ext.apply(this, config, {
			xtype: 'zarafa.buttongroup'
		});

		Zarafa.core.ui.ButtonGroup.superclass.constructor.call(this, config);
	},

	/**
	 * Function will check whether buttongroup has any visible buttons or not.
	 *
	 * @return {Boolean} function will return true if buttonGroups has any visible items,
	 * It will return false if buttonGroup is empty or buttonGroup has all hidden buttons.
	 */
	hasVisibleButtons: function()
	{
		var buttonsArray = this.items.getRange();
		if(!Ext.isEmpty(buttonsArray)){
			for (var i = 0; i < buttonsArray.length; i++)
			{
				//isVisible function is not working properly in current version of ExtJs.
				if(buttonsArray[i].hidden === false) {
					return true;
				}
			}
		}
		return false;
	}
});

Ext.reg('zarafa.buttongroup', Zarafa.core.ui.ButtonGroup);
Ext.namespace('Zarafa.core.ui');

/**
 * @class Zarafa.core.ui.ContentPanel
 * @extends Ext.Container
 * @xtype zarafa.contentpanel
 *
 * FIXME: this documentation needs to be completely rewritten
 * The {@link Zarafa.core.ui.ContentPanel ContentPanel} class a base class for content panels and is therefore not intended to be instantiated
 * directly. Developers must extend this class to implement a dialog window that can be shown as either an ExtJS
 * window or a separate browser window.
 * <p>
 * All dialog classes must be registered using the static register method:
 * <p>
 * <code><pre>
 * Zarafa.core.ui.ContentPanel.register(MyDialog, 'mydialog');
 * </pre></code>.
 * <p>
 * The dialog can then be instantiated using the create() method:
 * <p>
 * <code><pre>
 * MyDialog.create(config, {
 *  browser: true,
 *  width: 500,
 *  height: 300,
 * });
 * </pre></code>.
 *
 */
Zarafa.core.ui.ContentPanel = Ext.extend(Ext.Container, {
	/**
	 * @cfg {String} title The title for the ContentPanel
	 */
	title: undefined,

	/**
	 * @cfg {String} The action to take when the close header tool is clicked.
	 * Only used when using {@link Zarafa.core.data.UIFactoryWindowLayer UIFactoryWindowLayer} to
	 * display this {@link Zarafa.core.ui.ContentPanel ContentPanel}.
	 */
	closeAction: 'closeWrap',

	/**
	 * @cfg {Boolean} standalone If true, the {@link Zarafa.core.ui.ContentPanel contentpanel}
	 * will not be hooked into the {@link Zarafa.core.data.ContentPanelMgr}. This will prevent
	 * listening to events coming from the {@link Zarafa.core.data.ContentPanelMgr ContentPanelMgr}.
	 * Defaults to false.
	 */
	standalone: false,

	/**
	 * @cfg {Number} width The width for the ContentPanel
	 */
	width: 800,

	/**
	 * @cfg {Number} height The height for the ContentPanel
	 */
	height: 550,

	/**
	 * @cfg {String} iconCls Icon class for the tab
	 */
	iconCls: undefined,

	/**
	 * @cfg {Boolean} useInputAutoFocusPlugin True to use the inputautofocusplugin for
	 * this panel
	 */
	useInputAutoFocusPlugin: true,

	/**
	 * @cfg {Boolean} forceFullyOpenInMainWindow True to render panel on main browser window
	 */
	forceFullyOpenInMainWindow: false,

	/**
	 * @constructor
	 * @param {Object} config configuration object.
	 */
	constructor: function(config)
	{
		config = config || {};

		if ( config.useInputAutoFocusPlugin !== false ) {
			config.plugins = Ext.value(config.plugins, []);
			config.plugins.push({
				ptype: 'zarafa.inputautofocusplugin'
			});
		}

		Ext.applyIf(config, {
			xtype: 'zarafa.contentpanel',
			stateful: true
		});

		Zarafa.core.ui.ContentPanel.superclass.constructor.call(this, config);

		/**
		 * @event beforeclose
		 * Fires before the {@link Zarafa.core.ui.ContentPanel ContentPanel} is closed.
		 * A handler can return false to cancel the close.
		 * @param {Ext.Panel} panel The Panel being closed.
		 */
		/**
		 * @event close
		 * Fires after the Panel is closed.
		 * @param {Ext.Panel} panel The Panel that has been closed.
		 */
		this.initializeChildComponent(this);

		if (Ext.isDefined(this.title)) {
			this.setTitle(this.title);
		}

		// register with the ContentPanelMgr unless specifically instructed not to
		if (!this.standalone) {
			Zarafa.core.data.ContentPanelMgr.register(this);

		}
	},

	/**
	 * Called when the ContentPanel has been rendered.
	 * This activate the keymap on the element of this component after the normal operations of
	 * afterRender have been completed. It will activate by getting the xtype hierarchy from
	 * {@link #getXTypes} and format it into a string usable by the
	 * {@link Zarafa.core.KeyMapMgr KeyMapMgr}.
	 * @private
	 */
	afterRender: function()
	{
		Zarafa.core.ui.ContentPanel.superclass.afterRender.apply(this, arguments);
		var xtypes = this.getXTypes();

		// The first part leading up to zarafa.contentpanel will be stripped
		xtypes = xtypes.replace('component/box/container/zarafa.contentpanel','');

		// Then the "zarafa." will be stripped off from all the xtypes like "zarafa.somecontentpanel".
		xtypes = xtypes.replace(/\/zarafa\./g,'.');

		// Finally we strip string the "contentpanel" from all the xtypes. Otherwise each level will have
		// that "contentpanel" mentioned in them. Also we add contentpanel to the start as that sets
		// it apart from other components in the key mapping.
		xtypes = 'contentpanel' + xtypes.replace(/contentpanel/g, '');

		// This will activate keymaps with 'contentpanel.some.somemore' so the component
		// will have all key events register with 'contentpanel', 'some', 'somemore'
		Zarafa.core.KeyMapMgr.activate(this, xtypes);
	},

	/**
	 * This will initialize the {@link Ext.Container container} which is
	 * placed within the {@link Zarafa.core.ui.ContentPanel ContentPanel}. The
	 * {@link Ext.Container container} will receive a reference to the
	 * {@link Zarafa.core.ui.ContentPanel ContentPanel} into which it has been embedded.
	 * @param {Ext.Component} component The component which must be initialized
	 * @private
	 */
	initializeChildComponent: function(component)
	{
		// Empty objects (undefined, null, []) cannot be initialized.
		if (Ext.isEmpty(component)) {
			return;
		}

		// If it is an array, just recursively call
		// this function again for each individual item.
		var recursive = function(c) {
			this.initializeChildComponent(c);
		};

		if (Array.isArray(component)) {
			Ext.each(component, recursive, this);
			return;
		} else if (component instanceof Ext.util.MixedCollection) {
			component.each(recursive, this);
			return;
		}

		// We only initialize containers (and their subclasses).
		if (!component.isXType('container')) {
			return;
		}

		component.dialog = this;

		// Toolbars include the top- and bottomtoolbars
		if (!Ext.isEmpty(component.toolbars)) {
			this.initializeChildComponent(component.toolbars);
		}

		// Initialize all child items
		if (!Ext.isEmpty(component.items)) {
			this.initializeChildComponent(component.items);
		}
	},

	/**
	 * Closes the panel. Destroys all child components, rendering the panel unusable.
	 */
	close: function()
	{
		if (this.fireEvent('beforeclose', this) !== false) {
			this.doClose();
		}
	},

	/**
	 * Close handler for the {@link #close} function.
	 * @private
	 */
	doClose: function()
	{
		this.fireEvent('close', this);
		Zarafa.core.data.ContentPanelMgr.unregister(this);
	},

	/**
	 * Function will be called when {@link Ext.Window#closeAction} config will be used to
	 * close the window. This is hardcoded to close the dialog instead of hiding it to make sure that
	 * we fire proper events to notify dialog is closed.
	 * @protected
	 */
	closeWrap: function()
	{
		this.close();
	},

	/**
	 * Sets the title of the panel.
	 * @param {String} title the new window title.
	 */
	setTitle: function(title)
	{
		this.title = title;
		this.fireEvent('titlechange', this, title);
	},

	/**
	 * @param {String} iconCls Icon class of the panel
	 */
	setIcon: function(iconCls)
	{
		var oldIcon = this.iconCls;
		this.iconCls = iconCls;
		this.fireEvent('iconchange', this, iconCls, oldIcon);
	},

	/**
	 * @return {Boolean} true iff the content panel is a modal dialog
	 */
	isModal: function()
	{
		return this.modal;
	},

	/**
	 * Obtain the path in which the {@link #getState state} must be saved.
	 * This option is only used when the {@link Zarafa.core.data.SettingsStateProvider SettingsStateProvider} is
	 * used in the {@link Ext.state.Manager}. This returns {@link #statefulName} if provided, or else generates
	 * a custom name.
	 * @return {String} The unique name for this component by which the {@link #getState state} must be saved.
	 */
	getStateName: function()
	{
		return 'dialogs/' + Zarafa.core.ui.ContentPanel.superclass.getStateName.call(this);
	},

	/**
	 * When {@link #stateful} the State object which should be saved into the
	 * {@link Ext.state.Manager}.
	 * @return {Object} The state object
	 * @protected
	 */
	getState: function()
	{
		var state = Zarafa.core.ui.ContentPanel.superclass.getState.call(this) || {};
		return Ext.apply(state, this.getSize());
	}
});

Ext.reg('zarafa.contentpanel', Zarafa.core.ui.ContentPanel);
Ext.namespace('Zarafa.core.ui');
/**
 * @class Zarafa.core.ui.ContextNavigationPanel
 * @extends Ext.Panel
 * @xtype zarafa.contextnavigation
 *
 * ContextNavigationPanel provides custom navigation options to context through {@link Zarafa.hierarchy.ui.HierarchyTreePanel}.
 */
Zarafa.core.ui.ContextNavigationPanel = Ext.extend(Ext.Panel, {
	/**
	 * For this Context this panel will be visible in the {@link Zarafa.core.ui.NavigationPanel NavigationPanel}.
	 * @cfg {Zarafa.core.Context} Related Context
	 */
	context: null,

	/**
	 * @cfg {Boolean} true to avoid to {@link Zarafa.core.ui.NavigationPanel#getAllFoldersPanel show all folders} Panel.
	 * even if {@link Zarafa.core.ui.NavigationPanel#showFolderList showFolderList} config is true
	 */
	restrictToShowAllFolderList: false,

	/**
	 * @constructor
	 * @param {Object} config configuration object
	 */
	constructor: function (config) {
		config = config || {};

		// Config options for component itself.
		Ext.applyIf(config, {
			border: false,
			layout: 'fit',
			defaults: {
				border: false,
				autoScroll: false,
				defaults: { cls: 'zarafa-context-navigation-item-body' }
			}
		});

		Zarafa.core.ui.ContextNavigationPanel.superclass.constructor.call(this, config);
	},

	/**
	 * @return {Zarafa.core.Context}
	 */
	getContext: function() {
		return this.context || false;
	}
});

Ext.reg('zarafa.contextnavigation', Zarafa.core.ui.ContextNavigationPanel);
Ext.namespace('Zarafa.core.ui');

/**
 * @class Zarafa.core.ui.MainContentTabPanel
 * @extends Ext.TabPanel
 * @xtype zarafa.maincontenttabpanel
 * This subclass is used in the main content area, and contains the ContextContainer and dialogs that the user opens
 * Initialized in MainViewport.CreateContentContainer
 */
Zarafa.core.ui.MainContentTabPanel = Ext.extend(Ext.TabPanel, {

	/**
	 * The list of opened tab. It contains the list with the entryid and tab id.
	 * @property
	 * @type Ext.util.MixedCollection
	 * @private
	 */
	openedTabs: new Ext.util.MixedCollection(),

	/**
	 * Overridden to modify the tab depending on whether the record has been edited or not
	 * This method is called when a contained component fires the 'titlechange' event
	 * @param {Component} item
	 * @param {String} title
	 * @override
	 * @private
	 */
	onItemTitleChanged: function(item, title, oldTitle)
	{
		var el = this.getTabEl(item);
		if (el) {
			// now elide title if it exceeds 20 character length
			var tab = Ext.get(el).child('.x-tab-strip-text', true);

			// Change tooltip, and give full title in tab tool tip
			tab.qtip = title;

			tab.innerHTML = Ext.util.Format.htmlEncodeElide(title, 20, 0);
		}
	},

	/**
	 * This method is called when a contained component fires the 'iconchange' event
	 * @param {Component} item
	 * @param {String} iconCls icon class to apply
	 * @param {String} oldCls The previous icon class
	 * @private
	 */
	onItemIconChanged: function(item, iconCls, oldCls)
	{
		var tabEl = this.getTabEl(item);
		if (!Ext.isEmpty(tabEl)) {
			// removeClass only works when the CSS classes have been split
			// into an array, as it will not do it manually. For addClass,
			// we do not have this restriction.
			if (oldCls) {
				oldCls = oldCls.split(' ');
			}

			var tabText = Ext.get(tabEl).child('.x-tab-strip-text');
			if (iconCls) {
				tabText.addClass('x-tab-strip-icon');
				tabText.replaceClass(oldCls, iconCls);
			} else {
				tabText.removeClass('x-tab-strip-icon');
				tabText.removeClass(oldCls);
			}
		}
	},

	/**
	 * This method is called when contained component fires the 'userupdaterecord' event
	 * @param {Component} item The item which fired the event
	 * @param {Ext.data.Record} record Which was updated by the user
	 * @param {Boolean} changed True if the item has been changed by the user
	 * @private
	 */
	onItemUserUpdateRecord: function(item, record, changed)
	{
		var el = this.getTabEl(item);
		if (el) {
			var tab = Ext.get(el).child('.x-tab-strip-text');
			if (record.phantom || changed) {
				tab.addClass('zarafa-tab-edited');
			} else {
				tab.removeClass('zarafa-tab-edited');
			}
		}

		// If record get saved then add record id
		if (!record.phantom && !this.getOpenedTab(record)) {
			this.registeredOpenTab(record, item.getId());
		}
	},

	/**
	 * Overridden in order to listen to close event of child component
	 * @param {Component} item
	 * @param {Number} index
	 * @override
	 * @private
	 */
	initTab: function(item, index)
	{
		var title = item.title;
		if(!Ext.isEmpty(title)) {
			// provide a tooltip for tab titles
			item.tabTip = title;
			// now we can shorten the length of title if its exceeding 20 characters
			item.title = Ext.util.Format.htmlEncodeElide(title, 20, 0);
		}

		Zarafa.core.ui.MainContentTabPanel.superclass.initTab.call(this, item, index);

		item.on({
			scope: this,
			render: this.applyTooltip,
			iconchange: this.onItemIconChanged,
			userupdaterecord: this.onItemUserUpdateRecord,
			close: this.onTabClose
		});
	},

	/**
	 * This will apply tooltip on close button ('X') of {@link Ext.Panel Panel}.
	 * @param {Component} item.
	 */
	applyTooltip: function(item)
	{
		var el = item.tabEl;
		var closeTab = Ext.get(el).child('.x-tab-strip-close', true);
		if(closeTab) {
			closeTab.qtip = _('Close') + ' (Ctrl + Alt + W)';
		}
	},

	/**
	 * Handler for closing a tab when a child component has fired its close event
	 * For instance when a mail is sent, the MailCreateContentPanel needs to be closed
	 * @param {Component} tab
	 */
	onTabClose: function(tab)
	{
		if (this.fireEvent('beforeclose', tab)!==false) {
			this.remove(tab);
			this.openedTabs = this.openedTabs.filterBy(function (item) {
				return item !== tab.getId();
			});
			this.fireEvent('close', tab);
		}
	},

	/**
	 * handler for the '+' button in the tab strip
	 * adds a new item of type depending on current context
	 * @param {Ext.EventObjectImpl} e Event object
	 * @param {Element} t Event target
	 * @param {Object} o Configuration object
	 */
	onTabAddClick: function(e, t, o)
	{
		var model = container.getCurrentContext().getModel();

		if(model){

			if(model.createRecord === Ext.emptyFn) {
				//if unable to create record from the current context model, try to get the model based on the scope of the mainToolbar button
				var button = container.getMainPanel().mainToolbar.newButton;
				model = button.scope.model;
			}

			if (model.checkCreateRights === Ext.emptyFn) {
				var record = model.createRecord();
				if (!record) {
					//if unable to create record from the current context model, invoke the handler of the first item in the 'new' menu
					var button = container.getMainPanel().mainToolbar.newButton;
					button.handler.call(button.scope);
					return;
				} else {
					// This will always use tab layer only, no matter what layer is configured in settings
					Zarafa.core.data.UIFactory.openCreateRecord(record, {layerType: 'tab'});
				}
			} else {
				model.createRecord(function(record){
					if (!record) {
						//if unable to create record from the current context model, invoke the handler of the first item in the 'new' menu
						var button = container.getMainPanel().mainToolbar.newButton;
						button.handler.call(button.scope);
						return;
					} else {
						// This will always use tab layer only, no matter what layer is configured in settings
						Zarafa.core.data.UIFactory.openCreateRecord(record, {layerType: 'tab'});
					}
				}.createDelegate(this));
			}
		}
	},

	/**
	 * Overridden in order to add the '+' button to the edge of the tabstrip
	 * @param {Ext.Element} ct Container in which the panel is created
	 * @param {Element} position Element at which the panel is created (relative to its position)
	 * @override
	 */
	onRender: function(ct, position)
	{
		Zarafa.core.ui.MainContentTabPanel.superclass.onRender.call(this, ct, position);

		// insert add button into the edge element
		var edge = this.edge.update('<span id="zarafa-mainpanel-addtabbutton" class=\'x-tab-add\'></span>');
		this.mon(edge, 'click', this.onTabAddClick, this);

		// set tooltip on add button
		var addBtn = edge.child('.x-tab-add', true);
		addBtn.qtip = _('New item') + ' (Ctrl + Alt + N)';
	},

	/**
	 * Overridden in order to call close() on the item, instead of removing it immediately
	 * This allows the contained panel to fire a confirmation dialog
	 * @param {Ext.EventObjectImpl} e Event
	 * @private
	 * @override
	 */
	onStripMouseDown: function(e)
	{
		if (e.button !== 0) {
			return;
		}

		var target = this.findTargets(e);
		if (target.close) {
			target.item.close();
			return;
		}
		if (target.item && target.item != this.activeTab) {
			this.setActiveTab(target.item);
		}
	},

	/**
	 * Overridden so that '+' sign for adding tabs remains visible when there are scroll buttons
	 * @private
	 * @override
	 */
	getScrollWidth: function()
	{
		return this.edge.getOffsetsTo(this.stripWrap)[0] + this.getScrollPos() + this.edge.getWidth();
	},

	/**
	 * Fires before Ext.Component is added or inserted into the Zarafa.core.ui.MainContentTabPanel
	 * It will verify if the record is already opened in tab. If yes then moved focus on that tab,
	 * Otherwise add tab id into openedTabs.
	 * @param {Component} tab
	 */
	onBeforeAdd: function (tab)
	{
		if (tab.name === "main.content" || !tab.dialog) {
			return;
		}

		var record = tab.dialog.record;
		if (!Ext.isEmpty(record) && !Ext.isEmpty(record.get("entryid"))) {
			var openedTab = this.getOpenedTab(record);
			if (openedTab) {
				container.getTabPanel().setActiveTab(openedTab);
				return false;
			} else {
				this.registeredOpenTab(record,tab.getId());
			}
		}
	},

	/**
	 * Register a newly created tab with the {@link #openedTabs}.
	 * If record is {@link Zarafa.core.data.AppointmentRecord AppointmentRecord} recurring occurrence appointment then
	 * append basedate with entryid.
	 *
	 * @param {Ext.data.Record} record
	 * @param {String} tabId id of tab component.
	 */
	registeredOpenTab: function (record, tabId)
	{
		var entryid = record.get("entryid");

		// If record is attachment record then take id instead of entryid because
		// Attachment record id contains entryid with attach_num
		if (Array.isArray(record.get('attach_num'))) {
			entryid = record.id;
		} else if (!Ext.isEmpty(record.get("basedate"))) {
			entryid += "_" + record.get('basedate').getTime();
		}
		this.openedTabs.add(entryid, tabId);
	},

	/**
	 * Function which is use to find tab id of given record.
	 * @param {Zarafa.core.data.MAPIRecord} record
	 * @returns {String | Boolean} Tab id if record is already opened, false otherwise.
	 */
	getOpenedTab: function (record)
	{
		return this.openedTabs.find(function (key, item) {
			if (!Ext.isEmpty(record.get("basedate"))) {
				if (item.indexOf(record.get("basedate").getTime()) > -1) {
					item = item.split("_")[0];
				} else {
					item = false;
				}
			}
			var entryid = record.get("entryid");

			// If record is attachment record then take id instead of entryid because
			// Attachment record id contains entryid with attach_num
			if (Array.isArray(record.get('attach_num'))) {
				entryid = record.id;
			}

			if (Zarafa.core.EntryId.compareEntryIds(item, entryid)) {
				return true;
			}
		});
	}
});

Ext.reg('zarafa.maincontenttabpanel', Zarafa.core.ui.MainContentTabPanel);
Ext.namespace('Zarafa.core.ui');

/**
 * @class Zarafa.core.ui.MainTab
 * @extends Ext.Toolbar.Item
 * @xtype zarafa.maintab
 *
 * Is used to render the tabs in the {@link Zarafa.core.ui.MainTabBar MainTabBar}. It will relate to
 * a context and based on what is the active context according to the
 * {@link Zarafa.core.Container Container} it will mark itself as active.
 */
Zarafa.core.ui.MainTab = Ext.extend( Ext.Button, {
	/**
	 * @cfg {String} context Holds the name of the {@link Zarafa.core.Context Context} that this tab is related to.
	 */
	context: null,

	/**
	 * @constructor
	 * @param config Configuration structure
	 */
	constructor: function(config)
	{
		config = config || {};

		Ext.applyIf(config, {
			// Override from Ext.Component
			xtype: 'zarafa.maintab',
			cls: 'zarafa-maintab',
			buttonActiveCls: 'zarafa-maintabbar-maintab-active',
			handler: this.selectContext
		});

		Zarafa.core.ui.MainTab.superclass.constructor.call(this, config);

		this.mon(container, 'contextswitch', this.onContextSwitch, this);
		this.on('render', this.onRenderButton, this);
	},

	/**
	 * When switching to another context the state of the tab is re-evaluated.
	 * @private
	 */
	onContextSwitch: function(parameters, oldContext, newContext)
	{
		this.setContextActivityState(newContext);
	},

	/**
	 * When the tab is rendered the correct state is set.
	 * @private
	 */
	onRenderButton: function()
	{
		this.setContextActivityState(container.getCurrentContext());
	},

	/**
	 * Set the state of the tab based on the what the currently active context is. It will add the
	 * {@link #buttonActiveCls} as CSS class for the tab that is related to the active context. It
	 * will be removed if the related context is not active.
	 * @param {Zarafa.core.Context} currentContext The current context
	 * @private
	 */
	setContextActivityState: function(currentContext)
	{
		if(this.context == currentContext.getName()){
			this.addClass(this.buttonActiveCls);
		} else {
			this.removeClass(this.buttonActiveCls);
		}
	},


	/**
	 * Selects the context that this tab is related to set by the {@link #context} property. It will
	 * grab the {@link Zarafa.core.ContextModel ContextModel} and retrieve the default folder if the
	 * ContextModel exists. The default folder is opened for that context. The navigation panel is
	 * also just showing the folders related to that context.
	 * @private
	 */
	selectContext: function()
	{
		var context = container.getContextByName(this.context);

		if(context) {
			if(context === container.getCurrentContext()) {
				// if we are loading same context then don't do anything
				return;
			}

			var contextModel = context.getModel();
			var contextFolder;
			if (contextModel) {
				// Try to determine the folders, if previously
				// no folders were selected, we should select
				// the default folder.
				contextFolder = contextModel.getFolders();
				if (Ext.isEmpty(contextFolder)) {
					contextFolder = contextModel.getDefaultFolder();
				}
			}

			container.switchContext(context, contextFolder);
		}
	}
});

Ext.reg('zarafa.maintab', Zarafa.core.ui.MainTab);
Ext.namespace('Zarafa.core.ui');

/**
 * @class Zarafa.core.ui.MainTabBar
 * @extends Ext.Toolbar
 * @xtype zarafa.maintabbar
 *
 * The MainTabBar shows the tabs at the top of the application. It can be filled by two insertion
 * points that populate the left and the right side of the bar. It will use instances of the
 * {@link Zarafa.core.ui.MainTab MainTab} to represent the tabs.
 */
Zarafa.core.ui.MainTabBar = Ext.extend(Ext.Toolbar, {
	// Insertion points for this class
	/**
	 * @insert main.maintabbar.left
	 * Insertion point for populating main tabbar to the left. The tabOrderIndex is used to
	 * determine the order of the tabs. The lower the number the more to the left the button is
	 * added.
	 * @param {Zarafa.core.ui.MainTabBar} toolbar This toolbar
	 */
	/**
	 * @insert main.maintabbar.right
	 * Insertion point for populating main tabbar to the right. The tabOrderIndex is used to
	 * determine the order of the tabs. The lower the number the more to the right the button is
	 * added.
	 * @param {Zarafa.core.ui.MainTabBar} toolbar This toolbar
	 */

	/**
	 * @constructor
	 * @param config Configuration structure
	 */
	constructor: function(config)
	{
		config = config || {};

		Ext.applyIf(config, {
			// Override from Ext.Component
			xtype: 'zarafa.maintabbar',
			id: 'zarafa-mainmenu',
			cls: 'zarafa-maintabbar',
			defaultType: 'zarafa.maintab'
		});

		Zarafa.core.ui.MainTabBar.superclass.constructor.call(this, config);

		this.initBar();
	},

	/**
	 * Add items to this toolbar by using the main.maintabbar.left and main.maintabbar.right
	 * insertion points. Also the text who is logged in and the log out button is added. The buttons
	 * added through the insertion points are sorted using the tabOrderIndex set on the objects in
	 * the list returned by the insertion point.
	 * @private
	 */
	initBar: function()
	{

		var leftItems = container.populateInsertionPoint('main.maintabbar.left', this) || [];
		var rightItems = container.populateInsertionPoint('main.maintabbar.right', this) || [];

		// Make sure the items are properly sorted by priority.
		leftItems = Zarafa.core.Util.sortArray(leftItems, 'ASC', 'tabOrderIndex');
		// The right items are sorted so that the first item appears to the most right
		rightItems = Zarafa.core.Util.sortArray(rightItems, 'DESC', 'tabOrderIndex');

		this.addTooltip(leftItems, rightItems);

		var loginText = {
				xtype: 'tbtext',
				width: 'auto',
				cls: 'zarafa-maintabbar-logintext',
				text: container.getUser().getDisplayName(),
				id: 'mainmenu-logintext'
		};

		// Adding reminder button with bell icon.
		var reminder = {
			width: 30,
			id: 'mainmenu-button-reminder',
			ref: 'reminder',
			handler: function() {
				var store = container.getReminderStore();
				Zarafa.common.Actions.openReminderContent(store.getRange());
			},
			listeners: {
				afterRender: function(reminderBtn) {
					var store = container.getReminderStore();
					var recordLength = store.getRange().length;
					reminderBtn.getEl().setStyle('backgroundImage', 'url(\'' + Zarafa.common.ui.IconClass.getReminderSvgIcon( recordLength ) + '\')');
					var noReminder = recordLength === 0;
					reminderBtn.setDisabled(noReminder);
					reminderBtn.setTooltip(noReminder? _('There are no reminders'): '');
				},
				scope: this
			},
			style: {
				backgroundImage: 'url(\'' + Zarafa.common.ui.IconClass.getReminderSvgIcon() + '\')',
				backgroundRepeat: 'no-repeat',
				backgroundPosition: 'center'
			},
			scope: this
		};

		this.add(leftItems, {xtype: 'tbfill'}, loginText, reminder, rightItems);

		// Don't show the logout button when using SSO, but always show it in DeskApp
		if ( !container.getServerConfig().usingSSO() || Zarafa.isDeskApp ){
			var logoutButton = {
				text: _('Logout'),
				handler: this.onLogoutButton,
				id: 'mainmenu-button-logout'
			};

			this.add(logoutButton);
		}
	},

	/**
	 * Used to apply key shortcut and context name in tooltip.
	 * @param {Array} leftItems The leftItems contains the left context items which are properly sorted by priority.
	 * @param {Array} rightItems The rightItems contains the right context items.
	 */
	addTooltip: function(leftItems, rightItems)
	{
		var contextItems = [];
		contextItems = contextItems.concat(leftItems, rightItems);
		Ext.each(contextItems, function(context, index) {
			context.tooltip = context.text + ' (Ctrl + '+index+')';
		});
	},

	/**
	 * Event handler which is called when the user presses the 'logout' button
	 * @private
	 */
	onLogoutButton: function()
	{
		container.logout();
	}
});

Ext.reg('zarafa.maintabbar', Zarafa.core.ui.MainTabBar);
Ext.namespace('Zarafa.core.ui');

/**
 * @class Zarafa.core.ui.MainViewSidebar
 * @extends Ext.Panel
 * @xtype mainviewsidebar
 *
 * The sidebars for the {@link Zarafa.core.ui.MainViewport Main Viewport}
 * which support {@link #collapse collapsing} and {@link #stateful}.
 */
Zarafa.core.ui.MainViewSidebar = Ext.extend(Ext.Panel, {

	/**
	 * {@link Ext.QuickTips tooltip} message for {@link Zarafa.core.ui.MainViewSidebar panel}
	 * collapse {@link Ext.Button button}
	 * @property
	 * @type String
	 */
	collapseQuickTip: undefined,

	/**
	 * {@link Ext.QuickTips tooltip} message for {@link Zarafa.core.ui.MainViewSidebar panel}
	 * expand {@link Ext.Button button}
	 * @property
	 * @type String
	 */
	expandQuickTip: undefined,

	/**

	/**
	 * @constructor
	 * @param {Object} config Configuration object
	 */
	constructor: function(config)
	{
		config = config || {};

		config.plugins = Ext.value(config.plugins, []);

		Ext.applyIf(config, {
			cls: 'zarafa-context-mainpanel',
			border: false,
			collapsible: true,
			collapsed: false,
			split: true,
			width: 300,
			minSize: 150
		});

		Zarafa.core.ui.MainViewSidebar.superclass.constructor.call(this, config);
	},

	/**
	 * Called during rendering of the panel, this will initialize all events.
	 * @private
	 */
	initEvents: function ()
	{
		this.on('afterlayout', this.onAfterLayoutPanel, this, {single: true});
		this.on('collapse', this.onBeforeCollapsePanel, this, {single: true});
	},

	/**
	 * Event handler which is called after {@link Zarafa.core.ui.MainViewSidebar panel} get layout
	 * This will set {@link Ext.QuickTips} on {@link Ext.Button collapse} button
	 * @private
	 */
	onAfterLayoutPanel: function ()
	{
		if (this.collapseQuickTip) {
			Ext.QuickTips.register({
				target: this.tools.toggle,
				text: this.collapseQuickTip
			});
		}
	},

	/**
	 * Event handler which is called before {@link Zarafa.core.ui.MainViewSidebar panel} collapse
	 * This will set {@link Ext.QuickTips} on {@link Ext.Button expand} button
	 * @private
	 */
	onBeforeCollapsePanel: function ()
	{
		if (this.expandQuickTip) {
			Ext.QuickTips.register({
				target: Ext.get(this.id + '-xcollapsed').first(),
				text: this.expandQuickTip
			});
		}
	},

	/**
	 * Obtain the path in which the {@link #getState state} must be saved.
	 * This option is only used when the {@link Zarafa.core.data.SettingsStateProvider SettingsStateProvider} is
	 * used in the {@link Ext.state.Manager}. This returns {@link #statefulName} if provided, or else generates
	 * a custom name.
	 * @return {String} The unique name for this component by which the {@link #getState state} must be saved.
	 */
	getStateName: function()
	{
		return 'sidebars/' + Zarafa.core.ui.MainViewSidebar.superclass.getStateName.call(this);
	}
});

Ext.reg('mainviewsidebar', Zarafa.core.ui.MainViewSidebar);
Ext.namespace('Zarafa.core.ui');

/**
 * @class Zarafa.core.ui.MainViewport
 * @extends Ext.Viewport
 * The main viewport for the application. It defines the basic switching context
 * containers (toolbar, main content panel)
 */
Zarafa.core.ui.MainViewport = Ext.extend(Ext.Viewport, {
	// Insertion points for this class
	/**
	 * @insert main.content
	 * Insertion point for the main panel content area.
	 */

	/**
	 * The {@link Zarafa.core.ui.NavigationPanel NavigationPanel}.
	 * @property
	 * @type Zarafa.core.ui.NavigationPanel
	 */
	navigationPanel: undefined,

	/**
	 * The {@link Zarafa.core.ui.widget.WidgetPanel WidgetPanel}
	 * @property
	 * @type Zarafa.core.ui.widget.WidgetPanel
	 */
	widgetPanel: undefined,

	/**
	 * The {@link Zarafa.core.ui.ContextContainer ContextContainer}
	 * @property
	 * @type Zarafa.core.ui.ContextContainer
	 */
	contentPanel: undefined,

	/**
	 * @constructor
	 * @param {Object} config Configuration object
	 */
	constructor: function(config)
	{
		config = config || {};
		config = Ext.applyIf(config, {
			layout: 'fit',
			items: [{
				xtype: 'container',
				id: 'zarafa-mainview',
				layout: 'border',
				cls: 'zarafa-panel-body',
				border: false,
				items: [
					this.createTopbarContainer(),
					this.createNavigationPanel(),
					this.createTodayPanel(),
					this.createContentContainer()				]
			}]
		});

		// initialise the viewport with some pre-defined settings
		Zarafa.core.ui.MainViewport.superclass.constructor.call(this, config);

		// Activate global key events.
		Zarafa.core.KeyMapMgr.activate(null, 'global', Ext.getBody());
		// Don't allow propagation of all other key events which are registered in KeyMapMgr.
		Zarafa.core.KeyMapMgr.activate(null, 'globaldisable', Ext.getBody());
	},

	/**
	 * Create the {@link Zarafa.core.ui.NavigationPanel NavigationPanel} for
	 * the west region of the client in which the treePanel can be shown.
	 * @return {Zarafa.core.ui.NavigationPanel} NavigationPanel.
	 * @private
	 */
	createNavigationPanel: function()
	{
		this.navigationPanel = new Zarafa.core.ui.NavigationPanel({
			region: 'west',
			stateful: true,
			statefulName: 'hierarchybar',
			statefulRelativeDimensions: false
		});
		return this.navigationPanel;
	},

	/**
	 * Returns the {@link Zarafa.core.ui.NavigationPanel NavigationPanel} for
	 * the west region of the client in which the treePanel can be shown.
	 * @return {Zarafa.core.ui.NavigationPanel} NavigationPanel.
	 */
	getNavigationPanel: function()
	{
		return this.navigationPanel;
	},

	/**
	 * Create the {@link Zarafa.core.ui.widget.WidgetPanel WidgetPanel} for
	 * the east region of the client in which the Today view can be shown.
	 * @return {Object} Configuration object for the WidgetPanel.
	 * @private
	 */
	createTodayPanel: function()
	{
		var hide = container.getServerConfig().isWidgetEnabled() ? container.getSettingsModel().get('zarafa/v1/widgets/sidebar/hide_widgetpanel', true, false) : true;
		this.widgetPanel = new Zarafa.core.ui.widget.WidgetPanel({
			region: 'east',
			title: _('Widgets'),
			numColumns: 1,
			stateful: true,
			statefulName: 'todaybar',
			statefulRelativeDimensions: false,
			settingsPath: 'zarafa/v1/contexts/today/sidebar',
			hidden: hide,
			collapsed: true
		});
		return this.widgetPanel;
	},

	/**
	 * Returns the {@link Zarafa.core.ui.widget.WidgetPanel WidgetPanel} for
	 * the east region of the client in which the widgets can be shown.
	 * @return {Zarafa.core.ui.widget.WidgetPanel} widgetPanel
	 */
	getWidgetPanel: function()
	{
		return this.widgetPanel;
	},

	/**
	 * Create the {@link Zarafa.core.ui.ContextContainer ContextContainer} for
	 * the center region of the client in which the main contents can be shown.
	 * @return {Object} Configuration object for the ContextContainer
	 * @private
	 */
	createContentContainer: function()
	{
		var cc = new Zarafa.core.ui.ContextContainer({
			name: 'main.content',
			id: 'zarafa-mainpanel-content'
		});
		//get items from insertion point
		//TODO: create separate insertion points for front and back of tab strip?
		var lazyItems = container.populateInsertionPoint('main.content.tabpanel', this);

		this.contentPanel = new Zarafa.core.ui.MainContentTabPanel({
			id: 'zarafa-mainpanel',
			activeTab: 0,
			region: 'center',
			enableTabScroll: true,
			layoutOnTabChange: true,
			items: [ cc ].concat(lazyItems),
			plugins: [ 'zarafa.tabclosemenuplugin' ],
			cls: 'zarafa-body-tabbar',
			listeners: {
				"afterrender": function() {
					container.fireEvent('afterrendercontentpanel', this);
				}
			}
		});
		return this.contentPanel;
	},

	/**
	 * Returns the {@link Zarafa.core.ui.ContextContainer ContentPanel} for
	 * the center region of the client in which the context will display its contents
	 * @return {Zarafa.core.ui.ContextContainer} contentPanel
	 */
	getContentPanel: function()
	{
		return this.contentPanel;
	},

	/**
	 * Create a {@link Ext.Container Container} that contains the
	 * {@link Zarafa.core.ui.MainTabBar MainTabBar} and the {@link Zarafa.core.ui.MainTab MainTab}
	 * at the top of the client.
	 * @return {Ext.Container} Container holding the MainTabBar and MainTab
	 * @private
	 */
	createTopbarContainer: function()
	{
		return new Ext.Container({
			name: 'main.topbar',
			region: 'north',
			layout: 'border',
			height: 36+54,
			items: [
				new Zarafa.core.ui.MainTabBar({
					name: 'main.maintabbar',
					region: 'center',
					height: 36,
					boxMinHeight: 36,
					boxMaxHeight: 36,
					ref: '../../mainTabBar'
				}),
				new Zarafa.core.ui.MainToolbar({
					name: 'main.toolbar',
					region: 'south',
					height: 54,
					boxMinHeight: 54,
					boxMaxHeight: 54,
					ref: '../../mainToolbar'
				})
			]
		});
	}
});
Ext.namespace('Zarafa.core.ui');

/**
 * @class Zarafa.core.ui.PreviewPanel
 * @extends Ext.Panel
 * @xtype zarafa.previewpanel
 *
 * Panel that shows a preview of a {@link Zarafa.core.data.IPMRecord message}.
 */
Zarafa.core.ui.PreviewPanel = Ext.extend(Ext.Panel, {
	// Insertion points for this class
	/**
	 * @insert previewpanel.toolbar.left
	 * Insertions point for inserting buttons in previewpane's toolbar on left side.
	 * @param {Zarafa.core.ui.PreviewPanel} panel This panel
	 * @param {Zarafa.core.ContextModel} model The contextmodel ({@link #model}).
	 */

	/**
	 * @insert previewpanel.toolbar.right
	 * Insertions point for inserting buttons in previewpane's toolbar on right side.
	 * @param {Zarafa.core.ui.PreviewPanel} panel This panel
	 * @param {Zarafa.core.ContextModel} model The contextmodel ({@link #model}).
	 */

	/**
	 * @cfg {Zarafa.core.ContextModel} model The model for which this
	 * PreviewPanel is being loaded.
	 */
	model: undefined,

	/**
	 * Reference to the {@link Zarafa.core.plugins.RecordComponentPlugin RecordComponent} plugin
	 * which is used to send update events to all child {@link Ext.Container containers}
	 * in this container. This field is initialized by the
	 * {@link Zarafa.core.plugins.RecordComponentPlugin RecordComponent} itself.
	 *
	 * @property
	 * @type Zarafa.core.plugins.RecordComponentPlugin
	 */
	recordComponentPlugin: undefined,

	/**
	 * @cfg {Object} Configuration object which will be used
	 * to instantiate the {@link Zarafa.core.plugins.RecordComponentPlugin RecordComponent}.
	 * See the plugin for the available configuration options.
	 */
	recordComponentPluginConfig: undefined,

	/**
	 * @cfg {Zarafa.core.data.IPMRecord} record (See {@link Zarafa.core.plugins.RecordComponentPlugin#record}).
	 */
	record: undefined,

	/**
	 * @cfg {Boolean} isLoadMaskShown true if load mask should be shown else false.
	 */
	isLoadMaskShown: false,

	/**
	 * The LoadMask object which will be shown when the {@link #record} is being opened, and
	 * the dialog is waiting for the server to respond with the desired data. This will only
	 * be set if {@link #showLoadMask} is true.
	 * @property
	 * @type Zarafa.common.ui.LoadMask
	 */
	loadMask: undefined,

	/**
	 * @constructor
	 * @param {Object} config configuration object.
	 */
	constructor: function(config)
	{
		config = config || {};

		config = Ext.applyIf(config, {
			layout: 'fit',
			stateful: true,
			minWidth: 200,
			minHeight: 200,
			xtype: 'zarafa.previewpanel'
		});

		config.tbar = Ext.applyIf(config.tbar || {}, {
			cls: 'zarafa-previewpanel-toolbar',
			xtype: 'zarafa.toolbar',
			hidden: true,
			items: []
		});

		var tbarItems = [
			container.populateInsertionPoint('previewpanel.toolbar.left', this, config.model),
			config.tbar.items, // Default items in toolbar should be left aligned.
			{xtype: 'tbfill'},
			container.populateInsertionPoint('previewpanel.toolbar.right.first', {scope: this, model: config.model}),
			container.populateInsertionPoint('previewpanel.toolbar.right', {scope: this, model: config.model})
		];

		config.tbar.items = Ext.flatten(tbarItems);
		config.plugins = Ext.value(config.plugins, []);
		config.plugins.push(Ext.applyIf(config.recordComponentPluginConfig || {}, {
			ptype: 'zarafa.recordcomponentplugin',
			useShadowStore: true,
			allowWrite: false
		}));

		config.plugins.push({
			ptype: 'zarafa.recordcomponentupdaterplugin'
		}, {
			ptype: 'zarafa.enablefocusplugin'
		});

		if (container.getSettingsModel().get('zarafa/v1/contexts/mail/readflag_time_enable') === true) {
			config.plugins.push({
				ptype: 'zarafa.markasreadplugin'
			});
		}

		Zarafa.core.ui.PreviewPanel.superclass.constructor.call(this, config);

		if (this.model) {
			this.mon(this.model, 'previewrecordchange', this.onPreviewRecordChange, this);
		}

		this.on({
			'beforeloadrecord': this.onBeforeLoadRecord,
			'loadrecord': this.onLoadRecord,
			'exceptionrecord': this.onExceptionRecord,
			'show': this.onPreviewPanelShow,
			'expand': this.onPreviewPanelShow,
			'scope': this
		});
	},

	/**
	 * Called when the ContentPanel has been rendered.
	 * This activate the keymap on the element of this component after the normal operations of
	 * afterRender have been completed. It will activate by getting the xtype hierarchy from
	 * {@link #getXTypes} and format it into a string usable by the
	 * {@link Zarafa.core.KeyMapMgr KeyMapMgr}.
	 * @private
	 */
	afterRender: function()
	{
		Zarafa.core.ui.PreviewPanel.superclass.afterRender.apply(this, arguments);
		var xtypes = this.getXTypes();

		// The first part leading up to zarafa.previewpanel will be stripped
		xtypes = xtypes.replace('component/box/container/panel/zarafa.previewpanel','');

		// Then the "zarafa." will be stripped off from start, the xtype like "zarafa.previewpanel".
		xtypes = xtypes.replace(/\/zarafa\./g,'.');

		// Finally we string the "previewpanel" from all the xtypes. Otherwise each level will have
		// that "previewpanel" mentioned in them. Also we add previewpanel to the start as that sets
		// it apart from other components in the key mapping.
		xtypes = 'previewpanel' + xtypes.replace(/previewpanel/g, '');

		// This will activate keymaps with 'previewpanel.mail' so the component
		// will have all key events register with 'previewpanel', 'mail'
		Zarafa.core.KeyMapMgr.activate(this, xtypes);
	},

	/**
	 * See {@link Zarafa.core.plugins.RecordComponentPlugin#setRecord}.
	 *
	 * @param {Zarafa.core.data.MAPIRecord} record The record to set
	 */
	setRecord: function(record)
	{
		for (var i = 0; i < this.toolbars.length; i++) {
			this.toolbars[i].setVisible(!!record);
		}

		if (this.recordComponentPlugin) {
			this.recordComponentPlugin.setRecord(record);
		}

		// Remove the update handler of the record in the preview panel
		// before we set a new one
		if ( this.record && this.record.store ){
			this.record.store.un('update', this.updatePreviewPanel, this);
		}
		if ( record && record.store ){
			record.store.on('update', this.updatePreviewPanel, this);
		}
	},

	/**
	 * Updates the categories in the preview panel when the data of the record changes.
	 *
	 * @param {Zarafa.core.data.IPMStore} store The store to which the record belongs
	 * that is shown in the preview panel
	 * @param {Zarafa.core.data.IPMRecord} record The Record that was updated
	 */
	updatePreviewPanel: function(store, record)
	{
		// Only update when we are showing a record in the preview panel
		if ( this.record && record.equals(this.record) ){
			this.record.set('categories', record.get('categories'));
			this.doLayout();
		}
	},

	/**
	 * Update the components with the given record.
	 *
	 * @param {Zarafa.core.data.MAPIRecord} record The record to update in this component
	 * @param {Boolean} contentReset force the component to perform a full update of the data.
	 */
	update: function(record, contentReset)
	{
		this.record = record;
	},

	/**
	 * Function for 'previewrecordchange' and 'show' events before setting record into component
	 * @param {Zarafa.core.data.MAPIRecord} record
	 * @private
	 */
	showRecordInPanel: function(record)
	{
		var panelConstructor;

		// Record is already loaded...
		if (this.record === record) {
			return;
		}

		// Check if same record is loaded again
		if (this.record && record && this.record.equals(record)) {
			return;
		}

		if (Ext.isDefined(record)) {
			// TODO: Find a proper way to suppress exception popup
			record.suppressException();
			panelConstructor = container.getSharedComponent(Zarafa.core.data.SharedComponentType['common.preview'], record);
			if (panelConstructor && this.get(0) instanceof panelConstructor) {
				if(this.isLoadMaskShown){
					this.hideLoadMask();
				}
				this.setRecord(record);
				return;
			}
		}

		// Do a complete refresh, clearing all UI components
		// which might be active inside the panel.
		this.removeAll();

		if (panelConstructor) {
			this.add(new panelConstructor());
		}

		this.setRecord(record);
		this.doLayout();
		this.hideLoadMask();
	},

	/**
	 * Removes all components from this container. Additionally this function will also remove all the items from
	 * {@link Ext.Toolbar toolbar} also.
	 * @param {Boolean} autoDestroy (optional) True to automatically invoke the removed Component's {@link Ext.Component#destroy} function.
	 * Defaults to the value of this Container's {@link #autoDestroy} config.
	 * @return {Array} Array of the destroyed components
	 */
	removeAll: function(autoDestroy)
	{
		// remove toolbar items first
		var destroyedItems = [];
		if(this.getTopToolbar()) {
			destroyedItems.concat(this.getTopToolbar().removeAll.apply(this, arguments));
		}

		if(this.getBottomToolbar()) {
			destroyedItems.concat(this.getBottomToolbar().removeAll.apply(this, arguments));
		}

		destroyedItems.concat(Zarafa.core.ui.PreviewPanel.superclass.removeAll.apply(this, arguments));

		return destroyedItems;
	},

	/**
	 * Event handler which is trigggerd when the {@link Zarafa.core.ContextModel ContextModel} receives a new
	 * record for previewing (it fired the {@link Zarafa.core.ContextModel#previewrecordchange previewrecordchange} event).
	 * This updates the {@link Ext.Container previewpanel} with the selected record to be previewed.
	 *
	 * @param {Zarafa.core.ContextModel} model The context model.
	 * @param {Ext.data.Record} record The record which is selected for preview.
	 * @private
	 */
	onPreviewRecordChange: function(contextModel, record)
	{
		// Load record in preview panel
		if(this.isVisible()) {
			this.showRecordInPanel(record);
		}
	},

	/**
	 * If {@link #showLoadMask} is enabled, this function will display
	 * the {@link #loadMask}.
	 * @param {Boolean} errorMask True to show an error mask instead of the loading mask.
	 * @protected
	 */
	showLoadMask: function(errorMask)
	{
		if (this.isLoadMaskShown && !errorMask) {
			return;
		}
		if (!this.loadMask) {
			this.loadMask = new Zarafa.common.ui.LoadMask(this.el);
		}

		if (errorMask) {
			this.loadMask.showError();
		} else {
			this.loadMask.show();
			this.isLoadMaskShown = true;
		}
	},

	/**
	 * If {@link #showLoadMask} is enabled, and {@link #showLoadMask} has been
	 * called to display the {@link #loadMask} this function will disable the
	 * loadMask.
	 * @protected
	 */
	hideLoadMask: function()
	{
		if (this.isLoadMaskShown === false) {
			return;
		}

		if (this.loadMask) {
			this.loadMask.hide();
			this.isLoadMaskShown = false;
		}
	},

	/**
	 * Event handler for the {@link Zarafa.core.plugins.RecordComponentPlugin#beforeloadrecord beforeloadrecord} event.
	 * This will call the {@link #showLoadMask} function to show the loadmask.
	 *
	 * @param {Ext.Container} panel The panel to which the record belongs
	 * @param {Zarafa.core.data.IPMRecord} record The record which was updated
	 * @private
	 */
	onBeforeLoadRecord: function(panel, record)
	{
		this.showLoadMask();
	},

	/**
	 * Event handler for the {@link Zarafa.core.plugins.RecordComponentPlugin#loadrecord loadrecord} event.
	 * This will call the {@link #hideLoadMask} function to hide the loadmask.
	 *
	 * @param {Ext.Container} panel The panel to which the record belongs
	 * @param {Zarafa.core.data.IPMRecord} record The record which was updated
	 * @private
	 */
	onLoadRecord: function(panel, record)
	{
		this.hideLoadMask();
	},

	/**
	 * Event handler for the {@link Zarafa.core.plugins.RecordComponentPlugin#exceptionrecord} event.
	 * This will call {@link #showLoadMask} to update it with a new error message.
	 *
	 * @param {String} type See {@link Ext.data.DataProxy}.{@link Ext.data.DataProxy#exception exception}
	 * for description.
	 * @param {String} action See {@link Ext.data.DataProxy}.{@link Ext.data.DataProxy#exception exception}
	 * for description.
	 * @param {Object} options See {@link Ext.data.DataProxy}.{@link Ext.data.DataProxy#exception exception}
	 * for description.
	 * @param {Object} response See {@link Ext.data.DataProxy}.{@link Ext.data.DataProxy#exception exception}
	 * for description.
	 * @param {Zarafa.core.data.MAPIRecord} record The record which was subject of the request
	 * that encountered an exception.
	 * @param {String} error (Optional) Passed when a thrown JS Exception or JS Error is
	 */
	onExceptionRecord: function(type, action, options, response, record)
	{
		this.showLoadMask(true);
	},

	/**
	 * Event handler which is fired when this panel has been set to visible. Fetches previewRecord from
	 * the {@link Zarafa.core.ContextModel ContextModel} and displays into panel.
	 */
	onPreviewPanelShow: function()
	{
		if (Ext.isDefined(this.model)) {
			var record = this.model.getPreviewRecord();
			if (Ext.isDefined(record)) {
				this.showRecordInPanel(record);
			}
		}
	},

	/**
	 * Obtain the path in which the {@link #getState state} must be saved.
	 * This option is only used when the {@link Zarafa.core.data.SettingsStateProvider SettingsStateProvider} is
	 * used in the {@link Ext.state.Manager}. This returns {@link #statefulName} if provided, or else generates
	 * a custom name.
	 * @return {String} The unique name for this component by which the {@link #getState state} must be saved.
	 */
	getStateName: function()
	{
		return 'preview/' + Zarafa.core.ui.PreviewPanel.superclass.getStateName.call(this);
	}
});
Ext.reg('zarafa.previewpanel', Zarafa.core.ui.PreviewPanel);
Ext.namespace('Zarafa.core.ui');

/**
 * @class Zarafa.core.ui.SwitchViewContentContainer
 * @extends Ext.Container
 * @xtype zarafa.switchviewcontentcontainer
 *
 * This class represents an {@link Ext.Panel panel} which contains multiple views which
 * can be enabled/disabled at any time by the user. Using lazy loading each view will only
 * be allocated when the user switches to the given view for the first time.
 */
Zarafa.core.ui.SwitchViewContentContainer = Ext.extend(Ext.Container, {
	/**
	 * @cfg {Object[]} lazyItems List of {@link Ext.Containers containers} which
	 * act as view for this {@link Ext.Container container}. This array consists of configuration
	 * objects which will be allocated when the user switches to the given view for the
	 * first time.
	 * All items added to this list must always contain the 'id' property,
	 * used in switchView. Without this property it is not possible to switch
	 * to this view.
	 */
	lazyItems: undefined,

	/**
	 * @cfg {Boolean} autoClean If true this container will automatically remove
	 * and delete the previously selected {@link Ext.Container container} when switching
	 * to a new active {@link Ext.Container container}.
	 */
	autoClean: true,

	/**
	 * @constructor
	 * @param {Object} config Configuration object
	 */
	constructor: function(config)
	{
		config = config || {};

		// Always ensure that at least 1 item is non-lazy
		if (Ext.isEmpty(config.items) && !Ext.isEmpty(config.lazyItems)) {
			config.items = [ config.lazyItems[0] ];
		}
		// Ensure that the non-lazy item is marked as active
		if (Ext.isEmpty(config.activeItem)) {
			config.activeItem = config.items[0].id;
		}

		Ext.applyIf(config, {
			autoDestroy: true
		});

		this.addEvents(
			/**
			 * @event switchview
			 * Fires when the active view is being changed
			 * @param {Ext.Container} this The {@link Zarafa.core.ui.SwitchViewContentContainer switchcontainer}.
			 * @param {Ext.Container} newView The new {@link Ext.Container view} which is shown
			 * @param {Ext.Container} oldView The old {@link Ext.Container view} which was shown
			 */
			'switchview'
		);

		Zarafa.core.ui.SwitchViewContentContainer.superclass.constructor.call(this, config);
	},

	/**
	 * Called by Extjs when the container is being {@link #doLayout laid out}. This will obtain
	 * the {@link Ext.layout.CardLayout#activeItem} and {@link Ext.Panel#doLayout update the layout}
	 * on that component as well.
	 * @private
	 */
	onLayout: function()
	{
		Zarafa.core.ui.SwitchViewContentContainer.superclass.onLayout.apply(this, arguments);

		// If the activeItem contains a layout, it should be laid out as well
		var item = this.getActiveItem();
		if (Ext.isFunction(item.doLayout)) {
			item.doLayout();
		}
	},

	/**
	 * This function will be used to switch between different views.
	 * It will attempt to find the view within this panel, if found,
	 * then the layout manager will be updated with the new view.
	 * @param {String} viewId id of the view that should be shown
	 */
	switchView: function(viewId)
	{
		var oldView = this.getActiveItem();
		var newView = this.findById(viewId);

		if (!Ext.isDefined(newView) || oldView == newView) {
			return;
		}

		// Check if the layout has been created yet, if not
		// then we store the activeItem inside the current
		// panel so it can be applied to the layout when it
		// is being created.
		var layout = this.getLayout();
		if (!Ext.isFunction(layout.setActiveItem)) {
			this.activeItem = viewId;
		} else {
			layout.setActiveItem(viewId);
		}

		this.fireEvent('switchview', this, newView, oldView);

		// TODO: We should enable some timeout mechanism which
		// removes and deletes the oldView after a particular timeout.
		// This should increase performance when switching between 2
		// views often.
		if (this.autoClean === true && oldView && oldView != this.getActiveItem()) {
			this.remove(oldView);
			delete oldView;
		}
	},

	/**
	 * This function returns the currently active item
	 * @return {Ext.Component} The active item
	 */
	getActiveItem: function()
	{
		var layout = this.getLayout();
		if (!Ext.isFunction(layout.setActiveItem)) {
			return this.activeItem;
		} else {
			return layout.activeItem;
		}
	},

	/**
	 * Find a component under this container at any level by id.
	 * This extension will read all lazy items as well, if the required id, is
	 * one of the lazy items, then the item will be created and added to the panel.
	 * @param {String} id
	 * @return Ext.Component
	 */
	findById: function(id)
	{
		var retval = Zarafa.core.ui.SwitchViewContentContainer.superclass.findById.call(this, id);
		if (!retval) {
			retval = this.findBy(function(item) { return item.id === id; });
			if (!Ext.isEmpty(retval)) {
				retval = retval[0];
			}
		}

		return retval;
	},

	/**
	 * Find a component under this container at any level by a custom function.
	 * If the passed function returns true, the component will be included in the results.
	 * The passed function is called with the arguments (component, this container).
	 *
	 * This function will not only search through {@link #items} but also through {@link #lazyItems}.
	 *
	 * @param {Function} fn The function to call
	 * @param {Object} scope (optional)
	 * @return {Array} Array of Ext.Components
	 */
	findBy: function(fn, scope)
	{
		var retval = Zarafa.core.ui.SwitchViewContentContainer.superclass.findBy.apply(this, arguments);

		if (Ext.isDefined(this.lazyItems)) {
			for (var i = 0; i < this.lazyItems.length; i++) {
				var item = this.lazyItems[i];

				if (fn.call(scope || item, item, this) === true) {
					/*
					 * Make an exact clone of the lazyItems entry. When we pass the config
					 * object to the Component constructor, we must consider the fact that
					 * this would be a reference. What we pass here, might be modified by
					 * the constructor. However we do not want those changes to be saved
					 * back into our lazyItems object, since that might cause problems
					 * when we instantiate the lazyItem for the second time.
					 */
					item = Ext.create(Ext.apply({}, item));
					this.add(item);
					retval.push(item);
				}
			}
		}

		return retval;
	}
});

Ext.reg('zarafa.switchviewcontentcontainer', Zarafa.core.ui.SwitchViewContentContainer);
Ext.namespace('Zarafa.core.ui');

/**
 * @class Zarafa.core.ui.Toolbar
 * @extends Ext.Toolbar
 * @xtype zarafa.toolbar
 */
Zarafa.core.ui.Toolbar = Ext.extend(Ext.Toolbar, {
	// Private
	// Used to keep track of whether the toolbar buttongroups have been corrected in size yet.
	initialSizeCorrectionDone: false,

	/**
	 * @constructor
	 * @param config
	 */
	constructor: function(config)
	{
		Ext.apply(this, config, {
			// Override from Ext.Component
			xtype: 'zarafa.toolbar',
			// Override from Ext.Toolbar
			enableOverflow: true
		});

		Zarafa.core.ui.Toolbar.superclass.constructor.call(this, config);

		this.on('afterlayout', this.onAfterLayout, this);
	},

	/**
	 * Add a new {@link Ext.ButtonGroup} to the {@link Ext.Toolbar} using an insertionpoint to support
	 * the addition of extra {@link Ext.Button} elements to the {@link Ext.ButtonGroup}. If the insertion
	 * point contains a {@link Ext.ButtonGroup} it will be added outside the given {@link Ext.ButtonGroup}
	 * to prevent nesting.
	 *
	 * @param {Object/String} group The {@link Ext.ButtonGroup} to which individual {@link Ext.Button} elements must be added.
	 * If this argument is a {@link String} it will be used as title for the {@link Ext.ButtonGroup}.
	 * @param {String} insertion (optional) The name of the insertion point used to add additional {@link Ext.Button} and
	 * {@link Ext.ButtonGroup} elements.
	 * @protected
	 */
	addItems: function(buttons, insertion)
	{
		var items = [];

		if (Ext.isDefined(buttons)) {
			if (!Ext.isEmpty(buttons)) {
				items = items.concat(buttons);
			}
		}

		if (Ext.isString(insertion)) {
			var insert = container.populateInsertionPoint(insertion);
			if (!Ext.isEmpty(insert)) {
				items = items.concat(insert);
			}
		}

		// Make sure we get a single array containing all buttons
		items = Ext.flatten(items);

		this.add(items);
	},

	/**
	 * When the toolbar is laid out the sizes for the different buttongroups may differ. When the
	 * afterlayout event is fired the heights are corrected to have all the buttongroups match in
	 * height. This is only done the first time the toolbar is shown.
	 * @private
	 */
	onAfterLayout: function()
	{
		// Only do it the first time the toolbar is shown.
		if(!this.initialSizeCorrectionDone && this.el.getHeight() > 0){

			var elements = [];
			var maxHeight = 0;
			this.items.each(function(item){
				if (item.isXType('buttongroup')) {
					var elem = item.body;
					maxHeight = Math.max(maxHeight, elem.getHeight());
					elements.push(elem);
				}
			});

			if(maxHeight > 0){
				Ext.each(elements, function(elem){
					elem.setHeight(maxHeight);
				}, this);
				this.initialSizeCorrectionDone = true;
			}
		}
	},

	onAddItem: function(toolbar, item, index)
	{
		if ( item.isXType('menuitem') && Ext.isDefined(item.recordComponentUpdaterPlugin) ){
			// The plugin has called the update function of the item, but (for overflown toolbars)
			// when buttons have been changed to menuitems by Ext they do not have the defined
			// update function anymore, and use the update function of Ext.Component. This function
			// will change the text and we don't want this, so we set it back
			var html = item.itemTpl.apply({
				id: item.id,
				cls: item.cls,
				href: '#',
				hrefTarget: '',
				iconCls: item.iconCls,
				text: item.overflowText
			});
			var tmp = new Ext.Element(document.createElement('div'));
			tmp.dom.outerHTML = html;
			html = tmp.dom.innerHTML;
			item.getEl().dom.innerHtml = html;
		}
	}
});

Ext.reg('zarafa.toolbar', Zarafa.core.ui.Toolbar);
Ext.namespace('Zarafa.core.ui');

/**
 * @class Zarafa.core.ui.ToolbarButton
 * @extends Ext.Button
 * @xtype zarafa.toolbarbutton
 *
 * Special implementation for the {@link Ext.Button button}
 * which connects to the Mail {@link Ext.Panel panel} for the
 * currently active {@link Zarafa.core.Context context}.
 * Using this link, the {@link Ext.Button button} can be disabled
 * or enabled depending on the number of currently selected
 * {@link Zarafa.core.data.IPMRecord records}.
 *
 * Also this class makes it easier for the {@link Ext.Button button}
 * to request the currently selected {@link Zarafa.core.data.IPMRecord records}
 * on which it can perform an action.
 *
 * FIXME: This should somehow be merged with Zarafa.core.ui.menu.ConditionalItem.
 */
Zarafa.core.ui.ToolbarButton = Ext.extend(Ext.Button, {
	/**
	 * @cfg {Boolean} emptySelectOnly This button must only be enabled
	 * if no record is selected
	 */
	emptySelectOnly: false,
	/**
	 * @cfg {Boolean} nonEmptySelectOnly This button must only be enabled
	 * if one or more records are selected.
	 */
	nonEmptySelectOnly: false,
	/**
	 * @cfg {Boolean} singleSelectOnly This button must only be enabled
	 * if a single record is selected
	 */
	singleSelectOnly: false,
	/**
	 * @cfg {Boolean} multiSelectOnly This button must only be enabled
	 * if multiple records are selected.
	 */
	multiSelectOnly: false,
	/**
	 * @constructor
	 * @param {Object} config Configuration object
	 */
	constructor: function(config)
	{
		config = config || {};

		Ext.applyIf(config, {
			xtype: 'zarafa.toolbarbutton'
		});

		Ext.apply(this, config);

		Zarafa.core.ui.ToolbarButton.superclass.constructor.call(this, config);

		var model = this.model || container.getCurrentContext().getModel();
		if (Ext.isDefined(model)) {
			this.onRecordSelectionChange(model, model.getSelectedRecords());
			this.mon(model, 'recordselectionchange', this.onRecordSelectionChange, this);
		}
	},

	/**
	 * Event handler which is triggered when the current {@link Zarafa.core.data.IPMRecord record}
	 * selection changes. This will then evaluate if the {@link Ext.Button button} must be
	 * enabled or disabled.
	 *
	 * @param {Zarafa.core.ContextModel} model this model.
	 * @param {Zarafa.core.data.Record[]} records The selected records
	 */
	onRecordSelectionChange: function(model, records)
	{
		if (this.emptySelectOnly) {
			if (Ext.isDefined(records) && (!Array.isArray(records) || records.length > 0)) {
				this.setDisabled(true);
			} else {
				this.setDisabled(false);
			}
		} else if (this.nonEmptySelectOnly) {
			if (!Ext.isDefined(records) || (Array.isArray(records) && records.length === 0)) {
				this.setDisabled(true);
			} else {
				this.setDisabled(false);
			}
		} else if (this.singleSelectOnly) {
			if (!Ext.isDefined(records) || (Array.isArray(records) && records.length !== 1)) {
				this.setDisabled(true);
			} else {
				this.setDisabled(false);
			}
		} else if (this.multiSelectOnly) {
			if (!Ext.isDefined(records) || (!Array.isArray(records) || records.length === 1)) {
				this.setDisabled(true);
			} else {
				this.setDisabled(false);
			}
		}
	}
});

Ext.reg('zarafa.toolbarbutton', Zarafa.core.ui.ToolbarButton);
Ext.namespace("Zarafa.core.ui");

/**
 * @class Zarafa.core.ui.View
 * @extends Ext.util.Observable
 * The View class is a component for building views. Views abstract away the details
 * of manipulating HTML and the DOM from underlying controls such as grid panels
 * and {@link Zarafa.calendar.ui.CalendarPanel CalendarPanel}.
 *
 * The View class supports convenience methods for creating elements, managing
 * several child views, etc.
 * <p>
 * The createDiv() method can be used to create div elements. These elements are then
 * automatically destroyed by the destroy() method.
 * Child views can be added using addChildView(). These are then also destroyed
 * automatically by the default destroy() implementation.
 */
Zarafa.core.ui.View = Ext.extend(Ext.util.Observable, {
	/**
	 * @cfg {String} baseCls The base CSS class to apply to this panel's element.
	 */
	baseCls: undefined,

	/**
	 * @cfg {String} itemCls The item CSS class to apply to this panel's element.
	 */
	itemCls: undefined,

	/**
	 * @cfg {String} themeCls The CSS theme which must be applied to this view. The name will
	 * be added to the various CSS classes which are generated using {@link #getClassName}.
	 * will be used.
	 */
	themeCls: undefined,

	/**
	 * The parent element for this view
	 * @property
	 * @type Object
	 */
	parentView: undefined,

	/**
	 * The {@link Ext.Element} in which this view has been rendered.
	 * This is passed as argument to {@link #render}.
	 * @property
	 * @type Ext.Element
	 */
	container: undefined,

	/**
	 * The list of child objects which were created for this view. When
	 * this object is destroyed all registered children will be automatically
	 * cleaned up as well.
	 * @property
	 * @type Array
	 */
	children: undefined,

	/**
	 * The list of child elements which were created by this view using
	 * the {@link #create} and {@link #createDiv} functions. When this
	 * object is destroyed all registered elements will be automatically
	 * cleaned up as well.
	 * @property
	 * @type Array
	 */
	elements: undefined,

	/**
	 * Indicates if this view has been {@link #render rendered}.
	 * @property
	 * @type Boolean
	 */
	rendered: false,

	/**
	 * True if the view has been destroyed already. Read only
	 * @property isDestroyed
	 * @type Boolean
	 */
	isDestroyed: false,

	/**
	 * @constructor
	 * @param {Object} config Configuration object
	 */
	constructor: function(config)
	{
		Ext.apply(this, config);

		this.addEvents(
			/**
			 * @event beforedestroy
			 * Fires before the component is {@link #destroy}ed. Return false from an event handler to stop the {@link #destroy}.
			 * @param {Ext.Component} this
			 */
			'beforedestroy',
			/**
			 * @event destroy
			 * Fires after the component is {@link #destroy}ed.
			 * @param {Ext.Component} this
			 */
			'destroy'
		);

		Zarafa.core.ui.View.superclass.constructor.call(this, config);

		this.init();
	},

	/**
	 * Initialises the view. When a {@link #parentView} is defined,
	 * this will automatically register this view to the given parentView.
	 * @protected
	 */
	init: function()
	{
		this.children = [];
		this.elements = [];

		if (Ext.isDefined(this.parentView)) {
			this.parentView.addChildView(this);
		}
	},

	/**
	 * Returns the base className which must be applied to all {@link Ext.Element elements} within this
	 * view. This will also be the prefix of any additional CSS classes which will be applied to those elements.
	 * The exact base classname depends on the {@link #baseCls} and the availability of the {@link #itemCls}.
	 * @return {String} The base class.
	 * @private
	 */
	getBaseClassName: function()
	{
		return this.itemCls ? (this.baseCls + '-' + this.itemCls) : this.baseCls;
	},

	/**
	 * Helper function for generating a string with CSS class names. Each {@link Ext.Element Element} must
	 * at least contain the {@link #baseCls} CSS class and CSS classes containing the {@link #itemCls} and
	 * {@link #themeCls} but additional names will be generated for specific elements. Example:
	 * <pre>
	 *  baseCls = 'zarafa-calendar';
	 *  getClassName('body', 'image');
	 * </pre>
	 * The outcome will be:
	 * <pre>
	 *  'zarafa-calendar zarafa-calendar-body zarafa-calendar-body-image zarafa-calendar-blue zarafa-calendar-blue-body zarafa-calendar-blue-body-image'
	 * </pre>
	 *
	 * @param {String} name The main name for the CSS element, this will be appended to the {@link #baseCls}.
	 * @param {String/Array} postfix (optional) If provided extra CSS classNames will be generated with the name and postfix. If the postfix
	 * is an array, all elements from the array will be used as postfix.
	 * @return {String} the CSS class names
	 * @param {String/Array} extraCls (optional) If provided extra CSS classNames will be generated with this extraCls. If the postfix
	 * is an array, all elements from the array will be used as extraCls.
	 * @private
	 */
	getClassName: function(name, postfix, extraCls)
	{
		var baseCls = this.getBaseClassName();
		var className = this.baseCls;

		if (!Ext.isEmpty(this.themeCls)) {
			className += ' ' + this.baseCls + '-' + this.themeCls;
		}

		className += ' ' + baseCls + '-' + name;

		if (Ext.isDefined(postfix) && !Array.isArray(postfix)) {
			postfix = [ postfix ];
		}

		if (!Ext.isEmpty(postfix)) {
			for (var i = 0, len = postfix.length; i < len; i++) {
				className += ' ' + baseCls + '-' + name + '-' + postfix[i];
			}
		}

		if (Ext.isDefined(extraCls) && !Array.isArray(extraCls)) {
			extraCls = [ extraCls ];
		}

		if (!Ext.isEmpty(extraCls)) {
			for (var i = 0, len = extraCls.length; i < len; i++) {
				className += ' ' + baseCls + '-' + extraCls[i];
			}
		}

		return className;
	},

	/**
	 * A convenience method for creating DIV elements. The new DIV element will be
	 * wrapped in an {@link Ext.Element} and added to the view as a property called 'name'.
	 * Equivalent to <code>create('div', parent, name, className)</code>.
	 * @param {Mixed} parent element
	 * @param {String} name the name of the property to create
	 * @param {String} className (optional) class name for the new DIV
	 * @return {Ext.Element} The created div element
	 * @private
	 */
	createDiv: function(parent, name, className)
	{
		return this.create('div', parent, name, className);
	},

	/**
	 * A convenience method for creating DOM elements. The new element will be
	 * wrapped in an {@link Ext.Element} and added to the view as a property called 'name'.
	 * Calling <pre>this.create('div', 'test')</pre> for instance will create a new
	 * property <pre>this.test</pre>, and the DIV can be manipulated with code such
	 * as <pre>this.test.setSize(100, 100);</pre>. If a property already exists
	 * with the given name, the property will be converted into an array and
	 * the new element will the appended to that array.
	 * @param {String|Object} tagName type of HTML tag to generate (i.e. 'div', 'img', etc.)
	 * @param {Mixed} parent element
	 * @param {String} name the name of the property to create
	 * @param {String} className (optional) class name for the new DIV
	 * @return {Ext.Element} The created element
	 * @private
	 */
	create: function(tagName, parent, name, className)
	{
		var config = Ext.isObject(tagName) ? tagName : { tag: tagName };

		Ext.applyIf(config, { cls: className || this.baseCls });

		// create a new HTML element and add it to the parent element
		var element = Ext.get(parent).createChild(config);

		// add the element as a property to this
		if (!this[name]) {
			this[name] = element;
		} else if (Array.isArray(this[name])) {
			this[name].push(element);
		} else {
			this[name] = [ this[name], element ];
		}

		// add the element to the elements list so that we may destroy it easily later on
		this.elements.push(element);

		return element;
	},

	/**
	 * Removes a child element from the view and destroys it.
	 * @param {Ext.Element} element DOM element
	 */
	remove: function(element)
	{
		element = Ext.get(element);
		this.elements.remove(element);
		element.remove();
	},

	/**
	 * @return {Ext.Element} The main element of the view. This is the element the
	 * view is rendered <i>into</i>. The container of a view may be shared with other
	 * views, so manipulating the container of a given view might affect other views
	 * as well.
	 */
	getContainer: function()
	{
		return this.container;
	},

	/**
	 * Calls 'render' on each of the child views
	 * @param {Ext.Element} container The container into which the children must be rendered. Defaults to {@link #el}.
	 * @private
	 */
	renderChildren: function(container)
	{
		for (var i = 0, len = this.children.length; i < len; i++) {
			this.children[i].render(container || this.container);
		}
	},

	/**
	 * Renders the view.
	 * Rendering a view is done once after construction, and creates all the DOM elements needed for the visual
	 * representation of the view.
	 * @param {Ext.Element} container The Ext.Element into which the view must be rendered.
	 */
	render: function(container)
	{
		this.container = container;
		// TODO Can't we call renderChildren here?
		this.rendered = true;
	},

	/**
	 * Calls 'layout' on each of the child views.
	 * @private
	 */
	layoutChildren: function()
	{
		for (var i = 0, len = this.children.length; i < len; i++) {
			this.children[i].layout();
		}
	},

	/**
	 * Called just before this View will be {@link #layout laid out}.
	 * @protected
	 */
	onBeforeLayout: Ext.emptyFn,

	/**
	 * Called when this view is being {@link #layout laid out}.
	 * @protected
	 */
	onLayout: Ext.emptyFn,

	/**
	 * Called just after this View has been {@link #layout laid out}.
	 * @protected
	 */
	onAfterLayout: Ext.emptyFn,

	/**
	 * Lays out the view, setting the position and size of the individual DOM elements.
	 */
	layout: function()
	{
		this.onBeforeLayout();
		this.onLayout();
		this.onAfterLayout();
	},

	/**
	 * Calls 'destroy' on each of the child views.
	 * @private
	 */
	destroyChildren: function()
	{
		for (var i = 0, len = this.children.length; i < len; i++) {
			this.children[i].destroy();
		}
	},

	/**
	 * Destroys the view, removing all DOM elements generated by render() from the DOM tree and unhooks
	 * any events.
	 */
	destroy: function()
	{
		if (!this.isDestroyed) {
			if (this.fireEvent('beforedestroy', this) !== false) {
				// remove all elements generated with createDiv() from the DOM
				for (var i = 0, len = this.elements.length; i < len; i++) {
					this.elements[i].remove();
				}

				// destroy child views
				this.destroyChildren();

				this.onDestroy();

				this.fireEvent('destroy', this);
				this.purgeListeners();
			}

			this.isDestroyed = true;
		}
	},

	/**
	 * Set the {@link #parentView} field. This will be called by {@link #addChildView}
	 * to update the parent to this view.
	 * @param {Zarafa.core.ui.View} parentView The parent view
	 * @private
	 */
	setParentView: function(parentView)
	{
		this.parentView = parentView;
	},

	/**
	 * Adds a child view to this view.
	 * @param {Zarafa.core.ui.View} child child view to be added
	 */
	addChildView: function(child)
	{
		child.setParentView(this);
		this.children.push(child);
		return child;
	},

	/**
	 * Removes a child view from this view.
	 * @param {Zarafa.core.ui.View} child child view to be added
	 * @param {Boolean} destroy if true, destroy the child view
	 */
	removeChildView: function(child, destroy)
	{
		this.children.remove(child);
		child.setParentView(undefined);
		if (destroy) {
			child.destroy();
		}
	},

	/**
	 * Purge all event listeners.
	 * @private
	 */
	purgeListeners: Ext.Container.prototype.purgeListeners,

	/**
	 * Clear all monitored event listeners. This will call
	 * {@link Ext.util.Observable.un un()} on each previously
	 * registered event handler which was registered with
	 * {@link #mon}.
	 * @private
	 */
	clearMons: Ext.Container.prototype.clearMons,

	/**
	 * Create the tracking object in which we store
	 * each event handler which was registered with {@link #mon}.
	 * @private
	 */
	createMons: Ext.Container.prototype.createMons,

	/**
	 * Adds listeners to any Observable object (or Elements) which are automatically removed when this View
	 * is destroyed.
	 * @param {Observable|Element} item The item to which to add a listener/listeners.
	 * @param {Object|String} ename The event name, or an object containing event name properties.
	 * @param {Function} fn Optional. If the <code>ename</code> parameter was an event name, this
	 * is the handler function.
	 * @param {Object} scope Optional. If the <code>ename</code> parameter was an event name, this
	 * is the scope (<code>this</code> reference) in which the handler function is executed.
	 * @param {Object} opt Optional. If the <code>ename</code> parameter was an event name, this
	 * is the {@link Ext.util.Observable#addListener addListener} options.
	 */
	mon: Ext.Container.prototype.mon,

	/**
	 * Removes listeners that were added by the {@link #mon} method.
	 * @param {Observable|Element} item The item from which to remove a listener/listeners.
	 * @param {Object|String} ename The event name, or an object containing event name properties.
	 * @param {Function} fn Optional. If the <code>ename</code> parameter was an event name, this
	 * is the handler function.
	 * @param {Object} scope Optional. If the <code>ename</code> parameter was an event name, this
	 * is the scope (<code>this</code> reference) in which the handler function is executed.
	 */
	mun: Ext.Container.prototype.mun,

	/**
	 * Called when the View is being destroyed.
	 * @protected
	 */
	onDestroy: Ext.emptyFn
});
Ext.namespace('Zarafa.core.ui');

/**
 * @class Zarafa.core.ui.WelcomeViewport
 * @extends Ext.Viewport
 * The viewport to be used as welcome page for first time users, this will show
 * a welcome message, and allow the user to configure to initial settings
 * before continuing to the {@link Zarafa.core.ui.MainViewport Main viewport}.
 */
Zarafa.core.ui.WelcomeViewport = Ext.extend(Ext.Viewport, {

	/**
	 * The reference as returned by {@link Zarafa.core.ui.notifier.Notifier#notify} to reference the
	 * message in order to remove the message as soon as the save was completed.
	 * @property
	 * @type Ext.Element
	 * @private
	 */
	savingEl: undefined,

	/**
	 * @constructor
	 * @param {Object} config Configuration object
	 */
	constructor: function(config)
	{
		config = config || {};

		config = Ext.applyIf(config, {
			layout: {
				cls: 'grommunio-welcome-viewport',
				type: 'vbox',
				align: 'stretch'
			},
			items: [{
				xtype: 'container',
				flex: 0.5
			},{
				layout: {
					type: 'hbox'
				},
				xtype: 'container',
				items: [{
					xtype: 'container',
					flex: 0.5
				},{
					xtype: 'panel',
					cls: 'zarafa-welcome-body',
					border: false,
					items: [{
						xtype: 'displayfield',
						cls: 'zarafa-welcome-title',
						value : _('Welcome to grommunio Web')
					},{
						xtype: 'displayfield',
						cls: 'zarafa-welcome-message',
						value: _('This is the first time you are using grommunio Web. ') + '<br />' + _('Please check the following settings before continuing.')
					},{
						xtype: 'zarafa.settingswelcomecategory',
						ref: '../../settingsCategory'
					}],
					buttonAlign: 'right',
					buttons: [{
						cls: 'zarafa-action',
						text: _('Continue'),
						handler: this.onContinueButton,
						scope: this
					}]
				},{
					xtype: 'container',
					flex: 0.5
				}]
			},{
				xtype: 'container',
				flex: 0.5
			}]
		});

		Zarafa.core.ui.WelcomeViewport.superclass.constructor.call(this, config);

		this.settingsCategory.update(container.getSettingsModel());

		// Disable autoSave, we want to call the save function manually,
		// so we can supply a callback function.
		container.getSettingsModel().autoSave = false;
	},

	/**
	 * Event handler which is fired when the user clicks the 'Continue' button
	 * This will save all settings, and reload the page to continue to the
	 * {@link Zarafa.core.ui.MainViewport}.
	 * @private
	 */
	onContinueButton: function()
	{
		var model = container.getSettingsModel();

		model.beginEdit();

		// Load settings from UI
		this.settingsCategory.updateSettings(model);

		// Disable the welcome message for next logon
		model.set('zarafa/v1/main/show_welcome', false);

		// set the default keyboard controls to 'basic'
		// We must do this explicitly because there is some logic
		// for backward compatibility that will assume 'disabled'
		// when no keycontrols are set.
		model.set('zarafa/v1/main/keycontrols', 'basic');

		model.endEdit();

		// Register event listener, so we can redirect the user
		// once the save has completed.
		this.mon(model, 'save', this.onSettingsSave, this, { single: true });
		this.mon(model, 'exception', this.onSettingsException, this, { single: true });

		// Show an information box indicating that the settings are being saved.
		this.savingEl = container.getNotifier().notify('info.saving', '', _('Saving') + '...', {
			container: this.getEl(),
			persistent: true
		});

		// Save settings
		model.save();
	},

	/**
	 * Called when the {@link Zarafa.settings.SettingsModel} fires the {@link Zarafa.settings.SettingsModel#save save}
	 * event to indicate the settings were successfully saved.
	 * @param {Zarafa.settings.SettingsModel} model The model which fired the event.
	 * @param {Object} parameters The key-value object containing the action and the corresponding
	 * settings which were saved to the server.
	 * @private
	 */
	onSettingsSave: function(model, parameters)
	{
		container.getNotifier().notify('info.saving', null, null, {
			container: this.getEl(),
			destroy: true,
			reference: this.savingEl
		});

		Zarafa.core.Util.disableLeaveRequester();
		window.location.reload();
	},

	/**
	 * Called when the {@link Zarafa.settings.SettingsModel} fires the {@link Zarafa.settings.SettingsModel#exception exception}
	 * event to indicate the settings were not successfully saved.
	 * @private
	 */
	onSettingsException: function()
	{
		container.getNotifier().notify('info.saving', null, null, {
			container: this.getEl(),
			destroy: true,
			reference: this.savingEl
		});
	}
});
Ext.namespace('Zarafa.core.ui.menu');

/**
 * @class Zarafa.core.ui.menu.ConditionalItem
 * @extends Ext.menu.Item
 * @xtype zarafa.conditionalitem
 *
 * Extends the {@link Ext.menu.Item} class and will check if the
 * item should be enabled or disabled each time before the
 * context menu is being shown.
 */
Zarafa.core.ui.menu.ConditionalItem = Ext.extend(Ext.menu.Item, {
	/**
	 * @cfg {Boolean} hideOnDisabled This item must be hidden rather
	 * then be marked disabled.
	 */
	hideOnDisabled: true,
	/**
	 * @cfg {Boolean} emptySelectOnly This item must only be enabled
	 * if no record is selected
	 */
	emptySelectOnly: false,
	/**
	 * @cfg {Boolean} nonEmptySelectOnly This item must only be enabled
	 * if one or more records are selected
	 */
	nonEmptySelectOnly: false,
	/**
	 * @cfg {Boolean} singleSelectOnly This item must only be enabled
	 * if a single record is selected
	 */
	singleSelectOnly: false,
	/**
	 * @cfg {Boolean} multiSelectOnly This item must only be enabled
	 * if multiple records are selected.
	 */
	multiSelectOnly: false,
	/**
	 * Override of {@link Ext.menu.Item#itemTpl} to add the possibility of
	 * styling the icon.
	 * @property
	 * @Type Ext.XTemplate
	 */
  itemTpl: new Ext.XTemplate(
    '<a id="{id}" class="{cls} x-unselectable" hidefocus="true" unselectable="on" href="{href}"',
      '<tpl if="hrefTarget">',
        ' target="{hrefTarget}"',
      '</tpl>',
     '>',
       '<img alt="{altText}" src="{icon}" class="x-menu-item-icon {iconCls}" {iconStyle}/>',
       '<span class="x-menu-item-text">{text}</span>',
     '</a>'
  ),
  /**
   * Override of {@link Ext.menu.Item#getTemplateArgs} to add the possibility of
   * styling the icon.
   * @return {Object}
   */
  getTemplateArgs: function() {
  	// Get the original template arguments from the original function
  	var templateArgs = Zarafa.core.ui.menu.ConditionalItem.superclass.getTemplateArgs.call(this);
  	// Add the argument for the icon style
  	templateArgs.iconStyle = this.iconBG ? 'style="background-color:'+this.iconBG+';"' : '';
		return templateArgs;
	},
	/**
	 * @constructor
	 * @param {Object} config Configuration object
	 */
	constructor: function(config)
	{
		Ext.applyIf(config, {
			xtype: 'zarafa.conditionalitem'
		});

		Ext.applyIf(this, config);

		Zarafa.core.ui.menu.ConditionalItem.superclass.constructor.call(this, config);
	},

	/**
	 * Initialize the component
	 * @private
	 */
	initComponent: function()
	{
		Zarafa.core.ui.menu.ConditionalItem.superclass.initComponent.apply(this, arguments);

		// Reset the enable/disable functions to their
		// show/hide counterparts if we should not display the item as disabled.
		if (this.hideOnDisabled) {
			this.enable = this.show;
			this.disable = this.hide;
		}
	},

	/**
	 * Apply the {@link #emptySelectOnly}, {@link #nonEmptySelectOnly}, {@link #singleSelectOnly}
	 * and {@link #multiSelectOnly} filters to determine if the item must be {@link #setDisabled disabled}
	 * or not.
	 * @private
	 */
	applySelectionFilter: function()
	{
		var records = this.getRecords();

		if (this.emptySelectOnly) {
			if (Ext.isDefined(records) && (!Array.isArray(records) || records.length > 0)) {
				this.setDisabled(true);
			} else {
				this.setDisabled(false);
			}
		}

		if (this.nonEmptySelectOnly) {
			if (Ext.isDefined(records) && Array.isArray(records) && records.length === 0) {
				this.setDisabled(true);
			} else {
				this.setDisabled(false);
			}
		}

		if (this.singleSelectOnly) {
			if (!Ext.isDefined(records) || (Array.isArray(records) && records.length !== 1)) {
				this.setDisabled(true);
			} else {
				this.setDisabled(false);
			}
		}

		if (this.multiSelectOnly) {
			if (!Ext.isDefined(records) || !Array.isArray(records) || records.length === 1) {
				this.setDisabled(true);
			} else {
				this.setDisabled(false);
			}
		}
	},

	/**
	 * Called by the {@link #parentMenu} when it is about to be shown, this can be
	 * overridden by subclasses to add extra filters on the visibility of this item.
	 * @param {Zarafa.core.ui.menu.ConditionalItem} item The item which is about to be shown.
	 * @param {Zarafa.core.data.MAPIRecord} record The record which is being shown for this menu
	 */
	beforeShow: function(item, record)
	{
		item.applySelectionFilter();
	},

	/**
	 * Obtain the record/records which are attached to the {@link Zarafa.core.ui.menu.ConditionalMenu menu}
	 * to which this item belongs.
	 * @return {Zarafa.core.data.MAPIRecord[]} records
	 */
	getRecords: function()
	{
		return this.getRootMenu().records;
	},

	/**
	 * Obtain the reference to the root menu.
	 * This will go through all parents of the {@link Zarafa.core.ui.ConditionalItem item}
	 * until no {@link Ext.menu.menu} is found as parent. The last parent is considered the
	 * root and will be returned to the caller.
	 *
	 * @return {Ext.menu.menu} The Root menu object
	 */
	getRootMenu: function()
	{
		var menu = this.parentMenu;
		if (!menu && this.ownerCt instanceof Ext.menu.Menu) {
			menu = this.ownerCt;
		}

		while (menu && (Ext.isDefined(menu.parentMenu) || menu.ownerCt instanceof Ext.menu.Menu)) {
			menu = menu.parentMenu || menu.ownerCt;
		}

		return menu;
	}
});

Ext.reg('zarafa.conditionalitem', Zarafa.core.ui.menu.ConditionalItem);
Ext.namespace('Zarafa.core.ui.menu');

/**
 * @class Zarafa.core.ui.menu.ConditionalMenu
 * @extends Ext.menu.Menu
 * @xtype zarafa.conditionalmenu
 *
 * Extends the {@link Ext.menu.Menu} class and allows menu options to determine whether to display themselves.
 */
Zarafa.core.ui.menu.ConditionalMenu = Ext.extend(Ext.menu.Menu, {
	/**
	 * @cfg {Zarafa.core.data.IPMRecord[]} records The records on which this contextmenu was requested
	 */
	records: undefined,

	/**
	 * @constructor
	 * @param {Object} config Configuration object
	 */
	constructor: function(config)
	{
		config = config || {};

		Ext.applyIf(config, {
			xtype: 'zarafa.conditionalmenu',
			shadow: false
		});

		Zarafa.core.ui.menu.ConditionalMenu.superclass.constructor.call(this, config);

		this.on('beforeshow', this.onMenuBeforeShow, this);
	},

	/**
	 * Event handler for the {@link #beforeshow} event. This will go through all
	 * {@link Zarafa.core.ui.menu.ConditionalItem items} in the menu and call the
	 * {@link Zarafa.core.ui.menu.ConditionalItem#beforeShow} function.
	 *
	 * Optionally all duplicate menuseparators will be removed. This can happen when an
	 * item which was surrounded by separators has been hidden.
	 *
	 * @param {Zarafa.core.ui.menu.ConditionalMenu} menu The menu which is being opened.
	 * @private
	 */
	onMenuBeforeShow: function(menu)
	{
		var records = this.records;
		var allowSeparator = false;
		var lastItemIndex = -1;

		// move over the items list and call 'beforeOpen' on each item if that function exists
		menu.items.each(function(item, index) {
			if (item.isXType('zarafa.conditionalitem')) {
				if (Ext.isFunction(item.beforeShow)) {
					item.beforeShow.call(item.scope || item, item, records);
				}

				if (Ext.isDefined(item.menu)) {
					this.onMenuBeforeShow(item.menu);
				}
			}

			// A separator is being added, check if we actually want to display it.
			if (item.isXType('menuseparator')) {
				item.setVisible(allowSeparator === true);
				allowSeparator = false;
			} else {
				// If the non-separator item is visible,
				// we are allowed to display a separator when requested.
				if (item.hidden === false) {
					allowSeparator = true;
					lastItemIndex = index;
				}
			}
		}, this);

		// The menu is empty, or we have hidden everything.
		if (lastItemIndex === -1) {
			return false;
		}

		// Remove all separators which are visible as last items in the menu.
		for (var i = lastItemIndex, len = menu.items.getCount(); i < len; i++) {
			var item = menu.items.items[i];
			if (item.isXType('menuseparator')) {
				item.setVisible(false);
			}
		}

		// But what if we just removed everything?
		if (lastItemIndex === -1) {
			return false;
		}
	}
});

Ext.reg('zarafa.conditionalmenu', Zarafa.core.ui.menu.ConditionalMenu);
Ext.namespace('Zarafa.core.ui.notifier');

/**
 * @class Zarafa.core.ui.notifier.Notifier
 * @extends Object
 */
Zarafa.core.ui.notifier.Notifier = Ext.extend(Object, {
	/**
	 * The key-value list of all available plugins which
	 * might be used to send notify messages to.
	 * @property
	 * @type Object
	 */
	availablePlugins: undefined,

	/**
	 * @constructor
	 * @param {Object} config Configuration object
	 */
	constructor: function(config)
	{
		Ext.apply(this, config);

		this.availablePlugins = {};
	},

	/**
	 * Register the {@link Zarafa.core.ui.notifier.NotifyPlugin NotifyPlugin} to the
	 * Notifier instance. This will add the plugin for the given name on the
	 * {@link #availablePlugins} array. It will also make the name available
	 * to be set in the {@link Zarafa.core.SettingsManager Settings}.
	 *
	 * @param {String} name The name how the plugin must be registered.
	 * @param {Zarafa.core.ui.notifier.NotifyPlugin} plugin The plugin which must be registered.
	 */
	registerPlugin: function(name, plugin)
	{
		if (!Ext.isDefined(this.availablePlugins[name])) {
			this.availablePlugins[name] = plugin;
		}
	},

	/**
	 * Obtain the {@link Zarafa.core.ui.notifier.NotifyPlugin plugin} which should be used
	 * for Notifications for the given category.
	 *
	 * The category should be either "error", "warning", "info" or "debug", or a subtype thereof (e.g. "info.newmail"). When
	 * searching for a valid plugin, the {@link Zarafa.core.SettingsManager Settings} is consulted to find the
	 * correct pluginname. If no pluginname is found for the given category, a supertype will be tried. For example,
	 * if no pluginname is found for "info.alfresco", then it will try to find the pluginname for "info" instead.
	 * If no pluginname can be found, then 'default' will be used as categoryname. Finally, if that also has no
	 * plugin registered, then this function will return 'undefined'.
	 *
	 * @param {String} category The category for which the plugin is searched for.
	 * @return {Zarafa.core.ui.notifier.NotifyPlugin} The plugin.
	 */
	getPlugin: function(category)
	{
		var setting = 'zarafa/v1/main/notifier/' + category.replace(/\./g, '/') + (category != 'default' ? '/value' : '');
		var pluginName = container.getSettingsModel().get(setting);
		var plugin = this.availablePlugins[pluginName];

		if (!plugin && category != 'default') {
			var index = category.lastIndexOf('.');
			if (index > 0) {
				category = category.substr(0, index);
			} else {
				category = 'default';
			}

			plugin = this.getPlugin(category);
		}

		return plugin;
	},

	/**
	 * Send a notification to the user. This {@link #getPlugin requests} the plugin for the provided category.
	 * On the plugin the {@link Zarafa.core.ui.notifier.NotifyPlugin#notify notify} function is called to display
	 * the message to the user. If no plugin could be found for
	 * the given category, then no message will be shown to the user.
	 *
	 * @param {String} category The category which applies to the notification.
	 * @param {String} title The title which must be shown in the message.
	 * @param {String} message The message which should be displayed.
	 * @param {Object} config Configuration object which can be applied to the notifier
	 * This object can contain keys like:
	 * - container: Which is the container to which the notifier should be restricted
	 * - persistent: True to make the message persistent and don't disappear automatically
	 * - destroy: Don't create new message, but destroy previous one
	 * - update: Don't create a new message, but update previous one
	 * - reference: The original message which must be updated by this action
	 * - listeners: Event handlers which must be registered on the element
	 * @return {Mixed} A reference to the message which was created, this can be used
	 * as value for 'reference' in the config argument.
	 */
	notify: function(category, title, message, config)
	{
		var categoryName = category.toLowerCase();
		var plugin = this.getPlugin(categoryName);
		if (plugin) {
			return plugin.notify(categoryName, title, message, config);
		}
	}
});
Ext.namespace('Zarafa.core.ui.notifier');

/**
 * @class Zarafa.core.ui.notifier.NotifyPlugin
 * @extends Object
 *
 * Base class for any plugins which can be registered to the
 * {@link Zarafa.core.ui.notifier.Notifier Notifier}.
 */
Zarafa.core.ui.notifier.NotifyPlugin = Ext.extend(Object, {
	/**
	 * @constructor
	 * @param {Object} config Configuration object
	 */
	constructor: function(config)
	{
		Ext.apply(this, config);
	},

	/**
	 * Notify the user with a message.
	 *
	 * The category can be either "error", "warning", "info" or "debug", or a subtype thereof (e.g. "info.newmail").
	 *
	 * @param {String} category The category which applies to the notification.
	 * @param {String} title The title which must be shown in the message.
	 * @param {String} message The message which should be displayed.
	 * @param {Object} config Configuration object which can be applied to the notifier
	 * This object can contain keys like:
	 * - container: Which is the container to which the notifier should be restricted
	 * - destroy: Don't create new message, but destroy previous one
	 * - reference: The original message which must be updated by this action
	 * @return {Mixed} A reference to the message which was created, this can be used
	 * as value for 'reference' in the config argument.
	 */
	notify: Ext.emptyFn
});
Ext.namespace('Zarafa.core.ui.notifier');

/**
 * @class Zarafa.core.ui.notifier.SliderContainer
 * @extends Object
 *
 * Special container to be used by the {@link Zarafa.core.ui.notifier.SliderNotifyPlugin}.
 * This supports holding multiple notification messages within a single DIV element,
 * and slide new ones into the container, and old ones out. The code is structured so only
 * one animation occurs at a time, and the container can be positioned anywhere on the screen.
 */
Zarafa.core.ui.notifier.SliderContainer = Ext.extend(Object, {
	/**
	 * @cfg {Ext.Element} parentContainer The parent container in which the
	 * slider will be positioned.
	 */
	parentContainer: undefined,

	/**
	 * @cfg {String} containerCls The CSS class to be applied on the {@link #container}.
	 */
	containerCls: 'zarafa-notifier-container',

	/**
	 * @cfg {String} itemCls The CSS class to be applied to the DIV wrapper around
	 * the messages added in {@link #createMessage}.
	 */
	itemCls: 'zarafa-notifier-container-item',

	/**
	 * @cfg {String} containerPosition The position of the container
	 * in which the notifier will be shown. This can be any of the following values:
	 * - 'tl': The top left corner
	 * - 't': The center of the top edge (default)
	 * - 'tr': The top right corner
	 * - 'l': The center of the left edge
	 * - 'c': In the center of the element
	 * - 'r': The center of the right edge
	 * - 'bl': The bottom left corner
	 * - 'b': The center of the bottom edge
	 * - 'br': The bottom right corner
	 */
	containerPosition: 't',

	/**
	 * @cfg {Char} slideinDirection the animation slidein direction for notification
	 */
	slideinDirection: 't',

	/**
	 * @cfg {Char} slideoutDirection the animation slideout direction for notification
	 */
	slideoutDirection: 't',

	/**
	 * @cfg {Boolean} animatedContainerResize True to animate the resizing of the container
	 * when a new notification is added into the container. This defaults to 'false' if
	 * {@link #slideinDirection} is 't', 'true' otherwise.
	 */
	animatedContainerResize: false,

	/**
	 * The container in which the sliders will be placed.
	 * @property
	 * @type Ext.Element
	 * @private
	 */
	container: undefined,

	/**
	 * The array of {@link Ext.Element Elements} which are pending to be displayed in the {@link #container}.
	 * Because we cannot animate two messages simultaneously, we keep the list of pending messages here, which
	 * serves as FIFO queue. Whenever a message has been animated, this list will be checked to see if a message
	 * is in the queue to be animated.
	 * @property
	 * @type Array
	 * @private
	 */
	pendingMessages: undefined,

	/**
	 * The array of {@link Ext.Element Elements} which are currently displayed in the {@link #container}.
	 * @property
	 * @type Array
	 * @private
	 */
	stackedMessages: undefined,

	/**
	 * Used by {@link #showNextMessage} to check if there currently is a Message which is being animated.
	 * When this field is set to 'true', then no other message is allowed to start animating.
	 * @property
	 * @type Array
	 * @private
	 */
	animating: false,

	/**
	 * @constructor
	 * @param {Object} config Configuration object
	 */
	constructor: function(config)
	{
		Ext.apply(this, config);

		this.container = this.getContainer();
		this.pendingMessages = [];
		this.stackedMessages = [];

		// update animatedContainerResize depending on configuration
		if (!Ext.isDefined(config.animatedContainerResize)) {
			this.animatedContainerResize = this.slideinDirection !== 't';
		}
	},

	/**
	 * Add a new slider to the container. This will animate the slider into the {@link #container}.
	 * @param {String} html The HTML string for the slider to create
	 * @param {Number} timeout (optional) If provided, the message will be removed automatically after the timeout
	 * @return {Ext.Element} The slider element which was created
	 */
	createMessage: function(html, timeout)
	{
		// Construct the Ext.Element object
		var element = Ext.DomHelper.append(this.container, { html: html, cls: this.itemCls, style: 'visibility: hidden;' }, true);

		// Push it to pending messages
		this.pendingMessages.push(element);
		element.timeout = timeout;
		element.slider = this;

		// Check if there are no pending messages, and we can direct display this new one.
		this.showNextMessage();

		return element;
	},

	/**
	 * Update a message which was previously created using {@link #createMessage}.
	 * @param {Ext.Element} The element to update
	 * @param {String} html The HTML string for the slider to update
	 * @return {Ext.Element} The slider element which was updated
	 * @private
	 */
	updateMessage: function(element, html)
	{
		element.dom.innerHTML = html;
		return element;
	},

	/**
	 * Remove a message which was previously created using {@link #createMessage}.
	 * @param {Ext.Element} element The element to remove
	 */
	removeMessage: function(element)
	{
		this.animating = true;

		if (this.stackedMessages.indexOf(element) >= 0) {
			element.ghost(this.slideoutDirection, {
				remove: true,
				stopFx: true,
				callback: this.onRemoveComplete.createDelegate(this, [ element ]),
				scope: this
			});
		}
	},

	/**
	 * Called whenever a message has been added to the {@link #pendingMessages} queue, or when
	 * a message has been removed from the {@link #stackedMessages}. When we still have pending
	 * messages in {@link #pendingMessages}, we first check if we are currently busy
	 * {@link #animating animating} a message, or if there is no room for a new message to appear.
	 * When neither problem occurs, then we start animating the new message into the screen.
	 *
	 * @private
	 */
	showNextMessage: function()
	{
		var element = this.pendingMessages[0];
		var msgHeight = element.getHeight();
		var ctHeight = this.container.getHeight();
		var maxHeight = this.parentContainer.getHeight();

		// The <body> element might sometimes have height 0, compensate this
		// by asking for the document height in that case.
		if (maxHeight === 0 && this.parentContainer.dom.tagName === 'BODY') {
			maxHeight = document.height;
		}

		if (this.animating || (ctHeight + msgHeight) > maxHeight) {
			return;
		}
		this.animating = true;

		// Update the dimensions of the container to make room for the new element.
		// All existing messages will be nicely shifted the require height higher.
		var newHeight = ctHeight + msgHeight;
		this.updateContainer(newHeight, this.animatedContainerResize);

		// Store the original height of the message for later,
		// we need it for onMessageSlideOutComplete() when the real height has been reset,
		// and we need to resize the container.
		element.height = msgHeight;

		// Move the element to the next queue
		this.pendingMessages.remove(element);
		this.stackedMessages.push(element);

		this.animMessage(element);
	},

	/**
	 * Start the animation which will {@link Ext.Fx#slideIn slidein} the given element.
	 * If the slider was configured with a timeout, it will {@link Ext.Fx#pause show} the
	 * element for the given time and then {@link Ext.Fx#ghost remove} it again.
	 * @param {Ext.Element} element The element to show
	 * @private
	 */
	animMessage: function(element)
	{
		element = element.slideIn(this.slideinDirection, {
			callback: this.onShowComplete,
			scope: this
		});

		// Check if this element must automatically be removed
		// after a certain timeout.
		if (Ext.isNumber(element.timeout)) {
			element.pause(element.timeout, {
				callback: this.onPauseComplete,
				scope: this
			}).ghost(this.slideoutDirection, {
				remove: true,
				callback: this.onRemoveComplete.createDelegate(this, [ element ]),
				scope: this
			});
		}
	},

	/**
	 * Event handler which is called when the {@link Ext.Fx#slideIn slideIn} animation has been
	 * completed. This will reset the {@link #animating} flag, and call {@link #showNextMessage}
	 * to start animating any pending messages.
	 *
	 * @private
	 */
	onShowComplete: function()
	{
		this.animating = false;

		// Show the next message if there are pending messages,
		// and the parent container still exists (it might have been
		// deleted while the element was animating).
		if (!Ext.isEmpty(this.pendingMessages) && this.parentContainer.dom) {
			this.showNextMessage();
		}
	},

	/**
	 * Event handler which is called when the {@link Ext.Fx#pause pause} has been completed.
	 * This will set the {@link #animating} field again to prepare for the {@link Ext.Fx#ghost ghost}
	 * animation.
	 *
	 * @private
	 */
	onPauseComplete: function()
	{
		this.animating = true;
	},

	/**
	 * Event handler which is called when the {@link Ext.Fx#ghost ghost} animation has been
	 * completed. This will resize the {@link #container} to make sure it will only
	 * contain enough room for the visible {@link #stackedMessages messages}.
	 * @param {Ext.Element} element The element which was removed
	 * @private
	 */
	onRemoveComplete: function(element)
	{
		var newHeight = this.container.getHeight() - element.height;
		var index = this.stackedMessages.indexOf(element);

		// Technically shouldn't happen...
		if (index < 0) {
			return;
		}

		// Remove the element from stack
		this.stackedMessages.splice(index, 1);

		this.animating = false;

		if (newHeight < 0) {
			newHeight = 0;
		}

		// After cleaning up, we need to check if the parent container
		// still exists, as it might have been deleted while the element
		// was animating.
		if (!this.parentContainer.dom) {
			return;
		}

		// Don't animate the repositioning, otherwise all visible
		// messages will be bumped above and then animated down again.
		// By not animating, the messages will remain exactly where they
		// are.
		this.updateContainer(newHeight, false);

		if (!Ext.isEmpty(this.pendingMessages)) {
			this.showNextMessage();
		}
	},

	/**
	 * Obtain the {@link Ext.Element container} in which this slider will operate.
	 * This uses the {@link #parentContainer} to find if any slider container already
	 * exists, if not, it will create one at the {@link #containerPosition}.
	 * @return {Ext.Element} The slider container
	 * @private
	 */
	getContainer: function()
	{
		var parentId = this.parentContainer.id;
		var sliderCls = this.containerCls + ' ' + this.containerCls + '-' + this.containerPosition;
		var sliderId = parentId + '-' + this.containerCls + '-' + this.containerPosition;

		var ct = Ext.DomQuery.select('#' + sliderId).shift();
		if (!ct) {
			// Insert the notifier message into the given container
			ct = Ext.DomHelper.insertFirst(this.parentContainer, { id: sliderId, cls: sliderCls }, true);

			// Set initial height to 0
			ct.setHeight(0);

			// Position the container to the correct location
			ct.alignTo(this.parentContainer, this.containerPosition);
		} else {
			ct = Ext.get(ct);
		}

		return ct;
	},

	/**
	 * Update the {@link #container} with a new height. This will recalculate the
	 * position of the {@link #container} to make sure the container remains fixed
	 * to the {@link #containerPosition.
	 *
	 * @param {Number} height The new height which must be applied to the container
	 * @param {Boolean} animate True when the resize/repositioning of the container
	 * should be animated.
	 * @private
	 */
	updateContainer: function(height, animate)
	{
		// The position of the element should be: br-br, t-t, etc, to ensure
		// that the bottom-right corner of the container is in the bottom-right
		// corner of the parentContainer.
		var position = this.containerPosition + '-' + this.containerPosition;

		// Update positioning of the container
		this.container.setHeight(height);
		this.container.alignTo(this.parentContainer, position, undefined, animate);
	}
});
Ext.namespace('Zarafa.core.ui.widget');

/**
 * @class Zarafa.core.ui.widget.Widget
 * @extends Ext.ux.Portlet
 * @xtype zarafa.widget
 *
 * A 'widget' is a plug-in that users can instantiate and put on their today view or tool bar.
 * Examples of widgets are the clock, the weather widget, public notes, etc.
 *
 * Users can add new instances of a widget freely, and create multiple instances if desired. Each
 * widget instance has a unique GUID, and settings for that instance are stored in the widgets/[GUID]
 * settings folder. When a widget is destroyed, that folder is deleted.
 */
Zarafa.core.ui.widget.Widget = Ext.extend(Ext.ux.Portlet, {
	/**
	 * @cfg {Zarafa.core.WidgetInfo} info The Widget Meta Data object
	 * which was used to {@link Zarafa.core.Container#registerWidget register}
	 * this Widget to the {@link Zarafa.core.Container container}.
	 */
	info: undefined,

	/**
	 * @cfg {String} guid The unique identifier used for reference to this particular
	 * widget in side a {@link Zarafa.core.ui.widget.WidgetPanel}. This is used for
	 * {@link #set setting} and {@link #get getting} {@link Zarafa.settings.SettingsModel settings}.
	 */
	guid: undefined,

	/**
	 * @cfg {Boolean} hasConfig True if the {@link #config} function
	 * has been implemented and a 'gear' icon should be shown besides
	 * the close button.
	 */
	hasConfig: false,

	/**
	 * @cfg {String} about The about text. If provided, {@link #registerAboutText}
	 * will be automatically called during {@link #initWidget initialization}.
	 */
	about: undefined,

	/**
	 * The widget panel on which this widget is located
	 * @property
	 * @type Zarafa.core.ui.widget.WidgetPanel
	 */
	widgetPanel: undefined,

	/**
	 * {@link Ext.QuickTips tooltip} message for the widget's collapse {@link Ext.Button button}
	 * @property
	 * @type String
	 */
	collapseQuickTip: undefined,

	/**
	 * {@link Ext.QuickTips tooltip} message for the widget's expand {@link Ext.Button button}
	 * @property
	 * @type String
	 */
	expandQuickTip: undefined,

	/**
	 * @constructor
	 * @param {Object} config Configuration object
	 */
	constructor: function(config)
	{
		config = config || {};

		var tools = config.tools || [];

		// Check if the configuration function has been
		// implemented. If it is the case, create the config tool.
		if (config.hasConfig === true) {
			tools.push({
				id: 'gear',
				qtip: _('Configure widget'),
				handler: this.config,
				scope: this
			});
		}

		// Always add the close tool.
		tools.push({
			id: 'close',
			qtip: _('Remove widget'),
			scope: this,
			handler: this.close
		});

		Ext.applyIf(config, {
			title: config.info.getDisplayName(),
			anchor: '100%',
			frame: true,
			collapsible: true,
			collapseQuickTip: _('Collapse widget'),
			expandQuickTip: _('Expand widget'),
			draggable: {
				ddGroup: 'dd.widget'
			},
			tools: tools
		});

		Zarafa.core.ui.widget.Widget.superclass.constructor.call(this, config);

		this.initWidget();
	},

	/**
	 * Called during rendering of the panel, this will initialize all events.
	 * @private
	 */
	initEvents: function ()
	{
		Zarafa.core.ui.widget.Widget.superclass.initEvents.call(this);
		this.on('afterlayout', this.setCollapseQuickTip, this, {single: true});
		this.on('expand', this.setCollapseQuickTip, this);
		this.on('collapse', this.setExpandQuickTip, this);
	},

	/**
	 * Event handler which is called after {@link Zarafa.core.ui.MainViewSidebar panel} get layout
	 * This will set {@link Ext.QuickTips} on {@link Ext.Button collapse} button
	 * @private
	 */
	setCollapseQuickTip: function ()
	{
		this.tools['toggle'].dom.qtip = this.collapseQuickTip;
	},

	/**
	 * Event handler which is called before {@link Zarafa.core.ui.MainViewSidebar panel} collapse
	 * This will set {@link Ext.QuickTips} on {@link Ext.Button expand} button
	 * @private
	 */
	setExpandQuickTip: function ()
	{
		this.tools['toggle'].dom.qtip = this.expandQuickTip;
	},

	/**
	 * Function to be implemented by Widget subclasses to initialize the widget.
	 * @protected
	 */
	initWidget: function()
	{
		// If the about text is provided, automatically register it
		if (!Ext.isEmpty(this.about)) {
			this.registerAboutText(this.title, this.about);
		}
	},

	/**
	 * Add a About/Copyright notice to the Widget.
	 * @param {String} title The title for the About widget
	 * @param {String} text The text which should be shown in the About text (may contain HTML)
	 * @protected
	 */
	registerAboutText: function(title, text)
	{
		this.tools.splice(this.tools.length - 1, 0, {
			id: 'help',
			qtip: _('About this widget'),
			handler: this.showAbout,
			scope: this,
			title: title,
			text: text
		});
	},

	/**
	 * Checks if this widget is currently visible. This returns true when
	 * the widget itself is {@link #isVisible visible} and the {@link #widgetPanel}
	 * is currently not {@link Ext.Panel#collapsed}.
	 */
	isWidgetVisible: function()
	{
		return this.isVisible() && !this.widgetPanel.collapsed;
	},

	/**
	 * Called when the widget has been rendered.
	 * This will initialize the {@link #widgetPanel}.
	 */
	onRender: function()
	{
		Zarafa.core.ui.widget.Widget.superclass.onRender.apply(this, arguments);
		this.widgetPanel = this.findParentByType('zarafa.widgetpanel');
	},

	/**
	 * Get a settings property..
	 * @param {String} key settings path. This is path relative to where the widget's settings are stored.
	 * @return {String} value.
	 */
	get: function(key)
	{
		return container.getSettingsModel().get(Zarafa.core.ui.widget.Widget.settingsPath(this.guid, key));
	},

	/**
	 * Set a settings property.
	 * @param {String} key settings path. This is path relative to where the widget's settings are stored.
	 * @param {String} value value.
	 */
	set: function(key, value)
	{
		container.getSettingsModel().set(Zarafa.core.ui.widget.Widget.settingsPath(this.guid, key), value);
	},

	/**
	 * Closes and destroys the widget, removing its settings from the settings tree.
	 */
	close: function(e, target, panel)
	{
		this.widgetPanel.destroyWidget(this);
	},

	/**
	 * Called when a user clicks the config button on the widget panel.
	 * Should be overridden by child classes.
	 * @protected
	 */
	config: Ext.emptyFn,

	/**
	 * Called when a user clicks the about button on the widget panel.
	 * This will show a {@link Ext.Window} containing the {@link #about} text.
	 * @protected
	 */
	showAbout: function(event, toolEl, panel, tc)
	{
		var win = new Ext.Window({
			title: tc.title,
			width: 320,
			height: 200,
			padding: 5,
			autoScroll: true,
			items: [{
				xtype: 'panel',
				layout: 'form',
				border: false,
				items: [{
					xtype: 'displayfield',
					value: tc.text,
					hideLabel: true,
					htmlEncode: false
				}]
			}]
		});

		win.show(this);
	}
});

Ext.reg('zarafa.widget', Zarafa.core.ui.widget.Widget);

/**
 * @param {String} guid The unique identifier for a {@link Zarafa.core.ui.widget.Widget widget}.
 * @param {String} key The setting key which should be updated
 * @return {String} The settings path for the guid & key combination.
 * @static
 */
Zarafa.core.ui.widget.Widget.settingsPath = function(guid, key)
{
	return ('zarafa/v1/widgets/' + guid + '/' + key).replace(/(\/)+/,'/');
};
Ext.namespace('Zarafa.core.ui.widget');

/**
 * @class Zarafa.core.ui.widget.WidgetMetaData
 * @extends Object
 *
 * The Meta Data object containing the registration details
 * of a {@link Zarafa.core.ui.widget.Widget}. An instance of this object
 * must be passed to {@link Zarafa.core.Container#registerWidget}.
 */
Zarafa.core.ui.widget.WidgetMetaData = Ext.extend(Object, {
	/**
	 * @cfg {String} name (required) The unique name for this widget.
	 * For a user-friendly name for UI components, see {@link #displayName}
	 */
	name: '',

	/**
	 * @cfg {String} displayName The display name for this widget. This
	 * will be used in places where the widget is referenced in UI components.
	 * If not provided, {@link #name} will be used.
	 */
	displayName: '',

	/**
	 * @cfg {String} iconCls The icon to be used in places where the widget is referenced
	 * in UI components.
	 */
	iconCls: '',

	/**
	 * @cfg {String} iconPath (deprecated) The path to the icon to be used in places where the widget is referenced
	 * in UI components. This configuration option is deprecated, it is preferred to use {@link #iconCls} instead.
	 */
	iconPath: '',

	/**
	 * @cfg {String} about The about text. If provided, {@link Zarafa.core.ui.widget.Widget#registerAboutText}
	 * will be automatically called during {@link Zarafa.core.ui.widget.Widget#initWidget initialization}.
	 */
	about: undefined,

	/**
	 * @cfg {Constructor} WidgetConstructor (required) The constructor of the
	 * {@link Zarafa.core.ui.widgetWidget} which is described by this WidgetMetaData instance.
	 */
	widgetConstructor: undefined,

	/**
	 * @constructor
	 * @param {Object} config Configuration object
	 */
	constructor: function(config)
	{
		config = config || {};

		Ext.apply(this, config);

		Zarafa.core.ui.widget.WidgetMetaData.superclass.constructor.call(this, config);

		// Initialize displayName if not initialized from config
		if (Ext.isEmpty(this.displayName)) {
			this.displayName = this.name;
		}
	},

	/**
	 * Obtain the unique name for this widget
	 * @return {String} The unique name for this widget
	 */
	getName: function()
	{
		return this.name;
	},

	/**
	 * Obtain the display name for this widget
	 * @return {String} The display name for this widget
	 */
	getDisplayName: function()
	{
		return this.displayName;
	},

	/**
	 * Obtain the CSS classname for this widget
	 * @return {String} The CSS classname for this widget
	 */
	getIconCls: function()
	{
		return this.iconCls;
	},

	/**
	 * Obtain the path to the image for this widget
	 * @return {String} The path to the icon for this widget
	 */
	getIconPath: function()
	{
		return this.iconPath;
	},

	/**
	 * Obtain the About text containing the copyright and other disclaimers.
	 * @return {String} The about text for this widget
	 */
	getAbout: function()
	{
		return this.about;
	},

	/**
	 * Obtain the instance of the {@link Zarafa.core.ui.widget.Widget} which is instantiated
	 * using the {@link #widgetConstructor}.
	 * @param {Object} config The configuration object to apply
	 * @return {Zarafa.core.ui.widget.Widget} The Widget instance
	 */
	getInstance: function(config)
	{
		return new this.widgetConstructor(Ext.apply({ info: this }, config));
	}
});
Ext.namespace('Zarafa');

/**
 * @class Zarafa.ABOUT
 * @extends String
 *
 * The copyright string holding the copyright notice for grommunio Web.
 */

Zarafa.ABOUT = ""
	+ "<style>div.atxt h1{font-weight:inherit;font-style:italic;}</style><div class=\"atxt\">"
	+ "<p>Copyright (C) 2020 - 2025 grommunio GmbH</p>"

	+ "<p>This program is free software: you can redistribute it and/or modify "
	+ "it under the terms of the GNU Affero General Public License as "
	+ "published by the Free Software Foundation, either version 3 of the "
	+ "License, or (at your option) any later version.</p>"

	+ "<p>This program is distributed in the hope that it will be useful, "
	+ "but WITHOUT ANY WARRANTY; without even the implied warranty of "
	+ "MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the "
	+ "GNU Affero General Public License for more details.</p>"

	+ "<p>You should have received a copy of the GNU Affero General Public License "
	+ "along with this program. If not, see <a href=\"https://www.gnu.org/licenses/\" target=\"_blank\">https://www.gnu.org/licenses/</a>.</p>"

	+ "<hr />"

	+ "<p>grommunio Web contains the following components:</p>"

	+ "<h1>grommunio Web</h1>"

	+ "<p>Copyright (C) 2020 - 2025 grommunio GmbH<br>"
	+ "Copyright (C) 2016 Kopano and its licensors<br>"
	+ "Copyright (C) 2005 - 2016 Zarafa B.V. and its licensors</p>"

	+ "<h1>grommunio Web File Previewer Plugin</h1>"

	+ "<p>Copyright (C) 2020 - 2025 grommunio GmbH<br>"
	+ "Copyright (C) 2016 Kopano and its licensors<br>"
	+ "Copyright (C) 2016 Zarafa B.V. and its licensors</p>"

	+ "<h1>grommunio Web Desktop Notifications Plugin</h1>"

	+ "<p>Copyright (C) 2020 - 2025 grommunio GmbH<br>"
	+ "Copyright (C) 2016 Kopano and its licensors<br>"
	+ "Copyright (C) 2013 Saket Patel</p>"

	+ "<h1>grommunio Web Meet Plugin</h1>"

	+ "<p>Copyright (C) 2020 - 2025 grommunio GmbH<br>"
	+ "Copyright (C) 2020 Siedl Networks<br>"
	+ "Copyright (C) 2017 Kopano and its licensors</p>"

	+ "<h1>grommunio Web Password Plugin</h1>"

	+ "<p>Copyright (C) 2020 - 2025 grommunio GmbH<br>"
	+ "Copyright (C) 2017 Kopano and its licensors<br>"
	+ "Copyright (C) 2013 - 2017 Saket Patel</p>"

	+ "<h1>grommunio Web Device Management Plugin</h1>"

	+ "<p>Copyright (C) 2020 - 2025 grommunio GmbH<br>"
	+ "Copyright (C) 2016 Kopano and its licensors<br>"
	+ "Copyright (C) 2005 - 2016 Zarafa B.V. and its licensors</p>"

	+ "<h1>grommunio Web Chat Plugin</h1>"

	+ "<p>Copyright (C) 2020 - 2025 grommunio GmbH<br>"
	+ "Copyright (C) 2017 Kopano and its licensors</p>"

	+ "<h1>grommunio Web Maps Plugin</h1>"

	+ "<p>Copyright (C) 2020 - 2025 grommunio GmbH<br>"
	+ "Copyright (C) 2016 Kopano and its licensors<br>"
	+ "Copyright (C) 2005 - 2016 Zarafa B.V. and its licensors</p>"

	+ "<h1>grommunio Web Files & Backend Plugin</h1>"

	+ "<p>Copyright (C) 2020 - 2025 grommunio GmbH<br>"
	+ "Copyright (C) 2016 Kopano and its licensors<br>"
	+ "Copyright (C) 2016 Zarafa B.V. and its licensors</p>"

	+ "<h1>grommunio Web Intranet Plugin</h1>"

	+ "<p>Copyright (C) 2020 - 2025 grommunio GmbH<br>"
	+ "Copyright (C) 2017 Kopano and its licensors</p>"

	+ "<h1>grommunio Web Kendox Plugin</h1>"

	+ "<p>Copyright (C) 2020 - 2025 grommunio GmbH<br>"
	+ "Copyright (C) 2021 Kendox and its licensors</br>"
        + "Copyright (C) 2021 NDG IT-Systeme GmbH</p>"

	+ "<p>This program is free software: you can redistribute it and/or modify "
	+ "it under the terms of the GNU Affero General Public License as "
	+ "published by the Free Software Foundation, either version 3 of the "
	+ "License, or (at your option) any later version.</p>"

	+ "<p>This program is distributed in the hope that it will be useful, "
	+ "but WITHOUT ANY WARRANTY; without even the implied warranty of "
	+ "MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the "
	+ "GNU Affero General Public License for more details.</p>"

	+ "<p>You should have received a copy of the GNU Affero General Public License "
	+ "along with this program. "
	+ "If not, see <a href=\"https://www.gnu.org/licenses/\" target=\"_blank\">https://www.gnu.org/licenses/</a>.</p>"


	+ "<h1>Ext JS</h1>"

	+ "<p>Copyright (C) 2011-2013 Sencha Inc</p>"

	+ "<p>Ext JS is free software: you can redistribute it and/or modify "
	+ "it under the terms of the GNU General Public License version 3.0 as "
	+ "published by the Free Software Foundation."
	+ "<a href=\"https://www.gnu.org/copyleft/gpl.html\" target=\"_blank\">https://www.gnu.org/copyleft/gpl.html</a></p>"


	+ "<h1>Ext.ux.form.Metaform</h1>"

	+ "<p>Copyright (C) 2008 Ing. Jozef Sakáloša.</p>"

	+ "<p>Ext.ux.form.MetaForm is licensed under the terms of the Open Source LGPL 3.0 "
	+ "license. Commercial use is permitted to the extent that the code/component(s)"
	+ "do NOT become part of another Open Source or Commercially licensed development"
	+ "library or toolkit without explicit permission."
	+ "<a href=\"https://www.gnu.org/copyleft/lgpl.html\" target=\"_blank\">https://www.gnu.org/copyleft/lgpl.html</a></p>"


	+ "<h1>Phpfastcache</h1>"

	+ "<p>Copyright (c) 2023 Phpfastcache & its Extensions</p>"

	+ "<p>Permission is hereby granted, free of charge, to any person obtaining a copy of "
	+ "this software and associated documentation files (the Software), to deal in the "
	+ "Software without restriction, including without limitation the rights to use, "
	+ "copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the" 
	+ "Software, and to permit persons to whom the Software is furnished to do so, "
	+ "subject to the following conditions:</p>"

	+ "<p>The above copyright notice and this permission notice shall be included in all "
	+ "copies or substantial portions of the Software.</p>"

	+ "<p>THE SOFTWARE IS PROVIDED AS IS, WITHOUT WARRANTY OF ANY KIND, EXPRESS OR "
	+ "IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, "
	+ "FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE "
	+ "AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER "
	+ "LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, "
	+ "OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE "
	+ "SOFTWARE.</p>"


	+ "<h1>Sabre-dav</h1>"

	+ "<p>Copyright (C) 2007-2016 fruux GmbH (https://fruux.com/)</p>"
	+ "<p>Sabre-dav is licensed under the BSD3 license.</p>"

	+ "<p>Permission is hereby granted, free of charge, to any person obtaining a copy "
	+ "of this software and associated documentation files (the \"Software\"), to deal "
	+ "in the Software without restriction, including without limitation the rights "
	+ "to use, copy, modify, merge, publish, distribute, sublicense, and/or sell "
	+ "copies of the Software, and to permit persons to whom the Software is "
	+ "furnished to do so, subject to the following conditions:</p>"

	+ "<p>The above copyright notice and this permission notice shall be included in "
	+ "all copies or substantial portions of the Software.</p>"

	+ "<p>THE SOFTWARE IS PROVIDED \"AS IS\", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR "
	+ "IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, "
	+ "FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE "
	+ "AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER "
	+ "LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, "
	+ "OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN "
	+ "THE SOFTWARE.</p>"

	+ "<p>Redistribution and use in source and binary forms, with or without "
	+ "modification, are permitted provided that the following conditions are met:</p>"

	+ "<p><ul><li>Redistributions of source code must retain the above copyright notice, this "
	+ "list of conditions and the following disclaimer.</li>"
	+ "<li>Redistributions in binary form must reproduce the above copyright notice, "
	+ "this list of conditions and the following disclaimer in the documentation "
	+ "and/or other materials provided with the distribution.</li>"
	+ "<li>Neither the name of SabreDAV nor the names of its contributors may be used "
	+ "to endorse or promote products derived from this software without specific "
	+ "prior written permission.</li></p>"

	+ "<p>THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS \"AS IS\" AND "
	+ "ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED "
	+ "WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE "
	+ "DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR "
	+ "ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES "
	+ "(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; "
	+ "LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON "
	+ "ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT "
	+ "(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS "
	+ "SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.</p>"


	+ "<h1>Printer rendering</h1>"

	+ "<p>Copyright (C) 2010 Ed Spencer</p>"

	+ "<p>Permission is hereby granted, free of charge, to any person obtaining a copy "
	+ "of this software and associated documentation files (the \"Software\"), to deal "
	+ "in the Software without restriction, including without limitation the rights "
	+ "to use, copy, modify, merge, publish, distribute, sublicense, and/or sell "
	+ "copies of the Software, and to permit persons to whom the Software is "
	+ "furnished to do so, subject to the following conditions:</p>"

	+ "<p>The above copyright notice and this permission notice shall be included in "
	+ "all copies or substantial portions of the Software.</p>"

	+ "<p>THE SOFTWARE IS PROVIDED \"AS IS\", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR "
	+ "IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, "
	+ "FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE "
	+ "AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER "
	+ "LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, "
	+ "OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN "
	+ "THE SOFTWARE.</p>"


	+ "<h1>TinyMCE</h1>"

	+ "<p>Copyright (c) 2025, Ephox Corporation DBA Tiny Technologies, Inc.</p>"

	+ "<p>TinyMCE is free software: you can redistribute it and/or modify "
	+ "under the terms of the GNU General Public License "
	+ "as published by the Free Software Foundation; either version 2 "
	+ "of the License, or (at your option) any later version.</p>"

	+ "<p>TinyMCE is distributed in the hope that it will be useful, "
	+ "but WITHOUT ANY WARRANTY; without even the implied warranty of "
	+ "MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU "
	+ "General Public License for more details.</p>"

	+ "<p>You should have received a copy of the GNU General Public "
	+ "license along with program; "
	+ "if not, see <a href=\"https://www.gnu.org/licenses/\" target=\"_blank\">https://www.gnu.org/licenses/</a>.</p>"


	+ "<h1>Tokenizr</h1>"

	+ "<p>Copyright (c) 2015-2025 Dr. Ralf S. Engelschall <a href=\"mailto:rse@engelschall.com\">rse@engelschall.com</a></p>"

	+ "<p>Permission is hereby granted, free of charge, to any person obtaining"
	+ "a copy of this software and associated documentation files (the"
	+ "\"Software\"), to deal in the Software without restriction, including"
	+ "without limitation the rights to use, copy, modify, merge, publish,"
	+ "distribute, sublicense, and/or sell copies of the Software, and to"
	+ "permit persons to whom the Software is furnished to do so, subject to"
	+ "the following conditions:</p>"

	+ "<p>The above copyright notice and this permission notice shall be included"
	+ "in all copies or substantial portions of the Software.</p>"

	+ "<p>THE SOFTWARE IS PROVIDED \"AS IS\", WITHOUT WARRANTY OF ANY KIND,"
	+ "EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF"
	+ "MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT."
	+ "IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY"
	+ "CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,"
	+ "TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE"
	+ "SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.</p>"


	+ "<h1>DOMPurify</h1>"

	+ "<p>Copyright 2025 Dr.-Ing. Mario Heiderich, Cure53</p><p>Licensed under the Apache License, version 2.0.</p>"


	+ "<h1>ViewerJS</h1>"

	+ "<p>Copyright (C) 2015 viewerjs.org</p>"


	+ "<h1>Leaflet</h1>"

	+ "<p>Copyright (C) 2010-2025 Vladimir Agafonkin</p>"
	+ "<p>Copyright (C) 2010-2011 CloudMade</p>"

	+ "<p>Redistribution and use in source and binary forms, with or without"
	+ "modification, are permitted provided that the following conditions are met:</p>"

	+ "<p>1. Redistributions of source code must retain the above copyright notice, this"
	+ "list of conditions and the following disclaimer.</p>"

	+ "<p>2. Redistributions in binary form must reproduce the above copyright notice,"
	+ "this list of conditions and the following disclaimer in the documentation"
	+ "and/or other materials provided with the distribution.</p>"

	+ "<p>THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS \"AS IS\""
	+ "AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE"
	+ "IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE"
	+ "DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE"
	+ "FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL"
	+ "DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR"
	+ "SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER"
	+ "CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,"
	+ "OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE"
	+ "OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.</p>"


	+ "<h1>Leaflet.GeoSearch</h1>"

	+ "<p>Copyright (c) 2010-2016 Stephan Meijer</p>"

	+ "<p>Permission is hereby granted, free of charge, to any person obtaining"
	+ "a copy of this software and associated documentation files (the"
	+ "\"Software\"), to deal in the Software without restriction, including"
	+ "without limitation the rights to use, copy, modify, merge, publish,"
	+ "distribute, sublicense, and/or sell copies of the Software, and to"
	+ "permit persons to whom the Software is furnished to do so, subject to"
	+ "the following conditions:</p>"

	+ "<p>The above copyright notice and this permission notice shall be included"
	+ "in all copies or substantial portions of the Software.</p>"

	+ "<p>THE SOFTWARE IS PROVIDED \"AS IS\", WITHOUT WARRANTY OF ANY KIND,"
	+ "EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF"
	+ "MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT."
	+ "IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY"
	+ "CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,"
	+ "TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE"
	+ "SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.</p>"


	+ "<h1>PDF.js</h1>"

	+ "<p>Copyright (C) Mozilla and individual contributors</p><p>Licensed under the Apache License, version 2.0.</p>"

	+ "</div>";
/**
 * @class CanvasRenderingContext2D
 * @extends Object
 * 
 * Some convenience methods for the default canvas 2D context. These allow of drawing
 * circles, lines, text with auto-wrapping, etc. 
 */
if (!!document.createElement('canvas').getContext)
{
	/**
	 * Creates a circular path on the canvas.
	 * @param {Number} x center horizontal position.
	 * @param {Number} y center vertical position.
	 * @param {Number} radius circle radius.
	 */
	CanvasRenderingContext2D.prototype.circle = function(x, y, radius)
	{
		this.beginPath();
		this.arc(x, y, radius, 0, Math.PI*2, true); 
		this.closePath();
	};
	
	/**
	 * Strokes a straight line between two points.
	 * @param {Number} x1 horizontal component of the start point. 
	 * @param {Number} y1 vertical component of the start point.
	 * @param {Number} x2 horizontal component of the end point.
	 * @param {Number} y2 vertical component of the end point.
	 */
	CanvasRenderingContext2D.prototype.strokeLine = function(x1, y1, x2, y2)
	{
		this.beginPath();
		this.moveTo(x1, y1);
		this.lineTo(x2, y2);
		this.stroke();
	};

	/**
	 * Gets the font size from CSS font specifier string. font property of canvas object contains font specification.
	 * @return {Number} current font size of the text
	 */
	CanvasRenderingContext2D.prototype.getFontSize = function()
	{
		var fontSizeRegExp = new RegExp('([0-9]*)p[tx]');
		return parseInt(fontSizeRegExp.exec(this.font)[1], 10);
	};

	/**
	 * Draws a string of text.
	 * @param {String} text text to draw.
	 * @param {Number} x horizontal position.
	 * @param {Number} y vertical position.
	 * @param {Number} maxWidth (optional) maximum width in pixels. 
	 */
	CanvasRenderingContext2D.prototype.drawText = function(text, x, y, maxWidth)
	{
		if (maxWidth) {
			this.fillText(text, x, y, maxWidth);
		} else {
			this.fillText(text, x, y);
		}
	};

	/**
	 * Draws a string of wrapped text. 
	 * @param {String} text text to draw.
	 * @param {Number} x horizontal position.
	 * @param {Number} y vertical position.
	 * @param {Number} width width.
	 * @param {Number} lineHeight number of pixels to advance down after each line break.
	 * @param {Number} maxHeight (optional) maximum height in pixels
	 * @param {Number} clipX (optional) The number of pixels horizontally that should be clipped (from the left-top corner).
	 * @param {Number} clipY (optional) The number of pixels vertically that should be clipped (from the left-top corner).
	 * @return {Number} Total height of the drawn text
	 */
	CanvasRenderingContext2D.prototype.drawWrappedText = function(text, x, y, width, lineHeight, maxHeight, clipX, clipY)
	{
		// If the text contains '\n' characters, we have to split the text up
		// into individual lines which we can then draw on the canvas.
		while (text[text.length - 1] === '\n') {
			text = text.substr(0, text.length - 1);
		}
		var lines = text.split('\n');
		if (lines.length > 1) {
			var totalHeight = 0;

			for (var i = 0; i < lines.length; i++) {
				var offset = this.drawWrappedText(lines[i], x, y, width, lineHeight, maxHeight, clipX, clipY);
				y += offset;
				totalHeight += offset;
				maxHeight -= offset;
			}

			return totalHeight;
		}

		// split the text into words
		var words = text.split(' ');
		
		// number of pixels to advance horizontally after drawing a single word
		var spaceWidth = this.textWidth(' ');

		// Check the clipping options
		clipX = clipX || 0;
		clipY = clipY || 0;

		// render each word individually
		var tx = x + clipX;
		var ty = y;
		for (var i = 0, len = words.length; i < len; i++) {
			var word = words[i];
			var wordWidth = this.textWidth(word);

			// Test if the word will fit on this line
			if (((tx - x) + wordWidth) >= width && tx !== x) {
				var moveLine = true;

				// It doesn't fit, check if it will
				// fit if we move to the next line
				if (wordWidth >= width) {
					// It won't, lets see if at least half
					// the word will fit the current line
					if (((tx - x) + (wordWidth / 2)) < width) {
						// It will, just print it
						moveLine = false;
					}
				}

				if (moveLine) {
					ty += lineHeight;
					if (ty > clipY) {
						clipX=0;
					}
					tx = x + clipX;
					// See if we are going to write the next line
					// across the limit
					if ((ty - y) > maxHeight) {
						break;
					}
				}
			}

			this.drawText(word, tx, ty);
			tx += wordWidth + spaceWidth;
		}

		// Calculate the written height, this means that the current
		// offset must be subtracted by the start offset. And finally
		// we need to add a lineHeight, as we have written at least
		// 1 line.
		return ty - y + lineHeight;
	};

	/**
	 * Measures the height of a text when wrapped to a certain width. 
	 * @param {String} text text to draw.
	 * @param {Number} width width.
	 * @param {Number} lineHeight number of pixels to advance down after each line break.
	 * @return {Number} height of the text
	 */
	CanvasRenderingContext2D.prototype.textHeight = function(text, width, lineHeight)
	{
		// If the text contains '\n' characters, we have to split the text up
		// into individual lines which we can then draw on the canvas.
		while(text[text.length - 1] === '\n') {
			text = text.substr(0, text.length - 1);
		}
		var lines = text.split('\n');
		if (lines.length > 1) {
			var totalHeight = 0;

			for (var i = 0; i < lines.length; i++) {
				var offset = this.textHeight(lines[i], width, lineHeight);
				totalHeight += offset;
			}

			return totalHeight;
		}

		// split the text into words
		var words = text.split(' ');
		
		// number of pixels to advance horizontally after drawing a single word
		var spaceWidth = this.textWidth(' ');
		
		// render each word individually
		var tx = 0;
		var ty = 0;
		for (var i = 0, len = words.length; i < len; i++) {
			var word = words[i];
			var wordWidth = this.textWidth(word);

			// Test if the word will fit on this line
			if ((tx + wordWidth) >= width && tx !== 0) {
				var moveLine = true;

				// It doesn't fit, check if it will
				// fit if we move to the next line
				if (wordWidth >= width) {
					// It won't, lets see if at least half
					// the word will fit the current line
					if ((tx + (wordWidth / 2)) < width) {
						// It will, just print it
						moveLine = false;
					}
				}

				if (moveLine) {
					tx = 0;
					ty += lineHeight;
				}
			}

			tx += wordWidth + spaceWidth;
		}
		ty += lineHeight;
		return ty;
	};

	/**
	 * Measures a piece of text.
	 * @param {String} text text to measure.
	 * @return {Number} the width in pixels of the given text string if rendered with the current font. 
	 */
	CanvasRenderingContext2D.prototype.textWidth = function(text)
	{
		return this.measureText(text).width;
	};
	
	/**
	 * Converts the hexadecimal RGB notation (#FFAA00) to a decimal notation using the rgba 
	 * function. It will output a string that can be used in defining colors and opacity in Canvas. 
	 * By default it will return rgba(0,0,0,0) if no or incorrect values are passed. This is a black
	 * and completely transparent rgba() value.
	 * @param {String} str Hexadecimal RGB notation like #FFAA00. The '#' is optional, the rest of 
	 * the strings needs to contain 6 characters.
	 * @param {Number} opacity (Optional) Will set the opacity in the rgba function. Defaults to 1.
	 * @return {String} The rgba() notation. 
	 */
	CanvasRenderingContext2D.prototype.convertHexRgbToDecRgba = function(str, opacity)
	{
		// The regex ensures that the input has at least 6 RGB characters and the preceding # is optional
		if(Ext.isString(str) && str.search(/#?[A-F0-9]{6}/i) === 0){
			if(!Ext.isNumber(opacity)){
				opacity = 1;
			}

			var decRGB = [];
			str = str.replace('#','');
			decRGB[0] = parseInt(str.substr(0, 2), 16);
			decRGB[1] = parseInt(str.substr(2, 2), 16);
			decRGB[2] = parseInt(str.substr(4, 2), 16);
			return 'rgba('+decRGB[0]+','+decRGB[1]+','+decRGB[2]+','+opacity+')';
		}

		// Will return a black, completely transparent rgba() value
		return 'rgba(0,0,0,0)';
	};
}
Ext.namespace('Zarafa.addressbook');

/**
 * @class Zarafa.addressbook.Actions
 * Common actions which can be used within {@link Ext.Button buttons}
 * or other {@link Ext.Component components} with action handlers.
 * @singleton
 */
Zarafa.addressbook.Actions = {
	/**
	 * Open the address book
	 * @param {Object} config (optional) Configuration object used to create the ContentPanel
	 */
	openAddressBook: function(config)
	{
		config = Ext.applyIf(config || {}, {
			modal: true
		});

		var componentType = Zarafa.core.data.SharedComponentType['addressbook.dialog.addressbook'];
		Zarafa.core.data.UIFactory.openLayerComponent(componentType, undefined, config);
	},

	/**
	 * Open the details panel for the selected records
	 * @param {Zarafa.core.data.IPMRecord} records The records for which
	 * the details {@link Zarafa.core.ui.ContentPanel contentpanel} must be shown
	 * @param {Object} config (optional) Configuration object used to create the ContentPanel
	 */
	openDetailsContent: function(records, config)
	{
		if (Array.isArray(records) && !Ext.isEmpty(records)) {
			records = records[0];
		}
		if (records.isPersonalContact() || records.isSharedContact()) {
			// A personal contact needs to be converted to a contact so the correct panel can be shown.
			records = records.convertToContactRecord();
			// FIXME: We put the abRecord into the ShadowStore to be able
			// to open it, and obtain all details. However, we also need to
			// find a point where we can remove it again.
			container.getShadowStore().add(records);
		} else if (records.isPersonalDistList()) {
			// A personal distlist needs to be converted to a distlist so the correct panel can be shown.
			records = records.convertToDistListRecord();
			// FIXME: We put the abRecord into the ShadowStore to be able
			// to open it, and obtain all details. However, we also need to
			// find a point where we can remove it again.
			container.getShadowStore().add(records);
		}

		config = Ext.applyIf(config || {}, {
			modal: true
		});

		Zarafa.core.data.UIFactory.openViewRecord(records, config);
	}
};
Ext.namespace('Zarafa.addressbook.dialogs');

/**
 * @class Zarafa.addressbook.dialogs.ABEmailAddressTab
 * @extends Ext.form.FormPanel
 * @xtype zarafa.abemailaddresstab
 *
 * This class is used to create layout of email address tab panel.
 */
Zarafa.addressbook.dialogs.ABEmailAddressTab = Ext.extend(Ext.form.FormPanel, {
	/**
	 * @constructor
	 * @param {Object} config configuration object.
	 */
	constructor: function(config)
	{
		config = config || {};

		config.plugins = Ext.value(config.plugins, []);
		config.plugins.push('zarafa.recordcomponentupdaterplugin');

		Ext.applyIf(config, {
			xtype: 'zarafa.abemailaddresstab',
			title: _('Email Addresses'),
			style: 'padding: 8px;',
			layout: {
				type: 'vbox',
				pack: 'start',
				align: 'stretch'
			},
			items: [{
				xtype: 'displayfield',
				value: _('Email addresses') + ':',
				hideLabel: true
			},{
				xtype: 'panel',
				flex: 1,
				autoScroll: true,
				items: [{
					xtype: 'listview',
					// initialize a dummy store
					store: new Ext.data.Store(),
					ref: '../emailList',
					hideHeaders: true,
					singleSelect: false,
					anchor: '100% 100%',
					columns: [{
						dataIndex: 'address',
						tpl: '{address:htmlEncode}'
					}]
				}]
			}]
		});

		Zarafa.addressbook.dialogs.ABEmailAddressTab.superclass.constructor.call(this, config);
	},

	/**
	 * Function is used to update values of form fields when ever
	 * an updated {@link Zarafa.core.data.MAPIRecord record} is received
	 * @param {Zarafa.core.data.MAPIRecord} record The record update the panel with.
	 * @param {Boolean} contentReset force the component to perform a full update of the data.
	 * @private
	 */
	update: function(record, contentReset)
	{
		this.getForm().loadRecord(record);

		var proxyAddressSubStore = record.getSubStore('ems_ab_proxy_addresses');
		if (proxyAddressSubStore && this.emailList.getStore() !== proxyAddressSubStore) {
			this.emailList.bindStore(proxyAddressSubStore);
		}
	}
});

Ext.reg('zarafa.abemailaddresstab', Zarafa.addressbook.dialogs.ABEmailAddressTab);
Ext.namespace('Zarafa.addressbook.dialogs');

/**
 * @class Zarafa.addressbook.dialogs.ABGroupDetailPanel
 * @extends Ext.TabPanel
 * @xtype zarafa.abgroupdetailpanel
 */
Zarafa.addressbook.dialogs.ABGroupDetailPanel = Ext.extend(Ext.TabPanel, {
	// Insertion points for this class
	/**
	 * @insert context.addressbook.abgroupdetailcontentpanel.tabs
	 * can be used to add extra tabs to addressbook group details dialog by 3rd party plugins
	 * @param {Zarafa.addressbook.dialogs.ABGroupDetailPanel} panel This contactpanel
	 */

	/**
	 * @constructor
	 * @param {Object} config configuration object.
	 */
	constructor: function(config)
	{
		config = config || {};

		Ext.applyIf(config, {
			xtype: 'zarafa.abgroupdetailpanel',
			border: false,
			activeTab: 0,
			items: [{
				xtype: 'zarafa.abgroupgeneraltab'
			}, {
				xtype: 'zarafa.abmemberoftab'
			},{
				xtype: 'zarafa.abemailaddresstab'
			},
			// Add insertion point
			container.populateInsertionPoint('context.addressbook.abgroupdetailcontentpanel.tabs', this)
			]
		});

		Zarafa.addressbook.dialogs.ABGroupDetailPanel.superclass.constructor.call(this, config);
	}
});

Ext.reg('zarafa.abgroupdetailpanel', Zarafa.addressbook.dialogs.ABGroupDetailPanel);
Ext.namespace('Zarafa.addressbook.dialogs');

/**
 * @class Zarafa.addressbook.dialogs.ABGroupGeneralTab
 * @extends Ext.form.FormPanel
 * @xtype zarafa.abgroupgeneraltab
 *
 * This class is used to create layout of general tab in tab panel.
 */
Zarafa.addressbook.dialogs.ABGroupGeneralTab = Ext.extend(Ext.form.FormPanel, {

	/**
	 * @constructor
	 * @param {Object} config configuration object.
	 */
	constructor: function(config)
	{
		config = config || {};

		config.plugins = Ext.value(config.plugins, []);
		config.plugins.push('zarafa.recordcomponentupdaterplugin');

		Ext.applyIf(config, {
			xtype: 'zarafa.abgroupgeneraltab',
			title: _('General'),
			layout: 'column',
			autoScroll: true,
			items: [
				this.createNameFieldset(),
				this.createMembersFieldset()
			]
		});

		Zarafa.addressbook.dialogs.ABGroupGeneralTab.superclass.constructor.call(this, config);
	},

	/**
	 * Creates fieldset for general tab of tab panel.
	 * @return {Object} config object for creating {@link Ext.form.FieldSet FieldSet}.
	 * @private
	 */
	createNameFieldset: function()
	{
		return {
			xtype: 'fieldset',
			columnWidth: 0.5,
			border: false,
			cls: 'zarafa-fieldset',
			autoHeight: true,
			items: [{
				xtype: 'textfield',
				fieldLabel: _('Display Name'),
				name: 'display_name',
				anchor: '100%',
				border: false,
				readOnly: true
			},{
				xtype: 'textfield',
				fieldLabel: _('Alias'),
				name: 'account',
				anchor: '100%',
				border: false,
				readOnly: true
			},{
				xtype: 'displayfield',
				value: _('Owner') + ':',
				hideLabel: true
			},{
				xtype: 'zarafa.abitemgrid',
				ref: '../ownerList',
				anchor: '100%',
				bwrapStyle: 'width: 100%',
				autoHeight: true,
				border: false
			},{
				xtype: 'displayfield',
				value: _('Notes') + ':',
				hideLabel: true
			},{
				xtype: 'textarea',
				hideLabel: true,
				name: 'comment',
				flex: 1,
				anchor: '100%',
				border: false,
				readOnly: true
			}]
		};
	},

	/**
	 * Creates fieldset for general tab of tab panel.
	 * @return {Object} config object for creating {@link Ext.form.FieldSet FieldSet}.
	 * @private
	 */
	createMembersFieldset: function()
	{
		return {
			xtype: 'fieldset',
			columnWidth: 0.5,
			border: true,
			title: _('Members'),
			cls: 'zarafa-fieldset',
			autoHeight: true,
			items: [{
				xtype: 'zarafa.abitemgrid',
				autoWidth: true,
				border: true,
				height: 225,
				ref: '../groupMembersList',
				flex: 1,
				readOnly: true
			}]
		};
	},

	/**
	 * Function is used to update values of form fields when ever
	 * an updated {@link Zarafa.core.data.MAPIRecord record} is received
	 * @param {Zarafa.core.data.MAPIRecord} record The record update the panel with.
	 * @param {Boolean} contentReset force the component to perform a full update of the data.
	 * @private
	 */
	update: function(record, contentReset)
	{
		this.getForm().loadRecord(record);

		var membersSubStore = record.getSubStore('members');
		if (membersSubStore && this.groupMembersList.getStore() !== membersSubStore) {
			this.groupMembersList.reconfigure(membersSubStore, this.groupMembersList.getColumnModel());
		}

		var ownerSubStore = record.getSubStore('ems_ab_owner');
		if (ownerSubStore && this.ownerList.getStore() !== ownerSubStore) {
			this.ownerList.reconfigure(ownerSubStore, this.ownerList.getColumnModel());
		}
	}
});

Ext.reg('zarafa.abgroupgeneraltab', Zarafa.addressbook.dialogs.ABGroupGeneralTab);
Ext.namespace('Zarafa.addressbook.dialogs');

/**
 * @class Zarafa.addressbook.dialogs.ABItemGrid
 * @extends Ext.grid.GridPanel
 * @xtype zarafa.abitemgrid
 *
 * A gridPanel which is used in the Addressbook detail dialogs
 * for showing one or more addressbook items.
 */
Zarafa.addressbook.dialogs.ABItemGrid = Ext.extend(Ext.grid.GridPanel, {

	/**
	 * @constructor
	 * @param {Object} config Configuration object
	 */
	constructor: function(config)
	{
		config = config || {};

		Ext.applyIf(config, {
			viewConfig: {
				forceFit: true
			},
			store: new Zarafa.addressbook.AddressBookSubStore(),
			colModel: new Ext.grid.ColumnModel({
				columns: [{
					dataIndex: 'display_type',
					header: _('Icon'),
					headerCls: 'zarafa-icon-column icon',
					width: 25,
					fixed: true,
					renderer: Zarafa.common.ui.grid.Renderers.icon
				},{
					header: _('Display Name'),
					dataIndex: 'display_name',
					renderer: Ext.util.Format.htmlEncode,
					headerCls: 'k-unsortable',
					sortable: false
				},{
					header: _('Email Address'),
					dataIndex: 'smtp_address',
					renderer: Ext.util.Format.htmlEncode,
					headerCls: 'k-unsortable',
					sortable: false
				},{
					header: _('Account'),
					dataIndex: 'account',
					renderer: Ext.util.Format.htmlEncode,
					headerCls: 'k-unsortable',
					sortable: false
				}]
			})
		});

		Zarafa.addressbook.dialogs.ABItemGrid.superclass.constructor.call(this, config);

		this.on('rowdblclick', this.onRowDblClick, this);
	},

	/**
	 * Event handler which is fired when a row in the grid has been double-clicked.
	 * This will open the selected addressbook item in a new dialog.
	 * @param {Ext.grid.GridPanel} grid The grid which fired the event
	 * @param {Number} rowIndex The index of the row which was double clicked
	 * @param {Ext.EventObject} event The event
	 */
	onRowDblClick: function(grid, rowIndex, event)
	{
		var item = this.getStore().getAt(rowIndex);
		if (item) {
			Zarafa.core.data.UIFactory.openViewRecord(item);
		}
	}
});

Ext.reg('zarafa.abitemgrid', Zarafa.addressbook.dialogs.ABItemGrid);
Ext.namespace('Zarafa.addressbook.dialogs');

/**
 * @class Zarafa.addressbook.dialogs.ABMemberOfTab
 * @extends Ext.form.FormPanel
 * @xtype zarafa.abmemberoftab
 *
 * This class is used to create layout of MemberOf tab panel.
 */
Zarafa.addressbook.dialogs.ABMemberOfTab = Ext.extend(Ext.form.FormPanel, {
	/**
	 * @constructor
	 * @param {Object} config configuration object.
	 */
	constructor: function(config)
	{
		config = config || {};

		config.plugins = Ext.value(config.plugins, []);
		config.plugins.push('zarafa.recordcomponentupdaterplugin');

		Ext.applyIf(config, {
			xtype: 'zarafa.abmemberoftab',
			title: _('Member Of'),
			style: 'padding: 8px;',
			layout: {
				type: 'vbox',
				align: 'stretch'
			},
			items: [{
				xtype: 'displayfield',
				value: _('Group Membership') + ':',
				hideLabel: true
			},{
				xtype: 'zarafa.abitemgrid',
				ref: 'memberOfList',
				flex: 1
			}]
		});

		Zarafa.addressbook.dialogs.ABMemberOfTab.superclass.constructor.call(this, config);
	},

	/**
	 * Function is used to update values of form fields when ever
	 * an updated {@link Zarafa.core.data.MAPIRecord record} is received
	 * @param {Zarafa.core.data.MAPIRecord} record The record update the panel with.
	 * @param {Boolean} contentReset force the component to perform a full update of the data.
	 * @private
	 */
	update: function(record, contentReset)
	{
		this.getForm().loadRecord(record);
		var memberOfSubStore = record.getSubStore('ems_ab_is_member_of_dl');
		if (memberOfSubStore && this.memberOfList.getStore() !== memberOfSubStore) {
			this.memberOfList.reconfigure(memberOfSubStore, this.memberOfList.getColumnModel());
		}
	}
});

Ext.reg('zarafa.abmemberoftab', Zarafa.addressbook.dialogs.ABMemberOfTab);
Ext.namespace('Zarafa.addressbook.dialogs');

/**
 * @class Zarafa.addressbook.dialogs.ABUserDetailPanel
 * @extends Ext.TabPanel
 * This class is used as wrapper class for all tabs, individual tab will have its own class
 * @xtype zarafa.abuserdetailpanel
 */
Zarafa.addressbook.dialogs.ABUserDetailPanel = Ext.extend(Ext.TabPanel,{
	// Insertion points for this class
	/**
	 * @insert context.addressbook.abuserdetailcontentpanel.tabs
	 * can be used to add extra tabs to addressbook user details dialog by 3rd party plugins
	 * @param {Zarafa.addressbook.dialogs.ABUserDetailPanel} panel This contactpanel
	 */

	/**
	 * @constructor
	 * @param {Object} config configuration object.
	 */
	constructor: function(config)
	{
		config = config || {};

		Ext.applyIf(config, {
			xtype: 'zarafa.abuserdetailpanel',
			border: false,
			activeTab: 0,
			items: [{
				xtype: 'zarafa.abusergeneraltab'
			}, {
				xtype: 'zarafa.abuserorganizationtab'
			},{
				xtype: 'zarafa.abuserphonetab'
			}, {
				xtype: 'zarafa.abmemberoftab'
			},{
				xtype: 'zarafa.abemailaddresstab'
			},
			// Add insertion point
			container.populateInsertionPoint('context.addressbook.abuserdetailcontentpanel.tabs', this)
			]
		});

		Zarafa.addressbook.dialogs.ABUserDetailPanel.superclass.constructor.call(this, config);
	}
});

Ext.reg('zarafa.abuserdetailpanel', Zarafa.addressbook.dialogs.ABUserDetailPanel);
Ext.namespace('Zarafa.addressbook.dialogs');

/**
 * @class Zarafa.addressbook.dialogs.ABUserGeneralTab
 * @extends Ext.form.FormPanel
 * @xtype zarafa.abusergeneraltab
 *
 * This class is used to create layout of general tab in tab panel.
 */
Zarafa.addressbook.dialogs.ABUserGeneralTab = Ext.extend(Ext.form.FormPanel, {

	/**
	 * @constructor
	 * @param {Object} config configuration object.
	 */
	constructor: function(config)
	{
		config = config || {};

		config.plugins = Ext.value(config.plugins, []);
		config.plugins.push('zarafa.recordcomponentupdaterplugin');

		Ext.applyIf(config, {
			xtype: 'zarafa.abusergeneraltab',
			title: _('General'),	
			layout: 'column',
			autoHeight: true,
			autoScroll: true,
			border: false,
			layoutConfig: {
				columns: 2
			},
			defaults: {
				xtype: 'fieldset',
				columnWidth: 0.5,
				cls: 'k-fieldset',
				hideBorders: true
			},
			items: [
				this.createNameFieldset(),
				this.createPhotoFieldset(),
				this.createAddressFieldset(),
				this.createOfficeFieldset()
			]
		});

		Zarafa.addressbook.dialogs.ABUserGeneralTab.superclass.constructor.call(this, config);
	},

	/**
	 * Creates the name fieldset for general tab of form panel.
	 * @return {Object} config object for creating {@link Ext.form.FieldSet FieldSet}.
	 * @private
	 */
	createNameFieldset: function()
	{
		return {
			defaults: {
				anchor: '100%',
				readOnly: true,
			},
			items: [{
				xtype: 'textfield',
				fieldLabel: _('First name'),
				name: 'given_name',
			}, {
				xtype: 'textfield',
				flex: 1,
				fieldLabel: _('Last name'),
				name: 'surname'
			},{
				xtype: 'textfield',
				flex: 1,
				fieldLabel: _('Initials'),
				name: 'initials'
			},{
				xtype: 'textfield',
				flex: 1,
				fieldLabel:_('Display'),
				name: 'display_name'
			},{
				xtype: 'textfield',
				flex: 1,
				fieldLabel: _('Alias'),
				name: 'account'
			}]
		};
	},

	/**
	 * Creates the photo fieldset for general tab of form panel.
	 * @return {Object} config object for creating {@link Ext.form.FieldSet FieldSet}.
	 * @private
	 */
	createPhotoFieldset: function()
	{
		return {
			border: false,
			layout: {
				type: 'hbox',
				flex: 1
			},
			items: {
				xtype: 'box',
				cls: 'contact_photo_box default_contact_photo',
				ctCls: 'contact_photo_box_ct',
				autoEl: {
					tag: 'img',
					height: 135,
					src: Ext.BLANK_IMAGE_URL
				},
				ref: '../contactPhotoBox'
			}
		};
	},

	/**
	 * Creates fieldset for general tab of tab panel.
	 * @return {Object} config object for creating {@link Ext.form.FieldSet FieldSet}.
	 * @private
	 */
	createAddressFieldset: function()
	{
		return {
			xtype: 'fieldset',
			defaultType: 'textfield',
			title: _('Addresses'),
			border: false,
			defaults: {
				anchor:'100%',
				readOnly: true
			},
			items: [{
				xtype: 'textarea',
				fieldLabel: _('Address'),
				name: 'street_address',
				flex: 1,
				height: 51
			},{
				fieldLabel: _('City'),
				flex: 1,
				name: 'locality'
			},{
				fieldLabel: _('State'),
				flex: 1,
				name: 'state_or_province'
			},{
				fieldLabel: _('Zip Code'),
				flex: 1,
				name: 'postal_code'
			},{
				fieldLabel: _('Country'),
				flex: 1,
				name: 'country'
			}]
		};
	},

	/**
	 * Creates fieldset for general tab of tab panel.
	 * @return {Object} config object for creating {@link Ext.form.FieldSet FieldSet}.
	 * @private
	 */
	createOfficeFieldset: function()
	{
		return {
			xtype: 'fieldset',
			defaultType: 'textfield',
			title: _('Professional'),
			defaults: {
				anchor: '100%',
				readOnly: true
			},
			items: [{
				fieldLabel: _('Title'),
				flex: 1,
				name: 'title'
			},{
				fieldLabel: _('Company'),
				flex: 1,
				name: 'company_name'
			},{
				fieldLabel: _('Department'),
				flex: 1,
				name: 'department_name'
			},{
				fieldLabel: _('Office'),
				flex: 1,
				name: 'office_location'
			},{
				fieldLabel: _('Assistant'),
				flex: 1,
				name: 'assistant'
			},{
				fieldLabel: _('Phone'),
				flex: 1,
				name: 'business_telephone_number'
			}]
		};
	},

	/**
	 * Function is used to update values of form fields when ever
	 * an updated {@link Zarafa.core.data.MAPIRecord record} is received
	 * @param {Zarafa.core.data.MAPIRecord} record The record update the panel with.
	 * @param {Boolean} contentReset force the component to perform a full update of the data.
	 * @private
	 */
	update: function(record, contentReset)
	{
		if(Ext.isEmpty(record)) {
			return;
		}

		this.getForm().loadRecord(record);

		if (record.isOpened() && contentReset) {
			// Show GAB contact photo.
			var abThumbnail = record.get('ems_ab_thumbnail_photo');
			if (!Ext.isEmpty(abThumbnail)) {
				var imageField = this.contactPhotoBox.getEl();
				imageField.dom.setAttribute('src', abThumbnail);
				imageField.removeClass('default_contact_photo');
			}
		}
	}
});

Ext.reg('zarafa.abusergeneraltab', Zarafa.addressbook.dialogs.ABUserGeneralTab);
Ext.namespace('Zarafa.addressbook.dialogs');

/**
 * @class Zarafa.addressbook.dialogs.ABUserOrganizationTab
 * @extends Ext.form.FormPanel
 * @xtype zarafa.abuserorganizationtab
 *
 * This class is used to create layout of Organization tab of tab panel.
 */
Zarafa.addressbook.dialogs.ABUserOrganizationTab = Ext.extend(Ext.form.FormPanel, {
	/**
	 * @constructor
	 * @param {Object} config configuration object.
	 */
	constructor: function(config)
	{
		config = config || {};

		config.plugins = Ext.value(config.plugins, []);
		config.plugins.push('zarafa.recordcomponentupdaterplugin');

		Ext.applyIf(config, {
			xtype: 'zarafa.abuserorganizationtab',
			title: _('Organization'),
			selModel: this.initSelectionModel(),
			padding: 5,
			layout: {
				type: 'vbox',
				align: 'stretch'
			},
			autoScroll: true,
			items: [
				this.createManagerFieldset(),
				this.createDirectReportFieldset()
			]
		});

		Zarafa.addressbook.dialogs.ABUserOrganizationTab.superclass.constructor.call(this, config);
	},

	/**
	 * Creates manager fieldset for organization tab
	 * @return {Object} config object for creating {@link Ext.form.FieldSet FieldSet}.
	 * @private
	 */
	createManagerFieldset: function()
	{
		return [{
			xtype: 'displayfield',
			margins: '0 0 5 0',
			value: _('Manager') + ':',
			hideLabel: true
		},{
			xtype: 'zarafa.abitemgrid',
			ref: 'managerList',
			autoHeight: true
		}];
	},

	/**
	 * creates and returns a selection model object, used in {@link Ext.grid.GridPanel.selModel selModel} config
	 * @return {Ext.grid.RowSelectionModel} selection model object
	 * @private
	 */
	initSelectionModel: function()
	{
		return new Ext.grid.RowSelectionModel({
			singleSelect: false
		});
	},

	/**
	 * Creates the direct report fieldset for organization tab of form panel.
	 * @return {Object} config object for creating {@link Ext.form.FieldSet FieldSet}.
	 * @private
	 */
	createDirectReportFieldset: function()
	{
		return [{
			xtype: 'displayfield',
			margins: '5 0',
			value: _('Direct reports') + ':',
			hideLabel: true
		},{
			xtype: 'zarafa.abitemgrid',
			ref: 'reportList',
			flex: 1
		}];
	},

	/**
	 * Function is used to update values of form fields when ever
	 * an updated {@link Zarafa.core.data.MAPIRecord record} is received
	 * @param {Zarafa.core.data.MAPIRecord} record The record update the panel with.
	 * @param {Boolean} contentReset force the component to perform a full update of the data.
	 * @private
	 */
	update: function(record, contentReset)
	{
		this.getForm().loadRecord(record);
		var managerSubStore = record.getSubStore('ems_ab_manager');
		if (managerSubStore && this.managerList.getStore() !== managerSubStore) {
			this.managerList.reconfigure(managerSubStore, this.managerList.getColumnModel());
		}

		var reportsSubStore = record.getSubStore('ems_ab_reports');
		if (reportsSubStore && this.reportList.getStore() !== reportsSubStore) {
			this.reportList.reconfigure(reportsSubStore, this.reportList.getColumnModel());
		}
	}
});

Ext.reg('zarafa.abuserorganizationtab', Zarafa.addressbook.dialogs.ABUserOrganizationTab);
Ext.namespace('Zarafa.addressbook.dialogs');

/**
 * @class Zarafa.addressbook.dialogs.ABUserPhoneTab
 * @extends Ext.form.FormPanel
 * @xtype zarafa.abuserphonetab
 *
 * This class is used to create layout of phone tab of tab panel.
 */
Zarafa.addressbook.dialogs.ABUserPhoneTab = Ext.extend(Ext.form.FormPanel, {

	/**
	 * @constructor
	 * @param {Object} config configuration object.
	 */
	constructor: function(config)
	{
		config = config || {};

		config.plugins = Ext.value(config.plugins, []);
		config.plugins.push('zarafa.recordcomponentupdaterplugin');

		Ext.applyIf(config, {
			xtype: 'zarafa.abuserphonetab',
			title: _('Phone'),
			layout: {
				type: 'vbox',
				align: 'stretch'
			},
			autoScroll: true,
			items: [
				this.createPhonenumberFieldset(),
				this.createNoteFieldset()
			]
		});

		Zarafa.addressbook.dialogs.ABUserPhoneTab.superclass.constructor.call(this, config);
	},

	/**
	 * Creates the Phonenumber fieldset for phone tab
	 * @return {Object} config object for creating {@link Ext.form.FieldSet FieldSet}.
	 * @private
	 */
	createPhonenumberFieldset: function()
	{
		return {
			xtype: 'fieldset',
			title: _('Phone numbers'),
			border: true,
			cls: 'zarafa-fieldset',
			layout: 'column',
			columnWidth: 1,
			autoHeight: true,
			anchor: '100%',
			items: [{
				xtype: 'container',
				layout: 'form',
				columnWidth	: 0.5,
				border: false,
				defaults: {
					anchor: '100%',
					editable: false,
					readOnly: true
				},
				items: [{
					xtype: 'textfield',
					fieldLabel:_('Business'),
					name: 'business_telephone_number'
				},{
					xtype: 'combo',
					mode: 'local',
					fieldLabel:_('Business2'),
					store: new Zarafa.addressbook.AddressBookTelephoneNumberSubStore(),
					autoSelect: true,
					forceSelection: true,
					lazyInit: false,
					editable: false,
					readOnly: false,
					triggerAction	: 'all',
					displayField: 'number',
					valueField: 'number',
					ref: '../../business2PhoneCombo'
				},{
					xtype: 'textfield',
					fieldLabel:_('Fax'),
					name: 'primary_fax_number'
				},{
					xtype: 'textfield',
					fieldLabel: _('Assistant'),
					name: 'assistant'
				}]
			},{
				xtype: 'container',
				layout: 'form',
				columnWidth	: 0.5,
				border: false,
				defaults: {
					anchor: '100%',
					editable: false,
					readOnly: true
				},
				items: [{
					xtype: 'textfield',
					fieldLabel: _('Home'),
					name: 'home_telephone_number'
				},{
					xtype: 'combo',
					mode: 'local',
					fieldLabel:_('Home2'),
					store: new Zarafa.addressbook.AddressBookTelephoneNumberSubStore(),
					forceSelection: true,
					lazyInit: false,
					editable: false,
					readOnly: false,
					triggerAction	: 'all',
					displayField: 'number',
					valueField: 'number',
					ref: '../../home2PhoneCombo'
				},{
					xtype: 'textfield',
					fieldLabel: _('Mobile'),
					name: 'mobile_telephone_number'
				},{
					xtype: 'textfield',
					fieldLabel: _('Pager'),
					name: 'pager_telephone_number'
				}]
			}]
		};
	},

	/**
	 * Creates the note fieldset for phone tab
	 * @return {Object} config object for creating {@link Ext.form.FieldSet FieldSet}.
	 * @private
	 */
	createNoteFieldset: function()
	{
		return [{
			xtype: 'fieldset',
			title: _('Notes'),
			border: true,
			cls: 'zarafa-fieldset',
			layout: 'column',
			columnWidth: 1,
			autoHeight: true,
			anchor: '100%',
			items: [{
				xtype: 'textarea',
				hideLabel: true,
				name: 'comment',
				readOnly: true,
				columnWidth: 1,
				flex: 1
			}]
		}];
	},

	/**
	 * Function is used to update values of form fields when ever
	 * an updated {@link Zarafa.core.data.MAPIRecord record} is received
	 * @param {Zarafa.core.data.MAPIRecord} record The record update the panel with.
	 * @param {Boolean} contentReset force the component to perform a full update of the data.
	 * @private
	 */
	update: function(record, contentReset)
	{
		this.getForm().loadRecord(record);

		var businessPhoneSubStore = record.getSubStore('business2_telephone_numbers');
		if (businessPhoneSubStore && this.business2PhoneCombo.getStore() !== businessPhoneSubStore) {
			this.business2PhoneCombo.bindStore(businessPhoneSubStore);
			var select = businessPhoneSubStore.getAt(0);
			if (select) {
				this.business2PhoneCombo.setValue(select.get(this.business2PhoneCombo.valueField));
			}
		}

		var homePhoneSubStore = record.getSubStore('home2_telephone_numbers');
		if (homePhoneSubStore && this.home2PhoneCombo.getStore() !== homePhoneSubStore) {
			this.home2PhoneCombo.bindStore(homePhoneSubStore);
			var select = homePhoneSubStore.getAt(0);
			if (select) {
				this.home2PhoneCombo.setValue(select.get(this.home2PhoneCombo.valueField));
			}
		}
	}
});

Ext.reg('zarafa.abuserphonetab', Zarafa.addressbook.dialogs.ABUserPhoneTab);
Ext.namespace('Zarafa.addressbook.ui');

/**
 * @class Zarafa.addressbook.ui.ABMultiUserSelectionPanel
 * @extends Ext.Panel
 * @xtype zarafa.abmultiuserselectionpanel
 */
Zarafa.addressbook.ui.ABMultiUserSelectionPanel = Ext.extend(Ext.Panel, {
	/**
	 * @cfg {Function} callback the callback function to return to after selecting user from AdressBook.
	 */
	callback: Ext.emptyFn,

	/**
	 * @cfg {Function} convert the convert function which converts an
	 * {@link Zarafa.addressbook.AddressBookRecord user} to the correct type
	 * which can be placed in the {@link #store}.
	 * This function receives the selected AddressBookRecord as first argument,
	 * and optionally passes the {@link Ext.Component} which was generated from the
	 * {@link #selectionCfg} which was used to select the recipient as second argument.
	 */
	convert: Ext.emptyFn,

	/**
	 * @cfg {Function} scope the scope in which the {@link #callback} will be called
	 */
	scope: undefined,

	/**
	 * @cfg {Ext.data.Store} store The store in which all records should be placed.
	 */
	store: undefined,

	/**
	 * @cfg {Array} selectionCfg Array of {@link Zarafa.common.ui.BoxField} configuration
	 * objects which are created below the User list. These will show which users
	 * the user has selected.
	 */
	selectionCfg: undefined,

	/**
	 * @constructor
	 * @param {Object} config Configuration structure
	 */
	constructor: function(config)
	{
		config = config || {};

		var items = [{
			xtype: 'zarafa.addressbookmainpanel',
			ref: 'abPanel',
			flex: 1,
			hideContactsFolders: config.hideContactsFolders,
			listRestriction: config.listRestriction
		}];

		Ext.each(config.selectionCfg, function(cfg) {
			var btn = {
				xtype: 'button',
				text: cfg.fieldLabel,
				//width: 100,
				autoHeight: true,
				handler: this.onSelectionBtnClick,
				scope: this
			};

			items.push({
				xtype: 'zarafa.resizablecompositefield',
				cls: 'zarafa-addressbook-dialog-compositefield',
				hideLabel: true,
				anchor: '100%',
				autoHeight: false,
				items: [ btn, cfg ]
			});
		}, this);

		Ext.applyIf(config, {
			xtype: 'zarafa.abmultiuserselectionpanel',
			cls: 'k-abmultiuserselectionpanel',
			layout: {
				type: 'vbox',
				align: 'stretch'
			},
			items: items,
			buttons: [{
				text: _('Ok'),
				handler: this.onSubmit,
				scope: this
			},{
				text: _('Cancel'),
				handler: this.onCancel,
				scope: this
			}]
		});

		Zarafa.addressbook.ui.ABMultiUserSelectionPanel.superclass.constructor.call(this, config);
	},

	/**
	 * Calls callback function
	 */
	doCallBack: function() {
		var grid = this.abPanel.getGridPanel();
		var sm = grid.getSelectionModel();
		var records = sm.getSelections();

		if (!Ext.isEmpty(records)) {
			Ext.each(records, function(r) {
				this.store.add(this.convert(r));
			}, this);
		}
	},

	/**
	 * Initialize events
	 * @private
	 */
	initEvents: function()
	{
		Zarafa.addressbook.ui.ABMultiUserSelectionPanel.superclass.initEvents.apply(this, arguments);

		this.mon(this.abPanel.getGridPanel(), 'rowdblclick', this.onAddressBookRowDblClick, this);
		this.mon(this.abPanel.getGridPanel(), 'rowcontextmenu', this.onAddressBookRowContextMenu, this);
	},

	/**
	 * Event handler which is triggered when the user doubleclicks on a
	 * row within the {@link Ext.grid.GridPanel gridpanel}. This will add
	 * the selected user or group to the {@link Zarafa.core.ui.RecipientField recipientfield}
	 * @private
	 */
	onAddressBookRowDblClick: function()
	{
		this.onSelectionBtnClick();
	},

	/**
	 * Event handler which is triggered when the user rightclicks
	 * on a row in the {@link Ext.grid.GridPanel gridpanel}. This will
	 * open a {@link Zarafa.core.ui.menu.ConditionalMenu contextmenu}
	 * for the selected row.
	 * @param {Ext.grid.GridPanel} grid The grid on which the user clicked
	 * @param {Number} rowIndex the row on which was doubleclicked
	 * @param {Ext.EventObject} event The event information
	 * @private
	 */
	onAddressBookRowContextMenu: function(grid, rowIndex, event)
	{
		var sm = grid.getSelectionModel();

		if (sm.hasSelection()) {
			// Some records were selected...
			if (!sm.isSelected(rowIndex)) {
				// But none of them was the record on which the
				// context menu was invoked. Reset selection.
				sm.clearSelections();
				sm.selectRow(rowIndex);
			}
		} else {
			// No records were selected,
			// select row on which context menu was invoked
			sm.selectRow(rowIndex);
		}

		Zarafa.core.data.UIFactory.openDefaultContextMenu(sm.getSelections(), {
			position: event.getXY(),
			dialog: this.dialog
		});
	},

	/**
	 * Event handler which is fired when one of the Selection Buttons generated
	 * from the {@link #selectionCfg} was called. This will call {@link #convert}
	 * on the selected record, and then add it to the {@link #store}.
	 * @param {Ext.Button} btn The button which was clicked
	 * @private
	 */
	onSelectionBtnClick: function(btn)
	{
		var field = btn ? btn.ownerCt.get(1) : undefined;
		var grid = this.abPanel.getGridPanel();
		var sm = grid.getSelectionModel();
		var records = sm.getSelections();

		if (Ext.isFunction(this.convert) && !Ext.isEmpty(records)) {
			Ext.each(records, function(r) {
				r = this.convert.call(this.scope || this, r, field);
				if (r) {
					this.store.add(r);
				}
			}, this);
		}
	},

	/**
	 * Event handler which is called when the user presses the "Ok" button.
	 * Function will store changed data in record and close the dialog
	 * @private
	 */
	onSubmit: function()
	{
		if (Ext.isFunction(this.callback)) {
			if (this.callback.call(this.scope || this) !== false) {
				this.dialog.close();
			} else {
				Ext.MessageBox.show({
					title: _('No user selected'),
					msg: _('You must select one or more users.'),
					buttons: Ext.MessageBox.OK
				});
			}
		}
	},

	/**
	 * Closes {@link Zarafa.core.ui.CreateFolderDialog CreateFolder} dialog
	 * @private
	 */
	onCancel: function()
	{
		this.dialog.close();
	}
});

Ext.reg('zarafa.abmultiuserselectionpanel', Zarafa.addressbook.ui.ABMultiUserSelectionPanel);
Ext.namespace('Zarafa.addressbook.ui');

/**
 * @class Zarafa.addressbook.ui.AddressBookMainPanel
 * @extends Ext.Panel
 * @xtype zarafa.addressbookmainpanel
 */
Zarafa.addressbook.ui.AddressBookMainPanel = Ext.extend(Ext.Panel, {
	/**
	 * @cfg {Zarafa.addressbook.AddressBookStore} addressBookStore
	 * The store which should be used for displaying the contents of the
	 * addressbook inside the grid. If not provided, a new store will
	 * allocated.
	 */
	addressBookStore: undefined,

	/**
	 * @cfg (Boolean) Set to true to hide contacts folders in the address book
	 * hierarchy dropdown.
	 */
	hideContactsFolders: false,

	/**
	 * @cfg {Object} listRestriction The default restriction which
	 * must be send to the server side when obtaining a fresh list
	 * from the server. This can be used to restrict the visibility
	 * of users, groups, companies etc.
	 */
	listRestriction: undefined,

	/**
	 * @cfg {Boolean} singleSelect true to allow selection of only one row at a time (defaults to false allowing multiple selections)
	 */
	singleSelect: false,

	/**
	 * The text that will be shown in the grid when the user has to do a search before
	 * results are shown. (for the GAB when the admin has set ENABLE_FULL_GAB to false)
	 * @property
	 * @type {String}
	 */
	emptyGridText: _('Use the search bar to get results'),

	/**
	 * The text that will be shown in the grid when a search gave no results.
	 * @property
	 * @type {String}
	 */
	noResultsGridText: _('There are no items to show in this list'),

	/**
 	 * @constructor
	 * @param {Object} config Configuration structure
	 */
	constructor: function(config)
	{
		// Let's remove the state settings of the address book panel once so
		// everyone will start again with the new default settings (januari 2017)
		var sm = container.getSettingsModel();
		var stateUpdated = sm.get('zarafa/v1/contexts/addressbook/stateUpdated');
		if ( Ext.isEmpty(stateUpdated) ) {
			sm.remove('zarafa/v1/state/dialogs/addressbookcontentpanel');
			sm.remove('zarafa/v1/state/dialogs/abuserselectioncontentpanel');
			sm.remove('zarafa/v1/state/gab/contacts');
			sm.remove('zarafa/v1/state/gab/globaladdressbook');
			sm.set('zarafa/v1/contexts/addressbook/stateUpdated', 1);
		}

		config = config || {};

		if (!Ext.isDefined(config.addressBookStore)) {
			config.addressBookStore = new Zarafa.addressbook.AddressBookStore();
		}

		var items = config.items || [];
		// whatever items passed in config should be added at last place
		config.items = [
			this.createHeaderPanel(config.hideContactsFolders),
			this.createViewPanel(config.addressBookStore, { singleSelect: Ext.isDefined(config.singleSelect) ? config.singleSelect : this.singleSelect})
		].concat(items);

		Ext.applyIf(config, {
			xtype: 'zarafa.addressbookmainpanel',
			border: false,
			cls: 'k-addressbookmainpanel',
			layout: {
				type:'vbox',
				align:'stretch'
			}
		});

		// Call parent constructor
		Zarafa.addressbook.ui.AddressBookMainPanel.superclass.constructor.call(this, config);

		// Load the address book
		this.initDialog();

		// When the MainPanel is destroyed, also destroy the store,
		// this ensures that any pending requests will be cancelled.
		this.on('destroy', this.addressBookStore.destroy, this.addressBookStore);
	},

	/**
	 * Initialize the header panel in which the search bar, and the
	 * container selection components are placed which can filter the
	 * contents of the {@link Ext.grid.GridPanel gridpanel}.
	 * @return {Object} Configuration object for the header panel
	 * @private
	 */
	createHeaderPanel: function(hideContactsFolders)
	{
		var hierarchyTpl = new Ext.XTemplate(
			'<tpl for=".">',
				'<div class="x-combo-list-item<tpl if="group_header"> k-combo-list-item-header</tpl>">',
					'{depth:indent}{display_name:htmlEncode}',
				'</div>',
			'</tpl>',
			{
				compiled: true
			}
		);

		return {
			xtype: 'panel',
			cls: 'k-addressbookmainpanel-header',
			border: false,
			layout: 'hbox',
			items: [{
				xtype: 'trigger',
				ref: '../searchText',
				flex: 1,
				hideFieldLabel: true,
				enableKeyEvents: true,
				triggerClass: 'icon_magnifier',
				triggerScope: this,
				onTriggerClick: this.onSearchButtonClick.createDelegate(this),
				wrapFocusClass: '',
				listeners:{
					scope: this,
					render: this.onRenderSearchField,
					keyup: this.onSearchTextFiledKeyUp
				}
			},{
				xtype: 'spacer',
				width: 30,
				height: 10
			},{
				xtype: 'container',
				width: 355,
				items: [{
					xtype: 'combo',
					width: 200,
					plugins: [ 'zarafa.fieldlabeler', 'zarafa.comboautowidth' ],
					fieldLabel: _('Show Names from the'),
					labelWidth: 150,
					listClass: 'k-addressbook-combo-list',
					editable: false,
					mode: 'local',
					triggerAction: 'all',
					store: Zarafa.addressbook.AddressBookHierarchyStore,
					displayField: 'display_name',
					valueField: 'entryid',
					ref: '../../addressBookSelectionCB',
					tpl: hierarchyTpl,
					autoSelect: true,
					minListWidth: 150,
					listeners:{
						beforeselect: this.onBeforeSelectAddressBook,
						select: this.onAddressBookChange,
						scope: this
					},
					onLoad: this.onAddressBookComboLoad.createDelegate(this, [hideContactsFolders])
				}]
			}]
		};
	},

	/**
	 * Initialize View Panel which shows the grid of the address book items
	 * in the selected container.
	 * @param {Zarafa.addressbook.AddressBookStore} addressBookStore The store which must
	 * be displayed in the GridPanel.
	 * @param {Object} viewConfig config options for view panel
	 * be displayed in the GridPanel.
	 * @return {Object} The Configuration object for the {@link Ext.grid.GridPanel gridpanel}
	 * @private
	 */
	createViewPanel: function(addressBookStore, viewConfig)
	{
		return Ext.apply(viewConfig, {
			xtype: 'zarafa.addressbookgrid',
			hideLabel: true,
			name: 'viewpanel',
			cls: 'k-addressbookmainpanel-grid',
			viewConfig: {
				emptyText: this.emptyGridText,
				deferEmptyText: false
			},
			store: addressBookStore,
			border: false,
			ref: 'viewPanel',
			flex: 1
		});
	},

	/**
	 * Obtain the {@link Ext.grid.GridPanel gridpanel} which is used within this
	 * {@link Ext.Panel panel}.
	 * @return {Zarafa.addressbook.ui.AddressBookGrid} The GridPanel for this panel
	 */
	getGridPanel: function()
	{
		return this.viewPanel;
	},

	/**
	 * Event handler for the render event of the searchfield. Will add a placeholder
	 * attribute to the input.
	 */
	onRenderSearchField: function(triggerField)
	{
		triggerField.getEl().set({'placeholder': _('Search…')});
	},

	/**
	 * Event handler for the onbeforeselect event of the Address Book combo. Will
	 * make sure group headers cannot be selected.
	 *
	 * @param {Ext.form.ComboBox} combo The Address Book combobox
	 * @param {Zarafa.core.data.IPMRecord IPMRecord} record The selected Address Book record
	 * @param {Number} index The index of the selected record in the combo
	 */
	onBeforeSelectAddressBook: function(combo, record, index)
	{
		return !record.get('group_header');
	},

	/**
	 * Event handler which is triggered when the addressbook
	 * combobox has been used to change the selected addressbook
	 *
	 * @param {Ext.form.ComboBox} field The combobox which was selected
	 * @param {Zarafa.core.data.IPMRecord} record The selected Record
	 * @param {Number} The index number of the selected record.
	 * @private
	 */
	onAddressBookChange: function(field, record, index)
	{
		// Trigger a search
		this.onSearchButtonClick();
	},

	/**
	 * Overriding the onLoad function of the combobox to be able to use the filter
	 * on the store of the combobox. The doQuery function of {@Ext.form.ComboBox}
	 * will clear the filter before calling onLoad, so we set it in this
	 * override.
	 */
	onAddressBookComboLoad: function(hideContactsFolders) {
		if ( hideContactsFolders === true ) {
			Zarafa.addressbook.AddressBookHierarchyStore.filter('type', 'gab');
		} else {
			Zarafa.addressbook.AddressBookHierarchyStore.clearFilter();
		}

		Ext.form.ComboBox.prototype.onLoad.call(this.addressBookSelectionCB);
	},

	/**
	 * Initializes the dialog by selecting the default address book in the dropdown
	 * and triggering a search to fill the grid.
	 * @private
	 */
	initDialog: function()
	{
		var record;

		// Check that addressBookSelectionCB is created
		if (!Ext.isDefined(this.addressBookSelectionCB)) {
			return;
		}

		// Check that we have at least obtained one item
		if (Zarafa.addressbook.AddressBookHierarchyStore.getCount() === 0) {
			return;
		}

		// Get the entryId of default address book configured in setting
		var folderEntryId = container.getSettingsModel().get('zarafa/v1/main/default_addressbook');

		// If there is no configuration for default address book into setting,
		// than we by default displays the 'Global Address Book'
		if (!Ext.isEmpty(folderEntryId)) {
			record = Zarafa.addressbook.AddressBookHierarchyStore.getById(folderEntryId);
		}

		if (Ext.isEmpty(record)) {
			record = Zarafa.addressbook.AddressBookHierarchyStore.getAt(0);
		}

		var entryid = record.get('entryid');
		if (!Ext.isDefined(entryid)) {
			return;
		}

		this.addressBookSelectionCB.setValue(entryid);

		// Trigger a search when the grid has been rendered. We do this
		// because Ext 'forgets' to add the scrollOfset to an empty grid
		// when the store fires the load event, but it does do this when
		// rendering an empty grid on start. Since we don't want the
		// emptyText to jump because of this we will make sure the store
		// fires the load event when the grid is rendered and not before
		// it is rendered.
		this.viewPanel.on('render', this.onSearchButtonClick, this);
	},

	/**
	 * Event handler which is triggered when the searchButton is
	 * clicked, The handler will send request to server to search
	 * in addressbook
	 * @private
	 */
	onSearchButtonClick: function()
	{
		var selectedFolder = this.getSelectedFolderRecord();
		if ( !Ext.isDefined(selectedFolder) ) {
			return;
		}

		var folderType = selectedFolder.get('type');
		var fullGabDisabled = container.getServerConfig().isFullGabDisabled();
		var searchText = (this.searchText.getValue() || '').trim();

		// Do not load when we are doing a GAB load without a search text
		// and the admin has configured to not load the GAB in that case
		if ( folderType === 'gab' && fullGabDisabled && Ext.isEmpty(searchText) ) {
			this.viewPanel.getView().emptyText = '<div class="emptytext">' + this.emptyGridText + '</div>';
			this.addressBookStore.removeAll();
			return;
		}

		this.viewPanel.getView().emptyText = '<div class="emptytext">' + this.noResultsGridText + '</div>';

		var params = {
			subActionType: Zarafa.core.Actions['globaladdressbook'],
			entryid: selectedFolder.get('entryid'),
			store_entryid: selectedFolder.get('store_entryid'),
			restriction: Ext.applyIf({searchstring: searchText}, this.listRestriction),
			folderType: folderType
		};

		if (folderType === 'contacts' || folderType === 'sharedcontacts') {
			var folderId = selectedFolder.get("entryid");
			var unwrappedEntryID = Zarafa.core.EntryId.unwrapContactProviderEntryId(folderId);
			var folder = folderType === 'contacts' ?
				container.getHierarchyStore().getFolder(unwrappedEntryID) :
				container.getHierarchyStore().getFolder(folderId);
			var isSharedStore = folder.getMAPIStore().isSharedStore() || folder.getMAPIStore().isPublicStore();
			if (isSharedStore) {
				params["isSharedFolder"] = isSharedStore;
				params["sharedFolder"] = {
					store_entryid: folder.get("store_entryid")
				};
			}
		}

		this.addressBookStore.load({
			actionType: Zarafa.core.Actions['list'],
			params: params
		});
	},

	/**
	 * Returns the currently selected folder in the hierarchy combobox
	 *
	 *@return {Zarafa.core.data.IPMRecord IPMRecord} The selected record from the
	 * {#hierarchyStore}
	 */
	getSelectedFolderRecord: function()
	{
		var entryid = this.addressBookSelectionCB.getValue();
		var index = Zarafa.addressbook.AddressBookHierarchyStore.find('entryid', entryid);
		return Zarafa.addressbook.AddressBookHierarchyStore.getAt(index);
	},

	/**
	 * Event handler which is triggered when
	 * a key is pressed in the searchTextField
	 *
	 * @param {Ext.form.TextField} field
	 * @param {Ext.EventObject} e
	 * @private
	 */
	onSearchTextFiledKeyUp: function(field, e)
	{
		if (e.getKey() === e.ENTER) {
			this.onSearchButtonClick();
		}
	}
});

Ext.reg('zarafa.addressbookmainpanel', Zarafa.addressbook.ui.AddressBookMainPanel);
Ext.namespace('Zarafa.advancesearch');

/**
 * @class Zarafa.advancesearch.Actions
 *
 * Advance search actions which can be used within {@link Ext.Button buttons}
 * or other {@link Ext.Component components} with action handlers.
 * @singleton
 */
Zarafa.advancesearch.Actions = {
	/**
	 * Open a {@link Zarafa.advancesearch.dialogs.SelectFolderContentPanel SelectFolderContentPanel} for
	 * select the {@link Zarafa.core.data.IPFRecord folder} on which search gets performs.
	 *
	 * @param {Object} config (optional) Configuration object to create the ContentPanel
	 */
	openSelectSearchFolderDialog: function(config)
	{
		config = Ext.applyIf(config || {}, {
			modal: true
		});
		var componentType = Zarafa.core.data.SharedComponentType['search.dialog.selectfolder'];
		Zarafa.core.data.UIFactory.openLayerComponent(componentType, [], config);
	},

	/**
	 * Open a dialog in which a new {@link Zarafa.common.favorites.data.FavoritesFolderRecord search} folder record can be
	 * further edited.
	 *
	 * @param {Zarafa.advancesearch.AdvanceSearchContextModel} model Context Model object that will be used
	 * to {@link Zarafa.advancesearch.AdvanceSearchContextModel#createSearchFolderRecord create} the search folder.
	 * @param {Object} config configuration options for {@link Zarafa.advancesearch.dialogs.CreateSearchFolderContentPanel CreateSearchFolderContentPanel}.
	 */
	openCreateSearchFolderContentPanel: function(model, config)
	{
		config = Ext.apply(config||{}, {
			modal: true,
			manager: Ext.WindowMgr,
			iconCls: 'icon_favorites',
			closable: false,
			resizable: false,
			showModalWithoutParent: true
		});

		var record = model.createSearchFolderRecord(config.searchText);

		Zarafa.core.data.UIFactory.openCreateRecord(record, config);
	},

	/**
	 * Open a {@link Zarafa.advancesearch.dialogs.SearchCategoriesContentPanel dialog}
	 * @param {Object} config configuration options.
	 */
	openSearchCategoryContentPanel: function (config)
	{
		var componentType = Zarafa.core.data.SharedComponentType['search.dialog.searchcategory'];
		Zarafa.core.data.UIFactory.openLayerComponent(componentType, [], config);
	}
};
Ext.namespace('Zarafa.advancesearch');

/**
 * @class Zarafa.advancesearch.KQLParser
 * @singleton
 * @extends Object
 */
Zarafa.advancesearch.KQLParser = Ext.extend(Object, {
	lexer: null,

	/**
	 * Creates a tokenizer with rules to split strings into KQL tokens.
	 * The tokenizer is an instance of Tokenizr.
	 * See {@link https://github.com/rse/tokenizr}
	 * @return {Tokenizr} The tokenizer
	 */
	getLexer: function() {
		if ( this.lexer ) {
			return this.lexer;
		}

		// eslint-disable-next-line no-undef
		this.lexer = new Tokenizr();

		var keys = [
			'body',
			'attachment',
			'subject',
			'sender',
			'from',
			'to',
			'cc',
			'bcc',
			'category'
		];
		var ops = [':', '=', '<>'];
		var boolOps = ['AND', 'OR', 'NOT', '+', '-'];

		var keyword = false;
		var operator = false;
		this.lexer.before(function(ctx, match, rule) {
			switch (rule.name) {
				case 'phrase':
				case 'word':
					if ( keyword === false ) {
						var op = match[1].toUpperCase();
					if ( boolOps.indexOf(op) > -1 ) {
						ctx.accept('operator', {op: op});
					} else {
						ctx.accept(rule.name, match[1]);
					}
					} else {
						// We found an expression
						ctx.accept('expression', {key: keyword, op: operator, val: match[1]});
						keyword = false;
						operator = false;
					}
					break;
			default:
				if ( keyword ) {
					ctx.accept('word', keyword);
					keyword = false;
					operator = false;
				}
			}
		});

		this.lexer.rule(/\(/, function(ctx, match) {
			ctx.accept('parenthesis', 'open');
			ctx.push('parens');
		}, 'parens');
		this.lexer.rule('parens', /\)/, function(ctx, match) {
			ctx.accept('parenthesis', 'close');
			ctx.pop();
		}, 'parensclose');
		this.lexer.rule(new RegExp('(' + keys.join('|') + ')(' + ops.join('|') + ')', 'i'), function(ctx, match) {
			ctx.ignore();
			keyword = match[1].toLowerCase();
			operator = match[2];
		}, 'keyword');
		this.lexer.rule(/"(.*?)"/, function(ctx, match) {
			ctx.ignore();
		}, 'phrase');
		this.lexer.rule(this.getWordRuleRegExp(), function(ctx, match) {
			ctx.ignore();
		}, 'word');
		this.lexer.rule(/[ \t\r\n]+/, function(ctx, match) {
			ctx.ignore();
		}, 'whitespace');
		// Ignore anything else
		this.lexer.rule(/./, function(ctx, match) {
			ctx.ignore();
		});

		return this.lexer;
	},

	/**
	 * Builds the Regular Expression that will be used by the lexer to find 'words'.
	 * It supports unicode characters, like letters with umlauts.
	 * @suppress {checkRegExp|suspiciousCode}
	 *
	 * @return {RegExp} A regular expression object
	 */
	getWordRuleRegExp: function() {
		// Test if we can use unicode regexps. (IE11 does not support it)
		// Otherwise we will match anything up till white space
		if ( typeof (/bla/).unicode === 'boolean' ) {
			var flags = 'iu';
			// Not all browsers that support unicode also support the \p{L} syntax,
			// so we will test for that too.
			try {
				// The following lines might look strange, but they written like this to
				// trick the closure compiler, because we can't seem to suppress the
				// checkRegExp and suspiousCode warnings.
				var f = 'ui';
				f = f.substr(0,1);
				var re = new RegExp('\\p{L}', f);
				re = '((?:\\p{L}|[a-z0-9_\\/@.!#$%&\'*+-=?^`{|}~])+)';
			} catch (e) {
				// The browser supports unicode, but not the shorthand syntax, so we must
				// explicitly tell which characters to match.
				// See https://stackoverflow.com/a/37668315 for more information on the following regexp
				re = '([a-z0-9\u00AA\u00B5\u00BA\u00C0-\u00D6\u00D8-\u00F6\u00F8-\u02C1\u02C6-\u02D1\u02E0-\u02E4\u02EC\u02EE\u0370-\u0374\u0376\u0377\u037A-\u037D\u037F\u0386\u0388-\u038A\u038C\u038E-\u03A1\u03A3-\u03F5\u03F7-\u0481\u048A-\u052F\u0531-\u0556\u0559\u0561-\u0587\u05D0-\u05EA\u05F0-\u05F2\u0620-\u064A\u066E\u066F\u0671-\u06D3\u06D5\u06E5\u06E6\u06EE\u06EF\u06FA-\u06FC\u06FF\u0710\u0712-\u072F\u074D-\u07A5\u07B1\u07CA-\u07EA\u07F4\u07F5\u07FA\u0800-\u0815\u081A\u0824\u0828\u0840-\u0858\u08A0-\u08B4\u0904-\u0939\u093D\u0950\u0958-\u0961\u0971-\u0980\u0985-\u098C\u098F\u0990\u0993-\u09A8\u09AA-\u09B0\u09B2\u09B6-\u09B9\u09BD\u09CE\u09DC\u09DD\u09DF-\u09E1\u09F0\u09F1\u0A05-\u0A0A\u0A0F\u0A10\u0A13-\u0A28\u0A2A-\u0A30\u0A32\u0A33\u0A35\u0A36\u0A38\u0A39\u0A59-\u0A5C\u0A5E\u0A72-\u0A74\u0A85-\u0A8D\u0A8F-\u0A91\u0A93-\u0AA8\u0AAA-\u0AB0\u0AB2\u0AB3\u0AB5-\u0AB9\u0ABD\u0AD0\u0AE0\u0AE1\u0AF9\u0B05-\u0B0C\u0B0F\u0B10\u0B13-\u0B28\u0B2A-\u0B30\u0B32\u0B33\u0B35-\u0B39\u0B3D\u0B5C\u0B5D\u0B5F-\u0B61\u0B71\u0B83\u0B85-\u0B8A\u0B8E-\u0B90\u0B92-\u0B95\u0B99\u0B9A\u0B9C\u0B9E\u0B9F\u0BA3\u0BA4\u0BA8-\u0BAA\u0BAE-\u0BB9\u0BD0\u0C05-\u0C0C\u0C0E-\u0C10\u0C12-\u0C28\u0C2A-\u0C39\u0C3D\u0C58-\u0C5A\u0C60\u0C61\u0C85-\u0C8C\u0C8E-\u0C90\u0C92-\u0CA8\u0CAA-\u0CB3\u0CB5-\u0CB9\u0CBD\u0CDE\u0CE0\u0CE1\u0CF1\u0CF2\u0D05-\u0D0C\u0D0E-\u0D10\u0D12-\u0D3A\u0D3D\u0D4E\u0D5F-\u0D61\u0D7A-\u0D7F\u0D85-\u0D96\u0D9A-\u0DB1\u0DB3-\u0DBB\u0DBD\u0DC0-\u0DC6\u0E01-\u0E30\u0E32\u0E33\u0E40-\u0E46\u0E81\u0E82\u0E84\u0E87\u0E88\u0E8A\u0E8D\u0E94-\u0E97\u0E99-\u0E9F\u0EA1-\u0EA3\u0EA5\u0EA7\u0EAA\u0EAB\u0EAD-\u0EB0\u0EB2\u0EB3\u0EBD\u0EC0-\u0EC4\u0EC6\u0EDC-\u0EDF\u0F00\u0F40-\u0F47\u0F49-\u0F6C\u0F88-\u0F8C\u1000-\u102A\u103F\u1050-\u1055\u105A-\u105D\u1061\u1065\u1066\u106E-\u1070\u1075-\u1081\u108E\u10A0-\u10C5\u10C7\u10CD\u10D0-\u10FA\u10FC-\u1248\u124A-\u124D\u1250-\u1256\u1258\u125A-\u125D\u1260-\u1288\u128A-\u128D\u1290-\u12B0\u12B2-\u12B5\u12B8-\u12BE\u12C0\u12C2-\u12C5\u12C8-\u12D6\u12D8-\u1310\u1312-\u1315\u1318-\u135A\u1380-\u138F\u13A0-\u13F5\u13F8-\u13FD\u1401-\u166C\u166F-\u167F\u1681-\u169A\u16A0-\u16EA\u16F1-\u16F8\u1700-\u170C\u170E-\u1711\u1720-\u1731\u1740-\u1751\u1760-\u176C\u176E-\u1770\u1780-\u17B3\u17D7\u17DC\u1820-\u1877\u1880-\u18A8\u18AA\u18B0-\u18F5\u1900-\u191E\u1950-\u196D\u1970-\u1974\u1980-\u19AB\u19B0-\u19C9\u1A00-\u1A16\u1A20-\u1A54\u1AA7\u1B05-\u1B33\u1B45-\u1B4B\u1B83-\u1BA0\u1BAE\u1BAF\u1BBA-\u1BE5\u1C00-\u1C23\u1C4D-\u1C4F\u1C5A-\u1C7D\u1CE9-\u1CEC\u1CEE-\u1CF1\u1CF5\u1CF6\u1D00-\u1DBF\u1E00-\u1F15\u1F18-\u1F1D\u1F20-\u1F45\u1F48-\u1F4D\u1F50-\u1F57\u1F59\u1F5B\u1F5D\u1F5F-\u1F7D\u1F80-\u1FB4\u1FB6-\u1FBC\u1FBE\u1FC2-\u1FC4\u1FC6-\u1FCC\u1FD0-\u1FD3\u1FD6-\u1FDB\u1FE0-\u1FEC\u1FF2-\u1FF4\u1FF6-\u1FFC\u2071\u207F\u2090-\u209C\u2102\u2107\u210A-\u2113\u2115\u2119-\u211D\u2124\u2126\u2128\u212A-\u212D\u212F-\u2139\u213C-\u213F\u2145-\u2149\u214E\u2183\u2184\u2C00-\u2C2E\u2C30-\u2C5E\u2C60-\u2CE4\u2CEB-\u2CEE\u2CF2\u2CF3\u2D00-\u2D25\u2D27\u2D2D\u2D30-\u2D67\u2D6F\u2D80-\u2D96\u2DA0-\u2DA6\u2DA8-\u2DAE\u2DB0-\u2DB6\u2DB8-\u2DBE\u2DC0-\u2DC6\u2DC8-\u2DCE\u2DD0-\u2DD6\u2DD8-\u2DDE\u2E2F\u3005\u3006\u3031-\u3035\u303B\u303C\u3041-\u3096\u309D-\u309F\u30A1-\u30FA\u30FC-\u30FF\u3105-\u312D\u3131-\u318E\u31A0-\u31BA\u31F0-\u31FF\u3400-\u4DB5\u4E00-\u9FD5\uA000-\uA48C\uA4D0-\uA4FD\uA500-\uA60C\uA610-\uA61F\uA62A\uA62B\uA640-\uA66E\uA67F-\uA69D\uA6A0-\uA6E5\uA717-\uA71F\uA722-\uA788\uA78B-\uA7AD\uA7B0-\uA7B7\uA7F7-\uA801\uA803-\uA805\uA807-\uA80A\uA80C-\uA822\uA840-\uA873\uA882-\uA8B3\uA8F2-\uA8F7\uA8FB\uA8FD\uA90A-\uA925\uA930-\uA946\uA960-\uA97C\uA984-\uA9B2\uA9CF\uA9E0-\uA9E4\uA9E6-\uA9EF\uA9FA-\uA9FE\uAA00-\uAA28\uAA40-\uAA42\uAA44-\uAA4B\uAA60-\uAA76\uAA7A\uAA7E-\uAAAF\uAAB1\uAAB5\uAAB6\uAAB9-\uAABD\uAAC0\uAAC2\uAADB-\uAADD\uAAE0-\uAAEA\uAAF2-\uAAF4\uAB01-\uAB06\uAB09-\uAB0E\uAB11-\uAB16\uAB20-\uAB26\uAB28-\uAB2E\uAB30-\uAB5A\uAB5C-\uAB65\uAB70-\uABE2\uAC00-\uD7A3\uD7B0-\uD7C6\uD7CB-\uD7FB\uF900-\uFA6D\uFA70-\uFAD9\uFB00-\uFB06\uFB13-\uFB17\uFB1D\uFB1F-\uFB28\uFB2A-\uFB36\uFB38-\uFB3C\uFB3E\uFB40\uFB41\uFB43\uFB44\uFB46-\uFBB1\uFBD3-\uFD3D\uFD50-\uFD8F\uFD92-\uFDC7\uFDF0-\uFDFB\uFE70-\uFE74\uFE76-\uFEFC\uFF21-\uFF3A\uFF41-\uFF5A\uFF66-\uFFBE\uFFC2-\uFFC7\uFFCA-\uFFCF\uFFD2-\uFFD7\uFFDA-\uFFDC_\\/@.!#$%&\'*+-=?^`{|}~]+)';
			}
		} else {
			re = '([^\t\r\n\s]+)';
			flags = 'i';
		}

		return new RegExp(re, flags);
	},

	/**
	 * Helper function to tokenize a string without explicitly
	 * creating a {@link getLexer lexer}
	 * @param {String} query
	 * @return {Array|Boolean} An array of token objects or false when no KQL tokens were found
	 */
	tokenize: function(query) {
		var lexer = this.getLexer();

		lexer.input(query);

		// Turn off debugging of the tokenizr library
		// Turn it on when needed during development
		lexer.debug(false);

		var tokens = lexer.tokens();

		if ( !this.isKqlQuery(tokens) ) {
			return false;
		}

		return tokens;
	},

	/**
	 * Scans the tokens to see if the tokenized query can be considered a KQL query.
	 * It is regarded as one when an expression or operator is found.
	 *
	 * @param {String|Array} tokens The search query, either tokenized or just a plain string
	 * @return {Boolean} True if the query is a KQL query, false otherwise
	 */
	isKqlQuery: function(tokens) {
		if ( typeof tokens === 'string' ) {
			return Boolean(this.tokenize(tokens));
		}

		if ( !Array.isArray(tokens) ) {
			return false;
		}


		tokens = this.normalize(tokens);

		var expressionOrOperatorFound = tokens.some(function(t) {
			return typeof t === 'object' && (t.type === 'expression' || t.type === 'subquery' || t.type === 'operator');
		});

		return expressionOrOperatorFound;
	},

	usesExplicitSyntax: function(tokens) {
		if ( typeof tokens === 'string' ) {
			tokens = this.tokenize(tokens);
		}

		if ( !Array.isArray(tokens) ) {
			return false;
		}

		return tokens.some(function(token) {
			if (!token || typeof token !== 'object') {
				return false;
			}

			if (token.type === 'operator') {
				var op = token.value && token.value.op;
				return op === 'AND' || op === 'OR' || op === 'NOT' || op === '+' || op === '-';
			}

			if (token.type === 'expression') {
				return true;
			}

			if (token.type === 'word' && typeof token.value === 'string') {
				return /^[+-]/.test(token.value);
			}

			return false;
		});
	},

	/**
	 * Normalize the tokens if necessary
	 * We want something like <expression> <operator> <expression> <operator> etc
	 * but users are allowed to omit the AND between expressions and NOT is a special
	 * operator that needs to be preceded by another operator.
	 * Query parts within parenthesis will be moved into a so-called 'subquery' token
	 *
	 * @param {Object[]} Array of token objects
	 */
	normalize: function(tokens) {
		// make sure we don't change the original tokens
		tokens = JSON.parse(JSON.stringify(tokens));

		// Convert plain words or quoted phrases into expressions that
		// search across all indexed fields. This allows queries like
		// "foo AND bar" without specifying a field name.
		tokens = tokens.map(function(token) {
			if ( token.type === 'word' || token.type === 'phrase' ) {
				return {
					type: 'expression',
					value: {key: 'any', op: ':', val: token.value}
				};
			}
			return token;
		});

		var t = [];
		var lastToken = null;
		while (tokens.length) {
			var token = tokens.shift();

			// First rewrite + to AND and - to NOT
			if ( token.type === 'operator') {
				if ( token.value.op === '+' ) {
					token.value.op = 'AND';
				} else if ( token.value.op === '-' ) {
					token.value.op = 'NOT';
				}
			}

			if ( token.type === 'operator' && token.value.op !== 'NOT' ) {
				if ( lastToken && (lastToken.type === 'expression' || lastToken.type === 'subquery') ) {
					// This is fine, add the token
					t.push(token);
				} else {
					// This is wrong. We'll should probably generate an error
					// Another option is to just drop this token. Let's do this
					// for now.
				}
			} else if ( token.type === 'operator' ) { // This is a NOT
				if ( lastToken && (lastToken.type === 'expression' || lastToken.type === 'subquery' || lastToken.type === 'parenthesis') ) {
					// An expression followed by a negated expression is regarded as joined
					// by an 'AND', so first push that
					t.push({type:'operator', value: {op: 'AND'}});
				}
				// We have a NOT, so we need an expression next and no NOT before this.
				if ( tokens.length && (tokens[0].type === 'operator' && tokens[0].value.op === 'NOT' ) ) {
					// Drop this NOT and the next one
					t.pop();
					tokens.shift();
				} else if ( tokens.length && (tokens[0].type === 'expression' || tokens[0].type === 'subquery' || (tokens[0].type === 'parenthesis' && tokens[0].value === 'open')) ) {
					tokens[0].negate = true;
				}
			} else if ( (token.type === 'expression' || token.type === 'subquery') ) {
				if ( lastToken && (lastToken.type === 'expression' || lastToken.type === 'subquery') ) {
					// An expression or subquery followed by an expression or subquery is
					// regarded as joined by an 'AND', so first push that
					t.push({type:'operator', value: {op: 'AND'}});
				}
				t.push(token);
			} else if ( token.type ==='parenthesis' && token.value === 'open' ) {
				// Find all tokens up till the closing parenthesis
				var subqueryTokens = [];
				var parensOpenCount = 1;
				while ( tokens.length && !(tokens[0].type === 'parenthesis' && tokens[0].value === 'close' && parensOpenCount === 1) ) {
					var tmpToken = tokens.shift();
					if ( tmpToken.type === 'parenthesis' && tmpToken.value === 'open' ) {
						parensOpenCount++;
					}
					if ( tmpToken.type === 'parenthesis' && tmpToken.value === 'close' ) {
						parensOpenCount--;
					}
					subqueryTokens.push(tmpToken);
				}
				if ( subqueryTokens.length === 0 || subqueryTokens[subqueryTokens.length -1].type === 'EOF') {
					// Something is wrong with the query! There is nothing in the parentheses or
					// we couldn't find the closing parenthesis. Let's just drop the subquery for now.
					// TODO: Show a message

				} else {
					// Create a subquery, but only if our subquery tokens contain more than one token.
					// Otherwise we can just drop the parenthesis
					subqueryTokens = this.normalize(subqueryTokens);
					if ( subqueryTokens.length > 1 ) {
						t.push({
							type: 'subquery',
							value: this.normalize(subqueryTokens),
							negate: token.negate
						});
					} else if (subqueryTokens.length === 1 ) {
						t.push(subqueryTokens[0]);
					}
				}
			}

			lastToken = t.length ? t[t.length - 1] : null;
		}

		// We can't end with an operator
		while ( t.length && t[t.length-1].type === 'operator' ) {
			t.pop();
		}

		return t;
	},

	/**
	 * Will process the tokens and move AND parts into subqueries.
	 * We now have tokens like <expression> OR <subquery> AND <expression>
	 * Since the AND takes presendence over OR we will move the AND with
	 * its expressions into a subquery so we will only have something like
	 * <expression> OR <subquery> OR <subquery>
	 * 					or
	 * <expression> AND <subquery> AND <subquery>
	 * (so only with the same operator)
	 * @param {Object[]} tokens Array of token objects
	 * @return {Object[]} Array of token objects
	 */
	flatten: function(tokens) {
		var pos = 0;
		var lastOperator;
		var lastOperatorPos;
		while ( pos < tokens.length ) {
			var token = tokens[pos];
			if ( token.type === 'subquery' ) {
				token.value = this.flatten(token.value);
			}
			if ( token.type === 'operator' && (token.value.op === 'AND' || token.value.op === 'OR') ) {
				if ( lastOperator && lastOperator !== token.value.op ) {
					// We must create a subquery
					if ( lastOperator === 'AND' ) {
						var subqueryTokens = tokens.splice(lastOperatorPos-1, pos+1);
						tokens.splice(lastOperatorPos-1, 0, {
							type: 'subquery',
							value: subqueryTokens
						});
						pos = pos - subqueryTokens.length + 1;
					} else {
						subqueryTokens = [];
						// eslint-disable-next-line max-depth
						while ( tokens[pos-1] && (!tokens[pos] || !(tokens[pos].type === 'operator' && tokens[pos].value === 'OR')) ) {
							subqueryTokens = subqueryTokens.concat(tokens.splice(pos-1, 1));
						}
						tokens.splice(pos-1, 0, {
							type: 'subquery',
							value: subqueryTokens
						});
						pos--;
					}
				} else {
					lastOperator = token.value.op;
					lastOperatorPos = pos;
				}
			}
			pos++;
		}

		return tokens;
	},

	/**
	 * Creates a restriction from a set of tokens
	 *
	 * @param {Array} tokens An array of token objects
	 * @return {Array} A Restriction array
	 */
	createTokenRestriction: function(tokens, anyFieldPreference) {
		tokens = this.flatten(this.normalize(tokens));

		var propMap = {
			subject: ['subject'],
			body: ['body'],
			sender: ['sender_name', 'sender_email_address'],
			from: ['sender_name', 'sender_email_address'],
			to: [{type:'recipient', recipientType: Zarafa.core.mapi.RecipientType.MAPI_TO}],
			cc: [{type:'recipient', recipientType: Zarafa.core.mapi.RecipientType.MAPI_CC}],
			bcc: [{type:'recipient', recipientType: Zarafa.core.mapi.RecipientType.MAPI_BCC}],
			category: ['categories'],
			attachment: [{type: 'attachment'}],
			// default search across multiple fields
			any: [
				'subject',
				'body',
				'sender_name',
				'sender_email_address',
				{type:'recipient', recipientType: Zarafa.core.mapi.RecipientType.MAPI_TO},
			{type:'recipient', recipientType: Zarafa.core.mapi.RecipientType.MAPI_CC},
			{type:'recipient', recipientType: Zarafa.core.mapi.RecipientType.MAPI_BCC},
			]
		};
		if (Ext.isArray(anyFieldPreference) && anyFieldPreference.length) {
			var anyFields = [];
			var seen = {};
			Ext.each(anyFieldPreference, function(field) {
				if (Ext.isEmpty(field)) {
					return;
				}
				var entry;
				switch (field) {
					case 'display_to':
						entry = {type:'recipient', recipientType: Zarafa.core.mapi.RecipientType.MAPI_TO};
						break;
					case 'display_cc':
						entry = {type:'recipient', recipientType: Zarafa.core.mapi.RecipientType.MAPI_CC};
						break;
					case 'display_bcc':
						entry = {type:'recipient', recipientType: Zarafa.core.mapi.RecipientType.MAPI_BCC};
						break;
					default:
						entry = field;
				}
				var key = Ext.isObject(entry) ? entry.type + ':' + entry.recipientType : entry;
				if (!seen[key]) {
					anyFields.push(entry);
					seen[key] = true;
				}
			});
			if (anyFields.length) {
				propMap.any = anyFields;
			}
		}
		var restrictions = [];

		// find the operator used in the restriction
		var op = tokens.filter(function(t) {
			return t.type === 'operator';
		});
		if ( op.length ) {
			op = op[0].value.op;
		} else {
			op = false;
		}

		tokens.forEach(function(token) {
			var curRes = [];
			if ( token.type === 'expression' ) {
				if ( !propMap[token.value.key] ) {
					return;
				}
				propMap[token.value.key].forEach(function(prop) {
					var r;
					if ( typeof prop === 'string' ) {
						r = this.createStringRestriction(prop, token.value.val);
					} else if ( typeof prop === 'object' && prop.type === 'recipient' ) {
						r = this.createRecipientRestriction(prop.recipientType, token.value.val);
					} else if ( typeof prop === 'object' && prop.type === 'attachment' ) {
						r = this.createAttachmentRestriction(token.value.val);
					}

					curRes.push(r);
				}, this);
			} else if ( token.type === 'subquery' ) {
				curRes.push(this.createTokenRestriction(token.value, anyFieldPreference));
			}

			if ( curRes.length === 0 ) {
				return;
			}

			if ( curRes.length > 1 ) {
				curRes = Zarafa.core.data.RestrictionFactory.createResOr(curRes);
			} else {
				curRes = curRes[0];
			}

			if ( token.negate ) {
				curRes = Zarafa.core.data.RestrictionFactory.createResNot(curRes);
			}

			restrictions.push(curRes);
		}, this);

		if ( restrictions.length === 0 ) {
			return false;
		}

		if ( op === 'OR' ) {
			restrictions = Zarafa.core.data.RestrictionFactory.createResOr(restrictions);
		} else {
			restrictions = Zarafa.core.data.RestrictionFactory.createResAnd(restrictions);
		}

		return restrictions;
	},

	/**
	 * Creates a text restriction on the given property
	 *
	 * @param {String} prop The property to set the restriction on
	 * @param {String} value The value the restriction will search for
	 * @return {Array} A Restriction array
	 */
	createStringRestriction: function(prop, value) {
		return Zarafa.core.data.RestrictionFactory.dataResContent(
			prop,
			Zarafa.core.mapi.Restrictions.FL_SUBSTRING | Zarafa.core.mapi.Restrictions.FL_IGNORECASE,
			value
		);
	},

	/**
	 * Creates a restriction on a recipient type
	 *
	 * @param {Number} recipientType The {@link Zarafa.core.mapi.RecipientType RecipientType}
	 * to put the restriction on
	 * @param {String} value The value the restriction will search for
	 * @return {Array} A Restriction array
	 */
	createRecipientRestriction: function(recipientType, value) {
		return Zarafa.core.data.RestrictionFactory.createResSubRestriction('PR_MESSAGE_RECIPIENTS',
			Zarafa.core.data.RestrictionFactory.createResAnd([
				Zarafa.core.data.RestrictionFactory.dataResProperty('PR_RECIPIENT_TYPE',
					Zarafa.core.mapi.Restrictions.RELOP_EQ,
					recipientType
				),
				Zarafa.core.data.RestrictionFactory.createResOr([
					this.createStringRestriction('PR_DISPLAY_NAME', value),
					this.createStringRestriction('PR_EMAIL_ADDRESS', value),
					this.createStringRestriction('PR_SMTP_ADDRESS', value)
				])
			])
		);
	},

	/**
	 * Creates a restriction on a attachment name
	 *
	 * @param {String} value The value the restriction will search for
	 * @return {Array} A Restriction array
	 */
	createAttachmentRestriction: function(value) {
		return Zarafa.core.data.RestrictionFactory.createResSubRestriction('PR_MESSAGE_ATTACHMENTS',
			this.createStringRestriction('PR_ATTACH_LONG_FILENAME', value)
		);
	}

});

Zarafa.advancesearch.KQLParser = new Zarafa.advancesearch.KQLParser();
Ext.namespace('Zarafa.advancesearch.data');

/**
 * @class Zarafa.advancesearch.data.DateRangeFields
 * Search date range fields for search toolBox
 * @singleton
 */
Zarafa.advancesearch.data.DateRangeFields = [{
    'name': _('Any date'),
    'value': 'all_dates'
}, {
    'name': _('Past week'),
    'value': 'past_week'
}, {
    'name': _('Past 2 weeks'),
    'value': 'past_two_weeks'
}, {
    'name': _('Past month'),
    'value': 'past_month'
}, {
    'name': _('Past 6 month'),
    'value': 'past_six_month'
}, {
    'name': _('Past year'),
    'value': 'past_year'
}, {
    'name': _('Custom date'),
    'value': 'custom_date'
}];
Ext.namespace('Zarafa.advancesearch.data');
/**
 * @class Zarafa.advancesearch.data.SearchCategoriesStore
 * @extends Ext.data.ArrayStore
 * @xtype zarafa.searchcategoriesstore
 *
 * Store which will bind with category filter
 * and manage categories which is added in category filter.
 */
Zarafa.advancesearch.data.SearchCategoriesStore = Ext.extend(Ext.data.ArrayStore, {

	/**
	 * @constructor
	 * @param {Object} config Configuration structure
	 */
	constructor: function (config)
	{
		config = config || {};

		Ext.applyIf(config, {
			fields: ['name', 'backgroundColor', 'colorClass'],
			autoDestroy: true
		});

		Ext.apply(this, config);

		Zarafa.advancesearch.data.SearchCategoriesStore.superclass.constructor.call(this, config);
	},

	/**
	 * Helper function which is use to add categories in to this store.
	 * @param {Array} categories list of categories which will add in category filter.
	 */
	addCategories: function (categories)
	{
		categories.forEach(function (category) {
			if (this.findExactCaseInsensitive('name', category) === -1) {
				// Add the category to the store
				var backgroundColor = Zarafa.common.categories.Util.getCategoryColor(category);
				this.add(new this.recordType({
					name: category,
					backgroundColor: Zarafa.common.categories.Util.getCategoryColor(category),
					colorClass: Zarafa.core.ColorSchemes.getLuma(backgroundColor) < 200 ? 'zarafa-dark' : ''
				}));
			}
		}, this);
	},

	/**
	 * Helper function which is use to remove categories from this store.
	 * @param {Array} categories list of categories which will remove from category filter.
	 */
	removeCategories: function (categories)
	{
		categories.forEach(function (category) {
			this.removeAt(this.findExactCaseInsensitive('name', category));
		}, this);
	},

	/**
	 * Finds the index of the first matching Record in this store by a specific field value.
	 * Matches case-insensitive.
	 * @param {String} fieldName The field that should be used to match the record
	 * @param {String} value The value that should be used to match the records
	 *
	 * @return {Number} The index of the first matching record or -1 if no match was found
	 */
	findExactCaseInsensitive: function (fieldName, value)
	{
		return this.findBy(function (category, index) {
			return category.get(fieldName).toLowerCase() === value.toLowerCase();
		}, this);
	},

	/**
	 * Helper function which is use to get list of categories.
	 * @returns {Array} list of categories which is added in category filter.
	 */
	getCategories: function ()
	{
		var categories = [];
		this.each(function (record) {
			categories.push(record.get('name'));
		});
		return categories;
	}
});

Ext.reg('zarafa.searchcategoriesstore', Zarafa.advancesearch.data.SearchCategoriesStore);
Ext.namespace('Zarafa.advancesearch.dialogs');

/**
 * @class Zarafa.advancesearch.dialogs.CreateSearchFolderPanel
 * @extends Ext.form.FormPanel
 * @xtype zarafa.createsearchfolderpanel
 *
 * Panel that is used to create new search folder under the favorites folder list.
 */
Zarafa.advancesearch.dialogs.CreateSearchFolderPanel = Ext.extend(Ext.form.FormPanel, {

	/**
	 * @constructor
	 * @param {Object} config Configuration object
	 */
	constructor: function(config)
	{
		config = config || {};

		Ext.applyIf(config, {
			cls: 'k-create-search-folder-panel',
			border: false,
			items: [{
				xtype: 'displayfield',
				value: _('Add a folder to favorites based on search query'),
				hideLabel: true,
				margins: '0 0 9 0',
				anchor: '100%'
			}, {
				xtype: 'container',
				layout: 'form',
				labelAlign: 'top',
				items:[{
					xtype: 'textfield',
					ref: '../../searchFolderTextField',
					emptyText: _('Folder name'),
					labelSeparator: '',
					anchor: '100%'
				}]
			}]
		});

		Zarafa.advancesearch.dialogs.CreateSearchFolderPanel.superclass.constructor.call(this, config);
	}
});

Ext.reg('zarafa.createsearchfolderpanel', Zarafa.advancesearch.dialogs.CreateSearchFolderPanel);
Ext.namespace('Zarafa.advancesearch.dialogs');

/**
 * @class Zarafa.advancesearch.dialogs.SearchCenterPanel
 * @extends Ext.Panel
 * @xtype zarafa.searchcenterpanel
 *
 */
Zarafa.advancesearch.dialogs.SearchCenterPanel = Ext.extend(Ext.Panel, {

	/**
	 * @constructor
	 * @param {Object} config configuration object.
	 */
	constructor: function (config)
	{
		config = config || {};

		if (!Ext.isDefined(config.model) && Ext.isDefined(config.searchContext)) {
			config.model = config.searchContext.getModel();
		}

		Ext.applyIf(config, {
			xtype: 'zarafa.searchcenterpanel',
			cls: 'k-searchcenterpanel',
			region: 'center',
			layout: 'fit',
			ref: 'centerRegion',
			unstyled: true,
			items: [{
				xtype: 'panel',
				layout: 'zarafa.switchborder',
				ref: 'switchBorder',
				border: false,
				unstyled: true,
				items: [{
					layout: 'fit',
					cls: 'zarafa-context-mainpanel',
					collapsible: false,
					region: 'center',
					items: [{
						xtype: 'zarafa.switchviewcontentcontainer',
						ref: '../viewPanel',
						layout: 'card',
						activeItem: 0,
						items: [{
							xtype: 'zarafa.searchgrid',
							flex: 1,
							id: 'search-grid' + (++Ext.Component.AUTO_ID),
							searchTabId: config.searchTabId,
							anchor: '100%',
							searchContext: config.searchContext,
							ref: '../../searchGrid',
							searchCenterPanel: this
						}]
					}]
				},{
					region: 'south',
					xtype: 'zarafa.searchresultpreviewpanel',
					ref: '../searchResultPreviewPanel',
					split: true,
					width: 400,
					height: 400,
					searchContext: config.searchContext
				}]
			}]
		});

		Zarafa.advancesearch.dialogs.SearchCenterPanel.superclass.constructor.call(this, config);
	},

	/**
	 * Function called by Extjs when the panel has been {@link #render rendered}.
	 * At this time all events can be registered.
	 * @private
	 */
	initEvents: function ()
	{
		if (Ext.isDefined(this.searchContext)) {
			this.switchBorder.mon(this.searchContext,{
				viewchange: this.onViewChange,
				viewmodechange: this.onViewModeChange,
				scope: this
			});

			this.switchBorder.on('afterlayout', this.onAfterLayout, this, {single: true});
		}
	},

	/**
	 * Function is used to get the {@link Zarafa.advancesearch.ui.SearchResultPreviewPanel searchResultPreviewPanel}
	 * @return {Object } return {@link Zarafa.advancesearch.ui.SearchResultPreviewPanel searchResultPreviewPanel}
	 */
	getSearchResultPreviewPanel: function ()
	{
		return this.searchResultPreviewPanel;
	},

	/**
	 * Event handler triggered when {@link Zarafa.core.ui.SwitchViewContentContainer Switch view content container}
	 * layout has been initialized
	 */
	onAfterLayout: function ()
	{
		this.onViewModeChange(this.searchContext, this.searchContext.getCurrentViewMode());
	},

	/**
	 * Event handler which is fired when the currently active view inside the {@link #context}
	 * has been updated. This will update the call
	 * {@link #viewPanel}#{@link Zarafa.core.ui.SwitchViewContentContainer#switchView}
	 * to make the requested view active.
	 *
	 * @param {Zarafa.core.Context} context The context which fired the event.
	 * @param {Zarafa.common.data.Views} newView The ID of the selected view.
	 * @param {Zarafa.common.data.Views} oldView The ID of the previously selected view.
	 */
	onViewChange: function (context, newView, oldView)
	{
		if(newView === Zarafa.common.data.Views.LIST) {
			var searchGridId = this.switchBorder.searchGrid.getId();
			this.switchBorder.viewPanel.switchView(searchGridId);
		}
	},

	/**
	 * Event handler which is fired when the {@link Zarafa.core.Context} fires the
	 * {@link Zarafa.core.Context#viewmodechange viewmodechange} event. This will
	 * convert the configured {@link Zarafa.common.data.ViewModes mode} to a
	 * {@link Zarafa.common.ui.layout.SwitchBorderLayout.Orientation orientation}
	 * to be {@link Zarafa.common.ui.layout.SwitchBorderLayout.setOrientation applied}
	 * to the {@link #layout}.
	 * @param {Zarafa.core.Context} context The context which fired the event
	 * @param {Zarafa.common.data.ViewModes} newViewMode The new active mode
	 * @param {Zarafa.common.data.ViewModes} oldViewMode The previous mode
	 * @private
	 */
	onViewModeChange: function (context, newViewMode, oldViewMode)
	{
		var orientation;

		switch (newViewMode) {
			case Zarafa.common.data.ViewModes.NO_PREVIEW:
				orientation = Zarafa.common.ui.layout.SwitchBorderLayout.Orientation.OFF;
				break;
			case Zarafa.common.data.ViewModes.RIGHT_PREVIEW:
				orientation = Zarafa.common.ui.layout.SwitchBorderLayout.Orientation.HORIZONTAL;
				// hide the toolbar when right preview panel is enabled.
				this.getSearchResultPreviewPanel().getTopToolbar().onHide();
				break;
			case Zarafa.common.data.ViewModes.BOTTOM_PREVIEW:
				orientation = Zarafa.common.ui.layout.SwitchBorderLayout.Orientation.VERTICAL;
				// show the toolbar when bottom preview panel is enabled and at lest one record
				// is selected in search grid.
				var record = this.model.getPreviewRecord();
				if (Ext.isDefined(record)) {
					this.getSearchResultPreviewPanel().getTopToolbar().onShow();
				}
				break;
			case Zarafa.common.data.ViewModes.SEARCH:
			case Zarafa.common.data.ViewModes.LIVESCROLL:
				return;
		}

		// This function could be called when the layout has not yet
		// been instantiated. In that case we update the layoutConfig
		// so it will be automatically picked up by the layout when
		// it needs it.
		var layout = this.switchBorder.getLayout();
		if (!Ext.isFunction(layout.setOrientation)) {
			if (Ext.isString(layout)) {
				this.layoutConfig = Ext.apply(this.layoutConfig || {}, {orientation: orientation});
			} else {
				this.layout.orientation = orientation;
			}
		} else {
			layout.setOrientation(orientation);
		}
	}
});

Ext.reg('zarafa.searchcenterpanel', Zarafa.advancesearch.dialogs.SearchCenterPanel);

Ext.namespace('Zarafa.advancesearch.dialogs');

/**
 * @class Zarafa.advancesearch.dialogs.SearchPanel
 * @extends Ext.Panel
 * @xtype zarafa.searchpanel
 *
 */
Zarafa.advancesearch.dialogs.SearchPanel = Ext.extend(Ext.Panel, {

	/**
	 * @cfg{Object} searchContentPanel which contains {@link Zarafa.advancesearch.dialogs.SearchPanel searchPanel}
	 */
	searchContentPanel: undefined,

	/**
	 * @constructor
	 * @param {Object} config configuration object.
	 */
	constructor: function(config)
	{
		config = config || {};

		var searchContext = container.getContextByName('advancesearch');
		if (!Ext.isDefined(config.model) && Ext.isDefined(searchContext)) {
			config.model = searchContext.getModel();
			if(Ext.isDefined(config.searchFolder)) {
				config.model.store.searchFolder[config.searchTabId] = config.searchFolder;
			}
			var parentModel = container.getCurrentContext().getModel();
			searchContext.enable(parentModel.getDefaultFolder(), true);
		}

		Ext.applyIf(config, {
			xtype: 'zarafa.searchpanel',
			layout: 'border',
			border: false,
			items: [{
				xtype: 'zarafa.searchtoolboxpanel',
				searchContext: searchContext,
				searchTabId: config.searchTabId,
				collapsed: Ext.isDefined(config.searchFolder),
				region:'west',
				scope: this
			},{
				xtype: 'zarafa.searchcenterpanel',
				searchContext: searchContext,
				searchTabId: config.searchTabId,
				region: 'center',
				searchPanel: this,
				scope: this
			},{
				xtype: 'zarafa.searchtoolbarpanel',
				searchContext: searchContext,
				searchText: config.searchText,
				region: 'north',
				scope: this
			}]
		});

		Zarafa.advancesearch.dialogs.SearchPanel.superclass.constructor.call(this, config);
	},

	/**
	 * Function called by Extjs when the panel has been {@link #render rendered}.
	 * At this time all events can be registered.
	 * @private
	 */
	initEvents: function()
	{
		this.mon(this.centerRegion.getSearchResultPreviewPanel(),{
			afterupdatesearchpreviewpanel: this.onAfterUpdateSearchPreviewPanel,
			scope: this
		});

		this.mon(this.searchContentPanel,{
			beforeclose: this.onBeforeCloseContentPanel,
			scope: this
		});

		// Events registered on Search grid.
		this.mon(this.centerRegion.switchBorder.searchGrid,{
			resize: this.onSearchGridResize,
			scope: this
		});

		this.mon(this.searchToolBox,{
			afterupdaterestriction: this.onAfterUpdateRestriction,
			scope: this
		});

		this.mon(this.searchToolBox, 'afterlayout',this.onAfterRenderSearchToolBox, this, {single: true});

		this.mon(this.model, {
			searchfinished: this.onModelSearchFinished,
			searchexception: this.onModelSearchException,
			scope: this
		});

		this.mon(this.searchToolBox.includeSubFolderFieldSet, {
			beforerender: this.onBeforeRenderSubFolderFieldSet,
			scope: this
		});

		this.mon(this.searchToolBox.includeSubFolder, {
			check: this.onCheckIncludeSubFolder,
			render: this.onRenderSubFolderCheckbox,
			scope: this
		});

		// Events registered on Advance Search field.
		this.searchToolbar.mon(this.searchToolbar.getAdvanceSearchField(),{
			render: this.onRenderSearchTextField,
			valid: this.onValidSearchTextField,
			change: this.onChangeSearchTextField,
			start: this.onSearchStart,
			stop: this.onSearchStop,
			scope: this
		});

		this.searchToolbar.mon(this.searchToolbar.getSearchFolderCombo(),{
			render: this.onRenderSearchFolderCombo,
			select: this.onSelectSearchFolderComboValue,
			beforeselect: this.onBeforeSelectSearchFolder,
			scope: this
		});

		this.mon(container,{
			 'aftercontextswitch': this.onAfterContextSwitch,
			 scope: this
		});

		this.mon(container.getHierarchyStore(), 'addFolder', this.onHierarchyAddFolder, this);
		this.mon(container.getHierarchyStore(), 'removeFolder', this.onHierarchyRemoveFolder, this);
	},

	/**
	 * Event handler triggers when folder was added in hierarchy. function was
	 * responsible to save the search criteria in settings.
	 *
	 * @param {Zarafa.hierarchy.data.HierarchyStore} store The store which fired the event
	 * @param {Zarafa.hierarchy.data.MAPIStoreRecord} mapiStore mapi store in which new folders are added.
	 * @param {Zarafa.hierarchy.data.MAPIFolderRecord/Zarafa.hierarchy.data.MAPIFolderRecord[]} record folder record(s) which are added in hierarchy.
	 * @private
	 */
	onHierarchyAddFolder: function(store, mapiStore, record)
	{
		if (Ext.isArray(record)) {
			record.forEach(function (record) {
				this.onHierarchyAddFolder(store, mapiStore, record);
			},this);
			return;
		}
		var searchStore = this.model.getActiveStore();
		var searchStoreUniqueId = searchStore.searchStoreUniqueId;
		var searchToolBox = this.searchToolBox;
		if (record.isSearchFolder() && searchStoreUniqueId === searchToolBox.searchTabId) {
			searchStore.searchFolder[searchStore.searchStoreUniqueId] = record;

			this.suspendEvents();
			var obj = {};
			var searchInCheckBoxGroup = searchToolBox.searchInCheckboxGroup.getValue();
			var messageTypeCheckBoxGroup = searchToolBox.messageTypeCheckboxGroup.getValue();
			var filterCheckBoxGroup = searchToolBox.filterCheckBoxGroup.getValue();
			var dateRangeDropdown = searchToolBox.dateRangeCombo;

			obj['searchInCheckBoxGroup'] = {};
			obj['messageTypeCheckBoxGroup'] = {};
			obj['filterCheckBoxGroup'] = {};
			obj["search_text"] = this.searchText;
			obj["search_folder_combo"] = {};
			obj['categories'] = searchToolBox.searchCategoriesStore.getCategories();

			searchInCheckBoxGroup.forEach(function(item){
				obj['searchInCheckBoxGroup'][item.itemId] = true;
			}, this);

			messageTypeCheckBoxGroup.forEach(function(item){
				obj['messageTypeCheckBoxGroup'][item.name] = true;
			}, this);

			filterCheckBoxGroup.forEach(function(item){
				obj['filterCheckBoxGroup'][item.name] = true;
			}, this);

			if (dateRangeDropdown.getValue() === 'custom_date') {
				obj["date_range"] = {};
				Ext.apply(obj["date_range"], {
					start: searchToolBox.dateField.getValue().getStartDate().getTime(),
					due: searchToolBox.dateField.getValue().getDueDate().getTime()
				});
			} else {
				obj["date_range"] = dateRangeDropdown.getValue();
			}

			var searchFolderCombo = this.searchToolbar.getSearchFolderCombo();
			var folder = searchFolderCombo.getStore().getAt(searchFolderCombo.getStore().find('value',searchFolderCombo.getValue()));
			if(folder.get('flag') === Zarafa.advancesearch.data.SearchComboBoxFieldsFlags.ALL_FOLDERS) {
				obj["search_folder_combo"]["folder_type"] = folder.get('flag');
			} else {
				obj["search_folder_combo"]["folder_entryid"] = folder.get('value');
				obj["search_folder_combo"]["folder_name"] = folder.get('name');
				obj["search_folder_combo"]["folder_type"] = folder.get('flag');
				obj["search_folder_combo"]["include_subfolder"] = folder.get('include_subfolder');
			}
			container.getSettingsModel().set('zarafa/v1/contexts/search/search_criteria/'+record.get('entryid'), obj);
			this.resumeEvents();
		}
	},

	/**
	 * Event handler triggers when folder was remove in hierarchy. function was
	 * responsible to verify that search tab is still exist of folder which was removed.
	 * If search tab is still open then set 'keepSearchFolder' flags true.
	 *
	 * @param {Zarafa.hierarchy.data.HierarchyStore} store The store which fired the event
	 * @param {Zarafa.hierarchy.data.MAPIStoreRecord} mapiStore mapi store in which new folders was removed.
	 * @param {Zarafa.hierarchy.data.MAPIFolderRecord/Zarafa.hierarchy.data.MAPIFolderRecord[]} record folder record(s) which was removed in hierarchy.
	 * @private
	 */
	onHierarchyRemoveFolder: function (store, mapiStore, record)
	{
		var searchStore = this.model.getActiveStore();
		var searchStoreRecord = searchStore.searchFolder[this.searchToolBox.searchTabId];
		if (searchStoreRecord && Zarafa.core.EntryId.compareEntryIds(searchStoreRecord.get('entryid'), record.get('entryid'))) {
			record.addMessageAction('keepSearchFolder', true);
		}
	},

	/**
	 * Event handler triggers when context was switch successfully.
	 * It was set the parent search field because after the switching context
	 * search field of parent context gets destroy so we need to update the parent
	 * search field object and also required to {#reinitializeEvents}
	 *
	 * @param {Object} folders contains folder details
	 * @param {Context} oldParentContext previously selected context
	 * @param {Context} newParentContext selected context
	 */
	onAfterContextSwitch: function(folders, oldParentContext, newParentContext)
	{
		var activeItem = container.getTabPanel().getActiveTab().getActiveItem();
		var topToolbar = activeItem.getTopToolbar();
		var parentSearchField;

		if (Ext.isDefined(topToolbar) && Ext.isDefined(parentSearchField = topToolbar.searchTextfield)) {
			this.searchContentPanel.setParentSearchField(parentSearchField);
		} else {
			this.searchContentPanel.setParentSearchField(undefined);
		}
	},

	/**
	 * Event handler triggered before {@link Zarafa.advancesearch.dialogs.SearchContentPanel searchcontentpanel}
	 * gets close. it will call the {#onSearchStop} which stop the search.
	 * @param {Zarafa.advancesearch.dialogs.SearchContentPanel} contentPanel which gets the close
	 */
	onBeforeCloseContentPanel: function(contentPanel)
	{
		this.resetParentSearchField();
		var parentSearchField = this.searchContentPanel.getParentSearchField();
		if (parentSearchField) {
			parentSearchField.searchPanelRendered = false;
		}

		/**
		 * when user tries to close the search tab which is not currently active tab,
		 * it will find the closing search tab store and set it to active store in model, so when stopSearch
		 * function of model is call to remove the search folder on server side it will send correct search folder
		 * entry id
		 */
		var currentSearchStore = false;
		var model = this.model;
		if (model.getActiveStore().getSearchStoreUniqueId() !== contentPanel.name){
			currentSearchStore = this.model.store;
			var store = model.stores[contentPanel.name];
			model.setActiveStore(store);
		}

		this.onSearchStop();

		/**
		 * Model has hold the selected record in Zarafa.core.ContextModel.selectedRecords config
		 * so when close the search panel we have to clear the selection manually.
		 */
		if (Ext.isDefined(model.getSelectedRecords())) {
			model.setSelectedRecords(undefined);
		}

		// search tab is going to close so remove the search store from stores object
		model.discardStore(contentPanel.name);

		/**
		 * After removed search folder on server side and close the search tab we need
		 * to again set the active search tab store in model.
 		 */
		if(currentSearchStore !== false) {
			model.setActiveStore(currentSearchStore);
		}
	},

	/**
	 * Event handler which will be called when the {@link #model} fires the
	 * {@link Zarafa.core.ContextModel#searchfinished} event. This will update
	 * the UI to indicate that the search is no longer running and enable the
	 * components that a new search might be created.
	 *
	 * @param {Zarafa.core.ContextModel} model The model which fired the event
	 * @private
	 */
	onModelSearchFinished: function(model)
	{
		this.resetParentSearchField();
		this.searchToolbar.getAdvanceSearchField().hideMask();
	},

	/**
	 * Function is reset the {@link Zarafa.common.searchfield.ui.SearchTextField SearchTextField}
	 * if parent context has.
	 */
	resetParentSearchField: function()
	{
		var parentSearchField = this.searchContentPanel.getParentSearchField();
		if (Ext.isDefined(parentSearchField)) {
			parentSearchField.reset();
			parentSearchField.hideMask();
		}
	},

	/**
	 * Event handler which will be called when the {@link #model} fires the
	 * {@link Zarafa.core.ContextModel#searchexception} event. This will
	 * show an error message indicating that the search has failed.
	 *
	 * @param {Zarafa.core.ContextModel} model The model which fired the event
	 * @param {Zarafa.core.data.IPMProxy} proxy object that received the error
	 * and which fired exception event.
	 * @param {String} type 'request' if an invalid response from server received,
	 * 'remote' if valid response received from server but with succuessProperty === false.
	 * @param {String} action Name of the action {@link Ext.data.Api.actions}.
	 * @param {Object} options The options for the action that were specified in the request.
	 * @param {Object} response response received from server depends on type.
	 * @param {Mixed} args
	 * @private
	 */
	onModelSearchException: function(model, proxy, type, action, options, response, args)
	{
		var searchTextfield = this.searchToolbar.getAdvanceSearchField();
		searchTextfield.hideMask();
		searchTextfield.focus();
	},

	/**
	 * Function will be used to start actual search on {@link Zarafa.core.data.ListModuleStore ListModuleStore},
	 * and also it will register event on {@link Zarafa.core.data.ListModuleStore ListModuleStore} to get
	 * updated status of search.
	 * @param {Zarafa.common.searchfield.ui.SearchTextField} advanceSearchField the advance search field which
	 * performs the search.
	 * @private
	 */
	onSearchStart: function(advanceSearchField)
	{
		var searchText = advanceSearchField.getValue();
		this.searchContentPanel.searchText = searchText;
		var searchField = this.searchToolbar.getAdvanceSearchField();

		this.searchContentPanel.setTitle(searchText);
		if(searchField.getValue() !== searchText) {
			searchField.setValue(searchText);
		}

		var restriction = this.searchToolBox.createRestriction(searchText);
		var searchFolderCombo = this.searchToolbar.getSearchFolderCombo();
		var folder = container.getHierarchyStore().getFolder(searchFolderCombo.getValue());

		var record = searchFolderCombo.findRecord('value', searchFolderCombo.getValue());
		var includeSubFolders = record.get('include_subfolder');

		this.model.startSearch(restriction , includeSubFolders, {'folder': folder});

		// if search is performed from the parent search field then, it will set the search tab
		// to active tab.
		var tabPanel = container.getTabPanel();
		tabPanel.setActiveTab(this.searchContentPanel);
	},

	/**
	 * Function will be used to execute stop search request on {@link Zarafa.core.data.ListModuleStore ListModuleStore},
	 * it will also unregister event on store to for getting updates of search status.
	 * @private
	 */
	onSearchStop: function()
	{
		var store = this.model.getActiveStore();
		if (!Ext.isDefined(store.searchFolder[store.searchStoreUniqueId])) {
			this.model.stopSearch();
		} else {
			// If the search folder does not exists in the hierarchy,
			// Then execute the stop search request.
			var hierarchyStore = container.getHierarchyStore();
			var defaultStore = hierarchyStore.getDefaultStore();
			if (defaultStore.getFavoritesStore().findExact('entryid', store.searchFolderEntryId) === -1) {
				this.model.stopSearch();
			}
			delete store.searchFolder[store.searchStoreUniqueId];
		}

		if(Ext.isDefined(this.searchToolbar)) {
			this.searchToolbar.getAdvanceSearchField().focus();
		}
	},

	/**
	 * Event handler triggers when {@link Zarafa.advancesearch.dialogs.SearchToolBoxPanel search tool box}
	 * layout was rendered also it will start the search.
	 * @param {Zarafa.advancesearch.dialogs.SearchToolBoxPanel} searchToolBox The search tool box panel.
	 */
	onAfterRenderSearchToolBox: function()
	{
		var searchField = this.searchToolbar.getAdvanceSearchField();
		searchField.searchPanelRendered = this.rendered;

		var store = this.model.getActiveStore();
		if (Ext.isDefined(store.searchFolder[store.searchStoreUniqueId])) {
			var folder = store.searchFolder[store.searchStoreUniqueId];

			// Load a new set of folders from the store.
			var data = Ext.applyIf({}, {
				'folder': folder
			});

			store.load(data);
			store.hasSearchResults = true;
		} else {
			// Trigger search operation.
			searchField.onTriggerClick();
		}
	},

	/**
	 * Event handler triggers when {@link Zarafa.advancesearch.dialogs.SearchToolBoxPanel#searchCriteria}
	 * gets update. it will trigger the advance search.
	 */
	onAfterUpdateRestriction: function()
	{
		this.searchToolbar.getAdvanceSearchField().onTriggerClick();
	},

	/**
	 * Event handler triggers when {@link Zarafa.common.searchfield.ui.SearchTextField SearchTextField}
	 * gets render and also it will set the search text in search field.
	 *
	 * @param {Zarafa.common.searchfield.ui.SearchTextField} searchTextField the search text field which
	 * contains the search string.
	 */
	onRenderSearchTextField: function(searchTextField)
	{
		var searchFolderSettingObj = this.searchToolBox.getSearchFolderSettings();
		if (searchFolderSettingObj) {
			searchTextField.setValue(searchFolderSettingObj.search_text);
		} else {
			searchTextField.setValue(this.searchText);
		}
	},

	/**
	 * Event handler triggered when value of {@link Zarafa.common.searchfield.ui.SearchTextField Search text field}
	 * was validated.
	 *
	 * @param {Zarafa.common.searchfield.ui.SearchTextField} searchTextField search text field which contains
	 * search text.
	 */
	onValidSearchTextField: function(searchTextField)
	{
		this.searchText = searchTextField.getValue();
	},

	/**
	 * Event handler triggered when value of search text field has been changed.
	 *
	 * @param {Zarafa.common.searchfield.ui.SearchTextField} searchTextField search text field which contains
	 * @param {String} newValue new value of search text field.
	 * @param {String} oldValue old value of search text field.
	 */
	onChangeSearchTextField: function(searchTextField, newValue, oldValue)
	{
		this.searchText = newValue;
	},

	/**
	 * Event handler triggers when {@link Zarafa.common.searchfield.ui.SearchFolderCombo SearchFolderCombo}
	 * is render. it will update the {@link Zarafa.common.searchfield.ui.SearchFolderCombo SearchFolderCombo}
	 * store with parent search folder
	 *
	 * @param {Zarafa.common.searchfield.ui.SearchTextField} searchTextField the search text field which
	 * contains the search string.
	 */
	onRenderSearchFolderCombo: function(searchFolderCombo)
	{
		var searchFolderSettingObj = this.searchToolBox.getSearchFolderSettings();
		if (searchFolderSettingObj) {
			var searchComboSettingObj = searchFolderSettingObj['search_folder_combo'];
			var store = searchFolderCombo.getStore();
			var value;

			if (searchComboSettingObj['folder_type'] === Zarafa.advancesearch.data.SearchComboBoxFieldsFlags.ALL_FOLDERS) {
				var folder = store.getAt(store.find('flag', searchComboSettingObj['folder_type']));
				value = folder.get('value');
			} else {
				var index = 0;
				if (searchComboSettingObj['folder_type'] === Zarafa.advancesearch.data.SearchComboBoxFieldsFlags.CURRENT_SELECTED_FOLDER) {
					index = 1;
					store.removeAt(index);
				} else if (store.getAt(0).get('flag') === Zarafa.advancesearch.data.SearchComboBoxFieldsFlags.IMPORTED_FOLDER) {
					store.removeAt(index);
				}
				var folder = new Ext.data.Record({
					'name': searchComboSettingObj['folder_name'],
					'value': searchComboSettingObj['folder_entryid'],
					'flag': searchComboSettingObj['folder_type'],
					'include_subfolder': searchComboSettingObj['include_subfolder']
				});
				store.insert(index, folder);
				value = searchComboSettingObj['folder_entryid'];
			}
			searchFolderCombo.setValue(value);
		} else {
			searchFolderCombo.store.clearData();
			var parentSearchFolderCombo = this.searchContentPanel.getParentSearchFolderCombo();
			parentSearchFolderCombo.store.getRange().forEach(function (record) {
				searchFolderCombo.store.add(record.copy());
			});
			var value = parentSearchFolderCombo.getValue();
			searchFolderCombo.setValue(value);
		}

	},

	/**
	 * Event handler triggered when selection was performed on
	 * {@link Zarafa.common.searchfield.ui.SearchFolderCombo SearchFolderCombo}.
	 * function was used to check/un-check "Include sub folder" checkbox which belongs to
	 * {@link Zarafa.advancesearch.dialogs.SearchToolBoxPanel SearchToolBox}.
	 *
	 * @param {Zarafa.common.searchfield.ui.SearchFolderCombo} combo The combo box which fired the event.
	 * @param {Ext.data.Record} record The data record returned from the underlying store
	 * @param {number} index The index of the selected item in the drop down list
	 */
	onSelectSearchFolderComboValue: function(combo, record, index)
	{
		var subFolderCheckBox = this.searchToolBox.includeSubFolder;
		subFolderCheckBox.setValue(record.get('include_subfolder'));
		this.onRenderSubFolderCheckbox(subFolderCheckBox);
	},

	/**
	 * Event handler which is raised just before the {@link Ext.form.FieldSet IncludeSubFolderFieldSet}
	 * is being rendered. it will hide the {@link Ext.form.FieldSet IncludeSubFolderFieldSet} if
	 * selected folder is does not support the search folder.
	 *
	 * @param {Ext.form.FieldSet} fieldSet field set which fire this event.
	 */
	onBeforeRenderSubFolderFieldSet: function (fieldSet)
	{
		var searchFolderCombo = this.searchToolbar.getSearchFolderCombo();
		var folder = container.getHierarchyStore().getFolder(searchFolderCombo.getValue());
		fieldSet.hidden = !this.model.supportsSearchFolder(folder);
	},

	/**
	 * Event handler triggered when "Include sub folder" checkbox which belongs to
	 * {@link Zarafa.advancesearch.dialogs.SearchToolBoxPanel SearchToolBox}
	 * was checked/un-checked. it will also update the "include_subfolder" property of
	 * {@link Ext.data.Record searchFolder} record of search folder combo box.
	 *
	 * @param {Ext.form.Checkbox} checkbox The checkbox which fired the event
	 * @param {Boolean} check True if the checkbox is currently checked
	 * @private
	 */
	onCheckIncludeSubFolder: function (checkBox, checked)
	{
		var searchFolderCombo = this.searchToolbar.getSearchFolderCombo();
		var record = searchFolderCombo.findRecord('value', searchFolderCombo.getValue());

		if (checked !== record.get('include_subfolder')) {
			record.set('include_subfolder', checked);
			record.commit();
			this.searchToolBox.afterUpdateRestriction();
		}
	},

	/**
	 * Event handle triggered when "Include sub folder" checkbox which belongs to
	 * {@link Zarafa.advancesearch.dialogs.SearchToolBoxPanel SearchToolBox} was rendered
	 * also it will call when search folder change in {@link Zarafa.common.searchfield.ui.SearchFolderCombo SearchFolderCombo}.
	 * Also function is responsible to disable "Include sub folder" check box
	 * and apply the tooltip when "All folders" option in {@link Zarafa.common.searchfield.ui.SearchFolderCombo SearchFolderCombo}
	 * is selected else enable checkbox and remove the tooltip from "Include Sub folder" checkbox.
	 *
	 * @param {Ext.form.Checkbox} checkBox The checkbox which was rendered.
	 */
	onRenderSubFolderCheckbox: function (checkBox)
	{
		var searchFolderCombo = this.searchToolbar.getSearchFolderCombo();
		var record = searchFolderCombo.findRecord('value', searchFolderCombo.getValue());
		var isAllFolderSelected = record.get('flag') === Zarafa.advancesearch.data.SearchComboBoxFieldsFlags.ALL_FOLDERS;
		checkBox.setDisabled(isAllFolderSelected);
		checkBox.setValue(record.get('include_subfolder'));

		if(checkBox.rendered) {
			// Add tooltip on "include sub folder" check box when "All folders"
			// was selected in search folder combo box else remove tooltip from
			// "include sub folder" check box
			if (isAllFolderSelected) {
				checkBox.wrap.dom.qtip = _("All folders are selected");
			} else {
				delete(checkBox.wrap.dom.qtip);
			}
		}
	},

	/**
	 * Event handler triggered before selection performs in {@link Zarafa.common.searchfield.ui.SearchFolderCombo SearchFolderCombo}
	 * Will open {@link Zarafa.advancesearch.dialogs.SelectFolderContentPanel SelectFolderContentPanel}, if
	 * "Other…" option was selected. also it will check select folder from {@link Zarafa.common.searchfield.ui.SearchFolderCombo SearchFolderCombo}
	 * is supports search folder if not we hide the {@link Ext.form.FieldSet IncludeSubFolderFieldSet}.
	 *
	 * @param {Zarafa.common.searchfield.ui.SearchFolderCombo} combo The combo which fired the event.
	 * @param {Ext.data.Record} record The data record returned from the underlying store
	 * @param {number} index The index of the selected item in the dropdown list
	 * @return {boolean} true if selected record is not 'Other…' else false.
	 */
	onBeforeSelectSearchFolder: function (combo, record, index)
	{
		if(record.get('value') === 'other') {
			combo.collapse();
			Zarafa.advancesearch.Actions.openSelectSearchFolderDialog({
				searchFolderCombo: combo,
				model: this.model
			});
			return false;
		} else {
			var folder = container.getHierarchyStore().getFolder(record.get('value'));
			this.searchToolBox.includeSubFolderFieldSet.setVisible(this.model.supportsSearchFolder(folder));
			this.doLayout();
		}
		return true;
	},

	/**
	 * Event handler triggered when Search result grid is gets resize.
	 * also it will set the width of the left and right toolbar of the
	 * {@link Zarafa.advancesearch.dialogs.SearchToolbarPanel search toolbar panel}
	 *
	 * @param {Ext.grid.GridPanel} grid which holds the search result.
	 * @param {Number} adjWidth The box-adjusted width that was set
	 * @param {Number} adjHeight The box-adjusted height that was set
	 * @param {Number} rawWidth The width that was originally specified
	 * @param {Number} rawHeight The height that was originally specified
	 */
	onSearchGridResize: function(grid, adjWidth, adjHeight, rawWidth, rawHeight )
	{
		var leftToolbar = this.searchToolbar.contextMainPanelToolbar;
		var rightToolbar = this.searchToolbar.getRightSearchToolbar();

		var searchToolBox = this.getLayout().west;
		var searchToolBoxMargins = searchToolBox.getMargins();

		var margins = searchToolBoxMargins.left + searchToolBoxMargins.right;

		leftToolbar.setWidth(searchToolBox.getSize().width + margins + adjWidth);

		rightToolbar.setWidth(this.centerRegion.switchBorder.getLayout().south.getSize().width);
		rightToolbar.setPosition(leftToolbar.getWidth());
	},

	/**
	 * Event handler which triggered when {@link Zarafa.advancesearch.ui.SearchResultPreviewPanel SearchResultPreviewPanel}
	 * was updated.
	 *
	 * @param {Zarafa.advancesearch.ui.SearchResultPreviewPanel} SearchResultPreviewPanel The SearchResultPreviewPanel which fired the event
	 * @param {Zarafa.core.data.MAPIRecord} record The record to update in this component
	 * @param {Boolean} contentReset force the component to perform a full update of the data.
	 */
	onAfterUpdateSearchPreviewPanel: function(SearchResultPreviewPanel,record, contentReset)
	{
		var rightToolbar = this.searchToolbar.getRightSearchToolbar();
		rightToolbar.setVisible(!!record);

		if (record) {
			var isFaultyMessage = record.isFaultyMessage();
			var isMessageReplyable = Zarafa.core.MessageClass.isClass(record.get('message_class'), ['IPM.NOTE', 'REPORT.IPM', 'IPM.SCHEDULE', 'IPM.APPOINTMENT']);

			// Additional check when the message is IPM.Appointment and not a meeting request
			// but a simple appointment which can not be replied as there is no reply-to recipients available.
			if (isMessageReplyable && Zarafa.core.MessageClass.isClass(record.get('message_class'), ['IPM.APPOINTMENT'])) {
				if(!record.isMeeting()){
					isMessageReplyable = false;
				}
			}

			rightToolbar.replyBtn.setVisible(!isFaultyMessage && isMessageReplyable);
			rightToolbar.replyAllBtn.setVisible(!isFaultyMessage && isMessageReplyable);
			rightToolbar.forwardBtn.setVisible(!isFaultyMessage && isMessageReplyable);

			// Currently pop-out functionality is not available for
			// contact, sticky note, distribution list, appointment and task
			// So disable showing popout button in search results preview panel for those context item
			// TODO Remove when we support popout for all context
			if (Zarafa.supportsPopOut()) {
				var isSupportPopout = Zarafa.core.MessageClass.isClass(record.get('message_class'), ['IPM.NOTE', 'REPORT.IPM.Note', 'IPM.Schedule.Meeting'], true);
				rightToolbar.popoutBtn.setVisible(isSupportPopout);
			}

			// Only show the "Edit as New" button in the toolbar when the item is in the Sent folder
			var defaultFolder = SearchResultPreviewPanel.model.getDefaultFolder();
			rightToolbar.editAsNewBtn.setVisible(defaultFolder.getDefaultFolderKey() === 'sent' && !isFaultyMessage && isMessageReplyable);

			this.searchToolbar.recordComponentPlugin.setRecord(record);
		}
	}
});

Ext.reg('zarafa.searchpanel', Zarafa.advancesearch.dialogs.SearchPanel);
Ext.namespace('Zarafa.advancesearch.dialogs');

/**
 * @class Zarafa.advancesearch.dialogs.SearchToolBoxPanel
 * @extends Ext.Panel
 * @xtype zarafa.searchtoolboxpanel
 *
 */
Zarafa.advancesearch.dialogs.SearchToolBoxPanel = Ext.extend(Ext.Panel, {

	/**
	 * The {@link Zarafa.advancesearch.AdvanceSearchContextModel} which is obtained from
	 * the {@link #context}.
	 *
	 * @property
	 * @type Zarafa.advancesearch.AdvanceSearchContextModel
	 */
	model: undefined,

	/**
	 * The template of the category blocks
	 * @property
	 * @type {Ext.Template/String}
	 */
	categoriesHtmlTemplate:
		'<tpl for=".">' +
			'<span class="k-category-block {colorClass}" '+
				'<tpl if="!Ext.isEmpty(backgroundColor)">style="background-color:{backgroundColor};"</tpl>'+
				'<tpl if="!Ext.isEmpty(hoverString)">ext:qtip = "{hoverString}" ext:qwidth="100%"</tpl>'+
				'>' +
				'{categoryName}' + '<span class="k-category-close"></span>'+
			'</span>' +
		'</tpl>',

	/**
	 * @cfg {Zarafa.advancesearch.data.SearchCategoriesStore} store The store which contain categories,
	 * That added in category filter.
	 */
	searchCategoriesStore: undefined,

	/**
	 * @constructor
	 * @param {Object} config configuration object.
	 */
	constructor: function(config)
	{
		config = config || {};

		if (!Ext.isDefined(config.model) && Ext.isDefined(config.searchContext)) {
			config.model = config.searchContext.getModel();
		}

		this.searchCategoriesStore = new Zarafa.advancesearch.data.SearchCategoriesStore();
		var messageType = {};
		var filterSetting = {};
		var searchInCheckBoxSetting = {};
		var searchCriteria = this.getSearchFolderSettings(config.model, config.searchTabId);
		if (searchCriteria) {
			messageType = searchCriteria.messageTypeCheckBoxGroup || {};
			filterSetting = searchCriteria.filterCheckBoxGroup || {};
			searchInCheckBoxSetting = searchCriteria.searchInCheckBoxGroup || {};
			this.searchCategoriesStore.addCategories(searchCriteria.categories);
		}

		/**
		 * messageClasses contains all message classes which used in advance search.
		 */
		config.messageClasses = {};

		/**
		 * folder on which search gets performed.
		 */
		config.folder = undefined;

		/**
		 * search fields which used to match with search string.
		 */
		config.searchFields = {};

		/**
		 * searchCriteria which contains the search criteria based on this
		 * search criteria search restriction will be form.
		 */
		config.searchCriteria = {};
		config.searchCriteria['date_range'] = {};
		config.searchCriteria['date_range']['start'] = 0;
		config.searchCriteria['date_range']['end'] = 0;

		var dateRangeStore = {
			xtype: 'jsonstore',
			autoDestroy: true,
			fields: ['name', 'value'],
			autoLoad: true,
			data: Zarafa.advancesearch.data.DateRangeFields
		};

		Ext.applyIf(config, {
			xtype: 'zarafa.searchtoolboxpanel',
			title: _('Search tools'),
			width: 175,
			iconCls: 'icon_magnifier',
			cls: 'zarafa-search-toolbox',
			plugins: [{
				ptype: 'zarafa.recordcomponentplugin'
			},{
				ptype: 'zarafa.recordcomponentupdaterplugin'
			}],
			collapsible: true,
			layout: 'fit',
			unstyled: true,
			ref: 'searchToolBox',
			items: [{
				xtype: 'container',
				autoScroll: true,
				items: [
					this.createFoldersFieldset(),
					this.createMessageTypeFieldset(messageType),
					this.createFilterFieldset(filterSetting),
					this.createDateRangeFieldset(dateRangeStore),
					this.createSearchInFieldset(searchInCheckBoxSetting),
					this.createCategoryFilterFieldset(dateRangeStore),
					this.createFavoritesContainer(config)
				]
			}],
			listeners: {
				afterrender: this.onAfterRender,
				scope: this
			}
		});

		this.addEvents(
			/**
			 * @event afterupdaterestriction fired after the {@link #searchCriteria} gets updated by the
			 * {@link Zarafa.advancesearch.dialogs.SearchToolBoxPanel searchToolBox}.
			 * @param {Zarafa.advancesearch.dialogs.SearchToolBoxPanel} searchToolBox which used to triggers/update the search restriction.
			 */
			'afterupdaterestriction'
		);

		Zarafa.advancesearch.dialogs.SearchToolBoxPanel.superclass.constructor.call(this, config);
	},

	/**
	 * Event handler for the render event of the SearchToolBoxPanel. Will add an event listener to the
	 * input element of the {@link Zarafa.common.searchfield.ui.SearchTextField}
	 */
	onAfterRender: function()
	{
		var searchTextField = this.ownerCt.searchToolbar.contextMainPanelToolbar.searchFieldContainer.searchTextField;

		// Because the input event is not relayed by the Ext.form.TextField (like e.g. keyup) we must listen
		// to the input element itself. We can only add the listener once it has been rendered, hence the
		// double mon()
		this.mon(searchTextField, 'render', function() {
			this.mon(searchTextField.getEl(), 'input', this.onSearchTextFieldChange, this);

			// Call the listener also upon rendering of the searchTextField, since we might need to disable the
			// search fields of the toolbox
			this.onSearchTextFieldChange();
		}, this, {single: true});
	},

	/**
	 * Event handler for the input event of the input of the {@link Zarafa.common.searchfield.ui.SearchTextField}
	 * Will disable the search field panels when the search query is a KQL query, or enable them otherwise
	 * @param {Zarafa.common.searchfield.ui.SearchTextField} searchTextField The text field where the
	 * search queries are entered.
	 */
	onSearchTextFieldChange: function()
	{
		var searchTextField = this.ownerCt.searchToolbar.contextMainPanelToolbar.searchFieldContainer.searchTextField;
		var query = searchTextField.getValue();
		var tokens = Zarafa.advancesearch.KQLParser.tokenize(query);
		var usesAdvancedSyntax = Zarafa.advancesearch.KQLParser.usesExplicitSyntax(tokens);

		if ( usesAdvancedSyntax ) {
			this.searchInFieldset.disable();
			this.categoryFilterFieldSet.disable();
		} else {
			this.searchInFieldset.enable();
			this.categoryFilterFieldSet.enable();
		}
	},

	/**
	 * Creates the folders fieldset for search tool box of form panel.
	 * @return {Object} config object for creating {@link Ext.form.FieldSet FieldSet}.
	 * @private
	 */
	createFoldersFieldset: function()
	{
		return {
			layout: 'form',
			xtype:'fieldset',
			width: 156,
			border: false,
			title: _('Folders'),
			ref: '../includeSubFolderFieldSet',
			items: [{
				xtype: "checkbox",
				hideLabel: true,
				ref: '../../includeSubFolder',
				boxLabel: _('Include subfolders')
			}]
		};
	},

	/**
	 * Creates the message type fieldset for search tool box of form panel.
	 *
	 * @param {Object} messageType setting object which used to pre-select the check box when
	 * user trying to open saved search folder.
	 *
	 * @return {Object} config object for creating {@link Ext.form.FieldSet FieldSet}.
	 * @private
	 */
	createMessageTypeFieldset: function(messageType)
	{
		return {
			layout: 'form',
			xtype:'fieldset',
			width: 156,
			border: false,
			title: _('Show…'),
			items: [{
				xtype: 'checkboxgroup',
				ref: '../../messageTypeCheckboxGroup',
				columns: 1,
				name: 'messageTypeCheckboxGroup',
				hideLabel: true,
				listeners: {
					change: this.onMessageTypeCheckboxChange,
					scope: this
				},
				items: [{
					name: 'mail',
					boxLabel: _('Mails'),
					checked: Ext.isDefined(messageType['mail'])
				},{
					name: 'calendar',
					boxLabel: _('Appointments'),
					checked: Ext.isDefined(messageType['calendar'])
				},{
					name: 'contact',
					boxLabel: _('Contacts'),
					checked: Ext.isDefined(messageType['contact'])
				},{
					name: 'task',
					boxLabel: _('Tasks'),
					checked: Ext.isDefined(messageType['task'])
				},{
					name: 'note',
					boxLabel: _('Notes'),
					checked: Ext.isDefined(messageType['note'])
				}]
			}]
		};
	},

	/**
	 * Creates the filter fieldset for search tool box of form panel.
	 *
	 * @param {Object} filterSetting setting object which used to pre-select the check box when
	 * user trying to open saved search folder.
	 *
	 * @return {Object} config object for creating {@link Ext.form.FieldSet FieldSet}.
	 * @private
	 */
	createFilterFieldset: function(filterSetting)
	{
		return {
			layout: 'form',
			xtype:'fieldset',
			width: 156,
			border: false,
			title: _('Filter…'),
			items: [{
				xtype: 'checkboxgroup',
				columns: 1,
				ref: '../../filterCheckBoxGroup',
				hideLabel: true,
				name: 'filterCheckBoxGroup',
				listeners: {
					change: this.onFilterCheckBoxGroup,
					render: this.onRenderCheckboxGroup,
					scope: this
				},
				items: [{
					name: 'message_flags',
					boxLabel: _('Unread'),
					checked: Ext.isDefined(filterSetting['message_flags'])
				},{
					name: 'hasattach',
					boxLabel: _('Attachments'),
					checked: Ext.isDefined(filterSetting['hasattach'])
				}]
			}]
		};
	},

	/**
	 * Creates the date range fieldset for search tool box of form panel.
	 *
	 * @param {Ext.data.JsonStore} dateRangeStore store which contains different date range for
	 * date range combo box.
	 *
	 * @return {Object} config object for creating {@link Ext.form.FieldSet FieldSet}.
	 * @private
	 */
	createDateRangeFieldset: function(dateRangeStore)
	{
		return {
			layout: 'form',
			xtype:'fieldset',
			border: false,
			title: _('Date'),
			items: [{
				xtype: 'combo',
				displayField: 'name',
				editable: false,
				hideLabel: true,
				ref: '../../dateRangeCombo',
				store: dateRangeStore,
				valueField: 'value',
				value: dateRangeStore.data[0].value,
				mode: 'local',
				triggerAction: 'all',
				width: 146,
				listeners: {
					select: this.onSelectCombo,
					beforerender: this.onBeforeRenderDateRangeCombo,
					scope: this
				}
			},{
				xtype: 'zarafa.dateperiodfield',
				ref: '../../dateField',
				hidden: true,
				allowBlank: false,
				onStartChange: this.onStartChange,
				onEndChange: this.onEndChange,
				defaultValue: new Zarafa.core.DateRange({
					allowBlank: false ,
					startDate: new Date().add(Date.MONTH, -1),
					dueDate: new Date()
				}),
				startFieldConfig: {
					labelSeparator: "",
					fieldLabel: pgettext('search.date', 'From'),
					labelStyle: 'width: 35px',
					itemCls: 'zarafa-dateperiodfield-itemsCls',
					labelWidth: 50,
					width: 110
				},
				endFieldConfig: {
					labelSeparator: "",
					fieldLabel: pgettext('search.date', 'To'),
					labelStyle: 'width: 35px',
					itemCls: 'zarafa-dateperiodfield-itemsCls',
					labelWidth: 50,
					width: 110
				}
			}]
		};
	},

	/**
	 * Creates the category filter fieldset for search tool box of form panel.
	 *
	 * @return {Object} config object for creating {@link Ext.form.FieldSet FieldSet}.
	 * @private
	 */
	createCategoryFilterFieldset: function ()
	{
		return {
			layout: 'form',
			xtype: 'fieldset',
			border: false,
			cls: 'k-category-filter',
			title: _('Filter category…'),
			autoHeight: true,
			ref: '../categoryFilterFieldSet',
			items: [{
				xtype: 'button',
				iconCls: 'icon_category_add',
				cls: 'k-category-add-button',
				handler: this.onSelectCategory,
				scope: this
			}, {
				xtype: 'button',
				text: _('Select Category'),
				cls: 'k-category-filter-label',
				width: 'auto',
				hidden: this.searchCategoriesStore.getCount(),
				ref: '../../categoryFilterLabel',
				handler: this.onSelectCategory,
				scope: this
			}, {
				xtype: 'dataview',
				anchor: '100% 100%',
				autoHeight: true,
				tpl: this.categoriesHtmlTemplate,
				prepareData: function (data) {
					Ext.apply(data, {
						hoverString: Ext.util.Format.htmlEncode(data.name.length > 20 ? data.name : '').replace(/\s/g, '&nbsp;'),
						categoryName: Ext.util.Format.ellipsis(data.name, 20)
					});
					return data;
				},
				store: this.searchCategoriesStore,
				itemSelector: 'span.k-category-block',
				listeners: {
					click: this.onCategoryRemove,
					scope: this
				}
			}]
		};
	},

	/**
	 * Create the "Search in" {@link Ext.form.CheckboxGroup checkboxgroup} which specifies
	 * which fields search has to look in.
	 *
	 * @param {Object} searchInCheckBoxSetting setting object which used to pre-select the check box when
	 * user trying to open saved search folder.
	 *
	 * @return {Object} config object for creating {@link Ext.form.FieldSet FieldSet}.
	 * @private
	 */
	createSearchInFieldset: function(searchInCheckBoxSetting)
	{
		return {
			layout: 'form',
			xtype:'fieldset',
			width: 160,
			border: false,
			title: _('Search…'),
			ref: '../searchInFieldset',
			items: [{
				xtype: 'checkboxgroup',
				columns: 1,
				ref: '../../searchInCheckboxGroup',
				name: 'searchInCheckboxGroup',
				hideLabel: true,
				listeners: {
					change: this.onSearchInCheckboxChange,
					render: this.onRenderCheckboxGroup,
					scope: this
				},
				items: [{
					name: ['sender_name', 'sender_email_address'],
					itemId: 'sender',
					boxLabel: _('Sender'),
					checked: Ext.isDefined(searchInCheckBoxSetting['sender'])
				},{
					name: ['display_to', 'display_cc', 'display_bcc'],
					itemId: 'recipients',
					boxLabel: _('Recipients'),
					checked: Ext.isDefined(searchInCheckBoxSetting['recipients'])
				},{
					name: 'subject',
					itemId: 'subject',
					boxLabel: _('Subject'),
					checked: Ext.isDefined(searchInCheckBoxSetting['subject'])
				},{
					name: 'body',
					itemId: 'body',
					boxLabel: _('Body & Attachments'),
					checked: Ext.isDefined(searchInCheckBoxSetting['body'])
				}]
			}]
		};
	},

	/**
	 * Create the "Search in" {@link Ext.form.CheckboxGroup checkboxgroup} which specifies
	 * which fields search has to look in.
	 * @return {Object} config object for creating {@link Ext.form.FieldSet FieldSet}.
	 * @private
	 */
	createFavoritesContainer: function ()
	{
		return {
			xtype: 'container',
			cls: 'zarafa-search-toolbox-favoritesbutton-container',
			layout:'fit',
			hidden: container.getSettingsModel().get('zarafa/v1/contexts/hierarchy/hide_favorites', true, false),
			items: [{
				cls: 'search-toolbox-favorites-button',
				iconCls: 'icon_favorites',
				xtype:'button',
				text: '<span>' + _('Favorites') + '</span>',
				handler: this.onClickFavorites,
				tooltip: {
					text: _('Add a folder to favorites based on search query'),
					width: 350
				},
				scope: this
			}]
		};
	},

	/**
	 * Initialize events
	 * @private
	 */
	initEvents: function()
	{
		this.mon(this.messageTypeCheckboxGroup, {
			render: this.onRenderCheckboxGroup,
			scope: this
		});

		this.mon(this.dateRangeCombo, {
			enable: this.onEnableCombo,
			scope: this
		});

		this.mon(this.dateField,{
			change: this.onChangeDateField,
			scope: this
		});

		this.dateField.mon(this.dateField.startField, 'specialkey', this.onSpecialKey, this);
		this.dateField.mon(this.dateField.endField, 'specialkey', this.onSpecialKey, this);
		this.mon(this.searchCategoriesStore, 'add', this.onSearchCategoryUpdate, this);
		this.mon(this.searchCategoriesStore, 'remove', this.onSearchCategoryUpdate, this);
	},

	/**
	 * Overridden event handler which is called when the start date has been changed.
	 *
	 * @param {Ext.form.Field} field The field which has changed
	 * @param {Mixed} newValue The new value for the field
	 * @param {Mixed} oldValue The old value for the field
	 */
	onStartChange: function(field, newValue, oldValue)
	{
		var range = this.defaultValue;
		var oldRange = this.defaultValue.clone();

		if (range.getStartDate() !== newValue) {
			if (Ext.isEmpty(newValue)) {
				range.setStartDate(null);
			} else {
				var dueTime = range.getDueTime();
				// If the start date is after the due date, then set due date same as the start date
				if (newValue.getTime() > dueTime) {
					range.set(newValue, newValue.clone());
				} else {
					range.setStartDate(newValue);
				}
			}
		}

		this.fireEvent('change', this, range.clone(), oldRange);
	},

	/**
	 * Overridden handler which is called when the due date has been changed.
	 *
	 * @param {Ext.form.Field} field The field which has changed
	 * @param {Mixed} newValue The new value for the field
	 * @param {Mixed} oldValue The old value for the field
	 */
	onEndChange: function(field, newValue, oldValue)
	{
		var range = this.defaultValue;
		var oldRange = this.defaultValue.clone();

		if (range.getDueDate() !== newValue) {
			if (Ext.isEmpty(newValue)) {
				range.set(null, null);
			} else {
				var startTime = range.getStartTime();
				if (newValue.getTime() < startTime) {
					// If the due date is before the start date, then set start date same as the due date
					range.set(newValue, newValue.clone());
				} else {
					range.setDueDate(newValue);
				}
			}
		}

		this.fireEvent('change', this, range.clone(), oldRange);
	},

	/**
	 * Event handler which is raised just before the {@link Ext.form.ComboBox ComboBox}
	 * is being rendered. it will call {@link #getSearchFolderSettings} which
	 * provide saved search folder related setting objects which used to pre-select
	 * date range combo box.
	 *
	 * @param {Ext.form.ComboBox} combo the combo box component.
	 */
	onBeforeRenderDateRangeCombo: function(combo)
	{
		var searchCriteria = this.getSearchFolderSettings();
		if (searchCriteria) {
			var dateRange = searchCriteria.date_range;
			var record = combo.findRecord(combo.valueField, dateRange);
			this.setDateRangeRestriction(combo, record);
			if (!Ext.isObject(dateRange)) {
				combo.setValue(dateRange);
			} else {
				combo.setValue('custom_date');
				this.dateField.hidden = false;
				var startDate = new Date(dateRange.start);
				var dueDate = new Date(dateRange.due);

				this.dateField.startField.setValue(startDate);
				this.dateField.endField.setValue(dueDate);
			}
		}
	},

	/**
	 * Function which used to retrieve the saved search folder related settings object
	 * if we are trying to open it.
	 *
	 * @param {Zarafa.advancesearch.AdvanceSearchContextModel} contextModel (optional) the advance search context model
	 * @param {String} searchStoreUniqueId (optional) searchStoreUniqueId is represent the unique id of
	 * {@link Zarafa.advancesearch.AdvanceSearchStore AdvanceSearchStore}.
	 *
	 * @returns {Object|Boolean} return settings object of saved search folder else false.
	 */
	getSearchFolderSettings: function(contextModel, searchStoreUniqueId)
	{
		var model = Ext.isDefined(contextModel) ? contextModel : this.model;
		var store = model.store;
		searchStoreUniqueId = Ext.isDefined(searchStoreUniqueId) ? searchStoreUniqueId : store.searchStoreUniqueId;
		if (Ext.isDefined(store.searchFolder[searchStoreUniqueId])) {
			var folder = store.searchFolder[searchStoreUniqueId];
			return container.getSettingsModel().getSettingsObject('zarafa/v1/contexts/search/search_criteria/'+folder.get('entryid'));
		}
		return false;
	},

	/**
	 * Event handler is fired for each special key, but it only handles the {@link Ext.EventObjectImp#ENTER} key.
	 * it was call the triggerBlur function of updated date field. which internally fire the
	 * blur event and blur event fire the change event, which handled by
	 * {@link Zarafa.common.ui.DatePeriodField#onStartChange} or {@link Zarafa.common.ui.DatePeriodField#onEndChange}
	 * which fire the {@link Zarafa.common.ui.DateRangeField#change} event of {@link Zarafa.common.ui.DateRangeField date rage field}
	 * and it was handled by the {@link #onChangeDateField}.
	 *
	 * @param {Ext.form.Field} field The field which fired the event
	 * @param {Ext.EventObject} eventObj The event object for this event
	 */
	onSpecialKey: function(field, eventObj)
	{
		if (eventObj.getKey() === eventObj.ENTER) {
			field.triggerBlur();
		}
	},

	/**
	 * Function which is used to set the date range related restriction
	 * in {@link #searchCriteria} object.
	 *
	 * @param {Ext.form.ComboBox} combo The field which fired the event
	 * @param {Ext.data.Record} record The selected record
	 * @private
	 */
	setDateRangeRestriction: function(combo, record)
	{
		var value = record.get('value');
		var today = new Date();
		// Add a day to implement until
		this.searchCriteria['date_range']['end'] = today.add(Date.DAY, 1).getTime() / 1000;

		if (value !== 'custom_date' && this.dateField.isVisible()) {
			this.dateField.hide();
			this.doLayout();
		}

		switch(value) {
			case 'past_week':
				this.searchCriteria['date_range']['start'] = today.add(Date.DAY, -7).getTime() / 1000;
			break;
			case 'past_two_weeks':
				this.searchCriteria['date_range']['start'] = today.add(Date.DAY, -14).getTime() / 1000;
			break;
			case 'past_month':
				this.searchCriteria['date_range']['start'] = today.add(Date.MONTH, -1).getTime() / 1000;
				break;
			case 'past_six_month':
				this.searchCriteria['date_range']['start'] = today.add(Date.MONTH, -6).getTime() / 1000;
				break;
			case 'past_year':
				this.searchCriteria['date_range']['start'] = today.add(Date.YEAR, -1).getTime() / 1000;
				break;
			case 'custom_date':
				this.dateField.show();
				this.doLayout();
				this.searchCriteria['date_range']['start'] = this.dateField.startField.getValue().getTime() / 1000;
				// Add a day to implement until
				this.searchCriteria['date_range']['end'] = this.dateField.endField.getValue().add(Date.DAY, 1).getTime() / 1000;
				break;
			default:
				this.searchCriteria['date_range']['start'] = 0;
				this.searchCriteria['date_range']['end'] = 0;
		}
	},

	/**
	 * Event handler which is called when a selection has been made in the
	 * {@link Ext.form.ComboBox combobox}.
	 * @param {Ext.form.ComboBox} combo The field which fired the event
	 * @param {Ext.data.Record} record The selected record
	 * @private
	 */
	onSelectCombo: function(combo, record)
	{
		this.setDateRangeRestriction(combo, record);
		this.afterUpdateRestriction();
	},

	/**
	 * Event handler for the {@link Ext.form.CheckboxGroup#change change} event, this will
	 * update the {@link #searchCriteria}, which used in advance search request.
	 *
	 * @param {Ext.form.CheckboxGroup} group the checkboxgroup
	 * @param {Array} checked an array of {Ext.form.Checkbox} items which are selected
	 */
	onFilterCheckBoxGroup: function(group, checked)
	{
		this.setFilterRestriction(group, checked);
		this.afterUpdateRestriction();
	},

	/**
	 * Event handler for the {@link Ext.form.CheckboxGroup#change change} event, this will
	 * update the {@link #searchCriteria}, which used in advance search request.
	 *
	 * @param {Ext.form.CheckboxGroup} group the checkboxgroup
	 * @param {Array} checked an array of {Ext.form.Checkbox} items which are selected
	 */
	onMessageTypeCheckboxChange: function(group, checked)
	{
		this.setMessageClassRestriction(group, checked);
		this.afterUpdateRestriction();
	},

	/**
	 * Event handler for the {@link Ext.form.CheckboxGroup#change change} event, this will
	 * update the {@link #searchCriteria} and restricts the "search_fields" so that it only
	 * contains fields which are selected.
	 *
	 * @param {Ext.form.CheckboxGroup} group the checkboxgroup
	 * @param {Array} checked an array of {Ext.form.Checkbox} items which are selected
	 */
	onSearchInCheckboxChange: function(group, checked)
	{
		this.setSearchInRestriction(group, checked);
		this.afterUpdateRestriction();
	},

	/**
	 * Sets the search restriction for Search filtering fields.
	 *
	 * @param {Ext.form.CheckboxGroup} group the checkboxgroup
	 * @param {Array} checked an array of {Ext.form.Checkbox} items which are selected
	 */
	setSearchInRestriction: function (group, checked)
	{
		if (Ext.isEmpty(checked)) {
			// Set the search_fields restriction based the current selected messageClasses.
			this.setMessageClassRestriction(this.messageTypeCheckboxGroup, this.messageTypeCheckboxGroup.getValue());
		} else {
			var searchFields = [];
			checked.forEach(function(checkBox) {
				searchFields = searchFields.concat(checkBox.name);
			});
			this.searchCriteria['search_fields'] = searchFields;
		}
	},

	/**
	 * Function call after the {@link #searchCriteria} gets updated
	 * by {@link Zarafa.advancesearch.dialogs.SearchToolBoxPanel search tool box}. This
	 * will fire the {@link #afterupdaterestriction} which triggers the advance search.
	 */
	afterUpdateRestriction: function()
	{
		this.fireEvent('afterupdaterestriction' , this);
	},

	/**
	 * Event handler was fire when message type/ filter/ search check box group gets rendered.
	 * @param {Ext.form.CheckboxGroup} group the group is {@link Ext.form.CheckboxGroup checkbox}
	 * @private
	 */
	onRenderCheckboxGroup: function(group)
	{
		switch(group.name) {
			case 'filterCheckBoxGroup':
				this.setFilterRestriction(group, group.getValue());
				break;
			case 'searchInCheckboxGroup':
				this.setSearchInRestriction(group, group.getValue());
				break;
			default:
				this.setMessageClassRestriction(group, group.getValue());
		}
	},

	/**
	 * Sets the search restriction for extra filtering fields.
	 *
	 * @param {Ext.form.CheckboxGroup} group the group is {@link Ext.form.CheckboxGroup checkbox}
	 * @param {Array} checked an array of {Ext.form.Checkbox} items which are selected
	 */
	setFilterRestriction: function(group, checked)
	{
		this.searchCriteria['extra_fields'] = checked.map(function(checkbox) { return checkbox.name; });
	},

	/**
	 * Sets the search restriction for message classes based on the checkboxes which are available,
	 * if no checkboxes are selected we want to search through all available message and searchfields.
	 *
	 * @param {Ext.form.CheckboxGroup} group the {@link Ext.form.CheckboxGroup checkbox} group
	 * @param {Array} checked a list of checkboxes which are enabled
	 */
	setMessageClassRestriction: function(group, checked)
	{
		// Helper to filter out duplicates
		const onlyUnique = function(value, index, arr) {
			return arr.indexOf(value) === index;
		};

		var messageClasses = [];
		var searchFields = [];

		if (Ext.isEmpty(checked)) {
			checked = group.items.items;
		}

		var searchInCheckBox = this.searchInCheckboxGroup.getValue();
		var searchInCheckBoxFields = [];
		if (!Ext.isEmpty(searchInCheckBox)) {
			searchInCheckBox.forEach(function (checkBox) {
				searchInCheckBoxFields = searchInCheckBoxFields.concat(checkBox.name);
			}, this);
		}

		checked.forEach(function(checkBox) {
			messageClasses = messageClasses.concat(this.getMessageClass(checkBox.name));
			// searchInCheckBox has high priority, If any of the checkBox selected from that
			// then don't add/contact default search fields in searchFields array.
			if (Ext.isEmpty(searchInCheckBoxFields)) {
				searchFields = searchFields.concat(Zarafa[checkBox.name].data.SearchFields[0].value.split(' '));
			}
		}, this);

		if (!Ext.isEmpty(searchInCheckBoxFields)) {
			searchFields = searchInCheckBoxFields;
		}

		this.searchCriteria['message_class'] = messageClasses.filter(onlyUnique);
		this.searchCriteria['search_fields'] = searchFields.filter(onlyUnique);
	},

	/**
	 * Function is use to retive the message class based on the selected
	 * {@link #createMessageTypeFieldset}.
	 * @param {String} checkBoxName The checkBoxName of the selected check box from check box list
	 * @return {Array} return and array of message classes.
	 */
	getMessageClass: function(checkBoxName)
	{
		switch(checkBoxName) {
			case 'mail':
				return ['IPM.Note', 'REPORT.IPM.Note'];
			case 'calendar':
				return ['IPM.Appointment', 'IPM.Schedule'];
			case 'contact':
				return ['IPM.Contact', 'IPM.DistList'];
			case 'task':
				return ['IPM.Task'];
			case 'note':
				return ['IPM.StickyNote'];
		}
	},

	/**
	 * Event handler which is fired when the {@link Zarafa.common.ui.DateRangeField} has been changed.
	 * This will update the start and due date inside the {@link #searchCriteria} accordingly.
	 *
	 * @param {Ext.form.Field} field The field which has changed
	 * @param {Mixed} newRange The new date range
	 * @param {Mixed} oldRange The old date range
	 * @private
	 */
	onChangeDateField: function(field, newRange, oldRange)
	{
		var newStartDate = newRange.startDate.getTime()/1000;
		// Add a day to implement until
		var newDueDate = newRange.dueDate.add(Date.DAY, 1).getTime()/1000;

		this.searchCriteria['date_range']['start'] = newStartDate;
		this.searchCriteria['date_range']['end'] = newDueDate;

		if(newRange.compare(oldRange) !== 0) {
			this.afterUpdateRestriction();
		}
	},

	/**
	 * Event handler triggers after the date range combo box gets enabled.
	 * also it will update the {@link #searchCriteria} based on the selected
	 * value of the combo box.
	 *
	 * @param {Ext.form.ComboBox} combo which gets enabled.
	 * @private
	 */
	onEnableCombo: function(combo)
	{
		var index = combo.getStore().find('value', combo.getValue());
		var record = combo.getStore().getAt(index);
		this.onSelectCombo(combo, record);
	},

	/**
	 * Event handler triggered when "Favorites" button was pressed. it will open
	 * {@link Zarafa.advancesearch.dialogs.CreateSearchFolderPanel CreateSearchFolderPanel}.
	 */
	onClickFavorites: function ()
	{
		var config = {
			searchText: this.dialog.searchText,
			searchStoreEntryId: this.model.getStore().searchStoreEntryId,
			includeSubFolder: this.includeSubFolder.getValue(),
			searchFolderEntryId: this.model.store.searchFolderEntryId
		};
		Zarafa.advancesearch.Actions.openCreateSearchFolderContentPanel(this.model, config);
	},


	/**
	 * Function will be used to create search restriction based on value entered in
	 * search textfield and {@link Zarafa.common.search.dialogs.SearchToolBoxPanel SearchToolBox}.
	 *
	 * In words: all terms must occur at least once, but it doesn't matter in which of the fields they occur.
	 *
	 * @param {String} textFieldValue value of search text field.
	 * @return {Object} Object that will be passed as restriction to server.
	 * @private
	 */
	createRestriction: function(textFieldValue)
	{
		if (Ext.isEmpty(textFieldValue)) {
			return [];
		}

		var searchFieldPreference = Ext.isArray(this.searchCriteria['search_fields']) ? this.searchCriteria['search_fields'].slice(0) : [];
		var tokens = Zarafa.advancesearch.KQLParser.tokenize(textFieldValue);
		var usesAdvancedSyntax = Zarafa.advancesearch.KQLParser.usesExplicitSyntax(tokens);
		var andRes = [];
		if ( tokens ) {
			var tokenRes = Zarafa.advancesearch.KQLParser.createTokenRestriction(tokens, usesAdvancedSyntax ? null : searchFieldPreference);
			if ( tokenRes ) {
				andRes = [tokenRes];
			} else {
				// treat malformed KQL as plain search text
				tokens = false;
			}
		}

		var finalRes = [];
		var orResDate = [];
		var orResSearchField = [];
		var orResMessageClass = [];
		var andResCategory = [];
		var orFilters = [];

		Ext.iterate(this.searchCriteria, function(key, values) {
			if ( !tokens ) {
				// search field restriction
				if(key === 'search_fields') {
					Ext.each(values, function(value){
						orResSearchField.push(
							Zarafa.core.data.RestrictionFactory.dataResContent(
								value,
								Zarafa.core.mapi.Restrictions.FL_SUBSTRING | Zarafa.core.mapi.Restrictions.FL_IGNORECASE,
								textFieldValue
							)
						);
					}, this);
				}
			}

			if (key === 'extra_fields') {
				Ext.each(values, function(value) {
					if (value === 'hasattach') {
						orFilters.push(
							Zarafa.core.data.RestrictionFactory.createResAnd([
								Zarafa.core.data.RestrictionFactory.dataResProperty(
									'hide_attachments',
									Zarafa.core.mapi.Restrictions.RELOP_NE,
									true
								),
								Zarafa.core.data.RestrictionFactory.createResSubRestriction(
									'PR_MESSAGE_ATTACHMENTS',
									Zarafa.core.data.RestrictionFactory.dataResProperty(
										'PR_ATTACHMENT_HIDDEN',
										Zarafa.core.mapi.Restrictions.RELOP_NE,
										true
									)
								)
							])
						);

					}
					if (value === 'message_flags') {
						orFilters.push(
							Zarafa.core.data.RestrictionFactory.dataResBitmask(
								value,
								Zarafa.core.mapi.Restrictions.BMR_EQZ,
								Zarafa.core.mapi.MessageFlags.MSGFLAG_READ
							)
						);
					}
				});
			}

			// Date Range restriction
			if(key === 'date_range') {
				if(values.start !== 0 && values.end !== 0) {
					// Modification date
					orResDate = Zarafa.core.data.RestrictionFactory.createResOr([
						Zarafa.core.data.RestrictionFactory.createResAnd([
							Zarafa.core.data.RestrictionFactory.createResNot(
								Zarafa.core.data.RestrictionFactory.dataResExist('PR_MESSAGE_DELIVERY_TIME')
							),
							Zarafa.core.data.RestrictionFactory.dataResProperty(
								'last_modification_time',
								Zarafa.core.mapi.Restrictions.RELOP_GE,
								values.start
							),
							Zarafa.core.data.RestrictionFactory.dataResProperty(
								'last_modification_time',
								Zarafa.core.mapi.Restrictions.RELOP_LT,
								values.end
							)
						]),
						Zarafa.core.data.RestrictionFactory.createResAnd([
							Zarafa.core.data.RestrictionFactory.dataResExist('PR_MESSAGE_DELIVERY_TIME'),
							Zarafa.core.data.RestrictionFactory.dataResProperty(
								'message_delivery_time',
								Zarafa.core.mapi.Restrictions.RELOP_GE,
								values.start
							),
							Zarafa.core.data.RestrictionFactory.dataResProperty(
								'message_delivery_time',
								Zarafa.core.mapi.Restrictions.RELOP_LT,
								values.end
							)
						])
					]);
				}
			}

			// message class restriction
			if(key === 'message_class' && !Ext.isEmpty(values)) {
				Ext.each(values, function(value){
					orResMessageClass.push(
						Zarafa.core.data.RestrictionFactory.dataResContent(
							key,
							Zarafa.core.mapi.Restrictions.FL_PREFIX | Zarafa.core.mapi.Restrictions.FL_IGNORECASE,
							value
						)
					);
				}, this);
			}

			// category restriction
			if (key === 'categories' && !Ext.isEmpty(values)) {
				Ext.each(values, function (value) {
					andResCategory.push(
						Zarafa.core.data.RestrictionFactory.dataResContent(
							key,
							Zarafa.core.mapi.Restrictions.FL_FULLSTRING | Zarafa.core.mapi.Restrictions.FL_IGNORECASE,
							value
						)
					);
				}, this);
			}
		}, this);

		/**
		 * If date-information is present in search criteria then create search restriction
		 * something like this.
		 * AND
		 * 		AND
		 * 			OR
		 * 				AND
		 * 					Not PR_MESSAGE_DELIVERY_TIME exists
		 * 					start date (last_modification_time)
		 * 					end date (last_modification_time)
		 * 				AND
		 * 					PR_MESSAGE_DELIVERY_TIME exists
		 * 					start date (message_delivery_time)
		 * 					end date (message_delivery_time)
		 * 			OR
		 * 				searchFields
		 * 		OR
		 * 			message class
		 * 		OR
		 * 			search filters
		 */
		if(!Ext.isEmpty(orResDate)) {
			var andResDateSearchField = [];
			andResDateSearchField.push(orResDate);
			if ( orResSearchField.length ) {
				andResDateSearchField.push(Zarafa.core.data.RestrictionFactory.createResOr(orResSearchField));
			}
			andRes.push(Zarafa.core.data.RestrictionFactory.createResAnd(andResDateSearchField));
		} else if ( orResSearchField.length ) {
			/**
			 * If date information is not present in search criteria then create search restriction
			 * something like this.
			 * AND
			 * 		OR
			 * 			searchFields
			 * 		OR
			 * 			message class
			 * 		OR
			 * 			search filters
			 */
			andRes.push(Zarafa.core.data.RestrictionFactory.createResOr(orResSearchField));
		}

		// Message class restriction which indicates which type of message you want to search.
		andRes.push(Zarafa.core.data.RestrictionFactory.createResOr(orResMessageClass));

		if (!Ext.isEmpty(andResCategory)) {
			andRes.push(Zarafa.core.data.RestrictionFactory.createResAnd(andResCategory));
		}

		if (!Ext.isEmpty(orFilters)) {
			andRes.push(Zarafa.core.data.RestrictionFactory.createResAnd(orFilters));
		}

		if(!Ext.isEmpty(andRes)) {
			finalRes = Zarafa.core.data.RestrictionFactory.createResAnd(andRes);
		}


		return finalRes;
	},

	/**
	 * Function which is handle click event of select category.
	 * It will show {@link Zarafa.advancesearch.dialogs.SearchCategoriesContentPanel search category panel}.
	 */
	onSelectCategory: function ()
	{
		Zarafa.advancesearch.Actions.openSearchCategoryContentPanel({
			modal: true,
			searchCategoryStore: this.searchCategoriesStore,
			scope: this
		});
	},

	/**
	 * Function which is handle click event of category box.
	 * It will identify that is user click on close button, If yes then
	 * Remove that category from {@link Zarafa.advancesearch.data.SearchCategoriesStore}
	 *
	 * @param {Ext.DataView} item Categories data view.
	 * @param {Number} index The index of the target node in {@link Zarafa.advancesearch.data.SearchCategoriesStore}.
	 * @param {HTMLElement} node html element.
	 * @param {Ext.EventObject} e event object.
	 */
	onCategoryRemove: function (item, index, node, e)
	{
		var element = e.getTarget();
		if (element.className === "k-category-close") {
			this.searchCategoriesStore.removeAt(index);
		}
	},

	/**
	 * Sets the search restriction for categories
	 * based on categories available in {@link #searchCategoriesStore},
	 *
	 * @param {Array} categories a list of categories.
	 */
	setCategoriesRestriction: function(categories)
	{
		this.searchCriteria['categories'] = categories;
	},

	/**
	 * Handler which is call while category will be add or remove.
	 * It will update the search restriction with categories available in {@link #searchCategoriesStore}
	 */
	onSearchCategoryUpdate: function ()
	{
		var categories = this.searchCategoriesStore.getCategories();
		this.setCategoriesRestriction(categories);
		this.afterUpdateRestriction();
		this.categoryFilterLabel.setVisible(!this.searchCategoriesStore.getCount());
	}
});

Ext.reg('zarafa.searchtoolboxpanel', Zarafa.advancesearch.dialogs.SearchToolBoxPanel);

Ext.namespace('Zarafa.advancesearch.dialogs');

/**
 * @class Zarafa.advancesearch.dialogs.SearchToolbarPanel
 * @extends Ext.Panel
 * @xtype zarafa.searchtoolbarpanel
 *
 */
Zarafa.advancesearch.dialogs.SearchToolbarPanel = Ext.extend(Ext.Panel, {

	/**
	 * @constructor
	 * @param {Object} config configuration object.
	 */
	constructor: function(config)
	{
		config = config || {};

		if (!Ext.isDefined(config.model) && Ext.isDefined(config.searchContext)) {
			config.model = config.searchContext.getModel();
		}

		Ext.applyIf(config, {
			xtype: 'zarafa.searchtoolbarpanel',
			layout: 'hbox',
			ref: 'searchToolbar',
			cls: 'k-search-toolbar-panel',
			border: false,
			plugins: [{
				ptype: 'zarafa.recordcomponentplugin',
				enableOpenLoadTask: true,
				autoOpenLoadTaskDefer: 0
			},{
				ptype: 'zarafa.recordcomponentupdaterplugin'
			}],
			height: 35,
			items: [{
				xtype: 'zarafa.contextmainpaneltoolbar',
				style: 'border-style: none',
				searchText: config.searchText,
				context: config.searchContext
			},{
				xtype: 'zarafa.toolbar',
				style: 'border-style: none; margin-left:5px;',
				cls: 'zarafa-previewpanel-toolbar zarafa-search-previewpanel-toolbar zarafa-context-mainpanel', // change the css class name
				ref: 'rightSearchToolbar',
				hidden: true,
				items: [container.populateInsertionPoint('previewpanel.toolbar.left', {scope: this, model: config.model}),
				{
					xtype: 'button',
					tooltip: _('Reply') + ' (Ctrl + R)',
					overflowText: _('Reply'),
					text: _('Reply'),
					iconCls: 'icon_reply',
					ref: 'replyBtn',
					responseMode: Zarafa.mail.data.ActionTypes.REPLY,
					handler: this.onResponse,
					scope: this
				},{
					xtype: 'button',
					tooltip: _('Reply All') + ' (Ctrl + Alt + R)',
					overflowText: _('Reply All'),
					text: _('Reply All'),
					iconCls: 'icon_reply_all',
					ref: 'replyAllBtn',
					responseMode: Zarafa.mail.data.ActionTypes.REPLYALL,
					handler: this.onResponse,
					scope: this
				},{
					xtype: 'button',
					tooltip: _('Forward') + ' (Ctrl + F)',
					overflowText: _('Forward'),
					text: _('Forward'),
					iconCls: 'icon_forward',
					ref: 'forwardBtn',
					responseMode: Zarafa.mail.data.ActionTypes.FORWARD,
					handler: this.onResponse,
					scope: this
				},{
					xtype: 'tbfill'
				},
				container.populateInsertionPoint('previewpanel.toolbar.right.first', {scope: this, model: config.model}),
				{
					xtype: 'button',
					tooltip: _('Edit as New') + ' (Ctrl + E)',
					overflowText: _('Edit as New'),
					iconCls: 'icon_edit_as_new_mail',
					ref: 'editAsNewBtn',
					responseMode: Zarafa.mail.data.ActionTypes.EDIT_AS_NEW,
					handler: this.onResponse,
					scope: this
				},container.populateInsertionPoint('previewpanel.toolbar.right', {scope: this, model: config.model})]
			}]
		});

		Zarafa.advancesearch.dialogs.SearchToolbarPanel.superclass.constructor.call(this, config);
	},

	/**
	 * Function is used to retrieve {@link Zarafa.common.searchfield.ui.SearchFieldContainer SearchFieldContainer}
	 *
	 * @return {Zarafa.common.searchfield.ui.SearchFieldContainer} Search field container.
	 */
	getSearchFieldContainer: function()
	{
		return this.contextMainPanelToolbar.searchFieldContainer;
	},

	/**
	 * Function is used to retrieve the {@link Zarafa.common.searchfield.ui.SearchTextField SearchTextField}.
	 * @return {Zarafa.common.searchfield.ui.SearchTextField} Search text field.
	 */
	getAdvanceSearchField: function()
	{
		return this.getSearchFieldContainer().searchTextField;
	},

	/**
	 * Function is used to retrieve the {@link Zarafa.common.searchfield.ui.SearchFolderCombo SearchFolderCombo}.
	 * @return {Zarafa.common.searchfield.ui.SearchFolderCombo} Search folder combo
	 */
	getSearchFolderCombo: function()
	{
		return this.getSearchFieldContainer().searchFolderCombo;
	},

	/**
	 * Update the components with the given record.
	 *
	 * @param {Zarafa.core.data.MAPIRecord} record The record to update in this component
	 * @param {Boolean} contentReset force the component to perform a full update of the data.
	 * @private
	 */
	update: function(record , contentReset)
	{
		this.record = record;
	},

	/**
	 * Function was used to get the right search toolbar.
	 * @returns {Object} return right search tool bar
	 */
	getRightSearchToolbar: function()
	{
		return this.rightSearchToolbar;
	},

	/**
	 * Called when one of the "Reply"/"Reply All"/"Forward"/"Edit as New" menuitems are clicked from
	 * right toolbar of search tool bar.
	 * @param {Ext.Button} button The button which was clicked
	 * @private
	 */
	onResponse: function(button)
	{
		var mailContextModel = container.getContextByName('mail').getModel();
		Zarafa.mail.Actions.openCreateMailResponseContent(this.record, mailContextModel, button.responseMode);
	}
});

Ext.reg('zarafa.searchtoolbarpanel', Zarafa.advancesearch.dialogs.SearchToolbarPanel);

Ext.namespace('Zarafa.advancesearch.dialogs');

/**
 * @class Zarafa.advancesearch.dialogs.SelectFolderPanel
 * @extends Ext.Panel
 * @xtype zarafa.selectfolderpanel
 *
 * Panel for users to select the {@link Zarafa.core.data.IPFRecord folder}
 * on which search can perform.
 */
Zarafa.advancesearch.dialogs.SelectFolderPanel = Ext.extend(Ext.Panel, {
	/**
	 * {@link Zarafa.common.searchfield.ui.SearchFolderCombo SearchFolderCombo} contains the search folders
	 * which used in search operation.
	 * @type Object
	 * @property searchFolderCombo
	 */
	searchFolderCombo: undefined,

	/**
	 * @constructor
	 * @param {Object} config Configuration structure
	 */
	constructor: function(config)
	{
		config = config || {};

		Ext.applyIf(config, {
			xtype: 'zarafa.selectfolderpanel',
			layout: {
				type: 'fit',
				align: 'stretch'
			},
			border: false,
			searchFolderCombo: config.searchFolderCombo,
			header: false,
			items: [
				this.createTreePanel()
			],
			buttonAlign: 'right',
			buttons: [{
				text: _('Ok'),
				handler: this.onOk,
				disabled: true,
				ref: '../okButton',
				cls: 'zarafa-action',
				scope: this
			},{
				text: _('Cancel'),
				disabled: true,
				ref: '../cancelButton',
				handler: this.onCancel,
				scope: this
			}]
		});

		Zarafa.advancesearch.dialogs.SelectFolderPanel.superclass.constructor.call(this, config);
	},

	/**
	 * Initialize events
	 * @private
	 */
	initEvents: function ()
	{
		Zarafa.advancesearch.dialogs.SelectFolderPanel.superclass.initEvents.apply(this, arguments);
		this.mon(this.hierarchyTree, 'load', this.onTreeNodeLoad, this);
		this.mon(this.hierarchyTree.getSelectionModel(), 'selectionchange', this.onSelectionChange, this);
	},

	/**
	 * Creates a {@link Zarafa.hierarchy.ui.Tree treepanel}
	 * which contains all the {@link Zarafa.hierarchy.data.MAPIFolderRecord folders}
	 * on which search get perform.
	 * @return {Object} Configuration object for the tree panel.
	 * @private
	 */
	createTreePanel: function()
	{
		return {
			xtype: 'panel',
			layout: {
				type: 'vbox',
				align: 'stretch'
			},
			defaults: {
				margins: "0 0 5 0"
			},
			border: false,
			flex: 1,
			bodyStyle: 'background-color: inherit;',
			items: [{
				xtype: 'zarafa.hierarchytree',
				flex: 1,
				border: true,
				treeSorter: true,
				hideFavorites: true,
				enableDD: false,
				anchor: '100% 90%',
				ref: '../hierarchyTree'
			},{
				xtype: "checkbox",
				hideLabel: true,
				ref: '../includeSubFolder',
				boxLabel: _('Include subfolders')
			}]
		};
	},

	/**
	 * Event handler which is triggered when the user select a {@link Zarafa.hierarchy.data.MAPIFolderRecord folder}
	 * from the {@link Zarafa.hierarchy.ui.Tree tree}. This will determine if a valid
	 * {@link Zarafa.hierarchy.data.MAPIFolderRecord folder} is selected on which search gets performed.
	 *
	 * @param {DefaultSelectionModel} selectionModel The selectionModel for the treepanel
	 * @param {TreeNode} node The selected tree node
	 * @private
	 */
	onSelectionChange: function(selectionModel, node)
	{
		if (!Ext.isDefined(node) || (node.getFolder().isIPMSubTree() && this.objectType == Zarafa.core.mapi.ObjectType.MAPI_MESSAGE)) {
			this.okButton.disable();
			this.cancelButton.disable();
		} else {
			this.okButton.enable();
			this.cancelButton.enable();
			this.updateIncludeSubFolderCheckBox(node);
		}
	},

	/**
	 * Function is used to disable "Include sub folder" checkbox and apply tooltip when "All folders" option is
	 * selected from {@link Zarafa.common.searchfield.ui.SearchFolderCombo SearchFolderCombo} or IPM_SUBTREE folder
	 * selected from hierarchy and enable as well as remove tooltip from "Include Sub folder" checkbox if other then
	 * "All folders" option is selected in {@link Zarafa.common.searchfield.ui.SearchFolderCombo SearchFolderCombo} or
	 * hierarchy
	 *
	 * @param {Ext.tree.TreeNode} node The selected tree node
	 */
	updateIncludeSubFolderCheckBox: function(node)
	{
		var supportSearchFolder = this.model.supportsSearchFolder(node.getFolder());
		this.includeSubFolder.setVisible(supportSearchFolder);
		if (supportSearchFolder) {
			var record = this.searchFolderCombo.findRecord('value', node.getFolder().get('entryid'));
			var isChecked = false;
			var isDisabled = false;
			if(Ext.isDefined(record)) {
				isChecked = record.get('include_subfolder');
				isDisabled = record.get('flag') === Zarafa.advancesearch.data.SearchComboBoxFieldsFlags.ALL_FOLDERS;
			}
			var subFolderCheckBox = this.includeSubFolder;
			subFolderCheckBox.setValue(isChecked);
			subFolderCheckBox.setDisabled(isDisabled);

			if(subFolderCheckBox.rendered) {
				// Add tooltip on "include subfolder" check box when "All folders"
				// was selected in search folder combo box else remove tooltip from
				// "include subfolder" check box
				if (isDisabled) {
					subFolderCheckBox.wrap.dom.qtip = _("All folders are selected");
				} else {
					delete(subFolderCheckBox.wrap.dom.qtip);
				}
			}
		}
	},

	/**
	 * Fired when the {@link Zarafa.hierarchy.ui.Tree Tree} fires the {@link Zarafa.hierarchy.ui.Tree#load load}
	 * event. This function will try to select the {@link Ext.tree.TreeNode TreeNode} in
	 * {@link Zarafa.hierarchy.ui.Tree Tree} initially. When the given node is not loaded yet, it will try again
	 * later when the event is fired again.
	 *
	 * @private
	 */
	onTreeNodeLoad: function()
	{
		// Select folder in hierarchy tree.
		var folder = container.getHierarchyStore().getFolder(this.searchFolderCombo.getValue());

		// If the folder could be selected, then unregister the event handler.
		if (this.hierarchyTree.selectFolderInTree(folder)) {
			this.mun(this.hierarchyTree, 'load', this.onTreeNodeLoad, this);
		}
	},

	/**
	 * Event handler which is triggered when the user presses the ok
	 * {@link Ext.Button button}. This will add selected {@link Zarafa.core.data.IPFRecord folder}
	 * into {@link Zarafa.common.searchfield.ui.SearchFolderCombo SearchFolderCombo}, if folder
	 * is not already exists in {@link Zarafa.common.searchfield.ui.SearchFolderCombo SearchFolderCombo}.
	 * Also it will use to check/un-check "Include sub folder" checkbox which belongs to
	 * {@link Zarafa.advancesearch.dialogs.SearchToolBoxPanel SearchToolBox}.
	 *
	 * @private
	 */
	onOk: function ()
	{
		var folder = this.hierarchyTree.getSelectionModel().getSelectedNode().getFolder();

		if (!Ext.isDefined(folder)) {
			return;
		}
		var includeSubFolder = this.includeSubFolder.checked;
		var store = this.searchFolderCombo.getStore();
		var record = store.getAt(store.findExact("value", folder.get('entryid')));
		if (!Ext.isDefined(record)) {
			var importedFolderFlag = Zarafa.advancesearch.data.SearchComboBoxFieldsFlags.IMPORTED_FOLDER;
			if (store.getAt(0).get('flag') === importedFolderFlag) {
				store.removeAt(0);
			}
			record = new Ext.data.Record({
				'name': folder.get('display_name'),
				'value': folder.get('entryid'),
				'flag': importedFolderFlag,
				'include_subfolder': includeSubFolder
			});
			store.insert(0, record);
		} else {
			record.set('include_subfolder', includeSubFolder);
		}
		this.searchFolderCombo.setValue(record.get('value'));
		this.searchFolderCombo.onSelect(record, 0);
		this.dialog.close();
	},

	/**
	 * Event handler which is triggered when the user presses the cancel
	 * {@link Ext.Button button}. This will close the {@link Zarafa.advancesearch.dialogs.SelectFolderPanel dialog}
	 * without adding any {@link Ext.data.Record records} in search combo box.
	 * @private
	 */
	onCancel: function()
	{
		this.dialog.close();
	}
});

Ext.reg('zarafa.selectfolderpanel', Zarafa.advancesearch.dialogs.SelectFolderPanel);
Ext.namespace('Zarafa.advancesearch.ui');

/**
 * @class Zarafa.advancesearch.ui.AdvanceSearchRowSelectionModel
 * @extends Ext.grid.RowSelectionModel
 *
 * The {@link Zarafa.advancesearch.ui.MailRowSelectionModel MailRowSelectionModel}
 * is the {@link Ext.grid.RowSelectionModel RowSelectionModel} used in the
 * {@link Zarafa.advancesearch.ui.SearchGrid SearchGrid}.
 * It checks columns that have the preventRowSelection property and triggers actions depending on it.
 * See {@link #handleMouseDown}
 */
Zarafa.advancesearch.ui.AdvanceSearchRowSelectionModel = Ext.extend(Ext.grid.RowSelectionModel, {
	/**
	 * If one of the columns of interest is clicked, then the row should not be selected.
	 * Otherwise call original handler.
	 * 'cellmousedown' does not fire when drag&drop is installed on a component,
	 * otherwise it may have been possible to cancel selection from there by returning false.
	 *
	 * @param {Zarafa.advancesearch.ui.SearchGrid} grid The search grid from which the event came
	 * @param {Number} rowIndex Index of the row that was clicked
	 * @param {Ext.EventObject} event The mouse event
	 *
	 * @override
	 * @private
	 */
	handleMouseDown: function(grid, rowIndex, event)
	{
		// boolean to determine what we are going to do
		var preventRowSelection = false;
		if(event.target.className.indexOf('icon') !== -1){
			 preventRowSelection = true;
		}

		if (preventRowSelection !== true) {
			Zarafa.advancesearch.ui.AdvanceSearchRowSelectionModel.superclass.handleMouseDown.call(this, grid, rowIndex, event);
		}
	}
});

Ext.reg('zarafa.advancesearchrowselectionmodel', Zarafa.advancesearch.ui.AdvanceSearchRowSelectionModel);
Ext.namespace('Zarafa.advancesearch.ui');

/**
 * @class Zarafa.advancesearch.ui.SearchGridRenderers
 * Methods of this object can be used as renderers for search grid panels, to render
 * cells in custom format according to data type
 * @singleton
 */
Zarafa.advancesearch.ui.SearchGridRenderers = {

	/**
	 * Render the subject and body information of record
	 *
	 * @param {Object} value The data value for the cell.
	 * @param {Object} p An object with metadata
	 * @param {Ext.data.record} record The {Ext.data.Record} from which the data was extracted.
	 * @return {String} The formatted string
	 */
	subjectWithBodyColumn: function (value, p, record)
	{
		p.css = 'search-data';

		if (Ext.isEmpty(value)) {
			// if value is empty then add extra css class for empty cell
			p.css += ' zarafa-grid-empty-cell';
		}

		var messageClass = record.get('message_class');

		var subject = '';
		var body = '';

		switch (messageClass) {
			case 'IPM.Contact':
				subject = Zarafa.advancesearch.ui.SearchGridRenderers.phoneNumber(record.get('home_telephone_number'), {}, record);
				body = Ext.util.Format.htmlEncode(record.get('email_address_1'));
				break;
			case 'IPM.Task':
				body = Ext.util.Format.htmlEncode(record.get('body'));
				break;
			case 'IPM.StickyNote':
				subject = Ext.util.Format.htmlEncode(record.get('body'));
				break;
			default:
				//case 'IPM.Note':
				//case 'IPM.Appointment':
				//case 'IPM.Schedule':
				subject = Ext.util.Format.htmlEncode(record.get('subject'));
				body = Ext.util.Format.htmlEncode(record.get('body'));
				break;
		}
		return '<div>' + ( subject ? '<span class="subject">' + subject + '</span>' : '' ) + ( body ? '<span class="body">' + body + '</span>' : '' ) + '</div>';
	},

	/**
	 * Render the name information of record
	 *
	 * @param {Object} value The data value for the cell.
	 * @param {Object} p An object with metadata
	 * @param {Ext.data.record} record The {Ext.data.Record} from which the data was extracted.
	 * @return {String} The formatted string
	 */
	nameColumn: function (value, p, record)
	{
		var messageClass = record.get('message_class');
		var userRecord = false;
		var isItemSentByUser = Ext.isFunction(record.senderIsReceiver) ? !record.senderIsReceiver() && record.senderIsUser() : false;

		if (isItemSentByUser && !Ext.isEmpty(record.get('display_to'))) {
			value = record.get('display_to');
		} else {
			value = record.get('sent_representing_name');
		}

		if (Ext.isEmpty(value) && Ext.isFunction(record.getSender)) {
			value = record.get('sender_name');
			userRecord = record.getSender();
		} else if (Ext.isFunction(record.getSentRepresenting)) {
			userRecord = record.getSentRepresenting();
		}

		switch (messageClass) {
			case 'IPM.Contact':
				return Ext.util.Format.htmlEncode(record.get('display_name'));
			case 'IPM.StickyNote':
				return Ext.util.Format.htmlEncode(record.get('subject'));
			case 'IPM.Schedule.Meeting.Request':
				value = _('With') + ': ' + value;
				break;
			case 'IPM.TaskRequest':
				if (!isItemSentByUser) {
					value = record.get('display_to');
				}
				break;
			case 'IPM.Note':
				if (isItemSentByUser) {
					value = _('To') + ': ' + value;
				}
		}
		var userName = Ext.util.Format.htmlEncode(value);
		return Zarafa.common.ui.grid.Renderers.presenceStatus(userName, p, userRecord);
	},

	/**
	 * Render the date field.
	 *
	 * @param {Object} value The data value for the cell.
	 * @param {Object} p An object with metadata
	 * @param {Ext.data.record} record The {Ext.data.Record} from which the data was extracted.
	 * @return {String} The formatted string
	 */
	dateColumn: function (value, p, record)
	{
		p.css = 'search-date';

		var date = '';
		if (Ext.isDate(value)) {
			if ( container.getSettingsModel().get('zarafa/v1/main/datetime_display_format') === 'short' ){
				// Add one class that the tooltip can use to recognize a 'nice' date.
				// Add one class so the tooltip can easily get the timestamp of the date.
				p.css += ' k-date-nice k-ts-'+value.getTime();

				date = value.getNiceFormat();
			} else {
				date = value.format(_('d/m/Y'));
			}
		} else if (record.isMessageClass('IPM.Task')){
			date = _('No date');
		}

		return '<table cellpadding=0 cellspacing=0 style="width:100%"><tr>' +
			'<td class="date"><div>' + date + '</div></td>' +
			'</tr></table>';
	},

	/**
	 * Render the phone number field.
	 *
	 * @param {Object} value The data value for the cell.
	 * @param {Object} p An object with metadata
	 * @param {Ext.data.record} record The {Ext.data.Record} from which the data was extracted.
	 * @return {String} The formatted string
	 */
	phoneNumber: function (value, p, record)
	{
		p.css = '';
		var phoneNumberProps = ['business_telephone_number', 'home_telephone_number', 'business_fax_number', 'cellular_telephone_number'];

		var phoneNumberProp = phoneNumberProps.find(function (prop) {
			if (!Ext.isEmpty(record.get(prop))) {
				return true;
			}
		});

		if (phoneNumberProp) {
			return Ext.util.Format.htmlEncode(record.get(phoneNumberProp));
		} else {
			return '';
		}
	}
};
Ext.namespace('Zarafa.calendar');

/**
 * @class Zarafa.calendar.Actions
 * Common actions which can be used within {@link Ext.Button buttons}
 * or other {@link Ext.Component components} with action handlers.
 * @singleton
 */
Zarafa.calendar.Actions = {
	/**
	 * Opens a {@link Zarafa.calendar.dialogs.AppointmentContentPanel AppointmentContentPanel} for
	 * viewing an appointment.
	 *
	 * @param {Zarafa.core.data.IPMRecord} records The record, or records to which will be responded.
	 * @param {Object} config (optional) configuration object used to create the ContentPanel
	 */
	openAppointmentContent: function(records, config)
	{
		Ext.each(records, function(record) {
			// If the appointment is a series, then we need to ask the user
			// if he wants to open the occurrence or the series.
			if (Ext.isDefined(record.isRecurringOccurrence) && record.isRecurringOccurrence()) {
				Zarafa.common.Actions.openRecurringSelectionContent(record, function(button, radio) {
					// Action cancelled.
					if (button != 'ok') {
						return;
					}
					if (Ext.isEmpty(record.getStore())) {
						record = this.getById(record.get('entryid'));
					}
					// Convert the record to the requested type
					if (radio.id !== 'recurrence_series') {
						record = record.convertToOccurrenceRecord();
					} else {
						record = record.convertToSeriesRecord();
					}

					Zarafa.core.data.UIFactory.openViewRecord(record, config);
				}, record.getStore());
			} else {
				Zarafa.core.data.UIFactory.openViewRecord(record, config);
			}
		}, this);
	},

	/**
	 * Opens a {@link Zarafa.calendar.dialogs.AppointmentContentPanel AppointmentContentPanel} for creating a new appointment
	 *
	 * @param {Zarafa.calendar.CalendarContextModel} model The context model, which is used to create a new record. The record is blank but contains default values, etc.
	 * @param {Object} config (optional) Configuration object used to create the ContentPanel
	 */
	openCreateAppointmentContent: function(model, config)
	{
		model.createRecord(function(record){
			Zarafa.core.data.UIFactory.openCreateRecord(record, config);
		}.createDelegate(this, [config], true));
	},

	/**
	 * Open a Panel in which the {@link Zarafa.core.data.