/**
 * TinyMCE version 8.2.1 (2025-11-06)
 */

(function () {
    'use strict';

    /* eslint-disable @typescript-eslint/no-wrapper-object-types */
    const getPrototypeOf$2 = Object.getPrototypeOf;
    const hasProto = (v, constructor, predicate) => {
        if (predicate(v, constructor.prototype)) {
            return true;
        }
        else {
            // String-based fallback time
            return v.constructor?.name === constructor.name;
        }
    };
    const typeOf = (x) => {
        const t = typeof x;
        if (x === null) {
            return 'null';
        }
        else if (t === 'object' && Array.isArray(x)) {
            return 'array';
        }
        else if (t === 'object' && hasProto(x, String, (o, proto) => proto.isPrototypeOf(o))) {
            return 'string';
        }
        else {
            return t;
        }
    };
    const isType$1 = (type) => (value) => typeOf(value) === type;
    const isSimpleType = (type) => (value) => typeof value === type;
    const eq$1 = (t) => (a) => t === a;
    const is$2 = (value, constructor) => isObject(value) && hasProto(value, constructor, (o, proto) => getPrototypeOf$2(o) === proto);
    const isString = isType$1('string');
    const isObject = isType$1('object');
    const isPlainObject = (value) => is$2(value, Object);
    const isArray = isType$1('array');
    const isNull = eq$1(null);
    const isBoolean = isSimpleType('boolean');
    const isUndefined = eq$1(undefined);
    const isNullable = (a) => a === null || a === undefined;
    const isNonNullable = (a) => !isNullable(a);
    const isFunction = isSimpleType('function');
    const isNumber = isSimpleType('number');
    const isArrayOf = (value, pred) => {
        if (isArray(value)) {
            for (let i = 0, len = value.length; i < len; ++i) {
                if (!(pred(value[i]))) {
                    return false;
                }
            }
            return true;
        }
        return false;
    };

    const noop = () => { };
    const noarg = (f) => () => f();
    /** Compose a unary function with an n-ary function */
    const compose = (fa, fb) => {
        return (...args) => {
            return fa(fb.apply(null, args));
        };
    };
    /** Compose two unary functions. Similar to compose, but avoids using Function.prototype.apply. */
    const compose1 = (fbc, fab) => (a) => fbc(fab(a));
    const constant$1 = (value) => {
        return () => {
            return value;
        };
    };
    const identity = (x) => {
        return x;
    };
    const tripleEquals = (a, b) => {
        return a === b;
    };
    function curry(fn, ...initialArgs) {
        return (...restArgs) => {
            const all = initialArgs.concat(restArgs);
            return fn.apply(null, all);
        };
    }
    const not = (f) => (t) => !f(t);
    const die = (msg) => {
        return () => {
            throw new Error(msg);
        };
    };
    const apply$1 = (f) => {
        return f();
    };
    const never = constant$1(false);
    const always = constant$1(true);

    /**
     * The `Optional` type represents a value (of any type) that potentially does
     * not exist. Any `Optional<T>` can either be a `Some<T>` (in which case the
     * value does exist) or a `None` (in which case the value does not exist). This
     * module defines a whole lot of FP-inspired utility functions for dealing with
     * `Optional` objects.
     *
     * Comparison with null or undefined:
     * - We don't get fancy null coalescing operators with `Optional`
     * - We do get fancy helper functions with `Optional`
     * - `Optional` support nesting, and allow for the type to still be nullable (or
     * another `Optional`)
     * - There is no option to turn off strict-optional-checks like there is for
     * strict-null-checks
     */
    class Optional {
        tag;
        value;
        // Sneaky optimisation: every instance of Optional.none is identical, so just
        // reuse the same object
        static singletonNone = new Optional(false);
        // The internal representation has a `tag` and a `value`, but both are
        // private: able to be console.logged, but not able to be accessed by code
        constructor(tag, value) {
            this.tag = tag;
            this.value = value;
        }
        // --- Identities ---
        /**
         * Creates a new `Optional<T>` that **does** contain a value.
         */
        static some(value) {
            return new Optional(true, value);
        }
        /**
         * Create a new `Optional<T>` that **does not** contain a value. `T` can be
         * any type because we don't actually have a `T`.
         */
        static none() {
            return Optional.singletonNone;
        }
        /**
         * Perform a transform on an `Optional` type. Regardless of whether this
         * `Optional` contains a value or not, `fold` will return a value of type `U`.
         * If this `Optional` does not contain a value, the `U` will be created by
         * calling `onNone`. If this `Optional` does contain a value, the `U` will be
         * created by calling `onSome`.
         *
         * For the FP enthusiasts in the room, this function:
         * 1. Could be used to implement all of the functions below
         * 2. Forms a catamorphism
         */
        fold(onNone, onSome) {
            if (this.tag) {
                return onSome(this.value);
            }
            else {
                return onNone();
            }
        }
        /**
         * Determine if this `Optional` object contains a value.
         */
        isSome() {
            return this.tag;
        }
        /**
         * Determine if this `Optional` object **does not** contain a value.
         */
        isNone() {
            return !this.tag;
        }
        // --- Functor (name stolen from Haskell / maths) ---
        /**
         * Perform a transform on an `Optional` object, **if** there is a value. If
         * you provide a function to turn a T into a U, this is the function you use
         * to turn an `Optional<T>` into an `Optional<U>`. If this **does** contain
         * a value then the output will also contain a value (that value being the
         * output of `mapper(this.value)`), and if this **does not** contain a value
         * then neither will the output.
         */
        map(mapper) {
            if (this.tag) {
                return Optional.some(mapper(this.value));
            }
            else {
                return Optional.none();
            }
        }
        // --- Monad (name stolen from Haskell / maths) ---
        /**
         * Perform a transform on an `Optional` object, **if** there is a value.
         * Unlike `map`, here the transform itself also returns an `Optional`.
         */
        bind(binder) {
            if (this.tag) {
                return binder(this.value);
            }
            else {
                return Optional.none();
            }
        }
        // --- Traversable (name stolen from Haskell / maths) ---
        /**
         * For a given predicate, this function finds out if there **exists** a value
         * inside this `Optional` object that meets the predicate. In practice, this
         * means that for `Optional`s that do not contain a value it returns false (as
         * no predicate-meeting value exists).
         */
        exists(predicate) {
            return this.tag && predicate(this.value);
        }
        /**
         * For a given predicate, this function finds out if **all** the values inside
         * this `Optional` object meet the predicate. In practice, this means that
         * for `Optional`s that do not contain a value it returns true (as all 0
         * objects do meet the predicate).
         */
        forall(predicate) {
            return !this.tag || predicate(this.value);
        }
        filter(predicate) {
            if (!this.tag || predicate(this.value)) {
                return this;
            }
            else {
                return Optional.none();
            }
        }
        // --- Getters ---
        /**
         * Get the value out of the inside of the `Optional` object, using a default
         * `replacement` value if the provided `Optional` object does not contain a
         * value.
         */
        getOr(replacement) {
            return this.tag ? this.value : replacement;
        }
        /**
         * Get the value out of the inside of the `Optional` object, using a default
         * `replacement` value if the provided `Optional` object does not contain a
         * value.  Unlike `getOr`, in this method the `replacement` object is also
         * `Optional` - meaning that this method will always return an `Optional`.
         */
        or(replacement) {
            return this.tag ? this : replacement;
        }
        /**
         * Get the value out of the inside of the `Optional` object, using a default
         * `replacement` value if the provided `Optional` object does not contain a
         * value. Unlike `getOr`, in this method the `replacement` value is
         * "thunked" - that is to say that you don't pass a value to `getOrThunk`, you
         * pass a function which (if called) will **return** the `value` you want to
         * use.
         */
        getOrThunk(thunk) {
            return this.tag ? this.value : thunk();
        }
        /**
         * Get the value out of the inside of the `Optional` object, using a default
         * `replacement` value if the provided Optional object does not contain a
         * value.
         *
         * Unlike `or`, in this method the `replacement` value is "thunked" - that is
         * to say that you don't pass a value to `orThunk`, you pass a function which
         * (if called) will **return** the `value` you want to use.
         *
         * Unlike `getOrThunk`, in this method the `replacement` value is also
         * `Optional`, meaning that this method will always return an `Optional`.
         */
        orThunk(thunk) {
            return this.tag ? this : thunk();
        }
        /**
         * Get the value out of the inside of the `Optional` object, throwing an
         * exception if the provided `Optional` object does not contain a value.
         *
         * WARNING:
         * You should only be using this function if you know that the `Optional`
         * object **is not** empty (otherwise you're throwing exceptions in production
         * code, which is bad).
         *
         * In tests this is more acceptable.
         *
         * Prefer other methods to this, such as `.each`.
         */
        getOrDie(message) {
            if (!this.tag) {
                throw new Error(message ?? 'Called getOrDie on None');
            }
            else {
                return this.value;
            }
        }
        // --- Interop with null and undefined ---
        /**
         * Creates an `Optional` value from a nullable (or undefined-able) input.
         * Null, or undefined, is converted to `None`, and anything else is converted
         * to `Some`.
         */
        static from(value) {
            return isNonNullable(value) ? Optional.some(value) : Optional.none();
        }
        /**
         * Converts an `Optional` to a nullable type, by getting the value if it
         * exists, or returning `null` if it does not.
         */
        getOrNull() {
            return this.tag ? this.value : null;
        }
        /**
         * Converts an `Optional` to an undefined-able type, by getting the value if
         * it exists, or returning `undefined` if it does not.
         */
        getOrUndefined() {
            return this.value;
        }
        // --- Utilities ---
        /**
         * If the `Optional` contains a value, perform an action on that value.
         * Unlike the rest of the methods on this type, `.each` has side-effects. If
         * you want to transform an `Optional<T>` **into** something, then this is not
         * the method for you. If you want to use an `Optional<T>` to **do**
         * something, then this is the method for you - provided you're okay with not
         * doing anything in the case where the `Optional` doesn't have a value inside
         * it. If you're not sure whether your use-case fits into transforming
         * **into** something or **doing** something, check whether it has a return
         * value. If it does, you should be performing a transform.
         */
        each(worker) {
            if (this.tag) {
                worker(this.value);
            }
        }
        /**
         * Turn the `Optional` object into an array that contains all of the values
         * stored inside the `Optional`. In practice, this means the output will have
         * either 0 or 1 elements.
         */
        toArray() {
            return this.tag ? [this.value] : [];
        }
        /**
         * Turn the `Optional` object into a string for debugging or printing. Not
         * recommended for production code, but good for debugging. Also note that
         * these days an `Optional` object can be logged to the console directly, and
         * its inner value (if it exists) will be visible.
         */
        toString() {
            return this.tag ? `some(${this.value})` : 'none()';
        }
    }

    const nativeSlice = Array.prototype.slice;
    const nativeIndexOf = Array.prototype.indexOf;
    const nativePush = Array.prototype.push;
    const rawIndexOf = (ts, t) => nativeIndexOf.call(ts, t);
    const indexOf = (xs, x) => {
        // The rawIndexOf method does not wrap up in an option. This is for performance reasons.
        const r = rawIndexOf(xs, x);
        return r === -1 ? Optional.none() : Optional.some(r);
    };
    const contains$2 = (xs, x) => rawIndexOf(xs, x) > -1;
    const exists = (xs, pred) => {
        for (let i = 0, len = xs.length; i < len; i++) {
            const x = xs[i];
            if (pred(x, i)) {
                return true;
            }
        }
        return false;
    };
    const range$2 = (num, f) => {
        const r = [];
        for (let i = 0; i < num; i++) {
            r.push(f(i));
        }
        return r;
    };
    // It's a total micro optimisation, but these do make some difference.
    // Particularly for browsers other than Chrome.
    // - length caching
    // http://jsperf.com/browser-diet-jquery-each-vs-for-loop/69
    // - not using push
    // http://jsperf.com/array-direct-assignment-vs-push/2
    const chunk$1 = (array, size) => {
        const r = [];
        for (let i = 0; i < array.length; i += size) {
            const s = nativeSlice.call(array, i, i + size);
            r.push(s);
        }
        return r;
    };
    const map$2 = (xs, f) => {
        // pre-allocating array size when it's guaranteed to be known
        // http://jsperf.com/push-allocated-vs-dynamic/22
        const len = xs.length;
        const r = new Array(len);
        for (let i = 0; i < len; i++) {
            const x = xs[i];
            r[i] = f(x, i);
        }
        return r;
    };
    // Unwound implementing other functions in terms of each.
    // The code size is roughly the same, and it should allow for better optimisation.
    // const each = function<T, U>(xs: T[], f: (x: T, i?: number, xs?: T[]) => void): void {
    const each$1 = (xs, f) => {
        for (let i = 0, len = xs.length; i < len; i++) {
            const x = xs[i];
            f(x, i);
        }
    };
    const eachr = (xs, f) => {
        for (let i = xs.length - 1; i >= 0; i--) {
            const x = xs[i];
            f(x, i);
        }
    };
    const partition$3 = (xs, pred) => {
        const pass = [];
        const fail = [];
        for (let i = 0, len = xs.length; i < len; i++) {
            const x = xs[i];
            const arr = pred(x, i) ? pass : fail;
            arr.push(x);
        }
        return { pass, fail };
    };
    const filter$2 = (xs, pred) => {
        const r = [];
        for (let i = 0, len = xs.length; i < len; i++) {
            const x = xs[i];
            if (pred(x, i)) {
                r.push(x);
            }
        }
        return r;
    };
    const foldr = (xs, f, acc) => {
        eachr(xs, (x, i) => {
            acc = f(acc, x, i);
        });
        return acc;
    };
    const foldl = (xs, f, acc) => {
        each$1(xs, (x, i) => {
            acc = f(acc, x, i);
        });
        return acc;
    };
    const findUntil = (xs, pred, until) => {
        for (let i = 0, len = xs.length; i < len; i++) {
            const x = xs[i];
            if (pred(x, i)) {
                return Optional.some(x);
            }
            else if (until(x, i)) {
                break;
            }
        }
        return Optional.none();
    };
    const find$5 = (xs, pred) => {
        return findUntil(xs, pred, never);
    };
    const findIndex$1 = (xs, pred) => {
        for (let i = 0, len = xs.length; i < len; i++) {
            const x = xs[i];
            if (pred(x, i)) {
                return Optional.some(i);
            }
        }
        return Optional.none();
    };
    const flatten = (xs) => {
        // Note, this is possible because push supports multiple arguments:
        // http://jsperf.com/concat-push/6
        // Note that in the past, concat() would silently work (very slowly) for array-like objects.
        // With this change it will throw an error.
        const r = [];
        for (let i = 0, len = xs.length; i < len; ++i) {
            // Ensure that each value is an array itself
            if (!isArray(xs[i])) {
                throw new Error('Arr.flatten item ' + i + ' was not an array, input: ' + xs);
            }
            nativePush.apply(r, xs[i]);
        }
        return r;
    };
    const bind$3 = (xs, f) => flatten(map$2(xs, f));
    const forall = (xs, pred) => {
        for (let i = 0, len = xs.length; i < len; ++i) {
            const x = xs[i];
            if (pred(x, i) !== true) {
                return false;
            }
        }
        return true;
    };
    const reverse = (xs) => {
        const r = nativeSlice.call(xs, 0);
        r.reverse();
        return r;
    };
    const difference = (a1, a2) => filter$2(a1, (x) => !contains$2(a2, x));
    const mapToObject = (xs, f) => {
        const r = {};
        for (let i = 0, len = xs.length; i < len; i++) {
            const x = xs[i];
            r[String(x)] = f(x, i);
        }
        return r;
    };
    const pure$2 = (x) => [x];
    const sort = (xs, comparator) => {
        const copy = nativeSlice.call(xs, 0);
        copy.sort(comparator);
        return copy;
    };
    const get$i = (xs, i) => i >= 0 && i < xs.length ? Optional.some(xs[i]) : Optional.none();
    const head = (xs) => get$i(xs, 0);
    const last$1 = (xs) => get$i(xs, xs.length - 1);
    const from = isFunction(Array.from) ? Array.from : (x) => nativeSlice.call(x);
    const findMap = (arr, f) => {
        for (let i = 0; i < arr.length; i++) {
            const r = f(arr[i], i);
            if (r.isSome()) {
                return r;
            }
        }
        return Optional.none();
    };

    // There are many variations of Object iteration that are faster than the 'for-in' style:
    // http://jsperf.com/object-keys-iteration/107
    //
    // Use the native keys if it is available (IE9+), otherwise fall back to manually filtering
    const keys = Object.keys;
    const hasOwnProperty = Object.hasOwnProperty;
    const each = (obj, f) => {
        const props = keys(obj);
        for (let k = 0, len = props.length; k < len; k++) {
            const i = props[k];
            const x = obj[i];
            f(x, i);
        }
    };
    const map$1 = (obj, f) => {
        return tupleMap(obj, (x, i) => ({
            k: i,
            v: f(x, i)
        }));
    };
    const tupleMap = (obj, f) => {
        const r = {};
        each(obj, (x, i) => {
            const tuple = f(x, i);
            r[tuple.k] = tuple.v;
        });
        return r;
    };
    const objAcc = (r) => (x, i) => {
        r[i] = x;
    };
    const internalFilter = (obj, pred, onTrue, onFalse) => {
        each(obj, (x, i) => {
            (pred(x, i) ? onTrue : onFalse)(x, i);
        });
    };
    const bifilter = (obj, pred) => {
        const t = {};
        const f = {};
        internalFilter(obj, pred, objAcc(t), objAcc(f));
        return { t, f };
    };
    const filter$1 = (obj, pred) => {
        const t = {};
        internalFilter(obj, pred, objAcc(t), noop);
        return t;
    };
    const mapToArray = (obj, f) => {
        const r = [];
        each(obj, (value, name) => {
            r.push(f(value, name));
        });
        return r;
    };
    const find$4 = (obj, pred) => {
        const props = keys(obj);
        for (let k = 0, len = props.length; k < len; k++) {
            const i = props[k];
            const x = obj[i];
            if (pred(x, i, obj)) {
                return Optional.some(x);
            }
        }
        return Optional.none();
    };
    const values = (obj) => {
        return mapToArray(obj, identity);
    };
    const get$h = (obj, key) => {
        return has$2(obj, key) ? Optional.from(obj[key]) : Optional.none();
    };
    const has$2 = (obj, key) => hasOwnProperty.call(obj, key);
    const hasNonNullableKey = (obj, key) => has$2(obj, key) && obj[key] !== undefined && obj[key] !== null;

    /*
     * Generates a church encoded ADT (https://en.wikipedia.org/wiki/Church_encoding)
     * For syntax and use, look at the test code.
     */
    const generate$7 = (cases) => {
        // validation
        if (!isArray(cases)) {
            throw new Error('cases must be an array');
        }
        if (cases.length === 0) {
            throw new Error('there must be at least one case');
        }
        const constructors = [];
        // adt is mutated to add the individual cases
        const adt = {};
        each$1(cases, (acase, count) => {
            const keys$1 = keys(acase);
            // validation
            if (keys$1.length !== 1) {
                throw new Error('one and only one name per case');
            }
            const key = keys$1[0];
            const value = acase[key];
            // validation
            if (adt[key] !== undefined) {
                throw new Error('duplicate key detected:' + key);
            }
            else if (key === 'cata') {
                throw new Error('cannot have a case named cata (sorry)');
            }
            else if (!isArray(value)) {
                // this implicitly checks if acase is an object
                throw new Error('case arguments must be an array');
            }
            constructors.push(key);
            //
            // constructor for key
            //
            adt[key] = (...args) => {
                const argLength = args.length;
                // validation
                if (argLength !== value.length) {
                    throw new Error('Wrong number of arguments to case ' + key + '. Expected ' + value.length + ' (' + value + '), got ' + argLength);
                }
                const match = (branches) => {
                    const branchKeys = keys(branches);
                    if (constructors.length !== branchKeys.length) {
                        throw new Error('Wrong number of arguments to match. Expected: ' + constructors.join(',') + '\nActual: ' + branchKeys.join(','));
                    }
                    const allReqd = forall(constructors, (reqKey) => {
                        return contains$2(branchKeys, reqKey);
                    });
                    if (!allReqd) {
                        throw new Error('Not all branches were specified when using match. Specified: ' + branchKeys.join(', ') + '\nRequired: ' + constructors.join(', '));
                    }
                    return branches[key].apply(null, args);
                };
                //
                // the fold function for key
                //
                return {
                    fold: (...foldArgs) => {
                        // runtime validation
                        if (foldArgs.length !== cases.length) {
                            throw new Error('Wrong number of arguments to fold. Expected ' + cases.length + ', got ' + foldArgs.length);
                        }
                        const target = foldArgs[count];
                        return target.apply(null, args);
                    },
                    match,
                    // NOTE: Only for debugging.
                    log: (label) => {
                        // eslint-disable-next-line no-console
                        console.log(label, {
                            constructors,
                            constructor: key,
                            params: args
                        });
                    }
                };
            };
        });
        return adt;
    };
    const Adt = {
        generate: generate$7
    };

    const Cell = (initial) => {
        let value = initial;
        const get = () => {
            return value;
        };
        const set = (v) => {
            value = v;
        };
        return {
            get,
            set
        };
    };

    const nu$d = (baseFn) => {
        let data = Optional.none();
        let callbacks = [];
        /** map :: this LazyValue a -> (a -> b) -> LazyValue b */
        const map = (f) => nu$d((nCallback) => {
            get((data) => {
                nCallback(f(data));
            });
        });
        const get = (nCallback) => {
            if (isReady()) {
                call(nCallback);
            }
            else {
                callbacks.push(nCallback);
            }
        };
        const set = (x) => {
            if (!isReady()) {
                data = Optional.some(x);
                run(callbacks);
                callbacks = [];
            }
        };
        const isReady = () => data.isSome();
        const run = (cbs) => {
            each$1(cbs, call);
        };
        const call = (cb) => {
            data.each((x) => {
                setTimeout(() => {
                    cb(x);
                }, 0);
            });
        };
        // Lazy values cache the value and kick off immediately
        baseFn(set);
        return {
            get,
            map,
            isReady
        };
    };
    const pure$1 = (a) => nu$d((callback) => {
        callback(a);
    });
    const LazyValue = {
        nu: nu$d,
        pure: pure$1
    };

    const errorReporter = (err) => {
        // we can not throw the error in the reporter as it will just be black-holed
        // by the Promise so we use a setTimeout to escape the Promise.
        setTimeout(() => {
            throw err;
        }, 0);
    };
    const make$8 = (run) => {
        const get = (callback) => {
            run().then(callback, errorReporter);
        };
        /** map :: this Future a -> (a -> b) -> Future b */
        const map = (fab) => {
            return make$8(() => run().then(fab));
        };
        /** bind :: this Future a -> (a -> Future b) -> Future b */
        const bind = (aFutureB) => {
            return make$8(() => run().then((v) => aFutureB(v).toPromise()));
        };
        /** anonBind :: this Future a -> Future b -> Future b
         *  Returns a future, which evaluates the first future, ignores the result, then evaluates the second.
         */
        const anonBind = (futureB) => {
            return make$8(() => run().then(() => futureB.toPromise()));
        };
        const toLazy = () => {
            return LazyValue.nu(get);
        };
        const toCached = () => {
            let cache = null;
            return make$8(() => {
                if (cache === null) {
                    cache = run();
                }
                return cache;
            });
        };
        const toPromise = run;
        return {
            map,
            bind,
            anonBind,
            toLazy,
            toCached,
            toPromise,
            get
        };
    };
    const nu$c = (baseFn) => {
        return make$8(() => new Promise(baseFn));
    };
    /** a -> Future a */
    const pure = (a) => {
        return make$8(() => Promise.resolve(a));
    };
    const Future = {
        nu: nu$c,
        pure
    };

    /**
     * Creates a new `Result<T, E>` that **does** contain a value.
     */
    const value$4 = (value) => {
        const applyHelper = (fn) => fn(value);
        const constHelper = constant$1(value);
        const outputHelper = () => output;
        const output = {
            // Debug info
            tag: true,
            inner: value,
            // Actual Result methods
            fold: (_onError, onValue) => onValue(value),
            isValue: always,
            isError: never,
            map: (mapper) => Result.value(mapper(value)),
            mapError: outputHelper,
            bind: applyHelper,
            exists: applyHelper,
            forall: applyHelper,
            getOr: constHelper,
            or: outputHelper,
            getOrThunk: constHelper,
            orThunk: outputHelper,
            getOrDie: constHelper,
            each: (fn) => {
                // Can't write the function inline because we don't want to return something by mistake
                fn(value);
            },
            toOptional: () => Optional.some(value),
        };
        return output;
    };
    /**
     * Creates a new `Result<T, E>` that **does not** contain a value, and therefore
     * contains an error.
     */
    const error$1 = (error) => {
        const outputHelper = () => output;
        const output = {
            // Debug info
            tag: false,
            inner: error,
            // Actual Result methods
            fold: (onError, _onValue) => onError(error),
            isValue: never,
            isError: always,
            map: outputHelper,
            mapError: (mapper) => Result.error(mapper(error)),
            bind: outputHelper,
            exists: never,
            forall: always,
            getOr: identity,
            or: identity,
            getOrThunk: apply$1,
            orThunk: apply$1,
            getOrDie: die(String(error)),
            each: noop,
            toOptional: Optional.none,
        };
        return output;
    };
    /**
     * Creates a new `Result<T, E>` from an `Optional<T>` and an `E`. If the
     * `Optional` contains a value, so will the outputted `Result`. If it does not,
     * the outputted `Result` will contain an error (and that error will be the
     * error passed in).
     */
    const fromOption = (optional, err) => optional.fold(() => error$1(err), value$4);
    const Result = {
        value: value$4,
        error: error$1,
        fromOption
    };

    const wrap$2 = (delegate) => {
        const toCached = () => {
            return wrap$2(delegate.toCached());
        };
        const bindFuture = (f) => {
            return wrap$2(delegate.bind((resA) => resA.fold((err) => (Future.pure(Result.error(err))), (a) => f(a))));
        };
        const bindResult = (f) => {
            return wrap$2(delegate.map((resA) => resA.bind(f)));
        };
        const mapResult = (f) => {
            return wrap$2(delegate.map((resA) => resA.map(f)));
        };
        const mapError = (f) => {
            return wrap$2(delegate.map((resA) => resA.mapError(f)));
        };
        const foldResult = (whenError, whenValue) => {
            return delegate.map((res) => res.fold(whenError, whenValue));
        };
        const withTimeout = (timeout, errorThunk) => {
            return wrap$2(Future.nu((callback) => {
                let timedOut = false;
                const timer = setTimeout(() => {
                    timedOut = true;
                    callback(Result.error(errorThunk()));
                }, timeout);
                delegate.get((result) => {
                    if (!timedOut) {
                        clearTimeout(timer);
                        callback(result);
                    }
                });
            }));
        };
        return {
            ...delegate,
            toCached,
            bindFuture,
            bindResult,
            mapResult,
            mapError,
            foldResult,
            withTimeout
        };
    };
    const nu$b = (worker) => {
        return wrap$2(Future.nu(worker));
    };
    const value$3 = (value) => {
        return wrap$2(Future.pure(Result.value(value)));
    };
    const error = (error) => {
        return wrap$2(Future.pure(Result.error(error)));
    };
    const fromResult$1 = (result) => {
        return wrap$2(Future.pure(result));
    };
    const fromFuture = (future) => {
        return wrap$2(future.map(Result.value));
    };
    const fromPromise = (promise) => {
        return nu$b((completer) => {
            promise.then((value) => {
                completer(Result.value(value));
            }, (error) => {
                completer(Result.error(error));
            });
        });
    };
    const FutureResult = {
        nu: nu$b,
        wrap: wrap$2,
        pure: value$3,
        value: value$3,
        error,
        fromResult: fromResult$1,
        fromFuture,
        fromPromise
    };

    // Use window object as the global if it's available since CSP will block script evals
    // eslint-disable-next-line @typescript-eslint/no-implied-eval
    const Global = typeof window !== 'undefined' ? window : Function('return this;')();

    /**
     * Adds two numbers, and wrap to a range.
     * If the result overflows to the right, snap to the left.
     * If the result overflows to the left, snap to the right.
     */
    const cycleBy = (value, delta, min, max) => {
        const r = value + delta;
        if (r > max) {
            return min;
        }
        else if (r < min) {
            return max;
        }
        else {
            return r;
        }
    };
    // ASSUMPTION: Max will always be larger than min
    const clamp = (value, min, max) => Math.min(Math.max(value, min), max);
    // the division is meant to get a number between 0 and 1 for more information check this discussion: https://stackoverflow.com/questions/58285941/how-to-replace-math-random-with-crypto-getrandomvalues-and-keep-same-result
    const random = () => window.crypto.getRandomValues(new Uint32Array(1))[0] / 4294967295;

    /**
     * Generate a unique identifier.
     *
     * The unique portion of the identifier only contains an underscore
     * and digits, so that it may safely be used within HTML attributes.
     *
     * The chance of generating a non-unique identifier has been minimized
     * by combining the current time, a random number and a one-up counter.
     *
     * generate :: String -> String
     */
    let unique = 0;
    const generate$6 = (prefix) => {
        const date = new Date();
        const time = date.getTime();
        const random$1 = Math.floor(random() * 1000000000);
        unique++;
        return prefix + '_' + random$1 + unique + String(time);
    };

    const shallow$1 = (old, nu) => {
        return nu;
    };
    const deep$1 = (old, nu) => {
        const bothObjects = isPlainObject(old) && isPlainObject(nu);
        return bothObjects ? deepMerge(old, nu) : nu;
    };
    const baseMerge = (merger) => {
        return (...objects) => {
            if (objects.length === 0) {
                throw new Error(`Can't merge zero objects`);
            }
            const ret = {};
            for (let j = 0; j < objects.length; j++) {
                const curObject = objects[j];
                for (const key in curObject) {
                    if (has$2(curObject, key)) {
                        ret[key] = merger(ret[key], curObject[key]);
                    }
                }
            }
            return ret;
        };
    };
    const deepMerge = baseMerge(deep$1);
    const merge$1 = baseMerge(shallow$1);

    /**
     * **Is** the value stored inside this Optional object equal to `rhs`?
     */
    const is$1 = (lhs, rhs, comparator = tripleEquals) => lhs.exists((left) => comparator(left, rhs));
    /**
     * Are these two Optional objects equal? Equality here means either they're both
     * `Some` (and the values are equal under the comparator) or they're both `None`.
     */
    const equals = (lhs, rhs, comparator = tripleEquals) => lift2(lhs, rhs, comparator).getOr(lhs.isNone() && rhs.isNone());
    const cat = (arr) => {
        const r = [];
        const push = (x) => {
            r.push(x);
        };
        for (let i = 0; i < arr.length; i++) {
            arr[i].each(push);
        }
        return r;
    };
    const sequence = (arr) => {
        const r = [];
        for (let i = 0; i < arr.length; i++) {
            const x = arr[i];
            if (x.isSome()) {
                r.push(x.getOrDie());
            }
            else {
                return Optional.none();
            }
        }
        return Optional.some(r);
    };
    /*
    Notes on the lift functions:
    - We used to have a generic liftN, but we were concerned about its type-safety, and the below variants were faster in microbenchmarks.
    - The getOrDie calls are partial functions, but are checked beforehand. This is faster and more convenient (but less safe) than folds.
    - && is used instead of a loop for simplicity and performance.
    */
    const lift2 = (oa, ob, f) => oa.isSome() && ob.isSome() ? Optional.some(f(oa.getOrDie(), ob.getOrDie())) : Optional.none();
    const lift3 = (oa, ob, oc, f) => oa.isSome() && ob.isSome() && oc.isSome() ? Optional.some(f(oa.getOrDie(), ob.getOrDie(), oc.getOrDie())) : Optional.none();
    const mapFrom = (a, f) => (a !== undefined && a !== null) ? Optional.some(f(a)) : Optional.none();
    // This can help with type inference, by specifying the type param on the none case, so the caller doesn't have to.
    const someIf = (b, a) => b ? Optional.some(a) : Optional.none();

    // https://developer.mozilla.org/en-US/docs/Web/JavaScript/Guide/Regular_Expressions#escaping
    const escape = (text) => text.replace(/[.*+?^${}()|[\]\\]/g, '\\$&');

    /** path :: ([String], JsObj?) -> JsObj */
    const path$1 = (parts, scope) => {
        let o = scope !== undefined && scope !== null ? scope : Global;
        for (let i = 0; i < parts.length && o !== undefined && o !== null; ++i) {
            o = o[parts[i]];
        }
        return o;
    };
    /** resolve :: (String, JsObj?) -> JsObj */
    const resolve = (p, scope) => {
        const parts = p.split('.');
        return path$1(parts, scope);
    };

    Adt.generate([
        { bothErrors: ['error1', 'error2'] },
        { firstError: ['error1', 'value2'] },
        { secondError: ['value1', 'error2'] },
        { bothValues: ['value1', 'value2'] }
    ]);
    /** partition :: [Result a] -> { errors: [String], values: [a] } */
    const partition$2 = (results) => {
        const errors = [];
        const values = [];
        each$1(results, (result) => {
            result.fold((err) => {
                errors.push(err);
            }, (value) => {
                values.push(value);
            });
        });
        return { errors, values };
    };

    const singleton$1 = (doRevoke) => {
        const subject = Cell(Optional.none());
        const revoke = () => subject.get().each(doRevoke);
        const clear = () => {
            revoke();
            subject.set(Optional.none());
        };
        const isSet = () => subject.get().isSome();
        const get = () => subject.get();
        const set = (s) => {
            revoke();
            subject.set(Optional.some(s));
        };
        return {
            clear,
            isSet,
            get,
            set
        };
    };
    const destroyable = () => singleton$1((s) => s.destroy());
    const unbindable = () => singleton$1((s) => s.unbind());
    const value$2 = () => {
        const subject = singleton$1(noop);
        const on = (f) => subject.get().each(f);
        return {
            ...subject,
            on
        };
    };

    const addToEnd = (str, suffix) => {
        return str + suffix;
    };
    const removeFromStart = (str, numChars) => {
        return str.substring(numChars);
    };

    const checkRange = (str, substr, start) => substr === '' || str.length >= substr.length && str.substr(start, start + substr.length) === substr;
    const removeLeading = (str, prefix) => {
        return startsWith(str, prefix) ? removeFromStart(str, prefix.length) : str;
    };
    const ensureTrailing = (str, suffix) => {
        return endsWith(str, suffix) ? str : addToEnd(str, suffix);
    };
    const contains$1 = (str, substr, start = 0, end) => {
        const idx = str.indexOf(substr, start);
        if (idx !== -1) {
            return isUndefined(end) ? true : idx + substr.length <= end;
        }
        else {
            return false;
        }
    };
    /** Does 'str' start with 'prefix'?
     *  Note: all strings start with the empty string.
     *        More formally, for all strings x, startsWith(x, "").
     *        This is so that for all strings x and y, startsWith(y + x, y)
     */
    const startsWith = (str, prefix) => {
        return checkRange(str, prefix, 0);
    };
    /** Does 'str' end with 'suffix'?
     *  Note: all strings end with the empty string.
     *        More formally, for all strings x, endsWith(x, "").
     *        This is so that for all strings x and y, endsWith(x + y, y)
     */
    const endsWith = (str, suffix) => {
        return checkRange(str, suffix, str.length - suffix.length);
    };
    const blank = (r) => (s) => s.replace(r, '');
    /** removes all leading and trailing spaces */
    const trim$1 = blank(/^\s+|\s+$/g);
    const isNotEmpty = (s) => s.length > 0;
    const isEmpty = (s) => !isNotEmpty(s);
    const toFloat = (value) => {
        const num = parseFloat(value);
        return isNaN(num) ? Optional.none() : Optional.some(num);
    };

    // Run a function fn after rate ms. If another invocation occurs
    // during the time it is waiting, update the arguments f will run
    // with (but keep the current schedule)
    const adaptable = (fn, rate) => {
        let timer = null;
        let args = null;
        const cancel = () => {
            if (!isNull(timer)) {
                clearTimeout(timer);
                timer = null;
                args = null;
            }
        };
        const throttle = (...newArgs) => {
            args = newArgs;
            if (isNull(timer)) {
                timer = setTimeout(() => {
                    const tempArgs = args;
                    timer = null;
                    args = null;
                    fn.apply(null, tempArgs);
                }, rate);
            }
        };
        return {
            cancel,
            throttle
        };
    };
    // Run a function fn after rate ms. If another invocation occurs
    // during the time it is waiting, ignore it completely.
    const first$1 = (fn, rate) => {
        let timer = null;
        const cancel = () => {
            if (!isNull(timer)) {
                clearTimeout(timer);
                timer = null;
            }
        };
        const throttle = (...args) => {
            if (isNull(timer)) {
                timer = setTimeout(() => {
                    timer = null;
                    fn.apply(null, args);
                }, rate);
            }
        };
        return {
            cancel,
            throttle
        };
    };
    // Run a function fn after rate ms. If another invocation occurs
    // during the time it is waiting, reschedule the function again
    // with the new arguments.
    const last = (fn, rate) => {
        let timer = null;
        const cancel = () => {
            if (!isNull(timer)) {
                clearTimeout(timer);
                timer = null;
            }
        };
        const throttle = (...args) => {
            cancel();
            timer = setTimeout(() => {
                timer = null;
                fn.apply(null, args);
            }, rate);
        };
        return {
            cancel,
            throttle
        };
    };

    const cached = (f) => {
        let called = false;
        let r;
        return (...args) => {
            if (!called) {
                called = true;
                r = f.apply(null, args);
            }
            return r;
        };
    };

    const zeroWidth = '\uFEFF';
    const nbsp = '\u00A0';

    const fromHtml$2 = (html, scope) => {
        const doc = scope || document;
        const div = doc.createElement('div');
        div.innerHTML = html;
        if (!div.hasChildNodes() || div.childNodes.length > 1) {
            const message = 'HTML does not have a single root node';
            // eslint-disable-next-line no-console
            console.error(message, html);
            throw new Error(message);
        }
        return fromDom(div.childNodes[0]);
    };
    const fromTag = (tag, scope) => {
        const doc = scope || document;
        const node = doc.createElement(tag);
        return fromDom(node);
    };
    const fromText = (text, scope) => {
        const doc = scope || document;
        const node = doc.createTextNode(text);
        return fromDom(node);
    };
    const fromDom = (node) => {
        // TODO: Consider removing this check, but left atm for safety
        if (node === null || node === undefined) {
            throw new Error('Node cannot be null or undefined');
        }
        return {
            dom: node
        };
    };
    const fromPoint = (docElm, x, y) => Optional.from(docElm.dom.elementFromPoint(x, y)).map(fromDom);
    // tslint:disable-next-line:variable-name
    const SugarElement = {
        fromHtml: fromHtml$2,
        fromTag,
        fromText,
        fromDom,
        fromPoint
    };

    // NOTE: Mutates the range.
    const setStart = (rng, situ) => {
        situ.fold((e) => {
            rng.setStartBefore(e.dom);
        }, (e, o) => {
            rng.setStart(e.dom, o);
        }, (e) => {
            rng.setStartAfter(e.dom);
        });
    };
    const setFinish = (rng, situ) => {
        situ.fold((e) => {
            rng.setEndBefore(e.dom);
        }, (e, o) => {
            rng.setEnd(e.dom, o);
        }, (e) => {
            rng.setEndAfter(e.dom);
        });
    };
    const relativeToNative = (win, startSitu, finishSitu) => {
        const range = win.document.createRange();
        setStart(range, startSitu);
        setFinish(range, finishSitu);
        return range;
    };
    const exactToNative = (win, start, soffset, finish, foffset) => {
        const rng = win.document.createRange();
        rng.setStart(start.dom, soffset);
        rng.setEnd(finish.dom, foffset);
        return rng;
    };
    const toRect = (rect) => ({
        left: rect.left,
        top: rect.top,
        right: rect.right,
        bottom: rect.bottom,
        width: rect.width,
        height: rect.height
    });
    const getFirstRect$1 = (rng) => {
        const rects = rng.getClientRects();
        // ASSUMPTION: The first rectangle is the start of the selection
        const rect = rects.length > 0 ? rects[0] : rng.getBoundingClientRect();
        return rect.width > 0 || rect.height > 0 ? Optional.some(rect).map(toRect) : Optional.none();
    };
    const getBounds$3 = (rng) => {
        const rect = rng.getBoundingClientRect();
        return rect.width > 0 || rect.height > 0 ? Optional.some(rect).map(toRect) : Optional.none();
    };

    const adt$a = Adt.generate([
        { ltr: ['start', 'soffset', 'finish', 'foffset'] },
        { rtl: ['start', 'soffset', 'finish', 'foffset'] }
    ]);
    const fromRange = (win, type, range) => type(SugarElement.fromDom(range.startContainer), range.startOffset, SugarElement.fromDom(range.endContainer), range.endOffset);
    const getRanges = (win, selection) => selection.match({
        domRange: (rng) => {
            return {
                ltr: constant$1(rng),
                rtl: Optional.none
            };
        },
        relative: (startSitu, finishSitu) => {
            return {
                ltr: cached(() => relativeToNative(win, startSitu, finishSitu)),
                rtl: cached(() => Optional.some(relativeToNative(win, finishSitu, startSitu)))
            };
        },
        exact: (start, soffset, finish, foffset) => {
            return {
                ltr: cached(() => exactToNative(win, start, soffset, finish, foffset)),
                rtl: cached(() => Optional.some(exactToNative(win, finish, foffset, start, soffset)))
            };
        }
    });
    const doDiagnose = (win, ranges) => {
        // If we cannot create a ranged selection from start > finish, it could be RTL
        const rng = ranges.ltr();
        if (rng.collapsed) {
            // Let's check if it's RTL ... if it is, then reversing the direction will not be collapsed
            const reversed = ranges.rtl().filter((rev) => rev.collapsed === false);
            return reversed.map((rev) => 
            // We need to use "reversed" here, because the original only has one point (collapsed)
            adt$a.rtl(SugarElement.fromDom(rev.endContainer), rev.endOffset, SugarElement.fromDom(rev.startContainer), rev.startOffset)).getOrThunk(() => fromRange(win, adt$a.ltr, rng));
        }
        else {
            return fromRange(win, adt$a.ltr, rng);
        }
    };
    const diagnose = (win, selection) => {
        const ranges = getRanges(win, selection);
        return doDiagnose(win, ranges);
    };
    const asLtrRange = (win, selection) => {
        const diagnosis = diagnose(win, selection);
        return diagnosis.match({
            ltr: (start, soffset, finish, foffset) => {
                const rng = win.document.createRange();
                rng.setStart(start.dom, soffset);
                rng.setEnd(finish.dom, foffset);
                return rng;
            },
            rtl: (start, soffset, finish, foffset) => {
                // NOTE: Reversing start and finish
                const rng = win.document.createRange();
                rng.setStart(finish.dom, foffset);
                rng.setEnd(start.dom, soffset);
                return rng;
            }
        });
    };
    adt$a.ltr;
    adt$a.rtl;

    const DOCUMENT = 9;
    const DOCUMENT_FRAGMENT = 11;
    const ELEMENT = 1;
    const TEXT = 3;

    const is = (element, selector) => {
        const dom = element.dom;
        if (dom.nodeType !== ELEMENT) {
            return false;
        }
        else {
            const elem = dom;
            if (elem.matches !== undefined) {
                return elem.matches(selector);
            }
            else if (elem.msMatchesSelector !== undefined) {
                return elem.msMatchesSelector(selector);
            }
            else if (elem.webkitMatchesSelector !== undefined) {
                return elem.webkitMatchesSelector(selector);
            }
            else if (elem.mozMatchesSelector !== undefined) {
                // cast to any as mozMatchesSelector doesn't exist in TS DOM lib
                return elem.mozMatchesSelector(selector);
            }
            else {
                throw new Error('Browser lacks native selectors');
            } // unfortunately we can't throw this on startup :(
        }
    };
    const bypassSelector = (dom) => 
    // Only elements, documents and shadow roots support querySelector
    // shadow root element type is DOCUMENT_FRAGMENT
    dom.nodeType !== ELEMENT && dom.nodeType !== DOCUMENT && dom.nodeType !== DOCUMENT_FRAGMENT ||
        // IE fix for complex queries on empty nodes: http://jsfiddle.net/spyder/fv9ptr5L/
        dom.childElementCount === 0;
    const all$3 = (selector, scope) => {
        const base = scope === undefined ? document : scope.dom;
        return bypassSelector(base) ? [] : map$2(base.querySelectorAll(selector), SugarElement.fromDom);
    };
    const one = (selector, scope) => {
        const base = scope === undefined ? document : scope.dom;
        return bypassSelector(base) ? Optional.none() : Optional.from(base.querySelector(selector)).map(SugarElement.fromDom);
    };

    const eq = (e1, e2) => e1.dom === e2.dom;
    // Returns: true if node e1 contains e2, otherwise false.
    // (returns false if e1===e2: A node does not contain itself).
    const contains = (e1, e2) => {
        const d1 = e1.dom;
        const d2 = e2.dom;
        return d1 === d2 ? false : d1.contains(d2);
    };

    const DeviceType = (os, browser, userAgent, mediaMatch) => {
        const isiPad = os.isiOS() && /ipad/i.test(userAgent) === true;
        const isiPhone = os.isiOS() && !isiPad;
        const isMobile = os.isiOS() || os.isAndroid();
        const isTouch = isMobile || mediaMatch('(pointer:coarse)');
        const isTablet = isiPad || !isiPhone && isMobile && mediaMatch('(min-device-width:768px)');
        const isPhone = isiPhone || isMobile && !isTablet;
        const iOSwebview = browser.isSafari() && os.isiOS() && /safari/i.test(userAgent) === false;
        const isDesktop = !isPhone && !isTablet && !iOSwebview;
        return {
            isiPad: constant$1(isiPad),
            isiPhone: constant$1(isiPhone),
            isTablet: constant$1(isTablet),
            isPhone: constant$1(isPhone),
            isTouch: constant$1(isTouch),
            isAndroid: os.isAndroid,
            isiOS: os.isiOS,
            isWebView: constant$1(iOSwebview),
            isDesktop: constant$1(isDesktop)
        };
    };

    const firstMatch = (regexes, s) => {
        for (let i = 0; i < regexes.length; i++) {
            const x = regexes[i];
            if (x.test(s)) {
                return x;
            }
        }
        return undefined;
    };
    const find$3 = (regexes, agent) => {
        const r = firstMatch(regexes, agent);
        if (!r) {
            return { major: 0, minor: 0 };
        }
        const group = (i) => {
            return Number(agent.replace(r, '$' + i));
        };
        return nu$a(group(1), group(2));
    };
    const detect$4 = (versionRegexes, agent) => {
        const cleanedAgent = String(agent).toLowerCase();
        if (versionRegexes.length === 0) {
            return unknown$3();
        }
        return find$3(versionRegexes, cleanedAgent);
    };
    const unknown$3 = () => {
        return nu$a(0, 0);
    };
    const nu$a = (major, minor) => {
        return { major, minor };
    };
    const Version = {
        nu: nu$a,
        detect: detect$4,
        unknown: unknown$3
    };

    const detectBrowser$1 = (browsers, userAgentData) => {
        return findMap(userAgentData.brands, (uaBrand) => {
            const lcBrand = uaBrand.brand.toLowerCase();
            return find$5(browsers, (browser) => lcBrand === browser.brand?.toLowerCase())
                .map((info) => ({
                current: info.name,
                version: Version.nu(parseInt(uaBrand.version, 10), 0)
            }));
        });
    };

    const detect$3 = (candidates, userAgent) => {
        const agent = String(userAgent).toLowerCase();
        return find$5(candidates, (candidate) => {
            return candidate.search(agent);
        });
    };
    // They (browser and os) are the same at the moment, but they might
    // not stay that way.
    const detectBrowser = (browsers, userAgent) => {
        return detect$3(browsers, userAgent).map((browser) => {
            const version = Version.detect(browser.versionRegexes, userAgent);
            return {
                current: browser.name,
                version
            };
        });
    };
    const detectOs = (oses, userAgent) => {
        return detect$3(oses, userAgent).map((os) => {
            const version = Version.detect(os.versionRegexes, userAgent);
            return {
                current: os.name,
                version
            };
        });
    };

    const normalVersionRegex = /.*?version\/\ ?([0-9]+)\.([0-9]+).*/;
    const checkContains = (target) => {
        return (uastring) => {
            return contains$1(uastring, target);
        };
    };
    const browsers = [
        // This is legacy Edge
        {
            name: 'Edge',
            versionRegexes: [/.*?edge\/ ?([0-9]+)\.([0-9]+)$/],
            search: (uastring) => {
                return contains$1(uastring, 'edge/') && contains$1(uastring, 'chrome') && contains$1(uastring, 'safari') && contains$1(uastring, 'applewebkit');
            }
        },
        // This is Google Chrome and Chromium Edge
        {
            name: 'Chromium',
            brand: 'Chromium',
            versionRegexes: [/.*?chrome\/([0-9]+)\.([0-9]+).*/, normalVersionRegex],
            search: (uastring) => {
                return contains$1(uastring, 'chrome') && !contains$1(uastring, 'chromeframe');
            }
        },
        {
            name: 'IE',
            versionRegexes: [/.*?msie\ ?([0-9]+)\.([0-9]+).*/, /.*?rv:([0-9]+)\.([0-9]+).*/],
            search: (uastring) => {
                return contains$1(uastring, 'msie') || contains$1(uastring, 'trident');
            }
        },
        // INVESTIGATE: Is this still the Opera user agent?
        {
            name: 'Opera',
            versionRegexes: [normalVersionRegex, /.*?opera\/([0-9]+)\.([0-9]+).*/],
            search: checkContains('opera')
        },
        {
            name: 'Firefox',
            versionRegexes: [/.*?firefox\/\ ?([0-9]+)\.([0-9]+).*/],
            search: checkContains('firefox')
        },
        {
            name: 'Safari',
            versionRegexes: [normalVersionRegex, /.*?cpu os ([0-9]+)_([0-9]+).*/],
            search: (uastring) => {
                return (contains$1(uastring, 'safari') || contains$1(uastring, 'mobile/')) && contains$1(uastring, 'applewebkit');
            }
        }
    ];
    const oses = [
        {
            name: 'Windows',
            search: checkContains('win'),
            versionRegexes: [/.*?windows\ nt\ ?([0-9]+)\.([0-9]+).*/]
        },
        {
            name: 'iOS',
            search: (uastring) => {
                return contains$1(uastring, 'iphone') || contains$1(uastring, 'ipad');
            },
            versionRegexes: [/.*?version\/\ ?([0-9]+)\.([0-9]+).*/, /.*cpu os ([0-9]+)_([0-9]+).*/, /.*cpu iphone os ([0-9]+)_([0-9]+).*/]
        },
        {
            name: 'Android',
            search: checkContains('android'),
            versionRegexes: [/.*?android\ ?([0-9]+)\.([0-9]+).*/]
        },
        {
            name: 'macOS',
            search: checkContains('mac os x'),
            versionRegexes: [/.*?mac\ os\ x\ ?([0-9]+)_([0-9]+).*/]
        },
        {
            name: 'Linux',
            search: checkContains('linux'),
            versionRegexes: []
        },
        { name: 'Solaris',
            search: checkContains('sunos'),
            versionRegexes: []
        },
        {
            name: 'FreeBSD',
            search: checkContains('freebsd'),
            versionRegexes: []
        },
        {
            name: 'ChromeOS',
            search: checkContains('cros'),
            versionRegexes: [/.*?chrome\/([0-9]+)\.([0-9]+).*/]
        }
    ];
    const PlatformInfo = {
        browsers: constant$1(browsers),
        oses: constant$1(oses)
    };

    const edge = 'Edge';
    const chromium = 'Chromium';
    const ie = 'IE';
    const opera = 'Opera';
    const firefox = 'Firefox';
    const safari = 'Safari';
    const unknown$2 = () => {
        return nu$9({
            current: undefined,
            version: Version.unknown()
        });
    };
    const nu$9 = (info) => {
        const current = info.current;
        const version = info.version;
        const isBrowser = (name) => () => current === name;
        return {
            current,
            version,
            isEdge: isBrowser(edge),
            isChromium: isBrowser(chromium),
            // NOTE: isIe just looks too weird
            isIE: isBrowser(ie),
            isOpera: isBrowser(opera),
            isFirefox: isBrowser(firefox),
            isSafari: isBrowser(safari)
        };
    };
    const Browser = {
        unknown: unknown$2,
        nu: nu$9,
        edge: constant$1(edge),
        chromium: constant$1(chromium),
        ie: constant$1(ie),
        opera: constant$1(opera),
        firefox: constant$1(firefox),
        safari: constant$1(safari)
    };

    const windows = 'Windows';
    const ios = 'iOS';
    const android = 'Android';
    const linux = 'Linux';
    const macos = 'macOS';
    const solaris = 'Solaris';
    const freebsd = 'FreeBSD';
    const chromeos = 'ChromeOS';
    // Though there is a bit of dupe with this and Browser, trying to
    // reuse code makes it much harder to follow and change.
    const unknown$1 = () => {
        return nu$8({
            current: undefined,
            version: Version.unknown()
        });
    };
    const nu$8 = (info) => {
        const current = info.current;
        const version = info.version;
        const isOS = (name) => () => current === name;
        return {
            current,
            version,
            isWindows: isOS(windows),
            // TODO: Fix capitalisation
            isiOS: isOS(ios),
            isAndroid: isOS(android),
            isMacOS: isOS(macos),
            isLinux: isOS(linux),
            isSolaris: isOS(solaris),
            isFreeBSD: isOS(freebsd),
            isChromeOS: isOS(chromeos)
        };
    };
    const OperatingSystem = {
        unknown: unknown$1,
        nu: nu$8,
        windows: constant$1(windows),
        ios: constant$1(ios),
        android: constant$1(android),
        linux: constant$1(linux),
        macos: constant$1(macos),
        solaris: constant$1(solaris),
        freebsd: constant$1(freebsd),
        chromeos: constant$1(chromeos)
    };

    const detect$2 = (userAgent, userAgentDataOpt, mediaMatch) => {
        const browsers = PlatformInfo.browsers();
        const oses = PlatformInfo.oses();
        const browser = userAgentDataOpt.bind((userAgentData) => detectBrowser$1(browsers, userAgentData))
            .orThunk(() => detectBrowser(browsers, userAgent))
            .fold(Browser.unknown, Browser.nu);
        const os = detectOs(oses, userAgent).fold(OperatingSystem.unknown, OperatingSystem.nu);
        const deviceType = DeviceType(os, browser, userAgent, mediaMatch);
        return {
            browser,
            os,
            deviceType
        };
    };
    const PlatformDetection = {
        detect: detect$2
    };

    const mediaMatch = (query) => window.matchMedia(query).matches;
    // IMPORTANT: Must be in a thunk, otherwise rollup thinks calling this immediately
    // causes side effects and won't tree shake this away
    // Note: navigator.userAgentData is not part of the native typescript types yet
    let platform = cached(() => PlatformDetection.detect(window.navigator.userAgent, Optional.from((window.navigator.userAgentData)), mediaMatch));
    const detect$1 = () => platform();

    const unsafe = (name, scope) => {
        return resolve(name, scope);
    };
    const getOrDie$1 = (name, scope) => {
        const actual = unsafe(name, scope);
        if (actual === undefined || actual === null) {
            throw new Error(name + ' not available on this browser');
        }
        return actual;
    };

    const getPrototypeOf$1 = Object.getPrototypeOf;
    /*
     * IE9 and above
     *
     * MDN no use on this one, but here's the link anyway:
     * https://developer.mozilla.org/en/docs/Web/API/HTMLElement
     */
    const sandHTMLElement = (scope) => {
        return getOrDie$1('HTMLElement', scope);
    };
    const isPrototypeOf = (x) => {
        // use Resolve to get the window object for x and just return undefined if it can't find it.
        // undefined scope later triggers using the global window.
        const scope = resolve('ownerDocument.defaultView', x);
        // TINY-7374: We can't rely on looking at the owner window HTMLElement as the element may have
        // been constructed in a different window and then appended to the current window document.
        return isObject(x) && (sandHTMLElement(scope).prototype.isPrototypeOf(x) || /^HTML\w*Element$/.test(getPrototypeOf$1(x).constructor.name));
    };

    const name$3 = (element) => {
        const r = element.dom.nodeName;
        return r.toLowerCase();
    };
    const type$1 = (element) => element.dom.nodeType;
    const isType = (t) => (element) => type$1(element) === t;
    const isHTMLElement = (element) => isElement$1(element) && isPrototypeOf(element.dom);
    const isElement$1 = isType(ELEMENT);
    const isText = isType(TEXT);
    const isDocument = isType(DOCUMENT);
    const isDocumentFragment = isType(DOCUMENT_FRAGMENT);
    const isTag = (tag) => (e) => isElement$1(e) && name$3(e) === tag;

    /**
     * The document associated with the current element
     * NOTE: this will throw if the owner is null.
     */
    const owner$4 = (element) => SugarElement.fromDom(element.dom.ownerDocument);
    /**
     * If the element is a document, return it. Otherwise, return its ownerDocument.
     * @param dos
     */
    const documentOrOwner = (dos) => isDocument(dos) ? dos : owner$4(dos);
    const documentElement = (element) => SugarElement.fromDom(documentOrOwner(element).dom.documentElement);
    /**
     * The window element associated with the element
     * NOTE: this will throw if the defaultView is null.
     */
    const defaultView = (element) => SugarElement.fromDom(documentOrOwner(element).dom.defaultView);
    const parent = (element) => Optional.from(element.dom.parentNode).map(SugarElement.fromDom);
    // Cast down to just be SugarElement<Node>
    const parentNode = (element) => parent(element);
    const parentElement = (element) => Optional.from(element.dom.parentElement).map(SugarElement.fromDom);
    const parents = (element, isRoot) => {
        const stop = isFunction(isRoot) ? isRoot : never;
        // This is used a *lot* so it needs to be performant, not recursive
        let dom = element.dom;
        const ret = [];
        while (dom.parentNode !== null && dom.parentNode !== undefined) {
            const rawParent = dom.parentNode;
            const p = SugarElement.fromDom(rawParent);
            ret.push(p);
            if (stop(p) === true) {
                break;
            }
            else {
                dom = rawParent;
            }
        }
        return ret;
    };
    const offsetParent = (element) => Optional.from(element.dom.offsetParent).map(SugarElement.fromDom);
    const prevSibling = (element) => Optional.from(element.dom.previousSibling).map(SugarElement.fromDom);
    const nextSibling = (element) => Optional.from(element.dom.nextSibling).map(SugarElement.fromDom);
    const children = (element) => map$2(element.dom.childNodes, SugarElement.fromDom);
    const child$2 = (element, index) => {
        const cs = element.dom.childNodes;
        return Optional.from(cs[index]).map(SugarElement.fromDom);
    };
    const firstChild = (element) => child$2(element, 0);
    const spot = (element, offset) => ({
        element,
        offset
    });
    const leaf = (element, offset) => {
        const cs = children(element);
        return cs.length > 0 && offset < cs.length ? spot(cs[offset], 0) : spot(element, offset);
    };

    const makeRange = (start, soffset, finish, foffset) => {
        const doc = owner$4(start);
        // TODO: We need to think about a better place to put native range creation code. Does it even belong in sugar?
        // Could the `Compare` checks (node.compareDocumentPosition) handle these situations better?
        const rng = doc.dom.createRange();
        rng.setStart(start.dom, soffset);
        rng.setEnd(finish.dom, foffset);
        return rng;
    };
    const after$2 = (start, soffset, finish, foffset) => {
        const r = makeRange(start, soffset, finish, foffset);
        const same = eq(start, finish) && soffset === foffset;
        return r.collapsed && !same;
    };

    /**
     * Is the element a ShadowRoot?
     *
     * Note: this is insufficient to test if any element is a shadow root, but it is sufficient to differentiate between
     * a Document and a ShadowRoot.
     */
    const isShadowRoot = (dos) => isDocumentFragment(dos) && isNonNullable(dos.dom.host);
    const getRootNode = (e) => SugarElement.fromDom(e.dom.getRootNode());
    /** Where content needs to go. ShadowRoot or document body */
    const getContentContainer = (dos) => 
    // Can't use SugarBody.body without causing a circular module reference (since SugarBody.inBody uses SugarShadowDom)
    isShadowRoot(dos) ? dos : SugarElement.fromDom(documentOrOwner(dos).dom.body);
    /** Is this element either a ShadowRoot or a descendent of a ShadowRoot. */
    const isInShadowRoot = (e) => getShadowRoot(e).isSome();
    /** If this element is in a ShadowRoot, return it. */
    const getShadowRoot = (e) => {
        const r = getRootNode(e);
        return isShadowRoot(r) ? Optional.some(r) : Optional.none();
    };
    /** Return the host of a ShadowRoot.
     *
     * This function will throw if Shadow DOM is unsupported in the browser, or if the host is null.
     * If you actually have a ShadowRoot, this shouldn't happen.
     */
    const getShadowHost = (e) => SugarElement.fromDom(e.dom.host);
    /**
     * When Events bubble up through a ShadowRoot, the browser changes the target to be the shadow host.
     * This function gets the "original" event target if possible.
     * This only works if the shadow tree is open - if the shadow tree is closed, event.target is returned.
     * See: https://developers.google.com/web/fundamentals/web-components/shadowdom#events
     */
    const getOriginalEventTarget = (event) => {
        if (isNonNullable(event.target)) {
            const el = SugarElement.fromDom(event.target);
            if (isElement$1(el) && isOpenShadowHost(el)) {
                // When target element is inside Shadow DOM we need to take first element from composedPath
                // otherwise we'll get Shadow Root parent, not actual target element.
                if (event.composed && event.composedPath) {
                    const composedPath = event.composedPath();
                    if (composedPath) {
                        return head(composedPath);
                    }
                }
            }
        }
        return Optional.from(event.target);
    };
    /** Return true if the element is a host of an open shadow root.
     *  Return false if the element is a host of a closed shadow root, or if the element is not a host.
     */
    const isOpenShadowHost = (element) => isNonNullable(element.dom.shadowRoot);

    const mkEvent = (target, x, y, stop, prevent, kill, raw) => ({
        target,
        x,
        y,
        stop,
        prevent,
        kill,
        raw
    });
    /** Wraps an Event in an EventArgs structure.
     * The returned EventArgs structure has its target set to the "original" target if possible.
     * See SugarShadowDom.getOriginalEventTarget
     */
    const fromRawEvent$1 = (rawEvent) => {
        const target = SugarElement.fromDom(getOriginalEventTarget(rawEvent).getOr(rawEvent.target));
        const stop = () => rawEvent.stopPropagation();
        const prevent = () => rawEvent.preventDefault();
        const kill = compose(prevent, stop); // more of a sequence than a compose, but same effect
        // FIX: Don't just expose the raw event. Need to identify what needs standardisation.
        return mkEvent(target, rawEvent.clientX, rawEvent.clientY, stop, prevent, kill, rawEvent);
    };
    const handle = (filter, handler) => (rawEvent) => {
        if (filter(rawEvent)) {
            handler(fromRawEvent$1(rawEvent));
        }
    };
    const binder = (element, event, filter, handler, useCapture) => {
        const wrapped = handle(filter, handler);
        // IE9 minimum
        element.dom.addEventListener(event, wrapped, useCapture);
        return {
            unbind: curry(unbind, element, event, wrapped, useCapture)
        };
    };
    const bind$2 = (element, event, filter, handler) => binder(element, event, filter, handler, false);
    const capture$1 = (element, event, filter, handler) => binder(element, event, filter, handler, true);
    const unbind = (element, event, handler, useCapture) => {
        // IE9 minimum
        element.dom.removeEventListener(event, handler, useCapture);
    };

    const filter = always; // no filter on plain DomEvents
    const bind$1 = (element, event, handler) => bind$2(element, event, filter, handler);
    const capture = (element, event, handler) => capture$1(element, event, filter, handler);
    const fromRawEvent = fromRawEvent$1;

    const getDocument = () => SugarElement.fromDom(document);

    const focus$4 = (element, preventScroll = false) => element.dom.focus({ preventScroll });
    const blur$1 = (element) => element.dom.blur();
    const hasFocus = (element) => {
        const root = getRootNode(element).dom;
        return element.dom === root.activeElement;
    };
    // Note: assuming that activeElement will always be a HTMLElement (maybe we should add a runtime check?)
    const active$1 = (root = getDocument()) => Optional.from(root.dom.activeElement).map(SugarElement.fromDom);
    /**
     * Return the descendant element that has focus.
     * Use instead of SelectorFind.descendant(container, ':focus')
     *  because the :focus selector relies on keyboard focus.
     */
    const search = (element) => active$1(getRootNode(element))
        .filter((e) => element.dom.contains(e.dom));

    const before$1 = (marker, element) => {
        const parent$1 = parent(marker);
        parent$1.each((v) => {
            v.dom.insertBefore(element.dom, marker.dom);
        });
    };
    const after$1 = (marker, element) => {
        const sibling = nextSibling(marker);
        sibling.fold(() => {
            const parent$1 = parent(marker);
            parent$1.each((v) => {
                append$2(v, element);
            });
        }, (v) => {
            before$1(v, element);
        });
    };
    const prepend$1 = (parent, element) => {
        const firstChild$1 = firstChild(parent);
        firstChild$1.fold(() => {
            append$2(parent, element);
        }, (v) => {
            parent.dom.insertBefore(element.dom, v.dom);
        });
    };
    const append$2 = (parent, element) => {
        parent.dom.appendChild(element.dom);
    };
    const appendAt = (parent, element, index) => {
        child$2(parent, index).fold(() => {
            append$2(parent, element);
        }, (v) => {
            before$1(v, element);
        });
    };

    const append$1 = (parent, elements) => {
        each$1(elements, (x) => {
            append$2(parent, x);
        });
    };

    const rawSet = (dom, key, value) => {
        /*
         * JQuery coerced everything to a string, and silently did nothing on text node/null/undefined.
         *
         * We fail on those invalid cases, only allowing numbers and booleans.
         */
        if (isString(value) || isBoolean(value) || isNumber(value)) {
            dom.setAttribute(key, value + '');
        }
        else {
            // eslint-disable-next-line no-console
            console.error('Invalid call to Attribute.set. Key ', key, ':: Value ', value, ':: Element ', dom);
            throw new Error('Attribute value was not simple');
        }
    };
    const set$9 = (element, key, value) => {
        rawSet(element.dom, key, value);
    };
    const setAll$1 = (element, attrs) => {
        const dom = element.dom;
        each(attrs, (v, k) => {
            rawSet(dom, k, v);
        });
    };
    const get$g = (element, key) => {
        const v = element.dom.getAttribute(key);
        // undefined is the more appropriate value for JS, and this matches JQuery
        return v === null ? undefined : v;
    };
    const getOpt = (element, key) => Optional.from(get$g(element, key));
    const has$1 = (element, key) => {
        const dom = element.dom;
        // return false for non-element nodes, no point in throwing an error
        return dom && dom.hasAttribute ? dom.hasAttribute(key) : false;
    };
    const remove$8 = (element, key) => {
        element.dom.removeAttribute(key);
    };
    const clone$2 = (element) => foldl(element.dom.attributes, (acc, attr) => {
        acc[attr.name] = attr.value;
        return acc;
    }, {});

    const empty = (element) => {
        // shortcut "empty node" trick. Requires IE 9.
        element.dom.textContent = '';
        // If the contents was a single empty text node, the above doesn't remove it. But, it's still faster in general
        // than removing every child node manually.
        // The following is (probably) safe for performance as 99.9% of the time the trick works and
        // Traverse.children will return an empty array.
        each$1(children(element), (rogue) => {
            remove$7(rogue);
        });
    };
    const remove$7 = (element) => {
        const dom = element.dom;
        if (dom.parentNode !== null) {
            dom.parentNode.removeChild(dom);
        }
    };

    const clone$1 = (original, isDeep) => SugarElement.fromDom(original.dom.cloneNode(isDeep));
    /** Shallow clone - just the tag, no children */
    const shallow = (original) => clone$1(original, false);
    /** Deep clone - everything copied including children */
    const deep = (original) => clone$1(original, true);

    const fromHtml$1 = (html, scope) => {
        const doc = scope || document;
        const div = doc.createElement('div');
        div.innerHTML = html;
        return children(SugarElement.fromDom(div));
    };

    const get$f = (element) => element.dom.innerHTML;
    const set$8 = (element, content) => {
        const owner = owner$4(element);
        const docDom = owner.dom;
        // FireFox has *terrible* performance when using innerHTML = x
        const fragment = SugarElement.fromDom(docDom.createDocumentFragment());
        const contentElements = fromHtml$1(content, docDom);
        append$1(fragment, contentElements);
        empty(element);
        append$2(element, fragment);
    };
    const getOuter$2 = (element) => {
        const container = SugarElement.fromTag('div');
        const clone = SugarElement.fromDom(element.dom.cloneNode(true));
        append$2(container, clone);
        return get$f(container);
    };

    const getHtml = (element) => {
        if (isShadowRoot(element)) {
            return '#shadow-root';
        }
        else {
            const clone = shallow(element);
            return getOuter$2(clone);
        }
    };

    const image = (image) => new Promise((resolve, reject) => {
        const loaded = () => {
            destroy();
            resolve(image);
        };
        const listeners = [
            bind$1(image, 'load', loaded),
            bind$1(image, 'error', () => {
                destroy();
                reject('Unable to load data from image: ' + image.dom.src);
            }),
        ];
        const destroy = () => each$1(listeners, (l) => l.unbind());
        if (image.dom.complete) {
            loaded();
        }
    });

    // some elements, such as mathml, don't have style attributes
    // others, such as angular elements, have style attributes that aren't a CSSStyleDeclaration
    const isSupported = (dom) => dom.style !== undefined && isFunction(dom.style.getPropertyValue);

    // Node.contains() is very, very, very good performance
    // http://jsperf.com/closest-vs-contains/5
    const inBody = (element) => {
        // Technically this is only required on IE, where contains() returns false for text nodes.
        // But it's cheap enough to run everywhere and Sugar doesn't have platform detection (yet).
        const dom = isText(element) ? element.dom.parentNode : element.dom;
        // use ownerDocument.body to ensure this works inside iframes.
        // Normally contains is bad because an element "contains" itself, but here we want that.
        if (dom === undefined || dom === null || dom.ownerDocument === null) {
            return false;
        }
        const doc = dom.ownerDocument;
        return getShadowRoot(SugarElement.fromDom(dom)).fold(() => doc.body.contains(dom), compose1(inBody, getShadowHost));
    };
    const body = () => getBody(SugarElement.fromDom(document));
    const getBody = (doc) => {
        const b = doc.dom.body;
        if (b === null || b === undefined) {
            throw new Error('Body is not available yet');
        }
        return SugarElement.fromDom(b);
    };

    const internalSet = (dom, property, value) => {
        // This is going to hurt. Apologies.
        // JQuery coerces numbers to pixels for certain property names, and other times lets numbers through.
        // we're going to be explicit; strings only.
        if (!isString(value)) {
            // eslint-disable-next-line no-console
            console.error('Invalid call to CSS.set. Property ', property, ':: Value ', value, ':: Element ', dom);
            throw new Error('CSS value must be a string: ' + value);
        }
        // removed: support for dom().style[property] where prop is camel case instead of normal property name
        if (isSupported(dom)) {
            dom.style.setProperty(property, value);
        }
    };
    const internalRemove = (dom, property) => {
        /*
         * IE9 and above - MDN doesn't have details, but here's a couple of random internet claims
         *
         * http://help.dottoro.com/ljopsjck.php
         * http://stackoverflow.com/a/7901886/7546
         */
        if (isSupported(dom)) {
            dom.style.removeProperty(property);
        }
    };
    const set$7 = (element, property, value) => {
        const dom = element.dom;
        internalSet(dom, property, value);
    };
    const setAll = (element, css) => {
        const dom = element.dom;
        each(css, (v, k) => {
            internalSet(dom, k, v);
        });
    };
    const setOptions = (element, css) => {
        const dom = element.dom;
        each(css, (v, k) => {
            v.fold(() => {
                internalRemove(dom, k);
            }, (value) => {
                internalSet(dom, k, value);
            });
        });
    };
    /*
     * NOTE: For certain properties, this returns the "used value" which is subtly different to the "computed value" (despite calling getComputedStyle).
     * Blame CSS 2.0.
     *
     * https://developer.mozilla.org/en-US/docs/Web/CSS/used_value
     */
    const get$e = (element, property) => {
        const dom = element.dom;
        /*
         * IE9 and above per
         * https://developer.mozilla.org/en/docs/Web/API/window.getComputedStyle
         *
         * Not in numerosity, because it doesn't memoize and looking this up dynamically in performance critical code would be horrendous.
         *
         * JQuery has some magic here for IE popups, but we don't really need that.
         * It also uses element.ownerDocument.defaultView to handle iframes but that hasn't been required since FF 3.6.
         */
        const styles = window.getComputedStyle(dom);
        const r = styles.getPropertyValue(property);
        // jquery-ism: If r is an empty string, check that the element is not in a document. If it isn't, return the raw value.
        // Turns out we do this a lot.
        return (r === '' && !inBody(element)) ? getUnsafeProperty(dom, property) : r;
    };
    // removed: support for dom().style[property] where prop is camel case instead of normal property name
    // empty string is what the browsers (IE11 and Chrome) return when the propertyValue doesn't exists.
    const getUnsafeProperty = (dom, property) => isSupported(dom) ? dom.style.getPropertyValue(property) : '';
    /*
     * Gets the raw value from the style attribute. Useful for retrieving "used values" from the DOM:
     * https://developer.mozilla.org/en-US/docs/Web/CSS/used_value
     *
     * Returns NONE if the property isn't set, or the value is an empty string.
     */
    const getRaw = (element, property) => {
        const dom = element.dom;
        const raw = getUnsafeProperty(dom, property);
        return Optional.from(raw).filter((r) => r.length > 0);
    };
    const getAllRaw = (element) => {
        const css = {};
        const dom = element.dom;
        if (isSupported(dom)) {
            for (let i = 0; i < dom.style.length; i++) {
                const ruleName = dom.style.item(i);
                css[ruleName] = dom.style[ruleName];
            }
        }
        return css;
    };
    const isValidValue$1 = (tag, property, value) => {
        const element = SugarElement.fromTag(tag);
        set$7(element, property, value);
        const style = getRaw(element, property);
        return style.isSome();
    };
    const remove$6 = (element, property) => {
        const dom = element.dom;
        internalRemove(dom, property);
        if (is$1(getOpt(element, 'style').map(trim$1), '')) {
            // No more styles left, remove the style attribute as well
            remove$8(element, 'style');
        }
    };
    /* NOTE: This function is here for the side effect it triggers.
    The value itself is not used.
    Be sure to not use the return value, and that it is not removed by a minifier.
     */
    const reflow = (e) => e.dom.offsetWidth;

    const Dimension = (name, getOffset) => {
        const set = (element, h) => {
            if (!isNumber(h) && !h.match(/^[0-9]+$/)) {
                throw new Error(name + '.set accepts only positive integer values. Value was ' + h);
            }
            const dom = element.dom;
            if (isSupported(dom)) {
                dom.style[name] = h + 'px';
            }
        };
        /*
         * jQuery supports querying width and height on the document and window objects.
         *
         * TBIO doesn't do this, so the code is removed to save space, but left here just in case.
         */
        /*
        var getDocumentWidth = (element) => {
          var dom = element.dom;
          if (Node.isDocument(element)) {
            var body = dom.body;
            var doc = dom.documentElement;
            return Math.max(
              body.scrollHeight,
              doc.scrollHeight,
              body.offsetHeight,
              doc.offsetHeight,
              doc.clientHeight
            );
          }
        };
      
        var getWindowWidth = (element) => {
          var dom = element.dom;
          if (dom.window === dom) {
            // There is no offsetHeight on a window, so use the clientHeight of the document
            return dom.document.documentElement.clientHeight;
          }
        };
      */
        const get = (element) => {
            const r = getOffset(element);
            // zero or null means non-standard or disconnected, fall back to CSS
            if (r <= 0 || r === null) {
                const css = get$e(element, name);
                // ugh this feels dirty, but it saves cycles
                return parseFloat(css) || 0;
            }
            return r;
        };
        // in jQuery, getOuter replicates (or uses) box-sizing: border-box calculations
        // although these calculations only seem relevant for quirks mode, and edge cases TBIO doesn't rely on
        const getOuter = get;
        const aggregate = (element, properties) => foldl(properties, (acc, property) => {
            const val = get$e(element, property);
            const value = val === undefined ? 0 : parseInt(val, 10);
            return isNaN(value) ? acc : acc + value;
        }, 0);
        const max = (element, value, properties) => {
            const cumulativeInclusions = aggregate(element, properties);
            // if max-height is 100px and your cumulativeInclusions is 150px, there is no way max-height can be 100px, so we return 0.
            const absoluteMax = value > cumulativeInclusions ? value - cumulativeInclusions : 0;
            return absoluteMax;
        };
        return {
            set,
            get,
            getOuter,
            aggregate,
            max
        };
    };

    const api$2 = Dimension('height', (element) => {
        // getBoundingClientRect gives better results than offsetHeight for tables with captions on Firefox
        const dom = element.dom;
        return inBody(element) ? dom.getBoundingClientRect().height : dom.offsetHeight;
    });
    const get$d = (element) => api$2.get(element);
    const getOuter$1 = (element) => api$2.getOuter(element);
    const setMax$1 = (element, value) => {
        // These properties affect the absolute max-height, they are not counted natively, we want to include these properties.
        const inclusions = ['margin-top', 'border-top-width', 'padding-top', 'padding-bottom', 'border-bottom-width', 'margin-bottom'];
        const absMax = api$2.max(element, value, inclusions);
        set$7(element, 'max-height', absMax + 'px');
    };

    const isHidden$1 = (dom) => dom.offsetWidth <= 0 && dom.offsetHeight <= 0;
    const isVisible = (element) => !isHidden$1(element.dom);

    const api$1 = Dimension('width', (element) => {
        const dom = element.dom;
        return inBody(element) ? dom.getBoundingClientRect().width : dom.offsetWidth;
    });
    const set$6 = (element, h) => api$1.set(element, h);
    const get$c = (element) => api$1.get(element);
    const getOuter = (element) => api$1.getOuter(element);
    const setMax = (element, value) => {
        // These properties affect the absolute max-height, they are not counted natively, we want to include these properties.
        const inclusions = ['margin-left', 'border-left-width', 'padding-left', 'padding-right', 'border-right-width', 'margin-right'];
        const absMax = api$1.max(element, value, inclusions);
        set$7(element, 'max-width', absMax + 'px');
    };

    const r$1 = (left, top) => {
        const translate = (x, y) => r$1(left + x, top + y);
        return {
            left,
            top,
            translate
        };
    };
    // tslint:disable-next-line:variable-name
    const SugarPosition = r$1;

    const boxPosition = (dom) => {
        const box = dom.getBoundingClientRect();
        return SugarPosition(box.left, box.top);
    };
    // Avoids falsy false fallthrough
    const firstDefinedOrZero = (a, b) => {
        if (a !== undefined) {
            return a;
        }
        else {
            return b !== undefined ? b : 0;
        }
    };
    const absolute$3 = (element) => {
        const doc = element.dom.ownerDocument;
        const body = doc.body;
        const win = doc.defaultView;
        const html = doc.documentElement;
        if (body === element.dom) {
            return SugarPosition(body.offsetLeft, body.offsetTop);
        }
        const scrollTop = firstDefinedOrZero(win?.pageYOffset, html.scrollTop);
        const scrollLeft = firstDefinedOrZero(win?.pageXOffset, html.scrollLeft);
        const clientTop = firstDefinedOrZero(html.clientTop, body.clientTop);
        const clientLeft = firstDefinedOrZero(html.clientLeft, body.clientLeft);
        return viewport$1(element).translate(scrollLeft - clientLeft, scrollTop - clientTop);
    };
    const viewport$1 = (element) => {
        const dom = element.dom;
        const doc = dom.ownerDocument;
        const body = doc.body;
        if (body === dom) {
            return SugarPosition(body.offsetLeft, body.offsetTop);
        }
        if (!inBody(element)) {
            return SugarPosition(0, 0);
        }
        return boxPosition(dom);
    };

    // get scroll position (x,y) relative to document _doc (or global if not supplied)
    const get$b = (_DOC) => {
        const doc = _DOC !== undefined ? _DOC.dom : document;
        // ASSUMPTION: This is for cross-browser support, body works for Safari & EDGE, and when we have an iframe body scroller
        const x = doc.body.scrollLeft || doc.documentElement.scrollLeft;
        const y = doc.body.scrollTop || doc.documentElement.scrollTop;
        return SugarPosition(x, y);
    };
    // Scroll content to (x,y) relative to document _doc (or global if not supplied)
    const to = (x, y, _DOC) => {
        const doc = _DOC !== undefined ? _DOC.dom : document;
        const win = doc.defaultView;
        if (win) {
            win.scrollTo(x, y);
        }
    };

    const NodeValue = (is, name) => {
        const get = (element) => {
            if (!is(element)) {
                throw new Error('Can only get ' + name + ' value of a ' + name + ' node');
            }
            return getOption(element).getOr('');
        };
        const getOption = (element) => is(element) ? Optional.from(element.dom.nodeValue) : Optional.none();
        const set = (element, value) => {
            if (!is(element)) {
                throw new Error('Can only set raw ' + name + ' value of a ' + name + ' node');
            }
            element.dom.nodeValue = value;
        };
        return {
            get,
            getOption,
            set
        };
    };

    const api = NodeValue(isText, 'text');
    const get$a = (element) => api.get(element);

    const onDirection = (isLtr, isRtl) => (element) => getDirection(element) === 'rtl' ? isRtl : isLtr;
    const getDirection = (element) => get$e(element, 'direction') === 'rtl' ? 'rtl' : 'ltr';

    // Methods for handling attributes that contain a list of values <div foo="alpha beta theta">
    const read$2 = (element, attr) => {
        const value = get$g(element, attr);
        return value === undefined || value === '' ? [] : value.split(' ');
    };
    const add$4 = (element, attr, id) => {
        const old = read$2(element, attr);
        const nu = old.concat([id]);
        set$9(element, attr, nu.join(' '));
        return true;
    };
    const remove$5 = (element, attr, id) => {
        const nu = filter$2(read$2(element, attr), (v) => v !== id);
        if (nu.length > 0) {
            set$9(element, attr, nu.join(' '));
        }
        else {
            remove$8(element, attr);
        }
        return false;
    };

    var ClosestOrAncestor = (is, ancestor, scope, a, isRoot) => {
        if (is(scope, a)) {
            return Optional.some(scope);
        }
        else if (isFunction(isRoot) && isRoot(scope)) {
            return Optional.none();
        }
        else {
            return ancestor(scope, a, isRoot);
        }
    };

    const ancestor$2 = (scope, predicate, isRoot) => {
        let element = scope.dom;
        const stop = isFunction(isRoot) ? isRoot : never;
        while (element.parentNode) {
            element = element.parentNode;
            const el = SugarElement.fromDom(element);
            if (predicate(el)) {
                return Optional.some(el);
            }
            else if (stop(el)) {
                break;
            }
        }
        return Optional.none();
    };
    const closest$4 = (scope, predicate, isRoot) => {
        // This is required to avoid ClosestOrAncestor passing the predicate to itself
        const is = (s, test) => test(s);
        return ClosestOrAncestor(is, ancestor$2, scope, predicate, isRoot);
    };
    const sibling$1 = (scope, predicate) => {
        const element = scope.dom;
        if (!element.parentNode) {
            return Optional.none();
        }
        return child$1(SugarElement.fromDom(element.parentNode), (x) => !eq(scope, x) && predicate(x));
    };
    const child$1 = (scope, predicate) => {
        const pred = (node) => predicate(SugarElement.fromDom(node));
        const result = find$5(scope.dom.childNodes, pred);
        return result.map(SugarElement.fromDom);
    };
    const descendant$1 = (scope, predicate) => {
        const descend = (node) => {
            // tslint:disable-next-line:prefer-for-of
            for (let i = 0; i < node.childNodes.length; i++) {
                const child = SugarElement.fromDom(node.childNodes[i]);
                if (predicate(child)) {
                    return Optional.some(child);
                }
                const res = descend(node.childNodes[i]);
                if (res.isSome()) {
                    return res;
                }
            }
            return Optional.none();
        };
        return descend(scope.dom);
    };

    // TODO: An internal SelectorFilter module that doesn't SugarElement.fromDom() everything
    const first = (selector) => one(selector);
    const ancestor$1 = (scope, selector, isRoot) => ancestor$2(scope, (e) => is(e, selector), isRoot);
    const sibling = (scope, selector) => sibling$1(scope, (e) => is(e, selector));
    const child = (scope, selector) => child$1(scope, (e) => is(e, selector));
    const descendant = (scope, selector) => one(selector, scope);
    // Returns Some(closest ancestor element (sugared)) matching 'selector' up to isRoot, or None() otherwise
    const closest$3 = (scope, selector, isRoot) => {
        const is$1 = (element, selector) => is(element, selector);
        return ClosestOrAncestor(is$1, ancestor$1, scope, selector, isRoot);
    };

    const set$5 = (element, status) => {
        element.dom.checked = status;
    };
    const get$9 = (element) => element.dom.checked;

    // IE11 Can return undefined for a classList on elements such as math, so we make sure it's not undefined before attempting to use it.
    const supports = (element) => element.dom.classList !== undefined;
    const get$8 = (element) => read$2(element, 'class');
    const add$3 = (element, clazz) => add$4(element, 'class', clazz);
    const remove$4 = (element, clazz) => remove$5(element, 'class', clazz);
    const toggle$5 = (element, clazz) => {
        if (contains$2(get$8(element), clazz)) {
            return remove$4(element, clazz);
        }
        else {
            return add$3(element, clazz);
        }
    };

    /*
     * ClassList is IE10 minimum:
     * https://developer.mozilla.org/en-US/docs/Web/API/Element.classList
     *
     * Note that IE doesn't support the second argument to toggle (at all).
     * If it did, the toggler could be better.
     */
    const add$2 = (element, clazz) => {
        if (supports(element)) {
            element.dom.classList.add(clazz);
        }
        else {
            add$3(element, clazz);
        }
    };
    const cleanClass = (element) => {
        const classList = supports(element) ? element.dom.classList : get$8(element);
        // classList is a "live list", so this is up to date already
        if (classList.length === 0) {
            // No more classes left, remove the class attribute as well
            remove$8(element, 'class');
        }
    };
    const remove$3 = (element, clazz) => {
        if (supports(element)) {
            const classList = element.dom.classList;
            classList.remove(clazz);
        }
        else {
            remove$4(element, clazz);
        }
        cleanClass(element);
    };
    const toggle$4 = (element, clazz) => {
        const result = supports(element) ? element.dom.classList.toggle(clazz) : toggle$5(element, clazz);
        cleanClass(element);
        return result;
    };
    const has = (element, clazz) => supports(element) && element.dom.classList.contains(clazz);

    /*
     * ClassList is IE10 minimum:
     * https://developer.mozilla.org/en-US/docs/Web/API/Element.classList
     */
    const add$1 = (element, classes) => {
        each$1(classes, (x) => {
            add$2(element, x);
        });
    };
    const remove$2 = (element, classes) => {
        each$1(classes, (x) => {
            remove$3(element, x);
        });
    };
    const toggle$3 = (element, classes) => {
        each$1(classes, (x) => {
            toggle$4(element, x);
        });
    };
    const hasAll = (element, classes) => forall(classes, (clazz) => has(element, clazz));
    const getNative = (element) => {
        const classList = element.dom.classList;
        const r = new Array(classList.length);
        for (let i = 0; i < classList.length; i++) {
            const item = classList.item(i);
            if (item !== null) {
                r[i] = item;
            }
        }
        return r;
    };
    const get$7 = (element) => supports(element) ? getNative(element) : get$8(element);

    const get$6 = (element) => element.dom.textContent;

    const get$5 = (element) => element.dom.value;
    const set$4 = (element, value) => {
        if (value === undefined) {
            throw new Error('Value.set was undefined');
        }
        element.dom.value = value;
    };

    const ancestors = (scope, predicate, isRoot) => filter$2(parents(scope, isRoot), predicate);

    const descendants = (scope, selector) => all$3(selector, scope);

    const closest$2 = (scope, predicate, isRoot) => closest$4(scope, predicate, isRoot).isSome();

    const closest$1 = (scope, selector, isRoot) => closest$3(scope, selector, isRoot).isSome();

    const ensureIsRoot = (isRoot) => isFunction(isRoot) ? isRoot : never;
    const ancestor = (scope, transform, isRoot) => {
        let element = scope.dom;
        const stop = ensureIsRoot(isRoot);
        while (element.parentNode) {
            element = element.parentNode;
            const el = SugarElement.fromDom(element);
            const transformed = transform(el);
            if (transformed.isSome()) {
                return transformed;
            }
            else if (stop(el)) {
                break;
            }
        }
        return Optional.none();
    };
    const closest = (scope, transform, isRoot) => {
        const current = transform(scope);
        const stop = ensureIsRoot(isRoot);
        return current.orThunk(() => stop(scope) ? Optional.none() : ancestor(scope, transform, stop));
    };

    const create$5 = (start, soffset, finish, foffset) => ({
        start,
        soffset,
        finish,
        foffset
    });
    // tslint:disable-next-line:variable-name
    const SimRange = {
        create: create$5
    };

    const adt$9 = Adt.generate([
        { before: ['element'] },
        { on: ['element', 'offset'] },
        { after: ['element'] }
    ]);
    // Probably don't need this given that we now have "match"
    const cata$2 = (subject, onBefore, onOn, onAfter) => subject.fold(onBefore, onOn, onAfter);
    const getStart$1 = (situ) => situ.fold(identity, identity, identity);
    const before = adt$9.before;
    const on$1 = adt$9.on;
    const after = adt$9.after;
    // tslint:disable-next-line:variable-name
    const Situ = {
        before,
        on: on$1,
        after,
        cata: cata$2,
        getStart: getStart$1
    };

    // Consider adding a type for "element"
    const adt$8 = Adt.generate([
        { domRange: ['rng'] },
        { relative: ['startSitu', 'finishSitu'] },
        { exact: ['start', 'soffset', 'finish', 'foffset'] }
    ]);
    const exactFromRange = (simRange) => adt$8.exact(simRange.start, simRange.soffset, simRange.finish, simRange.foffset);
    const getStart = (selection) => selection.match({
        domRange: (rng) => SugarElement.fromDom(rng.startContainer),
        relative: (startSitu, _finishSitu) => Situ.getStart(startSitu),
        exact: (start, _soffset, _finish, _foffset) => start
    });
    const domRange = adt$8.domRange;
    const relative$1 = adt$8.relative;
    const exact = adt$8.exact;
    const getWin = (selection) => {
        const start = getStart(selection);
        return defaultView(start);
    };
    // This is out of place but it's API so I can't remove it
    const range$1 = SimRange.create;
    // tslint:disable-next-line:variable-name
    const SimSelection = {
        domRange,
        relative: relative$1,
        exact,
        exactFromRange,
        getWin,
        range: range$1
    };

    const getNativeSelection = (win) => Optional.from(win.getSelection());
    // NOTE: We are still reading the range because it gives subtly different behaviour
    // than using the anchorNode and focusNode. I'm not sure if this behaviour is any
    // better or worse; it's just different.
    const readRange = (selection) => {
        if (selection.rangeCount > 0) {
            const firstRng = selection.getRangeAt(0);
            const lastRng = selection.getRangeAt(selection.rangeCount - 1);
            return Optional.some(SimRange.create(SugarElement.fromDom(firstRng.startContainer), firstRng.startOffset, SugarElement.fromDom(lastRng.endContainer), lastRng.endOffset));
        }
        else {
            return Optional.none();
        }
    };
    const doGetExact = (selection) => {
        if (selection.anchorNode === null || selection.focusNode === null) {
            return readRange(selection);
        }
        else {
            const anchor = SugarElement.fromDom(selection.anchorNode);
            const focus = SugarElement.fromDom(selection.focusNode);
            // if this returns true anchor is _after_ focus, so we need a custom selection object to maintain the RTL selection
            return after$2(anchor, selection.anchorOffset, focus, selection.focusOffset) ? Optional.some(SimRange.create(anchor, selection.anchorOffset, focus, selection.focusOffset)) : readRange(selection);
        }
    };
    const getExact = (win) => 
    // We want to retrieve the selection as it is.
    getNativeSelection(win)
        .filter((sel) => sel.rangeCount > 0)
        .bind(doGetExact);
    const getFirstRect = (win, selection) => {
        const rng = asLtrRange(win, selection);
        return getFirstRect$1(rng);
    };
    const getBounds$2 = (win, selection) => {
        const rng = asLtrRange(win, selection);
        return getBounds$3(rng);
    };

    const units = {
        // we don't really support all of these different ways to express a length
        unsupportedLength: [
            'em',
            'ex',
            'cap',
            'ch',
            'ic',
            'rem',
            'lh',
            'rlh',
            'vw',
            'vh',
            'vi',
            'vb',
            'vmin',
            'vmax',
            'cm',
            'mm',
            'Q',
            'in',
            'pc',
            'pt',
            'px'
        ],
        // these are the length values we do support
        fixed: ['px', 'pt'],
        relative: ['%'],
        empty: ['']
    };
    // Built from https://tc39.es/ecma262/#prod-StrDecimalLiteral
    // Matches a float followed by a trailing set of characters
    const pattern = (() => {
        const decimalDigits = '[0-9]+';
        const signedInteger = '[+-]?' + decimalDigits;
        const exponentPart = '[eE]' + signedInteger;
        const dot = '\\.';
        const opt = (input) => `(?:${input})?`;
        const unsignedDecimalLiteral = [
            'Infinity',
            decimalDigits + dot + opt(decimalDigits) + opt(exponentPart),
            dot + decimalDigits + opt(exponentPart),
            decimalDigits + opt(exponentPart)
        ].join('|');
        const float = `[+-]?(?:${unsignedDecimalLiteral})`;
        return new RegExp(`^(${float})(.*)$`);
    })();
    const isUnit = (unit, accepted) => exists(accepted, (acc) => exists(units[acc], (check) => unit === check));
    const parse = (input, accepted) => {
        const match = Optional.from(pattern.exec(input));
        return match.bind((array) => {
            const value = Number(array[1]);
            const unitRaw = array[2];
            if (isUnit(unitRaw, accepted)) {
                return Optional.some({
                    value,
                    unit: unitRaw
                });
            }
            else {
                return Optional.none();
            }
        });
    };
    const normalise = (input, accepted) => parse(input, accepted).map(({ value, unit }) => value + unit);

    const get$4 = (_win) => {
        const win = _win === undefined ? window : _win;
        if (detect$1().browser.isFirefox()) {
            // TINY-7984: Firefox 91 is returning incorrect values for visualViewport.pageTop, so disable it for now
            return Optional.none();
        }
        else {
            return Optional.from(win.visualViewport);
        }
    };
    const bounds$1 = (x, y, width, height) => ({
        x,
        y,
        width,
        height,
        right: x + width,
        bottom: y + height
    });
    const getBounds$1 = (_win) => {
        const win = _win === undefined ? window : _win;
        const doc = win.document;
        const scroll = get$b(SugarElement.fromDom(doc));
        return get$4(win).fold(() => {
            const html = win.document.documentElement;
            // Don't use window.innerWidth/innerHeight here, as we don't want to include scrollbars
            // since the right/bottom position is based on the edge of the scrollbar not the window
            const width = html.clientWidth;
            const height = html.clientHeight;
            return bounds$1(scroll.left, scroll.top, width, height);
        }, (visualViewport) => 
        // iOS doesn't update the pageTop/pageLeft when element.scrollIntoView() is called, so we need to fallback to the
        // scroll position which will always be less than the page top/left values when page top/left are accurate/correct.
        bounds$1(Math.max(visualViewport.pageLeft, scroll.left), Math.max(visualViewport.pageTop, scroll.top), visualViewport.width, visualViewport.height));
    };

    const walkUp = (navigation, doc) => {
        const frame = navigation.view(doc);
        return frame.fold(constant$1([]), (f) => {
            const parent = navigation.owner(f);
            const rest = walkUp(navigation, parent);
            return [f].concat(rest);
        });
    };
    // TODO: Why is this an option if it is always some?
    const pathTo = (element, navigation) => {
        const d = navigation.owner(element);
        const paths = walkUp(navigation, d);
        return Optional.some(paths);
    };

    const view = (doc) => {
        // Only walk up to the document this script is defined in.
        // This prevents walking up to the parent window when the editor is in an iframe.
        const element = doc.dom === document ? Optional.none() : Optional.from(doc.dom.defaultView?.frameElement);
        return element.map(SugarElement.fromDom);
    };
    const owner$3 = (element) => owner$4(element);

    var Navigation = /*#__PURE__*/Object.freeze({
        __proto__: null,
        view: view,
        owner: owner$3
    });

    const find$2 = (element) => {
        const doc = getDocument();
        const scroll = get$b(doc);
        // Get the path of iframe elements to this element.
        const path = pathTo(element, Navigation);
        return path.fold(curry(absolute$3, element), (frames) => {
            const offset = viewport$1(element);
            const r = foldr(frames, (b, a) => {
                const loc = viewport$1(a);
                return {
                    left: b.left + loc.left,
                    top: b.top + loc.top
                };
            }, { left: 0, top: 0 });
            return SugarPosition(r.left + offset.left + scroll.left, r.top + offset.top + scroll.top);
        });
    };

    const pointed = (point, width, height) => ({
        point,
        width,
        height
    });
    const rect = (x, y, width, height) => ({
        x,
        y,
        width,
        height
    });
    const bounds = (x, y, width, height) => ({
        x,
        y,
        width,
        height,
        right: x + width,
        bottom: y + height
    });
    const box$1 = (element) => {
        const xy = absolute$3(element);
        const w = getOuter(element);
        const h = getOuter$1(element);
        return bounds(xy.left, xy.top, w, h);
    };
    // NOTE: We used to use AriaFocus.preserve here, but there is no reason to do that now that
    // we are not changing the visibility of the element. Hopefully (2015-09-29).
    const absolute$2 = (element) => {
        const position = find$2(element);
        const width = getOuter(element);
        const height = getOuter$1(element);
        return bounds(position.left, position.top, width, height);
    };
    const constrain = (original, constraint) => {
        const left = Math.max(original.x, constraint.x);
        const top = Math.max(original.y, constraint.y);
        const right = Math.min(original.right, constraint.right);
        const bottom = Math.min(original.bottom, constraint.bottom);
        const width = right - left;
        const height = bottom - top;
        return bounds(left, top, width, height);
    };
    const constrainByMany = (original, constraints) => {
        return foldl(constraints, (acc, c) => constrain(acc, c), original);
    };
    const win = () => getBounds$1(window);

    const isSource = (component, simulatedEvent) => eq(component.element, simulatedEvent.event.target);

    const getOffsetParent = (element) => {
        // Firefox sets the offsetParent to the body when fixed instead of null like
        // all other browsers. So we need to check if the element is fixed and if so then
        // disregard the elements offsetParent.
        const isFixed = is$1(getRaw(element, 'position'), 'fixed');
        const offsetParent$1 = isFixed ? Optional.none() : offsetParent(element);
        return offsetParent$1.orThunk(() => {
            const marker = SugarElement.fromTag('span');
            // PERFORMANCE: Append the marker to the parent element, as adding it before the current element will
            // trigger the styles to be recalculated which is a little costly (particularly in scroll/resize events)
            return parent(element).bind((parent) => {
                append$2(parent, marker);
                const offsetParent$1 = offsetParent(marker);
                remove$7(marker);
                return offsetParent$1;
            });
        });
    };
    /*
     * This allows the absolute coordinates to be obtained by adding the
     * origin to the offset coordinates and not needing to know scroll.
     */
    const getOrigin = (element) => getOffsetParent(element).map(absolute$3).getOrThunk(() => SugarPosition(0, 0));

    const describedBy = (describedElement, describeElement) => {
        const describeId = Optional.from(get$g(describedElement, 'id'))
            .getOrThunk(() => {
            const id = generate$6('aria');
            set$9(describeElement, 'id', id);
            return id;
        });
        set$9(describedElement, 'aria-describedby', describeId);
    };
    const remove$1 = (describedElement) => {
        remove$8(describedElement, 'aria-describedby');
    };

    var SimpleResultType;
    (function (SimpleResultType) {
        SimpleResultType[SimpleResultType["Error"] = 0] = "Error";
        SimpleResultType[SimpleResultType["Value"] = 1] = "Value";
    })(SimpleResultType || (SimpleResultType = {}));
    const fold$1 = (res, onError, onValue) => res.stype === SimpleResultType.Error ? onError(res.serror) : onValue(res.svalue);
    const partition$1 = (results) => {
        const values = [];
        const errors = [];
        each$1(results, (obj) => {
            fold$1(obj, (err) => errors.push(err), (val) => values.push(val));
        });
        return { values, errors };
    };
    const mapError = (res, f) => {
        if (res.stype === SimpleResultType.Error) {
            return { stype: SimpleResultType.Error, serror: f(res.serror) };
        }
        else {
            return res;
        }
    };
    const map = (res, f) => {
        if (res.stype === SimpleResultType.Value) {
            return { stype: SimpleResultType.Value, svalue: f(res.svalue) };
        }
        else {
            return res;
        }
    };
    const bind = (res, f) => {
        if (res.stype === SimpleResultType.Value) {
            return f(res.svalue);
        }
        else {
            return res;
        }
    };
    const bindError = (res, f) => {
        if (res.stype === SimpleResultType.Error) {
            return f(res.serror);
        }
        else {
            return res;
        }
    };
    const svalue = (v) => ({ stype: SimpleResultType.Value, svalue: v });
    const serror = (e) => ({ stype: SimpleResultType.Error, serror: e });
    const toResult$1 = (res) => fold$1(res, Result.error, Result.value);
    const fromResult = (res) => res.fold(serror, svalue);
    const SimpleResult = {
        fromResult,
        toResult: toResult$1,
        svalue,
        partition: partition$1,
        serror,
        bind,
        bindError,
        map,
        mapError,
        fold: fold$1
    };

    const formatObj = (input) => {
        return isObject(input) && keys(input).length > 100 ? ' removed due to size' : JSON.stringify(input, null, 2);
    };
    const formatErrors = (errors) => {
        const es = errors.length > 10 ? errors.slice(0, 10).concat([
            {
                path: [],
                getErrorInfo: constant$1('... (only showing first ten failures)')
            }
        ]) : errors;
        // TODO: Work out a better split between PrettyPrinter and SchemaError
        return map$2(es, (e) => {
            return 'Failed path: (' + e.path.join(' > ') + ')\n' + e.getErrorInfo();
        });
    };

    const nu$7 = (path, getErrorInfo) => {
        return SimpleResult.serror([{
                path,
                // This is lazy so that it isn't calculated unnecessarily
                getErrorInfo
            }]);
    };
    const missingRequired = (path, key, obj) => nu$7(path, () => 'Could not find valid *required* value for "' + key + '" in ' + formatObj(obj));
    const missingKey = (path, key) => nu$7(path, () => 'Choice schema did not contain choice key: "' + key + '"');
    const missingBranch = (path, branches, branch) => nu$7(path, () => 'The chosen schema: "' + branch + '" did not exist in branches: ' + formatObj(branches));
    const unsupportedFields = (path, unsupported) => nu$7(path, () => 'There are unsupported fields: [' + unsupported.join(', ') + '] specified');
    const custom = (path, err) => nu$7(path, constant$1(err));

    const value$1 = (validator) => {
        const extract = (path, val) => {
            return SimpleResult.bindError(validator(val), (err) => custom(path, err));
        };
        const toString = constant$1('val');
        return {
            extract,
            toString
        };
    };
    const anyValue$1 = value$1(SimpleResult.svalue);

    const anyValue = constant$1(anyValue$1);
    const typedValue = (validator, expectedType) => value$1((a) => {
        const actualType = typeof a;
        return validator(a) ? SimpleResult.svalue(a) : SimpleResult.serror(`Expected type: ${expectedType} but got: ${actualType}`);
    });
    const number = typedValue(isNumber, 'number');
    const string = typedValue(isString, 'string');
    const boolean = typedValue(isBoolean, 'boolean');
    const functionProcessor = typedValue(isFunction, 'function');
    // Test if a value can be copied by the structured clone algorithm and hence sendable via postMessage
    // https://developer.mozilla.org/en-US/docs/Web/API/Web_Workers_API/Structured_clone_algorithm
    // from https://stackoverflow.com/a/32673910/7377237 with adjustments for typescript
    const isPostMessageable = (val) => {
        if (Object(val) !== val) { // Primitive value
            return true;
        }
        switch ({}.toString.call(val).slice(8, -1)) { // Class
            case 'Boolean':
            case 'Number':
            case 'String':
            case 'Date':
            case 'RegExp':
            case 'Blob':
            case 'FileList':
            case 'ImageData':
            case 'ImageBitmap':
            case 'ArrayBuffer':
                return true;
            case 'Array':
            case 'Object':
                return Object.keys(val).every((prop) => isPostMessageable(val[prop]));
            default:
                return false;
        }
    };
    const postMessageable = value$1((a) => {
        if (isPostMessageable(a)) {
            return SimpleResult.svalue(a);
        }
        else {
            return SimpleResult.serror('Expected value to be acceptable for sending via postMessage');
        }
    });

    const required$2 = () => ({ tag: "required" /* FieldPresenceTag.Required */, process: {} });
    const defaultedThunk = (fallbackThunk) => ({ tag: "defaultedThunk" /* FieldPresenceTag.DefaultedThunk */, process: fallbackThunk });
    const defaulted$1 = (fallback) => defaultedThunk(constant$1(fallback));
    const asOption = () => ({ tag: "option" /* FieldPresenceTag.Option */, process: {} });
    const mergeWithThunk = (baseThunk) => ({ tag: "mergeWithThunk" /* FieldPresenceTag.MergeWithThunk */, process: baseThunk });
    const mergeWith = (base) => mergeWithThunk(constant$1(base));

    const field$2 = (key, newKey, presence, prop) => ({ tag: "field" /* FieldTag.Field */, key, newKey, presence, prop });
    const customField$1 = (newKey, instantiator) => ({ tag: "custom" /* FieldTag.CustomField */, newKey, instantiator });
    const fold = (value, ifField, ifCustom) => {
        switch (value.tag) {
            case "field" /* FieldTag.Field */:
                return ifField(value.key, value.newKey, value.presence, value.prop);
            case "custom" /* FieldTag.CustomField */:
                return ifCustom(value.newKey, value.instantiator);
        }
    };

    const mergeValues$1 = (values, base) => {
        return SimpleResult.svalue(deepMerge(base, merge$1.apply(undefined, values)));
    };
    const mergeErrors$1 = (errors) => compose(SimpleResult.serror, flatten)(errors);
    const consolidateObj = (objects, base) => {
        const partition = SimpleResult.partition(objects);
        return partition.errors.length > 0 ? mergeErrors$1(partition.errors) : mergeValues$1(partition.values, base);
    };
    const consolidateArr = (objects) => {
        const partitions = SimpleResult.partition(objects);
        return partitions.errors.length > 0 ? mergeErrors$1(partitions.errors) : SimpleResult.svalue(partitions.values);
    };
    const ResultCombine = {
        consolidateObj,
        consolidateArr
    };

    const requiredAccess = (path, obj, key, bundle) => 
    // In required mode, if it is undefined, it is an error.
    get$h(obj, key).fold(() => missingRequired(path, key, obj), bundle);
    const fallbackAccess = (obj, key, fallback, bundle) => {
        const v = get$h(obj, key).getOrThunk(() => fallback(obj));
        return bundle(v);
    };
    const optionAccess = (obj, key, bundle) => bundle(get$h(obj, key));
    const optionDefaultedAccess = (obj, key, fallback, bundle) => {
        const opt = get$h(obj, key).map((val) => val === true ? fallback(obj) : val);
        return bundle(opt);
    };
    const extractField = (field, path, obj, key, prop) => {
        const bundle = (av) => prop.extract(path.concat([key]), av);
        const bundleAsOption = (optValue) => optValue.fold(() => SimpleResult.svalue(Optional.none()), (ov) => {
            const result = prop.extract(path.concat([key]), ov);
            return SimpleResult.map(result, Optional.some);
        });
        switch (field.tag) {
            case "required" /* FieldPresenceTag.Required */:
                return requiredAccess(path, obj, key, bundle);
            case "defaultedThunk" /* FieldPresenceTag.DefaultedThunk */:
                return fallbackAccess(obj, key, field.process, bundle);
            case "option" /* FieldPresenceTag.Option */:
                return optionAccess(obj, key, bundleAsOption);
            case "defaultedOptionThunk" /* FieldPresenceTag.DefaultedOptionThunk */:
                return optionDefaultedAccess(obj, key, field.process, bundleAsOption);
            case "mergeWithThunk" /* FieldPresenceTag.MergeWithThunk */: {
                return fallbackAccess(obj, key, constant$1({}), (v) => {
                    const result = deepMerge(field.process(obj), v);
                    return bundle(result);
                });
            }
        }
    };
    const extractFields = (path, obj, fields) => {
        const success = {};
        const errors = [];
        // PERFORMANCE: We use a for loop here instead of Arr.each as this is a hot code path
        for (const field of fields) {
            fold(field, (key, newKey, presence, prop) => {
                const result = extractField(presence, path, obj, key, prop);
                SimpleResult.fold(result, (err) => {
                    errors.push(...err);
                }, (res) => {
                    success[newKey] = res;
                });
            }, (newKey, instantiator) => {
                success[newKey] = instantiator(obj);
            });
        }
        return errors.length > 0 ? SimpleResult.serror(errors) : SimpleResult.svalue(success);
    };
    const valueThunk = (getDelegate) => {
        const extract = (path, val) => getDelegate().extract(path, val);
        const toString = () => getDelegate().toString();
        return {
            extract,
            toString
        };
    };
    // This is because Obj.keys can return things where the key is set to undefined.
    const getSetKeys = (obj) => keys(filter$1(obj, isNonNullable));
    const objOfOnly = (fields) => {
        const delegate = objOf(fields);
        const fieldNames = foldr(fields, (acc, value) => {
            return fold(value, (key) => deepMerge(acc, { [key]: true }), constant$1(acc));
        }, {});
        const extract = (path, o) => {
            const keys = isBoolean(o) ? [] : getSetKeys(o);
            const extra = filter$2(keys, (k) => !hasNonNullableKey(fieldNames, k));
            return extra.length === 0 ? delegate.extract(path, o) : unsupportedFields(path, extra);
        };
        return {
            extract,
            toString: delegate.toString
        };
    };
    const objOf = (values) => {
        const extract = (path, o) => extractFields(path, o, values);
        const toString = () => {
            const fieldStrings = map$2(values, (value) => fold(value, (key, _okey, _presence, prop) => key + ' -> ' + prop.toString(), (newKey, _instantiator) => 'state(' + newKey + ')'));
            return 'obj{\n' + fieldStrings.join('\n') + '}';
        };
        return {
            extract,
            toString
        };
    };
    const arrOf = (prop) => {
        const extract = (path, array) => {
            const results = map$2(array, (a, i) => prop.extract(path.concat(['[' + i + ']']), a));
            return ResultCombine.consolidateArr(results);
        };
        const toString = () => 'array(' + prop.toString() + ')';
        return {
            extract,
            toString
        };
    };
    const oneOf = (props, rawF) => {
        // If f is not supplied, then use identity.
        const f = rawF !== undefined ? rawF : identity;
        const extract = (path, val) => {
            const errors = [];
            // Return on first match
            for (const prop of props) {
                const res = prop.extract(path, val);
                if (res.stype === SimpleResultType.Value) {
                    return {
                        stype: SimpleResultType.Value,
                        svalue: f(res.svalue)
                    };
                }
                errors.push(res);
            }
            // All failed, return errors
            return ResultCombine.consolidateArr(errors);
        };
        const toString = () => 'oneOf(' + map$2(props, (prop) => prop.toString()).join(', ') + ')';
        return {
            extract,
            toString
        };
    };
    const setOf$1 = (validator, prop) => {
        const validateKeys = (path, keys) => arrOf(value$1(validator)).extract(path, keys);
        const extract = (path, o) => {
            //
            const keys$1 = keys(o);
            const validatedKeys = validateKeys(path, keys$1);
            return SimpleResult.bind(validatedKeys, (validKeys) => {
                const schema = map$2(validKeys, (vk) => {
                    return field$2(vk, vk, required$2(), prop);
                });
                return objOf(schema).extract(path, o);
            });
        };
        const toString = () => 'setOf(' + prop.toString() + ')';
        return {
            extract,
            toString
        };
    };
    const thunk = (_desc, processor) => {
        const getP = cached(processor);
        const extract = (path, val) => getP().extract(path, val);
        const toString = () => getP().toString();
        return {
            extract,
            toString
        };
    };
    const arrOfObj = compose(arrOf, objOf);

    const chooseFrom = (path, input, branches, ch) => {
        const fields = get$h(branches, ch);
        return fields.fold(() => missingBranch(path, branches, ch), (vp) => vp.extract(path.concat(['branch: ' + ch]), input));
    };
    // The purpose of choose is to have a key which picks which of the schemas to follow.
    // The key will index into the object of schemas: branches
    const choose$2 = (key, branches) => {
        const extract = (path, input) => {
            const choice = get$h(input, key);
            return choice.fold(() => missingKey(path, key), (chosen) => chooseFrom(path, input, branches, chosen));
        };
        const toString = () => 'chooseOn(' + key + '). Possible values: ' + keys(branches);
        return {
            extract,
            toString
        };
    };

    const arrOfVal = () => arrOf(anyValue$1);
    const valueOf = (validator) => value$1((v) => validator(v).fold(SimpleResult.serror, SimpleResult.svalue));
    const setOf = (validator, prop) => setOf$1((v) => SimpleResult.fromResult(validator(v)), prop);
    const extractValue = (label, prop, obj) => {
        const res = prop.extract([label], obj);
        return SimpleResult.mapError(res, (errs) => ({ input: obj, errors: errs }));
    };
    const asRaw = (label, prop, obj) => SimpleResult.toResult(extractValue(label, prop, obj));
    const getOrDie = (extraction) => {
        return extraction.fold((errInfo) => {
            // A readable version of the error.
            throw new Error(formatError(errInfo));
        }, identity);
    };
    const asRawOrDie$1 = (label, prop, obj) => getOrDie(asRaw(label, prop, obj));
    const formatError = (errInfo) => {
        return 'Errors: \n' + formatErrors(errInfo.errors).join('\n') +
            '\n\nInput object: ' + formatObj(errInfo.input);
    };
    const choose$1 = (key, branches) => choose$2(key, map$1(branches, objOf));
    const thunkOf = (desc, schema) => thunk(desc, schema);

    const field$1 = field$2;
    const customField = customField$1;
    const validateEnum = (values) => valueOf((value) => contains$2(values, value) ?
        Result.value(value) :
        Result.error(`Unsupported value: "${value}", choose one of "${values.join(', ')}".`));
    const required$1 = (key) => field$1(key, key, required$2(), anyValue());
    const requiredOf = (key, schema) => field$1(key, key, required$2(), schema);
    const requiredNumber = (key) => requiredOf(key, number);
    const requiredString = (key) => requiredOf(key, string);
    const requiredStringEnum = (key, values) => field$1(key, key, required$2(), validateEnum(values));
    const requiredFunction = (key) => requiredOf(key, functionProcessor);
    const forbid = (key, message) => field$1(key, key, asOption(), value$1((_v) => SimpleResult.serror('The field: ' + key + ' is forbidden. ' + message)));
    const requiredObjOf = (key, objSchema) => field$1(key, key, required$2(), objOf(objSchema));
    const requiredArrayOfObj = (key, objFields) => field$1(key, key, required$2(), arrOfObj(objFields));
    const requiredArrayOf = (key, schema) => field$1(key, key, required$2(), arrOf(schema));
    const option$3 = (key) => field$1(key, key, asOption(), anyValue());
    const optionOf = (key, schema) => field$1(key, key, asOption(), schema);
    const optionNumber = (key) => optionOf(key, number);
    const optionString = (key) => optionOf(key, string);
    const optionStringEnum = (key, values) => optionOf(key, validateEnum(values));
    const optionBoolean = (key) => optionOf(key, boolean);
    const optionFunction = (key) => optionOf(key, functionProcessor);
    const optionArrayOf = (key, schema) => optionOf(key, arrOf(schema));
    const optionObjOf = (key, objSchema) => optionOf(key, objOf(objSchema));
    const optionObjOfOnly = (key, objSchema) => optionOf(key, objOfOnly(objSchema));
    const defaulted = (key, fallback) => field$1(key, key, defaulted$1(fallback), anyValue());
    const defaultedOf = (key, fallback, schema) => field$1(key, key, defaulted$1(fallback), schema);
    const defaultedNumber = (key, fallback) => defaultedOf(key, fallback, number);
    const defaultedString = (key, fallback) => defaultedOf(key, fallback, string);
    const defaultedStringEnum = (key, fallback, values) => defaultedOf(key, fallback, validateEnum(values));
    const defaultedBoolean = (key, fallback) => defaultedOf(key, fallback, boolean);
    const defaultedFunction = (key, fallback) => defaultedOf(key, fallback, functionProcessor);
    const defaultedPostMsg = (key, fallback) => defaultedOf(key, fallback, postMessageable);
    const defaultedArrayOf = (key, fallback, schema) => defaultedOf(key, fallback, arrOf(schema));
    const defaultedObjOf = (key, fallback, objSchema) => defaultedOf(key, fallback, objOf(objSchema));

    const exclude$1 = (obj, fields) => {
        const r = {};
        each(obj, (v, k) => {
            if (!contains$2(fields, k)) {
                r[k] = v;
            }
        });
        return r;
    };

    const wrap$1 = (key, value) => ({ [key]: value });
    const wrapAll$1 = (keyvalues) => {
        const r = {};
        each$1(keyvalues, (kv) => {
            r[kv.key] = kv.value;
        });
        return r;
    };

    const exclude = (obj, fields) => exclude$1(obj, fields);
    const wrap = (key, value) => wrap$1(key, value);
    const wrapAll = (keyvalues) => wrapAll$1(keyvalues);
    const mergeValues = (values, base) => {
        return values.length === 0 ? Result.value(base) : Result.value(deepMerge(base, merge$1.apply(undefined, values))
        // Merger.deepMerge.apply(undefined, [ base ].concat(values))
        );
    };
    const mergeErrors = (errors) => Result.error(flatten(errors));
    const consolidate = (objs, base) => {
        const partitions = partition$2(objs);
        return partitions.errors.length > 0 ? mergeErrors(partitions.errors) : mergeValues(partitions.values, base);
    };

    const constant = constant$1;
    const touchstart = constant('touchstart');
    const touchmove = constant('touchmove');
    const touchend = constant('touchend');
    const touchcancel = constant('touchcancel');
    const mousedown = constant('mousedown');
    const mousemove = constant('mousemove');
    const mouseout = constant('mouseout');
    const mouseup = constant('mouseup');
    const mouseover = constant('mouseover');
    // Not really a native event as it has to be simulated
    const focusin = constant('focusin');
    const focusout = constant('focusout');
    const keydown = constant('keydown');
    const keyup = constant('keyup');
    const input = constant('input');
    const change = constant('change');
    const click = constant('click');
    const transitioncancel = constant('transitioncancel');
    const transitionend = constant('transitionend');
    const transitionstart = constant('transitionstart');
    const selectstart = constant('selectstart');

    const prefixName = (name) => constant$1('alloy.' + name);
    const alloy = { tap: prefixName('tap') };
    // This is used to pass focus to a component. A component might interpret
    // this event and pass the DOM focus to one of its children, depending on its
    // focus model.
    const focus$3 = prefixName('focus');
    // This event is fired a small amount of time after the blur has fired. This
    // allows the handler to know what was the focused element, and what is now.
    const postBlur = prefixName('blur.post');
    // This event is fired a small amount of time after the paste event has fired.
    const postPaste = prefixName('paste.post');
    // This event is fired by gui.broadcast*. It is defined by 'receivers'
    const receive = prefixName('receive');
    // This event is for executing buttons and things that have (mostly) enter actions
    const execute$5 = prefixName('execute');
    // This event is used by a menu to tell an item to focus itself because it has been
    // selected. This might automatically focus inside the item, it might focus the outer
    // part of the widget etc.
    const focusItem = prefixName('focus.item');
    // This event represents a touchstart and touchend on the same location, and fires on
    // the touchend
    const tap = alloy.tap;
    // This event represents a longpress on the same location
    const longpress = prefixName('longpress');
    // Fire by a child element to tell the outer element to close
    const sandboxClose = prefixName('sandbox.close');
    // Tell the typeahead to cancel any pending fetches (that haven't already executed)
    const typeaheadCancel = prefixName('typeahead.cancel');
    // Fired when adding to a world
    const systemInit = prefixName('system.init');
    // Fired when a touchmove on the document happens
    const documentTouchmove = prefixName('system.touchmove');
    // Fired when a touchend on the document happens
    const documentTouchend = prefixName('system.touchend');
    // Fired when the window scrolls
    const windowScroll = prefixName('system.scroll');
    // Fired when the window resizes
    const windowResize = prefixName('system.resize');
    const attachedToDom = prefixName('system.attached');
    const detachedFromDom = prefixName('system.detached');
    const dismissRequested = prefixName('system.dismissRequested');
    const repositionRequested = prefixName('system.repositionRequested');
    const focusShifted = prefixName('focusmanager.shifted');
    // Fired when slots are made hidden/shown
    const slotVisibility = prefixName('slotcontainer.visibility');
    // Used for containers outside the mothership that scroll. Used by docking.
    const externalElementScroll = prefixName('system.external.element.scroll');
    const changeTab = prefixName('change.tab');
    const dismissTab = prefixName('dismiss.tab');
    const highlight$1 = prefixName('highlight');
    const dehighlight$1 = prefixName('dehighlight');

    const element = (elem) => getHtml(elem);

    const unknown = 'unknown';
    /*
      typescipt qwerk:
      const debugging: boolean = true;
      if (boolean === false) {  -> this throws a type error! // TS2365:Operator '===' cannot be applied to types 'false' and 'true'
        https://www.typescriptlang.org/play/#src=const%20foo%3A%20boolean%20%3D%20true%3B%0D%0A%0D%0Aif%20(foo%20%3D%3D%3D%20false)%20%7B%0D%0A%20%20%20%20%0D%0A%7D
      }
    */
    const debugging = true;
    var EventConfiguration;
    (function (EventConfiguration) {
        EventConfiguration[EventConfiguration["STOP"] = 0] = "STOP";
        EventConfiguration[EventConfiguration["NORMAL"] = 1] = "NORMAL";
        EventConfiguration[EventConfiguration["LOGGING"] = 2] = "LOGGING";
    })(EventConfiguration || (EventConfiguration = {}));
    const eventConfig = Cell({});
    const makeEventLogger = (eventName, initialTarget) => {
        const sequence = [];
        const startTime = new Date().getTime();
        return {
            logEventCut: (_name, target, purpose) => {
                sequence.push({ outcome: 'cut', target, purpose });
            },
            logEventStopped: (_name, target, purpose) => {
                sequence.push({ outcome: 'stopped', target, purpose });
            },
            logNoParent: (_name, target, purpose) => {
                sequence.push({ outcome: 'no-parent', target, purpose });
            },
            logEventNoHandlers: (_name, target) => {
                sequence.push({ outcome: 'no-handlers-left', target });
            },
            logEventResponse: (_name, target, purpose) => {
                sequence.push({ outcome: 'response', purpose, target });
            },
            write: () => {
                const finishTime = new Date().getTime();
                if (contains$2(['mousemove', 'mouseover', 'mouseout', systemInit()], eventName)) {
                    return;
                }
                // eslint-disable-next-line no-console
                console.log(eventName, {
                    event: eventName,
                    time: finishTime - startTime,
                    target: initialTarget.dom,
                    sequence: map$2(sequence, (s) => {
                        if (!contains$2(['cut', 'stopped', 'response'], s.outcome)) {
                            return s.outcome;
                        }
                        else {
                            return '{' + s.purpose + '} ' + s.outcome + ' at (' + element(s.target) + ')';
                        }
                    })
                });
            }
        };
    };
    const processEvent = (eventName, initialTarget, f) => {
        const status = get$h(eventConfig.get(), eventName).orThunk(() => {
            const patterns = keys(eventConfig.get());
            return findMap(patterns, (p) => eventName.indexOf(p) > -1 ? Optional.some(eventConfig.get()[p]) : Optional.none());
        }).getOr(EventConfiguration.NORMAL);
        switch (status) {
            case EventConfiguration.NORMAL:
                return f(noLogger());
            case EventConfiguration.LOGGING: {
                const logger = makeEventLogger(eventName, initialTarget);
                const output = f(logger);
                logger.write();
                return output;
            }
            case EventConfiguration.STOP:
                // Does not even run the function to trigger event and listen to handlers
                return true;
        }
    };
    // Ignore these files in the error stack
    const path = [
        'alloy/data/Fields',
        'alloy/debugging/Debugging'
    ];
    const getTrace = () => {
        if (debugging === false) {
            return unknown;
        }
        const err = new Error();
        if (err.stack !== undefined) {
            const lines = err.stack.split('\n');
            return find$5(lines, (line) => line.indexOf('alloy') > 0 && !exists(path, (p) => line.indexOf(p) > -1)).getOr(unknown);
        }
        else {
            return unknown;
        }
    };
    const ignoreEvent = {
        logEventCut: noop,
        logEventStopped: noop,
        logNoParent: noop,
        logEventNoHandlers: noop,
        logEventResponse: noop,
        write: noop
    };
    const monitorEvent = (eventName, initialTarget, f) => processEvent(eventName, initialTarget, f);
    const noLogger = constant$1(ignoreEvent);

    const menuFields = constant$1([
        required$1('menu'),
        required$1('selectedMenu')
    ]);
    const itemFields = constant$1([
        required$1('item'),
        required$1('selectedItem')
    ]);
    constant$1(objOf(itemFields().concat(menuFields())));
    const itemSchema$3 = constant$1(objOf(itemFields()));

    const _initSize = requiredObjOf('initSize', [
        required$1('numColumns'),
        required$1('numRows')
    ]);
    const itemMarkers = () => requiredOf('markers', itemSchema$3());
    const tieredMenuMarkers = () => requiredObjOf('markers', [
        required$1('backgroundMenu')
    ].concat(menuFields()).concat(itemFields()));
    const markers$1 = (required) => requiredObjOf('markers', map$2(required, required$1));
    const onPresenceHandler = (label, fieldName, presence) => {
        // We care about where the handler was declared (in terms of which schema)
        getTrace();
        return field$1(fieldName, fieldName, presence, 
        // Apply some wrapping to their supplied function
        valueOf((f) => Result.value((...args) => {
            return f.apply(undefined, args);
        })));
    };
    const onHandler = (fieldName) => onPresenceHandler('onHandler', fieldName, defaulted$1(noop));
    const onKeyboardHandler = (fieldName) => onPresenceHandler('onKeyboardHandler', fieldName, defaulted$1(Optional.none));
    const onStrictHandler = (fieldName) => onPresenceHandler('onHandler', fieldName, required$2());
    const onStrictKeyboardHandler = (fieldName) => onPresenceHandler('onKeyboardHandler', fieldName, required$2());
    const output$1 = (name, value) => customField(name, constant$1(value));
    const snapshot = (name) => customField(name, identity);
    const initSize = constant$1(_initSize);

    const markAsBehaviourApi = (f, apiName, apiFunction) => {
        const delegate = apiFunction.toString();
        const endIndex = delegate.indexOf(')') + 1;
        const openBracketIndex = delegate.indexOf('(');
        const parameters = delegate.substring(openBracketIndex + 1, endIndex - 1).split(/,\s*/);
        f.toFunctionAnnotation = () => ({
            name: apiName,
            parameters: cleanParameters(parameters.slice(0, 1).concat(parameters.slice(3)))
        });
        return f;
    };
    // Remove any comment (/*) at end of parameter names
    const cleanParameters = (parameters) => map$2(parameters, (p) => endsWith(p, '/*') ? p.substring(0, p.length - '/*'.length) : p);
    const markAsExtraApi = (f, extraName) => {
        const delegate = f.toString();
        const endIndex = delegate.indexOf(')') + 1;
        const openBracketIndex = delegate.indexOf('(');
        const parameters = delegate.substring(openBracketIndex + 1, endIndex - 1).split(/,\s*/);
        f.toFunctionAnnotation = () => ({
            name: extraName,
            parameters: cleanParameters(parameters)
        });
        return f;
    };
    const markAsSketchApi = (f, apiFunction) => {
        const delegate = apiFunction.toString();
        const endIndex = delegate.indexOf(')') + 1;
        const openBracketIndex = delegate.indexOf('(');
        const parameters = delegate.substring(openBracketIndex + 1, endIndex - 1).split(/,\s*/);
        f.toFunctionAnnotation = () => ({
            name: 'OVERRIDE',
            parameters: cleanParameters(parameters.slice(1))
        });
        return f;
    };

    const DelayedFunction = (fun, delay) => {
        let ref = null;
        const schedule = (...args) => {
            ref = setTimeout(() => {
                fun.apply(null, args);
                ref = null;
            }, delay);
        };
        const cancel = () => {
            if (ref !== null) {
                clearTimeout(ref);
                ref = null;
            }
        };
        return {
            cancel,
            schedule
        };
    };

    const SIGNIFICANT_MOVE = 5;
    const LONGPRESS_DELAY = 400;
    const getTouch = (event) => {
        const raw = event.raw;
        if (raw.touches === undefined || raw.touches.length !== 1) {
            return Optional.none();
        }
        return Optional.some(raw.touches[0]);
    };
    // Check to see if the touch has changed a *significant* amount
    const isFarEnough = (touch, data) => {
        const distX = Math.abs(touch.clientX - data.x);
        const distY = Math.abs(touch.clientY - data.y);
        return distX > SIGNIFICANT_MOVE || distY > SIGNIFICANT_MOVE;
    };
    const monitor = (settings) => {
        /* A tap event is a combination of touchstart and touchend on the same element
         * without a *significant* touchmove in between.
         */
        const startData = value$2();
        const longpressFired = Cell(false);
        const longpress$1 = DelayedFunction((event) => {
            settings.triggerEvent(longpress(), event);
            longpressFired.set(true);
        }, LONGPRESS_DELAY);
        const handleTouchstart = (event) => {
            getTouch(event).each((touch) => {
                longpress$1.cancel();
                const data = {
                    x: touch.clientX,
                    y: touch.clientY,
                    target: event.target
                };
                longpress$1.schedule(event);
                longpressFired.set(false);
                startData.set(data);
            });
            return Optional.none();
        };
        const handleTouchmove = (event) => {
            longpress$1.cancel();
            getTouch(event).each((touch) => {
                startData.on((data) => {
                    if (isFarEnough(touch, data)) {
                        startData.clear();
                    }
                });
            });
            return Optional.none();
        };
        const handleTouchend = (event) => {
            longpress$1.cancel();
            const isSame = (data) => eq(data.target, event.target);
            return startData.get().filter(isSame).map((_data) => {
                if (longpressFired.get()) {
                    event.prevent();
                    return false;
                }
                else {
                    return settings.triggerEvent(tap(), event);
                }
            });
        };
        const handlers = wrapAll([
            { key: touchstart(), value: handleTouchstart },
            { key: touchmove(), value: handleTouchmove },
            { key: touchend(), value: handleTouchend }
        ]);
        const fireIfReady = (event, type) => get$h(handlers, type).bind((handler) => handler(event));
        return {
            fireIfReady
        };
    };

    var FocusInsideModes;
    (function (FocusInsideModes) {
        FocusInsideModes["OnFocusMode"] = "onFocus";
        FocusInsideModes["OnEnterOrSpaceMode"] = "onEnterOrSpace";
        FocusInsideModes["OnApiMode"] = "onApi";
    })(FocusInsideModes || (FocusInsideModes = {}));

    const _placeholder = 'placeholder';
    const adt$7 = Adt.generate([
        { single: ['required', 'valueThunk'] },
        { multiple: ['required', 'valueThunks'] }
    ]);
    const isSubstituted = (spec) => has$2(spec, 'uiType');
    const subPlaceholder = (owner, detail, compSpec, placeholders) => {
        if (owner.exists((o) => o !== compSpec.owner)) {
            return adt$7.single(true, constant$1(compSpec));
        }
        // Ignore having to find something for the time being.
        return get$h(placeholders, compSpec.name).fold(() => {
            throw new Error('Unknown placeholder component: ' + compSpec.name + '\nKnown: [' +
                keys(placeholders) + ']\nNamespace: ' + owner.getOr('none') + '\nSpec: ' + JSON.stringify(compSpec, null, 2));
        }, (newSpec) => 
        // Must return a single/multiple type
        newSpec.replace());
    };
    const scan = (owner, detail, compSpec, placeholders) => {
        if (isSubstituted(compSpec) && compSpec.uiType === _placeholder) {
            return subPlaceholder(owner, detail, compSpec, placeholders);
        }
        else {
            return adt$7.single(false, constant$1(compSpec));
        }
    };
    const substitute = (owner, detail, compSpec, placeholders) => {
        const base = scan(owner, detail, compSpec, placeholders);
        return base.fold((req, valueThunk) => {
            const value = isSubstituted(compSpec) ? valueThunk(detail, compSpec.config, compSpec.validated) : valueThunk(detail);
            const childSpecs = get$h(value, 'components').getOr([]);
            const substituted = bind$3(childSpecs, (c) => substitute(owner, detail, c, placeholders));
            return [
                {
                    ...value,
                    components: substituted
                }
            ];
        }, (req, valuesThunk) => {
            if (isSubstituted(compSpec)) {
                const values = valuesThunk(detail, compSpec.config, compSpec.validated);
                // Allow a preprocessing step for groups before returning the components
                const preprocessor = compSpec.validated.preprocess.getOr(identity);
                return preprocessor(values);
            }
            else {
                return valuesThunk(detail);
            }
        });
    };
    const substituteAll = (owner, detail, components, placeholders) => bind$3(components, (c) => substitute(owner, detail, c, placeholders));
    const oneReplace = (label, replacements) => {
        let called = false;
        const used = () => called;
        const replace = () => {
            if (called) {
                throw new Error('Trying to use the same placeholder more than once: ' + label);
            }
            called = true;
            return replacements;
        };
        const required = () => replacements.fold((req, _) => req, (req, _) => req);
        return {
            name: constant$1(label),
            required,
            used,
            replace
        };
    };
    const substitutePlaces = (owner, detail, components, placeholders) => {
        const ps = map$1(placeholders, (ph, name) => oneReplace(name, ph));
        const outcome = substituteAll(owner, detail, components, ps);
        each(ps, (p) => {
            if (p.used() === false && p.required()) {
                throw new Error('Placeholder: ' + p.name() + ' was not found in components list\nNamespace: ' + owner.getOr('none') + '\nComponents: ' +
                    JSON.stringify(detail.components, null, 2));
            }
        });
        return outcome;
    };
    const single$2 = adt$7.single;
    const multiple = adt$7.multiple;
    const placeholder = constant$1(_placeholder);

    const adt$6 = Adt.generate([
        { required: ['data'] },
        { external: ['data'] },
        { optional: ['data'] },
        { group: ['data'] }
    ]);
    const fFactory = defaulted('factory', { sketch: identity });
    const fSchema = defaulted('schema', []);
    const fName = required$1('name');
    const fPname = field$1('pname', 'pname', defaultedThunk((typeSpec) => '<alloy.' + generate$6(typeSpec.name) + '>'), anyValue());
    // Groups cannot choose their schema.
    const fGroupSchema = customField('schema', () => [
        option$3('preprocess')
    ]);
    const fDefaults = defaulted('defaults', constant$1({}));
    const fOverrides = defaulted('overrides', constant$1({}));
    const requiredSpec = objOf([
        fFactory, fSchema, fName, fPname, fDefaults, fOverrides
    ]);
    const externalSpec = objOf([
        fFactory, fSchema, fName, fDefaults, fOverrides
    ]);
    const optionalSpec = objOf([
        fFactory, fSchema, fName, fPname, fDefaults, fOverrides
    ]);
    const groupSpec = objOf([
        fFactory, fGroupSchema, fName,
        required$1('unit'),
        fPname, fDefaults, fOverrides
    ]);
    const asNamedPart = (part) => {
        return part.fold(Optional.some, Optional.none, Optional.some, Optional.some);
    };
    const name$2 = (part) => {
        const get = (data) => data.name;
        return part.fold(get, get, get, get);
    };
    const asCommon = (part) => {
        return part.fold(identity, identity, identity, identity);
    };
    const convert = (adtConstructor, partSchema) => (spec) => {
        const data = asRawOrDie$1('Converting part type', partSchema, spec);
        return adtConstructor(data);
    };
    const required = convert(adt$6.required, requiredSpec);
    const external$1 = convert(adt$6.external, externalSpec);
    const optional = convert(adt$6.optional, optionalSpec);
    const group = convert(adt$6.group, groupSpec);
    const original = constant$1('entirety');

    var PartType = /*#__PURE__*/Object.freeze({
        __proto__: null,
        required: required,
        external: external$1,
        optional: optional,
        group: group,
        asNamedPart: asNamedPart,
        name: name$2,
        asCommon: asCommon,
        original: original
    });

    const combine$2 = (detail, data, partSpec, partValidated) => 
    // Extremely confusing names and types :(
    deepMerge(data.defaults(detail, partSpec, partValidated), partSpec, { uid: detail.partUids[data.name] }, data.overrides(detail, partSpec, partValidated));
    const subs = (owner, detail, parts) => {
        const internals = {};
        const externals = {};
        each$1(parts, (part) => {
            part.fold(
            // Internal
            (data) => {
                internals[data.pname] = single$2(true, (detail, partSpec, partValidated) => data.factory.sketch(combine$2(detail, data, partSpec, partValidated)));
            }, 
            // External
            (data) => {
                const partSpec = detail.parts[data.name];
                externals[data.name] = constant$1(data.factory.sketch(combine$2(detail, data, partSpec[original()]), partSpec) // This is missing partValidated
                );
                // no placeholders
            }, 
            // Optional
            (data) => {
                internals[data.pname] = single$2(false, (detail, partSpec, partValidated) => data.factory.sketch(combine$2(detail, data, partSpec, partValidated)));
            }, 
            // Group
            (data) => {
                internals[data.pname] = multiple(true, (detail, _partSpec, _partValidated) => {
                    const units = detail[data.name];
                    return map$2(units, (u) => 
                    // Group multiples do not take the uid because there is more than one.
                    data.factory.sketch(deepMerge(data.defaults(detail, u, _partValidated), u, data.overrides(detail, u))));
                });
            });
        });
        return {
            internals: constant$1(internals),
            externals: constant$1(externals)
        };
    };

    // TODO: Make more functional if performance isn't an issue.
    const generate$5 = (owner, parts) => {
        const r = {};
        each$1(parts, (part) => {
            asNamedPart(part).each((np) => {
                const g = doGenerateOne(owner, np.pname);
                r[np.name] = (config) => {
                    const validated = asRawOrDie$1('Part: ' + np.name + ' in ' + owner, objOf(np.schema), config);
                    return {
                        ...g,
                        config,
                        validated
                    };
                };
            });
        });
        return r;
    };
    // Does not have the config.
    const doGenerateOne = (owner, pname) => ({
        uiType: placeholder(),
        owner,
        name: pname
    });
    const generateOne$1 = (owner, pname, config) => ({
        uiType: placeholder(),
        owner,
        name: pname,
        config,
        validated: {}
    });
    const schemas = (parts) => 
    // This actually has to change. It needs to return the schemas for things that will
    // not appear in the components list, which is only externals
    bind$3(parts, (part) => part.fold(Optional.none, Optional.some, Optional.none, Optional.none).map((data) => requiredObjOf(data.name, data.schema.concat([
        snapshot(original())
    ]))).toArray());
    const names = (parts) => map$2(parts, name$2);
    const substitutes = (owner, detail, parts) => subs(owner, detail, parts);
    const components$1 = (owner, detail, internals) => substitutePlaces(Optional.some(owner), detail, detail.components, internals);
    const getPart = (component, detail, partKey) => {
        const uid = detail.partUids[partKey];
        return component.getSystem().getByUid(uid).toOptional();
    };
    const getPartOrDie = (component, detail, partKey) => getPart(component, detail, partKey).getOrDie('Could not find part: ' + partKey);
    const getParts = (component, detail, partKeys) => {
        const r = {};
        const uids = detail.partUids;
        const system = component.getSystem();
        each$1(partKeys, (pk) => {
            r[pk] = constant$1(system.getByUid(uids[pk]));
        });
        return r;
    };
    const getAllParts = (component, detail) => {
        const system = component.getSystem();
        return map$1(detail.partUids, (pUid, _k) => constant$1(system.getByUid(pUid)));
    };
    const getAllPartNames = (detail) => keys(detail.partUids);
    const getPartsOrDie = (component, detail, partKeys) => {
        const r = {};
        const uids = detail.partUids;
        const system = component.getSystem();
        each$1(partKeys, (pk) => {
            r[pk] = constant$1(system.getByUid(uids[pk]).getOrDie());
        });
        return r;
    };
    const defaultUids = (baseUid, partTypes) => {
        const partNames = names(partTypes);
        return wrapAll(map$2(partNames, (pn) => ({ key: pn, value: baseUid + '-' + pn })));
    };
    const defaultUidsSchema = (partTypes) => field$1('partUids', 'partUids', mergeWithThunk((spec) => defaultUids(spec.uid, partTypes)), anyValue());

    var AlloyParts = /*#__PURE__*/Object.freeze({
        __proto__: null,
        generate: generate$5,
        generateOne: generateOne$1,
        schemas: schemas,
        names: names,
        substitutes: substitutes,
        components: components$1,
        defaultUids: defaultUids,
        defaultUidsSchema: defaultUidsSchema,
        getAllParts: getAllParts,
        getAllPartNames: getAllPartNames,
        getPart: getPart,
        getPartOrDie: getPartOrDie,
        getParts: getParts,
        getPartsOrDie: getPartsOrDie
    });

    const allAlignments = [
        'valignCentre',
        'alignLeft',
        'alignRight',
        'alignCentre',
        'top',
        'bottom',
        'left',
        'right',
        'inset'
    ];
    const nu$6 = (xOffset, yOffset, classes, insetModifier = 1) => {
        const insetXOffset = xOffset * insetModifier;
        const insetYOffset = yOffset * insetModifier;
        const getClasses = (prop) => get$h(classes, prop).getOr([]);
        const make = (xDelta, yDelta, alignmentsOn) => {
            const alignmentsOff = difference(allAlignments, alignmentsOn);
            return {
                offset: SugarPosition(xDelta, yDelta),
                classesOn: bind$3(alignmentsOn, getClasses),
                classesOff: bind$3(alignmentsOff, getClasses)
            };
        };
        return {
            southeast: () => make(-xOffset, yOffset, ['top', 'alignLeft']),
            southwest: () => make(xOffset, yOffset, ['top', 'alignRight']),
            south: () => make(-xOffset / 2, yOffset, ['top', 'alignCentre']),
            northeast: () => make(-xOffset, -yOffset, ['bottom', 'alignLeft']),
            northwest: () => make(xOffset, -yOffset, ['bottom', 'alignRight']),
            north: () => make(-xOffset / 2, -yOffset, ['bottom', 'alignCentre']),
            east: () => make(xOffset, -yOffset / 2, ['valignCentre', 'left']),
            west: () => make(-xOffset, -yOffset / 2, ['valignCentre', 'right']),
            insetNortheast: () => make(insetXOffset, insetYOffset, ['top', 'alignLeft', 'inset']),
            insetNorthwest: () => make(-insetXOffset, insetYOffset, ['top', 'alignRight', 'inset']),
            insetNorth: () => make(-insetXOffset / 2, insetYOffset, ['top', 'alignCentre', 'inset']),
            insetSoutheast: () => make(insetXOffset, -insetYOffset, ['bottom', 'alignLeft', 'inset']),
            insetSouthwest: () => make(-insetXOffset, -insetYOffset, ['bottom', 'alignRight', 'inset']),
            insetSouth: () => make(-insetXOffset / 2, -insetYOffset, ['bottom', 'alignCentre', 'inset']),
            insetEast: () => make(-insetXOffset, -insetYOffset / 2, ['valignCentre', 'right', 'inset']),
            insetWest: () => make(insetXOffset, -insetYOffset / 2, ['valignCentre', 'left', 'inset'])
        };
    };
    const fallback = () => nu$6(0, 0, {});

    const nu$5 = (x, y, bubble, direction, placement, boundsRestriction, labelPrefix, alwaysFit = false) => ({
        x,
        y,
        bubble,
        direction,
        placement,
        restriction: boundsRestriction,
        label: `${labelPrefix}-${placement}`,
        alwaysFit
    });

    const adt$5 = Adt.generate([
        { southeast: [] },
        { southwest: [] },
        { northeast: [] },
        { northwest: [] },
        { south: [] },
        { north: [] },
        { east: [] },
        { west: [] }
    ]);
    const cata$1 = (subject, southeast, southwest, northeast, northwest, south, north, east, west) => subject.fold(southeast, southwest, northeast, northwest, south, north, east, west);
    const cataVertical = (subject, south, middle, north) => subject.fold(south, south, north, north, south, north, middle, middle);
    const cataHorizontal = (subject, east, middle, west) => subject.fold(east, west, east, west, middle, middle, east, west);
    // TODO: Simplify with the typescript approach.
    const southeast$3 = adt$5.southeast;
    const southwest$3 = adt$5.southwest;
    const northeast$3 = adt$5.northeast;
    const northwest$3 = adt$5.northwest;
    const south$3 = adt$5.south;
    const north$3 = adt$5.north;
    const east$3 = adt$5.east;
    const west$3 = adt$5.west;

    const getRestriction = (anchor, restriction) => {
        switch (restriction) {
            case 1 /* AnchorBoxBounds.LeftEdge */:
                return anchor.x;
            case 0 /* AnchorBoxBounds.RightEdge */:
                return anchor.x + anchor.width;
            case 2 /* AnchorBoxBounds.TopEdge */:
                return anchor.y;
            case 3 /* AnchorBoxBounds.BottomEdge */:
                return anchor.y + anchor.height;
        }
    };
    const boundsRestriction = (anchor, restrictions) => mapToObject(['left', 'right', 'top', 'bottom'], (dir) => get$h(restrictions, dir).map((restriction) => getRestriction(anchor, restriction)));
    const adjustBounds = (bounds$1, restriction, bubbleOffset) => {
        const applyRestriction = (dir, current) => restriction[dir].map((pos) => {
            const isVerticalAxis = dir === 'top' || dir === 'bottom';
            const offset = isVerticalAxis ? bubbleOffset.top : bubbleOffset.left;
            const comparator = dir === 'left' || dir === 'top' ? Math.max : Math.min;
            const newPos = comparator(pos, current) + offset;
            // Ensure the new restricted position is within the current bounds
            return isVerticalAxis ? clamp(newPos, bounds$1.y, bounds$1.bottom) : clamp(newPos, bounds$1.x, bounds$1.right);
        }).getOr(current);
        const adjustedLeft = applyRestriction('left', bounds$1.x);
        const adjustedTop = applyRestriction('top', bounds$1.y);
        const adjustedRight = applyRestriction('right', bounds$1.right);
        const adjustedBottom = applyRestriction('bottom', bounds$1.bottom);
        return bounds(adjustedLeft, adjustedTop, adjustedRight - adjustedLeft, adjustedBottom - adjustedTop);
    };

    /*
      Layout for menus and inline context dialogs;
      Either above or below. Never left or right.
      Aligned to the left or right of the anchor as appropriate.
     */
    const labelPrefix$2 = 'layout';
    // display element to the right, left edge against the anchor
    const eastX$1 = (anchor) => anchor.x;
    // element centre aligned horizontally with the anchor
    const middleX$1 = (anchor, element) => anchor.x + (anchor.width / 2) - (element.width / 2);
    // display element to the left, right edge against the right of the anchor
    const westX$1 = (anchor, element) => anchor.x + anchor.width - element.width;
    // display element above, bottom edge against the top of the anchor
    const northY$2 = (anchor, element) => anchor.y - element.height;
    // display element below, top edge against the bottom of the anchor
    const southY$2 = (anchor) => anchor.y + anchor.height;
    // display element below, top edge against the bottom of the anchor
    const centreY$1 = (anchor, element) => anchor.y + (anchor.height / 2) - (element.height / 2);
    const eastEdgeX$1 = (anchor) => anchor.x + anchor.width;
    const westEdgeX$1 = (anchor, element) => anchor.x - element.width;
    const southeast$2 = (anchor, element, bubbles) => nu$5(eastX$1(anchor), southY$2(anchor), bubbles.southeast(), southeast$3(), "southeast" /* Placement.Southeast */, boundsRestriction(anchor, { left: 1 /* AnchorBoxBounds.LeftEdge */, top: 3 /* AnchorBoxBounds.BottomEdge */ }), labelPrefix$2);
    const southwest$2 = (anchor, element, bubbles) => nu$5(westX$1(anchor, element), southY$2(anchor), bubbles.southwest(), southwest$3(), "southwest" /* Placement.Southwest */, boundsRestriction(anchor, { right: 0 /* AnchorBoxBounds.RightEdge */, top: 3 /* AnchorBoxBounds.BottomEdge */ }), labelPrefix$2);
    const northeast$2 = (anchor, element, bubbles) => nu$5(eastX$1(anchor), northY$2(anchor, element), bubbles.northeast(), northeast$3(), "northeast" /* Placement.Northeast */, boundsRestriction(anchor, { left: 1 /* AnchorBoxBounds.LeftEdge */, bottom: 2 /* AnchorBoxBounds.TopEdge */ }), labelPrefix$2);
    const northwest$2 = (anchor, element, bubbles) => nu$5(westX$1(anchor, element), northY$2(anchor, element), bubbles.northwest(), northwest$3(), "northwest" /* Placement.Northwest */, boundsRestriction(anchor, { right: 0 /* AnchorBoxBounds.RightEdge */, bottom: 2 /* AnchorBoxBounds.TopEdge */ }), labelPrefix$2);
    const north$2 = (anchor, element, bubbles) => nu$5(middleX$1(anchor, element), northY$2(anchor, element), bubbles.north(), north$3(), "north" /* Placement.North */, boundsRestriction(anchor, { bottom: 2 /* AnchorBoxBounds.TopEdge */ }), labelPrefix$2);
    const south$2 = (anchor, element, bubbles) => nu$5(middleX$1(anchor, element), southY$2(anchor), bubbles.south(), south$3(), "south" /* Placement.South */, boundsRestriction(anchor, { top: 3 /* AnchorBoxBounds.BottomEdge */ }), labelPrefix$2);
    const east$2 = (anchor, element, bubbles) => nu$5(eastEdgeX$1(anchor), centreY$1(anchor, element), bubbles.east(), east$3(), "east" /* Placement.East */, boundsRestriction(anchor, { left: 0 /* AnchorBoxBounds.RightEdge */ }), labelPrefix$2);
    const west$2 = (anchor, element, bubbles) => nu$5(westEdgeX$1(anchor, element), centreY$1(anchor, element), bubbles.west(), west$3(), "west" /* Placement.West */, boundsRestriction(anchor, { right: 1 /* AnchorBoxBounds.LeftEdge */ }), labelPrefix$2);
    const all$2 = () => [southeast$2, southwest$2, northeast$2, northwest$2, south$2, north$2, east$2, west$2];
    const allRtl$1 = () => [southwest$2, southeast$2, northwest$2, northeast$2, south$2, north$2, east$2, west$2];
    const aboveOrBelow = () => [northeast$2, northwest$2, southeast$2, southwest$2, north$2, south$2];
    const aboveOrBelowRtl = () => [northwest$2, northeast$2, southwest$2, southeast$2, north$2, south$2];
    const belowOrAbove = () => [southeast$2, southwest$2, northeast$2, northwest$2, south$2, north$2];
    const belowOrAboveRtl = () => [southwest$2, southeast$2, northwest$2, northeast$2, south$2, north$2];

    const placementAttribute = 'data-alloy-placement';
    const setPlacement$1 = (element, placement) => {
        set$9(element, placementAttribute, placement);
    };
    const getPlacement = (element) => getOpt(element, placementAttribute);
    const reset$2 = (element) => remove$8(element, placementAttribute);

    /*
      Layouts for things that overlay over the anchor element/box. These are designed to mirror
      the `Layout` logic.

      As an example `Layout.north` will appear horizontally centered above the anchor, whereas
      `LayoutInset.north` will appear horizontally centered overlapping the top of the anchor.
     */
    const labelPrefix$1 = 'layout-inset';
    // returns left edge of anchor - used to display element to the left, left edge against the anchor
    const westEdgeX = (anchor) => anchor.x;
    // returns middle of anchor minus half the element width - used to horizontally centre element to the anchor
    const middleX = (anchor, element) => anchor.x + (anchor.width / 2) - (element.width / 2);
    // returns right edge of anchor minus element width - used to display element to the right, right edge against the anchor
    const eastEdgeX = (anchor, element) => anchor.x + anchor.width - element.width;
    // returns top edge - used to display element to the top, top edge against the anchor
    const northY$1 = (anchor) => anchor.y;
    // returns bottom edge minus element height - used to display element at the bottom, bottom edge against the anchor
    const southY$1 = (anchor, element) => anchor.y + anchor.height - element.height;
    // returns centre of anchor minus half the element height - used to vertically centre element to the anchor
    const centreY = (anchor, element) => anchor.y + (anchor.height / 2) - (element.height / 2);
    // positions element relative to the bottom right of the anchor
    const southwest$1 = (anchor, element, bubbles) => nu$5(eastEdgeX(anchor, element), southY$1(anchor, element), bubbles.insetSouthwest(), northwest$3(), "southwest" /* Placement.Southwest */, boundsRestriction(anchor, { right: 0 /* AnchorBoxBounds.RightEdge */, bottom: 3 /* AnchorBoxBounds.BottomEdge */ }), labelPrefix$1);
    // positions element relative to the bottom left of the anchor
    const southeast$1 = (anchor, element, bubbles) => nu$5(westEdgeX(anchor), southY$1(anchor, element), bubbles.insetSoutheast(), northeast$3(), "southeast" /* Placement.Southeast */, boundsRestriction(anchor, { left: 1 /* AnchorBoxBounds.LeftEdge */, bottom: 3 /* AnchorBoxBounds.BottomEdge */ }), labelPrefix$1);
    // positions element relative to the top right of the anchor
    const northwest$1 = (anchor, element, bubbles) => nu$5(eastEdgeX(anchor, element), northY$1(anchor), bubbles.insetNorthwest(), southwest$3(), "northwest" /* Placement.Northwest */, boundsRestriction(anchor, { right: 0 /* AnchorBoxBounds.RightEdge */, top: 2 /* AnchorBoxBounds.TopEdge */ }), labelPrefix$1);
    // positions element relative to the top left of the anchor
    const northeast$1 = (anchor, element, bubbles) => nu$5(westEdgeX(anchor), northY$1(anchor), bubbles.insetNortheast(), southeast$3(), "northeast" /* Placement.Northeast */, boundsRestriction(anchor, { left: 1 /* AnchorBoxBounds.LeftEdge */, top: 2 /* AnchorBoxBounds.TopEdge */ }), labelPrefix$1);
    // positions element relative to the top of the anchor, horizontally centered
    const north$1 = (anchor, element, bubbles) => nu$5(middleX(anchor, element), northY$1(anchor), bubbles.insetNorth(), south$3(), "north" /* Placement.North */, boundsRestriction(anchor, { top: 2 /* AnchorBoxBounds.TopEdge */ }), labelPrefix$1);
    // positions element relative to the bottom of the anchor, horizontally centered
    const south$1 = (anchor, element, bubbles) => nu$5(middleX(anchor, element), southY$1(anchor, element), bubbles.insetSouth(), north$3(), "south" /* Placement.South */, boundsRestriction(anchor, { bottom: 3 /* AnchorBoxBounds.BottomEdge */ }), labelPrefix$1);
    // positions element with the right edge against the anchor, vertically centered
    const east$1 = (anchor, element, bubbles) => nu$5(eastEdgeX(anchor, element), centreY(anchor, element), bubbles.insetEast(), west$3(), "east" /* Placement.East */, boundsRestriction(anchor, { right: 0 /* AnchorBoxBounds.RightEdge */ }), labelPrefix$1);
    // positions element with the left each against the anchor, vertically centered
    const west$1 = (anchor, element, bubbles) => nu$5(westEdgeX(anchor), centreY(anchor, element), bubbles.insetWest(), east$3(), "west" /* Placement.West */, boundsRestriction(anchor, { left: 1 /* AnchorBoxBounds.LeftEdge */ }), labelPrefix$1);
    const lookupPreserveLayout = (lastPlacement) => {
        switch (lastPlacement) {
            case "north" /* Placement.North */:
                return north$1;
            case "northeast" /* Placement.Northeast */:
                return northeast$1;
            case "northwest" /* Placement.Northwest */:
                return northwest$1;
            case "south" /* Placement.South */:
                return south$1;
            case "southeast" /* Placement.Southeast */:
                return southeast$1;
            case "southwest" /* Placement.Southwest */:
                return southwest$1;
            case "east" /* Placement.East */:
                return east$1;
            case "west" /* Placement.West */:
                return west$1;
        }
    };
    const preserve$1 = (anchor, element, bubbles, placee, bounds) => {
        const layout = getPlacement(placee).map(lookupPreserveLayout).getOr(north$1);
        return layout(anchor, element, bubbles, placee, bounds);
    };
    const lookupFlippedLayout = (lastPlacement) => {
        switch (lastPlacement) {
            case "north" /* Placement.North */:
                return south$1;
            case "northeast" /* Placement.Northeast */:
                return southeast$1;
            case "northwest" /* Placement.Northwest */:
                return southwest$1;
            case "south" /* Placement.South */:
                return north$1;
            case "southeast" /* Placement.Southeast */:
                return northeast$1;
            case "southwest" /* Placement.Southwest */:
                return northwest$1;
            case "east" /* Placement.East */:
                return west$1;
            case "west" /* Placement.West */:
                return east$1;
        }
    };
    const flip = (anchor, element, bubbles, placee, bounds) => {
        const layout = getPlacement(placee).map(lookupFlippedLayout).getOr(north$1);
        return layout(anchor, element, bubbles, placee, bounds);
    };

    // applies the max-height as determined by Bounder
    const setMaxHeight = (element, maxHeight) => {
        setMax$1(element, Math.floor(maxHeight));
    };
    // adds both max-height and overflow to constrain it
    const anchored = constant$1((element, available) => {
        setMaxHeight(element, available);
        setAll(element, {
            'overflow-x': 'hidden',
            'overflow-y': 'auto'
        });
    });
    /*
     * This adds max height, but not overflow - the effect of this is that elements can grow beyond the max height,
     * but if they run off the top they're pushed down.
     *
     * If the element expands below the screen height it will be cut off, but we were already doing that.
     */
    const expandable$1 = constant$1((element, available) => {
        setMaxHeight(element, available);
    });

    // applies the max-width as determined by Bounder
    const expandable = constant$1((element, available) => {
        setMax(element, Math.floor(available));
    });

    var AttributeValue;
    (function (AttributeValue) {
        AttributeValue["TopToBottom"] = "toptobottom";
        AttributeValue["BottomToTop"] = "bottomtotop";
    })(AttributeValue || (AttributeValue = {}));
    const Attribute = 'data-alloy-vertical-dir';
    const isBottomToTopDir = (el) => closest$2(el, (current) => isElement$1(current) && get$g(current, 'data-alloy-vertical-dir') === AttributeValue.BottomToTop);

    var HighlightOnOpen;
    (function (HighlightOnOpen) {
        HighlightOnOpen[HighlightOnOpen["HighlightMenuAndItem"] = 0] = "HighlightMenuAndItem";
        HighlightOnOpen[HighlightOnOpen["HighlightJustMenu"] = 1] = "HighlightJustMenu";
        HighlightOnOpen[HighlightOnOpen["HighlightNone"] = 2] = "HighlightNone";
    })(HighlightOnOpen || (HighlightOnOpen = {}));

    const NoState = {
        init: () => nu$4({
            readState: constant$1('No State required')
        })
    };
    const nu$4 = (spec) => spec;

    const defaultEventHandler = {
        can: always,
        abort: never,
        run: noop
    };
    const nu$3 = (parts) => {
        if (!hasNonNullableKey(parts, 'can') && !hasNonNullableKey(parts, 'abort') && !hasNonNullableKey(parts, 'run')) {
            throw new Error('EventHandler defined by: ' + JSON.stringify(parts, null, 2) + ' does not have can, abort, or run!');
        }
        return {
            ...defaultEventHandler,
            ...parts
        };
    };
    const all$1 = (handlers, f) => (...args) => foldl(handlers, (acc, handler) => acc && f(handler).apply(undefined, args), true);
    const any = (handlers, f) => (...args) => foldl(handlers, (acc, handler) => acc || f(handler).apply(undefined, args), false);
    const read$1 = (handler) => isFunction(handler) ? {
        can: always,
        abort: never,
        run: handler
    } : handler;
    const fuse$1 = (handlers) => {
        const can = all$1(handlers, (handler) => handler.can);
        const abort = any(handlers, (handler) => handler.abort);
        const run = (...args) => {
            each$1(handlers, (handler) => {
                // ASSUMPTION: Return value is unimportant.
                handler.run.apply(undefined, args);
            });
        };
        return {
            can,
            abort,
            run
        };
    };

    const emit = (component, event) => {
        dispatchWith(component, component.element, event, {});
    };
    const emitWith = (component, event, properties) => {
        dispatchWith(component, component.element, event, properties);
    };
    const emitExecute = (component) => {
        emit(component, execute$5());
    };
    const dispatch = (component, target, event) => {
        dispatchWith(component, target, event, {});
    };
    const dispatchWith = (component, target, event, properties) => {
        // NOTE: The order of spreading here means that it will maintain any target that
        // exists in the current properties. Because this function has been used for situations where
        // properties is either an emulated SugarEvent with no target (see TouchEvent) or
        // for emitting custom events that have no target, this likely hasn't been a problem.
        // But until we verify that nothing is relying on this ordering, there is an alternate
        // function below called retargetAndDispatchWith, which spreads in the other direction.
        const data = {
            target,
            ...properties
        };
        component.getSystem().triggerEvent(event, target, data);
    };
    const retargetAndDispatchWith = (component, target, eventName, properties) => {
        // This is essentially the same as dispatchWith, except the spreading order
        // means that it clobbers anything in the nativeEvent with "target". It also
        // expects what is being passed in to be a real sugar event, not just a data
        // blob
        const data = {
            ...properties,
            target
        };
        component.getSystem().triggerEvent(eventName, target, data);
    };
    const dispatchEvent = (component, target, event, simulatedEvent) => {
        component.getSystem().triggerEvent(event, target, simulatedEvent.event);
    };

    const derive$2 = (configs) => wrapAll(configs);
    // const combine = (configs...);
    const abort = (name, predicate) => {
        return {
            key: name,
            value: nu$3({
                abort: predicate
            })
        };
    };
    const can = (name, predicate) => {
        return {
            key: name,
            value: nu$3({
                can: predicate
            })
        };
    };
    const preventDefault = (name) => {
        return {
            key: name,
            value: nu$3({
                run: (component, simulatedEvent) => {
                    simulatedEvent.event.prevent();
                }
            })
        };
    };
    const run$1 = (name, handler) => {
        return {
            key: name,
            value: nu$3({
                run: handler
            })
        };
    };
    // Extra can be used when your handler needs more context, and is declared in one spot
    // It's really just convenient partial application.
    const runActionExtra = (name, action, extra) => {
        return {
            key: name,
            value: nu$3({
                run: (component, simulatedEvent) => {
                    action.apply(undefined, [component, simulatedEvent].concat(extra));
                }
            })
        };
    };
    const runOnName = (name) => {
        return (handler) => run$1(name, handler);
    };
    const runOnSourceName = (name) => {
        return (handler) => ({
            key: name,
            value: nu$3({
                run: (component, simulatedEvent) => {
                    if (isSource(component, simulatedEvent)) {
                        handler(component, simulatedEvent);
                    }
                }
            })
        });
    };
    const redirectToUid = (name, uid) => {
        return run$1(name, (component, simulatedEvent) => {
            component.getSystem().getByUid(uid).each((redirectee) => {
                dispatchEvent(redirectee, redirectee.element, name, simulatedEvent);
            });
        });
    };
    const redirectToPart = (name, detail, partName) => {
        const uid = detail.partUids[partName];
        return redirectToUid(name, uid);
    };
    const runWithTarget = (name, f) => {
        return run$1(name, (component, simulatedEvent) => {
            const ev = simulatedEvent.event;
            const target = component.getSystem().getByDom(ev.target).getOrThunk(
            // If we don't find an alloy component for the target, I guess we go up the tree
            // until we find an alloy component? Performance concern?
            // TODO: Write tests for this.
            () => {
                const closest$1 = closest(ev.target, (el) => component.getSystem().getByDom(el).toOptional(), never);
                // If we still found nothing ... fire on component itself;
                return closest$1.getOr(component);
            });
            f(component, target, simulatedEvent);
        });
    };
    const cutter = (name) => {
        return run$1(name, (component, simulatedEvent) => {
            simulatedEvent.cut();
        });
    };
    const stopper = (name) => {
        return run$1(name, (component, simulatedEvent) => {
            simulatedEvent.stop();
        });
    };
    const runOnSource = (name, f) => {
        return runOnSourceName(name)(f);
    };
    const runOnAttached = runOnSourceName(attachedToDom());
    const runOnDetached = runOnSourceName(detachedFromDom());
    const runOnInit = runOnSourceName(systemInit());
    const runOnExecute$1 = runOnName(execute$5());

    // Maybe we'll need to allow add/remove
    const nu$2 = (s) => ({
        classes: isUndefined(s.classes) ? [] : s.classes,
        attributes: isUndefined(s.attributes) ? {} : s.attributes,
        styles: isUndefined(s.styles) ? {} : s.styles
    });
    const merge = (defnA, mod) => ({
        ...defnA,
        attributes: { ...defnA.attributes, ...mod.attributes },
        styles: { ...defnA.styles, ...mod.styles },
        classes: defnA.classes.concat(mod.classes)
    });

    const executeEvent = (bConfig, bState, executor) => runOnExecute$1((component) => {
        executor(component, bConfig, bState);
    });
    const loadEvent = (bConfig, bState, f) => runOnInit((component, _simulatedEvent) => {
        f(component, bConfig, bState);
    });
    const create$4 = (schema, name, active, apis, extra, state) => {
        const configSchema = objOfOnly(schema);
        const schemaSchema = optionObjOf(name, [
            optionObjOfOnly('config', schema)
        ]);
        return doCreate(configSchema, schemaSchema, name, active, apis, extra, state);
    };
    const createModes$1 = (modes, name, active, apis, extra, state) => {
        const configSchema = modes;
        const schemaSchema = optionObjOf(name, [
            optionOf('config', modes)
        ]);
        return doCreate(configSchema, schemaSchema, name, active, apis, extra, state);
    };
    const wrapApi = (bName, apiFunction, apiName) => {
        const f = (component, ...rest) => {
            const args = [component].concat(rest);
            return component.config({
                name: constant$1(bName)
            }).fold(() => {
                throw new Error('We could not find any behaviour configuration for: ' + bName + '. Using API: ' + apiName);
            }, (info) => {
                const rest = Array.prototype.slice.call(args, 1);
                return apiFunction.apply(undefined, [component, info.config, info.state].concat(rest));
            });
        };
        return markAsBehaviourApi(f, apiName, apiFunction);
    };
    // I think the "revoke" idea is fragile at best.
    const revokeBehaviour = (name) => ({
        key: name,
        value: undefined
    });
    const doCreate = (configSchema, schemaSchema, name, active, apis, extra, state) => {
        const getConfig = (info) => hasNonNullableKey(info, name) ? info[name]() : Optional.none();
        const wrappedApis = map$1(apis, (apiF, apiName) => wrapApi(name, apiF, apiName));
        const wrappedExtra = map$1(extra, (extraF, extraName) => markAsExtraApi(extraF, extraName));
        const me = {
            ...wrappedExtra,
            ...wrappedApis,
            revoke: curry(revokeBehaviour, name),
            config: (spec) => {
                const prepared = asRawOrDie$1(name + '-config', configSchema, spec);
                return {
                    key: name,
                    value: {
                        config: prepared,
                        me,
                        configAsRaw: cached(() => asRawOrDie$1(name + '-config', configSchema, spec)),
                        initialConfig: spec,
                        state
                    }
                };
            },
            schema: constant$1(schemaSchema),
            exhibit: (info, base) => {
                return lift2(getConfig(info), get$h(active, 'exhibit'), (behaviourInfo, exhibitor) => {
                    return exhibitor(base, behaviourInfo.config, behaviourInfo.state);
                }).getOrThunk(() => nu$2({}));
            },
            name: constant$1(name),
            handlers: (info) => {
                return getConfig(info).map((behaviourInfo) => {
                    const getEvents = get$h(active, 'events').getOr(() => ({}));
                    return getEvents(behaviourInfo.config, behaviourInfo.state);
                }).getOr({});
            }
        };
        return me;
    };

    const derive$1 = (capabilities) => wrapAll(capabilities);
    const simpleSchema = objOfOnly([
        required$1('fields'),
        required$1('name'),
        defaulted('active', {}),
        defaulted('apis', {}),
        defaulted('state', NoState),
        defaulted('extra', {})
    ]);
    const create$3 = (data) => {
        const value = asRawOrDie$1('Creating behaviour: ' + data.name, simpleSchema, data);
        return create$4(value.fields, value.name, value.active, value.apis, value.extra, value.state);
    };
    const modeSchema = objOfOnly([
        required$1('branchKey'),
        required$1('branches'),
        required$1('name'),
        defaulted('active', {}),
        defaulted('apis', {}),
        defaulted('state', NoState),
        defaulted('extra', {})
    ]);
    const createModes = (data) => {
        const value = asRawOrDie$1('Creating behaviour: ' + data.name, modeSchema, data);
        return createModes$1(choose$1(value.branchKey, value.branches), value.name, value.active, value.apis, value.extra, value.state);
    };
    const revoke = constant$1(undefined);

    // AlloyEventKeyAndHandler type argument needs to be any here to satisfy an array of handlers
    // where each item can be any subtype of EventFormat we can't use <T extends EventFormat> since
    // then each item would have to be the same type
    const events$i = (name, eventHandlers) => {
        const events = derive$2(eventHandlers);
        return create$3({
            fields: [
                required$1('enabled')
            ],
            name,
            active: {
                events: constant$1(events)
            }
        });
    };
    const config = (name, eventHandlers) => {
        const me = events$i(name, eventHandlers);
        return {
            key: name,
            value: {
                config: {},
                me,
                configAsRaw: constant$1({}),
                initialConfig: {},
                state: NoState
            }
        };
    };

    const SetupBehaviourCellState = (initialState) => {
        const init = () => {
            const cell = Cell(initialState);
            const get = () => cell.get();
            const set = (newState) => cell.set(newState);
            const clear = () => cell.set(initialState);
            const readState = () => cell.get();
            return {
                get,
                set,
                clear,
                readState
            };
        };
        return {
            init
        };
    };

    const focus$2 = (component, focusConfig) => {
        if (!focusConfig.ignore) {
            focus$4(component.element);
            focusConfig.onFocus(component);
        }
    };
    const blur = (component, focusConfig) => {
        if (!focusConfig.ignore) {
            blur$1(component.element);
        }
    };
    const isFocused = (component) => hasFocus(component.element);

    var FocusApis = /*#__PURE__*/Object.freeze({
        __proto__: null,
        focus: focus$2,
        blur: blur,
        isFocused: isFocused
    });

    // TODO: DomModification types
    const exhibit$6 = (base, focusConfig) => {
        const mod = focusConfig.ignore ? {} : {
            attributes: {
                tabindex: '-1'
            }
        };
        return nu$2(mod);
    };
    const events$h = (focusConfig) => derive$2([
        run$1(focus$3(), (component, simulatedEvent) => {
            focus$2(component, focusConfig);
            simulatedEvent.stop();
        })
    ].concat(focusConfig.stopMousedown ? [
        run$1(mousedown(), (_, simulatedEvent) => {
            // This setting is often used in tandem with ignoreFocus. Basically, if you
            // don't prevent default on a menu that has fake focus, then it can transfer
            // focus to the outer body when they click on it, which can break things
            // which dismiss on blur (e.g. typeahead)
            simulatedEvent.event.prevent();
        })
    ] : []));

    var ActiveFocus = /*#__PURE__*/Object.freeze({
        __proto__: null,
        exhibit: exhibit$6,
        events: events$h
    });

    var FocusSchema = [
        // TODO: Work out when we want to  call this. Only when it is has changed?
        onHandler('onFocus'),
        defaulted('stopMousedown', false),
        defaulted('ignore', false)
    ];

    const Focusing = create$3({
        fields: FocusSchema,
        name: 'focusing',
        active: ActiveFocus,
        apis: FocusApis
        // Consider adding isFocused an an extra
    });

    const BACKSPACE = [8];
    const TAB = [9];
    const ENTER = [13];
    const ESCAPE = [27];
    const SPACE = [32];
    const LEFT = [37];
    const UP = [38];
    const RIGHT = [39];
    const DOWN = [40];

    const closeTooltips = constant$1('tooltipping.close.all');
    const dismissPopups = constant$1('dismiss.popups');
    const repositionPopups = constant$1('reposition.popups');
    const mouseReleased = constant$1('mouse.released');

    const cyclePrev = (values, index, predicate) => {
        const before = reverse(values.slice(0, index));
        const after = reverse(values.slice(index + 1));
        return find$5(before.concat(after), predicate);
    };
    const tryPrev = (values, index, predicate) => {
        const before = reverse(values.slice(0, index));
        return find$5(before, predicate);
    };
    const cycleNext = (values, index, predicate) => {
        const before = values.slice(0, index);
        const after = values.slice(index + 1);
        return find$5(after.concat(before), predicate);
    };
    const tryNext = (values, index, predicate) => {
        const after = values.slice(index + 1);
        return find$5(after, predicate);
    };

    const inSet = (keys) => (event) => {
        const raw = event.raw;
        return contains$2(keys, raw.which);
    };
    const and = (preds) => (event) => forall(preds, (pred) => pred(event));
    const isShift$1 = (event) => {
        const raw = event.raw;
        return raw.shiftKey === true;
    };
    const isControl = (event) => {
        const raw = event.raw;
        return raw.ctrlKey === true;
    };
    const isNotShift = not(isShift$1);

    const rule = (matches, action) => ({
        matches,
        classification: action
    });
    const choose = (transitions, event) => {
        const transition = find$5(transitions, (t) => t.matches(event));
        return transition.map((t) => t.classification);
    };

    // THIS IS NOT API YET
    const dehighlightAllExcept = (component, hConfig, hState, skip) => {
        const highlighted = descendants(component.element, '.' + hConfig.highlightClass);
        each$1(highlighted, (h) => {
            // We don't want to dehighlight anything that should be skipped.
            // Generally, this is because we are about to highlight that thing.
            const shouldSkip = exists(skip, (skipComp) => eq(skipComp.element, h));
            if (!shouldSkip) {
                remove$3(h, hConfig.highlightClass);
                component.getSystem().getByDom(h).each((target) => {
                    hConfig.onDehighlight(component, target);
                    emit(target, dehighlight$1());
                });
            }
        });
    };
    const dehighlightAll = (component, hConfig, hState) => dehighlightAllExcept(component, hConfig, hState, []);
    const dehighlight = (component, hConfig, hState, target) => {
        // Only act if it was highlighted.
        if (isHighlighted(component, hConfig, hState, target)) {
            remove$3(target.element, hConfig.highlightClass);
            hConfig.onDehighlight(component, target);
            emit(target, dehighlight$1());
        }
    };
    const highlight = (component, hConfig, hState, target) => {
        // If asked to highlight something, dehighlight everything else first except
        // for the new thing we are going to highlight. It's a rare case, but we don't
        // want to get an onDehighlight, onHighlight for the same item on a highlight call.
        // We also don't want to call onHighlight if it was already highlighted.
        //
        // Note, that there is an important distinction here: highlight is NOT a no-op
        // if target is already highlighted, because it will still dehighlight everything else.
        // However, it won't fire any onHighlight or onDehighlight handlers for the already
        // highlighted item. I'm not sure if this is behaviour we need to maintain, but it is now
        // tested. A simpler approach might just be to not do anything if it's already highlighted,
        // but that could leave us in an inconsistent state, where multiple items have highlights
        // even after a highlight call. This way, highlight validates the highlights in the
        // component, and ensures there is only one thing highlighted.
        dehighlightAllExcept(component, hConfig, hState, [target]);
        if (!isHighlighted(component, hConfig, hState, target)) {
            add$2(target.element, hConfig.highlightClass);
            hConfig.onHighlight(component, target);
            emit(target, highlight$1());
        }
    };
    const highlightFirst = (component, hConfig, hState) => {
        getFirst(component, hConfig).each((firstComp) => {
            highlight(component, hConfig, hState, firstComp);
        });
    };
    const highlightLast = (component, hConfig, hState) => {
        getLast(component, hConfig).each((lastComp) => {
            highlight(component, hConfig, hState, lastComp);
        });
    };
    const highlightAt = (component, hConfig, hState, index) => {
        getByIndex(component, hConfig, hState, index).fold((err) => {
            throw err;
        }, (firstComp) => {
            highlight(component, hConfig, hState, firstComp);
        });
    };
    const highlightBy = (component, hConfig, hState, predicate) => {
        const candidates = getCandidates(component, hConfig);
        const targetComp = find$5(candidates, predicate);
        targetComp.each((c) => {
            highlight(component, hConfig, hState, c);
        });
    };
    const isHighlighted = (component, hConfig, hState, queryTarget) => has(queryTarget.element, hConfig.highlightClass);
    const getHighlighted = (component, hConfig, _hState) => descendant(component.element, '.' + hConfig.highlightClass).bind((e) => component.getSystem().getByDom(e).toOptional());
    const getByIndex = (component, hConfig, hState, index) => {
        const items = descendants(component.element, '.' + hConfig.itemClass);
        return Optional.from(items[index]).fold(() => Result.error(new Error('No element found with index ' + index)), component.getSystem().getByDom);
    };
    const getFirst = (component, hConfig, _hState) => descendant(component.element, '.' + hConfig.itemClass).bind((e) => component.getSystem().getByDom(e).toOptional());
    const getLast = (component, hConfig, _hState) => {
        const items = descendants(component.element, '.' + hConfig.itemClass);
        const last = items.length > 0 ? Optional.some(items[items.length - 1]) : Optional.none();
        return last.bind((c) => component.getSystem().getByDom(c).toOptional());
    };
    const getDelta$2 = (component, hConfig, hState, delta) => {
        const items = descendants(component.element, '.' + hConfig.itemClass);
        const current = findIndex$1(items, (item) => has(item, hConfig.highlightClass));
        return current.bind((selected) => {
            const dest = cycleBy(selected, delta, 0, items.length - 1);
            return component.getSystem().getByDom(items[dest]).toOptional();
        });
    };
    const getPrevious = (component, hConfig, hState) => getDelta$2(component, hConfig, hState, -1);
    const getNext = (component, hConfig, hState) => getDelta$2(component, hConfig, hState, +1);
    const getCandidates = (component, hConfig, _hState) => {
        const items = descendants(component.element, '.' + hConfig.itemClass);
        return cat(map$2(items, (i) => component.getSystem().getByDom(i).toOptional()));
    };

    var HighlightApis = /*#__PURE__*/Object.freeze({
        __proto__: null,
        dehighlightAll: dehighlightAll,
        dehighlight: dehighlight,
        highlight: highlight,
        highlightFirst: highlightFirst,
        highlightLast: highlightLast,
        highlightAt: highlightAt,
        highlightBy: highlightBy,
        isHighlighted: isHighlighted,
        getHighlighted: getHighlighted,
        getFirst: getFirst,
        getLast: getLast,
        getPrevious: getPrevious,
        getNext: getNext,
        getCandidates: getCandidates
    });

    var HighlightSchema = [
        required$1('highlightClass'),
        required$1('itemClass'),
        onHandler('onHighlight'),
        onHandler('onDehighlight')
    ];

    const Highlighting = create$3({
        fields: HighlightSchema,
        name: 'highlighting',
        apis: HighlightApis
    });

    const reportFocusShifting = (component, prevFocus, newFocus) => {
        const noChange = prevFocus.exists((p) => newFocus.exists((n) => eq(n, p)));
        if (!noChange) {
            emitWith(component, focusShifted(), {
                prevFocus,
                newFocus
            });
        }
    };
    const dom$2 = () => {
        const get = (component) => search(component.element);
        const set = (component, focusee) => {
            const prevFocus = get(component);
            component.getSystem().triggerFocus(focusee, component.element);
            const newFocus = get(component);
            reportFocusShifting(component, prevFocus, newFocus);
        };
        return {
            get,
            set
        };
    };
    const highlights = () => {
        const get = (component) => Highlighting.getHighlighted(component).map((item) => item.element);
        const set = (component, element) => {
            const prevFocus = get(component);
            component.getSystem().getByDom(element).fold(noop, (item) => {
                Highlighting.highlight(component, item);
            });
            const newFocus = get(component);
            reportFocusShifting(component, prevFocus, newFocus);
        };
        return {
            get,
            set
        };
    };

    const typical = (infoSchema, stateInit, getKeydownRules, getKeyupRules, optFocusIn) => {
        const schema = () => infoSchema.concat([
            defaulted('focusManager', dom$2()),
            defaultedOf('focusInside', 'onFocus', valueOf((val) => contains$2(['onFocus', 'onEnterOrSpace', 'onApi'], val) ? Result.value(val) : Result.error('Invalid value for focusInside'))),
            output$1('handler', me),
            output$1('state', stateInit),
            output$1('sendFocusIn', optFocusIn)
        ]);
        const processKey = (component, simulatedEvent, getRules, keyingConfig, keyingState) => {
            const rules = getRules(component, simulatedEvent, keyingConfig, keyingState);
            return choose(rules, simulatedEvent.event).bind((rule) => rule(component, simulatedEvent, keyingConfig, keyingState));
        };
        const toEvents = (keyingConfig, keyingState) => {
            const onFocusHandler = keyingConfig.focusInside !== FocusInsideModes.OnFocusMode
                ? Optional.none()
                : optFocusIn(keyingConfig).map((focusIn) => run$1(focus$3(), (component, simulatedEvent) => {
                    focusIn(component, keyingConfig, keyingState);
                    simulatedEvent.stop();
                }));
            // On enter or space on root element, if using EnterOrSpace focus mode, fire a focusIn on the component
            const tryGoInsideComponent = (component, simulatedEvent) => {
                const isEnterOrSpace = inSet(SPACE.concat(ENTER))(simulatedEvent.event);
                if (keyingConfig.focusInside === FocusInsideModes.OnEnterOrSpaceMode && isEnterOrSpace && isSource(component, simulatedEvent)) {
                    optFocusIn(keyingConfig).each((focusIn) => {
                        focusIn(component, keyingConfig, keyingState);
                        simulatedEvent.stop();
                    });
                }
            };
            const keyboardEvents = [
                run$1(keydown(), (component, simulatedEvent) => {
                    processKey(component, simulatedEvent, getKeydownRules, keyingConfig, keyingState).fold(() => {
                        // Key wasn't handled ... so see if we should enter into the component (focusIn)
                        tryGoInsideComponent(component, simulatedEvent);
                    }, (_) => {
                        simulatedEvent.stop();
                    });
                }),
                run$1(keyup(), (component, simulatedEvent) => {
                    processKey(component, simulatedEvent, getKeyupRules, keyingConfig, keyingState).each((_) => {
                        simulatedEvent.stop();
                    });
                })
            ];
            return derive$2(onFocusHandler.toArray().concat(keyboardEvents));
        };
        const me = {
            schema,
            processKey,
            toEvents
        };
        return me;
    };

    const create$2 = (cyclicField) => {
        const schema = [
            option$3('onEscape'),
            option$3('onEnter'),
            defaulted('selector', '[data-alloy-tabstop="true"]:not(:disabled)'),
            defaulted('firstTabstop', 0),
            defaulted('useTabstopAt', always),
            // Maybe later we should just expose isVisible
            option$3('visibilitySelector')
        ].concat([
            cyclicField
        ]);
        // TODO: Test this
        const isVisible = (tabbingConfig, element) => {
            const target = tabbingConfig.visibilitySelector
                .bind((sel) => closest$3(element, sel))
                .getOr(element);
            // NOTE: We can't use Visibility.isVisible, because the toolbar has width when it has closed, just not height.
            return get$d(target) > 0;
        };
        const findInitial = (component, tabbingConfig) => {
            const tabstops = descendants(component.element, tabbingConfig.selector);
            const visibles = filter$2(tabstops, (elem) => isVisible(tabbingConfig, elem));
            return Optional.from(visibles[tabbingConfig.firstTabstop]);
        };
        const findCurrent = (component, tabbingConfig) => tabbingConfig.focusManager.get(component)
            .bind((elem) => closest$3(elem, tabbingConfig.selector));
        const isTabstop = (tabbingConfig, element) => isVisible(tabbingConfig, element) && tabbingConfig.useTabstopAt(element);
        // Fire an alloy focus on the first visible element that matches the selector
        const focusIn = (component, tabbingConfig, _tabbingState) => {
            findInitial(component, tabbingConfig).each((target) => {
                tabbingConfig.focusManager.set(component, target);
            });
        };
        const goFromTabstop = (component, tabstops, stopIndex, tabbingConfig, cycle) => cycle(tabstops, stopIndex, (elem) => isTabstop(tabbingConfig, elem))
            .fold(
        // Even if there is only one, still capture the event if cycling
        () => tabbingConfig.cyclic ? Optional.some(true) : Optional.none(), (target) => {
            tabbingConfig.focusManager.set(component, target);
            // Kill the event
            return Optional.some(true);
        });
        const go = (component, _simulatedEvent, tabbingConfig, cycle) => {
            // 1. Find our current tabstop
            // 2. Find the index of that tabstop
            // 3. Cycle the tabstop
            // 4. Fire alloy focus on the resultant tabstop
            const tabstops = filter$2(descendants(component.element, tabbingConfig.selector), (element) => isVisible(tabbingConfig, element));
            return findCurrent(component, tabbingConfig).bind((tabstop) => {
                // focused component
                const optStopIndex = findIndex$1(tabstops, curry(eq, tabstop));
                return optStopIndex.bind((stopIndex) => goFromTabstop(component, tabstops, stopIndex, tabbingConfig, cycle));
            });
        };
        const goBackwards = (component, simulatedEvent, tabbingConfig) => {
            const navigate = tabbingConfig.cyclic ? cyclePrev : tryPrev;
            return go(component, simulatedEvent, tabbingConfig, navigate);
        };
        const goForwards = (component, simulatedEvent, tabbingConfig) => {
            const navigate = tabbingConfig.cyclic ? cycleNext : tryNext;
            return go(component, simulatedEvent, tabbingConfig, navigate);
        };
        const isFirstChild = (elem) => parentNode(elem).bind(firstChild).exists((child) => eq(child, elem));
        const goFromPseudoTabstop = (component, simulatedEvent, tabbingConfig) => findCurrent(component, tabbingConfig).filter((elem) => !tabbingConfig.useTabstopAt(elem))
            .bind((elem) => (isFirstChild(elem) ? goBackwards : goForwards)(component, simulatedEvent, tabbingConfig));
        const execute = (component, simulatedEvent, tabbingConfig) => tabbingConfig.onEnter.bind((f) => f(component, simulatedEvent));
        const exit = (component, simulatedEvent, tabbingConfig) => {
            component.getSystem().broadcastOn([closeTooltips()], {
                closedTooltip: () => {
                    simulatedEvent.stop();
                }
            });
            if (!simulatedEvent.isStopped()) {
                return tabbingConfig.onEscape.bind((f) => f(component, simulatedEvent));
            }
            else {
                return Optional.none();
            }
        };
        const getKeydownRules = constant$1([
            rule(and([isShift$1, inSet(TAB)]), goBackwards),
            rule(inSet(TAB), goForwards),
            rule(and([isNotShift, inSet(ENTER)]), execute)
        ]);
        const getKeyupRules = constant$1([
            rule(inSet(ESCAPE), exit),
            rule(inSet(TAB), goFromPseudoTabstop),
        ]);
        return typical(schema, NoState.init, getKeydownRules, getKeyupRules, () => Optional.some(focusIn));
    };

    var AcyclicType = create$2(customField('cyclic', never));

    var CyclicType = create$2(customField('cyclic', always));

    const inside = (target) => ((isTag('input')(target) && get$g(target, 'type') !== 'radio') ||
        isTag('textarea')(target));

    const doDefaultExecute = (component, _simulatedEvent, focused) => {
        // Note, we use to pass through simulatedEvent here and make target: component. This simplification
        // may be a problem
        dispatch(component, focused, execute$5());
        return Optional.some(true);
    };
    const defaultExecute = (component, simulatedEvent, focused) => {
        const isComplex = inside(focused) && inSet(SPACE)(simulatedEvent.event);
        return isComplex ? Optional.none() : doDefaultExecute(component, simulatedEvent, focused);
    };
    // On Firefox, pressing space fires a click event if the element maintains focus and fires a keyup. This
    // stops the keyup, which should stop the click. We might want to make this only work for buttons and Firefox etc,
    // but at this stage it's cleaner to just always do it. It makes sense that Keying that handles space should handle
    // keyup also. This does make the name confusing, though.
    const stopEventForFirefox = (_component, _simulatedEvent) => Optional.some(true);

    const schema$y = [
        defaulted('execute', defaultExecute),
        defaulted('useSpace', false),
        defaulted('useEnter', true),
        defaulted('useControlEnter', false),
        defaulted('useDown', false)
    ];
    const execute$4 = (component, simulatedEvent, executeConfig) => executeConfig.execute(component, simulatedEvent, component.element);
    const getKeydownRules$5 = (component, _simulatedEvent, executeConfig, _executeState) => {
        const spaceExec = executeConfig.useSpace && !inside(component.element) ? SPACE : [];
        const enterExec = executeConfig.useEnter ? ENTER : [];
        const downExec = executeConfig.useDown ? DOWN : [];
        const execKeys = spaceExec.concat(enterExec).concat(downExec);
        return [
            rule(inSet(execKeys), execute$4)
        ].concat(executeConfig.useControlEnter ? [
            rule(and([isControl, inSet(ENTER)]), execute$4)
        ] : []);
    };
    const getKeyupRules$5 = (component, _simulatedEvent, executeConfig, _executeState) => executeConfig.useSpace && !inside(component.element) ?
        [rule(inSet(SPACE), stopEventForFirefox)] :
        [];
    var ExecutionType = typical(schema$y, NoState.init, getKeydownRules$5, getKeyupRules$5, () => Optional.none());

    const flatgrid$1 = () => {
        const dimensions = value$2();
        const setGridSize = (numRows, numColumns) => {
            dimensions.set({ numRows, numColumns });
        };
        const getNumRows = () => dimensions.get().map((d) => d.numRows);
        const getNumColumns = () => dimensions.get().map((d) => d.numColumns);
        return nu$4({
            readState: () => dimensions.get().map((d) => ({
                numRows: String(d.numRows),
                numColumns: String(d.numColumns)
            })).getOr({
                numRows: '?',
                numColumns: '?'
            }),
            setGridSize,
            getNumRows,
            getNumColumns
        });
    };
    const init$g = (spec) => spec.state(spec);

    var KeyingState = /*#__PURE__*/Object.freeze({
        __proto__: null,
        flatgrid: flatgrid$1,
        init: init$g
    });

    // Looks up direction (considering LTR and RTL), finds the focused element,
    // and tries to move. If it succeeds, triggers focus and kills the event.
    const useH = (movement) => (component, simulatedEvent, config, state) => {
        const move = movement(component.element);
        return use(move, component, simulatedEvent, config, state);
    };
    const west = (moveLeft, moveRight) => {
        const movement = onDirection(moveLeft, moveRight);
        return useH(movement);
    };
    const east = (moveLeft, moveRight) => {
        const movement = onDirection(moveRight, moveLeft);
        return useH(movement);
    };
    const useV = (move) => (component, simulatedEvent, config, state) => use(move, component, simulatedEvent, config, state);
    const use = (move, component, simulatedEvent, config, state) => {
        const outcome = config.focusManager.get(component).bind((focused) => move(component.element, focused, config, state));
        return outcome.map((newFocus) => {
            config.focusManager.set(component, newFocus);
            return true;
        });
    };
    const north = useV;
    const south = useV;
    const move$1 = useV;

    const locate = (candidates, predicate) => findIndex$1(candidates, predicate).map((index) => ({
        index,
        candidates
    }));

    const locateVisible = (container, current, selector) => {
        const predicate = (x) => eq(x, current);
        const candidates = descendants(container, selector);
        const visible = filter$2(candidates, isVisible);
        return locate(visible, predicate);
    };
    const findIndex = (elements, target) => findIndex$1(elements, (elem) => eq(target, elem));

    const withGrid = (values, index, numCols, f) => {
        const oldRow = Math.floor(index / numCols);
        const oldColumn = index % numCols;
        return f(oldRow, oldColumn).bind((address) => {
            const newIndex = address.row * numCols + address.column;
            return newIndex >= 0 && newIndex < values.length ? Optional.some(values[newIndex]) : Optional.none();
        });
    };
    const cycleHorizontal$1 = (values, index, numRows, numCols, delta) => withGrid(values, index, numCols, (oldRow, oldColumn) => {
        const onLastRow = oldRow === numRows - 1;
        const colsInRow = onLastRow ? values.length - (oldRow * numCols) : numCols;
        const newColumn = cycleBy(oldColumn, delta, 0, colsInRow - 1);
        return Optional.some({
            row: oldRow,
            column: newColumn
        });
    });
    const cycleVertical$1 = (values, index, numRows, numCols, delta) => withGrid(values, index, numCols, (oldRow, oldColumn) => {
        const newRow = cycleBy(oldRow, delta, 0, numRows - 1);
        const onLastRow = newRow === numRows - 1;
        const colsInRow = onLastRow ? values.length - (newRow * numCols) : numCols;
        const newCol = clamp(oldColumn, 0, colsInRow - 1);
        return Optional.some({
            row: newRow,
            column: newCol
        });
    });
    const cycleRight$1 = (values, index, numRows, numCols) => cycleHorizontal$1(values, index, numRows, numCols, +1);
    const cycleLeft$1 = (values, index, numRows, numCols) => cycleHorizontal$1(values, index, numRows, numCols, -1);
    const cycleUp$1 = (values, index, numRows, numCols) => cycleVertical$1(values, index, numRows, numCols, -1);
    const cycleDown$1 = (values, index, numRows, numCols) => cycleVertical$1(values, index, numRows, numCols, +1);

    const schema$x = [
        required$1('selector'),
        defaulted('execute', defaultExecute),
        onKeyboardHandler('onEscape'),
        defaulted('captureTab', false),
        initSize()
    ];
    const focusIn$4 = (component, gridConfig, _gridState) => {
        descendant(component.element, gridConfig.selector).each((first) => {
            gridConfig.focusManager.set(component, first);
        });
    };
    const findCurrent$1 = (component, gridConfig) => gridConfig.focusManager.get(component).bind((elem) => closest$3(elem, gridConfig.selector));
    const execute$3 = (component, simulatedEvent, gridConfig, _gridState) => findCurrent$1(component, gridConfig)
        .bind((focused) => gridConfig.execute(component, simulatedEvent, focused));
    const doMove$2 = (cycle) => (element, focused, gridConfig, gridState) => locateVisible(element, focused, gridConfig.selector)
        .bind((identified) => cycle(identified.candidates, identified.index, gridState.getNumRows().getOr(gridConfig.initSize.numRows), gridState.getNumColumns().getOr(gridConfig.initSize.numColumns)));
    const handleTab = (_component, _simulatedEvent, gridConfig) => gridConfig.captureTab ? Optional.some(true) : Optional.none();
    const doEscape$1 = (component, simulatedEvent, gridConfig) => gridConfig.onEscape(component, simulatedEvent);
    const moveLeft$3 = doMove$2(cycleLeft$1);
    const moveRight$3 = doMove$2(cycleRight$1);
    const moveNorth$1 = doMove$2(cycleUp$1);
    const moveSouth$1 = doMove$2(cycleDown$1);
    const getKeydownRules$4 = constant$1([
        rule(inSet(LEFT), west(moveLeft$3, moveRight$3)),
        rule(inSet(RIGHT), east(moveLeft$3, moveRight$3)),
        rule(inSet(UP), north(moveNorth$1)),
        rule(inSet(DOWN), south(moveSouth$1)),
        rule(and([isShift$1, inSet(TAB)]), handleTab),
        rule(and([isNotShift, inSet(TAB)]), handleTab),
        // Probably should make whether space is used configurable
        rule(inSet(SPACE.concat(ENTER)), execute$3)
    ]);
    const getKeyupRules$4 = constant$1([
        rule(inSet(ESCAPE), doEscape$1),
        rule(inSet(SPACE), stopEventForFirefox)
    ]);
    var FlatgridType = typical(schema$x, flatgrid$1, getKeydownRules$4, getKeyupRules$4, () => Optional.some(focusIn$4));

    const f = (container, selector, current, delta, getNewIndex) => {
        const isDisabledButton = (candidate) => name$3(candidate) === 'button' && get$g(candidate, 'disabled') === 'disabled';
        const tryNewIndex = (initial, index, candidates) => getNewIndex(initial, index, delta, 0, candidates.length - 1, candidates[index], (newIndex) => isDisabledButton(candidates[newIndex]) ?
            tryNewIndex(initial, newIndex, candidates) :
            Optional.from(candidates[newIndex]));
        // I wonder if this will be a problem when the focused element is invisible (shouldn't happen)
        return locateVisible(container, current, selector).bind((identified) => {
            const index = identified.index;
            const candidates = identified.candidates;
            return tryNewIndex(index, index, candidates);
        });
    };
    const horizontalWithoutCycles = (container, selector, current, delta) => f(container, selector, current, delta, (prevIndex, v, d, min, max, oldCandidate, onNewIndex) => {
        const newIndex = clamp(v + d, min, max);
        return newIndex === prevIndex ? Optional.from(oldCandidate) : onNewIndex(newIndex);
    });
    const horizontal = (container, selector, current, delta) => f(container, selector, current, delta, (prevIndex, v, d, min, max, _oldCandidate, onNewIndex) => {
        const newIndex = cycleBy(v, d, min, max);
        // If we've cycled back to the original index, we've failed to find a new valid candidate
        return newIndex === prevIndex ? Optional.none() : onNewIndex(newIndex);
    });

    const schema$w = [
        required$1('selector'),
        defaulted('getInitial', Optional.none),
        defaulted('execute', defaultExecute),
        onKeyboardHandler('onEscape'),
        defaulted('executeOnMove', false),
        defaulted('allowVertical', true),
        defaulted('allowHorizontal', true),
        defaulted('cycles', true)
    ];
    // TODO: Remove dupe.
    // TODO: Probably use this for not just execution.
    const findCurrent = (component, flowConfig) => flowConfig.focusManager.get(component).bind((elem) => closest$3(elem, flowConfig.selector));
    const execute$2 = (component, simulatedEvent, flowConfig) => findCurrent(component, flowConfig).bind((focused) => flowConfig.execute(component, simulatedEvent, focused));
    const focusIn$3 = (component, flowConfig, _state) => {
        flowConfig.getInitial(component).orThunk(() => descendant(component.element, flowConfig.selector)).each((first) => {
            flowConfig.focusManager.set(component, first);
        });
    };
    const moveLeft$2 = (element, focused, info) => (info.cycles ? horizontal : horizontalWithoutCycles)(element, info.selector, focused, -1);
    const moveRight$2 = (element, focused, info) => (info.cycles ? horizontal : horizontalWithoutCycles)(element, info.selector, focused, +1);
    const doMove$1 = (movement) => (component, simulatedEvent, flowConfig, flowState) => movement(component, simulatedEvent, flowConfig, flowState).bind(() => flowConfig.executeOnMove ?
        execute$2(component, simulatedEvent, flowConfig) :
        Optional.some(true));
    const doEscape = (component, simulatedEvent, flowConfig) => flowConfig.onEscape(component, simulatedEvent);
    const getKeydownRules$3 = (_component, _se, flowConfig, _flowState) => {
        const westMovers = [...flowConfig.allowHorizontal ? LEFT : []].concat(flowConfig.allowVertical ? UP : []);
        const eastMovers = [...flowConfig.allowHorizontal ? RIGHT : []].concat(flowConfig.allowVertical ? DOWN : []);
        return [
            rule(inSet(westMovers), doMove$1(west(moveLeft$2, moveRight$2))),
            rule(inSet(eastMovers), doMove$1(east(moveLeft$2, moveRight$2))),
            rule(inSet(ENTER), execute$2),
            rule(inSet(SPACE), execute$2)
        ];
    };
    const getKeyupRules$3 = constant$1([
        rule(inSet(SPACE), stopEventForFirefox),
        rule(inSet(ESCAPE), doEscape)
    ]);
    var FlowType = typical(schema$w, NoState.init, getKeydownRules$3, getKeyupRules$3, () => Optional.some(focusIn$3));

    const toCell = (matrix, rowIndex, columnIndex) => Optional.from(matrix[rowIndex]).bind((row) => Optional.from(row[columnIndex]).map((cell) => ({
        rowIndex,
        columnIndex,
        cell
    })));
    const cycleHorizontal = (matrix, rowIndex, startCol, deltaCol) => {
        const row = matrix[rowIndex];
        const colsInRow = row.length;
        const newColIndex = cycleBy(startCol, deltaCol, 0, colsInRow - 1);
        return toCell(matrix, rowIndex, newColIndex);
    };
    const cycleVertical = (matrix, colIndex, startRow, deltaRow) => {
        const nextRowIndex = cycleBy(startRow, deltaRow, 0, matrix.length - 1);
        const colsInNextRow = matrix[nextRowIndex].length;
        const nextColIndex = clamp(colIndex, 0, colsInNextRow - 1);
        return toCell(matrix, nextRowIndex, nextColIndex);
    };
    const moveHorizontal = (matrix, rowIndex, startCol, deltaCol) => {
        const row = matrix[rowIndex];
        const colsInRow = row.length;
        const newColIndex = clamp(startCol + deltaCol, 0, colsInRow - 1);
        return toCell(matrix, rowIndex, newColIndex);
    };
    const moveVertical = (matrix, colIndex, startRow, deltaRow) => {
        const nextRowIndex = clamp(startRow + deltaRow, 0, matrix.length - 1);
        const colsInNextRow = matrix[nextRowIndex].length;
        const nextColIndex = clamp(colIndex, 0, colsInNextRow - 1);
        return toCell(matrix, nextRowIndex, nextColIndex);
    };
    // return address(Math.floor(index / columns), index % columns);
    const cycleRight = (matrix, startRow, startCol) => cycleHorizontal(matrix, startRow, startCol, +1);
    const cycleLeft = (matrix, startRow, startCol) => cycleHorizontal(matrix, startRow, startCol, -1);
    const cycleUp = (matrix, startRow, startCol) => cycleVertical(matrix, startCol, startRow, -1);
    const cycleDown = (matrix, startRow, startCol) => cycleVertical(matrix, startCol, startRow, +1);
    const moveLeft$1 = (matrix, startRow, startCol) => moveHorizontal(matrix, startRow, startCol, -1);
    const moveRight$1 = (matrix, startRow, startCol) => moveHorizontal(matrix, startRow, startCol, +1);
    const moveUp$1 = (matrix, startRow, startCol) => moveVertical(matrix, startCol, startRow, -1);
    const moveDown$1 = (matrix, startRow, startCol) => moveVertical(matrix, startCol, startRow, +1);

    const schema$v = [
        requiredObjOf('selectors', [
            required$1('row'),
            required$1('cell')
        ]),
        // Used to determine whether pressing right/down at the end cycles back to the start/top
        defaulted('cycles', true),
        defaulted('previousSelector', Optional.none),
        defaulted('execute', defaultExecute)
    ];
    const focusIn$2 = (component, matrixConfig, _state) => {
        const focused = matrixConfig.previousSelector(component).orThunk(() => {
            const selectors = matrixConfig.selectors;
            return descendant(component.element, selectors.cell);
        });
        focused.each((cell) => {
            matrixConfig.focusManager.set(component, cell);
        });
    };
    const execute$1 = (component, simulatedEvent, matrixConfig) => search(component.element).bind((focused) => matrixConfig.execute(component, simulatedEvent, focused));
    const toMatrix = (rows, matrixConfig) => map$2(rows, (row) => descendants(row, matrixConfig.selectors.cell));
    const doMove = (ifCycle, ifMove) => (element, focused, matrixConfig) => {
        const move = matrixConfig.cycles ? ifCycle : ifMove;
        return closest$3(focused, matrixConfig.selectors.row).bind((inRow) => {
            const cellsInRow = descendants(inRow, matrixConfig.selectors.cell);
            return findIndex(cellsInRow, focused).bind((colIndex) => {
                const allRows = descendants(element, matrixConfig.selectors.row);
                return findIndex(allRows, inRow).bind((rowIndex) => {
                    // Now, make the matrix.
                    const matrix = toMatrix(allRows, matrixConfig);
                    return move(matrix, rowIndex, colIndex).map((next) => next.cell);
                });
            });
        });
    };
    const moveLeft = doMove(cycleLeft, moveLeft$1);
    const moveRight = doMove(cycleRight, moveRight$1);
    const moveNorth = doMove(cycleUp, moveUp$1);
    const moveSouth = doMove(cycleDown, moveDown$1);
    const getKeydownRules$2 = constant$1([
        rule(inSet(LEFT), west(moveLeft, moveRight)),
        rule(inSet(RIGHT), east(moveLeft, moveRight)),
        rule(inSet(UP), north(moveNorth)),
        rule(inSet(DOWN), south(moveSouth)),
        rule(inSet(SPACE.concat(ENTER)), execute$1)
    ]);
    const getKeyupRules$2 = constant$1([
        rule(inSet(SPACE), stopEventForFirefox)
    ]);
    var MatrixType = typical(schema$v, NoState.init, getKeydownRules$2, getKeyupRules$2, () => Optional.some(focusIn$2));

    const schema$u = [
        required$1('selector'),
        defaulted('execute', defaultExecute),
        defaulted('moveOnTab', false)
    ];
    const execute = (component, simulatedEvent, menuConfig) => menuConfig.focusManager.get(component).bind((focused) => menuConfig.execute(component, simulatedEvent, focused));
    const focusIn$1 = (component, menuConfig, _state) => {
        // Maybe keep selection if it was there before
        descendant(component.element, menuConfig.selector).each((first) => {
            menuConfig.focusManager.set(component, first);
        });
    };
    const moveUp = (element, focused, info) => horizontal(element, info.selector, focused, -1);
    const moveDown = (element, focused, info) => horizontal(element, info.selector, focused, +1);
    const fireShiftTab = (component, simulatedEvent, menuConfig, menuState) => menuConfig.moveOnTab ? move$1(moveUp)(component, simulatedEvent, menuConfig, menuState) : Optional.none();
    const fireTab = (component, simulatedEvent, menuConfig, menuState) => menuConfig.moveOnTab ? move$1(moveDown)(component, simulatedEvent, menuConfig, menuState) : Optional.none();
    const getKeydownRules$1 = constant$1([
        rule(inSet(UP), move$1(moveUp)),
        rule(inSet(DOWN), move$1(moveDown)),
        rule(and([isShift$1, inSet(TAB)]), fireShiftTab),
        rule(and([isNotShift, inSet(TAB)]), fireTab),
        rule(inSet(ENTER), execute),
        rule(inSet(SPACE), execute)
    ]);
    const getKeyupRules$1 = constant$1([
        rule(inSet(SPACE), stopEventForFirefox)
    ]);
    var MenuType = typical(schema$u, NoState.init, getKeydownRules$1, getKeyupRules$1, () => Optional.some(focusIn$1));

    const schema$t = [
        onKeyboardHandler('onSpace'),
        onKeyboardHandler('onEnter'),
        onKeyboardHandler('onShiftEnter'),
        onKeyboardHandler('onLeft'),
        onKeyboardHandler('onRight'),
        onKeyboardHandler('onTab'),
        onKeyboardHandler('onShiftTab'),
        onKeyboardHandler('onUp'),
        onKeyboardHandler('onDown'),
        onKeyboardHandler('onEscape'),
        defaulted('stopSpaceKeyup', false),
        option$3('focusIn')
    ];
    const getKeydownRules = (component, simulatedEvent, specialInfo) => [
        rule(inSet(SPACE), specialInfo.onSpace),
        rule(and([isNotShift, inSet(ENTER)]), specialInfo.onEnter),
        rule(and([isShift$1, inSet(ENTER)]), specialInfo.onShiftEnter),
        rule(and([isShift$1, inSet(TAB)]), specialInfo.onShiftTab),
        rule(and([isNotShift, inSet(TAB)]), specialInfo.onTab),
        rule(inSet(UP), specialInfo.onUp),
        rule(inSet(DOWN), specialInfo.onDown),
        rule(inSet(LEFT), specialInfo.onLeft),
        rule(inSet(RIGHT), specialInfo.onRight),
        rule(inSet(SPACE), specialInfo.onSpace)
    ];
    const getKeyupRules = (component, simulatedEvent, specialInfo) => [
        ...(specialInfo.stopSpaceKeyup ? [rule(inSet(SPACE), stopEventForFirefox)] : []),
        rule(inSet(ESCAPE), specialInfo.onEscape)
    ];
    var SpecialType = typical(schema$t, NoState.init, getKeydownRules, getKeyupRules, (specialInfo) => specialInfo.focusIn);

    const acyclic = AcyclicType.schema();
    const cyclic = CyclicType.schema();
    const flow = FlowType.schema();
    const flatgrid = FlatgridType.schema();
    const matrix = MatrixType.schema();
    const execution = ExecutionType.schema();
    const menu = MenuType.schema();
    const special = SpecialType.schema();

    var KeyboardBranches = /*#__PURE__*/Object.freeze({
        __proto__: null,
        acyclic: acyclic,
        cyclic: cyclic,
        flow: flow,
        flatgrid: flatgrid,
        matrix: matrix,
        execution: execution,
        menu: menu,
        special: special
    });

    const isFlatgridState = (keyState) => hasNonNullableKey(keyState, 'setGridSize');
    const Keying = createModes({
        branchKey: 'mode',
        branches: KeyboardBranches,
        name: 'keying',
        active: {
            events: (keyingConfig, keyingState) => {
                const handler = keyingConfig.handler;
                return handler.toEvents(keyingConfig, keyingState);
            }
        },
        apis: {
            focusIn: (component, keyConfig, keyState) => {
                // If we have a custom sendFocusIn function, use that.
                // Otherwise, we just trigger focus on the outer element.
                keyConfig.sendFocusIn(keyConfig).fold(() => {
                    component.getSystem().triggerFocus(component.element, component.element);
                }, (sendFocusIn) => {
                    sendFocusIn(component, keyConfig, keyState);
                });
            },
            // These APIs are going to be interesting because they are not
            // available for all keying modes
            setGridSize: (component, keyConfig, keyState, numRows, numColumns) => {
                if (!isFlatgridState(keyState)) {
                    // eslint-disable-next-line no-console
                    console.error('Layout does not support setGridSize');
                }
                else {
                    keyState.setGridSize(numRows, numColumns);
                }
            }
        },
        state: KeyingState
    });

    const premadeTag = generate$6('alloy-premade');
    const premade$1 = (comp) => {
        Object.defineProperty(comp.element.dom, premadeTag, {
            value: comp.uid,
            writable: true
        });
        return wrap(premadeTag, comp);
    };
    const isPremade = (element) => has$2(element.dom, premadeTag);
    const getPremade = (spec) => get$h(spec, premadeTag);
    const makeApi = (f) => markAsSketchApi((component, ...rest) => f(component.getApis(), component, ...rest), f);

    const isConnected = (comp) => comp.getSystem().isConnected();
    const fireDetaching = (component) => {
        emit(component, detachedFromDom());
        const children = component.components();
        each$1(children, fireDetaching);
    };
    const fireAttaching = (component) => {
        const children = component.components();
        each$1(children, fireAttaching);
        emit(component, attachedToDom());
    };
    // Unlike attach, a virtualAttach makes no actual DOM changes.
    // This is because it should only be used in a situation
    // where we are patching an existing element.
    const virtualAttach = (parent, child) => {
        // So we still add it to the world
        parent.getSystem().addToWorld(child);
        // And we fire attaching ONLY if it's already in the DOM
        if (inBody(parent.element)) {
            fireAttaching(child);
        }
    };
    // Unlike detach, a virtualDetach makes no actual DOM changes.
    // This is because it's used in patching circumstances.
    const virtualDetach = (comp) => {
        fireDetaching(comp);
        comp.getSystem().removeFromWorld(comp);
    };
    const attach$1 = (parent, child) => {
        append$2(parent.element, child.element);
    };
    const detachChildren$1 = (component) => {
        // This will not detach the component, but will detach its children and sync at the end.
        each$1(component.components(), (childComp) => remove$7(childComp.element));
        // Clear the component also.
        empty(component.element);
        component.syncComponents();
    };
    const replaceChildren = (component, newSpecs, buildNewChildren) => {
        // Detach all existing children
        const subs = component.components();
        detachChildren$1(component);
        const newChildren = buildNewChildren(newSpecs);
        // Determine which components have been deleted and remove them from the world
        const deleted = difference(subs, newChildren);
        each$1(deleted, (comp) => {
            fireDetaching(comp);
            component.getSystem().removeFromWorld(comp);
        });
        // Add all new components
        each$1(newChildren, (childComp) => {
            // If the component isn't connected, ie is new, then we also need to add it to the world
            if (!isConnected(childComp)) {
                component.getSystem().addToWorld(childComp);
                attach$1(component, childComp);
                if (inBody(component.element)) {
                    fireAttaching(childComp);
                }
            }
            else {
                attach$1(component, childComp);
            }
        });
        component.syncComponents();
    };
    const virtualReplaceChildren = (component, newSpecs, buildNewChildren) => {
        // When replacing we don't want to fire detachedFromDom and attachedToDom again for a premade that has just had its position in the children moved around,
        // so we only detach initially if we aren't a premade. Premades will be detached later, but only if they are no longer in the child list.
        const subs = component.components();
        const existingComps = bind$3(newSpecs, (spec) => getPremade(spec).toArray());
        each$1(subs, (childComp) => {
            if (!contains$2(existingComps, childComp)) {
                virtualDetach(childComp);
            }
        });
        const newChildren = buildNewChildren(newSpecs);
        // Determine which components have been deleted and remove them from the world
        // It's probable the component has already been detached beforehand so only
        // detach what's still attached to the world (i.e removed premades)
        const deleted = difference(subs, newChildren);
        each$1(deleted, (deletedComp) => {
            if (isConnected(deletedComp)) {
                virtualDetach(deletedComp);
            }
        });
        // Add all new components
        each$1(newChildren, (childComp) => {
            // If the component isn't connected, ie is new, then we also need to add it to the world
            if (!isConnected(childComp)) {
                virtualAttach(component, childComp);
            }
        });
        component.syncComponents();
    };

    const attach = (parent, child) => {
        attachWith(parent, child, append$2);
    };
    const attachWith = (parent, child, insertion) => {
        parent.getSystem().addToWorld(child);
        insertion(parent.element, child.element);
        if (inBody(parent.element)) {
            fireAttaching(child);
        }
        parent.syncComponents();
    };
    const doDetach = (component) => {
        fireDetaching(component);
        remove$7(component.element);
        component.getSystem().removeFromWorld(component);
    };
    const detach = (component) => {
        const parent$1 = parent(component.element).bind((p) => component.getSystem().getByDom(p).toOptional());
        doDetach(component);
        parent$1.each((p) => {
            p.syncComponents();
        });
    };
    const detachChildren = (component) => {
        // This will not detach the component, but will detach its children and sync at the end.
        const subs = component.components();
        each$1(subs, doDetach);
        // Clear the component also.
        empty(component.element);
        component.syncComponents();
    };
    const attachSystem = (element, guiSystem) => {
        attachSystemWith(element, guiSystem, append$2);
    };
    const attachSystemAfter = (element, guiSystem) => {
        attachSystemWith(element, guiSystem, after$1);
    };
    const attachSystemWith = (element, guiSystem, inserter) => {
        inserter(element, guiSystem.element);
        const children$1 = children(guiSystem.element);
        each$1(children$1, (child) => {
            guiSystem.getByDom(child).each(fireAttaching);
        });
    };
    const detachSystem = (guiSystem) => {
        const children$1 = children(guiSystem.element);
        each$1(children$1, (child) => {
            guiSystem.getByDom(child).each(fireDetaching);
        });
        remove$7(guiSystem.element);
    };

    const determineObsoleted = (parent, index, oldObsoleted) => {
        // When dealing with premades, the process of building something may have moved existing nodes around, so we see
        // if the child at the index position is still the same. If it isn't, we need to introduce some complex behaviour
        //
        // Example:
        // ```<div><premade></premade><span></span></div>```
        // and then moving the premade inside a blockquote
        // ```<div><blockquote><premade></premade></blockquote><span></span></div>```
        //
        // so when you go to replace the first thing it would think there is only 1 child which would be the span, so in
        // this case we insert a marker to keep the span in the same spot.
        const newObsoleted = child$2(parent, index);
        return newObsoleted.map((newObs) => {
            const elemChanged = oldObsoleted.exists((o) => !eq(o, newObs));
            // Adding a marker prevents the case where a premade is added to something shifting it from where
            // it was. That in turn un-synced all trailing children and made it so they couldn't be patched.
            if (elemChanged) {
                const oldTag = oldObsoleted.map(name$3).getOr('span');
                const marker = SugarElement.fromTag(oldTag);
                before$1(newObs, marker);
                return marker;
            }
            else {
                return newObs;
            }
        });
    };
    const ensureInDom = (parent, child, obsoleted) => {
        obsoleted.fold(
        // There is nothing here, so just append to the parent
        () => append$2(parent, child), (obs) => {
            if (!eq(obs, child)) {
                // This situation occurs when the DOM element that has been patched when building it is no
                // longer the one that we need to replace. This is probably caused by premades.
                before$1(obs, child);
                remove$7(obs);
            }
        });
    };
    const patchChildrenWith = (parent, nu, f) => {
        const builtChildren = map$2(nu, f);
        // Need to regather the children in case some of the previous children have moved
        // to an earlier index. So this just prunes any leftover children in the dom.
        const currentChildren = children(parent);
        each$1(currentChildren.slice(builtChildren.length), remove$7);
        return builtChildren;
    };
    const patchSpecChild = (parent, index, spec, build) => {
        // Before building anything, this is the DOM element we are going to try to use.
        const oldObsoleted = child$2(parent, index);
        const childComp = build(spec, oldObsoleted);
        const obsoleted = determineObsoleted(parent, index, oldObsoleted);
        ensureInDom(parent, childComp.element, obsoleted);
        return childComp;
    };
    const patchSpecChildren = (parent, specs, build) => patchChildrenWith(parent, specs, (spec, index) => patchSpecChild(parent, index, spec, build));
    const patchDomChildren = (parent, nodes) => patchChildrenWith(parent, nodes, (node, index) => {
        const optObsoleted = child$2(parent, index);
        ensureInDom(parent, node, optObsoleted);
        return node;
    });

    const preserve = (f, container) => {
        const dos = getRootNode(container);
        const refocus = active$1(dos).bind((focused) => {
            const hasFocus = (elem) => eq(focused, elem);
            return hasFocus(container) ? Optional.some(container) : descendant$1(container, hasFocus);
        });
        const result = f(container);
        // If there is a focussed element, the F function may cause focus to be lost (such as by hiding elements). Restore it afterwards.
        refocus.each((oldFocus) => {
            active$1(dos).filter((newFocus) => eq(newFocus, oldFocus)).fold(() => {
                // Only refocus if the focus has changed, otherwise we break IE
                focus$4(oldFocus);
            }, noop);
        });
        return result;
    };

    const withoutReuse = (parent, data) => {
        preserve(() => {
            replaceChildren(parent, data, () => map$2(data, parent.getSystem().build));
        }, parent.element);
    };
    const withReuse = (parent, data) => {
        // Note: We shouldn't need AriaPreserve since we're trying to keep the existing elements,
        // but let's just do it for now just to be safe.
        preserve(() => {
            virtualReplaceChildren(parent, data, () => {
                // Build the new children
                return patchSpecChildren(parent.element, data, parent.getSystem().buildOrPatch);
            });
        }, parent.element);
    };

    const virtualReplace = (component, replacee, replaceeIndex, childSpec) => {
        virtualDetach(replacee);
        const child = patchSpecChild(component.element, replaceeIndex, childSpec, component.getSystem().buildOrPatch);
        virtualAttach(component, child);
        component.syncComponents();
    };
    const insert = (component, insertion, childSpec) => {
        const child = component.getSystem().build(childSpec);
        attachWith(component, child, insertion);
    };
    const replace = (component, replacee, replaceeIndex, childSpec) => {
        detach(replacee);
        insert(component, (p, c) => appendAt(p, c, replaceeIndex), childSpec);
    };
    const set$3 = (component, replaceConfig, replaceState, data) => {
        const replacer = replaceConfig.reuseDom ? withReuse : withoutReuse;
        return replacer(component, data);
    };
    const append = (component, replaceConfig, replaceState, appendee) => {
        insert(component, append$2, appendee);
    };
    const prepend = (component, replaceConfig, replaceState, prependee) => {
        insert(component, prepend$1, prependee);
    };
    // NOTE: Removee is going to be a component, not a spec.
    const remove = (component, replaceConfig, replaceState, removee) => {
        const children = contents(component);
        const foundChild = find$5(children, (child) => eq(removee.element, child.element));
        foundChild.each(detach);
    };
    // TODO: Rename
    const contents = (component, _replaceConfig) => component.components();
    const replaceAt = (component, replaceConfig, replaceState, replaceeIndex, replacer) => {
        const children = contents(component);
        return Optional.from(children[replaceeIndex]).map((replacee) => {
            replacer.fold(() => detach(replacee), (r) => {
                const replacer = replaceConfig.reuseDom ? virtualReplace : replace;
                replacer(component, replacee, replaceeIndex, r);
            });
            return replacee;
        });
    };
    const replaceBy = (component, replaceConfig, replaceState, replaceePred, replacer) => {
        const children = contents(component);
        return findIndex$1(children, replaceePred).bind((replaceeIndex) => replaceAt(component, replaceConfig, replaceState, replaceeIndex, replacer));
    };

    var ReplaceApis = /*#__PURE__*/Object.freeze({
        __proto__: null,
        append: append,
        prepend: prepend,
        remove: remove,
        replaceAt: replaceAt,
        replaceBy: replaceBy,
        set: set$3,
        contents: contents
    });

    const Replacing = create$3({
        fields: [
            defaultedBoolean('reuseDom', true)
        ],
        name: 'replacing',
        apis: ReplaceApis
    });

    // The purpose of this check is to ensure that a simulated focus call is not going
    // to recurse infinitely. Essentially, if the originator of the focus call is the same
    // as the element receiving it, and it wasn't its own target, then stop the focus call
    // and log a warning.
    const isRecursive = (component, originator, target) => eq(originator, component.element) && !eq(originator, target);
    const events$g = derive$2([
        can(focus$3(), (component, simulatedEvent) => {
            // originator may not always be there. Will need to check this.
            const event = simulatedEvent.event;
            const originator = event.originator;
            const target = event.target;
            if (isRecursive(component, originator, target)) {
                // eslint-disable-next-line no-console
                console.warn(focus$3() + ' did not get interpreted by the desired target. ' +
                    '\nOriginator: ' + element(originator) +
                    '\nTarget: ' + element(target) +
                    '\nCheck the ' + focus$3() + ' event handlers');
                return false;
            }
            else {
                return true;
            }
        })
    ]);

    var DefaultEvents = /*#__PURE__*/Object.freeze({
        __proto__: null,
        events: events$g
    });

    const prefix$1 = constant$1('alloy-id-');
    const idAttr$1 = constant$1('data-alloy-id');

    const prefix = prefix$1();
    const idAttr = idAttr$1();
    const write = (label, elem) => {
        const id = generate$6(prefix + label);
        writeOnly(elem, id);
        return id;
    };
    const writeOnly = (elem, uid) => {
        Object.defineProperty(elem.dom, idAttr, {
            value: uid,
            writable: true
        });
    };
    const read = (elem) => {
        const id = isElement$1(elem) ? elem.dom[idAttr] : null;
        return Optional.from(id);
    };
    const generate$4 = (prefix) => generate$6(prefix);

    const make$7 = identity;

    const NoContextApi = (getComp) => {
        const getMessage = (event) => `The component must be in a context to execute: ${event}` +
            (getComp ? '\n' + element(getComp().element) + ' is not in context.' : '');
        const fail = (event) => () => {
            throw new Error(getMessage(event));
        };
        const warn = (event) => () => {
            // eslint-disable-next-line no-console
            console.warn(getMessage(event));
        };
        return {
            debugInfo: constant$1('fake'),
            triggerEvent: warn('triggerEvent'),
            triggerFocus: warn('triggerFocus'),
            triggerEscape: warn('triggerEscape'),
            broadcast: warn('broadcast'),
            broadcastOn: warn('broadcastOn'),
            broadcastEvent: warn('broadcastEvent'),
            build: fail('build'),
            buildOrPatch: fail('buildOrPatch'),
            addToWorld: fail('addToWorld'),
            removeFromWorld: fail('removeFromWorld'),
            addToGui: fail('addToGui'),
            removeFromGui: fail('removeFromGui'),
            getByUid: fail('getByUid'),
            getByDom: fail('getByDom'),
            isConnected: never
        };
    };
    const singleton = NoContextApi();

    const generateFrom$1 = (spec, all) => {
        /*
         * This takes a basic record of configured behaviours, defaults their state
         * and ensures that all the behaviours were valid. Will need to document
         * this entire process. Let's see where this is used.
         */
        const schema = map$2(all, (a) => 
        // Optional here probably just due to ForeignGui listing everything it supports. Can most likely
        // change it to strict once I fix the other errors.
        optionObjOf(a.name(), [
            required$1('config'),
            defaulted('state', NoState)
        ]));
        const validated = asRaw('component.behaviours', objOf(schema), spec.behaviours).fold((errInfo) => {
            throw new Error(formatError(errInfo) + '\nComplete spec:\n' +
                JSON.stringify(spec, null, 2));
        }, identity);
        return {
            list: all,
            data: map$1(validated, (optBlobThunk) => {
                const output = optBlobThunk.map((blob) => ({
                    config: blob.config,
                    state: blob.state.init(blob.config)
                }));
                return constant$1(output);
            })
        };
    };
    const getBehaviours$3 = (bData) => bData.list;
    const getData$2 = (bData) => bData.data;

    const byInnerKey = (data, tuple) => {
        const r = {};
        each(data, (detail, key) => {
            each(detail, (value, indexKey) => {
                const chain = get$h(r, indexKey).getOr([]);
                r[indexKey] = chain.concat([
                    tuple(key, value)
                ]);
            });
        });
        return r;
    };

    // Based on all the behaviour exhibits, and the original dom modification, identify
    // the overall combined dom modification that needs to occur
    const combine$1 = (info, baseMod, behaviours, base) => {
        // Clone the object so we can change it.
        const modsByBehaviour = { ...baseMod };
        each$1(behaviours, (behaviour) => {
            modsByBehaviour[behaviour.name()] = behaviour.exhibit(info, base);
        });
        // byAspect format: { classes: [ { name: Toggling, modification: [ 'selected' ] } ] }
        const byAspect = byInnerKey(modsByBehaviour, (name, modification) => ({ name, modification }));
        const combineObjects = (objects) => foldr(objects, (b, a) => ({ ...a.modification, ...b }), {});
        const combinedClasses = foldr(byAspect.classes, (b, a) => a.modification.concat(b), []);
        const combinedAttributes = combineObjects(byAspect.attributes);
        const combinedStyles = combineObjects(byAspect.styles);
        return nu$2({
            classes: combinedClasses,
            attributes: combinedAttributes,
            styles: combinedStyles
        });
    };

    const sortKeys = (label, keyName, array, order) => {
        try {
            const sorted = sort(array, (a, b) => {
                const aKey = a[keyName];
                const bKey = b[keyName];
                const aIndex = order.indexOf(aKey);
                const bIndex = order.indexOf(bKey);
                if (aIndex === -1) {
                    throw new Error('The ordering for ' + label + ' does not have an entry for ' + aKey +
                        '.\nOrder specified: ' + JSON.stringify(order, null, 2));
                }
                if (bIndex === -1) {
                    throw new Error('The ordering for ' + label + ' does not have an entry for ' + bKey +
                        '.\nOrder specified: ' + JSON.stringify(order, null, 2));
                }
                if (aIndex < bIndex) {
                    return -1;
                }
                else if (bIndex < aIndex) {
                    return 1;
                }
                else {
                    return 0;
                }
            });
            return Result.value(sorted);
        }
        catch (err) {
            return Result.error([err]);
        }
    };

    const uncurried = (handler, purpose) => ({
        handler,
        purpose
    });
    const curried = (handler, purpose) => ({
        cHandler: handler,
        purpose
    });
    const curryArgs = (descHandler, extraArgs) => curried(curry.apply(undefined, [descHandler.handler].concat(extraArgs)), descHandler.purpose);
    const getCurried = (descHandler) => descHandler.cHandler;

    const behaviourTuple = (name, handler) => ({
        name,
        handler
    });
    const nameToHandlers = (behaviours, info) => {
        const r = {};
        each$1(behaviours, (behaviour) => {
            r[behaviour.name()] = behaviour.handlers(info);
        });
        return r;
    };
    const groupByEvents = (info, behaviours, base) => {
        const behaviourEvents = {
            ...base,
            ...nameToHandlers(behaviours, info)
        };
        // Now, with all of these events, we need to index by event name
        return byInnerKey(behaviourEvents, behaviourTuple);
    };
    const combine = (info, eventOrder, behaviours, base) => {
        const byEventName = groupByEvents(info, behaviours, base);
        return combineGroups(byEventName, eventOrder);
    };
    const assemble = (rawHandler) => {
        const handler = read$1(rawHandler);
        return (component, simulatedEvent, ...rest) => {
            const args = [component, simulatedEvent].concat(rest);
            if (handler.abort.apply(undefined, args)) {
                simulatedEvent.stop();
            }
            else if (handler.can.apply(undefined, args)) {
                handler.run.apply(undefined, args);
            }
        };
    };
    const missingOrderError = (eventName, tuples) => Result.error([
        'The event (' + eventName + ') has more than one behaviour that listens to it.\nWhen this occurs, you must ' +
            'specify an event ordering for the behaviours in your spec (e.g. [ "listing", "toggling" ]).\nThe behaviours that ' +
            'can trigger it are: ' + JSON.stringify(map$2(tuples, (c) => c.name), null, 2)
    ]);
    const fuse = (tuples, eventOrder, eventName) => {
        // ASSUMPTION: tuples.length will never be 0, because it wouldn't have an entry if it was 0
        const order = eventOrder[eventName];
        if (!order) {
            return missingOrderError(eventName, tuples);
        }
        else {
            return sortKeys('Event: ' + eventName, 'name', tuples, order).map((sortedTuples) => {
                const handlers = map$2(sortedTuples, (tuple) => tuple.handler);
                return fuse$1(handlers);
            });
        }
    };
    const combineGroups = (byEventName, eventOrder) => {
        const r = mapToArray(byEventName, (tuples, eventName) => {
            const combined = tuples.length === 1 ? Result.value(tuples[0].handler) : fuse(tuples, eventOrder, eventName);
            return combined.map((handler) => {
                const assembled = assemble(handler);
                const purpose = tuples.length > 1 ? filter$2(eventOrder[eventName], (o) => exists(tuples, (t) => t.name === o)).join(' > ') : tuples[0].name;
                return wrap(eventName, uncurried(assembled, purpose));
            });
        });
        return consolidate(r, {});
    };

    const baseBehaviour = 'alloy.base.behaviour';
    const schema$s = objOf([
        field$1('dom', 'dom', required$2(), objOf([
            // Note, no children.
            required$1('tag'),
            defaulted('styles', {}),
            defaulted('classes', []),
            defaulted('attributes', {}),
            option$3('value'),
            option$3('innerHtml')
        ])),
        required$1('components'),
        required$1('uid'),
        defaulted('events', {}),
        defaulted('apis', {}),
        // Use mergeWith in the future when pre-built behaviours conflict
        field$1('eventOrder', 'eventOrder', mergeWith({
            // Note, not using constant behaviour names to avoid code size of unused behaviours
            [execute$5()]: ['disabling', baseBehaviour, 'toggling', 'typeaheadevents'],
            [focus$3()]: [baseBehaviour, 'focusing', 'keying'],
            [systemInit()]: [baseBehaviour, 'disabling', 'toggling', 'representing', 'tooltipping'],
            [input()]: [baseBehaviour, 'representing', 'streaming', 'invalidating'],
            [detachedFromDom()]: [baseBehaviour, 'representing', 'item-events', 'toolbar-button-events', 'tooltipping'],
            [mousedown()]: ['focusing', baseBehaviour, 'item-type-events'],
            [touchstart()]: ['focusing', baseBehaviour, 'item-type-events'],
            [mouseover()]: ['item-type-events', 'tooltipping'],
            [receive()]: ['receiving', 'reflecting', 'tooltipping']
        }), anyValue()),
        option$3('domModification')
    ]);
    const toInfo = (spec) => asRaw('custom.definition', schema$s, spec);
    const toDefinition = (detail) => 
    // EFFICIENCY: Consider not merging here.
    ({
        ...detail.dom,
        uid: detail.uid,
        domChildren: map$2(detail.components, (comp) => comp.element)
    });
    const toModification = (detail) => detail.domModification.fold(() => nu$2({}), nu$2);
    const toEvents = (info) => info.events;

    const diffKeyValueSet = (newObj, oldObj) => {
        const newKeys = keys(newObj);
        const oldKeys = keys(oldObj);
        const toRemove = difference(oldKeys, newKeys);
        const toSet = bifilter(newObj, (v, k) => {
            return !has$2(oldObj, k) || v !== oldObj[k];
        }).t;
        return { toRemove, toSet };
    };
    const reconcileToDom = (definition, obsoleted) => {
        const { class: clazz, style, ...existingAttributes } = clone$2(obsoleted);
        const { toSet: attrsToSet, toRemove: attrsToRemove } = diffKeyValueSet(definition.attributes, existingAttributes);
        const updateAttrs = () => {
            each$1(attrsToRemove, (a) => remove$8(obsoleted, a));
            setAll$1(obsoleted, attrsToSet);
        };
        const existingStyles = getAllRaw(obsoleted);
        const { toSet: stylesToSet, toRemove: stylesToRemove } = diffKeyValueSet(definition.styles, existingStyles);
        const updateStyles = () => {
            each$1(stylesToRemove, (s) => remove$6(obsoleted, s));
            setAll(obsoleted, stylesToSet);
        };
        const existingClasses = get$7(obsoleted);
        const classesToRemove = difference(existingClasses, definition.classes);
        const classesToAdd = difference(definition.classes, existingClasses);
        const updateClasses = () => {
            add$1(obsoleted, classesToAdd);
            remove$2(obsoleted, classesToRemove);
        };
        const updateHtml = (html) => {
            set$8(obsoleted, html);
        };
        const updateChildren = () => {
            const children = definition.domChildren;
            patchDomChildren(obsoleted, children);
        };
        const updateValue = () => {
            const valueElement = obsoleted;
            const value = definition.value.getOrUndefined();
            if (value !== get$5(valueElement)) {
                // TINY-8736: Value.set throws an error in case the value is undefined
                set$4(valueElement, value ?? '');
            }
        };
        updateAttrs();
        updateClasses();
        updateStyles();
        // Patching can only support one form of children, so we only update the html or the children, but never both
        definition.innerHtml.fold(updateChildren, updateHtml);
        updateValue();
        return obsoleted;
    };

    const introduceToDom = (definition) => {
        const subject = SugarElement.fromTag(definition.tag);
        setAll$1(subject, definition.attributes);
        add$1(subject, definition.classes);
        setAll(subject, definition.styles);
        // Remember: Order of innerHtml vs children is important.
        definition.innerHtml.each((html) => set$8(subject, html));
        // Children are already elements.
        const children = definition.domChildren;
        append$1(subject, children);
        definition.value.each((value) => {
            set$4(subject, value);
        });
        return subject;
    };
    const attemptPatch = (definition, obsoleted) => {
        try {
            const e = reconcileToDom(definition, obsoleted);
            return Optional.some(e);
        }
        catch {
            return Optional.none();
        }
    };
    // If a component has both innerHtml and children then we can't patch it
    const hasMixedChildren = (definition) => definition.innerHtml.isSome() && definition.domChildren.length > 0;
    const renderToDom = (definition, optObsoleted) => {
        // If the current tag doesn't match, let's not try to add anything further down the tree.
        // If it does match though and we don't have mixed children then attempt to patch attributes etc...
        const canBePatched = (candidate) => name$3(candidate) === definition.tag && !hasMixedChildren(definition) && !isPremade(candidate);
        const elem = optObsoleted
            .filter(canBePatched)
            .bind((obsoleted) => attemptPatch(definition, obsoleted))
            .getOrThunk(() => introduceToDom(definition));
        writeOnly(elem, definition.uid);
        return elem;
    };

    // This goes through the list of behaviours defined for a particular spec (removing anything
    // that has been revoked), and returns the BehaviourType (e.g. Sliding)
    const getBehaviours$2 = (spec) => {
        const behaviours = get$h(spec, 'behaviours').getOr({});
        return bind$3(keys(behaviours), (name) => {
            const behaviour = behaviours[name];
            return isNonNullable(behaviour) ? [behaviour.me] : [];
        });
    };
    const generateFrom = (spec, all) => generateFrom$1(spec, all);
    const generate$3 = (spec) => {
        const all = getBehaviours$2(spec);
        return generateFrom(spec, all);
    };

    // This is probably far too complicated. I think DomModification is probably
    // questionable as a concept. Maybe it should be deprecated.
    const getDomDefinition = (info, bList, bData) => {
        // Get the current DOM definition from the spec
        const definition = toDefinition(info);
        // Get the current DOM modification definition from the spec
        const infoModification = toModification(info);
        // Treat the DOM modification like it came from a behaviour
        const baseModification = {
            'alloy.base.modification': infoModification
        };
        // Combine the modifications from any defined behaviours
        const modification = bList.length > 0 ? combine$1(bData, baseModification, bList, definition) : infoModification;
        // Transform the DOM definition with the combined dom modifications to make a new DOM definition
        return merge(definition, modification);
    };
    const getEvents = (info, bList, bData) => {
        const baseEvents = {
            'alloy.base.behaviour': toEvents(info)
        };
        return combine(bData, info.eventOrder, bList, baseEvents).getOrDie();
    };
    const build$2 = (spec, obsoleted) => {
        const getMe = () => me;
        const systemApi = Cell(singleton);
        const info = getOrDie(toInfo(spec));
        const bBlob = generate$3(spec);
        const bList = getBehaviours$3(bBlob);
        const bData = getData$2(bBlob);
        const modDefinition = getDomDefinition(info, bList, bData);
        const item = renderToDom(modDefinition, obsoleted);
        const events = getEvents(info, bList, bData);
        const subcomponents = Cell(info.components);
        const connect = (newApi) => {
            systemApi.set(newApi);
        };
        const disconnect = () => {
            systemApi.set(NoContextApi(getMe));
        };
        const syncComponents = () => {
            // Update the component list with the current children
            const children$1 = children(item);
            // INVESTIGATE: Not sure about how to handle text nodes here.
            const subs = bind$3(children$1, (child) => systemApi.get().getByDom(child).fold(() => [], pure$2));
            subcomponents.set(subs);
        };
        // TYPIFY (any here is for the info.apis() pathway)
        const config = (behaviour) => {
            const b = bData;
            const f = isFunction(b[behaviour.name()]) ? b[behaviour.name()] : () => {
                throw new Error('Could not find ' + behaviour.name() + ' in ' + JSON.stringify(spec, null, 2));
            };
            return f();
        };
        const hasConfigured = (behaviour) => isFunction(bData[behaviour.name()]);
        const getApis = () => info.apis;
        const readState = (behaviourName) => bData[behaviourName]().map((b) => b.state.readState()).getOr('not enabled');
        const me = {
            uid: spec.uid,
            getSystem: systemApi.get,
            config,
            hasConfigured,
            spec,
            readState,
            getApis,
            connect,
            disconnect,
            element: item,
            syncComponents,
            components: subcomponents.get,
            events
        };
        return me;
    };

    const buildSubcomponents = (spec, obsoleted) => {
        const components = get$h(spec, 'components').getOr([]);
        return obsoleted.fold(() => map$2(components, build$1), (obs) => map$2(components, (c, i) => {
            return buildOrPatch(c, child$2(obs, i));
        }));
    };
    const buildFromSpec = (userSpec, obsoleted) => {
        const { events: specEvents, ...spec } = make$7(userSpec);
        // Build the subcomponents. A spec hierarchy is built from the bottom up.
        // obsoleted is used to define which element we are attempting to replace
        // so that it might be used to patch the DOM instead of recreate it.
        const components = buildSubcomponents(spec, obsoleted);
        const completeSpec = {
            ...spec,
            events: { ...DefaultEvents, ...specEvents },
            components
        };
        return Result.value(
        // Note, this isn't a spec any more, because it has built children
        build$2(completeSpec, obsoleted));
    };
    const text$2 = (textContent) => {
        const element = SugarElement.fromText(textContent);
        return external({
            element
        });
    };
    const external = (spec) => {
        const extSpec = asRawOrDie$1('external.component', objOfOnly([
            required$1('element'),
            option$3('uid')
        ]), spec);
        const systemApi = Cell(NoContextApi());
        const connect = (newApi) => {
            systemApi.set(newApi);
        };
        const disconnect = () => {
            systemApi.set(NoContextApi(() => me));
        };
        const uid = extSpec.uid.getOrThunk(() => generate$4('external'));
        writeOnly(extSpec.element, uid);
        const me = {
            uid,
            getSystem: systemApi.get,
            config: Optional.none,
            hasConfigured: never,
            connect,
            disconnect,
            getApis: () => ({}),
            element: extSpec.element,
            spec,
            readState: constant$1('No state'),
            syncComponents: noop,
            components: constant$1([]),
            events: {}
        };
        return premade$1(me);
    };
    // We experimented with just having a counter for efficiency, but that fails for situations
    // where an external JS file is using alloy, and is contained within another
    // alloy root container. The ids can conflict, because the counters do not
    // know about each other (being parts of separate scripts).
    //
    // There are other solutions than this ... not sure if they are going to have better performance, though
    const uids = generate$4;
    const isSketchSpec$1 = (spec) => has$2(spec, 'uid');
    // INVESTIGATE: A better way to provide 'meta-specs'
    const buildOrPatch = (spec, obsoleted) => getPremade(spec).getOrThunk(() => {
        // EFFICIENCY: Consider not merging here, and passing uid through separately
        const userSpecWithUid = isSketchSpec$1(spec) ? spec : {
            uid: uids(''),
            ...spec
        };
        return buildFromSpec(userSpecWithUid, obsoleted).getOrDie();
    });
    const build$1 = (spec) => buildOrPatch(spec, Optional.none());
    const premade = premade$1;

    // Mark this component as busy, or blocked.
    const block = (component, config, state, 
    // This works in conjunction with the 'getRoot' function in the config. To
    // attach a blocker component to the dom, ensure that 'getRoot' returns a
    // component, and this function returns the specification of the component to
    // attach.
    getBusySpec) => {
        set$9(component.element, 'aria-busy', true);
        const root = config.getRoot(component).getOr(component);
        const blockerBehaviours = derive$1([
            // Trap the "Tab" key and don't let it escape.
            Keying.config({
                mode: 'special',
                onTab: () => Optional.some(true),
                onShiftTab: () => Optional.some(true)
            }),
            Focusing.config({})
        ]);
        const blockSpec = getBusySpec(root, blockerBehaviours);
        const blocker = root.getSystem().build(blockSpec);
        Replacing.append(root, premade(blocker));
        if (blocker.hasConfigured(Keying) && config.focus) {
            Keying.focusIn(blocker);
        }
        if (!state.isBlocked()) {
            config.onBlock(component);
        }
        state.blockWith(() => Replacing.remove(root, blocker));
    };
    // Mark this component as unblocked, or not busy. This is a noop on a component
    // that isn't blocked.
    const unblock = (component, config, state) => {
        remove$8(component.element, 'aria-busy');
        if (state.isBlocked()) {
            config.onUnblock(component);
        }
        state.clear();
    };
    const isBlocked = (component, blockingConfig, blockingState) => blockingState.isBlocked();

    var BlockingApis = /*#__PURE__*/Object.freeze({
        __proto__: null,
        block: block,
        unblock: unblock,
        isBlocked: isBlocked
    });

    var BlockingSchema = [
        // The blocking behaviour places a blocking element over the DOM while the
        // component is in the blocked state. If a function is provided here that
        // returns Some, then the blocking element will be added as a child of the
        // element returned. Otherwise, it will be added as a child of the main
        // component.
        defaultedFunction('getRoot', Optional.none),
        // This boolean, if provided, will specify whether the blocking element is
        // focused when the component is first blocked
        defaultedBoolean('focus', true),
        // This function, if provided, will be called any time the component is
        // blocked (unless it was already blocked).
        onHandler('onBlock'),
        // This function, if provided, will be called any time the component is
        // unblocked (unless it was already unblocked).
        onHandler('onUnblock')
    ];

    const init$f = () => {
        const blocker = destroyable();
        const blockWith = (destroy) => {
            blocker.set({ destroy });
        };
        return nu$4({
            readState: blocker.isSet,
            blockWith,
            clear: blocker.clear,
            isBlocked: blocker.isSet
        });
    };

    var BlockingState = /*#__PURE__*/Object.freeze({
        __proto__: null,
        init: init$f
    });

    // Mark a component as able to be "Blocked" or able to enter a busy state. See
    // BlockingSchema and BlockingApis for more details on how to configure this.
    const Blocking = create$3({
        fields: BlockingSchema,
        name: 'blocking',
        apis: BlockingApis,
        state: BlockingState
    });

    const getCurrent = (component, composeConfig, _composeState) => composeConfig.find(component);

    var ComposeApis = /*#__PURE__*/Object.freeze({
        __proto__: null,
        getCurrent: getCurrent
    });

    const ComposeSchema = [
        required$1('find')
    ];

    const Composing = create$3({
        fields: ComposeSchema,
        name: 'composing',
        apis: ComposeApis
    });

    const getCoupled = (component, coupleConfig, coupleState, name) => coupleState.getOrCreate(component, coupleConfig, name);
    const getExistingCoupled = (component, coupleConfig, coupleState, name) => coupleState.getExisting(component, coupleConfig, name);

    var CouplingApis = /*#__PURE__*/Object.freeze({
        __proto__: null,
        getCoupled: getCoupled,
        getExistingCoupled: getExistingCoupled
    });

    var CouplingSchema = [
        requiredOf('others', setOf(Result.value, anyValue()))
    ];

    // Unfortunately, the Coupling APIs currently throw errors when the coupled name
    // is not recognised. This is because if the wrong name is used, it is a
    // non-recoverable error, and the developer should be notified. However, there are
    // better ways to do this: (removing this API and only returning Optionals/Results)
    const init$e = () => {
        const coupled = {};
        const lookupCoupled = (coupleConfig, coupledName) => {
            const available = keys(coupleConfig.others);
            if (available.length === 0) {
                throw new Error('Cannot find any known coupled components');
            }
            else {
                return get$h(coupled, coupledName);
            }
        };
        const getOrCreate = (component, coupleConfig, name) => {
            return lookupCoupled(coupleConfig, name).getOrThunk(() => {
                // TODO: TINY-9014 Likely type error. coupleConfig.others[key] is
                // `() => ((comp: AlloyComponent) => AlloySpec)`,
                // but builder is being treated as a `(comp: AlloyComponent) => AlloySpec`
                const builder = get$h(coupleConfig.others, name).getOrDie('No information found for coupled component: ' + name);
                const spec = builder(component);
                const built = component.getSystem().build(spec);
                coupled[name] = built;
                return built;
            });
        };
        const getExisting = (component, coupleConfig, name) => {
            return lookupCoupled(coupleConfig, name).orThunk(() => {
                // Validate we recognise this coupled component's name.
                get$h(coupleConfig.others, name).getOrDie('No information found for coupled component: ' + name);
                // It's a valid name, so return None, because it hasn't been built yet.
                return Optional.none();
            });
        };
        const readState = constant$1({});
        return nu$4({
            readState,
            getExisting,
            getOrCreate
        });
    };

    var CouplingState = /*#__PURE__*/Object.freeze({
        __proto__: null,
        init: init$e
    });

    const Coupling = create$3({
        fields: CouplingSchema,
        name: 'coupling',
        apis: CouplingApis,
        state: CouplingState
    });

    // Just use "disabled" attribute for these, not "aria-disabled"
    const nativeDisabled = [
        'input',
        'button',
        'textarea',
        'select'
    ];
    const onLoad$5 = (component, disableConfig, disableState) => {
        const f = disableConfig.disabled() ? disable : enable;
        f(component, disableConfig);
    };
    const hasNative = (component, config) => config.useNative === true && contains$2(nativeDisabled, name$3(component.element));
    const nativeIsDisabled = (component) => has$1(component.element, 'disabled');
    const nativeDisable = (component) => {
        set$9(component.element, 'disabled', 'disabled');
    };
    const nativeEnable = (component) => {
        remove$8(component.element, 'disabled');
    };
    const ariaIsDisabled = (component) => get$g(component.element, 'aria-disabled') === 'true';
    const ariaDisable = (component) => {
        set$9(component.element, 'aria-disabled', 'true');
    };
    const ariaEnable = (component) => {
        set$9(component.element, 'aria-disabled', 'false');
    };
    const disable = (component, disableConfig, _disableState) => {
        disableConfig.disableClass.each((disableClass) => {
            add$2(component.element, disableClass);
        });
        const f = hasNative(component, disableConfig) ? nativeDisable : ariaDisable;
        f(component);
        disableConfig.onDisabled(component);
    };
    const enable = (component, disableConfig, _disableState) => {
        disableConfig.disableClass.each((disableClass) => {
            remove$3(component.element, disableClass);
        });
        const f = hasNative(component, disableConfig) ? nativeEnable : ariaEnable;
        f(component);
        disableConfig.onEnabled(component);
    };
    const isDisabled$1 = (component, disableConfig) => hasNative(component, disableConfig) ? nativeIsDisabled(component) : ariaIsDisabled(component);
    const set$2 = (component, disableConfig, disableState, disabled) => {
        const f = disabled ? disable : enable;
        f(component, disableConfig);
    };

    var DisableApis = /*#__PURE__*/Object.freeze({
        __proto__: null,
        enable: enable,
        disable: disable,
        isDisabled: isDisabled$1,
        onLoad: onLoad$5,
        set: set$2
    });

    const exhibit$5 = (base, disableConfig) => nu$2({
        // Do not add the attribute yet, because it will depend on the node name
        // if we use "aria-disabled" or just "disabled"
        classes: disableConfig.disabled() ? disableConfig.disableClass.toArray() : []
    });
    const events$f = (disableConfig, disableState) => derive$2([
        abort(execute$5(), (component, _simulatedEvent) => isDisabled$1(component, disableConfig)),
        loadEvent(disableConfig, disableState, onLoad$5)
    ]);

    var ActiveDisable = /*#__PURE__*/Object.freeze({
        __proto__: null,
        exhibit: exhibit$5,
        events: events$f
    });

    var DisableSchema = [
        defaultedFunction('disabled', never),
        defaulted('useNative', true),
        option$3('disableClass'),
        onHandler('onDisabled'),
        onHandler('onEnabled')
    ];

    const Disabling = create$3({
        fields: DisableSchema,
        name: 'disabling',
        active: ActiveDisable,
        apis: DisableApis
    });

    const NuPositionCss = (position, left, top, right, bottom) => {
        const toPx = (num) => num + 'px';
        return {
            position,
            left: left.map(toPx),
            top: top.map(toPx),
            right: right.map(toPx),
            bottom: bottom.map(toPx)
        };
    };
    const toOptions = (position) => ({
        ...position,
        position: Optional.some(position.position)
    });
    const applyPositionCss = (element, position) => {
        setOptions(element, toOptions(position));
    };

    const appear = (component, contextualInfo) => {
        const elem = component.element;
        add$2(elem, contextualInfo.transitionClass);
        remove$3(elem, contextualInfo.fadeOutClass);
        add$2(elem, contextualInfo.fadeInClass);
        contextualInfo.onShow(component);
    };
    const disappear = (component, contextualInfo) => {
        const elem = component.element;
        add$2(elem, contextualInfo.transitionClass);
        remove$3(elem, contextualInfo.fadeInClass);
        add$2(elem, contextualInfo.fadeOutClass);
        contextualInfo.onHide(component);
    };
    const isPartiallyVisible = (box, bounds) => box.y < bounds.bottom && box.bottom > bounds.y;
    const isTopCompletelyVisible = (box, bounds) => box.y >= bounds.y;
    const isBottomCompletelyVisible = (box, bounds) => box.bottom <= bounds.bottom;
    const forceTopPosition = (winBox, leftX, viewport) => ({
        location: 'top',
        leftX,
        topY: viewport.bounds.y - winBox.y
    });
    const forceBottomPosition = (winBox, leftX, viewport) => ({
        location: 'bottom',
        leftX,
        bottomY: winBox.bottom - viewport.bounds.bottom
    });
    const getDockedLeftPosition = (bounds) => {
        // Essentially, we are just getting the bounding client rect left here,
        // because winBox.x will be the scroll value.
        return bounds.box.x - bounds.win.x;
    };
    const tryDockingPosition = (modes, bounds, viewport) => {
        const winBox = bounds.win;
        const box = bounds.box;
        const leftX = getDockedLeftPosition(bounds);
        return findMap(modes, (mode) => {
            switch (mode) {
                case 'bottom':
                    return !isBottomCompletelyVisible(box, viewport.bounds) ? Optional.some(forceBottomPosition(winBox, leftX, viewport)) : Optional.none();
                case 'top':
                    return !isTopCompletelyVisible(box, viewport.bounds) ? Optional.some(forceTopPosition(winBox, leftX, viewport)) : Optional.none();
                default:
                    return Optional.none();
            }
        }).getOr({
            location: 'no-dock'
        });
    };
    const isVisibleForModes = (modes, box, viewport) => forall(modes, (mode) => {
        switch (mode) {
            case 'bottom':
                return isBottomCompletelyVisible(box, viewport.bounds);
            case 'top':
                return isTopCompletelyVisible(box, viewport.bounds);
        }
    });
    const getXYForRestoring = (pos, viewport) => {
        const priorY = viewport.optScrollEnv.fold(constant$1(pos.bounds.y), (scrollEnv) => scrollEnv.scrollElmTop + (pos.bounds.y - scrollEnv.currentScrollTop));
        return SugarPosition(pos.bounds.x, priorY);
    };
    const getXYForSaving = (box, viewport) => {
        const priorY = viewport.optScrollEnv.fold(constant$1(box.y), (scrollEnv) => box.y + scrollEnv.currentScrollTop - scrollEnv.scrollElmTop);
        return SugarPosition(box.x, priorY);
    };
    const getPrior = (elem, viewport, state) => state.getInitialPos().map((pos) => {
        const xy = getXYForRestoring(pos, viewport);
        return {
            box: bounds(xy.left, xy.top, get$c(elem), get$d(elem)),
            location: pos.location
        };
    });
    const storePrior = (elem, box, viewport, state, decision) => {
        const xy = getXYForSaving(box, viewport);
        const bounds$1 = bounds(xy.left, xy.top, box.width, box.height);
        state.setInitialPos({
            style: getAllRaw(elem),
            position: get$e(elem, 'position') || 'static',
            bounds: bounds$1,
            location: decision.location
        });
    };
    // When we are using APIs like forceDockToTop, then we only want to store the previous position
    // if we weren't already docked. Otherwise, we still want to move the component, but keep its old
    // restore values
    const storePriorIfNone = (elem, box, viewport, state, decision) => {
        state.getInitialPos().fold(() => storePrior(elem, box, viewport, state, decision), () => noop);
    };
    const revertToOriginal = (elem, box, state) => state.getInitialPos().bind((position) => {
        state.clearInitialPos();
        switch (position.position) {
            case 'static':
                return Optional.some({
                    morph: 'static'
                });
            case 'absolute':
                const offsetParent = getOffsetParent(elem).getOr(body());
                const offsetBox = box$1(offsetParent);
                // Adding the scrollDelta here may not be the right solution. The basic problem is that the
                // rest of the code isn't considering whether its absolute or not, and where the offset parent
                // is. In the situation where the offset parent is *inside* the scrolling environment, then
                // we don't need to consider the scroll, and that's what getXYForRestoring does ... it removes
                // the scroll. We don't need to consider the scroll because the sink is already affected by the
                // scroll. However, when the sink IS the scroller, its position is not moved by scrolling. But the
                // positions of everything inside it needs to consider the scroll. So we add the scroll value.
                //
                // This might also be a bit naive. It's possible that we need to check that the offsetParent
                // is THE scroller, not just that it has a scroll value. For example, if the offset parent
                // was the body, and the body had a scroll, this might give unexpected results. That's somewhat
                // countered by the fact that if the offset parent is outside the scroller, then you don't really
                // have a scrolling environment any more, because the offset parent isn't going to be impacted
                // at all by the scroller
                const scrollDelta = offsetParent.dom.scrollTop ?? 0;
                return Optional.some({
                    morph: 'absolute',
                    positionCss: NuPositionCss('absolute', get$h(position.style, 'left').map((_left) => box.x - offsetBox.x), get$h(position.style, 'top').map((_top) => box.y - offsetBox.y + scrollDelta), get$h(position.style, 'right').map((_right) => offsetBox.right - box.right), get$h(position.style, 'bottom').map((_bottom) => offsetBox.bottom - box.bottom))
                });
            default:
                return Optional.none();
        }
    });
    const tryMorphToOriginal = (elem, viewport, state) => getPrior(elem, viewport, state)
        .filter(({ box }) => isVisibleForModes(state.getModes(), box, viewport))
        .bind(({ box }) => revertToOriginal(elem, box, state));
    const tryDecisionToFixedMorph = (decision) => {
        switch (decision.location) {
            case 'top': {
                // We store our current position so we can revert to it once it's
                // visible again.
                return Optional.some({
                    morph: 'fixed',
                    positionCss: NuPositionCss('fixed', Optional.some(decision.leftX), Optional.some(decision.topY), Optional.none(), Optional.none())
                });
            }
            case 'bottom': {
                // We store our current position so we can revert to it once it's
                // visible again.
                return Optional.some({
                    morph: 'fixed',
                    positionCss: NuPositionCss('fixed', Optional.some(decision.leftX), Optional.none(), Optional.none(), Optional.some(decision.bottomY))
                });
            }
            default:
                return Optional.none();
        }
    };
    const tryMorphToFixed = (elem, viewport, state) => {
        const box = box$1(elem);
        const winBox = win();
        const decision = tryDockingPosition(state.getModes(), {
            win: winBox,
            box
        }, viewport);
        if (decision.location === 'top' || decision.location === 'bottom') {
            // We are moving from undocked to docked, so store the previous location
            // so that we can restore it when we switch out of docking (back to undocked)
            storePrior(elem, box, viewport, state, decision);
            return tryDecisionToFixedMorph(decision);
        }
        else {
            return Optional.none();
        }
    };
    const tryMorphToOriginalOrUpdateFixed = (elem, viewport, state) => {
        // When a "docked" element is docked to the top of a scroll container (due to optScrollEnv in
        // viewport), we need to reposition its fixed if the scroll container has itself moved its top position.
        // This isn't required when the docking is to the top of the window, because the entire window cannot
        // be scrolled up and down the page - it is the page.
        //
        // Imagine a situation where the toolbar has docked to the top of the scroll container, which is at
        // y = 200. Now, when the user scrolls the page another 50px down the page, the top of the scroll
        // container will now be 150px, but the "fixed" toolbar will still be at "200px". So this is a morph
        // from "fixed" to "fixed", but with new coordinates. So if we can't morph to original from "fixed",
        // we try to update our "fixed" position (if we have a scrolling environment in the viewport)
        return tryMorphToOriginal(elem, viewport, state)
            .orThunk(() => {
            // Importantly, we don't update our stored position for the element before "docking", because
            // this is a transition between "docked" and "docked", not "undocked" and "docked". We want to
            // keep our undocked position in our store, not a docked position.
            // So we don't change our stored position. We just improve our fixed.
            return viewport.optScrollEnv
                .bind((_) => getPrior(elem, viewport, state))
                .bind(({ box, location }) => {
                const winBox = win();
                const leftX = getDockedLeftPosition({ win: winBox, box });
                // Keep the same docking location
                const decision = location === 'top'
                    ? forceTopPosition(winBox, leftX, viewport)
                    : forceBottomPosition(winBox, leftX, viewport);
                return tryDecisionToFixedMorph(decision);
            });
        });
    };
    const tryMorph = (component, viewport, state) => {
        const elem = component.element;
        const isDocked = is$1(getRaw(elem, 'position'), 'fixed');
        return isDocked
            ? tryMorphToOriginalOrUpdateFixed(elem, viewport, state)
            : tryMorphToFixed(elem, viewport, state);
    };
    // The difference between the "calculate" functions and the "try" functions is that the "try" functions
    // will first consider whether there is a need to morph, whereas the "calculate" functions will just
    // give you the morph details, bypassing the check to see if it's needed
    const calculateMorphToOriginal = (component, viewport, state) => {
        const elem = component.element;
        return getPrior(elem, viewport, state)
            .bind(({ box }) => revertToOriginal(elem, box, state));
    };
    const forceDockWith = (elem, viewport, state, getDecision) => {
        const box = box$1(elem);
        const winBox = win();
        const leftX = getDockedLeftPosition({ win: winBox, box });
        const decision = getDecision(winBox, leftX, viewport);
        if (decision.location === 'bottom' || decision.location === 'top') {
            // We only want to store the values if we aren't already docking. If we are already docking, then
            // we just want to move the element, without updating where it started originally
            storePriorIfNone(elem, box, viewport, state, decision);
            return tryDecisionToFixedMorph(decision);
        }
        else {
            return Optional.none();
        }
    };

    const morphToStatic = (component, config, state) => {
        state.setDocked(false);
        each$1(['left', 'right', 'top', 'bottom', 'position'], (prop) => remove$6(component.element, prop));
        config.onUndocked(component);
    };
    const morphToCoord = (component, config, state, position) => {
        const isDocked = position.position === 'fixed';
        state.setDocked(isDocked);
        applyPositionCss(component.element, position);
        const method = isDocked ? config.onDocked : config.onUndocked;
        method(component);
    };
    const updateVisibility = (component, config, state, viewport, morphToDocked = false) => {
        config.contextual.each((contextInfo) => {
            // Make the dockable component disappear if the context is outside the viewport
            contextInfo.lazyContext(component).each((box) => {
                const isVisible = isPartiallyVisible(box, viewport.bounds);
                if (isVisible !== state.isVisible()) {
                    state.setVisible(isVisible);
                    // If morphing to docked and the context isn't visible then immediately set
                    // the fadeout class and don't worry about transitioning, as the context
                    // would never have been in view while docked
                    if (morphToDocked && !isVisible) {
                        add$1(component.element, [contextInfo.fadeOutClass]);
                        contextInfo.onHide(component);
                    }
                    else {
                        const method = isVisible ? appear : disappear;
                        method(component, contextInfo);
                    }
                }
            });
        });
    };
    const applyFixedMorph = (component, config, state, viewport, morph) => {
        // This "updateVisibility" call is potentially duplicated with the
        // call in refreshInternal for isDocked. We might want to consolidate them.
        // The difference between them is the "morphToDocked" flag.
        updateVisibility(component, config, state, viewport, true);
        morphToCoord(component, config, state, morph.positionCss);
    };
    const applyMorph = (component, config, state, viewport, morph) => {
        // Apply the morph result depending on its type
        switch (morph.morph) {
            case 'static': {
                return morphToStatic(component, config, state);
            }
            case 'absolute': {
                return morphToCoord(component, config, state, morph.positionCss);
            }
            case 'fixed': {
                return applyFixedMorph(component, config, state, viewport, morph);
            }
        }
    };
    const refreshInternal = (component, config, state) => {
        // Absolute coordinates (considers scroll)
        const viewport = config.lazyViewport(component);
        updateVisibility(component, config, state, viewport);
        tryMorph(component, viewport, state).each((morph) => {
            applyMorph(component, config, state, viewport, morph);
        });
    };
    const resetInternal = (component, config, state) => {
        // Morph back to the original position
        const elem = component.element;
        state.setDocked(false);
        const viewport = config.lazyViewport(component);
        calculateMorphToOriginal(component, viewport, state).each((staticOrAbsoluteMorph) => {
            // This code is very similar to the "applyMorph" function above. The main difference
            // is that it doesn't consider fixed position, because something that is docking
            // can't currently start with fixed position
            switch (staticOrAbsoluteMorph.morph) {
                case 'static': {
                    morphToStatic(component, config, state);
                    break;
                }
                case 'absolute': {
                    morphToCoord(component, config, state, staticOrAbsoluteMorph.positionCss);
                    break;
                }
            }
        });
        // Remove contextual visibility classes
        state.setVisible(true);
        config.contextual.each((contextInfo) => {
            remove$2(elem, [contextInfo.fadeInClass, contextInfo.fadeOutClass, contextInfo.transitionClass]);
            contextInfo.onShow(component);
        });
        // Apply docking again to reset the position
        refresh$4(component, config, state);
    };
    const refresh$4 = (component, config, state) => {
        // Ensure the component is attached to the document/world, if not then do nothing as we can't
        // check if the component should be docked or not when in a detached state
        if (component.getSystem().isConnected()) {
            refreshInternal(component, config, state);
        }
    };
    const reset$1 = (component, config, state) => {
        // If the component is not docked then there's no need to reset the state,
        // so only reset when docked
        if (state.isDocked()) {
            resetInternal(component, config, state);
        }
    };
    const forceDockWithDecision = (getDecision) => (component, config, state) => {
        const viewport = config.lazyViewport(component);
        const optMorph = forceDockWith(component.element, viewport, state, getDecision);
        optMorph.each((morph) => {
            // ASSUMPTION: This "applyFixedMorph" sets state.setDocked to true.
            applyFixedMorph(component, config, state, viewport, morph);
        });
    };
    const forceDockToTop = forceDockWithDecision(forceTopPosition);
    const forceDockToBottom = forceDockWithDecision(forceBottomPosition);
    const isDocked$2 = (component, config, state) => state.isDocked();
    const setModes = (component, config, state, modes) => state.setModes(modes);
    const getModes = (component, config, state) => state.getModes();

    var DockingApis = /*#__PURE__*/Object.freeze({
        __proto__: null,
        refresh: refresh$4,
        reset: reset$1,
        isDocked: isDocked$2,
        getModes: getModes,
        setModes: setModes,
        forceDockToTop: forceDockToTop,
        forceDockToBottom: forceDockToBottom
    });

    const events$e = (dockInfo, dockState) => derive$2([
        runOnSource(transitionend(), (component, simulatedEvent) => {
            dockInfo.contextual.each((contextInfo) => {
                if (has(component.element, contextInfo.transitionClass)) {
                    remove$2(component.element, [contextInfo.transitionClass, contextInfo.fadeInClass]);
                    const notify = dockState.isVisible() ? contextInfo.onShown : contextInfo.onHidden;
                    notify(component);
                }
                simulatedEvent.stop();
            });
        }),
        run$1(windowScroll(), (component, _) => {
            refresh$4(component, dockInfo, dockState);
        }),
        run$1(externalElementScroll(), (component, _) => {
            refresh$4(component, dockInfo, dockState);
        }),
        run$1(windowResize(), (component, _) => {
            reset$1(component, dockInfo, dockState);
        })
    ]);

    var ActiveDocking = /*#__PURE__*/Object.freeze({
        __proto__: null,
        events: events$e
    });

    var DockingSchema = [
        optionObjOf('contextual', [
            requiredString('fadeInClass'),
            requiredString('fadeOutClass'),
            requiredString('transitionClass'),
            requiredFunction('lazyContext'),
            onHandler('onShow'),
            onHandler('onShown'),
            onHandler('onHide'),
            onHandler('onHidden')
        ]),
        defaultedFunction('lazyViewport', () => ({
            bounds: win(),
            optScrollEnv: Optional.none()
        })),
        defaultedArrayOf('modes', ['top', 'bottom'], string),
        onHandler('onDocked'),
        onHandler('onUndocked')
    ];

    const init$d = (spec) => {
        const docked = Cell(false);
        const visible = Cell(true);
        const initialBounds = value$2();
        const modes = Cell(spec.modes);
        const readState = () => `docked:  ${docked.get()}, visible: ${visible.get()}, modes: ${modes.get().join(',')}`;
        return nu$4({
            isDocked: docked.get,
            setDocked: docked.set,
            getInitialPos: initialBounds.get,
            setInitialPos: initialBounds.set,
            clearInitialPos: initialBounds.clear,
            isVisible: visible.get,
            setVisible: visible.set,
            getModes: modes.get,
            setModes: modes.set,
            readState
        });
    };

    var DockingState = /*#__PURE__*/Object.freeze({
        __proto__: null,
        init: init$d
    });

    const Docking = create$3({
        fields: DockingSchema,
        name: 'docking',
        active: ActiveDocking,
        apis: DockingApis,
        state: DockingState
    });

    /*
     * origin: the position (without scroll) of the offset parent
     * scroll: the scrolling position of the window
     *
     * fixed: the fixed coordinates to show for css
     * offset: the absolute coordinates to show for css when inside an offset parent
     * absolute: the absolute coordinates to show before considering the offset parent
     */
    const adt$4 = Adt.generate([
        { offset: ['x', 'y'] },
        { absolute: ['x', 'y'] },
        { fixed: ['x', 'y'] }
    ]);
    const subtract = (change) => (point) => point.translate(-change.left, -change.top);
    const add = (change) => (point) => point.translate(change.left, change.top);
    const transform = (changes) => (x, y) => foldl(changes, (rest, f) => f(rest), SugarPosition(x, y));
    const asFixed = (coord, scroll, origin) => coord.fold(
    // offset to fixed
    transform([add(origin), subtract(scroll)]), 
    // absolute to fixed
    transform([subtract(scroll)]), 
    // fixed to fixed
    transform([]));
    const asAbsolute = (coord, scroll, origin) => coord.fold(
    // offset to absolute
    transform([add(origin)]), 
    // absolute to absolute
    transform([]), 
    // fixed to absolute
    transform([add(scroll)]));
    const asOffset = (coord, scroll, origin) => coord.fold(
    // offset to offset
    transform([]), 
    // absolute to offset
    transform([subtract(origin)]), 
    // fixed to offset
    transform([add(scroll), subtract(origin)]));
    const withinRange = (coord1, coord2, xRange, yRange, scroll, origin) => {
        const a1 = asAbsolute(coord1, scroll, origin);
        const a2 = asAbsolute(coord2, scroll, origin);
        // console.log(`a1.left: ${a1.left}, a2.left: ${a2.left}, leftDelta: ${a1.left - a2.left}, xRange: ${xRange}, lD <= xRange: ${Math.abs(a1.left - a2.left) <= xRange}`);
        // console.log(`a1.top: ${a1.top}, a2.top: ${a2.top}, topDelta: ${a1.top - a2.top}, yRange: ${yRange}, lD <= xRange: ${Math.abs(a1.top - a2.top) <= yRange}`);
        return Math.abs(a1.left - a2.left) <= xRange &&
            Math.abs(a1.top - a2.top) <= yRange;
    };
    const getDeltas = (coord1, coord2, xRange, yRange, scroll, origin) => {
        const a1 = asAbsolute(coord1, scroll, origin);
        const a2 = asAbsolute(coord2, scroll, origin);
        const left = Math.abs(a1.left - a2.left);
        const top = Math.abs(a1.top - a2.top);
        return SugarPosition(left, top);
    };
    const toStyles = (coord, scroll, origin) => {
        const stylesOpt = coord.fold((x, y) => ({ position: Optional.some('absolute'), left: Optional.some(x + 'px'), top: Optional.some(y + 'px') }), // offset
        (x, y) => ({ position: Optional.some('absolute'), left: Optional.some((x - origin.left) + 'px'), top: Optional.some((y - origin.top) + 'px') }), // absolute
        (x, y) => ({ position: Optional.some('fixed'), left: Optional.some(x + 'px'), top: Optional.some(y + 'px') }) // fixed
        );
        return { right: Optional.none(), bottom: Optional.none(), ...stylesOpt };
    };
    const translate$2 = (coord, deltaX, deltaY) => coord.fold((x, y) => offset(x + deltaX, y + deltaY), (x, y) => absolute$1(x + deltaX, y + deltaY), (x, y) => fixed$1(x + deltaX, y + deltaY));
    const absorb = (partialCoord, originalCoord, scroll, origin) => {
        const absorbOne = (stencil, nu) => (optX, optY) => {
            const original = stencil(originalCoord, scroll, origin);
            return nu(optX.getOr(original.left), optY.getOr(original.top));
        };
        return partialCoord.fold(absorbOne(asOffset, offset), absorbOne(asAbsolute, absolute$1), absorbOne(asFixed, fixed$1));
    };
    const offset = adt$4.offset;
    const absolute$1 = adt$4.absolute;
    const fixed$1 = adt$4.fixed;

    const parseAttrToInt = (element, name) => {
        const value = get$g(element, name);
        return isUndefined(value) ? NaN : parseInt(value, 10);
    };
    // NOTE: Moved from ego with some parameterisation
    const get$3 = (component, snapsInfo) => {
        const element = component.element;
        const x = parseAttrToInt(element, snapsInfo.leftAttr);
        const y = parseAttrToInt(element, snapsInfo.topAttr);
        return isNaN(x) || isNaN(y) ? Optional.none() : Optional.some(SugarPosition(x, y));
    };
    const set$1 = (component, snapsInfo, pt) => {
        const element = component.element;
        set$9(element, snapsInfo.leftAttr, pt.left + 'px');
        set$9(element, snapsInfo.topAttr, pt.top + 'px');
    };
    const clear = (component, snapsInfo) => {
        const element = component.element;
        remove$8(element, snapsInfo.leftAttr);
        remove$8(element, snapsInfo.topAttr);
    };

    // Types of coordinates
    // SugarLocation: This is the position on the screen including scroll.
    // Absolute: This is the css setting that would be applied. Therefore, it subtracts
    // the origin of the relative offsetParent.
    // Fixed: This is the fixed position.
    /*
     So in attempt to make this more understandable, let's use offset, absolute, and fixed.
     and try and model individual combinators
    */
    /*

     Relationships:
       - location -> absolute: should just need to subtract the position of the offset parent (origin)
       - location -> fixed: subtract the scrolling
       - absolute -> fixed: add the origin, and subtract the scrolling
       - absolute -> location: add the origin
       - fixed -> absolute: add the scrolling, remove the origin
       - fixed -> location: add the scrolling

    /*
     * When the user is dragging around the element, and it snaps into place, it is important
     * for the next movement to be from its pre-snapped location, rather than the snapped location.
     * This is because if it is from the snapped location the next delta movement may not actually
     * be high enough to get it out of the snap area, and hence, it will just snap again (and again).
     */
    // This identifies the position of the draggable element as either its current position, or the position
    // that we put on it before we snapped it into place (before dropping). Once it's dropped, the presnap
    // position will go away. It is used to avoid the situation where you can't escape the snap unless you
    // move the mouse really quickly :)
    const getCoords = (component, snapInfo, coord, delta) => get$3(component, snapInfo).fold(() => coord, (fixed) => 
    // We have a pre-snap position, so we have to apply the delta ourselves
    fixed$1(fixed.left + delta.left, fixed.top + delta.top));
    const moveOrSnap = (component, snapInfo, coord, delta, scroll, origin) => {
        const newCoord = getCoords(component, snapInfo, coord, delta);
        const snap = snapInfo.mustSnap ? findClosestSnap(component, snapInfo, newCoord, scroll, origin) :
            findSnap(component, snapInfo, newCoord, scroll, origin);
        const fixedCoord = asFixed(newCoord, scroll, origin);
        set$1(component, snapInfo, fixedCoord);
        return snap.fold(() => ({
            coord: fixed$1(fixedCoord.left, fixedCoord.top),
            extra: Optional.none()
        })
        // No snap.
        // var newfixed = graph.boundToFixed(theatre, element, loc.left, loc.top, fixed.left, fixed.top, height);
        // presnaps.set(element, 'fixed', newfixed.left, newfixed.top);
        // return { position: 'fixed', left: newfixed.left + 'px', top: newfixed.top + 'px' };
        , (spanned) => ({
            coord: spanned.output,
            extra: spanned.extra
        }));
    };
    const stopDrag = (component, snapInfo) => {
        clear(component, snapInfo);
    };
    const findMatchingSnap = (snaps, newCoord, scroll, origin) => findMap(snaps, (snap) => {
        const sensor = snap.sensor;
        const inRange = withinRange(newCoord, sensor, snap.range.left, snap.range.top, scroll, origin);
        return inRange ? Optional.some({
            output: absorb(snap.output, newCoord, scroll, origin),
            extra: snap.extra
        }) : Optional.none();
    });
    const findClosestSnap = (component, snapInfo, newCoord, scroll, origin) => {
        // You need to pass in the absX and absY so that they can be used for things which only care about snapping one axis and keeping the other one.
        const snaps = snapInfo.getSnapPoints(component);
        const matchSnap = findMatchingSnap(snaps, newCoord, scroll, origin);
        return matchSnap.orThunk(() => {
            const bestSnap = foldl(snaps, (acc, snap) => {
                const sensor = snap.sensor;
                const deltas = getDeltas(newCoord, sensor, snap.range.left, snap.range.top, scroll, origin);
                return acc.deltas.fold(() => ({
                    deltas: Optional.some(deltas),
                    snap: Optional.some(snap)
                }), (bestDeltas) => {
                    const currAvg = (deltas.left + deltas.top) / 2;
                    const bestAvg = (bestDeltas.left + bestDeltas.top) / 2;
                    if (currAvg <= bestAvg) {
                        return {
                            deltas: Optional.some(deltas),
                            snap: Optional.some(snap)
                        };
                    }
                    else {
                        return acc;
                    }
                });
            }, {
                deltas: Optional.none(),
                snap: Optional.none()
            });
            return bestSnap.snap.map((snap) => ({
                output: absorb(snap.output, newCoord, scroll, origin),
                extra: snap.extra
            }));
        });
    };
    // x: the absolute position.left of the draggable element
    // y: the absolute position.top of the draggable element
    // deltaX: the amount the mouse has moved horizontally
    // deltaY: the amount the mouse has moved vertically
    const findSnap = (component, snapInfo, newCoord, scroll, origin) => {
        // You need to pass in the absX and absY so that they can be used for things which only care about snapping one axis and keeping the other one.
        const snaps = snapInfo.getSnapPoints(component);
        // HERE
        return findMatchingSnap(snaps, newCoord, scroll, origin);
    };
    const snapTo$1 = (snap, scroll, origin) => ({
        // TODO: This looks to be incorrect and needs fixing as DragCoord definitely needs a number
        // based drag coord for the second argument here, so this is probably a bug.
        coord: absorb(snap.output, snap.output, scroll, origin),
        extra: snap.extra
    });

    const snapTo = (component, dragConfig, _state, snap) => {
        const target = dragConfig.getTarget(component.element);
        if (dragConfig.repositionTarget) {
            const doc = owner$4(component.element);
            const scroll = get$b(doc);
            const origin = getOrigin(target);
            const snapPin = snapTo$1(snap, scroll, origin);
            const styles = toStyles(snapPin.coord, scroll, origin);
            setOptions(target, styles);
        }
    };

    var DraggingApis = /*#__PURE__*/Object.freeze({
        __proto__: null,
        snapTo: snapTo
    });

    const field = (name, forbidden) => defaultedObjOf(name, {}, map$2(forbidden, (f) => forbid(f.name(), 'Cannot configure ' + f.name() + ' for ' + name)).concat([
        customField('dump', identity)
    ]));
    const get$2 = (data) => data.dump;
    const augment = (data, original) => ({
        ...derive$1(original),
        ...data.dump
    });
    // Is this used?
    const SketchBehaviours = {
        field,
        augment,
        get: get$2
    };

    const base = (partSchemas, partUidsSchemas) => {
        const ps = partSchemas.length > 0 ? [
            requiredObjOf('parts', partSchemas)
        ] : [];
        return ps.concat([
            required$1('uid'),
            defaulted('dom', {}), // Maybe get rid of.
            defaulted('components', []),
            snapshot('originalSpec'),
            defaulted('debug.sketcher', {})
        ]).concat(partUidsSchemas);
    };
    const asRawOrDie = (label, schema, spec, partSchemas, partUidsSchemas) => {
        const baseS = base(partSchemas, partUidsSchemas);
        return asRawOrDie$1(label + ' [SpecSchema]', objOfOnly(baseS.concat(schema)), spec);
    };

    const single$1 = (owner, schema, factory, spec) => {
        const specWithUid = supplyUid(spec);
        const detail = asRawOrDie(owner, schema, specWithUid, [], []);
        return factory(detail, specWithUid);
    };
    const composite$1 = (owner, schema, partTypes, factory, spec) => {
        const specWithUid = supplyUid(spec);
        // Identify any information required for external parts
        const partSchemas = schemas(partTypes);
        // Generate partUids for all parts (external and otherwise)
        const partUidsSchema = defaultUidsSchema(partTypes);
        const detail = asRawOrDie(owner, schema, specWithUid, partSchemas, [partUidsSchema]);
        // Create (internals, externals) substitutions
        const subs = substitutes(owner, detail, partTypes);
        // Work out the components by substituting internals
        const components = components$1(owner, detail, subs.internals());
        return factory(detail, components, specWithUid, subs.externals());
    };
    const hasUid = (spec) => has$2(spec, 'uid');
    const supplyUid = (spec) => {
        return hasUid(spec) ? spec : {
            ...spec,
            uid: generate$4('uid')
        };
    };

    const isSketchSpec = (spec) => {
        return spec.uid !== undefined;
    };
    const singleSchema = objOfOnly([
        required$1('name'),
        required$1('factory'),
        required$1('configFields'),
        defaulted('apis', {}),
        defaulted('extraApis', {})
    ]);
    const compositeSchema = objOfOnly([
        required$1('name'),
        required$1('factory'),
        required$1('configFields'),
        required$1('partFields'),
        defaulted('apis', {}),
        defaulted('extraApis', {})
    ]);
    const single = (rawConfig) => {
        const config = asRawOrDie$1('Sketcher for ' + rawConfig.name, singleSchema, rawConfig);
        const sketch = (spec) => single$1(config.name, config.configFields, config.factory, spec);
        const apis = map$1(config.apis, makeApi);
        const extraApis = map$1(config.extraApis, (f, k) => markAsExtraApi(f, k));
        return {
            name: config.name,
            configFields: config.configFields,
            sketch,
            ...apis,
            ...extraApis
        };
    };
    const composite = (rawConfig) => {
        const config = asRawOrDie$1('Sketcher for ' + rawConfig.name, compositeSchema, rawConfig);
        const sketch = (spec) => composite$1(config.name, config.configFields, config.partFields, config.factory, spec);
        // These are constructors that will store their configuration.
        const parts = generate$5(config.name, config.partFields);
        const apis = map$1(config.apis, makeApi);
        const extraApis = map$1(config.extraApis, (f, k) => markAsExtraApi(f, k));
        return {
            name: config.name,
            partFields: config.partFields,
            configFields: config.configFields,
            sketch,
            parts,
            ...apis,
            ...extraApis
        };
    };

    const factory$n = (detail) => {
        const { attributes, ...domWithoutAttributes } = detail.dom;
        return {
            uid: detail.uid,
            dom: {
                tag: 'div',
                attributes: {
                    role: 'presentation',
                    ...attributes
                },
                ...domWithoutAttributes
            },
            components: detail.components,
            behaviours: get$2(detail.containerBehaviours),
            events: detail.events,
            domModification: detail.domModification,
            eventOrder: detail.eventOrder
        };
    };
    const Container = single({
        name: 'Container',
        factory: factory$n,
        configFields: [
            defaulted('components', []),
            field('containerBehaviours', []),
            // TODO: Deprecate
            defaulted('events', {}),
            defaulted('domModification', {}),
            defaulted('eventOrder', {})
        ]
    });

    const initialAttribute = 'data-initial-z-index';
    // We have to alter the z index of the alloy root of the blocker so that
    // it can have a z-index high enough to act as the "blocker". Just before
    // discarding it, we need to reset those z-indices back to what they
    // were. ASSUMPTION: the blocker has been added as a direct child of the root
    const resetZIndex = (blocker) => {
        parent(blocker.element).filter(isElement$1).each((root) => {
            getOpt(root, initialAttribute).fold(() => remove$6(root, 'z-index'), (zIndex) => set$7(root, 'z-index', zIndex));
            remove$8(root, initialAttribute);
        });
    };
    const changeZIndex = (blocker) => {
        parent(blocker.element).filter(isElement$1).each((root) => {
            getRaw(root, 'z-index').each((zindex) => {
                set$9(root, initialAttribute, zindex);
            });
            // Used to be a really high number, but it probably just has
            // to match the blocker
            set$7(root, 'z-index', get$e(blocker.element, 'z-index'));
        });
    };
    const instigate = (anyComponent, blocker) => {
        anyComponent.getSystem().addToGui(blocker);
        changeZIndex(blocker);
    };
    const discard = (blocker) => {
        resetZIndex(blocker);
        blocker.getSystem().removeFromGui(blocker);
    };
    const createComponent = (component, blockerClass, blockerEvents) => component.getSystem().build(Container.sketch({
        dom: {
            // Probably consider doing with classes?
            styles: {
                'left': '0px',
                'top': '0px',
                'width': '100%',
                'height': '100%',
                'position': 'fixed',
                'z-index': '1000000000000000'
            },
            classes: [blockerClass]
        },
        events: blockerEvents
    }));

    var SnapSchema = optionObjOf('snaps', [
        required$1('getSnapPoints'),
        onHandler('onSensor'),
        required$1('leftAttr'),
        required$1('topAttr'),
        defaulted('lazyViewport', win),
        defaulted('mustSnap', false)
    ]);

    const schema$r = [
        // Is this used?
        defaulted('useFixed', never),
        required$1('blockerClass'),
        defaulted('getTarget', identity),
        defaulted('onDrag', noop),
        defaulted('repositionTarget', true),
        defaulted('onDrop', noop),
        defaultedFunction('getBounds', win),
        SnapSchema
    ];

    const getCurrentCoord = (target) => lift3(getRaw(target, 'left'), getRaw(target, 'top'), getRaw(target, 'position'), (left, top, position) => {
        const nu = position === 'fixed' ? fixed$1 : offset;
        return nu(parseInt(left, 10), parseInt(top, 10));
    }).getOrThunk(() => {
        const location = absolute$3(target);
        return absolute$1(location.left, location.top);
    });
    const clampCoords = (component, coords, scroll, origin, startData) => {
        const bounds = startData.bounds;
        const absoluteCoord = asAbsolute(coords, scroll, origin);
        const newX = clamp(absoluteCoord.left, bounds.x, bounds.x + bounds.width - startData.width);
        const newY = clamp(absoluteCoord.top, bounds.y, bounds.y + bounds.height - startData.height);
        const newCoords = absolute$1(newX, newY);
        // Translate the absolute coord back into the previous type
        return coords.fold(
        // offset
        () => {
            const offset$1 = asOffset(newCoords, scroll, origin);
            return offset(offset$1.left, offset$1.top);
        }, 
        // absolute
        constant$1(newCoords), 
        // fixed
        () => {
            const fixed = asFixed(newCoords, scroll, origin);
            return fixed$1(fixed.left, fixed.top);
        });
    };
    const calcNewCoord = (component, optSnaps, currentCoord, scroll, origin, delta, startData) => {
        const newCoord = optSnaps.fold(() => {
            // When not docking, use fixed coordinates.
            const translated = translate$2(currentCoord, delta.left, delta.top);
            const fixedCoord = asFixed(translated, scroll, origin);
            return fixed$1(fixedCoord.left, fixedCoord.top);
        }, (snapInfo) => {
            const snapping = moveOrSnap(component, snapInfo, currentCoord, delta, scroll, origin);
            snapping.extra.each((extra) => {
                snapInfo.onSensor(component, extra);
            });
            return snapping.coord;
        });
        // Clamp the coords so that they are within the bounds
        return clampCoords(component, newCoord, scroll, origin, startData);
    };
    const dragBy = (component, dragConfig, startData, delta) => {
        const target = dragConfig.getTarget(component.element);
        if (dragConfig.repositionTarget) {
            const doc = owner$4(component.element);
            const scroll = get$b(doc);
            const origin = getOrigin(target);
            const currentCoord = getCurrentCoord(target);
            const newCoord = calcNewCoord(component, dragConfig.snaps, currentCoord, scroll, origin, delta, startData);
            const styles = toStyles(newCoord, scroll, origin);
            setOptions(target, styles);
        }
        // NOTE: On drag just goes with the original delta. It does not know about snapping.
        dragConfig.onDrag(component, target, delta);
    };

    const calcStartData = (dragConfig, comp) => ({
        bounds: dragConfig.getBounds(),
        height: getOuter$1(comp.element),
        width: getOuter(comp.element)
    });
    const move = (component, dragConfig, dragState, dragMode, event) => {
        const delta = dragState.update(dragMode, event);
        const dragStartData = dragState.getStartData().getOrThunk(() => calcStartData(dragConfig, component));
        delta.each((dlt) => {
            dragBy(component, dragConfig, dragStartData, dlt);
        });
    };
    const stop = (component, blocker, dragConfig, dragState) => {
        blocker.each(discard);
        dragConfig.snaps.each((snapInfo) => {
            stopDrag(component, snapInfo);
        });
        const target = dragConfig.getTarget(component.element);
        dragState.reset();
        dragConfig.onDrop(component, target);
    };
    const handlers = (events) => (dragConfig, dragState) => {
        const updateStartState = (comp) => {
            dragState.setStartData(calcStartData(dragConfig, comp));
        };
        return derive$2([
            run$1(windowScroll(), (comp) => {
                // Only update if we have some start data
                dragState.getStartData().each(() => updateStartState(comp));
            }),
            ...events(dragConfig, dragState, updateStartState)
        ]);
    };

    const init$c = (dragApi) => derive$2([
        // When the user clicks on the blocker, something has probably gone slightly
        // wrong, so we'll just drop for safety. The blocker should really only
        // be there when the mouse is already down and not released, so a 'click'
        run$1(mousedown(), dragApi.forceDrop),
        // When the user releases the mouse on the blocker, that is a drop
        run$1(mouseup(), dragApi.drop),
        // As the user moves the mouse around (while pressed down), we move the
        // component around
        run$1(mousemove(), (comp, simulatedEvent) => {
            dragApi.move(simulatedEvent.event);
        }),
        // When the use moves outside the range, schedule a block to occur but
        // give it a chance to be cancelled.
        run$1(mouseout(), dragApi.delayDrop)
    ]);

    const getData$1 = (event) => Optional.from(SugarPosition(event.x, event.y));
    // When dragging with the mouse, the delta is simply the difference
    // between the two position (previous/old and next/nu)
    const getDelta$1 = (old, nu) => SugarPosition(nu.left - old.left, nu.top - old.top);

    var MouseData = /*#__PURE__*/Object.freeze({
        __proto__: null,
        getData: getData$1,
        getDelta: getDelta$1
    });

    const events$d = (dragConfig, dragState, updateStartState) => [
        run$1(mousedown(), (component, simulatedEvent) => {
            const raw = simulatedEvent.event.raw;
            if (raw.button !== 0) {
                return;
            }
            simulatedEvent.stop();
            const stop$1 = () => stop(component, Optional.some(blocker), dragConfig, dragState);
            // If the user has moved something outside the area, and has not come back within
            // 200 ms, then drop
            const delayDrop = DelayedFunction(stop$1, 200);
            const dragApi = {
                drop: stop$1,
                delayDrop: delayDrop.schedule,
                forceDrop: stop$1,
                move: (event) => {
                    // Stop any pending drops caused by mouseout
                    delayDrop.cancel();
                    move(component, dragConfig, dragState, MouseData, event);
                }
            };
            const blocker = createComponent(component, dragConfig.blockerClass, init$c(dragApi));
            const start = () => {
                updateStartState(component);
                instigate(component, blocker);
            };
            start();
        })
    ];
    const schema$q = [
        ...schema$r,
        output$1('dragger', {
            handlers: handlers(events$d)
        })
    ];

    const init$b = (dragApi) => derive$2([
        // When the user taps on the blocker, something has probably gone slightly
        // wrong, so we'll just drop for safety. The blocker should really only
        // be there when their finger is already down and not released, so a 'tap'
        run$1(touchstart(), dragApi.forceDrop),
        // When the user releases their finger on the blocker, that is a drop
        run$1(touchend(), dragApi.drop),
        run$1(touchcancel(), dragApi.drop),
        // As the user moves their finger around (while pressed down), we move the
        // component around
        run$1(touchmove(), (comp, simulatedEvent) => {
            dragApi.move(simulatedEvent.event);
        })
    ]);

    const getDataFrom = (touches) => {
        const touch = touches[0];
        return Optional.some(SugarPosition(touch.clientX, touch.clientY));
    };
    const getData = (event) => {
        const raw = event.raw;
        const touches = raw.touches;
        return touches.length === 1 ? getDataFrom(touches) : Optional.none();
    };
    // When dragging the touch, the delta is simply the difference
    // between the two touch positions (previous/old and next/nu)
    const getDelta = (old, nu) => SugarPosition(nu.left - old.left, nu.top - old.top);

    var TouchData = /*#__PURE__*/Object.freeze({
        __proto__: null,
        getData: getData,
        getDelta: getDelta
    });

    const events$c = (dragConfig, dragState, updateStartState) => {
        const blockerSingleton = value$2();
        const stopBlocking = (component) => {
            stop(component, blockerSingleton.get(), dragConfig, dragState);
            blockerSingleton.clear();
        };
        // Android fires events on the component at all times, while iOS initially fires on the component
        // but once moved off the component then fires on the element behind. As such we need to use
        // a blocker and then listen to both touchmove/touchend on both the component and blocker.
        return [
            run$1(touchstart(), (component, simulatedEvent) => {
                simulatedEvent.stop();
                const stop = () => stopBlocking(component);
                const dragApi = {
                    drop: stop,
                    // delayDrop is not used by touch
                    delayDrop: noop,
                    forceDrop: stop,
                    move: (event) => {
                        move(component, dragConfig, dragState, TouchData, event);
                    }
                };
                const blocker = createComponent(component, dragConfig.blockerClass, init$b(dragApi));
                blockerSingleton.set(blocker);
                const start = () => {
                    updateStartState(component);
                    instigate(component, blocker);
                };
                start();
            }),
            run$1(touchmove(), (component, simulatedEvent) => {
                simulatedEvent.stop();
                move(component, dragConfig, dragState, TouchData, simulatedEvent.event);
            }),
            run$1(touchend(), (component, simulatedEvent) => {
                simulatedEvent.stop();
                stopBlocking(component);
            }),
            run$1(touchcancel(), stopBlocking)
        ];
    };
    const schema$p = [
        ...schema$r,
        output$1('dragger', {
            handlers: handlers(events$c)
        })
    ];

    const events$b = (dragConfig, dragState, updateStartState) => [
        ...events$d(dragConfig, dragState, updateStartState),
        ...events$c(dragConfig, dragState, updateStartState)
    ];
    const schema$o = [
        ...schema$r,
        output$1('dragger', {
            handlers: handlers(events$b)
        })
    ];

    const mouse = schema$q;
    const touch = schema$p;
    const mouseOrTouch = schema$o;

    var DraggingBranches = /*#__PURE__*/Object.freeze({
        __proto__: null,
        mouse: mouse,
        touch: touch,
        mouseOrTouch: mouseOrTouch
    });

    // NOTE: mode refers to the way that information is retrieved from
    // the user interaction. It can be things like MouseData, TouchData etc.
    const init$a = () => {
        // Dragging operates on the difference between the previous user
        // interaction and the next user interaction. Therefore, we store
        // the previous interaction so that we can compare it.
        let previous = Optional.none();
        // Dragging requires calculating the bounds, so we store that data initially
        // to reduce the amount of computation each mouse movement
        let startData = Optional.none();
        const reset = () => {
            previous = Optional.none();
            startData = Optional.none();
        };
        // Return position delta between previous position and nu position,
        // or None if this is the first. Set the previous position to nu.
        const calculateDelta = (mode, nu) => {
            const result = previous.map((old) => mode.getDelta(old, nu));
            previous = Optional.some(nu);
            return result;
        };
        // NOTE: This dragEvent is the DOM touch event or mouse event
        const update = (mode, dragEvent) => mode.getData(dragEvent).bind((nuData) => calculateDelta(mode, nuData));
        const setStartData = (data) => {
            startData = Optional.some(data);
        };
        const getStartData = () => startData;
        const readState = constant$1({});
        return nu$4({
            readState,
            reset,
            update,
            getStartData,
            setStartData
        });
    };

    var DragState = /*#__PURE__*/Object.freeze({
        __proto__: null,
        init: init$a
    });

    const Dragging = createModes({
        branchKey: 'mode',
        branches: DraggingBranches,
        name: 'dragging',
        active: {
            events: (dragConfig, dragState) => {
                const dragger = dragConfig.dragger;
                return dragger.handlers(dragConfig, dragState);
            }
        },
        extra: {
            // Extra. Does not need component as input.
            snap: (sConfig) => ({
                sensor: sConfig.sensor,
                range: sConfig.range,
                output: sConfig.output,
                extra: Optional.from(sConfig.extra)
            })
        },
        state: DragState,
        apis: DraggingApis
    });

    const ariaElements = [
        'input',
        'textarea'
    ];
    const isAriaElement = (elem) => {
        const name = name$3(elem);
        return contains$2(ariaElements, name);
    };
    const markValid = (component, invalidConfig) => {
        const elem = invalidConfig.getRoot(component).getOr(component.element);
        remove$3(elem, invalidConfig.invalidClass);
        invalidConfig.notify.each((notifyInfo) => {
            if (isAriaElement(component.element)) {
                set$9(component.element, 'aria-invalid', false);
            }
            notifyInfo.getContainer(component).each((container) => {
                set$8(container, notifyInfo.validHtml);
            });
            notifyInfo.onValid(component);
        });
    };
    const markInvalid = (component, invalidConfig, invalidState, text) => {
        const elem = invalidConfig.getRoot(component).getOr(component.element);
        add$2(elem, invalidConfig.invalidClass);
        invalidConfig.notify.each((notifyInfo) => {
            if (isAriaElement(component.element)) {
                set$9(component.element, 'aria-invalid', true);
            }
            notifyInfo.getContainer(component).each((container) => {
                // TODO: Should we just use Text here, not HTML?
                set$8(container, text);
            });
            notifyInfo.onInvalid(component, text);
        });
    };
    const query = (component, invalidConfig, _invalidState) => invalidConfig.validator.fold(() => Future.pure(Result.value(true)), (validatorInfo) => validatorInfo.validate(component));
    const run = (component, invalidConfig, invalidState) => {
        invalidConfig.notify.each((notifyInfo) => {
            notifyInfo.onValidate(component);
        });
        return query(component, invalidConfig).map((valid) => {
            if (component.getSystem().isConnected()) {
                return valid.fold((err) => {
                    markInvalid(component, invalidConfig, invalidState, err);
                    return Result.error(err);
                }, (v) => {
                    markValid(component, invalidConfig);
                    return Result.value(v);
                });
            }
            else {
                return Result.error('No longer in system');
            }
        });
    };
    const isInvalid = (component, invalidConfig) => {
        const elem = invalidConfig.getRoot(component).getOr(component.element);
        return has(elem, invalidConfig.invalidClass);
    };

    var InvalidateApis = /*#__PURE__*/Object.freeze({
        __proto__: null,
        markValid: markValid,
        markInvalid: markInvalid,
        query: query,
        run: run,
        isInvalid: isInvalid
    });

    const events$a = (invalidConfig, invalidState) => invalidConfig.validator.map((validatorInfo) => derive$2([
        run$1(validatorInfo.onEvent, (component) => {
            run(component, invalidConfig, invalidState).get(identity);
        })
    ].concat(validatorInfo.validateOnLoad ? [
        runOnAttached((component) => {
            run(component, invalidConfig, invalidState).get(noop);
        })
    ] : []))).getOr({});

    var ActiveInvalidate = /*#__PURE__*/Object.freeze({
        __proto__: null,
        events: events$a
    });

    var InvalidateSchema = [
        required$1('invalidClass'),
        defaulted('getRoot', Optional.none),
        // TODO: Completely rework the notify API
        optionObjOf('notify', [
            defaulted('aria', 'alert'),
            // Maybe we should use something else.
            defaulted('getContainer', Optional.none),
            defaulted('validHtml', ''),
            onHandler('onValid'),
            onHandler('onInvalid'),
            onHandler('onValidate')
        ]),
        optionObjOf('validator', [
            required$1('validate'),
            defaulted('onEvent', 'input'),
            defaulted('validateOnLoad', true)
        ])
    ];

    const onLoad$4 = (component, repConfig, repState) => {
        repConfig.store.manager.onLoad(component, repConfig, repState);
    };
    const onUnload$2 = (component, repConfig, repState) => {
        repConfig.store.manager.onUnload(component, repConfig, repState);
    };
    const setValue$3 = (component, repConfig, repState, data) => {
        repConfig.store.manager.setValue(component, repConfig, repState, data);
    };
    const getValue$3 = (component, repConfig, repState) => repConfig.store.manager.getValue(component, repConfig, repState);
    const getState$2 = (component, repConfig, repState) => repState;

    var RepresentApis = /*#__PURE__*/Object.freeze({
        __proto__: null,
        onLoad: onLoad$4,
        onUnload: onUnload$2,
        setValue: setValue$3,
        getValue: getValue$3,
        getState: getState$2
    });

    const events$9 = (repConfig, repState) => {
        const es = repConfig.resetOnDom ? [
            runOnAttached((comp, _se) => {
                onLoad$4(comp, repConfig, repState);
            }),
            runOnDetached((comp, _se) => {
                onUnload$2(comp, repConfig, repState);
            })
        ] : [
            loadEvent(repConfig, repState, onLoad$4)
        ];
        return derive$2(es);
    };

    var ActiveRepresenting = /*#__PURE__*/Object.freeze({
        __proto__: null,
        events: events$9
    });

    const memory$1 = () => {
        const data = Cell(null);
        const readState = () => ({
            mode: 'memory',
            value: data.get()
        });
        const isNotSet = () => data.get() === null;
        const clear = () => {
            data.set(null);
        };
        return nu$4({
            set: data.set,
            get: data.get,
            isNotSet,
            clear,
            readState
        });
    };
    const manual = () => {
        const readState = noop;
        return nu$4({
            readState
        });
    };
    const dataset = () => {
        const dataByValue = Cell({});
        const dataByText = Cell({});
        const readState = () => ({
            mode: 'dataset',
            dataByValue: dataByValue.get(),
            dataByText: dataByText.get()
        });
        const clear = () => {
            dataByValue.set({});
            dataByText.set({});
        };
        // itemString can be matching value or text.
        // TODO: type problem - impossible to correctly return value when type parameter only exists in return type
        const lookup = (itemString) => get$h(dataByValue.get(), itemString).orThunk(() => get$h(dataByText.get(), itemString));
        const update = (items) => {
            const currentDataByValue = dataByValue.get();
            const currentDataByText = dataByText.get();
            const newDataByValue = {};
            const newDataByText = {};
            each$1(items, (item) => {
                newDataByValue[item.value] = item;
                get$h(item, 'meta').each((meta) => {
                    get$h(meta, 'text').each((text) => {
                        newDataByText[text] = item;
                    });
                });
            });
            dataByValue.set({
                ...currentDataByValue,
                ...newDataByValue
            });
            dataByText.set({
                ...currentDataByText,
                ...newDataByText
            });
        };
        return nu$4({
            readState,
            lookup,
            update,
            clear
        });
    };
    const init$9 = (spec) => spec.store.manager.state(spec);

    var RepresentState = /*#__PURE__*/Object.freeze({
        __proto__: null,
        memory: memory$1,
        dataset: dataset,
        manual: manual,
        init: init$9
    });

    const setValue$2 = (component, repConfig, repState, data) => {
        const store = repConfig.store;
        repState.update([data]);
        store.setValue(component, data);
        repConfig.onSetValue(component, data);
    };
    const getValue$2 = (component, repConfig, repState) => {
        const store = repConfig.store;
        const key = store.getDataKey(component);
        return repState.lookup(key).getOrThunk(() => store.getFallbackEntry(key));
    };
    const onLoad$3 = (component, repConfig, repState) => {
        const store = repConfig.store;
        store.initialValue.each((data) => {
            setValue$2(component, repConfig, repState, data);
        });
    };
    const onUnload$1 = (component, repConfig, repState) => {
        repState.clear();
    };
    var DatasetStore = [
        option$3('initialValue'),
        required$1('getFallbackEntry'),
        required$1('getDataKey'),
        required$1('setValue'),
        output$1('manager', {
            setValue: setValue$2,
            getValue: getValue$2,
            onLoad: onLoad$3,
            onUnload: onUnload$1,
            state: dataset
        })
    ];

    const getValue$1 = (component, repConfig, _repState) => repConfig.store.getValue(component);
    const setValue$1 = (component, repConfig, _repState, data) => {
        repConfig.store.setValue(component, data);
        repConfig.onSetValue(component, data);
    };
    const onLoad$2 = (component, repConfig, _repState) => {
        repConfig.store.initialValue.each((data) => {
            repConfig.store.setValue(component, data);
        });
    };
    var ManualStore = [
        required$1('getValue'),
        defaulted('setValue', noop),
        option$3('initialValue'),
        output$1('manager', {
            setValue: setValue$1,
            getValue: getValue$1,
            onLoad: onLoad$2,
            onUnload: noop,
            state: NoState.init
        })
    ];

    const setValue = (component, repConfig, repState, data) => {
        repState.set(data);
        repConfig.onSetValue(component, data);
    };
    const getValue = (component, repConfig, repState) => repState.get();
    const onLoad$1 = (component, repConfig, repState) => {
        repConfig.store.initialValue.each((initVal) => {
            if (repState.isNotSet()) {
                repState.set(initVal);
            }
        });
    };
    const onUnload = (component, repConfig, repState) => {
        repState.clear();
    };
    var MemoryStore = [
        option$3('initialValue'),
        output$1('manager', {
            setValue,
            getValue,
            onLoad: onLoad$1,
            onUnload,
            state: memory$1
        })
    ];

    var RepresentSchema = [
        defaultedOf('store', { mode: 'memory' }, choose$1('mode', {
            memory: MemoryStore,
            manual: ManualStore,
            dataset: DatasetStore
        })),
        onHandler('onSetValue'),
        defaulted('resetOnDom', false)
    ];

    // The self-reference is clumsy.
    const Representing = create$3({
        fields: RepresentSchema,
        name: 'representing',
        active: ActiveRepresenting,
        apis: RepresentApis,
        extra: {
            setValueFrom: (component, source) => {
                const value = Representing.getValue(source);
                Representing.setValue(component, value);
            }
        },
        state: RepresentState
    });

    const Invalidating = create$3({
        fields: InvalidateSchema,
        name: 'invalidating',
        active: ActiveInvalidate,
        apis: InvalidateApis,
        extra: {
            // Note, this requires representing to be on the validatee
            validation: (validator) => {
                return (component) => {
                    const v = Representing.getValue(component);
                    return Future.pure(validator(v));
                };
            }
        }
    });

    const exhibit$4 = (base, posConfig) => nu$2({
        classes: [],
        styles: posConfig.useFixed() ? {} : { position: 'relative' }
    });

    var ActivePosition = /*#__PURE__*/Object.freeze({
        __proto__: null,
        exhibit: exhibit$4
    });

    const adt$3 = Adt.generate([
        { none: [] },
        { relative: ['x', 'y', 'width', 'height'] },
        { fixed: ['x', 'y', 'width', 'height'] }
    ]);
    const positionWithDirection = (posName, decision, x, y, width, height) => {
        const decisionRect = decision.rect;
        const decisionX = decisionRect.x - x;
        const decisionY = decisionRect.y - y;
        const decisionWidth = decisionRect.width;
        const decisionHeight = decisionRect.height;
        const decisionRight = width - (decisionX + decisionWidth);
        const decisionBottom = height - (decisionY + decisionHeight);
        const left = Optional.some(decisionX);
        const top = Optional.some(decisionY);
        const right = Optional.some(decisionRight);
        const bottom = Optional.some(decisionBottom);
        const none = Optional.none();
        return cata$1(decision.direction, () => NuPositionCss(posName, left, top, none, none), // southeast
        () => NuPositionCss(posName, none, top, right, none), // southwest
        () => NuPositionCss(posName, left, none, none, bottom), // northeast
        () => NuPositionCss(posName, none, none, right, bottom), // northwest
        () => NuPositionCss(posName, left, top, none, none), // south
        () => NuPositionCss(posName, left, none, none, bottom), // north
        () => NuPositionCss(posName, left, top, none, none), // east
        () => NuPositionCss(posName, none, top, right, none) // west
        );
    };
    const reposition = (origin, decision) => origin.fold(() => {
        const decisionRect = decision.rect;
        return NuPositionCss('absolute', Optional.some(decisionRect.x), Optional.some(decisionRect.y), Optional.none(), Optional.none());
    }, (x, y, width, height) => {
        return positionWithDirection('absolute', decision, x, y, width, height);
    }, (x, y, width, height) => {
        return positionWithDirection('fixed', decision, x, y, width, height);
    });
    const toBox = (origin, element) => {
        const rel = curry(find$2, element);
        const position = origin.fold(rel, rel, () => {
            const scroll = get$b();
            // TODO: Make adding the scroll in OuterPosition.find optional.
            return find$2(element).translate(-scroll.left, -scroll.top);
        });
        const width = getOuter(element);
        const height = getOuter$1(element);
        return bounds(position.left, position.top, width, height);
    };
    const viewport = (origin, optBounds) => optBounds.fold(
    /* There are no bounds supplied */
    () => origin.fold(win, win, bounds), (bounds$1) => 
    /* Use any bounds supplied or remove the scroll position of the bounds for fixed. */
    origin.fold(constant$1(bounds$1), constant$1(bounds$1), () => {
        const pos = translate$1(origin, bounds$1.x, bounds$1.y);
        return bounds(pos.left, pos.top, bounds$1.width, bounds$1.height);
    }));
    const translate$1 = (origin, x, y) => {
        const pos = SugarPosition(x, y);
        const removeScroll = () => {
            const outerScroll = get$b();
            return pos.translate(-outerScroll.left, -outerScroll.top);
        };
        // This could use cata if it wasn't a circular reference
        return origin.fold(constant$1(pos), constant$1(pos), removeScroll);
    };
    const cata = (subject, onNone, onRelative, onFixed) => subject.fold(onNone, onRelative, onFixed);
    adt$3.none;
    const relative = adt$3.relative;
    const fixed = adt$3.fixed;

    const anchor = (anchorBox, origin) => ({
        anchorBox,
        origin
    });
    const box = (anchorBox, origin) => anchor(anchorBox, origin);

    const adt$2 = Adt.generate([
        { fit: ['reposition'] },
        { nofit: ['reposition', 'visibleW', 'visibleH', 'isVisible'] }
    ]);
    /**
     * This will attempt to determine if the box will fit within the specified bounds or if it needs to be repositioned.
     * It will return the following details:
     *  - if the original rect was in bounds (originInBounds & sizeInBounds). This is used to determine if we fitted
     *    without having to make adjustments.
     *  - the height and width that would be visible in the original location. (ie the overlap between the rect and
     *    the bounds or the distance between the boxes if there is no overlap)
     */
    const determinePosition = (box, bounds) => {
        const { x: boundsX, y: boundsY, right: boundsRight, bottom: boundsBottom } = bounds;
        const { x, y, right, bottom, width, height } = box;
        // simple checks for "is the top left inside the view"
        const xInBounds = x >= boundsX && x <= boundsRight;
        const yInBounds = y >= boundsY && y <= boundsBottom;
        const originInBounds = xInBounds && yInBounds;
        // simple checks for "is the bottom right inside the view"
        const rightInBounds = right <= boundsRight && right >= boundsX;
        const bottomInBounds = bottom <= boundsBottom && bottom >= boundsY;
        const sizeInBounds = rightInBounds && bottomInBounds;
        // measure how much of the width and height are visible. This should never be larger than the actual width or height
        // however it can be a negative value when offscreen. These values are generally are only needed for the "nofit" case
        const visibleW = Math.min(width, x >= boundsX ? boundsRight - x : right - boundsX);
        const visibleH = Math.min(height, y >= boundsY ? boundsBottom - y : bottom - boundsY);
        return {
            originInBounds,
            sizeInBounds,
            visibleW,
            visibleH
        };
    };
    /**
     * This will attempt to calculate and adjust the position of the box so that is stays within the specified bounds.
     * The end result will be a new restricted box of where it can safely be placed within the bounds.
     */
    const calcReposition = (box, bounds$1) => {
        const { x: boundsX, y: boundsY, right: boundsRight, bottom: boundsBottom } = bounds$1;
        const { x, y, width, height } = box;
        // measure the maximum x and y taking into account the height and width of the box
        const maxX = Math.max(boundsX, boundsRight - width);
        const maxY = Math.max(boundsY, boundsBottom - height);
        // Futz with the X value to ensure that we're not off the left or right of the screen
        const restrictedX = clamp(x, boundsX, maxX);
        // Futz with the Y value to ensure that we're not off the top or bottom of the screen
        const restrictedY = clamp(y, boundsY, maxY);
        // Determine the new height and width based on the restricted X/Y to keep the element in bounds
        const restrictedWidth = Math.min(restrictedX + width, boundsRight) - restrictedX;
        const restrictedHeight = Math.min(restrictedY + height, boundsBottom) - restrictedY;
        return bounds(restrictedX, restrictedY, restrictedWidth, restrictedHeight);
    };
    /**
     * Determine the maximum height and width available for where the box is positioned in the bounds, making sure
     * to account for which direction it's rendering in.
     */
    const calcMaxSizes = (direction, box, bounds) => {
        // Futz with the "height" of the popup to ensure if it doesn't fit it's capped at the available height.
        const upAvailable = constant$1(box.bottom - bounds.y);
        const downAvailable = constant$1(bounds.bottom - box.y);
        const maxHeight = cataVertical(direction, downAvailable, /* middle */ downAvailable, upAvailable);
        // Futz with the "width" of the popup to ensure if it doesn't fit it's capped at the available width.
        const westAvailable = constant$1(box.right - bounds.x);
        const eastAvailable = constant$1(bounds.right - box.x);
        const maxWidth = cataHorizontal(direction, eastAvailable, /* middle */ eastAvailable, westAvailable);
        return {
            maxWidth,
            maxHeight
        };
    };
    const attempt = (candidate, width, height, bounds$1) => {
        const bubble = candidate.bubble;
        const bubbleOffset = bubble.offset;
        // adjust the bounds to account for the layout and bubble restrictions
        const adjustedBounds = adjustBounds(bounds$1, candidate.restriction, bubbleOffset);
        // candidate position is excluding the bubble, so add those values as well
        const newX = candidate.x + bubbleOffset.left;
        const newY = candidate.y + bubbleOffset.top;
        const box = bounds(newX, newY, width, height);
        // determine the position of the box in relation to the bounds
        const { originInBounds, sizeInBounds, visibleW, visibleH } = determinePosition(box, adjustedBounds);
        // restrict the box if it won't fit in the bounds
        const fits = originInBounds && sizeInBounds;
        const fittedBox = fits ? box : calcReposition(box, adjustedBounds);
        // Determine if the box is at least partly visible in the bounds after applying the restrictions
        const isPartlyVisible = fittedBox.width > 0 && fittedBox.height > 0;
        // Determine the maximum height and width available in the bounds
        const { maxWidth, maxHeight } = calcMaxSizes(candidate.direction, fittedBox, bounds$1);
        const reposition = {
            rect: fittedBox,
            maxHeight,
            maxWidth,
            direction: candidate.direction,
            placement: candidate.placement,
            classes: {
                on: bubble.classesOn,
                off: bubble.classesOff
            },
            layout: candidate.label,
            testY: newY
        };
        // useful debugging that I don't want to lose
        // console.log(candidate.label);
        // console.table([{
        //   newY,
        //   limitY: fittedBox.y,
        //   boundsY: bounds.y,
        //   boundsBottom: bounds.bottom,
        //   newX,
        //   limitX: fittedBox.x,
        //   boundsX: bounds.x,
        //   boundsRight: bounds.right,
        //   candidateX: candidate.x,
        //   candidateY: candidate.y,
        //   width,
        //   height,
        //   isPartlyVisible
        // }]);
        // console.log(`maxWidth: ${maxWidth}, visibleW: ${visibleW}`);
        // console.log(`maxHeight: ${maxHeight}, visibleH: ${visibleH}`);
        // console.log('originInBounds:', originInBounds);
        // console.log('sizeInBounds:', sizeInBounds);
        // console.log(originInBounds && sizeInBounds ? 'fit' : 'nofit');
        // Take special note that we don't use the futz values in the nofit case; whether this position is a good fit is separate
        // to ensuring that if we choose it the popup is actually on screen properly.
        return fits || candidate.alwaysFit ? adt$2.fit(reposition) : adt$2.nofit(reposition, visibleW, visibleH, isPartlyVisible);
    };
    /**
     * Attempts to fit a box (generally a menu).
     *
     * candidates: an array of layout generators, generally obtained via api.Layout or api.LinkedLayout
     * anchorBox: the box on screen that triggered the menu, we must touch one of the edges as defined by the candidate layouts
     * elementBox: the popup (only width and height matter)
     * bubbles: the bubbles for the popup (see api.Bubble)
     * bounds: the screen
     */
    const attempts = (element, candidates, anchorBox, elementBox, bubbles, bounds) => {
        const panelWidth = elementBox.width;
        const panelHeight = elementBox.height;
        const attemptBestFit = (layout, reposition, visibleW, visibleH, isVisible) => {
            const next = layout(anchorBox, elementBox, bubbles, element, bounds);
            const attemptLayout = attempt(next, panelWidth, panelHeight, bounds);
            return attemptLayout.fold(constant$1(attemptLayout), (newReposition, newVisibleW, newVisibleH, newIsVisible) => {
                // console.log(`label: ${next.label}, newVisibleW: ${newVisibleW}, visibleW: ${visibleW}, newVisibleH: ${newVisibleH}, visibleH: ${visibleH}, newIsVisible: ${newIsVisible}, isVisible: ${isVisible}`);
                const improved = isVisible === newIsVisible ? (newVisibleH > visibleH || newVisibleW > visibleW) : (!isVisible && newIsVisible);
                // console.log('improved? ', improved);
                return improved ? attemptLayout : adt$2.nofit(reposition, visibleW, visibleH, isVisible);
            });
        };
        const abc = foldl(candidates, (b, a) => {
            const bestNext = curry(attemptBestFit, a);
            return b.fold(constant$1(b), bestNext);
        }, 
        // fold base case: No candidates, it's never going to be correct, so do whatever
        adt$2.nofit({
            rect: anchorBox,
            maxHeight: elementBox.height,
            maxWidth: elementBox.width,
            direction: southeast$3(),
            placement: "southeast" /* Placement.Southeast */,
            classes: {
                on: [],
                off: []
            },
            layout: 'none',
            testY: anchorBox.y
        }, -1, -1, false));
        // unwrapping 'reposition' from the adt, for both fit & nofit the first arg is the one we need,
        // so we can cheat and use Fun.identity
        return abc.fold(identity, identity);
    };

    const properties = ['top', 'bottom', 'right', 'left'];
    const timerAttr = 'data-alloy-transition-timer';
    const isTransitioning$1 = (element, transition) => hasAll(element, transition.classes);
    const shouldApplyTransitionCss = (transition, decision, lastPlacement) => {
        // Don't apply transitions if there was no previous placement as it's transitioning from offscreen
        return lastPlacement.exists((placer) => {
            const mode = transition.mode;
            return mode === 'all' ? true : placer[mode] !== decision[mode];
        });
    };
    const hasChanges = (position, intermediate) => {
        // Round to 3 decimal points
        const round = (value) => parseFloat(value).toFixed(3);
        return find$4(intermediate, (value, key) => {
            const newValue = position[key].map(round);
            const val = value.map(round);
            return !equals(newValue, val);
        }).isSome();
    };
    const getTransitionDuration = (element) => {
        const get = (name) => {
            const style = get$e(element, name);
            const times = style.split(/\s*,\s*/);
            return filter$2(times, isNotEmpty);
        };
        const parse = (value) => {
            if (isString(value) && /^[\d.]+/.test(value)) {
                const num = parseFloat(value);
                return endsWith(value, 'ms') ? num : num * 1000;
            }
            else {
                return 0;
            }
        };
        const delay = get('transition-delay');
        const duration = get('transition-duration');
        return foldl(duration, (acc, dur, i) => {
            const time = parse(delay[i]) + parse(dur);
            return Math.max(acc, time);
        }, 0);
    };
    const setupTransitionListeners = (element, transition) => {
        const transitionEnd = unbindable();
        const transitionCancel = unbindable();
        let timer;
        const isSourceTransition = (e) => {
            // Ensure the transition event isn't from a pseudo element
            const pseudoElement = e.raw.pseudoElement ?? '';
            return eq(e.target, element) && isEmpty(pseudoElement) && contains$2(properties, e.raw.propertyName);
        };
        const transitionDone = (e) => {
            if (isNullable(e) || isSourceTransition(e)) {
                transitionEnd.clear();
                transitionCancel.clear();
                // Only cleanup the class/timer on transitionend not on a cancel. This is done as cancel
                // means the element has been repositioned and would need to keep transitioning
                const type = e?.raw.type;
                if (isNullable(type) || type === transitionend()) {
                    clearTimeout(timer);
                    remove$8(element, timerAttr);
                    remove$2(element, transition.classes);
                }
            }
        };
        const transitionStart = bind$1(element, transitionstart(), (e) => {
            if (isSourceTransition(e)) {
                transitionStart.unbind();
                transitionEnd.set(bind$1(element, transitionend(), transitionDone));
                transitionCancel.set(bind$1(element, transitioncancel(), transitionDone));
            }
        });
        // Request the next animation frame so we can roughly determine when the transition starts and then ensure
        // the transition is cleaned up. In addition add ~17ms to the delay as that's about about 1 frame at 60fps
        const duration = getTransitionDuration(element);
        window.requestAnimationFrame(() => {
            timer = setTimeout(transitionDone, duration + 17);
            set$9(element, timerAttr, timer);
        });
    };
    const startTransitioning = (element, transition) => {
        add$1(element, transition.classes);
        // Clear any existing cleanup timers
        getOpt(element, timerAttr).each((timerId) => {
            clearTimeout(parseInt(timerId, 10));
            remove$8(element, timerAttr);
        });
        setupTransitionListeners(element, transition);
    };
    const applyTransitionCss = (element, origin, position, transition, decision, lastPlacement) => {
        const shouldTransition = shouldApplyTransitionCss(transition, decision, lastPlacement);
        if (shouldTransition || isTransitioning$1(element, transition)) {
            // Set the new position first so we can calculate the computed position
            set$7(element, 'position', position.position);
            // Get the computed positions for the current element based on the new position CSS being applied
            const rect = toBox(origin, element);
            const intermediatePosition = reposition(origin, { ...decision, rect });
            const intermediateCssOptions = mapToObject(properties, (prop) => intermediatePosition[prop]);
            // Apply the intermediate styles and transition classes if something has changed
            if (hasChanges(position, intermediateCssOptions)) {
                setOptions(element, intermediateCssOptions);
                if (shouldTransition) {
                    startTransitioning(element, transition);
                }
                reflow(element);
            }
        }
        else {
            remove$2(element, transition.classes);
        }
    };

    /*
     * This is the old repartee API. It is retained in a similar structure to the original form,
     * in case we decide to bring back the flexibility of working with non-standard positioning.
     */
    const elementSize = (p) => ({
        width: Math.ceil(getOuter(p)),
        height: getOuter$1(p)
    });
    const layout = (anchorBox, element, bubbles, options) => {
        // clear the potentially limiting factors before measuring
        remove$6(element, 'max-height');
        remove$6(element, 'max-width');
        const elementBox = elementSize(element);
        return attempts(element, options.preference, anchorBox, elementBox, bubbles, options.bounds);
    };
    const setClasses = (element, decision) => {
        const classInfo = decision.classes;
        remove$2(element, classInfo.off);
        add$1(element, classInfo.on);
    };
    /*
     * maxHeightFunction is a MaxHeight instance.
     * max-height is usually the distance between the edge of the popup and the screen; top of popup to bottom of screen for south, bottom of popup to top of screen for north.
     *
     * There are a few cases where we specifically don't want a max-height, which is why it's optional.
     */
    const setHeight = (element, decision, options) => {
        // The old API enforced MaxHeight.anchored() for fixed position. That no longer seems necessary.
        const maxHeightFunction = options.maxHeightFunction;
        maxHeightFunction(element, decision.maxHeight);
    };
    const setWidth = (element, decision, options) => {
        const maxWidthFunction = options.maxWidthFunction;
        maxWidthFunction(element, decision.maxWidth);
    };
    const position$2 = (element, decision, options) => {
        // This is a point of difference between Alloy and Repartee. Repartee appears to use Measure to calculate the available space for fixed origin
        // That is not ported yet.
        const positionCss = reposition(options.origin, decision);
        options.transition.each((transition) => {
            applyTransitionCss(element, options.origin, positionCss, transition, decision, options.lastPlacement);
        });
        applyPositionCss(element, positionCss);
    };
    const setPlacement = (element, decision) => {
        setPlacement$1(element, decision.placement);
    };

    const defaultOr = (options, key, dephault) => options[key] === undefined ? dephault : options[key];
    // This takes care of everything when you are positioning UI that can go anywhere on the screen
    const simple = (anchor, element, bubble, layouts, lastPlacement, optBounds, overrideOptions, transition) => {
        // the only supported override at the moment. Once relative has been deleted, maybe this can be optional in the bag
        const maxHeightFunction = defaultOr(overrideOptions, 'maxHeightFunction', anchored());
        const maxWidthFunction = defaultOr(overrideOptions, 'maxWidthFunction', noop);
        const anchorBox = anchor.anchorBox;
        const origin = anchor.origin;
        const options = {
            bounds: viewport(origin, optBounds),
            origin,
            preference: layouts,
            maxHeightFunction,
            maxWidthFunction,
            lastPlacement,
            transition
        };
        return go(anchorBox, element, bubble, options);
    };
    // This is the old public API. If we ever need full customisability again, this is how to expose it
    const go = (anchorBox, element, bubble, options) => {
        const decision = layout(anchorBox, element, bubble, options);
        position$2(element, decision, options);
        setPlacement(element, decision);
        setClasses(element, decision);
        setHeight(element, decision, options);
        setWidth(element, decision, options);
        return {
            layout: decision.layout,
            placement: decision.placement
        };
    };

    const nu$1 = identity;

    const schema$n = () => optionObjOf('layouts', [
        required$1('onLtr'),
        required$1('onRtl'),
        option$3('onBottomLtr'),
        option$3('onBottomRtl')
    ]);
    const get$1 = (elem, info, defaultLtr, defaultRtl, defaultBottomLtr, defaultBottomRtl, dirElement) => {
        const isBottomToTop = dirElement.map(isBottomToTopDir).getOr(false);
        const customLtr = info.layouts.map((ls) => ls.onLtr(elem));
        const customRtl = info.layouts.map((ls) => ls.onRtl(elem));
        const ltr = isBottomToTop ?
            info.layouts.bind((ls) => ls.onBottomLtr.map((f) => f(elem)))
                .or(customLtr)
                .getOr(defaultBottomLtr) :
            customLtr.getOr(defaultLtr);
        const rtl = isBottomToTop ?
            info.layouts.bind((ls) => ls.onBottomRtl.map((f) => f(elem)))
                .or(customRtl)
                .getOr(defaultBottomRtl) :
            customRtl.getOr(defaultRtl);
        const f = onDirection(ltr, rtl);
        return f(elem);
    };

    const placement$4 = (component, anchorInfo, origin) => {
        const hotspot = anchorInfo.hotspot;
        const anchorBox = toBox(origin, hotspot.element);
        const layouts = get$1(component.element, anchorInfo, belowOrAbove(), belowOrAboveRtl(), aboveOrBelow(), aboveOrBelowRtl(), Optional.some(anchorInfo.hotspot.element));
        return Optional.some(nu$1({
            anchorBox,
            bubble: anchorInfo.bubble.getOr(fallback()),
            overrides: anchorInfo.overrides,
            layouts
        }));
    };
    var HotspotAnchor = [
        required$1('hotspot'),
        option$3('bubble'),
        defaulted('overrides', {}),
        schema$n(),
        output$1('placement', placement$4)
    ];

    const placement$3 = (component, anchorInfo, origin) => {
        const pos = translate$1(origin, anchorInfo.x, anchorInfo.y);
        const anchorBox = bounds(pos.left, pos.top, anchorInfo.width, anchorInfo.height);
        const layouts = get$1(component.element, anchorInfo, all$2(), allRtl$1(), 
        // No default bottomToTop layouts currently needed
        all$2(), allRtl$1(), Optional.none());
        return Optional.some(nu$1({
            anchorBox,
            bubble: anchorInfo.bubble,
            overrides: anchorInfo.overrides,
            layouts
        }));
    };
    var MakeshiftAnchor = [
        required$1('x'),
        required$1('y'),
        defaulted('height', 0),
        defaulted('width', 0),
        defaulted('bubble', fallback()),
        defaulted('overrides', {}),
        schema$n(),
        output$1('placement', placement$3)
    ];

    const adt$1 = Adt.generate([
        { screen: ['point'] },
        { absolute: ['point', 'scrollLeft', 'scrollTop'] }
    ]);
    const toFixed = (pos) => 
    // TODO: Use new ADT methods
    pos.fold(identity, (point, scrollLeft, scrollTop) => point.translate(-scrollLeft, -scrollTop));
    const toAbsolute = (pos) => pos.fold(identity, identity);
    const sum = (points) => foldl(points, (b, a) => b.translate(a.left, a.top), SugarPosition(0, 0));
    const sumAsFixed = (positions) => {
        const points = map$2(positions, toFixed);
        return sum(points);
    };
    const sumAsAbsolute = (positions) => {
        const points = map$2(positions, toAbsolute);
        return sum(points);
    };
    const screen = adt$1.screen;
    const absolute = adt$1.absolute;

    // In one mode, the window is inside an iframe. If that iframe is in the
    // same document as the positioning element (component), then identify the offset
    // difference between the iframe and the component.
    const getOffset = (component, origin, anchorInfo) => {
        const win = defaultView(anchorInfo.root).dom;
        const hasSameOwner = (frame) => {
            const frameOwner = owner$4(frame);
            const compOwner = owner$4(component.element);
            return eq(frameOwner, compOwner);
        };
        return Optional.from(win.frameElement).map(SugarElement.fromDom)
            .filter(hasSameOwner).map(absolute$3);
    };
    const getRootPoint = (component, origin, anchorInfo) => {
        const doc = owner$4(component.element);
        const outerScroll = get$b(doc);
        const offset = getOffset(component, origin, anchorInfo).getOr(outerScroll);
        return absolute(offset, outerScroll.left, outerScroll.top);
    };

    const getBox = (left, top, width, height) => {
        const point = screen(SugarPosition(left, top));
        return Optional.some(pointed(point, width, height));
    };
    const calcNewAnchor = (optBox, rootPoint, anchorInfo, origin, elem) => optBox.map((box) => {
        const points = [rootPoint, box.point];
        const topLeft = cata(origin, () => sumAsAbsolute(points), () => sumAsAbsolute(points), () => sumAsFixed(points));
        const anchorBox = rect(topLeft.left, topLeft.top, box.width, box.height);
        const layoutsLtr = anchorInfo.showAbove ?
            aboveOrBelow() :
            belowOrAbove();
        const layoutsRtl = anchorInfo.showAbove ?
            aboveOrBelowRtl() :
            belowOrAboveRtl();
        const layouts = get$1(elem, anchorInfo, layoutsLtr, layoutsRtl, layoutsLtr, layoutsRtl, Optional.none());
        return nu$1({
            anchorBox,
            bubble: anchorInfo.bubble.getOr(fallback()),
            overrides: anchorInfo.overrides,
            layouts
        });
    });

    const placement$2 = (component, anchorInfo, origin) => {
        const rootPoint = getRootPoint(component, origin, anchorInfo);
        return anchorInfo.node
            // Ensure the node is still attached, otherwise we can't get a valid client rect for a detached node
            .filter(inBody)
            .bind((target) => {
            const rect = target.dom.getBoundingClientRect();
            const nodeBox = getBox(rect.left, rect.top, rect.width, rect.height);
            const elem = anchorInfo.node.getOr(component.element);
            return calcNewAnchor(nodeBox, rootPoint, anchorInfo, origin, elem);
        });
    };
    var NodeAnchor = [
        required$1('node'),
        required$1('root'),
        option$3('bubble'),
        schema$n(),
        // chiefly MaxHeight.expandable()
        defaulted('overrides', {}),
        defaulted('showAbove', false),
        output$1('placement', placement$2)
    ];

    const point = (element, offset) => ({
        element,
        offset
    });
    // NOTE: This only descends once.
    const descendOnce$1 = (element, offset) => {
        const children$1 = children(element);
        if (children$1.length === 0) {
            return point(element, offset);
        }
        else if (offset < children$1.length) {
            return point(children$1[offset], 0);
        }
        else {
            const last = children$1[children$1.length - 1];
            const len = isText(last) ? get$a(last).length : children(last).length;
            return point(last, len);
        }
    };

    // A range from (a, 1) to (body, end) was giving the wrong bounds.
    const descendOnce = (element, offset) => isText(element) ? point(element, offset) : descendOnce$1(element, offset);
    const isSimRange = (detail) => detail.foffset !== undefined;
    const getAnchorSelection = (win, anchorInfo) => {
        // FIX TEST Test both providing a getSelection and not providing a getSelection
        const getSelection = anchorInfo.getSelection.getOrThunk(() => () => getExact(win));
        return getSelection().map((sel) => {
            if (isSimRange(sel)) {
                const modStart = descendOnce(sel.start, sel.soffset);
                const modFinish = descendOnce(sel.finish, sel.foffset);
                return SimSelection.range(modStart.element, modStart.offset, modFinish.element, modFinish.offset);
            }
            else {
                return sel;
            }
        });
    };
    const placement$1 = (component, anchorInfo, origin) => {
        const win = defaultView(anchorInfo.root).dom;
        const rootPoint = getRootPoint(component, origin, anchorInfo);
        const selectionBox = getAnchorSelection(win, anchorInfo).bind((sel) => {
            // This represents the *visual* rectangle of the selection.
            if (isSimRange(sel)) {
                const optRect = getBounds$2(win, SimSelection.exactFromRange(sel)).orThunk(() => {
                    const zeroWidth$1 = SugarElement.fromText(zeroWidth);
                    before$1(sel.start, zeroWidth$1);
                    // Certain things like <p><br/></p> with (p, 0) or <br>) as collapsed selection do not return a client rectangle
                    const rect = getFirstRect(win, SimSelection.exact(zeroWidth$1, 0, zeroWidth$1, 1));
                    remove$7(zeroWidth$1);
                    return rect;
                });
                return optRect.bind((rawRect) => {
                    return getBox(rawRect.left, rawRect.top, rawRect.width, rawRect.height);
                });
            }
            else {
                const selectionRect = map$1(sel, (cell) => cell.dom.getBoundingClientRect());
                const bounds = {
                    left: Math.min(selectionRect.firstCell.left, selectionRect.lastCell.left),
                    right: Math.max(selectionRect.firstCell.right, selectionRect.lastCell.right),
                    top: Math.min(selectionRect.firstCell.top, selectionRect.lastCell.top),
                    bottom: Math.max(selectionRect.firstCell.bottom, selectionRect.lastCell.bottom)
                };
                return getBox(bounds.left, bounds.top, bounds.right - bounds.left, bounds.bottom - bounds.top);
            }
        });
        const targetElement = getAnchorSelection(win, anchorInfo)
            .bind((sel) => {
            if (isSimRange(sel)) {
                return isElement$1(sel.start) ? Optional.some(sel.start) : parentElement(sel.start);
            }
            else {
                return Optional.some(sel.firstCell);
            }
        });
        const elem = targetElement.getOr(component.element);
        return calcNewAnchor(selectionBox, rootPoint, anchorInfo, origin, elem);
    };
    var SelectionAnchor = [
        option$3('getSelection'),
        required$1('root'),
        option$3('bubble'),
        schema$n(),
        defaulted('overrides', {}),
        defaulted('showAbove', false),
        output$1('placement', placement$1)
    ];

    /*
      Layout for submenus;
      Either left or right of the anchor menu item. Never above or below.
      Aligned to the top or bottom of the anchor as appropriate.
     */
    const labelPrefix = 'link-layout';
    // display element to the right, left edge against the right of the menu
    const eastX = (anchor) => anchor.x + anchor.width;
    // display element to the left, right edge against the left of the menu
    const westX = (anchor, element) => anchor.x - element.width;
    // display element pointing up, bottom edge against the bottom of the menu (usually to one side)
    const northY = (anchor, element) => anchor.y - element.height + anchor.height;
    // display element pointing down, top edge against the top of the menu (usually to one side)
    const southY = (anchor) => anchor.y;
    const southeast = (anchor, element, bubbles) => nu$5(eastX(anchor), southY(anchor), bubbles.southeast(), southeast$3(), "southeast" /* Placement.Southeast */, boundsRestriction(anchor, { left: 0 /* AnchorBoxBounds.RightEdge */, top: 2 /* AnchorBoxBounds.TopEdge */ }), labelPrefix);
    const southwest = (anchor, element, bubbles) => nu$5(westX(anchor, element), southY(anchor), bubbles.southwest(), southwest$3(), "southwest" /* Placement.Southwest */, boundsRestriction(anchor, { right: 1 /* AnchorBoxBounds.LeftEdge */, top: 2 /* AnchorBoxBounds.TopEdge */ }), labelPrefix);
    const northeast = (anchor, element, bubbles) => nu$5(eastX(anchor), northY(anchor, element), bubbles.northeast(), northeast$3(), "northeast" /* Placement.Northeast */, boundsRestriction(anchor, { left: 0 /* AnchorBoxBounds.RightEdge */, bottom: 3 /* AnchorBoxBounds.BottomEdge */ }), labelPrefix);
    const northwest = (anchor, element, bubbles) => nu$5(westX(anchor, element), northY(anchor, element), bubbles.northwest(), northwest$3(), "northwest" /* Placement.Northwest */, boundsRestriction(anchor, { right: 1 /* AnchorBoxBounds.LeftEdge */, bottom: 3 /* AnchorBoxBounds.BottomEdge */ }), labelPrefix);
    const all = () => [southeast, southwest, northeast, northwest];
    const allRtl = () => [southwest, southeast, northwest, northeast];

    const placement = (component, submenuInfo, origin) => {
        const anchorBox = toBox(origin, submenuInfo.item.element);
        const layouts = get$1(component.element, submenuInfo, all(), allRtl(), 
        // No default bottomToTop layouts currently needed
        all(), allRtl(), Optional.none());
        return Optional.some(nu$1({
            anchorBox,
            bubble: fallback(),
            overrides: submenuInfo.overrides,
            layouts
        }));
    };
    var SubmenuAnchor = [
        required$1('item'),
        schema$n(),
        defaulted('overrides', {}),
        output$1('placement', placement)
    ];

    var AnchorSchema = choose$1('type', {
        selection: SelectionAnchor,
        node: NodeAnchor,
        hotspot: HotspotAnchor,
        submenu: SubmenuAnchor,
        makeshift: MakeshiftAnchor
    });

    const TransitionSchema = [
        requiredArrayOf('classes', string),
        defaultedStringEnum('mode', 'all', ['all', 'layout', 'placement'])
    ];
    const PositionSchema = [
        defaulted('useFixed', never),
        option$3('getBounds')
    ];
    const PlacementSchema = [
        requiredOf('anchor', AnchorSchema),
        optionObjOf('transition', TransitionSchema)
    ];

    const getFixedOrigin = () => {
        // Don't use window.innerWidth/innerHeight here, as we don't want to include scrollbars
        // since the right/bottom position is based on the edge of the scrollbar not the window
        const html = document.documentElement;
        return fixed(0, 0, html.clientWidth, html.clientHeight);
    };
    const getRelativeOrigin = (component) => {
        const position = absolute$3(component.element);
        const bounds = component.element.dom.getBoundingClientRect();
        // We think that this just needs to be kept consistent with Boxes.win. If we remove the scroll values from Boxes.win, we
        // should change this to just bounds.left and bounds.top from getBoundingClientRect
        return relative(position.left, position.top, bounds.width, bounds.height);
    };
    const place = (origin, anchoring, optBounds, placee, lastPlace, transition) => {
        const anchor = box(anchoring.anchorBox, origin);
        return simple(anchor, placee.element, anchoring.bubble, anchoring.layouts, lastPlace, optBounds, anchoring.overrides, transition);
    };
    const position$1 = (component, posConfig, posState, placee, placementSpec) => {
        const optWithinBounds = Optional.none();
        positionWithinBounds(component, posConfig, posState, placee, placementSpec, optWithinBounds);
    };
    const positionWithinBounds = (component, posConfig, posState, placee, placementSpec, optWithinBounds) => {
        const placeeDetail = asRawOrDie$1('placement.info', objOf(PlacementSchema), placementSpec);
        const anchorage = placeeDetail.anchor;
        const element = placee.element;
        const placeeState = posState.get(placee.uid);
        // Preserve the focus as IE 11 loses it when setting visibility to hidden
        preserve(() => {
            // We set it to be fixed, so that it doesn't interfere with the layout of anything
            // when calculating anchors
            set$7(element, 'position', 'fixed');
            const oldVisibility = getRaw(element, 'visibility');
            set$7(element, 'visibility', 'hidden');
            // We need to calculate the origin (esp. the bounding client rect) *after* we have done
            // all the preprocessing of the component and placee. Otherwise, the relative positions
            // (bottom and right) will be using the wrong dimensions
            const origin = posConfig.useFixed() ? getFixedOrigin() : getRelativeOrigin(component);
            anchorage.placement(component, anchorage, origin).each((anchoring) => {
                // If "within bounds" is specified, it overrides any Positioning config. Otherwise, we
                // use the Positioning config. We don't try to combine automatically here because they are
                // sometimes serving different purposes. If the Positioning config getBounds needs to be
                // combined with the optWithinBounds bounds, then it is the responsibility of the calling
                // code to combine them, and pass in the combined value as optWithinBounds. The optWithinBounds
                // will *always* override the Positioning config.
                const optBounds = optWithinBounds.orThunk(() => posConfig.getBounds.map(apply$1));
                // Place the element and then update the state for the placee
                const newState = place(origin, anchoring, optBounds, placee, placeeState, placeeDetail.transition);
                posState.set(placee.uid, newState);
            });
            oldVisibility.fold(() => {
                remove$6(element, 'visibility');
            }, (vis) => {
                set$7(element, 'visibility', vis);
            });
            // We need to remove position: fixed put on by above code if it is not needed.
            if (getRaw(element, 'left').isNone() &&
                getRaw(element, 'top').isNone() &&
                getRaw(element, 'right').isNone() &&
                getRaw(element, 'bottom').isNone() &&
                is$1(getRaw(element, 'position'), 'fixed')) {
                remove$6(element, 'position');
            }
        }, element);
    };
    const getMode = (component, pConfig, _pState) => pConfig.useFixed() ? 'fixed' : 'absolute';
    const reset = (component, pConfig, posState, placee) => {
        const element = placee.element;
        each$1(['position', 'left', 'right', 'top', 'bottom'], (prop) => remove$6(element, prop));
        reset$2(element);
        posState.clear(placee.uid);
    };

    var PositionApis = /*#__PURE__*/Object.freeze({
        __proto__: null,
        position: position$1,
        positionWithinBounds: positionWithinBounds,
        getMode: getMode,
        reset: reset
    });

    const init$8 = () => {
        let state = {};
        const set = (id, data) => {
            state[id] = data;
        };
        const get = (id) => get$h(state, id);
        const clear = (id) => {
            if (isNonNullable(id)) {
                delete state[id];
            }
            else {
                state = {};
            }
        };
        return nu$4({
            readState: () => state,
            clear,
            set,
            get
        });
    };

    var PositioningState = /*#__PURE__*/Object.freeze({
        __proto__: null,
        init: init$8
    });

    const Positioning = create$3({
        fields: PositionSchema,
        name: 'positioning',
        active: ActivePosition,
        apis: PositionApis,
        state: PositioningState
    });

    const chooseChannels = (channels, message) => message.universal ? channels : filter$2(channels, (ch) => contains$2(message.channels, ch));
    const events$8 = (receiveConfig) => derive$2([
        run$1(receive(), (component, message) => {
            const channelMap = receiveConfig.channels;
            const channels = keys(channelMap);
            // NOTE: Receiving event ignores the whole simulated event part.
            // TODO: Think about the types for this, or find a better way for this to rely on receiving.
            const receivingData = message;
            const targetChannels = chooseChannels(channels, receivingData);
            each$1(targetChannels, (ch) => {
                const channelInfo = channelMap[ch];
                const channelSchema = channelInfo.schema;
                const data = asRawOrDie$1('channel[' + ch + '] data\nReceiver: ' + element(component.element), channelSchema, receivingData.data);
                channelInfo.onReceive(component, data);
            });
        })
    ]);

    var ActiveReceiving = /*#__PURE__*/Object.freeze({
        __proto__: null,
        events: events$8
    });

    var ReceivingSchema = [
        requiredOf('channels', setOf(
        // Allow any keys.
        Result.value, objOfOnly([
            onStrictHandler('onReceive'),
            defaulted('schema', anyValue())
        ])))
    ];

    const Receiving = create$3({
        fields: ReceivingSchema,
        name: 'receiving',
        active: ActiveReceiving
    });

    const events$7 = (reflectingConfig, reflectingState) => {
        const update = (component, data) => {
            reflectingConfig.updateState.each((updateState) => {
                const newState = updateState(component, data);
                reflectingState.set(newState);
            });
            // FIX: Partial duplication of Replacing + Receiving
            reflectingConfig.renderComponents.each((renderComponents) => {
                const newComponents = renderComponents(data, reflectingState.get());
                const replacer = reflectingConfig.reuseDom ? withReuse : withoutReuse;
                replacer(component, newComponents);
            });
        };
        return derive$2([
            run$1(receive(), (component, message) => {
                // NOTE: Receiving event ignores the whole simulated event part.
                // TODO: Think about the types for this, or find a better way for this to rely on receiving.
                const receivingData = message;
                if (!receivingData.universal) {
                    const channel = reflectingConfig.channel;
                    if (contains$2(receivingData.channels, channel)) {
                        update(component, receivingData.data);
                    }
                }
            }),
            runOnAttached((comp, _se) => {
                reflectingConfig.initialData.each((rawData) => {
                    update(comp, rawData);
                });
            })
        ]);
    };

    var ActiveReflecting = /*#__PURE__*/Object.freeze({
        __proto__: null,
        events: events$7
    });

    const getState$1 = (component, replaceConfig, reflectState) => reflectState;

    var ReflectingApis = /*#__PURE__*/Object.freeze({
        __proto__: null,
        getState: getState$1
    });

    var ReflectingSchema = [
        required$1('channel'),
        option$3('renderComponents'),
        option$3('updateState'),
        option$3('initialData'),
        defaultedBoolean('reuseDom', true)
    ];

    const init$7 = () => {
        const cell = Cell(Optional.none());
        const clear = () => cell.set(Optional.none());
        const readState = () => cell.get().getOr('none');
        return {
            readState,
            get: cell.get,
            set: cell.set,
            clear
        };
    };

    var ReflectingState = /*#__PURE__*/Object.freeze({
        __proto__: null,
        init: init$7
    });

    const Reflecting = create$3({
        fields: ReflectingSchema,
        name: 'reflecting',
        active: ActiveReflecting,
        apis: ReflectingApis,
        state: ReflectingState
    });

    // NOTE: A sandbox should not start as part of the world. It is expected to be
    // added to the sink on rebuild.
    const rebuild = (sandbox, sConfig, sState, data) => {
        sState.get().each((_data) => {
            // If currently has data, so it hasn't been removed yet. It is
            // being "re-opened"
            detachChildren(sandbox);
        });
        const point = sConfig.getAttachPoint(sandbox);
        attach(point, sandbox);
        // Must be after the sandbox is in the system
        const built = sandbox.getSystem().build(data);
        attach(sandbox, built);
        sState.set(built);
        return built;
    };
    // Open sandbox transfers focus to the opened menu
    const open$1 = (sandbox, sConfig, sState, data) => {
        const newState = rebuild(sandbox, sConfig, sState, data);
        sConfig.onOpen(sandbox, newState);
        return newState;
    };
    const setContent = (sandbox, sConfig, sState, data) => sState.get().map(() => rebuild(sandbox, sConfig, sState, data));
    // TODO AP-191 write a test for openWhileCloaked
    const openWhileCloaked = (sandbox, sConfig, sState, data, transaction) => {
        cloak(sandbox, sConfig);
        open$1(sandbox, sConfig, sState, data);
        transaction();
        decloak(sandbox, sConfig);
    };
    const close$1 = (sandbox, sConfig, sState) => {
        sState.get().each((data) => {
            detachChildren(sandbox);
            detach(sandbox);
            sConfig.onClose(sandbox, data);
            sState.clear();
        });
    };
    const isOpen$1 = (_sandbox, _sConfig, sState) => sState.isOpen();
    const isPartOf$1 = (sandbox, sConfig, sState, queryElem) => isOpen$1(sandbox, sConfig, sState) && sState.get().exists((data) => sConfig.isPartOf(sandbox, data, queryElem));
    const getState = (_sandbox, _sConfig, sState) => sState.get();
    const store = (sandbox, cssKey, attr, newValue) => {
        getRaw(sandbox.element, cssKey).fold(() => {
            remove$8(sandbox.element, attr);
        }, (v) => {
            set$9(sandbox.element, attr, v);
        });
        set$7(sandbox.element, cssKey, newValue);
    };
    const restore = (sandbox, cssKey, attr) => {
        getOpt(sandbox.element, attr).fold(() => remove$6(sandbox.element, cssKey), (oldValue) => set$7(sandbox.element, cssKey, oldValue));
    };
    const cloak = (sandbox, sConfig, _sState) => {
        const sink = sConfig.getAttachPoint(sandbox);
        // Use the positioning mode of the sink, so that it does not interfere with the sink's positioning
        // We add it here to stop it causing layout problems.
        set$7(sandbox.element, 'position', Positioning.getMode(sink));
        store(sandbox, 'visibility', sConfig.cloakVisibilityAttr, 'hidden');
    };
    const hasPosition = (element) => exists(['top', 'left', 'right', 'bottom'], (pos) => getRaw(element, pos).isSome());
    const decloak = (sandbox, sConfig, _sState) => {
        if (!hasPosition(sandbox.element)) {
            // If a position value was not added to the sandbox during cloaking, remove it
            // otherwise certain position values (absolute, relative) will impact the child that _was_ positioned
            remove$6(sandbox.element, 'position');
        }
        restore(sandbox, 'visibility', sConfig.cloakVisibilityAttr);
    };

    var SandboxApis = /*#__PURE__*/Object.freeze({
        __proto__: null,
        cloak: cloak,
        decloak: decloak,
        open: open$1,
        openWhileCloaked: openWhileCloaked,
        close: close$1,
        isOpen: isOpen$1,
        isPartOf: isPartOf$1,
        getState: getState,
        setContent: setContent
    });

    const events$6 = (sandboxConfig, sandboxState) => derive$2([
        run$1(sandboxClose(), (sandbox, _simulatedEvent) => {
            close$1(sandbox, sandboxConfig, sandboxState);
        })
    ]);

    var ActiveSandbox = /*#__PURE__*/Object.freeze({
        __proto__: null,
        events: events$6
    });

    var SandboxSchema = [
        onHandler('onOpen'),
        onHandler('onClose'),
        // Maybe this should be optional
        required$1('isPartOf'),
        required$1('getAttachPoint'),
        defaulted('cloakVisibilityAttr', 'data-precloak-visibility')
    ];

    const init$6 = () => {
        const contents = value$2();
        const readState = constant$1('not-implemented');
        return nu$4({
            readState,
            isOpen: contents.isSet,
            clear: contents.clear,
            set: contents.set,
            get: contents.get
        });
    };

    var SandboxState = /*#__PURE__*/Object.freeze({
        __proto__: null,
        init: init$6
    });

    const Sandboxing = create$3({
        fields: SandboxSchema,
        name: 'sandboxing',
        active: ActiveSandbox,
        apis: SandboxApis,
        state: SandboxState
    });

    const getAnimationRoot = (component, slideConfig) => slideConfig.getAnimationRoot.fold(() => component.element, (get) => get(component));

    const getDimensionProperty = (slideConfig) => slideConfig.dimension.property;
    const getDimension = (slideConfig, elem) => slideConfig.dimension.getDimension(elem);
    const disableTransitions = (component, slideConfig) => {
        const root = getAnimationRoot(component, slideConfig);
        remove$2(root, [slideConfig.shrinkingClass, slideConfig.growingClass]);
    };
    const setShrunk = (component, slideConfig) => {
        remove$3(component.element, slideConfig.openClass);
        add$2(component.element, slideConfig.closedClass);
        set$7(component.element, getDimensionProperty(slideConfig), '0px');
        reflow(component.element);
    };
    const setGrown = (component, slideConfig) => {
        remove$3(component.element, slideConfig.closedClass);
        add$2(component.element, slideConfig.openClass);
        remove$6(component.element, getDimensionProperty(slideConfig));
    };
    const doImmediateShrink = (component, slideConfig, slideState, _calculatedSize) => {
        slideState.setCollapsed();
        // Force current dimension to begin transition
        set$7(component.element, getDimensionProperty(slideConfig), getDimension(slideConfig, component.element));
        // TINY-8710: we don't think reflow is required (as has been done elsewhere) as the animation is not needed
        disableTransitions(component, slideConfig);
        setShrunk(component, slideConfig);
        slideConfig.onStartShrink(component);
        slideConfig.onShrunk(component);
    };
    const doStartShrink = (component, slideConfig, slideState, calculatedSize) => {
        const size = calculatedSize.getOrThunk(() => getDimension(slideConfig, component.element));
        slideState.setCollapsed();
        // Force current dimension to begin transition
        set$7(component.element, getDimensionProperty(slideConfig), size);
        reflow(component.element);
        const root = getAnimationRoot(component, slideConfig);
        remove$3(root, slideConfig.growingClass);
        add$2(root, slideConfig.shrinkingClass); // enable transitions
        setShrunk(component, slideConfig);
        slideConfig.onStartShrink(component);
    };
    // A "smartShrink" will do an immediate shrink if no shrinking is scheduled to happen
    const doStartSmartShrink = (component, slideConfig, slideState) => {
        const size = getDimension(slideConfig, component.element);
        const shrinker = size === '0px' ? doImmediateShrink : doStartShrink;
        shrinker(component, slideConfig, slideState, Optional.some(size));
    };
    // Showing is complex due to the inability to transition to "auto".
    // We also can't cache the dimension as the parents may have resized since it was last shown.
    const doStartGrow = (component, slideConfig, slideState) => {
        // Start the growing animation styles
        const root = getAnimationRoot(component, slideConfig);
        // Record whether this is interrupting a shrink and its current size
        const wasShrinking = has(root, slideConfig.shrinkingClass);
        const beforeSize = getDimension(slideConfig, component.element);
        setGrown(component, slideConfig);
        const fullSize = getDimension(slideConfig, component.element);
        // If the grow is interrupting a shrink, use the size from before the grow as the start size
        // And reflow so that the animation works.
        const startPartialGrow = () => {
            set$7(component.element, getDimensionProperty(slideConfig), beforeSize);
            reflow(component.element);
        };
        // If the grow is not interrupting a shrink, start from 0 (shrunk)
        const startCompleteGrow = () => {
            setShrunk(component, slideConfig);
        };
        // Determine what the initial size for the grow operation should be.
        const setStartSize = wasShrinking ? startPartialGrow : startCompleteGrow;
        setStartSize();
        remove$3(root, slideConfig.shrinkingClass);
        add$2(root, slideConfig.growingClass);
        setGrown(component, slideConfig);
        set$7(component.element, getDimensionProperty(slideConfig), fullSize);
        slideState.setExpanded();
        slideConfig.onStartGrow(component);
    };
    const refresh$3 = (component, slideConfig, slideState) => {
        if (slideState.isExpanded()) {
            remove$6(component.element, getDimensionProperty(slideConfig));
            const fullSize = getDimension(slideConfig, component.element);
            set$7(component.element, getDimensionProperty(slideConfig), fullSize);
        }
    };
    const grow = (component, slideConfig, slideState) => {
        if (!slideState.isExpanded()) {
            doStartGrow(component, slideConfig, slideState);
        }
    };
    const shrink = (component, slideConfig, slideState) => {
        if (slideState.isExpanded()) {
            doStartSmartShrink(component, slideConfig, slideState);
        }
    };
    const immediateShrink = (component, slideConfig, slideState) => {
        if (slideState.isExpanded()) {
            doImmediateShrink(component, slideConfig, slideState);
        }
    };
    const hasGrown = (component, slideConfig, slideState) => slideState.isExpanded();
    const hasShrunk = (component, slideConfig, slideState) => slideState.isCollapsed();
    const isGrowing = (component, slideConfig, _slideState) => {
        const root = getAnimationRoot(component, slideConfig);
        return has(root, slideConfig.growingClass) === true;
    };
    const isShrinking = (component, slideConfig, _slideState) => {
        const root = getAnimationRoot(component, slideConfig);
        return has(root, slideConfig.shrinkingClass) === true;
    };
    const isTransitioning = (component, slideConfig, slideState) => isGrowing(component, slideConfig) || isShrinking(component, slideConfig);
    const toggleGrow = (component, slideConfig, slideState) => {
        const f = slideState.isExpanded() ? doStartSmartShrink : doStartGrow;
        f(component, slideConfig, slideState);
    };
    const immediateGrow = (component, slideConfig, slideState) => {
        if (!slideState.isExpanded()) {
            setGrown(component, slideConfig);
            set$7(component.element, getDimensionProperty(slideConfig), getDimension(slideConfig, component.element));
            // TINY-8710: we don't think reflow is required (as has been done elsewhere) as the animation is not needed
            // Keep disableTransition to handle the case where it's part way through transitioning
            disableTransitions(component, slideConfig);
            slideState.setExpanded();
            slideConfig.onStartGrow(component);
            slideConfig.onGrown(component);
        }
    };

    var SlidingApis = /*#__PURE__*/Object.freeze({
        __proto__: null,
        refresh: refresh$3,
        grow: grow,
        shrink: shrink,
        immediateShrink: immediateShrink,
        hasGrown: hasGrown,
        hasShrunk: hasShrunk,
        isGrowing: isGrowing,
        isShrinking: isShrinking,
        isTransitioning: isTransitioning,
        toggleGrow: toggleGrow,
        disableTransitions: disableTransitions,
        immediateGrow: immediateGrow
    });

    const exhibit$3 = (base, slideConfig, _slideState) => {
        const expanded = slideConfig.expanded;
        return expanded ? nu$2({
            classes: [slideConfig.openClass],
            styles: {}
        }) : nu$2({
            classes: [slideConfig.closedClass],
            styles: wrap(slideConfig.dimension.property, '0px')
        });
    };
    const events$5 = (slideConfig, slideState) => derive$2([
        runOnSource(transitionend(), (component, simulatedEvent) => {
            const raw = simulatedEvent.event.raw;
            // This will fire for all transitions, we're only interested in the dimension completion on source
            if (raw.propertyName === slideConfig.dimension.property) {
                disableTransitions(component, slideConfig); // disable transitions immediately (Safari animates the dimension removal below)
                if (slideState.isExpanded()) {
                    remove$6(component.element, slideConfig.dimension.property);
                } // when showing, remove the dimension so it is responsive
                const notify = slideState.isExpanded() ? slideConfig.onGrown : slideConfig.onShrunk;
                notify(component);
            }
        })
    ]);

    var ActiveSliding = /*#__PURE__*/Object.freeze({
        __proto__: null,
        exhibit: exhibit$3,
        events: events$5
    });

    var SlidingSchema = [
        required$1('closedClass'),
        required$1('openClass'),
        required$1('shrinkingClass'),
        required$1('growingClass'),
        // Element which shrinking and growing animations
        option$3('getAnimationRoot'),
        onHandler('onShrunk'),
        onHandler('onStartShrink'),
        onHandler('onGrown'),
        onHandler('onStartGrow'),
        defaulted('expanded', false),
        requiredOf('dimension', choose$1('property', {
            width: [
                output$1('property', 'width'),
                output$1('getDimension', (elem) => get$c(elem) + 'px')
            ],
            height: [
                output$1('property', 'height'),
                output$1('getDimension', (elem) => get$d(elem) + 'px')
            ]
        }))
    ];

    const init$5 = (spec) => {
        const state = Cell(spec.expanded);
        const readState = () => 'expanded: ' + state.get();
        return nu$4({
            isExpanded: () => state.get() === true,
            isCollapsed: () => state.get() === false,
            setCollapsed: curry(state.set, false),
            setExpanded: curry(state.set, true),
            readState
        });
    };

    var SlidingState = /*#__PURE__*/Object.freeze({
        __proto__: null,
        init: init$5
    });

    const Sliding = create$3({
        fields: SlidingSchema,
        name: 'sliding',
        active: ActiveSliding,
        apis: SlidingApis,
        state: SlidingState
    });

    const events$4 = (streamConfig, streamState) => {
        const streams = streamConfig.stream.streams;
        const processor = streams.setup(streamConfig, streamState);
        return derive$2([
            run$1(streamConfig.event, processor),
            runOnDetached(() => streamState.cancel())
        ].concat(streamConfig.cancelEvent.map((e) => [
            run$1(e, () => streamState.cancel())
        ]).getOr([])));
    };

    var ActiveStreaming = /*#__PURE__*/Object.freeze({
        __proto__: null,
        events: events$4
    });

    const throttle = (_config) => {
        const state = Cell(null);
        const readState = () => ({
            timer: state.get() !== null ? 'set' : 'unset'
        });
        const setTimer = (t) => {
            state.set(t);
        };
        const cancel = () => {
            const t = state.get();
            if (t !== null) {
                t.cancel();
            }
        };
        return nu$4({
            readState,
            setTimer,
            cancel
        });
    };
    const init$4 = (spec) => spec.stream.streams.state(spec);

    var StreamingState = /*#__PURE__*/Object.freeze({
        __proto__: null,
        throttle: throttle,
        init: init$4
    });

    const setup$e = (streamInfo, streamState) => {
        const sInfo = streamInfo.stream;
        const throttler = last(streamInfo.onStream, sInfo.delay);
        streamState.setTimer(throttler);
        return (component, simulatedEvent) => {
            throttler.throttle(component, simulatedEvent);
            if (sInfo.stopEvent) {
                simulatedEvent.stop();
            }
        };
    };
    var StreamingSchema = [
        requiredOf('stream', choose$1('mode', {
            throttle: [
                required$1('delay'),
                defaulted('stopEvent', true),
                output$1('streams', {
                    setup: setup$e,
                    state: throttle
                })
            ]
        })),
        defaulted('event', 'input'),
        option$3('cancelEvent'),
        onStrictHandler('onStream')
    ];

    const Streaming = create$3({
        fields: StreamingSchema,
        name: 'streaming',
        active: ActiveStreaming,
        state: StreamingState
    });

    const exhibit$2 = (base, tabConfig) => nu$2({
        attributes: wrapAll([
            { key: tabConfig.tabAttr, value: 'true' }
        ])
    });

    var ActiveTabstopping = /*#__PURE__*/Object.freeze({
        __proto__: null,
        exhibit: exhibit$2
    });

    var TabstopSchema = [
        defaulted('tabAttr', 'data-alloy-tabstop')
    ];

    const Tabstopping = create$3({
        fields: TabstopSchema,
        name: 'tabstopping',
        active: ActiveTabstopping
    });

    const updateAriaState = (component, toggleConfig, toggleState) => {
        const ariaInfo = toggleConfig.aria;
        ariaInfo.update(component, ariaInfo, toggleState.get());
    };
    const updateClass = (component, toggleConfig, toggleState) => {
        toggleConfig.toggleClass.each((toggleClass) => {
            if (toggleState.get()) {
                add$2(component.element, toggleClass);
            }
            else {
                remove$3(component.element, toggleClass);
            }
        });
    };
    const set = (component, toggleConfig, toggleState, state) => {
        const initialState = toggleState.get();
        toggleState.set(state);
        updateClass(component, toggleConfig, toggleState);
        updateAriaState(component, toggleConfig, toggleState);
        if (initialState !== state) {
            toggleConfig.onToggled(component, state);
        }
    };
    const toggle$2 = (component, toggleConfig, toggleState) => {
        set(component, toggleConfig, toggleState, !toggleState.get());
    };
    const on = (component, toggleConfig, toggleState) => {
        set(component, toggleConfig, toggleState, true);
    };
    const off = (component, toggleConfig, toggleState) => {
        set(component, toggleConfig, toggleState, false);
    };
    const isOn = (component, toggleConfig, toggleState) => toggleState.get();
    const onLoad = (component, toggleConfig, toggleState) => {
        // There used to be a bit of code in here that would only overwrite
        // the attribute if it didn't have a current value. I can't remember
        // what case that was for, so I'm removing it until it is required.
        set(component, toggleConfig, toggleState, toggleConfig.selected);
    };

    var ToggleApis = /*#__PURE__*/Object.freeze({
        __proto__: null,
        onLoad: onLoad,
        toggle: toggle$2,
        isOn: isOn,
        on: on,
        off: off,
        set: set
    });

    const exhibit$1 = () => nu$2({});
    const events$3 = (toggleConfig, toggleState) => {
        const execute = executeEvent(toggleConfig, toggleState, toggle$2);
        const load = loadEvent(toggleConfig, toggleState, onLoad);
        return derive$2(flatten([
            toggleConfig.toggleOnExecute ? [execute] : [],
            [load]
        ]));
    };

    var ActiveToggle = /*#__PURE__*/Object.freeze({
        __proto__: null,
        exhibit: exhibit$1,
        events: events$3
    });

    const updatePressed = (component, ariaInfo, status) => {
        set$9(component.element, 'aria-pressed', status);
        if (ariaInfo.syncWithExpanded) {
            updateExpanded(component, ariaInfo, status);
        }
    };
    const updateSelected = (component, ariaInfo, status) => {
        set$9(component.element, 'aria-selected', status);
    };
    const updateChecked = (component, ariaInfo, status) => {
        set$9(component.element, 'aria-checked', status);
    };
    const updateExpanded = (component, ariaInfo, status) => {
        set$9(component.element, 'aria-expanded', status);
    };

    var ToggleSchema = [
        defaulted('selected', false),
        option$3('toggleClass'),
        defaulted('toggleOnExecute', true),
        onHandler('onToggled'),
        defaultedOf('aria', {
            mode: 'none'
        }, choose$1('mode', {
            pressed: [
                defaulted('syncWithExpanded', false),
                output$1('update', updatePressed)
            ],
            checked: [
                output$1('update', updateChecked)
            ],
            expanded: [
                output$1('update', updateExpanded)
            ],
            selected: [
                output$1('update', updateSelected)
            ],
            none: [
                output$1('update', noop)
            ]
        }))
    ];

    const Toggling = create$3({
        fields: ToggleSchema,
        name: 'toggling',
        active: ActiveToggle,
        apis: ToggleApis,
        state: SetupBehaviourCellState(false)
    });

    const ExclusivityChannel = generate$6('tooltip.exclusive');
    const ShowTooltipEvent = generate$6('tooltip.show');
    const HideTooltipEvent = generate$6('tooltip.hide');
    const ImmediateHideTooltipEvent = generate$6('tooltip.immediateHide');
    const ImmediateShowTooltipEvent = generate$6('tooltip.immediateShow');

    const hideAllExclusive = (component, _tConfig, _tState) => {
        component.getSystem().broadcastOn([ExclusivityChannel], {});
    };
    const setComponents = (_component, _tConfig, tState, specs) => {
        tState.getTooltip().each((tooltip) => {
            if (tooltip.getSystem().isConnected()) {
                Replacing.set(tooltip, specs);
            }
        });
    };
    const isEnabled = (_component, _tConfig, tState) => tState.isEnabled();
    const setEnabled = (_component, _tConfig, tState, enabled) => tState.setEnabled(enabled);
    const immediateOpenClose = (component, _tConfig, _tState, open) => emit(component, open ? ImmediateShowTooltipEvent : ImmediateHideTooltipEvent);

    var TooltippingApis = /*#__PURE__*/Object.freeze({
        __proto__: null,
        hideAllExclusive: hideAllExclusive,
        immediateOpenClose: immediateOpenClose,
        isEnabled: isEnabled,
        setComponents: setComponents,
        setEnabled: setEnabled
    });

    const events$2 = (tooltipConfig, state) => {
        const hide = (comp) => {
            state.getTooltip().each((p) => {
                if (p.getSystem().isConnected()) {
                    detach(p);
                    tooltipConfig.onHide(comp, p);
                    state.clearTooltip();
                }
            });
            state.clearTimer();
        };
        const show = (comp) => {
            if (!state.isShowing() && state.isEnabled()) {
                hideAllExclusive(comp);
                const sink = tooltipConfig.lazySink(comp).getOrDie();
                const popup = comp.getSystem().build({
                    dom: tooltipConfig.tooltipDom,
                    components: tooltipConfig.tooltipComponents,
                    events: derive$2(tooltipConfig.mode === 'normal'
                        ? [
                            run$1(mouseover(), (_) => {
                                emit(comp, ShowTooltipEvent);
                            }),
                            run$1(mouseout(), (_) => {
                                emit(comp, HideTooltipEvent);
                            })
                        ]
                        : []),
                    behaviours: derive$1([
                        Replacing.config({})
                    ])
                });
                state.setTooltip(popup);
                attach(sink, popup);
                tooltipConfig.onShow(comp, popup);
                Positioning.position(sink, popup, { anchor: tooltipConfig.anchor(comp) });
            }
        };
        const reposition = (comp) => {
            state.getTooltip().each((tooltip) => {
                const sink = tooltipConfig.lazySink(comp).getOrDie();
                Positioning.position(sink, tooltip, { anchor: tooltipConfig.anchor(comp) });
            });
        };
        const getEvents = () => {
            switch (tooltipConfig.mode) {
                case 'normal':
                    return [
                        run$1(focusin(), (comp) => {
                            emit(comp, ImmediateShowTooltipEvent);
                        }),
                        run$1(postBlur(), (comp) => {
                            emit(comp, ImmediateHideTooltipEvent);
                        }),
                        run$1(mouseover(), (comp) => {
                            emit(comp, ShowTooltipEvent);
                        }),
                        run$1(mouseout(), (comp) => {
                            emit(comp, HideTooltipEvent);
                        })
                    ];
                case 'follow-highlight':
                    return [
                        run$1(highlight$1(), (comp, _se) => {
                            emit(comp, ShowTooltipEvent);
                        }),
                        run$1(dehighlight$1(), (comp) => {
                            emit(comp, HideTooltipEvent);
                        })
                    ];
                case 'children-normal':
                    return [
                        run$1(focusin(), (comp, se) => {
                            search(comp.element).each((_) => {
                                if (is(se.event.target, '[data-mce-tooltip]')) {
                                    state.getTooltip().fold(() => {
                                        emit(comp, ImmediateShowTooltipEvent);
                                    }, (tooltip) => {
                                        if (state.isShowing()) {
                                            tooltipConfig.onShow(comp, tooltip);
                                            reposition(comp);
                                        }
                                    });
                                }
                            });
                        }),
                        run$1(postBlur(), (comp) => {
                            search(comp.element).fold(() => {
                                emit(comp, ImmediateHideTooltipEvent);
                            }, noop);
                        }),
                        run$1(mouseover(), (comp) => {
                            descendant(comp.element, '[data-mce-tooltip]:hover').each((_) => {
                                state.getTooltip().fold(() => {
                                    emit(comp, ShowTooltipEvent);
                                }, (tooltip) => {
                                    if (state.isShowing()) {
                                        tooltipConfig.onShow(comp, tooltip);
                                        reposition(comp);
                                    }
                                });
                            });
                        }),
                        run$1(mouseout(), (comp) => {
                            descendant(comp.element, '[data-mce-tooltip]:hover').fold(() => {
                                emit(comp, HideTooltipEvent);
                            }, noop);
                        }),
                    ];
                default:
                    return [
                        run$1(focusin(), (comp, se) => {
                            search(comp.element).each((_) => {
                                if (is(se.event.target, '[data-mce-tooltip]')) {
                                    state.getTooltip().fold(() => {
                                        emit(comp, ImmediateShowTooltipEvent);
                                    }, (tooltip) => {
                                        if (state.isShowing()) {
                                            tooltipConfig.onShow(comp, tooltip);
                                            reposition(comp);
                                        }
                                    });
                                }
                            });
                        }),
                        run$1(postBlur(), (comp) => {
                            search(comp.element).fold(() => {
                                emit(comp, ImmediateHideTooltipEvent);
                            }, noop);
                        }),
                    ];
            }
        };
        return derive$2(flatten([
            [
                runOnInit((component) => {
                    tooltipConfig.onSetup(component);
                }),
                run$1(ShowTooltipEvent, (comp) => {
                    state.resetTimer(() => {
                        show(comp);
                    }, tooltipConfig.delayForShow());
                }),
                run$1(HideTooltipEvent, (comp) => {
                    state.resetTimer(() => {
                        hide(comp);
                    }, tooltipConfig.delayForHide());
                }),
                run$1(ImmediateShowTooltipEvent, (comp) => {
                    state.resetTimer(() => {
                        show(comp);
                    }, 0);
                }),
                run$1(ImmediateHideTooltipEvent, (comp) => {
                    state.resetTimer(() => {
                        hide(comp);
                    }, 0);
                }),
                run$1(receive(), (comp, message) => {
                    // TODO: Think about the types for this, or find a better way for this
                    // to rely on receiving.
                    const receivingData = message;
                    if (!receivingData.universal) {
                        if (contains$2(receivingData.channels, ExclusivityChannel) || contains$2(receivingData.channels, closeTooltips())) {
                            if (receivingData.data.closedTooltip && state.isShowing()) {
                                receivingData.data.closedTooltip();
                            }
                            hide(comp);
                        }
                    }
                }),
                runOnDetached((comp) => {
                    hide(comp);
                })
            ],
            (getEvents())
        ]));
    };

    var ActiveTooltipping = /*#__PURE__*/Object.freeze({
        __proto__: null,
        events: events$2
    });

    var TooltippingSchema = [
        required$1('lazySink'),
        required$1('tooltipDom'),
        defaulted('exclusive', true),
        defaulted('tooltipComponents', []),
        defaultedFunction('delayForShow', constant$1(300)),
        defaultedFunction('delayForHide', constant$1(100)),
        defaultedFunction('onSetup', noop),
        defaultedStringEnum('mode', 'normal', ['normal', 'follow-highlight', 'children-keyboard-focus', 'children-normal']),
        defaulted('anchor', (comp) => ({
            type: 'hotspot',
            hotspot: comp,
            layouts: {
                onLtr: constant$1([south$2, north$2, southeast$2, northeast$2, southwest$2, northwest$2]),
                onRtl: constant$1([south$2, north$2, southeast$2, northeast$2, southwest$2, northwest$2])
            },
            bubble: nu$6(0, -2, {}),
        })),
        onHandler('onHide'),
        onHandler('onShow'),
    ];

    const init$3 = () => {
        const enabled = Cell(true);
        const timer = value$2();
        const popup = value$2();
        const clearTimer = () => {
            timer.on(clearTimeout);
        };
        const resetTimer = (f, delay) => {
            clearTimer();
            timer.set(setTimeout(f, delay));
        };
        const readState = constant$1('not-implemented');
        return nu$4({
            getTooltip: popup.get,
            isShowing: popup.isSet,
            setTooltip: popup.set,
            clearTooltip: popup.clear,
            clearTimer,
            resetTimer,
            readState,
            isEnabled: () => enabled.get(),
            setEnabled: (setToEnabled) => enabled.set(setToEnabled)
        });
    };

    var TooltippingState = /*#__PURE__*/Object.freeze({
        __proto__: null,
        init: init$3
    });

    const Tooltipping = create$3({
        fields: TooltippingSchema,
        name: 'tooltipping',
        active: ActiveTooltipping,
        state: TooltippingState,
        apis: TooltippingApis
    });

    const exhibit = () => nu$2({
        styles: {
            '-webkit-user-select': 'none',
            'user-select': 'none',
            '-ms-user-select': 'none',
            '-moz-user-select': '-moz-none'
        },
        attributes: {
            unselectable: 'on'
        }
    });
    const events$1 = () => derive$2([
        abort(selectstart(), always)
    ]);

    var ActiveUnselecting = /*#__PURE__*/Object.freeze({
        __proto__: null,
        events: events$1,
        exhibit: exhibit
    });

    const Unselecting = create$3({
        fields: [],
        name: 'unselecting',
        active: ActiveUnselecting
    });

    const getAttrs = (elem) => {
        const attributes = elem.dom.attributes !== undefined ? elem.dom.attributes : [];
        return foldl(attributes, (b, attr) => {
            // Make class go through the class path. Do not list it as an attribute.
            if (attr.name === 'class') {
                return b;
            }
            else {
                return { ...b, [attr.name]: attr.value };
            }
        }, {});
    };
    const getClasses = (elem) => Array.prototype.slice.call(elem.dom.classList, 0);
    const fromHtml = (html) => {
        const elem = SugarElement.fromHtml(html);
        const children$1 = children(elem);
        const attrs = getAttrs(elem);
        const classes = getClasses(elem);
        const contents = children$1.length === 0 ? {} : { innerHtml: get$f(elem) };
        return {
            tag: name$3(elem),
            classes,
            attributes: attrs,
            ...contents
        };
    };

    const record = (spec) => {
        const uid = isSketchSpec(spec) && hasNonNullableKey(spec, 'uid') ? spec.uid : generate$4('memento');
        const get = (anyInSystem) => anyInSystem.getSystem().getByUid(uid).getOrDie();
        const getOpt = (anyInSystem) => anyInSystem.getSystem().getByUid(uid).toOptional();
        const asSpec = () => ({
            ...spec,
            uid
        });
        return {
            get,
            getOpt,
            asSpec
        };
    };

    // TODO: ^ rename the parts/ api to composites, it will break mobile alloy now if we do
    const parts$g = AlloyParts;
    const partType$1 = PartType;

    const fromSource = (event, source) => {
        const stopper = Cell(false);
        const cutter = Cell(false);
        const stop = () => {
            stopper.set(true);
        };
        const cut = () => {
            cutter.set(true);
        };
        return {
            stop,
            cut,
            isStopped: stopper.get,
            isCut: cutter.get,
            event,
            // Used only for tiered menu at the moment. It is an element, not a component
            setSource: source.set,
            getSource: source.get
        };
    };
    // Events that come from outside of the alloy root (e.g. window scroll)
    const fromExternal = (event) => {
        const stopper = Cell(false);
        const stop = () => {
            stopper.set(true);
        };
        return {
            stop,
            cut: noop, // cutting has no meaning for a broadcasted event
            isStopped: stopper.get,
            isCut: never,
            event,
            // Nor do targets really
            setSource: die('Cannot set source of a broadcasted event'),
            getSource: die('Cannot get source of a broadcasted event')
        };
    };

    const isDangerous = (event) => {
        // Will trigger the Back button in the browser
        const keyEv = event.raw;
        return keyEv.which === BACKSPACE[0] && !contains$2(['input', 'textarea'], name$3(event.target)) && !closest$1(event.target, '[contenteditable="true"]');
    };
    const setup$d = (container, rawSettings) => {
        const settings = {
            stopBackspace: true,
            ...rawSettings
        };
        const pointerEvents = [
            'touchstart',
            'touchmove',
            'touchend',
            'touchcancel',
            'gesturestart',
            'mousedown',
            'mouseup',
            'mouseover',
            'mousemove',
            'mouseout',
            'click'
        ];
        const tapEvent = monitor(settings);
        // These events are just passed through ... no additional processing
        const simpleEvents = map$2(pointerEvents.concat([
            'selectstart',
            'input',
            'contextmenu',
            'change',
            'transitionend',
            'transitioncancel',
            // Test the drag events
            'drag',
            'dragstart',
            'dragend',
            'dragenter',
            'dragleave',
            'dragover',
            'drop',
            'keyup'
        ]), (type) => bind$1(container, type, (event) => {
            tapEvent.fireIfReady(event, type).each((tapStopped) => {
                if (tapStopped) {
                    event.kill();
                }
            });
            const stopped = settings.triggerEvent(type, event);
            if (stopped) {
                event.kill();
            }
        }));
        const pasteTimeout = value$2();
        const onPaste = bind$1(container, 'paste', (event) => {
            tapEvent.fireIfReady(event, 'paste').each((tapStopped) => {
                if (tapStopped) {
                    event.kill();
                }
            });
            const stopped = settings.triggerEvent('paste', event);
            if (stopped) {
                event.kill();
            }
            pasteTimeout.set(setTimeout(() => {
                settings.triggerEvent(postPaste(), event);
            }, 0));
        });
        const onKeydown = bind$1(container, 'keydown', (event) => {
            // Prevent default of backspace when not in input fields.
            const stopped = settings.triggerEvent('keydown', event);
            if (stopped) {
                event.kill();
            }
            else if (settings.stopBackspace && isDangerous(event)) {
                event.prevent();
            }
        });
        const onFocusIn = bind$1(container, 'focusin', (event) => {
            const stopped = settings.triggerEvent('focusin', event);
            if (stopped) {
                event.kill();
            }
        });
        const focusoutTimeout = value$2();
        const onFocusOut = bind$1(container, 'focusout', (event) => {
            const stopped = settings.triggerEvent('focusout', event);
            if (stopped) {
                event.kill();
            }
            // INVESTIGATE: Come up with a better way of doing this. Related target can be used, but not on FF.
            // It allows the active element to change before firing the blur that we will listen to
            // for things like closing popups
            focusoutTimeout.set(setTimeout(() => {
                settings.triggerEvent(postBlur(), event);
            }, 0));
        });
        const unbind = () => {
            each$1(simpleEvents, (e) => {
                e.unbind();
            });
            onKeydown.unbind();
            onFocusIn.unbind();
            onFocusOut.unbind();
            onPaste.unbind();
            pasteTimeout.on(clearTimeout);
            focusoutTimeout.on(clearTimeout);
        };
        return {
            unbind
        };
    };

    const derive = (rawEvent, rawTarget) => {
        const source = get$h(rawEvent, 'target').getOr(rawTarget);
        return Cell(source);
    };

    const adt = Adt.generate([
        { stopped: [] },
        { resume: ['element'] },
        { complete: [] }
    ]);
    const doTriggerHandler = (lookup, eventType, rawEvent, target, source, logger) => {
        const handler = lookup(eventType, target);
        const simulatedEvent = fromSource(rawEvent, source);
        return handler.fold(() => {
            // No handler, so complete.
            logger.logEventNoHandlers(eventType, target);
            return adt.complete();
        }, (handlerInfo) => {
            const descHandler = handlerInfo.descHandler;
            const eventHandler = getCurried(descHandler);
            eventHandler(simulatedEvent);
            // Now, check if the event was stopped.
            if (simulatedEvent.isStopped()) {
                logger.logEventStopped(eventType, handlerInfo.element, descHandler.purpose);
                return adt.stopped();
            }
            else if (simulatedEvent.isCut()) {
                logger.logEventCut(eventType, handlerInfo.element, descHandler.purpose);
                return adt.complete();
            }
            else {
                return parent(handlerInfo.element).fold(() => {
                    logger.logNoParent(eventType, handlerInfo.element, descHandler.purpose);
                    // No parent, so complete.
                    return adt.complete();
                }, (parent) => {
                    logger.logEventResponse(eventType, handlerInfo.element, descHandler.purpose);
                    // Resume at parent
                    return adt.resume(parent);
                });
            }
        });
    };
    const doTriggerOnUntilStopped = (lookup, eventType, rawEvent, rawTarget, source, logger) => doTriggerHandler(lookup, eventType, rawEvent, rawTarget, source, logger).fold(
    // stopped.
    always, 
    // Go again.
    (parent) => doTriggerOnUntilStopped(lookup, eventType, rawEvent, parent, source, logger), 
    // completed
    never);
    const triggerHandler = (lookup, eventType, rawEvent, target, logger) => {
        const source = derive(rawEvent, target);
        return doTriggerHandler(lookup, eventType, rawEvent, target, source, logger);
    };
    const broadcast = (listeners, rawEvent, _logger) => {
        const simulatedEvent = fromExternal(rawEvent);
        each$1(listeners, (listener) => {
            const descHandler = listener.descHandler;
            const handler = getCurried(descHandler);
            handler(simulatedEvent);
        });
        return simulatedEvent.isStopped();
    };
    const triggerUntilStopped = (lookup, eventType, rawEvent, logger) => triggerOnUntilStopped(lookup, eventType, rawEvent, rawEvent.target, logger);
    const triggerOnUntilStopped = (lookup, eventType, rawEvent, rawTarget, logger) => {
        const source = derive(rawEvent, rawTarget);
        return doTriggerOnUntilStopped(lookup, eventType, rawEvent, rawTarget, source, logger);
    };

    const eventHandler = (element, descHandler) => ({
        element,
        descHandler
    });
    const broadcastHandler = (id, handler) => ({
        id,
        descHandler: handler
    });
    const EventRegistry = () => {
        const registry = {};
        const registerId = (extraArgs, id, events) => {
            each(events, (v, k) => {
                const handlers = registry[k] !== undefined ? registry[k] : {};
                handlers[id] = curryArgs(v, extraArgs);
                registry[k] = handlers;
            });
        };
        const findHandler = (handlers, elem) => read(elem)
            .bind((id) => get$h(handlers, id))
            .map((descHandler) => eventHandler(elem, descHandler));
        // Given just the event type, find all handlers regardless of element
        const filterByType = (type) => get$h(registry, type)
            .map((handlers) => mapToArray(handlers, (f, id) => broadcastHandler(id, f)))
            .getOr([]);
        // Given event type, and element, find the handler.
        const find = (isAboveRoot, type, target) => get$h(registry, type)
            .bind((handlers) => closest(target, (elem) => findHandler(handlers, elem), isAboveRoot));
        const unregisterId = (id) => {
            // INVESTIGATE: Find a better way than mutation if we can.
            each(registry, (handlersById, _eventName) => {
                if (has$2(handlersById, id)) {
                    delete handlersById[id];
                }
            });
        };
        return {
            registerId,
            unregisterId,
            filterByType,
            find
        };
    };

    const Registry = () => {
        const events = EventRegistry();
        // An index of uid -> built components
        const components = {};
        const readOrTag = (component) => {
            const elem = component.element;
            return read(elem).getOrThunk(() => 
            // No existing tag, so add one.
            write('uid-', component.element));
        };
        const failOnDuplicate = (component, tagId) => {
            const conflict = components[tagId];
            if (conflict === component) {
                unregister(component);
            }
            else {
                throw new Error('The tagId "' + tagId + '" is already used by: ' + element(conflict.element) + '\nCannot use it for: ' + element(component.element) + '\n' +
                    'The conflicting element is' + (inBody(conflict.element) ? ' ' : ' not ') + 'already in the DOM');
            }
        };
        const register = (component) => {
            const tagId = readOrTag(component);
            if (hasNonNullableKey(components, tagId)) {
                failOnDuplicate(component, tagId);
            }
            // Component is passed through an an extra argument to all events
            const extraArgs = [component];
            events.registerId(extraArgs, tagId, component.events);
            components[tagId] = component;
        };
        const unregister = (component) => {
            read(component.element).each((tagId) => {
                delete components[tagId];
                events.unregisterId(tagId);
            });
        };
        const filter = (type) => events.filterByType(type);
        const find = (isAboveRoot, type, target) => events.find(isAboveRoot, type, target);
        const getById = (id) => get$h(components, id);
        return {
            find,
            filter,
            register,
            unregister,
            getById
        };
    };

    const takeover = (root) => {
        const isAboveRoot = (el) => parent(root.element).fold(always, (parent) => eq(el, parent));
        const registry = Registry();
        const lookup = (eventName, target) => registry.find(isAboveRoot, eventName, target);
        const domEvents = setup$d(root.element, {
            triggerEvent: (eventName, event) => {
                return monitorEvent(eventName, event.target, (logger) => triggerUntilStopped(lookup, eventName, event, logger));
            }
        });
        const systemApi = {
            // This is a real system
            debugInfo: constant$1('real'),
            triggerEvent: (eventName, target, data) => {
                monitorEvent(eventName, target, (logger) => 
                // The return value is not used because this is a fake event.
                triggerOnUntilStopped(lookup, eventName, data, target, logger));
            },
            triggerFocus: (target, originator) => {
                read(target).fold(() => {
                    // When the target is not within the alloy system, dispatch a normal focus event.
                    focus$4(target);
                }, (_alloyId) => {
                    monitorEvent(focus$3(), target, (logger) => {
                        // NOTE: This will stop at first handler.
                        triggerHandler(lookup, focus$3(), {
                            // originator is used by the default events to ensure that focus doesn't
                            // get called infinitely
                            originator,
                            kill: noop,
                            prevent: noop,
                            target
                        }, target, logger);
                        return false;
                    });
                });
            },
            triggerEscape: (comp, simulatedEvent) => {
                systemApi.triggerEvent('keydown', comp.element, simulatedEvent.event);
            },
            getByUid: (uid) => {
                return getByUid(uid);
            },
            getByDom: (elem) => {
                return getByDom(elem);
            },
            build: build$1,
            buildOrPatch: buildOrPatch,
            addToGui: (c) => {
                add(c);
            },
            removeFromGui: (c) => {
                remove(c);
            },
            addToWorld: (c) => {
                addToWorld(c);
            },
            removeFromWorld: (c) => {
                removeFromWorld(c);
            },
            broadcast: (message) => {
                broadcast$1(message);
            },
            broadcastOn: (channels, message) => {
                broadcastOn(channels, message);
            },
            broadcastEvent: (eventName, event) => {
                broadcastEvent(eventName, event);
            },
            isConnected: always
        };
        const addToWorld = (component) => {
            component.connect(systemApi);
            if (!isText(component.element)) {
                registry.register(component);
                each$1(component.components(), addToWorld);
                systemApi.triggerEvent(systemInit(), component.element, { target: component.element });
            }
        };
        const removeFromWorld = (component) => {
            if (!isText(component.element)) {
                each$1(component.components(), removeFromWorld);
                registry.unregister(component);
            }
            component.disconnect();
        };
        const add = (component) => {
            attach(root, component);
        };
        const remove = (component) => {
            detach(component);
        };
        const destroy = () => {
            // INVESTIGATE: something with registry?
            domEvents.unbind();
            remove$7(root.element);
        };
        const broadcastData = (data) => {
            const receivers = registry.filter(receive());
            each$1(receivers, (receiver) => {
                const descHandler = receiver.descHandler;
                const handler = getCurried(descHandler);
                handler(data);
            });
        };
        const broadcast$1 = (message) => {
            broadcastData({
                universal: true,
                data: message
            });
        };
        const broadcastOn = (channels, message) => {
            broadcastData({
                universal: false,
                channels,
                data: message
            });
        };
        // This doesn't follow usual DOM bubbling. It will just dispatch on all
        // targets that have the event. It is the general case of the more specialised
        // "message". "messages" may actually just go away. This is used for things
        // like window scroll.
        const broadcastEvent = (eventName, event) => {
            const listeners = registry.filter(eventName);
            return broadcast(listeners, event);
        };
        const getByUid = (uid) => registry.getById(uid).fold(() => Result.error(new Error('Could not find component with uid: "' + uid + '" in system.')), Result.value);
        const getByDom = (elem) => {
            const uid = read(elem).getOr('not found');
            return getByUid(uid);
        };
        addToWorld(root);
        return {
            root,
            element: root.element,
            destroy,
            add,
            remove,
            getByUid,
            getByDom,
            addToWorld,
            removeFromWorld,
            broadcast: broadcast$1,
            broadcastOn,
            broadcastEvent
        };
    };

    const pointerEvents = () => {
        const onClick = (component, simulatedEvent) => {
            simulatedEvent.stop();
            emitExecute(component);
        };
        return [
            // Trigger execute when clicked
            run$1(click(), onClick),
            run$1(tap(), onClick),
            // Other mouse down listeners above this one should not get mousedown behaviour (like dragging)
            cutter(touchstart()),
            cutter(mousedown())
        ];
    };
    const events = (optAction) => {
        const executeHandler = (action) => runOnExecute$1((component, simulatedEvent) => {
            action(component);
            simulatedEvent.stop();
        });
        return derive$2(flatten([
            // Only listen to execute if it is supplied
            optAction.map(executeHandler).toArray(),
            pointerEvents()
        ]));
    };

    const factory$m = (detail) => {
        const events$1 = events(detail.action);
        const tag = detail.dom.tag;
        const lookupAttr = (attr) => get$h(detail.dom, 'attributes').bind((attrs) => get$h(attrs, attr));
        // Button tags should not have a default role of button, and only buttons should
        // get a type of button.
        const getModAttributes = () => {
            if (tag === 'button') {
                // Default to type button, unless specified otherwise
                const type = lookupAttr('type').getOr('button');
                // Only use a role if it is specified
                const roleAttrs = lookupAttr('role').map((role) => ({ role })).getOr({});
                return {
                    type,
                    ...roleAttrs
                };
            }
            else {
                // We are not a button, so type is irrelevant (unless specified)
                // Default role to button
                const role = detail.role.getOr(lookupAttr('role').getOr('button'));
                return { role };
            }
        };
        return {
            uid: detail.uid,
            dom: detail.dom,
            components: detail.components,
            events: events$1,
            behaviours: SketchBehaviours.augment(detail.buttonBehaviours, [
                Focusing.config({}),
                Keying.config({
                    mode: 'execution',
                    // Note execution will capture keyup when the focus is on the button
                    // on Firefox, because otherwise it will fire a click event and double
                    // up on the action
                    useSpace: true,
                    useEnter: true
                })
            ]),
            domModification: {
                attributes: getModAttributes()
            },
            eventOrder: detail.eventOrder
        };
    };
    const Button = single({
        name: 'Button',
        factory: factory$m,
        configFields: [
            defaulted('uid', undefined),
            required$1('dom'),
            defaulted('components', []),
            SketchBehaviours.field('buttonBehaviours', [Focusing, Keying]),
            option$3('action'),
            option$3('role'),
            defaulted('eventOrder', {})
        ]
    });

    const schema$m = constant$1([
        defaulted('shell', false),
        required$1('makeItem'),
        defaulted('setupItem', noop),
        SketchBehaviours.field('listBehaviours', [Replacing])
    ]);
    const customListDetail = () => ({
        behaviours: derive$1([
            Replacing.config({})
        ])
    });
    const itemsPart = optional({
        name: 'items',
        overrides: customListDetail
    });
    const parts$f = constant$1([
        itemsPart
    ]);
    const name$1 = constant$1('CustomList');

    const factory$l = (detail, components, _spec, _external) => {
        const setItems = (list, items) => {
            getListContainer(list).fold(() => {
                // check that the group container existed. It may not have if the components
                // did not list anything, and shell was false.
                // eslint-disable-next-line no-console
                console.error('Custom List was defined to not be a shell, but no item container was specified in components');
                throw new Error('Custom List was defined to not be a shell, but no item container was specified in components');
            }, (container) => {
                // Get all the children of container, because they will be items.
                // And then use the item setGroup api
                const itemComps = Replacing.contents(container);
                const numListsRequired = items.length;
                const numListsToAdd = numListsRequired - itemComps.length;
                const itemsToAdd = numListsToAdd > 0 ?
                    range$2(numListsToAdd, () => detail.makeItem()) : [];
                const itemsToRemove = itemComps.slice(numListsRequired);
                each$1(itemsToRemove, (item) => Replacing.remove(container, item));
                each$1(itemsToAdd, (item) => Replacing.append(container, item));
                const builtLists = Replacing.contents(container);
                each$1(builtLists, (item, i) => {
                    detail.setupItem(list, item, items[i], i);
                });
            });
        };
        // In shell mode, the group overrides need to be added to the main container, and there can be no children
        const extra = detail.shell ? { behaviours: [Replacing.config({})], components: [] } : { behaviours: [], components };
        const getListContainer = (component) => detail.shell ? Optional.some(component) : getPart(component, detail, 'items');
        return {
            uid: detail.uid,
            dom: detail.dom,
            components: extra.components,
            behaviours: augment(detail.listBehaviours, extra.behaviours),
            apis: {
                setItems
            }
        };
    };
    const CustomList = composite({
        name: name$1(),
        configFields: schema$m(),
        partFields: parts$f(),
        factory: factory$l,
        apis: {
            setItems: (apis, list, items) => {
                apis.setItems(list, items);
            }
        }
    });

    const attribute = 'aria-controls';
    const find$1 = (queryElem) => {
        const dependent = closest$4(queryElem, (elem) => {
            if (!isElement$1(elem)) {
                return false;
            }
            const id = get$g(elem, 'id');
            return id !== undefined && id.indexOf(attribute) > -1;
        });
        return dependent.bind((dep) => {
            const id = get$g(dep, 'id');
            const dos = getRootNode(dep);
            return descendant(dos, `[${attribute}="${id}"]`);
        });
    };
    const manager = () => {
        const ariaId = generate$6(attribute);
        const link = (elem) => {
            set$9(elem, attribute, ariaId);
        };
        const unlink = (elem) => {
            remove$8(elem, attribute);
        };
        return {
            id: ariaId,
            link,
            unlink,
        };
    };

    const isAriaPartOf = (component, queryElem) => find$1(queryElem).exists((owner) => isPartOf(component, owner));
    const isPartOf = (component, queryElem) => closest$2(queryElem, (el) => eq(el, component.element), never) || isAriaPartOf(component, queryElem);

    const hoverEvent = 'alloy.item-hover';
    const focusEvent = 'alloy.item-focus';
    const toggledEvent = 'alloy.item-toggled';
    const onHover = (item) => {
        // Firstly, check that the focus isn't already inside the item. This
        // is to handle situations like widgets where the widget is inside the item
        // and it has the focus, so as you slightly adjust the mouse, you don't
        // want to lose focus on the widget. Note, that because this isn't API based
        // (i.e. we are manually searching for focus), it may not be that flexible.
        if (search(item.element).isNone() || Focusing.isFocused(item)) {
            if (!Focusing.isFocused(item)) {
                Focusing.focus(item);
            }
            emitWith(item, hoverEvent, { item });
        }
    };
    const onFocus$1 = (item) => {
        emitWith(item, focusEvent, { item });
    };
    const onToggled = (item, state) => {
        emitWith(item, toggledEvent, { item, state });
    };
    const hover = constant$1(hoverEvent);
    const focus$1 = constant$1(focusEvent);
    const toggled = constant$1(toggledEvent);

    // eslint-disable-next-line @typescript-eslint/no-redundant-type-constituents
    const getItemRole = (detail) => detail.role.fold(() => detail.toggling
        .map((toggling) => toggling.exclusive ? 'menuitemradio' : 'menuitemcheckbox')
        .getOr('menuitem'), identity);
    const getTogglingSpec = (tConfig, isOption) => ({
        aria: {
            mode: isOption ? 'selected' : 'checked'
        },
        // Filter out the additional properties that are not in Toggling Behaviour's configuration (e.g. exclusive)
        ...filter$1(tConfig, (_value, name) => name !== 'exclusive'),
        onToggled: (component, state) => {
            if (isFunction(tConfig.onToggled)) {
                tConfig.onToggled(component, state);
            }
            onToggled(component, state);
        }
    });
    const builder$2 = (detail) => ({
        dom: detail.dom,
        domModification: {
            // INVESTIGATE: If more efficient, destructure attributes out
            ...detail.domModification,
            attributes: {
                'role': getItemRole(detail),
                ...detail.domModification.attributes,
                'aria-haspopup': detail.hasSubmenu,
                ...(detail.hasSubmenu ? { 'aria-expanded': false } : {})
            }
        },
        behaviours: SketchBehaviours.augment(detail.itemBehaviours, [
            // Investigate, is the Toggling.revoke still necessary here?
            detail.toggling.fold(Toggling.revoke, (tConfig) => Toggling.config(getTogglingSpec(tConfig, detail.role.exists((role) => role === 'option')))),
            Focusing.config({
                ignore: detail.ignoreFocus,
                // Rationale: because nothing is focusable, when you click
                // on the items to choose them, the focus jumps to the first
                // focusable outer container ... often the body. If we prevent
                // mouseDown ... that doesn't happen. But only tested on Chrome/FF.
                stopMousedown: detail.ignoreFocus,
                onFocus: (component) => {
                    onFocus$1(component);
                }
            }),
            Keying.config({
                mode: 'execution'
            }),
            Representing.config({
                store: {
                    mode: 'memory',
                    initialValue: detail.data
                }
            }),
            config('item-type-events', [
                // Treat clicks the same as a button
                ...pointerEvents(),
                run$1(mouseover(), onHover),
                run$1(focusItem(), Focusing.focus)
            ])
        ]),
        components: detail.components,
        eventOrder: detail.eventOrder
    });
    const schema$l = [
        required$1('data'),
        required$1('components'),
        required$1('dom'),
        defaulted('hasSubmenu', false),
        option$3('toggling'),
        option$3('role'),
        // Maybe this needs to have fewer behaviours
        SketchBehaviours.field('itemBehaviours', [Toggling, Focusing, Keying, Representing]),
        defaulted('ignoreFocus', false),
        defaulted('domModification', {}),
        output$1('builder', builder$2),
        defaulted('eventOrder', {})
    ];
    var ItemType = schema$l;

    const builder$1 = (detail) => ({
        dom: detail.dom,
        components: detail.components,
        events: derive$2([
            stopper(focusItem())
        ])
    });
    const schema$k = [
        required$1('dom'),
        required$1('components'),
        output$1('builder', builder$1)
    ];
    var SeparatorType = schema$k;

    const owner$2 = constant$1('item-widget');
    const parts$e = constant$1([
        required({
            name: 'widget',
            overrides: (detail) => {
                return {
                    behaviours: derive$1([
                        Representing.config({
                            store: {
                                mode: 'manual',
                                getValue: (_component) => {
                                    return detail.data;
                                },
                                setValue: noop
                            }
                        })
                    ])
                };
            }
        })
    ]);

    const builder = (detail) => {
        const subs = substitutes(owner$2(), detail, parts$e());
        const components = components$1(owner$2(), detail, subs.internals());
        const focusWidget = (component) => getPart(component, detail, 'widget').map((widget) => {
            Keying.focusIn(widget);
            return widget;
        });
        const onHorizontalArrow = (component, simulatedEvent) => inside(simulatedEvent.event.target) ? Optional.none() : (() => {
            if (detail.autofocus) {
                simulatedEvent.setSource(component.element);
                return Optional.none();
            }
            else {
                return Optional.none();
            }
        })();
        return {
            dom: detail.dom,
            components,
            domModification: detail.domModification,
            events: derive$2([
                runOnExecute$1((component, simulatedEvent) => {
                    focusWidget(component).each((_widget) => {
                        simulatedEvent.stop();
                    });
                }),
                run$1(mouseover(), onHover),
                run$1(focusItem(), (component, _simulatedEvent) => {
                    if (detail.autofocus) {
                        focusWidget(component);
                    }
                    else {
                        Focusing.focus(component);
                    }
                })
            ]),
            behaviours: SketchBehaviours.augment(detail.widgetBehaviours, [
                Representing.config({
                    store: {
                        mode: 'memory',
                        initialValue: detail.data
                    }
                }),
                Focusing.config({
                    ignore: detail.ignoreFocus,
                    // What about stopMousedown from ItemType?
                    onFocus: (component) => {
                        onFocus$1(component);
                    }
                }),
                Keying.config({
                    mode: 'special',
                    // This is required as long as Highlighting tries to focus the first thing (after focusItem fires)
                    focusIn: detail.autofocus ? (component) => {
                        focusWidget(component);
                    } : revoke(),
                    onLeft: onHorizontalArrow,
                    onRight: onHorizontalArrow,
                    onEscape: (component, simulatedEvent) => {
                        // If the outer list item didn't have focus,
                        // then focus it (i.e. escape the inner widget). Only do if not autofocusing
                        // Autofocusing should treat the widget like it is the only item, so it should
                        // let its outer menu handle escape
                        if (!Focusing.isFocused(component) && !detail.autofocus) {
                            Focusing.focus(component);
                            return Optional.some(true);
                        }
                        else if (detail.autofocus) {
                            simulatedEvent.setSource(component.element);
                            return Optional.none();
                        }
                        else {
                            return Optional.none();
                        }
                    }
                })
            ])
        };
    };
    const schema$j = [
        required$1('uid'),
        required$1('data'),
        required$1('components'),
        required$1('dom'),
        defaulted('autofocus', false),
        defaulted('ignoreFocus', false),
        SketchBehaviours.field('widgetBehaviours', [Representing, Focusing, Keying]),
        defaulted('domModification', {}),
        // We don't have the uid at this point
        defaultUidsSchema(parts$e()),
        output$1('builder', builder)
    ];
    var WidgetType = schema$j;

    const itemSchema$2 = choose$1('type', {
        widget: WidgetType,
        item: ItemType,
        separator: SeparatorType
    });
    const configureGrid = (detail, movementInfo) => ({
        mode: 'flatgrid',
        selector: '.' + detail.markers.item,
        initSize: {
            numColumns: movementInfo.initSize.numColumns,
            numRows: movementInfo.initSize.numRows
        },
        focusManager: detail.focusManager
    });
    const configureMatrix = (detail, movementInfo) => ({
        mode: 'matrix',
        selectors: {
            row: movementInfo.rowSelector,
            cell: '.' + detail.markers.item
        },
        previousSelector: movementInfo.previousSelector,
        focusManager: detail.focusManager
    });
    const configureMenu = (detail, movementInfo) => ({
        mode: 'menu',
        selector: '.' + detail.markers.item,
        moveOnTab: movementInfo.moveOnTab,
        focusManager: detail.focusManager
    });
    const parts$d = constant$1([
        group({
            factory: {
                sketch: (spec) => {
                    const itemInfo = asRawOrDie$1('menu.spec item', itemSchema$2, spec);
                    return itemInfo.builder(itemInfo);
                }
            },
            name: 'items',
            unit: 'item',
            defaults: (detail, u) => {
                // Switch this to a common library
                // The WidgetItemSpec is just because it has uid, and the others don't
                // for some reason. So there is nothing guaranteeing that `u` is a WidgetItemSpec,
                // so we should probably rework this code.
                return has$2(u, 'uid') ? u : {
                    ...u,
                    uid: generate$4('item')
                };
            },
            overrides: (detail, u) => {
                return {
                    type: u.type,
                    ignoreFocus: detail.fakeFocus,
                    domModification: {
                        classes: [detail.markers.item]
                    }
                };
            }
        })
    ]);
    const schema$i = constant$1([
        optionString('role'),
        required$1('value'),
        required$1('items'),
        required$1('dom'),
        required$1('components'),
        defaulted('eventOrder', {}),
        field('menuBehaviours', [Highlighting, Representing, Composing, Keying]),
        defaultedOf('movement', {
            // When you don't specify movement for a Menu, this is what you get
            // a "menu" type of movement that moves on tab. If you want finer-grained
            // control, like disabling moveOnTab, then you need to specify
            // your entire movement configuration when creating your MenuSpec.
            mode: 'menu',
            moveOnTab: true
        }, choose$1('mode', {
            grid: [
                initSize(),
                output$1('config', configureGrid)
            ],
            matrix: [
                output$1('config', configureMatrix),
                required$1('rowSelector'),
                defaulted('previousSelector', Optional.none),
            ],
            menu: [
                defaulted('moveOnTab', true),
                output$1('config', configureMenu)
            ]
        })),
        itemMarkers(),
        defaulted('fakeFocus', false),
        defaulted('focusManager', dom$2()),
        onHandler('onHighlight'),
        onHandler('onDehighlight'),
        defaulted('showMenuRole', true),
    ]);

    const focus = constant$1('alloy.menu-focus');

    const deselectOtherRadioItems = (menu, item) => {
        // TODO: TINY-8812 - This ideally should be done in a way such that a menu can have multiple radio groups.
        const checkedRadioItems = descendants(menu.element, '[role="menuitemradio"][aria-checked="true"]');
        each$1(checkedRadioItems, (ele) => {
            if (!eq(ele, item.element)) {
                menu.getSystem().getByDom(ele).each((c) => {
                    Toggling.off(c);
                });
            }
        });
    };
    const make$6 = (detail, components, _spec, _externals) => ({
        uid: detail.uid,
        dom: detail.dom,
        markers: detail.markers,
        behaviours: augment(detail.menuBehaviours, [
            Highlighting.config({
                // Highlighting for a menu is selecting items inside the menu
                highlightClass: detail.markers.selectedItem,
                itemClass: detail.markers.item,
                onHighlight: detail.onHighlight,
                onDehighlight: detail.onDehighlight
            }),
            Representing.config({
                store: {
                    mode: 'memory',
                    initialValue: detail.value
                }
            }),
            Composing.config({
                find: Optional.some
            }),
            Keying.config(detail.movement.config(detail, detail.movement))
        ]),
        events: derive$2([
            // This is dispatched from a menu to tell an item to be highlighted.
            run$1(focus$1(), (menu, simulatedEvent) => {
                // Highlight the item
                const event = simulatedEvent.event;
                menu.getSystem().getByDom(event.target).each((item) => {
                    Highlighting.highlight(menu, item);
                    simulatedEvent.stop();
                    // Trigger the focus event on the menu.
                    emitWith(menu, focus(), { menu, item });
                });
            }),
            // Highlight the item that the cursor is over. The onHighlight
            // code needs to handle updating focus if required
            run$1(hover(), (menu, simulatedEvent) => {
                const item = simulatedEvent.event.item;
                Highlighting.highlight(menu, item);
            }),
            // Enforce only a single radio menu item is toggled by finding any other toggled
            // radio menu items and untoggling them when a certain item is toggled
            run$1(toggled(), (menu, simulatedEvent) => {
                const { item, state } = simulatedEvent.event;
                if (state && get$g(item.element, 'role') === 'menuitemradio') {
                    deselectOtherRadioItems(menu, item);
                }
            })
        ]),
        components,
        eventOrder: detail.eventOrder,
        ...detail.showMenuRole ? {
            domModification: {
                attributes: {
                    role: detail.role.getOr('menu')
                }
            }
        } : {}
    });

    const Menu = composite({
        name: 'Menu',
        configFields: schema$i(),
        partFields: parts$d(),
        factory: make$6
    });

    const transpose$1 = (obj) => 
    // Assumes no duplicate fields.
    tupleMap(obj, (v, k) => ({ k: v, v: k }));
    const trace = (items, byItem, byMenu, finish) => 
    // Given a finishing submenu (which will be the value of expansions),
    // find the triggering item, find its menu, and repeat the process. If there
    // is no triggering item, we are done.
    get$h(byMenu, finish).bind((triggerItem) => get$h(items, triggerItem).bind((triggerMenu) => {
        const rest = trace(items, byItem, byMenu, triggerMenu);
        return Optional.some([triggerMenu].concat(rest));
    })).getOr([]);
    const generate$2 = (menus, expansions) => {
        const items = {};
        each(menus, (menuItems, menu) => {
            each$1(menuItems, (item) => {
                items[item] = menu;
            });
        });
        const byItem = expansions;
        const byMenu = transpose$1(expansions);
        // For each menu, calculate the backlog of submenus to get to it.
        const menuPaths = map$1(byMenu, (_triggerItem, submenu) => [submenu].concat(trace(items, byItem, byMenu, submenu)));
        return map$1(items, (menu) => get$h(menuPaths, menu).getOr([menu]));
    };

    const init$2 = () => {
        const expansions = Cell({});
        const menus = Cell({});
        const paths = Cell({});
        const primary = value$2();
        // Probably think of a better way to store this information.
        const directory = Cell({});
        const clear = () => {
            expansions.set({});
            menus.set({});
            paths.set({});
            primary.clear();
        };
        const isClear = () => primary.get().isNone();
        const setMenuBuilt = (menuName, built) => {
            menus.set({
                ...menus.get(),
                [menuName]: {
                    type: 'prepared',
                    menu: built
                }
            });
        };
        const setContents = (sPrimary, sMenus, sExpansions, dir) => {
            primary.set(sPrimary);
            expansions.set(sExpansions);
            menus.set(sMenus);
            directory.set(dir);
            const sPaths = generate$2(dir, sExpansions);
            paths.set(sPaths);
        };
        const getTriggeringItem = (menuValue) => find$4(expansions.get(), (v, _k) => v === menuValue);
        const getTriggerData = (menuValue, getItemByValue, path) => getPreparedMenu(menuValue).bind((menu) => getTriggeringItem(menuValue).bind((triggeringItemValue) => getItemByValue(triggeringItemValue).map((triggeredItem) => ({
            triggeredMenu: menu,
            triggeringItem: triggeredItem,
            triggeringPath: path
        }))));
        const getTriggeringPath = (itemValue, getItemByValue) => {
            // Get the path up to the last item
            const extraPath = filter$2(lookupItem(itemValue).toArray(), (menuValue) => getPreparedMenu(menuValue).isSome());
            return get$h(paths.get(), itemValue).bind((path) => {
                // remember the path is [ most-recent-menu, next-most-recent-menu ]
                // convert each menu identifier into { triggeringItem: comp, menu: comp }
                // could combine into a fold ... probably a left to reverse ... but we'll take the
                // straightforward version when prototyping
                const revPath = reverse(extraPath.concat(path));
                const triggers = bind$3(revPath, (menuValue, menuIndex) => 
                // finding menuValue, it should match the trigger
                getTriggerData(menuValue, getItemByValue, revPath.slice(0, menuIndex + 1)).fold(() => is$1(primary.get(), menuValue) ? [] : [Optional.none()], (data) => [Optional.some(data)]));
                // Convert List<Optional<X>> to Optional<List<X>> if ALL are Some
                return sequence(triggers);
            });
        };
        // Given an item, return a list of all menus including the one that it triggered (if there is one)
        const expand = (itemValue) => get$h(expansions.get(), itemValue).map((menu) => {
            const current = get$h(paths.get(), itemValue).getOr([]);
            return [menu].concat(current);
        });
        const collapse = (itemValue) => 
        // Look up which key has the itemValue
        get$h(paths.get(), itemValue).bind((path) => path.length > 1 ? Optional.some(path.slice(1)) : Optional.none());
        const refresh = (itemValue) => get$h(paths.get(), itemValue);
        const getPreparedMenu = (menuValue) => lookupMenu(menuValue).bind(extractPreparedMenu);
        const lookupMenu = (menuValue) => get$h(menus.get(), menuValue);
        const lookupItem = (itemValue) => get$h(expansions.get(), itemValue);
        const otherMenus = (path) => {
            const menuValues = directory.get();
            return difference(keys(menuValues), path);
        };
        const getPrimary = () => primary.get().bind(getPreparedMenu);
        const getMenus = () => menus.get();
        return {
            setMenuBuilt,
            setContents,
            expand,
            refresh,
            collapse,
            lookupMenu,
            lookupItem,
            otherMenus,
            getPrimary,
            getMenus,
            clear,
            isClear,
            getTriggeringPath
        };
    };
    const extractPreparedMenu = (prep) => prep.type === 'prepared' ? Optional.some(prep.menu) : Optional.none();
    const LayeredState = {
        init: init$2,
        extractPreparedMenu
    };

    const onMenuItemHighlightedEvent = generate$6('tiered-menu-item-highlight');
    const onMenuItemDehighlightedEvent = generate$6('tiered-menu-item-dehighlight');

    const make$5 = (detail, _rawUiSpec) => {
        const submenuParentItems = value$2();
        // So the way to provide extra configuration for the menus that tiered menus create is just
        // to provide different menu specs when building up the TieredData. The TieredMenu itself
        // does not control it, except to set: markers, fakeFocus, onHighlight, and focusManager
        const buildMenus = (container, primaryName, menus) => map$1(menus, (spec, name) => {
            const makeSketch = () => Menu.sketch({
                ...spec,
                value: name,
                // The TieredMenu markers should be inherited by the Menu. "Markers" are things like
                // what is the class for the currently selected item
                markers: detail.markers,
                // If the TieredMenu has been configured with FakeFocus, it needs the menus that it generates
                // to preserve that configuration. Generally, FakeFocus is used for situations where the user
                // wants to keep focus inside some editable element (like an input, or editor content)
                fakeFocus: detail.fakeFocus,
                // The TieredMenu detail.onHighlight function only relates to selecting an item,
                // not a menu, and the menuComp it is passed is the menu, not the tiered menu.
                // This makes it a difficult handler to use for a tieredmenu, so we are
                // deprecating it.
                onHighlight: (menuComp, itemComp) => {
                    // Trigger an internal event so that we can listen to it at the tieredmenu
                    // level, and call detail.onHighlightItem handler with tmenu, menu, and item.
                    const highlightData = {
                        menuComp,
                        itemComp
                    };
                    emitWith(menuComp, onMenuItemHighlightedEvent, highlightData);
                },
                onDehighlight: (menuComp, itemComp) => {
                    const dehighlightData = {
                        menuComp,
                        itemComp
                    };
                    // Trigger an internal event so that we can listen to it at the tieredmenu
                    // level, and call detail.onDehighlightItem handler with tmenu, menu, and item.
                    emitWith(menuComp, onMenuItemDehighlightedEvent, dehighlightData);
                },
                // The Menu itself doesn't set the focusManager based on the value of fakeFocus. It only uses
                // its fakeFocus configuration for creating items that ignore focus, but it still needs to be
                // told which focusManager to use. Perhaps we should change this, though it does allow for more
                // complex focusManagers in single menus.
                focusManager: detail.fakeFocus ? highlights() : dom$2()
            });
            // Only build the primary at first. Build the others as needed.
            return name === primaryName ? {
                type: 'prepared',
                menu: container.getSystem().build(makeSketch())
            } : {
                type: 'notbuilt',
                nbMenu: makeSketch
            };
        });
        const layeredState = LayeredState.init();
        const setup = (container) => {
            const componentMap = buildMenus(container, detail.data.primary, detail.data.menus);
            const directory = toDirectory();
            layeredState.setContents(detail.data.primary, componentMap, detail.data.expansions, directory);
            return layeredState.getPrimary();
        };
        const getItemValue = (item) => Representing.getValue(item).value;
        // Find the first item with value `itemValue` in any of the menus inside this tiered menu structure
        const getItemByValue = (_container, menus, itemValue) => 
        // Can *greatly* improve the performance of this by calculating things up front.
        findMap(menus, (menu) => {
            if (!menu.getSystem().isConnected()) {
                return Optional.none();
            }
            const candidates = Highlighting.getCandidates(menu);
            return find$5(candidates, (c) => getItemValue(c) === itemValue);
        });
        const toDirectory = (_container) => map$1(detail.data.menus, (data, _menuName) => bind$3(data.items, (item) => item.type === 'separator' ? [] : [item.data.value]));
        // This just sets the active menu. It will not set any active items.
        const setActiveMenu = Highlighting.highlight;
        // The item highlighted as active is either the currently active item in the menu,
        // or the first one.
        const setActiveMenuAndItem = (container, menu) => {
            // Firstly, choose the active menu
            setActiveMenu(container, menu);
            // Then, choose the active item inside the active menu
            Highlighting.getHighlighted(menu).orThunk(() => Highlighting.getFirst(menu)).each((item) => {
                if (detail.fakeFocus) {
                    // When using fakeFocus, the items won't have a tab-index, so calling focusItem on them
                    // won't do anything. So we need to manually call highlighting, which is what fakeFocus
                    // uses. It would probably be better to use the focusManager specified.
                    Highlighting.highlight(menu, item);
                }
                else {
                    // We don't just use Focusing.focus here, because some items can have slightly different
                    // handling when they respond to a focusItem event. Widgets with autofocus, for example,
                    // will trigger a Keying.focusIn instead of Focusing.focus call, because they want to move
                    // the focus _inside_ the widget, not just to its outer level. The focusItem event
                    // performs a similar purpose to SystemEvents.focus() and potentially, could be consolidated.
                    dispatch(container, item.element, focusItem());
                }
            });
        };
        const getMenus = (state, menuValues) => cat(map$2(menuValues, (mv) => state.lookupMenu(mv).bind((prep) => prep.type === 'prepared' ? Optional.some(prep.menu) : Optional.none())));
        const closeOthers = (container, state, path) => {
            const others = getMenus(state, state.otherMenus(path));
            each$1(others, (o) => {
                // May not need to do the active menu thing.
                remove$2(o.element, [detail.markers.backgroundMenu]);
                if (!detail.stayInDom) {
                    Replacing.remove(container, o);
                }
            });
        };
        const getSubmenuParents = (container) => submenuParentItems.get().getOrThunk(() => {
            const r = {};
            const items = descendants(container.element, `.${detail.markers.item}`);
            const parentItems = filter$2(items, (i) => get$g(i, 'aria-haspopup') === 'true');
            each$1(parentItems, (i) => {
                container.getSystem().getByDom(i).each((itemComp) => {
                    const key = getItemValue(itemComp);
                    r[key] = itemComp;
                });
            });
            submenuParentItems.set(r);
            return r;
        });
        // Not ideal. Ideally, we would like a map of item keys to components.
        const updateAriaExpansions = (container, path) => {
            const parentItems = getSubmenuParents(container);
            each(parentItems, (v, k) => {
                // Really should turn path into a Set
                const expanded = contains$2(path, k);
                set$9(v.element, 'aria-expanded', expanded);
            });
        };
        const updateMenuPath = (container, state, path) => Optional.from(path[0]).bind((latestMenuName) => state.lookupMenu(latestMenuName).bind((menuPrep) => {
            if (menuPrep.type === 'notbuilt') {
                return Optional.none();
            }
            else {
                const activeMenu = menuPrep.menu;
                const rest = getMenus(state, path.slice(1));
                each$1(rest, (r) => {
                    add$2(r.element, detail.markers.backgroundMenu);
                });
                if (!inBody(activeMenu.element)) {
                    Replacing.append(container, premade(activeMenu));
                }
                // Remove the background-menu class from the active menu
                remove$2(activeMenu.element, [detail.markers.backgroundMenu]);
                setActiveMenuAndItem(container, activeMenu);
                closeOthers(container, state, path);
                return Optional.some(activeMenu);
            }
        }));
        let ExpandHighlightDecision;
        (function (ExpandHighlightDecision) {
            ExpandHighlightDecision[ExpandHighlightDecision["HighlightSubmenu"] = 0] = "HighlightSubmenu";
            ExpandHighlightDecision[ExpandHighlightDecision["HighlightParent"] = 1] = "HighlightParent";
        })(ExpandHighlightDecision || (ExpandHighlightDecision = {}));
        const buildIfRequired = (container, menuName, menuPrep) => {
            if (menuPrep.type === 'notbuilt') {
                const menu = container.getSystem().build(menuPrep.nbMenu());
                layeredState.setMenuBuilt(menuName, menu);
                return menu;
            }
            else {
                return menuPrep.menu;
            }
        };
        const expandRight = (container, item, decision = ExpandHighlightDecision.HighlightSubmenu) => {
            if (item.hasConfigured(Disabling) && Disabling.isDisabled(item)) {
                return Optional.some(item);
            }
            else {
                const value = getItemValue(item);
                return layeredState.expand(value).bind((path) => {
                    // Called when submenus are opened by keyboard AND hovering navigation
                    updateAriaExpansions(container, path);
                    // When expanding, always select the first.
                    return Optional.from(path[0]).bind((menuName) => layeredState.lookupMenu(menuName).bind((activeMenuPrep) => {
                        const activeMenu = buildIfRequired(container, menuName, activeMenuPrep);
                        // DUPE with above. Fix later.
                        if (!inBody(activeMenu.element)) {
                            Replacing.append(container, premade(activeMenu));
                        }
                        // updateMenuPath is the code which changes the active menu. We don't always
                        // want to change the active menu. Sometimes, we just want to show it (e.g. hover)
                        detail.onOpenSubmenu(container, item, activeMenu, reverse(path));
                        if (decision === ExpandHighlightDecision.HighlightSubmenu) {
                            Highlighting.highlightFirst(activeMenu);
                            return updateMenuPath(container, layeredState, path);
                        }
                        else {
                            Highlighting.dehighlightAll(activeMenu);
                            return Optional.some(item);
                        }
                    }));
                });
            }
        };
        const collapseLeft = (container, item) => {
            const value = getItemValue(item);
            return layeredState.collapse(value).bind((path) => {
                // Called when submenus are closed because of KEYBOARD navigation
                updateAriaExpansions(container, path);
                return updateMenuPath(container, layeredState, path).map((activeMenu) => {
                    detail.onCollapseMenu(container, item, activeMenu);
                    return activeMenu;
                });
            });
        };
        const updateView = (container, item) => {
            const value = getItemValue(item);
            return layeredState.refresh(value).bind((path) => {
                // Only this function collapses irrelevant submenus when navigating by HOVERING.
                // Does mean this is called twice when navigating by hovering, since both
                // updateView and expandRight are called by the ItemEvents.hover() handler
                updateAriaExpansions(container, path);
                return updateMenuPath(container, layeredState, path);
            });
        };
        const onRight = (container, item) => inside(item.element) ? Optional.none() : expandRight(container, item, ExpandHighlightDecision.HighlightSubmenu);
        const onLeft = (container, item) => 
        // Exclude inputs, textareas etc.
        inside(item.element) ? Optional.none() : collapseLeft(container, item);
        const onEscape = (container, item) => collapseLeft(container, item).orThunk(() => detail.onEscape(container, item).map(() => container) // This should only fire when the user presses ESC ... not any other close.
        );
        const keyOnItem = (f) => (container, simulatedEvent) => {
            // 2022-08-16 This seems to be the only code in alloy that actually uses
            // the getSource aspect of an event. Remember, that this code is firing
            // when an event bubbles up the tiered menu, e.g. left arrow key.
            // The only current code that sets the source manually is in the Widget item
            // type, and it only sets the source when it is using autofocus. Autofocus
            // is used to essentially treat the widget like it is the top-level item, so
            // when events originate from *within* the widget, their source is changed to
            // the top-level item. Consider removing EventSource from alloy altogether.
            return closest$3(simulatedEvent.getSource(), `.${detail.markers.item}`)
                .bind((target) => container.getSystem().getByDom(target).toOptional().bind((item) => f(container, item).map(always)));
        };
        // NOTE: Many of these events rely on identifying the current item by information
        // sent with the event. However, in situations where you are using fakeFocus, but
        // the real focus is still somewhere in the menu (e.g. search bar), this will lead to
        // an incorrect identification of the active item. Ideally, instead of pulling the
        // item from the event, we should just use Highlighting to identify the active item,
        // and operate on it. However, not all events will necessarily have to happen on the
        // active item, so we need to consider all the cases before making this change. For now,
        // there will be a known limitation that if the real focus is still inside the TieredMenu,
        // but the menu is using fakeFocus, then the actions will operate on the wrong targets.
        // A workaround for that is to stop or cut or redispatch the events in whichever
        // component has the real focus.
        // TODO: TINY-9011 Introduce proper handling of fakeFocus in TieredMenu
        const events = derive$2([
            // Set "active-menu" for the menu with focus
            run$1(focus(), (tmenu, simulatedEvent) => {
                // Ensure the item is actually part of this menu structure, and not part of another menu structure that's bubbling.
                const item = simulatedEvent.event.item;
                layeredState.lookupItem(getItemValue(item)).each(() => {
                    const menu = simulatedEvent.event.menu;
                    Highlighting.highlight(tmenu, menu);
                    const value = getItemValue(simulatedEvent.event.item);
                    layeredState.refresh(value).each((path) => closeOthers(tmenu, layeredState, path));
                });
            }),
            runOnExecute$1((component, simulatedEvent) => {
                // Trigger on execute on the targeted element
                // I.e. clicking on menu item
                const target = simulatedEvent.event.target;
                component.getSystem().getByDom(target).each((item) => {
                    const itemValue = getItemValue(item);
                    // INVESTIGATE: I don't know if this is doing anything any more. Check.
                    if (itemValue.indexOf('collapse-item') === 0) {
                        collapseLeft(component, item);
                    }
                    expandRight(component, item, ExpandHighlightDecision.HighlightSubmenu).fold(() => {
                        detail.onExecute(component, item);
                    }, noop);
                });
            }),
            // Open the menu as soon as it is added to the DOM
            runOnAttached((container, _simulatedEvent) => {
                setup(container).each((primary) => {
                    Replacing.append(container, premade(primary));
                    detail.onOpenMenu(container, primary);
                    if (detail.highlightOnOpen === HighlightOnOpen.HighlightMenuAndItem) {
                        setActiveMenuAndItem(container, primary);
                    }
                    else if (detail.highlightOnOpen === HighlightOnOpen.HighlightJustMenu) {
                        setActiveMenu(container, primary);
                    }
                });
            }),
            // Listen to the events bubbling up from menu about highlighting, and trigger
            // our handlers with tmenu, menu and item
            run$1(onMenuItemHighlightedEvent, (tmenuComp, se) => {
                detail.onHighlightItem(tmenuComp, se.event.menuComp, se.event.itemComp);
            }),
            run$1(onMenuItemDehighlightedEvent, (tmenuComp, se) => {
                detail.onDehighlightItem(tmenuComp, se.event.menuComp, se.event.itemComp);
            }),
            ...(detail.navigateOnHover ? [
                // Hide any irrelevant submenus and expand any submenus based
                // on hovered item
                run$1(hover(), (tmenu, simulatedEvent) => {
                    const item = simulatedEvent.event.item;
                    updateView(tmenu, item);
                    expandRight(tmenu, item, ExpandHighlightDecision.HighlightParent);
                    detail.onHover(tmenu, item);
                })
            ] : [])
        ]);
        const getActiveItem = (container) => Highlighting.getHighlighted(container).bind(Highlighting.getHighlighted);
        const collapseMenuApi = (container) => {
            getActiveItem(container).each((currentItem) => {
                collapseLeft(container, currentItem);
            });
        };
        const highlightPrimary = (container) => {
            layeredState.getPrimary().each((primary) => {
                setActiveMenuAndItem(container, primary);
            });
        };
        const extractMenuFromContainer = (container) => Optional.from(container.components()[0]).filter((comp) => get$g(comp.element, 'role') === 'menu');
        const repositionMenus = (container) => {
            // Get the primary menu
            const maybeActivePrimary = layeredState.getPrimary().bind((primary) => 
            // Get the triggering path (item, menu) up to the active item
            getActiveItem(container).bind((currentItem) => {
                const itemValue = getItemValue(currentItem);
                const allMenus = values(layeredState.getMenus());
                const preparedMenus = cat(map$2(allMenus, LayeredState.extractPreparedMenu));
                return layeredState.getTriggeringPath(itemValue, (v) => getItemByValue(container, preparedMenus, v));
            }).map((triggeringPath) => ({ primary, triggeringPath })));
            maybeActivePrimary.fold(() => {
                // When a menu is open but there is no activeItem, we get the menu from the container.
                extractMenuFromContainer(container).each((primaryMenu) => {
                    detail.onRepositionMenu(container, primaryMenu, []);
                });
            }, ({ primary, triggeringPath }) => {
                // Refresh all the menus up to the active item
                detail.onRepositionMenu(container, primary, triggeringPath);
            });
        };
        const apis = {
            collapseMenu: collapseMenuApi,
            highlightPrimary,
            repositionMenus
        };
        return {
            uid: detail.uid,
            dom: detail.dom,
            markers: detail.markers,
            behaviours: augment(detail.tmenuBehaviours, [
                Keying.config({
                    mode: 'special',
                    onRight: keyOnItem(onRight),
                    onLeft: keyOnItem(onLeft),
                    onEscape: keyOnItem(onEscape),
                    focusIn: (container, _keyInfo) => {
                        layeredState.getPrimary().each((primary) => {
                            dispatch(container, primary.element, focusItem());
                        });
                    }
                }),
                // Highlighting is used for highlighting the active menu
                Highlighting.config({
                    highlightClass: detail.markers.selectedMenu,
                    itemClass: detail.markers.menu
                }),
                Composing.config({
                    find: (container) => {
                        return Highlighting.getHighlighted(container);
                    }
                }),
                Replacing.config({})
            ]),
            eventOrder: detail.eventOrder,
            apis,
            events
        };
    };
    const collapseItem$1 = constant$1('collapse-item');

    const tieredData = (primary, menus, expansions) => ({
        primary,
        menus,
        expansions
    });
    const singleData = (name, menu) => ({
        primary: name,
        menus: wrap(name, menu),
        expansions: {}
    });
    const collapseItem = (text) => ({
        value: generate$6(collapseItem$1()),
        meta: {
            text
        }
    });
    const tieredMenu = single({
        name: 'TieredMenu',
        configFields: [
            onStrictKeyboardHandler('onExecute'),
            onStrictKeyboardHandler('onEscape'),
            onStrictHandler('onOpenMenu'),
            onStrictHandler('onOpenSubmenu'),
            onHandler('onRepositionMenu'),
            onHandler('onCollapseMenu'),
            // Ideally, we should validate that this is a valid value, but
            // this is an number-based enum, so it would just be a number.
            defaulted('highlightOnOpen', HighlightOnOpen.HighlightMenuAndItem),
            requiredObjOf('data', [
                required$1('primary'),
                required$1('menus'),
                required$1('expansions')
            ]),
            defaulted('fakeFocus', false),
            onHandler('onHighlightItem'),
            onHandler('onDehighlightItem'),
            onHandler('onHover'),
            tieredMenuMarkers(),
            required$1('dom'),
            defaulted('navigateOnHover', true),
            defaulted('stayInDom', false),
            field('tmenuBehaviours', [Keying, Highlighting, Composing, Replacing]),
            defaulted('eventOrder', {})
        ],
        apis: {
            collapseMenu: (apis, tmenu) => {
                apis.collapseMenu(tmenu);
            },
            // This will highlight the primary menu AND an item in the primary menu
            // Do not use just to set the active menu.
            highlightPrimary: (apis, tmenu) => {
                apis.highlightPrimary(tmenu);
            },
            repositionMenus: (apis, tmenu) => {
                apis.repositionMenus(tmenu);
            }
        },
        factory: make$5,
        extraApis: {
            tieredData,
            singleData,
            collapseItem
        }
    });

    const suffix = constant$1('sink');
    const partType = constant$1(optional({
        name: suffix(),
        overrides: constant$1({
            dom: {
                tag: 'div'
            },
            behaviours: derive$1([
                Positioning.config({
                    // TODO: Make an internal sink also be able to be used with relative layouts
                    useFixed: always
                })
            ]),
            events: derive$2([
                // Sinks should not let keydown or click propagate
                cutter(keydown()),
                cutter(mousedown()),
                cutter(click())
            ])
        })
    }));

    const schema$h = objOfOnly([
        defaulted('isExtraPart', never),
        optionObjOf('fireEventInstead', [
            defaulted('event', dismissRequested())
        ])
    ]);
    const receivingChannel$1 = (rawSpec) => {
        const detail = asRawOrDie$1('Dismissal', schema$h, rawSpec);
        return {
            [dismissPopups()]: {
                schema: objOfOnly([
                    required$1('target')
                ]),
                onReceive: (sandbox, data) => {
                    if (Sandboxing.isOpen(sandbox)) {
                        const isPart = Sandboxing.isPartOf(sandbox, data.target) || detail.isExtraPart(sandbox, data.target);
                        if (!isPart) {
                            detail.fireEventInstead.fold(() => Sandboxing.close(sandbox), (fe) => emit(sandbox, fe.event));
                        }
                    }
                }
            }
        };
    };

    const schema$g = objOfOnly([
        optionObjOf('fireEventInstead', [
            defaulted('event', repositionRequested())
        ]),
        requiredFunction('doReposition')
    ]);
    const receivingChannel = (rawSpec) => {
        const detail = asRawOrDie$1('Reposition', schema$g, rawSpec);
        return {
            [repositionPopups()]: {
                onReceive: (sandbox) => {
                    if (Sandboxing.isOpen(sandbox)) {
                        detail.fireEventInstead.fold(() => detail.doReposition(sandbox), (fe) => emit(sandbox, fe.event));
                    }
                }
            }
        };
    };

    const getAnchor = (detail, component) => {
        const hotspot = detail.getHotspot(component).getOr(component);
        const type = 'hotspot';
        const overrides = detail.getAnchorOverrides();
        return detail.layouts.fold(() => ({ type, hotspot, overrides }), (layouts) => ({ type, hotspot, overrides, layouts }));
    };
    const fetch$1 = (detail, mapFetch, component) => {
        const fetcher = detail.fetch;
        return fetcher(component).map(mapFetch);
    };
    const openF = (detail, mapFetch, anchor, component, sandbox, externals, highlightOnOpen) => {
        const futureData = fetch$1(detail, mapFetch, component);
        const getLazySink = getSink(component, detail);
        // TODO: Make this potentially a single menu also
        return futureData.map((tdata) => tdata.bind((data) => {
            const primaryMenu = data.menus[data.primary];
            Optional.from(primaryMenu).each((menu) => {
                detail.listRole.each((listRole) => {
                    menu.role = listRole;
                });
            });
            return Optional.from(tieredMenu.sketch({
                // Externals are configured by the "menu" part. It's called external because it isn't contained
                // within the DOM descendants of the dropdown. You can configure things like `fakeFocus` here.
                ...externals.menu(),
                uid: generate$4(''),
                data,
                highlightOnOpen,
                onOpenMenu: (tmenu, menu) => {
                    const sink = getLazySink().getOrDie();
                    Positioning.position(sink, menu, { anchor });
                    Sandboxing.decloak(sandbox);
                },
                onOpenSubmenu: (tmenu, item, submenu) => {
                    const sink = getLazySink().getOrDie();
                    Positioning.position(sink, submenu, {
                        anchor: {
                            type: 'submenu',
                            item
                        }
                    });
                    Sandboxing.decloak(sandbox);
                },
                onRepositionMenu: (tmenu, primaryMenu, submenuTriggers) => {
                    const sink = getLazySink().getOrDie();
                    Positioning.position(sink, primaryMenu, { anchor });
                    each$1(submenuTriggers, (st) => {
                        Positioning.position(sink, st.triggeredMenu, {
                            anchor: { type: 'submenu', item: st.triggeringItem }
                        });
                    });
                },
                onEscape: () => {
                    // Focus the triggering component after escaping the menu
                    Focusing.focus(component);
                    Sandboxing.close(sandbox);
                    return Optional.some(true);
                }
            }));
        }));
    };
    // onOpenSync is because some operations need to be applied immediately, not wrapped in a future
    // It can avoid things like flickering due to asynchronous bouncing
    const open = (detail, mapFetch, hotspot, sandbox, externals, onOpenSync, highlightOnOpen) => {
        const anchor = getAnchor(detail, hotspot);
        const processed = openF(detail, mapFetch, anchor, hotspot, sandbox, externals, highlightOnOpen);
        return processed.map((tdata) => {
            // If we have data, display a menu. Else, close the menu if it was open
            tdata.fold(() => {
                if (Sandboxing.isOpen(sandbox)) {
                    Sandboxing.close(sandbox);
                }
            }, (data) => {
                Sandboxing.cloak(sandbox);
                Sandboxing.open(sandbox, data);
                onOpenSync(sandbox);
            });
            return sandbox;
        });
    };
    const close = (detail, mapFetch, component, sandbox, _externals, _onOpenSync, _highlightOnOpen) => {
        Sandboxing.close(sandbox);
        return Future.pure(sandbox);
    };
    const togglePopup = (detail, mapFetch, hotspot, externals, onOpenSync, highlightOnOpen) => {
        const sandbox = Coupling.getCoupled(hotspot, 'sandbox');
        const showing = Sandboxing.isOpen(sandbox);
        const action = showing ? close : open;
        return action(detail, mapFetch, hotspot, sandbox, externals, onOpenSync, highlightOnOpen);
    };
    const matchWidth = (hotspot, container, useMinWidth) => {
        const menu = Composing.getCurrent(container).getOr(container);
        const buttonWidth = get$c(hotspot.element);
        if (useMinWidth) {
            set$7(menu.element, 'min-width', buttonWidth + 'px');
        }
        else {
            set$6(menu.element, buttonWidth);
        }
    };
    const getSink = (anyInSystem, sinkDetail) => anyInSystem
        .getSystem()
        .getByUid(sinkDetail.uid + '-' + suffix())
        .map((internalSink) => () => Result.value(internalSink))
        .getOrThunk(() => sinkDetail.lazySink.fold(() => () => Result.error(new Error('No internal sink is specified, nor could an external sink be found')), (lazySinkFn) => () => lazySinkFn(anyInSystem)));
    const doRepositionMenus = (sandbox) => {
        Sandboxing.getState(sandbox).each((tmenu) => {
            tieredMenu.repositionMenus(tmenu);
        });
    };
    const makeSandbox$1 = (detail, hotspot, extras) => {
        const ariaControls = manager();
        const onOpen = (component, menu) => {
            const anchor = getAnchor(detail, hotspot);
            ariaControls.link(hotspot.element);
            if (detail.matchWidth) {
                matchWidth(anchor.hotspot, menu, detail.useMinWidth);
            }
            detail.onOpen(anchor, component, menu);
            if (extras !== undefined && extras.onOpen !== undefined) {
                extras.onOpen(component, menu);
            }
        };
        const onClose = (component, menu) => {
            ariaControls.unlink(hotspot.element);
            lazySink().getOr(menu).element.dom.dispatchEvent(new window.FocusEvent('focusout'));
            if (extras !== undefined && extras.onClose !== undefined) {
                extras.onClose(component, menu);
            }
        };
        const lazySink = getSink(hotspot, detail);
        return {
            dom: {
                tag: 'div',
                classes: detail.sandboxClasses,
                // TODO: Add aria-selected attribute
                attributes: {
                    id: ariaControls.id,
                }
            },
            behaviours: SketchBehaviours.augment(detail.sandboxBehaviours, [
                Representing.config({
                    store: {
                        mode: 'memory',
                        initialValue: hotspot
                    }
                }),
                Sandboxing.config({
                    onOpen,
                    onClose,
                    isPartOf: (container, data, queryElem) => {
                        return isPartOf(data, queryElem) || isPartOf(hotspot, queryElem);
                    },
                    getAttachPoint: () => {
                        return lazySink().getOrDie();
                    }
                }),
                // The Composing of the dropdown here is the the active menu of the TieredMenu
                // inside the sandbox.
                Composing.config({
                    find: (sandbox) => {
                        return Sandboxing.getState(sandbox).bind((menu) => Composing.getCurrent(menu));
                    }
                }),
                Receiving.config({
                    channels: {
                        ...receivingChannel$1({
                            isExtraPart: never
                        }),
                        ...receivingChannel({
                            doReposition: doRepositionMenus
                        })
                    }
                })
            ])
        };
    };
    const repositionMenus = (comp) => {
        const sandbox = Coupling.getCoupled(comp, 'sandbox');
        doRepositionMenus(sandbox);
    };

    // TODO: Roll this back into Fields at some point
    // Unfortunately there appears to be a cyclical dependency or something that's preventing it, but for now this will do as it's home
    const sandboxFields = () => [
        defaulted('sandboxClasses', []),
        SketchBehaviours.field('sandboxBehaviours', [Composing, Receiving, Sandboxing, Representing])
    ];

    const schema$f = constant$1([
        required$1('dom'),
        required$1('fetch'),
        onHandler('onOpen'),
        onKeyboardHandler('onExecute'),
        defaulted('getHotspot', Optional.some),
        defaulted('getAnchorOverrides', constant$1({})),
        schema$n(),
        field('dropdownBehaviours', [Toggling, Coupling, Keying, Focusing]),
        required$1('toggleClass'),
        defaulted('eventOrder', {}),
        option$3('lazySink'),
        defaulted('matchWidth', false),
        defaulted('useMinWidth', false),
        option$3('role'),
        option$3('listRole'),
    ].concat(sandboxFields()));
    const parts$c = constant$1([
        external$1({
            schema: [
                tieredMenuMarkers(),
                // Defining a defaulted field isn't necessary when dealing with
                // external parts, because the post-boulder part spec is not passed
                // through to any of these functions (defaults, overrides etc.). So all
                // this does is make it a bit clearer what you should expect, but remember
                // that the default value here is irrelevant!
                defaulted('fakeFocus', false)
            ],
            name: 'menu',
            defaults: (detail) => {
                return {
                    onExecute: detail.onExecute
                };
            }
        }),
        partType()
    ]);

    const factory$k = (detail, components, _spec, externals) => {
        const lookupAttr = (attr) => get$h(detail.dom, 'attributes').bind((attrs) => get$h(attrs, attr));
        const switchToMenu = (sandbox) => {
            Sandboxing.getState(sandbox).each((tmenu) => {
                // This will highlight the menu AND the item
                tieredMenu.highlightPrimary(tmenu);
            });
        };
        const togglePopup$1 = (dropdownComp, onOpenSync, highlightOnOpen) => {
            return togglePopup(detail, identity, dropdownComp, externals, onOpenSync, highlightOnOpen);
        };
        const action = (component) => {
            const onOpenSync = switchToMenu;
            togglePopup$1(component, onOpenSync, HighlightOnOpen.HighlightMenuAndItem).get(noop);
        };
        const apis = {
            expand: (comp) => {
                if (!Toggling.isOn(comp)) {
                    togglePopup$1(comp, noop, HighlightOnOpen.HighlightNone).get(noop);
                }
            },
            open: (comp) => {
                if (!Toggling.isOn(comp)) {
                    togglePopup$1(comp, noop, HighlightOnOpen.HighlightMenuAndItem).get(noop);
                }
            },
            refetch: (comp) => {
                // Generally, the triggers for a refetch should make it so that the
                // sandbox has been created, but it's not guaranteed, so we still handle the
                // case where there isn't yet a sandbox.
                const optSandbox = Coupling.getExistingCoupled(comp, 'sandbox');
                return optSandbox.fold(() => {
                    // If we don't have a sandbox, refetch is the same as open,
                    // except we return when it is completed.
                    return togglePopup$1(comp, noop, HighlightOnOpen.HighlightMenuAndItem)
                        .map(noop);
                }, (sandboxComp) => {
                    // We are intentionally not preserving the selected items when
                    // triggering a refetch, and will just highlight the first item.
                    // Note: this will mean that submenus will close. If we want to start
                    // preserving the selected items, we can't rely on the components themselves,
                    // so we'd need to use the item and menu values through Representing.
                    // However, be aware that alloy menus and items often have randomised values,
                    // so these might not be reliable either.
                    // NOTE: We use DropdownUtils.open directly, because we want it to 'open',
                    // even if it's already open. If we just used apis.open, it wouldn't do
                    // anything if it was already open, which means we wouldn't see the new
                    // refetched data.
                    return open(detail, identity, comp, 
                    // NOTE: The TieredMenu is inside the sandbox. They aren't the same component.
                    sandboxComp, externals, noop, HighlightOnOpen.HighlightMenuAndItem).map(noop);
                });
            },
            isOpen: Toggling.isOn,
            close: (comp) => {
                if (Toggling.isOn(comp)) {
                    togglePopup$1(comp, noop, HighlightOnOpen.HighlightMenuAndItem).get(noop);
                }
            },
            // If we are open, refresh the menus in the tiered menu system
            repositionMenus: (comp) => {
                if (Toggling.isOn(comp)) {
                    repositionMenus(comp);
                }
            }
        };
        const triggerExecute = (comp, _se) => {
            emitExecute(comp);
            return Optional.some(true);
        };
        return {
            uid: detail.uid,
            dom: detail.dom,
            components,
            behaviours: augment(detail.dropdownBehaviours, [
                Toggling.config({
                    toggleClass: detail.toggleClass,
                    aria: {
                        mode: 'expanded'
                    }
                }),
                Coupling.config({
                    others: {
                        sandbox: (hotspot) => {
                            return makeSandbox$1(detail, hotspot, {
                                onOpen: () => Toggling.on(hotspot),
                                onClose: () => Toggling.off(hotspot)
                            });
                        }
                    }
                }),
                Keying.config({
                    mode: 'special',
                    onSpace: triggerExecute,
                    onEnter: triggerExecute,
                    onDown: (comp, _se) => {
                        if (Dropdown.isOpen(comp)) {
                            const sandbox = Coupling.getCoupled(comp, 'sandbox');
                            switchToMenu(sandbox);
                        }
                        else {
                            Dropdown.open(comp);
                        }
                        return Optional.some(true);
                    },
                    onEscape: (comp, _se) => {
                        if (Dropdown.isOpen(comp)) {
                            Dropdown.close(comp);
                            return Optional.some(true);
                        }
                        else {
                            return Optional.none();
                        }
                    }
                }),
                Focusing.config({})
            ]),
            events: events(Optional.some(action)),
            eventOrder: {
                ...detail.eventOrder,
                // Order, the button state is toggled first, so assumed !selected means close.
                [execute$5()]: ['disabling', 'toggling', 'alloy.base.behaviour']
            },
            apis,
            domModification: {
                attributes: {
                    'aria-haspopup': detail.listRole.getOr('true'),
                    ...detail.role.fold(() => ({}), (role) => ({ role })),
                    ...detail.dom.tag === 'button' ? { type: lookupAttr('type').getOr('button') } : {}
                }
            }
        };
    };
    const Dropdown = composite({
        name: 'Dropdown',
        configFields: schema$f(),
        partFields: parts$c(),
        factory: factory$k,
        apis: {
            open: (apis, comp) => apis.open(comp),
            refetch: (apis, comp) => apis.refetch(comp),
            expand: (apis, comp) => apis.expand(comp),
            close: (apis, comp) => apis.close(comp),
            isOpen: (apis, comp) => apis.isOpen(comp),
            repositionMenus: (apis, comp) => apis.repositionMenus(comp)
        }
    });

    const owner$1 = 'form';
    const schema$e = [
        field('formBehaviours', [Representing])
    ];
    const getPartName$1 = (name) => '<alloy.field.' + name + '>';
    const sketch$2 = (fSpec) => {
        const parts = (() => {
            const record = [];
            const field = (name, config) => {
                record.push(name);
                return generateOne$1(owner$1, getPartName$1(name), config);
            };
            return {
                field,
                record: constant$1(record)
            };
        })();
        const spec = fSpec(parts);
        const partNames = parts.record();
        // Unlike other sketches, a form does not know its parts in advance (as they represent each field
        // in a particular form). Therefore, it needs to calculate the part names on the fly
        const fieldParts = map$2(partNames, (n) => required({ name: n, pname: getPartName$1(n) }));
        return composite$1(owner$1, schema$e, fieldParts, make$4, spec);
    };
    const toResult = (o, e) => o.fold(() => Result.error(e), Result.value);
    const make$4 = (detail, components) => ({
        uid: detail.uid,
        dom: detail.dom,
        components,
        // Form has an assumption that every field must have composing, and that the composed element has representing.
        behaviours: augment(detail.formBehaviours, [
            Representing.config({
                store: {
                    mode: 'manual',
                    getValue: (form) => {
                        const resPs = getAllParts(form, detail);
                        return map$1(resPs, (resPThunk, pName) => resPThunk().bind((v) => {
                            const opt = Composing.getCurrent(v);
                            return toResult(opt, new Error(`Cannot find a current component to extract the value from for form part '${pName}': ` + element(v.element)));
                        }).map(Representing.getValue));
                    },
                    setValue: (form, values) => {
                        each(values, (newValue, key) => {
                            getPart(form, detail, key).each((wrapper) => {
                                Composing.getCurrent(wrapper).each((field) => {
                                    Representing.setValue(field, newValue);
                                });
                            });
                        });
                    }
                }
            })
        ]),
        apis: {
            getField: (form, key) => {
                // Returns an Optional (not a result);
                return getPart(form, detail, key).bind(Composing.getCurrent);
            }
        }
    });
    const Form = {
        getField: makeApi((apis, component, key) => apis.getField(component, key)),
        sketch: sketch$2
    };

    const schema$d = constant$1([
        required$1('dom'),
        defaulted('shell', true),
        field('toolbarBehaviours', [Replacing])
    ]);
    // TODO: Dupe with Toolbar
    const enhanceGroups = () => ({
        behaviours: derive$1([
            Replacing.config({})
        ])
    });
    const parts$b = constant$1([
        // Note, is the container for putting all the groups in, not a group itself.
        optional({
            name: 'groups',
            overrides: enhanceGroups
        })
    ]);

    const factory$j = (detail, components, _spec, _externals) => {
        const setGroups = (toolbar, groups) => {
            getGroupContainer(toolbar).fold(() => {
                // check that the group container existed. It may not have if the components
                // did not list anything, and shell was false.
                // eslint-disable-next-line no-console
                console.error('Toolbar was defined to not be a shell, but no groups container was specified in components');
                throw new Error('Toolbar was defined to not be a shell, but no groups container was specified in components');
            }, (container) => {
                Replacing.set(container, groups);
            });
        };
        const getGroupContainer = (component) => detail.shell ? Optional.some(component) : getPart(component, detail, 'groups');
        // In shell mode, the group overrides need to be added to the main container, and there can be no children
        const extra = detail.shell ? { behaviours: [Replacing.config({})], components: [] } : { behaviours: [], components };
        return {
            uid: detail.uid,
            dom: detail.dom,
            components: extra.components,
            behaviours: augment(detail.toolbarBehaviours, extra.behaviours),
            apis: {
                setGroups,
                refresh: noop
            },
            domModification: {
                attributes: {
                    role: 'group'
                }
            }
        };
    };
    const Toolbar = composite({
        name: 'Toolbar',
        configFields: schema$d(),
        partFields: parts$b(),
        factory: factory$j,
        apis: {
            setGroups: (apis, toolbar, groups) => {
                apis.setGroups(toolbar, groups);
            }
        }
    });

    const schema$c = constant$1([
        markers$1(['toggledClass']),
        required$1('lazySink'),
        requiredFunction('fetch'),
        optionFunction('getBounds'),
        optionObjOf('fireDismissalEventInstead', [
            defaulted('event', dismissRequested())
        ]),
        schema$n(),
        onHandler('onToggled'),
    ]);
    const parts$a = constant$1([
        external$1({
            name: 'button',
            overrides: (detail) => ({
                dom: {
                    attributes: {
                        'aria-haspopup': 'true'
                    }
                },
                buttonBehaviours: derive$1([
                    Toggling.config({
                        toggleClass: detail.markers.toggledClass,
                        aria: {
                            mode: 'expanded'
                        },
                        toggleOnExecute: false,
                        /**
                         * For FloatingToolbars, we can hook up our `onToggled` handler directly to the Toggling
                         * because we don't have to worry about any animations.
                         *
                         * Unfortunately, for SlidingToolbars, Toggling is more directly hooked into the animation for growing,
                         * so to have an event `onToggled` that doesn't care about the animation, we can't just hook into the Toggling config.
                         */
                        onToggled: detail.onToggled
                    })
                ])
            })
        }),
        external$1({
            factory: Toolbar,
            schema: schema$d(),
            name: 'toolbar',
            overrides: (detail) => {
                return {
                    toolbarBehaviours: derive$1([
                        Keying.config({
                            mode: 'cyclic',
                            onEscape: (comp) => {
                                getPart(comp, detail, 'button').each(Focusing.focus);
                                // Don't return true here, as we need to allow the sandbox to handle the escape to close the overflow
                                return Optional.none();
                            }
                        })
                    ])
                };
            }
        })
    ]);

    const shouldSkipFocus = value$2();
    const toggleWithoutFocusing = (button, externals) => {
        shouldSkipFocus.set(true);
        toggle$1(button, externals);
        shouldSkipFocus.clear();
    };
    const toggle$1 = (button, externals) => {
        const toolbarSandbox = Coupling.getCoupled(button, 'toolbarSandbox');
        if (Sandboxing.isOpen(toolbarSandbox)) {
            Sandboxing.close(toolbarSandbox);
        }
        else {
            Sandboxing.open(toolbarSandbox, externals.toolbar());
        }
    };
    const position = (button, toolbar, detail, layouts) => {
        const bounds = detail.getBounds.map((bounder) => bounder());
        const sink = detail.lazySink(button).getOrDie();
        Positioning.positionWithinBounds(sink, toolbar, {
            anchor: {
                type: 'hotspot',
                hotspot: button,
                layouts,
                overrides: {
                    maxWidthFunction: expandable()
                }
            }
        }, bounds);
    };
    const setGroups$1 = (button, toolbar, detail, layouts, groups) => {
        Toolbar.setGroups(toolbar, groups);
        position(button, toolbar, detail, layouts);
        Toggling.on(button);
    };
    const makeSandbox = (button, spec, detail) => {
        const ariaControls = manager();
        const onOpen = (sandbox, toolbar) => {
            const skipFocus = shouldSkipFocus.get().getOr(false);
            detail.fetch().get((groups) => {
                setGroups$1(button, toolbar, detail, spec.layouts, groups);
                ariaControls.link(button.element);
                if (!skipFocus) {
                    Keying.focusIn(toolbar);
                }
            });
        };
        const onClose = () => {
            // Toggle and focus the button
            Toggling.off(button);
            if (!shouldSkipFocus.get().getOr(false)) {
                Focusing.focus(button);
            }
            ariaControls.unlink(button.element);
        };
        return {
            dom: {
                tag: 'div',
                attributes: {
                    id: ariaControls.id
                }
            },
            behaviours: derive$1([
                Keying.config({
                    mode: 'special',
                    onEscape: (comp) => {
                        Sandboxing.close(comp);
                        return Optional.some(true);
                    }
                }),
                Sandboxing.config({
                    onOpen,
                    onClose,
                    isPartOf: (container, data, queryElem) => {
                        return isPartOf(data, queryElem) || isPartOf(button, queryElem);
                    },
                    getAttachPoint: () => {
                        return detail.lazySink(button).getOrDie();
                    }
                }),
                Receiving.config({
                    channels: {
                        ...receivingChannel$1({
                            isExtraPart: never,
                            ...detail.fireDismissalEventInstead.map((fe) => ({ fireEventInstead: { event: fe.event } })).getOr({})
                        }),
                        ...receivingChannel({
                            doReposition: () => {
                                Sandboxing.getState(Coupling.getCoupled(button, 'toolbarSandbox')).each((toolbar) => {
                                    position(button, toolbar, detail, spec.layouts);
                                });
                            }
                        })
                    }
                })
            ])
        };
    };
    const factory$i = (detail, components, spec, externals) => ({
        ...Button.sketch({
            ...externals.button(),
            action: (button) => {
                toggle$1(button, externals);
            },
            buttonBehaviours: SketchBehaviours.augment({ dump: externals.button().buttonBehaviours }, [
                Coupling.config({
                    others: {
                        toolbarSandbox: (button) => {
                            return makeSandbox(button, spec, detail);
                        }
                    }
                })
            ])
        }),
        apis: {
            setGroups: (button, groups) => {
                Sandboxing.getState(Coupling.getCoupled(button, 'toolbarSandbox')).each((toolbar) => {
                    setGroups$1(button, toolbar, detail, spec.layouts, groups);
                });
            },
            reposition: (button) => {
                Sandboxing.getState(Coupling.getCoupled(button, 'toolbarSandbox')).each((toolbar) => {
                    position(button, toolbar, detail, spec.layouts);
                });
            },
            toggle: (button) => {
                toggle$1(button, externals);
            },
            toggleWithoutFocusing: (button) => {
                toggleWithoutFocusing(button, externals);
            },
            getToolbar: (button) => {
                return Sandboxing.getState(Coupling.getCoupled(button, 'toolbarSandbox'));
            },
            isOpen: (button) => {
                return Sandboxing.isOpen(Coupling.getCoupled(button, 'toolbarSandbox'));
            }
        }
    });
    const FloatingToolbarButton = composite({
        name: 'FloatingToolbarButton',
        factory: factory$i,
        configFields: schema$c(),
        partFields: parts$a(),
        apis: {
            setGroups: (apis, button, groups) => {
                apis.setGroups(button, groups);
            },
            reposition: (apis, button) => {
                apis.reposition(button);
            },
            toggle: (apis, button) => {
                apis.toggle(button);
            },
            toggleWithoutFocusing: (apis, button) => {
                apis.toggleWithoutFocusing(button);
            },
            getToolbar: (apis, button) => apis.getToolbar(button),
            isOpen: (apis, button) => apis.isOpen(button)
        }
    });

    const schema$b = constant$1([
        defaulted('prefix', 'form-field'),
        field('fieldBehaviours', [Composing, Representing])
    ]);
    const parts$9 = constant$1([
        optional({
            schema: [required$1('dom')],
            name: 'label'
        }),
        optional({
            factory: {
                sketch: (spec) => {
                    return {
                        uid: spec.uid,
                        dom: {
                            tag: 'span',
                            styles: {
                                display: 'none'
                            },
                            attributes: {
                                'aria-hidden': 'true'
                            },
                            innerHtml: spec.text
                        }
                    };
                }
            },
            schema: [required$1('text')],
            name: 'aria-descriptor'
        }),
        required({
            factory: {
                sketch: (spec) => {
                    const excludeFactory = exclude(spec, ['factory']);
                    return spec.factory.sketch(excludeFactory);
                }
            },
            schema: [required$1('factory')],
            name: 'field'
        })
    ]);

    const factory$h = (detail, components, _spec, _externals) => {
        const behaviours = augment(detail.fieldBehaviours, [
            Composing.config({
                find: (container) => {
                    return getPart(container, detail, 'field');
                }
            }),
            Representing.config({
                store: {
                    mode: 'manual',
                    getValue: (field) => {
                        return Composing.getCurrent(field).bind(Representing.getValue);
                    },
                    setValue: (field, value) => {
                        Composing.getCurrent(field).each((current) => {
                            Representing.setValue(current, value);
                        });
                    }
                }
            })
        ]);
        const events = derive$2([
            // Used to be systemInit
            runOnAttached((component, _simulatedEvent) => {
                const ps = getParts(component, detail, ['label', 'field', 'aria-descriptor']);
                ps.field().each((field) => {
                    const id = generate$6(detail.prefix);
                    ps.label().each((label) => {
                        // TODO: Find a nicer way of doing this.
                        set$9(label.element, 'for', id);
                        set$9(field.element, 'id', id);
                    });
                    ps['aria-descriptor']().each((descriptor) => {
                        const descriptorId = generate$6(detail.prefix);
                        set$9(descriptor.element, 'id', descriptorId);
                        set$9(field.element, 'aria-describedby', descriptorId);
                    });
                });
            })
        ]);
        const apis = {
            getField: (container) => getPart(container, detail, 'field'),
            getLabel: (container) => 
            // TODO: Use constants for part names
            getPart(container, detail, 'label')
        };
        return {
            uid: detail.uid,
            dom: detail.dom,
            components,
            behaviours,
            events,
            apis
        };
    };
    const FormField = composite({
        name: 'FormField',
        configFields: schema$b(),
        partFields: parts$9(),
        factory: factory$h,
        apis: {
            getField: (apis, comp) => apis.getField(comp),
            getLabel: (apis, comp) => apis.getLabel(comp)
        }
    });

    const schema$a = constant$1([
        defaulted('field1Name', 'field1'),
        defaulted('field2Name', 'field2'),
        onStrictHandler('onLockedChange'),
        markers$1(['lockClass']),
        defaulted('locked', false),
        SketchBehaviours.field('coupledFieldBehaviours', [Composing, Representing]),
        defaultedFunction('onInput', noop)
    ]);
    const getField = (comp, detail, partName) => getPart(comp, detail, partName).bind(Composing.getCurrent);
    const coupledPart = (selfName, otherName) => required({
        factory: FormField,
        name: selfName,
        overrides: (detail) => {
            return {
                fieldBehaviours: derive$1([
                    config('coupled-input-behaviour', [
                        run$1(input(), (me) => {
                            getField(me, detail, otherName).each((other) => {
                                getPart(me, detail, 'lock').each((lock) => {
                                    // TODO IMPROVEMENT: Allow locker to fire onLockedChange if it is turned on after being off.
                                    if (Toggling.isOn(lock)) {
                                        detail.onLockedChange(me, other, lock);
                                    }
                                    detail.onInput(me);
                                });
                            });
                        })
                    ])
                ])
            };
        }
    });
    const parts$8 = constant$1([
        coupledPart('field1', 'field2'),
        coupledPart('field2', 'field1'),
        required({
            factory: Button,
            schema: [
                required$1('dom')
            ],
            name: 'lock',
            overrides: (detail) => {
                return {
                    buttonBehaviours: derive$1([
                        Toggling.config({
                            selected: detail.locked,
                            toggleClass: detail.markers.lockClass,
                            aria: {
                                mode: 'pressed'
                            }
                        })
                    ])
                };
            }
        })
    ]);

    const factory$g = (detail, components, _spec, _externals) => ({
        uid: detail.uid,
        dom: detail.dom,
        components,
        behaviours: SketchBehaviours.augment(detail.coupledFieldBehaviours, [
            Composing.config({ find: Optional.some }),
            Representing.config({
                store: {
                    mode: 'manual',
                    getValue: (comp) => {
                        const parts = getPartsOrDie(comp, detail, ['field1', 'field2']);
                        return {
                            [detail.field1Name]: Representing.getValue(parts.field1()),
                            [detail.field2Name]: Representing.getValue(parts.field2())
                        };
                    },
                    setValue: (comp, value) => {
                        const parts = getPartsOrDie(comp, detail, ['field1', 'field2']);
                        if (hasNonNullableKey(value, detail.field1Name)) {
                            Representing.setValue(parts.field1(), value[detail.field1Name]);
                        }
                        if (hasNonNullableKey(value, detail.field2Name)) {
                            Representing.setValue(parts.field2(), value[detail.field2Name]);
                        }
                    }
                }
            })
        ]),
        apis: {
            getField1: (component) => getPart(component, detail, 'field1'),
            getField2: (component) => getPart(component, detail, 'field2'),
            getLock: (component) => getPart(component, detail, 'lock')
        }
    });
    const FormCoupledInputs = composite({
        name: 'FormCoupledInputs',
        configFields: schema$a(),
        partFields: parts$8(),
        factory: factory$g,
        apis: {
            getField1: (apis, component) => apis.getField1(component),
            getField2: (apis, component) => apis.getField2(component),
            getLock: (apis, component) => apis.getLock(component)
        }
    });

    const factory$f = (detail, _spec) => {
        const options = map$2(detail.options, (option) => ({
            dom: {
                tag: 'option',
                value: option.value,
                innerHtml: option.text
            }
        }));
        const initialValues = detail.data.map((v) => wrap('initialValue', v)).getOr({});
        return {
            uid: detail.uid,
            dom: {
                tag: 'select',
                classes: detail.selectClasses,
                attributes: detail.selectAttributes
            },
            components: options,
            behaviours: augment(detail.selectBehaviours, [
                Focusing.config({}),
                Representing.config({
                    store: {
                        mode: 'manual',
                        getValue: (select) => {
                            return get$5(select.element);
                        },
                        setValue: (select, newValue) => {
                            const firstOption = head(detail.options);
                            // This is probably generically useful ... may become a part of Representing.
                            const found = find$5(detail.options, (opt) => opt.value === newValue);
                            if (found.isSome()) {
                                set$4(select.element, newValue);
                            }
                            else if (select.element.dom.selectedIndex === -1 && newValue === '') {
                                /*
                                  Sometimes after a redial alloy tries to set a new value, but if no value has been set in the data this used to fail. Now we set the value to the first option in the list if:
                                    The index is out of range, indicating that the list of options have changed, or was never set.
                                    The user is not trying to set a specific value (which would be user error)
                                */
                                firstOption.each((value) => set$4(select.element, value.value));
                            }
                        },
                        ...initialValues
                    }
                })
            ])
        };
    };
    const HtmlSelect = single({
        name: 'HtmlSelect',
        configFields: [
            required$1('options'),
            field('selectBehaviours', [Focusing, Representing]),
            defaulted('selectClasses', []),
            defaulted('selectAttributes', {}),
            option$3('data')
        ],
        factory: factory$f
    });

    const makeMenu = (detail, menuSandbox, placementSpec, menuSpec, getBounds) => {
        const lazySink = () => detail.lazySink(menuSandbox);
        const layouts = menuSpec.type === 'horizontal' ? { layouts: {
                onLtr: () => belowOrAbove(),
                onRtl: () => belowOrAboveRtl()
            } } : {};
        const isFirstTierSubmenu = (triggeringPaths) => triggeringPaths.length === 2; // primary and first tier menu === 2 items
        const getSubmenuLayouts = (triggeringPaths) => isFirstTierSubmenu(triggeringPaths) ? layouts : {};
        return tieredMenu.sketch({
            dom: {
                tag: 'div'
            },
            data: menuSpec.data,
            markers: menuSpec.menu.markers,
            highlightOnOpen: menuSpec.menu.highlightOnOpen,
            fakeFocus: menuSpec.menu.fakeFocus,
            onEscape: () => {
                // Note for the future: this should possibly also call detail.onHide
                Sandboxing.close(menuSandbox);
                detail.onEscape.map((handler) => handler(menuSandbox));
                return Optional.some(true);
            },
            onExecute: () => {
                return Optional.some(true);
            },
            onOpenMenu: (tmenu, menu) => {
                Positioning.positionWithinBounds(lazySink().getOrDie(), menu, placementSpec, getBounds());
            },
            onOpenSubmenu: (tmenu, item, submenu, triggeringPaths) => {
                const sink = lazySink().getOrDie();
                Positioning.position(sink, submenu, {
                    anchor: {
                        type: 'submenu',
                        item,
                        ...getSubmenuLayouts(triggeringPaths)
                    }
                });
            },
            onRepositionMenu: (tmenu, primaryMenu, submenuTriggers) => {
                const sink = lazySink().getOrDie();
                Positioning.positionWithinBounds(sink, primaryMenu, placementSpec, getBounds());
                each$1(submenuTriggers, (st) => {
                    const submenuLayouts = getSubmenuLayouts(st.triggeringPath);
                    Positioning.position(sink, st.triggeredMenu, {
                        anchor: { type: 'submenu', item: st.triggeringItem, ...submenuLayouts }
                    });
                });
            }
        });
    };
    const factory$e = (detail, spec) => {
        const isPartOfRelated = (sandbox, queryElem) => {
            const related = detail.getRelated(sandbox);
            return related.exists((rel) => isPartOf(rel, queryElem));
        };
        const setContent = (sandbox, thing) => {
            // Keep the same location, and just change the content.
            Sandboxing.setContent(sandbox, thing);
        };
        const showAt = (sandbox, thing, placementSpec) => {
            const getBounds = Optional.none;
            showWithinBounds(sandbox, thing, placementSpec, getBounds);
        };
        const showWithinBounds = (sandbox, thing, placementSpec, getBounds) => {
            const sink = detail.lazySink(sandbox).getOrDie();
            Sandboxing.openWhileCloaked(sandbox, thing, () => Positioning.positionWithinBounds(sink, sandbox, placementSpec, getBounds()));
            Representing.setValue(sandbox, Optional.some({
                mode: 'position',
                config: placementSpec,
                getBounds
            }));
        };
        // TODO AP-191 write a test for showMenuAt
        const showMenuAt = (sandbox, placementSpec, menuSpec) => {
            showMenuWithinBounds(sandbox, placementSpec, menuSpec, Optional.none);
        };
        const showMenuWithinBounds = (sandbox, placementSpec, menuSpec, getBounds) => {
            const menu = makeMenu(detail, sandbox, placementSpec, menuSpec, getBounds);
            Sandboxing.open(sandbox, menu);
            Representing.setValue(sandbox, Optional.some({
                mode: 'menu',
                menu
            }));
        };
        const hide = (sandbox) => {
            if (Sandboxing.isOpen(sandbox)) {
                Representing.setValue(sandbox, Optional.none());
                Sandboxing.close(sandbox);
            }
        };
        const getContent = (sandbox) => Sandboxing.getState(sandbox);
        const reposition = (sandbox) => {
            if (Sandboxing.isOpen(sandbox)) {
                Representing.getValue(sandbox).each((state) => {
                    switch (state.mode) {
                        case 'menu':
                            Sandboxing.getState(sandbox).each(tieredMenu.repositionMenus);
                            break;
                        case 'position':
                            const sink = detail.lazySink(sandbox).getOrDie();
                            Positioning.positionWithinBounds(sink, sandbox, state.config, state.getBounds());
                            break;
                    }
                });
            }
        };
        const apis = {
            setContent,
            showAt,
            showWithinBounds,
            showMenuAt,
            showMenuWithinBounds,
            hide,
            getContent,
            reposition,
            isOpen: Sandboxing.isOpen
        };
        return {
            uid: detail.uid,
            dom: detail.dom,
            behaviours: augment(detail.inlineBehaviours, [
                Sandboxing.config({
                    isPartOf: (sandbox, data, queryElem) => {
                        return isPartOf(data, queryElem) || isPartOfRelated(sandbox, queryElem);
                    },
                    getAttachPoint: (sandbox) => {
                        return detail.lazySink(sandbox).getOrDie();
                    },
                    onOpen: (sandbox) => {
                        detail.onShow(sandbox);
                    },
                    onClose: (sandbox) => {
                        detail.onHide(sandbox);
                    }
                }),
                Representing.config({
                    store: {
                        mode: 'memory',
                        initialValue: Optional.none()
                    }
                }),
                Receiving.config({
                    channels: {
                        ...receivingChannel$1({
                            isExtraPart: spec.isExtraPart,
                            ...detail.fireDismissalEventInstead.map((fe) => ({ fireEventInstead: { event: fe.event } })).getOr({})
                        }),
                        ...receivingChannel({
                            ...detail.fireRepositionEventInstead.map((fe) => ({ fireEventInstead: { event: fe.event } })).getOr({}),
                            doReposition: reposition
                        })
                    }
                })
            ]),
            eventOrder: detail.eventOrder,
            apis
        };
    };
    const InlineView = single({
        name: 'InlineView',
        configFields: [
            required$1('lazySink'),
            onHandler('onShow'),
            onHandler('onHide'),
            optionFunction('onEscape'),
            field('inlineBehaviours', [Sandboxing, Representing, Receiving]),
            optionObjOf('fireDismissalEventInstead', [
                defaulted('event', dismissRequested())
            ]),
            optionObjOf('fireRepositionEventInstead', [
                defaulted('event', repositionRequested())
            ]),
            defaulted('getRelated', Optional.none),
            defaulted('isExtraPart', never),
            defaulted('eventOrder', Optional.none)
        ],
        factory: factory$e,
        apis: {
            showAt: (apis, component, anchor, thing) => {
                apis.showAt(component, anchor, thing);
            },
            showWithinBounds: (apis, component, anchor, thing, bounds) => {
                apis.showWithinBounds(component, anchor, thing, bounds);
            },
            showMenuAt: (apis, component, anchor, menuSpec) => {
                apis.showMenuAt(component, anchor, menuSpec);
            },
            showMenuWithinBounds: (apis, component, anchor, menuSpec, bounds) => {
                apis.showMenuWithinBounds(component, anchor, menuSpec, bounds);
            },
            hide: (apis, component) => {
                apis.hide(component);
            },
            isOpen: (apis, component) => apis.isOpen(component),
            getContent: (apis, component) => apis.getContent(component),
            setContent: (apis, component, thing) => {
                apis.setContent(component, thing);
            },
            reposition: (apis, component) => {
                apis.reposition(component);
            }
        }
    });

    const schema$9 = constant$1([
        defaultedString('type', 'text'),
        option$3('data'),
        defaulted('inputAttributes', {}),
        defaulted('inputStyles', {}),
        defaulted('tag', 'input'),
        defaulted('inputClasses', []),
        onHandler('onSetValue'),
        defaultedFunction('fromInputValue', identity),
        defaultedFunction('toInputValue', identity),
        defaulted('styles', {}),
        defaulted('eventOrder', {}),
        field('inputBehaviours', [Representing, Focusing]),
        defaulted('selectOnFocus', true)
    ]);
    const focusBehaviours = (detail) => derive$1([
        Focusing.config({
            onFocus: !detail.selectOnFocus ? noop : (component) => {
                const input = component.element;
                const value = get$5(input);
                // TODO: There are probably more types that can't handle setSelectionRange
                if (detail.type !== 'range') {
                    input.dom.setSelectionRange(0, value.length);
                }
            }
        })
    ]);
    const behaviours = (detail) => ({
        ...focusBehaviours(detail),
        ...augment(detail.inputBehaviours, [
            Representing.config({
                store: {
                    mode: 'manual',
                    // Propagating its Optional
                    ...detail.data.map((data) => ({ initialValue: data })).getOr({}),
                    getValue: (input) => {
                        return detail.fromInputValue(get$5(input.element));
                    },
                    setValue: (input, data) => {
                        const current = get$5(input.element);
                        // Only set it if it has changed ... otherwise the cursor goes to the end.
                        if (current !== data) {
                            set$4(input.element, detail.toInputValue(data));
                        }
                    }
                },
                onSetValue: detail.onSetValue
            })
        ])
    });
    const dom$1 = (detail) => ({
        tag: detail.tag,
        attributes: {
            type: detail.type,
            ...detail.inputAttributes
        },
        styles: detail.inputStyles,
        classes: detail.inputClasses
    });

    const factory$d = (detail, _spec) => ({
        uid: detail.uid,
        dom: dom$1(detail),
        // No children.
        components: [],
        behaviours: behaviours(detail),
        eventOrder: detail.eventOrder
    });
    const Input = single({
        name: 'Input',
        configFields: schema$9(),
        factory: factory$d
    });

    const parts$7 = generate$5(owner$2(), parts$e());

    const labelledBy = (labelledElement, labelElement) => {
        const labelId = getOpt(labelledElement, 'id')
            .fold(() => {
            const id = generate$6('dialog-label');
            set$9(labelElement, 'id', id);
            return id;
        }, identity);
        set$9(labelledElement, 'aria-labelledby', labelId);
    };

    const schema$8 = constant$1([
        required$1('lazySink'),
        option$3('dragBlockClass'),
        defaultedFunction('getBounds', win),
        defaulted('useTabstopAt', always),
        defaulted('firstTabstop', 0),
        defaulted('eventOrder', {}),
        field('modalBehaviours', [Keying]),
        onKeyboardHandler('onExecute'),
        onStrictKeyboardHandler('onEscape')
    ]);
    const basic = { sketch: identity };
    const parts$6 = constant$1([
        optional({
            name: 'draghandle',
            overrides: (detail, spec) => {
                return {
                    behaviours: derive$1([
                        Dragging.config({
                            mode: 'mouse',
                            getTarget: (handle) => {
                                return ancestor$1(handle, '[role="dialog"]').getOr(handle);
                            },
                            blockerClass: detail.dragBlockClass.getOrDie(
                            // TODO: Support errors in Optional getOrDie.
                            new Error('The drag blocker class was not specified for a dialog with a drag handle: \n' +
                                JSON.stringify(spec, null, 2)).message),
                            getBounds: detail.getDragBounds
                        })
                    ])
                };
            }
        }),
        required({
            schema: [required$1('dom')],
            name: 'title'
        }),
        required({
            factory: basic,
            schema: [required$1('dom')],
            name: 'close'
        }),
        required({
            factory: basic,
            schema: [required$1('dom')],
            name: 'body'
        }),
        optional({
            factory: basic,
            schema: [required$1('dom')],
            name: 'footer'
        }),
        external$1({
            factory: {
                sketch: (spec, detail) => 
                // Merging should take care of the uid
                ({
                    ...spec,
                    dom: detail.dom,
                    components: detail.components
                })
            },
            schema: [
                defaulted('dom', {
                    tag: 'div',
                    styles: {
                        position: 'fixed',
                        left: '0px',
                        top: '0px',
                        right: '0px',
                        bottom: '0px'
                    }
                }),
                defaulted('components', [])
            ],
            name: 'blocker'
        })
    ]);

    const factory$c = (detail, components, spec, externals) => {
        const dialogComp = value$2();
        // TODO IMPROVEMENT: Make close actually close the dialog by default!
        const showDialog = (dialog) => {
            dialogComp.set(dialog);
            const sink = detail.lazySink(dialog).getOrDie();
            const externalBlocker = externals.blocker();
            const blocker = sink.getSystem().build({
                ...externalBlocker,
                components: externalBlocker.components.concat([
                    premade(dialog)
                ]),
                behaviours: derive$1([
                    Focusing.config({}),
                    config('dialog-blocker-events', [
                        // Ensure we use runOnSource otherwise this would cause an infinite loop, as `focusIn` would fire a `focusin` which would then get responded to and so forth
                        runOnSource(focusin(), () => {
                            Blocking.isBlocked(dialog) ? noop() : Keying.focusIn(dialog);
                        })
                    ])
                ])
            });
            attach(sink, blocker);
            Keying.focusIn(dialog);
        };
        const hideDialog = (dialog) => {
            dialogComp.clear();
            parent(dialog.element).each((blockerDom) => {
                dialog.getSystem().getByDom(blockerDom).each((blocker) => {
                    detach(blocker);
                });
            });
        };
        const getDialogBody = (dialog) => getPartOrDie(dialog, detail, 'body');
        const getDialogFooter = (dialog) => getPart(dialog, detail, 'footer');
        const setBusy = (dialog, getBusySpec) => {
            Blocking.block(dialog, getBusySpec);
        };
        const setIdle = (dialog) => {
            Blocking.unblock(dialog);
        };
        const modalEventsId = generate$6('modal-events');
        const eventOrder = {
            ...detail.eventOrder,
            [attachedToDom()]: [modalEventsId].concat(detail.eventOrder['alloy.system.attached'] || [])
        };
        const browser = detect$1();
        return {
            uid: detail.uid,
            dom: detail.dom,
            components,
            apis: {
                show: showDialog,
                hide: hideDialog,
                getBody: getDialogBody,
                getFooter: getDialogFooter,
                setIdle,
                setBusy
            },
            eventOrder,
            domModification: {
                attributes: {
                    'role': 'dialog',
                    'aria-modal': 'true'
                }
            },
            behaviours: augment(detail.modalBehaviours, [
                Replacing.config({}),
                Keying.config({
                    mode: 'cyclic',
                    onEnter: detail.onExecute,
                    onEscape: detail.onEscape,
                    useTabstopAt: detail.useTabstopAt,
                    firstTabstop: detail.firstTabstop
                }),
                Blocking.config({
                    getRoot: dialogComp.get
                }),
                config(modalEventsId, [
                    runOnAttached((c) => {
                        // TINY-10808 - Workaround to address the dialog header not being announced on VoiceOver with aria-labelledby, ideally we should use the aria-labelledby
                        const titleElm = getPartOrDie(c, detail, 'title').element;
                        const title = get$6(titleElm);
                        if (browser.os.isMacOS() && isNonNullable(title)) {
                            set$9(c.element, 'aria-label', title);
                        }
                        else {
                            labelledBy(c.element, titleElm);
                        }
                    })
                ])
            ])
        };
    };
    const ModalDialog = composite({
        name: 'ModalDialog',
        configFields: schema$8(),
        partFields: parts$6(),
        factory: factory$c,
        apis: {
            show: (apis, dialog) => {
                apis.show(dialog);
            },
            hide: (apis, dialog) => {
                apis.hide(dialog);
            },
            getBody: (apis, dialog) => apis.getBody(dialog),
            getFooter: (apis, dialog) => apis.getFooter(dialog),
            setBusy: (apis, dialog, getBusySpec) => {
                apis.setBusy(dialog, getBusySpec);
            },
            setIdle: (apis, dialog) => {
                apis.setIdle(dialog);
            }
        }
    });

    const labelPart = optional({
        schema: [required$1('dom')],
        name: 'label'
    });
    const edgePart = (name) => optional({
        name: '' + name + '-edge',
        overrides: (detail) => {
            const action = detail.model.manager.edgeActions[name];
            // Not all edges have actions for all sliders.
            // A horizontal slider will only have left and right, for instance,
            // ignoring top, bottom and diagonal edges as they don't make sense in context of those sliders.
            return action.fold(() => ({}), (a) => ({
                events: derive$2([
                    runActionExtra(touchstart(), (comp, se, d) => a(comp, d), [detail]),
                    runActionExtra(mousedown(), (comp, se, d) => a(comp, d), [detail]),
                    runActionExtra(mousemove(), (comp, se, det) => {
                        if (det.mouseIsDown.get()) {
                            a(comp, det);
                        }
                    }, [detail])
                ])
            }));
        }
    });
    // When the user touches the top left edge, it should move the thumb
    const tlEdgePart = edgePart('top-left');
    // When the user touches the top edge, it should move the thumb
    const tedgePart = edgePart('top');
    // When the user touches the top right edge, it should move the thumb
    const trEdgePart = edgePart('top-right');
    // When the user touches the right edge, it should move the thumb
    const redgePart = edgePart('right');
    // When the user touches the bottom right edge, it should move the thumb
    const brEdgePart = edgePart('bottom-right');
    // When the user touches the bottom edge, it should move the thumb
    const bedgePart = edgePart('bottom');
    // When the user touches the bottom left edge, it should move the thumb
    const blEdgePart = edgePart('bottom-left');
    // When the user touches the left edge, it should move the thumb
    const ledgePart = edgePart('left');
    // The thumb part needs to have position absolute to be positioned correctly
    const thumbPart = required({
        name: 'thumb',
        defaults: constant$1({
            dom: {
                styles: { position: 'absolute' }
            }
        }),
        overrides: (detail) => {
            return {
                events: derive$2([
                    // If the user touches the thumb itself, pretend they touched the spectrum instead. This
                    // allows sliding even when they touchstart the current value
                    redirectToPart(touchstart(), detail, 'spectrum'),
                    redirectToPart(touchmove(), detail, 'spectrum'),
                    redirectToPart(touchend(), detail, 'spectrum'),
                    redirectToPart(mousedown(), detail, 'spectrum'),
                    redirectToPart(mousemove(), detail, 'spectrum'),
                    redirectToPart(mouseup(), detail, 'spectrum')
                ])
            };
        }
    });
    const isShift = (event) => isShift$1(event.event);
    const spectrumPart = required({
        schema: [
            customField('mouseIsDown', () => Cell(false))
        ],
        name: 'spectrum',
        overrides: (detail) => {
            const modelDetail = detail.model;
            const model = modelDetail.manager;
            const setValueFrom = (component, simulatedEvent) => model.getValueFromEvent(simulatedEvent).map((value) => model.setValueFrom(component, detail, value));
            return {
                behaviours: derive$1([
                    // Move left and right along the spectrum
                    Keying.config({
                        mode: 'special',
                        onLeft: (spectrum, event) => model.onLeft(spectrum, detail, isShift(event)),
                        onRight: (spectrum, event) => model.onRight(spectrum, detail, isShift(event)),
                        onUp: (spectrum, event) => model.onUp(spectrum, detail, isShift(event)),
                        onDown: (spectrum, event) => model.onDown(spectrum, detail, isShift(event))
                    }),
                    Tabstopping.config({}),
                    Focusing.config({})
                ]),
                events: derive$2([
                    run$1(touchstart(), setValueFrom),
                    run$1(touchmove(), setValueFrom),
                    run$1(mousedown(), setValueFrom),
                    run$1(mousemove(), (spectrum, se) => {
                        if (detail.mouseIsDown.get()) {
                            setValueFrom(spectrum, se);
                        }
                    })
                ])
            };
        }
    });
    var SliderParts = [
        labelPart,
        ledgePart,
        redgePart,
        tedgePart,
        bedgePart,
        tlEdgePart,
        trEdgePart,
        blEdgePart,
        brEdgePart,
        thumbPart,
        spectrumPart
    ];

    const _sliderChangeEvent = 'slider.change.value';
    const sliderChangeEvent = constant$1(_sliderChangeEvent);
    const isTouchEvent$2 = (evt) => evt.type.indexOf('touch') !== -1;
    const getEventSource = (simulatedEvent) => {
        const evt = simulatedEvent.event.raw;
        if (isTouchEvent$2(evt)) {
            const touchEvent = evt;
            return touchEvent.touches !== undefined && touchEvent.touches.length === 1 ?
                Optional.some(touchEvent.touches[0]).map((t) => SugarPosition(t.clientX, t.clientY)) : Optional.none();
        }
        else {
            const mouseEvent = evt;
            return mouseEvent.clientX !== undefined ? Optional.some(mouseEvent).map((me) => SugarPosition(me.clientX, me.clientY)) : Optional.none();
        }
    };

    const t = 'top', r = 'right', b = 'bottom', l = 'left';
    // Values
    const minX = (detail) => detail.model.minX;
    const minY = (detail) => detail.model.minY;
    const min1X = (detail) => detail.model.minX - 1;
    const min1Y = (detail) => detail.model.minY - 1;
    const maxX = (detail) => detail.model.maxX;
    const maxY = (detail) => detail.model.maxY;
    const max1X = (detail) => detail.model.maxX + 1;
    const max1Y = (detail) => detail.model.maxY + 1;
    const range = (detail, max, min) => max(detail) - min(detail);
    const xRange = (detail) => range(detail, maxX, minX);
    const yRange = (detail) => range(detail, maxY, minY);
    const halfX = (detail) => xRange(detail) / 2;
    const halfY = (detail) => yRange(detail) / 2;
    const step = (detail, useMultiplier) => useMultiplier ? detail.stepSize * detail.speedMultiplier : detail.stepSize;
    const snap = (detail) => detail.snapToGrid;
    const snapStart = (detail) => detail.snapStart;
    const rounded = (detail) => detail.rounded;
    // Not great but... /shrug
    const hasEdge = (detail, edgeName) => detail[edgeName + '-edge'] !== undefined;
    const hasLEdge = (detail) => hasEdge(detail, l);
    const hasREdge = (detail) => hasEdge(detail, r);
    const hasTEdge = (detail) => hasEdge(detail, t);
    const hasBEdge = (detail) => hasEdge(detail, b);
    // Ew, any
    const currentValue = (detail) => detail.model.value.get();

    const xyValue = (x, y) => ({
        x,
        y
    });
    const fireSliderChange$3 = (component, value) => {
        emitWith(component, sliderChangeEvent(), { value });
    };
    // North West XY
    const setToTLEdgeXY = (edge, detail) => {
        fireSliderChange$3(edge, xyValue(min1X(detail), min1Y(detail)));
    };
    // North
    const setToTEdge = (edge, detail) => {
        fireSliderChange$3(edge, min1Y(detail));
    };
    // North XY
    const setToTEdgeXY = (edge, detail) => {
        fireSliderChange$3(edge, xyValue(halfX(detail), min1Y(detail)));
    };
    // North East XY
    const setToTREdgeXY = (edge, detail) => {
        fireSliderChange$3(edge, xyValue(max1X(detail), min1Y(detail)));
    };
    // East
    const setToREdge = (edge, detail) => {
        fireSliderChange$3(edge, max1X(detail));
    };
    // East XY
    const setToREdgeXY = (edge, detail) => {
        fireSliderChange$3(edge, xyValue(max1X(detail), halfY(detail)));
    };
    // South East XY
    const setToBREdgeXY = (edge, detail) => {
        fireSliderChange$3(edge, xyValue(max1X(detail), max1Y(detail)));
    };
    // South
    const setToBEdge = (edge, detail) => {
        fireSliderChange$3(edge, max1Y(detail));
    };
    // South XY
    const setToBEdgeXY = (edge, detail) => {
        fireSliderChange$3(edge, xyValue(halfX(detail), max1Y(detail)));
    };
    // South West XY
    const setToBLEdgeXY = (edge, detail) => {
        fireSliderChange$3(edge, xyValue(min1X(detail), max1Y(detail)));
    };
    // West
    const setToLEdge = (edge, detail) => {
        fireSliderChange$3(edge, min1X(detail));
    };
    // West XY
    const setToLEdgeXY = (edge, detail) => {
        fireSliderChange$3(edge, xyValue(min1X(detail), halfY(detail)));
    };

    const reduceBy = (value, min, max, step) => {
        if (value < min) {
            return value;
        }
        else if (value > max) {
            return max;
        }
        else if (value === min) {
            return min - 1;
        }
        else {
            return Math.max(min, value - step);
        }
    };
    const increaseBy = (value, min, max, step) => {
        if (value > max) {
            return value;
        }
        else if (value < min) {
            return min;
        }
        else if (value === max) {
            return max + 1;
        }
        else {
            return Math.min(max, value + step);
        }
    };
    const capValue = (value, min, max) => Math.max(min, Math.min(max, value));
    const snapValueOf = (value, min, max, step, snapStart) => 
    // We are snapping by the step size. Therefore, find the nearest multiple of
    // the step
    snapStart.fold(() => {
        // There is no initial snapping start, so just go from the minimum
        const initValue = value - min;
        const extraValue = Math.round(initValue / step) * step;
        return capValue(min + extraValue, min - 1, max + 1);
    }, (start) => {
        // There is an initial snapping start, so using that as the starting point,
        // calculate the nearest snap position based on the value
        const remainder = (value - start) % step;
        const adjustment = Math.round(remainder / step);
        const rawSteps = Math.floor((value - start) / step);
        const maxSteps = Math.floor((max - start) / step);
        const numSteps = Math.min(maxSteps, rawSteps + adjustment);
        const r = start + (numSteps * step);
        return Math.max(start, r);
    });
    const findOffsetOf = (value, min, max) => Math.min(max, Math.max(value, min)) - min;
    const findValueOf = (args) => {
        const { min, max, range, value, step, snap, snapStart, rounded, hasMinEdge, hasMaxEdge, minBound, maxBound, screenRange } = args;
        const capMin = hasMinEdge ? min - 1 : min;
        const capMax = hasMaxEdge ? max + 1 : max;
        if (value < minBound) {
            return capMin;
        }
        else if (value > maxBound) {
            return capMax;
        }
        else {
            const offset = findOffsetOf(value, minBound, maxBound);
            const newValue = capValue(((offset / screenRange) * range) + min, capMin, capMax);
            if (snap && newValue >= min && newValue <= max) {
                return snapValueOf(newValue, min, max, step, snapStart);
            }
            else if (rounded) {
                return Math.round(newValue);
            }
            else {
                return newValue;
            }
        }
    };
    const findOffsetOfValue$2 = (args) => {
        const { min, max, range, value, hasMinEdge, hasMaxEdge, maxBound, maxOffset, centerMinEdge, centerMaxEdge } = args;
        if (value < min) {
            return hasMinEdge ? 0 : centerMinEdge;
        }
        else if (value > max) {
            return hasMaxEdge ? maxBound : centerMaxEdge;
        }
        else {
            // position along the slider
            return (value - min) / range * maxOffset;
        }
    };

    const top = 'top', right = 'right', bottom = 'bottom', left = 'left', width = 'width', height = 'height';
    // Screen offsets from bounding client rect
    const getBounds = (component) => component.element.dom.getBoundingClientRect();
    const getBoundsProperty = (bounds, property) => bounds[property];
    const getMinXBounds = (component) => {
        const bounds = getBounds(component);
        return getBoundsProperty(bounds, left);
    };
    const getMaxXBounds = (component) => {
        const bounds = getBounds(component);
        return getBoundsProperty(bounds, right);
    };
    const getMinYBounds = (component) => {
        const bounds = getBounds(component);
        return getBoundsProperty(bounds, top);
    };
    const getMaxYBounds = (component) => {
        const bounds = getBounds(component);
        return getBoundsProperty(bounds, bottom);
    };
    const getXScreenRange = (component) => {
        const bounds = getBounds(component);
        return getBoundsProperty(bounds, width);
    };
    const getYScreenRange = (component) => {
        const bounds = getBounds(component);
        return getBoundsProperty(bounds, height);
    };
    const getCenterOffsetOf = (componentMinEdge, componentMaxEdge, spectrumMinEdge) => (componentMinEdge + componentMaxEdge) / 2 - spectrumMinEdge;
    const getXCenterOffSetOf = (component, spectrum) => {
        const componentBounds = getBounds(component);
        const spectrumBounds = getBounds(spectrum);
        const componentMinEdge = getBoundsProperty(componentBounds, left);
        const componentMaxEdge = getBoundsProperty(componentBounds, right);
        const spectrumMinEdge = getBoundsProperty(spectrumBounds, left);
        return getCenterOffsetOf(componentMinEdge, componentMaxEdge, spectrumMinEdge);
    };
    const getYCenterOffSetOf = (component, spectrum) => {
        const componentBounds = getBounds(component);
        const spectrumBounds = getBounds(spectrum);
        const componentMinEdge = getBoundsProperty(componentBounds, top);
        const componentMaxEdge = getBoundsProperty(componentBounds, bottom);
        const spectrumMinEdge = getBoundsProperty(spectrumBounds, top);
        return getCenterOffsetOf(componentMinEdge, componentMaxEdge, spectrumMinEdge);
    };

    // fire slider change event with x value
    const fireSliderChange$2 = (spectrum, value) => {
        emitWith(spectrum, sliderChangeEvent(), { value });
    };
    // find the value of the x offset of where the mouse was clicked from the model.
    const findValueOfOffset$1 = (spectrum, detail, left) => {
        const args = {
            min: minX(detail),
            max: maxX(detail),
            range: xRange(detail),
            value: left,
            step: step(detail),
            snap: snap(detail),
            snapStart: snapStart(detail),
            rounded: rounded(detail),
            hasMinEdge: hasLEdge(detail),
            hasMaxEdge: hasREdge(detail),
            minBound: getMinXBounds(spectrum),
            maxBound: getMaxXBounds(spectrum),
            screenRange: getXScreenRange(spectrum)
        };
        return findValueOf(args);
    };
    // find the value and fire a slider change event, returning the value
    const setValueFrom$2 = (spectrum, detail, value) => {
        const xValue = findValueOfOffset$1(spectrum, detail, value);
        const sliderVal = xValue;
        fireSliderChange$2(spectrum, sliderVal);
        return xValue;
    };
    // fire a slider change event with the minimum value
    const setToMin$2 = (spectrum, detail) => {
        const min = minX(detail);
        fireSliderChange$2(spectrum, min);
    };
    // fire a slider change event with the maximum value
    const setToMax$2 = (spectrum, detail) => {
        const max = maxX(detail);
        fireSliderChange$2(spectrum, max);
    };
    // move in a direction by step size. Fire change at the end
    const moveBy$2 = (direction, spectrum, detail, useMultiplier) => {
        const f = (direction > 0) ? increaseBy : reduceBy;
        const xValue = f(currentValue(detail), minX(detail), maxX(detail), step(detail, useMultiplier));
        fireSliderChange$2(spectrum, xValue);
        return Optional.some(xValue);
    };
    const handleMovement$2 = (direction) => (spectrum, detail, useMultiplier) => moveBy$2(direction, spectrum, detail, useMultiplier).map(always);
    // get x offset from event
    const getValueFromEvent$2 = (simulatedEvent) => {
        const pos = getEventSource(simulatedEvent);
        return pos.map((p) => p.left);
    };
    // find the x offset of a given value from the model
    const findOffsetOfValue$1 = (spectrum, detail, value, minEdge, maxEdge) => {
        const minOffset = 0;
        const maxOffset = getXScreenRange(spectrum);
        const centerMinEdge = minEdge.bind((edge) => Optional.some(getXCenterOffSetOf(edge, spectrum))).getOr(minOffset);
        const centerMaxEdge = maxEdge.bind((edge) => Optional.some(getXCenterOffSetOf(edge, spectrum))).getOr(maxOffset);
        const args = {
            min: minX(detail),
            max: maxX(detail),
            range: xRange(detail),
            value,
            hasMinEdge: hasLEdge(detail),
            hasMaxEdge: hasREdge(detail),
            minBound: getMinXBounds(spectrum),
            minOffset,
            maxBound: getMaxXBounds(spectrum),
            maxOffset,
            centerMinEdge,
            centerMaxEdge
        };
        return findOffsetOfValue$2(args);
    };
    // find left offset for absolute positioning from a given value
    const findPositionOfValue$1 = (slider, spectrum, value, minEdge, maxEdge, detail) => {
        const offset = findOffsetOfValue$1(spectrum, detail, value, minEdge, maxEdge);
        return (getMinXBounds(spectrum) - getMinXBounds(slider)) + offset;
    };
    // update the position of the thumb from the slider's current value
    const setPositionFromValue$2 = (slider, thumb, detail, edges) => {
        const value = currentValue(detail);
        const pos = findPositionOfValue$1(slider, edges.getSpectrum(slider), value, edges.getLeftEdge(slider), edges.getRightEdge(slider), detail);
        const thumbRadius = get$c(thumb.element) / 2;
        set$7(thumb.element, 'left', (pos - thumbRadius) + 'px');
    };
    // Key Events
    const onLeft$2 = handleMovement$2(-1);
    const onRight$2 = handleMovement$2(1);
    const onUp$2 = Optional.none;
    const onDown$2 = Optional.none;
    // Edge Click Actions
    const edgeActions$2 = {
        'top-left': Optional.none(),
        'top': Optional.none(),
        'top-right': Optional.none(),
        'right': Optional.some(setToREdge),
        'bottom-right': Optional.none(),
        'bottom': Optional.none(),
        'bottom-left': Optional.none(),
        'left': Optional.some(setToLEdge)
    };

    var HorizontalModel = /*#__PURE__*/Object.freeze({
        __proto__: null,
        setValueFrom: setValueFrom$2,
        setToMin: setToMin$2,
        setToMax: setToMax$2,
        findValueOfOffset: findValueOfOffset$1,
        getValueFromEvent: getValueFromEvent$2,
        findPositionOfValue: findPositionOfValue$1,
        setPositionFromValue: setPositionFromValue$2,
        onLeft: onLeft$2,
        onRight: onRight$2,
        onUp: onUp$2,
        onDown: onDown$2,
        edgeActions: edgeActions$2
    });

    // fire slider change event with y value
    const fireSliderChange$1 = (spectrum, value) => {
        emitWith(spectrum, sliderChangeEvent(), { value });
    };
    // find the value of the y offset of where the mouse was clicked from the model.
    const findValueOfOffset = (spectrum, detail, top) => {
        const args = {
            min: minY(detail),
            max: maxY(detail),
            range: yRange(detail),
            value: top,
            step: step(detail),
            snap: snap(detail),
            snapStart: snapStart(detail),
            rounded: rounded(detail),
            hasMinEdge: hasTEdge(detail),
            hasMaxEdge: hasBEdge(detail),
            minBound: getMinYBounds(spectrum),
            maxBound: getMaxYBounds(spectrum),
            screenRange: getYScreenRange(spectrum)
        };
        return findValueOf(args);
    };
    // find the value and fire a slider change event, returning the value
    const setValueFrom$1 = (spectrum, detail, value) => {
        const yValue = findValueOfOffset(spectrum, detail, value);
        const sliderVal = yValue;
        fireSliderChange$1(spectrum, sliderVal);
        return yValue;
    };
    // fire a slider change event with the minimum value
    const setToMin$1 = (spectrum, detail) => {
        const min = minY(detail);
        fireSliderChange$1(spectrum, min);
    };
    // fire a slider change event with the maximum value
    const setToMax$1 = (spectrum, detail) => {
        const max = maxY(detail);
        fireSliderChange$1(spectrum, max);
    };
    // move in a direction by step size. Fire change at the end
    const moveBy$1 = (direction, spectrum, detail, useMultiplier) => {
        const f = (direction > 0) ? increaseBy : reduceBy;
        const yValue = f(currentValue(detail), minY(detail), maxY(detail), step(detail, useMultiplier));
        fireSliderChange$1(spectrum, yValue);
        return Optional.some(yValue);
    };
    const handleMovement$1 = (direction) => (spectrum, detail, useMultiplier) => moveBy$1(direction, spectrum, detail, useMultiplier).map(always);
    // get y offset from event
    const getValueFromEvent$1 = (simulatedEvent) => {
        const pos = getEventSource(simulatedEvent);
        return pos.map((p) => {
            return p.top;
        });
    };
    // find the y offset of a given value from the model
    const findOffsetOfValue = (spectrum, detail, value, minEdge, maxEdge) => {
        const minOffset = 0;
        const maxOffset = getYScreenRange(spectrum);
        const centerMinEdge = minEdge.bind((edge) => Optional.some(getYCenterOffSetOf(edge, spectrum))).getOr(minOffset);
        const centerMaxEdge = maxEdge.bind((edge) => Optional.some(getYCenterOffSetOf(edge, spectrum))).getOr(maxOffset);
        const args = {
            min: minY(detail),
            max: maxY(detail),
            range: yRange(detail),
            value,
            hasMinEdge: hasTEdge(detail),
            hasMaxEdge: hasBEdge(detail),
            minBound: getMinYBounds(spectrum),
            minOffset,
            maxBound: getMaxYBounds(spectrum),
            maxOffset,
            centerMinEdge,
            centerMaxEdge
        };
        return findOffsetOfValue$2(args);
    };
    // find left offset for absolute positioning from a given value
    const findPositionOfValue = (slider, spectrum, value, minEdge, maxEdge, detail) => {
        const offset = findOffsetOfValue(spectrum, detail, value, minEdge, maxEdge);
        return (getMinYBounds(spectrum) - getMinYBounds(slider)) + offset;
    };
    // update the position of the thumb from the slider's current value
    const setPositionFromValue$1 = (slider, thumb, detail, edges) => {
        const value = currentValue(detail);
        const pos = findPositionOfValue(slider, edges.getSpectrum(slider), value, edges.getTopEdge(slider), edges.getBottomEdge(slider), detail);
        const thumbRadius = get$d(thumb.element) / 2;
        set$7(thumb.element, 'top', (pos - thumbRadius) + 'px');
    };
    // Key Events
    const onLeft$1 = Optional.none;
    const onRight$1 = Optional.none;
    const onUp$1 = handleMovement$1(-1);
    const onDown$1 = handleMovement$1(1);
    // Edge Click Actions
    const edgeActions$1 = {
        'top-left': Optional.none(),
        'top': Optional.some(setToTEdge),
        'top-right': Optional.none(),
        'right': Optional.none(),
        'bottom-right': Optional.none(),
        'bottom': Optional.some(setToBEdge),
        'bottom-left': Optional.none(),
        'left': Optional.none()
    };

    var VerticalModel = /*#__PURE__*/Object.freeze({
        __proto__: null,
        setValueFrom: setValueFrom$1,
        setToMin: setToMin$1,
        setToMax: setToMax$1,
        findValueOfOffset: findValueOfOffset,
        getValueFromEvent: getValueFromEvent$1,
        findPositionOfValue: findPositionOfValue,
        setPositionFromValue: setPositionFromValue$1,
        onLeft: onLeft$1,
        onRight: onRight$1,
        onUp: onUp$1,
        onDown: onDown$1,
        edgeActions: edgeActions$1
    });

    // fire slider change event with xy value
    const fireSliderChange = (spectrum, value) => {
        emitWith(spectrum, sliderChangeEvent(), { value });
    };
    const sliderValue = (x, y) => ({
        x,
        y
    });
    // find both values of x and y offsets of where the mouse was clicked from the model.
    // then fire a slider change event with those values, returning the values
    const setValueFrom = (spectrum, detail, value) => {
        const xValue = findValueOfOffset$1(spectrum, detail, value.left);
        const yValue = findValueOfOffset(spectrum, detail, value.top);
        const val = sliderValue(xValue, yValue);
        fireSliderChange(spectrum, val);
        return val;
    };
    // move in a direction by step size. Fire change at the end
    const moveBy = (direction, isVerticalMovement, spectrum, detail, useMultiplier) => {
        const f = (direction > 0) ? increaseBy : reduceBy;
        const xValue = isVerticalMovement ? currentValue(detail).x :
            f(currentValue(detail).x, minX(detail), maxX(detail), step(detail, useMultiplier));
        const yValue = !isVerticalMovement ? currentValue(detail).y :
            f(currentValue(detail).y, minY(detail), maxY(detail), step(detail, useMultiplier));
        fireSliderChange(spectrum, sliderValue(xValue, yValue));
        return Optional.some(xValue);
    };
    const handleMovement = (direction, isVerticalMovement) => (spectrum, detail, useMultiplier) => moveBy(direction, isVerticalMovement, spectrum, detail, useMultiplier).map(always);
    // fire a slider change event with the minimum value
    const setToMin = (spectrum, detail) => {
        const mX = minX(detail);
        const mY = minY(detail);
        fireSliderChange(spectrum, sliderValue(mX, mY));
    };
    // fire a slider change event with the maximum value
    const setToMax = (spectrum, detail) => {
        const mX = maxX(detail);
        const mY = maxY(detail);
        fireSliderChange(spectrum, sliderValue(mX, mY));
    };
    // get event data as a SugarPosition
    const getValueFromEvent = (simulatedEvent) => getEventSource(simulatedEvent);
    // update the position of the thumb from the slider's current value
    const setPositionFromValue = (slider, thumb, detail, edges) => {
        const value = currentValue(detail);
        const xPos = findPositionOfValue$1(slider, edges.getSpectrum(slider), value.x, edges.getLeftEdge(slider), edges.getRightEdge(slider), detail);
        const yPos = findPositionOfValue(slider, edges.getSpectrum(slider), value.y, edges.getTopEdge(slider), edges.getBottomEdge(slider), detail);
        const thumbXRadius = get$c(thumb.element) / 2;
        const thumbYRadius = get$d(thumb.element) / 2;
        set$7(thumb.element, 'left', (xPos - thumbXRadius) + 'px');
        set$7(thumb.element, 'top', (yPos - thumbYRadius) + 'px');
    };
    // Key Events
    const onLeft = handleMovement(-1, false);
    const onRight = handleMovement(1, false);
    const onUp = handleMovement(-1, true);
    const onDown = handleMovement(1, true);
    // Edge Click Actions
    const edgeActions = {
        'top-left': Optional.some(setToTLEdgeXY),
        'top': Optional.some(setToTEdgeXY),
        'top-right': Optional.some(setToTREdgeXY),
        'right': Optional.some(setToREdgeXY),
        'bottom-right': Optional.some(setToBREdgeXY),
        'bottom': Optional.some(setToBEdgeXY),
        'bottom-left': Optional.some(setToBLEdgeXY),
        'left': Optional.some(setToLEdgeXY)
    };

    var TwoDModel = /*#__PURE__*/Object.freeze({
        __proto__: null,
        setValueFrom: setValueFrom,
        setToMin: setToMin,
        setToMax: setToMax,
        getValueFromEvent: getValueFromEvent,
        setPositionFromValue: setPositionFromValue,
        onLeft: onLeft,
        onRight: onRight,
        onUp: onUp,
        onDown: onDown,
        edgeActions: edgeActions
    });

    const SliderSchema = [
        defaulted('stepSize', 1),
        defaulted('speedMultiplier', 10),
        defaulted('onChange', noop),
        defaulted('onChoose', noop),
        defaulted('onInit', noop),
        defaulted('onDragStart', noop),
        defaulted('onDragEnd', noop),
        defaulted('snapToGrid', false),
        defaulted('rounded', true),
        option$3('snapStart'),
        requiredOf('model', choose$1('mode', {
            x: [
                defaulted('minX', 0),
                defaulted('maxX', 100),
                customField('value', (spec) => Cell(spec.mode.minX)),
                required$1('getInitialValue'),
                output$1('manager', HorizontalModel)
            ],
            y: [
                defaulted('minY', 0),
                defaulted('maxY', 100),
                customField('value', (spec) => Cell(spec.mode.minY)),
                required$1('getInitialValue'),
                output$1('manager', VerticalModel)
            ],
            xy: [
                defaulted('minX', 0),
                defaulted('maxX', 100),
                defaulted('minY', 0),
                defaulted('maxY', 100),
                customField('value', (spec) => Cell({
                    x: spec.mode.minX,
                    y: spec.mode.minY
                })),
                required$1('getInitialValue'),
                output$1('manager', TwoDModel)
            ]
        })),
        field('sliderBehaviours', [Keying, Representing]),
        customField('mouseIsDown', () => Cell(false))
    ];

    const sketch$1 = (detail, components, _spec, _externals) => {
        const getThumb = (component) => getPartOrDie(component, detail, 'thumb');
        const getSpectrum = (component) => getPartOrDie(component, detail, 'spectrum');
        const getLeftEdge = (component) => getPart(component, detail, 'left-edge');
        const getRightEdge = (component) => getPart(component, detail, 'right-edge');
        const getTopEdge = (component) => getPart(component, detail, 'top-edge');
        const getBottomEdge = (component) => getPart(component, detail, 'bottom-edge');
        const modelDetail = detail.model;
        const model = modelDetail.manager;
        const refresh = (slider, thumb) => {
            model.setPositionFromValue(slider, thumb, detail, {
                getLeftEdge,
                getRightEdge,
                getTopEdge,
                getBottomEdge,
                getSpectrum
            });
        };
        const setValue = (slider, newValue) => {
            modelDetail.value.set(newValue);
            const thumb = getThumb(slider);
            refresh(slider, thumb);
        };
        const changeValue = (slider, newValue) => {
            setValue(slider, newValue);
            const thumb = getThumb(slider);
            detail.onChange(slider, thumb, newValue);
            return Optional.some(true);
        };
        const resetToMin = (slider) => {
            model.setToMin(slider, detail);
        };
        const resetToMax = (slider) => {
            model.setToMax(slider, detail);
        };
        const choose = (slider) => {
            const fireOnChoose = () => {
                getPart(slider, detail, 'thumb').each((thumb) => {
                    const value = modelDetail.value.get();
                    detail.onChoose(slider, thumb, value);
                });
            };
            const wasDown = detail.mouseIsDown.get();
            detail.mouseIsDown.set(false);
            // We don't want this to fire if the mouse wasn't pressed down over anything other than the slider.
            if (wasDown) {
                fireOnChoose();
            }
        };
        const onDragStart = (slider, simulatedEvent) => {
            simulatedEvent.stop();
            detail.mouseIsDown.set(true);
            detail.onDragStart(slider, getThumb(slider));
        };
        const onDragEnd = (slider, simulatedEvent) => {
            simulatedEvent.stop();
            detail.onDragEnd(slider, getThumb(slider));
            choose(slider);
        };
        const focusWidget = (component) => {
            getPart(component, detail, 'spectrum').map(Keying.focusIn);
        };
        return {
            uid: detail.uid,
            dom: detail.dom,
            components,
            behaviours: augment(detail.sliderBehaviours, [
                Keying.config({
                    mode: 'special',
                    focusIn: focusWidget
                }),
                Representing.config({
                    store: {
                        mode: 'manual',
                        getValue: (_) => {
                            return modelDetail.value.get();
                        },
                        setValue
                    }
                }),
                Receiving.config({
                    channels: {
                        [mouseReleased()]: {
                            onReceive: choose
                        }
                    }
                })
            ]),
            events: derive$2([
                run$1(sliderChangeEvent(), (slider, simulatedEvent) => {
                    changeValue(slider, simulatedEvent.event.value);
                }),
                runOnAttached((slider, _simulatedEvent) => {
                    // Set the initial value
                    const getInitial = modelDetail.getInitialValue();
                    modelDetail.value.set(getInitial);
                    const thumb = getThumb(slider);
                    refresh(slider, thumb);
                    const spectrum = getSpectrum(slider);
                    // Call onInit instead of onChange for the first value.
                    detail.onInit(slider, thumb, spectrum, modelDetail.value.get());
                }),
                run$1(touchstart(), onDragStart),
                run$1(touchend(), onDragEnd),
                run$1(mousedown(), (component, event) => {
                    focusWidget(component);
                    onDragStart(component, event);
                }),
                run$1(mouseup(), onDragEnd),
            ]),
            apis: {
                resetToMin,
                resetToMax,
                setValue,
                refresh
            },
            domModification: {
                styles: {
                    position: 'relative'
                }
            }
        };
    };

    const Slider = composite({
        name: 'Slider',
        configFields: SliderSchema,
        partFields: SliderParts,
        factory: sketch$1,
        apis: {
            setValue: (apis, slider, value) => {
                apis.setValue(slider, value);
            },
            resetToMin: (apis, slider) => {
                apis.resetToMin(slider);
            },
            resetToMax: (apis, slider) => {
                apis.resetToMax(slider);
            },
            refresh: (apis, slider) => {
                apis.refresh(slider);
            }
        }
    });

    const owner = 'container';
    const schema$7 = [
        field('slotBehaviours', [])
    ];
    const getPartName = (name) => '<alloy.field.' + name + '>';
    const sketch = (sSpec) => {
        // As parts.slot is called, record all of the parts that are registered
        // as part of this SlotContainer.
        const parts = (() => {
            const record = [];
            const slot = (name, config) => {
                record.push(name);
                return generateOne$1(owner, getPartName(name), config);
            };
            return {
                slot,
                record: constant$1(record)
            };
        })();
        const spec = sSpec(parts);
        const partNames = parts.record();
        // Like a Form, a SlotContainer does not know its parts in advance. So the
        // record lists the names of the parts to put in the schema.
        // TODO: Find a nice way to remove dupe with Form
        const fieldParts = map$2(partNames, (n) => required({ name: n, pname: getPartName(n) }));
        return composite$1(owner, schema$7, fieldParts, make$3, spec);
    };
    const make$3 = (detail, components) => {
        const getSlotNames = (_) => getAllPartNames(detail);
        const getSlot = (container, key) => getPart(container, detail, key);
        const onSlot = (f, def) => (container, key) => getPart(container, detail, key).map((slot) => f(slot, key)).getOr(def);
        const onSlots = (f) => (container, keys) => {
            each$1(keys, (key) => f(container, key));
        };
        const doShowing = (comp, _key) => get$g(comp.element, 'aria-hidden') !== 'true';
        const doShow = (comp, key) => {
            // NOTE: May need to restore old values.
            if (!doShowing(comp)) {
                const element = comp.element;
                remove$6(element, 'display');
                remove$8(element, 'aria-hidden');
                emitWith(comp, slotVisibility(), { name: key, visible: true });
            }
        };
        const doHide = (comp, key) => {
            // NOTE: May need to save old values.
            if (doShowing(comp)) {
                const element = comp.element;
                set$7(element, 'display', 'none');
                set$9(element, 'aria-hidden', 'true');
                emitWith(comp, slotVisibility(), { name: key, visible: false });
            }
        };
        const isShowing = onSlot(doShowing, false);
        const hideSlot = onSlot(doHide);
        const hideSlots = onSlots(hideSlot);
        const hideAllSlots = (container) => hideSlots(container, getSlotNames());
        const showSlot = onSlot(doShow);
        const apis = {
            getSlotNames,
            getSlot,
            isShowing,
            hideSlot,
            hideAllSlots,
            showSlot
        };
        return {
            uid: detail.uid,
            dom: detail.dom,
            components,
            behaviours: get$2(detail.slotBehaviours),
            apis
        };
    };
    // No type safety doing it this way. But removes dupe.
    // We could probably use spread operator to help here.
    const slotApis = map$1({
        getSlotNames: (apis, c) => apis.getSlotNames(c),
        getSlot: (apis, c, key) => apis.getSlot(c, key),
        isShowing: (apis, c, key) => apis.isShowing(c, key),
        hideSlot: (apis, c, key) => apis.hideSlot(c, key),
        hideAllSlots: (apis, c) => apis.hideAllSlots(c),
        showSlot: (apis, c, key) => apis.showSlot(c, key)
    }, (value) => makeApi(value));
    const SlotContainer = {
        ...slotApis,
        ...{ sketch }
    };

    const generate$1 = (xs, f) => {
        const init = {
            len: 0,
            list: []
        };
        const r = foldl(xs, (b, a) => {
            const value = f(a, b.len);
            return value.fold(constant$1(b), (v) => ({
                len: v.finish,
                list: b.list.concat([v])
            }));
        }, init);
        return r.list;
    };

    const output = (within, extra, withinWidth) => ({
        within,
        extra,
        withinWidth
    });
    const apportion = (units, total, len) => {
        const parray = generate$1(units, (unit, current) => {
            const width = len(unit);
            return Optional.some({
                element: unit,
                start: current,
                finish: current + width,
                width
            });
        });
        const within = filter$2(parray, (unit) => unit.finish <= total);
        const withinWidth = foldr(within, (acc, el) => acc + el.width, 0);
        const extra = parray.slice(within.length);
        return {
            within,
            extra,
            withinWidth
        };
    };
    const toUnit = (parray) => map$2(parray, (unit) => unit.element);
    const fitLast = (within, extra, withinWidth) => {
        const fits = toUnit(within.concat(extra));
        return output(fits, [], withinWidth);
    };
    const overflow = (within, extra, overflower, withinWidth) => {
        const fits = toUnit(within).concat([overflower]);
        return output(fits, toUnit(extra), withinWidth);
    };
    const fitAll = (within, extra, withinWidth) => output(toUnit(within), [], withinWidth);
    const tryFit = (total, units, len) => {
        const divide = apportion(units, total, len);
        return divide.extra.length === 0 ? Optional.some(divide) : Optional.none();
    };
    const partition = (total, units, len, overflower) => {
        // Firstly, we try without the overflower.
        const divide = tryFit(total, units, len).getOrThunk(() => 
        // If that doesn't work, overflow
        apportion(units, total - len(overflower), len));
        const within = divide.within;
        const extra = divide.extra;
        const withinWidth = divide.withinWidth;
        if (extra.length === 1 && extra[0].width <= len(overflower)) {
            return fitLast(within, extra, withinWidth);
        }
        else if (extra.length >= 1) {
            return overflow(within, extra, overflower, withinWidth);
        }
        else {
            return fitAll(within, extra, withinWidth);
        }
    };

    const setGroups = (toolbar, storedGroups) => {
        const bGroups = map$2(storedGroups, (g) => premade(g));
        Toolbar.setGroups(toolbar, bGroups);
    };
    const findFocusedComp = (comps) => findMap(comps, (comp) => search(comp.element).bind((focusedElm) => comp.getSystem().getByDom(focusedElm).toOptional()));
    const refresh$2 = (toolbar, detail, setOverflow) => {
        // Ensure we have toolbar groups to render
        const builtGroups = detail.builtGroups.get();
        if (builtGroups.length === 0) {
            return;
        }
        const primary = getPartOrDie(toolbar, detail, 'primary');
        const overflowGroup = Coupling.getCoupled(toolbar, 'overflowGroup');
        // Set the primary toolbar to have visibility hidden;
        set$7(primary.element, 'visibility', 'hidden');
        const groups = builtGroups.concat([overflowGroup]);
        // Store the current focus state
        const focusedComp = findFocusedComp(groups);
        // Clear the overflow toolbar
        setOverflow([]);
        // Put all the groups inside the primary toolbar
        setGroups(primary, groups);
        const availableWidth = get$c(primary.element);
        const overflows = partition(availableWidth, detail.builtGroups.get(), (comp) => Math.ceil(comp.element.dom.getBoundingClientRect().width), overflowGroup);
        if (overflows.extra.length === 0) {
            // Not ideal. Breaking abstraction somewhat, though remove is better than insert
            // Can just reset the toolbar groups also ... but may be a bit slower.
            Replacing.remove(primary, overflowGroup);
            setOverflow([]);
        }
        else {
            setGroups(primary, overflows.within);
            setOverflow(overflows.extra);
        }
        remove$6(primary.element, 'visibility');
        reflow(primary.element);
        // Restore the focus
        focusedComp.each(Focusing.focus);
    };

    const schema$6 = constant$1([
        field('splitToolbarBehaviours', [Coupling]),
        customField('builtGroups', () => Cell([]))
    ]);

    const schema$5 = constant$1([
        markers$1(['overflowToggledClass']),
        optionFunction('getOverflowBounds'),
        required$1('lazySink'),
        customField('overflowGroups', () => Cell([])),
        onHandler('onOpened'),
        onHandler('onClosed')
    ].concat(schema$6()));
    const parts$5 = constant$1([
        required({
            factory: Toolbar,
            schema: schema$d(),
            name: 'primary'
        }),
        external$1({
            schema: schema$d(),
            name: 'overflow'
        }),
        external$1({
            name: 'overflow-button'
        }),
        external$1({
            name: 'overflow-group'
        })
    ]);

    const schema$4 = constant$1([
        required$1('items'),
        markers$1(['itemSelector']),
        field('tgroupBehaviours', [Keying])
    ]);
    const parts$4 = constant$1([
        group({
            name: 'items',
            unit: 'item'
        })
    ]);

    const factory$b = (detail, components, _spec, _externals) => ({
        uid: detail.uid,
        dom: detail.dom,
        components,
        behaviours: augment(detail.tgroupBehaviours, [
            Keying.config({
                mode: 'flow',
                selector: detail.markers.itemSelector
            })
        ]),
        domModification: {
            attributes: {
                role: 'toolbar'
            }
        }
    });
    const ToolbarGroup = composite({
        name: 'ToolbarGroup',
        configFields: schema$4(),
        partFields: parts$4(),
        factory: factory$b
    });

    const buildGroups = (comps) => map$2(comps, (g) => premade(g));
    const refresh$1 = (toolbar, memFloatingToolbarButton, detail) => {
        refresh$2(toolbar, detail, (overflowGroups) => {
            detail.overflowGroups.set(overflowGroups);
            memFloatingToolbarButton.getOpt(toolbar).each((floatingToolbarButton) => {
                FloatingToolbarButton.setGroups(floatingToolbarButton, buildGroups(overflowGroups));
            });
        });
    };
    const factory$a = (detail, components, spec, externals) => {
        const memFloatingToolbarButton = record(FloatingToolbarButton.sketch({
            fetch: () => Future.nu((resolve) => {
                resolve(buildGroups(detail.overflowGroups.get()));
            }),
            layouts: {
                onLtr: () => [southwest$2, southeast$2],
                onRtl: () => [southeast$2, southwest$2],
                onBottomLtr: () => [northwest$2, northeast$2],
                onBottomRtl: () => [northeast$2, northwest$2]
            },
            getBounds: spec.getOverflowBounds,
            lazySink: detail.lazySink,
            fireDismissalEventInstead: {},
            markers: {
                toggledClass: detail.markers.overflowToggledClass
            },
            parts: {
                button: externals['overflow-button'](),
                toolbar: externals.overflow()
            },
            onToggled: (comp, state) => detail[state ? 'onOpened' : 'onClosed'](comp)
        }));
        return {
            uid: detail.uid,
            dom: detail.dom,
            components,
            behaviours: augment(detail.splitToolbarBehaviours, [
                Coupling.config({
                    others: {
                        overflowGroup: () => {
                            return ToolbarGroup.sketch({
                                ...externals['overflow-group'](),
                                items: [
                                    memFloatingToolbarButton.asSpec()
                                ]
                            });
                        }
                    }
                })
            ]),
            apis: {
                setGroups: (toolbar, groups) => {
                    detail.builtGroups.set(map$2(groups, toolbar.getSystem().build));
                    refresh$1(toolbar, memFloatingToolbarButton, detail);
                },
                refresh: (toolbar) => refresh$1(toolbar, memFloatingToolbarButton, detail),
                toggle: (toolbar) => {
                    memFloatingToolbarButton.getOpt(toolbar).each((floatingToolbarButton) => {
                        FloatingToolbarButton.toggle(floatingToolbarButton);
                    });
                },
                toggleWithoutFocusing: (toolbar) => {
                    memFloatingToolbarButton.getOpt(toolbar).each(FloatingToolbarButton.toggleWithoutFocusing);
                },
                isOpen: (toolbar) => memFloatingToolbarButton.getOpt(toolbar).map(FloatingToolbarButton.isOpen).getOr(false),
                reposition: (toolbar) => {
                    memFloatingToolbarButton.getOpt(toolbar).each((floatingToolbarButton) => {
                        FloatingToolbarButton.reposition(floatingToolbarButton);
                    });
                },
                getOverflow: (toolbar) => memFloatingToolbarButton.getOpt(toolbar).bind(FloatingToolbarButton.getToolbar)
            },
            domModification: {
                attributes: { role: 'group' }
            }
        };
    };
    const SplitFloatingToolbar = composite({
        name: 'SplitFloatingToolbar',
        configFields: schema$5(),
        partFields: parts$5(),
        factory: factory$a,
        apis: {
            setGroups: (apis, toolbar, groups) => {
                apis.setGroups(toolbar, groups);
            },
            refresh: (apis, toolbar) => {
                apis.refresh(toolbar);
            },
            reposition: (apis, toolbar) => {
                apis.reposition(toolbar);
            },
            toggle: (apis, toolbar) => {
                apis.toggle(toolbar);
            },
            toggleWithoutFocusing: (apis, toolbar) => {
                apis.toggle(toolbar);
            },
            isOpen: (apis, toolbar) => apis.isOpen(toolbar),
            getOverflow: (apis, toolbar) => apis.getOverflow(toolbar)
        }
    });

    const schema$3 = constant$1([
        markers$1(['closedClass', 'openClass', 'shrinkingClass', 'growingClass', 'overflowToggledClass']),
        onHandler('onOpened'),
        onHandler('onClosed')
    ].concat(schema$6()));
    const parts$3 = constant$1([
        required({
            factory: Toolbar,
            schema: schema$d(),
            name: 'primary'
        }),
        required({
            factory: Toolbar,
            schema: schema$d(),
            name: 'overflow',
            overrides: (detail) => {
                return {
                    toolbarBehaviours: derive$1([
                        Sliding.config({
                            dimension: {
                                property: 'height'
                            },
                            closedClass: detail.markers.closedClass,
                            openClass: detail.markers.openClass,
                            shrinkingClass: detail.markers.shrinkingClass,
                            growingClass: detail.markers.growingClass,
                            onShrunk: (comp) => {
                                getPart(comp, detail, 'overflow-button').each((button) => {
                                    Toggling.off(button);
                                });
                                detail.onClosed(comp);
                            },
                            onGrown: (comp) => {
                                detail.onOpened(comp);
                            },
                            onStartGrow: (comp) => {
                                getPart(comp, detail, 'overflow-button').each(Toggling.on);
                            }
                        }),
                        Keying.config({
                            mode: 'acyclic',
                            onEscape: (comp) => {
                                getPart(comp, detail, 'overflow-button').each(Focusing.focus);
                                return Optional.some(true);
                            }
                        })
                    ])
                };
            }
        }),
        external$1({
            name: 'overflow-button',
            overrides: (detail) => ({
                buttonBehaviours: derive$1([
                    Toggling.config({
                        toggleClass: detail.markers.overflowToggledClass,
                        aria: {
                            mode: 'expanded'
                        },
                        toggleOnExecute: false
                    })
                ])
            })
        }),
        external$1({
            name: 'overflow-group'
        })
    ]);

    const isOpen = (toolbar, detail) => getPart(toolbar, detail, 'overflow').map(Sliding.hasGrown).getOr(false);
    const toggleToolbar = (toolbar, detail, skipFocus) => {
        // Make sure that the toolbar needs to toggled by checking for overflow button presence
        getPart(toolbar, detail, 'overflow-button')
            .each((oveflowButton) => {
            getPart(toolbar, detail, 'overflow').each((overf) => {
                refresh(toolbar, detail);
                if (Sliding.hasShrunk(overf)) {
                    const fn = detail.onOpened;
                    detail.onOpened = (comp) => {
                        if (!skipFocus) {
                            Keying.focusIn(overf);
                        }
                        fn(comp);
                        detail.onOpened = fn;
                    };
                }
                else {
                    const fn = detail.onClosed;
                    detail.onClosed = (comp) => {
                        if (!skipFocus) {
                            Focusing.focus(oveflowButton);
                        }
                        fn(comp);
                        detail.onClosed = fn;
                    };
                }
                Sliding.toggleGrow(overf);
            });
        });
    };
    const refresh = (toolbar, detail) => {
        getPart(toolbar, detail, 'overflow').each((overflow) => {
            refresh$2(toolbar, detail, (groups) => {
                const builtGroups = map$2(groups, (g) => premade(g));
                Toolbar.setGroups(overflow, builtGroups);
            });
            getPart(toolbar, detail, 'overflow-button').each((button) => {
                if (Sliding.hasGrown(overflow)) {
                    Toggling.on(button);
                }
            });
            Sliding.refresh(overflow);
        });
    };
    const factory$9 = (detail, components, spec, externals) => {
        const toolbarToggleEvent = 'alloy.toolbar.toggle';
        const doSetGroups = (toolbar, groups) => {
            const built = map$2(groups, toolbar.getSystem().build);
            detail.builtGroups.set(built);
        };
        return {
            uid: detail.uid,
            dom: detail.dom,
            components,
            behaviours: augment(detail.splitToolbarBehaviours, [
                Coupling.config({
                    others: {
                        overflowGroup: (toolbar) => {
                            return ToolbarGroup.sketch({
                                ...externals['overflow-group'](),
                                items: [
                                    Button.sketch({
                                        ...externals['overflow-button'](),
                                        action: (_button) => {
                                            emit(toolbar, toolbarToggleEvent);
                                        }
                                    })
                                ]
                            });
                        }
                    }
                }),
                config('toolbar-toggle-events', [
                    run$1(toolbarToggleEvent, (toolbar) => {
                        toggleToolbar(toolbar, detail, false);
                    })
                ])
            ]),
            apis: {
                setGroups: (toolbar, groups) => {
                    doSetGroups(toolbar, groups);
                    refresh(toolbar, detail);
                },
                refresh: (toolbar) => refresh(toolbar, detail),
                toggle: (toolbar) => {
                    toggleToolbar(toolbar, detail, false);
                },
                toggleWithoutFocusing: (toolbar) => {
                    toggleToolbar(toolbar, detail, true);
                },
                isOpen: (toolbar) => isOpen(toolbar, detail)
            },
            domModification: {
                attributes: { role: 'group' }
            }
        };
    };
    const SplitSlidingToolbar = composite({
        name: 'SplitSlidingToolbar',
        configFields: schema$3(),
        partFields: parts$3(),
        factory: factory$9,
        apis: {
            setGroups: (apis, toolbar, groups) => {
                apis.setGroups(toolbar, groups);
            },
            refresh: (apis, toolbar) => {
                apis.refresh(toolbar);
            },
            toggle: (apis, toolbar) => {
                apis.toggle(toolbar);
            },
            isOpen: (apis, toolbar) => apis.isOpen(toolbar)
        }
    });

    const factory$8 = (detail, _spec) => ({
        uid: detail.uid,
        dom: detail.dom,
        components: detail.components,
        events: events(detail.action),
        behaviours: augment(detail.tabButtonBehaviours, [
            Focusing.config({}),
            Keying.config({
                mode: 'execution',
                useSpace: true,
                useEnter: true
            }),
            Representing.config({
                store: {
                    mode: 'memory',
                    initialValue: detail.value
                }
            })
        ]),
        domModification: detail.domModification
    });
    const TabButton = single({
        name: 'TabButton',
        configFields: [
            defaulted('uid', undefined),
            required$1('value'),
            field$1('dom', 'dom', mergeWithThunk(() => ({
                attributes: {
                    'role': 'tab',
                    // NOTE: This is used in TabSection to connect "labelledby"
                    'id': generate$6('aria'),
                    'aria-selected': 'false'
                }
            })), anyValue()),
            option$3('action'),
            defaulted('domModification', {}),
            field('tabButtonBehaviours', [Focusing, Keying, Representing]),
            required$1('view')
        ],
        factory: factory$8
    });

    const schema$2 = constant$1([
        required$1('tabs'),
        required$1('dom'),
        defaulted('clickToDismiss', false),
        field('tabbarBehaviours', [Highlighting, Keying]),
        markers$1(['tabClass', 'selectedClass'])
    ]);
    const tabsPart = group({
        factory: TabButton,
        name: 'tabs',
        unit: 'tab',
        overrides: (barDetail) => {
            const dismissTab$1 = (tabbar, button) => {
                Highlighting.dehighlight(tabbar, button);
                emitWith(tabbar, dismissTab(), {
                    tabbar,
                    button
                });
            };
            const changeTab$1 = (tabbar, button) => {
                Highlighting.highlight(tabbar, button);
                emitWith(tabbar, changeTab(), {
                    tabbar,
                    button
                });
            };
            return {
                action: (button) => {
                    const tabbar = button.getSystem().getByUid(barDetail.uid).getOrDie();
                    const activeButton = Highlighting.isHighlighted(tabbar, button);
                    const response = (() => {
                        if (activeButton && barDetail.clickToDismiss) {
                            return dismissTab$1;
                        }
                        else if (!activeButton) {
                            return changeTab$1;
                        }
                        else {
                            return noop;
                        }
                    })();
                    response(tabbar, button);
                },
                domModification: {
                    classes: [barDetail.markers.tabClass]
                }
            };
        }
    });
    const parts$2 = constant$1([
        tabsPart
    ]);

    const factory$7 = (detail, components, _spec, _externals) => ({
        'uid': detail.uid,
        'dom': detail.dom,
        components,
        'debug.sketcher': 'Tabbar',
        'domModification': {
            attributes: {
                role: 'tablist'
            }
        },
        'behaviours': augment(detail.tabbarBehaviours, [
            Highlighting.config({
                highlightClass: detail.markers.selectedClass,
                itemClass: detail.markers.tabClass,
                // https://www.w3.org/TR/2010/WD-wai-aria-practices-20100916/#tabpanel
                // Consider a more seam-less way of combining highlighting and toggling
                onHighlight: (tabbar, tab) => {
                    // TODO: Integrate highlighting and toggling in a nice way
                    set$9(tab.element, 'aria-selected', 'true');
                },
                onDehighlight: (tabbar, tab) => {
                    set$9(tab.element, 'aria-selected', 'false');
                }
            }),
            Keying.config({
                mode: 'flow',
                getInitial: (tabbar) => {
                    // Restore focus to the previously highlighted tab.
                    return Highlighting.getHighlighted(tabbar).map((tab) => tab.element);
                },
                selector: '.' + detail.markers.tabClass,
                executeOnMove: true
            })
        ])
    });
    const Tabbar = composite({
        name: 'Tabbar',
        configFields: schema$2(),
        partFields: parts$2(),
        factory: factory$7
    });

    const factory$6 = (detail, _spec) => ({
        uid: detail.uid,
        dom: detail.dom,
        behaviours: augment(detail.tabviewBehaviours, [
            Replacing.config({})
        ]),
        domModification: {
            attributes: { role: 'tabpanel' }
        }
    });
    const Tabview = single({
        name: 'Tabview',
        configFields: [
            field('tabviewBehaviours', [Replacing])
        ],
        factory: factory$6
    });

    const schema$1 = constant$1([
        defaulted('selectFirst', true),
        onHandler('onChangeTab'),
        onHandler('onDismissTab'),
        defaulted('tabs', []),
        field('tabSectionBehaviours', [])
    ]);
    const barPart = required({
        factory: Tabbar,
        schema: [
            required$1('dom'),
            requiredObjOf('markers', [
                required$1('tabClass'),
                required$1('selectedClass')
            ])
        ],
        name: 'tabbar',
        defaults: (detail) => {
            return {
                tabs: detail.tabs
            };
        }
    });
    const viewPart = required({
        factory: Tabview,
        name: 'tabview'
    });
    const parts$1 = constant$1([
        barPart,
        viewPart
    ]);

    const factory$5 = (detail, components, _spec, _externals) => {
        const changeTab$1 = (button) => {
            const tabValue = Representing.getValue(button);
            getPart(button, detail, 'tabview').each((tabview) => {
                const tabWithValue = find$5(detail.tabs, (t) => t.value === tabValue);
                tabWithValue.each((tabData) => {
                    const panel = tabData.view();
                    // Update the tabview to refer to the current tab.
                    getOpt(button.element, 'id').each((id) => {
                        set$9(tabview.element, 'aria-labelledby', id);
                    });
                    Replacing.set(tabview, panel);
                    detail.onChangeTab(tabview, button, panel);
                });
            });
        };
        const changeTabBy = (section, byPred) => {
            getPart(section, detail, 'tabbar').each((tabbar) => {
                byPred(tabbar).each(emitExecute);
            });
        };
        return {
            uid: detail.uid,
            dom: detail.dom,
            components,
            behaviours: get$2(detail.tabSectionBehaviours),
            events: derive$2(flatten([
                detail.selectFirst ? [
                    runOnAttached((section, _simulatedEvent) => {
                        changeTabBy(section, Highlighting.getFirst);
                    })
                ] : [],
                [
                    run$1(changeTab(), (section, simulatedEvent) => {
                        const button = simulatedEvent.event.button;
                        changeTab$1(button);
                    }),
                    run$1(dismissTab(), (section, simulatedEvent) => {
                        const button = simulatedEvent.event.button;
                        detail.onDismissTab(section, button);
                    })
                ]
            ])),
            apis: {
                getViewItems: (section) => {
                    return getPart(section, detail, 'tabview').map((tabview) => Replacing.contents(tabview)).getOr([]);
                },
                // How should "clickToDismiss" interact with this? At the moment, it will never dismiss
                showTab: (section, tabKey) => {
                    // We only change the tab if it isn't currently active because that takes
                    // the whole "dismiss" issue out of the equation.
                    const getTabIfNotActive = (tabbar) => {
                        const candidates = Highlighting.getCandidates(tabbar);
                        const optTab = find$5(candidates, (c) => Representing.getValue(c) === tabKey);
                        return optTab.filter((tab) => !Highlighting.isHighlighted(tabbar, tab));
                    };
                    changeTabBy(section, getTabIfNotActive);
                }
            }
        };
    };
    const TabSection = composite({
        name: 'TabSection',
        configFields: schema$1(),
        partFields: parts$1(),
        factory: factory$5,
        apis: {
            getViewItems: (apis, component) => apis.getViewItems(component),
            showTab: (apis, component, tabKey) => {
                apis.showTab(component, tabKey);
            }
        }
    });

    // When showing a value in an input field, which part of the item do we use?
    const setValueFromItem = (model, input, item) => {
        const itemData = Representing.getValue(item);
        Representing.setValue(input, itemData);
        setCursorAtEnd(input);
    };
    const setSelectionOn = (input, f) => {
        const el = input.element;
        const value = get$5(el);
        const node = el.dom;
        // Only do for valid input types.
        if (get$g(el, 'type') !== 'number') {
            f(node, value);
        }
    };
    const setCursorAtEnd = (input) => {
        setSelectionOn(input, (node, value) => node.setSelectionRange(value.length, value.length));
    };
    const setSelectionToEnd = (input, startOffset) => {
        setSelectionOn(input, (node, value) => node.setSelectionRange(startOffset, value.length));
    };
    const attemptSelectOver = (model, input, item) => {
        if (!model.selectsOver) {
            return Optional.none();
        }
        else {
            const currentValue = Representing.getValue(input);
            const inputDisplay = model.getDisplayText(currentValue);
            const itemValue = Representing.getValue(item);
            const itemDisplay = model.getDisplayText(itemValue);
            return itemDisplay.indexOf(inputDisplay) === 0 ?
                Optional.some(() => {
                    setValueFromItem(model, input, item);
                    setSelectionToEnd(input, inputDisplay.length);
                })
                : Optional.none();
        }
    };

    const itemExecute = constant$1('alloy.typeahead.itemexecute');

    // TODO: Fix this.
    const make$2 = (detail, components, spec, externals) => {
        const navigateList = (comp, simulatedEvent, highlighter) => {
            /*
             * If we have an open Sandbox with an active menu,
             * but no highlighted item, then highlight the menu
             *
             * If we have an open Sandbox with an active menu,
             * and there is a highlighted item, simulated a keydown
             * on the menu
             *
             * If we have a closed sandbox, open the sandbox
             *
             * Regardless, this is a user initiated action. End previewing.
             */
            detail.previewing.set(false);
            const sandbox = Coupling.getCoupled(comp, 'sandbox');
            if (Sandboxing.isOpen(sandbox)) {
                Composing.getCurrent(sandbox).each((menu) => {
                    Highlighting.getHighlighted(menu).fold(() => {
                        highlighter(menu);
                    }, () => {
                        dispatchEvent(sandbox, menu.element, 'keydown', simulatedEvent);
                    });
                });
            }
            else {
                const onOpenSync = (sandbox) => {
                    Composing.getCurrent(sandbox).each(highlighter);
                };
                open(detail, mapFetch(comp), comp, sandbox, externals, onOpenSync, HighlightOnOpen.HighlightMenuAndItem).get(noop);
            }
        };
        // Due to the fact that typeahead probably need to separate value from text, they can't reuse
        // (easily) the same representing logic as input fields.
        const focusBehaviours$1 = focusBehaviours(detail);
        const mapFetch = (comp) => (tdata) => tdata.map((data) => {
            const menus = values(data.menus);
            const items = bind$3(menus, (menu) => filter$2(menu.items, (item) => item.type === 'item'));
            const repState = Representing.getState(comp);
            repState.update(map$2(items, (item) => item.data));
            return data;
        });
        // This function (getActiveMenu) is intended to make it easier to read what is happening
        // without having to decipher the Highlighting and Composing calls.
        const getActiveMenu = (sandboxComp) => Composing.getCurrent(sandboxComp);
        const typeaheadCustomEvents = 'typeaheadevents';
        const behaviours = [
            Focusing.config({}),
            Representing.config({
                onSetValue: detail.onSetValue,
                store: {
                    mode: 'dataset',
                    getDataKey: (comp) => get$5(comp.element),
                    // This really needs to be configurable
                    getFallbackEntry: (itemString) => ({
                        value: itemString,
                        meta: {}
                    }),
                    setValue: (comp, data) => {
                        set$4(comp.element, detail.model.getDisplayText(data));
                    },
                    ...detail.initialData.map((d) => wrap('initialValue', d)).getOr({})
                }
            }),
            Streaming.config({
                stream: {
                    mode: 'throttle',
                    delay: detail.responseTime,
                    stopEvent: false
                },
                onStream: (component, _simulatedEvent) => {
                    const sandbox = Coupling.getCoupled(component, 'sandbox');
                    const focusInInput = Focusing.isFocused(component);
                    // You don't want it to change when something else has triggered the change.
                    if (focusInInput) {
                        if (get$5(component.element).length >= detail.minChars) {
                            // Get the value of the previously active (selected/highlighted) item. We
                            // are going to try to preserve this.
                            const previousValue = getActiveMenu(sandbox).bind((activeMenu) => Highlighting.getHighlighted(activeMenu).map(Representing.getValue));
                            // Turning previewing ON here every keystroke is unnecessary, but relies
                            // on the fact that it will be turned off if required by highlighting events.
                            // So even if previewing was supposed to be off, turning it on here is
                            // just temporary, because the onOpenSync below will trigger a highlight
                            // if there was meant to be one, which will turn it off if required.
                            detail.previewing.set(true);
                            const onOpenSync = (_sandbox) => {
                                // This getActiveMenu relies on a menu being highlighted / active
                                getActiveMenu(sandbox).each((activeMenu) => {
                                    // The folds can make this hard to follow, but the basic gist of it is
                                    // that we want to see if we need to highlight one of the items in the
                                    // menu that we just opened. If we do highlight an item, then that
                                    // highlighting action will clear previewing (handled by the TieredMenu
                                    // part configuration for onHighlight). Note: that onOpenSync runs
                                    // *after* the highlightOnOpen setting.
                                    //
                                    // 1. If in "selectsOver" mode and we don't have a previous item,
                                    // then highlight the first one. This one will be used as the basis
                                    // for the "selectsOver" text selection. The act of highlighting the
                                    // first item will take us out of previewing mode. If the "selectsOver"
                                    // operation fails, it should clear the highlight, and restore previewing
                                    // 2. If not in "selectsOver" mode, and we don't have a previous item,
                                    // then we don't highlight anything. This will keep us in previewing
                                    // mode until the menu is interacted with (hover, navigation etc.)
                                    // 3. If we have a previous item, then try and rehighlight it. But if
                                    // we can't, the just highlight the first. Either action will take us
                                    // out of previewing mode.
                                    previousValue.fold(() => {
                                        // We are using "selectOver", so we need *something* to highlight
                                        if (detail.model.selectsOver) {
                                            Highlighting.highlightFirst(activeMenu);
                                        }
                                        // We aren't using "selectOver", so don't highlight anything
                                        // to preserve our "previewing" mode.
                                    }, (pv) => {
                                        // We have a previous item, so if we can't rehighlight it, then
                                        // we'll change to the first item. We want to keep some selection.
                                        Highlighting.highlightBy(activeMenu, (item) => {
                                            const itemData = Representing.getValue(item);
                                            return itemData.value === pv.value;
                                        });
                                        // Highlight first if could not find it?
                                        Highlighting.getHighlighted(activeMenu).orThunk(() => {
                                            Highlighting.highlightFirst(activeMenu);
                                            return Optional.none();
                                        });
                                    });
                                });
                            };
                            open(detail, mapFetch(component), component, sandbox, externals, onOpenSync, 
                            // The onOpenSync takes care of what should be given the highlights, but
                            // we want to highlight just the menu so that the onOpenSync can find the
                            // activeMenu.
                            HighlightOnOpen.HighlightJustMenu).get(noop);
                        }
                    }
                },
                cancelEvent: typeaheadCancel()
            }),
            Keying.config({
                mode: 'special',
                onDown: (comp, simulatedEvent) => {
                    // The navigation here will stop the "previewing" mode, because
                    // now the menu will get focus (fake focus, but focus nevertheless)
                    navigateList(comp, simulatedEvent, Highlighting.highlightFirst);
                    return Optional.some(true);
                },
                onEscape: (comp) => {
                    // Escape only has handling if the sandbox is visible. It has no meaning
                    // to the input itself.
                    const sandbox = Coupling.getCoupled(comp, 'sandbox');
                    if (Sandboxing.isOpen(sandbox)) {
                        Sandboxing.close(sandbox);
                        return Optional.some(true);
                    }
                    return Optional.none();
                },
                onUp: (comp, simulatedEvent) => {
                    // The navigation here will stop the "previewing" mode, because
                    // now the menu will get focus (fake focus, but focus nevertheless)
                    navigateList(comp, simulatedEvent, Highlighting.highlightLast);
                    return Optional.some(true);
                },
                onEnter: (comp) => {
                    const sandbox = Coupling.getCoupled(comp, 'sandbox');
                    const sandboxIsOpen = Sandboxing.isOpen(sandbox);
                    // 'Previewing' means that items are shown but none has been actively selected by the user.
                    // When previewing, all keyboard input should still be processed by the
                    // input itself, not the menu. The menu is not considered to have focus.
                    // 'Previewing' is turned on by (streaming) keystrokes, and turned off by
                    // successful interaction with the menu (navigation, highlighting, hovering).
                    // So if we aren't previewing, and the dropdown sandbox is open, then
                    // we process <enter> keys on the items in the menu. All this will do
                    // is trigger an itemExecute event. The typeahead events (in the spec below)
                    // are responsible for doing something with that event.
                    if (sandboxIsOpen && !detail.previewing.get()) {
                        return getActiveMenu(sandbox).bind((activeMenu) => Highlighting.getHighlighted(activeMenu)).map((item) => {
                            // And item was selected, so trigger execute and consider the
                            // <enter> key 'handled'
                            emitWith(comp, itemExecute(), { item });
                            return true;
                        });
                    }
                    else {
                        // We are either previewing, or the sandbox isn't open, so we should
                        // process the <enter> key inside the input itself. This should cancel
                        // any attempt to fetch data (the typeaheadCancel), and trigger the execute.
                        // We also close the sandbox if it's open.
                        const currentValue = Representing.getValue(comp);
                        emit(comp, typeaheadCancel());
                        detail.onExecute(sandbox, comp, currentValue);
                        // If we're open and previewing, close the sandbox after firing execute.
                        if (sandboxIsOpen) {
                            Sandboxing.close(sandbox);
                        }
                        return Optional.some(true);
                    }
                }
            }),
            Toggling.config({
                toggleClass: detail.markers.openClass,
                aria: {
                    mode: 'expanded'
                }
            }),
            Coupling.config({
                others: {
                    sandbox: (hotspot) => {
                        return makeSandbox$1(detail, hotspot, {
                            onOpen: () => Toggling.on(hotspot),
                            onClose: () => {
                                // TINY-9280: Remove aria-activedescendant that is set when menu item is highlighted
                                detail.lazyTypeaheadComp.get().each((input) => remove$8(input.element, 'aria-activedescendant'));
                                Toggling.off(hotspot);
                            }
                        });
                    }
                }
            }),
            config(typeaheadCustomEvents, [
                runOnAttached((typeaheadComp) => {
                    // Set up the reference to the typeahead, so that it can retrieved from
                    // the tiered menu part, even if the tieredmenu is in a different
                    // system / alloy root / mothership.
                    detail.lazyTypeaheadComp.set(Optional.some(typeaheadComp));
                }),
                runOnDetached((_typeaheadComp) => {
                    detail.lazyTypeaheadComp.set(Optional.none());
                }),
                runOnExecute$1((comp) => {
                    const onOpenSync = noop;
                    togglePopup(detail, mapFetch(comp), comp, externals, onOpenSync, HighlightOnOpen.HighlightMenuAndItem).get(noop);
                }),
                run$1(itemExecute(), (comp, se) => {
                    const sandbox = Coupling.getCoupled(comp, 'sandbox');
                    // Copy the value from the executed item into the input, because it was "chosen"
                    setValueFromItem(detail.model, comp, se.event.item);
                    emit(comp, typeaheadCancel());
                    detail.onItemExecute(comp, sandbox, se.event.item, Representing.getValue(comp));
                    Sandboxing.close(sandbox);
                    setCursorAtEnd(comp);
                })
            ].concat(detail.dismissOnBlur ? [
                run$1(postBlur(), (typeahead) => {
                    const sandbox = Coupling.getCoupled(typeahead, 'sandbox');
                    // Only close the sandbox if the focus isn't inside it!
                    if (search(sandbox.element).isNone()) {
                        Sandboxing.close(sandbox);
                    }
                })
            ] : []))
        ];
        // The order specified here isn't important. Alloy just requires a
        // deterministic order for the configured behaviours.
        const eventOrder = {
            [detachedFromDom()]: [
                Representing.name(),
                Streaming.name(),
                typeaheadCustomEvents
            ],
            ...detail.eventOrder,
        };
        return {
            uid: detail.uid,
            dom: dom$1(deepMerge(detail, {
                // TODO: Add aria-activedescendant attribute
                inputAttributes: {
                    'role': 'combobox',
                    'aria-autocomplete': 'list',
                    'aria-haspopup': 'true'
                }
            })),
            behaviours: {
                ...focusBehaviours$1,
                ...augment(detail.typeaheadBehaviours, behaviours)
            },
            eventOrder
        };
    };

    const schema = constant$1([
        option$3('lazySink'),
        required$1('fetch'),
        defaulted('minChars', 5),
        defaulted('responseTime', 1000),
        onHandler('onOpen'),
        // TODO: Remove dupe with Dropdown
        defaulted('getHotspot', Optional.some),
        defaulted('getAnchorOverrides', constant$1({})),
        defaulted('layouts', Optional.none()),
        defaulted('eventOrder', {}),
        // Information about what these model settings do can be found in TypeaheadTypes
        defaultedObjOf('model', {}, [
            defaulted('getDisplayText', (itemData) => itemData.meta !== undefined && itemData.meta.text !== undefined ? itemData.meta.text : itemData.value),
            defaulted('selectsOver', true),
            defaulted('populateFromBrowse', true)
        ]),
        onHandler('onSetValue'),
        onKeyboardHandler('onExecute'),
        onHandler('onItemExecute'),
        defaulted('inputClasses', []),
        defaulted('inputAttributes', {}),
        defaulted('inputStyles', {}),
        defaulted('matchWidth', true),
        defaulted('useMinWidth', false),
        defaulted('dismissOnBlur', true),
        markers$1(['openClass']),
        option$3('initialData'),
        option$3('listRole'),
        field('typeaheadBehaviours', [
            Focusing, Representing, Streaming, Keying, Toggling, Coupling
        ]),
        customField('lazyTypeaheadComp', () => Cell(Optional.none)),
        customField('previewing', () => Cell(true))
    ].concat(schema$9()).concat(sandboxFields()));
    const parts = constant$1([
        external$1({
            schema: [
                tieredMenuMarkers()
            ],
            name: 'menu',
            overrides: (detail) => {
                return {
                    fakeFocus: true,
                    onHighlightItem: (_tmenu, menu, item) => {
                        if (!detail.previewing.get()) {
                            // We need to use this type of reference, rather than just looking
                            // it up from the system by uid, because the input and the tieredmenu
                            // might be in different systems.
                            detail.lazyTypeaheadComp.get().each((input) => {
                                if (detail.model.populateFromBrowse) {
                                    setValueFromItem(detail.model, input, item);
                                }
                                // The focus is retained on the input element when the menu is shown, unlike the combobox, in which the focus is passed to the menu.
                                // This results in screen readers not being able to announce the menu or highlighted item.
                                // The solution is to tell screen readers which menu item is highlighted using the `aria-activedescendant` attribute.
                                // TINY-9280: The aria attribute is removed when the menu is closed.
                                // Since `onDehighlight` is called only when highlighting a new menu item, this will be handled in
                                // https://github.com/tinymce/tinymce/blob/2d8c1c034e8aa484b868a0c44605489ee0ca9cd4/modules/alloy/src/main/ts/ephox/alloy/ui/composite/TypeaheadSpec.ts#L282
                                getOpt(item.element, 'id').each((id) => set$9(input.element, 'aria-activedescendant', id));
                            });
                        }
                        else {
                            // ASSUMPTION: Currently, any interaction with the menu via the keyboard or the mouse
                            // will firstly clear previewing mode before triggering any highlights
                            // so if we are still in previewing mode by the time we get to the highlight call,
                            // that means that the highlight was triggered NOT by the user interacting
                            // with the menu, but instead by the Highlighting API call that happens automatically
                            // when a streamed keyboard input event is updating its results. That call will
                            // try to keep any active highlight if there already was one (defaulting to first
                            // if it can't find the original), but if there wasn't an active highlight, but
                            // it is using "selectsOver", it will just highlight the first item. In this
                            // latter case, it is only doing that so that selectsOver has something to copy.
                            // So all of the complex code below is trying to handle whether we should stay
                            // in previewing mode after this highlight, and the ONLY case where we should stay
                            // in previewing mode is that we were in previewing mode, we are using selectsOver,
                            // and the selectsOver failed to succeed. In that case, to stay in previewing mode,
                            // we want to cancel the highlight that we just made via the highlighting API
                            // and reset previewing to true. Otherwise, all codepaths should set previewing
                            // to false, because now we have a valid highlight.
                            //
                            // As of 2022-08-18, the selectsOver model is not in use by TinyMCE, so
                            // this subtle interaction is unfortunately largely untested. Also, if we can't
                            // get a reference to the typeahead input by lazyTypeaheadComp, then we don't
                            // change previewing, either. Note also, that it is likely that if we checked
                            // if selectsOver would succeed before setting the highlight in the streaming
                            // response, this could might be a lot easier to follow.
                            detail.lazyTypeaheadComp.get().each((input) => {
                                attemptSelectOver(detail.model, input, item).fold(
                                // If we are in "previewing" mode and we can't select over the
                                // thing that is first, then clear the highlight.
                                // Hopefully, this doesn't cause a flicker. Find a better
                                // way to do this.
                                () => {
                                    // If using "selectOver", we essentially want to cancel the highlight
                                    // that was only invoked just so that we'd have something to selectOver,
                                    // so we dehighlight, and then, importantly, *DON'T* clear previewing.
                                    // We'll set it to be true to be explicit, although it should
                                    // always be true if it reached here (unless an above function changed
                                    // it)
                                    if (detail.model.selectsOver) {
                                        Highlighting.dehighlight(menu, item);
                                        detail.previewing.set(true);
                                    }
                                    else {
                                        // Because we aren't using selectsOver mode, we now want to keep
                                        // whatever highlight we just made, and because we have a highlighted
                                        // item in the menu, we are no longer previewing.
                                        detail.previewing.set(false);
                                    }
                                }, ((selectOverTextInInput) => {
                                    // We have made a selection in the menu, and have selected over text
                                    // in the input, so clear previewing.
                                    selectOverTextInInput();
                                    detail.previewing.set(false);
                                }));
                            });
                        }
                    },
                    // Because the focus stays inside the input, this onExecute is fired when the
                    // user "clicks" on an item. The focusing behaviour should be configured
                    // so that items don't get focus, but they prevent a mousedown event from
                    // firing so that the typeahead doesn't lose focus. This is the handler
                    // for clicking on an item. We need to close the sandbox, update the typeahead
                    // to show the item clicked on, and fire an execute.
                    onExecute: (_menu, item) => {
                        // Note: This will only work when the typeahead and menu are in the same system.
                        return detail.lazyTypeaheadComp.get().map((typeahead) => {
                            emitWith(typeahead, itemExecute(), { item });
                            return true;
                        });
                    },
                    onHover: (menu, item) => {
                        // Hovering is also a user-initiated action, so previewing mode is over.
                        // TODO: Have a better API for managing state in between parts.
                        detail.previewing.set(false);
                        detail.lazyTypeaheadComp.get().each((input) => {
                            if (detail.model.populateFromBrowse) {
                                setValueFromItem(detail.model, input, item);
                            }
                        });
                    }
                };
            }
        })
    ]);

    const Typeahead = composite({
        name: 'Typeahead',
        configFields: schema(),
        partFields: parts(),
        factory: make$2
    });

    var global$b = tinymce.util.Tools.resolve('tinymce.ThemeManager');

    var global$a = tinymce.util.Tools.resolve('tinymce.util.Delay');

    var global$9 = tinymce.util.Tools.resolve('tinymce.dom.DOMUtils');

    var global$8 = tinymce.util.Tools.resolve('tinymce.EditorManager');

    var global$7 = tinymce.util.Tools.resolve('tinymce.Env');

    var ToolbarMode$1;
    (function (ToolbarMode) {
        ToolbarMode["default"] = "wrap";
        ToolbarMode["floating"] = "floating";
        ToolbarMode["sliding"] = "sliding";
        ToolbarMode["scrolling"] = "scrolling";
    })(ToolbarMode$1 || (ToolbarMode$1 = {}));
    var ToolbarLocation$1;
    (function (ToolbarLocation) {
        ToolbarLocation["auto"] = "auto";
        ToolbarLocation["top"] = "top";
        ToolbarLocation["bottom"] = "bottom";
    })(ToolbarLocation$1 || (ToolbarLocation$1 = {}));
    const option$2 = (name) => (editor) => editor.options.get(name);
    const wrapOptional = (fn) => (editor) => Optional.from(fn(editor));
    const register$f = (editor) => {
        const isPhone = global$7.deviceType.isPhone();
        const isMobile = global$7.deviceType.isTablet() || isPhone;
        const registerOption = editor.options.register;
        const stringOrFalseProcessor = (value) => isString(value) || value === false;
        const stringOrNumberProcessor = (value) => isString(value) || isNumber(value);
        registerOption('skin', {
            processor: (value) => isString(value) || value === false,
            default: 'oxide'
        });
        registerOption('skin_url', {
            processor: 'string'
        });
        registerOption('height', {
            processor: stringOrNumberProcessor,
            default: Math.max(editor.getElement().offsetHeight, 400)
        });
        registerOption('width', {
            processor: stringOrNumberProcessor,
            default: global$9.DOM.getStyle(editor.getElement(), 'width')
        });
        registerOption('min_height', {
            processor: 'number',
            default: 100
        });
        registerOption('min_width', {
            processor: 'number'
        });
        registerOption('max_height', {
            processor: 'number'
        });
        registerOption('max_width', {
            processor: 'number'
        });
        registerOption('style_formats', {
            processor: 'object[]'
        });
        registerOption('style_formats_merge', {
            processor: 'boolean',
            default: false
        });
        registerOption('style_formats_autohide', {
            processor: 'boolean',
            default: false
        });
        registerOption('line_height_formats', {
            processor: 'string',
            default: '1 1.1 1.2 1.3 1.4 1.5 2'
        });
        registerOption('font_family_formats', {
            processor: 'string',
            default: 'Andale Mono=andale mono,monospace;' +
                'Arial=arial,helvetica,sans-serif;' +
                'Arial Black=arial black,sans-serif;' +
                'Book Antiqua=book antiqua,palatino,serif;' +
                'Comic Sans MS=comic sans ms,sans-serif;' +
                'Courier New=courier new,courier,monospace;' +
                'Georgia=georgia,palatino,serif;' +
                'Helvetica=helvetica,arial,sans-serif;' +
                'Impact=impact,sans-serif;' +
                'Symbol=symbol;' +
                'Tahoma=tahoma,arial,helvetica,sans-serif;' +
                'Terminal=terminal,monaco,monospace;' +
                'Times New Roman=times new roman,times,serif;' +
                'Trebuchet MS=trebuchet ms,geneva,sans-serif;' +
                'Verdana=verdana,geneva,sans-serif;' +
                'Webdings=webdings;' +
                'Wingdings=wingdings,zapf dingbats'
        });
        registerOption('font_size_formats', {
            processor: 'string',
            default: '8pt 10pt 12pt 14pt 18pt 24pt 36pt'
        });
        registerOption('font_size_input_default_unit', {
            processor: 'string',
            default: 'pt'
        });
        registerOption('block_formats', {
            processor: 'string',
            default: 'Paragraph=p;' +
                'Heading 1=h1;' +
                'Heading 2=h2;' +
                'Heading 3=h3;' +
                'Heading 4=h4;' +
                'Heading 5=h5;' +
                'Heading 6=h6;' +
                'Preformatted=pre'
        });
        registerOption('content_langs', {
            processor: 'object[]'
        });
        registerOption('removed_menuitems', {
            processor: 'string',
            default: ''
        });
        registerOption('menubar', {
            processor: (value) => isString(value) || isBoolean(value),
            // Phones don't have a lot of screen space so disable the menubar
            default: !isPhone
        });
        registerOption('menu', {
            processor: 'object',
            default: {}
        });
        registerOption('toolbar', {
            processor: (value) => {
                if (isBoolean(value) || isString(value) || isArray(value)) {
                    return { value, valid: true };
                }
                else {
                    return { valid: false, message: 'Must be a boolean, string or array.' };
                }
            },
            default: true
        });
        // Register the toolbarN variations: toolbar1 -> toolbar9
        range$2(9, (num) => {
            registerOption('toolbar' + (num + 1), {
                processor: 'string'
            });
        });
        registerOption('toolbar_mode', {
            processor: 'string',
            // Use the default side-scrolling toolbar for tablets/phones
            default: isMobile ? 'scrolling' : 'floating'
        });
        registerOption('toolbar_groups', {
            processor: 'object',
            default: {}
        });
        registerOption('toolbar_location', {
            processor: 'string',
            default: ToolbarLocation$1.auto
        });
        registerOption('toolbar_persist', {
            processor: 'boolean',
            default: false
        });
        registerOption('toolbar_sticky', {
            processor: 'boolean',
            default: editor.inline
        });
        registerOption('toolbar_sticky_offset', {
            processor: 'number',
            default: 0
        });
        registerOption('fixed_toolbar_container', {
            processor: 'string',
            default: ''
        });
        registerOption('fixed_toolbar_container_target', {
            processor: 'object'
        });
        registerOption('ui_mode', {
            processor: 'string',
            default: 'combined'
        });
        registerOption('file_picker_callback', {
            processor: 'function'
        });
        registerOption('file_picker_validator_handler', {
            processor: 'function'
        });
        registerOption('file_picker_types', {
            processor: 'string'
        });
        registerOption('typeahead_urls', {
            processor: 'boolean',
            default: true
        });
        registerOption('anchor_top', {
            processor: stringOrFalseProcessor,
            default: '#top'
        });
        registerOption('anchor_bottom', {
            processor: stringOrFalseProcessor,
            default: '#bottom'
        });
        registerOption('draggable_modal', {
            processor: 'boolean',
            default: false
        });
        registerOption('statusbar', {
            processor: 'boolean',
            default: true
        });
        registerOption('elementpath', {
            processor: 'boolean',
            default: true
        });
        registerOption('branding', {
            processor: 'boolean',
            default: true
        });
        registerOption('promotion', {
            processor: 'boolean',
            default: true
        });
        registerOption('resize', {
            processor: (value) => value === 'both' || isBoolean(value),
            // Editor resize doesn't work on touch devices at this stage
            default: !global$7.deviceType.isTouch()
        });
        registerOption('sidebar_show', {
            processor: 'string'
        });
        // This option is being registered in the theme instead of the help plugin as it cannot be accessed from the theme when registered there
        registerOption('help_accessibility', {
            processor: 'boolean',
            default: editor.hasPlugin('help')
        });
        registerOption('default_font_stack', {
            processor: 'string[]',
            default: []
        });
    };
    const isReadOnly = option$2('readonly');
    const isDisabled = option$2('disabled');
    const getHeightOption = option$2('height');
    const getWidthOption = option$2('width');
    const getMinWidthOption = wrapOptional(option$2('min_width'));
    const getMinHeightOption = wrapOptional(option$2('min_height'));
    const getMaxWidthOption = wrapOptional(option$2('max_width'));
    const getMaxHeightOption = wrapOptional(option$2('max_height'));
    const getUserStyleFormats = wrapOptional(option$2('style_formats'));
    const shouldMergeStyleFormats = option$2('style_formats_merge');
    const shouldAutoHideStyleFormats = option$2('style_formats_autohide');
    const getContentLanguages = option$2('content_langs');
    const getRemovedMenuItems = option$2('removed_menuitems');
    const getToolbarMode = option$2('toolbar_mode');
    const getToolbarGroups = option$2('toolbar_groups');
    const getToolbarLocation = option$2('toolbar_location');
    const fixedContainerSelector = option$2('fixed_toolbar_container');
    const fixedToolbarContainerTarget = option$2('fixed_toolbar_container_target');
    const isToolbarPersist = option$2('toolbar_persist');
    const getStickyToolbarOffset = option$2('toolbar_sticky_offset');
    const getMenubar = option$2('menubar');
    const getToolbar = option$2('toolbar');
    const getFilePickerCallback = option$2('file_picker_callback');
    const getFilePickerValidatorHandler = option$2('file_picker_validator_handler');
    const getFontSizeInputDefaultUnit = option$2('font_size_input_default_unit');
    const getFilePickerTypes = option$2('file_picker_types');
    const useTypeaheadUrls = option$2('typeahead_urls');
    const getAnchorTop = option$2('anchor_top');
    const getAnchorBottom = option$2('anchor_bottom');
    const isDraggableModal$1 = option$2('draggable_modal');
    const useStatusBar = option$2('statusbar');
    const useElementPath = option$2('elementpath');
    const useBranding = option$2('branding');
    const getResize = option$2('resize');
    const getPasteAsText = option$2('paste_as_text');
    const getSidebarShow = option$2('sidebar_show');
    const promotionEnabled = option$2('promotion');
    const useHelpAccessibility = option$2('help_accessibility');
    const getDefaultFontStack = option$2('default_font_stack');
    const getSkin = option$2('skin');
    const isSkinDisabled = (editor) => editor.options.get('skin') === false;
    const isMenubarEnabled = (editor) => editor.options.get('menubar') !== false;
    const getSkinUrl = (editor) => {
        const skinUrl = editor.options.get('skin_url');
        if (isSkinDisabled(editor)) {
            return skinUrl;
        }
        else {
            if (skinUrl) {
                return editor.documentBaseURI.toAbsolute(skinUrl);
            }
            else {
                const skin = editor.options.get('skin');
                return global$8.baseURL + '/skins/ui/' + skin;
            }
        }
    };
    const getSkinUrlOption = (editor) => Optional.from(editor.options.get('skin_url'));
    const getLineHeightFormats = (editor) => editor.options.get('line_height_formats').split(' ');
    const isToolbarEnabled = (editor) => {
        const toolbar = getToolbar(editor);
        const isToolbarString = isString(toolbar);
        const isToolbarObjectArray = isArray(toolbar) && toolbar.length > 0;
        // Toolbar is enabled if its value is true, a string or non-empty object array, but not string array
        return !isMultipleToolbars(editor) && (isToolbarObjectArray || isToolbarString || toolbar === true);
    };
    // Convert toolbar<n> into toolbars array
    const getMultipleToolbarsOption = (editor) => {
        const toolbars = range$2(9, (num) => editor.options.get('toolbar' + (num + 1)));
        const toolbarArray = filter$2(toolbars, isString);
        return someIf(toolbarArray.length > 0, toolbarArray);
    };
    // Check if multiple toolbars is enabled
    // Multiple toolbars is enabled if toolbar value is a string array or if toolbar<n> is present
    const isMultipleToolbars = (editor) => getMultipleToolbarsOption(editor).fold(() => {
        const toolbar = getToolbar(editor);
        return isArrayOf(toolbar, isString) && toolbar.length > 0;
    }, always);
    const isToolbarLocationBottom = (editor) => getToolbarLocation(editor) === ToolbarLocation$1.bottom;
    const fixedContainerTarget = (editor) => {
        if (!editor.inline) {
            // fixed_toolbar_container(_target) is only available in inline mode
            return Optional.none();
        }
        const selector = fixedContainerSelector(editor) ?? '';
        if (selector.length > 0) {
            // If we have a valid selector
            return descendant(body(), selector);
        }
        const element = fixedToolbarContainerTarget(editor);
        if (isNonNullable(element)) {
            // If we have a valid target
            return Optional.some(SugarElement.fromDom(element));
        }
        return Optional.none();
    };
    const useFixedContainer = (editor) => editor.inline && fixedContainerTarget(editor).isSome();
    const getUiContainer = (editor) => {
        const fixedContainer = fixedContainerTarget(editor);
        return fixedContainer.getOrThunk(() => getContentContainer(getRootNode(SugarElement.fromDom(editor.getElement()))));
    };
    const isDistractionFree = (editor) => editor.inline && !isMenubarEnabled(editor) && !isToolbarEnabled(editor) && !isMultipleToolbars(editor);
    const isStickyToolbar = (editor) => {
        const isStickyToolbar = editor.options.get('toolbar_sticky');
        return (isStickyToolbar || editor.inline) && !useFixedContainer(editor) && !isDistractionFree(editor);
    };
    const isSplitUiMode = (editor) => !useFixedContainer(editor) && editor.options.get('ui_mode') === 'split';
    const getMenus = (editor) => {
        const menu = editor.options.get('menu');
        return map$1(menu, (menu) => ({ ...menu, items: menu.items }));
    };

    var Options = /*#__PURE__*/Object.freeze({
        __proto__: null,
        get ToolbarMode () { return ToolbarMode$1; },
        get ToolbarLocation () { return ToolbarLocation$1; },
        register: register$f,
        getSkinUrl: getSkinUrl,
        getSkinUrlOption: getSkinUrlOption,
        isReadOnly: isReadOnly,
        isDisabled: isDisabled,
        getSkin: getSkin,
        isSkinDisabled: isSkinDisabled,
        getHeightOption: getHeightOption,
        getWidthOption: getWidthOption,
        getMinWidthOption: getMinWidthOption,
        getMinHeightOption: getMinHeightOption,
        getMaxWidthOption: getMaxWidthOption,
        getMaxHeightOption: getMaxHeightOption,
        getUserStyleFormats: getUserStyleFormats,
        shouldMergeStyleFormats: shouldMergeStyleFormats,
        shouldAutoHideStyleFormats: shouldAutoHideStyleFormats,
        getLineHeightFormats: getLineHeightFormats,
        getContentLanguages: getContentLanguages,
        getRemovedMenuItems: getRemovedMenuItems,
        isMenubarEnabled: isMenubarEnabled,
        isMultipleToolbars: isMultipleToolbars,
        isToolbarEnabled: isToolbarEnabled,
        isToolbarPersist: isToolbarPersist,
        getMultipleToolbarsOption: getMultipleToolbarsOption,
        getUiContainer: getUiContainer,
        useFixedContainer: useFixedContainer,
        isSplitUiMode: isSplitUiMode,
        getToolbarMode: getToolbarMode,
        isDraggableModal: isDraggableModal$1,
        isDistractionFree: isDistractionFree,
        isStickyToolbar: isStickyToolbar,
        getStickyToolbarOffset: getStickyToolbarOffset,
        getToolbarLocation: getToolbarLocation,
        isToolbarLocationBottom: isToolbarLocationBottom,
        getToolbarGroups: getToolbarGroups,
        getMenus: getMenus,
        getMenubar: getMenubar,
        getToolbar: getToolbar,
        getFilePickerCallback: getFilePickerCallback,
        getFilePickerTypes: getFilePickerTypes,
        useTypeaheadUrls: useTypeaheadUrls,
        getAnchorTop: getAnchorTop,
        getAnchorBottom: getAnchorBottom,
        getFilePickerValidatorHandler: getFilePickerValidatorHandler,
        getFontSizeInputDefaultUnit: getFontSizeInputDefaultUnit,
        useStatusBar: useStatusBar,
        useElementPath: useElementPath,
        promotionEnabled: promotionEnabled,
        useBranding: useBranding,
        getResize: getResize,
        getPasteAsText: getPasteAsText,
        getSidebarShow: getSidebarShow,
        useHelpAccessibility: useHelpAccessibility,
        getDefaultFontStack: getDefaultFontStack
    });

    // See https://developer.mozilla.org/en-US/docs/Glossary/Scroll_container for what makes an element scrollable
    const nonScrollingOverflows = ['visible', 'hidden', 'clip'];
    const isScrollingOverflowValue = (value) => trim$1(value).length > 0 && !contains$2(nonScrollingOverflows, value);
    const isScroller = (elem) => {
        if (isHTMLElement(elem)) {
            const overflowX = get$e(elem, 'overflow-x');
            const overflowY = get$e(elem, 'overflow-y');
            return isScrollingOverflowValue(overflowX) || isScrollingOverflowValue(overflowY);
        }
        else {
            return false;
        }
    };
    const isFullscreen = (editor) => editor.plugins.fullscreen && editor.plugins.fullscreen.isFullscreen();
    // NOTE: Calculating the list of scrolling ancestors each time this function is called might
    // be unnecessary. It will depend on its usage.
    const detect = (editor, popupSinkElem) => {
        const ancestorsScrollers = ancestors(popupSinkElem, isScroller);
        // If there is no scrollable container, we try to see if it's in a shadow root, and try to traverse beyond the host of shadow root to retrieve the scrollable container
        // If it is not within a ShadowRoot, since if there's a scrollable container as the ancestors, then it would not execute the code below, or return an empty array if it's not in a ShadowRoot
        const scrollers = ancestorsScrollers.length === 0
            ? getShadowRoot(popupSinkElem).map(getShadowHost).map((x) => ancestors(x, isScroller)).getOr([])
            : ancestorsScrollers;
        return head(scrollers)
            .map((element) => ({
            element,
            // A list of all scrolling elements above the nearest scroller,
            // ordered from closest to popup -> closest to top of document
            others: scrollers.slice(1),
            isFullscreen: () => isFullscreen(editor)
        }));
    };
    const detectWhenSplitUiMode = (editor, popupSinkElem) => isSplitUiMode(editor) ? detect(editor, popupSinkElem) : Optional.none();
    // Using all the scrolling viewports in the ancestry, limit the absolute
    // coordinates of window so that the bounds are limited by all the scrolling
    // viewports.
    const getBoundsFrom = (sc) => {
        const scrollableBoxes = [
            // sc.element is the main scroller, others are *additional* scrollers above that
            // we need to combine all of them to constrain the bounds
            ...map$2(sc.others, box$1),
            win()
        ];
        return sc.isFullscreen() ? win() : constrainByMany(box$1(sc.element), scrollableBoxes);
    };

    /*! @license DOMPurify 3.2.6 | (c) Cure53 and other contributors | Released under the Apache license 2.0 and Mozilla Public License 2.0 | github.com/cure53/DOMPurify/blob/3.2.6/LICENSE */

    const {
      entries,
      setPrototypeOf,
      isFrozen,
      getPrototypeOf,
      getOwnPropertyDescriptor
    } = Object;
    let {
      freeze,
      seal,
      create: create$1
    } = Object; // eslint-disable-line import/no-mutable-exports
    let {
      apply,
      construct
    } = typeof Reflect !== 'undefined' && Reflect;
    if (!freeze) {
      freeze = function freeze(x) {
        return x;
      };
    }
    if (!seal) {
      seal = function seal(x) {
        return x;
      };
    }
    if (!apply) {
      apply = function apply(fun, thisValue, args) {
        return fun.apply(thisValue, args);
      };
    }
    if (!construct) {
      construct = function construct(Func, args) {
        return new Func(...args);
      };
    }
    const arrayForEach = unapply(Array.prototype.forEach);
    const arrayLastIndexOf = unapply(Array.prototype.lastIndexOf);
    const arrayPop = unapply(Array.prototype.pop);
    const arrayPush = unapply(Array.prototype.push);
    const arraySplice = unapply(Array.prototype.splice);
    const stringToLowerCase = unapply(String.prototype.toLowerCase);
    const stringToString = unapply(String.prototype.toString);
    const stringMatch = unapply(String.prototype.match);
    const stringReplace = unapply(String.prototype.replace);
    const stringIndexOf = unapply(String.prototype.indexOf);
    const stringTrim = unapply(String.prototype.trim);
    const objectHasOwnProperty = unapply(Object.prototype.hasOwnProperty);
    const regExpTest = unapply(RegExp.prototype.test);
    const typeErrorCreate = unconstruct(TypeError);
    /**
     * Creates a new function that calls the given function with a specified thisArg and arguments.
     *
     * @param func - The function to be wrapped and called.
     * @returns A new function that calls the given function with a specified thisArg and arguments.
     */
    function unapply(func) {
      return function (thisArg) {
        if (thisArg instanceof RegExp) {
          thisArg.lastIndex = 0;
        }
        for (var _len = arguments.length, args = new Array(_len > 1 ? _len - 1 : 0), _key = 1; _key < _len; _key++) {
          args[_key - 1] = arguments[_key];
        }
        return apply(func, thisArg, args);
      };
    }
    /**
     * Creates a new function that constructs an instance of the given constructor function with the provided arguments.
     *
     * @param func - The constructor function to be wrapped and called.
     * @returns A new function that constructs an instance of the given constructor function with the provided arguments.
     */
    function unconstruct(func) {
      return function () {
        for (var _len2 = arguments.length, args = new Array(_len2), _key2 = 0; _key2 < _len2; _key2++) {
          args[_key2] = arguments[_key2];
        }
        return construct(func, args);
      };
    }
    /**
     * Add properties to a lookup table
     *
     * @param set - The set to which elements will be added.
     * @param array - The array containing elements to be added to the set.
     * @param transformCaseFunc - An optional function to transform the case of each element before adding to the set.
     * @returns The modified set with added elements.
     */
    function addToSet(set, array) {
      let transformCaseFunc = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : stringToLowerCase;
      if (setPrototypeOf) {
        // Make 'in' and truthy checks like Boolean(set.constructor)
        // independent of any properties defined on Object.prototype.
        // Prevent prototype setters from intercepting set as a this value.
        setPrototypeOf(set, null);
      }
      let l = array.length;
      while (l--) {
        let element = array[l];
        if (typeof element === 'string') {
          const lcElement = transformCaseFunc(element);
          if (lcElement !== element) {
            // Config presets (e.g. tags.js, attrs.js) are immutable.
            if (!isFrozen(array)) {
              array[l] = lcElement;
            }
            element = lcElement;
          }
        }
        set[element] = true;
      }
      return set;
    }
    /**
     * Clean up an array to harden against CSPP
     *
     * @param array - The array to be cleaned.
     * @returns The cleaned version of the array
     */
    function cleanArray(array) {
      for (let index = 0; index < array.length; index++) {
        const isPropertyExist = objectHasOwnProperty(array, index);
        if (!isPropertyExist) {
          array[index] = null;
        }
      }
      return array;
    }
    /**
     * Shallow clone an object
     *
     * @param object - The object to be cloned.
     * @returns A new object that copies the original.
     */
    function clone(object) {
      const newObject = create$1(null);
      for (const [property, value] of entries(object)) {
        const isPropertyExist = objectHasOwnProperty(object, property);
        if (isPropertyExist) {
          if (Array.isArray(value)) {
            newObject[property] = cleanArray(value);
          } else if (value && typeof value === 'object' && value.constructor === Object) {
            newObject[property] = clone(value);
          } else {
            newObject[property] = value;
          }
        }
      }
      return newObject;
    }
    /**
     * This method automatically checks if the prop is function or getter and behaves accordingly.
     *
     * @param object - The object to look up the getter function in its prototype chain.
     * @param prop - The property name for which to find the getter function.
     * @returns The getter function found in the prototype chain or a fallback function.
     */
    function lookupGetter(object, prop) {
      while (object !== null) {
        const desc = getOwnPropertyDescriptor(object, prop);
        if (desc) {
          if (desc.get) {
            return unapply(desc.get);
          }
          if (typeof desc.value === 'function') {
            return unapply(desc.value);
          }
        }
        object = getPrototypeOf(object);
      }
      function fallbackValue() {
        return null;
      }
      return fallbackValue;
    }

    const html$1 = freeze(['a', 'abbr', 'acronym', 'address', 'area', 'article', 'aside', 'audio', 'b', 'bdi', 'bdo', 'big', 'blink', 'blockquote', 'body', 'br', 'button', 'canvas', 'caption', 'center', 'cite', 'code', 'col', 'colgroup', 'content', 'data', 'datalist', 'dd', 'decorator', 'del', 'details', 'dfn', 'dialog', 'dir', 'div', 'dl', 'dt', 'element', 'em', 'fieldset', 'figcaption', 'figure', 'font', 'footer', 'form', 'h1', 'h2', 'h3', 'h4', 'h5', 'h6', 'head', 'header', 'hgroup', 'hr', 'html', 'i', 'img', 'input', 'ins', 'kbd', 'label', 'legend', 'li', 'main', 'map', 'mark', 'marquee', 'menu', 'menuitem', 'meter', 'nav', 'nobr', 'ol', 'optgroup', 'option', 'output', 'p', 'picture', 'pre', 'progress', 'q', 'rp', 'rt', 'ruby', 's', 'samp', 'section', 'select', 'shadow', 'small', 'source', 'spacer', 'span', 'strike', 'strong', 'style', 'sub', 'summary', 'sup', 'table', 'tbody', 'td', 'template', 'textarea', 'tfoot', 'th', 'thead', 'time', 'tr', 'track', 'tt', 'u', 'ul', 'var', 'video', 'wbr']);
    const svg$1 = freeze(['svg', 'a', 'altglyph', 'altglyphdef', 'altglyphitem', 'animatecolor', 'animatemotion', 'animatetransform', 'circle', 'clippath', 'defs', 'desc', 'ellipse', 'filter', 'font', 'g', 'glyph', 'glyphref', 'hkern', 'image', 'line', 'lineargradient', 'marker', 'mask', 'metadata', 'mpath', 'path', 'pattern', 'polygon', 'polyline', 'radialgradient', 'rect', 'stop', 'style', 'switch', 'symbol', 'text', 'textpath', 'title', 'tref', 'tspan', 'view', 'vkern']);
    const svgFilters = freeze(['feBlend', 'feColorMatrix', 'feComponentTransfer', 'feComposite', 'feConvolveMatrix', 'feDiffuseLighting', 'feDisplacementMap', 'feDistantLight', 'feDropShadow', 'feFlood', 'feFuncA', 'feFuncB', 'feFuncG', 'feFuncR', 'feGaussianBlur', 'feImage', 'feMerge', 'feMergeNode', 'feMorphology', 'feOffset', 'fePointLight', 'feSpecularLighting', 'feSpotLight', 'feTile', 'feTurbulence']);
    // List of SVG elements that are disallowed by default.
    // We still need to know them so that we can do namespace
    // checks properly in case one wants to add them to
    // allow-list.
    const svgDisallowed = freeze(['animate', 'color-profile', 'cursor', 'discard', 'font-face', 'font-face-format', 'font-face-name', 'font-face-src', 'font-face-uri', 'foreignobject', 'hatch', 'hatchpath', 'mesh', 'meshgradient', 'meshpatch', 'meshrow', 'missing-glyph', 'script', 'set', 'solidcolor', 'unknown', 'use']);
    const mathMl$1 = freeze(['math', 'menclose', 'merror', 'mfenced', 'mfrac', 'mglyph', 'mi', 'mlabeledtr', 'mmultiscripts', 'mn', 'mo', 'mover', 'mpadded', 'mphantom', 'mroot', 'mrow', 'ms', 'mspace', 'msqrt', 'mstyle', 'msub', 'msup', 'msubsup', 'mtable', 'mtd', 'mtext', 'mtr', 'munder', 'munderover', 'mprescripts']);
    // Similarly to SVG, we want to know all MathML elements,
    // even those that we disallow by default.
    const mathMlDisallowed = freeze(['maction', 'maligngroup', 'malignmark', 'mlongdiv', 'mscarries', 'mscarry', 'msgroup', 'mstack', 'msline', 'msrow', 'semantics', 'annotation', 'annotation-xml', 'mprescripts', 'none']);
    const text$1 = freeze(['#text']);

    const html = freeze(['accept', 'action', 'align', 'alt', 'autocapitalize', 'autocomplete', 'autopictureinpicture', 'autoplay', 'background', 'bgcolor', 'border', 'capture', 'cellpadding', 'cellspacing', 'checked', 'cite', 'class', 'clear', 'color', 'cols', 'colspan', 'controls', 'controlslist', 'coords', 'crossorigin', 'datetime', 'decoding', 'default', 'dir', 'disabled', 'disablepictureinpicture', 'disableremoteplayback', 'download', 'draggable', 'enctype', 'enterkeyhint', 'face', 'for', 'headers', 'height', 'hidden', 'high', 'href', 'hreflang', 'id', 'inputmode', 'integrity', 'ismap', 'kind', 'label', 'lang', 'list', 'loading', 'loop', 'low', 'max', 'maxlength', 'media', 'method', 'min', 'minlength', 'multiple', 'muted', 'name', 'nonce', 'noshade', 'novalidate', 'nowrap', 'open', 'optimum', 'pattern', 'placeholder', 'playsinline', 'popover', 'popovertarget', 'popovertargetaction', 'poster', 'preload', 'pubdate', 'radiogroup', 'readonly', 'rel', 'required', 'rev', 'reversed', 'role', 'rows', 'rowspan', 'spellcheck', 'scope', 'selected', 'shape', 'size', 'sizes', 'span', 'srclang', 'start', 'src', 'srcset', 'step', 'style', 'summary', 'tabindex', 'title', 'translate', 'type', 'usemap', 'valign', 'value', 'width', 'wrap', 'xmlns', 'slot']);
    const svg = freeze(['accent-height', 'accumulate', 'additive', 'alignment-baseline', 'amplitude', 'ascent', 'attributename', 'attributetype', 'azimuth', 'basefrequency', 'baseline-shift', 'begin', 'bias', 'by', 'class', 'clip', 'clippathunits', 'clip-path', 'clip-rule', 'color', 'color-interpolation', 'color-interpolation-filters', 'color-profile', 'color-rendering', 'cx', 'cy', 'd', 'dx', 'dy', 'diffuseconstant', 'direction', 'display', 'divisor', 'dur', 'edgemode', 'elevation', 'end', 'exponent', 'fill', 'fill-opacity', 'fill-rule', 'filter', 'filterunits', 'flood-color', 'flood-opacity', 'font-family', 'font-size', 'font-size-adjust', 'font-stretch', 'font-style', 'font-variant', 'font-weight', 'fx', 'fy', 'g1', 'g2', 'glyph-name', 'glyphref', 'gradientunits', 'gradienttransform', 'height', 'href', 'id', 'image-rendering', 'in', 'in2', 'intercept', 'k', 'k1', 'k2', 'k3', 'k4', 'kerning', 'keypoints', 'keysplines', 'keytimes', 'lang', 'lengthadjust', 'letter-spacing', 'kernelmatrix', 'kernelunitlength', 'lighting-color', 'local', 'marker-end', 'marker-mid', 'marker-start', 'markerheight', 'markerunits', 'markerwidth', 'maskcontentunits', 'maskunits', 'max', 'mask', 'media', 'method', 'mode', 'min', 'name', 'numoctaves', 'offset', 'operator', 'opacity', 'order', 'orient', 'orientation', 'origin', 'overflow', 'paint-order', 'path', 'pathlength', 'patterncontentunits', 'patterntransform', 'patternunits', 'points', 'preservealpha', 'preserveaspectratio', 'primitiveunits', 'r', 'rx', 'ry', 'radius', 'refx', 'refy', 'repeatcount', 'repeatdur', 'restart', 'result', 'rotate', 'scale', 'seed', 'shape-rendering', 'slope', 'specularconstant', 'specularexponent', 'spreadmethod', 'startoffset', 'stddeviation', 'stitchtiles', 'stop-color', 'stop-opacity', 'stroke-dasharray', 'stroke-dashoffset', 'stroke-linecap', 'stroke-linejoin', 'stroke-miterlimit', 'stroke-opacity', 'stroke', 'stroke-width', 'style', 'surfacescale', 'systemlanguage', 'tabindex', 'tablevalues', 'targetx', 'targety', 'transform', 'transform-origin', 'text-anchor', 'text-decoration', 'text-rendering', 'textlength', 'type', 'u1', 'u2', 'unicode', 'values', 'viewbox', 'visibility', 'version', 'vert-adv-y', 'vert-origin-x', 'vert-origin-y', 'width', 'word-spacing', 'wrap', 'writing-mode', 'xchannelselector', 'ychannelselector', 'x', 'x1', 'x2', 'xmlns', 'y', 'y1', 'y2', 'z', 'zoomandpan']);
    const mathMl = freeze(['accent', 'accentunder', 'align', 'bevelled', 'close', 'columnsalign', 'columnlines', 'columnspan', 'denomalign', 'depth', 'dir', 'display', 'displaystyle', 'encoding', 'fence', 'frame', 'height', 'href', 'id', 'largeop', 'length', 'linethickness', 'lspace', 'lquote', 'mathbackground', 'mathcolor', 'mathsize', 'mathvariant', 'maxsize', 'minsize', 'movablelimits', 'notation', 'numalign', 'open', 'rowalign', 'rowlines', 'rowspacing', 'rowspan', 'rspace', 'rquote', 'scriptlevel', 'scriptminsize', 'scriptsizemultiplier', 'selection', 'separator', 'separators', 'stretchy', 'subscriptshift', 'supscriptshift', 'symmetric', 'voffset', 'width', 'xmlns']);
    const xml = freeze(['xlink:href', 'xml:id', 'xlink:title', 'xml:space', 'xmlns:xlink']);

    // eslint-disable-next-line unicorn/better-regex
    const MUSTACHE_EXPR = seal(/\{\{[\w\W]*|[\w\W]*\}\}/gm); // Specify template detection regex for SAFE_FOR_TEMPLATES mode
    const ERB_EXPR = seal(/<%[\w\W]*|[\w\W]*%>/gm);
    const TMPLIT_EXPR = seal(/\$\{[\w\W]*/gm); // eslint-disable-line unicorn/better-regex
    const DATA_ATTR = seal(/^data-[\-\w.\u00B7-\uFFFF]+$/); // eslint-disable-line no-useless-escape
    const ARIA_ATTR = seal(/^aria-[\-\w]+$/); // eslint-disable-line no-useless-escape
    const IS_ALLOWED_URI = seal(/^(?:(?:(?:f|ht)tps?|mailto|tel|callto|sms|cid|xmpp|matrix):|[^a-z]|[a-z+.\-]+(?:[^a-z+.\-:]|$))/i // eslint-disable-line no-useless-escape
    );
    const IS_SCRIPT_OR_DATA = seal(/^(?:\w+script|data):/i);
    const ATTR_WHITESPACE = seal(/[\u0000-\u0020\u00A0\u1680\u180E\u2000-\u2029\u205F\u3000]/g // eslint-disable-line no-control-regex
    );
    const DOCTYPE_NAME = seal(/^html$/i);
    const CUSTOM_ELEMENT = seal(/^[a-z][.\w]*(-[.\w]+)+$/i);

    var EXPRESSIONS = /*#__PURE__*/Object.freeze({
      __proto__: null,
      ARIA_ATTR: ARIA_ATTR,
      ATTR_WHITESPACE: ATTR_WHITESPACE,
      CUSTOM_ELEMENT: CUSTOM_ELEMENT,
      DATA_ATTR: DATA_ATTR,
      DOCTYPE_NAME: DOCTYPE_NAME,
      ERB_EXPR: ERB_EXPR,
      IS_ALLOWED_URI: IS_ALLOWED_URI,
      IS_SCRIPT_OR_DATA: IS_SCRIPT_OR_DATA,
      MUSTACHE_EXPR: MUSTACHE_EXPR,
      TMPLIT_EXPR: TMPLIT_EXPR
    });

    /* eslint-disable @typescript-eslint/indent */
    // https://developer.mozilla.org/en-US/docs/Web/API/Node/nodeType
    const NODE_TYPE = {
      element: 1,
      attribute: 2,
      text: 3,
      cdataSection: 4,
      entityReference: 5,
      // Deprecated
      entityNode: 6,
      // Deprecated
      progressingInstruction: 7,
      comment: 8,
      document: 9,
      documentType: 10,
      documentFragment: 11,
      notation: 12 // Deprecated
    };
    const getGlobal = function getGlobal() {
      return typeof window === 'undefined' ? null : window;
    };
    /**
     * Creates a no-op policy for internal use only.
     * Don't export this function outside this module!
     * @param trustedTypes The policy factory.
     * @param purifyHostElement The Script element used to load DOMPurify (to determine policy name suffix).
     * @return The policy created (or null, if Trusted Types
     * are not supported or creating the policy failed).
     */
    const _createTrustedTypesPolicy = function _createTrustedTypesPolicy(trustedTypes, purifyHostElement) {
      if (typeof trustedTypes !== 'object' || typeof trustedTypes.createPolicy !== 'function') {
        return null;
      }
      // Allow the callers to control the unique policy name
      // by adding a data-tt-policy-suffix to the script element with the DOMPurify.
      // Policy creation with duplicate names throws in Trusted Types.
      let suffix = null;
      const ATTR_NAME = 'data-tt-policy-suffix';
      if (purifyHostElement && purifyHostElement.hasAttribute(ATTR_NAME)) {
        suffix = purifyHostElement.getAttribute(ATTR_NAME);
      }
      const policyName = 'dompurify' + (suffix ? '#' + suffix : '');
      try {
        return trustedTypes.createPolicy(policyName, {
          createHTML(html) {
            return html;
          },
          createScriptURL(scriptUrl) {
            return scriptUrl;
          }
        });
      } catch (_) {
        // Policy creation failed (most likely another DOMPurify script has
        // already run). Skip creating the policy, as this will only cause errors
        // if TT are enforced.
        console.warn('TrustedTypes policy ' + policyName + ' could not be created.');
        return null;
      }
    };
    const _createHooksMap = function _createHooksMap() {
      return {
        afterSanitizeAttributes: [],
        afterSanitizeElements: [],
        afterSanitizeShadowDOM: [],
        beforeSanitizeAttributes: [],
        beforeSanitizeElements: [],
        beforeSanitizeShadowDOM: [],
        uponSanitizeAttribute: [],
        uponSanitizeElement: [],
        uponSanitizeShadowNode: []
      };
    };
    function createDOMPurify() {
      let window = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : getGlobal();
      const DOMPurify = root => createDOMPurify(root);
      DOMPurify.version = '3.2.6';
      DOMPurify.removed = [];
      if (!window || !window.document || window.document.nodeType !== NODE_TYPE.document || !window.Element) {
        // Not running in a browser, provide a factory function
        // so that you can pass your own Window
        DOMPurify.isSupported = false;
        return DOMPurify;
      }
      let {
        document
      } = window;
      const originalDocument = document;
      const currentScript = originalDocument.currentScript;
      const {
        DocumentFragment,
        HTMLTemplateElement,
        Node,
        Element,
        NodeFilter,
        NamedNodeMap = window.NamedNodeMap || window.MozNamedAttrMap,
        HTMLFormElement,
        DOMParser,
        trustedTypes
      } = window;
      const ElementPrototype = Element.prototype;
      const cloneNode = lookupGetter(ElementPrototype, 'cloneNode');
      const remove = lookupGetter(ElementPrototype, 'remove');
      const getNextSibling = lookupGetter(ElementPrototype, 'nextSibling');
      const getChildNodes = lookupGetter(ElementPrototype, 'childNodes');
      const getParentNode = lookupGetter(ElementPrototype, 'parentNode');
      // As per issue #47, the web-components registry is inherited by a
      // new document created via createHTMLDocument. As per the spec
      // (http://w3c.github.io/webcomponents/spec/custom/#creating-and-passing-registries)
      // a new empty registry is used when creating a template contents owner
      // document, so we use that as our parent document to ensure nothing
      // is inherited.
      if (typeof HTMLTemplateElement === 'function') {
        const template = document.createElement('template');
        if (template.content && template.content.ownerDocument) {
          document = template.content.ownerDocument;
        }
      }
      let trustedTypesPolicy;
      let emptyHTML = '';
      const {
        implementation,
        createNodeIterator,
        createDocumentFragment,
        getElementsByTagName
      } = document;
      const {
        importNode
      } = originalDocument;
      let hooks = _createHooksMap();
      /**
       * Expose whether this browser supports running the full DOMPurify.
       */
      DOMPurify.isSupported = typeof entries === 'function' && typeof getParentNode === 'function' && implementation && implementation.createHTMLDocument !== undefined;
      const {
        MUSTACHE_EXPR,
        ERB_EXPR,
        TMPLIT_EXPR,
        DATA_ATTR,
        ARIA_ATTR,
        IS_SCRIPT_OR_DATA,
        ATTR_WHITESPACE,
        CUSTOM_ELEMENT
      } = EXPRESSIONS;
      let {
        IS_ALLOWED_URI: IS_ALLOWED_URI$1
      } = EXPRESSIONS;
      /**
       * We consider the elements and attributes below to be safe. Ideally
       * don't add any new ones but feel free to remove unwanted ones.
       */
      /* allowed element names */
      let ALLOWED_TAGS = null;
      const DEFAULT_ALLOWED_TAGS = addToSet({}, [...html$1, ...svg$1, ...svgFilters, ...mathMl$1, ...text$1]);
      /* Allowed attribute names */
      let ALLOWED_ATTR = null;
      const DEFAULT_ALLOWED_ATTR = addToSet({}, [...html, ...svg, ...mathMl, ...xml]);
      /*
       * Configure how DOMPurify should handle custom elements and their attributes as well as customized built-in elements.
       * @property {RegExp|Function|null} tagNameCheck one of [null, regexPattern, predicate]. Default: `null` (disallow any custom elements)
       * @property {RegExp|Function|null} attributeNameCheck one of [null, regexPattern, predicate]. Default: `null` (disallow any attributes not on the allow list)
       * @property {boolean} allowCustomizedBuiltInElements allow custom elements derived from built-ins if they pass CUSTOM_ELEMENT_HANDLING.tagNameCheck. Default: `false`.
       */
      let CUSTOM_ELEMENT_HANDLING = Object.seal(create$1(null, {
        tagNameCheck: {
          writable: true,
          configurable: false,
          enumerable: true,
          value: null
        },
        attributeNameCheck: {
          writable: true,
          configurable: false,
          enumerable: true,
          value: null
        },
        allowCustomizedBuiltInElements: {
          writable: true,
          configurable: false,
          enumerable: true,
          value: false
        }
      }));
      /* Explicitly forbidden tags (overrides ALLOWED_TAGS/ADD_TAGS) */
      let FORBID_TAGS = null;
      /* Explicitly forbidden attributes (overrides ALLOWED_ATTR/ADD_ATTR) */
      let FORBID_ATTR = null;
      /* Decide if ARIA attributes are okay */
      let ALLOW_ARIA_ATTR = true;
      /* Decide if custom data attributes are okay */
      let ALLOW_DATA_ATTR = true;
      /* Decide if unknown protocols are okay */
      let ALLOW_UNKNOWN_PROTOCOLS = false;
      /* Decide if self-closing tags in attributes are allowed.
       * Usually removed due to a mXSS issue in jQuery 3.0 */
      let ALLOW_SELF_CLOSE_IN_ATTR = true;
      /* Output should be safe for common template engines.
       * This means, DOMPurify removes data attributes, mustaches and ERB
       */
      let SAFE_FOR_TEMPLATES = false;
      /* Output should be safe even for XML used within HTML and alike.
       * This means, DOMPurify removes comments when containing risky content.
       */
      let SAFE_FOR_XML = true;
      /* Decide if document with <html>... should be returned */
      let WHOLE_DOCUMENT = false;
      /* Track whether config is already set on this instance of DOMPurify. */
      let SET_CONFIG = false;
      /* Decide if all elements (e.g. style, script) must be children of
       * document.body. By default, browsers might move them to document.head */
      let FORCE_BODY = false;
      /* Decide if a DOM `HTMLBodyElement` should be returned, instead of a html
       * string (or a TrustedHTML object if Trusted Types are supported).
       * If `WHOLE_DOCUMENT` is enabled a `HTMLHtmlElement` will be returned instead
       */
      let RETURN_DOM = false;
      /* Decide if a DOM `DocumentFragment` should be returned, instead of a html
       * string  (or a TrustedHTML object if Trusted Types are supported) */
      let RETURN_DOM_FRAGMENT = false;
      /* Try to return a Trusted Type object instead of a string, return a string in
       * case Trusted Types are not supported  */
      let RETURN_TRUSTED_TYPE = false;
      /* Output should be free from DOM clobbering attacks?
       * This sanitizes markups named with colliding, clobberable built-in DOM APIs.
       */
      let SANITIZE_DOM = true;
      /* Achieve full DOM Clobbering protection by isolating the namespace of named
       * properties and JS variables, mitigating attacks that abuse the HTML/DOM spec rules.
       *
       * HTML/DOM spec rules that enable DOM Clobbering:
       *   - Named Access on Window (§7.3.3)
       *   - DOM Tree Accessors (§3.1.5)
       *   - Form Element Parent-Child Relations (§4.10.3)
       *   - Iframe srcdoc / Nested WindowProxies (§4.8.5)
       *   - HTMLCollection (§4.2.10.2)
       *
       * Namespace isolation is implemented by prefixing `id` and `name` attributes
       * with a constant string, i.e., `user-content-`
       */
      let SANITIZE_NAMED_PROPS = false;
      const SANITIZE_NAMED_PROPS_PREFIX = 'user-content-';
      /* Keep element content when removing element? */
      let KEEP_CONTENT = true;
      /* If a `Node` is passed to sanitize(), then performs sanitization in-place instead
       * of importing it into a new Document and returning a sanitized copy */
      let IN_PLACE = false;
      /* Allow usage of profiles like html, svg and mathMl */
      let USE_PROFILES = {};
      /* Tags to ignore content of when KEEP_CONTENT is true */
      let FORBID_CONTENTS = null;
      const DEFAULT_FORBID_CONTENTS = addToSet({}, ['annotation-xml', 'audio', 'colgroup', 'desc', 'foreignobject', 'head', 'iframe', 'math', 'mi', 'mn', 'mo', 'ms', 'mtext', 'noembed', 'noframes', 'noscript', 'plaintext', 'script', 'style', 'svg', 'template', 'thead', 'title', 'video', 'xmp']);
      /* Tags that are safe for data: URIs */
      let DATA_URI_TAGS = null;
      const DEFAULT_DATA_URI_TAGS = addToSet({}, ['audio', 'video', 'img', 'source', 'image', 'track']);
      /* Attributes safe for values like "javascript:" */
      let URI_SAFE_ATTRIBUTES = null;
      const DEFAULT_URI_SAFE_ATTRIBUTES = addToSet({}, ['alt', 'class', 'for', 'id', 'label', 'name', 'pattern', 'placeholder', 'role', 'summary', 'title', 'value', 'style', 'xmlns']);
      const MATHML_NAMESPACE = 'http://www.w3.org/1998/Math/MathML';
      const SVG_NAMESPACE = 'http://www.w3.org/2000/svg';
      const HTML_NAMESPACE = 'http://www.w3.org/1999/xhtml';
      /* Document namespace */
      let NAMESPACE = HTML_NAMESPACE;
      let IS_EMPTY_INPUT = false;
      /* Allowed XHTML+XML namespaces */
      let ALLOWED_NAMESPACES = null;
      const DEFAULT_ALLOWED_NAMESPACES = addToSet({}, [MATHML_NAMESPACE, SVG_NAMESPACE, HTML_NAMESPACE], stringToString);
      let MATHML_TEXT_INTEGRATION_POINTS = addToSet({}, ['mi', 'mo', 'mn', 'ms', 'mtext']);
      let HTML_INTEGRATION_POINTS = addToSet({}, ['annotation-xml']);
      // Certain elements are allowed in both SVG and HTML
      // namespace. We need to specify them explicitly
      // so that they don't get erroneously deleted from
      // HTML namespace.
      const COMMON_SVG_AND_HTML_ELEMENTS = addToSet({}, ['title', 'style', 'font', 'a', 'script']);
      /* Parsing of strict XHTML documents */
      let PARSER_MEDIA_TYPE = null;
      const SUPPORTED_PARSER_MEDIA_TYPES = ['application/xhtml+xml', 'text/html'];
      const DEFAULT_PARSER_MEDIA_TYPE = 'text/html';
      let transformCaseFunc = null;
      /* Keep a reference to config to pass to hooks */
      let CONFIG = null;
      /* Ideally, do not touch anything below this line */
      /* ______________________________________________ */
      const formElement = document.createElement('form');
      const isRegexOrFunction = function isRegexOrFunction(testValue) {
        return testValue instanceof RegExp || testValue instanceof Function;
      };
      /**
       * _parseConfig
       *
       * @param cfg optional config literal
       */
      // eslint-disable-next-line complexity
      const _parseConfig = function _parseConfig() {
        let cfg = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {};
        if (CONFIG && CONFIG === cfg) {
          return;
        }
        /* Shield configuration object from tampering */
        if (!cfg || typeof cfg !== 'object') {
          cfg = {};
        }
        /* Shield configuration object from prototype pollution */
        cfg = clone(cfg);
        PARSER_MEDIA_TYPE =
        // eslint-disable-next-line unicorn/prefer-includes
        SUPPORTED_PARSER_MEDIA_TYPES.indexOf(cfg.PARSER_MEDIA_TYPE) === -1 ? DEFAULT_PARSER_MEDIA_TYPE : cfg.PARSER_MEDIA_TYPE;
        // HTML tags and attributes are not case-sensitive, converting to lowercase. Keeping XHTML as is.
        transformCaseFunc = PARSER_MEDIA_TYPE === 'application/xhtml+xml' ? stringToString : stringToLowerCase;
        /* Set configuration parameters */
        ALLOWED_TAGS = objectHasOwnProperty(cfg, 'ALLOWED_TAGS') ? addToSet({}, cfg.ALLOWED_TAGS, transformCaseFunc) : DEFAULT_ALLOWED_TAGS;
        ALLOWED_ATTR = objectHasOwnProperty(cfg, 'ALLOWED_ATTR') ? addToSet({}, cfg.ALLOWED_ATTR, transformCaseFunc) : DEFAULT_ALLOWED_ATTR;
        ALLOWED_NAMESPACES = objectHasOwnProperty(cfg, 'ALLOWED_NAMESPACES') ? addToSet({}, cfg.ALLOWED_NAMESPACES, stringToString) : DEFAULT_ALLOWED_NAMESPACES;
        URI_SAFE_ATTRIBUTES = objectHasOwnProperty(cfg, 'ADD_URI_SAFE_ATTR') ? addToSet(clone(DEFAULT_URI_SAFE_ATTRIBUTES), cfg.ADD_URI_SAFE_ATTR, transformCaseFunc) : DEFAULT_URI_SAFE_ATTRIBUTES;
        DATA_URI_TAGS = objectHasOwnProperty(cfg, 'ADD_DATA_URI_TAGS') ? addToSet(clone(DEFAULT_DATA_URI_TAGS), cfg.ADD_DATA_URI_TAGS, transformCaseFunc) : DEFAULT_DATA_URI_TAGS;
        FORBID_CONTENTS = objectHasOwnProperty(cfg, 'FORBID_CONTENTS') ? addToSet({}, cfg.FORBID_CONTENTS, transformCaseFunc) : DEFAULT_FORBID_CONTENTS;
        FORBID_TAGS = objectHasOwnProperty(cfg, 'FORBID_TAGS') ? addToSet({}, cfg.FORBID_TAGS, transformCaseFunc) : clone({});
        FORBID_ATTR = objectHasOwnProperty(cfg, 'FORBID_ATTR') ? addToSet({}, cfg.FORBID_ATTR, transformCaseFunc) : clone({});
        USE_PROFILES = objectHasOwnProperty(cfg, 'USE_PROFILES') ? cfg.USE_PROFILES : false;
        ALLOW_ARIA_ATTR = cfg.ALLOW_ARIA_ATTR !== false; // Default true
        ALLOW_DATA_ATTR = cfg.ALLOW_DATA_ATTR !== false; // Default true
        ALLOW_UNKNOWN_PROTOCOLS = cfg.ALLOW_UNKNOWN_PROTOCOLS || false; // Default false
        ALLOW_SELF_CLOSE_IN_ATTR = cfg.ALLOW_SELF_CLOSE_IN_ATTR !== false; // Default true
        SAFE_FOR_TEMPLATES = cfg.SAFE_FOR_TEMPLATES || false; // Default false
        SAFE_FOR_XML = cfg.SAFE_FOR_XML !== false; // Default true
        WHOLE_DOCUMENT = cfg.WHOLE_DOCUMENT || false; // Default false
        RETURN_DOM = cfg.RETURN_DOM || false; // Default false
        RETURN_DOM_FRAGMENT = cfg.RETURN_DOM_FRAGMENT || false; // Default false
        RETURN_TRUSTED_TYPE = cfg.RETURN_TRUSTED_TYPE || false; // Default false
        FORCE_BODY = cfg.FORCE_BODY || false; // Default false
        SANITIZE_DOM = cfg.SANITIZE_DOM !== false; // Default true
        SANITIZE_NAMED_PROPS = cfg.SANITIZE_NAMED_PROPS || false; // Default false
        KEEP_CONTENT = cfg.KEEP_CONTENT !== false; // Default true
        IN_PLACE = cfg.IN_PLACE || false; // Default false
        IS_ALLOWED_URI$1 = cfg.ALLOWED_URI_REGEXP || IS_ALLOWED_URI;
        NAMESPACE = cfg.NAMESPACE || HTML_NAMESPACE;
        MATHML_TEXT_INTEGRATION_POINTS = cfg.MATHML_TEXT_INTEGRATION_POINTS || MATHML_TEXT_INTEGRATION_POINTS;
        HTML_INTEGRATION_POINTS = cfg.HTML_INTEGRATION_POINTS || HTML_INTEGRATION_POINTS;
        CUSTOM_ELEMENT_HANDLING = cfg.CUSTOM_ELEMENT_HANDLING || {};
        if (cfg.CUSTOM_ELEMENT_HANDLING && isRegexOrFunction(cfg.CUSTOM_ELEMENT_HANDLING.tagNameCheck)) {
          CUSTOM_ELEMENT_HANDLING.tagNameCheck = cfg.CUSTOM_ELEMENT_HANDLING.tagNameCheck;
        }
        if (cfg.CUSTOM_ELEMENT_HANDLING && isRegexOrFunction(cfg.CUSTOM_ELEMENT_HANDLING.attributeNameCheck)) {
          CUSTOM_ELEMENT_HANDLING.attributeNameCheck = cfg.CUSTOM_ELEMENT_HANDLING.attributeNameCheck;
        }
        if (cfg.CUSTOM_ELEMENT_HANDLING && typeof cfg.CUSTOM_ELEMENT_HANDLING.allowCustomizedBuiltInElements === 'boolean') {
          CUSTOM_ELEMENT_HANDLING.allowCustomizedBuiltInElements = cfg.CUSTOM_ELEMENT_HANDLING.allowCustomizedBuiltInElements;
        }
        if (SAFE_FOR_TEMPLATES) {
          ALLOW_DATA_ATTR = false;
        }
        if (RETURN_DOM_FRAGMENT) {
          RETURN_DOM = true;
        }
        /* Parse profile info */
        if (USE_PROFILES) {
          ALLOWED_TAGS = addToSet({}, text$1);
          ALLOWED_ATTR = [];
          if (USE_PROFILES.html === true) {
            addToSet(ALLOWED_TAGS, html$1);
            addToSet(ALLOWED_ATTR, html);
          }
          if (USE_PROFILES.svg === true) {
            addToSet(ALLOWED_TAGS, svg$1);
            addToSet(ALLOWED_ATTR, svg);
            addToSet(ALLOWED_ATTR, xml);
          }
          if (USE_PROFILES.svgFilters === true) {
            addToSet(ALLOWED_TAGS, svgFilters);
            addToSet(ALLOWED_ATTR, svg);
            addToSet(ALLOWED_ATTR, xml);
          }
          if (USE_PROFILES.mathMl === true) {
            addToSet(ALLOWED_TAGS, mathMl$1);
            addToSet(ALLOWED_ATTR, mathMl);
            addToSet(ALLOWED_ATTR, xml);
          }
        }
        /* Merge configuration parameters */
        if (cfg.ADD_TAGS) {
          if (ALLOWED_TAGS === DEFAULT_ALLOWED_TAGS) {
            ALLOWED_TAGS = clone(ALLOWED_TAGS);
          }
          addToSet(ALLOWED_TAGS, cfg.ADD_TAGS, transformCaseFunc);
        }
        if (cfg.ADD_ATTR) {
          if (ALLOWED_ATTR === DEFAULT_ALLOWED_ATTR) {
            ALLOWED_ATTR = clone(ALLOWED_ATTR);
          }
          addToSet(ALLOWED_ATTR, cfg.ADD_ATTR, transformCaseFunc);
        }
        if (cfg.ADD_URI_SAFE_ATTR) {
          addToSet(URI_SAFE_ATTRIBUTES, cfg.ADD_URI_SAFE_ATTR, transformCaseFunc);
        }
        if (cfg.FORBID_CONTENTS) {
          if (FORBID_CONTENTS === DEFAULT_FORBID_CONTENTS) {
            FORBID_CONTENTS = clone(FORBID_CONTENTS);
          }
          addToSet(FORBID_CONTENTS, cfg.FORBID_CONTENTS, transformCaseFunc);
        }
        /* Add #text in case KEEP_CONTENT is set to true */
        if (KEEP_CONTENT) {
          ALLOWED_TAGS['#text'] = true;
        }
        /* Add html, head and body to ALLOWED_TAGS in case WHOLE_DOCUMENT is true */
        if (WHOLE_DOCUMENT) {
          addToSet(ALLOWED_TAGS, ['html', 'head', 'body']);
        }
        /* Add tbody to ALLOWED_TAGS in case tables are permitted, see #286, #365 */
        if (ALLOWED_TAGS.table) {
          addToSet(ALLOWED_TAGS, ['tbody']);
          delete FORBID_TAGS.tbody;
        }
        if (cfg.TRUSTED_TYPES_POLICY) {
          if (typeof cfg.TRUSTED_TYPES_POLICY.createHTML !== 'function') {
            throw typeErrorCreate('TRUSTED_TYPES_POLICY configuration option must provide a "createHTML" hook.');
          }
          if (typeof cfg.TRUSTED_TYPES_POLICY.createScriptURL !== 'function') {
            throw typeErrorCreate('TRUSTED_TYPES_POLICY configuration option must provide a "createScriptURL" hook.');
          }
          // Overwrite existing TrustedTypes policy.
          trustedTypesPolicy = cfg.TRUSTED_TYPES_POLICY;
          // Sign local variables required by `sanitize`.
          emptyHTML = trustedTypesPolicy.createHTML('');
        } else {
          // Uninitialized policy, attempt to initialize the internal dompurify policy.
          if (trustedTypesPolicy === undefined) {
            trustedTypesPolicy = _createTrustedTypesPolicy(trustedTypes, currentScript);
          }
          // If creating the internal policy succeeded sign internal variables.
          if (trustedTypesPolicy !== null && typeof emptyHTML === 'string') {
            emptyHTML = trustedTypesPolicy.createHTML('');
          }
        }
        // Prevent further manipulation of configuration.
        // Not available in IE8, Safari 5, etc.
        if (freeze) {
          freeze(cfg);
        }
        CONFIG = cfg;
      };
      /* Keep track of all possible SVG and MathML tags
       * so that we can perform the namespace checks
       * correctly. */
      const ALL_SVG_TAGS = addToSet({}, [...svg$1, ...svgFilters, ...svgDisallowed]);
      const ALL_MATHML_TAGS = addToSet({}, [...mathMl$1, ...mathMlDisallowed]);
      /**
       * @param element a DOM element whose namespace is being checked
       * @returns Return false if the element has a
       *  namespace that a spec-compliant parser would never
       *  return. Return true otherwise.
       */
      const _checkValidNamespace = function _checkValidNamespace(element) {
        let parent = getParentNode(element);
        // In JSDOM, if we're inside shadow DOM, then parentNode
        // can be null. We just simulate parent in this case.
        if (!parent || !parent.tagName) {
          parent = {
            namespaceURI: NAMESPACE,
            tagName: 'template'
          };
        }
        const tagName = stringToLowerCase(element.tagName);
        const parentTagName = stringToLowerCase(parent.tagName);
        if (!ALLOWED_NAMESPACES[element.namespaceURI]) {
          return false;
        }
        if (element.namespaceURI === SVG_NAMESPACE) {
          // The only way to switch from HTML namespace to SVG
          // is via <svg>. If it happens via any other tag, then
          // it should be killed.
          if (parent.namespaceURI === HTML_NAMESPACE) {
            return tagName === 'svg';
          }
          // The only way to switch from MathML to SVG is via`
          // svg if parent is either <annotation-xml> or MathML
          // text integration points.
          if (parent.namespaceURI === MATHML_NAMESPACE) {
            return tagName === 'svg' && (parentTagName === 'annotation-xml' || MATHML_TEXT_INTEGRATION_POINTS[parentTagName]);
          }
          // We only allow elements that are defined in SVG
          // spec. All others are disallowed in SVG namespace.
          return Boolean(ALL_SVG_TAGS[tagName]);
        }
        if (element.namespaceURI === MATHML_NAMESPACE) {
          // The only way to switch from HTML namespace to MathML
          // is via <math>. If it happens via any other tag, then
          // it should be killed.
          if (parent.namespaceURI === HTML_NAMESPACE) {
            return tagName === 'math';
          }
          // The only way to switch from SVG to MathML is via
          // <math> and HTML integration points
          if (parent.namespaceURI === SVG_NAMESPACE) {
            return tagName === 'math' && HTML_INTEGRATION_POINTS[parentTagName];
          }
          // We only allow elements that are defined in MathML
          // spec. All others are disallowed in MathML namespace.
          return Boolean(ALL_MATHML_TAGS[tagName]);
        }
        if (element.namespaceURI === HTML_NAMESPACE) {
          // The only way to switch from SVG to HTML is via
          // HTML integration points, and from MathML to HTML
          // is via MathML text integration points
          if (parent.namespaceURI === SVG_NAMESPACE && !HTML_INTEGRATION_POINTS[parentTagName]) {
            return false;
          }
          if (parent.namespaceURI === MATHML_NAMESPACE && !MATHML_TEXT_INTEGRATION_POINTS[parentTagName]) {
            return false;
          }
          // We disallow tags that are specific for MathML
          // or SVG and should never appear in HTML namespace
          return !ALL_MATHML_TAGS[tagName] && (COMMON_SVG_AND_HTML_ELEMENTS[tagName] || !ALL_SVG_TAGS[tagName]);
        }
        // For XHTML and XML documents that support custom namespaces
        if (PARSER_MEDIA_TYPE === 'application/xhtml+xml' && ALLOWED_NAMESPACES[element.namespaceURI]) {
          return true;
        }
        // The code should never reach this place (this means
        // that the element somehow got namespace that is not
        // HTML, SVG, MathML or allowed via ALLOWED_NAMESPACES).
        // Return false just in case.
        return false;
      };
      /**
       * _forceRemove
       *
       * @param node a DOM node
       */
      const _forceRemove = function _forceRemove(node) {
        arrayPush(DOMPurify.removed, {
          element: node
        });
        try {
          // eslint-disable-next-line unicorn/prefer-dom-node-remove
          getParentNode(node).removeChild(node);
        } catch (_) {
          remove(node);
        }
      };
      /**
       * _removeAttribute
       *
       * @param name an Attribute name
       * @param element a DOM node
       */
      const _removeAttribute = function _removeAttribute(name, element) {
        try {
          arrayPush(DOMPurify.removed, {
            attribute: element.getAttributeNode(name),
            from: element
          });
        } catch (_) {
          arrayPush(DOMPurify.removed, {
            attribute: null,
            from: element
          });
        }
        element.removeAttribute(name);
        // We void attribute values for unremovable "is" attributes
        if (name === 'is') {
          if (RETURN_DOM || RETURN_DOM_FRAGMENT) {
            try {
              _forceRemove(element);
            } catch (_) {}
          } else {
            try {
              element.setAttribute(name, '');
            } catch (_) {}
          }
        }
      };
      /**
       * _initDocument
       *
       * @param dirty - a string of dirty markup
       * @return a DOM, filled with the dirty markup
       */
      const _initDocument = function _initDocument(dirty) {
        /* Create a HTML document */
        let doc = null;
        let leadingWhitespace = null;
        if (FORCE_BODY) {
          dirty = '<remove></remove>' + dirty;
        } else {
          /* If FORCE_BODY isn't used, leading whitespace needs to be preserved manually */
          const matches = stringMatch(dirty, /^[\r\n\t ]+/);
          leadingWhitespace = matches && matches[0];
        }
        if (PARSER_MEDIA_TYPE === 'application/xhtml+xml' && NAMESPACE === HTML_NAMESPACE) {
          // Root of XHTML doc must contain xmlns declaration (see https://www.w3.org/TR/xhtml1/normative.html#strict)
          dirty = '<html xmlns="http://www.w3.org/1999/xhtml"><head></head><body>' + dirty + '</body></html>';
        }
        const dirtyPayload = trustedTypesPolicy ? trustedTypesPolicy.createHTML(dirty) : dirty;
        /*
         * Use the DOMParser API by default, fallback later if needs be
         * DOMParser not work for svg when has multiple root element.
         */
        if (NAMESPACE === HTML_NAMESPACE) {
          try {
            doc = new DOMParser().parseFromString(dirtyPayload, PARSER_MEDIA_TYPE);
          } catch (_) {}
        }
        /* Use createHTMLDocument in case DOMParser is not available */
        if (!doc || !doc.documentElement) {
          doc = implementation.createDocument(NAMESPACE, 'template', null);
          try {
            doc.documentElement.innerHTML = IS_EMPTY_INPUT ? emptyHTML : dirtyPayload;
          } catch (_) {
            // Syntax error if dirtyPayload is invalid xml
          }
        }
        const body = doc.body || doc.documentElement;
        if (dirty && leadingWhitespace) {
          body.insertBefore(document.createTextNode(leadingWhitespace), body.childNodes[0] || null);
        }
        /* Work on whole document or just its body */
        if (NAMESPACE === HTML_NAMESPACE) {
          return getElementsByTagName.call(doc, WHOLE_DOCUMENT ? 'html' : 'body')[0];
        }
        return WHOLE_DOCUMENT ? doc.documentElement : body;
      };
      /**
       * Creates a NodeIterator object that you can use to traverse filtered lists of nodes or elements in a document.
       *
       * @param root The root element or node to start traversing on.
       * @return The created NodeIterator
       */
      const _createNodeIterator = function _createNodeIterator(root) {
        return createNodeIterator.call(root.ownerDocument || root, root,
        // eslint-disable-next-line no-bitwise
        NodeFilter.SHOW_ELEMENT | NodeFilter.SHOW_COMMENT | NodeFilter.SHOW_TEXT | NodeFilter.SHOW_PROCESSING_INSTRUCTION | NodeFilter.SHOW_CDATA_SECTION, null);
      };
      /**
       * _isClobbered
       *
       * @param element element to check for clobbering attacks
       * @return true if clobbered, false if safe
       */
      const _isClobbered = function _isClobbered(element) {
        return element instanceof HTMLFormElement && (typeof element.nodeName !== 'string' || typeof element.textContent !== 'string' || typeof element.removeChild !== 'function' || !(element.attributes instanceof NamedNodeMap) || typeof element.removeAttribute !== 'function' || typeof element.setAttribute !== 'function' || typeof element.namespaceURI !== 'string' || typeof element.insertBefore !== 'function' || typeof element.hasChildNodes !== 'function');
      };
      /**
       * Checks whether the given object is a DOM node.
       *
       * @param value object to check whether it's a DOM node
       * @return true is object is a DOM node
       */
      const _isNode = function _isNode(value) {
        return typeof Node === 'function' && value instanceof Node;
      };
      function _executeHooks(hooks, currentNode, data) {
        arrayForEach(hooks, hook => {
          hook.call(DOMPurify, currentNode, data, CONFIG);
        });
      }
      /**
       * _sanitizeElements
       *
       * @protect nodeName
       * @protect textContent
       * @protect removeChild
       * @param currentNode to check for permission to exist
       * @return true if node was killed, false if left alive
       */
      const _sanitizeElements = function _sanitizeElements(currentNode) {
        let content = null;
        /* Execute a hook if present */
        _executeHooks(hooks.beforeSanitizeElements, currentNode, null);
        /* Check if element is clobbered or can clobber */
        if (_isClobbered(currentNode)) {
          _forceRemove(currentNode);
          return true;
        }
        /* Now let's check the element's type and name */
        const tagName = transformCaseFunc(currentNode.nodeName);
        /* Execute a hook if present */
        _executeHooks(hooks.uponSanitizeElement, currentNode, {
          tagName,
          allowedTags: ALLOWED_TAGS
        });
        /* Detect mXSS attempts abusing namespace confusion */
        if (SAFE_FOR_XML && currentNode.hasChildNodes() && !_isNode(currentNode.firstElementChild) && regExpTest(/<[/\w!]/g, currentNode.innerHTML) && regExpTest(/<[/\w!]/g, currentNode.textContent)) {
          _forceRemove(currentNode);
          return true;
        }
        /* Remove any occurrence of processing instructions */
        if (currentNode.nodeType === NODE_TYPE.progressingInstruction) {
          _forceRemove(currentNode);
          return true;
        }
        /* Remove any kind of possibly harmful comments */
        if (SAFE_FOR_XML && currentNode.nodeType === NODE_TYPE.comment && regExpTest(/<[/\w]/g, currentNode.data)) {
          _forceRemove(currentNode);
          return true;
        }
        /* Remove element if anything forbids its presence */
        if (!ALLOWED_TAGS[tagName] || FORBID_TAGS[tagName]) {
          /* Check if we have a custom element to handle */
          if (!FORBID_TAGS[tagName] && _isBasicCustomElement(tagName)) {
            if (CUSTOM_ELEMENT_HANDLING.tagNameCheck instanceof RegExp && regExpTest(CUSTOM_ELEMENT_HANDLING.tagNameCheck, tagName)) {
              return false;
            }
            if (CUSTOM_ELEMENT_HANDLING.tagNameCheck instanceof Function && CUSTOM_ELEMENT_HANDLING.tagNameCheck(tagName)) {
              return false;
            }
          }
          /* Keep content except for bad-listed elements */
          if (KEEP_CONTENT && !FORBID_CONTENTS[tagName]) {
            const parentNode = getParentNode(currentNode) || currentNode.parentNode;
            const childNodes = getChildNodes(currentNode) || currentNode.childNodes;
            if (childNodes && parentNode) {
              const childCount = childNodes.length;
              for (let i = childCount - 1; i >= 0; --i) {
                const childClone = cloneNode(childNodes[i], true);
                childClone.__removalCount = (currentNode.__removalCount || 0) + 1;
                parentNode.insertBefore(childClone, getNextSibling(currentNode));
              }
            }
          }
          _forceRemove(currentNode);
          return true;
        }
        /* Check whether element has a valid namespace */
        if (currentNode instanceof Element && !_checkValidNamespace(currentNode)) {
          _forceRemove(currentNode);
          return true;
        }
        /* Make sure that older browsers don't get fallback-tag mXSS */
        if ((tagName === 'noscript' || tagName === 'noembed' || tagName === 'noframes') && regExpTest(/<\/no(script|embed|frames)/i, currentNode.innerHTML)) {
          _forceRemove(currentNode);
          return true;
        }
        /* Sanitize element content to be template-safe */
        if (SAFE_FOR_TEMPLATES && currentNode.nodeType === NODE_TYPE.text) {
          /* Get the element's text content */
          content = currentNode.textContent;
          arrayForEach([MUSTACHE_EXPR, ERB_EXPR, TMPLIT_EXPR], expr => {
            content = stringReplace(content, expr, ' ');
          });
          if (currentNode.textContent !== content) {
            arrayPush(DOMPurify.removed, {
              element: currentNode.cloneNode()
            });
            currentNode.textContent = content;
          }
        }
        /* Execute a hook if present */
        _executeHooks(hooks.afterSanitizeElements, currentNode, null);
        return false;
      };
      /**
       * _isValidAttribute
       *
       * @param lcTag Lowercase tag name of containing element.
       * @param lcName Lowercase attribute name.
       * @param value Attribute value.
       * @return Returns true if `value` is valid, otherwise false.
       */
      // eslint-disable-next-line complexity
      const _isValidAttribute = function _isValidAttribute(lcTag, lcName, value) {
        /* Make sure attribute cannot clobber */
        if (SANITIZE_DOM && (lcName === 'id' || lcName === 'name') && (value in document || value in formElement)) {
          return false;
        }
        /* Allow valid data-* attributes: At least one character after "-"
            (https://html.spec.whatwg.org/multipage/dom.html#embedding-custom-non-visible-data-with-the-data-*-attributes)
            XML-compatible (https://html.spec.whatwg.org/multipage/infrastructure.html#xml-compatible and http://www.w3.org/TR/xml/#d0e804)
            We don't need to check the value; it's always URI safe. */
        if (ALLOW_DATA_ATTR && !FORBID_ATTR[lcName] && regExpTest(DATA_ATTR, lcName)) ; else if (ALLOW_ARIA_ATTR && regExpTest(ARIA_ATTR, lcName)) ; else if (!ALLOWED_ATTR[lcName] || FORBID_ATTR[lcName]) {
          if (
          // First condition does a very basic check if a) it's basically a valid custom element tagname AND
          // b) if the tagName passes whatever the user has configured for CUSTOM_ELEMENT_HANDLING.tagNameCheck
          // and c) if the attribute name passes whatever the user has configured for CUSTOM_ELEMENT_HANDLING.attributeNameCheck
          _isBasicCustomElement(lcTag) && (CUSTOM_ELEMENT_HANDLING.tagNameCheck instanceof RegExp && regExpTest(CUSTOM_ELEMENT_HANDLING.tagNameCheck, lcTag) || CUSTOM_ELEMENT_HANDLING.tagNameCheck instanceof Function && CUSTOM_ELEMENT_HANDLING.tagNameCheck(lcTag)) && (CUSTOM_ELEMENT_HANDLING.attributeNameCheck instanceof RegExp && regExpTest(CUSTOM_ELEMENT_HANDLING.attributeNameCheck, lcName) || CUSTOM_ELEMENT_HANDLING.attributeNameCheck instanceof Function && CUSTOM_ELEMENT_HANDLING.attributeNameCheck(lcName)) ||
          // Alternative, second condition checks if it's an `is`-attribute, AND
          // the value passes whatever the user has configured for CUSTOM_ELEMENT_HANDLING.tagNameCheck
          lcName === 'is' && CUSTOM_ELEMENT_HANDLING.allowCustomizedBuiltInElements && (CUSTOM_ELEMENT_HANDLING.tagNameCheck instanceof RegExp && regExpTest(CUSTOM_ELEMENT_HANDLING.tagNameCheck, value) || CUSTOM_ELEMENT_HANDLING.tagNameCheck instanceof Function && CUSTOM_ELEMENT_HANDLING.tagNameCheck(value))) ; else {
            return false;
          }
          /* Check value is safe. First, is attr inert? If so, is safe */
        } else if (URI_SAFE_ATTRIBUTES[lcName]) ; else if (regExpTest(IS_ALLOWED_URI$1, stringReplace(value, ATTR_WHITESPACE, ''))) ; else if ((lcName === 'src' || lcName === 'xlink:href' || lcName === 'href') && lcTag !== 'script' && stringIndexOf(value, 'data:') === 0 && DATA_URI_TAGS[lcTag]) ; else if (ALLOW_UNKNOWN_PROTOCOLS && !regExpTest(IS_SCRIPT_OR_DATA, stringReplace(value, ATTR_WHITESPACE, ''))) ; else if (value) {
          return false;
        } else ;
        return true;
      };
      /**
       * _isBasicCustomElement
       * checks if at least one dash is included in tagName, and it's not the first char
       * for more sophisticated checking see https://github.com/sindresorhus/validate-element-name
       *
       * @param tagName name of the tag of the node to sanitize
       * @returns Returns true if the tag name meets the basic criteria for a custom element, otherwise false.
       */
      const _isBasicCustomElement = function _isBasicCustomElement(tagName) {
        return tagName !== 'annotation-xml' && stringMatch(tagName, CUSTOM_ELEMENT);
      };
      /**
       * _sanitizeAttributes
       *
       * @protect attributes
       * @protect nodeName
       * @protect removeAttribute
       * @protect setAttribute
       *
       * @param currentNode to sanitize
       */
      const _sanitizeAttributes = function _sanitizeAttributes(currentNode) {
        /* Execute a hook if present */
        _executeHooks(hooks.beforeSanitizeAttributes, currentNode, null);
        const {
          attributes
        } = currentNode;
        /* Check if we have attributes; if not we might have a text node */
        if (!attributes || _isClobbered(currentNode)) {
          return;
        }
        const hookEvent = {
          attrName: '',
          attrValue: '',
          keepAttr: true,
          allowedAttributes: ALLOWED_ATTR,
          forceKeepAttr: undefined
        };
        let l = attributes.length;
        /* Go backwards over all attributes; safely remove bad ones */
        while (l--) {
          const attr = attributes[l];
          const {
            name,
            namespaceURI,
            value: attrValue
          } = attr;
          const lcName = transformCaseFunc(name);
          const initValue = attrValue;
          let value = name === 'value' ? initValue : stringTrim(initValue);
          /* Execute a hook if present */
          hookEvent.attrName = lcName;
          hookEvent.attrValue = value;
          hookEvent.keepAttr = true;
          hookEvent.forceKeepAttr = undefined; // Allows developers to see this is a property they can set
          _executeHooks(hooks.uponSanitizeAttribute, currentNode, hookEvent);
          value = hookEvent.attrValue;
          /* Full DOM Clobbering protection via namespace isolation,
           * Prefix id and name attributes with `user-content-`
           */
          if (SANITIZE_NAMED_PROPS && (lcName === 'id' || lcName === 'name')) {
            // Remove the attribute with this value
            _removeAttribute(name, currentNode);
            // Prefix the value and later re-create the attribute with the sanitized value
            value = SANITIZE_NAMED_PROPS_PREFIX + value;
          }
          /* Work around a security issue with comments inside attributes */
          if (SAFE_FOR_XML && regExpTest(/((--!?|])>)|<\/(style|title)/i, value)) {
            _removeAttribute(name, currentNode);
            continue;
          }
          /* Did the hooks approve of the attribute? */
          if (hookEvent.forceKeepAttr) {
            continue;
          }
          /* Did the hooks approve of the attribute? */
          if (!hookEvent.keepAttr) {
            _removeAttribute(name, currentNode);
            continue;
          }
          /* Work around a security issue in jQuery 3.0 */
          if (!ALLOW_SELF_CLOSE_IN_ATTR && regExpTest(/\/>/i, value)) {
            _removeAttribute(name, currentNode);
            continue;
          }
          /* Sanitize attribute content to be template-safe */
          if (SAFE_FOR_TEMPLATES) {
            arrayForEach([MUSTACHE_EXPR, ERB_EXPR, TMPLIT_EXPR], expr => {
              value = stringReplace(value, expr, ' ');
            });
          }
          /* Is `value` valid for this attribute? */
          const lcTag = transformCaseFunc(currentNode.nodeName);
          if (!_isValidAttribute(lcTag, lcName, value)) {
            _removeAttribute(name, currentNode);
            continue;
          }
          /* Handle attributes that require Trusted Types */
          if (trustedTypesPolicy && typeof trustedTypes === 'object' && typeof trustedTypes.getAttributeType === 'function') {
            if (namespaceURI) ; else {
              switch (trustedTypes.getAttributeType(lcTag, lcName)) {
                case 'TrustedHTML':
                  {
                    value = trustedTypesPolicy.createHTML(value);
                    break;
                  }
                case 'TrustedScriptURL':
                  {
                    value = trustedTypesPolicy.createScriptURL(value);
                    break;
                  }
              }
            }
          }
          /* Handle invalid data-* attribute set by try-catching it */
          if (value !== initValue) {
            try {
              if (namespaceURI) {
                currentNode.setAttributeNS(namespaceURI, name, value);
              } else {
                /* Fallback to setAttribute() for browser-unrecognized namespaces e.g. "x-schema". */
                currentNode.setAttribute(name, value);
              }
              if (_isClobbered(currentNode)) {
                _forceRemove(currentNode);
              } else {
                arrayPop(DOMPurify.removed);
              }
            } catch (_) {
              _removeAttribute(name, currentNode);
            }
          }
        }
        /* Execute a hook if present */
        _executeHooks(hooks.afterSanitizeAttributes, currentNode, null);
      };
      /**
       * _sanitizeShadowDOM
       *
       * @param fragment to iterate over recursively
       */
      const _sanitizeShadowDOM = function _sanitizeShadowDOM(fragment) {
        let shadowNode = null;
        const shadowIterator = _createNodeIterator(fragment);
        /* Execute a hook if present */
        _executeHooks(hooks.beforeSanitizeShadowDOM, fragment, null);
        while (shadowNode = shadowIterator.nextNode()) {
          /* Execute a hook if present */
          _executeHooks(hooks.uponSanitizeShadowNode, shadowNode, null);
          /* Sanitize tags and elements */
          _sanitizeElements(shadowNode);
          /* Check attributes next */
          _sanitizeAttributes(shadowNode);
          /* Deep shadow DOM detected */
          if (shadowNode.content instanceof DocumentFragment) {
            _sanitizeShadowDOM(shadowNode.content);
          }
        }
        /* Execute a hook if present */
        _executeHooks(hooks.afterSanitizeShadowDOM, fragment, null);
      };
      // eslint-disable-next-line complexity
      DOMPurify.sanitize = function (dirty) {
        let cfg = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
        let body = null;
        let importedNode = null;
        let currentNode = null;
        let returnNode = null;
        /* Make sure we have a string to sanitize.
          DO NOT return early, as this will return the wrong type if
          the user has requested a DOM object rather than a string */
        IS_EMPTY_INPUT = !dirty;
        if (IS_EMPTY_INPUT) {
          dirty = '<!-->';
        }
        /* Stringify, in case dirty is an object */
        if (typeof dirty !== 'string' && !_isNode(dirty)) {
          if (typeof dirty.toString === 'function') {
            dirty = dirty.toString();
            if (typeof dirty !== 'string') {
              throw typeErrorCreate('dirty is not a string, aborting');
            }
          } else {
            throw typeErrorCreate('toString is not a function');
          }
        }
        /* Return dirty HTML if DOMPurify cannot run */
        if (!DOMPurify.isSupported) {
          return dirty;
        }
        /* Assign config vars */
        if (!SET_CONFIG) {
          _parseConfig(cfg);
        }
        /* Clean up removed elements */
        DOMPurify.removed = [];
        /* Check if dirty is correctly typed for IN_PLACE */
        if (typeof dirty === 'string') {
          IN_PLACE = false;
        }
        if (IN_PLACE) {
          /* Do some early pre-sanitization to avoid unsafe root nodes */
          if (dirty.nodeName) {
            const tagName = transformCaseFunc(dirty.nodeName);
            if (!ALLOWED_TAGS[tagName] || FORBID_TAGS[tagName]) {
              throw typeErrorCreate('root node is forbidden and cannot be sanitized in-place');
            }
          }
        } else if (dirty instanceof Node) {
          /* If dirty is a DOM element, append to an empty document to avoid
             elements being stripped by the parser */
          body = _initDocument('<!---->');
          importedNode = body.ownerDocument.importNode(dirty, true);
          if (importedNode.nodeType === NODE_TYPE.element && importedNode.nodeName === 'BODY') {
            /* Node is already a body, use as is */
            body = importedNode;
          } else if (importedNode.nodeName === 'HTML') {
            body = importedNode;
          } else {
            // eslint-disable-next-line unicorn/prefer-dom-node-append
            body.appendChild(importedNode);
          }
        } else {
          /* Exit directly if we have nothing to do */
          if (!RETURN_DOM && !SAFE_FOR_TEMPLATES && !WHOLE_DOCUMENT &&
          // eslint-disable-next-line unicorn/prefer-includes
          dirty.indexOf('<') === -1) {
            return trustedTypesPolicy && RETURN_TRUSTED_TYPE ? trustedTypesPolicy.createHTML(dirty) : dirty;
          }
          /* Initialize the document to work on */
          body = _initDocument(dirty);
          /* Check we have a DOM node from the data */
          if (!body) {
            return RETURN_DOM ? null : RETURN_TRUSTED_TYPE ? emptyHTML : '';
          }
        }
        /* Remove first element node (ours) if FORCE_BODY is set */
        if (body && FORCE_BODY) {
          _forceRemove(body.firstChild);
        }
        /* Get node iterator */
        const nodeIterator = _createNodeIterator(IN_PLACE ? dirty : body);
        /* Now start iterating over the created document */
        while (currentNode = nodeIterator.nextNode()) {
          /* Sanitize tags and elements */
          _sanitizeElements(currentNode);
          /* Check attributes next */
          _sanitizeAttributes(currentNode);
          /* Shadow DOM detected, sanitize it */
          if (currentNode.content instanceof DocumentFragment) {
            _sanitizeShadowDOM(currentNode.content);
          }
        }
        /* If we sanitized `dirty` in-place, return it. */
        if (IN_PLACE) {
          return dirty;
        }
        /* Return sanitized string or DOM */
        if (RETURN_DOM) {
          if (RETURN_DOM_FRAGMENT) {
            returnNode = createDocumentFragment.call(body.ownerDocument);
            while (body.firstChild) {
              // eslint-disable-next-line unicorn/prefer-dom-node-append
              returnNode.appendChild(body.firstChild);
            }
          } else {
            returnNode = body;
          }
          if (ALLOWED_ATTR.shadowroot || ALLOWED_ATTR.shadowrootmode) {
            /*
              AdoptNode() is not used because internal state is not reset
              (e.g. the past names map of a HTMLFormElement), this is safe
              in theory but we would rather not risk another attack vector.
              The state that is cloned by importNode() is explicitly defined
              by the specs.
            */
            returnNode = importNode.call(originalDocument, returnNode, true);
          }
          return returnNode;
        }
        let serializedHTML = WHOLE_DOCUMENT ? body.outerHTML : body.innerHTML;
        /* Serialize doctype if allowed */
        if (WHOLE_DOCUMENT && ALLOWED_TAGS['!doctype'] && body.ownerDocument && body.ownerDocument.doctype && body.ownerDocument.doctype.name && regExpTest(DOCTYPE_NAME, body.ownerDocument.doctype.name)) {
          serializedHTML = '<!DOCTYPE ' + body.ownerDocument.doctype.name + '>\n' + serializedHTML;
        }
        /* Sanitize final string template-safe */
        if (SAFE_FOR_TEMPLATES) {
          arrayForEach([MUSTACHE_EXPR, ERB_EXPR, TMPLIT_EXPR], expr => {
            serializedHTML = stringReplace(serializedHTML, expr, ' ');
          });
        }
        return trustedTypesPolicy && RETURN_TRUSTED_TYPE ? trustedTypesPolicy.createHTML(serializedHTML) : serializedHTML;
      };
      DOMPurify.setConfig = function () {
        let cfg = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {};
        _parseConfig(cfg);
        SET_CONFIG = true;
      };
      DOMPurify.clearConfig = function () {
        CONFIG = null;
        SET_CONFIG = false;
      };
      DOMPurify.isValidAttribute = function (tag, attr, value) {
        /* Initialize shared config vars if necessary. */
        if (!CONFIG) {
          _parseConfig({});
        }
        const lcTag = transformCaseFunc(tag);
        const lcName = transformCaseFunc(attr);
        return _isValidAttribute(lcTag, lcName, value);
      };
      DOMPurify.addHook = function (entryPoint, hookFunction) {
        if (typeof hookFunction !== 'function') {
          return;
        }
        arrayPush(hooks[entryPoint], hookFunction);
      };
      DOMPurify.removeHook = function (entryPoint, hookFunction) {
        if (hookFunction !== undefined) {
          const index = arrayLastIndexOf(hooks[entryPoint], hookFunction);
          return index === -1 ? undefined : arraySplice(hooks[entryPoint], index, 1)[0];
        }
        return arrayPop(hooks[entryPoint]);
      };
      DOMPurify.removeHooks = function (entryPoint) {
        hooks[entryPoint] = [];
      };
      DOMPurify.removeAllHooks = function () {
        hooks = _createHooksMap();
      };
      return DOMPurify;
    }
    var purify = createDOMPurify();

    const sanitizeHtmlString = (html) => purify().sanitize(html);

    var global$6 = tinymce.util.Tools.resolve('tinymce.util.I18n');

    // Icons that need to be transformed in RTL
    const rtlTransform = {
        'indent': true,
        'outdent': true,
        'table-insert-column-after': true,
        'table-insert-column-before': true,
        'paste-column-after': true,
        'paste-column-before': true,
        'unordered-list': true,
        'list-bull-circle': true,
        'list-bull-disc': true,
        'list-bull-default': true,
        'list-bull-square': true
    };
    const defaultIconName = 'temporary-placeholder';
    const defaultIcon = (icons) => () => get$h(icons, defaultIconName).getOr('!not found!');
    const getIconName = (name, icons) => {
        const lcName = name.toLowerCase();
        // If in rtl mode then try to see if we have a rtl icon to use instead
        if (global$6.isRtl()) {
            const rtlName = ensureTrailing(lcName, '-rtl');
            return has$2(icons, rtlName) ? rtlName : lcName;
        }
        else {
            return lcName;
        }
    };
    const lookupIcon = (name, icons) => get$h(icons, getIconName(name, icons));
    const get = (name, iconProvider) => {
        const icons = iconProvider();
        return lookupIcon(name, icons).getOrThunk(defaultIcon(icons));
    };
    const getOr = (name, iconProvider, fallbackIcon) => {
        const icons = iconProvider();
        return lookupIcon(name, icons).or(fallbackIcon).getOrThunk(defaultIcon(icons));
    };
    const needsRtlTransform = (iconName) => global$6.isRtl() ? has$2(rtlTransform, iconName) : false;
    const addFocusableBehaviour = () => config('add-focusable', [
        runOnAttached((comp) => {
            // set focusable=false on SVGs to prevent focusing the toolbar when tabbing into the editor
            child(comp.element, 'svg').each((svg) => set$9(svg, 'focusable', 'false'));
        })
    ]);
    const renderIcon$3 = (spec, iconName, icons, fallbackIcon) => {
        // If RTL, add the flip icon class if the icon doesn't have a `-rtl` icon available.
        const rtlIconClasses = needsRtlTransform(iconName) ? ['tox-icon--flip'] : [];
        const iconHtml = get$h(icons, getIconName(iconName, icons)).or(fallbackIcon).getOrThunk(defaultIcon(icons));
        return {
            dom: {
                tag: spec.tag,
                attributes: spec.attributes ?? {},
                classes: spec.classes.concat(rtlIconClasses),
                innerHtml: iconHtml
            },
            behaviours: derive$1([
                ...spec.behaviours ?? [],
                addFocusableBehaviour()
            ]),
            eventOrder: spec.eventOrder ?? {}
        };
    };
    const render$4 = (iconName, spec, iconProvider, fallbackIcon = Optional.none()) => renderIcon$3(spec, iconName, iconProvider(), fallbackIcon);
    const renderFirst = (iconNames, spec, iconProvider) => {
        const icons = iconProvider();
        const iconName = find$5(iconNames, (name) => has$2(icons, getIconName(name, icons)));
        return renderIcon$3(spec, iconName.getOr(defaultIconName), icons, Optional.none());
    };

    const notificationIconMap = {
        success: 'checkmark',
        error: 'warning',
        err: 'error',
        warning: 'warning',
        warn: 'warning',
        info: 'info'
    };
    const factory$4 = (detail) => {
        // For using the alert banner as a standalone banner
        const notificationTextId = generate$6('notification-text');
        const memBannerText = record({
            dom: fromHtml(`<p id=${notificationTextId}>${sanitizeHtmlString(detail.backstageProvider.translate(detail.text))}</p>`),
            behaviours: derive$1([
                Replacing.config({})
            ])
        });
        const renderPercentBar = (percent) => ({
            dom: {
                tag: 'div',
                classes: ['tox-bar'],
                styles: {
                    width: `${percent}%`
                }
            }
        });
        const renderPercentText = (percent) => ({
            dom: {
                tag: 'div',
                classes: ['tox-text'],
                innerHtml: `${percent}%`
            }
        });
        const memBannerProgress = record({
            dom: {
                tag: 'div',
                classes: detail.progress ? ['tox-progress-bar', 'tox-progress-indicator'] : ['tox-progress-bar']
            },
            components: [
                {
                    dom: {
                        tag: 'div',
                        classes: ['tox-bar-container']
                    },
                    components: [
                        renderPercentBar(0)
                    ]
                },
                renderPercentText(0)
            ],
            behaviours: derive$1([
                Replacing.config({})
            ])
        });
        const updateProgress = (comp, percent) => {
            if (comp.getSystem().isConnected()) {
                memBannerProgress.getOpt(comp).each((progress) => {
                    Replacing.set(progress, [
                        {
                            dom: {
                                tag: 'div',
                                classes: ['tox-bar-container']
                            },
                            components: [
                                renderPercentBar(percent)
                            ]
                        },
                        renderPercentText(percent)
                    ]);
                });
            }
        };
        const updateText = (comp, text) => {
            if (comp.getSystem().isConnected()) {
                const banner = memBannerText.get(comp);
                Replacing.set(banner, [
                    text$2(text)
                ]);
            }
        };
        const apis = {
            updateProgress,
            updateText
        };
        const iconChoices = flatten([
            detail.icon.toArray(),
            [detail.level],
            Optional.from(notificationIconMap[detail.level]).toArray()
        ]);
        const memButton = record(Button.sketch({
            dom: {
                tag: 'button',
                classes: ['tox-notification__dismiss', 'tox-button', 'tox-button--naked', 'tox-button--icon'],
                attributes: {
                    'aria-label': detail.backstageProvider.translate('Close')
                }
            },
            components: [
                render$4('close', {
                    tag: 'span',
                    classes: ['tox-icon'],
                }, detail.iconProvider)
            ],
            buttonBehaviours: derive$1([
                Tabstopping.config({}),
                Tooltipping.config({
                    ...detail.backstageProvider.tooltips.getConfig({
                        tooltipText: detail.backstageProvider.translate('Close')
                    })
                })
            ]),
            action: (comp) => {
                detail.onAction(comp);
            }
        }));
        const notificationIconSpec = renderFirst(iconChoices, { tag: 'div', classes: ['tox-notification__icon'] }, detail.iconProvider);
        const notificationBodySpec = {
            dom: {
                tag: 'div',
                classes: ['tox-notification__body']
            },
            components: [
                memBannerText.asSpec()
            ],
            behaviours: derive$1([
                Replacing.config({})
            ])
        };
        const components = [notificationIconSpec, notificationBodySpec];
        return {
            uid: detail.uid,
            dom: {
                tag: 'div',
                attributes: {
                    'role': 'alert',
                    'aria-labelledby': notificationTextId
                },
                classes: ['tox-notification', 'tox-notification--in', `tox-notification--${detail.level}`],
            },
            behaviours: derive$1([
                Tabstopping.config({}),
                Focusing.config({}),
                Keying.config({
                    mode: 'special',
                    onEscape: (comp) => {
                        detail.onAction(comp);
                        return Optional.some(true);
                    }
                })
            ]),
            components: components
                .concat(detail.progress ? [memBannerProgress.asSpec()] : [])
                .concat([memButton.asSpec()]),
            apis
        };
    };
    const Notification = single({
        name: 'Notification',
        factory: factory$4,
        configFields: [
            defaultedStringEnum('level', 'info', ['success', 'error', 'warning', 'warn', 'info']),
            required$1('progress'),
            option$3('icon'),
            required$1('onAction'),
            required$1('text'),
            required$1('iconProvider'),
            required$1('backstageProvider'),
        ],
        apis: {
            updateProgress: (apis, comp, percent) => {
                apis.updateProgress(comp, percent);
            },
            updateText: (apis, comp, text) => {
                apis.updateText(comp, text);
            }
        }
    });

    var NotificationManagerImpl = (editor, extras, uiMothership, notificationRegion) => {
        const sharedBackstage = extras.backstage.shared;
        const getBoundsContainer = () => SugarElement.fromDom(editor.queryCommandValue('ToggleView') === '' ? editor.getContentAreaContainer() : editor.getContainer());
        const getBounds = () => {
            const contentArea = box$1(getBoundsContainer());
            return Optional.some(contentArea);
        };
        const clampComponentsToBounds = (components) => {
            getBounds().each((bounds) => {
                each$1(components, (comp) => {
                    remove$6(comp.element, 'width');
                    if (get$c(comp.element) > bounds.width) {
                        set$7(comp.element, 'width', bounds.width + 'px');
                    }
                });
            });
        };
        const open = (settings, closeCallback, isEditorOrUIFocused) => {
            const close = () => {
                const removeNotificationAndReposition = (region) => {
                    Replacing.remove(region, notification);
                    reposition();
                };
                const manageRegionVisibility = (region, editorOrUiFocused) => {
                    if (children(region.element).length === 0) {
                        handleEmptyRegion(region, editorOrUiFocused);
                    }
                    else {
                        handleRegionWithChildren(region, editorOrUiFocused);
                    }
                };
                const handleEmptyRegion = (region, editorOrUIFocused) => {
                    InlineView.hide(region);
                    notificationRegion.clear();
                    if (editorOrUIFocused) {
                        editor.focus();
                    }
                };
                const handleRegionWithChildren = (region, editorOrUIFocused) => {
                    if (editorOrUIFocused) {
                        Keying.focusIn(region);
                    }
                };
                notificationRegion.on((region) => {
                    closeCallback();
                    const editorOrUIFocused = isEditorOrUIFocused();
                    removeNotificationAndReposition(region);
                    manageRegionVisibility(region, editorOrUIFocused);
                });
            };
            const shouldApplyDocking = () => !isStickyToolbar(editor) || !sharedBackstage.header.isPositionedAtTop();
            const notification = build$1(Notification.sketch({
                text: settings.text,
                level: contains$2(['success', 'error', 'warning', 'warn', 'info'], settings.type) ? settings.type : undefined,
                progress: settings.progressBar === true,
                icon: settings.icon,
                onAction: close,
                iconProvider: sharedBackstage.providers.icons,
                backstageProvider: sharedBackstage.providers,
            }));
            if (!notificationRegion.isSet()) {
                const notificationWrapper = build$1(InlineView.sketch({
                    dom: {
                        tag: 'div',
                        classes: ['tox-notifications-container'],
                        attributes: {
                            'aria-label': 'Notifications',
                            'role': 'region'
                        }
                    },
                    lazySink: sharedBackstage.getSink,
                    fireDismissalEventInstead: {},
                    ...sharedBackstage.header.isPositionedAtTop() ? {} : { fireRepositionEventInstead: {} },
                    inlineBehaviours: derive$1([
                        Keying.config({
                            mode: 'cyclic',
                            selector: '.tox-notification, .tox-notification a, .tox-notification button',
                        }),
                        Replacing.config({}),
                        ...(shouldApplyDocking()
                            ? [
                                Docking.config({
                                    contextual: {
                                        lazyContext: () => Optional.some(box$1(getBoundsContainer())),
                                        fadeInClass: 'tox-notification-container-dock-fadein',
                                        fadeOutClass: 'tox-notification-container-dock-fadeout',
                                        transitionClass: 'tox-notification-container-dock-transition'
                                    },
                                    modes: ['top'],
                                    lazyViewport: (comp) => {
                                        const optScrollingContext = detectWhenSplitUiMode(editor, comp.element);
                                        return optScrollingContext
                                            .map((sc) => {
                                            const combinedBounds = getBoundsFrom(sc);
                                            return {
                                                bounds: combinedBounds,
                                                optScrollEnv: Optional.some({
                                                    currentScrollTop: sc.element.dom.scrollTop,
                                                    scrollElmTop: absolute$3(sc.element).top
                                                })
                                            };
                                        }).getOrThunk(() => ({
                                            bounds: win(),
                                            optScrollEnv: Optional.none()
                                        }));
                                    }
                                })
                            ] : [])
                    ])
                }));
                const notificationSpec = premade(notification);
                const anchorOverrides = {
                    maxHeightFunction: expandable$1()
                };
                const anchor = {
                    ...sharedBackstage.anchors.banner(),
                    overrides: anchorOverrides
                };
                notificationRegion.set(notificationWrapper);
                uiMothership.add(notificationWrapper);
                InlineView.showWithinBounds(notificationWrapper, notificationSpec, { anchor }, getBounds);
            }
            else {
                const notificationSpec = premade(notification);
                notificationRegion.on((notificationWrapper) => {
                    Replacing.append(notificationWrapper, notificationSpec);
                    InlineView.reposition(notificationWrapper);
                    if (notification.hasConfigured(Docking)) {
                        Docking.refresh(notificationWrapper);
                    }
                    clampComponentsToBounds(notificationWrapper.components());
                });
            }
            if (isNumber(settings.timeout) && settings.timeout > 0) {
                global$a.setEditorTimeout(editor, () => {
                    close();
                }, settings.timeout);
            }
            const reposition = () => {
                notificationRegion.on((region) => {
                    InlineView.reposition(region);
                    if (region.hasConfigured(Docking)) {
                        Docking.refresh(region);
                    }
                    clampComponentsToBounds(region.components());
                });
            };
            const thisNotification = {
                close,
                reposition,
                text: (nuText) => {
                    // check if component is still mounted
                    Notification.updateText(notification, nuText);
                },
                settings,
                getEl: () => notification.element.dom,
                progressBar: {
                    value: (percent) => {
                        Notification.updateProgress(notification, percent);
                    }
                }
            };
            return thisNotification;
        };
        const close = (notification) => {
            notification.close();
        };
        const getArgs = (notification) => {
            return notification.settings;
        };
        return {
            open,
            close,
            getArgs
        };
    };

    const setup$c = (api, editor) => {
        const redirectKeyToItem = (item, e) => {
            emitWith(item, keydown(), { raw: e });
        };
        const getItem = () => api.getMenu().bind(Highlighting.getHighlighted);
        editor.on('keydown', (e) => {
            const keyCode = e.which;
            // If the autocompleter isn't activated then do nothing
            if (!api.isActive()) {
                return;
            }
            if (api.isMenuOpen()) {
                // Pressing <enter> executes any item currently selected, or does nothing
                if (keyCode === 13) {
                    getItem().each(emitExecute);
                    e.preventDefault();
                    // Pressing <down> either highlights the first option, or moves down the menu
                }
                else if (keyCode === 40) {
                    getItem().fold(
                    // No current item, so highlight the first one
                    () => {
                        api.getMenu().each(Highlighting.highlightFirst);
                    }, 
                    // There is a current item, so move down in the menu
                    (item) => {
                        redirectKeyToItem(item, e);
                    });
                    e.preventDefault();
                    e.stopImmediatePropagation();
                    // Pressing <up>, <left>, <right> gets redirected to the selected item
                }
                else if (keyCode === 37 || keyCode === 38 || keyCode === 39) {
                    getItem().each((item) => {
                        redirectKeyToItem(item, e);
                        e.preventDefault();
                        e.stopImmediatePropagation();
                    });
                }
            }
            else {
                // Pressing <enter>, <down> or <up> closes the autocompleter when it's active but the menu isn't open
                if (keyCode === 13 || keyCode === 38 || keyCode === 40) {
                    api.cancelIfNecessary();
                }
            }
        });
        editor.on('NodeChange', () => {
            // Close if active, not in the middle of an onAction callback and we're no longer inside the autocompleter span
            if (api.isActive() && !api.isProcessingAction() && !editor.queryCommandState('mceAutoCompleterInRange')) {
                api.cancelIfNecessary();
            }
        });
    };
    const AutocompleterEditorEvents = {
        setup: setup$c
    };

    var ItemResponse;
    (function (ItemResponse) {
        ItemResponse[ItemResponse["CLOSE_ON_EXECUTE"] = 0] = "CLOSE_ON_EXECUTE";
        ItemResponse[ItemResponse["BUBBLE_TO_SANDBOX"] = 1] = "BUBBLE_TO_SANDBOX";
    })(ItemResponse || (ItemResponse = {}));
    var ItemResponse$1 = ItemResponse;

    const navClass = 'tox-menu-nav__js';
    const selectableClass = 'tox-collection__item';
    const colorClass = 'tox-swatch';
    const presetClasses = {
        normal: navClass,
        color: colorClass
    };
    const tickedClass = 'tox-collection__item--enabled';
    const groupHeadingClass = 'tox-collection__group-heading';
    const iconClass = 'tox-collection__item-icon';
    const imageClass = 'tox-collection__item-image';
    const imageSelectorClasll = 'tox-collection__item-image-selector';
    const textClass = 'tox-collection__item-label';
    const accessoryClass = 'tox-collection__item-accessory';
    const caretClass = 'tox-collection__item-caret';
    const checkmarkClass = 'tox-collection__item-checkmark';
    const activeClass = 'tox-collection__item--active';
    const containerClass = 'tox-collection__item-container';
    const containerColumnClass = 'tox-collection__item-container--column';
    const containerRowClass = 'tox-collection__item-container--row';
    const containerAlignRightClass = 'tox-collection__item-container--align-right';
    const containerAlignLeftClass = 'tox-collection__item-container--align-left';
    const containerValignTopClass = 'tox-collection__item-container--valign-top';
    const containerValignMiddleClass = 'tox-collection__item-container--valign-middle';
    const containerValignBottomClass = 'tox-collection__item-container--valign-bottom';
    const classForPreset = (presets) => get$h(presetClasses, presets).getOr(navClass);

    const forMenu = (presets) => {
        if (presets === 'color') {
            return 'tox-swatches';
        }
        else {
            return 'tox-menu';
        }
    };
    const classes = (presets) => ({
        backgroundMenu: 'tox-background-menu',
        selectedMenu: 'tox-selected-menu',
        selectedItem: 'tox-collection__item--active',
        hasIcons: 'tox-menu--has-icons',
        menu: forMenu(presets),
        tieredMenu: 'tox-tiered-menu'
    });

    const markers = (presets) => {
        const menuClasses = classes(presets);
        return {
            backgroundMenu: menuClasses.backgroundMenu,
            selectedMenu: menuClasses.selectedMenu,
            menu: menuClasses.menu,
            selectedItem: menuClasses.selectedItem,
            item: classForPreset(presets)
        };
    };
    const dom = (hasIcons, columns, presets) => {
        const menuClasses = classes(presets);
        return {
            tag: 'div',
            classes: flatten([
                [menuClasses.menu, `tox-menu-${columns}-column`],
                hasIcons ? [menuClasses.hasIcons] : []
            ])
        };
    };
    const components = [
        Menu.parts.items({})
    ];
    // NOTE: Up to here.
    const part = (hasIcons, columns, presets) => {
        const menuClasses = classes(presets);
        const d = {
            tag: 'div',
            classes: flatten([
                [menuClasses.tieredMenu]
            ])
        };
        return {
            dom: d,
            markers: markers(presets)
        };
    };

    // This event is triggered by a menu item from a dropdown when it wants the
    // dropdown to refetch its contents based on a search string.
    const refetchTriggerEvent = generate$6('refetch-trigger-event');
    // This event is triggerd by a menu item from a dropdown, when it wants to
    // redispatch that event to the currently active item of that dropdown menu. It will
    // be used in situations where the event should be firing on the item with fake focus,
    // but instead it is firing on the item with real focus (e.g of real focus:
    // menu search field)
    const redirectMenuItemInteractionEvent = generate$6('redirect-menu-item-interaction');

    // This is not stored in ItemClasses, because the searcher is not actually
    // contained within items. It isn't part of their navigation, and it
    // isn't maintained by menus. It is just part of the first menu, but
    // not its items.
    const menuSearcherClass = 'tox-menu__searcher';
    // Ideally, we'd be using mementos to find it again, but we'd need to pass
    // that memento onto the dropdown, which isn't going to have it. Especially,
    // because the dropdown isn't responsible for putting this searcher component
    // into the menu, NestedMenus is.
    const findWithinSandbox = (sandboxComp) => {
        return descendant(sandboxComp.element, `.${menuSearcherClass}`).bind((inputElem) => sandboxComp.getSystem().getByDom(inputElem).toOptional());
    };
    // There is nothing sandbox-specific about this code. It just needs to be
    // a container that wraps the search field.
    const findWithinMenu = findWithinSandbox;
    const restoreState = (inputComp, searcherState) => {
        Representing.setValue(inputComp, searcherState.fetchPattern);
        inputComp.element.dom.selectionStart = searcherState.selectionStart;
        inputComp.element.dom.selectionEnd = searcherState.selectionEnd;
    };
    const saveState = (inputComp) => {
        const fetchPattern = Representing.getValue(inputComp);
        const selectionStart = inputComp.element.dom.selectionStart;
        const selectionEnd = inputComp.element.dom.selectionEnd;
        return {
            fetchPattern,
            selectionStart,
            selectionEnd
        };
    };
    // Make sure there is ARIA communicating the currently active item in the results.
    const setActiveDescendant = (inputComp, active) => {
        getOpt(active.element, 'id')
            .each((id) => set$9(inputComp.element, 'aria-activedescendant', id));
    };
    const renderMenuSearcher = (spec) => {
        const handleByBrowser = (comp, se) => {
            // We "cut" this event, so that the browser still handles it, but it is not processed
            // by any of the above alloy components. We could also do this by stopping propagation,
            // but not preventing default, but it's probably good to allow some overarching thing
            // in the DOM (outside of alloy) to stop it if they want to.
            se.cut();
            // Returning a Some here (regardless of boolean value) is going to call `stop` on the
            // simulated event, which is going to call: preventDefault and stopPropagation. We want
            // neither of these things to happen, so we return None here to say that it hasn't been
            // handled. But because we've cut it, it will not propagate to any other alloy components
            return Optional.none();
        };
        const handleByHighlightedItem = (comp, se) => {
            // Because we need to redispatch based on highlighted items that we don't know about here,
            // we are going to emit an event, that the sandbox listens to, and the sandbox will
            // redispatch the event.
            const eventData = {
                interactionEvent: se.event,
                eventType: se.event.raw.type
            };
            emitWith(comp, redirectMenuItemInteractionEvent, eventData);
            return Optional.some(true);
        };
        const customSearcherEventsName = 'searcher-events';
        return {
            dom: {
                tag: 'div',
                // NOTE: This is very intentionally NOT the navigation class, because
                // we don't want the searcher to be part of the navigation. This class
                // is just for styling consistency. Perhaps it should be its own class.
                classes: [selectableClass]
            },
            components: [
                Input.sketch({
                    inputClasses: [menuSearcherClass, 'tox-textfield'],
                    inputAttributes: {
                        ...(spec.placeholder.map((placeholder) => ({ placeholder: spec.i18n(placeholder) })).getOr({})),
                        // This ARIA is based on the algolia example documented in TINY-8952
                        'type': 'search',
                        'aria-autocomplete': 'list'
                    },
                    inputBehaviours: derive$1([
                        config(customSearcherEventsName, [
                            // When the user types into the search field, we want to retrigger
                            // a fetch on the dropdown. This will be fired from within the
                            // dropdown's sandbox, so the dropdown is going to have to listen
                            // for it there. See CommonDropdown.ts.
                            run$1(
                            // Use "input" to handle keydown, paste etc.
                            input(), (inputComp) => {
                                emit(inputComp, refetchTriggerEvent);
                            }),
                            run$1(keydown(), (inputComp, se) => {
                                // The Special Keying config type since TINY-7005 processes the Escape
                                // key on keyup, not keydown. We need to stop the keydown event for this
                                // input, because some browsers (e.g. Chrome) will process a keydown
                                // for Escape inside an input[type=search] by clearing the input value,
                                // and then triggering an "input" event. This "input" event will trigger
                                // a refetch, which if it completes before the keyup is fired for Escape,
                                // will go back to only showing one level of menu. Then, when the escape
                                // keyup is processed by Keying, it will close the single remaining menu.
                                // This has the effect of closing *all* menus that are open when Escape is
                                // pressed instead of the last one. So, instead, we are going to kill the
                                // keydown event, so that it doesn't have the default browser behaviour, and
                                // won't trigger an input (and then Refetch). Then the keyup will still fire
                                // so just one level of the menu will close. This is all based on the underlying
                                // assumption that preventDefault and/or stop on a keydown does not suppress
                                // the related keyup. All of the documentation found so far, suggests it should
                                // only suppress the keypress, not the keyup, but that might not be across all
                                // browsers, or implemented consistently.
                                if (se.event.raw.key === 'Escape') {
                                    se.stop();
                                }
                            })
                        ]),
                        // In addition to input handling, we want special handling for
                        // Up/Down/Left/Right/Enter/Escape/Space. We can divide these into two categories
                        //  - events that we don't want to allow the overall menu system to process (left and right and space)
                        //  - events that we want to redispatch on the "highlighted item" based on the
                        // current fake focus.
                        Keying.config({
                            mode: 'special',
                            onLeft: handleByBrowser,
                            onRight: handleByBrowser,
                            onSpace: handleByBrowser,
                            onEnter: handleByHighlightedItem,
                            onEscape: handleByHighlightedItem,
                            onUp: handleByHighlightedItem,
                            onDown: handleByHighlightedItem
                        })
                    ]),
                    // Because we have customised handling for keydown, and we are configuring
                    // Keying, we need to specify which "behaviour" (custom events or keying) gets to
                    // process the keydown event first. In this situation, we want to stop escape before
                    // anything happens (although it really isn't necessary)
                    eventOrder: {
                        keydown: [customSearcherEventsName, Keying.name()]
                    }
                })
            ]
        };
    };

    const searchResultsClass = 'tox-collection--results__js';
    // NOTE: this is operating on the the final AlloySpec
    const augmentWithAria = (item) => {
        if (item.dom) {
            return {
                ...item,
                dom: {
                    ...item.dom,
                    attributes: {
                        ...item.dom.attributes ?? {},
                        'id': generate$6('aria-item-search-result-id'),
                        'aria-selected': 'false'
                    }
                }
            };
        }
        else {
            return item;
        }
    };

    const widgetAriaLabel = 'Use arrow keys to navigate.';
    const chunk = (rowDom, numColumns) => (items) => {
        const chunks = chunk$1(items, numColumns);
        return map$2(chunks, (c) => ({
            dom: rowDom,
            components: c
        }));
    };
    const forSwatch = (columns) => ({
        dom: {
            tag: 'div',
            classes: ['tox-menu', 'tox-swatches-menu'],
            attributes: {
                'aria-label': global$6.translate(widgetAriaLabel)
            }
        },
        components: [
            {
                dom: {
                    tag: 'div',
                    classes: ['tox-swatches']
                },
                components: [
                    Menu.parts.items({
                        preprocess: columns !== 'auto' ? chunk({
                            tag: 'div',
                            classes: ['tox-swatches__row']
                        }, columns) : identity
                    })
                ]
            }
        ]
    });
    const forImageSelector = (columns) => ({
        dom: {
            tag: 'div',
            classes: ['tox-menu', 'tox-image-selector-menu']
        },
        components: [
            {
                dom: {
                    tag: 'div',
                    classes: ['tox-image-selector']
                },
                components: [
                    Menu.parts.items({
                        preprocess: columns !== 'auto' ? chunk({
                            tag: 'div',
                            classes: ['tox-image-selector__row']
                        }, columns) : identity
                    })
                ]
            }
        ]
    });
    const forToolbar = (columns) => ({
        dom: {
            tag: 'div',
            // TODO: Configurable lg setting?
            classes: ['tox-menu', 'tox-collection', 'tox-collection--toolbar', 'tox-collection--toolbar-lg']
        },
        components: [
            Menu.parts.items({
                preprocess: chunk({
                    tag: 'div',
                    classes: ['tox-collection__group']
                }, columns)
            })
        ]
    });
    // NOTE: That type signature isn't quite true.
    const preprocessCollection = (items, isSeparator) => {
        const allSplits = [];
        let currentSplit = [];
        each$1(items, (item, i) => {
            if (isSeparator(item, i)) {
                if (currentSplit.length > 0) {
                    allSplits.push(currentSplit);
                }
                currentSplit = [];
                if (has$2(item.dom, 'innerHtml') || item.components && item.components.length > 0) {
                    currentSplit.push(item);
                }
            }
            else {
                currentSplit.push(item);
            }
        });
        if (currentSplit.length > 0) {
            allSplits.push(currentSplit);
        }
        return map$2(allSplits, (s) => ({
            dom: {
                tag: 'div',
                classes: ['tox-collection__group']
            },
            components: s
        }));
    };
    const insertItemsPlaceholder = (columns, initItems, onItem) => {
        return Menu.parts.items({
            preprocess: (rawItems) => {
                // Add any information to the items that is required. For example
                // when the items are results in a searchable menu, we need them to have
                // an ID that can be referenced by aria-activedescendant
                const enrichedItems = map$2(rawItems, onItem);
                if (columns !== 'auto' && columns > 1) {
                    return chunk({
                        tag: 'div',
                        classes: ['tox-collection__group']
                    }, columns)(enrichedItems);
                }
                else {
                    return preprocessCollection(enrichedItems, (_item, i) => initItems[i].type === 'separator');
                }
            }
        });
    };
    const hasWidget = (items) => exists(items, (item) => item.type === 'widget');
    const forCollection = (columns, initItems, _hasIcons = true) => ({
        dom: {
            tag: 'div',
            classes: ['tox-menu', 'tox-collection'].concat(columns === 1 ? ['tox-collection--list'] : ['tox-collection--grid']),
            attributes: {
                // widget item can be inserttable, colorswatch or imageselect - all of them are navigated with arrow keys
                ...hasWidget(initItems) ? { 'aria-label': global$6.translate(widgetAriaLabel) } : {}
            },
        },
        components: [
            // We don't need to add IDs for each item because there are no
            // aria relationships we need to maintain
            insertItemsPlaceholder(columns, initItems, identity)
        ]
    });
    const forCollectionWithSearchResults = (columns, initItems, _hasIcons = true) => {
        // A collection with results is exactly like a collection, except it also has
        // an ID and class on its outer div to allow for aria-controls relationships, and ids
        // on its items.
        // This connects the search bar with the list box.
        const ariaControlsSearchResults = generate$6('aria-controls-search-results');
        return {
            dom: {
                tag: 'div',
                classes: ['tox-menu', 'tox-collection', searchResultsClass].concat(columns === 1 ? ['tox-collection--list'] : ['tox-collection--grid']),
                attributes: {
                    id: ariaControlsSearchResults
                }
            },
            components: [
                // For each item, it needs to have an ID, so that we can refer to it
                // by the aria-activedescendant attribute
                insertItemsPlaceholder(columns, initItems, augmentWithAria)
            ]
        };
    };
    // Does a searchable menu *really* support columns !== 1 ?
    const forCollectionWithSearchField = (columns, initItems, searchField) => {
        // This connects the search bar with the list box.
        const ariaControlsSearchResults = generate$6('aria-controls-search-results');
        return {
            dom: {
                tag: 'div',
                classes: ['tox-menu', 'tox-collection'].concat(columns === 1 ? ['tox-collection--list'] : ['tox-collection--grid'])
            },
            components: [
                // Importantly, the search bar is not in the "items" part, which means that it is
                // not given any of the item decorations by default. In order to ensure that is
                // not part of the navigation, however, we need to prevent it from getting the nav
                // class. For general collection menu items, it is navClass, which is:
                // tox-menu-nav__js. So simply, do not add this class when creating
                // the search, so that it isn't in the navigation. Ideally, it would only ever look
                // inside its items section, but the items aren't guaranteed to have a separate
                // container, and navigation candidates are found anywhere inside the menu
                // container. We could add configuration to alloy's Menu movement, where there was
                // a 'navigation container' that all items would be in. That could be another
                // way to solve the problem. For now, we'll just manually avoid adding the navClass
                renderMenuSearcher({
                    i18n: global$6.translate,
                    placeholder: searchField.placeholder
                }),
                {
                    // We need a separate container for the items, because this is the container
                    // that multiple tox-collection__groups might go into, and will be the container
                    // that the search bar controls.
                    dom: {
                        tag: 'div',
                        classes: [
                            ...(columns === 1 ? ['tox-collection--list'] : ['tox-collection--grid']),
                            searchResultsClass
                        ],
                        attributes: {
                            id: ariaControlsSearchResults
                        }
                    },
                    components: [
                        // For each item, it needs to have an ID, so that we can refer to it
                        // by the aria-activedescendant attribute
                        insertItemsPlaceholder(columns, initItems, augmentWithAria)
                    ]
                }
            ]
        };
    };
    const forHorizontalCollection = (initItems, _hasIcons = true) => ({
        dom: {
            tag: 'div',
            classes: ['tox-collection', 'tox-collection--horizontal']
        },
        components: [
            Menu.parts.items({
                preprocess: (items) => preprocessCollection(items, (_item, i) => initItems[i].type === 'separator')
            })
        ]
    });

    const menuHasIcons = (xs) => exists(xs, (item) => 'icon' in item && item.icon !== undefined);
    const handleError = (error) => {
        // eslint-disable-next-line no-console
        console.error(formatError(error));
        // eslint-disable-next-line no-console
        console.log(error);
        return Optional.none();
    };
    const createHorizontalPartialMenuWithAlloyItems = (value, _hasIcons, items, _columns, _menuLayout) => {
        // Horizontal collections do not support different menu layout structures currently.
        const structure = forHorizontalCollection(items);
        return {
            value,
            dom: structure.dom,
            components: structure.components,
            items
        };
    };
    const createPartialMenuWithAlloyItems = (value, hasIcons, items, columns, menuLayout) => {
        const getNormalStructure = () => {
            if (menuLayout.menuType !== 'searchable') {
                return forCollection(columns, items);
            }
            else {
                return menuLayout.searchMode.searchMode === 'search-with-field'
                    ? forCollectionWithSearchField(columns, items, menuLayout.searchMode)
                    : forCollectionWithSearchResults(columns, items);
            }
        };
        if (menuLayout.menuType === 'color') {
            const structure = forSwatch(columns);
            return {
                value,
                dom: structure.dom,
                components: structure.components,
                items
            };
        }
        else if (menuLayout.menuType === 'imageselector' && columns !== 'auto') {
            const structure = forImageSelector(columns);
            return {
                value,
                dom: structure.dom,
                components: structure.components,
                items
            };
        }
        else if (menuLayout.menuType === 'normal' && columns === 'auto') {
            const structure = forCollection(columns, items);
            return {
                value,
                dom: structure.dom,
                components: structure.components,
                items
            };
        }
        else if (menuLayout.menuType === 'normal' || menuLayout.menuType === 'searchable') {
            const structure = getNormalStructure();
            return {
                value,
                dom: structure.dom,
                components: structure.components,
                items
            };
        }
        else if (menuLayout.menuType === 'listpreview' && columns !== 'auto') {
            const structure = forToolbar(columns);
            return {
                value,
                dom: structure.dom,
                components: structure.components,
                items
            };
        }
        else {
            return {
                value,
                dom: dom(hasIcons, columns, menuLayout.menuType),
                components: components,
                items
            };
        }
    };

    const type = requiredString('type');
    const name = requiredString('name');
    const label = requiredString('label');
    const text = requiredString('text');
    const title = requiredString('title');
    const icon = requiredString('icon');
    const url = requiredString('url');
    const value = requiredString('value');
    const fetch = requiredFunction('fetch');
    const getSubmenuItems = requiredFunction('getSubmenuItems');
    const onAction = requiredFunction('onAction');
    const onItemAction = requiredFunction('onItemAction');
    const onSetup = defaultedFunction('onSetup', () => noop);
    const optionalName = optionString('name');
    const optionalText = optionString('text');
    const optionalRole = optionString('role');
    const optionalIcon = optionString('icon');
    const optionalTooltip = optionString('tooltip');
    const optionalChevronTooltip = optionString('chevronTooltip');
    const optionalLabel = optionString('label');
    const optionalShortcut = optionString('shortcut');
    const optionalSelect = optionFunction('select');
    const active = defaultedBoolean('active', false);
    const borderless = defaultedBoolean('borderless', false);
    const enabled = defaultedBoolean('enabled', true);
    const primary = defaultedBoolean('primary', false);
    const defaultedColumns = (num) => defaulted('columns', num);
    const defaultedMeta = defaulted('meta', {});
    const defaultedOnAction = defaultedFunction('onAction', noop);
    const defaultedType = (type) => defaultedString('type', type);
    const generatedName = (namePrefix) => field$1('name', 'name', defaultedThunk(() => generate$6(`${namePrefix}-name`)), string);
    const generatedValue = (valuePrefix) => field$1('value', 'value', defaultedThunk(() => generate$6(`${valuePrefix}-value`)), anyValue());

    const alertBannerFields = [
        type,
        text,
        requiredStringEnum('level', ['info', 'warn', 'error', 'success']),
        icon,
        defaulted('url', '')
    ];
    const alertBannerSchema = objOf(alertBannerFields);

    const createBarFields = (itemsField) => [
        type,
        itemsField
    ];

    const buttonFields = [
        type,
        text,
        enabled,
        generatedName('button'),
        optionalIcon,
        borderless,
        // this should be defaulted to `secondary` but the implementation needs to manage the deprecation
        optionStringEnum('buttonType', ['primary', 'secondary', 'toolbar']),
        // this should be removed, but must live here because FieldSchema doesn't have a way to manage deprecated fields
        primary,
        defaultedString('context', 'mode:design')
    ];
    const buttonSchema = objOf(buttonFields);

    const formComponentFields = [
        type,
        name
    ];
    const formComponentWithLabelFields = formComponentFields.concat([
        optionalLabel
    ]);

    const checkboxFields = formComponentFields.concat([
        label,
        enabled,
        defaultedString('context', 'mode:design')
    ]);
    const checkboxSchema = objOf(checkboxFields);
    const checkboxDataProcessor = boolean;

    const collectionFields = formComponentWithLabelFields.concat([
        defaultedColumns('auto'),
        defaultedString('context', 'mode:design')
    ]);
    const collectionSchema = objOf(collectionFields);
    // TODO: Make type for CollectionItem
    const collectionDataProcessor = arrOfObj([
        value,
        text,
        icon
    ]);

    const colorInputFields = formComponentWithLabelFields.concat([
        defaultedString('storageKey', 'default'),
        defaultedString('context', 'mode:design'),
    ]);
    const colorInputSchema = objOf(colorInputFields);
    const colorInputDataProcessor = string;

    const colorPickerFields = formComponentWithLabelFields;
    const colorPickerSchema = objOf(colorPickerFields);
    const colorPickerDataProcessor = string;

    const customEditorFields = formComponentFields.concat([
        defaultedString('tag', 'textarea'),
        requiredString('scriptId'),
        requiredString('scriptUrl'),
        optionFunction('onFocus'),
        defaultedPostMsg('settings', undefined)
    ]);
    const customEditorFieldsOld = formComponentFields.concat([
        defaultedString('tag', 'textarea'),
        requiredFunction('init')
    ]);
    const customEditorSchema = valueOf((v) => asRaw('customeditor.old', objOfOnly(customEditorFieldsOld), v).orThunk(() => asRaw('customeditor.new', objOfOnly(customEditorFields), v)));
    const customEditorDataProcessor = string;

    const commonMenuItemFields = [
        enabled,
        optionalText,
        optionalRole,
        optionalShortcut,
        generatedValue('menuitem'),
        defaultedMeta,
        defaultedString('context', 'mode:design')
    ];

    const dialogToggleMenuItemSchema = objOf([
        type,
        name
    ].concat(commonMenuItemFields));
    const dialogToggleMenuItemDataProcessor = boolean;

    const baseFooterButtonFields = [
        generatedName('button'),
        optionalIcon,
        defaultedStringEnum('align', 'end', ['start', 'end']),
        // this should be removed, but must live here because FieldSchema doesn't have a way to manage deprecated fields
        primary,
        enabled,
        // this should be defaulted to `secondary` but the implementation needs to manage the deprecation
        optionStringEnum('buttonType', ['primary', 'secondary']),
        defaultedString('context', 'mode:design')
    ];
    const dialogFooterButtonFields = [
        ...baseFooterButtonFields,
        text
    ];
    const normalFooterButtonFields = [
        requiredStringEnum('type', ['submit', 'cancel', 'custom']),
        ...dialogFooterButtonFields
    ];
    const menuFooterButtonFields = [
        requiredStringEnum('type', ['menu']),
        optionalText,
        optionalTooltip,
        optionalIcon,
        requiredArrayOf('items', dialogToggleMenuItemSchema),
        ...baseFooterButtonFields
    ];
    const toggleButtonSpecFields = [
        ...baseFooterButtonFields,
        requiredStringEnum('type', ['togglebutton']),
        optionalTooltip,
        optionalIcon,
        optionalText,
        defaultedBoolean('active', false)
    ];
    const dialogFooterButtonSchema = choose$1('type', {
        submit: normalFooterButtonFields,
        cancel: normalFooterButtonFields,
        custom: normalFooterButtonFields,
        menu: menuFooterButtonFields,
        togglebutton: toggleButtonSpecFields
    });

    const dropZoneFields = formComponentWithLabelFields.concat([
        defaultedString('context', 'mode:design'),
    ]);
    const dropZoneSchema = objOf(dropZoneFields);
    const dropZoneDataProcessor = arrOfVal();

    const createGridFields = (itemsField) => [
        type,
        requiredNumber('columns'),
        itemsField
    ];

    const htmlPanelFields = [
        type,
        requiredString('html'),
        defaultedStringEnum('presets', 'presentation', ['presentation', 'document']),
        defaultedFunction('onInit', noop),
        defaultedBoolean('stretched', false),
    ];
    const htmlPanelSchema = objOf(htmlPanelFields);

    const iframeFields = formComponentWithLabelFields.concat([
        defaultedBoolean('border', false),
        defaultedBoolean('sandboxed', true),
        defaultedBoolean('streamContent', false),
        defaultedBoolean('transparent', true)
    ]);
    const iframeSchema = objOf(iframeFields);
    const iframeDataProcessor = string;

    const imagePreviewSchema = objOf(formComponentFields.concat([
        optionString('height'),
    ]));
    const imagePreviewDataProcessor = objOf([
        requiredString('url'),
        optionNumber('zoom'),
        optionNumber('cachedWidth'),
        optionNumber('cachedHeight'),
    ]);

    const inputFields = formComponentWithLabelFields.concat([
        optionString('inputMode'),
        optionString('placeholder'),
        defaultedBoolean('maximized', false),
        enabled,
        defaultedString('context', 'mode:design'),
    ]);
    const inputSchema = objOf(inputFields);
    const inputDataProcessor = string;

    const createLabelFields = (itemsField) => [
        type,
        label,
        itemsField,
        defaultedStringEnum('align', 'start', ['start', 'center', 'end']),
        optionString('for')
    ];

    const listBoxSingleItemFields = [
        text,
        value
    ];
    const listBoxNestedItemFields = [
        text,
        requiredArrayOf('items', thunkOf('items', () => listBoxItemSchema))
    ];
    const listBoxItemSchema = oneOf([
        objOf(listBoxSingleItemFields),
        objOf(listBoxNestedItemFields)
    ]);
    const listBoxFields = formComponentWithLabelFields.concat([
        requiredArrayOf('items', listBoxItemSchema),
        enabled,
        defaultedString('context', 'mode:design')
    ]);
    const listBoxSchema = objOf(listBoxFields);
    const listBoxDataProcessor = string;

    const selectBoxFields = formComponentWithLabelFields.concat([
        requiredArrayOfObj('items', [
            text,
            value
        ]),
        defaultedNumber('size', 1),
        enabled,
        defaultedString('context', 'mode:design')
    ]);
    const selectBoxSchema = objOf(selectBoxFields);
    const selectBoxDataProcessor = string;

    const sizeInputFields = formComponentWithLabelFields.concat([
        defaultedBoolean('constrain', true),
        enabled,
        defaultedString('context', 'mode:design')
    ]);
    const sizeInputSchema = objOf(sizeInputFields);
    const sizeInputDataProcessor = objOf([
        requiredString('width'),
        requiredString('height')
    ]);

    const sliderFields = formComponentFields.concat([
        label,
        defaultedNumber('min', 0),
        defaultedNumber('max', 0),
    ]);
    const sliderSchema = objOf(sliderFields);
    const sliderInputDataProcessor = number;

    const tableFields = [
        type,
        requiredArrayOf('header', string),
        requiredArrayOf('cells', arrOf(string))
    ];
    const tableSchema = objOf(tableFields);

    const textAreaFields = formComponentWithLabelFields.concat([
        optionString('placeholder'),
        defaultedBoolean('maximized', false),
        enabled,
        defaultedString('context', 'mode:design'),
        optionBoolean('spellcheck'),
    ]);
    const textAreaSchema = objOf(textAreaFields);
    const textAreaDataProcessor = string;

    const baseMenuButtonFields = [
        defaultedString('buttonType', 'default'),
        optionString('text'),
        optionString('tooltip'),
        optionString('icon'),
        defaultedOf('search', false, 
        // So our boulder validation are:
        // a) boolean -> we need to map it into an Option
        // b) object -> we need to map it into a Some
        oneOf([
            // Unfortunately, due to objOf not checking to see that the
            // input is an object, the boolean check MUST be first
            boolean,
            objOf([
                optionString('placeholder')
            ])
        ], 
        // This function allows you to standardise the output.
        (x) => {
            if (isBoolean(x)) {
                return x ? Optional.some({ placeholder: Optional.none() }) : Optional.none();
            }
            else {
                return Optional.some(x);
            }
        })),
        requiredFunction('fetch'),
        defaultedFunction('onSetup', () => noop),
        defaultedString('context', 'mode:design')
    ];

    const MenuButtonSchema = objOf([
        type,
        ...baseMenuButtonFields
    ]);
    const createMenuButton = (spec) => asRaw('menubutton', MenuButtonSchema, spec);

    const baseTreeItemFields = [
        requiredStringEnum('type', ['directory', 'leaf']),
        title,
        requiredString('id'),
        optionOf('menu', MenuButtonSchema),
        optionString('customStateIcon'),
        optionString('customStateIconTooltip'),
    ];
    const treeItemLeafFields = baseTreeItemFields;
    const treeItemLeafSchema = objOf(treeItemLeafFields);
    const treeItemDirectoryFields = baseTreeItemFields.concat([
        requiredArrayOf('children', thunkOf('children', () => {
            return choose$2('type', {
                directory: treeItemDirectorySchema,
                leaf: treeItemLeafSchema,
            });
        })),
    ]);
    const treeItemDirectorySchema = objOf(treeItemDirectoryFields);
    const treeItemSchema = choose$2('type', {
        directory: treeItemDirectorySchema,
        leaf: treeItemLeafSchema,
    });
    const treeFields = [
        type,
        requiredArrayOf('items', treeItemSchema),
        optionFunction('onLeafAction'),
        optionFunction('onToggleExpand'),
        defaultedArrayOf('defaultExpandedIds', [], string),
        optionString('defaultSelectedId'),
    ];
    const treeSchema = objOf(treeFields);

    const urlInputFields = formComponentWithLabelFields.concat([
        defaultedStringEnum('filetype', 'file', ['image', 'media', 'file']),
        enabled,
        optionString('picker_text'),
        defaultedString('context', 'mode:design')
    ]);
    const urlInputSchema = objOf(urlInputFields);
    const urlInputDataProcessor = objOf([
        value,
        defaultedMeta
    ]);

    const createItemsField = (name) => field$1('items', 'items', required$2(), arrOf(valueOf((v) => asRaw(`Checking item of ${name}`, itemSchema$1, v).fold((sErr) => Result.error(formatError(sErr)), (passValue) => Result.value(passValue)))));
    // We're using a thunk here so we can refer to panel fields
    const itemSchema$1 = valueThunk(() => choose$2('type', {
        alertbanner: alertBannerSchema,
        bar: objOf(createBarFields(createItemsField('bar'))),
        button: buttonSchema,
        checkbox: checkboxSchema,
        colorinput: colorInputSchema,
        colorpicker: colorPickerSchema,
        dropzone: dropZoneSchema,
        grid: objOf(createGridFields(createItemsField('grid'))),
        iframe: iframeSchema,
        input: inputSchema,
        listbox: listBoxSchema,
        selectbox: selectBoxSchema,
        sizeinput: sizeInputSchema,
        slider: sliderSchema,
        textarea: textAreaSchema,
        urlinput: urlInputSchema,
        customeditor: customEditorSchema,
        htmlpanel: htmlPanelSchema,
        imagepreview: imagePreviewSchema,
        collection: collectionSchema,
        label: objOf(createLabelFields(createItemsField('label'))),
        table: tableSchema,
        tree: treeSchema,
        panel: panelSchema
    }));
    const panelFields = [
        type,
        defaulted('classes', []),
        requiredArrayOf('items', itemSchema$1)
    ];
    const panelSchema = objOf(panelFields);

    const tabFields = [
        generatedName('tab'),
        title,
        requiredArrayOf('items', itemSchema$1)
    ];
    const tabPanelFields = [
        type,
        requiredArrayOfObj('tabs', tabFields)
    ];
    const tabPanelSchema = objOf(tabPanelFields);

    const dialogButtonFields = dialogFooterButtonFields;
    const dialogButtonSchema = dialogFooterButtonSchema;
    const dialogSchema = objOf([
        requiredString('title'),
        requiredOf('body', choose$2('type', {
            panel: panelSchema,
            tabpanel: tabPanelSchema
        })),
        defaultedString('size', 'normal'),
        defaultedArrayOf('buttons', [], dialogButtonSchema),
        defaulted('initialData', {}),
        defaultedFunction('onAction', noop),
        defaultedFunction('onChange', noop),
        defaultedFunction('onSubmit', noop),
        defaultedFunction('onClose', noop),
        defaultedFunction('onCancel', noop),
        defaultedFunction('onTabChange', noop)
    ]);
    const createDialog = (spec) => asRaw('dialog', dialogSchema, spec);

    const urlDialogButtonSchema = objOf([
        requiredStringEnum('type', ['cancel', 'custom']),
        ...dialogButtonFields
    ]);
    const urlDialogSchema = objOf([
        requiredString('title'),
        requiredString('url'),
        optionNumber('height'),
        optionNumber('width'),
        optionArrayOf('buttons', urlDialogButtonSchema),
        defaultedFunction('onAction', noop),
        defaultedFunction('onCancel', noop),
        defaultedFunction('onClose', noop),
        defaultedFunction('onMessage', noop)
    ]);
    const createUrlDialog = (spec) => asRaw('dialog', urlDialogSchema, spec);

    // eslint-disable-next-line @typescript-eslint/no-wrapper-object-types
    const getAllObjects = (obj) => {
        if (isObject(obj)) {
            return [obj].concat(bind$3(values(obj), getAllObjects));
        }
        else if (isArray(obj)) {
            return bind$3(obj, getAllObjects);
        }
        else {
            return [];
        }
    };

    const isNamedItem = (obj) => isString(obj.type) && isString(obj.name);
    const dataProcessors = {
        checkbox: checkboxDataProcessor,
        colorinput: colorInputDataProcessor,
        colorpicker: colorPickerDataProcessor,
        dropzone: dropZoneDataProcessor,
        input: inputDataProcessor,
        iframe: iframeDataProcessor,
        imagepreview: imagePreviewDataProcessor,
        selectbox: selectBoxDataProcessor,
        sizeinput: sizeInputDataProcessor,
        slider: sliderInputDataProcessor,
        listbox: listBoxDataProcessor,
        size: sizeInputDataProcessor,
        textarea: textAreaDataProcessor,
        urlinput: urlInputDataProcessor,
        customeditor: customEditorDataProcessor,
        collection: collectionDataProcessor,
        togglemenuitem: dialogToggleMenuItemDataProcessor
    };
    const getDataProcessor = (item) => Optional.from(dataProcessors[item.type]);
    const getNamedItems = (structure) => filter$2(getAllObjects(structure), isNamedItem);

    const createDataValidator = (structure) => {
        const namedItems = getNamedItems(structure);
        const fields = bind$3(namedItems, (item) => getDataProcessor(item).fold(() => [], (schema) => [requiredOf(item.name, schema)]));
        return objOf(fields);
    };

    const extract = (structure) => {
        const internalDialog = getOrDie(createDialog(structure));
        const dataValidator = createDataValidator(structure);
        // We used to validate data here, but it's done when loading the dialog in tinymce
        const initialData = structure.initialData ?? {};
        return {
            internalDialog,
            dataValidator,
            initialData
        };
    };
    const DialogManager = {
        open: (factory, structure) => {
            const extraction = extract(structure);
            return factory(extraction.internalDialog, extraction.initialData, extraction.dataValidator);
        },
        openUrl: (factory, structure) => {
            const internalDialog = getOrDie(createUrlDialog(structure));
            return factory(internalDialog);
        },
        redial: (structure) => extract(structure)
    };

    const separatorMenuItemSchema = objOf([
        type,
        optionalText
    ]);
    const createSeparatorMenuItem = (spec) => asRaw('separatormenuitem', separatorMenuItemSchema, spec);

    const autocompleterItemSchema = objOf([
        // Currently, autocomplete items don't support configuring type, active, disabled, meta
        defaultedType('autocompleteitem'),
        active,
        enabled,
        defaultedMeta,
        value,
        optionalText,
        optionalIcon
    ]);
    objOf([
        type,
        requiredString('trigger'),
        defaultedNumber('minChars', 1),
        defaultedColumns(1),
        defaultedNumber('maxResults', 10),
        optionFunction('matches'),
        fetch,
        onAction,
        defaultedArrayOf('highlightOn', [], string)
    ]);
    const createSeparatorItem = (spec) => asRaw('Autocompleter.Separator', separatorMenuItemSchema, spec);
    const createAutocompleterItem = (spec) => asRaw('Autocompleter.Item', autocompleterItemSchema, spec);

    const baseToolbarButtonFields = [
        enabled,
        optionalTooltip,
        optionalIcon,
        optionalText,
        onSetup,
        defaultedString('context', 'mode:design')
    ];
    const toolbarButtonSchema = objOf([
        type,
        onAction,
        optionalShortcut
    ].concat(baseToolbarButtonFields));
    const createToolbarButton = (spec) => asRaw('toolbarbutton', toolbarButtonSchema, spec);

    const baseToolbarToggleButtonFields = [
        active
    ].concat(baseToolbarButtonFields);
    const toggleButtonSchema = objOf(baseToolbarToggleButtonFields.concat([
        type,
        onAction,
        optionalShortcut
    ]));
    const createToggleButton = (spec) => asRaw('ToggleButton', toggleButtonSchema, spec);

    const contextBarFields = [
        defaultedFunction('predicate', never),
        defaultedStringEnum('scope', 'node', ['node', 'editor']),
        defaultedStringEnum('position', 'selection', ['node', 'selection', 'line'])
    ];

    const contextButtonFields = baseToolbarButtonFields.concat([
        defaultedType('contextformbutton'),
        defaultedString('align', 'end'),
        primary,
        onAction,
        customField('original', identity)
    ]);
    const contextToggleButtonFields = baseToolbarToggleButtonFields.concat([
        defaultedType('contextformbutton'),
        defaultedString('align', 'end'),
        primary,
        onAction,
        customField('original', identity)
    ]);
    const launchButtonFields$1 = baseToolbarButtonFields.concat([
        defaultedType('contextformbutton')
    ]);
    const launchToggleButtonFields = baseToolbarToggleButtonFields.concat([
        defaultedType('contextformtogglebutton')
    ]);
    const toggleOrNormal = choose$1('type', {
        contextformbutton: contextButtonFields,
        contextformtogglebutton: contextToggleButtonFields
    });
    const baseContextFormFields = [
        optionalLabel,
        requiredArrayOf('commands', toggleOrNormal),
        optionOf('launch', choose$1('type', {
            contextformbutton: launchButtonFields$1,
            contextformtogglebutton: launchToggleButtonFields
        })),
        defaultedFunction('onInput', noop),
        defaultedFunction('onSetup', noop)
    ];
    const contextFormFields = [
        ...contextBarFields,
        ...baseContextFormFields,
        requiredStringEnum('type', ['contextform']),
        defaultedFunction('initValue', constant$1('')),
        optionString('placeholder'),
    ];
    const contextSliderFormFields = [
        ...contextBarFields,
        ...baseContextFormFields,
        requiredStringEnum('type', ['contextsliderform']),
        defaultedFunction('initValue', constant$1(0)),
        defaultedFunction('min', constant$1(0)),
        defaultedFunction('max', constant$1(100))
    ];
    const contextSizeInputFormFields = [
        ...contextBarFields,
        ...baseContextFormFields,
        requiredStringEnum('type', ['contextsizeinputform']),
        defaultedFunction('initValue', constant$1({ width: '', height: '' }))
    ];
    const contextFormSchema = choose$1('type', {
        contextform: contextFormFields,
        contextsliderform: contextSliderFormFields,
        contextsizeinputform: contextSizeInputFormFields
    });
    const createContextForm = (spec) => asRaw('ContextForm', contextFormSchema, spec);

    const launchButtonFields = baseToolbarButtonFields.concat([
        defaultedType('contexttoolbarbutton')
    ]);
    const contextToolbarSchema = objOf([
        defaultedType('contexttoolbar'),
        optionObjOf('launch', launchButtonFields),
        requiredOf('items', oneOf([
            string,
            arrOfObj([
                optionString('name'),
                optionString('label'),
                requiredArrayOf('items', string)
            ])
        ])),
    ].concat(contextBarFields));
    const toolbarGroupBackToSpec = (toolbarGroup) => ({
        name: toolbarGroup.name.getOrUndefined(),
        label: toolbarGroup.label.getOrUndefined(),
        items: toolbarGroup.items
    });
    const contextToolbarToSpec = (contextToolbar) => ({
        ...contextToolbar,
        launch: contextToolbar.launch.getOrUndefined(),
        items: isString(contextToolbar.items) ? contextToolbar.items : map$2(contextToolbar.items, toolbarGroupBackToSpec)
    });
    const createContextToolbar = (spec) => asRaw('ContextToolbar', contextToolbarSchema, spec);

    const cardImageFields = [
        type,
        requiredString('src'),
        optionString('alt'),
        defaultedArrayOf('classes', [], string)
    ];
    const cardImageSchema = objOf(cardImageFields);

    const cardTextFields = [
        type,
        text,
        optionalName,
        defaultedArrayOf('classes', ['tox-collection__item-label'], string)
    ];
    const cardTextSchema = objOf(cardTextFields);

    const itemSchema = valueThunk(() => choose$2('type', {
        cardimage: cardImageSchema,
        cardtext: cardTextSchema,
        cardcontainer: cardContainerSchema
    }));
    const cardContainerSchema = objOf([
        type,
        defaultedString('direction', 'horizontal'),
        defaultedString('align', 'left'),
        defaultedString('valign', 'middle'),
        requiredArrayOf('items', itemSchema)
    ]);

    const cardMenuItemSchema = objOf([
        type,
        optionalLabel,
        requiredArrayOf('items', itemSchema),
        onSetup,
        defaultedOnAction
    ].concat(commonMenuItemFields));
    const createCardMenuItem = (spec) => asRaw('cardmenuitem', cardMenuItemSchema, spec);

    const choiceMenuItemSchema = objOf([
        type,
        active,
        optionalIcon,
        optionalLabel
    ].concat(commonMenuItemFields));
    const createChoiceMenuItem = (spec) => asRaw('choicemenuitem', choiceMenuItemSchema, spec);

    const baseFields = [
        type,
        requiredString('fancytype'),
        defaultedOnAction
    ];
    const insertTableFields = [
        defaulted('initData', {})
    ].concat(baseFields);
    const colorSwatchFields = [
        optionFunction('select'),
        defaultedObjOf('initData', {}, [
            defaultedBoolean('allowCustomColors', true),
            defaultedString('storageKey', 'default'),
            // Note: We don't validate the colors as they are instead validated by choiceschema when rendering
            optionArrayOf('colors', anyValue())
        ])
    ].concat(baseFields);
    const imageSelectFields = [
        optionFunction('select'),
        requiredObjOf('initData', [
            requiredNumber('columns'),
            // Note: We don't validate the items as they are instead validated by imageMenuItemSchema when rendering
            defaultedArrayOf('items', [], anyValue())
        ])
    ].concat(baseFields);
    const fancyMenuItemSchema = choose$1('fancytype', {
        inserttable: insertTableFields,
        colorswatch: colorSwatchFields,
        imageselect: imageSelectFields
    });
    const createFancyMenuItem = (spec) => asRaw('fancymenuitem', fancyMenuItemSchema, spec);

    const imageMenuItemSchema = objOf([
        type,
        active,
        url,
        optionalLabel,
        optionalTooltip
    ].concat(commonMenuItemFields));
    const resetImageItemSchema = objOf([
        type,
        active,
        icon,
        label,
        optionalTooltip,
        value
    ].concat(commonMenuItemFields));
    const createImageMenuItem = (spec) => asRaw('imagemenuitem', imageMenuItemSchema, spec);
    const createResetImageItem = (spec) => asRaw('resetimageitem', resetImageItemSchema, spec);

    const menuItemSchema = objOf([
        type,
        onSetup,
        defaultedOnAction,
        optionalIcon
    ].concat(commonMenuItemFields));
    const createMenuItem = (spec) => asRaw('menuitem', menuItemSchema, spec);

    const nestedMenuItemSchema = objOf([
        type,
        getSubmenuItems,
        onSetup,
        optionalIcon
    ].concat(commonMenuItemFields));
    const createNestedMenuItem = (spec) => asRaw('nestedmenuitem', nestedMenuItemSchema, spec);

    const toggleMenuItemSchema = objOf([
        type,
        optionalIcon,
        active,
        onSetup,
        onAction
    ].concat(commonMenuItemFields));
    const createToggleMenuItem = (spec) => asRaw('togglemenuitem', toggleMenuItemSchema, spec);

    const sidebarSchema = objOf([
        optionalIcon,
        optionalTooltip,
        defaultedFunction('onShow', noop),
        defaultedFunction('onHide', noop),
        onSetup
    ]);
    const createSidebar = (spec) => asRaw('sidebar', sidebarSchema, spec);

    const groupToolbarButtonSchema = objOf([
        type,
        requiredOf('items', oneOf([
            arrOfObj([
                name,
                requiredArrayOf('items', string)
            ]),
            string
        ]))
    ].concat(baseToolbarButtonFields));
    const createGroupToolbarButton = (spec) => asRaw('GroupToolbarButton', groupToolbarButtonSchema, spec);

    const splitButtonSchema = objOf([
        type,
        optionalTooltip,
        optionalChevronTooltip,
        optionalIcon,
        optionalText,
        optionalSelect,
        fetch,
        onSetup,
        // TODO: Validate the allowed presets
        defaultedStringEnum('presets', 'normal', ['normal', 'color', 'listpreview']),
        defaultedColumns(1),
        onAction,
        onItemAction,
        defaultedString('context', 'mode:design')
    ]);
    const createSplitButton = (spec) => asRaw('SplitButton', splitButtonSchema, spec);

    const baseButtonFields = [
        optionalText,
        optionalIcon,
        optionString('tooltip'),
        defaultedStringEnum('buttonType', 'secondary', ['primary', 'secondary']),
        defaultedBoolean('borderless', false),
        requiredFunction('onAction'),
        defaultedString('context', 'mode:design')
    ];
    const normalButtonFields = [
        ...baseButtonFields,
        text,
        requiredStringEnum('type', ['button']),
    ];
    const toggleButtonFields = [
        ...baseButtonFields,
        defaultedBoolean('active', false),
        requiredStringEnum('type', ['togglebutton'])
    ];
    const schemaWithoutGroupButton = {
        button: normalButtonFields,
        togglebutton: toggleButtonFields,
    };
    const groupFields = [
        requiredStringEnum('type', ['group']),
        defaultedArrayOf('buttons', [], choose$1('type', schemaWithoutGroupButton))
    ];
    const viewButtonSchema = choose$1('type', {
        ...schemaWithoutGroupButton,
        group: groupFields
    });

    const viewSchema = objOf([
        defaultedArrayOf('buttons', [], viewButtonSchema),
        requiredFunction('onShow'),
        requiredFunction('onHide')
    ]);
    const createView = (spec) => asRaw('view', viewSchema, spec);

    const detectSize = (comp, margin, selectorClass) => {
        const descendants$1 = descendants(comp.element, '.' + selectorClass);
        // TODO: This seems to cause performance issues in the emoji dialog
        if (descendants$1.length > 0) {
            const columnLength = findIndex$1(descendants$1, (c) => {
                const thisTop = c.dom.getBoundingClientRect().top;
                const cTop = descendants$1[0].dom.getBoundingClientRect().top;
                return Math.abs(thisTop - cTop) > margin;
            }).getOr(descendants$1.length);
            return Optional.some({
                numColumns: columnLength,
                numRows: Math.ceil(descendants$1.length / columnLength)
            });
        }
        else {
            return Optional.none();
        }
    };

    // Consider moving to alloy once it takes shape.
    const namedEvents = (name, handlers) => derive$1([
        config(name, handlers)
    ]);
    const unnamedEvents = (handlers) => namedEvents(generate$6('unnamed-events'), handlers);
    const SimpleBehaviours = {
        namedEvents,
        unnamedEvents
    };

    const item = (disabled) => Disabling.config({
        disabled,
        disableClass: 'tox-collection__item--state-disabled'
    });
    const button = (disabled) => Disabling.config({
        disabled
    });
    const splitButton = (disabled) => Disabling.config({
        disabled,
        disableClass: 'tox-tbtn--disabled'
    });
    const toolbarButton = (disabled) => Disabling.config({
        disabled,
        disableClass: 'tox-tbtn--disabled',
        useNative: false
    });
    const DisablingConfigs = {
        item,
        button,
        splitButton,
        toolbarButton
    };

    const runWithApi = (info, comp) => {
        const api = info.getApi(comp);
        return (f) => {
            f(api);
        };
    };
    // These handlers are used for providing common onAttached and onDetached handlers.
    // Essentially, the `editorOffCell` is used store the onDestroy function returned
    // by onSetup. The reason onControlAttached doesn't create the cell itself, is because
    // it also has to be passed into onControlDetached. We could make this function return
    // the cell and the onAttachedHandler, but that would provide too much complexity.
    const onControlAttached = (info, editorOffCell) => runOnAttached((comp) => {
        if (isFunction(info.onBeforeSetup)) {
            info.onBeforeSetup(comp);
        }
        const run = runWithApi(info, comp);
        run((api) => {
            const onDestroy = info.onSetup(api);
            if (isFunction(onDestroy)) {
                editorOffCell.set(onDestroy);
            }
        });
    });
    const onControlDetached = (getApi, editorOffCell) => runOnDetached((comp) => runWithApi(getApi, comp)(editorOffCell.get()));
    const onContextFormControlDetached = (getApi, editorOffCell, valueState) => runOnDetached((comp) => {
        valueState.set(Representing.getValue(comp));
        return runWithApi(getApi, comp)(editorOffCell.get());
    });

    const UiStateChannel = 'silver.uistate';
    const messageSetDisabled = 'setDisabled';
    const messageSetEnabled = 'setEnabled';
    const messageInit = 'init';
    const messageSwitchMode = 'switchmode';
    const modeContextMessages = [messageSwitchMode, messageInit];
    const broadcastEvents = (uiRefs, messageType) => {
        const outerContainer = uiRefs.mainUi.outerContainer;
        const motherships = [uiRefs.mainUi.mothership, ...uiRefs.uiMotherships];
        if (messageType === messageSetDisabled) {
            each$1(motherships, (m) => {
                m.broadcastOn([dismissPopups()], { target: outerContainer.element });
            });
        }
        each$1(motherships, (m) => {
            m.broadcastOn([UiStateChannel], messageType);
        });
    };
    const setupEventsForUi = (editor, uiRefs) => {
        editor.on('init SwitchMode', (event) => {
            broadcastEvents(uiRefs, event.type);
        });
        editor.on('DisabledStateChange', (event) => {
            if (!event.isDefaultPrevented()) {
                // When the event state indicates the editor is **enabled** (`event.state` is false),
                // we send an 'init' message instead of 'setEnabled' because the editor might be in read-only mode.
                // Sending 'setEnabled' would enable all the toolbar buttons, which is undesirable if the editor is read-only.
                const messageType = event.state ? messageSetDisabled : messageInit;
                broadcastEvents(uiRefs, messageType);
                // After refreshing the state of the buttons, trigger a NodeChange event.
                if (!event.state) {
                    editor.nodeChanged();
                }
            }
        });
        editor.on('NodeChange', (e) => {
            const messageType = editor.ui.isEnabled() ? e.type : messageSetDisabled;
            broadcastEvents(uiRefs, messageType);
        });
        if (isReadOnly(editor)) {
            editor.mode.set('readonly');
        }
    };
    const toggleOnReceive = (getContext) => Receiving.config({
        channels: {
            [UiStateChannel]: {
                onReceive: (comp, messageType) => {
                    if (messageType === messageSetDisabled || messageType === messageSetEnabled) {
                        Disabling.set(comp, messageType === messageSetDisabled);
                        return;
                    }
                    const { contextType, shouldDisable } = getContext();
                    if (contextType === 'mode' && !contains$2(modeContextMessages, messageType)) {
                        return;
                    }
                    Disabling.set(comp, shouldDisable);
                }
            }
        }
    });

    // Perform `action` when an item is clicked on, close menus, and stop event
    const onMenuItemExecute = (info, itemResponse) => runOnExecute$1((comp, simulatedEvent) => {
        // If there is an action, run the action
        runWithApi(info, comp)(info.onAction);
        if (!info.triggersSubmenu && itemResponse === ItemResponse$1.CLOSE_ON_EXECUTE) {
            if (comp.getSystem().isConnected()) {
                emit(comp, sandboxClose());
            }
            simulatedEvent.stop();
        }
    });
    const menuItemEventOrder = {
        // TODO: use the constants provided by behaviours.
        [execute$5()]: ['disabling', 'alloy.base.behaviour', 'toggling', 'item-events']
    };

    const componentRenderPipeline = cat;
    const renderCommonItem = (spec, structure, itemResponse, providersBackstage) => {
        const editorOffCell = Cell(noop);
        return {
            type: 'item',
            dom: structure.dom,
            components: componentRenderPipeline(structure.optComponents),
            data: spec.data,
            eventOrder: menuItemEventOrder,
            hasSubmenu: spec.triggersSubmenu,
            itemBehaviours: derive$1([
                config('item-events', [
                    onMenuItemExecute(spec, itemResponse),
                    onControlAttached(spec, editorOffCell),
                    onControlDetached(spec, editorOffCell)
                ]),
                DisablingConfigs.item(() => !spec.enabled || providersBackstage.checkUiComponentContext(spec.context).shouldDisable),
                toggleOnReceive(() => providersBackstage.checkUiComponentContext(spec.context)),
                Replacing.config({})
            ].concat(spec.itemBehaviours))
        };
    };
    const buildData = (source) => ({
        value: source.value,
        meta: {
            text: source.text.getOr(''),
            ...source.meta
        }
    });

    const renderImage$1 = (spec, imageUrl) => {
        const spinnerElement = SugarElement.fromTag('div');
        add$2(spinnerElement, 'tox-image-selector-loading-spinner');
        const addSpinnerElement = (loadingElement) => {
            add$2(loadingElement, 'tox-image-selector-loading-spinner-wrapper');
            append$2(loadingElement, spinnerElement);
        };
        const removeSpinnerElement = (loadingElement) => {
            remove$3(loadingElement, 'tox-image-selector-loading-spinner-wrapper');
            remove$7(spinnerElement);
        };
        return {
            dom: {
                tag: spec.tag,
                attributes: spec.attributes ?? {},
                classes: spec.classes,
            },
            components: [
                {
                    dom: {
                        tag: 'div',
                        classes: ['tox-image-selector-image-wrapper']
                    },
                    components: [
                        {
                            dom: {
                                tag: 'img',
                                attributes: { src: imageUrl },
                                classes: ['tox-image-selector-image-img']
                            }
                        },
                    ]
                },
                ...spec.checkMark.toArray()
            ],
            behaviours: derive$1([
                ...spec.behaviours ?? [],
                config('render-image-events', [
                    runOnAttached((component) => {
                        addSpinnerElement(component.element);
                        descendant(component.element, 'img').each((image$1) => {
                            image(image$1).catch((e) => {
                                // eslint-disable-next-line no-console
                                console.error(e);
                            }).finally(() => {
                                removeSpinnerElement(component.element);
                            });
                        });
                    })
                ]),
            ])
        };
    };
    const render$3 = (imageUrl, spec) => renderImage$1(spec, imageUrl);

    // Converts shortcut format to Mac/PC variants
    // Note: This is different to the help shortcut converter, as it doesn't padd the + symbol with spaces
    //       so as to not take up large amounts of space in the menus
    const convertText = (source) => {
        const isMac = global$7.os.isMacOS() || global$7.os.isiOS();
        const mac = {
            alt: '\u2325',
            ctrl: '\u2303',
            shift: '\u21E7',
            meta: '\u2318',
            access: '\u2303\u2325'
        };
        const other = {
            meta: 'Ctrl',
            access: 'Shift+Alt'
        };
        const replace = isMac ? mac : other;
        const shortcut = source.split('+');
        const updated = map$2(shortcut, (segment) => {
            // search lowercase, but if not found use the original
            const search = segment.toLowerCase().trim();
            return has$2(replace, search) ? replace[search] : segment;
        });
        return isMac ? updated.join('') : updated.join('+');
    };

    const renderIcon$2 = (name, icons, classes = [iconClass]) => render$4(name, { tag: 'div', classes }, icons);
    const renderText = (text) => ({
        dom: {
            tag: 'div',
            classes: [textClass]
        },
        components: [text$2(global$6.translate(text))]
    });
    const renderHtml = (html, classes) => ({
        dom: {
            tag: 'div',
            classes,
            innerHtml: html
        }
    });
    const renderStyledText = (style, text) => ({
        dom: {
            tag: 'div',
            classes: [textClass]
        },
        components: [
            {
                dom: {
                    tag: style.tag,
                    styles: style.styles
                },
                components: [text$2(global$6.translate(text))]
            }
        ]
    });
    const renderShortcut = (shortcut) => ({
        dom: {
            tag: 'div',
            classes: [accessoryClass]
        },
        components: [
            text$2(convertText(shortcut))
        ]
    });
    const renderCheckmark = (icons) => renderIcon$2('checkmark', icons, [checkmarkClass]);
    const renderSubmenuCaret = (icons) => renderIcon$2('chevron-right', icons, [caretClass]);
    const renderDownwardsCaret = (icons) => renderIcon$2('chevron-down', icons, [caretClass]);
    const renderContainer = (container, components) => {
        const directionClass = container.direction === 'vertical' ? containerColumnClass : containerRowClass;
        const alignClass = container.align === 'left' ? containerAlignLeftClass : containerAlignRightClass;
        const getValignClass = () => {
            switch (container.valign) {
                case 'top':
                    return containerValignTopClass;
                case 'middle':
                    return containerValignMiddleClass;
                case 'bottom':
                    return containerValignBottomClass;
            }
        };
        return {
            dom: {
                tag: 'div',
                classes: [
                    containerClass,
                    directionClass,
                    alignClass,
                    getValignClass()
                ]
            },
            components
        };
    };
    const renderImage = (src, classes, alt) => ({
        dom: {
            tag: 'img',
            classes,
            attributes: {
                src,
                alt: alt.getOr('')
            }
        }
    });

    const renderColorStructure = (item, providerBackstage, fallbackIcon) => {
        const colorPickerCommand = 'custom';
        const removeColorCommand = 'remove';
        const itemValue = item.value;
        const iconSvg = item.iconContent.map((name) => getOr(name, providerBackstage.icons, fallbackIcon));
        const attributes = item.ariaLabel.map((al) => ({
            'aria-label': providerBackstage.translate(al),
            'data-mce-name': al
        })).getOr({});
        const getDom = () => {
            const common = colorClass;
            const icon = iconSvg.getOr('');
            const baseDom = {
                tag: 'div',
                attributes,
                classes: [common]
            };
            if (itemValue === colorPickerCommand) {
                return {
                    ...baseDom,
                    tag: 'button',
                    classes: [...baseDom.classes, 'tox-swatches__picker-btn'],
                    innerHtml: icon
                };
            }
            else if (itemValue === removeColorCommand) {
                return {
                    ...baseDom,
                    classes: [...baseDom.classes, 'tox-swatch--remove'],
                    innerHtml: icon
                };
            }
            else if (isNonNullable(itemValue)) {
                return {
                    ...baseDom,
                    attributes: {
                        ...baseDom.attributes,
                        'data-mce-color': itemValue
                    },
                    styles: {
                        'background-color': itemValue
                    },
                    innerHtml: icon
                };
            }
            else {
                return baseDom;
            }
        };
        return {
            dom: getDom(),
            optComponents: []
        };
    };
    const renderItemDomStructure = (ariaLabel, classes) => {
        const domTitle = ariaLabel.map((label) => ({
            attributes: {
                'id': generate$6('menu-item'),
                'aria-label': global$6.translate(label)
            }
        })).getOr({});
        return {
            tag: 'div',
            classes: [navClass, selectableClass].concat(classes),
            ...domTitle
        };
    };
    const createLabel = (label) => {
        return {
            dom: {
                tag: 'label'
            },
            components: [
                text$2(label)
            ]
        };
    };
    const renderNormalItemStructure = (info, providersBackstage, renderIcons, fallbackIcon) => {
        // TODO: TINY-3036 Work out a better way of dealing with custom icons
        const iconSpec = { tag: 'div', classes: [iconClass] };
        const renderIcon = (iconName) => render$4(iconName, iconSpec, providersBackstage.icons, fallbackIcon);
        const renderEmptyIcon = () => Optional.some({ dom: iconSpec });
        // Note: renderIcons indicates if any icons are present in the menu - if false then the icon column will not be present for the whole menu
        const leftIcon = renderIcons ? info.iconContent.map(renderIcon).orThunk(renderEmptyIcon) : Optional.none();
        // TINY-3345: Dedicated columns for icon and checkmark if applicable
        const checkmark = info.checkMark;
        // Style items and autocompleter both have meta. Need to branch on style
        // This could probably be more stable...
        const textRender = Optional.from(info.meta).fold(() => renderText, (meta) => has$2(meta, 'style') ? curry(renderStyledText, meta.style) : renderText);
        const content = info.htmlContent.fold(() => info.textContent.map(textRender), (html) => Optional.some(renderHtml(html, [textClass])));
        const menuItem = {
            dom: renderItemDomStructure(info.ariaLabel, []),
            optComponents: [
                leftIcon,
                content,
                info.shortcutContent.map(renderShortcut),
                checkmark,
                info.caret,
                info.labelContent.map(createLabel)
            ]
        };
        return menuItem;
    };
    const renderImgItemStructure = (info) => {
        const menuItem = {
            dom: renderItemDomStructure(info.ariaLabel, [imageSelectorClasll]),
            optComponents: [
                Optional.some(render$3(info.iconContent.getOrDie(), { tag: 'div', classes: [imageClass], checkMark: info.checkMark })),
                info.labelContent.map(createLabel)
            ]
        };
        return menuItem;
    };
    // TODO: Maybe need aria-label
    const renderItemStructure = (info, providersBackstage, renderIcons, fallbackIcon = Optional.none()) => {
        if (info.presets === 'color') {
            return renderColorStructure(info, providersBackstage, fallbackIcon);
        }
        else if (info.presets === 'img') {
            return renderImgItemStructure(info);
        }
        else {
            return renderNormalItemStructure(info, providersBackstage, renderIcons, fallbackIcon);
        }
    };

    // Use meta to pass through special information about the tooltip
    // (yes this is horrible but it is not yet public API)
    const tooltipBehaviour = (meta, sharedBackstage, tooltipText) => get$h(meta, 'tooltipWorker')
        .map((tooltipWorker) => [
        Tooltipping.config({
            lazySink: sharedBackstage.getSink,
            tooltipDom: {
                tag: 'div',
                classes: ['tox-tooltip-worker-container']
            },
            tooltipComponents: [],
            anchor: (comp) => ({
                type: 'submenu',
                item: comp,
                overrides: {
                    // NOTE: this avoids it setting overflow and max-height.
                    maxHeightFunction: expandable$1
                }
            }),
            mode: 'follow-highlight',
            onShow: (component, _tooltip) => {
                tooltipWorker((elm) => {
                    Tooltipping.setComponents(component, [
                        external({ element: SugarElement.fromDom(elm) })
                    ]);
                });
            }
        })
    ])
        .getOrThunk(() => {
        return tooltipText.map((text) => [
            Tooltipping.config({
                ...sharedBackstage.providers.tooltips.getConfig({
                    tooltipText: text
                }),
                mode: 'follow-highlight'
            })
        ]).getOr([]);
    });
    const encodeText = (text) => global$9.DOM.encode(text);
    const replaceText = (text, matchText) => {
        const translated = global$6.translate(text);
        const encoded = encodeText(translated);
        if (matchText.length > 0) {
            const escapedMatchRegex = new RegExp(escape(matchText), 'gi');
            return encoded.replace(escapedMatchRegex, (match) => `<span class="tox-autocompleter-highlight">${match}</span>`);
        }
        else {
            return encoded;
        }
    };
    const renderAutocompleteItem = (spec, matchText, useText, presets, onItemValueHandler, itemResponse, sharedBackstage, renderIcons = true) => {
        const structure = renderItemStructure({
            presets,
            textContent: Optional.none(),
            htmlContent: useText ? spec.text.map((text) => replaceText(text, matchText)) : Optional.none(),
            ariaLabel: spec.text,
            labelContent: Optional.none(),
            iconContent: spec.icon,
            shortcutContent: Optional.none(),
            checkMark: Optional.none(),
            caret: Optional.none(),
            value: spec.value
        }, sharedBackstage.providers, renderIcons, spec.icon);
        const tooltipString = spec.text.filter((text) => !useText && text !== '');
        return renderCommonItem({
            context: 'mode:design',
            data: buildData(spec),
            enabled: spec.enabled,
            getApi: constant$1({}),
            onAction: (_api) => onItemValueHandler(spec.value, spec.meta),
            onSetup: constant$1(noop),
            triggersSubmenu: false,
            itemBehaviours: tooltipBehaviour(spec, sharedBackstage, tooltipString)
        }, structure, itemResponse, sharedBackstage.providers);
    };

    const render$2 = (items, extras) => map$2(items, (item) => {
        switch (item.type) {
            case 'cardcontainer':
                return renderContainer(item, render$2(item.items, extras));
            case 'cardimage':
                return renderImage(item.src, item.classes, item.alt);
            case 'cardtext':
                // Only highlight targeted text components
                const shouldHighlight = item.name.exists((name) => contains$2(extras.cardText.highlightOn, name));
                const matchText = shouldHighlight ? Optional.from(extras.cardText.matchText).getOr('') : '';
                return renderHtml(replaceText(item.text, matchText), item.classes);
        }
    });
    const renderCardMenuItem = (spec, itemResponse, sharedBackstage, extras) => {
        const getApi = (component) => ({
            isEnabled: () => !Disabling.isDisabled(component),
            setEnabled: (state) => {
                Disabling.set(component, !state);
                // Disable sub components
                each$1(descendants(component.element, '*'), (elm) => {
                    component.getSystem().getByDom(elm).each((comp) => {
                        if (comp.hasConfigured(Disabling)) {
                            Disabling.set(comp, !state);
                        }
                    });
                });
            }
        });
        const structure = {
            dom: renderItemDomStructure(spec.label, []),
            optComponents: [
                Optional.some({
                    dom: {
                        tag: 'div',
                        classes: [containerClass, containerRowClass]
                    },
                    components: render$2(spec.items, extras)
                })
            ]
        };
        return renderCommonItem({
            context: 'mode:design',
            data: buildData({ text: Optional.none(), ...spec }),
            enabled: spec.enabled,
            getApi,
            onAction: spec.onAction,
            onSetup: spec.onSetup,
            triggersSubmenu: false,
            itemBehaviours: Optional.from(extras.itemBehaviours).getOr([])
        }, structure, itemResponse, sharedBackstage.providers);
    };

    const renderChoiceItem = (spec, useText, presets, onItemValueHandler, isSelected, itemResponse, providersBackstage, renderIcons = true) => {
        const getApi = (component) => ({
            setActive: (state) => {
                Toggling.set(component, state);
            },
            isActive: () => Toggling.isOn(component),
            isEnabled: () => !Disabling.isDisabled(component),
            setEnabled: (state) => Disabling.set(component, !state)
        });
        const structure = renderItemStructure({
            presets,
            textContent: useText ? spec.text : Optional.none(),
            htmlContent: Optional.none(),
            labelContent: spec.label,
            ariaLabel: spec.text,
            iconContent: spec.icon,
            shortcutContent: useText ? spec.shortcut : Optional.none(),
            // useText essentially says that we have one column. In one column lists, we should show a tick
            // The tick is controlled by the tickedClass (via css). It is always present
            // but is hidden unless the tickedClass is present.
            checkMark: useText ? Optional.some(renderCheckmark(providersBackstage.icons)) : Optional.none(),
            caret: Optional.none(),
            value: spec.value
        }, providersBackstage, renderIcons);
        const optTooltipping = spec.text
            .filter(constant$1(!useText))
            .map((t) => Tooltipping.config(providersBackstage.tooltips.getConfig({
            tooltipText: providersBackstage.translate(t)
        })));
        return deepMerge(renderCommonItem({
            context: spec.context,
            data: buildData(spec),
            enabled: spec.enabled,
            getApi,
            onAction: (_api) => onItemValueHandler(spec.value),
            onSetup: (api) => {
                api.setActive(isSelected);
                return noop;
            },
            triggersSubmenu: false,
            itemBehaviours: [
                ...optTooltipping.toArray()
            ]
        }, structure, itemResponse, providersBackstage), {
            toggling: {
                toggleClass: tickedClass,
                toggleOnExecute: false,
                selected: spec.active,
                exclusive: true
            }
        });
    };

    const hexColour = (value) => ({
        value: normalizeHex(value)
    });
    const shorthandRegex = /^#?([a-f\d])([a-f\d])([a-f\d])$/i;
    const longformRegex = /^#?([a-f\d]{2})([a-f\d]{2})([a-f\d]{2})$/i;
    const isHexString = (hex) => shorthandRegex.test(hex) || longformRegex.test(hex);
    const normalizeHex = (hex) => removeLeading(hex, '#').toUpperCase();
    const fromString$1 = (hex) => isHexString(hex) ? Optional.some({ value: normalizeHex(hex) }) : Optional.none();
    // Expand shorthand form (e.g. "03F") to full form (e.g. "0033FF")
    const getLongForm = (hex) => {
        const hexString = hex.value.replace(shorthandRegex, (m, r, g, b) => r + r + g + g + b + b);
        return { value: hexString };
    };
    const extractValues = (hex) => {
        const longForm = getLongForm(hex);
        const splitForm = longformRegex.exec(longForm.value);
        return splitForm === null ? ['FFFFFF', 'FF', 'FF', 'FF'] : splitForm;
    };
    const toHex = (component) => {
        const hex = component.toString(16);
        return (hex.length === 1 ? '0' + hex : hex).toUpperCase();
    };
    const fromRgba = (rgbaColour) => {
        const value = toHex(rgbaColour.red) + toHex(rgbaColour.green) + toHex(rgbaColour.blue);
        return hexColour(value);
    };

    const hsvColour = (hue, saturation, value) => ({
        hue,
        saturation,
        value
    });
    const fromRgb = (rgbaColour) => {
        let h = 0;
        let s = 0;
        let v = 0;
        const r = rgbaColour.red / 255;
        const g = rgbaColour.green / 255;
        const b = rgbaColour.blue / 255;
        const minRGB = Math.min(r, Math.min(g, b));
        const maxRGB = Math.max(r, Math.max(g, b));
        if (minRGB === maxRGB) {
            v = minRGB;
            return hsvColour(0, 0, v * 100);
        }
        /* eslint no-nested-ternary:0 */
        const d = (r === minRGB) ? g - b : ((b === minRGB) ? r - g : b - r);
        h = (r === minRGB) ? 3 : ((b === minRGB) ? 1 : 5);
        h = 60 * (h - d / (maxRGB - minRGB));
        s = (maxRGB - minRGB) / maxRGB;
        v = maxRGB;
        return hsvColour(Math.round(h), Math.round(s * 100), Math.round(v * 100));
    };

    const min = Math.min;
    const max = Math.max;
    const round$1 = Math.round;
    const rgbRegex = /^\s*rgb\s*\(\s*(\d+)\s*[,\s]\s*(\d+)\s*[,\s]\s*(\d+)\s*\)\s*$/i;
    // This regex will match rgba(0, 0, 0, 0.5) or rgba(0, 0, 0, 50%) , or without commas
    const rgbaRegex = /^\s*rgba\s*\(\s*(\d+)\s*[,\s]\s*(\d+)\s*[,\s]\s*(\d+)\s*[,\s]\s*((?:\d?\.\d+|\d+)%?)\s*\)\s*$/i;
    const rgbaColour = (red, green, blue, alpha) => ({
        red,
        green,
        blue,
        alpha
    });
    const isRgbaComponent = (value) => {
        const num = parseInt(value, 10);
        return num.toString() === value && num >= 0 && num <= 255;
    };
    const fromHsv = (hsv) => {
        let r;
        let g;
        let b;
        const hue = (hsv.hue || 0) % 360;
        let saturation = hsv.saturation / 100;
        let brightness = hsv.value / 100;
        saturation = max(0, min(saturation, 1));
        brightness = max(0, min(brightness, 1));
        if (saturation === 0) {
            r = g = b = round$1(255 * brightness);
            return rgbaColour(r, g, b, 1);
        }
        const side = hue / 60;
        const chroma = brightness * saturation;
        const x = chroma * (1 - Math.abs(side % 2 - 1));
        const match = brightness - chroma;
        switch (Math.floor(side)) {
            case 0:
                r = chroma;
                g = x;
                b = 0;
                break;
            case 1:
                r = x;
                g = chroma;
                b = 0;
                break;
            case 2:
                r = 0;
                g = chroma;
                b = x;
                break;
            case 3:
                r = 0;
                g = x;
                b = chroma;
                break;
            case 4:
                r = x;
                g = 0;
                b = chroma;
                break;
            case 5:
                r = chroma;
                g = 0;
                b = x;
                break;
            default:
                r = g = b = 0;
        }
        r = round$1(255 * (r + match));
        g = round$1(255 * (g + match));
        b = round$1(255 * (b + match));
        return rgbaColour(r, g, b, 1);
    };
    // Temporarily using: https://stackoverflow.com/questions/5623838/rgb-to-hex-and-hex-to-rgb
    const fromHex = (hexColour) => {
        const result = extractValues(hexColour);
        const red = parseInt(result[1], 16);
        const green = parseInt(result[2], 16);
        const blue = parseInt(result[3], 16);
        return rgbaColour(red, green, blue, 1);
    };
    const fromStringValues = (red, green, blue, alpha) => {
        const r = parseInt(red, 10);
        const g = parseInt(green, 10);
        const b = parseInt(blue, 10);
        const a = parseFloat(alpha);
        return rgbaColour(r, g, b, a);
    };
    const fromString = (rgbaString) => {
        const rgbMatch = rgbRegex.exec(rgbaString);
        if (rgbMatch !== null) {
            return Optional.some(fromStringValues(rgbMatch[1], rgbMatch[2], rgbMatch[3], '1'));
        }
        const rgbaMatch = rgbaRegex.exec(rgbaString);
        if (rgbaMatch !== null) {
            return Optional.some(fromStringValues(rgbaMatch[1], rgbaMatch[2], rgbaMatch[3], rgbaMatch[4]));
        }
        return Optional.none();
    };
    const toString = (rgba) => `rgba(${rgba.red},${rgba.green},${rgba.blue},${rgba.alpha})`;
    const red = rgbaColour(255, 0, 0, 1);

    const hexToHsv = (hex) => fromRgb(fromHex(hex));
    const hsvToHex = (hsv) => fromRgba(fromHsv(hsv));
    const anyToHex = (color) => fromString$1(color)
        .orThunk(() => fromString(color).map(fromRgba))
        .getOrThunk(() => {
        // Not dealing with Hex or RGBA so use a canvas to parse the color
        const canvas = document.createElement('canvas');
        canvas.height = 1;
        canvas.width = 1;
        const canvasContext = canvas.getContext('2d');
        // all valid colors after this point
        canvasContext.clearRect(0, 0, canvas.width, canvas.height);
        // invalid colors will be shown as white - the first assignment will pass and the second may be ignored
        canvasContext.fillStyle = '#FFFFFF';
        canvasContext.fillStyle = color;
        canvasContext.fillRect(0, 0, 1, 1);
        const rgba = canvasContext.getImageData(0, 0, 1, 1).data;
        const r = rgba[0];
        const g = rgba[1];
        const b = rgba[2];
        const a = rgba[3];
        return fromRgba(rgbaColour(r, g, b, a));
    });

    const fireSkinLoaded$1 = (editor) => {
        editor.dispatch('SkinLoaded');
    };
    const fireSkinLoadError$1 = (editor, error) => {
        editor.dispatch('SkinLoadError', error);
    };
    const fireResizeEditor = (editor) => {
        editor.dispatch('ResizeEditor');
    };
    const fireResizeContent = (editor, e) => {
        editor.dispatch('ResizeContent', e);
    };
    const fireScrollContent = (editor, e) => {
        editor.dispatch('ScrollContent', e);
    };
    const fireTextColorChange = (editor, data) => {
        editor.dispatch('TextColorChange', data);
    };
    const fireAfterProgressState = (editor, state) => {
        editor.dispatch('AfterProgressState', { state });
    };
    const fireResolveName = (editor, node) => editor.dispatch('ResolveName', {
        name: node.nodeName.toLowerCase(),
        target: node
    });
    const fireToggleToolbarDrawer = (editor, state) => {
        editor.dispatch('ToggleToolbarDrawer', { state });
    };
    const fireStylesTextUpdate = (editor, data) => {
        editor.dispatch('StylesTextUpdate', data);
    };
    const fireAlignTextUpdate = (editor, data) => {
        editor.dispatch('AlignTextUpdate', data);
    };
    const fireFontSizeTextUpdate = (editor, data) => {
        editor.dispatch('FontSizeTextUpdate', data);
    };
    const fireFontSizeInputTextUpdate = (editor, data) => {
        editor.dispatch('FontSizeInputTextUpdate', data);
    };
    const fireBlocksTextUpdate = (editor, data) => {
        editor.dispatch('BlocksTextUpdate', data);
    };
    const fireFontFamilyTextUpdate = (editor, data) => {
        editor.dispatch('FontFamilyTextUpdate', data);
    };
    const fireToggleSidebar = (editor) => {
        editor.dispatch('ToggleSidebar');
    };
    const fireToggleView = (editor) => {
        editor.dispatch('ToggleView');
    };
    const fireContextToolbarClose = (editor) => {
        editor.dispatch('ContextToolbarClose');
    };
    const fireContextFormSlideBack = (editor) => {
        editor.dispatch('ContextFormSlideBack');
    };

    const composeUnbinders = (f, g) => () => {
        f();
        g();
    };
    const onSetupEditableToggle = (editor, enabledPredicate = always) => onSetupEvent(editor, 'NodeChange', (api) => {
        api.setEnabled(editor.selection.isEditable() && enabledPredicate());
    });
    const onSetupFormatToggle = (editor, name) => (api) => {
        const boundFormatChangeCallback = unbindable();
        const init = () => {
            api.setActive(editor.formatter.match(name));
            const binding = editor.formatter.formatChanged(name, api.setActive);
            boundFormatChangeCallback.set(binding);
        };
        // The editor may or may not have been setup yet, so check for that
        editor.initialized ? init() : editor.once('init', init);
        return () => {
            editor.off('init', init);
            boundFormatChangeCallback.clear();
        };
    };
    const onSetupStateToggle = (editor, name) => (api) => {
        const unbindEditableToogle = onSetupEditableToggle(editor)(api);
        const unbindFormatToggle = onSetupFormatToggle(editor, name)(api);
        return () => {
            unbindEditableToogle();
            unbindFormatToggle();
        };
    };
    const onSetupEvent = (editor, event, f) => (api) => {
        const handleEvent = () => f(api);
        const init = () => {
            f(api);
            editor.on(event, handleEvent);
        };
        // The editor may or may not have been setup yet, so check for that
        editor.initialized ? init() : editor.once('init', init);
        return () => {
            editor.off('init', init);
            editor.off(event, handleEvent);
        };
    };
    const onActionToggleFormat$1 = (editor) => (rawItem) => () => {
        editor.undoManager.transact(() => {
            editor.focus();
            editor.execCommand('mceToggleFormat', false, rawItem.format);
        });
    };
    const onActionExecCommand = (editor, command) => () => editor.execCommand(command);

    var global$5 = tinymce.util.Tools.resolve('tinymce.util.LocalStorage');

    const cacheStorage = {};
    const ColorCache = (storageId, max = 10) => {
        const storageString = global$5.getItem(storageId);
        const localstorage = isString(storageString) ? JSON.parse(storageString) : [];
        const prune = (list) => {
            // When the localStorage cache is too big,
            // remove the difference from the tail (head is fresh, tail is stale!)
            const diff = max - list.length;
            return (diff < 0) ? list.slice(0, max) : list;
        };
        const cache = prune(localstorage);
        const add = (key) => {
            // Remove duplicates first.
            indexOf(cache, key).each(remove);
            cache.unshift(key);
            // When max size is exceeded, the oldest colors will be removed
            if (cache.length > max) {
                cache.pop();
            }
            global$5.setItem(storageId, JSON.stringify(cache));
        };
        const remove = (idx) => {
            cache.splice(idx, 1);
        };
        const state = () => cache.slice(0);
        return {
            add,
            state
        };
    };
    const getCacheForId = (id) => get$h(cacheStorage, id).getOrThunk(() => {
        const storageId = `tinymce-custom-colors-${id}`;
        const currentData = global$5.getItem(storageId);
        if (isNullable(currentData)) {
            const legacyDefault = global$5.getItem('tinymce-custom-colors');
            global$5.setItem(storageId, isNonNullable(legacyDefault) ? legacyDefault : '[]');
        }
        const storage = ColorCache(storageId, 10);
        cacheStorage[id] = storage;
        return storage;
    });
    const getCurrentColors = (id) => map$2(getCacheForId(id).state(), (color) => ({
        type: 'choiceitem',
        text: color,
        icon: 'checkmark',
        value: color
    }));
    const addColor = (id, color) => {
        getCacheForId(id).add(color);
    };

    const foregroundId = 'forecolor';
    const backgroundId = 'hilitecolor';
    const fallbackCols = 5;
    const mapColors = (colorMap) => mapColorsRaw(colorMap.map((color, index) => {
        if (index % 2 === 0) {
            return '#' + anyToHex(color).value;
        }
        return color;
    }));
    const mapColorsRaw = (colorMap) => {
        const colors = [];
        for (let i = 0; i < colorMap.length; i += 2) {
            colors.push({
                text: colorMap[i + 1],
                value: colorMap[i],
                icon: 'checkmark',
                type: 'choiceitem'
            });
        }
        return colors;
    };
    const option$1 = (name) => (editor) => editor.options.get(name);
    const fallbackColor = '#000000';
    const register$e = (editor) => {
        const registerOption = editor.options.register;
        const colorProcessor = (value) => {
            if (isArrayOf(value, isString)) {
                return { value: mapColors(value), valid: true };
            }
            else {
                return { valid: false, message: 'Must be an array of strings.' };
            }
        };
        const colorProcessorRaw = (value) => {
            if (isArrayOf(value, isString)) {
                return { value: mapColorsRaw(value), valid: true };
            }
            else {
                return { valid: false, message: 'Must be an array of strings.' };
            }
        };
        const colorColsProcessor = (value) => {
            if (isNumber(value) && value > 0) {
                return { value, valid: true };
            }
            else {
                return { valid: false, message: 'Must be a positive number.' };
            }
        };
        registerOption('color_map', {
            processor: colorProcessor,
            default: [
                '#BFEDD2', 'Light Green',
                '#FBEEB8', 'Light Yellow',
                '#F8CAC6', 'Light Red',
                '#ECCAFA', 'Light Purple',
                '#C2E0F4', 'Light Blue',
                '#2DC26B', 'Green',
                '#F1C40F', 'Yellow',
                '#E03E2D', 'Red',
                '#B96AD9', 'Purple',
                '#3598DB', 'Blue',
                '#169179', 'Dark Turquoise',
                '#E67E23', 'Orange',
                '#BA372A', 'Dark Red',
                '#843FA1', 'Dark Purple',
                '#236FA1', 'Dark Blue',
                '#ECF0F1', 'Light Gray',
                '#CED4D9', 'Medium Gray',
                '#95A5A6', 'Gray',
                '#7E8C8D', 'Dark Gray',
                '#34495E', 'Navy Blue',
                '#000000', 'Black',
                '#ffffff', 'White'
            ]
        });
        registerOption('color_map_raw', {
            processor: colorProcessorRaw,
        });
        registerOption('color_map_background', {
            processor: colorProcessor
        });
        registerOption('color_map_foreground', {
            processor: colorProcessor
        });
        registerOption('color_cols', {
            processor: colorColsProcessor,
            default: calcCols(editor)
        });
        registerOption('color_cols_foreground', {
            processor: colorColsProcessor,
            default: defaultCols(editor, foregroundId)
        });
        registerOption('color_cols_background', {
            processor: colorColsProcessor,
            default: defaultCols(editor, backgroundId)
        });
        registerOption('custom_colors', {
            processor: 'boolean',
            default: true
        });
        registerOption('color_default_foreground', {
            processor: 'string',
            default: fallbackColor
        });
        registerOption('color_default_background', {
            processor: 'string',
            default: fallbackColor
        });
    };
    const getColors$2 = (editor, id) => {
        if (id === foregroundId && editor.options.isSet('color_map_foreground')) {
            return option$1('color_map_foreground')(editor);
        }
        else if (id === backgroundId && editor.options.isSet('color_map_background')) {
            return option$1('color_map_background')(editor);
        }
        else if (editor.options.isSet('color_map_raw')) {
            return option$1('color_map_raw')(editor);
        }
        else {
            return option$1('color_map')(editor);
        }
    };
    const calcCols = (editor, id = 'default') => Math.max(fallbackCols, Math.ceil(Math.sqrt(getColors$2(editor, id).length)));
    const defaultCols = (editor, id) => {
        const defaultCols = option$1('color_cols')(editor);
        const calculatedCols = calcCols(editor, id);
        if (defaultCols === calcCols(editor)) {
            return calculatedCols;
        }
        else {
            return defaultCols;
        }
    };
    const getColorCols$1 = (editor, id = 'default') => {
        const getCols = () => {
            if (id === foregroundId) {
                return option$1('color_cols_foreground')(editor);
            }
            else if (id === backgroundId) {
                return option$1('color_cols_background')(editor);
            }
            else {
                return option$1('color_cols')(editor);
            }
        };
        return Math.round(getCols());
    };
    const hasCustomColors$1 = option$1('custom_colors');
    const getDefaultForegroundColor = option$1('color_default_foreground');
    const getDefaultBackgroundColor = option$1('color_default_background');

    const defaultBackgroundColor = 'rgba(0, 0, 0, 0)';
    const isValidBackgroundColor = (value) => fromString(value).exists((c) => c.alpha !== 0);
    // Climb up the tree to find the value of the background until finding a non-transparent value or defaulting.
    const getClosestCssBackgroundColorValue = (scope) => {
        return closest(scope, (node) => {
            if (isElement$1(node)) {
                const color = get$e(node, 'background-color');
                return someIf(isValidBackgroundColor(color), color);
            }
            else {
                return Optional.none();
            }
        }).getOr(defaultBackgroundColor);
    };
    const getCurrentColor = (editor, format) => {
        const node = SugarElement.fromDom(editor.selection.getStart());
        const cssRgbValue = format === 'hilitecolor'
            ? getClosestCssBackgroundColorValue(node)
            : get$e(node, 'color');
        return fromString(cssRgbValue).map((rgba) => '#' + fromRgba(rgba).value);
    };
    const applyFormat = (editor, format, value) => {
        editor.undoManager.transact(() => {
            editor.focus();
            editor.formatter.apply(format, { value });
            editor.nodeChanged();
        });
    };
    const removeFormat = (editor, format) => {
        editor.undoManager.transact(() => {
            editor.focus();
            editor.formatter.remove(format, { value: null }, undefined, true);
            editor.nodeChanged();
        });
    };
    const registerCommands = (editor) => {
        editor.addCommand('mceApplyTextcolor', (format, value) => {
            applyFormat(editor, format, value);
        });
        editor.addCommand('mceRemoveTextcolor', (format) => {
            removeFormat(editor, format);
        });
    };
    const getAdditionalColors = (hasCustom) => {
        const type = 'choiceitem';
        const remove = {
            type,
            text: 'Remove color',
            icon: 'color-swatch-remove-color',
            value: 'remove'
        };
        const custom = {
            type,
            text: 'Custom color',
            icon: 'color-picker',
            value: 'custom'
        };
        return hasCustom ? [
            remove,
            custom
        ] : [remove];
    };
    const applyColor = (editor, format, value, onChoice) => {
        if (value === 'custom') {
            const dialog = colorPickerDialog(editor);
            dialog((colorOpt) => {
                colorOpt.each((color) => {
                    addColor(format, color);
                    editor.execCommand('mceApplyTextcolor', format, color);
                    onChoice(color);
                });
            }, getCurrentColor(editor, format).getOr(fallbackColor));
        }
        else if (value === 'remove') {
            onChoice('');
            editor.execCommand('mceRemoveTextcolor', format);
        }
        else {
            onChoice(value);
            editor.execCommand('mceApplyTextcolor', format, value);
        }
    };
    const getColors$1 = (colors, id, hasCustom) => colors.concat(getCurrentColors(id).concat(getAdditionalColors(hasCustom)));
    const getFetch$1 = (colors, id, hasCustom) => (callback) => {
        callback(getColors$1(colors, id, hasCustom));
    };
    const setIconColor = (splitButtonApi, name, newColor) => {
        const id = name === 'forecolor' ? 'tox-icon-text-color__color' : 'tox-icon-highlight-bg-color__color';
        splitButtonApi.setIconFill(id, newColor);
    };
    const setTooltip = (buttonApi, tooltip) => {
        buttonApi.setTooltip(tooltip);
    };
    const select$1 = (editor, format) => (value) => {
        const optCurrentHex = getCurrentColor(editor, format);
        return is$1(optCurrentHex, value.toUpperCase());
    };
    // Selecting `Remove Color` would set the lastColor to ''
    const getToolTipText = (editor, format, lastColor) => {
        if (isEmpty(lastColor)) {
            return format === 'forecolor' ? 'Text color' : 'Background color';
        }
        const tooltipPrefix = format === 'forecolor' ? 'Text color {0}' : 'Background color {0}';
        const colors = getColors$1(getColors$2(editor, format), format, false);
        const colorText = find$5(colors, (c) => c.value === lastColor).getOr({ text: '' }).text;
        return editor.translate([tooltipPrefix, editor.translate(colorText)]);
    };
    const registerTextColorButton = (editor, name, format, lastColor) => {
        editor.ui.registry.addSplitButton(name, {
            tooltip: getToolTipText(editor, format, lastColor.get()),
            chevronTooltip: name === 'forecolor' ? 'Text color menu' : 'Background color menu',
            presets: 'color',
            icon: name === 'forecolor' ? 'text-color' : 'highlight-bg-color',
            select: select$1(editor, format),
            columns: getColorCols$1(editor, format),
            fetch: getFetch$1(getColors$2(editor, format), format, hasCustomColors$1(editor)),
            onAction: (_splitButtonApi) => {
                applyColor(editor, format, lastColor.get(), noop);
            },
            onItemAction: (_splitButtonApi, value) => {
                applyColor(editor, format, value, (newColor) => {
                    lastColor.set(newColor);
                    fireTextColorChange(editor, {
                        name,
                        color: newColor
                    });
                });
            },
            onSetup: (splitButtonApi) => {
                setIconColor(splitButtonApi, name, lastColor.get());
                const handler = (e) => {
                    if (e.name === name) {
                        setIconColor(splitButtonApi, e.name, e.color);
                        setTooltip(splitButtonApi, getToolTipText(editor, format, e.color));
                    }
                };
                editor.on('TextColorChange', handler);
                return composeUnbinders(onSetupEditableToggle(editor)(splitButtonApi), () => {
                    editor.off('TextColorChange', handler);
                });
            }
        });
    };
    const registerTextColorMenuItem = (editor, name, format, text, lastColor) => {
        editor.ui.registry.addNestedMenuItem(name, {
            text,
            icon: name === 'forecolor' ? 'text-color' : 'highlight-bg-color',
            onSetup: (api) => {
                setTooltip(api, getToolTipText(editor, format, lastColor.get()));
                setIconColor(api, name, lastColor.get());
                return onSetupEditableToggle(editor)(api);
            },
            getSubmenuItems: () => [
                {
                    type: 'fancymenuitem',
                    fancytype: 'colorswatch',
                    select: select$1(editor, format),
                    initData: {
                        storageKey: format,
                    },
                    onAction: (data) => {
                        applyColor(editor, format, data.value, (newColor) => {
                            lastColor.set(newColor);
                            fireTextColorChange(editor, {
                                name,
                                color: newColor
                            });
                        });
                    },
                }
            ]
        });
    };
    const colorPickerDialog = (editor) => (callback, value) => {
        let isValid = false;
        const onSubmit = (api) => {
            const data = api.getData();
            const hex = data.colorpicker;
            if (isValid) {
                callback(Optional.from(hex));
                api.close();
            }
            else {
                editor.windowManager.alert(editor.translate(['Invalid hex color code: {0}', hex]));
            }
        };
        const onAction = (_api, details) => {
            if (details.name === 'hex-valid') {
                isValid = details.value;
            }
        };
        const initialData = {
            colorpicker: value
        };
        editor.windowManager.open({
            title: 'Color Picker',
            size: 'normal',
            body: {
                type: 'panel',
                items: [
                    {
                        type: 'colorpicker',
                        name: 'colorpicker',
                        label: 'Color'
                    }
                ]
            },
            buttons: [
                {
                    type: 'cancel',
                    name: 'cancel',
                    text: 'Cancel'
                },
                {
                    type: 'submit',
                    name: 'save',
                    text: 'Save',
                    primary: true
                }
            ],
            initialData,
            onAction,
            onSubmit,
            onClose: noop,
            onCancel: () => {
                callback(Optional.none());
            }
        });
    };
    const register$d = (editor) => {
        registerCommands(editor);
        const fallbackColorForeground = getDefaultForegroundColor(editor);
        const fallbackColorBackground = getDefaultBackgroundColor(editor);
        const lastForeColor = Cell(fallbackColorForeground);
        const lastBackColor = Cell(fallbackColorBackground);
        registerTextColorButton(editor, 'forecolor', 'forecolor', lastForeColor);
        registerTextColorButton(editor, 'backcolor', 'hilitecolor', lastBackColor);
        registerTextColorMenuItem(editor, 'forecolor', 'forecolor', 'Text color', lastForeColor);
        registerTextColorMenuItem(editor, 'backcolor', 'hilitecolor', 'Background color', lastBackColor);
    };

    const renderImgItem = (spec, onItemValueHandler, isSelected, itemResponse, providersBackstage) => {
        const getApi = (component) => ({
            setActive: (state) => {
                Toggling.set(component, state);
            },
            isActive: () => Toggling.isOn(component),
            isEnabled: () => !Disabling.isDisabled(component),
            setEnabled: (state) => Disabling.set(component, !state)
        });
        const structure = renderItemStructure({
            presets: 'img',
            textContent: Optional.none(),
            htmlContent: Optional.none(),
            ariaLabel: spec.tooltip,
            iconContent: Optional.some(spec.url),
            labelContent: spec.label,
            shortcutContent: Optional.none(),
            checkMark: Optional.some(renderCheckmark(providersBackstage.icons)),
            caret: Optional.none(),
            value: spec.value
        }, providersBackstage, true);
        const optTooltipping = spec.tooltip
            .map((t) => Tooltipping.config(providersBackstage.tooltips.getConfig({
            tooltipText: providersBackstage.translate(t)
        })));
        return deepMerge(renderCommonItem({
            context: spec.context,
            data: buildData(spec),
            enabled: spec.enabled,
            getApi,
            onAction: (api) => {
                onItemValueHandler(spec.value);
                api.setActive(true);
            },
            onSetup: (api) => {
                api.setActive(isSelected);
                return noop;
            },
            triggersSubmenu: false,
            itemBehaviours: [
                ...optTooltipping.toArray()
            ]
        }, structure, itemResponse, providersBackstage), {
            toggling: {
                toggleClass: tickedClass,
                toggleOnExecute: false,
                selected: spec.active,
                exclusive: true
            }
        });
    };

    const createPartialChoiceMenu = (value, items, onItemValueHandler, columns, presets, itemResponse, select, providersBackstage) => {
        const hasIcons = menuHasIcons(items);
        const presetItemTypes = presets !== 'color' ? 'normal' : 'color';
        const alloyItems = createChoiceItems(items, onItemValueHandler, columns, presetItemTypes, itemResponse, select, providersBackstage);
        const menuLayout = {
            menuType: presets
        };
        return createPartialMenuWithAlloyItems(value, hasIcons, alloyItems, columns, menuLayout);
    };
    const createChoiceItems = (items, onItemValueHandler, columns, itemPresets, itemResponse, select, providersBackstage) => cat(map$2(items, (item) => {
        if (item.type === 'choiceitem') {
            return createChoiceMenuItem(item).fold(handleError, (d) => Optional.some(renderChoiceItem(d, columns === 1, itemPresets, onItemValueHandler, select(d.value), itemResponse, providersBackstage, menuHasIcons(items))));
        }
        else if (item.type === 'imageitem') {
            return createImageMenuItem(item).fold(handleError, (d) => Optional.some(renderImgItem(d, onItemValueHandler, select(d.value), itemResponse, providersBackstage)));
        }
        else if (item.type === 'resetimage') {
            return createResetImageItem(item).fold(handleError, (d) => Optional.some(renderChoiceItem(({
                ...d,
                type: 'choiceitem',
                text: d.tooltip,
                icon: Optional.some(d.icon),
                label: Optional.some(d.label),
            }), columns === 1, itemPresets, onItemValueHandler, select(d.value), itemResponse, providersBackstage, menuHasIcons(items))));
        }
        else {
            return Optional.none();
        }
    }));

    const deriveMenuMovement = (columns, presets) => {
        const menuMarkers = markers(presets);
        if (columns === 1) {
            return { mode: 'menu', moveOnTab: true };
        }
        else if (columns === 'auto') {
            return {
                mode: 'grid',
                selector: '.' + menuMarkers.item,
                initSize: {
                    numColumns: 1,
                    numRows: 1
                }
            };
        }
        else {
            const rowClass = {
                color: 'tox-swatches__row',
                imageselector: 'tox-image-selector__row',
                listpreview: 'tox-collection__group',
                normal: 'tox-collection__group'
            }[presets];
            return {
                mode: 'matrix',
                rowSelector: '.' + rowClass,
                previousSelector: (menu) => {
                    // We only want the navigation to start on the selected item if we are in color-mode (The colorswatch)
                    return presets === 'color'
                        ? descendant(menu.element, '[aria-checked=true]')
                        : Optional.none();
                }
            };
        }
    };
    const deriveCollectionMovement = (columns, presets) => {
        if (columns === 1) {
            return {
                mode: 'menu',
                moveOnTab: false,
                selector: '.tox-collection__item'
            };
        }
        else if (columns === 'auto') {
            return {
                mode: 'flatgrid',
                selector: '.' + 'tox-collection__item',
                initSize: {
                    numColumns: 1,
                    numRows: 1
                }
            };
        }
        else {
            return {
                mode: 'matrix',
                selectors: {
                    row: presets === 'color' ? '.tox-swatches__row' : '.tox-collection__group',
                    cell: presets === 'color' ? `.${colorClass}` : `.${selectableClass}`
                }
            };
        }
    };

    const renderColorSwatchItem = (spec, backstage) => {
        const items = getColorItems(spec, backstage);
        const columns = backstage.colorinput.getColorCols(spec.initData.storageKey);
        const presets = 'color';
        const menuSpec = createPartialChoiceMenu(generate$6('menu-value'), items, (value) => {
            spec.onAction({ value });
        }, columns, presets, ItemResponse$1.CLOSE_ON_EXECUTE, spec.select.getOr(never), backstage.shared.providers);
        const widgetSpec = {
            ...menuSpec,
            markers: markers(presets),
            movement: deriveMenuMovement(columns, presets),
            // TINY-10806: Avoid duplication of ARIA role="menu" in the accessibility tree for Color Swatch menu item.
            showMenuRole: false
        };
        return {
            type: 'widget',
            data: { value: generate$6('widget-id') },
            dom: {
                tag: 'div',
                classes: ['tox-fancymenuitem']
            },
            autofocus: true,
            components: [
                parts$7.widget(Menu.sketch(widgetSpec))
            ]
        };
    };
    const getColorItems = (spec, backstage) => {
        const useCustomColors = spec.initData.allowCustomColors && backstage.colorinput.hasCustomColors();
        return spec.initData.colors.fold(() => getColors$1(backstage.colorinput.getColors(spec.initData.storageKey), spec.initData.storageKey, useCustomColors), (colors) => colors.concat(getAdditionalColors(useCustomColors)));
    };

    const renderImageSelector = (spec, backstage) => {
        const presets = 'imageselector';
        const columns = spec.initData.columns;
        const menuSpec = createPartialChoiceMenu(generate$6('menu-value'), spec.initData.items, (value) => {
            spec.onAction({ value });
        }, columns, presets, ItemResponse$1.CLOSE_ON_EXECUTE, spec.select.getOr(never), backstage.shared.providers);
        const widgetSpec = {
            ...menuSpec,
            markers: markers(presets),
            movement: deriveMenuMovement(columns, presets),
            // TINY-10806: Avoid duplication of ARIA role="menu" in the accessibility tree for Image Selector menu item.
            showMenuRole: false
        };
        return {
            type: 'widget',
            data: { value: generate$6('widget-id') },
            dom: {
                tag: 'div',
                classes: ['tox-fancymenuitem', 'tox-collection--toolbar']
            },
            autofocus: true,
            components: [
                parts$7.widget(Menu.sketch(widgetSpec))
            ]
        };
    };

    const cellOverEvent = generate$6('cell-over');
    const cellExecuteEvent = generate$6('cell-execute');
    const makeAnnouncementText = (backstage) => (row, col) => backstage.shared.providers.translate(['{0} columns, {1} rows', col, row]);
    const makeCell = (row, col, label) => {
        const emitCellOver = (c) => emitWith(c, cellOverEvent, { row, col });
        const emitExecute = (c) => emitWith(c, cellExecuteEvent, { row, col });
        const onClick = (c, se) => {
            se.stop();
            emitExecute(c);
        };
        return build$1({
            dom: {
                tag: 'div',
                attributes: {
                    role: 'button',
                    ['aria-label']: label
                }
            },
            behaviours: derive$1([
                config('insert-table-picker-cell', [
                    run$1(mouseover(), Focusing.focus),
                    run$1(execute$5(), emitExecute),
                    run$1(click(), onClick),
                    run$1(tap(), onClick)
                ]),
                Toggling.config({
                    toggleClass: 'tox-insert-table-picker__selected',
                    toggleOnExecute: false
                }),
                Focusing.config({ onFocus: emitCellOver })
            ])
        });
    };
    const makeCells = (getCellLabel, numRows, numCols) => {
        const cells = [];
        for (let i = 0; i < numRows; i++) {
            const row = [];
            for (let j = 0; j < numCols; j++) {
                const label = getCellLabel(i + 1, j + 1);
                row.push(makeCell(i, j, label));
            }
            cells.push(row);
        }
        return cells;
    };
    const selectCells = (cells, selectedRow, selectedColumn, numRows, numColumns) => {
        for (let i = 0; i < numRows; i++) {
            for (let j = 0; j < numColumns; j++) {
                Toggling.set(cells[i][j], i <= selectedRow && j <= selectedColumn);
            }
        }
    };
    const makeComponents = (cells) => bind$3(cells, (cellRow) => map$2(cellRow, premade));
    const makeLabelText = (row, col) => text$2(`${col}x${row}`);
    const renderInsertTableMenuItem = (spec, backstage) => {
        const numRows = 10;
        const numColumns = 10;
        const getCellLabel = makeAnnouncementText(backstage);
        const cells = makeCells(getCellLabel, numRows, numColumns);
        const emptyLabelText = makeLabelText(0, 0);
        const memLabel = record({
            dom: {
                tag: 'span',
                classes: ['tox-insert-table-picker__label'],
            },
            components: [emptyLabelText],
            behaviours: derive$1([
                Replacing.config({})
            ])
        });
        return {
            type: 'widget',
            data: { value: generate$6('widget-id') },
            dom: {
                tag: 'div',
                classes: ['tox-fancymenuitem']
            },
            autofocus: true,
            components: [parts$7.widget({
                    dom: {
                        tag: 'div',
                        classes: ['tox-insert-table-picker']
                    },
                    components: makeComponents(cells).concat(memLabel.asSpec()),
                    behaviours: derive$1([
                        config('insert-table-picker', [
                            runOnAttached((c) => {
                                // Restore the empty label when opened, otherwise it may still be using an old label from last time it was opened
                                Replacing.set(memLabel.get(c), [emptyLabelText]);
                            }),
                            runWithTarget(cellOverEvent, (c, t, e) => {
                                const { row, col } = e.event;
                                selectCells(cells, row, col, numRows, numColumns);
                                Replacing.set(memLabel.get(c), [makeLabelText(row + 1, col + 1)]);
                            }),
                            runWithTarget(cellExecuteEvent, (c, _, e) => {
                                const { row, col } = e.event;
                                // Close the sandbox before triggering the action
                                emit(c, sandboxClose());
                                spec.onAction({ numRows: row + 1, numColumns: col + 1 });
                            })
                        ]),
                        Keying.config({
                            initSize: {
                                numRows,
                                numColumns
                            },
                            mode: 'flatgrid',
                            selector: '[role="button"]'
                        })
                    ])
                })]
        };
    };

    const fancyMenuItems = {
        inserttable: renderInsertTableMenuItem,
        colorswatch: renderColorSwatchItem,
        imageselect: renderImageSelector
    };
    const renderFancyMenuItem = (spec, backstage) => get$h(fancyMenuItems, spec.fancytype).map((render) => render(spec, backstage));

    // Note, this does not create a valid SketchSpec.
    const renderNestedItem = (spec, itemResponse, providersBackstage, renderIcons = true, downwardsCaret = false) => {
        const caret = downwardsCaret ? renderDownwardsCaret(providersBackstage.icons) : renderSubmenuCaret(providersBackstage.icons);
        const getApi = (component) => ({
            isEnabled: () => !Disabling.isDisabled(component),
            setEnabled: (state) => Disabling.set(component, !state),
            setIconFill: (id, value) => {
                descendant(component.element, `svg path[class="${id}"], rect[class="${id}"]`).each((underlinePath) => {
                    set$9(underlinePath, 'fill', value);
                });
            },
            setTooltip: (tooltip) => {
                const translatedTooltip = providersBackstage.translate(tooltip);
                set$9(component.element, 'aria-label', translatedTooltip);
            }
        });
        const structure = renderItemStructure({
            presets: 'normal',
            iconContent: spec.icon,
            textContent: spec.text,
            htmlContent: Optional.none(),
            ariaLabel: spec.text,
            labelContent: Optional.none(),
            caret: Optional.some(caret),
            checkMark: Optional.none(),
            shortcutContent: spec.shortcut
        }, providersBackstage, renderIcons);
        return renderCommonItem({
            context: spec.context,
            data: buildData(spec),
            getApi,
            enabled: spec.enabled,
            onAction: noop,
            onSetup: spec.onSetup,
            triggersSubmenu: true,
            itemBehaviours: []
        }, structure, itemResponse, providersBackstage);
    };

    // Note, this does not create a valid SketchSpec.
    const renderNormalItem = (spec, itemResponse, providersBackstage, renderIcons = true) => {
        const getApi = (component) => ({
            isEnabled: () => !Disabling.isDisabled(component),
            setEnabled: (state) => Disabling.set(component, !state)
        });
        const structure = renderItemStructure({
            presets: 'normal',
            iconContent: spec.icon,
            textContent: spec.text,
            htmlContent: Optional.none(),
            labelContent: Optional.none(),
            ariaLabel: spec.text,
            caret: Optional.none(),
            checkMark: Optional.none(),
            shortcutContent: spec.shortcut
        }, providersBackstage, renderIcons);
        return renderCommonItem({
            context: spec.context,
            data: buildData(spec),
            getApi,
            enabled: spec.enabled,
            onAction: spec.onAction,
            onSetup: spec.onSetup,
            triggersSubmenu: false,
            itemBehaviours: []
        }, structure, itemResponse, providersBackstage);
    };

    const renderSeparatorItem = (spec) => ({
        type: 'separator',
        dom: {
            tag: 'div',
            classes: [selectableClass, groupHeadingClass]
        },
        components: spec.text.map(text$2).toArray()
    });

    const renderToggleMenuItem = (spec, itemResponse, providersBackstage, renderIcons = true) => {
        const getApi = (component) => ({
            setActive: (state) => {
                Toggling.set(component, state);
            },
            isActive: () => Toggling.isOn(component),
            isEnabled: () => !Disabling.isDisabled(component),
            setEnabled: (state) => Disabling.set(component, !state)
        });
        // BespokeSelects use meta to pass through styling information. Bespokes should only
        // be togglemenuitems hence meta is only passed through in this MenuItem.
        const structure = renderItemStructure({
            iconContent: spec.icon,
            textContent: spec.text,
            htmlContent: Optional.none(),
            labelContent: Optional.none(),
            ariaLabel: spec.text,
            checkMark: Optional.some(renderCheckmark(providersBackstage.icons)),
            caret: Optional.none(),
            shortcutContent: spec.shortcut,
            presets: 'normal',
            meta: spec.meta
        }, providersBackstage, renderIcons);
        return deepMerge(renderCommonItem({
            context: spec.context,
            data: buildData(spec),
            enabled: spec.enabled,
            getApi,
            onAction: spec.onAction,
            onSetup: spec.onSetup,
            triggersSubmenu: false,
            itemBehaviours: []
        }, structure, itemResponse, providersBackstage), {
            toggling: {
                toggleClass: tickedClass,
                toggleOnExecute: false,
                selected: spec.active
            },
            role: spec.role.getOrUndefined()
        });
    };

    const autocomplete = renderAutocompleteItem;
    const separator$3 = renderSeparatorItem;
    const normal = renderNormalItem;
    const nested = renderNestedItem;
    const toggle = renderToggleMenuItem;
    const fancy = renderFancyMenuItem;
    const card = renderCardMenuItem;

    const identifyMenuLayout = (searchMode) => {
        switch (searchMode.searchMode) {
            case 'no-search': {
                return {
                    menuType: 'normal'
                };
            }
            default: {
                return {
                    menuType: 'searchable',
                    searchMode
                };
            }
        }
    };
    const handleRefetchTrigger = (originalSandboxComp) => {
        // At the moment, a Sandbox is "Represented" by its triggering Dropdown.
        // We'll want to make this an official API, in case we change it later.
        const dropdown = Representing.getValue(originalSandboxComp);
        // Because refetch will replace the entire menu, we need to store the
        // original version of the searcher state, so that we can reinstate it
        // after the fetch completes (which is async)
        const optSearcherState = findWithinSandbox(originalSandboxComp).map(saveState);
        Dropdown.refetch(dropdown).get(() => {
            // It has completed, so now find the searcher and set its value
            // again. We can't just use the originalSandbox, because that will
            // have been thrown away and recreated by now.
            const newSandboxComp = Coupling.getCoupled(dropdown, 'sandbox');
            optSearcherState.each((searcherState) => findWithinSandbox(newSandboxComp).each((inputComp) => restoreState(inputComp, searcherState)));
        });
    };
    // This event is triggered by the searcher for key events
    // that should be handled by the currently selected item
    // (that is, the one with *fake* focus, not real focus). So we
    // need to redispatch them to the selected item in the sandbox.
    const handleRedirectToMenuItem = (sandboxComp, se) => {
        getActiveMenuItemFrom(sandboxComp).each((activeItem) => {
            retargetAndDispatchWith(sandboxComp, activeItem.element, se.event.eventType, se.event.interactionEvent);
        });
    };
    // This function is useful when you have fakeFocus, so you can't just find the
    // currently focused item (or the item that triggered a key event). It relies on
    // the following relationships between components
    // The Sandbox creates a tieredmenu, so Sandboxing.getState returns the TieredMenu
    // The TieredMenu uses Highlighting for managing which menus are active, so
    // Highlighting.getHighlighted(tmenu) is the current active menu
    // The Menu uses highlighting to manage the active item, so use
    // Highlighting.getHighlighted(menu) to get the current item.
    const getActiveMenuItemFrom = (sandboxComp) => {
        // Consider moving some of these things into shared APIs. For example, make an extra API
        // for TieredMenu to get the highlighted item.
        return Sandboxing.getState(sandboxComp)
            .bind(Highlighting.getHighlighted)
            .bind(Highlighting.getHighlighted);
    };
    const getSearchResults = (activeMenuComp) => {
        // Depending on the menu layout, the search results will either be the entire
        // menu, or something within the menu.
        return has(activeMenuComp.element, searchResultsClass)
            ? Optional.some(activeMenuComp.element)
            : descendant(activeMenuComp.element, '.' + searchResultsClass);
    };
    // Model the interaction with ARIA
    const updateAriaOnHighlight = (tmenuComp, menuComp, itemComp) => {
        // This ARIA behaviour is based on the algolia example documented in TINY-8952
        findWithinMenu(tmenuComp).each((inputComp) => {
            setActiveDescendant(inputComp, itemComp);
            const optActiveResults = getSearchResults(menuComp);
            optActiveResults.each((resultsElem) => {
                // Link aria-controls of the input to the id of the results container.
                getOpt(resultsElem, 'id')
                    .each((controlledId) => set$9(inputComp.element, 'aria-controls', controlledId));
            });
        });
        // Update the aria-selected on the item. The removal is handled by onDehighlight
        set$9(itemComp.element, 'aria-selected', 'true');
    };
    const updateAriaOnDehighlight = (tmenuComp, menuComp, itemComp) => {
        // This ARIA behaviour is based on the algolia example documented in TINY-8952
        set$9(itemComp.element, 'aria-selected', 'false');
    };
    const focusSearchField = (tmenuComp) => {
        findWithinMenu(tmenuComp).each((searcherComp) => Focusing.focus(searcherComp));
    };
    const getSearchPattern = (dropdownComp) => {
        // Dropdowns are "coupled" with their sandbox and generally, create them on demand.
        // When using "getExistingCoupled" of Coupling, it only returns the coupled
        // component (here: the sandbox) if it already exists ... it won't do any creation.
        // So here, we are trying to get possible fetchContext information for our fetch
        // callback. If there is no sandbox, then there is no open menu, and we
        // don't have any search context, so use an empty string. Otherwise, dive into
        // the sandbox, and find the search field's current pattern.
        const optSandboxComp = Coupling.getExistingCoupled(dropdownComp, 'sandbox');
        return optSandboxComp
            .bind(findWithinSandbox)
            .map(saveState)
            .map((state) => state.fetchPattern)
            .getOr('');
    };

    var FocusMode;
    (function (FocusMode) {
        FocusMode[FocusMode["ContentFocus"] = 0] = "ContentFocus";
        FocusMode[FocusMode["UiFocus"] = 1] = "UiFocus";
    })(FocusMode || (FocusMode = {}));
    const createMenuItemFromBridge = (item, itemResponse, backstage, menuHasIcons, isHorizontalMenu) => {
        const providersBackstage = backstage.shared.providers;
        // If we're making a horizontal menu (mobile context menu) we want text OR icons
        // to simplify the UI. We also don't want shortcut text.
        const parseForHorizontalMenu = (menuitem) => !isHorizontalMenu ? menuitem : ({
            ...menuitem,
            shortcut: Optional.none(),
            icon: menuitem.text.isSome() ? Optional.none() : menuitem.icon
        });
        switch (item.type) {
            case 'menuitem':
                return createMenuItem(item).fold(handleError, (d) => Optional.some(normal(parseForHorizontalMenu(d), itemResponse, providersBackstage, menuHasIcons)));
            case 'nestedmenuitem':
                return createNestedMenuItem(item).fold(handleError, (d) => Optional.some(nested(parseForHorizontalMenu(d), itemResponse, providersBackstage, menuHasIcons, isHorizontalMenu)));
            case 'togglemenuitem':
                return createToggleMenuItem(item).fold(handleError, (d) => Optional.some(toggle(parseForHorizontalMenu(d), itemResponse, providersBackstage, menuHasIcons)));
            case 'separator':
                return createSeparatorMenuItem(item).fold(handleError, (d) => Optional.some(separator$3(d)));
            case 'fancymenuitem':
                return createFancyMenuItem(item).fold(handleError, 
                // Fancy menu items don't have shortcuts or icons
                (d) => fancy(d, backstage));
            default: {
                // eslint-disable-next-line no-console
                console.error('Unknown item in general menu', item);
                return Optional.none();
            }
        }
    };
    const createAutocompleteItems = (items, matchText, onItemValueHandler, columns, itemResponse, sharedBackstage, highlightOn) => {
        // Render text and icons if we're using a single column, otherwise only render icons
        const renderText = columns === 1;
        const renderIcons = !renderText || menuHasIcons(items);
        return cat(map$2(items, (item) => {
            switch (item.type) {
                case 'separator':
                    return createSeparatorItem(item).fold(handleError, (d) => Optional.some(separator$3(d)));
                case 'cardmenuitem':
                    return createCardMenuItem(item).fold(handleError, (d) => Optional.some(card({
                        ...d,
                        // Intercept action
                        onAction: (api) => {
                            d.onAction(api);
                            onItemValueHandler(d.value, d.meta);
                        }
                    }, itemResponse, sharedBackstage, {
                        itemBehaviours: tooltipBehaviour(d.meta, sharedBackstage, Optional.none()),
                        cardText: {
                            matchText,
                            highlightOn
                        }
                    })));
                case 'autocompleteitem':
                default:
                    return createAutocompleterItem(item).fold(handleError, (d) => Optional.some(autocomplete(d, matchText, renderText, 'normal', onItemValueHandler, itemResponse, sharedBackstage, renderIcons)));
            }
        }));
    };
    const createPartialMenu = (value, items, itemResponse, backstage, isHorizontalMenu, searchMode) => {
        const hasIcons = menuHasIcons(items);
        const alloyItems = cat(map$2(items, (item) => {
            // Have to check each item for an icon, instead of as part of hasIcons above,
            // else in horizontal menus, items with an icon but without text will display
            // with neither
            const itemHasIcon = (i) => isHorizontalMenu ? !has$2(i, 'text') : hasIcons;
            const createItem = (i) => createMenuItemFromBridge(i, itemResponse, backstage, itemHasIcon(i), isHorizontalMenu);
            if (item.type === 'nestedmenuitem' && item.getSubmenuItems().length <= 0) {
                return createItem({ ...item, enabled: false });
            }
            else {
                return createItem(item);
            }
        }));
        // The menu layout is dependent upon our search mode.
        const menuLayout = identifyMenuLayout(searchMode);
        const createPartial = isHorizontalMenu ?
            createHorizontalPartialMenuWithAlloyItems :
            createPartialMenuWithAlloyItems;
        return createPartial(value, hasIcons, alloyItems, 1, menuLayout);
    };
    const createTieredDataFrom = (partialMenu) => tieredMenu.singleData(partialMenu.value, partialMenu);
    const createInlineMenuFrom = (partialMenu, columns, focusMode, presets) => {
        const movement = deriveMenuMovement(columns, presets);
        const menuMarkers = markers(presets);
        return {
            data: createTieredDataFrom({
                ...partialMenu,
                movement,
                menuBehaviours: SimpleBehaviours.unnamedEvents(columns !== 'auto' ? [] : [
                    runOnAttached((comp, _se) => {
                        detectSize(comp, 4, menuMarkers.item).each(({ numColumns, numRows }) => {
                            Keying.setGridSize(comp, numRows, numColumns);
                        });
                    })
                ])
            }),
            menu: {
                markers: markers(presets),
                fakeFocus: focusMode === FocusMode.ContentFocus
            }
        };
    };

    const rangeToSimRange = (r) => SimRange.create(SugarElement.fromDom(r.startContainer), r.startOffset, SugarElement.fromDom(r.endContainer), r.endOffset);
    const register$c = (editor, sharedBackstage) => {
        const autocompleterId = generate$6('autocompleter');
        const processingAction = Cell(false);
        const activeState = Cell(false);
        const activeRange = value$2();
        const autocompleter = build$1(InlineView.sketch({
            dom: {
                tag: 'div',
                classes: ['tox-autocompleter'],
                attributes: {
                    id: autocompleterId
                }
            },
            components: [],
            fireDismissalEventInstead: {},
            inlineBehaviours: derive$1([
                config('dismissAutocompleter', [
                    run$1(dismissRequested(), () => cancelIfNecessary()),
                    run$1(highlight$1(), (_, se) => {
                        getOpt(se.event.target, 'id').each((id) => set$9(SugarElement.fromDom(editor.getBody()), 'aria-activedescendant', id));
                    }),
                ])
            ]),
            lazySink: sharedBackstage.getSink
        }));
        const isMenuOpen = () => InlineView.isOpen(autocompleter);
        const isActive = activeState.get;
        const hideIfNecessary = () => {
            if (isMenuOpen()) {
                InlineView.hide(autocompleter);
                editor.dom.remove(autocompleterId, false);
                const editorBody = SugarElement.fromDom(editor.getBody());
                getOpt(editorBody, 'aria-owns')
                    .filter((ariaOwnsAttr) => ariaOwnsAttr === autocompleterId)
                    .each(() => {
                    remove$8(editorBody, 'aria-owns');
                    remove$8(editorBody, 'aria-activedescendant');
                });
            }
        };
        const getMenu = () => InlineView.getContent(autocompleter).bind((tmenu) => {
            // The autocompleter menu will be the first child component of the tiered menu.
            // Unfortunately a memento can't be used to do this lookup because the component
            // id is changed while generating the tiered menu.
            return get$i(tmenu.components(), 0);
        });
        const cancelIfNecessary = () => editor.execCommand('mceAutocompleterClose');
        const getCombinedItems = (matches) => {
            const columns = findMap(matches, (m) => Optional.from(m.columns)).getOr(1);
            return bind$3(matches, (match) => {
                const choices = match.items;
                return createAutocompleteItems(choices, match.matchText, (itemValue, itemMeta) => {
                    const autocompleterApi = {
                        hide: () => cancelIfNecessary(),
                        reload: (fetchOptions) => {
                            hideIfNecessary();
                            editor.execCommand('mceAutocompleterReload', false, { fetchOptions });
                        }
                    };
                    // Asks the editor for a new active range that emits an event that updates
                    // the activeRange state not ideal but trying to avoid direct method calls to the core.
                    // We need to get a fresh range since when you hit enter the IME commits and the updates the DOM so we then need to rescan.
                    editor.execCommand('mceAutocompleterRefreshActiveRange');
                    activeRange.get().each((range) => {
                        processingAction.set(true);
                        match.onAction(autocompleterApi, range, itemValue, itemMeta);
                        processingAction.set(false);
                    });
                }, columns, ItemResponse$1.BUBBLE_TO_SANDBOX, sharedBackstage, match.highlightOn);
            });
        };
        const display = (lookupData, items) => {
            // Display the autocompleter menu
            const columns = findMap(lookupData, (ld) => Optional.from(ld.columns)).getOr(1);
            InlineView.showMenuAt(autocompleter, {
                anchor: {
                    type: 'selection',
                    getSelection: () => activeRange.get().map(rangeToSimRange),
                    root: SugarElement.fromDom(editor.getBody()),
                }
            }, createInlineMenuFrom(createPartialMenuWithAlloyItems('autocompleter-value', true, items, columns, { menuType: 'normal' }), columns, FocusMode.ContentFocus, 
            // Use the constant.
            'normal'));
            getMenu().each(Highlighting.highlightFirst);
        };
        const updateDisplay = (lookupData) => {
            const combinedItems = getCombinedItems(lookupData);
            // Open the autocompleter if there are items to show
            if (combinedItems.length > 0) {
                display(lookupData, combinedItems);
                set$9(SugarElement.fromDom(editor.getBody()), 'aria-owns', autocompleterId);
                if (!editor.inline) {
                    cloneAutocompleterToEditorDoc();
                }
            }
            else {
                hideIfNecessary();
            }
        };
        const cloneAutocompleterToEditorDoc = () => {
            if (editor.dom.get(autocompleterId)) {
                editor.dom.remove(autocompleterId, false);
            }
            const docElm = editor.getDoc().documentElement;
            const selection = editor.selection.getNode();
            const newElm = deep(autocompleter.element);
            setAll(newElm, {
                border: '0',
                clip: 'rect(0 0 0 0)',
                height: '1px',
                margin: '-1px',
                overflow: 'hidden',
                padding: '0',
                position: 'absolute',
                width: '1px',
                top: `${selection.offsetTop}px`,
                left: `${selection.offsetLeft}px`,
            });
            editor.dom.add(docElm, newElm.dom);
            // Clean up positioning styles so that the "hidden" autocompleter is around the selection
            descendant(newElm, '[role="menu"]').each((child) => {
                remove$6(child, 'position');
                remove$6(child, 'max-height');
            });
        };
        editor.on('AutocompleterStart', ({ lookupData }) => {
            activeState.set(true);
            processingAction.set(false);
            updateDisplay(lookupData);
        });
        editor.on('AutocompleterUpdate', ({ lookupData }) => updateDisplay(lookupData));
        editor.on('AutocompleterUpdateActiveRange', ({ range }) => activeRange.set(range));
        editor.on('AutocompleterEnd', () => {
            // Hide the menu and reset
            hideIfNecessary();
            activeState.set(false);
            processingAction.set(false);
            activeRange.clear();
        });
        const autocompleterUiApi = {
            cancelIfNecessary,
            isMenuOpen,
            isActive,
            isProcessingAction: processingAction.get,
            getMenu
        };
        AutocompleterEditorEvents.setup(autocompleterUiApi, editor);
    };
    const Autocompleter = {
        register: register$c
    };

    const renderBar = (spec, backstage) => ({
        dom: {
            tag: 'div',
            classes: ['tox-bar', 'tox-form__controls-h-stack']
        },
        components: map$2(spec.items, backstage.interpreter)
    });

    var global$4 = tinymce.util.Tools.resolve('tinymce.html.Entities');

    const renderFormFieldWith = (pLabel, pField, extraClasses, extraBehaviours) => {
        const spec = renderFormFieldSpecWith(pLabel, pField, extraClasses, extraBehaviours);
        return FormField.sketch(spec);
    };
    const renderFormField = (pLabel, pField) => renderFormFieldWith(pLabel, pField, [], []);
    const renderFormFieldSpecWith = (pLabel, pField, extraClasses, extraBehaviours) => ({
        dom: renderFormFieldDomWith(extraClasses),
        components: pLabel.toArray().concat([pField]),
        fieldBehaviours: derive$1(extraBehaviours)
    });
    const renderFormFieldDom = () => renderFormFieldDomWith([]);
    const renderFormFieldDomWith = (extraClasses) => ({
        tag: 'div',
        classes: ['tox-form__group'].concat(extraClasses)
    });
    const renderLabel$3 = (label, providersBackstage) => FormField.parts.label({
        dom: {
            tag: 'label',
            classes: ['tox-label']
        },
        components: [
            text$2(providersBackstage.translate(label))
        ]
    });

    const formChangeEvent = generate$6('form-component-change');
    const formInputEvent = generate$6('form-component-input');
    const formCloseEvent = generate$6('form-close');
    const formCancelEvent = generate$6('form-cancel');
    const formActionEvent = generate$6('form-action');
    const formSubmitEvent = generate$6('form-submit');
    const formBlockEvent = generate$6('form-block');
    const formUnblockEvent = generate$6('form-unblock');
    const formTabChangeEvent = generate$6('form-tabchange');
    const formResizeEvent = generate$6('form-resize');

    const renderCollection = (spec, providersBackstage, initialData) => {
        // DUPE with TextField.
        const pLabel = spec.label.map((label) => renderLabel$3(label, providersBackstage));
        const icons = providersBackstage.icons();
        // TINY-10174: Icon string is either in icon pack or displayed directly
        const getIcon = (icon) => icons[icon] ?? icon;
        const runOnItem = (f) => (comp, se) => {
            closest$3(se.event.target, '[data-collection-item-value]').each((target) => {
                f(comp, se, target, get$g(target, 'data-collection-item-value'));
            });
        };
        const setContents = (comp, items) => {
            // Giving it a default `mode:design` context, these shouldn't run at all in mode:readonly
            const disabled = providersBackstage.checkUiComponentContext('mode:design').shouldDisable || providersBackstage.isDisabled();
            const disabledClass = disabled ? ' tox-collection__item--state-disabled' : '';
            const htmlLines = map$2(items, (item) => {
                const itemText = global$6.translate(item.text);
                const textContent = spec.columns === 1 ? `<div class="tox-collection__item-label">${itemText}</div>` : '';
                const iconContent = `<div class="tox-collection__item-icon">${getIcon(item.icon)}</div>`;
                // Replacing the hyphens and underscores in collection items with spaces
                // to ensure the screen readers pronounce the words correctly.
                // This is only for aria purposes. Emoticon and Special Character names will still use _ and - for autocompletion.
                const mapItemName = {
                    '_': ' ',
                    ' - ': ' ',
                    '-': ' '
                };
                // Using aria-label here overrides the Apple description of emojis and special characters in Mac/ MS description in Windows.
                // But if only the title attribute is used instead, the names are read out twice. i.e., the description followed by the item.text.
                const ariaLabel = itemText.replace(/\_| \- |\-/g, (match) => mapItemName[match]);
                return `<div data-mce-tooltip="${ariaLabel}" class="tox-collection__item${disabledClass}" tabindex="-1" data-collection-item-value="${global$4.encodeAllRaw(item.value)}" aria-label="${ariaLabel}">${iconContent}${textContent}</div>`;
            });
            const chunks = spec.columns !== 'auto' && spec.columns > 1 ? chunk$1(htmlLines, spec.columns) : [htmlLines];
            const html = map$2(chunks, (ch) => `<div class="tox-collection__group">${ch.join('')}</div>`);
            set$8(comp.element, html.join(''));
        };
        const onClick = runOnItem((comp, se, tgt, itemValue) => {
            se.stop();
            if (!(providersBackstage.checkUiComponentContext('mode:design').shouldDisable || providersBackstage.isDisabled())) {
                emitWith(comp, formActionEvent, {
                    name: spec.name,
                    value: itemValue
                });
            }
        });
        const collectionEvents = [
            run$1(mouseover(), runOnItem((comp, se, tgt) => {
                focus$4(tgt, true);
            })),
            run$1(click(), onClick),
            run$1(tap(), onClick),
            run$1(focusin(), runOnItem((comp, se, tgt) => {
                descendant(comp.element, '.' + activeClass).each((currentActive) => {
                    remove$3(currentActive, activeClass);
                });
                add$2(tgt, activeClass);
            })),
            run$1(focusout(), runOnItem((comp) => {
                descendant(comp.element, '.' + activeClass).each((currentActive) => {
                    remove$3(currentActive, activeClass);
                    blur$1(currentActive);
                });
            })),
            runOnExecute$1(runOnItem((comp, se, tgt, itemValue) => {
                emitWith(comp, formActionEvent, {
                    name: spec.name,
                    value: itemValue
                });
            })),
        ];
        const iterCollectionItems = (comp, applyAttributes) => map$2(descendants(comp.element, '.tox-collection__item'), applyAttributes);
        const pField = FormField.parts.field({
            dom: {
                tag: 'div',
                // FIX: Read from columns
                classes: ['tox-collection'].concat(spec.columns !== 1 ? ['tox-collection--grid'] : ['tox-collection--list'])
            },
            components: [],
            factory: { sketch: identity },
            behaviours: derive$1([
                Disabling.config({
                    disabled: () => providersBackstage.checkUiComponentContext(spec.context).shouldDisable,
                    onDisabled: (comp) => {
                        iterCollectionItems(comp, (childElm) => {
                            add$2(childElm, 'tox-collection__item--state-disabled');
                            set$9(childElm, 'aria-disabled', true);
                        });
                    },
                    onEnabled: (comp) => {
                        iterCollectionItems(comp, (childElm) => {
                            remove$3(childElm, 'tox-collection__item--state-disabled');
                            remove$8(childElm, 'aria-disabled');
                        });
                    }
                }),
                toggleOnReceive(() => providersBackstage.checkUiComponentContext(spec.context)),
                Replacing.config({}),
                Tooltipping.config({
                    ...providersBackstage.tooltips.getConfig({
                        tooltipText: '',
                        onShow: (comp) => {
                            descendant(comp.element, '.' + activeClass + '[data-mce-tooltip]').each((current) => {
                                getOpt(current, 'data-mce-tooltip').each((text) => {
                                    Tooltipping.setComponents(comp, providersBackstage.tooltips.getComponents({ tooltipText: text }));
                                });
                            });
                        }
                    }),
                    mode: 'children-keyboard-focus',
                    anchor: (comp) => ({
                        type: 'node',
                        node: descendant(comp.element, '.' + activeClass).orThunk(() => first('.tox-collection__item')),
                        root: comp.element,
                        layouts: {
                            onLtr: constant$1([south$2, north$2, southeast$2, northeast$2, southwest$2, northwest$2]),
                            onRtl: constant$1([south$2, north$2, southeast$2, northeast$2, southwest$2, northwest$2])
                        },
                        bubble: nu$6(0, -2, {}),
                    })
                }),
                Representing.config({
                    store: {
                        mode: 'memory',
                        initialValue: initialData.getOr([])
                    },
                    onSetValue: (comp, items) => {
                        setContents(comp, items);
                        if (spec.columns === 'auto') {
                            detectSize(comp, 5, 'tox-collection__item').each(({ numRows, numColumns }) => {
                                Keying.setGridSize(comp, numRows, numColumns);
                            });
                        }
                        emit(comp, formResizeEvent);
                    }
                }),
                Tabstopping.config({}),
                Keying.config(deriveCollectionMovement(spec.columns, 'normal')),
                config('collection-events', collectionEvents)
            ]),
            eventOrder: {
                [execute$5()]: ['disabling', 'alloy.base.behaviour', 'collection-events'],
                [focusin()]: ['collection-events', 'tooltipping'],
            }
        });
        const extraClasses = ['tox-form__group--collection'];
        return renderFormFieldWith(pLabel, pField, extraClasses, []);
    };

    const renderPanelButton = (spec, sharedBackstage) => Dropdown.sketch({
        dom: spec.dom,
        components: spec.components,
        toggleClass: 'mce-active',
        dropdownBehaviours: derive$1([
            DisablingConfigs.button(() => sharedBackstage.providers.isDisabled() || sharedBackstage.providers.checkUiComponentContext(spec.context).shouldDisable),
            toggleOnReceive(() => sharedBackstage.providers.checkUiComponentContext(spec.context)),
            Unselecting.config({}),
            Tabstopping.config({})
        ]),
        layouts: spec.layouts,
        sandboxClasses: ['tox-dialog__popups'],
        lazySink: sharedBackstage.getSink,
        fetch: (comp) => Future.nu((callback) => spec.fetch(callback)).map((items) => Optional.from(createTieredDataFrom(deepMerge(createPartialChoiceMenu(generate$6('menu-value'), items, (value) => {
            spec.onItemAction(comp, value);
        }, spec.columns, spec.presets, ItemResponse$1.CLOSE_ON_EXECUTE, 
        // No colour is ever selected on opening
        never, sharedBackstage.providers), {
            movement: deriveMenuMovement(spec.columns, spec.presets)
        })))),
        parts: {
            menu: part(false, 1, spec.presets)
        }
    });

    const colorInputChangeEvent = generate$6('color-input-change');
    const colorSwatchChangeEvent = generate$6('color-swatch-change');
    const colorPickerCancelEvent = generate$6('color-picker-cancel');
    const renderColorInput = (spec, sharedBackstage, colorInputBackstage, initialData) => {
        const pField = FormField.parts.field({
            factory: Input,
            inputClasses: ['tox-textfield'],
            data: initialData,
            onSetValue: (c) => Invalidating.run(c).get(noop),
            inputBehaviours: derive$1([
                Disabling.config({
                    disabled: () => sharedBackstage.providers.isDisabled() || sharedBackstage.providers.checkUiComponentContext(spec.context).shouldDisable
                }),
                toggleOnReceive(() => sharedBackstage.providers.checkUiComponentContext(spec.context)),
                Tabstopping.config({}),
                Invalidating.config({
                    invalidClass: 'tox-textbox-field-invalid',
                    getRoot: (comp) => parentElement(comp.element),
                    notify: {
                        onValid: (comp) => {
                            // onValid should pass through the value here
                            // We need a snapshot of the value validated.
                            const val = Representing.getValue(comp);
                            emitWith(comp, colorInputChangeEvent, {
                                color: val
                            });
                        }
                    },
                    validator: {
                        validateOnLoad: false,
                        validate: (input) => {
                            const inputValue = Representing.getValue(input);
                            // Consider empty strings valid colours
                            if (inputValue.length === 0) {
                                return Future.pure(Result.value(true));
                            }
                            else {
                                const span = SugarElement.fromTag('span');
                                set$7(span, 'background-color', inputValue);
                                const res = getRaw(span, 'background-color').fold(
                                // TODO: Work out what we want to do here.
                                () => Result.error('blah'), (_) => Result.value(inputValue));
                                return Future.pure(res);
                            }
                        }
                    }
                })
            ]),
            selectOnFocus: false
        });
        const pLabel = spec.label.map((label) => renderLabel$3(label, sharedBackstage.providers));
        const emitSwatchChange = (colorBit, value) => {
            emitWith(colorBit, colorSwatchChangeEvent, {
                value
            });
        };
        const onItemAction = (comp, value) => {
            memColorButton.getOpt(comp).each((colorBit) => {
                if (value === 'custom') {
                    colorInputBackstage.colorPicker((valueOpt) => {
                        valueOpt.fold(() => emit(colorBit, colorPickerCancelEvent), (value) => {
                            emitSwatchChange(colorBit, value);
                            addColor(spec.storageKey, value);
                        });
                    }, '#ffffff');
                }
                else if (value === 'remove') {
                    emitSwatchChange(colorBit, '');
                }
                else {
                    emitSwatchChange(colorBit, value);
                }
            });
        };
        const memColorButton = record(renderPanelButton({
            dom: {
                tag: 'span',
                attributes: {
                    'aria-label': sharedBackstage.providers.translate('Color swatch')
                }
            },
            layouts: {
                onRtl: () => [southwest$2, southeast$2, south$2],
                onLtr: () => [southeast$2, southwest$2, south$2]
            },
            components: [],
            fetch: getFetch$1(colorInputBackstage.getColors(spec.storageKey), spec.storageKey, colorInputBackstage.hasCustomColors()),
            columns: colorInputBackstage.getColorCols(spec.storageKey),
            presets: 'color',
            onItemAction,
            context: spec.context
        }, sharedBackstage));
        return FormField.sketch({
            dom: {
                tag: 'div',
                classes: ['tox-form__group']
            },
            components: pLabel.toArray().concat([
                {
                    dom: {
                        tag: 'div',
                        classes: ['tox-color-input']
                    },
                    components: [
                        pField,
                        memColorButton.asSpec()
                    ]
                }
            ]),
            fieldBehaviours: derive$1([
                config('form-field-events', [
                    run$1(colorInputChangeEvent, (comp, se) => {
                        memColorButton.getOpt(comp).each((colorButton) => {
                            set$7(colorButton.element, 'background-color', se.event.color);
                        });
                        emitWith(comp, formChangeEvent, { name: spec.name });
                    }),
                    run$1(colorSwatchChangeEvent, (comp, se) => {
                        FormField.getField(comp).each((field) => {
                            Representing.setValue(field, se.event.value);
                            // Focus the field now that we've set its value
                            Composing.getCurrent(comp).each(Focusing.focus);
                        });
                    }),
                    run$1(colorPickerCancelEvent, (comp, _se) => {
                        FormField.getField(comp).each((_field) => {
                            Composing.getCurrent(comp).each(Focusing.focus);
                        });
                    })
                ])
            ])
        });
    };

    // TODO: Move this to alloy if the concept works out
    // eslint-disable-next-line consistent-this
    const self = () => Composing.config({
        find: Optional.some
    });
    const memento$1 = (mem) => Composing.config({
        find: mem.getOpt
    });
    const childAt = (index) => Composing.config({
        find: (comp) => child$2(comp.element, index)
            .bind((element) => comp.getSystem().getByDom(element).toOptional())
    });
    const ComposingConfigs = {
        self,
        memento: memento$1,
        childAt
    };

    const processors = objOf([
        defaulted('preprocess', identity),
        defaulted('postprocess', identity)
    ]);
    const memento = (mem, rawProcessors) => {
        const ps = asRawOrDie$1('RepresentingConfigs.memento processors', processors, rawProcessors);
        return Representing.config({
            store: {
                mode: 'manual',
                getValue: (comp) => {
                    const other = mem.get(comp);
                    const rawValue = Representing.getValue(other);
                    return ps.postprocess(rawValue);
                },
                setValue: (comp, rawValue) => {
                    const newValue = ps.preprocess(rawValue);
                    const other = mem.get(comp);
                    Representing.setValue(other, newValue);
                }
            }
        });
    };
    const withComp = (optInitialValue, getter, setter) => Representing.config({
        store: {
            mode: 'manual',
            ...optInitialValue.map((initialValue) => ({ initialValue })).getOr({}),
            getValue: getter,
            setValue: setter
        }
    });
    const withElement = (initialValue, getter, setter) => withComp(initialValue, (c) => getter(c.element), (c, v) => setter(c.element, v));
    const domHtml = (optInitialValue) => withElement(optInitialValue, get$f, set$8);
    const memory = (initialValue) => Representing.config({
        store: {
            mode: 'memory',
            initialValue
        }
    });

    const fieldsUpdate = generate$6('rgb-hex-update');
    const sliderUpdate = generate$6('slider-update');
    const paletteUpdate = generate$6('palette-update');

    const sliderFactory = (translate, getClass) => {
        const spectrum = Slider.parts.spectrum({
            dom: {
                tag: 'div',
                classes: [getClass('hue-slider-spectrum')],
                attributes: {
                    role: 'presentation'
                }
            }
        });
        const thumb = Slider.parts.thumb({
            dom: {
                tag: 'div',
                classes: [getClass('hue-slider-thumb')],
                attributes: {
                    role: 'presentation'
                }
            }
        });
        return Slider.sketch({
            dom: {
                tag: 'div',
                classes: [getClass('hue-slider')],
                attributes: {
                    'role': 'slider',
                    'aria-valuemin': 0,
                    'aria-valuemax': 360,
                    'aria-valuenow': 120,
                }
            },
            rounded: false,
            model: {
                mode: 'y',
                getInitialValue: constant$1(0)
            },
            components: [
                spectrum,
                thumb
            ],
            sliderBehaviours: derive$1([
                Focusing.config({})
            ]),
            onChange: (slider, _thumb, value) => {
                set$9(slider.element, 'aria-valuenow', Math.floor(360 - (value * 3.6)));
                emitWith(slider, sliderUpdate, {
                    value
                });
            }
        });
    };

    const validInput = generate$6('valid-input');
    const invalidInput = generate$6('invalid-input');
    const validatingInput = generate$6('validating-input');
    const translatePrefix = 'colorcustom.rgb.';
    const uninitiatedTooltipApi = {
        isEnabled: always,
        setEnabled: noop,
        immediatelyShow: noop,
        immediatelyHide: noop,
    };
    const rgbFormFactory = (translate, getClass, onValidHexx, onInvalidHexx, tooltipGetConfig, makeIcon) => {
        const setTooltipEnabled = (enabled, tooltipApi) => {
            const api = tooltipApi.get();
            if (enabled === api.isEnabled()) {
                return;
            }
            api.setEnabled(enabled);
            if (enabled) {
                api.immediatelyShow();
            }
            else {
                api.immediatelyHide();
            }
        };
        const invalidation = (label, isValid, tooltipApi) => Invalidating.config({
            invalidClass: getClass('invalid'),
            notify: {
                onValidate: (comp) => {
                    emitWith(comp, validatingInput, {
                        type: label
                    });
                },
                onValid: (comp) => {
                    setTooltipEnabled(false, tooltipApi);
                    emitWith(comp, validInput, {
                        type: label,
                        value: Representing.getValue(comp)
                    });
                },
                onInvalid: (comp) => {
                    setTooltipEnabled(true, tooltipApi);
                    emitWith(comp, invalidInput, {
                        type: label,
                        value: Representing.getValue(comp)
                    });
                }
            },
            validator: {
                validate: (comp) => {
                    const value = Representing.getValue(comp);
                    const res = isValid(value) ? Result.value(true) : Result.error(translate('aria.input.invalid'));
                    return Future.pure(res);
                },
                validateOnLoad: false
            }
        });
        const renderTextField = (isValid, name, label, description, data) => {
            const tooltipApi = Cell(uninitiatedTooltipApi);
            const helptext = translate(translatePrefix + 'range');
            const pLabel = FormField.parts.label({
                dom: { tag: 'label' },
                components: [text$2(label)]
            });
            const pField = FormField.parts.field({
                data,
                factory: Input,
                inputAttributes: {
                    'type': 'text',
                    'aria-label': description,
                    ...name === 'hex' ? { 'aria-live': 'polite' } : {}
                },
                inputClasses: [getClass('textfield')],
                // Have basic invalidating and tabstopping behaviour.
                inputBehaviours: derive$1([
                    invalidation(name, isValid, tooltipApi),
                    Tabstopping.config({}),
                    Tooltipping.config({
                        ...tooltipGetConfig({
                            tooltipText: '',
                            onSetup: (comp) => {
                                tooltipApi.set({
                                    isEnabled: () => {
                                        return Tooltipping.isEnabled(comp);
                                    },
                                    setEnabled: (enabled) => {
                                        return Tooltipping.setEnabled(comp, enabled);
                                    },
                                    immediatelyShow: () => {
                                        return Tooltipping.immediateOpenClose(comp, true);
                                    },
                                    immediatelyHide: () => {
                                        return Tooltipping.immediateOpenClose(comp, false);
                                    },
                                });
                                Tooltipping.setEnabled(comp, false);
                            },
                            onShow: (component, _tooltip) => {
                                Tooltipping.setComponents(component, [
                                    {
                                        dom: {
                                            tag: 'p',
                                            classes: [
                                                getClass('rgb-warning-note')
                                            ]
                                        },
                                        components: [text$2(translate(name === 'hex' ? 'colorcustom.rgb.invalidHex' : 'colorcustom.rgb.invalid'))]
                                    }
                                ]);
                            },
                        })
                    })
                ]),
                // If it was invalid, and the value was set, run validation against it.
                onSetValue: (input) => {
                    if (Invalidating.isInvalid(input)) {
                        const run = Invalidating.run(input);
                        run.get(noop);
                    }
                }
            });
            const errorId = generate$6('aria-invalid');
            const memInvalidIcon = record(makeIcon('invalid', Optional.some(errorId), 'warning'));
            const memStatus = record({
                dom: {
                    tag: 'div',
                    classes: [getClass('invalid-icon')]
                },
                components: [
                    memInvalidIcon.asSpec()
                ]
            });
            const comps = [pLabel, pField, memStatus.asSpec()];
            const concats = name !== 'hex' ? [FormField.parts['aria-descriptor']({
                    text: helptext
                })] : [];
            const components = comps.concat(concats);
            return {
                dom: {
                    tag: 'div',
                    attributes: {
                        role: 'presentation'
                    },
                    classes: [
                        getClass('rgb-container'),
                    ]
                },
                components
            };
        };
        const copyRgbToHex = (form, rgba) => {
            const hex = fromRgba(rgba);
            Form.getField(form, 'hex').each((hexField) => {
                // Not amazing, but it turns out that if we have an invalid RGB field, and no hex code
                // and then type in a valid three digit hex code, the RGB field will be overriden, then validate and then set
                // the hex field to be the six digit version of that same three digit hex code. This is incorrect.
                if (!Focusing.isFocused(hexField)) {
                    Representing.setValue(form, {
                        hex: hex.value
                    });
                }
            });
            return hex;
        };
        const copyRgbToForm = (form, rgb) => {
            const red = rgb.red;
            const green = rgb.green;
            const blue = rgb.blue;
            Representing.setValue(form, { red, green, blue });
        };
        const memPreview = record({
            dom: {
                tag: 'div',
                classes: [getClass('rgba-preview')],
                styles: {
                    'background-color': 'white'
                },
                attributes: {
                    role: 'presentation'
                }
            }
        });
        const updatePreview = (anyInSystem, hex) => {
            memPreview.getOpt(anyInSystem).each((preview) => {
                set$7(preview.element, 'background-color', '#' + hex.value);
            });
        };
        const factory = () => {
            const state = {
                red: Cell(Optional.some(255)),
                green: Cell(Optional.some(255)),
                blue: Cell(Optional.some(255)),
                hex: Cell(Optional.some('ffffff'))
            };
            const copyHexToRgb = (form, hex) => {
                const rgb = fromHex(hex);
                copyRgbToForm(form, rgb);
                setValueRgb(rgb);
            };
            const get = (prop) => state[prop].get();
            const set = (prop, value) => {
                state[prop].set(value);
            };
            const getValueRgb = () => get('red').bind((red) => get('green').bind((green) => get('blue').map((blue) => rgbaColour(red, green, blue, 1))));
            // TODO: Find way to use this for palette and slider updates
            const setValueRgb = (rgb) => {
                const red = rgb.red;
                const green = rgb.green;
                const blue = rgb.blue;
                set('red', Optional.some(red));
                set('green', Optional.some(green));
                set('blue', Optional.some(blue));
            };
            const onInvalidInput = (form, simulatedEvent) => {
                const data = simulatedEvent.event;
                if (data.type !== 'hex') {
                    set(data.type, Optional.none());
                }
                else {
                    onInvalidHexx(form);
                }
            };
            const onValidHex = (form, value) => {
                onValidHexx(form);
                const hex = hexColour(value);
                set('hex', Optional.some(hex.value));
                const rgb = fromHex(hex);
                copyRgbToForm(form, rgb);
                setValueRgb(rgb);
                emitWith(form, fieldsUpdate, {
                    hex
                });
                updatePreview(form, hex);
            };
            const onValidRgb = (form, prop, value) => {
                const val = parseInt(value, 10);
                set(prop, Optional.some(val));
                getValueRgb().each((rgb) => {
                    const hex = copyRgbToHex(form, rgb);
                    emitWith(form, fieldsUpdate, {
                        hex
                    });
                    updatePreview(form, hex);
                });
            };
            const isHexInputEvent = (data) => data.type === 'hex';
            const onValidInput = (form, simulatedEvent) => {
                const data = simulatedEvent.event;
                if (isHexInputEvent(data)) {
                    onValidHex(form, data.value);
                }
                else {
                    onValidRgb(form, data.type, data.value);
                }
            };
            const formPartStrings = (key) => ({
                label: translate(translatePrefix + key + '.label'),
                description: translate(translatePrefix + key + '.description')
            });
            const redStrings = formPartStrings('red');
            const greenStrings = formPartStrings('green');
            const blueStrings = formPartStrings('blue');
            const hexStrings = formPartStrings('hex');
            // TODO: Provide a nice way of adding APIs to existing sketchers
            return deepMerge(Form.sketch((parts) => ({
                dom: {
                    tag: 'form',
                    classes: [getClass('rgb-form')],
                    attributes: { 'aria-label': translate('aria.color.picker') }
                },
                components: [
                    parts.field('red', FormField.sketch(renderTextField(isRgbaComponent, 'red', redStrings.label, redStrings.description, 255))),
                    parts.field('green', FormField.sketch(renderTextField(isRgbaComponent, 'green', greenStrings.label, greenStrings.description, 255))),
                    parts.field('blue', FormField.sketch(renderTextField(isRgbaComponent, 'blue', blueStrings.label, blueStrings.description, 255))),
                    parts.field('hex', FormField.sketch(renderTextField(isHexString, 'hex', hexStrings.label, hexStrings.description, 'ffffff'))),
                    memPreview.asSpec()
                ],
                formBehaviours: derive$1([
                    Invalidating.config({
                        invalidClass: getClass('form-invalid')
                    }),
                    config('rgb-form-events', [
                        run$1(validInput, onValidInput),
                        run$1(invalidInput, onInvalidInput),
                        run$1(validatingInput, onInvalidInput)
                    ])
                ])
            })), {
                apis: {
                    updateHex: (form, hex) => {
                        Representing.setValue(form, {
                            hex: hex.value
                        });
                        copyHexToRgb(form, hex);
                        updatePreview(form, hex);
                    }
                }
            });
        };
        const rgbFormSketcher = single({
            factory,
            name: 'RgbForm',
            configFields: [],
            apis: {
                updateHex: (apis, form, hex) => {
                    apis.updateHex(form, hex);
                }
            },
            extraApis: {}
        });
        return rgbFormSketcher;
    };

    const paletteFactory = (translate, getClass) => {
        const spectrumPart = Slider.parts.spectrum({
            dom: {
                tag: 'canvas',
                attributes: {
                    role: 'presentation'
                },
                classes: [getClass('sv-palette-spectrum')]
            }
        });
        const thumbPart = Slider.parts.thumb({
            dom: {
                tag: 'div',
                attributes: {
                    role: 'presentation'
                },
                classes: [getClass('sv-palette-thumb')],
                innerHtml: `<div class=${getClass('sv-palette-inner-thumb')} role="presentation"></div>`
            }
        });
        const setColour = (canvas, rgba) => {
            const { width, height } = canvas;
            const ctx = canvas.getContext('2d');
            if (ctx === null) {
                return;
            }
            ctx.fillStyle = rgba;
            ctx.fillRect(0, 0, width, height);
            const grdWhite = ctx.createLinearGradient(0, 0, width, 0);
            grdWhite.addColorStop(0, 'rgba(255,255,255,1)');
            grdWhite.addColorStop(1, 'rgba(255,255,255,0)');
            ctx.fillStyle = grdWhite;
            ctx.fillRect(0, 0, width, height);
            const grdBlack = ctx.createLinearGradient(0, 0, 0, height);
            grdBlack.addColorStop(0, 'rgba(0,0,0,0)');
            grdBlack.addColorStop(1, 'rgba(0,0,0,1)');
            ctx.fillStyle = grdBlack;
            ctx.fillRect(0, 0, width, height);
        };
        const setPaletteHue = (slider, hue) => {
            const canvas = slider.components()[0].element.dom;
            const hsv = hsvColour(hue, 100, 100);
            const rgba = fromHsv(hsv);
            setColour(canvas, toString(rgba));
        };
        const setPaletteThumb = (slider, hex) => {
            const hsv = fromRgb(fromHex(hex));
            Slider.setValue(slider, { x: hsv.saturation, y: 100 - hsv.value });
            set$9(slider.element, 'aria-valuetext', translate(['Saturation {0}%, Brightness {1}%', hsv.saturation, hsv.value]));
        };
        const factory = (_detail) => {
            const getInitialValue = constant$1({
                x: 0,
                y: 0
            });
            const onChange = (slider, _thumb, value) => {
                if (!isNumber(value)) {
                    set$9(slider.element, 'aria-valuetext', translate(['Saturation {0}%, Brightness {1}%', Math.floor(value.x), Math.floor(100 - value.y)]));
                }
                emitWith(slider, paletteUpdate, {
                    value
                });
            };
            const onInit = (_slider, _thumb, spectrum, _value) => {
                // Maybe make this initial value configurable?
                setColour(spectrum.element.dom, toString(red));
            };
            const sliderBehaviours = derive$1([
                Composing.config({
                    find: Optional.some
                }),
                Focusing.config({})
            ]);
            return Slider.sketch({
                dom: {
                    tag: 'div',
                    attributes: {
                        'role': 'slider',
                        'aria-valuetext': translate(['Saturation {0}%, Brightness {1}%', 0, 0])
                    },
                    classes: [getClass('sv-palette')]
                },
                model: {
                    mode: 'xy',
                    getInitialValue,
                },
                rounded: false,
                components: [
                    spectrumPart,
                    thumbPart
                ],
                onChange,
                onInit,
                sliderBehaviours
            });
        };
        const saturationBrightnessPaletteSketcher = single({
            factory,
            name: 'SaturationBrightnessPalette',
            configFields: [],
            apis: {
                setHue: (_apis, slider, hue) => {
                    setPaletteHue(slider, hue);
                },
                setThumb: (_apis, slider, hex) => {
                    setPaletteThumb(slider, hex);
                }
            },
            extraApis: {}
        });
        return saturationBrightnessPaletteSketcher;
    };

    const makeFactory = (translate, getClass, tooltipConfig, makeIcon) => {
        const factory = (detail) => {
            const rgbForm = rgbFormFactory(translate, getClass, detail.onValidHex, detail.onInvalidHex, tooltipConfig, makeIcon);
            const sbPalette = paletteFactory(translate, getClass);
            const hueSliderToDegrees = (hue) => (100 - hue) / 100 * 360;
            const hueDegreesToSlider = (hue) => 100 - (hue / 360) * 100;
            const state = {
                paletteRgba: Cell(red),
                paletteHue: Cell(0)
            };
            const memSlider = record(sliderFactory(translate, getClass));
            const memPalette = record(sbPalette.sketch({}));
            const memRgb = record(rgbForm.sketch({}));
            const updatePalette = (anyInSystem, _hex, hue) => {
                memPalette.getOpt(anyInSystem).each((palette) => {
                    sbPalette.setHue(palette, hue);
                });
            };
            const updateFields = (anyInSystem, hex) => {
                memRgb.getOpt(anyInSystem).each((form) => {
                    rgbForm.updateHex(form, hex);
                });
            };
            const updateSlider = (anyInSystem, _hex, hue) => {
                memSlider.getOpt(anyInSystem).each((slider) => {
                    Slider.setValue(slider, hueDegreesToSlider(hue));
                });
            };
            const updatePaletteThumb = (anyInSystem, hex) => {
                memPalette.getOpt(anyInSystem).each((palette) => {
                    sbPalette.setThumb(palette, hex);
                });
            };
            const updateState = (hex, hue) => {
                const rgba = fromHex(hex);
                state.paletteRgba.set(rgba);
                state.paletteHue.set(hue);
            };
            const runUpdates = (anyInSystem, hex, hue, updates) => {
                updateState(hex, hue);
                each$1(updates, (update) => {
                    update(anyInSystem, hex, hue);
                });
            };
            const onPaletteUpdate = () => {
                const updates = [updateFields];
                return (form, simulatedEvent) => {
                    const value = simulatedEvent.event.value;
                    const oldHue = state.paletteHue.get();
                    const newHsv = hsvColour(oldHue, value.x, (100 - value.y));
                    const newHex = hsvToHex(newHsv);
                    runUpdates(form, newHex, oldHue, updates);
                };
            };
            const onSliderUpdate = () => {
                const updates = [updatePalette, updateFields];
                return (form, simulatedEvent) => {
                    const hue = hueSliderToDegrees(simulatedEvent.event.value);
                    const oldRgb = state.paletteRgba.get();
                    const oldHsv = fromRgb(oldRgb);
                    const newHsv = hsvColour(hue, oldHsv.saturation, oldHsv.value);
                    const newHex = hsvToHex(newHsv);
                    runUpdates(form, newHex, hue, updates);
                };
            };
            const onFieldsUpdate = () => {
                const updates = [updatePalette, updateSlider, updatePaletteThumb];
                return (form, simulatedEvent) => {
                    const hex = simulatedEvent.event.hex;
                    const hsv = hexToHsv(hex);
                    runUpdates(form, hex, hsv.hue, updates);
                };
            };
            return {
                uid: detail.uid,
                dom: detail.dom,
                components: [
                    memPalette.asSpec(),
                    memSlider.asSpec(),
                    memRgb.asSpec()
                ],
                behaviours: derive$1([
                    config('colour-picker-events', [
                        run$1(fieldsUpdate, onFieldsUpdate()),
                        run$1(paletteUpdate, onPaletteUpdate()),
                        run$1(sliderUpdate, onSliderUpdate())
                    ]),
                    Composing.config({
                        find: (comp) => memRgb.getOpt(comp)
                    }),
                    Keying.config({
                        mode: 'acyclic'
                    })
                ])
            };
        };
        const colourPickerSketcher = single({
            name: 'ColourPicker',
            configFields: [
                required$1('dom'),
                defaulted('onValidHex', noop),
                defaulted('onInvalidHex', noop)
            ],
            factory
        });
        return colourPickerSketcher;
    };

    const english = {
        'colorcustom.rgb.red.label': 'R',
        'colorcustom.rgb.red.description': 'Red channel',
        'colorcustom.rgb.green.label': 'G',
        'colorcustom.rgb.green.description': 'Green channel',
        'colorcustom.rgb.blue.label': 'B',
        'colorcustom.rgb.blue.description': 'Blue channel',
        'colorcustom.rgb.hex.label': '#',
        'colorcustom.rgb.hex.description': 'Hex color code',
        'colorcustom.rgb.range': 'Range 0 to 255',
        'colorcustom.rgb.invalid': 'Numbers only, 0 to 255',
        'colorcustom.rgb.invalidHex': 'Hexadecimal only, 000000 to FFFFFF',
        'aria.color.picker': 'Color Picker',
        'aria.input.invalid': 'Invalid input'
    };
    const translate = (providerBackstage) => (key) => {
        if (isString(key)) {
            return providerBackstage.translate(english[key]);
        }
        else {
            return providerBackstage.translate(key);
        }
    };
    const renderColorPicker = (_spec, providerBackstage, initialData) => {
        const getClass = (key) => 'tox-' + key;
        const renderIcon = (name, errId, icon = name, label = name) => render$4(icon, {
            tag: 'div',
            classes: ['tox-icon', 'tox-control-wrap__status-icon-' + name],
            attributes: {
                'title': providerBackstage.translate(label),
                'aria-live': 'polite',
                ...errId.fold(() => ({}), (id) => ({ id }))
            }
        }, providerBackstage.icons);
        const colourPickerFactory = makeFactory(translate(providerBackstage), getClass, providerBackstage.tooltips.getConfig, renderIcon);
        const onValidHex = (form) => {
            emitWith(form, formActionEvent, { name: 'hex-valid', value: true });
        };
        const onInvalidHex = (form) => {
            emitWith(form, formActionEvent, { name: 'hex-valid', value: false });
        };
        const memPicker = record(colourPickerFactory.sketch({
            dom: {
                tag: 'div',
                classes: [getClass('color-picker-container')],
                attributes: {
                    role: 'presentation'
                }
            },
            onValidHex,
            onInvalidHex
        }));
        return {
            dom: {
                tag: 'div'
            },
            components: [
                memPicker.asSpec()
            ],
            behaviours: derive$1([
                // We'll allow invalid values
                withComp(initialData, (comp) => {
                    const picker = memPicker.get(comp);
                    const optRgbForm = Composing.getCurrent(picker);
                    const optHex = optRgbForm.bind((rgbForm) => {
                        const formValues = Representing.getValue(rgbForm);
                        return formValues.hex;
                    });
                    return optHex.map((hex) => '#' + removeLeading(hex, '#')).getOr('');
                }, (comp, newValue) => {
                    const pattern = /^#([a-fA-F0-9]{3}(?:[a-fA-F0-9]{3})?)/;
                    const valOpt = Optional.from(pattern.exec(newValue)).bind((matches) => get$i(matches, 1));
                    const picker = memPicker.get(comp);
                    const optRgbForm = Composing.getCurrent(picker);
                    optRgbForm.fold(() => {
                        // eslint-disable-next-line no-console
                        console.log('Can not find form');
                    }, (rgbForm) => {
                        Representing.setValue(rgbForm, {
                            hex: valOpt.getOr('')
                        });
                        // So not the way to do this.
                        Form.getField(rgbForm, 'hex').each((hexField) => {
                            emit(hexField, input());
                        });
                    });
                }),
                ComposingConfigs.self()
            ])
        };
    };

    var global$3 = tinymce.util.Tools.resolve('tinymce.Resource');

    const isOldCustomEditor = (spec) => has$2(spec, 'init');
    const renderCustomEditor = (spec) => {
        const editorApi = value$2();
        const memReplaced = record({
            dom: {
                tag: spec.tag
            }
        });
        const initialValue = value$2();
        const focusBehaviour = !isOldCustomEditor(spec) && spec.onFocus.isSome() ? [
            Focusing.config({
                onFocus: (comp) => {
                    spec.onFocus.each((onFocusFn) => {
                        onFocusFn(comp.element.dom);
                    });
                }
            }),
            Tabstopping.config({})
        ] : [];
        return {
            dom: {
                tag: 'div',
                classes: ['tox-custom-editor']
            },
            behaviours: derive$1([
                config('custom-editor-events', [
                    runOnAttached((component) => {
                        memReplaced.getOpt(component).each((ta) => {
                            // eslint-disable-next-line @typescript-eslint/no-floating-promises
                            (isOldCustomEditor(spec)
                                ? spec.init(ta.element.dom)
                                : global$3.load(spec.scriptId, spec.scriptUrl).then((init) => init(ta.element.dom, spec.settings))).then((ea) => {
                                initialValue.on((cvalue) => {
                                    ea.setValue(cvalue);
                                });
                                initialValue.clear();
                                editorApi.set(ea);
                            });
                        });
                    })
                ]),
                withComp(Optional.none(), () => editorApi.get().fold(() => initialValue.get().getOr(''), (ed) => ed.getValue()), (_component, value) => {
                    editorApi.get().fold(() => initialValue.set(value), (ed) => ed.setValue(value));
                }),
                ComposingConfigs.self()
            ].concat(focusBehaviour)),
            components: [memReplaced.asSpec()]
        };
    };

    var global$2 = tinymce.util.Tools.resolve('tinymce.util.Tools');

    const browseFilesEvent = generate$6('browse.files.event');
    const filterByExtension = (files, providersBackstage) => {
        const allowedImageFileTypes = global$2.explode(providersBackstage.getOption('images_file_types'));
        const isFileInAllowedTypes = (file) => exists(allowedImageFileTypes, (type) => endsWith(file.name.toLowerCase(), `.${type.toLowerCase()}`));
        return filter$2(from(files), isFileInAllowedTypes);
    };
    const renderDropZone = (spec, providersBackstage, initialData) => {
        // TODO: Consider moving to alloy
        const stopper = (_, se) => {
            se.stop();
        };
        // TODO: Consider moving to alloy
        const sequence = (actions) => (comp, se) => {
            each$1(actions, (a) => {
                a(comp, se);
            });
        };
        const onDrop = (comp, se) => {
            if (!Disabling.isDisabled(comp)) {
                const transferEvent = se.event.raw;
                emitWith(comp, browseFilesEvent, { files: transferEvent.dataTransfer?.files });
            }
        };
        const onSelect = (component, simulatedEvent) => {
            const input = simulatedEvent.event.raw.target;
            emitWith(component, browseFilesEvent, { files: input.files });
        };
        const handleFiles = (component, files) => {
            if (files) {
                Representing.setValue(component, filterByExtension(files, providersBackstage));
                emitWith(component, formChangeEvent, { name: spec.name });
            }
        };
        const memInput = record({
            dom: {
                tag: 'input',
                attributes: {
                    type: 'file',
                    accept: 'image/*'
                },
                styles: {
                    display: 'none'
                }
            },
            behaviours: derive$1([
                config('input-file-events', [
                    cutter(click()),
                    cutter(tap())
                ])
            ])
        });
        const pLabel = spec.label.map((label) => renderLabel$3(label, providersBackstage));
        const pField = FormField.parts.field({
            factory: Button,
            dom: {
                tag: 'button',
                styles: {
                    position: 'relative'
                },
                classes: ['tox-button', 'tox-button--secondary']
            },
            components: [
                text$2(providersBackstage.translate('Browse for an image')),
                memInput.asSpec()
            ],
            action: (comp) => {
                const inputComp = memInput.get(comp);
                inputComp.element.dom.click();
            },
            buttonBehaviours: derive$1([
                ComposingConfigs.self(),
                memory(initialData.getOr([])),
                Tabstopping.config({}),
                DisablingConfigs.button(() => providersBackstage.checkUiComponentContext(spec.context).shouldDisable),
                toggleOnReceive(() => providersBackstage.checkUiComponentContext(spec.context))
            ])
        });
        const wrapper = {
            dom: {
                tag: 'div',
                classes: ['tox-dropzone-container']
            },
            behaviours: derive$1([
                Disabling.config({
                    disabled: () => providersBackstage.checkUiComponentContext(spec.context).shouldDisable
                }),
                toggleOnReceive(() => providersBackstage.checkUiComponentContext(spec.context)),
                Toggling.config({
                    toggleClass: 'dragenter',
                    toggleOnExecute: false
                }),
                config('dropzone-events', [
                    run$1('dragenter', sequence([stopper, Toggling.toggle])),
                    run$1('dragleave', sequence([stopper, Toggling.toggle])),
                    run$1('dragover', stopper),
                    run$1('drop', sequence([stopper, onDrop])),
                    run$1(change(), onSelect)
                ])
            ]),
            components: [
                {
                    dom: {
                        tag: 'div',
                        classes: ['tox-dropzone'],
                        styles: {}
                    },
                    components: [
                        {
                            dom: {
                                tag: 'p'
                            },
                            components: [
                                text$2(providersBackstage.translate('Drop an image here'))
                            ]
                        },
                        pField
                    ]
                }
            ]
        };
        return renderFormFieldWith(pLabel, wrapper, ['tox-form__group--stretched'], [config('handle-files', [
                run$1(browseFilesEvent, (comp, se) => {
                    FormField.getField(comp).each((field) => {
                        handleFiles(field, se.event.files);
                    });
                })
            ])]);
    };

    const renderGrid = (spec, backstage) => ({
        dom: {
            tag: 'div',
            classes: ['tox-form__grid', `tox-form__grid--${spec.columns}col`]
        },
        components: map$2(spec.items, backstage.interpreter)
    });

    const beforeObject = generate$6('alloy-fake-before-tabstop');
    const afterObject = generate$6('alloy-fake-after-tabstop');
    const craftWithClasses = (classes) => {
        return {
            dom: {
                tag: 'div',
                styles: {
                    width: '1px',
                    height: '1px',
                    outline: 'none'
                },
                attributes: {
                    tabindex: '0' // Capture native tabbing in the appropriate order
                },
                classes
            },
            behaviours: derive$1([
                Focusing.config({ ignore: true }),
                Tabstopping.config({})
            ])
        };
    };
    const craft = (containerClasses, spec) => {
        return {
            dom: {
                tag: 'div',
                classes: ['tox-navobj', ...containerClasses.getOr([])]
            },
            components: [
                craftWithClasses([beforeObject]),
                spec,
                craftWithClasses([afterObject])
            ],
            behaviours: derive$1([
                ComposingConfigs.childAt(1)
            ])
        };
    };
    // TODO: Create an API in alloy to do this.
    const triggerTab = (placeholder, shiftKey) => {
        emitWith(placeholder, keydown(), {
            raw: {
                which: 9,
                shiftKey
            }
        });
    };
    const onFocus = (container, targetComp) => {
        const target = targetComp.element;
        // If focus has shifted naturally to a before object, the tab direction is backwards.
        if (has(target, beforeObject)) {
            triggerTab(container, true);
        }
        else if (has(target, afterObject)) {
            triggerTab(container, false);
        }
    };
    const isPseudoStop = (element) => {
        return closest$1(element, ['.' + beforeObject, '.' + afterObject].join(','), never);
    };

    const dialogChannel = generate$6('update-dialog');
    const titleChannel = generate$6('update-title');
    const bodyChannel = generate$6('update-body');
    const footerChannel = generate$6('update-footer');
    const bodySendMessageChannel = generate$6('body-send-message');
    const dialogFocusShiftedChannel = generate$6('dialog-focus-shifted');

    const browser = detect$1().browser;
    const isSafari = browser.isSafari();
    const isFirefox = browser.isFirefox();
    const isSafariOrFirefox = isSafari || isFirefox;
    const isChromium = browser.isChromium();
    const isElementScrollAtBottom = ({ scrollTop, scrollHeight, clientHeight }) => Math.ceil(scrollTop) + clientHeight >= scrollHeight;
    const scrollToY = (win, y) => 
    // TINY-10128: The iframe body is occasionally null when we attempt to scroll, so instead of using body.scrollHeight, use a
    // fallback value of 99999999. To minimise the potential impact of future browser changes, this fallback is significantly smaller
    // than the minimum of the maximum value Window.scrollTo would take on supported browsers:
    // Chromium: > Number.MAX_SAFE_INTEGER
    // Safari: 2^31 - 1 = 2147483647
    // Firefox: 2147483583
    win.scrollTo(0, y === 'bottom' ? 99999999 : y);
    const getScrollingElement = (doc, html) => {
        // TINY-10110: The scrolling element can change between body and documentElement depending on whether there
        // is a doctype declaration. However, this behavior is inconsistent on Chrome and Safari so checking for
        // the scroll properties is the most reliable way to determine which element is the scrolling element, at
        // least for the purposes of determining whether scroll is at bottom.
        const body = doc.body;
        return Optional.from(!/^<!DOCTYPE (html|HTML)/.test(html) &&
            (!isChromium && !isSafari || isNonNullable(body) && (body.scrollTop !== 0 || Math.abs(body.scrollHeight - body.clientHeight) > 1))
            ? body : doc.documentElement);
    };
    const writeValue = (iframeElement, html, fallbackFn) => {
        const iframe = iframeElement.dom;
        Optional.from(iframe.contentDocument).fold(fallbackFn, (doc) => {
            let lastScrollTop = 0;
            // TINY-10032: If documentElement (or body) is nullable, we assume document is empty and so scroll is at bottom.
            const isScrollAtBottom = getScrollingElement(doc, html).map((el) => {
                lastScrollTop = el.scrollTop;
                return el;
            }).forall(isElementScrollAtBottom);
            const scrollAfterWrite = () => {
                const win = iframe.contentWindow;
                if (isNonNullable(win)) {
                    if (isScrollAtBottom) {
                        scrollToY(win, 'bottom');
                    }
                    else if (!isScrollAtBottom && isSafariOrFirefox && lastScrollTop !== 0) {
                        // TINY-10078: Safari and Firefox reset scroll to top on each document.write(), so we need to restore scroll manually
                        scrollToY(win, lastScrollTop);
                    }
                }
            };
            // TINY-10109: On Safari, attempting to scroll before the iframe has finished loading will cause scroll to reset to top upon load.
            // TINY-10128: We will not wait for the load event on Chrome and Firefox since doing so causes the scroll to jump around erratically,
            // especially on Firefox. However, not waiting for load has the trade-off of potentially losing bottom scroll when updating at a very
            // rapid rate, as attempting to scroll before the iframe body is loaded will not work.
            if (isSafari) {
                iframe.addEventListener('load', scrollAfterWrite, { once: true });
            }
            doc.open();
            doc.write(html);
            doc.close();
            if (!isSafari) {
                scrollAfterWrite();
            }
        });
    };
    // TINY-10078: On Firefox, throttle to 200ms to improve scrolling experience. Since we are manually maintaining previous scroll position
    // on each update, when updating rapidly without a throttle, attempting to scroll around the iframe can feel stuck.
    // TINY-10097: On Safari, throttle to 500ms to reduce flickering as the document.write() method still observes significant flickering.
    // Also improves scrolling, as scroll positions are maintained manually similar to Firefox.
    const throttleInterval = someIf(isSafariOrFirefox, isSafari ? 500 : 200);
    // TINY-10078: Use Throttler.adaptable to ensure that any content added during the waiting period is not lost.
    const writeValueThrottler = throttleInterval.map((interval) => adaptable(writeValue, interval));
    const getDynamicSource = (initialData, stream) => {
        const cachedValue = Cell(initialData.getOr(''));
        return {
            getValue: (_frameComponent) => 
            // Ideally we should fetch data from the iframe...innerHtml, this triggers Cors errors
            cachedValue.get(),
            setValue: (frameComponent, html) => {
                if (cachedValue.get() !== html) {
                    const iframeElement = frameComponent.element;
                    const setSrcdocValue = () => set$9(iframeElement, 'srcdoc', html);
                    if (stream) {
                        writeValueThrottler.fold(constant$1(writeValue), (throttler) => throttler.throttle)(iframeElement, html, setSrcdocValue);
                    }
                    else {
                        // TINY-3769: We need to use srcdoc here, instead of src with a data URI, otherwise browsers won't retain the Origin.
                        // See https://bugs.chromium.org/p/chromium/issues/detail?id=58999#c11
                        setSrcdocValue();
                    }
                }
                cachedValue.set(html);
            }
        };
    };
    const renderIFrame = (spec, providersBackstage, initialData) => {
        const baseClass = 'tox-dialog__iframe';
        const opaqueClass = spec.transparent ? [] : [`${baseClass}--opaque`];
        const containerBorderedClass = spec.border ? [`tox-navobj-bordered`] : [];
        const attributes = {
            ...spec.label.map((title) => ({ title })).getOr({}),
            ...initialData.map((html) => ({ srcdoc: html })).getOr({}),
            ...spec.sandboxed ? { sandbox: 'allow-scripts allow-same-origin' } : {}
        };
        const sourcing = getDynamicSource(initialData, spec.streamContent);
        const pLabel = spec.label.map((label) => renderLabel$3(label, providersBackstage));
        const factory = (newSpec) => craft(Optional.from(containerBorderedClass), {
            // We need to use the part uid or the label and field won't be linked with ARIA
            uid: newSpec.uid,
            dom: {
                tag: 'iframe',
                attributes,
                classes: [
                    baseClass,
                    ...opaqueClass
                ]
            },
            behaviours: derive$1([
                Tabstopping.config({}),
                Focusing.config({}),
                withComp(initialData, sourcing.getValue, sourcing.setValue),
                Receiving.config({
                    channels: {
                        [dialogFocusShiftedChannel]: {
                            onReceive: (comp, message) => {
                                message.newFocus.each((newFocus) => {
                                    parentElement(comp.element).each((parent) => {
                                        const f = eq(comp.element, newFocus) ? add$2 : remove$3;
                                        f(parent, 'tox-navobj-bordered-focus');
                                    });
                                });
                            }
                        }
                    }
                })
            ])
        });
        // Note, it's not going to handle escape at this point.
        const pField = FormField.parts.field({
            factory: { sketch: factory }
        });
        return renderFormFieldWith(pLabel, pField, ['tox-form__group--stretched'], []);
    };

    const calculateImagePosition = (panelWidth, panelHeight, imageWidth, imageHeight, zoom) => {
        const width = imageWidth * zoom;
        const height = imageHeight * zoom;
        const left = Math.max(0, panelWidth / 2 - width / 2);
        const top = Math.max(0, panelHeight / 2 - height / 2);
        return {
            left: left.toString() + 'px',
            top: top.toString() + 'px',
            width: width.toString() + 'px',
            height: height.toString() + 'px',
        };
    };
    const zoomToFit = (panel, width, height) => {
        const panelW = get$c(panel);
        const panelH = get$d(panel);
        return Math.min(panelW / width, panelH / height, 1);
    };
    const renderImagePreview = (spec, initialData) => {
        const cachedData = Cell(initialData.getOr({ url: '' }));
        const memImage = record({
            dom: {
                tag: 'img',
                classes: ['tox-imagepreview__image'],
                attributes: initialData.map((data) => ({ src: data.url })).getOr({})
            },
        });
        const memContainer = record({
            dom: {
                tag: 'div',
                classes: ['tox-imagepreview__container'],
                attributes: {
                    role: 'presentation'
                },
            },
            components: [
                memImage.asSpec()
            ]
        });
        const setValue = (frameComponent, data) => {
            const translatedData = {
                url: data.url
            };
            // update properties that are set by the data
            data.zoom.each((z) => translatedData.zoom = z);
            data.cachedWidth.each((z) => translatedData.cachedWidth = z);
            data.cachedHeight.each((z) => translatedData.cachedHeight = z);
            cachedData.set(translatedData);
            const applyFramePositioning = () => {
                const { cachedWidth, cachedHeight, zoom } = translatedData;
                if (!isUndefined(cachedWidth) && !isUndefined(cachedHeight)) {
                    if (isUndefined(zoom)) {
                        const z = zoomToFit(frameComponent.element, cachedWidth, cachedHeight);
                        // sneaky mutation since we own the object
                        translatedData.zoom = z;
                    }
                    const position = calculateImagePosition(get$c(frameComponent.element), get$d(frameComponent.element), cachedWidth, cachedHeight, translatedData.zoom);
                    memContainer.getOpt(frameComponent).each((container) => {
                        setAll(container.element, position);
                    });
                }
            };
            memImage.getOpt(frameComponent).each((imageComponent) => {
                const img = imageComponent.element;
                if (data.url !== get$g(img, 'src')) {
                    set$9(img, 'src', data.url);
                    remove$3(frameComponent.element, 'tox-imagepreview__loaded');
                }
                applyFramePositioning();
                // eslint-disable-next-line @typescript-eslint/no-floating-promises
                image(img).then((img) => {
                    // Ensure the component hasn't been removed while the image was loading
                    // if it is disconnected, just do nothing
                    if (frameComponent.getSystem().isConnected()) {
                        add$2(frameComponent.element, 'tox-imagepreview__loaded');
                        // sneaky mutation since we own the object
                        translatedData.cachedWidth = img.dom.naturalWidth;
                        translatedData.cachedHeight = img.dom.naturalHeight;
                        applyFramePositioning();
                    }
                });
            });
        };
        const styles = {};
        spec.height.each((h) => styles.height = h);
        // TODO: TINY-8393 Use the initial data properly once it's validated
        const fakeValidatedData = initialData.map((d) => ({
            url: d.url,
            zoom: Optional.from(d.zoom),
            cachedWidth: Optional.from(d.cachedWidth),
            cachedHeight: Optional.from(d.cachedHeight),
        }));
        return {
            dom: {
                tag: 'div',
                classes: ['tox-imagepreview'],
                styles,
                attributes: {
                    role: 'presentation'
                }
            },
            components: [
                memContainer.asSpec(),
            ],
            behaviours: derive$1([
                ComposingConfigs.self(),
                withComp(fakeValidatedData, () => 
                /*
                  NOTE: This is intentionally returning the cached image width and height.
      
                  Including those details in the dialog data helps when `setData` only changes the URL, as
                  the old image must continue to be displayed at the old size until the new image has loaded.
                */
                cachedData.get(), setValue),
            ])
        };
    };

    const renderLabel$2 = (spec, backstageShared, getCompByName) => {
        const baseClass = 'tox-label';
        const centerClass = spec.align === 'center' ? [`${baseClass}--center`] : [];
        const endClass = spec.align === 'end' ? [`${baseClass}--end`] : [];
        const label = record({
            dom: {
                tag: 'label',
                classes: [baseClass, ...centerClass, ...endClass]
            },
            components: [
                text$2(backstageShared.providers.translate(spec.label))
            ]
        });
        const comps = map$2(spec.items, backstageShared.interpreter);
        return {
            dom: {
                tag: 'div',
                classes: ['tox-form__group']
            },
            components: [
                label.asSpec(),
                ...comps
            ],
            behaviours: derive$1([
                ComposingConfigs.self(),
                Replacing.config({}),
                domHtml(Optional.none()),
                Keying.config({
                    mode: 'acyclic'
                }),
                config('label', [
                    runOnAttached((comp) => {
                        spec.for.each((name) => {
                            getCompByName(name).each((target) => {
                                label.getOpt(comp).each((labelComp) => {
                                    const id = get$g(target.element, 'id') ?? generate$6('form-field');
                                    set$9(target.element, 'id', id);
                                    set$9(labelComp.element, 'for', id);
                                });
                            });
                        });
                    })
                ]),
            ])
        };
    };

    const internalToolbarButtonExecute = generate$6('toolbar.button.execute');
    // Perform `action` when an item is clicked on, close menus, and stop event
    const onToolbarButtonExecute = (info) => runOnExecute$1((comp, _simulatedEvent) => {
        // If there is an action, run the action
        runWithApi(info, comp)((itemApi) => {
            emitWith(comp, internalToolbarButtonExecute, {
                buttonApi: itemApi
            });
            info.onAction(itemApi);
        });
    });
    const commonButtonDisplayEvent = generate$6('common-button-display-events');
    const toolbarButtonEventOrder = {
        // TODO: use the constants provided by behaviours.
        [execute$5()]: ['disabling', 'alloy.base.behaviour', 'toggling', 'toolbar-button-events', 'tooltipping'],
        [attachedToDom()]: [
            'toolbar-button-events',
            commonButtonDisplayEvent
        ],
        [detachedFromDom()]: ['toolbar-button-events', 'dropdown-events', 'tooltipping'],
        [mousedown()]: [
            'focusing',
            'alloy.base.behaviour',
            commonButtonDisplayEvent
        ]
    };

    const forceInitialSize = (comp) => set$7(comp.element, 'width', get$e(comp.element, 'width'));

    const renderIcon$1 = (iconName, iconsProvider, behaviours) => render$4(iconName, {
        tag: 'span',
        classes: ["tox-icon" /* ToolbarButtonClasses.Icon */, "tox-tbtn__icon-wrap" /* ToolbarButtonClasses.IconWrap */],
        behaviours
    }, iconsProvider);
    const renderIconFromPack$1 = (iconName, iconsProvider) => renderIcon$1(iconName, iconsProvider, []);
    const renderReplaceableIconFromPack = (iconName, iconsProvider) => renderIcon$1(iconName, iconsProvider, [Replacing.config({})]);
    const renderLabel$1 = (text, prefix, providersBackstage) => ({
        dom: {
            tag: 'span',
            classes: [`${prefix}__select-label`]
        },
        components: [
            text$2(providersBackstage.translate(text))
        ],
        behaviours: derive$1([
            Replacing.config({})
        ])
    });

    const updateMenuText = generate$6('update-menu-text');
    const updateMenuIcon = generate$6('update-menu-icon');
    const updateTooltiptext = generate$6('update-tooltip-text');
    // TODO: Use renderCommonStructure here.
    const renderCommonDropdown = (spec, prefix, sharedBackstage, btnName) => {
        const editorOffCell = Cell(noop);
        const tooltip = Cell(spec.tooltip);
        // We need mementos for display text and display icon because on the events
        // updateMenuText and updateMenuIcon respectively, their contents are changed
        // via Replacing. These events are generally emitted by dropdowns that want the
        // main text and icon to match the current selection (e.g. bespokes like font family)
        const optMemDisplayText = spec.text.map((text) => record(renderLabel$1(text, prefix, sharedBackstage.providers)));
        const optMemDisplayIcon = spec.icon.map((iconName) => record(renderReplaceableIconFromPack(iconName, sharedBackstage.providers.icons)));
        /*
         * The desired behaviour here is:
         *
         *   when left or right is pressed, and it isn't associated with expanding or
         *   collapsing a submenu, then it should navigate to the next menu item, and
         *   expand it (without highlighting any items in the expanded menu).
         *   It also needs to close the previous menu
         */
        const onLeftOrRightInMenu = (comp, se) => {
            // The originating dropdown is stored on the sandbox itself. This is just an
            // implementation detail of alloy. We really need to make it a fully-fledged API.
            // TODO: TINY-9014 Make SandboxAPI have a function that just delegates to Representing
            const dropdown = Representing.getValue(comp);
            // Focus the dropdown. Current workaround required to make FlowLayout recognise the current focus.
            // The triggering keydown is going to try to move the focus left or
            // right of the current menu, so it needs to know what the current menu dropdown is. It
            // can't work it out by the current focus, because the current focus is *in* the menu, so
            // we help it by moving the focus to the button, so it can work out what the next menu to
            // the left or right is.
            Focusing.focus(dropdown);
            emitWith(dropdown, 'keydown', {
                raw: se.event.raw
            });
            // Because we have just navigated off this open menu, we want to close it.
            // INVESTIGATE: TINY-9014: Is this handling situations where there were no menus
            // to move to? Does it matter if we still close it when there are no other menus?
            Dropdown.close(dropdown);
            // The Optional.some(true) tells the keyboard handler that this event was handled,
            // which will do things like stopPropagation and preventDefault.
            return Optional.some(true);
        };
        const role = spec.role.fold(() => ({}), (role) => ({ role }));
        const listRole = Optional.from(spec.listRole).map((listRole) => ({ listRole })).getOr({});
        const ariaLabelAttribute = spec.ariaLabel.fold(() => ({}), (ariaLabel) => {
            const translatedAriaLabel = sharedBackstage.providers.translate(ariaLabel);
            return {
                'aria-label': translatedAriaLabel
            };
        });
        const iconSpec = render$4('chevron-down', {
            tag: 'div',
            classes: [`${prefix}__select-chevron`]
        }, sharedBackstage.providers.icons);
        const fixWidthBehaviourName = generate$6('common-button-display-events');
        // Should we use Id.generate here?
        const customEventsName = 'dropdown-events';
        const memDropdown = record(Dropdown.sketch({
            ...spec.uid ? { uid: spec.uid } : {},
            ...role,
            ...listRole,
            dom: {
                tag: 'button',
                classes: [prefix, `${prefix}--select`].concat(map$2(spec.classes, (c) => `${prefix}--${c}`)),
                attributes: {
                    ...ariaLabelAttribute,
                    ...(isNonNullable(btnName) ? { 'data-mce-name': btnName } : {})
                }
            },
            components: componentRenderPipeline([
                optMemDisplayIcon.map((mem) => mem.asSpec()),
                optMemDisplayText.map((mem) => mem.asSpec()),
                Optional.some(iconSpec)
            ]),
            matchWidth: true,
            useMinWidth: true,
            // When the dropdown opens, if we are in search mode, then we want to
            // focus our searcher.
            onOpen: (anchor, dropdownComp, tmenuComp) => {
                if (spec.searchable) {
                    focusSearchField(tmenuComp);
                }
            },
            dropdownBehaviours: derive$1([
                ...spec.dropdownBehaviours,
                DisablingConfigs.button(() => spec.disabled || sharedBackstage.providers.checkUiComponentContext(spec.context).shouldDisable),
                toggleOnReceive(() => sharedBackstage.providers.checkUiComponentContext(spec.context)),
                // INVESTIGATE (TINY-9012): There was a old comment here about something not quite working, and that
                // we can still get the button focused. It was probably related to Unselecting.
                Unselecting.config({}),
                Replacing.config({}),
                ...(spec.tooltip.map((t) => Tooltipping.config(sharedBackstage.providers.tooltips.getConfig({
                    tooltipText: sharedBackstage.providers.translate(t),
                    onShow: (comp) => {
                        if (lift2(tooltip.get(), spec.tooltip, (tooltipStr, tt) => tt !== tooltipStr).getOr(false)) {
                            const translatedTooltip = sharedBackstage.providers.translate(tooltip.get().getOr(''));
                            Tooltipping.setComponents(comp, sharedBackstage.providers.tooltips.getComponents({ tooltipText: translatedTooltip }));
                        }
                    }
                })))).toArray(),
                // This is the generic way to make onSetup and onDestroy call as the component is attached /
                // detached from the page/DOM.
                config(customEventsName, [
                    onControlAttached(spec, editorOffCell),
                    onControlDetached(spec, editorOffCell)
                ]),
                config(fixWidthBehaviourName, [
                    runOnAttached((comp, _se) => {
                        if (spec.listRole !== 'listbox') {
                            forceInitialSize(comp);
                        }
                    }),
                ]),
                config('update-dropdown-width-variable', [
                    run$1(windowResize(), (comp, _se) => Dropdown.close(comp)),
                ]),
                config('menubutton-update-display-text', [
                    // These handlers are just using Replacing to replace either the menu
                    // text or the icon.
                    run$1(updateMenuText, (comp, se) => {
                        optMemDisplayText.bind((mem) => mem.getOpt(comp)).each((displayText) => {
                            Replacing.set(displayText, [text$2(sharedBackstage.providers.translate(se.event.text))]);
                        });
                    }),
                    run$1(updateMenuIcon, (comp, se) => {
                        optMemDisplayIcon.bind((mem) => mem.getOpt(comp)).each((displayIcon) => {
                            Replacing.set(displayIcon, [
                                renderReplaceableIconFromPack(se.event.icon, sharedBackstage.providers.icons)
                            ]);
                        });
                    }),
                    run$1(updateTooltiptext, (comp, se) => {
                        const translatedTooltip = sharedBackstage.providers.translate(se.event.text);
                        set$9(comp.element, 'aria-label', translatedTooltip);
                        tooltip.set(Optional.some(se.event.text));
                    })
                ])
            ]),
            eventOrder: deepMerge(toolbarButtonEventOrder, {
                // INVESTIGATE (TINY-9014): Explain why we need the events in this order.
                // Ideally, have a test that fails when they are in a different order if order
                // is important
                [mousedown()]: ['focusing', 'alloy.base.behaviour', 'item-type-events', 'normal-dropdown-events'],
                [attachedToDom()]: [
                    '