# Copyright © The Debusine Developers
# See the AUTHORS file at the top-level directory of this distribution
#
# This file is part of Debusine. It is subject to the license terms
# in the LICENSE file found in the top-level directory of this
# distribution. No part of Debusine, including this file, may be copied,
# modified, propagated, or distributed except according to the terms
# contained in the LICENSE file.

"""UI helpers for collections."""

from functools import cached_property

from django.urls import reverse

from debusine.db.models import Collection, CollectionItem
from debusine.web.views.places import IndexPlace, ResourcePlace
from debusine.web.views.ui.base import UI


class CollectionUI(UI[Collection]):
    """UI helpers for Collection instances."""

    @cached_property
    def place(self) -> ResourcePlace:
        """Return a place to show this collection."""
        place = ResourcePlace(
            title=f"{self.instance.name}@{self.instance.category} collection",
            breadcrumb=self.instance.name,
            url=self.instance.get_absolute_url(),
            description=f"{self.instance.name}@{self.instance.category}"
            " collection",
        )
        place.parent = self.instance.workspace.ui(
            self.request
        ).place_collection_list
        return place

    @cached_property
    def place_search(self) -> IndexPlace:
        """Return a place to search this collection."""
        return IndexPlace(
            title=f"Search {self.instance.name}@{self.instance.category}",
            url=self.instance.get_absolute_url_search(),
            description=f"{self.instance} collection search",
            parent=self.instance.workspace.ui(
                self.request
            ).place_collection_list,
        )

    @cached_property
    def place_update(self) -> IndexPlace:
        """Return a place to configure this collection."""
        return IndexPlace(
            title="Configure",
            url=reverse(
                "workspaces:collections:update",
                kwargs={
                    "wname": self.instance.workspace.name,
                    "cname": self.instance.name,
                    "ccat": self.instance.category,
                },
            ),
            parent=self.place,
        )

    @cached_property
    def place_delete(self) -> IndexPlace:
        """Return a place to delete this collection."""
        return IndexPlace(
            title="Delete",
            url=reverse(
                "workspaces:collections:delete",
                kwargs={
                    "wname": self.instance.workspace.name,
                    "cname": self.instance.name,
                    "ccat": self.instance.category,
                },
            ),
            parent=self.place,
        )


class CollectionItemUI(UI[CollectionItem]):
    """UI helpers for CollectionItem instances."""

    @cached_property
    def place(self) -> ResourcePlace:
        """Return a place to show this collection item."""
        return ResourcePlace(
            title=self.instance.name,
            breadcrumb=self.instance.name,
            url=self.instance.get_absolute_url(),
            parent=self.instance.parent_collection.ui(self.request).place,
        )
