/*
 * Copyright (c) Members of the EGEE Collaboration. 2004-2010. 
 * See http://www.eu-egee.org/partners/ for details on the copyright
 * holders.  
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); 
 * you may not use this file except in compliance with the License. 
 * You may obtain a copy of the License at 
 * 
 *     http://www.apache.org/licenses/LICENSE-2.0 
 * 
 * Unless required by applicable law or agreed to in writing, software 
 * distributed under the License is distributed on an "AS IS" BASIS, 
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. 
 * See the License for the specific language governing permissions and 
 * limitations under the License.
 *
 */

/**
   \defgroup LcmapsOpensslInterface LCMAPS openssl dependent interfaces
   
    \brief  This is a collection of interfaces that use openssl types
            and functions.

    The interface function is declared as a type, since the function is not
    pre-defined when using dlopen().

    -# lcmaps_run_with_stack_of_x509_and_return_account():
        To do the user mapping, based on a certificate chain and return the account information

    \ingroup LcmapsInterface
*/

/**
   \file   lcmaps_openssl.h
   \brief  Stack of X509 interface of the LCMAPS library.
   \author Oscar Koeroo for the EGEE project.
   \author Martijn Steenbakkers for the EU DataGrid.
  
    This header contains the declarations of the LCMAPS library functions:
    -# lcmaps_run_with_stack_of_x509_and_return_account():
	To do the user mapping, based on a certificate chain and return the
	account information
    \ingroup LcmapsOpensslInterface
*/

#ifndef LCMAPS_X509_H
#define LCMAPS_X509_H

/******************************************************************************
                             Include header files
******************************************************************************/
#include <sys/types.h>
#include <openssl/x509.h>
#include "lcmaps_basic.h"

/******************************************************************************
 *                 Module definition
 *****************************************************************************/


/**
   \brief Run LCMAPS and map to an account based on the supplied certificate chain.
   
   LCMAPS runs receiving a certificate chain, containing at least an
   End-Entity Certificate.  A list of policies may be provided. The
   allocated uid, gids and the poolindex will be returned to the
   calling application.

    \param cert_chain   The certificate chain to use for the mapping
    \param request      RSL string (input)
    \param npols        number of policies to be considered for evaluation (input)
    \param policynames  the names of the policies to be considered for evaluation (input)
    \param puid         the uid of the account
    \param ppgid_list   the list of primary gids of the account
    \param pnpgid       the number of primary gids of the account
    \param psgid_list   the list of secondary gids of the account
    \param pnsgid       the number of secondary gids of the account
    \param poolindexp   poolindex string of the account

    \retval 0 mapping succeeded
    \retval 1 mapping failed
*/
typedef int lcmaps_run_with_stack_of_x509_and_return_account_t(
        STACK_OF(X509) *  cert_chain,
        int               mapcounter,
        lcmaps_request_t  request,
        int               npols,
        char **           policynames,
        uid_t *           puid,
        gid_t **          ppgid_list,
        int *             pnpgid,
        gid_t **          psgid_list,
        int *             pnsgid,
        char **           poolindexp
);
#ifndef LCMAPS_USE_DLOPEN
lcmaps_run_with_stack_of_x509_and_return_account_t lcmaps_run_with_stack_of_x509_and_return_account;
#endif /* LCMAPS_USE_DLOPEN */

#endif /* LCMAPS_X509_H */
