/**
* @file protoList.cpp
* 
* @brief The ProtoList class provides a simple double linked-list
* class with a "ProtoList::Item" base class to use for
* deriving your own classes you wish to store in a 
* ProtoList
*/

#include "protoList.h"

ProtoList::Item::Item()
{
}

ProtoList::Item::~Item()
{
}

ProtoList::ProtoList()
 : head(NULL), tail(NULL)
{
}

ProtoList::~ProtoList()
{
}

void ProtoList::Prepend(Item& item)
{
    item.plist_prev = NULL;
    if (NULL != head)
        head->plist_prev = &item;
    else
        tail = &item;
    item.plist_next = head;
    head = &item;
}  // end ProtoList::Prepend()

void ProtoList::Append(Item& item)
{
    item.plist_next = NULL;
    if (NULL != tail)
        tail->plist_next = &item;
    else
        head = &item;
    item.plist_prev = tail;
    tail = &item;
}  // end ProtoList::Append()

void ProtoList::Remove(Item& item)
{
    if (NULL == item.plist_prev)
        head = item.plist_next;
    else
        item.plist_prev->plist_next = item.plist_next;
    
    if (NULL == item.plist_next)
        tail = item.plist_prev;
    else
        item.plist_next->plist_prev = item.plist_prev;
    item.plist_prev = item.plist_next = NULL;
}  // end ProtoList::Remove()

ProtoList::Item* ProtoList::RemoveHead()
{
    Item* item = head;
    if (NULL != item) Remove(*item);
    return item;
}  // end ProtoList::RemoveHead()


void ProtoList::Destroy()
{
    Item* item = head;
    while (NULL != item)
    {
        Remove(*item);
        delete item;
        item = head;
    }
}  // end ProtoList::Destroy()

ProtoList::Iterator::Iterator(const ProtoList& theList, bool reverse)
 : list(theList)
{
    Reset(reverse);
}

ProtoList::Iterator::~Iterator()
{
}

ProtoList::Item* ProtoList::Iterator::GetNextItem()
{
    if (reversed)
    {
        item = (NULL != item) ? item->plist_next : list.head; 
        reversed = false;
    }
    if (NULL != item)  
    {
        Item* next = item;
        item = item->plist_next;
        return next;
    }
    else
    {
        return NULL;
    }
}  // end ProtoList::Iterator::GetNextItem()    

ProtoList::Item* ProtoList::Iterator::GetPrevItem()
{
    if (!reversed)
    {
        item = (NULL != item) ? item->plist_prev : list.tail; 
        reversed = true;
    }
    if (NULL != item)  
    {
        Item* prev = item;
        item = item->plist_prev;
        return prev;
    }
    else
    {
        return NULL;
    }
}  // end ProtoList::Iterator::GetPrevItem()  

ProtoStack::ProtoStack()
 : head(NULL), tail(NULL)
{
}

ProtoStack::~ProtoStack()
{
}


void ProtoStack::Push(Item& item)
{
    item.pstack_next = head;
    head = &item;
    if (NULL == tail) tail = &item;
}  // end ProtoStack::Push(Item& item)


ProtoStack::Item* ProtoStack::Pop()
{
    Item* item = head;  
    if (NULL != item)
    {
        if (NULL != item->pstack_next)
            head = item->pstack_next;
        else
            head = tail = NULL;
    }   
    return item;
}  // end ProtoStack::Pop()

void ProtoStack::Put(Item& item)
{
    item.pstack_next = NULL;
    if (NULL != tail)
        tail->pstack_next = &item;
    else
        head = tail = &item;
}  // end ProtoStack::Put()

void ProtoStack::Destroy()
{
    Item* item;
    while (NULL != (item = Pop()))
        delete item;
}  // end ProtoStack::Destroy()

ProtoStack::Item::Item()
 : pstack_next(NULL)
{
}

ProtoStack::Item::~Item()
{
}

ProtoStack::Iterator::Iterator(const ProtoStack& theStack)
 : stack(theStack), next(theStack.GetHead())
{
}

ProtoStack::Iterator::~Iterator()
{
}

ProtoStack::Item* ProtoStack::Iterator::GetNextItem()
{
    Item* item = next;
    if (NULL != item)
        next = item->pstack_next;
    return item;
}  // end ProtoStack::Iterator::GetNextItem()
