<?php

/****************************************
 RefDB interface library
 
 This file defines an object-oriented interface to RefDB. The current
 implementation uses calls to the C clients instead of a direct
 communication with refdbd through sockets.
 
 Markus Hoenicka <mhoenicka@users.sourceforge.net> 2007-11-02

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
   
   You should have received a copy of the GNU General Public License
   along with this program; if not, see <http://www.gnu.org/licenses/>

********************************/

class RefDB {
  var $usernamearg; /* username for db authentication */
  var $passwordarg; /* password for db authentication */
  var $dbarg;       /* name of database to connect to */
  var $pdf_rootarg; /* base path of offprint repository */
  var $client;      /* full path of refdbc or refdba binary */


  function __construct($username, $password, $db, $pdf_root, $client) {
    $this->usernamearg = strlen($username) ? " -u ".$username : "";
    $this->passwordarg = strlen($password) ? " -w ".$password : "";
    $this->dbarg = strlen($db) ? " -d ".$db : "";
    $this->pdf_rootarg = strlen($pdf_root) ? " -R ".$pdf_root : "";
    $this->client = $client;

    /* this is the full path of the binary with all mandatory
     arguments. We send the output to stdout to bypass any pager that
     may lurk in the user's config files */
    $this->clientcmd = $this->client.$this->usernamearg.$this->passwordarg.$this->dbarg." -c stdout ";
  }

  protected function _run_command($command) {
    $lines = "";
    $exit_state = 0;

    //        echo "<p>DEBUG: running command $command</p>";
    exec($command, $lines, $exit_state);
    //        echo "<p>DEBUG: exit state went to $exit_state</p>";
    switch($exit_state) {
    case 0:
      return $lines;
      break;
    default:
      return array("ERROR");
    }
  }

  public function addstyle_from_file($file) {
    $command = $this->clientcmd." -C addstyle ".escapeshellarg($file);
    return $this->_run_command($command);
  }

  public function adduser($host, $database, $newuserpasswd, $username) {
    $command = $this->clientcmd." -C adduser -d ".escapeshellarg($database);

    if (strlen($newuserpasswd) > 0) {
      $command .= " -W ".$newuserpasswd;
    }

    if (strlen($host) > 0) {
      $command .= " -H ".$host;
    }

    $command .= $username;

    return $this->_run_command($command);
  }

  public function deleteuser($host, $database, $username) {
    $command = $this->clientcmd." -C deleteuser -d ".escapeshellarg($database);

    if (strlen($host) > 0) {
      $command .= " -H ".$host;
    }

    $command .= $username;

    return $this->_run_command($command);
  }

  public function addword($words) {
    $command = $this->clientcmd." -C addword ".escapeshellarg($words);
    return $this->_run_command($command);
  }

  public function deleteword($words) {
    $command = $this->clientcmd." -C deleteword ".escapeshellarg($words);
    return $this->_run_command($command);
  }

  public function confserv($command) {
    $command = $this->clientcmd." -C confserv ".escapeshellarg($command);
    return $this->_run_command($command);
  }

  public function createdb($dbname, $encoding) {
    $command = $this->clientcmd." -C createdb ";
    
    if (strlen($encoding) > 0) {
      $command .= " -E ".escapeshellarg($encoding);
    }

    $command .= " ".escapeshellarg($dbname);

    return $this->_run_command($command);
  }

  public function deletedb($dbname) {
    $command = $this->clientcmd." -C deletedb ".escapeshellarg($dbname);
    return $this->_run_command($command);
  }

  public function deletestyle($style) {
    $command = $this->clientcmd." -C deletestyle ".escapeshellarg($style);
    return $this->_run_command($command);
  }

  public function getstyle($style) {
    $command = $this->clientcmd." -C getstyle ".escapeshellarg($style);
    return $this->_run_command($command);
  }

  public function listdb($dbname) {
    $command = $this->clientcmd." -C listdb ".escapeshellarg($dbname);
    return $this->_run_command($command);
  }

  public function listuser($dbname, $user) {
    $command = $this->clientcmd." -C listuser -d ".escapeshellarg($dbname)." ".escapeshellarg($user);
    return $this->_run_command($command);
  }

  public function listword($word) {
    $command = $this->clientcmd." -C listword ".escapeshellarg($word);
    return $this->_run_command($command);
  }

  public function liststyle($style) {
    $command = $this->clientcmd." -C liststyle ".escapeshellarg($style);
    return $this->_run_command($command);
  }

  public function viewstat() {
    $command = $this->clientcmd." -C viewstat";
    return $this->_run_command($command);
  }

  public function scankw($dbname) {
    $command = $this->clientcmd." -C scankw ".escapeshellarg($dbname);
    return $this->_run_command($command);
  }

  public function addref($owner, $refdata, $type, $encoding) {
    return array("ERROR");
  }

  public function addref_from_file($owner, $file, $type, $encoding) {
    $command = $this->clientcmd." -C addref -A $type ".escapeshellarg($file);
    if (strlen($encoding) > 0) {
      $command .= " -E $encoding";
    }
    return $this->_run_command($command);
  }

  public function updateref($owner, $is_personal, $refdata, $type, $encoding) {
    return array("ERROR");
  }

  public function updateref_from_file($owner, $file, $type, $encoding) {
    $command = $this->clientcmd." -C updateref -A $type ".escapeshellarg($file);
    if (strlen($encoding) > 0) {
      $command .= " -E $encoding";
    }
    return $this->_run_command($command);
  }

  public function checkref($risdata, $type, $encoding, $outtype) {
    return array("ERROR");
  }

  public function checkref_from_file($owner, $file, $type, $encoding) {
    $command = $this->clientcmd." -C checkref -A $type ".escapeshellarg($file);
    if (strlen($encoding) > 0) {
      $command .= " -E $encoding";
    }
    return $this->_run_command($command);
  }

  public function deleteref($idlist) {
    $command = $this->clientcmd." -C deleteref ".escapeshellarg($idlist);
    return $this->_run_command($command);
  }

  public function addnote($owner, $xnotedata) {
    return array("ERROR");
  }

  public function addnote_from_file($owner, $file) {
    $command = $this->clientcmd." -C addnote ".escapeshellarg($file);
    return $this->_run_command($command);
  }

  public function updatenote($owner, $xnotedata) {
    return array("ERROR");
  }

  public function updatenote_from_file($owner, $file) {
    $command = $this->clientcmd." -C note ".escapeshellarg($file);
    return $this->_run_command($command);
  }

  public function deletenote($idlist) {
    $command = $this->clientcmd." -C deletenote ".escapeshellarg($idlist);
    return $this->_run_command($command);
  }

  public function addlink($linkspec) {
    $command = $this->clientcmd." -C addlink ".escapeshellarg($idlist);
    return $this->_run_command($command);
  }

  public function deletelink($linkspec) {
    $command = $this->clientcmd." -C deletelink ".escapeshellarg($idlist);
    return $this->_run_command($command);
  }

  public function getas($limit_string, $freq, $name) {
    $command = $this->clientcmd." -C getas ".escapeshellarg($name);
    if (strlen($freq) > 0) {
      $command .= " -s $freq";
    }
    if (strlen($limit_string) > 0) {
      $command .= " -N $limit_string";
    }
    return $this->_run_command($command);
  }

  public function getau($limit_string, $freq, $name) {
    $command = $this->clientcmd." -C getau ".escapeshellarg($name);
    if (strlen($freq) > 0) {
      $command .= " -s $freq";
    }
    if (strlen($limit_string) > 0) {
      $command .= " -N $limit_string";
    }
    return $this->_run_command($command);
  }

  public function getax($limit_string, $freq, $name) {
    $command = $this->clientcmd." -C getax ".escapeshellarg($name);
    if (strlen($freq) > 0) {
      $command .= " -s $freq";
    }
    if (strlen($limit_string) > 0) {
      $command .= " -N $limit_string";
    }
    return $this->_run_command($command);
  }

  public function geted($limit_string, $freq, $name) {
    $command = $this->clientcmd." -C geted ".escapeshellarg($name);
    if (strlen($freq) > 0) {
      $command .= " -s $freq";
    }
    if (strlen($limit_string) > 0) {
      $command .= " -N $limit_string";
    }
    return $this->_run_command($command);
  }

  public function getkw($limit_string, $freq, $name) {
    $command = $this->clientcmd." -C getkw ".escapeshellarg($name);
    if (strlen($freq) > 0) {
      $command .= " -s $freq";
    }
    if (strlen($limit_string) > 0) {
      $command .= " -N $limit_string";
    }
    return $this->_run_command($command);
  }

  public function getjf($is_all, $limit_string, $freq, $name) {
    $command = $this->clientcmd." -C getjf ".escapeshellarg($name);
    if (strlen($freq) > 0) {
      $command .= " -s $freq";
    }
    if (strlen($limit_string) > 0) {
      $command .= " -N $limit_string";
    }
    if ($is_all=="t") {
      $command .= " -a ";
    }
    return $this->_run_command($command);
  }

  public function getjo($is_all, $limit_string, $freq, $name) {
    $command = $this->clientcmd." -C getjo ".escapeshellarg($name);
    if (strlen($freq) > 0) {
      $command .= " -s $freq";
    }
    if (strlen($limit_string) > 0) {
      $command .= " -N $limit_string";
    }
    if ($is_all=="t") {
      $command .= " -a ";
    }
    return $this->_run_command($command);
  }

  public function getj1($is_all, $limit_string, $freq, $name) {
    $command = $this->clientcmd." -C getj1 ".escapeshellarg($name);
    if (strlen($freq) > 0) {
      $command .= " -s $freq";
    }
    if (strlen($limit_string) > 0) {
      $command .= " -N $limit_string";
    }
    if ($is_all=="t") {
      $command .= " -a ";
    }
    return $this->_run_command($command);
  }

  public function getj2($is_all, $limit_string, $freq, $name) {
    $command = $this->clientcmd." -C getj2 ".escapeshellarg($name);
    if (strlen($freq) > 0) {
      $command .= " -s $freq";
    }
    if (strlen($limit_string) > 0) {
      $command .= " -N $limit_string";
    }
    if ($is_all=="t") {
      $command .= " -a ";
    }
    return $this->_run_command($command);
  }

  public function getref($type, $format_string, $sort_string, $listname, $encoding, $limit_string, $namespace, $frequency, $query) {
    $command = $this->clientcmd." -C getref ";
    if (strlen($type) > 0) {
      $command .= " -t ".escapeshellarg($type);
    }
    if (strlen($format_string) > 0) {
      $command .= " -s ".escapeshellarg($format_string);
    }
    if (strlen($sort_string) > 0) {
      $command .= " -S ".escapeshellarg($sort_string);
    }
    if (strlen($listname) > 0) {
      $command .= " -b ".escapeshellarg($listname);
    }
    if (strlen($encoding) > 0) {
      $command .= " -E ".escapeshellarg($encoding);
    }
    if (strlen($limit_string) > 0) {
      $command .= " -N ".escapeshellarg($limit_string);
    }
    if (strlen($namespace) > 0) {
      $command .= " -n ".escapeshellarg($namespace);
    }
    if (strlen($query) > 0) {
      $command .= " ".escapeshellarg($query);
      //      $command .= " ".$query;
    }
    if (strlen($frequency) > 0 && $frequency != 0) {
      $command .= " -Q";
    }

    return $this->_run_command($command);
  }

  public function countref($listname, $limit_string, $query) {
    $command = $this->clientcmd." -C countref ";
    if (strlen($listname) > 0) {
      $command .= " -b ".escapeshellarg($listname);
    }
    if (strlen($limit_string) > 0) {
      $command .= " -N ".escapeshellarg($limit_string);
    }
    if (strlen($query) > 0) {
      $command .= " ".escapeshellarg($query);
    }
    
    // countref returns nothing on stdout, the summary appears
    // on stderr. Redirect the latter to stdout so run_command
    // will read it as regular output
    $command .= " 2>&1";
    return $this->_run_command($command);
  }

  public function pickref($idlist, $listname) {
    $command = $this->clientcmd." -C pickref ";
    if (strlen($listname) > 0) {
      $command .= " -b ".escapeshellarg($listname);
    }

    $command .= " ".escapeshellarg($idlist);
    
    return $this->_run_command($command);
  }

  public function dumpref($idlist, $listname) {
    $command = $this->clientcmd." -C dumpref ";
    if (strlen($listname) > 0) {
      $command .= " -b ".escapeshellarg($listname);
    }

    $command .= " ".escapeshellarg($idlist);
    
    return $this->_run_command($command);
  }

  public function getnote($type, $format_string, $sort_string, $encoding, $limit_string, $namespace, $query) {
    $command = $this->clientcmd." -C getnote ";
    if (strlen($type) > 0) {
      $command .= " -t ".escapeshellarg($type);
    }
    if (strlen($format_string) > 0) {
      $command .= " -s ".escapeshellarg($format_string);
    }
    if (strlen($sort_string) > 0) {
      $command .= " -S ".escapeshellarg($sort_string);
    }
    if (strlen($encoding) > 0) {
      $command .= " -E ".escapeshellarg($encoding);
    }
    if (strlen($limit_string) > 0) {
      $command .= " -N ".escapeshellarg($limit_string);
    }
    if (strlen($namespace) > 0) {
      $command .= " -n ".escapeshellarg($namespace);
    }
    if (strlen($query) > 0) {
      $command .= " ".escapeshellarg($query);
    }
    
    return $this->_run_command($command);
  }

  public function countnote($listname, $limit_string, $query) {
    $command = $this->clientcmd." -C countnote ";
    if (strlen($limit_string) > 0) {
      $command .= " -N ".escapeshellarg($limit_string);
    }
    if (strlen($query) > 0) {
      $command .= " ".escapeshellarg($query);
    }
    
    return $this->_run_command($command);
  }

  public function selectdb($dbname) {
    $this->dbarg = strlen($dbname) ? " -d ".$dbname : "";
    $this->clientcmd = $this->client.$this->usernamearg.$this->passwordarg.$this->dbarg." -c stdout ";
    return array($dbname);
  }

  public function whichdb() {
    $command = $this->clientcmd." -C whichdb";
    return $this->_run_command($command);
  }

  public function updatejo($names) {
    $command = $this->clientcmd." -C updatejo ".escapeshellarg($names);
    return $this->_run_command($command);
  }

  public function texbib($style, $cite_data) {
    return array("ERROR");
  }

  public function dbib($style, $encoding, $cite_data) {
    return array("ERROR");
  }

  public function getrefx($type, $encoding, $cite_data) {
    return array("ERROR");
  }

  public function getversion() {
    $command = $this->clientcmd." -v";
    return $this->_run_command($command);
  }

} /* end class RefDB */

?>
