# Copyright © The Debusine Developers
# See the AUTHORS file at the top-level directory of this distribution
#
# This file is part of Debusine. It is subject to the license terms
# in the LICENSE file found in the top-level directory of this
# distribution. No part of Debusine, including this file, may be copied,
# modified, propagated, or distributed except according to the terms
# contained in the LICENSE file.

"""UI helpers for workspaces."""

from functools import cached_property

from django.urls import reverse

from debusine.db.context import context
from debusine.db.models import Workspace
from debusine.web.icons import Icons
from debusine.web.views.places import (
    ContainerPlace,
    IndexPlace,
    Place,
    ResourcePlace,
)
from debusine.web.views.ui.base import UI


class WorkspaceUI(UI[Workspace]):
    """UI helpers for Workspace instances."""

    @cached_property
    def place(self) -> ResourcePlace:
        """Return a place to show this workspace."""
        public = " public" if self.instance.public else ""
        place = ResourcePlace(
            title=self.instance.name,
            description=f"{self.instance.name}{public} workspace",
            url=self.instance.get_absolute_url(),
            icon=Icons.WORKSPACE,
            breadcrumb=self.instance.name,
        )
        place.parent = self.instance.scope.ui(self.request).place
        return place

    @cached_property
    def place_update(self) -> IndexPlace:
        """Return a place to update this workspace."""
        parent = self.place
        assert parent.description is not None
        return IndexPlace(
            title="Configure",
            url=self.instance.get_absolute_url_configure(),
            description=parent.description + " configuration",
            parent=parent,
        )

    @cached_property
    def place_config_inspector(self) -> IndexPlace:
        """Return a place to inspect this workspace task configuration."""
        from debusine.server.scopes import urlconf_scope

        with urlconf_scope(self.instance.scope.name):
            url = reverse(
                "workspaces:task_configuration_inspector",
                kwargs={"wname": self.instance.name},
            )

        parent = self.place
        assert parent.description is not None
        return IndexPlace(
            title="Task configuration inspector",
            url=url,
            description=parent.description + " task configuration inspector",
            parent=parent,
        )

    @cached_property
    def place_update_inheritance(self) -> Place:
        """Return a place to update this workspace's inheritance."""
        from debusine.server.scopes import urlconf_scope

        with urlconf_scope(self.instance.scope.name):
            url = reverse(
                "workspaces:update_inheritance",
                kwargs={"wname": self.instance.name},
            )

        parent = self.place
        assert parent.description is not None
        return IndexPlace(
            title="Configure inheritance",
            url=url,
            description=parent.description + " inheritance configuration",
            parent=parent,
        )

    @cached_property
    def place_workflow_list(self) -> ContainerPlace:
        """Return a place to list this workspace's workflows."""
        from debusine.server.scopes import urlconf_scope

        with urlconf_scope(self.instance.scope.name):
            url = reverse(
                "workspaces:workflows:list",
                kwargs={"wname": self.instance.name},
            )

        parent = self.place
        assert parent.description is not None
        return ContainerPlace(
            title="Workflow list",
            url=url,
            breadcrumb="workflows",
            description=parent.description + " workflow list",
            parent=parent,
        )

    @cached_property
    def place_collection_create(self) -> IndexPlace:
        """Return a place to create a collection in this workspace."""
        from debusine.server.scopes import urlconf_scope

        with urlconf_scope(self.instance.scope.name):
            url = reverse(
                "workspaces:collections:create",
                kwargs={"wname": self.instance.name},
            )

        parent = self.place
        assert parent.description is not None
        place = IndexPlace(
            title="Create collection",
            url=url,
            description="Create a new collection in " + parent.description,
            parent=parent,
        )
        return place

    @cached_property
    def place_collection_list(self) -> ContainerPlace:
        """Return a place to list this workspace's collections."""
        from debusine.server.scopes import urlconf_scope

        with urlconf_scope(self.instance.scope.name):
            url = reverse(
                "workspaces:collections:list",
                kwargs={"wname": self.instance.name},
            )

        parent = self.place
        assert parent.description is not None
        place = ContainerPlace(
            title="Collections",
            url=url,
            description=parent.description + " collection list",
            parent=parent,
            breadcrumb="collections",
        )
        return place

    @cached_property
    def candidate_parents(self) -> list[Workspace]:
        """Return a list of workspaces that can be parents to this one."""
        base_qs = Workspace.objects.select_related("scope")
        allowed = base_qs.can_display(context.user).exclude(pk=self.instance.pk)
        allowed = allowed.union(
            base_qs.filter(pk__in=self.instance.chain_parents.values("parent"))
        )
        return list(allowed.order_by("scope", "name"))
