# Copyright © The Debusine Developers
# See the AUTHORS file at the top-level directory of this distribution
#
# This file is part of Debusine. It is subject to the license terms
# in the LICENSE file found in the top-level directory of this
# distribution. No part of Debusine, including this file, may be copied,
# modified, propagated, or distributed except according to the terms
# contained in the LICENSE file.

"""UI helpers for artifacts."""

from functools import cached_property

from debusine.artifacts.models import ArtifactCategory
from debusine.db.models import Artifact, FileInArtifact
from debusine.db.models.artifacts import ARTIFACT_CATEGORY_ICON_NAMES
from debusine.web.views.places import ResourcePlace
from debusine.web.views.ui.base import UI


class ArtifactUI(UI[Artifact]):
    """UI helpers for Artifact instances."""

    @cached_property
    def place(self) -> ResourcePlace:
        """Return a place to show this artifact."""
        return ResourcePlace(
            title=f"Artifact {self.instance.get_label()}",
            breadcrumb=self.instance.get_label(),
            url=self.instance.get_absolute_url(),
            icon=ARTIFACT_CATEGORY_ICON_NAMES.get(
                ArtifactCategory(self.instance.category), "file"
            ),
            description=f"{self.instance.category} artifact"
            f" {self.instance.get_label()}",
            parent=self.instance.workspace.ui(self.request).place,
        )


class FileInArtifactUI(UI[FileInArtifact]):
    """UI helpers for FileInArtifact instances."""

    @cached_property
    def place(self) -> ResourcePlace:
        """Return a place to show this file."""
        title = str(self.instance.path)
        if not self.instance.complete:
            title += " (incomplete)"
        return ResourcePlace(
            title=title,
            breadcrumb=str(self.instance.path),
            url=self.instance.get_absolute_url(),
            parent=self.instance.artifact.ui(self.request).place,
        )

    @cached_property
    def place_raw(self) -> ResourcePlace:
        """Return a place to show this file (raw view)."""
        title = str(self.instance.path)
        if not self.instance.complete:
            title += " (incomplete)"
        return ResourcePlace(
            title=title,
            breadcrumb=str(self.instance.path),
            url=self.instance.get_absolute_url_raw(),
            parent=self.instance.artifact.ui(self.request).place,
        )

    @cached_property
    def place_download(self) -> ResourcePlace:
        """Return a place to show this file (download view)."""
        title = str(self.instance.path)
        if not self.instance.complete:
            title += " (incomplete)"
        return ResourcePlace(
            title=title,
            breadcrumb=str(self.instance.path),
            url=self.instance.get_absolute_url_download(),
            parent=self.instance.artifact.ui(self.request).place,
        )
