/**
 * Copyright (C) 2003 Billy Biggs <vektor@dumbterm.net>.
 *
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, sublicense, and/or sell copies of the Software, and to
 * permit persons to whom the Software is furnished to do so, subject to
 * the following conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
 * ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include "exroutput.h"
#include "pnginput.h"

static double voltage_to_intensity_srgb( double val )
{
    if( val <= 0.04045 ) {
        return val / 12.92;
    }
    return pow( ( val + 0.055 ) / 1.055, 2.4 );
}

int main( int argc, char **argv )
{
    exroutput_t *exrout;
    pnginput_t *pngin;
    double *scanline;
    int width, height;
    int y;

    if( argc < 3 ) {
        fprintf( stderr, "Usage: pngtoexr input.png output.exr\n" );
        return 1;
    }

    pngin = pnginput_new( argv[ 1 ] );
    if( !pngin ) {
        fprintf( stderr, "pngtoexr: Cannot open %s\n", argv[ 1 ] );
        return 1;
    }
    width = pnginput_get_width( pngin );
    height = pnginput_get_height( pngin );
    fprintf( stdout, "pngtoexr: %s [%dx%d]\n", argv[ 1 ], width, height );

    scanline = malloc( width * 4 * sizeof( double ) );
    if( !scanline ) {
        fprintf( stderr, "pngtoexr: Cannot allocate memory.\n" );
        pnginput_delete( pngin );
        return 1;
    }

    exrout = exroutput_new( argv[ 2 ], width, height );
    if( !exrout ) {
        fprintf( stderr, "pngtoexr: Cannot open %s\n", argv[ 2 ] );
        pnginput_delete( pngin );
        free( scanline );
        return 1;
    }

    for( y = 0; y < height; y++ ) {
        uint8_t *curin = pnginput_get_scanline( pngin, y );
        int x;

        fprintf( stderr, "pngtoexr: Writing scanline %d\r", y );

        for( x = 0; x < width; x++ ) {
            double rp, gp, bp, a;

            if( pnginput_has_alpha( pngin ) ) {
                rp = ((double) curin[ (x * 4) + 0 ]) / 255.0;
                gp = ((double) curin[ (x * 4) + 1 ]) / 255.0;
                bp = ((double) curin[ (x * 4) + 2 ]) / 255.0;
                a  = ((double) curin[ (x * 4) + 3 ]) / 255.0;
            } else {
                rp = ((double) curin[ (x * 3) + 0 ]) / 255.0;
                gp = ((double) curin[ (x * 3) + 1 ]) / 255.0;
                bp = ((double) curin[ (x * 3) + 2 ]) / 255.0;
                a = 1.0;
            }

            scanline[ (x * 4) + 0 ] = voltage_to_intensity_srgb( rp );
            scanline[ (x * 4) + 1 ] = voltage_to_intensity_srgb( gp );
            scanline[ (x * 4) + 2 ] = voltage_to_intensity_srgb( bp );
            scanline[ (x * 4) + 3 ] = a;
        }

        exroutput_scanline( exrout, scanline );
    }
    fprintf( stderr, "\npngtoexr: Done.\n" );

    exroutput_delete( exrout );
    pnginput_delete( pngin );
    free( scanline );
    return 0;
}

