/***************************************************************************
 *   Copyright (C) 2004-2006 by Jim Campbell                               *
 *   ifpgui@gmail.com                                                   *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include "radiofreqdlg.h"
#include "gui_user.h"
#include "backend/ifpinterface.h"

#include <qmessagebox.h>


RadioFreqWindow::RadioFreqWindow(QWidget *parent, Qt::WFlags f)
        : QDialog( parent, f | Qt::CustomizeWindowHint|Qt::WindowTitleHint|Qt::WindowSystemMenuHint)
{
    QHBoxLayout *WindowFrame = new QHBoxLayout(this);
    QFrame *frame = new QFrame();
    WindowFrame->addWidget(frame);

    setWindowTitle(tr("Set Radio Frequencies"));

    // todo hide min/max buttons on window

    freqLayout = new QGridLayout();
    frame->setLayout(freqLayout);
    frame->setMinimumSize(280, 430);
    frame->setMaximumSize(400, 800);

    freqTable = new QTableWidget(IFP_PRESET_TOTAL, 2, this);
    freqTable->setHorizontalHeaderItem(0, new QTableWidgetItem(tr("Station")));
    freqTable->setHorizontalHeaderItem(1, new QTableWidgetItem(tr("Frequency (MHz)")));
    freqTable->setColumnWidth (0, 100);
    freqTable->setColumnWidth (1, 120);
    freqLayout->addWidget(freqTable, 0, 0);

    QHBoxLayout *Btnframe = new QHBoxLayout();
    btnDone   = new QPushButton(tr("Done"), this);
    btnDone->setMinimumSize(80, 30);
    Btnframe->addWidget(btnDone);

    btnCancel = new QPushButton(tr("Cancel"), this);
    btnCancel->setMinimumSize(80, 30);
    Btnframe->addWidget(btnCancel);
    freqLayout->addLayout(Btnframe, 1, 0);

    setMinimumSize(290, 485);
    setMaximumSize(350, 700);
    resize( QSize(290, 485).expandedTo(minimumSizeHint()) );
    //clearWState( WState_Polished );

    initalizeDisplaySettings();

    // signals and slots connections
    connect( btnDone, SIGNAL( clicked() ), this, SLOT( btnDone_clicked() ) );
    connect( btnCancel, SIGNAL( clicked() ), this, SLOT( btnCancel_clicked() ) );
    connect( freqTable, SIGNAL( cellChanged(int,int) ), this, SLOT( tableValueChanged(int,int) ) );

    // tab order
    setTabOrder( freqTable, btnDone );
    setTabOrder( btnDone, btnCancel );
}

/*
 *  Destroys the object and frees any allocated resources
 */
RadioFreqWindow::~RadioFreqWindow()
{
    // no need to delete child widgets, Qt does it all for us
}



void RadioFreqWindow::btnDone_clicked()
{
    finalizeDisplaySettings();
    close();
}

void RadioFreqWindow::btnCancel_clicked()
{
    reject();
}


void RadioFreqWindow::initalizeDisplaySettings(void)
{
    char label[IFP_TUNER_LABEL + 1];
    int freq;
    int ret;
    int cnt;
    QString err;
    QTableWidgetItem *newItem;

    if (ifp_dh == NULL)
        return;
    ret = ifp_get_tuner_presets(&ifp_dev, buf, sizeof(buf));
    if (ret != IFP_OK)
    {
        err = tr("ERROR: Could not GET radio frequency presets.");
        showInfoMsgBox(tr("Retrieve Radio Frequency Status"), err);
        return;
    }

    for (cnt = 0; cnt != IFP_PRESET_TOTAL; cnt++)
    {
        ret = ifpgui_ifp_get_station(cnt, buf, label, &freq);
        if (ret != IFP_OK)
        {
            err = tr("Could not get station #%1.").arg(cnt + 1);
            showInfoMsgBox(tr("Retrieve Radio Frequency Status"), err);
            return;
        }
        newItem = new QTableWidgetItem(label);
        freqTable->setItem(cnt, 0, newItem);

        newItem = new QTableWidgetItem(QString("%1.%2").arg(freq/100).arg(freq%100));
        freqTable->setItem(cnt, 1, newItem);
    }
}


void RadioFreqWindow::finalizeDisplaySettings(void)
{
    int ret;
    QString err;

    if (ifp_dh == NULL)
        return;
    ret = ifpgui_ifp_set_tuner_presets(&ifp_dev, buf, sizeof(buf));
    if (ret != IFP_OK)
    {
        err = QObject::tr("ERROR: Could not SET radio frequency presets.");
        showInfoMsgBox(tr("Retrieve Radio Frequency Status"), err);
        return;
    }
}

void RadioFreqWindow::tableValueChanged(int row, int col)
{
    bool ok;
    double dbl;
    int freq, ret;
    QString slabel, err;
    char label[IFP_TUNER_LABEL + 1];

    if (col)
    {
        // satisfy compilier
    }

    if (ifp_dh == NULL)
        return;

    {
        dbl = freqTable->item(row, 1)->text().toDouble(&ok);
        if (!ok)
        {
            err = tr("Error: frequency is not valid.\n\nEnter a value between 87.5 and 108.0 (MHz)");
            showInfoMsgBox(tr("Set Radio Frequency Status"), err);

            ret = ifpgui_ifp_get_station(row, buf, label, &freq);
            if (ret == IFP_OK)
            {
                freqTable->item(row, 1)->setText(QString("%1.%2").arg(freq/100).arg(freq%100));
            }
            else
            {
                freq = IFP_FREQ_MIN;
            }
        }
        else
        {
            freq = (int)((dbl + .005) * 100);
        }
        if ((freq < IFP_FREQ_MIN) || (freq > IFP_FREQ_MAX))
        {
            err = QObject::tr("Error: frequency out of range.\n\nEnter a value between 87.5 and 108.0 (MHz)");
            showInfoMsgBox(tr("Set Radio Frequency Status"), err);
            if (freq < IFP_FREQ_MIN)
            {
                freq = IFP_FREQ_MIN;
            }
            else
            {
                freq = IFP_FREQ_MAX;
            }
            freqTable->item(row, 1)->setText(QString("%1.%2").arg(freq/100).arg(freq%100));
        }
        else
        {
            // make sure formatted properly
            freqTable->item(row, 1)->setText(QString("%1.%2").arg(freq/100).arg(freq%100));
        }
    }

    slabel = freqTable->item(row, 0)->text();
    if (slabel.length() > 6)
    {
        err = tr("Warning: station label can only be 6 character.");
        showInfoMsgBox(tr("Set Radio Frequency Status"), err);
        slabel.truncate(IFP_TUNER_LABEL);
        freqTable->item(row, 0)->setText(slabel);
    }
    //strcpy(label, slabel);

    //------------------------------------------------------
    // finally store the validated settings
    //------------------------------------------------------
    ret = ifpgui_ifp_set_station(row, buf, slabel.toUtf8().data(), freq);
    if (ret != IFP_OK)
    {
        err = tr("Error: Could not store settings for station %1").arg(row);
        showInfoMsgBox(tr("Set Radio Frequency Status"), err);
    }
}

