/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2012 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "iconfigure.h"
#if ISHELL_INCLUDED(ISHELL_GG)


#include "iggpagevolume.h"


#include "icontrolmodule.h"
#include "iimagefactory.h"
#include "ishell.h"
#include "iviewmodule.h"
#include "ivolumeviewsubject.h"

#include "iggdatatypeprovider.h"
#include "iggframedatavariablelist.h"
#include "iggframefunctionmapping.h"
#include "iggframematerialproperties.h"
#include "iggframepaletteselection.h"
#include "iggframereplicate.h"
#include "iggmainwindow.h"
#include "iggrenderwindow.h"
#include "iggwidgetkeybutton.h"
#include "iggwidgetkeyselectionbox.h"
#include "iggwidgetkeyslider.h"
#include "iggwidgetmisc.h"
#include "iggwidgetotherbutton.h"

#include "ibgframesubject.h"
#include "ibgwidgetbuttonsubject.h"

#include "iggsubjectfactory.h"

//
//  Templates (needed for some compilers)
//
#include "iarraytemplate.h"
#include"iggwidgetkeyslidertemplate.h"


using namespace iParameter;
using namespace iParameter;


namespace iggPageVolume_Private
{
	//
	//  Helper widgets
	//
	class WindowSizeFrame : public iggFrame
	{
		
	public:
		
		WindowSizeFrame(iggFrame *parent) : iggFrame("Window size",parent,1)
		{
			mSubject1 = iggSubjectFactory::CreateWidgetButtonSubject(this,ButtonType::PushButton,"256x256",1);
			mSubject2 = iggSubjectFactory::CreateWidgetButtonSubject(this,ButtonType::PushButton,"512x512",2);
			mSubject3 = iggSubjectFactory::CreateWidgetButtonSubject(this,ButtonType::PushButton,"1024x1024",3);

			mSubject->PlaceWidget(0,0,mSubject1,1,true);
			mSubject->PlaceWidget(0,1,mSubject2,1,true);
			mSubject->PlaceWidget(0,2,mSubject3,1,true);

			this->SetBaloonHelp("Sets the size of the current visualization window to a power of 2","When using the VolumePro board for volume rendering, IFrIT relies on OpenGL textures to display the scene. Since OpenGL textures must have dimensions that are powers of two, the performance is optimal when the visualization window size corresponds exactly to the texture size. These buttons will quickly resize the visualization window to be a power of two size.");
		}

	protected:
		
		virtual void OnVoid1Body()
		{
			static int iv[2] = { 256, 256};

			iString ws;
			this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeySize(),iv,2);
			this->GetShell()->GetControlModule()->Execute(ws,this->GetMainWindow()->IsAutoRender(),this->GetExecuteFlags());
		}

		virtual void OnVoid2Body()
		{
			static int iv[2] = { 512, 512};

			iString ws;
			this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeySize(),iv,2);
			this->GetShell()->GetControlModule()->Execute(ws,this->GetMainWindow()->IsAutoRender(),this->GetExecuteFlags());
		}

		virtual void OnVoid3Body()
		{
			static int iv[2] = { 1024, 1024};

			iString ws;
			this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeySize(),iv,2);
			this->GetShell()->GetControlModule()->Execute(ws,this->GetMainWindow()->IsAutoRender(),this->GetExecuteFlags());
		}

		ibgWidgetButtonSubject *mSubject1, *mSubject2, *mSubject3;
	};


	class MethodBox : public iggWidgetKeyComboBox
	{

	public:

		MethodBox(iggFrame *parent) : iggWidgetKeyComboBox("",0,iVolumeViewSubject::KeyMethod(),parent)
		{
			this->InsertItem("A volume rendering method");
		}

	protected:

		virtual void UpdateWidgetBody()
		{
			int i;
			iString s;
			//
			//  Optimization: direct access
			//
			iVolumeViewSubject *vo = this->GetShell()->GetControlModule()->GetViewModule()->GetVolumeViewSubject();

			this->Clear();
			i = 0;
			while(!(s = vo->GetMethodName(i)).IsEmpty())
			{
				this->SetItem(s,i,vo->IsMethodAvailable(i));
				i++;
			}

			this->iggWidgetKeyComboBox::UpdateWidgetBody();
		}
	};


	//
	//  Main page
	// ************************************************
	//
	class MainPage : public iggFrameDC
	{

	public:

		MainPage(iggFrameBase *parent, iggDataTypeProvider *provider) : iggFrameDC(parent,3), mProvider(provider)
		{
			if(this->ImmediateConstruction()) this->CompleteConstruction();
		}

	protected:

		iggDataTypeProvider *mProvider;

		virtual void CompleteConstructionBody()
		{
			//
			//  Show
			//
			iggWidget *show = new iggWidgetShowButton(iVolumeViewSubject::Type(),this);
			this->AddLine(show);
			this->AddSpace(2);
			//
			//  Field & method
			//
			iggFrame *tmp = new iggFrame("Method",this);
			tmp->AddLine(new MethodBox(tmp));
			this->AddLine(tmp);

			iggFrameDataVariableList *pl = new iggFrameDataVariableList(mProvider,"Field",iVolumeViewSubject::KeyVar(),0,this,0);
			pl->Complete();
			pl->AddDependent(show);
#ifdef VTK_USE_VOLUMEPRO
			this->AddLine(pl,new iggPageVolume_Private::WindowSizeFrame(this));
#else
			this->AddLine(pl);
#endif
			this->AddSpace(10);

			this->SetColStretch(2,10);
		}
	};


	//
	//  Paint page
	// ************************************************
	//
	class PaintPage : public iggFrameDC
	{

	public:

		PaintPage(iggFrameBase *parent) : iggFrameDC(parent,1)
		{
			if(this->ImmediateConstruction()) this->CompleteConstruction();
		}

	protected:

		virtual void CompleteConstructionBody()
		{
			const iImage *icon = iImageFactory::FindIcon("volv.png");
			//
			//  Book
			//
			iggFrameBook *pb = new iggFrameBook(this);
			this->AddLine(pb);
			//
			//  Palette page
			//
			iggFrame *pbpage0 = new iggFrame(pb,2);
			pb->AddPage("Palette",icon,pbpage0);
			{
				iggFramePaletteSelection *ps = new iggFramePaletteSelection(false,iVolumeViewSubject::KeyPalette(),pbpage0,0);
				pbpage0->AddLine(ps);
				pbpage0->AddSpace(20);

				pbpage0->SetColStretch(0,10);
				pbpage0->SetColStretch(1,3);
			}
			//
			//  Material page
			//
			iggFrame *pbpage1 = new iggFrame(pb,2);
			pb->AddPage("Material",icon,pbpage1);
			{
				iggFrameMaterialProperties *mp = new iggFrameMaterialProperties(true,false,iVolumeViewSubject::Type(),pbpage1);
				pbpage1->AddLine(mp);

				pbpage1->AddSpace(10);
				pbpage1->SetColStretch(0,10);
				pbpage1->SetColStretch(1,3);
			}
		}
	};


	//
	//  Opacity page
	// ************************************************
	//
	class OpacityPage : public iggFrameDC
	{

	public:

		OpacityPage(iggFrameBase *parent) : iggFrameDC(parent,1)
		{
			if(this->ImmediateConstruction()) this->CompleteConstruction();
		}

	protected:

		virtual void CompleteConstructionBody()
		{
			//
			//  FunctionMapping widget
			//
			this->AddLine(new iggFrameFunctionMapping(iVolumeViewSubject::KeyOpacityFunction(),this));
		}
	};


	//
	//  Replicate page
	// ************************************************
	//
	class ReplicatePage : public iggFrameDC
	{

	public:

		ReplicatePage(iggFrameBase *parent) : iggFrameDC(parent,2)
		{
			if(this->ImmediateConstruction()) this->CompleteConstruction();
		}

	protected:

		virtual void CompleteConstructionBody()
		{
			//
			//  Replicate
			//
			this->AddLine(new iggFrameReplicate(iVolumeViewSubject::Type(),this));
			this->AddSpace(10);
			this->SetColStretch(1,3);
		}
	};


	//
	//  Extras page
	// ************************************************
	//
	class ExtrasPage : public iggFrameDC
	{

	public:

		ExtrasPage(iggFrameBase *parent) : iggFrameDC(parent,3)
		{
			if(this->ImmediateConstruction()) this->CompleteConstruction();
		}

	protected:

		virtual void CompleteConstructionBody()
		{
			//
			//  Ray casting settings
			//
			iggFrame *rc = new iggFrame("Rendering settings",this,2);
			iggWidgetKeyRadioBox *it = new iggWidgetKeyRadioBox(1,"Interpolation type",0,iVolumeViewSubject::KeyInterpolationType(),rc);
			it->InsertItem("Nearest neighbor");
			it->InsertItem("Linear");
			iggWidgetKeyRadioBox *bm = new iggWidgetKeyRadioBox(1,"Blending mode",0,iVolumeViewSubject::KeyBlendMode(),rc);
			bm->InsertItem("Composite");
			bm->InsertItem("Maximum intensity");
			rc->AddLine(it,bm);
			this->AddLine(rc);
			this->AddSpace(10);

			iggFrame *rs = new iggFrame("Downsampling settings",this,1);
			iggWidgetKeyFloatSlider *fs = new iggWidgetKeyFloatSlider(0.1f,10.0f,20,1,4,"Image downsampling",iVolumeViewSubject::KeyImageDownsampleFactor(),RenderMode::UseGlobal,rs);
			fs->SetStretch(3,10);
			rs->AddLine(fs);
			fs = new iggWidgetKeyFloatSlider(0.1f,10.0f,20,1,4,"Depth downsampling",iVolumeViewSubject::KeyDepthDownsampleFactor(),RenderMode::UseGlobal,rs);
			fs->SetStretch(3,10);
			rs->AddLine(fs);

			this->AddLine(rs,2);
			this->AddSpace(10);

			this->SetColStretch(1,10);
			this->SetColStretch(2,3);
		}
	};
};


using namespace iggPageVolume_Private;


iggPageVolume::iggPageVolume(iggFrameBase *parent) : iggPageObject(parent,ViewSubject::Id::Volume)
{
	const iImage *icon = iImageFactory::FindIcon("volv.png");

	//
	//  Main page
	// ************************************************
	//
	MainPage *page0 = new MainPage(mBook,mProvider);
	mBook->AddPage("Main",icon,page0);

	//
	//  Paint page
	// ************************************************
	//
	PaintPage *page1 = new PaintPage(mBook);
	mBook->AddPage("Paint",icon,page1);

	//
	//  Opacity page
	// ************************************************
	//
	OpacityPage *page2 = new OpacityPage(mBook);
	mBook->AddPage("Opacity",icon,page2);

	//
	//  Replicate page
	// ************************************************
	//
	ReplicatePage *page3 = new ReplicatePage(mBook);
	mBook->AddPage("Replicate",icon,page3);

	//
	//  Extras page
	// ************************************************
	//
	ExtrasPage *page4 = new ExtrasPage(mBook);
	mBook->AddPage("Extras",icon,page4);
}

#endif
