/*****************************************************************************
 * Copyright (C) 2004-2009 Christoph Thielecke <crissi99@gmx.de>             *
 *                                                                           *
 * This program is free software; you can redistribute it and/or modify      *
 * it under the terms of the GNU General Public License as published by      *
 * the Free Software Foundation; either version 2 of the License, or         *
 * (at your option) any later version.                                       *
 *                                                                           *
 * This package is distributed in the hope that it will be useful,           *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of            *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the             *
 * GNU General Public License for more details.                              *
 *                                                                           *
 * You should have received a copy of the GNU General Public License         *
 * along with this package; if not, write to the Free Software               *
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA *
 *****************************************************************************/

#ifndef NETWORKINTERFACE_H
#define NETWORKINTERFACE_H

#include <QHttp>
#include <QProcess>
#include <QtCore/QObject>
#include <QtCore/QStringList>
#include <QtGui/QApplication>
#include <QtGui/QWidget>

#include <ktemporaryfile.h>

#include "kvpncconfig.h"

/**
 * Class for interaction with network interface
 */
class NetworkInterface : public QObject
{
    Q_OBJECT

public:
    /**
     * Constructor
     * @param parent Pointer to parent widget
     * @param name Name of the object
     * @return nothing
     */
    NetworkInterface(KVpncConfig* config, QApplication *app, QObject *parent = 0);

    /**
     * Destructor
     * @return nothing
     */
    ~NetworkInterface();

    /**
     * Tests if the interface exists
     * @param Interface The interface to test
     * @return True if interface exist
     */
    bool interfaceExist(QString Interface);

    /**
     * Retrieves a list of all availiable interfaces
     * @return List with all interfaces
     */
    QStringList getAllNetworkInterfaces();

    /**
     * Gets the IP from a given interface
     * @param Interface The interface from which the IP whould be given
     * @return The IP
     */
    QString getInterfaceIP(QString Interface);

    QString getInterfaceIP2(QString Interface); // old style

    /**
     * Gets the device from a given IP
     * @param IP The IP which the target interface has
     * @return The interface where the IP is set
     */
    QString getInterfaceAddress(QString IP);

    /**
     * Gets the interface where the default route points
     * @return The interface
     */
    QString getDefaultInterface();

    /**
     * Gets the gateway of a given interface
     * @return The interface
     */
    QString getGatewayOfInterface(QString interface);

    QString getNetmaskOfInterface(QString interface);

    bool inSameNetwork(QString ip1, QString Netmask1, QString ip2, QString Netmask2);

    /**
     * Gets the gateway of the default route
     * @return The interface
     */
    QString getGatewayOfDefaultInterface();

    /**
     * Gets the count of default route entries
     * @return count
     */
    int getDefaultRouteCount();

    /**
     * Gets the IP address which is used for communication with internet
     * connections (same as local address if not behind router)
     * @return The external IP address
     */
    QString  getExternalIpAddress();

    bool inSameNetwork(QString ip1, QString ip2);

public slots:
    void readFromStdout();
    void readFromStderr();
    void processHasFinished(int exitCode, QProcess::ExitStatus exitStatus);
    void readFromStdout_interfaceip();
    void readFromStderr_interfaceip();
    void externalIpDataRecieved(const QHttpResponseHeader &resp);

protected:
    QProcess *proc; //< The process for starting external programs
    QProcess *InterfaceExitsProc;
    QProcess *InterfaceIpProc;
    bool interfaceTest; //< True if interface test is running
    bool retrieveInterfaceIP; //< True if retrieve interface IP is running
    bool retrieveInterfaceAddress; //< True if retrieve interface address is running
    QStringList InterfaceList; //< List with interfaces
    QString InterfaceIP; //< IP for interface
    QString InterfaceAddress; //< Device for IP
    QString GatewayAddress; //< Gateway of Device
    QString GatewayOfInterfaceAddress;
    QString GatewayOfDefaultInterfaceAddress; //< Gateway of Device
    bool interfaceExists; //< True if interface exists
    QString ifconfigbin; //< Path to ifconfig
    QString routebin; //< Path to route
    QString defaultinterface; //< Interface with defaultroute
    QString IPforInterface; //< The Ip for interface
    QString tmpInterface; //< Interface for getting devname
    QApplication *app;
    bool readOutput, getExternalIpAddressRunning;
    KTemporaryFile *tmpfile;
    QHttp *http;
    QString ExternalIpAddress;
    QStringList *env;
    KVpncConfig *GlobalConfig;
};

#endif
