package org.jboss.cache.eviction.minttl;

import org.jboss.cache.Cache;
import org.jboss.cache.DefaultCacheFactory;
import org.jboss.cache.Fqn;
import org.jboss.cache.util.TestingUtil;
import org.jboss.cache.config.EvictionConfig;
import org.jboss.cache.config.EvictionRegionConfig;
import org.jboss.cache.eviction.EvictionPolicyConfigBase;
import org.testng.annotations.AfterMethod;
import org.testng.annotations.BeforeMethod;
import org.testng.annotations.Test;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.CountDownLatch;

/**
 * This test exercises the minimum time to live for any element in the cache
 *
 * @author <a href="mailto:manik@jboss.org">Manik Surtani</a>
 * @since 2.1.0
 */
@Test(groups = {"functional"})
public abstract class MinTTLTestBase
{
   // this should ideally be in an eviction test base class so all eviction policies can be tested

   protected Cache cache;
   protected Fqn region = Fqn.fromString("/test-region");
   protected Fqn fqn = Fqn.fromRelativeElements(region, "a");
   // allows the test methods to notify any support threads in subclasses that data is in the cache and the test is about to begin
   protected CountDownLatch cacheInitialisedLatch;

   protected abstract EvictionPolicyConfigBase getEvictionPolicyConfig();

   @BeforeMethod
   public void setUp()
   {
      cacheInitialisedLatch = new CountDownLatch(1);

      // the LRU policy cfg
      EvictionPolicyConfigBase cfg = getEvictionPolicyConfig();

      // the region configuration
      EvictionRegionConfig regionCfg = new EvictionRegionConfig();
      regionCfg.setRegionFqn(region);
      regionCfg.setRegionName(region.toString());
      regionCfg.setEvictionPolicyConfig(cfg);

      // set regions in a list
      List<EvictionRegionConfig> evictionRegionConfigs = new ArrayList<EvictionRegionConfig>();
      evictionRegionConfigs.add(regionCfg);

      // cache-wide
      EvictionConfig ec = new EvictionConfig();
      ec.setWakeupIntervalSeconds(1);
      ec.setEvictionRegionConfigs(evictionRegionConfigs);

      cache = new DefaultCacheFactory().createCache(false);
      cache.getConfiguration().setEvictionConfig(ec);
   }

   @AfterMethod
   public void tearDown()
   {
      cache.stop();
   }

   public void testNoMinimumTTL()
   {
      cache.start();
      cache.put(fqn, "k", "v");
      // in case any waiting threads in subclasses are waiting for the cache to be initialised
      cacheInitialisedLatch.countDown();

      assert cache.get(fqn, "k") != null : "Node should be in the cache";

      TestingUtil.sleepThread(3000);

      assert cache.get(fqn, "k") == null : "Node should have been evicted";
   }

   public void testWithMinimumTTL()
   {
      ((EvictionPolicyConfigBase) cache.getConfiguration().getEvictionConfig().getEvictionRegionConfigs().get(0).getEvictionPolicyConfig()).setMinTimeToLiveSeconds(3);

      cache.start();
      cache.put(fqn, "k", "v");
      // in case any waiting threads in subclasses are waiting for the cache to be initialised
      cacheInitialisedLatch.countDown();

      assert cache.get(fqn, "k") != null : "Node should be in the cache";

      TestingUtil.sleepThread(3000);

      assert cache.get(fqn, "k") != null : "Node should still be in cache due to a minTTL of 3 secs";

      // the last cache.get() would have updated the last modified tstamp so we need to wait at least 3 secs (+1 sec maybe for the eviction thread)
      // to make sure this is evicted.

      TestingUtil.sleepThread(5000);

      assert cache.get(fqn, "k") == null : "Node should have been evicted";
   }

}
