#include "osl/container/moveVector.h"
#include "osl/hash/hashKey.h"
#include "osl/state/numEffectState.h"
#include "osl/state/historyState.h"
#include "osl/record/kisen.h"
#include "osl/record/csaRecord.h"
#include "osl/checkmate/dualDfpn.h"
#include "osl/eval/see.h"
#include "osl/misc/filePath.h"

#include <boost/accumulators/accumulators.hpp>
#include <boost/accumulators/statistics/mean.hpp>
#include <boost/accumulators/statistics/max.hpp>
#include <boost/scoped_ptr.hpp>
#include <boost/program_options.hpp>
#include <boost/filesystem/convenience.hpp>
#include <boost/foreach.hpp>
#include <boost/format.hpp>
#include <iostream>
#include <fstream>

static void 
convert_from_first(const osl::NumEffectState& initial,
		   const osl::vector<osl::Move> &in,
		   osl::vector<osl::Move> &out, size_t checkmate_limit)
{
  osl::DualDfpn dfpn;
  osl::NumEffectState state(initial);
  BOOST_FOREACH(osl::Move move, in) 
  {
    const int see = osl::See::see
      (state, move, state.pin(state.turn()), state.pin(alt(state.turn())));
    out.push_back(move);
    state.makeMove(move);
    if (state.inCheck() && see < 0
	&& dfpn.isLosingState(checkmate_limit, state, 
			      osl::HashKey(state), osl::PathEncoding(state.turn())))
      break;
  }
}

static void 
trim_last(const osl::NumEffectState& initial,
	  const osl::vector<osl::Move> &in,
	  osl::vector<osl::Move> &out, size_t checkmate_limit)
{
  if (in.empty())
    return;
  osl::DualDfpn dfpn;
  osl::HistoryState history(initial);
  BOOST_FOREACH(osl::Move move, in) 
    history.makeMove(move);
  const osl::Player last_played = in.back().player();
  int length = in.size();
  for (; length > 0; length -= 2)
  {
    osl::NumEffectState current = history.state();
    assert(current.turn() == alt(last_played));
    if (! current.inCheck())
      break;
    if (! dfpn.isLosingState(checkmate_limit, current, 
			     osl::HashKey(current), osl::PathEncoding(last_played)))
      break;
    history.unmakeMove();
    history.unmakeMove();
  }
  out = in;
  out.resize(length);
}

static void convert(const std::vector<std::string> &input_filename,
		    const std::string &output_kisen_filename,
		    size_t checkmate_limit, bool output_ipx, bool trim)
{
  namespace acc = boost::accumulators;
  acc::accumulator_set<double, acc::features<acc::tag::max, acc::tag::mean> > accumulator;
  std::ofstream ofs(output_kisen_filename.c_str());
  osl::record::OKisenStream ks(ofs);

  boost::scoped_ptr<osl::record::KisenIpxWriter> ipx_writer;
  boost::scoped_ptr<std::ofstream> ipx_ofs;
  if (output_ipx)
  {
    const boost::filesystem::path ipx_path =
      boost::filesystem::change_extension(boost::filesystem::path(output_kisen_filename), ".ipx");
    const std::string ipx = osl::misc::file_string(ipx_path);
    ipx_ofs.reset(new std::ofstream(ipx.c_str()));
    ipx_writer.reset(new osl::record::KisenIpxWriter(*ipx_ofs));
  }

  for (size_t i = 0; i < input_filename.size(); ++i)
  {
    osl::KisenFile kisen(input_filename[i]);
    osl::KisenIpxFile ipx(kisen.ipxFileName());
    for (size_t j=0; j<kisen.size(); ++j) 
    {
      osl::NumEffectState state = kisen.getInitialState();
      osl::vector<osl::Move> moves = kisen.getMoves(j);
      osl::vector<osl::Move> new_moves;
      if (trim)
	trim_last(state, moves, new_moves, checkmate_limit);
      else
	convert_from_first(state, moves, new_moves, checkmate_limit);

      osl::record::Record new_record;
      new_record.setPlayer(osl::BLACK, ipx.getPlayer(j, osl::BLACK));
      new_record.setPlayer(osl::WHITE, ipx.getPlayer(j, osl::WHITE));
      new_record.setDate(ipx.getStartDate(j));
      osl::SimpleState record_state = state;
      osl::record::RecordVisitor visitor;
      visitor.setState(&record_state);
      visitor.setRecord(&new_record);
      for (size_t k=0; k<new_moves.size(); ++k) 
      {
	visitor.addMoveAndAdvance(new_moves[k]);
	state.makeMove(new_moves[k]);
      }
      new_record.setResult(state.turn() == osl::BLACK
			   ? osl::Record::WHITE_WIN : osl::Record::BLACK_WIN);
      accumulator(moves.size() - new_moves.size());
      if (new_moves.size() >= 256)
	std::cerr << "long record " << j << ' ' << new_moves.size() << "\n";
      ks.save(&new_record);
      if (output_ipx)
      {
	ipx_writer->save(new_record,
			 ipx.getRating(j, osl::BLACK),ipx.getRating(j, osl::WHITE), 
			 ipx.getTitle(j, osl::BLACK), ipx.getTitle(j, osl::WHITE));
      }
      if ((j % 1000) == 999)
	std::cerr << input_filename[i] << " " << j
		  << " max " << acc::max(accumulator)
		  << " mean " << acc::mean(accumulator) << "\n";
    }
    std::cerr << input_filename[i]
	      << " max " << acc::max(accumulator)
	      << " mean " << acc::mean(accumulator) << "\n";
  }
}

int main(int argc, char **argv)
{
  bool output_ipx, trim;
  std::string kisen_filename;
  size_t checkmate_limit;
  boost::program_options::options_description command_line_options;
  command_line_options.add_options()
    ("trim-from-last",
     boost::program_options::value<bool>(&trim)->default_value(true),
     "trim last checkmate sequence")
    ("output-ipx",
     boost::program_options::value<bool>(&output_ipx)->default_value(true),
     "Whether output IPX file in addition to KIF file")
    ("output-kisen-filename,o",
     boost::program_options::value<std::string>(&kisen_filename)->
     default_value("test.kif"),
     "Output filename of Kisen file")
    ("checkmate-limit,l",
     boost::program_options::value<size_t>(&checkmate_limit)->default_value(1000),
     "Whether output IPX file in addition to KIF file")
    ("input-file", boost::program_options::value< std::vector<std::string> >(),
     "input files in kisen format")
    ("help", "Show help message");
  boost::program_options::variables_map vm;
  boost::program_options::positional_options_description p;
  p.add("input-file", -1);

  try
  {
    boost::program_options::store(
      boost::program_options::command_line_parser(
	argc, argv).options(command_line_options).positional(p).run(), vm);
    boost::program_options::notify(vm);
    if (vm.count("help"))
    {
      std::cerr << "Usage: " << argv[0] << " [options] kisen-files\n";
      std::cerr << "       " << argv[0] << " [options]\n";
      std::cout << command_line_options << std::endl;
      return 0;
    }
  }
  catch (std::exception &e)
  {
    std::cerr << "error in parsing options" << std::endl
	      << e.what() << std::endl;
    std::cerr << "Usage: " << argv[0] << " [options] kisen-files\n";
    std::cerr << "       " << argv[0] << " [options]\n";
    std::cerr << command_line_options << std::endl;
    return 1;
  }

  std::vector<std::string> files;
  if (vm.count("input-file"))
    files = vm["input-file"].as<std::vector<std::string> >();

  convert(files, kisen_filename, checkmate_limit, output_ipx, trim);
  return 0;
}
// ;;; Local Variables:
// ;;; mode:c++
// ;;; c-basic-offset:2
// ;;; End:
