///////////////////////////////////////////////////////////////////////////////
// 
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <scripting/Scripting.h>

#include <core/scene/animation/AnimManager.h>
#include <core/scene/animation/AnimationSettings.h>
#include <core/scene/animation/controller/Controller.h>
#include <core/scene/animation/controller/StandardControllers.h>
#include <core/scene/animation/controller/LookAtController.h>
#include <core/scene/animation/controller/TransformationController.h>

namespace Scripting {

using namespace boost::python;

void ExportAnimation()
{  
	class_<AnimationSettings, bases<RefTarget>, intrusive_ptr<AnimationSettings>, noncopyable>("AnimationSettings", init< optional<bool> >())
		.add_property("Time", &AnimationSettings::time, &AnimationSettings::setTime)
		.add_property("AnimationInterval", make_function(&AnimationSettings::animationInterval, return_value_policy<copy_const_reference>()), &AnimationSettings::setAnimationInterval)
		.add_property("FramesPerSecond", &AnimationSettings::framesPerSecond)
	;	

	class_<AnimManager, noncopyable>("AnimManager", no_init)
		.add_static_property("Instance", make_function(&AnimManager::getSingletonInstance, return_value_policy<reference_existing_object>()))
		.def("SuspendAnim", &AnimManager::suspendAnim)
		.def("ResumeAnim", &AnimManager::resumeAnim)
		.add_property("IsAnimating", &AnimManager::isAnimating)
		.add_property("AnimationMode", &AnimManager::animationMode, &AnimManager::setAnimationMode)
		.add_property("Time", &AnimManager::time, &AnimManager::setTime)
		.add_property("AnimationInterval", make_function(&AnimManager::animationInterval, return_value_policy<copy_const_reference>()), &AnimManager::setAnimationInterval)
		.def("TimeToString", &AnimManager::timeToString)
		.def("FrameToTime", &AnimManager::frameToTime)
		.def("TimeToFrame", &AnimManager::timeToFrame)
	;
	
	class_<Controller, bases<RefTarget>, intrusive_ptr<Controller>, noncopyable>("Controller", no_init)
		.def("ValidityInterval", pure_virtual((TimeInterval (Controller::*)(TimeTicks))&Controller::validityInterval))
	;

	class_<FloatController, bases<Controller>, intrusive_ptr<FloatController>, noncopyable>("FloatController", no_init)
		.add_property("Value", &FloatController::getCurrentValue, &FloatController::setCurrentValue)
		.def("GetValue", pure_virtual(&FloatController::getValue))
		.def("SetValue", pure_virtual(&FloatController::setValue))
		.def("GetValueAtTime", &FloatController::getValueAtTime)
		.def("ApplyValue", &FloatController::applyValue)
		.def("ValidityInterval", &FloatController::validityInterval)
	;

	class_<IntegerController, bases<Controller>, intrusive_ptr<IntegerController>, noncopyable>("IntegerController", no_init)
		.add_property("Value", &IntegerController::getCurrentValue, &IntegerController::setCurrentValue)
		.def("GetValue", pure_virtual(&IntegerController::getValue))
		.def("SetValue", pure_virtual(&IntegerController::setValue))
		.def("GetValueAtTime", &IntegerController::getValueAtTime)
		.def("ApplyValue", &IntegerController::applyValue)
		.def("ValidityInterval", &IntegerController::validityInterval)
	;

	class_<BooleanController, bases<Controller>, intrusive_ptr<BooleanController>, noncopyable>("BooleanController", no_init)
		.add_property("Value", &BooleanController::getCurrentValue, &BooleanController::setCurrentValue)
		.def("GetValue", pure_virtual(&BooleanController::getValue))
		.def("SetValue", pure_virtual(&BooleanController::setValue))
		.def("GetValueAtTime", &BooleanController::getValueAtTime)
		.def("ApplyValue", &BooleanController::applyValue)
		.def("ValidityInterval", &BooleanController::validityInterval)
	;

	class_<VectorController, bases<Controller>, intrusive_ptr<VectorController>, noncopyable>("VectorController", no_init)
		.add_property("Value", &VectorController::getCurrentValue, &VectorController::setCurrentValue)
		.def("GetValue", pure_virtual((void (VectorController::*)(TimeTicks, Vector3&, TimeInterval&))&VectorController::getValue))
		.def("SetValue", pure_virtual(&VectorController::setValue))
		.def("GetValueAtTime", &VectorController::getValueAtTime)
		.def("ApplyValue", &VectorController::applyValue)
		.def("ValidityInterval", &VectorController::validityInterval)
	;

	class_<PositionController, bases<Controller>, intrusive_ptr<PositionController>, noncopyable>("PositionController", no_init)
		.add_property("Value", &PositionController::getCurrentValue, &PositionController::setCurrentValue)
		.def("GetValue", pure_virtual(&PositionController::getValue))
		.def("SetValue", pure_virtual(&PositionController::setValue))
		.def("GetValueAtTime", &PositionController::getValueAtTime)
		.def("ApplyValue", &PositionController::applyValue)
		.def("ValidityInterval", &PositionController::validityInterval)
		.def("ChangeParent", &PositionController::changeParent)
	;

	class_<RotationController, bases<Controller>, intrusive_ptr<RotationController>, noncopyable>("RotationController", no_init)
		.add_property("Value", &RotationController::getCurrentValue, &RotationController::setCurrentValue)
		.def("GetValue", pure_virtual(&RotationController::getValue))
		.def("SetValue", pure_virtual(&RotationController::setValue))
		.def("GetValueAtTime", &RotationController::getValueAtTime)
		.def("ApplyValue", &RotationController::applyValue)
		.def("ValidityInterval", &RotationController::validityInterval)
		.def("ChangeParent", &RotationController::changeParent)
	;

	class_<ScalingController, bases<Controller>, intrusive_ptr<ScalingController>, noncopyable>("ScalingController", no_init)
		.add_property("Value", &ScalingController::getCurrentValue, &ScalingController::setCurrentValue)
		.def("GetValue", pure_virtual(&ScalingController::getValue))
		.def("SetValue", pure_virtual(&ScalingController::setValue))
		.def("GetValueAtTime", &ScalingController::getValueAtTime)
		.def("ApplyValue", &ScalingController::applyValue)
		.def("ValidityInterval", &ScalingController::validityInterval)
		.def("ChangeParent", &ScalingController::changeParent)
	;

	class_<TransformationController, bases<Controller>, intrusive_ptr<TransformationController>, noncopyable>("TransformationController", no_init)
		.def("GetValue", pure_virtual(&TransformationController::getValue))
		.def("GetValueAtTime", &TransformationController::getValueAtTime)
		.add_property("CurrentValue", &TransformationController::getCurrentValue, &TransformationController::setCurrentValue)
		.def("SetValue", pure_virtual(&TransformationController::setValue))
		.def("ApplyValue", &TransformationController::applyValue)
		.def("Translate", &TransformationController::translate)
		.def("Rotate", &TransformationController::rotate)
		.def("Scale", &TransformationController::scale)
		.def("ValidityInterval", &TransformationController::validityInterval)
		.def("ChangeParent", &TransformationController::changeParent)
	;
	
	class_<ControllerManager, noncopyable>("ControllerManager", no_init)
		.add_static_property("Instance", make_function(&ControllerManager::getSingletonInstance, return_value_policy<reference_existing_object>()))
	;

	class_<ConstFloatController, bases<FloatController>, intrusive_ptr<ConstFloatController>, noncopyable>("ConstFloatController", init<>())
	;

	class_<ConstIntegerController, bases<IntegerController>, intrusive_ptr<ConstIntegerController>, noncopyable>("ConstIntegerController", init<>())
	;

	class_<ConstVectorController, bases<VectorController>, intrusive_ptr<ConstVectorController>, noncopyable>("ConstVectorController", init<>())
	;

	class_<ConstBooleanController, bases<BooleanController>, intrusive_ptr<ConstBooleanController>, noncopyable>("ConstBooleanController", init<>())
	;

	class_<LinearFloatController, bases<FloatController>, intrusive_ptr<LinearFloatController>, noncopyable>("LinearFloatController", init<>())
	;

	class_<LinearIntegerController, bases<IntegerController>, intrusive_ptr<LinearIntegerController>, noncopyable>("LinearIntegerController", init<>())
	;

	class_<LinearVectorController, bases<VectorController>, intrusive_ptr<LinearVectorController>, noncopyable>("LinearVectorController", init<>())
	;

	class_<LinearPositionController, bases<PositionController>, intrusive_ptr<LinearPositionController>, noncopyable>("LinearPositionController", init<>())
	;

	class_<LinearRotationController, bases<RotationController>, intrusive_ptr<LinearRotationController>, noncopyable>("LinearRotationController", init<>())
	;

	class_<LinearScalingController, bases<ScalingController>, intrusive_ptr<LinearScalingController>, noncopyable>("LinearScalingController", init<>())
	;

	class_<PRSTransformationController, bases<TransformationController>, intrusive_ptr<PRSTransformationController>, noncopyable>("PRSTransformationController", init<>())
		.add_property("Position", make_function(&PRSTransformationController::positionController, return_internal_reference<>()), (void (PRSTransformationController::*)(PositionController*))&PRSTransformationController::setPositionController)
		.add_property("Rotation", make_function(&PRSTransformationController::rotationController, return_internal_reference<>()), (void (PRSTransformationController::*)(RotationController*))&PRSTransformationController::setRotationController)
		.add_property("Scaling", make_function(&PRSTransformationController::scalingController, return_internal_reference<>()), (void (PRSTransformationController::*)(ScalingController*))&PRSTransformationController::setScalingController)
	;
}

};
