% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mcmc-recover.R
\name{MCMC-recover}
\alias{MCMC-recover}
\alias{mcmc_recover_intervals}
\alias{mcmc_recover_scatter}
\alias{mcmc_recover_hist}
\title{Compare MCMC estimates to "true" parameter values}
\usage{
mcmc_recover_intervals(
  x,
  true,
  batch = rep(1, length(true)),
  ...,
  facet_args = list(),
  prob = 0.5,
  prob_outer = 0.9,
  point_est = c("median", "mean", "none"),
  size = 4,
  alpha = 1
)

mcmc_recover_scatter(
  x,
  true,
  batch = rep(1, length(true)),
  ...,
  facet_args = list(),
  point_est = c("median", "mean"),
  size = 3,
  alpha = 1
)

mcmc_recover_hist(
  x,
  true,
  ...,
  facet_args = list(),
  binwidth = NULL,
  bins = NULL,
  breaks = NULL
)
}
\arguments{
\item{x}{An object containing MCMC draws:
\itemize{
\item A 3-D array, matrix, list of matrices, or data frame. The \link{MCMC-overview}
page provides details on how to specify each these.
\item A \code{draws} object from the \pkg{posterior} package (e.g.,
\code{draws_array}, \code{draws_rvars}, etc.).
\item An object with an \code{as.array()} method that returns the same kind of 3-D
array described on the \link{MCMC-overview} page.
}}

\item{true}{A numeric vector of "true" values of the parameters in \code{x}.
There should be one value in \code{true} for each parameter included in
\code{x} and the order of the parameters in \code{true} should be the same
as the order of the parameters in \code{x}.}

\item{batch}{Optionally, a vector-like object (numeric, character, integer,
factor) used to split the parameters into batches. If \code{batch} is
specified, it must have the same length as \code{true} and be in the same
order as \code{true}. Parameters in the same batch will be grouped together
in the same facet in the plot (see the \strong{Examples} section, below).
The default is to group all parameters together into a single batch.
Changing the default is most useful when parameters are on very different
scales, in which case \code{batch} can be used to group them into batches
within which it makes sense to use the same y-axis.}

\item{...}{Currently unused.}

\item{facet_args}{A named list of arguments (other than \code{facets}) passed
to \code{\link[ggplot2:facet_wrap]{ggplot2::facet_wrap()}} or \code{\link[ggplot2:facet_grid]{ggplot2::facet_grid()}}
to control faceting. Note: if \code{scales} is not included in \code{facet_args}
then \strong{bayesplot} may use \code{scales="free"} as the default (depending
on the plot) instead of the \strong{ggplot2} default of \code{scales="fixed"}.}

\item{prob}{The probability mass to include in the inner interval. The
default is \code{0.5} (50\% interval).}

\item{prob_outer}{The probability mass to include in the outer interval. The
default is \code{0.9} (90\% interval).}

\item{point_est}{The point estimate to show. Either \code{"median"} (the
default), \code{"mean"}, or \code{"none"}.}

\item{size, alpha}{Passed to \code{\link[ggplot2:geom_point]{ggplot2::geom_point()}} to control the
appearance of plotted points.}

\item{binwidth}{Passed to \code{\link[ggplot2:geom_histogram]{ggplot2::geom_histogram()}} to override
the default binwidth.}

\item{bins}{Passed to \code{\link[ggplot2:geom_histogram]{ggplot2::geom_histogram()}} to override
the default binwidth.}

\item{breaks}{Passed to \code{\link[ggplot2:geom_histogram]{ggplot2::geom_histogram()}} as an
alternative to \code{binwidth}.}
}
\value{
A ggplot object that can be further customized using the \strong{ggplot2} package.
}
\description{
Plots comparing MCMC estimates to "true" parameter values. Before fitting a
model to real data it is useful to simulate data according to the model using
known (fixed) parameter values and to check that these "true" parameter
values are (approximately) recovered by fitting the model to the simulated
data. See the \strong{Plot Descriptions} section, below, for details on the
available plots.
}
\section{Plot Descriptions}{

\describe{
\item{\code{mcmc_recover_intervals()}}{
Central intervals and point estimates computed from MCMC draws, with
"true" values plotted using a different shape.
}
\item{\code{mcmc_recover_scatter()}}{
Scatterplot of posterior means (or medians) against "true" values.
}
\item{\code{mcmc_recover_hist()}}{
Histograms of the draws for each parameter with the "true" value overlaid
as a vertical line.
}
}
}

\examples{
\dontrun{
library(rstanarm)
alpha <- 1; beta <- rnorm(10, 0, 3); sigma <- 2
X <- matrix(rnorm(1000), 100, 10)
y <- rnorm(100, mean = c(alpha + X \%*\% beta), sd = sigma)
fit <- stan_glm(y ~ ., data = data.frame(y, X), refresh = 0)
draws <- as.matrix(fit)
print(colnames(draws))
true <- c(alpha, beta, sigma)

mcmc_recover_intervals(draws, true)

# put the coefficients on X into the same batch
mcmc_recover_intervals(draws, true, batch = c(1, rep(2, 10), 1))
# equivalent
mcmc_recover_intervals(draws, true, batch = grepl("X", colnames(draws)))
# same but facets stacked vertically
mcmc_recover_intervals(draws, true,
                       batch = grepl("X", colnames(draws)),
                       facet_args = list(ncol = 1),
                       size = 3)

# each parameter in its own facet
mcmc_recover_intervals(draws, true, batch = 1:ncol(draws))
# same but in a different order
mcmc_recover_intervals(draws, true, batch = c(1, 3, 4, 2, 5:12))
# present as bias by centering with true values
mcmc_recover_intervals(sweep(draws, 2, true), rep(0, ncol(draws))) + hline_0()


# scatterplot of posterior means vs true values
mcmc_recover_scatter(draws, true, point_est = "mean")


# histograms of parameter draws with true value added as vertical line
color_scheme_set("brightblue")
mcmc_recover_hist(draws[, 1:4], true[1:4])
}

}
\seealso{
Other MCMC: 
\code{\link{MCMC-combos}},
\code{\link{MCMC-diagnostics}},
\code{\link{MCMC-distributions}},
\code{\link{MCMC-intervals}},
\code{\link{MCMC-nuts}},
\code{\link{MCMC-overview}},
\code{\link{MCMC-parcoord}},
\code{\link{MCMC-scatterplots}},
\code{\link{MCMC-traces}}
}
\concept{MCMC}
