/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.clients.consumer.internals;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.IntStream;
import org.apache.kafka.clients.consumer.ConsumerConfig;
import org.apache.kafka.clients.consumer.internals.CoordinatorRequestManager;
import org.apache.kafka.clients.consumer.internals.HeartbeatRequestState;
import org.apache.kafka.clients.consumer.internals.MemberState;
import org.apache.kafka.clients.consumer.internals.NetworkClientDelegate;
import org.apache.kafka.clients.consumer.internals.RequestManager;
import org.apache.kafka.clients.consumer.internals.StreamsMembershipManager;
import org.apache.kafka.clients.consumer.internals.StreamsRebalanceData;
import org.apache.kafka.clients.consumer.internals.events.BackgroundEventHandler;
import org.apache.kafka.clients.consumer.internals.events.ErrorEvent;
import org.apache.kafka.clients.consumer.internals.metrics.HeartbeatMetricsManager;
import org.apache.kafka.common.TopicPartition;
import org.apache.kafka.common.errors.GroupAuthorizationException;
import org.apache.kafka.common.errors.RetriableException;
import org.apache.kafka.common.errors.UnsupportedVersionException;
import org.apache.kafka.common.message.StreamsGroupHeartbeatRequestData;
import org.apache.kafka.common.message.StreamsGroupHeartbeatResponseData;
import org.apache.kafka.common.metrics.Metrics;
import org.apache.kafka.common.protocol.Errors;
import org.apache.kafka.common.requests.StreamsGroupHeartbeatRequest;
import org.apache.kafka.common.requests.StreamsGroupHeartbeatResponse;
import org.apache.kafka.common.utils.LogContext;
import org.apache.kafka.common.utils.Time;
import org.apache.kafka.common.utils.Timer;
import org.slf4j.Logger;

public class StreamsGroupHeartbeatRequestManager
implements RequestManager {
    private static final String UNSUPPORTED_VERSION_ERROR_MESSAGE = "The cluster does not support the STREAMS group protocol or does not support the versions of the STREAMS group protocol used by this client (used versions: 0 to 0).";
    private final Logger logger;
    private final int maxPollIntervalMs;
    private final CoordinatorRequestManager coordinatorRequestManager;
    private final HeartbeatRequestState heartbeatRequestState;
    private final HeartbeatState heartbeatState;
    private final StreamsMembershipManager membershipManager;
    private final BackgroundEventHandler backgroundEventHandler;
    private final HeartbeatMetricsManager metricsManager;
    private StreamsRebalanceData streamsRebalanceData;
    private final Timer pollTimer;

    public StreamsGroupHeartbeatRequestManager(LogContext logContext, Time time, ConsumerConfig config, CoordinatorRequestManager coordinatorRequestManager, StreamsMembershipManager membershipManager, BackgroundEventHandler backgroundEventHandler, Metrics metrics, StreamsRebalanceData streamsRebalanceData) {
        this.logger = logContext.logger(this.getClass());
        this.coordinatorRequestManager = Objects.requireNonNull(coordinatorRequestManager, "Coordinator request manager cannot be null");
        this.membershipManager = Objects.requireNonNull(membershipManager, "Streams membership manager cannot be null");
        this.backgroundEventHandler = Objects.requireNonNull(backgroundEventHandler, "Background event handler cannot be null");
        this.metricsManager = new HeartbeatMetricsManager(Objects.requireNonNull(metrics, "Metrics cannot be null"));
        this.streamsRebalanceData = Objects.requireNonNull(streamsRebalanceData, "Streams rebalance data cannot be null");
        this.maxPollIntervalMs = config.getInt("max.poll.interval.ms");
        long retryBackoffMs = config.getLong("retry.backoff.ms");
        long retryBackoffMaxMs = config.getLong("retry.backoff.max.ms");
        this.heartbeatState = new HeartbeatState(streamsRebalanceData, membershipManager, this.maxPollIntervalMs);
        this.heartbeatRequestState = new HeartbeatRequestState(logContext, time, 0L, retryBackoffMs, retryBackoffMaxMs, (double)this.maxPollIntervalMs);
        this.pollTimer = time.timer(this.maxPollIntervalMs);
    }

    @Override
    public NetworkClientDelegate.PollResult poll(long currentTimeMs) {
        if (this.coordinatorRequestManager.coordinator().isEmpty() || this.membershipManager.shouldSkipHeartbeat()) {
            this.membershipManager.onHeartbeatRequestSkipped();
            this.maybePropagateCoordinatorFatalErrorEvent();
            return NetworkClientDelegate.PollResult.EMPTY;
        }
        this.pollTimer.update(currentTimeMs);
        if (this.pollTimer.isExpired() && !this.membershipManager.isLeavingGroup()) {
            this.logger.warn("Consumer poll timeout has expired. This means the time between subsequent calls to poll() was longer than the configured max.poll.interval.ms, which typically implies that the poll loop is spending too much time processing messages. You can address this either by increasing max.poll.interval.ms or by reducing the maximum size of batches returned in poll() with max.poll.records.");
            this.membershipManager.onPollTimerExpired();
            NetworkClientDelegate.UnsentRequest leaveHeartbeat = this.makeHeartbeatRequestAndLogResponse(currentTimeMs);
            this.heartbeatRequestState.reset();
            this.heartbeatState.reset();
            return new NetworkClientDelegate.PollResult(this.heartbeatRequestState.heartbeatIntervalMs(), Collections.singletonList(leaveHeartbeat));
        }
        if (this.shouldHeartbeatBeforeIntervalExpires() || this.heartbeatRequestState.canSendRequest(currentTimeMs)) {
            NetworkClientDelegate.UnsentRequest request = this.makeHeartbeatRequestAndHandleResponse(currentTimeMs);
            return new NetworkClientDelegate.PollResult(this.heartbeatRequestState.heartbeatIntervalMs(), Collections.singletonList(request));
        }
        return new NetworkClientDelegate.PollResult(this.heartbeatRequestState.timeToNextHeartbeatMs(currentTimeMs));
    }

    @Override
    public NetworkClientDelegate.PollResult pollOnClose(long currentTimeMs) {
        if (this.membershipManager.isLeavingGroup()) {
            NetworkClientDelegate.UnsentRequest request = this.makeHeartbeatRequestAndLogResponse(currentTimeMs);
            return new NetworkClientDelegate.PollResult(this.heartbeatRequestState.heartbeatIntervalMs(), List.of(request));
        }
        return NetworkClientDelegate.PollResult.EMPTY;
    }

    public StreamsMembershipManager membershipManager() {
        return this.membershipManager;
    }

    @Override
    public long maximumTimeToWait(long currentTimeMs) {
        this.pollTimer.update(currentTimeMs);
        if (this.pollTimer.isExpired() || this.membershipManager.shouldNotWaitForHeartbeatInterval() && !this.heartbeatRequestState.requestInFlight()) {
            return 0L;
        }
        return Math.min(this.pollTimer.remainingMs() / 2L, this.heartbeatRequestState.timeToNextHeartbeatMs(currentTimeMs));
    }

    public void resetPollTimer(long pollMs) {
        this.pollTimer.update(pollMs);
        if (this.pollTimer.isExpired()) {
            this.logger.warn("Time between subsequent calls to poll() was longer than the configured max.poll.interval.ms, exceeded approximately by {} ms. Member {} will rejoin the group now.", (Object)this.pollTimer.isExpiredBy(), (Object)this.membershipManager.memberId());
            this.membershipManager.maybeRejoinStaleMember();
        }
        this.pollTimer.reset(this.maxPollIntervalMs);
    }

    private boolean shouldHeartbeatBeforeIntervalExpires() {
        return this.membershipManager.state() == MemberState.LEAVING || (this.membershipManager.state() == MemberState.JOINING || this.membershipManager.state() == MemberState.ACKNOWLEDGING) && !this.heartbeatRequestState.requestInFlight();
    }

    private void maybePropagateCoordinatorFatalErrorEvent() {
        this.coordinatorRequestManager.getAndClearFatalError().ifPresent(fatalError -> this.backgroundEventHandler.add(new ErrorEvent((Throwable)fatalError)));
    }

    private NetworkClientDelegate.UnsentRequest makeHeartbeatRequestAndLogResponse(long currentTimeMs) {
        return this.makeHeartbeatRequest(currentTimeMs).whenComplete((response, exception) -> {
            if (response != null) {
                this.metricsManager.recordRequestLatency(response.requestLatencyMs());
                Errors error = Errors.forCode(((StreamsGroupHeartbeatResponse)response.responseBody()).data().errorCode());
                if (error == Errors.NONE) {
                    this.logger.debug("StreamsGroupHeartbeatRequest responded successfully: {}", response);
                } else {
                    this.logger.error("StreamsGroupHeartbeatRequest failed because of {}: {}", (Object)error, response);
                }
            } else {
                this.logger.error("StreamsGroupHeartbeatRequest failed because of unexpected exception.", exception);
            }
        });
    }

    private NetworkClientDelegate.UnsentRequest makeHeartbeatRequestAndHandleResponse(long currentTimeMs) {
        NetworkClientDelegate.UnsentRequest request = this.makeHeartbeatRequest(currentTimeMs);
        return request.whenComplete((response, exception) -> {
            long completionTimeMs = request.handler().completionTimeMs();
            if (response != null) {
                this.metricsManager.recordRequestLatency(response.requestLatencyMs());
                this.onResponse((StreamsGroupHeartbeatResponse)response.responseBody(), completionTimeMs);
            } else {
                this.onFailure((Throwable)exception, completionTimeMs);
            }
        });
    }

    private NetworkClientDelegate.UnsentRequest makeHeartbeatRequest(long currentTimeMs) {
        NetworkClientDelegate.UnsentRequest request = new NetworkClientDelegate.UnsentRequest(new StreamsGroupHeartbeatRequest.Builder(this.heartbeatState.buildRequestData(), true), this.coordinatorRequestManager.coordinator());
        this.heartbeatRequestState.onSendAttempt(currentTimeMs);
        this.membershipManager.onHeartbeatRequestGenerated();
        this.metricsManager.recordHeartbeatSentMs(currentTimeMs);
        this.heartbeatRequestState.resetTimer();
        return request;
    }

    private void onResponse(StreamsGroupHeartbeatResponse response, long currentTimeMs) {
        if (Errors.forCode(response.data().errorCode()) == Errors.NONE) {
            this.onSuccessResponse(response, currentTimeMs);
        } else {
            this.onErrorResponse(response, currentTimeMs);
        }
    }

    private void onSuccessResponse(StreamsGroupHeartbeatResponse response, long currentTimeMs) {
        List<StreamsGroupHeartbeatResponseData.Status> statuses;
        StreamsGroupHeartbeatResponseData data = response.data();
        this.heartbeatRequestState.updateHeartbeatIntervalMs(data.heartbeatIntervalMs());
        this.heartbeatRequestState.onSuccessfulAttempt(currentTimeMs);
        this.heartbeatState.setEndpointInformationEpoch(data.endpointInformationEpoch());
        if (data.partitionsByUserEndpoint() != null) {
            this.streamsRebalanceData.setPartitionsByHost(StreamsGroupHeartbeatRequestManager.convertHostInfoMap(data));
        }
        if ((statuses = data.status()) != null) {
            this.streamsRebalanceData.setStatuses(statuses);
            if (!statuses.isEmpty()) {
                String statusDetails = statuses.stream().map(status -> "(" + status.statusCode() + ") " + status.statusDetail()).collect(Collectors.joining(", "));
                this.logger.warn("Membership is in the following statuses: {}", (Object)statusDetails);
            }
        }
        this.membershipManager.onHeartbeatSuccess(response);
    }

    private void onErrorResponse(StreamsGroupHeartbeatResponse response, long currentTimeMs) {
        Errors error = Errors.forCode(response.data().errorCode());
        String errorMessage = response.data().errorMessage();
        this.heartbeatState.reset();
        this.heartbeatRequestState.onFailedAttempt(currentTimeMs);
        switch (error) {
            case NOT_COORDINATOR: {
                this.logInfo(String.format("StreamsGroupHeartbeatRequest failed because the group coordinator %s is incorrect. Will attempt to find the coordinator again and retry", this.coordinatorRequestManager.coordinator()), response, currentTimeMs);
                this.coordinatorRequestManager.markCoordinatorUnknown(errorMessage, currentTimeMs);
                this.heartbeatRequestState.reset();
                break;
            }
            case COORDINATOR_NOT_AVAILABLE: {
                this.logInfo(String.format("StreamsGroupHeartbeatRequest failed because the group coordinator %s is not available. Will attempt to find the coordinator again and retry", this.coordinatorRequestManager.coordinator()), response, currentTimeMs);
                this.coordinatorRequestManager.markCoordinatorUnknown(errorMessage, currentTimeMs);
                this.heartbeatRequestState.reset();
                break;
            }
            case COORDINATOR_LOAD_IN_PROGRESS: {
                this.logInfo(String.format("StreamsGroupHeartbeatRequest failed because the group coordinator %s is still loading. Will retry", this.coordinatorRequestManager.coordinator()), response, currentTimeMs);
                break;
            }
            case GROUP_AUTHORIZATION_FAILED: {
                GroupAuthorizationException exception = GroupAuthorizationException.forGroupId(this.membershipManager.groupId());
                this.logger.error("StreamsGroupHeartbeatRequest failed due to group authorization failure: {}", (Object)exception.getMessage());
                this.handleFatalFailure(error.exception(exception.getMessage()));
                break;
            }
            case TOPIC_AUTHORIZATION_FAILED: {
                this.logger.error("StreamsGroupHeartbeatRequest failed for member {} with state {} due to {}: {}", new Object[]{this.membershipManager.memberId(), this.membershipManager.state(), error, errorMessage});
                this.backgroundEventHandler.add(new ErrorEvent(error.exception()));
                break;
            }
            case INVALID_REQUEST: 
            case GROUP_MAX_SIZE_REACHED: 
            case STREAMS_INVALID_TOPOLOGY: 
            case STREAMS_INVALID_TOPOLOGY_EPOCH: 
            case STREAMS_TOPOLOGY_FENCED: {
                this.logger.error("StreamsGroupHeartbeatRequest failed due to {}: {}", (Object)error, (Object)errorMessage);
                this.handleFatalFailure(error.exception(errorMessage));
                break;
            }
            case FENCED_MEMBER_EPOCH: {
                this.logInfo(String.format("StreamsGroupHeartbeatRequest failed for member %s because epoch %s is fenced.", this.membershipManager.memberId(), this.membershipManager.memberEpoch()), response, currentTimeMs);
                this.membershipManager.onFenced();
                this.heartbeatRequestState.reset();
                break;
            }
            case UNKNOWN_MEMBER_ID: {
                this.logInfo(String.format("StreamsGroupHeartbeatRequest failed because member %s is unknown.", this.membershipManager.memberId()), response, currentTimeMs);
                this.membershipManager.onFenced();
                this.heartbeatRequestState.reset();
                break;
            }
            case UNSUPPORTED_VERSION: {
                this.logger.error("StreamsGroupHeartbeatRequest failed due to {}: {}", (Object)error, (Object)UNSUPPORTED_VERSION_ERROR_MESSAGE);
                this.handleFatalFailure(error.exception(UNSUPPORTED_VERSION_ERROR_MESSAGE));
                break;
            }
            default: {
                this.logger.error("StreamsGroupHeartbeatRequest failed due to unexpected error {}: {}", (Object)error, (Object)errorMessage);
                this.handleFatalFailure(error.exception(errorMessage));
            }
        }
        this.membershipManager.onFatalHeartbeatFailure();
    }

    private void logInfo(String message, StreamsGroupHeartbeatResponse response, long currentTimeMs) {
        this.logger.info("{} in {}ms: {}", new Object[]{message, this.heartbeatRequestState.remainingBackoffMs(currentTimeMs), response.data().errorMessage()});
    }

    private void onFailure(Throwable exception, long responseTimeMs) {
        this.heartbeatRequestState.onFailedAttempt(responseTimeMs);
        this.heartbeatState.reset();
        if (exception instanceof RetriableException) {
            this.coordinatorRequestManager.handleCoordinatorDisconnect(exception, responseTimeMs);
            String message = String.format("StreamsGroupHeartbeatRequest failed because of a retriable exception. Will retry in %s ms: %s", this.heartbeatRequestState.remainingBackoffMs(responseTimeMs), exception.getMessage());
            this.logger.debug(message);
            this.membershipManager.onRetriableHeartbeatFailure();
        } else {
            if (exception instanceof UnsupportedVersionException) {
                this.logger.error("StreamsGroupHeartbeatRequest failed because of an unsupported version exception: {}", (Object)exception.getMessage());
                this.handleFatalFailure(new UnsupportedVersionException(UNSUPPORTED_VERSION_ERROR_MESSAGE));
            } else {
                this.logger.error("StreamsGroupHeartbeatRequest failed because of a fatal exception while sending request: {}", (Object)exception.getMessage());
                this.handleFatalFailure(exception);
            }
            this.membershipManager.onFatalHeartbeatFailure();
        }
    }

    private void handleFatalFailure(Throwable error) {
        this.backgroundEventHandler.add(new ErrorEvent(error));
        this.membershipManager.transitionToFatal();
    }

    private static Map<StreamsRebalanceData.HostInfo, StreamsRebalanceData.EndpointPartitions> convertHostInfoMap(StreamsGroupHeartbeatResponseData data) {
        HashMap<StreamsRebalanceData.HostInfo, StreamsRebalanceData.EndpointPartitions> partitionsByHost = new HashMap<StreamsRebalanceData.HostInfo, StreamsRebalanceData.EndpointPartitions>();
        data.partitionsByUserEndpoint().forEach(endpoint -> {
            List<TopicPartition> activeTopicPartitions = StreamsGroupHeartbeatRequestManager.getTopicPartitionList(endpoint.activePartitions());
            List<TopicPartition> standbyTopicPartitions = StreamsGroupHeartbeatRequestManager.getTopicPartitionList(endpoint.standbyPartitions());
            StreamsGroupHeartbeatResponseData.Endpoint userEndpoint = endpoint.userEndpoint();
            StreamsRebalanceData.EndpointPartitions endpointPartitions = new StreamsRebalanceData.EndpointPartitions(activeTopicPartitions, standbyTopicPartitions);
            partitionsByHost.put(new StreamsRebalanceData.HostInfo(userEndpoint.host(), userEndpoint.port()), endpointPartitions);
        });
        return partitionsByHost;
    }

    static List<TopicPartition> getTopicPartitionList(List<StreamsGroupHeartbeatResponseData.TopicPartition> topicPartitions) {
        return topicPartitions.stream().flatMap(partition -> partition.partitions().stream().map(partitionId -> new TopicPartition(partition.topic(), (int)partitionId))).collect(Collectors.toList());
    }

    static class HeartbeatState {
        private final StreamsMembershipManager membershipManager;
        private final int rebalanceTimeoutMs;
        private final StreamsRebalanceData streamsRebalanceData;
        private final LastSentFields lastSentFields = new LastSentFields();
        private int endpointInformationEpoch = -1;

        public HeartbeatState(StreamsRebalanceData streamsRebalanceData, StreamsMembershipManager membershipManager, int rebalanceTimeoutMs) {
            this.membershipManager = membershipManager;
            this.streamsRebalanceData = streamsRebalanceData;
            this.rebalanceTimeoutMs = rebalanceTimeoutMs;
        }

        public void reset() {
            this.lastSentFields.reset();
        }

        public int endpointInformationEpoch() {
            return this.endpointInformationEpoch;
        }

        public void setEndpointInformationEpoch(int endpointInformationEpoch) {
            this.endpointInformationEpoch = endpointInformationEpoch;
        }

        public StreamsGroupHeartbeatRequestData buildRequestData() {
            boolean joining;
            StreamsGroupHeartbeatRequestData data = new StreamsGroupHeartbeatRequestData();
            data.setGroupId(this.membershipManager.groupId());
            data.setMemberId(this.membershipManager.memberId());
            data.setMemberEpoch(this.membershipManager.memberEpoch());
            data.setEndpointInformationEpoch(this.endpointInformationEpoch);
            this.membershipManager.groupInstanceId().ifPresent(data::setInstanceId);
            boolean bl = joining = this.membershipManager.state() == MemberState.JOINING;
            if (joining) {
                StreamsGroupHeartbeatRequestData.Topology topology = new StreamsGroupHeartbeatRequestData.Topology();
                topology.setSubtopologies(HeartbeatState.fromStreamsToHeartbeatRequest(this.streamsRebalanceData.subtopologies()));
                topology.setEpoch(this.streamsRebalanceData.topologyEpoch());
                data.setTopology(topology);
                data.setRebalanceTimeoutMs(this.rebalanceTimeoutMs);
                data.setProcessId(this.streamsRebalanceData.processId().toString());
                this.streamsRebalanceData.endpoint().ifPresent(userEndpoint -> data.setUserEndpoint(new StreamsGroupHeartbeatRequestData.Endpoint().setHost(userEndpoint.host()).setPort(userEndpoint.port())));
                data.setClientTags(this.streamsRebalanceData.clientTags().entrySet().stream().map(entry -> new StreamsGroupHeartbeatRequestData.KeyValue().setKey((String)entry.getKey()).setValue((String)entry.getValue())).collect(Collectors.toList()));
                data.setActiveTasks(HeartbeatState.fromStreamsToHeartbeatRequest(Set.of()));
                data.setStandbyTasks(HeartbeatState.fromStreamsToHeartbeatRequest(Set.of()));
                data.setWarmupTasks(HeartbeatState.fromStreamsToHeartbeatRequest(Set.of()));
            } else {
                StreamsRebalanceData.Assignment reconciledAssignment = this.streamsRebalanceData.reconciledAssignment();
                if (!reconciledAssignment.equals(this.lastSentFields.assignment)) {
                    data.setActiveTasks(HeartbeatState.fromStreamsToHeartbeatRequest(reconciledAssignment.activeTasks()));
                    data.setStandbyTasks(HeartbeatState.fromStreamsToHeartbeatRequest(reconciledAssignment.standbyTasks()));
                    data.setWarmupTasks(HeartbeatState.fromStreamsToHeartbeatRequest(reconciledAssignment.warmupTasks()));
                    this.lastSentFields.assignment = reconciledAssignment;
                }
            }
            data.setShutdownApplication(this.streamsRebalanceData.shutdownRequested());
            return data;
        }

        private static List<StreamsGroupHeartbeatRequestData.TaskIds> fromStreamsToHeartbeatRequest(Set<StreamsRebalanceData.TaskId> tasks) {
            return tasks.stream().collect(Collectors.groupingBy(StreamsRebalanceData.TaskId::subtopologyId, Collectors.mapping(StreamsRebalanceData.TaskId::partitionId, Collectors.toList()))).entrySet().stream().map(entry -> new StreamsGroupHeartbeatRequestData.TaskIds().setSubtopologyId((String)entry.getKey()).setPartitions((List)entry.getValue())).collect(Collectors.toList());
        }

        private static List<StreamsGroupHeartbeatRequestData.Subtopology> fromStreamsToHeartbeatRequest(Map<String, StreamsRebalanceData.Subtopology> subtopologies) {
            ArrayList<StreamsGroupHeartbeatRequestData.Subtopology> subtopologiesForRequest = new ArrayList<StreamsGroupHeartbeatRequestData.Subtopology>(subtopologies.size());
            for (Map.Entry<String, StreamsRebalanceData.Subtopology> subtopology : subtopologies.entrySet()) {
                subtopologiesForRequest.add(HeartbeatState.fromStreamsToHeartbeatRequest(subtopology.getKey(), subtopology.getValue()));
            }
            subtopologiesForRequest.sort(Comparator.comparing(StreamsGroupHeartbeatRequestData.Subtopology::subtopologyId));
            return subtopologiesForRequest;
        }

        private static StreamsGroupHeartbeatRequestData.Subtopology fromStreamsToHeartbeatRequest(String subtopologyId, StreamsRebalanceData.Subtopology subtopology) {
            StreamsGroupHeartbeatRequestData.Subtopology subtopologyData = new StreamsGroupHeartbeatRequestData.Subtopology();
            subtopologyData.setSubtopologyId(subtopologyId);
            ArrayList<String> sortedSourceTopics = new ArrayList<String>(subtopology.sourceTopics());
            Collections.sort(sortedSourceTopics);
            subtopologyData.setSourceTopics(sortedSourceTopics);
            ArrayList<String> sortedSinkTopics = new ArrayList<String>(subtopology.repartitionSinkTopics());
            Collections.sort(sortedSinkTopics);
            subtopologyData.setRepartitionSinkTopics(sortedSinkTopics);
            subtopologyData.setRepartitionSourceTopics(HeartbeatState.getRepartitionTopicsInfoFromStreams(subtopology));
            subtopologyData.setStateChangelogTopics(HeartbeatState.getChangelogTopicsInfoFromStreams(subtopology));
            subtopologyData.setCopartitionGroups(HeartbeatState.getCopartitionGroupsFromStreams(subtopology.copartitionGroups(), subtopologyData));
            return subtopologyData;
        }

        private static List<StreamsGroupHeartbeatRequestData.CopartitionGroup> getCopartitionGroupsFromStreams(Collection<Set<String>> copartitionGroups, StreamsGroupHeartbeatRequestData.Subtopology subtopologyData) {
            Map<String, Short> sourceTopicsMap = IntStream.range(0, subtopologyData.sourceTopics().size()).boxed().collect(Collectors.toMap(subtopologyData.sourceTopics()::get, Integer::shortValue));
            Map<String, Short> repartitionSourceTopics = IntStream.range(0, subtopologyData.repartitionSourceTopics().size()).boxed().collect(Collectors.toMap(x -> subtopologyData.repartitionSourceTopics().get((int)x).name(), Integer::shortValue));
            return copartitionGroups.stream().map(x -> HeartbeatState.getCopartitionGroupFromStreams(x, sourceTopicsMap, repartitionSourceTopics)).collect(Collectors.toList());
        }

        private static StreamsGroupHeartbeatRequestData.CopartitionGroup getCopartitionGroupFromStreams(Set<String> topicNames, Map<String, Short> sourceTopicsMap, Map<String, Short> repartitionSourceTopics) {
            StreamsGroupHeartbeatRequestData.CopartitionGroup copartitionGroup = new StreamsGroupHeartbeatRequestData.CopartitionGroup();
            topicNames.forEach(topicName -> {
                if (sourceTopicsMap.containsKey(topicName)) {
                    copartitionGroup.sourceTopics().add((Short)sourceTopicsMap.get(topicName));
                } else if (repartitionSourceTopics.containsKey(topicName)) {
                    copartitionGroup.repartitionSourceTopics().add((Short)repartitionSourceTopics.get(topicName));
                } else {
                    throw new IllegalStateException("Source topic not found in subtopology: " + topicName);
                }
            });
            return copartitionGroup;
        }

        private static List<StreamsGroupHeartbeatRequestData.TopicInfo> getRepartitionTopicsInfoFromStreams(StreamsRebalanceData.Subtopology subtopologyDataFromStreams) {
            ArrayList<StreamsGroupHeartbeatRequestData.TopicInfo> repartitionTopicsInfo = new ArrayList<StreamsGroupHeartbeatRequestData.TopicInfo>();
            for (Map.Entry<String, StreamsRebalanceData.TopicInfo> repartitionTopic : subtopologyDataFromStreams.repartitionSourceTopics().entrySet()) {
                StreamsGroupHeartbeatRequestData.TopicInfo repartitionTopicInfo = new StreamsGroupHeartbeatRequestData.TopicInfo();
                repartitionTopicInfo.setName(repartitionTopic.getKey());
                repartitionTopic.getValue().numPartitions().ifPresent(repartitionTopicInfo::setPartitions);
                repartitionTopic.getValue().replicationFactor().ifPresent(repartitionTopicInfo::setReplicationFactor);
                repartitionTopic.getValue().topicConfigs().forEach((k, v) -> repartitionTopicInfo.topicConfigs().add(new StreamsGroupHeartbeatRequestData.KeyValue().setKey((String)k).setValue((String)v)));
                repartitionTopicsInfo.add(repartitionTopicInfo);
                repartitionTopicInfo.topicConfigs().sort(Comparator.comparing(StreamsGroupHeartbeatRequestData.KeyValue::key));
            }
            repartitionTopicsInfo.sort(Comparator.comparing(StreamsGroupHeartbeatRequestData.TopicInfo::name));
            return repartitionTopicsInfo;
        }

        private static List<StreamsGroupHeartbeatRequestData.TopicInfo> getChangelogTopicsInfoFromStreams(StreamsRebalanceData.Subtopology subtopologyDataFromStreams) {
            ArrayList<StreamsGroupHeartbeatRequestData.TopicInfo> changelogTopicsInfo = new ArrayList<StreamsGroupHeartbeatRequestData.TopicInfo>();
            for (Map.Entry<String, StreamsRebalanceData.TopicInfo> changelogTopic : subtopologyDataFromStreams.stateChangelogTopics().entrySet()) {
                StreamsGroupHeartbeatRequestData.TopicInfo changelogTopicInfo = new StreamsGroupHeartbeatRequestData.TopicInfo();
                changelogTopicInfo.setName(changelogTopic.getKey());
                changelogTopic.getValue().replicationFactor().ifPresent(changelogTopicInfo::setReplicationFactor);
                changelogTopic.getValue().topicConfigs().forEach((k, v) -> changelogTopicInfo.topicConfigs().add(new StreamsGroupHeartbeatRequestData.KeyValue().setKey((String)k).setValue((String)v)));
                changelogTopicInfo.topicConfigs().sort(Comparator.comparing(StreamsGroupHeartbeatRequestData.KeyValue::key));
                changelogTopicsInfo.add(changelogTopicInfo);
            }
            changelogTopicsInfo.sort(Comparator.comparing(StreamsGroupHeartbeatRequestData.TopicInfo::name));
            return changelogTopicsInfo;
        }

        static class LastSentFields {
            private StreamsRebalanceData.Assignment assignment = null;

            LastSentFields() {
            }

            void reset() {
                this.assignment = null;
            }
        }
    }
}

