/*
 *  $Id: cell-renderer-color.c 15501 2013-10-26 20:11:05Z yeti-dn $
 *  Copyright (C) 2012-2025 David Nečas (Yeti).
 *  E-mail: yeti@gwyddion.net.
 *
 *  This program is free software; you can redistribute it and/or modify it under the terms of the GNU General Public
 *  License as published by the Free Software Foundation; either version 2 of the License, or (at your option) any
 *  later version.
 *
 *  This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied
 *  warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 *  details.
 *
 *  You should have received a copy of the GNU General Public License along with this program; if not, write to the
 *  Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

#include "libgwyddion/macros.h"
#include "libgwyddion/math.h"
#include "libgwyddion/utils.h"

#include "libgwyui/cell-renderer-color.h"
#include "libgwyui/widget-impl-utils.h"
#include "libgwyui/cairo-utils.h"

#define ASPECT_RATIO 1.618

enum {
    MIN_HEIGHT = 1,
};

enum {
    PROP_0,
    PROP_COLOR,
    NUM_PROPERTIES
};

struct _GwyCellRendererColorPrivate {
    GwyRGBA color;
    guint height;
};

static void               get_property        (GObject *object,
                                               guint prop_id,
                                               GValue *value,
                                               GParamSpec *pspec);
static void               set_property        (GObject *object,
                                               guint prop_id,
                                               const GValue *value,
                                               GParamSpec *pspec);
static GtkSizeRequestMode get_request_mode    (GtkCellRenderer *cell);
static void               get_preferred_width (GtkCellRenderer *cell,
                                               GtkWidget *widget,
                                               gint *minimum_size,
                                               gint *natural_size);
static void               get_preferred_height(GtkCellRenderer *cell,
                                               GtkWidget *widget,
                                               gint *minimum_size,
                                               gint *natural_size);
static void               render              (GtkCellRenderer *cellrenderer,
                                               cairo_t *cr,
                                               GtkWidget *widget,
                                               const GdkRectangle *background_area,
                                               const GdkRectangle *cell_area,
                                               GtkCellRendererState flags);
static void               set_color           (GwyCellRendererColor *renderer,
                                               GwyRGBA *color);

static GParamSpec *properties[NUM_PROPERTIES] = { NULL, };
static GtkCellRendererClass *parent_class = NULL;

G_DEFINE_TYPE_WITH_CODE(GwyCellRendererColor, gwy_cell_renderer_color, GTK_TYPE_CELL_RENDERER,
                        G_ADD_PRIVATE(GwyCellRendererColor));

static void
gwy_cell_renderer_color_class_init(GwyCellRendererColorClass *class)
{
    GObjectClass *gobject_class = G_OBJECT_CLASS(class);
    GtkCellRendererClass *cell_class = GTK_CELL_RENDERER_CLASS(class);

    parent_class = gwy_cell_renderer_color_parent_class;

    gobject_class->get_property = get_property;
    gobject_class->set_property = set_property;

    cell_class->get_request_mode = get_request_mode;
    cell_class->get_preferred_width = get_preferred_width;
    cell_class->get_preferred_height = get_preferred_height;
    cell_class->render = render;

    properties[PROP_COLOR] = g_param_spec_boxed("color", NULL,
                                                "Color to display",
                                                GWY_TYPE_RGBA,
                                                GWY_GPARAM_RWE);

    g_object_class_install_properties(gobject_class, NUM_PROPERTIES, properties);
}

static void
gwy_cell_renderer_color_init(GwyCellRendererColor *renderer)
{
    GwyCellRendererColorPrivate *priv;

    renderer->priv = priv = gwy_cell_renderer_color_get_instance_private(renderer);

    gint w, h;
    /* FIXME: This is wrong. Use a font-based estimate for the natural size. */
    gtk_icon_size_lookup(GTK_ICON_SIZE_MENU, &w, &h);
    priv->height = h;
}

static void
set_property(GObject *object,
             guint prop_id,
             const GValue *value,
             GParamSpec *pspec)
{
    GwyCellRendererColor *renderer = GWY_CELL_RENDERER_COLOR(object);

    switch (prop_id) {
        case PROP_COLOR:
        set_color(renderer, g_value_get_boxed(value));
        break;

        default:
        G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
        break;
    }
}

static void
get_property(GObject *object,
             guint prop_id,
             GValue *value,
             GParamSpec *pspec)
{
    GwyCellRendererColorPrivate *priv = GWY_CELL_RENDERER_COLOR(object)->priv;

    switch (prop_id) {
        case PROP_COLOR:
        g_value_set_boxed(value, &priv->color);
        break;

        default:
        G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
        break;
    }
}

static GtkSizeRequestMode
get_request_mode(G_GNUC_UNUSED GtkCellRenderer *cell)
{
    return GTK_SIZE_REQUEST_CONSTANT_SIZE;
}

static gint
get_natural_height(GwyCellRendererColor *renderer, GtkWidget *widget)
{
    GwyCellRendererColorPrivate *priv = renderer->priv;
    if (!priv->height)
        priv->height = get_natural_text_height(widget);
    return priv->height;
}

static void
get_preferred_width(GtkCellRenderer *renderer, GtkWidget *widget, gint *minimum, gint *natural)
{
    gint xpad;
    gtk_cell_renderer_get_padding(renderer, &xpad, NULL);
    if (minimum)
        *minimum = MIN_HEIGHT + 2*xpad;
    if (natural) {
        gint height = get_natural_height(GWY_CELL_RENDERER_COLOR(renderer), widget);
        *natural = GWY_ROUND(ASPECT_RATIO*height) + 2*xpad;
    }
}

static void
get_preferred_height(GtkCellRenderer *renderer, GtkWidget *widget, gint *minimum, gint *natural)
{
    gint ypad;
    gtk_cell_renderer_get_padding(renderer, &ypad, NULL);
    if (minimum)
        *minimum = MIN_HEIGHT + 2*ypad;
    if (natural) {
        gint height = get_natural_height(GWY_CELL_RENDERER_COLOR(renderer), widget);
        *natural = height + 2*ypad;
    }
}

// FIXME: Do we want to change the rendering depending on style flags?
static void
render(GtkCellRenderer *cellrenderer,
       cairo_t *cr,
       G_GNUC_UNUSED GtkWidget *widget,
       G_GNUC_UNUSED const GdkRectangle *background_area,
       const GdkRectangle *cell_area,
       G_GNUC_UNUSED GtkCellRendererState flags)
{
    gint xpad, ypad;
    gtk_cell_renderer_get_padding(cellrenderer, &xpad, &ypad);
    GdkRectangle rect = {
        .x = cell_area->x + xpad,
        .y = cell_area->y + ypad,
        .width = cell_area->width - 2*xpad,
        .height = cell_area->height - 2*ypad,
    };
    if (!gdk_rectangle_intersect(cell_area, &rect, NULL))
        return;

    GwyCellRendererColor *renderer = GWY_CELL_RENDERER_COLOR(cellrenderer);
    const GwyRGBA *color = &renderer->priv->color;

    cairo_save(cr);
    cairo_rectangle(cr, rect.x, rect.y, rect.width, rect.height);
    cairo_set_source_rgba(cr, color->r, color->g, color->b, color->a);
    cairo_fill(cr);
    cairo_restore(cr);
}

/**
 * gwy_cell_renderer_color_new:
 *
 * Creates a new color cell renderer.
 *
 * Return value: A new color cell renderer.
 **/
GtkCellRenderer*
gwy_cell_renderer_color_new(void)
{
    return g_object_new(GWY_TYPE_CELL_RENDERER_COLOR, NULL);
}

static void
set_color(GwyCellRendererColor *renderer, GwyRGBA *color)
{
    GwyCellRendererColorPrivate *priv = renderer->priv;
    priv->color = *color;
}

/**
 * SECTION:cell-renderer-color
 * @title: GwyCellRendererColor
 * @short_description: Renders a color in a cell
 *
 * A #GwyCellRendererColor can be used to render a color swatch in a cell of a #GtkTreeView or a similar cell-view
 * widget.
 */

/* vim: set cin columns=120 tw=118 et ts=4 sw=4 cino=>1s,e0,n0,f0,{0,}0,^0,\:1s,=0,g1s,h0,t0,+1s,c3,(0,u0 : */
