/* BEGIN software license
 *
 * msXpertSuite - mass spectrometry software suite
 * -----------------------------------------------
 * Copyright (C) 2009--2020 Filippo Rusconi
 *
 * http://www.msxpertsuite.org
 *
 * This file is part of the msXpertSuite project.
 *
 * The msXpertSuite project is the successor of the massXpert project. This
 * project now includes various independent modules:
 *
 * - massXpert, model polymer chemistries and simulate mass spectrometric data;
 * - mineXpert, a powerful TIC chromatogram/mass spectrum viewer/miner;
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 *
 * END software license
 */


/////////////////////// libXpertMassCore includes


/////////////////////// libXpertMassGUI includes
#include "MsXpS/libXpertMassGui/JavaScriptQObjectExposureRegistry.hpp"

namespace MsXpS
{
namespace libXpertMassGui
{
JavaScriptQObjectExposureRegistry::JavaScriptQObjectExposureRegistry(
  QObject *parent)
  : QObject(parent)
{
}

JavaScriptQObjectExposureRegistry::~JavaScriptQObjectExposureRegistry()
{
}

JavaScriptQObjectExposureRecordQSPtr
JavaScriptQObjectExposureRegistry::getRecordByObject(QObject *object_p)
{
  Q_ASSERT(object_p != nullptr);

  foreach(JavaScriptQObjectExposureRecordQSPtr record_qsp, m_records)
    {
      if(record_qsp->getObject() == object_p)
        return record_qsp;
    }

  return nullptr;
}

JavaScriptQObjectExposureRecordQSPtr
JavaScriptQObjectExposureRegistry::getRecordByObjectParent(QObject *object_p)
{
  Q_ASSERT(object_p != nullptr);

  foreach(JavaScriptQObjectExposureRecordQSPtr record_qsp, m_records)
    {
      if(record_qsp->getObjectParent() == object_p)
        return record_qsp;
    }

  return nullptr;
}

JavaScriptQObjectExposureRecordQSPtr
JavaScriptQObjectExposureRegistry::getRecordByName(const QString object_name)
{
  foreach(JavaScriptQObjectExposureRecordQSPtr record_qsp, m_records)
    {
      if(record_qsp->getObjectName() == object_name)
        return record_qsp;
    }

  return nullptr;
}

JavaScriptQObjectExposureRecordQSPtr
JavaScriptQObjectExposureRegistry::getRecordByAlias(const QString object_alias)
{
  foreach(JavaScriptQObjectExposureRecordQSPtr record_qsp, m_records)
    {
      if(record_qsp->getObjectName() == object_alias)
        return record_qsp;
    }

  return nullptr;
}

/*!
\brief Registers \a object_p with all the available data.

\list

\li \a object_parent_p, the object that is the parent of \a object_p
\li \a object_name, the name with which the object is exposed (property name)
\li \a object_alias, the alias with which the object can be named
\li \a object_description, the description of the object
\endlist

The only compulsory argument is \a object_p.

Returns 0 if no new record was made because one by the object was found already
or 1 if a new record was made. If an error occurs -1 is returned.
*/
int
JavaScriptQObjectExposureRegistry::registerQObject(
  QObject *object_p,
  QObject *object_parent_p,
  const QString &object_name,
  const QString &object_alias,
  const QString &object_description)
{
  if(getRecordByObject(object_p))
    {
      qDebug() << "One record exists already by the same object";
      return 0;
    }

  m_records.append(QSharedPointer<JavaScriptQObjectExposureRecord>::create(
    object_p, object_parent_p, object_name, object_alias, object_description));

  connect(object_p, &QObject::destroyed, [&]() {
    unregisterQObject(object_p);
  });

  return 1;
}

/*!
\brief Removes the records for \a object_p from this registry.

Returns the count of removed records.
*/
int
JavaScriptQObjectExposureRegistry::unregisterQObject(QObject *object_p)
{
  qsizetype removed_count = 0;

  if(object_p == nullptr)
    {
      qDebug() << "The passed pointer is nullptr.";
      return removed_count;
    }

  JavaScriptQObjectExposureRecordQSPtr record_qsp = getRecordByObject(object_p);

  if(record_qsp != nullptr)
    removed_count = m_records.removeAll(record_qsp);

  return removed_count;
}


} // namespace libXpertMassGui
} // namespace MsXpS
