# pyOCD debugger
# Copyright (c) 2021 Huada Semiconductor Corporation
# Copyright (c) 2021 Chris Reed
# SPDX-License-Identifier: Apache-2.0
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

from ...coresight.coresight_target import CoreSightTarget
from ...core.memory_map import (FlashRegion, RamRegion, MemoryMap)
from ...debug.svd.loader import SVDFile


class DBGMCU:
    STCTL = 0xE0042020
    STCTL_VALUE = 0x3

    TRACECTL = 0xE0042024
    TRACECTL_VALUE = 0x0

FLASH_ALGO = {
    'load_address' : 0x20000000,

    # Flash algorithm as a hex string. HDSC.HC32F460.1.0.12.pack 10965
    'instructions': [
    0xE00ABE00,
    0x4770ba40, 0x4770ba40, 0x4770ba40, 0x4770bac0, 0x4770bac0, 0x4770bac0, 0x0030ea4f, 0x00004770,
    0x0030ea4f, 0x00004770, 0x0030ea4f, 0x00004770, 0x5001f24a, 0x80084908, 0x49082000, 0x0036f881,
    0x0026f881, 0x68004806, 0x60084905, 0x49032001, 0x002af881, 0x00004770, 0x400543fe, 0x40054000,
    0x40010408, 0x5101f24a, 0x80114a17, 0xf8914917, 0xf0411036, 0x4a150101, 0x1036f882, 0x4104f240,
    0xf0016809, 0xb1480001, 0x4a124911, 0x49126011, 0xf0416809, 0x4a100140, 0xe0086011, 0x4a0d490f,
    0x490d6011, 0xf0416809, 0x4a0b0150, 0x490c6011, 0x62114a06, 0x4a052100, 0x7011322a, 0x4a032105,
    0x1026f882, 0x00004770, 0x400543fe, 0x40054000, 0x10102781, 0x40054100, 0x40010408, 0x10101f81,
    0x00116310, 0x4603b570, 0xe003460c, 0x5b01f814, 0x5b01f803, 0xf1a21e15, 0xd1f70201, 0xb510bd70,
    0xf840f000, 0xb510bd10, 0x46204604, 0xf882f000, 0x0000bd10, 0xf000b510, 0x2000f9eb, 0x60084913,
    0x4912203f, 0x6008310c, 0x49102000, 0x60083908, 0x490f2001, 0x480f7008, 0x3926490d, 0x0100f8c1,
    0x62082000, 0x490a2001, 0x70081d09, 0x490b480a, 0x12c06008, 0x480a6048, 0xf44f6108, 0x49044025,
    0xf8a13926, 0xf00003fe, 0xbd10f9c3, 0x40010408, 0x40054026, 0x11101300, 0xfffffaee, 0x40048000,
    0xa5a50000, 0xf000b510, 0x4821f9b3, 0xf0206800, 0x491f0001, 0x46086008, 0xf0406800, 0x60080001,
    0x68004608, 0x0070f020, 0x46086008, 0xf0406800, 0x60080050, 0x68004608, 0x7080f420, 0x46086008,
    0xf4406800, 0x60087080, 0x2034f241, 0x60082100, 0xf000e001, 0x480ef98d, 0x68001d00, 0x7080f400,
    0x7f80f5b0, 0x480ad1f5, 0xf0206800, 0x49080070, 0x46086008, 0x60086800, 0x68004608, 0x0001f020,
    0x46086008, 0x60086800, 0xf972f000, 0xbd102000, 0x4001040c, 0x4604b570, 0x4540f649, 0xf968f000,
    0x68004824, 0x0001f020, 0x60084922, 0x68004608, 0x0001f040, 0x46086008, 0xf0206800, 0x60080070,
    0x68004608, 0x0040f040, 0x46086008, 0xf4206800, 0x60087080, 0x68004608, 0x7080f440, 0xf0006008,
    0xbf00f947, 0x60202000, 0xf942f000, 0xf000e001, 0x4810f93f, 0x68001d00, 0x7080f400, 0x7f80f5b0,
    0x1e68d002, 0xd1f21e05, 0x6800480a, 0x0070f020, 0x60084908, 0x68004608, 0x46086008, 0xf0206800,
    0x60080001, 0x68004608, 0xf0006008, 0x2000f921, 0x0000bd70, 0x4001040c, 0xf240b510, 0x49071023,
    0xf2436008, 0x60082010, 0x5001f24a, 0x80084904, 0xfec8f7ff, 0xf90cf000, 0x0000bd10, 0x40010400,
    0x400543fe, 0x45f8e92d, 0x460e4680, 0xf04f4615, 0x900030ff, 0x4740f649, 0x494f203f, 0xf0006008,
    0x484df8f7, 0x68003808, 0x0001f020, 0x3908494a, 0x46086008, 0xf0406800, 0x60080001, 0x68004608,
    0x0070f020, 0x46086008, 0xf0406800, 0x60080030, 0x68004608, 0x7080f420, 0x46086008, 0xf4406800,
    0x60087080, 0xbf004644, 0xf8d2f000, 0x0a00f04f, 0xf649e02f, 0x48394740, 0xd2024284, 0x60206828,
    0x6829e003, 0x7044f04f, 0x1d245101, 0xf0001d2d, 0xe001f8bf, 0xf8bcf000, 0x1f00482f, 0xf0006800,
    0x28100010, 0x1e78d002, 0xd1f31e07, 0x2001b917, 0x85f8e8bd, 0x1f004828, 0xf0006800, 0xb118000f,
    0xf8a6f000, 0xe7f32001, 0x49232010, 0xf10a6008, 0xebba0a01, 0xd3cc0f96, 0x0003f006, 0xf006b138,
    0x46290203, 0xf7ff4668, 0x9800fe8d, 0xf000c401, 0x4819f88f, 0x68003808, 0x7080f420, 0x39084916,
    0x46086008, 0x60086800, 0x68004608, 0x0070f020, 0x46086008, 0x60086800, 0x68004608, 0x0001f020,
    0x46086008, 0x60086800, 0x4740f649, 0xf000e001, 0x4809f86f, 0x68001f00, 0x7080f400, 0x7f80f5b0,
    0x1e78d002, 0xd1f21e07, 0x2001b90f, 0xf000e7b0, 0x2000f85f, 0x0000e7ac, 0x40010414, 0x0007ffe0,
    0x4604b570, 0x4616460d, 0xff36f7ff, 0xbd702000, 0x4604b570, 0x4616460d, 0x46294632, 0xf7ff4620,
    0xbd70ff41, 0x49034802, 0x48036008, 0x47706008, 0xffff0123, 0x40049408, 0xffff3210, 0x4604b510,
    0xfe48f7ff, 0xbd102000, 0x4604b5f0, 0x2300460d, 0x27002600, 0x21004626, 0xf856e007, 0x6810cb04,
    0xd0004584, 0x1d12e004, 0xebb11c49, 0xd3f40f95, 0x4637bf00, 0xe0062300, 0xcb01f817, 0x45845cd0,
    0xe004d000, 0xf0051c5b, 0x42980003, 0xbf00d8f4, 0x0081eb04, 0xbdf04418, 0x49034802, 0x48036088,
    0x47706088, 0xffff0123, 0x40049000, 0xffff3210, 0x4807b510, 0xf4006800, 0xb9083080, 0xf84ef000,
    0x68004803, 0x0001f000, 0xf000b908, 0xbd10f803, 0x40010680, 0x481fb570, 0xb2846800, 0x6800481e,
    0x0681f3c0, 0x6800481c, 0x2503f3c0, 0x11a4b90e, 0x2e01e008, 0x12a4d101, 0x2e02e004, 0x1324d101,
    0x13a4e000, 0x2d0fb10d, 0xf7ffd102, 0xe020ff8b, 0x0001f005, 0xb9e4b118, 0xff84f7ff, 0xf005e019,
    0x28020002, 0x2c01d104, 0xf7ffd113, 0xe010ff7b, 0x0004f005, 0xd1042804, 0xd10a2c02, 0xff72f7ff,
    0xf005e007, 0x28080008, 0x2c03d103, 0xf7ffd101, 0xbd70ff69, 0x40049404, 0x40010680, 0x4823b570,
    0xb2846840, 0x68004822, 0x4681f3c0, 0x68004820, 0x6503f3c0, 0x11a4b90e, 0x2e01e008, 0x12a4d101,
    0x2e02e004, 0x1324d101, 0x13a4e000, 0x2001b90d, 0x2000e000, 0xd1012d0f, 0xe0002101, 0x43082100,
    0xf7ffb110, 0xe020ff79, 0x0001f005, 0xb9e4b118, 0xff72f7ff, 0xf005e019, 0x28020002, 0x2c01d104,
    0xf7ffd113, 0xe010ff69, 0x0004f005, 0xd1042804, 0xd10a2c02, 0xff60f7ff, 0xf005e007, 0x28080008,
    0x2c03d103, 0xf7ffd101, 0xbd70ff57, 0x40049000, 0x40010680, 0x00000000
    ],

    # Relative function addresses
    'pc_init': 0x20000445,
    'pc_unInit': 0x20000481,
    'pc_program_page': 0x20000455,
    'pc_erase_sector': 0x2000010b,
    'pc_eraseAll': 0x20000103,

    'static_base' : 0x20000000 + 0x00000004 + 0x00000634,
    'begin_stack' : 0x20000900,
    'begin_data' : 0x20000000 + 0x1000,
    'page_size' : 0x200,
    'analyzer_supported' : False,
    'analyzer_address' : 0x00000000,
    'page_buffers' : [0x20001000, 0x20001200],   # Enable double buffering
    'min_program_length' : 0x200,

    # Flash information
    'flash_start': 0x0,
    'flash_size': 0x80000,
    'sector_sizes': (
        (0x0, 0x2000),
    )
}


FLASH_ALGO_OTP = {
    'load_address' : 0x20000000,

    # Flash algorithm as a hex string. HDSC.HC32F460.1.0.12.pack 10965
    'instructions': [
    0xE00ABE00,
    0x4770ba40, 0x4770ba40, 0x4770ba40, 0x4770bac0, 0x4770bac0, 0x4770bac0, 0x0030ea4f, 0x00004770,
    0x0030ea4f, 0x00004770, 0x0030ea4f, 0x00004770, 0x5001f24a, 0x80084908, 0x49082000, 0x0036f881,
    0x0026f881, 0x68004806, 0x60084905, 0x49032001, 0x002af881, 0x00004770, 0x400543fe, 0x40054000,
    0x40010408, 0x5101f24a, 0x80114a17, 0xf8914917, 0xf0411036, 0x4a150101, 0x1036f882, 0x4104f240,
    0xf0016809, 0xb1480001, 0x4a124911, 0x49126011, 0xf0416809, 0x4a100140, 0xe0086011, 0x4a0d490f,
    0x490d6011, 0xf0416809, 0x4a0b0150, 0x490c6011, 0x62114a06, 0x4a052100, 0x7011322a, 0x4a032105,
    0x1026f882, 0x00004770, 0x400543fe, 0x40054000, 0x10102781, 0x40054100, 0x40010408, 0x10101f81,
    0x00116310, 0x4603b570, 0xe003460c, 0x5b01f814, 0x5b01f803, 0xf1a21e15, 0xd1f70201, 0xb510bd70,
    0xf840f000, 0xb510bd10, 0x46204604, 0xf83cf000, 0x0000bd10, 0xf000b510, 0x2000f957, 0x60084913,
    0x4912203f, 0x6008310c, 0x49102000, 0x60083908, 0x490f2001, 0x480f7008, 0x3926490d, 0x0100f8c1,
    0x62082000, 0x490a2001, 0x70081d09, 0x490b480a, 0x12c06008, 0x480a6048, 0xf44f6108, 0x49044025,
    0xf8a13926, 0xf00003fe, 0xbd10f92f, 0x40010408, 0x40054026, 0x11101300, 0xfffffaee, 0x40048000,
    0xa5a50000, 0x47702000, 0x20004601, 0x00004770, 0xf240b510, 0x49071023, 0xf2436008, 0x60082010,
    0x5001f24a, 0x80084904, 0xff5cf7ff, 0xf90cf000, 0x0000bd10, 0x40010400, 0x400543fe, 0x45f8e92d,
    0x460e4680, 0xf04f4615, 0x900030ff, 0x4740f649, 0x494f203f, 0xf0006008, 0x484df8f7, 0x68003808,
    0x0001f020, 0x3908494a, 0x46086008, 0xf0406800, 0x60080001, 0x68004608, 0x0070f020, 0x46086008,
    0xf0406800, 0x60080030, 0x68004608, 0x7080f420, 0x46086008, 0xf4406800, 0x60087080, 0xbf004644,
    0xf8d2f000, 0x0a00f04f, 0xf649e02f, 0x48394740, 0xd2024284, 0x60206828, 0x6829e003, 0x7044f04f,
    0x1d245101, 0xf0001d2d, 0xe001f8bf, 0xf8bcf000, 0x1f00482f, 0xf0006800, 0x28100010, 0x1e78d002,
    0xd1f31e07, 0x2001b917, 0x85f8e8bd, 0x1f004828, 0xf0006800, 0xb118000f, 0xf8a6f000, 0xe7f32001,
    0x49232010, 0xf10a6008, 0xebba0a01, 0xd3cc0f96, 0x0003f006, 0xf006b138, 0x46290203, 0xf7ff4668,
    0x9800ff21, 0xf000c401, 0x4819f88f, 0x68003808, 0x7080f420, 0x39084916, 0x46086008, 0x60086800,
    0x68004608, 0x0070f020, 0x46086008, 0x60086800, 0x68004608, 0x0001f020, 0x46086008, 0x60086800,
    0x4740f649, 0xf000e001, 0x4809f86f, 0x68001f00, 0x7080f400, 0x7f80f5b0, 0x1e78d002, 0xd1f21e07,
    0x2001b90f, 0xf000e7b0, 0x2000f85f, 0x0000e7ac, 0x40010414, 0x0007ffe0, 0x4604b570, 0x4616460d,
    0xff36f7ff, 0xbd702000, 0x4604b570, 0x4616460d, 0x46294632, 0xf7ff4620, 0xbd70ff41, 0x49034802,
    0x48036008, 0x47706008, 0xffff0123, 0x40049408, 0xffff3210, 0x4604b510, 0xfedcf7ff, 0xbd102000,
    0x4604b5f0, 0x2300460d, 0x27002600, 0x21004626, 0xf856e007, 0x6810cb04, 0xd0004584, 0x1d12e004,
    0xebb11c49, 0xd3f40f95, 0x4637bf00, 0xe0062300, 0xcb01f817, 0x45845cd0, 0xe004d000, 0xf0051c5b,
    0x42980003, 0xbf00d8f4, 0x0081eb04, 0xbdf04418, 0x49034802, 0x48036088, 0x47706088, 0xffff0123,
    0x40049000, 0xffff3210, 0x4807b510, 0xf4006800, 0xb9083080, 0xf84ef000, 0x68004803, 0x0001f000,
    0xf000b908, 0xbd10f803, 0x40010680, 0x481fb570, 0xb2846800, 0x6800481e, 0x0681f3c0, 0x6800481c,
    0x2503f3c0, 0x11a4b90e, 0x2e01e008, 0x12a4d101, 0x2e02e004, 0x1324d101, 0x13a4e000, 0x2d0fb10d,
    0xf7ffd102, 0xe020ff8b, 0x0001f005, 0xb9e4b118, 0xff84f7ff, 0xf005e019, 0x28020002, 0x2c01d104,
    0xf7ffd113, 0xe010ff7b, 0x0004f005, 0xd1042804, 0xd10a2c02, 0xff72f7ff, 0xf005e007, 0x28080008,
    0x2c03d103, 0xf7ffd101, 0xbd70ff69, 0x40049404, 0x40010680, 0x4823b570, 0xb2846840, 0x68004822,
    0x4681f3c0, 0x68004820, 0x6503f3c0, 0x11a4b90e, 0x2e01e008, 0x12a4d101, 0x2e02e004, 0x1324d101,
    0x13a4e000, 0x2001b90d, 0x2000e000, 0xd1012d0f, 0xe0002101, 0x43082100, 0xf7ffb110, 0xe020ff79,
    0x0001f005, 0xb9e4b118, 0xff72f7ff, 0xf005e019, 0x28020002, 0x2c01d104, 0xf7ffd113, 0xe010ff69,
    0x0004f005, 0xd1042804, 0xd10a2c02, 0xff60f7ff, 0xf005e007, 0x28080008, 0x2c03d103, 0xf7ffd101,
    0xbd70ff57, 0x40049000, 0x40010680, 0x00000000
    ],

    # Relative function addresses
    'pc_init': 0x2000031d,
    'pc_unInit': 0x20000359,
    'pc_program_page': 0x2000032d,
    'pc_erase_sector': 0x2000010b,
    'pc_eraseAll': 0x20000103,

    'static_base' : 0x20000000 + 0x00000004 + 0x0000050c,
    'begin_stack' : 0x20000800,
    'begin_data' : 0x20000000 + 0x1000,
    'page_size' : 0x3fc,
    'analyzer_supported' : False,
    'analyzer_address' : 0x00000000,
    'page_buffers' : [0x20001000, 0x200013fc],   # Enable double buffering
    'min_program_length' : 0x3fc,

    # Flash information
    'flash_start': 0x3000c00,
    'flash_size': 0x3fc,
    'sector_sizes': (
        (0x0, 0x3fc),
    )
}


class HC32F460xC(CoreSightTarget):

    VENDOR = "HDSC"

    MEMORY_MAP = MemoryMap(
        FlashRegion( start=0x00000000, length=0x40000, page_size=0x200, sector_size=0x2000,
                        is_boot_memory=True,
                        algo=FLASH_ALGO),
        FlashRegion( start=0x03000C00, length=0x3FC, sector_size=0x3FC,
                        is_boot_memory=False,
                        is_default=False,
                        algo=FLASH_ALGO_OTP),
        RamRegion(   start=0x1FFF8000, length=0x2F000),
        RamRegion(   start=0x200F0000, length=0x1000)
        )

    def __init__(self, session):
        super(HC32F460xC, self).__init__(session, self.MEMORY_MAP)
        self._svd_location = SVDFile.from_builtin("HC32F460.svd")

    def post_connect_hook(self):
        self.write32(DBGMCU.STCTL, DBGMCU.STCTL_VALUE)
        self.write32(DBGMCU.TRACECTL, DBGMCU.TRACECTL_VALUE)


class HC32F460xE(CoreSightTarget):

    VENDOR = "HDSC"

    MEMORY_MAP = MemoryMap(
        FlashRegion( start=0x00000000, length=0x80000, page_size=0x200, sector_size=0x2000,
                        is_boot_memory=True,
                        algo=FLASH_ALGO),
        FlashRegion( start=0x03000C00, length=0x3FC, sector_size=0x3FC,
                        is_boot_memory=False,
                        is_default=False,
                        algo=FLASH_ALGO_OTP),
        RamRegion(   start=0x1FFF8000, length=0x2F000),
        RamRegion(   start=0x200F0000, length=0x1000)
        )

    def __init__(self, session):
        super(HC32F460xE, self).__init__(session, self.MEMORY_MAP)
        self._svd_location = SVDFile.from_builtin("HC32F460.svd")

    def post_connect_hook(self):
        self.write32(DBGMCU.STCTL, DBGMCU.STCTL_VALUE)
        self.write32(DBGMCU.TRACECTL, DBGMCU.TRACECTL_VALUE)

