/*
 * Asterisk -- An open source telephony toolkit.
 *
 * Copyright (C) 1999 - 2008, Digium, Inc.
 *
 * Mark Spencer <markster@digium.com>
 *
 * See http://www.asterisk.org for more information about
 * the Asterisk project. Please do not directly contact
 * any of the maintainers of this project for assistance;
 * the project provides a web site, mailing lists and IRC
 * channels for your use.
 *
 * This program is free software, distributed under the terms of
 * the GNU General Public License Version 2. See the LICENSE file
 * at the top of the source tree.
 */

/*! \file
 *
 * \brief DAHDI for Pseudo TDM
 *
 * \author Mark Spencer <markster@digium.com>
 *
 * Connects to the DAHDI telephony library as well as
 * libpri. Libpri is optional and needed only if you are
 * going to use ISDN connections.
 *
 * You need to install libraries before you attempt to compile
 * and install the DAHDI channel.
 *
 * \ingroup channel_drivers
 *
 * \todo Deprecate the "musiconhold" configuration option post 1.4
 */

/*! \li \ref chan_dahdi.c uses the configuration file \ref chan_dahdi.conf
 * \addtogroup configuration_file
 */

/*! \page chan_dahdi.conf chan_dahdi.conf
 * \verbinclude chan_dahdi.conf.sample
 */

/*** MODULEINFO
	<depend>dahdi</depend>
	<depend>tonezone</depend>
	<use type="module">res_smdi</use>
	<use type="external">pri</use>
	<use type="external">ss7</use>
	<use type="external">openr2</use>
	<support_level>core</support_level>
 ***/

#include "asterisk.h"

#if defined(__NetBSD__) || defined(__FreeBSD__)
#include <pthread.h>
#else
#include <sys/sysmacros.h>
#endif
#include <signal.h>
#include <sys/stat.h>
#include <math.h>

#include "sig_analog.h"
/* Analog signaling is currently still present in chan_dahdi for use with
 * radio. Sig_analog does not currently handle any radio operations. If
 * radio only uses analog signaling, then the radio handling logic could
 * be placed in sig_analog and the duplicated code could be removed.
 */

#if defined(HAVE_PRI)
#include "sig_pri.h"
#ifndef PRI_RESTART
#error "Upgrade your libpri"
#endif
#endif	/* defined(HAVE_PRI) */

#if defined(HAVE_SS7)
#include "sig_ss7.h"
#if !defined(LIBSS7_ABI_COMPATIBILITY)
#error "Upgrade your libss7"
#elif LIBSS7_ABI_COMPATIBILITY != 2
#error "Your installed libss7 is not compatible"
#endif
#endif	/* defined(HAVE_SS7) */

#if defined(HAVE_OPENR2)
/* put this here until sig_mfcr2 comes along */
#define SIG_MFCR2_MAX_CHANNELS	672		/*!< No more than a DS3 per trunk group */
#endif	/* defined(HAVE_OPENR2) */

#include "asterisk/lock.h"
#include "asterisk/channel.h"
#include "asterisk/config.h"
#include "asterisk/module.h"
#include "asterisk/pbx.h"
#include "asterisk/file.h"
#include "asterisk/ulaw.h"
#include "asterisk/alaw.h"
#include "asterisk/callerid.h"
#include "asterisk/adsi.h"
#include "asterisk/cli.h"
#include "asterisk/pickup.h"
#include "asterisk/features.h"
#include "asterisk/musiconhold.h"
#include "asterisk/say.h"
#include "asterisk/tdd.h"
#include "asterisk/mwi.h"
#include "asterisk/dsp.h"
#include "asterisk/astdb.h"
#include "asterisk/manager.h"
#include "asterisk/causes.h"
#include "asterisk/term.h"
#include "asterisk/utils.h"
#include "asterisk/transcap.h"
#include "asterisk/stringfields.h"
#include "asterisk/abstract_jb.h"
#include "asterisk/smdi.h"
#include "asterisk/devicestate.h"
#include "asterisk/paths.h"
#include "asterisk/ccss.h"
#include "asterisk/features_config.h"
#include "asterisk/bridge.h"
#include "asterisk/stasis_channels.h"
#include "asterisk/parking.h"
#include "asterisk/format_cache.h"
#include "chan_dahdi.h"
#include "dahdi/bridge_native_dahdi.h"

/*** DOCUMENTATION
	<application name="DAHDISendKeypadFacility" language="en_US">
		<since>
			<version>1.4.22</version>
		</since>
		<synopsis>
			Send digits out of band over a PRI.
		</synopsis>
		<syntax>
			<parameter name="digits" required="true" />
		</syntax>
		<description>
			<para>This application will send the given string of digits in a Keypad
			Facility IE over the current channel.</para>
		</description>
	</application>
	<application name="DAHDISendCallreroutingFacility" language="en_US">
		<since>
			<version>1.6.2.0</version>
		</since>
		<synopsis>
			Send an ISDN call rerouting/deflection facility message.
		</synopsis>
		<syntax argsep=",">
			<parameter name="destination" required="true">
				<para>Destination number.</para>
			</parameter>
			<parameter name="original">
				<para>Original called number.</para>
			</parameter>
			<parameter name="reason">
				<para>Diversion reason, if not specified defaults to <literal>unknown</literal></para>
			</parameter>
		</syntax>
		<description>
			<para>This application will send an ISDN switch specific call
			rerouting/deflection facility message over the current channel.
			Supported switches depend upon the version of libpri in use.</para>
		</description>
	</application>
	<application name="DAHDIAcceptR2Call" language="en_US">
		<since>
			<version>1.6.1.0</version>
		</since>
		<synopsis>
			Accept an R2 call if its not already accepted (you still need to answer it)
		</synopsis>
		<syntax>
			<parameter name="charge" required="true">
				<para>Yes or No.</para>
				<para>Whether you want to accept the call with charge or without charge.</para>
			</parameter>
		</syntax>
		<description>
			<para>This application will Accept the R2 call either with charge or no charge.</para>
		</description>
	</application>
	<function name="POLARITY" language="en_US">
		<since>
			<version>16.28.0</version>
			<version>18.14.0</version>
			<version>19.6.0</version>
		</since>
		<synopsis>
			Set or get the polarity of a DAHDI channel.
		</synopsis>
		<syntax />
		<description>
			<para>The POLARITY function can be used to set the polarity of a DAHDI channel.</para>
			<para>Applies only to FXS channels (using FXO signalling) with supporting hardware.</para>
			<para>The polarity can be set to the following numeric or named values:</para>
			<enumlist>
				<enum name="0" />
				<enum name="idle" />
				<enum name="1" />
				<enum name="reverse" />
			</enumlist>
			<para>However, when read, the function will always return 0 or 1.</para>
			<example title="Set idle polarity">
			same => n,Set(POLARITY()=0)
			</example>
			<example title="Set reverse polarity">
			same => n,NoOp(Current Polarity: ${POLARITY()})
			same => n,Set(POLARITY()=reverse)
			same => n,NoOp(New Polarity: ${POLARITY()})
			</example>
			<example title="Reverse the polarity from whatever it is currently">
			same => n,Set(POLARITY()=${IF($[ "${POLARITY()}" = "1" ]?0:1)})
			</example>
		</description>
	</function>
	<function name="DAHDI_CHANNEL" language="en_US">
		<synopsis>
			Set or get a property of a DAHDI channel.
		</synopsis>
		<syntax>
			<parameter name="property" required="true">
				<para>The property to set or get.</para>
				<enumlist>
					<enum name="owner">
						<para>R/O The name of the active channel on this DAHDI device.</para>
					</enum>
					<enum name="callwait">
						<para>R/O The name of the call waiting channel on this DAHDI device.</para>
					</enum>
					<enum name="threeway">
						<para>R/O The name of the three-way channel on this DAHDI device.</para>
					</enum>
					<enum name="polarity">
						<para>R/W The polarity of the channel (0 or 1, idle or reverse can also be used for setting).</para>
						<para>Only valid on FXO-signalled channels.</para>
					</enum>
					<enum name="dnd">
						<para>R/W Whether Do Not Disturb is active.</para>
						<para>Only valid on FXO-signalled channels.</para>
					</enum>
					<enum name="callforward">
						<para>R/W The call forwarding target number.</para>
						<para>Only valid on FXO-signalled channels.</para>
					</enum>
					<enum name="lastexten">
						<para>R/W The last number dialed.</para>
						<para>Only valid on FXO-signalled channels.</para>
					</enum>
				</enumlist>
			</parameter>
			<parameter name="channel" required="false">
				<para>The DAHDI channel number.</para>
				<para>If not specified, the DAHDI channel number of the current channel
				will be used (in which case it must be a DAHDI channel).</para>
			</parameter>
		</syntax>
		<description>
			<para>The DAHDI_CHANNEL function can be used to set or get properties of DAHDI channels
			that are not specific to Asterisk channels.</para>
			<para>This function may also be called from non-DAHDI channels.</para>
			<example title="Set reverse polarity on current DAHDI channel">
			same => n,Set(DAHDI_CHANNEL(polarity)=1)
			same => n,NoOp(${DAHDI_CHANNEL(polarity)})
			</example>
			<example title="Set call forwarding target for channel 1">
			same => n,Set(DAHDI_CHANNEL(callforwardnumber,1)=5552368)
			same => n,NoOp(Calls now forwarding to ${DAHDI_CHANNEL(callforwardnumber,1)})
			</example>
		</description>
	</function>
	<info name="CHANNEL" language="en_US" tech="DAHDI">
		<enumlist>
			<enum name="dahdi_channel">
				<para>R/O DAHDI channel related to this channel.</para>
			</enum>
			<enum name="dahdi_span">
				<para>R/O DAHDI span related to this channel.</para>
			</enum>
			<enum name="dahdi_group">
				<para>R/O DAHDI logical group related to this channel.</para>
			</enum>
			<enum name="dahdi_type">
				<para>R/O DAHDI channel type, one of:</para>
				<enumlist>
					<enum name="analog" />
					<enum name="mfc/r2" />
					<enum name="pri" />
					<enum name="pseudo" />
					<enum name="ss7" />
				</enumlist>
			</enum>
			<enum name="keypad_digits">
				<para>R/O PRI Keypad digits that came in with the SETUP message.</para>
			</enum>
			<enum name="reversecharge">
				<para>R/O PRI Reverse Charging Indication, one of:</para>
				<enumlist>
					<enum name="-1"> <para>None</para></enum>
					<enum name=" 1"> <para>Reverse Charging Requested</para></enum>
				</enumlist>
			</enum>
			<enum name="no_media_path">
				<para>R/O PRI Nonzero if the channel has no B channel.
				The channel is either on hold or a call waiting call.</para>
			</enum>
			<enum name="buffers">
				<para>W/O Change the channel's buffer policy (for the current call only)</para>
				<para>This option takes two arguments:</para>
				<para>	Number of buffers,</para>
				<para>	Buffer policy being one of:</para>
				<para>	    <literal>full</literal></para>
				<para>	    <literal>immediate</literal></para>
				<para>	    <literal>half</literal></para>
			</enum>
			<enum name="echocan_mode">
				<para>W/O Change the configuration of the active echo
				canceller on the channel (if any), for the current call
				only.</para>
				<para>Possible values are:</para>
				<para>	<literal>on</literal>	Normal mode (the echo canceller is actually reinitialized)</para>
				<para>	<literal>off</literal>	Disabled</para>
				<para>	<literal>fax</literal>	FAX/data mode (NLP disabled if possible, otherwise
					completely disabled)</para>
				<para>	<literal>voice</literal>	Voice mode (returns from FAX mode, reverting the changes that were made)</para>
			</enum>
			<enum name="dialmode">
				<para>R/W Pulse and tone dialing mode of the channel.</para>
				<para>Disabling tone dialing using this option will not automatically disable the DSP used for DTMF detection.
				To do that, also set the <literal>digitdetect</literal> option. If digit detection is disabled,
				DTMF will not be detected, regardless of the <literal>dialmode</literal> setting.
				The <literal>digitdetect</literal> setting has no impact on pulse dialing detection.</para>
				<para>If set, overrides the setting in <literal>chan_dahdi.conf</literal> for that channel.</para>
				<para>The <literal>dialmode</literal> setting applies to the DAHDI channel as a whole, but is reset for each call,
				so modifications made using the <literal>CHANNEL</literal> function apply temporarily per-call.
				The <literal>digitdetect</literal> setting applies to the entire DAHDI channel,
				so any changes made to this setting will affect all calls concurrently on the same DAHDI channel.
				<literal>digitdetect</literal> is reset once all calls on the line have cleared.</para>
				<enumlist>
					<enum name="both" />
					<enum name="pulse" />
					<enum name="dtmf" />
					<enum name="tone" />
					<enum name="none" />
				</enumlist>
			</enum>
			<enum name="waitfordialtone">
				<para>W/O Duration in ms for which to wait for dial tone on the current call.</para>
				<para>This setting is will temporarily override the <literal>waitfordialtone</literal>
				setting in <literal>chan_dahdi.conf</literal> (typically if that setting is disabled).
				You must call this in a pre-dial handler when making a call on an analog trunk
				(e.g. FXS-signalled interface).</para>
				<para>This allows, for example, being able to barge in on an in-use trunk,
				if dialed specifically, but allows skipping the trunk when routing calls
				if dial tone is not present on a channel.</para>
				<para>This setting will only apply to the current (next) call made on the
				DAHDI channel, and will not persist for future calls.</para>
				<para>Please keep in mind that due to the way that chan_dahdi implements dial tone detection,
				DTMF digits on an in-use channel will temporarily relay to any other channels attempting to use the channel for a call.
				However, voice transmission will not leak.</para>
			</enum>
		</enumlist>
	</info>
	<info name="Dial_Resource" language="en_US" tech="DAHDI">
		<para>DAHDI allows several modifiers to be specified as part of the resource.</para>
		<para>The general syntax is :</para>
		<para><literal>Dial(DAHDI/pseudo[/extension])</literal></para>
		<para><literal>Dial(DAHDI/&lt;channel#&gt;[c|r&lt;cadence#&gt;|d][/extension])</literal></para>
		<para><literal>Dial(DAHDI/(g|G|r|R)&lt;group#(0-63)&gt;[c|r&lt;cadence#&gt;|d][/extension])</literal></para>
		<para>The following modifiers may be used before the channel number:</para>
		<enumlist>
		<enum name="g">
			<para>Search forward, dialing on first available channel in group (lowest to highest).</para>
		</enum>
		<enum name="G">
			<para>Search backward, dialing on first available channel in group (highest to lowest).</para>
		</enum>
		<enum name="r">
			<para>Round robin search forward, picking up from where last left off (lowest to highest).</para>
		</enum>
		<enum name="R">
			<para>Round robin search backward, picking up from where last left off (highest to lowest).</para>
		</enum>
		</enumlist>
		<para>The following modifiers may be used after the channel number:</para>
		<enumlist>
		<enum name="c">
			<para>Wait for DTMF digit <literal>#</literal> before providing answer supervision.</para>
			<para>This can be useful on outbound calls via FXO ports, as otherwise
			they would indicate answer immediately.</para>
		</enum>
		<enum name="d">
			<para>Force bearer capability for ISDN/SS7 call to digital.</para>
		</enum>
		<enum name="i">
			<para>ISDN span channel restriction.</para>
			<para>Used by CC to ensure that the CC recall goes out the same span.
			Also to make ISDN channel names dialable when the sequence number
			is stripped off.  (Used by DTMF attended transfer feature.)</para>
		</enum>
		<enum name="r">
			<para>Specifies the distinctive ring cadence number to use immediately after
			specifying this option. There are 4 default built-in cadences, and up to 24
			total cadences may be configured.</para>
		</enum>
		</enumlist>
		<example title="Dial 555-1212 on first available channel in group 1, searching from highest to lowest">
		same => n,Dial(DAHDI/g1/5551212)
		</example>
		<example title="Ringing FXS channel 4 with ring cadence 2">
		same => n,Dial(DAHDI/4r2)
		</example>
		<example title="Dial 555-1212 on channel 3 and require answer confirmation">
		same => n,Dial(DAHDI/3c/5551212)
		</example>
	</info>
	<manager name="DAHDITransfer" language="en_US">
		<since>
			<version>1.4.22</version>
		</since>
		<synopsis>
			Transfer DAHDI Channel.
		</synopsis>
		<syntax>
			<xi:include xpointer="xpointer(/docs/manager[@name='Login']/syntax/parameter[@name='ActionID'])" />
			<parameter name="DAHDIChannel" required="true">
				<para>DAHDI channel number to transfer.</para>
			</parameter>
		</syntax>
		<description>
			<para>Simulate a flash hook event by the user connected to the channel.</para>
			<note><para>Valid only for analog channels.</para></note>
		</description>
	</manager>
	<manager name="DAHDIHangup" language="en_US">
		<since>
			<version>1.4.22</version>
		</since>
		<synopsis>
			Hangup DAHDI Channel.
		</synopsis>
		<syntax>
			<xi:include xpointer="xpointer(/docs/manager[@name='Login']/syntax/parameter[@name='ActionID'])" />
			<parameter name="DAHDIChannel" required="true">
				<para>DAHDI channel number to hangup.</para>
			</parameter>
		</syntax>
		<description>
			<para>Simulate an on-hook event by the user connected to the channel.</para>
			<note><para>Valid only for analog channels.</para></note>
		</description>
	</manager>
	<manager name="DAHDIDialOffhook" language="en_US">
		<since>
			<version>1.4.22</version>
		</since>
		<synopsis>
			Dial over DAHDI channel while offhook.
		</synopsis>
		<syntax>
			<xi:include xpointer="xpointer(/docs/manager[@name='Login']/syntax/parameter[@name='ActionID'])" />
			<parameter name="DAHDIChannel" required="true">
				<para>DAHDI channel number to dial digits.</para>
			</parameter>
			<parameter name="Number" required="true">
				<para>Digits to dial.</para>
			</parameter>
		</syntax>
		<description>
			<para>Generate DTMF control frames to the bridged peer.</para>
		</description>
	</manager>
	<manager name="DAHDIDNDon" language="en_US">
		<since>
			<version>1.4.22</version>
		</since>
		<synopsis>
			Toggle DAHDI channel Do Not Disturb status ON.
		</synopsis>
		<syntax>
			<xi:include xpointer="xpointer(/docs/manager[@name='Login']/syntax/parameter[@name='ActionID'])" />
			<parameter name="DAHDIChannel" required="true">
				<para>DAHDI channel number to set DND on.</para>
			</parameter>
		</syntax>
		<description>
			<para>Equivalent to the CLI command "dahdi set dnd <variable>channel</variable> on".</para>
			<note><para>Feature only supported by analog channels.</para></note>
		</description>
	</manager>
	<manager name="DAHDIDNDoff" language="en_US">
		<since>
			<version>1.4.22</version>
		</since>
		<synopsis>
			Toggle DAHDI channel Do Not Disturb status OFF.
		</synopsis>
		<syntax>
			<xi:include xpointer="xpointer(/docs/manager[@name='Login']/syntax/parameter[@name='ActionID'])" />
			<parameter name="DAHDIChannel" required="true">
				<para>DAHDI channel number to set DND off.</para>
			</parameter>
		</syntax>
		<description>
			<para>Equivalent to the CLI command "dahdi set dnd <variable>channel</variable> off".</para>
			<note><para>Feature only supported by analog channels.</para></note>
		</description>
	</manager>
	<manager name="DAHDIShowChannels" language="en_US">
		<since>
			<version>1.4.22</version>
		</since>
		<synopsis>
			Show status of DAHDI channels.
		</synopsis>
		<syntax>
			<xi:include xpointer="xpointer(/docs/manager[@name='Login']/syntax/parameter[@name='ActionID'])" />
			<parameter name="DAHDIChannel">
				<para>Specify the specific channel number to show.  Show all channels if zero or not present.</para>
			</parameter>
		</syntax>
		<description>
			<para>Similar to the CLI command "dahdi show channels".</para>
		</description>
	</manager>
	<manager name="DAHDIShowStatus" language="en_US">
		<since>
			<version>21.3.0</version>
			<version>20.8.0</version>
			<version>18.23.0</version>
		</since>
		<synopsis>
			Show status of DAHDI spans.
		</synopsis>
		<syntax/>
		<description>
			<para>Similar to the CLI command "dahdi show status".</para>
		</description>
	</manager>
	<manager name="DAHDIRestart" language="en_US">
		<since>
			<version>1.4.22</version>
		</since>
		<synopsis>
			Fully Restart DAHDI channels (terminates calls).
		</synopsis>
		<syntax>
			<xi:include xpointer="xpointer(/docs/manager[@name='Login']/syntax/parameter[@name='ActionID'])" />
		</syntax>
		<description>
			<para>Equivalent to the CLI command "dahdi restart".</para>
		</description>
	</manager>
	<manager name="PRIShowSpans" language="en_US">
		<since>
			<version>10.0.0</version>
		</since>
		<synopsis>
			Show status of PRI spans.
		</synopsis>
		<syntax>
			<xi:include xpointer="xpointer(/docs/manager[@name='Login']/syntax/parameter[@name='ActionID'])" />
			<parameter name="Span">
				<para>Specify the specific span to show.  Show all spans if zero or not present.</para>
			</parameter>
		</syntax>
		<description>
			<para>Similar to the CLI command "pri show spans".</para>
		</description>
	</manager>
	<manager name="PRIDebugSet" language="en_US">
		<since>
			<version>13.0.0</version>
		</since>
		<synopsis>
			Set PRI debug levels for a span
		</synopsis>
		<syntax>
			<xi:include xpointer="xpointer(/docs/manager[@name='Login']/syntax/parameter[@name='ActionID'])" />
			<parameter name="Span" required="true">
				<para>Which span to affect.</para>
			</parameter>
			<parameter name="Level" required="true">
				<para>What debug level to set. May be a numerical value or a text value from the list below</para>
				<enumlist>
					<enum name="off" />
					<enum name="on" />
					<enum name="hex" />
					<enum name="intense" />
				</enumlist>
			</parameter>
		</syntax>
		<description>
			<para>Equivalent to the CLI command "pri set debug &lt;level&gt; span &lt;span&gt;".</para>
		</description>
	</manager>
	<manager name="PRIDebugFileSet" language="en_US">
		<since>
			<version>13.0.0</version>
		</since>
		<synopsis>
			Set the file used for PRI debug message output
		</synopsis>
		<syntax>
			<xi:include xpointer="xpointer(/docs/manager[@name='Login']/syntax/parameter[@name='ActionID'])" />
			<parameter name="File" required="true">
				<para>Path of file to write debug output.</para>
			</parameter>
		</syntax>
		<description>
			<para>Equivalent to the CLI command "pri set debug file &lt;output-file&gt;"</para>
		</description>
	</manager>
	<manager name="PRIDebugFileUnset" language="en_US">
		<since>
			<version>13.0.0</version>
		</since>
		<synopsis>
			Disables file output for PRI debug messages
		</synopsis>
		<syntax>
			<xi:include xpointer="xpointer(/docs/manager[@name='Login']/syntax/parameter[@name='ActionID'])" />
		</syntax>
	</manager>
	<managerEvent language="en_US" name="AlarmClear">
		<managerEventInstance class="EVENT_FLAG_SYSTEM">
			<since>
				<version>12.0.0</version>
			</since>
			<synopsis>Raised when an alarm is cleared on a DAHDI channel.</synopsis>
			<syntax>
				<parameter name="DAHDIChannel">
					<para>The DAHDI channel on which the alarm was cleared.</para>
					<note><para>This is not an Asterisk channel identifier.</para></note>
				</parameter>
			</syntax>
		</managerEventInstance>
	</managerEvent>
	<managerEvent language="en_US" name="SpanAlarmClear">
		<managerEventInstance class="EVENT_FLAG_SYSTEM">
			<since>
				<version>12.0.0</version>
			</since>
			<synopsis>Raised when an alarm is cleared on a DAHDI span.</synopsis>
			<syntax>
				<parameter name="Span">
					<para>The span on which the alarm was cleared.</para>
				</parameter>
			</syntax>
		</managerEventInstance>
	</managerEvent>
	<managerEvent language="en_US" name="DNDState">
		<managerEventInstance class="EVENT_FLAG_SYSTEM">
			<since>
				<version>12.0.0</version>
			</since>
			<synopsis>Raised when the Do Not Disturb state is changed on a DAHDI channel.</synopsis>
			<syntax>
				<parameter name="DAHDIChannel">
					<para>The DAHDI channel on which DND status changed.</para>
					<note><para>This is not an Asterisk channel identifier.</para></note>
				</parameter>
				<parameter name="Status">
					<enumlist>
						<enum name="enabled"/>
						<enum name="disabled"/>
					</enumlist>
				</parameter>
			</syntax>
		</managerEventInstance>
	</managerEvent>
	<managerEvent language="en_US" name="Alarm">
		<managerEventInstance class="EVENT_FLAG_SYSTEM">
			<since>
				<version>12.0.0</version>
			</since>
			<synopsis>Raised when an alarm is set on a DAHDI channel.</synopsis>
			<syntax>
				<parameter name="DAHDIChannel">
					<para>The channel on which the alarm occurred.</para>
					<note><para>This is not an Asterisk channel identifier.</para></note>
				</parameter>
				<parameter name="Alarm">
					<para>A textual description of the alarm that occurred.</para>
				</parameter>
			</syntax>
		</managerEventInstance>
	</managerEvent>
	<managerEvent language="en_US" name="SpanAlarm">
		<managerEventInstance class="EVENT_FLAG_SYSTEM">
			<since>
				<version>12.0.0</version>
			</since>
			<synopsis>Raised when an alarm is set on a DAHDI span.</synopsis>
			<syntax>
				<parameter name="Span">
					<para>The span on which the alarm occurred.</para>
				</parameter>
				<parameter name="Alarm">
					<para>A textual description of the alarm that occurred.</para>
				</parameter>
			</syntax>
		</managerEventInstance>
	</managerEvent>
	<managerEvent language="en_US" name="DAHDIChannel">
		<managerEventInstance class="EVENT_FLAG_CALL">
			<since>
				<version>12.0.0</version>
			</since>
			<synopsis>Raised when a DAHDI channel is created or an underlying technology is associated with a DAHDI channel.</synopsis>
			<syntax>
				<channel_snapshot/>
				<parameter name="DAHDIGroup">
					<para>The DAHDI logical group associated with this channel.</para>
				</parameter>
				<parameter name="DAHDISpan">
					<para>The DAHDI span associated with this channel.</para>
				</parameter>
				<parameter name="DAHDIChannel">
					<para>The DAHDI channel associated with this channel.</para>
				</parameter>
			</syntax>
		</managerEventInstance>
	</managerEvent>
 ***/

#define SMDI_MD_WAIT_TIMEOUT 1500 /* 1.5 seconds */

static const char * const lbostr[] = {
"0 db (CSU)/0-133 feet (DSX-1)",
"133-266 feet (DSX-1)",
"266-399 feet (DSX-1)",
"399-533 feet (DSX-1)",
"533-655 feet (DSX-1)",
"-7.5db (CSU)",
"-15db (CSU)",
"-22.5db (CSU)"
};

/*! Global jitterbuffer configuration - by default, jb is disabled
 *  \note Values shown here match the defaults shown in chan_dahdi.conf.sample */
static struct ast_jb_conf default_jbconf =
{
	.flags = 0,
	.max_size = 200,
	.resync_threshold = 1000,
	.impl = "fixed",
	.target_extra = 40,
};
static struct ast_jb_conf global_jbconf;

/*!
 * \note Define ZHONE_HACK to cause us to go off hook and then back on hook when
 * the user hangs up to reset the state machine so ring works properly.
 * This is used to be able to support kewlstart by putting the zhone in
 * groundstart mode since their forward disconnect supervision is entirely
 * broken even though their documentation says it isn't and their support
 * is entirely unwilling to provide any assistance with their channel banks
 * even though their web site says they support their products for life.
 */
/* #define ZHONE_HACK */

/*! \brief Typically, how many rings before we should send Caller*ID */
#define DEFAULT_CIDRINGS 1

#define AST_LAW(p) (((p)->law == DAHDI_LAW_ALAW) ? ast_format_alaw : ast_format_ulaw)


/*! \brief Signaling types that need to use MF detection should be placed in this macro */
#define NEED_MFDETECT(p) (((p)->sig == SIG_FEATDMF) || ((p)->sig == SIG_FEATDMF_TA) || ((p)->sig == SIG_E911) || ((p)->sig == SIG_FGC_CAMA) || ((p)->sig == SIG_FGC_CAMAMF) || ((p)->sig == SIG_FEATB))

static const char tdesc[] = "DAHDI Telephony"
#if defined(HAVE_PRI) || defined(HAVE_SS7) || defined(HAVE_OPENR2)
	" w/"
	#if defined(HAVE_PRI)
		"PRI"
	#endif	/* defined(HAVE_PRI) */
	#if defined(HAVE_SS7)
		#if defined(HAVE_PRI)
		" & "
		#endif	/* defined(HAVE_PRI) */
		"SS7"
	#endif	/* defined(HAVE_SS7) */
	#if defined(HAVE_OPENR2)
		#if defined(HAVE_PRI) || defined(HAVE_SS7)
		" & "
		#endif	/* defined(HAVE_PRI) || defined(HAVE_SS7) */
		"MFC/R2"
	#endif	/* defined(HAVE_OPENR2) */
#endif	/* defined(HAVE_PRI) || defined(HAVE_SS7) || defined(HAVE_OPENR2) */
;

static const char config[] = "chan_dahdi.conf";

#ifdef LOTS_OF_SPANS
#define NUM_SPANS	DAHDI_MAX_SPANS
#else
#define NUM_SPANS 		32
#endif

#define CHAN_PSEUDO	-2

#define CALLPROGRESS_PROGRESS		1
#define CALLPROGRESS_FAX_OUTGOING	2
#define CALLPROGRESS_FAX_INCOMING	4
#define CALLPROGRESS_FAX		(CALLPROGRESS_FAX_INCOMING | CALLPROGRESS_FAX_OUTGOING)

#define NUM_CADENCE_MAX 25
static int num_cadence = 4;
static int user_has_defined_cadences = 0;

static int has_pseudo;

static struct dahdi_ring_cadence cadences[NUM_CADENCE_MAX] = {
	{ { 125, 125, 2000, 4000 } },			/*!< Quick chirp followed by normal ring */
	{ { 250, 250, 500, 1000, 250, 250, 500, 4000 } }, /*!< British style ring */
	{ { 125, 125, 125, 125, 125, 4000 } },	/*!< Three short bursts */
	{ { 1000, 500, 2500, 5000 } },	/*!< Long ring */
};

/*! \brief cidrings says in which pause to transmit the cid information, where the first pause
 * is 1, the second pause is 2 and so on.
 */

static int cidrings[NUM_CADENCE_MAX] = {
	2,										/*!< Right after first long ring */
	4,										/*!< Right after long part */
	3,										/*!< After third chirp */
	2,										/*!< Second spell */
};

/* ETSI EN300 659-1 specifies the ring pulse between 200 and 300 mS */
static struct dahdi_ring_cadence AS_RP_cadence = {{250, 10000}};

#define ISTRUNK(p) ((p->sig == SIG_FXSLS) || (p->sig == SIG_FXSKS) || \
			(p->sig == SIG_FXSGS) || (p->sig == SIG_PRI))

#define CANBUSYDETECT(p) (ISTRUNK(p) || (p->sig & (SIG_EM | SIG_EM_E1 | SIG_SF)) /* || (p->sig & __DAHDI_SIG_FXO) */)
#define CANPROGRESSDETECT(p) (ISTRUNK(p) || (p->sig & (SIG_EM | SIG_EM_E1 | SIG_SF)) /* || (p->sig & __DAHDI_SIG_FXO) */)

static char defaultcic[64] = "";
static char defaultozz[64] = "";

/*! Run this script when the MWI state changes on an FXO line, if mwimonitor is enabled */
static char mwimonitornotify[PATH_MAX] = "";
#ifndef HAVE_DAHDI_LINEREVERSE_VMWI
static int  mwisend_rpas = 0;
#endif

static char progzone[10] = "";

static int usedistinctiveringdetection = 0;
static int distinctiveringaftercid = 0;

static int numbufs = 4;

static int mwilevel = 512;
static int dtmfcid_level = 256;

#define REPORT_CHANNEL_ALARMS 1
#define REPORT_SPAN_ALARMS    2
static int report_alarms = REPORT_CHANNEL_ALARMS;

#ifdef HAVE_PRI
static int pridebugfd = -1;
static char pridebugfilename[1024] = "";
#endif

/*! \brief Protect the interface list (of dahdi_pvt's) */
AST_MUTEX_DEFINE_STATIC(iflock);


static int ifcount = 0;

#ifdef HAVE_PRI
AST_MUTEX_DEFINE_STATIC(pridebugfdlock);
#endif

/*! \brief Protect the monitoring thread, so only one process can kill or start it, and not
   when it's doing something critical. */
AST_MUTEX_DEFINE_STATIC(monlock);

/*! \brief This is the thread for the monitor which checks for input on the channels
   which are not currently in use. */
static pthread_t monitor_thread = AST_PTHREADT_NULL;
static ast_cond_t ss_thread_complete;
AST_MUTEX_DEFINE_STATIC(ss_thread_lock);
AST_MUTEX_DEFINE_STATIC(restart_lock);
static int ss_thread_count = 0;
static int num_restart_pending = 0;

static int restart_monitor(void);

static int dahdi_sendtext(struct ast_channel *c, const char *text);

/*! \brief Avoid the silly dahdi_getevent which ignores a bunch of events */
static inline int dahdi_get_event(int fd)
{
	int j;
	if (ioctl(fd, DAHDI_GETEVENT, &j) == -1)
		return -1;
	return j;
}

/*! \brief Avoid the silly dahdi_waitevent which ignores a bunch of events */
static inline int dahdi_wait_event(int fd)
{
	int i, j = 0;
	i = DAHDI_IOMUX_SIGEVENT;
	if (ioctl(fd, DAHDI_IOMUX, &i) == -1)
		return -1;
	if (ioctl(fd, DAHDI_GETEVENT, &j) == -1)
		return -1;
	return j;
}

/*! Chunk size to read -- we use 20ms chunks to make things happy. */
#define READ_SIZE 160

#define MASK_AVAIL		(1 << 0)	/*!< Channel available for PRI use */
#define MASK_INUSE		(1 << 1)	/*!< Channel currently in use */

#define CALLWAITING_SILENT_SAMPLES		((300 * 8) / READ_SIZE) /*!< 300 ms */
#define CALLWAITING_REPEAT_SAMPLES		((10000 * 8) / READ_SIZE) /*!< 10,000 ms */
#define CALLWAITING_SUPPRESS_SAMPLES	((100 * 8) / READ_SIZE) /*!< 100 ms */
#define CIDCW_EXPIRE_SAMPLES			((500 * 8) / READ_SIZE) /*!< 500 ms */
#define MIN_MS_SINCE_FLASH				((2000) )	/*!< 2000 ms */
#define DEFAULT_RINGT 					((8000 * 8) / READ_SIZE) /*!< 8,000 ms */
#define DEFAULT_DIALTONE_DETECT_TIMEOUT ((10000 * 8) / READ_SIZE) /*!< 10,000 ms */

/*!
 * \brief Configured ring timeout base.
 * \note Value computed from "ringtimeout" read in from chan_dahdi.conf if it exists.
 */
static int ringt_base = DEFAULT_RINGT;

#if defined(HAVE_SS7)

struct dahdi_ss7 {
	struct sig_ss7_linkset ss7;
};

static struct dahdi_ss7 linksets[NUM_SPANS];

static int cur_ss7type = -1;
static int cur_slc = -1;
static int cur_linkset = -1;
static int cur_pointcode = -1;
static int cur_cicbeginswith = -1;
static int cur_adjpointcode = -1;
static int cur_networkindicator = -1;
static int cur_defaultdpc = -1;
#endif	/* defined(HAVE_SS7) */

#ifdef HAVE_OPENR2
struct dahdi_mfcr2_conf {
	openr2_variant_t variant;
	int mfback_timeout;
	int metering_pulse_timeout;
	int max_ani;
	int max_dnis;
#if defined(OR2_LIB_INTERFACE) && OR2_LIB_INTERFACE > 2
	int dtmf_time_on;
	int dtmf_time_off;
#endif
#if defined(OR2_LIB_INTERFACE) && OR2_LIB_INTERFACE > 3
	int dtmf_end_timeout;
#endif
	signed int get_ani_first:2;
#if defined(OR2_LIB_INTERFACE) && OR2_LIB_INTERFACE > 1
	signed int skip_category_request:2;
#endif
	unsigned int call_files:1;
	unsigned int allow_collect_calls:1;
	unsigned int charge_calls:1;
	unsigned int accept_on_offer:1;
	unsigned int forced_release:1;
	unsigned int double_answer:1;
	signed int immediate_accept:2;
#if defined(OR2_LIB_INTERFACE) && OR2_LIB_INTERFACE > 2
	signed int dtmf_dialing:2;
	signed int dtmf_detection:2;
#endif
	char logdir[OR2_MAX_PATH];
	char r2proto_file[OR2_MAX_PATH];
	openr2_log_level_t loglevel;
	openr2_calling_party_category_t category;
};

/* MFC-R2 pseudo-link structure */
struct dahdi_mfcr2 {
	int index;                             /*!< Unique index for CLI */
	pthread_t r2master;		       /*!< Thread of master */
	openr2_context_t *protocol_context;    /*!< OpenR2 context handle */
	struct dahdi_pvt *pvts[SIG_MFCR2_MAX_CHANNELS];     /*!< Member channel pvt structs */
	int numchans;                          /*!< Number of channels in this R2 block */
	int live_chans;                        /*!< Number of unremoved channels in this R2 block */
	int nodev;                             /*!< Link disconnected? */
	struct dahdi_mfcr2_conf conf;          /*!< Configuration used to setup this pseudo-link */
};

struct r2link_entry {
	struct dahdi_mfcr2 mfcr2;
	AST_LIST_ENTRY(r2link_entry) list;
};
static AST_LIST_HEAD_STATIC(r2links, r2link_entry);
static struct r2links nodev_r2links = AST_LIST_HEAD_INIT_VALUE;


/* how many r2links have been malloc'd */
static int r2links_count = 0;

#endif /* HAVE_OPENR2 */

#ifdef HAVE_PRI

struct dahdi_pri {
	int dchannels[SIG_PRI_NUM_DCHANS];		/*!< What channel are the dchannels on */
	int mastertrunkgroup;					/*!< What trunk group is our master */
	int prilogicalspan;						/*!< Logical span number within trunk group */
	struct sig_pri_span pri;
};

static struct dahdi_pri pris[NUM_SPANS];

#if defined(HAVE_PRI_CCSS)
/*! DAHDI PRI CCSS agent and monitor type name. */
static const char dahdi_pri_cc_type[] = "DAHDI/PRI";
#endif	/* defined(HAVE_PRI_CCSS) */

#else
/*! Shut up the compiler */
struct dahdi_pri;
#endif

/* Polarity states */
#define POLARITY_IDLE   0
#define POLARITY_REV    1

const char * const subnames[] = {
	"Real",
	"Callwait",
	"Threeway"
};

static struct dahdi_pvt *iflist = NULL;	/*!< Main interface list start */
static struct dahdi_pvt *ifend = NULL;	/*!< Main interface list end */

#if defined(HAVE_PRI)
struct doomed_pri {
	struct sig_pri_span *pri;
	AST_LIST_ENTRY(doomed_pri) list;
};
static AST_LIST_HEAD_STATIC(doomed_pris, doomed_pri);

static void pri_destroy_span(struct sig_pri_span *pri);

static struct dahdi_parms_pseudo {
	int buf_no;					/*!< Number of buffers */
	int buf_policy;				/*!< Buffer policy */
	int faxbuf_no;              /*!< Number of Fax buffers */
	int faxbuf_policy;          /*!< Fax buffer policy */
} dahdi_pseudo_parms;
#endif	/* defined(HAVE_PRI) */

/*! \brief Channel configuration from chan_dahdi.conf .
 * This struct is used for parsing the [channels] section of chan_dahdi.conf.
 * Generally there is a field here for every possible configuration item.
 *
 * The state of fields is saved along the parsing and whenever a 'channel'
 * statement is reached, the current dahdi_chan_conf is used to configure the
 * channel (struct dahdi_pvt)
 *
 * \see dahdi_chan_init for the default values.
 */
struct dahdi_chan_conf {
	struct dahdi_pvt chan;
#ifdef HAVE_PRI
	struct dahdi_pri pri;
#endif

#if defined(HAVE_SS7)
	struct dahdi_ss7 ss7;
#endif	/* defined(HAVE_SS7) */

#ifdef HAVE_OPENR2
	struct dahdi_mfcr2_conf mfcr2;
#endif
	struct dahdi_params timing;
	int is_sig_auto; /*!< Use channel signalling from DAHDI? */
	/*! Continue configuration even if a channel is not there. */
	int ignore_failed_channels;

	/*!
	 * \brief The serial port to listen for SMDI data on
	 * \note Set from the "smdiport" string read in from chan_dahdi.conf
	 */
	char smdi_port[SMDI_MAX_FILENAME_LEN];

	/*!
	 * \brief Don't create channels below this number
	 * \note by default is 0 (no limit)
	 */
	int wanted_channels_start;

	/*!
	 * \brief Don't create channels above this number (infinity by default)
	 * \note by default is 0 (special value that means "no limit").
	 */
	int wanted_channels_end;
};

/*! returns a new dahdi_chan_conf with default values (by-value) */
static struct dahdi_chan_conf dahdi_chan_conf_default(void)
{
	/* recall that if a field is not included here it is initialized
	 * to 0 or equivalent
	 */
	struct dahdi_chan_conf conf = {
#ifdef HAVE_PRI
		.pri.pri = {
			.nsf = PRI_NSF_NONE,
			.switchtype = PRI_SWITCH_NI2,
			.dialplan = PRI_UNKNOWN + 1,
			.localdialplan = PRI_NATIONAL_ISDN + 1,
			.nodetype = PRI_CPE,
			.qsigchannelmapping = DAHDI_CHAN_MAPPING_PHYSICAL,

#if defined(HAVE_PRI_CCSS)
			.cc_ptmp_recall_mode = 1,/* specificRecall */
			.cc_qsig_signaling_link_req = 1,/* retain */
			.cc_qsig_signaling_link_rsp = 1,/* retain */
#endif	/* defined(HAVE_PRI_CCSS) */

			.minunused = 2,
			.idleext = "",
			.idledial = "",
			.internationalprefix = "",
			.nationalprefix = "",
			.localprefix = "",
			.privateprefix = "",
			.unknownprefix = "",
			.colp_send = SIG_PRI_COLP_UPDATE,
			.resetinterval = -1,
		},
#endif
#if defined(HAVE_SS7)
		.ss7.ss7 = {
			.called_nai = SS7_NAI_NATIONAL,
			.calling_nai = SS7_NAI_NATIONAL,
			.internationalprefix = "",
			.nationalprefix = "",
			.subscriberprefix = "",
			.unknownprefix = "",
			.networkroutedprefix = ""
		},
#endif	/* defined(HAVE_SS7) */
#ifdef HAVE_OPENR2
		.mfcr2 = {
			.variant = OR2_VAR_ITU,
			.mfback_timeout = -1,
			.metering_pulse_timeout = -1,
			.max_ani = 10,
			.max_dnis = 4,
			.get_ani_first = -1,
#if defined(OR2_LIB_INTERFACE) && OR2_LIB_INTERFACE > 1
			.skip_category_request = -1,
#endif
			.call_files = 0,
			.allow_collect_calls = 0,
			.charge_calls = 1,
			.accept_on_offer = 1,
			.forced_release = 0,
			.double_answer = 0,
			.immediate_accept = -1,
#if defined(OR2_LIB_INTERFACE) && OR2_LIB_INTERFACE > 2
			.dtmf_dialing = -1,
			.dtmf_detection = -1,
			.dtmf_time_on = OR2_DEFAULT_DTMF_ON,
			.dtmf_time_off = OR2_DEFAULT_DTMF_OFF,
#endif
#if defined(OR2_LIB_INTERFACE) && OR2_LIB_INTERFACE > 3
			.dtmf_end_timeout = -1,
#endif
			.logdir = "",
			.r2proto_file = "",
			.loglevel = OR2_LOG_ERROR | OR2_LOG_WARNING,
			.category = OR2_CALLING_PARTY_CATEGORY_NATIONAL_SUBSCRIBER
		},
#endif
		.chan = {
			.context = "default",
			.immediatering = 1,
			.cid_num = "",
			.cid_name = "",
			.cid_tag = "",
			.mohinterpret = "default",
			.mohsuggest = "",
			.parkinglot = "",
			.transfertobusy = 1,
			.permdialmode = ANALOG_DIALMODE_BOTH,

			.ani_info_digits = 2,
			.ani_wink_time = 1000,
			.ani_timeout = 10000,

			.cid_signalling = CID_SIG_BELL,
			.cid_start = CID_START_RING,
			.dahditrcallerid = 0,
			.use_callerid = 1,
			.sig = -1,
			.outsigmod = -1,

			.cid_rxgain = +5.0,

			.tonezone = -1,

			.echocancel.head.tap_length = 1,

			.busycount = 3,

			.accountcode = "",

			.mailbox = "",

#ifdef HAVE_DAHDI_LINEREVERSE_VMWI
			.mwisend_fsk = 1,
#endif
			.polarityonanswerdelay = 600,

			.sendcalleridafter = DEFAULT_CIDRINGS,

			.buf_policy = DAHDI_POLICY_IMMEDIATE,
			.buf_no = numbufs,
			.usefaxbuffers = 0,
			.cc_params = ast_cc_config_params_init(),
			.firstdigit_timeout = ANALOG_FIRST_DIGIT_TIMEOUT,
			.interdigit_timeout = ANALOG_INTER_DIGIT_TIMEOUT,
			.matchdigit_timeout = ANALOG_MATCH_DIGIT_TIMEOUT,
		},
		.timing = {
			.prewinktime = -1,
			.preflashtime = -1,
			.winktime = -1,
			.flashtime = -1,
			.starttime = -1,
			.rxwinktime = -1,
			.rxflashtime = -1,
			.debouncetime = -1
		},
		.is_sig_auto = 1,
		.ignore_failed_channels = 1,
		.smdi_port = "/dev/ttyS0",
	};

	return conf;
}


static struct ast_channel *dahdi_request(const char *type, struct ast_format_cap *cap,
	const struct ast_assigned_ids *assignedids, const struct ast_channel *requestor,
	const char *data, int *cause);
static int dahdi_digit_begin(struct ast_channel *ast, char digit);
static int dahdi_digit_end(struct ast_channel *ast, char digit, unsigned int duration);
static int dahdi_sendtext(struct ast_channel *c, const char *text);
static int dahdi_call(struct ast_channel *ast, const char *rdest, int timeout);
static int dahdi_hangup(struct ast_channel *ast);
static int dahdi_answer(struct ast_channel *ast);
static struct ast_frame *dahdi_read(struct ast_channel *ast);
static int dahdi_write(struct ast_channel *ast, struct ast_frame *frame);
static struct ast_frame *dahdi_exception(struct ast_channel *ast);
static int dahdi_indicate(struct ast_channel *chan, int condition, const void *data, size_t datalen);
static int dahdi_fixup(struct ast_channel *oldchan, struct ast_channel *newchan);
static int dahdi_setoption(struct ast_channel *chan, int option, void *data, int datalen);
static int dahdi_queryoption(struct ast_channel *chan, int option, void *data, int *datalen);
static int dahdi_func_read(struct ast_channel *chan, const char *function, char *data, char *buf, size_t len);
static int dahdi_func_write(struct ast_channel *chan, const char *function, char *data, const char *value);
static int dahdi_devicestate(const char *data);
static int dahdi_cc_callback(struct ast_channel *inbound, const char *dest, ast_cc_callback_fn callback);

static struct ast_channel_tech dahdi_tech = {
	.type = "DAHDI",
	.description = tdesc,
	.requester = dahdi_request,
	.send_digit_begin = dahdi_digit_begin,
	.send_digit_end = dahdi_digit_end,
	.send_text = dahdi_sendtext,
	.call = dahdi_call,
	.hangup = dahdi_hangup,
	.answer = dahdi_answer,
	.read = dahdi_read,
	.write = dahdi_write,
	.exception = dahdi_exception,
	.indicate = dahdi_indicate,
	.fixup = dahdi_fixup,
	.setoption = dahdi_setoption,
	.queryoption = dahdi_queryoption,
	.func_channel_read = dahdi_func_read,
	.func_channel_write = dahdi_func_write,
	.devicestate = dahdi_devicestate,
	.cc_callback = dahdi_cc_callback,
};

#define GET_CHANNEL(p) ((p)->channel)

static enum analog_sigtype dahdisig_to_analogsig(int sig)
{
	switch (sig) {
	case SIG_FXOLS:
		return ANALOG_SIG_FXOLS;
	case SIG_FXOGS:
		return ANALOG_SIG_FXOGS;
	case SIG_FXOKS:
		return ANALOG_SIG_FXOKS;
	case SIG_FXSLS:
		return ANALOG_SIG_FXSLS;
	case SIG_FXSGS:
		return ANALOG_SIG_FXSGS;
	case SIG_FXSKS:
		return ANALOG_SIG_FXSKS;
	case SIG_EMWINK:
		return ANALOG_SIG_EMWINK;
	case SIG_EM:
		return ANALOG_SIG_EM;
	case SIG_EM_E1:
		return ANALOG_SIG_EM_E1;
	case SIG_FEATD:
		return ANALOG_SIG_FEATD;
	case SIG_FEATDMF:
		return ANALOG_SIG_FEATDMF;
	case SIG_E911:
		return SIG_E911;
	case SIG_FGC_CAMA:
		return ANALOG_SIG_FGC_CAMA;
	case SIG_FGC_CAMAMF:
		return ANALOG_SIG_FGC_CAMAMF;
	case SIG_FEATB:
		return ANALOG_SIG_FEATB;
	case SIG_SFWINK:
		return ANALOG_SIG_SFWINK;
	case SIG_SF:
		return ANALOG_SIG_SF;
	case SIG_SF_FEATD:
		return ANALOG_SIG_SF_FEATD;
	case SIG_SF_FEATDMF:
		return ANALOG_SIG_SF_FEATDMF;
	case SIG_FEATDMF_TA:
		return ANALOG_SIG_FEATDMF_TA;
	case SIG_SF_FEATB:
		return ANALOG_SIG_FEATB;
	default:
		return -1;
	}
}


static int analog_tone_to_dahditone(enum analog_tone tone)
{
	switch (tone) {
	case ANALOG_TONE_RINGTONE:
		return DAHDI_TONE_RINGTONE;
	case ANALOG_TONE_STUTTER:
		return DAHDI_TONE_STUTTER;
	case ANALOG_TONE_CONGESTION:
		return DAHDI_TONE_CONGESTION;
	case ANALOG_TONE_DIALTONE:
		return DAHDI_TONE_DIALTONE;
	case ANALOG_TONE_DIALRECALL:
		return DAHDI_TONE_DIALRECALL;
	case ANALOG_TONE_INFO:
		return DAHDI_TONE_INFO;
	default:
		return -1;
	}
}

static int analogsub_to_dahdisub(enum analog_sub analogsub)
{
	int index;

	switch (analogsub) {
	case ANALOG_SUB_REAL:
		index = SUB_REAL;
		break;
	case ANALOG_SUB_CALLWAIT:
		index = SUB_CALLWAIT;
		break;
	case ANALOG_SUB_THREEWAY:
		index = SUB_THREEWAY;
		break;
	default:
		ast_log(LOG_ERROR, "Unidentified sub!\n");
		index = SUB_REAL;
	}

	return index;
}

/*!
 * \internal
 * \brief release all members on the doomed pris list
 * \since 13.0
 *
 * Called periodically by the monitor threads to release spans marked for
 * removal.
 */
static void release_doomed_pris(void)
{
#ifdef HAVE_PRI
	struct doomed_pri *entry;

	AST_LIST_LOCK(&doomed_pris);
	while ((entry = AST_LIST_REMOVE_HEAD(&doomed_pris, list))) {
		/* The span destruction must be done with this lock not held */
		AST_LIST_UNLOCK(&doomed_pris);
		ast_debug(4, "Destroying span %d from doomed queue.\n",
				entry->pri->span);
		pri_destroy_span(entry->pri);
		ast_free(entry);
		AST_LIST_LOCK(&doomed_pris);
	}
	AST_LIST_UNLOCK(&doomed_pris);
#endif
}

#ifdef HAVE_PRI
/*!
 * \brief Queue a span for destruction
 * \since 13.0
 *
 * \param pri the span to destroy
 *
 * Add a span to the list of spans to be destroyed later on
 * by the monitor thread. Allows destroying a span while holding its
 * lock.
 */
static void pri_queue_for_destruction(struct sig_pri_span *pri)
{
	struct doomed_pri *entry;

	AST_LIST_LOCK(&doomed_pris);
	AST_LIST_TRAVERSE(&doomed_pris, entry, list) {
		if (entry->pri == pri) {
			AST_LIST_UNLOCK(&doomed_pris);
			return;
		}
	}
	entry = ast_calloc(sizeof(struct doomed_pri), 1);
	if (!entry) {
		/* Nothing useful to do here. Panic? */
		ast_log(LOG_WARNING, "Failed allocating memory for a doomed_pri.\n");
		AST_LIST_UNLOCK(&doomed_pris);
		return;
	}
	entry->pri = pri;
	ast_debug(4, "Queue span %d for destruction.\n", pri->span);
	AST_LIST_INSERT_TAIL(&doomed_pris, entry, list);
	AST_LIST_UNLOCK(&doomed_pris);
}
#endif

/*!
 * \internal
 * \brief Send a dial string to DAHDI.
 * \since 12.0.0
 *
 * \param pvt DAHDI private pointer
 * \param operation DAHDI dial operation to do to string
 * \param dial_str Dial string to send
 *
 * \retval 0 on success.
 * \retval non-zero on error.
 */
static int dahdi_dial_str(struct dahdi_pvt *pvt, int operation, const char *dial_str)
{
	int res;
	int offset;
	const char *pos;
	struct dahdi_dialoperation zo = {
		.op = operation,
	};

	/* Convert the W's to ww. */
	pos = dial_str;
	for (offset = 0; offset < sizeof(zo.dialstr) - 1; ++offset) {
		if (!*pos) {
			break;
		}
		if (*pos == 'W') {
			/* Convert 'W' to "ww" */
			++pos;
			if (offset >= sizeof(zo.dialstr) - 3) {
				/* No room to expand */
				break;
			}
			zo.dialstr[offset] = 'w';
			++offset;
			zo.dialstr[offset] = 'w';
			continue;
		}
		zo.dialstr[offset] = *pos++;
	}
	/* The zo initialization has already terminated the dialstr. */

	ast_debug(1, "Channel %d: Dial str '%s' expanded to '%s' sent to DAHDI_DIAL.\n",
		pvt->channel, dial_str, zo.dialstr);
	res = ioctl(pvt->subs[SUB_REAL].dfd, DAHDI_DIAL, &zo);
	if (res) {
		ast_log(LOG_WARNING, "Channel %d: Couldn't dial '%s': %s\n",
			pvt->channel, dial_str, strerror(errno));
	}

	return res;
}

static enum analog_event dahdievent_to_analogevent(int event);
static int bump_gains(struct dahdi_pvt *p);
static int dahdi_setlinear(int dfd, int linear);

static int my_start_cid_detect(void *pvt, int cid_signalling)
{
	struct dahdi_pvt *p = pvt;
	int index = SUB_REAL;
	p->cs = callerid_new(cid_signalling);
	if (!p->cs) {
		ast_log(LOG_ERROR, "Unable to alloc callerid\n");
		return -1;
	}
	bump_gains(p);
	dahdi_setlinear(p->subs[index].dfd, 0);

	return 0;
}

static int restore_gains(struct dahdi_pvt *p);

static int my_stop_cid_detect(void *pvt)
{
	struct dahdi_pvt *p = pvt;
	int index = SUB_REAL;

	if (p->cs) {
		callerid_free(p->cs);
	}

	/* Restore linear mode after Caller*ID processing */
	dahdi_setlinear(p->subs[index].dfd, p->subs[index].linear);
	restore_gains(p);

	return 0;
}

static int my_get_callerid(void *pvt, char *namebuf, char *numbuf, enum analog_event *ev, size_t timeout)
{
	struct dahdi_pvt *p = pvt;
	struct analog_pvt *analog_p = p->sig_pvt;
	struct pollfd poller;
	char *name, *num;
	int index = SUB_REAL;
	int res;
	unsigned char buf[256];
	int flags;
	int redirecting;

	poller.fd = p->subs[SUB_REAL].dfd;
	poller.events = POLLPRI | POLLIN;
	poller.revents = 0;

	res = poll(&poller, 1, timeout);

	if (poller.revents & POLLPRI) {
		*ev = dahdievent_to_analogevent(dahdi_get_event(p->subs[SUB_REAL].dfd));
		return 1;
	}

	if (poller.revents & POLLIN) {
		/*** NOTES ***/
		/* Change API: remove cid_signalling from get_callerid, add a new start_cid_detect and stop_cid_detect function
		 * to enable slin mode and allocate cid detector. get_callerid should be able to be called any number of times until
		 * either a timeout occurs or CID is detected (returns 0). returning 1 should be event received, and -1 should be
		 * a failure and die, and returning 2 means no event was received. */
		res = read(p->subs[index].dfd, buf, sizeof(buf));
		if (res < 0) {
			ast_log(LOG_WARNING, "read returned error: %s\n", strerror(errno));
			return -1;
		}

		if (analog_p->ringt > 0) {
			if (!(--analog_p->ringt)) {
				/* only return if we timeout from a ring event */
				return -1;
			}
		}

		if (p->cid_signalling == CID_SIG_V23_JP) {
			res = callerid_feed_jp(p->cs, buf, res, AST_LAW(p));
		} else {
			res = callerid_feed(p->cs, buf, res, AST_LAW(p));
		}
		if (res < 0) {
			/*
			 * The previous diagnostic message output likely
			 * explains why it failed.
			 */
			ast_log(LOG_WARNING, "Failed to decode CallerID\n");
			return -1;
		}

		if (res == 1) {
			struct ast_channel *chan = analog_p->ss_astchan;
			callerid_get_with_redirecting(p->cs, &name, &num, &flags, &redirecting);
			if (name)
				ast_copy_string(namebuf, name, ANALOG_MAX_CID);
			if (num)
				ast_copy_string(numbuf, num, ANALOG_MAX_CID);

			if (flags & (CID_PRIVATE_NUMBER | CID_UNKNOWN_NUMBER)) {
				/* If we got a presentation, we must set it on the channel */
				struct ast_party_caller caller;

				ast_party_caller_set_init(&caller, ast_channel_caller(chan));
				caller.id.name.presentation = caller.id.number.presentation = (flags & CID_PRIVATE_NUMBER) ?
					AST_PRES_RESTRICTED | AST_PRES_USER_NUMBER_UNSCREENED : AST_PRES_UNAVAILABLE | AST_PRES_USER_NUMBER_UNSCREENED;
				ast_party_caller_set(ast_channel_caller(chan), &caller, NULL);
				ast_party_caller_free(&caller);
			}
			if (redirecting) {
				/* There is a redirecting reason available in the Caller*ID received.
				 * No idea what the redirecting number is, since the Caller*ID protocol
				 * has no parameter for that, but at least we know WHY it was redirected. */
				ast_channel_redirecting(chan)->reason.code = redirecting;
			}

			if (flags & CID_QUALIFIER) {
				/* This is the inverse of how the qualifier is set in sig_analog */
				pbx_builtin_setvar_helper(chan, "CALL_QUALIFIER", "1");
			}

			ast_debug(1, "CallerID number: %s, name: %s, flags=%d, redirecting=%s\n", num, name, flags, ast_redirecting_reason_name(&ast_channel_redirecting(chan)->reason));
			return 0;
		}
	}

	*ev = ANALOG_EVENT_NONE;
	return 2;
}

static const char *event2str(int event);

static int my_distinctive_ring(struct ast_channel *chan, void *pvt, int idx, int *ringdata)
{
	unsigned char buf[256];
	int distMatches;
	int curRingData[RING_PATTERNS];
	int receivedRingT;
	int counter1;
	int counter;
	int i;
	int res;
	int checkaftercid = 0;
	const char *matched_context;
	struct dahdi_pvt *p = pvt;
	struct analog_pvt *analog_p = p->sig_pvt;

	if (ringdata == NULL) {
		ringdata = curRingData;
	} else {
		checkaftercid = 1;
	}

	/* We must have a ring by now so lets try to listen for distinctive ringing */
	if ((checkaftercid && distinctiveringaftercid) || !checkaftercid) {
		/* Clear the current ring data array so we don't have old data in it. */
		for (receivedRingT = 0; receivedRingT < RING_PATTERNS; receivedRingT++)
			ringdata[receivedRingT] = 0;
		receivedRingT = 0;

		if (checkaftercid && distinctiveringaftercid) {
			ast_verb(3, "Detecting post-CID distinctive ring\n");
		}

		for (;;) {
			i = DAHDI_IOMUX_READ | DAHDI_IOMUX_SIGEVENT;
			res = ioctl(p->subs[idx].dfd, DAHDI_IOMUX, &i);
			if (res) {
				ast_log(LOG_WARNING, "I/O MUX failed: %s\n", strerror(errno));
				ast_hangup(chan);
				return 1;
			}
			if (i & DAHDI_IOMUX_SIGEVENT) {
				res = dahdi_get_event(p->subs[idx].dfd);
				ast_debug(3, "Got event %d (%s)...\n", res, event2str(res));
				if (res == DAHDI_EVENT_NOALARM) {
					p->inalarm = 0;
					analog_p->inalarm = 0;
				} else if (res == DAHDI_EVENT_RINGOFFHOOK) {
					/* Let us detect distinctive ring */
					ringdata[receivedRingT] = analog_p->ringt;

					if (analog_p->ringt < analog_p->ringt_base / 2) {
						break;
					}
					/* Increment the ringT counter so we can match it against
					   values in chan_dahdi.conf for distinctive ring */
					if (++receivedRingT == RING_PATTERNS) {
						break;
					}
				}
			} else if (i & DAHDI_IOMUX_READ) {
				res = read(p->subs[idx].dfd, buf, sizeof(buf));
				if (res < 0) {
					if (errno != ELAST) {
						ast_log(LOG_WARNING, "read returned error: %s\n", strerror(errno));
						ast_hangup(chan);
						return 1;
					}
					break;
				}
				if (analog_p->ringt > 0) {
					if (!(--analog_p->ringt)) {
						break;
					}
				}
			}
		}
	}

	/* Check to see if the rings we received match any of the ones in chan_dahdi.conf for this channel */
	ast_verb(3, "Detected ring pattern: %d,%d,%d\n", ringdata[0], ringdata[1], ringdata[2]);
	matched_context = p->defcontext;
	for (counter = 0; counter < 3; counter++) {
		int range = p->drings.ringnum[counter].range;

		distMatches = 0;
		ast_verb(3, "Checking %d,%d,%d with +/- %d range\n",
			p->drings.ringnum[counter].ring[0],
			p->drings.ringnum[counter].ring[1],
			p->drings.ringnum[counter].ring[2],
			range);
		for (counter1 = 0; counter1 < 3; counter1++) {
			int ring = p->drings.ringnum[counter].ring[counter1];

			if (ring == -1) {
				ast_verb(3, "Pattern ignore (-1) detected, so matching pattern %d regardless.\n",
					ringdata[counter1]);
				distMatches++;
			} else if (ring - range <= ringdata[counter1] && ringdata[counter1] <= ring + range) {
				ast_verb(3, "Ring pattern %d is in range: %d to %d\n",
					ringdata[counter1], ring - range, ring + range);
				distMatches++;
			} else {
				/* The current dring pattern cannot match. */
				break;
			}
		}

		if (distMatches == 3) {
			/* The ring matches, set the context to whatever is for distinctive ring.. */
			matched_context = S_OR(p->drings.ringContext[counter].contextData, p->defcontext);
			ast_verb(3, "Matched Distinctive Ring context %s\n", matched_context);
			break;
		}
	}

	/* Set selected distinctive ring context if not already set. */
	if (strcmp(p->context, matched_context) != 0) {
		ast_copy_string(p->context, matched_context, sizeof(p->context));
		ast_channel_context_set(chan, matched_context);
	}

	return 0;
}

static int my_stop_callwait(void *pvt)
{
	struct dahdi_pvt *p = pvt;
	p->callwaitingrepeat = 0;
	p->cidcwexpire = 0;
	p->cid_suppress_expire = 0;

	return 0;
}

static int send_callerid(struct dahdi_pvt *p);
static int save_conference(struct dahdi_pvt *p);
static int restore_conference(struct dahdi_pvt *p);

static int my_callwait(void *pvt)
{
	struct dahdi_pvt *p = pvt;

	p->callwaitingrepeat = CALLWAITING_REPEAT_SAMPLES;
	if (p->cidspill) {
		ast_log(LOG_WARNING, "Spill already exists?!?\n");
		ast_free(p->cidspill);
	}

	/*
	 * SAS: Subscriber Alert Signal, 440Hz for 300ms
	 * CAS: CPE Alert Signal, 2130Hz * 2750Hz sine waves
	 */
	if (!(p->cidspill = ast_malloc(2400 /* SAS */ + 680 /* CAS */ + READ_SIZE * 4)))
		return -1;
	save_conference(p);
	/* Silence */
	memset(p->cidspill, 0x7f, 2400 + 600 + READ_SIZE * 4);
	if (!p->callwaitrings && p->callwaitingcallerid) {
		ast_gen_cas(p->cidspill, 1, 2400 + 680, AST_LAW(p));
		p->callwaitcas = 1;
		p->cidlen = 2400 + 680 + READ_SIZE * 4;
	} else {
		ast_gen_cas(p->cidspill, 1, 2400, AST_LAW(p));
		p->callwaitcas = 0;
		p->cidlen = 2400 + READ_SIZE * 4;
	}
	p->cidpos = 0;
	send_callerid(p);

	return 0;
}

static int my_send_callerid(void *pvt, int cwcid, struct ast_party_caller *caller)
{
	struct dahdi_pvt *p = pvt;
	struct analog_pvt *analog_p = p->sig_pvt;

	ast_debug(2, "Starting cid spill\n");

	if (p->cidspill) {
		ast_log(LOG_WARNING, "cidspill already exists??\n");
		ast_free(p->cidspill);
	}

	if ((p->cidspill = ast_malloc(MAX_CALLERID_SIZE))) {
		int pres = ast_party_id_presentation(&caller->id);
		if (cwcid == 0) {
			/* Some CPE support additional parameters for on-hook Caller*ID,
			 * such as redirecting reason and call qualifier, so send those
			 * if available.
			 * I don't know of any CPE that supports this for Call Waiting (unfortunately),
			 * so don't send those for call waiting as that will just lengthen the CID spill
			 * for no good reason.
			 */
			p->cidlen = ast_callerid_full_generate(p->cidspill,
				caller->id.name.str,
				caller->id.number.str,
				NULL,
				analog_p->redirecting_reason,
				pres,
				analog_p->call_qualifier,
				CID_TYPE_MDMF,
				AST_LAW(p));
		} else {
			ast_verb(3, "CPE supports Call Waiting Caller*ID.  Sending '%s/%s'\n",
				caller->id.name.str, caller->id.number.str);
			p->callwaitcas = 0;
			p->cidcwexpire = 0;
			p->cidlen = ast_callerid_callwaiting_full_generate(p->cidspill,
				caller->id.name.str,
				caller->id.number.str,
				NULL,
				-1,
				pres,
				0,
				AST_LAW(p));
			p->cidlen += READ_SIZE * 4;
		}
		p->cidpos = 0;
		p->cid_suppress_expire = 0;
		send_callerid(p);
	}
	return 0;
}

static int my_dsp_reset_and_flush_digits(void *pvt)
{
	struct dahdi_pvt *p = pvt;
	if (p->dsp)
		ast_dsp_digitreset(p->dsp);

	return 0;
}

static int my_dsp_set_digitmode(void *pvt, enum analog_dsp_digitmode mode)
{
	struct dahdi_pvt *p = pvt;

	if (p->channel == CHAN_PSEUDO)
		ast_log(LOG_ERROR, "You have assumed incorrectly sir!\n");

	if (mode == ANALOG_DIGITMODE_DTMF) {
		/* If we do hardware dtmf, no need for a DSP */
		if (p->hardwaredtmf) {
			if (p->dsp) {
				ast_dsp_free(p->dsp);
				p->dsp = NULL;
			}
			return 0;
		}

		if (!p->dsp) {
			p->dsp = ast_dsp_new();
			if (!p->dsp) {
				ast_log(LOG_ERROR, "Unable to allocate DSP\n");
				return -1;
			}
		}

		ast_dsp_set_digitmode(p->dsp, DSP_DIGITMODE_DTMF | p->dtmfrelax);
	} else if (mode == ANALOG_DIGITMODE_MF) {
		if (!p->dsp) {
			p->dsp = ast_dsp_new();
			if (!p->dsp) {
				ast_log(LOG_ERROR, "Unable to allocate DSP\n");
				return -1;
			}
		}
		ast_dsp_set_digitmode(p->dsp, DSP_DIGITMODE_MF | p->dtmfrelax);
	}
	return 0;
}

static int dahdi_wink(struct dahdi_pvt *p, int index);

static int my_wink(void *pvt, enum analog_sub sub)
{
	struct dahdi_pvt *p = pvt;
	int index = analogsub_to_dahdisub(sub);
	if (index != SUB_REAL) {
		ast_log(LOG_ERROR, "We used a sub other than SUB_REAL (incorrect assumption sir)\n");
	}
	return dahdi_wink(p, index);
}

static void wakeup_sub(struct dahdi_pvt *p, int a);

static int reset_conf(struct dahdi_pvt *p);

static inline int dahdi_confmute(struct dahdi_pvt *p, int muted);

static void my_handle_dtmf(void *pvt, struct ast_channel *ast, enum analog_sub analog_index, struct ast_frame **dest)
{
	struct ast_frame *f = *dest;
	struct dahdi_pvt *p = pvt;
	int idx = analogsub_to_dahdisub(analog_index);

	ast_debug(1, "%s DTMF digit: 0x%02X '%c' on %s\n",
		f->frametype == AST_FRAME_DTMF_BEGIN ? "Begin" : "End",
		(unsigned)f->subclass.integer, f->subclass.integer, ast_channel_name(ast));

	if (f->subclass.integer == 'f') {
		if (f->frametype == AST_FRAME_DTMF_END) {
			/* Fax tone -- Handle and return NULL */
			if ((p->callprogress & CALLPROGRESS_FAX) && !p->faxhandled) {
				/* If faxbuffers are configured, use them for the fax transmission */
				if (p->usefaxbuffers && !p->bufferoverrideinuse) {
					struct dahdi_bufferinfo bi = {
						.txbufpolicy = p->faxbuf_policy,
						.bufsize = p->bufsize,
						.numbufs = p->faxbuf_no
					};
					int res;

					if ((res = ioctl(p->subs[idx].dfd, DAHDI_SET_BUFINFO, &bi)) < 0) {
						ast_log(LOG_WARNING, "Channel '%s' unable to set buffer policy, reason: %s\n", ast_channel_name(ast), strerror(errno));
					} else {
						p->bufferoverrideinuse = 1;
					}
				}
				p->faxhandled = 1;
				if (p->dsp) {
					p->dsp_features &= ~DSP_FEATURE_FAX_DETECT;
					ast_dsp_set_features(p->dsp, p->dsp_features);
					ast_debug(1, "Disabling FAX tone detection on %s after tone received\n", ast_channel_name(ast));
				}
				if (strcmp(ast_channel_exten(ast), "fax")) {
					const char *target_context = ast_channel_context(ast);

					/*
					 * We need to unlock 'ast' here because ast_exists_extension has the
					 * potential to start autoservice on the channel. Such action is prone
					 * to deadlock if the channel is locked.
					 *
					 * ast_async_goto() has its own restriction on not holding the
					 * channel lock.
					 */
					ast_mutex_unlock(&p->lock);
					ast_channel_unlock(ast);
					if (ast_exists_extension(ast, target_context, "fax", 1,
						S_COR(ast_channel_caller(ast)->id.number.valid, ast_channel_caller(ast)->id.number.str, NULL))) {
						ast_verb(3, "Redirecting %s to fax extension\n", ast_channel_name(ast));
						/* Save the DID/DNIS when we transfer the fax call to a "fax" extension */
						pbx_builtin_setvar_helper(ast, "FAXEXTEN", ast_channel_exten(ast));
						if (ast_async_goto(ast, target_context, "fax", 1))
							ast_log(LOG_WARNING, "Failed to async goto '%s' into fax of '%s'\n", ast_channel_name(ast), target_context);
					} else {
						ast_log(LOG_NOTICE, "Fax detected, but no fax extension\n");
					}
					ast_channel_lock(ast);
					ast_mutex_lock(&p->lock);
				} else {
					ast_debug(1, "Already in a fax extension, not redirecting\n");
				}
			} else {
				ast_debug(1, "Fax already handled\n");
			}
			dahdi_confmute(p, 0);
		}
		p->subs[idx].f.frametype = AST_FRAME_NULL;
		p->subs[idx].f.subclass.integer = 0;
		*dest = &p->subs[idx].f;
	}
}

static void my_lock_private(void *pvt)
{
	struct dahdi_pvt *p = pvt;
	ast_mutex_lock(&p->lock);
}

static void my_unlock_private(void *pvt)
{
	struct dahdi_pvt *p = pvt;
	ast_mutex_unlock(&p->lock);
}

static void my_deadlock_avoidance_private(void *pvt)
{
	struct dahdi_pvt *p = pvt;

	DEADLOCK_AVOIDANCE(&p->lock);
}

static struct ast_manager_event_blob *dahdichannel_to_ami(struct stasis_message *msg)
{
	RAII_VAR(struct ast_str *, channel_string, NULL, ast_free);
	struct ast_channel_blob *obj = stasis_message_data(msg);
	struct ast_json *group, *span, *channel;

	channel_string = ast_manager_build_channel_state_string(obj->snapshot);
	if (!channel_string) {
		return NULL;
	}

	group = ast_json_object_get(obj->blob, "group");
	span = ast_json_object_get(obj->blob, "span");
	channel = ast_json_object_get(obj->blob, "channel");

	return ast_manager_event_blob_create(EVENT_FLAG_CALL, "DAHDIChannel",
		"%s"
		"DAHDIGroup: %llu\r\n"
		"DAHDISpan: %u\r\n"
		"DAHDIChannel: %s\r\n",
		ast_str_buffer(channel_string),
		(ast_group_t)ast_json_integer_get(group),
		(unsigned int)ast_json_integer_get(span),
		ast_json_string_get(channel));
}

STASIS_MESSAGE_TYPE_DEFN_LOCAL(dahdichannel_type,
	.to_ami = dahdichannel_to_ami,
	);

/*! \brief Sends a DAHDIChannel channel blob used to produce DAHDIChannel AMI messages */
static void publish_dahdichannel(struct ast_channel *chan, ast_group_t group, int span, const char *dahdi_channel)
{
	RAII_VAR(struct ast_json *, blob, NULL, ast_json_unref);

	ast_assert(dahdi_channel != NULL);

	blob = ast_json_pack("{s: I, s: i, s: s}",
		"group", (ast_json_int_t)group,
		"span", span,
		"channel", dahdi_channel);
	if (!blob) {
		return;
	}

	ast_channel_lock(chan);
	ast_channel_publish_blob(chan, dahdichannel_type(), blob);
	ast_channel_unlock(chan);
}

/*!
 * \internal
 * \brief Post an AMI DAHDI channel association event.
 * \since 1.8
 *
 * \param p DAHDI private pointer
 * \param chan Channel associated with the private pointer
 */
static void dahdi_ami_channel_event(struct dahdi_pvt *p, struct ast_channel *chan)
{
	char ch_name[23];

	if (p->channel < CHAN_PSEUDO) {
		/* No B channel */
		snprintf(ch_name, sizeof(ch_name), "no-media (%d)", p->channel);
	} else if (p->channel == CHAN_PSEUDO) {
		/* Pseudo channel */
		strcpy(ch_name, "pseudo");
	} else {
		/* Real channel */
		snprintf(ch_name, sizeof(ch_name), "%d", p->channel);
	}
	publish_dahdichannel(chan, p->group, p->span, ch_name);
}

#ifdef HAVE_PRI
/*!
 * \internal
 * \brief Post an AMI DAHDI channel association event.
 * \since 1.8
 *
 * \param pvt DAHDI private pointer
 * \param chan Channel associated with the private pointer
 */
static void my_ami_channel_event(void *pvt, struct ast_channel *chan)
{
	struct dahdi_pvt *p = pvt;

	dahdi_ami_channel_event(p, chan);
}
#endif

/* linear_mode = 0 - turn linear mode off, >0 - turn linear mode on
* 	returns the last value of the linear setting
*/
static int my_set_linear_mode(void *pvt, enum analog_sub sub, int linear_mode)
{
	struct dahdi_pvt *p = pvt;
	int oldval;
	int idx = analogsub_to_dahdisub(sub);

	dahdi_setlinear(p->subs[idx].dfd, linear_mode);
	oldval = p->subs[idx].linear;
	p->subs[idx].linear = linear_mode ? 1 : 0;
	return oldval;
}

static void my_set_inthreeway(void *pvt, enum analog_sub sub, int inthreeway)
{
	struct dahdi_pvt *p = pvt;
	int idx = analogsub_to_dahdisub(sub);

	p->subs[idx].inthreeway = inthreeway;
}

static int get_alarms(struct dahdi_pvt *p);
static void handle_alarms(struct dahdi_pvt *p, int alms);
static void my_get_and_handle_alarms(void *pvt)
{
	int res;
	struct dahdi_pvt *p = pvt;

	res = get_alarms(p);
	handle_alarms(p, res);
}

static void *my_get_sigpvt_bridged_channel(struct ast_channel *chan)
{
	RAII_VAR(struct ast_channel *, bridged, ast_channel_bridge_peer(chan), ast_channel_cleanup);

	if (bridged && ast_channel_tech(bridged) == &dahdi_tech) {
		struct dahdi_pvt *p = ast_channel_tech_pvt(bridged);

		if (dahdi_analog_lib_handles(p->sig, p->radio, p->oprmode)) {
			return p->sig_pvt;
		}
	}
	return NULL;
}

static int my_get_sub_fd(void *pvt, enum analog_sub sub)
{
	struct dahdi_pvt *p = pvt;
	int dahdi_sub = analogsub_to_dahdisub(sub);
	return p->subs[dahdi_sub].dfd;
}

static void my_set_cadence(void *pvt, int *cid_rings, struct ast_channel *ast)
{
	struct dahdi_pvt *p = pvt;

	/* Choose proper cadence */
	if ((p->distinctivering > 0) && (p->distinctivering <= num_cadence)) {
		if (ioctl(p->subs[SUB_REAL].dfd, DAHDI_SETCADENCE, &cadences[p->distinctivering - 1]))
			ast_log(LOG_WARNING, "Unable to set distinctive ring cadence %d on '%s': %s\n", p->distinctivering, ast_channel_name(ast), strerror(errno));
		*cid_rings = cidrings[p->distinctivering - 1];
	} else {
		if (p->distinctivering > 0) {
			ast_log(LOG_WARNING, "Cadence %d is not defined, falling back to default ring cadence\n", p->distinctivering);
		}
		if (ioctl(p->subs[SUB_REAL].dfd, DAHDI_SETCADENCE, NULL))
			ast_log(LOG_WARNING, "Unable to reset default ring on '%s': %s\n", ast_channel_name(ast), strerror(errno));
		*cid_rings = p->sendcalleridafter;
	}
}

static void my_set_alarm(void *pvt, int in_alarm)
{
	struct dahdi_pvt *p = pvt;

	p->inalarm = in_alarm;
}

static void my_set_dialing(void *pvt, int is_dialing)
{
	struct dahdi_pvt *p = pvt;

	p->dialing = is_dialing;
}

static void my_set_outgoing(void *pvt, int is_outgoing)
{
	struct dahdi_pvt *p = pvt;

	p->outgoing = is_outgoing;
}

#if defined(HAVE_PRI) || defined(HAVE_SS7)
static void my_set_digital(void *pvt, int is_digital)
{
	struct dahdi_pvt *p = pvt;

	p->digital = is_digital;
}
#endif	/* defined(HAVE_PRI) || defined(HAVE_SS7) */

#if defined(HAVE_SS7)
static void my_set_inservice(void *pvt, int is_inservice)
{
	struct dahdi_pvt *p = pvt;

	p->inservice = is_inservice;
}
#endif	/* defined(HAVE_SS7) */

#if defined(HAVE_SS7)
static void my_set_locallyblocked(void *pvt, int is_blocked)
{
	struct dahdi_pvt *p = pvt;

	p->locallyblocked = is_blocked;
}
#endif	/* defined(HAVE_SS7) */

#if defined(HAVE_SS7)
static void my_set_remotelyblocked(void *pvt, int is_blocked)
{
	struct dahdi_pvt *p = pvt;

	p->remotelyblocked = is_blocked;
}
#endif	/* defined(HAVE_SS7) */

static void my_set_ringtimeout(void *pvt, int ringt)
{
	struct dahdi_pvt *p = pvt;
	p->ringt = ringt;
}

static void my_set_waitingfordt(void *pvt, struct ast_channel *ast)
{
	struct dahdi_pvt *p = pvt;

	/* We reset p->waitfordialtonetemp here, to prevent leaking to future calls,
	 * but we also need to check against this value until we get dialtone
	 * or the timer expires, since waitingfordt is when the timer started,
	 * not when it should expire.
	 *
	 * Critically, we only set p->waitingfordt here if waitfordialtone or waitfordialtonetemp
	 * has already been set, as waitingfordt is what is checked at runtime to determine
	 * if we should be waiting for dial tone. This ensures that if a second call
	 * is initiated concurrently, the first one "consumes" waitfordialtonetemp and resets it,
	 * preventing leaking to other calls while remaining available to check on the first one while dialing.
	 */
	p->waitfordialtoneduration = p->waitfordialtonetemp ? p->waitfordialtonetemp : p->waitfordialtone;
	p->waitfordialtonetemp = 0;

	if (!(p->waitfordialtoneduration && CANPROGRESSDETECT(p))) {
		return;
	}

	/* Because the DSP is allocated when the channel is created,
	 * if we requested waitfordialtone later (in a predial handler),
	 * we need to create it now */
	if (!p->dsp) {
		p->dsp = ast_dsp_new();
		if (!p->dsp) {
			ast_log(LOG_ERROR, "Unable to allocate DSP\n");
			return;
		}
	}
	p->dsp_features |= DSP_FEATURE_WAITDIALTONE;
	ast_dsp_set_features(p->dsp, p->dsp_features);

	ast_debug(1, "Defer dialing for %dms or dialtone\n", p->waitfordialtoneduration);
	gettimeofday(&p->waitingfordt, NULL);
	ast_setstate(ast, AST_STATE_OFFHOOK);
}

static int my_check_waitingfordt(void *pvt)
{
	struct dahdi_pvt *p = pvt;

	if (p->waitingfordt.tv_sec) {
		return 1;
	}

	return 0;
}

static void my_set_confirmanswer(void *pvt, int flag)
{
	struct dahdi_pvt *p = pvt;
	p->confirmanswer = flag;
}

static int my_check_confirmanswer(void *pvt)
{
	struct dahdi_pvt *p = pvt;
	if (p->confirmanswer) {
		return 1;
	}

	return 0;
}

static void my_set_callwaiting(void *pvt, int callwaiting_enable)
{
	struct dahdi_pvt *p = pvt;

	p->callwaiting = callwaiting_enable;
}

static void my_cancel_cidspill(void *pvt)
{
	struct dahdi_pvt *p = pvt;

	ast_free(p->cidspill);
	p->cidspill = NULL;
	restore_conference(p);
}

static int my_confmute(void *pvt, int mute)
{
	struct dahdi_pvt *p = pvt;
	return dahdi_confmute(p, mute);
}

static void my_set_pulsedial(void *pvt, int flag)
{
	struct dahdi_pvt *p = pvt;
	p->pulsedial = flag;
}

static void my_set_new_owner(void *pvt, struct ast_channel *new_owner)
{
	struct dahdi_pvt *p = pvt;

	p->owner = new_owner;
}

static const char *my_get_orig_dialstring(void *pvt)
{
	struct dahdi_pvt *p = pvt;

	return p->dialstring;
}

static void my_increase_ss_count(void)
{
	ast_mutex_lock(&ss_thread_lock);
	ss_thread_count++;
	ast_mutex_unlock(&ss_thread_lock);
}

static void my_decrease_ss_count(void)
{
	ast_mutex_lock(&ss_thread_lock);
	ss_thread_count--;
	ast_cond_signal(&ss_thread_complete);
	ast_mutex_unlock(&ss_thread_lock);
}

static void my_all_subchannels_hungup(void *pvt)
{
	struct dahdi_pvt *p = pvt;
	int res, law;

	p->faxhandled = 0;
	p->didtdd = 0;

	if (p->dsp) {
		ast_dsp_free(p->dsp);
		p->dsp = NULL;
	}

	p->law = p->law_default;
	law = p->law_default;
	res = ioctl(p->subs[SUB_REAL].dfd, DAHDI_SETLAW, &law);
	if (res < 0)
		ast_log(LOG_WARNING, "Unable to set law on channel %d to default: %s\n", p->channel, strerror(errno));

	dahdi_setlinear(p->subs[SUB_REAL].dfd, 0);

#if 1
	{
	int i;
	p->owner = NULL;
	/* Cleanup owners here */
	for (i = 0; i < 3; i++) {
		p->subs[i].owner = NULL;
	}
	}
#endif

	reset_conf(p);
	if (num_restart_pending == 0) {
		restart_monitor();
	}
}

static int conf_del(struct dahdi_pvt *p, struct dahdi_subchannel *c, int index);

static int my_conf_del(void *pvt, enum analog_sub sub)
{
	struct dahdi_pvt *p = pvt;
	int x = analogsub_to_dahdisub(sub);

	return conf_del(p, &p->subs[x], x);
}

static int conf_add(struct dahdi_pvt *p, struct dahdi_subchannel *c, int index, int slavechannel);

static int my_conf_add(void *pvt, enum analog_sub sub)
{
	struct dahdi_pvt *p = pvt;
	int x = analogsub_to_dahdisub(sub);

	return conf_add(p, &p->subs[x], x, 0);
}

static int isslavenative(struct dahdi_pvt *p, struct dahdi_pvt **out);

static int my_complete_conference_update(void *pvt, int needconference)
{
	struct dahdi_pvt *p = pvt;
	int needconf = needconference;
	int x;
	int useslavenative;
	struct dahdi_pvt *slave = NULL;

	useslavenative = isslavenative(p, &slave);

	/* If we have a slave, add him to our conference now. or DAX
	   if this is slave native */
	for (x = 0; x < MAX_SLAVES; x++) {
		if (p->slaves[x]) {
			if (useslavenative)
				conf_add(p, &p->slaves[x]->subs[SUB_REAL], SUB_REAL, GET_CHANNEL(p));
			else {
				conf_add(p, &p->slaves[x]->subs[SUB_REAL], SUB_REAL, 0);
				needconf++;
			}
		}
	}
	/* If we're supposed to be in there, do so now */
	if (p->inconference && !p->subs[SUB_REAL].inthreeway) {
		if (useslavenative)
			conf_add(p, &p->subs[SUB_REAL], SUB_REAL, GET_CHANNEL(slave));
		else {
			conf_add(p, &p->subs[SUB_REAL], SUB_REAL, 0);
			needconf++;
		}
	}
	/* If we have a master, add ourselves to his conference */
	if (p->master) {
		if (isslavenative(p->master, NULL)) {
			conf_add(p->master, &p->subs[SUB_REAL], SUB_REAL, GET_CHANNEL(p->master));
		} else {
			conf_add(p->master, &p->subs[SUB_REAL], SUB_REAL, 0);
		}
	}
	if (!needconf) {
		/* Nobody is left (or should be left) in our conference.
		   Kill it. */
		p->confno = -1;
	}

	return 0;
}

static int check_for_conference(struct dahdi_pvt *p);

static int my_check_for_conference(void *pvt)
{
	struct dahdi_pvt *p = pvt;
	return check_for_conference(p);
}

static void my_swap_subchannels(void *pvt, enum analog_sub a, struct ast_channel *ast_a,  enum analog_sub b, struct ast_channel *ast_b)
{
	struct dahdi_pvt *p = pvt;
	int da, db;
	int tchan;
	int tinthreeway;

	da = analogsub_to_dahdisub(a);
	db = analogsub_to_dahdisub(b);

	tchan = p->subs[da].chan;
	p->subs[da].chan = p->subs[db].chan;
	p->subs[db].chan = tchan;

	tinthreeway = p->subs[da].inthreeway;
	p->subs[da].inthreeway = p->subs[db].inthreeway;
	p->subs[db].inthreeway = tinthreeway;

	p->subs[da].owner = ast_a;
	p->subs[db].owner = ast_b;

	if (ast_a)
		ast_channel_set_fd(ast_a, 0, p->subs[da].dfd);
	if (ast_b)
		ast_channel_set_fd(ast_b, 0, p->subs[db].dfd);

	wakeup_sub(p, a);
	wakeup_sub(p, b);

	return;
}

/*!
 * \internal
 * \brief performs duties of dahdi_new, but also removes and possibly unbinds (if callid_created is 1) before returning
 * \note this variant of dahdi should only be used in conjunction with ast_callid_threadstorage_auto()
 *
 * \param callid_created value returned from ast_callid_threadstorage_auto()
 * \param i, state, startpbx, idx, law, assignedids, requestor, callid
 */
static struct ast_channel *dahdi_new_callid_clean(struct dahdi_pvt *i, int state, int startpbx, int idx, int law, const struct ast_assigned_ids *assignedids, const struct ast_channel *requestor, ast_callid callid, int callid_created);

static struct ast_channel *dahdi_new(struct dahdi_pvt *i, int state, int startpbx, int idx, int law, const struct ast_assigned_ids *assignedids, const struct ast_channel *requestor, ast_callid callid);

static struct ast_channel *my_new_analog_ast_channel(void *pvt, int state, int startpbx, enum analog_sub sub, const struct ast_channel *requestor)
{
	ast_callid callid = 0;
	int callid_created = ast_callid_threadstorage_auto(&callid);
	struct dahdi_pvt *p = pvt;
	int dsub = analogsub_to_dahdisub(sub);

	return dahdi_new_callid_clean(p, state, startpbx, dsub, 0, NULL, requestor, callid, callid_created);
}

#if defined(HAVE_PRI) || defined(HAVE_SS7)
static int dahdi_setlaw(int dfd, int law)
{
	int res;
	res = ioctl(dfd, DAHDI_SETLAW, &law);
	if (res)
		return res;
	return 0;
}
#endif	/* defined(HAVE_PRI) || defined(HAVE_SS7) */

#if defined(HAVE_PRI)
static struct ast_channel *my_new_pri_ast_channel(void *pvt, int state,
	enum sig_pri_law law, char *exten, const struct ast_assigned_ids *assignedids,
	const struct ast_channel *requestor)
{
	struct dahdi_pvt *p = pvt;
	int audio;
	int newlaw = -1;
	ast_callid callid = 0;
	int callid_created = ast_callid_threadstorage_auto(&callid);

	switch (p->sig) {
	case SIG_PRI_LIB_HANDLE_CASES:
		if (((struct sig_pri_chan *) p->sig_pvt)->no_b_channel) {
			/* PRI nobch pseudo channel.  Does not handle ioctl(DAHDI_AUDIOMODE) */
			break;
		}
		/* Fall through */
	default:
		/* Set to audio mode at this point */
		audio = 1;
		if (ioctl(p->subs[SUB_REAL].dfd, DAHDI_AUDIOMODE, &audio) == -1) {
			ast_log(LOG_WARNING, "Unable to set audio mode on channel %d to %d: %s\n",
				p->channel, audio, strerror(errno));
		}
		break;
	}

	if (law != SIG_PRI_DEFLAW) {
		dahdi_setlaw(p->subs[SUB_REAL].dfd, (law == SIG_PRI_ULAW) ? DAHDI_LAW_MULAW : DAHDI_LAW_ALAW);
	}

	ast_copy_string(p->exten, exten, sizeof(p->exten));

	switch (law) {
		case SIG_PRI_DEFLAW:
			newlaw = 0;
			break;
		case SIG_PRI_ALAW:
			newlaw = DAHDI_LAW_ALAW;
			break;
		case SIG_PRI_ULAW:
			newlaw = DAHDI_LAW_MULAW;
			break;
	}

	return dahdi_new_callid_clean(p, state, 0, SUB_REAL, newlaw, assignedids, requestor, callid, callid_created);
}
#endif	/* defined(HAVE_PRI) */

static int set_actual_gain(int fd, float rxgain, float txgain, float rxdrc, float txdrc, int law);

#if defined(HAVE_PRI) || defined(HAVE_SS7)
/*!
 * \internal
 * \brief Open the PRI/SS7 channel media path.
 * \since 1.8
 *
 * \param p Channel private control structure.
 */
static void my_pri_ss7_open_media(void *p)
{
	struct dahdi_pvt *pvt = p;
	int res;
	int dfd;
	int set_val;

	dfd = pvt->subs[SUB_REAL].dfd;

	/* Open the media path. */
	set_val = 1;
	res = ioctl(dfd, DAHDI_AUDIOMODE, &set_val);
	if (res < 0) {
		ast_log(LOG_WARNING, "Unable to enable audio mode on channel %d (%s)\n",
			pvt->channel, strerror(errno));
	}

	/* Set correct companding law for this call. */
	res = dahdi_setlaw(dfd, pvt->law);
	if (res < 0) {
		ast_log(LOG_WARNING, "Unable to set law on channel %d\n", pvt->channel);
	}

	/* Set correct gain for this call. */
	if (pvt->digital) {
		res = set_actual_gain(dfd, 0, 0, pvt->rxdrc, pvt->txdrc, pvt->law);
	} else {
		res = set_actual_gain(dfd, pvt->rxgain, pvt->txgain, pvt->rxdrc, pvt->txdrc,
			pvt->law);
	}
	if (res < 0) {
		ast_log(LOG_WARNING, "Unable to set gains on channel %d\n", pvt->channel);
	}

	if (pvt->dsp_features && pvt->dsp) {
		ast_dsp_set_features(pvt->dsp, pvt->dsp_features);
	}
}
#endif	/* defined(HAVE_PRI) || defined(HAVE_SS7) */

#if defined(HAVE_PRI)
/*!
 * \internal
 * \brief Ask DAHDI to dial the given dial string.
 * \since 1.8.11
 *
 * \param p Channel private control structure.
 * \param dial_string String to pass to DAHDI to dial.
 *
 * \note The channel private lock needs to be held when calling.
 */
static void my_pri_dial_digits(void *p, const char *dial_string)
{
	char dial_str[DAHDI_MAX_DTMF_BUF];
	struct dahdi_pvt *pvt = p;
	int res;

	snprintf(dial_str, sizeof(dial_str), "T%s", dial_string);
	res = dahdi_dial_str(pvt, DAHDI_DIAL_OP_APPEND, dial_str);
	if (!res) {
		pvt->dialing = 1;
	}
}
#endif	/* defined(HAVE_PRI) */

static int unalloc_sub(struct dahdi_pvt *p, int x);

static int my_unallocate_sub(void *pvt, enum analog_sub analogsub)
{
	struct dahdi_pvt *p = pvt;

	return unalloc_sub(p, analogsub_to_dahdisub(analogsub));
}

static int alloc_sub(struct dahdi_pvt *p, int x);

static int my_allocate_sub(void *pvt, enum analog_sub analogsub)
{
	struct dahdi_pvt *p = pvt;

	return alloc_sub(p, analogsub_to_dahdisub(analogsub));
}

static int has_voicemail(struct dahdi_pvt *p);

static int my_has_voicemail(void *pvt)
{
	struct dahdi_pvt *p = pvt;

	return has_voicemail(p);
}

static int my_play_tone(void *pvt, enum analog_sub sub, enum analog_tone tone)
{
	struct dahdi_pvt *p = pvt;
	int index;

	index = analogsub_to_dahdisub(sub);

	return tone_zone_play_tone(p->subs[index].dfd, analog_tone_to_dahditone(tone));
}

static enum analog_event dahdievent_to_analogevent(int event)
{
	enum analog_event res;

	switch (event) {
	case DAHDI_EVENT_ONHOOK:
		res = ANALOG_EVENT_ONHOOK;
		break;
	case DAHDI_EVENT_RINGOFFHOOK:
		res = ANALOG_EVENT_RINGOFFHOOK;
		break;
	case DAHDI_EVENT_WINKFLASH:
		res = ANALOG_EVENT_WINKFLASH;
		break;
	case DAHDI_EVENT_ALARM:
		res = ANALOG_EVENT_ALARM;
		break;
	case DAHDI_EVENT_NOALARM:
		res = ANALOG_EVENT_NOALARM;
		break;
	case DAHDI_EVENT_DIALCOMPLETE:
		res = ANALOG_EVENT_DIALCOMPLETE;
		break;
	case DAHDI_EVENT_RINGERON:
		res = ANALOG_EVENT_RINGERON;
		break;
	case DAHDI_EVENT_RINGEROFF:
		res = ANALOG_EVENT_RINGEROFF;
		break;
	case DAHDI_EVENT_HOOKCOMPLETE:
		res = ANALOG_EVENT_HOOKCOMPLETE;
		break;
	case DAHDI_EVENT_PULSE_START:
		res = ANALOG_EVENT_PULSE_START;
		break;
	case DAHDI_EVENT_POLARITY:
		res = ANALOG_EVENT_POLARITY;
		break;
	case DAHDI_EVENT_RINGBEGIN:
		res = ANALOG_EVENT_RINGBEGIN;
		break;
	case DAHDI_EVENT_EC_DISABLED:
		res = ANALOG_EVENT_EC_DISABLED;
		break;
	case DAHDI_EVENT_REMOVED:
		res = ANALOG_EVENT_REMOVED;
		break;
	case DAHDI_EVENT_NEONMWI_ACTIVE:
		res = ANALOG_EVENT_NEONMWI_ACTIVE;
		break;
	case DAHDI_EVENT_NEONMWI_INACTIVE:
		res = ANALOG_EVENT_NEONMWI_INACTIVE;
		break;
#ifdef HAVE_DAHDI_ECHOCANCEL_FAX_MODE
	case DAHDI_EVENT_TX_CED_DETECTED:
		res = ANALOG_EVENT_TX_CED_DETECTED;
		break;
	case DAHDI_EVENT_RX_CED_DETECTED:
		res = ANALOG_EVENT_RX_CED_DETECTED;
		break;
	case DAHDI_EVENT_EC_NLP_DISABLED:
		res = ANALOG_EVENT_EC_NLP_DISABLED;
		break;
	case DAHDI_EVENT_EC_NLP_ENABLED:
		res = ANALOG_EVENT_EC_NLP_ENABLED;
		break;
#endif
	case DAHDI_EVENT_PULSEDIGIT:
		res = ANALOG_EVENT_PULSEDIGIT;
		break;
	case DAHDI_EVENT_DTMFDOWN:
		res = ANALOG_EVENT_DTMFDOWN;
		break;
	case DAHDI_EVENT_DTMFUP:
		res = ANALOG_EVENT_DTMFUP;
		break;
	default:
		switch(event & 0xFFFF0000) {
		case DAHDI_EVENT_PULSEDIGIT:
		case DAHDI_EVENT_DTMFDOWN:
		case DAHDI_EVENT_DTMFUP:
			/* The event includes a digit number in the low word.
			 * Converting it to a 'enum analog_event' would remove
			 * that information. Thus it is returned as-is.
			 */
			return event;
		}

		res = ANALOG_EVENT_ERROR;
		break;
	}

	return res;
}

static inline int dahdi_wait_event(int fd);

static int my_wait_event(void *pvt)
{
	struct dahdi_pvt *p = pvt;

	return dahdi_wait_event(p->subs[SUB_REAL].dfd);
}

static int my_get_event(void *pvt)
{
	struct dahdi_pvt *p = pvt;
	int res;

	if (p->fake_event) {
		res = p->fake_event;
		p->fake_event = 0;
	} else
		res = dahdi_get_event(p->subs[SUB_REAL].dfd);

	return dahdievent_to_analogevent(res);
}

static int my_is_off_hook(void *pvt)
{
	struct dahdi_pvt *p = pvt;
	int res;
	struct dahdi_params par;

	memset(&par, 0, sizeof(par));

	if (p->subs[SUB_REAL].dfd > -1)
		res = ioctl(p->subs[SUB_REAL].dfd, DAHDI_GET_PARAMS, &par);
	else {
		/* Assume not off hook on CVRS */
		res = 0;
		par.rxisoffhook = 0;
	}
	if (res) {
		ast_log(LOG_WARNING, "Unable to check hook state on channel %d: %s\n", p->channel, strerror(errno));
	}

	if ((p->sig == SIG_FXSKS) || (p->sig == SIG_FXSGS)) {
		/* When "onhook" that means no battery on the line, and thus
		it is out of service..., if it's on a TDM card... If it's a channel
		bank, there is no telling... */
		return (par.rxbits > -1) || par.rxisoffhook;
	}

	return par.rxisoffhook;
}

static int my_set_echocanceller(void *pvt, int enable)
{
	struct dahdi_pvt *p = pvt;

	if (enable)
		dahdi_ec_enable(p);
	else
		dahdi_ec_disable(p);

	return 0;
}

static int dahdi_ring_phone(struct dahdi_pvt *p);

static int my_ring(void *pvt)
{
	struct dahdi_pvt *p = pvt;

	return dahdi_ring_phone(p);
}

static int my_flash(void *pvt)
{
	struct dahdi_pvt *p = pvt;
	int func = DAHDI_FLASH;
	return ioctl(p->subs[SUB_REAL].dfd, DAHDI_HOOK, &func);
}

static inline int dahdi_set_hook(int fd, int hs);

static int my_off_hook(void *pvt)
{
	struct dahdi_pvt *p = pvt;
	return dahdi_set_hook(p->subs[SUB_REAL].dfd, DAHDI_OFFHOOK);
}

static void my_set_needringing(void *pvt, int value)
{
	struct dahdi_pvt *p = pvt;
	p->subs[SUB_REAL].needringing = value;
}

static void my_set_polarity(void *pvt, int value)
{
	struct dahdi_pvt *p = pvt;

	if (p->channel == CHAN_PSEUDO) {
		return;
	}
	p->polarity = value;
	ioctl(p->subs[SUB_REAL].dfd, DAHDI_SETPOLARITY, &value);
}

static void my_start_polarityswitch(void *pvt)
{
	struct dahdi_pvt *p = pvt;

	if (p->answeronpolarityswitch || p->hanguponpolarityswitch) {
		my_set_polarity(pvt, 0);
	}
}

static void my_answer_polarityswitch(void *pvt)
{
	struct dahdi_pvt *p = pvt;

	if (!p->answeronpolarityswitch) {
		return;
	}

	my_set_polarity(pvt, 1);
}

static void my_hangup_polarityswitch(void *pvt)
{
	struct dahdi_pvt *p = pvt;

	if (!p->hanguponpolarityswitch) {
		return;
	}

	if (p->answeronpolarityswitch) {
		my_set_polarity(pvt, 0);
	} else {
		my_set_polarity(pvt, 1);
	}
}

static int my_start(void *pvt)
{
	struct dahdi_pvt *p = pvt;
	int x = DAHDI_START;

	return ioctl(p->subs[SUB_REAL].dfd, DAHDI_HOOK, &x);
}

static int my_dial_digits(void *pvt, enum analog_sub sub, struct analog_dialoperation *dop)
{
	struct dahdi_pvt *p = pvt;

	if (dop->op != ANALOG_DIAL_OP_REPLACE) {
		ast_log(LOG_ERROR, "Fix the dial_digits callback!\n");
		return -1;
	}

	if (sub != ANALOG_SUB_REAL) {
		ast_log(LOG_ERROR, "Trying to dial_digits '%s' on channel %d subchannel %u\n",
			dop->dialstr, p->channel, sub);
		return -1;
	}

	return dahdi_dial_str(p, DAHDI_DIAL_OP_REPLACE, dop->dialstr);
}

static void dahdi_train_ec(struct dahdi_pvt *p);

static int my_train_echocanceller(void *pvt)
{
	struct dahdi_pvt *p = pvt;

	dahdi_train_ec(p);

	return 0;
}

static int my_is_dialing(void *pvt, enum analog_sub sub)
{
	struct dahdi_pvt *p = pvt;
	int index;
	int x;

	index = analogsub_to_dahdisub(sub);

	if (ioctl(p->subs[index].dfd, DAHDI_DIALING, &x)) {
		ast_debug(1, "DAHDI_DIALING ioctl failed!\n");
		return -1;
	}

	return x;
}

static int my_on_hook(void *pvt)
{
	struct dahdi_pvt *p = pvt;
	return dahdi_set_hook(p->subs[ANALOG_SUB_REAL].dfd, DAHDI_ONHOOK);
}

#if defined(HAVE_PRI)
static void my_pri_fixup_chans(void *chan_old, void *chan_new)
{
	struct dahdi_pvt *old_chan = chan_old;
	struct dahdi_pvt *new_chan = chan_new;

	new_chan->owner = old_chan->owner;
	old_chan->owner = NULL;
	if (new_chan->owner) {
		ast_channel_tech_pvt_set(new_chan->owner, new_chan);
		ast_channel_internal_fd_set(new_chan->owner, 0, new_chan->subs[SUB_REAL].dfd);
		new_chan->subs[SUB_REAL].owner = old_chan->subs[SUB_REAL].owner;
		old_chan->subs[SUB_REAL].owner = NULL;
	}
	/* Copy any DSP that may be present */
	new_chan->dsp = old_chan->dsp;
	new_chan->dsp_features = old_chan->dsp_features;
	old_chan->dsp = NULL;
	old_chan->dsp_features = 0;

	/* Transfer flags from the old channel. */
	new_chan->dialing = old_chan->dialing;
	new_chan->digital = old_chan->digital;
	new_chan->outgoing = old_chan->outgoing;
	old_chan->dialing = 0;
	old_chan->digital = 0;
	old_chan->outgoing = 0;

	/* More stuff to transfer to the new channel. */
	new_chan->law = old_chan->law;
	strcpy(new_chan->dialstring, old_chan->dialstring);
}
#endif	/* defined(HAVE_PRI) */

#if defined(HAVE_PRI)
static int sig_pri_tone_to_dahditone(enum sig_pri_tone tone)
{
	switch (tone) {
	case SIG_PRI_TONE_RINGTONE:
		return DAHDI_TONE_RINGTONE;
	case SIG_PRI_TONE_STUTTER:
		return DAHDI_TONE_STUTTER;
	case SIG_PRI_TONE_CONGESTION:
		return DAHDI_TONE_CONGESTION;
	case SIG_PRI_TONE_DIALTONE:
		return DAHDI_TONE_DIALTONE;
	case SIG_PRI_TONE_DIALRECALL:
		return DAHDI_TONE_DIALRECALL;
	case SIG_PRI_TONE_INFO:
		return DAHDI_TONE_INFO;
	case SIG_PRI_TONE_BUSY:
		return DAHDI_TONE_BUSY;
	default:
		return -1;
	}
}
#endif	/* defined(HAVE_PRI) */

#if defined(HAVE_PRI)
static void my_handle_dchan_exception(struct sig_pri_span *pri, int index)
{
	int x;

	ioctl(pri->fds[index], DAHDI_GETEVENT, &x);
	switch (x) {
	case DAHDI_EVENT_NONE:
		break;
	case DAHDI_EVENT_ALARM:
	case DAHDI_EVENT_NOALARM:
		if (sig_pri_is_alarm_ignored(pri)) {
			break;
		}
		/* Fall through */
	default:
		ast_log(LOG_NOTICE, "Got DAHDI event: %s (%d) on D-channel of span %d\n",
			event2str(x), x, pri->span);
		break;
	}
	/* Keep track of alarm state */
	switch (x) {
	case DAHDI_EVENT_ALARM:
		pri_event_alarm(pri, index, 0);
		break;
	case DAHDI_EVENT_NOALARM:
		pri_event_noalarm(pri, index, 0);
		break;
	case DAHDI_EVENT_REMOVED:
		pri_queue_for_destruction(pri);
		break;
	default:
		break;
	}
}
#endif	/* defined(HAVE_PRI) */

#if defined(HAVE_PRI)
static int my_pri_play_tone(void *pvt, enum sig_pri_tone tone)
{
	struct dahdi_pvt *p = pvt;

	return tone_zone_play_tone(p->subs[SUB_REAL].dfd, sig_pri_tone_to_dahditone(tone));
}
#endif	/* defined(HAVE_PRI) */

#if defined(HAVE_PRI) || defined(HAVE_SS7)
/*!
 * \internal
 * \brief Set the caller id information.
 * \since 1.8
 *
 * \param pvt DAHDI private structure
 * \param caller Caller-id information to set.
 */
static void my_set_callerid(void *pvt, const struct ast_party_caller *caller)
{
	struct dahdi_pvt *p = pvt;

	ast_copy_string(p->cid_num,
		S_COR(caller->id.number.valid, caller->id.number.str, ""),
		sizeof(p->cid_num));
	ast_copy_string(p->cid_name,
		S_COR(caller->id.name.valid, caller->id.name.str, ""),
		sizeof(p->cid_name));
	ast_copy_string(p->cid_subaddr,
		S_COR(caller->id.subaddress.valid, caller->id.subaddress.str, ""),
		sizeof(p->cid_subaddr));
	p->cid_ton = caller->id.number.plan;
	p->callingpres = ast_party_id_presentation(&caller->id);
	if (caller->id.tag) {
		ast_copy_string(p->cid_tag, caller->id.tag, sizeof(p->cid_tag));
	}
	ast_copy_string(p->cid_ani,
		S_COR(caller->ani.number.valid, caller->ani.number.str, ""),
		sizeof(p->cid_ani));
	p->cid_ani2 = caller->ani2;
}
#endif	/* defined(HAVE_PRI) || defined(HAVE_SS7) */

#if defined(HAVE_PRI) || defined(HAVE_SS7)
/*!
 * \internal
 * \brief Set the Dialed Number Identifier.
 * \since 1.8
 *
 * \param pvt DAHDI private structure
 * \param dnid Dialed Number Identifier string.
 */
static void my_set_dnid(void *pvt, const char *dnid)
{
	struct dahdi_pvt *p = pvt;

	ast_copy_string(p->dnid, dnid, sizeof(p->dnid));
}
#endif	/* defined(HAVE_PRI) || defined(HAVE_SS7) */

#if defined(HAVE_PRI)
/*!
 * \internal
 * \brief Set the Redirecting Directory Number Information Service (RDNIS).
 * \since 1.8
 *
 * \param pvt DAHDI private structure
 * \param rdnis Redirecting Directory Number Information Service (RDNIS) string.
 */
static void my_set_rdnis(void *pvt, const char *rdnis)
{
	struct dahdi_pvt *p = pvt;

	ast_copy_string(p->rdnis, rdnis, sizeof(p->rdnis));
}
#endif	/* defined(HAVE_PRI) */

#if defined(HAVE_PRI)
/*!
 * \internal
 * \brief Make a dialstring for native ISDN CC to recall properly.
 * \since 1.8
 *
 * \param priv Channel private control structure.
 * \param buf Where to put the modified dialstring.
 * \param buf_size Size of modified dialstring buffer.
 *
 * \details
 * original dialstring:
 * \verbatim
   DAHDI/[i<span>-](g|G|r|R)<group#(0-63)>[c|r<cadence#>|d][/extension[/options]]
   \endverbatim
 *
 * The modified dialstring will have prefixed the channel-group section
 * with the ISDN channel restriction.
 *
 * buf:
 * \verbatim
   DAHDI/i<span>-(g|G|r|R)<group#(0-63)>[c|r<cadence#>|d][/extension[/options]]
   \endverbatim
 *
 * The routine will check to see if the ISDN channel restriction is already
 * in the original dialstring.
 */
static void my_pri_make_cc_dialstring(void *priv, char *buf, size_t buf_size)
{
	char *dial;
	struct dahdi_pvt *pvt;
	AST_DECLARE_APP_ARGS(args,
		AST_APP_ARG(tech);	/* channel technology token */
		AST_APP_ARG(group);	/* channel/group token */
		//AST_APP_ARG(ext);	/* extension token */
		//AST_APP_ARG(opts);	/* options token */
		//AST_APP_ARG(other);	/* Any remaining unused arguments */
	);

	pvt = priv;
	dial = ast_strdupa(pvt->dialstring);
	AST_NONSTANDARD_APP_ARGS(args, dial, '/');
	if (!args.tech) {
		ast_copy_string(buf, pvt->dialstring, buf_size);
		return;
	}
	if (!args.group) {
		/* Append the ISDN span channel restriction to the dialstring. */
		snprintf(buf, buf_size, "%s/i%d-", args.tech, pvt->pri->span);
		return;
	}
	if (isdigit(args.group[0]) || args.group[0] == 'i' || strchr(args.group, '!')) {
		/* The ISDN span channel restriction is not needed or already
		 * in the dialstring. */
		ast_copy_string(buf, pvt->dialstring, buf_size);
		return;
	}
	/* Insert the ISDN span channel restriction into the dialstring. */
	snprintf(buf, buf_size, "%s/i%d-%s", args.tech, pvt->pri->span, args.group);
}
#endif	/* defined(HAVE_PRI) */

#if defined(HAVE_PRI)
/*!
 * \internal
 * \brief Reevaluate the PRI span device state.
 * \since 1.8
 *
 * \param pri Asterisk D channel control structure.
 *
 * \note Assumes the pri->lock is already obtained.
 */
static void dahdi_pri_update_span_devstate(struct sig_pri_span *pri)
{
	unsigned idx;
	unsigned num_b_chans;	/* Number of B channels provisioned on the span. */
	unsigned in_use;		/* Number of B channels in use on the span. */
	unsigned in_alarm;		/* TRUE if the span is in alarm condition. */
	enum ast_device_state new_state;

	/* Count the number of B channels and the number of B channels in use. */
	num_b_chans = 0;
	in_use = 0;
	in_alarm = 1;
	for (idx = pri->numchans; idx--;) {
		if (pri->pvts[idx] && !pri->pvts[idx]->no_b_channel) {
			/* This is a B channel interface. */
			++num_b_chans;
			if (!sig_pri_is_chan_available(pri->pvts[idx])) {
				++in_use;
			}
			if (!pri->pvts[idx]->inalarm) {
				/* There is a channel that is not in alarm. */
				in_alarm = 0;
			}
		}
	}

	/* Update the span congestion device state and report any change. */
	if (in_alarm) {
		new_state = AST_DEVICE_UNAVAILABLE;
	} else {
		new_state = num_b_chans == in_use ? AST_DEVICE_BUSY : AST_DEVICE_NOT_INUSE;
	}
	if (pri->congestion_devstate != new_state) {
		pri->congestion_devstate = new_state;
		ast_devstate_changed(AST_DEVICE_UNKNOWN, AST_DEVSTATE_NOT_CACHABLE, "DAHDI/I%d/congestion", pri->span);
	}
#if defined(THRESHOLD_DEVSTATE_PLACEHOLDER)
	/* Update the span threshold device state and report any change. */
	if (in_alarm) {
		new_state = AST_DEVICE_UNAVAILABLE;
	} else if (!in_use) {
		new_state = AST_DEVICE_NOT_INUSE;
	} else if (!pri->user_busy_threshold) {
		new_state = in_use < num_b_chans ? AST_DEVICE_INUSE : AST_DEVICE_BUSY;
	} else {
		new_state = in_use < pri->user_busy_threshold ? AST_DEVICE_INUSE
			: AST_DEVICE_BUSY;
	}
	if (pri->threshold_devstate != new_state) {
		pri->threshold_devstate = new_state;
		ast_devstate_changed(AST_DEVICE_UNKNOWN, AST_DEVSTATE_NOT_CACHABLE, "DAHDI/I%d/threshold", pri->span);
	}
#endif	/* defined(THRESHOLD_DEVSTATE_PLACEHOLDER) */
}
#endif	/* defined(HAVE_PRI) */

#if defined(HAVE_PRI)
/*!
 * \internal
 * \brief Reference this module.
 * \since 1.8
 */
static void my_module_ref(void)
{
	ast_module_ref(ast_module_info->self);
}
#endif	/* defined(HAVE_PRI) */

#if defined(HAVE_PRI)
/*!
 * \internal
 * \brief Unreference this module.
 * \since 1.8
 */
static void my_module_unref(void)
{
	ast_module_unref(ast_module_info->self);
}
#endif	/* defined(HAVE_PRI) */

#if defined(HAVE_PRI)
#if defined(HAVE_PRI_CALL_WAITING)
static void my_pri_init_config(void *priv, struct sig_pri_span *pri);
#endif	/* defined(HAVE_PRI_CALL_WAITING) */
static int dahdi_new_pri_nobch_channel(struct sig_pri_span *pri);

struct sig_pri_callback sig_pri_callbacks =
{
	.handle_dchan_exception = my_handle_dchan_exception,
	.play_tone = my_pri_play_tone,
	.set_echocanceller = my_set_echocanceller,
	.dsp_reset_and_flush_digits = my_dsp_reset_and_flush_digits,
	.lock_private = my_lock_private,
	.unlock_private = my_unlock_private,
	.deadlock_avoidance_private = my_deadlock_avoidance_private,
	.new_ast_channel = my_new_pri_ast_channel,
	.fixup_chans = my_pri_fixup_chans,
	.set_alarm = my_set_alarm,
	.set_dialing = my_set_dialing,
	.set_outgoing = my_set_outgoing,
	.set_digital = my_set_digital,
	.set_callerid = my_set_callerid,
	.set_dnid = my_set_dnid,
	.set_rdnis = my_set_rdnis,
	.new_nobch_intf = dahdi_new_pri_nobch_channel,
#if defined(HAVE_PRI_CALL_WAITING)
	.init_config = my_pri_init_config,
#endif	/* defined(HAVE_PRI_CALL_WAITING) */
	.get_orig_dialstring = my_get_orig_dialstring,
	.make_cc_dialstring = my_pri_make_cc_dialstring,
	.update_span_devstate = dahdi_pri_update_span_devstate,
	.module_ref = my_module_ref,
	.module_unref = my_module_unref,
	.dial_digits = my_pri_dial_digits,
	.open_media = my_pri_ss7_open_media,
	.ami_channel_event = my_ami_channel_event,
	.destroy_later = pri_queue_for_destruction,
};
#endif	/* defined(HAVE_PRI) */

#if defined(HAVE_SS7)
/*!
 * \internal
 * \brief Handle the SS7 link exception.
 * \since 1.8
 *
 * \param linkset Controlling linkset for the channel.
 * \param which Link index of the signaling channel.
 */
static void my_handle_link_exception(struct sig_ss7_linkset *linkset, int which)
{
	int event;

	if (ioctl(linkset->fds[which], DAHDI_GETEVENT, &event)) {
		ast_log(LOG_ERROR, "SS7: Error in exception retrieval on span %d/%d!\n",
			linkset->span, which);
		return;
	}
	switch (event) {
	case DAHDI_EVENT_NONE:
		break;
	case DAHDI_EVENT_ALARM:
		ast_log(LOG_ERROR, "SS7 got event: %s(%d) on span %d/%d\n",
			event2str(event), event, linkset->span, which);
		sig_ss7_link_alarm(linkset, which);
		break;
	case DAHDI_EVENT_NOALARM:
		ast_log(LOG_ERROR, "SS7 got event: %s(%d) on span %d/%d\n",
			event2str(event), event, linkset->span, which);
		sig_ss7_link_noalarm(linkset, which);
		break;
	default:
		ast_log(LOG_NOTICE, "SS7 got event: %s(%d) on span %d/%d\n",
			event2str(event), event, linkset->span, which);
		break;
	}
}
#endif	/* defined(HAVE_SS7) */

#if defined(HAVE_SS7)
static void my_ss7_set_loopback(void *pvt, int enable)
{
	struct dahdi_pvt *p = pvt;

	if (ioctl(p->subs[SUB_REAL].dfd, DAHDI_LOOPBACK, &enable)) {
		ast_log(LOG_WARNING, "Unable to set loopback on channel %d: %s\n", p->channel,
			strerror(errno));
	}
}
#endif	/* defined(HAVE_SS7) */

#if defined(HAVE_SS7)
/*!
 * \internal
 * \brief Find the linkset to which SS7 belongs.
 * \since 11.0
 *
 * \param ss7 structure to match on.
 *
 * \retval linkset if found.
 * \retval NULL if not found.
 */
static struct sig_ss7_linkset *my_ss7_find_linkset(struct ss7 *ss7)
{
	int idx;

	if (!ss7) {
		return NULL;
	}

	for (idx = 0; idx < NUM_SPANS; ++idx) {
		if (linksets[idx].ss7.ss7 == ss7) {
			return &linksets[idx].ss7;
		}
	}
	return NULL;
}
#endif	/* defined(HAVE_SS7) */

#if defined(HAVE_SS7)
/*!
 * \internal
 * \brief Create a new asterisk channel structure for SS7.
 * \since 1.8
 *
 * \param pvt Private channel structure.
 * \param state Initial state of new channel.
 * \param law Companding law to use.
 * \param exten Dialplan extension for incoming call.
 * \param requestor Channel requesting this new channel.
 * \param assignedids
 *
 * \retval ast_channel on success.
 * \retval NULL on error.
 */
static struct ast_channel *my_new_ss7_ast_channel(void *pvt, int state, enum sig_ss7_law law, char *exten, const struct ast_assigned_ids *assignedids, const struct ast_channel *requestor)
{
	struct dahdi_pvt *p = pvt;
	int audio;
	int newlaw;
	ast_callid callid = 0;
	int callid_created = ast_callid_threadstorage_auto(&callid);

	/* Set to audio mode at this point */
	audio = 1;
	if (ioctl(p->subs[SUB_REAL].dfd, DAHDI_AUDIOMODE, &audio) == -1)
		ast_log(LOG_WARNING, "Unable to set audio mode on channel %d to %d: %s\n",
			p->channel, audio, strerror(errno));

	if (law != SIG_SS7_DEFLAW) {
		dahdi_setlaw(p->subs[SUB_REAL].dfd,
			(law == SIG_SS7_ULAW) ? DAHDI_LAW_MULAW : DAHDI_LAW_ALAW);
	}

	ast_copy_string(p->exten, exten, sizeof(p->exten));

	newlaw = -1;
	switch (law) {
	case SIG_SS7_DEFLAW:
		newlaw = 0;
		break;
	case SIG_SS7_ALAW:
		newlaw = DAHDI_LAW_ALAW;
		break;
	case SIG_SS7_ULAW:
		newlaw = DAHDI_LAW_MULAW;
		break;
	}
	return dahdi_new_callid_clean(p, state, 0, SUB_REAL, newlaw, assignedids, requestor, callid, callid_created);
}
#endif	/* defined(HAVE_SS7) */

#if defined(HAVE_SS7)
static int sig_ss7_tone_to_dahditone(enum sig_ss7_tone tone)
{
	switch (tone) {
	case SIG_SS7_TONE_RINGTONE:
		return DAHDI_TONE_RINGTONE;
	case SIG_SS7_TONE_STUTTER:
		return DAHDI_TONE_STUTTER;
	case SIG_SS7_TONE_CONGESTION:
		return DAHDI_TONE_CONGESTION;
	case SIG_SS7_TONE_DIALTONE:
		return DAHDI_TONE_DIALTONE;
	case SIG_SS7_TONE_DIALRECALL:
		return DAHDI_TONE_DIALRECALL;
	case SIG_SS7_TONE_INFO:
		return DAHDI_TONE_INFO;
	case SIG_SS7_TONE_BUSY:
		return DAHDI_TONE_BUSY;
	default:
		return -1;
	}
}
#endif	/* defined(HAVE_SS7) */

#if defined(HAVE_SS7)
static int my_ss7_play_tone(void *pvt, enum sig_ss7_tone tone)
{
	struct dahdi_pvt *p = pvt;

	return tone_zone_play_tone(p->subs[SUB_REAL].dfd, sig_ss7_tone_to_dahditone(tone));
}
#endif	/* defined(HAVE_SS7) */

#if defined(HAVE_SS7)
struct sig_ss7_callback sig_ss7_callbacks =
{
	.lock_private = my_lock_private,
	.unlock_private = my_unlock_private,
	.deadlock_avoidance_private = my_deadlock_avoidance_private,

	.set_echocanceller = my_set_echocanceller,
	.set_loopback = my_ss7_set_loopback,

	.new_ast_channel = my_new_ss7_ast_channel,
	.play_tone = my_ss7_play_tone,

	.handle_link_exception = my_handle_link_exception,
	.set_alarm = my_set_alarm,
	.set_dialing = my_set_dialing,
	.set_outgoing = my_set_outgoing,
	.set_digital = my_set_digital,
	.set_inservice = my_set_inservice,
	.set_locallyblocked = my_set_locallyblocked,
	.set_remotelyblocked = my_set_remotelyblocked,
	.set_callerid = my_set_callerid,
	.set_dnid = my_set_dnid,
	.open_media = my_pri_ss7_open_media,
	.find_linkset = my_ss7_find_linkset,
};
#endif	/* defined(HAVE_SS7) */

/*!
 * \brief Send MWI state change
 *
 * \param mailbox This is the mailbox associated with the FXO line that the
 *      MWI state has changed on.
 * \param thereornot This argument should simply be set to 1 or 0, to indicate
 *      whether there are messages waiting or not.
 *
 * This function does two things:
 *
 * 1) It generates an internal Asterisk event notifying any other module that
 *    cares about MWI that the state of a mailbox has changed.
 *
 * 2) It runs the script specified by the mwimonitornotify option to allow
 *    some custom handling of the state change.
 */
static void notify_message(char *mailbox, int thereornot)
{
	char s[sizeof(mwimonitornotify) + 164];

	if (ast_strlen_zero(mailbox)) {
		return;
	}

	ast_publish_mwi_state(mailbox, NULL, thereornot, thereornot);
	if (!ast_strlen_zero(mwimonitornotify)) {
		snprintf(s, sizeof(s), "%s %s %d", mwimonitornotify, mailbox, thereornot);
		ast_safe_system(s);
	}
}

static void my_handle_notify_message(struct ast_channel *chan, void *pvt, int cid_flags, int neon_mwievent)
{
	struct dahdi_pvt *p = pvt;

	if (neon_mwievent > -1 && !p->mwimonitor_neon)
		return;

	if (neon_mwievent == ANALOG_EVENT_NEONMWI_ACTIVE || cid_flags & CID_MSGWAITING) {
		ast_log(LOG_NOTICE, "MWI: Channel %d message waiting, mailbox %s\n", p->channel, p->mailbox);
		notify_message(p->mailbox, 1);
	} else if (neon_mwievent == ANALOG_EVENT_NEONMWI_INACTIVE || cid_flags & CID_NOMSGWAITING) {
		ast_log(LOG_NOTICE, "MWI: Channel %d no message waiting, mailbox %s\n", p->channel, p->mailbox);
		notify_message(p->mailbox, 0);
	}
	/* If the CID had Message waiting payload, assume that this for MWI only and hangup the call */
	/* If generated using Ring Pulse Alert, then ring has been answered as a call and needs to be hungup */
	if (neon_mwievent == -1 && p->mwimonitor_rpas) {
		ast_hangup(chan);
		return;
	}
}

static int my_have_progressdetect(void *pvt)
{
	struct dahdi_pvt *p = pvt;

	if ((p->callprogress & CALLPROGRESS_PROGRESS)
		&& CANPROGRESSDETECT(p) && p->dsp && p->outgoing) {
		return 1;
	} else {
		/* Don't have progress detection. */
		return 0;
	}
}

#define gen_pvt_field_callback(type, field) \
	static type my_get_##field(void *pvt) \
	{ \
		struct dahdi_pvt *p = pvt; \
		return p->field; \
	}

gen_pvt_field_callback(int, firstdigit_timeout);
gen_pvt_field_callback(int, interdigit_timeout);
gen_pvt_field_callback(int, matchdigit_timeout);

#undef gen_pvt_field_callback

struct analog_callback analog_callbacks =
{
	.play_tone = my_play_tone,
	.get_event = my_get_event,
	.wait_event = my_wait_event,
	.is_off_hook = my_is_off_hook,
	.set_echocanceller = my_set_echocanceller,
	.ring = my_ring,
	.flash = my_flash,
	.off_hook = my_off_hook,
	.dial_digits = my_dial_digits,
	.train_echocanceller = my_train_echocanceller,
	.on_hook = my_on_hook,
	.is_dialing = my_is_dialing,
	.allocate_sub = my_allocate_sub,
	.unallocate_sub = my_unallocate_sub,
	.swap_subs = my_swap_subchannels,
	.has_voicemail = my_has_voicemail,
	.check_for_conference = my_check_for_conference,
	.conf_add = my_conf_add,
	.conf_del = my_conf_del,
	.complete_conference_update = my_complete_conference_update,
	.start = my_start,
	.all_subchannels_hungup = my_all_subchannels_hungup,
	.lock_private = my_lock_private,
	.unlock_private = my_unlock_private,
	.deadlock_avoidance_private = my_deadlock_avoidance_private,
	.handle_dtmf = my_handle_dtmf,
	.wink = my_wink,
	.new_ast_channel = my_new_analog_ast_channel,
	.dsp_set_digitmode = my_dsp_set_digitmode,
	.dsp_reset_and_flush_digits = my_dsp_reset_and_flush_digits,
	.send_callerid = my_send_callerid,
	.callwait = my_callwait,
	.stop_callwait = my_stop_callwait,
	.get_callerid = my_get_callerid,
	.start_cid_detect = my_start_cid_detect,
	.stop_cid_detect = my_stop_cid_detect,
	.handle_notify_message = my_handle_notify_message,
	.increase_ss_count = my_increase_ss_count,
	.decrease_ss_count = my_decrease_ss_count,
	.distinctive_ring = my_distinctive_ring,
	.set_linear_mode = my_set_linear_mode,
	.set_inthreeway = my_set_inthreeway,
	.get_and_handle_alarms = my_get_and_handle_alarms,
	.get_sigpvt_bridged_channel = my_get_sigpvt_bridged_channel,
	.get_sub_fd = my_get_sub_fd,
	.set_cadence = my_set_cadence,
	.set_alarm = my_set_alarm,
	.set_dialing = my_set_dialing,
	.set_outgoing = my_set_outgoing,
	.set_ringtimeout = my_set_ringtimeout,
	.set_waitingfordt = my_set_waitingfordt,
	.check_waitingfordt = my_check_waitingfordt,
	.set_confirmanswer = my_set_confirmanswer,
	.check_confirmanswer = my_check_confirmanswer,
	.set_callwaiting = my_set_callwaiting,
	.cancel_cidspill = my_cancel_cidspill,
	.confmute = my_confmute,
	.set_pulsedial = my_set_pulsedial,
	.set_new_owner = my_set_new_owner,
	.get_orig_dialstring = my_get_orig_dialstring,
	.set_needringing = my_set_needringing,
	.set_polarity = my_set_polarity,
	.start_polarityswitch = my_start_polarityswitch,
	.answer_polarityswitch = my_answer_polarityswitch,
	.hangup_polarityswitch = my_hangup_polarityswitch,
	.have_progressdetect = my_have_progressdetect,
	.get_firstdigit_timeout = my_get_firstdigit_timeout,
	.get_matchdigit_timeout = my_get_matchdigit_timeout,
	.get_interdigit_timeout = my_get_interdigit_timeout,
};

/*! Round robin search locations. */
static struct dahdi_pvt *round_robin[64]; /* groups can range from 0-63 */

int _dahdi_get_index(struct ast_channel *ast, struct dahdi_pvt *p, int nullok, const char *fname, unsigned long line)
{
	int res;
	if (p->subs[SUB_REAL].owner == ast)
		res = 0;
	else if (p->subs[SUB_CALLWAIT].owner == ast)
		res = 1;
	else if (p->subs[SUB_THREEWAY].owner == ast)
		res = 2;
	else {
		res = -1;
		if (!nullok)
			ast_log(LOG_WARNING,
				"Unable to get index for '%s' on channel %d (%s(), line %lu)\n",
				ast ? ast_channel_name(ast) : "", p->channel, fname, line);
	}
	return res;
}

/*!
 * \internal
 * \brief Obtain the specified subchannel owner lock if the owner exists.
 *
 * \param pvt Channel private struct.
 * \param sub_idx Subchannel owner to lock.
 *
 * \note Assumes the pvt->lock is already obtained.
 *
 * \note
 * Because deadlock avoidance may have been necessary, you need to confirm
 * the state of things before continuing.
 */
static void dahdi_lock_sub_owner(struct dahdi_pvt *pvt, int sub_idx)
{
	for (;;) {
		if (!pvt->subs[sub_idx].owner) {
			/* No subchannel owner pointer */
			break;
		}
		if (!ast_channel_trylock(pvt->subs[sub_idx].owner)) {
			/* Got subchannel owner lock */
			break;
		}
		/* We must unlock the private to avoid the possibility of a deadlock */
		DEADLOCK_AVOIDANCE(&pvt->lock);
	}
}

static void wakeup_sub(struct dahdi_pvt *p, int a)
{
	dahdi_lock_sub_owner(p, a);
	if (p->subs[a].owner) {
		ast_queue_frame(p->subs[a].owner, &ast_null_frame);
		ast_channel_unlock(p->subs[a].owner);
	}
}

static void dahdi_queue_frame(struct dahdi_pvt *p, struct ast_frame *f)
{
	for (;;) {
		if (p->owner) {
			if (ast_channel_trylock(p->owner)) {
				DEADLOCK_AVOIDANCE(&p->lock);
			} else {
				ast_queue_frame(p->owner, f);
				ast_channel_unlock(p->owner);
				break;
			}
		} else
			break;
	}
}

static void publish_channel_alarm_clear(int channel)
{
	RAII_VAR(struct ast_json *, body, NULL, ast_json_unref);
	RAII_VAR(struct ast_str *, dahdi_chan, ast_str_create(32), ast_free);
	if (!dahdi_chan) {
		return;
	}

	ast_str_set(&dahdi_chan, 0, "%d", channel);
	ast_log(LOG_NOTICE, "Alarm cleared on channel DAHDI/%d\n", channel);
	body = ast_json_pack("{s: s}", "DAHDIChannel", ast_str_buffer(dahdi_chan));
	if (!body) {
		return;
	}

	ast_manager_publish_event("AlarmClear", EVENT_FLAG_SYSTEM, body);
}

static void publish_span_alarm_clear(int span)
{
	RAII_VAR(struct ast_json *, body, NULL, ast_json_unref);

	ast_log(LOG_NOTICE, "Alarm cleared on span %d\n", span);
	body = ast_json_pack("{s: i}", "Span", span);
	if (!body) {
		return;
	}

	ast_manager_publish_event("SpanAlarmClear", EVENT_FLAG_SYSTEM, body);
}

static void handle_clear_alarms(struct dahdi_pvt *p)
{
#if defined(HAVE_PRI)
	if (dahdi_sig_pri_lib_handles(p->sig) && sig_pri_is_alarm_ignored(p->pri)) {
		return;
	}
#endif	/* defined(HAVE_PRI) */

	if (report_alarms & REPORT_CHANNEL_ALARMS) {
		publish_channel_alarm_clear(p->channel);
	}
	if (report_alarms & REPORT_SPAN_ALARMS && p->manages_span_alarms) {
		publish_span_alarm_clear(p->span);
	}
}

#ifdef HAVE_OPENR2
static void mfcr2_queue_for_destruction(const struct dahdi_pvt *p)
{
	const struct dahdi_mfcr2 *r2link = p->mfcr2;
	struct r2link_entry *cur;
	AST_LIST_LOCK(&r2links);
	AST_LIST_TRAVERSE_SAFE_BEGIN(&r2links, cur, list) {
		if (r2link == &cur->mfcr2) {
			ast_debug(3, "MFC/R2 channel %d queued for destruction\n", p->channel);
			AST_LIST_MOVE_CURRENT(&nodev_r2links, list);
			break;
		}
	}
	AST_LIST_TRAVERSE_SAFE_END;
	AST_LIST_UNLOCK(&r2links);
}

static int dahdi_r2_answer(struct dahdi_pvt *p)
{
	int res = 0;
	/* openr2 1.1.0 and older does not even define OR2_LIB_INTERFACE
	* and does not has support for openr2_chan_answer_call_with_mode
	*  */
#if defined(OR2_LIB_INTERFACE) && OR2_LIB_INTERFACE > 1
	const char *double_answer = pbx_builtin_getvar_helper(p->owner, "MFCR2_DOUBLE_ANSWER");
	int wants_double_answer = ast_true(double_answer) ? 1 : 0;
	if (!double_answer) {
		/* this still can result in double answer if the channel context
		* was configured that way */
		res = openr2_chan_answer_call(p->r2chan);
	} else if (wants_double_answer) {
		res = openr2_chan_answer_call_with_mode(p->r2chan, OR2_ANSWER_DOUBLE);
	} else {
		res = openr2_chan_answer_call_with_mode(p->r2chan, OR2_ANSWER_SIMPLE);
	}
#else
	res = openr2_chan_answer_call(p->r2chan);
#endif
	return res;
}



/* should be called with the ast_channel locked */
static openr2_calling_party_category_t dahdi_r2_get_channel_category(struct ast_channel *c)
{
	openr2_calling_party_category_t cat;
	const char *catstr = pbx_builtin_getvar_helper(c, "MFCR2_CATEGORY");
	struct dahdi_pvt *p = ast_channel_tech_pvt(c);
	if (ast_strlen_zero(catstr)) {
		ast_debug(1, "No MFC/R2 category specified for chan %s, using default %s\n",
				ast_channel_name(c), openr2_proto_get_category_string(p->mfcr2_category));
		return p->mfcr2_category;
	}
	if ((cat = openr2_proto_get_category(catstr)) == OR2_CALLING_PARTY_CATEGORY_UNKNOWN) {
		ast_log(LOG_WARNING, "Invalid category specified '%s' for chan %s, using default %s\n",
				catstr, ast_channel_name(c), openr2_proto_get_category_string(p->mfcr2_category));
		return p->mfcr2_category;
	}
	ast_debug(1, "Using category %s\n", catstr);
	return cat;
}

static void dahdi_r2_on_call_init(openr2_chan_t *r2chan)
{
	struct dahdi_pvt *p = openr2_chan_get_client_data(r2chan);
	ast_mutex_lock(&p->lock);
	if (p->mfcr2call) {
		ast_mutex_unlock(&p->lock);
		/* TODO: This can happen when some other thread just finished dahdi_request requesting this very same
		   interface but has not yet seized the line (dahdi_call), and the far end wins and seize the line,
		   can we avoid this somehow?, at this point when dahdi_call send the seize, it is likely that since
		   the other end will see our seize as a forced release and drop the call, we will see an invalid
		   pattern that will be seen and treated as protocol error. */
		ast_log(LOG_ERROR, "Collision of calls on chan %d detected!.\n", openr2_chan_get_number(r2chan));
		return;
	}
	p->mfcr2call = 1;
	/* better safe than sorry ... */
	p->cid_name[0] = '\0';
	p->cid_num[0] = '\0';
	p->cid_subaddr[0] = '\0';
	p->rdnis[0] = '\0';
	p->exten[0] = '\0';
	p->mfcr2_ani_index = '\0';
	p->mfcr2_dnis_index = '\0';
	p->mfcr2_dnis_matched = 0;
	p->mfcr2_answer_pending = 0;
	p->mfcr2_call_accepted = 0;
	ast_mutex_unlock(&p->lock);
	ast_verbose("New MFC/R2 call detected on chan %d.\n", openr2_chan_get_number(r2chan));
}

static void dahdi_r2_on_hardware_alarm(openr2_chan_t *r2chan, int alarm)
{
	int res;
	struct dahdi_pvt *p = openr2_chan_get_client_data(r2chan);
	ast_mutex_lock(&p->lock);
	p->inalarm = alarm ? 1 : 0;
	if (p->inalarm) {
		res = get_alarms(p);
		if (res == DAHDI_ALARM_NOTOPEN) {
			mfcr2_queue_for_destruction(p);
		}
		handle_alarms(p, res);
	} else {
		handle_clear_alarms(p);
	}
	ast_mutex_unlock(&p->lock);
}

static void dahdi_r2_on_os_error(openr2_chan_t *r2chan, int errorcode)
{
	struct dahdi_pvt *p = openr2_chan_get_client_data(r2chan);

	ast_log(LOG_ERROR, "OS error on chan %d: %s\n", openr2_chan_get_number(r2chan), strerror(errorcode));
	ast_mutex_lock(&p->lock);
	/* Disconnected? */
	if (errorcode == ENODEV) {
		struct dahdi_mfcr2 *r2link = p->mfcr2;
		p->mfcr2call = 0;
		if (r2link) {
			r2link->nodev = 1;
		}
	}
	ast_mutex_unlock(&p->lock);
}

static void dahdi_r2_on_protocol_error(openr2_chan_t *r2chan, openr2_protocol_error_t reason)
{
	struct dahdi_pvt *p = openr2_chan_get_client_data(r2chan);
	ast_log(LOG_ERROR, "MFC/R2 protocol error on chan %d: %s\n", openr2_chan_get_number(r2chan), openr2_proto_get_error(reason));
	if (p->owner) {
		ast_channel_hangupcause_set(p->owner, AST_CAUSE_PROTOCOL_ERROR);
		ast_channel_softhangup_internal_flag_add(p->owner, AST_SOFTHANGUP_DEV);
	}
	ast_mutex_lock(&p->lock);
	p->mfcr2call = 0;
	ast_mutex_unlock(&p->lock);
}

static void dahdi_r2_disconnect_call(struct dahdi_pvt *p, openr2_call_disconnect_cause_t cause)
{
	if (openr2_chan_disconnect_call(p->r2chan, cause)) {
		ast_log(LOG_NOTICE, "Bad! failed to disconnect call on channel %d with reason %s, hope for the best!\n",
		   p->channel, openr2_proto_get_disconnect_string(cause));
		/* force the chan to idle and release the call flag now since we will not see a clean on_call_end */
		openr2_chan_set_idle(p->r2chan);
		ast_mutex_lock(&p->lock);
		p->mfcr2call = 0;
		ast_mutex_unlock(&p->lock);
	}
}

static void dahdi_r2_on_call_offered(openr2_chan_t *r2chan, const char *ani, const char *dnis, openr2_calling_party_category_t category)
{
	struct dahdi_pvt *p;
	struct ast_channel *c;
	ast_callid callid = 0;
	int callid_created = ast_callid_threadstorage_auto(&callid);
	ast_verbose("MFC/R2 call offered on chan %d. ANI = %s, DNIS = %s, Category = %s\n",
			openr2_chan_get_number(r2chan), ani ? ani : "(restricted)", dnis,
			openr2_proto_get_category_string(category));
	p = openr2_chan_get_client_data(r2chan);
	/* if collect calls are not allowed and this is a collect call, reject it! */
	if (!p->mfcr2_allow_collect_calls && category == OR2_CALLING_PARTY_CATEGORY_COLLECT_CALL) {
		ast_log(LOG_NOTICE, "Rejecting MFC/R2 collect call\n");
		dahdi_r2_disconnect_call(p, OR2_CAUSE_COLLECT_CALL_REJECTED);
		goto dahdi_r2_on_call_offered_cleanup;
	}
	ast_mutex_lock(&p->lock);
	p->mfcr2_recvd_category = category;
	/* if we're not supposed to use CID, clear whatever we have */
	if (!p->use_callerid) {
		ast_debug(1, "No CID allowed in configuration, CID is being cleared!\n");
		p->cid_num[0] = 0;
		p->cid_name[0] = 0;
	}
	/* if we're supposed to answer immediately, clear DNIS and set 's' exten */
	if (p->immediate || !openr2_context_get_max_dnis(openr2_chan_get_context(r2chan))) {
		ast_debug(1, "Setting exten => s because of immediate or 0 DNIS configured\n");
		p->exten[0] = 's';
		p->exten[1] = 0;
	}
	ast_mutex_unlock(&p->lock);
	if (!ast_exists_extension(NULL, p->context, p->exten, 1, p->cid_num)) {
		ast_log(LOG_NOTICE, "MFC/R2 call on channel %d requested non-existent extension '%s' in context '%s'. Rejecting call.\n",
				p->channel, p->exten, p->context);
		dahdi_r2_disconnect_call(p, OR2_CAUSE_UNALLOCATED_NUMBER);
		goto dahdi_r2_on_call_offered_cleanup;
	}
	if (!p->mfcr2_accept_on_offer) {
		/* The user wants us to start the PBX thread right away without accepting the call first */
		c = dahdi_new(p, AST_STATE_RING, 1, SUB_REAL, DAHDI_LAW_ALAW, NULL, NULL, callid);
		if (c) {
			/* Done here, don't disable reading now since we still need to generate MF tones to accept
			   the call or reject it and detect the tone off condition of the other end, all of this
			   will be done in the PBX thread now */
			goto dahdi_r2_on_call_offered_cleanup;
		}
		ast_log(LOG_WARNING, "Unable to create PBX channel in DAHDI channel %d\n", p->channel);
		dahdi_r2_disconnect_call(p, OR2_CAUSE_OUT_OF_ORDER);
	} else if (p->mfcr2_charge_calls) {
		ast_debug(1, "Accepting MFC/R2 call with charge on chan %d\n", p->channel);
		openr2_chan_accept_call(r2chan, OR2_CALL_WITH_CHARGE);
	} else {
		ast_debug(1, "Accepting MFC/R2 call with no charge on chan %d\n", p->channel);
		openr2_chan_accept_call(r2chan, OR2_CALL_NO_CHARGE);
	}

dahdi_r2_on_call_offered_cleanup:
	ast_callid_threadstorage_auto_clean(callid, callid_created);
}

static void dahdi_r2_on_call_end(openr2_chan_t *r2chan)
{
	struct dahdi_pvt *p = openr2_chan_get_client_data(r2chan);
	ast_verbose("MFC/R2 call end on channel %d\n", p->channel);
	ast_mutex_lock(&p->lock);
	p->mfcr2call = 0;
	ast_mutex_unlock(&p->lock);
}

static void dahdi_r2_on_call_accepted(openr2_chan_t *r2chan, openr2_call_mode_t mode)
{
	struct dahdi_pvt *p = NULL;
	struct ast_channel *c = NULL;
	ast_callid callid = 0;
	int callid_created = ast_callid_threadstorage_auto(&callid);
	p = openr2_chan_get_client_data(r2chan);
	dahdi_ec_enable(p);
	p->mfcr2_call_accepted = 1;
	/* if it's an incoming call ... */
	if (OR2_DIR_BACKWARD == openr2_chan_get_direction(r2chan)) {
		ast_verbose("MFC/R2 call has been accepted on backward channel %d\n", openr2_chan_get_number(r2chan));
		/* If accept on offer is not set, it means at this point the PBX thread is already
		   launched (was launched in the 'on call offered' handler) and therefore this callback
		   is being executed already in the PBX thread rather than the monitor thread, don't launch
		   any other thread, just disable the openr2 reading and answer the call if needed */
		if (!p->mfcr2_accept_on_offer) {
			openr2_chan_disable_read(r2chan);
			if (p->mfcr2_answer_pending) {
				ast_debug(1, "Answering MFC/R2 call after accepting it on chan %d\n", openr2_chan_get_number(r2chan));
				dahdi_r2_answer(p);
			}
			goto dahdi_r2_on_call_accepted_cleanup;
		}
		c = dahdi_new(p, AST_STATE_RING, 1, SUB_REAL, DAHDI_LAW_ALAW, NULL, NULL, callid);
		if (c) {
			/* chan_dahdi will take care of reading from now on in the PBX thread, tell the
			   library to forget about it */
			openr2_chan_disable_read(r2chan);
			goto dahdi_r2_on_call_accepted_cleanup;
		}
		ast_log(LOG_WARNING, "Unable to create PBX channel in DAHDI channel %d\n", p->channel);
		/* failed to create the channel, bail out and report it as an out of order line */
		dahdi_r2_disconnect_call(p, OR2_CAUSE_OUT_OF_ORDER);
		goto dahdi_r2_on_call_accepted_cleanup;
	}
	/* this is an outgoing call, no need to launch the PBX thread, most likely we're in one already */
	ast_verbose("MFC/R2 call has been accepted on forward channel %d\n", p->channel);
	p->subs[SUB_REAL].needringing = 1;
	p->dialing = 0;
	/* chan_dahdi will take care of reading from now on in the PBX thread, tell the library to forget about it */
	openr2_chan_disable_read(r2chan);

dahdi_r2_on_call_accepted_cleanup:
	ast_callid_threadstorage_auto_clean(callid, callid_created);
}

static void dahdi_r2_on_call_answered(openr2_chan_t *r2chan)
{
	struct dahdi_pvt *p = openr2_chan_get_client_data(r2chan);
	ast_verbose("MFC/R2 call has been answered on channel %d\n", openr2_chan_get_number(r2chan));
	p->subs[SUB_REAL].needanswer = 1;
}

static void dahdi_r2_on_call_read(openr2_chan_t *r2chan, const unsigned char *buf, int buflen)
{
	/*ast_debug(1, "Read data from dahdi channel %d\n", openr2_chan_get_number(r2chan));*/
}

static int dahdi_r2_cause_to_ast_cause(openr2_call_disconnect_cause_t cause)
{
	switch (cause) {
	case OR2_CAUSE_BUSY_NUMBER:
		return AST_CAUSE_BUSY;
	case OR2_CAUSE_NETWORK_CONGESTION:
		return AST_CAUSE_CONGESTION;
	case OR2_CAUSE_OUT_OF_ORDER:
		return AST_CAUSE_DESTINATION_OUT_OF_ORDER;
	case OR2_CAUSE_UNALLOCATED_NUMBER:
		return AST_CAUSE_UNREGISTERED;
	case OR2_CAUSE_NO_ANSWER:
		return AST_CAUSE_NO_ANSWER;
	case OR2_CAUSE_NORMAL_CLEARING:
		return AST_CAUSE_NORMAL_CLEARING;
	case OR2_CAUSE_UNSPECIFIED:
	default:
		return AST_CAUSE_NOTDEFINED;
	}
}

static void dahdi_r2_on_call_disconnect(openr2_chan_t *r2chan, openr2_call_disconnect_cause_t cause)
{
	struct dahdi_pvt *p = openr2_chan_get_client_data(r2chan);
	char cause_str[50];
	struct ast_control_pvt_cause_code *cause_code;
	int datalen = sizeof(*cause_code);

	ast_verbose("MFC/R2 call disconnected on channel %d\n", openr2_chan_get_number(r2chan));
	ast_mutex_lock(&p->lock);
	if (!p->owner) {
		ast_mutex_unlock(&p->lock);
		/* no owner, therefore we can't use dahdi_hangup to disconnect, do it right now */
		dahdi_r2_disconnect_call(p, OR2_CAUSE_NORMAL_CLEARING);
		return;
	}

	snprintf(cause_str, sizeof(cause_str), "R2 DISCONNECT (%s)", openr2_proto_get_disconnect_string(cause));
	datalen += strlen(cause_str);
	cause_code = ast_alloca(datalen);
	memset(cause_code, 0, datalen);
	cause_code->ast_cause = dahdi_r2_cause_to_ast_cause(cause);
	ast_copy_string(cause_code->chan_name, ast_channel_name(p->owner), AST_CHANNEL_NAME);
	ast_copy_string(cause_code->code, cause_str, datalen + 1 - sizeof(*cause_code));
	ast_queue_control_data(p->owner, AST_CONTROL_PVT_CAUSE_CODE, cause_code, datalen);
	ast_channel_hangupcause_hash_set(p->owner, cause_code, datalen);
	ast_channel_hangupcause_set(p->owner, cause_code->ast_cause);

	/* when we have an owner we don't call dahdi_r2_disconnect_call here, that will
	   be done in dahdi_hangup */
	if (ast_channel_state(p->owner) == AST_STATE_UP) {
		ast_channel_softhangup_internal_flag_add(p->owner, AST_SOFTHANGUP_DEV);
		ast_mutex_unlock(&p->lock);
	} else if (openr2_chan_get_direction(r2chan) == OR2_DIR_FORWARD) {
		/* being the forward side we must report what happened to the call to whoever requested it */
		switch (cause) {
		case OR2_CAUSE_BUSY_NUMBER:
			p->subs[SUB_REAL].needbusy = 1;
			break;
		case OR2_CAUSE_NETWORK_CONGESTION:
		case OR2_CAUSE_OUT_OF_ORDER:
		case OR2_CAUSE_UNALLOCATED_NUMBER:
		case OR2_CAUSE_NO_ANSWER:
		case OR2_CAUSE_UNSPECIFIED:
		case OR2_CAUSE_NORMAL_CLEARING:
			p->subs[SUB_REAL].needcongestion = 1;
			break;
		default:
			ast_channel_softhangup_internal_flag_add(p->owner, AST_SOFTHANGUP_DEV);
		}
		ast_mutex_unlock(&p->lock);
	} else {
		ast_mutex_unlock(&p->lock);
		/* being the backward side and not UP yet, we only need to request hangup */
		/* TODO: what about doing this same thing when were AST_STATE_UP? */
		ast_queue_hangup_with_cause(p->owner, dahdi_r2_cause_to_ast_cause(cause));
	}
}

static void dahdi_r2_write_log(openr2_log_level_t level, char *logmessage)
{
	switch (level) {
	case OR2_LOG_NOTICE:
		ast_verbose("%s", logmessage);
		break;
	case OR2_LOG_WARNING:
		ast_log(LOG_WARNING, "%s", logmessage);
		break;
	case OR2_LOG_ERROR:
		ast_log(LOG_ERROR, "%s", logmessage);
		break;
	case OR2_LOG_STACK_TRACE:
	case OR2_LOG_MF_TRACE:
	case OR2_LOG_CAS_TRACE:
	case OR2_LOG_DEBUG:
	case OR2_LOG_EX_DEBUG:
		ast_debug(1, "%s", logmessage);
		break;
	default:
		ast_log(LOG_WARNING, "We should handle logging level %d here.\n", level);
		ast_debug(1, "%s", logmessage);
		break;
	}
}

static void dahdi_r2_on_line_blocked(openr2_chan_t *r2chan)
{
	struct dahdi_pvt *p = openr2_chan_get_client_data(r2chan);
	ast_mutex_lock(&p->lock);
	p->remotelyblocked = 1;
	ast_mutex_unlock(&p->lock);
	ast_log(LOG_NOTICE, "Far end blocked on chan %d\n", openr2_chan_get_number(r2chan));
}

static void dahdi_r2_on_line_idle(openr2_chan_t *r2chan)
{
	struct dahdi_pvt *p = openr2_chan_get_client_data(r2chan);
	ast_mutex_lock(&p->lock);
	p->remotelyblocked = 0;
	ast_mutex_unlock(&p->lock);
	ast_log(LOG_NOTICE, "Far end unblocked on chan %d\n", openr2_chan_get_number(r2chan));
}

static void dahdi_r2_on_context_log(openr2_context_t *r2context, openr2_log_level_t level, const char *fmt, va_list ap)
	__attribute__((format (printf, 3, 0)));
static void dahdi_r2_on_context_log(openr2_context_t *r2context, openr2_log_level_t level, const char *fmt, va_list ap)
{
#define CONTEXT_TAG "Context - "
	char logmsg[256];
	char completemsg[sizeof(logmsg) * 2];
	vsnprintf(logmsg, sizeof(logmsg), fmt, ap);
	snprintf(completemsg, sizeof(completemsg), CONTEXT_TAG "%s", logmsg);
	dahdi_r2_write_log(level, completemsg);
#undef CONTEXT_TAG
}

static void dahdi_r2_on_chan_log(openr2_chan_t *r2chan, openr2_log_level_t level, const char *fmt, va_list ap)
	__attribute__((format (printf, 3, 0)));
static void dahdi_r2_on_chan_log(openr2_chan_t *r2chan, openr2_log_level_t level, const char *fmt, va_list ap)
{
#define CHAN_TAG "Chan "
	char logmsg[256];
	char completemsg[sizeof(logmsg) * 2];
	vsnprintf(logmsg, sizeof(logmsg), fmt, ap);
	snprintf(completemsg, sizeof(completemsg), CHAN_TAG "%d - %s", openr2_chan_get_number(r2chan), logmsg);
	dahdi_r2_write_log(level, completemsg);
#undef CHAN_TAG
}

static int dahdi_r2_on_dnis_digit_received(openr2_chan_t *r2chan, char digit)
{
	struct dahdi_pvt *p = openr2_chan_get_client_data(r2chan);
	/* if 'immediate' is set, let's stop requesting DNIS */
	if (p->immediate) {
		return 0;
	}
	p->exten[p->mfcr2_dnis_index] = digit;
	p->rdnis[p->mfcr2_dnis_index] = digit;
	p->mfcr2_dnis_index++;
	p->exten[p->mfcr2_dnis_index] = 0;
	p->rdnis[p->mfcr2_dnis_index] = 0;
	/* if the DNIS is a match and cannot match more, stop requesting DNIS */
	if ((p->mfcr2_dnis_matched ||
	    (ast_exists_extension(NULL, p->context, p->exten, 1, p->cid_num) && (p->mfcr2_dnis_matched = 1))) &&
	    !ast_matchmore_extension(NULL, p->context, p->exten, 1, p->cid_num)) {
		return 0;
	}
	/* otherwise keep going */
	return 1;
}

static void dahdi_r2_on_ani_digit_received(openr2_chan_t *r2chan, char digit)
{
	struct dahdi_pvt *p = openr2_chan_get_client_data(r2chan);
	p->cid_num[p->mfcr2_ani_index] = digit;
	p->cid_name[p->mfcr2_ani_index] = digit;
	p->mfcr2_ani_index++;
	p->cid_num[p->mfcr2_ani_index] = 0;
	p->cid_name[p->mfcr2_ani_index] = 0;
}

static void dahdi_r2_on_billing_pulse_received(openr2_chan_t *r2chan)
{
	ast_verbose("MFC/R2 billing pulse received on channel %d\n", openr2_chan_get_number(r2chan));
}

static openr2_event_interface_t dahdi_r2_event_iface = {
	.on_call_init = dahdi_r2_on_call_init,
	.on_call_offered = dahdi_r2_on_call_offered,
	.on_call_accepted = dahdi_r2_on_call_accepted,
	.on_call_answered = dahdi_r2_on_call_answered,
	.on_call_disconnect = dahdi_r2_on_call_disconnect,
	.on_call_end = dahdi_r2_on_call_end,
	.on_call_read = dahdi_r2_on_call_read,
	.on_hardware_alarm = dahdi_r2_on_hardware_alarm,
	.on_os_error = dahdi_r2_on_os_error,
	.on_protocol_error = dahdi_r2_on_protocol_error,
	.on_line_blocked = dahdi_r2_on_line_blocked,
	.on_line_idle = dahdi_r2_on_line_idle,
	/* cast seems to be needed to get rid of the annoying warning regarding format attribute  */
	.on_context_log = (openr2_handle_context_logging_func)dahdi_r2_on_context_log,
	.on_dnis_digit_received = dahdi_r2_on_dnis_digit_received,
	.on_ani_digit_received = dahdi_r2_on_ani_digit_received,
	/* so far we do nothing with billing pulses */
	.on_billing_pulse_received = dahdi_r2_on_billing_pulse_received
};

static inline int16_t dahdi_r2_alaw_to_linear(uint8_t sample)
{
	return AST_ALAW(sample);
}

static inline uint8_t dahdi_r2_linear_to_alaw(int sample)
{
	return AST_LIN2A(sample);
}

static openr2_transcoder_interface_t dahdi_r2_transcode_iface = {
	dahdi_r2_alaw_to_linear,
	dahdi_r2_linear_to_alaw
};

#endif /* HAVE_OPENR2 */

static void swap_subs(struct dahdi_pvt *p, int a, int b)
{
	int tchan;
	int tinthreeway;
	struct ast_channel *towner;

	ast_debug(1, "Swapping %d and %d\n", a, b);

	tchan = p->subs[a].chan;
	towner = p->subs[a].owner;
	tinthreeway = p->subs[a].inthreeway;

	p->subs[a].chan = p->subs[b].chan;
	p->subs[a].owner = p->subs[b].owner;
	p->subs[a].inthreeway = p->subs[b].inthreeway;

	p->subs[b].chan = tchan;
	p->subs[b].owner = towner;
	p->subs[b].inthreeway = tinthreeway;

	if (p->subs[a].owner)
		ast_channel_set_fd(p->subs[a].owner, 0, p->subs[a].dfd);
	if (p->subs[b].owner)
		ast_channel_set_fd(p->subs[b].owner, 0, p->subs[b].dfd);
	wakeup_sub(p, a);
	wakeup_sub(p, b);
}

static int dahdi_open(char *fn)
{
	int fd;
	int isnum;
	int chan = 0;
	int bs;
	int x;
	isnum = 1;
	for (x = 0; x < strlen(fn); x++) {
		if (!isdigit(fn[x])) {
			isnum = 0;
			break;
		}
	}
	if (isnum) {
		chan = atoi(fn);
		if (chan < 1) {
			ast_log(LOG_WARNING, "Invalid channel number '%s'\n", fn);
			return -1;
		}
		fn = "/dev/dahdi/channel";
	}
	fd = open(fn, O_RDWR | O_NONBLOCK);
	if (fd < 0) {
		ast_log(LOG_WARNING, "Unable to open '%s': %s\n", fn, strerror(errno));
		return -1;
	}
	if (chan) {
		if (ioctl(fd, DAHDI_SPECIFY, &chan)) {
			x = errno;
			close(fd);
			errno = x;
			ast_log(LOG_WARNING, "Unable to specify channel %d: %s\n", chan, strerror(errno));
			return -1;
		}
	}
	bs = READ_SIZE;
	if (ioctl(fd, DAHDI_SET_BLOCKSIZE, &bs) == -1) {
		ast_log(LOG_WARNING, "Unable to set blocksize '%d': %s\n", bs,  strerror(errno));
		x = errno;
		close(fd);
		errno = x;
		return -1;
	}
	return fd;
}

static void dahdi_close(int fd)
{
	if (fd > 0)
		close(fd);
}

static void dahdi_close_sub(struct dahdi_pvt *chan_pvt, int sub_num)
{
	dahdi_close(chan_pvt->subs[sub_num].dfd);
	chan_pvt->subs[sub_num].dfd = -1;
}

#if defined(HAVE_PRI)
static void dahdi_close_pri_fd(struct dahdi_pri *pri, int fd_num)
{
	dahdi_close(pri->pri.fds[fd_num]);
	pri->pri.fds[fd_num] = -1;
}
#endif	/* defined(HAVE_PRI) */

#if defined(HAVE_SS7)
static void dahdi_close_ss7_fd(struct dahdi_ss7 *ss7, int fd_num)
{
	dahdi_close(ss7->ss7.fds[fd_num]);
	ss7->ss7.fds[fd_num] = -1;
}
#endif	/* defined(HAVE_SS7) */

static int dahdi_setlinear(int dfd, int linear)
{
	return ioctl(dfd, DAHDI_SETLINEAR, &linear);
}


static int alloc_sub(struct dahdi_pvt *p, int x)
{
	struct dahdi_bufferinfo bi;
	int res;
	if (p->subs[x].dfd >= 0) {
		ast_log(LOG_WARNING, "%s subchannel of %d already in use\n", subnames[x], p->channel);
		return -1;
	}

	p->subs[x].dfd = dahdi_open("/dev/dahdi/pseudo");
	if (p->subs[x].dfd <= -1) {
		ast_log(LOG_WARNING, "Unable to open pseudo channel: %s\n", strerror(errno));
		return -1;
	}

	res = ioctl(p->subs[x].dfd, DAHDI_GET_BUFINFO, &bi);
	if (!res) {
		bi.txbufpolicy = p->buf_policy;
		bi.rxbufpolicy = p->buf_policy;
		bi.numbufs = p->buf_no;
		res = ioctl(p->subs[x].dfd, DAHDI_SET_BUFINFO, &bi);
		if (res < 0) {
			ast_log(LOG_WARNING, "Unable to set buffer policy on channel %d: %s\n", x, strerror(errno));
		}
	} else
		ast_log(LOG_WARNING, "Unable to check buffer policy on channel %d: %s\n", x, strerror(errno));

	if (ioctl(p->subs[x].dfd, DAHDI_CHANNO, &p->subs[x].chan) == 1) {
		ast_log(LOG_WARNING, "Unable to get channel number for pseudo channel on FD %d: %s\n", p->subs[x].dfd, strerror(errno));
		dahdi_close_sub(p, x);
		p->subs[x].dfd = -1;
		return -1;
	}
	ast_debug(1, "Allocated %s subchannel on FD %d channel %d\n", subnames[x], p->subs[x].dfd, p->subs[x].chan);
	return 0;
}

static int unalloc_sub(struct dahdi_pvt *p, int x)
{
	if (!x) {
		ast_log(LOG_WARNING, "Trying to unalloc the real channel %d?!?\n", p->channel);
		return -1;
	}
	ast_debug(1, "Released sub %d of channel %d\n", x, p->channel);
	dahdi_close_sub(p, x);
	p->subs[x].linear = 0;
	p->subs[x].chan = 0;
	p->subs[x].owner = NULL;
	p->subs[x].inthreeway = 0;
	p->polarity = POLARITY_IDLE;
	memset(&p->subs[x].curconf, 0, sizeof(p->subs[x].curconf));
	return 0;
}

static int digit_to_dtmfindex(char digit)
{
	if (isdigit(digit))
		return DAHDI_TONE_DTMF_BASE + (digit - '0');
	else if (digit >= 'A' && digit <= 'D')
		return DAHDI_TONE_DTMF_A + (digit - 'A');
	else if (digit >= 'a' && digit <= 'd')
		return DAHDI_TONE_DTMF_A + (digit - 'a');
	else if (digit == '*')
		return DAHDI_TONE_DTMF_s;
	else if (digit == '#')
		return DAHDI_TONE_DTMF_p;
	else
		return -1;
}

static int dahdi_digit_begin(struct ast_channel *chan, char digit)
{
	struct dahdi_pvt *pvt;
	int idx;
	int dtmf;
	int res;

	pvt = ast_channel_tech_pvt(chan);

	ast_mutex_lock(&pvt->lock);

	idx = dahdi_get_index(chan, pvt, 0);

	if ((idx != SUB_REAL) || !pvt->owner)
		goto out;

#ifdef HAVE_PRI
	switch (pvt->sig) {
	case SIG_PRI_LIB_HANDLE_CASES:
		res = sig_pri_digit_begin(pvt->sig_pvt, chan, digit);
		if (!res)
			goto out;
		break;
	default:
		break;
	}
#endif
	dtmf = digit_to_dtmfindex(digit);
	if (dtmf == -1) {
		/* Not a valid DTMF digit */
		goto out;
	}

	if (pvt->pulse || ioctl(pvt->subs[SUB_REAL].dfd, DAHDI_SENDTONE, &dtmf)) {
		char dial_str[] = { 'T', digit, '\0' };

		res = dahdi_dial_str(pvt, DAHDI_DIAL_OP_APPEND, dial_str);
		if (!res) {
			pvt->dialing = 1;
		}
	} else {
		pvt->dialing = 1;
		pvt->begindigit = digit;

		/* Flush the write buffer in DAHDI to start sending the digit immediately. */
		dtmf = DAHDI_FLUSH_WRITE;
		res = ioctl(pvt->subs[SUB_REAL].dfd, DAHDI_FLUSH, &dtmf);
		if (res) {
			ast_log(LOG_WARNING, "Unable to flush the DAHDI write buffer to send DTMF on channel %d: %s\n",
				pvt->channel, strerror(errno));
		}

		ast_debug(1, "Channel %s started VLDTMF digit '%c'\n",
			ast_channel_name(chan), digit);
	}

out:
	ast_mutex_unlock(&pvt->lock);

	return 0;
}

static int dahdi_digit_end(struct ast_channel *chan, char digit, unsigned int duration)
{
	struct dahdi_pvt *pvt;
	int res = 0;
	int idx;
	int x;

	pvt = ast_channel_tech_pvt(chan);

	ast_mutex_lock(&pvt->lock);

	idx = dahdi_get_index(chan, pvt, 0);

	if ((idx != SUB_REAL) || !pvt->owner || pvt->pulse)
		goto out;

#ifdef HAVE_PRI
	/* This means that the digit was already sent via PRI signalling */
	if (dahdi_sig_pri_lib_handles(pvt->sig) && !pvt->begindigit) {
		goto out;
	}
#endif

	if (pvt->begindigit) {
		x = -1;
		ast_debug(1, "Channel %s ending VLDTMF digit '%c'\n",
			ast_channel_name(chan), digit);
		res = ioctl(pvt->subs[SUB_REAL].dfd, DAHDI_SENDTONE, &x);
		pvt->dialing = 0;
		pvt->begindigit = 0;
	}

out:
	ast_mutex_unlock(&pvt->lock);

	return res;
}

static const char * const events[] = {
	"No event",
	"On hook",
	"Ring/Answered",
	"Wink/Flash",
	"Alarm",
	"No more alarm",
	"HDLC Abort",
	"HDLC Overrun",
	"HDLC Bad FCS",
	"Dial Complete",
	"Ringer On",
	"Ringer Off",
	"Hook Transition Complete",
	"Bits Changed",
	"Pulse Start",
	"Timer Expired",
	"Timer Ping",
	"Polarity Reversal",
	"Ring Begin",
};

static struct {
	int alarm;
	char *name;
} alarms[] = {
	{ DAHDI_ALARM_RED, "Red Alarm" },
	{ DAHDI_ALARM_YELLOW, "Yellow Alarm" },
	{ DAHDI_ALARM_BLUE, "Blue Alarm" },
	{ DAHDI_ALARM_RECOVER, "Recovering" },
	{ DAHDI_ALARM_LOOPBACK, "Loopback" },
	{ DAHDI_ALARM_NOTOPEN, "Not Open" },
	{ DAHDI_ALARM_NONE, "None" },
};

static char *alarm2str(int alm)
{
	int x;
	for (x = 0; x < ARRAY_LEN(alarms); x++) {
		if (alarms[x].alarm & alm)
			return alarms[x].name;
	}
	return alm ? "Unknown Alarm" : "No Alarm";
}

static const char *event2str(int event)
{
	static char buf[256];
	if ((event > -1) && (event < (ARRAY_LEN(events))) )
		return events[event];
	sprintf(buf, "Event %d", event); /* safe */
	return buf;
}

static char *dahdi_sig2str(int sig)
{
	static char buf[256];
	switch (sig) {
	case SIG_EM:
		return "E & M Immediate";
	case SIG_EMWINK:
		return "E & M Wink";
	case SIG_EM_E1:
		return "E & M E1";
	case SIG_FEATD:
		return "Feature Group D (DTMF)";
	case SIG_FEATDMF:
		return "Feature Group D (MF)";
	case SIG_FEATDMF_TA:
		return "Feature Group D (MF) Tandem Access";
	case SIG_FEATB:
		return "Feature Group B (MF)";
	case SIG_E911:
		return "E911 (MF)";
	case SIG_FGC_CAMA:
		return "FGC/CAMA (Dialpulse)";
	case SIG_FGC_CAMAMF:
		return "FGC/CAMA (MF)";
	case SIG_FXSLS:
		return "FXS Loopstart";
	case SIG_FXSGS:
		return "FXS Groundstart";
	case SIG_FXSKS:
		return "FXS Kewlstart";
	case SIG_FXOLS:
		return "FXO Loopstart";
	case SIG_FXOGS:
		return "FXO Groundstart";
	case SIG_FXOKS:
		return "FXO Kewlstart";
	case SIG_PRI:
		return "ISDN PRI";
	case SIG_BRI:
		return "ISDN BRI Point to Point";
	case SIG_BRI_PTMP:
		return "ISDN BRI Point to MultiPoint";
	case SIG_SS7:
		return "SS7";
	case SIG_MFCR2:
		return "MFC/R2";
	case SIG_SF:
		return "SF (Tone) Immediate";
	case SIG_SFWINK:
		return "SF (Tone) Wink";
	case SIG_SF_FEATD:
		return "SF (Tone) with Feature Group D (DTMF)";
	case SIG_SF_FEATDMF:
		return "SF (Tone) with Feature Group D (MF)";
	case SIG_SF_FEATB:
		return "SF (Tone) with Feature Group B (MF)";
	case 0:
		return "Pseudo";
	default:
		snprintf(buf, sizeof(buf), "Unknown signalling %d", sig);
		return buf;
	}
}

#define sig2str dahdi_sig2str

static int conf_add(struct dahdi_pvt *p, struct dahdi_subchannel *c, int idx, int slavechannel)
{
	/* If the conference already exists, and we're already in it
	   don't bother doing anything */
	struct dahdi_confinfo zi;

	memset(&zi, 0, sizeof(zi));
	zi.chan = 0;

	if (slavechannel > 0) {
		/* If we have only one slave, do a digital mon */
		zi.confmode = DAHDI_CONF_DIGITALMON;
		zi.confno = slavechannel;
	} else {
		if (!idx) {
			/* Real-side and pseudo-side both participate in conference */
			zi.confmode = DAHDI_CONF_REALANDPSEUDO | DAHDI_CONF_TALKER | DAHDI_CONF_LISTENER |
				DAHDI_CONF_PSEUDO_TALKER | DAHDI_CONF_PSEUDO_LISTENER;
		} else
			zi.confmode = DAHDI_CONF_CONF | DAHDI_CONF_TALKER | DAHDI_CONF_LISTENER;
		zi.confno = p->confno;
	}
	if ((zi.confno == c->curconf.confno) && (zi.confmode == c->curconf.confmode))
		return 0;
	if (c->dfd < 0)
		return 0;
	if (ioctl(c->dfd, DAHDI_SETCONF, &zi)) {
		ast_log(LOG_WARNING, "Failed to add %d to conference %d/%d: %s\n", c->dfd, zi.confmode, zi.confno, strerror(errno));
		return -1;
	}
	if (slavechannel < 1) {
		p->confno = zi.confno;
	}
	c->curconf = zi;
	ast_debug(1, "Added %d to conference %d/%d\n", c->dfd, c->curconf.confmode, c->curconf.confno);
	return 0;
}

static int isourconf(struct dahdi_pvt *p, struct dahdi_subchannel *c)
{
	/* If they're listening to our channel, they're ours */
	if ((p->channel == c->curconf.confno) && (c->curconf.confmode == DAHDI_CONF_DIGITALMON))
		return 1;
	/* If they're a talker on our (allocated) conference, they're ours */
	if ((p->confno > 0) && (p->confno == c->curconf.confno) && (c->curconf.confmode & DAHDI_CONF_TALKER))
		return 1;
	return 0;
}

static int conf_del(struct dahdi_pvt *p, struct dahdi_subchannel *c, int idx)
{
	struct dahdi_confinfo zi;
	if (/* Can't delete if there's no dfd */
		(c->dfd < 0) ||
		/* Don't delete from the conference if it's not our conference */
		!isourconf(p, c)
		/* Don't delete if we don't think it's conferenced at all (implied) */
		) return 0;
	memset(&zi, 0, sizeof(zi));
	if (ioctl(c->dfd, DAHDI_SETCONF, &zi)) {
		ast_log(LOG_WARNING, "Failed to drop %d from conference %d/%d: %s\n", c->dfd, c->curconf.confmode, c->curconf.confno, strerror(errno));
		return -1;
	}
	ast_debug(1, "Removed %d from conference %d/%d\n", c->dfd, c->curconf.confmode, c->curconf.confno);
	memcpy(&c->curconf, &zi, sizeof(c->curconf));
	return 0;
}

static int isslavenative(struct dahdi_pvt *p, struct dahdi_pvt **out)
{
	int x;
	int useslavenative;
	struct dahdi_pvt *slave = NULL;
	/* Start out optimistic */
	useslavenative = 1;
	/* Update conference state in a stateless fashion */
	for (x = 0; x < 3; x++) {
		/* Any three-way calling makes slave native mode *definitely* out
		   of the question */
		if ((p->subs[x].dfd > -1) && p->subs[x].inthreeway)
			useslavenative = 0;
	}
	/* If we don't have any 3-way calls, check to see if we have
	   precisely one slave */
	if (useslavenative) {
		for (x = 0; x < MAX_SLAVES; x++) {
			if (p->slaves[x]) {
				if (slave) {
					/* Whoops already have a slave!  No
					   slave native and stop right away */
					slave = NULL;
					useslavenative = 0;
					break;
				} else {
					/* We have one slave so far */
					slave = p->slaves[x];
				}
			}
		}
	}
	/* If no slave, slave native definitely out */
	if (!slave)
		useslavenative = 0;
	else if (slave->law != p->law) {
		useslavenative = 0;
		slave = NULL;
	}
	if (out)
		*out = slave;
	return useslavenative;
}

static int reset_conf(struct dahdi_pvt *p)
{
	p->confno = -1;
	memset(&p->subs[SUB_REAL].curconf, 0, sizeof(p->subs[SUB_REAL].curconf));
	if (p->subs[SUB_REAL].dfd > -1) {
		struct dahdi_confinfo zi;

		memset(&zi, 0, sizeof(zi));
		if (ioctl(p->subs[SUB_REAL].dfd, DAHDI_SETCONF, &zi))
			ast_log(LOG_WARNING, "Failed to reset conferencing on channel %d: %s\n", p->channel, strerror(errno));
	}
	return 0;
}

void dahdi_conf_update(struct dahdi_pvt *p)
{
	int needconf = 0;
	int x;
	int useslavenative;
	struct dahdi_pvt *slave = NULL;

	useslavenative = isslavenative(p, &slave);
	/* Start with the obvious, general stuff */
	for (x = 0; x < 3; x++) {
		/* Look for three way calls */
		if ((p->subs[x].dfd > -1) && p->subs[x].inthreeway) {
			conf_add(p, &p->subs[x], x, 0);
			needconf++;
		} else {
			conf_del(p, &p->subs[x], x);
		}
	}
	/* If we have a slave, add him to our conference now. or DAX
	   if this is slave native */
	for (x = 0; x < MAX_SLAVES; x++) {
		if (p->slaves[x]) {
			if (useslavenative)
				conf_add(p, &p->slaves[x]->subs[SUB_REAL], SUB_REAL, GET_CHANNEL(p));
			else {
				conf_add(p, &p->slaves[x]->subs[SUB_REAL], SUB_REAL, 0);
				needconf++;
			}
		}
	}
	/* If we're supposed to be in there, do so now */
	if (p->inconference && !p->subs[SUB_REAL].inthreeway) {
		if (useslavenative)
			conf_add(p, &p->subs[SUB_REAL], SUB_REAL, GET_CHANNEL(slave));
		else {
			conf_add(p, &p->subs[SUB_REAL], SUB_REAL, 0);
			needconf++;
		}
	}
	/* If we have a master, add ourselves to his conference */
	if (p->master) {
		if (isslavenative(p->master, NULL)) {
			conf_add(p->master, &p->subs[SUB_REAL], SUB_REAL, GET_CHANNEL(p->master));
		} else {
			conf_add(p->master, &p->subs[SUB_REAL], SUB_REAL, 0);
		}
	}
	if (!needconf) {
		/* Nobody is left (or should be left) in our conference.
		   Kill it. */
		p->confno = -1;
	}
	ast_debug(1, "Updated conferencing on %d, with %d conference users\n", p->channel, needconf);
}

void dahdi_ec_enable(struct dahdi_pvt *p)
{
	int res;
	if (!p)
		return;
	if (p->echocanon) {
		ast_debug(1, "Echo cancellation already on\n");
		return;
	}
	if (p->digital) {
		ast_debug(1, "Echo cancellation isn't required on digital connection\n");
		return;
	}
	if (p->echocancel.head.tap_length) {
#if defined(HAVE_PRI) || defined(HAVE_SS7)
		switch (p->sig) {
#if defined(HAVE_PRI)
		case SIG_PRI_LIB_HANDLE_CASES:
			if (((struct sig_pri_chan *) p->sig_pvt)->no_b_channel) {
				/*
				 * PRI nobch pseudo channel.  Does not need ec anyway.
				 * Does not handle ioctl(DAHDI_AUDIOMODE)
				 */
				return;
			}
			/* Fall through */
#endif	/* defined(HAVE_PRI) */
#if defined(HAVE_SS7)
		case SIG_SS7:
#endif	/* defined(HAVE_SS7) */
			{
				int x = 1;

				res = ioctl(p->subs[SUB_REAL].dfd, DAHDI_AUDIOMODE, &x);
				if (res)
					ast_log(LOG_WARNING,
						"Unable to enable audio mode on channel %d (%s)\n",
						p->channel, strerror(errno));
			}
			break;
		default:
			break;
		}
#endif	/* defined(HAVE_PRI) || defined(HAVE_SS7) */
		res = ioctl(p->subs[SUB_REAL].dfd, DAHDI_ECHOCANCEL_PARAMS, &p->echocancel);
		if (res) {
			ast_log(LOG_WARNING, "Unable to enable echo cancellation on channel %d (%s)\n", p->channel, strerror(errno));
		} else {
			p->echocanon = 1;
			ast_debug(1, "Enabled echo cancellation on channel %d\n", p->channel);
		}
	} else
		ast_debug(1, "No echo cancellation requested\n");
}

static void dahdi_train_ec(struct dahdi_pvt *p)
{
	int x;
	int res;

	if (p && p->echocanon && p->echotraining) {
		x = p->echotraining;
		res = ioctl(p->subs[SUB_REAL].dfd, DAHDI_ECHOTRAIN, &x);
		if (res)
			ast_log(LOG_WARNING, "Unable to request echo training on channel %d: %s\n", p->channel, strerror(errno));
		else
			ast_debug(1, "Engaged echo training on channel %d\n", p->channel);
	} else {
		ast_debug(1, "No echo training requested\n");
	}
}

void dahdi_ec_disable(struct dahdi_pvt *p)
{
	int res;

	if (p->echocanon) {
		struct dahdi_echocanparams ecp = { .tap_length = 0 };

		res = ioctl(p->subs[SUB_REAL].dfd, DAHDI_ECHOCANCEL_PARAMS, &ecp);

		if (res)
			ast_log(LOG_WARNING, "Unable to disable echo cancellation on channel %d: %s\n", p->channel, strerror(errno));
		else
			ast_debug(1, "Disabled echo cancellation on channel %d\n", p->channel);
	}

	p->echocanon = 0;
}

static int set_hwgain(int fd, float gain, int tx_direction)
{
	struct dahdi_hwgain hwgain;

	hwgain.newgain = gain * 10.0;
	hwgain.tx = tx_direction;
	return ioctl(fd, DAHDI_SET_HWGAIN, &hwgain) < 0;
}

/* perform a dynamic range compression transform on the given sample */
static int drc_sample(int sample, float drc)
{
	float neg;
	float shallow, steep;
	float max = SHRT_MAX;

	neg = (sample < 0 ? -1 : 1);
	steep = drc*sample;
	shallow = neg*(max-max/drc)+(float)sample/drc;
	if (fabsf(steep) < fabsf(shallow)) {
		sample = steep;
	}
	else {
		sample = shallow;
	}

	return sample;
}


static void fill_txgain(struct dahdi_gains *g, float gain, float drc, int law)
{
	int j;
	int k;

	float linear_gain = pow(10.0, gain / 20.0);

	switch (law) {
	case DAHDI_LAW_ALAW:
		for (j = 0; j < ARRAY_LEN(g->txgain); j++) {
			if (gain || drc) {
				k = AST_ALAW(j);
				if (drc) {
					k = drc_sample(k, drc);
				}
				k = (float)k * linear_gain;
				if (k > 32767) {
					k = 32767;
				} else if (k < -32768) {
					k = -32768;
				}
				g->txgain[j] = AST_LIN2A(k);
			} else {
				g->txgain[j] = j;
			}
		}
		break;
	case DAHDI_LAW_MULAW:
		for (j = 0; j < ARRAY_LEN(g->txgain); j++) {
			if (gain || drc) {
				k = AST_MULAW(j);
				if (drc) {
					k = drc_sample(k, drc);
				}
				k = (float)k * linear_gain;
				if (k > 32767) {
					k = 32767;
				} else if (k < -32768) {
					k = -32768;
				}
				g->txgain[j] = AST_LIN2MU(k);

			} else {
				g->txgain[j] = j;
			}
		}
		break;
	}
}

static void fill_rxgain(struct dahdi_gains *g, float gain, float drc, int law)
{
	int j;
	int k;
	float linear_gain = pow(10.0, gain / 20.0);

	switch (law) {
	case DAHDI_LAW_ALAW:
		for (j = 0; j < ARRAY_LEN(g->rxgain); j++) {
			if (gain || drc) {
				k = AST_ALAW(j);
				if (drc) {
					k = drc_sample(k, drc);
				}
				k = (float)k * linear_gain;
				if (k > 32767) {
					k = 32767;
				} else if (k < -32768) {
					k = -32768;
				}
				g->rxgain[j] = AST_LIN2A(k);
			} else {
				g->rxgain[j] = j;
			}
		}
		break;
	case DAHDI_LAW_MULAW:
		for (j = 0; j < ARRAY_LEN(g->rxgain); j++) {
			if (gain || drc) {
				k = AST_MULAW(j);
				if (drc) {
					k = drc_sample(k, drc);
				}
				k = (float)k * linear_gain;
				if (k > 32767) {
					k = 32767;
				} else if (k < -32768) {
					k = -32768;
				}
				g->rxgain[j] = AST_LIN2MU(k);
			} else {
				g->rxgain[j] = j;
			}
		}
		break;
	}
}

static int set_actual_txgain(int fd, float gain, float drc, int law)
{
	struct dahdi_gains g;
	int res;

	memset(&g, 0, sizeof(g));
	res = ioctl(fd, DAHDI_GETGAINS, &g);
	if (res) {
		ast_debug(1, "Failed to read gains: %s\n", strerror(errno));
		return res;
	}

	fill_txgain(&g, gain, drc, law);

	return ioctl(fd, DAHDI_SETGAINS, &g);
}

static int set_actual_rxgain(int fd, float gain, float drc, int law)
{
	struct dahdi_gains g;
	int res;

	memset(&g, 0, sizeof(g));
	res = ioctl(fd, DAHDI_GETGAINS, &g);
	if (res) {
		ast_debug(1, "Failed to read gains: %s\n", strerror(errno));
		return res;
	}

	fill_rxgain(&g, gain, drc, law);

	return ioctl(fd, DAHDI_SETGAINS, &g);
}

static int set_actual_gain(int fd, float rxgain, float txgain, float rxdrc, float txdrc, int law)
{
	return set_actual_txgain(fd, txgain, txdrc, law) | set_actual_rxgain(fd, rxgain, rxdrc, law);
}

static int bump_gains(struct dahdi_pvt *p)
{
	int res;

	/* Bump receive gain by value stored in cid_rxgain */
	res = set_actual_gain(p->subs[SUB_REAL].dfd, p->rxgain + p->cid_rxgain, p->txgain, p->rxdrc, p->txdrc, p->law);
	if (res) {
		ast_log(LOG_WARNING, "Unable to bump gain: %s\n", strerror(errno));
		return -1;
	}

	return 0;
}

static int restore_gains(struct dahdi_pvt *p)
{
	int res;

	res = set_actual_gain(p->subs[SUB_REAL].dfd, p->rxgain, p->txgain, p->rxdrc, p->txdrc, p->law);
	if (res) {
		ast_log(LOG_WARNING, "Unable to restore gains: %s\n", strerror(errno));
		return -1;
	}

	return 0;
}

static inline int dahdi_set_hook(int fd, int hs)
{
	int x, res;

	x = hs;
	res = ioctl(fd, DAHDI_HOOK, &x);

	if (res < 0) {
		if (errno == EINPROGRESS)
			return 0;
		ast_log(LOG_WARNING, "DAHDI hook failed returned %d (trying %d): %s\n", res, hs, strerror(errno));
		/* will expectedly fail if phone is off hook during operation, such as during a restart */
	}

	return res;
}

static inline int dahdi_confmute(struct dahdi_pvt *p, int muted)
{
	int x, res;

	x = muted;
#if defined(HAVE_PRI) || defined(HAVE_SS7)
	switch (p->sig) {
#if defined(HAVE_PRI)
	case SIG_PRI_LIB_HANDLE_CASES:
		if (((struct sig_pri_chan *) p->sig_pvt)->no_b_channel) {
			/* PRI nobch pseudo channel.  Does not handle ioctl(DAHDI_AUDIOMODE) */
			break;
		}
		/* Fall through */
#endif	/* defined(HAVE_PRI) */
#if defined(HAVE_SS7)
	case SIG_SS7:
#endif	/* defined(HAVE_SS7) */
		{
			int y = 1;

			res = ioctl(p->subs[SUB_REAL].dfd, DAHDI_AUDIOMODE, &y);
			if (res)
				ast_log(LOG_WARNING, "Unable to set audio mode on %d: %s\n",
					p->channel, strerror(errno));
		}
		break;
	default:
		break;
	}
#endif	/* defined(HAVE_PRI) || defined(HAVE_SS7) */
	res = ioctl(p->subs[SUB_REAL].dfd, DAHDI_CONFMUTE, &x);
	if (res < 0)
		ast_log(LOG_WARNING, "DAHDI confmute(%d) failed on channel %d: %s\n", muted, p->channel, strerror(errno));
	return res;
}

static int save_conference(struct dahdi_pvt *p)
{
	struct dahdi_confinfo c;
	int res;
	if (p->saveconf.confmode) {
		ast_log(LOG_WARNING, "Can't save conference -- already in use\n");
		return -1;
	}
	p->saveconf.chan = 0;
	res = ioctl(p->subs[SUB_REAL].dfd, DAHDI_GETCONF, &p->saveconf);
	if (res) {
		ast_log(LOG_WARNING, "Unable to get conference info: %s\n", strerror(errno));
		p->saveconf.confmode = 0;
		return -1;
	}
	memset(&c, 0, sizeof(c));
	c.confmode = DAHDI_CONF_NORMAL;
	res = ioctl(p->subs[SUB_REAL].dfd, DAHDI_SETCONF, &c);
	if (res) {
		ast_log(LOG_WARNING, "Unable to set conference info: %s\n", strerror(errno));
		return -1;
	}
	ast_debug(1, "Disabled conferencing\n");
	return 0;
}

static int restore_conference(struct dahdi_pvt *p)
{
	int res;
	if (p->saveconf.confmode) {
		res = ioctl(p->subs[SUB_REAL].dfd, DAHDI_SETCONF, &p->saveconf);
		p->saveconf.confmode = 0;
		if (res) {
			ast_log(LOG_WARNING, "Unable to restore conference info: %s\n", strerror(errno));
			return -1;
		}
		ast_debug(1, "Restored conferencing\n");
	}
	return 0;
}

static int send_cwcidspill(struct dahdi_pvt *p)
{
	p->callwaitcas = 0;
	p->cidcwexpire = 0;
	p->cid_suppress_expire = 0;
	if (!(p->cidspill = ast_malloc(MAX_CALLERID_SIZE)))
		return -1;
	p->cidlen = ast_callerid_callwaiting_generate(p->cidspill, p->callwait_name, p->callwait_num, AST_LAW(p));
	/* Make sure we account for the end */
	p->cidlen += READ_SIZE * 4;
	p->cidpos = 0;
	send_callerid(p);
	ast_verb(3, "CPE supports Call Waiting Caller*ID.  Sending '%s/%s'\n", p->callwait_name, p->callwait_num);
	return 0;
}

static int has_voicemail(struct dahdi_pvt *p)
{
	int new_msgs;
	RAII_VAR(struct stasis_message *, mwi_message, NULL, ao2_cleanup);

	/* A manual MWI disposition has been requested, use that instead
	 * if this is for sending the new MWI indication. */
	if (p->mwioverride_active) {
		/* We don't clear p->mwioverride_active automatically,
		 * because otherwise do_monitor would just change it back to the way it was.
		 * We need to keep the override active until explicitly disabled by the user,
		 * so that we can keep returning the correct answer in subsequent calls to do_monitor. */
		ast_debug(6, "MWI manual override active on channel %d: pretending that it should be %s\n",
			p->channel, p->mwioverride_disposition ? "active" : "inactive");
		return p->mwioverride_disposition;
	}

	mwi_message = stasis_cache_get(ast_mwi_state_cache(), ast_mwi_state_type(), p->mailbox);
	if (mwi_message) {
		struct ast_mwi_state *mwi_state = stasis_message_data(mwi_message);
		new_msgs = mwi_state->new_msgs;
	} else {
		new_msgs = ast_app_has_voicemail(p->mailbox, NULL);
	}

	return new_msgs;
}



static int send_callerid(struct dahdi_pvt *p)
{
	/* Assumes spill in p->cidspill, p->cidlen in length and we're p->cidpos into it */
	int res;
	/* Take out of linear mode if necessary */
	if (p->subs[SUB_REAL].linear) {
		p->subs[SUB_REAL].linear = 0;
		dahdi_setlinear(p->subs[SUB_REAL].dfd, 0);
	}
	while (p->cidpos < p->cidlen) {
		res = write(p->subs[SUB_REAL].dfd, p->cidspill + p->cidpos, p->cidlen - p->cidpos);
		ast_debug(4, "writing callerid at pos %d of %d, res = %d\n", p->cidpos, p->cidlen, res);
		if (res < 0) {
			if (errno == EAGAIN)
				return 0;
			else {
				ast_log(LOG_WARNING, "write failed: %s\n", strerror(errno));
				return -1;
			}
		}
		if (!res)
			return 0;
		p->cidpos += res;
	}
	p->cid_suppress_expire = CALLWAITING_SUPPRESS_SAMPLES;
	ast_free(p->cidspill);
	p->cidspill = NULL;
	if (p->callwaitcas) {
		/* Wait for CID/CW to expire */
		p->cidcwexpire = CIDCW_EXPIRE_SAMPLES;
		p->cid_suppress_expire = p->cidcwexpire;
	} else
		restore_conference(p);
	return 0;
}

static int dahdi_callwait(struct ast_channel *ast)
{
	struct dahdi_pvt *p = ast_channel_tech_pvt(ast);

	p->callwaitingrepeat = CALLWAITING_REPEAT_SAMPLES;
	if (p->cidspill) {
		ast_log(LOG_WARNING, "Spill already exists?!?\n");
		ast_free(p->cidspill);
	}

	/*
	 * SAS: Subscriber Alert Signal, 440Hz for 300ms
	 * CAS: CPE Alert Signal, 2130Hz * 2750Hz sine waves
	 */
	if (!(p->cidspill = ast_malloc(2400 /* SAS */ + 680 /* CAS */ + READ_SIZE * 4)))
		return -1;
	save_conference(p);
	/* Silence */
	memset(p->cidspill, 0x7f, 2400 + 600 + READ_SIZE * 4);
	if (!p->callwaitrings && p->callwaitingcallerid) {
		ast_gen_cas(p->cidspill, 1, 2400 + 680, AST_LAW(p));
		p->callwaitcas = 1;
		p->cidlen = 2400 + 680 + READ_SIZE * 4;
	} else {
		ast_gen_cas(p->cidspill, 1, 2400, AST_LAW(p));
		p->callwaitcas = 0;
		p->cidlen = 2400 + READ_SIZE * 4;
	}
	p->cidpos = 0;
	send_callerid(p);

	return 0;
}

static int dahdi_call(struct ast_channel *ast, const char *rdest, int timeout)
{
	struct dahdi_pvt *p = ast_channel_tech_pvt(ast);
	int x, res, mysig;
	char *dest;
	AST_DECLARE_APP_ARGS(args,
		AST_APP_ARG(group);	/* channel/group token */
		AST_APP_ARG(ext);	/* extension token */
		//AST_APP_ARG(opts);	/* options token */
		AST_APP_ARG(other);	/* Any remaining unused arguments */
	);

	ast_mutex_lock(&p->lock);
	ast_copy_string(p->dialdest, rdest, sizeof(p->dialdest));

	/* Split the dialstring */
	dest = ast_strdupa(rdest);
	AST_NONSTANDARD_APP_ARGS(args, dest, '/');
	if (!args.ext) {
		args.ext = "";
	}

#if defined(HAVE_PRI)
	if (dahdi_sig_pri_lib_handles(p->sig)) {
		char *subaddr;

		sig_pri_extract_called_num_subaddr(p->sig_pvt, rdest, p->exten, sizeof(p->exten));

		/* Remove any subaddress for uniformity with incoming calls. */
		subaddr = strchr(p->exten, ':');
		if (subaddr) {
			*subaddr = '\0';
		}
	} else
#endif	/* defined(HAVE_PRI) */
	{
		ast_copy_string(p->exten, args.ext, sizeof(p->exten));
	}

	if ((ast_channel_state(ast) == AST_STATE_BUSY)) {
		p->subs[SUB_REAL].needbusy = 1;
		ast_mutex_unlock(&p->lock);
		return 0;
	}
	if ((ast_channel_state(ast) != AST_STATE_DOWN) && (ast_channel_state(ast) != AST_STATE_RESERVED)) {
		ast_log(LOG_WARNING, "dahdi_call called on %s, neither down nor reserved\n", ast_channel_name(ast));
		ast_mutex_unlock(&p->lock);
		return -1;
	}
	p->waitingfordt.tv_sec = 0;
	p->dialednone = 0;
	if ((p->radio || (p->oprmode < 0)))  /* if a radio channel, up immediately */
	{
		/* Special pseudo -- automatically up */
		ast_setstate(ast, AST_STATE_UP);
		ast_mutex_unlock(&p->lock);
		return 0;
	}
	x = DAHDI_FLUSH_READ | DAHDI_FLUSH_WRITE;
	res = ioctl(p->subs[SUB_REAL].dfd, DAHDI_FLUSH, &x);
	if (res)
		ast_log(LOG_WARNING, "Unable to flush input on channel %d: %s\n", p->channel, strerror(errno));
	p->outgoing = 1;

	if (IS_DIGITAL(ast_channel_transfercapability(ast))){
		set_actual_gain(p->subs[SUB_REAL].dfd, 0, 0, p->rxdrc, p->txdrc, p->law);
	} else {
		set_actual_gain(p->subs[SUB_REAL].dfd, p->rxgain, p->txgain, p->rxdrc, p->txdrc, p->law);
	}

#ifdef HAVE_PRI
	if (dahdi_sig_pri_lib_handles(p->sig)) {
		res = sig_pri_call(p->sig_pvt, ast, rdest, timeout,
			(p->law == DAHDI_LAW_ALAW) ? PRI_LAYER_1_ALAW : PRI_LAYER_1_ULAW);
		ast_mutex_unlock(&p->lock);
		return res;
	}
#endif

#if defined(HAVE_SS7)
	if (p->sig == SIG_SS7) {
		res = sig_ss7_call(p->sig_pvt, ast, rdest);
		ast_mutex_unlock(&p->lock);
		return res;
	}
#endif	/* defined(HAVE_SS7) */

	/* If this is analog signalling we can exit here */
	if (dahdi_analog_lib_handles(p->sig, p->radio, p->oprmode)) {
		p->callwaitrings = 0;
		res = analog_call(p->sig_pvt, ast, rdest, timeout);
		ast_mutex_unlock(&p->lock);
		return res;
	}

	mysig = p->outsigmod > -1 ? p->outsigmod : p->sig;
	switch (mysig) {
	case 0:
		/* Special pseudo -- automatically up*/
		ast_setstate(ast, AST_STATE_UP);
		break;
	case SIG_MFCR2:
		break;
	default:
		ast_debug(1, "not yet implemented\n");
		ast_mutex_unlock(&p->lock);
		return -1;
	}

#ifdef HAVE_OPENR2
	if (p->mfcr2) {
		openr2_calling_party_category_t chancat;
		int callres = 0;
		char *c, *l;

		/* We'll get it in a moment -- but use dialdest to store pre-setup_ack digits */
		p->dialdest[0] = '\0';

		c = args.ext;
		if (!p->hidecallerid) {
			l = ast_channel_connected(ast)->id.number.valid ? ast_channel_connected(ast)->id.number.str : NULL;
		} else {
			l = NULL;
		}
		if (strlen(c) < p->stripmsd) {
			ast_log(LOG_WARNING, "Number '%s' is shorter than stripmsd (%d)\n", c, p->stripmsd);
			ast_mutex_unlock(&p->lock);
			return -1;
		}
		p->dialing = 1;
		chancat = dahdi_r2_get_channel_category(ast);
		callres = openr2_chan_make_call(p->r2chan, l, (c + p->stripmsd), chancat);
		if (-1 == callres) {
			ast_mutex_unlock(&p->lock);
			ast_log(LOG_ERROR, "unable to make new MFC/R2 call!\n");
			return -1;
		}
		p->mfcr2_call_accepted = 0;
		p->mfcr2_progress_sent = 0;
		ast_setstate(ast, AST_STATE_DIALING);
	}
#endif /* HAVE_OPENR2 */
	ast_mutex_unlock(&p->lock);
	return 0;
}

/*!
 * \internal
 * \brief Insert the given chan_dahdi interface structure into the interface list.
 * \since 1.8
 *
 * \param pvt chan_dahdi private interface structure to insert.
 *
 * \details
 * The interface list is a doubly linked list sorted by the chan_dahdi channel number.
 * Any duplicates are inserted after the existing entries.
 *
 * \note The new interface must not already be in the list.
 */
static void dahdi_iflist_insert(struct dahdi_pvt *pvt)
{
	struct dahdi_pvt *cur;

	pvt->which_iflist = DAHDI_IFLIST_MAIN;

	/* Find place in middle of list for the new interface. */
	for (cur = iflist; cur; cur = cur->next) {
		if (pvt->channel < cur->channel) {
			/* New interface goes before the current interface. */
			pvt->prev = cur->prev;
			pvt->next = cur;
			if (cur->prev) {
				/* Insert into the middle of the list. */
				cur->prev->next = pvt;
			} else {
				/* Insert at head of list. */
				iflist = pvt;
			}
			cur->prev = pvt;
			return;
		}
	}

	/* New interface goes onto the end of the list */
	pvt->prev = ifend;
	pvt->next = NULL;
	if (ifend) {
		ifend->next = pvt;
	}
	ifend = pvt;
	if (!iflist) {
		/* List was empty */
		iflist = pvt;
	}
}

/*!
 * \internal
 * \brief Extract the given chan_dahdi interface structure from the interface list.
 * \since 1.8
 *
 * \param pvt chan_dahdi private interface structure to extract.
 *
 * \note
 * The given interface structure can be either in the interface list or a stand alone
 * structure that has not been put in the list if the next and prev pointers are NULL.
 */
static void dahdi_iflist_extract(struct dahdi_pvt *pvt)
{
	/* Extract from the forward chain. */
	if (pvt->prev) {
		pvt->prev->next = pvt->next;
	} else if (iflist == pvt) {
		/* Node is at the head of the list. */
		iflist = pvt->next;
	}

	/* Extract from the reverse chain. */
	if (pvt->next) {
		pvt->next->prev = pvt->prev;
	} else if (ifend == pvt) {
		/* Node is at the end of the list. */
		ifend = pvt->prev;
	}

	/* Node is no longer in the list. */
	pvt->which_iflist = DAHDI_IFLIST_NONE;
	pvt->prev = NULL;
	pvt->next = NULL;
}

#if defined(HAVE_PRI)
/*!
 * \internal
 * \brief Insert the given chan_dahdi interface structure into the no B channel list.
 * \since 1.8
 *
 * \param pri sig_pri span control structure holding no B channel list.
 * \param pvt chan_dahdi private interface structure to insert.
 *
 * \details
 * The interface list is a doubly linked list sorted by the chan_dahdi channel number.
 * Any duplicates are inserted after the existing entries.
 *
 * \note The new interface must not already be in the list.
 */
static void dahdi_nobch_insert(struct sig_pri_span *pri, struct dahdi_pvt *pvt)
{
	struct dahdi_pvt *cur;

	pvt->which_iflist = DAHDI_IFLIST_NO_B_CHAN;

	/* Find place in middle of list for the new interface. */
	for (cur = pri->no_b_chan_iflist; cur; cur = cur->next) {
		if (pvt->channel < cur->channel) {
			/* New interface goes before the current interface. */
			pvt->prev = cur->prev;
			pvt->next = cur;
			if (cur->prev) {
				/* Insert into the middle of the list. */
				cur->prev->next = pvt;
			} else {
				/* Insert at head of list. */
				pri->no_b_chan_iflist = pvt;
			}
			cur->prev = pvt;
			return;
		}
	}

	/* New interface goes onto the end of the list */
	pvt->prev = pri->no_b_chan_end;
	pvt->next = NULL;
	if (pri->no_b_chan_end) {
		((struct dahdi_pvt *) pri->no_b_chan_end)->next = pvt;
	}
	pri->no_b_chan_end = pvt;
	if (!pri->no_b_chan_iflist) {
		/* List was empty */
		pri->no_b_chan_iflist = pvt;
	}
}
#endif	/* defined(HAVE_PRI) */

#if defined(HAVE_PRI)
/*!
 * \internal
 * \brief Extract the given chan_dahdi interface structure from the no B channel list.
 * \since 1.8
 *
 * \param pri sig_pri span control structure holding no B channel list.
 * \param pvt chan_dahdi private interface structure to extract.
 *
 * \note
 * The given interface structure can be either in the interface list or a stand alone
 * structure that has not been put in the list if the next and prev pointers are NULL.
 */
static void dahdi_nobch_extract(struct sig_pri_span *pri, struct dahdi_pvt *pvt)
{
	/* Extract from the forward chain. */
	if (pvt->prev) {
		pvt->prev->next = pvt->next;
	} else if (pri->no_b_chan_iflist == pvt) {
		/* Node is at the head of the list. */
		pri->no_b_chan_iflist = pvt->next;
	}

	/* Extract from the reverse chain. */
	if (pvt->next) {
		pvt->next->prev = pvt->prev;
	} else if (pri->no_b_chan_end == pvt) {
		/* Node is at the end of the list. */
		pri->no_b_chan_end = pvt->prev;
	}

	/* Node is no longer in the list. */
	pvt->which_iflist = DAHDI_IFLIST_NONE;
	pvt->prev = NULL;
	pvt->next = NULL;
}
#endif	/* defined(HAVE_PRI) */

#if defined(HAVE_PRI)
/*!
 * \internal
 * \brief Unlink the channel interface from the PRI private pointer array.
 * \since 1.8
 *
 * \param pvt chan_dahdi private interface structure to unlink.
 */
static void dahdi_unlink_pri_pvt(struct dahdi_pvt *pvt)
{
	unsigned idx;
	struct sig_pri_span *pri;

	pri = pvt->pri;
	if (!pri) {
		/* Not PRI signaling so cannot be in a PRI private pointer array. */
		return;
	}
	ast_mutex_lock(&pri->lock);
	for (idx = 0; idx < pri->numchans; ++idx) {
		if (pri->pvts[idx] == pvt->sig_pvt) {
			pri->pvts[idx] = NULL;
			ast_mutex_unlock(&pri->lock);
			return;
		}
	}
	ast_mutex_unlock(&pri->lock);
}
#endif	/* defined(HAVE_PRI) */

#if defined(HAVE_SS7)
/*!
 * \internal
 * \brief Unlink the channel interface from the SS7 private pointer array.
 * \since 1.8
 *
 * \param pvt chan_dahdi private interface structure to unlink.
 */
static void dahdi_unlink_ss7_pvt(struct dahdi_pvt *pvt)
{
	unsigned idx;
	struct sig_ss7_linkset *ss7;

	ss7 = pvt->ss7;
	if (!ss7) {
		/* Not SS7 signaling so cannot be in a SS7 private pointer array. */
		return;
	}
	ast_mutex_lock(&ss7->lock);
	for (idx = 0; idx < ss7->numchans; ++idx) {
		if (ss7->pvts[idx] == pvt->sig_pvt) {
			ss7->pvts[idx] = NULL;
			ast_mutex_unlock(&ss7->lock);
			return;
		}
	}
	ast_mutex_unlock(&ss7->lock);
}
#endif	/* defined(HAVE_SS7) */

#if defined(HAVE_OPENR2)
/*!
 * \internal
 * \brief Unlink the channel interface from the MFC/R2 private pointer array.
 *
 * \param pvt chan_dahdi private interface structure to unlink.
 */
static void dahdi_unlink_mfcr2_pvt(struct dahdi_pvt *pvt)
{
	unsigned idx;
	struct dahdi_mfcr2 *mfcr2;
	int should_destroy_link = 0;

	ast_mutex_lock(&pvt->lock);
	if (pvt->r2chan) {
		ast_debug(1, "Disable MFC/R2 channel %d read\n", pvt->channel);
		openr2_chan_disable_read(pvt->r2chan);
	}
	mfcr2 = pvt->mfcr2;
	if (mfcr2) {
		for (idx = 0; idx < mfcr2->numchans; ++idx) {
			if (mfcr2->pvts[idx] == pvt) {
				ast_debug(1, "Removing MFC/R2 channel %d from the mfcr2 link\n", pvt->channel);
				mfcr2->pvts[idx] = NULL;
				mfcr2->live_chans--;
				break;
			}
		}
		if (!mfcr2->live_chans) {
			ast_debug(1, "MFC/R2 link is now empty\n");
			should_destroy_link = 1;
		}
	}
	ast_mutex_unlock(&pvt->lock);
	if (should_destroy_link) {
		ast_debug(1, "MFC/R2 link is now empty\n");
		mfcr2_queue_for_destruction(pvt);
	}
}
#endif	/* defined(HAVE_OPENR2) */

static struct dahdi_pvt *find_next_iface_in_span(struct dahdi_pvt *cur)
{
	if (cur->next && cur->next->span == cur->span) {
		return cur->next;
	} else if (cur->prev && cur->prev->span == cur->span) {
		return cur->prev;
	}

	return NULL;
}

static void destroy_dahdi_pvt(struct dahdi_pvt *pvt)
{
	struct dahdi_pvt *p = pvt;

	if (p->manages_span_alarms) {
		struct dahdi_pvt *next = find_next_iface_in_span(p);
		if (next) {
			next->manages_span_alarms = 1;
		}
	}

	/* Remove channel from the list */
#if defined(HAVE_PRI)
	dahdi_unlink_pri_pvt(p);
#endif	/* defined(HAVE_PRI) */
#if defined(HAVE_SS7)
	dahdi_unlink_ss7_pvt(p);
#endif	/* defined(HAVE_SS7) */
#if defined(HAVE_OPENR2)
	dahdi_unlink_mfcr2_pvt(p);
#endif	/* defined(HAVE_SS7) */
	switch (pvt->which_iflist) {
	case DAHDI_IFLIST_NONE:
		break;
	case DAHDI_IFLIST_MAIN:
		dahdi_iflist_extract(p);
		break;
#if defined(HAVE_PRI)
	case DAHDI_IFLIST_NO_B_CHAN:
		if (p->pri) {
			dahdi_nobch_extract(p->pri, p);
		}
		break;
#endif	/* defined(HAVE_PRI) */
	}

	if (p->sig_pvt) {
		if (dahdi_analog_lib_handles(p->sig, 0, 0)) {
			analog_delete(p->sig_pvt);
		}
		switch (p->sig) {
#if defined(HAVE_PRI)
		case SIG_PRI_LIB_HANDLE_CASES:
			sig_pri_chan_delete(p->sig_pvt);
			break;
#endif	/* defined(HAVE_PRI) */
#if defined(HAVE_SS7)
		case SIG_SS7:
			sig_ss7_chan_delete(p->sig_pvt);
			break;
#endif	/* defined(HAVE_SS7) */
		default:
			break;
		}
	}
	ast_free(p->cidspill);
	if (p->use_smdi) {
		ao2_cleanup(p->smdi_iface);
	}
	if (p->mwi_event_sub) {
		p->mwi_event_sub = ast_mwi_unsubscribe(p->mwi_event_sub);
	}
	if (p->vars) {
		ast_variables_destroy(p->vars);
	}
	if (p->cc_params) {
		ast_cc_config_params_destroy(p->cc_params);
	}

	p->named_callgroups = ast_unref_namedgroups(p->named_callgroups);
	p->named_pickupgroups = ast_unref_namedgroups(p->named_pickupgroups);

	ast_mutex_destroy(&p->lock);
	dahdi_close_sub(p, SUB_REAL);
	if (p->owner) {
		ast_channel_tech_pvt_set(p->owner, NULL);
	}
	ast_free(p);
}

static void destroy_channel(struct dahdi_pvt *cur, int now)
{
	int i;

	if (!now) {
		/* Do not destroy the channel now if it is owned by someone. */
		if (cur->owner) {
			return;
		}
		for (i = 0; i < 3; i++) {
			if (cur->subs[i].owner) {
				return;
			}
		}
	}
	destroy_dahdi_pvt(cur);
}

static void destroy_all_channels(void)
{
	int chan;
#if defined(HAVE_PRI)
	unsigned span;
	struct sig_pri_span *pri;
#endif	/* defined(HAVE_PRI) */
	struct dahdi_pvt *p;

	while (num_restart_pending) {
		usleep(1);
	}

	ast_mutex_lock(&iflock);
	/* Destroy all the interfaces and free their memory */
	while (iflist) {
		p = iflist;

		chan = p->channel;
#if defined(HAVE_PRI_SERVICE_MESSAGES)
		{
			char db_chan_name[20];
			char db_answer[5];
			char state;
			int why = -1;

			snprintf(db_chan_name, sizeof(db_chan_name), "%s/%d:%d", dahdi_db, p->span, chan);
			if (!ast_db_get(db_chan_name, SRVST_DBKEY, db_answer, sizeof(db_answer))) {
				sscanf(db_answer, "%1c:%30d", &state, &why);
			}
			if (!why) {
				/* SRVST persistence is not required */
				ast_db_del(db_chan_name, SRVST_DBKEY);
			}
		}
#endif	/* defined(HAVE_PRI_SERVICE_MESSAGES) */
		/* Free associated memory */
		destroy_dahdi_pvt(p);
		ast_verb(3, "Unregistered channel %d\n", chan);
	}
	ifcount = 0;
	ast_mutex_unlock(&iflock);

#if defined(HAVE_PRI)
	/* Destroy all of the no B channel interface lists */
	for (span = 0; span < NUM_SPANS; ++span) {
		if (!pris[span].dchannels[0]) {
			break;
		}
		pri = &pris[span].pri;
		ast_mutex_lock(&pri->lock);
		while (pri->no_b_chan_iflist) {
			p = pri->no_b_chan_iflist;

			/* Free associated memory */
			destroy_dahdi_pvt(p);
		}
		ast_mutex_unlock(&pri->lock);
	}
#endif	/* defined(HAVE_PRI) */
}

#if defined(HAVE_PRI)
static char *dahdi_send_keypad_facility_app = "DAHDISendKeypadFacility";

static int dahdi_send_keypad_facility_exec(struct ast_channel *chan, const char *digits)
{
	/* Data will be our digit string */
	struct dahdi_pvt *p;

	if (ast_strlen_zero(digits)) {
		ast_debug(1, "No digit string sent to application!\n");
		return -1;
	}

	p = (struct dahdi_pvt *)ast_channel_tech_pvt(chan);

	if (!p) {
		ast_debug(1, "Unable to find technology private\n");
		return -1;
	}

	pri_send_keypad_facility_exec(p->sig_pvt, digits);

	return 0;
}
#endif	/* defined(HAVE_PRI) */

#if defined(HAVE_PRI)
#if defined(HAVE_PRI_PROG_W_CAUSE)
static char *dahdi_send_callrerouting_facility_app = "DAHDISendCallreroutingFacility";

static int dahdi_send_callrerouting_facility_exec(struct ast_channel *chan, const char *data)
{
	/* Data will be our digit string */
	struct dahdi_pvt *pvt;
	char *parse;
	int res;
	AST_DECLARE_APP_ARGS(args,
		AST_APP_ARG(destination);
		AST_APP_ARG(original);
		AST_APP_ARG(reason);
	);

	if (ast_strlen_zero(data)) {
		ast_debug(1, "No data sent to application!\n");
		return -1;
	}
	if (ast_channel_tech(chan) != &dahdi_tech) {
		ast_debug(1, "Only DAHDI technology accepted!\n");
		return -1;
	}
	pvt = (struct dahdi_pvt *) ast_channel_tech_pvt(chan);
	if (!pvt) {
		ast_debug(1, "Unable to find technology private\n");
		return -1;
	}
	switch (pvt->sig) {
	case SIG_PRI_LIB_HANDLE_CASES:
		break;
	default:
		ast_debug(1, "callrerouting attempted on non-ISDN channel %s\n",
			ast_channel_name(chan));
		return -1;
	}

	parse = ast_strdupa(data);
	AST_STANDARD_APP_ARGS(args, parse);

	if (ast_strlen_zero(args.destination)) {
		ast_log(LOG_WARNING, "callrerouting facility requires at least destination number argument\n");
		return -1;
	}

	if (ast_strlen_zero(args.original)) {
		ast_log(LOG_WARNING, "Callrerouting Facility without original called number argument\n");
		args.original = NULL;
	}

	if (ast_strlen_zero(args.reason)) {
		ast_log(LOG_NOTICE, "Callrerouting Facility without diversion reason argument, defaulting to unknown\n");
		args.reason = NULL;
	}

	res = pri_send_callrerouting_facility_exec(pvt->sig_pvt, ast_channel_state(chan),
		args.destination, args.original, args.reason);
	if (!res) {
		/*
		 * Wait up to 5 seconds for a reply before hanging up this call
		 * leg if the peer does not disconnect first.
		 */
		ast_safe_sleep(chan, 5000);
	}

	return -1;
}
#endif	/* defined(HAVE_PRI_PROG_W_CAUSE) */
#endif	/* defined(HAVE_PRI) */

#if defined(HAVE_OPENR2)
static const char * const dahdi_accept_r2_call_app = "DAHDIAcceptR2Call";

static int dahdi_accept_r2_call_exec(struct ast_channel *chan, const char *data)
{
	/* data is whether to accept with charge or no charge */
	openr2_call_mode_t accept_mode;
	int res, timeout, maxloops;
	struct ast_frame *f;
	struct dahdi_pvt *p;
	char *parse;
	AST_DECLARE_APP_ARGS(args,
			AST_APP_ARG(charge);
	);

	if (ast_strlen_zero(data)) {
		ast_debug(1, "No data sent to application!\n");
		return -1;
	}

	if (ast_channel_tech(chan) != &dahdi_tech) {
		ast_debug(1, "Only DAHDI technology accepted!\n");
		return -1;
	}

	p = (struct dahdi_pvt *)ast_channel_tech_pvt(chan);
	if (!p) {
		ast_debug(1, "Unable to find technology private!\n");
		return -1;
	}

	parse = ast_strdupa(data);
	AST_STANDARD_APP_ARGS(args, parse);

	if (ast_strlen_zero(args.charge)) {
		ast_log(LOG_WARNING, "DAHDIAcceptR2Call requires 'yes' or 'no' for the charge parameter\n");
		return -1;
	}

	ast_mutex_lock(&p->lock);
	if (!p->mfcr2 || !p->mfcr2call) {
		ast_mutex_unlock(&p->lock);
		ast_debug(1, "Channel %s does not seems to be an R2 active channel!\n", ast_channel_name(chan));
		return -1;
	}

	if (p->mfcr2_call_accepted) {
		ast_mutex_unlock(&p->lock);
		ast_debug(1, "MFC/R2 call already accepted on channel %s!\n", ast_channel_name(chan));
		return 0;
	}
	accept_mode = ast_true(args.charge) ? OR2_CALL_WITH_CHARGE : OR2_CALL_NO_CHARGE;
	if (openr2_chan_accept_call(p->r2chan, accept_mode)) {
		ast_mutex_unlock(&p->lock);
		ast_log(LOG_WARNING, "Failed to accept MFC/R2 call!\n");
		return -1;
	}
	ast_mutex_unlock(&p->lock);

	res = 0;
	timeout = 100;
	maxloops = 50; /* wait up to 5 seconds */
	/* we need to read() until the call is accepted */
	while (maxloops > 0) {
		maxloops--;
		if (ast_check_hangup(chan)) {
			break;
		}
		res = ast_waitfor(chan, timeout);
		if (res < 0) {
			ast_debug(1, "ast_waitfor failed on channel %s, going out ...\n", ast_channel_name(chan));
			res = -1;
			break;
		}
		if (res == 0) {
			continue;
		}
		res = 0;
		f = ast_read(chan);
		if (!f) {
			ast_debug(1, "No frame read on channel %s, going out ...\n", ast_channel_name(chan));
			res = -1;
			break;
		}
		if (f->frametype == AST_FRAME_CONTROL && f->subclass.integer == AST_CONTROL_HANGUP) {
			ast_debug(1, "Got HANGUP frame on channel %s, going out ...\n", ast_channel_name(chan));
			ast_frfree(f);
			res = -1;
			break;
		}
		ast_frfree(f);
		ast_mutex_lock(&p->lock);
		if (p->mfcr2_call_accepted) {
			ast_mutex_unlock(&p->lock);
			ast_debug(1, "Accepted MFC/R2 call!\n");
			break;
		}
		ast_mutex_unlock(&p->lock);
	}
	if (res == -1) {
		ast_log(LOG_WARNING, "Failed to accept MFC/R2 call!\n");
	}
	return res;
}

static openr2_call_disconnect_cause_t dahdi_ast_cause_to_r2_cause(int cause)
{
	openr2_call_disconnect_cause_t r2cause = OR2_CAUSE_NORMAL_CLEARING;
	switch (cause) {
	case AST_CAUSE_USER_BUSY:
	case AST_CAUSE_CALL_REJECTED:
	case AST_CAUSE_INTERWORKING: /* I don't know wtf is this but is used sometimes when ekiga rejects a call */
		r2cause = OR2_CAUSE_BUSY_NUMBER;
		break;

	case AST_CAUSE_NORMAL_CIRCUIT_CONGESTION:
	case AST_CAUSE_SWITCH_CONGESTION:
		r2cause = OR2_CAUSE_NETWORK_CONGESTION;
		break;

	case AST_CAUSE_UNALLOCATED:
		r2cause = OR2_CAUSE_UNALLOCATED_NUMBER;
		break;

	case AST_CAUSE_NETWORK_OUT_OF_ORDER:
	case AST_CAUSE_DESTINATION_OUT_OF_ORDER:
		r2cause = OR2_CAUSE_OUT_OF_ORDER;
		break;

	case AST_CAUSE_NO_ANSWER:
	case AST_CAUSE_NO_USER_RESPONSE:
		r2cause = OR2_CAUSE_NO_ANSWER;
		break;

	default:
		r2cause = OR2_CAUSE_NORMAL_CLEARING;
		break;
	}
	ast_debug(1, "ast cause %d resulted in openr2 cause %d/%s\n",
			cause, r2cause, openr2_proto_get_disconnect_string(r2cause));
	return r2cause;
}
#endif

static int revert_fax_buffers(struct dahdi_pvt *p, struct ast_channel *ast)
{
	if (p->bufferoverrideinuse) {
		/* faxbuffers are in use, revert them */
		struct dahdi_bufferinfo bi = {
			.txbufpolicy = p->buf_policy,
			.rxbufpolicy = p->buf_policy,
			.bufsize = p->bufsize,
			.numbufs = p->buf_no
		};
		int bpres;

		if ((bpres = ioctl(p->subs[SUB_REAL].dfd, DAHDI_SET_BUFINFO, &bi)) < 0) {
			ast_log(LOG_WARNING, "Channel '%s' unable to revert buffer policy: %s\n", ast_channel_name(ast), strerror(errno));
		}
		p->bufferoverrideinuse = 0;
		return bpres;
	}

	return -1;
}

static int dahdi_hangup(struct ast_channel *ast)
{
	int res = 0;
	int idx,x;
	int law;
	/*static int restore_gains(struct dahdi_pvt *p);*/
	struct dahdi_pvt *p = ast_channel_tech_pvt(ast);
	struct dahdi_params par;

	ast_debug(1, "dahdi_hangup(%s)\n", ast_channel_name(ast));
	if (!ast_channel_tech_pvt(ast)) {
		ast_log(LOG_WARNING, "Asked to hangup channel not connected\n");
		return 0;
	}

	ast_mutex_lock(&p->lock);
	p->exten[0] = '\0';
	/* Always use sig_analog hangup handling for operator mode */
	if (dahdi_analog_lib_handles(p->sig, p->radio, 0)) {
		p->oprmode = 0;
		dahdi_confmute(p, 0);
		restore_gains(p);
		p->ignoredtmf = 0;
		p->waitingfordt.tv_sec = 0;

		res = analog_hangup(p->sig_pvt, ast);
		revert_fax_buffers(p, ast);

		goto hangup_out;
	} else {
		p->cid_num[0] = '\0';
		p->cid_name[0] = '\0';
		p->cid_subaddr[0] = '\0';
	}

#if defined(HAVE_PRI)
	if (dahdi_sig_pri_lib_handles(p->sig)) {
		x = 1;
		ast_channel_setoption(ast, AST_OPTION_AUDIO_MODE, &x, sizeof(char), 0);

		dahdi_confmute(p, 0);
		p->muting = 0;
		restore_gains(p);
		if (p->dsp) {
			ast_dsp_free(p->dsp);
			p->dsp = NULL;
		}
		p->ignoredtmf = 0;

		/* Real channel, do some fixup */
		p->subs[SUB_REAL].owner = NULL;
		p->subs[SUB_REAL].needbusy = 0;
		dahdi_setlinear(p->subs[SUB_REAL].dfd, 0);

		p->owner = NULL;
		p->cid_tag[0] = '\0';
		p->ringt = 0;/* Probably not used in this mode.  Reset anyway. */
		p->distinctivering = 0;/* Probably not used in this mode. Reset anyway. */
		p->confirmanswer = 0;/* Probably not used in this mode. Reset anyway. */
		p->outgoing = 0;
		p->digital = 0;
		p->faxhandled = 0;
		p->pulsedial = 0;/* Probably not used in this mode. Reset anyway. */

		revert_fax_buffers(p, ast);

		p->law = p->law_default;
		law = p->law_default;
		res = ioctl(p->subs[SUB_REAL].dfd, DAHDI_SETLAW, &law);
		if (res < 0) {
			ast_log(LOG_WARNING, "Unable to set law on channel %d to default: %s\n",
				p->channel, strerror(errno));
		}

		sig_pri_hangup(p->sig_pvt, ast);

		tone_zone_play_tone(p->subs[SUB_REAL].dfd, -1);
		dahdi_ec_disable(p);

		x = 0;
		ast_channel_setoption(ast, AST_OPTION_TDD, &x, sizeof(char), 0);
		p->didtdd = 0;/* Probably not used in this mode. Reset anyway. */

		p->rdnis[0] = '\0';
		dahdi_conf_update(p);
		reset_conf(p);

		/* Restore data mode */
		x = 0;
		ast_channel_setoption(ast, AST_OPTION_AUDIO_MODE, &x, sizeof(char), 0);

		if (num_restart_pending == 0) {
			restart_monitor();
		}
		goto hangup_out;
	}
#endif	/* defined(HAVE_PRI) */

#if defined(HAVE_SS7)
	if (p->sig == SIG_SS7) {
		x = 1;
		ast_channel_setoption(ast, AST_OPTION_AUDIO_MODE, &x, sizeof(char), 0);

		dahdi_confmute(p, 0);
		p->muting = 0;
		restore_gains(p);
		if (p->dsp) {
			ast_dsp_free(p->dsp);
			p->dsp = NULL;
		}
		p->ignoredtmf = 0;

		/* Real channel, do some fixup */
		p->subs[SUB_REAL].owner = NULL;
		p->subs[SUB_REAL].needbusy = 0;
		dahdi_setlinear(p->subs[SUB_REAL].dfd, 0);

		p->owner = NULL;
		p->ringt = 0;/* Probably not used in this mode.  Reset anyway. */
		p->distinctivering = 0;/* Probably not used in this mode. Reset anyway. */
		p->confirmanswer = 0;/* Probably not used in this mode. Reset anyway. */
		p->outgoing = 0;
		p->digital = 0;
		p->faxhandled = 0;
		p->pulsedial = 0;/* Probably not used in this mode. Reset anyway. */

		revert_fax_buffers(p, ast);

		p->law = p->law_default;
		law = p->law_default;
		res = ioctl(p->subs[SUB_REAL].dfd, DAHDI_SETLAW, &law);
		if (res < 0) {
			ast_log(LOG_WARNING, "Unable to set law on channel %d to default: %s\n",
				p->channel, strerror(errno));
		}

		sig_ss7_hangup(p->sig_pvt, ast);

		tone_zone_play_tone(p->subs[SUB_REAL].dfd, -1);
		dahdi_ec_disable(p);

		x = 0;
		ast_channel_setoption(ast, AST_OPTION_TDD, &x, sizeof(char), 0);
		p->didtdd = 0;/* Probably not used in this mode. Reset anyway. */

		dahdi_conf_update(p);
		reset_conf(p);

		/* Restore data mode */
		x = 0;
		ast_channel_setoption(ast, AST_OPTION_AUDIO_MODE, &x, sizeof(char), 0);

		if (num_restart_pending == 0) {
			restart_monitor();
		}
		goto hangup_out;
	}
#endif	/* defined(HAVE_SS7) */

	idx = dahdi_get_index(ast, p, 1);

	dahdi_confmute(p, 0);
	p->muting = 0;
	restore_gains(p);
	if (p->origcid_num) {
		ast_copy_string(p->cid_num, p->origcid_num, sizeof(p->cid_num));
		ast_free(p->origcid_num);
		p->origcid_num = NULL;
	}
	if (p->origcid_name) {
		ast_copy_string(p->cid_name, p->origcid_name, sizeof(p->cid_name));
		ast_free(p->origcid_name);
		p->origcid_name = NULL;
	}
	if (p->dsp)
		ast_dsp_set_digitmode(p->dsp, DSP_DIGITMODE_DTMF | p->dtmfrelax);

	ast_debug(1, "Hangup: channel: %d index = %d, normal = %d, callwait = %d, thirdcall = %d\n",
		p->channel, idx, p->subs[SUB_REAL].dfd, p->subs[SUB_CALLWAIT].dfd, p->subs[SUB_THREEWAY].dfd);
	p->ignoredtmf = 0;

	if (idx > -1) {
		/* Real channel, do some fixup */
		p->subs[idx].owner = NULL;
		p->subs[idx].needanswer = 0;
		p->subs[idx].needflash = 0;
		p->subs[idx].needringing = 0;
		p->subs[idx].needbusy = 0;
		p->subs[idx].needcongestion = 0;
		p->subs[idx].linear = 0;
		p->polarity = POLARITY_IDLE;
		dahdi_setlinear(p->subs[idx].dfd, 0);
		if (idx == SUB_REAL) {
			if ((p->subs[SUB_CALLWAIT].dfd > -1) && (p->subs[SUB_THREEWAY].dfd > -1)) {
				ast_debug(1, "Normal call hung up with both three way call and a call waiting call in place?\n");
				if (p->subs[SUB_CALLWAIT].inthreeway) {
					/* We had flipped over to answer a callwait and now it's gone */
					ast_debug(1, "We were flipped over to the callwait, moving back and not owning.\n");
					/* Move to the call-wait, but un-own us until they flip back. */
					swap_subs(p, SUB_CALLWAIT, SUB_REAL);
					unalloc_sub(p, SUB_CALLWAIT);
					p->owner = NULL;
				} else {
					/* The three way hung up, but we still have a call wait */
					ast_debug(1, "We were in the threeway and have a callwait still.  Ditching the threeway.\n");
					swap_subs(p, SUB_THREEWAY, SUB_REAL);
					unalloc_sub(p, SUB_THREEWAY);
					if (p->subs[SUB_REAL].inthreeway) {
						/* This was part of a three way call.  Immediately make way for
						   another call */
						ast_debug(1, "Call was complete, setting owner to former third call\n");
						p->owner = p->subs[SUB_REAL].owner;
					} else {
						/* This call hasn't been completed yet...  Set owner to NULL */
						ast_debug(1, "Call was incomplete, setting owner to NULL\n");
						p->owner = NULL;
					}
					p->subs[SUB_REAL].inthreeway = 0;
				}
			} else if (p->subs[SUB_CALLWAIT].dfd > -1) {
				/* Move to the call-wait and switch back to them. */
				swap_subs(p, SUB_CALLWAIT, SUB_REAL);
				unalloc_sub(p, SUB_CALLWAIT);
				p->owner = p->subs[SUB_REAL].owner;
				if (ast_channel_state(p->owner) != AST_STATE_UP)
					p->subs[SUB_REAL].needanswer = 1;
				ast_queue_unhold(p->subs[SUB_REAL].owner);
			} else if (p->subs[SUB_THREEWAY].dfd > -1) {
				swap_subs(p, SUB_THREEWAY, SUB_REAL);
				unalloc_sub(p, SUB_THREEWAY);
				if (p->subs[SUB_REAL].inthreeway) {
					/* This was part of a three way call.  Immediately make way for
					   another call */
					ast_debug(1, "Call was complete, setting owner to former third call\n");
					p->owner = p->subs[SUB_REAL].owner;
				} else {
					/* This call hasn't been completed yet...  Set owner to NULL */
					ast_debug(1, "Call was incomplete, setting owner to NULL\n");
					p->owner = NULL;
				}
				p->subs[SUB_REAL].inthreeway = 0;
			}
		} else if (idx == SUB_CALLWAIT) {
			/* Ditch the holding callwait call, and immediately make it available */
			if (p->subs[SUB_CALLWAIT].inthreeway) {
				/* This is actually part of a three way, placed on hold.  Place the third part
				   on music on hold now */
				if (p->subs[SUB_THREEWAY].owner) {
					ast_queue_hold(p->subs[SUB_THREEWAY].owner, p->mohsuggest);
				}
				p->subs[SUB_THREEWAY].inthreeway = 0;
				/* Make it the call wait now */
				swap_subs(p, SUB_CALLWAIT, SUB_THREEWAY);
				unalloc_sub(p, SUB_THREEWAY);
			} else
				unalloc_sub(p, SUB_CALLWAIT);
		} else if (idx == SUB_THREEWAY) {
			if (p->subs[SUB_CALLWAIT].inthreeway) {
				/* The other party of the three way call is currently in a call-wait state.
				   Start music on hold for them, and take the main guy out of the third call */
				if (p->subs[SUB_CALLWAIT].owner) {
					ast_queue_hold(p->subs[SUB_CALLWAIT].owner, p->mohsuggest);
				}
				p->subs[SUB_CALLWAIT].inthreeway = 0;
			}
			p->subs[SUB_REAL].inthreeway = 0;
			/* If this was part of a three way call index, let us make
			   another three way call */
			unalloc_sub(p, SUB_THREEWAY);
		} else {
			/* This wasn't any sort of call, but how are we an index? */
			ast_log(LOG_WARNING, "Index found but not any type of call?\n");
		}
	}

	if (!p->subs[SUB_REAL].owner && !p->subs[SUB_CALLWAIT].owner && !p->subs[SUB_THREEWAY].owner) {
		p->owner = NULL;
		p->ringt = 0;
		p->distinctivering = 0;
		p->confirmanswer = 0;
		p->outgoing = 0;
		p->digital = 0;
		p->faxhandled = 0;
		p->pulsedial = 0;
		if (p->dsp) {
			ast_dsp_free(p->dsp);
			p->dsp = NULL;
		}

		revert_fax_buffers(p, ast);

		p->law = p->law_default;
		law = p->law_default;
		res = ioctl(p->subs[SUB_REAL].dfd, DAHDI_SETLAW, &law);
		if (res < 0)
			ast_log(LOG_WARNING, "Unable to set law on channel %d to default: %s\n", p->channel, strerror(errno));
		/* Perform low level hangup if no owner left */
#ifdef HAVE_OPENR2
		if (p->mfcr2 && p->mfcr2call && openr2_chan_get_direction(p->r2chan) != OR2_DIR_STOPPED) {
			ast_debug(1, "disconnecting MFC/R2 call on chan %d\n", p->channel);
			/* If it's an incoming call, check the mfcr2_forced_release setting */
			if (openr2_chan_get_direction(p->r2chan) == OR2_DIR_BACKWARD && p->mfcr2_forced_release) {
				dahdi_r2_disconnect_call(p, OR2_CAUSE_FORCED_RELEASE);
			} else {
				const char *r2causestr = pbx_builtin_getvar_helper(ast, "MFCR2_CAUSE");
				int r2cause_user = r2causestr ? atoi(r2causestr) : 0;
				openr2_call_disconnect_cause_t r2cause = r2cause_user ? dahdi_ast_cause_to_r2_cause(r2cause_user)
					                                              : dahdi_ast_cause_to_r2_cause(ast_channel_hangupcause(ast));
				dahdi_r2_disconnect_call(p, r2cause);
			}
		} else if (p->mfcr2call) {
			ast_debug(1, "Clearing call request on channel %d\n", p->channel);
			/* since ast_request() was called but not ast_call() we have not yet dialed
			and the openr2 stack will not call on_call_end callback, we need to unset
			the mfcr2call flag and bump the monitor count so the monitor thread can take
			care of this channel events from now on */
			p->mfcr2call = 0;
		}
#endif
		switch (p->sig) {
		case SIG_SS7:
		case SIG_MFCR2:
		case SIG_PRI_LIB_HANDLE_CASES:
		case 0:
			break;
		default:
			res = dahdi_set_hook(p->subs[SUB_REAL].dfd, DAHDI_ONHOOK);
			break;
		}
		if (res < 0) {
			ast_log(LOG_WARNING, "Unable to hangup line %s\n", ast_channel_name(ast));
		}
		switch (p->sig) {
		case SIG_FXOGS:
		case SIG_FXOLS:
		case SIG_FXOKS:
			memset(&par, 0, sizeof(par));
			res = ioctl(p->subs[SUB_REAL].dfd, DAHDI_GET_PARAMS, &par);
			if (!res) {
				struct analog_pvt *analog_p = p->sig_pvt;
#if 0
				ast_debug(1, "Hanging up channel %d, offhook = %d\n", p->channel, par.rxisoffhook);
#endif
				/* If they're off hook, try playing congestion */
				if ((par.rxisoffhook) && (!(p->radio || (p->oprmode < 0))))
					tone_zone_play_tone(p->subs[SUB_REAL].dfd, DAHDI_TONE_CONGESTION);
				else
					tone_zone_play_tone(p->subs[SUB_REAL].dfd, -1);
				analog_p->fxsoffhookstate = par.rxisoffhook;
			}
			break;
		case SIG_FXSGS:
		case SIG_FXSLS:
		case SIG_FXSKS:
			/* Make sure we're not made available for at least two seconds assuming
			we were actually used for an inbound or outbound call. */
			if (ast_channel_state(ast) != AST_STATE_RESERVED) {
				time(&p->guardtime);
				p->guardtime += 2;
			}
			break;
		default:
			tone_zone_play_tone(p->subs[SUB_REAL].dfd, -1);
			break;
		}
		if (p->sig)
			dahdi_ec_disable(p);
		x = 0;
		ast_channel_setoption(ast,AST_OPTION_TONE_VERIFY,&x,sizeof(char),0);
		ast_channel_setoption(ast,AST_OPTION_TDD,&x,sizeof(char),0);
		p->didtdd = 0;
		p->callwaitcas = 0;
		p->callwaiting = p->permcallwaiting;
		p->hidecallerid = p->permhidecallerid;
		if (dahdi_analog_lib_handles(p->sig, p->radio, 0) && !p->owner) {
			/* The code in sig_analog handles resetting to permdialmode on originations;
			 * this addresses the edge case of multiple calls that do not involve
			 * origination inbetween, i.e. multiple incoming calls. */
			struct analog_pvt *analog_p = p->sig_pvt;
			/* If no calls remain, reset dialmode.
			 * This way, if the next call is an incoming call,
			 * it's already been reset. */
			analog_p->dialmode = analog_p->permdialmode;
		}
		p->waitingfordt.tv_sec = 0;
		p->dialing = 0;
		p->rdnis[0] = '\0';
		dahdi_conf_update(p);
		reset_conf(p);
		/* Restore data mode */
		switch (p->sig) {
		case SIG_PRI_LIB_HANDLE_CASES:
		case SIG_SS7:
			x = 0;
			ast_channel_setoption(ast,AST_OPTION_AUDIO_MODE,&x,sizeof(char),0);
			break;
		default:
			break;
		}
		if (num_restart_pending == 0)
			restart_monitor();
	}

	p->callwaitingrepeat = 0;
	p->cidcwexpire = 0;
	p->cid_suppress_expire = 0;
	p->oprmode = 0;
hangup_out:
	ast_channel_tech_pvt_set(ast, NULL);
	ast_free(p->cidspill);
	p->cidspill = NULL;

	if (p->reoriginate && p->sig == SIG_FXOKS && dahdi_analog_lib_handles(p->sig, p->radio, 0)) {
		/* Automatic reorigination: if all calls towards a user have hung up,
		 * give dial tone again, so user doesn't need to cycle the hook state manually. */
		if (my_is_off_hook(p) && !p->owner) {
			/* 2 important criteria: channel must be off-hook, with no calls remaining (no owner) */
			ast_debug(1, "Queuing reorigination for channel %d\n", p->channel);
			my_play_tone(p, SUB_REAL, -1); /* Stop any congestion tone that may be present. */
			/* Must wait for the loop disconnect to end.
			 * Sadly, these definitions are in dahdi/kernel.h, not dahdi/user.h
			 * Calling usleep on an active DAHDI channel is a no-no, but this is okay.
			 */
			usleep(800000); /* DAHDI_KEWLTIME + DAHDI_AFTERKEWLTIME */
			/* If the line is still off-hook and ownerless, actually queue the reorigination.
			 * do_monitor will actually go ahead and do it. */
			if (!p->owner && my_is_off_hook(p)) {
				p->doreoriginate = 1; /* Tell do_monitor to reoriginate this channel */
				/* Note, my_off_hook will fail if called before the loop disconnect has finished
				 * (important for FXOKS signaled channels). This is because DAHDI will reject
				 * DAHDI_OFFHOOK while the channel is in TXSTATE_KEWL or TXSTATE_AFTERKEWL,
				 * so we have to wait for that to finish (see comment above).
				 * do_monitor itself cannot block, so make the blocking usleep call
				 * here in the channel thread instead.
				 */
				my_off_hook(p); /* Now, go ahead and take the channel back off hook (sig_analog put it on hook) */
			} else {
				ast_debug(1, "Channel %d is no longer eligible for reorigination (went back on hook or became in use)\n", p->channel);
			}
		}
	}

	ast_mutex_unlock(&p->lock);
	ast_verb(3, "Hungup '%s'\n", ast_channel_name(ast));

	ast_mutex_lock(&iflock);
	if (p->restartpending) {
		num_restart_pending--;
	}

	if (p->destroy) {
		destroy_channel(p, 0);
	}
	ast_mutex_unlock(&iflock);

	ast_module_unref(ast_module_info->self);
	return 0;
}

static int dahdi_answer(struct ast_channel *ast)
{
	struct dahdi_pvt *p = ast_channel_tech_pvt(ast);
	int res = 0;
	int idx;
	ast_setstate(ast, AST_STATE_UP);/*! \todo XXX this is redundantly set by the analog and PRI submodules! */
	ast_mutex_lock(&p->lock);
	idx = dahdi_get_index(ast, p, 0);
	if (idx < 0)
		idx = SUB_REAL;
	/* nothing to do if a radio channel */
	if ((p->radio || (p->oprmode < 0))) {
		ast_mutex_unlock(&p->lock);
		return 0;
	}

	if (dahdi_analog_lib_handles(p->sig, p->radio, p->oprmode)) {
		res = analog_answer(p->sig_pvt, ast);
		ast_mutex_unlock(&p->lock);
		return res;
	}

	switch (p->sig) {
#if defined(HAVE_PRI)
	case SIG_PRI_LIB_HANDLE_CASES:
		res = sig_pri_answer(p->sig_pvt, ast);
		break;
#endif	/* defined(HAVE_PRI) */
#if defined(HAVE_SS7)
	case SIG_SS7:
		res = sig_ss7_answer(p->sig_pvt, ast);
		break;
#endif	/* defined(HAVE_SS7) */
#ifdef HAVE_OPENR2
	case SIG_MFCR2:
		if (!p->mfcr2_call_accepted) {
			/* The call was not accepted on offer nor the user, so it must be accepted now before answering,
			   openr2_chan_answer_call will be called when the callback on_call_accepted is executed */
			p->mfcr2_answer_pending = 1;
			if (p->mfcr2_charge_calls) {
				ast_debug(1, "Accepting MFC/R2 call with charge before answering on chan %d\n", p->channel);
				openr2_chan_accept_call(p->r2chan, OR2_CALL_WITH_CHARGE);
			} else {
				ast_debug(1, "Accepting MFC/R2 call with no charge before answering on chan %d\n", p->channel);
				openr2_chan_accept_call(p->r2chan, OR2_CALL_NO_CHARGE);
			}
		} else {
			ast_debug(1, "Answering MFC/R2 call on chan %d\n", p->channel);
			dahdi_r2_answer(p);
		}
		break;
#endif
	case 0:
		ast_mutex_unlock(&p->lock);
		return 0;
	default:
		ast_log(LOG_WARNING, "Don't know how to answer signalling %d (channel %d)\n", p->sig, p->channel);
		res = -1;
		break;
	}
	ast_mutex_unlock(&p->lock);
	return res;
}

void dahdi_dtmf_detect_disable(struct dahdi_pvt *p)
{
	int val = 0;

	p->ignoredtmf = 1;

	ioctl(p->subs[SUB_REAL].dfd, DAHDI_TONEDETECT, &val);

	if (!p->hardwaredtmf && p->dsp) {
		p->dsp_features &= ~DSP_FEATURE_DIGIT_DETECT;
		ast_dsp_set_features(p->dsp, p->dsp_features);
	}
}

void dahdi_dtmf_detect_enable(struct dahdi_pvt *p)
{
	int val = DAHDI_TONEDETECT_ON | DAHDI_TONEDETECT_MUTE;

	if (p->channel == CHAN_PSEUDO)
		return;

	p->ignoredtmf = 0;

	ioctl(p->subs[SUB_REAL].dfd, DAHDI_TONEDETECT, &val);

	if (!p->hardwaredtmf && p->dsp) {
		p->dsp_features |= DSP_FEATURE_DIGIT_DETECT;
		ast_dsp_set_features(p->dsp, p->dsp_features);
	}
}

static int dahdi_queryoption(struct ast_channel *chan, int option, void *data, int *datalen)
{
	char *cp;
	struct dahdi_pvt *p = ast_channel_tech_pvt(chan);

	/* all supported options require data */
	if (!p || !data || (*datalen < 1)) {
		errno = EINVAL;
		return -1;
	}

	switch (option) {
	case AST_OPTION_TDD:
		cp = (char *) data;
		if (p->mate) {
			*cp = 2;
		} else {
			*cp = p->tdd ? 1 : 0;
		}
		break;
	case AST_OPTION_DIGIT_DETECT:
		cp = (char *) data;
		*cp = p->ignoredtmf ? 0 : 1;
		ast_debug(1, "Reporting digit detection %sabled on %s\n", *cp ? "en" : "dis", ast_channel_name(chan));
		break;
	case AST_OPTION_FAX_DETECT:
		cp = (char *) data;
		*cp = (p->dsp_features & DSP_FEATURE_FAX_DETECT) ? 0 : 1;
		ast_debug(1, "Reporting fax tone detection %sabled on %s\n", *cp ? "en" : "dis", ast_channel_name(chan));
		break;
	case AST_OPTION_CC_AGENT_TYPE:
#if defined(HAVE_PRI)
#if defined(HAVE_PRI_CCSS)
		if (dahdi_sig_pri_lib_handles(p->sig)) {
			ast_copy_string((char *) data, dahdi_pri_cc_type, *datalen);
			break;
		}
#endif	/* defined(HAVE_PRI_CCSS) */
#endif	/* defined(HAVE_PRI) */
		return -1;
	default:
		return -1;
	}

	errno = 0;

	return 0;
}

static int dahdi_setoption(struct ast_channel *chan, int option, void *data, int datalen)
{
	char *cp;
	signed char *scp;
	int x;
	int idx;
	struct dahdi_pvt *p = ast_channel_tech_pvt(chan), *pp;
	struct oprmode *oprmode;


	/* all supported options require data */
	if (!p || !data || (datalen < 1)) {
		errno = EINVAL;
		return -1;
	}

	switch (option) {
	case AST_OPTION_TXGAIN:
		scp = (signed char *) data;
		idx = dahdi_get_index(chan, p, 0);
		if (idx < 0) {
			ast_log(LOG_WARNING, "No index in TXGAIN?\n");
			return -1;
		}
		ast_debug(1, "Setting actual tx gain on %s to %f\n", ast_channel_name(chan), p->txgain + (float) *scp);
		return set_actual_txgain(p->subs[idx].dfd, p->txgain + (float) *scp, p->txdrc, p->law);
	case AST_OPTION_RXGAIN:
		scp = (signed char *) data;
		idx = dahdi_get_index(chan, p, 0);
		if (idx < 0) {
			ast_log(LOG_WARNING, "No index in RXGAIN?\n");
			return -1;
		}
		ast_debug(1, "Setting actual rx gain on %s to %f\n", ast_channel_name(chan), p->rxgain + (float) *scp);
		return set_actual_rxgain(p->subs[idx].dfd, p->rxgain + (float) *scp, p->rxdrc, p->law);
	case AST_OPTION_TONE_VERIFY:
		if (!p->dsp)
			break;
		cp = (char *) data;
		switch (*cp) {
		case 1:
			ast_debug(1, "Set option TONE VERIFY, mode: MUTECONF(1) on %s\n",ast_channel_name(chan));
			ast_dsp_set_digitmode(p->dsp, DSP_DIGITMODE_MUTECONF | p->dtmfrelax);  /* set mute mode if desired */
			break;
		case 2:
			ast_debug(1, "Set option TONE VERIFY, mode: MUTECONF/MAX(2) on %s\n",ast_channel_name(chan));
			ast_dsp_set_digitmode(p->dsp, DSP_DIGITMODE_MUTECONF | DSP_DIGITMODE_MUTEMAX | p->dtmfrelax);  /* set mute mode if desired */
			break;
		default:
			ast_debug(1, "Set option TONE VERIFY, mode: OFF(0) on %s\n",ast_channel_name(chan));
			ast_dsp_set_digitmode(p->dsp, DSP_DIGITMODE_DTMF | p->dtmfrelax);  /* set mute mode if desired */
			break;
		}
		break;
	case AST_OPTION_TDD:
		/* turn on or off TDD */
		cp = (char *) data;
		p->mate = 0;
		if (!*cp) { /* turn it off */
			ast_debug(1, "Set option TDD MODE, value: OFF(0) on %s\n",ast_channel_name(chan));
			if (p->tdd)
				tdd_free(p->tdd);
			p->tdd = 0;
			break;
		}
		ast_debug(1, "Set option TDD MODE, value: %s(%d) on %s\n",
			(*cp == 2) ? "MATE" : "ON", (int) *cp, ast_channel_name(chan));
		dahdi_ec_disable(p);
		/* otherwise, turn it on */
		if (!p->didtdd) { /* if haven't done it yet */
			unsigned char mybuf[41000];/*! \todo XXX This is an abuse of the stack!! */
			unsigned char *buf;
			int size, res, fd, len;
			struct pollfd fds[1];

			buf = mybuf;
			memset(buf, 0x7f, sizeof(mybuf)); /* set to silence */
			ast_tdd_gen_ecdisa(buf + 16000, 16000);  /* put in tone */
			len = 40000;
			idx = dahdi_get_index(chan, p, 0);
			if (idx < 0) {
				ast_log(LOG_WARNING, "No index in TDD?\n");
				return -1;
			}
			fd = p->subs[idx].dfd;
			while (len) {
				if (ast_check_hangup(chan))
					return -1;
				size = len;
				if (size > READ_SIZE)
					size = READ_SIZE;
				fds[0].fd = fd;
				fds[0].events = POLLPRI | POLLOUT;
				fds[0].revents = 0;
				res = poll(fds, 1, -1);
				if (!res) {
					ast_debug(1, "poll (for write) ret. 0 on channel %d\n", p->channel);
					continue;
				}
				/* if got exception */
				if (fds[0].revents & POLLPRI)
					return -1;
				if (!(fds[0].revents & POLLOUT)) {
					ast_debug(1, "write fd not ready on channel %d\n", p->channel);
					continue;
				}
				res = write(fd, buf, size);
				if (res != size) {
					if (res == -1) return -1;
					ast_debug(1, "Write returned %d (%s) on channel %d\n", res, strerror(errno), p->channel);
					break;
				}
				len -= size;
				buf += size;
			}
			p->didtdd = 1; /* set to have done it now */
		}
		if (*cp == 2) { /* Mate mode */
			if (p->tdd)
				tdd_free(p->tdd);
			p->tdd = 0;
			p->mate = 1;
			break;
		}
		if (!p->tdd) { /* if we don't have one yet */
			p->tdd = tdd_new(); /* allocate one */
		}
		break;
	case AST_OPTION_RELAXDTMF:  /* Relax DTMF decoding (or not) */
		if (!p->dsp)
			break;
		cp = (char *) data;
		ast_debug(1, "Set option RELAX DTMF, value: %s(%d) on %s\n",
			*cp ? "ON" : "OFF", (int) *cp, ast_channel_name(chan));
		ast_dsp_set_digitmode(p->dsp, ((*cp) ? DSP_DIGITMODE_RELAXDTMF : DSP_DIGITMODE_DTMF) | p->dtmfrelax);
		break;
	case AST_OPTION_AUDIO_MODE:  /* Set AUDIO mode (or not) */
#if defined(HAVE_PRI)
		if (dahdi_sig_pri_lib_handles(p->sig)
			&& ((struct sig_pri_chan *) p->sig_pvt)->no_b_channel) {
			/* PRI nobch pseudo channel.  Does not handle ioctl(DAHDI_AUDIOMODE) */
			break;
		}
#endif	/* defined(HAVE_PRI) */

		cp = (char *) data;
		if (!*cp) {
			ast_debug(1, "Set option AUDIO MODE, value: OFF(0) on %s\n", ast_channel_name(chan));
			x = 0;
			dahdi_ec_disable(p);
		} else {
			ast_debug(1, "Set option AUDIO MODE, value: ON(1) on %s\n", ast_channel_name(chan));
			x = 1;
		}
		if (ioctl(p->subs[SUB_REAL].dfd, DAHDI_AUDIOMODE, &x) == -1)
			ast_log(LOG_WARNING, "Unable to set audio mode on channel %d to %d: %s\n", p->channel, x, strerror(errno));
		break;
	case AST_OPTION_OPRMODE:  /* Operator services mode */
		oprmode = (struct oprmode *) data;
		/* We don't support operator mode across technologies */
		if (strcasecmp(ast_channel_tech(chan)->type, ast_channel_tech(oprmode->peer)->type)) {
			ast_log(LOG_NOTICE, "Operator mode not supported on %s to %s calls.\n",
					ast_channel_tech(chan)->type, ast_channel_tech(oprmode->peer)->type);
			errno = EINVAL;
			return -1;
		}
		pp = ast_channel_tech_pvt(oprmode->peer);
		p->oprmode = pp->oprmode = 0;
		/* setup peers */
		p->oprpeer = pp;
		pp->oprpeer = p;
		/* setup modes, if any */
		if (oprmode->mode)
		{
			pp->oprmode = oprmode->mode;
			p->oprmode = -oprmode->mode;
		}
		ast_debug(1, "Set Operator Services mode, value: %d on %s/%s\n",
			oprmode->mode, ast_channel_name(chan),ast_channel_name(oprmode->peer));
		break;
	case AST_OPTION_ECHOCAN:
		cp = (char *) data;
		if (*cp) {
			ast_debug(1, "Enabling echo cancellation on %s\n", ast_channel_name(chan));
			dahdi_ec_enable(p);
		} else {
			ast_debug(1, "Disabling echo cancellation on %s\n", ast_channel_name(chan));
			dahdi_ec_disable(p);
		}
		break;
	case AST_OPTION_DIGIT_DETECT:
		cp = (char *) data;
		ast_debug(1, "%sabling digit detection on %s\n", *cp ? "En" : "Dis", ast_channel_name(chan));
		if (*cp) {
			dahdi_dtmf_detect_enable(p);
		} else {
			dahdi_dtmf_detect_disable(p);
		}
		break;
	case AST_OPTION_FAX_DETECT:
		cp = (char *) data;
		if (p->dsp) {
			ast_debug(1, "%sabling fax tone detection on %s\n", *cp ? "En" : "Dis", ast_channel_name(chan));
			if (*cp) {
				p->dsp_features |= DSP_FEATURE_FAX_DETECT;
			} else {
				p->dsp_features &= ~DSP_FEATURE_FAX_DETECT;
			}
			ast_dsp_set_features(p->dsp, p->dsp_features);
		}
		break;
	default:
		return -1;
	}
	errno = 0;

	return 0;
}

static int dahdi_func_read(struct ast_channel *chan, const char *function, char *data, char *buf, size_t len)
{
	struct dahdi_pvt *p = ast_channel_tech_pvt(chan);
	int res = 0;

	if (!p) {
		/* No private structure! */
		*buf = '\0';
		return -1;
	}

	if (!strcasecmp(data, "rxgain")) {
		ast_mutex_lock(&p->lock);
		snprintf(buf, len, "%f", p->rxgain);
		ast_mutex_unlock(&p->lock);
	} else if (!strcasecmp(data, "txgain")) {
		ast_mutex_lock(&p->lock);
		snprintf(buf, len, "%f", p->txgain);
		ast_mutex_unlock(&p->lock);
	} else if (!strcasecmp(data, "dahdi_channel")) {
		ast_mutex_lock(&p->lock);
		snprintf(buf, len, "%d", p->channel);
		ast_mutex_unlock(&p->lock);
	} else if (!strcasecmp(data, "dahdi_span")) {
		ast_mutex_lock(&p->lock);
		snprintf(buf, len, "%d", p->span);
		ast_mutex_unlock(&p->lock);
	} else if (!strcasecmp(data, "dahdi_group")) {
		ast_mutex_lock(&p->lock);
		snprintf(buf, len, "%llu", p->group);
		ast_mutex_unlock(&p->lock);
	} else if (!strcasecmp(data, "dahdi_type")) {
		ast_mutex_lock(&p->lock);
		switch (p->sig) {
#if defined(HAVE_OPENR2)
		case SIG_MFCR2:
			ast_copy_string(buf, "mfc/r2", len);
			break;
#endif	/* defined(HAVE_OPENR2) */
#if defined(HAVE_PRI)
		case SIG_PRI_LIB_HANDLE_CASES:
			ast_copy_string(buf, "pri", len);
			break;
#endif	/* defined(HAVE_PRI) */
		case 0:
			ast_copy_string(buf, "pseudo", len);
			break;
#if defined(HAVE_SS7)
		case SIG_SS7:
			ast_copy_string(buf, "ss7", len);
			break;
#endif	/* defined(HAVE_SS7) */
		default:
			/* The only thing left is analog ports. */
			ast_copy_string(buf, "analog", len);
			break;
		}
		ast_mutex_unlock(&p->lock);
#if defined(HAVE_PRI)
#if defined(HAVE_PRI_REVERSE_CHARGE)
	} else if (!strcasecmp(data, "reversecharge")) {
		ast_mutex_lock(&p->lock);
		switch (p->sig) {
		case SIG_PRI_LIB_HANDLE_CASES:
			snprintf(buf, len, "%d", ((struct sig_pri_chan *) p->sig_pvt)->reverse_charging_indication);
			break;
		default:
			*buf = '\0';
			res = -1;
			break;
		}
		ast_mutex_unlock(&p->lock);
#endif
#if defined(HAVE_PRI_SETUP_KEYPAD)
	} else if (!strcasecmp(data, "keypad_digits")) {
		ast_mutex_lock(&p->lock);
		switch (p->sig) {
		case SIG_PRI_LIB_HANDLE_CASES:
			ast_copy_string(buf, ((struct sig_pri_chan *) p->sig_pvt)->keypad_digits,
				len);
			break;
		default:
			*buf = '\0';
			res = -1;
			break;
		}
		ast_mutex_unlock(&p->lock);
#endif	/* defined(HAVE_PRI_SETUP_KEYPAD) */
	} else if (!strcasecmp(data, "no_media_path")) {
		ast_mutex_lock(&p->lock);
		switch (p->sig) {
		case SIG_PRI_LIB_HANDLE_CASES:
			/*
			 * TRUE if the call is on hold or is call waiting because
			 * there is no media path available.
			 */
			snprintf(buf, len, "%d", ((struct sig_pri_chan *) p->sig_pvt)->no_b_channel);
			break;
		default:
			*buf = '\0';
			res = -1;
			break;
		}
		ast_mutex_unlock(&p->lock);
#endif	/* defined(HAVE_PRI) */
	} else if (!strcasecmp(data, "dialmode")) {
		struct analog_pvt *analog_p;
		ast_mutex_lock(&p->lock);
		analog_p = p->sig_pvt;
		/* Hardcode p->radio and p->oprmode as 0 since we're using this to check for analogness, not the handler */
		if (dahdi_analog_lib_handles(p->sig, 0, 0) && analog_p) {
			switch (analog_p->dialmode) {
			case ANALOG_DIALMODE_BOTH:
				ast_copy_string(buf, "both", len);
				break;
			case ANALOG_DIALMODE_PULSE:
				ast_copy_string(buf, "pulse", len);
				break;
			case ANALOG_DIALMODE_DTMF:
				ast_copy_string(buf, "dtmf", len);
				break;
			case ANALOG_DIALMODE_NONE:
				ast_copy_string(buf, "none", len);
				break;
			}
		} else {
			ast_log(LOG_WARNING, "%s only supported on analog channels\n", data);
			*buf = '\0';
			res = -1;
		}
		ast_mutex_unlock(&p->lock);
	} else {
		*buf = '\0';
		res = -1;
	}

	return res;
}


static int parse_buffers_policy(const char *parse, int *num_buffers, int *policy)
{
	int res;
	char policy_str[21] = "";

	if ((res = sscanf(parse, "%30d,%20s", num_buffers, policy_str)) != 2) {
		ast_log(LOG_WARNING, "Parsing buffer string '%s' failed.\n", parse);
		return 1;
	}
	if (*num_buffers < 0) {
		ast_log(LOG_WARNING, "Invalid buffer count given '%d'.\n", *num_buffers);
		return -1;
	}
	if (!strcasecmp(policy_str, "full")) {
		*policy = DAHDI_POLICY_WHEN_FULL;
	} else if (!strcasecmp(policy_str, "immediate")) {
		*policy = DAHDI_POLICY_IMMEDIATE;
#if defined(HAVE_DAHDI_HALF_FULL)
	} else if (!strcasecmp(policy_str, "half")) {
		*policy = DAHDI_POLICY_HALF_FULL;
#endif
	} else {
		ast_log(LOG_WARNING, "Invalid policy name given '%s'.\n", policy_str);
		return -1;
	}

	return 0;
}

static int dahdi_func_write(struct ast_channel *chan, const char *function, char *data, const char *value)
{
	struct dahdi_pvt *p = ast_channel_tech_pvt(chan);
	int res = 0;

	if (!p) {
		/* No private structure! */
		return -1;
	}

	if (!strcasecmp(data, "buffers")) {
		int num_bufs, policy;

		if (!(parse_buffers_policy(value, &num_bufs, &policy))) {
			struct dahdi_bufferinfo bi = {
				.txbufpolicy = policy,
				.rxbufpolicy = policy,
				.bufsize = p->bufsize,
				.numbufs = num_bufs,
			};
			int bpres;

			if ((bpres = ioctl(p->subs[SUB_REAL].dfd, DAHDI_SET_BUFINFO, &bi)) < 0) {
				ast_log(LOG_WARNING, "Channel '%d' unable to override buffer policy: %s\n", p->channel, strerror(errno));
			} else {
				p->bufferoverrideinuse = 1;
			}
		} else {
			res = -1;
		}
	} else if (!strcasecmp(data, "echocan_mode")) {
		if (!strcasecmp(value, "on")) {
			ast_mutex_lock(&p->lock);
			dahdi_ec_enable(p);
			ast_mutex_unlock(&p->lock);
		} else if (!strcasecmp(value, "off")) {
			ast_mutex_lock(&p->lock);
			dahdi_ec_disable(p);
			ast_mutex_unlock(&p->lock);
#ifdef HAVE_DAHDI_ECHOCANCEL_FAX_MODE
		} else if (!strcasecmp(value, "fax")) {
			int blah = 1;

			ast_mutex_lock(&p->lock);
			if (!p->echocanon) {
				dahdi_ec_enable(p);
			}
			if (ioctl(p->subs[SUB_REAL].dfd, DAHDI_ECHOCANCEL_FAX_MODE, &blah)) {
				ast_log(LOG_WARNING, "Unable to place echocan into fax mode on channel %d: %s\n", p->channel, strerror(errno));
			}
			ast_mutex_unlock(&p->lock);
		} else if (!strcasecmp(value, "voice")) {
			int blah = 0;

			ast_mutex_lock(&p->lock);
			if (!p->echocanon) {
				dahdi_ec_enable(p);
			}
			if (ioctl(p->subs[SUB_REAL].dfd, DAHDI_ECHOCANCEL_FAX_MODE, &blah)) {
				ast_log(LOG_WARNING, "Unable to place echocan into voice mode on channel %d: %s\n", p->channel, strerror(errno));
			}
			ast_mutex_unlock(&p->lock);
#endif
		} else {
			ast_log(LOG_WARNING, "Unsupported value '%s' provided for '%s' item.\n", value, data);
			res = -1;
		}
	} else if (!strcasecmp(data, "dialmode")) {
		struct analog_pvt *analog_p;

		ast_mutex_lock(&p->lock);
		analog_p = p->sig_pvt;
		if (!dahdi_analog_lib_handles(p->sig, 0, 0) || !analog_p) {
			ast_log(LOG_WARNING, "%s only supported on analog channels\n", data);
			ast_mutex_unlock(&p->lock);
			return -1;
		}
		/* analog pvt is used for pulse dialing, so update both */
		if (!strcasecmp(value, "pulse")) {
			analog_p->dialmode = ANALOG_DIALMODE_PULSE;
		} else if (!strcasecmp(value, "dtmf") || !strcasecmp(value, "tone")) {
			analog_p->dialmode = ANALOG_DIALMODE_DTMF;
		} else if (!strcasecmp(value, "none")) {
			analog_p->dialmode = ANALOG_DIALMODE_NONE;
		} else if (!strcasecmp(value, "both")) {
			analog_p->dialmode = ANALOG_DIALMODE_BOTH;
		} else {
			ast_log(LOG_WARNING, "'%s' is an invalid setting for %s\n", value, data);
			res = -1;
		}
		ast_mutex_unlock(&p->lock);
	} else if (!strcasecmp(data, "waitfordialtone")) {
		if (ast_strlen_zero(value)) {
			ast_log(LOG_WARNING, "waitfordialtone requires a duration in ms\n");
			return -1;
		}

		ast_mutex_lock(&p->lock);
		if (!CANPROGRESSDETECT(p)) {
			ast_log(LOG_WARNING, "%s only supported on analog trunks\n", data);
			ast_mutex_unlock(&p->lock);
			return -1;
		}
		/* Only set the temp waitfordialtone setting, not the permanent one. */
		p->waitfordialtonetemp = atoi(value);
		ast_mutex_unlock(&p->lock);
	} else {
		res = -1;
	}

	return res;
}

void dahdi_master_slave_unlink(struct dahdi_pvt *slave, struct dahdi_pvt *master, int needlock)
{
	/* Unlink a specific slave or all slaves/masters from a given master */
	int x;
	int hasslaves;
	if (!master)
		return;
	if (needlock) {
		ast_mutex_lock(&master->lock);
		if (slave) {
			while (ast_mutex_trylock(&slave->lock)) {
				DEADLOCK_AVOIDANCE(&master->lock);
			}
		}
	}
	hasslaves = 0;
	for (x = 0; x < MAX_SLAVES; x++) {
		if (master->slaves[x]) {
			if (!slave || (master->slaves[x] == slave)) {
				/* Take slave out of the conference */
				ast_debug(1, "Unlinking slave %d from %d\n", master->slaves[x]->channel, master->channel);
				conf_del(master, &master->slaves[x]->subs[SUB_REAL], SUB_REAL);
				conf_del(master->slaves[x], &master->subs[SUB_REAL], SUB_REAL);
				master->slaves[x]->master = NULL;
				master->slaves[x] = NULL;
			} else
				hasslaves = 1;
		}
		if (!hasslaves)
			master->inconference = 0;
	}
	if (!slave) {
		if (master->master) {
			/* Take master out of the conference */
			conf_del(master->master, &master->subs[SUB_REAL], SUB_REAL);
			conf_del(master, &master->master->subs[SUB_REAL], SUB_REAL);
			hasslaves = 0;
			for (x = 0; x < MAX_SLAVES; x++) {
				if (master->master->slaves[x] == master)
					master->master->slaves[x] = NULL;
				else if (master->master->slaves[x])
					hasslaves = 1;
			}
			if (!hasslaves)
				master->master->inconference = 0;
		}
		master->master = NULL;
	}
	dahdi_conf_update(master);
	if (needlock) {
		if (slave)
			ast_mutex_unlock(&slave->lock);
		ast_mutex_unlock(&master->lock);
	}
}

void dahdi_master_slave_link(struct dahdi_pvt *slave, struct dahdi_pvt *master)
{
	int x;
	if (!slave || !master) {
		ast_log(LOG_WARNING, "Tried to link to/from NULL??\n");
		return;
	}
	for (x = 0; x < MAX_SLAVES; x++) {
		if (!master->slaves[x]) {
			master->slaves[x] = slave;
			break;
		}
	}
	if (x >= MAX_SLAVES) {
		ast_log(LOG_WARNING, "Replacing slave %d with new slave, %d\n", master->slaves[MAX_SLAVES - 1]->channel, slave->channel);
		master->slaves[MAX_SLAVES - 1] = slave;
	}
	if (slave->master)
		ast_log(LOG_WARNING, "Replacing master %d with new master, %d\n", slave->master->channel, master->channel);
	slave->master = master;

	ast_debug(1, "Making %d slave to master %d at %d\n", slave->channel, master->channel, x);
}

static int dahdi_fixup(struct ast_channel *oldchan, struct ast_channel *newchan)
{
	struct dahdi_pvt *p = ast_channel_tech_pvt(newchan);
	int x;

	ast_mutex_lock(&p->lock);

	ast_debug(1, "New owner for channel %d is %s\n", p->channel, ast_channel_name(newchan));
	if (p->owner == oldchan) {
		p->owner = newchan;
	}
	for (x = 0; x < 3; x++) {
		if (p->subs[x].owner == oldchan) {
			if (!x) {
				dahdi_master_slave_unlink(NULL, p, 0);
			}
			p->subs[x].owner = newchan;
		}
	}
	if (dahdi_analog_lib_handles(p->sig, p->radio, p->oprmode)) {
		analog_fixup(oldchan, newchan, p->sig_pvt);
#if defined(HAVE_PRI)
	} else if (dahdi_sig_pri_lib_handles(p->sig)) {
		sig_pri_fixup(oldchan, newchan, p->sig_pvt);
#endif	/* defined(HAVE_PRI) */
#if defined(HAVE_SS7)
	} else if (p->sig == SIG_SS7) {
		sig_ss7_fixup(oldchan, newchan, p->sig_pvt);
#endif	/* defined(HAVE_SS7) */
	}
	dahdi_conf_update(p);

	ast_mutex_unlock(&p->lock);

	if (ast_channel_state(newchan) == AST_STATE_RINGING) {
		dahdi_indicate(newchan, AST_CONTROL_RINGING, NULL, 0);
	}
	return 0;
}

static int dahdi_ring_phone(struct dahdi_pvt *p)
{
	int x;
	int res;
	/* Make sure our transmit state is on hook */
	x = 0;
	x = DAHDI_ONHOOK;
	res = ioctl(p->subs[SUB_REAL].dfd, DAHDI_HOOK, &x);
	do {
		x = DAHDI_RING;
		res = ioctl(p->subs[SUB_REAL].dfd, DAHDI_HOOK, &x);
		if (res) {
			switch (errno) {
			case EBUSY:
			case EINTR:
				/* Wait just in case */
				usleep(10000);
				continue;
			case EINPROGRESS:
				res = 0;
				break;
			default:
				ast_log(LOG_WARNING, "Couldn't ring the phone: %s\n", strerror(errno));
				res = 0;
			}
		}
	} while (res);
	return res;
}

static void *analog_ss_thread(void *data);

/*!
 * \internal
 * \brief Attempt to transfer 3-way call.
 *
 * \param p DAHDI private structure.
 *
 * \note On entry these locks are held: real-call, private, 3-way call.
 * \note On exit these locks are held: real-call, private.
 *
 * \retval 0 on success.
 * \retval -1 on error.
 */
static int attempt_transfer(struct dahdi_pvt *p)
{
	struct ast_channel *owner_real;
	struct ast_channel *owner_3way;
	enum ast_transfer_result xfer_res;
	int res = 0;

	owner_real = ast_channel_ref(p->subs[SUB_REAL].owner);
	owner_3way = ast_channel_ref(p->subs[SUB_THREEWAY].owner);

	ast_verb(3, "TRANSFERRING %s to %s\n",
		ast_channel_name(owner_3way), ast_channel_name(owner_real));

	ast_channel_unlock(owner_real);
	ast_channel_unlock(owner_3way);
	ast_mutex_unlock(&p->lock);

	xfer_res = ast_bridge_transfer_attended(owner_3way, owner_real);
	if (xfer_res != AST_BRIDGE_TRANSFER_SUCCESS) {
		ast_softhangup(owner_3way, AST_SOFTHANGUP_DEV);
		res = -1;
	}

	/* Must leave with these locked. */
	ast_channel_lock(owner_real);
	ast_mutex_lock(&p->lock);

	ast_channel_unref(owner_real);
	ast_channel_unref(owner_3way);

	return res;
}

static int check_for_conference(struct dahdi_pvt *p)
{
	struct dahdi_confinfo ci;
	/* Fine if we already have a master, etc */
	if (p->master || (p->confno > -1))
		return 0;
	memset(&ci, 0, sizeof(ci));
	if (ioctl(p->subs[SUB_REAL].dfd, DAHDI_GETCONF, &ci)) {
		ast_log(LOG_WARNING, "Failed to get conference info on channel %d: %s\n", p->channel, strerror(errno));
		return 0;
	}
	/* If we have no master and don't have a confno, then
	   if we're in a conference, it's probably a MeetMe room or
	   some such, so don't let us 3-way out! */
	if ((p->subs[SUB_REAL].curconf.confno != ci.confno) || (p->subs[SUB_REAL].curconf.confmode != ci.confmode)) {
		ast_verb(3, "Avoiding 3-way call when in an external conference\n");
		return 1;
	}
	return 0;
}

/*! Checks channel for alarms
 * \param p a channel to check for alarms.
 * \returns the alarms on the span to which the channel belongs, or alarms on
 *          the channel if no span alarms.
 */
static int get_alarms(struct dahdi_pvt *p)
{
	int res;
	struct dahdi_spaninfo zi;
	struct dahdi_params params;

	memset(&zi, 0, sizeof(zi));
	zi.spanno = p->span;

	if ((res = ioctl(p->subs[SUB_REAL].dfd, DAHDI_SPANSTAT, &zi)) >= 0) {
		if (zi.alarms != DAHDI_ALARM_NONE)
			return zi.alarms;
	} else {
		ast_log(LOG_WARNING, "Unable to determine alarm on channel %d: %s\n", p->channel, strerror(errno));
		return 0;
	}

	/* No alarms on the span. Check for channel alarms. */
	memset(&params, 0, sizeof(params));
	if ((res = ioctl(p->subs[SUB_REAL].dfd, DAHDI_GET_PARAMS, &params)) >= 0)
		return params.chan_alarms;

	ast_log(LOG_WARNING, "Unable to determine alarm on channel %d\n", p->channel);

	return DAHDI_ALARM_NONE;
}

static void dahdi_handle_dtmf(struct ast_channel *ast, int idx, struct ast_frame **dest)
{
	struct dahdi_pvt *p = ast_channel_tech_pvt(ast);
	struct ast_frame *f = *dest;

	ast_debug(1, "%s DTMF digit: 0x%02X '%c' on %s\n",
		f->frametype == AST_FRAME_DTMF_BEGIN ? "Begin" : "End",
		(unsigned)f->subclass.integer, f->subclass.integer, ast_channel_name(ast));

	if (p->confirmanswer) {
		if (f->frametype == AST_FRAME_DTMF_END) {
			ast_debug(1, "Confirm answer on %s!\n", ast_channel_name(ast));
			/* Upon receiving a DTMF digit, consider this an answer confirmation instead
			   of a DTMF digit */
			p->subs[idx].f.frametype = AST_FRAME_CONTROL;
			p->subs[idx].f.subclass.integer = AST_CONTROL_ANSWER;
			/* Reset confirmanswer so DTMF's will behave properly for the duration of the call */
			p->confirmanswer = 0;
		} else {
			p->subs[idx].f.frametype = AST_FRAME_NULL;
			p->subs[idx].f.subclass.integer = 0;
		}
		*dest = &p->subs[idx].f;
	} else if (p->callwaitcas) {
		if (f->frametype == AST_FRAME_DTMF_END) {
			if ((f->subclass.integer == 'A') || (f->subclass.integer == 'D')) {
				ast_debug(1, "Got some DTMF, but it's for the CAS\n");
				ast_free(p->cidspill);
				p->cidspill = NULL;
				send_cwcidspill(p);
			}
			p->callwaitcas = 0;
		}
		p->subs[idx].f.frametype = AST_FRAME_NULL;
		p->subs[idx].f.subclass.integer = 0;
		*dest = &p->subs[idx].f;
	} else if (f->subclass.integer == 'f') {
		if (f->frametype == AST_FRAME_DTMF_END) {
			/* Fax tone -- Handle and return NULL */
			if ((p->callprogress & CALLPROGRESS_FAX) && !p->faxhandled) {
				/* If faxbuffers are configured, use them for the fax transmission */
				if (p->usefaxbuffers && !p->bufferoverrideinuse) {
					struct dahdi_bufferinfo bi = {
						.txbufpolicy = p->faxbuf_policy,
						.bufsize = p->bufsize,
						.numbufs = p->faxbuf_no
					};
					int res;

					if ((res = ioctl(p->subs[idx].dfd, DAHDI_SET_BUFINFO, &bi)) < 0) {
						ast_log(LOG_WARNING, "Channel '%s' unable to set buffer policy, reason: %s\n", ast_channel_name(ast), strerror(errno));
					} else {
						p->bufferoverrideinuse = 1;
					}
				}
				p->faxhandled = 1;
				if (p->dsp) {
					p->dsp_features &= ~DSP_FEATURE_FAX_DETECT;
					ast_dsp_set_features(p->dsp, p->dsp_features);
					ast_debug(1, "Disabling FAX tone detection on %s after tone received\n", ast_channel_name(ast));
				}
				if (strcmp(ast_channel_exten(ast), "fax")) {
					const char *target_context = ast_channel_context(ast);

					/*
					 * We need to unlock 'ast' here because ast_exists_extension has the
					 * potential to start autoservice on the channel. Such action is prone
					 * to deadlock if the channel is locked.
					 *
					 * ast_async_goto() has its own restriction on not holding the
					 * channel lock.
					 */
					ast_mutex_unlock(&p->lock);
					ast_channel_unlock(ast);
					if (ast_exists_extension(ast, target_context, "fax", 1,
						S_COR(ast_channel_caller(ast)->id.number.valid, ast_channel_caller(ast)->id.number.str, NULL))) {
						ast_verb(3, "Redirecting %s to fax extension\n", ast_channel_name(ast));
						/* Save the DID/DNIS when we transfer the fax call to a "fax" extension */
						pbx_builtin_setvar_helper(ast, "FAXEXTEN", ast_channel_exten(ast));
						if (ast_async_goto(ast, target_context, "fax", 1))
							ast_log(LOG_WARNING, "Failed to async goto '%s' into fax of '%s'\n", ast_channel_name(ast), target_context);
					} else {
						ast_log(LOG_NOTICE, "Fax detected, but no fax extension\n");
					}
					ast_channel_lock(ast);
					ast_mutex_lock(&p->lock);
				} else {
					ast_debug(1, "Already in a fax extension, not redirecting\n");
				}
			} else {
				ast_debug(1, "Fax already handled\n");
			}
			dahdi_confmute(p, 0);
		}
		p->subs[idx].f.frametype = AST_FRAME_NULL;
		p->subs[idx].f.subclass.integer = 0;
		*dest = &p->subs[idx].f;
	}
}

static void publish_span_alarm(int span, const char *alarm_txt)
{
	RAII_VAR(struct ast_json *, body, NULL, ast_json_unref);

	body = ast_json_pack("{s: i, s: s}",
		"Span", span,
		"Alarm", alarm_txt);
	if (!body) {
		return;
	}

	ast_manager_publish_event("SpanAlarm", EVENT_FLAG_SYSTEM, body);
}

static void publish_channel_alarm(int channel, const char *alarm_txt)
{
	RAII_VAR(struct ast_json *, body, NULL, ast_json_unref);
	RAII_VAR(struct ast_str *, dahdi_chan, ast_str_create(32), ast_free);
	if (!dahdi_chan) {
		return;
	}

	ast_str_set(&dahdi_chan, 0, "%d", channel);
	body = ast_json_pack("{s: s, s: s}",
		"DAHDIChannel", ast_str_buffer(dahdi_chan),
		"Alarm", alarm_txt);
	if (!body) {
		return;
	}

	ast_manager_publish_event("Alarm", EVENT_FLAG_SYSTEM, body);
}

static void handle_alarms(struct dahdi_pvt *p, int alms)
{
	const char *alarm_str;

#if defined(HAVE_PRI)
	if (dahdi_sig_pri_lib_handles(p->sig) && sig_pri_is_alarm_ignored(p->pri)) {
		return;
	}
#endif	/* defined(HAVE_PRI) */

	alarm_str = alarm2str(alms);
	if (report_alarms & REPORT_CHANNEL_ALARMS) {
		ast_log(LOG_WARNING, "Detected alarm on channel %d: %s\n", p->channel, alarm_str);
		publish_channel_alarm(p->channel, alarm_str);
	}

	if (report_alarms & REPORT_SPAN_ALARMS && p->manages_span_alarms) {
		ast_log(LOG_WARNING, "Detected alarm on span %d: %s\n", p->span, alarm_str);
		publish_span_alarm(p->span, alarm_str);
	}
}

static struct ast_frame *dahdi_handle_event(struct ast_channel *ast)
{
	int res, x;
	int idx, mysig;
	char *c;
	struct dahdi_pvt *p = ast_channel_tech_pvt(ast);
	pthread_t threadid;
	struct ast_channel *chan;
	struct ast_frame *f;

	idx = dahdi_get_index(ast, p, 0);
	if (idx < 0) {
		return &ast_null_frame;
	}
	mysig = p->sig;
	if (p->outsigmod > -1)
		mysig = p->outsigmod;
	p->subs[idx].f.frametype = AST_FRAME_NULL;
	p->subs[idx].f.subclass.integer = 0;
	p->subs[idx].f.datalen = 0;
	p->subs[idx].f.samples = 0;
	p->subs[idx].f.mallocd = 0;
	p->subs[idx].f.offset = 0;
	p->subs[idx].f.src = "dahdi_handle_event";
	p->subs[idx].f.data.ptr = NULL;
	f = &p->subs[idx].f;

	if (p->fake_event) {
		res = p->fake_event;
		p->fake_event = 0;
	} else
		res = dahdi_get_event(p->subs[idx].dfd);

	ast_debug(1, "Got event %s(%d) on channel %d (index %d)\n", event2str(res), res, p->channel, idx);

	if (res & (DAHDI_EVENT_PULSEDIGIT | DAHDI_EVENT_DTMFUP)) {
		p->pulsedial = (res & DAHDI_EVENT_PULSEDIGIT) ? 1 : 0;
		ast_debug(1, "Detected %sdigit '%c'\n", p->pulsedial ? "pulse ": "", res & 0xff);
#if defined(HAVE_PRI)
		if (dahdi_sig_pri_lib_handles(p->sig)
			&& ((struct sig_pri_chan *) p->sig_pvt)->call_level < SIG_PRI_CALL_LEVEL_PROCEEDING
			&& p->pri
			&& (p->pri->overlapdial & DAHDI_OVERLAPDIAL_INCOMING)) {
			/* absorb event */
		} else
#endif	/* defined(HAVE_PRI) */
		{
			/* Unmute conference */
			dahdi_confmute(p, 0);
			p->subs[idx].f.frametype = AST_FRAME_DTMF_END;
			p->subs[idx].f.subclass.integer = res & 0xff;
			dahdi_handle_dtmf(ast, idx, &f);
		}
		return f;
	}

	if (res & DAHDI_EVENT_DTMFDOWN) {
		ast_debug(1, "DTMF Down '%c'\n", res & 0xff);
#if defined(HAVE_PRI)
		if (dahdi_sig_pri_lib_handles(p->sig)
			&& ((struct sig_pri_chan *) p->sig_pvt)->call_level < SIG_PRI_CALL_LEVEL_PROCEEDING
			&& p->pri
			&& (p->pri->overlapdial & DAHDI_OVERLAPDIAL_INCOMING)) {
			/* absorb event */
		} else
#endif	/* defined(HAVE_PRI) */
		{
			/* Mute conference */
			dahdi_confmute(p, 1);
			p->subs[idx].f.frametype = AST_FRAME_DTMF_BEGIN;
			p->subs[idx].f.subclass.integer = res & 0xff;
			dahdi_handle_dtmf(ast, idx, &f);
		}
		return &p->subs[idx].f;
	}

	switch (res) {
	case DAHDI_EVENT_EC_DISABLED:
		ast_verb(3, "Channel %d echo canceler disabled.\n", p->channel);
		p->echocanon = 0;
		break;
#ifdef HAVE_DAHDI_ECHOCANCEL_FAX_MODE
	case DAHDI_EVENT_TX_CED_DETECTED:
		ast_verb(3, "Channel %d detected a CED tone towards the network.\n", p->channel);
		break;
	case DAHDI_EVENT_RX_CED_DETECTED:
		ast_verb(3, "Channel %d detected a CED tone from the network.\n", p->channel);
		break;
	case DAHDI_EVENT_EC_NLP_DISABLED:
		ast_verb(3, "Channel %d echo canceler disabled its NLP.\n", p->channel);
		break;
	case DAHDI_EVENT_EC_NLP_ENABLED:
		ast_verb(3, "Channel %d echo canceler enabled its NLP.\n", p->channel);
		break;
#endif
	case DAHDI_EVENT_BITSCHANGED:
#ifdef HAVE_OPENR2
		if (p->sig != SIG_MFCR2) {
			ast_log(LOG_WARNING, "Received bits changed on %s signalling?\n", sig2str(p->sig));
		} else {
			ast_debug(1, "bits changed in chan %d\n", p->channel);
			openr2_chan_handle_cas(p->r2chan);
		}
#else
		ast_log(LOG_WARNING, "Received bits changed on %s signalling?\n", sig2str(p->sig));
#endif
		break;
	case DAHDI_EVENT_PULSE_START:
		/* Stop tone if there's a pulse start and the PBX isn't started */
		if (!ast_channel_pbx(ast))
			tone_zone_play_tone(p->subs[idx].dfd, -1);
		break;
	case DAHDI_EVENT_DIALCOMPLETE:
		/* DAHDI has completed dialing all digits sent using DAHDI_DIAL. */
#if defined(HAVE_PRI)
		if (dahdi_sig_pri_lib_handles(p->sig)) {
			if (p->inalarm) {
				break;
			}
			if (ioctl(p->subs[idx].dfd, DAHDI_DIALING, &x) == -1) {
				ast_debug(1, "DAHDI_DIALING ioctl failed on %s: %s\n",
					ast_channel_name(ast), strerror(errno));
				return NULL;
			}
			if (x) {
				/* Still dialing in DAHDI driver */
				break;
			}
			/*
			 * The ast channel is locked and the private may be locked more
			 * than once.
			 */
			sig_pri_dial_complete(p->sig_pvt, ast);
			break;
		}
#endif	/* defined(HAVE_PRI) */
#ifdef HAVE_OPENR2
		if ((p->sig & SIG_MFCR2) && p->r2chan && ast_channel_state(ast) != AST_STATE_UP) {
			/* we don't need to do anything for this event for R2 signaling
			   if the call is being setup */
			break;
		}
#endif
		if (p->inalarm) break;
		if ((p->radio || (p->oprmode < 0))) break;
		if (ioctl(p->subs[idx].dfd,DAHDI_DIALING,&x) == -1) {
			ast_debug(1, "DAHDI_DIALING ioctl failed on %s: %s\n",ast_channel_name(ast), strerror(errno));
			return NULL;
		}
		if (!x) { /* if not still dialing in driver */
			dahdi_ec_enable(p);
			if (p->echobreak) {
				dahdi_train_ec(p);
				ast_copy_string(p->dop.dialstr, p->echorest, sizeof(p->dop.dialstr));
				p->dop.op = DAHDI_DIAL_OP_REPLACE;
				res = dahdi_dial_str(p, p->dop.op, p->dop.dialstr);
				p->echobreak = 0;
			} else {
				p->dialing = 0;
				if ((mysig == SIG_E911) || (mysig == SIG_FGC_CAMA) || (mysig == SIG_FGC_CAMAMF)) {
					/* if thru with dialing after offhook */
					if (ast_channel_state(ast) == AST_STATE_DIALING_OFFHOOK) {
						ast_setstate(ast, AST_STATE_UP);
						p->subs[idx].f.frametype = AST_FRAME_CONTROL;
						p->subs[idx].f.subclass.integer = AST_CONTROL_ANSWER;
						break;
					} else { /* if to state wait for offhook to dial rest */
						/* we now wait for off hook */
						ast_setstate(ast,AST_STATE_DIALING_OFFHOOK);
					}
				}
				if (ast_channel_state(ast) == AST_STATE_DIALING) {
					if ((p->callprogress & CALLPROGRESS_PROGRESS) && CANPROGRESSDETECT(p) && p->dsp && p->outgoing) {
						ast_debug(1, "Done dialing, but waiting for progress detection before doing more...\n");
					} else if (p->confirmanswer || (!p->dialednone
						&& ((mysig == SIG_EM) || (mysig == SIG_EM_E1)
							|| (mysig == SIG_EMWINK) || (mysig == SIG_FEATD)
							|| (mysig == SIG_FEATDMF_TA) || (mysig == SIG_FEATDMF)
							|| (mysig == SIG_E911) || (mysig == SIG_FGC_CAMA)
							|| (mysig == SIG_FGC_CAMAMF) || (mysig == SIG_FEATB)
							|| (mysig == SIG_SF) || (mysig == SIG_SFWINK)
							|| (mysig == SIG_SF_FEATD) || (mysig == SIG_SF_FEATDMF)
							|| (mysig == SIG_SF_FEATB)))) {
						ast_setstate(ast, AST_STATE_RINGING);
					} else if (!p->answeronpolarityswitch) {
						ast_setstate(ast, AST_STATE_UP);
						p->subs[idx].f.frametype = AST_FRAME_CONTROL;
						p->subs[idx].f.subclass.integer = AST_CONTROL_ANSWER;
						/* If aops=0 and hops=1, this is necessary */
						p->polarity = POLARITY_REV;
					} else {
						/* Start clean, so we can catch the change to REV polarity when party answers */
						p->polarity = POLARITY_IDLE;
					}
				}
			}
		}
		break;
	case DAHDI_EVENT_ALARM:
		switch (p->sig) {
#if defined(HAVE_PRI)
		case SIG_PRI_LIB_HANDLE_CASES:
			sig_pri_chan_alarm_notify(p->sig_pvt, 0);
			break;
#endif	/* defined(HAVE_PRI) */
#if defined(HAVE_SS7)
		case SIG_SS7:
			sig_ss7_set_alarm(p->sig_pvt, 1);
			break;
#endif	/* defined(HAVE_SS7) */
		default:
			p->inalarm = 1;
			break;
		}
		res = get_alarms(p);
		handle_alarms(p, res);
#ifdef HAVE_PRI
		if (!p->pri || !p->pri->pri || pri_get_timer(p->pri->pri, PRI_TIMER_T309) < 0) {
			/* fall through intentionally */
		} else {
			break;
		}
#endif
#if defined(HAVE_SS7)
		if (p->sig == SIG_SS7)
			break;
#endif	/* defined(HAVE_SS7) */
#ifdef HAVE_OPENR2
		if (p->sig == SIG_MFCR2)
			break;
#endif
	case DAHDI_EVENT_ONHOOK:
		if (p->radio) {
			p->subs[idx].f.frametype = AST_FRAME_CONTROL;
			p->subs[idx].f.subclass.integer = AST_CONTROL_RADIO_UNKEY;
			break;
		}
		if (p->oprmode < 0)
		{
			if (p->oprmode != -1) { /* Operator flash recall */
				ast_verb(4, "Operator mode enabled on channel %d, holding line for channel %d\n", p->channel, p->oprpeer->channel);
				break;
			}
			/* Otherwise, immediate recall */
			if ((p->sig == SIG_FXOLS) || (p->sig == SIG_FXOKS) || (p->sig == SIG_FXOGS))
			{
				/* Make sure it starts ringing */
				dahdi_set_hook(p->subs[SUB_REAL].dfd, DAHDI_RINGOFF);
				dahdi_set_hook(p->subs[SUB_REAL].dfd, DAHDI_RING);
				save_conference(p->oprpeer);
				tone_zone_play_tone(p->oprpeer->subs[SUB_REAL].dfd, DAHDI_TONE_RINGTONE);
				ast_verb(4, "Operator recall, channel %d ringing back channel %d\n", p->oprpeer->channel, p->channel);
			}
			break;
		}
		switch (p->sig) {
		case SIG_FXOLS:
		case SIG_FXOGS:
		case SIG_FXOKS:
			/* Check for some special conditions regarding call waiting */
			if (idx == SUB_REAL) {
				/* The normal line was hung up */
				if (p->subs[SUB_CALLWAIT].owner) {
					/* There's a call waiting call, so ring the phone, but make it unowned in the mean time */
					swap_subs(p, SUB_CALLWAIT, SUB_REAL);
					ast_verb(3, "Channel %d still has (callwait) call, ringing phone\n", p->channel);
					unalloc_sub(p, SUB_CALLWAIT);
#if 0
					p->subs[idx].needanswer = 0;
					p->subs[idx].needringing = 0;
#endif
					p->callwaitingrepeat = 0;
					p->cidcwexpire = 0;
					p->cid_suppress_expire = 0;
					p->owner = NULL;
					/* Don't start streaming audio yet if the incoming call isn't up yet */
					if (ast_channel_state(p->subs[SUB_REAL].owner) != AST_STATE_UP)
						p->dialing = 1;
					dahdi_ring_phone(p);
				} else if (p->subs[SUB_THREEWAY].owner) {
					unsigned int mssinceflash;
					/* Here we have to retain the lock on both the main channel, the 3-way channel, and
					   the private structure -- not especially easy or clean */
					while (p->subs[SUB_THREEWAY].owner && ast_channel_trylock(p->subs[SUB_THREEWAY].owner)) {
						/* Yuck, didn't get the lock on the 3-way, gotta release everything and re-grab! */
						DLA_UNLOCK(&p->lock);
						CHANNEL_DEADLOCK_AVOIDANCE(ast);
						/* We can grab ast and p in that order, without worry.  We should make sure
						   nothing seriously bad has happened though like some sort of bizarre double
						   masquerade! */
						DLA_LOCK(&p->lock);
						if (p->owner != ast) {
							ast_log(LOG_WARNING, "This isn't good...\n");
							return NULL;
						}
					}
					if (!p->subs[SUB_THREEWAY].owner) {
						ast_log(LOG_NOTICE, "Whoa, threeway disappeared kinda randomly.\n");
						return NULL;
					}
					mssinceflash = ast_tvdiff_ms(ast_tvnow(), p->flashtime);
					ast_debug(1, "Last flash was %u ms ago\n", mssinceflash);
					if (mssinceflash < MIN_MS_SINCE_FLASH) {
						/* It hasn't been long enough since the last flashook.  This is probably a bounce on
						   hanging up.  Hangup both channels now */
						if (p->subs[SUB_THREEWAY].owner)
							ast_queue_hangup_with_cause(p->subs[SUB_THREEWAY].owner, AST_CAUSE_NO_ANSWER);
						ast_channel_softhangup_internal_flag_add(p->subs[SUB_THREEWAY].owner, AST_SOFTHANGUP_DEV);
						ast_debug(1, "Looks like a bounced flash, hanging up both calls on %d\n", p->channel);
						ast_channel_unlock(p->subs[SUB_THREEWAY].owner);
					} else if ((ast_channel_pbx(ast)) || (ast_channel_state(ast) == AST_STATE_UP)) {
						if (p->transfer) {
							/* In any case this isn't a threeway call anymore */
							p->subs[SUB_REAL].inthreeway = 0;
							p->subs[SUB_THREEWAY].inthreeway = 0;
							/* Only attempt transfer if the phone is ringing; why transfer to busy tone eh? */
							if (!p->transfertobusy && ast_channel_state(ast) == AST_STATE_BUSY) {
								ast_channel_unlock(p->subs[SUB_THREEWAY].owner);
								/* Swap subs and dis-own channel */
								swap_subs(p, SUB_THREEWAY, SUB_REAL);
								p->owner = NULL;
								/* Ring the phone */
								dahdi_ring_phone(p);
							} else if (!attempt_transfer(p)) {
								/*
								 * Transfer successful.  Don't actually hang up at this point.
								 * Let our channel legs of the calls die off as the transfer
								 * percolates through the core.
								 */
								break;
							}
						} else {
							ast_channel_softhangup_internal_flag_add(p->subs[SUB_THREEWAY].owner, AST_SOFTHANGUP_DEV);
							if (p->subs[SUB_THREEWAY].owner)
								ast_channel_unlock(p->subs[SUB_THREEWAY].owner);
						}
					} else {
						ast_channel_unlock(p->subs[SUB_THREEWAY].owner);
						/* Swap subs and dis-own channel */
						swap_subs(p, SUB_THREEWAY, SUB_REAL);
						p->owner = NULL;
						/* Ring the phone */
						dahdi_ring_phone(p);
					}
				}
			} else {
				ast_log(LOG_WARNING, "Got a hangup and my index is %d?\n", idx);
			}
			/* Fall through */
		default:
			dahdi_ec_disable(p);
			return NULL;
		}
		break;
	case DAHDI_EVENT_RINGOFFHOOK:
		if (p->inalarm) break;
		if (p->oprmode < 0)
		{
			if ((p->sig == SIG_FXOLS) || (p->sig == SIG_FXOKS) || (p->sig == SIG_FXOGS))
			{
				/* Make sure it stops ringing */
				dahdi_set_hook(p->subs[SUB_REAL].dfd, DAHDI_RINGOFF);
				tone_zone_play_tone(p->oprpeer->subs[SUB_REAL].dfd, -1);
				restore_conference(p->oprpeer);
				ast_debug(1, "Operator recall by channel %d for channel %d complete\n", p->oprpeer->channel, p->channel);
			}
			break;
		}
		if (p->radio)
		{
			p->subs[idx].f.frametype = AST_FRAME_CONTROL;
			p->subs[idx].f.subclass.integer = AST_CONTROL_RADIO_KEY;
			break;
 		}
		/* for E911, its supposed to wait for offhook then dial
		   the second half of the dial string */
		if (((mysig == SIG_E911) || (mysig == SIG_FGC_CAMA) || (mysig == SIG_FGC_CAMAMF)) && (ast_channel_state(ast) == AST_STATE_DIALING_OFFHOOK)) {
			c = strchr(p->dialdest, '/');
			if (c)
				c++;
			else
				c = p->dialdest;

			if (*c) {
				int numchars = snprintf(p->dop.dialstr, sizeof(p->dop.dialstr), "M*0%s#", c);
				if (numchars >= sizeof(p->dop.dialstr)) {
					ast_log(LOG_WARNING, "Dial string '%s' truncated\n", c);
				}
			} else {
				ast_copy_string(p->dop.dialstr,"M*2#", sizeof(p->dop.dialstr));
			}

			if (strlen(p->dop.dialstr) > 4) {
				memset(p->echorest, 'w', sizeof(p->echorest) - 1);
				strcpy(p->echorest + (p->echotraining / 401) + 1, p->dop.dialstr + strlen(p->dop.dialstr) - 2);
				p->echorest[sizeof(p->echorest) - 1] = '\0';
				p->echobreak = 1;
				p->dop.dialstr[strlen(p->dop.dialstr)-2] = '\0';
			} else
				p->echobreak = 0;
			if (dahdi_dial_str(p, p->dop.op, p->dop.dialstr)) {
				x = DAHDI_ONHOOK;
				ioctl(p->subs[SUB_REAL].dfd, DAHDI_HOOK, &x);
				return NULL;
			}
			p->dialing = 1;
			return &p->subs[idx].f;
		}
		switch (p->sig) {
		case SIG_FXOLS:
		case SIG_FXOGS:
		case SIG_FXOKS:
			switch (ast_channel_state(ast)) {
			case AST_STATE_RINGING:
				dahdi_ec_enable(p);
				dahdi_train_ec(p);
				p->subs[idx].f.frametype = AST_FRAME_CONTROL;
				p->subs[idx].f.subclass.integer = AST_CONTROL_ANSWER;
				/* Make sure it stops ringing */
				p->subs[SUB_REAL].needringing = 0;
				dahdi_set_hook(p->subs[idx].dfd, DAHDI_OFFHOOK);
				ast_debug(1, "channel %d answered\n", p->channel);

				/* Cancel any running CallerID spill */
				ast_free(p->cidspill);
				p->cidspill = NULL;
				restore_conference(p);

				p->dialing = 0;
				p->callwaitcas = 0;
				if (p->confirmanswer) {
					/* Ignore answer if "confirm answer" is enabled */
					p->subs[idx].f.frametype = AST_FRAME_NULL;
					p->subs[idx].f.subclass.integer = 0;
				} else if (!ast_strlen_zero(p->dop.dialstr)) {
					/* nick@dccinc.com 4/3/03 - fxo should be able to do deferred dialing */
					res = dahdi_dial_str(p, p->dop.op, p->dop.dialstr);
					if (res) {
						p->dop.dialstr[0] = '\0';
						return NULL;
					} else {
						ast_debug(1, "Sent FXO deferred digit string: %s\n", p->dop.dialstr);
						p->subs[idx].f.frametype = AST_FRAME_NULL;
						p->subs[idx].f.subclass.integer = 0;
						p->dialing = 1;
					}
					p->dop.dialstr[0] = '\0';
					ast_setstate(ast, AST_STATE_DIALING);
				} else
					ast_setstate(ast, AST_STATE_UP);
				return &p->subs[idx].f;
			case AST_STATE_DOWN:
				ast_setstate(ast, AST_STATE_RING);
				ast_channel_rings_set(ast, 1);
				p->subs[idx].f.frametype = AST_FRAME_CONTROL;
				p->subs[idx].f.subclass.integer = AST_CONTROL_OFFHOOK;
				ast_debug(1, "channel %d picked up\n", p->channel);
				return &p->subs[idx].f;
			case AST_STATE_UP:
				/* Make sure it stops ringing */
				dahdi_set_hook(p->subs[idx].dfd, DAHDI_OFFHOOK);
				/* Okay -- probably call waiting*/
				ast_queue_unhold(p->owner);
				p->subs[idx].needunhold = 1;
				break;
			case AST_STATE_RESERVED:
				/* Start up dialtone */
				if (has_voicemail(p))
					res = tone_zone_play_tone(p->subs[SUB_REAL].dfd, DAHDI_TONE_STUTTER);
				else
					res = tone_zone_play_tone(p->subs[SUB_REAL].dfd, DAHDI_TONE_DIALTONE);
				break;
			default:
				ast_log(LOG_WARNING, "FXO phone off hook in weird state %u??\n", ast_channel_state(ast));
			}
			break;
		case SIG_FXSLS:
		case SIG_FXSGS:
		case SIG_FXSKS:
			if (ast_channel_state(ast) == AST_STATE_RING) {
				p->ringt = p->ringt_base;
			}

			/* If we get a ring then we cannot be in
			 * reversed polarity. So we reset to idle */
			ast_debug(1, "Setting IDLE polarity due "
				"to ring. Old polarity was %d\n",
				p->polarity);
			p->polarity = POLARITY_IDLE;

			/* Fall through */
		case SIG_EM:
		case SIG_EM_E1:
		case SIG_EMWINK:
		case SIG_FEATD:
		case SIG_FEATDMF:
		case SIG_FEATDMF_TA:
		case SIG_E911:
		case SIG_FGC_CAMA:
		case SIG_FGC_CAMAMF:
		case SIG_FEATB:
		case SIG_SF:
		case SIG_SFWINK:
		case SIG_SF_FEATD:
		case SIG_SF_FEATDMF:
		case SIG_SF_FEATB:
			if (ast_channel_state(ast) == AST_STATE_PRERING)
				ast_setstate(ast, AST_STATE_RING);
			if ((ast_channel_state(ast) == AST_STATE_DOWN) || (ast_channel_state(ast) == AST_STATE_RING)) {
				ast_debug(1, "Ring detected\n");
				p->subs[idx].f.frametype = AST_FRAME_CONTROL;
				p->subs[idx].f.subclass.integer = AST_CONTROL_RING;
			} else if (p->outgoing && ((ast_channel_state(ast) == AST_STATE_RINGING) || (ast_channel_state(ast) == AST_STATE_DIALING))) {
				ast_debug(1, "Line answered\n");
				if (p->confirmanswer) {
					p->subs[idx].f.frametype = AST_FRAME_NULL;
					p->subs[idx].f.subclass.integer = 0;
				} else {
					p->subs[idx].f.frametype = AST_FRAME_CONTROL;
					p->subs[idx].f.subclass.integer = AST_CONTROL_ANSWER;
					ast_setstate(ast, AST_STATE_UP);
				}
			} else if (ast_channel_state(ast) != AST_STATE_RING)
				ast_log(LOG_WARNING, "Ring/Off-hook in strange state %u on channel %d\n", ast_channel_state(ast), p->channel);
			break;
		default:
			ast_log(LOG_WARNING, "Don't know how to handle ring/off hook for signalling %d\n", p->sig);
		}
		break;
	case DAHDI_EVENT_RINGBEGIN:
		switch (p->sig) {
		case SIG_FXSLS:
		case SIG_FXSGS:
		case SIG_FXSKS:
			if (ast_channel_state(ast) == AST_STATE_RING) {
				p->ringt = p->ringt_base;
			}
			break;
		}
		break;
	case DAHDI_EVENT_RINGERON:
		break;
	case DAHDI_EVENT_NOALARM:
		switch (p->sig) {
#if defined(HAVE_PRI)
		case SIG_PRI_LIB_HANDLE_CASES:
			sig_pri_chan_alarm_notify(p->sig_pvt, 1);
			break;
#endif	/* defined(HAVE_PRI) */
#if defined(HAVE_SS7)
		case SIG_SS7:
			sig_ss7_set_alarm(p->sig_pvt, 0);
			break;
#endif	/* defined(HAVE_SS7) */
		default:
			p->inalarm = 0;
			break;
		}
		handle_clear_alarms(p);
		break;
	case DAHDI_EVENT_WINKFLASH:
		if (p->inalarm) break;
		if (p->radio) break;
		if (p->oprmode < 0) break;
		if (p->oprmode > 1)
		{
			struct dahdi_params par;

			memset(&par, 0, sizeof(par));
			if (ioctl(p->oprpeer->subs[SUB_REAL].dfd, DAHDI_GET_PARAMS, &par) != -1)
			{
				if (!par.rxisoffhook)
				{
					/* Make sure it stops ringing */
					dahdi_set_hook(p->oprpeer->subs[SUB_REAL].dfd, DAHDI_RINGOFF);
					dahdi_set_hook(p->oprpeer->subs[SUB_REAL].dfd, DAHDI_RING);
					save_conference(p);
					tone_zone_play_tone(p->subs[SUB_REAL].dfd, DAHDI_TONE_RINGTONE);
					ast_verb(4, "Operator flash recall, channel %d ringing back channel %d\n", p->oprpeer->channel, p->channel);
				}
			}
			break;
		}
		/* Remember last time we got a flash-hook */
		p->flashtime = ast_tvnow();
		switch (mysig) {
		case SIG_FXOLS:
		case SIG_FXOGS:
		case SIG_FXOKS:
			ast_debug(1, "Winkflash, index: %d, normal: %d, callwait: %d, thirdcall: %d\n",
				idx, p->subs[SUB_REAL].dfd, p->subs[SUB_CALLWAIT].dfd, p->subs[SUB_THREEWAY].dfd);

			/* Cancel any running CallerID spill */
			ast_free(p->cidspill);
			p->cidspill = NULL;
			restore_conference(p);
			p->callwaitcas = 0;

			if (idx != SUB_REAL) {
				ast_log(LOG_WARNING, "Got flash hook with index %d on channel %d?!?\n", idx, p->channel);
				goto winkflashdone;
			}

			if (p->subs[SUB_CALLWAIT].owner) {
				/* Swap to call-wait */
				swap_subs(p, SUB_REAL, SUB_CALLWAIT);
				tone_zone_play_tone(p->subs[SUB_REAL].dfd, -1);
				p->owner = p->subs[SUB_REAL].owner;
				ast_debug(1, "Making %s the new owner\n", ast_channel_name(p->owner));
				if (ast_channel_state(p->owner) == AST_STATE_RINGING) {
					ast_setstate(p->owner, AST_STATE_UP);
					p->subs[SUB_REAL].needanswer = 1;
				}
				p->callwaitingrepeat = 0;
				p->cidcwexpire = 0;
				p->cid_suppress_expire = 0;
				/* Start music on hold if appropriate */
				if (!p->subs[SUB_CALLWAIT].inthreeway) {
					ast_queue_hold(p->subs[SUB_CALLWAIT].owner, p->mohsuggest);
				}
				p->subs[SUB_CALLWAIT].needhold = 1;
				ast_queue_hold(p->subs[SUB_REAL].owner, p->mohsuggest);
				p->subs[SUB_REAL].needunhold = 1;
			} else if (!p->subs[SUB_THREEWAY].owner) {
				if (!p->threewaycalling) {
					/* Just send a flash if no 3-way calling */
					p->subs[SUB_REAL].needflash = 1;
					goto winkflashdone;
				} else if (!check_for_conference(p)) {
					ast_callid callid = 0;
					int callid_created;
					char cid_num[256];
					char cid_name[256];

					cid_num[0] = 0;
					cid_name[0] = 0;
					if (p->dahditrcallerid && p->owner) {
						if (ast_channel_caller(p->owner)->id.number.valid
							&& ast_channel_caller(p->owner)->id.number.str) {
							ast_copy_string(cid_num, ast_channel_caller(p->owner)->id.number.str,
								sizeof(cid_num));
						}
						if (ast_channel_caller(p->owner)->id.name.valid
							&& ast_channel_caller(p->owner)->id.name.str) {
							ast_copy_string(cid_name, ast_channel_caller(p->owner)->id.name.str,
								sizeof(cid_name));
						}
					}
					/* XXX This section needs much more error checking!!! XXX */
					/* Start a 3-way call if feasible */
					if (!((ast_channel_pbx(ast)) ||
						(ast_channel_state(ast) == AST_STATE_UP) ||
						(ast_channel_state(ast) == AST_STATE_RING))) {
						ast_debug(1, "Flash when call not up or ringing\n");
						goto winkflashdone;
					}
					if (alloc_sub(p, SUB_THREEWAY)) {
						ast_log(LOG_WARNING, "Unable to allocate three-way subchannel\n");
						goto winkflashdone;
					}
					callid_created = ast_callid_threadstorage_auto(&callid);
					/*
					 * Make new channel
					 *
					 * We cannot hold the p or ast locks while creating a new
					 * channel.
					 */
					ast_mutex_unlock(&p->lock);
					ast_channel_unlock(ast);
					chan = dahdi_new(p, AST_STATE_RESERVED, 0, SUB_THREEWAY, 0, NULL, NULL, callid);
					ast_channel_lock(ast);
					ast_mutex_lock(&p->lock);
					if (p->dahditrcallerid) {
						if (!p->origcid_num)
							p->origcid_num = ast_strdup(p->cid_num);
						if (!p->origcid_name)
							p->origcid_name = ast_strdup(p->cid_name);
						ast_copy_string(p->cid_num, cid_num, sizeof(p->cid_num));
						ast_copy_string(p->cid_name, cid_name, sizeof(p->cid_name));
					}
					/* Swap things around between the three-way and real call */
					swap_subs(p, SUB_THREEWAY, SUB_REAL);
					/* Disable echo canceller for better dialing */
					dahdi_ec_disable(p);
					res = tone_zone_play_tone(p->subs[SUB_REAL].dfd, DAHDI_TONE_DIALRECALL);
					if (res)
						ast_log(LOG_WARNING, "Unable to start dial recall tone on channel %d\n", p->channel);
					p->owner = chan;
					if (!chan) {
						ast_log(LOG_WARNING, "Cannot allocate new structure on channel %d\n", p->channel);
					} else if (ast_pthread_create_detached(&threadid, NULL, analog_ss_thread, chan)) {
						ast_log(LOG_WARNING, "Unable to start simple switch on channel %d\n", p->channel);
						res = tone_zone_play_tone(p->subs[SUB_REAL].dfd, DAHDI_TONE_CONGESTION);
						dahdi_ec_enable(p);
						ast_hangup(chan);
					} else {
						ast_verb(3, "Started three way call on channel %d\n", p->channel);

						/* Start music on hold */
						ast_queue_hold(p->subs[SUB_THREEWAY].owner, p->mohsuggest);
						p->subs[SUB_THREEWAY].needhold = 1;
					}
					ast_callid_threadstorage_auto_clean(callid, callid_created);
				}
			} else {
				/* Already have a 3 way call */
				if (p->subs[SUB_THREEWAY].inthreeway) {
					/* Call is already up, drop the last person */
					ast_debug(1, "Got flash with three way call up, dropping last call on %d\n", p->channel);
					/* If the primary call isn't answered yet, use it */
					if ((ast_channel_state(p->subs[SUB_REAL].owner) != AST_STATE_UP) && (ast_channel_state(p->subs[SUB_THREEWAY].owner) == AST_STATE_UP)) {
						/* Swap back -- we're dropping the real 3-way that isn't finished yet*/
						swap_subs(p, SUB_THREEWAY, SUB_REAL);
						p->owner = p->subs[SUB_REAL].owner;
					}
					/* Drop the last call and stop the conference */
					ast_verb(3, "Dropping three-way call on %s\n", ast_channel_name(p->subs[SUB_THREEWAY].owner));
					ast_channel_softhangup_internal_flag_add(p->subs[SUB_THREEWAY].owner, AST_SOFTHANGUP_DEV);
					p->subs[SUB_REAL].inthreeway = 0;
					p->subs[SUB_THREEWAY].inthreeway = 0;
				} else {
					/* Lets see what we're up to */
					if (((ast_channel_pbx(ast)) || (ast_channel_state(ast) == AST_STATE_UP)) &&
						(p->transfertobusy || (ast_channel_state(ast) != AST_STATE_BUSY))) {
						int otherindex = SUB_THREEWAY;

						ast_verb(3, "Building conference call with %s and %s\n",
							ast_channel_name(p->subs[SUB_THREEWAY].owner),
							ast_channel_name(p->subs[SUB_REAL].owner));
						/* Put them in the threeway, and flip */
						p->subs[SUB_THREEWAY].inthreeway = 1;
						p->subs[SUB_REAL].inthreeway = 1;
						if (ast_channel_state(ast) == AST_STATE_UP) {
							swap_subs(p, SUB_THREEWAY, SUB_REAL);
							otherindex = SUB_REAL;
						}
						if (p->subs[otherindex].owner) {
							ast_queue_unhold(p->subs[otherindex].owner);
						}
						p->subs[otherindex].needunhold = 1;
						p->owner = p->subs[SUB_REAL].owner;
					} else {
						ast_verb(3, "Dumping incomplete call on %s\n", ast_channel_name(p->subs[SUB_THREEWAY].owner));
						swap_subs(p, SUB_THREEWAY, SUB_REAL);
						ast_channel_softhangup_internal_flag_add(p->subs[SUB_THREEWAY].owner, AST_SOFTHANGUP_DEV);
						p->owner = p->subs[SUB_REAL].owner;
						if (p->subs[SUB_REAL].owner) {
							ast_queue_unhold(p->subs[SUB_REAL].owner);
						}
						p->subs[SUB_REAL].needunhold = 1;
						dahdi_ec_enable(p);
					}
				}
			}
winkflashdone:
			dahdi_conf_update(p);
			break;
		case SIG_EM:
		case SIG_EM_E1:
		case SIG_FEATD:
		case SIG_SF:
		case SIG_SFWINK:
		case SIG_SF_FEATD:
		case SIG_FXSLS:
		case SIG_FXSGS:
			if (p->dialing)
				ast_debug(1, "Ignoring wink on channel %d\n", p->channel);
			else
				ast_debug(1, "Got wink in weird state %u on channel %d\n", ast_channel_state(ast), p->channel);
			break;
		case SIG_FEATDMF_TA:
			switch (p->whichwink) {
			case 0:
				ast_debug(1, "ANI2 set to '%d' and ANI is '%s'\n", ast_channel_caller(p->owner)->ani2,
					S_COR(ast_channel_caller(p->owner)->ani.number.valid,
						ast_channel_caller(p->owner)->ani.number.str, ""));
				snprintf(p->dop.dialstr, sizeof(p->dop.dialstr), "M*%d%s#",
					ast_channel_caller(p->owner)->ani2,
					S_COR(ast_channel_caller(p->owner)->ani.number.valid,
						ast_channel_caller(p->owner)->ani.number.str, ""));
				break;
			case 1:
				ast_copy_string(p->dop.dialstr, p->finaldial, sizeof(p->dop.dialstr));
				break;
			case 2:
				ast_log(LOG_WARNING, "Received unexpected wink on channel of type SIG_FEATDMF_TA\n");
				return NULL;
			}
			p->whichwink++;
			/* Fall through */
		case SIG_FEATDMF:
		case SIG_E911:
		case SIG_FGC_CAMAMF:
		case SIG_FGC_CAMA:
		case SIG_FEATB:
		case SIG_SF_FEATDMF:
		case SIG_SF_FEATB:
		case SIG_EMWINK:
			/* FGD MF and EMWINK *Must* wait for wink */
			if (!ast_strlen_zero(p->dop.dialstr)) {
				res = dahdi_dial_str(p, p->dop.op, p->dop.dialstr);
				if (res) {
					p->dop.dialstr[0] = '\0';
					return NULL;
				} else
					ast_debug(1, "Sent deferred digit string: %s\n", p->dop.dialstr);
			}
			p->dop.dialstr[0] = '\0';
			break;
		default:
			ast_log(LOG_WARNING, "Don't know how to handle ring/off hook for signalling %d\n", p->sig);
		}
		break;
	case DAHDI_EVENT_HOOKCOMPLETE:
		if (p->inalarm) break;
		if ((p->radio || (p->oprmode < 0))) break;
		if (p->waitingfordt.tv_sec) break;
		switch (mysig) {
		case SIG_FXSLS:  /* only interesting for FXS */
		case SIG_FXSGS:
		case SIG_FXSKS:
		case SIG_EM:
		case SIG_EM_E1:
		case SIG_EMWINK:
		case SIG_FEATD:
		case SIG_SF:
		case SIG_SFWINK:
		case SIG_SF_FEATD:
			if (!ast_strlen_zero(p->dop.dialstr)) {
				res = dahdi_dial_str(p, p->dop.op, p->dop.dialstr);
				if (res) {
					p->dop.dialstr[0] = '\0';
					return NULL;
				} else
					ast_debug(1, "Sent deferred digit string: %s\n", p->dop.dialstr);
			}
			p->dop.dialstr[0] = '\0';
			p->dop.op = DAHDI_DIAL_OP_REPLACE;
			break;
		case SIG_FEATDMF:
		case SIG_FEATDMF_TA:
		case SIG_E911:
		case SIG_FGC_CAMA:
		case SIG_FGC_CAMAMF:
		case SIG_FEATB:
		case SIG_SF_FEATDMF:
		case SIG_SF_FEATB:
			ast_debug(1, "Got hook complete in MF FGD, waiting for wink now on channel %d\n",p->channel);
			break;
		default:
			break;
		}
		break;
	case DAHDI_EVENT_POLARITY:
		/*
		 * If we get a Polarity Switch event, check to see
		 * if we should change the polarity state and
		 * mark the channel as UP or if this is an indication
		 * of remote end disconnect.
		 */
		if (p->polarity == POLARITY_IDLE) {
			p->polarity = POLARITY_REV;
			if (p->answeronpolarityswitch &&
				((ast_channel_state(ast) == AST_STATE_DIALING) ||
				(ast_channel_state(ast) == AST_STATE_RINGING))) {
				ast_debug(1, "Answering on polarity switch!\n");
				ast_setstate(p->owner, AST_STATE_UP);
				if (p->hanguponpolarityswitch) {
					p->polaritydelaytv = ast_tvnow();
				}
			} else
				ast_debug(1, "Ignore switch to REVERSED Polarity on channel %d, state %u\n", p->channel, ast_channel_state(ast));
		}
		/* Removed else statement from here as it was preventing hangups from ever happening*/
		/* Added AST_STATE_RING in if statement below to deal with calling party hangups that take place when ringing */
		if (p->hanguponpolarityswitch &&
			(p->polarityonanswerdelay > 0) &&
			(p->polarity == POLARITY_REV) &&
			((ast_channel_state(ast) == AST_STATE_UP) || (ast_channel_state(ast) == AST_STATE_RING)) ) {
			/* Added log_debug information below to provide a better indication of what is going on */
			ast_debug(1, "Polarity Reversal event occurred - DEBUG 1: channel %d, state %u, pol= %d, aonp= %d, honp= %d, pdelay= %d, tv= %" PRIi64 "\n", p->channel, ast_channel_state(ast), p->polarity, p->answeronpolarityswitch, p->hanguponpolarityswitch, p->polarityonanswerdelay, ast_tvdiff_ms(ast_tvnow(), p->polaritydelaytv) );

			if (ast_tvdiff_ms(ast_tvnow(), p->polaritydelaytv) > p->polarityonanswerdelay) {
				ast_debug(1, "Polarity Reversal detected and now Hanging up on channel %d\n", p->channel);
				ast_softhangup(p->owner, AST_SOFTHANGUP_EXPLICIT);
				p->polarity = POLARITY_IDLE;
			} else
				ast_debug(1, "Polarity Reversal detected but NOT hanging up (too close to answer event) on channel %d, state %u\n", p->channel, ast_channel_state(ast));

		} else {
			p->polarity = POLARITY_IDLE;
			ast_debug(1, "Ignoring Polarity switch to IDLE on channel %d, state %u\n", p->channel, ast_channel_state(ast));
		}
		/* Added more log_debug information below to provide a better indication of what is going on */
		ast_debug(1, "Polarity Reversal event occurred - DEBUG 2: channel %d, state %u, pol= %d, aonp= %d, honp= %d, pdelay= %d, tv= %" PRIi64 "\n", p->channel, ast_channel_state(ast), p->polarity, p->answeronpolarityswitch, p->hanguponpolarityswitch, p->polarityonanswerdelay, ast_tvdiff_ms(ast_tvnow(), p->polaritydelaytv) );
		break;
	default:
		ast_debug(1, "Dunno what to do with event %d on channel %d\n", res, p->channel);
	}
	return &p->subs[idx].f;
}

static struct ast_frame *__dahdi_exception(struct ast_channel *ast)
{
	int res;
	int idx;
	struct ast_frame *f;
	int usedindex = -1;
	struct dahdi_pvt *p = ast_channel_tech_pvt(ast);

	if ((idx = dahdi_get_index(ast, p, 0)) < 0) {
		idx = SUB_REAL;
	}

	p->subs[idx].f.frametype = AST_FRAME_NULL;
	p->subs[idx].f.datalen = 0;
	p->subs[idx].f.samples = 0;
	p->subs[idx].f.mallocd = 0;
	p->subs[idx].f.offset = 0;
	p->subs[idx].f.subclass.integer = 0;
	p->subs[idx].f.delivery = ast_tv(0,0);
	p->subs[idx].f.src = "dahdi_exception";
	p->subs[idx].f.data.ptr = NULL;


	if ((!p->owner) && (!(p->radio || (p->oprmode < 0)))) {
		/* If nobody owns us, absorb the event appropriately, otherwise
		   we loop indefinitely.  This occurs when, during call waiting, the
		   other end hangs up our channel so that it no longer exists, but we
		   have neither FLASH'd nor ONHOOK'd to signify our desire to
		   change to the other channel. */
		if (p->fake_event) {
			res = p->fake_event;
			p->fake_event = 0;
		} else
			res = dahdi_get_event(p->subs[SUB_REAL].dfd);
		/* Switch to real if there is one and this isn't something really silly... */
		if ((res != DAHDI_EVENT_RINGEROFF) && (res != DAHDI_EVENT_RINGERON) &&
			(res != DAHDI_EVENT_HOOKCOMPLETE)) {
			ast_debug(1, "Restoring owner of channel %d on event %d\n", p->channel, res);
			p->owner = p->subs[SUB_REAL].owner;
			if (p->owner) {
				ast_queue_unhold(p->owner);
			}
			p->subs[SUB_REAL].needunhold = 1;
		}
		switch (res) {
		case DAHDI_EVENT_ONHOOK:
			dahdi_ec_disable(p);
			if (p->owner) {
				ast_verb(3, "Channel %s still has call, ringing phone\n", ast_channel_name(p->owner));
				dahdi_ring_phone(p);
				p->callwaitingrepeat = 0;
				p->cidcwexpire = 0;
				p->cid_suppress_expire = 0;
			} else
				ast_log(LOG_WARNING, "Absorbed on hook, but nobody is left!?!?\n");
			dahdi_conf_update(p);
			break;
		case DAHDI_EVENT_RINGOFFHOOK:
			dahdi_ec_enable(p);
			dahdi_set_hook(p->subs[SUB_REAL].dfd, DAHDI_OFFHOOK);
			if (p->owner && (ast_channel_state(p->owner) == AST_STATE_RINGING)) {
				p->subs[SUB_REAL].needanswer = 1;
				p->dialing = 0;
			}
			break;
		case DAHDI_EVENT_HOOKCOMPLETE:
		case DAHDI_EVENT_RINGERON:
		case DAHDI_EVENT_RINGEROFF:
			/* Do nothing */
			break;
		case DAHDI_EVENT_WINKFLASH:
			p->flashtime = ast_tvnow();
			if (p->owner) {
				ast_verb(3, "Channel %d flashed to other channel %s\n", p->channel, ast_channel_name(p->owner));
				if (ast_channel_state(p->owner) != AST_STATE_UP) {
					/* Answer if necessary */
					usedindex = dahdi_get_index(p->owner, p, 0);
					if (usedindex > -1) {
						p->subs[usedindex].needanswer = 1;
					}
					ast_setstate(p->owner, AST_STATE_UP);
				}
				p->callwaitingrepeat = 0;
				p->cidcwexpire = 0;
				p->cid_suppress_expire = 0;
				ast_queue_unhold(p->owner);
				p->subs[SUB_REAL].needunhold = 1;
			} else
				ast_log(LOG_WARNING, "Absorbed on hook, but nobody is left!?!?\n");
			dahdi_conf_update(p);
			break;
		default:
			ast_log(LOG_WARNING, "Don't know how to absorb event %s\n", event2str(res));
		}
		f = &p->subs[idx].f;
		return f;
	}
	if (!(p->radio || (p->oprmode < 0)))
		ast_debug(1, "Exception on %d, channel %d\n", ast_channel_fd(ast, 0), p->channel);
	/* If it's not us, return NULL immediately */
	if (ast != p->owner) {
		if (p->owner) {
			ast_log(LOG_WARNING, "We're %s, not %s\n", ast_channel_name(ast), ast_channel_name(p->owner));
		}
		f = &p->subs[idx].f;
		return f;
	}

	f = dahdi_handle_event(ast);
	if (!f) {
		const char *name = ast_strdupa(ast_channel_name(ast));

		/* Tell the CDR this DAHDI device hung up */
		ast_mutex_unlock(&p->lock);
		ast_channel_unlock(ast);
		ast_set_hangupsource(ast, name, 0);
		ast_channel_lock(ast);
		ast_mutex_lock(&p->lock);
	}
	return f;
}

static struct ast_frame *dahdi_exception(struct ast_channel *ast)
{
	struct dahdi_pvt *p = ast_channel_tech_pvt(ast);
	struct ast_frame *f;
	ast_mutex_lock(&p->lock);
	if (dahdi_analog_lib_handles(p->sig, p->radio, p->oprmode)) {
		struct analog_pvt *analog_p = p->sig_pvt;
		f = analog_exception(analog_p, ast);
	} else {
		f = __dahdi_exception(ast);
	}
	ast_mutex_unlock(&p->lock);
	return f;
}

static struct ast_frame *dahdi_read(struct ast_channel *ast)
{
	struct dahdi_pvt *p;
	int res;
	int idx;
	void *readbuf;
	struct ast_frame *f;

	/*
	 * For analog channels, we must do deadlock avoidance because
	 * analog ports can have more than one Asterisk channel using
	 * the same private structure.
	 */
	p = ast_channel_tech_pvt(ast);
	while (ast_mutex_trylock(&p->lock)) {
		CHANNEL_DEADLOCK_AVOIDANCE(ast);

		/*
		 * Check to see if the channel is still associated with the same
		 * private structure.  While the Asterisk channel was unlocked
		 * the following events may have occurred:
		 *
		 * 1) A masquerade may have associated the channel with another
		 * technology or private structure.
		 *
		 * 2) For PRI calls, call signaling could change the channel
		 * association to another B channel (private structure).
		 */
		if (ast_channel_tech_pvt(ast) != p) {
			/* The channel is no longer associated.  Quit gracefully. */
			return &ast_null_frame;
		}
	}

	idx = dahdi_get_index(ast, p, 0);

	/* Hang up if we don't really exist */
	if (idx < 0)	{
		ast_log(LOG_WARNING, "We don't exist?\n");
		ast_mutex_unlock(&p->lock);
		return NULL;
	}

	if ((p->radio || (p->oprmode < 0)) && p->inalarm) {
		ast_mutex_unlock(&p->lock);
		return NULL;
	}

	p->subs[idx].f.frametype = AST_FRAME_NULL;
	p->subs[idx].f.datalen = 0;
	p->subs[idx].f.samples = 0;
	p->subs[idx].f.mallocd = 0;
	p->subs[idx].f.offset = 0;
	p->subs[idx].f.subclass.integer = 0;
	p->subs[idx].f.delivery = ast_tv(0,0);
	p->subs[idx].f.src = "dahdi_read";
	p->subs[idx].f.data.ptr = NULL;

	/* make sure it sends initial key state as first frame */
	if ((p->radio || (p->oprmode < 0)) && (!p->firstradio))
	{
		struct dahdi_params ps;

		memset(&ps, 0, sizeof(ps));
		if (ioctl(p->subs[SUB_REAL].dfd, DAHDI_GET_PARAMS, &ps) < 0) {
			ast_mutex_unlock(&p->lock);
			return NULL;
		}
		p->firstradio = 1;
		p->subs[idx].f.frametype = AST_FRAME_CONTROL;
		if (ps.rxisoffhook)
		{
			p->subs[idx].f.subclass.integer = AST_CONTROL_RADIO_KEY;
		}
		else
		{
			p->subs[idx].f.subclass.integer = AST_CONTROL_RADIO_UNKEY;
		}
		ast_mutex_unlock(&p->lock);
		return &p->subs[idx].f;
	}
	if (p->ringt > 0) {
		if (!(--p->ringt)) {
			ast_mutex_unlock(&p->lock);
			return NULL;
		}
	}

#ifdef HAVE_OPENR2
	if (p->mfcr2) {
		openr2_chan_process_event(p->r2chan);
		if (OR2_DIR_FORWARD == openr2_chan_get_direction(p->r2chan)) {
			struct ast_frame fr = { AST_FRAME_CONTROL, { AST_CONTROL_PROGRESS } };
			/* if the call is already accepted and we already delivered AST_CONTROL_RINGING
			 * now enqueue a progress frame to bridge the media up */
			if (p->mfcr2_call_accepted &&
				!p->mfcr2_progress_sent &&
				ast_channel_state(ast) == AST_STATE_RINGING) {
				ast_debug(1, "Enqueuing progress frame after R2 accept in chan %d\n", p->channel);
				ast_queue_frame(p->owner, &fr);
				p->mfcr2_progress_sent = 1;
			}
		}
	}
#endif

	if (p->subs[idx].needringing) {
		/* Send ringing frame if requested */
		p->subs[idx].needringing = 0;
		p->subs[idx].f.frametype = AST_FRAME_CONTROL;
		p->subs[idx].f.subclass.integer = AST_CONTROL_RINGING;
		ast_setstate(ast, AST_STATE_RINGING);
		ast_mutex_unlock(&p->lock);
		return &p->subs[idx].f;
	}

	if (p->subs[idx].needbusy) {
		/* Send busy frame if requested */
		p->subs[idx].needbusy = 0;
		p->subs[idx].f.frametype = AST_FRAME_CONTROL;
		p->subs[idx].f.subclass.integer = AST_CONTROL_BUSY;
		ast_mutex_unlock(&p->lock);
		return &p->subs[idx].f;
	}

	if (p->subs[idx].needcongestion) {
		/* Send congestion frame if requested */
		p->subs[idx].needcongestion = 0;
		p->subs[idx].f.frametype = AST_FRAME_CONTROL;
		p->subs[idx].f.subclass.integer = AST_CONTROL_CONGESTION;
		ast_mutex_unlock(&p->lock);
		return &p->subs[idx].f;
	}

	if (p->subs[idx].needanswer) {
		/* Send answer frame if requested */
		p->subs[idx].needanswer = 0;
		p->subs[idx].f.frametype = AST_FRAME_CONTROL;
		p->subs[idx].f.subclass.integer = AST_CONTROL_ANSWER;
		ast_mutex_unlock(&p->lock);
		return &p->subs[idx].f;
	}
#ifdef HAVE_OPENR2
	if (p->mfcr2 && openr2_chan_get_read_enabled(p->r2chan)) {
		/* openr2 took care of reading and handling any event
		  (needanswer, needbusy etc), if we continue we will read()
		  twice, lets just return a null frame. This should only
		  happen when openr2 is dialing out */
		ast_mutex_unlock(&p->lock);
		return &ast_null_frame;
	}
#endif

	if (p->subs[idx].needflash) {
		/* Send answer frame if requested */
		p->subs[idx].needflash = 0;
		p->subs[idx].f.frametype = AST_FRAME_CONTROL;
		p->subs[idx].f.subclass.integer = AST_CONTROL_FLASH;
		ast_mutex_unlock(&p->lock);
		return &p->subs[idx].f;
	}

	if (p->subs[idx].needhold) {
		/* Send answer frame if requested */
		p->subs[idx].needhold = 0;
		p->subs[idx].f.frametype = AST_FRAME_CONTROL;
		p->subs[idx].f.subclass.integer = AST_CONTROL_HOLD;
		ast_mutex_unlock(&p->lock);
		ast_debug(1, "Sending hold on '%s'\n", ast_channel_name(ast));
		return &p->subs[idx].f;
	}

	if (p->subs[idx].needunhold) {
		/* Send answer frame if requested */
		p->subs[idx].needunhold = 0;
		p->subs[idx].f.frametype = AST_FRAME_CONTROL;
		p->subs[idx].f.subclass.integer = AST_CONTROL_UNHOLD;
		ast_mutex_unlock(&p->lock);
		ast_debug(1, "Sending unhold on '%s'\n", ast_channel_name(ast));
		return &p->subs[idx].f;
	}

	/*
	 * If we have a fake_event, fake an exception to handle it only
	 * if this channel owns the private.
	 */
	if (p->fake_event && p->owner == ast) {
		if (dahdi_analog_lib_handles(p->sig, p->radio, p->oprmode)) {
			struct analog_pvt *analog_p = p->sig_pvt;

			f = analog_exception(analog_p, ast);
		} else {
			f = __dahdi_exception(ast);
		}
		ast_mutex_unlock(&p->lock);
		return f;
	}

	if (ast_format_cmp(ast_channel_rawreadformat(ast), ast_format_slin) == AST_FORMAT_CMP_EQUAL) {
		if (!p->subs[idx].linear) {
			p->subs[idx].linear = 1;
			res = dahdi_setlinear(p->subs[idx].dfd, p->subs[idx].linear);
			if (res)
				ast_log(LOG_WARNING, "Unable to set channel %d (index %d) to linear mode.\n", p->channel, idx);
		}
	} else {
		if (p->subs[idx].linear) {
			p->subs[idx].linear = 0;
			res = dahdi_setlinear(p->subs[idx].dfd, p->subs[idx].linear);
			if (res)
				ast_log(LOG_WARNING, "Unable to set channel %d (index %d) to companded mode.\n", p->channel, idx);
		}
	}
	readbuf = ((unsigned char *)p->subs[idx].buffer) + AST_FRIENDLY_OFFSET;
	CHECK_BLOCKING(ast);
	res = read(p->subs[idx].dfd, readbuf, p->subs[idx].linear ? READ_SIZE * 2 : READ_SIZE);
	ast_clear_flag(ast_channel_flags(ast), AST_FLAG_BLOCKING);
	/* Check for hangup */
	if (res < 0) {
		f = NULL;
		if (res == -1) {
			if (errno == EAGAIN) {
				/* Return "NULL" frame if there is nobody there */
				ast_mutex_unlock(&p->lock);
				return &p->subs[idx].f;
			} else if (errno == ELAST) {
				if (dahdi_analog_lib_handles(p->sig, p->radio, p->oprmode)) {
					struct analog_pvt *analog_p = p->sig_pvt;
					f = analog_exception(analog_p, ast);
				} else {
					f = __dahdi_exception(ast);
				}
			} else
				ast_log(LOG_WARNING, "dahdi_rec: %s\n", strerror(errno));
		}
		ast_mutex_unlock(&p->lock);
		return f;
	}
	if (res != (p->subs[idx].linear ? READ_SIZE * 2 : READ_SIZE)) {
		ast_debug(1, "Short read (%d/%d), must be an event...\n", res, p->subs[idx].linear ? READ_SIZE * 2 : READ_SIZE);
		if (dahdi_analog_lib_handles(p->sig, p->radio, p->oprmode)) {
			struct analog_pvt *analog_p = p->sig_pvt;
			f = analog_exception(analog_p, ast);
		} else {
			f = __dahdi_exception(ast);
		}
		ast_mutex_unlock(&p->lock);
		return f;
	}
	if (p->tdd) { /* if in TDD mode, see if we receive that */
		int c;

		c = tdd_feed(p->tdd,readbuf,READ_SIZE);
		if (c < 0) {
			ast_debug(1,"tdd_feed failed\n");
			ast_mutex_unlock(&p->lock);
			return NULL;
		}
		if (c) { /* if a char to return */
			p->subs[idx].f.subclass.integer = 0;
			p->subs[idx].f.frametype = AST_FRAME_TEXT;
			p->subs[idx].f.mallocd = 0;
			p->subs[idx].f.offset = AST_FRIENDLY_OFFSET;
			p->subs[idx].f.data.ptr = p->subs[idx].buffer + AST_FRIENDLY_OFFSET;
			p->subs[idx].f.datalen = 1;
			*((char *) p->subs[idx].f.data.ptr) = c;
			ast_mutex_unlock(&p->lock);
			return &p->subs[idx].f;
		}
	}
	if (idx == SUB_REAL) {
		/* Ensure the CW timers decrement only on a single subchannel */
		if (p->cidcwexpire) {
			if (!--p->cidcwexpire) {
				/* Expired CID/CW */
				ast_verb(3, "CPE does not support Call Waiting Caller*ID.\n");
				restore_conference(p);
			}
		}
		if (p->cid_suppress_expire) {
			--p->cid_suppress_expire;
		}
		if (p->callwaitingrepeat) {
			if (!--p->callwaitingrepeat) {
				/* Expired, Repeat callwaiting tone */
				++p->callwaitrings;
				dahdi_callwait(ast);
			}
		}
	}
	if (p->subs[idx].linear) {
		p->subs[idx].f.datalen = READ_SIZE * 2;
	} else
		p->subs[idx].f.datalen = READ_SIZE;

	/* Handle CallerID Transmission */
	if ((p->owner == ast) && p->cidspill) {
		send_callerid(p);
	}

	p->subs[idx].f.frametype = AST_FRAME_VOICE;
	p->subs[idx].f.subclass.format = ast_channel_rawreadformat(ast);
	p->subs[idx].f.samples = READ_SIZE;
	p->subs[idx].f.mallocd = 0;
	p->subs[idx].f.offset = AST_FRIENDLY_OFFSET;
	p->subs[idx].f.data.ptr = p->subs[idx].buffer + AST_FRIENDLY_OFFSET / sizeof(p->subs[idx].buffer[0]);
#if 0
	ast_debug(1, "Read %d of voice on %s\n", p->subs[idx].f.datalen, ast->name);
#endif
	if ((p->dialing && !p->waitingfordt.tv_sec) ||  p->radio || /* Transmitting something */
		(idx && (ast_channel_state(ast) != AST_STATE_UP)) || /* Three-way or callwait that isn't up */
		((idx == SUB_CALLWAIT) && !p->subs[SUB_CALLWAIT].inthreeway) /* Inactive and non-confed call-wait */
		) {
		/* Whoops, we're still dialing, or in a state where we shouldn't transmit....
		   don't send anything */
		p->subs[idx].f.frametype = AST_FRAME_NULL;
		p->subs[idx].f.subclass.integer = 0;
		p->subs[idx].f.samples = 0;
		p->subs[idx].f.mallocd = 0;
		p->subs[idx].f.offset = 0;
		p->subs[idx].f.data.ptr = NULL;
		p->subs[idx].f.datalen= 0;
	}
	if (p->dsp && (!p->ignoredtmf || p->callwaitcas || p->busydetect || p->callprogress || p->waitingfordt.tv_sec || p->dialtone_detect) && !idx) {
		/* Perform busy detection etc on the dahdi line */
		int mute;

		if ((p->dsp_features & DSP_FEATURE_FAX_DETECT)
			&& p->faxdetect_timeout
			&& p->faxdetect_timeout <= ast_channel_get_up_time(ast)) {
			p->dsp_features &= ~DSP_FEATURE_FAX_DETECT;
			ast_dsp_set_features(p->dsp, p->dsp_features);
			ast_debug(1, "Channel driver fax CNG detection timeout on %s\n",
				ast_channel_name(ast));
		}

		f = ast_dsp_process(ast, p->dsp, &p->subs[idx].f);

		/* Check if DSP code thinks we should be muting this frame and mute the conference if so */
		mute = ast_dsp_was_muted(p->dsp);
		if (p->muting != mute) {
			p->muting = mute;
			dahdi_confmute(p, mute);
		}

		if (f) {
			if ((p->dsp_features & DSP_FEATURE_WAITDIALTONE) && (p->dialtone_detect > 0)
				&& !p->outgoing && ast_channel_state(ast) == AST_STATE_UP) {
				if (++p->dialtone_scanning_time_elapsed >= p->dialtone_detect) {
					p->dsp_features &= ~DSP_FEATURE_WAITDIALTONE;
					ast_dsp_set_features(p->dsp, p->dsp_features);
				}
			}
			if ((f->frametype == AST_FRAME_CONTROL) && (f->subclass.integer == AST_CONTROL_BUSY)) {
				if ((ast_channel_state(ast) == AST_STATE_UP) && !p->outgoing) {
					/*
					 * Treat this as a "hangup" instead of a "busy" on the
					 * assumption that a busy means the incoming call went away.
					 */
					ast_frfree(f);
					f = NULL;
				}
			} else if (p->dialtone_detect && !p->outgoing && f->frametype == AST_FRAME_VOICE) {
				if ((ast_dsp_get_tstate(p->dsp) == DSP_TONE_STATE_DIALTONE) && (ast_dsp_get_tcount(p->dsp) > 9)) {
					/* Dialtone detected on inbound call; hangup the channel */
					ast_frfree(f);
					f = NULL;
				}
			} else if (f->frametype == AST_FRAME_DTMF_BEGIN
				|| f->frametype == AST_FRAME_DTMF_END) {
#ifdef HAVE_PRI
				if (dahdi_sig_pri_lib_handles(p->sig)
					&& ((struct sig_pri_chan *) p->sig_pvt)->call_level < SIG_PRI_CALL_LEVEL_PROCEEDING
					&& p->pri
					&& ((!p->outgoing && (p->pri->overlapdial & DAHDI_OVERLAPDIAL_INCOMING))
						|| (p->outgoing && (p->pri->overlapdial & DAHDI_OVERLAPDIAL_OUTGOING)))) {
					/* Don't accept in-band DTMF when in overlap dial mode */
					ast_debug(1, "Absorbing inband %s DTMF digit: 0x%02X '%c' on %s\n",
						f->frametype == AST_FRAME_DTMF_BEGIN ? "begin" : "end",
						(unsigned)f->subclass.integer, f->subclass.integer, ast_channel_name(ast));

					f->frametype = AST_FRAME_NULL;
					f->subclass.integer = 0;
				}
#endif
				/* DSP clears us of being pulse */
				p->pulsedial = 0;
			} else if (p->waitingfordt.tv_sec) {
				if (ast_tvdiff_ms(ast_tvnow(), p->waitingfordt) >= p->waitfordialtoneduration) {
					p->waitingfordt.tv_sec = 0;
					ast_log(LOG_NOTICE, "Never saw dialtone on channel %d\n", p->channel);
					ast_frfree(f);
					f = NULL;
				} else if (f->frametype == AST_FRAME_VOICE) {
					f->frametype = AST_FRAME_NULL;
					f->subclass.integer = 0;
					if ((ast_dsp_get_tstate(p->dsp) == DSP_TONE_STATE_DIALTONE || ast_dsp_get_tstate(p->dsp) == DSP_TONE_STATE_RINGING) && ast_dsp_get_tcount(p->dsp) > 9) {
						p->waitingfordt.tv_sec = 0;
						p->dsp_features &= ~DSP_FEATURE_WAITDIALTONE;
						ast_dsp_set_features(p->dsp, p->dsp_features);
						ast_debug(1, "Got 10 samples of dialtone!\n");
						if (!ast_strlen_zero(p->dop.dialstr)) { /* Dial deferred digits */
							res = dahdi_dial_str(p, p->dop.op, p->dop.dialstr);
							if (res) {
								p->dop.dialstr[0] = '\0';
								ast_mutex_unlock(&p->lock);
								ast_frfree(f);
								return NULL;
							} else {
								ast_debug(1, "Sent deferred digit string: %s\n", p->dop.dialstr);
								p->dialing = 1;
								p->dop.dialstr[0] = '\0';
								p->dop.op = DAHDI_DIAL_OP_REPLACE;
								ast_setstate(ast, AST_STATE_DIALING);
							}
						}
					}
				}
			}
		}
	} else
		f = &p->subs[idx].f;

	if (f) {
		struct analog_pvt *analog_p = p->sig_pvt;
		switch (f->frametype) {
		case AST_FRAME_DTMF_BEGIN:
		case AST_FRAME_DTMF_END:
			if (dahdi_analog_lib_handles(p->sig, p->radio, p->oprmode)) {
				analog_handle_dtmf(p->sig_pvt, ast, idx, &f);
			} else {
				dahdi_handle_dtmf(ast, idx, &f);
			}
			if (!(analog_p->dialmode == ANALOG_DIALMODE_BOTH || analog_p->dialmode == ANALOG_DIALMODE_DTMF)) {
				if (f->frametype == AST_FRAME_DTMF_END) { /* only show this message when the key is let go of */
					ast_debug(1, "Dropping DTMF digit '%c' because tone dialing is disabled\n", f->subclass.integer);
				}
				f->frametype = AST_FRAME_NULL;
				f->subclass.integer = 0;
			}
			break;
		case AST_FRAME_VOICE:
			if (p->cidspill || p->cid_suppress_expire) {
				/* We are/were sending a caller id spill.  Suppress any echo. */
				p->subs[idx].f.frametype = AST_FRAME_NULL;
				p->subs[idx].f.subclass.integer = 0;
				p->subs[idx].f.samples = 0;
				p->subs[idx].f.mallocd = 0;
				p->subs[idx].f.offset = 0;
				p->subs[idx].f.data.ptr = NULL;
				p->subs[idx].f.datalen= 0;
			}
			break;
		default:
			break;
		}
	}

	ast_mutex_unlock(&p->lock);
	return f;
}

static int my_dahdi_write(struct dahdi_pvt *p, unsigned char *buf, int len, int idx, int linear)
{
	int sent=0;
	int size;
	int res;
	int fd;
	fd = p->subs[idx].dfd;
	while (len) {
		size = len;
		if (size > (linear ? READ_SIZE * 2 : READ_SIZE))
			size = (linear ? READ_SIZE * 2 : READ_SIZE);
		res = write(fd, buf, size);
		if (res != size) {
			ast_debug(1, "Write returned %d (%s) on channel %d\n", res, strerror(errno), p->channel);
			return sent;
		}
		len -= size;
		buf += size;
	}
	return sent;
}

static int dahdi_write(struct ast_channel *ast, struct ast_frame *frame)
{
	struct dahdi_pvt *p;
	int res;
	int idx;

	/* Write a frame of (presumably voice) data */
	if (frame->frametype != AST_FRAME_VOICE) {
		if (frame->frametype != AST_FRAME_IMAGE) {
			ast_log(LOG_WARNING, "Don't know what to do with frame type '%u'\n",
				frame->frametype);
		}
		return 0;
	}

	/* Return if it's not valid data */
	if (!frame->data.ptr || !frame->datalen) {
		return 0;
	}

	p = ast_channel_tech_pvt(ast);
	ast_mutex_lock(&p->lock);

	idx = dahdi_get_index(ast, p, 0);
	if (idx < 0) {
		ast_mutex_unlock(&p->lock);
		ast_log(LOG_WARNING, "%s doesn't really exist?\n", ast_channel_name(ast));
		return -1;
	}

	if (p->sig == SIG_FXOLS || p->sig == SIG_FXOKS || p->sig == SIG_FXOGS) {
		struct analog_pvt *analog_p = p->sig_pvt;
		if (analog_p->callwaitingdeluxepending) {
			unsigned int mssinceflash = ast_tvdiff_ms(ast_tvnow(), analog_p->flashtime);
			if (mssinceflash >= 1000) {
				/* Timer expired: the user hasn't yet selected an option. Take the default action and get on with it. */
				/* Note: If in the future Advanced Call Waiting Deluxe (*76) is supported, then as part of the
				 * dialing code, we'll need to automatically invoke the preselected behavior about 2-3 seconds after
				 * the call waiting begins (this allows for the SAS, CAS, and CWCID spill to be sent first).
				 */
				analog_p->callwaitingdeluxepending = 0;
				analog_callwaiting_deluxe(analog_p, 0);
			}
			ast_mutex_unlock(&p->lock);
			/* The user shouldn't hear anything after hook flashing, until a decision is made, by the user or when the timer expires. */
			ast_debug(5, "Dropping frame since Call Waiting Deluxe pending on %s\n", ast_channel_name(ast));
			return 0;
		}
	}

	if (p->dialing) {
		ast_mutex_unlock(&p->lock);
		ast_debug(5, "Dropping frame since I'm still dialing on %s...\n",
			ast_channel_name(ast));
		return 0;
	}
	if (!p->owner) {
		ast_mutex_unlock(&p->lock);
		ast_debug(5, "Dropping frame since there is no active owner on %s...\n",
			ast_channel_name(ast));
		return 0;
	}
	if (p->cidspill) {
		ast_mutex_unlock(&p->lock);
		ast_debug(5, "Dropping frame since I've still got a callerid spill on %s...\n",
			ast_channel_name(ast));
		return 0;
	}

	if (ast_format_cmp(frame->subclass.format, ast_format_slin) == AST_FORMAT_CMP_EQUAL) {
		if (!p->subs[idx].linear) {
			p->subs[idx].linear = 1;
			res = dahdi_setlinear(p->subs[idx].dfd, p->subs[idx].linear);
			if (res)
				ast_log(LOG_WARNING, "Unable to set linear mode on channel %d\n", p->channel);
		}
		res = my_dahdi_write(p, (unsigned char *)frame->data.ptr, frame->datalen, idx, 1);
	} else if (ast_format_cmp(frame->subclass.format, ast_format_ulaw) == AST_FORMAT_CMP_EQUAL
		|| ast_format_cmp(frame->subclass.format, ast_format_alaw) == AST_FORMAT_CMP_EQUAL) {
		/* x-law already */
		if (p->subs[idx].linear) {
			p->subs[idx].linear = 0;
			res = dahdi_setlinear(p->subs[idx].dfd, p->subs[idx].linear);
			if (res)
				ast_log(LOG_WARNING, "Unable to set companded mode on channel %d\n", p->channel);
		}
		res = my_dahdi_write(p, (unsigned char *)frame->data.ptr, frame->datalen, idx, 0);
	} else {
		ast_mutex_unlock(&p->lock);
		ast_log(LOG_WARNING, "Cannot handle frames in %s format\n",
			ast_format_get_name(frame->subclass.format));
		return -1;
	}
	ast_mutex_unlock(&p->lock);
	if (res < 0) {
		ast_log(LOG_WARNING, "write failed: %s\n", strerror(errno));
		return -1;
	}
	return 0;
}

static int dahdi_indicate(struct ast_channel *chan, int condition, const void *data, size_t datalen)
{
	struct dahdi_pvt *p = ast_channel_tech_pvt(chan);
	int res=-1;
	int idx;
	int func = DAHDI_FLASH;

	ast_mutex_lock(&p->lock);
	ast_debug(1, "Requested indication %d on channel %s\n", condition, ast_channel_name(chan));
	switch (p->sig) {
#if defined(HAVE_PRI)
	case SIG_PRI_LIB_HANDLE_CASES:
		res = sig_pri_indicate(p->sig_pvt, chan, condition, data, datalen);
		ast_mutex_unlock(&p->lock);
		return res;
#endif	/* defined(HAVE_PRI) */
#if defined(HAVE_SS7)
	case SIG_SS7:
		res = sig_ss7_indicate(p->sig_pvt, chan, condition, data, datalen);
		ast_mutex_unlock(&p->lock);
		return res;
#endif	/* defined(HAVE_SS7) */
	default:
		break;
	}
#ifdef HAVE_OPENR2
	if (p->mfcr2 && !p->mfcr2_call_accepted) {
		ast_mutex_unlock(&p->lock);
		/* if this is an R2 call and the call is not yet accepted, we don't want the
		   tone indications to mess up with the MF tones */
		return 0;
	}
#endif
	idx = dahdi_get_index(chan, p, 0);
	if (idx == SUB_REAL) {
		switch (condition) {
		case AST_CONTROL_BUSY:
			res = tone_zone_play_tone(p->subs[idx].dfd, DAHDI_TONE_BUSY);
			break;
		case AST_CONTROL_RINGING:
			res = tone_zone_play_tone(p->subs[idx].dfd, DAHDI_TONE_RINGTONE);

			if (ast_channel_state(chan) != AST_STATE_UP) {
				if ((ast_channel_state(chan) != AST_STATE_RING) ||
					((p->sig != SIG_FXSKS) &&
				 (p->sig != SIG_FXSLS) &&
				 (p->sig != SIG_FXSGS))) {
					/* We're playing audible ringback tone on the channel,
					 * so set state to AST_STATE_RING, not AST_STATE_RINGING. */
					ast_setstate(chan, AST_STATE_RING);
				}
			}
			break;
		case AST_CONTROL_INCOMPLETE:
			ast_debug(1, "Received AST_CONTROL_INCOMPLETE on %s\n", ast_channel_name(chan));
			/* act as a progress or proceeding, allowing the caller to enter additional numbers */
			res = 0;
			break;
		case AST_CONTROL_PROCEEDING:
			ast_debug(1, "Received AST_CONTROL_PROCEEDING on %s\n", ast_channel_name(chan));
			/* don't continue in ast_indicate */
			res = 0;
			break;
		case AST_CONTROL_PROGRESS:
			ast_debug(1, "Received AST_CONTROL_PROGRESS on %s\n", ast_channel_name(chan));
			/* don't continue in ast_indicate */
			res = 0;
			break;
		case AST_CONTROL_CONGESTION:
			/* There are many cause codes that generate an AST_CONTROL_CONGESTION. */
			switch (ast_channel_hangupcause(chan)) {
			case AST_CAUSE_USER_BUSY:
			case AST_CAUSE_NORMAL_CLEARING:
			case 0:/* Cause has not been set. */
				/* Supply a more appropriate cause. */
				ast_channel_hangupcause_set(chan, AST_CAUSE_CONGESTION);
				break;
			default:
				break;
			}
			break;
		case AST_CONTROL_HOLD:
			ast_moh_start(chan, data, p->mohinterpret);
			break;
		case AST_CONTROL_UNHOLD:
			ast_moh_stop(chan);
			break;
		case AST_CONTROL_RADIO_KEY:
			if (p->radio)
				res = dahdi_set_hook(p->subs[idx].dfd, DAHDI_OFFHOOK);
			res = 0;
			break;
		case AST_CONTROL_RADIO_UNKEY:
			if (p->radio)
				res = dahdi_set_hook(p->subs[idx].dfd, DAHDI_RINGOFF);
			res = 0;
			break;
		case AST_CONTROL_FLASH:
			/* flash hookswitch */
			if (ISTRUNK(p) && (p->sig != SIG_PRI)) {
				/* Clear out the dial buffer */
				p->dop.dialstr[0] = '\0';
				if ((ioctl(p->subs[SUB_REAL].dfd,DAHDI_HOOK,&func) == -1) && (errno != EINPROGRESS)) {
					ast_log(LOG_WARNING, "Unable to flash external trunk on channel %s: %s\n",
						ast_channel_name(chan), strerror(errno));
				} else
					res = 0;
			} else
				res = 0;
			break;
		case AST_CONTROL_SRCUPDATE:
			res = 0;
			break;
		case -1:
			res = tone_zone_play_tone(p->subs[idx].dfd, -1);
			break;
		}
	} else {
		res = 0;
	}
	ast_mutex_unlock(&p->lock);
	return res;
}

#if defined(HAVE_PRI)
static struct ast_str *create_channel_name(struct dahdi_pvt *i, int is_outgoing, char *address)
#else
static struct ast_str *create_channel_name(struct dahdi_pvt *i)
#endif	/* defined(HAVE_PRI) */
{
	struct ast_str *chan_name;
	int x, y;

	/* Create the new channel name tail. */
	if (!(chan_name = ast_str_create(32))) {
		return NULL;
	}
	if (i->channel == CHAN_PSEUDO) {
		ast_str_set(&chan_name, 0, "pseudo-%ld", ast_random());
#if defined(HAVE_PRI)
	} else if (i->pri) {
		ast_mutex_lock(&i->pri->lock);
		y = ++i->pri->new_chan_seq;
		if (is_outgoing) {
			ast_str_set(&chan_name, 0, "i%d/%s-%x", i->pri->span, address, (unsigned)y);
			address[0] = '\0';
		} else if (ast_strlen_zero(i->cid_subaddr)) {
			/* Put in caller-id number only since there is no subaddress. */
			ast_str_set(&chan_name, 0, "i%d/%s-%x", i->pri->span, i->cid_num, (unsigned)y);
		} else {
			/* Put in caller-id number and subaddress. */
			ast_str_set(&chan_name, 0, "i%d/%s:%s-%x", i->pri->span, i->cid_num,
				i->cid_subaddr, (unsigned)y);
		}
		ast_mutex_unlock(&i->pri->lock);
#endif	/* defined(HAVE_PRI) */
	} else {
		y = 1;
		do {
			ast_str_set(&chan_name, 0, "%d-%d", i->channel, y);
			for (x = 0; x < 3; ++x) {
				if (i->subs[x].owner && !strcasecmp(ast_str_buffer(chan_name),
					ast_channel_name(i->subs[x].owner) + 6)) {
					break;
				}
			}
			++y;
		} while (x < 3);
	}
	return chan_name;
}

static struct ast_channel *dahdi_new_callid_clean(struct dahdi_pvt *i, int state, int startpbx, int idx, int law, const struct ast_assigned_ids *assignedids, const struct ast_channel *requestor, ast_callid callid, int callid_created)
{
	struct ast_channel *new_channel = dahdi_new(i, state, startpbx, idx, law, assignedids, requestor, callid);

	ast_callid_threadstorage_auto_clean(callid, callid_created);

	return new_channel;
}

static struct ast_channel *dahdi_new(struct dahdi_pvt *i, int state, int startpbx, int idx, int law, const struct ast_assigned_ids *assignedids, const struct ast_channel *requestor, ast_callid callid)
{
	struct ast_channel *tmp;
	struct ast_format_cap *caps;
	struct ast_format *deflaw;
	int x;
	int features;
	struct ast_str *chan_name;
	struct ast_variable *v;
	char *dashptr;
	char device_name[AST_CHANNEL_NAME];

	if (i->subs[idx].owner) {
		ast_log(LOG_WARNING, "Channel %d already has a %s call\n", i->channel,subnames[idx]);
		return NULL;
	}

#if defined(HAVE_PRI)
	/*
	 * The dnid has been stuffed with the called-number[:subaddress]
	 * by dahdi_request() for outgoing calls.
	 */
	chan_name = create_channel_name(i, i->outgoing, i->dnid);
#else
	chan_name = create_channel_name(i);
#endif	/* defined(HAVE_PRI) */
	if (!chan_name) {
		return NULL;
	}

	caps = ast_format_cap_alloc(AST_FORMAT_CAP_FLAG_DEFAULT);
	if (!caps) {
		ast_free(chan_name);
		return NULL;
	}

	tmp = ast_channel_alloc(0, state, i->cid_num, i->cid_name, i->accountcode, i->exten, i->context, assignedids, requestor, i->amaflags, "DAHDI/%s", ast_str_buffer(chan_name));
	ast_free(chan_name);
	if (!tmp) {
		ao2_ref(caps, -1);
		return NULL;
	}

	ast_channel_stage_snapshot(tmp);

	if (callid) {
		ast_channel_callid_set(tmp, callid);
	}

	ast_channel_tech_set(tmp, &dahdi_tech);
#if defined(HAVE_PRI)
	if (i->pri) {
		ast_cc_copy_config_params(i->cc_params, i->pri->cc_params);
	}
#endif	/* defined(HAVE_PRI) */
	ast_channel_cc_params_init(tmp, i->cc_params);
	if (law) {
		i->law = law;
		if (law == DAHDI_LAW_ALAW) {
			deflaw = ast_format_alaw;
		} else {
			deflaw = ast_format_ulaw;
		}
	} else {
		switch (i->sig) {
		case SIG_PRI_LIB_HANDLE_CASES:
			/* Make sure companding law is known. */
			i->law = (i->law_default == DAHDI_LAW_ALAW)
				? DAHDI_LAW_ALAW : DAHDI_LAW_MULAW;
			break;
		default:
			i->law = i->law_default;
			break;
		}
		if (i->law_default == DAHDI_LAW_ALAW) {
			deflaw = ast_format_alaw;
		} else {
			deflaw = ast_format_ulaw;
		}
	}
	ast_channel_set_fd(tmp, 0, i->subs[idx].dfd);
	ast_format_cap_append(caps, deflaw, 0);
	ast_channel_nativeformats_set(tmp, caps);
	ao2_ref(caps, -1);
	/* Start out assuming ulaw since it's smaller :) */
	ast_channel_set_rawreadformat(tmp, deflaw);
	ast_channel_set_readformat(tmp, deflaw);
	ast_channel_set_rawwriteformat(tmp, deflaw);
	ast_channel_set_writeformat(tmp, deflaw);
	i->subs[idx].linear = 0;
	dahdi_setlinear(i->subs[idx].dfd, i->subs[idx].linear);
	features = 0;
	if (idx == SUB_REAL) {
		if (i->busydetect && CANBUSYDETECT(i))
			features |= DSP_FEATURE_BUSY_DETECT;
		if ((i->callprogress & CALLPROGRESS_PROGRESS) && CANPROGRESSDETECT(i))
			features |= DSP_FEATURE_CALL_PROGRESS;
		if ((i->waitfordialtone || i->dialtone_detect) && CANPROGRESSDETECT(i))
			features |= DSP_FEATURE_WAITDIALTONE;
		if ((!i->outgoing && (i->callprogress & CALLPROGRESS_FAX_INCOMING)) ||
			(i->outgoing && (i->callprogress & CALLPROGRESS_FAX_OUTGOING))) {
			features |= DSP_FEATURE_FAX_DETECT;
		}
		x = DAHDI_TONEDETECT_ON | DAHDI_TONEDETECT_MUTE;
		if (ioctl(i->subs[idx].dfd, DAHDI_TONEDETECT, &x)) {
			i->hardwaredtmf = 0;
			features |= DSP_FEATURE_DIGIT_DETECT;
		} else if (NEED_MFDETECT(i)) {
			i->hardwaredtmf = 1;
			features |= DSP_FEATURE_DIGIT_DETECT;
		}
	}
	if (features) {
		if (i->dsp) {
			ast_debug(1, "Already have a dsp on %s?\n", ast_channel_name(tmp));
		} else {
			if (i->channel != CHAN_PSEUDO)
				i->dsp = ast_dsp_new();
			else
				i->dsp = NULL;
			if (i->dsp) {
				i->dsp_features = features;
#if defined(HAVE_PRI) || defined(HAVE_SS7)
				/* We cannot do progress detection until receive PROGRESS message */
				if (i->outgoing && (dahdi_sig_pri_lib_handles(i->sig) || (i->sig == SIG_SS7))) {
					/* Remember requested DSP features, don't treat
					   talking as ANSWER */
					i->dsp_features = features & ~DSP_PROGRESS_TALK;
					features = 0;
				}
#endif	/* defined(HAVE_PRI) || defined(HAVE_SS7) */
				ast_dsp_set_features(i->dsp, features);
				ast_dsp_set_digitmode(i->dsp, DSP_DIGITMODE_DTMF | i->dtmfrelax);
				if (!ast_strlen_zero(progzone))
					ast_dsp_set_call_progress_zone(i->dsp, progzone);
				if (i->busydetect && CANBUSYDETECT(i)) {
					ast_dsp_set_busy_count(i->dsp, i->busycount);
					ast_dsp_set_busy_pattern(i->dsp, &i->busy_cadence);
				}
			}
		}
	}

	i->dialtone_scanning_time_elapsed = 0;

	if (state == AST_STATE_RING)
		ast_channel_rings_set(tmp, 1);
	ast_channel_tech_pvt_set(tmp, i);
	if ((i->sig == SIG_FXOKS) || (i->sig == SIG_FXOGS) || (i->sig == SIG_FXOLS)) {
		/* Only FXO signalled stuff can be picked up */
		ast_channel_callgroup_set(tmp, i->callgroup);
		ast_channel_pickupgroup_set(tmp, i->pickupgroup);
		ast_channel_named_callgroups_set(tmp, i->named_callgroups);
		ast_channel_named_pickupgroups_set(tmp, i->named_pickupgroups);
	}
	if (!ast_strlen_zero(i->parkinglot))
		ast_channel_parkinglot_set(tmp, i->parkinglot);
	if (!ast_strlen_zero(i->language))
		ast_channel_language_set(tmp, i->language);
	if (!i->owner)
		i->owner = tmp;
	if (!ast_strlen_zero(i->accountcode))
		ast_channel_accountcode_set(tmp, i->accountcode);
	if (i->amaflags)
		ast_channel_amaflags_set(tmp, i->amaflags);
	i->subs[idx].owner = tmp;
	ast_channel_context_set(tmp, i->context);
	if (!dahdi_analog_lib_handles(i->sig, i->radio, i->oprmode)) {
		ast_channel_call_forward_set(tmp, i->call_forward);
	}
	/* If we've been told "no ADSI" then enforce it */
	if (!i->adsi)
		ast_channel_adsicpe_set(tmp, AST_ADSI_UNAVAILABLE);
	if (!ast_strlen_zero(i->exten))
		ast_channel_exten_set(tmp, i->exten);
	if (!ast_strlen_zero(i->rdnis)) {
		ast_channel_redirecting(tmp)->from.number.valid = 1;
		ast_channel_redirecting(tmp)->from.number.str = ast_strdup(i->rdnis);
	}
	if (!ast_strlen_zero(i->dnid)) {
		ast_channel_dialed(tmp)->number.str = ast_strdup(i->dnid);
	}

	/* Don't use ast_set_callerid() here because it will
	 * generate a needless NewCallerID event */
#if defined(HAVE_PRI) || defined(HAVE_SS7)
	if (!ast_strlen_zero(i->cid_ani)) {
		ast_channel_caller(tmp)->ani.number.valid = 1;
		ast_channel_caller(tmp)->ani.number.str = ast_strdup(i->cid_ani);
	} else if (!ast_strlen_zero(i->cid_num)) {
		ast_channel_caller(tmp)->ani.number.valid = 1;
		ast_channel_caller(tmp)->ani.number.str = ast_strdup(i->cid_num);
	}
#else
	if (!ast_strlen_zero(i->cid_num)) {
		ast_channel_caller(tmp)->ani.number.valid = 1;
		ast_channel_caller(tmp)->ani.number.str = ast_strdup(i->cid_num);
	}
#endif	/* defined(HAVE_PRI) || defined(HAVE_SS7) */
	ast_channel_caller(tmp)->id.name.presentation = i->callingpres;
	ast_channel_caller(tmp)->id.number.presentation = i->callingpres;
	ast_channel_caller(tmp)->id.number.plan = i->cid_ton;
	ast_channel_caller(tmp)->ani2 = i->cid_ani2;
	ast_channel_caller(tmp)->id.tag = ast_strdup(i->cid_tag);
	/* clear the fake event in case we posted one before we had ast_channel */
	i->fake_event = 0;
	/* Assure there is no confmute on this channel */
	dahdi_confmute(i, 0);
	i->muting = 0;
	/* Configure the new channel jb */
	ast_jb_configure(tmp, &global_jbconf);

	/* Set initial device state */
	ast_copy_string(device_name, ast_channel_name(tmp), sizeof(device_name));
	dashptr = strrchr(device_name, '-');
	if (dashptr) {
		*dashptr = '\0';
	}
	ast_set_flag(ast_channel_flags(tmp), AST_FLAG_DISABLE_DEVSTATE_CACHE);
	ast_devstate_changed_literal(AST_DEVICE_UNKNOWN, AST_DEVSTATE_NOT_CACHABLE, device_name);

	for (v = i->vars ; v ; v = v->next)
		pbx_builtin_setvar_helper(tmp, v->name, v->value);

	ast_channel_stage_snapshot_done(tmp);

	ast_channel_unlock(tmp);

	ast_module_ref(ast_module_info->self);

	dahdi_ami_channel_event(i, tmp);
	if (startpbx) {
#ifdef HAVE_OPENR2
		if (i->mfcr2call) {
			pbx_builtin_setvar_helper(tmp, "MFCR2_CATEGORY", openr2_proto_get_category_string(i->mfcr2_recvd_category));
		}
#endif
		if (ast_pbx_start(tmp)) {
			ast_log(LOG_WARNING, "Unable to start PBX on %s\n", ast_channel_name(tmp));
			ast_hangup(tmp);
			return NULL;
		}
	}
	return tmp;
}


static int my_getsigstr(struct ast_channel *chan, char *str, const char *term, int ms)
{
	char c;

	*str = 0; /* start with empty output buffer */
	for (;;)
	{
		/* Wait for the first digit (up to specified ms). */
		c = ast_waitfordigit(chan, ms);
		/* if timeout, hangup or error, return as such */
		if (c < 1)
			return c;
		*str++ = c;
		*str = 0;
		if (strchr(term, c))
			return 1;
	}
}

static int dahdi_wink(struct dahdi_pvt *p, int idx)
{
	int j;
	dahdi_set_hook(p->subs[idx].dfd, DAHDI_WINK);
	for (;;)
	{
		/* set bits of interest */
		j = DAHDI_IOMUX_SIGEVENT;
		/* wait for some happening */
		if (ioctl(p->subs[idx].dfd,DAHDI_IOMUX,&j) == -1) return(-1);
		/* exit loop if we have it */
		if (j & DAHDI_IOMUX_SIGEVENT) break;
	}
	/* get the event info */
	if (ioctl(p->subs[idx].dfd,DAHDI_GETEVENT,&j) == -1) return(-1);
	return 0;
}

static void publish_dnd_state(int channel, const char *status)
{
	RAII_VAR(struct ast_json *, body, NULL, ast_json_unref);
	RAII_VAR(struct ast_str *, dahdichan, ast_str_create(32), ast_free);
	if (!dahdichan) {
		return;
	}

	ast_str_set(&dahdichan, 0, "%d", channel);

	body = ast_json_pack("{s: s, s: s}",
		"DAHDIChannel", ast_str_buffer(dahdichan),
		"Status", status);
	if (!body) {
		return;
	}

	ast_manager_publish_event("DNDState", EVENT_FLAG_SYSTEM, body);
}

/*! \brief enable or disable the chan_dahdi Do-Not-Disturb mode for a DAHDI channel
 * \param dahdichan "Physical" DAHDI channel (e.g: DAHDI/5)
 * \param flag on 1 to enable, 0 to disable, -1 return dnd value
 *
 * chan_dahdi has a DND (Do Not Disturb) mode for each dahdichan (physical
 * DAHDI channel). Use this to enable or disable it.
 *
 * \bug the use of the word "channel" for those dahdichans is really confusing.
 */
static int dahdi_dnd(struct dahdi_pvt *dahdichan, int flag)
{
	if (dahdi_analog_lib_handles(dahdichan->sig, dahdichan->radio, dahdichan->oprmode)) {
		return analog_dnd(dahdichan->sig_pvt, flag);
	}

	if (flag == -1) {
		return dahdichan->dnd;
	}

	/* Do not disturb */
	dahdichan->dnd = flag;
	ast_verb(3, "%s DND on channel %d\n",
			flag? "Enabled" : "Disabled",
			dahdichan->channel);
	publish_dnd_state(dahdichan->channel, flag ? "enabled" : "disabled");
	return 0;
}

static int canmatch_featurecode(const char *pickupexten, const char *exten)
{
	int extlen = strlen(exten);

	if (!extlen) {
		return 1;
	}

	if (extlen < strlen(pickupexten) && !strncmp(pickupexten, exten, extlen)) {
		return 1;
	}
	/* hardcoded features are *60, *67, *69, *70, *72, *73, *78, *79, *82, *0 */
	if (exten[0] == '*' && extlen < 3) {
		if (extlen == 1) {
			return 1;
		}
		/* "*0" should be processed before it gets here */
		switch (exten[1]) {
		case '6':
		case '7':
		case '8':
			return 1;
		}
	}
	return 0;
}

static void *analog_ss_thread(void *data)
{
	struct ast_channel *chan = data;
	struct dahdi_pvt *p = ast_channel_tech_pvt(chan);
	char exten[AST_MAX_EXTENSION] = "";
	char exten2[AST_MAX_EXTENSION] = "";
	unsigned char buf[256];
	char dtmfcid[300];
	char dtmfbuf[300];
	struct callerid_state *cs = NULL;
	char *name = NULL, *number = NULL;
	int distMatches;
	int curRingData[3];
	int receivedRingT;
	int counter1;
	int counter;
	int samples = 0;
	struct ast_smdi_md_message *smdi_msg = NULL;
	int flags = 0;
	int i;
	int timeout;
	int getforward = 0;
	char *s1, *s2;
	int len = 0;
	int res;
	int idx;
	RAII_VAR(struct ast_features_pickup_config *, pickup_cfg, NULL, ao2_cleanup);
	const char *pickupexten;

	ast_mutex_lock(&ss_thread_lock);
	ss_thread_count++;
	ast_mutex_unlock(&ss_thread_lock);
	/* in the bizarre case where the channel has become a zombie before we
	   even get started here, abort safely
	*/
	if (!p) {
		ast_log(LOG_WARNING, "Channel became a zombie before simple switch could be started (%s)\n", ast_channel_name(chan));
		ast_hangup(chan);
		goto quit;
	}
	ast_verb(3, "Starting simple switch on '%s'\n", ast_channel_name(chan));
	idx = dahdi_get_index(chan, p, 1);
	if (idx < 0) {
		ast_log(LOG_WARNING, "Huh?\n");
		ast_hangup(chan);
		goto quit;
	}

	ast_channel_lock(chan);
	pickup_cfg = ast_get_chan_features_pickup_config(chan);
	if (!pickup_cfg) {
		ast_log(LOG_ERROR, "Unable to retrieve pickup configuration options. Unable to detect call pickup extension\n");
		pickupexten = "";
	} else {
		pickupexten = ast_strdupa(pickup_cfg->pickupexten);
	}
	ast_channel_unlock(chan);

	if (p->dsp)
		ast_dsp_digitreset(p->dsp);
	switch (p->sig) {
	case SIG_FEATD:
	case SIG_FEATDMF:
	case SIG_FEATDMF_TA:
	case SIG_E911:
	case SIG_FGC_CAMAMF:
	case SIG_FEATB:
	case SIG_EMWINK:
	case SIG_SF_FEATD:
	case SIG_SF_FEATDMF:
	case SIG_SF_FEATB:
	case SIG_SFWINK:
		if (dahdi_wink(p, idx))
			goto quit;
		/* Fall through */
	case SIG_EM:
	case SIG_EM_E1:
	case SIG_SF:
	case SIG_FGC_CAMA:
		res = tone_zone_play_tone(p->subs[idx].dfd, -1);
		if (p->dsp)
			ast_dsp_digitreset(p->dsp);
		/* set digit mode appropriately */
		if (p->dsp) {
			if (NEED_MFDETECT(p))
				ast_dsp_set_digitmode(p->dsp, DSP_DIGITMODE_MF | p->dtmfrelax);
			else
				ast_dsp_set_digitmode(p->dsp, DSP_DIGITMODE_DTMF | p->dtmfrelax);
		}
		memset(dtmfbuf, 0, sizeof(dtmfbuf));
		/* Wait for the first digit only if immediate=no */
		if (!p->immediate)
			/* Wait for the first digit (up to 5 seconds). */
			res = ast_waitfordigit(chan, 5000);
		else
			res = 0;
		if (res > 0) {
			/* save first char */
			dtmfbuf[0] = res;
			switch (p->sig) {
			case SIG_FEATD:
			case SIG_SF_FEATD:
				res = my_getsigstr(chan, dtmfbuf + 1, "*", 3000);
				if (res > 0)
					res = my_getsigstr(chan, dtmfbuf + strlen(dtmfbuf), "*", 3000);
				if ((res < 1) && (p->dsp)) ast_dsp_digitreset(p->dsp);
				break;
			case SIG_FEATDMF_TA:
				res = my_getsigstr(chan, dtmfbuf + 1, "#", 3000);
				if ((res < 1) && (p->dsp)) ast_dsp_digitreset(p->dsp);
				if (dahdi_wink(p, idx)) goto quit;
				dtmfbuf[0] = 0;
				/* Wait for the first digit (up to 5 seconds). */
				res = ast_waitfordigit(chan, 5000);
				if (res <= 0) break;
				dtmfbuf[0] = res;
				/* fall through intentionally */
			case SIG_FEATDMF:
			case SIG_E911:
			case SIG_FGC_CAMAMF:
			case SIG_SF_FEATDMF:
				res = my_getsigstr(chan, dtmfbuf + 1, "#", 3000);
				/* if international caca, do it again to get real ANO */
				if ((p->sig == SIG_FEATDMF) && (dtmfbuf[1] != '0') && (strlen(dtmfbuf) != 14))
				{
					if (dahdi_wink(p, idx)) goto quit;
					dtmfbuf[0] = 0;
					/* Wait for the first digit (up to 5 seconds). */
					res = ast_waitfordigit(chan, 5000);
					if (res <= 0) break;
					dtmfbuf[0] = res;
					res = my_getsigstr(chan, dtmfbuf + 1, "#", 3000);
				}
				if (res > 0) {
					/* if E911, take off hook */
					if (p->sig == SIG_E911)
						dahdi_set_hook(p->subs[SUB_REAL].dfd, DAHDI_OFFHOOK);
					res = my_getsigstr(chan, dtmfbuf + strlen(dtmfbuf), "#", 3000);
				}
				if ((res < 1) && (p->dsp)) ast_dsp_digitreset(p->dsp);
				break;
			case SIG_FEATB:
			case SIG_SF_FEATB:
				res = my_getsigstr(chan, dtmfbuf + 1, "#", 3000);
				if ((res < 1) && (p->dsp)) ast_dsp_digitreset(p->dsp);
				break;
			case SIG_EMWINK:
				/* if we received a '*', we are actually receiving Feature Group D
				   dial syntax, so use that mode; otherwise, fall through to normal
				   mode
				*/
				if (res == '*') {
					res = my_getsigstr(chan, dtmfbuf + 1, "*", 3000);
					if (res > 0)
						res = my_getsigstr(chan, dtmfbuf + strlen(dtmfbuf), "*", 3000);
					if ((res < 1) && (p->dsp)) ast_dsp_digitreset(p->dsp);
					break;
				}
			default:
				/* If we got the first digit, get the rest */
				len = 1;
				dtmfbuf[len] = '\0';
				while ((len < AST_MAX_EXTENSION-1) && ast_matchmore_extension(chan, ast_channel_context(chan), dtmfbuf, 1, p->cid_num)) {
					if (ast_exists_extension(chan, ast_channel_context(chan), dtmfbuf, 1, p->cid_num)) {
						timeout = p->matchdigit_timeout;
					} else {
						timeout = p->interdigit_timeout;
					}
					res = ast_waitfordigit(chan, timeout);
					if (res < 0) {
						ast_debug(1, "waitfordigit returned < 0...\n");
						ast_hangup(chan);
						goto quit;
					} else if (res) {
						dtmfbuf[len++] = res;
						dtmfbuf[len] = '\0';
					} else {
						break;
					}
				}
				break;
			}
		}
		if (res == -1) {
			ast_log(LOG_WARNING, "getdtmf on channel %d: %s\n", p->channel, strerror(errno));
			ast_hangup(chan);
			goto quit;
		} else if (res < 0) {
			ast_debug(1, "Got hung up before digits finished\n");
			ast_hangup(chan);
			goto quit;
		}

		if (p->sig == SIG_FGC_CAMA) {
			char anibuf[100];

			if (ast_safe_sleep(chan,1000) == -1) {
				ast_hangup(chan);
				goto quit;
			}
			dahdi_set_hook(p->subs[SUB_REAL].dfd, DAHDI_OFFHOOK);
			ast_dsp_set_digitmode(p->dsp, DSP_DIGITMODE_MF | p->dtmfrelax);
			res = my_getsigstr(chan, anibuf, "#", 10000);
			if ((res > 0) && (strlen(anibuf) > 2)) {
				if (anibuf[strlen(anibuf) - 1] == '#')
					anibuf[strlen(anibuf) - 1] = 0;
				ast_set_callerid(chan, anibuf + 2, NULL, anibuf + 2);
			}
			ast_dsp_set_digitmode(p->dsp, DSP_DIGITMODE_DTMF | p->dtmfrelax);
		}

		ast_copy_string(exten, dtmfbuf, sizeof(exten));
		if (ast_strlen_zero(exten))
			ast_copy_string(exten, "s", sizeof(exten));
		if (p->sig == SIG_FEATD || p->sig == SIG_EMWINK) {
			/* Look for Feature Group D on all E&M Wink and Feature Group D trunks */
			if (exten[0] == '*') {
				char *stringp=NULL;
				ast_copy_string(exten2, exten, sizeof(exten2));
				/* Parse out extension and callerid */
				stringp=exten2 +1;
				s1 = strsep(&stringp, "*");
				s2 = strsep(&stringp, "*");
				if (s2) {
					if (!ast_strlen_zero(p->cid_num))
						ast_set_callerid(chan, p->cid_num, NULL, p->cid_num);
					else
						ast_set_callerid(chan, s1, NULL, s1);
					ast_copy_string(exten, s2, sizeof(exten));
				} else
					ast_copy_string(exten, s1, sizeof(exten));
			} else if (p->sig == SIG_FEATD)
				ast_log(LOG_WARNING, "Got a non-Feature Group D input on channel %d.  Assuming E&M Wink instead\n", p->channel);
		}
		if ((p->sig == SIG_FEATDMF) || (p->sig == SIG_FEATDMF_TA)) {
			if (exten[0] == '*') {
				char *stringp=NULL;
				ast_copy_string(exten2, exten, sizeof(exten2));
				/* Parse out extension and callerid */
				stringp=exten2 +1;
				s1 = strsep(&stringp, "#");
				s2 = strsep(&stringp, "#");
				if (s2) {
					if (!ast_strlen_zero(p->cid_num))
						ast_set_callerid(chan, p->cid_num, NULL, p->cid_num);
					else
						if (*(s1 + 2))
							ast_set_callerid(chan, s1 + 2, NULL, s1 + 2);
					ast_copy_string(exten, s2 + 1, sizeof(exten));
				} else
					ast_copy_string(exten, s1 + 2, sizeof(exten));
			} else
				ast_log(LOG_WARNING, "Got a non-Feature Group D input on channel %d.  Assuming E&M Wink instead\n", p->channel);
		}
		if ((p->sig == SIG_E911) || (p->sig == SIG_FGC_CAMAMF)) {
			if (exten[0] == '*') {
				char *stringp=NULL;
				ast_copy_string(exten2, exten, sizeof(exten2));
				/* Parse out extension and callerid */
				stringp=exten2 +1;
				s1 = strsep(&stringp, "#");
				s2 = strsep(&stringp, "#");
				if (s2 && (*(s2 + 1) == '0')) {
					if (*(s2 + 2))
						ast_set_callerid(chan, s2 + 2, NULL, s2 + 2);
				}
				if (s1)	ast_copy_string(exten, s1, sizeof(exten));
				else ast_copy_string(exten, "911", sizeof(exten));
			} else
				ast_log(LOG_WARNING, "Got a non-E911/FGC CAMA input on channel %d.  Assuming E&M Wink instead\n", p->channel);
		}
		if (p->sig == SIG_FEATB) {
			if (exten[0] == '*') {
				char *stringp=NULL;
				ast_copy_string(exten2, exten, sizeof(exten2));
				/* Parse out extension and callerid */
				stringp=exten2 +1;
				s1 = strsep(&stringp, "#");
				ast_copy_string(exten, exten2 + 1, sizeof(exten));
			} else
				ast_log(LOG_WARNING, "Got a non-Feature Group B input on channel %d.  Assuming E&M Wink instead\n", p->channel);
		}
		if ((p->sig == SIG_FEATDMF) || (p->sig == SIG_FEATDMF_TA)) {
			dahdi_wink(p, idx);
			/* some switches require a minimum guard time between
			   the last FGD wink and something that answers
			   immediately. This ensures it */
			if (ast_safe_sleep(chan, 100)) {
				ast_hangup(chan);
				goto quit;
			}
		}
		dahdi_ec_enable(p);
		if (NEED_MFDETECT(p)) {
			if (p->dsp) {
				if (!p->hardwaredtmf)
					ast_dsp_set_digitmode(p->dsp, DSP_DIGITMODE_DTMF | p->dtmfrelax);
				else {
					ast_dsp_free(p->dsp);
					p->dsp = NULL;
				}
			}
		}

		if (ast_exists_extension(chan, ast_channel_context(chan), exten, 1,
			S_COR(ast_channel_caller(chan)->id.number.valid, ast_channel_caller(chan)->id.number.str, NULL))) {
			ast_channel_exten_set(chan, exten);
			if (p->dsp) ast_dsp_digitreset(p->dsp);
			res = ast_pbx_run(chan);
			if (res) {
				ast_log(LOG_WARNING, "PBX exited non-zero\n");
				res = tone_zone_play_tone(p->subs[idx].dfd, DAHDI_TONE_CONGESTION);
			}
			goto quit;
		} else {
			ast_verb(2, "Unknown extension '%s' in context '%s' requested\n", exten, ast_channel_context(chan));
			sleep(2);
			res = tone_zone_play_tone(p->subs[idx].dfd, DAHDI_TONE_INFO);
			if (res < 0)
				ast_log(LOG_WARNING, "Unable to start special tone on %d\n", p->channel);
			else
				sleep(1);
			res = ast_streamfile(chan, "ss-noservice", ast_channel_language(chan));
			if (res >= 0)
				ast_waitstream(chan, "");
			res = tone_zone_play_tone(p->subs[idx].dfd, DAHDI_TONE_CONGESTION);
			ast_hangup(chan);
			goto quit;
		}
		break;
	case SIG_FXOLS:
	case SIG_FXOGS:
	case SIG_FXOKS:
		/* Read the first digit */
		timeout = p->firstdigit_timeout;
		/* If starting a threeway call, never timeout on the first digit so someone
		   can use flash-hook as a "hold" feature */
		if (p->subs[SUB_THREEWAY].owner)
			timeout = INT_MAX;
		while (len < AST_MAX_EXTENSION-1) {
			int is_exten_parking = 0;

			/* Read digit unless it's supposed to be immediate, in which case the
			   only answer is 's' */
			if (p->immediate)
				res = 's';
			else
				res = ast_waitfordigit(chan, timeout);
			timeout = 0;
			if (res < 0) {
				ast_debug(1, "waitfordigit returned < 0...\n");
				res = tone_zone_play_tone(p->subs[idx].dfd, -1);
				ast_hangup(chan);
				goto quit;
			} else if (res) {
				ast_debug(1,"waitfordigit returned '%c' (%d), timeout = %d\n", res, res, timeout);
				exten[len++]=res;
				exten[len] = '\0';
			}
			if (!ast_ignore_pattern(ast_channel_context(chan), exten)) {
				tone_zone_play_tone(p->subs[idx].dfd, -1);
			} else {
				tone_zone_play_tone(p->subs[idx].dfd, DAHDI_TONE_DIALTONE);
			}
			if (ast_parking_provider_registered()) {
				is_exten_parking = ast_parking_is_exten_park(ast_channel_context(chan), exten);
			}
			if (ast_exists_extension(chan, ast_channel_context(chan), exten, 1, p->cid_num) && !is_exten_parking) {
				if (!res || !ast_matchmore_extension(chan, ast_channel_context(chan), exten, 1, p->cid_num)) {
					if (getforward) {
						/* Record this as the forwarding extension */
						ast_copy_string(p->call_forward, exten, sizeof(p->call_forward));
						ast_verb(3, "Setting call forward to '%s' on channel %d\n", p->call_forward, p->channel);
						res = tone_zone_play_tone(p->subs[idx].dfd, DAHDI_TONE_DIALRECALL);
						if (res)
							break;
						usleep(500000);
						res = tone_zone_play_tone(p->subs[idx].dfd, -1);
						sleep(1);
						memset(exten, 0, sizeof(exten));
						res = tone_zone_play_tone(p->subs[idx].dfd, DAHDI_TONE_DIALTONE);
						len = 0;
						getforward = 0;
					} else {
						res = tone_zone_play_tone(p->subs[idx].dfd, -1);
						ast_channel_lock(chan);
						ast_channel_exten_set(chan, exten);
						if (!ast_strlen_zero(p->cid_num)) {
							if (!p->hidecallerid)
								ast_set_callerid(chan, p->cid_num, NULL, p->cid_num);
							else
								ast_set_callerid(chan, NULL, NULL, p->cid_num);
						}
						if (!ast_strlen_zero(p->cid_name)) {
							if (!p->hidecallerid)
								ast_set_callerid(chan, NULL, p->cid_name, NULL);
						}
						ast_setstate(chan, AST_STATE_RING);
						ast_channel_unlock(chan);
						dahdi_ec_enable(p);
						res = ast_pbx_run(chan);
						if (res) {
							ast_log(LOG_WARNING, "PBX exited non-zero\n");
							res = tone_zone_play_tone(p->subs[idx].dfd, DAHDI_TONE_CONGESTION);
						}
						goto quit;
					}
				} else {
					/* It's a match, but they just typed a digit, and there is an ambiguous match,
					   so just set the timeout to matchdigit_timeout and wait some more */
					timeout = p->matchdigit_timeout;
				}
			} else if (res == 0) {
				ast_debug(1, "not enough digits (and no ambiguous match)...\n");
				res = tone_zone_play_tone(p->subs[idx].dfd, DAHDI_TONE_CONGESTION);
				dahdi_wait_event(p->subs[idx].dfd);
				ast_hangup(chan);
				goto quit;
			} else if (p->callwaiting && !strcmp(exten, "*70")) {
				ast_verb(3, "Disabling call waiting on %s\n", ast_channel_name(chan));
				/* Disable call waiting if enabled */
				p->callwaiting = 0;
				res = tone_zone_play_tone(p->subs[idx].dfd, DAHDI_TONE_DIALRECALL);
				if (res) {
					ast_log(LOG_WARNING, "Unable to do dial recall on channel %s: %s\n",
						ast_channel_name(chan), strerror(errno));
				}
				len = 0;
				ioctl(p->subs[idx].dfd,DAHDI_CONFDIAG,&len);
				memset(exten, 0, sizeof(exten));
				timeout = p->firstdigit_timeout;

			} else if (!strcmp(exten, pickupexten)) {
				/* Scan all channels and see if there are any
				 * ringing channels that have call groups
				 * that equal this channels pickup group
				 */
				if (idx == SUB_REAL) {
					/* Switch us from Third call to Call Wait */
					if (p->subs[SUB_THREEWAY].owner) {
						/* If you make a threeway call and the *8# a call, it should actually
						   look like a callwait */
						alloc_sub(p, SUB_CALLWAIT);
						swap_subs(p, SUB_CALLWAIT, SUB_THREEWAY);
						unalloc_sub(p, SUB_THREEWAY);
					}
					dahdi_ec_enable(p);
					if (ast_pickup_call(chan)) {
						ast_debug(1, "No call pickup possible...\n");
						res = tone_zone_play_tone(p->subs[idx].dfd, DAHDI_TONE_CONGESTION);
						dahdi_wait_event(p->subs[idx].dfd);
					}
					ast_hangup(chan);
					goto quit;
				} else {
					ast_log(LOG_WARNING, "Huh?  Got *8# on call not on real\n");
					ast_hangup(chan);
					goto quit;
				}

			} else if (!p->hidecallerid && !strcmp(exten, "*67")) {
				ast_verb(3, "Disabling Caller*ID on %s\n", ast_channel_name(chan));
				/* Disable Caller*ID if enabled */
				p->hidecallerid = 1;
				ast_party_number_free(&ast_channel_caller(chan)->id.number);
				ast_party_number_init(&ast_channel_caller(chan)->id.number);
				ast_party_name_free(&ast_channel_caller(chan)->id.name);
				ast_party_name_init(&ast_channel_caller(chan)->id.name);
				res = tone_zone_play_tone(p->subs[idx].dfd, DAHDI_TONE_DIALRECALL);
				if (res) {
					ast_log(LOG_WARNING, "Unable to do dial recall on channel %s: %s\n",
						ast_channel_name(chan), strerror(errno));
				}
				len = 0;
				memset(exten, 0, sizeof(exten));
				timeout = p->firstdigit_timeout;
			} else if (p->callreturn && !strcmp(exten, "*69")) {
				res = tone_zone_play_tone(p->subs[idx].dfd, DAHDI_TONE_DIALRECALL);
				break;
			} else if (!strcmp(exten, "*78")) {
				dahdi_dnd(p, 1);
				/* Do not disturb */
				res = tone_zone_play_tone(p->subs[idx].dfd, DAHDI_TONE_DIALRECALL);
				getforward = 0;
				memset(exten, 0, sizeof(exten));
				len = 0;
			} else if (!strcmp(exten, "*79")) {
				dahdi_dnd(p, 0);
				/* Do not disturb */
				res = tone_zone_play_tone(p->subs[idx].dfd, DAHDI_TONE_DIALRECALL);
				getforward = 0;
				memset(exten, 0, sizeof(exten));
				len = 0;
			} else if (p->cancallforward && !strcmp(exten, "*72")) {
				res = tone_zone_play_tone(p->subs[idx].dfd, DAHDI_TONE_DIALRECALL);
				getforward = 1;
				memset(exten, 0, sizeof(exten));
				len = 0;
			} else if (p->cancallforward && !strcmp(exten, "*73")) {
				ast_verb(3, "Cancelling call forwarding on channel %d\n", p->channel);
				res = tone_zone_play_tone(p->subs[idx].dfd, DAHDI_TONE_DIALRECALL);
				memset(p->call_forward, 0, sizeof(p->call_forward));
				getforward = 0;
				memset(exten, 0, sizeof(exten));
				len = 0;
			} else if ((p->transfer || p->canpark) && is_exten_parking
				&& p->subs[SUB_THREEWAY].owner) {
				struct ast_bridge_channel *bridge_channel;

				/*
				 * This is a three way call, the main call being a real channel,
				 * and we're parking the first call.
				 */
				ast_channel_lock(p->subs[SUB_THREEWAY].owner);
				bridge_channel = ast_channel_get_bridge_channel(p->subs[SUB_THREEWAY].owner);
				ast_channel_unlock(p->subs[SUB_THREEWAY].owner);
				if (bridge_channel) {
					if (!ast_parking_blind_transfer_park(bridge_channel, ast_channel_context(chan), exten, NULL, NULL)) {
						/*
						 * Swap things around between the three-way and real call so we
						 * can hear where the channel got parked.
						 */
						ast_mutex_lock(&p->lock);
						p->owner = p->subs[SUB_THREEWAY].owner;
						swap_subs(p, SUB_THREEWAY, SUB_REAL);
						ast_mutex_unlock(&p->lock);

						ast_verb(3, "%s: Parked call\n", ast_channel_name(chan));
						ast_hangup(chan);
						ao2_ref(bridge_channel, -1);
						goto quit;
					}
					ao2_ref(bridge_channel, -1);
				}
				break;
			} else if (p->hidecallerid && !strcmp(exten, "*82")) {
				ast_verb(3, "Enabling Caller*ID on %s\n", ast_channel_name(chan));
				/* Enable Caller*ID if enabled */
				p->hidecallerid = 0;
				ast_set_callerid(chan, p->cid_num, p->cid_name, NULL);
				res = tone_zone_play_tone(p->subs[idx].dfd, DAHDI_TONE_DIALRECALL);
				if (res) {
					ast_log(LOG_WARNING, "Unable to do dial recall on channel %s: %s\n",
						ast_channel_name(chan), strerror(errno));
				}
				len = 0;
				memset(exten, 0, sizeof(exten));
				timeout = p->firstdigit_timeout;
			} else if (!strcmp(exten, "*0")) {
				struct ast_channel *nbridge =
					p->subs[SUB_THREEWAY].owner;
				struct dahdi_pvt *pbridge = NULL;
				RAII_VAR(struct ast_channel *, bridged, nbridge ? ast_channel_bridge_peer(nbridge) : NULL, ast_channel_cleanup);

				/* set up the private struct of the bridged one, if any */
				if (nbridge && bridged) {
					pbridge = ast_channel_tech_pvt(bridged);
				}
				if (nbridge && pbridge &&
					(ast_channel_tech(nbridge) == &dahdi_tech) &&
					(ast_channel_tech(bridged) == &dahdi_tech) &&
					ISTRUNK(pbridge)) {
					int func = DAHDI_FLASH;
					/* Clear out the dial buffer */
					p->dop.dialstr[0] = '\0';
					/* flash hookswitch */
					if ((ioctl(pbridge->subs[SUB_REAL].dfd,DAHDI_HOOK,&func) == -1) && (errno != EINPROGRESS)) {
						ast_log(LOG_WARNING, "Unable to flash external trunk on channel %s: %s\n",
							ast_channel_name(nbridge), strerror(errno));
					}
					swap_subs(p, SUB_REAL, SUB_THREEWAY);
					unalloc_sub(p, SUB_THREEWAY);
					p->owner = p->subs[SUB_REAL].owner;
					ast_queue_unhold(p->subs[SUB_REAL].owner);
					ast_hangup(chan);
					goto quit;
				} else {
					tone_zone_play_tone(p->subs[idx].dfd, DAHDI_TONE_CONGESTION);
					dahdi_wait_event(p->subs[idx].dfd);
					tone_zone_play_tone(p->subs[idx].dfd, -1);
					swap_subs(p, SUB_REAL, SUB_THREEWAY);
					unalloc_sub(p, SUB_THREEWAY);
					p->owner = p->subs[SUB_REAL].owner;
					ast_hangup(chan);
					goto quit;
				}
			} else if (!ast_canmatch_extension(chan, ast_channel_context(chan), exten, 1,
				S_COR(ast_channel_caller(chan)->id.number.valid, ast_channel_caller(chan)->id.number.str, NULL))
				&& !canmatch_featurecode(pickupexten, exten)) {
				ast_debug(1, "Can't match %s from '%s' in context %s\n", exten,
					S_COR(ast_channel_caller(chan)->id.number.valid, ast_channel_caller(chan)->id.number.str, "<Unknown Caller>"),
					ast_channel_context(chan));
				break;
			}
			if (!timeout)
				timeout = p->interdigit_timeout;
			if (len && !ast_ignore_pattern(ast_channel_context(chan), exten))
				tone_zone_play_tone(p->subs[idx].dfd, -1);
		}
		break;
	case SIG_FXSLS:
	case SIG_FXSGS:
	case SIG_FXSKS:
		/* check for SMDI messages */
		if (p->use_smdi && p->smdi_iface) {
			smdi_msg = ast_smdi_md_message_wait(p->smdi_iface, SMDI_MD_WAIT_TIMEOUT);

			if (smdi_msg != NULL) {
				ast_channel_exten_set(chan, smdi_msg->fwd_st);

				if (smdi_msg->type == 'B')
					pbx_builtin_setvar_helper(chan, "_SMDI_VM_TYPE", "b");
				else if (smdi_msg->type == 'N')
					pbx_builtin_setvar_helper(chan, "_SMDI_VM_TYPE", "u");

				ast_debug(1, "Received SMDI message on %s\n", ast_channel_name(chan));
			} else {
				ast_log(LOG_WARNING, "SMDI enabled but no SMDI message present\n");
			}
		}

		if (p->use_callerid && (p->cid_signalling == CID_SIG_SMDI && smdi_msg)) {
			number = smdi_msg->calling_st;

		/* If we want caller id, we're in a prering state due to a polarity reversal
		 * and we're set to use a polarity reversal to trigger the start of caller id,
		 * grab the caller id and wait for ringing to start... */
		} else if (p->use_callerid && (ast_channel_state(chan) == AST_STATE_PRERING &&
						 (p->cid_start == CID_START_POLARITY || p->cid_start == CID_START_POLARITY_IN || p->cid_start == CID_START_DTMF_NOALERT))) {
			/* If set to use DTMF CID signalling, listen for DTMF */
			if (p->cid_signalling == CID_SIG_DTMF) {
				int k = 0;
				int off_ms;
				struct timeval start = ast_tvnow();
				int ms;
				cs = NULL;
				ast_debug(1, "Receiving DTMF cid on channel %s\n", ast_channel_name(chan));
				dahdi_setlinear(p->subs[idx].dfd, 0);
				/*
				 * We are the only party interested in the Rx stream since
				 * we have not answered yet.  We don't need or even want DTMF
				 * emulation.  The DTMF digits can come so fast that emulation
				 * can drop some of them.
				 */
				ast_channel_lock(chan);
				ast_set_flag(ast_channel_flags(chan), AST_FLAG_END_DTMF_ONLY);
				ast_channel_unlock(chan);
				off_ms = 4000;/* This is a typical OFF time between rings. */
				for (;;) {
					struct ast_frame *f;

					ms = ast_remaining_ms(start, off_ms);
					res = ast_waitfor(chan, ms);
					if (res <= 0) {
						/*
						 * We do not need to restore the dahdi_setlinear()
						 * or AST_FLAG_END_DTMF_ONLY flag settings since we
						 * are hanging up the channel.
						 */
						ast_log(LOG_WARNING, "DTMFCID timed out waiting for ring. "
							"Exiting simple switch\n");
						ast_hangup(chan);
						goto quit;
					}
					f = ast_read(chan);
					if (!f)
						break;
					if (f->frametype == AST_FRAME_DTMF) {
						if (k < ARRAY_LEN(dtmfbuf) - 1) {
							dtmfbuf[k++] = f->subclass.integer;
						}
						ast_debug(1, "CID got digit '%c'\n", f->subclass.integer);
						start = ast_tvnow();
					}
					ast_frfree(f);
					if (ast_channel_state(chan) == AST_STATE_RING ||
						ast_channel_state(chan) == AST_STATE_RINGING)
						break; /* Got ring */
				}
				ast_channel_lock(chan);
				ast_clear_flag(ast_channel_flags(chan), AST_FLAG_END_DTMF_ONLY);
				ast_channel_unlock(chan);
				dtmfbuf[k] = '\0';
				dahdi_setlinear(p->subs[idx].dfd, p->subs[idx].linear);
				/* Got cid and ring. */
				ast_debug(1, "CID got string '%s'\n", dtmfbuf);
				callerid_get_dtmf(dtmfbuf, dtmfcid, &flags);
				ast_debug(1, "CID is '%s', flags %d\n", dtmfcid, flags);
				/* If first byte is NULL, we have no cid */
				if (!ast_strlen_zero(dtmfcid))
					number = dtmfcid;
				else
					number = NULL;
			/* If set to use V23 Signalling, launch our FSK gubbins and listen for it */
			} else if ((p->cid_signalling == CID_SIG_V23) || (p->cid_signalling == CID_SIG_V23_JP)) {
				cs = callerid_new(p->cid_signalling);
				if (cs) {
					int off_ms;
					struct timeval start;
					int ms;
					samples = 0;
#if 1
					bump_gains(p);
#endif
					/* Take out of linear mode for Caller*ID processing */
					dahdi_setlinear(p->subs[idx].dfd, 0);

					/* First we wait and listen for the Caller*ID */
					for (;;) {
						i = DAHDI_IOMUX_READ | DAHDI_IOMUX_SIGEVENT;
						if ((res = ioctl(p->subs[idx].dfd, DAHDI_IOMUX, &i))) {
							ast_log(LOG_WARNING, "I/O MUX failed: %s\n", strerror(errno));
							callerid_free(cs);
							ast_hangup(chan);
							goto quit;
						}
						if (i & DAHDI_IOMUX_SIGEVENT) {
							res = dahdi_get_event(p->subs[idx].dfd);
							ast_log(LOG_NOTICE, "Got event %d (%s)...\n", res, event2str(res));
							if (res == DAHDI_EVENT_NOALARM) {
								p->inalarm = 0;
							}

							if (p->cid_signalling == CID_SIG_V23_JP) {
								if (res == DAHDI_EVENT_RINGBEGIN) {
									res = dahdi_set_hook(p->subs[SUB_REAL].dfd, DAHDI_OFFHOOK);
									usleep(1);
								}
							} else {
								res = 0;
								break;
							}
						} else if (i & DAHDI_IOMUX_READ) {
							res = read(p->subs[idx].dfd, buf, sizeof(buf));
							if (res < 0) {
								if (errno != ELAST) {
									ast_log(LOG_WARNING, "read returned error: %s\n", strerror(errno));
									callerid_free(cs);
									ast_hangup(chan);
									goto quit;
								}
								break;
							}
							samples += res;

							if (p->cid_signalling == CID_SIG_V23_JP) {
								res = callerid_feed_jp(cs, buf, res, AST_LAW(p));
							} else {
								res = callerid_feed(cs, buf, res, AST_LAW(p));
							}
							if (res < 0) {
								/*
								 * The previous diagnostic message output likely
								 * explains why it failed.
								 */
								ast_log(LOG_WARNING,
									"Failed to decode CallerID on channel '%s'\n",
									ast_channel_name(chan));
								break;
							} else if (res)
								break;
							else if (samples > (8000 * 10))
								break;
						}
					}
					if (res == 1) {
						callerid_get(cs, &name, &number, &flags);
						ast_log(LOG_NOTICE, "CallerID number: %s, name: %s, flags=%d\n", number, name, flags);
					}

					if (p->cid_signalling == CID_SIG_V23_JP) {
						res = dahdi_set_hook(p->subs[SUB_REAL].dfd, DAHDI_ONHOOK);
						usleep(1);
					}

					/* Finished with Caller*ID, now wait for a ring to make sure there really is a call coming */
					start = ast_tvnow();
					off_ms = 4000;/* This is a typical OFF time between rings. */
					for (;;) {
						struct ast_frame *f;

						ms = ast_remaining_ms(start, off_ms);
						res = ast_waitfor(chan, ms);
						if (res <= 0) {
							ast_log(LOG_WARNING, "CID timed out waiting for ring. "
								"Exiting simple switch\n");
							ast_hangup(chan);
							goto quit;
						}
						if (!(f = ast_read(chan))) {
							ast_log(LOG_WARNING, "Hangup received waiting for ring. Exiting simple switch\n");
							ast_hangup(chan);
							goto quit;
						}
						ast_frfree(f);
						if (ast_channel_state(chan) == AST_STATE_RING ||
							ast_channel_state(chan) == AST_STATE_RINGING)
							break; /* Got ring */
					}

					/* We must have a ring by now, so, if configured, lets try to listen for
					 * distinctive ringing */
					if (p->usedistinctiveringdetection) {
						len = 0;
						distMatches = 0;
						/* Clear the current ring data array so we don't have old data in it. */
						for (receivedRingT = 0; receivedRingT < ARRAY_LEN(curRingData); receivedRingT++)
							curRingData[receivedRingT] = 0;
						receivedRingT = 0;
						counter = 0;
						counter1 = 0;
						/* Check to see if context is what it should be, if not set to be. */
						if (strcmp(p->context,p->defcontext) != 0) {
							ast_copy_string(p->context, p->defcontext, sizeof(p->context));
							ast_channel_context_set(chan, p->defcontext);
						}

						for (;;) {
							i = DAHDI_IOMUX_READ | DAHDI_IOMUX_SIGEVENT;
							if ((res = ioctl(p->subs[idx].dfd, DAHDI_IOMUX, &i))) {
								ast_log(LOG_WARNING, "I/O MUX failed: %s\n", strerror(errno));
								callerid_free(cs);
								ast_hangup(chan);
								goto quit;
							}
							if (i & DAHDI_IOMUX_SIGEVENT) {
								res = dahdi_get_event(p->subs[idx].dfd);
								ast_log(LOG_NOTICE, "Got event %d (%s)...\n", res, event2str(res));
								if (res == DAHDI_EVENT_NOALARM) {
									p->inalarm = 0;
								}
								res = 0;
								/* Let us detect distinctive ring */

								curRingData[receivedRingT] = p->ringt;

								if (p->ringt < p->ringt_base/2)
									break;
								/* Increment the ringT counter so we can match it against
								   values in chan_dahdi.conf for distinctive ring */
								if (++receivedRingT == ARRAY_LEN(curRingData))
									break;
							} else if (i & DAHDI_IOMUX_READ) {
								res = read(p->subs[idx].dfd, buf, sizeof(buf));
								if (res < 0) {
									if (errno != ELAST) {
										ast_log(LOG_WARNING, "read returned error: %s\n", strerror(errno));
										callerid_free(cs);
										ast_hangup(chan);
										goto quit;
									}
									break;
								}
								if (p->ringt > 0) {
									if (!(--p->ringt)) {
										res = -1;
										break;
									}
								}
							}
						}
							/* this only shows up if you have n of the dring patterns filled in */
						ast_verb(3, "Detected ring pattern: %d,%d,%d\n",curRingData[0],curRingData[1],curRingData[2]);
						for (counter = 0; counter < 3; counter++) {
							/* Check to see if the rings we received match any of the ones in chan_dahdi.conf for this
							channel */
							distMatches = 0;
							for (counter1 = 0; counter1 < 3; counter1++) {
								ast_verb(3, "Ring pattern check range: %d\n", p->drings.ringnum[counter].range);
								if (p->drings.ringnum[counter].ring[counter1] == -1) {
									ast_verb(3, "Pattern ignore (-1) detected, so matching pattern %d regardless.\n",
									curRingData[counter1]);
									distMatches++;
								} else if (curRingData[counter1] <= (p->drings.ringnum[counter].ring[counter1] + p->drings.ringnum[counter].range) &&
										curRingData[counter1] >= (p->drings.ringnum[counter].ring[counter1] - p->drings.ringnum[counter].range)) {
									ast_verb(3, "Ring pattern matched in range: %d to %d\n",
									(p->drings.ringnum[counter].ring[counter1] - p->drings.ringnum[counter].range),
									(p->drings.ringnum[counter].ring[counter1] + p->drings.ringnum[counter].range));
									distMatches++;
								}
							}

							if (distMatches == 3) {
								/* The ring matches, set the context to whatever is for distinctive ring.. */
								ast_copy_string(p->context, S_OR(p->drings.ringContext[counter].contextData, p->defcontext), sizeof(p->context));
								ast_channel_context_set(chan, S_OR(p->drings.ringContext[counter].contextData, p->defcontext));
								ast_verb(3, "Distinctive Ring matched context %s\n",p->context);
								break;
							}
						}
					}
					/* Restore linear mode (if appropriate) for Caller*ID processing */
					dahdi_setlinear(p->subs[idx].dfd, p->subs[idx].linear);
#if 1
					restore_gains(p);
#endif
				} else
					ast_log(LOG_WARNING, "Unable to get caller ID space\n");
			} else {
				ast_log(LOG_WARNING, "Channel %s in prering "
					"state, but I have nothing to do. "
					"Terminating simple switch, should be "
					"restarted by the actual ring.\n",
					ast_channel_name(chan));
				ast_hangup(chan);
				goto quit;
			}
		} else if (p->use_callerid && p->cid_start == CID_START_RING) {
			if (p->cid_signalling == CID_SIG_DTMF) {
				int k = 0;
				int off_ms;
				struct timeval start;
				int ms;
				cs = NULL;
				dahdi_setlinear(p->subs[idx].dfd, 0);
				off_ms = 2000;
				start = ast_tvnow();
				for (;;) {
					struct ast_frame *f;

					ms = ast_remaining_ms(start, off_ms);
					res = ast_waitfor(chan, ms);
					if (res <= 0) {
						ast_log(LOG_WARNING, "DTMFCID timed out waiting for ring. "
							"Exiting simple switch\n");
						ast_hangup(chan);
						goto quit;
					}
					f = ast_read(chan);
					if (!f) {
						/* Hangup received waiting for DTMFCID. Exiting simple switch. */
						ast_hangup(chan);
						goto quit;
					}
					if (f->frametype == AST_FRAME_DTMF) {
						dtmfbuf[k++] = f->subclass.integer;
						ast_debug(1, "CID got digit '%c'\n", f->subclass.integer);
						start = ast_tvnow();
					}
					ast_frfree(f);

					if (p->ringt_base == p->ringt)
						break;
				}
				dtmfbuf[k] = '\0';
				dahdi_setlinear(p->subs[idx].dfd, p->subs[idx].linear);
				/* Got cid and ring. */
				callerid_get_dtmf(dtmfbuf, dtmfcid, &flags);
				ast_debug(1, "CID is '%s', flags %d\n",
					dtmfcid, flags);
				/* If first byte is NULL, we have no cid */
				if (!ast_strlen_zero(dtmfcid))
					number = dtmfcid;
				else
					number = NULL;
				/* If set to use V23 Signalling, launch our FSK gubbins and listen for it */
			} else {
				/* FSK Bell202 callerID */
				cs = callerid_new(p->cid_signalling);
				if (cs) {
#if 1
					bump_gains(p);
#endif
					samples = 0;
					len = 0;
					distMatches = 0;
					/* Clear the current ring data array so we don't have old data in it. */
					for (receivedRingT = 0; receivedRingT < ARRAY_LEN(curRingData); receivedRingT++)
						curRingData[receivedRingT] = 0;
					receivedRingT = 0;
					counter = 0;
					counter1 = 0;
					/* Check to see if context is what it should be, if not set to be. */
					if (strcmp(p->context,p->defcontext) != 0) {
						ast_copy_string(p->context, p->defcontext, sizeof(p->context));
						ast_channel_context_set(chan, p->defcontext);
					}

					/* Take out of linear mode for Caller*ID processing */
					dahdi_setlinear(p->subs[idx].dfd, 0);
					for (;;) {
						i = DAHDI_IOMUX_READ | DAHDI_IOMUX_SIGEVENT;
						if ((res = ioctl(p->subs[idx].dfd, DAHDI_IOMUX, &i))) {
							ast_log(LOG_WARNING, "I/O MUX failed: %s\n", strerror(errno));
							callerid_free(cs);
							ast_hangup(chan);
							goto quit;
						}
						if (i & DAHDI_IOMUX_SIGEVENT) {
							res = dahdi_get_event(p->subs[idx].dfd);
							ast_log(LOG_NOTICE, "Got event %d (%s)...\n", res, event2str(res));
							if (res == DAHDI_EVENT_NOALARM) {
								p->inalarm = 0;
							}
							/* If we get a PR event, they hung up while processing calerid */
							if ( res == DAHDI_EVENT_POLARITY && p->hanguponpolarityswitch && p->polarity == POLARITY_REV) {
								ast_debug(1, "Hanging up due to polarity reversal on channel %d while detecting callerid\n", p->channel);
								p->polarity = POLARITY_IDLE;
								callerid_free(cs);
								ast_hangup(chan);
								goto quit;
							}
							res = 0;
							/* Let us detect callerid when the telco uses distinctive ring */

							curRingData[receivedRingT] = p->ringt;

							if (p->ringt < p->ringt_base/2)
								break;
							/* Increment the ringT counter so we can match it against
							   values in chan_dahdi.conf for distinctive ring */
							if (++receivedRingT == ARRAY_LEN(curRingData))
								break;
						} else if (i & DAHDI_IOMUX_READ) {
							res = read(p->subs[idx].dfd, buf, sizeof(buf));
							if (res < 0) {
								if (errno != ELAST) {
									ast_log(LOG_WARNING, "read returned error: %s\n", strerror(errno));
									callerid_free(cs);
									ast_hangup(chan);
									goto quit;
								}
								break;
							}
							if (p->ringt > 0) {
								if (!(--p->ringt)) {
									res = -1;
									break;
								}
							}
							samples += res;
							res = callerid_feed(cs, buf, res, AST_LAW(p));
							if (res < 0) {
								/*
								 * The previous diagnostic message output likely
								 * explains why it failed.
								 */
								ast_log(LOG_WARNING,
									"Failed to decode CallerID on channel '%s'\n",
									ast_channel_name(chan));
								break;
							} else if (res)
								break;
							else if (samples > (8000 * 10))
								break;
						}
					}
					if (res == 1) {
						callerid_get(cs, &name, &number, &flags);
						ast_debug(1, "CallerID number: %s, name: %s, flags=%d\n", number, name, flags);
					}
					if (distinctiveringaftercid == 1) {
						/* Clear the current ring data array so we don't have old data in it. */
						for (receivedRingT = 0; receivedRingT < 3; receivedRingT++) {
							curRingData[receivedRingT] = 0;
						}
						receivedRingT = 0;
						ast_verb(3, "Detecting post-CID distinctive ring\n");
						for (;;) {
							i = DAHDI_IOMUX_READ | DAHDI_IOMUX_SIGEVENT;
							if ((res = ioctl(p->subs[idx].dfd, DAHDI_IOMUX, &i))) {
								ast_log(LOG_WARNING, "I/O MUX failed: %s\n", strerror(errno));
								callerid_free(cs);
								ast_hangup(chan);
								goto quit;
							}
							if (i & DAHDI_IOMUX_SIGEVENT) {
								res = dahdi_get_event(p->subs[idx].dfd);
								ast_log(LOG_NOTICE, "Got event %d (%s)...\n", res, event2str(res));
								if (res == DAHDI_EVENT_NOALARM) {
									p->inalarm = 0;
								}
								res = 0;
								/* Let us detect callerid when the telco uses distinctive ring */

								curRingData[receivedRingT] = p->ringt;

								if (p->ringt < p->ringt_base/2)
									break;
								/* Increment the ringT counter so we can match it against
								   values in chan_dahdi.conf for distinctive ring */
								if (++receivedRingT == ARRAY_LEN(curRingData))
									break;
							} else if (i & DAHDI_IOMUX_READ) {
								res = read(p->subs[idx].dfd, buf, sizeof(buf));
								if (res < 0) {
									if (errno != ELAST) {
										ast_log(LOG_WARNING, "read returned error: %s\n", strerror(errno));
										callerid_free(cs);
										ast_hangup(chan);
										goto quit;
									}
									break;
								}
								if (p->ringt > 0) {
									if (!(--p->ringt)) {
										res = -1;
										break;
									}
								}
							}
						}
					}
					if (p->usedistinctiveringdetection) {
						/* this only shows up if you have n of the dring patterns filled in */
						ast_verb(3, "Detected ring pattern: %d,%d,%d\n",curRingData[0],curRingData[1],curRingData[2]);

						for (counter = 0; counter < 3; counter++) {
							/* Check to see if the rings we received match any of the ones in chan_dahdi.conf for this
							channel */
							/* this only shows up if you have n of the dring patterns filled in */
							ast_verb(3, "Checking %d,%d,%d\n",
									p->drings.ringnum[counter].ring[0],
									p->drings.ringnum[counter].ring[1],
									p->drings.ringnum[counter].ring[2]);
							distMatches = 0;
							for (counter1 = 0; counter1 < 3; counter1++) {
								ast_verb(3, "Ring pattern check range: %d\n", p->drings.ringnum[counter].range);
								if (p->drings.ringnum[counter].ring[counter1] == -1) {
									ast_verb(3, "Pattern ignore (-1) detected, so matching pattern %d regardless.\n",
									curRingData[counter1]);
									distMatches++;
								}
								else if (curRingData[counter1] <= (p->drings.ringnum[counter].ring[counter1] + p->drings.ringnum[counter].range) &&
									curRingData[counter1] >= (p->drings.ringnum[counter].ring[counter1] - p->drings.ringnum[counter].range)) {
									ast_verb(3, "Ring pattern matched in range: %d to %d\n",
									(p->drings.ringnum[counter].ring[counter1] - p->drings.ringnum[counter].range),
									(p->drings.ringnum[counter].ring[counter1] + p->drings.ringnum[counter].range));
									distMatches++;
								}
							}
							if (distMatches == 3) {
								/* The ring matches, set the context to whatever is for distinctive ring.. */
								ast_copy_string(p->context, S_OR(p->drings.ringContext[counter].contextData, p->defcontext), sizeof(p->context));
								ast_channel_context_set(chan, S_OR(p->drings.ringContext[counter].contextData, p->defcontext));
								ast_verb(3, "Distinctive Ring matched context %s\n",p->context);
								break;
							}
						}
					}
					/* Restore linear mode (if appropriate) for Caller*ID processing */
					dahdi_setlinear(p->subs[idx].dfd, p->subs[idx].linear);
#if 1
					restore_gains(p);
#endif
					if (res < 0) {
						ast_log(LOG_WARNING, "CallerID returned with error on channel '%s'\n", ast_channel_name(chan));
					}
				} else
					ast_log(LOG_WARNING, "Unable to get caller ID space\n");
			}
		} else
			cs = NULL;

		if (number)
			ast_shrink_phone_number(number);
		ast_set_callerid(chan, number, name, number);

		ao2_cleanup(smdi_msg);

		if (cs)
			callerid_free(cs);

		my_handle_notify_message(chan, p, flags, -1);

		ast_channel_lock(chan);
		ast_setstate(chan, AST_STATE_RING);
		ast_channel_rings_set(chan, 1);
		ast_channel_unlock(chan);
		p->ringt = p->ringt_base;
		res = ast_pbx_run(chan);
		if (res) {
			ast_hangup(chan);
			ast_log(LOG_WARNING, "PBX exited non-zero\n");
		}
		goto quit;
	default:
		ast_log(LOG_WARNING, "Don't know how to handle simple switch with signalling %s on channel %d\n", sig2str(p->sig), p->channel);
		res = tone_zone_play_tone(p->subs[idx].dfd, DAHDI_TONE_CONGESTION);
		if (res < 0)
				ast_log(LOG_WARNING, "Unable to play congestion tone on channel %d\n", p->channel);
	}
	res = tone_zone_play_tone(p->subs[idx].dfd, DAHDI_TONE_CONGESTION);
	if (res < 0)
			ast_log(LOG_WARNING, "Unable to play congestion tone on channel %d\n", p->channel);
	ast_hangup(chan);
quit:
	ast_mutex_lock(&ss_thread_lock);
	ss_thread_count--;
	ast_cond_signal(&ss_thread_complete);
	ast_mutex_unlock(&ss_thread_lock);
	return NULL;
}

struct mwi_thread_data {
	struct dahdi_pvt *pvt;
	unsigned char buf[READ_SIZE];
	size_t len;
};

static int calc_energy(const unsigned char *buf, int len, struct ast_format *law)
{
	int x;
	int sum = 0;

	if (!len)
		return 0;

	for (x = 0; x < len; x++)
		sum += abs(law == ast_format_ulaw ? AST_MULAW(buf[x]) : AST_ALAW(buf[x]));

	return sum / len;
}

static void *mwi_thread(void *data)
{
	struct mwi_thread_data *mtd = data;
	struct callerid_state *cs;
	pthread_t threadid;
	int samples = 0;
	char *name, *number;
	int flags;
	int i, res;
	unsigned int spill_done = 0;
	int spill_result = -1;

	if (!(cs = callerid_new(mtd->pvt->cid_signalling))) {
		goto quit_no_clean;
	}

	callerid_feed(cs, mtd->buf, mtd->len, AST_LAW(mtd->pvt));

	bump_gains(mtd->pvt);

	for (;;) {
		i = DAHDI_IOMUX_READ | DAHDI_IOMUX_SIGEVENT;
		if ((res = ioctl(mtd->pvt->subs[SUB_REAL].dfd, DAHDI_IOMUX, &i))) {
			ast_log(LOG_WARNING, "I/O MUX failed: %s\n", strerror(errno));
			goto quit;
		}

		if (i & DAHDI_IOMUX_SIGEVENT) {
			struct ast_channel *chan;
			ast_callid callid = 0;
			int callid_created;

			/* If we get an event, screen out events that we do not act on.
			 * Otherwise, cancel and go to the simple switch to let it deal with it.
			 */
			res = dahdi_get_event(mtd->pvt->subs[SUB_REAL].dfd);

			switch (res) {
			case DAHDI_EVENT_NEONMWI_ACTIVE:
			case DAHDI_EVENT_NEONMWI_INACTIVE:
			case DAHDI_EVENT_NONE:
			case DAHDI_EVENT_BITSCHANGED:
				break;
			case DAHDI_EVENT_NOALARM:
				if (dahdi_analog_lib_handles(mtd->pvt->sig, mtd->pvt->radio, mtd->pvt->oprmode)) {
					struct analog_pvt *analog_p = mtd->pvt->sig_pvt;

					analog_p->inalarm = 0;
				}
				mtd->pvt->inalarm = 0;
				handle_clear_alarms(mtd->pvt);
				break;
			case DAHDI_EVENT_ALARM:
				if (dahdi_analog_lib_handles(mtd->pvt->sig, mtd->pvt->radio, mtd->pvt->oprmode)) {
					struct analog_pvt *analog_p = mtd->pvt->sig_pvt;

					analog_p->inalarm = 1;
				}
				mtd->pvt->inalarm = 1;
				res = get_alarms(mtd->pvt);
				handle_alarms(mtd->pvt, res);
				break; /* What to do on channel alarm ???? -- fall thru intentionally?? */
			default:
				callid_created = ast_callid_threadstorage_auto(&callid);
				ast_log(LOG_NOTICE, "Got event %d (%s)...  Passing along to analog_ss_thread\n", res, event2str(res));
				callerid_free(cs);

				restore_gains(mtd->pvt);
				mtd->pvt->ringt = mtd->pvt->ringt_base;

				if ((chan = dahdi_new(mtd->pvt, AST_STATE_RING, 0, SUB_REAL, 0, NULL, NULL, callid))) {
					int result;

					if (dahdi_analog_lib_handles(mtd->pvt->sig, mtd->pvt->radio, mtd->pvt->oprmode)) {
						result = analog_ss_thread_start(mtd->pvt->sig_pvt, chan);
					} else {
						result = ast_pthread_create_detached(&threadid, NULL, analog_ss_thread, chan);
					}
					if (result) {
						ast_log(LOG_WARNING, "Unable to start simple switch thread on channel %d\n", mtd->pvt->channel);
						res = tone_zone_play_tone(mtd->pvt->subs[SUB_REAL].dfd, DAHDI_TONE_CONGESTION);
						if (res < 0)
							ast_log(LOG_WARNING, "Unable to play congestion tone on channel %d\n", mtd->pvt->channel);
						ast_hangup(chan);
					}
				} else {
					ast_log(LOG_WARNING, "Could not create channel to handle call\n");
				}

				ast_callid_threadstorage_auto_clean(callid, callid_created);
				goto quit_no_clean;
			}
		} else if (i & DAHDI_IOMUX_READ) {
			if ((res = read(mtd->pvt->subs[SUB_REAL].dfd, mtd->buf, sizeof(mtd->buf))) < 0) {
				if (errno != ELAST) {
					ast_log(LOG_WARNING, "read returned error: %s\n", strerror(errno));
					goto quit;
				}
				break;
			}
			samples += res;
			if (!spill_done) {
				if ((spill_result = callerid_feed(cs, mtd->buf, res, AST_LAW(mtd->pvt))) < 0) {
					/*
					 * The previous diagnostic message output likely
					 * explains why it failed.
					 */
					ast_log(LOG_WARNING, "Failed to decode CallerID\n");
					break;
				} else if (spill_result) {
					spill_done = 1;
				}
			} else {
				/* keep reading data until the energy level drops below the threshold
				   so we don't get another 'trigger' on the remaining carrier signal
				*/
				if (calc_energy(mtd->buf, res, AST_LAW(mtd->pvt)) <= mwilevel)
					break;
			}
			if (samples > (8000 * 4)) /*Termination case - time to give up*/
				break;
		}
	}

	if (spill_result == 1) {
		callerid_get(cs, &name, &number, &flags);
		if (flags & CID_MSGWAITING) {
			ast_log(LOG_NOTICE, "mwi: Have Messages on channel %d\n", mtd->pvt->channel);
			notify_message(mtd->pvt->mailbox, 1);
		} else if (flags & CID_NOMSGWAITING) {
			ast_log(LOG_NOTICE, "mwi: No Messages on channel %d\n", mtd->pvt->channel);
			notify_message(mtd->pvt->mailbox, 0);
		} else {
			ast_log(LOG_NOTICE, "mwi: Status unknown on channel %d\n", mtd->pvt->channel);
		}
	}


quit:
	callerid_free(cs);

	restore_gains(mtd->pvt);

quit_no_clean:
	mtd->pvt->mwimonitoractive = 0;
	ast_free(mtd);

	return NULL;
}

/*
* The following three functions (mwi_send_init, mwi_send_process_buffer,
* mwi_send_process_event) work with the do_monitor thread to generate mwi spills
* that are sent out via FXS port on voicemail state change.  The execution of
* the mwi send is state driven and can either generate a ring pulse prior to
* sending the fsk spill or simply send an fsk spill.
*/
static int mwi_send_init(struct dahdi_pvt * pvt)
{
	int x;

#ifdef HAVE_DAHDI_LINEREVERSE_VMWI
	/* Determine how this spill is to be sent */
	if (pvt->mwisend_rpas) {
		pvt->mwisend_data.mwisend_current = MWI_SEND_SA;
		pvt->mwisendactive = 1;
	} else if (pvt->mwisend_fsk) {
		pvt->mwisend_data.mwisend_current = MWI_SEND_SPILL;
		pvt->mwisendactive = 1;
	} else {
		pvt->mwisendactive = 0;
		return 0;
	}
#else
	if (mwisend_rpas) {
		pvt->mwisend_data.mwisend_current = MWI_SEND_SA;
	} else {
		pvt->mwisend_data.mwisend_current = MWI_SEND_SPILL;
	}
	pvt->mwisendactive = 1;
#endif

	if (pvt->cidspill) {
		ast_log(LOG_WARNING, "cidspill already exists when trying to send FSK MWI\n");
		ast_free(pvt->cidspill);
		pvt->cidspill = NULL;
		pvt->cidpos = 0;
		pvt->cidlen = 0;
	}
	pvt->cidspill = ast_calloc(1, MAX_CALLERID_SIZE);
	if (!pvt->cidspill) {
		pvt->mwisendactive = 0;
		return -1;
	}
	x = DAHDI_FLUSH_BOTH;
	ioctl(pvt->subs[SUB_REAL].dfd, DAHDI_FLUSH, &x);
	x = 3000;
	ioctl(pvt->subs[SUB_REAL].dfd, DAHDI_ONHOOKTRANSFER, &x);
#ifdef HAVE_DAHDI_LINEREVERSE_VMWI
	if (pvt->mwisend_fsk) {
#endif
		pvt->cidlen = ast_callerid_vmwi_generate(pvt->cidspill, has_voicemail(pvt),
			CID_MWI_TYPE_MDMF_FULL, AST_LAW(pvt), pvt->cid_name, pvt->cid_num, 0);
		pvt->cidpos = 0;
#ifdef HAVE_DAHDI_LINEREVERSE_VMWI
	}
#endif
	return 0;
}

static int mwi_send_process_buffer(struct dahdi_pvt * pvt, int num_read)
{
	struct timeval 	now;
	int 			res;

	/* sanity check to catch if this had been interrupted previously
	*	i.e. state says there is more to do but there is no spill allocated
	*/
	if (MWI_SEND_DONE != pvt->mwisend_data.mwisend_current && !pvt->cidspill) {
		pvt->mwisend_data.mwisend_current = MWI_SEND_DONE;
	} else if (MWI_SEND_DONE != pvt->mwisend_data.mwisend_current) {
		/* Normal processing -- Perform mwi send action */
		switch ( pvt->mwisend_data.mwisend_current) {
		case MWI_SEND_SA:
			/* Send the Ring Pulse Signal Alert */
			res = ioctl(pvt->subs[SUB_REAL].dfd, DAHDI_SETCADENCE, &AS_RP_cadence);
			if (res) {
				ast_log(LOG_WARNING, "Unable to set RP-AS ring cadence: %s\n", strerror(errno));
				goto quit;
			}
			res = dahdi_set_hook(pvt->subs[SUB_REAL].dfd, DAHDI_RING);
			pvt->mwisend_data.mwisend_current = MWI_SEND_SA_WAIT;
			break;
		case MWI_SEND_SA_WAIT:  /* do nothing until I get RINGEROFF event */
			break;
		case MWI_SEND_PAUSE:  /* Wait between alert and spill - min of 500 mS*/
#ifdef HAVE_DAHDI_LINEREVERSE_VMWI
			if (pvt->mwisend_fsk) {
#endif
				gettimeofday(&now, NULL);
				if ((int)(now.tv_sec - pvt->mwisend_data.pause.tv_sec) * 1000000 + (int)now.tv_usec - (int)pvt->mwisend_data.pause.tv_usec > 500000) {
					pvt->mwisend_data.mwisend_current = MWI_SEND_SPILL;
				}
#ifdef HAVE_DAHDI_LINEREVERSE_VMWI
			} else { /* support for mwisendtype=nofsk */
				pvt->mwisend_data.mwisend_current = MWI_SEND_CLEANUP;
			}
#endif
			break;
		case MWI_SEND_SPILL:
			/* We read some number of bytes.  Write an equal amount of data */
			if (0 < num_read) {
				if (num_read > pvt->cidlen - pvt->cidpos) {
					num_read = pvt->cidlen - pvt->cidpos;
				}
				res = write(pvt->subs[SUB_REAL].dfd, pvt->cidspill + pvt->cidpos, num_read);
				if (res > 0) {
					pvt->cidpos += res;
					if (pvt->cidpos >= pvt->cidlen) {
						pvt->mwisend_data.mwisend_current = MWI_SEND_CLEANUP;
					}
				} else {
					ast_log(LOG_WARNING, "MWI FSK Send Write failed: %s\n", strerror(errno));
					goto quit;
				}
			}
			break;
		case MWI_SEND_CLEANUP:
			/* For now, do nothing */
			pvt->mwisend_data.mwisend_current = MWI_SEND_DONE;
			break;
		default:
			/* Should not get here, punt*/
			goto quit;
		}
	}

	if (MWI_SEND_DONE == pvt->mwisend_data.mwisend_current) {
		if (pvt->cidspill) {
			ast_free(pvt->cidspill);
			pvt->cidspill = NULL;
			pvt->cidpos = 0;
			pvt->cidlen = 0;
		}
		pvt->mwisendactive = 0;
	}
	return 0;
quit:
	if (pvt->cidspill) {
		ast_free(pvt->cidspill);
		pvt->cidspill = NULL;
		pvt->cidpos = 0;
		pvt->cidlen = 0;
	}
	pvt->mwisendactive = 0;
	return -1;
}

static int mwi_send_process_event(struct dahdi_pvt * pvt, int event)
{
	int handled = 0;

	if (MWI_SEND_DONE != pvt->mwisend_data.mwisend_current) {
		switch (event) {
		case DAHDI_EVENT_RINGEROFF:
			if (pvt->mwisend_data.mwisend_current == MWI_SEND_SA_WAIT) {
				handled = 1;

				if (dahdi_set_hook(pvt->subs[SUB_REAL].dfd, DAHDI_RINGOFF) ) {
					ast_log(LOG_WARNING, "Unable to finish RP-AS: %s mwi send aborted\n", strerror(errno));
					ast_free(pvt->cidspill);
					pvt->cidspill = NULL;
					pvt->mwisend_data.mwisend_current = MWI_SEND_DONE;
					pvt->mwisendactive = 0;
				} else {
					pvt->mwisend_data.mwisend_current = MWI_SEND_PAUSE;
					gettimeofday(&pvt->mwisend_data.pause, NULL);
				}
			}
			break;
		/* Going off hook, I need to punt this spill */
		case DAHDI_EVENT_RINGOFFHOOK:
			if (pvt->cidspill) {
				ast_free(pvt->cidspill);
				pvt->cidspill = NULL;
				pvt->cidpos = 0;
				pvt->cidlen = 0;
			}
			pvt->mwisend_data.mwisend_current = MWI_SEND_DONE;
			pvt->mwisendactive = 0;
			break;
		case DAHDI_EVENT_RINGERON:
		case DAHDI_EVENT_HOOKCOMPLETE:
			break;
		default:
			break;
		}
	}
	return handled;
}

/* destroy a range DAHDI channels, identified by their number */
static void dahdi_destroy_channel_range(int start, int end)
{
	struct dahdi_pvt *cur;
	struct dahdi_pvt *next;
	int destroyed_first = 0;
	int destroyed_last = 0;

	ast_mutex_lock(&iflock);
	ast_debug(1, "range: %d-%d\n", start, end);
	for (cur = iflist; cur; cur = next) {
		next = cur->next;
		if (cur->channel >= start && cur->channel <= end) {
			int x = DAHDI_FLASH;

			if (cur->channel > destroyed_last) {
				destroyed_last = cur->channel;
			}
			if (destroyed_first < 1 || cur->channel < destroyed_first) {
				destroyed_first = cur->channel;
			}
			ast_debug(3, "Destroying %d\n", cur->channel);
			/* important to create an event for dahdi_wait_event to register so that all analog_ss_threads terminate */
			ioctl(cur->subs[SUB_REAL].dfd, DAHDI_HOOK, &x);

			destroy_channel(cur, 1);
			ast_module_unref(ast_module_info->self);
		}
	}
	ast_mutex_unlock(&iflock);
	if (destroyed_first > start || destroyed_last < end) {
		ast_debug(1, "Asked to destroy %d-%d, destroyed %d-%d,\n",
			start, end, destroyed_first, destroyed_last);
	}
}

#ifdef HAVE_OPENR2
static void dahdi_r2_destroy_nodev(void)
{
	struct r2link_entry *cur;
	AST_LIST_LOCK(&nodev_r2links);
	AST_LIST_TRAVERSE_SAFE_BEGIN(&nodev_r2links, cur, list) {
		int i;
		struct dahdi_mfcr2 *r2 = &cur->mfcr2;
		ast_debug(3, "About to destroy %d DAHDI channels of MFC/R2 link.\n", r2->numchans);
		for (i = 0; i < r2->numchans; i++) {
			int channel;
			struct dahdi_pvt *pvt = r2->pvts[i];
			if (!pvt) {
				continue;
			}
			channel = pvt->channel;
			ast_debug(3, "About to destroy B-channel %d.\n", channel);
			dahdi_destroy_channel_range(channel, channel);
		}
		ast_debug(3, "Destroying R2 link\n");
		AST_LIST_REMOVE(&nodev_r2links, cur, list);
		if (r2->r2master != AST_PTHREADT_NULL) {
			pthread_cancel(r2->r2master);
			pthread_join(r2->r2master, NULL);
			r2->r2master = AST_PTHREADT_NULL;
			openr2_context_delete(r2->protocol_context);
		}
		ast_free(cur);
	}
	AST_LIST_TRAVERSE_SAFE_END;
	AST_LIST_UNLOCK(&nodev_r2links);
}
#endif

static int setup_dahdi(int reload);
static int setup_dahdi_int(int reload, struct dahdi_chan_conf *default_conf, struct dahdi_chan_conf *base_conf, struct dahdi_chan_conf *conf);

/*!
 * \internal
 * \brief create a range of new DAHDI channels
 *
 * \param start first channel in the range
 * \param end last channel in the range
 *
 * \retval RESULT_SUCCESS on success.
 * \retval RESULT_FAILURE on error.
 */
static int dahdi_create_channel_range(int start, int end)
{
	struct dahdi_pvt *cur;
	struct dahdi_chan_conf default_conf = dahdi_chan_conf_default();
	struct dahdi_chan_conf base_conf = dahdi_chan_conf_default();
	struct dahdi_chan_conf conf = dahdi_chan_conf_default();
	int ret = RESULT_FAILURE; /* be pessimistic */

	ast_debug(1, "channel range caps: %d - %d\n", start, end);
	ast_mutex_lock(&iflock);
	for (cur = iflist; cur; cur = cur->next) {
		if (cur->channel >= start && cur->channel <= end) {
			ast_log(LOG_ERROR,
				"channel range %d-%d is occupied\n",
				start, end);
			goto out;
		}
	}
#ifdef HAVE_PRI
	{
		int i, x;
		for (x = 0; x < NUM_SPANS; x++) {
			struct dahdi_pri *pri = pris + x;

			if (!pris[x].pri.pvts[0]) {
				break;
			}
			for (i = 0; i < SIG_PRI_NUM_DCHANS; i++) {
				int channo = pri->dchannels[i];

				if (!channo) {
					break;
				}
				if (!pri->pri.fds[i]) {
					break;
				}
				if (channo >= start && channo <= end) {
					ast_log(LOG_ERROR,
							"channel range %d-%d is occupied by span %d\n",
							start, end, x + 1);
					goto out;
				}
			}
		}
	}
#endif
	if (!default_conf.chan.cc_params || !base_conf.chan.cc_params ||
		!conf.chan.cc_params) {
		goto out;
	}
	default_conf.wanted_channels_start = start;
	base_conf.wanted_channels_start = start;
	conf.wanted_channels_start = start;
	default_conf.wanted_channels_end = end;
	base_conf.wanted_channels_end = end;
	conf.wanted_channels_end = end;
	if (setup_dahdi_int(0, &default_conf, &base_conf, &conf) == 0) {
		ret = RESULT_SUCCESS;
	}
out:
	ast_cc_config_params_destroy(default_conf.chan.cc_params);
	ast_cc_config_params_destroy(base_conf.chan.cc_params);
	ast_cc_config_params_destroy(conf.chan.cc_params);
	ast_mutex_unlock(&iflock);
	return ret;
}


static struct dahdi_pvt *handle_init_event(struct dahdi_pvt *i, int event)
{
	int res;
	pthread_t threadid;
	struct ast_channel *chan;
	ast_callid callid = 0;
	int callid_created;

	/* Handle an event on a given channel for the monitor thread. */

	switch (event) {
	case DAHDI_EVENT_NONE:
	case DAHDI_EVENT_BITSCHANGED:
		break;
	case DAHDI_EVENT_WINKFLASH:
	case DAHDI_EVENT_RINGOFFHOOK:
		if (i->inalarm) break;
		if (i->radio) break;
		/* Got a ring/answer.  What kind of channel are we? */
		switch (i->sig) {
		case SIG_FXOLS:
		case SIG_FXOGS:
		case SIG_FXOKS:
			res = dahdi_set_hook(i->subs[SUB_REAL].dfd, DAHDI_OFFHOOK);
			if (res && (errno == EBUSY)) {
				break;
			}

			callid_created = ast_callid_threadstorage_auto(&callid);

			/* Cancel VMWI spill */
			ast_free(i->cidspill);
			i->cidspill = NULL;
			restore_conference(i);

			if (i->immediate) {
				dahdi_ec_enable(i);
				/* The channel is immediately up.  Start right away */
				res = tone_zone_play_tone(i->subs[SUB_REAL].dfd, DAHDI_TONE_RINGTONE);
				chan = dahdi_new(i, AST_STATE_RING, 1, SUB_REAL, 0, NULL, NULL, callid);
				if (!chan) {
					ast_log(LOG_WARNING, "Unable to start PBX on channel %d\n", i->channel);
					res = tone_zone_play_tone(i->subs[SUB_REAL].dfd, DAHDI_TONE_CONGESTION);
					if (res < 0)
						ast_log(LOG_WARNING, "Unable to play congestion tone on channel %d\n", i->channel);
				}
			} else {
				/* Check for callerid, digits, etc */
				chan = dahdi_new(i, AST_STATE_RESERVED, 0, SUB_REAL, 0, NULL, NULL, callid);
				if (chan) {
					if (has_voicemail(i))
						res = tone_zone_play_tone(i->subs[SUB_REAL].dfd, DAHDI_TONE_STUTTER);
					else
						res = tone_zone_play_tone(i->subs[SUB_REAL].dfd, DAHDI_TONE_DIALTONE);
					if (res < 0)
						ast_log(LOG_WARNING, "Unable to play dialtone on channel %d, do you have defaultzone and loadzone defined?\n", i->channel);
					if (ast_pthread_create_detached(&threadid, NULL, analog_ss_thread, chan)) {
						ast_log(LOG_WARNING, "Unable to start simple switch thread on channel %d\n", i->channel);
						res = tone_zone_play_tone(i->subs[SUB_REAL].dfd, DAHDI_TONE_CONGESTION);
						if (res < 0)
							ast_log(LOG_WARNING, "Unable to play congestion tone on channel %d\n", i->channel);
						ast_hangup(chan);
					}
				} else
					ast_log(LOG_WARNING, "Unable to create channel\n");
			}

			ast_callid_threadstorage_auto_clean(callid, callid_created);
			break;
		case SIG_FXSLS:
		case SIG_FXSGS:
		case SIG_FXSKS:
				i->ringt = i->ringt_base;
				/* Fall through */
		case SIG_EMWINK:
		case SIG_FEATD:
		case SIG_FEATDMF:
		case SIG_FEATDMF_TA:
		case SIG_E911:
		case SIG_FGC_CAMA:
		case SIG_FGC_CAMAMF:
		case SIG_FEATB:
		case SIG_EM:
		case SIG_EM_E1:
		case SIG_SFWINK:
		case SIG_SF_FEATD:
		case SIG_SF_FEATDMF:
		case SIG_SF_FEATB:
		case SIG_SF:
			/* Check for callerid, digits, etc */
			callid_created = ast_callid_threadstorage_auto(&callid);
			if (i->cid_start == CID_START_POLARITY_IN) {
				chan = dahdi_new(i, AST_STATE_PRERING, 0, SUB_REAL, 0, NULL, NULL, callid);
			} else {
				chan = dahdi_new(i, AST_STATE_RING, 0, SUB_REAL, 0, NULL, NULL, callid);
			}

			if (!chan) {
				ast_log(LOG_WARNING, "Cannot allocate new structure on channel %d\n", i->channel);
			} else if (ast_pthread_create_detached(&threadid, NULL, analog_ss_thread, chan)) {
				ast_log(LOG_WARNING, "Unable to start simple switch thread on channel %d\n", i->channel);
				res = tone_zone_play_tone(i->subs[SUB_REAL].dfd, DAHDI_TONE_CONGESTION);
				if (res < 0) {
					ast_log(LOG_WARNING, "Unable to play congestion tone on channel %d\n", i->channel);
				}
				ast_hangup(chan);
			}

			ast_callid_threadstorage_auto_clean(callid, callid_created);
			break;
		default:
			ast_log(LOG_WARNING, "Don't know how to handle ring/answer with signalling %s on channel %d\n", sig2str(i->sig), i->channel);
			res = tone_zone_play_tone(i->subs[SUB_REAL].dfd, DAHDI_TONE_CONGESTION);
			if (res < 0)
				ast_log(LOG_WARNING, "Unable to play congestion tone on channel %d\n", i->channel);
			return NULL;
		}
		break;
	case DAHDI_EVENT_NOALARM:
		switch (i->sig) {
#if defined(HAVE_PRI)
		case SIG_PRI_LIB_HANDLE_CASES:
			ast_mutex_lock(&i->lock);
			sig_pri_chan_alarm_notify(i->sig_pvt, 1);
			ast_mutex_unlock(&i->lock);
			break;
#endif	/* defined(HAVE_PRI) */
#if defined(HAVE_SS7)
		case SIG_SS7:
			sig_ss7_set_alarm(i->sig_pvt, 0);
			break;
#endif	/* defined(HAVE_SS7) */
		default:
			i->inalarm = 0;
			break;
		}
		handle_clear_alarms(i);
		break;
	case DAHDI_EVENT_ALARM:
		switch (i->sig) {
#if defined(HAVE_PRI)
		case SIG_PRI_LIB_HANDLE_CASES:
			ast_mutex_lock(&i->lock);
			sig_pri_chan_alarm_notify(i->sig_pvt, 0);
			ast_mutex_unlock(&i->lock);
			break;
#endif	/* defined(HAVE_PRI) */
#if defined(HAVE_SS7)
		case SIG_SS7:
			sig_ss7_set_alarm(i->sig_pvt, 1);
			break;
#endif	/* defined(HAVE_SS7) */
		default:
			i->inalarm = 1;
			break;
		}
		res = get_alarms(i);
		handle_alarms(i, res);
		/* fall thru intentionally */
	case DAHDI_EVENT_ONHOOK:
		if (i->radio)
			break;
		/* Back on hook.  Hang up. */
		switch (i->sig) {
		case SIG_FXOLS:
		case SIG_FXOGS:
		case SIG_FEATD:
		case SIG_FEATDMF:
		case SIG_FEATDMF_TA:
		case SIG_E911:
		case SIG_FGC_CAMA:
		case SIG_FGC_CAMAMF:
		case SIG_FEATB:
		case SIG_EM:
		case SIG_EM_E1:
		case SIG_EMWINK:
		case SIG_SF_FEATD:
		case SIG_SF_FEATDMF:
		case SIG_SF_FEATB:
		case SIG_SF:
		case SIG_SFWINK:
		case SIG_FXSLS:
		case SIG_FXSGS:
		case SIG_FXSKS:
		case SIG_FXOKS:
			dahdi_ec_disable(i);
			/* Diddle the battery for the zhone */
#ifdef ZHONE_HACK
			dahdi_set_hook(i->subs[SUB_REAL].dfd, DAHDI_OFFHOOK);
			usleep(1);
#endif
			res = tone_zone_play_tone(i->subs[SUB_REAL].dfd, -1);
			dahdi_set_hook(i->subs[SUB_REAL].dfd, DAHDI_ONHOOK);
			break;
		case SIG_SS7:
		case SIG_PRI_LIB_HANDLE_CASES:
			dahdi_ec_disable(i);
			res = tone_zone_play_tone(i->subs[SUB_REAL].dfd, -1);
			break;
		default:
			ast_log(LOG_WARNING, "Don't know how to handle on hook with signalling %s on channel %d\n", sig2str(i->sig), i->channel);
			res = tone_zone_play_tone(i->subs[SUB_REAL].dfd, -1);
			return NULL;
		}
		break;
	case DAHDI_EVENT_POLARITY:
		switch (i->sig) {
		case SIG_FXSLS:
		case SIG_FXSKS:
		case SIG_FXSGS:
			/* We have already got a PR before the channel was
			   created, but it wasn't handled. We need polarity
			   to be REV for remote hangup detection to work.
			   At least in Spain */
			callid_created = ast_callid_threadstorage_auto(&callid);
			if (i->hanguponpolarityswitch)
				i->polarity = POLARITY_REV;
			if (i->cid_start == CID_START_POLARITY || i->cid_start == CID_START_POLARITY_IN) {
				i->polarity = POLARITY_REV;
				ast_verb(2, "Starting post polarity "
					"CID detection on channel %d\n",
					i->channel);
				chan = dahdi_new(i, AST_STATE_PRERING, 0, SUB_REAL, 0, NULL, NULL, callid);
				if (!chan) {
					ast_log(LOG_WARNING, "Cannot allocate new structure on channel %d\n", i->channel);
				} else if (ast_pthread_create_detached(&threadid, NULL, analog_ss_thread, chan)) {
					ast_log(LOG_WARNING, "Unable to start simple switch thread on channel %d\n", i->channel);
					ast_hangup(chan);
				}
			}
			ast_callid_threadstorage_auto_clean(callid, callid_created);
			break;
		default:
			ast_log(LOG_WARNING, "handle_init_event detected "
				"polarity reversal on non-FXO (SIG_FXS) "
				"interface %d\n", i->channel);
		}
		break;
	case DAHDI_EVENT_REMOVED: /* destroy channel, will actually do so in do_monitor */
		ast_log(LOG_NOTICE,
				"Got DAHDI_EVENT_REMOVED. Destroying channel %d\n",
				i->channel);
		return i;
	case DAHDI_EVENT_NEONMWI_ACTIVE:
		if (i->mwimonitor_neon) {
			notify_message(i->mailbox, 1);
			ast_log(LOG_NOTICE, "NEON MWI set for channel %d, mailbox %s \n", i->channel, i->mailbox);
		}
		break;
	case DAHDI_EVENT_NEONMWI_INACTIVE:
		if (i->mwimonitor_neon) {
			notify_message(i->mailbox, 0);
			ast_log(LOG_NOTICE, "NEON MWI cleared for channel %d, mailbox %s\n", i->channel, i->mailbox);
		}
		break;
	}
	return NULL;
}

static void monitor_pfds_clean(void *arg) {
	struct pollfd **pfds = arg;
	ast_free(*pfds);
}

static void *do_monitor(void *data)
{
	int count, res, res2, spoint, pollres=0;
	struct dahdi_pvt *i;
	struct dahdi_pvt *last = NULL;
	struct dahdi_pvt *doomed;
	time_t thispass = 0, lastpass = 0;
	int found;
	char buf[1024];
	struct pollfd *pfds=NULL;
	int lastalloc = -1;
	/* This thread monitors all the frame relay interfaces which are not yet in use
	   (and thus do not have a separate thread) indefinitely */
	/* From here on out, we die whenever asked */
#if 0
	if (pthread_setcanceltype(PTHREAD_CANCEL_ASYNCHRONOUS, NULL)) {
		ast_log(LOG_WARNING, "Unable to set cancel type to asynchronous\n");
		return NULL;
	}
	ast_debug(1, "Monitor starting...\n");
#endif
	pthread_setcancelstate(PTHREAD_CANCEL_DISABLE, NULL);

	pthread_cleanup_push(monitor_pfds_clean, &pfds);
	for (;;) {
		/* Lock the interface list */
		ast_mutex_lock(&iflock);
		if (!pfds || (lastalloc != ifcount)) {
			if (pfds) {
				ast_free(pfds);
				pfds = NULL;
			}
			if (ifcount) {
				if (!(pfds = ast_calloc(1, ifcount * sizeof(*pfds)))) {
					ast_mutex_unlock(&iflock);
					return NULL;
				}
			}
			lastalloc = ifcount;
		}
		/* Build the stuff we're going to poll on, that is the socket of every
		   dahdi_pvt that does not have an associated owner channel */
		count = 0;
		for (i = iflist; i; i = i->next) {
			ast_mutex_lock(&i->lock);
			if (pfds && (i->subs[SUB_REAL].dfd > -1) && i->sig && (!i->radio) && !(i->sig & SIG_MFCR2)) {
				if (dahdi_analog_lib_handles(i->sig, i->radio, i->oprmode)) {
					struct analog_pvt *p = i->sig_pvt;

					if (!p) {
						ast_log(LOG_ERROR, "No sig_pvt?\n");
					} else if (!p->owner && !p->subs[SUB_REAL].owner) {
						/* This needs to be watched, as it lacks an owner */
						pfds[count].fd = i->subs[SUB_REAL].dfd;
						pfds[count].events = POLLPRI;
						pfds[count].revents = 0;
						/* Message waiting or r2 channels also get watched for reading */
						if (i->cidspill || i->mwisendactive || i->mwimonitor_fsk ||
							(i->cid_start == CID_START_DTMF_NOALERT && (i->sig == SIG_FXSLS || i->sig == SIG_FXSGS || i->sig == SIG_FXSKS))) {
							pfds[count].events |= POLLIN;
						}
						count++;
					}
				} else {
					if (!i->owner && !i->subs[SUB_REAL].owner && !i->mwimonitoractive ) {
						/* This needs to be watched, as it lacks an owner */
						pfds[count].fd = i->subs[SUB_REAL].dfd;
						pfds[count].events = POLLPRI;
						pfds[count].revents = 0;
						/* If we are monitoring for VMWI or sending CID, we need to
						   read from the channel as well */
						if (i->cidspill || i->mwisendactive || i->mwimonitor_fsk ||
							(i->cid_start == CID_START_DTMF_NOALERT && (i->sig == SIG_FXSLS || i->sig == SIG_FXSGS || i->sig == SIG_FXSKS))) {
							pfds[count].events |= POLLIN;
						}
						count++;
					}
				}
			}
			ast_mutex_unlock(&i->lock);
		}
		/* Okay, now that we know what to do, release the interface lock */
		ast_mutex_unlock(&iflock);

		pthread_setcancelstate(PTHREAD_CANCEL_ENABLE, NULL);
		pthread_testcancel();
		/* Wait at least a second for something to happen */
		res = poll(pfds, count, 1000);
		pthread_testcancel();
		pthread_setcancelstate(PTHREAD_CANCEL_DISABLE, NULL);

		/* Okay, poll has finished.  Let's see what happened.  */
		if (res < 0) {
			if ((errno != EAGAIN) && (errno != EINTR))
				ast_log(LOG_WARNING, "poll return %d: %s\n", res, strerror(errno));
			continue;
		}
		/* Alright, lock the interface list again, and let's look and see what has
		   happened */
		ast_mutex_lock(&iflock);
		found = 0;
		spoint = 0;
		lastpass = thispass;
		thispass = time(NULL);
		doomed = NULL;
		for (i = iflist;; i = i->next) {
			if (doomed) {
				dahdi_destroy_channel_range(doomed->channel, doomed->channel);
				doomed = NULL;
			}
			if (!i) {
				break;
			}

			if (thispass != lastpass) {
				if (!found && ((i == last) || ((i == iflist) && !last))) {
					last = i;
					if (last) {
						struct analog_pvt *analog_p = last->sig_pvt;
						/* Only allow MWI to be initiated on a quiescent fxs port */
						if (analog_p
							&& !last->mwisendactive
							&& (last->sig & __DAHDI_SIG_FXO)
							&& !analog_p->fxsoffhookstate
							&& !last->owner
							&& (!ast_strlen_zero(last->mailbox) || last->mwioverride_active)
							&& !analog_p->subs[SUB_REAL].owner /* could be a recall ring from a flash hook hold */
							&& (thispass - analog_p->onhooktime > 3)
							/* In some cases, all of the above checks will pass even if the line is really off-hook.
							 * This last check will give the right answer 100% of the time, but is relatively
							 * "expensive" (it requires an ioctl), so it is last to avoid unnecessary system calls. */
							&& !my_is_off_hook(last)) {
							res = has_voicemail(last);
							if (analog_p->msgstate != res) {
								/* Set driver resources for signalling VMWI */
								res2 = ioctl(last->subs[SUB_REAL].dfd, DAHDI_VMWI, &res);
								if (res2) {
									/* TODO: This message will ALWAYS be generated on some cards; any way to restrict it to those cards where it is interesting? */
									ast_debug(3, "Unable to control message waiting led on channel %d: %s\n", last->channel, strerror(errno));
								}
								/* If enabled for FSK spill then initiate it */
								ast_debug(5, "Initiating MWI FSK spill on channel %d\n", last->channel);
								if (mwi_send_init(last)) {
									ast_log(LOG_WARNING, "Unable to initiate mwi send sequence on channel %d\n", last->channel);
								}
								analog_p->msgstate = res;
								found ++;
							}
						}
						last = last->next;
					}
				}
			}
			if ((i->subs[SUB_REAL].dfd > -1) && i->sig) {
				if (i->radio && !i->owner)
				{
					res = dahdi_get_event(i->subs[SUB_REAL].dfd);
					if (res)
					{
						ast_debug(1, "Monitor doohicky got event %s on radio channel %d\n", event2str(res), i->channel);
						/* Don't hold iflock while handling init events */
						ast_mutex_unlock(&iflock);
						if (dahdi_analog_lib_handles(i->sig, i->radio, i->oprmode))
							doomed = (struct dahdi_pvt *) analog_handle_init_event(i->sig_pvt, dahdievent_to_analogevent(res));
						else
							doomed = handle_init_event(i, res);
						ast_mutex_lock(&iflock);
					}
					continue;
				}
				pollres = ast_fdisset(pfds, i->subs[SUB_REAL].dfd, count, &spoint);
				if (pollres & POLLIN) {
					if (i->owner || i->subs[SUB_REAL].owner) {
#ifdef HAVE_PRI
						if (!i->pri)
#endif
							ast_log(LOG_WARNING, "Whoa....  I'm owned but found (%d) in read...\n", i->subs[SUB_REAL].dfd);
						continue;
					}
					if (!i->mwimonitor_fsk && !i->mwisendactive  && i->cid_start != CID_START_DTMF_NOALERT) {
						ast_log(LOG_WARNING, "Whoa....  I'm not looking for MWI or sending MWI but am reading (%d)...\n", i->subs[SUB_REAL].dfd);
						continue;
					}
					res = read(i->subs[SUB_REAL].dfd, buf, sizeof(buf));
					if (res > 0) {
						if (i->mwimonitor_fsk) {
							if (calc_energy((unsigned char *) buf, res, AST_LAW(i)) > mwilevel) {
								pthread_attr_t attr;
								pthread_t threadid;
								struct mwi_thread_data *mtd;

								pthread_attr_init(&attr);
								pthread_attr_setdetachstate(&attr, PTHREAD_CREATE_DETACHED);

								ast_debug(1, "Maybe some MWI on port %d!\n", i->channel);
								if ((mtd = ast_calloc(1, sizeof(*mtd)))) {
									mtd->pvt = i;
									memcpy(mtd->buf, buf, res);
									mtd->len = res;
									i->mwimonitoractive = 1;
									if (ast_pthread_create_background(&threadid, &attr, mwi_thread, mtd)) {
										ast_log(LOG_WARNING, "Unable to start mwi thread on channel %d\n", i->channel);
										i->mwimonitoractive = 0;
										ast_free(mtd);
									}
								}
							}
						/* If configured to check for a DTMF CID spill that comes without alert (e.g no polarity reversal) */
						} else if (i->cid_start == CID_START_DTMF_NOALERT) {
							int energy;
							struct timeval now;
							/* State machine dtmfcid_holdoff_state allows for the line to settle
							 * before checking again for dtmf energy.  Presently waits for 500 mS before checking again
							*/
							if (1 == i->dtmfcid_holdoff_state) {
								gettimeofday(&i->dtmfcid_delay, NULL);
								i->dtmfcid_holdoff_state = 2;
							} else if (2 == i->dtmfcid_holdoff_state) {
								gettimeofday(&now, NULL);
								if ((int)(now.tv_sec - i->dtmfcid_delay.tv_sec) * 1000000 + (int)now.tv_usec - (int)i->dtmfcid_delay.tv_usec > 500000) {
									i->dtmfcid_holdoff_state = 0;
								}
							} else {
								energy = calc_energy((unsigned char *) buf, res, AST_LAW(i));
								if (!i->mwisendactive && energy > dtmfcid_level) {
									pthread_t threadid;
									struct ast_channel *chan;
									ast_mutex_unlock(&iflock);
									if (dahdi_analog_lib_handles(i->sig, i->radio, i->oprmode)) {
										/* just in case this event changes or somehow destroys a channel, set doomed here too */
										doomed = analog_handle_init_event(i->sig_pvt, ANALOG_EVENT_DTMFCID);
										i->dtmfcid_holdoff_state = 1;
									} else {
										ast_callid callid = 0;
										int callid_created = ast_callid_threadstorage_auto(&callid);
										chan = dahdi_new(i, AST_STATE_PRERING, 0, SUB_REAL, 0, NULL, NULL, callid);
										if (!chan) {
											ast_log(LOG_WARNING, "Cannot allocate new structure on channel %d\n", i->channel);
										} else {
											res = ast_pthread_create_detached(&threadid, NULL, analog_ss_thread, chan);
											if (res) {
												ast_log(LOG_WARNING, "Unable to start simple switch thread on channel %d\n", i->channel);
												ast_hangup(chan);
											} else {
												i->dtmfcid_holdoff_state = 1;
											}
										}
										ast_callid_threadstorage_auto_clean(callid, callid_created);
									}
									ast_mutex_lock(&iflock);
								}
							}
						}
						if (i->mwisendactive) {
							mwi_send_process_buffer(i, res);
						}
					} else {
						ast_log(LOG_WARNING, "Read failed with %d: %s\n", res, strerror(errno));
					}
				}
				if (pollres & POLLPRI) {
					if (i->owner || i->subs[SUB_REAL].owner) {
#ifdef HAVE_PRI
						if (!i->pri)
#endif
							ast_log(LOG_WARNING, "Whoa....  I'm owned but found (%d)...\n", i->subs[SUB_REAL].dfd);
						continue;
					}
					res = dahdi_get_event(i->subs[SUB_REAL].dfd);
					ast_debug(1, "Monitor doohicky got event %s on channel %d\n", event2str(res), i->channel);
					/* Don't hold iflock while handling init events */
					ast_mutex_unlock(&iflock);
					if (0 == i->mwisendactive || 0 == mwi_send_process_event(i, res)) {
						if (dahdi_analog_lib_handles(i->sig, i->radio, i->oprmode))
							doomed = (struct dahdi_pvt *) analog_handle_init_event(i->sig_pvt, dahdievent_to_analogevent(res));
						else
							doomed = handle_init_event(i, res);
					}
					if (i->doreoriginate && res == DAHDI_EVENT_HOOKCOMPLETE) {
						/* Actually automatically reoriginate this FXS line, if directed to.
						 * We should get a DAHDI_EVENT_HOOKCOMPLETE from the loop disconnect
						 * doing its thing (one reason why this is for FXOKS only: FXOLS
						 * hangups don't give us any DAHDI events to piggyback off of)*/
						i->doreoriginate = 0;
						/* Double check the channel is still off-hook. There's only about a millisecond
						 * between when doreoriginate is set high and we see that here, but just to be safe. */
						if (!my_is_off_hook(i)) {
							ast_debug(1, "Woah! Went back on hook before reoriginate could happen on channel %d\n", i->channel);
						} else {
							ast_verb(3, "Automatic reorigination on channel %d\n", i->channel);
							res = DAHDI_EVENT_RINGOFFHOOK; /* Pretend that the physical channel just went off hook */
							if (dahdi_analog_lib_handles(i->sig, i->radio, i->oprmode)) {
								doomed = (struct dahdi_pvt *) analog_handle_init_event(i->sig_pvt, dahdievent_to_analogevent(res));
							} else {
								doomed = handle_init_event(i, res);
							}
						}
					}
					ast_mutex_lock(&iflock);
				}
			}
		}
		ast_mutex_unlock(&iflock);
		release_doomed_pris();
#ifdef HAVE_OPENR2
		dahdi_r2_destroy_nodev();
#endif
	}
	/* Never reached */
	pthread_cleanup_pop(1);
	return NULL;

}

static int restart_monitor(void)
{
	/* If we're supposed to be stopped -- stay stopped */
	if (monitor_thread == AST_PTHREADT_STOP)
		return 0;
	ast_mutex_lock(&monlock);
	if (monitor_thread == pthread_self()) {
		ast_mutex_unlock(&monlock);
		ast_log(LOG_WARNING, "Cannot kill myself\n");
		return -1;
	}
	if (monitor_thread != AST_PTHREADT_NULL) {
		/* Wake up the thread */
		pthread_kill(monitor_thread, SIGURG);
	} else {
		/* Start a new monitor */
		if (ast_pthread_create_background(&monitor_thread, NULL, do_monitor, NULL) < 0) {
			ast_mutex_unlock(&monlock);
			ast_log(LOG_ERROR, "Unable to start monitor thread.\n");
			return -1;
		}
	}
	ast_mutex_unlock(&monlock);
	return 0;
}

#if defined(HAVE_PRI)
static int pri_resolve_span(int *span, int channel, int offset, struct dahdi_spaninfo *si)
{
	int x;
	int trunkgroup;
	/* Get appropriate trunk group if there is one */
	trunkgroup = pris[*span].mastertrunkgroup;
	if (trunkgroup) {
		/* Select a specific trunk group */
		for (x = 0; x < NUM_SPANS; x++) {
			if (pris[x].pri.trunkgroup == trunkgroup) {
				*span = x;
				return 0;
			}
		}
		ast_log(LOG_WARNING, "Channel %d on span %d configured to use nonexistent trunk group %d\n", channel, *span, trunkgroup);
		*span = -1;
	} else {
		if (pris[*span].pri.trunkgroup) {
			ast_log(LOG_WARNING, "Unable to use span %d implicitly since it is trunk group %d (please use spanmap)\n", *span, pris[*span].pri.trunkgroup);
			*span = -1;
		} else if (pris[*span].mastertrunkgroup) {
			ast_log(LOG_WARNING, "Unable to use span %d implicitly since it is already part of trunk group %d\n", *span, pris[*span].mastertrunkgroup);
			*span = -1;
		} else {
			if (si->totalchans == 31) {
				/* E1 */
				pris[*span].dchannels[0] = 16 + offset;
			} else if (si->totalchans == 24) {
				/* T1 or J1 */
				pris[*span].dchannels[0] = 24 + offset;
			} else if (si->totalchans == 3) {
				/* BRI */
				pris[*span].dchannels[0] = 3 + offset;
			} else {
				ast_log(LOG_WARNING, "Unable to use span %d, since the D-channel cannot be located (unexpected span size of %d channels)\n", *span, si->totalchans);
				*span = -1;
				return 0;
			}
			pris[*span].pri.span = *span + 1;
		}
	}
	return 0;
}
#endif	/* defined(HAVE_PRI) */

#if defined(HAVE_PRI)
static int pri_create_trunkgroup(int trunkgroup, int *channels)
{
	struct dahdi_spaninfo si;
	struct dahdi_params p;
	int fd;
	int span;
	int ospan=0;
	int x,y;
	for (x = 0; x < NUM_SPANS; x++) {
		if (pris[x].pri.trunkgroup == trunkgroup) {
			ast_log(LOG_WARNING, "Trunk group %d already exists on span %d, Primary d-channel %d\n", trunkgroup, x + 1, pris[x].dchannels[0]);
			return -1;
		}
	}
	for (y = 0; y < SIG_PRI_NUM_DCHANS; y++) {
		if (!channels[y])
			break;
		memset(&si, 0, sizeof(si));
		memset(&p, 0, sizeof(p));
		fd = open("/dev/dahdi/channel", O_RDWR);
		if (fd < 0) {
			ast_log(LOG_WARNING, "Failed to open channel: %s\n", strerror(errno));
			return -1;
		}
		x = channels[y];
		if (ioctl(fd, DAHDI_SPECIFY, &x)) {
			ast_log(LOG_WARNING, "Failed to specify channel %d: %s\n", channels[y], strerror(errno));
			close(fd);
			return -1;
		}
		if (ioctl(fd, DAHDI_GET_PARAMS, &p)) {
			ast_log(LOG_WARNING, "Failed to get channel parameters for channel %d: %s\n", channels[y], strerror(errno));
			close(fd);
			return -1;
		}
		if (ioctl(fd, DAHDI_SPANSTAT, &si)) {
			ast_log(LOG_WARNING, "Failed go get span information on channel %d (span %d): %s\n", channels[y], p.spanno, strerror(errno));
			close(fd);
			return -1;
		}
		span = p.spanno - 1;
		if (pris[span].pri.trunkgroup) {
			ast_log(LOG_WARNING, "Span %d is already provisioned for trunk group %d\n", span + 1, pris[span].pri.trunkgroup);
			close(fd);
			return -1;
		}
		if (pris[span].pri.pvts[0]) {
			ast_log(LOG_WARNING, "Span %d is already provisioned with channels (implicit PRI maybe?)\n", span + 1);
			close(fd);
			return -1;
		}
		if (!y) {
			pris[span].pri.trunkgroup = trunkgroup;
			ospan = span;
		}
		pris[ospan].dchannels[y] = channels[y];
		pris[span].pri.span = span + 1;
		close(fd);
	}
	return 0;
}
#endif	/* defined(HAVE_PRI) */

#if defined(HAVE_PRI)
static int pri_create_spanmap(int span, int trunkgroup, int logicalspan)
{
	if (pris[span].mastertrunkgroup) {
		ast_log(LOG_WARNING, "Span %d is already part of trunk group %d, cannot add to trunk group %d\n", span + 1, pris[span].mastertrunkgroup, trunkgroup);
		return -1;
	}
	pris[span].mastertrunkgroup = trunkgroup;
	pris[span].prilogicalspan = logicalspan;
	return 0;
}
#endif	/* defined(HAVE_PRI) */

#if defined(HAVE_SS7)
static unsigned int parse_pointcode(const char *pcstring)
{
	unsigned int code1, code2, code3;
	int numvals;

	numvals = sscanf(pcstring, "%30d-%30d-%30d", &code1, &code2, &code3);
	if (numvals == 1)
		return code1;
	if (numvals == 3)
		return (code1 << 16) | (code2 << 8) | code3;

	return 0;
}
#endif	/* defined(HAVE_SS7) */

#if defined(HAVE_SS7)
static struct dahdi_ss7 * ss7_resolve_linkset(int linkset)
{
	if ((linkset < 0) || (linkset >= NUM_SPANS))
		return NULL;
	else
		return &linksets[linkset - 1];
}
#endif	/* defined(HAVE_SS7) */

#ifdef HAVE_OPENR2
static void dahdi_r2_destroy_links(void)
{
	struct r2link_entry *cur;

	/* Queue everything for removal */
	AST_LIST_LOCK(&r2links);
	AST_LIST_TRAVERSE_SAFE_BEGIN(&r2links, cur, list) {
		ast_debug(3, "MFC/R2 link #%d queued for destruction\n", cur->mfcr2.index);
		AST_LIST_MOVE_CURRENT(&nodev_r2links, list);
	}
	AST_LIST_TRAVERSE_SAFE_END;
	AST_LIST_UNLOCK(&r2links);
	/* Now destroy properly */
	dahdi_r2_destroy_nodev();
}

/* This is an artificial convenient capacity, to keep at most a full E1 of channels in a single thread */
#define R2_LINK_CAPACITY 30
static struct r2link_entry *dahdi_r2_get_link(const struct dahdi_chan_conf *conf)
{
	struct r2link_entry *cur = NULL;
	/* Only create a new R2 link if
	   1. This is the first link requested
	   2. Configuration changed
	   3. We got more channels than supported per link */
	AST_LIST_LOCK(&r2links);
	if (! AST_LIST_EMPTY(&r2links)) {
		cur = AST_LIST_LAST(&r2links);
		if (memcmp(&conf->mfcr2, &cur->mfcr2.conf, sizeof(conf->mfcr2))) {
			ast_debug(3, "Need new R2 link because of: Configuration change\n");
			cur = NULL;
		} else if (cur->mfcr2.numchans == R2_LINK_CAPACITY) {
			ast_debug(3, "Need new R2 link because of: Capacity (%d)\n", R2_LINK_CAPACITY);
			cur = NULL;
		}
	}
	if (!cur) {
		struct r2link_entry *tmp = NULL;
		int new_idx = r2links_count + 1;
		int i;
		for (i = 1; i <= r2links_count; i++) {
			int i_unused = 1;
			AST_LIST_TRAVERSE(&r2links, tmp, list) {
				if (i == tmp->mfcr2.index) {
					i_unused = 0;
					break;
				}
			}
			if (i_unused) {
				new_idx = i;
				break;
			}
		}
		cur = ast_calloc(1, sizeof(*cur));
		if (!cur) {
			ast_log(LOG_ERROR, "Cannot allocate R2 link!\n");
			return NULL;
		}
		cur->mfcr2.index = new_idx;
		cur->mfcr2.r2master = AST_PTHREADT_NULL;
		r2links_count++;
		ast_debug(3, "Created new R2 link #%d (now have %d)\n", new_idx, r2links_count);
		AST_LIST_INSERT_TAIL(&r2links, cur, list);
	}
	AST_LIST_UNLOCK(&r2links);
	return cur;
}

static int dahdi_r2_set_context(struct dahdi_mfcr2 *r2_link, const struct dahdi_chan_conf *conf)
{
	char tmplogdir[] = "/tmp";
	char logdir[OR2_MAX_PATH];
	int threshold = 0;
	int snres = 0;
	r2_link->protocol_context = openr2_context_new(NULL, &dahdi_r2_event_iface,
			&dahdi_r2_transcode_iface, conf->mfcr2.variant, conf->mfcr2.max_ani,
			conf->mfcr2.max_dnis);
	if (!r2_link->protocol_context) {
		return -1;
	}
	openr2_context_set_log_level(r2_link->protocol_context, conf->mfcr2.loglevel);
	openr2_context_set_ani_first(r2_link->protocol_context, conf->mfcr2.get_ani_first);
#if defined(OR2_LIB_INTERFACE) && OR2_LIB_INTERFACE > 1
	openr2_context_set_skip_category_request(r2_link->protocol_context, conf->mfcr2.skip_category_request);
#endif
	openr2_context_set_mf_threshold(r2_link->protocol_context, threshold);
	openr2_context_set_mf_back_timeout(r2_link->protocol_context, conf->mfcr2.mfback_timeout);
	openr2_context_set_metering_pulse_timeout(r2_link->protocol_context, conf->mfcr2.metering_pulse_timeout);
	openr2_context_set_double_answer(r2_link->protocol_context, conf->mfcr2.double_answer);
	openr2_context_set_immediate_accept(r2_link->protocol_context, conf->mfcr2.immediate_accept);
#if defined(OR2_LIB_INTERFACE) && OR2_LIB_INTERFACE > 2
	openr2_context_set_dtmf_dialing(r2_link->protocol_context, conf->mfcr2.dtmf_dialing, conf->mfcr2.dtmf_time_on, conf->mfcr2.dtmf_time_off);
	openr2_context_set_dtmf_detection(r2_link->protocol_context, conf->mfcr2.dtmf_detection);
#endif
#if defined(OR2_LIB_INTERFACE) && OR2_LIB_INTERFACE > 3
	openr2_context_set_dtmf_detection_end_timeout(r2_link->protocol_context, conf->mfcr2.dtmf_end_timeout);
#endif
	if (ast_strlen_zero(conf->mfcr2.logdir)) {
		if (openr2_context_set_log_directory(r2_link->protocol_context, tmplogdir)) {
			ast_log(LOG_ERROR, "Failed setting default MFC/R2 log directory %s\n", tmplogdir);
		}
	} else {
		snres = snprintf(logdir, sizeof(logdir), "%s/%s/%s", ast_config_AST_LOG_DIR, "mfcr2", conf->mfcr2.logdir);
		if (snres >= sizeof(logdir)) {
			ast_log(LOG_ERROR, "MFC/R2 logging directory truncated, using %s\n", tmplogdir);
			if (openr2_context_set_log_directory(r2_link->protocol_context, tmplogdir)) {
				ast_log(LOG_ERROR, "Failed setting default MFC/R2 log directory %s\n", tmplogdir);
			}
		} else {
			if (openr2_context_set_log_directory(r2_link->protocol_context, logdir)) {
				ast_log(LOG_ERROR, "Failed setting MFC/R2 log directory %s\n", logdir);
			}
		}
	}
	if (!ast_strlen_zero(conf->mfcr2.r2proto_file)) {
		if (openr2_context_configure_from_advanced_file(r2_link->protocol_context, conf->mfcr2.r2proto_file)) {
			ast_log(LOG_ERROR, "Failed to configure r2context from advanced configuration file %s\n", conf->mfcr2.r2proto_file);
		}
	}
	/* Save the configuration used to setup this link */
	memcpy(&r2_link->conf, &conf->mfcr2, sizeof(r2_link->conf));
	return 0;
}
#endif

/* converts a DAHDI sigtype to signalling as can be configured from
 * chan_dahdi.conf.
 * While both have basically the same values, this will later be the
 * place to add filters and sanity checks
 */
static int sigtype_to_signalling(int sigtype)
{
	return sigtype;
}

/*!
 * \internal
 * \brief Initialize/create a channel interface.
 *
 * \param channel Channel interface number to initialize/create.
 * \param conf Configuration parameters to initialize interface with.
 * \param reloading What we are doing now:
 * 0 - initial module load,
 * 1 - module reload,
 * 2 - module restart
 *
 * \retval Interface-pointer initialized/created
 * \retval NULL if error
 */
static struct dahdi_pvt *mkintf(int channel, const struct dahdi_chan_conf *conf, int reloading)
{
	/* Make a dahdi_pvt structure for this interface */
	struct dahdi_pvt *tmp;/*!< Current channel structure initializing */
	char fn[80];
	struct dahdi_bufferinfo bi;

	int res;
#if defined(HAVE_PRI)
	int span = 0;
#endif	/* defined(HAVE_PRI) */
	int here = 0;/*!< TRUE if the channel interface already exists. */
	int x;
	struct analog_pvt *analog_p = NULL;
	struct dahdi_params p;
#if defined(HAVE_PRI)
	struct dahdi_spaninfo si;
	struct sig_pri_chan *pri_chan = NULL;
#endif	/* defined(HAVE_PRI) */
#if defined(HAVE_SS7)
	struct sig_ss7_chan *ss7_chan = NULL;
#endif	/* defined(HAVE_SS7) */

	/* Search channel interface list to see if it already exists. */
	for (tmp = iflist; tmp; tmp = tmp->next) {
		if (!tmp->destroy) {
			if (tmp->channel == channel) {
				/* The channel interface already exists. */
				here = 1;
				break;
			}
			if (tmp->channel > channel) {
				/* No way it can be in the sorted list. */
				tmp = NULL;
				break;
			}
		}
	}

	if (!here && reloading != 1) {
		tmp = ast_calloc(1, sizeof(*tmp));
		if (!tmp) {
			return NULL;
		}
		tmp->cc_params = ast_cc_config_params_init();
		if (!tmp->cc_params) {
			ast_free(tmp);
			return NULL;
		}
		ast_mutex_init(&tmp->lock);
		ifcount++;
		for (x = 0; x < 3; x++)
			tmp->subs[x].dfd = -1;
		tmp->channel = channel;
		tmp->priindication_oob = conf->chan.priindication_oob;
	}

	if (tmp) {
		int chan_sig = conf->chan.sig;

		/* If there are variables in tmp before it is updated to match the new config, clear them */
		if (reloading && tmp->vars) {
			ast_variables_destroy(tmp->vars);
			tmp->vars = NULL;
		}

		if (!here) {
			/* Can only get here if this is a new channel interface being created. */
			if ((channel != CHAN_PSEUDO)) {
				int count = 0;

				snprintf(fn, sizeof(fn), "%d", channel);
				/* Open non-blocking */
				tmp->subs[SUB_REAL].dfd = dahdi_open(fn);
				/* Retry open on restarts, but don't keep retrying if the channel doesn't exist (e.g. not configured) */
				while (tmp->subs[SUB_REAL].dfd < 0 && reloading == 2 && count < 1000 && errno != ENXIO) { /* the kernel may not call dahdi_release fast enough for the open flagbit to be cleared in time */
					usleep(1);
					tmp->subs[SUB_REAL].dfd = dahdi_open(fn);
					count++;
				}
				/* Allocate a DAHDI structure */
				if (tmp->subs[SUB_REAL].dfd < 0) {
					ast_log(LOG_ERROR, "Unable to open channel %d: %s\nhere = %d, tmp->channel = %d, channel = %d\n", channel, strerror(errno), here, tmp->channel, channel);
					destroy_dahdi_pvt(tmp);
					return NULL;
				}
				memset(&p, 0, sizeof(p));
				res = ioctl(tmp->subs[SUB_REAL].dfd, DAHDI_GET_PARAMS, &p);
				if (res < 0) {
					ast_log(LOG_ERROR, "Unable to get parameters: %s\n", strerror(errno));
					destroy_dahdi_pvt(tmp);
					return NULL;
				}
				if (conf->is_sig_auto)
					chan_sig = sigtype_to_signalling(p.sigtype);
				if (p.sigtype != (chan_sig & 0x3ffff)) {
					ast_log(LOG_ERROR, "Signalling requested on channel %d is %s but line is in %s signalling\n", channel, sig2str(chan_sig), sig2str(p.sigtype));
					destroy_dahdi_pvt(tmp);
					return NULL;
				}
				tmp->law_default = p.curlaw;
				tmp->law = p.curlaw;
				tmp->span = p.spanno;
#if defined(HAVE_PRI)
				span = p.spanno - 1;
#endif	/* defined(HAVE_PRI) */
			} else {
				chan_sig = 0;
			}
			tmp->sig = chan_sig;
			tmp->outsigmod = conf->chan.outsigmod;

			if (dahdi_analog_lib_handles(chan_sig, tmp->radio, tmp->oprmode)) {
				analog_p = analog_new(dahdisig_to_analogsig(chan_sig), tmp);
				if (!analog_p) {
					destroy_dahdi_pvt(tmp);
					return NULL;
				}
				tmp->sig_pvt = analog_p;
			}
#if defined(HAVE_SS7)
			if (chan_sig == SIG_SS7) {
				struct dahdi_ss7 *ss7;
				int clear = 0;

				if (ioctl(tmp->subs[SUB_REAL].dfd, DAHDI_AUDIOMODE, &clear)) {
					ast_log(LOG_ERROR, "Unable to set clear mode on clear channel %d of span %d: %s\n", channel, p.spanno, strerror(errno));
					destroy_dahdi_pvt(tmp);
					return NULL;
				}

				ss7 = ss7_resolve_linkset(cur_linkset);
				if (!ss7) {
					ast_log(LOG_ERROR, "Unable to find linkset %d\n", cur_linkset);
					destroy_dahdi_pvt(tmp);
					return NULL;
				}
				ss7->ss7.span = cur_linkset;
				if (cur_cicbeginswith < 0) {
					ast_log(LOG_ERROR, "Need to set cicbeginswith for the channels!\n");
					destroy_dahdi_pvt(tmp);
					return NULL;
				}
				ss7_chan = sig_ss7_chan_new(tmp, &ss7->ss7);
				if (!ss7_chan) {
					destroy_dahdi_pvt(tmp);
					return NULL;
				}
				tmp->sig_pvt = ss7_chan;
				tmp->ss7 = &ss7->ss7;

				ss7_chan->channel = tmp->channel;
				ss7_chan->cic = cur_cicbeginswith++;

				/* DB: Add CIC's DPC information */
				ss7_chan->dpc = cur_defaultdpc;

				ss7->ss7.pvts[ss7->ss7.numchans++] = ss7_chan;

				ast_copy_string(ss7->ss7.internationalprefix, conf->ss7.ss7.internationalprefix, sizeof(ss7->ss7.internationalprefix));
				ast_copy_string(ss7->ss7.nationalprefix, conf->ss7.ss7.nationalprefix, sizeof(ss7->ss7.nationalprefix));
				ast_copy_string(ss7->ss7.subscriberprefix, conf->ss7.ss7.subscriberprefix, sizeof(ss7->ss7.subscriberprefix));
				ast_copy_string(ss7->ss7.unknownprefix, conf->ss7.ss7.unknownprefix, sizeof(ss7->ss7.unknownprefix));
				ast_copy_string(ss7->ss7.networkroutedprefix, conf->ss7.ss7.networkroutedprefix, sizeof(ss7->ss7.networkroutedprefix));

				ss7->ss7.called_nai = conf->ss7.ss7.called_nai;
				ss7->ss7.calling_nai = conf->ss7.ss7.calling_nai;
			}
#endif	/* defined(HAVE_SS7) */
#ifdef HAVE_OPENR2
			if (chan_sig == SIG_MFCR2) {
				struct dahdi_mfcr2 *r2_link;
				struct r2link_entry *r2_le = dahdi_r2_get_link(conf);
				r2_link = &r2_le->mfcr2;
				if (!r2_link) {
					ast_log(LOG_WARNING, "Cannot get another R2 DAHDI context!\n");
					destroy_dahdi_pvt(tmp);
					return NULL;
				}
				if (!r2_link->protocol_context && dahdi_r2_set_context(r2_link, conf)) {
					ast_log(LOG_ERROR, "Cannot create OpenR2 protocol context.\n");
					destroy_dahdi_pvt(tmp);
					return NULL;
				}
				if (r2_link->numchans == ARRAY_LEN(r2_link->pvts)) {
					ast_log(LOG_ERROR, "Cannot add more channels to this link!\n");
					destroy_dahdi_pvt(tmp);
					return NULL;
				}
				r2_link->pvts[r2_link->numchans++] = tmp;
				tmp->r2chan = openr2_chan_new_from_fd(r2_link->protocol_context,
						                      tmp->subs[SUB_REAL].dfd,
						                      NULL, NULL);
				if (!tmp->r2chan) {
					openr2_liberr_t err = openr2_context_get_last_error(r2_link->protocol_context);
					ast_log(LOG_ERROR, "Cannot create OpenR2 channel: %s\n", openr2_context_error_string(err));
					destroy_dahdi_pvt(tmp);
					return NULL;
				}
				r2_link->live_chans++;
				tmp->mfcr2 = r2_link;
				if (conf->mfcr2.call_files) {
					openr2_chan_enable_call_files(tmp->r2chan);
				}
				openr2_chan_set_client_data(tmp->r2chan, tmp);
				/* cast seems to be needed to get rid of the annoying warning regarding format attribute  */
				openr2_chan_set_logging_func(tmp->r2chan, (openr2_logging_func_t)dahdi_r2_on_chan_log);
				openr2_chan_set_log_level(tmp->r2chan, conf->mfcr2.loglevel);
				tmp->mfcr2_category = conf->mfcr2.category;
				tmp->mfcr2_charge_calls = conf->mfcr2.charge_calls;
				tmp->mfcr2_allow_collect_calls = conf->mfcr2.allow_collect_calls;
				tmp->mfcr2_forced_release = conf->mfcr2.forced_release;
				tmp->mfcr2_accept_on_offer = conf->mfcr2.accept_on_offer;
				tmp->mfcr2call = 0;
				tmp->mfcr2_dnis_index = 0;
				tmp->mfcr2_ani_index = 0;
			}
#endif
#ifdef HAVE_PRI
			if (dahdi_sig_pri_lib_handles(chan_sig)) {
				int offset;
				int matchesdchan;
				int x,y;
				int myswitchtype = 0;

				offset = 0;
				if (ioctl(tmp->subs[SUB_REAL].dfd, DAHDI_AUDIOMODE, &offset)) {
					ast_log(LOG_ERROR, "Unable to set clear mode on clear channel %d of span %d: %s\n", channel, p.spanno, strerror(errno));
					destroy_dahdi_pvt(tmp);
					return NULL;
				}
				if (span >= NUM_SPANS) {
					ast_log(LOG_ERROR, "Channel %d does not lie on a span I know of (%d)\n", channel, span);
					destroy_dahdi_pvt(tmp);
					return NULL;
				} else {
					si.spanno = 0;
					if (ioctl(tmp->subs[SUB_REAL].dfd,DAHDI_SPANSTAT,&si) == -1) {
						ast_log(LOG_ERROR, "Unable to get span status: %s\n", strerror(errno));
						destroy_dahdi_pvt(tmp);
						return NULL;
					}
					/* Store the logical span first based upon the real span */
					tmp->logicalspan = pris[span].prilogicalspan;
					pri_resolve_span(&span, channel, (channel - p.chanpos), &si);
					if (span < 0) {
						ast_log(LOG_WARNING, "Channel %d: Unable to find locate channel/trunk group!\n", channel);
						destroy_dahdi_pvt(tmp);
						return NULL;
					}
					myswitchtype = conf->pri.pri.switchtype;
					/* Make sure this isn't a d-channel */
					matchesdchan=0;
					for (x = 0; x < NUM_SPANS; x++) {
						for (y = 0; y < SIG_PRI_NUM_DCHANS; y++) {
							if (pris[x].dchannels[y] == tmp->channel) {
								matchesdchan = 1;
								break;
							}
						}
					}
					if (!matchesdchan) {
						if (pris[span].pri.nodetype && (pris[span].pri.nodetype != conf->pri.pri.nodetype)) {
							ast_log(LOG_ERROR, "Span %d is already a %s node\n", span + 1, pri_node2str(pris[span].pri.nodetype));
							destroy_dahdi_pvt(tmp);
							return NULL;
						}
						if (pris[span].pri.switchtype && (pris[span].pri.switchtype != myswitchtype)) {
							ast_log(LOG_ERROR, "Span %d is already a %s switch\n", span + 1, pri_switch2str(pris[span].pri.switchtype));
							destroy_dahdi_pvt(tmp);
							return NULL;
						}
						if ((pris[span].pri.dialplan) && (pris[span].pri.dialplan != conf->pri.pri.dialplan)) {
							ast_log(LOG_ERROR, "Span %d is already a %s dialing plan\n", span + 1, pris[span].pri.dialplan == -1 ? "Dynamically set dialplan in ISDN" : pri_plan2str(pris[span].pri.dialplan));
							destroy_dahdi_pvt(tmp);
							return NULL;
						}
						if (!ast_strlen_zero(pris[span].pri.idledial) && strcmp(pris[span].pri.idledial, conf->pri.pri.idledial)) {
							ast_log(LOG_ERROR, "Span %d already has idledial '%s'.\n", span + 1, conf->pri.pri.idledial);
							destroy_dahdi_pvt(tmp);
							return NULL;
						}
						if (!ast_strlen_zero(pris[span].pri.idleext) && strcmp(pris[span].pri.idleext, conf->pri.pri.idleext)) {
							ast_log(LOG_ERROR, "Span %d already has idleext '%s'.\n", span + 1, conf->pri.pri.idleext);
							destroy_dahdi_pvt(tmp);
							return NULL;
						}
						if (pris[span].pri.minunused && (pris[span].pri.minunused != conf->pri.pri.minunused)) {
							ast_log(LOG_ERROR, "Span %d already has minunused of %d.\n", span + 1, conf->pri.pri.minunused);
							destroy_dahdi_pvt(tmp);
							return NULL;
						}
						if (pris[span].pri.minidle && (pris[span].pri.minidle != conf->pri.pri.minidle)) {
							ast_log(LOG_ERROR, "Span %d already has minidle of %d.\n", span + 1, conf->pri.pri.minidle);
							destroy_dahdi_pvt(tmp);
							return NULL;
						}
						if (pris[span].pri.numchans >= ARRAY_LEN(pris[span].pri.pvts)) {
							ast_log(LOG_ERROR, "Unable to add channel %d: Too many channels in trunk group %d!\n", channel,
								pris[span].pri.trunkgroup);
							destroy_dahdi_pvt(tmp);
							return NULL;
						}

						pri_chan = sig_pri_chan_new(tmp, &pris[span].pri, tmp->logicalspan, p.chanpos, pris[span].mastertrunkgroup);
						if (!pri_chan) {
							destroy_dahdi_pvt(tmp);
							return NULL;
						}
						tmp->sig_pvt = pri_chan;
						tmp->pri = &pris[span].pri;

						tmp->priexclusive = conf->chan.priexclusive;

						if (!tmp->pri->cc_params) {
							tmp->pri->cc_params = ast_cc_config_params_init();
							if (!tmp->pri->cc_params) {
								destroy_dahdi_pvt(tmp);
								return NULL;
							}
						}
						ast_cc_copy_config_params(tmp->pri->cc_params,
							conf->chan.cc_params);

						pris[span].pri.sig = chan_sig;
						pris[span].pri.nodetype = conf->pri.pri.nodetype;
						pris[span].pri.switchtype = myswitchtype;
						pris[span].pri.nsf = conf->pri.pri.nsf;
						pris[span].pri.dialplan = conf->pri.pri.dialplan;
						pris[span].pri.localdialplan = conf->pri.pri.localdialplan;
						pris[span].pri.cpndialplan = conf->pri.pri.cpndialplan;
						pris[span].pri.pvts[pris[span].pri.numchans++] = tmp->sig_pvt;
						pris[span].pri.minunused = conf->pri.pri.minunused;
						pris[span].pri.minidle = conf->pri.pri.minidle;
						pris[span].pri.overlapdial = conf->pri.pri.overlapdial;
						pris[span].pri.qsigchannelmapping = conf->pri.pri.qsigchannelmapping;
						pris[span].pri.discardremoteholdretrieval = conf->pri.pri.discardremoteholdretrieval;
#if defined(HAVE_PRI_SERVICE_MESSAGES)
						pris[span].pri.enable_service_message_support = conf->pri.pri.enable_service_message_support;
#endif	/* defined(HAVE_PRI_SERVICE_MESSAGES) */
#ifdef HAVE_PRI_INBANDDISCONNECT
						pris[span].pri.inbanddisconnect = conf->pri.pri.inbanddisconnect;
#endif
#if defined(HAVE_PRI_CALL_HOLD)
						pris[span].pri.hold_disconnect_transfer =
							conf->pri.pri.hold_disconnect_transfer;
#endif	/* defined(HAVE_PRI_CALL_HOLD) */
#if defined(HAVE_PRI_CCSS)
						pris[span].pri.cc_ptmp_recall_mode =
							conf->pri.pri.cc_ptmp_recall_mode;
						pris[span].pri.cc_qsig_signaling_link_req =
							conf->pri.pri.cc_qsig_signaling_link_req;
						pris[span].pri.cc_qsig_signaling_link_rsp =
							conf->pri.pri.cc_qsig_signaling_link_rsp;
#endif	/* defined(HAVE_PRI_CCSS) */
#if defined(HAVE_PRI_CALL_WAITING)
						pris[span].pri.max_call_waiting_calls =
							conf->pri.pri.max_call_waiting_calls;
						pris[span].pri.allow_call_waiting_calls =
							conf->pri.pri.allow_call_waiting_calls;
#endif	/* defined(HAVE_PRI_CALL_WAITING) */
						pris[span].pri.transfer = conf->chan.transfer;
						pris[span].pri.facilityenable = conf->pri.pri.facilityenable;
#if defined(HAVE_PRI_L2_PERSISTENCE)
						pris[span].pri.l2_persistence = conf->pri.pri.l2_persistence;
#endif	/* defined(HAVE_PRI_L2_PERSISTENCE) */
						pris[span].pri.colp_send = conf->pri.pri.colp_send;
#if defined(HAVE_PRI_AOC_EVENTS)
						pris[span].pri.aoc_passthrough_flag = conf->pri.pri.aoc_passthrough_flag;
						pris[span].pri.aoce_delayhangup = conf->pri.pri.aoce_delayhangup;
#endif	/* defined(HAVE_PRI_AOC_EVENTS) */
						if (chan_sig == SIG_BRI_PTMP) {
							pris[span].pri.layer1_ignored = conf->pri.pri.layer1_ignored;
						} else {
							/* Option does not apply to this line type. */
							pris[span].pri.layer1_ignored = 0;
						}
						pris[span].pri.append_msn_to_user_tag = conf->pri.pri.append_msn_to_user_tag;
						pris[span].pri.inband_on_setup_ack = conf->pri.pri.inband_on_setup_ack;
						pris[span].pri.inband_on_proceeding = conf->pri.pri.inband_on_proceeding;
						ast_copy_string(pris[span].pri.initial_user_tag, conf->chan.cid_tag, sizeof(pris[span].pri.initial_user_tag));
						ast_copy_string(pris[span].pri.msn_list, conf->pri.pri.msn_list, sizeof(pris[span].pri.msn_list));
#if defined(HAVE_PRI_MWI)
						ast_copy_string(pris[span].pri.mwi_mailboxes,
							conf->pri.pri.mwi_mailboxes,
							sizeof(pris[span].pri.mwi_mailboxes));
						ast_copy_string(pris[span].pri.mwi_vm_boxes,
							conf->pri.pri.mwi_vm_boxes,
							sizeof(pris[span].pri.mwi_vm_boxes));
						ast_copy_string(pris[span].pri.mwi_vm_numbers,
							conf->pri.pri.mwi_vm_numbers,
							sizeof(pris[span].pri.mwi_vm_numbers));
#endif	/* defined(HAVE_PRI_MWI) */
						ast_copy_string(pris[span].pri.idledial, conf->pri.pri.idledial, sizeof(pris[span].pri.idledial));
						ast_copy_string(pris[span].pri.idleext, conf->pri.pri.idleext, sizeof(pris[span].pri.idleext));
						ast_copy_string(pris[span].pri.internationalprefix, conf->pri.pri.internationalprefix, sizeof(pris[span].pri.internationalprefix));
						ast_copy_string(pris[span].pri.nationalprefix, conf->pri.pri.nationalprefix, sizeof(pris[span].pri.nationalprefix));
						ast_copy_string(pris[span].pri.localprefix, conf->pri.pri.localprefix, sizeof(pris[span].pri.localprefix));
						ast_copy_string(pris[span].pri.privateprefix, conf->pri.pri.privateprefix, sizeof(pris[span].pri.privateprefix));
						ast_copy_string(pris[span].pri.unknownprefix, conf->pri.pri.unknownprefix, sizeof(pris[span].pri.unknownprefix));
						pris[span].pri.moh_signaling = conf->pri.pri.moh_signaling;
						pris[span].pri.resetinterval = conf->pri.pri.resetinterval;
#if defined(HAVE_PRI_DISPLAY_TEXT)
						pris[span].pri.display_flags_send = conf->pri.pri.display_flags_send;
						pris[span].pri.display_flags_receive = conf->pri.pri.display_flags_receive;
#endif	/* defined(HAVE_PRI_DISPLAY_TEXT) */
#if defined(HAVE_PRI_MCID)
						pris[span].pri.mcid_send = conf->pri.pri.mcid_send;
#endif	/* defined(HAVE_PRI_MCID) */
						pris[span].pri.force_restart_unavailable_chans = conf->pri.pri.force_restart_unavailable_chans;
#if defined(HAVE_PRI_DATETIME_SEND)
						pris[span].pri.datetime_send = conf->pri.pri.datetime_send;
#endif	/* defined(HAVE_PRI_DATETIME_SEND) */

						for (x = 0; x < PRI_MAX_TIMERS; x++) {
							pris[span].pri.pritimers[x] = conf->pri.pri.pritimers[x];
						}

#if defined(HAVE_PRI_CALL_WAITING)
						/* Channel initial config parameters. */
						pris[span].pri.ch_cfg.stripmsd = conf->chan.stripmsd;
						pris[span].pri.ch_cfg.hidecallerid = conf->chan.hidecallerid;
						pris[span].pri.ch_cfg.hidecalleridname = conf->chan.hidecalleridname;
						pris[span].pri.ch_cfg.immediate = conf->chan.immediate;
						pris[span].pri.ch_cfg.priexclusive = conf->chan.priexclusive;
						pris[span].pri.ch_cfg.priindication_oob = conf->chan.priindication_oob;
						pris[span].pri.ch_cfg.use_callerid = conf->chan.use_callerid;
						pris[span].pri.ch_cfg.use_callingpres = conf->chan.use_callingpres;
						ast_copy_string(pris[span].pri.ch_cfg.context, conf->chan.context, sizeof(pris[span].pri.ch_cfg.context));
						ast_copy_string(pris[span].pri.ch_cfg.mohinterpret, conf->chan.mohinterpret, sizeof(pris[span].pri.ch_cfg.mohinterpret));
#endif	/* defined(HAVE_PRI_CALL_WAITING) */
					} else {
						ast_log(LOG_ERROR, "Channel %d is reserved for D-channel.\n", p.chanpos);
						destroy_dahdi_pvt(tmp);
						return NULL;
					}
				}
			}
#endif
		} else {
			/* already exists in interface list */
			ast_log(LOG_WARNING, "Attempt to configure channel %d with signaling %s ignored because it is already configured to be %s.\n", tmp->channel, dahdi_sig2str(chan_sig), dahdi_sig2str(tmp->sig));
			chan_sig = tmp->sig;
			if (tmp->subs[SUB_REAL].dfd > -1) {
				memset(&p, 0, sizeof(p));
				res = ioctl(tmp->subs[SUB_REAL].dfd, DAHDI_GET_PARAMS, &p);
			}
		}
		/* Adjust starttime on loopstart and kewlstart trunks to reasonable values */
		switch (chan_sig) {
		case SIG_FXSKS:
		case SIG_FXSLS:
		case SIG_EM:
		case SIG_EM_E1:
		case SIG_EMWINK:
		case SIG_FEATD:
		case SIG_FEATDMF:
		case SIG_FEATDMF_TA:
		case SIG_FEATB:
		case SIG_E911:
		case SIG_SF:
		case SIG_SFWINK:
		case SIG_FGC_CAMA:
		case SIG_FGC_CAMAMF:
		case SIG_SF_FEATD:
		case SIG_SF_FEATDMF:
		case SIG_SF_FEATB:
			p.starttime = 250;
			break;
		}

		if (tmp->radio) {
			/* XXX Waiting to hear back from Jim if these should be adjustable XXX */
			p.channo = channel;
			p.rxwinktime = 1;
			p.rxflashtime = 1;
			p.starttime = 1;
			p.debouncetime = 5;
		} else {
			p.channo = channel;
			/* Override timing settings based on config file */
			if (conf->timing.prewinktime >= 0)
				p.prewinktime = conf->timing.prewinktime;
			if (conf->timing.preflashtime >= 0)
				p.preflashtime = conf->timing.preflashtime;
			if (conf->timing.winktime >= 0)
				p.winktime = conf->timing.winktime;
			if (conf->timing.flashtime >= 0)
				p.flashtime = conf->timing.flashtime;
			if (conf->timing.starttime >= 0)
				p.starttime = conf->timing.starttime;
			if (conf->timing.rxwinktime >= 0)
				p.rxwinktime = conf->timing.rxwinktime;
			if (conf->timing.rxflashtime >= 0)
				p.rxflashtime = conf->timing.rxflashtime;
			if (conf->timing.debouncetime >= 0)
				p.debouncetime = conf->timing.debouncetime;
		}

		/* don't set parms on a pseudo-channel */
		if (tmp->subs[SUB_REAL].dfd >= 0)
		{
			res = ioctl(tmp->subs[SUB_REAL].dfd, DAHDI_SET_PARAMS, &p);
			if (res < 0) {
				ast_log(LOG_ERROR, "Unable to set parameters: %s\n", strerror(errno));
				destroy_dahdi_pvt(tmp);
				return NULL;
			}
		}
#if 1
		if (!here && (tmp->subs[SUB_REAL].dfd > -1)) {
			memset(&bi, 0, sizeof(bi));
			res = ioctl(tmp->subs[SUB_REAL].dfd, DAHDI_GET_BUFINFO, &bi);
			if (!res) {
				bi.txbufpolicy = conf->chan.buf_policy;
				bi.rxbufpolicy = conf->chan.buf_policy;
				bi.numbufs = conf->chan.buf_no;
				res = ioctl(tmp->subs[SUB_REAL].dfd, DAHDI_SET_BUFINFO, &bi);
				if (res < 0) {
					ast_log(LOG_WARNING, "Unable to set buffer policy on channel %d: %s\n", channel, strerror(errno));
				}
			} else {
				ast_log(LOG_WARNING, "Unable to check buffer policy on channel %d: %s\n", channel, strerror(errno));
			}
			tmp->buf_policy = conf->chan.buf_policy;
			tmp->buf_no = conf->chan.buf_no;
			tmp->usefaxbuffers = conf->chan.usefaxbuffers;
			tmp->faxbuf_policy = conf->chan.faxbuf_policy;
			tmp->faxbuf_no = conf->chan.faxbuf_no;
			/* This is not as gnarly as it may first appear.  If the ioctl above failed, we'd be setting
			 * tmp->bufsize to zero which would cause subsequent faxbuffer-related ioctl calls to fail.
			 * The reason the ioctl call above failed should to be determined before worrying about the
			 * faxbuffer-related ioctl calls */
			tmp->bufsize = bi.bufsize;
		}
#endif
		tmp->immediate = conf->chan.immediate;
		tmp->immediatering = conf->chan.immediatering;
		tmp->transfertobusy = conf->chan.transfertobusy;
		tmp->permdialmode = conf->chan.permdialmode;
		if (chan_sig & __DAHDI_SIG_FXS) {
			tmp->mwimonitor_fsk = conf->chan.mwimonitor_fsk;
			tmp->mwimonitor_neon = conf->chan.mwimonitor_neon;
			tmp->mwimonitor_rpas = conf->chan.mwimonitor_rpas;
		}
		tmp->ringt_base = ringt_base;
		tmp->firstradio = 0;
		if ((chan_sig == SIG_FXOKS) || (chan_sig == SIG_FXOLS) || (chan_sig == SIG_FXOGS))
			tmp->permcallwaiting = conf->chan.callwaiting;
		else
			tmp->permcallwaiting = 0;
		/* Flag to destroy the channel must be cleared on new mkif.  Part of changes for reload to work */
		tmp->destroy = 0;
		tmp->drings = conf->chan.drings;

		/* 10 is a nice default. */
		if (tmp->drings.ringnum[0].range == 0)
			tmp->drings.ringnum[0].range = 10;
		if (tmp->drings.ringnum[1].range == 0)
			tmp->drings.ringnum[1].range = 10;
		if (tmp->drings.ringnum[2].range == 0)
			tmp->drings.ringnum[2].range = 10;

		tmp->usedistinctiveringdetection = usedistinctiveringdetection;
		tmp->callwaitingcallerid = conf->chan.callwaitingcallerid;
		tmp->callwaitingdeluxe = conf->chan.callwaitingdeluxe; /* Not used in DAHDI pvt, only analog pvt */
		tmp->threewaycalling = conf->chan.threewaycalling;
		tmp->threewaysilenthold = conf->chan.threewaysilenthold;
		tmp->calledsubscriberheld = conf->chan.calledsubscriberheld; /* Not used in chan_dahdi.c, just analog pvt, but must exist on the DAHDI pvt anyways */
		tmp->adsi = conf->chan.adsi;
		tmp->use_smdi = conf->chan.use_smdi;
		tmp->permhidecallerid = conf->chan.hidecallerid;
		tmp->hidecalleridname = conf->chan.hidecalleridname;
		tmp->callreturn = conf->chan.callreturn;
		tmp->lastnumredial = conf->chan.lastnumredial; /* Not used in DAHDI pvt, only analog pvt */
		tmp->echocancel = conf->chan.echocancel;
		tmp->echotraining = conf->chan.echotraining;
		tmp->pulse = conf->chan.pulse;
		if (tmp->echocancel.head.tap_length) {
			tmp->echocanbridged = conf->chan.echocanbridged;
		} else {
			if (conf->chan.echocanbridged)
				ast_log(LOG_NOTICE, "echocancelwhenbridged requires echocancel to be enabled; ignoring\n");
			tmp->echocanbridged = 0;
		}
		tmp->busydetect = conf->chan.busydetect;
		tmp->busycount = conf->chan.busycount;
		tmp->busy_cadence = conf->chan.busy_cadence;
		tmp->callprogress = conf->chan.callprogress;
		tmp->waitfordialtone = conf->chan.waitfordialtone;
		tmp->dialtone_detect = conf->chan.dialtone_detect;
		tmp->faxdetect_timeout = conf->chan.faxdetect_timeout;
		tmp->firstdigit_timeout = conf->chan.firstdigit_timeout;
		tmp->interdigit_timeout = conf->chan.interdigit_timeout;
		tmp->matchdigit_timeout = conf->chan.matchdigit_timeout;
		tmp->cancallforward = conf->chan.cancallforward;
		tmp->dtmfrelax = conf->chan.dtmfrelax;
		tmp->callwaiting = tmp->permcallwaiting;
		tmp->hidecallerid = tmp->permhidecallerid;
		tmp->channel = channel;
		tmp->stripmsd = conf->chan.stripmsd;
		tmp->use_callerid = conf->chan.use_callerid;
		tmp->cid_signalling = conf->chan.cid_signalling;
		tmp->cid_start = conf->chan.cid_start;
		tmp->dahditrcallerid = conf->chan.dahditrcallerid;
		tmp->restrictcid = conf->chan.restrictcid;
		tmp->use_callingpres = conf->chan.use_callingpres;
		if (tmp->usedistinctiveringdetection) {
			if (!tmp->use_callerid) {
				ast_log(LOG_NOTICE, "Distinctive Ring detect requires 'usecallerid' be on\n");
				tmp->use_callerid = 1;
			}
		}

		if (tmp->cid_signalling == CID_SIG_SMDI) {
			if (!tmp->use_smdi) {
				ast_log(LOG_WARNING, "SMDI callerid requires SMDI to be enabled, enabling...\n");
				tmp->use_smdi = 1;
			}
		}
		if (tmp->use_smdi) {
			tmp->smdi_iface = ast_smdi_interface_find(conf->smdi_port);
			if (!(tmp->smdi_iface)) {
				ast_log(LOG_ERROR, "Invalid SMDI port specified, disabling SMDI support\n");
				tmp->use_smdi = 0;
			}
		}

		ast_copy_string(tmp->accountcode, conf->chan.accountcode, sizeof(tmp->accountcode));
		tmp->amaflags = conf->chan.amaflags;
		if (!here) {
			tmp->confno = -1;
			tmp->propconfno = -1;
		}
		tmp->canpark = conf->chan.canpark;
		tmp->transfer = conf->chan.transfer;
		ast_copy_string(tmp->defcontext,conf->chan.context,sizeof(tmp->defcontext));
		ast_copy_string(tmp->language, conf->chan.language, sizeof(tmp->language));
		ast_copy_string(tmp->mohinterpret, conf->chan.mohinterpret, sizeof(tmp->mohinterpret));
		ast_copy_string(tmp->mohsuggest, conf->chan.mohsuggest, sizeof(tmp->mohsuggest));
		ast_copy_string(tmp->context, conf->chan.context, sizeof(tmp->context));
		ast_copy_string(tmp->description, conf->chan.description, sizeof(tmp->description));
		ast_copy_string(tmp->parkinglot, conf->chan.parkinglot, sizeof(tmp->parkinglot));
		tmp->cid_ton = 0;
		if (dahdi_analog_lib_handles(tmp->sig, tmp->radio, tmp->oprmode)) {
			ast_copy_string(tmp->cid_num, conf->chan.cid_num, sizeof(tmp->cid_num));
			ast_copy_string(tmp->cid_name, conf->chan.cid_name, sizeof(tmp->cid_name));
		} else {
			tmp->cid_num[0] = '\0';
			tmp->cid_name[0] = '\0';
		}
#if defined(HAVE_PRI)
		if (dahdi_sig_pri_lib_handles(tmp->sig)) {
			tmp->cid_tag[0] = '\0';
		} else
#endif	/* defined(HAVE_PRI) */
		{
			ast_copy_string(tmp->cid_tag, conf->chan.cid_tag, sizeof(tmp->cid_tag));
		}
		tmp->cid_subaddr[0] = '\0';
		ast_copy_string(tmp->mailbox, conf->chan.mailbox, sizeof(tmp->mailbox));
		if (channel != CHAN_PSEUDO && !ast_strlen_zero(tmp->mailbox)) {
			/* This module does not handle MWI in an event-based manner.  However, it
			 * subscribes to MWI for each mailbox that is configured so that the core
			 * knows that we care about it.  Then, chan_dahdi will get the MWI from the
			 * event cache instead of checking the mailbox directly. */
			tmp->mwi_event_sub = ast_mwi_subscribe_pool(tmp->mailbox, stasis_subscription_cb_noop, NULL);
		}
#ifdef HAVE_DAHDI_LINEREVERSE_VMWI
		tmp->mwisend_setting = conf->chan.mwisend_setting;
		tmp->mwisend_fsk  = conf->chan.mwisend_fsk;
		tmp->mwisend_rpas = conf->chan.mwisend_rpas;
#endif

		tmp->group = conf->chan.group;
		tmp->callgroup = conf->chan.callgroup;
		tmp->pickupgroup= conf->chan.pickupgroup;
		ast_unref_namedgroups(tmp->named_callgroups);
		tmp->named_callgroups = ast_ref_namedgroups(conf->chan.named_callgroups);
		ast_unref_namedgroups(tmp->named_pickupgroups);
		tmp->named_pickupgroups = ast_ref_namedgroups(conf->chan.named_pickupgroups);
		if (conf->chan.vars) {
			struct ast_variable *v, *tmpvar;
			for (v = conf->chan.vars ; v ; v = v->next) {
				if ((tmpvar = ast_variable_new(v->name, v->value, v->file))) {
					if (ast_variable_list_replace(&tmp->vars, tmpvar)) {
						tmpvar->next = tmp->vars;
						tmp->vars = tmpvar;
					}
				}
			}
		}
		tmp->hwrxgain_enabled = conf->chan.hwrxgain_enabled;
		tmp->hwtxgain_enabled = conf->chan.hwtxgain_enabled;
		tmp->hwrxgain = conf->chan.hwrxgain;
		tmp->hwtxgain = conf->chan.hwtxgain;
		tmp->cid_rxgain = conf->chan.cid_rxgain;
		tmp->rxgain = conf->chan.rxgain;
		tmp->txgain = conf->chan.txgain;
		tmp->txdrc = conf->chan.txdrc;
		tmp->rxdrc = conf->chan.rxdrc;
		tmp->tonezone = conf->chan.tonezone;
		if (tmp->subs[SUB_REAL].dfd > -1) {
			if (tmp->hwrxgain_enabled) {
				tmp->hwrxgain_enabled = !set_hwgain(tmp->subs[SUB_REAL].dfd, tmp->hwrxgain, 0);
			}
			if (tmp->hwtxgain_enabled) {
				tmp->hwtxgain_enabled = !set_hwgain(tmp->subs[SUB_REAL].dfd, tmp->hwtxgain, 1);
			}
			set_actual_gain(tmp->subs[SUB_REAL].dfd, tmp->rxgain, tmp->txgain, tmp->rxdrc, tmp->txdrc, tmp->law);
			if (tmp->dsp)
				ast_dsp_set_digitmode(tmp->dsp, DSP_DIGITMODE_DTMF | tmp->dtmfrelax);
			dahdi_conf_update(tmp);
			if (!here) {
				switch (chan_sig) {
				case SIG_PRI_LIB_HANDLE_CASES:
				case SIG_SS7:
				case SIG_MFCR2:
					break;
				default:
					/* Hang it up to be sure it's good */
					dahdi_set_hook(tmp->subs[SUB_REAL].dfd, DAHDI_ONHOOK);
					break;
				}
			}
			ioctl(tmp->subs[SUB_REAL].dfd,DAHDI_SETTONEZONE,&tmp->tonezone);
			if ((res = get_alarms(tmp)) != DAHDI_ALARM_NONE) {
				/* the dchannel is down so put the channel in alarm */
				switch (tmp->sig) {
#ifdef HAVE_PRI
				case SIG_PRI_LIB_HANDLE_CASES:
					sig_pri_set_alarm(tmp->sig_pvt, 1);
					break;
#endif
#if defined(HAVE_SS7)
				case SIG_SS7:
					sig_ss7_set_alarm(tmp->sig_pvt, 1);
					break;
#endif	/* defined(HAVE_SS7) */
				default:
					/* The only sig submodule left should be sig_analog. */
					analog_p = tmp->sig_pvt;
					if (analog_p) {
						analog_p->inalarm = 1;
					}
					tmp->inalarm = 1;
					break;
				}
				handle_alarms(tmp, res);
			}
		}

		tmp->polarityonanswerdelay = conf->chan.polarityonanswerdelay;
		tmp->answeronpolarityswitch = conf->chan.answeronpolarityswitch;
		tmp->ani_info_digits = conf->chan.ani_info_digits;
		tmp->ani_wink_time = conf->chan.ani_wink_time;
		tmp->ani_timeout = conf->chan.ani_timeout;
		tmp->hanguponpolarityswitch = conf->chan.hanguponpolarityswitch;
		tmp->reoriginate = conf->chan.reoriginate;
		tmp->sendcalleridafter = conf->chan.sendcalleridafter;
		ast_cc_copy_config_params(tmp->cc_params, conf->chan.cc_params);

		if (!here) {
			tmp->locallyblocked = 0;
			tmp->remotelyblocked = 0;
			switch (tmp->sig) {
#if defined(HAVE_PRI)
			case SIG_PRI_LIB_HANDLE_CASES:
				tmp->inservice = 1;/* Inservice until actually implemented. */
#if defined(HAVE_PRI_SERVICE_MESSAGES)
				((struct sig_pri_chan *) tmp->sig_pvt)->service_status = 0;
				if (chan_sig == SIG_PRI) {
					char db_chan_name[20];
					char db_answer[5];

					/*
					 * Initialize the active out-of-service status
					 * and delete any record if the feature is not enabled.
					 */
					snprintf(db_chan_name, sizeof(db_chan_name), "%s/%d:%d", dahdi_db, tmp->span, tmp->channel);
					if (!ast_db_get(db_chan_name, SRVST_DBKEY, db_answer, sizeof(db_answer))) {
						unsigned *why;

						why = &((struct sig_pri_chan *) tmp->sig_pvt)->service_status;
						if (tmp->pri->enable_service_message_support) {
							char state;

							sscanf(db_answer, "%1c:%30u", &state, why);

							/* Ensure that only the implemented bits could be set.*/
							*why &= (SRVST_NEAREND | SRVST_FAREND);
						}
						if (!*why) {
							ast_db_del(db_chan_name, SRVST_DBKEY);
						}
					}
				}
#endif	/* defined(HAVE_PRI_SERVICE_MESSAGES) */
				break;
#endif	/* defined(HAVE_PRI) */
#if defined(HAVE_SS7)
			case SIG_SS7:
				tmp->inservice = 0;
				if (tmp->ss7->flags & LINKSET_FLAG_INITIALHWBLO) {
					tmp->remotelyblocked |= SS7_BLOCKED_HARDWARE;
				}
				break;
#endif	/* defined(HAVE_SS7) */
			default:
				 /* We default to in service on protocols that don't have a reset */
				tmp->inservice = 1;
				break;
			}
		}

		switch (tmp->sig) {
#if defined(HAVE_PRI)
		case SIG_PRI_LIB_HANDLE_CASES:
			if (pri_chan) {
				pri_chan->channel = tmp->channel;
				pri_chan->hidecallerid = tmp->hidecallerid;
				pri_chan->hidecalleridname = tmp->hidecalleridname;
				pri_chan->immediate = tmp->immediate;
				pri_chan->inalarm = tmp->inalarm;
				pri_chan->priexclusive = tmp->priexclusive;
				pri_chan->priindication_oob = tmp->priindication_oob;
				pri_chan->use_callerid = tmp->use_callerid;
				pri_chan->use_callingpres = tmp->use_callingpres;
				ast_copy_string(pri_chan->context, tmp->context,
					sizeof(pri_chan->context));
				ast_copy_string(pri_chan->mohinterpret, tmp->mohinterpret,
					sizeof(pri_chan->mohinterpret));
				pri_chan->stripmsd = tmp->stripmsd;
			}
			break;
#endif	/* defined(HAVE_PRI) */
#if defined(HAVE_SS7)
		case SIG_SS7:
			if (ss7_chan) {
				ss7_chan->inalarm = tmp->inalarm;
				ss7_chan->inservice = tmp->inservice;

				ss7_chan->stripmsd = tmp->stripmsd;
				ss7_chan->hidecallerid = tmp->hidecallerid;
				ss7_chan->use_callerid = tmp->use_callerid;
				ss7_chan->use_callingpres = tmp->use_callingpres;
				ss7_chan->immediate = tmp->immediate;
				ss7_chan->locallyblocked = tmp->locallyblocked;
				ss7_chan->remotelyblocked = tmp->remotelyblocked;
				ast_copy_string(ss7_chan->context, tmp->context,
					sizeof(ss7_chan->context));
				ast_copy_string(ss7_chan->mohinterpret, tmp->mohinterpret,
					sizeof(ss7_chan->mohinterpret));
			}
			break;
#endif	/* defined(HAVE_SS7) */
		default:
			/* The only sig submodule left should be sig_analog. */
			analog_p = tmp->sig_pvt;
			if (analog_p) {
				analog_p->channel = tmp->channel;
				analog_p->polarityonanswerdelay = conf->chan.polarityonanswerdelay;
				analog_p->answeronpolarityswitch = conf->chan.answeronpolarityswitch;
				analog_p->ani_info_digits = conf->chan.ani_info_digits;
				analog_p->ani_timeout = conf->chan.ani_timeout;
				analog_p->ani_wink_time = conf->chan.ani_wink_time;
				analog_p->hanguponpolarityswitch = conf->chan.hanguponpolarityswitch;
				analog_p->permcallwaiting = conf->chan.callwaiting; /* permcallwaiting possibly modified in analog_config_complete */
				analog_p->calledsubscriberheld = conf->chan.calledsubscriberheld; /* Only actually used in analog pvt, not DAHDI pvt */
				analog_p->callreturn = conf->chan.callreturn;
				analog_p->lastnumredial = conf->chan.lastnumredial; /* Only actually used in analog pvt, not DAHDI pvt */
				analog_p->cancallforward = conf->chan.cancallforward;
				analog_p->canpark = conf->chan.canpark;
				analog_p->dahditrcallerid = conf->chan.dahditrcallerid;
				analog_p->immediate = conf->chan.immediate;
				analog_p->immediatering = conf->chan.immediatering;
				analog_p->permhidecallerid = conf->chan.hidecallerid; /* hidecallerid is the config setting, not permhidecallerid (~permcallwaiting above) */
				/* It's not necessary to set analog_p->hidecallerid here, sig_analog will set hidecallerid=permhidecaller before each call */
				analog_p->pulse = conf->chan.pulse;
				analog_p->threewaycalling = conf->chan.threewaycalling;
				analog_p->transfer = conf->chan.transfer;
				analog_p->transfertobusy = conf->chan.transfertobusy;
				analog_p->permdialmode = conf->chan.permdialmode;
				analog_p->use_callerid = tmp->use_callerid;
				analog_p->usedistinctiveringdetection = tmp->usedistinctiveringdetection;
				analog_p->use_smdi = tmp->use_smdi;
				analog_p->smdi_iface = tmp->smdi_iface;
				analog_p->outsigmod = ANALOG_SIG_NONE;
				analog_p->echotraining = conf->chan.echotraining;
				analog_p->cid_signalling = conf->chan.cid_signalling;
				analog_p->stripmsd = conf->chan.stripmsd;
				switch (conf->chan.cid_start) {
				case CID_START_POLARITY:
					analog_p->cid_start = ANALOG_CID_START_POLARITY;
					break;
				case CID_START_POLARITY_IN:
					analog_p->cid_start = ANALOG_CID_START_POLARITY_IN;
					break;
				case CID_START_DTMF_NOALERT:
					analog_p->cid_start = ANALOG_CID_START_DTMF_NOALERT;
					break;
				default:
					analog_p->cid_start = ANALOG_CID_START_RING;
					break;
				}
				analog_p->callwaitingcallerid = conf->chan.callwaitingcallerid;
				analog_p->callwaitingdeluxe = conf->chan.callwaitingdeluxe;
				analog_p->ringt = conf->chan.ringt;
				analog_p->ringt_base = ringt_base;
				analog_p->onhooktime = time(NULL);
				if (chan_sig & __DAHDI_SIG_FXO) {
					memset(&p, 0, sizeof(p));
					res = ioctl(tmp->subs[SUB_REAL].dfd, DAHDI_GET_PARAMS, &p);
					if (!res) {
						analog_p->fxsoffhookstate = p.rxisoffhook;
					}
#ifdef HAVE_DAHDI_LINEREVERSE_VMWI
					res = ioctl(tmp->subs[SUB_REAL].dfd, DAHDI_VMWI_CONFIG, &tmp->mwisend_setting);
#endif
				}
				analog_p->msgstate = -1;

				ast_copy_string(analog_p->mohsuggest, conf->chan.mohsuggest, sizeof(analog_p->mohsuggest));
				ast_copy_string(analog_p->cid_num, conf->chan.cid_num, sizeof(analog_p->cid_num));
				ast_copy_string(analog_p->cid_name, conf->chan.cid_name, sizeof(analog_p->cid_name));

				analog_config_complete(analog_p);
			}
			break;
		}
#if defined(HAVE_PRI)
		if (tmp->channel == CHAN_PSEUDO) {
			/*
			 * Save off pseudo channel buffer policy values for dynamic creation of
			 * no B channel interfaces.
			 */
			dahdi_pseudo_parms.buf_no = tmp->buf_no;
			dahdi_pseudo_parms.buf_policy = tmp->buf_policy;
			dahdi_pseudo_parms.faxbuf_no = tmp->faxbuf_no;
			dahdi_pseudo_parms.faxbuf_policy = tmp->faxbuf_policy;
		}
#endif	/* defined(HAVE_PRI) */
	}
	if (tmp && !here) {
		/* Add the new channel interface to the sorted channel interface list. */
		dahdi_iflist_insert(tmp);
	}
	return tmp;
}

static int is_group_or_channel_match(struct dahdi_pvt *p, int span, ast_group_t groupmatch, int *groupmatched, int channelmatch, int *channelmatched)
{
#if defined(HAVE_PRI)
	if (0 < span) {
		/* The channel must be on the specified PRI span. */
		if (!p->pri || p->pri->span != span) {
			return 0;
		}
		if (!groupmatch && channelmatch == -1) {
			/* Match any group since it only needs to be on the PRI span. */
			*groupmatched = 1;
			return 1;
		}
	}
#endif	/* defined(HAVE_PRI) */
	/* check group matching */
	if (groupmatch) {
		if ((p->group & groupmatch) != groupmatch)
			/* Doesn't match the specified group, try the next one */
			return 0;
		*groupmatched = 1;
	}
	/* Check to see if we have a channel match */
	if (channelmatch != -1) {
		if (p->channel != channelmatch)
			/* Doesn't match the specified channel, try the next one */
			return 0;
		*channelmatched = 1;
	}

	return 1;
}

static int available(struct dahdi_pvt **pvt, int is_specific_channel)
{
	struct dahdi_pvt *p = *pvt;

	if (p->inalarm)
		return 0;

	if (dahdi_analog_lib_handles(p->sig, p->radio, p->oprmode))
		return analog_available(p->sig_pvt);

	switch (p->sig) {
#if defined(HAVE_PRI)
	case SIG_PRI_LIB_HANDLE_CASES:
		{
			struct sig_pri_chan *pvt_chan;
			int res;

			pvt_chan = p->sig_pvt;
			res = sig_pri_available(&pvt_chan, is_specific_channel);
			*pvt = pvt_chan->chan_pvt;
			return res;
		}
#endif	/* defined(HAVE_PRI) */
#if defined(HAVE_SS7)
	case SIG_SS7:
		return sig_ss7_available(p->sig_pvt);
#endif	/* defined(HAVE_SS7) */
	default:
		break;
	}

	if (p->locallyblocked || p->remotelyblocked) {
		return 0;
	}

	/* If no owner definitely available */
	if (!p->owner) {
#ifdef HAVE_OPENR2
		/* Trust MFC/R2 */
		if (p->mfcr2) {
			if (p->mfcr2call) {
				return 0;
			} else {
				return 1;
			}
		}
#endif
		return 1;
	}

	return 0;
}

#if defined(HAVE_PRI)
#if defined(HAVE_PRI_CALL_WAITING)
/*!
 * \internal
 * \brief Init the private channel configuration using the span controller.
 * \since 1.8
 *
 * \param priv Channel to init the configuration.
 * \param pri sig_pri PRI control structure.
 *
 * \note Assumes the pri->lock is already obtained.
 */
static void my_pri_init_config(void *priv, struct sig_pri_span *pri)
{
	struct dahdi_pvt *pvt = priv;

	pvt->stripmsd = pri->ch_cfg.stripmsd;
	pvt->hidecallerid = pri->ch_cfg.hidecallerid;
	pvt->hidecalleridname = pri->ch_cfg.hidecalleridname;
	pvt->immediate = pri->ch_cfg.immediate;
	pvt->priexclusive = pri->ch_cfg.priexclusive;
	pvt->priindication_oob = pri->ch_cfg.priindication_oob;
	pvt->use_callerid = pri->ch_cfg.use_callerid;
	pvt->use_callingpres = pri->ch_cfg.use_callingpres;
	ast_copy_string(pvt->context, pri->ch_cfg.context, sizeof(pvt->context));
	ast_copy_string(pvt->mohinterpret, pri->ch_cfg.mohinterpret, sizeof(pvt->mohinterpret));
}
#endif	/* defined(HAVE_PRI_CALL_WAITING) */
#endif	/* defined(HAVE_PRI) */

#if defined(HAVE_PRI)
/*!
 * \internal
 * \brief Create a no B channel interface.
 * \since 1.8
 *
 * \param pri sig_pri span controller to add interface.
 *
 * \note Assumes the pri->lock is already obtained.
 *
 * \retval array-index into private pointer array on success.
 * \retval -1 on error.
 */
static int dahdi_new_pri_nobch_channel(struct sig_pri_span *pri)
{
	int pvt_idx;
	int res;
	unsigned idx;
	struct dahdi_pvt *pvt;
	struct sig_pri_chan *chan;
	struct dahdi_bufferinfo bi;

	static int nobch_channel = CHAN_PSEUDO;

	/* Find spot in the private pointer array for new interface. */
	for (pvt_idx = 0; pvt_idx < pri->numchans; ++pvt_idx) {
		if (!pri->pvts[pvt_idx]) {
			break;
		}
	}
	if (pri->numchans == pvt_idx) {
		if (ARRAY_LEN(pri->pvts) <= pvt_idx) {
			ast_log(LOG_ERROR, "Unable to add a no-B-channel interface!\n");
			return -1;
		}

		/* Add new spot to the private pointer array. */
		pri->pvts[pvt_idx] = NULL;
		++pri->numchans;
	}

	pvt = ast_calloc(1, sizeof(*pvt));
	if (!pvt) {
		return -1;
	}
	pvt->cc_params = ast_cc_config_params_init();
	if (!pvt->cc_params) {
		ast_free(pvt);
		return -1;
	}
	ast_mutex_init(&pvt->lock);
	for (idx = 0; idx < ARRAY_LEN(pvt->subs); ++idx) {
		pvt->subs[idx].dfd = -1;
	}
	pvt->buf_no = dahdi_pseudo_parms.buf_no;
	pvt->buf_policy = dahdi_pseudo_parms.buf_policy;
	pvt->faxbuf_no = dahdi_pseudo_parms.faxbuf_no;
	pvt->faxbuf_policy = dahdi_pseudo_parms.faxbuf_policy;

	chan = sig_pri_chan_new(pvt, pri, 0, 0, 0);
	if (!chan) {
		destroy_dahdi_pvt(pvt);
		return -1;
	}
	chan->no_b_channel = 1;

	/*
	 * Pseudo channel companding law.
	 * Needed for outgoing call waiting calls.
	 * XXX May need to make this determined by switchtype or user option.
	 */
	pvt->law_default = DAHDI_LAW_ALAW;

	pvt->sig = pri->sig;
	pvt->outsigmod = -1;
	pvt->pri = pri;
	pvt->sig_pvt = chan;
	pri->pvts[pvt_idx] = chan;

	pvt->subs[SUB_REAL].dfd = dahdi_open("/dev/dahdi/pseudo");
	if (pvt->subs[SUB_REAL].dfd < 0) {
		ast_log(LOG_ERROR, "Unable to open no B channel interface pseudo channel: %s\n",
			strerror(errno));
		destroy_dahdi_pvt(pvt);
		return -1;
	}
	memset(&bi, 0, sizeof(bi));
	res = ioctl(pvt->subs[SUB_REAL].dfd, DAHDI_GET_BUFINFO, &bi);
	if (!res) {
		pvt->bufsize = bi.bufsize;
		bi.txbufpolicy = pvt->buf_policy;
		bi.rxbufpolicy = pvt->buf_policy;
		bi.numbufs = pvt->buf_no;
		res = ioctl(pvt->subs[SUB_REAL].dfd, DAHDI_SET_BUFINFO, &bi);
		if (res < 0) {
			ast_log(LOG_WARNING,
				"Unable to set buffer policy on no B channel interface: %s\n",
				strerror(errno));
		}
	} else
		ast_log(LOG_WARNING,
			"Unable to check buffer policy on no B channel interface: %s\n",
			strerror(errno));

	--nobch_channel;
	if (CHAN_PSEUDO < nobch_channel) {
		nobch_channel = CHAN_PSEUDO - 1;
	}
	pvt->channel = nobch_channel;
	pvt->span = pri->span;
	chan->channel = pvt->channel;

	dahdi_nobch_insert(pri, pvt);

	return pvt_idx;
}
#endif	/* defined(HAVE_PRI) */

/* This function can *ONLY* be used for copying pseudo (CHAN_PSEUDO) private
   structures; it makes no attempt to safely copy regular channel private
   structures that might contain reference-counted object pointers and other
   scary bits
*/
static struct dahdi_pvt *duplicate_pseudo(struct dahdi_pvt *src)
{
	struct dahdi_pvt *p;
	struct dahdi_bufferinfo bi;
	int res;

	p = ast_malloc(sizeof(*p));
	if (!p) {
		return NULL;
	}
	*p = *src;

	/* Must deep copy the cc_params. */
	p->cc_params = ast_cc_config_params_init();
	if (!p->cc_params) {
		ast_free(p);
		return NULL;
	}
	ast_cc_copy_config_params(p->cc_params, src->cc_params);

	p->which_iflist = DAHDI_IFLIST_NONE;
	p->next = NULL;
	p->prev = NULL;
	ast_mutex_init(&p->lock);
	p->subs[SUB_REAL].dfd = dahdi_open("/dev/dahdi/pseudo");
	if (p->subs[SUB_REAL].dfd < 0) {
		ast_log(LOG_ERROR, "Unable to dup channel: %s\n", strerror(errno));
		destroy_dahdi_pvt(p);
		return NULL;
	}
	res = ioctl(p->subs[SUB_REAL].dfd, DAHDI_GET_BUFINFO, &bi);
	if (!res) {
		bi.txbufpolicy = src->buf_policy;
		bi.rxbufpolicy = src->buf_policy;
		bi.numbufs = src->buf_no;
		res = ioctl(p->subs[SUB_REAL].dfd, DAHDI_SET_BUFINFO, &bi);
		if (res < 0) {
			ast_log(LOG_WARNING, "Unable to set buffer policy on dup channel: %s\n", strerror(errno));
		}
	} else
		ast_log(LOG_WARNING, "Unable to check buffer policy on dup channel: %s\n", strerror(errno));
	p->destroy = 1;
	dahdi_iflist_insert(p);
	return p;
}

struct dahdi_starting_point {
	/*! Group matching mask.  Zero if not specified. */
	ast_group_t groupmatch;
	/*! DAHDI channel to match with.  -1 if not specified. */
	int channelmatch;
	/*! Round robin saved search location index. (Valid if roundrobin TRUE) */
	int rr_starting_point;
	/*! ISDN span where channels can be picked (Zero if not specified) */
	int span;
	/*! Analog channel distinctive ring cadence index. */
	int cadence;
	/*! Dialing option. c/r/d if present and valid. */
	char opt;
	/*! TRUE if to search the channel list backwards. */
	char backwards;
	/*! TRUE if search is done with round robin sequence. */
	char roundrobin;
};
static struct dahdi_pvt *determine_starting_point(const char *data, struct dahdi_starting_point *param)
{
	char *dest;
	char *s;
	int x;
	int res = 0;
	struct dahdi_pvt *p;
	char *subdir = NULL;
	AST_DECLARE_APP_ARGS(args,
		AST_APP_ARG(group);	/* channel/group token */
		//AST_APP_ARG(ext);	/* extension token */
		//AST_APP_ARG(opts);	/* options token */
		AST_APP_ARG(other);	/* Any remaining unused arguments */
	);

	/*
	 * data is ---v
	 * Dial(DAHDI/pseudo[/extension[/options]])
	 * Dial(DAHDI/<channel#>[c|r<cadence#>|d][/extension[/options]])
	 * Dial(DAHDI/<subdir>!<channel#>[c|r<cadence#>|d][/extension[/options]])
	 * Dial(DAHDI/i<span>[/extension[/options]])
	 * Dial(DAHDI/[i<span>-](g|G|r|R)<group#(0-63)>[c|r<cadence#>|d][/extension[/options]])
	 *
	 * i - ISDN span channel restriction.
	 *     Used by CC to ensure that the CC recall goes out the same span.
	 *     Also to make ISDN channel names dialable when the sequence number
	 *     is stripped off.  (Used by DTMF attended transfer feature.)
	 *
	 * g - channel group allocation search forward
	 * G - channel group allocation search backward
	 * r - channel group allocation round robin search forward
	 * R - channel group allocation round robin search backward
	 *
	 * c - Wait for DTMF digit to confirm answer
	 * r<cadence#> - Set distinctive ring cadence number
	 * d - Force bearer capability for ISDN/SS7 call to digital.
	 */

	if (data) {
		dest = ast_strdupa(data);
	} else {
		ast_log(LOG_WARNING, "Channel requested with no data\n");
		return NULL;
	}
	AST_NONSTANDARD_APP_ARGS(args, dest, '/');
	if (!args.argc || ast_strlen_zero(args.group)) {
		ast_log(LOG_WARNING, "No channel/group specified\n");
		return NULL;
	}

	/* Initialize the output parameters */
	memset(param, 0, sizeof(*param));
	param->channelmatch = -1;

	if (strchr(args.group, '!') != NULL) {
		char *prev = args.group;
		while ((s = strchr(prev, '!')) != NULL) {
			*s++ = '/';
			prev = s;
		}
		*(prev - 1) = '\0';
		subdir = args.group;
		args.group = prev;
	} else if (args.group[0] == 'i') {
		/* Extract the ISDN span channel restriction specifier. */
		res = sscanf(args.group + 1, "%30d", &x);
		if (res < 1) {
			ast_log(LOG_WARNING, "Unable to determine ISDN span for data %s\n", data);
			return NULL;
		}
		param->span = x;

		/* Remove the ISDN span channel restriction specifier. */
		s = strchr(args.group, '-');
		if (!s) {
			/* Search all groups since we are ISDN span restricted. */
			return iflist;
		}
		args.group = s + 1;
		res = 0;
	}
	if (toupper(args.group[0]) == 'G' || toupper(args.group[0])=='R') {
		/* Retrieve the group number */
		s = args.group + 1;
		res = sscanf(s, "%30d%1c%30d", &x, &param->opt, &param->cadence);
		if (res < 1) {
			ast_log(LOG_WARNING, "Unable to determine group for data %s\n", data);
			return NULL;
		}
		param->groupmatch = ((ast_group_t) 1 << x);

		if (toupper(args.group[0]) == 'G') {
			if (args.group[0] == 'G') {
				param->backwards = 1;
				p = ifend;
			} else
				p = iflist;
		} else {
			if (ARRAY_LEN(round_robin) <= x) {
				ast_log(LOG_WARNING, "Round robin index %d out of range for data %s\n",
					x, data);
				return NULL;
			}
			if (args.group[0] == 'R') {
				param->backwards = 1;
				p = round_robin[x] ? round_robin[x]->prev : ifend;
				if (!p)
					p = ifend;
			} else {
				p = round_robin[x] ? round_robin[x]->next : iflist;
				if (!p)
					p = iflist;
			}
			param->roundrobin = 1;
			param->rr_starting_point = x;
		}
	} else {
		s = args.group;
		if (!strcasecmp(s, "pseudo")) {
			/* Special case for pseudo */
			x = CHAN_PSEUDO;
			param->channelmatch = x;
		} else {
			res = sscanf(s, "%30d%1c%30d", &x, &param->opt, &param->cadence);
			if (res < 1) {
				ast_log(LOG_WARNING, "Unable to determine channel for data %s\n", data);
				return NULL;
			} else {
				param->channelmatch = x;
			}
		}
		if (subdir) {
			char path[PATH_MAX];
			struct stat stbuf;

			snprintf(path, sizeof(path), "/dev/dahdi/%s/%d",
					subdir, param->channelmatch);
			if (stat(path, &stbuf) < 0) {
				ast_log(LOG_WARNING, "stat(%s) failed: %s\n",
						path, strerror(errno));
				return NULL;
			}
			if (!S_ISCHR(stbuf.st_mode)) {
				ast_log(LOG_ERROR, "%s: Not a character device file\n",
						path);
				return NULL;
			}
			param->channelmatch = minor(stbuf.st_rdev);
		}

		p = iflist;
	}

	if (param->opt == 'r' && res < 3) {
		ast_log(LOG_WARNING, "Distinctive ring missing identifier in '%s'\n", data);
		param->opt = '\0';
	}

	return p;
}

static struct ast_channel *dahdi_request(const char *type, struct ast_format_cap *cap,
	const struct ast_assigned_ids *assignedids, const struct ast_channel *requestor,
	const char *data, int *cause)
{
	int callwait = 0;
	struct dahdi_pvt *p;
	struct ast_channel *tmp = NULL;
	struct dahdi_pvt *exitpvt;
	int channelmatched = 0;
	int foundowner = 0;
	int groupmatched = 0;
#if defined(HAVE_PRI) || defined(HAVE_SS7)
	int transcapdigital = 0;
#endif	/* defined(HAVE_PRI) || defined(HAVE_SS7) */
	struct dahdi_starting_point start;
	ast_callid callid = 0;
	int callid_created = ast_callid_threadstorage_auto(&callid);

	ast_mutex_lock(&iflock);
	p = determine_starting_point(data, &start);
	if (!p) {
		/* We couldn't determine a starting point, which likely means badly-formatted channel name. Abort! */
		ast_mutex_unlock(&iflock);
		ast_callid_threadstorage_auto_clean(callid, callid_created);
		return NULL;
	}

	/* Search for an unowned channel */
	exitpvt = p;
	while (p && !tmp) {
		if (start.roundrobin)
			round_robin[start.rr_starting_point] = p;

		if (p->owner) {
			foundowner++;
		}

		if (is_group_or_channel_match(p, start.span, start.groupmatch, &groupmatched, start.channelmatch, &channelmatched)
			&& available(&p, channelmatched)) {
			ast_debug(1, "Using channel %d\n", p->channel);

			callwait = (p->owner != NULL);
#ifdef HAVE_OPENR2
			if (p->mfcr2) {
				ast_mutex_lock(&p->lock);
				if (p->mfcr2call) {
					ast_mutex_unlock(&p->lock);
					ast_debug(1, "Yay!, someone just beat us in the race for channel %d.\n", p->channel);
					goto next;
				}
				p->mfcr2call = 1;
				ast_mutex_unlock(&p->lock);
			}
#endif
			if (p->channel == CHAN_PSEUDO) {
				p = duplicate_pseudo(p);
				if (!p) {
					break;
				}
			}

			p->distinctivering = 0;
			/* Make special notes */
			switch (start.opt) {
			case '\0':
				/* No option present. */
				break;
			case 'c':
				/* Confirm answer */
				p->confirmanswer = 1;
				break;
			case 'r':
				/* Distinctive ring */
				p->distinctivering = start.cadence;
				break;
			case 'd':
#if defined(HAVE_PRI) || defined(HAVE_SS7)
				/* If this is an ISDN call, make it digital */
				transcapdigital = AST_TRANS_CAP_DIGITAL;
#endif	/* defined(HAVE_PRI) || defined(HAVE_SS7) */
				break;
			default:
				ast_log(LOG_WARNING, "Unknown option '%c' in '%s'\n", start.opt, data);
				break;
			}

			p->outgoing = 1;
			if (dahdi_analog_lib_handles(p->sig, p->radio, p->oprmode)) {
				tmp = analog_request(p->sig_pvt, &callwait, requestor);
#ifdef HAVE_PRI
			} else if (dahdi_sig_pri_lib_handles(p->sig)) {
				/*
				 * We already have the B channel reserved for this call.  We
				 * just need to make sure that dahdi_hangup() has completed
				 * cleaning up before continuing.
				 */
				ast_mutex_lock(&p->lock);
				ast_mutex_unlock(&p->lock);

				sig_pri_extract_called_num_subaddr(p->sig_pvt, data, p->dnid,
					sizeof(p->dnid));
				tmp = sig_pri_request(p->sig_pvt, SIG_PRI_DEFLAW, assignedids, requestor, transcapdigital);
#endif
#if defined(HAVE_SS7)
			} else if (p->sig == SIG_SS7) {
				tmp = sig_ss7_request(p->sig_pvt, SIG_SS7_DEFLAW, assignedids, requestor, transcapdigital);
#endif	/* defined(HAVE_SS7) */
			} else {
				tmp = dahdi_new(p, AST_STATE_RESERVED, 0, p->owner ? SUB_CALLWAIT : SUB_REAL, 0, assignedids, requestor, callid);
			}
			if (!tmp) {
				p->outgoing = 0;
#if defined(HAVE_PRI)
				switch (p->sig) {
				case SIG_PRI_LIB_HANDLE_CASES:
#if defined(HAVE_PRI_CALL_WAITING)
					if (((struct sig_pri_chan *) p->sig_pvt)->is_call_waiting) {
						((struct sig_pri_chan *) p->sig_pvt)->is_call_waiting = 0;
						ast_atomic_fetchadd_int(&p->pri->num_call_waiting_calls, -1);
					}
#endif	/* defined(HAVE_PRI_CALL_WAITING) */
					/*
					 * This should be the last thing to clear when we are done with
					 * the channel.
					 */
					((struct sig_pri_chan *) p->sig_pvt)->allocated = 0;
					break;
				default:
					break;
				}
#endif	/* defined(HAVE_PRI) */
			} else {
				snprintf(p->dialstring, sizeof(p->dialstring), "DAHDI/%s", data);
			}
			break;
		}
#ifdef HAVE_OPENR2
next:
#endif
		if (start.backwards) {
			p = p->prev;
			if (!p)
				p = ifend;
		} else {
			p = p->next;
			if (!p)
				p = iflist;
		}
		/* stop when you roll to the one that we started from */
		if (p == exitpvt)
			break;
	}
	ast_mutex_unlock(&iflock);
	restart_monitor();
	if (cause && !tmp) {
		if (callwait || (channelmatched && foundowner)) {
			*cause = AST_CAUSE_BUSY;
		} else if (groupmatched) {
			*cause = AST_CAUSE_CONGESTION;
		} else {
			/*
			 * We did not match any channel requested.
			 * Dialplan error requesting non-existant channel?
			 */
		}
	}

	ast_callid_threadstorage_auto_clean(callid, callid_created);
	return tmp;
}

/*!
 * \internal
 * \brief Determine the device state for a given DAHDI device if we can.
 * \since 1.8
 *
 * \param data DAHDI device name after "DAHDI/".
 *
 * \retval device_state enum ast_device_state value.
 * \retval AST_DEVICE_UNKNOWN if we could not determine the device's state.
 */
static int dahdi_devicestate(const char *data)
{
#if defined(HAVE_PRI)
	const char *device;
	unsigned span;
	int res;

	device = data;

	if (*device != 'I') {
		/* The request is not for an ISDN span device. */
		return AST_DEVICE_UNKNOWN;
	}
	res = sscanf(device, "I%30u", &span);
	if (res != 1 || !span || NUM_SPANS < span) {
		/* Bad format for ISDN span device name. */
		return AST_DEVICE_UNKNOWN;
	}
	device = strchr(device, '/');
	if (!device) {
		/* Bad format for ISDN span device name. */
		return AST_DEVICE_UNKNOWN;
	}

	/*
	 * Since there are currently no other span devstate's defined,
	 * it must be congestion.
	 */
#if defined(THRESHOLD_DEVSTATE_PLACEHOLDER)
	++device;
	if (!strcmp(device, "congestion"))
#endif	/* defined(THRESHOLD_DEVSTATE_PLACEHOLDER) */
	{
		return pris[span - 1].pri.congestion_devstate;
	}
#if defined(THRESHOLD_DEVSTATE_PLACEHOLDER)
	else if (!strcmp(device, "threshold")) {
		return pris[span - 1].pri.threshold_devstate;
	}
	return AST_DEVICE_UNKNOWN;
#endif	/* defined(THRESHOLD_DEVSTATE_PLACEHOLDER) */
#else
	return AST_DEVICE_UNKNOWN;
#endif	/* defined(HAVE_PRI) */
}

/*!
 * \brief Callback made when dial failed to get a channel out of dahdi_request().
 * \since 1.8
 *
 * \param inbound Incoming asterisk channel.
 * \param dest Same dial string passed to dahdi_request().
 * \param callback Callback into CC core to announce a busy channel available for CC.
 *
 * \details
 * This callback acts like a forked dial with all prongs of the fork busy.
 * Essentially, for each channel that could have taken the call, indicate that
 * it is busy.
 *
 * \retval 0 on success.
 * \retval -1 on error.
 */
static int dahdi_cc_callback(struct ast_channel *inbound, const char *dest, ast_cc_callback_fn callback)
{
	struct dahdi_pvt *p;
	struct dahdi_pvt *exitpvt;
	struct dahdi_starting_point start;
	int groupmatched = 0;
	int channelmatched = 0;

	ast_mutex_lock(&iflock);
	p = determine_starting_point(dest, &start);
	if (!p) {
		ast_mutex_unlock(&iflock);
		return -1;
	}
	exitpvt = p;
	for (;;) {
		if (is_group_or_channel_match(p, start.span, start.groupmatch, &groupmatched, start.channelmatch, &channelmatched)) {
			/* We found a potential match. call the callback */
			struct ast_str *device_name;
			char *dash;
			const char *monitor_type;
			char dialstring[AST_CHANNEL_NAME];
			char full_device_name[AST_CHANNEL_NAME];

			switch (ast_get_cc_monitor_policy(p->cc_params)) {
			case AST_CC_MONITOR_NEVER:
				break;
			case AST_CC_MONITOR_NATIVE:
			case AST_CC_MONITOR_ALWAYS:
			case AST_CC_MONITOR_GENERIC:
#if defined(HAVE_PRI)
				if (dahdi_sig_pri_lib_handles(p->sig)) {
					/*
					 * ISDN is in a trunk busy condition so we need to monitor
					 * the span congestion device state.
					 */
					snprintf(full_device_name, sizeof(full_device_name),
						"DAHDI/I%d/congestion", p->pri->span);
				} else
#endif	/* defined(HAVE_PRI) */
				{
#if defined(HAVE_PRI)
					device_name = create_channel_name(p, 1, "");
#else
					device_name = create_channel_name(p);
#endif	/* defined(HAVE_PRI) */
					snprintf(full_device_name, sizeof(full_device_name), "DAHDI/%s",
						device_name ? ast_str_buffer(device_name) : "");
					ast_free(device_name);
					/*
					 * The portion after the '-' in the channel name is either a random
					 * number, a sequence number, or a subchannel number. None are
					 * necessary so strip them off.
					 */
					dash = strrchr(full_device_name, '-');
					if (dash) {
						*dash = '\0';
					}
				}
				snprintf(dialstring, sizeof(dialstring), "DAHDI/%s", dest);

				/*
				 * Analog can only do generic monitoring.
				 * ISDN is in a trunk busy condition and any "device" is going
				 * to be busy until a B channel becomes available.  The generic
				 * monitor can do this task.
				 */
				monitor_type = AST_CC_GENERIC_MONITOR_TYPE;
				callback(inbound,
#if defined(HAVE_PRI)
					p->pri ? p->pri->cc_params : p->cc_params,
#else
					p->cc_params,
#endif	/* defined(HAVE_PRI) */
					monitor_type, full_device_name, dialstring, NULL);
				break;
			}
		}
		p = start.backwards ? p->prev : p->next;
		if (!p) {
			p = start.backwards ? ifend : iflist;
		}
		if (p == exitpvt) {
			break;
		}
	}
	ast_mutex_unlock(&iflock);
	return 0;
}

#if defined(HAVE_SS7)
static void dahdi_ss7_message(struct ss7 *ss7, char *s)
{
	int i;

	if (ss7) {
		for (i = 0; i < NUM_SPANS; i++) {
			if (linksets[i].ss7.ss7 == ss7) {
				ast_verbose_callid(0, "[%d] %s", i + 1, s);
				return;
			}
		}
	}
	ast_verbose_callid(0, "%s", s);
}
#endif	/* defined(HAVE_SS7) */

#if defined(HAVE_SS7)
static void dahdi_ss7_error(struct ss7 *ss7, char *s)
{
	int i;

	if (ss7) {
		for (i = 0; i < NUM_SPANS; i++) {
			if (linksets[i].ss7.ss7 == ss7) {
				ast_log_callid(LOG_ERROR, 0, "[%d] %s", i + 1, s);
				return;
			}
		}
	}
	ast_log_callid(LOG_ERROR, 0, "%s", s);
}
#endif	/* defined(HAVE_SS7) */

#if defined(HAVE_OPENR2)
static void *mfcr2_monitor(void *data)
{
	struct dahdi_mfcr2 *mfcr2 = data;
	struct dahdi_pvt *pvt;

	/* we should be using pthread_key_create
	   and allocate pollers dynamically.
	   I think do_monitor() could be leaking, since it
	   could be cancelled at any time and is not
	   using thread keys, why?, */
	struct pollfd pollers[ARRAY_LEN(mfcr2->pvts)];
	int res = 0;
	int i = 0;
	int oldstate = 0;
	int quit_loop = 0;
	int maxsleep = 20;
	int was_idle = 0;
	int pollsize = 0;
	/* now that we're ready to get calls, unblock our side and
	   get current line state */
	for (i = 0; i < mfcr2->numchans; i++) {
		pvt = mfcr2->pvts[i];
		if (!pvt) {
			continue;
		}
		openr2_chan_set_idle(pvt->r2chan);
		openr2_chan_handle_cas(pvt->r2chan);
	}
	while (1) {
		/* we trust here that the mfcr2 channel list will not ever change once
		   the module is loaded */
		pollsize = 0;
		for (i = 0; i < mfcr2->numchans; i++) {
			pollers[i].revents = 0;
			pollers[i].events = 0;
			pvt = mfcr2->pvts[i];
			if (!pvt) {
				continue;
			}
			if (pvt->owner) {
				continue;
			}
			if (mfcr2->nodev) {
				continue;
			}
			if (!pvt->r2chan) {
				ast_debug(1, "Wow, no r2chan on channel %d\n", pvt->channel);
				quit_loop = 1;
				break;
			}
			openr2_chan_enable_read(pvt->r2chan);
			pollers[i].events = POLLIN | POLLPRI;
			pollers[i].fd = pvt->subs[SUB_REAL].dfd;
			pollsize++;
		}
		if (quit_loop) {
			break;
		}
		if (pollsize == 0) {
			if (!was_idle) {
				ast_debug(1, "Monitor thread going idle since everybody has an owner\n");
				was_idle = 1;
			}
			poll(NULL, 0, maxsleep);
			continue;
		}
		was_idle = 0;
		/* probably poll() is a valid cancel point, lets just be on the safe side
		   by calling pthread_testcancel */
		pthread_testcancel();
		res = poll(pollers, mfcr2->numchans, maxsleep);
		pthread_testcancel();
		if ((res < 0) && (errno != EINTR)) {
			ast_log(LOG_ERROR, "going out, poll failed: %s\n", strerror(errno));
			break;
		}
		/* do we want to allow to cancel while processing events? */
		pthread_setcancelstate(PTHREAD_CANCEL_DISABLE, &oldstate);
		for (i = 0; i < mfcr2->numchans; i++) {
			pvt = mfcr2->pvts[i];
			if (!pvt) {
				continue;
			}
			if (pollers[i].revents & POLLPRI || pollers[i].revents & POLLIN) {
				openr2_chan_process_event(pvt->r2chan);
			}
		}
		pthread_setcancelstate(PTHREAD_CANCEL_ENABLE, &oldstate);
	}
	ast_log(LOG_NOTICE, "Quitting MFC/R2 monitor thread\n");
	return 0;
}
#endif /* HAVE_OPENR2 */

#if defined(HAVE_PRI)
static void dahdi_pri_message(struct pri *pri, char *s)
{
	int x;
	int y;
	int dchan = -1;
	int span = -1;
	int dchancount = 0;

	if (pri) {
		for (x = 0; x < NUM_SPANS; x++) {
			for (y = 0; y < SIG_PRI_NUM_DCHANS; y++) {
				if (pris[x].pri.dchans[y]) {
					dchancount++;
				}

				if (pris[x].pri.dchans[y] == pri) {
					dchan = y;
				}
			}
			if (dchan >= 0) {
				span = x;
				break;
			}
			dchancount = 0;
		}
		if (-1 < span) {
			if (1 < dchancount) {
				ast_verbose_callid(0, "[PRI Span: %d D-Channel: %d] %s", span + 1, dchan, s);
			} else {
				ast_verbose_callid(0, "PRI Span: %d %s", span + 1, s);
			}
		} else {
			ast_verbose_callid(0, "PRI Span: ? %s", s);
		}
	} else {
		ast_verbose_callid(0, "PRI Span: ? %s", s);
	}

	ast_mutex_lock(&pridebugfdlock);

	if (pridebugfd >= 0) {
		if (write(pridebugfd, s, strlen(s)) < 0) {
			ast_log_callid(LOG_WARNING, 0, "write() failed: %s\n", strerror(errno));
		}
	}

	ast_mutex_unlock(&pridebugfdlock);
}
#endif	/* defined(HAVE_PRI) */

#if defined(HAVE_PRI)
static void dahdi_pri_error(struct pri *pri, char *s)
{
	int x;
	int y;
	int dchan = -1;
	int span = -1;
	int dchancount = 0;

	if (pri) {
		for (x = 0; x < NUM_SPANS; x++) {
			for (y = 0; y < SIG_PRI_NUM_DCHANS; y++) {
				if (pris[x].pri.dchans[y]) {
					dchancount++;
				}

				if (pris[x].pri.dchans[y] == pri) {
					dchan = y;
				}
			}
			if (dchan >= 0) {
				span = x;
				break;
			}
			dchancount = 0;
		}
		if (-1 < span) {
			if (1 < dchancount) {
				ast_log_callid(LOG_ERROR, 0, "[PRI Span: %d D-Channel: %d] %s", span + 1, dchan, s);
			} else {
				ast_log_callid(LOG_ERROR, 0, "PRI Span: %d %s", span + 1, s);
			}
		} else {
			ast_log_callid(LOG_ERROR, 0, "PRI Span: ? %s", s);
		}
	} else {
		ast_log_callid(LOG_ERROR, 0, "PRI Span: ? %s", s);
	}

	ast_mutex_lock(&pridebugfdlock);

	if (pridebugfd >= 0) {
		if (write(pridebugfd, s, strlen(s)) < 0) {
			ast_log_callid(LOG_WARNING, 0, "write() failed: %s\n", strerror(errno));
		}
	}

	ast_mutex_unlock(&pridebugfdlock);
}
#endif	/* defined(HAVE_PRI) */

#if defined(HAVE_PRI)
static int prepare_pri(struct dahdi_pri *pri)
{
	int i, res, x;
	struct dahdi_params p;
	struct dahdi_bufferinfo bi;
	struct dahdi_spaninfo si;

	for (i = 0; i < SIG_PRI_NUM_DCHANS; i++) {
		if (!pri->dchannels[i])
			break;
		if (pri->pri.fds[i] >= 0) {
			/* A partial range addition. Not a complete setup. */
			break;
		}
		pri->pri.fds[i] = open("/dev/dahdi/channel", O_RDWR);
		if ((pri->pri.fds[i] < 0)) {
			ast_log(LOG_ERROR, "Unable to open D-channel (fd=%d) (%s)\n",
				pri->pri.fds[i], strerror(errno));
			return -1;
		}
		x = pri->dchannels[i];
		res = ioctl(pri->pri.fds[i], DAHDI_SPECIFY, &x);
		if (res) {
			dahdi_close_pri_fd(pri, i);
			ast_log(LOG_ERROR, "Unable to SPECIFY channel %d (%s)\n", x, strerror(errno));
			return -1;
		}
		memset(&p, 0, sizeof(p));
		res = ioctl(pri->pri.fds[i], DAHDI_GET_PARAMS, &p);
		if (res) {
			dahdi_close_pri_fd(pri, i);
			ast_log(LOG_ERROR, "Unable to get parameters for D-channel %d (%s)\n", x, strerror(errno));
			return -1;
		}
		if ((p.sigtype != DAHDI_SIG_HDLCFCS) && (p.sigtype != DAHDI_SIG_HARDHDLC)) {
			dahdi_close_pri_fd(pri, i);
			ast_log(LOG_ERROR, "D-channel %d is not in HDLC/FCS mode.\n", x);
			return -1;
		}
		memset(&si, 0, sizeof(si));
		res = ioctl(pri->pri.fds[i], DAHDI_SPANSTAT, &si);
		if (res) {
			dahdi_close_pri_fd(pri, i);
			ast_log(LOG_ERROR, "Unable to get span state for D-channel %d (%s)\n", x, strerror(errno));
		}
		if (!si.alarms) {
			pri_event_noalarm(&pri->pri, i, 1);
		} else {
			pri_event_alarm(&pri->pri, i, 1);
		}
		memset(&bi, 0, sizeof(bi));
		bi.txbufpolicy = DAHDI_POLICY_IMMEDIATE;
		bi.rxbufpolicy = DAHDI_POLICY_IMMEDIATE;
		bi.numbufs = 32;
		bi.bufsize = 1024;
		if (ioctl(pri->pri.fds[i], DAHDI_SET_BUFINFO, &bi)) {
			ast_log(LOG_ERROR, "Unable to set appropriate buffering on channel %d: %s\n", x, strerror(errno));
			dahdi_close_pri_fd(pri, i);
			return -1;
		}
		pri->pri.dchan_logical_span[i] = pris[p.spanno - 1].prilogicalspan;
	}
	return 0;
}
#endif	/* defined(HAVE_PRI) */

#if defined(HAVE_PRI)
static char *complete_span_helper(const char *line, const char *word, int pos, int state, int rpos)
{
	int which, span;
	char *ret = NULL;

	if (pos != rpos)
		return ret;

	for (which = span = 0; span < NUM_SPANS; span++) {
		if (pris[span].pri.pri && ++which > state) {
			if (ast_asprintf(&ret, "%d", span + 1) < 0) {	/* user indexes start from 1 */
				ret = NULL;
			}
			break;
		}
	}
	return ret;
}
#endif	/* defined(HAVE_PRI) */

#if defined(HAVE_PRI)
static char *complete_span_4(const char *line, const char *word, int pos, int state)
{
	return complete_span_helper(line,word,pos,state,3);
}
#endif	/* defined(HAVE_PRI) */

#if defined(HAVE_PRI)
static char *handle_pri_set_debug_file(struct ast_cli_entry *e, int cmd, struct ast_cli_args *a)
{
	int myfd;
	switch (cmd) {
	case CLI_INIT:
		e->command = "pri set debug file";
		e->usage = "Usage: pri set debug file [output-file]\n"
			"       Sends PRI debug output to the specified output file\n";
		return NULL;
	case CLI_GENERATE:
		return NULL;
	}
	if (a->argc < 5)
		return CLI_SHOWUSAGE;

	if (ast_strlen_zero(a->argv[4]))
		return CLI_SHOWUSAGE;

	myfd = open(a->argv[4], O_CREAT|O_WRONLY, AST_FILE_MODE);
	if (myfd < 0) {
		ast_cli(a->fd, "Unable to open '%s' for writing\n", a->argv[4]);
		return CLI_SUCCESS;
	}

	ast_mutex_lock(&pridebugfdlock);

	if (pridebugfd >= 0)
		close(pridebugfd);

	pridebugfd = myfd;
	ast_copy_string(pridebugfilename,a->argv[4],sizeof(pridebugfilename));
	ast_mutex_unlock(&pridebugfdlock);
	ast_cli(a->fd, "PRI debug output will be sent to '%s'\n", a->argv[4]);
	return CLI_SUCCESS;
}
#endif	/* defined(HAVE_PRI) */

#if defined(HAVE_PRI)
static int action_pri_debug_file_set(struct mansession *s, const struct message *m)
{
	const char *output_file = astman_get_header(m, "File");
	int myfd;

	if (ast_strlen_zero(output_file)) {
		astman_send_error(s, m, "Action must define a 'File'");
	}

	myfd = open(output_file, O_CREAT|O_WRONLY, AST_FILE_MODE);
	if (myfd < 0) {
		astman_send_error(s, m, "Unable to open requested file for writing");
		return 0;
	}

	ast_mutex_lock(&pridebugfdlock);

	if (pridebugfd >= 0) {
		close(pridebugfd);
	}

	pridebugfd = myfd;
	ast_copy_string(pridebugfilename, output_file, sizeof(pridebugfilename));
	ast_mutex_unlock(&pridebugfdlock);
	astman_send_ack(s, m, "PRI debug output will now be sent to requested file.");

	return 0;
}
#endif	/* defined(HAVE_PRI) */

#if defined(HAVE_PRI)
static int action_pri_debug_file_unset(struct mansession *s, const struct message *m)
{
	ast_mutex_lock(&pridebugfdlock);

	if (pridebugfd >= 0) {
		close(pridebugfd);
	}

	pridebugfd = -1;

	ast_mutex_unlock(&pridebugfdlock);

	astman_send_ack(s, m, "PRI Debug output to file disabled");
	return 0;
}
#endif	/* defined(HAVE_PRI) */

#if defined(HAVE_PRI)
static char *handle_pri_debug(struct ast_cli_entry *e, int cmd, struct ast_cli_args *a)
{
	int span;
	int x;
	int debugmask = 0;
	int level = 0;
	switch (cmd) {
	case CLI_INIT:
		e->command = "pri set debug {on|off|hex|intense|0|1|2|3|4|5|6|7|8|9|10|11|12|13|14|15} span";
		e->usage =
			"Usage: pri set debug {<level>|on|off|hex|intense} span <span>\n"
			"       Enables debugging on a given PRI span\n"
			"	Level is a bitmap of the following values:\n"
			"	1 General debugging incl. state changes\n"
			"	2 Decoded Q.931 messages\n"
			"	4 Decoded Q.921 messages\n"
			"	8 Raw hex dumps of Q.921 frames\n"
			"       on - equivalent to 3\n"
			"       hex - equivalent to 8\n"
			"       intense - equivalent to 15\n";
		return NULL;
	case CLI_GENERATE:
		return complete_span_4(a->line, a->word, a->pos, a->n);
	}
	if (a->argc < 6) {
		return CLI_SHOWUSAGE;
	}

	if (!strcasecmp(a->argv[3], "on")) {
		level = 3;
	} else if (!strcasecmp(a->argv[3], "off")) {
		level = 0;
	} else if (!strcasecmp(a->argv[3], "intense")) {
		level = 15;
	} else if (!strcasecmp(a->argv[3], "hex")) {
		level = 8;
	} else {
		level = atoi(a->argv[3]);
	}
	span = atoi(a->argv[5]);
	if ((span < 1) || (span > NUM_SPANS)) {
		ast_cli(a->fd, "Invalid span %s.  Should be a number %d to %d\n", a->argv[5], 1, NUM_SPANS);
		return CLI_SUCCESS;
	}
	if (!pris[span-1].pri.pri) {
		ast_cli(a->fd, "No PRI running on span %d\n", span);
		return CLI_SUCCESS;
	}

	if (level & 1) debugmask |= SIG_PRI_DEBUG_NORMAL;
	if (level & 2) debugmask |= PRI_DEBUG_Q931_DUMP;
	if (level & 4) debugmask |= PRI_DEBUG_Q921_DUMP;
	if (level & 8) debugmask |= PRI_DEBUG_Q921_RAW;

	/* Set debug level in libpri */
	for (x = 0; x < SIG_PRI_NUM_DCHANS; x++) {
		if (pris[span - 1].pri.dchans[x]) {
			pri_set_debug(pris[span - 1].pri.dchans[x], debugmask);
		}
	}
	if (level == 0) {
		/* Close the debugging file if it's set */
		ast_mutex_lock(&pridebugfdlock);
		if (0 <= pridebugfd) {
			close(pridebugfd);
			pridebugfd = -1;
			ast_cli(a->fd, "Disabled PRI debug output to file '%s'\n",
				pridebugfilename);
		}
		ast_mutex_unlock(&pridebugfdlock);
	}
	pris[span - 1].pri.debug = (level) ? 1 : 0;
	ast_cli(a->fd, "%s debugging on span %d\n", (level) ? "Enabled" : "Disabled", span);
	return CLI_SUCCESS;
}
#endif	/* defined(HAVE_PRI) */

#if defined(HAVE_PRI)
static int action_pri_debug_set(struct mansession *s, const struct message *m)
{
	const char *level = astman_get_header(m, "Level");
	const char *span = astman_get_header(m, "Span");
	int level_val;
	int span_val;
	int x;
	int debugmask = 0;

	if (ast_strlen_zero(level)) {
		astman_send_error(s, m, "'Level' was not specified");
		return 0;
	}

	if (ast_strlen_zero(span)) {
		astman_send_error(s, m, "'Span' was not specified");
		return 0;
	}

	if (!strcasecmp(level, "on")) {
		level_val = 3;
	} else if (!strcasecmp(level, "off")) {
		level_val = 0;
	} else if (!strcasecmp(level, "intense")) {
		level_val = 15;
	} else if (!strcasecmp(level, "hex")) {
		level_val = 8;
	} else {
		if (sscanf(level, "%30d", &level_val) != 1) {
			astman_send_error(s, m, "Invalid value for 'Level'");
			return 0;
		}
	}

	if (sscanf(span, "%30d", &span_val) != 1) {
		astman_send_error(s, m, "Invalid value for 'Span'");
	}

	if ((span_val < 1) || (span_val > NUM_SPANS)) {
		const char *id = astman_get_header(m, "ActionID");
		char id_text[256] = "";

		if (!ast_strlen_zero(id)) {
			snprintf(id_text, sizeof(id_text), "ActionID: %s\r\n", id);
		}

		astman_append(s, "Response: Error\r\n"
			"%s" /* id_text */
			"Message: Invalid span '%s' - Should be a number from 1 to %d\r\n"
			"\r\n",
			id_text,
			span, NUM_SPANS);

		return 0;
	}

	if (!pris[span_val-1].pri.pri) {
		astman_send_error(s, m, "No PRI running on requested span");
		return 0;
	}

	if (level_val & 1) {
		debugmask |= SIG_PRI_DEBUG_NORMAL;
	}
	if (level_val & 2) {
		debugmask |= PRI_DEBUG_Q931_DUMP;
	}
	if (level_val & 4) {
		debugmask |= PRI_DEBUG_Q921_DUMP;
	}
	if (level_val & 8) {
		debugmask |= PRI_DEBUG_Q921_RAW;
	}

	/* Set debug level in libpri */
	for (x = 0; x < SIG_PRI_NUM_DCHANS; x++) {
		if (pris[span_val - 1].pri.dchans[x]) {
			pri_set_debug(pris[span_val - 1].pri.dchans[x], debugmask);
		}
	}

	pris[span_val - 1].pri.debug = (level_val) ? 1 : 0;
	astman_send_ack(s, m, "Debug level set for requested span");

	return 0;
}
#endif	/* defined(HAVE_PRI) */

#if defined(HAVE_PRI)
#if defined(HAVE_PRI_SERVICE_MESSAGES)
static char *handle_pri_service_generic(struct ast_cli_entry *e, int cmd, struct ast_cli_args *a, int changestatus)
{
	unsigned *why;
	int channel;
	int trunkgroup;
	int x, y, fd = a->fd;
	int interfaceid = 0;
	char db_chan_name[20], db_answer[15];
	struct dahdi_pvt *tmp;
	struct dahdi_pri *pri;

	if (a->argc < 5 || a->argc > 6)
		return CLI_SHOWUSAGE;
	if (strchr(a->argv[4], ':')) {
		if (sscanf(a->argv[4], "%30d:%30d", &trunkgroup, &channel) != 2)
			return CLI_SHOWUSAGE;
		if ((trunkgroup < 1) || (channel < 1))
			return CLI_SHOWUSAGE;
		pri = NULL;
		for (x=0;x<NUM_SPANS;x++) {
			if (pris[x].pri.trunkgroup == trunkgroup) {
				pri = pris + x;
				break;
			}
		}
		if (!pri) {
			ast_cli(fd, "No such trunk group %d\n", trunkgroup);
			return CLI_FAILURE;
		}
	} else
		channel = atoi(a->argv[4]);

	if (a->argc == 6)
		interfaceid = atoi(a->argv[5]);

	/* either servicing a D-Channel */
	for (x = 0; x < NUM_SPANS; x++) {
		for (y = 0; y < SIG_PRI_NUM_DCHANS; y++) {
			if (pris[x].dchannels[y] == channel) {
				pri = pris + x;
				if (pri->pri.enable_service_message_support) {
					ast_mutex_lock(&pri->pri.lock);
					pri_maintenance_service(pri->pri.pri, interfaceid, -1, changestatus);
					ast_mutex_unlock(&pri->pri.lock);
				} else {
					ast_cli(fd,
						"\n\tThis operation has not been enabled in chan_dahdi.conf, set 'service_message_support=yes' to use this operation.\n"
						"\tNote only 4ESS, 5ESS, and NI2 switch types are supported.\n\n");
				}
				return CLI_SUCCESS;
			}
		}
	}

	/* or servicing a B-Channel */
	ast_mutex_lock(&iflock);
	for (tmp = iflist; tmp; tmp = tmp->next) {
		if (tmp->pri && tmp->channel == channel) {
			ast_mutex_unlock(&iflock);
			ast_mutex_lock(&tmp->pri->lock);
			if (!tmp->pri->enable_service_message_support) {
				ast_mutex_unlock(&tmp->pri->lock);
				ast_cli(fd,
					"\n\tThis operation has not been enabled in chan_dahdi.conf, set 'service_message_support=yes' to use this operation.\n"
					"\tNote only 4ESS, 5ESS, and NI2 switch types are supported.\n\n");
				return CLI_SUCCESS;
			}
			snprintf(db_chan_name, sizeof(db_chan_name), "%s/%d:%d", dahdi_db, tmp->span, channel);
			why = &((struct sig_pri_chan *) tmp->sig_pvt)->service_status;
			switch(changestatus) {
			case 0: /* enable */
				/* Near end wants to be in service now. */
				ast_db_del(db_chan_name, SRVST_DBKEY);
				*why &= ~SRVST_NEAREND;
				if (*why) {
					snprintf(db_answer, sizeof(db_answer), "%s:%u", SRVST_TYPE_OOS, *why);
					ast_db_put(db_chan_name, SRVST_DBKEY, db_answer);
				} else {
					dahdi_pri_update_span_devstate(tmp->pri);
				}
				break;
			/* case 1:  -- loop */
			case 2: /* disable */
				/* Near end wants to be out-of-service now. */
				ast_db_del(db_chan_name, SRVST_DBKEY);
				*why |= SRVST_NEAREND;
				snprintf(db_answer, sizeof(db_answer), "%s:%u", SRVST_TYPE_OOS, *why);
				ast_db_put(db_chan_name, SRVST_DBKEY, db_answer);
				dahdi_pri_update_span_devstate(tmp->pri);
				break;
			/* case 3:  -- continuity */
			/* case 4:  -- shutdown */
			default:
				ast_log(LOG_WARNING, "Unsupported changestatus: '%d'\n", changestatus);
				break;
			}
			pri_maintenance_bservice(tmp->pri->pri, tmp->sig_pvt, changestatus);
			ast_mutex_unlock(&tmp->pri->lock);
			return CLI_SUCCESS;
		}
	}
	ast_mutex_unlock(&iflock);

	ast_cli(fd, "Unable to find given channel %d, possibly not a PRI\n", channel);
	return CLI_FAILURE;
}

static char *handle_pri_service_enable_channel(struct ast_cli_entry *e, int cmd, struct ast_cli_args *a)
{
	switch (cmd) {
	case CLI_INIT:
		e->command = "pri service enable channel";
		e->usage =
			"Usage: pri service enable channel <channel> [<interface id>]\n"
			"       Send an AT&T / NFAS / CCS ANSI T1.607 maintenance message\n"
			"	to restore a channel to service, with optional interface id\n"
			"	as agreed upon with remote switch operator\n";
		return NULL;
	case CLI_GENERATE:
		return NULL;
	}
	return handle_pri_service_generic(e, cmd, a, 0);
}

static char *handle_pri_service_disable_channel(struct ast_cli_entry *e, int cmd, struct ast_cli_args *a)
{
	switch (cmd) {
	case CLI_INIT:
		e->command = "pri service disable channel";
		e->usage =
			"Usage: pri service disable channel <chan num> [<interface id>]\n"
			"	Send an AT&T / NFAS / CCS ANSI T1.607 maintenance message\n"
			"	to remove a channel from service, with optional interface id\n"
			"	as agreed upon with remote switch operator\n";
		return NULL;
	case CLI_GENERATE:
		return NULL;
	}
	return handle_pri_service_generic(e, cmd, a, 2);
}
#endif	/* defined(HAVE_PRI_SERVICE_MESSAGES) */
#endif	/* defined(HAVE_PRI) */

#if defined(HAVE_PRI)
static char *handle_pri_show_channels(struct ast_cli_entry *e, int cmd, struct ast_cli_args *a)
{
	int span;

	switch (cmd) {
	case CLI_INIT:
		e->command = "pri show channels";
		e->usage =
			"Usage: pri show channels\n"
			"       Displays PRI channel information such as the current mapping\n"
			"       of DAHDI B channels to Asterisk channel names and which calls\n"
			"       are on hold or call-waiting.  Calls on hold or call-waiting\n"
			"       are not associated with any B channel.\n";
		return NULL;
	case CLI_GENERATE:
		return NULL;
	}

	if (a->argc != 3)
		return CLI_SHOWUSAGE;

	sig_pri_cli_show_channels_header(a->fd);
	for (span = 0; span < NUM_SPANS; ++span) {
		if (pris[span].pri.pri) {
			sig_pri_cli_show_channels(a->fd, &pris[span].pri);
		}
	}
	return CLI_SUCCESS;
}
#endif	/* defined(HAVE_PRI) */

#if defined(HAVE_PRI)
static char *handle_pri_show_spans(struct ast_cli_entry *e, int cmd, struct ast_cli_args *a)
{
	int span;

	switch (cmd) {
	case CLI_INIT:
		e->command = "pri show spans";
		e->usage =
			"Usage: pri show spans\n"
			"       Displays PRI span information\n";
		return NULL;
	case CLI_GENERATE:
		return NULL;
	}

	if (a->argc != 3)
		return CLI_SHOWUSAGE;

	for (span = 0; span < NUM_SPANS; span++) {
		if (pris[span].pri.pri) {
			sig_pri_cli_show_spans(a->fd, span + 1, &pris[span].pri);
		}
	}
	return CLI_SUCCESS;
}
#endif	/* defined(HAVE_PRI) */

#if defined(HAVE_PRI)
#define container_of(ptr, type, member) \
	((type *)((char *)(ptr) - offsetof(type, member)))
/*!
 * \internal
 * \brief Destroy a D-Channel of a PRI span
 * \since 12
 *
 * \param pri the pri span
 *
 * Shuts down a span and destroys its D-Channel. Further destruction
 * of the B-channels using dahdi_destroy_channel() would probably be required
 * for the B-Channels.
 */
static void pri_destroy_span(struct sig_pri_span *pri)
{
	int i;
	int res;
	int cancel_code;
	struct dahdi_pri* dahdi_pri;
	pthread_t master = pri->master;

	if (!master || (master == AST_PTHREADT_NULL)) {
		return;
	}
	ast_debug(2, "About to destroy DAHDI channels of span %d.\n", pri->span);
	for (i = 0; i < pri->numchans; i++) {
		int channel;
		struct sig_pri_chan *pvt = pri->pvts[i];

		if (!pvt) {
			continue;
		}
		channel = pvt->channel;
		ast_debug(2, "About to destroy B-channel %d.\n", channel);
		dahdi_destroy_channel_range(channel, channel);
	}

	cancel_code = pthread_cancel(master);
	pthread_kill(master, SIGURG);
	ast_debug(4,
		"Waiting to join thread of span %d "
		"with pid=%p cancel_code=%d\n",
		pri->span, (void *)master, cancel_code);
	res = pthread_join(master, NULL);
	if (res != 0) {
		ast_log(LOG_NOTICE, "pthread_join failed: %d\n", res);
	}
	pri->master = AST_PTHREADT_NULL;

	/* The 'struct dahdi_pri' that contains our 'struct sig_pri_span' */
	dahdi_pri = container_of(pri, struct dahdi_pri, pri);
	for (i = 0; i < SIG_PRI_NUM_DCHANS; i++) {
		ast_debug(4, "closing pri_fd %d\n", i);
		dahdi_close_pri_fd(dahdi_pri, i);
		dahdi_pri->dchannels[i] = 0;
	}
	sig_pri_init_pri(pri);
	ast_debug(1, "PRI span %d destroyed\n", pri->span);
}

static char *handle_pri_destroy_span(struct ast_cli_entry *e, int cmd,
		struct ast_cli_args *a)
{
	int span;
	int res;
	struct sig_pri_span *pri;

	switch (cmd) {
	case CLI_INIT:
		e->command = "pri destroy span";
		e->usage =
			"Usage: pri destroy span <span>\n"
			"       Destroys D-channel of span and its B-channels.\n"
			"	DON'T USE THIS UNLESS YOU KNOW WHAT YOU ARE DOING.\n";
		return NULL;
	case CLI_GENERATE:
		return complete_span_4(a->line, a->word, a->pos, a->n);
	}

	if (a->argc < 4) {
		return CLI_SHOWUSAGE;
	}
	res = sscanf(a->argv[3], "%30d", &span);
	if ((res != 1) || span < 1 || span > NUM_SPANS) {
		ast_cli(a->fd,
			"Invalid span '%s'.  Should be a number from %d to %d\n",
			a->argv[3], 1, NUM_SPANS);
		return CLI_SUCCESS;
	}
	pri = &pris[span - 1].pri;
	if (!pri->pri) {
		ast_cli(a->fd, "No PRI running on span %d\n", span);
		return CLI_SUCCESS;
	}

	pri_destroy_span(pri);
	return CLI_SUCCESS;
}

#endif	/* defined(HAVE_PRI) */

#if defined(HAVE_PRI)
static char *handle_pri_show_span(struct ast_cli_entry *e, int cmd, struct ast_cli_args *a)
{
	int span;

	switch (cmd) {
	case CLI_INIT:
		e->command = "pri show span";
		e->usage =
			"Usage: pri show span <span>\n"
			"       Displays PRI Information on a given PRI span\n";
		return NULL;
	case CLI_GENERATE:
		return complete_span_4(a->line, a->word, a->pos, a->n);
	}

	if (a->argc < 4)
		return CLI_SHOWUSAGE;
	span = atoi(a->argv[3]);
	if ((span < 1) || (span > NUM_SPANS)) {
		ast_cli(a->fd, "Invalid span '%s'.  Should be a number from %d to %d\n", a->argv[3], 1, NUM_SPANS);
		return CLI_SUCCESS;
	}
	if (!pris[span-1].pri.pri) {
		ast_cli(a->fd, "No PRI running on span %d\n", span);
		return CLI_SUCCESS;
	}

	sig_pri_cli_show_span(a->fd, pris[span-1].dchannels, &pris[span-1].pri);

	return CLI_SUCCESS;
}
#endif	/* defined(HAVE_PRI) */

#if defined(HAVE_PRI)
static char *handle_pri_show_debug(struct ast_cli_entry *e, int cmd, struct ast_cli_args *a)
{
	int x;
	int span;
	int count=0;
	int debug;

	switch (cmd) {
	case CLI_INIT:
		e->command = "pri show debug";
		e->usage =
			"Usage: pri show debug\n"
			"	Show the debug state of pri spans\n";
		return NULL;
	case CLI_GENERATE:
		return NULL;
	}

	for (span = 0; span < NUM_SPANS; span++) {
		if (pris[span].pri.pri) {
			for (x = 0; x < SIG_PRI_NUM_DCHANS; x++) {
				if (pris[span].pri.dchans[x]) {
					debug = pri_get_debug(pris[span].pri.dchans[x]);
					ast_cli(a->fd, "Span %d: Debug: %s\tIntense: %s\n", span+1, (debug&PRI_DEBUG_Q931_STATE)? "Yes" : "No" ,(debug&PRI_DEBUG_Q921_RAW)? "Yes" : "No" );
					count++;
				}
			}
		}

	}
	ast_mutex_lock(&pridebugfdlock);
	if (pridebugfd >= 0)
		ast_cli(a->fd, "Logging PRI debug to file %s\n", pridebugfilename);
	ast_mutex_unlock(&pridebugfdlock);

	if (!count)
		ast_cli(a->fd, "No PRI running\n");
	return CLI_SUCCESS;
}
#endif	/* defined(HAVE_PRI) */

#if defined(HAVE_PRI)
static char *handle_pri_version(struct ast_cli_entry *e, int cmd, struct ast_cli_args *a)
{
	switch (cmd) {
	case CLI_INIT:
		e->command = "pri show version";
		e->usage =
			"Usage: pri show version\n"
			"Show libpri version information\n";
		return NULL;
	case CLI_GENERATE:
		return NULL;
	}

	ast_cli(a->fd, "libpri version: %s\n", pri_get_version());

	return CLI_SUCCESS;
}
#endif	/* defined(HAVE_PRI) */

#if defined(HAVE_PRI)
static struct ast_cli_entry dahdi_pri_cli[] = {
	AST_CLI_DEFINE(handle_pri_debug, "Enables PRI debugging on a span"),
#if defined(HAVE_PRI_SERVICE_MESSAGES)
 	AST_CLI_DEFINE(handle_pri_service_enable_channel, "Return a channel to service"),
 	AST_CLI_DEFINE(handle_pri_service_disable_channel, "Remove a channel from service"),
#endif	/* defined(HAVE_PRI_SERVICE_MESSAGES) */
	AST_CLI_DEFINE(handle_pri_show_channels, "Displays PRI channel information"),
	AST_CLI_DEFINE(handle_pri_show_spans, "Displays PRI span information"),
	AST_CLI_DEFINE(handle_pri_show_span, "Displays PRI span information"),
	AST_CLI_DEFINE(handle_pri_destroy_span, "Destroy a PRI span"),
	AST_CLI_DEFINE(handle_pri_show_debug, "Displays current PRI debug settings"),
	AST_CLI_DEFINE(handle_pri_set_debug_file, "Sends PRI debug output to the specified file"),
	AST_CLI_DEFINE(handle_pri_version, "Displays libpri version"),
};
#endif	/* defined(HAVE_PRI) */

#ifdef HAVE_OPENR2

static char *handle_mfcr2_version(struct ast_cli_entry *e, int cmd, struct ast_cli_args *a)
{
	switch (cmd) {
	case CLI_INIT:
		e->command = "mfcr2 show version";
		e->usage =
			"Usage: mfcr2 show version\n"
			"       Shows the version of the OpenR2 library being used.\n";
		return NULL;
	case CLI_GENERATE:
		return NULL;
	}
	ast_cli(a->fd, "OpenR2 version: %s, revision: %s\n", openr2_get_version(), openr2_get_revision());
	return CLI_SUCCESS;
}

static char *handle_mfcr2_show_variants(struct ast_cli_entry *e, int cmd, struct ast_cli_args *a)
{
#define FORMAT "%4s %40s\n"
	int i = 0;
	int numvariants = 0;
	const openr2_variant_entry_t *variants;
	switch (cmd) {
	case CLI_INIT:
		e->command = "mfcr2 show variants";
		e->usage =
			"Usage: mfcr2 show variants\n"
			"       Shows the list of MFC/R2 variants supported.\n";
		return NULL;
	case CLI_GENERATE:
		return NULL;
	}
	if (!(variants = openr2_proto_get_variant_list(&numvariants))) {
		ast_cli(a->fd, "Failed to get list of variants.\n");
		return CLI_FAILURE;
	}
	ast_cli(a->fd, FORMAT, "Variant Code", "Country");
	for (i = 0; i < numvariants; i++) {
		ast_cli(a->fd, FORMAT, variants[i].name, variants[i].country);
	}
	return CLI_SUCCESS;
#undef FORMAT
}

static char *handle_mfcr2_show_channels(struct ast_cli_entry *e, int cmd, struct ast_cli_args *a)
{
#define FORMAT "%4s %4s %-7.7s %-7.7s %-8.8s %-9.9s %-16.16s %-8.8s %-8.8s\n"
	int filtertype = 0;
	int targetnum = 0;
	char channo[5];
	char linkno[5];
	char anino[5];
	char dnisno[5];
	struct dahdi_pvt *p;
	openr2_context_t *r2context;
	openr2_variant_t r2variant;
	switch (cmd) {
	case CLI_INIT:
		e->command = "mfcr2 show channels [group|context]";
		e->usage =
			"Usage: mfcr2 show channels [group <group> | context <context>]\n"
			"       Shows the DAHDI channels configured with MFC/R2 signaling.\n";
		return NULL;
	case CLI_GENERATE:
		return NULL;
	}
	if (!((a->argc == 3) || (a->argc == 5))) {
		return CLI_SHOWUSAGE;
	}
	if (a->argc == 5) {
		if (!strcasecmp(a->argv[3], "group")) {
			targetnum = atoi(a->argv[4]);
			if ((targetnum < 0) || (targetnum > 63))
				return CLI_SHOWUSAGE;
			targetnum = 1 << targetnum;
			filtertype = 1;
		} else if (!strcasecmp(a->argv[3], "context")) {
			filtertype = 2;
		} else {
			return CLI_SHOWUSAGE;
		}
	}
	ast_cli(a->fd, FORMAT, "Chan", "Link#", "Variant", "Max ANI", "Max DNIS", "ANI First", "Immediate Accept", "Tx CAS", "Rx CAS");
	ast_mutex_lock(&iflock);
	for (p = iflist; p; p = p->next) {
		if (!(p->sig & SIG_MFCR2) || !p->r2chan) {
			continue;
		}
		if (filtertype) {
			switch(filtertype) {
			case 1: /* mfcr2 show channels group <group> */
				if (p->group != targetnum) {
					continue;
				}
				break;
			case 2: /* mfcr2 show channels context <context> */
				if (strcasecmp(p->context, a->argv[4])) {
					continue;
				}
				break;
			default:
				;
			}
		}
		r2context = openr2_chan_get_context(p->r2chan);
		r2variant = openr2_context_get_variant(r2context);
		snprintf(channo, sizeof(channo), "%d", p->channel);
		snprintf(linkno, sizeof(linkno), "%d", p->mfcr2->index);
		snprintf(anino, sizeof(anino), "%d", openr2_context_get_max_ani(r2context));
		snprintf(dnisno, sizeof(dnisno), "%d", openr2_context_get_max_dnis(r2context));
		ast_cli(a->fd, FORMAT, channo, linkno, openr2_proto_get_variant_string(r2variant),
				anino, dnisno, openr2_context_get_ani_first(r2context) ? "Yes" : "No",
				openr2_context_get_immediate_accept(r2context) ? "Yes" : "No",
				openr2_chan_get_tx_cas_string(p->r2chan), openr2_chan_get_rx_cas_string(p->r2chan));
	}
	ast_mutex_unlock(&iflock);
	return CLI_SUCCESS;
#undef FORMAT
}

static char *handle_mfcr2_set_debug(struct ast_cli_entry *e, int cmd, struct ast_cli_args *a)
{
	struct dahdi_pvt *p = NULL;
	int channo = 0;
	char *toklevel = NULL;
	char *saveptr = NULL;
	char *logval = NULL;
	openr2_log_level_t loglevel = OR2_LOG_NOTHING;
	openr2_log_level_t tmplevel = OR2_LOG_NOTHING;
	switch (cmd) {
	case CLI_INIT:
		e->command = "mfcr2 set debug";
		e->usage =
			"Usage: mfcr2 set debug <loglevel> <channel>\n"
			"       Set a new logging level for the specified channel.\n"
			"       If no channel is specified the logging level will be applied to all channels.\n";
		return NULL;
	case CLI_GENERATE:
		return NULL;
	}
	if (a->argc < 4) {
		return CLI_SHOWUSAGE;
	}
	channo = (a->argc == 5) ? atoi(a->argv[4]) : -1;
	logval = ast_strdupa(a->argv[3]);
	toklevel = strtok_r(logval, ",", &saveptr);
	if (-1 == (tmplevel = openr2_log_get_level(toklevel))) {
		ast_cli(a->fd, "Invalid MFC/R2 logging level '%s'.\n", a->argv[3]);
		return CLI_FAILURE;
	} else if (OR2_LOG_NOTHING == tmplevel) {
		loglevel = tmplevel;
	} else {
		loglevel |= tmplevel;
		while ((toklevel = strtok_r(NULL, ",", &saveptr))) {
			if (-1 == (tmplevel = openr2_log_get_level(toklevel))) {
				ast_cli(a->fd, "Ignoring invalid logging level: '%s'.\n", toklevel);
				continue;
			}
			loglevel |= tmplevel;
		}
	}
	ast_mutex_lock(&iflock);
	for (p = iflist; p; p = p->next) {
		if (!(p->sig & SIG_MFCR2) || !p->r2chan) {
			continue;
		}
		if ((channo != -1) && (p->channel != channo )) {
			continue;
		}
		openr2_chan_set_log_level(p->r2chan, loglevel);
		if (channo != -1) {
			ast_cli(a->fd, "MFC/R2 debugging set to '%s' for channel %d.\n", a->argv[3], p->channel);
			break;
		}
	}
	if ((channo != -1) && !p) {
		ast_cli(a->fd, "MFC/R2 channel %d not found.\n", channo);
	}
	if (channo == -1) {
		ast_cli(a->fd, "MFC/R2 debugging set to '%s' for all channels.\n", a->argv[3]);
	}
	ast_mutex_unlock(&iflock);
	return CLI_SUCCESS;
}

static char *handle_mfcr2_call_files(struct ast_cli_entry *e, int cmd, struct ast_cli_args *a)
{
	struct dahdi_pvt *p = NULL;
	int channo = 0;
	switch (cmd) {
	case CLI_INIT:
		e->command = "mfcr2 call files [on|off]";
		e->usage =
			"Usage: mfcr2 call files [on|off] <channel>\n"
			"       Enable call files creation on the specified channel.\n"
			"       If no channel is specified call files creation policy will be applied to all channels.\n";
		return NULL;
	case CLI_GENERATE:
		return NULL;
	}
	if (a->argc < 4) {
		return CLI_SHOWUSAGE;
	}
	channo = (a->argc == 5) ? atoi(a->argv[4]) : -1;
	ast_mutex_lock(&iflock);
	for (p = iflist; p; p = p->next) {
		if (!(p->sig & SIG_MFCR2) || !p->r2chan) {
			continue;
		}
		if ((channo != -1) && (p->channel != channo )) {
			continue;
		}
		if (ast_true(a->argv[3])) {
			openr2_chan_enable_call_files(p->r2chan);
		} else {
			openr2_chan_disable_call_files(p->r2chan);
		}
		if (channo != -1) {
			if (ast_true(a->argv[3])) {
				ast_cli(a->fd, "MFC/R2 call files enabled for channel %d.\n", p->channel);
			} else {
				ast_cli(a->fd, "MFC/R2 call files disabled for channel %d.\n", p->channel);
			}
			break;
		}
	}
	if ((channo != -1) && !p) {
		ast_cli(a->fd, "MFC/R2 channel %d not found.\n", channo);
	}
	if (channo == -1) {
		if (ast_true(a->argv[3])) {
			ast_cli(a->fd, "MFC/R2 Call files enabled for all channels.\n");
		} else {
			ast_cli(a->fd, "MFC/R2 Call files disabled for all channels.\n");
		}
	}
	ast_mutex_unlock(&iflock);
	return CLI_SUCCESS;
}

static char *handle_mfcr2_set_idle(struct ast_cli_entry *e, int cmd, struct ast_cli_args *a)
{
	struct dahdi_pvt *p = NULL;
	int channo = 0;
	switch (cmd) {
	case CLI_INIT:
		e->command = "mfcr2 set idle";
		e->usage =
			"Usage: mfcr2 set idle <channel>\n"
			"       DON'T USE THIS UNLESS YOU KNOW WHAT YOU ARE DOING.\n"
			"       Force the given channel into IDLE state.\n"
			"       If no channel is specified, all channels will be set to IDLE.\n";
		return NULL;
	case CLI_GENERATE:
		return NULL;
	}
	channo = (a->argc == 4) ? atoi(a->argv[3]) : -1;
	ast_mutex_lock(&iflock);
	for (p = iflist; p; p = p->next) {
		if (!(p->sig & SIG_MFCR2) || !p->r2chan) {
			continue;
		}
		if ((channo != -1) && (p->channel != channo )) {
			continue;
		}
		openr2_chan_set_idle(p->r2chan);
		ast_mutex_lock(&p->lock);
		p->locallyblocked = 0;
		p->mfcr2call = 0;
		ast_mutex_unlock(&p->lock);
		if (channo != -1) {
			break;
		}
	}
	if ((channo != -1) && !p) {
		ast_cli(a->fd, "MFC/R2 channel %d not found.\n", channo);
	}
	ast_mutex_unlock(&iflock);
	return CLI_SUCCESS;
}

static char *handle_mfcr2_set_blocked(struct ast_cli_entry *e, int cmd, struct ast_cli_args *a)
{
	struct dahdi_pvt *p = NULL;
	int channo = 0;
	switch (cmd) {
	case CLI_INIT:
		e->command = "mfcr2 set blocked";
		e->usage =
			"Usage: mfcr2 set blocked <channel>\n"
			"       DON'T USE THIS UNLESS YOU KNOW WHAT YOU ARE DOING.\n"
			"       Force the given channel into BLOCKED state.\n"
			"       If no channel is specified, all channels will be set to BLOCKED.\n";
		return NULL;
	case CLI_GENERATE:
		return NULL;
	}
	channo = (a->argc == 4) ? atoi(a->argv[3]) : -1;
	ast_mutex_lock(&iflock);
	for (p = iflist; p; p = p->next) {
		if (!(p->sig & SIG_MFCR2) || !p->r2chan) {
			continue;
		}
		if ((channo != -1) && (p->channel != channo )) {
			continue;
		}
		openr2_chan_set_blocked(p->r2chan);
		ast_mutex_lock(&p->lock);
		p->locallyblocked = 1;
		ast_mutex_unlock(&p->lock);
		if (channo != -1) {
			break;
		}
	}
	if ((channo != -1) && !p) {
		ast_cli(a->fd, "MFC/R2 channel %d not found.\n", channo);
	}
	ast_mutex_unlock(&iflock);
	return CLI_SUCCESS;
}

static void mfcr2_show_links_of(struct ast_cli_args *a, struct r2links *list_head, const char *title)
{
#define FORMAT "%-5s %-10s %-15s %-10s %s\n"
	AST_LIST_LOCK(list_head);
	if (! AST_LIST_EMPTY(list_head)) {
		int x = 0;
		char index[5];
		char live_chans_str[5];
		char channel_list[R2_LINK_CAPACITY * 4];
		struct r2link_entry *cur;
		ast_cli(a->fd, "%s\n", title);
		ast_cli(a->fd, FORMAT, "Index", "Thread", "Dahdi-Device", "Channels", "Channel-List");
		AST_LIST_TRAVERSE(list_head, cur, list) {
			struct dahdi_mfcr2 *mfcr2 = &cur->mfcr2;
			const char *thread_status = NULL;
			int i;
			int len;
			int inside_range;
			int channo;
			int prev_channo;
			x++;
			if (mfcr2->r2master == 0L) {
				thread_status = "zero";
			} else if (mfcr2->r2master == AST_PTHREADT_NULL) {
				thread_status = "none";
			} else {
				thread_status = "created";
			}
			snprintf(index, sizeof(index), "%d", mfcr2->index);
			snprintf(live_chans_str, sizeof(live_chans_str), "%d", mfcr2->live_chans);
			channo = 0;
			prev_channo = 0;
			inside_range = 0;
			len = 0;
			/* Prepare nice string in channel_list[] */
			for (i = 0; i < mfcr2->numchans && len < sizeof(channel_list) - 1; i++) {
				struct dahdi_pvt *p = mfcr2->pvts[i];
				if (!p) {
					continue;
				}
				channo = p->channel;
				/* Don't show a range until we know the last channel number */
				if (prev_channo && prev_channo == channo - 1) {
					prev_channo = channo;
					inside_range = 1;
					continue;
				}
				if (inside_range) {
					/* Close range */
					len += snprintf(channel_list + len, sizeof(channel_list) - len - 1, "-%d,%d", prev_channo, channo);
					inside_range = 0;
				} else if (prev_channo) {
					/* Non-sequential channel numbers */
					len += snprintf(channel_list + len, sizeof(channel_list) - len - 1, ",%d", channo);
				} else {
					/* First channel number */
					len += snprintf(channel_list + len, sizeof(channel_list) - len - 1, "%d", channo);
				}
				prev_channo = channo;
			}
			/* Handle leftover channels */
			if (inside_range) {
				/* Close range */
				len += snprintf(channel_list + len, sizeof(channel_list) - len - 1, "-%d", channo);
				inside_range = 0;
			} else if (prev_channo) {
				/* Non-sequential channel numbers */
				len += snprintf(channel_list + len, sizeof(channel_list) - len - 1, ",%d", channo);
			}
			// channel_list[len] = '\0';
			ast_cli(a->fd, FORMAT,
				index,
				thread_status,
				(mfcr2->nodev) ? "MISSING" : "OK",
				live_chans_str,
				channel_list);
		}
	}
	AST_LIST_UNLOCK(list_head);
#undef FORMAT
}

static char *handle_mfcr2_show_links(struct ast_cli_entry *e, int cmd, struct ast_cli_args *a)
{
	switch (cmd) {
	case CLI_INIT:
		e->command = "mfcr2 show links";
		e->usage =
			"Usage: mfcr2 show links\n"
			"       Shows the DAHDI MFC/R2 links.\n";
		return NULL;
	case CLI_GENERATE:
		return NULL;
	}
	if (a->argc != 3) {
		return CLI_SHOWUSAGE;
	}
	mfcr2_show_links_of(a, &r2links, "Live links\n");
	mfcr2_show_links_of(a, &nodev_r2links, "Links to be removed (device missing)\n");
	return CLI_SUCCESS;
}

static char *handle_mfcr2_destroy_link(struct ast_cli_entry *e, int cmd, struct ast_cli_args *a)
{
	int res;
	int wanted_link_index;
	int found_link = 0;
	struct r2link_entry *cur = NULL;

	switch (cmd) {
	case CLI_INIT:
		e->command = "mfcr2 destroy link";
		e->usage =
			"Usage: mfcr2 destroy link <index-number>\n"
			"       Destroys D-channel of link and its B-channels.\n"
			"	DON'T USE THIS UNLESS YOU KNOW WHAT YOU ARE DOING.\n";
		return NULL;
	case CLI_GENERATE:
		return NULL;
	}
	if (a->argc < 4) {
		return CLI_SHOWUSAGE;
	}
	res = sscanf(a->argv[3], "%30d", &wanted_link_index);
	if ((res != 1) || wanted_link_index < 1) {
		ast_cli(a->fd,
			"Invalid link index '%s'.  Should be a positive number\n", a->argv[3]);
		return CLI_SUCCESS;
	}
	AST_LIST_LOCK(&r2links);
	AST_LIST_TRAVERSE_SAFE_BEGIN(&r2links, cur, list) {
		struct dahdi_mfcr2 *mfcr2 = &cur->mfcr2;
		if (wanted_link_index == mfcr2->index) {
			AST_LIST_MOVE_CURRENT(&nodev_r2links, list);
			r2links_count--;
			break;
		}
	}
	AST_LIST_TRAVERSE_SAFE_END;
	AST_LIST_UNLOCK(&r2links);
	if (! found_link) {
		ast_cli(a->fd, "No link found with index %d.\n", wanted_link_index);
		return CLI_FAILURE;
	}
	return CLI_SUCCESS;
}

static struct ast_cli_entry dahdi_mfcr2_cli[] = {
	AST_CLI_DEFINE(handle_mfcr2_version, "Show OpenR2 library version"),
	AST_CLI_DEFINE(handle_mfcr2_show_variants, "Show supported MFC/R2 variants"),
	AST_CLI_DEFINE(handle_mfcr2_show_channels, "Show MFC/R2 channels"),
	AST_CLI_DEFINE(handle_mfcr2_show_links, "Show MFC/R2 links"),
	AST_CLI_DEFINE(handle_mfcr2_set_debug, "Set MFC/R2 channel logging level"),
	AST_CLI_DEFINE(handle_mfcr2_call_files, "Enable/Disable MFC/R2 call files"),
	AST_CLI_DEFINE(handle_mfcr2_set_idle, "Reset MFC/R2 channel forcing it to IDLE"),
	AST_CLI_DEFINE(handle_mfcr2_set_blocked, "Reset MFC/R2 channel forcing it to BLOCKED"),
	AST_CLI_DEFINE(handle_mfcr2_destroy_link, "Destroy given MFC/R2 link"),
};

#endif /* HAVE_OPENR2 */

static char *dahdi_destroy_channels(struct ast_cli_entry *e, int cmd, struct ast_cli_args *a)
{
	int start;
	int end;
	switch (cmd) {
	case CLI_INIT:
		e->command = "dahdi destroy channels";
		e->usage =
			"Usage: dahdi destroy channels <from_channel> [<to_channel>]\n"
			"	DON'T USE THIS UNLESS YOU KNOW WHAT YOU ARE DOING.  Immediately removes a given channel, whether it is in use or not\n";
		return NULL;
	case CLI_GENERATE:
		return NULL;
	}
	if ((a->argc < 4) || a->argc > 5) {
		return CLI_SHOWUSAGE;
	}
	start = atoi(a->argv[3]);
	if (start < 1) {
		ast_cli(a->fd, "Invalid starting channel number %s.\n",
				a->argv[4]);
		return CLI_FAILURE;
	}
	if (a->argc == 5) {
		end = atoi(a->argv[4]);
		if (end < 1) {
			ast_cli(a->fd, "Invalid ending channel number %s.\n",
					a->argv[4]);
			return CLI_FAILURE;
		}
	} else {
		end = start;
	}

	if (end < start) {
		ast_cli(a->fd,
			"range end (%d) is smaller than range start (%d)\n",
			end, start);
		return CLI_FAILURE;
	}
	dahdi_destroy_channel_range(start, end);
	return CLI_SUCCESS;
}

static char *dahdi_create_channels(struct ast_cli_entry *e, int cmd, struct ast_cli_args *a)
{
	int start;
	int end;
	int ret;

	switch (cmd) {
	case CLI_INIT:
		e->command = "dahdi create channels";
		e->usage = "Usage: dahdi create channels <from> [<to>] - a range of channels\n"
			   "       dahdi create channels new           - add channels not yet created\n"
			   "For ISDN  and SS7 the range should include complete spans.\n";
		return NULL;
	case CLI_GENERATE:
		return NULL;
	}
	if ((a->argc < 4) || a->argc > 5) {
		return CLI_SHOWUSAGE;
	}
	if (a->argc == 4 && !strcmp(a->argv[3], "new")) {
		ret = dahdi_create_channel_range(0, 0);
		return (RESULT_SUCCESS == ret) ? CLI_SUCCESS : CLI_FAILURE;
	}
	start = atoi(a->argv[3]);
	if (start <= 0) {
		ast_cli(a->fd, "Invalid starting channel number '%s'.\n",
				a->argv[3]);
		return CLI_FAILURE;
	}
	if (a->argc == 5) {
		end = atoi(a->argv[4]);
		if (end <= 0) {
			ast_cli(a->fd, "Invalid ending channel number '%s'.\n",
					a->argv[4]);
			return CLI_FAILURE;
		}
	} else {
		end = start;
	}
	if (end < start) {
		ast_cli(a->fd,
			"range end (%d) is smaller than range start (%d)\n",
			end, start);
		return CLI_FAILURE;
	}
	ret = dahdi_create_channel_range(start, end);
	return (RESULT_SUCCESS == ret) ? CLI_SUCCESS : CLI_FAILURE;
}

static void dahdi_softhangup_all(void)
{
	struct dahdi_pvt *p;
retry:
	ast_mutex_lock(&iflock);
	for (p = iflist; p; p = p->next) {
		ast_mutex_lock(&p->lock);
		if (p->owner && !p->restartpending) {
			if (ast_channel_trylock(p->owner)) {
				if (DEBUG_ATLEAST(3))
					ast_verbose("Avoiding deadlock\n");
				/* Avoid deadlock since you're not supposed to lock iflock or pvt before a channel */
				ast_mutex_unlock(&p->lock);
				ast_mutex_unlock(&iflock);
				goto retry;
			}
			if (DEBUG_ATLEAST(3))
				ast_verbose("Softhanging up on %s\n", ast_channel_name(p->owner));
			ast_softhangup_nolock(p->owner, AST_SOFTHANGUP_EXPLICIT);
			p->restartpending = 1;
			num_restart_pending++;
			ast_channel_unlock(p->owner);
		}
		ast_mutex_unlock(&p->lock);
	}
	ast_mutex_unlock(&iflock);
}

static int dahdi_restart(void)
{
#if defined(HAVE_PRI) || defined(HAVE_SS7)
	int i, j;
#endif	/* defined(HAVE_PRI) || defined(HAVE_SS7) */
	int cancel_code;
	struct dahdi_pvt *p;

	ast_mutex_lock(&restart_lock);
	ast_verb(1, "Destroying channels and reloading DAHDI configuration.\n");
	dahdi_softhangup_all();
	ast_verb(4, "Initial softhangup of all DAHDI channels complete.\n");
#ifdef HAVE_OPENR2
	dahdi_r2_destroy_links();
#endif

#if defined(HAVE_PRI)
	for (i = 0; i < NUM_SPANS; i++) {
		if (pris[i].pri.master && (pris[i].pri.master != AST_PTHREADT_NULL)) {
			cancel_code = pthread_cancel(pris[i].pri.master);
			pthread_kill(pris[i].pri.master, SIGURG);
			ast_debug(4, "Waiting to join thread of span %d with pid=%p, cancel_code=%d\n", i, (void *) pris[i].pri.master, cancel_code);
			pthread_join(pris[i].pri.master, NULL);
			ast_debug(4, "Joined thread of span %d\n", i);
		}
	}
#endif

#if defined(HAVE_SS7)
	for (i = 0; i < NUM_SPANS; i++) {
		if (linksets[i].ss7.master && (linksets[i].ss7.master != AST_PTHREADT_NULL)) {
			cancel_code = pthread_cancel(linksets[i].ss7.master);
			pthread_kill(linksets[i].ss7.master, SIGURG);
			ast_debug(4, "Waiting to join thread of span %d with pid=%p, cancel_code=%d\n", i, (void *) linksets[i].ss7.master, cancel_code);
			pthread_join(linksets[i].ss7.master, NULL);
			ast_debug(4, "Joined thread of span %d\n", i);
		}
	}
#endif	/* defined(HAVE_SS7) */

	ast_mutex_lock(&monlock);
	if (monitor_thread && (monitor_thread != AST_PTHREADT_STOP) && (monitor_thread != AST_PTHREADT_NULL)) {
		cancel_code = pthread_cancel(monitor_thread);
		pthread_kill(monitor_thread, SIGURG);
		ast_debug(4, "Waiting to join monitor thread with pid=%p, cancel_code=%d\n", (void *) monitor_thread, cancel_code);
		pthread_join(monitor_thread, NULL);
		ast_debug(4, "Joined monitor thread\n");
	}
	monitor_thread = AST_PTHREADT_NULL; /* prepare to restart thread in setup_dahdi once channels are reconfigured */

	ast_mutex_lock(&ss_thread_lock);
	while (ss_thread_count > 0) { /* let ss_threads finish and run dahdi_hangup before dahvi_pvts are destroyed */
		int x = DAHDI_FLASH;
		ast_debug(3, "Waiting on %d analog_ss_thread(s) to finish\n", ss_thread_count);

		ast_mutex_lock(&iflock);
		for (p = iflist; p; p = p->next) {
			if (p->owner) {
				/* important to create an event for dahdi_wait_event to register so that all analog_ss_threads terminate */
				ioctl(p->subs[SUB_REAL].dfd, DAHDI_HOOK, &x);
			}
		}
		ast_mutex_unlock(&iflock);
		ast_cond_wait(&ss_thread_complete, &ss_thread_lock);
	}

	/* ensure any created channels before monitor threads were stopped are hungup */
	dahdi_softhangup_all();
	ast_verb(4, "Final softhangup of all DAHDI channels complete.\n");
	destroy_all_channels();
	memset(round_robin, 0, sizeof(round_robin));
	ast_debug(1, "Channels destroyed. Now re-reading config. %d active channels remaining.\n", ast_active_channels());

	ast_mutex_unlock(&monlock);

#ifdef HAVE_PRI
	for (i = 0; i < NUM_SPANS; i++) {
		for (j = 0; j < SIG_PRI_NUM_DCHANS; j++)
			dahdi_close_pri_fd(&(pris[i]), j);
	}

	memset(pris, 0, sizeof(pris));
	for (i = 0; i < NUM_SPANS; i++) {
		sig_pri_init_pri(&pris[i].pri);
	}
	pri_set_error(dahdi_pri_error);
	pri_set_message(dahdi_pri_message);
#endif
#if defined(HAVE_SS7)
	for (i = 0; i < NUM_SPANS; i++) {
		for (j = 0; j < SIG_SS7_NUM_DCHANS; j++)
			dahdi_close_ss7_fd(&(linksets[i]), j);
	}

	memset(linksets, 0, sizeof(linksets));
	for (i = 0; i < NUM_SPANS; i++) {
		sig_ss7_init_linkset(&linksets[i].ss7);
	}
	ss7_set_error(dahdi_ss7_error);
	ss7_set_message(dahdi_ss7_message);
	ss7_set_hangup(sig_ss7_cb_hangup);
	ss7_set_notinservice(sig_ss7_cb_notinservice);
	ss7_set_call_null(sig_ss7_cb_call_null);
#endif	/* defined(HAVE_SS7) */

	if (setup_dahdi(2) != 0) {
		ast_log(LOG_WARNING, "Reload channels from dahdi config failed!\n");
		ast_mutex_unlock(&ss_thread_lock);
		return 1;
	}
	ast_mutex_unlock(&ss_thread_lock);
	ast_mutex_unlock(&restart_lock);
	return 0;
}

static char *dahdi_restart_cmd(struct ast_cli_entry *e, int cmd, struct ast_cli_args *a)
{
	switch (cmd) {
	case CLI_INIT:
		e->command = "dahdi restart";
		e->usage =
			"Usage: dahdi restart\n"
			"	Restarts the DAHDI channels: destroys them all and then\n"
			"	re-reads them from chan_dahdi.conf.\n"
			"	Note that this will STOP any running CALL on DAHDI channels.\n"
			"";
		return NULL;
	case CLI_GENERATE:
		return NULL;
	}
	if (a->argc != 2)
		return CLI_SHOWUSAGE;

	if (dahdi_restart() != 0)
		return CLI_FAILURE;
	return CLI_SUCCESS;
}

static int action_dahdirestart(struct mansession *s, const struct message *m)
{
	if (dahdi_restart() != 0) {
		astman_send_error(s, m, "Failed rereading DAHDI configuration");
		return 1;
	}
	astman_send_ack(s, m, "DAHDIRestart: Success");
	return 0;
}

static char *dahdi_show_channels(struct ast_cli_entry *e, int cmd, struct ast_cli_args *a)
{
#define FORMAT "%7s %4d %-20.20s %-10.10s %-15.15s %-8.8s %-20.20s %-10.10s %-10.10s %-12.12s %-32.32s\n"
#define FORMAT2 "%7s %4s %-20.20s %-10.10s %-15.15s %-8.8s %-20.20s %-10.10s %-10.10s %-12.12s %-32.32s\n"
	ast_group_t targetnum = 0;
	int filtertype = 0;
	struct dahdi_pvt *tmp = NULL;
	char tmps[20];
	char blockstr[20];

	switch (cmd) {
	case CLI_INIT:
		e->command = "dahdi show channels [group|context]";
		e->usage =
			"Usage: dahdi show channels [ group <group> | context <context> ]\n"
			"	Shows a list of available channels with optional filtering\n"
			"	<group> must be a number between 0 and 63\n";
		return NULL;
	case CLI_GENERATE:
		return NULL;
	}

	/* syntax: dahdi show channels [ group <group> | context <context> ] */

	if (!((a->argc == 3) || (a->argc == 5))) {
		return CLI_SHOWUSAGE;
	}

	if (a->argc == 5) {
		if (!strcasecmp(a->argv[3], "group")) {
			targetnum = atoi(a->argv[4]);
			if (63 < targetnum) {
				return CLI_SHOWUSAGE;
			}
			targetnum = ((ast_group_t) 1) << targetnum;
			filtertype = 1;
		} else if (!strcasecmp(a->argv[3], "context")) {
			filtertype = 2;
		}
	}

	ast_cli(a->fd, FORMAT2, "Chan", "Span", "Signalling", "Extension", "Context", "Language", "MOH Interpret", "Blocked", "In Service", "Alarms", "Description");
	ast_mutex_lock(&iflock);
	for (tmp = iflist; tmp; tmp = tmp->next) {
		int alm = 0;
		if (filtertype) {
			switch(filtertype) {
			case 1: /* dahdi show channels group <group> */
				if (!(tmp->group & targetnum)) {
					continue;
				}
				break;
			case 2: /* dahdi show channels context <context> */
				if (strcasecmp(tmp->context, a->argv[4])) {
					continue;
				}
				break;
			default:
				break;
			}
		}
		if (tmp->channel > 0) {
			snprintf(tmps, sizeof(tmps), "%d", tmp->channel);
			alm = get_alarms(tmp);
		} else {
			ast_copy_string(tmps, "pseudo", sizeof(tmps));
		}

		blockstr[0] = tmp->locallyblocked ? 'L' : ' ';
		blockstr[1] = tmp->remotelyblocked ? 'R' : ' ';
		blockstr[2] = '\0';

		ast_cli(a->fd, FORMAT, tmps, tmp->span, sig2str(tmp->sig), tmp->exten, tmp->context, tmp->language, tmp->mohinterpret, blockstr, tmp->inservice ? "Yes" : "No",
			alarm2str(alm), tmp->description);
	}
	ast_mutex_unlock(&iflock);
	return CLI_SUCCESS;
#undef FORMAT
#undef FORMAT2
}

static char *dahdi_show_channel(struct ast_cli_entry *e, int cmd, struct ast_cli_args *a)
{
	int channel;
	struct dahdi_pvt *tmp = NULL;
	struct dahdi_confinfo ci;
	struct dahdi_params ps;
	int x;
	char hwrxgain[15];
	char hwtxgain[15];

	switch (cmd) {
	case CLI_INIT:
		e->command = "dahdi show channel";
		e->usage =
			"Usage: dahdi show channel <chan num>\n"
			"	Detailed information about a given channel\n";
		return NULL;
	case CLI_GENERATE:
		return NULL;
	}

	if (a->argc != 4)
		return CLI_SHOWUSAGE;

	channel = atoi(a->argv[3]);

	ast_mutex_lock(&iflock);
	for (tmp = iflist; tmp; tmp = tmp->next) {
		if (tmp->channel == channel) {
			ast_cli(a->fd, "Channel: %d\n", tmp->channel);
			ast_cli(a->fd, "Description: %s\n", tmp->description);
			ast_cli(a->fd, "File Descriptor: %d\n", tmp->subs[SUB_REAL].dfd);
			ast_cli(a->fd, "Span: %d\n", tmp->span);
			ast_cli(a->fd, "Extension: %s\n", tmp->exten);
			ast_cli(a->fd, "Dialing: %s\n", tmp->dialing ? "yes" : "no");
			ast_cli(a->fd, "Context: %s\n", tmp->context);
			ast_cli(a->fd, "Caller ID: %s\n", tmp->cid_num);
			ast_cli(a->fd, "Calling TON: %d\n", tmp->cid_ton);
#if defined(HAVE_PRI)
#if defined(HAVE_PRI_SUBADDR)
			ast_cli(a->fd, "Caller ID subaddress: %s\n", tmp->cid_subaddr);
#endif	/* defined(HAVE_PRI_SUBADDR) */
#endif	/* defined(HAVE_PRI) */
			ast_cli(a->fd, "Caller ID name: %s\n", tmp->cid_name);
			ast_cli(a->fd, "Mailbox: %s\n", S_OR(tmp->mailbox, "none"));
			if (tmp->vars) {
				struct ast_variable *v;
				ast_cli(a->fd, "Variables:\n");
				for (v = tmp->vars ; v ; v = v->next)
					ast_cli(a->fd, "       %s = %s\n", v->name, v->value);
			}
			ast_cli(a->fd, "Destroy: %d\n", tmp->destroy);
			ast_cli(a->fd, "InAlarm: %d\n", tmp->inalarm);
			ast_cli(a->fd, "Signalling Type: %s\n", sig2str(tmp->sig));
			ast_cli(a->fd, "Radio: %d\n", tmp->radio);
			ast_cli(a->fd, "Owner: %s\n", tmp->owner ? ast_channel_name(tmp->owner) : "<None>");
			ast_cli(a->fd, "Real: %s%s%s\n", tmp->subs[SUB_REAL].owner ? ast_channel_name(tmp->subs[SUB_REAL].owner) : "<None>", tmp->subs[SUB_REAL].inthreeway ? " (Confed)" : "", tmp->subs[SUB_REAL].linear ? " (Linear)" : "");
			ast_cli(a->fd, "Callwait: %s%s%s\n", tmp->subs[SUB_CALLWAIT].owner ? ast_channel_name(tmp->subs[SUB_CALLWAIT].owner) : "<None>", tmp->subs[SUB_CALLWAIT].inthreeway ? " (Confed)" : "", tmp->subs[SUB_CALLWAIT].linear ? " (Linear)" : "");
			ast_cli(a->fd, "Threeway: %s%s%s\n", tmp->subs[SUB_THREEWAY].owner ? ast_channel_name(tmp->subs[SUB_THREEWAY].owner) : "<None>", tmp->subs[SUB_THREEWAY].inthreeway ? " (Confed)" : "", tmp->subs[SUB_THREEWAY].linear ? " (Linear)" : "");
			ast_cli(a->fd, "Confno: %d\n", tmp->confno);
			ast_cli(a->fd, "Propagated Conference: %d\n", tmp->propconfno);
			ast_cli(a->fd, "Real in conference: %d\n", tmp->inconference);
			ast_cli(a->fd, "DSP: %s\n", tmp->dsp ? "yes" : "no");
			ast_cli(a->fd, "Busy Detection: %s\n", tmp->busydetect ? "yes" : "no");
			if (tmp->busydetect) {
#if defined(BUSYDETECT_TONEONLY)
				ast_cli(a->fd, "    Busy Detector Helper: BUSYDETECT_TONEONLY\n");
#elif defined(BUSYDETECT_COMPARE_TONE_AND_SILENCE)
				ast_cli(a->fd, "    Busy Detector Helper: BUSYDETECT_COMPARE_TONE_AND_SILENCE\n");
#endif
#ifdef BUSYDETECT_DEBUG
				ast_cli(a->fd, "    Busy Detector Debug: Enabled\n");
#endif
				ast_cli(a->fd, "    Busy Count: %d\n", tmp->busycount);
				ast_cli(a->fd, "    Busy Pattern: %d,%d,%d,%d\n", tmp->busy_cadence.pattern[0], tmp->busy_cadence.pattern[1], (tmp->busy_cadence.length == 4) ? tmp->busy_cadence.pattern[2] : 0, (tmp->busy_cadence.length == 4) ? tmp->busy_cadence.pattern[3] : 0);
			}
			ast_cli(a->fd, "TDD: %s\n", tmp->tdd ? "yes" : "no");
			ast_cli(a->fd, "Relax DTMF: %s\n", tmp->dtmfrelax ? "yes" : "no");
			ast_cli(a->fd, "Dialing/CallwaitCAS: %d/%d\n", tmp->dialing, tmp->callwaitcas);
			ast_cli(a->fd, "Default law: %s\n", tmp->law_default == DAHDI_LAW_MULAW ? "ulaw" : tmp->law_default == DAHDI_LAW_ALAW ? "alaw" : "unknown");
			ast_cli(a->fd, "Fax Handled: %s\n", tmp->faxhandled ? "yes" : "no");
			ast_cli(a->fd, "Pulse phone: %s\n", tmp->pulsedial ? "yes" : "no");
			if (tmp->hwrxgain_enabled) {
				snprintf(hwrxgain, sizeof(hwrxgain), "%.1f", tmp->hwrxgain);
			} else {
				ast_copy_string(hwrxgain, "Disabled", sizeof(hwrxgain));
			}
			if (tmp->hwtxgain_enabled) {
				snprintf(hwtxgain, sizeof(hwtxgain), "%.1f", tmp->hwtxgain);
			} else {
				ast_copy_string(hwtxgain, "Disabled", sizeof(hwtxgain));
			}
			ast_cli(a->fd, "HW Gains (RX/TX): %s/%s\n", hwrxgain, hwtxgain);
			ast_cli(a->fd, "SW Gains (RX/TX): %.2f/%.2f\n", tmp->rxgain, tmp->txgain);
			ast_cli(a->fd, "Dynamic Range Compression (RX/TX): %.2f/%.2f\n", tmp->rxdrc, tmp->txdrc);
			ast_cli(a->fd, "DND: %s\n", dahdi_dnd(tmp, -1) ? "yes" : "no");
			ast_cli(a->fd, "Echo Cancellation:\n");

			if (tmp->echocancel.head.tap_length) {
				ast_cli(a->fd, "\t%u taps\n", tmp->echocancel.head.tap_length);
				for (x = 0; x < tmp->echocancel.head.param_count; x++) {
					ast_cli(a->fd, "\t\t%s: %dd\n", tmp->echocancel.params[x].name, tmp->echocancel.params[x].value);
				}
				ast_cli(a->fd, "\t%scurrently %s\n", tmp->echocanbridged ? "" : "(unless TDM bridged) ", tmp->echocanon ? "ON" : "OFF");
			} else {
				ast_cli(a->fd, "\tnone\n");
			}
			ast_cli(a->fd, "Wait for dialtone: %dms\n", tmp->waitfordialtone);
			if (tmp->master)
				ast_cli(a->fd, "Master Channel: %d\n", tmp->master->channel);
			for (x = 0; x < MAX_SLAVES; x++) {
				if (tmp->slaves[x])
					ast_cli(a->fd, "Slave Channel: %d\n", tmp->slaves[x]->channel);
			}
#ifdef HAVE_OPENR2
			if (tmp->mfcr2) {
				char calldir[OR2_MAX_PATH];
				openr2_context_t *r2context = openr2_chan_get_context(tmp->r2chan);
				openr2_variant_t r2variant = openr2_context_get_variant(r2context);
				ast_cli(a->fd, "MFC/R2 MF State: %s\n", openr2_chan_get_mf_state_string(tmp->r2chan));
				ast_cli(a->fd, "MFC/R2 MF Group: %s\n", openr2_chan_get_mf_group_string(tmp->r2chan));
				ast_cli(a->fd, "MFC/R2 State: %s\n", openr2_chan_get_r2_state_string(tmp->r2chan));
				ast_cli(a->fd, "MFC/R2 Call State: %s\n", openr2_chan_get_call_state_string(tmp->r2chan));
				ast_cli(a->fd, "MFC/R2 Call Files Enabled: %s\n", openr2_chan_get_call_files_enabled(tmp->r2chan) ? "Yes" : "No");
				ast_cli(a->fd, "MFC/R2 Variant: %s\n", openr2_proto_get_variant_string(r2variant));
				ast_cli(a->fd, "MFC/R2 Max ANI: %d\n", openr2_context_get_max_ani(r2context));
				ast_cli(a->fd, "MFC/R2 Max DNIS: %d\n", openr2_context_get_max_dnis(r2context));
#if defined(OR2_LIB_INTERFACE) && OR2_LIB_INTERFACE > 2
				ast_cli(a->fd, "MFC/R2 DTMF Dialing: %s\n", openr2_context_get_dtmf_dialing(r2context, NULL, NULL) ? "Yes" : "No");
				ast_cli(a->fd, "MFC/R2 DTMF Detection: %s\n", openr2_context_get_dtmf_detection(r2context) ? "Yes" : "No");
#endif
				ast_cli(a->fd, "MFC/R2 Get ANI First: %s\n", openr2_context_get_ani_first(r2context) ? "Yes" : "No");
#if defined(OR2_LIB_INTERFACE) && OR2_LIB_INTERFACE > 1
				ast_cli(a->fd, "MFC/R2 Skip Category Request: %s\n", openr2_context_get_skip_category_request(r2context) ? "Yes" : "No");
#endif
				ast_cli(a->fd, "MFC/R2 Immediate Accept: %s\n", openr2_context_get_immediate_accept(r2context) ? "Yes" : "No");
				ast_cli(a->fd, "MFC/R2 Accept on Offer: %s\n", tmp->mfcr2_accept_on_offer ? "Yes" : "No");
				ast_cli(a->fd, "MFC/R2 Charge Calls: %s\n", tmp->mfcr2_charge_calls ? "Yes" : "No");
				ast_cli(a->fd, "MFC/R2 Allow Collect Calls: %s\n", tmp->mfcr2_allow_collect_calls ? "Yes" : "No");
				ast_cli(a->fd, "MFC/R2 Forced Release: %s\n", tmp->mfcr2_forced_release ? "Yes" : "No");
				ast_cli(a->fd, "MFC/R2 MF Back Timeout: %dms\n", openr2_context_get_mf_back_timeout(r2context));
				ast_cli(a->fd, "MFC/R2 R2 Metering Pulse Timeout: %dms\n", openr2_context_get_metering_pulse_timeout(r2context));
				ast_cli(a->fd, "MFC/R2 Rx CAS: %s\n", openr2_chan_get_rx_cas_string(tmp->r2chan));
				ast_cli(a->fd, "MFC/R2 Tx CAS: %s\n", openr2_chan_get_tx_cas_string(tmp->r2chan));
				ast_cli(a->fd, "MFC/R2 MF Tx Signal: %d\n", openr2_chan_get_tx_mf_signal(tmp->r2chan));
				ast_cli(a->fd, "MFC/R2 MF Rx Signal: %d\n", openr2_chan_get_rx_mf_signal(tmp->r2chan));
				ast_cli(a->fd, "MFC/R2 Call Files Directory: %s\n", openr2_context_get_log_directory(r2context, calldir, sizeof(calldir)));
			}
#endif
#if defined(HAVE_SS7)
			if (tmp->ss7) {
				struct sig_ss7_chan *chan = tmp->sig_pvt;

				ast_cli(a->fd, "CIC: %d\n", chan->cic);
			}
#endif	/* defined(HAVE_SS7) */
#ifdef HAVE_PRI
			if (tmp->pri) {
				struct sig_pri_chan *chan = tmp->sig_pvt;

				ast_cli(a->fd, "PRI Flags: ");
				if (chan->resetting != SIG_PRI_RESET_IDLE) {
					ast_cli(a->fd, "Resetting=%u ", chan->resetting);
				}
				if (chan->call)
					ast_cli(a->fd, "Call ");
				if (chan->allocated) {
					ast_cli(a->fd, "Allocated ");
				}
				ast_cli(a->fd, "\n");
				if (tmp->logicalspan)
					ast_cli(a->fd, "PRI Logical Span: %d\n", tmp->logicalspan);
				else
					ast_cli(a->fd, "PRI Logical Span: Implicit\n");
			}
#endif
			memset(&ci, 0, sizeof(ci));
			ps.channo = tmp->channel;
			if (tmp->subs[SUB_REAL].dfd > -1) {
				memset(&ci, 0, sizeof(ci));
				if (!ioctl(tmp->subs[SUB_REAL].dfd, DAHDI_GETCONF, &ci)) {
					ast_cli(a->fd, "Actual Confinfo: Num/%d, Mode/0x%04x\n", ci.confno, (unsigned)ci.confmode);
				}
				if (!ioctl(tmp->subs[SUB_REAL].dfd, DAHDI_GETCONFMUTE, &x)) {
					ast_cli(a->fd, "Actual Confmute: %s\n", x ? "Yes" : "No");
				}
				memset(&ps, 0, sizeof(ps));
				if (ioctl(tmp->subs[SUB_REAL].dfd, DAHDI_GET_PARAMS, &ps) < 0) {
					ast_log(LOG_WARNING, "Failed to get parameters on channel %d: %s\n", tmp->channel, strerror(errno));
				} else {
					ast_cli(a->fd, "Hookstate (FXS only): %s\n", ps.rxisoffhook ? "Offhook" : "Onhook");
				}
			}
			ast_mutex_unlock(&iflock);
			return CLI_SUCCESS;
		}
	}
	ast_mutex_unlock(&iflock);

	ast_cli(a->fd, "Unable to find given channel %d\n", channel);
	return CLI_FAILURE;
}

static char *handle_dahdi_show_cadences(struct ast_cli_entry *e, int cmd, struct ast_cli_args *a)
{
	int i, j;
	switch (cmd) {
	case CLI_INIT:
		e->command = "dahdi show cadences";
		e->usage =
			"Usage: dahdi show cadences\n"
			"       Shows all cadences currently defined\n";
		return NULL;
	case CLI_GENERATE:
		return NULL;
	}
	for (i = 0; i < num_cadence; i++) {
		char output[1024];
		char tmp[16], tmp2[64];
		snprintf(tmp, sizeof(tmp), "r%d: ", i + 1);
		term_color(output, tmp, COLOR_GREEN, COLOR_BLACK, sizeof(output));

		for (j = 0; j < 16; j++) {
			if (cadences[i].ringcadence[j] == 0)
				break;
			snprintf(tmp, sizeof(tmp), "%d", cadences[i].ringcadence[j]);
			if (cidrings[i] * 2 - 1 == j)
				term_color(tmp2, tmp, COLOR_MAGENTA, COLOR_BLACK, sizeof(tmp2) - 1);
			else
				term_color(tmp2, tmp, COLOR_GREEN, COLOR_BLACK, sizeof(tmp2) - 1);
			if (j != 0)
				strncat(output, ",", sizeof(output) - strlen(output) - 1);
			strncat(output, tmp2, sizeof(output) - strlen(output) - 1);
		}
		ast_cli(a->fd,"%s\n",output);
	}
	return CLI_SUCCESS;
}

static void build_alarm_info(char *restrict alarmstr, struct dahdi_spaninfo *spaninfo)
{
	alarmstr[0] = '\0';
	if (spaninfo->alarms > 0) {
		if (spaninfo->alarms & DAHDI_ALARM_BLUE) {
			strcat(alarmstr, "BLU/");
		}
		if (spaninfo->alarms & DAHDI_ALARM_YELLOW) {
			strcat(alarmstr, "YEL/");
		}
		if (spaninfo->alarms & DAHDI_ALARM_RED) {
			strcat(alarmstr, "RED/");
		}
		if (spaninfo->alarms & DAHDI_ALARM_LOOPBACK) {
			strcat(alarmstr, "LB/");
		}
		if (spaninfo->alarms & DAHDI_ALARM_RECOVER) {
			strcat(alarmstr, "REC/");
		}
		if (spaninfo->alarms & DAHDI_ALARM_NOTOPEN) {
			strcat(alarmstr, "NOP/");
		}
		if (!strlen(alarmstr)) {
			strcat(alarmstr, "UUU/");
		}
		if (strlen(alarmstr)) {
			/* Strip trailing / */
			alarmstr[strlen(alarmstr) - 1] = '\0';
		}
	} else {
		if (spaninfo->numchans) {
			strcpy(alarmstr, "OK");
		} else {
			strcpy(alarmstr, "UNCONFIGURED");
		}
	}
}

/* Based on irqmiss.c */
static char *dahdi_show_status(struct ast_cli_entry *e, int cmd, struct ast_cli_args *a)
{
	#define FORMAT "%4d %-40.40s %-7.7s %-6d %-6d %-6d %-3.3s %-4.4s %-8.8s %s\n"
	#define FORMAT2 "%4s %-40.40s %-7.7s %-6.6s %-6.6s %-6.6s %-3.3s %-4.4s %-8.8s %s\n"
	int span;
	int res;
	char alarmstr[50];

	int ctl;
	struct dahdi_spaninfo s;

	switch (cmd) {
	case CLI_INIT:
		e->command = "dahdi show status";
		e->usage =
			"Usage: dahdi show status\n"
			"       Shows a list of DAHDI cards with status\n";
		return NULL;
	case CLI_GENERATE:
		return NULL;
	}
	ctl = open("/dev/dahdi/ctl", O_RDWR);
	if (ctl < 0) {
		ast_cli(a->fd, "No DAHDI found. Unable to open /dev/dahdi/ctl: %s\n", strerror(errno));
		return CLI_FAILURE;
	}
	ast_cli(a->fd, FORMAT2, "Span", "Description", "Alarms", "IRQ", "bpviol", "CRC", "Framing", "Coding", "Options", "LBO");

	for (span = 1; span < DAHDI_MAX_SPANS; ++span) {
		s.spanno = span;
		res = ioctl(ctl, DAHDI_SPANSTAT, &s);
		if (res) {
			continue;
		}
		build_alarm_info(alarmstr, &s);
		ast_cli(a->fd, FORMAT, span, s.desc, alarmstr, s.irqmisses, s.bpvcount, s.crc4count,
			s.lineconfig & DAHDI_CONFIG_D4 ? "D4" :
			s.lineconfig & DAHDI_CONFIG_ESF ? "ESF" :
			s.lineconfig & DAHDI_CONFIG_CCS ? "CCS" :
			"CAS",
			s.lineconfig & DAHDI_CONFIG_B8ZS ? "B8ZS" :
			s.lineconfig & DAHDI_CONFIG_HDB3 ? "HDB3" :
			s.lineconfig & DAHDI_CONFIG_AMI ? "AMI" :
			"Unknown",
			s.lineconfig & DAHDI_CONFIG_CRC4 ?
				s.lineconfig & DAHDI_CONFIG_NOTOPEN ? "CRC4/YEL" : "CRC4" :
				s.lineconfig & DAHDI_CONFIG_NOTOPEN ? "YEL" : "",
			lbostr[s.lbo]
			);
	}
	close(ctl);

	return CLI_SUCCESS;
#undef FORMAT
#undef FORMAT2
}

static char *dahdi_show_version(struct ast_cli_entry *e, int cmd, struct ast_cli_args *a)
{
	int pseudo_fd = -1;
	struct dahdi_versioninfo vi;

	switch (cmd) {
	case CLI_INIT:
		e->command = "dahdi show version";
		e->usage =
			"Usage: dahdi show version\n"
			"       Shows the DAHDI version in use\n";
		return NULL;
	case CLI_GENERATE:
		return NULL;
	}
	if ((pseudo_fd = open("/dev/dahdi/ctl", O_RDONLY)) < 0) {
		ast_cli(a->fd, "Failed to open control file to get version.\n");
		return CLI_SUCCESS;
	}

	strcpy(vi.version, "Unknown");
	strcpy(vi.echo_canceller, "Unknown");

	if (ioctl(pseudo_fd, DAHDI_GETVERSION, &vi))
		ast_cli(a->fd, "Failed to get DAHDI version: %s\n", strerror(errno));
	else
		ast_cli(a->fd, "DAHDI Version: %s Echo Canceller: %s\n", vi.version, vi.echo_canceller);

	close(pseudo_fd);

	return CLI_SUCCESS;
}

static char *dahdi_set_hwgain(struct ast_cli_entry *e, int cmd, struct ast_cli_args *a)
{
	int channel;
	float gain;
	int tx;
	struct dahdi_pvt *tmp = NULL;

	switch (cmd) {
	case CLI_INIT:
		e->command = "dahdi set hwgain {rx|tx}";
		e->usage =
			"Usage: dahdi set hwgain <rx|tx> <chan#> <gain>\n"
			"   Sets the hardware gain on a given channel and overrides the\n"
			"   value provided at module loadtime.  Changes take effect\n"
			"   immediately whether the channel is in use or not.\n"
			"\n"
			"   <rx|tx> which direction do you want to change (relative to our module)\n"
			"   <chan num> is the channel number relative to the device\n"
			"   <gain> is the gain in dB (e.g. -3.5 for -3.5dB)\n"
			"\n"
			"   Please note:\n"
			"   * hwgain is only supportable by hardware with analog ports because\n"
			"     hwgain works on the analog side of an analog-digital conversion.\n";
		return NULL;
	case CLI_GENERATE:
		return NULL;
	}

	if (a->argc != 6)
		return CLI_SHOWUSAGE;

	if (!strcasecmp("rx", a->argv[3]))
		tx = 0; /* rx */
	else if (!strcasecmp("tx", a->argv[3]))
		tx = 1; /* tx */
	else
		return CLI_SHOWUSAGE;

	channel = atoi(a->argv[4]);
	gain = atof(a->argv[5]);

	ast_mutex_lock(&iflock);

	for (tmp = iflist; tmp; tmp = tmp->next) {

		if (tmp->channel != channel)
			continue;

		if (tmp->subs[SUB_REAL].dfd == -1)
			break;

		if (set_hwgain(tmp->subs[SUB_REAL].dfd, gain, tx)) {
			ast_cli(a->fd, "Unable to set the hardware gain for channel %d: %s\n", channel, strerror(errno));
			ast_mutex_unlock(&iflock);
			return CLI_FAILURE;
		}
		ast_cli(a->fd, "Hardware %s gain set to %.1f dB on channel %d.\n",
			tx ? "tx" : "rx", gain, channel);

		if (tx) {
			tmp->hwtxgain_enabled = 1;
			tmp->hwtxgain = gain;
		} else {
			tmp->hwrxgain_enabled = 1;
			tmp->hwrxgain = gain;
		}
		break;
	}

	ast_mutex_unlock(&iflock);

	if (tmp)
		return CLI_SUCCESS;

	ast_cli(a->fd, "Unable to find given channel %d\n", channel);
	return CLI_FAILURE;

}

static char *dahdi_set_swgain(struct ast_cli_entry *e, int cmd, struct ast_cli_args *a)
{
	int channel;
	float gain;
	int tx;
	int res;
	struct dahdi_pvt *tmp = NULL;

	switch (cmd) {
	case CLI_INIT:
		e->command = "dahdi set swgain {rx|tx}";
		e->usage =
			"Usage: dahdi set swgain <rx|tx> <chan#> <gain>\n"
			"   Sets the software gain on a given channel and overrides the\n"
			"   value provided at module loadtime.  Changes take effect\n"
			"   immediately whether the channel is in use or not.\n"
			"\n"
			"   <rx|tx> which direction do you want to change (relative to our module)\n"
			"   <chan num> is the channel number relative to the device\n"
			"   <gain> is the gain in dB (e.g. -3.5 for -3.5dB)\n";
		return NULL;
	case CLI_GENERATE:
		return NULL;
	}

	if (a->argc != 6)
		return CLI_SHOWUSAGE;

	if (!strcasecmp("rx", a->argv[3]))
		tx = 0; /* rx */
	else if (!strcasecmp("tx", a->argv[3]))
		tx = 1; /* tx */
	else
		return CLI_SHOWUSAGE;

	channel = atoi(a->argv[4]);
	gain = atof(a->argv[5]);

	ast_mutex_lock(&iflock);
	for (tmp = iflist; tmp; tmp = tmp->next) {

		if (tmp->channel != channel)
			continue;

		if (tmp->subs[SUB_REAL].dfd == -1)
			break;

		if (tx)
			res = set_actual_txgain(tmp->subs[SUB_REAL].dfd, gain, tmp->txdrc, tmp->law);
		else
			res = set_actual_rxgain(tmp->subs[SUB_REAL].dfd, gain, tmp->rxdrc, tmp->law);

		if (res) {
			ast_cli(a->fd, "Unable to set the software gain for channel %d\n", channel);
			ast_mutex_unlock(&iflock);
			return CLI_FAILURE;
		}

		ast_cli(a->fd, "Software %s gain set to %.2f dB on channel %d.\n",
			tx ? "tx" : "rx", gain, channel);

		if (tx) {
			tmp->txgain = gain;
		} else {
			tmp->rxgain = gain;
		}
		break;
	}
	ast_mutex_unlock(&iflock);

	if (tmp)
		return CLI_SUCCESS;

	ast_cli(a->fd, "Unable to find given channel %d\n", channel);
	return CLI_FAILURE;

}

static char *dahdi_set_dnd(struct ast_cli_entry *e, int cmd, struct ast_cli_args *a)
{
	int channel;
	int on;
	struct dahdi_pvt *dahdi_chan = NULL;

	switch (cmd) {
	case CLI_INIT:
		e->command = "dahdi set dnd";
		e->usage =
			"Usage: dahdi set dnd <chan#> <on|off>\n"
			"	Sets/resets DND (Do Not Disturb) mode on a channel.\n"
			"	Changes take effect immediately.\n"
			"	<chan num> is the channel number\n"
			" 	<on|off> Enable or disable DND mode?\n"
			;
		return NULL;
	case CLI_GENERATE:
		return NULL;
	}

	if (a->argc != 5)
		return CLI_SHOWUSAGE;

	if ((channel = atoi(a->argv[3])) <= 0) {
		ast_cli(a->fd, "Expected channel number, got '%s'\n", a->argv[3]);
		return CLI_SHOWUSAGE;
	}

	if (ast_true(a->argv[4]))
		on = 1;
	else if (ast_false(a->argv[4]))
		on = 0;
	else {
		ast_cli(a->fd, "Expected 'on' or 'off', got '%s'\n", a->argv[4]);
		return CLI_SHOWUSAGE;
	}

	ast_mutex_lock(&iflock);
	for (dahdi_chan = iflist; dahdi_chan; dahdi_chan = dahdi_chan->next) {
		if (dahdi_chan->channel != channel)
			continue;

		/* Found the channel. Actually set it */
		dahdi_dnd(dahdi_chan, on);
		break;
	}
	ast_mutex_unlock(&iflock);

	if (!dahdi_chan) {
		ast_cli(a->fd, "Unable to find given channel %d\n", channel);
		return CLI_FAILURE;
	}

	return CLI_SUCCESS;
}

static char *dahdi_set_mwi(struct ast_cli_entry *e, int cmd, struct ast_cli_args *a)
{
	int channel;
	int on;
	int override = 1;
	struct dahdi_pvt *dahdi_chan = NULL;

	switch (cmd) {
	case CLI_INIT:
		e->command = "dahdi set mwi";
		e->usage =
			"Usage: dahdi set mwi <chan#> <on|off|reset>\n"
			"	Sets/unsets MWI (Message Waiting Indicator) manually on a channel.\n"
			"   This may be used regardless of whether the channel is assigned any mailboxes.\n"
			"   When active, this setting will override the voicemail status to set MWI.\n"
			"   Once cleared, the voicemail status will resume control of MWI.\n"
			"	Changes are queued for when the channel is idle and persist until cleared.\n"
			"	<chan num> is the channel number\n"
			" 	<on|off|reset> Enable, disable, or reset Message Waiting Indicator override?\n"
			;
		return NULL;
	case CLI_GENERATE:
		return NULL;
	}

	if (a->argc != 5)
		return CLI_SHOWUSAGE;

	if ((channel = atoi(a->argv[3])) <= 0) {
		ast_cli(a->fd, "Expected channel number, got '%s'\n", a->argv[3]);
		return CLI_SHOWUSAGE;
	}

	if (ast_true(a->argv[4])) {
		on = 1;
	} else if (ast_false(a->argv[4])) {
		on = 0;
	} else if (!strcmp(a->argv[4], "reset")) {
		override = 0;
	} else {
		ast_cli(a->fd, "Expected 'on' or 'off' or 'reset', got '%s'\n", a->argv[4]);
		return CLI_SHOWUSAGE;
	}

	ast_mutex_lock(&iflock);
	for (dahdi_chan = iflist; dahdi_chan; dahdi_chan = dahdi_chan->next) {
		if (dahdi_chan->channel != channel)
			continue;

		/* Found the channel. Actually set it */
		if (override) {
			dahdi_chan->mwioverride_disposition = on;
			ast_cli(a->fd, "MWI '%s' queued for channel %d\n", on ? "enable" : "disable", channel);
		}
		dahdi_chan->mwioverride_active = override;
		/* The do_monitor thread will take care of actually sending the MWI
		 * at an appropriate time for the channel. */
		break;
	}
	ast_mutex_unlock(&iflock);

	if (!dahdi_chan) {
		ast_cli(a->fd, "Unable to find given channel %d\n", channel);
		return CLI_FAILURE;
	}

	return CLI_SUCCESS;
}

static struct ast_cli_entry dahdi_cli[] = {
	AST_CLI_DEFINE(handle_dahdi_show_cadences, "List cadences"),
	AST_CLI_DEFINE(dahdi_show_channels, "Show active DAHDI channels"),
	AST_CLI_DEFINE(dahdi_show_channel, "Show information on a channel"),
	AST_CLI_DEFINE(dahdi_destroy_channels, "Destroy channels"),
	AST_CLI_DEFINE(dahdi_create_channels, "Create channels"),
	AST_CLI_DEFINE(dahdi_restart_cmd, "Fully restart DAHDI channels"),
	AST_CLI_DEFINE(dahdi_show_status, "Show all DAHDI cards status"),
	AST_CLI_DEFINE(dahdi_show_version, "Show the DAHDI version in use"),
	AST_CLI_DEFINE(dahdi_set_hwgain, "Set hardware gain on a channel"),
	AST_CLI_DEFINE(dahdi_set_swgain, "Set software gain on a channel"),
	AST_CLI_DEFINE(dahdi_set_dnd, "Sets/resets DND (Do Not Disturb) mode on a channel"),
	AST_CLI_DEFINE(dahdi_set_mwi, "Sets/unsets MWI (Message Waiting Indicator) manually on a channel"),
};

#define TRANSFER	0
#define HANGUP		1

static int dahdi_fake_event(struct dahdi_pvt *p, int mode)
{
	if (p) {
		switch (mode) {
		case TRANSFER:
			p->fake_event = DAHDI_EVENT_WINKFLASH;
			break;
		case HANGUP:
			p->fake_event = DAHDI_EVENT_ONHOOK;
			break;
		default:
			ast_log(LOG_WARNING, "I don't know how to handle transfer event with this: %d on channel %s\n",mode, ast_channel_name(p->owner));
		}
	}
	return 0;
}
static struct dahdi_pvt *find_channel(int channel)
{
	struct dahdi_pvt *p;

	ast_mutex_lock(&iflock);
	for (p = iflist; p; p = p->next) {
		if (p->channel == channel) {
			break;
		}
	}
	ast_mutex_unlock(&iflock);
	return p;
}

/*!
 * \internal
 * \brief Get private struct using given numeric channel string.
 *
 * \param channel Numeric channel number string get private struct.
 *
 * \retval pvt on success.
 * \retval NULL on error.
 */
static struct dahdi_pvt *find_channel_from_str(const char *channel)
{
	int chan_num;

	if (sscanf(channel, "%30d", &chan_num) != 1) {
		/* Not numeric string. */
		return NULL;
	}

	return find_channel(chan_num);
}

static int print_subchannel(struct dahdi_pvt *p, int subchan, char *buffer, size_t len)
{
	if (!p->subs[subchan].owner) {
		return -1;
	}
	ast_channel_lock(p->subs[subchan].owner);
	snprintf(buffer, len, "%s", ast_channel_name(p->subs[subchan].owner));
	ast_channel_unlock(p->subs[subchan].owner);
	return 0;
}

#define REQUIRE_FXO_SIG() \
	if (!(dahdip->sigtype & __DAHDI_SIG_FXO)) { \
		ast_log(LOG_WARNING, "DAHDI channel %d is not FXO signalled\n", p->channel); \
		return -1; \
	}

static int dahdichan_read_property(struct dahdi_pvt *p, struct dahdi_params *dahdip, const char *property, char *buffer, size_t len)
{
	struct analog_pvt *analog_p = p->sig_pvt;

	/* R/O properties */
	if (!strcasecmp(property, "owner")) {
		return print_subchannel(p, SUB_REAL, buffer, len);
	} else if (!strcasecmp(property, "callwait")) {
		return print_subchannel(p, SUB_CALLWAIT, buffer, len);
	} else if (!strcasecmp(property, "threeway")) {
		return print_subchannel(p, SUB_THREEWAY, buffer, len);
	/* R/W properties */
	} else if (!strcasecmp(property, "polarity")) {
		REQUIRE_FXO_SIG();
		snprintf(buffer, len, "%d", p->polarity);
	} else if (!strcasecmp(property, "dnd")) {
		REQUIRE_FXO_SIG();
		snprintf(buffer, len, "%d", analog_p->dnd);
	} else if (!strcasecmp(property, "callforward")) {
		REQUIRE_FXO_SIG();
		snprintf(buffer, len, "%s", analog_p->call_forward);
	} else if (!strcasecmp(property, "lastexten")) {
		REQUIRE_FXO_SIG();
		snprintf(buffer, len, "%s", analog_p->lastexten);
	} else {
		ast_log(LOG_ERROR, "Unknown DAHDI_CHANNEL property '%s'\n", property);
		return -1;
	}
	return 0;
}

static int dahdichan_write_property(struct dahdi_pvt *p, struct dahdi_params *dahdip, const char *property, const char *value)
{
	struct analog_pvt *analog_p = p->sig_pvt;

	/* We don't need to check ast_strlen_zero(value) because it's obviously not NULL.
	 * It may even be okay for it to be an empty string, but that's a per-setting thing. */

	/* R/O properties */
	if (!strcasecmp(property, "owner") || !strcasecmp(property, "callwait") || !strcasecmp(property, "threeway")) {
		ast_log(LOG_ERROR, "DAHDI subchannel names are R/O\n");
		return -1;
	/* R/W properties */
	} else if (!strcasecmp(property, "polarity")) {
		int polarity = atoi(value);
		REQUIRE_FXO_SIG();
		if (polarity != POLARITY_IDLE && polarity != POLARITY_REV) {
			ast_log(LOG_ERROR, "Invalid polarity: '%s'\n", value);
			return -1;
		}
		my_set_polarity(p, polarity);
	} else if (!strcasecmp(property, "dnd")) {
		int dnd = atoi(value);
		REQUIRE_FXO_SIG();
		analog_dnd(analog_p, dnd ? 1 : 0);
	} else if (!strcasecmp(property, "callforward")) {
		REQUIRE_FXO_SIG();
		if (strlen(value) >= sizeof(analog_p->call_forward) - 1) {
			ast_log(LOG_ERROR, "Provided call forwarding target '%s' is too long\n", value);
		}
		ast_copy_string(analog_p->call_forward, value, sizeof(analog_p->call_forward)); /* Could be empty to clear value */
	} else if (!strcasecmp(property, "lastexten")) {
		REQUIRE_FXO_SIG();
		if (strlen(value) >= sizeof(analog_p->lastexten) - 1) {
			ast_log(LOG_ERROR, "Provided lastexten target '%s' is too long\n", value);
		}
		ast_copy_string(analog_p->lastexten, value, sizeof(analog_p->lastexten)); /* Could be empty to clear value */
	} else {
		ast_log(LOG_ERROR, "Unknown DAHDI_CHANNEL property '%s'\n", property);
		return -1;
	}
	return 0;
}
#undef REQUIRE_FXO_SIG

static int dahdichan_helper(struct ast_channel *chan, char *data, const char *value, char *buffer, size_t buflen)
{
	char *parse;
	struct dahdi_pvt *pvt;
	struct dahdi_params dahdip;
	int res;
	AST_DECLARE_APP_ARGS(args,
		AST_APP_ARG(property);
		AST_APP_ARG(dahdichan);
	);

	parse = ast_strdupa(data);
	AST_STANDARD_APP_ARGS(args, parse);

	if (buflen > 0) {
		*buffer = '\0';
	}

	if (!ast_strlen_zero(args.dahdichan)) {
		/* DAHDI channel number explicitly provided, find it. */
		int channo = atoi(args.dahdichan);
		pvt = find_channel(channo);
		if (!pvt) {
			ast_log(LOG_ERROR, "DAHDI channel %d does not exist\n", channo);
			return -1;
		}
	} else {
		/* No channel specified explicitly, so implicitly use the current channel, in which case it must be a DAHDI channel. */
		if (!chan || !ast_channel_tech(chan) || strcasecmp(ast_channel_tech(chan)->type, "DAHDI")) {
			ast_log(LOG_WARNING, "%s is not a DAHDI channel, and no DAHDI channel specified\n", ast_channel_name(chan));
			return -1;
		}
		pvt = ast_channel_tech_pvt(chan);
	}

	memset(&dahdip, 0, sizeof(dahdip));
	if (ioctl(pvt->subs[SUB_REAL].dfd, DAHDI_GET_PARAMS, &dahdip)) {
		ast_log(LOG_WARNING, "Unable to get parameters of DAHDI channel %d: %s\n", pvt->channel, strerror(errno));
		return -1;
	}

	/* We have the channel private to use. */
	ast_mutex_lock(&pvt->lock);
	if (value) {
		res = dahdichan_write_property(pvt, &dahdip, args.property, value);
	} else {
		res = dahdichan_read_property(pvt, &dahdip, args.property, buffer, buflen);
	}
	ast_mutex_unlock(&pvt->lock);
	return res;
}

static int dahdichan_read(struct ast_channel *chan, const char *cmd, char *data, char *buffer, size_t buflen)
{
	return dahdichan_helper(chan, data, NULL, buffer, buflen);
}

static int dahdichan_write(struct ast_channel *chan, const char *cmd, char *data, const char *value)
{
	return dahdichan_helper(chan, data, value, NULL, 0);
}

static struct ast_custom_function dahdichan_function = {
	.name = "DAHDI_CHANNEL",
	.write = dahdichan_write,
	.read = dahdichan_read,
};

/*! \todo The standalone POLARITY function can and should be deprecated/removed, since its functionality is now part of DAHDI_CHANNEL. */

static int polarity_read(struct ast_channel *chan, const char *cmd, char *data, char *buffer, size_t buflen)
{
	struct dahdi_params dahdip;
	struct dahdi_pvt *pvt = ast_channel_tech_pvt(chan);
	if (strcasecmp(ast_channel_tech(chan)->type, "DAHDI")) {
		ast_log(LOG_WARNING, "%s is not a DAHDI channel\n", ast_channel_name(chan));
		return -1;
	}
	memset(&dahdip, 0, sizeof(dahdip));
	if (ioctl(pvt->subs[SUB_REAL].dfd, DAHDI_GET_PARAMS, &dahdip)) {
		ast_log(LOG_WARNING, "Unable to get parameters of DAHDI channel %d: %s\n", pvt->subs[SUB_REAL].dfd, strerror(errno));
		return -1;
	}
	return dahdichan_read_property(ast_channel_tech_pvt(chan), &dahdip, "polarity", buffer, buflen);
}

static int polarity_write(struct ast_channel *chan, const char *cmd, char *data, const char *value)
{
	struct dahdi_params dahdip;
	struct dahdi_pvt *pvt = ast_channel_tech_pvt(chan);
	if (strcasecmp(ast_channel_tech(chan)->type, "DAHDI")) {
		ast_log(LOG_WARNING, "%s is not a DAHDI channel\n", ast_channel_name(chan));
		return -1;
	}
	memset(&dahdip, 0, sizeof(dahdip));
	if (ioctl(pvt->subs[SUB_REAL].dfd, DAHDI_GET_PARAMS, &dahdip)) {
		ast_log(LOG_WARNING, "Unable to get parameters of DAHDI channel %d: %s\n", pvt->subs[SUB_REAL].dfd, strerror(errno));
		return -1;
	}
	return dahdichan_write_property(ast_channel_tech_pvt(chan), &dahdip, "polarity", value);
}

static struct ast_custom_function polarity_function = {
	.name = "POLARITY",
	.write = polarity_write,
	.read = polarity_read,
};

static int action_dahdidndon(struct mansession *s, const struct message *m)
{
	struct dahdi_pvt *p;
	const char *channel = astman_get_header(m, "DAHDIChannel");

	if (ast_strlen_zero(channel)) {
		astman_send_error(s, m, "No channel specified");
		return 0;
	}
	p = find_channel_from_str(channel);
	if (!p) {
		astman_send_error(s, m, "No such channel");
		return 0;
	}
	dahdi_dnd(p, 1);
	astman_send_ack(s, m, "DND Enabled");
	return 0;
}

static int action_dahdidndoff(struct mansession *s, const struct message *m)
{
	struct dahdi_pvt *p;
	const char *channel = astman_get_header(m, "DAHDIChannel");

	if (ast_strlen_zero(channel)) {
		astman_send_error(s, m, "No channel specified");
		return 0;
	}
	p = find_channel_from_str(channel);
	if (!p) {
		astman_send_error(s, m, "No such channel");
		return 0;
	}
	dahdi_dnd(p, 0);
	astman_send_ack(s, m, "DND Disabled");
	return 0;
}

static int action_transfer(struct mansession *s, const struct message *m)
{
	struct dahdi_pvt *p;
	const char *channel = astman_get_header(m, "DAHDIChannel");

	if (ast_strlen_zero(channel)) {
		astman_send_error(s, m, "No channel specified");
		return 0;
	}
	p = find_channel_from_str(channel);
	if (!p) {
		astman_send_error(s, m, "No such channel");
		return 0;
	}
	if (!dahdi_analog_lib_handles(p->sig, 0, 0)) {
		astman_send_error(s, m, "Channel signaling is not analog");
		return 0;
	}
	dahdi_fake_event(p,TRANSFER);
	astman_send_ack(s, m, "DAHDITransfer");
	return 0;
}

static int action_transferhangup(struct mansession *s, const struct message *m)
{
	struct dahdi_pvt *p;
	const char *channel = astman_get_header(m, "DAHDIChannel");

	if (ast_strlen_zero(channel)) {
		astman_send_error(s, m, "No channel specified");
		return 0;
	}
	p = find_channel_from_str(channel);
	if (!p) {
		astman_send_error(s, m, "No such channel");
		return 0;
	}
	if (!dahdi_analog_lib_handles(p->sig, 0, 0)) {
		astman_send_error(s, m, "Channel signaling is not analog");
		return 0;
	}
	dahdi_fake_event(p,HANGUP);
	astman_send_ack(s, m, "DAHDIHangup");
	return 0;
}

static int action_dahdidialoffhook(struct mansession *s, const struct message *m)
{
	struct dahdi_pvt *p;
	const char *channel = astman_get_header(m, "DAHDIChannel");
	const char *number = astman_get_header(m, "Number");
	int i;

	if (ast_strlen_zero(channel)) {
		astman_send_error(s, m, "No channel specified");
		return 0;
	}
	if (ast_strlen_zero(number)) {
		astman_send_error(s, m, "No number specified");
		return 0;
	}
	p = find_channel_from_str(channel);
	if (!p) {
		astman_send_error(s, m, "No such channel");
		return 0;
	}
	if (!p->owner) {
		astman_send_error(s, m, "Channel does not have it's owner");
		return 0;
	}
	for (i = 0; i < strlen(number); i++) {
		struct ast_frame f = { AST_FRAME_DTMF, .subclass.integer = number[i] };
		dahdi_queue_frame(p, &f);
	}
	astman_send_ack(s, m, "DAHDIDialOffhook");
	return 0;
}

static int action_dahdishowchannels(struct mansession *s, const struct message *m)
{
	struct dahdi_pvt *tmp = NULL;
	const char *id = astman_get_header(m, "ActionID");
	const char *dahdichannel = astman_get_header(m, "DAHDIChannel");
	char idText[256];
	int channels = 0;
	int dahdichanquery;

	if (!dahdichannel || sscanf(dahdichannel, "%30d", &dahdichanquery) != 1) {
		/* Not numeric string. */
		dahdichanquery = -1;
	}

	idText[0] = '\0';
	if (!ast_strlen_zero(id)) {
		snprintf(idText, sizeof(idText), "ActionID: %s\r\n", id);
	}

	astman_send_listack(s, m, "DAHDI channel status will follow", "start");

	ast_mutex_lock(&iflock);

	for (tmp = iflist; tmp; tmp = tmp->next) {
		if (tmp->channel > 0) {
			int alm;

			/* If a specific channel is queried for, only deliver status for that channel */
			if (dahdichanquery > 0 && tmp->channel != dahdichanquery)
				continue;

			alm = get_alarms(tmp);
			channels++;
			if (tmp->owner) {
				/* Add data if we have a current call */
				astman_append(s,
					"Event: DAHDIShowChannels\r\n"
					"DAHDIChannel: %d\r\n"
					"Channel: %s\r\n"
					"Uniqueid: %s\r\n"
					"AccountCode: %s\r\n"
					"Signalling: %s\r\n"
					"SignallingCode: %d\r\n"
					"Context: %s\r\n"
					"DND: %s\r\n"
					"Alarm: %s\r\n"
					"Description: %s\r\n"
					"%s"
					"\r\n",
					tmp->channel,
					ast_channel_name(tmp->owner),
					ast_channel_uniqueid(tmp->owner),
					ast_channel_accountcode(tmp->owner),
					sig2str(tmp->sig),
					tmp->sig,
					tmp->context,
					dahdi_dnd(tmp, -1) ? "Enabled" : "Disabled",
					alarm2str(alm),
					tmp->description, idText);
			} else {
				astman_append(s,
					"Event: DAHDIShowChannels\r\n"
					"DAHDIChannel: %d\r\n"
					"Signalling: %s\r\n"
					"SignallingCode: %d\r\n"
					"Context: %s\r\n"
					"DND: %s\r\n"
					"Alarm: %s\r\n"
					"Description: %s\r\n"
					"%s"
					"\r\n",
					tmp->channel, sig2str(tmp->sig), tmp->sig,
					tmp->context,
					dahdi_dnd(tmp, -1) ? "Enabled" : "Disabled",
					alarm2str(alm),
					tmp->description, idText);
			}
		}
	}

	ast_mutex_unlock(&iflock);

	astman_send_list_complete_start(s, m, "DAHDIShowChannelsComplete", channels);
	astman_append(s, "Items: %d\r\n", channels);
	astman_send_list_complete_end(s);
	return 0;
}

static int action_dahdishowstatus(struct mansession *s, const struct message *m)
{
	const char *id = astman_get_header(m, "ActionID");
	int span;
	int res;
	char alarmstr[50];
	int ctl;
	char idText[256];
	int numspans = 0;
	struct dahdi_spaninfo spaninfo;

	ctl = open("/dev/dahdi/ctl", O_RDWR);
	if (ctl < 0) {
		astman_send_error(s, m, "No DAHDI detected");
		return 0;
	}

	idText[0] = '\0';
	if (!ast_strlen_zero(id)) {
		snprintf(idText, sizeof(idText), "ActionID: %s\r\n", id);
	}
	astman_send_listack(s, m, "DAHDI span statuses will follow", "start");

	for (span = 1; span < DAHDI_MAX_SPANS; ++span) {
		spaninfo.spanno = span;
		res = ioctl(ctl, DAHDI_SPANSTAT, &spaninfo);
		if (res) {
			continue;
		}
		numspans++;
		build_alarm_info(alarmstr, &spaninfo);
		astman_append(s,
			"Event: DAHDIShowStatus\r\n"
			"Span: %d\r\n"
			"Description: %s\r\n"
			"Alarms: %s\r\n"
			"IRQ: %d\r\n"
			"bpviol: %d\r\n"
			"CRC: %d\r\n"
			"Framing: %s\r\n"
			"Coding: %s\r\n"
			"Options: %s\r\n"
			"LBO: %s\r\n"
			"%s"
			"\r\n",
			span, spaninfo.desc, alarmstr, spaninfo.irqmisses, spaninfo.bpvcount, spaninfo.crc4count,
			spaninfo.lineconfig & DAHDI_CONFIG_D4 ? "D4" :
			spaninfo.lineconfig & DAHDI_CONFIG_ESF ? "ESF" :
			spaninfo.lineconfig & DAHDI_CONFIG_CCS ? "CCS" :
			"CAS",
			spaninfo.lineconfig & DAHDI_CONFIG_B8ZS ? "B8ZS" :
			spaninfo.lineconfig & DAHDI_CONFIG_HDB3 ? "HDB3" :
			spaninfo.lineconfig & DAHDI_CONFIG_AMI ? "AMI" :
			"Unk",
			spaninfo.lineconfig & DAHDI_CONFIG_CRC4 ?
				spaninfo.lineconfig & DAHDI_CONFIG_NOTOPEN ? "CRC4/YEL" : "CRC4" :
				spaninfo.lineconfig & DAHDI_CONFIG_NOTOPEN ? "YEL" : "",
			lbostr[spaninfo.lbo],
			idText);
	}
	close(ctl);

	astman_send_list_complete_start(s, m, "DAHDIShowStatusComplete", numspans);
	astman_append(s, "Items: %d\r\n", numspans);
	astman_send_list_complete_end(s);
	return 0;
}

#if defined(HAVE_PRI)
static int action_prishowspans(struct mansession *s, const struct message *m)
{
	int count;
	int idx;
	int span_query;
	struct dahdi_pri *dspan;
	const char *id = astman_get_header(m, "ActionID");
	const char *span_str = astman_get_header(m, "Span");
	char action_id[256];
	const char *show_cmd = "PRIShowSpans";

	/* NOTE: Asking for span 0 gets all spans. */
	if (!ast_strlen_zero(span_str)) {
		span_query = atoi(span_str);
	} else {
		span_query = 0;
	}

	if (!ast_strlen_zero(id)) {
		snprintf(action_id, sizeof(action_id), "ActionID: %s\r\n", id);
	} else {
		action_id[0] = '\0';
	}

	astman_send_listack(s, m, "Span status will follow", "start");

	count = 0;
	for (idx = 0; idx < ARRAY_LEN(pris); ++idx) {
		dspan = &pris[idx];

		/* If a specific span is asked for, only deliver status for that span. */
		if (0 < span_query && dspan->pri.span != span_query) {
			continue;
		}

		if (dspan->pri.pri) {
			count += sig_pri_ami_show_spans(s, show_cmd, &dspan->pri, dspan->dchannels,
				action_id);
		}
	}

	astman_send_list_complete_start(s, m, "PRIShowSpansComplete", count);
	astman_append(s, "Items: %d\r\n", count);
	astman_send_list_complete_end(s);
	return 0;
}
#endif	/* defined(HAVE_PRI) */

#if defined(HAVE_SS7)
static int linkset_addsigchan(int sigchan)
{
	struct dahdi_ss7 *link;
	int res;
	int curfd;
	struct dahdi_params params;
	struct dahdi_bufferinfo bi;
	struct dahdi_spaninfo si;

	if (sigchan < 0) {
		ast_log(LOG_ERROR, "Invalid sigchan!\n");
		return -1;
	}
	if (cur_ss7type < 0) {
		ast_log(LOG_ERROR, "Unspecified or invalid ss7type\n");
		return -1;
	}
	if (cur_pointcode < 0) {
		ast_log(LOG_ERROR, "Unspecified pointcode!\n");
		return -1;
	}
	if (cur_adjpointcode < 0) {
		ast_log(LOG_ERROR, "Unspecified adjpointcode!\n");
		return -1;
	}
	if (cur_defaultdpc < 0) {
		ast_log(LOG_ERROR, "Unspecified defaultdpc!\n");
		return -1;
	}
	if (cur_networkindicator < 0) {
		ast_log(LOG_ERROR, "Invalid networkindicator!\n");
		return -1;
	}
	link = ss7_resolve_linkset(cur_linkset);
	if (!link) {
		ast_log(LOG_ERROR, "Invalid linkset number.  Must be between 1 and %d\n", NUM_SPANS + 1);
		return -1;
	}
	if (link->ss7.numsigchans >= SIG_SS7_NUM_DCHANS) {
		ast_log(LOG_ERROR, "Too many sigchans on linkset %d\n", cur_linkset);
		return -1;
	}

	curfd = link->ss7.numsigchans;

	/* Open signaling channel */
	link->ss7.fds[curfd] = open("/dev/dahdi/channel", O_RDWR, 0600);
	if (link->ss7.fds[curfd] < 0) {
		ast_log(LOG_ERROR, "Unable to open SS7 sigchan %d (%s)\n", sigchan,
			strerror(errno));
		return -1;
	}
	if (ioctl(link->ss7.fds[curfd], DAHDI_SPECIFY, &sigchan) == -1) {
		dahdi_close_ss7_fd(link, curfd);
		ast_log(LOG_ERROR, "Unable to specify SS7 sigchan %d (%s)\n", sigchan,
			strerror(errno));
		return -1;
	}

	/* Get signaling channel parameters */
	memset(&params, 0, sizeof(params));
	res = ioctl(link->ss7.fds[curfd], DAHDI_GET_PARAMS, &params);
	if (res) {
		dahdi_close_ss7_fd(link, curfd);
		ast_log(LOG_ERROR, "Unable to get parameters for sigchan %d (%s)\n", sigchan,
			strerror(errno));
		return -1;
	}
	if (params.sigtype != DAHDI_SIG_HDLCFCS
		&& params.sigtype != DAHDI_SIG_HARDHDLC
		&& params.sigtype != DAHDI_SIG_MTP2) {
		dahdi_close_ss7_fd(link, curfd);
		ast_log(LOG_ERROR, "sigchan %d is not in HDLC/FCS mode.\n", sigchan);
		return -1;
	}

	/* Set signaling channel buffer policy. */
	memset(&bi, 0, sizeof(bi));
	bi.txbufpolicy = DAHDI_POLICY_IMMEDIATE;
	bi.rxbufpolicy = DAHDI_POLICY_IMMEDIATE;
	bi.numbufs = 32;
	bi.bufsize = 512;
	if (ioctl(link->ss7.fds[curfd], DAHDI_SET_BUFINFO, &bi)) {
		ast_log(LOG_ERROR, "Unable to set appropriate buffering on channel %d: %s\n",
			sigchan, strerror(errno));
		dahdi_close_ss7_fd(link, curfd);
		return -1;
	}

	/* Get current signaling channel alarm status. */
	memset(&si, 0, sizeof(si));
	res = ioctl(link->ss7.fds[curfd], DAHDI_SPANSTAT, &si);
	if (res) {
		dahdi_close_ss7_fd(link, curfd);
		ast_log(LOG_ERROR, "Unable to get span state for sigchan %d (%s)\n", sigchan,
			strerror(errno));
	}

	res = sig_ss7_add_sigchan(&link->ss7, curfd, cur_ss7type,
		(params.sigtype == DAHDI_SIG_MTP2)
			? SS7_TRANSPORT_DAHDIMTP2
			: SS7_TRANSPORT_DAHDIDCHAN,
		si.alarms, cur_networkindicator, cur_pointcode, cur_adjpointcode, cur_slc);
	if (res) {
		dahdi_close_ss7_fd(link, curfd);
		return -1;
	}

	++link->ss7.numsigchans;

	return 0;
}
#endif	/* defined(HAVE_SS7) */

#if defined(HAVE_SS7)
static char *handle_ss7_debug(struct ast_cli_entry *e, int cmd, struct ast_cli_args *a)
{
	int span;
	switch (cmd) {
	case CLI_INIT:
		e->command = "ss7 set debug {on|off} linkset";
		e->usage =
			"Usage: ss7 set debug {on|off} linkset <linkset>\n"
			"       Enables debugging on a given SS7 linkset\n";
		return NULL;
	case CLI_GENERATE:
		return NULL;
	}

	if (a->argc < 6) {
		return CLI_SHOWUSAGE;
	}

	span = atoi(a->argv[5]);
	if ((span < 1) || (span > NUM_SPANS)) {
		ast_cli(a->fd, "Invalid linkset %s.  Should be a number from %d to %d\n", a->argv[5], 1, NUM_SPANS);
		return CLI_SUCCESS;
	}
	if (!linksets[span-1].ss7.ss7) {
		ast_cli(a->fd, "No SS7 running on linkset %d\n", span);
	} else {
		if (!strcasecmp(a->argv[3], "on")) {
			linksets[span - 1].ss7.debug = 1;
			ss7_set_debug(linksets[span-1].ss7.ss7, SIG_SS7_DEBUG);
			ast_cli(a->fd, "Enabled debugging on linkset %d\n", span);
		} else {
			linksets[span - 1].ss7.debug = 0;
			ss7_set_debug(linksets[span-1].ss7.ss7, 0);
			ast_cli(a->fd, "Disabled debugging on linkset %d\n", span);
		}
	}

	return CLI_SUCCESS;
}
#endif	/* defined(HAVE_SS7) */

#if defined(HAVE_SS7)
static char *handle_ss7_cic_blocking(struct ast_cli_entry *e, int cmd, struct ast_cli_args *a)
{
	int linkset, cic;
	int blocked, i;
	int do_block = 0;
	unsigned int dpc;

	switch (cmd) {
	case CLI_INIT:
		e->command = "ss7 {block|unblock} cic";
		e->usage =
			"Usage: ss7 {block|unblock} cic <linkset> <dpc> <CIC>\n"
			"       Sends a remote {blocking|unblocking} request for the given CIC on the specified linkset\n";
		return NULL;
	case CLI_GENERATE:
		return NULL;
	}

	if (a->argc == 6) {
		linkset = atoi(a->argv[3]);
	} else {
		return CLI_SHOWUSAGE;
	}

	if (!strcasecmp(a->argv[1], "block")) {
		do_block = 1;
	} else if (strcasecmp(a->argv[1], "unblock")) {
		return CLI_SHOWUSAGE;
	}

	if ((linkset < 1) || (linkset > NUM_SPANS)) {
		ast_cli(a->fd, "Invalid linkset %s.  Should be a number %d to %d\n", a->argv[3], 1, NUM_SPANS);
		return CLI_SUCCESS;
	}

	if (!linksets[linkset-1].ss7.ss7) {
		ast_cli(a->fd, "No SS7 running on linkset %d\n", linkset);
		return CLI_SUCCESS;
	}

	cic = atoi(a->argv[5]);
	if (cic < 1) {
		ast_cli(a->fd, "Invalid CIC specified!\n");
		return CLI_SUCCESS;
	}

	dpc = atoi(a->argv[4]);
	if (dpc < 1) {
		ast_cli(a->fd, "Invalid DPC specified!\n");
		return CLI_SUCCESS;
	}

	for (i = 0; i < linksets[linkset-1].ss7.numchans; i++) {
		if (linksets[linkset-1].ss7.pvts[i] && linksets[linkset-1].ss7.pvts[i]->cic == cic && linksets[linkset-1].ss7.pvts[i]->dpc == dpc) {
			blocked = linksets[linkset-1].ss7.pvts[i]->locallyblocked;
			if (!do_block ^ !(blocked & SS7_BLOCKED_MAINTENANCE)) {
				if (sig_ss7_cic_blocking(&linksets[linkset-1].ss7, do_block, i) < 0) {
					ast_cli(a->fd, "Unable to allocate new ss7call\n");
				} else {
					ast_cli(a->fd, "Sent %sblocking request for linkset %d on CIC %d DPC %d\n", (do_block) ? "" : "un", linkset, cic, dpc);
				}
			} else if (!do_block && blocked) {
				ast_cli(a->fd, "CIC %d is hardware locally blocked!\n", cic);
			} else {
				ast_cli(a->fd, "CIC %d %s locally blocked\n", cic, do_block ? "already" : "is not");
			}
			return CLI_SUCCESS;
		}
	}

	ast_cli(a->fd, "Invalid CIC specified!\n");
	return CLI_SUCCESS;
}
#endif	/* defined(HAVE_SS7) */

#if defined(HAVE_SS7)
static char *handle_ss7_linkset_mng(struct ast_cli_entry *e, int cmd, struct ast_cli_args *a)
{
	int linkset, i;
	enum {
		DO_BLOCK,
		DO_UNBLOCK,
		DO_RESET,
	} do_what;

	switch (cmd) {
	case CLI_INIT:
		e->command = "ss7 {reset|block|unblock} linkset";
		e->usage =
			"Usage: ss7 {reset|block|unblock} linkset <linkset number>\n"
			"       Sends a remote {reset|blocking|unblocking} request for all CICs on the given linkset\n";
		return NULL;
	case CLI_GENERATE:
		return NULL;
	}

	if (a->argc == 4) {
		linkset = atoi(a->argv[3]);
	} else {
		return CLI_SHOWUSAGE;
	}

	if (!strcasecmp(a->argv[1], "block")) {
		do_what = DO_BLOCK;
	} else if (!strcasecmp(a->argv[1], "unblock")) {
		do_what = DO_UNBLOCK;
	} else if (!strcasecmp(a->argv[1], "reset")) {
		do_what = DO_RESET;
	} else {
		return CLI_SHOWUSAGE;
	}

	if ((linkset < 1) || (linkset > NUM_SPANS)) {
		ast_cli(a->fd, "Invalid linkset %s.  Should be a number %d to %d\n", a->argv[3], 1, NUM_SPANS);
		return CLI_SUCCESS;
	}

	if (!linksets[linkset - 1].ss7.ss7) {
		ast_cli(a->fd, "No SS7 running on linkset %d\n", linkset);
		return CLI_SUCCESS;
	}

	for (i = 0; i < linksets[linkset - 1].ss7.numchans; i++) {
		/* XXX Should be done with GRS/CGB/CGU instead - see ss7_reset_linkset() */
		if (linksets[linkset - 1].ss7.pvts[i]) {
			switch (do_what) {
			case DO_BLOCK:
			case DO_UNBLOCK:
				if (sig_ss7_cic_blocking(&linksets[linkset - 1].ss7, do_what == DO_BLOCK, i)) {
					ast_cli(a->fd, "Sent remote %s request on CIC %d\n",
						(do_what == DO_BLOCK) ? "blocking" : "unblocking",
						linksets[linkset - 1].ss7.pvts[i]->cic);
				}
				break;
			case DO_RESET:
				if (sig_ss7_reset_cic(&linksets[linkset - 1].ss7,
					linksets[linkset - 1].ss7.pvts[i]->cic,
					linksets[linkset - 1].ss7.pvts[i]->dpc)) {
					ast_cli(a->fd, "Sent reset request on CIC %d\n",
						linksets[linkset - 1].ss7.pvts[i]->cic);
				}
				break;
			}
		}
	}

	return CLI_SUCCESS;
}
#endif	/* defined(HAVE_SS7) */

#if defined(HAVE_SS7)
static char *handle_ss7_group_blocking(struct ast_cli_entry *e, int cmd, struct ast_cli_args *a)
{
	int linkset, cic, range, chanpos;
	int i, dpc, orient = 0;
	int do_block = 0;
	unsigned char state[255];

	switch (cmd) {
	case CLI_INIT:
		e->command = "ss7 {block|unblock} group";
		e->usage =
			"Usage: ss7 {block|unblock} group <linkset> <dpc> <1st. CIC> <range> [H]\n"
			"       Sends a remote {blocking|unblocking} request for CIC range on the specified linkset\n";
		return NULL;
	case CLI_GENERATE:
		return NULL;
	}

	if (a->argc == 7 || a->argc == 8) {
		linkset = atoi(a->argv[3]);
	} else {
		return CLI_SHOWUSAGE;
	}

	if (!strcasecmp(a->argv[1], "block")) {
		do_block = 1;
	} else if (strcasecmp(a->argv[1], "unblock")) {
		return CLI_SHOWUSAGE;
	}

	if (a->argc == 8) {
		if (!strcasecmp(a->argv[7], "H")) {
			orient = 1;
		} else {
			return CLI_SHOWUSAGE;
		}
	}

	if ((linkset < 1) || (linkset > NUM_SPANS)) {
		ast_cli(a->fd, "Invalid linkset %s.  Should be a number %d to %d\n", a->argv[4], 1, NUM_SPANS);
		return CLI_SUCCESS;
	}

	if (!linksets[linkset-1].ss7.ss7) {
		ast_cli(a->fd, "No SS7 running on linkset %d\n", linkset);
		return CLI_SUCCESS;
	}

	cic = atoi(a->argv[5]);
	if (cic < 1) {
		ast_cli(a->fd, "Invalid CIC specified!\n");
		return CLI_SUCCESS;
	}

	range = atoi(a->argv[6]);
	/* ITU-T Q.763 3.43 - range 0 is reserved, which makes a range of 2 CICs a minimum group */
	if (range < 1 || range > (linksets[linkset - 1].ss7.type == SS7_ANSI ? 24 : 31)) {
		ast_cli(a->fd, "Invalid range specified!\n");
		return CLI_SUCCESS;
	}

	dpc = atoi(a->argv[4]);
	if (dpc < 1) {
		ast_cli(a->fd, "Invalid DPC specified!\n");
		return CLI_SUCCESS;
	}

	ast_mutex_lock(&linksets[linkset-1].ss7.lock);
	if (!sig_ss7_find_cic_range(&linksets[linkset-1].ss7, cic, cic + range, dpc)) {
		ast_mutex_unlock(&linksets[linkset-1].ss7.lock);
		ast_cli(a->fd, "Invalid CIC/RANGE\n");
		return CLI_SHOWUSAGE;
	}

	memset(state, 0, sizeof(state));
	for (i = 0; i <= range; ++i) {
		state[i] = 1;
	}

	/* We are guaranteed to find chanpos because of sig_ss7_find_cic_range() includes it. */
	chanpos = sig_ss7_find_cic(&linksets[linkset-1].ss7, cic, dpc);
	if (sig_ss7_group_blocking(&linksets[linkset-1].ss7, do_block, chanpos, cic + range, state, orient)) {
		ast_cli(a->fd, "Unable allocate new ss7call\n");
	} else {
		ast_cli(a->fd, "Sending remote%s %sblocking request linkset %d on CIC %d range %d\n",
			orient ? " hardware" : "", do_block ? "" : "un", linkset, cic, range);
	}

	ast_mutex_unlock(&linksets[linkset-1].ss7.lock);

	/* Break poll on the linkset so it sends our messages */
	if (linksets[linkset-1].ss7.master != AST_PTHREADT_NULL) {
		pthread_kill(linksets[linkset-1].ss7.master, SIGURG);
	}
	return CLI_SUCCESS;
}
#endif	/* defined(HAVE_SS7) */

#if defined(HAVE_SS7)
static char *handle_ss7_group_reset(struct ast_cli_entry *e, int cmd, struct ast_cli_args *a)
{
	int linkset, cic, range;
	unsigned int dpc;

	switch (cmd) {
	case CLI_INIT:
		e->command = "ss7 reset group";
		e->usage =
			"Usage: ss7 reset group <linkset> <dpc> <1st CIC> <range>\n"
			"       Send a GRS for the given CIC range on the specified linkset\n";
		return NULL;
	case CLI_GENERATE:
		return NULL;
	}

	if (a->argc == 7) {
		linkset = atoi(a->argv[3]);
	} else {
		return CLI_SHOWUSAGE;
	}

	if ((linkset < 1) || (linkset > NUM_SPANS)) {
		ast_cli(a->fd, "Invalid linkset %s.  Should be a number %d to %d\n", a->argv[4], 1, NUM_SPANS);
		return CLI_SUCCESS;
	}

	if (!linksets[linkset-1].ss7.ss7) {
		ast_cli(a->fd, "No SS7 running on linkset %d\n", linkset);
		return CLI_SUCCESS;
	}

	cic = atoi(a->argv[5]);

	if (cic < 1) {
		ast_cli(a->fd, "Invalid CIC specified!\n");
		return CLI_SUCCESS;
	}

	range = atoi(a->argv[6]);
	if (range < 1 || range > (linksets[linkset - 1].ss7.type == SS7_ANSI ? 24 : 31)) {
		ast_cli(a->fd, "Invalid range specified!\n");
		return CLI_SUCCESS;
	}

	dpc = atoi(a->argv[4]);
	if (dpc < 1) {
		ast_cli(a->fd, "Invalid DPC specified!\n");
		return CLI_SUCCESS;
	}

	ast_mutex_lock(&linksets[linkset-1].ss7.lock);
	if (!sig_ss7_find_cic_range(&linksets[linkset-1].ss7, cic, cic + range, dpc)) {
		ast_mutex_unlock(&linksets[linkset-1].ss7.lock);
		ast_cli(a->fd, "Invalid CIC/RANGE\n");
		return CLI_SHOWUSAGE;
	}

	if (sig_ss7_reset_group(&linksets[linkset-1].ss7, cic, dpc, range)) {
		ast_cli(a->fd, "Unable to allocate new ss7call\n");
	} else {
		ast_cli(a->fd, "GRS sent ... \n");
	}

	ast_mutex_unlock(&linksets[linkset-1].ss7.lock);

	/* Break poll on the linkset so it sends our messages */
	if (linksets[linkset-1].ss7.master != AST_PTHREADT_NULL) {
		pthread_kill(linksets[linkset-1].ss7.master, SIGURG);
	}
	return CLI_SUCCESS;
}
#endif	/* defined(HAVE_SS7) */

#if defined(HAVE_SS7)
static char *handle_ss7_show_calls(struct ast_cli_entry *e, int cmd, struct ast_cli_args *a)
{
	int linkset;

	switch (cmd) {
	case CLI_INIT:
		e->command = "ss7 show calls";
		e->usage =
			"Usage: ss7 show calls <linkset>\n"
			"       Show SS7 calls on the specified linkset\n";
		return NULL;
	case CLI_GENERATE:
		return NULL;
	}

	if (a->argc == 4) {
		linkset = atoi(a->argv[3]);
	} else {
		return CLI_SHOWUSAGE;
	}

	if ((linkset < 1) || (linkset > NUM_SPANS)) {
		ast_cli(a->fd, "Invalid linkset %s.  Should be a number %d to %d\n", a->argv[3], 1, NUM_SPANS);
		return CLI_SUCCESS;
	}

	if (!linksets[linkset-1].ss7.ss7) {
		ast_cli(a->fd, "No SS7 running on linkset %d\n", linkset);
		return CLI_SUCCESS;
	}

	ast_mutex_lock(&linksets[linkset-1].ss7.lock);
	isup_show_calls(linksets[linkset-1].ss7.ss7, &ast_cli, a->fd);
	ast_mutex_unlock(&linksets[linkset-1].ss7.lock);

	return CLI_SUCCESS;
}
#endif	/* defined(HAVE_SS7) */

#if defined(HAVE_SS7)
static char *handle_ss7_reset_cic(struct ast_cli_entry *e, int cmd, struct ast_cli_args *a)
{
	int linkset, cic, res;
	unsigned int dpc;

	switch (cmd) {
	case CLI_INIT:
		e->command = "ss7 reset cic";
		e->usage =
			"Usage: ss7 reset cic <linkset> <dpc> <CIC>\n"
			"       Send a RSC for the given CIC on the specified linkset\n";
		return NULL;
	case CLI_GENERATE:
		return NULL;
	}

	if (a->argc == 6) {
		linkset = atoi(a->argv[3]);
	} else {
		return CLI_SHOWUSAGE;
	}

	if ((linkset < 1) || (linkset > NUM_SPANS)) {
		ast_cli(a->fd, "Invalid linkset %s.  Should be a number %d to %d\n", a->argv[3], 1, NUM_SPANS);
		return CLI_SUCCESS;
	}

	if (!linksets[linkset-1].ss7.ss7) {
		ast_cli(a->fd, "No SS7 running on linkset %d\n", linkset);
		return CLI_SUCCESS;
	}

	cic = atoi(a->argv[5]);

	if (cic < 1) {
		ast_cli(a->fd, "Invalid CIC specified!\n");
		return CLI_SUCCESS;
	}

	dpc = atoi(a->argv[4]);
	if (dpc < 1) {
		ast_cli(a->fd, "Invalid DPC specified!\n");
		return CLI_SUCCESS;
	}

	res = sig_ss7_reset_cic(&linksets[linkset-1].ss7, cic, dpc);

	ast_cli(a->fd, "%s RSC for linkset %d on CIC %d DPC %d\n", res ? "Sent" : "Failed", linkset, cic, dpc);

	return CLI_SUCCESS;
}
#endif	/* defined(HAVE_SS7) */

#if defined(HAVE_SS7)
static char *handle_ss7_net_mng(struct ast_cli_entry *e, int cmd, struct ast_cli_args *a)
{
	int linkset;
	unsigned int slc;
	unsigned int arg = 0;
	const char *res;

	switch (cmd) {
	case CLI_INIT:
		e->command = "ss7 mtp3";
		e->usage =
			"Usage: ss7 mtp3 <linkset> <slc> coo|coa|cbd|cba|eco|eca|tfp|tfa|lin|lun|lia|lua|lid|lfu <arg>\n"
			"       Send a NET MNG message\n"
			"       WARNING!!! WARNING!!! We are not a STP, just for testing/development purposes\n";
		return NULL;
	case CLI_GENERATE:
		return NULL;
	}

	if (a->argc < 5) {
		return CLI_SHOWUSAGE;
	}

	linkset = atoi(a->argv[2]);
	if ((linkset < 1) || (linkset > NUM_SPANS)) {
		ast_cli(a->fd, "Invalid linkset %s.  Should be a number %d to %d\n", a->argv[2], 1, NUM_SPANS);
		return CLI_SUCCESS;
	}
	if (!linksets[linkset-1].ss7.ss7) {
		ast_cli(a->fd, "No SS7 running on linkset %d\n", linkset);
		return CLI_SUCCESS;
	}

	slc = atoi(a->argv[3]);

	if (a->argc == 6) {
		arg = atoi(a->argv[5]);
	}

	ast_mutex_lock(&linksets[linkset-1].ss7.lock);
	res = mtp3_net_mng(linksets[linkset-1].ss7.ss7, slc, a->argv[4], arg);
	ast_mutex_unlock(&linksets[linkset-1].ss7.lock);

	/* Break poll on the linkset so it sends our messages */
	if (linksets[linkset-1].ss7.master != AST_PTHREADT_NULL) {
		pthread_kill(linksets[linkset-1].ss7.master, SIGURG);
	}

	ast_cli(a->fd, "%s", res);

	return CLI_SUCCESS;
}
#endif	/* defined(HAVE_SS7) */

#if defined(HAVE_SS7)
static char *handle_ss7_mtp3_restart(struct ast_cli_entry *e, int cmd, struct ast_cli_args *a)
{
	int linkset;
	unsigned int slc = 0;

	switch (cmd) {
	case CLI_INIT:
		e->command = "ss7 restart mtp3";
		e->usage =
			"Usage: ss7 restart mtp3 <linkset> <slc>\n"
			"       Restart link\n";
		return NULL;
	case CLI_GENERATE:
		return NULL;
	}

	if (a->argc < 5) {
		return CLI_SHOWUSAGE;
	}

	linkset = atoi(a->argv[3]);
	if ((linkset < 1) || (linkset > NUM_SPANS)) {
		ast_cli(a->fd, "Invalid linkset %s.  Should be a number %d to %d\n", a->argv[2], 1, NUM_SPANS);
		return CLI_SUCCESS;
	}
	if (!linksets[linkset-1].ss7.ss7) {
		ast_cli(a->fd, "No SS7 running on linkset %d\n", linkset);
		return CLI_SUCCESS;
	}

	slc = atoi(a->argv[4]);

	ast_mutex_lock(&linksets[linkset-1].ss7.lock);
	mtp3_init_restart(linksets[linkset-1].ss7.ss7, slc);
	ast_mutex_unlock(&linksets[linkset-1].ss7.lock);

	/* Break poll on the linkset so it sends our messages */
	if (linksets[linkset-1].ss7.master != AST_PTHREADT_NULL) {
		pthread_kill(linksets[linkset-1].ss7.master, SIGURG);
	}

	return CLI_SUCCESS;
}
#endif	/* defined(HAVE_SS7) */

#if defined(HAVE_SS7)
static char *handle_ss7_show_linkset(struct ast_cli_entry *e, int cmd, struct ast_cli_args *a)
{
	int linkset;
	struct sig_ss7_linkset *ss7;
	switch (cmd) {
	case CLI_INIT:
		e->command = "ss7 show linkset";
		e->usage =
			"Usage: ss7 show linkset <span>\n"
			"       Shows the status of an SS7 linkset.\n";
		return NULL;
	case CLI_GENERATE:
		return NULL;
	}

	if (a->argc < 4) {
		return CLI_SHOWUSAGE;
	}

	linkset = atoi(a->argv[3]);
	if ((linkset < 1) || (linkset > NUM_SPANS)) {
		ast_cli(a->fd, "Invalid linkset %s.  Should be a number %d to %d\n", a->argv[3], 1, NUM_SPANS);
		return CLI_SUCCESS;
	}
	ss7 = &linksets[linkset - 1].ss7;
	if (!ss7->ss7) {
		ast_cli(a->fd, "No SS7 running on linkset %d\n", linkset);
		return CLI_SUCCESS;
	}

	ast_cli(a->fd, "SS7 flags: 0x%x\n", ss7->flags);
	ast_cli(a->fd, "SS7 linkset %d status: %s\n", linkset, (ss7->state == LINKSET_STATE_UP) ? "Up" : "Down");
	ast_cli(a->fd, "SS7 calling nai: %i\n", ss7->calling_nai);
	ast_cli(a->fd, "SS7 called nai: %i\n", ss7->called_nai);
	ast_cli(a->fd, "SS7 nationalprefix: %s\n", ss7->nationalprefix);
	ast_cli(a->fd, "SS7 internationalprefix: %s\n", ss7->internationalprefix);
	ast_cli(a->fd, "SS7 unknownprefix: %s\n", ss7->unknownprefix);
	ast_cli(a->fd, "SS7 networkroutedprefix: %s\n", ss7->networkroutedprefix);
	ast_cli(a->fd, "SS7 subscriberprefix: %s\n", ss7->subscriberprefix);
	ss7_show_linkset(ss7->ss7, &ast_cli, a->fd);

	return CLI_SUCCESS;
}
#endif	/* defined(HAVE_SS7) */

#if defined(HAVE_SS7)
static char *handle_ss7_show_channels(struct ast_cli_entry *e, int cmd, struct ast_cli_args *a)
{
	int linkset;

	switch (cmd) {
	case CLI_INIT:
		e->command = "ss7 show channels";
		e->usage =
			"Usage: ss7 show channels\n"
			"       Displays SS7 channel information at a glance.\n";
		return NULL;
	case CLI_GENERATE:
		return NULL;
	}

	if (a->argc != 3) {
		return CLI_SHOWUSAGE;
	}

	sig_ss7_cli_show_channels_header(a->fd);
	for (linkset = 0; linkset < NUM_SPANS; ++linkset) {
		if (linksets[linkset].ss7.ss7) {
			sig_ss7_cli_show_channels(a->fd, &linksets[linkset].ss7);
		}
	}
	return CLI_SUCCESS;
}
#endif	/* defined(HAVE_SS7) */

#if defined(HAVE_SS7)
static char *handle_ss7_show_cics(struct ast_cli_entry *e, int cmd, struct ast_cli_args *a)
{
#define FORMAT "%5s %5s %6s %12s   %-12s\n"
#define FORMAT2 "%5i %5i %6i %12s   %-12s\n"
	int i, linkset, dpc = 0;
	struct sig_ss7_linkset *ss7;
	char *state;
	char blocking[12];

	switch (cmd) {
	case CLI_INIT:
		e->command = "ss7 show cics";
		e->usage =
			"Usage: ss7 show cics <linkset> [dpc]\n"
			"       Shows the cics of an SS7 linkset.\n";
		return NULL;
	case CLI_GENERATE:
		return NULL;
	}

	if (a->argc < 4 || a->argc > 5) {
		return CLI_SHOWUSAGE;
	}

	linkset = atoi(a->argv[3]);

	if ((linkset < 1) || (linkset > NUM_SPANS)) {
		ast_cli(a->fd, "Invalid linkset %s.  Should be a number %d to %d\n", a->argv[3], 1, NUM_SPANS);
		return CLI_SUCCESS;
	}

	if (!linksets[linkset-1].ss7.ss7) {
		ast_cli(a->fd, "No SS7 running on linkset %d\n", linkset);
		return CLI_SUCCESS;
	}
	ss7 = &linksets[linkset-1].ss7;

	if (a->argc == 5) {
		dpc = atoi(a->argv[4]);
		if (dpc < 1) {
			ast_cli(a->fd, "Invalid DPC specified!\n");
			return CLI_SUCCESS;
		}
	}

	ast_cli(a->fd, FORMAT, "CIC", "DPC", "DAHDI", "STATE", "BLOCKING");

	for (i = 0; i < ss7->numchans; i++) {
		if (!dpc || (ss7->pvts[i] && ss7->pvts[i]->dpc == dpc)) {
			struct dahdi_pvt *p = ss7->pvts[i]->chan_pvt;

			if (ss7->pvts[i]->owner) {
				state = "Used";
			} else if (ss7->pvts[i]->ss7call) {
				state = "Pending";
			} else if (!p->inservice) {
				state = "NotInServ";
			} else {
				state = "Idle";
			}

			if (p->locallyblocked) {
				strcpy(blocking, "L:");
				if (p->locallyblocked & SS7_BLOCKED_MAINTENANCE) {
					strcat(blocking, "M");
				} else {
					strcat(blocking, " ");
				}

				if (p->locallyblocked & SS7_BLOCKED_HARDWARE) {
					strcat(blocking, "H");
				} else {
					strcat(blocking, " ");
				}
			} else {
				strcpy(blocking, "    ");
			}

			if (p->remotelyblocked) {
				strcat(blocking, " R:");
				if (p->remotelyblocked & SS7_BLOCKED_MAINTENANCE) {
					strcat(blocking, "M");
				} else {
					strcat(blocking, " ");
				}

				if (p->remotelyblocked & SS7_BLOCKED_HARDWARE) {
					strcat(blocking, "H");
				} else {
					strcat(blocking, " ");
				}
			}

			ast_cli(a->fd, FORMAT2, ss7->pvts[i]->cic, ss7->pvts[i]->dpc, ss7->pvts[i]->channel, state, blocking);
		}
	}

	return CLI_SUCCESS;
#undef FORMAT
#undef FORMAT2
}
#endif	/* defined(HAVE_SS7) */

#if defined(HAVE_SS7)
static char *handle_ss7_version(struct ast_cli_entry *e, int cmd, struct ast_cli_args *a)
{
	switch (cmd) {
	case CLI_INIT:
		e->command = "ss7 show version";
		e->usage =
			"Usage: ss7 show version\n"
			"	Show the libss7 version\n";
		return NULL;
	case CLI_GENERATE:
		return NULL;
	}

	ast_cli(a->fd, "libss7 version: %s\n", ss7_get_version());

	return CLI_SUCCESS;
}
#endif	/* defined(HAVE_SS7) */

#if defined(HAVE_SS7)
static struct ast_cli_entry dahdi_ss7_cli[] = {
	AST_CLI_DEFINE(handle_ss7_debug, "Enables SS7 debugging on a linkset"),
	AST_CLI_DEFINE(handle_ss7_cic_blocking, "Blocks/Unblocks the given CIC"),
	AST_CLI_DEFINE(handle_ss7_linkset_mng, "Resets/Blocks/Unblocks all CICs on a linkset"),
	AST_CLI_DEFINE(handle_ss7_group_blocking, "Blocks/Unblocks the given CIC range"),
	AST_CLI_DEFINE(handle_ss7_reset_cic, "Resets the given CIC"),
	AST_CLI_DEFINE(handle_ss7_group_reset, "Resets the given CIC range"),
	AST_CLI_DEFINE(handle_ss7_mtp3_restart, "Restart a link"),
	AST_CLI_DEFINE(handle_ss7_net_mng, "Send an NET MNG message"),
	AST_CLI_DEFINE(handle_ss7_show_linkset, "Shows the status of a linkset"),
	AST_CLI_DEFINE(handle_ss7_show_channels, "Displays SS7 channel information"),
	AST_CLI_DEFINE(handle_ss7_show_calls, "Show ss7 calls"),
	AST_CLI_DEFINE(handle_ss7_show_cics, "Show cics on a linkset"),
	AST_CLI_DEFINE(handle_ss7_version, "Displays libss7 version"),
};
#endif	/* defined(HAVE_SS7) */

#if defined(HAVE_PRI)
#if defined(HAVE_PRI_CCSS)
/*!
 * \internal
 * \brief CC agent initialization.
 * \since 1.8
 *
 * \param agent CC core agent control.
 * \param chan Original channel the agent will attempt to recall.
 *
 * \details
 * This callback is called when the CC core is initialized.  Agents should allocate
 * any private data necessary for the call and assign it to the private_data
 * on the agent.  Additionally, if any ast_cc_agent_flags are pertinent to the
 * specific agent type, they should be set in this function as well.
 *
 * \retval 0 on success.
 * \retval -1 on error.
 */
static int dahdi_pri_cc_agent_init(struct ast_cc_agent *agent, struct ast_channel *chan)
{
	struct dahdi_pvt *pvt;
	struct sig_pri_chan *pvt_chan;
	int res;

	ast_assert(!strcmp(ast_channel_tech(chan)->type, "DAHDI"));

	pvt = ast_channel_tech_pvt(chan);
	if (dahdi_sig_pri_lib_handles(pvt->sig)) {
		pvt_chan = pvt->sig_pvt;
	} else {
		pvt_chan = NULL;
	}
	if (!pvt_chan) {
		return -1;
	}

	ast_module_ref(ast_module_info->self);

	res = sig_pri_cc_agent_init(agent, pvt_chan);
	if (res) {
		ast_module_unref(ast_module_info->self);
	}
	return res;
}
#endif	/* defined(HAVE_PRI_CCSS) */
#endif	/* defined(HAVE_PRI) */

#if defined(HAVE_PRI)
#if defined(HAVE_PRI_CCSS)
/*!
 * \internal
 * \brief Destroy private data on the agent.
 * \since 1.8
 *
 * \param agent CC core agent control.
 *
 * \details
 * The core will call this function upon completion
 * or failure of CC.
 */
static void dahdi_pri_cc_agent_destructor(struct ast_cc_agent *agent)
{
	sig_pri_cc_agent_destructor(agent);

	ast_module_unref(ast_module_info->self);
}
#endif	/* defined(HAVE_PRI_CCSS) */
#endif	/* defined(HAVE_PRI) */

#if defined(HAVE_PRI)
#if defined(HAVE_PRI_CCSS)
static struct ast_cc_agent_callbacks dahdi_pri_cc_agent_callbacks = {
	.type = dahdi_pri_cc_type,
	.init = dahdi_pri_cc_agent_init,
	.start_offer_timer = sig_pri_cc_agent_start_offer_timer,
	.stop_offer_timer = sig_pri_cc_agent_stop_offer_timer,
	.respond = sig_pri_cc_agent_req_rsp,
	.status_request = sig_pri_cc_agent_status_req,
	.stop_ringing = sig_pri_cc_agent_stop_ringing,
	.party_b_free = sig_pri_cc_agent_party_b_free,
	.start_monitoring = sig_pri_cc_agent_start_monitoring,
	.callee_available = sig_pri_cc_agent_callee_available,
	.destructor = dahdi_pri_cc_agent_destructor,
};
#endif	/* defined(HAVE_PRI_CCSS) */
#endif	/* defined(HAVE_PRI) */

#if defined(HAVE_PRI)
#if defined(HAVE_PRI_CCSS)
static struct ast_cc_monitor_callbacks dahdi_pri_cc_monitor_callbacks = {
	.type = dahdi_pri_cc_type,
	.request_cc = sig_pri_cc_monitor_req_cc,
	.suspend = sig_pri_cc_monitor_suspend,
	.unsuspend = sig_pri_cc_monitor_unsuspend,
	.status_response = sig_pri_cc_monitor_status_rsp,
	.cancel_available_timer = sig_pri_cc_monitor_cancel_available_timer,
	.destructor = sig_pri_cc_monitor_destructor,
};
#endif	/* defined(HAVE_PRI_CCSS) */
#endif	/* defined(HAVE_PRI) */

static int __unload_module(void)
{
	struct dahdi_pvt *p;
#if defined(HAVE_PRI) || defined(HAVE_SS7)
	int i, j;
#endif	/* defined(HAVE_PRI) || defined(HAVE_SS7) */

#ifdef HAVE_PRI
	for (i = 0; i < NUM_SPANS; i++) {
		if (pris[i].pri.master != AST_PTHREADT_NULL) {
			pthread_cancel(pris[i].pri.master);
			pthread_kill(pris[i].pri.master, SIGURG);
		}
	}
	ast_cli_unregister_multiple(dahdi_pri_cli, ARRAY_LEN(dahdi_pri_cli));
	ast_unregister_application(dahdi_send_keypad_facility_app);
#ifdef HAVE_PRI_PROG_W_CAUSE
	ast_unregister_application(dahdi_send_callrerouting_facility_app);
#endif
#endif
#if defined(HAVE_SS7)
	for (i = 0; i < NUM_SPANS; i++) {
		if (linksets[i].ss7.master != AST_PTHREADT_NULL) {
			pthread_cancel(linksets[i].ss7.master);
			pthread_kill(linksets[i].ss7.master, SIGURG);
		}
	}
	ast_cli_unregister_multiple(dahdi_ss7_cli, ARRAY_LEN(dahdi_ss7_cli));
#endif	/* defined(HAVE_SS7) */
#if defined(HAVE_OPENR2)
	dahdi_r2_destroy_links();
	ast_cli_unregister_multiple(dahdi_mfcr2_cli, ARRAY_LEN(dahdi_mfcr2_cli));
	ast_unregister_application(dahdi_accept_r2_call_app);
#endif

	ast_custom_function_unregister(&dahdichan_function);
	ast_custom_function_unregister(&polarity_function);

	ast_cli_unregister_multiple(dahdi_cli, ARRAY_LEN(dahdi_cli));
	ast_manager_unregister("DAHDIDialOffhook");
	ast_manager_unregister("DAHDIHangup");
	ast_manager_unregister("DAHDITransfer");
	ast_manager_unregister("DAHDIDNDoff");
	ast_manager_unregister("DAHDIDNDon");
	ast_manager_unregister("DAHDIShowChannels");
	ast_manager_unregister("DAHDIShowStatus");
	ast_manager_unregister("DAHDIRestart");
#if defined(HAVE_PRI)
	ast_manager_unregister("PRIShowSpans");
	ast_manager_unregister("PRIDebugSet");
	ast_manager_unregister("PRIDebugFileSet");
	ast_manager_unregister("PRIDebugFileUnset");
#endif	/* defined(HAVE_PRI) */
	ast_channel_unregister(&dahdi_tech);

	/* Hangup all interfaces if they have an owner */
	ast_mutex_lock(&iflock);
	for (p = iflist; p; p = p->next) {
		if (p->owner)
			ast_softhangup(p->owner, AST_SOFTHANGUP_APPUNLOAD);
	}
	ast_mutex_unlock(&iflock);

	ast_mutex_lock(&monlock);
	if (monitor_thread && (monitor_thread != AST_PTHREADT_STOP) && (monitor_thread != AST_PTHREADT_NULL)) {
		pthread_cancel(monitor_thread);
		pthread_kill(monitor_thread, SIGURG);
		pthread_join(monitor_thread, NULL);
	}
	monitor_thread = AST_PTHREADT_STOP;
	ast_mutex_unlock(&monlock);

	destroy_all_channels();

#if defined(HAVE_PRI)
	for (i = 0; i < NUM_SPANS; i++) {
		if (pris[i].pri.master && (pris[i].pri.master != AST_PTHREADT_NULL)) {
			pthread_join(pris[i].pri.master, NULL);
		}
		for (j = 0; j < SIG_PRI_NUM_DCHANS; j++) {
			dahdi_close_pri_fd(&(pris[i]), j);
		}
		sig_pri_stop_pri(&pris[i].pri);
	}
#if defined(HAVE_PRI_CCSS)
	ast_cc_agent_unregister(&dahdi_pri_cc_agent_callbacks);
	ast_cc_monitor_unregister(&dahdi_pri_cc_monitor_callbacks);
#endif	/* defined(HAVE_PRI_CCSS) */
	sig_pri_unload();
#endif

#if defined(HAVE_SS7)
	for (i = 0; i < NUM_SPANS; i++) {
		if (linksets[i].ss7.master && (linksets[i].ss7.master != AST_PTHREADT_NULL)) {
			pthread_join(linksets[i].ss7.master, NULL);
		}
		for (j = 0; j < SIG_SS7_NUM_DCHANS; j++) {
			dahdi_close_ss7_fd(&(linksets[i]), j);
		}
		if (linksets[i].ss7.ss7) {
			ss7_destroy(linksets[i].ss7.ss7);
			linksets[i].ss7.ss7 = NULL;
		}
	}
#endif	/* defined(HAVE_SS7) */
	ast_cond_destroy(&ss_thread_complete);

	dahdi_native_unload();

	ao2_cleanup(dahdi_tech.capabilities);
	dahdi_tech.capabilities = NULL;
	STASIS_MESSAGE_TYPE_CLEANUP(dahdichannel_type);
	return 0;
}

static int unload_module(void)
{
#if defined(HAVE_PRI) || defined(HAVE_SS7)
	int y;
#endif	/* defined(HAVE_PRI) || defined(HAVE_SS7) */
#ifdef HAVE_PRI
	for (y = 0; y < NUM_SPANS; y++)
		ast_mutex_destroy(&pris[y].pri.lock);
#endif
#if defined(HAVE_SS7)
	for (y = 0; y < NUM_SPANS; y++)
		ast_mutex_destroy(&linksets[y].ss7.lock);
#endif	/* defined(HAVE_SS7) */
	return __unload_module();
}

static int build_channels(struct dahdi_chan_conf *conf, const char *value, int reload, int lineno)
{
	char *c, *chan;
	int x, start, finish;
	struct dahdi_pvt *tmp;

	if ((reload == 0) && (conf->chan.sig < 0) && !conf->is_sig_auto) {
		ast_log(LOG_ERROR, "Signalling must be specified before any channels are.\n");
		return -1;
	}

	c = ast_strdupa(value);

	while ((chan = strsep(&c, ","))) {
		if (sscanf(chan, "%30d-%30d", &start, &finish) == 2) {
			/* Range */
		} else if (sscanf(chan, "%30d", &start)) {
			/* Just one */
			finish = start;
		} else if (!strcasecmp(chan, "pseudo")) {
			finish = start = CHAN_PSEUDO;
		} else {
			ast_log(LOG_ERROR, "Syntax error parsing '%s' at '%s'\n", value, chan);
			return -1;
		}
		if (finish < start) {
			ast_log(LOG_WARNING, "Silliness: %d < %d\n", start, finish);
			x = finish;
			finish = start;
			start = x;
		}

		for (x = start; x <= finish; x++) {
			if (conf->wanted_channels_start &&
				(x < conf->wanted_channels_start ||
				 x > conf->wanted_channels_end)
			   ) {
				continue;
			}
			tmp = mkintf(x, conf, reload);

			if (tmp) {
				ast_verb(3, "%s channel %d, %s signalling\n", reload ? "Reconfigured" : "Registered", x, sig2str(tmp->sig));
			} else {
				ast_log(LOG_ERROR, "Unable to %s channel '%s'\n",
						(reload == 1) ? "reconfigure" : "register", value);
				return -1;
			}
			if (x == CHAN_PSEUDO) {
				has_pseudo = 1;
			}
		}
	}

	return 0;
}

/** The length of the parameters list of 'dahdichan'.
 * \todo Move definition of MAX_CHANLIST_LEN to a proper place. */
#define MAX_CHANLIST_LEN 80

static void process_echocancel(struct dahdi_chan_conf *confp, const char *data, unsigned int line)
{
	char *parse = ast_strdupa(data);
	char *params[DAHDI_MAX_ECHOCANPARAMS + 1];
	unsigned int param_count;
	unsigned int x;

	if (!(param_count = ast_app_separate_args(parse, ',', params, ARRAY_LEN(params))))
		return;

	memset(&confp->chan.echocancel, 0, sizeof(confp->chan.echocancel));

	/* first parameter is tap length, process it here */

	x = ast_strlen_zero(params[0]) ? 0 : atoi(params[0]);

	if ((x == 32) || (x == 64) || (x == 128) || (x == 256) || (x == 512) || (x == 1024))
		confp->chan.echocancel.head.tap_length = x;
	else if ((confp->chan.echocancel.head.tap_length = ast_true(params[0])))
		confp->chan.echocancel.head.tap_length = 128;

	/* now process any remaining parameters */

	for (x = 1; x < param_count; x++) {
		struct {
			char *name;
			char *value;
		} param;

		if (ast_app_separate_args(params[x], '=', (char **) &param, 2) < 1) {
			ast_log(LOG_WARNING, "Invalid echocancel parameter supplied at line %u: '%s'\n", line, params[x]);
			continue;
		}

		if (ast_strlen_zero(param.name) || (strlen(param.name) > sizeof(confp->chan.echocancel.params[0].name)-1)) {
			ast_log(LOG_WARNING, "Invalid echocancel parameter supplied at line %u: '%s'\n", line, param.name);
			continue;
		}

		strcpy(confp->chan.echocancel.params[confp->chan.echocancel.head.param_count].name, param.name);

		if (param.value) {
			if (sscanf(param.value, "%30d", &confp->chan.echocancel.params[confp->chan.echocancel.head.param_count].value) != 1) {
				ast_log(LOG_WARNING, "Invalid echocancel parameter value supplied at line %u: '%s'\n", line, param.value);
				continue;
			}
		}
		confp->chan.echocancel.head.param_count++;
	}
}

#if defined(HAVE_PRI)
#if defined(HAVE_PRI_DISPLAY_TEXT)
/*!
 * \internal
 * \brief Determine the configured display text options.
 * \since 10.0
 *
 * \param value Configuration value string.
 *
 * \return Configured display text option flags.
 */
static unsigned long dahdi_display_text_option(const char *value)
{
	char *val_str;
	char *opt_str;
	unsigned long options;

	options = 0;
	val_str = ast_strdupa(value);

	for (;;) {
		opt_str = strsep(&val_str, ",");
		if (!opt_str) {
			break;
		}
		opt_str = ast_strip(opt_str);
		if (!*opt_str) {
			continue;
		}

		if (!strcasecmp(opt_str, "block")) {
			options |= PRI_DISPLAY_OPTION_BLOCK;
		} else if (!strcasecmp(opt_str, "name_initial")) {
			options |= PRI_DISPLAY_OPTION_NAME_INITIAL;
		} else if (!strcasecmp(opt_str, "name_update")) {
			options |= PRI_DISPLAY_OPTION_NAME_UPDATE;
		} else if (!strcasecmp(opt_str, "name")) {
			options |= (PRI_DISPLAY_OPTION_NAME_INITIAL | PRI_DISPLAY_OPTION_NAME_UPDATE);
		} else if (!strcasecmp(opt_str, "text")) {
			options |= PRI_DISPLAY_OPTION_TEXT;
		}
	}
	return options;
}
#endif	/* defined(HAVE_PRI_DISPLAY_TEXT) */
#endif	/* defined(HAVE_PRI) */

#if defined(HAVE_PRI)
#if defined(HAVE_PRI_DATETIME_SEND)
/*!
 * \internal
 * \brief Determine the configured date/time send policy option.
 * \since 10.0
 *
 * \param value Configuration value string.
 *
 * \return Configured date/time send policy option.
 */
static int dahdi_datetime_send_option(const char *value)
{
	int option;

	option = PRI_DATE_TIME_SEND_DEFAULT;

	if (ast_false(value)) {
		option = PRI_DATE_TIME_SEND_NO;
	} else if (!strcasecmp(value, "date")) {
		option = PRI_DATE_TIME_SEND_DATE;
	} else if (!strcasecmp(value, "date_hh")) {
		option = PRI_DATE_TIME_SEND_DATE_HH;
	} else if (!strcasecmp(value, "date_hhmm")) {
		option = PRI_DATE_TIME_SEND_DATE_HHMM;
	} else if (!strcasecmp(value, "date_hhmmss")) {
		option = PRI_DATE_TIME_SEND_DATE_HHMMSS;
	}

	return option;
}
#endif	/* defined(HAVE_PRI_DATETIME_SEND) */
#endif	/* defined(HAVE_PRI) */

/*! process_dahdi() - ignore keyword 'channel' and similar */
#define PROC_DAHDI_OPT_NOCHAN  (1 << 0)
/*! process_dahdi() - No warnings on non-existing configuration keywords */
#define PROC_DAHDI_OPT_NOWARN  (1 << 1)

static void parse_busy_pattern(struct ast_variable *v, struct ast_dsp_busy_pattern *busy_cadence)
{
	int count_pattern = 0;
	int norval = 0;
	char *temp = NULL;

	for (; ;) {
		/* Scans the string for the next value in the pattern. If none, it checks to see if any have been entered so far. */
		if (!sscanf(v->value, "%30d", &norval) && count_pattern == 0) {
			ast_log(LOG_ERROR, "busypattern= expects either busypattern=tonelength,quietlength or busypattern=t1length, q1length, t2length, q2length at line %d.\n", v->lineno);
			break;
		}

		busy_cadence->pattern[count_pattern] = norval;

		count_pattern++;
		if (count_pattern == 4) {
			break;
		}

		temp = strchr(v->value, ',');
		if (temp == NULL) {
			break;
		}
		v->value = temp + 1;
	}
	busy_cadence->length = count_pattern;

	if (count_pattern % 2 != 0) {
		/* The pattern length must be divisible by two */
		ast_log(LOG_ERROR, "busypattern= expects either busypattern=tonelength,quietlength or busypattern=t1length, q1length, t2length, q2length at line %d.\n", v->lineno);
	}

}

static int process_dahdi(struct dahdi_chan_conf *confp, const char *cat, struct ast_variable *v, int reload, int options)
{
	struct dahdi_pvt *tmp;
	int y;
	struct ast_variable *dahdichan = NULL;

	/* Re-parse any cadences from beginning, rather than appending until we run out of room */
	user_has_defined_cadences = 0;

	for (; v; v = v->next) {
		if (!ast_jb_read_conf(&global_jbconf, v->name, v->value))
			continue;

		/* Create the interface list */
		if (!strcasecmp(v->name, "channel") || !strcasecmp(v->name, "channels")) {
			if (options & PROC_DAHDI_OPT_NOCHAN) {
				ast_log(LOG_WARNING, "Channel '%s' ignored.\n", v->value);
				continue;
			}
			if (build_channels(confp, v->value, reload, v->lineno)) {
				if (confp->ignore_failed_channels) {
					ast_log(LOG_WARNING, "Channel '%s' failure ignored: ignore_failed_channels.\n", v->value);
					continue;
				} else {
					return -1;
				}
			}
			ast_debug(1, "Channel '%s' configured.\n", v->value);
		} else if (!strcasecmp(v->name, "ignore_failed_channels")) {
			confp->ignore_failed_channels = ast_true(v->value);
		} else if (!strcasecmp(v->name, "buffers")) {
			if (parse_buffers_policy(v->value, &confp->chan.buf_no, &confp->chan.buf_policy)) {
				ast_log(LOG_WARNING, "Using default buffer policy.\n");
				confp->chan.buf_no = numbufs;
				confp->chan.buf_policy = DAHDI_POLICY_IMMEDIATE;
			}
		} else if (!strcasecmp(v->name, "faxbuffers")) {
			if (!parse_buffers_policy(v->value, &confp->chan.faxbuf_no, &confp->chan.faxbuf_policy)) {
				confp->chan.usefaxbuffers = 1;
			}
 		} else if (!strcasecmp(v->name, "dahdichan")) {
			/* Only process the last dahdichan value. */
			dahdichan = v;
		} else if (!strcasecmp(v->name, "usedistinctiveringdetection")) {
			usedistinctiveringdetection = ast_true(v->value);
		} else if (!strcasecmp(v->name, "distinctiveringaftercid")) {
			distinctiveringaftercid = ast_true(v->value);
		} else if (!strcasecmp(v->name, "dring1context")) {
			ast_copy_string(confp->chan.drings.ringContext[0].contextData,v->value,sizeof(confp->chan.drings.ringContext[0].contextData));
		} else if (!strcasecmp(v->name, "dring2context")) {
			ast_copy_string(confp->chan.drings.ringContext[1].contextData,v->value,sizeof(confp->chan.drings.ringContext[1].contextData));
		} else if (!strcasecmp(v->name, "dring3context")) {
			ast_copy_string(confp->chan.drings.ringContext[2].contextData,v->value,sizeof(confp->chan.drings.ringContext[2].contextData));
		} else if (!strcasecmp(v->name, "dring1range")) {
			confp->chan.drings.ringnum[0].range = atoi(v->value);
		} else if (!strcasecmp(v->name, "dring2range")) {
			confp->chan.drings.ringnum[1].range = atoi(v->value);
		} else if (!strcasecmp(v->name, "dring3range")) {
			confp->chan.drings.ringnum[2].range = atoi(v->value);
		} else if (!strcasecmp(v->name, "dring1")) {
			sscanf(v->value, "%30d,%30d,%30d", &confp->chan.drings.ringnum[0].ring[0], &confp->chan.drings.ringnum[0].ring[1], &confp->chan.drings.ringnum[0].ring[2]);
		} else if (!strcasecmp(v->name, "dring2")) {
			sscanf(v->value, "%30d,%30d,%30d", &confp->chan.drings.ringnum[1].ring[0], &confp->chan.drings.ringnum[1].ring[1], &confp->chan.drings.ringnum[1].ring[2]);
		} else if (!strcasecmp(v->name, "dring3")) {
			sscanf(v->value, "%30d,%30d,%30d", &confp->chan.drings.ringnum[2].ring[0], &confp->chan.drings.ringnum[2].ring[1], &confp->chan.drings.ringnum[2].ring[2]);
		} else if (!strcasecmp(v->name, "usecallerid")) {
			confp->chan.use_callerid = ast_true(v->value);
		} else if (!strcasecmp(v->name, "cidsignalling")) {
			if (!strcasecmp(v->value, "bell"))
				confp->chan.cid_signalling = CID_SIG_BELL;
			else if (!strcasecmp(v->value, "v23"))
				confp->chan.cid_signalling = CID_SIG_V23;
			else if (!strcasecmp(v->value, "dtmf"))
				confp->chan.cid_signalling = CID_SIG_DTMF;
			else if (!strcasecmp(v->value, "smdi"))
				confp->chan.cid_signalling = CID_SIG_SMDI;
			else if (!strcasecmp(v->value, "v23_jp"))
				confp->chan.cid_signalling = CID_SIG_V23_JP;
			else if (ast_true(v->value))
				confp->chan.cid_signalling = CID_SIG_BELL;
		} else if (!strcasecmp(v->name, "cidstart")) {
			if (!strcasecmp(v->value, "ring"))
				confp->chan.cid_start = CID_START_RING;
			else if (!strcasecmp(v->value, "polarity_in"))
				confp->chan.cid_start = CID_START_POLARITY_IN;
			else if (!strcasecmp(v->value, "polarity"))
				confp->chan.cid_start = CID_START_POLARITY;
			else if (!strcasecmp(v->value, "dtmf"))
				confp->chan.cid_start = CID_START_DTMF_NOALERT;
			else if (ast_true(v->value))
				confp->chan.cid_start = CID_START_RING;
		} else if (!strcasecmp(v->name, "threewaycalling")) {
			confp->chan.threewaycalling = ast_true(v->value);
		} else if (!strcasecmp(v->name, "threewaysilenthold")) {
			confp->chan.threewaysilenthold = ast_true(v->value);
		} else if (!strcasecmp(v->name, "cancallforward")) {
			confp->chan.cancallforward = ast_true(v->value);
		} else if (!strcasecmp(v->name, "relaxdtmf")) {
			if (ast_true(v->value))
				confp->chan.dtmfrelax = DSP_DIGITMODE_RELAXDTMF;
			else
				confp->chan.dtmfrelax = 0;
		} else if (!strcasecmp(v->name, "mailbox")) {
			ast_copy_string(confp->chan.mailbox, v->value, sizeof(confp->chan.mailbox));
		} else if (!strcasecmp(v->name, "description")) {
			ast_copy_string(confp->chan.description, v->value, sizeof(confp->chan.description));
		} else if (!strcasecmp(v->name, "hasvoicemail")) {
			if (ast_true(v->value) && ast_strlen_zero(confp->chan.mailbox)) {
				/*
				 * hasvoicemail is a users.conf legacy voicemail enable method.
				 * hasvoicemail is only going to work for app_voicemail mailboxes.
				 */
				if (strchr(cat, '@')) {
					ast_copy_string(confp->chan.mailbox, cat, sizeof(confp->chan.mailbox));
				} else {
					snprintf(confp->chan.mailbox, sizeof(confp->chan.mailbox),
						"%s@default", cat);
				}
			}
		} else if (!strcasecmp(v->name, "adsi")) {
			confp->chan.adsi = ast_true(v->value);
		} else if (!strcasecmp(v->name, "usesmdi")) {
			confp->chan.use_smdi = ast_true(v->value);
		} else if (!strcasecmp(v->name, "smdiport")) {
			ast_copy_string(confp->smdi_port, v->value, sizeof(confp->smdi_port));
		} else if (!strcasecmp(v->name, "transfer")) {
			confp->chan.transfer = ast_true(v->value);
		} else if (!strcasecmp(v->name, "canpark")) {
			confp->chan.canpark = ast_true(v->value);
		} else if (!strcasecmp(v->name, "echocancelwhenbridged")) {
			confp->chan.echocanbridged = ast_true(v->value);
		} else if (!strcasecmp(v->name, "busydetect")) {
			confp->chan.busydetect = ast_true(v->value);
		} else if (!strcasecmp(v->name, "busycount")) {
			confp->chan.busycount = atoi(v->value);
		} else if (!strcasecmp(v->name, "busypattern")) {
			parse_busy_pattern(v, &confp->chan.busy_cadence);
		} else if (!strcasecmp(v->name, "calledsubscriberheld")) {
			confp->chan.calledsubscriberheld = ast_true(v->value);
		} else if (!strcasecmp(v->name, "lastnumredial")) {
			confp->chan.lastnumredial = ast_true(v->value);
		} else if (!strcasecmp(v->name, "callprogress")) {
			confp->chan.callprogress &= ~CALLPROGRESS_PROGRESS;
			if (ast_true(v->value))
				confp->chan.callprogress |= CALLPROGRESS_PROGRESS;
		} else if (!strcasecmp(v->name, "waitfordialtone")) {
			confp->chan.waitfordialtone = atoi(v->value);
		} else if (!strcasecmp(v->name, "dialtone_detect")) {
			if (!strcasecmp(v->value, "always")) {
				confp->chan.dialtone_detect = -1;
			} else if (ast_true(v->value)) {
				confp->chan.dialtone_detect = DEFAULT_DIALTONE_DETECT_TIMEOUT;
			} else if (ast_false(v->value)) {
				confp->chan.dialtone_detect = 0;
			} else {
				confp->chan.dialtone_detect = ast_strlen_zero(v->value) ? 0 : (8 * atoi(v->value)) / READ_SIZE;
			}
		} else if (!strcasecmp(v->name, "faxdetect")) {
			confp->chan.callprogress &= ~CALLPROGRESS_FAX;
			if (!strcasecmp(v->value, "incoming")) {
				confp->chan.callprogress |= CALLPROGRESS_FAX_INCOMING;
			} else if (!strcasecmp(v->value, "outgoing")) {
				confp->chan.callprogress |= CALLPROGRESS_FAX_OUTGOING;
			} else if (!strcasecmp(v->value, "both") || ast_true(v->value))
				confp->chan.callprogress |= CALLPROGRESS_FAX_INCOMING | CALLPROGRESS_FAX_OUTGOING;
		} else if (!strcasecmp(v->name, "faxdetect_timeout")) {
			if (sscanf(v->value, "%30u", &confp->chan.faxdetect_timeout) != 1) {
				confp->chan.faxdetect_timeout = 0;
			}
		} else if (!strcasecmp(v->name, "firstdigit_timeout")) {
			if (sscanf(v->value, "%30d", &confp->chan.firstdigit_timeout) != 1
				|| confp->chan.firstdigit_timeout <= 0) {
				confp->chan.firstdigit_timeout = ANALOG_FIRST_DIGIT_TIMEOUT;
			}
		} else if (!strcasecmp(v->name, "interdigit_timeout")) {
			if (sscanf(v->value, "%30d", &confp->chan.interdigit_timeout) != 1
				|| confp->chan.interdigit_timeout <= 0) {
				confp->chan.interdigit_timeout = ANALOG_INTER_DIGIT_TIMEOUT;
			}
		} else if (!strcasecmp(v->name, "matchdigit_timeout")) {
			if (sscanf(v->value, "%30d", &confp->chan.matchdigit_timeout) != 1
				|| confp->chan.matchdigit_timeout <= 0) {
				confp->chan.matchdigit_timeout = ANALOG_MATCH_DIGIT_TIMEOUT;
			}
		} else if (!strcasecmp(v->name, "echocancel")) {
			process_echocancel(confp, v->value, v->lineno);
		} else if (!strcasecmp(v->name, "echotraining")) {
			if (sscanf(v->value, "%30d", &y) == 1) {
				if ((y < 10) || (y > 4000)) {
					ast_log(LOG_WARNING, "Echo training time must be within the range of 10 to 4000 ms at line %d.\n", v->lineno);
				} else {
					confp->chan.echotraining = y;
				}
			} else if (ast_true(v->value)) {
				confp->chan.echotraining = 400;
			} else
				confp->chan.echotraining = 0;
		} else if (!strcasecmp(v->name, "hidecallerid")) {
			confp->chan.hidecallerid = ast_true(v->value);
		} else if (!strcasecmp(v->name, "hidecalleridname")) {
			confp->chan.hidecalleridname = ast_true(v->value);
 		} else if (!strcasecmp(v->name, "pulsedial")) {
 			confp->chan.pulse = ast_true(v->value);
		} else if (!strcasecmp(v->name, "callreturn")) {
			confp->chan.callreturn = ast_true(v->value);
		} else if (!strcasecmp(v->name, "callwaiting")) {
			confp->chan.callwaiting = ast_true(v->value);
		} else if (!strcasecmp(v->name, "callwaitingcallerid")) {
			confp->chan.callwaitingcallerid = ast_true(v->value);
		} else if (!strcasecmp(v->name, "callwaitingdeluxe")) {
			confp->chan.callwaitingdeluxe = ast_true(v->value);
		} else if (!strcasecmp(v->name, "context")) {
			ast_copy_string(confp->chan.context, v->value, sizeof(confp->chan.context));
		} else if (!strcasecmp(v->name, "language")) {
			ast_copy_string(confp->chan.language, v->value, sizeof(confp->chan.language));
		} else if (!strcasecmp(v->name, "progzone")) {
			ast_copy_string(progzone, v->value, sizeof(progzone));
		} else if (!strcasecmp(v->name, "mohinterpret")
			||!strcasecmp(v->name, "musiconhold") || !strcasecmp(v->name, "musicclass")) {
			ast_copy_string(confp->chan.mohinterpret, v->value, sizeof(confp->chan.mohinterpret));
		} else if (!strcasecmp(v->name, "mohsuggest")) {
			ast_copy_string(confp->chan.mohsuggest, v->value, sizeof(confp->chan.mohsuggest));
		} else if (!strcasecmp(v->name, "parkinglot")) {
			ast_copy_string(confp->chan.parkinglot, v->value, sizeof(confp->chan.parkinglot));
		} else if (!strcasecmp(v->name, "stripmsd")) {
			ast_log(LOG_NOTICE, "Configuration option \"%s\" has been deprecated. Please use dialplan instead\n", v->name);
			confp->chan.stripmsd = atoi(v->value);
		} else if (!strcasecmp(v->name, "jitterbuffers")) {
			numbufs = atoi(v->value);
		} else if (!strcasecmp(v->name, "group")) {
			confp->chan.group = ast_get_group(v->value);
		} else if (!strcasecmp(v->name, "callgroup")) {
			if (!((confp->chan.sig == SIG_FXOKS) || (confp->chan.sig == SIG_FXOGS) || (confp->chan.sig == SIG_FXOLS))) {
				ast_log(LOG_WARNING, "Only FXO signalled channels may belong to a call group\n");
			}
			if (!strcasecmp(v->value, "none"))
				confp->chan.callgroup = 0;
			else
				confp->chan.callgroup = ast_get_group(v->value);
		} else if (!strcasecmp(v->name, "pickupgroup")) {
			if (!((confp->chan.sig == SIG_FXOKS) || (confp->chan.sig == SIG_FXOGS) || (confp->chan.sig == SIG_FXOLS))) {
				ast_log(LOG_WARNING, "Only FXO signalled channels may belong to a pickup group\n");
			}
			if (!strcasecmp(v->value, "none"))
				confp->chan.pickupgroup = 0;
			else
				confp->chan.pickupgroup = ast_get_group(v->value);
		} else if (!strcasecmp(v->name, "namedcallgroup")) {
			if (!((confp->chan.sig == SIG_FXOKS) || (confp->chan.sig == SIG_FXOGS) || (confp->chan.sig == SIG_FXOLS))) {
				ast_log(LOG_WARNING, "Only FXO signalled channels may belong to a named call group\n");
			}
			confp->chan.named_callgroups = ast_get_namedgroups(v->value);
		} else if (!strcasecmp(v->name, "namedpickupgroup")) {
			if (!((confp->chan.sig == SIG_FXOKS) || (confp->chan.sig == SIG_FXOGS) || (confp->chan.sig == SIG_FXOLS))) {
				ast_log(LOG_WARNING, "Only FXO signalled channels may belong to a named pickup group\n");
			}
			confp->chan.named_pickupgroups = ast_get_namedgroups(v->value);
		} else if (!strcasecmp(v->name, "setvar")) {
			if (v->value) {
				char *varval = NULL;
				struct ast_variable *tmpvar;
				char varname[strlen(v->value) + 1];
				strcpy(varname, v->value); /* safe */
				if ((varval = strchr(varname, '='))) {
					*varval++ = '\0';
					if ((tmpvar = ast_variable_new(varname, varval, ""))) {
						if (ast_variable_list_replace(&confp->chan.vars, tmpvar)) {
							tmpvar->next = confp->chan.vars;
							confp->chan.vars = tmpvar;
						}
					}
				}
			}
		} else if (!strcasecmp(v->name, "immediate")) {
			confp->chan.immediate = ast_true(v->value);
		} else if (!strcasecmp(v->name, "immediatering")) {
			confp->chan.immediatering = ast_true(v->value);
		} else if (!strcasecmp(v->name, "transfertobusy")) {
			confp->chan.transfertobusy = ast_true(v->value);
		} else if (!strcasecmp(v->name, "dialmode")) {
			if (!strcasecmp(v->value, "pulse")) {
				confp->chan.permdialmode = ANALOG_DIALMODE_PULSE;
			} else if (!strcasecmp(v->value, "dtmf") || !strcasecmp(v->value, "tone")) {
				confp->chan.permdialmode = ANALOG_DIALMODE_DTMF;
			} else if (!strcasecmp(v->value, "none")) {
				confp->chan.permdialmode = ANALOG_DIALMODE_NONE;
			} else {
				confp->chan.permdialmode = ANALOG_DIALMODE_BOTH;
			}
		} else if (!strcasecmp(v->name, "mwimonitor")) {
			confp->chan.mwimonitor_neon = 0;
			confp->chan.mwimonitor_fsk = 0;
			confp->chan.mwimonitor_rpas = 0;
			if (strcasestr(v->value, "fsk")) {
				confp->chan.mwimonitor_fsk = 1;
			}
			if (strcasestr(v->value, "rpas")) {
				confp->chan.mwimonitor_rpas = 1;
			}
			if (strcasestr(v->value, "neon")) {
				confp->chan.mwimonitor_neon = 1;
			}
			/* If set to true or yes, assume that simple fsk is desired */
			if (ast_true(v->value)) {
				confp->chan.mwimonitor_fsk = 1;
			}
		} else if (!strcasecmp(v->name, "hwrxgain")) {
			confp->chan.hwrxgain_enabled = 0;
			if (strcasecmp(v->value, "disabled")) {
				if (sscanf(v->value, "%30f", &confp->chan.hwrxgain) == 1) {
					confp->chan.hwrxgain_enabled = 1;
				} else {
					ast_log(LOG_WARNING, "Invalid hwrxgain: %s at line %d.\n", v->value, v->lineno);
				}
			}
		} else if (!strcasecmp(v->name, "hwtxgain")) {
			confp->chan.hwtxgain_enabled = 0;
			if (strcasecmp(v->value, "disabled")) {
				if (sscanf(v->value, "%30f", &confp->chan.hwtxgain) == 1) {
					confp->chan.hwtxgain_enabled = 1;
				} else {
					ast_log(LOG_WARNING, "Invalid hwtxgain: %s at line %d.\n", v->value, v->lineno);
				}
			}
		} else if (!strcasecmp(v->name, "cid_rxgain")) {
			if (sscanf(v->value, "%30f", &confp->chan.cid_rxgain) != 1) {
				ast_log(LOG_WARNING, "Invalid cid_rxgain: %s at line %d.\n", v->value, v->lineno);
			}
		} else if (!strcasecmp(v->name, "rxgain")) {
			if (sscanf(v->value, "%30f", &confp->chan.rxgain) != 1) {
				ast_log(LOG_WARNING, "Invalid rxgain: %s at line %d.\n", v->value, v->lineno);
			}
		} else if (!strcasecmp(v->name, "txgain")) {
			if (sscanf(v->value, "%30f", &confp->chan.txgain) != 1) {
				ast_log(LOG_WARNING, "Invalid txgain: %s at line %d.\n", v->value, v->lineno);
			}
		} else if (!strcasecmp(v->name, "txdrc")) {
			if (sscanf(v->value, "%f", &confp->chan.txdrc) != 1) {
				ast_log(LOG_WARNING, "Invalid txdrc: %s\n", v->value);
			}
		} else if (!strcasecmp(v->name, "rxdrc")) {
			if (sscanf(v->value, "%f", &confp->chan.rxdrc) != 1) {
				ast_log(LOG_WARNING, "Invalid rxdrc: %s\n", v->value);
			}
		} else if (!strcasecmp(v->name, "tonezone")) {
			if (sscanf(v->value, "%30d", &confp->chan.tonezone) != 1) {
				ast_log(LOG_WARNING, "Invalid tonezone: %s at line %d.\n", v->value, v->lineno);
			}
		} else if (!strcasecmp(v->name, "callerid")) {
			if (!strcasecmp(v->value, "asreceived")) {
				confp->chan.cid_num[0] = '\0';
				confp->chan.cid_name[0] = '\0';
			} else {
				ast_callerid_split(v->value, confp->chan.cid_name, sizeof(confp->chan.cid_name), confp->chan.cid_num, sizeof(confp->chan.cid_num));
			}
		} else if (!strcasecmp(v->name, "fullname")) {
			ast_copy_string(confp->chan.cid_name, v->value, sizeof(confp->chan.cid_name));
		} else if (!strcasecmp(v->name, "cid_number")) {
			ast_copy_string(confp->chan.cid_num, v->value, sizeof(confp->chan.cid_num));
		} else if (!strcasecmp(v->name, "cid_tag")) {
			ast_copy_string(confp->chan.cid_tag, v->value, sizeof(confp->chan.cid_tag));
		} else if (!strcasecmp(v->name, "useincomingcalleridondahditransfer")) {
			confp->chan.dahditrcallerid = ast_true(v->value);
		} else if (!strcasecmp(v->name, "restrictcid")) {
			confp->chan.restrictcid = ast_true(v->value);
		} else if (!strcasecmp(v->name, "usecallingpres")) {
			confp->chan.use_callingpres = ast_true(v->value);
		} else if (!strcasecmp(v->name, "accountcode")) {
			ast_copy_string(confp->chan.accountcode, v->value, sizeof(confp->chan.accountcode));
		} else if (!strcasecmp(v->name, "amaflags")) {
			y = ast_channel_string2amaflag(v->value);
			if (y < 0)
				ast_log(LOG_WARNING, "Invalid AMA flags: %s at line %d.\n", v->value, v->lineno);
			else
				confp->chan.amaflags = y;
		} else if (!strcasecmp(v->name, "polarityonanswerdelay")) {
			confp->chan.polarityonanswerdelay = atoi(v->value);
		} else if (!strcasecmp(v->name, "answeronpolarityswitch")) {
			confp->chan.answeronpolarityswitch = ast_true(v->value);
		} else if (!strcasecmp(v->name, "ani_info_digits")) {
			confp->chan.ani_info_digits = atoi(v->value);
		} else if (!strcasecmp(v->name, "ani_wink_time")) {
			confp->chan.ani_wink_time = atoi(v->value);
		} else if (!strcasecmp(v->name, "ani_timeout")) {
			confp->chan.ani_timeout = atoi(v->value);
		} else if (!strcasecmp(v->name, "hanguponpolarityswitch")) {
			confp->chan.hanguponpolarityswitch = ast_true(v->value);
		} else if (!strcasecmp(v->name, "autoreoriginate")) {
			confp->chan.reoriginate = ast_true(v->value);
		} else if (!strcasecmp(v->name, "sendcalleridafter")) {
			confp->chan.sendcalleridafter = atoi(v->value);
		} else if (!strcasecmp(v->name, "mwimonitornotify")) {
			ast_copy_string(mwimonitornotify, v->value, sizeof(mwimonitornotify));
		} else if (ast_cc_is_config_param(v->name)) {
			ast_cc_set_param(confp->chan.cc_params, v->name, v->value);
		} else if (!strcasecmp(v->name, "mwisendtype")) {
#ifndef HAVE_DAHDI_LINEREVERSE_VMWI  /* backward compatibility for older dahdi VMWI implementation */
			if (!strcasecmp(v->value, "rpas")) { /* Ring Pulse Alert Signal */
				mwisend_rpas = 1;
			} else {
				mwisend_rpas = 0;
			}
#else
			/* Default is fsk, to turn it off you must specify nofsk */
			memset(&confp->chan.mwisend_setting, 0, sizeof(confp->chan.mwisend_setting));
			if (strcasestr(v->value, "nofsk")) {		/* NoFSK */
				confp->chan.mwisend_fsk = 0;
			} else {					/* Default FSK */
				confp->chan.mwisend_fsk = 1;
			}
			if (strcasestr(v->value, "rpas")) {		/* Ring Pulse Alert Signal, normally followed by FSK */
				confp->chan.mwisend_rpas = 1;
			} else {
				confp->chan.mwisend_rpas = 0;
			}
			if (strcasestr(v->value, "lrev")) {		/* Line Reversal */
				confp->chan.mwisend_setting.vmwi_type |= DAHDI_VMWI_LREV;
			}
			if (strcasestr(v->value, "hvdc")) {		/* HV 90VDC */
				confp->chan.mwisend_setting.vmwi_type |= DAHDI_VMWI_HVDC;
			}
			if ( (strcasestr(v->value, "neon")) || (strcasestr(v->value, "hvac")) ) {	/* 90V DC pulses */
				confp->chan.mwisend_setting.vmwi_type |= DAHDI_VMWI_HVAC;
			}
#endif
		} else if (reload != 1) {
			 if (!strcasecmp(v->name, "signalling") || !strcasecmp(v->name, "signaling")) {
				int orig_radio = confp->chan.radio;
				int orig_outsigmod = confp->chan.outsigmod;
				int orig_auto = confp->is_sig_auto;

				confp->chan.radio = 0;
				confp->chan.outsigmod = -1;
				confp->is_sig_auto = 0;
				if (!strcasecmp(v->value, "em")) {
					confp->chan.sig = SIG_EM;
				} else if (!strcasecmp(v->value, "em_e1")) {
					confp->chan.sig = SIG_EM_E1;
				} else if (!strcasecmp(v->value, "em_w")) {
					confp->chan.sig = SIG_EMWINK;
				} else if (!strcasecmp(v->value, "fxs_ls")) {
					confp->chan.sig = SIG_FXSLS;
				} else if (!strcasecmp(v->value, "fxs_gs")) {
					confp->chan.sig = SIG_FXSGS;
				} else if (!strcasecmp(v->value, "fxs_ks")) {
					confp->chan.sig = SIG_FXSKS;
				} else if (!strcasecmp(v->value, "fxo_ls")) {
					confp->chan.sig = SIG_FXOLS;
				} else if (!strcasecmp(v->value, "fxo_gs")) {
					confp->chan.sig = SIG_FXOGS;
				} else if (!strcasecmp(v->value, "fxo_ks")) {
					confp->chan.sig = SIG_FXOKS;
				} else if (!strcasecmp(v->value, "fxs_rx")) {
					confp->chan.sig = SIG_FXSKS;
					confp->chan.radio = 1;
				} else if (!strcasecmp(v->value, "fxo_rx")) {
					confp->chan.sig = SIG_FXOLS;
					confp->chan.radio = 1;
				} else if (!strcasecmp(v->value, "fxs_tx")) {
					confp->chan.sig = SIG_FXSLS;
					confp->chan.radio = 1;
				} else if (!strcasecmp(v->value, "fxo_tx")) {
					confp->chan.sig = SIG_FXOGS;
					confp->chan.radio = 1;
				} else if (!strcasecmp(v->value, "em_rx")) {
					confp->chan.sig = SIG_EM;
					confp->chan.radio = 1;
				} else if (!strcasecmp(v->value, "em_tx")) {
					confp->chan.sig = SIG_EM;
					confp->chan.radio = 1;
				} else if (!strcasecmp(v->value, "em_rxtx")) {
					confp->chan.sig = SIG_EM;
					confp->chan.radio = 2;
				} else if (!strcasecmp(v->value, "em_txrx")) {
					confp->chan.sig = SIG_EM;
					confp->chan.radio = 2;
				} else if (!strcasecmp(v->value, "sf")) {
					confp->chan.sig = SIG_SF;
				} else if (!strcasecmp(v->value, "sf_w")) {
					confp->chan.sig = SIG_SFWINK;
				} else if (!strcasecmp(v->value, "sf_featd")) {
					confp->chan.sig = SIG_FEATD;
				} else if (!strcasecmp(v->value, "sf_featdmf")) {
					confp->chan.sig = SIG_FEATDMF;
				} else if (!strcasecmp(v->value, "sf_featb")) {
					confp->chan.sig = SIG_SF_FEATB;
				} else if (!strcasecmp(v->value, "sf")) {
					confp->chan.sig = SIG_SF;
				} else if (!strcasecmp(v->value, "sf_rx")) {
					confp->chan.sig = SIG_SF;
					confp->chan.radio = 1;
				} else if (!strcasecmp(v->value, "sf_tx")) {
					confp->chan.sig = SIG_SF;
					confp->chan.radio = 1;
				} else if (!strcasecmp(v->value, "sf_rxtx")) {
					confp->chan.sig = SIG_SF;
					confp->chan.radio = 2;
				} else if (!strcasecmp(v->value, "sf_txrx")) {
					confp->chan.sig = SIG_SF;
					confp->chan.radio = 2;
				} else if (!strcasecmp(v->value, "featd")) {
					confp->chan.sig = SIG_FEATD;
				} else if (!strcasecmp(v->value, "featdmf")) {
					confp->chan.sig = SIG_FEATDMF;
				} else if (!strcasecmp(v->value, "featdmf_ta")) {
					confp->chan.sig = SIG_FEATDMF_TA;
				} else if (!strcasecmp(v->value, "e911")) {
					confp->chan.sig = SIG_E911;
				} else if (!strcasecmp(v->value, "fgccama")) {
					confp->chan.sig = SIG_FGC_CAMA;
				} else if (!strcasecmp(v->value, "fgccamamf")) {
					confp->chan.sig = SIG_FGC_CAMAMF;
				} else if (!strcasecmp(v->value, "featb")) {
					confp->chan.sig = SIG_FEATB;
#ifdef HAVE_PRI
				} else if (!strcasecmp(v->value, "pri_net")) {
					confp->chan.sig = SIG_PRI;
					confp->pri.pri.nodetype = PRI_NETWORK;
				} else if (!strcasecmp(v->value, "pri_cpe")) {
					confp->chan.sig = SIG_PRI;
					confp->pri.pri.nodetype = PRI_CPE;
				} else if (!strcasecmp(v->value, "bri_cpe")) {
					confp->chan.sig = SIG_BRI;
					confp->pri.pri.nodetype = PRI_CPE;
				} else if (!strcasecmp(v->value, "bri_net")) {
					confp->chan.sig = SIG_BRI;
					confp->pri.pri.nodetype = PRI_NETWORK;
				} else if (!strcasecmp(v->value, "bri_cpe_ptmp")) {
					confp->chan.sig = SIG_BRI_PTMP;
					confp->pri.pri.nodetype = PRI_CPE;
				} else if (!strcasecmp(v->value, "bri_net_ptmp")) {
#if defined(HAVE_PRI_CALL_HOLD)
					confp->chan.sig = SIG_BRI_PTMP;
					confp->pri.pri.nodetype = PRI_NETWORK;
#else
					ast_log(LOG_WARNING, "How cool would it be if someone implemented this mode!  For now, sucks for you. (line %d)\n", v->lineno);
#endif	/* !defined(HAVE_PRI_CALL_HOLD) */
#endif
#if defined(HAVE_SS7)
				} else if (!strcasecmp(v->value, "ss7")) {
					confp->chan.sig = SIG_SS7;
#endif	/* defined(HAVE_SS7) */
#ifdef HAVE_OPENR2
				} else if (!strcasecmp(v->value, "mfcr2")) {
					confp->chan.sig = SIG_MFCR2;
#endif
				} else if (!strcasecmp(v->value, "auto")) {
					confp->is_sig_auto = 1;
				} else {
					confp->chan.outsigmod = orig_outsigmod;
					confp->chan.radio = orig_radio;
					confp->is_sig_auto = orig_auto;
					ast_log(LOG_ERROR, "Unknown signalling method '%s' at line %d.\n", v->value, v->lineno);
				}
			 } else if (!strcasecmp(v->name, "outsignalling") || !strcasecmp(v->name, "outsignaling")) {
				if (!strcasecmp(v->value, "em")) {
					confp->chan.outsigmod = SIG_EM;
				} else if (!strcasecmp(v->value, "em_e1")) {
					confp->chan.outsigmod = SIG_EM_E1;
				} else if (!strcasecmp(v->value, "em_w")) {
					confp->chan.outsigmod = SIG_EMWINK;
				} else if (!strcasecmp(v->value, "sf")) {
					confp->chan.outsigmod = SIG_SF;
				} else if (!strcasecmp(v->value, "sf_w")) {
					confp->chan.outsigmod = SIG_SFWINK;
				} else if (!strcasecmp(v->value, "sf_featd")) {
					confp->chan.outsigmod = SIG_FEATD;
				} else if (!strcasecmp(v->value, "sf_featdmf")) {
					confp->chan.outsigmod = SIG_FEATDMF;
				} else if (!strcasecmp(v->value, "sf_featb")) {
					confp->chan.outsigmod = SIG_SF_FEATB;
				} else if (!strcasecmp(v->value, "sf")) {
					confp->chan.outsigmod = SIG_SF;
				} else if (!strcasecmp(v->value, "featd")) {
					confp->chan.outsigmod = SIG_FEATD;
				} else if (!strcasecmp(v->value, "featdmf")) {
					confp->chan.outsigmod = SIG_FEATDMF;
				} else if (!strcasecmp(v->value, "featdmf_ta")) {
					confp->chan.outsigmod = SIG_FEATDMF_TA;
				} else if (!strcasecmp(v->value, "e911")) {
					confp->chan.outsigmod = SIG_E911;
				} else if (!strcasecmp(v->value, "fgccama")) {
					confp->chan.outsigmod = SIG_FGC_CAMA;
				} else if (!strcasecmp(v->value, "fgccamamf")) {
					confp->chan.outsigmod = SIG_FGC_CAMAMF;
				} else if (!strcasecmp(v->value, "featb")) {
					confp->chan.outsigmod = SIG_FEATB;
				} else {
					ast_log(LOG_ERROR, "Unknown signalling method '%s' at line %d.\n", v->value, v->lineno);
				}
#ifdef HAVE_PRI
			} else if (!strcasecmp(v->name, "pridialplan")) {
				if (!strcasecmp(v->value, "national")) {
					confp->pri.pri.dialplan = PRI_NATIONAL_ISDN + 1;
				} else if (!strcasecmp(v->value, "unknown")) {
					confp->pri.pri.dialplan = PRI_UNKNOWN + 1;
				} else if (!strcasecmp(v->value, "private")) {
					confp->pri.pri.dialplan = PRI_PRIVATE + 1;
				} else if (!strcasecmp(v->value, "international")) {
					confp->pri.pri.dialplan = PRI_INTERNATIONAL_ISDN + 1;
				} else if (!strcasecmp(v->value, "local")) {
					confp->pri.pri.dialplan = PRI_LOCAL_ISDN + 1;
	 			} else if (!strcasecmp(v->value, "dynamic")) {
 					confp->pri.pri.dialplan = -1;
				} else if (!strcasecmp(v->value, "redundant")) {
					confp->pri.pri.dialplan = -2;
				} else {
					ast_log(LOG_WARNING, "Unknown PRI dialplan '%s' at line %d.\n", v->value, v->lineno);
				}
			} else if (!strcasecmp(v->name, "prilocaldialplan")) {
				if (!strcasecmp(v->value, "national")) {
					confp->pri.pri.localdialplan = PRI_NATIONAL_ISDN + 1;
				} else if (!strcasecmp(v->value, "unknown")) {
					confp->pri.pri.localdialplan = PRI_UNKNOWN + 1;
				} else if (!strcasecmp(v->value, "private")) {
					confp->pri.pri.localdialplan = PRI_PRIVATE + 1;
				} else if (!strcasecmp(v->value, "international")) {
					confp->pri.pri.localdialplan = PRI_INTERNATIONAL_ISDN + 1;
				} else if (!strcasecmp(v->value, "local")) {
					confp->pri.pri.localdialplan = PRI_LOCAL_ISDN + 1;
				} else if (!strcasecmp(v->value, "from_channel")) {
					confp->pri.pri.localdialplan = 0;
				} else if (!strcasecmp(v->value, "dynamic")) {
					confp->pri.pri.localdialplan = -1;
				} else if (!strcasecmp(v->value, "redundant")) {
					confp->pri.pri.localdialplan = -2;
				} else {
					ast_log(LOG_WARNING, "Unknown PRI localdialplan '%s' at line %d.\n", v->value, v->lineno);
				}
			} else if (!strcasecmp(v->name, "pricpndialplan")) {
				if (!strcasecmp(v->value, "national")) {
					confp->pri.pri.cpndialplan = PRI_NATIONAL_ISDN + 1;
				} else if (!strcasecmp(v->value, "unknown")) {
					confp->pri.pri.cpndialplan = PRI_UNKNOWN + 1;
				} else if (!strcasecmp(v->value, "private")) {
					confp->pri.pri.cpndialplan = PRI_PRIVATE + 1;
				} else if (!strcasecmp(v->value, "international")) {
					confp->pri.pri.cpndialplan = PRI_INTERNATIONAL_ISDN + 1;
				} else if (!strcasecmp(v->value, "local")) {
					confp->pri.pri.cpndialplan = PRI_LOCAL_ISDN + 1;
				} else if (!strcasecmp(v->value, "from_channel")) {
					confp->pri.pri.cpndialplan = 0;
				} else if (!strcasecmp(v->value, "dynamic")) {
					confp->pri.pri.cpndialplan = -1;
				} else if (!strcasecmp(v->value, "redundant")) {
					confp->pri.pri.cpndialplan = -2;
				} else {
					ast_log(LOG_WARNING, "Unknown PRI cpndialplan '%s' at line %d.\n", v->value, v->lineno);
				}
			} else if (!strcasecmp(v->name, "switchtype")) {
				if (!strcasecmp(v->value, "national"))
					confp->pri.pri.switchtype = PRI_SWITCH_NI2;
				else if (!strcasecmp(v->value, "ni1"))
					confp->pri.pri.switchtype = PRI_SWITCH_NI1;
				else if (!strcasecmp(v->value, "dms100"))
					confp->pri.pri.switchtype = PRI_SWITCH_DMS100;
				else if (!strcasecmp(v->value, "4ess"))
					confp->pri.pri.switchtype = PRI_SWITCH_ATT4ESS;
				else if (!strcasecmp(v->value, "5ess"))
					confp->pri.pri.switchtype = PRI_SWITCH_LUCENT5E;
				else if (!strcasecmp(v->value, "euroisdn"))
					confp->pri.pri.switchtype = PRI_SWITCH_EUROISDN_E1;
				else if (!strcasecmp(v->value, "qsig"))
					confp->pri.pri.switchtype = PRI_SWITCH_QSIG;
				else {
					ast_log(LOG_ERROR, "Unknown switchtype '%s' at line %d.\n", v->value, v->lineno);
					return -1;
				}
			} else if (!strcasecmp(v->name, "msn")) {
				ast_copy_string(confp->pri.pri.msn_list, v->value,
					sizeof(confp->pri.pri.msn_list));
			} else if (!strcasecmp(v->name, "nsf")) {
				if (!strcasecmp(v->value, "sdn"))
					confp->pri.pri.nsf = PRI_NSF_SDN;
				else if (!strcasecmp(v->value, "megacom"))
					confp->pri.pri.nsf = PRI_NSF_MEGACOM;
				else if (!strcasecmp(v->value, "tollfreemegacom"))
					confp->pri.pri.nsf = PRI_NSF_TOLL_FREE_MEGACOM;
				else if (!strcasecmp(v->value, "accunet"))
					confp->pri.pri.nsf = PRI_NSF_ACCUNET;
				else if (!strcasecmp(v->value, "none"))
					confp->pri.pri.nsf = PRI_NSF_NONE;
				else {
					ast_log(LOG_WARNING, "Unknown network-specific facility '%s' at line %d.\n", v->value, v->lineno);
					confp->pri.pri.nsf = PRI_NSF_NONE;
				}
			} else if (!strcasecmp(v->name, "priindication")) {
				if (!strcasecmp(v->value, "outofband"))
					confp->chan.priindication_oob = 1;
				else if (!strcasecmp(v->value, "inband"))
					confp->chan.priindication_oob = 0;
				else
					ast_log(LOG_WARNING, "'%s' is not a valid pri indication value, should be 'inband' or 'outofband' at line %d.\n",
						v->value, v->lineno);
			} else if (!strcasecmp(v->name, "priexclusive")) {
				confp->chan.priexclusive = ast_true(v->value);
			} else if (!strcasecmp(v->name, "internationalprefix")) {
				ast_copy_string(confp->pri.pri.internationalprefix, v->value, sizeof(confp->pri.pri.internationalprefix));
			} else if (!strcasecmp(v->name, "nationalprefix")) {
				ast_copy_string(confp->pri.pri.nationalprefix, v->value, sizeof(confp->pri.pri.nationalprefix));
			} else if (!strcasecmp(v->name, "localprefix")) {
				ast_copy_string(confp->pri.pri.localprefix, v->value, sizeof(confp->pri.pri.localprefix));
			} else if (!strcasecmp(v->name, "privateprefix")) {
				ast_copy_string(confp->pri.pri.privateprefix, v->value, sizeof(confp->pri.pri.privateprefix));
			} else if (!strcasecmp(v->name, "unknownprefix")) {
				ast_copy_string(confp->pri.pri.unknownprefix, v->value, sizeof(confp->pri.pri.unknownprefix));
			} else if (!strcasecmp(v->name, "resetinterval")) {
				if (!strcasecmp(v->value, "never"))
					confp->pri.pri.resetinterval = -1;
				else if (atoi(v->value) >= 60)
					confp->pri.pri.resetinterval = atoi(v->value);
				else
					ast_log(LOG_WARNING, "'%s' is not a valid reset interval, should be >= 60 seconds or 'never' at line %d.\n",
						v->value, v->lineno);
			} else if (!strcasecmp(v->name, "force_restart_unavailable_chans")) {
				confp->pri.pri.force_restart_unavailable_chans = ast_true(v->value);
			} else if (!strcasecmp(v->name, "minunused")) {
				confp->pri.pri.minunused = atoi(v->value);
			} else if (!strcasecmp(v->name, "minidle")) {
				confp->pri.pri.minidle = atoi(v->value);
			} else if (!strcasecmp(v->name, "idleext")) {
				ast_copy_string(confp->pri.pri.idleext, v->value, sizeof(confp->pri.pri.idleext));
			} else if (!strcasecmp(v->name, "idledial")) {
				ast_copy_string(confp->pri.pri.idledial, v->value, sizeof(confp->pri.pri.idledial));
			} else if (!strcasecmp(v->name, "overlapdial")) {
				if (ast_true(v->value)) {
					confp->pri.pri.overlapdial = DAHDI_OVERLAPDIAL_BOTH;
				} else if (!strcasecmp(v->value, "incoming")) {
					confp->pri.pri.overlapdial = DAHDI_OVERLAPDIAL_INCOMING;
				} else if (!strcasecmp(v->value, "outgoing")) {
					confp->pri.pri.overlapdial = DAHDI_OVERLAPDIAL_OUTGOING;
				} else if (!strcasecmp(v->value, "both") || ast_true(v->value)) {
					confp->pri.pri.overlapdial = DAHDI_OVERLAPDIAL_BOTH;
				} else {
					confp->pri.pri.overlapdial = DAHDI_OVERLAPDIAL_NONE;
				}
#ifdef HAVE_PRI_PROG_W_CAUSE
			} else if (!strcasecmp(v->name, "qsigchannelmapping")) {
				if (!strcasecmp(v->value, "logical")) {
					confp->pri.pri.qsigchannelmapping = DAHDI_CHAN_MAPPING_LOGICAL;
				} else if (!strcasecmp(v->value, "physical")) {
					confp->pri.pri.qsigchannelmapping = DAHDI_CHAN_MAPPING_PHYSICAL;
				} else {
					confp->pri.pri.qsigchannelmapping = DAHDI_CHAN_MAPPING_PHYSICAL;
				}
#endif
			} else if (!strcasecmp(v->name, "discardremoteholdretrieval")) {
				confp->pri.pri.discardremoteholdretrieval = ast_true(v->value);
#if defined(HAVE_PRI_SERVICE_MESSAGES)
			} else if (!strcasecmp(v->name, "service_message_support")) {
				/* assuming switchtype for this channel group has been configured already */
				if ((confp->pri.pri.switchtype == PRI_SWITCH_ATT4ESS
					|| confp->pri.pri.switchtype == PRI_SWITCH_LUCENT5E
					|| confp->pri.pri.switchtype == PRI_SWITCH_NI2) && ast_true(v->value)) {
					confp->pri.pri.enable_service_message_support = 1;
				} else {
					confp->pri.pri.enable_service_message_support = 0;
				}
#endif	/* defined(HAVE_PRI_SERVICE_MESSAGES) */
#ifdef HAVE_PRI_INBANDDISCONNECT
			} else if (!strcasecmp(v->name, "inbanddisconnect")) {
				confp->pri.pri.inbanddisconnect = ast_true(v->value);
#endif
			} else if (!strcasecmp(v->name, "pritimer")) {
#ifdef PRI_GETSET_TIMERS
				char tmp[20];
				char *timerc;
				char *c;
				int timer;
				int timeridx;

				ast_copy_string(tmp, v->value, sizeof(tmp));
				c = tmp;
				timerc = strsep(&c, ",");
				if (!ast_strlen_zero(timerc) && !ast_strlen_zero(c)) {
					timeridx = pri_timer2idx(timerc);
					timer = atoi(c);
					if (timeridx < 0 || PRI_MAX_TIMERS <= timeridx) {
						ast_log(LOG_WARNING,
							"'%s' is not a valid ISDN timer at line %d.\n", timerc,
							v->lineno);
					} else if (!timer) {
						ast_log(LOG_WARNING,
							"'%s' is not a valid value for ISDN timer '%s' at line %d.\n",
							c, timerc, v->lineno);
					} else {
						confp->pri.pri.pritimers[timeridx] = timer;
					}
				} else {
					ast_log(LOG_WARNING,
						"'%s' is not a valid ISDN timer configuration string at line %d.\n",
						v->value, v->lineno);
				}
#endif /* PRI_GETSET_TIMERS */
			} else if (!strcasecmp(v->name, "facilityenable")) {
				confp->pri.pri.facilityenable = ast_true(v->value);
#if defined(HAVE_PRI_AOC_EVENTS)
			} else if (!strcasecmp(v->name, "aoc_enable")) {
				confp->pri.pri.aoc_passthrough_flag = 0;
				if (strchr(v->value, 's') || strchr(v->value, 'S')) {
					confp->pri.pri.aoc_passthrough_flag |= SIG_PRI_AOC_GRANT_S;
				}
				if (strchr(v->value, 'd') || strchr(v->value, 'D')) {
					confp->pri.pri.aoc_passthrough_flag |= SIG_PRI_AOC_GRANT_D;
				}
				if (strchr(v->value, 'e') || strchr(v->value, 'E')) {
					confp->pri.pri.aoc_passthrough_flag |= SIG_PRI_AOC_GRANT_E;
				}
			} else if (!strcasecmp(v->name, "aoce_delayhangup")) {
				confp->pri.pri.aoce_delayhangup = ast_true(v->value);
#endif	/* defined(HAVE_PRI_AOC_EVENTS) */
#if defined(HAVE_PRI_CALL_HOLD)
			} else if (!strcasecmp(v->name, "hold_disconnect_transfer")) {
				confp->pri.pri.hold_disconnect_transfer = ast_true(v->value);
#endif	/* defined(HAVE_PRI_CALL_HOLD) */
			} else if (!strcasecmp(v->name, "moh_signaling")
				|| !strcasecmp(v->name, "moh_signalling")) {
				if (!strcasecmp(v->value, "moh")) {
					confp->pri.pri.moh_signaling = SIG_PRI_MOH_SIGNALING_MOH;
				} else if (!strcasecmp(v->value, "notify")) {
					confp->pri.pri.moh_signaling = SIG_PRI_MOH_SIGNALING_NOTIFY;
#if defined(HAVE_PRI_CALL_HOLD)
				} else if (!strcasecmp(v->value, "hold")) {
					confp->pri.pri.moh_signaling = SIG_PRI_MOH_SIGNALING_HOLD;
#endif	/* defined(HAVE_PRI_CALL_HOLD) */
				} else {
					confp->pri.pri.moh_signaling = SIG_PRI_MOH_SIGNALING_MOH;
				}
#if defined(HAVE_PRI_CCSS)
			} else if (!strcasecmp(v->name, "cc_ptmp_recall_mode")) {
				if (!strcasecmp(v->value, "global")) {
					confp->pri.pri.cc_ptmp_recall_mode = 0;/* globalRecall */
				} else if (!strcasecmp(v->value, "specific")) {
					confp->pri.pri.cc_ptmp_recall_mode = 1;/* specificRecall */
				} else {
					confp->pri.pri.cc_ptmp_recall_mode = 1;/* specificRecall */
				}
			} else if (!strcasecmp(v->name, "cc_qsig_signaling_link_req")) {
				if (!strcasecmp(v->value, "release")) {
					confp->pri.pri.cc_qsig_signaling_link_req = 0;/* release */
				} else if (!strcasecmp(v->value, "retain")) {
					confp->pri.pri.cc_qsig_signaling_link_req = 1;/* retain */
				} else if (!strcasecmp(v->value, "do_not_care")) {
					confp->pri.pri.cc_qsig_signaling_link_req = 2;/* do-not-care */
				} else {
					confp->pri.pri.cc_qsig_signaling_link_req = 1;/* retain */
				}
			} else if (!strcasecmp(v->name, "cc_qsig_signaling_link_rsp")) {
				if (!strcasecmp(v->value, "release")) {
					confp->pri.pri.cc_qsig_signaling_link_rsp = 0;/* release */
				} else if (!strcasecmp(v->value, "retain")) {
					confp->pri.pri.cc_qsig_signaling_link_rsp = 1;/* retain */
				} else {
					confp->pri.pri.cc_qsig_signaling_link_rsp = 1;/* retain */
				}
#endif	/* defined(HAVE_PRI_CCSS) */
#if defined(HAVE_PRI_CALL_WAITING)
			} else if (!strcasecmp(v->name, "max_call_waiting_calls")) {
				confp->pri.pri.max_call_waiting_calls = atoi(v->value);
				if (confp->pri.pri.max_call_waiting_calls < 0) {
					/* Negative values are not allowed. */
					confp->pri.pri.max_call_waiting_calls = 0;
				}
			} else if (!strcasecmp(v->name, "allow_call_waiting_calls")) {
				confp->pri.pri.allow_call_waiting_calls = ast_true(v->value);
#endif	/* defined(HAVE_PRI_CALL_WAITING) */
#if defined(HAVE_PRI_MWI)
			} else if (!strcasecmp(v->name, "mwi_mailboxes")) {
				ast_copy_string(confp->pri.pri.mwi_mailboxes, v->value,
					sizeof(confp->pri.pri.mwi_mailboxes));
			} else if (!strcasecmp(v->name, "mwi_vm_boxes")) {
				ast_copy_string(confp->pri.pri.mwi_vm_boxes, v->value,
					sizeof(confp->pri.pri.mwi_vm_boxes));
			} else if (!strcasecmp(v->name, "mwi_vm_numbers")) {
				ast_copy_string(confp->pri.pri.mwi_vm_numbers, v->value,
					sizeof(confp->pri.pri.mwi_vm_numbers));
#endif	/* defined(HAVE_PRI_MWI) */
			} else if (!strcasecmp(v->name, "append_msn_to_cid_tag")) {
				confp->pri.pri.append_msn_to_user_tag = ast_true(v->value);
			} else if (!strcasecmp(v->name, "inband_on_setup_ack")) {
				confp->pri.pri.inband_on_setup_ack = ast_true(v->value);
			} else if (!strcasecmp(v->name, "inband_on_proceeding")) {
				confp->pri.pri.inband_on_proceeding = ast_true(v->value);
#if defined(HAVE_PRI_DISPLAY_TEXT)
			} else if (!strcasecmp(v->name, "display_send")) {
				confp->pri.pri.display_flags_send = dahdi_display_text_option(v->value);
			} else if (!strcasecmp(v->name, "display_receive")) {
				confp->pri.pri.display_flags_receive = dahdi_display_text_option(v->value);
#endif	/* defined(HAVE_PRI_DISPLAY_TEXT) */
#if defined(HAVE_PRI_MCID)
			} else if (!strcasecmp(v->name, "mcid_send")) {
				confp->pri.pri.mcid_send = ast_true(v->value);
#endif	/* defined(HAVE_PRI_MCID) */
#if defined(HAVE_PRI_DATETIME_SEND)
			} else if (!strcasecmp(v->name, "datetime_send")) {
				confp->pri.pri.datetime_send = dahdi_datetime_send_option(v->value);
#endif	/* defined(HAVE_PRI_DATETIME_SEND) */
			} else if (!strcasecmp(v->name, "layer1_presence")) {
				if (!strcasecmp(v->value, "required")) {
					confp->pri.pri.layer1_ignored = 0;
				} else if (!strcasecmp(v->value, "ignore")) {
					confp->pri.pri.layer1_ignored = 1;
				} else {
					/* Default */
					confp->pri.pri.layer1_ignored = 0;
				}
#if defined(HAVE_PRI_L2_PERSISTENCE)
			} else if (!strcasecmp(v->name, "layer2_persistence")) {
				if (!strcasecmp(v->value, "keep_up")) {
					confp->pri.pri.l2_persistence = PRI_L2_PERSISTENCE_KEEP_UP;
				} else if (!strcasecmp(v->value, "leave_down")) {
					confp->pri.pri.l2_persistence = PRI_L2_PERSISTENCE_LEAVE_DOWN;
				} else {
					confp->pri.pri.l2_persistence = PRI_L2_PERSISTENCE_DEFAULT;
				}
#endif	/* defined(HAVE_PRI_L2_PERSISTENCE) */
			} else if (!strcasecmp(v->name, "colp_send")) {
				if (!strcasecmp(v->value, "block")) {
					confp->pri.pri.colp_send = SIG_PRI_COLP_BLOCK;
				} else if (!strcasecmp(v->value, "connect")) {
					confp->pri.pri.colp_send = SIG_PRI_COLP_CONNECT;
				} else if (!strcasecmp(v->value, "update")) {
					confp->pri.pri.colp_send = SIG_PRI_COLP_UPDATE;
				} else {
					confp->pri.pri.colp_send = SIG_PRI_COLP_UPDATE;
				}
#endif /* HAVE_PRI */
#if defined(HAVE_SS7)
			} else if (!strcasecmp(v->name, "ss7type")) {
				if (!strcasecmp(v->value, "itu")) {
					cur_ss7type = SS7_ITU;
				} else if (!strcasecmp(v->value, "ansi")) {
					cur_ss7type = SS7_ANSI;
				} else {
					ast_log(LOG_WARNING, "'%s' is an unknown ss7 switch type at line %d.!\n", v->value, v->lineno);
				}
			} else if (!strcasecmp(v->name, "slc")) {
				cur_slc = atoi(v->value);
			} else if (!strcasecmp(v->name, "linkset")) {
				cur_linkset = atoi(v->value);
			} else if (!strcasecmp(v->name, "pointcode")) {
				cur_pointcode = parse_pointcode(v->value);
			} else if (!strcasecmp(v->name, "adjpointcode")) {
				cur_adjpointcode = parse_pointcode(v->value);
			} else if (!strcasecmp(v->name, "defaultdpc")) {
				cur_defaultdpc = parse_pointcode(v->value);
			} else if (!strcasecmp(v->name, "cicbeginswith")) {
				cur_cicbeginswith = atoi(v->value);
			} else if (!strcasecmp(v->name, "networkindicator")) {
				if (!strcasecmp(v->value, "national")) {
					cur_networkindicator = SS7_NI_NAT;
				} else if (!strcasecmp(v->value, "national_spare")) {
					cur_networkindicator = SS7_NI_NAT_SPARE;
				} else if (!strcasecmp(v->value, "international")) {
					cur_networkindicator = SS7_NI_INT;
				} else if (!strcasecmp(v->value, "international_spare")) {
					cur_networkindicator = SS7_NI_INT_SPARE;
				} else {
					cur_networkindicator = -1;
				}
			} else if (!strcasecmp(v->name, "ss7_internationalprefix")) {
				ast_copy_string(confp->ss7.ss7.internationalprefix, v->value, sizeof(confp->ss7.ss7.internationalprefix));
			} else if (!strcasecmp(v->name, "ss7_nationalprefix")) {
				ast_copy_string(confp->ss7.ss7.nationalprefix, v->value, sizeof(confp->ss7.ss7.nationalprefix));
			} else if (!strcasecmp(v->name, "ss7_subscriberprefix")) {
				ast_copy_string(confp->ss7.ss7.subscriberprefix, v->value, sizeof(confp->ss7.ss7.subscriberprefix));
			} else if (!strcasecmp(v->name, "ss7_unknownprefix")) {
				ast_copy_string(confp->ss7.ss7.unknownprefix, v->value, sizeof(confp->ss7.ss7.unknownprefix));
			} else if (!strcasecmp(v->name, "ss7_networkroutedprefix")) {
				ast_copy_string(confp->ss7.ss7.networkroutedprefix, v->value, sizeof(confp->ss7.ss7.networkroutedprefix));
			} else if (!strcasecmp(v->name, "ss7_called_nai")) {
				if (!strcasecmp(v->value, "national")) {
					confp->ss7.ss7.called_nai = SS7_NAI_NATIONAL;
				} else if (!strcasecmp(v->value, "international")) {
					confp->ss7.ss7.called_nai = SS7_NAI_INTERNATIONAL;
				} else if (!strcasecmp(v->value, "subscriber")) {
					confp->ss7.ss7.called_nai = SS7_NAI_SUBSCRIBER;
				} else if (!strcasecmp(v->value, "unknown")) {
					confp->ss7.ss7.called_nai = SS7_NAI_UNKNOWN;
				} else if (!strcasecmp(v->value, "dynamic")) {
					confp->ss7.ss7.called_nai = SS7_NAI_DYNAMIC;
				} else {
					ast_log(LOG_WARNING, "Unknown SS7 called_nai '%s' at line %d.\n", v->value, v->lineno);
				}
			} else if (!strcasecmp(v->name, "ss7_calling_nai")) {
				if (!strcasecmp(v->value, "national")) {
					confp->ss7.ss7.calling_nai = SS7_NAI_NATIONAL;
				} else if (!strcasecmp(v->value, "international")) {
					confp->ss7.ss7.calling_nai = SS7_NAI_INTERNATIONAL;
				} else if (!strcasecmp(v->value, "subscriber")) {
					confp->ss7.ss7.calling_nai = SS7_NAI_SUBSCRIBER;
				} else if (!strcasecmp(v->value, "unknown")) {
					confp->ss7.ss7.calling_nai = SS7_NAI_UNKNOWN;
				} else if (!strcasecmp(v->value, "dynamic")) {
					confp->ss7.ss7.calling_nai = SS7_NAI_DYNAMIC;
				} else {
					ast_log(LOG_WARNING, "Unknown SS7 calling_nai '%s' at line %d.\n", v->value, v->lineno);
				}
			} else if (!strcasecmp(v->name, "sigchan")) {
				int sigchan, res;
				sigchan = atoi(v->value);
				res = linkset_addsigchan(sigchan);
				if (res < 0) {
					return -1;
				}
			} else if (!strcasecmp(v->name, "ss7_explicitacm")) {
				struct dahdi_ss7 *link;
				link = ss7_resolve_linkset(cur_linkset);
				if (!link) {
					ast_log(LOG_ERROR, "Invalid linkset number.  Must be between 1 and %d\n", NUM_SPANS + 1);
					return -1;
				}
				if (ast_true(v->value)) {
					link->ss7.flags |= LINKSET_FLAG_EXPLICITACM;
				} else {
					link->ss7.flags &= ~LINKSET_FLAG_EXPLICITACM;
				}
			} else if (!strcasecmp(v->name, "ss7_autoacm")) {
				struct dahdi_ss7 *link;
				link = ss7_resolve_linkset(cur_linkset);
				if (!link) {
					ast_log(LOG_ERROR, "Invalid linkset number.  Must be between 1 and %d\n", NUM_SPANS + 1);
					return -1;
				}
				if (ast_true(v->value)) {
					link->ss7.flags |= LINKSET_FLAG_AUTOACM;
				} else {
					link->ss7.flags &= ~LINKSET_FLAG_AUTOACM;
				}
			} else if (!strcasecmp(v->name, "ss7_initialhwblo")) {
				struct dahdi_ss7 *link;
				link = ss7_resolve_linkset(cur_linkset);
				if (!link) {
					ast_log(LOG_ERROR, "Invalid linkset number.  Must be between 1 and %d\n", NUM_SPANS + 1);
					return -1;
				}
				if (ast_true(v->value)) {
					link->ss7.flags |= LINKSET_FLAG_INITIALHWBLO;
				} else {
					link->ss7.flags &= ~LINKSET_FLAG_INITIALHWBLO;
				}
			} else if (!strcasecmp(v->name, "ss7_use_echocontrol")) {
				struct dahdi_ss7 *link;
				link = ss7_resolve_linkset(cur_linkset);
				if (!link) {
					ast_log(LOG_ERROR, "Invalid linkset number.  Must be between 1 and %d\n", NUM_SPANS + 1);
					return -1;
				}
				if (ast_true(v->value)) {
					link->ss7.flags |= LINKSET_FLAG_USEECHOCONTROL;
				} else {
					link->ss7.flags &= ~LINKSET_FLAG_USEECHOCONTROL;
				}
			} else if (!strcasecmp(v->name, "ss7_default_echocontrol")) {
				struct dahdi_ss7 *link;
				link = ss7_resolve_linkset(cur_linkset);
				if (!link) {
					ast_log(LOG_ERROR, "Invalid linkset number.  Must be between 1 and %d\n", NUM_SPANS + 1);
					return -1;
				}
				if (ast_true(v->value)) {
					link->ss7.flags |= LINKSET_FLAG_DEFAULTECHOCONTROL;
				} else {
					link->ss7.flags &= ~LINKSET_FLAG_DEFAULTECHOCONTROL;
				}
			} else if (!strncasecmp(v->name, "isup_timer.", 11)) {
				struct dahdi_ss7 *link;
				link = ss7_resolve_linkset(cur_linkset);
				if (!link) {
					ast_log(LOG_ERROR, "Invalid linkset number.  Must be between 1 and %d\n", NUM_SPANS + 1);
					return -1;
				}
				if (!link->ss7.ss7) {
					ast_log(LOG_ERROR, "Please specify isup timers after sigchan!\n");
				} else if (!ss7_set_isup_timer(link->ss7.ss7, strstr(v->name, ".") + 1, atoi(v->value))) {
					ast_log(LOG_ERROR, "Invalid isup timer %s\n", v->name);
				}
			} else if (!strncasecmp(v->name, "mtp3_timer.", 11)) {
				struct dahdi_ss7 *link;
				link = ss7_resolve_linkset(cur_linkset);
				if (!link) {
					ast_log(LOG_ERROR, "Invalid linkset number.  Must be between 1 and %d\n", NUM_SPANS + 1);
					return -1;
				}
				if (!link->ss7.ss7) {
					ast_log(LOG_ERROR, "Please specify mtp3 timers after sigchan!\n");
				} else if (!ss7_set_mtp3_timer(link->ss7.ss7, strstr(v->name, ".") + 1, atoi(v->value))) {
					ast_log(LOG_ERROR, "Invalid mtp3 timer %s\n", v->name);
				}
			} else if (!strcasecmp(v->name, "inr_if_no_calling")) {
				struct dahdi_ss7 *link;
				link = ss7_resolve_linkset(cur_linkset);
				if (!link) {
					ast_log(LOG_ERROR, "Invalid linkset number.  Must be between 1 and %d\n", NUM_SPANS + 1);
					return -1;
				}
				if (!link->ss7.ss7) {
					ast_log(LOG_ERROR, "Please specify inr_if_no_calling after sigchan!\n");
				} else if (ast_true(v->value)) {
					ss7_set_flags(link->ss7.ss7, SS7_INR_IF_NO_CALLING);
				} else {
					ss7_clear_flags(link->ss7.ss7, SS7_INR_IF_NO_CALLING);
				}
			} else if (!strcasecmp(v->name, "non_isdn_access")) {
				struct dahdi_ss7 *link;
				link = ss7_resolve_linkset(cur_linkset);
				if (!link) {
					ast_log(LOG_ERROR, "Invalid linkset number.  Must be between 1 and %d\n", NUM_SPANS + 1);
					return -1;
				}
				if (!link->ss7.ss7) {
					ast_log(LOG_ERROR, "Please specify non_isdn_access after sigchan!\n");
				} else if (ast_true(v->value)) {
					ss7_clear_flags(link->ss7.ss7, SS7_ISDN_ACCESS_INDICATOR);
				} else {
					ss7_set_flags(link->ss7.ss7, SS7_ISDN_ACCESS_INDICATOR);
				}
			} else if (!strcasecmp(v->name, "sls_shift")) {
				struct dahdi_ss7 *link;
				int sls_shift = atoi(v->value);

				if (sls_shift < 0 || sls_shift > 7) {
					ast_log(LOG_ERROR, "Invalid sls_shift value.  Must be between 0 and 7\n");
					return -1;
				}

				link = ss7_resolve_linkset(cur_linkset);
				if (!link) {
					ast_log(LOG_ERROR, "Invalid linkset number.  Must be between 1 and %d\n", NUM_SPANS + 1);
					return -1;
				}
				if (!link->ss7.ss7) {
					ast_log(LOG_ERROR, "Please specify sls_shift after sigchan!\n");
				} else {
					ss7_set_sls_shift(link->ss7.ss7, sls_shift);
				}
			} else if (!strcasecmp(v->name, "cause_location")) {
				struct dahdi_ss7 *link;
				int cause_location = atoi(v->value);

				if (cause_location < 0 || cause_location > 15) {
					ast_log(LOG_ERROR, "Invalid cause_location value.  Must be between 0 and 15\n");
					return -1;
				}
				link = ss7_resolve_linkset(cur_linkset);
				if (!link) {
					ast_log(LOG_ERROR, "Invalid linkset number.  Must be between 1 and %d\n", NUM_SPANS + 1);
					return -1;
				}
				if (!link->ss7.ss7) {
					ast_log(LOG_ERROR, "Please specify cause_location after sigchan!\n");
				} else {
					ss7_set_cause_location(link->ss7.ss7, cause_location);
				}
#endif	/* defined(HAVE_SS7) */
#ifdef HAVE_OPENR2
			} else if (!strcasecmp(v->name, "mfcr2_advanced_protocol_file")) {
				ast_copy_string(confp->mfcr2.r2proto_file, v->value, sizeof(confp->mfcr2.r2proto_file));
				ast_log(LOG_WARNING, "MFC/R2 Protocol file '%s' will be used, you only should use this if you *REALLY KNOW WHAT YOU ARE DOING*.\n", confp->mfcr2.r2proto_file);
			} else if (!strcasecmp(v->name, "mfcr2_logdir")) {
				ast_copy_string(confp->mfcr2.logdir, v->value, sizeof(confp->mfcr2.logdir));
			} else if (!strcasecmp(v->name, "mfcr2_variant")) {
				confp->mfcr2.variant = openr2_proto_get_variant(v->value);
				if (OR2_VAR_UNKNOWN == confp->mfcr2.variant) {
					ast_log(LOG_WARNING, "Unknown MFC/R2 variant '%s' at line %d, defaulting to ITU.\n", v->value, v->lineno);
					confp->mfcr2.variant = OR2_VAR_ITU;
				}
			} else if (!strcasecmp(v->name, "mfcr2_mfback_timeout")) {
				confp->mfcr2.mfback_timeout = atoi(v->value);
				if (!confp->mfcr2.mfback_timeout) {
					ast_log(LOG_WARNING, "MF timeout of 0? hum, I will protect you from your ignorance. Setting default.\n");
					confp->mfcr2.mfback_timeout = -1;
				} else if (confp->mfcr2.mfback_timeout > 0 && confp->mfcr2.mfback_timeout < 500) {
					ast_log(LOG_WARNING, "MF timeout less than 500ms is not recommended, you have been warned!\n");
				}
			} else if (!strcasecmp(v->name, "mfcr2_metering_pulse_timeout")) {
				confp->mfcr2.metering_pulse_timeout = atoi(v->value);
				if (confp->mfcr2.metering_pulse_timeout > 500) {
					ast_log(LOG_WARNING, "Metering pulse timeout greater than 500ms is not recommended, you have been warned!\n");
				}
#if defined(OR2_LIB_INTERFACE) && OR2_LIB_INTERFACE > 2
			} else if (!strcasecmp(v->name, "mfcr2_dtmf_detection")) {
				confp->mfcr2.dtmf_detection = ast_true(v->value) ? 1 : 0;
			} else if (!strcasecmp(v->name, "mfcr2_dtmf_dialing")) {
				confp->mfcr2.dtmf_dialing = ast_true(v->value) ? 1 : 0;
			} else if (!strcasecmp(v->name, "mfcr2_dtmf_time_on")) {
				confp->mfcr2.dtmf_time_on = atoi(v->value);
			} else if (!strcasecmp(v->name, "mfcr2_dtmf_time_off")) {
				confp->mfcr2.dtmf_time_off = atoi(v->value);
#endif
#if defined(OR2_LIB_INTERFACE) && OR2_LIB_INTERFACE > 3
			} else if (!strcasecmp(v->name, "mfcr2_dtmf_end_timeout")) {
				confp->mfcr2.dtmf_end_timeout = atoi(v->value);
#endif
			} else if (!strcasecmp(v->name, "mfcr2_get_ani_first")) {
				confp->mfcr2.get_ani_first = ast_true(v->value) ? 1 : 0;
			} else if (!strcasecmp(v->name, "mfcr2_double_answer")) {
				confp->mfcr2.double_answer = ast_true(v->value) ? 1 : 0;
			} else if (!strcasecmp(v->name, "mfcr2_charge_calls")) {
				confp->mfcr2.charge_calls = ast_true(v->value) ? 1 : 0;
			} else if (!strcasecmp(v->name, "mfcr2_accept_on_offer")) {
				confp->mfcr2.accept_on_offer = ast_true(v->value) ? 1 : 0;
			} else if (!strcasecmp(v->name, "mfcr2_allow_collect_calls")) {
				confp->mfcr2.allow_collect_calls = ast_true(v->value) ? 1 : 0;
			} else if (!strcasecmp(v->name, "mfcr2_forced_release")) {
				confp->mfcr2.forced_release = ast_true(v->value) ? 1 : 0;
			} else if (!strcasecmp(v->name, "mfcr2_immediate_accept")) {
				confp->mfcr2.immediate_accept = ast_true(v->value) ? 1 : 0;
#if defined(OR2_LIB_INTERFACE) && OR2_LIB_INTERFACE > 1
			} else if (!strcasecmp(v->name, "mfcr2_skip_category")) {
				confp->mfcr2.skip_category_request = ast_true(v->value) ? 1 : 0;
#endif
			} else if (!strcasecmp(v->name, "mfcr2_call_files")) {
				confp->mfcr2.call_files = ast_true(v->value) ? 1 : 0;
			} else if (!strcasecmp(v->name, "mfcr2_max_ani")) {
				confp->mfcr2.max_ani = atoi(v->value);
				if (confp->mfcr2.max_ani >= AST_MAX_EXTENSION) {
					confp->mfcr2.max_ani = AST_MAX_EXTENSION - 1;
				}
			} else if (!strcasecmp(v->name, "mfcr2_max_dnis")) {
				confp->mfcr2.max_dnis = atoi(v->value);
				if (confp->mfcr2.max_dnis >= AST_MAX_EXTENSION) {
					confp->mfcr2.max_dnis = AST_MAX_EXTENSION - 1;
				}
			} else if (!strcasecmp(v->name, "mfcr2_category")) {
				confp->mfcr2.category = openr2_proto_get_category(v->value);
				if (OR2_CALLING_PARTY_CATEGORY_UNKNOWN == confp->mfcr2.category) {
					confp->mfcr2.category = OR2_CALLING_PARTY_CATEGORY_NATIONAL_SUBSCRIBER;
					ast_log(LOG_WARNING, "Invalid MFC/R2 caller category '%s' at line %d. Using national subscriber as default.\n",
							v->value, v->lineno);
				}
			} else if (!strcasecmp(v->name, "mfcr2_logging")) {
				openr2_log_level_t tmplevel;
				char *clevel;
				char *logval;
				char copy[strlen(v->value) + 1];
				strcpy(copy, v->value); /* safe */
				logval = copy;
				while (logval) {
					clevel = strsep(&logval,",");
					if (-1 == (tmplevel = openr2_log_get_level(clevel))) {
						ast_log(LOG_WARNING, "Ignoring invalid logging level: '%s' at line %d.\n", clevel, v->lineno);
						continue;
					}
					confp->mfcr2.loglevel |= tmplevel;
				}
#endif /* HAVE_OPENR2 */
			} else if (!strcasecmp(v->name, "cadence")) {
				/* setup to scan our argument */
				int element_count, c[16] = {0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0};
				int i;
				struct dahdi_ring_cadence new_cadence;
				int cid_location = -1;
				int firstcadencepos = 0;
				char original_args[80];
				int cadence_is_ok = 1;

				ast_copy_string(original_args, v->value, sizeof(original_args));
				/* 16 cadences allowed (8 pairs) */
				element_count = sscanf(v->value, "%30d,%30d,%30d,%30d,%30d,%30d,%30d,%30d,%30d,%30d,%30d,%30d,%30d,%30d,%30d,%30d", &c[0], &c[1], &c[2], &c[3], &c[4], &c[5], &c[6], &c[7], &c[8], &c[9], &c[10], &c[11], &c[12], &c[13], &c[14], &c[15]);

				/* Cadence must be even (on/off) */
				if (element_count % 2 == 1) {
					ast_log(LOG_ERROR, "Must be a silence duration for each ring duration: %s at line %d.\n", original_args, v->lineno);
					cadence_is_ok = 0;
				}

				/* This check is only needed to satisfy the compiler that element_count can't cause an out of bounds */
				if (element_count > ARRAY_LEN(c)) {
					element_count = ARRAY_LEN(c);
				}

				/* Ring cadences cannot be negative */
				for (i = 0; i < element_count; i++) {
					if (c[i] == 0) {
						ast_log(LOG_ERROR, "Ring or silence duration cannot be zero: %s at line %d.\n", original_args, v->lineno);
						cadence_is_ok = 0;
						break;
					} else if (c[i] < 0) {
						if (i % 2 == 1) {
							/* Silence duration, negative possibly okay */
							if (cid_location == -1) {
								cid_location = i;
								c[i] *= -1;
							} else {
								ast_log(LOG_ERROR, "CID location specified twice: %s at line %d.\n", original_args, v->lineno);
								cadence_is_ok = 0;
								break;
							}
						} else {
							if (firstcadencepos == 0) {
								firstcadencepos = i; /* only recorded to avoid duplicate specification */
											/* duration will be passed negative to the DAHDI driver */
							} else {
								 ast_log(LOG_ERROR, "First cadence position specified twice: %s at line %d.\n", original_args, v->lineno);
								cadence_is_ok = 0;
								break;
							}
						}
					}
				}

				/* Substitute our scanned cadence */
				for (i = 0; i < 16; i++) {
					new_cadence.ringcadence[i] = c[i];
				}

				if (cadence_is_ok) {
					/* ---we scanned it without getting annoyed; now some sanity checks--- */
					if (element_count < 2) {
						ast_log(LOG_ERROR, "Minimum cadence is ring,pause: %s at line %d.\n", original_args, v->lineno);
					} else {
						if (cid_location == -1) {
							/* user didn't say; default to first pause */
							cid_location = 1;
						} else {
							/* convert element_index to cidrings value */
							cid_location = (cid_location + 1) / 2;
						}
						/* ---we like their cadence; try to install it--- */
						if (!user_has_defined_cadences++)
							/* this is the first user-defined cadence; clear the default user cadences */
							num_cadence = 0;
						if ((num_cadence+1) >= NUM_CADENCE_MAX)
							ast_log(LOG_ERROR, "Already %d cadences; can't add another: %s at line %d.\n", NUM_CADENCE_MAX, original_args, v->lineno);
						else {
							cadences[num_cadence] = new_cadence;
							cidrings[num_cadence++] = cid_location;
							ast_verb(3, "cadence 'r%d' added: %s\n",num_cadence,original_args);
						}
					}
				}
			} else if (!strcasecmp(v->name, "ringtimeout")) {
				ringt_base = (atoi(v->value) * 8) / READ_SIZE;
			} else if (!strcasecmp(v->name, "prewink")) {
				confp->timing.prewinktime = atoi(v->value);
			} else if (!strcasecmp(v->name, "preflash")) {
				confp->timing.preflashtime = atoi(v->value);
			} else if (!strcasecmp(v->name, "wink")) {
				confp->timing.winktime = atoi(v->value);
			} else if (!strcasecmp(v->name, "flash")) {
				confp->timing.flashtime = atoi(v->value);
			} else if (!strcasecmp(v->name, "start")) {
				confp->timing.starttime = atoi(v->value);
			} else if (!strcasecmp(v->name, "rxwink")) {
				confp->timing.rxwinktime = atoi(v->value);
			} else if (!strcasecmp(v->name, "rxflash")) {
				confp->timing.rxflashtime = atoi(v->value);
			} else if (!strcasecmp(v->name, "debounce")) {
				confp->timing.debouncetime = atoi(v->value);
			} else if (!strcasecmp(v->name, "toneduration")) {
				int toneduration;
				int ctlfd;
				int res;
				struct dahdi_dialparams dps;

				ctlfd = open("/dev/dahdi/ctl", O_RDWR);
				if (ctlfd == -1) {
					ast_log(LOG_ERROR, "Unable to open /dev/dahdi/ctl to set toneduration at line %d.\n", v->lineno);
					return -1;
				}

				toneduration = atoi(v->value);
				if (toneduration > -1) {
					memset(&dps, 0, sizeof(dps));

					dps.dtmf_tonelen = dps.mfv1_tonelen = toneduration;
					res = ioctl(ctlfd, DAHDI_SET_DIALPARAMS, &dps);
					if (res < 0) {
						ast_log(LOG_ERROR, "Invalid tone duration: %d ms at line %d: %s\n", toneduration, v->lineno, strerror(errno));
						close(ctlfd);
						return -1;
					}
				}
				close(ctlfd);
			} else if (!strcasecmp(v->name, "defaultcic")) {
				ast_copy_string(defaultcic, v->value, sizeof(defaultcic));
			} else if (!strcasecmp(v->name, "defaultozz")) {
				ast_copy_string(defaultozz, v->value, sizeof(defaultozz));
			} else if (!strcasecmp(v->name, "mwilevel")) {
				mwilevel = atoi(v->value);
			} else if (!strcasecmp(v->name, "dtmfcidlevel")) {
				dtmfcid_level = atoi(v->value);
			} else if (!strcasecmp(v->name, "reportalarms")) {
				if (!strcasecmp(v->value, "all"))
					report_alarms = REPORT_CHANNEL_ALARMS | REPORT_SPAN_ALARMS;
				if (!strcasecmp(v->value, "none"))
					report_alarms = 0;
				else if (!strcasecmp(v->value, "channels"))
					report_alarms = REPORT_CHANNEL_ALARMS;
			   else if (!strcasecmp(v->value, "spans"))
					report_alarms = REPORT_SPAN_ALARMS;
			 }
		} else if (!(options & PROC_DAHDI_OPT_NOWARN) )
			ast_log(LOG_NOTICE, "Ignoring any changes to '%s' (on reload) at line %d.\n", v->name, v->lineno);
	}

	if (dahdichan) {
		/* Process the deferred dahdichan value. */
		if (build_channels(confp, dahdichan->value, reload, dahdichan->lineno)) {
			if (confp->ignore_failed_channels) {
				ast_log(LOG_WARNING,
					"Dahdichan '%s' failure ignored: ignore_failed_channels.\n",
					dahdichan->value);
			} else {
				return -1;
			}
		}
	}

	/*
	 * Since confp has already filled individual dahdi_pvt objects with channels
	 * at this point, clear the variables in confp's pvt.
	 */
	if (confp->chan.vars) {
		ast_variables_destroy(confp->chan.vars);
		confp->chan.vars = NULL;
	}

	/* mark the first channels of each DAHDI span to watch for their span alarms */
	for (tmp = iflist, y=-1; tmp; tmp = tmp->next) {
		if (!tmp->destroy && tmp->span != y) {
			tmp->manages_span_alarms = 1;
			y = tmp->span;
		} else {
			tmp->manages_span_alarms = 0;
		}
	}

	/*< \todo why check for the pseudo in the per-channel section.
	 * Any actual use for manual setup of the pseudo channel? */
	if (!has_pseudo && reload != 1 && !(options & PROC_DAHDI_OPT_NOCHAN)) {
		/* use the default configuration for a channel, so
		   that any settings from real configured channels
		   don't "leak" into the pseudo channel config
		*/
		struct dahdi_chan_conf conf = dahdi_chan_conf_default();

		if (conf.chan.cc_params) {
			tmp = mkintf(CHAN_PSEUDO, &conf, reload);
		} else {
			tmp = NULL;
		}
		if (tmp) {
			ast_verb(3, "Automatically generated pseudo channel\n");
			has_pseudo = 1;
		} else {
			ast_log(LOG_WARNING, "Unable to register pseudo channel!\n");
		}
		ast_cc_config_params_destroy(conf.chan.cc_params);
	}

	/* Since named callgroup and named pickup group are ref'd to dahdi_pvt at this point, unref container in confp's pvt. */
	confp->chan.named_callgroups = ast_unref_namedgroups(confp->chan.named_callgroups);
	confp->chan.named_pickupgroups = ast_unref_namedgroups(confp->chan.named_pickupgroups);

	return 0;
}

/*!
 * \internal
 * \brief Deep copy struct dahdi_chan_conf.
 * \since 1.8
 *
 * \param dest Destination.
 * \param src Source.
 */
static void deep_copy_dahdi_chan_conf(struct dahdi_chan_conf *dest, const struct dahdi_chan_conf *src)
{
	struct ast_cc_config_params *cc_params;

	cc_params = dest->chan.cc_params;
	*dest = *src;
	dest->chan.cc_params = cc_params;
	ast_cc_copy_config_params(dest->chan.cc_params, src->chan.cc_params);
}

/*!
 * \internal
 * \brief Setup DAHDI channel driver.
 *
 * \param reload enum: load_module(0), reload(1), restart(2).
 * \param default_conf Default config parameters.  So cc_params can be properly destroyed.
 * \param base_conf Default config parameters per section.  So cc_params can be properly destroyed.
 * \param conf Local config parameters.  So cc_params can be properly destroyed.
 *
 * \retval 0 on success.
 * \retval -1 on error.
 */
static int setup_dahdi_int(int reload, struct dahdi_chan_conf *default_conf, struct dahdi_chan_conf *base_conf, struct dahdi_chan_conf *conf)
{
	struct ast_config *cfg;
	struct ast_config *ucfg;
	struct ast_variable *v;
	struct ast_flags config_flags = { reload == 1 ? CONFIG_FLAG_FILEUNCHANGED : 0 };
	const char *chans;
	const char *cat;
	int res;

#ifdef HAVE_PRI
	char *c;
	int spanno;
	int i;
	int logicalspan;
	int trunkgroup;
	int dchannels[SIG_PRI_NUM_DCHANS];
#endif
	int have_cfg_now;
	static int had_cfg_before = 1;/* So initial load will complain if we don't have cfg. */

	cfg = ast_config_load(config, config_flags);
	have_cfg_now = !!cfg;
	if (!cfg) {
		/* Error if we have no config file */
		if (had_cfg_before) {
			ast_log(LOG_ERROR, "Unable to load config %s\n", config);
			ast_clear_flag(&config_flags, CONFIG_FLAG_FILEUNCHANGED);
		}
		cfg = ast_config_new();/* Dummy config */
		if (!cfg) {
			return 0;
		}
		ucfg = ast_config_load("users.conf", config_flags);
		if (ucfg == CONFIG_STATUS_FILEUNCHANGED) {
			ast_config_destroy(cfg);
			return 0;
		}
		if (ucfg == CONFIG_STATUS_FILEINVALID) {
			ast_log(LOG_ERROR, "File users.conf cannot be parsed.  Aborting.\n");
			ast_config_destroy(cfg);
			return 0;
		}
	} else if (cfg == CONFIG_STATUS_FILEUNCHANGED) {
		ucfg = ast_config_load("users.conf", config_flags);
		if (ucfg == CONFIG_STATUS_FILEUNCHANGED) {
			return 0;
		}
		if (ucfg == CONFIG_STATUS_FILEINVALID) {
			ast_log(LOG_ERROR, "File users.conf cannot be parsed.  Aborting.\n");
			return 0;
		}
		ast_clear_flag(&config_flags, CONFIG_FLAG_FILEUNCHANGED);
		cfg = ast_config_load(config, config_flags);
		have_cfg_now = !!cfg;
		if (!cfg) {
			if (had_cfg_before) {
				/* We should have been able to load the config. */
				ast_log(LOG_ERROR, "Bad. Unable to load config %s\n", config);
				ast_config_destroy(ucfg);
				return 0;
			}
			cfg = ast_config_new();/* Dummy config */
			if (!cfg) {
				ast_config_destroy(ucfg);
				return 0;
			}
		} else if (cfg == CONFIG_STATUS_FILEINVALID) {
			ast_log(LOG_ERROR, "File %s cannot be parsed.  Aborting.\n", config);
			ast_config_destroy(ucfg);
			return 0;
		}
	} else if (cfg == CONFIG_STATUS_FILEINVALID) {
		ast_log(LOG_ERROR, "File %s cannot be parsed.  Aborting.\n", config);
		return 0;
	} else {
		ast_clear_flag(&config_flags, CONFIG_FLAG_FILEUNCHANGED);
		ucfg = ast_config_load("users.conf", config_flags);
		if (ucfg == CONFIG_STATUS_FILEINVALID) {
			ast_log(LOG_ERROR, "File users.conf cannot be parsed.  Aborting.\n");
			ast_config_destroy(cfg);
			return 0;
		}
	}
	had_cfg_before = have_cfg_now;

	/* It's a little silly to lock it, but we might as well just to be sure */
	ast_mutex_lock(&iflock);
#ifdef HAVE_PRI
	if (reload != 1) {
		/* Process trunkgroups first */
		v = ast_variable_browse(cfg, "trunkgroups");
		while (v) {
			if (!strcasecmp(v->name, "trunkgroup")) {
				trunkgroup = atoi(v->value);
				if (trunkgroup > 0) {
					if ((c = strchr(v->value, ','))) {
						i = 0;
						memset(dchannels, 0, sizeof(dchannels));
						while (c && (i < SIG_PRI_NUM_DCHANS)) {
							dchannels[i] = atoi(c + 1);
							if (dchannels[i] < 0) {
								ast_log(LOG_WARNING, "D-channel for trunk group %d must be a positive number at line %d of chan_dahdi.conf\n", trunkgroup, v->lineno);
							} else
								i++;
							c = strchr(c + 1, ',');
						}
						if (i) {
							if (pri_create_trunkgroup(trunkgroup, dchannels)) {
								ast_log(LOG_WARNING, "Unable to create trunk group %d with Primary D-channel %d at line %d of chan_dahdi.conf\n", trunkgroup, dchannels[0], v->lineno);
						} else
								ast_verb(2, "Created trunk group %d with Primary D-channel %d and %d backup%s\n", trunkgroup, dchannels[0], i - 1, (i == 1) ? "" : "s");
						} else
							ast_log(LOG_WARNING, "Trunk group %d lacks any valid D-channels at line %d of chan_dahdi.conf\n", trunkgroup, v->lineno);
					} else
						ast_log(LOG_WARNING, "Trunk group %d lacks a primary D-channel at line %d of chan_dahdi.conf\n", trunkgroup, v->lineno);
				} else
					ast_log(LOG_WARNING, "Trunk group identifier must be a positive integer at line %d of chan_dahdi.conf\n", v->lineno);
			} else if (!strcasecmp(v->name, "spanmap")) {
				spanno = atoi(v->value);
				if (spanno > 0) {
					if ((c = strchr(v->value, ','))) {
						trunkgroup = atoi(c + 1);
						if (trunkgroup > 0) {
							if ((c = strchr(c + 1, ',')))
								logicalspan = atoi(c + 1);
							else
								logicalspan = 0;
							if (logicalspan >= 0) {
								if (pri_create_spanmap(spanno - 1, trunkgroup, logicalspan)) {
									ast_log(LOG_WARNING, "Failed to map span %d to trunk group %d (logical span %d)\n", spanno, trunkgroup, logicalspan);
							} else
									ast_verb(2, "Mapped span %d to trunk group %d (logical span %d)\n", spanno, trunkgroup, logicalspan);
							} else
								ast_log(LOG_WARNING, "Logical span must be a positive number, or '0' (for unspecified) at line %d of chan_dahdi.conf\n", v->lineno);
						} else
							ast_log(LOG_WARNING, "Trunk group must be a positive number at line %d of chan_dahdi.conf\n", v->lineno);
					} else
						ast_log(LOG_WARNING, "Missing trunk group for span map at line %d of chan_dahdi.conf\n", v->lineno);
				} else
					ast_log(LOG_WARNING, "Span number must be a positive integer at line %d of chan_dahdi.conf\n", v->lineno);
			} else {
				ast_log(LOG_NOTICE, "Ignoring unknown keyword '%s' in trunkgroups\n", v->name);
			}
			v = v->next;
		}
	}
#endif

	/* Copy the default jb config over global_jbconf */
	memcpy(&global_jbconf, &default_jbconf, sizeof(global_jbconf));

	mwimonitornotify[0] = '\0';

	v = ast_variable_browse(cfg, "channels");
	if ((res = process_dahdi(base_conf,
		"" /* Must be empty for the channels category.  Silly voicemail mailbox. */,
		v, reload, 0))) {
		ast_mutex_unlock(&iflock);
		ast_config_destroy(cfg);
		if (ucfg) {
			ast_config_destroy(ucfg);
		}
		return res;
	}

	/* Now get configuration from all normal sections in chan_dahdi.conf: */
	for (cat = ast_category_browse(cfg, NULL); cat ; cat = ast_category_browse(cfg, cat)) {
		/* [channels] and [trunkgroups] are used. Let's also reserve
		 * [globals] and [general] for future use
		 */
		if (!strcasecmp(cat, "general") ||
			!strcasecmp(cat, "trunkgroups") ||
			!strcasecmp(cat, "globals") ||
			!strcasecmp(cat, "channels")) {
			continue;
		}

		chans = ast_variable_retrieve(cfg, cat, "dahdichan");
		if (ast_strlen_zero(chans)) {
			/* Section is useless without a dahdichan value present. */
			continue;
		}

		/* Copy base_conf to conf. */
		deep_copy_dahdi_chan_conf(conf, base_conf);

		if ((res = process_dahdi(conf, cat, ast_variable_browse(cfg, cat), reload, PROC_DAHDI_OPT_NOCHAN))) {
			ast_mutex_unlock(&iflock);
			ast_config_destroy(cfg);
			if (ucfg) {
				ast_config_destroy(ucfg);
			}
			return res;
		}
	}

	ast_config_destroy(cfg);

	if (ucfg) {
		/* Reset base_conf, so things don't leak from chan_dahdi.conf */
		deep_copy_dahdi_chan_conf(base_conf, default_conf);
		process_dahdi(base_conf,
			"" /* Must be empty for the general category.  Silly voicemail mailbox. */,
			ast_variable_browse(ucfg, "general"), 1, 0);

		for (cat = ast_category_browse(ucfg, NULL); cat ; cat = ast_category_browse(ucfg, cat)) {
			if (!strcasecmp(cat, "general")) {
				continue;
			}

			chans = ast_variable_retrieve(ucfg, cat, "dahdichan");
			if (ast_strlen_zero(chans)) {
				/* Section is useless without a dahdichan value present. */
				continue;
			}

			/* Copy base_conf to conf. */
			deep_copy_dahdi_chan_conf(conf, base_conf);

			if ((res = process_dahdi(conf, cat, ast_variable_browse(ucfg, cat), reload, PROC_DAHDI_OPT_NOCHAN | PROC_DAHDI_OPT_NOWARN))) {
				ast_config_destroy(ucfg);
				ast_mutex_unlock(&iflock);
				return res;
			}
		}
		ast_config_destroy(ucfg);
	}
	ast_mutex_unlock(&iflock);

#ifdef HAVE_PRI
	if (reload != 1) {
		int x;
		for (x = 0; x < NUM_SPANS; x++) {
			if (pris[x].pri.pvts[0] &&
					pris[x].pri.master == AST_PTHREADT_NULL) {
				prepare_pri(pris + x);
				if (sig_pri_start_pri(&pris[x].pri)) {
					ast_log(LOG_ERROR, "Unable to start D-channel on span %d\n", x + 1);
					return -1;
				} else
					ast_verb(2, "Starting D-Channel on span %d\n", x + 1);
			}
		}
	}
#endif
#if defined(HAVE_SS7)
	if (reload != 1) {
		int x;
		for (x = 0; x < NUM_SPANS; x++) {
			if (linksets[x].ss7.ss7) {
				if (ast_pthread_create(&linksets[x].ss7.master, NULL, ss7_linkset, &linksets[x].ss7)) {
					ast_log(LOG_ERROR, "Unable to start SS7 linkset on span %d\n", x + 1);
					return -1;
				} else
					ast_verb(2, "Starting SS7 linkset on span %d\n", x + 1);
			}
		}
	}
#endif	/* defined(HAVE_SS7) */
#ifdef HAVE_OPENR2
	if (reload != 1) {
		struct r2link_entry *cur;
		int x = 0;
		AST_LIST_LOCK(&r2links);
		AST_LIST_TRAVERSE(&r2links, cur, list) {
			struct dahdi_mfcr2 *r2 = &cur->mfcr2;
			if (r2->r2master == AST_PTHREADT_NULL) {
				if (ast_pthread_create(&r2->r2master, NULL, mfcr2_monitor, r2)) {
					ast_log(LOG_ERROR, "Unable to start R2 monitor on channel group %d\n", x + 1);
					return -1;
				} else {
					ast_verb(2, "Starting R2 monitor on channel group %d\n", x + 1);
				}
				x++;
			}
		}
		AST_LIST_UNLOCK(&r2links);
	}
#endif
	/* And start the monitor for the first time */
	restart_monitor();
	return 0;
}

/*!
 * \internal
 * \brief Setup DAHDI channel driver.
 *
 * \param reload enum: load_module(0), reload(1), restart(2).
 *
 * \retval 0 on success.
 * \retval -1 on error.
 */
static int setup_dahdi(int reload)
{
	int res;
	struct dahdi_chan_conf default_conf = dahdi_chan_conf_default();
	struct dahdi_chan_conf base_conf = dahdi_chan_conf_default();
	struct dahdi_chan_conf conf = dahdi_chan_conf_default();

	if (default_conf.chan.cc_params && base_conf.chan.cc_params && conf.chan.cc_params) {
		res = setup_dahdi_int(reload, &default_conf, &base_conf, &conf);
	} else {
		res = -1;
	}
	ast_cc_config_params_destroy(default_conf.chan.cc_params);
	ast_cc_config_params_destroy(base_conf.chan.cc_params);
	ast_cc_config_params_destroy(conf.chan.cc_params);

	return res;
}

/*!
 * \brief Load the module
 *
 * Module loading including tests for configuration or dependencies.
 * This function can return AST_MODULE_LOAD_FAILURE, AST_MODULE_LOAD_DECLINE,
 * or AST_MODULE_LOAD_SUCCESS. If a dependency or environment variable fails
 * tests return AST_MODULE_LOAD_FAILURE. If the module can not load the
 * configuration file or other non-critical problem return
 * AST_MODULE_LOAD_DECLINE. On success return AST_MODULE_LOAD_SUCCESS.
 */
static int load_module(void)
{
	int res;
#if defined(HAVE_PRI) || defined(HAVE_SS7)
	int y;
#endif	/* defined(HAVE_PRI) || defined(HAVE_SS7) */

	if (STASIS_MESSAGE_TYPE_INIT(dahdichannel_type)) {
		return AST_MODULE_LOAD_DECLINE;
	}

	if (!(dahdi_tech.capabilities = ast_format_cap_alloc(AST_FORMAT_CAP_FLAG_DEFAULT))) {
		return AST_MODULE_LOAD_DECLINE;
	}
	ast_format_cap_append(dahdi_tech.capabilities, ast_format_slin, 0);
	ast_format_cap_append(dahdi_tech.capabilities, ast_format_ulaw, 0);
	ast_format_cap_append(dahdi_tech.capabilities, ast_format_alaw, 0);

	if (dahdi_native_load(&dahdi_tech)) {
		ao2_ref(dahdi_tech.capabilities, -1);
		return AST_MODULE_LOAD_DECLINE;
	}

#ifdef HAVE_PRI
	memset(pris, 0, sizeof(pris));
	for (y = 0; y < NUM_SPANS; y++) {
		sig_pri_init_pri(&pris[y].pri);
	}
	pri_set_error(dahdi_pri_error);
	pri_set_message(dahdi_pri_message);
	ast_register_application_xml(dahdi_send_keypad_facility_app, dahdi_send_keypad_facility_exec);
#ifdef HAVE_PRI_PROG_W_CAUSE
	ast_register_application_xml(dahdi_send_callrerouting_facility_app, dahdi_send_callrerouting_facility_exec);
#endif
#if defined(HAVE_PRI_CCSS)
	if (ast_cc_agent_register(&dahdi_pri_cc_agent_callbacks)
		|| ast_cc_monitor_register(&dahdi_pri_cc_monitor_callbacks)) {
		__unload_module();
		return AST_MODULE_LOAD_DECLINE;
	}
#endif	/* defined(HAVE_PRI_CCSS) */
	if (sig_pri_load(
#if defined(HAVE_PRI_CCSS)
		dahdi_pri_cc_type
#else
		NULL
#endif	/* defined(HAVE_PRI_CCSS) */
		)) {
		__unload_module();
		return AST_MODULE_LOAD_DECLINE;
	}
#endif
#if defined(HAVE_SS7)
	memset(linksets, 0, sizeof(linksets));
	for (y = 0; y < NUM_SPANS; y++) {
		sig_ss7_init_linkset(&linksets[y].ss7);
	}
	ss7_set_error(dahdi_ss7_error);
	ss7_set_message(dahdi_ss7_message);
	ss7_set_hangup(sig_ss7_cb_hangup);
	ss7_set_notinservice(sig_ss7_cb_notinservice);
	ss7_set_call_null(sig_ss7_cb_call_null);
#endif	/* defined(HAVE_SS7) */
	res = setup_dahdi(0);
	/* Make sure we can register our DAHDI channel type */
	if (res) {
		__unload_module();
		return AST_MODULE_LOAD_DECLINE;
	}
	if (ast_channel_register(&dahdi_tech)) {
		ast_log(LOG_ERROR, "Unable to register channel class 'DAHDI'\n");
		__unload_module();
		return AST_MODULE_LOAD_DECLINE;
	}
#ifdef HAVE_PRI
	ast_cli_register_multiple(dahdi_pri_cli, ARRAY_LEN(dahdi_pri_cli));
#endif
#if defined(HAVE_SS7)
	ast_cli_register_multiple(dahdi_ss7_cli, ARRAY_LEN(dahdi_ss7_cli));
#endif	/* defined(HAVE_SS7) */
#ifdef HAVE_OPENR2
	ast_cli_register_multiple(dahdi_mfcr2_cli, ARRAY_LEN(dahdi_mfcr2_cli));
	ast_register_application_xml(dahdi_accept_r2_call_app, dahdi_accept_r2_call_exec);
#endif

	ast_custom_function_register(&dahdichan_function);
	ast_custom_function_register(&polarity_function);

	ast_cli_register_multiple(dahdi_cli, ARRAY_LEN(dahdi_cli));
	memset(round_robin, 0, sizeof(round_robin));
	ast_manager_register_xml("DAHDITransfer", 0, action_transfer);
	ast_manager_register_xml("DAHDIHangup", 0, action_transferhangup);
	ast_manager_register_xml("DAHDIDialOffhook", 0, action_dahdidialoffhook);
	ast_manager_register_xml("DAHDIDNDon", 0, action_dahdidndon);
	ast_manager_register_xml("DAHDIDNDoff", 0, action_dahdidndoff);
	ast_manager_register_xml("DAHDIShowChannels", 0, action_dahdishowchannels);
	ast_manager_register_xml("DAHDIShowStatus", 0, action_dahdishowstatus);
	ast_manager_register_xml("DAHDIRestart", 0, action_dahdirestart);
#if defined(HAVE_PRI)
	ast_manager_register_xml("PRIShowSpans", 0, action_prishowspans);
	ast_manager_register_xml("PRIDebugSet", 0, action_pri_debug_set);
	ast_manager_register_xml("PRIDebugFileSet", EVENT_FLAG_SYSTEM, action_pri_debug_file_set);
	ast_manager_register_xml("PRIDebugFileUnset", 0, action_pri_debug_file_unset);
#endif	/* defined(HAVE_PRI) */

	ast_cond_init(&ss_thread_complete, NULL);

	return res;
}

static int dahdi_sendtext(struct ast_channel *c, const char *text)
{
#define	END_SILENCE_LEN 400
#define	HEADER_MS 50
#define	TRAILER_MS 5
#define	HEADER_LEN ((HEADER_MS + TRAILER_MS) * 8)
#define	ASCII_BYTES_PER_CHAR 80

	unsigned char *buf,*mybuf;
	struct dahdi_pvt *p = ast_channel_tech_pvt(c);
	struct pollfd fds[1];
	int size,res,fd,len,x;
	int bytes=0;
	int idx;

	/*
	 * Initial carrier (imaginary)
	 *
	 * Note: The following float variables are used by the
	 * PUT_CLID_MARKMS and PUT_CLID() macros.
	 */
	float cr = 1.0;
	float ci = 0.0;
	float scont = 0.0;

	if (!text[0]) {
		return(0); /* if nothing to send, don't */
	}
	idx = dahdi_get_index(c, p, 0);
	if (idx < 0) {
		ast_log(LOG_WARNING, "Huh?  I don't exist?\n");
		return -1;
	}
	if ((!p->tdd) && (!p->mate)) {
#if defined(HAVE_PRI)
#if defined(HAVE_PRI_DISPLAY_TEXT)
		ast_mutex_lock(&p->lock);
		if (dahdi_sig_pri_lib_handles(p->sig)) {
			sig_pri_sendtext(p->sig_pvt, text);
		}
		ast_mutex_unlock(&p->lock);
#endif	/* defined(HAVE_PRI_DISPLAY_TEXT) */
#endif	/* defined(HAVE_PRI) */
		return(0);  /* if not in TDD mode, just return */
	}
	if (p->mate)
		buf = ast_malloc(((strlen(text) + 1) * ASCII_BYTES_PER_CHAR) + END_SILENCE_LEN + HEADER_LEN);
	else
		buf = ast_malloc(((strlen(text) + 1) * TDD_BYTES_PER_CHAR) + END_SILENCE_LEN);
	if (!buf)
		return -1;
	mybuf = buf;
	if (p->mate) {
		/* PUT_CLI_MARKMS is a macro and requires a format ptr called codec to be present */
		struct ast_format *codec = AST_LAW(p);

		for (x = 0; x < HEADER_MS; x++) {	/* 50 ms of Mark */
			PUT_CLID_MARKMS;
		}
		/* Put actual message */
		for (x = 0; text[x]; x++) {
			PUT_CLID(text[x]);
		}
		for (x = 0; x < TRAILER_MS; x++) {	/* 5 ms of Mark */
			PUT_CLID_MARKMS;
		}
		len = bytes;
		buf = mybuf;
	} else {
		len = tdd_generate(p->tdd, buf, text);
		if (len < 1) {
			ast_log(LOG_ERROR, "TDD generate (len %d) failed!!\n", (int)strlen(text));
			ast_free(mybuf);
			return -1;
		}
	}
	memset(buf + len, 0x7f, END_SILENCE_LEN);
	len += END_SILENCE_LEN;
	fd = p->subs[idx].dfd;
	while (len) {
		if (ast_check_hangup(c)) {
			ast_free(mybuf);
			return -1;
		}
		size = len;
		if (size > READ_SIZE)
			size = READ_SIZE;
		fds[0].fd = fd;
		fds[0].events = POLLOUT | POLLPRI;
		fds[0].revents = 0;
		res = poll(fds, 1, -1);
		if (!res) {
			ast_debug(1, "poll (for write) ret. 0 on channel %d\n", p->channel);
			continue;
		}
		/* if got exception */
		if (fds[0].revents & POLLPRI) {
			ast_free(mybuf);
			return -1;
		}
		if (!(fds[0].revents & POLLOUT)) {
			ast_debug(1, "write fd not ready on channel %d\n", p->channel);
			continue;
		}
		res = write(fd, buf, size);
		if (res != size) {
			if (res == -1) {
				ast_free(mybuf);
				return -1;
			}
			ast_debug(1, "Write returned %d (%s) on channel %d\n", res, strerror(errno), p->channel);
			break;
		}
		len -= size;
		buf += size;
	}
	ast_free(mybuf);
	return(0);
}


static int reload(void)
{
	int res = 0;

	res = setup_dahdi(1);
	if (res) {
		ast_log(LOG_WARNING, "Reload of chan_dahdi.so is unsuccessful!\n");
		return -1;
	}
	return 0;
}

/* This is a workaround so that menuselect displays a proper description
 * AST_MODULE_INFO(, , "DAHDI Telephony"
 */

AST_MODULE_INFO(ASTERISK_GPL_KEY, AST_MODFLAG_LOAD_ORDER, tdesc,
	.support_level = AST_MODULE_SUPPORT_CORE,
	.load = load_module,
	.unload = unload_module,
	.reload = reload,
	.load_pri = AST_MODPRI_CHANNEL_DRIVER,
	.requires = "ccss",
	.optional_modules = "res_smdi",
);
