%define OT_CovarianceModel_doc
"Covariance model.

Base class for covariance models."

%enddef
%feature("docstring") OT::CovarianceModelImplementation
OT_CovarianceModel_doc

// ---------------------------------------------------------------------

%define OT_CovarianceModel_computeAsScalar_doc
R"RAW(Compute the covariance function for scalar model.

Available usages:
    computeAsScalar(s, t)

    computeAsScalar(tau)

Parameters
----------
s, t : floats (if :math:`n=1`) or sequences of floats (any :math:`n`)
    Multivariate index :math:`(\vect{s}, \vect{t}) \in \cD \times \cD`
tau : float (if :math:`n=1`) or sequence of floats (any :math:`n`)
    Multivariate index :math:`\vect{\tau} \in \cD`

Returns
-------
covariance : float
    Covariance.

Notes
-----
The method makes sense only if the dimension of the process is :math:`d=1`.
It evaluates :math:`C(\vect{s}, \vect{t})`.

In the second usage, the covariance model must be stationary. Then we note
:math:`C^{stat}(\vect{\tau})` for :math:`C(\vect{s}, \vect{s}+\vect{\tau})` as
this quantity does not depend on :math:`\vect{s}`.)RAW"
%enddef
%feature("docstring") OT::CovarianceModelImplementation::computeAsScalar
OT_CovarianceModel_computeAsScalar_doc

// ---------------------------------------------------------------------

%define OT_CovarianceModel_computeCrossCovariance_doc
R"RAW(Compute the covariance function on a given mesh.

Parameters
----------
firstVertices : :class:`~openturns.Sample` or :class:`~openturns.Point`
    Container of the first discretization vertices

secondVertices : :class:`~openturns.Sample` or :class:`~openturns.Point`
    Container of the second discretization vertices

Returns
-------
Matrix : :class:`~openturns.Matrix`
    Cross covariance matrix

Notes
-----
This method computes a cross-covariance matrix.
The cross-covariance is the evaluation of the covariance model 
on both *firstVertices* and *secondVertices*.

If *firstVertices* contains :math:`n_1` points and *secondVertices* contains :math:`n_2` points,
the method returns an :math:`n_1 d \times n_2 d` matrix (:math:`d` being the output dimension).

To make things easier, let us focus on the :math:`d=1` case.
Let :math:`\vect{x}_0, \dots, \vect{x}_{n_1-1}` be the points of *firstVertices*
and let :math:`\vect{y}_0, \dots, \vect{y}_{n_2-1}` be the points of *secondVertices*.
The result is the :math:`n_1 \times n_2` matrix :math:`\mat{M}`
such that for any nonnegative integers :math:`i < n_1` and :math:`j < n_2`,
:math:`\mat{M}_{i,j} = \mathcal{C}(\vect{x}_i, \vect{y}_j)`.

)RAW"
%enddef
%feature("docstring") OT::CovarianceModelImplementation::computeCrossCovariance
OT_CovarianceModel_computeCrossCovariance_doc

// ---------------------------------------------------------------------

%define OT_CovarianceModel_discretize_doc
R"RAW(Discretize the covariance function on a given mesh.

Parameters
----------
where : :class:`~openturns.Mesh` or :class:`~openturns.RegularGrid` or :class:`~openturns.Sample`
    Container of the discretization vertices

Returns
-------
covarianceMatrix : :class:`~openturns.CovarianceMatrix`
    Covariance matrix :math:`\in \cS_{nd}^+(\Rset)` (if the process is of
    dimension :math:`d`)

Notes
-----
This method makes a discretization of the model on the given :class:`~openturns.Mesh`,
:class:`~openturns.RegularGrid` or :class:`~openturns.Sample` composed of
the vertices :math:`(\vect{t}_1, \dots, \vect{t}_{N-1})` and returns the
covariance matrix:

.. math ::

    \mat{C}_{1,\dots,k} = \left(
        \begin{array}{cccc}
        C(\vect{t}_1, \vect{t}_1) &C(\vect{t}_1, \vect{t}_2) & \dots &
        C(\vect{t}_1, \vect{t}_{k}) \\
        \dots & C(\vect{t}_2, \vect{t}_2)  & \dots &
        C(\vect{t}_2, \vect{t}_{k}) \\
        \dots & \dots & \dots & \dots \\
        \dots & \dots & \dots & C(\vect{t}_{k}, \vect{t}_{k})
        \end{array} \right)
)RAW"
%enddef
%feature("docstring") OT::CovarianceModelImplementation::discretize
OT_CovarianceModel_discretize_doc

// ---------------------------------------------------------------------

%define OT_CovarianceModel_discretizeAndFactorize_doc
R"RAW(Discretize and factorize the covariance function on a given mesh.

Parameters
----------
where : :class:`~openturns.Mesh` or :class:`~openturns.RegularGrid` or :class:`~openturns.Sample`
    Container of the discretization vertices

Returns
-------
CholeskyMatrix : :class:`~openturns.TriangularMatrix`
    Cholesky factor of the covariance matrix :math:`\in \cM_{nd\times nd}(\Rset)`
    (if the process is of dimension :math:`d`)

Notes
-----
This method makes a discretization of the model on the given :class:`~openturns.Mesh`,
:class:`~openturns.RegularGrid` or :class:`~openturns.Sample` composed of
the vertices :math:`(\vect{t}_1, \dots, \vect{t}_{N-1})` thanks to the
:py:meth:`discretize` method and returns its Cholesky factor.)RAW"
%enddef
%feature("docstring") OT::CovarianceModelImplementation::discretizeAndFactorize
OT_CovarianceModel_discretizeAndFactorize_doc

// ---------------------------------------------------------------------

%define OT_CovarianceModel_discretizeHMatrix_doc
R"RAW(Discretize the covariance function on a given mesh using HMatrix result.

Parameters
----------
where : :class:`~openturns.Mesh` or :class:`~openturns.RegularGrid` or :class:`~openturns.Sample`
    Container of the discretization vertices
hmatParam : :class:`~openturns.HMatrixParameters`
    Parameter values for the HMatrix

Returns
-------
HMatrix : :class:`~openturns.HMatrix`
    Covariance matrix :math:`\in\cS_{nd}^+(\Rset)` (if the process is of
    dimension :math:`d`), stored in hierarchical format (H-Matrix)

Notes
-----
This method is similar to the :py:meth:`discretize` method. This method requires that
OpenTURNS has been compiled with the hmat library.
The method is helpful for very large parameters (Mesh, grid, Sample)
because it compresses data.
)RAW"
%enddef
%feature("docstring") OT::CovarianceModelImplementation::discretizeHMatrix
OT_CovarianceModel_discretizeHMatrix_doc

// ---------------------------------------------------------------------

%define OT_CovarianceModel_discretizeAndFactorizeHMatrix_doc
R"RAW(Discretize and factorize the covariance function on a given mesh.

This uses HMatrix.

Parameters
----------
where : :class:`~openturns.Mesh` or :class:`~openturns.RegularGrid` or :class:`~openturns.Sample`
    Container of the discretization vertices
hmatParam : :class:`~openturns.HMatrixParameters`
    Parameter values for the HMatrix

Returns
-------
HMatrix : :class:`~openturns.HMatrix`
    Cholesk matrix :math:`\in \cS_{nd}^+(\Rset)` (if the process is of
    dimension :math:`d`), stored in hierarchical format (H-Matrix)

Notes
-----
This method is similar to the :py:meth:`discretizeAndFactorize` method. This method requires that
requires that OpenTURNS has been compiled with the hmat library.
The method is helpful for very large parameters (Mesh, grid, Sample)
because it compresses data.
)RAW"
%enddef
%feature("docstring") OT::CovarianceModelImplementation::discretizeAndFactorizeHMatrix
OT_CovarianceModel_discretizeAndFactorizeHMatrix_doc

// ---------------------------------------------------------------------

%define OT_CovarianceModel_discretizeRow_doc
"**(TODO)**"
%enddef
%feature("docstring") OT::CovarianceModelImplementation::discretizeRow
OT_CovarianceModel_discretizeRow_doc


// ---------------------------------------------------------------------

%define OT_CovarianceModel_getAmplitude_doc
R"RAW(Get the amplitude parameter :math:`\vect{\sigma}` of the covariance function.

Returns
-------
amplitude : :class:`~openturns.Point`
    The amplitude parameter :math:`\vect{\sigma} \in \Rset^d` of the covariance
    function.)RAW"
%enddef
%feature("docstring") OT::CovarianceModelImplementation::getAmplitude
OT_CovarianceModel_getAmplitude_doc

// ---------------------------------------------------------------------

%define OT_CovarianceModel_getOutputDimension_doc
R"RAW(Get the dimension :math:`d` of the covariance function.

Returns
-------
d : int
    Dimension :math:`d` such that :math:`C : \cD \times \cD \mapsto \cS_d^+(\Rset).`
    This is the dimension of the process :math:`X`.)RAW"
%enddef
%feature("docstring") OT::CovarianceModelImplementation::getOutputDimension
OT_CovarianceModel_getOutputDimension_doc

// ---------------------------------------------------------------------

%define OT_CovarianceModel_get_marginal
"Get the ith marginal of the model.

Returns
-------
marginal : int or sequence of int
    index of marginal of the model."
%enddef
%feature("docstring") OT::CovarianceModelImplementation::getMarginal
OT_CovarianceModel_get_marginal

// ---------------------------------------------------------------------

%define OT_CovarianceModel_get_nugget_factor_doc
"Accessor to the nugget factor.

This parameter allows smooth predictions from noisy data.
The nugget is added to the diagonal of the assumed training covariance
(thanks to discretize) and acts as a Tikhonov regularization in the
problem.

Returns
-------
nuggetFactor : float
    Nugget factor used to model the observation error variance."
%enddef
%feature("docstring") OT::CovarianceModelImplementation::getNuggetFactor
OT_CovarianceModel_get_nugget_factor_doc

// ---------------------------------------------------------------------

%define OT_CovarianceModel_getParameter_doc
R"RAW(Get the parameters of the covariance function.

Returns
-------
parameters : :class:`~openturns.Point`
    List of the scale parameter :math:`\vect{\theta} \in \Rset^n` and the
    amplitude parameter :math:`\vect{\sigma} \in \Rset^d` of the covariance
    function.

   The other specific parameters are not included.)RAW"
%enddef
%feature("docstring") OT::CovarianceModelImplementation::getParameter
OT_CovarianceModel_getParameter_doc

// ---------------------------------------------------------------------

%define OT_CovarianceModel_getParameterDescription_doc
"Get the description of the covariance function parameters.

Returns
-------
descriptionParam : :class:`~openturns.Description`
    Description of the components of the parameters obtained with the
    *getParameter* method.."
%enddef
%feature("docstring") OT::CovarianceModelImplementation::getParameterDescription
OT_CovarianceModel_getParameterDescription_doc

// ---------------------------------------------------------------------
%define OT_CovarianceModel_getScale_doc
R"RAW(Get the scale parameter :math:`\vect{\theta}` of the covariance function.

Returns
-------
scale : :class:`~openturns.Point`
    The scale parameter :math:`\vect{\theta} \in \Rset^n` used in the
    covariance function.)RAW"
%enddef
%feature("docstring") OT::CovarianceModelImplementation::getScale
OT_CovarianceModel_getScale_doc

// ---------------------------------------------------------------------

%define OT_CovarianceModel_getOutputCorrelation_doc
R"RAW(Get the spatial correlation matrix :math:`\mat{R}` of the covariance function.

Returns
-------
spatialCorrelation : :class:`~openturns.CorrelationMatrix`
    Correlation matrix :math:`\mat{R} \in \cS_d^+(\Rset)`.)RAW"
%enddef
%feature("docstring") OT::CovarianceModelImplementation::getOutputCorrelation
OT_CovarianceModel_getOutputCorrelation_doc

// ---------------------------------------------------------------------

%define OT_CovarianceModel_getInputDimension_doc
"Get the input dimension :math:`n` of the covariance function.

Returns
-------
inputDimension : int
    Spatial dimension :math:`n` of the covariance function."

%enddef
%feature("docstring") OT::CovarianceModelImplementation::getInputDimension
OT_CovarianceModel_getInputDimension_doc

// ---------------------------------------------------------------------

%define OT_CovarianceModel_isDiagonal_doc
"Test whether the model is diagonal or not.

Returns
-------
isDiagonal : bool
    *True* if the model is diagonal."
%enddef
%feature("docstring") OT::CovarianceModelImplementation::isDiagonal
OT_CovarianceModel_isDiagonal_doc

// ---------------------------------------------------------------------

%define OT_CovarianceModel_isStationary_doc
R"RAW(Test whether the model is stationary or not.

Returns
-------
isStationary : bool
    *True* if the model is stationary.

Notes
-----
The covariance function :math:`C` is stationary when it is invariant by
translation:

.. math::

    \forall(\vect{s},\vect{t},\vect{h}) \in \cD \times \cD, & \, \quad
    C(\vect{s}, \vect{s}+\vect{h}) = C(\vect{t}, \vect{t}+\vect{h})


We note :math:`C^{stat}(\vect{\tau})` for :math:`C(\vect{s}, \vect{s}+\vect{\tau})`.)RAW"
%enddef
%feature("docstring") OT::CovarianceModelImplementation::isStationary
OT_CovarianceModel_isStationary_doc

// ---------------------------------------------------------------------

%define OT_CovarianceModel_partialGradient_doc
R"RAW(Compute the gradient of the covariance function.

Parameters
----------
s, t : floats or sequences of float
    Multivariate index :math:`(\vect{s}, \vect{t}) \in \cD \times \cD`.

Returns
-------
gradient : :class:`~openturns.Matrix`
    Gradient of the covariance function.)RAW"
%enddef
%feature("docstring") OT::CovarianceModelImplementation::partialGradient
OT_CovarianceModel_partialGradient_doc

// ---------------------------------------------------------------------

%define OT_CovarianceModel_parameterGradient_doc
R"RAW(Compute the gradient according to the parameters.

Parameters
----------
s, t : sequences of float
    Multivariate index :math:`(\vect{s}, \vect{t}) \in \cD \times \cD`.

Returns
-------
gradient : :class:`~openturns.Matrix`
    Gradient of the function according to the parameters.)RAW"
%enddef
%feature("docstring") OT::CovarianceModelImplementation::parameterGradient
OT_CovarianceModel_parameterGradient_doc

// ---------------------------------------------------------------------

%define OT_CovarianceModel_setAmplitude_doc
R"RAW(Set the amplitude parameter :math:`\vect{\sigma}` of the covariance function.

Parameters
----------
amplitude : :class:`~openturns.Point`
    The amplitude parameter :math:`\vect{\sigma} \in \Rset^d` to be used in the
    covariance function.
    Its size must be equal to the dimension of the covariance function.)RAW"
%enddef
%feature("docstring") OT::CovarianceModelImplementation::setAmplitude
OT_CovarianceModel_setAmplitude_doc

// ---------------------------------------------------------------------

%define OT_CovarianceModel_setScale_doc
R"RAW(Set the scale parameter :math:`\vect{\theta}` of the covariance function.

Parameters
----------
scale : :class:`~openturns.Point`
    The scale parameter :math:`\vect{\theta} \in \Rset^n` to be used in the
    covariance function.
    Its size must be equal to the input dimension of the covariance function.)RAW"
%enddef
%feature("docstring") OT::CovarianceModelImplementation::setScale
OT_CovarianceModel_setScale_doc

// ---------------------------------------------------------------------

%define OT_CovarianceModel_set_nugget_factor_doc
"Set the nugget factor for the variance of the observation error.

Acts on the discretized covariance matrix.

Parameters
----------
nuggetFactor : float
    nugget factor to be used to model the variance of the observation error."
%enddef
%feature("docstring") OT::CovarianceModelImplementation::setNuggetFactor
OT_CovarianceModel_set_nugget_factor_doc

// ---------------------------------------------------------------------

%define OT_CovarianceModel_setParameter_doc
R"RAW(Set the parameters of the covariance function.

Parameters
----------
parameters : :class:`~openturns.Point`
    List of the scale parameter :math:`\vect{\theta} \in \Rset^n` and the
    amplitude parameter :math:`\vect{\sigma} \in \Rset^d` of the covariance
    function.

    Must be of dimension :math:`n+d`.)RAW"
%enddef
%feature("docstring") OT::CovarianceModelImplementation::setParameter
OT_CovarianceModel_setParameter_doc

// ---------------------------------------------------------------------

%define OT_CovarianceModel_setOutputCorrelation_doc
R"RAW(Set the spatial correlation matrix :math:`\mat{R}` of the covariance function.

Parameters
----------
spatialCorrelation : :class:`~openturns.CorrelationMatrix`
    Correlation matrix :math:`\mat{R} \in \cS_d^+([-1,1])`.)RAW"
%enddef
%feature("docstring") OT::CovarianceModelImplementation::setOutputCorrelation
OT_CovarianceModel_setOutputCorrelation_doc

// ---------------------------------------------------------------------

%define OT_CovarianceModel_operator_doc
R"RAW(Evaluate the covariance function.

Available usages:
    __call__(s, t)

    __call__(tau)

Parameters
----------
s, t : floats or sequences of float
    Multivariate index :math:`(\vect{s}, \vect{t}) \in \cD \times \cD`.
tau : float or sequence of float
    Multivariate index :math:`\vect{\tau} \in \cD`.

Returns
-------
covariance : CovarianceMatrix
    The evaluation of the covariance function.)RAW"
%enddef
%feature("docstring") OT::CovarianceModelImplementation::operator()
OT_CovarianceModel_operator_doc

// ---------------------------------------------------------------------

%define OT_CovarianceModel_setActiveParameter_doc
R"RAW(Accessor to the active parameter set.

This method defines which hyperparameters are tuned.

Parameters
----------
active : sequence of int
    Indices of the active parameters.

Notes
-----
In order to get the position of a parameter known by its name,  we have to first call the method
:py:meth:`getFullParameterDescription` and then call the method *find* as shown in the following
example.

Examples
--------
For example, we create a :class:`~openturns.MaternModel` covariance model. By default,
only the scale and the amplitude parameters are active and will be estimated.
We want to add the estimation of the :math:`\nu` parameter.

>>> import openturns as ot
>>> covModel = ot.MaternModel()

We get the default active parameters description: we can see that the :math:`\nu` parameter is not
included in the list.

>>> covModel.getParameterDescription()
[scale_0,amplitude_0]

We now get the active parameters. The parameter :math:`\nu`  is not in the list.

>>> active = covModel.getActiveParameter()
>>> print(active)
[0,2]

We want to search the index associated to the parameter :math:`\nu` in the full list of the
parameters indices.

>>> activeDesc = covModel.getFullParameterDescription()
>>> print(activeDesc)
[scale_0,nuggetFactor,amplitude_0,nu]
>>> index = activeDesc.find('nu')
>>> print(index)
3

Now, we add the index of :math:`\nu` in the initial list of active parameters. We sort the active
indices by increasing order. Then we update the list of the active parameters.

>>> active = covModel.getActiveParameter()
>>> active.add(index)
>>> covModel.setActiveParameter(sorted(active))

Check that the parameter :math:`\nu` is in the active parameters list!

>>> covModel.getActiveParameter()
[0,2,3]

>>> covModel.getFullParameterDescription()
[scale_0,nuggetFactor,amplitude_0,nu]
)RAW"
%enddef
%feature("docstring") OT::CovarianceModelImplementation::setActiveParameter
OT_CovarianceModel_setActiveParameter_doc

// ---------------------------------------------------------------------

%define OT_CovarianceModel_getActiveParameter_doc
"Accessor to the active parameter set.

In the context of kriging, it allows one to choose which hyperparameters are tuned.

Returns
-------
active : :class:`~openturns.Indices`
    Indices of the active parameters."
%enddef
%feature("docstring") OT::CovarianceModelImplementation::getActiveParameter
OT_CovarianceModel_getActiveParameter_doc

// ---------------------------------------------------------------------

%define OT_CovarianceModel_activateScale_doc
"Activate/deactivate the scale parameter(s).

In the context of Kriging, defines whether scale parameters should be tuned.

Parameters
----------
isScaleActive : bool
    If True, the scale parameters are all tuned.
    If False, none of them is tuned."
%enddef
%feature("docstring") OT::CovarianceModelImplementation::activateScale
OT_CovarianceModel_activateScale_doc

// ---------------------------------------------------------------------

%define OT_CovarianceModel_activateNuggetFactor_doc
"Activate/deactivate the nugget factor.

In the context of Kriging, defines whether the nugget factor should be tuned.

Parameters
----------
isNuggetFactorActive : bool
    If True (resp. False), the nugget factor is (resp. is not) tuned."
%enddef
%feature("docstring") OT::CovarianceModelImplementation::activateNuggetFactor
OT_CovarianceModel_activateNuggetFactor_doc

// ---------------------------------------------------------------------

%define OT_CovarianceModel_activateAmplitude_doc
"Activate/deactivate the amplitude parameter(s).

In the context of Kriging, defines whether amplitude parameters should be tuned.

Parameters
----------
isAmplitudeActive : bool
    If True, the amplitude parameters are all tuned.
    If False, none of them is tuned."
%enddef
%feature("docstring") OT::CovarianceModelImplementation::activateAmplitude
OT_CovarianceModel_activateAmplitude_doc

// ---------------------------------------------------------------------

%define OT_CovarianceModel_draw_doc
R"RAW(Draw a specific component of the covariance model with input dimension 1.

Parameters
----------
rowIndex : int, :math:`0 \leq rowIndex < dimension`
    The row index of the component to draw. Default value is 0.
columnIndex: int, :math:`0 \leq columnIndex < dimension`
    The column index of the component to draw. Default value is 0.
tMin : float
    The lower bound of the range over which the model is plotted. Default value is *CovarianceModel-DefaultTMin* in :class:`~openturns.ResourceMap`.
tMax : float
    The upper bound of the range over which the model is plotted. Default value is *CovarianceModel-DefaultTMax* in :class:`~openturns.ResourceMap`.
pointNumber : int, :math:`pointNumber \geq 2`
    The discretization of the range :math:`[tMin,tMax]` over which the model is plotted. Default value is *CovarianceModel-DefaultPointNumber* in  class:`~openturns.ResourceMap`.
asStationary : bool
    Flag to tell if the model has to be plotted as a stationary model, ie as a function of the lag :math:`\tau=t-s` if equals to *True*, or as a non-stationary model, ie as a function of :math:`(s,t)` if equals to *False*. Default value is *True*.
correlationFlag : bool
    Flag to tell if the model has to be plotted as a correlation function if equals to *True* or as a covariance function if equals to *False*. Default value is *False*.

Returns
-------
graph : :class:`~openturns.Graph`
    A graph containing a unique curve if *asStationary=True* and if the model is actually a stationary model, or containing the iso-values of the model if *asStationary=False* or if the model is nonstationary.

)RAW"
%enddef
%feature("docstring") OT::CovarianceModelImplementation::draw
OT_CovarianceModel_draw_doc

// ---------------------------------------------------------------------

%define OT_CovarianceModel_setFullParameter_doc
R"RAW(Set the full parameters of the covariance function.

Parameters
----------
parameter : :class:`~openturns.Point`
    List the full parameter of the covariance function i.e.
    scale parameter :math:`\vect{\theta} \in \Rset^n`, the
    the amplitude parameter :math:`\vect{\sigma} \in \Rset^d`,
    the Spatial correlation parameter  :math:`\mat{R} \in \cS_d^+([-1,1])`;
    and potential other parameter depending on the model;

    Must be at least of dimension :math:`n+\frac{d(d+1)}{2}`.)RAW"
%enddef
%feature("docstring") OT::CovarianceModelImplementation::setFullParameter
OT_CovarianceModel_setFullParameter_doc

// ---------------------------------------------------------------------

%define OT_CovarianceModel_getFullParameter_doc
R"RAW(Get the full parameters of the covariance function.

Returns
-------
parameter : :class:`~openturns.Point`
    List the full parameter of the covariance function i.e.
    scale parameter :math:`\vect{\theta} \in \Rset^n`, the
    the amplitude parameter :math:`\vect{\sigma} \in \Rset^d`,
    the Spatial correlation parameter  :math:`\mat{R} \in \cS_d^+([-1,1])`;
    and potential other parameter depending on the model;
)RAW"
%enddef
%feature("docstring") OT::CovarianceModelImplementation::getFullParameter
OT_CovarianceModel_getFullParameter_doc

// ---------------------------------------------------------------------

%define OT_CovarianceModel_getFullParameterDescription_doc
"Get the description full parameters of the covariance function.

Returns
-------
description : :class:`~openturns.Description`
    Description of the full parameter of the covariance function.
"
%enddef
%feature("docstring") OT::CovarianceModelImplementation::getFullParameterDescription
OT_CovarianceModel_getFullParameterDescription_doc

// ---------------------------------------------------------------------

%define OT_CovarianceModel_isParallel_doc
"Accessor to the parallel flag.

Returns
-------
isParallel : bool
    Whether the object is considered thread-safe.
"
%enddef
%feature("docstring") OT::CovarianceModelImplementation::isParallel
OT_CovarianceModel_isParallel_doc
