/* Copyright (c) 2017, Google Inc.
 *
 * Permission to use, copy, modify, and/or distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY
 * SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION
 * OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN
 * CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE. */

#include <openssl/crypto.h>

#include <stdio.h>
#include <stdlib.h>

#include <openssl/aead.h>
#include <openssl/aes.h>
#include <openssl/bn.h>
#include <openssl/ctrdrbg.h>
#include <openssl/curve25519.h>
#include <openssl/dh.h>
#include <openssl/digest.h>
#include <openssl/ec.h>
#include <openssl/ecdsa.h>
#include <openssl/ec_key.h>
#include <openssl/evp.h>
#include <openssl/hkdf.h>
#include <openssl/hmac.h>
#include <openssl/kdf.h>
#include <openssl/nid.h>
#include <openssl/rsa.h>
#include <openssl/sha.h>

#include "../../internal.h"
#include "../curve25519/internal.h"
#include "../dh/internal.h"
#include "../ec/internal.h"
#include "../ecdsa/internal.h"
#include "../ml_kem/ml_kem.h"
#include "../rand/internal.h"
#include "../rsa/internal.h"

// This should track the biggest input/output used in self_check.c
#define MAX_HEXDUMP_SIZE (MLDSA44_PRIVATE_KEY_BYTES * 2)
#define ERROR_FORMAT "%s failed.\nExpected:   %s\nCalculated: %s\n"
// The current max is "ML-KEM-encapsulate-shared-secret\0"
#define MAX_NAME 33
#define MAX_ERROR_MSG_SIZE (( 2 * MAX_HEXDUMP_SIZE) + sizeof(ERROR_FORMAT) + MAX_NAME)

static void hexdump(char buf[MAX_HEXDUMP_SIZE], const uint8_t *in, size_t in_len) {
    assert(in_len * 2 < MAX_HEXDUMP_SIZE);
    size_t pos = 0;
    for (size_t i = 0; i < in_len; i++) {
        pos += snprintf(buf + pos, MAX_HEXDUMP_SIZE - pos, "%02x", in[i]);
    }
}

static int check_test_optional_abort(const void *expected, const void *actual,
                      size_t expected_len, const char *name, const bool call_fips_failure) {
  char expected_hex[MAX_HEXDUMP_SIZE] = {0};
  char actual_hex[MAX_HEXDUMP_SIZE] = {0};
  char error_msg[MAX_ERROR_MSG_SIZE] = {0};
  if (OPENSSL_memcmp(actual, expected, expected_len) != 0) {
    assert(sizeof(name) < MAX_NAME);
    hexdump(expected_hex, expected, expected_len);
    hexdump(actual_hex, actual, expected_len);

    snprintf(error_msg, sizeof(error_msg),
               "%s failed.\nExpected:   %s\nCalculated: %s\n",
               name, expected_hex, actual_hex);
    if (call_fips_failure) {
      AWS_LC_FIPS_failure(error_msg);
    } else {
      fprintf(stderr, "%s\n", error_msg);
    }
    return 0;
  }
  return 1;
}

static int check_test(const void *expected, const void *actual,
                      size_t expected_len, const char *name) {
  return check_test_optional_abort(expected, actual, expected_len, name, true);
}

static int set_bignum(BIGNUM **out, const uint8_t *in, size_t len) {
  *out = BN_bin2bn(in, len, NULL);
  return *out != NULL;
}

static int serialize_ecdsa_sig(uint8_t *out, size_t out_len,
                               const ECDSA_SIG *sig) {
  if ((out_len & 1) ||  //
      !BN_bn2bin_padded(out, out_len / 2, sig->r) ||
      !BN_bn2bin_padded(out + out_len / 2, out_len / 2, sig->s)) {
    return 0;
  }
  return 1;
}

static ECDSA_SIG *parse_ecdsa_sig(const uint8_t *in, size_t in_len) {
  ECDSA_SIG *ret = ECDSA_SIG_new();
  if (!ret || //
      (in_len & 1) ||
      BN_bin2bn(in, in_len/2, ret->r) == NULL ||
      BN_bin2bn(in + in_len/2, in_len/2, ret->s) == NULL) {
    ECDSA_SIG_free(ret);
    ret = NULL;
  }
  return ret;
}

static RSA *self_test_rsa_key(void) {
  static const uint8_t kN[] = {
      0xd3, 0x3a, 0x62, 0x9f, 0x07, 0x77, 0xb0, 0x18, 0xf3, 0xff, 0xfe, 0xcc,
      0xc9, 0xa2, 0xc2, 0x3a, 0xa6, 0x1d, 0xd8, 0xf0, 0x26, 0x5b, 0x38, 0x90,
      0x17, 0x48, 0x15, 0xce, 0x21, 0xcd, 0xd6, 0x62, 0x99, 0xe2, 0xd7, 0xda,
      0x40, 0x80, 0x3c, 0xad, 0x18, 0xb7, 0x26, 0xe9, 0x30, 0x8a, 0x23, 0x3f,
      0x68, 0x9a, 0x9c, 0x31, 0x34, 0x91, 0x99, 0x06, 0x11, 0x36, 0xb2, 0x9e,
      0x3a, 0xd0, 0xbc, 0xb9, 0x93, 0x4e, 0xb8, 0x72, 0xa1, 0x9f, 0xb6, 0x8c,
      0xd5, 0x17, 0x1f, 0x7e, 0xaa, 0x75, 0xbb, 0xdf, 0xa1, 0x70, 0x48, 0xc4,
      0xec, 0x9a, 0x51, 0xed, 0x41, 0xc9, 0x74, 0xc0, 0x3e, 0x1e, 0x85, 0x2f,
      0xbe, 0x34, 0xc7, 0x65, 0x34, 0x8b, 0x4d, 0x55, 0x4b, 0xe1, 0x45, 0x54,
      0x0d, 0x75, 0x7e, 0x89, 0x4d, 0x0c, 0xf6, 0x33, 0xe5, 0xfc, 0xfb, 0x56,
      0x1b, 0xf2, 0x39, 0x9d, 0xe0, 0xff, 0x55, 0xcf, 0x02, 0x05, 0xb9, 0x74,
      0xd2, 0x91, 0xfc, 0x87, 0xe1, 0xbb, 0x97, 0x2a, 0xe4, 0xdd, 0x20, 0xc0,
      0x38, 0x47, 0xc0, 0x76, 0x3f, 0xa1, 0x9b, 0x5c, 0x20, 0xff, 0xff, 0xc7,
      0x49, 0x3b, 0x4c, 0xaf, 0x99, 0xa6, 0x3e, 0x82, 0x5c, 0x58, 0x27, 0xce,
      0x01, 0x03, 0xc3, 0x16, 0x35, 0x20, 0xe9, 0xf0, 0x15, 0x7a, 0x41, 0xd5,
      0x1f, 0x52, 0xea, 0xdf, 0xad, 0x4c, 0xbb, 0x0d, 0xcb, 0x04, 0x91, 0xb0,
      0x95, 0xa8, 0xce, 0x25, 0xfd, 0xd2, 0x62, 0x47, 0x77, 0xee, 0x13, 0xf1,
      0x48, 0x72, 0x9e, 0xd9, 0x2d, 0xe6, 0x5f, 0xa4, 0xc6, 0x9e, 0x5a, 0xb2,
      0xc6, 0xa2, 0xf7, 0x0a, 0x16, 0x17, 0xae, 0x6b, 0x1c, 0x30, 0x7c, 0x63,
      0x08, 0x83, 0xe7, 0x43, 0xec, 0x54, 0x5e, 0x2c, 0x08, 0x0b, 0x5e, 0x46,
      0xa7, 0x10, 0x93, 0x43, 0x53, 0x4e, 0xe3, 0x16, 0x73, 0x55, 0xce, 0xf2,
      0x94, 0xc0, 0xbe, 0xb3,
  };
  static const uint8_t kE[] = {0x01, 0x00, 0x01};  // 65537
  static const uint8_t kD[] = {
      0x2f, 0x2c, 0x1e, 0xd2, 0x3d, 0x2c, 0xb1, 0x9b, 0x21, 0x02, 0xce, 0xb8,
      0x95, 0x5f, 0x4f, 0xd9, 0x21, 0x38, 0x11, 0x36, 0xb0, 0x9a, 0x36, 0xab,
      0x97, 0x47, 0x75, 0xf7, 0x2e, 0xfd, 0x75, 0x1f, 0x58, 0x16, 0x9c, 0xf6,
      0x14, 0xe9, 0x8e, 0xa3, 0x69, 0x9d, 0x9d, 0x86, 0xfe, 0x5c, 0x1b, 0x3b,
      0x11, 0xf5, 0x55, 0x64, 0x77, 0xc4, 0xfc, 0x53, 0xaa, 0x8c, 0x78, 0x9f,
      0x75, 0xab, 0x20, 0x3a, 0xa1, 0x77, 0x37, 0x22, 0x02, 0x8e, 0x54, 0x8a,
      0x67, 0x1c, 0x5e, 0xe0, 0x3e, 0xd9, 0x44, 0x37, 0xd1, 0x29, 0xee, 0x56,
      0x6c, 0x30, 0x9a, 0x93, 0x4d, 0xd9, 0xdb, 0xc5, 0x03, 0x1a, 0x75, 0xcc,
      0x0f, 0xc2, 0x61, 0xb5, 0x6c, 0x62, 0x9f, 0xc6, 0xa8, 0xc7, 0x8a, 0x60,
      0x17, 0x11, 0x62, 0x4c, 0xef, 0x74, 0x31, 0x97, 0xad, 0x89, 0x2d, 0xe8,
      0x31, 0x1d, 0x8b, 0x58, 0x82, 0xe3, 0x03, 0x1a, 0x6b, 0xdf, 0x3f, 0x3e,
      0xa4, 0x27, 0x19, 0xef, 0x46, 0x7a, 0x90, 0xdf, 0xa7, 0xe7, 0xc9, 0x66,
      0xab, 0x41, 0x1d, 0x65, 0x78, 0x1c, 0x18, 0x40, 0x5c, 0xd6, 0x87, 0xb5,
      0xea, 0x29, 0x44, 0xb3, 0xf5, 0xb3, 0xd2, 0x4f, 0xce, 0x88, 0x78, 0x49,
      0x27, 0x4e, 0x0b, 0x30, 0x85, 0xfb, 0x73, 0xfd, 0x8b, 0x32, 0x15, 0xee,
      0x1f, 0xc9, 0x0e, 0x89, 0xb9, 0x43, 0x2f, 0xe9, 0x60, 0x8d, 0xda, 0xae,
      0x2b, 0x30, 0x99, 0xee, 0x88, 0x81, 0x20, 0x7b, 0x4a, 0xc3, 0x18, 0xf2,
      0x94, 0x02, 0x79, 0x94, 0xaa, 0x65, 0xd9, 0x1b, 0x45, 0x2a, 0xac, 0x6e,
      0x30, 0x48, 0x57, 0xea, 0xbe, 0x79, 0x7d, 0xfc, 0x67, 0xaa, 0x47, 0xc0,
      0xf7, 0x52, 0xfd, 0x0b, 0x63, 0x4e, 0x3d, 0x2e, 0xcc, 0x36, 0xa0, 0xdb,
      0x92, 0x0b, 0xa9, 0x1b, 0xeb, 0xc2, 0xd5, 0x08, 0xd3, 0x85, 0x87, 0xf8,
      0x5d, 0x1a, 0xf6, 0xc1,
  };
  static const uint8_t kP[] = {
      0xf7, 0x06, 0xa3, 0x98, 0x8a, 0x52, 0xf8, 0x63, 0x68, 0x27, 0x4f, 0x68,
      0x7f, 0x34, 0xec, 0x8e, 0x5d, 0xf8, 0x30, 0x92, 0xb3, 0x62, 0x4c, 0xeb,
      0xdb, 0x19, 0x6b, 0x09, 0xc5, 0xa3, 0xf0, 0xbb, 0xff, 0x0f, 0xc2, 0xd4,
      0x9b, 0xc9, 0x54, 0x4f, 0xb9, 0xf9, 0xe1, 0x4c, 0xf0, 0xe3, 0x4c, 0x90,
      0xda, 0x7a, 0x01, 0xc2, 0x9f, 0xc4, 0xc8, 0x8e, 0xb1, 0x1e, 0x93, 0x75,
      0x75, 0xc6, 0x13, 0x25, 0xc3, 0xee, 0x3b, 0xcc, 0xb8, 0x72, 0x6c, 0x49,
      0xb0, 0x09, 0xfb, 0xab, 0x44, 0xeb, 0x4d, 0x40, 0xf0, 0x61, 0x6b, 0xe5,
      0xe6, 0xfe, 0x3e, 0x0a, 0x77, 0x26, 0x39, 0x76, 0x3d, 0x4c, 0x3e, 0x9b,
      0x5b, 0xc0, 0xaf, 0xa2, 0x58, 0x76, 0xb0, 0xe9, 0xda, 0x7f, 0x0e, 0x78,
      0xc9, 0x76, 0x49, 0x5c, 0xfa, 0xb3, 0xb0, 0x15, 0x4b, 0x41, 0xc7, 0x27,
      0xa4, 0x75, 0x28, 0x5c, 0x30, 0x69, 0x50, 0x29,
  };
  static const uint8_t kQ[] = {
      0xda, 0xe6, 0xd2, 0xbb, 0x44, 0xff, 0x4f, 0xdf, 0x57, 0xc1, 0x11, 0xa3,
      0x51, 0xba, 0x17, 0x89, 0x4c, 0x01, 0xc0, 0x0c, 0x97, 0x34, 0x50, 0xcf,
      0x32, 0x1e, 0xc0, 0xbd, 0x7b, 0x35, 0xb5, 0x6a, 0x26, 0xcc, 0xea, 0x4c,
      0x8e, 0x87, 0x4a, 0x67, 0x8b, 0xd3, 0xe5, 0x4f, 0x3a, 0x60, 0x48, 0x59,
      0x04, 0x93, 0x39, 0xd7, 0x7c, 0xfb, 0x19, 0x1a, 0x34, 0xd5, 0xe8, 0xaf,
      0xe7, 0x22, 0x2c, 0x0d, 0xc2, 0x91, 0x69, 0xb6, 0xe9, 0x2a, 0xe9, 0x1c,
      0x4c, 0x6e, 0x8f, 0x40, 0xf5, 0xa8, 0x3e, 0x82, 0x69, 0x69, 0xbe, 0x9f,
      0x7d, 0x5c, 0x7f, 0x92, 0x78, 0x17, 0xa3, 0x6d, 0x41, 0x2d, 0x72, 0xed,
      0x3f, 0x71, 0xfa, 0x97, 0xb4, 0x63, 0xe4, 0x4f, 0xd9, 0x46, 0x03, 0xfb,
      0x00, 0xeb, 0x30, 0x70, 0xb9, 0x51, 0xd9, 0x0a, 0xd2, 0xf8, 0x50, 0xd4,
      0xfb, 0x43, 0x84, 0xf8, 0xac, 0x58, 0xc3, 0x7b,
  };
  static const uint8_t kDModPMinusOne[] = {
      0xf5, 0x50, 0x8f, 0x88, 0x7d, 0xdd, 0xb5, 0xb4, 0x2a, 0x8b, 0xd7, 0x4d,
      0x23, 0xfe, 0xaf, 0xe9, 0x16, 0x22, 0xd2, 0x41, 0xed, 0x88, 0xf2, 0x70,
      0xcb, 0x4d, 0xeb, 0xc1, 0x71, 0x97, 0xc4, 0x0b, 0x3e, 0x5a, 0x2d, 0x96,
      0xab, 0xfa, 0xfd, 0x12, 0x8b, 0xd3, 0x3e, 0x4e, 0x05, 0x6f, 0x04, 0xeb,
      0x59, 0x3c, 0x0e, 0xa1, 0x73, 0xbe, 0x9d, 0x99, 0x2f, 0x05, 0xf9, 0x54,
      0x8d, 0x98, 0x1e, 0x0d, 0xc4, 0x0c, 0xc3, 0x30, 0x23, 0xff, 0xe5, 0xd0,
      0x2b, 0xd5, 0x4e, 0x2b, 0xa0, 0xae, 0xb8, 0x32, 0x84, 0x45, 0x8b, 0x3c,
      0x6d, 0xf0, 0x10, 0x36, 0x9e, 0x6a, 0xc4, 0x67, 0xca, 0xa9, 0xfc, 0x06,
      0x96, 0xd0, 0xbc, 0xda, 0xd1, 0x55, 0x55, 0x8d, 0x77, 0x21, 0xf4, 0x82,
      0x39, 0x37, 0x91, 0xd5, 0x97, 0x56, 0x78, 0xc8, 0x3c, 0xcb, 0x5e, 0xf6,
      0xdc, 0x58, 0x48, 0xb3, 0x7c, 0x94, 0x29, 0x39,
  };
  static const uint8_t kDModQMinusOne[] = {
      0x64, 0x65, 0xbd, 0x7d, 0x1a, 0x96, 0x26, 0xa1, 0xfe, 0xf3, 0x94, 0x0d,
      0x5d, 0xec, 0x85, 0xe2, 0xf8, 0xb3, 0x4c, 0xcb, 0xf9, 0x85, 0x8b, 0x12,
      0x9c, 0xa0, 0x32, 0x32, 0x35, 0x92, 0x5a, 0x94, 0x47, 0x1b, 0x70, 0xd2,
      0x90, 0x04, 0x49, 0x01, 0xd8, 0xc5, 0xe4, 0xc4, 0x43, 0xb7, 0xe9, 0x36,
      0xba, 0xbc, 0x73, 0xa8, 0xfb, 0xaf, 0x86, 0xc1, 0xd8, 0x3d, 0xcb, 0xac,
      0xf1, 0xcb, 0x60, 0x7d, 0x27, 0x21, 0xde, 0x64, 0x7f, 0xe8, 0xa8, 0x65,
      0xcc, 0x40, 0x60, 0xff, 0xa0, 0x2b, 0xfc, 0x0f, 0x80, 0x1d, 0x79, 0xca,
      0x58, 0x8a, 0xd6, 0x0f, 0xed, 0x78, 0x9a, 0x02, 0x00, 0x04, 0xc2, 0x53,
      0x41, 0xe8, 0x1a, 0xd0, 0xfd, 0x71, 0x5b, 0x43, 0xac, 0x19, 0x4a, 0xb6,
      0x12, 0xa3, 0xcb, 0xe1, 0xc7, 0x7d, 0x5c, 0x98, 0x74, 0x4e, 0x63, 0x74,
      0x6b, 0x91, 0x7a, 0x29, 0x3b, 0x92, 0xb2, 0x85,
  };
  static const uint8_t kQInverseModP[] = {
      0xd0, 0xde, 0x19, 0xda, 0x1e, 0xa2, 0xd8, 0x8f, 0x1c, 0x92, 0x73, 0xb0,
      0xc9, 0x90, 0xc7, 0xf5, 0xec, 0xc5, 0x89, 0x01, 0x05, 0x78, 0x11, 0x2d,
      0x74, 0x34, 0x44, 0xad, 0xd5, 0xf7, 0xa4, 0xfe, 0x9f, 0x25, 0x4d, 0x0b,
      0x92, 0xe3, 0xb8, 0x7d, 0xd3, 0xfd, 0xa5, 0xca, 0x95, 0x60, 0xa3, 0xf9,
      0x55, 0x42, 0x14, 0xb2, 0x45, 0x51, 0x9f, 0x73, 0x88, 0x43, 0x8a, 0xd1,
      0x65, 0x9e, 0xd1, 0xf7, 0x82, 0x2a, 0x2a, 0x8d, 0x70, 0x56, 0xe3, 0xef,
      0xc9, 0x0e, 0x2a, 0x2c, 0x15, 0xaf, 0x7f, 0x97, 0x81, 0x66, 0xf3, 0xb5,
      0x00, 0xa9, 0x26, 0xcc, 0x1e, 0xc2, 0x98, 0xdd, 0xd3, 0x37, 0x06, 0x79,
      0xb3, 0x60, 0x58, 0x79, 0x99, 0x3f, 0xa3, 0x15, 0x1f, 0x31, 0xe3, 0x11,
      0x88, 0x4c, 0x35, 0x57, 0xfa, 0x79, 0xd7, 0xd8, 0x72, 0xee, 0x73, 0x95,
      0x89, 0x29, 0xc7, 0x05, 0x27, 0x68, 0x90, 0x15,
  };

  RSA *rsa = RSA_new();
  if (rsa == NULL ||
      !set_bignum(&rsa->n, kN, sizeof(kN)) ||
      !set_bignum(&rsa->e, kE, sizeof(kE)) ||
      !set_bignum(&rsa->d, kD, sizeof(kD)) ||
      !set_bignum(&rsa->p, kP, sizeof(kP)) ||
      !set_bignum(&rsa->q, kQ, sizeof(kQ)) ||
      !set_bignum(&rsa->dmp1, kDModPMinusOne, sizeof(kDModPMinusOne)) ||
      !set_bignum(&rsa->dmq1, kDModQMinusOne, sizeof(kDModQMinusOne)) ||
      !set_bignum(&rsa->iqmp, kQInverseModP, sizeof(kQInverseModP))) {
    RSA_free(rsa);
    return NULL;
  }

  return rsa;
}

static DH *self_test_ffc_dh_key(const uint8_t *p, size_t p_len,
                                const uint8_t *q, size_t q_len,
                                const uint8_t *g, size_t g_len,
                                const uint8_t *priv_key, size_t priv_key_len,
                                const uint8_t *pub_key, size_t pub_key_len) {
  DH *dh = DH_new();
  if (dh == NULL ||
      !set_bignum(&dh->p, p, p_len) ||
      !set_bignum(&dh->q, q, q_len) ||
      !set_bignum(&dh->g, g, g_len) ||
      !set_bignum(&dh->priv_key, priv_key, priv_key_len) ||
      !set_bignum(&dh->pub_key, pub_key, pub_key_len)) {
    DH_free(dh);
    return NULL;
  }
  return dh;
}

// domainParameterGenerationMode is FB
static DH *self_test_ffc_dh_fb_key(void) {
  // The data is fetched from ACVP data.
  // Details are available in CryptoAlg-851?selectedConversation=c0120d22-f2bd-4eae-8ae6-b04fcca86743
  // File name: 197488/683891/testvector-request.json
  // Other details: "tgId": 2, "tcId": 6
  static const uint8_t kDH_p[256] = {
      0xd9, 0xc4, 0x37, 0xc1, 0xa3, 0xe3, 0x5c, 0xb8, 0xbe, 0xc8, 0x0a, 0x83,
      0xa3, 0xd3, 0x04, 0x3f, 0xd1, 0x4c, 0x40, 0xbd, 0x3c, 0x3c, 0x29, 0x26,
      0xe5, 0x16, 0xde, 0xde, 0xd7, 0xdf, 0xe6, 0x68, 0x4f, 0x85, 0xa2, 0x46,
      0x65, 0xc1, 0xaf, 0xa8, 0x61, 0x99, 0xf8, 0xc3, 0xd9, 0xd3, 0xa9, 0xc9,
      0x15, 0xa9, 0x13, 0x5c, 0x0b, 0xac, 0xf6, 0x7c, 0xbc, 0xe9, 0x35, 0x6c,
      0xac, 0xf1, 0x28, 0xa2, 0x43, 0xef, 0xfd, 0x2a, 0x59, 0xd8, 0x5d, 0x89,
      0x7e, 0x28, 0x90, 0x46, 0xf5, 0x44, 0xa8, 0x07, 0x8e, 0x70, 0xef, 0x27,
      0x36, 0x78, 0x24, 0x3a, 0x22, 0xd4, 0x16, 0xa8, 0xb4, 0xea, 0x6c, 0x13,
      0x6c, 0xd9, 0x98, 0xc2, 0x59, 0xfe, 0xdb, 0x99, 0xf9, 0xeb, 0x07, 0x7d,
      0x43, 0xdd, 0x5e, 0xbf, 0x39, 0x59, 0xb4, 0xb8, 0x64, 0x88, 0xfe, 0xdb,
      0x6c, 0x8a, 0x43, 0xac, 0x67, 0x13, 0xa9, 0x78, 0x79, 0xa7, 0xff, 0x4e,
      0xc6, 0x1a, 0x3e, 0x13, 0x3e, 0xe8, 0x15, 0x57, 0xab, 0x29, 0xe7, 0x86,
      0x5e, 0x8c, 0xb3, 0x24, 0xb0, 0x3c, 0x6a, 0x3c, 0x7d, 0x4c, 0xc9, 0xb5,
      0x4d, 0x31, 0xb5, 0x0a, 0x28, 0xbb, 0x7d, 0x02, 0x75, 0x12, 0xa6, 0xca,
      0x9d, 0x1f, 0x28, 0x45, 0xb9, 0xaf, 0x63, 0x66, 0xbb, 0x49, 0x3f, 0x28,
      0x66, 0x8c, 0xb5, 0x1d, 0x81, 0x08, 0xbd, 0x1e, 0x6b, 0x43, 0x31, 0x4e,
      0x88, 0x48, 0x25, 0xe8, 0xc4, 0xd7, 0x83, 0x5f, 0xa3, 0x65, 0x1a, 0xc7,
      0x2d, 0x3b, 0xeb, 0xb0, 0xb2, 0xa9, 0x83, 0x96, 0xaa, 0x7c, 0x23, 0x15,
      0x0e, 0x2e, 0x6e, 0x46, 0x5d, 0x1b, 0x34, 0xc0, 0x70, 0xfd, 0xa8, 0x8d,
      0x82, 0x97, 0x10, 0xd4, 0x3e, 0x45, 0x16, 0x3e, 0x54, 0x42, 0x4a, 0x2a,
      0x76, 0x5c, 0x6f, 0x30, 0x44, 0x1d, 0xf8, 0xc7, 0x07, 0xe2, 0xb8, 0xd9,
      0xac, 0x74, 0x73, 0x09
  };
  static const uint8_t kDH_q[28] = {
      0xcc, 0x9c, 0x34, 0x91, 0x8e, 0x8b, 0xa0, 0x86, 0x23, 0xa2, 0x76, 0x82,
      0xeb, 0xd1, 0x98, 0x5f, 0xab, 0x27, 0x56, 0x44, 0x66, 0x70, 0x50, 0xc9,
      0x35, 0xdc, 0x14, 0xc7
  };
  static const uint8_t kDH_g[256] = {
      0x31, 0xee, 0x30, 0xb2, 0x17, 0x6f, 0x2a, 0xd1, 0x67, 0x09, 0xdf, 0x01,
      0x38, 0x0b, 0x81, 0xf6, 0x7a, 0x5d, 0xee, 0x06, 0xd0, 0xed, 0x20, 0x0e,
      0x80, 0xda, 0xe0, 0x97, 0x7d, 0xda, 0x02, 0xff, 0x86, 0xf2, 0x9f, 0x6a,
      0x30, 0x20, 0xe4, 0x3f, 0x71, 0x1a, 0x69, 0x04, 0x16, 0x60, 0xa6, 0xae,
      0x04, 0xb0, 0x33, 0xe5, 0xbd, 0xdb, 0x7a, 0x39, 0xaf, 0x70, 0xf7, 0x5c,
      0x55, 0xfb, 0x31, 0x56, 0xe7, 0xe4, 0xee, 0xa1, 0x15, 0x87, 0xf2, 0xdd,
      0x58, 0x65, 0x75, 0x09, 0xc1, 0x4d, 0xe9, 0xf4, 0x84, 0xb6, 0x8b, 0x66,
      0xf8, 0xde, 0xf0, 0x10, 0x0f, 0x8d, 0x6d, 0xc3, 0x7d, 0x82, 0x65, 0x3e,
      0x34, 0x99, 0xb6, 0x62, 0xe9, 0x71, 0x94, 0x5e, 0x6a, 0x55, 0x0f, 0x9e,
      0xc7, 0x7f, 0x53, 0xb3, 0x5c, 0x3b, 0x26, 0x83, 0xc1, 0xbc, 0x50, 0xc8,
      0xac, 0xbc, 0xd0, 0x40, 0xf9, 0x3c, 0x41, 0x28, 0xa0, 0xeb, 0xef, 0xb3,
      0x58, 0xe4, 0x57, 0xaa, 0xf8, 0x5d, 0x97, 0x1f, 0x29, 0xd5, 0x69, 0xcc,
      0x2d, 0xd3, 0xcf, 0xf9, 0x72, 0x8f, 0xae, 0xe3, 0x41, 0x93, 0x3f, 0x34,
      0x3e, 0x36, 0x8c, 0xdc, 0xd4, 0x37, 0x55, 0x8b, 0x81, 0xaa, 0xff, 0xa7,
      0x0b, 0xaf, 0xb3, 0x76, 0xe3, 0x12, 0xb4, 0x24, 0xcf, 0xfb, 0x42, 0x43,
      0x2e, 0xb7, 0x48, 0x01, 0xcf, 0x24, 0x03, 0x84, 0x74, 0x0a, 0x7a, 0xa2,
      0xaa, 0x3f, 0x38, 0xc0, 0x77, 0xa2, 0xb3, 0x60, 0x30, 0x00, 0x77, 0xdc,
      0x10, 0x5a, 0xd8, 0xc1, 0x82, 0xde, 0xb7, 0x48, 0x79, 0xa5, 0x0c, 0xd0,
      0x6c, 0x52, 0xde, 0x5d, 0xf0, 0x4a, 0x58, 0x3b, 0x8e, 0xde, 0xf1, 0xc0,
      0x42, 0x63, 0x1e, 0x4d, 0xcf, 0x26, 0x44, 0x9e, 0x50, 0x98, 0x03, 0xbc,
      0x5b, 0xfc, 0xef, 0x07, 0x3d, 0xae, 0xf7, 0xda, 0x9d, 0x76, 0x8a, 0x8d,
      0xa8, 0xb4, 0xe9, 0x79
  };
  const uint8_t kDH_private_key[28] = {
      0x75, 0x89, 0x8a, 0xbe, 0xc3, 0xc9, 0xc8, 0x7b, 0x04, 0x49, 0x47, 0xf6,
      0xc5, 0x1f, 0x9f, 0x71, 0x7f, 0x4a, 0x1d, 0x7c, 0xc3, 0x9a, 0xae, 0xcd,
      0x83, 0x53, 0xba, 0x25
  };
  const uint8_t kDH_public_key[256] = {
      0x80, 0xbc, 0xbe, 0xf0, 0xad, 0x46, 0xfe, 0x97, 0x79, 0x4b, 0xd1, 0x49,
      0x00, 0x04, 0xf0, 0x7f, 0x32, 0xac, 0x56, 0x17, 0x6b, 0xea, 0x84, 0xb5,
      0xdc, 0xc7, 0x2f, 0xec, 0x5e, 0x87, 0xd2, 0xd6, 0xa6, 0x9a, 0xbd, 0x73,
      0x12, 0x8b, 0x6a, 0x8a, 0x42, 0xe9, 0x21, 0x74, 0x28, 0xda, 0x64, 0xb2,
      0x8b, 0x05, 0x04, 0x80, 0x02, 0xcf, 0x8a, 0xab, 0xa0, 0xfe, 0x9c, 0xfb,
      0xfd, 0x3e, 0xb9, 0xa5, 0xfe, 0x08, 0xa5, 0xf3, 0x97, 0xdd, 0x28, 0x38,
      0x33, 0x41, 0x0d, 0x84, 0x1f, 0x46, 0xd3, 0x03, 0xfb, 0x6e, 0xac, 0x2f,
      0x17, 0x97, 0x5f, 0xad, 0x73, 0x23, 0xd2, 0xb3, 0x71, 0x26, 0xdb, 0x0e,
      0x97, 0xa4, 0x15, 0x36, 0x2c, 0x61, 0xc1, 0x21, 0xa6, 0xd1, 0x4d, 0xc6,
      0x54, 0xb4, 0xad, 0x5e, 0x9c, 0x0f, 0xe8, 0x00, 0xd5, 0x4a, 0x62, 0x32,
      0xfe, 0x62, 0x9e, 0x0b, 0x7c, 0xc8, 0x54, 0x8d, 0x83, 0xca, 0x36, 0x4d,
      0x0d, 0x5a, 0xbf, 0xe1, 0x5d, 0x5f, 0xfc, 0xc8, 0x3c, 0xc0, 0xec, 0xf4,
      0x35, 0x62, 0xd4, 0x16, 0xc3, 0x18, 0x0b, 0xf0, 0x41, 0x52, 0x9e, 0x57,
      0xa7, 0xf9, 0xfd, 0x14, 0xfc, 0x1b, 0x5c, 0xa5, 0x85, 0xf3, 0x05, 0xb6,
      0x6c, 0xde, 0xa0, 0x58, 0x49, 0xe0, 0xcc, 0x53, 0x25, 0x9f, 0xbd, 0xfb,
      0xab, 0x75, 0x83, 0x18, 0xc4, 0x9d, 0x2a, 0x95, 0xb0, 0x53, 0xc4, 0x7c,
      0xdc, 0x91, 0x55, 0x01, 0x93, 0xe1, 0x44, 0xd9, 0x64, 0x6c, 0xd9, 0xa5,
      0x70, 0xe9, 0x5b, 0x31, 0x83, 0x93, 0x1d, 0x79, 0x15, 0xfa, 0x8e, 0x84,
      0x4f, 0x04, 0x17, 0x8a, 0x12, 0x69, 0x83, 0x9c, 0xd6, 0x8b, 0x78, 0x58,
      0xfa, 0x2c, 0x6b, 0xeb, 0xe8, 0x47, 0xf8, 0x14, 0x0e, 0x33, 0x7a, 0x95,
      0xce, 0x34, 0x0f, 0x68, 0x32, 0x44, 0x76, 0xf6, 0xe8, 0x2e, 0x89, 0x72,
      0x11, 0x49, 0x04, 0x12
  };
  return self_test_ffc_dh_key(kDH_p, sizeof(kDH_p),
                              kDH_q, sizeof(kDH_q),
                              kDH_g, sizeof(kDH_g),
                              kDH_private_key, sizeof(kDH_private_key),
                              kDH_public_key, sizeof(kDH_public_key));
}

static EC_KEY *self_test_ecdsa_key(void) {
  static const uint8_t kQx[] = {
      0xc8, 0x15, 0x61, 0xec, 0xf2, 0xe5, 0x4e, 0xde, 0xfe, 0x66, 0x17,
      0xdb, 0x1c, 0x7a, 0x34, 0xa7, 0x07, 0x44, 0xdd, 0xb2, 0x61, 0xf2,
      0x69, 0xb8, 0x3d, 0xac, 0xfc, 0xd2, 0xad, 0xe5, 0xa6, 0x81,
  };
  static const uint8_t kQy[] = {
      0xe0, 0xe2, 0xaf, 0xa3, 0xf9, 0xb6, 0xab, 0xe4, 0xc6, 0x98, 0xef,
      0x64, 0x95, 0xf1, 0xbe, 0x49, 0xa3, 0x19, 0x6c, 0x50, 0x56, 0xac,
      0xb3, 0x76, 0x3f, 0xe4, 0x50, 0x7e, 0xec, 0x59, 0x6e, 0x88,
  };
  static const uint8_t kD[] = {
      0xc6, 0xc1, 0xaa, 0xda, 0x15, 0xb0, 0x76, 0x61, 0xf8, 0x14, 0x2c,
      0x6c, 0xaf, 0x0f, 0xdb, 0x24, 0x1a, 0xff, 0x2e, 0xfe, 0x46, 0xc0,
      0x93, 0x8b, 0x74, 0xf2, 0xbc, 0xc5, 0x30, 0x52, 0xb0, 0x77,
  };

  EC_KEY *ec_key = EC_KEY_new();
  BIGNUM *qx = BN_bin2bn(kQx, sizeof(kQx), NULL);
  BIGNUM *qy = BN_bin2bn(kQy, sizeof(kQy), NULL);
  BIGNUM *d = BN_bin2bn(kD, sizeof(kD), NULL);
  if (ec_key == NULL || qx == NULL || qy == NULL || d == NULL ||
      !EC_KEY_set_group(ec_key, EC_group_p256()) ||
      !EC_KEY_set_public_key_affine_coordinates(ec_key, qx, qy) ||
      !EC_KEY_set_private_key(ec_key, d)) {
    EC_KEY_free(ec_key);
    ec_key = NULL;
  }

  BN_free(qx);
  BN_free(qy);
  BN_free(d);
  return ec_key;
}

static DH *self_test_dh(void) {
  DH *dh = DH_get_rfc7919_2048();
  if (!dh) {
    return NULL;
  }

  BIGNUM *priv = BN_new();
  if (!priv) {
    goto err;
  }

  // kFFDHE2048PrivateKeyData is a 225-bit value. (225 because that's the
  // minimum private key size in
  // https://tools.ietf.org/html/rfc7919#appendix-A.1.)
  static const BN_ULONG kFFDHE2048PrivateKeyData[] = {
      TOBN(0x187be36b, 0xd38a4fa1),
      TOBN(0x0a152f39, 0x6458f3b8),
      TOBN(0x0570187e, 0xc422eeb7),
      TOBN(0x00000001, 0x91173f2a),
  };

  bn_set_static_words(priv, kFFDHE2048PrivateKeyData,
                      OPENSSL_ARRAY_SIZE(kFFDHE2048PrivateKeyData));

  if (!DH_set0_key(dh, NULL, priv)) {
    goto err;
  }
  return dh;

err:
  BN_free(priv);
  DH_free(dh);
  return NULL;
}


// Lazy self-tests
//
// Self tests that are slow are deferred until the corresponding algorithm is
// actually exercised, in FIPS mode. (In non-FIPS mode these tests are only run
// when requested by |BORINGSSL_self_test|.)

static OPENSSL_NOINLINE int boringssl_self_test_rsa(void) {
  int ret = 0;
  uint8_t output[256];

  RSA *const rsa_key = self_test_rsa_key();
  if (rsa_key == NULL) {
    fprintf(stderr, "RSA key construction failed\n");
    goto err;
  }
  // Disable blinding for the power-on tests because it's not needed and
  // triggers an entropy draw.
  rsa_key->flags |= RSA_FLAG_NO_BLINDING;

  // RSA Sign KAT

  static const uint8_t kRSASignPlaintext[32] = {
      0xd2, 0xb5, 0x6e, 0x53, 0x30, 0x6f, 0x72, 0x0d, 0x79, 0x29, 0xd8,
      0x70, 0x8b, 0xf4, 0x6f, 0x1c, 0x22, 0x30, 0x03, 0x05, 0x58, 0x2b,
      0x11, 0x5b, 0xed, 0xca, 0xc7, 0x22, 0xd8, 0xaa, 0x5a, 0xb2,
  };
  static const uint8_t kRSASignSignature[256] = {
      0xb7, 0xee, 0x25, 0x11, 0x66, 0xd4, 0xfd, 0x87, 0x10, 0x8f, 0x7c, 0x85,
      0x9a, 0x1d, 0x35, 0x42, 0x2b, 0x02, 0x30, 0xaa, 0x8a, 0x81, 0xf7, 0x50,
      0xb2, 0x64, 0xc0, 0x15, 0xe6, 0xbd, 0x5d, 0xe1, 0x03, 0x8b, 0xf4, 0xd3,
      0x5b, 0x4a, 0x8c, 0xf1, 0x7f, 0x3f, 0xcd, 0x1f, 0xe0, 0x90, 0x78, 0x3a,
      0xde, 0xef, 0xd2, 0x69, 0xe7, 0xe5, 0x59, 0xba, 0xc9, 0xde, 0x53, 0x74,
      0x84, 0xaa, 0x33, 0x06, 0x29, 0x47, 0xd6, 0x0f, 0x00, 0xca, 0xe2, 0xde,
      0xab, 0x6c, 0x86, 0x9d, 0xb2, 0xc6, 0x4f, 0xb6, 0x8e, 0xde, 0xbe, 0x1c,
      0x26, 0x46, 0x2f, 0xbb, 0x05, 0xc1, 0x2b, 0x79, 0xb0, 0x82, 0x4e, 0xc9,
      0x03, 0xb3, 0x1a, 0x27, 0xd0, 0xc8, 0x52, 0x2e, 0x22, 0x04, 0x70, 0x50,
      0x1d, 0xcf, 0xd6, 0x6d, 0x7a, 0xa0, 0x88, 0x90, 0x3a, 0x49, 0x64, 0x65,
      0x72, 0x88, 0x51, 0xe6, 0xdb, 0x8e, 0x8c, 0xbe, 0x1c, 0x50, 0xc7, 0x1a,
      0xc0, 0x8b, 0x44, 0x3d, 0x84, 0x46, 0xb0, 0xa2, 0x7b, 0x37, 0xc8, 0x03,
      0xf0, 0x01, 0x2e, 0x76, 0xc1, 0x0b, 0x7b, 0x0a, 0x5e, 0x62, 0xd9, 0x93,
      0x4b, 0x86, 0xaf, 0xa8, 0x44, 0x20, 0x3f, 0xca, 0xe7, 0x6b, 0x8d, 0x27,
      0x56, 0xf6, 0x03, 0xc1, 0xd5, 0x5e, 0xaa, 0x64, 0x34, 0xa4, 0xf6, 0xf2,
      0x0b, 0x86, 0xc1, 0xd4, 0xa4, 0xa9, 0x6e, 0x58, 0x8c, 0x0a, 0x09, 0xca,
      0xad, 0x4b, 0xd6, 0x6b, 0x3b, 0xfe, 0xfe, 0xfb, 0x09, 0x3c, 0x76, 0xdb,
      0x33, 0x2d, 0x62, 0x28, 0x44, 0xc4, 0xd5, 0x43, 0xe4, 0x86, 0x2e, 0xa8,
      0xa0, 0x26, 0xa8, 0x8c, 0xe9, 0x40, 0x7c, 0xf8, 0x59, 0x84, 0x07, 0xbd,
      0x90, 0xe0, 0xc7, 0xcb, 0x76, 0x8b, 0x41, 0x0f, 0xba, 0xed, 0xc9, 0xbf,
      0x14, 0x77, 0xc5, 0xdb, 0x16, 0xd1, 0xdb, 0x33, 0xbc, 0x20, 0xd0, 0xec,
      0x04, 0xbf, 0xf8, 0xe3,
  };

  unsigned sig_len;
  if (!rsa_digestsign_no_self_test(EVP_sha256(), kRSASignPlaintext,
                         sizeof(kRSASignPlaintext),output, &sig_len, rsa_key) ||
      !check_test(kRSASignSignature, output, sizeof(kRSASignSignature),
                  "RSA-sign KAT")) {
    goto err;
  }

  // RSA Verify KAT

  static const uint8_t kRSAVerifyPlaintext[32] = {
      0x09, 0x65, 0x2f, 0xd8, 0xed, 0x9d, 0xc2, 0x6d, 0xbc, 0xbf, 0xf2,
      0xa7, 0xa5, 0xed, 0xe1, 0x37, 0x13, 0x78, 0x21, 0x36, 0xcf, 0x8d,
      0x22, 0x3d, 0xab, 0x93, 0xb4, 0x12, 0xa8, 0xb5, 0x15, 0x53,
  };
  static const uint8_t kRSAVerifySignature[256] = {
      0xc5, 0xef, 0x03, 0x0d, 0x00, 0xa1, 0x3e, 0x3a, 0x70, 0x5b, 0x23, 0xe1,
      0xe3, 0xde, 0x3f, 0x2c, 0x8e, 0x84, 0xb2, 0xe8, 0x2d, 0x1b, 0xec, 0x14,
      0x11, 0x6f, 0x82, 0x45, 0xe5, 0xb6, 0xfa, 0x4b, 0x20, 0x7f, 0x12, 0xaf,
      0xe7, 0x2c, 0x8d, 0x36, 0x67, 0x5a, 0xcb, 0x7d, 0x67, 0x0f, 0x6a, 0x5c,
      0x59, 0x0e, 0x44, 0x71, 0x6c, 0x3d, 0xf3, 0x11, 0x04, 0xbe, 0xa8, 0x9f,
      0x61, 0xbe, 0xcd, 0x6c, 0xc1, 0x88, 0x81, 0x48, 0x01, 0xd3, 0x08, 0xce,
      0xec, 0x2a, 0x84, 0x3e, 0xc7, 0xf2, 0x5e, 0xbc, 0xdd, 0xe5, 0x88, 0xdd,
      0x89, 0x80, 0x32, 0x69, 0x28, 0xb1, 0x08, 0x43, 0xc4, 0xb3, 0x19, 0x03,
      0x38, 0xb0, 0xa0, 0x7c, 0x5a, 0x94, 0xa5, 0x3f, 0x6d, 0x84, 0xde, 0x79,
      0x47, 0xf3, 0xdb, 0x3d, 0x9f, 0x73, 0x06, 0x10, 0xbf, 0x46, 0x3c, 0x29,
      0x1f, 0xd9, 0x01, 0xab, 0x8a, 0x54, 0xe4, 0x7d, 0xbb, 0x19, 0x6d, 0x8a,
      0xf5, 0x3e, 0xf1, 0x5f, 0xd0, 0x6b, 0x7a, 0xd0, 0xdd, 0xb6, 0x5c, 0x83,
      0xc8, 0xfe, 0xe8, 0xd0, 0xa7, 0x08, 0x33, 0x47, 0x33, 0xcb, 0xe7, 0xb0,
      0x33, 0x22, 0x69, 0x5d, 0x9a, 0x40, 0x6a, 0x68, 0x78, 0x67, 0xb3, 0x82,
      0x94, 0xd8, 0x63, 0x48, 0x05, 0xd4, 0xcb, 0x68, 0x73, 0x73, 0x55, 0xd7,
      0x76, 0x48, 0x4d, 0xf3, 0xcd, 0x8b, 0x8f, 0xeb, 0x51, 0xfd, 0x94, 0x60,
      0x82, 0x18, 0xd8, 0x8e, 0xb2, 0xf2, 0xd0, 0x40, 0x64, 0x31, 0x2a, 0xd0,
      0x41, 0x2e, 0x7a, 0x96, 0xd4, 0x9a, 0x9d, 0x71, 0xd7, 0x66, 0x06, 0xab,
      0x7a, 0x5b, 0xd9, 0x9b, 0xc3, 0x1c, 0x37, 0x59, 0x3c, 0x83, 0x7f, 0x15,
      0xba, 0x86, 0x01, 0x92, 0x6a, 0x1f, 0x69, 0x19, 0xd1, 0x11, 0x0b, 0x55,
      0x90, 0x6e, 0x18, 0xe2, 0x9e, 0x2a, 0x94, 0x04, 0x5a, 0xe9, 0x21, 0x8b,
      0xc6, 0xc8, 0xda, 0x74,
  };
  if (!rsa_digestverify_no_self_test(EVP_sha256(), kRSAVerifyPlaintext,
                               sizeof(kRSAVerifyPlaintext), kRSAVerifySignature,
                               sizeof(kRSAVerifySignature), rsa_key)) {
    AWS_LC_FIPS_failure("RSA-verify KAT failed");
    goto err;
  }

  ret = 1;

err:
  RSA_free(rsa_key);

  return ret;
}

static OPENSSL_NOINLINE int boringssl_self_test_ecc(void) {
  int ret = 0;
  EC_KEY *ec_key = NULL;
  EC_POINT *ec_point_in = NULL;
  EC_POINT *ec_point_out = NULL;
  BIGNUM *ec_scalar = NULL;
  ECDSA_SIG *sig = NULL;

  ec_key = self_test_ecdsa_key();
  if (ec_key == NULL) {
    fprintf(stderr, "ECDSA KeyGen failed\n");
    goto err;
  }

  // ECDSA Sign/Verify KAT

  static const uint8_t kECDSASignPlaintext[32] = {
      0x1e, 0x35, 0x93, 0x0b, 0xe8, 0x60, 0xd0, 0x94, 0x2c, 0xa7, 0xbb,
      0xd6, 0xf6, 0xde, 0xd8, 0x7f, 0x15, 0x7e, 0x4d, 0xe2, 0x4f, 0x81,
      0xed, 0x4b, 0x87, 0x5c, 0x0e, 0x01, 0x8e, 0x89, 0xa8, 0x1f,
  };
  static const uint8_t kECDSASignSig[64] = {
      0x67, 0x80, 0xc5, 0xfc, 0x70, 0x27, 0x5e, 0x2c, 0x70, 0x61, 0xa0,
      0xe7, 0x87, 0x7b, 0xb1, 0x74, 0xde, 0xad, 0xeb, 0x98, 0x87, 0x02,
      0x7f, 0x3f, 0xa8, 0x36, 0x54, 0x15, 0x8b, 0xa7, 0xf5, 0x0c, 0x3a,
      0x82, 0xd9, 0x65, 0xb2, 0xa7, 0x2a, 0xc5, 0xa6, 0x07, 0xbc, 0x56,
      0xab, 0x37, 0x22, 0xc8, 0x42, 0xd9, 0x21, 0xc0, 0x49, 0x36, 0xe9,
      0xe2, 0x5f, 0xd7, 0xb6, 0x00, 0xe7, 0xd8, 0xdc, 0x80,
  };

  // The 'k' value for ECDSA is fixed to avoid an entropy draw.
  uint8_t ecdsa_k[32] = {0};
  ecdsa_k[31] = 42;

  sig = ecdsa_digestsign_no_self_test(EVP_sha256(),kECDSASignPlaintext, sizeof
                      (kECDSASignPlaintext), ec_key, ecdsa_k, sizeof(ecdsa_k));

  uint8_t ecdsa_sign_output[64];
  if (sig == NULL ||
      !serialize_ecdsa_sig(ecdsa_sign_output, sizeof(ecdsa_sign_output), sig) ||
      !check_test(kECDSASignSig, ecdsa_sign_output, sizeof(ecdsa_sign_output),
                  "ECDSA-sign signature")) {
    AWS_LC_FIPS_failure("ECDSA-sign KAT failed");
    goto err;
  }

  static const uint8_t kECDSAVerifyPlaintext[32] = {
      0x78, 0x7c, 0x50, 0x5c, 0x60, 0xc9, 0xe4, 0x13, 0x6c, 0xe4, 0x48,
      0xba, 0x93, 0xff, 0x71, 0xfa, 0x9c, 0x18, 0xf4, 0x17, 0x09, 0x4f,
      0xdf, 0x5a, 0xe2, 0x75, 0xc0, 0xcc, 0xd2, 0x67, 0x97, 0xad,
  };
  static const uint8_t kECDSAVerifySig[64] = {
      0x67, 0x80, 0xc5, 0xfc, 0x70, 0x27, 0x5e, 0x2c, 0x70, 0x61, 0xa0,
      0xe7, 0x87, 0x7b, 0xb1, 0x74, 0xde, 0xad, 0xeb, 0x98, 0x87, 0x02,
      0x7f, 0x3f, 0xa8, 0x36, 0x54, 0x15, 0x8b, 0xa7, 0xf5, 0x0c, 0x3c,
      0x77, 0xd1, 0xb6, 0xe0, 0x9e, 0x74, 0x7b, 0xc5, 0xab, 0x55, 0x01,
      0xd7, 0x5e, 0x61, 0x8d, 0x8e, 0x5b, 0x27, 0x2e, 0x15, 0x9f, 0xf3,
      0x41, 0x3c, 0xb7, 0x1a, 0x81, 0x40, 0x8d, 0x56, 0x05,
  };

  ECDSA_SIG_free(sig);
  sig = parse_ecdsa_sig(kECDSAVerifySig, sizeof(kECDSAVerifySig));
  if (!sig ||
      !ecdsa_digestverify_no_self_test(EVP_sha256(), kECDSAVerifyPlaintext,
                                  sizeof(kECDSAVerifyPlaintext), sig, ec_key)) {
    AWS_LC_FIPS_failure("ECDSA-verify KAT failed");
    goto err;
  }

  // Primitive Z Computation KAT (IG 9.6).

  // kP256Point is SHA256("Primitive Z Computation KAT")×G within P-256.
  static const uint8_t kP256Point[65] = {
      0x04, 0x4e, 0xc1, 0x94, 0x8c, 0x5c, 0xf4, 0x37, 0x35, 0x0d, 0xa3,
      0xf9, 0x55, 0xf9, 0x8b, 0x26, 0x23, 0x5c, 0x43, 0xe0, 0x83, 0x51,
      0x2b, 0x0d, 0x4b, 0x56, 0x24, 0xc3, 0xe4, 0xa5, 0xa8, 0xe2, 0xe9,
      0x95, 0xf2, 0xc4, 0xb9, 0xb7, 0x48, 0x7d, 0x2a, 0xae, 0xc5, 0xc0,
      0x0a, 0xcc, 0x1b, 0xd0, 0xec, 0xb8, 0xdc, 0xbe, 0x0c, 0xbe, 0x52,
      0x79, 0x93, 0x7c, 0x0b, 0x92, 0x2b, 0x7f, 0x17, 0xa5, 0x80,
  };
  // kP256Scalar is SHA256("Primitive Z Computation KAT scalar").
  static const uint8_t kP256Scalar[32] = {
      0xe7, 0x60, 0x44, 0x91, 0x26, 0x9a, 0xfb, 0x5b, 0x10, 0x2d, 0x6e,
      0xa5, 0x2c, 0xb5, 0x9f, 0xeb, 0x70, 0xae, 0xde, 0x6c, 0xe3, 0xbf,
      0xb3, 0xe0, 0x10, 0x54, 0x85, 0xab, 0xd8, 0x61, 0xd7, 0x7b,
  };
  // kP256PointResult is |kP256Scalar|×|kP256Point|.
  static const uint8_t kP256PointResult[65] = {
      0x04, 0xf1, 0x63, 0x00, 0x88, 0xc5, 0xd5, 0xe9, 0x05, 0x52, 0xac,
      0xb6, 0xec, 0x68, 0x76, 0xb8, 0x73, 0x7f, 0x0f, 0x72, 0x34, 0xe6,
      0xbb, 0x30, 0x32, 0x22, 0x37, 0xb6, 0x2a, 0x80, 0xe8, 0x9e, 0x6e,
      0x6f, 0x36, 0x02, 0xe7, 0x21, 0xd2, 0x31, 0xdb, 0x94, 0x63, 0xb7,
      0xd8, 0x19, 0x0e, 0xc2, 0xc0, 0xa7, 0x2f, 0x15, 0x49, 0x1a, 0xa2,
      0x7c, 0x41, 0x8f, 0xaf, 0x9c, 0x40, 0xaf, 0x2e, 0x4a, 0x0c,
  };

  const EC_GROUP *ec_group = EC_group_p256();
  ec_point_in = EC_POINT_new(ec_group);
  ec_point_out = EC_POINT_new(ec_group);
  ec_scalar = BN_new();
  uint8_t z_comp_result[65] = {0};
  if (ec_point_in == NULL || ec_point_out == NULL || ec_scalar == NULL ||
      !EC_POINT_oct2point(ec_group, ec_point_in, kP256Point, sizeof(kP256Point),
                          NULL) ||
      !BN_bin2bn(kP256Scalar, sizeof(kP256Scalar), ec_scalar) ||
      !ec_point_mul_no_self_test(ec_group, ec_point_out, NULL, ec_point_in,
                                 ec_scalar, NULL) ||
      !EC_POINT_point2oct(ec_group, ec_point_out, POINT_CONVERSION_UNCOMPRESSED,
                          z_comp_result, sizeof(z_comp_result), NULL) ||
      !check_test(kP256PointResult, z_comp_result, sizeof(z_comp_result),
                  "Z-computation")) {
    goto err;
  }

  ret = 1;

err:
  EC_KEY_free(ec_key);
  EC_POINT_free(ec_point_in);
  EC_POINT_free(ec_point_out);
  BN_free(ec_scalar);
  ECDSA_SIG_free(sig);

  return ret;
}

static OPENSSL_NOINLINE int boringssl_self_test_ffdh(void) {
  int ret = 0;
  DH *dh = NULL;
  DH *fb_dh = NULL;
  BIGNUM *ffdhe2048_value = NULL;
  BIGNUM *fb_peers_key = NULL;

  // FFC Diffie-Hellman KAT

  // kFFDHE2048PublicValueData is an arbitrary public value, mod
  // kFFDHE2048Data. (The private key happens to be 4096.)
  static const BN_ULONG kFFDHE2048PublicValueData[] = {
      TOBN(0x187be36b, 0xd38a4fa1), TOBN(0x0a152f39, 0x6458f3b8),
      TOBN(0x0570187e, 0xc422eeb7), TOBN(0x18af7482, 0x91173f2a),
      TOBN(0xe9fdac6a, 0xcff4eaaa), TOBN(0xf6afebb7, 0x6e589d6c),
      TOBN(0xf92f8e9a, 0xb7e33fb0), TOBN(0x70acf2aa, 0x4cf36ddd),
      TOBN(0x561ab426, 0xd07137fd), TOBN(0x5f57d037, 0x430ee91e),
      TOBN(0xe3e768c8, 0x60d10b8a), TOBN(0xb14884d8, 0xa18af8ce),
      TOBN(0xf8a98014, 0xa12b74e4), TOBN(0x748d407c, 0x3437b7a8),
      TOBN(0x627588c4, 0x9875d5a7), TOBN(0xdd24a127, 0x53c8f09d),
      TOBN(0x85a997d5, 0x0cd51aec), TOBN(0x44f0c619, 0xce348458),
      TOBN(0x9b894b24, 0x5f6b69a1), TOBN(0xae1302f2, 0xf6d4777e),
      TOBN(0xe6678eeb, 0x375db18e), TOBN(0x2674e1d6, 0x4fbcbdc8),
      TOBN(0xb297a823, 0x6fa93d28), TOBN(0x6a12fb70, 0x7c8c0510),
      TOBN(0x5c6d1aeb, 0xdb06f65b), TOBN(0xe8c2954e, 0x4c1804ca),
      TOBN(0x06bdeac1, 0xf5500fa7), TOBN(0x6a315604, 0x189cd76b),
      TOBN(0xbae7b0b3, 0x6e362dc0), TOBN(0xa57c73bd, 0xdc70fb82),
      TOBN(0xfaff50d2, 0x9d573457), TOBN(0x352bd399, 0xbe84058e),
  };
  static const uint8_t kDHOutput[2048 / 8] = {
      0x2a, 0xe6, 0xd3, 0xa6, 0x13, 0x58, 0x8e, 0xce, 0x53, 0xaa, 0xf6, 0x5d,
      0x9a, 0xae, 0x02, 0x12, 0xf5, 0x80, 0x3d, 0x06, 0x09, 0x76, 0xac, 0x57,
      0x37, 0x9e, 0xab, 0x38, 0x62, 0x25, 0x05, 0x1d, 0xf3, 0xa9, 0x39, 0x60,
      0xf6, 0xae, 0x90, 0xed, 0x1e, 0xad, 0x6e, 0xe9, 0xe3, 0xba, 0x27, 0xf6,
      0xdb, 0x54, 0xdf, 0xe2, 0xbd, 0xbb, 0x7f, 0xf1, 0x81, 0xac, 0x1a, 0xfa,
      0xdb, 0x87, 0x07, 0x98, 0x76, 0x90, 0x21, 0xf2, 0xae, 0xda, 0x0d, 0x84,
      0x97, 0x64, 0x0b, 0xbf, 0xb8, 0x8d, 0x10, 0x46, 0xe2, 0xd5, 0xca, 0x1b,
      0xbb, 0xe5, 0x37, 0xb2, 0x3b, 0x35, 0xd3, 0x1b, 0x65, 0xea, 0xae, 0xf2,
      0x03, 0xe2, 0xb6, 0xde, 0x22, 0xb7, 0x86, 0x49, 0x79, 0xfe, 0xd7, 0x16,
      0xf7, 0xdc, 0x9c, 0x59, 0xf5, 0xb7, 0x70, 0xc0, 0x53, 0x42, 0x6f, 0xb1,
      0xd2, 0x4e, 0x00, 0x25, 0x4b, 0x2d, 0x5a, 0x9b, 0xd0, 0xe9, 0x27, 0x43,
      0xcc, 0x00, 0x66, 0xea, 0x94, 0x7a, 0x0b, 0xb9, 0x89, 0x0c, 0x5e, 0x94,
      0xb8, 0x3a, 0x78, 0x9c, 0x4d, 0x84, 0xe6, 0x32, 0x2c, 0x38, 0x7c, 0xf7,
      0x43, 0x9c, 0xd8, 0xb8, 0x1c, 0xce, 0x24, 0x91, 0x20, 0x67, 0x7a, 0x54,
      0x1f, 0x7e, 0x86, 0x7f, 0xa1, 0xc1, 0x03, 0x4e, 0x2c, 0x26, 0x71, 0xb2,
      0x06, 0x30, 0xb3, 0x6c, 0x15, 0xcc, 0xac, 0x25, 0xe5, 0x37, 0x3f, 0x24,
      0x8f, 0x2a, 0x89, 0x5e, 0x3d, 0x43, 0x94, 0xc9, 0x36, 0xae, 0x40, 0x00,
      0x6a, 0x0d, 0xb0, 0x6e, 0x8b, 0x2e, 0x70, 0x57, 0xe1, 0x88, 0x53, 0xd6,
      0x06, 0x80, 0x2a, 0x4e, 0x5a, 0xf0, 0x1e, 0xaa, 0xcb, 0xab, 0x06, 0x0e,
      0x27, 0x0f, 0xd9, 0x88, 0xd9, 0x01, 0xe3, 0x07, 0xeb, 0xdf, 0xc3, 0x12,
      0xe3, 0x40, 0x88, 0x7b, 0x5f, 0x59, 0x78, 0x6e, 0x26, 0x20, 0xc3, 0xdf,
      0xc8, 0xe4, 0x5e, 0xb8,
  };

  // |kDH_fb_peer_public| and |kDH_fb_z| is fetched from ACVP data.
  // domainParameterGenerationMode is FB.
  // Details are available in
  // CryptoAlg-851?selectedConversation=c0120d22-f2bd-4eae-8ae6-b04fcca86743
  // File name: 197488/683891/testvector-request.json
  // Other details: "tgId": 2, "tcId": 6
  const uint8_t kDH_fb_peer_public[256] = {
    0x8f, 0xbc, 0x50, 0x66, 0x4b, 0x2c, 0x9e, 0x2e, 0x7d, 0x4c, 0x64, 0x1a,
    0xe2, 0xd4, 0xd2, 0xcc, 0x6a, 0xcf, 0xe6, 0xbd, 0xf3, 0x3d, 0x39, 0xf2,
    0x1d, 0xe4, 0xc3, 0x45, 0xb4, 0x51, 0x7a, 0xbd, 0x9e, 0x7d, 0x49, 0xf2,
    0xbd, 0x03, 0x4d, 0x54, 0xf3, 0x97, 0x84, 0xfe, 0x07, 0x31, 0x98, 0x0e,
    0x78, 0x5f, 0xe8, 0x5d, 0xf4, 0x6a, 0xf4, 0xf9, 0xef, 0x25, 0x6b, 0x3e,
    0x1a, 0xb2, 0x0a, 0x42, 0xec, 0x19, 0xad, 0xe9, 0x68, 0xa9, 0x8f, 0xfd,
    0x51, 0xd4, 0x95, 0x88, 0x09, 0x83, 0x28, 0xc8, 0xd6, 0x54, 0x05, 0xd1,
    0xc3, 0x75, 0xb2, 0xbf, 0x03, 0xdd, 0x5f, 0x01, 0x18, 0x6b, 0xbb, 0x8d,
    0x49, 0x75, 0x2d, 0x0d, 0xdf, 0x62, 0x0b, 0xbf, 0x70, 0xbc, 0x58, 0x25,
    0xdb, 0x37, 0xde, 0xb2, 0xea, 0xd5, 0x11, 0x57, 0xc1, 0x83, 0x26, 0x53,
    0x5d, 0x61, 0x42, 0xf9, 0xbf, 0x51, 0xf8, 0x38, 0x93, 0x7f, 0x2d, 0xdd,
    0x5f, 0x57, 0xab, 0x41, 0xf2, 0xda, 0x88, 0xe8, 0x9d, 0x0d, 0xca, 0x5d,
    0x54, 0xe6, 0x79, 0xdf, 0xe2, 0x63, 0x8a, 0x62, 0x9f, 0x48, 0x1c, 0xc3,
    0x09, 0x80, 0x32, 0x46, 0x9c, 0x76, 0xe1, 0xf3, 0xa5, 0xa8, 0x4d, 0xac,
    0xb0, 0x2e, 0x42, 0x3c, 0x1d, 0x68, 0xf2, 0x88, 0xad, 0xd7, 0x3e, 0xa7,
    0xac, 0x4c, 0x13, 0x91, 0xc1, 0x43, 0xce, 0xa5, 0x20, 0x38, 0x7d, 0x8c,
    0x05, 0x2c, 0x96, 0xd2, 0xd6, 0x2a, 0x75, 0xc1, 0xf0, 0x15, 0xa1, 0x5c,
    0xed, 0x80, 0xf9, 0x2e, 0x47, 0x11, 0x2c, 0x15, 0x6d, 0x97, 0x6f, 0x7a,
    0x2e, 0x73, 0xf7, 0x1f, 0xc8, 0x89, 0xd9, 0x34, 0x62, 0x8a, 0xdc, 0xae,
    0xe2, 0xdf, 0xda, 0x03, 0x6d, 0xce, 0x8a, 0xe2, 0x02, 0xf3, 0xd9, 0xb7,
    0x86, 0xf5, 0x3d, 0x5f, 0x28, 0xeb, 0x74, 0x81, 0xd0, 0x40, 0x6e, 0xa6,
    0x90, 0x1f, 0x97, 0xbe,
  };
  const uint8_t kDH_fb_z[256] = {
    0x8a, 0x03, 0x41, 0x31, 0x59, 0xda, 0x27, 0xff, 0x91, 0x0b, 0xd9, 0x46,
    0x08, 0x8f, 0x08, 0x67, 0x89, 0xa2, 0x0a, 0xac, 0x32, 0x9e, 0x8f, 0x05,
    0xde, 0x0d, 0x4f, 0xb8, 0x35, 0xf8, 0x5f, 0x56, 0x6f, 0x51, 0x75, 0xfd,
    0xa2, 0x50, 0x5e, 0x5e, 0x92, 0x42, 0x89, 0xd1, 0x6c, 0xe2, 0xab, 0x13,
    0x15, 0xd9, 0x72, 0x22, 0x66, 0x2e, 0x64, 0x83, 0x7d, 0x21, 0x51, 0x2b,
    0x8d, 0x79, 0xe5, 0x53, 0x4b, 0x7d, 0xf6, 0x6c, 0x8d, 0x13, 0x8a, 0xcc,
    0x9b, 0xed, 0x8d, 0xe9, 0x25, 0xd7, 0x31, 0x9b, 0x49, 0x0b, 0xc2, 0x5e,
    0x5e, 0xa7, 0x48, 0xb8, 0xf9, 0x66, 0xbc, 0x4e, 0x1e, 0x5b, 0xfe, 0x08,
    0x1f, 0x5f, 0x29, 0xdf, 0xfa, 0x27, 0x08, 0xad, 0x40, 0xff, 0x07, 0xd8,
    0xb6, 0xe8, 0x7e, 0x03, 0xc2, 0xe2, 0xdd, 0x29, 0xb1, 0x8d, 0x4b, 0x68,
    0x51, 0x94, 0xb9, 0x72, 0xb2, 0x49, 0x20, 0xa1, 0x80, 0x16, 0x09, 0x20,
    0x83, 0xa6, 0x13, 0x70, 0x0a, 0x42, 0x62, 0x8c, 0xd6, 0x1e, 0x9f, 0x64,
    0x18, 0x41, 0x48, 0x49, 0xe5, 0xcb, 0x3b, 0xd2, 0x86, 0x48, 0x92, 0x83,
    0x69, 0xc9, 0xa9, 0x99, 0xe6, 0xc7, 0xdc, 0x08, 0xee, 0xdc, 0x64, 0x43,
    0x42, 0xb7, 0x49, 0x39, 0x4b, 0x0d, 0x3a, 0xfc, 0x73, 0x63, 0xa7, 0x65,
    0x61, 0x9e, 0x45, 0xfd, 0x72, 0x0f, 0x6c, 0xef, 0x1a, 0x1d, 0xa7, 0xdc,
    0x81, 0xfd, 0x03, 0x62, 0x2a, 0x55, 0xbf, 0x88, 0x09, 0xf7, 0x1e, 0xd9,
    0xbc, 0xdd, 0x62, 0x33, 0xe7, 0xa0, 0xd5, 0xfa, 0x55, 0xca, 0xa0, 0xb8,
    0x47, 0xc7, 0xf4, 0xbc, 0x15, 0x98, 0x7d, 0x63, 0xf4, 0x71, 0xc0, 0x25,
    0x34, 0x96, 0x0f, 0xb5, 0xeb, 0xa9, 0x2e, 0x0c, 0xbf, 0x12, 0x99, 0xc0,
    0xbd, 0x0e, 0x65, 0xa3, 0xad, 0x77, 0x75, 0xc5, 0x99, 0xeb, 0x30, 0xe9,
    0x65, 0x90, 0xbc, 0x7e,
  };

  ffdhe2048_value = BN_new();
  if (ffdhe2048_value) {
    bn_set_static_words(ffdhe2048_value, kFFDHE2048PublicValueData,
                        OPENSSL_ARRAY_SIZE(kFFDHE2048PublicValueData));
  }

  dh = self_test_dh();
  uint8_t dh_out[sizeof(kDHOutput)];
  if (dh == NULL || ffdhe2048_value == NULL || sizeof(dh_out) != DH_size(dh) ||
      dh_compute_key_padded_no_self_test(dh_out, ffdhe2048_value, dh) !=
          sizeof(dh_out) ||
      !check_test(kDHOutput, dh_out, sizeof(dh_out), "FFC DH")) {
    goto err;
  }

  // FFC Diffie-Hellman KAT with FB domain parameters.
  set_bignum(&fb_peers_key, kDH_fb_peer_public, sizeof(kDH_fb_peer_public));
  fb_dh = self_test_ffc_dh_fb_key();
  uint8_t fb_dh_out[sizeof(kDH_fb_z)];
  if (fb_dh == NULL || fb_peers_key == NULL ||
      sizeof(fb_dh_out) != DH_size(fb_dh) ||
      dh_compute_key_padded_no_self_test(fb_dh_out, fb_peers_key, fb_dh) !=
          sizeof(fb_dh_out) ||
      !check_test(kDH_fb_z, fb_dh_out, sizeof(fb_dh_out), "FFC DH FB")) {
    goto err;
  }

  ret = 1;

err:
  DH_free(dh);
  BN_free(ffdhe2048_value);
  DH_free(fb_dh);
  BN_free(fb_peers_key);

  return ret;
}

static OPENSSL_NOINLINE int boringssl_self_test_ml_kem(void) {
  int ret = 0;

  static const uint8_t kKeyGenEKSeed[MLKEM512_KEYGEN_SEED_LEN] = {
      0xf8, 0x8c, 0xb2, 0x5f, 0x89, 0xa3, 0x55, 0x5f, 0xae, 0xc6, 0x71, 0xa1,
      0xdf, 0xc6, 0xf6, 0x1d, 0x60, 0xd0, 0x62, 0x22, 0x7d, 0x6a, 0x8f, 0xf6,
      0x2b, 0x3c, 0x6d, 0x7b, 0xd6, 0x14, 0x0f, 0x66, 0x24, 0xc0, 0x84, 0xa6,
      0x4d, 0xa7, 0x4c, 0x63, 0x32, 0x7e, 0x11, 0x77, 0x58, 0xaa, 0x33, 0x8a,
      0x02, 0xe4, 0x43, 0x74, 0x10, 0xb8, 0xf9, 0xf2, 0x00, 0x88, 0xa1, 0x29,
      0xc1, 0x68, 0x3d, 0xe7
  };
  static const uint8_t kKeyGenEK[MLKEM512_PUBLIC_KEY_BYTES] = {
      0xa7, 0xcc, 0x68, 0xf8, 0xd0, 0x21, 0x10, 0xca, 0x57, 0x20, 0x22, 0x3b,
      0x9e, 0x2a, 0x89, 0x87, 0xc8, 0xa2, 0x48, 0x35, 0xa2, 0x0d, 0xab, 0xcb,
      0xef, 0xa4, 0x30, 0xe7, 0x4a, 0x85, 0xaf, 0x80, 0xb9, 0xb7, 0x4b, 0x74,
      0x57, 0x4c, 0x5e, 0x5f, 0x58, 0x54, 0x59, 0xca, 0x36, 0x10, 0x94, 0x0f,
      0x0b, 0x57, 0xb3, 0x33, 0x44, 0xce, 0xac, 0xcc, 0xc1, 0x35, 0x55, 0x7b,
      0x82, 0xf4, 0x96, 0x86, 0x88, 0xa0, 0x16, 0x8c, 0x1a, 0xa2, 0x94, 0x0e,
      0x56, 0x04, 0x48, 0x2b, 0xf8, 0xb9, 0x00, 0xa4, 0x34, 0x30, 0x96, 0x44,
      0x63, 0x30, 0xcf, 0xee, 0x10, 0x91, 0x7c, 0x03, 0x38, 0x18, 0x1b, 0x7f,
      0xe8, 0xd3, 0x0f, 0x48, 0x16, 0x08, 0x7d, 0x62, 0x99, 0xf2, 0x25, 0x41,
      0x7f, 0x53, 0x3e, 0xe4, 0x04, 0x73, 0x89, 0x48, 0x47, 0xbd, 0x45, 0x29,
      0x13, 0x67, 0xbe, 0x6b, 0x1a, 0x7d, 0xee, 0x55, 0xbb, 0x21, 0xd6, 0x0e,
      0x38, 0x28, 0x55, 0x2f, 0x4c, 0x8a, 0x6f, 0x3c, 0x54, 0xfc, 0x74, 0xcf,
      0x67, 0xa6, 0x14, 0xee, 0xab, 0x00, 0x2a, 0x07, 0x68, 0x51, 0x87, 0x9c,
      0xef, 0x22, 0x18, 0xfd, 0xb3, 0x76, 0x61, 0x23, 0xc2, 0xde, 0x32, 0xa2,
      0x69, 0xb6, 0xaa, 0x46, 0x61, 0xb6, 0x93, 0x70, 0x31, 0x4c, 0x00, 0x44,
      0x46, 0xf7, 0x25, 0x8c, 0x40, 0xb2, 0xea, 0x78, 0x9f, 0x40, 0xca, 0x02,
      0x3b, 0xbb, 0x12, 0x17, 0xc2, 0xc4, 0x4b, 0x38, 0x0c, 0x6e, 0x31, 0x94,
      0xed, 0xd1, 0x29, 0xd0, 0x39, 0x21, 0x8d, 0x9b, 0x75, 0x19, 0x4a, 0x38,
      0x6d, 0x94, 0x4a, 0xcc, 0xe7, 0xa9, 0x72, 0x0a, 0xb0, 0x26, 0x36, 0x20,
      0x04, 0xe9, 0x5b, 0xf9, 0x22, 0x92, 0x90, 0xb5, 0x36, 0x13, 0x41, 0x60,
      0x82, 0xe8, 0x2b, 0xa8, 0xa4, 0x2a, 0x5f, 0xf1, 0x47, 0x59, 0xf5, 0x77,
      0x12, 0x39, 0x57, 0x06, 0xb3, 0x07, 0xf7, 0x63, 0x5e, 0xce, 0xe4, 0x23,
      0x17, 0xa4, 0x8e, 0xb3, 0xb9, 0x06, 0x83, 0xf3, 0xab, 0x53, 0xd1, 0x7c,
      0x50, 0xf5, 0x3c, 0x6c, 0xfb, 0x1c, 0x0c, 0xf5, 0x9d, 0x6f, 0x2a, 0x98,
      0x10, 0x21, 0x42, 0x8a, 0x7a, 0xc5, 0xed, 0xf1, 0x3b, 0x26, 0x84, 0x4d,
      0x83, 0xc3, 0x1d, 0x36, 0x08, 0x71, 0x0e, 0x62, 0x3a, 0xab, 0xb2, 0x4b,
      0x6c, 0x5b, 0x48, 0xba, 0xeb, 0xc1, 0xb3, 0x07, 0x89, 0x72, 0x58, 0x92,
      0x01, 0xb7, 0xa3, 0x0b, 0xc0, 0x93, 0x15, 0x61, 0x2b, 0x06, 0x76, 0x55,
      0xbe, 0xc4, 0x03, 0xa6, 0x9c, 0x89, 0xeb, 0x13, 0x7c, 0x31, 0x15, 0x79,
      0x71, 0x09, 0x8c, 0xb6, 0x93, 0xba, 0x4a, 0xe9, 0xae, 0x40, 0xa8, 0x03,
      0x1c, 0xec, 0x92, 0x58, 0x0b, 0xcc, 0x1b, 0x5a, 0xb3, 0xec, 0xd1, 0xaa,
      0x5f, 0x79, 0xaa, 0x2c, 0xd6, 0x92, 0x49, 0xd1, 0x38, 0xc8, 0x96, 0x5a,
      0x81, 0xc8, 0x7a, 0x07, 0xeb, 0x59, 0xa4, 0x61, 0x2e, 0x60, 0x65, 0x8f,
      0x4d, 0xf0, 0x28, 0xce, 0xf8, 0xaf, 0x1b, 0x83, 0x7e, 0x0a, 0xb0, 0xbf,
      0xed, 0xb7, 0x26, 0x90, 0x42, 0x90, 0xd0, 0xbc, 0x41, 0xdf, 0x6a, 0x67,
      0xf7, 0xa4, 0x16, 0x66, 0x09, 0x95, 0x24, 0x39, 0x63, 0x19, 0x60, 0x64,
      0x8e, 0x22, 0x9a, 0x21, 0xf2, 0xa4, 0xd8, 0x2a, 0xba, 0xd3, 0xec, 0x81,
      0x35, 0xdc, 0x9b, 0xb4, 0x3c, 0x70, 0x3d, 0x3b, 0x33, 0xe4, 0x37, 0xc9,
      0xef, 0x7b, 0xca, 0x91, 0xc3, 0x46, 0x56, 0x76, 0x74, 0x01, 0x25, 0xa1,
      0x5a, 0xd1, 0x70, 0x70, 0x88, 0xb1, 0x01, 0xb4, 0x27, 0x3d, 0x3c, 0x4b,
      0xf3, 0x01, 0x81, 0xb4, 0xb2, 0x57, 0x5d, 0xe7, 0x5c, 0xcf, 0xc1, 0x33,
      0x12, 0xa2, 0xa6, 0xbc, 0xeb, 0xc4, 0x77, 0xa9, 0x66, 0x8e, 0x75, 0x16,
      0x29, 0xb5, 0x69, 0xbf, 0xa2, 0x0b, 0xec, 0xa0, 0x98, 0x00, 0x99, 0x2c,
      0x63, 0xf0, 0x4b, 0x6b, 0x4a, 0x7d, 0xf9, 0x77, 0xa0, 0x01, 0x31, 0xc2,
      0xf8, 0x72, 0x2e, 0x51, 0x38, 0x77, 0x52, 0x35, 0xb5, 0x17, 0xa7, 0x09,
      0x85, 0x21, 0x67, 0xc1, 0xd4, 0x15, 0xfd, 0xc7, 0xad, 0x32, 0xf2, 0xaa,
      0xca, 0x43, 0x7e, 0x9c, 0xc6, 0xb2, 0x48, 0xd9, 0xca, 0x7c, 0x65, 0xb4,
      0x05, 0xe6, 0x8d, 0x24, 0xe8, 0x1b, 0x86, 0x88, 0xca, 0xa2, 0x2b, 0x3c,
      0xf5, 0xc9, 0xb1, 0x47, 0xf0, 0xcc, 0x27, 0xe6, 0x67, 0xa8, 0x0b, 0x83,
      0xcc, 0xbb, 0x4f, 0x41, 0x61, 0xa5, 0xff, 0xd0, 0x19, 0x4b, 0x1b, 0x57,
      0x20, 0xe6, 0x8e, 0xa0, 0xf5, 0x99, 0x97, 0xf2, 0x67, 0x40, 0x97, 0x2d,
      0x2c, 0x81, 0x24, 0xd7, 0xa6, 0xad, 0x83, 0x27, 0xc2, 0x07, 0x5a, 0x3f,
      0x76, 0xb9, 0x68, 0xd2, 0xaa, 0xad, 0x19, 0xc0, 0x06, 0x97, 0x59, 0x9e,
      0x0b, 0xe4, 0x9f, 0xa6, 0xd6, 0x5b, 0x40, 0x88, 0xb0, 0xbe, 0x69, 0x2d,
      0xcd, 0xa0, 0x28, 0x09, 0x58, 0x52, 0xa0, 0xd7, 0x20, 0x5e, 0x44, 0x17,
      0x40, 0x9c, 0x03, 0x17, 0x78, 0x03, 0x05, 0xa8, 0x78, 0xfb, 0x58, 0x29,
      0x63, 0xb6, 0x95, 0x3f, 0x6b, 0x8f, 0x08, 0x80, 0xb0, 0x50, 0x17, 0x83,
      0x01, 0xd6, 0x59, 0xbe, 0x3a, 0x4d, 0xb7, 0xe0, 0xbf, 0x25, 0x87, 0x12,
      0x91, 0x64, 0x17, 0x8f, 0x32, 0x70, 0x7d, 0x40, 0x39, 0x2d, 0x85, 0x71,
      0x3d, 0xea, 0x82, 0x91, 0x39, 0x99, 0xaa, 0x6c, 0x35, 0xaa, 0x94, 0xb3,
      0x54, 0x7a, 0xbe, 0x40, 0xe2, 0xb0, 0xba, 0x82, 0xb5, 0xb7, 0x83, 0x19,
      0x18, 0x2a, 0x5a, 0x47, 0xd1, 0x73, 0x17, 0x6b, 0xa6, 0xfa, 0x3a, 0x4d,
      0x70, 0xa8, 0x13, 0x0b, 0x31, 0x07, 0x43, 0xfa, 0x8a, 0xaa, 0xe3, 0x14,
      0x38, 0x1c, 0x9b, 0x7f, 0x49, 0x99, 0x1f, 0x19, 0xa4, 0xff, 0x83, 0x69,
      0x63, 0x8d, 0xe3, 0x80, 0xb5, 0xd8, 0x28, 0xb7, 0xb5, 0xfc, 0xfd, 0xd9,
      0x1f, 0xe6, 0x8e, 0xfe, 0x16, 0xd4, 0xcd, 0x9e, 0xed, 0x66, 0xd6, 0x5c,
    0x1d, 0x2d, 0x8c, 0xaf, 0x5a, 0xf4, 0xa6, 0x92};
  size_t public_len = MLKEM512_PUBLIC_KEY_BYTES;
  size_t secret_len = MLKEM512_SECRET_KEY_BYTES;
  uint8_t keygen_decaps[MLKEM512_SECRET_KEY_BYTES] = {0};
  uint8_t keygen_encaps[MLKEM512_PUBLIC_KEY_BYTES] = {0};

  if (ml_kem_512_keypair_deterministic_no_self_test(
          keygen_encaps, &public_len, keygen_decaps, &secret_len, kKeyGenEKSeed) ||
      !check_test(kKeyGenEK, keygen_encaps, sizeof(keygen_encaps),
              "ML-KEM-keyGen-encaps")) {
    goto err;
  }

  // Use a separate seed for the decaps key so break-kat.go doesn't get confused
  // about having two copies of the encaps key due to the way the decaps key is
  // defined: FIPS 203 section 6.1
  // dk = (dkPKE‖ek‖H(ek)‖𝑧)
  // where dkPKE is the secret key, ek is the public key, and z is the hash of
  // the public key
  public_len = MLKEM512_PUBLIC_KEY_BYTES;
  secret_len = MLKEM512_SECRET_KEY_BYTES;
  static const uint8_t kKeyGenDKSeed[MLKEM512_KEYGEN_SEED_LEN] = {
      0xec, 0xdb, 0x97, 0x62, 0xf4, 0x78, 0xb2, 0xfa, 0x26, 0x3d, 0xf4,
      0x6d, 0xe4, 0x47, 0xf3, 0xd1, 0x52, 0xa1, 0xbc, 0x0e, 0x02, 0xee,
      0x95, 0x36, 0x77, 0x30, 0x11, 0x64, 0xd1, 0x5d, 0x20, 0xd7, 0x1b,
      0x07, 0x4b, 0xff, 0x80, 0x44, 0x44, 0x5e, 0x11, 0x66, 0x0b, 0x1b,
      0x6b, 0x26, 0xdf, 0x24, 0x2b, 0x8f, 0xc0, 0x2b, 0x9e, 0x8d, 0xf5,
      0x38, 0xdb, 0x17, 0xa6, 0x39, 0xd7, 0xc4, 0x61, 0x32
  };
  static const uint8_t kKeyGenDK[MLKEM512_SECRET_KEY_BYTES] = {
      0x88, 0xc1, 0x2c, 0xea, 0xa6, 0xcb, 0x91, 0xf5, 0x89, 0xac, 0xb8, 0x6d,
      0x91, 0x3c, 0x7a, 0x60, 0xf7, 0xcd, 0xab, 0xe3, 0xb7, 0xb5, 0x90, 0x09,
      0x1d, 0x00, 0x84, 0xe2, 0x9a, 0x04, 0x9b, 0x43, 0x68, 0x41, 0xf2, 0x47,
      0x3b, 0x03, 0x16, 0x5a, 0xe9, 0xc6, 0xa9, 0x82, 0x6d, 0x6c, 0x65, 0x0d,
      0x04, 0xb3, 0x88, 0xef, 0xf5, 0x94, 0x50, 0x5b, 0x7e, 0x54, 0x70, 0x95,
      0x30, 0x54, 0x68, 0x25, 0xa0, 0x70, 0xa6, 0x25, 0xb0, 0xe5, 0xfa, 0x86,
      0x6e, 0x6a, 0xaf, 0x40, 0xc2, 0x41, 0x42, 0x46, 0x24, 0x09, 0x73, 0xc7,
      0x59, 0x8a, 0xae, 0x7c, 0x36, 0x3e, 0x43, 0x03, 0xab, 0xb7, 0xa1, 0x11,
      0x31, 0xb4, 0x64, 0xa9, 0x43, 0x99, 0x6d, 0xe7, 0x59, 0x2c, 0xa0, 0x49,
      0x22, 0xea, 0x8a, 0x4d, 0x73, 0xb4, 0x43, 0xea, 0x04, 0x8c, 0x06, 0xac,
      0xc4, 0xe5, 0x5a, 0x8f, 0x25, 0x4b, 0xf6, 0xd2, 0x71, 0xfd, 0x82, 0x71,
      0x19, 0xec, 0x5b, 0x55, 0x80, 0x49, 0x8b, 0xfc, 0xc0, 0x9e, 0xb0, 0x26,
      0x6f, 0x8c, 0x2b, 0x45, 0x98, 0x8a, 0xe9, 0x8c, 0x1e, 0x54, 0x02, 0xb7,
      0x00, 0x34, 0x63, 0xf2, 0x03, 0x59, 0x47, 0x01, 0x59, 0xc0, 0x50, 0x9f,
      0xa9, 0x71, 0x15, 0x34, 0x43, 0xce, 0x25, 0x80, 0xc0, 0xb2, 0x44, 0x3f,
      0x8a, 0xc2, 0xb0, 0x81, 0x04, 0x01, 0xe7, 0x30, 0x52, 0xd6, 0x26, 0xbf,
      0x58, 0xc6, 0x74, 0xee, 0x48, 0x88, 0x0c, 0x40, 0x8d, 0x1f, 0x31, 0x3a,
      0x94, 0xb1, 0x66, 0x7f, 0x89, 0x76, 0x28, 0xc5, 0x5a, 0x83, 0xe2, 0x86,
      0x34, 0xa2, 0x07, 0x10, 0xd2, 0x5d, 0xa8, 0x8b, 0x2a, 0xc9, 0x0c, 0x0f,
      0x5d, 0x6b, 0x0e, 0xd6, 0xe0, 0x80, 0xfd, 0x2c, 0x24, 0xbb, 0x11, 0x81,
      0x6b, 0x5c, 0x60, 0x79, 0x57, 0x78, 0x1d, 0xb2, 0x28, 0x79, 0x66, 0x71,
      0x7f, 0xfa, 0x50, 0x0a, 0x03, 0x02, 0x58, 0x39, 0x16, 0x41, 0x15, 0xba,
      0x5e, 0xa7, 0xb7, 0x17, 0x34, 0x45, 0x88, 0x16, 0x9e, 0x8a, 0x85, 0xa9,
      0xe8, 0x3c, 0x51, 0x6c, 0xab, 0x5e, 0xe6, 0xe6, 0x3b, 0x7c, 0x73, 0x6c,
      0xe9, 0x06, 0x28, 0xdd, 0xec, 0xa9, 0x9a, 0x6b, 0xc5, 0x47, 0x58, 0x86,
      0x81, 0xbc, 0x22, 0x23, 0xf3, 0x9f, 0xbc, 0x44, 0x64, 0xb4, 0x1c, 0x63,
      0xd9, 0x24, 0xaf, 0xeb, 0xa1, 0x09, 0x89, 0x60, 0xbd, 0x43, 0xc1, 0x2f,
      0xd0, 0x08, 0x0f, 0x49, 0x0c, 0x8c, 0x50, 0xf4, 0x84, 0x32, 0x22, 0x30,
      0x3c, 0x68, 0x21, 0xad, 0x7b, 0x66, 0x3d, 0x88, 0x44, 0xa1, 0x36, 0x8a,
      0x19, 0x54, 0x9e, 0x5c, 0xfc, 0x83, 0xd1, 0x4a, 0x84, 0x91, 0x20, 0x5c,
      0x44, 0xf5, 0x49, 0x2b, 0xf6, 0x09, 0xb6, 0x9c, 0x52, 0x6c, 0x95, 0xaa,
      0x5e, 0x50, 0x8e, 0xcf, 0x71, 0x4b, 0x79, 0x55, 0xb6, 0x1f, 0x6c, 0xab,
      0x34, 0x0b, 0x06, 0x20, 0x04, 0x32, 0xce, 0x3b, 0x1e, 0xcd, 0x0a, 0x07,
      0x5a, 0x83, 0x32, 0x04, 0x68, 0x65, 0x95, 0x9a, 0x25, 0x69, 0x8a, 0x08,
      0x1d, 0xf3, 0x53, 0x26, 0x91, 0x34, 0x8a, 0x25, 0x81, 0xd6, 0x06, 0xa0,
      0xe0, 0xb2, 0xcf, 0x2e, 0x8b, 0x3c, 0x9a, 0xfc, 0x59, 0x4e, 0x92, 0x6e,
      0xb0, 0xd7, 0xc7, 0x02, 0xc0, 0xc3, 0xe5, 0xb5, 0x4e, 0xe2, 0xc8, 0x6c,
      0xa2, 0x0c, 0xc4, 0x76, 0xaa, 0xac, 0xa3, 0x20, 0x8f, 0x32, 0xa2, 0x01,
      0x90, 0x56, 0x2c, 0x27, 0x33, 0x82, 0x02, 0x07, 0x1e, 0x11, 0x66, 0x5f,
      0x13, 0x42, 0x73, 0xbd, 0xaa, 0xcb, 0xc9, 0x52, 0xb9, 0x94, 0xba, 0x94,
      0x62, 0x67, 0x12, 0x26, 0x51, 0x4b, 0x44, 0x61, 0x13, 0xe7, 0xab, 0x7e,
      0xdb, 0x9c, 0x54, 0xc3, 0x11, 0xc4, 0xda, 0x94, 0x10, 0x4d, 0x26, 0x2a,
      0x80, 0x28, 0x0e, 0x39, 0x20, 0x1e, 0x75, 0x51, 0x91, 0x76, 0x39, 0x83,
      0xc4, 0x39, 0xa9, 0x5a, 0xea, 0xaf, 0xa7, 0x67, 0xc2, 0xcb, 0x59, 0x48,
      0x29, 0xe6, 0x31, 0x3e, 0x38, 0x69, 0x82, 0xd6, 0x62, 0x1a, 0xcc, 0x4b,
      0xb0, 0x99, 0x1a, 0x60, 0x79, 0x0a, 0x2b, 0x0c, 0x5f, 0x31, 0x39, 0xaa,
      0xdd, 0x70, 0x45, 0xf7, 0xd8, 0x49, 0xaa, 0x20, 0x9b, 0xf6, 0x0b, 0xc1,
      0x5e, 0xd8, 0x26, 0x67, 0x41, 0x4b, 0x70, 0xb7, 0x1a, 0x7e, 0x18, 0x4e,
      0x16, 0x42, 0x80, 0xaf, 0x00, 0xbf, 0x95, 0xa9, 0xad, 0x3d, 0xe4, 0x1d,
      0xcf, 0x19, 0x62, 0x36, 0x10, 0xcf, 0xb3, 0x06, 0x87, 0xa5, 0xa0, 0x82,
      0xa2, 0x45, 0x88, 0x70, 0xeb, 0x33, 0xd6, 0x49, 0xb3, 0xfc, 0xe3, 0x31,
      0x7e, 0x03, 0x62, 0xee, 0x61, 0x75, 0xfb, 0x81, 0x1c, 0x7f, 0xe3, 0x64,
      0x7c, 0xa2, 0x10, 0x20, 0x67, 0x3a, 0xed, 0xd2, 0x3b, 0xf0, 0x47, 0x0c,
      0xd8, 0x73, 0x5c, 0xe5, 0x3a, 0x78, 0x08, 0x26, 0x68, 0xe4, 0x9a, 0x51,
      0x32, 0xd1, 0x33, 0x6a, 0x08, 0x43, 0x60, 0x18, 0x7f, 0xcf, 0xf9, 0x22,
      0x7c, 0xdc, 0x0d, 0x7f, 0x20, 0x5b, 0x2a, 0xf2, 0xc8, 0x8d, 0xc6, 0xbc,
      0x9a, 0x04, 0xb4, 0x1f, 0x42, 0x9f, 0xa9, 0xa3, 0x86, 0xf5, 0x8b, 0x8f,
      0x21, 0x6c, 0xbf, 0x67, 0x14, 0xa6, 0x19, 0x59, 0x7c, 0xaa, 0x5a, 0x9b,
      0xf6, 0x68, 0x04, 0x8d, 0x71, 0x35, 0x10, 0x5a, 0x94, 0x25, 0xcb, 0x46,
      0xfe, 0x80, 0x7a, 0x40, 0x64, 0x59, 0xb0, 0x67, 0xb7, 0x16, 0xf5, 0xa4,
      0x77, 0x30, 0x97, 0x4b, 0x20, 0x78, 0x26, 0x71, 0x6f, 0xe3, 0xf7, 0x47,
      0xfd, 0x74, 0xcd, 0x8c, 0x02, 0x57, 0x20, 0xb1, 0x00, 0x3f, 0x27, 0xb2,
      0xde, 0xa6, 0x95, 0xf2, 0x21, 0x97, 0x1c, 0xd5, 0xaf, 0x81, 0xa1, 0xa7,
      0x95, 0x00, 0x1f, 0x12, 0x5c, 0x28, 0x5c, 0x15, 0x0d, 0xdc, 0x99, 0x59,
      0xe9, 0xe0, 0x22, 0x0a, 0x69, 0x45, 0x56, 0xbc, 0x0a, 0x4b, 0x5a, 0xad,
      0xe0, 0x32, 0xb2, 0xaa, 0x87, 0x7c, 0x0d, 0xa8, 0x16, 0xef, 0x32, 0xa8,
      0x4d, 0x77, 0xbb, 0xff, 0x46, 0x56, 0xe6, 0x80, 0x85, 0x4e, 0x69, 0x61,
      0xa4, 0x17, 0x08, 0x2b, 0xd4, 0x6f, 0x50, 0x23, 0xc4, 0x7c, 0xea, 0x09,
      0xbb, 0xa2, 0x14, 0x5d, 0x41, 0x51, 0x88, 0xe1, 0x38, 0x03, 0xbb, 0x5e,
      0x7f, 0xc1, 0x9d, 0x39, 0x81, 0x9e, 0x59, 0xca, 0x7f, 0xdc, 0x38, 0x05,
      0x1d, 0x96, 0x94, 0x01, 0xd4, 0xc4, 0x11, 0xda, 0xc0, 0x0a, 0x47, 0xab,
      0x57, 0x52, 0x7d, 0xaa, 0x89, 0x7a, 0x37, 0x75, 0x93, 0x51, 0xb4, 0xb1,
      0x18, 0x5b, 0x14, 0x96, 0xd7, 0x9a, 0xa3, 0x32, 0x29, 0xbe, 0xa3, 0xc0,
      0xe8, 0xdb, 0xcf, 0x1d, 0x7a, 0x64, 0x73, 0x09, 0x28, 0x77, 0x5c, 0x74,
      0xc9, 0x46, 0x18, 0x1b, 0xb8, 0x8e, 0x0a, 0x38, 0xa1, 0x65, 0xeb, 0xbd,
      0xc7, 0x86, 0x0b, 0x01, 0x4c, 0x59, 0x89, 0x4c, 0x72, 0x11, 0xbc, 0x23,
      0xd3, 0x90, 0x77, 0xe6, 0x29, 0x44, 0xd3, 0xc4, 0x8c, 0x43, 0x4b, 0x73,
      0x7b, 0x1a, 0x64, 0xb5, 0xf9, 0xa1, 0x64, 0x6a, 0x18, 0xa5, 0x19, 0x5a,
      0x54, 0x60, 0x3f, 0xfc, 0x28, 0x0b, 0xf5, 0xea, 0x1b, 0xf3, 0xd8, 0xcb,
      0x41, 0x12, 0xaa, 0x82, 0xfa, 0xb8, 0x9e, 0x26, 0x33, 0x66, 0x55, 0x97,
      0xbe, 0xd3, 0x0d, 0xe6, 0xc8, 0x89, 0x7a, 0x96, 0x13, 0xea, 0x8c, 0x11,
      0x73, 0x65, 0x24, 0xa7, 0x08, 0x62, 0x66, 0x52, 0x3f, 0xba, 0xb8, 0x87,
      0x46, 0x54, 0x94, 0xeb, 0xfc, 0x96, 0xfb, 0xe4, 0x46, 0x67, 0x75, 0xbc,
      0x0d, 0x70, 0x7e, 0x62, 0x4b, 0xb3, 0xde, 0x70, 0x0e, 0xaa, 0xc9, 0xb3,
      0x7c, 0x2b, 0xc1, 0x2b, 0x52, 0xaa, 0x2d, 0xfb, 0x73, 0xb7, 0x00, 0x6b,
      0xd9, 0x56, 0xc0, 0xb9, 0x32, 0x4c, 0x47, 0x29, 0x5e, 0x4e, 0x4c, 0x4c,
      0xc4, 0x7b, 0xa9, 0xb5, 0xa8, 0x99, 0x69, 0x4b, 0x38, 0xa6, 0x4a, 0x29,
      0x12, 0x06, 0x84, 0xbf, 0x64, 0xc6, 0x48, 0x21, 0x1b, 0x18, 0xd0, 0x17,
      0xce, 0x34, 0x84, 0x9c, 0x4a, 0x09, 0x2b, 0x24, 0xbc, 0xb8, 0x96, 0x0d,
      0x87, 0x04, 0x22, 0xeb, 0xd9, 0x54, 0xbc, 0x72, 0x64, 0xba, 0x95, 0xa7,
      0x27, 0xa3, 0x19, 0x26, 0xf3, 0x1c, 0x12, 0x38, 0x1d, 0x0d, 0x05, 0x1e,
      0x1e, 0x04, 0x19, 0x0b, 0x94, 0x93, 0x22, 0x44, 0x28, 0x45, 0xf3, 0xaa,
      0x6f, 0xb1, 0x79, 0x40, 0x44, 0x2a, 0x9c, 0x1a, 0x18, 0x3b, 0x23, 0x97,
      0xdc, 0xc0, 0x77, 0x0c, 0xb2, 0xb4, 0xa0, 0x7a, 0xaf, 0x0a, 0xf5, 0x26,
      0x36, 0x77, 0x8a, 0xc9, 0x94, 0x9d, 0x2d, 0xb4, 0x2f, 0x3d, 0x03, 0x6b,
      0xc8, 0x42, 0xb2, 0xf3, 0xb0, 0xb1, 0x1d, 0x27, 0x8e, 0xa0, 0x25, 0xcc,
      0x16, 0xac, 0x8b, 0x69, 0x86, 0xa9, 0xb1, 0x8b, 0x33, 0x56, 0x79, 0x0d,
      0xbc, 0x82, 0x1d, 0x0a, 0xc7, 0xb5, 0xea, 0xe9, 0x65, 0xfc, 0xf1, 0x4b,
      0x0e, 0x20, 0x2b, 0x00, 0xec, 0x5d, 0x70, 0x7c, 0x45, 0xaf, 0x52, 0xbe,
      0x2d, 0x97, 0x71, 0xbc, 0xc5, 0x3b, 0x31, 0xf2, 0x52, 0x5a, 0xb5, 0x3f,
      0x95, 0xa3, 0xb0, 0x02, 0x42, 0xc1, 0x1c, 0x49, 0x4e, 0x25, 0x54, 0x43,
      0x8b, 0x0a, 0x4f, 0x68, 0x19, 0x40, 0x19, 0x18, 0x1b, 0xbd, 0xb1, 0x94,
      0x91, 0x03, 0x07, 0xbc, 0x19, 0xaf, 0xae, 0x55, 0x3d, 0xd6, 0x66, 0x56,
      0x43, 0x32, 0xc0, 0x78, 0x60, 0x4f, 0xbd, 0x8b, 0xc4, 0xfd, 0xc9, 0x2f,
      0x92, 0x01, 0x80, 0x9a, 0xb4, 0x95, 0xc1, 0xb5, 0x35, 0x53, 0x0b, 0xad,
      0x2e, 0xb3, 0x57, 0x53, 0xf9, 0x19, 0x56, 0xb5, 0x8b, 0xfb, 0xf6, 0x2d,
      0x1e, 0x33, 0xc8, 0x61, 0xf7, 0x38, 0x46, 0x33, 0x4c, 0x4a, 0x2a, 0xc5,
      0x57, 0x69, 0x84, 0xef, 0x11, 0x63, 0xa5, 0x04, 0x54, 0xa3, 0xe4, 0xce,
      0x83, 0x88, 0x98, 0xf2, 0x25, 0x86, 0x81, 0x89, 0xbd, 0xae, 0x5a, 0x03,
      0xef, 0xd8, 0x00, 0x1d, 0x2a, 0x9f, 0x55, 0x7c, 0xb6, 0xcd, 0x93, 0xa4,
      0x6f, 0xbb, 0x13, 0x79, 0xd8, 0x16, 0x45, 0xb8, 0x53, 0xcb, 0x98, 0xb4,
      0x7b, 0xa8, 0x7a, 0x66, 0xfc, 0xbc, 0x0a, 0xda, 0x76, 0x50, 0x6b, 0x8d,
      0xcd, 0x65, 0x90, 0x42, 0x64, 0x0e, 0xf6, 0x16, 0x82, 0x0f, 0xf2, 0x0b,
      0x26, 0x43, 0x33, 0x99, 0x37, 0x1d, 0xc4, 0xc5, 0x02, 0x87, 0x34, 0x9c,
      0xa2, 0x76, 0x3e, 0x30, 0x73, 0x86, 0x5c, 0x5b, 0xc9, 0xaa, 0xa1, 0x43,
      0x7b, 0x86, 0xb2, 0x0b, 0x25, 0xb6, 0xf0, 0x88, 0xbd, 0x6a, 0x4c, 0x94,
      0x83, 0x84, 0x2d, 0x7c, 0x65, 0x25, 0x7c, 0xc4, 0x2a, 0xd7, 0xc7, 0xb2,
      0x6f, 0x58, 0xc8, 0xed, 0x19, 0x9b, 0x84, 0x8b, 0x8c, 0x6f, 0x40, 0x8d,
      0x48, 0x76, 0x37, 0x75, 0x18, 0x88, 0xa9, 0x5a, 0x44, 0x9c, 0xab, 0x47,
      0xb7, 0xa2, 0xac, 0xe6, 0xa8, 0x2c, 0x7d, 0x46, 0x91, 0x05, 0x26, 0x41,
      0x4a, 0xb1, 0x49, 0xb1, 0x13, 0x43, 0xb0, 0xea, 0x32, 0xd9, 0x69, 0xa5,
      0x08, 0xa8, 0x1c, 0xf8, 0x96, 0xb0, 0x81, 0x08, 0x7f, 0x4d, 0x52, 0xa2,
      0xa9, 0xe0, 0x77, 0x87, 0x8a, 0x43, 0xe5, 0x97, 0x1d, 0x74, 0x86, 0x3a,
      0x7b, 0xf2, 0x00, 0x37, 0xb2, 0x97, 0x47, 0x03, 0x14, 0x0e, 0xa8, 0x16,
      0x0a, 0x5b, 0xa7, 0x22, 0x27, 0xc1, 0x5b, 0xec, 0x5d, 0xf1, 0x71, 0x14,
      0x01, 0x98, 0x89, 0x6d, 0x31, 0x73, 0x2a, 0x24, 0x1f, 0x5b, 0x93, 0x8f,
      0xf2, 0x33, 0xcb, 0xa2, 0xdc, 0x82, 0xde, 0x91, 0xc6, 0xb4, 0x93, 0x78,
      0x2b, 0x61, 0x63, 0xbc, 0x12, 0x82, 0xdf, 0x85, 0x5e, 0xe6, 0xa6, 0xa6,
      0x59, 0x75, 0xb3, 0x3c, 0xb6, 0xa2, 0x7d, 0x25, 0x8b, 0xd3, 0x17, 0xd0,
      0x4c, 0xef, 0x8e, 0xb5, 0x57, 0xba, 0x02, 0xd4, 0x94, 0x71, 0x92, 0x3c,
      0xd6, 0x0e, 0x99, 0x17, 0x96, 0x6b, 0xe9, 0x0f, 0x56, 0xd2, 0xd5, 0x3f,
      0xa3, 0xbc, 0x26, 0x29, 0x74, 0x0a, 0xbb, 0xd1, 0x67, 0x20, 0xa9, 0xa7,
      0x06, 0x9a, 0x64, 0xde, 0x7a, 0x26, 0x32, 0x88, 0x17, 0xd7, 0x4a, 0x7c,
      0x2f, 0x55, 0xa2, 0x46, 0x1b, 0x07, 0x4b, 0xff, 0x80, 0x44, 0x44, 0x5e,
      0x11, 0x66, 0x0b, 0x1b, 0x6b, 0x26, 0xdf, 0x24, 0x2b, 0x8f, 0xc0, 0x2b,
      0x9e, 0x8d, 0xf5, 0x38, 0xdb, 0x17, 0xa6, 0x39, 0xd7, 0xc4, 0x61, 0x32};
  if (ml_kem_512_keypair_deterministic_no_self_test(
          keygen_encaps, &public_len, keygen_decaps, &secret_len, kKeyGenDKSeed) ||
      !check_test(kKeyGenDK, keygen_decaps, sizeof(keygen_decaps),
                  "ML-KEM-keyGen-decaps")) {
    goto err;
  }

  static const uint8_t kEncapEK[MLKEM512_PUBLIC_KEY_BYTES] = {
      0x57, 0xc3, 0xba, 0x4c, 0xd7, 0x81, 0xd8, 0x69, 0x0b, 0x4c, 0x39, 0x0d,
      0x9a, 0x58, 0xb3, 0x5d, 0x69, 0xa5, 0x2d, 0x52, 0xcd, 0x19, 0x01, 0x2a,
      0x25, 0xe1, 0x58, 0xa2, 0xc1, 0x9b, 0x75, 0x47, 0x0a, 0x03, 0x9a, 0x05,
      0xc5, 0x98, 0x33, 0xc5, 0xd2, 0xa8, 0x28, 0x0b, 0x33, 0xde, 0x95, 0xb6,
      0x0c, 0x1b, 0xb5, 0xc6, 0x33, 0xaf, 0x71, 0x13, 0x8f, 0xb0, 0x64, 0x28,
      0x14, 0x37, 0x12, 0xb7, 0xa0, 0x33, 0xeb, 0xaa, 0x3c, 0x0d, 0x6a, 0x84,
      0x12, 0x11, 0x20, 0x4c, 0x67, 0x56, 0x4b, 0xc5, 0x16, 0xf3, 0x43, 0x73,
      0xb3, 0x04, 0x70, 0xa1, 0xa5, 0x03, 0x3b, 0x85, 0x2a, 0x2c, 0x3a, 0x80,
      0xad, 0xfc, 0x8f, 0x8c, 0x17, 0x62, 0xfb, 0x06, 0x29, 0x0e, 0x55, 0x75,
      0x9d, 0x46, 0x6d, 0x67, 0xd2, 0x7e, 0xfd, 0x3b, 0xb8, 0xf6, 0x5c, 0x7f,
      0x1a, 0xd0, 0x11, 0xb9, 0x9a, 0xc6, 0x96, 0xe0, 0xba, 0x78, 0xf4, 0x56,
      0xc1, 0x91, 0x03, 0x30, 0x43, 0x0d, 0xa6, 0xcc, 0x91, 0x86, 0xcb, 0x21,
      0x9e, 0x47, 0xa8, 0xce, 0xfb, 0xc2, 0xbe, 0x58, 0x05, 0x52, 0xc3, 0x0a,
      0xf1, 0xe8, 0x73, 0x4c, 0x57, 0x2f, 0x48, 0x90, 0xc1, 0xec, 0x14, 0x22,
      0x5b, 0x2a, 0xc6, 0x41, 0xe6, 0xc2, 0x3c, 0x94, 0x94, 0x29, 0x84, 0x32,
      0x3f, 0xc5, 0x4a, 0xf9, 0x29, 0x9a, 0x72, 0xe9, 0x2d, 0x05, 0x68, 0x37,
      0xc4, 0x50, 0x54, 0x5a, 0x88, 0x49, 0x41, 0x7b, 0x40, 0xa3, 0x23, 0x68,
      0x62, 0xb4, 0x07, 0xd0, 0x9a, 0x70, 0x22, 0xdc, 0x54, 0x47, 0x55, 0x15,
      0x5b, 0x23, 0xaf, 0x0f, 0x30, 0x6d, 0xac, 0x83, 0x37, 0x34, 0x78, 0x03,
      0x41, 0x2c, 0xcd, 0x57, 0xdb, 0x4b, 0xeb, 0xf9, 0x78, 0x70, 0xd5, 0xb8,
      0x61, 0xdb, 0xac, 0x6c, 0x4a, 0x8d, 0x7f, 0xdb, 0xc9, 0x26, 0x92, 0x3f,
      0xae, 0xa1, 0x71, 0xbe, 0x17, 0x23, 0x54, 0xf5, 0x4c, 0x11, 0xb6, 0x8c,
      0x9f, 0xf6, 0xc0, 0xfe, 0x05, 0x86, 0x93, 0x24, 0x61, 0x16, 0x8b, 0xa9,
      0x01, 0x88, 0xc8, 0x6c, 0xba, 0xce, 0x80, 0x49, 0x82, 0xe1, 0x88, 0x12,
      0x16, 0x40, 0x11, 0x66, 0xc6, 0x5a, 0x55, 0x64, 0x25, 0xab, 0x21, 0x45,
      0x51, 0x6c, 0x76, 0x99, 0x03, 0x1d, 0xf5, 0x6b, 0x14, 0x78, 0x1b, 0xa2,
      0x44, 0x57, 0x29, 0x47, 0xf6, 0x3f, 0x7c, 0xe9, 0x35, 0x62, 0xb1, 0x09,
      0x01, 0x78, 0x04, 0x01, 0xe2, 0x44, 0x17, 0x47, 0x97, 0xcf, 0xe6, 0xab,
      0x5d, 0x5a, 0xca, 0xfd, 0x07, 0x48, 0x6d, 0x13, 0x36, 0x30, 0x63, 0x1e,
      0x3e, 0x00, 0x9f, 0xda, 0x23, 0xc4, 0x27, 0xf5, 0xb4, 0xc2, 0x05, 0x64,
      0x31, 0x47, 0xc0, 0xef, 0x17, 0xcc, 0x4a, 0x67, 0x4c, 0x96, 0x91, 0x77,
      0xb5, 0x0a, 0x37, 0x24, 0x25, 0xad, 0xbe, 0x65, 0x4a, 0x74, 0xca, 0xa8,
      0x87, 0xf8, 0x7c, 0xdd, 0xe4, 0x92, 0xec, 0x50, 0x27, 0x05, 0x60, 0x9b,
      0x9e, 0x31, 0x8a, 0xe6, 0xbb, 0x5c, 0xe4, 0xa9, 0xa2, 0x82, 0xf1, 0x24,
      0x44, 0x77, 0x9b, 0xba, 0xe0, 0x32, 0x0d, 0x12, 0x97, 0xd9, 0xcb, 0xbb,
      0x87, 0x61, 0x00, 0x03, 0x07, 0x89, 0xc9, 0x05, 0x21, 0x5f, 0x38, 0x69,
      0x03, 0xa9, 0x4e, 0xe6, 0x8c, 0xbc, 0x97, 0x73, 0x10, 0xcb, 0xa0, 0x2b,
      0x79, 0x37, 0x5c, 0x4a, 0x8a, 0x5e, 0xec, 0x55, 0x69, 0x52, 0x8a, 0x10,
      0x02, 0xc0, 0x8a, 0x90, 0x9b, 0xca, 0x05, 0xf1, 0x32, 0x3a, 0x34, 0xae,
      0xf2, 0x08, 0x1f, 0x66, 0x76, 0x93, 0x9b, 0x82, 0x6e, 0xb7, 0x74, 0x8b,
      0x80, 0xd4, 0x5a, 0x1e, 0xa1, 0x5a, 0x6f, 0xb3, 0x19, 0x63, 0xf6, 0x18,
      0x19, 0x96, 0x30, 0xa2, 0xd3, 0x30, 0x1a, 0xf1, 0x13, 0xde, 0xf4, 0x28,
      0xbf, 0x10, 0x75, 0x18, 0xfc, 0x30, 0x55, 0xf2, 0x4d, 0x28, 0xa3, 0x8c,
      0xb4, 0x54, 0x43, 0x48, 0x16, 0x0b, 0xc6, 0x30, 0x1d, 0xd3, 0xe7, 0x60,
      0x82, 0x03, 0x9c, 0x12, 0xb2, 0x92, 0x51, 0xe9, 0x03, 0xc6, 0x8c, 0x7b,
      0xd8, 0x41, 0xa4, 0x53, 0x5c, 0x5d, 0xd7, 0xc9, 0x5a, 0xac, 0xfa, 0x80,
      0x66, 0xf7, 0x3b, 0xa8, 0xc9, 0xac, 0xa4, 0xd3, 0x03, 0xa9, 0xf1, 0xb7,
      0x6f, 0xc0, 0x36, 0x92, 0xa9, 0xbf, 0x11, 0x15, 0xb8, 0x51, 0x86, 0x62,
      0x48, 0x66, 0x1c, 0xfa, 0xf5, 0x39, 0xc1, 0x39, 0x9e, 0x5d, 0xe0, 0x3b,
      0x86, 0xb0, 0x2c, 0xb9, 0x9b, 0x2d, 0xa7, 0xd0, 0x07, 0xce, 0x35, 0x43,
      0xd4, 0xbb, 0x94, 0xfb, 0x1c, 0xc9, 0x9e, 0x40, 0xcd, 0xe8, 0x71, 0x76,
      0x75, 0x29, 0x14, 0x4d, 0xa3, 0x29, 0x79, 0x47, 0x55, 0x56, 0x45, 0x33,
      0x3e, 0xf0, 0x58, 0x37, 0x70, 0x98, 0xb2, 0xa1, 0x51, 0xd5, 0x11, 0x0f,
      0xce, 0x09, 0x21, 0x7d, 0x5a, 0x00, 0x55, 0xa2, 0x08, 0xf3, 0xb5, 0x83,
      0xe2, 0xc7, 0x13, 0x67, 0x72, 0x24, 0x1b, 0x85, 0xb5, 0xbb, 0x36, 0x7d,
      0xb4, 0x3c, 0x07, 0xe5, 0x48, 0x09, 0xb7, 0xa3, 0x0c, 0x82, 0xd9, 0xaf,
      0x74, 0x79, 0xab, 0x37, 0xd8, 0x49, 0xd7, 0x78, 0x91, 0x39, 0xd6, 0xbc,
      0xfa, 0xf8, 0x31, 0x46, 0x77, 0x5f, 0x6f, 0x1a, 0x63, 0x2d, 0x17, 0x91,
      0x0f, 0x0a, 0x49, 0xf6, 0x74, 0x72, 0xc1, 0x8a, 0x34, 0x39, 0xcc, 0x03,
      0x7c, 0xa4, 0x2d, 0x7e, 0xb8, 0x61, 0x09, 0xe3, 0xc4, 0x7b, 0xd9, 0x4e,
      0xc7, 0xbc, 0x99, 0xd5, 0xb8, 0x2a, 0x8f, 0x4c, 0x4d, 0x1d, 0x72, 0x47,
      0xc8, 0x47, 0x99, 0xa9, 0x05, 0x8d, 0x01, 0x84, 0x2f, 0xed, 0x69, 0x57,
      0xce, 0xb4, 0x8d, 0xd2, 0x02, 0x91, 0x22, 0xaa, 0x8f, 0xf4, 0x38, 0x84,
      0x10, 0xe1, 0xb3, 0xbb, 0xa2, 0x76, 0xc2, 0xaa, 0x54, 0x4e, 0xf9, 0x91,
      0x97, 0x3f, 0x15, 0xe7, 0xc5, 0x9b, 0x89, 0x4a, 0xab, 0x27, 0xf5, 0xce,
      0xad, 0xdd, 0xe8, 0x25, 0x33, 0x3f, 0x44, 0x03, 0x9b, 0x02, 0xe7, 0xbd,
      0xec, 0x21, 0xfc, 0x0d, 0x8f, 0x9b, 0x3a, 0x22};
  static const uint8_t kEncapM[MLKEM512_ENCAPS_SEED_LEN] = {
      0x2c, 0x87, 0xaa, 0x8b, 0x11, 0x76, 0x75, 0x54, 0x74, 0xdf, 0x76,
      0x3b, 0x2a, 0xe0, 0x46, 0x35, 0x39, 0xe9, 0x53, 0xe0, 0x04, 0xc4,
      0x6a, 0x11, 0x83, 0xfd, 0x53, 0xcf, 0x84, 0xef, 0x81, 0x03};
  static const uint8_t kEncapCiphertext[MLKEM512_CIPHERTEXT_BYTES] = {
      0x43, 0x1a, 0x4f, 0x1b, 0x2d, 0x2c, 0x6c, 0x00, 0xf1, 0x69, 0x0b, 0xbe,
      0x48, 0x25, 0x41, 0xef, 0x3d, 0x56, 0x37, 0x74, 0xda, 0xff, 0x83, 0x20,
      0x7f, 0x96, 0xde, 0x7e, 0x5e, 0x4a, 0x59, 0xd5, 0xd9, 0x36, 0xd9, 0x44,
      0x3a, 0xd4, 0x22, 0xe6, 0x45, 0x79, 0x3e, 0x7a, 0x60, 0xa9, 0xb0, 0xa7,
      0x6c, 0xd6, 0x72, 0xd2, 0x0c, 0x69, 0xb8, 0x2a, 0x55, 0x63, 0xdf, 0x52,
      0xd9, 0x6f, 0x9a, 0x6c, 0xdf, 0xc5, 0x6f, 0xbd, 0x4f, 0xd8, 0xd5, 0xa8,
      0xaf, 0xeb, 0x2a, 0x09, 0xd9, 0x2e, 0xc8, 0x54, 0x09, 0x47, 0x94, 0xb4,
      0xed, 0x2d, 0xb3, 0x81, 0xf0, 0x4c, 0x68, 0x43, 0x96, 0x08, 0xaa, 0x99,
      0x02, 0xa4, 0xd1, 0x68, 0x9e, 0x2e, 0xb1, 0xe5, 0xf0, 0x7a, 0x4a, 0x1c,
      0x70, 0x92, 0x62, 0xd7, 0xc2, 0xff, 0x2f, 0x81, 0xf6, 0xee, 0xaa, 0xb2,
      0xa8, 0x6a, 0x41, 0xba, 0x21, 0x0e, 0xb1, 0xbf, 0x8e, 0x75, 0xfe, 0xbc,
      0xcd, 0x1a, 0x15, 0xb4, 0xd7, 0xa7, 0xb6, 0x02, 0x57, 0xc8, 0x9d, 0x00,
      0xbd, 0x81, 0xd3, 0x9f, 0xcb, 0x8d, 0x1c, 0xe3, 0x27, 0x81, 0x02, 0x59,
      0x5d, 0xd6, 0x52, 0xf7, 0xfb, 0x7d, 0x55, 0x84, 0x87, 0x4f, 0x33, 0x27,
      0xb1, 0x74, 0x04, 0x3b, 0x35, 0x0e, 0xbd, 0x4d, 0x41, 0xfe, 0x08, 0xbd,
      0x0e, 0x85, 0x4d, 0x41, 0xcb, 0xb0, 0x27, 0xc4, 0x81, 0xda, 0x64, 0xdc,
      0x61, 0x51, 0xb8, 0x8d, 0xec, 0xec, 0xcf, 0x02, 0x2d, 0xda, 0xc2, 0xe2,
      0x27, 0x36, 0xc1, 0x47, 0xe0, 0x77, 0x32, 0x94, 0x23, 0x1c, 0x05, 0x89,
      0x96, 0x71, 0x54, 0xc5, 0x26, 0xb0, 0xb7, 0xcd, 0xd5, 0x95, 0x68, 0xee,
      0xff, 0x57, 0x49, 0xa4, 0x0c, 0xb1, 0x00, 0xc6, 0x0c, 0x64, 0x80, 0x89,
      0x76, 0x55, 0xd9, 0x6e, 0x9f, 0x64, 0xd6, 0x16, 0x84, 0xc0, 0xb3, 0x15,
      0x06, 0x46, 0x73, 0x2c, 0x19, 0x40, 0x9f, 0xe5, 0x65, 0x54, 0x0a, 0x31,
      0x89, 0x47, 0x03, 0xcf, 0x01, 0x79, 0xca, 0xe8, 0x5b, 0xc8, 0xc1, 0xa5,
      0x73, 0x26, 0x49, 0x83, 0x6e, 0x48, 0xe6, 0x76, 0x40, 0x5b, 0x95, 0x91,
      0xb6, 0x5b, 0xa2, 0x5f, 0x9b, 0x48, 0x9b, 0x9e, 0x57, 0x72, 0xaa, 0x1e,
      0xd5, 0xa0, 0x01, 0x43, 0xcb, 0x9f, 0x54, 0x49, 0xfd, 0x01, 0x34, 0x57,
      0xa3, 0xc1, 0x38, 0x74, 0xcb, 0x58, 0xc7, 0x5b, 0x52, 0xc9, 0xb6, 0xa9,
      0xae, 0x49, 0x5c, 0xcb, 0x50, 0x4a, 0x89, 0xcb, 0x5f, 0x14, 0x56, 0x95,
      0xb9, 0x21, 0x63, 0x2f, 0xb8, 0x5b, 0x03, 0x16, 0xb3, 0x0d, 0x4a, 0xd1,
      0x7f, 0xef, 0x08, 0x62, 0xd6, 0xb1, 0xe6, 0xca, 0x6a, 0x61, 0x1c, 0x8a,
      0x6a, 0x72, 0x34, 0xb4, 0x36, 0x2c, 0x5c, 0xa0, 0xad, 0x9f, 0x76, 0x97,
      0x68, 0x77, 0x98, 0xcf, 0x62, 0x4d, 0xc9, 0xf3, 0x5f, 0xbb, 0x37, 0x6e,
      0x09, 0x95, 0x31, 0x56, 0x53, 0x2a, 0x90, 0x33, 0x70, 0x9d, 0xf7, 0x55,
      0xb4, 0x6c, 0xc6, 0xd8, 0x3d, 0xe3, 0xa1, 0x11, 0xe1, 0x9a, 0x76, 0xb3,
      0x61, 0xe0, 0xef, 0x14, 0xc9, 0x1d, 0xb8, 0xd9, 0x1c, 0x6c, 0x6d, 0x9e,
      0x3e, 0x46, 0xf4, 0x22, 0x91, 0xfd, 0x6c, 0xbf, 0x5c, 0xfd, 0x12, 0x27,
      0x16, 0xfb, 0x06, 0x75, 0x69, 0x8e, 0x60, 0x2a, 0xb3, 0x9e, 0xe9, 0x8e,
      0x0d, 0x81, 0x45, 0xee, 0xba, 0xaa, 0x93, 0x74, 0xf5, 0xb3, 0xbb, 0x0d,
      0xf4, 0xd0, 0xfd, 0x83, 0xa4, 0x0e, 0x0d, 0x25, 0x03, 0x8c, 0x39, 0xe9,
      0xbe, 0xe0, 0x1c, 0xf7, 0x9c, 0x86, 0xf3, 0x08, 0x61, 0x58, 0xd0, 0x31,
      0xd5, 0xc5, 0xe8, 0x6b, 0xc7, 0xe7, 0xeb, 0x16, 0xe6, 0x22, 0x50, 0x5f,
      0x28, 0x88, 0x21, 0x38, 0x84, 0xc0, 0xb5, 0x25, 0x22, 0x89, 0xb1, 0x1f,
      0xce, 0x5b, 0xfe, 0xeb, 0xfb, 0xef, 0x0a, 0x32, 0xce, 0xaf, 0x9c, 0x14,
      0xc6, 0x25, 0x00, 0x90, 0x02, 0x84, 0x63, 0xdb, 0x6f, 0x8d, 0x19, 0x68,
      0x4f, 0x54, 0x11, 0x08, 0xfe, 0x93, 0x4d, 0x88, 0xe7, 0xef, 0x5c, 0xce,
      0x9d, 0xae, 0xbb, 0x32, 0x70, 0x0b, 0x93, 0x97, 0x69, 0x1a, 0x68, 0x42,
      0x98, 0xc9, 0xbf, 0x1b, 0x7c, 0x22, 0xd1, 0xbc, 0xec, 0x3f, 0xca, 0xcf,
      0xbb, 0x17, 0xf2, 0xed, 0x2b, 0x98, 0xb8, 0x5e, 0x6a, 0x8f, 0xe2, 0x48,
      0x29, 0x96, 0xb5, 0xe0, 0x99, 0xe9, 0xd0, 0x21, 0x1c, 0xb9, 0x41, 0x26,
      0x14, 0xde, 0x87, 0xdc, 0x18, 0xd2, 0x36, 0x13, 0xed, 0x7f, 0x6c, 0x29,
      0xcc, 0x37, 0xb7, 0x27, 0x11, 0x6d, 0xd9, 0x01, 0xc2, 0x81, 0x79, 0x38,
      0xc2, 0x9f, 0xcd, 0x02, 0x60, 0x89, 0x33, 0x6a, 0xdd, 0xc0, 0x9e, 0xca,
      0x90, 0xde, 0x9a, 0x25, 0xa6, 0x37, 0x4f, 0xee, 0x86, 0xbc, 0xdd, 0x06,
      0xae, 0x3d, 0xaa, 0xf0, 0xb1, 0xbc, 0x5b, 0x3b, 0x27, 0x90, 0xd4, 0xd9,
      0xf7, 0x59, 0xbe, 0xf8, 0xac, 0x74, 0x36, 0x12, 0xa2, 0xbb, 0xf6, 0xe4,
      0x5d, 0xe8, 0xb2, 0x2e, 0xfa, 0x61, 0x22, 0x66, 0x25, 0xd4, 0xc3, 0x9f,
      0x34, 0x6b, 0x84, 0x4c, 0x5e, 0xbe, 0xc5, 0x35, 0x58, 0x66, 0xc0, 0x0b,
      0x72, 0x6c, 0xc1, 0x64, 0x0c, 0xb2, 0x37, 0xc3, 0x4a, 0x20, 0xa7, 0xc6,
      0x03, 0xd2, 0x51, 0xf4, 0x6e, 0x6b, 0x3b, 0x0f, 0xa7, 0x1b, 0x32, 0x76,
      0x83, 0x5e, 0x3e, 0x9d, 0xa5, 0xb9, 0x48, 0x5e, 0x78, 0x96, 0x14, 0xaf,
      0x49, 0xf1, 0xe9, 0x50, 0x4d, 0xb2, 0x52, 0x86, 0x31, 0xfb, 0xe1, 0xcd,
      0x7d, 0xbe, 0xe8, 0x51, 0x64, 0xe4, 0xc0, 0x99, 0xa2, 0x7a, 0x45, 0x83,
      0xe9, 0x24, 0x7d, 0x07, 0x8f, 0x88, 0x30, 0xb4, 0x68, 0x74, 0xc1, 0xb0,
      0x10, 0xbf, 0x3c, 0xd9, 0x0e, 0xb0, 0x77, 0x49, 0x61, 0xf2, 0x39, 0xba};
  static const uint8_t kEncapSharedSecret[MLKEM512_SHARED_SECRET_LEN] = {
      0xa7, 0x72, 0xdf, 0x2d, 0xe2, 0x50, 0xac, 0x7d, 0x89, 0x6b, 0xbb,
      0x82, 0x0b, 0x57, 0xf2, 0xae, 0x05, 0xf9, 0xa4, 0x12, 0xab, 0x55,
      0xba, 0xa4, 0x21, 0xd4, 0xaf, 0x6d, 0xac, 0x62, 0x66, 0x2a};

  size_t ciphertext_len = MLKEM512_CIPHERTEXT_BYTES;
  size_t shared_secret_len = MLKEM512_SHARED_SECRET_LEN;
  uint8_t ciphertext[MLKEM512_CIPHERTEXT_BYTES] = {0};
  uint8_t shared_secret[MLKEM512_SHARED_SECRET_LEN] = {0};

  if (ml_kem_512_encapsulate_deterministic_no_self_test(
          ciphertext, &ciphertext_len, shared_secret, &shared_secret_len, kEncapEK, kEncapM) ||
      !check_test(kEncapCiphertext, ciphertext, sizeof(kEncapCiphertext),
                  "ML-KEM-encapsulate-ciphertext") ||
      !check_test(kEncapSharedSecret, shared_secret, sizeof(kEncapSharedSecret),
                  "ML-KEM-encapsulate-shared-secret")) {
    goto err;
  }

  shared_secret_len = MLKEM512_SHARED_SECRET_LEN;
  static const uint8_t kDecapDK[MLKEM512_SECRET_KEY_BYTES] = {
      0x73, 0x9b, 0x8b, 0x1f, 0x6a, 0x57, 0x66, 0x31, 0x0b, 0x06, 0x19, 0x04,
      0x02, 0x14, 0x38, 0xbb, 0xd6, 0x1a, 0x14, 0xf0, 0x85, 0xfd, 0xe0, 0x29,
      0xb5, 0x33, 0x86, 0xec, 0x37, 0x61, 0xaa, 0xe7, 0x78, 0x28, 0xfb, 0x19,
      0xde, 0xdc, 0x50, 0xdd, 0xc1, 0xc3, 0x2d, 0x3a, 0x44, 0x4a, 0x15, 0x4b,
      0xf8, 0x33, 0xa8, 0x25, 0x71, 0x31, 0x5a, 0x56, 0x55, 0xfd, 0x3b, 0x06,
      0x65, 0x1c, 0x20, 0xf7, 0xb0, 0x37, 0x8d, 0x78, 0x24, 0x72, 0x68, 0x0c,
      0xfb, 0xb9, 0x1d, 0x07, 0xd2, 0x31, 0x10, 0xd8, 0xb1, 0x58, 0xe3, 0xa8,
      0xf3, 0x29, 0x5d, 0x92, 0x97, 0x64, 0x6d, 0xe5, 0x81, 0x4c, 0xe8, 0x2a,
      0xe8, 0x74, 0xa6, 0x45, 0x4c, 0x61, 0xf6, 0x5b, 0x0b, 0xed, 0x79, 0x77,
      0xce, 0x01, 0xc7, 0x69, 0x99, 0xce, 0x86, 0x75, 0x6d, 0xa2, 0x6a, 0x2f,
      0x5a, 0x09, 0x3c, 0x05, 0x70, 0x73, 0x16, 0xd9, 0x2a, 0xc2, 0xb2, 0x5e,
      0x72, 0xc2, 0x16, 0x0e, 0x60, 0x4c, 0x90, 0xc3, 0x39, 0xd8, 0xd0, 0x3f,
      0xa8, 0x45, 0x4d, 0x40, 0xc7, 0x07, 0x81, 0x33, 0x05, 0xe6, 0x0a, 0x48,
      0x89, 0x2c, 0x6d, 0x48, 0xfb, 0x18, 0x19, 0xf8, 0xa0, 0x92, 0x18, 0x62,
      0x49, 0xe5, 0x02, 0x1d, 0x83, 0x8b, 0xac, 0xe8, 0x19, 0xa1, 0xc7, 0xc7,
      0x2c, 0x5b, 0x45, 0xcf, 0x83, 0x98, 0xc5, 0x2a, 0x4b, 0xed, 0xd3, 0xa5,
      0x9e, 0xa4, 0x5e, 0x48, 0xf7, 0x6e, 0xc3, 0xe0, 0x78, 0x68, 0x71, 0x43,
      0x13, 0x72, 0xb3, 0x6f, 0xa7, 0xb5, 0x63, 0x97, 0xcb, 0x95, 0xe9, 0x20,
      0x0a, 0x99, 0x0b, 0x71, 0xbb, 0x4e, 0x10, 0xba, 0x3e, 0x7c, 0x08, 0xc4,
      0xa6, 0x28, 0x28, 0xb5, 0x49, 0xcc, 0xe4, 0xf5, 0x08, 0x7f, 0xd2, 0xa4,
      0xeb, 0x78, 0x64, 0xfd, 0xdc, 0x5f, 0x92, 0xa8, 0x68, 0xd5, 0xbb, 0x70,
      0xab, 0x51, 0x51, 0x0c, 0xb0, 0x43, 0xc1, 0x45, 0xb7, 0x4e, 0xb0, 0x5f,
      0x72, 0xe5, 0x96, 0x4b, 0xc7, 0x05, 0x00, 0xb0, 0xc8, 0x93, 0xe0, 0x1e,
      0xa1, 0xab, 0x77, 0xc4, 0x73, 0x7f, 0x23, 0x6a, 0xa8, 0xdd, 0x1c, 0x61,
      0xf2, 0xd7, 0xae, 0x5c, 0x93, 0x10, 0x24, 0x28, 0x12, 0xaa, 0x03, 0x4a,
      0xaa, 0xd3, 0xbc, 0xdf, 0x22, 0x4d, 0x42, 0xf3, 0x47, 0x09, 0xc0, 0x5f,
      0x12, 0x38, 0xb2, 0x18, 0xec, 0x33, 0xdd, 0x80, 0x4c, 0xb3, 0x31, 0x32,
      0xf3, 0x64, 0x7c, 0xbd, 0x48, 0xa6, 0xf5, 0xf5, 0x4a, 0x68, 0x66, 0xb2,
      0x4b, 0x51, 0x83, 0xeb, 0x66, 0x91, 0xd0, 0xa0, 0x84, 0x6b, 0x34, 0x6f,
      0x87, 0x79, 0x87, 0xf2, 0x6b, 0x76, 0xbe, 0xb3, 0x05, 0x50, 0x7b, 0x34,
      0xf2, 0xe1, 0xcc, 0x95, 0xcb, 0x81, 0x85, 0xbb, 0x5e, 0x35, 0x60, 0x67,
      0x6f, 0x30, 0x27, 0x52, 0xc3, 0x82, 0x4e, 0x90, 0xbd, 0x84, 0x08, 0x1b,
      0x28, 0x89, 0x1f, 0xcd, 0xda, 0x31, 0x3f, 0x06, 0x37, 0x65, 0x59, 0x4e,
      0x41, 0xe0, 0x85, 0x2a, 0x65, 0x55, 0x1f, 0xf2, 0x50, 0xb3, 0x18, 0x8a,
      0x2a, 0x61, 0xbc, 0x5f, 0x76, 0x50, 0x3b, 0x39, 0x16, 0xf0, 0x64, 0x61,
      0xca, 0x01, 0x1f, 0xe1, 0xd1, 0x91, 0x53, 0xe2, 0x64, 0x29, 0xb0, 0x37,
      0x3a, 0x45, 0x39, 0xdf, 0x32, 0x47, 0x47, 0x76, 0xbe, 0xcb, 0xec, 0xa6,
      0x08, 0xe9, 0x70, 0x6f, 0x65, 0xad, 0x92, 0x42, 0x2b, 0xbb, 0x28, 0xb6,
      0xb9, 0xc9, 0x6c, 0xa0, 0x2b, 0x61, 0x3d, 0x5c, 0x06, 0xee, 0xe3, 0x1f,
      0x9f, 0x32, 0x22, 0x33, 0x00, 0xb2, 0x22, 0xe3, 0x95, 0x5c, 0xb0, 0x5a,
      0x1b, 0xab, 0xac, 0x60, 0x2a, 0xb3, 0x1e, 0x18, 0x74, 0x15, 0x4a, 0x83,
      0xd4, 0xbc, 0x54, 0xdc, 0xdc, 0x94, 0xd6, 0xd0, 0x5a, 0xdf, 0xbc, 0x6f,
      0x46, 0x61, 0x3d, 0x85, 0x81, 0x45, 0xed, 0xeb, 0x63, 0xf9, 0x20, 0x44,
      0x55, 0x1b, 0xb9, 0xe8, 0xbb, 0x63, 0x07, 0x10, 0x1b, 0xef, 0x5b, 0xc2,
      0x68, 0x5b, 0xcb, 0x04, 0x35, 0x3c, 0x94, 0x84, 0x3e, 0xc9, 0x94, 0x28,
      0x69, 0x89, 0x99, 0xf9, 0x95, 0x80, 0x33, 0x6a, 0x90, 0x3e, 0xb3, 0x08,
      0x08, 0xe3, 0x01, 0xa1, 0x88, 0x79, 0xb0, 0x9b, 0x8b, 0x20, 0x59, 0x68,
      0xe6, 0x24, 0x7f, 0x3a, 0x14, 0xb9, 0x00, 0x93, 0x5d, 0x80, 0x8a, 0x7c,
      0xf8, 0x21, 0x85, 0x15, 0xf3, 0xb8, 0xc1, 0x3b, 0x54, 0x07, 0x29, 0xaa,
      0x74, 0x1b, 0x42, 0x46, 0x0b, 0x41, 0x17, 0xe0, 0x55, 0x56, 0x19, 0x46,
      0x0a, 0x5a, 0x48, 0x4c, 0x52, 0x15, 0x0f, 0x3b, 0x1b, 0x60, 0x00, 0x95,
      0xd2, 0x25, 0x9f, 0x45, 0x56, 0x7a, 0x3e, 0x03, 0x4e, 0x14, 0x39, 0xaf,
      0x5e, 0xd3, 0xbc, 0x41, 0x61, 0x93, 0x52, 0x77, 0x53, 0x05, 0x29, 0x24,
      0x26, 0x18, 0x49, 0x13, 0xc2, 0x53, 0xd9, 0x54, 0x46, 0xfc, 0x45, 0x33,
      0xfe, 0x12, 0x79, 0xf5, 0x5a, 0x04, 0x9b, 0xcc, 0x70, 0xad, 0x50, 0x86,
      0xad, 0x72, 0xc7, 0x40, 0xd5, 0x3d, 0x58, 0x5b, 0x4b, 0x93, 0x20, 0x42,
      0xa2, 0x84, 0x3b, 0x69, 0x5c, 0x21, 0x9f, 0xb1, 0x3c, 0xa3, 0xc0, 0xcf,
      0x62, 0x56, 0xca, 0x82, 0xe5, 0x8a, 0x98, 0xfb, 0x79, 0x88, 0x06, 0x91,
      0x27, 0x1c, 0x1a, 0xc5, 0x30, 0x38, 0x95, 0x3c, 0x0e, 0xc6, 0x07, 0x92,
      0x06, 0xc5, 0x0c, 0x5d, 0x24, 0xcd, 0xfe, 0x80, 0x23, 0x43, 0x91, 0xba,
      0x17, 0x62, 0x19, 0x7d, 0xf1, 0x2c, 0xcd, 0x01, 0x04, 0x7b, 0x74, 0x3b,
      0x9c, 0xab, 0x2d, 0xc8, 0xab, 0xb8, 0x61, 0x19, 0x0f, 0xfc, 0x92, 0xc6,
      0xa2, 0x6b, 0x86, 0x50, 0xf6, 0xa0, 0x0d, 0xf4, 0x40, 0x4d, 0xb2, 0x54,
      0x24, 0x68, 0x6a, 0x34, 0x55, 0xc6, 0x45, 0x90, 0x51, 0x1a, 0x34, 0x3a,
      0x8c, 0x73, 0x4c, 0x55, 0x8b, 0x42, 0x74, 0xb4, 0x39, 0x77, 0x12, 0x13,
      0xed, 0x5b, 0xc6, 0x28, 0x58, 0x5f, 0x75, 0x0b, 0x67, 0xd3, 0xc0, 0x42,
      0x96, 0xdc, 0x4b, 0xda, 0x56, 0x44, 0xec, 0xd5, 0x84, 0xee, 0x87, 0x0b,
      0x46, 0xa7, 0x04, 0xe5, 0xd1, 0x9f, 0xa1, 0x37, 0x6e, 0x66, 0x88, 0x09,
      0x73, 0xab, 0x56, 0xa5, 0x16, 0x0d, 0x8b, 0x99, 0xc2, 0xb4, 0xba, 0x4e,
      0x17, 0x7c, 0x25, 0xb2, 0x4a, 0x9b, 0xdf, 0x80, 0xba, 0x1b, 0x48, 0xc2,
      0x7c, 0x88, 0xbd, 0x2c, 0xab, 0xbd, 0x98, 0x80, 0x34, 0xf3, 0x5c, 0xa7,
      0x92, 0x54, 0x9e, 0x4d, 0x57, 0x9b, 0x20, 0xa5, 0x07, 0x45, 0x47, 0x43,
      0x11, 0x97, 0xa3, 0x62, 0x7b, 0x5d, 0x7c, 0x75, 0x73, 0xfa, 0x93, 0x06,
      0x52, 0xfa, 0x99, 0x9a, 0x12, 0x89, 0x15, 0xcc, 0x01, 0x73, 0x7a, 0x08,
      0x70, 0xa3, 0x02, 0xa1, 0x51, 0x38, 0xb3, 0x85, 0x48, 0xb5, 0x9a, 0x65,
      0xd3, 0x97, 0xb6, 0xa0, 0xfc, 0xb6, 0x61, 0x23, 0xad, 0x93, 0x47, 0xc8,
      0x9c, 0xf5, 0xbe, 0x90, 0xa9, 0xac, 0xcf, 0x4a, 0x61, 0x6f, 0x42, 0x36,
      0x9a, 0x83, 0x99, 0xcf, 0xc4, 0x6e, 0x9c, 0xc6, 0xa1, 0x01, 0xfc, 0x35,
      0x3f, 0xf2, 0x27, 0xc6, 0xf2, 0x62, 0xb7, 0x95, 0xa2, 0x15, 0x70, 0x49,
      0x7a, 0x9b, 0xad, 0x2d, 0x7c, 0xbf, 0x2e, 0x87, 0x7a, 0xde, 0x82, 0xc8,
      0x37, 0xea, 0x68, 0xbc, 0x42, 0x43, 0x8d, 0xf7, 0x9e, 0xc5, 0x96, 0x31,
      0x8b, 0x57, 0x98, 0x1c, 0x35, 0x96, 0x87, 0x15, 0xa7, 0x59, 0xf4, 0x53,
      0x83, 0x5c, 0xbc, 0x27, 0xd7, 0x59, 0x4c, 0xb8, 0x65, 0xdd, 0x51, 0x15,
      0x21, 0x72, 0xb0, 0xa1, 0x36, 0xbe, 0x2a, 0x4a, 0x95, 0xed, 0x0a, 0x80,
      0x27, 0xe1, 0x5f, 0xca, 0x37, 0x35, 0x13, 0xe9, 0x8a, 0x2a, 0x32, 0xc8,
      0xe2, 0x24, 0x62, 0xff, 0xb4, 0x46, 0x43, 0x54, 0x1e, 0x8c, 0x45, 0x83,
      0x5e, 0x67, 0x81, 0xd1, 0x99, 0x8e, 0x97, 0xa9, 0x4e, 0xf4, 0x98, 0xcf,
      0x41, 0xa0, 0x80, 0x20, 0xb0, 0x9f, 0x9c, 0xac, 0x36, 0xac, 0x42, 0x28,
      0x89, 0x45, 0x58, 0xf3, 0x29, 0x61, 0x0d, 0x52, 0x88, 0x20, 0x8a, 0x9b,
      0x44, 0xf6, 0x14, 0xd8, 0x26, 0x44, 0x0a, 0x87, 0xb9, 0x20, 0xe0, 0x7c,
      0x54, 0x70, 0xc1, 0x01, 0x24, 0x5a, 0xd7, 0x78, 0x5b, 0x30, 0xa4, 0xc5,
      0x4c, 0xd3, 0xa0, 0xec, 0x08, 0x8c, 0xf6, 0xb2, 0x7d, 0x43, 0x10, 0x4c,
      0x63, 0xf0, 0x56, 0x75, 0x68, 0x26, 0x38, 0xfb, 0xaf, 0x16, 0xa8, 0xaa,
      0xb8, 0x91, 0x04, 0x58, 0xb5, 0xc1, 0xe1, 0xdb, 0x60, 0xb3, 0xa2, 0x06,
      0xb7, 0xf2, 0x7f, 0x50, 0x44, 0x0b, 0xc2, 0x26, 0x09, 0x58, 0x45, 0x38,
      0xb3, 0x53, 0x70, 0x09, 0x01, 0x98, 0x8b, 0x44, 0x56, 0xda, 0xd6, 0x0c,
      0x98, 0xfa, 0x54, 0xc2, 0xba, 0xab, 0xd2, 0xbb, 0x38, 0x79, 0xb7, 0x2f,
      0xbc, 0xd5, 0x03, 0x15, 0x3a, 0x68, 0xa5, 0x28, 0x3e, 0x17, 0x60, 0x89,
      0x6e, 0x19, 0xcf, 0xdb, 0xbc, 0xb4, 0x3e, 0x86, 0x45, 0x88, 0x9a, 0x2a,
      0xe1, 0x1c, 0x0f, 0x9c, 0x74, 0xaa, 0x38, 0xb5, 0xa8, 0x13, 0x12, 0x0a,
      0x3f, 0xf5, 0x08, 0xf7, 0x53, 0xc8, 0x28, 0xf6, 0x46, 0x65, 0xf1, 0x1e,
      0xa6, 0x1c, 0x54, 0xc2, 0xa8, 0xb0, 0xaa, 0xb8, 0xc3, 0xd8, 0x35, 0x34,
      0xaf, 0x09, 0x38, 0xdc, 0x98, 0x59, 0x8d, 0xf0, 0x15, 0xc8, 0x37, 0xc6,
      0x9f, 0xb5, 0x53, 0x0d, 0x70, 0x01, 0xf5, 0x3b, 0x42, 0x9c, 0x0a, 0xbe,
      0x42, 0x30, 0x60, 0x87, 0xe5, 0x38, 0xdc, 0x27, 0xc8, 0xd6, 0x34, 0x80,
      0x96, 0xd7, 0xcf, 0x19, 0x41, 0x2e, 0x3e, 0x32, 0xb3, 0xd2, 0x53, 0xc8,
      0x23, 0x38, 0x19, 0x83, 0xe0, 0x88, 0xc1, 0xe0, 0xa1, 0x9f, 0x11, 0x51,
      0x07, 0x33, 0xaa, 0x07, 0xf6, 0x16, 0x02, 0xda, 0x8d, 0xbd, 0x20, 0x6a,
      0x62, 0xb6, 0xa9, 0xe1, 0x90, 0xad, 0x8d, 0xd9, 0xc1, 0x91, 0x52, 0x6d,
      0x26, 0xd7, 0x27, 0xfd, 0x15, 0x95, 0x46, 0x54, 0x71, 0xd2, 0xf0, 0x55,
      0xb1, 0x41, 0x74, 0xe7, 0x02, 0x1c, 0x84, 0x0a, 0x62, 0x17, 0x22, 0xac,
      0x0b, 0x76, 0xc3, 0xf0, 0xe4, 0x1e, 0x24, 0x9c, 0x8a, 0xa1, 0x06, 0x9c,
      0x52, 0x55, 0x38, 0x8d, 0xbb, 0xc8, 0xa9, 0xd6, 0xbd, 0xcb, 0x3b, 0x8a,
      0x2c, 0x19, 0x08, 0xf5, 0x40, 0x74, 0x54, 0x51, 0x70, 0x6d, 0x08, 0xbe,
      0xc0, 0x90, 0x2d, 0xbe, 0xca, 0x2e, 0xe9, 0x89, 0x0c, 0x2a, 0x24, 0x1d,
      0xde, 0x47, 0xc3, 0x62, 0x66, 0x16, 0xb4, 0x3c, 0x7d, 0x39, 0xf6, 0x3f,
      0x98, 0x9c, 0x62, 0x77, 0xc3, 0x0f, 0x70, 0xf9, 0x7b, 0xad, 0x09, 0x89,
      0xcb, 0x64, 0xac, 0xbe, 0xb4, 0xa4, 0x03, 0xd2, 0xa4, 0xb6, 0x57, 0x06,
      0x33, 0x89, 0x86, 0xc9, 0xa0, 0x79, 0xb1, 0x8b, 0x7d, 0x43, 0xf1, 0x19,
      0xa5, 0x14, 0xaa, 0x59, 0x06, 0x6c, 0xdc, 0x39, 0x11, 0xcc, 0x3c, 0xc4,
      0x8b, 0xec, 0x56, 0x45, 0x92, 0x7b, 0x25, 0x63, 0xa6, 0x51, 0x37, 0xb5,
      0x02, 0x56, 0xb0, 0xf3, 0xda, 0x25, 0xd5, 0xe4, 0x10, 0x92, 0x00, 0x9d,
      0x0c, 0x7b, 0xc5, 0x39, 0x8b, 0xb7, 0xcf, 0x15, 0x3c, 0xcb, 0x85, 0x4e,
      0xaf, 0xb2, 0x14, 0x40, 0x93, 0x73, 0x9b, 0x7a, 0x7c, 0xcd, 0x03, 0xc8,
      0xcb, 0xa8, 0xb2, 0xc2, 0x89, 0x2e, 0x57, 0xf9, 0x9c, 0xbd, 0x03, 0x23,
      0x4f, 0x09, 0x97, 0x52, 0xd7, 0xb1, 0x05, 0xec, 0x45, 0xc1, 0xa1, 0x53,
      0xbb, 0x30, 0x0d, 0xb7, 0x75, 0x3b, 0x3e, 0xb3, 0x0b, 0x76, 0x59, 0x56,
      0xe8, 0x37, 0xbb, 0xaa, 0x8c, 0x51, 0xab, 0xd8, 0xbe, 0x54, 0xeb, 0x64,
      0xab, 0x52, 0x29, 0x61, 0x1b, 0x4b, 0x8c, 0xee, 0xd4, 0xd7, 0x7a, 0xe3,
      0xb3, 0x93, 0x35, 0xa4, 0xa4, 0x7a, 0x68, 0xaa, 0x38, 0x50, 0x42, 0x50,
      0x82, 0x98, 0x74, 0xd0, 0xec, 0xc6, 0xdf, 0xf4, 0x98, 0x6d, 0x1a, 0xe5,
      0x8c, 0xfe, 0x53, 0x7a, 0xfd, 0x2e, 0x02, 0xb2, 0xc4, 0x4b, 0xa0, 0x88,
      0xe7, 0xf3, 0xdb, 0xd2, 0x1f, 0x9d, 0x1c, 0x02, 0x37, 0x17, 0x85, 0x74,
      0x6e, 0x66, 0x72, 0x63, 0xb5, 0x0d, 0x48, 0xc4, 0x85, 0x5a, 0xe2, 0x49,
      0xab, 0x4c, 0x80, 0xc9, 0xb4, 0x93, 0xd2, 0x2b, 0xf9, 0x59, 0xcd, 0x6e,
      0x85, 0xfd, 0x28, 0xc0, 0xcd, 0x9f, 0xae, 0xce, 0xd7, 0x0f, 0xa7, 0x90};
  static const uint8_t kDecapCiphertext[MLKEM512_CIPHERTEXT_BYTES] = {
      0xaf, 0x4a, 0x00, 0x68, 0xc3, 0x73, 0x44, 0xd7, 0xe1, 0x06, 0xa9, 0xcd,
      0x39, 0x77, 0x9c, 0xf8, 0xc7, 0x67, 0xd5, 0xb8, 0x1c, 0xb3, 0x44, 0x34,
      0x40, 0xda, 0xde, 0x14, 0xc2, 0xc4, 0x83, 0x27, 0xba, 0x34, 0x28, 0x45,
      0xdc, 0x8c, 0x58, 0x8b, 0xd2, 0x5d, 0x9b, 0x55, 0xe2, 0x7f, 0x33, 0x13,
      0x42, 0xe2, 0x46, 0xbd, 0xd5, 0x6d, 0x18, 0x3a, 0xc1, 0xe0, 0x78, 0x84,
      0x20, 0x57, 0x3f, 0x37, 0xe2, 0x29, 0xd4, 0x49, 0x0b, 0xf4, 0x17, 0xb3,
      0xfe, 0xda, 0x42, 0x20, 0xd7, 0x76, 0xa5, 0x29, 0xd9, 0x6f, 0x7c, 0xdd,
      0xfa, 0x1e, 0xce, 0x84, 0x81, 0x51, 0x56, 0x93, 0x54, 0x8b, 0xb7, 0x6f,
      0x5f, 0xb7, 0xda, 0xa6, 0x5d, 0xfb, 0x13, 0xbf, 0x84, 0xdd, 0x1c, 0xa4,
      0xe0, 0xef, 0x7e, 0x49, 0xe0, 0xd1, 0xe8, 0xa3, 0x91, 0x8e, 0x3c, 0xe9,
      0xa7, 0x84, 0xb3, 0x7d, 0xc2, 0xa4, 0xd2, 0xd2, 0xd3, 0x11, 0xf7, 0x06,
      0xe5, 0x05, 0xa1, 0xd9, 0x3e, 0x55, 0x23, 0x69, 0xe5, 0x10, 0xa0, 0xd2,
      0xca, 0x34, 0x7a, 0xec, 0xf8, 0x84, 0x62, 0xe2, 0xfd, 0xe0, 0xa6, 0xc9,
      0xfe, 0xb4, 0x95, 0x06, 0xa4, 0xfc, 0xeb, 0xdf, 0x98, 0x77, 0xab, 0xd1,
      0x8c, 0xa8, 0x1f, 0xea, 0x64, 0x11, 0x5f, 0xb1, 0x79, 0xe3, 0x35, 0xe0,
      0x06, 0x90, 0x5b, 0x72, 0x2a, 0x58, 0x88, 0xf7, 0x3d, 0x70, 0xed, 0x77,
      0x26, 0xe0, 0x72, 0x2f, 0x55, 0x24, 0x0e, 0xe2, 0xc2, 0x05, 0xd3, 0xe2,
      0xb2, 0xc5, 0x37, 0xa5, 0x2d, 0xf0, 0x2b, 0xb6, 0x93, 0xf4, 0xd7, 0x21,
      0x01, 0x17, 0x35, 0xce, 0x11, 0xb8, 0x07, 0x43, 0x76, 0x16, 0xff, 0x21,
      0x3d, 0x71, 0xc3, 0xa7, 0x32, 0x48, 0x4c, 0x5b, 0xc3, 0xda, 0xed, 0xea,
      0xf4, 0xfb, 0xe0, 0x2f, 0xc5, 0xe1, 0xb9, 0xb0, 0xcc, 0x82, 0xf6, 0xa6,
      0xe5, 0x1c, 0x0b, 0xca, 0x4f, 0x6d, 0x66, 0xfe, 0x19, 0x82, 0xcf, 0xdc,
      0x48, 0x04, 0xfb, 0xc8, 0xa7, 0x67, 0xbc, 0x44, 0x2e, 0x85, 0x34, 0xf6,
      0x3a, 0xf3, 0xb0, 0xd0, 0x57, 0xbc, 0x6c, 0x95, 0x0b, 0x5e, 0x0a, 0x07,
      0xae, 0x9f, 0x03, 0x85, 0x51, 0x0a, 0xe7, 0x8c, 0x11, 0xa2, 0xa1, 0x1e,
      0xbb, 0x84, 0x9c, 0x13, 0xe1, 0x77, 0xd9, 0x82, 0xdb, 0x7c, 0xd4, 0x7d,
      0x55, 0xf8, 0x08, 0x6c, 0x14, 0xe5, 0xc1, 0xc3, 0xe6, 0xc8, 0x20, 0x49,
      0x41, 0xbc, 0xa3, 0x79, 0x16, 0xfe, 0x20, 0x15, 0xc2, 0x99, 0x0d, 0x00,
      0xbd, 0x98, 0x64, 0x1c, 0xe6, 0x15, 0x04, 0x53, 0x86, 0x81, 0x29, 0x39,
      0xd0, 0xbc, 0xb7, 0x42, 0x77, 0xfc, 0xb7, 0x18, 0x34, 0xed, 0x29, 0x7e,
      0xda, 0x87, 0xdb, 0x1d, 0xf9, 0x14, 0x97, 0x85, 0x48, 0x95, 0xf6, 0xcd,
      0x8d, 0x94, 0xcf, 0xcb, 0x41, 0xed, 0xc1, 0xbe, 0x15, 0x1d, 0xf9, 0x14,
      0x73, 0xe3, 0x7e, 0xba, 0x54, 0x6e, 0x15, 0x62, 0x7a, 0x6d, 0xbd, 0x58,
      0x3d, 0x9b, 0xa0, 0xed, 0x34, 0xee, 0x51, 0x1a, 0x08, 0x31, 0xeb, 0xa1,
      0x35, 0x68, 0x29, 0x75, 0xa2, 0x39, 0xf4, 0x95, 0xe3, 0x09, 0x84, 0x2b,
      0xab, 0xee, 0xf6, 0xf4, 0x0e, 0x7b, 0xb4, 0xd6, 0xcd, 0x45, 0x09, 0x5e,
      0x3f, 0x91, 0xf9, 0xb6, 0x1b, 0x86, 0x35, 0x9c, 0xdd, 0x05, 0xd7, 0x9b,
      0xb7, 0x2f, 0x5e, 0xaa, 0x2e, 0xb9, 0x85, 0x4e, 0x21, 0xa0, 0x19, 0x4c,
      0x46, 0x8d, 0x9f, 0xe7, 0xe8, 0x9f, 0x3c, 0x0e, 0x74, 0xf5, 0x70, 0xf8,
      0x8b, 0x5b, 0x50, 0x15, 0xd4, 0xbb, 0x4c, 0x8b, 0xcb, 0x9e, 0xa6, 0x43,
      0xed, 0xce, 0x57, 0xba, 0x72, 0x11, 0x4c, 0xf5, 0x86, 0xcd, 0x6f, 0xb8,
      0xa4, 0xea, 0x5b, 0x05, 0x04, 0x47, 0xe0, 0x6b, 0xb7, 0x13, 0x89, 0x1e,
      0x13, 0x46, 0xcd, 0x22, 0x76, 0xf2, 0xb8, 0x63, 0x96, 0xe4, 0xcf, 0x42,
      0x41, 0x27, 0x6f, 0xec, 0xce, 0xec, 0x79, 0xe7, 0x38, 0xdc, 0xb4, 0x74,
      0x5e, 0x2d, 0xcd, 0x7d, 0x68, 0x98, 0x86, 0x25, 0x29, 0xf2, 0x4c, 0x38,
      0x79, 0xeb, 0xee, 0x42, 0xd8, 0x79, 0x67, 0x92, 0xdb, 0x7a, 0x7c, 0x86,
      0xd2, 0x8a, 0xb1, 0x8f, 0x26, 0xb9, 0x9b, 0x6c, 0xfc, 0x82, 0xf3, 0x9f,
      0xe7, 0xc0, 0xe5, 0x70, 0x73, 0x6d, 0x33, 0xa1, 0x77, 0x0b, 0xe8, 0x20,
      0x03, 0x66, 0x6e, 0x4f, 0x69, 0xb1, 0x18, 0x6b, 0x3d, 0x55, 0x83, 0x8a,
      0x85, 0x1b, 0x87, 0xbe, 0xb6, 0xe7, 0x22, 0x01, 0xfe, 0x3f, 0x8a, 0x25,
      0xf4, 0x86, 0x7e, 0xc6, 0x3e, 0xe8, 0xfe, 0xab, 0x05, 0xff, 0xf6, 0x08,
      0x39, 0x0f, 0xa4, 0x8d, 0x18, 0x60, 0x69, 0x80, 0x9c, 0xa6, 0x42, 0xa0,
      0xe7, 0x44, 0x8f, 0x3a, 0xd0, 0xc2, 0xd9, 0x11, 0xb3, 0x3c, 0x17, 0x5c,
      0x66, 0xa2, 0xea, 0xdf, 0xba, 0x79, 0xe2, 0xbc, 0x14, 0x0c, 0x03, 0xbf,
      0x96, 0x60, 0xec, 0x28, 0xb4, 0x1d, 0x07, 0x1c, 0x0f, 0xfb, 0xc7, 0x3c,
      0xe2, 0x58, 0x5f, 0x3d, 0xe5, 0xe1, 0x98, 0xf0, 0x25, 0xe7, 0xaf, 0xd4,
      0xbd, 0x12, 0x67, 0xe1, 0xf5, 0x4e, 0x06, 0x34, 0x96, 0xfe, 0xf3, 0xe9,
      0x66, 0xc5, 0xb7, 0x1a, 0x56, 0x5b, 0xf9, 0xbc, 0xfc, 0x70, 0x34, 0x21,
      0x6b, 0x5b, 0xf6, 0x21, 0xa2, 0xc2, 0x17, 0x9b, 0x52, 0xe9, 0xc9, 0x92,
      0x87, 0x1b, 0xd6, 0x61, 0x87, 0x39, 0x6d, 0xc6, 0x9f, 0x23, 0x94, 0xab,
      0x20, 0xc4, 0x17, 0x00, 0x2c, 0x83, 0x0e, 0x70, 0x5a, 0x08, 0xac, 0x20,
      0x5c, 0x61, 0x38, 0x04, 0xaf, 0x15, 0xac, 0x2c, 0x65, 0x55, 0x3b, 0x69,
      0xd5, 0xd0, 0x12, 0xcd, 0x90, 0xe5, 0x0a, 0x40, 0xfd, 0xdb, 0x8e, 0xda,
      0xa8, 0x33, 0x13, 0xb0, 0x08, 0x9b, 0x5d, 0x69, 0x97, 0x19, 0x3f, 0x21};
  static const uint8_t kDecapSharedSecret[MLKEM512_SHARED_SECRET_LEN] = {
      0xbe, 0x41, 0xc0, 0x7a, 0xd0, 0x59, 0x67, 0xab, 0xf7, 0x00, 0xb8,
      0xb2, 0xe6, 0x97, 0x5e, 0x3c, 0x1e, 0x87, 0x50, 0x3c, 0xa0, 0xb5,
      0x83, 0xe1, 0xd7, 0x30, 0x46, 0x23, 0x0e, 0x26, 0x5e, 0x5a};
  static const uint8_t kDecapCiphertextRejection[MLKEM512_CIPHERTEXT_BYTES] = {
      0x18, 0x1a, 0xf9, 0xb9, 0xf1, 0x63, 0xa3, 0x04, 0x9d, 0x97, 0xd8, 0x19,
      0x4b, 0x5c, 0x26, 0x35, 0x5f, 0xc9, 0xf9, 0xdc, 0xbf, 0x12, 0x05, 0x76,
      0xab, 0xe4, 0x4d, 0x71, 0xa0, 0xd9, 0x91, 0x06, 0xd5, 0x57, 0x5d, 0xb4,
      0xd1, 0x11, 0x89, 0xc0, 0xeb, 0xbd, 0x3b, 0x30, 0x9e, 0x2b, 0x7f, 0xb8,
      0x6b, 0xb2, 0x51, 0xfe, 0xbd, 0x0c, 0xb5, 0x1b, 0x34, 0x72, 0x89, 0x52,
      0xde, 0x93, 0xad, 0xf1, 0x52, 0xb9, 0xb2, 0x39, 0x27, 0x79, 0x04, 0x93,
      0x93, 0x74, 0xcd, 0x63, 0x89, 0xc9, 0x9e, 0xd2, 0x6d, 0xc9, 0xb7, 0xd7,
      0xa4, 0x82, 0x82, 0xe9, 0x1c, 0x8f, 0x51, 0x75, 0xca, 0xd4, 0xcb, 0x0c,
      0x49, 0xfb, 0x5b, 0x1d, 0x5b, 0x32, 0xfa, 0x89, 0x04, 0xd4, 0x84, 0x49,
      0xe1, 0x0a, 0x18, 0xa3, 0x99, 0x60, 0x25, 0xb1, 0x4e, 0xc7, 0x13, 0x7c,
      0x10, 0x49, 0xe8, 0x6f, 0xa0, 0xd1, 0x25, 0xb2, 0xb5, 0xe1, 0xa6, 0xd1,
      0xe8, 0xfa, 0x8d, 0x6d, 0xc1, 0xdb, 0x67, 0x25, 0x3c, 0xba, 0x44, 0xb2,
      0xa6, 0xb5, 0xf1, 0xa5, 0xe8, 0x7f, 0x05, 0x29, 0x1a, 0x03, 0xdb, 0xd2,
      0x75, 0x70, 0x70, 0x2c, 0xb7, 0x0c, 0x46, 0xe3, 0xe2, 0xd2, 0x60, 0x98,
      0x40, 0x65, 0xc9, 0x69, 0x18, 0x84, 0xd8, 0x86, 0x62, 0x32, 0x68, 0x8b,
      0x77, 0x43, 0xf2, 0x7f, 0x46, 0x92, 0xc9, 0x3e, 0x94, 0xe1, 0x05, 0x67,
      0xf6, 0x22, 0x6c, 0xef, 0xe8, 0x20, 0x30, 0xfd, 0x0a, 0x83, 0x54, 0x7a,
      0x8c, 0x0a, 0x1f, 0xf7, 0xca, 0x81, 0x26, 0x08, 0x76, 0x05, 0x68, 0xec,
      0x69, 0x28, 0xe0, 0xe6, 0x30, 0xd7, 0xe9, 0x37, 0xa9, 0x66, 0x99, 0x14,
      0x7e, 0x34, 0xbe, 0x04, 0xbb, 0xb7, 0xeb, 0xb5, 0x55, 0x07, 0xeb, 0x51,
      0x61, 0xd0, 0xe9, 0xdb, 0x13, 0xbe, 0x22, 0x15, 0xb7, 0x21, 0xe5, 0x47,
      0x41, 0xc7, 0x53, 0xe5, 0x0a, 0x50, 0xb6, 0x9c, 0x15, 0x97, 0x0c, 0xb1,
      0x2c, 0x92, 0xc4, 0x05, 0xa5, 0x08, 0xae, 0xa3, 0xce, 0x17, 0xb3, 0xa8,
      0x4e, 0x25, 0x36, 0x14, 0xfa, 0x67, 0x8b, 0x06, 0xf3, 0x0e, 0x19, 0x7c,
      0xff, 0x1f, 0x41, 0x24, 0x38, 0xe8, 0x00, 0x46, 0x70, 0xd8, 0x88, 0xde,
      0x0d, 0xf2, 0xc1, 0x13, 0x9c, 0xc2, 0x93, 0xee, 0x3a, 0xb9, 0x2f, 0xa9,
      0x8e, 0xdd, 0x4a, 0xbe, 0x6b, 0xe6, 0xca, 0x7d, 0xcd, 0x72, 0xe7, 0x27,
      0x30, 0xad, 0x9c, 0xaf, 0xc0, 0x93, 0x59, 0x4d, 0x57, 0x42, 0xdd, 0xf5,
      0x15, 0x4b, 0x63, 0x17, 0x34, 0x29, 0xdc, 0x5e, 0xf0, 0x33, 0x99, 0xc9,
      0xd1, 0x39, 0x51, 0x08, 0xc8, 0x94, 0xb2, 0x43, 0x59, 0xd0, 0xb1, 0xff,
      0x44, 0x9f, 0xb8, 0xde, 0xb0, 0xcb, 0x58, 0x45, 0xc7, 0x14, 0x75, 0xc5,
      0xc9, 0xa2, 0x4c, 0x7b, 0x77, 0x5e, 0x07, 0x99, 0x21, 0x49, 0xe4, 0xe5,
      0x7c, 0x79, 0x17, 0xc0, 0x1a, 0xed, 0x40, 0x29, 0x78, 0xa0, 0xf0, 0xa7,
      0xb5, 0x90, 0xca, 0xcf, 0xbf, 0x21, 0xa6, 0xcf, 0x59, 0x0a, 0x7d, 0x0d,
      0x0a, 0xd7, 0xa4, 0x55, 0x98, 0x2b, 0xe3, 0x2f, 0x94, 0xec, 0xb5, 0x42,
      0x9c, 0x24, 0x16, 0x2e, 0x13, 0x0d, 0x92, 0x13, 0x04, 0x8a, 0x65, 0x63,
      0x39, 0x0f, 0x63, 0xfe, 0x8f, 0xfb, 0x1a, 0xaf, 0x2d, 0x51, 0xdf, 0xee,
      0x47, 0x76, 0x06, 0xa3, 0x85, 0xf9, 0xa1, 0x6a, 0x00, 0x98, 0x06, 0x7f,
      0xf7, 0x89, 0x10, 0x2a, 0xac, 0xd5, 0x99, 0xee, 0x98, 0x79, 0xef, 0x4d,
      0xb7, 0xeb, 0xc0, 0xf1, 0xcc, 0x1e, 0x7d, 0xe5, 0xd2, 0x5b, 0x67, 0x65,
      0x15, 0x99, 0x4a, 0x00, 0x96, 0x91, 0xf4, 0xe2, 0xaf, 0x31, 0x1f, 0xf7,
      0x3d, 0x2b, 0xea, 0xcb, 0xdb, 0x2b, 0x28, 0x72, 0xd2, 0x7e, 0xe0, 0x2f,
      0x70, 0x50, 0xe8, 0x11, 0x98, 0xf4, 0x7f, 0xe7, 0x5e, 0x0a, 0xa0, 0x42,
      0x81, 0xf0, 0x9e, 0x5d, 0x5d, 0x19, 0x8b, 0x08, 0xbe, 0xc5, 0x61, 0x32,
      0x8e, 0xcb, 0x8d, 0x94, 0x32, 0x4e, 0xa6, 0xe1, 0xad, 0x9f, 0xad, 0x7a,
      0x8f, 0xbf, 0x4f, 0x4b, 0x1b, 0x91, 0xcc, 0x15, 0x0b, 0x0f, 0x6c, 0x86,
      0x40, 0xaf, 0x2e, 0x11, 0x2b, 0x6e, 0xeb, 0xbe, 0x8e, 0xa0, 0x33, 0x81,
      0x20, 0x09, 0x3f, 0x58, 0xbb, 0xe8, 0xae, 0xae, 0x7d, 0xf7, 0x28, 0x8f,
      0xef, 0x1d, 0xe8, 0xab, 0x32, 0x5a, 0x92, 0xb4, 0x2c, 0x0a, 0xa8, 0xed,
      0x3a, 0x62, 0xed, 0x9a, 0x86, 0xe9, 0x46, 0xff, 0xdd, 0xdc, 0x87, 0xaf,
      0x80, 0x15, 0xb4, 0xee, 0x64, 0xdf, 0x5f, 0xf4, 0x8e, 0x94, 0x11, 0x54,
      0x5a, 0x00, 0x78, 0xcf, 0x16, 0xd1, 0x49, 0xfb, 0xf3, 0x19, 0x38, 0xae,
      0xb1, 0x10, 0x92, 0x72, 0x7a, 0x45, 0x2c, 0xa8, 0x4b, 0xab, 0xb7, 0x03,
      0x89, 0x39, 0xa8, 0xdb, 0x3f, 0xa1, 0x04, 0x48, 0x17, 0xec, 0x03, 0x83,
      0x2f, 0x6c, 0x0a, 0x76, 0xe2, 0xb6, 0x16, 0xbb, 0x40, 0xb2, 0xe7, 0x62,
      0x11, 0xb9, 0x1f, 0xf9, 0x69, 0xdc, 0xe0, 0xad, 0x06, 0x9f, 0xb7, 0x29,
      0x47, 0x1e, 0x95, 0xba, 0xce, 0x35, 0x8e, 0x1b, 0xbf, 0xf2, 0xbd, 0xab,
      0xd7, 0x42, 0x98, 0x57, 0xdb, 0xab, 0x72, 0x21, 0x45, 0x47, 0xaf, 0x1a,
      0xbc, 0xdb, 0x08, 0x89, 0x6e, 0xe7, 0x7f, 0x13, 0xf2, 0xd7, 0x5b, 0x17,
      0x17, 0x44, 0x6f, 0xca, 0x7d, 0xf3, 0x2e, 0xfe, 0x1c, 0x2d, 0x09, 0xdc,
      0xfd, 0x5b, 0xfa, 0xff, 0xd1, 0x9e, 0xde, 0x7e, 0x50, 0x2b, 0x63, 0x64,
      0xab, 0xe3, 0x2e, 0x84, 0x49, 0x99, 0xb4, 0x47, 0x7c, 0x99, 0x8a, 0x9f,
      0xb3, 0xc9, 0xba, 0xbb, 0xe8, 0x3c, 0x6e, 0xc6, 0x13, 0x74, 0x0c, 0x2b,
      0x04, 0x75, 0xec, 0xb7, 0x32, 0xde, 0x51, 0x64, 0x38, 0x68, 0xeb, 0xb7};
  static const uint8_t kDecapSharedSecretRejection[MLKEM512_SHARED_SECRET_LEN] = {
      0x98, 0xed, 0x60, 0x0f, 0xfd, 0x9e, 0x01, 0x9f, 0x35, 0x0e, 0x0a,
      0x15, 0xd4, 0x69, 0x5b, 0xa0, 0x96, 0xce, 0x2b, 0x32, 0xc3, 0x75,
      0x24, 0x4f, 0x79, 0xa5, 0x74, 0xda, 0x06, 0xb4, 0xb1, 0xbd};

  if (ml_kem_512_decapsulate_no_self_test(shared_secret, &shared_secret_len, kDecapCiphertext,
                                          kDecapDK) ||
      !check_test(kDecapSharedSecret, shared_secret, sizeof(kDecapSharedSecret),
                  "ML-KEM decapsulate non-rejection") ||
      ml_kem_512_decapsulate_no_self_test(
          shared_secret, &shared_secret_len, kDecapCiphertextRejection, kDecapDK) ||
      !check_test(kDecapSharedSecretRejection, shared_secret,
                  sizeof(kDecapSharedSecretRejection),
                  "ML-KEM decapsulate implicit rejection")) {
    goto err;
  }

  ret = 1;
err:
  return ret;
}

static OPENSSL_NOINLINE int boringssl_self_test_ml_dsa(void) {
  int ret = 0;

  // Examples kMLDSAKeyGenSeed, kMLDSAKeyGenPublicKey, kMLDSAKeyGenPrivateKey from
  // https://github.com/usnistgov/ACVP-Server/blob/master/gen-val/json-files/ML-DSA-keyGen-FIPS204/prompt.json#L15
  static const uint8_t kMLDSAKeyGenSeed[MLDSA44_KEYGEN_SEED_BYTES] = {
    0x4B, 0xE7, 0xA0, 0x1A, 0x99, 0xA5, 0xE5, 0xBC, 0xFE, 0x3C, 0x06,
    0x78, 0x5D, 0x8E, 0x4E, 0xC6, 0x64, 0x08, 0x22, 0x27, 0xD8, 0x67,
    0x04, 0xE9, 0xE4, 0x48, 0x62, 0x62, 0x3A, 0x05, 0xC8, 0xB3
};
  // https://github.com/usnistgov/ACVP-Server/blob/master/gen-val/json-files/ML-DSA-keyGen-FIPS204/expectedResults.json#L13
  static const uint8_t kMLDSAKeyGenPublicKey[MLDSA44_PUBLIC_KEY_BYTES] = {
    0xad, 0xb0, 0xb3, 0x34, 0x64, 0x81, 0x60, 0x91, 0xf2, 0xa9, 0x59, 0x77,
    0xc6, 0x7f, 0x08, 0x5f, 0xdc, 0x24, 0xb3, 0x78, 0x54, 0xd4, 0xdb, 0x0a,
    0x57, 0x7a, 0xe9, 0x40, 0x1e, 0x40, 0x81, 0x48, 0xd8, 0x91, 0x7d, 0x21,
    0xaa, 0x49, 0x6b, 0xb1, 0x3c, 0x60, 0xb2, 0x95, 0xcb, 0x0a, 0x94, 0x23,
    0x22, 0xa0, 0x39, 0x97, 0x98, 0x28, 0xf4, 0x6a, 0x00, 0xa4, 0xe7, 0xde,
    0xeb, 0xa2, 0xbc, 0x06, 0x52, 0x16, 0xd9, 0x7d, 0x93, 0xfd, 0x6b, 0xc1,
    0xcd, 0x87, 0xdd, 0x38, 0x3f, 0x18, 0x96, 0x3c, 0xe5, 0xcf, 0xdd, 0x71,
    0xb5, 0x1d, 0xf0, 0x80, 0x86, 0xf1, 0x41, 0x5e, 0xa5, 0x12, 0xaf, 0xae,
    0x38, 0xce, 0x16, 0xe5, 0x17, 0x14, 0x2d, 0x3a, 0xe3, 0xed, 0xd3, 0x94,
    0x96, 0x70, 0x56, 0x79, 0x46, 0x6a, 0xa9, 0xcc, 0x3f, 0xfa, 0x14, 0x07,
    0x1d, 0xc5, 0x62, 0x44, 0xfd, 0x1c, 0xa4, 0xec, 0xa5, 0xa1, 0x40, 0x5d,
    0x5f, 0x1f, 0xc1, 0x2f, 0x3c, 0x5e, 0xca, 0xcc, 0x4b, 0x9b, 0x02, 0x1b,
    0xca, 0x72, 0x01, 0xc4, 0xea, 0x0d, 0x67, 0x00, 0x99, 0xb7, 0xbb, 0xd5,
    0x07, 0x07, 0x16, 0x4f, 0x3f, 0x39, 0x55, 0x69, 0x99, 0x0b, 0xc6, 0xe7,
    0xe3, 0x04, 0x24, 0xb3, 0x7a, 0xb0, 0x62, 0x1d, 0xb2, 0x95, 0x4e, 0x59,
    0x96, 0xb1, 0x56, 0x70, 0xdc, 0x4c, 0xe0, 0x3a, 0x14, 0x97, 0xe4, 0x04,
    0x65, 0xa4, 0xc3, 0x26, 0xf1, 0xe0, 0x65, 0xe2, 0xbf, 0xdf, 0x1a, 0x11,
    0xad, 0xba, 0x11, 0x93, 0x52, 0x27, 0x88, 0xee, 0xa0, 0x06, 0xb8, 0xb1,
    0xab, 0x29, 0x6a, 0xe1, 0xb1, 0x74, 0xc9, 0x56, 0xae, 0x95, 0x55, 0x18,
    0x02, 0x74, 0x29, 0x4c, 0xdc, 0x19, 0xcb, 0x4e, 0xef, 0xf2, 0xaa, 0xe1,
    0x26, 0x0a, 0xb1, 0xe0, 0x3c, 0x7c, 0xcf, 0x8d, 0x8a, 0x6d, 0xf4, 0x2b,
    0xa6, 0x7e, 0xea, 0x32, 0x28, 0xe9, 0xe8, 0x31, 0x98, 0x21, 0xfb, 0x06,
    0x53, 0x8c, 0x71, 0x06, 0x1b, 0x5e, 0x1d, 0x79, 0x27, 0x48, 0x16, 0x15,
    0x2a, 0x4f, 0x18, 0x8f, 0xe3, 0x97, 0x43, 0xd8, 0x6a, 0xde, 0xd0, 0xd4,
    0xe9, 0x02, 0x0d, 0x5d, 0xba, 0x05, 0x96, 0x52, 0x1e, 0x8e, 0x6a, 0x2d,
    0x50, 0x81, 0x1e, 0x6e, 0x98, 0x9a, 0xe0, 0xd3, 0x5b, 0x4d, 0x40, 0x0a,
    0x4a, 0x4c, 0x5c, 0x88, 0x31, 0x32, 0xf6, 0xf3, 0xdb, 0xd3, 0xf6, 0x51,
    0x7e, 0xcb, 0x27, 0x74, 0x59, 0x24, 0x89, 0x7f, 0xc1, 0x14, 0x1c, 0xbf,
    0xec, 0x4c, 0xdd, 0x4e, 0x87, 0x1e, 0x2d, 0x30, 0x2a, 0xe9, 0x56, 0x43,
    0xd3, 0x82, 0xca, 0xee, 0xf0, 0xde, 0xd3, 0xf1, 0xf0, 0x64, 0x67, 0x68,
    0xab, 0x18, 0x29, 0x7e, 0xee, 0xc4, 0x79, 0x6c, 0xf8, 0xa5, 0x96, 0x4f,
    0x21, 0xd5, 0x5b, 0x7b, 0x3c, 0x11, 0x17, 0x84, 0x3b, 0x1f, 0xae, 0x74,
    0xd8, 0x49, 0x63, 0x02, 0x18, 0x1e, 0xe6, 0x82, 0xd0, 0x76, 0xe7, 0x85,
    0x8b, 0x13, 0xbe, 0x61, 0x2e, 0xdb, 0xb2, 0x63, 0x5f, 0xfd, 0x51, 0xf3,
    0x96, 0x41, 0x40, 0x89, 0x0f, 0x10, 0x15, 0xc3, 0x2e, 0x3f, 0xfb, 0xdf,
    0xb0, 0xc7, 0x6e, 0x16, 0x31, 0x69, 0x12, 0x2b, 0x35, 0xf9, 0x1e, 0x1e,
    0x29, 0xc5, 0xb9, 0xc9, 0x2d, 0xc4, 0x0d, 0xc7, 0x38, 0xfb, 0x7a, 0xa4,
    0x49, 0x32, 0xb9, 0x37, 0xca, 0xfa, 0x03, 0xed, 0xd7, 0xe7, 0x11, 0xee,
    0xb1, 0x9a, 0x82, 0x80, 0x34, 0xec, 0x39, 0x81, 0xa2, 0x95, 0x03, 0x83,
    0xf6, 0xc6, 0xf7, 0x34, 0xbc, 0xff, 0xe6, 0x8d, 0x39, 0xb7, 0x1d, 0x7f,
    0x55, 0xfe, 0x35, 0xe4, 0xe2, 0x29, 0x9c, 0x0a, 0x9e, 0xdc, 0x40, 0xb6,
    0x6d, 0xb4, 0x1f, 0x7b, 0xca, 0x34, 0x11, 0x62, 0xde, 0x03, 0x50, 0xb5,
    0x4e, 0x72, 0x0b, 0x9a, 0x1a, 0xae, 0x0d, 0x99, 0xdb, 0x0a, 0x58, 0xd0,
    0x52, 0xb6, 0x9e, 0xe1, 0xa3, 0x02, 0x27, 0x43, 0x9b, 0xdf, 0xd8, 0xc1,
    0xd4, 0xfd, 0xbe, 0xb4, 0x69, 0x1f, 0x9c, 0xbd, 0x5b, 0xb6, 0x2b, 0x47,
    0xcb, 0x0e, 0x78, 0x2b, 0xf3, 0x10, 0xe2, 0x78, 0x7b, 0x3c, 0x25, 0x5b,
    0x67, 0x70, 0x8c, 0x5c, 0x7c, 0x75, 0x1f, 0x48, 0xbb, 0x79, 0x70, 0x69,
    0xc2, 0x90, 0x49, 0x3b, 0xb8, 0x56, 0x10, 0x72, 0xea, 0x17, 0xb1, 0x45,
    0xd2, 0x6b, 0x07, 0x20, 0x9e, 0x29, 0x2f, 0xa5, 0xed, 0x76, 0xca, 0xa3,
    0x1b, 0x2d, 0x7d, 0xb4, 0x5d, 0xdb, 0x09, 0x6d, 0x44, 0xcd, 0xa1, 0x3e,
    0xcd, 0x03, 0xe3, 0x54, 0x7b, 0x52, 0x1d, 0xc3, 0x0c, 0xe2, 0x7d, 0x66,
    0xa2, 0x88, 0xb4, 0x05, 0x97, 0x01, 0x59, 0x8d, 0x0a, 0xd3, 0x8a, 0x33,
    0x13, 0x0e, 0xbc, 0x49, 0x2b, 0xa0, 0x25, 0xd9, 0x3c, 0x58, 0xe8, 0xbd,
    0x11, 0x87, 0x3b, 0x37, 0xad, 0xb1, 0xb3, 0x90, 0x66, 0xfe, 0x83, 0xea,
    0xd6, 0x45, 0xae, 0x6b, 0x04, 0xf7, 0x58, 0x47, 0x0c, 0x16, 0x3a, 0xd9,
    0x6a, 0xed, 0xa9, 0x20, 0x84, 0x20, 0xc6, 0xe9, 0xc2, 0x74, 0xef, 0xbb,
    0x85, 0x35, 0x63, 0xe1, 0xcb, 0x36, 0x60, 0xd2, 0x75, 0xf8, 0x5b, 0xb5,
    0x51, 0xa2, 0x88, 0x8a, 0x1e, 0xf0, 0x5a, 0xaf, 0x32, 0x1f, 0x65, 0xd4,
    0xeb, 0x15, 0xda, 0xdc, 0x93, 0xc7, 0x6a, 0x4b, 0x52, 0xd0, 0xc3, 0x4f,
    0x55, 0x65, 0x92, 0x82, 0x49, 0x68, 0x20, 0x0e, 0xbd, 0xe6, 0x52, 0x74,
    0x07, 0x27, 0xe0, 0xcb, 0x0a, 0x6d, 0xe0, 0x28, 0x0f, 0xcd, 0x82, 0xcd,
    0x49, 0x6a, 0x1f, 0x5b, 0xaf, 0x95, 0x65, 0x7b, 0xa3, 0xc1, 0x09, 0x1f,
    0xcc, 0x36, 0xbf, 0x7e, 0xbe, 0x77, 0xfc, 0x34, 0x88, 0x9e, 0xa4, 0x38,
    0xa0, 0x7e, 0x92, 0xac, 0xea, 0xe4, 0x53, 0xc8, 0xee, 0xd9, 0x3d, 0x9a,
    0x70, 0xe8, 0x2c, 0xa3, 0xb8, 0xd0, 0x6e, 0xc9, 0xea, 0xdb, 0x5e, 0xaf,
    0x8b, 0x49, 0x1b, 0x66, 0x5b, 0x39, 0x03, 0xb8, 0xc8, 0x61, 0x3a, 0x24,
    0x07, 0xa6, 0xe5, 0xf3, 0x5f, 0x44, 0x67, 0x8b, 0x19, 0x35, 0x66, 0x3b,
    0xba, 0x75, 0x61, 0x09, 0x6c, 0x2e, 0x3d, 0x50, 0x2e, 0xf0, 0xf7, 0xa2,
    0x45, 0xa5, 0x22, 0x52, 0x28, 0x00, 0x16, 0xc0, 0xb7, 0x4e, 0x3b, 0x05,
    0xd9, 0x3a, 0xea, 0x24, 0xca, 0xeb, 0xa4, 0xd5, 0x95, 0xc3, 0x14, 0x34,
    0xd9, 0x9b, 0x46, 0xf5, 0xb5, 0xef, 0x4d, 0x3a, 0x3a, 0x62, 0x5e, 0xdd,
    0x7e, 0xb9, 0x5d, 0xcd, 0x27, 0xc8, 0x28, 0x94, 0x96, 0x99, 0xc5, 0xe5,
    0x67, 0xd7, 0xa8, 0x51, 0x26, 0x19, 0xe1, 0x4e, 0xcc, 0x6e, 0x1c, 0x47,
    0xca, 0x65, 0x67, 0x21, 0x1d, 0x12, 0x11, 0xfc, 0x6b, 0x94, 0x9b, 0x83,
    0x78, 0x3c, 0x7c, 0xb4, 0x17, 0x62, 0x16, 0x7b, 0xf9, 0x60, 0xe9, 0x35,
    0xfd, 0x85, 0x3e, 0x9e, 0xd3, 0xbb, 0xec, 0xcb, 0x75, 0xe8, 0xdf, 0xb4,
    0xe3, 0x5f, 0x85, 0x45, 0x66, 0xdc, 0xdb, 0x30, 0x0e, 0xb0, 0xd5, 0x1b,
    0x6c, 0x22, 0xe6, 0xeb, 0x1d, 0x85, 0x03, 0x0d, 0xaa, 0x5a, 0x2f, 0xe2,
    0xbb, 0xcc, 0x7c, 0x27, 0x45, 0x79, 0x08, 0x60, 0xad, 0x48, 0x03, 0x3d,
    0xa0, 0x38, 0x42, 0x10, 0x51, 0xd1, 0x61, 0x73, 0x9f, 0x52, 0x4b, 0xb1,
    0xc7, 0x16, 0x25, 0x06, 0x23, 0xa5, 0x7f, 0x64, 0xcb, 0x25, 0x06, 0xdf,
    0x49, 0xc6, 0x65, 0x37, 0xc2, 0xa4, 0x6c, 0xd0, 0xa6, 0x8d, 0x00, 0x17,
    0xbb, 0x33, 0xcb, 0xa4, 0xcf, 0x37, 0x35, 0x12, 0x91, 0xd2, 0x08, 0x03,
    0x37, 0x68, 0xc8, 0x67, 0x99, 0xe0, 0x16, 0xa9, 0x60, 0xe6, 0x82, 0x46,
    0x70, 0xfb, 0xff, 0x10, 0x61, 0x08, 0x7d, 0xa8, 0xfb, 0x74, 0xf9, 0x77,
    0x11, 0x02, 0x39, 0xde, 0xd4, 0x8c, 0xfc, 0x86, 0xa4, 0xfc, 0x4b, 0xae,
    0xf2, 0x20, 0xf0, 0x5a, 0xeb, 0x0c, 0xe3, 0x43, 0x57, 0xd7, 0xd3, 0x1d,
    0x20, 0x38, 0x8f, 0xaa, 0x30, 0x7f, 0x2d, 0x97, 0xf5, 0x93, 0xb9, 0xe2,
    0x87, 0x32, 0x2a, 0x9c, 0x32, 0x9e, 0x04, 0x32, 0x85, 0x22, 0xea, 0x79,
    0xaa, 0xc3, 0x1c, 0xf2, 0x76, 0x5f, 0xb3, 0x34, 0xab, 0xd9, 0xa1, 0x76,
    0x4d, 0xa3, 0x55, 0x09, 0x68, 0xfa, 0x69, 0x3c, 0xb8, 0x48, 0x52, 0x26,
    0x81, 0x70, 0x02, 0x28, 0x81, 0xcc, 0x38, 0x87, 0x45, 0xa5, 0x4a, 0x50,
    0x0f, 0xac, 0xb2, 0xb8, 0xdf, 0x36, 0x38, 0x3e, 0x21, 0x9b, 0x99, 0x17,
    0x7c, 0x3a, 0xf5, 0x1b, 0x1e, 0x33, 0xaf, 0x78, 0x3e, 0xfe, 0xae, 0x04,
    0xa0, 0xf9, 0x84, 0x5d, 0x79, 0xaa, 0x4a, 0x4d, 0xd9, 0xdd, 0xe5, 0xca,
    0xf7, 0xd5, 0x2d, 0x77, 0x9d, 0x59, 0xd1, 0x8f, 0x0e, 0x0e, 0x2b, 0x44,
    0xb6, 0x34, 0xa5, 0xd6, 0x30, 0xbe, 0x81, 0x98, 0xad, 0x8b, 0x20, 0x12,
    0xc5, 0x2a, 0x7b, 0x89, 0xf1, 0x33, 0xf0, 0x9c, 0x3e, 0x8d, 0x21, 0xf0,
    0xad, 0x11, 0x37, 0xaf, 0x09, 0x58, 0x93, 0xb9, 0x66, 0xff, 0xc0, 0x7e,
    0x89, 0x56, 0x58, 0x5f, 0xf5, 0xc1, 0x0b, 0x12, 0x3c, 0x6a, 0x4a, 0x5c,
    0x33, 0x67, 0xd8, 0x98, 0xac, 0xb1, 0x28, 0x91, 0x5f, 0xe7, 0x10, 0xdf,
    0x75, 0xfa, 0x23, 0xb4, 0x00, 0x5c, 0xaf, 0xfe, 0xe5, 0x18, 0xc1, 0x42,
    0x04, 0x03, 0x52, 0x3f, 0x0d, 0x7e, 0xb0, 0x36, 0x96, 0xbd, 0x21, 0x28,
    0x2e, 0xb0, 0xb1, 0xc1, 0x8a, 0x68, 0xf7, 0x0f, 0x66, 0x83, 0x7a, 0x58,
    0x45, 0x31, 0x90, 0xa0, 0x6a, 0xd0, 0xf6, 0xe2, 0xc2, 0x8c, 0x88, 0x18,
    0x35, 0x48, 0xc7, 0x02, 0x6e, 0x2e, 0xcc, 0x8a, 0x5d, 0xc5, 0x60, 0x6b,
    0xbb, 0x36, 0x79, 0xf2, 0x49, 0xb7, 0xa2, 0xef, 0x79, 0xfe, 0x70, 0x44,
    0x1c, 0x03, 0xfe, 0x0b, 0x54, 0x10, 0x16, 0xda, 0x53, 0xae, 0xdc, 0x40,
    0xcb, 0x69, 0xa4, 0xc9, 0x52, 0xcd, 0x07, 0x5a, 0x89, 0xd5, 0x03, 0x23,
    0x80, 0xe4, 0xa1, 0xc3
  };

  // https://github.com/usnistgov/ACVP-Server/blob/master/gen-val/json-files/ML-DSA-keyGen-FIPS204/expectedResults.json#L14
  static const uint8_t kMLDSAKeyGenPrivateKey[MLDSA44_PRIVATE_KEY_BYTES] = {
    0xad, 0xb0, 0xb3, 0x34, 0x64, 0x81, 0x60, 0x91, 0xf2, 0xa9, 0x59, 0x77,
    0xc6, 0x7f, 0x08, 0x5f, 0xdc, 0x24, 0xb3, 0x78, 0x54, 0xd4, 0xdb, 0x0a,
    0x57, 0x7a, 0xe9, 0x40, 0x1e, 0x40, 0x81, 0x48, 0xcf, 0x19, 0x5e, 0x6d,
    0x8d, 0xd3, 0x98, 0x71, 0x3e, 0x8f, 0x31, 0x7a, 0x5c, 0xb4, 0xd4, 0xf0,
    0x0b, 0x2b, 0x41, 0xf3, 0xa9, 0x58, 0x8a, 0x8f, 0xa4, 0xb8, 0x95, 0xd8,
    0xd8, 0xd1, 0xc9, 0xa1, 0x1f, 0xb7, 0x2b, 0x86, 0xf9, 0xb4, 0xa7, 0x51,
    0x32, 0x56, 0x7b, 0xe2, 0xc4, 0x5e, 0x87, 0x3d, 0x6b, 0x5d, 0x4a, 0x8b,
    0x8c, 0x59, 0xf4, 0x6f, 0x2d, 0xb8, 0x1b, 0x08, 0x57, 0x14, 0x4d, 0xb5,
    0x41, 0x6b, 0x07, 0x1d, 0xf2, 0xa8, 0xf7, 0x7d, 0x43, 0x7d, 0x47, 0xc3,
    0xbf, 0xe1, 0x1f, 0x16, 0xfe, 0xef, 0xe6, 0xc4, 0x70, 0xf4, 0x67, 0x62,
    0xc8, 0x81, 0xad, 0x19, 0xc0, 0x88, 0x4b, 0xb3, 0x02, 0x44, 0x26, 0x88,
    0x98, 0x69, 0x13, 0x45, 0x86, 0xd3, 0xb0, 0x41, 0xd3, 0x28, 0x46, 0x60,
    0xc2, 0x28, 0x18, 0x23, 0x02, 0x54, 0x28, 0x52, 0x03, 0x81, 0x0d, 0x04,
    0x87, 0x50, 0x20, 0x26, 0x22, 0x84, 0x12, 0x8e, 0x1c, 0x46, 0x2a, 0x20,
    0x03, 0x45, 0x48, 0x38, 0x29, 0x11, 0x90, 0x10, 0x80, 0x40, 0x50, 0x19,
    0xb3, 0x85, 0x08, 0x48, 0x48, 0xcc, 0x04, 0x90, 0x04, 0x88, 0x60, 0x89,
    0xa4, 0x89, 0x22, 0x05, 0x0a, 0x19, 0x87, 0x68, 0x51, 0x32, 0x2a, 0xa0,
    0x04, 0x41, 0x54, 0x22, 0x4e, 0xe3, 0x40, 0x29, 0x04, 0x41, 0x10, 0x02,
    0x13, 0x70, 0xdb, 0xa2, 0x69, 0xe1, 0x18, 0x72, 0xc4, 0x36, 0x26, 0x19,
    0x02, 0x66, 0x22, 0x96, 0x0d, 0x98, 0x84, 0x25, 0xc1, 0x02, 0x6e, 0x24,
    0x23, 0x31, 0x40, 0xb8, 0x70, 0x8a, 0x18, 0x09, 0x1c, 0x10, 0x68, 0x4c,
    0x12, 0x02, 0x09, 0xb7, 0x28, 0x59, 0x84, 0x0c, 0x01, 0x86, 0x4d, 0xc0,
    0x42, 0x62, 0xda, 0x84, 0x6d, 0x09, 0x26, 0x70, 0x8c, 0x16, 0x6e, 0xe2,
    0x44, 0x25, 0x51, 0x38, 0x68, 0xc2, 0x84, 0x25, 0x08, 0x39, 0x91, 0x90,
    0x88, 0x04, 0xa3, 0x46, 0x66, 0x03, 0x11, 0x02, 0x02, 0xc8, 0x64, 0x02,
    0x49, 0x24, 0x50, 0x92, 0x29, 0x20, 0x95, 0x81, 0xe4, 0x48, 0x61, 0x8c,
    0x48, 0x08, 0xe3, 0x18, 0x2d, 0x1c, 0xb8, 0x29, 0x42, 0xa2, 0x6c, 0x14,
    0xa2, 0x45, 0x48, 0x10, 0x71, 0x0b, 0x15, 0x44, 0x22, 0x48, 0x84, 0x58,
    0x42, 0x20, 0x18, 0x97, 0x69, 0x0a, 0xb5, 0x30, 0xd3, 0x30, 0x0c, 0xdc,
    0x20, 0x2c, 0x61, 0x36, 0x02, 0x14, 0x33, 0x8e, 0xe4, 0x88, 0x24, 0x13,
    0xb0, 0x44, 0x01, 0x06, 0x28, 0x13, 0x32, 0x4c, 0x13, 0x35, 0x4d, 0xc9,
    0xa8, 0x8d, 0xd0, 0x32, 0x2d, 0x0a, 0x11, 0x72, 0x0a, 0x24, 0x50, 0xc2,
    0x80, 0x49, 0x0a, 0xa7, 0x10, 0x23, 0x23, 0x6e, 0xdb, 0xb4, 0x41, 0xc8,
    0x96, 0x6c, 0x49, 0x44, 0x21, 0x18, 0x10, 0x20, 0x09, 0x28, 0x69, 0x64,
    0x18, 0x09, 0x18, 0x37, 0x6c, 0xc0, 0x12, 0x6e, 0xa2, 0x80, 0x8c, 0x0b,
    0x15, 0x80, 0x4b, 0xb6, 0x6c, 0x14, 0xb2, 0x41, 0x43, 0xa0, 0x84, 0x5c,
    0x38, 0x50, 0x03, 0x02, 0x8e, 0xa1, 0xc8, 0x0d, 0x1a, 0x12, 0x0e, 0x9a,
    0x20, 0x84, 0x00, 0x33, 0x01, 0x90, 0xa2, 0x05, 0x08, 0x15, 0x00, 0x02,
    0xa2, 0x11, 0x90, 0x48, 0x06, 0x19, 0x04, 0x08, 0xc3, 0x98, 0x91, 0x1c,
    0xb8, 0x00, 0x8a, 0x42, 0x65, 0xe2, 0x48, 0x40, 0x01, 0xa0, 0x01, 0x62,
    0x22, 0x84, 0x1b, 0x32, 0x28, 0x08, 0x38, 0x24, 0x09, 0x45, 0x6e, 0x03,
    0x35, 0x11, 0x14, 0x95, 0x80, 0xa1, 0x20, 0x51, 0x51, 0x34, 0x72, 0x03,
    0x92, 0x05, 0x61, 0x32, 0x71, 0xc1, 0x44, 0x02, 0xc4, 0x00, 0x90, 0x09,
    0x15, 0x8e, 0x48, 0xa0, 0x88, 0x00, 0xb8, 0x71, 0x52, 0xb0, 0x29, 0x02,
    0x28, 0x2e, 0x52, 0x96, 0x89, 0x1a, 0x21, 0x0d, 0x11, 0xa2, 0x2d, 0x64,
    0x26, 0x71, 0x8c, 0x00, 0x12, 0xc0, 0x26, 0x62, 0xe0, 0x30, 0x11, 0x98,
    0x36, 0x44, 0x61, 0xc2, 0x4c, 0x04, 0x30, 0x6c, 0xe0, 0x18, 0x4e, 0x0c,
    0xc2, 0x25, 0x9a, 0x30, 0x22, 0xe4, 0x88, 0x24, 0xd2, 0xc8, 0x60, 0xca,
    0xb8, 0x8c, 0x12, 0x46, 0x04, 0x64, 0x36, 0x6e, 0x18, 0xc8, 0x2d, 0x5b,
    0x34, 0x89, 0x8a, 0x22, 0x6a, 0x11, 0xb0, 0x30, 0x13, 0x84, 0x81, 0x04,
    0x11, 0x31, 0x62, 0xb4, 0x09, 0xe4, 0x22, 0x2c, 0x4b, 0x20, 0x2d, 0x0a,
    0x03, 0x70, 0xda, 0xc2, 0x61, 0xe1, 0x30, 0x8a, 0x9c, 0x96, 0x88, 0x03,
    0x45, 0x6e, 0x41, 0x24, 0x0d, 0x5c, 0x04, 0x00, 0x4c, 0x32, 0x4a, 0xe0,
    0x12, 0x4a, 0x1c, 0x25, 0x0a, 0x84, 0x36, 0x48, 0x90, 0x98, 0x2c, 0xc2,
    0xb4, 0x44, 0xc3, 0x48, 0x52, 0xcb, 0x96, 0x08, 0x54, 0x34, 0x28, 0x23,
    0x25, 0x26, 0x24, 0xb9, 0x51, 0x18, 0x37, 0x70, 0xe2, 0xc8, 0x64, 0x18,
    0xc6, 0x28, 0xc0, 0x42, 0x8d, 0xd9, 0xa6, 0x08, 0xa3, 0x30, 0x0e, 0x00,
    0xb4, 0x61, 0xca, 0xc2, 0x00, 0x4a, 0x32, 0x24, 0xcc, 0xc4, 0x2c, 0xd1,
    0x24, 0x46, 0x14, 0x10, 0x82, 0x09, 0xb2, 0x09, 0x89, 0x32, 0x28, 0x54,
    0x12, 0x66, 0x58, 0xc8, 0x69, 0xc2, 0x26, 0x49, 0x60, 0x96, 0x01, 0xd9,
    0x34, 0x60, 0x10, 0xc0, 0x51, 0x59, 0x44, 0x2c, 0x81, 0xc2, 0x69, 0x03,
    0x00, 0x8d, 0x9b, 0xc6, 0x40, 0x64, 0x40, 0x2e, 0xd8, 0x44, 0x0a, 0xe3,
    0x14, 0x25, 0xd3, 0x42, 0x32, 0x43, 0x22, 0x46, 0x09, 0xa2, 0x69, 0x94,
    0x24, 0x88, 0x8a, 0xc4, 0x8d, 0x04, 0x41, 0x11, 0x5b, 0x94, 0x11, 0x1c,
    0x33, 0x86, 0xa4, 0xa4, 0x61, 0x1a, 0xb5, 0x4c, 0x91, 0x22, 0x0e, 0x22,
    0xb6, 0x64, 0x22, 0x97, 0x68, 0x43, 0x38, 0x2e, 0xe1, 0xb4, 0x11, 0x63,
    0x90, 0x85, 0x52, 0xc0, 0x28, 0xe4, 0x44, 0x29, 0x5a, 0x26, 0x66, 0x01,
    0x06, 0x25, 0x80, 0x46, 0x61, 0x04, 0xa4, 0x2c, 0x94, 0x04, 0x52, 0x10,
    0x08, 0x24, 0x01, 0x05, 0x0d, 0x18, 0x24, 0x26, 0x18, 0x30, 0x84, 0x4b,
    0x04, 0x2d, 0xa0, 0xb4, 0x00, 0xd4, 0xb4, 0x90, 0xc3, 0x08, 0x46, 0x22,
    0xb1, 0x31, 0x14, 0x49, 0x71, 0xd3, 0x34, 0x61, 0x48, 0x02, 0x65, 0x00,
    0xa1, 0x20, 0x04, 0x44, 0x85, 0x11, 0x01, 0x4d, 0x84, 0xc2, 0x30, 0x61,
    0xc2, 0x89, 0x03, 0x46, 0x00, 0x22, 0xb2, 0x45, 0x93, 0xa0, 0x2d, 0x18,
    0x46, 0x48, 0x0b, 0x34, 0x88, 0x5c, 0xa2, 0x84, 0xc8, 0xa0, 0x64, 0x88,
    0xa4, 0x80, 0x12, 0x98, 0x21, 0x1b, 0x49, 0x6a, 0xd8, 0x33, 0x5f, 0xa7,
    0x9b, 0xf4, 0xec, 0x5f, 0xea, 0x39, 0xf1, 0xac, 0x7b, 0x7c, 0x58, 0x54,
    0xf0, 0xfb, 0x19, 0x85, 0xf4, 0x3d, 0x70, 0x7f, 0x56, 0x7c, 0xe1, 0x23,
    0x29, 0xe4, 0x3a, 0xbd, 0xaa, 0x9e, 0xd4, 0xc4, 0xb3, 0x9b, 0x63, 0x20,
    0xca, 0x70, 0xeb, 0xa0, 0x9a, 0x15, 0x97, 0x71, 0x7b, 0x72, 0xed, 0x5b,
    0x0a, 0xef, 0xd4, 0x6b, 0x7c, 0x5c, 0xbd, 0x56, 0xd8, 0x1c, 0x3a, 0x7a,
    0x6a, 0x9d, 0x35, 0xef, 0x4e, 0x5b, 0x87, 0xa9, 0xfb, 0xfd, 0x9b, 0x38,
    0x4c, 0x6b, 0x7f, 0x43, 0xec, 0x29, 0x63, 0x72, 0x37, 0x9b, 0x97, 0x05,
    0x78, 0x8b, 0xc7, 0xaa, 0x36, 0x68, 0x85, 0x96, 0xa8, 0x56, 0xc0, 0x24,
    0xfa, 0xb2, 0x40, 0x91, 0xf1, 0xb2, 0xad, 0x8d, 0xf6, 0x4f, 0xba, 0xba,
    0xc1, 0x8e, 0xff, 0x62, 0x61, 0xa9, 0x1b, 0x45, 0x48, 0x96, 0xd2, 0x3c,
    0xeb, 0x52, 0xef, 0x22, 0xd9, 0x3f, 0x2b, 0xf6, 0xd6, 0xec, 0x40, 0xce,
    0xe7, 0xfc, 0x83, 0x12, 0xd7, 0x87, 0xac, 0x9d, 0x45, 0xc6, 0xb2, 0xc1,
    0xc5, 0x2f, 0xdd, 0xea, 0x9a, 0x40, 0xdb, 0x0a, 0xf6, 0x32, 0x79, 0x2e,
    0x19, 0xe9, 0x8e, 0x04, 0xec, 0x44, 0x23, 0x52, 0xc9, 0x88, 0xac, 0x86,
    0xe1, 0xbb, 0x7d, 0x3f, 0x63, 0xa4, 0x12, 0xf9, 0xc7, 0xea, 0xb1, 0x74,
    0x63, 0xd1, 0x16, 0x31, 0x5d, 0x16, 0x07, 0x4c, 0x8e, 0x6c, 0xe0, 0x13,
    0x20, 0x03, 0x5b, 0x0a, 0x63, 0x51, 0x18, 0x6f, 0x3c, 0x7e, 0x17, 0xe0,
    0x91, 0x37, 0xce, 0x76, 0x3a, 0xd7, 0xc7, 0xbe, 0x49, 0x54, 0x1b, 0x74,
    0x05, 0xbb, 0xb8, 0xc7, 0x02, 0x6a, 0x5f, 0xa9, 0x77, 0xba, 0xe1, 0xb6,
    0x5b, 0x94, 0x86, 0x9e, 0xfe, 0xf0, 0x1b, 0x77, 0xcf, 0x17, 0x3e, 0x46,
    0x9a, 0x30, 0xda, 0x18, 0x2b, 0xd7, 0x66, 0x2a, 0xf0, 0x48, 0x18, 0xbd,
    0x62, 0x35, 0x24, 0xd5, 0x60, 0x0f, 0x23, 0xfd, 0x58, 0x68, 0xfa, 0x42,
    0xa4, 0x5f, 0x39, 0x67, 0x2e, 0x40, 0x1e, 0x2c, 0xf3, 0x36, 0xf4, 0x13,
    0x22, 0xf8, 0x23, 0x68, 0x4e, 0x6b, 0x87, 0x99, 0xb5, 0x5f, 0xb9, 0x6e,
    0xf9, 0x2f, 0x41, 0x0e, 0x23, 0xdd, 0xa7, 0x72, 0x48, 0xde, 0x65, 0x9f,
    0x70, 0x32, 0x28, 0xf9, 0x9c, 0x8a, 0xdf, 0x04, 0x52, 0x3c, 0x88, 0x10,
    0x57, 0x34, 0x49, 0x54, 0xa8, 0x1f, 0xdb, 0xae, 0x00, 0x0f, 0xae, 0x48,
    0x4c, 0xd8, 0x96, 0xc4, 0xe6, 0x54, 0x3c, 0xae, 0x00, 0x5a, 0x0e, 0x3d,
    0xf5, 0x1e, 0x37, 0xc0, 0x40, 0x1d, 0x10, 0x6e, 0x50, 0xdb, 0xa1, 0x41,
    0x68, 0xb5, 0x49, 0x6a, 0x3d, 0x6d, 0xaa, 0xb2, 0xe0, 0x10, 0xc2, 0xdc,
    0xe2, 0x15, 0x55, 0x3d, 0x10, 0x34, 0x7a, 0x95, 0x81, 0x98, 0x8d, 0x9e,
    0xdc, 0x25, 0x83, 0xfc, 0x10, 0x35, 0x0e, 0xc5, 0xef, 0x05, 0x18, 0xb3,
    0xfa, 0x9b, 0x54, 0x70, 0x2d, 0x93, 0xf2, 0xa9, 0x38, 0x06, 0x97, 0xe4,
    0x26, 0xc2, 0xa0, 0x17, 0xc1, 0xdb, 0xed, 0x69, 0x25, 0xc4, 0x4a, 0x77,
    0x50, 0x99, 0x95, 0xfe, 0x39, 0x0c, 0xab, 0xf2, 0x13, 0xd0, 0x13, 0xf2,
    0xcb, 0x9e, 0xbd, 0x9e, 0x09, 0xf9, 0xa0, 0x94, 0x98, 0x03, 0xde, 0x28,
    0x5c, 0x9c, 0x2e, 0x6a, 0x02, 0xcf, 0xfe, 0x7a, 0xea, 0xf0, 0x13, 0xea,
    0xab, 0x9b, 0x4e, 0x82, 0xe5, 0xd7, 0x5b, 0xbf, 0x8e, 0x0b, 0x84, 0x74,
    0xff, 0x63, 0xb2, 0xe7, 0xbe, 0xc2, 0xf1, 0xaa, 0x54, 0xef, 0xa0, 0xc1,
    0x94, 0xb6, 0x8f, 0xd4, 0x92, 0x46, 0x36, 0xd0, 0xe1, 0x88, 0xa3, 0x1e,
    0x32, 0xa4, 0x26, 0x5b, 0x24, 0x8f, 0xc9, 0x52, 0x5e, 0xd9, 0x46, 0x2f,
    0x9b, 0xf0, 0x4b, 0x86, 0xb4, 0x43, 0xbf, 0x5f, 0x86, 0xae, 0x66, 0x64,
    0x7e, 0x46, 0x36, 0x46, 0xd0, 0x2d, 0x0d, 0x8c, 0xe2, 0x01, 0xe0, 0xc2,
    0xce, 0x4a, 0x36, 0x23, 0xd0, 0xd1, 0x93, 0xa6, 0x64, 0x46, 0x34, 0xfe,
    0xba, 0x47, 0xa8, 0x55, 0x98, 0x9f, 0xcc, 0x8f, 0x3d, 0xcb, 0x81, 0xc5,
    0xf0, 0x5d, 0x9a, 0x0f, 0xe2, 0xe0, 0xe4, 0xdc, 0x09, 0x4a, 0x62, 0x1f,
    0x9f, 0xf4, 0x3f, 0x1f, 0x3e, 0x9a, 0x8f, 0x98, 0xcf, 0xd4, 0xe3, 0xc2,
    0x2f, 0x5f, 0xd7, 0xa0, 0xdc, 0x78, 0x8a, 0x13, 0x2e, 0x6f, 0x03, 0x42,
    0x7d, 0x29, 0x1e, 0xf9, 0xd6, 0x8a, 0xf3, 0xf3, 0xb6, 0x69, 0xa7, 0x65,
    0x63, 0x2d, 0xac, 0x5a, 0xa3, 0x8d, 0x57, 0x35, 0xca, 0x37, 0x0d, 0x4e,
    0xbc, 0xa8, 0xf0, 0x6f, 0x0f, 0x59, 0xd6, 0xc0, 0xd7, 0x49, 0x7d, 0x95,
    0x1f, 0x96, 0x68, 0xb5, 0x34, 0x03, 0x7b, 0x02, 0x7f, 0xa5, 0xa2, 0xfc,
    0x46, 0xdf, 0x7a, 0xf2, 0x3b, 0xe5, 0x61, 0x6d, 0xb2, 0x0a, 0xba, 0xce,
    0x02, 0xea, 0x19, 0xbe, 0x4b, 0x5d, 0xe6, 0x4e, 0x09, 0xa7, 0x1a, 0x7f,
    0x90, 0x72, 0x6e, 0x38, 0xb5, 0xa9, 0x68, 0xd7, 0xe5, 0x1f, 0x15, 0x46,
    0x6d, 0xa0, 0xaa, 0xc9, 0xb7, 0x4c, 0xd5, 0x0c, 0x54, 0x38, 0x19, 0xd0,
    0xce, 0xbe, 0x87, 0x98, 0x64, 0xd1, 0x45, 0x9e, 0x48, 0x08, 0x14, 0x5f,
    0x5f, 0x29, 0x52, 0x10, 0x16, 0xbb, 0x62, 0xb0, 0x5d, 0x8f, 0x71, 0x7b,
    0x12, 0x50, 0xcc, 0xef, 0x8a, 0x4c, 0x02, 0x96, 0x29, 0x2a, 0x86, 0x62,
    0x49, 0x9d, 0x94, 0xf7, 0xc3, 0xff, 0x83, 0xd7, 0xe1, 0x7d, 0x8d, 0x14,
    0x18, 0xec, 0x3f, 0x68, 0x43, 0xeb, 0xcb, 0xeb, 0xb9, 0x60, 0xf9, 0xf6,
    0x61, 0xe4, 0x56, 0xda, 0xdc, 0x48, 0x43, 0x72, 0x9d, 0x3f, 0xb3, 0x96,
    0xd8, 0xe8, 0x4e, 0xe1, 0x24, 0x56, 0x72, 0x32, 0x1e, 0x4d, 0x4c, 0x59,
    0x45, 0x4b, 0x53, 0x08, 0x44, 0x39, 0xd7, 0x66, 0x20, 0x42, 0xec, 0x1b,
    0xd2, 0x93, 0x05, 0x2e, 0x6a, 0x44, 0x46, 0x1b, 0xd3, 0x53, 0xcf, 0x32,
    0xa4, 0xb9, 0xfd, 0x4e, 0x95, 0x43, 0x45, 0x54, 0xce, 0xa0, 0x92, 0x2c,
    0xf0, 0xd5, 0x0b, 0x5f, 0x71, 0x8d, 0xe7, 0xb2, 0xf0, 0x1d, 0xe1, 0x89,
    0xef, 0xb8, 0x35, 0xf4, 0xb2, 0x9a, 0xb6, 0x2b, 0x99, 0xdc, 0x76, 0xcc,
    0xbe, 0xc2, 0x02, 0xa4, 0x82, 0x0d, 0xf9, 0x2a, 0x82, 0x03, 0x13, 0x87,
    0x9f, 0x8a, 0xf1, 0x9b, 0xfe, 0xe5, 0xb4, 0x57, 0xbf, 0x29, 0x87, 0xbe,
    0x48, 0x6c, 0xf3, 0xe8, 0x19, 0xa4, 0xfc, 0xe6, 0x7e, 0x64, 0xc6, 0xab,
    0xb4, 0xdd, 0x98, 0x11, 0xa5, 0xbd, 0x82, 0xbe, 0x73, 0x27, 0x43, 0xc5,
    0x2b, 0x1f, 0x84, 0xaf, 0x1b, 0x44, 0x14, 0x5c, 0x68, 0xa9, 0x6a, 0x06,
    0xed, 0xf5, 0xcf, 0xb7, 0xad, 0xc5, 0xbe, 0xc5, 0x5e, 0x60, 0xf6, 0x87,
    0x0b, 0xd1, 0x0d, 0xff, 0x60, 0x3b, 0x10, 0xe3, 0xf4, 0x54, 0xc0, 0xc9,
    0x7f, 0xb3, 0x37, 0xb1, 0x2c, 0x5d, 0xe0, 0x69, 0x38, 0xf6, 0x5d, 0x46,
    0x53, 0xf7, 0xd1, 0xed, 0xdd, 0x28, 0x1e, 0x4d, 0xdb, 0x1b, 0x7a, 0x5e,
    0x47, 0x53, 0x77, 0x69, 0x85, 0xee, 0x72, 0x15, 0x17, 0x62, 0x08, 0xb5,
    0x1c, 0x1d, 0x01, 0x38, 0xaf, 0x1d, 0x6a, 0x53, 0x54, 0x1f, 0x31, 0x29,
    0x51, 0x21, 0xaf, 0x68, 0x2d, 0x6f, 0xc0, 0x9f, 0x33, 0x5b, 0x88, 0x10,
    0x84, 0x07, 0xad, 0x78, 0x20, 0x94, 0xa7, 0xab, 0xc6, 0x64, 0xae, 0x1c,
    0x6b, 0x98, 0x1b, 0xf5, 0xd6, 0x46, 0xa0, 0xcb, 0xee, 0xaa, 0xa1, 0x7b,
    0x23, 0x53, 0xca, 0xf3, 0x1e, 0x4e, 0x1c, 0x72, 0x6e, 0x34, 0x0f, 0xf4,
    0x1c, 0xfc, 0x9e, 0x04, 0xbc, 0x77, 0x0a, 0xb7, 0xcc, 0x2e, 0xb0, 0x7f,
    0x01, 0x0b, 0x42, 0x98, 0x29, 0x2e, 0xfe, 0xac, 0xa1, 0x93, 0x34, 0x7c,
    0xd0, 0xbf, 0x18, 0x37, 0x04, 0x74, 0xd0, 0xdc, 0xd9, 0x5f, 0x5e, 0xb4,
    0x5e, 0x6a, 0xfd, 0x49, 0x36, 0xf8, 0x25, 0x1a, 0x1b, 0x19, 0x52, 0x1e,
    0xd7, 0xb2, 0x26, 0x79, 0xb5, 0x3c, 0x33, 0x1f, 0xa1, 0x48, 0x50, 0xd8,
    0x77, 0xaa, 0x12, 0xbe, 0xbc, 0x68, 0x90, 0x3f, 0x8c, 0x17, 0xc4, 0x6f,
    0x75, 0x47, 0x03, 0x8d, 0x22, 0xcd, 0xa0, 0xd6, 0x0c, 0x91, 0x2f, 0xb9,
    0xdc, 0xb0, 0xe4, 0xe8, 0xfb, 0x65, 0x6b, 0x90, 0xc4, 0x1f, 0x38, 0x20,
    0xb9, 0x09, 0x38, 0x26, 0xb0, 0x90, 0x41, 0x31, 0x85, 0x93, 0x4d, 0xbc,
    0x92, 0xfa, 0x2d, 0x20, 0x9f, 0x96, 0xbc, 0x58, 0xb1, 0x00, 0x86, 0x8c,
    0x24, 0xf8, 0x84, 0x5c, 0xb3, 0x2f, 0x28, 0x87, 0xee, 0x58, 0x26, 0xf6,
    0xb8, 0x8b, 0x47, 0xe6, 0x2b, 0x55, 0xe5, 0x39, 0x89, 0xaf, 0x38, 0x5c,
    0x1d, 0xc0, 0x68, 0x15, 0xa4, 0xf3, 0xe7, 0x3c, 0x69, 0x56, 0x94, 0xb7,
    0x64, 0x16, 0x64, 0x56, 0xe4, 0x6b, 0x4c, 0x05, 0x0a, 0x61, 0x22, 0x9a,
    0x87, 0xb3, 0x3d, 0xba, 0x7e, 0x56, 0xca, 0x77, 0xd8, 0x52, 0xcc, 0x58,
    0xba, 0xd1, 0x01, 0x08, 0x5a, 0x58, 0xc5, 0x58, 0x89, 0xa6, 0x1c, 0x09,
    0xb7, 0x5b, 0xcc, 0xd4, 0x2c, 0x80, 0x63, 0xc7, 0xaa, 0xc1, 0x32, 0x5c,
    0x9f, 0xca, 0x8f, 0x6e, 0xe4, 0x67, 0x3b, 0x08, 0x56, 0xc3, 0x3a, 0xa6,
    0x8c, 0x0e, 0x76, 0x9e, 0xbc, 0xaa, 0xc4, 0x70, 0x87, 0xd6, 0x21, 0x17,
    0xc0, 0xf5, 0xd1, 0xca, 0xd2, 0x16, 0xfe, 0x1d, 0x6c, 0xa3, 0xc4, 0x6d,
    0x94, 0xc1, 0x3e, 0xef, 0xb8, 0x11, 0x16, 0xe2, 0xa4, 0x94, 0x32, 0x4f,
    0xaf, 0x22, 0x3a, 0xba, 0x50, 0x38, 0x04, 0xc7, 0x36, 0x8c, 0xcd, 0xfc,
    0x94, 0x2e, 0x13, 0x09, 0xca, 0x8c, 0xf5, 0x5e, 0x4d, 0x3f, 0x77, 0xf9,
    0xa6, 0x5d, 0x84, 0x6a, 0x00, 0x3d, 0xd7, 0x9e, 0x91, 0xa3, 0x6f, 0x9a,
    0x86, 0xbe, 0xda, 0x5c, 0xac, 0xf6, 0xf8, 0xa8, 0xd3, 0x58, 0x26, 0xd3,
    0xab, 0x6e, 0x93, 0xd8, 0x6f, 0xab, 0xac, 0x91, 0x4e, 0x84, 0x00, 0x6d,
    0x62, 0x38, 0xa9, 0x9b, 0xde, 0xfb, 0xff, 0x93, 0x90, 0xa4, 0x55, 0xac,
    0xd8, 0x68, 0x11, 0x41, 0xb4, 0xa6, 0xea, 0x64, 0x41, 0xe6, 0x2e, 0x43,
    0x6c, 0xbb, 0x65, 0xbb, 0x10, 0x30, 0xab, 0xf3, 0xc5, 0x9e, 0xd1, 0x42,
    0xe2, 0xe2, 0x25, 0x44, 0x69, 0x3e, 0xb3, 0x4b, 0xbf, 0x0b, 0x12, 0xef,
    0xbf, 0x01, 0xbf, 0x65, 0x60, 0xf1, 0xa5, 0x59, 0xd2, 0xca, 0x9f, 0xdb,
    0x9d, 0x8c, 0x96, 0xff, 0x42, 0x1c, 0xc1, 0xcf, 0x34, 0xb4, 0xfb, 0xdc,
    0x85, 0xe8, 0xbb, 0x04, 0x53, 0xdd, 0x66, 0x78, 0xac, 0xf4, 0x00, 0xc3,
    0x1b, 0x3b, 0x40, 0xa2, 0xd8, 0xbd, 0x86, 0x39, 0xba, 0xbc, 0x12, 0xab,
    0xf9, 0xa3, 0x36, 0x59, 0x59, 0xf4, 0x6b, 0x8e, 0xf0, 0x66, 0x68, 0xf3,
    0x62, 0xc7, 0xea, 0xfa, 0x89, 0xc6, 0xc2, 0xc5, 0x25, 0xc7, 0xe9, 0xe5,
    0x84, 0x0b, 0x45, 0x26, 0xfd, 0x1f, 0xe9, 0x57, 0x5b, 0x3a, 0x44, 0x19,
    0x5f, 0x5d, 0xd3, 0x17, 0x92, 0x49, 0x0e, 0x76, 0xe2, 0x54, 0xc6, 0x13,
    0x1a, 0x98, 0xdd, 0x77, 0xf8, 0xd6, 0xac, 0x74, 0x9a, 0xc8, 0x2a, 0x98,
    0x38, 0x2a, 0x48, 0x82, 0xc4, 0x2a, 0xd7, 0xc6, 0x3d, 0xf2, 0x20, 0xda,
    0x0f, 0xbc, 0x46, 0x81, 0x5a, 0x21, 0xc4, 0x9f, 0x2a, 0xd1, 0x43, 0xb7,
    0x6f, 0x78, 0x9e, 0x02, 0xb6, 0x38, 0x21, 0x00, 0xde, 0xab, 0xdc, 0xcd,
    0xef, 0xc6, 0xe7, 0x4b, 0x87, 0x72, 0x4b, 0x95, 0x82, 0xde, 0xeb, 0x98,
    0xe2, 0xda, 0x6f, 0x31, 0x62, 0xd2, 0xd5, 0x68, 0xf3, 0x73, 0xc8, 0x7e,
    0x7b, 0x38, 0xf8, 0x58, 0xb7, 0x12, 0x19, 0x52, 0xd7, 0x8b, 0xb0, 0xc5,
    0x9a, 0x18, 0xd7, 0x60, 0x32, 0xd8, 0x28, 0x22, 0x15, 0x63, 0x4f, 0x73,
    0x1c, 0x86, 0x7f, 0x8a, 0xbd, 0xd1, 0x20, 0x03, 0xbd, 0x4c, 0xc1, 0xdc,
    0x75, 0x72, 0x78, 0x5a, 0xf6, 0x15, 0x35, 0xed, 0xad, 0x9e, 0x4f, 0x25,
    0x76, 0x9a, 0x40, 0xe3, 0x91, 0x7b, 0x27, 0x26, 0x59, 0xd1, 0x34, 0x8a,
    0xb8, 0xe6, 0x12, 0x42, 0xed, 0x8b, 0x97, 0xd5, 0x11, 0xd5, 0x79, 0x15,
    0xe5, 0x83, 0x67, 0x13, 0x69, 0xb6, 0x08, 0x8b, 0x4b, 0x4f, 0x88, 0x23,
    0xf6, 0x10, 0xd1, 0xb5, 0xc1, 0x57, 0x8e, 0x3f, 0xf4, 0x5d, 0xab, 0x7c,
    0xc4, 0x7d, 0x58, 0x9a, 0x68, 0x0a, 0x1d, 0xe6, 0x0a, 0xb9, 0xb5, 0x1f,
    0x1a, 0x53, 0x1f, 0xba, 0x56, 0xed, 0x21, 0xb3, 0xb5, 0xee, 0xee, 0x4f,
    0x75, 0x35, 0xf3, 0x06, 0xd5, 0x83, 0x23, 0x68, 0x7e, 0x20, 0x36, 0x14,
    0xdc, 0x64, 0xa6, 0x89
  };

  static const uint8_t kMLDSASignPlaintext[32] = {
    0x4a, 0x41, 0x4b, 0x45, 0x20, 0x4d, 0x41, 0x53, 0x53, 0x49, 0x4d, 0x4f,
    0x20, 0x41, 0x57, 0x53, 0x32, 0x30, 0x32, 0x35, 0x2e, 0x01, 0x02, 0x03,
    0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b,
  };

  static const uint8_t kMLDSAVerifyPlaintext[32] = {
    0x83, 0xf2, 0x1a, 0xc4, 0xb5, 0x9e, 0x7d, 0x2f, 0x48, 0xe6, 0x95, 0xd3,
    0x0c, 0x5b, 0xa1, 0x8d, 0x3f, 0x67, 0xb9, 0x14, 0x92, 0xc8, 0x4e, 0x76,
    0xf5, 0x21, 0xaa, 0x54, 0x89, 0x36, 0xdc, 0xeb,
};

  // ISO/IEC 19790:2012 Section 7.10.3.3 15. page 84 Requires the randomization
  // parameter rnd be fixed for the “hedged” algorithms (e.g., all zeros).
  static const uint8_t kMLDSASigGenSeed[MLDSA44_SIGNATURE_SEED_BYTES] = {
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  };

  static const uint8_t kMLDSASignSignature[MLDSA44_SIGNATURE_BYTES] = {
    0x34, 0x69, 0x80, 0xe5, 0x19, 0x14, 0x72, 0x2c, 0xb3, 0xdc, 0x40, 0x2d,
    0xca, 0xc9, 0xfa, 0x9a, 0xcb, 0xa9, 0xf1, 0x89, 0xe6, 0xe1, 0x2c, 0xba,
    0x7a, 0x15, 0x18, 0x1b, 0xf9, 0x5f, 0x88, 0x30, 0xe3, 0xb0, 0x27, 0x1b,
    0x4a, 0x9f, 0x84, 0xd4, 0x5a, 0x1a, 0x29, 0xc1, 0x8d, 0x15, 0xea, 0x25,
    0x4e, 0x9f, 0x5e, 0x6d, 0xbd, 0x0f, 0x12, 0xad, 0x8a, 0x70, 0x08, 0xdf,
    0x06, 0x7e, 0x93, 0x14, 0x69, 0xa9, 0xc3, 0x57, 0x59, 0x73, 0x66, 0x77,
    0x45, 0x6a, 0x90, 0x1c, 0xbf, 0xc7, 0x9b, 0xf7, 0x35, 0x1f, 0x55, 0x85,
    0xb7, 0xa2, 0x59, 0x04, 0x90, 0x83, 0x9e, 0x30, 0x13, 0xd3, 0x10, 0x37,
    0xa0, 0x8a, 0x1c, 0xd3, 0x50, 0x1e, 0x4b, 0x65, 0x82, 0x30, 0xf0, 0x31,
    0x84, 0x95, 0xc4, 0x98, 0xd7, 0x04, 0x40, 0xd5, 0x78, 0xc7, 0xf7, 0xbd,
    0x61, 0xc7, 0xa1, 0xa3, 0xda, 0x31, 0x5c, 0x40, 0x30, 0x69, 0x8b, 0xc7,
    0x0e, 0xb3, 0x31, 0xbb, 0xb8, 0x3b, 0xab, 0x5a, 0x6b, 0x63, 0x33, 0x8f,
    0x11, 0x31, 0x55, 0x84, 0x5d, 0x97, 0x38, 0x9d, 0x29, 0x4e, 0x1d, 0x3b,
    0x77, 0x65, 0x8c, 0x1e, 0x24, 0xa4, 0x38, 0xf9, 0xcd, 0xf1, 0x3b, 0x1e,
    0x01, 0x1c, 0x2e, 0x5b, 0x25, 0xdc, 0xc2, 0x18, 0xa9, 0x75, 0x76, 0x5b,
    0xa7, 0x9e, 0x63, 0x19, 0x12, 0xb0, 0xad, 0x10, 0xcc, 0x8b, 0xbd, 0x90,
    0xe7, 0xba, 0x25, 0x8a, 0x70, 0x2a, 0x9b, 0x5e, 0x6d, 0x72, 0xf5, 0x78,
    0xb5, 0xa9, 0x42, 0xb9, 0x91, 0x7a, 0x39, 0x89, 0xae, 0x8a, 0xe0, 0xb9,
    0x8c, 0xde, 0x12, 0x8b, 0x52, 0x95, 0x89, 0xc2, 0xfb, 0xba, 0xa8, 0x2a,
    0x81, 0x7e, 0xf2, 0xb3, 0xd9, 0x16, 0x97, 0x88, 0x38, 0x7c, 0x61, 0x2e,
    0x24, 0x88, 0xb6, 0x2a, 0xb9, 0x66, 0x4d, 0x2e, 0x06, 0xb3, 0xf2, 0x2d,
    0xe1, 0xeb, 0x09, 0x1e, 0x0e, 0x89, 0xaf, 0xe5, 0xb0, 0x90, 0xe1, 0xb8,
    0x1d, 0x68, 0x67, 0xb2, 0xbe, 0xa7, 0xc0, 0x19, 0x07, 0x9b, 0x29, 0x3d,
    0x8b, 0x10, 0x58, 0x22, 0xf0, 0xcb, 0x34, 0xfd, 0xc0, 0x3e, 0x12, 0x10,
    0x06, 0x2e, 0x4e, 0x1e, 0x28, 0x0c, 0xe3, 0xb5, 0x8d, 0x87, 0xf8, 0x43,
    0xde, 0x43, 0x08, 0x24, 0x08, 0x22, 0x03, 0x90, 0x67, 0x6b, 0x07, 0xe6,
    0xae, 0x5c, 0x1b, 0x6f, 0xa7, 0x7d, 0x57, 0x12, 0x49, 0xa3, 0x7e, 0xfd,
    0x41, 0xe9, 0x4d, 0xaf, 0x2d, 0x96, 0xa5, 0x40, 0x42, 0x65, 0x0f, 0x77,
    0xaa, 0xab, 0x3c, 0x43, 0xec, 0x5a, 0xfe, 0xa0, 0x8e, 0x55, 0xbb, 0x49,
    0x49, 0x52, 0xaa, 0x40, 0xb5, 0x09, 0x5e, 0x7c, 0x1b, 0x98, 0xd8, 0x10,
    0x4a, 0xb0, 0x7e, 0x7d, 0xbf, 0xc5, 0xf6, 0xf6, 0x82, 0x67, 0x61, 0xe6,
    0x85, 0xd0, 0x6a, 0x3a, 0x00, 0xf3, 0x88, 0x41, 0xa2, 0xba, 0x8d, 0x2c,
    0xcd, 0xe1, 0x14, 0x5c, 0xd4, 0xb0, 0x64, 0x3d, 0x95, 0x3d, 0x8f, 0xd4,
    0xc4, 0x85, 0x9b, 0xb7, 0x2f, 0x33, 0x8b, 0x9a, 0x1c, 0x03, 0x8f, 0x88,
    0xce, 0xfb, 0x5b, 0x33, 0xf8, 0x90, 0x78, 0xf7, 0xed, 0x8f, 0xa0, 0x50,
    0x53, 0x4b, 0x61, 0x97, 0xcc, 0xd2, 0x44, 0xb3, 0xbb, 0xd7, 0x06, 0x6f,
    0x6c, 0x4b, 0xbe, 0x74, 0x82, 0xec, 0xeb, 0xad, 0x20, 0x3e, 0x80, 0x79,
    0x59, 0x9d, 0x6a, 0xdc, 0xc2, 0x5e, 0xfc, 0x15, 0x1e, 0x3b, 0xe9, 0xc0,
    0xd7, 0x43, 0x5a, 0xc1, 0xcb, 0x3a, 0x57, 0x23, 0x81, 0xda, 0x12, 0x3a,
    0x67, 0x98, 0xcf, 0x21, 0xd1, 0x98, 0x35, 0xf9, 0xb9, 0x99, 0xd0, 0x9c,
    0x0b, 0x0a, 0x09, 0xf8, 0x6a, 0x9c, 0xf8, 0x7f, 0x2e, 0xd5, 0x13, 0x3e,
    0x36, 0xea, 0x6d, 0x69, 0x86, 0x18, 0x27, 0x96, 0x61, 0x2b, 0xd0, 0x61,
    0x91, 0xc9, 0xe4, 0x94, 0x05, 0x09, 0x53, 0x14, 0x1d, 0xa1, 0x74, 0x50,
    0x3c, 0x9c, 0x87, 0x4b, 0xf0, 0x7a, 0x3b, 0x64, 0x99, 0x45, 0x83, 0xc3,
    0xba, 0xb1, 0xaa, 0x90, 0xe0, 0x43, 0x31, 0x20, 0x0f, 0x86, 0xce, 0x5f,
    0x70, 0x5f, 0xe4, 0x67, 0xa4, 0x74, 0x1b, 0x0b, 0xd5, 0x7c, 0x34, 0xb8,
    0xd6, 0x66, 0x26, 0xc4, 0x2d, 0x72, 0xf7, 0x74, 0x39, 0xf5, 0xbc, 0xb4,
    0xed, 0x2e, 0x3f, 0x19, 0x72, 0x70, 0xa9, 0x3e, 0xfe, 0x68, 0xf7, 0xb2,
    0xa3, 0x0d, 0x6e, 0xbd, 0x9f, 0xa2, 0x29, 0x7f, 0x59, 0x77, 0x3e, 0x3d,
    0x8e, 0x81, 0xce, 0xc9, 0xea, 0x9d, 0xef, 0xd1, 0xb4, 0x43, 0x09, 0x75,
    0x9d, 0x0b, 0xfb, 0x0f, 0x43, 0xaa, 0x29, 0x38, 0x8d, 0xb9, 0x88, 0xf2,
    0xbe, 0x64, 0x5f, 0xfb, 0x3d, 0xe9, 0xed, 0xc1, 0x18, 0x1e, 0x93, 0x4f,
    0x14, 0x96, 0xce, 0xec, 0x5e, 0x31, 0x3f, 0xb9, 0xaa, 0xad, 0x37, 0xdd,
    0x6a, 0x8b, 0xe0, 0x9c, 0x7c, 0x7f, 0xd2, 0x63, 0xc2, 0xc9, 0x5c, 0x87,
    0xd5, 0xf8, 0x84, 0xbb, 0x75, 0xea, 0xd6, 0x77, 0xb3, 0xa0, 0xe4, 0xbc,
    0xc8, 0xbe, 0xc4, 0xf9, 0x9e, 0xd3, 0xee, 0x13, 0xfc, 0x89, 0x1b, 0x6c,
    0xc5, 0x86, 0x9a, 0xc8, 0x5d, 0x05, 0x75, 0x3c, 0x4c, 0xba, 0x0b, 0x32,
    0x91, 0x1a, 0xef, 0xab, 0x17, 0x67, 0xa7, 0xd0, 0x7c, 0x06, 0x1a, 0xf8,
    0x05, 0xc1, 0x76, 0x73, 0x34, 0x90, 0xf0, 0xe2, 0x01, 0x0b, 0xeb, 0xb1,
    0x8e, 0xb8, 0xc6, 0x1c, 0xec, 0x39, 0x20, 0xa3, 0x38, 0x66, 0xc2, 0xff,
    0x81, 0x97, 0xb0, 0x61, 0x5d, 0xae, 0x2d, 0x8e, 0x3b, 0x64, 0x12, 0xaf,
    0xd0, 0x68, 0x75, 0xf1, 0x9f, 0x5b, 0xd7, 0x0a, 0xd4, 0x78, 0xb8, 0x9a,
    0x7f, 0x3c, 0x7e, 0x0e, 0xd3, 0x3a, 0x12, 0xf0, 0x95, 0xbb, 0x0c, 0x2f,
    0x06, 0xad, 0x0c, 0x8f, 0x43, 0xba, 0x82, 0x22, 0x4b, 0x68, 0xc6, 0xc9,
    0xce, 0xc9, 0x92, 0x3c, 0x2f, 0x6c, 0xcd, 0x09, 0x95, 0xe5, 0x01, 0x5d,
    0x7f, 0x9e, 0x73, 0x07, 0x42, 0x62, 0xde, 0x21, 0xca, 0x59, 0x69, 0x71,
    0x36, 0xfd, 0x99, 0x92, 0x3d, 0x05, 0x7b, 0xe5, 0x00, 0x8b, 0x3b, 0xb0,
    0x18, 0xa5, 0x59, 0x43, 0x9b, 0xba, 0x5e, 0x33, 0xbb, 0xe2, 0x5b, 0x75,
    0xad, 0x95, 0xfc, 0x9e, 0xd4, 0x7c, 0x2a, 0x93, 0xce, 0x1e, 0x4a, 0xf3,
    0x0f, 0x0f, 0x68, 0x11, 0x51, 0x9d, 0x2a, 0xf2, 0x32, 0xf3, 0xba, 0x00,
    0x10, 0xa7, 0xbf, 0x23, 0x28, 0x4e, 0xbe, 0xd6, 0x7b, 0x1a, 0x66, 0x1f,
    0xcd, 0x3e, 0x4d, 0xad, 0x4f, 0xed, 0x42, 0x33, 0x0a, 0x5f, 0x8c, 0xb1,
    0x67, 0x72, 0x94, 0x3b, 0xd9, 0x3f, 0xb2, 0x5c, 0xe7, 0x7c, 0x06, 0x80,
    0xa4, 0xb7, 0x92, 0x24, 0xac, 0x4a, 0xff, 0x9a, 0xed, 0xf9, 0x5d, 0x04,
    0xf9, 0x2a, 0xe2, 0x91, 0x7f, 0xdd, 0x97, 0x14, 0x44, 0x56, 0xa3, 0x8e,
    0x47, 0x6b, 0xf8, 0x6f, 0xb5, 0x7f, 0x58, 0xae, 0x61, 0x1a, 0xc6, 0xcd,
    0x7b, 0x2a, 0xbd, 0x90, 0x8e, 0x78, 0xd5, 0x31, 0x2a, 0xe7, 0x2d, 0xc9,
    0xf7, 0xcb, 0x99, 0x9f, 0x3c, 0x42, 0xbe, 0xf6, 0x27, 0x1e, 0xdb, 0x57,
    0xf4, 0xc3, 0xd9, 0xac, 0x1e, 0xa2, 0xd9, 0x3c, 0xc0, 0x07, 0x3c, 0x65,
    0x66, 0x6b, 0x0f, 0x5a, 0x71, 0xcd, 0xbf, 0xff, 0x5e, 0x81, 0x80, 0x80,
    0x4b, 0x6e, 0x8c, 0x36, 0x53, 0x32, 0x35, 0x89, 0x93, 0x9f, 0xa7, 0x74,
    0x7a, 0x68, 0xea, 0x9f, 0x37, 0x27, 0x2f, 0x4c, 0xb9, 0xbb, 0xcc, 0x0d,
    0x52, 0x94, 0x31, 0xd8, 0x66, 0xe9, 0x29, 0xaa, 0xd9, 0xcf, 0x39, 0xcc,
    0x8a, 0x39, 0xca, 0x09, 0xba, 0x15, 0x20, 0xd7, 0xa8, 0x85, 0xe7, 0x9c,
    0x76, 0xef, 0xe5, 0xfe, 0xad, 0x4e, 0x37, 0x98, 0x7c, 0xe2, 0x61, 0x0f,
    0x02, 0x5b, 0xee, 0x4b, 0xee, 0xf2, 0x25, 0xe7, 0xd9, 0xc7, 0x85, 0x13,
    0x17, 0xeb, 0x79, 0x91, 0xe4, 0x6d, 0x68, 0xde, 0xd6, 0xde, 0x12, 0xa8,
    0x48, 0x87, 0x0b, 0x73, 0x32, 0x87, 0x32, 0xc0, 0x60, 0xdf, 0xda, 0xb2,
    0x91, 0x39, 0x5b, 0x8c, 0xce, 0x4d, 0xbf, 0xa9, 0xf7, 0xfd, 0x1e, 0x2d,
    0x2c, 0x3d, 0xee, 0xd6, 0xfe, 0x1b, 0xed, 0x9c, 0xc2, 0x5a, 0x36, 0x29,
    0xb4, 0x72, 0xd8, 0xe4, 0x84, 0xf0, 0x7d, 0xf8, 0xf0, 0x90, 0x84, 0x9d,
    0x78, 0xd2, 0x22, 0xe0, 0x68, 0xaf, 0xc9, 0xe9, 0x9d, 0x3c, 0xf6, 0xa6,
    0x15, 0xfd, 0xeb, 0x77, 0x83, 0x95, 0xa4, 0xbb, 0xf0, 0x53, 0x5c, 0x2d,
    0x97, 0x25, 0x7f, 0x01, 0x2a, 0x29, 0xe6, 0xfa, 0x03, 0x90, 0x44, 0x67,
    0x72, 0xca, 0x16, 0x15, 0x85, 0x20, 0x23, 0xd6, 0x15, 0x55, 0x02, 0xa0,
    0x12, 0xb2, 0x82, 0xc4, 0x08, 0xc7, 0x1c, 0x2f, 0x70, 0x33, 0x1b, 0xfe,
    0xc4, 0x6b, 0x09, 0x13, 0x6d, 0x4f, 0xd4, 0x95, 0xb0, 0x51, 0x42, 0x2b,
    0xd1, 0xea, 0x47, 0x87, 0xef, 0x3c, 0x8b, 0x00, 0x68, 0xa0, 0x73, 0x6f,
    0x8d, 0xe8, 0x26, 0x4c, 0x85, 0xb8, 0x47, 0x30, 0x36, 0x6a, 0xca, 0x64,
    0x92, 0x4f, 0x23, 0x1e, 0xb1, 0x69, 0xfc, 0x13, 0x20, 0x5c, 0xb4, 0xf1,
    0xea, 0xe7, 0x18, 0x98, 0x64, 0x2a, 0x5b, 0xfa, 0xf2, 0xc8, 0x1f, 0x7d,
    0x06, 0xaa, 0x7e, 0x5c, 0x98, 0x85, 0x2e, 0x55, 0x3a, 0xfe, 0xaf, 0x38,
    0x35, 0x54, 0xfa, 0x9e, 0xef, 0x32, 0xb2, 0xbf, 0x82, 0x45, 0x1a, 0xe7,
    0xc4, 0x05, 0xf1, 0x25, 0x30, 0xcb, 0x45, 0x23, 0xd4, 0x8e, 0xb4, 0x3d,
    0x72, 0x43, 0x6d, 0x97, 0xcd, 0xd0, 0x1f, 0x72, 0x84, 0xe6, 0xa8, 0xcb,
    0xeb, 0x6d, 0x67, 0xdb, 0x50, 0xb4, 0x3f, 0xa3, 0xe2, 0x72, 0x91, 0x43,
    0x83, 0xac, 0x59, 0x12, 0x03, 0xba, 0x48, 0x9c, 0xa2, 0x28, 0x2e, 0x42,
    0x6b, 0x05, 0xdb, 0xab, 0x35, 0xe7, 0x7c, 0x63, 0x6e, 0x13, 0xce, 0xb2,
    0x72, 0x5c, 0x91, 0x1f, 0xef, 0xc3, 0xd2, 0x80, 0xd1, 0xca, 0xa7, 0x4b,
    0x35, 0x6f, 0x18, 0x3d, 0x70, 0x91, 0xeb, 0x59, 0x05, 0xf2, 0xb6, 0x15,
    0xde, 0x61, 0x91, 0x21, 0x57, 0x85, 0xf6, 0xe2, 0xe1, 0xba, 0x07, 0x00,
    0xb1, 0xf9, 0xed, 0x74, 0x9b, 0xc5, 0x26, 0x5a, 0xc3, 0x18, 0xa4, 0x0c,
    0x99, 0xc6, 0x8c, 0xf9, 0x01, 0xfc, 0x81, 0xf4, 0x85, 0x9a, 0x91, 0xe4,
    0xd2, 0x48, 0xce, 0xca, 0xe1, 0xb8, 0x9a, 0x7b, 0x24, 0x5d, 0x1d, 0xf9,
    0xdb, 0x74, 0x31, 0x29, 0x27, 0x65, 0xd2, 0x3f, 0x4c, 0x65, 0x1b, 0x32,
    0x12, 0xc3, 0xe2, 0xee, 0x15, 0x9b, 0xfc, 0xa1, 0xd7, 0xb5, 0xae, 0xe7,
    0x4e, 0x53, 0x9f, 0x23, 0x36, 0x3c, 0x11, 0xc8, 0x0f, 0x07, 0x9f, 0x3e,
    0x62, 0x7b, 0x5c, 0x6b, 0xf0, 0x97, 0xc8, 0xf9, 0x28, 0x29, 0x9f, 0x28,
    0x32, 0x9e, 0xb9, 0x69, 0xf6, 0x4c, 0x9a, 0xad, 0x10, 0x89, 0xe4, 0xeb,
    0x39, 0x57, 0xa9, 0xe2, 0xa2, 0xa3, 0xdc, 0x6a, 0x1d, 0xdb, 0x84, 0xb6,
    0xc5, 0x15, 0x60, 0xe6, 0xcf, 0x26, 0xdd, 0x98, 0x68, 0xf5, 0xc7, 0xbc,
    0xb1, 0x07, 0x1c, 0xdd, 0x66, 0x54, 0x92, 0xae, 0xad, 0x80, 0xf6, 0x3e,
    0xd2, 0x2a, 0x23, 0x85, 0x7c, 0x23, 0x3b, 0xe5, 0xaf, 0x0d, 0x17, 0x2c,
    0x76, 0x50, 0xba, 0xbc, 0x6e, 0xad, 0x9d, 0x0c, 0x68, 0x3c, 0xa0, 0xb3,
    0x92, 0x4a, 0xc0, 0x6c, 0xde, 0xad, 0x8a, 0x70, 0xd6, 0xfa, 0x99, 0x6b,
    0xae, 0xf9, 0x97, 0x60, 0x23, 0xf0, 0x8b, 0x7b, 0x05, 0xb7, 0x82, 0x46,
    0x10, 0xeb, 0xbf, 0x9b, 0x0e, 0x9e, 0x18, 0xa0, 0x62, 0x46, 0x70, 0x6a,
    0xf5, 0xc0, 0x16, 0x7a, 0xcc, 0xda, 0x3d, 0xb0, 0xfe, 0x47, 0x5b, 0xe6,
    0x77, 0x5f, 0xee, 0xd7, 0x30, 0x96, 0xf4, 0xb4, 0xfa, 0x29, 0x57, 0x20,
    0xa6, 0xac, 0x64, 0xf1, 0x3d, 0x9f, 0x84, 0x57, 0xf2, 0xf3, 0x94, 0xe0,
    0x19, 0xa6, 0x08, 0xa5, 0xc2, 0xf9, 0x18, 0xf7, 0xe5, 0x3f, 0x81, 0xa3,
    0x77, 0x66, 0xfd, 0x3f, 0x0a, 0x9c, 0x83, 0xd0, 0xa9, 0x22, 0x6d, 0xad,
    0x28, 0x37, 0xc7, 0x0f, 0xa8, 0xc2, 0xca, 0x74, 0x89, 0xbc, 0x0e, 0xfe,
    0x37, 0xd5, 0x47, 0x98, 0xb3, 0x86, 0xeb, 0xe5, 0x80, 0x00, 0x7b, 0xcc,
    0xba, 0x84, 0xb1, 0x03, 0xbe, 0x53, 0x1e, 0xff, 0x0f, 0x9d, 0x97, 0x8a,
    0xa0, 0xb3, 0x30, 0x69, 0x53, 0x57, 0x3f, 0xfd, 0x0c, 0x37, 0xc6, 0x3a,
    0x63, 0x7c, 0xd7, 0xd2, 0x42, 0x5b, 0x6f, 0xa3, 0x85, 0x0c, 0xc7, 0xaa,
    0xb3, 0xc6, 0xe8, 0x75, 0x8d, 0x4d, 0xd2, 0x99, 0x39, 0xdb, 0xa4, 0x88,
    0x5f, 0x95, 0x73, 0xe2, 0x1b, 0x38, 0x1e, 0xdc, 0xa5, 0x80, 0xaa, 0xed,
    0x49, 0xae, 0x88, 0x87, 0x93, 0x76, 0xd3, 0x89, 0x86, 0x93, 0x6c, 0xf5,
    0xb3, 0xa3, 0x3d, 0x52, 0xa3, 0x2a, 0x37, 0x26, 0x07, 0x98, 0x4c, 0x55,
    0x0e, 0xd4, 0x34, 0xbb, 0x9b, 0x0d, 0x71, 0xe4, 0x45, 0x6b, 0x30, 0x4f,
    0x35, 0x9b, 0x8d, 0xb1, 0x13, 0xad, 0x14, 0xbc, 0xf5, 0x3e, 0x4e, 0x27,
    0x6d, 0x5f, 0x56, 0x2a, 0x90, 0xf3, 0xea, 0xce, 0xab, 0x03, 0x3b, 0xfe,
    0x38, 0x35, 0x09, 0xa5, 0xc5, 0xe9, 0x60, 0xa4, 0x94, 0x37, 0xb9, 0x7e,
    0xa6, 0x98, 0x9b, 0x06, 0xf6, 0x86, 0xbf, 0x50, 0xc7, 0xe1, 0x37, 0x33,
    0x24, 0xd8, 0x97, 0x47, 0xe0, 0x4b, 0x36, 0xaf, 0xcb, 0xf0, 0xe4, 0x07,
    0xdc, 0x4a, 0x8a, 0x60, 0xce, 0x01, 0xd0, 0x9e, 0xd5, 0x8f, 0x2d, 0x4a,
    0x0b, 0x99, 0xd9, 0x04, 0x8f, 0x2e, 0xfc, 0x75, 0xc1, 0x2f, 0xf5, 0x3b,
    0x7f, 0xe2, 0xf2, 0x65, 0x84, 0xd6, 0x0a, 0xee, 0x52, 0xbc, 0x1f, 0x48,
    0x4d, 0x0f, 0x99, 0xd9, 0xe8, 0x6b, 0x2c, 0x88, 0x2c, 0x5b, 0xe7, 0xc7,
    0x83, 0xf6, 0xb9, 0xc8, 0xd3, 0x5a, 0x71, 0xcf, 0x38, 0xfe, 0x1b, 0x03,
    0x6b, 0x38, 0x1f, 0x5e, 0x72, 0xed, 0x99, 0x28, 0xce, 0x65, 0x6f, 0xd8,
    0x98, 0xc2, 0xbf, 0x4e, 0xcc, 0x35, 0x17, 0x23, 0x52, 0x7d, 0x69, 0x2a,
    0x65, 0xc5, 0x54, 0x5c, 0x21, 0x80, 0xe6, 0x4f, 0x15, 0x20, 0xb9, 0x42,
    0xc6, 0xc3, 0x9c, 0x83, 0xa1, 0x01, 0x2e, 0x5e, 0x06, 0xaf, 0x35, 0xc0,
    0x7c, 0xfd, 0x32, 0x22, 0xa1, 0x4b, 0x6a, 0xc3, 0x3f, 0x3d, 0x7b, 0x5c,
    0x32, 0x80, 0x61, 0x7c, 0x62, 0xd3, 0x81, 0xfd, 0xf9, 0xda, 0x7b, 0xfc,
    0x04, 0xb5, 0x63, 0x2e, 0x2f, 0x02, 0x7d, 0x22, 0x61, 0xa5, 0x57, 0x13,
    0x8a, 0x84, 0xb5, 0x7c, 0x52, 0x74, 0x22, 0xbe, 0x2d, 0xe2, 0x4b, 0xc4,
    0x3b, 0x1c, 0x21, 0x99, 0xa5, 0x7e, 0x00, 0xe8, 0x31, 0xcc, 0x4c, 0xb8,
    0x2d, 0x83, 0x10, 0xf5, 0x3d, 0x2c, 0x7d, 0xfa, 0x6b, 0x29, 0x3a, 0x0e,
    0x2d, 0x81, 0x5e, 0xe3, 0x93, 0x4c, 0x51, 0x22, 0x86, 0x0f, 0xee, 0x42,
    0x31, 0x31, 0xb4, 0xbf, 0x2c, 0xb6, 0x66, 0x92, 0xc1, 0xdd, 0x0d, 0xa9,
    0xa4, 0xcd, 0x6d, 0xf1, 0x1e, 0xcc, 0x17, 0xe6, 0x4a, 0xc6, 0xbc, 0xa5,
    0xb6, 0xc9, 0x45, 0xf2, 0xa9, 0x62, 0xdd, 0x0d, 0x7e, 0x26, 0xc8, 0x95,
    0x08, 0x65, 0x72, 0xe9, 0xcb, 0xf3, 0x7f, 0x40, 0x3d, 0x39, 0xd7, 0xff,
    0xa2, 0xb3, 0x54, 0x85, 0xd5, 0xcf, 0xa6, 0x2a, 0x99, 0xc5, 0x65, 0xaa,
    0x8c, 0x1d, 0xe3, 0x12, 0xe8, 0xd7, 0x95, 0xfa, 0x34, 0x4b, 0xc8, 0x23,
    0x72, 0x33, 0x7f, 0x4e, 0x41, 0xae, 0xa7, 0xb2, 0xe6, 0xe9, 0x89, 0xcd,
    0x38, 0xf2, 0x00, 0x5a, 0xb3, 0x2d, 0x8c, 0x35, 0xaf, 0xbc, 0xae, 0x9e,
    0x0e, 0xd2, 0xd0, 0x3c, 0x1f, 0x9f, 0x0a, 0xf4, 0x29, 0xb0, 0xd8, 0xe8,
    0xd2, 0x99, 0xd7, 0x66, 0xdc, 0x07, 0x00, 0x50, 0x8e, 0x98, 0x6b, 0x3c,
    0x13, 0xa3, 0x8f, 0x8c, 0x03, 0x88, 0x0b, 0x18, 0x33, 0xda, 0xb4, 0xae,
    0x6f, 0xac, 0x36, 0xdb, 0x1b, 0x87, 0x2e, 0x26, 0x68, 0x5b, 0xc6, 0xa0,
    0x57, 0xf3, 0x06, 0xb1, 0x2f, 0xd7, 0x9c, 0x66, 0x19, 0xc2, 0x74, 0x7e,
    0xd7, 0x86, 0xd9, 0xa3, 0xe6, 0x9e, 0xaa, 0xdc, 0x85, 0xb4, 0xdd, 0x4c,
    0xdc, 0x8f, 0x6b, 0x10, 0x7f, 0xb6, 0x6c, 0xa0, 0x34, 0xf0, 0x65, 0x39,
    0xe9, 0xb2, 0x6e, 0x18, 0xff, 0xd5, 0xfc, 0x03, 0xb6, 0x0d, 0x6a, 0xf8,
    0x16, 0x56, 0x4a, 0x0c, 0xf0, 0x5f, 0x66, 0x47, 0x3f, 0x2f, 0xf7, 0x47,
    0xe9, 0x95, 0x3c, 0x22, 0x4b, 0x40, 0xae, 0xff, 0x88, 0xe8, 0x60, 0xd0,
    0x07, 0x7e, 0x85, 0x0b, 0x3a, 0x6e, 0x05, 0x44, 0x29, 0xec, 0x0d, 0x83,
    0xd0, 0x5c, 0xcd, 0x71, 0xd5, 0x24, 0x12, 0x55, 0x38, 0x20, 0x15, 0x6b,
    0xbd, 0x8b, 0x88, 0x8c, 0x30, 0xb3, 0x10, 0xae, 0x26, 0x5b, 0x8b, 0xb1,
    0x4d, 0x33, 0xe9, 0x9e, 0x48, 0x22, 0x58, 0x4e, 0xf8, 0xab, 0xca, 0x95,
    0x29, 0x2b, 0xcc, 0xd6, 0x81, 0xde, 0xf5, 0x2c, 0x68, 0x0b, 0xf5, 0x9c,
    0x94, 0xce, 0x99, 0x0e, 0xbf, 0xea, 0x6d, 0x74, 0x41, 0xdd, 0x3c, 0x13,
    0x1f, 0xd9, 0x0c, 0x13, 0xe5, 0xd1, 0xef, 0x2a, 0xe1, 0xf0, 0x70, 0x76,
    0x66, 0x9e, 0x50, 0xc9, 0xda, 0x58, 0xa1, 0x66, 0x06, 0x36, 0xab, 0xd3,
    0x68, 0xe3, 0x54, 0x52, 0xc5, 0x08, 0xe2, 0x42, 0x55, 0xfb, 0x8d, 0x48,
    0xa9, 0x08, 0x95, 0x9d, 0x66, 0x1c, 0x69, 0xda, 0x2e, 0x71, 0xc8, 0x7a,
    0xb2, 0x50, 0xd6, 0x49, 0xc0, 0x2e, 0x52, 0x60, 0x64, 0x95, 0x13, 0x63,
    0x21, 0x69, 0x2e, 0x5e, 0xd4, 0x8a, 0x3b, 0xa3, 0xf5, 0x34, 0xd0, 0x56,
    0x06, 0xb7, 0x96, 0xbe, 0x2c, 0x2c, 0x97, 0x7d, 0x83, 0x07, 0xde, 0xf5,
    0x46, 0xd5, 0xa4, 0xe0, 0x96, 0x88, 0x65, 0x5c, 0x08, 0x2a, 0x37, 0x8f,
    0x99, 0xa6, 0xaf, 0xc8, 0xde, 0xf4, 0xfe, 0x15, 0x1f, 0x2d, 0x32, 0x33,
    0x3a, 0x68, 0x7b, 0x7f, 0x8f, 0xa3, 0xa6, 0xa7, 0xaa, 0xb7, 0xbc, 0xc2,
    0xef, 0xf4, 0x0a, 0x24, 0x66, 0x70, 0x76, 0x82, 0x9a, 0x9f, 0xa6, 0xbe,
    0xd6, 0xdc, 0xe6, 0xeb, 0xf2, 0x35, 0x4b, 0x50, 0xa0, 0xbf, 0xdf, 0xfa,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x0b, 0x1e, 0x2d, 0x34
  };

  static const uint8_t kMLDSAVerifySignature[MLDSA44_SIGNATURE_BYTES] = {
    0x4b, 0x17, 0x18, 0xc2, 0xe9, 0x41, 0xfd, 0xe2, 0x33, 0x66, 0x89, 0x47,
    0xf9, 0xeb, 0x70, 0xe7, 0x6f, 0x4e, 0x42, 0xc8, 0x75, 0xba, 0x3f, 0xee,
    0x9b, 0xc7, 0xc5, 0xb0, 0xb6, 0x91, 0xfd, 0x39, 0x6c, 0xfa, 0xb0, 0x61,
    0x96, 0x6d, 0x68, 0x4b, 0xcb, 0x56, 0x29, 0xf8, 0x19, 0x5f, 0xbf, 0xbb,
    0x74, 0xe4, 0x32, 0xe5, 0xea, 0x9f, 0x6a, 0xd0, 0xba, 0xf9, 0x95, 0x88,
    0x25, 0x17, 0x82, 0x1e, 0x28, 0xee, 0xc0, 0xff, 0x4c, 0x52, 0x57, 0x27,
    0xdc, 0xb4, 0xde, 0x59, 0x72, 0x01, 0xad, 0xb3, 0x45, 0x75, 0xaa, 0xfe,
    0x96, 0x8b, 0x03, 0x6c, 0x14, 0x21, 0xde, 0x7e, 0x8e, 0x69, 0x35, 0x75,
    0x8b, 0x1a, 0x69, 0xa4, 0x00, 0x58, 0xec, 0xcc, 0x0d, 0x8d, 0xdb, 0x9b,
    0x50, 0x53, 0x8b, 0x70, 0xf6, 0xa5, 0xdc, 0x5e, 0x95, 0x6c, 0x69, 0x31,
    0x2d, 0x72, 0x0b, 0x47, 0x3d, 0x0b, 0x53, 0x75, 0x47, 0xfa, 0x6f, 0xb5,
    0x9c, 0x5d, 0x7a, 0xdf, 0x85, 0xb9, 0x18, 0x16, 0xa5, 0x27, 0x9e, 0x48,
    0x7f, 0xce, 0xe0, 0x0d, 0xcc, 0x53, 0x8d, 0x81, 0x15, 0x73, 0x36, 0x2b,
    0x4a, 0x48, 0xc6, 0x9d, 0x86, 0x29, 0xa4, 0xa3, 0x53, 0x11, 0x65, 0x43,
    0x0f, 0xba, 0x97, 0x16, 0x2e, 0xfb, 0x5d, 0x87, 0x7f, 0x5f, 0x5f, 0xd7,
    0xdb, 0x26, 0x8f, 0xa0, 0x65, 0xcc, 0xb3, 0x67, 0xbb, 0xbf, 0x52, 0xd7,
    0x21, 0x4c, 0x4a, 0x40, 0x02, 0x6f, 0x5d, 0xa8, 0xce, 0x00, 0x42, 0x52,
    0xee, 0x80, 0x27, 0xc3, 0x5c, 0xed, 0x54, 0x51, 0xf2, 0x52, 0xd3, 0x97,
    0xb9, 0x1e, 0xae, 0x84, 0x68, 0x0a, 0xa3, 0x23, 0xb6, 0x71, 0x7a, 0x2e,
    0x1a, 0x8c, 0xa8, 0xbf, 0x5e, 0x92, 0x38, 0x6f, 0x88, 0xf5, 0xaa, 0xc4,
    0x35, 0x21, 0x94, 0x34, 0x66, 0x1c, 0xb9, 0xd2, 0xc4, 0x2c, 0xea, 0xf4,
    0xc3, 0x20, 0x73, 0xc8, 0xb3, 0x72, 0x63, 0x7c, 0x46, 0x92, 0x92, 0x96,
    0xb0, 0x12, 0xd5, 0x5d, 0x73, 0x2d, 0x8d, 0x3a, 0xad, 0xe8, 0x91, 0x70,
    0x85, 0x08, 0x85, 0x9f, 0x6d, 0xc9, 0xa7, 0xa5, 0x59, 0xa4, 0xd7, 0xff,
    0xe4, 0x53, 0xf8, 0x1d, 0x4c, 0x75, 0x68, 0xa1, 0x0b, 0xa2, 0xc8, 0xfc,
    0xcd, 0xbf, 0x1f, 0xf7, 0xec, 0x4b, 0x63, 0xd4, 0x0c, 0x96, 0xa8, 0x51,
    0xf0, 0x23, 0x07, 0x6c, 0x19, 0xde, 0xf5, 0x81, 0xbf, 0xee, 0xe0, 0x88,
    0xc5, 0x63, 0xc9, 0x0e, 0x49, 0xa0, 0xc0, 0xe0, 0xb8, 0x2a, 0x04, 0x21,
    0x4f, 0x98, 0x5c, 0xa7, 0x9f, 0x12, 0xf7, 0x88, 0x7a, 0x17, 0xdb, 0x1f,
    0x71, 0xdb, 0x67, 0xb6, 0x58, 0xd3, 0x92, 0x70, 0x11, 0x02, 0xc9, 0x42,
    0xcb, 0x23, 0x8e, 0x32, 0xa5, 0x62, 0xf4, 0x23, 0x88, 0xaf, 0x02, 0x05,
    0x6f, 0xd8, 0x32, 0xd0, 0x24, 0xb0, 0x14, 0x34, 0x54, 0x25, 0xe1, 0xb5,
    0xb7, 0x25, 0x37, 0x91, 0x59, 0x74, 0xb7, 0x84, 0x41, 0xe3, 0x53, 0x4d,
    0xaa, 0xd8, 0xb5, 0xc3, 0x30, 0xc7, 0xf5, 0x75, 0x68, 0x74, 0x6c, 0xe7,
    0xc4, 0xbe, 0x90, 0x5c, 0x1b, 0xea, 0x3e, 0x80, 0xd6, 0x56, 0x37, 0x20,
    0xcd, 0x97, 0x59, 0xf1, 0x84, 0xeb, 0x1e, 0x4a, 0x03, 0x4f, 0x09, 0x29,
    0x07, 0x60, 0x3f, 0x32, 0x51, 0xec, 0x2e, 0x00, 0x0e, 0xce, 0xec, 0x0f,
    0xae, 0x13, 0x05, 0xf8, 0x61, 0x71, 0x61, 0xb1, 0x41, 0x38, 0x94, 0x57,
    0xda, 0x41, 0x26, 0xe1, 0xc0, 0x20, 0xcc, 0xd6, 0x7f, 0xfa, 0x34, 0x03,
    0x72, 0x8e, 0x0c, 0x68, 0x65, 0xfb, 0x8a, 0xda, 0x9a, 0x4e, 0x74, 0x15,
    0x21, 0x16, 0xa1, 0x50, 0x0e, 0x98, 0xca, 0x0e, 0xcf, 0xb7, 0xb8, 0x1d,
    0xe5, 0x19, 0x11, 0x22, 0x9b, 0x5b, 0xc3, 0x1c, 0x8e, 0x6c, 0xff, 0x95,
    0x22, 0x79, 0x65, 0x75, 0xd1, 0x78, 0x66, 0xe4, 0x89, 0x2b, 0x4b, 0xdb,
    0x82, 0x50, 0x2f, 0x28, 0x27, 0xf5, 0xfa, 0x50, 0xc6, 0x79, 0x73, 0xda,
    0x3a, 0x7e, 0x7f, 0x9a, 0x28, 0x21, 0x24, 0x29, 0x1a, 0xf1, 0x4e, 0x0e,
    0xc9, 0x26, 0x97, 0x62, 0x88, 0x50, 0xec, 0x79, 0x8e, 0x2b, 0x23, 0xf9,
    0x86, 0x1b, 0x04, 0x0e, 0x6e, 0x82, 0x80, 0x41, 0x4e, 0x2c, 0xee, 0x6e,
    0x82, 0xbd, 0xf9, 0x64, 0x89, 0x5d, 0x80, 0x00, 0xa4, 0x01, 0x8d, 0xf7,
    0x39, 0xea, 0x9f, 0xa2, 0x6c, 0xea, 0x9e, 0xf4, 0x9c, 0xe0, 0x47, 0x0c,
    0x90, 0x7e, 0x59, 0xe0, 0x19, 0x33, 0xd6, 0x3a, 0x57, 0xa4, 0xc2, 0x94,
    0x5b, 0xc6, 0x25, 0xc6, 0x1d, 0xbf, 0x45, 0x26, 0xeb, 0xb6, 0x8d, 0x43,
    0x78, 0xb1, 0x98, 0x2a, 0x1c, 0xfa, 0x7f, 0x5e, 0x8c, 0xdb, 0x8e, 0x8a,
    0x46, 0x19, 0x5c, 0x5e, 0xde, 0x8b, 0xc3, 0xf1, 0x49, 0xfd, 0xb6, 0x54,
    0x9d, 0xb1, 0x3f, 0x3f, 0x02, 0x04, 0x59, 0x4d, 0x12, 0x54, 0x70, 0x5f,
    0x32, 0xb8, 0x83, 0xb4, 0xd6, 0xd7, 0x5c, 0x41, 0x05, 0x56, 0x1f, 0x32,
    0xbc, 0xa6, 0x31, 0xa9, 0xe3, 0x18, 0x33, 0xde, 0x26, 0x53, 0x02, 0xf1,
    0xb4, 0xc8, 0x27, 0x8d, 0x28, 0xda, 0xfd, 0xdb, 0xfc, 0xf3, 0x2b, 0xe4,
    0x29, 0xa1, 0x2b, 0x25, 0xba, 0x34, 0x63, 0x3c, 0x2c, 0x2f, 0x55, 0xdf,
    0xa6, 0x74, 0x0c, 0x4e, 0x26, 0xf2, 0x87, 0x04, 0x81, 0x33, 0x61, 0x72,
    0x42, 0x53, 0x52, 0x37, 0xd6, 0x99, 0xf7, 0xb9, 0x95, 0x42, 0xe1, 0x60,
    0x0f, 0xaf, 0x7c, 0x4d, 0x4c, 0xbc, 0x30, 0x90, 0x61, 0xb3, 0x0b, 0x1c,
    0xd8, 0x7d, 0xc4, 0x88, 0x2e, 0xdd, 0xb1, 0x8a, 0x30, 0xe6, 0xb3, 0x3b,
    0x42, 0x23, 0xd1, 0x18, 0x43, 0x2a, 0x79, 0x51, 0x61, 0xdc, 0xb6, 0xf0,
    0xf7, 0x85, 0xf8, 0x56, 0x97, 0xf0, 0xdd, 0xd7, 0x26, 0x6a, 0xcc, 0x53,
    0x59, 0x9a, 0xed, 0xfe, 0x36, 0x03, 0xa5, 0xd9, 0xb3, 0x30, 0xc9, 0x37,
    0xaf, 0x0a, 0x38, 0xa3, 0x26, 0x8c, 0x49, 0x1a, 0xdf, 0x8d, 0xf0, 0x05,
    0x07, 0xfe, 0x4f, 0x44, 0xd9, 0x66, 0x76, 0x86, 0x55, 0xb6, 0x3d, 0x4b,
    0xf6, 0xb7, 0x10, 0x3f, 0x82, 0x10, 0x7a, 0x03, 0xa7, 0xc7, 0xb0, 0xc6,
    0x62, 0x98, 0x51, 0x62, 0xbe, 0x71, 0xac, 0x58, 0x52, 0xfd, 0x8d, 0xac,
    0x56, 0xb1, 0x9c, 0xd4, 0xd2, 0xcc, 0x81, 0x2f, 0xe6, 0x5a, 0xad, 0x01,
    0x0c, 0x5b, 0x10, 0x44, 0x95, 0x6b, 0xb2, 0xc9, 0xb4, 0x45, 0x6b, 0x51,
    0x42, 0xef, 0x8d, 0x55, 0x9f, 0x8d, 0xb2, 0xbe, 0xce, 0x05, 0xf7, 0x99,
    0x68, 0x90, 0x5a, 0x98, 0x68, 0xa0, 0x02, 0x48, 0xb5, 0x64, 0xab, 0x94,
    0xcc, 0xa0, 0x2a, 0x44, 0x41, 0x8a, 0x8a, 0x14, 0xb0, 0x45, 0x47, 0x08,
    0x77, 0x37, 0x25, 0xea, 0x3a, 0xb0, 0xbf, 0x28, 0x77, 0x2e, 0xe7, 0x63,
    0xc1, 0x75, 0xae, 0x5c, 0x70, 0xcd, 0x41, 0x72, 0x96, 0x51, 0x35, 0x33,
    0xbc, 0x3a, 0x3e, 0xd5, 0x69, 0x11, 0x8b, 0x1f, 0xc4, 0x76, 0xf8, 0xc0,
    0x50, 0x87, 0x36, 0x23, 0xce, 0x20, 0x6a, 0xc2, 0x18, 0xf2, 0x04, 0x76,
    0x43, 0xf3, 0xcb, 0xa0, 0xa0, 0x94, 0x90, 0x5c, 0x51, 0x3c, 0xa1, 0x3d,
    0x31, 0x96, 0xfb, 0x1f, 0x13, 0x35, 0x7b, 0x5e, 0x74, 0x08, 0x60, 0x19,
    0x11, 0x1a, 0x9e, 0x75, 0x1e, 0x64, 0x35, 0xb4, 0x5b, 0x81, 0x0a, 0x05,
    0xbb, 0xfe, 0xcf, 0xbc, 0xcf, 0x2a, 0x06, 0xf4, 0x8c, 0xee, 0xc0, 0x10,
    0x03, 0xa5, 0x6f, 0x24, 0xcc, 0xb7, 0xa6, 0x8f, 0x0d, 0x2d, 0x52, 0x0e,
    0x5e, 0xa8, 0xb6, 0x93, 0x8e, 0x00, 0x3e, 0xfb, 0x4e, 0x80, 0xf1, 0xbc,
    0xac, 0xbb, 0xce, 0xd5, 0x81, 0x94, 0xcb, 0xfa, 0xd1, 0x07, 0x53, 0x72,
    0xae, 0x67, 0x70, 0xa9, 0x93, 0x5b, 0xa1, 0xa4, 0x02, 0xde, 0x5b, 0xf2,
    0x9f, 0x4f, 0xc3, 0x76, 0xea, 0xa2, 0xee, 0x41, 0xfe, 0x60, 0x3e, 0xa8,
    0x2b, 0x3b, 0x0a, 0x62, 0xa7, 0x06, 0x36, 0xc8, 0xdb, 0xb4, 0x8f, 0x47,
    0xc5, 0x44, 0x29, 0x5e, 0x4f, 0x0c, 0xdd, 0xd4, 0x16, 0x2e, 0xc4, 0xc9,
    0x0b, 0xd1, 0x2c, 0x78, 0xc7, 0x80, 0x36, 0xb8, 0xdc, 0xcf, 0x57, 0x6d,
    0xac, 0xf7, 0x16, 0x33, 0x46, 0x39, 0x03, 0x09, 0x09, 0x3e, 0xda, 0x78,
    0xee, 0x82, 0x80, 0x24, 0xe5, 0x2a, 0x02, 0xe2, 0x76, 0xe9, 0x51, 0x0c,
    0xa3, 0xbf, 0x9d, 0x09, 0xd3, 0x36, 0xaa, 0xe2, 0xa3, 0x3a, 0xbb, 0x6d,
    0x27, 0x53, 0x4a, 0xb4, 0x08, 0x22, 0x49, 0x7d, 0xf5, 0xbc, 0x16, 0x70,
    0x65, 0x5c, 0x01, 0x01, 0x5e, 0xb5, 0xd2, 0x5c, 0x37, 0xb3, 0x2d, 0x7e,
    0x6f, 0x00, 0x63, 0x57, 0x6e, 0x17, 0x34, 0xf4, 0x74, 0x62, 0x85, 0x4b,
    0xaf, 0x93, 0xb7, 0xd7, 0x9d, 0x6b, 0x3f, 0x18, 0x6e, 0x31, 0x4c, 0x0e,
    0xec, 0xd3, 0x09, 0x35, 0x04, 0x94, 0xeb, 0x9e, 0xc8, 0x1f, 0xec, 0xd1,
    0xc6, 0xef, 0x2e, 0xfa, 0xe9, 0x86, 0xfa, 0xc4, 0xae, 0xae, 0x5f, 0x5f,
    0xee, 0xef, 0x51, 0xd4, 0x32, 0x3a, 0x49, 0x3a, 0x61, 0x44, 0x60, 0x0b,
    0x56, 0x4d, 0xa0, 0x91, 0x7c, 0x56, 0xb3, 0x34, 0x94, 0x37, 0x5b, 0x92,
    0x2c, 0x18, 0xdd, 0x91, 0xfd, 0x77, 0xb5, 0xf9, 0xdc, 0xd7, 0x2a, 0x75,
    0x7b, 0xcd, 0x71, 0xef, 0x04, 0x0a, 0xf2, 0x21, 0xf0, 0xb0, 0x5e, 0x3c,
    0x37, 0xb1, 0xc7, 0xb2, 0x73, 0xb1, 0xab, 0x53, 0x74, 0x6b, 0xe8, 0x4c,
    0x5a, 0x44, 0xf3, 0x93, 0xa7, 0x9c, 0x53, 0x84, 0x15, 0x08, 0x30, 0x05,
    0x49, 0xbb, 0xc6, 0xf8, 0xe7, 0x4a, 0x66, 0x41, 0xf4, 0x51, 0xe5, 0x3a,
    0xd7, 0xb4, 0x19, 0xac, 0xf2, 0x41, 0x2d, 0x6d, 0x43, 0x19, 0xa5, 0x61,
    0x16, 0x02, 0xb0, 0xba, 0xb4, 0xae, 0xfb, 0x42, 0x3a, 0x60, 0x37, 0x4a,
    0xcf, 0x20, 0x02, 0x0c, 0x8e, 0xb4, 0x2f, 0x1f, 0x3c, 0xce, 0xb9, 0x0e,
    0xe6, 0x60, 0xd5, 0xe1, 0xbb, 0x6f, 0xaf, 0xa0, 0xf2, 0x6c, 0x50, 0x3a,
    0x34, 0x84, 0x93, 0x9c, 0xcc, 0x6e, 0xa4, 0xbc, 0x96, 0x93, 0x19, 0x22,
    0x0c, 0x67, 0x10, 0xd0, 0x09, 0xcd, 0x77, 0xab, 0x11, 0xd8, 0x71, 0x8e,
    0xd4, 0x44, 0x59, 0x66, 0xb7, 0x6b, 0xc0, 0xd6, 0xcb, 0x2e, 0xd0, 0x55,
    0x2e, 0xe6, 0x78, 0x0f, 0x6a, 0xd0, 0x58, 0xcc, 0x09, 0xeb, 0xa9, 0xc2,
    0xa1, 0x67, 0xd9, 0xe8, 0x4b, 0x2b, 0xa8, 0xbe, 0x4f, 0xd3, 0x3c, 0x42,
    0x36, 0x77, 0xb6, 0x92, 0xf4, 0x38, 0xf6, 0x98, 0x8c, 0xc2, 0xb1, 0xe6,
    0x36, 0x3b, 0x78, 0x46, 0xfd, 0x00, 0xc4, 0x45, 0x02, 0x86, 0xc5, 0xbe,
    0xab, 0x53, 0x8c, 0xbb, 0x49, 0x6b, 0x4c, 0x51, 0xb8, 0x29, 0x11, 0x96,
    0x9a, 0x14, 0x34, 0x6a, 0x4b, 0x5c, 0xa8, 0x5b, 0x48, 0xc2, 0x5f, 0xc8,
    0x47, 0xc1, 0x10, 0xbb, 0x05, 0x42, 0x12, 0x32, 0xbb, 0xe5, 0x28, 0xf4,
    0x7b, 0xdb, 0x41, 0xbf, 0xc2, 0xb4, 0xe5, 0x26, 0x92, 0x7e, 0xbb, 0xf3,
    0x20, 0xee, 0x73, 0x98, 0xe3, 0x07, 0x29, 0x33, 0x98, 0x2c, 0x79, 0xb2,
    0xea, 0xa0, 0x2e, 0x28, 0xb4, 0xa7, 0x98, 0x17, 0xe2, 0x15, 0xff, 0x39,
    0x4b, 0xdc, 0x1d, 0x7b, 0x05, 0xd1, 0x15, 0xad, 0xc8, 0x63, 0x16, 0x11,
    0x65, 0x94, 0x6a, 0x85, 0x6f, 0x02, 0xff, 0xd5, 0x25, 0x7f, 0x02, 0xfc,
    0x4d, 0x00, 0x9b, 0x2e, 0x7e, 0x87, 0x63, 0x35, 0x92, 0xa2, 0xb4, 0xe1,
    0x08, 0xf3, 0xd8, 0x37, 0x86, 0x60, 0x3f, 0x43, 0x4b, 0x6a, 0x52, 0xb5,
    0x49, 0xb3, 0xfb, 0x4c, 0x61, 0x29, 0x78, 0x95, 0x54, 0xab, 0x4f, 0x41,
    0xa0, 0xe9, 0x0f, 0x59, 0x4f, 0x75, 0x51, 0xa1, 0xf8, 0xce, 0xc9, 0x82,
    0xdb, 0x83, 0xe3, 0xc5, 0x9d, 0x98, 0x22, 0x66, 0x64, 0x7b, 0xdf, 0x26,
    0x69, 0xa0, 0x84, 0x2a, 0x2b, 0x92, 0x47, 0xaa, 0x1f, 0xa4, 0x9b, 0x98,
    0x2a, 0x32, 0x76, 0x06, 0x8f, 0xd1, 0x62, 0x0a, 0x7c, 0x53, 0x53, 0x65,
    0x10, 0x16, 0xc5, 0x49, 0x42, 0x02, 0x55, 0xab, 0x33, 0xd1, 0x66, 0x69,
    0x15, 0x39, 0x91, 0xb3, 0x70, 0xde, 0x0b, 0xa1, 0x64, 0xd7, 0xed, 0x16,
    0x42, 0xee, 0x07, 0x3b, 0x19, 0xbd, 0xa5, 0x6a, 0x35, 0x2a, 0xe9, 0x5b,
    0x64, 0x33, 0x53, 0x58, 0x31, 0x06, 0x7d, 0x5f, 0xe6, 0x05, 0x88, 0x56,
    0x77, 0xf0, 0x16, 0xce, 0x20, 0xc0, 0x94, 0x44, 0x22, 0xae, 0x69, 0x42,
    0x30, 0xc6, 0xf2, 0xa9, 0x40, 0x81, 0x2c, 0xeb, 0x8c, 0x44, 0xcb, 0x0c,
    0x45, 0x0f, 0x23, 0xb2, 0x3f, 0x96, 0x15, 0x5d, 0x6d, 0x56, 0xde, 0x58,
    0xef, 0x86, 0x90, 0x58, 0x80, 0x5b, 0xd2, 0xd9, 0x8c, 0x07, 0xae, 0x9e,
    0x57, 0xcb, 0x0d, 0xc3, 0xe2, 0x99, 0xe3, 0x5f, 0x9c, 0x38, 0xca, 0xda,
    0x55, 0x73, 0xaf, 0xf6, 0xfd, 0x28, 0xbb, 0x20, 0x50, 0xbf, 0x98, 0x77,
    0x07, 0x48, 0xad, 0x92, 0x5f, 0xe2, 0xaa, 0x62, 0x3f, 0x24, 0xd3, 0xfe,
    0xc7, 0xb0, 0x91, 0x4b, 0x24, 0x02, 0x6f, 0x03, 0x3b, 0x8b, 0x13, 0xde,
    0x44, 0x93, 0xbb, 0xbe, 0x02, 0xac, 0xd8, 0x56, 0x83, 0xa5, 0xfc, 0x9b,
    0x33, 0x27, 0x28, 0xbd, 0xea, 0x5e, 0xa3, 0x17, 0x19, 0xd7, 0x51, 0x68,
    0x19, 0x8d, 0xd3, 0x2b, 0xb6, 0x75, 0x8f, 0xb3, 0xfd, 0x08, 0x16, 0x69,
    0xe5, 0x23, 0x43, 0x66, 0xa3, 0xc7, 0x3e, 0xf9, 0xe4, 0xa8, 0xe1, 0x1c,
    0x00, 0x4e, 0x06, 0xa2, 0xb0, 0xbe, 0xa6, 0xd6, 0x24, 0x4c, 0x2b, 0xd2,
    0xfd, 0x22, 0xdf, 0x49, 0x55, 0xc6, 0x6a, 0x6c, 0x5f, 0x7a, 0xe8, 0x2a,
    0xd7, 0xe4, 0x5d, 0x44, 0xc6, 0x7e, 0x85, 0x84, 0x32, 0xd1, 0x37, 0xff,
    0xe8, 0xcc, 0xab, 0x05, 0x99, 0xed, 0x43, 0x7c, 0xa9, 0x56, 0x60, 0x8d,
    0xb4, 0x53, 0xc3, 0x45, 0x56, 0x06, 0xea, 0x2d, 0x38, 0xbc, 0x4c, 0x16,
    0x7b, 0xd2, 0xfb, 0x0d, 0xae, 0xac, 0x1f, 0x32, 0x7a, 0x6b, 0x05, 0x87,
    0x0c, 0x6f, 0x8f, 0x1c, 0x1a, 0x26, 0x2f, 0xf1, 0xc7, 0x70, 0x48, 0x44,
    0x71, 0xc7, 0xe8, 0x8f, 0x96, 0x5b, 0x1d, 0x7f, 0xcf, 0xf5, 0x90, 0xb5,
    0x30, 0xb5, 0xd5, 0x33, 0x18, 0x56, 0xb6, 0x45, 0x17, 0x56, 0xb8, 0x3c,
    0xf8, 0x4e, 0xf5, 0xca, 0x11, 0x70, 0x39, 0x8c, 0xf7, 0x46, 0xbb, 0x37,
    0xb6, 0xb5, 0xd5, 0x05, 0x25, 0xb4, 0xe1, 0x5a, 0x75, 0x13, 0x36, 0x41,
    0x9e, 0x4a, 0x78, 0x16, 0x53, 0xf0, 0x2a, 0x77, 0x4e, 0x03, 0x1d, 0x10,
    0x41, 0x10, 0x97, 0x5f, 0xcf, 0x42, 0x25, 0xea, 0xff, 0xb6, 0x1d, 0x97,
    0x99, 0x53, 0xc3, 0x60, 0x17, 0x67, 0xf2, 0x20, 0x50, 0xb5, 0x2a, 0x5a,
    0xfc, 0xa8, 0x5d, 0xae, 0x98, 0x82, 0xaf, 0x3d, 0x74, 0x05, 0xce, 0x50,
    0x30, 0x51, 0xf5, 0xaf, 0x0b, 0x39, 0x00, 0x0b, 0xa9, 0xda, 0xf4, 0xd2,
    0x30, 0xad, 0xb6, 0x19, 0x6b, 0x29, 0xcf, 0x7a, 0x35, 0x55, 0xf4, 0x43,
    0x70, 0x8b, 0x57, 0x37, 0xf2, 0x2e, 0xcf, 0x4a, 0xe2, 0xb7, 0xb2, 0x73,
    0x71, 0x83, 0xe3, 0x2e, 0x60, 0x7e, 0x0b, 0x11, 0xe5, 0x4b, 0xb2, 0x65,
    0xbd, 0x85, 0x2d, 0x91, 0x80, 0x12, 0xa5, 0xf4, 0x2c, 0xe8, 0x27, 0xbb,
    0x60, 0x9b, 0x8c, 0x02, 0x62, 0xf5, 0x51, 0x61, 0x1f, 0xf0, 0x0a, 0x8e,
    0x3a, 0x0c, 0x0a, 0x0a, 0x58, 0x42, 0x2a, 0x5f, 0xfa, 0x3f, 0xed, 0xb5,
    0x62, 0x8a, 0xfb, 0xd0, 0xb7, 0x1c, 0x50, 0xab, 0xad, 0xa5, 0x0a, 0x1d,
    0x7c, 0x79, 0x90, 0x38, 0xb5, 0x02, 0x4d, 0x2a, 0xeb, 0xf3, 0xe3, 0x96,
    0x92, 0x21, 0x26, 0xc6, 0x3a, 0xbe, 0xfa, 0x01, 0x2a, 0xc9, 0x7b, 0x4b,
    0x40, 0xec, 0xfe, 0x9a, 0x4f, 0xf4, 0xbf, 0xb0, 0x83, 0xe9, 0x9e, 0x89,
    0xc4, 0xba, 0x20, 0x1b, 0x9e, 0x3d, 0x75, 0x51, 0xe2, 0x24, 0xc7, 0x1c,
    0x87, 0x1e, 0x94, 0x90, 0xd9, 0xe8, 0xb9, 0x4d, 0x7a, 0x58, 0xd0, 0xdd,
    0xe8, 0x98, 0x1c, 0x0b, 0xde, 0xf4, 0x42, 0xa3, 0x46, 0x15, 0xe4, 0xb4,
    0xd8, 0x8f, 0xcf, 0x0c, 0x5d, 0x16, 0xcb, 0xc1, 0x99, 0x15, 0x35, 0x93,
    0xd0, 0xae, 0x43, 0x06, 0x69, 0x34, 0xd2, 0x33, 0xdf, 0x9d, 0x38, 0x13,
    0xb9, 0x89, 0x8d, 0xa4, 0xbd, 0x11, 0xe9, 0x39, 0x1f, 0x94, 0x06, 0x4e,
    0x95, 0x2a, 0x92, 0xcb, 0x35, 0x06, 0x4b, 0x79, 0x40, 0xf5, 0x17, 0x72,
    0xe3, 0xc2, 0xf0, 0xe7, 0x3e, 0x0d, 0x07, 0x41, 0x21, 0x4f, 0x35, 0x2a,
    0x28, 0x9d, 0x57, 0x8e, 0xe8, 0xe6, 0x5b, 0x20, 0x4b, 0x65, 0xe0, 0xc4,
    0x50, 0x49, 0x60, 0x3e, 0x39, 0xa9, 0x20, 0x8d, 0x2c, 0x16, 0x66, 0x02,
    0xbc, 0xac, 0x47, 0xf8, 0x04, 0xfb, 0xdb, 0x3f, 0xa9, 0xc2, 0xcb, 0x48,
    0x6c, 0x0a, 0xc6, 0x2f, 0x17, 0xf9, 0x8a, 0x25, 0x76, 0x42, 0xd7, 0x7e,
    0x18, 0x8e, 0x62, 0xad, 0xd4, 0xdb, 0xaf, 0x3b, 0xfe, 0xee, 0x34, 0x18,
    0xb4, 0xc5, 0x3d, 0x1b, 0x64, 0xaa, 0xde, 0x44, 0x44, 0x66, 0xa3, 0x0e,
    0xf9, 0xf1, 0x7d, 0xe7, 0x65, 0x47, 0xa3, 0xb9, 0x75, 0x2d, 0x93, 0xde,
    0xbb, 0x67, 0xda, 0x22, 0x36, 0xd7, 0x52, 0x6a, 0xbb, 0x7f, 0x33, 0x27,
    0xdc, 0x2c, 0xc3, 0x15, 0x99, 0x38, 0x5e, 0x86, 0xfb, 0x8d, 0x20, 0x6c,
    0x26, 0xc9, 0x61, 0x9f, 0x4b, 0x76, 0x2c, 0x12, 0x7b, 0x0e, 0xfa, 0xc7,
    0x91, 0x5e, 0xdb, 0xd4, 0x3c, 0x32, 0xca, 0xf7, 0x4f, 0x75, 0xda, 0x87,
    0xb8, 0xb5, 0x7c, 0xa4, 0x63, 0x65, 0xbd, 0x0c, 0x58, 0xbd, 0xa3, 0xa5,
    0x89, 0xa6, 0x5d, 0xf1, 0x6b, 0xae, 0x0e, 0xd1, 0x01, 0x09, 0x2a, 0x31,
    0x50, 0x5d, 0x7b, 0x83, 0x87, 0x97, 0x9b, 0xa6, 0xb7, 0xbf, 0xc3, 0xcf,
    0xda, 0xe1, 0xe3, 0x0e, 0x15, 0x17, 0x18, 0x1b, 0x1c, 0x29, 0x42, 0x5b,
    0x70, 0x79, 0xba, 0xbb, 0xc0, 0xc1, 0xc8, 0xed, 0x0f, 0x13, 0x15, 0x17,
    0x1d, 0x1f, 0x2f, 0x4e, 0x50, 0x51, 0x5e, 0x75, 0x7d, 0x88, 0xb0, 0xb6,
    0xbc, 0xc2, 0xcc, 0xe2, 0xf0, 0x05, 0x2e, 0x46, 0x6a, 0x9b, 0xa4, 0xc9,
    0xca, 0xfd, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x13, 0x24, 0x39, 0x43
  };

  // Keygen
  uint8_t public_key[MLDSA44_PUBLIC_KEY_BYTES] = {0};
  uint8_t private_key[MLDSA44_PRIVATE_KEY_BYTES] = {0};

  if (!ml_dsa_44_keypair_internal_no_self_test(public_key, private_key, kMLDSAKeyGenSeed) ||
      !check_test(kMLDSAKeyGenPublicKey, public_key, sizeof(public_key), "ML-DSA-keyGen") ||
      !check_test(kMLDSAKeyGenPrivateKey, private_key, sizeof(private_key), "ML-DSA-keyGen")) {
    goto err;
  }

  // Sign
  uint8_t signature[MLDSA44_SIGNATURE_BYTES] = {0};
  size_t sig_len = MLDSA44_SIGNATURE_BYTES;
  size_t mlen_int = 32;

  if (!ml_dsa_44_sign_internal_no_self_test(private_key, signature, &sig_len, kMLDSASignPlaintext,
                                            mlen_int, NULL, 0, kMLDSASigGenSeed) ||
      !check_test(kMLDSASignSignature, signature, sizeof(signature), "ML-DSA-sign")) {
    goto err;
  }

  // Verify
  if (!ml_dsa_44_verify_internal_no_self_test(public_key, kMLDSAVerifySignature, sig_len, kMLDSAVerifyPlaintext,
                                              mlen_int, NULL, 0)) {
    AWS_LC_FIPS_failure("ML-DSA-verify failed");
    goto err;
    }

    ret = 1;
  err:
    return ret;
}

static OPENSSL_NOINLINE int boringssl_self_test_eddsa(void) {
  int ret = 0;

  static const uint8_t kEd25519PrivateKey[ED25519_PRIVATE_KEY_SEED_LEN] = {
      0xb3, 0x99, 0x05, 0xbf, 0x43, 0x0b, 0x2a, 0xd2, 0x1d, 0xb6, 0x5d,
      0x49, 0xa6, 0xab, 0x03, 0xc1, 0x7d, 0xdb, 0x72, 0xe7, 0xa9, 0x8e,
      0xb9, 0x8f, 0xae, 0x59, 0x91, 0x7a, 0xe2, 0x5f, 0x92, 0x14};
  static const uint8_t kEd25519PublicKey[ED25519_PUBLIC_KEY_LEN] = {
      0xe7, 0x75, 0xcf, 0x0e, 0x33, 0x48, 0x52, 0xa7, 0xe6, 0x99, 0xbe,
      0xba, 0x13, 0xbc, 0x24, 0xf8, 0x32, 0xf3, 0xc2, 0xa3, 0xa0, 0x3d,
      0xc9, 0x3c, 0x42, 0xb5, 0x92, 0x76, 0x15, 0xa5, 0x46, 0xba};

  uint8_t ed25519_private_key[ED25519_PRIVATE_KEY_LEN] = {0};
  OPENSSL_memcpy(ed25519_private_key, kEd25519PrivateKey,
                 ED25519_PRIVATE_KEY_SEED_LEN);
  OPENSSL_memcpy(ed25519_private_key + ED25519_PRIVATE_KEY_SEED_LEN,
                 kEd25519PublicKey, ED25519_PUBLIC_KEY_LEN);
  uint8_t ed25519_out_sig[ED25519_SIGNATURE_LEN] = {0};


  static const uint8_t kEd25519SignMessage[32] = {
      0x19, 0x61, 0xd1, 0xd5, 0x2d, 0x8c, 0x04, 0x5f, 0xdf, 0xc1, 0xc6, 0x82,
      0xb3, 0x5f, 0x07, 0xaa, 0xe1, 0xd3, 0xb6, 0xe5, 0x48, 0x63, 0x98, 0x30,
      0xee, 0xd9, 0x29, 0xbc, 0x12, 0x2d, 0x79, 0x9f};
  static const uint8_t kEd25519SignSignature[ED25519_SIGNATURE_LEN] = {
      0xa8, 0x81, 0xe8, 0xd9, 0x5d, 0xdb, 0xd5, 0xd1, 0x47, 0x60, 0xaf, 0x4e,
      0xcf, 0xce, 0x45, 0x96, 0xf7, 0x2e, 0x04, 0xd7, 0xee, 0xcc, 0xb9, 0xc6,
      0xa1, 0x93, 0xe2, 0x4d, 0xd7, 0x35, 0xb1, 0x3c, 0x18, 0xa5, 0x34, 0xc7,
      0x79, 0x31, 0x45, 0x46, 0x9d, 0xd1, 0x6f, 0x0c, 0x5e, 0x03, 0x71, 0xa3,
      0xfb, 0x85, 0x06, 0x35, 0x97, 0xc0, 0x92, 0x45, 0x97, 0xcb, 0x42, 0x75,
      0x60, 0xdb, 0x2a, 0x0b};

  if (!ED25519_sign_no_self_test(ed25519_out_sig, kEd25519SignMessage,
                                 sizeof(kEd25519SignMessage),
                                 ed25519_private_key) ||
      !check_test(kEd25519SignSignature, ed25519_out_sig,
                  ED25519_SIGNATURE_LEN, "ED25519-sign")) {
    AWS_LC_FIPS_failure("ED25519-sign failed");
    goto err;
  }

  static const uint8_t kEd25519VerifyMessage[32] = {
    0x71, 0x3a, 0x7a, 0xde, 0x3d, 0x9e, 0x10, 0x9f, 0x9f, 0xc1, 0x9b, 0xc3,
    0x24, 0xe0, 0x41, 0x72, 0xee, 0x7e, 0x4d, 0x4a, 0xc6, 0x36, 0x69, 0xb7,
    0xbc, 0xdb, 0xd6, 0xd2, 0xde, 0x87, 0xdf, 0x0e};
  static const uint8_t kEd25519VerifySignature[ED25519_SIGNATURE_LEN] = {
    0x44, 0xf2, 0x38, 0xf7, 0xea, 0x71, 0x54, 0xce, 0xdd, 0x95, 0x63, 0x11,
    0x44, 0x07, 0x8f, 0xfe, 0xc6, 0x55, 0x93, 0x8f, 0x73, 0xe2, 0x96, 0x76,
    0x72, 0x8b, 0x40, 0x0f, 0x8f, 0x46, 0xc8, 0x04, 0x8d, 0x5b, 0xf3, 0xab,
    0x12, 0x43, 0x42, 0xeb, 0xae, 0x54, 0xb6, 0xe0, 0x4f, 0x3f, 0x16, 0x7b,
    0x5e, 0xe8, 0xbd, 0xcf, 0xec, 0x9b, 0xe6, 0xff, 0x65, 0xbc, 0xc6, 0x9a,
    0x78, 0x89, 0x67, 0x0a};
  if (!ED25519_verify_no_self_test(kEd25519VerifyMessage, sizeof(kEd25519VerifyMessage),
                                   kEd25519VerifySignature, kEd25519PublicKey)) {
    AWS_LC_FIPS_failure("ED25519-verify failed");
    goto err;
  }

  ret = 1;
err:
  return ret;
}

static OPENSSL_NOINLINE int boringssl_self_test_hasheddsa(void) {
  int ret = 0;

  static const uint8_t kEd25519PrivateKey[ED25519_PRIVATE_KEY_SEED_LEN] = {
      0xc3, 0x53, 0x7a, 0x4f, 0x31, 0x5e, 0xc5, 0x8f, 0x5d, 0xe4, 0xc2,
      0x8d, 0xc5, 0x32, 0x7c, 0x79, 0xfb, 0x40, 0x7c, 0xb6, 0x70, 0xbe,
      0x05, 0xf1, 0x1b, 0x0f, 0x70, 0x06, 0x40, 0x70, 0x21, 0x27
  };
  static const uint8_t kEd25519PublicKey[ED25519_PUBLIC_KEY_LEN] = {
      0x63, 0x36, 0xa6, 0x15, 0xdf, 0x2d, 0xe9, 0x3b, 0x8d, 0xab, 0x78,
      0xe9, 0x7b, 0x82, 0x7b, 0x2d, 0x5c, 0xeb, 0xeb, 0xd7, 0xfa, 0xa7,
      0x7e, 0x3d, 0x97, 0xea, 0xf3, 0x6b, 0x12, 0xf7, 0x22, 0xe3
  };
  static const uint8_t kEd25519Context[32] = {
    0x76, 0x34, 0x2c, 0x15, 0xb7, 0x11, 0x97, 0x5d, 0x86, 0xd0, 0x11, 0xdd,
    0x28, 0xec, 0x76, 0xf9, 0xb9, 0xe7, 0x2a, 0xb1, 0x5a, 0x50, 0x15, 0xb0,
    0xdd, 0xca, 0xfa, 0x8f, 0xed, 0x54, 0x80, 0x66
  };

  static const uint8_t kEd25519SignMessage[32] = {
    0xf9, 0xbf, 0xec, 0x63, 0xc4, 0xe0, 0x73, 0xfa, 0x97, 0x1a, 0x80, 0x49,
    0x91, 0x47, 0xd0, 0x0a, 0xcd, 0x26, 0xaa, 0xe1, 0xff, 0x03, 0x64, 0xdb,
    0x20, 0xf8, 0xa7, 0xa4, 0x95, 0x4d, 0xb3, 0x87
  };
  static const uint8_t kEd25519SignSignature[ED25519_SIGNATURE_LEN] = {
    0x0b, 0x93, 0x3d, 0x3f, 0x59, 0x00, 0xe3, 0xa1, 0xe5, 0x39, 0x47, 0xce,
    0x97, 0x32, 0xc7, 0x01, 0x40, 0x37, 0xe9, 0xc9, 0x4b, 0x71, 0xcd, 0x3a,
    0xfb, 0x60, 0x46, 0xaa, 0x29, 0xfe, 0xa9, 0xbb, 0xd8, 0x1c, 0x50, 0x54,
    0x10, 0x64, 0xc6, 0x59, 0xd0, 0x07, 0x5f, 0xb3, 0x8c, 0x8b, 0x42, 0x0f,
    0x81, 0x48, 0x68, 0x2d, 0xc9, 0xf8, 0x38, 0x43, 0x55, 0x10, 0x5c, 0x39,
    0x70, 0xd2, 0x06, 0x09
};

  uint8_t ed25519_private_key[ED25519_PRIVATE_KEY_LEN] = {0};
  OPENSSL_memcpy(ed25519_private_key, kEd25519PrivateKey,
                 ED25519_PRIVATE_KEY_SEED_LEN);
  OPENSSL_memcpy(ed25519_private_key + ED25519_PRIVATE_KEY_SEED_LEN,
                 kEd25519PublicKey, ED25519_PUBLIC_KEY_LEN);
  uint8_t ed25519_out_sig[ED25519_SIGNATURE_LEN] = {0};

  if (!ED25519ph_sign_no_self_test(
          &ed25519_out_sig[0], kEd25519SignMessage, sizeof(kEd25519SignMessage),
          ed25519_private_key, kEd25519Context, sizeof(kEd25519Context)) ||
      !check_test(kEd25519SignSignature, ed25519_out_sig,
                  ED25519_SIGNATURE_LEN, "ED25519ph-sign")) {
    goto err;
  }

  static const uint8_t kEd25519VerifyMessage[32] = {
    0x36, 0xc7, 0xf4, 0x5a, 0x29, 0xa6, 0x84, 0xa8, 0x01, 0x37, 0x53, 0xb1,
    0xc6, 0x10, 0x09, 0x79, 0x1f, 0xbc, 0x6e, 0xd4, 0xaf, 0x81, 0x31, 0xaa,
    0x4b, 0xc2, 0x76, 0x3d, 0x7f, 0xd5, 0xf7, 0x50
  };
  static const uint8_t kEd25519VerifySignature[ED25519_SIGNATURE_LEN] = {
    0x10, 0x1b, 0xcc, 0xa2, 0x56, 0xef, 0x62, 0x0b, 0xb0, 0x87, 0x59, 0x2e,
    0x91, 0x73, 0x36, 0xd8, 0x54, 0x2b, 0x71, 0x72, 0x8e, 0x2a, 0x27, 0x48,
    0xc5, 0x5c, 0x71, 0x9b, 0x82, 0x5d, 0xad, 0x45, 0x21, 0xbf, 0xb3, 0x75,
    0x62, 0x4b, 0x27, 0xff, 0xf8, 0x1c, 0xbf, 0x71, 0x65, 0xe5, 0xba, 0x4a,
    0x98, 0xe8, 0xc4, 0x51, 0xb3, 0xc3, 0xc2, 0xfa, 0x23, 0x27, 0x8f, 0x2b,
    0xb7, 0x45, 0x81, 0x07
};
  if (!ED25519ph_verify_no_self_test(kEd25519VerifyMessage, sizeof(kEd25519VerifyMessage),
                                   kEd25519VerifySignature, kEd25519PublicKey, kEd25519Context, sizeof(kEd25519Context))) {
    AWS_LC_FIPS_failure("ED25519ph-verify failed");
    goto err;
  }

  ret = 1;
err:
  return ret;
}

#if defined(BORINGSSL_FIPS)

static void run_self_test_rsa(void) {
  if (!boringssl_self_test_rsa()) {
    AWS_LC_FIPS_failure("RSA self tests failed");
  }
}

DEFINE_STATIC_ONCE(g_self_test_once_rsa);

void boringssl_ensure_rsa_self_test(void) {
  CRYPTO_once(g_self_test_once_rsa_bss_get(), run_self_test_rsa);
}

static void run_self_test_ecc(void) {
  if (!boringssl_self_test_ecc()) {
    AWS_LC_FIPS_failure("ECC self tests failed");
  }
}

DEFINE_STATIC_ONCE(g_self_test_once_ecc);

void boringssl_ensure_ecc_self_test(void) {
  CRYPTO_once(g_self_test_once_ecc_bss_get(), run_self_test_ecc);
}

static void run_self_test_ffdh(void) {
  if (!boringssl_self_test_ffdh()) {
    AWS_LC_FIPS_failure("FFDH self tests failed");
  }
}

DEFINE_STATIC_ONCE(g_self_test_once_ffdh);

void boringssl_ensure_ffdh_self_test(void) {
  CRYPTO_once(g_self_test_once_ffdh_bss_get(), run_self_test_ffdh);
}

static void run_self_test_ml_kem(void) {
  if (!boringssl_self_test_ml_kem()) {
    AWS_LC_FIPS_failure("ML-KEM self tests failed");
  }
}

DEFINE_STATIC_ONCE(g_self_test_once_ml_kem);

void boringssl_ensure_ml_kem_self_test(void) {
  CRYPTO_once(g_self_test_once_ml_kem_bss_get(), run_self_test_ml_kem);
}

static void run_self_test_ml_dsa(void) {
  if (!boringssl_self_test_ml_dsa()) {
    AWS_LC_FIPS_failure("ML-DSA self tests failed");
  }
}

DEFINE_STATIC_ONCE(g_self_test_once_ml_dsa);

void boringssl_ensure_ml_dsa_self_test(void) {
  CRYPTO_once(g_self_test_once_ml_dsa_bss_get(), run_self_test_ml_dsa);
}

static void run_self_test_eddsa(void) {
  if (!boringssl_self_test_eddsa()) {
    AWS_LC_FIPS_failure("EdDSA self tests failed");
  }
}

DEFINE_STATIC_ONCE(g_self_test_once_eddsa);

void boringssl_ensure_eddsa_self_test(void) {
  CRYPTO_once(g_self_test_once_eddsa_bss_get(), run_self_test_eddsa);
}

static void run_self_test_hasheddsa(void) {
  if (!boringssl_self_test_hasheddsa()) {
    AWS_LC_FIPS_failure("EdDSA-ph self tests failed");
  }
}

DEFINE_STATIC_ONCE(g_self_test_once_hasheddsa);

void boringssl_ensure_hasheddsa_self_test(void) {
  CRYPTO_once(g_self_test_once_hasheddsa_bss_get(), run_self_test_hasheddsa);
}

#endif  // BORINGSSL_FIPS


// Startup self tests.
//
// These tests are run at process start when in FIPS mode. Note that the SHA256
// and HMAC-SHA256 tests are also used from bcm.c, so they can't be static.

OPENSSL_NOINLINE int boringssl_self_test_sha256(void) {
  static const uint8_t kInput[16] = {
      0xff, 0x3b, 0x85, 0x7d, 0xa7, 0x23, 0x6a, 0x2b,
      0xaa, 0x0f, 0x39, 0x6b, 0x51, 0x52, 0x22, 0x17,
  };
  static const uint8_t kPlaintextSHA256[32] = {
      0x7f, 0xe4, 0xd5, 0xf1, 0xa1, 0xe3, 0x82, 0x87, 0xd9, 0x58, 0xf5,
      0x11, 0xc7, 0x1d, 0x5e, 0x27, 0x5e, 0xcc, 0xd2, 0x66, 0xcf, 0xb9,
      0xc8, 0xc6, 0x60, 0xd8, 0x92, 0x1e, 0x57, 0xfd, 0x46, 0x75,
  };
  uint8_t output[SHA256_DIGEST_LENGTH];

  // SHA-256 KAT
  SHA256(kInput, sizeof(kInput), output);
  return check_test(kPlaintextSHA256, output, sizeof(kPlaintextSHA256),
                    "SHA-256 KAT");
}

static OPENSSL_NOINLINE int boringssl_self_test_sha512(void) {
  static const uint8_t kInput[16] = {
      0x21, 0x25, 0x12, 0xf8, 0xd2, 0xad, 0x83, 0x22,
      0x78, 0x1c, 0x6c, 0x4d, 0x69, 0xa9, 0xda, 0xa1,
  };
  static const uint8_t kPlaintextSHA512[64] = {
      0x29, 0x3c, 0x94, 0x35, 0x4e, 0x98, 0x83, 0xe5, 0xc2, 0x78, 0x36,
      0x7a, 0xe5, 0x18, 0x90, 0xbf, 0x35, 0x41, 0x01, 0x64, 0x19, 0x8d,
      0x26, 0xeb, 0xe1, 0xf8, 0x2f, 0x04, 0x8e, 0xfa, 0x8b, 0x2b, 0xc6,
      0xb2, 0x9d, 0x5d, 0x46, 0x76, 0x5a, 0xc8, 0xb5, 0x25, 0xa3, 0xea,
      0x52, 0x84, 0x47, 0x6d, 0x6d, 0xf4, 0xc9, 0x71, 0xf3, 0x3d, 0x89,
      0x4c, 0x3b, 0x20, 0x8c, 0x5b, 0x75, 0xe8, 0xf8, 0x7c,
  };
  uint8_t output[SHA512_DIGEST_LENGTH];

  // SHA-512 KAT
  SHA512(kInput, sizeof(kInput), output);
  return check_test(kPlaintextSHA512, output, sizeof(kPlaintextSHA512),
                    "SHA-512 KAT");
}

OPENSSL_NOINLINE int boringssl_self_test_hmac_sha256(void) {
  static const uint8_t kInput[16] = {
      0xda, 0xd9, 0x12, 0x93, 0xdf, 0xcf, 0x2a, 0x7c,
      0x8e, 0xcd, 0x13, 0xfe, 0x35, 0x3f, 0xa7, 0x5b,
  };
  static const uint8_t kPlaintextHMACSHA256[32] = {
      0x36, 0x5f, 0x5b, 0xd5, 0xf5, 0xeb, 0xfd, 0xc7, 0x6e, 0x53, 0xa5,
      0x73, 0x6d, 0x73, 0x20, 0x13, 0xaa, 0xd3, 0xbc, 0x86, 0x4b, 0xb8,
      0x84, 0x94, 0x16, 0x46, 0x88, 0x9c, 0x48, 0xee, 0xa9, 0x0e,
  };
  uint8_t output[EVP_MAX_MD_SIZE];

  unsigned output_len = 0;
  HMAC(EVP_sha256(), kInput, sizeof(kInput), kInput, sizeof(kInput), output,
       &output_len);
  return output_len == sizeof(kPlaintextHMACSHA256) &&
         check_test(kPlaintextHMACSHA256, output, sizeof(kPlaintextHMACSHA256),
                    "HMAC-SHA-256 KAT");
}

static OPENSSL_NOINLINE int boringssl_self_test_hkdf_sha256(void) {
  static const uint8_t kHKDF_ikm_tc1[] = {
      0x58, 0x3e, 0xa3, 0xcf, 0x8f, 0xcf, 0xc8, 0x08, 0x73, 0xcc, 0x7b, 0x88,
      0x00, 0x9d, 0x4a, 0xed, 0x07, 0xd8, 0xd8, 0x88, 0xae, 0x98, 0x76, 0x8d,
      0xca, 0x07, 0xcb, 0x1e, 0x4b, 0x33, 0x1e, 0xb9
  };
  static const uint8_t kHKDF_salt_tc1[] = {
      0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b,
      0x0c
  };
  static const uint8_t kHKDF_info_tc1[] = {
      0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7, 0xf8, 0xf9
  };
  static const uint8_t kHKDF_okm_tc1_sha256[] = {
      0xca, 0x5e, 0x64, 0x10, 0xe7, 0xa5, 0x23, 0x32, 0xfe, 0x0a, 0xb3, 0x60,
      0x12, 0x12, 0xa7, 0xd3, 0xdb, 0xdf, 0x55, 0xa1, 0x62, 0xaf, 0x42, 0xa5,
      0xda, 0xf3, 0x8b, 0x94, 0xf2, 0x45, 0x23, 0x47, 0x7e, 0x88, 0x0d, 0xd7,
      0x11, 0x50, 0x86, 0x84, 0xcc, 0x21
  };

  uint8_t output[sizeof(kHKDF_okm_tc1_sha256)];
  HKDF(output, sizeof(output), EVP_sha256(),
       kHKDF_ikm_tc1, sizeof(kHKDF_ikm_tc1),
       kHKDF_salt_tc1, sizeof(kHKDF_salt_tc1),
       kHKDF_info_tc1, sizeof(kHKDF_info_tc1));
  return check_test(kHKDF_okm_tc1_sha256, output, sizeof(output),
                    "HKDF-SHA-256 KAT");
}

static OPENSSL_NOINLINE int boringssl_self_test_sha3_256(void) {
  // From: SHA3_256ShortMsg.txt
  // Len = 128
  // Msg = d83c721ee51b060c5a41438a8221e040
  // MD  = b87d9e4722edd3918729ded9a6d03af8256998ee088a1ae662ef4bcaff142a96
  static const uint8_t kInput[16] = {
    0xd8, 0x3c, 0x72, 0x1e, 0xe5, 0x1b, 0x06, 0x0c,
    0x5a, 0x41, 0x43, 0x8a, 0x82, 0x21, 0xe0, 0x40,
};
  static const uint8_t kPlaintextSHA3_256[SHA3_256_DIGEST_LENGTH] = {
    0xb8, 0x7d, 0x9e, 0x47, 0x22, 0xed, 0xd3, 0x91, 0x87, 0x29, 0xde,
    0xd9, 0xa6, 0xd0, 0x3a, 0xf8, 0x25, 0x69, 0x98, 0xee, 0x08, 0x8a,
    0x1a, 0xe6, 0x62, 0xef, 0x4b, 0xca, 0xff, 0x14, 0x2a, 0x96,
};
  uint8_t output[SHA3_256_DIGEST_LENGTH];

  // SHA3-256 KAT
  SHA3_256(kInput, sizeof(kInput), output);
  return check_test(kPlaintextSHA3_256, output, sizeof(kPlaintextSHA3_256),
                    "SHA3-256 KAT");
}

static OPENSSL_NOINLINE int boringssl_self_test_fast(void) {
  static const uint8_t kAESKey[16] = {'B', 'o', 'r', 'i', 'n', 'g', 'C', 'r',
                                      'y', 'p', 't', 'o', ' ', 'K', 'e', 'y'};
  // Older versions of the gcc release build on ARM will optimize out the
  // assembly label for kAESIV, if we define it with {0}. The assembler
  // will set the value of kAESIV to another static constant in the
  // fipsmodule, kZeroIV, since they have the same value, then skip creating
  // a label for kAESIV. This does not mesh well with delocate.go, since we
  // use these labels to determine if the symbol is "known". Expanding the
  // definition of kAESIV gets around the unwanted assembler optimization.
  static const uint8_t kAESIV[16] = {
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  };

  EVP_AEAD_CTX aead_ctx;
  EVP_AEAD_CTX_zero(&aead_ctx);
  int ret = 0;

  AES_KEY aes_key;
  uint8_t aes_iv[16];
  uint8_t output[256];

  // AES-CBC Encryption KAT
  static const uint8_t kAESCBCEncPlaintext[32] = {
      0x07, 0x86, 0x09, 0xa6, 0xc5, 0xac, 0x25, 0x44, 0x69, 0x9a, 0xdf,
      0x68, 0x2f, 0xa3, 0x77, 0xf9, 0xbe, 0x8a, 0xb6, 0xae, 0xf5, 0x63,
      0xe8, 0xc5, 0x6a, 0x36, 0xb8, 0x4f, 0x55, 0x7f, 0xad, 0xd3,
  };
  static const uint8_t kAESCBCEncCiphertext[sizeof(kAESCBCEncPlaintext)] = {
      0x56, 0x46, 0xc1, 0x41, 0xf4, 0x13, 0xd6, 0xff, 0x62, 0x92, 0x41,
      0x7a, 0x26, 0xc6, 0x86, 0xbd, 0x30, 0x5f, 0xb6, 0x57, 0xa7, 0xd2,
      0x50, 0x3a, 0xc5, 0x5e, 0x8e, 0x93, 0x40, 0xf2, 0x10, 0xd8,
  };
  memcpy(aes_iv, kAESIV, sizeof(kAESIV));
  if (AES_set_encrypt_key(kAESKey, 8 * sizeof(kAESKey), &aes_key) != 0) {
    fprintf(stderr, "AES_set_encrypt_key failed.\n");
    goto err;
  }
  AES_cbc_encrypt(kAESCBCEncPlaintext, output, sizeof(kAESCBCEncPlaintext),
                  &aes_key, aes_iv, AES_ENCRYPT);
  if (!check_test(kAESCBCEncCiphertext, output, sizeof(kAESCBCEncCiphertext),
                  "AES-CBC-encrypt KAT")) {
    goto err;
  }

  // AES-CBC Decryption KAT
  static const uint8_t kAESCBCDecCiphertext[32] = {
      0x34, 0x7a, 0xa5, 0xa0, 0x24, 0xb2, 0x82, 0x57, 0xb3, 0x65, 0x10,
      0xbe, 0x58, 0x3d, 0x4f, 0x47, 0xad, 0xb7, 0xbb, 0xee, 0xdc, 0x60,
      0x05, 0xbb, 0xbd, 0x0d, 0x0a, 0x9f, 0x06, 0xbb, 0x7b, 0x10,
  };
  static const uint8_t kAESCBCDecPlaintext[sizeof(kAESCBCDecCiphertext)] = {
      0x51, 0xa7, 0xa0, 0x1f, 0x6b, 0x79, 0x6c, 0xcd, 0x48, 0x03, 0xa1,
      0x41, 0xdc, 0x56, 0xa6, 0xc2, 0x16, 0xb5, 0xd1, 0xd3, 0xb7, 0x06,
      0xb2, 0x25, 0x6f, 0xa6, 0xd0, 0xd2, 0x0e, 0x6f, 0x19, 0xb5,
  };
  memcpy(aes_iv, kAESIV, sizeof(kAESIV));
  if (AES_set_decrypt_key(kAESKey, 8 * sizeof(kAESKey), &aes_key) != 0) {
    fprintf(stderr, "AES_set_decrypt_key failed.\n");
    goto err;
  }
  AES_cbc_encrypt(kAESCBCDecCiphertext, output, sizeof(kAESCBCDecCiphertext),
                  &aes_key, aes_iv, AES_DECRYPT);
  if (!check_test(kAESCBCDecPlaintext, output, sizeof(kAESCBCDecPlaintext),
                  "AES-CBC-decrypt KAT")) {
    goto err;
  }

  size_t out_len;
  uint8_t nonce[EVP_AEAD_MAX_NONCE_LENGTH];
  OPENSSL_memset(nonce, 0, sizeof(nonce));
  if (!EVP_AEAD_CTX_init(&aead_ctx, EVP_aead_aes_128_gcm(), kAESKey,
                         sizeof(kAESKey), 0, NULL)) {
    fprintf(stderr, "EVP_AEAD_CTX_init for AES-128-GCM failed.\n");
    goto err;
  }

  // AES-GCM Encryption KAT
  static const uint8_t kAESGCMEncPlaintext[32] = {
      0x8f, 0xcc, 0x40, 0x99, 0x80, 0x8e, 0x75, 0xca, 0xaf, 0xf5, 0x82,
      0x89, 0x88, 0x48, 0xa8, 0x8d, 0x80, 0x8b, 0x55, 0xab, 0x4e, 0x93,
      0x70, 0x79, 0x7d, 0x94, 0x0b, 0xe8, 0xcc, 0x1d, 0x78, 0x84,
  };
  static const uint8_t kAESGCMCiphertext[sizeof(kAESGCMEncPlaintext) + 16] = {
      0x87, 0x7b, 0xd5, 0x8d, 0x96, 0x3e, 0x4b, 0xe6, 0x64, 0x94, 0x40, 0x2f,
      0x61, 0x9b, 0x7e, 0x56, 0x52, 0x7d, 0xa4, 0x5a, 0xf9, 0xa6, 0xe2, 0xdb,
      0x1c, 0x63, 0x2e, 0x97, 0x93, 0x0f, 0xfb, 0xed, 0xb5, 0x9e, 0x1c, 0x20,
      0xb2, 0xb0, 0x58, 0xda, 0x48, 0x07, 0x2d, 0xbd, 0x96, 0x0d, 0x34, 0xc6,
  };
  if (!EVP_AEAD_CTX_seal(&aead_ctx, output, &out_len, sizeof(output), nonce,
                         EVP_AEAD_nonce_length(EVP_aead_aes_128_gcm()),
                         kAESGCMEncPlaintext, sizeof(kAESGCMEncPlaintext), NULL,
                         0) ||
      !check_test(kAESGCMCiphertext, output, sizeof(kAESGCMCiphertext),
                  "AES-GCM-encrypt KAT")) {
    fprintf(stderr, "EVP_AEAD_CTX_seal for AES-128-GCM failed.\n");
    goto err;
  }

  // AES-GCM Decryption KAT
  static const uint8_t kAESGCMDecCiphertext[48] = {
      0x35, 0xf3, 0x05, 0x8f, 0x87, 0x57, 0x60, 0xff, 0x09, 0xd3, 0x12, 0x0f,
      0x70, 0xc4, 0xbc, 0x9e, 0xd7, 0xa8, 0x68, 0x72, 0xe1, 0x34, 0x52, 0x20,
      0x21, 0x76, 0xf7, 0x37, 0x1a, 0xe0, 0x4f, 0xaa, 0xe1, 0xdd, 0x39, 0x19,
      0x20, 0xf5, 0xd1, 0x39, 0x53, 0xd8, 0x96, 0x78, 0x59, 0x94, 0x82, 0x3c,
  };
  static const uint8_t kAESGCMDecPlaintext[sizeof(kAESGCMDecCiphertext) - 16] =
      {
          0x3d, 0x44, 0x90, 0x9b, 0x91, 0xe7, 0x5e, 0xd3, 0xc2, 0xb2, 0xd0,
          0xa9, 0x99, 0x17, 0x6a, 0x45, 0x05, 0x5e, 0x99, 0x83, 0x56, 0x01,
          0xc0, 0x82, 0x40, 0x81, 0xd2, 0x48, 0x45, 0xf2, 0xcc, 0xc3,
      };
  if (!EVP_AEAD_CTX_open(&aead_ctx, output, &out_len, sizeof(output), nonce,
                         EVP_AEAD_nonce_length(EVP_aead_aes_128_gcm()),
                         kAESGCMDecCiphertext, sizeof(kAESGCMDecCiphertext),
                         NULL, 0) ||
      !check_test(kAESGCMDecPlaintext, output, sizeof(kAESGCMDecPlaintext),
                  "AES-GCM-decrypt KAT")) {
    AWS_LC_FIPS_failure("AES-GCM-decrypt KAT failed because EVP_AEAD_CTX_open failed");
    goto err;
  }

  // SHA-1 KAT
  static const uint8_t kSHA1Input[16] = {
      0x13, 0x2f, 0xd9, 0xba, 0xd5, 0xc1, 0x82, 0x62,
      0x63, 0xba, 0xfb, 0xb6, 0x99, 0xf7, 0x07, 0xa5,
  };
  static const uint8_t kSHA1Digest[20] = {
      0x94, 0x19, 0x55, 0x93, 0x0a, 0x58, 0x29, 0x38, 0xeb, 0xf5,
      0x09, 0x11, 0x6d, 0x1a, 0xfd, 0x0f, 0x1e, 0x11, 0xe3, 0xcb,
  };
  SHA1(kSHA1Input, sizeof(kSHA1Input), output);
  if (!check_test(kSHA1Digest, output, sizeof(kSHA1Digest),
                  "SHA-1 KAT")) {
    goto err;
  }

  if (!boringssl_self_test_sha512() ||
      !boringssl_self_test_sha3_256() ||
      !boringssl_self_test_hkdf_sha256()) {
    goto err;
  }

  // DBRG KAT
  static const uint8_t kDRBGEntropy[48] = {
      0xc4, 0xda, 0x07, 0x40, 0xd5, 0x05, 0xf1, 0xee, 0x28, 0x0b, 0x95, 0xe5,
      0x8c, 0x49, 0x31, 0xac, 0x6d, 0xe8, 0x46, 0xa0, 0x15, 0x2f, 0xbb, 0x4a,
      0x3f, 0x17, 0x4c, 0xf4, 0x78, 0x7a, 0x4f, 0x1a, 0x40, 0xc2, 0xb5, 0x0b,
      0xab, 0xe1, 0x4a, 0xae, 0x53, 0x0b, 0xe5, 0x88, 0x6d, 0x91, 0x0a, 0x27,
  };
  static const uint8_t kDRBGPersonalization[18] = {
      'B', 'C', 'M', 'P', 'e', 'r', 's', 'o', 'n',
      'a', 'l', 'i', 'z', 'a', 't', 'i', 'o', 'n'};
  static const uint8_t kDRBGAD[16] = {'B', 'C', 'M', ' ', 'D', 'R', 'B', 'G',
                                      ' ', 'K', 'A', 'T', ' ', 'A', 'D', ' '};
  static const uint8_t kDRBGOutput[64] = {
      0x19, 0x1f, 0x2b, 0x49, 0x76, 0x85, 0xfd, 0x51, 0xb6, 0x56, 0xbc,
      0x1c, 0x7d, 0xd5, 0xdd, 0x44, 0x76, 0xa3, 0x5e, 0x17, 0x9b, 0x8e,
      0xb8, 0x98, 0x65, 0x12, 0xca, 0x35, 0x6c, 0xa0, 0x6f, 0xa0, 0x22,
      0xe4, 0xf6, 0xd8, 0x43, 0xed, 0x4e, 0x2d, 0x97, 0x39, 0x43, 0x3b,
      0x57, 0xfc, 0x23, 0x3f, 0x71, 0x0a, 0xe0, 0xed, 0xfe, 0xd5, 0xb8,
      0x67, 0x7a, 0x00, 0x39, 0xb2, 0x6e, 0xa9, 0x25, 0x97,
  };
  static const uint8_t kDRBGEntropy2[48] = {
      0xc7, 0x16, 0x1c, 0xa3, 0x6c, 0x23, 0x09, 0xb7, 0x16, 0xe9, 0x85, 0x9b,
      0xb9, 0x6c, 0x6d, 0x49, 0xbd, 0xc8, 0x35, 0x21, 0x03, 0xa1, 0x8c, 0xd2,
      0x4e, 0xf4, 0x2e, 0xc9, 0x7e, 0xf4, 0x6b, 0xf4, 0x46, 0xeb, 0x1a, 0x45,
      0x76, 0xc1, 0x86, 0xe9, 0x35, 0x18, 0x03, 0x76, 0x3a, 0x79, 0x12, 0xfe,
  };
  static const uint8_t kDRBGReseedOutput[64] = {
      0x00, 0xf2, 0x05, 0xaa, 0xfd, 0x11, 0x6c, 0x77, 0xbc, 0x81, 0x86,
      0x99, 0xca, 0x51, 0xcf, 0x80, 0x15, 0x9f, 0x02, 0x9e, 0x0b, 0xcd,
      0x26, 0xc8, 0x4b, 0x87, 0x8a, 0x15, 0x1a, 0xdd, 0xf2, 0xf3, 0xeb,
      0x94, 0x0b, 0x08, 0xc8, 0xc9, 0x57, 0xa4, 0x0b, 0x4b, 0x0f, 0x13,
      0xde, 0x7c, 0x0c, 0x6a, 0xac, 0x34, 0x4a, 0x9a, 0xf2, 0xd0, 0x83,
      0x02, 0x05, 0x17, 0xc9, 0x81, 0x8f, 0x2a, 0x81, 0x92,
  };
  CTR_DRBG_STATE drbg;
  if (!CTR_DRBG_init(&drbg, kDRBGEntropy, kDRBGPersonalization,
                     sizeof(kDRBGPersonalization)) ||
      !CTR_DRBG_generate(&drbg, output, sizeof(kDRBGOutput), kDRBGAD,
                         sizeof(kDRBGAD)) ||
      !check_test(kDRBGOutput, output, sizeof(kDRBGOutput),
                  "DRBG Generate KAT") ||
      !CTR_DRBG_reseed(&drbg, kDRBGEntropy2, kDRBGAD, sizeof(kDRBGAD)) ||
      !CTR_DRBG_generate(&drbg, output, sizeof(kDRBGReseedOutput), kDRBGAD,
                         sizeof(kDRBGAD)) ||
      !check_test(kDRBGReseedOutput, output, sizeof(kDRBGReseedOutput),
                  "DRBG-reseed KAT")) {
    AWS_LC_FIPS_failure("CTR-DRBG failed");
    goto err;
  }
  CTR_DRBG_clear(&drbg);

  CTR_DRBG_STATE kZeroDRBG;
  memset(&kZeroDRBG, 0, sizeof(kZeroDRBG));
  if (!check_test(&kZeroDRBG, &drbg, sizeof(drbg), "DRBG Clear KAT")) {
    goto err;
  }

  // TLS KDF KAT
  static const uint8_t kTLSSecret[32] = {
      0xab, 0xc3, 0x65, 0x7b, 0x09, 0x4c, 0x76, 0x28, 0xa0, 0xb2, 0x82,
      0x99, 0x6f, 0xe7, 0x5a, 0x75, 0xf4, 0x98, 0x4f, 0xd9, 0x4d, 0x4e,
      0xcc, 0x2f, 0xcf, 0x53, 0xa2, 0xc4, 0x69, 0xa3, 0xf7, 0x31,
  };
  static const char kTLSLabel[] = "FIPS self test";
  static const uint8_t kTLSSeed1[16] = {
      0x8f, 0x0d, 0xe8, 0xb6, 0x90, 0x8f, 0xb1, 0xd2,
      0x6d, 0x51, 0xf4, 0x79, 0x18, 0x63, 0x51, 0x65,
  };
  static const uint8_t kTLSSeed2[16] = {
      0x7d, 0x24, 0x1a, 0x9d, 0x3c, 0x59, 0xbf, 0x3c,
      0x31, 0x1e, 0x2b, 0x21, 0x41, 0x8d, 0x32, 0x81,
  };
  static const uint8_t kTLSOutput[32] = {
      0xe2, 0x1d, 0xd6, 0xc2, 0x68, 0xc7, 0x57, 0x03, 0x2c, 0x2c, 0xeb,
      0xbb, 0xb8, 0xa9, 0x7d, 0xe9, 0xee, 0xe6, 0xc9, 0x47, 0x83, 0x0a,
      0xbd, 0x11, 0x60, 0x5d, 0xd5, 0x2c, 0x47, 0xb6, 0x05, 0x88,
  };
  uint8_t tls_output[sizeof(kTLSOutput)];
  if (!CRYPTO_tls1_prf(EVP_sha256(), tls_output, sizeof(tls_output), kTLSSecret,
                       sizeof(kTLSSecret), kTLSLabel, sizeof(kTLSLabel),
                       kTLSSeed1, sizeof(kTLSSeed1), kTLSSeed2,
                       sizeof(kTLSSeed2)) ||
      !check_test(kTLSOutput, tls_output, sizeof(kTLSOutput), "TLS-KDF KAT")) {
    goto err;
  }

  static const uint8_t kPBKDF2Password[] = {
    'A', 'W', 'S', '-', 'L', 'C', 'F', 'I', 'P', 'S', 'p', 'a', 's', 's', 'w',
    'o', 'r', 'd'
  };
  static const uint8_t kPBKDF2Salt[] = {
    's', 'a', 'l', 't', 'S', 'A', 'L', 'T', 's', 'a', 'l', 't', 'S', 'A', 'L',
    'T', 's', 'a', 'l', 't', 'S', 'A', 'L', 'T', 's', 'a', 'l', 't', 'S', 'A',
    'L', 'T', 's', 'a', 'l', 't'
  };
  const unsigned kPBKDF2Iterations = 2;
  static const uint8_t kPBKDF2DerivedKey[] = {
    0xc6, 0xac, 0x07, 0x79, 0xe4, 0xa1, 0x17, 0xc9, 0x22, 0x28, 0x7f, 0x5e,
    0x10, 0xe7, 0xee, 0x6b, 0xa7, 0x4d, 0x8b, 0x19, 0x51, 0x9b, 0x4c, 0xc7,
    0x38
  };
  uint8_t pbkdf2_output[sizeof(kPBKDF2DerivedKey)];
  if (!PKCS5_PBKDF2_HMAC((const char *)kPBKDF2Password, sizeof(kPBKDF2Password),
                         kPBKDF2Salt, sizeof(kPBKDF2Salt), kPBKDF2Iterations,
                         EVP_sha256(), sizeof(kPBKDF2DerivedKey),
                         pbkdf2_output) ||
      !check_test(kPBKDF2DerivedKey, pbkdf2_output, sizeof(kPBKDF2DerivedKey),
                  "PBKDF2 KAT")) {
    goto err;
  }

  static const uint8_t kSSKDFDigestInfo[] = {
      0xaf, 0x85, 0xce, 0x3e, 0xa3, 0x03, 0x35, 0x0a, 0x2a, 0xa1, 0x31,
      0xb8, 0x79, 0xea, 0x95, 0x4e, 0x1e, 0xe1, 0xe6, 0x85, 0xd2, 0xbf,
      0x9d, 0xb7, 0x95, 0x42, 0x02, 0xe2, 0xf5, 0xec, 0x30, 0x96};
  static const uint8_t kSSKDFDigestSharedSecret[] = {
      0x39, 0xa1, 0xe2, 0xb3, 0x89, 0x9e, 0x87, 0xef, 0xec, 0xf6, 0x27,
      0x12, 0x82, 0xd8, 0xf8, 0x00, 0x8f, 0x25, 0x26, 0x86, 0xdd, 0x35,
      0xbf, 0xc3, 0x9a, 0x0f, 0x71, 0x47, 0x8d, 0xa4, 0x8c, 0x69, 0x15,
      0x65, 0xce, 0xe4, 0x31, 0x25, 0x4d, 0xd5, 0x0c, 0xab, 0x74, 0x62,
      0xc6, 0xcf, 0x19, 0x9b, 0xe9, 0xbf, 0x5c};
  static const uint8_t kSSKDFDigestDerivedKey[] = {
      0x5a, 0x2e, 0x26, 0x64, 0x4d, 0x16, 0x22, 0x2c, 0xd6, 0x36, 0xa1, 0xfd,
      0xb5, 0x7b, 0xfa, 0xa1, 0x7f, 0x94, 0x44, 0x91, 0x27, 0x61, 0x2b, 0xcd,
      0x7b, 0xe1, 0xbb, 0x39, 0xcc, 0x18, 0xf3, 0x28, 0x93, 0xd3, 0xc6, 0x48,
      0xc1, 0x63, 0x72, 0xfb, 0x6e, 0x9c, 0x63, 0xde, 0x54, 0x33, 0xb1, 0xcc,
      0xde, 0xb5, 0x1b, 0xb5, 0xf1, 0x53, 0x68, 0xc8, 0xa8, 0x49, 0xa1, 0xe5,
      0xa4, 0xef, 0xc6, 0x66, 0xfd, 0x33, 0xee, 0xb9, 0xf6, 0x72, 0x8b, 0x04,
      0x79, 0xf7, 0x66, 0x68, 0xcf, 0xaf, 0xc1, 0x3a, 0x91, 0x36, 0x70, 0x74,
      0xde, 0xf2, 0xb5, 0x0e, 0x9d, 0x9a, 0x91, 0x8a, 0x12, 0x02, 0x10, 0x82,
      0x41, 0x65, 0xd5, 0x96, 0xad, 0x4f, 0x94, 0xa3, 0x23, 0x6e, 0xf7, 0xcf,
      0x58, 0x43, 0x28, 0x2a, 0x0a, 0x57, 0xa4, 0x83, 0x81, 0x9f, 0x63, 0xe0,
      0xcf, 0xb2, 0x08, 0x1d, 0xaf, 0x9c, 0xcf, 0x35, 0xc6, 0x6a, 0x03, 0xe7,
      0xa0, 0x2d, 0x38, 0x91, 0xf4, 0x50, 0x22, 0xe1, 0xc8, 0x9d, 0x88, 0x8a,
      0xa8, 0x08, 0x7e, 0x08, 0xf4, 0x5b, 0xab, 0xbc, 0x52, 0x06, 0x2b, 0x18,
      0xe6, 0xfb, 0x70, 0xc1, 0x2d, 0xcb, 0x29, 0xa1, 0x94, 0xd2, 0x3a, 0xbc,
      0x35, 0x1c, 0xfb, 0x3c, 0xf4, 0xf1, 0x61, 0xcc, 0x77, 0x5a, 0x3e, 0x71,
      0x1b, 0xb1, 0x50, 0x2d, 0x69, 0x01, 0xf6, 0x93, 0x14, 0x07, 0xa9, 0xae,
      0x86, 0x84, 0x76, 0xf9, 0x98, 0xd1, 0xca, 0x4c, 0xca, 0x29, 0x6a, 0x9f,
      0x14, 0x75, 0x2d, 0x14, 0xf4, 0x74, 0x27, 0xe6, 0x66, 0x28, 0x9f, 0x80,
      0x89, 0x2a, 0x3d, 0x14, 0xa8, 0x4f, 0xe3, 0x43, 0xfd, 0x78, 0xd0, 0xda,
      0xdb, 0xde, 0x18, 0x19, 0xac, 0xa9, 0x15, 0xf7, 0xc0, 0xc0, 0x24, 0x37,
      0x6b, 0x40, 0xcb, 0x34, 0xba, 0xe2, 0xd2, 0x6e, 0x9f, 0x45, 0x52, 0xb6,
      0xb1, 0xa2, 0x6f, 0xa5};  // 256 bytes
  uint8_t sskdf_digest_output[sizeof(kSSKDFDigestDerivedKey)];
  if (!SSKDF_digest(&sskdf_digest_output[0], sizeof(kSSKDFDigestDerivedKey),
                    EVP_sha256(), &kSSKDFDigestSharedSecret[0],
                    sizeof(kSSKDFDigestSharedSecret), &kSSKDFDigestInfo[0],
                    sizeof(kSSKDFDigestInfo)) ||
      !check_test(kSSKDFDigestDerivedKey, sskdf_digest_output,
                  sizeof(kSSKDFDigestDerivedKey), "SSKDF_digest KAT")) {
    fprintf(stderr, "SSKDF_digest failed.\n");
    goto err;
  }

  // KBKDF counter HMAC-SHA-256
  static const uint8_t kKBKDF_ctr_hmac_secret[] = {
    0xdd, 0x1d, 0x91, 0xb7, 0xd9, 0x0b, 0x2b, 0xd3, 0x13, 0x85, 0x33, 0xce,
    0x92, 0xb2, 0x72, 0xfb, 0xf8, 0xa3, 0x69, 0x31, 0x6a, 0xef, 0xe2, 0x42,
    0xe6, 0x59, 0xcc, 0x0a, 0xe2, 0x38, 0xaf, 0xe0
  };
  static const uint8_t kKBKDF_ctr_hmac_info[] = {
    0x01, 0x32, 0x2b, 0x96, 0xb3, 0x0a, 0xcd, 0x19, 0x79, 0x79, 0x44, 0x4e,
    0x46, 0x8e, 0x1c, 0x5c, 0x68, 0x59, 0xbf, 0x1b, 0x1c, 0xf9, 0x51, 0xb7,
    0xe7, 0x25, 0x30, 0x3e, 0x23, 0x7e, 0x46, 0xb8, 0x64, 0xa1, 0x45, 0xfa,
    0xb2, 0x5e, 0x51, 0x7b, 0x08, 0xf8, 0x68, 0x3d, 0x03, 0x15, 0xbb, 0x29,
    0x11, 0xd8, 0x0a, 0x0e, 0x8a, 0xba, 0x17, 0xf3, 0xb4, 0x13, 0xfa, 0xac
  };
  static const uint8_t kKBKDF_ctr_hmac_output[] = {
    0x10, 0x62, 0x13, 0x42, 0xbf, 0xb0, 0xfd, 0x40, 0x04, 0x6c, 0x0e, 0x29,
    0xf2, 0xcf, 0xdb, 0xf0
  };

  uint8_t kbkdf_ctr_hmac_output[sizeof(kKBKDF_ctr_hmac_output)];
  if (!KBKDF_ctr_hmac(kbkdf_ctr_hmac_output, sizeof(kbkdf_ctr_hmac_output),
                      EVP_sha256(),
                      kKBKDF_ctr_hmac_secret, sizeof(kKBKDF_ctr_hmac_secret),
                      kKBKDF_ctr_hmac_info, sizeof(kKBKDF_ctr_hmac_info)) ||
      !check_test(kKBKDF_ctr_hmac_output, kbkdf_ctr_hmac_output,
                  sizeof(kbkdf_ctr_hmac_output),
                  "KBKDF-CTR-HMAC-SHA-256 KAT")) {
    goto err;
  }
  ret = 1;

err:
  EVP_AEAD_CTX_cleanup(&aead_ctx);

  return ret;
}

// BORINGSSL_self_test does not abort if any tests fail
int BORINGSSL_self_test(void) {
  if (!boringssl_self_test_fast() ||
      // When requested to run self tests, also run the lazy tests.
      !boringssl_self_test_rsa() ||
      !boringssl_self_test_ecc() ||
      !boringssl_self_test_ffdh() ||
      !boringssl_self_test_ml_kem() ||
      !boringssl_self_test_ml_dsa() ||
      !boringssl_self_test_eddsa() ||
      !boringssl_self_test_hasheddsa()) {
    return 0;
  }

  return 1;
}

#if defined(BORINGSSL_FIPS)
int boringssl_self_test_startup(void) {
  return boringssl_self_test_fast();
}
#endif
