/* -*- mode: C; c-file-style: "gnu"; indent-tabs-mode: nil; -*-
 *
 * Copyright 2025 GNOME Foundation, Inc.
 *
 * SPDX-License-Identifier: LGPL-2.1-or-later
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 *
 * Authors:
 *  - Philip Withnall <pwithnall@gnome.org>
 */

#pragma once

#include <glib.h>
#include <glib-object.h>
#include <gio/gio.h>
#include <libmalcontent/user-manager.h>

G_BEGIN_DECLS

/**
 * MctExtensionAgentObjectError:
 * @MCT_EXTENSION_AGENT_OBJECT_ERROR_FAILED: Unknown internal error.
 * @MCT_EXTENSION_AGENT_OBJECT_ERROR_IDENTIFYING_USER: The requesting user could
 *   not be identified.
 * @MCT_EXTENSION_AGENT_OBJECT_ERROR_CANCELLED: The request was cancelled before
 *   it could be answered.
 * @MCT_EXTENSION_AGENT_OBJECT_ERROR_INVALID_QUERY: A query parameter was
 *   invalid.
 *
 * Errors which can be returned by [class@Malcontent.ExtensionAgentObject].
 *
 * Since: 0.14.0
 */
typedef enum
{
  MCT_EXTENSION_AGENT_OBJECT_ERROR_FAILED = 0,
  MCT_EXTENSION_AGENT_OBJECT_ERROR_IDENTIFYING_USER,
  MCT_EXTENSION_AGENT_OBJECT_ERROR_CANCELLED,
  MCT_EXTENSION_AGENT_OBJECT_ERROR_INVALID_QUERY,
} MctExtensionAgentObjectError;
#define MCT_EXTENSION_AGENT_OBJECT_N_ERRORS (MCT_EXTENSION_AGENT_OBJECT_ERROR_INVALID_QUERY + 1)

GQuark mct_extension_agent_object_error_quark (void);
#define MCT_EXTENSION_AGENT_OBJECT_ERROR mct_extension_agent_object_error_quark ()

#define MCT_TYPE_EXTENSION_AGENT_OBJECT mct_extension_agent_object_get_type ()
G_DECLARE_DERIVABLE_TYPE (MctExtensionAgentObject, mct_extension_agent_object, MCT,
                          EXTENSION_AGENT_OBJECT, GObject)

/**
 * MctExtensionAgentObjectClass:
 * @request_extension_async: Virtual method called when an extension request
 *   is made to the agent. Agents should implement their main logic in this
 *   virtual method, including returning a value or error via @invocation.
 * @request_extension_finish: Virtual method called to extract results from a
 *   completed @request_extension_async closure.
 *
 * Since: 0.14.0
 */
struct _MctExtensionAgentObjectClass
{
  GObjectClass parent_class;

  void (* request_extension_async)  (MctExtensionAgentObject *agent,
                                     const char              *record_type,
                                     const char              *identifier,
                                     uint64_t                 duration_secs,
                                     GVariant                *extra_data,
                                     GVariant                *subject,
                                     GUnixFDList             *subject_fd_list,
                                     GDBusMethodInvocation   *invocation,
                                     GCancellable            *cancellable,
                                     GAsyncReadyCallback      callback,
                                     void                    *user_data);
  gboolean (* request_extension_finish) (MctExtensionAgentObject  *agent,
                                         GAsyncResult             *result,
                                         gboolean                 *out_granted,  /* (optional) */
                                         GVariant                **out_extra_data,  /* (nullable) (optional) */
                                         GError                  **error);

  void *padding[12];
};

gboolean mct_extension_agent_object_register (MctExtensionAgentObject  *self,
                                              GError                  **error);
void     mct_extension_agent_object_unregister (MctExtensionAgentObject *self);

gboolean mct_extension_agent_object_get_busy (MctExtensionAgentObject *self);

GDBusConnection *mct_extension_agent_object_get_connection (MctExtensionAgentObject *self);

G_END_DECLS
