/*
 * Copyright (C) 2000-2025 the xine project
 *
 * This file is part of xine, a unix video player.
 *
 * xine is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * xine is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110, USA
 *
 */
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h> /* unlink () */
#include <string.h>
#include <errno.h>

#include <xine/sorted_array.h>

#include "_xitk.h"
#include "slider.h"
#include "cfg_parse.h"
#include "skin.h"

#undef DEBUG_SKIN

/************************************************************************************
 *                                     PRIVATES
 ************************************************************************************/

static const uint8_t _tab_1[256] = {
  [0] = 100,
  ['\t'] = 101,
  [' '] = 101,
  ['&'] = 1,
  ['|'] = 2,
  ['$'] = 3,
  ['`'] = 4
};

static const uint8_t _tab_2[5][4] = {
  [1] = { 1, '&', 0, 0 },
  [2] = { 1, '|', 0, 0 },
  [3] = { 1, '(', 0, 0 }
};

static int _skin_first_cmd (const char *s) {
  const uint8_t *p = (const uint8_t *)s;
  uint8_t z;
  while ((z = _tab_1[p[0]]) == 101)
    p++;
  return p - (const uint8_t *)s;
}

static int _skin_next_cmd (const char *s) {
  const uint8_t *p = (const uint8_t *)s;
  uint8_t z;
  while (1) {
    while ((z = _tab_1[p[0]]) == 0)
      p++;
    if (z == 100)
      break;
    if (z < 100) {
      p++;
      if (_tab_2[z][0] && memcmp (p, &_tab_2[z][1], _tab_2[z][0]))
        continue;
      p += _tab_2[z][0];
      break;
    }
    p++;
  }
  while ((z = _tab_1[p[0]]) == 101)
    p++;
  return p - (const uint8_t *)s;
}

typedef struct {
#define _SKIN_NAME_SIZE 64
  /* this needs to stay first. */
  char                     section[_SKIN_NAME_SIZE];
  uint32_t                 users;
  /* offsets into sbuf */
  int                      pixmap_name,
                           pm_font_name,
                           pm_highlight_font_name,
                           pm_font_format,
                           slider_pm_pad_name;

  xitk_skin_element_info_t info;
} xitk_skin_element_t;

struct xitk_skin_config_s {
  xitk_t              *xitk;
  int                  version; /** << -1 (unused), >= 0 (skin loaded) */
  uint32_t             flags;
  uint32_t             plen; /** << path length */
  /* offsets into sbuf */
  unsigned int         meta[XITK_SKIN_LAST];

  xitk_image_t        *missing;

  xine_sarray_t       *imgs, *elements;
  xitk_skin_element_t *celement;

  uint32_t             sbuf_pos;
#define _SKIN_SBUF_SIZE 10240
  char                 sbuf[_SKIN_SBUF_SIZE + _SKIN_NAME_SIZE + 4];
};

static int _skin_ref (xitk_skin_config_t *skin, int add) {
  int refs;
  unsigned int fsize;
  FILE *f;
  char *name = skin->sbuf + skin->sbuf_pos + 2, buf[48];

  memcpy (name + skin->plen, "/.num_uses", 11);
  f = fopen (name, "rb+");
  if (f) {
    const char *w = buf;
    fsize = fread (buf, 1, sizeof (buf) - 32 - 1, f);
    buf[fsize] = 0;
    refs = xitk_str2int32 (&w, 0);
  } else {
    refs = 0;
    fsize = 0;
    f = fopen (name, "wb+");
    if (!f)
      return 0;
  }

  if (add) {
    int d;
    char *q;
    unsigned int v;
    refs += add;
    if (refs < 0)
      refs = 0;
    q = buf + 16;
    v = refs;
    do {
      *--q = v % 10u + '0';
      v /= 10u;
    } while (v);
    d = (int)fsize - (buf + 16 - q);
    if (d < 0)
      fsize -= d;
    else if (d > 0)
      memset (buf + 16, ' ', d);
    fseek (f, 0, SEEK_SET);
    fwrite (q, 1, fsize, f);
  }

  fclose (f);
  if (skin->xitk->verbosity >= 2)
    printf ("xitk.skin.refs (%s, %d).\n",
      skin->meta[XITK_SKIN_name] ? skin->sbuf + skin->meta[XITK_SKIN_name] : skin->sbuf + skin->meta[XITK_SKIN_path], refs);
  return refs;
}

static ATTR_INLINE_ALL_STRINGOPS uint32_t _skin_strdup (xitk_skin_config_t *skin, const char *s) {
  uint32_t r = skin->sbuf_pos, l;
  if (!s[0])
    return 0;
  l = strlen (s) + 1;
  if (l > _SKIN_SBUF_SIZE - r)
    return 0;
  memcpy (skin->sbuf + r, s, l);
  skin->sbuf_pos = r + l;
  return r;
}

typedef struct {
  const char *format;
  xitk_image_t *image;
  char name[1];
} xitk_skin_img_t;

/*
 *
 */
static void _skin_load_img (xitk_skin_config_t *skin, xitk_part_image_t *image, unsigned int pixmap, unsigned int format) {
  char *nbuf = skin->sbuf + skin->sbuf_pos, *key;
  const char *param = skin->sbuf + pixmap;
  size_t nlen = xitk_find_0_or_byte (param, '|');

  if (nlen >= _SKIN_NAME_SIZE)
    return;
  /* for part read and database keys, omit path. */
  key = nbuf + 2 + skin->plen + 1;
  memcpy (key, param, nlen);
  param += nlen;
  key[nlen] = 0;

  {
    xitk_skin_img_t *si;
    int pos;
    /* already loaded? */
#if defined(XINE_SARRAY_MODE_UNIQUE) && defined(XINE_SARRAY) && (XINE_SARRAY >= 2)
    pos = ~xine_sarray_add (skin->imgs, key);
#else
    pos = xine_sarray_binary_search (skin->imgs, key);
#endif
    if (pos >= 0) {
      char *name = xine_sarray_get (skin->imgs, pos);
      xitk_container (si, name, name[0]);
      if (format)
        si->format = skin->sbuf + format;
    } else {
      /* load now */
      si = malloc (xitk_offsetof (xitk_skin_img_t, name) + nlen + 1);
#if defined(XINE_SARRAY_MODE_UNIQUE) && defined(XINE_SARRAY) && (XINE_SARRAY >= 2)
      xine_sarray_move_location (skin->imgs, si->name, ~pos);
#endif
      if (!si) {
        if (image) {
          image->x = image->y = image->width = image->height = 0;
          image->num_states = 0;
          image->image = NULL;
        }
        return;
      }
      memcpy (si->name, key, nlen + 1);
      si->format = format ? skin->sbuf + format : NULL;
      si->image = NULL;
#if defined(XINE_SARRAY_MODE_UNIQUE) && defined(XINE_SARRAY) && (XINE_SARRAY >= 2)
#else
      xine_sarray_add (skin->imgs, si->name);
#endif
    }
    if (!image)
      return;
    do {
      if (si->image)
        break;
      key[-1] = '/';
      si->image = xitk_image_new (skin->xitk, nbuf + 2, 0, 0, 0);
      if (si->image)
        break;
      /* try fallback texture. */
      si->format = NULL;
      si->image = skin->missing;
      if (si->image)
        break;
      si->image = skin->missing = xitk_image_new (skin->xitk, XINE_SKINDIR "/missing.png", 0, 0, 0);
      if (si->image)
        break;
      image->x = image->y = image->width = image->height = 0;
      image->num_states = 0;
      image->image = NULL;
      return;
    } while (0);
    image->image = si->image;
    if (si->format)
      xitk_image_set_pix_font (si->image, si->format);
  }

  if (image->image != skin->missing) {
    image->num_states = (int)image->image->last_state + 1;
  } else {
    image->num_states = 3;
    param = "";
  }
  if (!*param) {
    image->x = image->y = 0;
    image->width = image->image->width;
    image->height = image->image->height;
    return;
  }

  image->x = image->y = image->width = image->height = 0;
  image->num_states = 0;
  {
    const uint8_t *p = (const uint8_t *)param + 1;
    uint32_t v, minus;
    uint8_t z;
    /* x */
    v = 0;
    while ((z = *p ^ '0') < 10)
      v = v * 10u + z, p++;
    image->x = v;
    if (*p == ',')
      p++;
    /* y */
    v = 0;
    while ((z = *p ^ '0') < 10)
      v = v * 10u + z, p++;
    image->y = v;
    if (*p == ',')
      p++;
    /* w */
    v = 0;
    while ((z = *p ^ '0') < 10)
      v = v * 10u + z, p++;
    image->width = v;
    if (*p == ',')
      p++;
    /* h */
    v = 0;
    while ((z = *p ^ '0') < 10)
      v = v * 10u + z, p++;
    image->height = v;
    if (*p == ',')
      p++;
    /* num_states */
    minus = 0;
    if (*p == '-')
      minus = 1, p++;
    v = 0;
    while ((z = *p ^ '0') < 10)
      v = v * 10u + z, p++;
    image->num_states = minus ? -(int)v : (int)v;
  }

  if (XITK_0_TO_MAX_MINUS_1 (image->num_states + 1, 3)) /* -1, 0, 1 */
    image->num_states = (int)image->image->last_state + 1;
  if (!XITK_0_TO_MAX_MINUS_1 (image->x, image->image->width))
    image->x = 0;
  if (!XITK_0_TO_MAX_MINUS_1 (image->y, image->image->height))
    image->y = 0;
  if (!XITK_0_TO_MAX_MINUS_1 (image->width - 1, image->image->width - image->x))
    image->width = image->image->width - image->x;
  if (!XITK_0_TO_MAX_MINUS_1 (image->height - 1, image->image->height - image->y))
    image->height = image->image->height - image->y;
}

/* 1 = \0; 2 = 0-9; 4 = A-Z,a-z,_; 8 = $; 16 = (,{; 32 = ),}; */
static const uint8_t _tab_char[256] = {
   1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
   0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
   0, 0, 0, 0, 8, 0, 0, 0,16,32, 0, 0, 0, 0, 0, 0,
   2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 0, 0, 0, 0, 0, 0,
   0, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4,
   4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 0, 0, 0, 0, 4,
   0, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4,
   4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4,16, 0,32, 0, 0,
   0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
   0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
   0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
   0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
   0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
   0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
   0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
   0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0
};

/*
 *
 */
//#warning FIXME
static uint32_t _skin_expanded (xitk_skin_config_t *skin, const char *cmd) {
  const uint8_t *p = (const uint8_t *)cmd;
  uint8_t buf[2048], *q = buf, *e = buf + sizeof (buf) - 1;
  int num_vars = 0;

  while (q < e) {
    const uint8_t *start, *stop;
    const char *val;
    int l;
    start = p;
    while (!(_tab_char[*p] & (1 | 8))) /* \0, $ */
      p++;
    /* literal part */
    l = p - start;
    if (l > e - q)
      l = e - q;
    if (l > 0) {
      memcpy (q, start, l);
      q += l;
    }
    /* eot */
    if (!*p)
      break;
    /* $ */
    if ((_tab_char[p[1]] & 16) && (_tab_char[p[2]] & 4)) {
      /* ${VARNAME}, $(VARNAME) */
      p += 2;
      start = p;
      while (_tab_char[*p] & (2 | 4)) /* 0-9, A-Z, a-z, _ */
        p++;
      stop = p;
      while (!(_tab_char[*p] & (1 | 32))) /* \0, ), } */
        p++;
      if (*p)
        p++;
    } else if (_tab_char[p[1]] & 4) {
      /* $VARNAME */
      p += 1;
      start = p;
      while (_tab_char[*p] & (2 | 4)) /* 0-9, A-Z, a-z, _ */
        p++;
      stop = p;
    } else {
      if (q < e)
        *q++ = *p;
      p++;
      continue;
    }
    num_vars += 1;
    val = NULL;
    l = stop - start;
    switch (l) {
      case 4:
        if (!memcmp (start, "HOME", 4)) {
          val = xine_get_homedir ();
        }
        break;
      case 8:
        if (!memcmp (start, "SKIN_URL", 8)) {
          val = skin->sbuf + skin->meta[XITK_SKIN_url];
        }
        break;
      case 9:
        if (!memcmp (start, "SKIN_", 5)) {
          if (!memcmp (start + 5, "PATH", 4)) {
            val = skin->sbuf + skin->meta[XITK_SKIN_path];
          } else if (!memcmp (start + 5, "NAME", 4)) {
            val = skin->sbuf + skin->meta[XITK_SKIN_name];
          } else if (!memcmp (start + 5, "DATE", 4)) {
            val = skin->sbuf + skin->meta[XITK_SKIN_date];
          }
        }
        break;
      case 11:
        if (!memcmp (start, "SKIN_AUTHOR", 11)) {
          val = skin->sbuf + skin->meta[XITK_SKIN_author];
        }
        break;
      case 12:
        if (!memcmp (start, "SKIN_VERSION", 12)) {
          if (skin->version >= 0)
            q += snprintf ((char *)q, e - q, "%d", skin->version);
        }
        break;
      case 16:
        if (!memcmp (start, "SKIN_PARENT_PATH", 16)) {
          if (skin->meta[XITK_SKIN_path]) {
            const char *z = strrchr (skin->sbuf + skin->meta[XITK_SKIN_path], '/');
            l = z ? z - (skin->sbuf + skin->meta[XITK_SKIN_path]) : (int)skin->plen;
            if (l > e - q)
              l = e - q;
            if (l > 0)
              memcpy (q, skin->sbuf + skin->meta[XITK_SKIN_path], l), q += l;
          }
        }
        break;
      default: ;
    }
    if (val) {
      l = xitk_find_byte (val, 0);
      if (l > e - q)
        l = e - q;
      if (l > 0) {
        memcpy (q, val, l); q += l;
      }
    }
  }
  *q = 0;
  if (num_vars > 0)
    return _skin_strdup (skin, (const char *)buf);
  return 0;
}

/*
 * Return alignement value.
 */
static int skin_get_align_value (const char *val) {
  union {
    char b[4];
    uint32_t v;
  } _left = {{'l','e','f','t'}},
    _right = {{'r','i','g','h'}},
    _have;

  /* _not_ an overread here. */
  memcpy (_have.b, val, 4);
  _have.v |= 0x20202020;
  if (_have.v == _left.v)
    return ALIGN_LEFT;
  if (_have.v == _right.v)
    return ALIGN_RIGHT;
  return ALIGN_CENTER;
}

/*
 * Return direction
 */
static int skin_get_direction (const char *val) {
  union {
    char b[4];
    uint32_t v;
  } _down  = {{'d','o','w','n'}},
    _right = {{'r','i','g','h'}},
    _up    = {{'u','p',' ',' '}},
    _have;

  /* _not_ an overread here. */
  memcpy (_have.b, val, 4);
  if (!_have.b[2])
    _have.b[3] = 0;
  _have.v |= 0x20202020;
  if (_have.v == _right.v)
    return DIRECTION_RIGHT;
  if (_have.v == _down.v)
    return DIRECTION_DOWN;
  if (_have.v == _up.v)
    return DIRECTION_UP;
  return DIRECTION_LEFT;
}

/*
 * Return slider typw
 */
static int skin_get_slider_type (const char *val) {
  union {
    char b[4];
    uint32_t v;
  } _vert = {{'v','e','r','t'}}, /* "vertical" */
    _rota = {{'r','o','t','a'}}, /* "rotate" */
    _have;

  /* _not_ an overread here. */
  memcpy (_have.b, val, 4);
  _have.v |= 0x20202020;
  if (_have.v == _vert.v)
    return XITK_VSLIDER;
  if (_have.v == _rota.v)
    return XITK_RSLIDER;
  return XITK_HSLIDER;
}

#ifdef DEBUG_SKIN
/*
 * Just to check list chained constitency.
 */
static void check_skin (xitk_skin_config_t *skin) {
  int n;

  ABORT_IF_NULL (skin);

  n = xine_sarray_size (skin->elements);
  if (n) {
    int i;

    printf("Skin name '%s'\n",      skin->sbuf + skin->meta[XITK_SKIN_name]);
    printf("     version   '%d'\n", skin->version);
    printf("     author    '%s'\n", skin->sbuf + skin->meta[XITK_SKIN_author]);
    printf("     date      '%s'\n", skin->sbuf + skin->meta[XITK_SKIN_date]);
    printf("     load cmd  '%s'\n", skin->sbuf + skin->meta[XITK_SKIN_load_command]);
    printf("     uload cmd '%s'\n", skin->sbuf + skin->meta[XITK_SKIN_unload_command]);
    printf("     URL       '%s'\n", skin->sbuf + skin->meta[XITK_SKIN_url]);
    printf("     logo      '%s'\n", skin->sbuf + skin->meta[XITK_SKIN_logo]);
    printf("     animation '%s'\n", skin->sbuf + skin->meta[XITK_SKIN_animation]);

    for (i = 0; i < n; i++) {
      xitk_skin_element_t *s = xine_sarray_get (skin->elements, i);

      printf("Section '%s'\n", s->section);
      printf("  enable      = %d\n", s->info.enability);
      printf("  visible     = %d\n", s->info.visibility);
      printf("  X           = %d\n", s->info.x);
      printf("  Y           = %d\n", s->info.y);
      printf("  type        = %d\n", s->info.type);
      printf("  size        = %d\n", s->info.size);
      printf("  pixmap      = '%s'\n", skin->sbuf + s->pixmap_name);

      if (s->info.type) {
	printf("  pad pixmap  = '%s'\n", skin->sbuf + s->slider_pm_pad_name);
      }

      printf("  height      = %d\n", s->info.label_height);
      printf("  animation   = %d\n", s->info.label_animation);
      printf("  step        = %d\n", s->info.label_animation_step);
      printf("  print       = %d\n", s->info.label_printable);
      printf("  static      = %d\n", s->info.label_staticity);
      printf("  length      = %d\n", s->info.label_length);
      printf("  color bg    = '#%06x'\n", (unsigned int)s->info.label_color_bg);
      printf("  color       = '#%06x'\n", (unsigned int)s->info.label_color);
      printf("  color focus = '#%06x'\n", (unsigned int)s->info.label_color_focus);
      printf("  color click = '#%06x'\n", (unsigned int)s->info.label_color_click);
      printf("  pixmap font = '%s'\n", skin->sbuf + s->pm_font_name);
      printf("  pixmap highlight_font = '%s'\n", skin->sbuf + s->pm_highlight_font_name);
      printf("  pixmap fmt  = '%s'\n", skin->sbuf + s->pm_font_format);
      printf("  font        = '%s'\n", s->info.label_fontname);
      printf("  max_buttons = %d\n", s->info.max_buttons);
    }

  }
}
#endif

/************************************************************************************
 *                                   END OF PRIVATES
 ************************************************************************************/

/*
 * Alloc a xitk_skin_config_t* memory area, nullify pointers.
 */
xitk_skin_config_t *xitk_skin_new (xitk_t *xitk) {
  xitk_skin_config_t *skin = (xitk_skin_config_t *)xitk_xmalloc (sizeof (*skin));

  if (!skin)
    return NULL;

  if (xitk_init_NULL ()) {
    skin->missing = NULL;
    skin->celement = NULL;
  }
#if 0
  skin->meta[XITK_SKIN_name]     =
  skin->meta[XITK_SKIN_author]   =
  skin->meta[XITK_SKIN_date]     =
  skin->meta[XITK_SKIN_url]      =
  skin->meta[XITK_SKIN_load_command] =
  skin->meta[XITK_SKIN_unload_command] =
  skin->meta[XITK_SKIN_logo]     =
  skin->meta[XITK_SKIN_animation] =
  skin->meta[XITK_SKIN_skinfile] =
  skin->meta[XITK_SKIN_path]     = 0;
  skin->plen     = 0;
#endif
  skin->sbuf_pos = 1;
  skin->sbuf[0] = 0;

  skin->elements = xine_sarray_new (128, (xine_sarray_comparator_t)strcmp);
  skin->imgs     = xine_sarray_new (128, (xine_sarray_comparator_t)strcmp);
#ifdef XINE_SARRAY_MODE_UNIQUE
  xine_sarray_set_mode (skin->imgs, XINE_SARRAY_MODE_UNIQUE);
#endif
#if defined(XINE_SARRAY) && (XINE_SARRAY >= 3)
  xine_sarray_set_hash (skin->elements, (unsigned int (*)(void *))xitk_string_hash_32, 32);
  xine_sarray_set_hash (skin->imgs,     (unsigned int (*)(void *))xitk_string_hash_32, 32);
#endif

  skin->xitk     = xitk;
  skin->version  = -1;

  return skin;
}

typedef enum {
  _K_NONE = 0,
  _K_align,
  _K_animation,
  _K_author,
  _K_browser,
  _K_color_bg,
  _K_color,
  _K_color_click,
  _K_color_focus,
  _K_coords,
  _K_date,
  _K_direction,
  _K_enable,
  _K_entries,
  _K_font,
  _K_label,
  _K_length,
  _K_load_command,
  _K_logo,
  _K_max_buttons,
  _K_name,
  _K_pixmap,
  _K_pixmap_focus,
  _K_pixmap_format,
  _K_print,
  _K_radius,
  _K_slider,
  _K_static,
  _K_step,
  _K_timer,
  _K_type,
  _K_unload_command,
  _K_url,
  _K_version,
  _K_visible,
  _K_x,
  _K_y,
  _K_h,
  _K_LAST
} _skin_key_t;

static _skin_key_t _skin_key_index (const char *key, unsigned int klen) {
  switch (klen) {
    int d;
    case 1:
      if (key[0] == 'x')
        return _K_x;
      if (key[0] == 'y')
        return _K_y;
      break;
    case 3:
      if (!memcmp (key, "url", 3))
        return _K_url;
      break;
    case 4:
      d = memcmp (key, "logo", 4);
      if (d == 0)
        return _K_logo;
      if (d < 0) {
        if (!memcmp (key, "date", 4))
          return _K_date;
        if (!memcmp (key, "font", 4))
          return _K_font;
      } else {
        if (!memcmp (key, "name", 4))
          return _K_name;
        if (!memcmp (key, "step", 4))
          return _K_step;
        if (!memcmp (key, "type", 4))
          return _K_type;
      }
      break;
    case 5:
      d = memcmp (key, "label", 5);
      if (d == 0)
        return _K_label;
      if (d < 0) {
        if (!memcmp (key, "align", 5))
          return _K_align;
        if (!memcmp (key, "color", 5))
          return _K_color;
      } else {
        if (!memcmp (key, "print", 5))
          return _K_print;
        if (!memcmp (key, "timer", 5))
          return _K_timer;
      }
      break;
    case 6:
      d = memcmp (key, "length", 6);
      if (d == 0)
        return _K_length;
      if (d < 0) {
        d = memcmp (key, "coords", 6);
        if (d == 0)
          return _K_coords;
        if (d < 0) {
          if (!memcmp (key, "author", 6))
            return _K_author;
        } else {
          if (!memcmp (key, "enable", 6))
            return _K_enable;
          if (!memcmp (key, "height", 6))
            return _K_h;
        }
      } else {
        d = memcmp (key, "slider", 6);
        if (d == 0)
          return _K_slider;
        if (d < 0) {
          if (!memcmp (key, "radius", 6))
            return _K_radius;
          if (!memcmp (key, "pixmap", 6))
            return _K_pixmap;
        } else {
          if (!memcmp (key, "static", 6))
            return _K_static;
        }
      }
      break;
    case 7:
      if (!memcmp (key, "browser", 7))
        return _K_browser;
      if (!memcmp (key, "entries", 7))
        return _K_entries;
      if (!memcmp (key, "version", 7))
        return _K_version;
      if (!memcmp (key, "visible", 7))
        return _K_visible;
      break;
    case 8:
      if (!memcmp (key, "color_bg", 8))
        return _K_color_bg;
      break;
    case 9:
      if (!memcmp (key, "animation", 9))
        return _K_animation;
      if (!memcmp (key, "direction", 9))
        return _K_direction;
      break;
    case 11:
      if (!memcmp (key, "color_click", 11))
        return _K_color_click;
      if (!memcmp (key, "color_focus", 11))
        return _K_color_focus;
      if (!memcmp (key, "max_buttons", 11))
        return _K_max_buttons;
      break;
    case 12:
      if (!memcmp (key, "load_command", 12))
        return _K_load_command;
      if (!memcmp (key, "pixmap_focus", 12))
        return _K_pixmap_focus;
      break;
    case 13:
      if (!memcmp (key, "pixmap_format", 13))
        return _K_pixmap_format;
      break;
    case 14:
      if (!memcmp (key, "unload_command", 14))
        return _K_unload_command;
      break;
    default: ;
  }
  return _K_NONE;
}

typedef enum {
  _KT_NONE = 0, /** << context dependent, container, or really no data. */
  _KT_string,
  _KT_expanded,
  _KT_align,
  _KT_direction,
  _KT_int,
  _KT_bool,
  _KT_color,
  _KT_LAST
} _skin_key_type_t;

static const uint8_t _skin_key_types[_K_LAST] = {
  [_K_NONE] = _KT_NONE,
  [_K_align] = _KT_align,
  [_K_animation] = _KT_NONE,
  [_K_author] = _KT_string,
  [_K_browser] = _KT_NONE,
  [_K_color_bg] = _KT_color,
  [_K_color] = _KT_color,
  [_K_color_click] = _KT_color,
  [_K_color_focus] = _KT_color,
  [_K_coords] = _KT_NONE,
  [_K_date] = _KT_string,
  [_K_direction] = _KT_direction,
  [_K_enable] = _KT_bool,
  [_K_entries] = _KT_int,
  [_K_font] = _KT_string,
  [_K_h] = _KT_int,
  [_K_label] = _KT_NONE,
  [_K_length] = _KT_int,
  [_K_load_command] = _KT_expanded,
  [_K_logo] = _KT_expanded,
  [_K_max_buttons] = _KT_int,
  [_K_name] = _KT_string,
  [_K_pixmap] = _KT_string,
  [_K_pixmap_focus] = _KT_string,
  [_K_pixmap_format] = _KT_string,
  [_K_print] = _KT_bool,
  [_K_radius] = _KT_int,
  [_K_slider] = _KT_NONE,
  [_K_static] = _KT_bool,
  [_K_step] = _KT_int,
  [_K_timer] = _KT_int,
  [_K_type] = _KT_NONE,
  [_K_unload_command] = _KT_expanded,
  [_K_url] = _KT_string,
  [_K_version] = _KT_int,
  [_K_visible] = _KT_bool,
  [_K_x] = _KT_int,
  [_K_y] = _KT_int
};

static const uint8_t _skin_element_offs_elem[_K_LAST] = {
  [_K_direction] = xitk_offsetof (xitk_skin_element_t, info.type),
  [_K_enable] = xitk_offsetof (xitk_skin_element_t, info.enability),
  [_K_max_buttons] = xitk_offsetof (xitk_skin_element_t, info.size),
  [_K_pixmap] = xitk_offsetof (xitk_skin_element_t, pixmap_name),
  [_K_visible] = xitk_offsetof (xitk_skin_element_t, info.visibility)
};

static const uint8_t _skin_element_offs_coords[_K_LAST] = {
  [_K_x] = xitk_offsetof (xitk_skin_element_t, info.x),
  [_K_y] = xitk_offsetof (xitk_skin_element_t, info.y)
};

static const uint8_t _skin_element_offs_label[_K_LAST] = {
  [_K_align] = xitk_offsetof (xitk_skin_element_t, info.label_alignment),
  [_K_color_bg] = xitk_offsetof (xitk_skin_element_t, info.label_color_bg),
  [_K_color] = xitk_offsetof (xitk_skin_element_t, info.label_color),
  [_K_color_click] = xitk_offsetof (xitk_skin_element_t, info.label_color_click),
  [_K_color_focus] = xitk_offsetof (xitk_skin_element_t, info.label_color_focus),
  [_K_length] = xitk_offsetof (xitk_skin_element_t, info.size),
  [_K_pixmap] = xitk_offsetof (xitk_skin_element_t, pm_font_name),
  [_K_pixmap_focus] = xitk_offsetof (xitk_skin_element_t, pm_highlight_font_name),
  [_K_pixmap_format] = xitk_offsetof (xitk_skin_element_t, pm_font_format),
  [_K_print] = xitk_offsetof (xitk_skin_element_t, info.label_printable),
  [_K_static] = xitk_offsetof (xitk_skin_element_t, info.label_staticity),
  [_K_step] = xitk_offsetof (xitk_skin_element_t, info.label_animation_step),
  [_K_timer] = xitk_offsetof (xitk_skin_element_t, info.label_animation_timer),
  [_K_y] = xitk_offsetof (xitk_skin_element_t, info.label_y),
  [_K_h] = xitk_offsetof (xitk_skin_element_t, info.label_height)
};

static const uint8_t _skin_element_offs_slider[_K_LAST] = {
  [_K_pixmap] = xitk_offsetof (xitk_skin_element_t, slider_pm_pad_name),
  [_K_radius] = xitk_offsetof (xitk_skin_element_t, info.size)
};

static void _skin_parse_2 (xitk_skin_config_t *skin, char *text, xitk_cfg_parse_t *tree, xitk_cfg_parse_t *sub) {
  xitk_skin_element_t *s = skin->celement;
  _skin_key_t n = sub->key;
  xitk_cfg_parse_t *sub2;

  if (_skin_element_offs_elem[n]) {
    union { int *i; uint8_t *s; xitk_skin_element_t *e; } u;
    u.e = s;
    u.s += _skin_element_offs_elem[n];
    *u.i = sub->value;
    return;
  }
  switch (n) {
    case _K_browser:
      for (sub2 = tree + sub->first_child; sub2 != tree; sub2 = tree + sub2->next) {
        if ((_skin_key_t)sub2->key == _K_entries)
          s->info.size = sub2->value;
      }
      break;
    case _K_coords:
      for (sub2 = tree + sub->first_child; sub2 != tree; sub2 = tree + sub2->next) {
        n = sub2->key;
        if (_skin_element_offs_coords[n]) {
          union { int *i; uint8_t *s; xitk_skin_element_t *e; } u;
          u.e = s;
	  u.s += _skin_element_offs_coords[n];
	  *u.i = sub2->value;
        }
      }
      break;
    case _K_label:
      /* for missing skin entries, set defaults... */
      s->info.label_y = 0;
      s->info.label_printable = 1;
      s->info.label_animation_step = 1;
      s->info.label_animation_timer = xitk_get_cfg_num (skin->xitk, XITK_TIMER_LABEL_ANIM);
      s->info.label_alignment = ALIGN_UNSET;
      /* ...or mark as unset here and really fix below. */
      s->info.label_color_click = ~0u;
      s->info.label_color_focus = ~0u;

      for (sub2 = tree + sub->first_child; sub2 != tree; sub2 = tree + sub2->next) {
        n = sub2->key;
        if (_skin_element_offs_label[n]) {
          union { int *i; uint8_t *s; xitk_skin_element_t *e; } u;
          u.e = s;
	  u.s += _skin_element_offs_label[n];
	  *u.i = sub2->value;
          continue;
        }
        switch (n) {
          case _K_animation:
            s->info.label_animation = xitk_get_bool_value (text + sub2->value);
            break;
          case _K_font:
            s->info.label_fontname = skin->sbuf + sub2->value;
            break;
          default: ;
        }
      }
      if (s->info.label_color_focus == ~0u)
        s->info.label_color_focus = s->info.label_color;
      if (s->info.label_color_click == ~0u)
        s->info.label_color_click = s->info.label_color_focus;
      if (s->pm_font_name)
        _skin_load_img (skin, NULL, s->pm_font_name, s->pm_font_format);
      if (s->pm_highlight_font_name)
        _skin_load_img (skin, NULL, s->pm_highlight_font_name, s->pm_font_format);
      break;
    case _K_slider:
      for (sub2 = tree + sub->first_child; sub2 != tree; sub2 = tree + sub2->next) {
        n = sub2->key;
        if (_skin_element_offs_slider[n]) {
          union { int *i; uint8_t *s; xitk_skin_element_t *e; } u;
          u.e = s;
	  u.s += _skin_element_offs_slider[n];
	  *u.i = sub2->value;
          continue;
        }
        switch (n) {
          case _K_type:
            s->info.type = skin_get_slider_type (text + sub2->value);
            break;
          default: ;
        }
      }
      break;
    default: ;
  }
}

static const uint8_t _skin_offs[_K_LAST] = {
  [_K_author] = xitk_offsetof (xitk_skin_config_t, meta[XITK_SKIN_author]),
  [_K_date] = xitk_offsetof (xitk_skin_config_t, meta[XITK_SKIN_date]),
  [_K_load_command] = xitk_offsetof (xitk_skin_config_t, meta[XITK_SKIN_load_command]),
  [_K_logo] = xitk_offsetof (xitk_skin_config_t, meta[XITK_SKIN_logo]),
  [_K_name] = xitk_offsetof (xitk_skin_config_t, meta[XITK_SKIN_name]),
  [_K_unload_command] = xitk_offsetof (xitk_skin_config_t, meta[XITK_SKIN_unload_command]),
  [_K_url] = xitk_offsetof (xitk_skin_config_t, meta[XITK_SKIN_url]),
  [_K_version] = xitk_offsetof (xitk_skin_config_t, version)
};

static void _skin_parse_1 (xitk_skin_config_t *skin, char *text, xitk_cfg_parse_t *tree) {
  xitk_cfg_parse_t *entry;

  for (entry = tree + tree->first_child; entry != tree; entry = tree + entry->next) {
    if (entry->key < 0)
      continue;
    if (entry->first_child) {
      const char *key = text + entry->key;
      xitk_cfg_parse_t *sub;
      xitk_skin_element_t *s = calloc (1, sizeof (*s));

      if (s) {
        if (xitk_init_NULL ()) {
          s->info.pixmap_img.image                 = NULL;
          s->info.slider_pixmap_pad_img.image      = NULL;
          s->info.label_pixmap_font_img            = NULL;
          s->info.label_pixmap_highlight_font_img  = NULL;
          s->info.label_fontname                   = NULL;
        }
        s->info.type = DIRECTION_LEFT; /* Compatibility */

        skin->celement = s;
        /* xitk_cfg_parse () already lowercased this key. */
        memcpy (s->section, key, xitk_min (sizeof (s->section) - 1, entry->klen));
        s->info.visibility = s->info.enability = 1;
        xine_sarray_add (skin->elements, s);

        for (sub = tree + entry->first_child; sub != tree; sub = tree + sub->next)
          _skin_parse_2 (skin, text, tree, sub);
      }
    } else {
      _skin_key_t n = entry->key;
      if (_skin_offs[n]) {
        union { int *i; uint8_t *b; xitk_skin_config_t *s; } u;
        u.s = skin;
        u.b += _skin_offs[n];
        *u.i = entry->value;
      } else {
        switch (n) {
          case _K_animation:
            skin->meta[XITK_SKIN_animation] = _skin_strdup (skin, text + entry->value);
            break;
          default: ;
        }
      }
    }
  }
}

static void _skin_unload (xitk_skin_config_t *skin) {
  int n;

  if (skin->xitk->verbosity >= 2)
    printf ("xitk.skin.unload (%s).\n", skin->sbuf + skin->meta[XITK_SKIN_name]);

  if (skin->meta[XITK_SKIN_unload_command] && (_skin_ref (skin, -1) == 0)) {
    char *cmd = skin->sbuf + skin->meta[XITK_SKIN_unload_command];
    if (!(skin->flags & XITK_SKIN_LOAD_TRUSTED)) {
      char *t = cmd + _skin_first_cmd (cmd);
      if (memcmp (t, "xset -fp ", 9)) {
        cmd = NULL;
      } else {
        t += _skin_next_cmd (t);
        if (*t)
          *t = 0;
      }
    }
    if (cmd) {
      int r = xitk_system (0, cmd);
      if (skin->xitk->verbosity + (r != 0) >= 2)
        printf ("xitk.skin.unload.system (%s) = %d: %s.\n",
          skin->sbuf + skin->meta[XITK_SKIN_name], r, cmd);
    }
  }

  for (n = xine_sarray_size (skin->elements) - 1; n >= 0; n--) {
    xitk_skin_element_t *s = xine_sarray_get (skin->elements, n);
    free (s);
  }
#if defined(XINE_SARRAY) && (XINE_SARRAY >= 4)
  if (skin->xitk->verbosity >= 2) {
    unsigned int q = xine_sarray_hash_quality (skin->elements);
    printf ("xitk.skin.elements.hash_quality (%u.%u%%).\n", q / 10u, q % 10u);
  }
#endif

  for (n = xine_sarray_size (skin->imgs) - 1; n >= 0; n--) {
    char *name = xine_sarray_get (skin->imgs, n);
    xitk_skin_img_t *simg;
    xitk_container (simg, name, name[0]);
    if (simg->image != skin->missing)
      xitk_image_free_image (&simg->image);
    free (simg);
  }
#if defined(XINE_SARRAY) && (XINE_SARRAY >= 4)
  if (skin->xitk->verbosity >= 2) {
    unsigned int q = xine_sarray_hash_quality (skin->imgs);
    printf ("xitk.skin.images.hash_quality (%u.%u%%).\n", q / 10u, q % 10u);
  }
#endif
}

/*
 * Load the skin configfile.
 */
int xitk_skin_load (xitk_skin_config_t *skin, const char *path, const char *filename, unsigned int flags) {
  char *text;
  size_t fsize = 2 << 20;
  xitk_cfg_parse_t *tree;

  if (!skin)
    return -2;
  if (skin->version >= 0) {
    unsigned int u;
    _skin_unload (skin);
    xine_sarray_clear (skin->imgs);
    xine_sarray_clear (skin->elements);
    skin->version = -1;
    skin->plen = 0;
    for (u = 0; u < XITK_SKIN_LAST; u++)
      skin->meta[u] = 0;
    skin->celement = NULL;
    skin->sbuf_pos = 1;
  }
  if (!path || !filename)
    return -1;

  skin->flags = flags;
  skin->meta[XITK_SKIN_path]     = _skin_strdup (skin, path);
  skin->plen     = skin->sbuf_pos - skin->meta[XITK_SKIN_path] - 1;
  skin->meta[XITK_SKIN_skinfile] = _skin_strdup (skin, filename);

  skin->sbuf[skin->meta[XITK_SKIN_path] + skin->plen] = '/';
  text = xitk_cfg_load (skin->sbuf + skin->meta[XITK_SKIN_path], &fsize);
  if (!text) {
    if (skin->xitk->verbosity >= 1) {
      int e = errno;
      printf ("xitk.skin.load.failed (%s, %s (%d)).\n", skin->sbuf + skin->meta[XITK_SKIN_path], strerror (e), e);
    }
    skin->sbuf[skin->meta[XITK_SKIN_path] + skin->plen] = 0;
    return -2;
  }
  skin->sbuf[skin->meta[XITK_SKIN_path] + skin->plen] = 0;

  tree = xitk_cfg_parse (text, (skin->xitk->verbosity >= 3) ? XITK_CFG_PARSE_DEBUG : 0);
  if (!tree) {
    xitk_cfg_unload (text);
    return -1;
  }

  xitk_load_cfg_file (skin->xitk, path);

  {
    int i;

    for (i = 1; i < tree[0].key; i++) {
      const char *v;
      /* printf ("xitk.skin.tree[%d] key (%s), value (%s).\n", i, text + tree[i].key, text + tree[i].value); */
      if (tree[i].level == 1) {
        if (memcmp (text + tree[i].key, "skin.", 5)) {
          tree[i].key = -1;
          continue;
        }
        tree[i].key += 5, tree[i].klen -= 5;
        if (tree[i].first_child)
          continue;
      }
      tree[i].key = _skin_key_index (text + tree[i].key, tree[i].klen);
      v = text + tree[i].value;
      switch (_skin_key_types[tree[i].key]) {
        case _KT_string:
          /* printf ("xitk.skin.tree[%d].strdup pos (%u), key (%d), value (%s).\n",
            i, (unsigned int)skin->sbuf_pos, tree[i].key, text + tree[i].value); */
          tree[i].value = _skin_strdup (skin, v);
          break;
        case _KT_expanded:
          /* printf ("xitk.skin.tree[%d].expanded pos (%u), key (%d), value (%s).\n",
            i, (unsigned int)skin->sbuf_pos, tree[i].key, text + tree[i].value); */
          tree[i].value = _skin_expanded (skin, v);
          break;
        case _KT_align:
          tree[i].value = skin_get_align_value (v);
          break;
        case _KT_direction:
          tree[i].value = skin_get_direction (v);
          break;
        case _KT_int:
          tree[i].value = xitk_str2int32 (&v, 0);
          break;
        case _KT_bool:
          tree[i].value = xitk_get_bool_value (v);
          break;
        case _KT_color:
          tree[i].value = xitk_get_color_name (v);
          break;
        default: ;
      }
    }
  }

  skin->version = 0;
  _skin_parse_1 (skin, text, tree);
  if (skin->version < 0)
    skin->version = 0;

  xitk_cfg_unparse (tree);
  xitk_cfg_unload (text);

  if (!skin->celement) {
    if (skin->xitk->verbosity >= 1) {
      skin->sbuf[skin->meta[XITK_SKIN_path] + skin->plen] = '/';
      printf ("skin.load.warning (%s): no valid skin element found.\n", skin->sbuf + skin->meta[XITK_SKIN_path]);
      skin->sbuf[skin->meta[XITK_SKIN_path] + skin->plen] = 0;
    }
    return -1;
  }

  if (skin->sbuf_pos + skin->plen >= _SKIN_SBUF_SIZE)
    skin->plen = _SKIN_SBUF_SIZE - skin->sbuf_pos;
  memset (skin->sbuf + skin->sbuf_pos, 0, 2);
  memcpy (skin->sbuf + skin->sbuf_pos + 2, skin->sbuf + skin->meta[XITK_SKIN_path], skin->plen);
  skin->sbuf[skin->sbuf_pos + 2 + skin->plen] = 0;

#ifdef DEBUG_SKIN
  check_skin (skin);
#endif

  if (skin->flags & XITK_SKIN_LOAD_FIRST) {
    char *name = skin->sbuf + skin->sbuf_pos + 2;
    /* security: ref counter is ours only. */
    memcpy (name + skin->plen, "/.num_uses", 11);
    unlink (name);
    skin->flags &= ~XITK_SKIN_LOAD_FIRST;
  }
  /*
   * Execute load command
   */
  if (skin->meta[XITK_SKIN_load_command] && (_skin_ref (skin, 1) <= 1)) {
    char *cmd = skin->sbuf + skin->meta[XITK_SKIN_load_command];
    if (!(skin->flags & XITK_SKIN_LOAD_TRUSTED)) {
      char *t = cmd + _skin_first_cmd (cmd);
      if (memcmp (t, "xset +fp ", 9)) {
        cmd = NULL;
      } else {
        t += _skin_next_cmd (t);
        if (*t)
          *t = 0;
      }
    }
    if (cmd) {
      int r = xitk_system (0, cmd);
      if (skin->xitk->verbosity + (r != 0) >= 2)
        printf ("xitk.skin.load.system (%s) = %d: %s.\n", skin->sbuf + skin->meta[XITK_SKIN_name], r, cmd);
    }
  }
    
  if (skin->xitk->verbosity >= 2)
    printf ("xitk.skin.load (%s): %u elements, %u string bytes.\n",
      skin->sbuf + skin->meta[XITK_SKIN_name],
      (unsigned int)xine_sarray_size (skin->elements), (unsigned int)skin->sbuf_pos);

  return skin->version;
}

/*
 * Unload (free) xitk_skin_config_t object.
 */
void xitk_skin_delete (xitk_skin_config_t **skin) {
  if (skin && *skin) {
    _skin_unload (*skin);
    xine_sarray_delete ((*skin)->elements);
    xitk_image_free_image (&(*skin)->missing);
    xine_sarray_delete ((*skin)->imgs);
    XITK_FREE (*skin);
    *skin = NULL;
  }
}

const char *xitk_skin_get_meta (xitk_skin_config_t *skin, xitk_skin_meta_t what) {
  return ((what < XITK_SKIN_LAST) && skin && skin->meta[what]) ? skin->sbuf + skin->meta[what] : NULL;
}

const xitk_skin_element_info_t *xitk_skin_get_info (xitk_skin_config_t *skin, const char *element_name) {
  xitk_skin_element_t *se;
  char name[64];
  int r;

  if (!skin || !element_name)
    return NULL;

  xitk_lower_strlcpy (name, element_name, sizeof (name));
  r = xine_sarray_binary_search (skin->elements, name);
  if (r < 0) {
    if (skin->xitk->verbosity >= 1)
      printf ("xitk.skin.section.missing (%s, %s).\n", skin->sbuf + skin->meta[XITK_SKIN_name], element_name);
    return NULL;
  }
  se = xine_sarray_get (skin->elements, r);

  if (!se->users) {
    xitk_part_image_t image;
    if (se->pixmap_name)
      _skin_load_img (skin, &se->info.pixmap_img, se->pixmap_name, 0);
    if (se->pm_font_name) {
      _skin_load_img (skin, &image, se->pm_font_name, se->pm_font_format);
      se->info.label_pixmap_font_img = image.image;
    }
    if (se->pm_highlight_font_name) {
      _skin_load_img (skin, &image, se->pm_highlight_font_name, se->pm_font_format);
      se->info.label_pixmap_highlight_font_img = image.image;
    }
    if (se->slider_pm_pad_name)
      _skin_load_img (skin, &se->info.slider_pixmap_pad_img, se->slider_pm_pad_name, 0);
  }
  se->users = (se->users & 0x7fffffff) + 1;
  return &se->info;
}
