#!/usr/bin/env python
# -*- coding: utf-8 -*-
# $Id: ArmBsdSpecCodeGen.py $

"""
ARM BSD / OpenSource specification code generator.
"""

from __future__ import print_function;

__copyright__ = \
"""
Copyright (C) 2025 Oracle and/or its affiliates.

This file is part of VirtualBox base platform packages, as
available from https://www.virtualbox.org.

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, in version 3 of the
License.

This program is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, see <https://www.gnu.org/licenses>.

SPDX-License-Identifier: GPL-3.0-only
"""
__version__ = "$Revision: 170501 $"

# pylint: disable=too-many-lines

# Standard python imports.
import argparse;
import collections;
import datetime;
import hashlib;
import operator;
import os;
import re;
import sys;
import time;

# Our imports:
from ArmAst import ArmAstAssignment;
from ArmAst import ArmAstBase;
from ArmAst import ArmAstBinaryOp;
from ArmAst import ArmAstBool;
from ArmAst import ArmAstConcat;
from ArmAst import ArmAstCppExpr;
from ArmAst import ArmAstCppExprBase;
from ArmAst import ArmAstCppCall;
from ArmAst import ArmAstDotAtom;
from ArmAst import ArmAstField;
from ArmAst import ArmAstFunction;
from ArmAst import ArmAstIdentifier;
from ArmAst import ArmAstIfList;
from ArmAst import ArmAstInteger;
from ArmAst import ArmAstReturn;
from ArmAst import ArmAstSet;
from ArmAst import ArmAstSlice;
from ArmAst import ArmAstSquareOp;
from ArmAst import ArmAstStatementList;
from ArmAst import ArmAstString;
from ArmAst import ArmAstTypeAnnotation;
from ArmAst import ArmAstUnaryOp;
from ArmAst import ArmAstValue;
import ArmAst;
import ArmBsdSpec as spec;

# Imports from the parent directory.
sys.path.append(os.path.dirname(os.path.dirname(os.path.abspath(__file__))));
import PyCommonVmm as pycmn;    # pylint: disable=import-error


## Program start time for logging.
g_nsProgStart = int(time.time_ns())


## Mapping from ARM FEAT_xxxx to CPUMFEATURESARMV8 member.
#
# The values are usually member names.  Alternatively, they can be fixed boolean
# values for things that are hardcoded (like mandatory feature, not in the spec).
# The special value -1 can be used as an alternative to False and will cause a
# todo comment to be generated.
#
# Sed script for extracting this stuff from cpum.h (-n option, sort output):
#
# # Match all comment lines with a (FEAT_XXX) in them.
# /[ (]FEAT_[A-Zn].*\*\//!d
#
# # Extract the feature string, quote it for dict key and pad to the value column.
# s,^.*[ (]\(FEAT_[A-Zn][^ )]*\)[ )].*\*\/.*$,'\1':,
# :morepadding
# s/$/ /
# /^................................/!b morepadding
#
# # Load the next line with the value, extract the member name and quote it for dict value.
# N
# s/\n *uint32_t  *\(f[a-zA-Z0-9][a-zA-Z0-9_]*\) * :.*$/'\1',/
# p
#
g_dSpecFeatToCpumFeat = {
    # Mandatory.
    'FEAT_EL0':                     True,
    'FEAT_EL1':                     True,

    # Generated by sed + sort:
    'FEAT_AA32':                    'fAa32',
    'FEAT_AA32BF16':                'fAa32Bf16',
    'FEAT_AA32EL0':                 'fAa32El0',
    'FEAT_AA32EL1':                 'fAa32El1',
    'FEAT_AA32EL2':                 'fAa32El2',
    'FEAT_AA32EL3':                 'fAa32El3',
    'FEAT_AA32HPD':                 'fAa32Hpd',
    'FEAT_AA32I8MM':                'fAa32I8mm',
    'FEAT_AA64':                    'fAa64',
    'FEAT_AA64EL0':                 'fAa64El0',
    'FEAT_AA64EL1':                 'fAa64El1',
    'FEAT_AA64EL2':                 'fAa64El2',
    'FEAT_AA64EL3':                 'fAa64El3',
    'FEAT_ABLE':                    'fAble',
    'FEAT_ADERR':                   'fAderr',
    'FEAT_AdvSIMD':                 'fAdvSimd',
    'FEAT_AES':                     'fAes',
    'FEAT_AFP':                     'fAfp',
    'FEAT_AIE':                     'fAie',
    'FEAT_AMUv1':                   'fAmuV1',
    'FEAT_AMUv1p1':                 'fAmuV1p1',
    'FEAT_ANERR':                   'fAnerr',
    'FEAT_ASID16':                  'fAsid16',
    'FEAT_ASID2':                   'fAsid2',
    'FEAT_ATS1A':                   'fATs1a',
    'FEAT_BBM':                     'fBbm',
    'FEAT_BF16':                    'fBf16',
    'FEAT_BRBE':                    'fBrbe',
    'FEAT_BRBEv1p1':                'fBrbeV1p1',
    'FEAT_BTI':                     'fBti',
    'FEAT_BWE':                     'fBwe',
    'FEAT_BWE2':                    'fBwe2',
    'FEAT_CCIDX':                   'fCcidx',
    'FEAT_CHK':                     'fChk',
    'FEAT_CLRBHB':                  'fClrBhb',
    'FEAT_CMOW':                    'fCmow',
    'FEAT_CMPBR':                   'fCmpBr',
    #'FEAT_CNTSC':                   'fCntsc',              # needs external registers for detection
    'FEAT_CONSTPACFIELD':           'fConstPacField',
    #'FEAT_CP15SDISABLE2':           'fCp15SDisable2', - can't be detected?
    'FEAT_CPA':                     'fCpa',
    'FEAT_CPA2':                    'fCpa2',
    'FEAT_CRC32':                   'fCrc32',
    'FEAT_CSSC':                    'fCssc',
    'FEAT_CSV2':                    'fCsv2',
    'FEAT_CSV2_1p1':                'fCsv21p1',
    'FEAT_CSV2_1p2':                'fCsv21p2',
    'FEAT_CSV2_2':                  'fCvs2_2',
    'FEAT_CSV2_3':                  'fCsv2v3',
    'FEAT_CSV3':                    'fCsv3',
    'FEAT_D128':                    'fD128',
    'FEAT_Debugv8p1':               'fDebugV8p1',
    'FEAT_Debugv8p2':               'fDebugV8p2',
    'FEAT_Debugv8p4':               'fDebugV8p4',
    'FEAT_Debugv8p8':               'fDebugV8p8',
    'FEAT_Debugv8p9':               'fDebugV8p9',
    'FEAT_DGH':                     'fDgh',
    'FEAT_DIT':                     'fDit',
    #'FEAT_DoPD':                    'fDopd',               # needs external registers for detection
    'FEAT_DotProd':                 'fDotProd',
    'FEAT_DoubleFault':             'fDoubleFault',
    'FEAT_DoubleFault2':            'fDoubleFault2',
    'FEAT_DoubleLock':              'fDoubleLock',
    'FEAT_DPB':                     'fDpb',
    'FEAT_DPB2':                    'fDpb2',
    'FEAT_E0PD':                    'fE0Pd',
    'FEAT_E2H0':                    'fE2H0',
    'FEAT_E3DSE':                   'fE3Dse',
    'FEAT_EBEP':                    'fEbep',
    'FEAT_EBF16':                   'fEbf16',
    'FEAT_ECBHB':                   'fEcBhb',
    'FEAT_ECV':                     'fEcv',
    'FEAT_ECV_POFF':                'fEcvPOff',
    #'FEAT_EDHSR':                   'fEdhsr',              # needs external registers for detection
    'FEAT_EL2':                     'fEl2',
    'FEAT_EL3':                     'fEl3',
    'FEAT_EPAC':                    'fEpac',
    'FEAT_ETE':                     'fEte',
    'FEAT_ETEv1p1':                 'fEteV1p1',
    'FEAT_ETEv1p2':                 'fEteV1p2',
    'FEAT_ETEv1p3':                 'fEteV1p3',
    'FEAT_ETMv4':                   'fEtmV4',
    'FEAT_ETMv4p1':                 'fEtmV4p1',
    'FEAT_ETMv4p2':                 'fEtmV4p2',
    'FEAT_ETMv4p3':                 'fEtmV4p3',
    'FEAT_ETMv4p4':                 'fEtmV4p4',
    'FEAT_ETMv4p5':                 'fEtmV4p5',
    'FEAT_ETMv4p6':                 'fEtmV4p6',
    'FEAT_ETS2':                    'fEts2',
    'FEAT_ETS3':                    'fEts3',
    'FEAT_EVT':                     'fEvt',
    'FEAT_ExS':                     'fExs',
    'FEAT_F32MM':                   'fF32mm',
    'FEAT_F64MM':                   'fF64mm',
    'FEAT_F8F16MM':                 'fF8F16mm',
    'FEAT_F8F32MM':                 'fF8F32mm',
    'FEAT_FAMINMAX':                'fFaMinMax',
    'FEAT_FCMA':                    'fFcma',
    'FEAT_FGT':                     'fFgt',
    'FEAT_FGT2':                    'fFgt2',
    'FEAT_FGWTE3':                  'fFgwtE3',
    'FEAT_FHM':                     'fFhm',
    'FEAT_FlagM':                   'fFlagM',
    'FEAT_FlagM2':                  'fFlagM2',
    'FEAT_FP':                      'fFp',
    'FEAT_FP16':                    'fFp16',
    'FEAT_FP8':                     'fFp8',
    'FEAT_FP8DOT2':                 'fFp8Dot2',
    'FEAT_FP8DOT4':                 'fFp8Dot4',
    'FEAT_FP8FMA':                  'fFp8Fma',
    'FEAT_FPAC':                    'fFpac',
    'FEAT_FPACC_SPEC':              'fFpaccSpec',
    'FEAT_FPACCOMBINE':             'fFpacCombine',
    'FEAT_FPMR':                    'fFpmr',
    'FEAT_FPRCVT':                  'fFpRcvt',
    'FEAT_FRINTTS':                 'fFrintts',
    'FEAT_GCS':                     'fGcs',
    'FEAT_GICv3':                   'fGicV3',
    'FEAT_GICv3_NMI':               'fGicV3Nmi',
    'FEAT_GICv3_TDIR':              'fGicV3Tdir',
    'FEAT_GICv3p1':                 'fGicV3p1',
    'FEAT_GICv4':                   'fGicV4',
    'FEAT_GICv4p1':                 'fGicV4p1',
    'FEAT_GTG':                     'fGtg',
    'FEAT_HACDBS':                  'fHacdbs',
    'FEAT_HAFDBS':                  'fHafdbs',
    'FEAT_HAFT':                    'fHaft',
    'FEAT_HBC':                     'fHbc',
    'FEAT_HCX':                     'fHcx',
    'FEAT_HDBSS':                   'fHdbss',
    'FEAT_HPDS':                    'fHpds',
    'FEAT_HPDS2':                   'fHpds2',
    'FEAT_HPMN0':                   'fHpmn0',
    'FEAT_I8MM':                    'fI8mm',
    'FEAT_IDST':                    'fIdst',
    'FEAT_IDTE3':                   'fIdte3',
    'FEAT_IESB':                    'fIesb',
    'FEAT_ITE':                     'fIte',
    'FEAT_IVIPT':                   'fIvipt',
    'FEAT_JSCVT':                   'fJscvt',
    'FEAT_LOR':                     'fLor',
    'FEAT_LPA':                     'fLpa',
    'FEAT_LPA2':                    'fLpa2',
    'FEAT_LRCPC':                   'fLrcpc',
    'FEAT_LRCPC2':                  'fLrcpc2',
    'FEAT_LRCPC3':                  'fLrcpc3',
    'FEAT_LS64':                    'fLs64',
    'FEAT_LS64_ACCDATA':            'fLs64Accdata',
    'FEAT_LS64_V':                  'fLs64V',
    'FEAT_LS64WB':                  'fLs64WB',
    'FEAT_LSE':                     'fLse',
    'FEAT_LSE128':                  'fLse128',
    'FEAT_LSE2':                    'fLse2',
    'FEAT_LSFE':                    'fLsfe',
    'FEAT_LSMAOC':                  'fLsmaoc',
    'FEAT_LSUI':                    'fLsui',
    'FEAT_LUT':                     'fLut',
    'FEAT_LVA':                     'fLva',
    'FEAT_LVA3':                    'fLva3',
    'FEAT_MEC':                     'fMec',
    'FEAT_MixedEnd':                'fMixedEnd',
    'FEAT_MixedEndEL0':             'fMixedEndEl0',
    'FEAT_MOPS':                    'fMops',
    'FEAT_MPAM':                    'fMpam',
    #'FEAT_MPAM_MSC_DCTRL':          'fMpamMscDCtrl',       # needs external registers for detection
    #'FEAT_MPAM_MSC_DOMAINS':        'fMpamMscDomains',     # needs external registers for detection
    'FEAT_MPAM_PE_BW_CTRL':         'fMpamPeBwCtrl',
    'FEAT_MPAMv0p1':                'fMpamV0p1',
    'FEAT_MPAMv1p1':                'fMpamV1p1',
    'FEAT_MTE':                     'fMte',
    'FEAT_MTE_ASYM_FAULT':          'fMteAsymFault',
    'FEAT_MTE_ASYNC':               'fMteAsync',
    'FEAT_MTE_CANONICAL_TAGS':      'fMteCanonicalTags',
    'FEAT_MTE_NO_ADDRESS_TAGS':     'fMteNoAddressTags',
    'FEAT_MTE_PERM':                'fMtePerm',
    #'FEAT_MTE_PERM_S1':             'fMtePermS1', # removed?
    'FEAT_MTE_STORE_ONLY':          'fMteStoreOnly',
    'FEAT_MTE_TAGGED_FAR':          'fMteTaggedFar',
    'FEAT_MTE2':                    'fMte2',
    'FEAT_MTE3':                    'fMte3',
    'FEAT_MTE4':                    'fMte4',
    'FEAT_MTPMU':                   'fMtPmu',
    'FEAT_NMI':                     'fNmi',
    'FEAT_nTLBPA':                  'fNTlbpa',
    'FEAT_NV':                      'fNv',
    'FEAT_NV2':                     'fNv2',
    'FEAT_NV2p1':                   'fNV2p1',
    'FEAT_OCCMO':                   'fOccmo',
    'FEAT_PACIMP':                  'fPacImp',
    'FEAT_PACQARMA3':               'fPacQarma3',
    'FEAT_PACQARMA5':               'fPacQarma5',
    'FEAT_PAN':                     'fPan',
    'FEAT_PAN2':                    'fPan2',
    'FEAT_PAN3':                    'fPan3',
    'FEAT_PAuth':                   'fPAuth',
    'FEAT_PAuth_LR':                'fPAuthLR',
    'FEAT_PAuth2':                  'fPAuth2',
    'FEAT_PCDPHINT':                'fPCDPHint',
    #'FEAT_PCSRv8':                  'fPcsrV8',             # needs external registers for detection
    #'FEAT_PCSRv8p2':                'fPcsrV8p2',           # needs external registers for detection
    #'FEAT_PCSRv8p9':                'fPcsrV8p9',           # needs external registers for detection
    'FEAT_PFAR':                    'fPfar',
    'FEAT_PMULL':                   'fPmull',
    'FEAT_PMUv3':                   'fPmuV3',
    'FEAT_PMUv3_EDGE':              'fPmuV3Edge',
    #'FEAT_PMUv3_EXT':               'fPmuV3Ext',           # difficult to detect
    #'FEAT_PMUv3_EXT32':             'fPmuV3Ext32',         # difficult to detect
    #'FEAT_PMUv3_EXT64':             'fPmuV3Ext64',         # difficult to detect
    'FEAT_PMUv3_ICNTR':             'fPmuV3Icntr',
    'FEAT_PMUv3_SME':               'fPmuV3Sme',
    'FEAT_PMUv3_SS':                'fPmuV3Ss',
    'FEAT_PMUv3_TH':                'fPmuV3Th',
    'FEAT_PMUv3_TH2':               'fPmuV3Th2',
    'FEAT_PMUv3p1':                 'fPmuV3p1',
    'FEAT_PMUv3p4':                 'fPmuV3p4',
    'FEAT_PMUv3p5':                 'fPmuV3p5',
    'FEAT_PMUv3p7':                 'fPmuV3p7',
    'FEAT_PMUv3p8':                 'fPmuV3p8',
    'FEAT_PMUv3p9':                 'fPmuV3p9',
    'FEAT_PRFMSLC':                 'fPrfmSlc',
    'FEAT_PoPS':                    'fPoPs',
    'FEAT_RAS':                     'fRas',
    #'FEAT_RASSAv1p1':               'fRassaV1p1',          # difficult to detect
    #'FEAT_RASSAv2':                 'fRasSaV2',            # difficult to detect
    'FEAT_RASv1p1':                 'fRasV1p1',
    'FEAT_RASv2':                   'fRasV2',
    'FEAT_RDM':                     'fRdm',
    'FEAT_RME':                     'fRme',
    'FEAT_RME_GDI':                 'fRmeGdi',
    'FEAT_RME_GPC2':                'fRmeGpc2',
    'FEAT_RME_GPC3':                'fRmeGpc3',
    'FEAT_RNG':                     'fRng',
    'FEAT_RNG_TRAP':                'fRngTrap',
    'FEAT_RPRES':                   'fRpres',
    'FEAT_RPRFM':                   'fRprfm',
    'FEAT_S1PIE':                   'fS1Pie',
    'FEAT_S1POE':                   'fS1Poe',
    'FEAT_S2FWB':                   'fS2Fwb',
    'FEAT_S2PIE':                   'fS2Pie',
    'FEAT_S2POE':                   'fS2Poe',
    'FEAT_S2TGran16K':              'fS2TGran16K',
    'FEAT_S2TGran4K':               'fS2TGran4K',
    'FEAT_S2TGran64K':              'fS2TGran64K',
    'FEAT_SB':                      'fSb',
    'FEAT_SCTLR2':                  'fSctlr2',
    'FEAT_SEBEP':                   'fSebep',
    'FEAT_SEL2':                    'fSecEl2',
    'FEAT_SHA1':                    'fSha1',
    'FEAT_SHA256':                  'fSha256',
    'FEAT_SHA3':                    'fSha3',
    'FEAT_SHA512':                  'fSha512',
    'FEAT_SM3':                     'fSm3',
    'FEAT_SM4':                     'fSm4',
    'FEAT_SME':                     'fSme',
    'FEAT_SME_B16B16':              'fSmeB16B16',
    'FEAT_SME_F16F16':              'fSmeF16F16',
    'FEAT_SME_F64F64':              'fSmeF64F64',
    'FEAT_SME_F8F16':               'fSmeF8F16',
    'FEAT_SME_F8F32':               'fSmeF8F32',
    'FEAT_SME_FA64':                'fSmeFA64',
    'FEAT_SME_I16I64':              'fSmeI16I64',
    'FEAT_SME_LUTv2':               'fSmeLutv2',
    'FEAT_SME_MOP4':                'fSmeMop4',
    'FEAT_SME_TMOP':                'fSmeTmop',
    'FEAT_SME2':                    'fSme2',
    'FEAT_SME2p1':                  'fSme2p1',
    'FEAT_SME2p2':                  'fSme2p2',
    'FEAT_SPE':                     'fSpe',
    'FEAT_SPE_ALTCLK':              'fSpeAltClk',
    'FEAT_SPE_CRR':                 'fSpeCrr',
    'FEAT_SPE_DPFZS':               'fSpeDpfzs',
    'FEAT_SPE_EFT':                 'fSpeEft',
    'FEAT_SPE_EXC':                 'fSpeExc',
    'FEAT_SPE_FDS':                 'fSpeFds',
    'FEAT_SPE_FnE':                 'fSpeFnE',
    'FEAT_SPE_FPF':                 'fSpeFpf',
    'FEAT_SPE_nVM':                 'fSpeNvm',
    'FEAT_SPE_PBT':                 'fSpePbt',
    'FEAT_SPE_SME':                 'fSpeSme',
    'FEAT_SPECRES':                 'fSpecres',
    'FEAT_SPECRES2':                'fSpecres2',
    'FEAT_SpecSEI':                 'fSpecSei',
    'FEAT_SPEv1p1':                 'fSpeV1p1',
    'FEAT_SPEv1p2':                 'fSpeV1p2',
    'FEAT_SPEv1p3':                 'fSpeV1p3',
    'FEAT_SPEv1p4':                 'fSpeV1p4',
    'FEAT_SPEv1p5':                 'fSpev1p5',
    'FEAT_SPMU':                    'fSpmu',
    'FEAT_SPMU2':                   'fSpmu2',
    'FEAT_SRMASK':                  'fSrMask',
    'FEAT_SSBS':                    'fSsbs',
    'FEAT_SSBS2':                   'fSsbs2',
    'FEAT_SSVE_AES':                'fSsveAes',
    'FEAT_SSVE_BitPerm':            'fSsveBitPerm',
    'FEAT_SSVE_FEXPA':              'fSsveFexpa',
    'FEAT_SSVE_FP8DOT2':            'fSsveFp8Dot2',
    'FEAT_SSVE_FP8DOT4':            'fSsveFp8Dot4',
    'FEAT_SSVE_FP8FMA':             'fSsveFp8Fma',
    'FEAT_STEP2':                   'fStep2',
    'FEAT_SVE':                     'fSve',
    'FEAT_SVE_AES':                 'fSveAes',
    'FEAT_SVE_AES2':                'fSveAes2',
    'FEAT_SVE_B16B16':              'fSveB16B16',
    'FEAT_SVE_BFSCALE':             'fSveBfscale',
    'FEAT_SVE_BitPerm':             'fSveBitPerm',
    'FEAT_SVE_F16F32MM':            'fSveF16F32mm',
    'FEAT_SVE_PMULL128':            'fSvePmull128',
    'FEAT_SVE_SHA3':                'fSveSha3',
    'FEAT_SVE_SM4':                 'fSveSm4',
    'FEAT_SVE2':                    'fSve2',
    'FEAT_SVE2p1':                  'fSve2p1',
    'FEAT_SVE2p2':                  'fSve2p2',
    'FEAT_SYSINSTR128':             'fSysInstr128',
    'FEAT_SYSREG128':               'fSysReg128',
    'FEAT_TCR2':                    'fTcr2',
    'FEAT_TGran16K':                'fTGran16K',
    'FEAT_TGran4K':                 'fTGran4K',
    'FEAT_TGran64K':                'fTGran64K',
    'FEAT_THE':                     'fThe',
    'FEAT_TIDCP1':                  'fTidcp1',
    'FEAT_TLBIOS':                  'fTlbios',
    'FEAT_TLBIRANGE':               'fTlbirange',
    'FEAT_TLBIW':                   'fTlbiW',
    'FEAT_TME':                     'fTme',
    'FEAT_TRBE':                    'fTrbe',
    'FEAT_TRBE_EXC':                'fTrbeExc',
    'FEAT_TRBE_EXT':                'fTrbeExt',
    'FEAT_TRBE_MPAM':               'fTrbeMpam',
    'FEAT_TRBEv1p1':                'fTrbev1p1',
    'FEAT_TRC_SR':                  'fTrcSr',
    'FEAT_TRF':                     'fTrf',
    'FEAT_TTCNP':                   'fTtcnp',
    'FEAT_TTL':                     'fTtl',
    'FEAT_TTST':                    'fTtst',
    'FEAT_TWED':                    'fTwed',
    'FEAT_UAO':                     'fUao',
    'FEAT_UINJ':                    'fUinj',
    'FEAT_VHE':                     'fVhe',
    'FEAT_VMID16':                  'fVmid16',
    'FEAT_VPIPT':                   'fVpipt',
    'FEAT_WFxT':                    'fWfxt',
    'FEAT_XNX':                     'fXnx',
    'FEAT_XS':                      'fXs',
};


## Mapping register names to CPUMCTX members.
## The value is: (sName|fReservedValue,)
g_dRegToCpumCtx = {
    'AArch64.AMUSERENR_EL0':    ( 0, ), # We don't support this, so report zero.
    'AArch64.CNTP_CTL_EL0':     ( 0, ), # We don't seem to support this, so report zero.
    'AArch64.GCSCRE0_EL1':      ( 0, ), # FEAT_GCS / ARMv9.4; we don't support this, returning zero for now.
    'AArch64.GCSCR_EL1':        ( 0, ), # FEAT_GCS / ARMv9.4; we don't support this, returning zero for now.
    'AArch64.GCSCR_EL2':        ( 0, ), # FEAT_GCS / ARMv9.4; we don't support this, returning zero for now.
    'AArch64.GCSCR_EL3':        ( 0, ), # FEAT_GCS / ARMv9.4; we don't support this, returning zero for now.
    'AArch64.HCRX_EL2':         ( 0, ), # FEAT_HCX / ARMv8.7; we don't support this, returning zero for now.
    'AArch64.HDFGRTR_EL2':      ( 0, ), # FEAT_FGT / ARMv8.6; we don't support this, returning zero for now.
    'AArch64.HDFGWTR_EL2':      ( 0, ), # FEAT_FGT / ARMv8.6; we don't support this, returning zero for now.
    'AArch64.HFGRTR_EL2':       ( 0, ), # FEAT_FGT / ARMv8.6; we don't support this, returning zero for now.
    'AArch64.HFGWTR_EL2':       ( 0, ), # FEAT_FGT / ARMv8.6; we don't support this, returning zero for now.
    'AArch64.HFGITR_EL2':       ( 0, ), # FEAT_FGT / ARMv8.6; we don't support this, returning zero for now.
    ## @todo 'AArch64.ICC_SRE_EL1': ( '.u64', ), - GIC has debug logging of this, but no handling.
    ## @todo 'AArch64.ICC_SRE_EL2': ( '.u64', ), - GIC lists this in the range, but not sure how to get at any value...
    ## @todo 'AArch64.ICH_HCR_EL2': ( '.u64', ), - This is part of GICv3, but our GIC doesn't mention it.
    ## @todo 'AArch64.MPAM2_EL2':   ( '.u64', ), - FEAT_MPAM / ARMv8.4; not sure if zero is a passable value here...
    ## @todo 'AArch64.MPAMBW2_EL2': ( '.u64', ), - FEAT_MPAM_PE_BW_CTRL / ARMv9.3; ditto.
    ## @todo 'AArch64.MPAMHCR_EL2': ( '.u64', ), - FEAT_MPAM / ARMv8.4; not sure if zero is a passable value here...
    ## @todo 'AArch64.MPAMIDR_EL1': ( '.u64', ), - FEAT_MPAM / ARMv8.4; not sure if zero is a passable value here...
    ## @todo 'AArch64.OSLSR_EL1':        ( 'fOsLck', ),  - this is a bool in our code...
    'AArch64.PMSCR_EL1':        ( 0, ), # FEAT_SPE / ARMv8.1
    ## @todo 'AArch64.PMSELR_EL0':  ( '.u64', ), - FEAT_PMUv3 / ARMv8.0; not sure if zero is a passable value here...
    ## @todo 'AArch64.PMUACR_EL1':  ( '.u64', ), - FEAT_PMUv3p9; not sure if zero is a passable value here...
    ## @todo 'AArch64.SCR_EL3':     ( '.u64', ), - we don't do EL3, so some stuff isn't eliminated properly...
    'AArch64.SCR_EL3':          ( 0, ), # CNTPCTSS_EL0 and others accesses it w/o HaveEL(EL3). ## @todo EL3
    ## @todo 'AArch64.SPMSELR_EL0': ( '.u64', ), - FEAT_SPMU / ARMv8.8
    ## @todo 'AArch64.TRCIDR0':     ( '.u64', ), - FEAT_ETE / ARMv9.0
    ## @todo 'AArch64.TRCIDR2':     ( '.u64', ), - FEAT_ETE / ARMv9.0
    ## @todo 'AArch64.TRCIDR3':     ( '.u64', ), - FEAT_ETE / ARMv9.0
    ## @todo 'AArch64.TRCIDR4':     ( '.u64', ), - FEAT_ETE / ARMv9.0
    ## @todo 'AArch64.TRCIDR5':     ( '.u64', ), - FEAT_ETE / ARMv9.0
    ## @todo 'AArch64.TRFCR_EL1':   ( '.u64', ), - FEAT_TRF  / ??

    'AArch64.SP_EL0':           ('aSpReg[0].u64',),
    'AArch64.SP_EL1':           ('aSpReg[1].u64',),
    'AArch64.SP_EL2':           ('aSpReg[2].u64',),

    'AArch64.SPSR_EL1':         ('Spsr.u64',),
    'AArch64.ELR_EL1':          ('Elr.u64',),
    'AArch64.SCTLR_EL1':        ('Sctlr.u64',),
    'AArch64.TCR_EL1':          ('Tcr.u64',),
    'AArch64.TTBR0_EL1':        ('Ttbr0.u64',),
    'AArch64.TTBR1_EL1':        ('Ttbr1.u64',),
    'AArch64.VBAR_EL1':         ('VBar.u64',),
    'AArch64.MDSCR_EL1':        ('Mdscr.u64',),

    'AArch64.DBGBCR0_EL1':      ('aBp[0].Ctrl.u64',),
    'AArch64.DBGBCR1_EL1':      ('aBp[1].Ctrl.u64',),
    'AArch64.DBGBCR2_EL1':      ('aBp[2].Ctrl.u64',),
    'AArch64.DBGBCR3_EL1':      ('aBp[3].Ctrl.u64',),
    'AArch64.DBGBCR4_EL1':      ('aBp[4].Ctrl.u64',),
    'AArch64.DBGBCR5_EL1':      ('aBp[5].Ctrl.u64',),
    'AArch64.DBGBCR6_EL1':      ('aBp[6].Ctrl.u64',),
    'AArch64.DBGBCR7_EL1':      ('aBp[7].Ctrl.u64',),
    'AArch64.DBGBCR8_EL1':      ('aBp[8].Ctrl.u64',),
    'AArch64.DBGBCR9_EL1':      ('aBp[9].Ctrl.u64',),
    'AArch64.DBGBCR10_EL1':     ('aBp[10].Ctrl.u64',),
    'AArch64.DBGBCR11_EL1':     ('aBp[11].Ctrl.u64',),
    'AArch64.DBGBCR12_EL1':     ('aBp[12].Ctrl.u64',),
    'AArch64.DBGBCR13_EL1':     ('aBp[13].Ctrl.u64',),
    'AArch64.DBGBCR14_EL1':     ('aBp[14].Ctrl.u64',),
    'AArch64.DBGBCR15_EL1':     ('aBp[15].Ctrl.u64',),

    'AArch64.DBGBVR0_EL1':      ('aBp[0].Value.u64',),
    'AArch64.DBGBVR1_EL1':      ('aBp[1].Value.u64',),
    'AArch64.DBGBVR2_EL1':      ('aBp[2].Value.u64',),
    'AArch64.DBGBVR3_EL1':      ('aBp[3].Value.u64',),
    'AArch64.DBGBVR4_EL1':      ('aBp[4].Value.u64',),
    'AArch64.DBGBVR5_EL1':      ('aBp[5].Value.u64',),
    'AArch64.DBGBVR6_EL1':      ('aBp[6].Value.u64',),
    'AArch64.DBGBVR7_EL1':      ('aBp[7].Value.u64',),
    'AArch64.DBGBVR8_EL1':      ('aBp[8].Value.u64',),
    'AArch64.DBGBVR9_EL1':      ('aBp[9].Value.u64',),
    'AArch64.DBGBVR10_EL1':     ('aBp[10].Value.u64',),
    'AArch64.DBGBVR11_EL1':     ('aBp[11].Value.u64',),
    'AArch64.DBGBVR12_EL1':     ('aBp[12].Value.u64',),
    'AArch64.DBGBVR13_EL1':     ('aBp[13].Value.u64',),
    'AArch64.DBGBVR14_EL1':     ('aBp[14].Value.u64',),
    'AArch64.DBGBVR15_EL1':     ('aBp[15].Value.u64',),

    'AArch64.DBGWCR0_EL1':      ('aWp[0].Ctrl.u64',),
    'AArch64.DBGWCR1_EL1':      ('aWp[1].Ctrl.u64',),
    'AArch64.DBGWCR2_EL1':      ('aWp[2].Ctrl.u64',),
    'AArch64.DBGWCR3_EL1':      ('aWp[3].Ctrl.u64',),
    'AArch64.DBGWCR4_EL1':      ('aWp[4].Ctrl.u64',),
    'AArch64.DBGWCR5_EL1':      ('aWp[5].Ctrl.u64',),
    'AArch64.DBGWCR6_EL1':      ('aWp[6].Ctrl.u64',),
    'AArch64.DBGWCR7_EL1':      ('aWp[7].Ctrl.u64',),
    'AArch64.DBGWCR8_EL1':      ('aWp[8].Ctrl.u64',),
    'AArch64.DBGWCR9_EL1':      ('aWp[9].Ctrl.u64',),
    'AArch64.DBGWCR10_EL1':     ('aWp[10].Ctrl.u64',),
    'AArch64.DBGWCR11_EL1':     ('aWp[11].Ctrl.u64',),
    'AArch64.DBGWCR12_EL1':     ('aWp[12].Ctrl.u64',),
    'AArch64.DBGWCR13_EL1':     ('aWp[13].Ctrl.u64',),
    'AArch64.DBGWCR14_EL1':     ('aWp[14].Ctrl.u64',),
    'AArch64.DBGWCR15_EL1':     ('aWp[15].Ctrl.u64',),

    'AArch64.DBGWVR0_EL1':      ('aWp[0].Value.u64',),
    'AArch64.DBGWVR1_EL1':      ('aWp[1].Value.u64',),
    'AArch64.DBGWVR2_EL1':      ('aWp[2].Value.u64',),
    'AArch64.DBGWVR3_EL1':      ('aWp[3].Value.u64',),
    'AArch64.DBGWVR4_EL1':      ('aWp[4].Value.u64',),
    'AArch64.DBGWVR5_EL1':      ('aWp[5].Value.u64',),
    'AArch64.DBGWVR6_EL1':      ('aWp[6].Value.u64',),
    'AArch64.DBGWVR7_EL1':      ('aWp[7].Value.u64',),
    'AArch64.DBGWVR8_EL1':      ('aWp[8].Value.u64',),
    'AArch64.DBGWVR9_EL1':      ('aWp[9].Value.u64',),
    'AArch64.DBGWVR10_EL1':     ('aWp[10].Value.u64',),
    'AArch64.DBGWVR11_EL1':     ('aWp[11].Value.u64',),
    'AArch64.DBGWVR12_EL1':     ('aWp[12].Value.u64',),
    'AArch64.DBGWVR13_EL1':     ('aWp[13].Value.u64',),
    'AArch64.DBGWVR14_EL1':     ('aWp[14].Value.u64',),
    'AArch64.DBGWVR15_EL1':     ('aWp[15].Value.u64',),

    'AArch64.APDAKeyLo_EL1':    ('Apda.Low.u64',),
    'AArch64.APDAKeyHi_EL1':    ('Apda.High.u64',),

    'AArch64.APDBKeyLo_EL1':    ('Apdb.Low.u64',),
    'AArch64.APDBKeyHi_EL1':    ('Apdb.High.u64',),

    'AArch64.APGAKeyLo_EL1':    ('Apga.Low.u64',),
    'AArch64.APGAKeyHi_EL1':    ('Apga.High.u64',),

    'AArch64.APIAKeyLo_EL1':    ('Apia.Low.u64',),
    'AArch64.APIAKeyHi_EL1':    ('Apia.High.u64',),

    'AArch64.APIBKeyLo_EL1':    ('Apib.Low.u64',),
    'AArch64.APIBKeyHi_EL1':    ('Apib.High.u64',),

    'AArch64.AFSR0_EL1':        ('Afsr0.u64',),
    'AArch64.AFSR1_EL1':        ('Afsr1.u64',),
    'AArch64.AMAIR_EL1':        ('Amair.u64',),
    'AArch64.CNTKCTL_EL1':      ('CntKCtl.u64',),
    'AArch64.CONTEXTIDR_EL1':   ('ContextIdr.u64',),
    'AArch64.CPACR_EL1':        ('Cpacr.u64',),
    'AArch64.CSSELR_EL1':       ('Csselr.u64',),
    'AArch64.ESR_EL1':          ('Esr.u64',),
    'AArch64.FAR_EL1':          ('Far.u64',),
    'AArch64.MAIR_EL1':         ('Mair.u64',),
    'AArch64.PAR_EL1':          ('Par.u64',),
    'AArch64.TPIDRRO_EL0':      ('TpIdrRoEl0.u64',),
    'AArch64.TPIDR_EL0':        ('aTpIdr[0].u64',),
    'AArch64.TPIDR_EL1':        ('aTpIdr[1].u64',),
    'AArch64.TPIDR_EL2':        ('aTpIdr[2].u64',),
    'AArch64.MDCCINT_EL1':      ('MDccInt.u64',),
    'AArch64.ACTLR_EL1':        ('Actlr.u64',),

    'AArch64.CNTHCTL_EL2':      ('CntHCtlEl2.u64',),
    'AArch64.CNTHP_CTL_EL2':    ('CntHpCtlEl2.u64',),
    'AArch64.CNTHP_CVAL_EL2':   ('CntHpCValEl2.u64',),
    'AArch64.CNTHP_TVAL_EL2':   ('CntHpTValEl2.u64',),
    'AArch64.CNTVOFF_EL2':      ('CntVOffEl2.u64',),
    'AArch64.CPTR_EL2':         ('CptrEl2.u64',),
    'AArch64.ELR_EL2':          ('ElrEl2.u64',),
    'AArch64.ESR_EL2':          ('EsrEl2.u64',),
    'AArch64.FAR_EL2':          ('FarEl2.u64',),
    'AArch64.HCR_EL2':          ('HcrEl2.u64',),
    'AArch64.HPFAR_EL2':        ('HpFarEl2.u64',),
    'AArch64.MAIR_EL2':         ('MairEl2.u64',),
    'AArch64.MDCR_EL2':         ('MdcrEl2.u64',),
    'AArch64.SCTLR_EL2':        ('SctlrEl2.u64',),
    'AArch64.SPSR_EL2':         ('SpsrEl2.u64',),
    'AArch64.TCR_EL2':          ('TcrEl2.u64',),
    'AArch64.TTBR0_EL2':        ('Ttbr0El2.u64',),
    'AArch64.TTBR1_EL2':        ('Ttbr1El2.u64',),
    'AArch64.VBAR_EL2':         ('VBarEl2.u64',),
    'AArch64.VMPIDR_EL2':       ('VMpidrEl2.u64',),
    'AArch64.VPIDR_EL2':        ('VPidrEl2.u64',),
    'AArch64.VTCR_EL2':         ('VTcrEl2.u64',),
    'AArch64.VTTBR_EL2':        ('VTtbrEl2.u64',),
    'AArch64.FPCR':             ('fpcr',),
    'AArch64.FPSR':             ('fpsr',),
    'AArch64.CNTV_CTL_EL0':     ('CntvCtlEl0',),
    'AArch64.CNTV_CVAL_EL0':    ('CntvCValEl0',),
};


#
# Enum mappings.
#

kdMapELx = {
    'EL0':                          '0',
    'EL1':                          '1',
    'EL2':                          '2',
    'EL3':                          '3',
};

kdMapTranslationStage = {
    'TranslationStage_1':           'kIemCImplA64TranslationStage_1',
    'TranslationStage_12':          'kIemCImplA64TranslationStage_12',
};
kdMapATAccess = {
    'ATAccess_Read':                'kIemCImplA64ATAccess_Read',
    'ATAccess_Write':               'kIemCImplA64ATAccess_Write',
    'ATAccess_Any':                 'kIemCImplA64ATAccess_Any',
    'ATAccess_ReadPAN':             'kIemCImplA64ATAccess_ReadPan',
    'ATAccess_WritePAN':            'kIemCImplA64ATAccess_WritePan',
};
kdMapCacheType = {
    'CacheType_Data':               'kIemCImplA64CacheType_Data',
    'CacheType_Tag':                'kIemCImplA64CacheType_Tag',
    'CacheType_Data_Tag':           'kIemCImplA64CacheType_DataTag',
    'CacheType_Instruction':        'kIemCImplA64CacheType_Instruction',
};
kdMapCacheOp = {
    'CacheOp_Clean':                'kIemCImplA64CacheOp_Clean',
    'CacheOp_Invalidate':           'kIemCImplA64CacheOp_Invalidate',
    'CacheOp_CleanInvalidate':      'kIemCImplA64CacheOp_CleanInvalidate',
};
kdMapCacheOpScope = {
    'CacheOpScope_SetWay':          'kIemCImplA64CacheOpScope_SetWay',
    'CacheOpScope_PoU':             'kIemCImplA64CacheOpScope_PoU',
    'CacheOpScope_PoC':             'kIemCImplA64CacheOpScope_PoC',
    'CacheOpScope_PoE':             'kIemCImplA64CacheOpScope_PoE',
    'CacheOpScope_PoP':             'kIemCImplA64CacheOpScope_PoP',
    'CacheOpScope_PoDP':            'kIemCImplA64CacheOpScope_PoDP',
    'CacheOpScope_PoPA':            'kIemCImplA64CacheOpScope_PoPA',
    'CacheOpScope_PoPS':            'kIemCImplA64CacheOpScope_PoPS',       #
    'CacheOpScope_ALLU':            'kIemCImplA64CacheOpScope_AllU',
    'CacheOpScope_ALLUIS':          'kIemCImplA64CacheOpScope_AllUIS',
    'CacheOpScope_OuterCache':      'kIemCImplA64CacheOpScope_OuterCache', #
};
kdMapRestrictType = {
    'RestrictType_CachePrefetch':   'kIemCImplA64RestrictType_CachePrefetch',
    'RestrictType_ControlFlow':     'kIemCImplA64RestrictType_ControlFlow',
    'RestrictType_DataValue':       'kIemCImplA64RestrictType_DataValue',
    'RestrictType_Other':           'kIemCImplA64RestrictType_Other',
};
kdMapRegime = {
    'Regime_EL10':                  'kIemCImplA64Regime_El10',
    'Regime_EL20':                  'kIemCImplA64Regime_El20',
    'Regime_EL2':                   'kIemCImplA64Regime_El2',
    'Regime_EL30':                  'kIemCImplA64Regime_El30',
    'Regime_EL3':                   'kIemCImplA64Regime_El3',
};
kdMapBroadcast = {
    'Broadcast_ISH':                'kIemCImplA64Broadcast_Ish',
    'Broadcast_ForcedISH':          'kIemCImplA64Broadcast_ForcedIsh',
    'Broadcast_OSH':                'kIemCImplA64Broadcast_Osh',
    'Broadcast_NSH':                'kIemCImplA64Broadcast_Nsh',
};
kdMapTlbiMemAttr = {
    'TLBI_AllAttr':                 'kIemCImplA64TlbiMemAttr_AllAttr',
    'TLBI_ExcludeXS':               'kIemCImplA64TlbiMemAttr_ExcludeXs',
};
kdMapTlbiLevel = {
    'TLBILevel_Any':                'kIemCImplA64TlbiLevel_Any',
    'TLBILevel_Last':               'kIemCImplA64TlbiLevel_Last',
};
kdMapVmId = {
    'VMID_NONE':                    '0 /*VMID_NONE*/'
};


#
# Decoder structure helpers.
#

## Populated by --decoder-hint0
g_dDecoderFilterDepth0 = { };

## Populated by --decoder-hint1
g_ddDecoderFilterDepth1 = { };


class MaskZipper(object):
    """
    This is mainly a class for putting static methods relating to mask
    packing and unpack.
    """

    def __init__(self):
        pass;

    @staticmethod
    def compileAlgo(fMask):
        """
        Returns an with instructions for extracting the bits from the mask into
        a compacted form. Each array entry is an array/tuple of source bit [0],
        destination bit [1], and bit counts [2].
        """
        aaiAlgo   = [];
        iSrcBit   = 0;
        iDstBit   = 0;
        while fMask > 0:
            # Skip leading zeros.
            cSkip    = (fMask & -fMask).bit_length() - 1;
            #assert (fMask & ((1 << cSkip) - 1)) == 0 and ((fMask >> cSkip) & 1), 'fMask=%#x cSkip=%d' % (fMask, cSkip)
            iSrcBit += cSkip;
            fMask  >>= cSkip;

            # Calculate leading ones the same way.
            cCount = (~fMask & -~fMask).bit_length() - 1;
            #assert (fMask & ((1 << cCount) - 1)) == ((1 << cCount) - 1) and (fMask & (1 << cCount)) == 0

            # Append to algo list.
            aaiAlgo.append((iSrcBit, iDstBit, (1 << cCount) - 1));

            # Advance.
            iDstBit += cCount;
            iSrcBit += cCount;
            fMask  >>= cCount;
        return aaiAlgo;

    @staticmethod
    def compileAlgoLimited(fMask):
        """
        Version of compileAlgo that returns an empty list if there are
        more than three sections.
        """
        #assert fMask;

        #
        # Chunk 0:
        #

        # Skip leading zeros.
        iSrcBit0 = (fMask & -fMask).bit_length() - 1;
        fMask  >>= iSrcBit0;
        # Calculate leading ones the same way.
        cCount0  = (~fMask & -~fMask).bit_length() - 1;
        fMask  >>= cCount0;
        if not fMask:
            return [(iSrcBit0, 0, (1 << cCount0) - 1)];

        #
        # Chunk 1:
        #

        # Skip leading zeros.
        cSrcGap1 = (fMask & -fMask).bit_length() - 1;
        fMask  >>= cSrcGap1;
        # Calculate leading ones the same way.
        cCount1  = (~fMask & -~fMask).bit_length() - 1;
        fMask  >>= cCount1;
        if not fMask:
            return [ (iSrcBit0, 0, (1 << cCount0) - 1),
                     (iSrcBit0 + cCount0 + cSrcGap1, cCount0, (1 << cCount1) - 1)];

        #
        # Chunk 2:
        #

        # Skip leading zeros.
        cSrcGap2 = (fMask & -fMask).bit_length() - 1;
        fMask  >>= cSrcGap2;
        # Calculate leading ones the same way.
        cCount2  = (~fMask & -~fMask).bit_length() - 1;
        fMask  >>= cCount2;
        if not fMask:
            iSrcBit1 = iSrcBit0 + cCount0 + cSrcGap1;
            return [ (iSrcBit0, 0, (1 << cCount0) - 1),
                     (iSrcBit1, cCount0, (1 << cCount1) - 1),
                     (iSrcBit1 + cCount1 + cSrcGap2, cCount0 + cCount1, (1 << cCount2) - 1), ];

        # Too many fragments.
        return [];

    @staticmethod
    def compileAlgoFromList(aiOrderedBits):
        """
        Returns an with instructions for extracting the bits from the mask into
        a compacted form. Each array entry is an array/tuple of source bit [0],
        destination bit [1], and mask (shifted to pos 0) [2].
        """
        aaiAlgo = [];
        iDstBit = 0;
        i       = 0;
        while i < len(aiOrderedBits):
            iSrcBit = aiOrderedBits[i];
            cCount  = 1;
            i      += 1;
            while i < len(aiOrderedBits) and aiOrderedBits[i] == iSrcBit + cCount:
                cCount += 1;
                i      += 1;
            aaiAlgo.append([iSrcBit, iDstBit, (1 << cCount) - 1])
            iDstBit += cCount;
        return aaiAlgo;

    @staticmethod
    def algoToBitList(aaiAlgo):
        aiRet = [];
        for iSrcBit, _, fMask in aaiAlgo:
            cCount = fMask.bit_count();
            aiRet += [iSrcBit + i for i in range(cCount)];
        return aiRet;

    @staticmethod
    def zipMask(uValue, aaiAlgo):
        idxRet = 0;
        for iSrcBit, iDstBit, fMask in aaiAlgo:
            idxRet |= ((uValue >> iSrcBit) & fMask) << iDstBit;
        return idxRet;

    @staticmethod
    def __zipMask1(uValue, aaiAlgo):
        iSrcBit, _, fMask = aaiAlgo[0];
        return (uValue >> iSrcBit) & fMask;

    @staticmethod
    def __zipMask2(uValue, aaiAlgo):
        iSrcBit0, _,        fMask0 = aaiAlgo[0];
        iSrcBit1, iDstBit1, fMask1 = aaiAlgo[1];
        return ((uValue >> iSrcBit0) & fMask0) | (((uValue >> iSrcBit1) & fMask1) << iDstBit1);

    @staticmethod
    def __zipMask3(uValue, aaiAlgo):
        iSrcBit0, _,        fMask0 = aaiAlgo[0];
        iSrcBit1, iDstBit1, fMask1 = aaiAlgo[1];
        iSrcBit2, iDstBit2, fMask2 = aaiAlgo[2];
        return ((uValue >> iSrcBit0) & fMask0) \
             | (((uValue >> iSrcBit1) & fMask1) << iDstBit1) \
             | (((uValue >> iSrcBit2) & fMask2) << iDstBit2);

    @staticmethod
    def algoToZipLambda(aaiAlgo, fAlgoMask, fCompileIt = True):
        assert aaiAlgo;
        if not fCompileIt:
            if len(aaiAlgo) == 1: return MaskZipper.__zipMask1;
            if len(aaiAlgo) == 2: return MaskZipper.__zipMask2;
            if len(aaiAlgo) == 3: return MaskZipper.__zipMask3;
            return MaskZipper.zipMask;
        # Compile it:
        sBody = '';
        for iSrcBit, iDstBit, fMask in aaiAlgo:
            if sBody:
                sBody += ' | ';
            assert iSrcBit >= iDstBit;
            if iDstBit == 0:
                if iSrcBit == 0:
                    sBody += '(uValue & %#x)' % (fMask,);
                else:
                    sBody += '((uValue >> %u) & %#x)' % (iSrcBit, fMask);
            else:
                sBody += '((uValue >> %u) & %#x)' % (iSrcBit - iDstBit, fMask << iDstBit);
        _ = fAlgoMask
        #sFn = 'zipMaskCompiled_%#010x' % (fAlgoMask,);
        #sFn = 'zipMaskCompiled';
        #dTmp = {};
        #exec('def %s(uValue,_): return %s' % (sFn, sBody), globals(), dTmp);
        #return dTmp[sFn];
        return eval('lambda uValue,_: ' + sBody);

    @staticmethod
    def unzipMask(uValue, aaiAlgo):
        fRet = 0;
        for iSrcBit, iDstBit, fMask in aaiAlgo:
            fRet |= ((uValue >> iDstBit) & fMask) << iSrcBit;
        return fRet;

    @staticmethod
    def __unzipMask1(uValue, aaiAlgo):
        return uValue << aaiAlgo[0][0];

    @staticmethod
    def __unzipMask2(uValue, aaiAlgo):
        iSrcBit0, _,        fMask0 = aaiAlgo[0];
        iSrcBit1, iDstBit1, fMask1 = aaiAlgo[1];
        return ((uValue & fMask0) << iSrcBit0) | (((uValue >> iDstBit1) & fMask1) << iSrcBit1);

    @staticmethod
    def __unzipMask3(uValue, aaiAlgo):
        iSrcBit0, _,        fMask0 = aaiAlgo[0];
        iSrcBit1, iDstBit1, fMask1 = aaiAlgo[1];
        iSrcBit2, iDstBit2, fMask2 = aaiAlgo[2];
        return ((uValue & fMask0) << iSrcBit0) \
             | (((uValue >> iDstBit1) & fMask1) << iSrcBit1) \
             | (((uValue >> iDstBit2) & fMask2) << iSrcBit2);

    @staticmethod
    def algoToUnzipLambda(aaiAlgo, fAlgoMask, fCompileIt = True):
        assert aaiAlgo;
        if not fCompileIt:
            if len(aaiAlgo) == 1: return MaskZipper.__unzipMask1;
            if len(aaiAlgo) == 2: return MaskZipper.__unzipMask2;
            if len(aaiAlgo) == 3: return MaskZipper.__unzipMask3;
            return MaskZipper.unzipMask;
        # Compile it:
        sBody = '';
        for iSrcBit, iDstBit, fMask in aaiAlgo:
            if sBody:
                sBody += ' | ';
            if iDstBit == 0:
                if iSrcBit == 0:
                    sBody += '(uIdx & %#x)' % (fMask,);
                else:
                    sBody += '((uIdx & %#x) << %u)' % (fMask, iSrcBit);
            else:
                sBody += '((uIdx << %u) & %#x)' % (iSrcBit - iDstBit, fMask << iSrcBit);

        _ = fAlgoMask
        #dTmp = {};
        #sFn = 'unzipMaskCompiled';
        #sFn = 'unzipMaskCompiled_%#010x' % (fAlgoMask,);
        #exec('def %s(uIdx,_): return %s' % (sFn, sBody), globals(), dTmp);
        #return dTmp[sFn];
        return eval('lambda uIdx,_: ' + sBody);


class MaskIterator(object):
    """ Helper class for DecoderNode.constructNextLevel(). """

    ## Maximum number of mask sub-parts.
    # Lower number means fewer instructions required to convert it into an index.
    # This is implied by the code in MaskZipper.compileAlgoLimited.
    kcMaxMaskParts = 3

    def __init__(self, fMask, cMinTableSizeInBits, cMaxTableSizeInBits, fMaskNotDoneYet):
        self.fMask               = fMask;
        self.aaiAlgo             = MaskZipper.compileAlgo(fMask);
        self.fCompactMask        = MaskZipper.zipMask(fMask, self.aaiAlgo);
        self.fnExpandMask        = MaskZipper.algoToUnzipLambda(self.aaiAlgo, fMask,
                                                                self.fCompactMask.bit_count() >= 8);
        self.cMinTableSizeInBits = cMinTableSizeInBits;
        self.cMaxTableSizeInBits = cMaxTableSizeInBits;
        self.fCompactMaskNotDoneYet = MaskZipper.zipMask(fMaskNotDoneYet, self.aaiAlgo);
        #print('debug: fMask=%#x -> fCompactMask=%#x aaiAlgo=%s' % (fMask, self.fCompactMask, self.aaiAlgo));
        #self.cReturned           = 0;

    def __iter__(self):
        return self;

    def __next__(self):
        fCompactMask           = self.fCompactMask;
        fCompactMaskNotDoneYet = self.fCompactMaskNotDoneYet;
        cMinTableSizeInBits    = self.cMinTableSizeInBits
        cMaxTableSizeInBits    = self.cMaxTableSizeInBits
        while fCompactMask != 0:
            if fCompactMask & fCompactMaskNotDoneYet:
                cCurBits = fCompactMask.bit_count();
                if cMinTableSizeInBits <= cCurBits <= cMaxTableSizeInBits:
                    fMask = self.fnExpandMask(fCompactMask, self.aaiAlgo);
                    aaiMaskAlgo = MaskZipper.compileAlgoLimited(fMask);
                    if aaiMaskAlgo:
                        #assert aaiMaskAlgo == MaskZipper.compileAlgo(fMask), \
                        #    '%s vs %s' % (aaiMaskAlgo, MaskZipper.compileAlgo(fMask));
                        #self.cReturned += 1;
                        self.fCompactMask = fCompactMask - 1;
                        return (fMask, cCurBits, aaiMaskAlgo);
            fCompactMask -= 1;
        self.fCompactMask = 0;
        #print('MaskIterator: fMask=%#x -> %u items returned' % (self.fMask, self.cReturned));
        raise StopIteration;


class DecoderNode(object):

    ## The absolute maximum table size in bits index by the log2 of the instruction count.
    kacMaxTableSizesInBits = (
        2,      # [2^0 =     1] =>     4
        4,      # [2^1 =     2] =>    16
        5,      # [2^2 =     4] =>    32
        6,      # [2^3 =     8] =>    64
        7,      # [2^4 =    16] =>   128
        7,      # [2^5 =    32] =>   128
        8,      # [2^6 =    64] =>   256
        9,      # [2^7 =   128] =>   512
        10,     # [2^8 =   256] =>  1024
        11,     # [2^9 =   512] =>  2048
        12,     # [2^10 = 1024] =>  4096
        13,     # [2^11 = 2048] =>  8192
        14,     # [2^12 = 4096] => 16384
        14,     # [2^13 = 8192] => 16384
        15,     # [2^14 =16384] => 32768
    );

    kfChildMaskOpcodeValueIf          = 0x7fffffff;
    kfChildMaskMultipleOpcodeValueIfs = 0xffffffff;

    class TooExpensive(Exception):
        def __init__(self):
            Exception.__init__(self, None);

    def __init__(self, aoInstructions, fCheckedMask, fCheckedValue):
        #assert (~fCheckedMask & fCheckedValue) == 0;
        #for idxInstr, oInstr in enumerate(aoInstructions):
        #    assert ((oInstr.fFixedValue ^ fCheckedValue) & fCheckedMask & oInstr.fFixedMask) == 0, \
        #            '%s: fFixedValue=%#x fFixedMask=%#x fCheckedValue=%#x fCheckedMask=%#x -> %#x\naoInstructions: len=%s\n %s' \
        #            % (idxInstr, oInstr.fFixedValue, oInstr.fFixedMask, fCheckedValue, fCheckedMask,
        #               (oInstr.fFixedValue ^ fCheckedValue) & fCheckedMask & oInstr.fFixedMask,
        #               len(aoInstructions),
        #               '\n '.join(['%s%s: %#010x/%#010x %s' % ('*' if i == idxInstr else ' ', i,
        #                                                       oInstr2.fFixedValue, oInstr2.fFixedMask, oInstr2.sName)
        #                           for i, oInstr2 in enumerate(aoInstructions[:idxInstr+8])]));

        self.aoInstructions     = aoInstructions;   ##< The instructions at this level.
        self.fCheckedMask       = fCheckedMask;     ##< The opcode bit mask covered thus far.
        self.fCheckedValue      = fCheckedValue;    ##< The value that goes with fCheckedMask.
        self.fChildMask         = 0;                ##< The mask used to separate the children.
        self.dChildren          = {};               ##< Children, sparsely populated by constructNextLevel().

    @staticmethod
    def popCount(uValue):
        cBits = 0;
        while uValue:
            cBits += 1;
            uValue &= uValue - 1;
        return cBits;

    s_uLogLine = 0;  # pylint: disable=invalid-name
    @staticmethod
    def dprint(uDepth, sMsg):
        msNow = (time.time_ns() - g_nsProgStart) // 1000000;
        print('%u.%03u: %u: debug/%u: %s%s' % (msNow // 1000, msNow % 1000, DecoderNode.s_uLogLine, uDepth, '  ' * uDepth, sMsg));
        DecoderNode.s_uLogLine += 1;

    def constructNextLevel(self, uDepth, uMaxCost): # pylint: disable=too-many-locals,too-many-statements
        """
        Recursively constructs the
        """
        if uDepth == 0:
            for i, oInstr in enumerate(self.aoInstructions):
                self.dprint(uDepth, '%4u: %s' % (i, oInstr.toString(cchName = 32),));

        #
        # Special cases: 4 or fewer entries.
        #
        cInstructions = len(self.aoInstructions)
        if cInstructions <= 4:
            assert not self.dChildren;
            uCost = 0;
            # Special case: 1 instruction - leaf.
            if cInstructions <= 1:
                if self.aoInstructions[0].fFixedMask & ~self.fCheckedMask != 0:
                    self.fChildMask = DecoderNode.kfChildMaskOpcodeValueIf;
                    uCost = 16;                                                         # 16 = kCostOpcodeValueIf
                else:
                    assert self.fChildMask == 0;

            # Special case: 2, 3 or 4 instructions - use a sequence of 'if ((uOpcode & fFixedMask) == fFixedValue)' checks.
            else:
                self.fChildMask = DecoderNode.kfChildMaskMultipleOpcodeValueIfs;
                uCost = 32 * cInstructions * 2;                                         # 32 = kCostMultipleOpcodeValueIfs
            return uCost;

        #
        # The cost of one indirect call is 256, so just bail if we don't have
        # the budget for any of that.
        #
        if uMaxCost <= 256:                                                             # 256 = kCostIndirectCall
            raise DecoderNode.TooExpensive();
        if uDepth > 5:                                                                  #   5 = kMaxDepth
            raise DecoderNode.TooExpensive();

        #
        # Do an inventory of the fixed masks used by the instructions.
        #
        dMaskCounts  = collections.Counter();
        fCheckedMask = self.fCheckedMask;    # (Cache it as a local variable for speed.)
        for oInstr in self.aoInstructions:
            dMaskCounts[oInstr.fFixedMask & ~fCheckedMask] += 1;
        #assert 0 not in dMaskCounts or dMaskCounts[0] <= 1, \
        #        'dMaskCounts=%s cInstructions=%s\n%s' % (dMaskCounts, cInstructions, self.aoInstructions);
        # 0x00011c00 & 0xfffee000  = 0x0 (0)

        #
        # HACK ALERT! For level 0 and 1 we offer ways to insert hints to reduce
        #             the runtime, since it's tedious to wait for 30 min for
        #             each code tweak...  See --decoder-hint0 and --decoder-hint1.
        #
        dSpeedupFilter = None;
        if uDepth <= 1:
            if uDepth == 1:
                dSpeedupFilter = g_ddDecoderFilterDepth1.get('%x/%x' % (self.fCheckedMask, self.fCheckedValue), None);
            elif g_dDecoderFilterDepth0:
                dSpeedupFilter = g_dDecoderFilterDepth0;

        #
        # Whether to bother compiling the mask zip/unzip functions.
        #
        # The goal here is to keep the {built-in method builtins.eval} line far
        # away from top of the profiler stats, while at the same time keeping the
        # __zipMaskN and __unzipMaskN methods from taking up too much time.
        #
        fCompileMaskZip   = cInstructions >= 256;
        fCompileMaskUnzip = cInstructions >= 32; #?

        #
        # Work thru the possible masks and test out the variations (brute force style).
        #
        uCostBest        = uMaxCost;
        cChildrenBits    = 0;
        fChildrenBest    = 0;
        dChildrenBest    = {};

        fMaskNotDoneYet  = 0xffffffff;
        fCheckedValue    = self.fCheckedValue; # (Cache it as a local variable for speed.)
        iOuterLoop       = -1;
        for fOrgMask, cOccurences in dMaskCounts.most_common(3):
            iOuterLoop += 1;

            # Determin the max and min table sizes (in bits) based on the instructions using the mask.
            cMinTableSizeInBits = cOccurences.bit_length() - 1;
            if (1 << cMinTableSizeInBits) < cOccurences:
                cMinTableSizeInBits += 1;
            cMaxTableSizeInBits = self.kacMaxTableSizesInBits[cMinTableSizeInBits]; # Not quite sure about this...
            cMinTableSizeInBits -= 1;

            if uDepth <= 2:
                self.dprint(uDepth,
                            '%s Start/%u: %#010x (%u) - %u/%u instructions - tab size %u-%u; fChecked=%#x/%#x uCostBest=%#x'
                            % (('=' if iOuterLoop == 0 else '-') * 5, iOuterLoop, fOrgMask,
                               self.popCount(fOrgMask), cOccurences, cInstructions, cMinTableSizeInBits, cMaxTableSizeInBits,
                               fCheckedValue, fCheckedMask, uCostBest,));

            # Skip pointless stuff and things we've already covered.
            if cOccurences >= 2 and fOrgMask > 0 and fOrgMask != 0xffffffff and (fOrgMask & fMaskNotDoneYet) != 0:
                #
                # Step 1: Brute force relevant mask variations and pick a few masks.
                #
                # The MaskIterator skips masks that are too wide, too fragmented or
                # already covered.
                #
                # The cost calculation is mainly based on distribution vs table size,
                # trying to favor masks with more target slots.
                #
                dCandidates = {};
                for fMask, cMaskBits, aaiMaskToIdxAlgo in MaskIterator(fOrgMask, cMinTableSizeInBits, cMaxTableSizeInBits,
                                                                       fMaskNotDoneYet):
                    #if uDepth <= 2:
                    #    self.dprint(uDepth, '1>> fMask=%#010x cMaskBits=%s aaiMaskToIdxAlgo=%s...'
                    #                         % (fMask, cMaskBits, aaiMaskToIdxAlgo));
                    #assert cMaskBits <= cMaxTableSizeInBits;

                    # HACK ALERT! Skip the mask if we have a selection filter and it isn't in it.
                    if dSpeedupFilter and fMask not in dSpeedupFilter:
                        continue;

                    # Calculate base cost and check it against uCostBest before continuing.
                    uCostTmp    = 256;                                                  # 256 = kCostIndirectCall
                    uCostTmp   += (len(aaiMaskToIdxAlgo) - 1) * 2;                      #   2 = kCostPerExtraIndexStep
                    if uCostTmp >= uCostBest:
                        #if uDepth <= 2:
                        #    self.dprint(uDepth, '!!! %#010x too expensive #1: %#x vs %#x' % (fMask, uCostTmp, uCostBest));
                        continue;

                    # Compile the indexing/unindexing functions.
                    fnToIndex   = MaskZipper.algoToZipLambda(aaiMaskToIdxAlgo, fMask, fCompileMaskZip);

                    # Insert the instructions into the temporary table.
                    daoTmp = collections.defaultdict(list);
                    for oInstr in self.aoInstructions:
                        idx = fnToIndex(oInstr.fFixedValue, aaiMaskToIdxAlgo);
                        #self.dprint(uDepth, '%#010x -> %#05x %s' % (oInstr.fFixedValue, idx, oInstr.sName));
                        fNonFixedMatches = ~oInstr.fFixedMask & fMask;
                        if fNonFixedMatches == 0:
                            daoTmp[idx].append(oInstr);
                        else:
                            fIdxNonFixedMatches  = fnToIndex(fNonFixedMatches, aaiMaskToIdxAlgo);
                            cBitsNonFixedMatches = fNonFixedMatches.bit_count();
                            if cBitsNonFixedMatches < 8:
                                idxStep         = fIdxNonFixedMatches & (~fIdxNonFixedMatches + 1);
                                idxInstrMask    = fnToIndex(oInstr.fFixedMask, aaiMaskToIdxAlgo);
                                idxInstrLast    = idx | fIdxNonFixedMatches;
                                for idx2 in range(idx, idxInstrLast + idxStep, idxStep):
                                    if (idx2 & idxInstrMask) == idx:
                                        daoTmp[idx2].append(oInstr);
                            else:
                                aaiNonFixedAlgo = MaskZipper.compileAlgo(fIdxNonFixedMatches);
                                fnNonFixedUnzip = MaskZipper.algoToUnzipLambda(aaiNonFixedAlgo, fIdxNonFixedMatches);
                                for idx3 in range(MaskZipper.zipMask(fIdxNonFixedMatches, aaiNonFixedAlgo) + 1):
                                    idx2 = idx | fnNonFixedUnzip(idx3, aaiNonFixedAlgo);
                                    daoTmp[idx2].append(oInstr);

                    ## @todo Account for entires causing instruction duplication...
                    ##       Perhaps by summing up the number of instructions for the next level?
                    cEffTmpSize = len(daoTmp);

                    # Reject anything that ends up putting all the stuff in a single slot.
                    if cEffTmpSize <= 1:
                        #if uDepth <= 2: self.dprint(uDepth, '!!! bad distribution #1: fMask=%#x' % (fMask,));
                        continue;

                    # Add cost for poor average distribution.
                    rdAvgLen = float(cInstructions) / cEffTmpSize;
                    if rdAvgLen > 1.2:
                        uCostTmp += int(rdAvgLen * 8)
                        if uCostTmp >= uCostBest:
                            #if uDepth <= 2:
                            #    self.dprint(uDepth, '!!! %#010x too expensive #2: %#x vs %#x (rdAvgLen=%s)'
                            #                        % (fMask, uCostTmp, uCostBest, rdAvgLen));
                            continue;

                    # Add the cost for unused entries under reasonable table population.
                    cNominalFill = 1 << (cMaskBits - 1); # 50% full or better is great.
                    if cEffTmpSize < cNominalFill:
                        uCostTmp += ((cNominalFill - cEffTmpSize) * 2);                 # 2 = kCostUnusedTabEntry
                        if uCostTmp >= uCostBest:
                            #if uDepth <= 2:
                            #    self.dprint(uDepth, '!!! %#010x too expensive #3: %#x vs %#x' % (fMask, uCostTmp, uCostBest));
                            continue;

                    # Record it as a candidate.
                    dCandidates[uCostTmp] = (fMask, cMaskBits, aaiMaskToIdxAlgo, daoTmp);
                    if len(dCandidates) > 64:
                        dOld = dCandidates;
                        dCandidates = { uKey:dOld[uKey] for uKey in sorted(dCandidates.keys())[:4] };
                        del dOld;

                #
                # Step 2: Process the top 4 candidates.
                #
                for uCostTmp in sorted(dCandidates.keys())[:4]:
                    fMask, cMaskBits, aaiMaskToIdxAlgo, daoTmp = dCandidates[uCostTmp];

                    #if uDepth <= 2:
                    #    self.dprint(uDepth, '2>> fMask=%#010x cMaskBits=%s aaiMaskToIdxAlgo=%s #daoTmp=%s...'
                    #                         % (fMask, cMaskBits, aaiMaskToIdxAlgo, len(daoTmp),));
                    #assert cMaskBits <= cMaxTableSizeInBits;

                    # Construct decoder nodes from the aaoTmp lists, construct sub-levels and calculate costs.
                    fnFromIndex  = MaskZipper.algoToUnzipLambda(aaiMaskToIdxAlgo, fMask, fCompileMaskUnzip);
                    dChildrenTmp = {};
                    try:
                        for idx, aoInstrs in daoTmp.items():
                            oChild = DecoderNode(aoInstrs,
                                                 fCheckedMask  | fMask,
                                                 fCheckedValue | fnFromIndex(idx, aaiMaskToIdxAlgo));
                            dChildrenTmp[idx] = oChild;
                            uCostTmp += oChild.constructNextLevel(uDepth + 1, uCostBest - uCostTmp);
                            if uCostTmp >= uCostBest:
                                break;
                    except DecoderNode.TooExpensive:
                        #if uDepth <= 2:
                        #    self.dprint(uDepth, '!!! %#010x too expensive #4: %#x+child vs %#x' % (fMask, uCostTmp, uCostBest));
                        continue;

                    # Is this mask better than the previous?
                    if uCostTmp < uCostBest:
                        if uDepth <= 2:
                            self.dprint(uDepth,
                                        '+++ %s best!  %#010x (%u) uCost=%#x; %u ins in %u slots (previous %#010x / %#x) ...'
                                        % ('New' if cChildrenBits else '1st', fMask, cMaskBits, uCostTmp,
                                           cInstructions, len(dChildrenTmp), fChildrenBest, uCostBest, ));
                        uCostBest      = uCostTmp;
                        cChildrenBits  = cMaskBits;
                        fChildrenBest  = fMask;
                        dChildrenBest  = dChildrenTmp;
                    #elif uDepth <= 2:
                    #    self.dprint(uDepth, '!!! %#010x too expensive #5: %#x vs %#x' % (fMask, uCostTmp, uCostBest));

                # Note that we've covered all the permutations in the given mask.
                fMaskNotDoneYet &= ~fOrgMask;

        # Drop it if too expensive.
        if uCostBest >= uMaxCost:
            raise DecoderNode.TooExpensive();

        if dChildrenBest is None:
            print('warning! No solution! depth=%u #Instruction=%u' % (uDepth, cInstructions));
            raise Exception('fixme')

        #assert fChildrenBest.bit_count() == cChildrenBits;
        #assert len(dChildrenBest) <= (1 << cChildrenBits)
        if uDepth <= 2:
            self.dprint(uDepth,
                        '===== Final: fMask=%#010x (%u) uCost=%#x #Instructions=%u in %u slots over %u entries...'
                        % (fChildrenBest, cChildrenBits, uCostBest, cInstructions, len(dChildrenBest), 1 << cChildrenBits));

        # Done.
        self.fChildMask = fChildrenBest;
        self.dChildren  = dChildrenBest;

        return uCostBest;

    def setInstrProps(self, uDepth):
        """
        Sets the fDecoderLeafCheckNeeded instruction property.
        """
        if not self.dChildren:
            assert len(self.aoInstructions) != 1 or self.fChildMask in (0, DecoderNode.kfChildMaskOpcodeValueIf);
            assert len(self.aoInstructions) == 1 or self.fChildMask == DecoderNode.kfChildMaskMultipleOpcodeValueIfs;
            for oInstr in self.aoInstructions:
                oInstr.fDecoderLeafCheckNeeded = self.fChildMask == DecoderNode.kfChildMaskOpcodeValueIf;
        else:
            for oChildNode in self.dChildren.values():
                oChildNode.setInstrProps(uDepth + 1);

    def getFuncName(self, sInstrSet, uDepth):
        """
        Returns the function name at the specific depth.
        """
        if not sInstrSet:
            sInstrSet = self.aoInstructions[0].getInstrSetName()
        if self.dChildren or len(self.aoInstructions) > 1:
            return 'iemDecode%s_%08x_%08x_%u' % (sInstrSet, self.fCheckedMask, self.fCheckedValue, uDepth,);
        return 'iemDecode%s_%s' % (sInstrSet, self.aoInstructions[0].getCName(),);


#
# System Register Code Morphing.
#

class VBoxAstCppField(ArmAstCppExpr):
    """ For converted ArmAstField so we optimize them later when processing
    an ArmAstBinaryOp or ArmAstUnaryOp parent node. """
    def __init__(self, sExpr, cBitsWidth, sNoShiftExpr = None, cBitsWidthNoShift = None):
        ArmAstCppExpr.__init__(self, sExpr, cBitsWidth);
        self.sNoShiftExpr      = sNoShiftExpr;
        self.cBitsWidthNoShift = cBitsWidthNoShift;

    def dropShift(self):
        if self.sNoShiftExpr:
            self.sExpr      = self.sNoShiftExpr;
            self.cBitsWidth = self.cBitsWidthNoShift;
        return self;

    def dropExtraParenthesis(self):
        if self.sExpr[0] == '(' and self.sExpr[-1] == ')':
            self.sExpr = self.sExpr[1:-1];
        return self;

    @staticmethod
    def fromFieldDetailsAndAccessExprOrInt(iFirstBit, cBitsWidth, sName, oAccessExprOrInt):
        """
        Returns an instance that accesses the given field.
        The oAccessExprOrInt parameter is either a string or integer. If it
        is an integer an ArmAstInteger instance is return instead.
        """
        if isinstance(oAccessExprOrInt, int):
            iValue  = oAccessExprOrInt >> iFirstBit;
            iValue &= (1 << cBitsWidth) - 1;
            return ArmAstInteger(iValue, cBitsWidth); ## @todo annotate

        if cBitsWidth == 1:
            if iFirstBit == 0:
                return VBoxAstCppField('(%s & UINT32_C(1)/*%s*/)' % (oAccessExprOrInt, sName,), 1);
            sExpr        = '((%s >> %2d) & UINT32_C(1)/*%s*/)' % (oAccessExprOrInt, iFirstBit, sName,);
            sNoShiftExpr = '(%s & RT_BIT_%u(%d)/*%s*/)' % (oAccessExprOrInt, 64 if iFirstBit >= 32 else 32, iFirstBit, sName,);
        else:
            if iFirstBit == 0:
                return VBoxAstCppField('(%s & UINT%u_C(%#x)/*%s*/)' % (oAccessExprOrInt, 64 if cBitsWidth > 32 else 32,
                                                                       ((1 << cBitsWidth) - 1) << iFirstBit, sName,),
                                       cBitsWidth);
            sExpr        = '((%s >> %2d) & UINT%u_C(%#x)/*%s*/)' \
                         % (oAccessExprOrInt, iFirstBit, 64 if cBitsWidth >= 32 else 32, ((1 << cBitsWidth) - 1), sName,);
            sNoShiftExpr = '(%s & UINT%u_C(%#x)/*%s*/)' \
                         % (oAccessExprOrInt, 64 if iFirstBit + cBitsWidth >= 32 else 32,
                            ((1 << cBitsWidth) - 1) << iFirstBit, sName,);
        return VBoxAstCppField(sExpr, cBitsWidth, sNoShiftExpr, cBitsWidth + iFirstBit);


class VBoxAstCppCast(ArmAstBase, ArmAstCppExprBase):
    """ C++ AST cast node. """
    def __init__(self, oExpr, cTargetBits, fSigned = False, sComment = None):
        ArmAstBase.__init__(self, 'c++-cast', sComment);
        ArmAstCppExprBase.__init__(self);
        self.oExpr       = oExpr;
        self.cTargetBits = cTargetBits;
        self.fSigned     = fSigned;

    def clone(self):
        return VBoxAstCppCast(self.oExpr, self.cTargetBits, self.fSigned, self.sComment);

    def isSame(self, oOther):
        if isinstance(oOther, VBoxAstCppCast):
            if self.cTargetBits == oOther.cTargetBits:
                if self.fSigned == oOther.fSigned:
                    return self.oExpr.isSame(oOther.oExpr);
        return False;

    def walk(self, fnCallback, oCallbackArg = None, fDepthFirst = True):
        return self._walker(fnCallback, oCallbackArg, fDepthFirst, self.oExpr);

    def transform(self, fnCallback, fEliminationAllowed, oCallbackArg, aoStack):
        oRet  = None;
        oExpr = self.oExpr.transform(fnCallback, fEliminationAllowed, oCallbackArg, aoStack);
        if oExpr:
            self.oExpr = oExpr;
            oRet = fnCallback(self, fEliminationAllowed, oCallbackArg, aoStack);
        return oRet;

    def toStringEx(self, sLang = None, cchMaxWidth = 120):
        sRet  = '(%sint%u_t)' % ('' if self.fSigned else 'u', self.cTargetBits,);
        cch   = len(sRet);
        sExpr = self.oExpr.toStringEx(sLang, max(cchMaxWidth - cch - 2, 60));
        fNeedParenthesis = sExpr[0] != '(' or sExpr[-1] != ')';
        if not fNeedParenthesis:
            cDepth = 1;
            off    = 1;
            offEnd = len(sExpr) - 1;
            while off < offEnd:
                if sExpr[off] == '(':
                    cDepth += 1;
                elif sExpr[off] == ')':
                    cDepth -= 1;
                    if cDepth <= 0:
                        fNeedParenthesis = False;
                        break;
                off += 1;
        if fNeedParenthesis:
            sRet = sRet + '(' + sExpr + ')';
            cch += 1;
        else:
            sRet = sRet + sExpr;
        if '\n' in sRet:
            sRet = sRet.replace('\n', '\n' + ' ' * cch);
        if self.sComment:
            sRet = self.formatInExprComment(sRet, self.sComment, sLang, cchMaxWidth);
        return sRet;

    def toCExpr(self, oHelper):
        _ = oHelper;
        return self.toStringEx(sLang = 'C');

    def getWidth(self, oHelper = None):
        _ = oHelper;
        return self.cTargetBits;



class SysRegAccessorInfo(object):
    """ Info about an accessor we're emitting code for. """
    def __init__(self, oAccessor, oReg, sEnc, sAsmValue = None):
        self.oAccessor  = oAccessor     # type: ArmAccessorSystem
        self.oReg       = oReg;
        self.sEnc       = sEnc;
        self.oCode      = None;
        self.sAsmValue  = sAsmValue if sAsmValue else oAccessor.oEncoding.sAsmValue;
        self.sRegName   = oReg.sName;
        # Stats for the code generator.
        self.cGstFeatsRefs                  = 0;
        self.cInstrEssenceRefs              = 0;  # References to uInstrEssence (idSysReg + idxGprReg + direction)
        self.cAssignmentsWithUnresolvedRegs = 0;  # Number of unresolved system register assignment reference.
        ## This is updated by morphCodeToC() and will be zero the code is ready for compilation.
        self.cIncompleteNodes             = -1;

    def toString(self):
        return 'SysRegAccessorInfo<sAsmValue=%s, sEnc=%s, sRegName=%s>' % (self.sAsmValue, self.sEnc, self.sRegName,);

    def __str__(self):
        return self.toString();

    def __repr__(self):
        return self.toString();


class SysRegGeneratorBase(object):
    """ Base class for the system register access code generators. """

    kdELxToNum = { 'EL0': 0, 'EL1': 1, 'EL2': 2, 'EL3': 3, }

    ## Register names that requires post modification state updates.
    kdRegsRequiringRecalcs = {
        'SCR_EL1':   'iemCImplHlpRecalcFlags',
        'SCR_EL2':   'iemCImplHlpRecalcFlags',
        'SCR_EL3':   'iemCImplHlpRecalcFlags',
        'SCTLR_EL1': 'iemCImplHlpRecalcFlagsAndPgmModeEl1',
        'SCTLR_EL2': 'iemCImplHlpRecalcFlagsAndPgmModeEl2',
        'SCTLR_EL3': 'iemCImplHlpRecalcFlagsAndPgmModeEl3',
        'HCR_EL2':   'iemCImplHlpRecalcFlags',
        'HCRX_EL2':  'iemCImplHlpRecalcFlags',
        'TCR_EL1':   'iemCImplHlpRecalcFlagsAndPgmModeEl1',
        'TCR_EL2':   'iemCImplHlpRecalcFlagsAndPgmModeEl2',
        'TCR_EL3':   'iemCImplHlpRecalcFlagsAndPgmModeEl3',
        'MDSCR_EL1': 'iemCImplHlpRecalcFlags',
    };

    ## Mapping of self.sType.
    kdTypeToGprSuff   = { 'read': 'Dst',         'write': 'Src' };
    kdTypeToGprDesc   = { 'read': 'destination', 'write': 'source' };
    kdTypeToParamDesc = {
        'read':     'Where to return the value.',
        'write':    'The value to write.'
    };

    def __init__(self, sInstr, sFuncPrefix, sParamType, sParamName, sType = 'read'):
        assert sType in {'write','read'}
        self.sInstr         = sInstr;
        self.sFuncPrefix    = sFuncPrefix;
        self.sParamType     = sParamType;
        self.sParamName     = sParamName;
        self.fIs128Bit      = sParamType.find('128') >= 0; ## @todo we can do better than this...
        self.sType          = sType;
        self.aoInfo         = []        # type: List[SysRegAccessorInfo]
        self.cComplete      = 0;
        self.cIncomplete    = 0;
        self.cUnresolvedRegs = 0;       # Number of info items with unresolved registers.
        self.dWarnings      = {};

    def isCppIdentifierInAst(self, oNode, sCppExpr):
        """ Checks if the given C++ identifier s part of the given AST. """
        dResult = {'ret': False,};
        def callback(oNode, dResult):
            if isinstance(oNode, ArmAstCppExpr):
                off = oNode.sExpr.find(sCppExpr);
                if (   off == 0
                    or (off > 0 and not oNode.sExpr[off - 1].isalnum() and oNode.sExpr[off - 1] != '_')):
                    offEnd = off + len(sCppExpr);
                    if (    offEnd == len(oNode.sExpr)
                        or (not oNode.sExpr[offEnd].isalnum() and oNode.sExpr[offEnd] != '_')):
                        dResult['ret'] = True;
        oNode.walk(callback, dResult);
        return dResult['ret'];

    def getDescriptionFromInfo(self, oInfo):
        return '%s - %s' % (oInfo.sAsmValue, oInfo.oAccessor.oEncoding.dNamedValues,);

    def generateOneHandler(self, oInfo):
        """ Generates one register access handler. """
        asLines = [
            '',
            '/**',
            ' * %s' % (self.getDescriptionFromInfo(oInfo),),
            ' * Transformation status: %u - %s'
            % (oInfo.cIncompleteNodes, 'complete' if oInfo.cIncompleteNodes == 0 else 'incomplete'),
            ' */',
            'static VBOXSTRICTRC %s_%s(PVMCPU pVCpu, %s %s%s) RT_NOEXCEPT'
            % (self.sFuncPrefix, oInfo.sAsmValue, self.sParamType, self.sParamName,
               ', uint32_t uInstrEssence' if oInfo.cInstrEssenceRefs > 0 else ''),
            '{',
        ];
        if oInfo.cIncompleteNodes > 0 or not oInfo.oCode:
            asLines += [
                '    RT_NOREF(pVCpu, %s%s);' % (self.sParamName, ', uInstrEssence' if oInfo.cInstrEssenceRefs > 0 else ''),
                '#ifdef IEM_SYSREG_TODO',
            ];
        elif not self.isCppIdentifierInAst(oInfo.oCode, self.sParamName):
            asLines.append('    RT_NOREF(%s);' % (self.sParamName,) );
        if oInfo.cGstFeatsRefs > 0:
            asLines.append('    const CPUMFEATURESARMV8 * const pGstFeats = IEM_GET_GUEST_CPU_FEATURES(pVCpu);');

        if oInfo.oCode:
            asLines.extend(oInfo.oCode.toStringList('    ', 'C'));

        if oInfo.cIncompleteNodes > 0 or not oInfo.oCode:
            asLines += [
                '#endif',
                '    return VERR_IEM_ASPECT_NOT_IMPLEMENTED;'
            ];

        asLines.append('    /* -------- Original code specification: -------- */');
        asLines.extend([ '    // ' + sLine for sLine in oInfo.oAccessor.oAccess.toStringList()]);

        asLines += [
            '}',
        ];
        return asLines;

    def generateMainFunctionBody(self):
        """ Generates the main CIMPL function body. """
        if self.sType == 'read':
            return [
                '    uint64_t         uZeroDummy;',
                '    uint64_t * const puDst = idxGprDst < ARMV8_A64_REG_XZR',
                '                           ? &pVCpu->cpum.GstCtx.aGRegs[idxGprDst].x : &uZeroDummy;',
                '    return %s_generic(pVCpu, idSysReg, idxGprDst, puDst);' % (self.sFuncPrefix,),
            ];

        return [
            '    uint64_t const uValue = idxGprSrc < ARMV8_A64_REG_XZR ? pVCpu->cpum.GstCtx.aGRegs[idxGprSrc].x : 0;',
            '    return %s_generic(pVCpu, idSysReg, idxGprSrc, uValue);' % (self.sFuncPrefix,),
        ];

    def generateMainFunction(self):
        asLines = [
            '',
            '/**',
            ' * Implements the %s instruction (generic).' % (self.sInstr),
            ' *',
            ' * @returns Strict VBox status code.',
            ' * @param   pVCpu       The cross context virtual CPU structure of the',
            ' *                      calling thread.',
            ' * @param   idSysReg    The system register to %s (IPRT format).' % (self.sType,),
            ' * @param   idxGpr%s   The %s GPR register number (for exceptions).'
            % (self.kdTypeToGprSuff[self.sType], self.kdTypeToGprDesc[self.sType],),
            ' * @param   %-11s %s' % (self.sParamName, self.kdTypeToParamDesc[self.sType],),
            ' */',
            'DECLHIDDEN(VBOXSTRICTRC) %s_generic(PVMCPU pVCpu, uint32_t idSysReg, uint32_t idxGpr%s, %s %s)'
            % (self.sFuncPrefix, self.kdTypeToGprSuff[self.sType],self.sParamType, self.sParamName,),
            '{',
        ];

        sLogFmt = '%.16Rhxs' if self.fIs128Bit else '%#RX64';
        if self.sType == 'read':
            asLines.append('    Log(("%s(%%#x)\\n", idSysReg));' % (self.sFuncPrefix, ));
        else:
            asLines.append('    Log(("%s(%%#x, %s)\\n", idSysReg, %s));'
                           % (self.sFuncPrefix, sLogFmt, self.sParamName,));

        for oInfo in self.aoInfo:
            if oInfo.cInstrEssenceRefs:
                asLines += [
                    '    uint32_t const uInstrEssence = IEM_CIMPL_SYSREG_INSTR_ESSENCE_MAKE(idSysReg, idxGpr%s, 0, 0xf);'
                    % (self.kdTypeToGprSuff[self.sType],),
                ];
                break;
        asLines += [
            '    switch (idSysReg)',
            '    {',
        ];
        for oInfo in self.aoInfo:
            if oInfo.sEnc[0] == 'A': ## @todo better filtering out of non A64/whatever stuff.
                asLines += [
                    '        case %s:' % (oInfo.sEnc,),
                    '        {',
                    '            VBOXSTRICTRC const rcStrict = %s_%s(pVCpu, %s%s);'
                    % (self.sFuncPrefix, oInfo.sAsmValue, self.sParamName,
                       ', uInstrEssence' if oInfo.cInstrEssenceRefs else '',),
                ];
                if self.sType == 'read':
                    asLines.append('            LogFlow(("%s_%s -> %%Rrc & *%s=%s\\n", VBOXSTRICTRC_VAL(rcStrict), *%s));'
                                   % (self.sFuncPrefix, oInfo.sAsmValue, self.sParamName, sLogFmt, self.sParamName,));
                else:
                    asLines.append('            LogFlow(("%s_%s(%s) -> %%Rrc\\n", %s, VBOXSTRICTRC_VAL(rcStrict)));'
                                   % (self.sFuncPrefix, oInfo.sAsmValue, sLogFmt, self.sParamName,));
                if oInfo.sRegName in self.kdRegsRequiringRecalcs:
                    asLines.append('            return %s(pVCpu, rcStrict);' % (self.kdRegsRequiringRecalcs[oInfo.sRegName],));
                else:
                    asLines.append('            return rcStrict;');
                asLines.append('        }');
        asLines += [
            '    }',
            '    /* Fall back on handcoded handler. */',
            '    return %s_fallback(pVCpu, idSysReg, idxGpr%s, %s);'
            % (self.sFuncPrefix, self.kdTypeToGprSuff[self.sType], self.sParamName,),
            '}',
            '',
            '/**',
            ' * Implements the %s instruction.' % (self.sInstr,),
            ' *',
            ' * @param   idSysReg    The system register to %s (IPRT format).' % (self.sType,),
            ' * @param   idxGpr%s   The %s GPR register number (for exceptions).'
            % (self.kdTypeToGprSuff[self.sType], self.kdTypeToGprDesc[self.sType],),
            ' */',
            'IEM_CIMPL_DEF_2(%s, uint32_t, idSysReg, uint32_t, idxGpr%s)' % (self.sFuncPrefix, self.kdTypeToGprSuff[self.sType],),
            '{',
        ]
        asLines += self.generateMainFunctionBody();
        asLines += [
            '}',
        ];
        return asLines;

    def isA64Instruction(self):
        return self.sInstr.startswith('A64');

    def getRegStateName(self):
        return 'AArch64' if self.isA64Instruction() else 'AArch32';

    def warn(self, sMsg, oRetValue = None):
        if sMsg not in self.dWarnings:
            self.dWarnings[sMsg]  = 1;
            #print('warning: %s' % (sMsg,)); # delay
        else:
            self.dWarnings[sMsg] += 1;
        return oRetValue;

    def warnXcpt(self, oXcpt, oRetValue):
        return self.warn(str(oXcpt), oRetValue);

    def printWarnings(self):
        """ Prints the warnings and returns a list of them for insertion into the C++ file. """
        asWarnings = [];
        for sMsg, cOccurences in sorted(self.dWarnings.items()):
            if cOccurences > 1:
                asWarnings.append('warning: %s (%u times)' % (sMsg, cOccurences,));
            else:
                asWarnings.append('warning: %s' % (sMsg,));
        if asWarnings:
            sys.stderr.write('\n'.join(asWarnings) + '\n');
        return asWarnings;

    def checkCConversionCallback(self, oNode, oInfo):
        """ Walker callback used by morphCodeToC() to look for nodes that aren't suitable for C. """
        if isinstance(oNode, ArmAstBinaryOp):
            if ArmAstBinaryOp.kdOps[oNode.sOp] in (ArmAstBinaryOp.ksOpTypeCompare, ArmAstBinaryOp.ksOpTypeLogical,
                                                   ArmAstBinaryOp.ksOpTypeArithmetical, ArmAstBinaryOp.ksOpTypeBitwise):
                return True;
        elif isinstance(oNode, ArmAstReturn):
            if oNode.oValue:
                return True;
        elif isinstance(oNode, ArmAstCppExpr):
            if oNode.sExpr == 'uInstrEssence':
                oInfo.cInstrEssenceRefs += 1;
            elif oNode.sExpr.startswith('pGstFeats'):
                oInfo.cGstFeatsRefs     += 1;
            return True;

        elif isinstance(oNode, (ArmAstBool, ArmAstCppExprBase, ArmAstIfList, ArmAstInteger, ArmAstUnaryOp,
                                ArmAstAssignment, ArmAstStatementList, self.VBoxAstCppConcat)):
            return True;

        oInfo.cIncompleteNodes += 1;
        if isinstance(oNode, ArmAstFunction):
            return self.warn('Call to untranslated function: %s' % (oNode.sName,), True);
        return True;

    def checkCConversion(self, oInfo, oNode):
        """
        Update the completion status and other statistics.
        Returns True if completely converted, False if not.

        Note! Can be used on sub-trees by supplying a root node other than oInfo.oCode.
        """
        oInfo.cIncompleteNodes  = 0;
        oInfo.cGstFeatsRefs     = 0;
        oInfo.cInstrEssenceRefs = 0;
        oNode.walk(self.checkCConversionCallback, oInfo);
        return oInfo.cIncompleteNodes == 0;

    def morphCodeToC(self, oInfo):
        """ Morphs the accessor code and assigns the result to self.oCode """
        assert oInfo.oCode is None;
        oInfo.oCode = self.transformCodePass0(oInfo, oInfo.oAccessor.oAccess.clone());
        oInfo.oCode = self.transformCodePass1(oInfo, oInfo.oCode);
        oInfo.oCode = self.transformCodePass2(oInfo, oInfo.oCode);

        # Add a return statement if necessary.
        if not oInfo.oCode.doAllPathsReturn():
            oInfo.oCode = ArmAstStatementList([
                oInfo.oCode,
                ArmAstReturn(ArmAstCppExpr('iemRegPc%sIncAndFinishingClearingFlags(pVCpu, VINF_SUCCESS)'
                                           % ('A64' if self.isA64Instruction() else 'A32',), 32)),
            ]);

        # Update the completion status and other statistics.
        self.checkCConversion(oInfo, oInfo.oCode);
        return True;


    #
    # Pass 0 - Resolve troublesome stuff before expand and eliminate anything.
    #

    def transformCodePass0Callback(self, oNode, fEliminationAllowed, oInfo, aoStack):
        """
        Callback for pass 0: Disambiguation of CNTHCTL_EL2.EL1PCTEN and others.
        """

        #
        # CNTHCTL_EL2.EL1PCTEN which can either be bit 0 or bit 10 depending on
        # whether ELIsInHost(EL2).  The specification of CNTPCT_EL0 __seems__ to
        # be checking the context and then referencing a field, probably
        # expecting the field to be picked depending on the context.  Sigh.
        #   - At EL0: EL2Enabled()    && !ELIsInHost(EL2)           && AArch64.CNTHCTL_EL2.EL1PCTEN == '0'
        #   - At EL0: ELIsInHost(EL2) && AArch64.HCR_EL2.TGE == '0' && AArch64.CNTHCTL_EL2.EL1PCTEN == '0'
        #   - At EL1: EL2Enabled()                                  && AArch64.CNTHCTL_EL2.EL1PCTEN == '0'
        #
        if oNode.isMatchingField('EL1PCTEN', 'CNTHCTL_EL2', 'AArch64'):
            # The first stack level is usually where we find the list of 'PSTATE.EL == ELx'
            # conditions.  Locate the relevant one and get our EL value.
            idxEl = -1;
            for oStackNode in aoStack:
                if isinstance(oStackNode, ArmAstIfList):
                    for idxIfStmt, oIfStmt in enumerate(oStackNode.aoIfStatements):
                        if oIfStmt.containsNode(oNode):
                            oIfCond = oStackNode.aoIfConditions[idxIfStmt];
                            if (    isinstance(oIfCond, ArmAstBinaryOp)
                                and oIfCond.sOp == '=='
                                and oIfCond.oLeft.isMatchingDotAtom('PSTATE', 'EL')):
                                idxEl = self.kdELxToNum.get(oIfCond.oRight.getIdentifierName(), -1);
                                if idxEl < 0: raise Exception('Unexpected EL value: %s' % (oIfCond.toString(),));
                                break;
            if idxEl < 0: raise Exception('Unable to determine EL for "%s" in %s' % (oNode.toString(), aoStack,));

            # ELIsInHost(ELx) cannot be true at EL1 or at EL3, so the field is at bit 0.
            if idxEl in (1, 3):
                oNode.sField = 'EL1PCTEN_AT_00';
                return oNode;

            # At EL0 and EL2 we have to look for ELIsInHost in sibiling BinaryOp
            # node.  Go up the stack till we encounter the if-list node and locate
            # the condition oNode belongs to (tedious).
            oIfListNode = None;
            for oStack in reversed(aoStack):
                if isinstance(oStack, ArmAstIfList):
                    oIfListNode = oStack;
                    break;
            if not oIfListNode:
                raise Exception('No if-list node on stack: %s' % (aoStack,));

            oIfCond = None;
            for oCurIfCond in oIfListNode.aoIfConditions:
                if oCurIfCond.containsNode(oNode):
                    oIfCond = oCurIfCond;
                    break;
            if not oIfCond:
                raise Exception('Node "%s" not found in any of the if-list conditions: %s'
                                % (oNode.toString(), oIfListNode.toString(),));

            # This is a bit weird, but it's easier to do the rest of the work
            # on a string representation of the expression.
            sIfCond = oIfCond.toStringEx(sLang = 'C', cchMaxWidth = 999999);
            if sIfCond.find('||') >= 0:
                raise Exception('Unexpectedly complicated condition: %s' % (sIfCond,));
            offInHost = sIfCond.find('ELIsInHost(EL');
            if offInHost < 0 or sIfCond[offInHost + 13] not in ('2', '0'):
                raise Exception('Unable to resolve ambigious CNTHCTL_EL2.EL1PCTEN access: %s' % (sIfCond,));

            if offInHost == 0 or sIfCond[offInHost - 1] == ' ':
                oNode.sField = 'EL1PCTEN_AT_10';
                return oNode;
            if sIfCond[offInHost - 1] == '!':
                oNode.sField = 'EL1PCTEN_AT_00';
                return oNode;

            _ = fEliminationAllowed; _ = oInfo; _ = aoStack;
            raise Exception('Unable to resolve ambigious CNTHCTL_EL2.EL1PCTEN access: %s' % (sIfCond,));

        return oNode;

    def transformCodePass0(self, oInfo, oCode):
        """ Code transformation pass 1: Field disambiguation. """
        return oCode.transform(self.transformCodePass0Callback, True, oInfo, []);


    #
    # Pass 1 - Expanding calls & code elimination.
    #

    kdAstForFunctionsWithoutArguments = {
        'EL2Enabled': # HaveEL(EL2) && (!Have(EL3) || SCR_curr[].NS || IsSecureEL2Enabled())
        ArmAstBinaryOp(ArmAstFunction('HaveEL', [ArmAstIdentifier('EL2'),]),
                                     '&&',
                                     ArmAstBinaryOp.orListToTree([
            ArmAstUnaryOp('!', ArmAstFunction('HaveEL', [ArmAstIdentifier('EL3'),])),
            ArmAstDotAtom([ArmAstIdentifier('SCR_curr'), ArmAstIdentifier('NS')]), # SCR_curr is SCR or SCR_EL3.
            ArmAstFunction('IsSecureEL2Enabled', []),
        ])),
        'IsHCRXEL2Enabled': # IsFeatureImplemented(FEAT_HCX) && (!HaveEL(EL3) || SCR_EL3.HXEn != 0) && EL2Enabled()
            ArmAstBinaryOp.andListToTree([
                ArmAstFunction('IsFeatureImplemented', [ArmAstIdentifier('FEAT_HCX')]),
                ArmAstBinaryOp(ArmAstUnaryOp('!', ArmAstFunction('HaveEL', [ArmAstIdentifier('EL3')])),
                               '||',
                               ArmAstBinaryOp(ArmAstField('HXEn', 'SCR_EL3'), '!=', ArmAstValue('\'0\''))),
                ArmAstFunction('EL2Enabled', []),
        ]),
    };

    def transformCodePass1_HaveAArch64(self):
        """ Pass 1: The HaveAArch64() must be true if we're in a A64 instruction handler. """
        if self.isA64Instruction():
            return ArmAstBool(True);
        return ArmAstFunction('IsFeatureImplemented', [ArmAstIdentifier('FEAT_AA64')]);

    def transformCodePass1_HaveEL(self, oNode):
        """ Pass 1: HaveEL(ELx) - Translate it into the corresponding feature checks. """
        if len(oNode.aoArgs) == 1:
            if oNode.aoArgs[0].isMatchingIdentifier('EL3'):
                return ArmAstBool(False); # EL3 is not implemented.
                #return ArmAstFunction('IsFeatureImplemented', [ArmAstIdentifier('FEAT_EL3')]); ## @todo EL3
            if oNode.aoArgs[0].isMatchingIdentifier('EL2'):
                return ArmAstFunction('IsFeatureImplemented', [ArmAstIdentifier('FEAT_EL2')]);
            if oNode.aoArgs[0].isMatchingIdentifier('EL1') or oNode.aoArgs[0].isMatchingIdentifier('EL0'):
                return ArmAstBool(True); # EL0 and EL1 are mandatory.
        raise Exception('Unexpected HaveEL call: %s' % (oNode.toString(),));

    def transformCodePass1_HaveAArch32EL(self, oNode):
        """ Pass 1: HaveAArch32EL(ELx) - Translate into corresponding feature checks. """
        if len(oNode.aoArgs) == 1:
            if oNode.aoArgs[0].isMatchingIdentifier('EL3'):
                return ArmAstBool(False); # EL3 is not implemented.
                #return ArmAstFunction('IsFeatureImplemented', [ArmAstIdentifier('FEAT_AA32EL3')]); ## @todo EL3
            if oNode.aoArgs[0].isMatchingIdentifier('EL2'):
                return ArmAstFunction('IsFeatureImplemented', [ArmAstIdentifier('FEAT_AA32EL2')]);
            if oNode.aoArgs[0].isMatchingIdentifier('EL1'):
                return ArmAstFunction('IsFeatureImplemented', [ArmAstIdentifier('FEAT_AA32EL1')]);
            if oNode.aoArgs[0].isMatchingIdentifier('EL0'):
                return ArmAstFunction('IsFeatureImplemented', [ArmAstIdentifier('FEAT_AA32EL0')]);
        raise Exception('Unexpected HaveAArch32EL call: %s' % (oNode.toString(),));

    def transformCodePass1_ELIsInHost(self, oNode):
        """ Pass 1: ELIsInHost(ELx) - Translate this into appropriate AST. """
        if oNode.aoArgs[0].isMatchingIdentifier('EL0') or oNode.aoArgs[0].isMatchingIdentifier('EL2'):
            ## @todo skipping the LEUsingAArch32(EL2) check
            oCommon = ArmAstBinaryOp.andListToTree([
                ArmAstFunction('IsFeatureImplemented', [ArmAstIdentifier('FEAT_VHE')]),
                ArmAstFunction('EL2Enabled', []),
                ArmAstBinaryOp(ArmAstUnaryOp('!', ArmAstFunction('IsFeatureImplemented', [ArmAstIdentifier('FEAT_E2H0')])),
                               '||', ArmAstBinaryOp(ArmAstField('E2H', 'HCR_EL2'), '!=', ArmAstInteger(0))),
            ]);
            if oNode.aoArgs[0].isMatchingIdentifier('EL0'):
                return ArmAstBinaryOp(oCommon, '&&', ArmAstBinaryOp(ArmAstField('TGE', 'HCR_EL2'), '!=', ArmAstInteger(0)));
            return oCommon;
        if oNode.aoArgs[0].isMatchingIdentifier('EL3') or oNode.aoArgs[0].isMatchingIdentifier('EL1'):
            return ArmAstBool(False); # Only EL0 and EL2 can be in hosted. mode.
        raise Exception('Unexpected ELIsInHost call: %s' % (oNode.toString(),));

    def transformCodePass1_IsFeatureImplemented(self, oNode):
        """ Pass 1: IsFeatureImplemented - replace with False for features we just don't implement. """
        if len(oNode.aoArgs) != 1 or not isinstance(oNode.aoArgs[0], ArmAstIdentifier):
            raise Exception('Unexpected: %s' % (oNode.toString(),));
        ## @todo EL3
        if oNode.aoArgs[0].sName in ('FEAT_AA64EL3', 'FEAT_AA32EL3', 'FEAT_EL3', 'FEAT_RME',
                                     'FEAT_MEC', # This feature implies FEAT_RME, which we don't implement because of no EL3.
                                     'FEAT_SEL2', 'FEAT_Secure', # No secure state w/o EL3...
                                     'FEAT_FGT2', # We don't have any of the associated registers like HDFGRTR2 yet.
                                     ):
            return ArmAstBool(False);

        # Some stuff that is true (probably unused):
        if oNode.aoArgs[0].sName in ('FEAT_AA64',) and self.isA64Instruction():
            return ArmAstBool(True);
        if oNode.aoArgs[0].sName in ('FEAT_EL0', 'FEAT_EL1',):
            return ArmAstBool(True);
        return oNode;

    def transformCodePass1_IsCurrentSecurityState(self, oNode):
        """ Pass 1: IsCurrentSecurityState(SS_Realm), IsCurrentSecurityState(SS_Secure). """
        if len(oNode.aoArgs) == 1 and isinstance(oNode.aoArgs[0], ArmAstIdentifier):
            if oNode.aoArgs[0].sName == 'SS_Realm':
                # Realm requires EL3. ## @todo EL3
                return ArmAstBool(False);
            if oNode.aoArgs[0].sName == 'SS_Secure':
                # SS_Secure requires EL3, unless SecureOnlyImplemeation() is true. ## @todo EL3
                return ArmAstBool(False);
            if oNode.aoArgs[0].sName == 'SS_NonSecure':
                # Without EL3, everything is non-secure.
                return ArmAstBool(True);
        raise Exception('Unexpected: %s' % (oNode.toString(),));

    def transformCodePass1_IsHighestEL(self, oNode):
        """ Pass 1: IsHighestEL(ELx). """
        ## @todo EL3
        if len(oNode.aoArgs) == 1:
            if isinstance(oNode.aoArgs[0], ArmAstIdentifier):
                # EL1 is the higest if we don't have EL2.
                if oNode.aoArgs[0].sName == 'EL1':
                    return ArmAstUnaryOp('!', ArmAstFunction('IsFeatureImplemented', [ArmAstIdentifier('FEAT_EL2')]));
                # EL2 is the higest if we have EL2.
                if oNode.aoArgs[0].sName == 'EL2':
                    return ArmAstFunction('IsFeatureImplemented', [ArmAstIdentifier('FEAT_EL2')]);
                # Neither EL0 nor EL3 can be the highest, as EL0 and EL1 are mandator and we don't implement EL3.
                if oNode.aoArgs[0].sName in ('EL0', 'EL3'):
                    return ArmAstBool(False);
            elif oNode.aoArgs[0].isMatchingDotAtom('PSTATE', 'EL'):
                return ArmAstBinaryOp(ArmAstBinaryOp(ArmAstFunction('IsFeatureImplemented', [ArmAstIdentifier('FEAT_EL2')]),
                                                     '&&',
                                                     ArmAstBinaryOp(oNode.aoArgs[0], '==', ArmAstIdentifier('EL2'))),
                                      '||',
                                      ArmAstBinaryOp(ArmAstUnaryOp('!', ArmAstFunction('IsFeatureImplemented',
                                                                                       [ArmAstIdentifier('FEAT_EL2')])),
                                                     '&&',
                                                     ArmAstBinaryOp(oNode.aoArgs[0], '==', ArmAstIdentifier('EL1'))));
        raise Exception('Unexpected: %s' % (oNode.toString(),));

    kasImpDefBoolRetTrue = (
    );
    kaoReImpDefBoolRetTrue = (
        re.compile('^ID_.+ trapped by HCR_EL2\\..*$'),  # e.g. ID_AA64ISAR2_EL1 trapped by HCR_EL2.TID3
    );
    kasImpDefBoolRetFalse = (
        'Trapped by MDCR_EL2.TDOSA',                    # Play it safe and don't trap.
        'IMPLEMENTED_ACTLR_ELx accessor behavior',      # We don't have ACTLR_EL2 or ACTLRMASK_EL2, so False suits us best here.
    );
    kaoReImpDefBoolRetFalse= (
    );

    def transformCodePass1_ImpDefBool(self, oNode):
        """ Pass 1: ImpDefBool("blah blah"). """
        if len(oNode.aoArgs) == 1 and isinstance(oNode.aoArgs[0], ArmAstString):
            sValue = oNode.aoArgs[0].sValue;
            # Exact matches.
            for sCandidate in self.kasImpDefBoolRetTrue:
                if sValue == sCandidate:
                    return ArmAstBool(True);
            for sCandidate in self.kasImpDefBoolRetFalse:
                if sValue == sCandidate:
                    return ArmAstBool(False);
            # Regular expression matches.
            for oRe in self.kaoReImpDefBoolRetTrue:
                if oRe.match(sValue):
                    return ArmAstBool(True);
            for oRe in self.kaoReImpDefBoolRetFalse:
                if oRe.match(sValue):
                    return ArmAstBool(False);
            return oNode;
        raise Exception('Unexpected: %s' % (oNode.toString(),));

    def transformCodePass1_TypeAnnotation(self, oNode, oInfo):
        """ Pass 1: REG_NAME = (bits(0x40)) UNKNOWN; Implementation specific behavior. """
        cBitsWidth = -1;
        if oNode.oType.isMatchingFunctionCall('bits', int):
            cBitsWidth = oNode.oType.aoArgs[0].getIntegerOrValue();
            if cBitsWidth not in (32, 64):
                raise Exception('Unexpected type width: %s' % (oNode.toString(),));

        # For now, just return zero for all of these. If we need anything
        # else, we can determin the context from oInfo.
        _ = oInfo;
        return ArmAstInteger(0, cBitsWidth);

    def transformCodePass1_IsZero(self, oNode):
        """ Pass 1: IsZero(EffectiveSCTLRMASK_EL1()). """
        if len(oNode.aoArgs) == 1:
            return ArmAstUnaryOp('!', oNode.aoArgs[0]);
        raise Exception('Unexpected: %s' % (oNode.toString(),));

    def transformCodePass1_AArch64_SystemAccessTrap(self, oNode): # pylint: disable=invalid-name
        """ Pass 1: AArch64_SystemAccessTrap(EL2,0x18) and such. """
        if len(oNode.aoArgs) == 2:
            idxEl  = self.kdELxToNum.get(oNode.aoArgs[0].getIdentifierName());
            iClass = oNode.aoArgs[1].getIntegerValue();
            if idxEl is not None and iClass is not None:
                assert idxEl > 0;
                if iClass == 0:
                    return ArmAstReturn(ArmAstCppCall('iemRaiseSystemAccessTrapUnknown',
                                                      [ ArmAstCppExpr('pVCpu'), ArmAstInteger(idxEl), ]));
                if iClass == 7 and self.isA64Instruction():
                    return ArmAstReturn(ArmAstCppCall('iemRaiseSystemAccessTrapAdvSimdFpAccessA64',
                                                      [ ArmAstCppExpr('pVCpu'), ArmAstInteger(idxEl), ]));
                if iClass == 7 and not self.isA64Instruction():
                    return ArmAstReturn(ArmAstCppCall('iemRaiseSystemAccessTrapAdvSimdFpAccess',
                                                      [ ArmAstCppExpr('pVCpu'),
                                                        ArmAstInteger(idxEl),
                                                        ArmAstCppExpr('uInstrEssence'), ]));
                if iClass == 20:
                    return ArmAstReturn(ArmAstCppCall('iemRaiseSystemAccessTrap128Bit',
                                                      [ ArmAstCppExpr('pVCpu'),
                                                        ArmAstInteger(idxEl),
                                                        ArmAstCppExpr('uInstrEssence'), ]));
                if iClass == 24:
                    return ArmAstReturn(ArmAstCppCall('iemRaiseSystemAccessTrap',
                                                      [ ArmAstCppExpr('pVCpu'),
                                                        ArmAstInteger(idxEl),
                                                        ArmAstCppExpr('uInstrEssence'), ]));
                if iClass == 25:
                    return ArmAstReturn(ArmAstCppCall('iemRaiseSystemAccessTrapSve',
                                                      [ ArmAstCppExpr('pVCpu'), ArmAstInteger(idxEl), ]));

                if iClass == 29:
                    return ArmAstReturn(ArmAstCppCall('iemRaiseSystemAccessTrapSme',
                                                      [ ArmAstCppExpr('pVCpu'), ArmAstInteger(idxEl), ]));

        raise Exception('Unexpected: %s' % (oNode.toString(),));


    def transformCodePass1Callback(self, oNode, fEliminationAllowed, oInfo, aoStack):
        """
        Callback for pass 1: Code flow adjustments; Optimizations.
        """
        if isinstance(oNode, ArmAstIfList):
            # If we have a complete series of PSTATE.EL == ELx checks,
            # turn the final one into an else case to help compilers make
            # better sense of the code flow.
            if len(oNode.aoIfConditions) >= 3 and not oNode.oElseStatement: ## @todo EL3
                aidxEl = [-1, -1, -1, -1];
                for idxCond, oCond in enumerate(oNode.aoIfConditions):
                    if isinstance(oCond, ArmAstBinaryOp):
                        if oCond.sOp == '==':
                            if oCond.oLeft.isMatchingDotAtom('PSTATE', 'EL'):
                                idxEl = self.kdELxToNum.get(oCond.oRight.getIdentifierName(), -1);
                                if idxEl >= 0:
                                    assert aidxEl[idxEl] == -1;
                                    aidxEl[idxEl] = idxCond;
                assert aidxEl[3] == -1; ## todo EL3
                if aidxEl[0] >= 0 and aidxEl[1] >= 0 and aidxEl[2] >= 0:
                    # We've found checks for each of the 4 EL levels.  Convert the last one into the else.
                    idxLast = max(aidxEl);
                    assert idxLast + 1 == len(oNode.aoIfStatements); # There shall not be anything after the final EL check.
                    oNode.oElseStatement = oNode.aoIfStatements[idxLast];
                    oNode.aoIfConditions = oNode.aoIfConditions[:idxLast];
                    oNode.aoIfStatements = oNode.aoIfStatements[:idxLast];

            # If the first condition is true, just use the corresponding statement.
            if oNode.aoIfConditions and oNode.aoIfConditions[0].isBoolAndTrue():
                return oNode.aoIfStatements[0];

            # Iff there is an else statement, eliminate any identical conditional
            # code blocks immediately preceding it.
            if oNode.oElseStatement and oNode.aoIfStatements:
                cIfStmts = len(oNode.aoIfStatements);
                while cIfStmts > 0 and oNode.oElseStatement.isSame(oNode.aoIfStatements[cIfStmts - 1]):
                    cIfStmts -= 1;
                if cIfStmts < len(oNode.aoIfStatements):
                    oNode.aoIfConditions = oNode.aoIfConditions[:cIfStmts];
                    oNode.aoIfStatements = oNode.aoIfStatements[:cIfStmts];
            if oNode.oElseStatement and not oNode.aoIfStatements:
                return oNode.oElseStatement;

        elif isinstance(oNode, ArmAstBinaryOp):
            # EL3 not implemented, so eliminate PSTATE.EL == 3 and similar. ## @todo EL3
            if oNode.oLeft.isMatchingDotAtom('PSTATE', 'EL'):
                if oNode.oRight.isMatchingIdentifier('EL3'):
                    if oNode.sOp in ('==', '>='):
                        return ArmAstBool(False);
                    if oNode.sOp in ('!=', '<'):
                        return ArmAstBool(True);
                elif oNode.sOp == '>' and oNode.oRight.isMatchingIdentifier('EL2'): # PSTATE.EL > EL2.
                    return ArmAstBool(False);

            # This isn't 100% safe, better do it later on when we generate the C code.
            ## Simplify boolean fields checks.
            #if isinstance(oNode.oLeft, ArmAstField) and isinstance(oNode.oRight, ArmAstValue):
            #    if oNode.oRight.getWidth(None) == 1:
            #        oReg = spec.g_ddoAllArmRegistersByStateByName[oNode.oLeft.sState][oNode.oLeft.sName]; # ArmRegister
            #        aoFields = oReg.daoFields.get(oNode.oLeft.sField); # List[ArmFieldsBase]
            #        if (    aoFields
            #            and len(aoFields[0].aoRanges) == 1
            #            and aoFields[0].aoRanges[0].cBitsWidth == 1
            #            and (   len(aoFields) == 1
            #                 or (    isinstance(aoFields[0].oParent, spec.ArmFieldsConditionalField)
            #                     and len(aoFields[0].oParent.aoRanges) == 1
            #                     and aoFields[0].oParent.aoRanges[0].cBitsWidth == 1))):
            #            if oNode.sOp == '==' and oNode.oRight.sValue == "'1'":
            #                return oNode.oLeft;
            #            if oNode.sOp == '==' and oNode.oRight.sValue == "'0'":
            #                return ArmAstUnaryOp('!', oNode.oLeft);
            #            if oNode.sOp == '!=' and oNode.oRight.sValue == "'0'":
            #                return oNode.oLeft;
            #            if oNode.sOp == '!=' and oNode.oRight.sValue == "'1'":
            #                return ArmAstUnaryOp('!', oNode.oLeft);

        elif isinstance(oNode, ArmAstFunction):
            # Since we don't implement any external debug state (no EDSCR.STATUS),
            # the Halted() and HaltingAllowed() functions always returns False.
            # Eliminate them.
            if (   oNode.isMatchingFunctionCall('Halted')
                or oNode.isMatchingFunctionCall('HaltingAllowed')):
                return ArmAstBool(False);

            # The EL3SDDUndefPriority() and EL3SDDUndef() can likewise be eliminated,
            # as they requires Halted() to be true and EDSCR.SDD to be set.
            if (   oNode.isMatchingFunctionCall('EL3SDDUndefPriority')
                or oNode.isMatchingFunctionCall('EL3SDDUndef')):
                return ArmAstBool(False);

            if oNode.sName == 'HaveAArch64':
                return self.transformCodePass1_HaveAArch64();
            if oNode.sName == 'HaveEL':
                return self.transformCodePass1_HaveEL(oNode);
            if oNode.sName == 'HaveAArch32EL':
                return self.transformCodePass1(oInfo, self.transformCodePass1_HaveAArch32EL(oNode));
            if oNode.sName == 'ELIsInHost':
                return self.transformCodePass1(oInfo, self.transformCodePass1_ELIsInHost(oNode));
            if oNode.sName == 'IsFeatureImplemented':
                return self.transformCodePass1_IsFeatureImplemented(oNode);
            if oNode.sName == 'IsCurrentSecurityState':
                return self.transformCodePass1_IsCurrentSecurityState(oNode);
            if oNode.sName == 'IsHighestEL':
                return self.transformCodePass1_IsHighestEL(oNode);
            if oNode.sName == 'ImpDefBool':
                return self.transformCodePass1_ImpDefBool(oNode);

            if oNode.sName == 'IsZero':
                return self.transformCodePass1_IsZero(oNode);

            if oNode.sName == 'AArch64_SystemAccessTrap':
                return self.transformCodePass1_AArch64_SystemAccessTrap(oNode);

            # Generic mapping of functions without any arguments:
            if len(oNode.aoArgs) == 0 and oNode.sName in self.kdAstForFunctionsWithoutArguments:
                oNode = self.kdAstForFunctionsWithoutArguments[oNode.sName].clone();
                return self.transformCodePass1(oInfo, oNode);

        elif isinstance(oNode, ArmAstDotAtom):
            # Rewrite lazy ones into Field accesses to avoid duplicating code..
            if len(oNode.aoValues) == 2:
                sReg   = oNode.aoValues[0].getIdentifierName();
                sField = oNode.aoValues[1].getIdentifierName();
                if sReg and sReg != 'PSTATE' and sField:
                    return ArmAstField(sField, sReg, self.getRegStateName());

        elif isinstance(oNode, ArmAstValue):
            # Convert non-wildcard Values.Value nodes into integer nodes with specific width.
            (fValue, _, fWildcard, cBitsWidth) = oNode.getValueDetails();
            if fWildcard == 0:
                return ArmAstInteger(fValue, cBitsWidth);

        elif isinstance(oNode, ArmAstTypeAnnotation):
            if aoStack and isinstance(aoStack[-1], ArmAstAssignment):
                # Typically these are 'REG_NAME = (bits(0x40)) UNKNOWN';
                return self.transformCodePass1_TypeAnnotation(oNode, oInfo);

        _ = fEliminationAllowed;
        return oNode;

    def transformCodePass1(self, oInfo, oCode):
        """ Code transformation pass 1: Code flow adjustments; Optimizations. """
        return oCode.transform(self.transformCodePass1Callback, True, oInfo, []);

    #
    # Pass 2 - C++ translation.
    #

    @staticmethod
    def isIdRegisterEncoding(oEncoding):
        """
        Checks if oEncoding (ArmRegEncoding) is for a known ID register and
        CPUM handleds by lookup.
        """
        # Note! This must match SUPDrv.cpp and CPUMAllCpuId.cpp...
        if (   oEncoding.compareCStyle(3, 0,  0,     range(8), None) == 0 # ID block
            or oEncoding.compareCStyle(3, 0,  5,            3,    0) == 0 # ERRIDR_EL1
            or oEncoding.compareCStyle(3, 0,  9,            9,    7) == 0 # PMSIDR_EL1
            or oEncoding.compareCStyle(3, 0,  9,           10,    7) == 0 # PMBIDR_EL1
            or oEncoding.compareCStyle(3, 0,  9,           11,    7) == 0 # TRBIDR_EL1
            or oEncoding.compareCStyle(3, 0,  9,           14,    6) == 0 # PMMIR_EL1
            or oEncoding.compareCStyle(3, 0, 10,            4,    5) == 0 # MPAMBWIDR_EL1
            or oEncoding.compareCStyle(3, 1,  0,            0,    4) == 0 # GMID_EL1
            or oEncoding.compareCStyle(3, 1,  0,            0,    6) == 0 # SMIDR_EL1
            or oEncoding.compareCStyle(2, 1,  0, range(8, 16),    7) == 0 # TRCIDR[0-7]
            or oEncoding.compareCStyle(2, 1,  0,     range(6),    6) == 0 # TRCIDR[8-13]
            or oEncoding.compareCStyle(2, 1,  7,           15,    6) == 0 # TRCDEVARCH
            # misc:
            or oEncoding.compareCStyle(3, 3,  0,            0,    1) == 0 # CTR_EL0
            or oEncoding.compareCStyle(3, 3,  0,            0,    7) == 0 # DCZID_EL0
            ):
            return True;
        return False;

    def lookupRegisterField(self, sState, sRegisterName, sField, sWhatFor, fWarnOnly = False):
        """
        Helper to lookup a register field, returning (oReg, oField, sAccessExpr|fReservedValue).
        Raises exceptions if not found or too complicated.
        """
        oReg = spec.g_ddoAllArmRegistersByStateByName[sState].get(sRegisterName); # ArmRegister
        if not oReg:
            raise Exception('%s: Register "%s" not found (field "%s" lookup)' % (sWhatFor, sRegisterName, sField,));

        aoFields = oReg.daoFields.get(sField); # List[ArmFieldsBase]
        if not aoFields:
            if sField in ('EL1PCTEN_AT_00', 'EL1PCTEN_AT_10') and sRegisterName == 'CNTHCTL_EL2': # HACK ALERT! See pass 0.
                aoFields = [spec.ArmFieldsField(None, [spec.ArmRange(int(sField[-2:]), 1),], sField),];
            else:
                oXcpt = Exception('%s: Field "%s" not found in register "%s"' % (sWhatFor, sField, sRegisterName,));
                if fWarnOnly: return self.warnXcpt(oXcpt, (None, None, None));
                raise oXcpt;

        tCpumCtxInfo = g_dRegToCpumCtx.get('%s.%s' % (sState, sRegisterName));
        if tCpumCtxInfo:
            oAccessExprOrInt = 'pVCpu->cpum.GstCtx.' + tCpumCtxInfo[0] if isinstance(tCpumCtxInfo[0], str) else tCpumCtxInfo[0];
        else:
            if sRegisterName == 'OSLSR_EL1' and sState == 'AArch64' and sField == 'OSLK':
                return (oReg, spec.ArmFieldsField(None, [spec.ArmRange(0, 1),], 'OSLK'),
                        'pVCpu->cpum.GstCtx.fOsLck'); # HACK ALERT: boolean field

            # Some hacks for unimplemented registers and gcc warnings.
            if sRegisterName == 'PMUSERENR_EL0' and sState == 'AArch64':
                oAccessExprOrInt = 'iemCImplHlpGetPmUserEnrEl0(pVCpu)';
            elif sRegisterName == 'AMUSERENR_EL0' and sState == 'AArch64':
                oAccessExprOrInt = 'iemCImplHlpGetAmUserEnrEl0(pVCpu)';
            elif sRegisterName == 'PMUACR_EL1' and sState == 'AArch64':
                oAccessExprOrInt = 'iemCImplHlpGetPmUacrEl1(pVCpu)';
            #elif sRegisterName == 'PMSELR_EL0' and sState == 'AArch64':
            #     oAccessExprOrInt = 'iemCImplHlpGetPmSelrEl0(pVCpu)';
            else:
                # If this is an ID register, we must a helper function to access it (assuming RES0).
                oAccessExprOrInt = None;
                for oAccessor in oReg.aoAccessors:
                    if isinstance(oAccessor, spec.ArmAccessorSystem):
                        if oAccessor.oEncoding.sAsmValue == sRegisterName:
                            if self.isIdRegisterEncoding(oAccessor.oEncoding):
                                oAccessExprOrInt = oAccessor.oEncoding.getSysRegIdCreate();
                                break;
                if not oAccessExprOrInt:
                    oXcpt = Exception('%s: No CPUMCTX mapping for register %s.%s (looking up field %s)'
                                      % (sWhatFor, sState, sRegisterName, sField,));
                    if fWarnOnly: return self.warnXcpt(oXcpt, (None, None, None));
                    raise oXcpt;
                oAccessExprOrInt = 'iemCImplHlpGetIdSysReg(pVCpu, %s)' % (oAccessExprOrInt,); ## @todo return AST object

        #
        # Iff this is an conditional field, there may be more than one entry in
        # aoFields and these all are relative to the ArmFieldsConditionalField
        # parent.  So, we just check that it's a simple one and return the
        # parent instaed of the children in aoFields.
        #
        oField = aoFields[-1]; # ArmFieldsBase
        if (    isinstance(oField.oParent, spec.ArmFieldsConditionalField)
            and len(oField.oParent.aoRanges)  == 1
            and len(oField.aoRanges)          == 1
            and oField.aoRanges[0].cBitsWidth == oField.oParent.aoRanges[0].cBitsWidth
            and oField.aoRanges[0].iFirstBit  == 0):
            oField = oField.oParent; # ArmFieldsConditionalField
            if oField.sName is None:
                oField.sName = sField; # HACK ALERT!!
            return (oReg, oField, oAccessExprOrInt,);

        # Try deal with multiple register views depending on context by check
        # if the field being accessed has the same place in all contexts.
        # CPTR_EL2.TCPAC is an example of this.
        if (    len(aoFields) > 1
            and not isinstance(oField.oParent, spec.ArmFieldsConditionalField)
            and len(aoFields[0].aoRanges) == 1):
            fSame = True;
            for idxField in range(1, len(aoFields)):
                fSame = (    fSame
                         and len(aoFields[idxField].aoRanges) == 1
                         and aoFields[idxField].aoRanges[0].isEqualTo(aoFields[0].aoRanges[0]));
            if fSame:
                aoFields = aoFields[:1];

        if len(aoFields) != 1 or isinstance(oField.oParent, spec.ArmFieldsConditionalField):
            oXcpt = Exception('%s: Ambigious register field in %s: %s (%s)'
                              % (sWhatFor, sRegisterName, sField,
                                 ['%s:%s (parent=%s:%s)'
                                  % (type(oField).__name__, oField.toString(),
                                     type(oField.oParent).__name__, oField.oParent.toString(),)
                                  for oField in aoFields] ));
            if fWarnOnly: return self.warnXcpt(oXcpt, (None, None, None));
            raise oXcpt;


        if len(oField.aoRanges) != 1:
            raise Exception('%s: Distributed register field in Concat() argument list: %s (%s)'
                            % (sWhatFor, sField, oField.toString(),));

        #print('reg.field: %s.%s' % (oReg.sName, oField.sName));
        return (oReg, oField, oAccessExprOrInt,);


    def transformCodePass2_IsFeatureImplemented(self, oNode):
        """ Pass 2: IsFeatureImplemented(FEAT_xxxx) -> pGstFeats->fXxxx. """
        if len(oNode.aoArgs) == 1 and isinstance(oNode.aoArgs[0], ArmAstIdentifier):
            sFeatureNm   = oNode.aoArgs[0].sName;
            sCpumFeature = g_dSpecFeatToCpumFeat.get(sFeatureNm, None);
            if sCpumFeature is None:
                raise Exception('Unknown IsFeatureImplemented parameter: %s (see g_dSpecFeatToCpumFeat)' % (sFeatureNm));
            if isinstance(sCpumFeature, str):
                return ArmAstCppExpr('pGstFeats->%s' % (sCpumFeature,), cBitsWidth = 1);
            if sCpumFeature is True:  return 'true /*%s*/' % (sFeatureNm,);
            if sCpumFeature is False: return 'false /*%s*/' % (sFeatureNm,);
            return ArmAstCppExpr('false /** @todo pGstFeats->%s */' % (sFeatureNm,), cBitsWidth = 1);
        raise Exception('Unexpected: %s' % (oNode,));

    def transformCodePass2_EffectiveHCR_EL2_NVx(self, oNode): # pylint: disable=invalid-name
        """ Pass 2: EffectiveHCR_EL2_NVx() -> helper call. """
        if len(oNode.aoArgs) == 0:
            return ArmAstCppCall('iemGetEffHcrEl2NVx', [ ArmAstCppExpr('pVCpu'), ArmAstCppExpr('pGstFeats'), ], cBitsWidth = 3);
        raise Exception('Unexpected: %s' % (oNode,));

    def transformCodePass2_EffectiveACTLRMASK_EL1(self, oNode): # pylint: disable=invalid-name
        """ Pass 2: EffectiveACTLRMASK_EL1() -> helper call. """
        if len(oNode.aoArgs) == 0:
            return ArmAstCppCall('iemGetEffActlrMaskEl1', [ ArmAstCppExpr('pVCpu'), ArmAstCppExpr('pGstFeats'), ],
                                 cBitsWidth = 64);
        raise Exception('Unexpected: %s' % (oNode,));

    def transformCodePass2_EffectiveCPACRMASK_EL1(self, oNode): # pylint: disable=invalid-name
        """ Pass 2: EffectiveCPACRMASK_EL1() -> helper call. """
        if len(oNode.aoArgs) == 0:
            return ArmAstCppCall('iemGetEffCpacrMaskEl1', [ ArmAstCppExpr('pVCpu'), ArmAstCppExpr('pGstFeats'), ],
                                 cBitsWidth = 64);
        raise Exception('Unexpected: %s' % (oNode,));

    def transformCodePass2_EffectiveCPTRMASK_EL2(self, oNode): # pylint: disable=invalid-name
        """ Pass 2: EffectiveCPTRMASK_EL2() -> helper call. """
        if len(oNode.aoArgs) == 0:
            return ArmAstCppCall('iemGetEffCptrMaskEl2', [ ArmAstCppExpr('pVCpu'), ArmAstCppExpr('pGstFeats'), ],
                                 cBitsWidth = 64);
        raise Exception('Unexpected: %s' % (oNode,));

    def transformCodePass2_EffectiveSCTLRMASK_EL1(self, oNode): # pylint: disable=invalid-name
        """ Pass 2: EffectiveSCTLRMASK_EL1() -> helper call. """
        if len(oNode.aoArgs) == 0:
            return ArmAstCppCall('iemGetEffSctlrMaskEl1', [ ArmAstCppExpr('pVCpu'), ArmAstCppExpr('pGstFeats'), ],
                                 cBitsWidth = 64);
        raise Exception('Unexpected: %s' % (oNode,));

    def transformCodePass2_EffectiveSCTLRMASK_EL2(self, oNode): # pylint: disable=invalid-name
        """ Pass 2: EffectiveSCTLRMASK_EL2() -> helper call. """
        if len(oNode.aoArgs) == 0:
            return ArmAstCppCall('iemGetEffSctlrMaskEl2', [ ArmAstCppExpr('pVCpu'), ArmAstCppExpr('pGstFeats'), ],
                                 cBitsWidth = 64);
        raise Exception('Unexpected: %s' % (oNode,));

    def transformCodePass2_GetCurrentEXLOCKEN(self, oNode): # pylint: disable=invalid-name
        """ Pass 2: GetCurrentEXLOCKEN() -> helper call. """
        if len(oNode.aoArgs) == 0:
            return ArmAstCppCall('iemGetCurrentExlockEn', [ ArmAstCppExpr('pVCpu'), ArmAstCppExpr('pGstFeats'), ],
                                 cBitsWidth = 1);
        raise Exception('Unexpected: %s' % (oNode,));

    def transformCodePass2_PhysicalCountInt(self, oNode):
        """ Pass 2: Translate PhysicalCountInt(). """
        if len(oNode.aoArgs) == 0:
            return ArmAstCppCall('iemCImplHlpGetPhysicalSystemTimerCount', [ ArmAstCppExpr('pVCpu'), ], cBitsWidth = 64);
        raise Exception('Unexpected: %s' % (oNode,));

    def transformCodePass2_SecurityStateAtEL(self, oNode):
        """ Pass 2: Translate SecurityStateAtEL(). """
        if len(oNode.aoArgs) == 1:
            idxEl = self.kdELxToNum.get(oNode.aoArgs[0].getIdentifierName(), -1);
            if idxEl >= 0:
                return ArmAstCppCall('iemCImplHlpGetSecurityStateAtEl',
                                     [ ArmAstCppExpr('pVCpu'), ArmAstCppExpr(str(idxEl)),], cBitsWidth = 64);
        raise Exception('Unexpected: %s' % (oNode,));

    def transformCodePass2_GCSEnabled(self, oNode):
        """ Pass 2: Translate GCSEnabled(EL0|EL1|EL2|EL3|PSTATE.EL). """
        if len(oNode.aoArgs) == 1:
            if oNode.aoArgs[0].isMatchingDotAtom('PSTATE', 'EL'):
                return ArmAstCppCall('iemCImplHlpIsGcsEnabledAtCurrentEl', [ ArmAstCppExpr('pVCpu'), ],  cBitsWidth = 1);
            idxEl = self.kdELxToNum.get(oNode.aoArgs[0].getIdentifierName(), -1);
            if idxEl >= 0:
                return ArmAstCppCall('iemCImplHlpIsGcsEnabled',
                                     [ ArmAstCppExpr('pVCpu'), ArmAstCppExpr(str(idxEl)) ],  cBitsWidth = 1);
        raise Exception('Unexpected: %s' % (oNode,));

    def transformCodePass2_UInt(self, oNode):
        """ Pass 2: Eliminate UInt. """
        if len(oNode.aoArgs) == 1:
            # If it's a C++ expression, we should be safe wrt signed-ness and width.
            if isinstance(oNode.aoArgs[0], ArmAstCppExprBase):
                return oNode.aoArgs[0];
            ## @todo Prepending (uint64_t) or similar isn't possible unless it's already an C++ expression...
            return oNode;
        raise Exception('Unexpected: %s' % (oNode,));

    def transformCodePass2_Zeros(self, oNode):
        """ Pass 2: Eliminate Zeros(bits). """
        if len(oNode.aoArgs) == 1:
            cBits = oNode.aoArgs[0].getIntegerOrValue();
            if cBits is not None and 0 < cBits <= 64:
                return ArmAstInteger(0, cBits);
        raise Exception('Unexpected: %s' % (oNode,));

    def transformCodePass2_ZeroExtend(self, oNode, oInfo):
        """ Pass 2: Eliminate ZeroExtend(value<x:y>, bits). """
        if len(oNode.aoArgs) == 2:
            cBits = oNode.aoArgs[1].getIntegerOrValue();
            if cBits is not None and cBits in (64, 32):
                oValue = oNode.aoArgs[0];
                if isinstance(oValue, ArmAstSquareOp):
                    if len(oValue.aoValues) == 1 and isinstance(oValue.aoValues[0], ArmAstSlice):
                        iFrom  = oValue.aoValues[0].oFrom.getIntegerOrValue();
                        iTo    = oValue.aoValues[0].oTo.getIntegerOrValue();
                        if iFrom is not None and iTo is not None and 0 <= iTo < iFrom < 63:
                            if self.checkCConversion(oInfo, oValue.oVar):
                                sValueExpr = oValue.oVar.toStringEx(sLang = 'C', cchMaxWidth = 99999);
                                if iTo == 0 and iFrom == 31:
                                    if cBits == 64:
                                        return ArmAstCppExpr('(uint64_t)(uint32_t)(%s)' % (sValueExpr,));
                                    if cBits == 32:
                                        return ArmAstCppExpr('(uint32_t)(%s)' % (sValueExpr,));
                                if iTo > 0:
                                    sValueExpr = '((%s) >> %u)' % (sValueExpr, iTo);
                                iFrom -= iTo;
                                if iFrom == 31:
                                    sValueExpr = '(uint32_t)(%s)' % (sValueExpr,);
                                    if cBits == 64:
                                        sValueExpr = '(uint64_t)%s' % (sValueExpr,);
                                else:
                                    sValueExpr = '((%s) & UINT%u_C(%#x)' % (sValueExpr, cBits, (1 << (iFrom + 1)) - 1,);
                                return ArmAstCppExpr(sValueExpr, cBitsWidth = cBits);
                            return oNode;
        raise Exception('Unexpected: %s' % (oNode,));

    def transformCodePass2_SignExtend(self, oNode):
        """ Pass 2: Eliminate SignExtend(value<x:y>, bits). """
        if len(oNode.aoArgs) == 2:
            cBits = oNode.aoArgs[1].getIntegerOrValue();
            if cBits is not None and cBits in (64, 32):
                oValue = oNode.aoArgs[0];
                if isinstance(oValue, ArmAstSquareOp):
                    if len(oValue.aoValues) == 1 and isinstance(oValue.aoValues[0], ArmAstSlice):
                        iFrom  = oValue.aoValues[0].oFrom.getIntegerOrValue();
                        iTo    = oValue.aoValues[0].oTo.getIntegerOrValue();
                        if iFrom is not None and iTo is not None and 0 <= iTo < iFrom < 63:
                            if isinstance(oValue.oVar, (ArmAstCppExpr,)):
                                sValueExpr = oValue.oVar.toStringEx(sLang = 'C', cchMaxWidth = 99999);
                                if iTo == 0 and iFrom == 31:
                                    if cBits == 64:
                                        return ArmAstCppExpr('(int64_t)(int32_t)(%s)' % (sValueExpr,));
                                    if cBits == 32:
                                        return ArmAstCppExpr('(int32_t)(%s)' % (sValueExpr,));
                                return ArmAstCppExpr('((int%u_t)((%s) << %d) >> %d)'
                                                     % (cBits, sValueExpr, cBits - iFrom - 1, cBits - iFrom - 1 + iTo, ),
                                                     cBitsWidth = cBits);
        raise Exception('Unexpected: %s' % (oNode,));


    class VBoxAstCppConcatEntry(object):
        """ An entry in a concat value list. """
        def __init__(self, oOrgValue, cBitsWidth, fValue = None, sAccessExpr = None, oField = None, iFirstBit = -1):
            self.oOrgValue  = oOrgValue # type: ArmAstFunction | ArmAstValue | ArmAstField | ArmAstDotAtom | ArmAstConcat
            ## Fixed field value or None.
            self.fValue     = fValue;
            ## Either a CPUMCTX field access expression or a getter function call expression. Usually resulting in uint64_t.
            self.sAccessExpr = sAccessExpr  # type: str | None
            ## The field details.
            self.oField     = oField    # type: spec.ArmFieldsBase
            ## The number of bits wide the field is.
            self.cBitsWidth = cBitsWidth;
            ## The position in the result.
            self.iFirstBit  = iFirstBit;

        def clone(self):
            return SysRegGeneratorBase.VBoxAstCppConcatEntry(self.oOrgValue, self.cBitsWidth, fValue = self.fValue,
                                                             sAccessExpr = self.sAccessExpr, oField = self.oField,
                                                             iFirstBit = self.iFirstBit);

    class VBoxAstCppConcat(ArmAstCppExpr):
        """ Converted ArmAstConcat. """
        def __init__(self, aoInfoEntries, oOriginal):
            ArmAstCppExpr.__init__(self, oOriginal.toString(), aoInfoEntries[0].iFirstBit + aoInfoEntries[0].cBitsWidth);
            self.aoInfoEntries = aoInfoEntries # type: List[VBoxAstCppConcatEntry]

            # Convert the info items to C++ code.
            asExprs = [];
            i       = 0;
            fFixed  = 0;
            while i < len(aoInfoEntries):
                oInfoEntry = aoInfoEntries[i];
                if oInfoEntry.fValue is not None:
                    fFixed |= oInfoEntry.fValue << oInfoEntry.iFirstBit;
                else:
                    assert oInfoEntry.sAccessExpr;
                    assert oInfoEntry.oField;
                    sNames      = oInfoEntry.oField.sName;
                    iFirstBit   = oInfoEntry.oField.aoRanges[0].iFirstBit;
                    cBitsWidth  = oInfoEntry.oField.aoRanges[0].cBitsWidth;
                    #print('debug/concat: i=%s %uL%u: %s/%uL%u -> %s' % (i, oInfoEntry.iFirstBit, oInfoEntry.cBitsWidth, sNames,
                    #                                                    iFirstBit, cBitsWidth, oInfoEntry.sAccessExpr));
                    while (    i + 1 < len(aoInfoEntries)
                           and aoInfoEntries[i + 1].sAccessExpr is oInfoEntry.sAccessExpr):
                        iCurFirstBit  = aoInfoEntries[i + 1].oField.aoRanges[0].iFirstBit;
                        cCurBitsWidth = aoInfoEntries[i + 1].oField.aoRanges[0].cBitsWidth;
                        if iCurFirstBit + cCurBitsWidth != iFirstBit:
                            break;
                        i += 1;
                        oInfoEntry = aoInfoEntries[i];
                        #print('debug/concat: i=%s %uL%u: %s/%uL%u ++' % (i, oInfoEntry.iFirstBit, oInfoEntry.cBitsWidth,
                        #                                                 oInfoEntry.oField.sName, iCurFirstBit, cCurBitsWidth));
                        iFirstBit   = iCurFirstBit;
                        cBitsWidth += cCurBitsWidth;
                        sNames     += ',' + aoInfoEntries[i].oField.sName;

                    fMask = ((1 << cBitsWidth) - 1) << iFirstBit;
                    if iFirstBit == oInfoEntry.iFirstBit:
                        asExprs.append('(%s & UINT64_C(%#x)/*%s*/)' % (oInfoEntry.sAccessExpr, fMask, sNames));
                    elif iFirstBit > oInfoEntry.iFirstBit:
                        asExprs.append('((%s & UINT64_C(%#x)/*%s*/) >> %d)'
                                       % (oInfoEntry.sAccessExpr, fMask, sNames, iFirstBit - oInfoEntry.iFirstBit));
                    else:
                        asExprs.append('((%s & UINT64_C(%#x)/*%s*/) << %d)'
                                       % (oInfoEntry.sAccessExpr, fMask, sNames, oInfoEntry.iFirstBit - iFirstBit));
                i += 1;
            if fFixed != 0:
                asExprs.append('UINT64_C(%#x)' % (fFixed,));

            self.asExprs = asExprs;
            self.sExpr   = '(%s)' % (' | '.join(asExprs),) if len(asExprs) > 1 else asExprs[0];

        def toStringEx(self, sLang = None, cchMaxWidth = 120):
            _ = sLang;
            if len(self.sExpr) <= cchMaxWidth:
                return self.sExpr;
            return '(  %s)' % ('\n | '.join(self.asExprs),);

        def areAllFromSameRegister(self):
            """ Checks if all the entries are from the same register or are zero. """
            sAccessExpr = self.aoInfoEntries[0].sAccessExpr;
            idx         = 1;
            while idx < len(self.aoInfoEntries) and not sAccessExpr:
                sAccessExpr = self.aoInfoEntries[idx].sAccessExpr;
                idx += 1;
            if sAccessExpr:
                for oInfoEntry in self.aoInfoEntries:
                    if oInfoEntry.sAccessExpr is None or oInfoEntry.sAccessExpr != sAccessExpr:
                        if oInfoEntry.fValue is None or oInfoEntry.fValue != 0:
                            return False;
                return True;
            return False;


    ## PSTATE field info.
    #   0. number of bits
    #   1. AArch64 bit position.
    kdPstateFields = {
        'SP':       (1,  0, ), ## @todo ?
        'EL':       (2,  2, ),
        'F':        (1,  6, ),
        'I':        (1,  7, ),
        'A':        (1,  8, ),
        'D':        (1,  9, ),
        'SSBS':     (1, 12, ),
        'ALLINT':   (1, 13, ),
        'PAN':      (1, 22, ),
        'UAO':      (1, 23, ),
        'DIT':      (1, 24, ),
        'TCO':      (1, 25, ),
        'V':        (1, 28, ),
        'C':        (1, 29, ),
        'Z':        (1, 30, ),
        'N':        (1, 31, ),
        'PM':       (1, 32, ),
        'EXLOCK':   (1, 34, ), # FEAT_GCS
    };

    def transformCodePass2_Concat(self, oNode):  # (ArmAstConcat) -> ArmAstBase
        """ Pass 2: Generic ArmAstConcat. """
        # Process the values and create a parallel list of VBoxAstCppConcatEntry objects.
        aoInfoEntries = []
        for oValue in oNode.aoValues:
            if isinstance(oValue, ArmAstFunction):
                if oValue.isMatchingFunctionCall('Zeros', int) and oValue.aoArgs[0].iValue > 0 and oValue.aoArgs[0].iValue <= 64:
                    aoInfoEntries.append(self.VBoxAstCppConcatEntry(oValue, oValue.aoArgs[0].iValue, fValue = 0));
                else:
                    raise Exception('Unexpected function in Concat() argument list: %s (%s)' % (oValue, oNode,));
            elif isinstance(oValue, (ArmAstValue, ArmAstInteger)):
                (fValue, _, fWildcard, cBitsWidth) = oValue.getValueDetails();
                if fWildcard != 0:
                    raise Exception('Unexpected wildcard value in Concat() argument list: %s (%s)' % (oValue, oNode,));
                aoInfoEntries.append(self.VBoxAstCppConcatEntry(oValue, cBitsWidth, fValue = fValue));
            elif isinstance(oValue, ArmAstField):
                (_, oField, oAccessExprOrInt) = self.lookupRegisterField(oValue.sState, oValue.sName, oValue.sField,
                                                                         'Concat/%s' % (oNode,));
                if isinstance(oAccessExprOrInt, int):
                    aoInfoEntries.append(self.VBoxAstCppConcatEntry(oValue, oField.aoRanges[0].cBitsWidth,
                                                                    fValue = oAccessExprOrInt, oField = oField));
                else:
                    aoInfoEntries.append(self.VBoxAstCppConcatEntry(oValue, oField.aoRanges[0].cBitsWidth,
                                                                    sAccessExpr = oAccessExprOrInt, oField = oField));
            elif isinstance(oValue, ArmAstDotAtom):
                if len(oValue.aoValues) == 2 and oValue.aoValues[0].isMatchingIdentifier('PSTATE'):
                    sField = oValue.aoValues[1].getIdentifierName();
                    tInfo  = self.kdPstateFields.get(sField);
                    if not tInfo:
                        raise Exception('Unexpected PSTAT field in Concat() argument list: %s (%s)' % (oValue, oNode,));
                    oField = spec.ArmFieldsBase(None, [spec.ArmRange(tInfo[1], tInfo[0])], sField); ## @todo aarch32
                    aoInfoEntries.append(self.VBoxAstCppConcatEntry(oValue, tInfo[0], sAccessExpr = 'pVCpu->cpum.GstCtx.fPState',
                                                                    oField = oField));
                else:
                    raise Exception('Unexpected DotAtom in Concat() argument list: %s (%s)' % (oValue, oNode,));
            elif isinstance(oValue, self.VBoxAstCppConcat):
                for oInfoEntry in oValue.aoInfoEntries:
                    aoInfoEntries.append(oInfoEntry.clone())
            else:
                raise Exception('Unexpected value in Concat() argument list: %s: %s (%s)' % (type(oValue), oValue, oNode,));

        # Assign bit positions.  This is done in reverse order as the Concat
        # arguments are in most to least significant bit order.
        iBit = 0;
        cNonValues = 0;
        for oInfoEntry in reversed(aoInfoEntries):
            oInfoEntry.iFirstBit = iBit;
            iBit += oInfoEntry.cBitsWidth;
            cNonValues += 1 if oInfoEntry.fValue is None else 0;

        # If all the entries have fixed values, we can convert this to a C++
        # constant of the appropriate size.
        if cNonValues == 0:
            fValue = 0;
            for oInfoEntry in aoInfoEntries:
                fValue |= oInfoEntry.fValue << oInfoEntry.iFirstBit;
            return ArmAstInteger(fValue, iBit);

        return self.VBoxAstCppConcat(aoInfoEntries, oNode);

    def transformCodePass2_BinaryOp_ConcatAndValueOrInt(self, oNode): # pylint: disable=invalid-name
        """ Pass 2: (AArch64.MDCR_EL2.TDE):(AArch64.MDCR_EL2.TDA) != '00' and similar """
        # First process the value.
        (fValue, _, fWildcard, _) = oNode.oRight.getValueDetails();
        #(fValue, fFixed, fWildcard, cBitsWidth) = oNode.oRight.getValueDetails();

        #
        # If all concatenated fields are from the same register, and the check
        # is either for any non-zero fields or that they are all non-zero, we
        # can drop the shifting and combine the masks.
        #
        if fValue == 0 and fWildcard == 0 and oNode.sOp in ('!=', '==',):
            if oNode.oLeft.areAllFromSameRegister():
                fMask       = 0;
                sAccessExpr = None;
                asNames     = [];
                for oInfoEntry in oNode.oLeft.aoInfoEntries:
                    if oInfoEntry.fValue is None:
                        fMask |= ((1 << oInfoEntry.oField.aoRanges[0].cBitsWidth) - 1) << oInfoEntry.oField.aoRanges[0].iFirstBit;
                        sAccessExpr = oInfoEntry.sAccessExpr;
                        assert oInfoEntry.oField.sName;
                        asNames.append(oInfoEntry.oField.sName);
                    else:
                        assert oInfoEntry.fValue == 0;
                oNode.oLeft.asExprs = ['%s & UINT64_C(%#x) /*%s*/' % (sAccessExpr, fMask, ','.join(asNames))];
                oNode.oLeft.sExpr = '(%s)' % (oNode.oLeft.asExprs[0],)

                if oNode.sOp == '!=':
                    return oNode.oLeft;
                return ArmAstUnaryOp('!', oNode.oLeft);

        return oNode;

    def transformCodePass2_BinaryOp_DotAtomAndValueOrInt(self, oNode): # pylint: disable=invalid-name
        """ Pass 2: PSTATE.SP == '0' and suchlike. """
        if len(oNode.oLeft.aoValues) == 2:
            (fValue, _, fWildcard, cBitsWidth) = oNode.oRight.getValueDetails();

            sField = oNode.oLeft.aoValues[1].getIdentifierName();
            if isinstance(sField, str) and sField:
                if oNode.oLeft.aoValues[0].isMatchingIdentifier('PSTATE'):
                    tInfo = self.kdPstateFields.get(sField);
                    if not tInfo:
                        raise Exception('Unexpected PSTATE field in binary op: %s (%s)' % (oNode.oLeft, oNode,));
                    if tInfo[0] != cBitsWidth:
                        raise Exception('Mismatching PSTATE field width and value: %s vs %s (%s)'
                                        % (oNode.oLeft, oNode.oRight, oNode,));

                    sExpr = '(pVCpu->cpum.GstCtx.fPState & UINT64_C(%#x)/*%s*/)' % (((1 << tInfo[0]) - 1) << tInfo[1], sField,);
                    if tInfo[0] == 1:
                        if fWildcard:
                            raise Exception('Wildcard value for single bit PSTATE field: %s (%s)' % (oNode.oLeft, oNode,));
                        if (oNode.sOp == '==' and fValue == 1) or (oNode.sOp == '!=' and fValue == 0):
                            return ArmAstCppExpr(sExpr, cBitsWidth = 1);
                        if (oNode.sOp == '==' and fValue == 0) or (oNode.sOp == '!=' and fValue == 1):
                            return ArmAstUnaryOp('!', ArmAstCppExpr(sExpr, cBitsWidth = 1));
                    elif fWildcard == 0:
                        raise Exception('Handle wildcard value for PSTATE field: %s (%s)' % (oNode.oLeft, oNode,));
                    if tInfo[1] != 0:
                        sExpr = '(%s >> %u)' % (sExpr, tInfo[1],);
                    oNode.oLeft  = ArmAstCppExpr(sExpr, cBitsWidth = tInfo[1]);
                    oNode.oRight = ArmAstInteger(fValue, cBitsWidth);
                    return oNode;
        raise Exception('Unexpected dot-atom + value binary op: %s' % (oNode,));

    def transformCodePass2_BinaryOp_InSet(self, oNode): # pylint: disable=invalid-name
        """ Pass 2: iemGetEffHcrEl2NVx(pVCpu, pGstFeats) IN ('1x1') and such. """
        oSet = oNode.oRight;
        if isinstance(oSet, ArmAstSet):
            for iValue, oValue in enumerate(oSet.aoValues):
                if not isinstance(oValue, (ArmAstValue, ArmAstInteger)):
                    raise Exception('Set value #%u is not an Values.Value node: %s' % (iValue, oNode.toString(),));
        elif isinstance(oSet, ArmAstValue):
            oSet = ArmAstSet([oNode.oRight,]);
        else:
            raise Exception('Unexpected use of operator "IN": %s' % (oNode.toString(),));

        if len(oSet.aoValues) == 1:
            (fValue, fFixed, fWildcard, cBitsWidth) = oSet.aoValues[0].getValueDetails();
            if fFixed == 0:
                raise Exception('Bogus wildcard set expression: %s' % (oNode.toString(),));

            if fWildcard != 0:
                oNode.oLeft = ArmAstBinaryOp(oNode.oLeft, 'AND', ArmAstInteger(fFixed, cBitsWidth))
            oNode.sOp = '==';
            oNode.oRight = ArmAstInteger(fValue, cBitsWidth);
            return oNode

        assert False, str(oNode);
        return oNode;

    def transformCodePass2_Field(self, oNode): # (ArmAstField) -> ArmAstBase
        """ Pass2: Deal with field accesses (except for AST.Concat). """
        # Most fields can be mapped directly to a CPUMCTX field.
        (_, oField, oAccessExprOrInt) = self.lookupRegisterField(oNode.sState, oNode.sName, oNode.sField, 'Generic field',
                                                                 fWarnOnly = True);
        if oField:
            assert len(oField.aoRanges) == 1;
            return VBoxAstCppField.fromFieldDetailsAndAccessExprOrInt(oField.aoRanges[0].iFirstBit,
                                                                      oField.aoRanges[0].cBitsWidth,
                                                                      oField.sName,
                                                                      oAccessExprOrInt);
        return oNode;

    def transformCodePass2_Identifier(self, oNode): # type: (ArmAstIdentifier, SysRegAccessorInfo)
        """ Pass 2: Deal with register identifiers during assignments. """
        sState = 'AArch64' if self.isA64Instruction() else 'AArch32';
        oReg   = spec.g_ddoAllArmRegistersByStateByName[sState].get(oNode.sName); # ArmRegister
        if not oReg:
            raise Exception('Assignment/Identifier: Register "%s" not found' % (oNode.sName, ));

        tCpumCtxInfo = g_dRegToCpumCtx.get('%s.%s' % (sState, oNode.sName));
        if not tCpumCtxInfo:
            # If this is an ID register, we use a helper getter function to access it (assuming RES0).
            for oAccessor in oReg.aoAccessors:
                if isinstance(oAccessor, spec.ArmAccessorSystem):
                    if oAccessor.oEncoding.sAsmValue == oNode.sName:
                        if self.isIdRegisterEncoding(oAccessor.oEncoding):
                            return ArmAstCppCall('iemCImplHlpGetIdSysReg',
                                                 [ ArmAstCppExpr('pVCpu'),
                                                   ArmAstCppExpr(oAccessor.oEncoding.getSysRegIdCreate()), ],
                                                 cBitsWidth = 64);

            oXcpt = Exception('Assignment/Identifier: No CPUMCTX mapping for register %s.%s' % (sState, oNode.sName,));
            return self.warnXcpt(oXcpt, oNode);
            #raise oXcpt;

        if isinstance(tCpumCtxInfo[0], int):
            return ArmAstInteger(tCpumCtxInfo[0], 64);
        return ArmAstCppExpr('pVCpu->cpum.GstCtx.%s' % (tCpumCtxInfo[0],), cBitsWidth = 64);

    def transformCodePass2_SlicedIdentifier(self, oNode):
        """ Pass 2: Deal with '*puDst = TTBR1_EL1[[0x3f:0]]' """
        assert len(oNode.aoValues) == 1 and isinstance(oNode.aoValues[0], ArmAstSlice);
        oSlice = oNode.aoValues[0] # type: ArmAstSlice
        iFrom  = oSlice.oFrom.getIntegerOrValue();
        iTo    = oSlice.oTo.getIntegerOrValue();
        if iFrom is not None and iTo is not None:
            if 0 <= iTo <= iFrom < 64:
                oVar = self.transformCodePass2_Identifier(oNode.oVar);
                if oVar is not oNode.oVar:
                    # A typical use it to slice out the lower 64 bits of a 128 bit system register.
                    if iTo == 0 and iFrom == 63:
                        sExpr = oVar.toString();
                        assert sExpr.endswith('u64');
                    else:
                        sExpr = '(%s & UINT64_C(%#x))' % (oVar.toString(), ((1 << (iFrom - iTo + 1)) - 1) << iTo);
                        if iTo != 0:
                            sExpr = '(%s >> %u)' % (sExpr, iTo,);
                    return ArmAstCppExpr(sExpr, cBitsWidth = iFrom - iTo + 1); # Width is important for sign & zero extending.
        return oNode;

    def transformCodePass2_ParseNVMem(self, oNode):
        if len(oNode.aoValues) in (1, 2):
            off = oNode.aoValues[0].getIntegerOrValue();
            if off is not None and 0 <= off < 4096:
                cBits = 64 if len(oNode.aoValues) == 1 else oNode.aoValues[1].getIntegerOrValue();
                if cBits is not None and cBits in (64, 128):
                    return (off, cBits);
        raise Exception('Bogus NVMem access: %s' % (oNode,));

    def transformCodePass2_AssignFromNVMem(self, oNode):
        """ Pass 2: Translate NVMem[off[,bits]] to function call. """
        (off, cBits) = self.transformCodePass2_ParseNVMem(oNode.oValue);

        # Assume the variable receiving the assignment is *puDst or similar:
        sValuePtr = oNode.oVar.toString();
        if sValuePtr[0] != '*':
            return oNode;
        sValuePtr = sValuePtr[1:];

        return ArmAstReturn(ArmAstCppCall('iemCImplHlpNVMemReadU%u' % (cBits,),
                                          [ ArmAstCppExpr('pVCpu'), ArmAstInteger(off, 32), ArmAstCppExpr(sValuePtr),]));

    def transformCodePass2_AssignToNVMem(self, oNode):
        """ Pass 2: Translate NVMem[off[,bits]] to function call. """
        (off, cBits) = self.transformCodePass2_ParseNVMem(oNode.oVar);
        return ArmAstReturn(ArmAstCppCall('iemCImplHlpNVMemWriteU%u' % (cBits,),
                                          [ ArmAstCppExpr('pVCpu'), ArmAstInteger(off, 32), oNode.oValue,]));

    def transformCodePass2_Assign_Read_DBGDTR_EL0(self, oNode): # pylint: disable=invalid-name
        """ Pass 2: Translates Read_DBGDTR_EL0() to a helper call. """
        if len(oNode.oValue.aoArgs) == 1:
            cBits = oNode.oValue.aoArgs[0].getIntegerOrValue();
            if cBits in (32, 64):
                # Assume the variable receiving the assignment is *puDst or similar:
                sValuePtr = oNode.oVar.toString();
                if sValuePtr[0] == '*':
                    sValuePtr = sValuePtr[1:];
                    return ArmAstReturn(ArmAstCppCall('iemCImplHlpReadDbgDtrEl0U%u' % (cBits,),
                                                      [ ArmAstCppExpr('pVCpu'), ArmAstCppExpr(sValuePtr), ]));
        raise Exception('Unexpected: %s' % (oNode.toString(),));

    def getRegisterInfo(self, sRegName, fRead, oInfo):
        """
        Pass 2 helper that resolves a register name to a C constant, function call or _novar arg.
        Returns (oAccessAst|None, sConstant|None).
        """
        #
        # Lookup the register and check for a direct CPUMCTX mapping.
        #
        sState = self.getRegStateName();
        oReg   = spec.g_ddoAllArmRegistersByStateByName[sState].get(sRegName); # ArmRegister
        if not oReg:
            raise Exception('Assignment/Identifier: Register "%s" not found' % (sRegName, ));

        tCpumCtxInfo = g_dRegToCpumCtx.get('%s.%s' % (sState, sRegName));
        if tCpumCtxInfo and (fRead or not isinstance(tCpumCtxInfo[0], int)):
            if isinstance(tCpumCtxInfo[0], int):
                return (ArmAstInteger(tCpumCtxInfo[0], 64), None);
            return (ArmAstCppExpr('pVCpu->cpum.GstCtx.%s' % (tCpumCtxInfo[0],), cBitsWidth = 64), None);

        #
        # No direct mapping. Maybe this is an ID register, otherwise we return
        # the encoding info for use in a call to _novar.
        #
        for oAccessor in oReg.aoAccessors:
            if isinstance(oAccessor, spec.ArmAccessorSystem):
                if oAccessor.oEncoding.sAsmValue == sRegName:
                    if self.isIdRegisterEncoding(oAccessor.oEncoding):
                        if not fRead:
                            raise Exception('ID register used in assignment! %s' % (sRegName,));
                        return (ArmAstCppCall('iemCImplHlpGetIdSysReg',
                                              [ ArmAstCppExpr('pVCpu'),
                                                ArmAstCppExpr(oAccessor.oEncoding.getSysRegIdCreate()), ],
                                              cBitsWidth = 64), None);
                    try:
                        return (None, oAccessor.oEncoding.getSysRegIdCreate());
                    except Exception:
                        pass;

        # As a last restort, see if it matches the oInfo.sEnc & sAsmValue.
        if oInfo.sAsmValue == sRegName:
            return (None, oInfo.sEnc);
        return self.warn('Assignment %s Identifier: Unable to map %s.%s to anything'
                         % ('from' if fRead else 'to', sState, sRegName,), (None, None));

    class AssignField(object):
        def __init__(self, iFirstBit, cBits, iSrcBit):
            self.iDstBit   = iFirstBit;
            self.cBits     = cBits;
            self.iSrcBit   = iSrcBit;

        def nextBit(self):
            return self.iDstBit + self.cBits;

    koReIdentifier = re.compile(r'^[a-zA-Z_][a-zA-Z0-9_]*$');

    def transformCodePass2_AssignToFieldsCommon(self, sDstVar, aoFields, oSrcVal):
        """ Worker for transformCodePass2_AssignToConcat and transformCodePass2_AssignToDotAtom. """
        #
        # Process the fields. a little.
        #
        cTotalBits = 0;
        fDstMask   = 0;
        for oField in aoFields:
            cTotalBits += oField.cBits;
            fDstMask   |= ((1 << oField.cBits) - 1) << oField.iDstBit;
        fDstMask = ~fDstMask & 0xffffffffffffffff;
        assert 0 < fDstMask <= 0xffffffffffffffff;

        #
        # Check out the value, in case we've got some SquareOp + Slice fun we
        # can use to simplify the translation.
        #
        if isinstance(oSrcVal, ArmAstSquareOp) and len(oSrcVal.aoValues) == 1:
            if isinstance(oSrcVal.aoValues[0], ArmAstSlice):
                oSlice = oSrcVal.aoValues[0];
                iFrom  = oSlice.oFrom.getIntegerValue();
                iTo    = oSlice.oTo.getIntegerValue();
                if iFrom is not None and iTo is not None and 0 <= iTo < iFrom < 64 and iFrom - iTo + 1 == cTotalBits:
                    for oField in aoFields:
                        oField.iSrcBit += iTo;
                    oSrcVal = oSrcVal.oVar;
                else:
                    self.warn('Bad oSlice=%s; iFrom=%s iTo=%s cTotalBits=%s' % (oSlice, iFrom, iTo, cTotalBits,));
            else:
                iTo = oSrcVal.aoValues[0].getIntegerValue();
                if iTo is not None and 0 <= iTo <= 64 and cTotalBits == 1:
                    assert len(aoFields) == 1
                    aoFields[0].iSrcBit += iTo;
                    oSrcVal = oSrcVal.oVar;
                else:
                    self.warn('Odd squareop=%s; iTo=%s cTotalBits=%s' % (oSrcVal.aoValues[0], iTo, cTotalBits,));

        #
        # Simple approach.
        #
        fSingleStmt = (    isinstance(oSrcVal, ArmAstCppExpr)
                       and oSrcVal.cBitsWidth == 64
                       and self.koReIdentifier.match(oSrcVal.sExpr));
        sValName    = 'uTmpVal' if not fSingleStmt else oSrcVal.sExpr;

        aoOrList = [];
        for oField in aoFields:
            oExtract = ArmAstBinaryOp(ArmAstCppExpr(sValName),
                                      'AND', ArmAstInteger(((1 << oField.cBits) - 1) << oField.iSrcBit, 64));
            if oField.iDstBit != oField.iSrcBit:
                oExtract = ArmAstBinaryOp(oExtract, '<<' if oField.iDstBit > oField.iSrcBit else '>>',
                                          ArmAstInteger(abs(oField.iDstBit - oField.iSrcBit), 32));
            aoOrList.append(oExtract);
        oAssignStmt = ArmAstAssignment(ArmAstCppExpr(sDstVar),
                                       ArmAstBinaryOp(ArmAstBinaryOp(ArmAstCppExpr(sDstVar), 'AND', ArmAstInteger(fDstMask)),
                                                      'OR', ArmAstBinaryOp.listToTree(aoOrList, 'OR')));
        if fSingleStmt:
            return oAssignStmt;
        return ArmAstStatementList([
            ArmAstAssignment(ArmAstCppExpr('uint64_t const uTmpVal', 64), oSrcVal),
            oAssignStmt,
        ]);


    def transformCodePass2_AssignToConcat(self, oNode):
        """ Pass 2: Assignment to a concatenation of fields (e.g. MSR NZCV, x0). """
        #
        # Check out the concatenation first.
        #
        sDstVar  = 'pVCpu->cpum.GstCtx.fPState';
        aoFields = [] # type: List[SysRegGeneratorBase.AssignField]
        iSrcBit  = 0;
        for oValue in reversed(oNode.oVar.aoValues):
            if isinstance(oValue, ArmAstDotAtom) and len(oValue.aoValues) == 2:
                sField = oValue.aoValues[1].getIdentifierName();
                if sField and oValue.aoValues[0].isMatchingIdentifier('PSTATE'):
                    (cBits, iFirstBit) = self.kdPstateFields[sField];
                    if aoFields and aoFields[-1].nextBit() == iFirstBit:
                        aoFields[-1].cBits += cBits;
                    else:
                        aoFields.append(SysRegGeneratorBase.AssignField(iFirstBit, cBits, iSrcBit));
                    iSrcBit += cBits;
                else:
                    assert False;
                    return oNode;
            else:
                assert False;
                return oNode;
        if not aoFields:
            assert False;
            return None;
        return self.transformCodePass2_AssignToFieldsCommon(sDstVar, aoFields, oNode.oValue);


    def transformCodePass2_AssignToDotAtom(self, oNode):
        """ Pass 2: Assignment to a single field (e.g. MSR DIT, x0). """
        oDotAtom = oNode.oVar;
        if len(oDotAtom.aoValues) == 2:
            if oDotAtom.aoValues[0].isMatchingIdentifier('PSTATE'):
                sField = oDotAtom.aoValues[1].getIdentifierName();
                if sField:
                    (cBits, iFirstBit) = self.kdPstateFields[sField];
                    return self.transformCodePass2_AssignToFieldsCommon('pVCpu->cpum.GstCtx.fPState',
                                                                        [SysRegGeneratorBase.AssignField(iFirstBit, cBits, 0),],
                                                                        oNode.oValue);
        return oNode;

    def transformCodePass2_AssignFromIdentifier(self, oNode, oInfo): # pylint: disable=invalid-name
        """ Pass 2: Assignment with an system register identifier on the right side (MRS). """
        sRegName = oNode.oValue.getIdentifierName();
        (oAccessAst, sRegConst) = self.getRegisterInfo(sRegName, True, oInfo);
        if oAccessAst:
            oNode.oValue = oAccessAst;
            return oNode;
        if not sRegConst:
            return oNode;

        # Assume the variable receiving the assignment is *puDst or similar:
        sValuePtr = oNode.oVar.toString();
        if sValuePtr[0] != '*':
            assert False;
            return oNode;
        sValuePtr = sValuePtr[1:];

        oInfo.cAssignmentsWithUnresolvedRegs += 1;
        return ArmAstReturn(ArmAstCppCall(self.sFuncPrefix + '_novar',
                                          [ ArmAstCppExpr('pVCpu'),
                                            ArmAstCppExpr(sRegConst),
                                            ArmAstCppExpr('"%s"' % (sRegName,)),
                                            ArmAstCppExpr(sValuePtr),
                                            ArmAstCppExpr('uInstrEssence'), ]));

    def transformCodePass2_AssignToIdentifier(self, oNode, oInfo): # pylint: disable=invalid-name
        """ Pass 2: Assignment with an system register identifier on the left side (MSR). """
        sRegName = oNode.oVar.getIdentifierName();
        (oAccessAst, sRegConst) = self.getRegisterInfo(sRegName, False, oInfo);
        if oAccessAst:
            assert not isinstance(oAccessAst, ArmAstInteger);
            oNode.oVar = oAccessAst;
            return oNode;
        if not sRegConst:
            return oNode;

        oInfo.cAssignmentsWithUnresolvedRegs += 1;
        return ArmAstReturn(ArmAstCppCall(self.sFuncPrefix + '_novar',
                                          [ ArmAstCppExpr('pVCpu'),
                                            ArmAstCppExpr(sRegConst),
                                            ArmAstCppExpr('"%s"' % (sRegName,)),
                                            oNode.oValue,
                                            ArmAstCppExpr('uInstrEssence'), ]));



    koReSkipIdentifierInBinOp = re.compile(r'^EL\d$');

    def transformCodePass2Callback(self, oNode, fEliminationAllowed, oInfo, aoStack):
        """ Callback for pass 2: C++ translation. """
        if isinstance(oNode, ArmAstFunction):
            # Undefined() -> return iemRaiseUndefined(pVCpu);
            if oNode.isMatchingFunctionCall('Undefined'):
                return ArmAstReturn(ArmAstCppCall('iemRaiseUndefined', [ArmAstCppExpr('pVCpu')]));
            if oNode.isMatchingFunctionCall('EXLOCKException'):
                return ArmAstReturn(ArmAstCppCall('iemRaiseExlockException', [ArmAstCppExpr('pVCpu')]));

            # IsFeatureImplemented(FEAT_xxxx) -> pGstFeat->fXxxx:
            if oNode.sName == 'IsFeatureImplemented':
                return self.transformCodePass2_IsFeatureImplemented(oNode);

            # Effective<register> getter functions:
            if oNode.sName == 'EffectiveHCR_EL2_NVx':
                return self.transformCodePass2_EffectiveHCR_EL2_NVx(oNode);
            if oNode.sName == 'EffectiveACTLRMASK_EL1':
                return self.transformCodePass2_EffectiveACTLRMASK_EL1(oNode);
            if oNode.sName == 'EffectiveCPACRMASK_EL1':
                return self.transformCodePass2_EffectiveCPACRMASK_EL1(oNode);
            if oNode.sName == 'EffectiveCPTRMASK_EL2':
                return self.transformCodePass2_EffectiveCPTRMASK_EL2(oNode);
            if oNode.sName == 'EffectiveSCTLRMASK_EL1':
                return self.transformCodePass2_EffectiveSCTLRMASK_EL1(oNode);
            if oNode.sName == 'EffectiveSCTLRMASK_EL2':
                return self.transformCodePass2_EffectiveSCTLRMASK_EL1(oNode);

            if oNode.sName == 'GetCurrentEXLOCKEN':
                return self.transformCodePass2_GetCurrentEXLOCKEN(oNode);

            if oNode.sName == 'PhysicalCountInt':
                return self.transformCodePass2_PhysicalCountInt(oNode);

            if oNode.sName == 'SecurityStateAtEL':
                return self.transformCodePass2_SecurityStateAtEL(oNode);

            if oNode.sName == 'GCSEnabled':
                return self.transformCodePass2_GCSEnabled(oNode);

            # UInt.
            if oNode.sName == 'UInt':
                return self.transformCodePass2_UInt(oNode);

            # Zeros.  Don't convert in concats as we want to handle those specially.
            if oNode.sName == 'Zeros' and (not aoStack or not isinstance(aoStack[-1], ArmAstConcat)):
                return self.transformCodePass2_Zeros(oNode);
            if oNode.sName == 'ZeroExtend':
                return self.transformCodePass2_ZeroExtend(oNode, oInfo);
            if oNode.sName == 'SignExtend':
                return self.transformCodePass2_SignExtend(oNode);

        elif isinstance(oNode, ArmAstBinaryOp):
            # PSTATE.EL == EL0 and similar:
            if oNode.oLeft.isMatchingDotAtom('PSTATE', 'EL'):
                idxEl = self.kdELxToNum.get(oNode.oRight.getIdentifierName(), -1);
                if idxEl >= 0:
                    oNode.oLeft  = ArmAstCppExpr('IEM_F_MODE_ARM_GET_EL(pVCpu->iem.s.fExec)', cBitsWidth = 2);
                    oNode.oRight = ArmAstInteger(idxEl, cBitsWidth = 2);
                return oNode;

            ## (AArch64.MDCR_EL2.TDE):(AArch64.MDCR_EL2.TDA) != '00' and similar:
            if (    isinstance(oNode.oLeft, self.VBoxAstCppConcat)
                and isinstance(oNode.oRight, (ArmAstValue, ArmAstInteger))
                and ArmAstBinaryOp.kdOps[oNode.sOp] in (ArmAstBinaryOp.ksOpTypeLogical, ArmAstBinaryOp.ksOpTypeCompare) ):
                return self.transformCodePass2_BinaryOp_ConcatAndValueOrInt(oNode);

            # PSTATE.SP == '0' and similar:
            if isinstance(oNode.oLeft, ArmAstDotAtom) and isinstance(oNode.oRight, (ArmAstValue, ArmAstInteger)):
                return self.transformCodePass2_BinaryOp_DotAtomAndValueOrInt(oNode);

            ## iemGetEffHcrEl2NVx(pVCpu, pGstFeats) IN ('1x1') and such fun stuff.
            if oNode.sOp == 'IN':
                return self.transformCodePass2_BinaryOp_InSet(oNode);

            # Drop unnecessary field shifting when and compares for non-zero field checks.
            if isinstance(oNode.oLeft, VBoxAstCppField):
                if (   (oNode.sOp == '==' and oNode.oRight.isMatchingIntegerOrValue(1) and oNode.oLeft.cBitsWidth == 1)
                    or (oNode.sOp == '!=' and oNode.oRight.isMatchingIntegerOrValue(0))):
                    return oNode.oLeft.dropShift();
                if (oNode.sOp == '==' and oNode.oRight.isMatchingIntegerOrValue(0)):
                    return ArmAstUnaryOp('!', oNode.oLeft.dropShift());
                if ArmAstBinaryOp.kdOps[oNode.sOp] == ArmAstBinaryOp.ksOpTypeLogical:
                    oNode.oLeft = oNode.oLeft.dropShift();
            elif isinstance(oNode.oRight, VBoxAstCppField):
                if ArmAstBinaryOp.kdOps[oNode.sOp] == ArmAstBinaryOp.ksOpTypeLogical:
                    oNode.oRight = oNode.oRight.dropShift();

        elif (    isinstance(oNode, ArmAstConcat)
              and (   not aoStack                                    # Not when on the left side of an assignment, please,
                   or not isinstance(aoStack[-1], ArmAstAssignment)  # that'll require special handling.
                   or oNode is not aoStack[-1].oVar)):
            return self.transformCodePass2_Concat(oNode);

        elif (    isinstance(oNode, ArmAstField)
              and (   not aoStack
                   or not isinstance(aoStack[-1], ArmAstConcat))):
            return self.transformCodePass2_Field(oNode);

        # Identifiers are tricky, as we don't want to prematurely process them in assignments,
        # concatenations, field access, dot-atoms and such. Thus the limited parent node and
        # name filtering hacks.
        elif (    isinstance(oNode, ArmAstIdentifier)
              and aoStack
              and (   isinstance(aoStack[-1], ArmAstUnaryOp)
                   or (isinstance(aoStack[-1], ArmAstBinaryOp) and not self.koReSkipIdentifierInBinOp.match(oNode.sName)) ) ):
            return self.transformCodePass2_Identifier(oNode);

        elif isinstance(oNode, ArmAstIfList):
            # Drop double parentheses around field extraction expressions when they are the sole if condition.
            for idxIfCond, oIfCond in enumerate(oNode.aoIfConditions):
                if isinstance(oIfCond, VBoxAstCppField):
                    oNode.aoIfConditions[idxIfCond] = oIfCond.dropExtraParenthesis();

        elif isinstance(oNode, ArmAstSquareOp):
            # Deal with '*puDst = TTBR1_EL1[[0x3f:0]]' and 'SignExtend(X[t,0x40][[0x1f:0]], 0x40)'.
            if (    isinstance(oNode.oVar, ArmAstIdentifier)
                and len(oNode.aoValues) == 1
                and isinstance(oNode.aoValues[0], ArmAstSlice)
                and aoStack
                and isinstance(aoStack[-1], (ArmAstAssignment, ArmAstFunction))):
                return self.transformCodePass2_SlicedIdentifier(oNode);

            # Weird VMID[] var/function.
            if oNode.oVar.isMatchingIdentifier('VMID') and len(oNode.aoValues) == 0:
                return ArmAstCppCall('iemCImplHlpGetCurrentVmId', [ArmAstCppExpr('pVCpu'),]);


        elif isinstance(oNode, ArmAstAssignment):
            if isinstance(oNode.oVar, ArmAstSquareOp):
                if oNode.oVar.oVar.isMatchingIdentifier('NVMem'):
                    return self.transformCodePass2_AssignToNVMem(oNode);
            elif isinstance(oNode.oValue, ArmAstSquareOp) and oNode.oValue.oVar.isMatchingIdentifier('NVMem'):
                return self.transformCodePass2_AssignFromNVMem(oNode);
            elif isinstance(oNode.oValue, ArmAstFunction):
                if oNode.oValue.sName == 'Read_DBGDTR_EL0':
                    return self.transformCodePass2_Assign_Read_DBGDTR_EL0(oNode);

            if isinstance(oNode.oVar, ArmAstConcat):  # MSR NZCV, x0
                return self.transformCodePass2_AssignToConcat(oNode);
            if isinstance(oNode.oVar, ArmAstDotAtom): # MSR DIT, x0
                return self.transformCodePass2_AssignToDotAtom(oNode);
            if isinstance(oNode.oVar, ArmAstIdentifier):
                return self.transformCodePass2_AssignToIdentifier(oNode, oInfo);
            if isinstance(oNode.oValue, ArmAstIdentifier):
                return self.transformCodePass2_AssignFromIdentifier(oNode, oInfo);

        _ = fEliminationAllowed;
        return oNode;

    def transformCodePass2(self, oInfo, oCode):
        """ Code transformation pass 2: C++ translation. """
        return oCode.transform(self.transformCodePass2Callback, True, oInfo, []);


class SysRegGeneratorA64Mrs(SysRegGeneratorBase):
    """
    Code generator specialization for the MRS (read system register) instruction.
    """

    def __init__(self):
        SysRegGeneratorBase.__init__(self, 'A64.MRS', 'iemCImplA64_mrs', 'uint64_t *', 'puDst', 'read');

    def transformCodePass2Callback(self, oNode, fEliminationAllowed, oInfo, aoStack):
        """ Callback used by the second pass."""
        if oNode.isMatchingSquareOp('X', 't', 64) or oNode.isMatchingSquareOp('X', 't', 32):
            return ArmAstCppExpr('*puDst', cBitsWidth = 64);

        return super().transformCodePass2Callback(oNode, fEliminationAllowed, oInfo, aoStack);


class SysRegGeneratorA64MsrReg(SysRegGeneratorBase):
    """
    Code generator specialization for the MSR (set system register) instruction.
    """

    def __init__(self):
        SysRegGeneratorBase.__init__(self, 'A64.MSRregister', 'iemCImplA64_msr', 'uint64_t', 'uValue', 'write');

    def transformCodePass2Callback(self, oNode, fEliminationAllowed, oInfo, aoStack):
        """ Callback used by the second pass."""
        # Replace X[t,64] references with uValue.
        if oNode.isMatchingSquareOp('X', 't', 64):# or oNode.oVar.isMatchingSquareOp('X', 't', 32):
            if aoStack and isinstance(aoStack[-1], (ArmAstAssignment, ArmAstBinaryOp, ArmAstSquareOp)):
                return ArmAstCppExpr('uValue', cBitsWidth = 64);

        # Return statements w/o a value are NOP branches, make them return VINF_SUCCESS.
        elif isinstance(oNode, ArmAstReturn):
            if not oNode.oValue:
                return ArmAstReturn(ArmAstCppExpr('iemRegPcA64IncAndFinishingClearingFlags(pVCpu, VINF_SUCCESS)', 32));

        return super().transformCodePass2Callback(oNode, fEliminationAllowed, oInfo, aoStack);


class SysRegGeneratorA64MsrImmediate(SysRegGeneratorBase):
    """
    Code generator specialization for the MSR (set system register) instruction.
    """

    def __init__(self):
        SysRegGeneratorBase.__init__(self, 'A64.MSRimmediate', 'iemCImplA64_msr', 'uint64_t', 'uValue', 'write');

    def transformCodePass2Callback(self, oNode, fEliminationAllowed, oInfo, aoStack):
        """ Callback used by the second pass."""
        # Replace X[t,64] references with uValue.
        if oNode.isMatchingSquareOp('X', 't', 64):# or oNode.oVar.isMatchingSquareOp('X', 't', 32):
            if aoStack and isinstance(aoStack[-1], (ArmAstAssignment, ArmAstBinaryOp, ArmAstSquareOp)):
                return ArmAstCppExpr('uValue', cBitsWidth = 64);

        # Return statements w/o a value are NOP branches, make them return VINF_SUCCESS.
        elif isinstance(oNode, ArmAstReturn):
            if not oNode.oValue:
                return ArmAstReturn(ArmAstCppExpr('iemRegPcA64IncAndFinishingClearingFlags(pVCpu, VINF_SUCCESS)', 32));

        return super().transformCodePass2Callback(oNode, fEliminationAllowed, oInfo, aoStack);


class SysRegGeneratorA64MsrrReg(SysRegGeneratorBase):
    """
    Code generator specialization for the MSRR (set 128-bit system register) instruction.
    """

    def __init__(self):
        SysRegGeneratorBase.__init__(self, 'A64.MSRRregister', 'iemCImplA64_msrr', 'PCRTUINT128U', 'puuValue', 'write');

    def transformCodePass2Callback(self, oNode, fEliminationAllowed, oInfo, aoStack):
        """ Callback used by the second pass."""
        ## @todo this needs lots more work and real 128-bit register variables.
        # Replace X[t,64] references with uValue.
        if oNode.isMatchingSquareOp('X', 't', 64):# or oNode.oVar.isMatchingSquareOp('X', 't', 32):
            if aoStack and isinstance(aoStack[-1], (ArmAstAssignment, ArmAstBinaryOp, ArmAstSquareOp)):
                return ArmAstCppExpr('uValue', cBitsWidth = 64);

        # Return statements w/o a value are NOP branches, make them return VINF_SUCCESS.
        elif isinstance(oNode, ArmAstReturn):
            if not oNode.oValue:
                return ArmAstReturn(ArmAstCppExpr('iemRegPcA64IncAndFinishingClearingFlags(pVCpu, VINF_SUCCESS)', 32));

        return super().transformCodePass2Callback(oNode, fEliminationAllowed, oInfo, aoStack);


class SysRegGeneratorA64Sys(SysRegGeneratorBase):
    """
    Code generator specialization for the SYS instruction.
    """

    def __init__(self, sInstr = 'A64.SYS', sFuncPrefix = 'iemCImplA64_sys', sParamType = 'uint64_t', sParamName = 'uValue'):
        SysRegGeneratorBase.__init__(self, sInstr, sFuncPrefix, sParamType, sParamName, 'write');

    def getDescriptionFromInfo(self, oInfo):
        return '%s %s - %s' % (oInfo.oAccessor.sName, oInfo.oAccessor.oEncoding.sAsmValue or '',
                               oInfo.oAccessor.oEncoding.dNamedValues,);

    #
    # Pass 2 function rewriters.
    #

    kdPass2FunctionHandlersEx = {
        # AArch64_AT(uValue, TranslationStage_1, EL1, ATAccess_Read)
        'AArch64_AT':           (('iemCImplHlpA64SysAt', [ArmAstCppExprBase, kdMapTranslationStage, kdMapELx, kdMapATAccess,]), ),
        # AArch64_DC(uValue, CacheType_Data, CacheOp_Clean, CacheOpScope_SetWay)
        'AArch64_DC':           (('iemCImplHlpA64SysDc', [ArmAstCppExprBase, kdMapCacheType, kdMapCacheOp, kdMapCacheOpScope,]),),
        # Arch64_IC(CacheOpScope_ALLUIS)
        # Arch64_IC(uValue, CacheOpScope_PoU)
        'AArch64_IC':           (('iemCImplHlpA64SysIc', [kdMapCacheOpScope,]),
                                 ('iemCImplHlpA64SysIcWithArg', [ArmAstCppExprBase, kdMapCacheOpScope,]), ),
        # AArch64_MemZero(uValue, CacheType_Data)
        'AArch64_MemZero':       (('iemCImplHlpA64MemZero', [ArmAstCppExprBase, kdMapCacheType,]), ),
        # AArch64_RestrictPrediction(uValue, RestrictType_DataValue)
        'AArch64_RestrictPrediction':   (('iemCImplHlpA64RestrictPrediction', [ArmAstCppExprBase, kdMapRestrictType,]),),
        # AArch64_TLBI_ALL(iemCImplHlpGetSecurityStateAtEl(pVCpu, 2), Regime_EL2, Broadcast_OSH, TLBI_AllAttr, uValue)
        'AArch64_TLBI_ALL':      (('iemCImplHlpA64TlbiAll', [ArmAstCppCall, kdMapRegime, kdMapBroadcast,
                                                             kdMapTlbiMemAttr, ArmAstCppExprBase,]), ),
        # AArch64_TLBI_ASID(iemCImplHlpGetSecurityStateAtEl(pVCpu, 1), Regime_EL10, VMID<> | VMID_NONE,
        #                   Broadcast_OSH, TLBI_AllAttr, uValue)
        'AArch64_TLBI_ASID':     (('iemCImplHlpA64TlbiAsid', [ArmAstCppCall, kdMapRegime, (kdMapVmId, ArmAstCppCall),
                                                              kdMapBroadcast, kdMapTlbiMemAttr, ArmAstCppExprBase,]),),
        # AArch64_TLBI_IPAS2(iemCImplHlpGetSecurityStateAtEl(pVCpu, 1), Regime_EL10, iemCImplHlpGetCurrentVmId(pVCpu) | VMID_NONE,
        #                     Broadcast_ISH, TLBILevel_Any, TLBI_AllAttr, uValue)
        'AArch64_TLBI_IPAS2':    (('iemCImplHlpA64TlbiIpAs2', [ArmAstCppCall, kdMapRegime, (kdMapVmId, ArmAstCppCall),
                                                               kdMapBroadcast, kdMapTlbiLevel, kdMapTlbiMemAttr,
                                                               ArmAstCppExprBase,]), ),
        # AArch64_TLBI_RIPAS2(iemCImplHlpGetSecurityStateAtEl(pVCpu, 1), Regime_EL10,
        #                     iemCImplHlpGetCurrentVmId(pVCpu) | VMID_NONE, Broadcast_ISH, TLBILevel_Any, TLBI_AllAttr, uValue)
        'AArch64_TLBI_RIPAS2':   (('iemCImplHlpA64TlbiRIpAs2', [ArmAstCppCall, kdMapRegime, (kdMapVmId, ArmAstCppCall),
                                                                kdMapBroadcast, kdMapTlbiLevel, kdMapTlbiMemAttr,
                                                                ArmAstCppExprBase,]), ),
        # AArch64_TLBI_RVA(iemCImplHlpGetSecurityStateAtEl(pVCpu, 2), Regime_EL20,
        #                  iemCImplHlpGetCurrentVmId(pVCpu) | VMID_NONE, Broadcast_ISH, TLBILevel_Any, TLBI_AllAttr, uValue)
        'AArch64_TLBI_RVA':      (('iemCImplHlpA64TlbiRva', [ArmAstCppCall, kdMapRegime, (kdMapVmId, ArmAstCppCall),
                                                             kdMapBroadcast, kdMapTlbiLevel, kdMapTlbiMemAttr,
                                                             ArmAstCppExprBase,]),),
        # AArch64_TLBI_RVAA(iemCImplHlpGetSecurityStateAtEl(pVCpu, 1), Regime_EL10,
        #                   iemCImplHlpGetCurrentVmId(pVCpu) | VMID_NONE, Broadcast_ISH, TLBILevel_Any, TLBI_AllAttr, uValue)
        'AArch64_TLBI_RVAA':     (('iemCImplHlpA64TlbiRvaa', [ArmAstCppCall, kdMapRegime, (kdMapVmId, ArmAstCppCall),
                                                              kdMapBroadcast, kdMapTlbiLevel, kdMapTlbiMemAttr,
                                                              ArmAstCppExprBase,]), ),
        # AArch64_TLBI_VA(iemCImplHlpGetSecurityStateAtEl(pVCpu, 1), Regime_EL10, iemCImplHlpGetCurrentVmId(pVCpu) | VMID_NONE,
        #                 Broadcast_ISH, TLBILevel_Any, TLBI_AllAttr, uValue)
        'AArch64_TLBI_VA':       (('iemCImplHlpA64TlbiVa', [ArmAstCppCall, kdMapRegime, (kdMapVmId, ArmAstCppCall),
                                                            kdMapBroadcast, kdMapTlbiLevel, kdMapTlbiMemAttr,
                                                            ArmAstCppExprBase,]), ),
        # AArch64_TLBI_VAA(iemCImplHlpGetSecurityStateAtEl(pVCpu, 1), Regime_EL10, iemCImplHlpGetCurrentVmId(pVCpu) | VMID_NONE,
        #                  Broadcast_ISH, TLBILevel_Any, TLBI_AllAttr, uValue)
        'AArch64_TLBI_VAA':      (('iemCImplHlpA64TlbiVaa', [ArmAstCppCall, kdMapRegime, (kdMapVmId, ArmAstCppCall),
                                                             kdMapBroadcast, kdMapTlbiLevel, kdMapTlbiMemAttr,
                                                             ArmAstCppExprBase,]),),
        # AArch64_TLBI_VMALL(iemCImplHlpGetSecurityStateAtEl(pVCpu, 1), Regime_EL10,
        #                    iemCImplHlpGetCurrentVmId(pVCpu) | VMID_NONE, Broadcast_ForcedISH, TLBI_AllAttr, uValue)
        'AArch64_TLBI_VMALL':    (('iemCImplHlpA64TlbiVmAll', [ArmAstCppCall, kdMapRegime, (kdMapVmId, ArmAstCppCall),
                                                               kdMapBroadcast, kdMapTlbiMemAttr, ArmAstCppExprBase,]),),

        # AArch64_TLBI_VMALLS12(iemCImplHlpGetSecurityStateAtEl(pVCpu, 1), Regime_EL10,
        #                       iemCImplHlpGetCurrentVmId(pVCpu) | VMID_NONE, Broadcast_OSH, TLBI_AllAttr, uValue)
        'AArch64_TLBI_VMALLS12': (('iemCImplHlpA64TlbiVmAllS12', [ArmAstCppCall, kdMapRegime, (kdMapVmId, ArmAstCppCall),
                                                                  kdMapBroadcast, kdMapTlbiMemAttr, ArmAstCppExprBase,]),),
        # AArch64_TLBI_VMALLWS2(iemCImplHlpGetSecurityStateAtEl(pVCpu, 1), Regime_EL10,
        #                       iemCImplHlpGetCurrentVmId(pVCpu) | VMID_NONE, Broadcast_ISH, TLBI_AllAttr, uValue)
        'AArch64_TLBI_VMALLWS2': (('iemCImplHlpA64TlbiVmAllWs2', [ArmAstCppCall, kdMapRegime, (kdMapVmId, ArmAstCppCall),
                                                                  kdMapBroadcast, kdMapTlbiMemAttr, ArmAstCppExprBase,]),),
        # AArch64_TRCIT(uValue)
        'AArch64_TRCIT':         (('iemCImplHlpA64TrcIt', [ArmAstCppExprBase,]),),  # 'trace-it'
        # BRB_IALL()
        'BRB_IALL':              (('iemCImplHlpA64BrbIAll', []),),              # branch buffers
        # BRB_INJ()
        'BRB_INJ':               (('iemCImplHlpA64BrbInj', []),),               # branch buffers
        # GCSPOPCX()
        'GCSPOPCX':              (('iemCImplHlpA64GcsPopCX', []),),
        # GCSPOPX()
        'GCSPOPX':               (('iemCImplHlpA64GcsPopX', []),),
        # GCSPUSHM(uValue)
        'GCSPUSHM':              (('iemCImplHlpA64GcsPushM', [ArmAstCppExprBase,]),),
        # GCSPUSHX()
        'GCSPUSHX':              (('iemCImplHlpA64GcsPushX', []),),
        # GCSSS1(uValue)
        'GCSSS1':                (('iemCImplHlpA64GcsSs1', [ArmAstCppExprBase,]),),

        #
        # Additional entries for SYSP:
        #

        # AArch64_TLBIP_IPAS2(iemCImplHlpGetSecurityStateAtEl(pVCpu, 1), Regime_EL10,
        #                     iemCImplHlpGetCurrentVmId(pVCpu) | VMID_NONE, Broadcast_ISH, TLBILevel_Any, TLBI_AllAttr, puValue)
        'AArch64_TLBIP_IPAS2':   (('iemCImplHlpA64TlbipIpAs2', [ArmAstCppCall, kdMapRegime, (kdMapVmId, ArmAstCppCall),
                                                                kdMapBroadcast, kdMapTlbiLevel, kdMapTlbiMemAttr,
                                                                ArmAstCppExprBase,]), ),
        # AArch64_TLBIP_RIPAS2(iemCImplHlpGetSecurityStateAtEl(pVCpu, 1), Regime_EL10,
        #                      iemCImplHlpGetCurrentVmId(pVCpu) | VMID_NONE, Broadcast_ISH, TLBILevel_Any, TLBI_AllAttr, puValue)
        'AArch64_TLBIP_RIPAS2':  (('iemCImplHlpA64TlbipRIpAs2', [ArmAstCppCall, kdMapRegime, (kdMapVmId, ArmAstCppCall),
                                                                 kdMapBroadcast, kdMapTlbiLevel, kdMapTlbiMemAttr,
                                                                 ArmAstCppExprBase,]), ),
        # AArch64_TLBIP_RVA(iemCImplHlpGetSecurityStateAtEl(pVCpu, 2), Regime_EL20,
        #                   iemCImplHlpGetCurrentVmId(pVCpu) | VMID_NONE, Broadcast_ISH, TLBILevel_Any, TLBI_AllAttr, puValue)
        'AArch64_TLBIP_RVA':     (('iemCImplHlpA64TlbipRva', [ArmAstCppCall, kdMapRegime, (kdMapVmId, ArmAstCppCall),
                                                              kdMapBroadcast, kdMapTlbiLevel, kdMapTlbiMemAttr,
                                                              ArmAstCppExprBase,]),),
        # AArch64_TLBIP_RVAA(iemCImplHlpGetSecurityStateAtEl(pVCpu, 1), Regime_EL10,
        #                    iemCImplHlpGetCurrentVmId(pVCpu) | VMID_NONE, Broadcast_ISH, TLBILevel_Any, TLBI_AllAttr, puValue)
        'AArch64_TLBIP_RVAA':    (('iemCImplHlpA64TlbipRvaa', [ArmAstCppCall, kdMapRegime, (kdMapVmId, ArmAstCppCall),
                                                               kdMapBroadcast, kdMapTlbiLevel, kdMapTlbiMemAttr,
                                                               ArmAstCppExprBase,]), ),
        # AArch64_TLBIP_VA(iemCImplHlpGetSecurityStateAtEl(pVCpu, 1), Regime_EL10, iemCImplHlpGetCurrentVmId(pVCpu) | VMID_NONE,
        #                  Broadcast_ISH, TLBILevel_Any, TLBI_AllAttr, puValue)
        'AArch64_TLBIP_VA':      (('iemCImplHlpA64TlbipVa', [ArmAstCppCall, kdMapRegime, (kdMapVmId, ArmAstCppCall),
                                                             kdMapBroadcast, kdMapTlbiLevel, kdMapTlbiMemAttr,
                                                             ArmAstCppExprBase,]), ),
        # AArch64_TLBIP_VAA(iemCImplHlpGetSecurityStateAtEl(pVCpu, 1), Regime_EL10, iemCImplHlpGetCurrentVmId(pVCpu) | VMID_NONE,
        #                   Broadcast_ISH, TLBILevel_Any, TLBI_AllAttr, puValue)
        'AArch64_TLBIP_VAA':     (('iemCImplHlpA64TlbipVaa', [ArmAstCppCall, kdMapRegime, (kdMapVmId, ArmAstCppCall),
                                                              kdMapBroadcast, kdMapTlbiLevel, kdMapTlbiMemAttr,
                                                              ArmAstCppExprBase,]),),
    };

    def transformCodePass2Callback(self, oNode, fEliminationAllowed, oInfo, aoStack):
        """ Callback used by the second pass."""
        # Replace X<t,64> references with uValue
        if oNode.isMatchingSquareOp('X', 't', 64):
            if aoStack and isinstance(aoStack[-1], (ArmAstFunction, ArmAstBinaryOp, ArmAstSquareOp)):
                return ArmAstCppExpr('uValue', cBitsWidth = 64);

        # Return statements w/o a value are NOP branches, make them return VINF_SUCCESS.
        elif isinstance(oNode, ArmAstReturn):
            if not oNode.oValue:
                return ArmAstReturn(ArmAstCppExpr('iemRegPcA64IncAndFinishingClearingFlags(pVCpu, VINF_SUCCESS)', 32));

        # Refactor function calls.
        elif isinstance(oNode, ArmAstFunction):
            if oNode.sName in self.kdPass2FunctionHandlersEx:
                asErrs = [];
                if oNode.isStatement():
                    for i, t in enumerate(self.kdPass2FunctionHandlersEx[oNode.sName]):
                        sCImplHlpName, aoArgTypeMap = t;
                        if len(aoArgTypeMap) == len(oNode.aoArgs):
                            aoCppArgs = [ ArmAstCppExpr('pVCpu'), ];
                            for iArg, aoTypesOrMap in enumerate(aoArgTypeMap):
                                if not isinstance(aoTypesOrMap, tuple):
                                    aoTypesOrMap = (aoTypesOrMap,);
                                for oTypesOrMap in aoTypesOrMap:
                                    if isinstance(oTypesOrMap, dict):
                                        sId = oNode.aoArgs[iArg].getIdentifierName();
                                        if sId and sId in oTypesOrMap:
                                            aoCppArgs.append(ArmAstCppExpr(oTypesOrMap[sId]));
                                            break;
                                        asErrs.append('%s: Mapping of call arg #%u failed: %s' % (i, iArg, oNode.aoArgs[iArg],));
                                    elif isinstance(oNode.aoArgs[iArg], oTypesOrMap):
                                        aoCppArgs.append(oNode.aoArgs[iArg].clone());
                                        break;
                                    else:
                                        asErrs.append('%s: Call arg #%u type mismatch: %s, expected %s'
                                                      % (i, iArg, type(oNode.aoArgs[iArg]), oTypesOrMap,));
                            if len(aoCppArgs) == len(oNode.aoArgs) + 1:
                                return ArmAstReturn(ArmAstCppCall(sCImplHlpName, aoCppArgs, cBitsWidth = 32));
                        else:
                            asErrs.append('%u: Argument count mismatch: %s vs %s' % (i, len(aoArgTypeMap), len(oNode.aoArgs),));
                else:
                    asErrs.append('Not statement!');
                raise Exception('Unexpected call: %s\n%s' % (oNode, '\n'.join(asErrs)))

        return super().transformCodePass2Callback(oNode, fEliminationAllowed, oInfo, aoStack);


class SysRegGeneratorA64SysP(SysRegGeneratorA64Sys):
    """
    Code generator specialization for the SYSP instruction (128-bit value in register pair).
    """

    def __init__(self):
        SysRegGeneratorA64Sys.__init__(self, 'A64.SYSP', 'iemCImplA64_sysp', 'PCRTUINT128U', 'puValue');

    def generateMainFunctionBody(self):
        return [
            '    RTUINT128U const uValue =',
            '    {',
            '        {',
            '            idxGprSrc     < ARMV8_A64_REG_XZR ? pVCpu->cpum.GstCtx.aGRegs[idxGprSrc].x     : 0,',
            '            idxGprSrc + 1 < ARMV8_A64_REG_XZR ? pVCpu->cpum.GstCtx.aGRegs[idxGprSrc + 1].x : 0,',
            '        }',
            '    };',
            '    return %s_generic(pVCpu, idSysReg, idxGprSrc, &uValue);' % (self.sFuncPrefix,),
        ];

    #
    # Pass 2 function rewriters.
    #

    def transformCodePass2Callback(self, oNode, fEliminationAllowed, oInfo, aoStack):
        """ Callback used by the second pass."""
        # Replace X<t/t2,64> references with puValue.s.Lo/Hi
        if isinstance(oNode, ArmAstSquareOp):
            if oNode.isMatchingSquareOp('X', 't', 64):
                if aoStack and isinstance(aoStack[-1], (ArmAstFunction, ArmAstBinaryOp, ArmAstSquareOp)):
                    return ArmAstCppExpr('puValue->s.Lo', cBitsWidth = 64);
            if oNode.isMatchingSquareOp('X', 't2', 64):
                if aoStack and isinstance(aoStack[-1], (ArmAstFunction, ArmAstBinaryOp, ArmAstSquareOp)):
                    return ArmAstCppExpr('puValue->s.Hi', cBitsWidth = 64);

        elif isinstance(oNode, ArmAstConcat):
            if len(oNode.aoValues) == 2:
                if (    oNode.aoValues[0].isMatchingSquareOp('X', 't2', 64)
                    and oNode.aoValues[1].isMatchingSquareOp('X', 't', 64)):
                    if aoStack and isinstance(aoStack[-1], ArmAstFunction):
                        return ArmAstCppExpr('puValue');

        return super().transformCodePass2Callback(oNode, fEliminationAllowed, oInfo, aoStack);


class SysRegGeneratorA64SysL(SysRegGeneratorBase):
    """
    Code generator specialization for the SYSL instruction.
    """

    def __init__(self):
        SysRegGeneratorBase.__init__(self, 'A64.SYSL', 'iemCImplA64_sysl', 'uint64_t *', 'puDst', 'read');

    def getDescriptionFromInfo(self, oInfo):
        return '%s %s - %s' % (oInfo.oAccessor.sName, oInfo.oAccessor.oEncoding.sAsmValue or '',
                               oInfo.oAccessor.oEncoding.dNamedValues,);

    def transformCodePass2Callback(self, oNode, fEliminationAllowed, oInfo, aoStack):
        """ Callback used by the second pass."""
        # Replace X[t,64] references with uValue, but not in assignments (handled below).
        if oNode.isMatchingSquareOp('X', 't', 64):
            if aoStack and isinstance(aoStack[-1], (ArmAstBinaryOp, ArmAstSquareOp)):
                return ArmAstCppExpr('uValue', cBitsWidth = 64);

        # Assignments to uValue
        elif isinstance(oNode, ArmAstAssignment):
            if oNode.oVar.isMatchingSquareOp('X', 't', 64):
                if oNode.oValue.isMatchingFunctionCall('GCSPOPM'):
                    return ArmAstReturn(ArmAstCppCall('iemCImplHlpA64GcsPopM',
                                                      [ArmAstCppExpr('pVCpu'), ArmAstCppExpr('puDst'),], cBitsWidth = 32));
                if oNode.oValue.isMatchingFunctionCall('GCSSS2'):
                    return ArmAstReturn(ArmAstCppCall('iemCImplHlpA64GcsSs2',
                                                      [ArmAstCppExpr('pVCpu'), ArmAstCppExpr('puDst'),], cBitsWidth = 32));
            raise Exception('Unexpected assignment: %s' % (oNode,));

        # Return statements w/o a value are NOP branches, make them return VINF_SUCCESS.
        elif isinstance(oNode, ArmAstReturn):
            if not oNode.oValue:
                return ArmAstReturn(ArmAstCppExpr('iemRegPcA64IncAndFinishingClearingFlags(pVCpu, VINF_SUCCESS)', 32));

        return super().transformCodePass2Callback(oNode, fEliminationAllowed, oInfo, aoStack);


class SysRegGeneratorNop(SysRegGeneratorBase):
    """
    Code generator specialization for the SYS instruction.
    """

    def __init__(self, sInstr, sFuncPrefix, sParamType = 'uint64_t', sParamName = 'uValue'):
        SysRegGeneratorBase.__init__(self, sInstr, sFuncPrefix, sParamType, sParamName);

    def generateOneHandler(self, oInfo):
        return [];

    def generateMainFunction(self):
        return [];


#
# Generators
#

class IEMArmGenerator(object):

    def __init__(self):
        self.oOptions        = None;
        self.dDecoderRoots   = {};
        self.dRootsIndexExpr = {};


    def constructDecoder(self):
        """
        Creates the decoder(s) to the best our abilities.
        """
        for oSet in spec.g_aoArmInstructionSets:
            oRoot = DecoderNode(sorted(oSet.aoAllInstructions,
                                       key = operator.attrgetter('fFixedMask','fFixedValue', 'sName')),#[:384],
                                0, 0);
            self.dDecoderRoots[oSet.sName] = oRoot;
            oRoot.constructNextLevel(0, sys.maxsize);

            # Set the fDecoderLeafCheckNeeded property of the instructions.
            oRoot.setInstrProps(0);


    def generateLicenseHeader(self, oVerInfo):
        """
        Returns the lines for a license header.
        """
        sDashYear = '-%s' % datetime.date.today().year;
        if sDashYear == '-2025':
            sDashYear = '';
        return [
            '/*',
            ' * Autogenerated by $Id: ArmBsdSpecCodeGen.py $',
            ' * from the open source %s specs, build %s (%s)'
            % (oVerInfo['architecture'], oVerInfo['build'], oVerInfo['ref'],),
            ' * dated %s.' % (oVerInfo['timestamp'],),
            ' *',
            ' * Do not edit!',
            ' */',
            '',
            '/*',
            ' * Copyright (C) 2025' + sDashYear + ' Oracle and/or its affiliates.',
            ' *',
            ' * This file is part of VirtualBox base platform packages, as',
            ' * available from https://www.virtualbox.org.',
            ' *',
            ' * This program is free software; you can redistribute it and/or',
            ' * modify it under the terms of the GNU General Public License',
            ' * as published by the Free Software Foundation, in version 3 of the',
            ' * License.',
            ' *',
            ' * This program is distributed in the hope that it will be useful, but',
            ' * WITHOUT ANY WARRANTY; without even the implied warranty of',
            ' * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU',
            ' * General Public License for more details.',
            ' *',
            ' * You should have received a copy of the GNU General Public License',
            ' * along with this program; if not, see <https://www.gnu.org/licenses>.',
            ' *',
            ' * SPDX-License-Identifier: GPL-3.0-only',
            ' */',
            '',
            '',
        ];

    def generateImplementationStubHdr(self, sInstrSet):
        """
        Generate implementation stubs.
        """
        asLines = self.generateLicenseHeader(spec.g_oArmInstructionVerInfo);

        # Organize this by instruction set, groups and instructions.
        sPrevCategory = '';
        for oInstr in sorted(spec.g_dArmInstructionSets[sInstrSet].aoAllInstructions,
                             key = spec.ArmInstruction.getSetAndGroupNames):
            # New group/category?
            sCategory = ' / '.join(oInstr.getSetAndGroupNames(),);
            if sCategory != sPrevCategory:
                asLines += [
                    '',
                    '',
                    '/*',
                    ' *',
                    ' * Instruction Set & Groups: %s' % (sCategory,),
                    ' *',
                    ' */',
                ];
                sPrevCategory = sCategory;

            # Emit the instruction stub.
            asArgs  = [ oField.sName for oField in oInstr.getNamedNonFixedFieldsSortedByPosition() ];
            asLines += [
                '',
                '/* %s (%08x/%08x) */' % (oInstr.sAsmDisplay, oInstr.fFixedMask, oInstr.fFixedValue,),
                '//#define IEM_INSTR_IMPL_%s__%s(%s)' % (sInstrSet, oInstr.getCName(), ', '.join(asArgs)),
                '',
            ]

        return (True, asLines);

    def generateA64ImplementationStubHdr(self, sFilename, iPartNo):
        _ = sFilename; _ = iPartNo;
        return self.generateImplementationStubHdr('A64');

    #
    # AST to C transformation.
    #

    class XformToCBase(object):
        """
        Transform AST to C - base class.
        """
        def xformIdentifier(self, oIdentifier, aoStack):
            """ Converts an identifier to appropriate C field expression. """
            _ = aoStack;
            raise Exception('Unexpected identifier: %s' % (oIdentifier,));

        def xformValue(self, oValue, aoStack):
            """ Converts the value to an integer. """
            (fValue, _, fWildcard, cBitsWidth) = oValue.getValueDetails();
            if not fWildcard:
                return ArmAstInteger(fValue, cBitsWidth);
            _ = aoStack;
            return oValue;

        def xformField(self, oField):
            """ Converts a sub-field reference. """
            # We don't support field references here.
            raise Exception('Unexpected field reference: %s' % (oField,));

        def xformBinOpInSet(self, oBinOp):
            """ Converts the value to an integer. """
            # Check & fix the input.
            assert oBinOp.sOp == 'IN';
            oSet = oBinOp.oRight;
            if isinstance(oSet, ArmAstSet):
                for iValue, oValue in enumerate(oSet.aoValues):
                    if not isinstance(oValue, (ArmAstValue, ArmAstInteger)):
                        raise Exception('Set value #%u is not an Values.Value node: %s' % (iValue, oBinOp.toString(),));
            elif isinstance(oSet, ArmAstValue):
                oSet = ArmAstSet([oBinOp.oRight,]);
            else:
                raise Exception('Unexpected use of operator "IN": %s' % (oBinOp.toString(),));

            # Create an OR list for matching each set value.
            asOrList = [];
            for oValue in oSet.aoValues:
                (fValue, fFixed, fWildcard, cBitsWidth) = oValue.getValueDetails();
                if fFixed == 0:
                    raise Exception('Bogus wildcard set expression: %s' % (oBinOp.toString(),));

                oCurOp = ArmAstBinaryOp(oBinOp.oLeft.clone(), '==', ArmAstInteger(fValue, cBitsWidth))
                if fWildcard != 0:
                    oCurOp.oLeft = ArmAstBinaryOp(oCurOp.oLeft, 'AND', ArmAstInteger(fFixed, cBitsWidth))
                asOrList.append(oCurOp);

            # Turn the list into a tree.
            return ArmAstBinaryOp.orListToTree(asOrList);

        def xformCallTo_IsFeatureImplemented(self, oCall):
            """ Converts a IsFeatureImplemented(FEAT_XXXX) call """
            if len(oCall.aoArgs) == 1:
                sFeatureNm = oCall.aoArgs[0].getIdentifierName();
                if sFeatureNm is not None:
                    sCpumFeature = g_dSpecFeatToCpumFeat.get(sFeatureNm, None);
                    if sCpumFeature is None:
                        raise Exception('Unknown IsFeatureImplemented parameter: %s (see g_dSpecFeatToCpumFeat)' % (sFeatureNm));
                    if isinstance(sCpumFeature, str):
                        return ArmAstCppExpr('IEM_GET_GUEST_CPU_FEATURES(pVCpu)->%s /*%s*/' % (sCpumFeature, sFeatureNm,), 1);
                    if sCpumFeature is True:
                        return ArmAstCppExpr('true /*%s*/'  % (sFeatureNm,), 1);
                    if sCpumFeature is False:
                        return ArmAstCppExpr('false /*%s*/' % (sFeatureNm,), 1);
                    return ArmAstCppExpr('false /** @todo IEM_GET_GUEST_CPU_FEATURES(pVCpu)->%s */' % (sFeatureNm,), 1);
            raise Exception('Expected call: %s' % (oCall,));

        def xformCallTo_SInt(self, oCall):
            if len(oCall.aoArgs) == 1:
                oArg  = oCall.aoArgs[0];
                cBits = oArg.getWidth();
                if 32 < cBits <= 64:
                    cTargetBits = 64;
                elif 0 < cBits <= 32:
                    cTargetBits = 32;
                else:
                    raise Exception('Unable to determin bit width of: %s' % (oCall,));
                if cTargetBits == cBits:
                    return VBoxAstCppCast(oArg, cTargetBits, fSigned = True);
                return ArmAstBinaryOp(VBoxAstCppCast(ArmAstBinaryOp(VBoxAstCppCast(oArg, cTargetBits),
                                                                    '<<', ArmAstInteger(cTargetBits - cBits, 32)),
                                                     cTargetBits, fSigned = True),
                                      '>>', ArmAstInteger(cTargetBits - cBits, 32));
            raise Exception('Expected call: %s' % (oCall,));

        def xformCallTo_UInt(self, oCall):
            if len(oCall.aoArgs) == 1:
                oArg  = oCall.aoArgs[0];
                cBits = oArg.getWidth();
                if 32 < cBits <= 64:
                    return VBoxAstCppCast(oArg, 64);
                if 0 < cBits <= 32:
                    return VBoxAstCppCast(oArg, 32);
                raise Exception('Unable to determin bit width of: %s' % (oCall,));
            raise Exception('Expected call: %s' % (oCall,));

        def callback(self, oNode, fEliminationAllowed, _, aoStack):
            """ Generic callback for transforming condition expressions to C. """
            if isinstance(oNode, ArmAstIdentifier):
                return self.xformIdentifier(oNode, aoStack);
            if isinstance(oNode, ArmAstValue):
                return self.xformValue(oNode, aoStack);
            if isinstance(oNode, ArmAstField):
                return self.xformField(oNode);
            if isinstance(oNode, ArmAstFunction):
                if oNode.sName == 'IsFeatureImplemented':
                    return self.xformCallTo_IsFeatureImplemented(oNode);
                if oNode.sName == 'UInt':
                    return self.xformCallTo_UInt(oNode);
                if oNode.sName == 'SInt':
                    return self.xformCallTo_SInt(oNode);
                raise Exception('Call to unsupported function: %s (%s)' % (oNode.sName, oNode.aoArgs,));
            if isinstance(oNode, ArmAstBinaryOp):
                if oNode.sOp == 'IN':
                    return self.xformBinOpInSet(oNode);
            _ = fEliminationAllowed; _ = aoStack;
            return oNode;


    class XformToCForDecoder(XformToCBase):
        """
        Transform AST to C for decoder functions.
        """
        def __init__(self, oInstr, fInLeafFn = True, oNode = None):
            IEMArmGenerator.XformToCBase.__init__(self);
            self.oInstr    = oInstr;        # ArmInstruction
            self.oNode     = oNode;         # DecoderNode || None
            self.fInLeafFn = fInLeafFn;

        def xformIdentifier(self, oIdentifier, aoStack):
            # Don't try convert just everything....
            if not aoStack or not isinstance(aoStack[-1], (ArmAstBinaryOp, ArmAstUnaryOp)):
                return oIdentifier;

            if oIdentifier.sName == 'uOpcode':
                return ArmAstCppExpr('uOpcode', 32);
            oField = self.oInstr.getFieldByName(oIdentifier.sName, False); # ArmEncodesetField
            if not oField:
                for oParent in self.oInstr.oParent.getUpIterator():
                    oField = oParent.getFieldByName(oIdentifier.sName, False);
                    if oField:
                        break;
                if not oField:
                    raise Exception('Identifier not found for instruction %s: %s' % (self.oInstr.sName, oIdentifier,));
            elif self.fInLeafFn:
                return ArmAstCppExpr(oIdentifier.sName, oField.cBitsWidth);

            # If the field is entirely within the fixed mask, return a constant integer value.
            fFieldOpMask = oField.getShiftedMask();
            if self.fInLeafFn:
                if (self.oInstr.fFixedMask & fFieldOpMask) == fFieldOpMask:
                    return ArmAstInteger((self.oInstr.fFixedValue & fFieldOpMask) >> oField.iFirstBit, oField.cBitsWidth);
            elif self.oNode and (self.oNode.fCheckedMask & fFieldOpMask) == fFieldOpMask:
                return ArmAstInteger((self.oNode.fCheckedValue & fFieldOpMask) >> oField.iFirstBit, oField.cBitsWidth);

            return VBoxAstCppField.fromFieldDetailsAndAccessExprOrInt(oField.iFirstBit, oField.cBitsWidth,
                                                                      oIdentifier.sName, 'uOpcode');

    #
    # Decoder.
    #

    def generateDecoderFunctions(self, sInstrSet):
        """
        Generates the leaf decoder functions.
        """

        asLines = [];
        for oInstr in sorted(spec.g_dArmInstructionSets[sInstrSet].aoAllInstructions,
                             key = operator.attrgetter('sName', 'sAsmDisplay')):
            sCName = oInstr.getCName();
            asLines += [
                '',
                '/* %08x/%08x: %s' % (oInstr.fFixedMask, oInstr.fFixedValue, oInstr.sAsmDisplay,),
                '   %s */'% (oInstr.getSetAndGroupNamesWithLabels(),),
                'FNIEMOP_DEF_1(iemDecode%s_%s, uint32_t, uOpcode)' % (sInstrSet, sCName,),
                '{',
            ];

            # The final decoding step, if needed.
            sIndent = '';
            asTail  = [];
            if oInstr.fDecoderLeafCheckNeeded:
                asLines += [
                    '    if ((uOpcode & UINT32_C(%#010x)) == UINT32_C(%#010x))' % (oInstr.fFixedMask, oInstr.fFixedValue,),
                    '    {',
                ];
                asTail  = [
                    '    Log(("Invalid instruction %#x at %x\\n", uOpcode, pVCpu->cpum.GstCtx.Pc.u64));',
                    '    IEMOP_RAISE_INVALID_OPCODE_RET();',
                    '}',
                ];
                sIndent = '    ';


            # Decode the fields and prepare for passing them as arguments.
            asArgs  = [];
            sLogFmt = '';
            for oField in oInstr.getNamedNonFixedFieldsSortedByPosition(): # ArmEncodesetField
                assert oField.sName and oField.fFixed != oField.getMask();
                asArgs.append(oField.sName);
                if oField.cBitsWidth < 4:
                    sLogFmt += ' %s=%%u' % (oField.sName,)
                else:
                    sLogFmt += ' %s=%%#x' % (oField.sName,)
                asLines.append('%s    uint32_t const %-10s = (uOpcode >> %2u) & %#010x;'
                               % (sIndent, oField.sName, oField.iFirstBit, (1 << oField.cBitsWidth) - 1,));

            # Any additional conditions for the instructions.
            oCondCode = oInstr.oCondition.clone().transform(IEMArmGenerator.XformToCForDecoder(oInstr).callback,
                                                            True, None, []);
            if oCondCode and not oCondCode.isBoolAndTrue():
                cchMax = 128 - len(sIndent) - 8;
                for i, sLine, fFinal in ArmAst.enumerateWithLookahead(oCondCode.toStringEx('C', cchMax).split('\n')):
                    asLines.append('%s%s%s%s' % (sIndent, '    if (' if i == 0 else '        ', sLine, ')' if fFinal else ''));
                asLines.append(sIndent + '    {');

                asTail = [
                    sIndent + '    Log(("Invalid instruction %#x at %x (cond)\\n", uOpcode, pVCpu->cpum.GstCtx.Pc.u64));',
                    sIndent + '    IEMOP_RAISE_INVALID_OPCODE_RET();',
                    sIndent + '}',
                ] + asTail;
                sIndent += '    ';

            # Log and call implementation.
            asLines += [
                '%s    LogFlow(("%%018x/%%010x: %s%s\\n", %s));'
                % (sIndent, sCName, sLogFmt, ', '.join(['pVCpu->cpum.GstCtx.Pc.u64', 'uOpcode',] + asArgs),),
                '#ifdef IEM_INSTR_IMPL_%s__%s' % (sInstrSet, sCName,),
                '%s    IEM_INSTR_IMPL_%s__%s(%s);' % (sIndent, sInstrSet, sCName, ', '.join(asArgs),),
                '#else',
                '%s    RT_NOREF(%s);' % (sIndent, ', '.join(asArgs + ['pVCpu', 'uOpcode',]),),
                '%s    return VERR_IEM_INSTR_NOT_IMPLEMENTED;' % (sIndent,),
                '#endif',
                '%s}' % (sIndent),
            ];

            asLines.extend(asTail);
        return asLines;


    class DecoderCodeBlock(object):
        """ A code block. """

        def __init__(self, sInstrSet, sName, sMatches, sStats = None):
            self.sHash     = sInstrSet;
            self.sName     = sName;
            self.asMatches = [sMatches,];
            self.sStats    = sStats;

        def getName(self):
            return self.sName;

        def getLines(self):
            return [
                '',
                '/* %s */' % ('\n   '.join(self.asMatches if not self.sStats else list(self.asMatches) + [self.sStats,]),),
            ];

        def getHash(self):
            return self.sHash;

        def _addLinesToHash(self, asLines):
            """ Adds the lines to the hash value. """
            oHash = hashlib.sha256();
            oHash.update(b'%u lines\n' % (len(asLines),));
            for sLine in asLines:
                oHash.update(sLine.encode('utf-8'));
            self.sHash += '-' + oHash.hexdigest();

    class DecoderCodeMultiIfFunc(DecoderCodeBlock):
        """ Helper class for deduplicating multiple-if functions. """
        def __init__(self, sInstrSet, oNode, uDepth):
            IEMArmGenerator.DecoderCodeBlock.__init__(self, sInstrSet, oNode.getFuncName(sInstrSet, uDepth),
                                                      '%08x/%08x level %u' % (oNode.fCheckedMask, oNode.fCheckedValue, uDepth,) );
            #
            # Check if the instructions have the same mask and all different values.
            # If the masks differs or if some have the same value, we have to include
            # any conditionals assoicated with the instructions and more carefully
            # order the tests.
            #
            # See LDRB_32BL_ldst_regoff vs LDRB_32B_ldst_regoff, and
            #     iemDecodeA64_RPRFM_R_ldst_regoff vs iemDecodeA64_PRFM_P_ldst_regoff.
            #
            dFixedMasks      = {};
            cDuplicateValues = 0;
            for oInstr in oNode.aoInstructions:
                fFixedMask  = oInstr.fFixedMask  & ~oNode.fCheckedMask
                fFixedValue = oInstr.fFixedValue & ~oNode.fCheckedMask
                if fFixedMask not in dFixedMasks:
                    dFixedMasks[fFixedMask] = { fFixedValue: 1 };
                else:
                    dValues = dFixedMasks[fFixedMask];
                    if fFixedValue not in dValues:
                        dValues[fFixedValue]  = 1;
                    else:
                        dValues[fFixedValue] += 1;
                        cDuplicateValues += 1;
            if len(dFixedMasks) == 1 and cDuplicateValues == 0:
                #
                # One mask and no duplicate values.
                #
                (fFixedMask, dValues) = next(iter(dFixedMasks.items()));
                assert (1 << fFixedMask.bit_count()) >= len(dValues);

                if fFixedMask.bit_count() == 1 and len(dValues) == 2:
                    # If there are exactly two and one deciding bit, we only need a single if-statement:
                    oInstr = oNode.aoInstructions[0];
                    self.asBody = [
                        '    if ((uOpcode & UINT32_C(%#010x)) == UINT32_C(%#010x))'
                        % (fFixedMask, oInstr.fFixedValue & ~oNode.fCheckedMask),
                        '        return iemDecode%s_%s(pVCpu, uOpcode);' % (sInstrSet, oInstr.getCName(),),
                        '    return iemDecode%s_%s(pVCpu, uOpcode);' % (sInstrSet, oNode.aoInstructions[1].getCName(),),
                    ];
                else:
                    # Otherwise use a switch.
                    self.asBody = [
                        '    switch (uOpcode & UINT32_C(%#010x))' % (fFixedMask,),
                        '    {',
                    ];
                    for oInstr in oNode.aoInstructions:
                        self.asBody += [
                            '        case UINT32_C(%#010x):' % (oInstr.fFixedValue & ~oNode.fCheckedMask,),
                            '            return iemDecode%s_%s(pVCpu, uOpcode);' % (sInstrSet, oInstr.getCName(),),
                        ];
                    if len(dValues) != (1 << fFixedMask.bit_count()):
                        self.asBody += [
                            '        default:',
                            '            return iemDecode%s_Invalid(pVCpu, uOpcode);' % (sInstrSet,),
                        ];
                    else:
                        # This is a hack to shut up the Visual C++ warning "C4715: not all control paths return a value"
                        # and the gcc warning 'control reaches end of non-void function [-Werror=return-type]' (v13.3.0).
                        asLastCase = self.asBody[-2:];
                        self.asBody = self.asBody[:-2];
                        self.asBody.append('        default:');
                        self.asBody += asLastCase;
                    self.asBody += [ '    }' ];
            else:
                #
                # Do the multiple-if stuff with any extra conditions.
                #
                # Just to be extra cautious, we sort instruction in reverse mask population
                # order, so that the more detailed masks are checked before the wider ones.
                # (The LDRB_32BL_ldst_regoff vs LDRB_32B_ldst_regoff issue could be solved
                # by this sorting trick alone, if required.)
                #
                class XformToCForDecoderWithoutFeatCheck(IEMArmGenerator.XformToCForDecoder):
                    """ Same as XformToCForDecoder, but IsFeatureImplemented calls are eliminated. """

                    def xformCallTo_IsFeatureImplemented(self, oCall):
                        if oCall.isMatchingFunctionCall('IsFeatureImplemented', str):
                            return ArmAstBool(True);
                        return IEMArmGenerator.XformToCForDecoder.xformCallTo_IsFeatureImplemented(self, oCall);

                # Assemble list of branches w/ conditions.
                atIfList = []
                for oInstr in sorted(oNode.aoInstructions,
                                     key = lambda o: ( (o.fFixedMask & ~oNode.fCheckedMask).bit_count(),
                                                       o.fFixedMask, fFixedValue ),
                                     reverse = True):
                    fFixedMask  = oInstr.fFixedMask  & ~oNode.fCheckedMask;
                    fFixedValue = oInstr.fFixedValue & ~oNode.fCheckedMask;
                    oCode = ArmAstBinaryOp(ArmAstBinaryOp(ArmAstIdentifier('uOpcode'), 'AND', ArmAstInteger(fFixedMask, 32)),
                                           '==', ArmAstInteger(fFixedValue, 32));
                    if oInstr.oCondition:
                        oCode = ArmAstBinaryOp(oCode, '&&', oInstr.oCondition.clone());
                    oOptCode = oCode.clone().transform(XformToCForDecoderWithoutFeatCheck(oInstr, False, oNode).callback,
                                                       True, None, []);

                    atIfList.append((oInstr, oOptCode, oCode,
                                     'return iemDecode%s_%s(pVCpu, uOpcode);' % (sInstrSet, oInstr.getCName(),)));

                # Add invalid opcode as the default.
                atIfList.append((None, ArmAstBool(True), None, 'return iemDecode%s_Invalid(pVCpu, uOpcode);' % (sInstrSet,)));

                # Output them.
                self.asBody = [];
                fDone       = False;
                ## @todo We have a few cases where the first two entries uses a single bit mask
                ##       and anything beyond that is unreachable.
                for oInstr, oOptCode, oCode, sRetStmt in atIfList:
                    if not fDone and oOptCode and not oOptCode.isBoolAndFalse():
                        if oInstr:
                            self.asBody.append('    /* %#08x/%#08x%s%s */'
                                               % (oInstr.fFixedMask, oInstr.fFixedValue,
                                                  ' org expr: ' if oCode else '', oCode.toString() if oCode else '',));
                        fDone = oOptCode.isBoolAndTrue();
                        if fDone:
                            self.asBody.append('    ' + sRetStmt);
                        else:
                            sPrefix = '    if ('
                            for sLine in oOptCode.toStringEx(sLang = 'C', cchMaxWidth = 120 - len(sPrefix)).split('\n'):
                                self.asBody.append(sPrefix + sLine);
                                sPrefix = '        ';
                            self.asBody[-1] += ')';
                            self.asBody.append('        ' + sRetStmt);
                    elif oInstr:
                        self.asBody.append('    /* skipping %#08x/%#08x %s%s%s */'
                                           % (oInstr.fFixedMask, oInstr.fFixedValue, oInstr.getCName(),
                                              ': ' if oCode else '', oCode.toString() if oCode else '',));

            self._addLinesToHash(self.asBody);

        def getLines(self):
            asLines  = IEMArmGenerator.DecoderCodeBlock.getLines(self);
            asLines += [
                'FNIEMOP_DEF_1(%s, uint32_t, uOpcode)' % (self.getName(),),
                '{',
            ];
            asLines += self.asBody;
            asLines += [
                '}',
            ];
            return asLines;

    class DecoderCodeTableLeafEntry(object):
        """ Special DecoderCodeTableFunc::dChildCode for leaf decoder functions. """
        def __init__(self, sName):
            self.sName = sName;

        def getHash(self):
            return self.sName;

        def getName(self):
            return self.sName;

    class DecoderCodeTableFunc(DecoderCodeBlock):
        """ Helper class for table based decoder function. """

        def __init__(self, sInstrSet, sName, sMatches, sStats, uDepth, cTabEntries, dChildCode, asIdx):
            IEMArmGenerator.DecoderCodeBlock.__init__(self, sInstrSet, sName, sMatches, sStats);
            self.sInstrSet   = sInstrSet;
            self.uDepth      = uDepth;
            self.cTabEntries = cTabEntries;
            self.dChildCode  = dChildCode;      # Note! DecoderCodeBlock or DecoderCodeTableLeafEntry instances.
            self.asIdx       = asIdx;

            self._addLinesToHash(asIdx);
            self._addLinesToHash(['%s-%s' % (idx, oCodeBlock.getHash(),) for idx, oCodeBlock in dChildCode.items()]);

        def getLines(self):
            # Generate the function.  For the top level we just do the table, as
            # the functions are static and we need the interpreter code to be able
            # to address the symbol and this is the speedier way.
            asLines = IEMArmGenerator.DecoderCodeBlock.getLines(self);
            if self.uDepth > 0:
                asLines += [
                    'FNIEMOP_DEF_1(%s, uint32_t, uOpcode)' % (self.getName(),),
                    '{',
                    '    static PFIEMOPU32 const s_apfn[] =',
                    '    {',
                ];
                sTabNm  = 's_apfn';
                sIndent = '    ';
            else:
                sTabNm  = 'g_apfnIemInterpretOnly' + self.sInstrSet;
                asLines += [
                    'PFIEMOPU32 const %s[] =' % (sTabNm,),
                    '{',
                ];
                sIndent = '';

            idxPrev = -1;
            for idx in sorted(self.dChildCode):
                idxPrev += 1;
                while idxPrev < idx:
                    asLines.append(sIndent + '    iemDecode%s_Invalid, // %s' % (self.sInstrSet, idxPrev,));
                    idxPrev += 1;
                asLines.append('%s    %s,' % (sIndent, self.dChildCode[idx].getName(),));

            while idxPrev + 1 < self.cTabEntries:
                idxPrev += 1;
                asLines.append(sIndent + '    iemDecode%s_Invalid, // %s' % (self.sInstrSet, idxPrev,));

            asLines += [
                '%s};' % (sIndent,),
                '%sAssertCompile(RT_ELEMENTS(%s) == %#x);' % (sIndent, sTabNm, self.cTabEntries,),
                '',
            ];

            if self.uDepth > 0:
                asLines += self.asIdx;
                asLines += [
                    '    return s_apfn[idx](pVCpu, uOpcode);',
                    '}'
                ];
            return asLines;


    def generateDecoderCode(self, sInstrSet, oNode, uDepth, dCodeCache):
        """
        Recursively generates the decoder code.
        """
        assert oNode.fChildMask != 0 and oNode.fChildMask not in (0x7fffffff, 0xffffffff, 0x4fffffff), \
            'fChildMask=%s #dChildren=%s aoInstr=%s' % (oNode.fChildMask, len(oNode.dChildren), oNode.aoInstructions,);
        assert oNode.dChildren;

        #
        # First recurse.
        #
        aoCodeBlocks    = [];
        dChildCode      = {};
        cLeafEntries    = 0;
        cMultiIfEntries = 0;
        cReusedCode     = 0;
        for idx in sorted(oNode.dChildren):
            oChildNode = oNode.dChildren[idx];
            if oChildNode.dChildren:
                aoSubCodeBlocks = self.generateDecoderCode(sInstrSet, oChildNode, uDepth + 1, dCodeCache);
            elif oChildNode.fChildMask == DecoderNode.kfChildMaskMultipleOpcodeValueIfs:
                assert len(oChildNode.aoInstructions) > 1;
                aoSubCodeBlocks = [IEMArmGenerator.DecoderCodeMultiIfFunc(sInstrSet, oChildNode, uDepth + 1),];
                cMultiIfEntries += 1;
            else:
                assert len(oChildNode.aoInstructions) == 1;
                assert oChildNode.fChildMask in [DecoderNode.kfChildMaskOpcodeValueIf, 0];
                cLeafEntries += 1;
                dChildCode[idx] = IEMArmGenerator.DecoderCodeTableLeafEntry(oChildNode.getFuncName(sInstrSet, -1));
                continue;

            oCodeBlock  = aoSubCodeBlocks[-1];
            oCachedCode = dCodeCache.get(oCodeBlock.getHash(), None);
            if len(aoSubCodeBlocks) != 1 or oCachedCode is None:
                assert oCachedCode is None; # Code shouldn't be in the hash if it has new dependencies!
                dChildCode[idx] = oCodeBlock;
                aoCodeBlocks.extend(aoSubCodeBlocks);
                dCodeCache[oCodeBlock.getHash()] = oCodeBlock; # comment out of disable code reuse.
            else:
                #print('debug: code cache hit');
                oCachedCode.asMatches += oCodeBlock.asMatches;
                dChildCode[idx] = oCachedCode;
                cReusedCode += 1;

        assert len(dChildCode) == len(oNode.dChildren);
        assert dChildCode;

        #
        # Match info w/ stats.
        #
        cTabEntries = 1 << oNode.fChildMask.bit_count();
        sStats   = 'mask=%#x entries=%#x children=%#x valid=%%%u (%#x) leaf=%%%u (%#x) multi-if=%%%u (%#x) reuse=%%%u (%#x)' \
                 % (oNode.fChildMask, cTabEntries, len(oNode.dChildren),
                    int(round(len(oNode.dChildren) * 100.0 / cTabEntries)), len(oNode.dChildren),
                    int(round(cLeafEntries         * 100.0 / cTabEntries)), cLeafEntries,
                    int(round(cMultiIfEntries      * 100.0 / cTabEntries)), cMultiIfEntries,
                    int(round(cReusedCode          * 100.0 / cTabEntries)), cReusedCode, );
        if uDepth == 0:
            sMatches = '--decoder-hint0 %#x' % (oNode.fChildMask,);
        elif uDepth == 1 and len(oNode.aoInstructions) >= self.oOptions.iDecoderL1Threshold:
            sMatches = '--decoder-hint1 %#x/%#x/%#x' % (oNode.fCheckedMask, oNode.fCheckedValue, oNode.fChildMask);
        else:
            sMatches = '%08x/%08x level %u' % (oNode.fCheckedMask, oNode.fCheckedValue, uDepth,);

        #
        # Code for extracting the index from uOpcode.
        #
        aaiAlgo = MaskZipper.compileAlgo(oNode.fChildMask);
        assert aaiAlgo, 'fChildMask=%s #children=%s instrs=%s' % (oNode.fChildMask, len(oNode.dChildren), oNode.aoInstructions,);
        asIdx = [
            '    /* fMask=%#010x -> %#010x */' % (oNode.fChildMask, cTabEntries - 1),
            '    uintptr_t const idx = ((uOpcode >> %2u) & UINT32_C(%#010x)) /* bit %2u L %u -> 0 */'
            % (aaiAlgo[0][0], aaiAlgo[0][2], aaiAlgo[0][0], aaiAlgo[0][2].bit_count(), ),
        ];
        for iSrcBit, iDstBit, fMask in aaiAlgo[1:]:
            asIdx.append('                        | ((uOpcode >> %2u) & UINT32_C(%#010x)) /* bit %2u L %u -> %u */'
                         % (iSrcBit - iDstBit, fMask << iDstBit, iSrcBit, fMask.bit_count(), iDstBit));
        asIdx[-1] += ';';

        # For the top level table, we save the expression so we can later put in a header file.
        if uDepth == 0:
            self.dRootsIndexExpr[sInstrSet] = asIdx;

        #
        # Create the code block for this table-based decoder function.
        #
        oCodeBlock = IEMArmGenerator.DecoderCodeTableFunc(sInstrSet, oNode.getFuncName(sInstrSet, uDepth), sMatches, sStats,
                                                          uDepth, cTabEntries, dChildCode, asIdx)
        return aoCodeBlocks + [oCodeBlock,];

    def generateDecoderCpp(self, sInstrSet):
        """ Generates the decoder data & code. """
        if sInstrSet not in self.dDecoderRoots:
            raise Exception('Instruction set not found: %s' % (sInstrSet,));

        asLines = self.generateLicenseHeader(spec.g_oArmInstructionVerInfo);
        asLines += [
            '#define LOG_GROUP LOG_GROUP_IEM',
            '#define VMCPU_INCL_CPUM_GST_CTX',
            '#include <iprt/asm.h> /* needed for Armv8A64ConvertImmRImmS2Mask32 and friends in iprt/armv8.h */',
            '#include "IEMInternal.h"',
            '#include <VBox/vmm/vm.h>',
            '#include "VBox/err.h"',
            '',
            '#include <iprt/armv8.h>',
            '',
            '#include "IEMMc.h"',
            '#include "IEMInline-armv8.h"',
            '',
            '#include "%s"' % (os.path.basename(self.oOptions.sFileDecoderHdr) if self.oOptions.sFileDecoderHdr
                               else 'IEMAllIntpr%sTables-armv8.h' % (sInstrSet),),
            '#include "%s"' % (os.path.basename(self.oOptions.sFileStubHdr) if self.oOptions.sFileStubHdr
                               else 'IEMAllInstr%sImpl.h' % (sInstrSet),),
            '',
            '',
            '/** Invalid instruction decoder function. */',
            'FNIEMOP_DEF_1(iemDecode%s_Invalid, uint32_t, uOpcode)' % (sInstrSet,),
            '{',
            '    Log(("Invalid instruction %#x at %x\\n", uOpcode, pVCpu->cpum.GstCtx.Pc.u64));',
            '    RT_NOREF_PV(uOpcode);',
            '    IEMOP_RAISE_INVALID_OPCODE_RET();',
            '}',
        ];

        asLines += self.generateDecoderFunctions(sInstrSet);

        dCodeCache   = {};
        assert self.dDecoderRoots[sInstrSet].dChildren;
        aoCodeBlocks = self.generateDecoderCode(sInstrSet, self.dDecoderRoots[sInstrSet], 0, dCodeCache);
        for oCodeBlock in aoCodeBlocks:
            asLines.extend(oCodeBlock.getLines());

        return (True, asLines);

    def generateA64DecoderCpp(self, sFilename, iPartNo):
        _ = sFilename; _ = iPartNo;
        return self.generateDecoderCpp('A64');


    def generateDecoderHdr(self, sFilename, iPartNo):
        """ Generates the decoder header file. """
        _ = iPartNo;

        asLines = self.generateLicenseHeader(spec.g_oArmInstructionVerInfo);
        sBlockerName = re.sub('[.-]', '_', os.path.basename(sFilename));
        asLines += [
            '#ifndef VMM_INCLUDED_SRC_VMMAll_target_armv8_%s' % (sBlockerName,),
            '#define VMM_INCLUDED_SRC_VMMAll_target_armv8_%s' % (sBlockerName,),
            '#ifndef RT_WITHOUT_PRAGMA_ONCE',
            '# pragma once',
            '#endif',
            '',
        ];
        for sInstrSet in sorted(self.dDecoderRoots.keys()):
            asLines += [
                '/** The top-level %s decoder table for the IEM interpreter. */' % (sInstrSet,),
                'extern PFIEMOPU32 const g_apfnIemInterpretOnly%s[%#x];'
                % (sInstrSet, 1 << self.dDecoderRoots[sInstrSet].fChildMask.bit_count()),
                '',
                '/**',
                ' * Calculates the index for @a uOpcode in g_apfnIemInterpretOnly%s.' % (sInstrSet,),
                ' */',
                'DECL_FORCE_INLINE(uintptr_t) iemInterpretOnly%sCalcIndex(uint32_t uOpcode)' % (sInstrSet,),
                '{',
            ];
            assert sInstrSet in self.dRootsIndexExpr and len(self.dRootsIndexExpr[sInstrSet]); # Set by generateDecoderCpp!
            asLines += self.dRootsIndexExpr[sInstrSet];
            asLines += [
                '    return idx;',
                '}',
                '',
            ];
        asLines += [
            '#endif /* !VMM_INCLUDED_SRC_VMMAll_target_armv8_%s */' % (sBlockerName,),
            '',
        ];
        return (True, asLines);


    #
    # System registers
    #

    def generateCImplSysRegCpp(self, sInstrSet, sState):
        """ Worker for generateA64CImplSysRegCpp. """
        _ = sInstrSet;

        #
        # Gather the relevant system register access code.
        #
        dAccessors = {
            'A64.MRS':          SysRegGeneratorA64Mrs(),
            'A64.MSRregister':  SysRegGeneratorA64MsrReg(),
            #'A64.MSRimmediate': SysRegGeneratorNop('A64.MSRimmediate', 'iemCImplA64_msr_imm'), # No code, must be handcoded.
            'A64.MRRS':         SysRegGeneratorNop('A64.MRRS',         'iemCImplA64_mrrs'), ## @todo requires 128-bit state.
            'A64.MSRRregister': SysRegGeneratorNop('A64.MSRRregister', 'iemCImplA64_msrr'), # SysRegGeneratorA64MsrrReg(),
            'A64.SYS':          SysRegGeneratorA64Sys(),
            'A64.SYSL':         SysRegGeneratorA64SysL(),
            'A64.SYSP':         SysRegGeneratorA64SysP(),
        } # type: Dict[str,SysRegGeneratorBase]

        for oReg in spec.g_daoAllArmRegistersByState[sState]: # type: ArmRegister
            assert oReg.sState == sState;
            for oAccessor in oReg.aoAccessors:  # type: ArmAccessorBase
                if isinstance(oAccessor, spec.ArmAccessorSystem):
                    if oAccessor.sName in { 'A64.SYS', 'A64.SYSL', 'A64.SYSP', 'A64.MSRimmediate'}:
                        pass; # Skip the generic system instruction entries and instruction w/o any access code.
                    elif oAccessor.sName in dAccessors:
                        # System Register.
                        sEncSortKey = 'encoding=%s' % (oAccessor.oEncoding.dNamedValues,);
                        if (    not isinstance(oAccessor, spec.ArmAccessorSystemArray)
                            and not oAccessor.oEncoding.fHasWildcard
                            and not oAccessor.oEncoding.fHasIndex
                            and oAccessor.sName in {'A64.MRS', 'A64.MSRregister',}):
                            sEncSortKey = oAccessor.oEncoding.getSysRegIdCreate();
                        dAccessors[oAccessor.sName].aoInfo.append(SysRegAccessorInfo(oAccessor, oReg, sEncSortKey));
                    else:
                        # System instruction - categorize by aliased instruction.
                        sAliasing = 'A64.SYS';
                        if oAccessor.oAccess.containsClassInstances(ArmAstAssignment):
                            sAliasing = 'A64.SYSL';
                        if oReg.aoFieldsets and oReg.aoFieldsets[0].cBitsWidth == 128:
                            assert sAliasing == 'A64.SYS';
                            sAliasing = 'A64.SYSP';
                        sEncSortKey = oAccessor.oEncoding.getSysRegIdCreate();
                        sAsmValue = oAccessor.sName[oAccessor.sName.find('.') + 1:];
                        if oAccessor.oEncoding.sAsmValue:
                            sAsmValue = '%s_%s' % (sAsmValue, oAccessor.oEncoding.sAsmValue,);
                        dAccessors[sAliasing].aoInfo.append(SysRegAccessorInfo(oAccessor, oReg, sEncSortKey, sAsmValue));


        for sKey, oGenerator in dAccessors.items():
            assert sKey == oGenerator.sInstr;
            oGenerator.aoInfo.sort(key = operator.attrgetter('sEnc', 'sRegName'));

            # Eliminate duplicate entries. See for instance SPMACCESSR_EL1 & SPMACCESSR_EL2.
            sEncPrev = '';
            idx      = 0;
            while idx < len(oGenerator.aoInfo):
                if oGenerator.aoInfo[idx].sEnc != sEncPrev:
                    sEncPrev = oGenerator.aoInfo[idx].sEnc;
                    idx += 1;
                else:
                    oThisAcc = oGenerator.aoInfo[idx].oAccessor;
                    oPrevAcc = oGenerator.aoInfo[idx - 1].oAccessor;
                    assert oThisAcc.oEncoding.toString() == oPrevAcc.oEncoding.toString();
                    if oThisAcc.oAccess and oPrevAcc.oAccess:
                        if not oThisAcc.oAccess.isSame(oPrevAcc.oAccess):
                            # A number of ICV_xxx variants of ICC_xxx registers have an extra !HaveEL(EL2) condition
                            # in the 2024-12 specs that seems to have been removed since. Pick the variant w/o it.
                            # The sorting on sRegName should make sure that ICC_xxx is prev and ICV_xxx is current.
                            if (    oThisAcc.oEncoding.sAsmValue.startswith('ICC_')
                                and oGenerator.aoInfo[idx].oReg.sName.startswith('ICV_')):
                                pass;
                            else:
                                asCodeThis = oThisAcc.oAccess.toStringList();
                                for i, sLine in enumerate(asCodeThis):
                                    asCodeThis[i] = '%2u: %s' % (i, sLine);

                                asCodePrev = oPrevAcc.oAccess.toStringList();
                                for i, sLine in enumerate(asCodePrev):
                                    asCodePrev[i] = '%2u: %s' % (i, sLine);

                                asDiff = [];
                                for i, sLine in enumerate(asCodeThis):
                                    if i < len(asCodePrev):
                                        if sLine != asCodePrev[i]:
                                            asDiff.extend(['cur:  %s' % (sLine,), 'prev: %s' % asCodePrev[i],]);
                                    else:
                                        asDiff.extend(['cur:  %s' % (sLine,), 'prev: <missing>',]);
                                for i in range(len(asCodeThis), len(asCodePrev)):
                                    asDiff.extend(['cur:  <missing>', 'prev: %s' % (asCodePrev[i]),]);
                                if asDiff:
                                    sCode = '%s\n------vs------\n%s\n-----diff-----\n%s' \
                                          % ('\n'.join(asCodeThis), '\n'.join(asCodePrev), '\n'.join(asDiff),);
                                    raise Exception('Duplicate entry %s (%s/%s, %s/%s) not matching!:\n%s'
                                                    % (sEncPrev,
                                                       oThisAcc.oEncoding.sAsmValue, oGenerator.aoInfo[idx].oReg.sName,
                                                       oPrevAcc.oEncoding.sAsmValue, oGenerator.aoInfo[idx - 1].oReg.sName,
                                                       sCode,));
                    elif oThisAcc.oAccess or oPrevAcc.oAccess:
                        raise Exception('Duplicate entry %s not matching! (#2)' % (sEncPrev,));
                    del oGenerator.aoInfo[idx];

        #
        # Morph the code into IEM suitable C code.
        #
        for oGenerator in dAccessors.values():
            for oInfo in oGenerator.aoInfo:
                if oInfo.sEnc[0] == 'A':
                    oGenerator.morphCodeToC(oInfo);
                    if oInfo.cIncompleteNodes > 0:
                        oGenerator.cIncomplete += 1;
                    else:
                        oGenerator.cComplete   += 1;
                    if oInfo.cAssignmentsWithUnresolvedRegs > 0:
                        oGenerator.cUnresolvedRegs += 1;

        #
        # File header.
        #
        asLines = self.generateLicenseHeader(spec.g_oArmRegistersVerInfo);
        asLines += [
            '#define LOG_GROUP LOG_GROUP_IEM_CIMPL',
            '#define VMCPU_INCL_CPUM_GST_CTX',
            '#include "IEMInternal.h"',
            '#include <VBox/vmm/vm.h>',
            '#include "VBox/err.h"',
            '',
            '#include "iprt/armv8.h"',
            '',
            '#include "IEMMc.h"',
            '#include "IEMInline-armv8.h"',
            '',
            '',
        ];

        #
        # Generate the real code, accessor type by accessor type.
        #
        for oGenerator in dAccessors.values():
            asLines += [
                '',
                '',
                '',
                '/*',
                ' * %s' % (oGenerator.sInstr,),
                ' *',
                ' * %4u registers' % (len(oGenerator.aoInfo),),
                ' * %4u complete - %u with unresolved registers'  % (oGenerator.cComplete, oGenerator.cUnresolvedRegs,),
                ' * %4u incomplete' % (oGenerator.cIncomplete,),
            ];
            asWarnings = oGenerator.printWarnings();
            if asWarnings:
                asLines += [
                    ' *',
                    ' * Warnings:',
                ];
                asLines += [ ' *   ' + sWarning[9:] for sWarning in asWarnings ];
            asLines += [
                ' */'
            ];

            # Individual handler functions.
            for oInfo in oGenerator.aoInfo:
                asLines.append('');
                if oInfo.sEnc[0] == 'A':
                    asLines += oGenerator.generateOneHandler(oInfo);
                else:
                    asLines += [
                        '// %s' % (oInfo.oAccessor.oEncoding.sAsmValue,),
                        '// %s' % (oInfo.sEnc,),
                    ];
                    if oInfo.oAccessor.oAccess:
                        asLines.extend(oInfo.oAccessor.oAccess.toStringList('//    '));
                    else:
                        asLines.append('// access is None!');

            # Main switch function.
            asLines += oGenerator.generateMainFunction();

        asLines.append(''); # make SCM happy.
        return (True, asLines);

    def generateA64CImplSysRegInsCpp(self, sFilename, iPartNo):
        """ Generates the IEMAllCImplA64SysRegsAndIns-armv8.cpp file. """
        _ = sFilename; _ = iPartNo;
        return self.generateCImplSysRegCpp('A64', 'AArch64');


    #
    # Features
    #

    def generateFeaturesHdr(self, sFilename, iPartNo):
        _ = iPartNo;

        asLines = self.generateLicenseHeader(spec.g_oArmFeaturesVerInfo);
        sBlockerName = re.sub('[.-]', '_', os.path.basename(sFilename));
        asLines += [
            '#ifndef VMM_INCLUDED_SRC_VMMR3_target_armv8_%s' % (sBlockerName,),
            '#define VMM_INCLUDED_SRC_VMMR3_target_armv8_%s' % (sBlockerName,),
            '#ifndef RT_WITHOUT_PRAGMA_ONCE',
            '# pragma once',
            '#endif',
            '',
        ];

        #
        # Gather the features we're listing and sort it so that we start with
        # expressions only requiring a single system register value and move
        # on to more complicated ones, ending with those including other
        # FEAT_XXXX references (relies on sorting order).
        #
        aoFeatures = [oFeature for oFeature in spec.g_aoAllArmFeatures
                      if    oFeature.sName.startswith('FEAT_')
                        #and oFeature.sName not in ('FEAT_EL0', 'FEAT_EL1', 'FEAT_EL2', 'FEAT_EL3') # removed in 2025
                        and oFeature.oSupportExpr
                        and oFeature.asSupportExprVars[0].split('.')[0] not in ('AArch32', 'PMU', 'AMU', 'ext', 'uext') ];
        aoFeatures = sorted(aoFeatures, key = lambda oFeature: (len(oFeature.asSupportExprVars), oFeature.asSupportExprVars));

        class XformToCForFeatures(IEMArmGenerator.XformToCBase):
            """
            Transform AST to C for the feature population function.
            """
            def __init__(self):
                IEMArmGenerator.XformToCBase.__init__(self);
                self.dVars        = {}      # Type: Dict[str, ArmRegister]
                self.idxFeatures  = 0;

            def xformIdentifier(self, oIdentifier, aoStack):
                sName = oIdentifier.sName;

                # Deal with FEAT_xxx:
                if sName.startswith('FEAT_'):
                    for i in range(self.idxFeatures):
                        if aoFeatures[i].sName == sName:
                            return ArmAstCppExpr('pFeatures->%s' % (g_dSpecFeatToCpumFeat.get(sName, sName),), 1);
                    raise Exception('Internal error: Feature %s has not yet been initialized! (%s)'
                                    % (sName, ', '.join([aoFeatures[i].sName for i in range(self.idxFeatures)]),));

                # The v8Ap4 style stuff we just set to false for now. (HACK)
                if sName in ('v8Ap4',):
                    return ArmAstBool(False, 'v8Ap4');

                return super().xformIdentifier(oIdentifier, aoStack);

            def xformField(self, oField): # ArmAstField
                if not oField.sSlices and not oField.sInstance and oField.sField and oField.sName and oField.sState:
                    if oField.sState == 'AArch64':  sCVarNm = 'u%s' % (oField.sName,);
                    else:                           sCVarNm = 'u%s_%s' % (oField.sState, oField.sName,)
                    oReg = self.dVars.get(sCVarNm) # Type: ArmRegister
                    if not oReg:
                        oReg = spec.g_ddoAllArmRegistersByStateByName[oField.sState][oField.sName];
                        self.dVars[sCVarNm] = oReg;

                    aoFields = oReg.daoFields.get(oField.sField);
                    if not aoFields:
                        raise Exception('Field %s was not found in register %s.%s (known fields: %s)'
                                        % (oField.sField, oField.sState, oField.sName, oReg.daoFields.keys(),));
                    if len(aoFields) > 1:
                        raise Exception('Ambigious field %s was in register %s.%s: %s'
                                        % (oField.sField, oField.sState, oField.sName, aoFields,));
                    o1stField = aoFields[0]    # Type: ArmFieldsBase

                    if len(o1stField.aoRanges) != 1:
                        raise Exception('TODO: Using complicated field %s in register %s.%s: %s'
                                        % (oField.sField, oField.sState, oField.sName, oField.aoRanges,));
                    iFirstBit  = o1stField.aoRanges[0].iFirstBit;
                    cBitsWidth = o1stField.aoRanges[0].cBitsWidth;
                    if (    o1stField.oParent
                        and isinstance(o1stField.oParent, spec.ArmFieldsConditionalField)): # Relative to parent range.
                        if len(o1stField.oParent.aoRanges) != 1:
                            raise Exception('TODO: Using complicated conditional field %s in register %s.%s: %s'
                                            % (oField.sField, oField.sState, oField.sName, o1stField.oParent.aoRanges,));
                        iFirstBit += o1stField.oParent.aoRanges[0].iFirstBit;
                        assert cBitsWidth <= o1stField.oParent.aoRanges[0].cBitsWidth;

                    return VBoxAstCppField.fromFieldDetailsAndAccessExprOrInt(iFirstBit, cBitsWidth, oField.sField, sCVarNm);
                return super().xformField(oField);


        asLines += [
            '',
            '',
            '/**',
            ' * Explodes ARMv8+ features from an array of system register values.',
            ' *',
            ' * @returns VBox status code',
            ' * @param   paSysRegs   The system registers and their values.',
            ' * @param   cSysRegs    Number of system register values.',
            ' * @param   pFeatures   The structure to explode the features into.',
            ' */',
            'VMMDECL(int) CPUMCpuIdExplodeFeaturesArmV8(PCSUPARMSYSREGVAL paSysRegs, uint32_t cSysRegs,',
            '                                           CPUMFEATURESARMV8 *pFeatures)',
            '{',
            '    RT_ZERO(*pFeatures);',
            '',
        ];
        asTodo  = [];
        oHelper = XformToCForFeatures();
        sIndent = '                                     ';
        cCVars  = 0;
        for oHelper.idxFeatures, oFeature in enumerate(aoFeatures):
            sFeatureMemberNm = g_dSpecFeatToCpumFeat.get(oFeature.sName, oFeature.sName);
            if not isinstance(sFeatureMemberNm, bool): # Skip fixed items.
                if sFeatureMemberNm is oFeature.sName:
                    asTodo.append(oFeature.sName);
                print('debug: %s/%s <-> %s' % (oFeature.sName, sFeatureMemberNm, oFeature.oSupportExpr.toString()))
                oCode = oFeature.oSupportExpr.clone().transform(oHelper.callback, True, None, []);
                if oCode.getWidth() != 1:
                    oCode = ArmAstUnaryOp('!', ArmAstUnaryOp('!', oCode));

                if cCVars < len(oHelper.dVars):
                    asLines.append('');
                    for sCVarNm in list(oHelper.dVars.keys())[cCVars:]: ## ASSUMES ordered dictionaries (py 3.7+)
                        asLines.append('    uint64_t const %-18s = cpumCpuIdLookupSysReg(paSysRegs, cSysRegs, %s);'
                                       % (sCVarNm, oHelper.dVars[sCVarNm].getVBoxConstant(),));
                    cCVars = len(oHelper.dVars);

                for iLine, sLine, fFinal in ArmAst.enumerateWithLookahead(oCode.toStringEx('C', 116 - 38).split('\n')):
                    if iLine == 0:
                        sLine = '    pFeatures->%-22s = %s%s' % (sFeatureMemberNm, sLine.lstrip(), ';' if fFinal else '');
                        sLine = '%-116s /* %s */' % (sLine, oFeature.sName,);
                    else:
                        sLine = sIndent + (' ' if sLine[0] != ' ' and sLine[1] == ' ' else '') + sLine + (';' if fFinal else '');
                    asLines.append(sLine);

        if asTodo:
            print('Error! Please add the features: %s' % (', '.join(asTodo),));
            for sFeature in asTodo:
                print('Error!   %-25s: %s -- %s'
                      % (sFeature, spec.g_dAllArmFeaturesByName[sFeature].asSupportExprVars,
                         ';  '.join(['#%u: %s' % (iExpr, oExpr.toString())
                                     for iExpr, oExpr in enumerate(spec.g_dAllArmFeaturesByName[sFeature].aoConstraints)]),));

        # Did we miss any features in CPUMFEATURESARMV8?
        asMissing = [];
        hsMissingFeatures = (  set({sNm for sNm, oMemb in g_dSpecFeatToCpumFeat.items() if not isinstance(oMemb, bool)} )
                             - set({oFeature.sName for oFeature in aoFeatures}) );
        if hsMissingFeatures:
            print('Error! The following CPUMFEATURESARMV8 members have not been initialized: %s'
                  % (', '.join(hsMissingFeatures),));
            asMissing += [
                '',
                '    /* Initializing "missing" members: */',
            ];
            for sFeature in sorted(hsMissingFeatures):
                oFeature = spec.g_dAllArmFeaturesByName.get(sFeature)
                if oFeature:
                    sExtra = ' (%s)' % ('; '.join(['#%u: %s' % (iExpr, oExpr.toString())
                                                   for iExpr, oExpr in enumerate(oFeature.aoConstraints)]),);
                else:
                    sExtra = '';
                asMissing.append('    pFeatures->%-22s = 0; /* %s%s */' % (g_dSpecFeatToCpumFeat[sFeature], sFeature, sExtra));

        asLines += asMissing;

        asLines += [
            '',
            '    return cpumCpuIdExplodeFeaturesArmV8Handcoded(paSysRegs, cSysRegs, pFeatures);',
            '}'
        ];

        #
        # Write a dumper function for the CPUMFEATURESARMV8 structure.
        #
        cchMaxFeatNm = max(len(sKey) for sKey in g_dSpecFeatToCpumFeat);
        asLines += [
            '',
            '#ifdef IN_RING3',
            '',
            '# include <VBox/vmm/dbgf.h> /* DBGFINFOHLP */',
            '',
            '/**',
            ' * Prints the ARMv8+ features in @a pFeatures.',
            ' *',
            ' * @param   pHlp            The output callback helper.',
            ' * @param   cchOutput       The width of the output (for multiple columns).',
            ' *                          Zero is taken to mean a single column.',
            ' * @param   pFeatures       The features to dump',
            ' * @param   pszLabel        The label.',
            ' * @param   pSecondary      Optional secondary feature set to compare with.',
            ' * @param   pszSecondary    The label for the secondary feature set.',
            ' */',
            'VMMR3DECL(void) CPUMR3CpuIdPrintArmV8Features(PCDBGFINFOHLP pHlp, uint32_t cchOutput,',
            '                                              CPUMFEATURESARMV8 const *pFeatures, const char *pszLabel,',
            '                                              CPUMFEATURESARMV8 const *pSecondary, const char *pszSecondary)',
            '{',
            '    unsigned const cchLabel  = (unsigned)strlen(pszLabel);',
            '    unsigned const cchLabel2 = pszSecondary ? (unsigned)strlen(pszSecondary) : 0;',
            '    unsigned const cchPad    = cchLabel - 1 + cchLabel2 - !!cchLabel2;',
            '    unsigned const cchColumn = 2 + %s + 3 + cchLabel + (pSecondary ? 3 + cchLabel2 : 0);' % (cchMaxFeatNm,),
            '    unsigned const cColumns  = cchOutput < cchColumn ? 1 : cchOutput / cchColumn;',
            '    if (pSecondary)',
            '        for (unsigned iColumn = 0; iColumn < cColumns; iColumn++)',
            '            pHlp->pfnPrintf(pHlp, "  %%%us = %%s (%%s)", "Features", pszLabel, pszSecondary);' % (cchMaxFeatNm,),
            '    else',
            '        for (unsigned iColumn = 0; iColumn < cColumns; iColumn++)',
            '            pHlp->pfnPrintf(pHlp, "  %%%us = %%s", "Features", pszLabel);' % (cchMaxFeatNm,),
            '    pHlp->pfnPrintf(pHlp, "\\n");',
            '',
            '    unsigned iColumn = 0;',
            '#define PRINT_FEATURE(a_Name, a_Member) do { \\',
            '            if (pSecondary) \\',
            '                pHlp->pfnPrintf(pHlp, "%%*s  %%%us = %%u (%%u)", \\' % (cchMaxFeatNm,),
            '                                iColumn ? cchPad : 0, "", #a_Name, pFeatures->a_Member, pSecondary->a_Member); \\',
            '            else \\',
            '                pHlp->pfnPrintf(pHlp, "%%*s  %%%us = %%u", \\' % (cchMaxFeatNm,),
            '                                iColumn ? cchPad : 0, "", #a_Name, pFeatures->a_Member); \\',
            '            iColumn += 1; \\',
            '            if (iColumn >= cColumns) \\',
            '            { \\',
            '                pHlp->pfnPrintf(pHlp, "\\n"); \\',
            '                iColumn = 0; \\',
            '            } \\',
            '        } while (0)',
        ];

        asLines.extend(['    PRINT_FEATURE(%-*s %s);' % (cchMaxFeatNm + 1, sFeature + ',', g_dSpecFeatToCpumFeat[sFeature])
                        for sFeature in sorted(g_dSpecFeatToCpumFeat.keys())
                        if not isinstance(g_dSpecFeatToCpumFeat[sFeature], bool)]);
        asLines += [
            '}',
            '#endif /* IN_RING3 */',
        ];


        asLines += [
            '',
            '#endif /* !VMM_INCLUDED_SRC_VMMR3_target_armv8_%s */' % (sBlockerName,),
            '',
        ];
        return (not(asTodo), asLines);


    def main(self, asArgs):
        """ Main function. """

        #
        # Parse arguments.
        #
        class MyArgParser(argparse.ArgumentParser):
            def convert_arg_line_to_args(self, arg_line):
                return arg_line.split();

        oArgParser = MyArgParser(fromfile_prefix_chars = '@',
                                 formatter_class = argparse.RawDescriptionHelpFormatter,
                                 epilog = '''
Hints can be extracted from existing code with the following sed command:
    kmk_sed -e "/--decoder-hint/!d" -e "s/^...//" IEMAllIntprA64Tables-armv8.cpp > hints.rsp
Then add @hints.rsp to the command line to make use of them.''');
        oArgParser.add_argument('--tar',
                                metavar = 'AARCHMRS_BSD_A_profile-2024-12.tar.gz',
                                dest    = 'sTarFile',
                                action  = 'store',
                                default = None,
                                help    = 'Specification TAR file to get the files from.');
        oArgParser.add_argument('--instructions',
                                metavar = 'Instructions.json',
                                dest    = 'sFileInstructions',
                                action  = 'store',
                                default = 'Instructions.json',
                                help    = 'The path to the instruction specficiation file.');
        oArgParser.add_argument('--features',
                                metavar = 'Features.json',
                                dest    = 'sFileFeatures',
                                action  = 'store',
                                default = 'Features.json',
                                help    = 'The path to the features specficiation file.');
        oArgParser.add_argument('--registers',
                                metavar = 'Registers.json',
                                dest    = 'sFileRegisters',
                                action  = 'store',
                                default = 'Registers.json',
                                help    = 'The path to the registers specficiation file.');
        oArgParser.add_argument('--spec-dir',
                                metavar = 'dir',
                                dest    = 'sSpecDir',
                                action  = 'store',
                                default = '',
                                help    = 'Specification directory to prefix the specficiation files with.');
        oArgParser.add_argument('--out-decoder-cpp',
                                metavar = 'file-decoder.cpp',
                                dest    = 'sFileDecoderCpp',
                                action  = 'store',
                                default = None,
                                help    = 'The output C++ file for the decoder.');
        oArgParser.add_argument('--out-decoder-hdr',
                                metavar = 'file-decoder.h',
                                dest    = 'sFileDecoderHdr',
                                action  = 'store',
                                default = None,
                                help    = 'The output header file for the decoder.');
        oArgParser.add_argument('--out-stub-hdr',
                                metavar = 'file-stub.h',
                                dest    = 'sFileStubHdr',
                                action  = 'store',
                                default = None,
                                help    = 'The output header file for the implementation stubs.');
        oArgParser.add_argument('--out-cimpl-sysregins-cpp',
                                metavar = 'file-cimpl-sysregins.cpp',
                                dest    = 'sFileCImplSysRegInsCpp',
                                action  = 'store',
                                default = None,
                                help    = 'The output C++ file for system register & instruction handling (MRS, SYS, et al).');
        oArgParser.add_argument('--out-features-hdr',
                                metavar = 'file-features.h',
                                dest    = 'sFileFeaturesHdr',
                                action  = 'store',
                                default = None,
                                help    = 'The output header file for the feature extraction & dumping code.');

        # debug:
        oArgParser.add_argument('--print-instructions',
                                dest    = 'fPrintInstructions',
                                action  = 'store_true',
                                default = False,
                                help    = 'List the instructions after loading.');
        oArgParser.add_argument('--print-instructions-with-conditions',
                                dest    = 'fPrintInstructionsWithConds',
                                action  = 'store_true',
                                default = False,
                                help    = 'List the instructions and conditions after loading.');
        oArgParser.add_argument('--print-instructions-with-encoding',
                                dest    = 'fPrintInstructionsWithEncoding',
                                action  = 'store_true',
                                default = False,
                                help    = 'List the instructions and encoding details after loading.');
        oArgParser.add_argument('--print-fixed-mask-stats',
                                dest    = 'fPrintFixedMaskStats',
                                action  = 'store_true',
                                default = False,
                                help    = 'List statistics on fixed bit masks.');
        oArgParser.add_argument('--print-fixed-mask-top-10',
                                dest    = 'fPrintFixedMaskTop10',
                                action  = 'store_true',
                                default = False,
                                help    = 'List the 10 top fixed bit masks.');
        oArgParser.add_argument('--print-sysregs',
                                dest    = 'fPrintSysRegs',
                                action  = 'store_true',
                                default = False,
                                help    = 'List system registers after loading.');
        # Hacks
        oArgParser.add_argument('--decoder-hint0',
                                metavar = 'mask-to-use',
                                dest    = 'asDecoderHintsL0',
                                action  = 'append',
                                default = [],
                                help    = 'Top level decoder hints used to shorten the runtime. Format: <mask-to-use>');
        oArgParser.add_argument('--decoder-hint1',
                                metavar = 'matched-mask/value/mask-to-use',
                                dest    = 'asDecoderHintsL1',
                                action  = 'append',
                                default = [],
                                help    = 'Level 1 decoder hints. Format: <matched-mask>/<matched-value/<mask-to-use>');
        oArgParser.add_argument('--decoder-hint1-threshold',
                                metavar = 'count',
                                dest    = 'iDecoderL1Threshold',
                                type    = int,
                                action  = 'store',
                                default = 20,
                                help    = 'The instruction threshold count for emitting --decoder-hint1 in the generated code.');

        # Do it!
        oOptions = oArgParser.parse_args(asArgs[1:]);
        self.oOptions = oOptions;

        # Process the decoder hints.
        for sValue in oOptions.asDecoderHintsL0:
            try:
                fTmp = int(sValue, 16);
                if fTmp <= 0 or fTmp >= 0x100000000 or fTmp.bit_count() <= 4 or fTmp.bit_count() >= 16:# pylint: disable=no-member
                    raise Exception();
            except:
                print('syntax error: Invalid --decoder-hint0 value: %s' % (sValue,))
                return 2;
            g_dDecoderFilterDepth0[fTmp] = True;

        for sValue in oOptions.asDecoderHintsL1:
            try:
                afVals = [int(sSub, 16) for sSub in sValue.split('/')];
                for iVal, fTmp in enumerate(afVals):
                    if fTmp < 0 or fTmp >= 0x100000000 or (fTmp == 0 and iVal != 1): raise Exception();
                if (afVals[0] & afVals[1]) != afVals[1] or (afVals[0] & afVals[2]) != 0: raise Exception();
            except:
                print('syntax error: Invalid --decoder-hint1 value: %s' % (sValue,))
                return 2;
            sKey = '%x/%x' % (afVals[0], afVals[1]);
            if sKey not in g_ddDecoderFilterDepth1:
                g_ddDecoderFilterDepth1[sKey] = {afVals[2]: True,};
            else:
                g_ddDecoderFilterDepth1[sKey][afVals[2]] = True;

        #
        # Load the specification.
        #
        if oOptions.sTarFile:
            fRc = spec.loadArmOpenSourceSpecificationFromTar(oOptions.sTarFile, oOptions.sFileInstructions,
                                                             oOptions.sFileFeatures, oOptions.sFileRegisters);
        elif oOptions.sSpecDir:
            fRc = spec.loadArmOpenSourceSpecificationFromDir(oOptions.sSpecDir, oOptions.sFileInstructions,
                                                             oOptions.sFileFeatures, oOptions.sFileRegisters);
        else:
            fRc = spec.loadArmOpenSourceSpecificationFromFiles(oOptions.sFileInstructions, oOptions.sFileFeatures,
                                                               oOptions.sFileRegisters);
        if fRc:
            spec.printSpecs(fPrintInstructions               = oOptions.fPrintInstructions,
                            fPrintInstructionsWithEncoding   = oOptions.fPrintInstructionsWithEncoding,
                            fPrintInstructionsWithConds      = oOptions.fPrintInstructionsWithConds,
                            fPrintFixedMaskStats             = oOptions.fPrintFixedMaskStats,
                            fPrintFixedMaskTop10             = oOptions.fPrintFixedMaskTop10,
                            fPrintSysRegs                    = oOptions.fPrintSysRegs);

            #
            # Check if we're generating any output before constructing the decoder.
            #
            aaoOutputFiles = [
                 ( oOptions.sFileDecoderCpp,        self.generateA64DecoderCpp,            0, 1, ),
                 ( oOptions.sFileDecoderHdr,        self.generateDecoderHdr,               0, 1, ), # after generateA64DecoderCpp!
                 ( oOptions.sFileStubHdr,           self.generateA64ImplementationStubHdr, 0, 1, ),
                 ( oOptions.sFileCImplSysRegInsCpp, self.generateA64CImplSysRegInsCpp,     0, 0, ),
                 ( oOptions.sFileFeaturesHdr,       self.generateFeaturesHdr,              0, 0, ),
            ];

            cOutputFiles        = 0;
            cDecoderOutputFiles = 0;
            for sOutFile, _, _, fDecoder in aaoOutputFiles:
                cOutputFiles        += sOutFile is not None;
                cDecoderOutputFiles += sOutFile is not None and fDecoder;

            fRc = True;
            if cOutputFiles > 0:
                #
                # Sort out the decoding if needed.
                #
                if cDecoderOutputFiles > 0:
                    self.constructDecoder();

                #
                # Output.
                #
                for sOutFile, fnGenMethod, iPartNo, _ in aaoOutputFiles:
                    if not sOutFile:
                        continue;
                    (fRc2, asLines) = fnGenMethod(sOutFile, iPartNo);
                    fRc = fRc2 and fRc;

                    if sOutFile == '-':
                        sys.stdout.write('\n'.join(asLines));
                    else:
                        try:
                            oOut = open(sOutFile, 'w', encoding = 'utf-8', errors = 'strict');
                        except Exception as oXcpt:
                            print('error! Failed open "%s" for writing: %s' % (sOutFile, oXcpt,), file = sys.stderr);
                            return 1;
                        with oOut:
                            oOut.write('\n'.join(asLines));
            if fRc:
                return 0;

        return 1;

if __name__ == '__main__':
    sys.exit(pycmn.mainWrapperCatchXcptAndDoProfiling(IEMArmGenerator().main));

