//===----------------------------------------------------------------------===//
//
// Part of the LLVM Project, under the Apache License v2.0 with LLVM Exceptions.
// See https://llvm.org/LICENSE.txt for license information.
// SPDX-License-Identifier: Apache-2.0 WITH LLVM-exception
//
//===----------------------------------------------------------------------===//

// <string>

// int compare(size_type pos, size_type n1, const charT *s, size_type n2) const;

// When back-deploying to macosx10.7, the RTTI for exception classes
// incorrectly provided by libc++.dylib is mixed with the one in
// libc++abi.dylib and exceptions are not caught properly.
// XFAIL: with_system_cxx_lib=macosx10.7

#include <string>
#include <stdexcept>
#include <cassert>

#include "min_allocator.h"

#include "test_macros.h"

int sign(int x)
{
    if (x == 0)
        return 0;
    if (x < 0)
        return -1;
    return 1;
}

template <class S>
void
test(const S& s, typename S::size_type pos, typename S::size_type n1,
     const typename S::value_type* str, typename S::size_type n2, int x)
{
    if (pos <= s.size())
        assert(sign(s.compare(pos, n1, str, n2)) == sign(x));
#ifndef TEST_HAS_NO_EXCEPTIONS
    else
    {
        try
        {
            TEST_IGNORE_NODISCARD s.compare(pos, n1, str, n2);
            assert(false);
        }
        catch (std::out_of_range&)
        {
            assert(pos > s.size());
        }
    }
#endif
}

template <class S>
void test0()
{
    test(S(""), 0, 0, "", 0, 0);
    test(S(""), 0, 0, "abcde", 0, 0);
    test(S(""), 0, 0, "abcde", 1, -1);
    test(S(""), 0, 0, "abcde", 2, -2);
    test(S(""), 0, 0, "abcde", 4, -4);
    test(S(""), 0, 0, "abcde", 5, -5);
    test(S(""), 0, 0, "abcdefghij", 0, 0);
    test(S(""), 0, 0, "abcdefghij", 1, -1);
    test(S(""), 0, 0, "abcdefghij", 5, -5);
    test(S(""), 0, 0, "abcdefghij", 9, -9);
    test(S(""), 0, 0, "abcdefghij", 10, -10);
    test(S(""), 0, 0, "abcdefghijklmnopqrst", 0, 0);
    test(S(""), 0, 0, "abcdefghijklmnopqrst", 1, -1);
    test(S(""), 0, 0, "abcdefghijklmnopqrst", 10, -10);
    test(S(""), 0, 0, "abcdefghijklmnopqrst", 19, -19);
    test(S(""), 0, 0, "abcdefghijklmnopqrst", 20, -20);
    test(S(""), 0, 1, "", 0, 0);
    test(S(""), 0, 1, "abcde", 0, 0);
    test(S(""), 0, 1, "abcde", 1, -1);
    test(S(""), 0, 1, "abcde", 2, -2);
    test(S(""), 0, 1, "abcde", 4, -4);
    test(S(""), 0, 1, "abcde", 5, -5);
    test(S(""), 0, 1, "abcdefghij", 0, 0);
    test(S(""), 0, 1, "abcdefghij", 1, -1);
    test(S(""), 0, 1, "abcdefghij", 5, -5);
    test(S(""), 0, 1, "abcdefghij", 9, -9);
    test(S(""), 0, 1, "abcdefghij", 10, -10);
    test(S(""), 0, 1, "abcdefghijklmnopqrst", 0, 0);
    test(S(""), 0, 1, "abcdefghijklmnopqrst", 1, -1);
    test(S(""), 0, 1, "abcdefghijklmnopqrst", 10, -10);
    test(S(""), 0, 1, "abcdefghijklmnopqrst", 19, -19);
    test(S(""), 0, 1, "abcdefghijklmnopqrst", 20, -20);
    test(S(""), 1, 0, "", 0, 0);
    test(S(""), 1, 0, "abcde", 0, 0);
    test(S(""), 1, 0, "abcde", 1, 0);
    test(S(""), 1, 0, "abcde", 2, 0);
    test(S(""), 1, 0, "abcde", 4, 0);
    test(S(""), 1, 0, "abcde", 5, 0);
    test(S(""), 1, 0, "abcdefghij", 0, 0);
    test(S(""), 1, 0, "abcdefghij", 1, 0);
    test(S(""), 1, 0, "abcdefghij", 5, 0);
    test(S(""), 1, 0, "abcdefghij", 9, 0);
    test(S(""), 1, 0, "abcdefghij", 10, 0);
    test(S(""), 1, 0, "abcdefghijklmnopqrst", 0, 0);
    test(S(""), 1, 0, "abcdefghijklmnopqrst", 1, 0);
    test(S(""), 1, 0, "abcdefghijklmnopqrst", 10, 0);
    test(S(""), 1, 0, "abcdefghijklmnopqrst", 19, 0);
    test(S(""), 1, 0, "abcdefghijklmnopqrst", 20, 0);
    test(S("abcde"), 0, 0, "", 0, 0);
    test(S("abcde"), 0, 0, "abcde", 0, 0);
    test(S("abcde"), 0, 0, "abcde", 1, -1);
    test(S("abcde"), 0, 0, "abcde", 2, -2);
    test(S("abcde"), 0, 0, "abcde", 4, -4);
    test(S("abcde"), 0, 0, "abcde", 5, -5);
    test(S("abcde"), 0, 0, "abcdefghij", 0, 0);
    test(S("abcde"), 0, 0, "abcdefghij", 1, -1);
    test(S("abcde"), 0, 0, "abcdefghij", 5, -5);
    test(S("abcde"), 0, 0, "abcdefghij", 9, -9);
    test(S("abcde"), 0, 0, "abcdefghij", 10, -10);
    test(S("abcde"), 0, 0, "abcdefghijklmnopqrst", 0, 0);
    test(S("abcde"), 0, 0, "abcdefghijklmnopqrst", 1, -1);
    test(S("abcde"), 0, 0, "abcdefghijklmnopqrst", 10, -10);
    test(S("abcde"), 0, 0, "abcdefghijklmnopqrst", 19, -19);
    test(S("abcde"), 0, 0, "abcdefghijklmnopqrst", 20, -20);
    test(S("abcde"), 0, 1, "", 0, 1);
    test(S("abcde"), 0, 1, "abcde", 0, 1);
    test(S("abcde"), 0, 1, "abcde", 1, 0);
    test(S("abcde"), 0, 1, "abcde", 2, -1);
    test(S("abcde"), 0, 1, "abcde", 4, -3);
    test(S("abcde"), 0, 1, "abcde", 5, -4);
    test(S("abcde"), 0, 1, "abcdefghij", 0, 1);
    test(S("abcde"), 0, 1, "abcdefghij", 1, 0);
    test(S("abcde"), 0, 1, "abcdefghij", 5, -4);
    test(S("abcde"), 0, 1, "abcdefghij", 9, -8);
    test(S("abcde"), 0, 1, "abcdefghij", 10, -9);
    test(S("abcde"), 0, 1, "abcdefghijklmnopqrst", 0, 1);
    test(S("abcde"), 0, 1, "abcdefghijklmnopqrst", 1, 0);
    test(S("abcde"), 0, 1, "abcdefghijklmnopqrst", 10, -9);
    test(S("abcde"), 0, 1, "abcdefghijklmnopqrst", 19, -18);
    test(S("abcde"), 0, 1, "abcdefghijklmnopqrst", 20, -19);
    test(S("abcde"), 0, 2, "", 0, 2);
    test(S("abcde"), 0, 2, "abcde", 0, 2);
    test(S("abcde"), 0, 2, "abcde", 1, 1);
    test(S("abcde"), 0, 2, "abcde", 2, 0);
    test(S("abcde"), 0, 2, "abcde", 4, -2);
    test(S("abcde"), 0, 2, "abcde", 5, -3);
    test(S("abcde"), 0, 2, "abcdefghij", 0, 2);
    test(S("abcde"), 0, 2, "abcdefghij", 1, 1);
    test(S("abcde"), 0, 2, "abcdefghij", 5, -3);
    test(S("abcde"), 0, 2, "abcdefghij", 9, -7);
    test(S("abcde"), 0, 2, "abcdefghij", 10, -8);
    test(S("abcde"), 0, 2, "abcdefghijklmnopqrst", 0, 2);
    test(S("abcde"), 0, 2, "abcdefghijklmnopqrst", 1, 1);
    test(S("abcde"), 0, 2, "abcdefghijklmnopqrst", 10, -8);
    test(S("abcde"), 0, 2, "abcdefghijklmnopqrst", 19, -17);
    test(S("abcde"), 0, 2, "abcdefghijklmnopqrst", 20, -18);
    test(S("abcde"), 0, 4, "", 0, 4);
    test(S("abcde"), 0, 4, "abcde", 0, 4);
    test(S("abcde"), 0, 4, "abcde", 1, 3);
    test(S("abcde"), 0, 4, "abcde", 2, 2);
}

template <class S>
void test1()
{
    test(S("abcde"), 0, 4, "abcde", 4, 0);
    test(S("abcde"), 0, 4, "abcde", 5, -1);
    test(S("abcde"), 0, 4, "abcdefghij", 0, 4);
    test(S("abcde"), 0, 4, "abcdefghij", 1, 3);
    test(S("abcde"), 0, 4, "abcdefghij", 5, -1);
    test(S("abcde"), 0, 4, "abcdefghij", 9, -5);
    test(S("abcde"), 0, 4, "abcdefghij", 10, -6);
    test(S("abcde"), 0, 4, "abcdefghijklmnopqrst", 0, 4);
    test(S("abcde"), 0, 4, "abcdefghijklmnopqrst", 1, 3);
    test(S("abcde"), 0, 4, "abcdefghijklmnopqrst", 10, -6);
    test(S("abcde"), 0, 4, "abcdefghijklmnopqrst", 19, -15);
    test(S("abcde"), 0, 4, "abcdefghijklmnopqrst", 20, -16);
    test(S("abcde"), 0, 5, "", 0, 5);
    test(S("abcde"), 0, 5, "abcde", 0, 5);
    test(S("abcde"), 0, 5, "abcde", 1, 4);
    test(S("abcde"), 0, 5, "abcde", 2, 3);
    test(S("abcde"), 0, 5, "abcde", 4, 1);
    test(S("abcde"), 0, 5, "abcde", 5, 0);
    test(S("abcde"), 0, 5, "abcdefghij", 0, 5);
    test(S("abcde"), 0, 5, "abcdefghij", 1, 4);
    test(S("abcde"), 0, 5, "abcdefghij", 5, 0);
    test(S("abcde"), 0, 5, "abcdefghij", 9, -4);
    test(S("abcde"), 0, 5, "abcdefghij", 10, -5);
    test(S("abcde"), 0, 5, "abcdefghijklmnopqrst", 0, 5);
    test(S("abcde"), 0, 5, "abcdefghijklmnopqrst", 1, 4);
    test(S("abcde"), 0, 5, "abcdefghijklmnopqrst", 10, -5);
    test(S("abcde"), 0, 5, "abcdefghijklmnopqrst", 19, -14);
    test(S("abcde"), 0, 5, "abcdefghijklmnopqrst", 20, -15);
    test(S("abcde"), 0, 6, "", 0, 5);
    test(S("abcde"), 0, 6, "abcde", 0, 5);
    test(S("abcde"), 0, 6, "abcde", 1, 4);
    test(S("abcde"), 0, 6, "abcde", 2, 3);
    test(S("abcde"), 0, 6, "abcde", 4, 1);
    test(S("abcde"), 0, 6, "abcde", 5, 0);
    test(S("abcde"), 0, 6, "abcdefghij", 0, 5);
    test(S("abcde"), 0, 6, "abcdefghij", 1, 4);
    test(S("abcde"), 0, 6, "abcdefghij", 5, 0);
    test(S("abcde"), 0, 6, "abcdefghij", 9, -4);
    test(S("abcde"), 0, 6, "abcdefghij", 10, -5);
    test(S("abcde"), 0, 6, "abcdefghijklmnopqrst", 0, 5);
    test(S("abcde"), 0, 6, "abcdefghijklmnopqrst", 1, 4);
    test(S("abcde"), 0, 6, "abcdefghijklmnopqrst", 10, -5);
    test(S("abcde"), 0, 6, "abcdefghijklmnopqrst", 19, -14);
    test(S("abcde"), 0, 6, "abcdefghijklmnopqrst", 20, -15);
    test(S("abcde"), 1, 0, "", 0, 0);
    test(S("abcde"), 1, 0, "abcde", 0, 0);
    test(S("abcde"), 1, 0, "abcde", 1, -1);
    test(S("abcde"), 1, 0, "abcde", 2, -2);
    test(S("abcde"), 1, 0, "abcde", 4, -4);
    test(S("abcde"), 1, 0, "abcde", 5, -5);
    test(S("abcde"), 1, 0, "abcdefghij", 0, 0);
    test(S("abcde"), 1, 0, "abcdefghij", 1, -1);
    test(S("abcde"), 1, 0, "abcdefghij", 5, -5);
    test(S("abcde"), 1, 0, "abcdefghij", 9, -9);
    test(S("abcde"), 1, 0, "abcdefghij", 10, -10);
    test(S("abcde"), 1, 0, "abcdefghijklmnopqrst", 0, 0);
    test(S("abcde"), 1, 0, "abcdefghijklmnopqrst", 1, -1);
    test(S("abcde"), 1, 0, "abcdefghijklmnopqrst", 10, -10);
    test(S("abcde"), 1, 0, "abcdefghijklmnopqrst", 19, -19);
    test(S("abcde"), 1, 0, "abcdefghijklmnopqrst", 20, -20);
    test(S("abcde"), 1, 1, "", 0, 1);
    test(S("abcde"), 1, 1, "abcde", 0, 1);
    test(S("abcde"), 1, 1, "abcde", 1, 1);
    test(S("abcde"), 1, 1, "abcde", 2, 1);
    test(S("abcde"), 1, 1, "abcde", 4, 1);
    test(S("abcde"), 1, 1, "abcde", 5, 1);
    test(S("abcde"), 1, 1, "abcdefghij", 0, 1);
    test(S("abcde"), 1, 1, "abcdefghij", 1, 1);
    test(S("abcde"), 1, 1, "abcdefghij", 5, 1);
    test(S("abcde"), 1, 1, "abcdefghij", 9, 1);
    test(S("abcde"), 1, 1, "abcdefghij", 10, 1);
    test(S("abcde"), 1, 1, "abcdefghijklmnopqrst", 0, 1);
    test(S("abcde"), 1, 1, "abcdefghijklmnopqrst", 1, 1);
    test(S("abcde"), 1, 1, "abcdefghijklmnopqrst", 10, 1);
    test(S("abcde"), 1, 1, "abcdefghijklmnopqrst", 19, 1);
    test(S("abcde"), 1, 1, "abcdefghijklmnopqrst", 20, 1);
    test(S("abcde"), 1, 2, "", 0, 2);
    test(S("abcde"), 1, 2, "abcde", 0, 2);
    test(S("abcde"), 1, 2, "abcde", 1, 1);
    test(S("abcde"), 1, 2, "abcde", 2, 1);
    test(S("abcde"), 1, 2, "abcde", 4, 1);
    test(S("abcde"), 1, 2, "abcde", 5, 1);
    test(S("abcde"), 1, 2, "abcdefghij", 0, 2);
    test(S("abcde"), 1, 2, "abcdefghij", 1, 1);
    test(S("abcde"), 1, 2, "abcdefghij", 5, 1);
    test(S("abcde"), 1, 2, "abcdefghij", 9, 1);
    test(S("abcde"), 1, 2, "abcdefghij", 10, 1);
    test(S("abcde"), 1, 2, "abcdefghijklmnopqrst", 0, 2);
    test(S("abcde"), 1, 2, "abcdefghijklmnopqrst", 1, 1);
    test(S("abcde"), 1, 2, "abcdefghijklmnopqrst", 10, 1);
    test(S("abcde"), 1, 2, "abcdefghijklmnopqrst", 19, 1);
    test(S("abcde"), 1, 2, "abcdefghijklmnopqrst", 20, 1);
    test(S("abcde"), 1, 3, "", 0, 3);
    test(S("abcde"), 1, 3, "abcde", 0, 3);
    test(S("abcde"), 1, 3, "abcde", 1, 1);
    test(S("abcde"), 1, 3, "abcde", 2, 1);
    test(S("abcde"), 1, 3, "abcde", 4, 1);
    test(S("abcde"), 1, 3, "abcde", 5, 1);
    test(S("abcde"), 1, 3, "abcdefghij", 0, 3);
    test(S("abcde"), 1, 3, "abcdefghij", 1, 1);
}

template <class S>
void test2()
{
    test(S("abcde"), 1, 3, "abcdefghij", 5, 1);
    test(S("abcde"), 1, 3, "abcdefghij", 9, 1);
    test(S("abcde"), 1, 3, "abcdefghij", 10, 1);
    test(S("abcde"), 1, 3, "abcdefghijklmnopqrst", 0, 3);
    test(S("abcde"), 1, 3, "abcdefghijklmnopqrst", 1, 1);
    test(S("abcde"), 1, 3, "abcdefghijklmnopqrst", 10, 1);
    test(S("abcde"), 1, 3, "abcdefghijklmnopqrst", 19, 1);
    test(S("abcde"), 1, 3, "abcdefghijklmnopqrst", 20, 1);
    test(S("abcde"), 1, 4, "", 0, 4);
    test(S("abcde"), 1, 4, "abcde", 0, 4);
    test(S("abcde"), 1, 4, "abcde", 1, 1);
    test(S("abcde"), 1, 4, "abcde", 2, 1);
    test(S("abcde"), 1, 4, "abcde", 4, 1);
    test(S("abcde"), 1, 4, "abcde", 5, 1);
    test(S("abcde"), 1, 4, "abcdefghij", 0, 4);
    test(S("abcde"), 1, 4, "abcdefghij", 1, 1);
    test(S("abcde"), 1, 4, "abcdefghij", 5, 1);
    test(S("abcde"), 1, 4, "abcdefghij", 9, 1);
    test(S("abcde"), 1, 4, "abcdefghij", 10, 1);
    test(S("abcde"), 1, 4, "abcdefghijklmnopqrst", 0, 4);
    test(S("abcde"), 1, 4, "abcdefghijklmnopqrst", 1, 1);
    test(S("abcde"), 1, 4, "abcdefghijklmnopqrst", 10, 1);
    test(S("abcde"), 1, 4, "abcdefghijklmnopqrst", 19, 1);
    test(S("abcde"), 1, 4, "abcdefghijklmnopqrst", 20, 1);
    test(S("abcde"), 1, 5, "", 0, 4);
    test(S("abcde"), 1, 5, "abcde", 0, 4);
    test(S("abcde"), 1, 5, "abcde", 1, 1);
    test(S("abcde"), 1, 5, "abcde", 2, 1);
    test(S("abcde"), 1, 5, "abcde", 4, 1);
    test(S("abcde"), 1, 5, "abcde", 5, 1);
    test(S("abcde"), 1, 5, "abcdefghij", 0, 4);
    test(S("abcde"), 1, 5, "abcdefghij", 1, 1);
    test(S("abcde"), 1, 5, "abcdefghij", 5, 1);
    test(S("abcde"), 1, 5, "abcdefghij", 9, 1);
    test(S("abcde"), 1, 5, "abcdefghij", 10, 1);
    test(S("abcde"), 1, 5, "abcdefghijklmnopqrst", 0, 4);
    test(S("abcde"), 1, 5, "abcdefghijklmnopqrst", 1, 1);
    test(S("abcde"), 1, 5, "abcdefghijklmnopqrst", 10, 1);
    test(S("abcde"), 1, 5, "abcdefghijklmnopqrst", 19, 1);
    test(S("abcde"), 1, 5, "abcdefghijklmnopqrst", 20, 1);
    test(S("abcde"), 2, 0, "", 0, 0);
    test(S("abcde"), 2, 0, "abcde", 0, 0);
    test(S("abcde"), 2, 0, "abcde", 1, -1);
    test(S("abcde"), 2, 0, "abcde", 2, -2);
    test(S("abcde"), 2, 0, "abcde", 4, -4);
    test(S("abcde"), 2, 0, "abcde", 5, -5);
    test(S("abcde"), 2, 0, "abcdefghij", 0, 0);
    test(S("abcde"), 2, 0, "abcdefghij", 1, -1);
    test(S("abcde"), 2, 0, "abcdefghij", 5, -5);
    test(S("abcde"), 2, 0, "abcdefghij", 9, -9);
    test(S("abcde"), 2, 0, "abcdefghij", 10, -10);
    test(S("abcde"), 2, 0, "abcdefghijklmnopqrst", 0, 0);
    test(S("abcde"), 2, 0, "abcdefghijklmnopqrst", 1, -1);
    test(S("abcde"), 2, 0, "abcdefghijklmnopqrst", 10, -10);
    test(S("abcde"), 2, 0, "abcdefghijklmnopqrst", 19, -19);
    test(S("abcde"), 2, 0, "abcdefghijklmnopqrst", 20, -20);
    test(S("abcde"), 2, 1, "", 0, 1);
    test(S("abcde"), 2, 1, "abcde", 0, 1);
    test(S("abcde"), 2, 1, "abcde", 1, 2);
    test(S("abcde"), 2, 1, "abcde", 2, 2);
    test(S("abcde"), 2, 1, "abcde", 4, 2);
    test(S("abcde"), 2, 1, "abcde", 5, 2);
    test(S("abcde"), 2, 1, "abcdefghij", 0, 1);
    test(S("abcde"), 2, 1, "abcdefghij", 1, 2);
    test(S("abcde"), 2, 1, "abcdefghij", 5, 2);
    test(S("abcde"), 2, 1, "abcdefghij", 9, 2);
    test(S("abcde"), 2, 1, "abcdefghij", 10, 2);
    test(S("abcde"), 2, 1, "abcdefghijklmnopqrst", 0, 1);
    test(S("abcde"), 2, 1, "abcdefghijklmnopqrst", 1, 2);
    test(S("abcde"), 2, 1, "abcdefghijklmnopqrst", 10, 2);
    test(S("abcde"), 2, 1, "abcdefghijklmnopqrst", 19, 2);
    test(S("abcde"), 2, 1, "abcdefghijklmnopqrst", 20, 2);
    test(S("abcde"), 2, 2, "", 0, 2);
    test(S("abcde"), 2, 2, "abcde", 0, 2);
    test(S("abcde"), 2, 2, "abcde", 1, 2);
    test(S("abcde"), 2, 2, "abcde", 2, 2);
    test(S("abcde"), 2, 2, "abcde", 4, 2);
    test(S("abcde"), 2, 2, "abcde", 5, 2);
    test(S("abcde"), 2, 2, "abcdefghij", 0, 2);
    test(S("abcde"), 2, 2, "abcdefghij", 1, 2);
    test(S("abcde"), 2, 2, "abcdefghij", 5, 2);
    test(S("abcde"), 2, 2, "abcdefghij", 9, 2);
    test(S("abcde"), 2, 2, "abcdefghij", 10, 2);
    test(S("abcde"), 2, 2, "abcdefghijklmnopqrst", 0, 2);
    test(S("abcde"), 2, 2, "abcdefghijklmnopqrst", 1, 2);
    test(S("abcde"), 2, 2, "abcdefghijklmnopqrst", 10, 2);
    test(S("abcde"), 2, 2, "abcdefghijklmnopqrst", 19, 2);
    test(S("abcde"), 2, 2, "abcdefghijklmnopqrst", 20, 2);
    test(S("abcde"), 2, 3, "", 0, 3);
    test(S("abcde"), 2, 3, "abcde", 0, 3);
    test(S("abcde"), 2, 3, "abcde", 1, 2);
    test(S("abcde"), 2, 3, "abcde", 2, 2);
    test(S("abcde"), 2, 3, "abcde", 4, 2);
    test(S("abcde"), 2, 3, "abcde", 5, 2);
    test(S("abcde"), 2, 3, "abcdefghij", 0, 3);
    test(S("abcde"), 2, 3, "abcdefghij", 1, 2);
    test(S("abcde"), 2, 3, "abcdefghij", 5, 2);
    test(S("abcde"), 2, 3, "abcdefghij", 9, 2);
    test(S("abcde"), 2, 3, "abcdefghij", 10, 2);
    test(S("abcde"), 2, 3, "abcdefghijklmnopqrst", 0, 3);
}

template <class S>
void test3()
{
    test(S("abcde"), 2, 3, "abcdefghijklmnopqrst", 1, 2);
    test(S("abcde"), 2, 3, "abcdefghijklmnopqrst", 10, 2);
    test(S("abcde"), 2, 3, "abcdefghijklmnopqrst", 19, 2);
    test(S("abcde"), 2, 3, "abcdefghijklmnopqrst", 20, 2);
    test(S("abcde"), 2, 4, "", 0, 3);
    test(S("abcde"), 2, 4, "abcde", 0, 3);
    test(S("abcde"), 2, 4, "abcde", 1, 2);
    test(S("abcde"), 2, 4, "abcde", 2, 2);
    test(S("abcde"), 2, 4, "abcde", 4, 2);
    test(S("abcde"), 2, 4, "abcde", 5, 2);
    test(S("abcde"), 2, 4, "abcdefghij", 0, 3);
    test(S("abcde"), 2, 4, "abcdefghij", 1, 2);
    test(S("abcde"), 2, 4, "abcdefghij", 5, 2);
    test(S("abcde"), 2, 4, "abcdefghij", 9, 2);
    test(S("abcde"), 2, 4, "abcdefghij", 10, 2);
    test(S("abcde"), 2, 4, "abcdefghijklmnopqrst", 0, 3);
    test(S("abcde"), 2, 4, "abcdefghijklmnopqrst", 1, 2);
    test(S("abcde"), 2, 4, "abcdefghijklmnopqrst", 10, 2);
    test(S("abcde"), 2, 4, "abcdefghijklmnopqrst", 19, 2);
    test(S("abcde"), 2, 4, "abcdefghijklmnopqrst", 20, 2);
    test(S("abcde"), 4, 0, "", 0, 0);
    test(S("abcde"), 4, 0, "abcde", 0, 0);
    test(S("abcde"), 4, 0, "abcde", 1, -1);
    test(S("abcde"), 4, 0, "abcde", 2, -2);
    test(S("abcde"), 4, 0, "abcde", 4, -4);
    test(S("abcde"), 4, 0, "abcde", 5, -5);
    test(S("abcde"), 4, 0, "abcdefghij", 0, 0);
    test(S("abcde"), 4, 0, "abcdefghij", 1, -1);
    test(S("abcde"), 4, 0, "abcdefghij", 5, -5);
    test(S("abcde"), 4, 0, "abcdefghij", 9, -9);
    test(S("abcde"), 4, 0, "abcdefghij", 10, -10);
    test(S("abcde"), 4, 0, "abcdefghijklmnopqrst", 0, 0);
    test(S("abcde"), 4, 0, "abcdefghijklmnopqrst", 1, -1);
    test(S("abcde"), 4, 0, "abcdefghijklmnopqrst", 10, -10);
    test(S("abcde"), 4, 0, "abcdefghijklmnopqrst", 19, -19);
    test(S("abcde"), 4, 0, "abcdefghijklmnopqrst", 20, -20);
    test(S("abcde"), 4, 1, "", 0, 1);
    test(S("abcde"), 4, 1, "abcde", 0, 1);
    test(S("abcde"), 4, 1, "abcde", 1, 4);
    test(S("abcde"), 4, 1, "abcde", 2, 4);
    test(S("abcde"), 4, 1, "abcde", 4, 4);
    test(S("abcde"), 4, 1, "abcde", 5, 4);
    test(S("abcde"), 4, 1, "abcdefghij", 0, 1);
    test(S("abcde"), 4, 1, "abcdefghij", 1, 4);
    test(S("abcde"), 4, 1, "abcdefghij", 5, 4);
    test(S("abcde"), 4, 1, "abcdefghij", 9, 4);
    test(S("abcde"), 4, 1, "abcdefghij", 10, 4);
    test(S("abcde"), 4, 1, "abcdefghijklmnopqrst", 0, 1);
    test(S("abcde"), 4, 1, "abcdefghijklmnopqrst", 1, 4);
    test(S("abcde"), 4, 1, "abcdefghijklmnopqrst", 10, 4);
    test(S("abcde"), 4, 1, "abcdefghijklmnopqrst", 19, 4);
    test(S("abcde"), 4, 1, "abcdefghijklmnopqrst", 20, 4);
    test(S("abcde"), 4, 2, "", 0, 1);
    test(S("abcde"), 4, 2, "abcde", 0, 1);
    test(S("abcde"), 4, 2, "abcde", 1, 4);
    test(S("abcde"), 4, 2, "abcde", 2, 4);
    test(S("abcde"), 4, 2, "abcde", 4, 4);
    test(S("abcde"), 4, 2, "abcde", 5, 4);
    test(S("abcde"), 4, 2, "abcdefghij", 0, 1);
    test(S("abcde"), 4, 2, "abcdefghij", 1, 4);
    test(S("abcde"), 4, 2, "abcdefghij", 5, 4);
    test(S("abcde"), 4, 2, "abcdefghij", 9, 4);
    test(S("abcde"), 4, 2, "abcdefghij", 10, 4);
    test(S("abcde"), 4, 2, "abcdefghijklmnopqrst", 0, 1);
    test(S("abcde"), 4, 2, "abcdefghijklmnopqrst", 1, 4);
    test(S("abcde"), 4, 2, "abcdefghijklmnopqrst", 10, 4);
    test(S("abcde"), 4, 2, "abcdefghijklmnopqrst", 19, 4);
    test(S("abcde"), 4, 2, "abcdefghijklmnopqrst", 20, 4);
    test(S("abcde"), 5, 0, "", 0, 0);
    test(S("abcde"), 5, 0, "abcde", 0, 0);
    test(S("abcde"), 5, 0, "abcde", 1, -1);
    test(S("abcde"), 5, 0, "abcde", 2, -2);
    test(S("abcde"), 5, 0, "abcde", 4, -4);
    test(S("abcde"), 5, 0, "abcde", 5, -5);
    test(S("abcde"), 5, 0, "abcdefghij", 0, 0);
    test(S("abcde"), 5, 0, "abcdefghij", 1, -1);
    test(S("abcde"), 5, 0, "abcdefghij", 5, -5);
    test(S("abcde"), 5, 0, "abcdefghij", 9, -9);
    test(S("abcde"), 5, 0, "abcdefghij", 10, -10);
    test(S("abcde"), 5, 0, "abcdefghijklmnopqrst", 0, 0);
    test(S("abcde"), 5, 0, "abcdefghijklmnopqrst", 1, -1);
    test(S("abcde"), 5, 0, "abcdefghijklmnopqrst", 10, -10);
    test(S("abcde"), 5, 0, "abcdefghijklmnopqrst", 19, -19);
    test(S("abcde"), 5, 0, "abcdefghijklmnopqrst", 20, -20);
    test(S("abcde"), 5, 1, "", 0, 0);
    test(S("abcde"), 5, 1, "abcde", 0, 0);
    test(S("abcde"), 5, 1, "abcde", 1, -1);
    test(S("abcde"), 5, 1, "abcde", 2, -2);
    test(S("abcde"), 5, 1, "abcde", 4, -4);
    test(S("abcde"), 5, 1, "abcde", 5, -5);
    test(S("abcde"), 5, 1, "abcdefghij", 0, 0);
    test(S("abcde"), 5, 1, "abcdefghij", 1, -1);
    test(S("abcde"), 5, 1, "abcdefghij", 5, -5);
    test(S("abcde"), 5, 1, "abcdefghij", 9, -9);
    test(S("abcde"), 5, 1, "abcdefghij", 10, -10);
    test(S("abcde"), 5, 1, "abcdefghijklmnopqrst", 0, 0);
    test(S("abcde"), 5, 1, "abcdefghijklmnopqrst", 1, -1);
    test(S("abcde"), 5, 1, "abcdefghijklmnopqrst", 10, -10);
    test(S("abcde"), 5, 1, "abcdefghijklmnopqrst", 19, -19);
    test(S("abcde"), 5, 1, "abcdefghijklmnopqrst", 20, -20);
}

template <class S>
void test4()
{
    test(S("abcde"), 6, 0, "", 0, 0);
    test(S("abcde"), 6, 0, "abcde", 0, 0);
    test(S("abcde"), 6, 0, "abcde", 1, 0);
    test(S("abcde"), 6, 0, "abcde", 2, 0);
    test(S("abcde"), 6, 0, "abcde", 4, 0);
    test(S("abcde"), 6, 0, "abcde", 5, 0);
    test(S("abcde"), 6, 0, "abcdefghij", 0, 0);
    test(S("abcde"), 6, 0, "abcdefghij", 1, 0);
    test(S("abcde"), 6, 0, "abcdefghij", 5, 0);
    test(S("abcde"), 6, 0, "abcdefghij", 9, 0);
    test(S("abcde"), 6, 0, "abcdefghij", 10, 0);
    test(S("abcde"), 6, 0, "abcdefghijklmnopqrst", 0, 0);
    test(S("abcde"), 6, 0, "abcdefghijklmnopqrst", 1, 0);
    test(S("abcde"), 6, 0, "abcdefghijklmnopqrst", 10, 0);
    test(S("abcde"), 6, 0, "abcdefghijklmnopqrst", 19, 0);
    test(S("abcde"), 6, 0, "abcdefghijklmnopqrst", 20, 0);
    test(S("abcdefghij"), 0, 0, "", 0, 0);
    test(S("abcdefghij"), 0, 0, "abcde", 0, 0);
    test(S("abcdefghij"), 0, 0, "abcde", 1, -1);
    test(S("abcdefghij"), 0, 0, "abcde", 2, -2);
    test(S("abcdefghij"), 0, 0, "abcde", 4, -4);
    test(S("abcdefghij"), 0, 0, "abcde", 5, -5);
    test(S("abcdefghij"), 0, 0, "abcdefghij", 0, 0);
    test(S("abcdefghij"), 0, 0, "abcdefghij", 1, -1);
    test(S("abcdefghij"), 0, 0, "abcdefghij", 5, -5);
    test(S("abcdefghij"), 0, 0, "abcdefghij", 9, -9);
    test(S("abcdefghij"), 0, 0, "abcdefghij", 10, -10);
    test(S("abcdefghij"), 0, 0, "abcdefghijklmnopqrst", 0, 0);
    test(S("abcdefghij"), 0, 0, "abcdefghijklmnopqrst", 1, -1);
    test(S("abcdefghij"), 0, 0, "abcdefghijklmnopqrst", 10, -10);
    test(S("abcdefghij"), 0, 0, "abcdefghijklmnopqrst", 19, -19);
    test(S("abcdefghij"), 0, 0, "abcdefghijklmnopqrst", 20, -20);
    test(S("abcdefghij"), 0, 1, "", 0, 1);
    test(S("abcdefghij"), 0, 1, "abcde", 0, 1);
    test(S("abcdefghij"), 0, 1, "abcde", 1, 0);
    test(S("abcdefghij"), 0, 1, "abcde", 2, -1);
    test(S("abcdefghij"), 0, 1, "abcde", 4, -3);
    test(S("abcdefghij"), 0, 1, "abcde", 5, -4);
    test(S("abcdefghij"), 0, 1, "abcdefghij", 0, 1);
    test(S("abcdefghij"), 0, 1, "abcdefghij", 1, 0);
    test(S("abcdefghij"), 0, 1, "abcdefghij", 5, -4);
    test(S("abcdefghij"), 0, 1, "abcdefghij", 9, -8);
    test(S("abcdefghij"), 0, 1, "abcdefghij", 10, -9);
    test(S("abcdefghij"), 0, 1, "abcdefghijklmnopqrst", 0, 1);
    test(S("abcdefghij"), 0, 1, "abcdefghijklmnopqrst", 1, 0);
    test(S("abcdefghij"), 0, 1, "abcdefghijklmnopqrst", 10, -9);
    test(S("abcdefghij"), 0, 1, "abcdefghijklmnopqrst", 19, -18);
    test(S("abcdefghij"), 0, 1, "abcdefghijklmnopqrst", 20, -19);
    test(S("abcdefghij"), 0, 5, "", 0, 5);
    test(S("abcdefghij"), 0, 5, "abcde", 0, 5);
    test(S("abcdefghij"), 0, 5, "abcde", 1, 4);
    test(S("abcdefghij"), 0, 5, "abcde", 2, 3);
    test(S("abcdefghij"), 0, 5, "abcde", 4, 1);
    test(S("abcdefghij"), 0, 5, "abcde", 5, 0);
    test(S("abcdefghij"), 0, 5, "abcdefghij", 0, 5);
    test(S("abcdefghij"), 0, 5, "abcdefghij", 1, 4);
    test(S("abcdefghij"), 0, 5, "abcdefghij", 5, 0);
    test(S("abcdefghij"), 0, 5, "abcdefghij", 9, -4);
    test(S("abcdefghij"), 0, 5, "abcdefghij", 10, -5);
    test(S("abcdefghij"), 0, 5, "abcdefghijklmnopqrst", 0, 5);
    test(S("abcdefghij"), 0, 5, "abcdefghijklmnopqrst", 1, 4);
    test(S("abcdefghij"), 0, 5, "abcdefghijklmnopqrst", 10, -5);
    test(S("abcdefghij"), 0, 5, "abcdefghijklmnopqrst", 19, -14);
    test(S("abcdefghij"), 0, 5, "abcdefghijklmnopqrst", 20, -15);
    test(S("abcdefghij"), 0, 9, "", 0, 9);
    test(S("abcdefghij"), 0, 9, "abcde", 0, 9);
    test(S("abcdefghij"), 0, 9, "abcde", 1, 8);
    test(S("abcdefghij"), 0, 9, "abcde", 2, 7);
    test(S("abcdefghij"), 0, 9, "abcde", 4, 5);
    test(S("abcdefghij"), 0, 9, "abcde", 5, 4);
    test(S("abcdefghij"), 0, 9, "abcdefghij", 0, 9);
    test(S("abcdefghij"), 0, 9, "abcdefghij", 1, 8);
    test(S("abcdefghij"), 0, 9, "abcdefghij", 5, 4);
    test(S("abcdefghij"), 0, 9, "abcdefghij", 9, 0);
    test(S("abcdefghij"), 0, 9, "abcdefghij", 10, -1);
    test(S("abcdefghij"), 0, 9, "abcdefghijklmnopqrst", 0, 9);
    test(S("abcdefghij"), 0, 9, "abcdefghijklmnopqrst", 1, 8);
    test(S("abcdefghij"), 0, 9, "abcdefghijklmnopqrst", 10, -1);
    test(S("abcdefghij"), 0, 9, "abcdefghijklmnopqrst", 19, -10);
    test(S("abcdefghij"), 0, 9, "abcdefghijklmnopqrst", 20, -11);
    test(S("abcdefghij"), 0, 10, "", 0, 10);
    test(S("abcdefghij"), 0, 10, "abcde", 0, 10);
    test(S("abcdefghij"), 0, 10, "abcde", 1, 9);
    test(S("abcdefghij"), 0, 10, "abcde", 2, 8);
    test(S("abcdefghij"), 0, 10, "abcde", 4, 6);
    test(S("abcdefghij"), 0, 10, "abcde", 5, 5);
    test(S("abcdefghij"), 0, 10, "abcdefghij", 0, 10);
    test(S("abcdefghij"), 0, 10, "abcdefghij", 1, 9);
    test(S("abcdefghij"), 0, 10, "abcdefghij", 5, 5);
    test(S("abcdefghij"), 0, 10, "abcdefghij", 9, 1);
    test(S("abcdefghij"), 0, 10, "abcdefghij", 10, 0);
    test(S("abcdefghij"), 0, 10, "abcdefghijklmnopqrst", 0, 10);
    test(S("abcdefghij"), 0, 10, "abcdefghijklmnopqrst", 1, 9);
    test(S("abcdefghij"), 0, 10, "abcdefghijklmnopqrst", 10, 0);
    test(S("abcdefghij"), 0, 10, "abcdefghijklmnopqrst", 19, -9);
    test(S("abcdefghij"), 0, 10, "abcdefghijklmnopqrst", 20, -10);
    test(S("abcdefghij"), 0, 11, "", 0, 10);
    test(S("abcdefghij"), 0, 11, "abcde", 0, 10);
    test(S("abcdefghij"), 0, 11, "abcde", 1, 9);
    test(S("abcdefghij"), 0, 11, "abcde", 2, 8);
}

template <class S>
void test5()
{
    test(S("abcdefghij"), 0, 11, "abcde", 4, 6);
    test(S("abcdefghij"), 0, 11, "abcde", 5, 5);
    test(S("abcdefghij"), 0, 11, "abcdefghij", 0, 10);
    test(S("abcdefghij"), 0, 11, "abcdefghij", 1, 9);
    test(S("abcdefghij"), 0, 11, "abcdefghij", 5, 5);
    test(S("abcdefghij"), 0, 11, "abcdefghij", 9, 1);
    test(S("abcdefghij"), 0, 11, "abcdefghij", 10, 0);
    test(S("abcdefghij"), 0, 11, "abcdefghijklmnopqrst", 0, 10);
    test(S("abcdefghij"), 0, 11, "abcdefghijklmnopqrst", 1, 9);
    test(S("abcdefghij"), 0, 11, "abcdefghijklmnopqrst", 10, 0);
    test(S("abcdefghij"), 0, 11, "abcdefghijklmnopqrst", 19, -9);
    test(S("abcdefghij"), 0, 11, "abcdefghijklmnopqrst", 20, -10);
    test(S("abcdefghij"), 1, 0, "", 0, 0);
    test(S("abcdefghij"), 1, 0, "abcde", 0, 0);
    test(S("abcdefghij"), 1, 0, "abcde", 1, -1);
    test(S("abcdefghij"), 1, 0, "abcde", 2, -2);
    test(S("abcdefghij"), 1, 0, "abcde", 4, -4);
    test(S("abcdefghij"), 1, 0, "abcde", 5, -5);
    test(S("abcdefghij"), 1, 0, "abcdefghij", 0, 0);
    test(S("abcdefghij"), 1, 0, "abcdefghij", 1, -1);
    test(S("abcdefghij"), 1, 0, "abcdefghij", 5, -5);
    test(S("abcdefghij"), 1, 0, "abcdefghij", 9, -9);
    test(S("abcdefghij"), 1, 0, "abcdefghij", 10, -10);
    test(S("abcdefghij"), 1, 0, "abcdefghijklmnopqrst", 0, 0);
    test(S("abcdefghij"), 1, 0, "abcdefghijklmnopqrst", 1, -1);
    test(S("abcdefghij"), 1, 0, "abcdefghijklmnopqrst", 10, -10);
    test(S("abcdefghij"), 1, 0, "abcdefghijklmnopqrst", 19, -19);
    test(S("abcdefghij"), 1, 0, "abcdefghijklmnopqrst", 20, -20);
    test(S("abcdefghij"), 1, 1, "", 0, 1);
    test(S("abcdefghij"), 1, 1, "abcde", 0, 1);
    test(S("abcdefghij"), 1, 1, "abcde", 1, 1);
    test(S("abcdefghij"), 1, 1, "abcde", 2, 1);
    test(S("abcdefghij"), 1, 1, "abcde", 4, 1);
    test(S("abcdefghij"), 1, 1, "abcde", 5, 1);
    test(S("abcdefghij"), 1, 1, "abcdefghij", 0, 1);
    test(S("abcdefghij"), 1, 1, "abcdefghij", 1, 1);
    test(S("abcdefghij"), 1, 1, "abcdefghij", 5, 1);
    test(S("abcdefghij"), 1, 1, "abcdefghij", 9, 1);
    test(S("abcdefghij"), 1, 1, "abcdefghij", 10, 1);
    test(S("abcdefghij"), 1, 1, "abcdefghijklmnopqrst", 0, 1);
    test(S("abcdefghij"), 1, 1, "abcdefghijklmnopqrst", 1, 1);
    test(S("abcdefghij"), 1, 1, "abcdefghijklmnopqrst", 10, 1);
    test(S("abcdefghij"), 1, 1, "abcdefghijklmnopqrst", 19, 1);
    test(S("abcdefghij"), 1, 1, "abcdefghijklmnopqrst", 20, 1);
    test(S("abcdefghij"), 1, 4, "", 0, 4);
    test(S("abcdefghij"), 1, 4, "abcde", 0, 4);
    test(S("abcdefghij"), 1, 4, "abcde", 1, 1);
    test(S("abcdefghij"), 1, 4, "abcde", 2, 1);
    test(S("abcdefghij"), 1, 4, "abcde", 4, 1);
    test(S("abcdefghij"), 1, 4, "abcde", 5, 1);
    test(S("abcdefghij"), 1, 4, "abcdefghij", 0, 4);
    test(S("abcdefghij"), 1, 4, "abcdefghij", 1, 1);
    test(S("abcdefghij"), 1, 4, "abcdefghij", 5, 1);
    test(S("abcdefghij"), 1, 4, "abcdefghij", 9, 1);
    test(S("abcdefghij"), 1, 4, "abcdefghij", 10, 1);
    test(S("abcdefghij"), 1, 4, "abcdefghijklmnopqrst", 0, 4);
    test(S("abcdefghij"), 1, 4, "abcdefghijklmnopqrst", 1, 1);
    test(S("abcdefghij"), 1, 4, "abcdefghijklmnopqrst", 10, 1);
    test(S("abcdefghij"), 1, 4, "abcdefghijklmnopqrst", 19, 1);
    test(S("abcdefghij"), 1, 4, "abcdefghijklmnopqrst", 20, 1);
    test(S("abcdefghij"), 1, 8, "", 0, 8);
    test(S("abcdefghij"), 1, 8, "abcde", 0, 8);
    test(S("abcdefghij"), 1, 8, "abcde", 1, 1);
    test(S("abcdefghij"), 1, 8, "abcde", 2, 1);
    test(S("abcdefghij"), 1, 8, "abcde", 4, 1);
    test(S("abcdefghij"), 1, 8, "abcde", 5, 1);
    test(S("abcdefghij"), 1, 8, "abcdefghij", 0, 8);
    test(S("abcdefghij"), 1, 8, "abcdefghij", 1, 1);
    test(S("abcdefghij"), 1, 8, "abcdefghij", 5, 1);
    test(S("abcdefghij"), 1, 8, "abcdefghij", 9, 1);
    test(S("abcdefghij"), 1, 8, "abcdefghij", 10, 1);
    test(S("abcdefghij"), 1, 8, "abcdefghijklmnopqrst", 0, 8);
    test(S("abcdefghij"), 1, 8, "abcdefghijklmnopqrst", 1, 1);
    test(S("abcdefghij"), 1, 8, "abcdefghijklmnopqrst", 10, 1);
    test(S("abcdefghij"), 1, 8, "abcdefghijklmnopqrst", 19, 1);
    test(S("abcdefghij"), 1, 8, "abcdefghijklmnopqrst", 20, 1);
    test(S("abcdefghij"), 1, 9, "", 0, 9);
    test(S("abcdefghij"), 1, 9, "abcde", 0, 9);
    test(S("abcdefghij"), 1, 9, "abcde", 1, 1);
    test(S("abcdefghij"), 1, 9, "abcde", 2, 1);
    test(S("abcdefghij"), 1, 9, "abcde", 4, 1);
    test(S("abcdefghij"), 1, 9, "abcde", 5, 1);
    test(S("abcdefghij"), 1, 9, "abcdefghij", 0, 9);
    test(S("abcdefghij"), 1, 9, "abcdefghij", 1, 1);
    test(S("abcdefghij"), 1, 9, "abcdefghij", 5, 1);
    test(S("abcdefghij"), 1, 9, "abcdefghij", 9, 1);
    test(S("abcdefghij"), 1, 9, "abcdefghij", 10, 1);
    test(S("abcdefghij"), 1, 9, "abcdefghijklmnopqrst", 0, 9);
    test(S("abcdefghij"), 1, 9, "abcdefghijklmnopqrst", 1, 1);
    test(S("abcdefghij"), 1, 9, "abcdefghijklmnopqrst", 10, 1);
    test(S("abcdefghij"), 1, 9, "abcdefghijklmnopqrst", 19, 1);
    test(S("abcdefghij"), 1, 9, "abcdefghijklmnopqrst", 20, 1);
    test(S("abcdefghij"), 1, 10, "", 0, 9);
    test(S("abcdefghij"), 1, 10, "abcde", 0, 9);
    test(S("abcdefghij"), 1, 10, "abcde", 1, 1);
    test(S("abcdefghij"), 1, 10, "abcde", 2, 1);
    test(S("abcdefghij"), 1, 10, "abcde", 4, 1);
    test(S("abcdefghij"), 1, 10, "abcde", 5, 1);
    test(S("abcdefghij"), 1, 10, "abcdefghij", 0, 9);
    test(S("abcdefghij"), 1, 10, "abcdefghij", 1, 1);
}

template <class S>
void test6()
{
    test(S("abcdefghij"), 1, 10, "abcdefghij", 5, 1);
    test(S("abcdefghij"), 1, 10, "abcdefghij", 9, 1);
    test(S("abcdefghij"), 1, 10, "abcdefghij", 10, 1);
    test(S("abcdefghij"), 1, 10, "abcdefghijklmnopqrst", 0, 9);
    test(S("abcdefghij"), 1, 10, "abcdefghijklmnopqrst", 1, 1);
    test(S("abcdefghij"), 1, 10, "abcdefghijklmnopqrst", 10, 1);
    test(S("abcdefghij"), 1, 10, "abcdefghijklmnopqrst", 19, 1);
    test(S("abcdefghij"), 1, 10, "abcdefghijklmnopqrst", 20, 1);
    test(S("abcdefghij"), 5, 0, "", 0, 0);
    test(S("abcdefghij"), 5, 0, "abcde", 0, 0);
    test(S("abcdefghij"), 5, 0, "abcde", 1, -1);
    test(S("abcdefghij"), 5, 0, "abcde", 2, -2);
    test(S("abcdefghij"), 5, 0, "abcde", 4, -4);
    test(S("abcdefghij"), 5, 0, "abcde", 5, -5);
    test(S("abcdefghij"), 5, 0, "abcdefghij", 0, 0);
    test(S("abcdefghij"), 5, 0, "abcdefghij", 1, -1);
    test(S("abcdefghij"), 5, 0, "abcdefghij", 5, -5);
    test(S("abcdefghij"), 5, 0, "abcdefghij", 9, -9);
    test(S("abcdefghij"), 5, 0, "abcdefghij", 10, -10);
    test(S("abcdefghij"), 5, 0, "abcdefghijklmnopqrst", 0, 0);
    test(S("abcdefghij"), 5, 0, "abcdefghijklmnopqrst", 1, -1);
    test(S("abcdefghij"), 5, 0, "abcdefghijklmnopqrst", 10, -10);
    test(S("abcdefghij"), 5, 0, "abcdefghijklmnopqrst", 19, -19);
    test(S("abcdefghij"), 5, 0, "abcdefghijklmnopqrst", 20, -20);
    test(S("abcdefghij"), 5, 1, "", 0, 1);
    test(S("abcdefghij"), 5, 1, "abcde", 0, 1);
    test(S("abcdefghij"), 5, 1, "abcde", 1, 5);
    test(S("abcdefghij"), 5, 1, "abcde", 2, 5);
    test(S("abcdefghij"), 5, 1, "abcde", 4, 5);
    test(S("abcdefghij"), 5, 1, "abcde", 5, 5);
    test(S("abcdefghij"), 5, 1, "abcdefghij", 0, 1);
    test(S("abcdefghij"), 5, 1, "abcdefghij", 1, 5);
    test(S("abcdefghij"), 5, 1, "abcdefghij", 5, 5);
    test(S("abcdefghij"), 5, 1, "abcdefghij", 9, 5);
    test(S("abcdefghij"), 5, 1, "abcdefghij", 10, 5);
    test(S("abcdefghij"), 5, 1, "abcdefghijklmnopqrst", 0, 1);
    test(S("abcdefghij"), 5, 1, "abcdefghijklmnopqrst", 1, 5);
    test(S("abcdefghij"), 5, 1, "abcdefghijklmnopqrst", 10, 5);
    test(S("abcdefghij"), 5, 1, "abcdefghijklmnopqrst", 19, 5);
    test(S("abcdefghij"), 5, 1, "abcdefghijklmnopqrst", 20, 5);
    test(S("abcdefghij"), 5, 2, "", 0, 2);
    test(S("abcdefghij"), 5, 2, "abcde", 0, 2);
    test(S("abcdefghij"), 5, 2, "abcde", 1, 5);
    test(S("abcdefghij"), 5, 2, "abcde", 2, 5);
    test(S("abcdefghij"), 5, 2, "abcde", 4, 5);
    test(S("abcdefghij"), 5, 2, "abcde", 5, 5);
    test(S("abcdefghij"), 5, 2, "abcdefghij", 0, 2);
    test(S("abcdefghij"), 5, 2, "abcdefghij", 1, 5);
    test(S("abcdefghij"), 5, 2, "abcdefghij", 5, 5);
    test(S("abcdefghij"), 5, 2, "abcdefghij", 9, 5);
    test(S("abcdefghij"), 5, 2, "abcdefghij", 10, 5);
    test(S("abcdefghij"), 5, 2, "abcdefghijklmnopqrst", 0, 2);
    test(S("abcdefghij"), 5, 2, "abcdefghijklmnopqrst", 1, 5);
    test(S("abcdefghij"), 5, 2, "abcdefghijklmnopqrst", 10, 5);
    test(S("abcdefghij"), 5, 2, "abcdefghijklmnopqrst", 19, 5);
    test(S("abcdefghij"), 5, 2, "abcdefghijklmnopqrst", 20, 5);
    test(S("abcdefghij"), 5, 4, "", 0, 4);
    test(S("abcdefghij"), 5, 4, "abcde", 0, 4);
    test(S("abcdefghij"), 5, 4, "abcde", 1, 5);
    test(S("abcdefghij"), 5, 4, "abcde", 2, 5);
    test(S("abcdefghij"), 5, 4, "abcde", 4, 5);
    test(S("abcdefghij"), 5, 4, "abcde", 5, 5);
    test(S("abcdefghij"), 5, 4, "abcdefghij", 0, 4);
    test(S("abcdefghij"), 5, 4, "abcdefghij", 1, 5);
    test(S("abcdefghij"), 5, 4, "abcdefghij", 5, 5);
    test(S("abcdefghij"), 5, 4, "abcdefghij", 9, 5);
    test(S("abcdefghij"), 5, 4, "abcdefghij", 10, 5);
    test(S("abcdefghij"), 5, 4, "abcdefghijklmnopqrst", 0, 4);
    test(S("abcdefghij"), 5, 4, "abcdefghijklmnopqrst", 1, 5);
    test(S("abcdefghij"), 5, 4, "abcdefghijklmnopqrst", 10, 5);
    test(S("abcdefghij"), 5, 4, "abcdefghijklmnopqrst", 19, 5);
    test(S("abcdefghij"), 5, 4, "abcdefghijklmnopqrst", 20, 5);
    test(S("abcdefghij"), 5, 5, "", 0, 5);
    test(S("abcdefghij"), 5, 5, "abcde", 0, 5);
    test(S("abcdefghij"), 5, 5, "abcde", 1, 5);
    test(S("abcdefghij"), 5, 5, "abcde", 2, 5);
    test(S("abcdefghij"), 5, 5, "abcde", 4, 5);
    test(S("abcdefghij"), 5, 5, "abcde", 5, 5);
    test(S("abcdefghij"), 5, 5, "abcdefghij", 0, 5);
    test(S("abcdefghij"), 5, 5, "abcdefghij", 1, 5);
    test(S("abcdefghij"), 5, 5, "abcdefghij", 5, 5);
    test(S("abcdefghij"), 5, 5, "abcdefghij", 9, 5);
    test(S("abcdefghij"), 5, 5, "abcdefghij", 10, 5);
    test(S("abcdefghij"), 5, 5, "abcdefghijklmnopqrst", 0, 5);
    test(S("abcdefghij"), 5, 5, "abcdefghijklmnopqrst", 1, 5);
    test(S("abcdefghij"), 5, 5, "abcdefghijklmnopqrst", 10, 5);
    test(S("abcdefghij"), 5, 5, "abcdefghijklmnopqrst", 19, 5);
    test(S("abcdefghij"), 5, 5, "abcdefghijklmnopqrst", 20, 5);
    test(S("abcdefghij"), 5, 6, "", 0, 5);
    test(S("abcdefghij"), 5, 6, "abcde", 0, 5);
    test(S("abcdefghij"), 5, 6, "abcde", 1, 5);
    test(S("abcdefghij"), 5, 6, "abcde", 2, 5);
    test(S("abcdefghij"), 5, 6, "abcde", 4, 5);
    test(S("abcdefghij"), 5, 6, "abcde", 5, 5);
    test(S("abcdefghij"), 5, 6, "abcdefghij", 0, 5);
    test(S("abcdefghij"), 5, 6, "abcdefghij", 1, 5);
    test(S("abcdefghij"), 5, 6, "abcdefghij", 5, 5);
    test(S("abcdefghij"), 5, 6, "abcdefghij", 9, 5);
    test(S("abcdefghij"), 5, 6, "abcdefghij", 10, 5);
    test(S("abcdefghij"), 5, 6, "abcdefghijklmnopqrst", 0, 5);
}

template <class S>
void test7()
{
    test(S("abcdefghij"), 5, 6, "abcdefghijklmnopqrst", 1, 5);
    test(S("abcdefghij"), 5, 6, "abcdefghijklmnopqrst", 10, 5);
    test(S("abcdefghij"), 5, 6, "abcdefghijklmnopqrst", 19, 5);
    test(S("abcdefghij"), 5, 6, "abcdefghijklmnopqrst", 20, 5);
    test(S("abcdefghij"), 9, 0, "", 0, 0);
    test(S("abcdefghij"), 9, 0, "abcde", 0, 0);
    test(S("abcdefghij"), 9, 0, "abcde", 1, -1);
    test(S("abcdefghij"), 9, 0, "abcde", 2, -2);
    test(S("abcdefghij"), 9, 0, "abcde", 4, -4);
    test(S("abcdefghij"), 9, 0, "abcde", 5, -5);
    test(S("abcdefghij"), 9, 0, "abcdefghij", 0, 0);
    test(S("abcdefghij"), 9, 0, "abcdefghij", 1, -1);
    test(S("abcdefghij"), 9, 0, "abcdefghij", 5, -5);
    test(S("abcdefghij"), 9, 0, "abcdefghij", 9, -9);
    test(S("abcdefghij"), 9, 0, "abcdefghij", 10, -10);
    test(S("abcdefghij"), 9, 0, "abcdefghijklmnopqrst", 0, 0);
    test(S("abcdefghij"), 9, 0, "abcdefghijklmnopqrst", 1, -1);
    test(S("abcdefghij"), 9, 0, "abcdefghijklmnopqrst", 10, -10);
    test(S("abcdefghij"), 9, 0, "abcdefghijklmnopqrst", 19, -19);
    test(S("abcdefghij"), 9, 0, "abcdefghijklmnopqrst", 20, -20);
    test(S("abcdefghij"), 9, 1, "", 0, 1);
    test(S("abcdefghij"), 9, 1, "abcde", 0, 1);
    test(S("abcdefghij"), 9, 1, "abcde", 1, 9);
    test(S("abcdefghij"), 9, 1, "abcde", 2, 9);
    test(S("abcdefghij"), 9, 1, "abcde", 4, 9);
    test(S("abcdefghij"), 9, 1, "abcde", 5, 9);
    test(S("abcdefghij"), 9, 1, "abcdefghij", 0, 1);
    test(S("abcdefghij"), 9, 1, "abcdefghij", 1, 9);
    test(S("abcdefghij"), 9, 1, "abcdefghij", 5, 9);
    test(S("abcdefghij"), 9, 1, "abcdefghij", 9, 9);
    test(S("abcdefghij"), 9, 1, "abcdefghij", 10, 9);
    test(S("abcdefghij"), 9, 1, "abcdefghijklmnopqrst", 0, 1);
    test(S("abcdefghij"), 9, 1, "abcdefghijklmnopqrst", 1, 9);
    test(S("abcdefghij"), 9, 1, "abcdefghijklmnopqrst", 10, 9);
    test(S("abcdefghij"), 9, 1, "abcdefghijklmnopqrst", 19, 9);
    test(S("abcdefghij"), 9, 1, "abcdefghijklmnopqrst", 20, 9);
    test(S("abcdefghij"), 9, 2, "", 0, 1);
    test(S("abcdefghij"), 9, 2, "abcde", 0, 1);
    test(S("abcdefghij"), 9, 2, "abcde", 1, 9);
    test(S("abcdefghij"), 9, 2, "abcde", 2, 9);
    test(S("abcdefghij"), 9, 2, "abcde", 4, 9);
    test(S("abcdefghij"), 9, 2, "abcde", 5, 9);
    test(S("abcdefghij"), 9, 2, "abcdefghij", 0, 1);
    test(S("abcdefghij"), 9, 2, "abcdefghij", 1, 9);
    test(S("abcdefghij"), 9, 2, "abcdefghij", 5, 9);
    test(S("abcdefghij"), 9, 2, "abcdefghij", 9, 9);
    test(S("abcdefghij"), 9, 2, "abcdefghij", 10, 9);
    test(S("abcdefghij"), 9, 2, "abcdefghijklmnopqrst", 0, 1);
    test(S("abcdefghij"), 9, 2, "abcdefghijklmnopqrst", 1, 9);
    test(S("abcdefghij"), 9, 2, "abcdefghijklmnopqrst", 10, 9);
    test(S("abcdefghij"), 9, 2, "abcdefghijklmnopqrst", 19, 9);
    test(S("abcdefghij"), 9, 2, "abcdefghijklmnopqrst", 20, 9);
    test(S("abcdefghij"), 10, 0, "", 0, 0);
    test(S("abcdefghij"), 10, 0, "abcde", 0, 0);
    test(S("abcdefghij"), 10, 0, "abcde", 1, -1);
    test(S("abcdefghij"), 10, 0, "abcde", 2, -2);
    test(S("abcdefghij"), 10, 0, "abcde", 4, -4);
    test(S("abcdefghij"), 10, 0, "abcde", 5, -5);
    test(S("abcdefghij"), 10, 0, "abcdefghij", 0, 0);
    test(S("abcdefghij"), 10, 0, "abcdefghij", 1, -1);
    test(S("abcdefghij"), 10, 0, "abcdefghij", 5, -5);
    test(S("abcdefghij"), 10, 0, "abcdefghij", 9, -9);
    test(S("abcdefghij"), 10, 0, "abcdefghij", 10, -10);
    test(S("abcdefghij"), 10, 0, "abcdefghijklmnopqrst", 0, 0);
    test(S("abcdefghij"), 10, 0, "abcdefghijklmnopqrst", 1, -1);
    test(S("abcdefghij"), 10, 0, "abcdefghijklmnopqrst", 10, -10);
    test(S("abcdefghij"), 10, 0, "abcdefghijklmnopqrst", 19, -19);
    test(S("abcdefghij"), 10, 0, "abcdefghijklmnopqrst", 20, -20);
    test(S("abcdefghij"), 10, 1, "", 0, 0);
    test(S("abcdefghij"), 10, 1, "abcde", 0, 0);
    test(S("abcdefghij"), 10, 1, "abcde", 1, -1);
    test(S("abcdefghij"), 10, 1, "abcde", 2, -2);
    test(S("abcdefghij"), 10, 1, "abcde", 4, -4);
    test(S("abcdefghij"), 10, 1, "abcde", 5, -5);
    test(S("abcdefghij"), 10, 1, "abcdefghij", 0, 0);
    test(S("abcdefghij"), 10, 1, "abcdefghij", 1, -1);
    test(S("abcdefghij"), 10, 1, "abcdefghij", 5, -5);
    test(S("abcdefghij"), 10, 1, "abcdefghij", 9, -9);
    test(S("abcdefghij"), 10, 1, "abcdefghij", 10, -10);
    test(S("abcdefghij"), 10, 1, "abcdefghijklmnopqrst", 0, 0);
    test(S("abcdefghij"), 10, 1, "abcdefghijklmnopqrst", 1, -1);
    test(S("abcdefghij"), 10, 1, "abcdefghijklmnopqrst", 10, -10);
    test(S("abcdefghij"), 10, 1, "abcdefghijklmnopqrst", 19, -19);
    test(S("abcdefghij"), 10, 1, "abcdefghijklmnopqrst", 20, -20);
    test(S("abcdefghij"), 11, 0, "", 0, 0);
    test(S("abcdefghij"), 11, 0, "abcde", 0, 0);
    test(S("abcdefghij"), 11, 0, "abcde", 1, 0);
    test(S("abcdefghij"), 11, 0, "abcde", 2, 0);
    test(S("abcdefghij"), 11, 0, "abcde", 4, 0);
    test(S("abcdefghij"), 11, 0, "abcde", 5, 0);
    test(S("abcdefghij"), 11, 0, "abcdefghij", 0, 0);
    test(S("abcdefghij"), 11, 0, "abcdefghij", 1, 0);
    test(S("abcdefghij"), 11, 0, "abcdefghij", 5, 0);
    test(S("abcdefghij"), 11, 0, "abcdefghij", 9, 0);
    test(S("abcdefghij"), 11, 0, "abcdefghij", 10, 0);
    test(S("abcdefghij"), 11, 0, "abcdefghijklmnopqrst", 0, 0);
    test(S("abcdefghij"), 11, 0, "abcdefghijklmnopqrst", 1, 0);
    test(S("abcdefghij"), 11, 0, "abcdefghijklmnopqrst", 10, 0);
    test(S("abcdefghij"), 11, 0, "abcdefghijklmnopqrst", 19, 0);
    test(S("abcdefghij"), 11, 0, "abcdefghijklmnopqrst", 20, 0);
}

template <class S>
void test8()
{
    test(S("abcdefghijklmnopqrst"), 0, 0, "", 0, 0);
    test(S("abcdefghijklmnopqrst"), 0, 0, "abcde", 0, 0);
    test(S("abcdefghijklmnopqrst"), 0, 0, "abcde", 1, -1);
    test(S("abcdefghijklmnopqrst"), 0, 0, "abcde", 2, -2);
    test(S("abcdefghijklmnopqrst"), 0, 0, "abcde", 4, -4);
    test(S("abcdefghijklmnopqrst"), 0, 0, "abcde", 5, -5);
    test(S("abcdefghijklmnopqrst"), 0, 0, "abcdefghij", 0, 0);
    test(S("abcdefghijklmnopqrst"), 0, 0, "abcdefghij", 1, -1);
    test(S("abcdefghijklmnopqrst"), 0, 0, "abcdefghij", 5, -5);
    test(S("abcdefghijklmnopqrst"), 0, 0, "abcdefghij", 9, -9);
    test(S("abcdefghijklmnopqrst"), 0, 0, "abcdefghij", 10, -10);
    test(S("abcdefghijklmnopqrst"), 0, 0, "abcdefghijklmnopqrst", 0, 0);
    test(S("abcdefghijklmnopqrst"), 0, 0, "abcdefghijklmnopqrst", 1, -1);
    test(S("abcdefghijklmnopqrst"), 0, 0, "abcdefghijklmnopqrst", 10, -10);
    test(S("abcdefghijklmnopqrst"), 0, 0, "abcdefghijklmnopqrst", 19, -19);
    test(S("abcdefghijklmnopqrst"), 0, 0, "abcdefghijklmnopqrst", 20, -20);
    test(S("abcdefghijklmnopqrst"), 0, 1, "", 0, 1);
    test(S("abcdefghijklmnopqrst"), 0, 1, "abcde", 0, 1);
    test(S("abcdefghijklmnopqrst"), 0, 1, "abcde", 1, 0);
    test(S("abcdefghijklmnopqrst"), 0, 1, "abcde", 2, -1);
    test(S("abcdefghijklmnopqrst"), 0, 1, "abcde", 4, -3);
    test(S("abcdefghijklmnopqrst"), 0, 1, "abcde", 5, -4);
    test(S("abcdefghijklmnopqrst"), 0, 1, "abcdefghij", 0, 1);
    test(S("abcdefghijklmnopqrst"), 0, 1, "abcdefghij", 1, 0);
    test(S("abcdefghijklmnopqrst"), 0, 1, "abcdefghij", 5, -4);
    test(S("abcdefghijklmnopqrst"), 0, 1, "abcdefghij", 9, -8);
    test(S("abcdefghijklmnopqrst"), 0, 1, "abcdefghij", 10, -9);
    test(S("abcdefghijklmnopqrst"), 0, 1, "abcdefghijklmnopqrst", 0, 1);
    test(S("abcdefghijklmnopqrst"), 0, 1, "abcdefghijklmnopqrst", 1, 0);
    test(S("abcdefghijklmnopqrst"), 0, 1, "abcdefghijklmnopqrst", 10, -9);
    test(S("abcdefghijklmnopqrst"), 0, 1, "abcdefghijklmnopqrst", 19, -18);
    test(S("abcdefghijklmnopqrst"), 0, 1, "abcdefghijklmnopqrst", 20, -19);
    test(S("abcdefghijklmnopqrst"), 0, 10, "", 0, 10);
    test(S("abcdefghijklmnopqrst"), 0, 10, "abcde", 0, 10);
    test(S("abcdefghijklmnopqrst"), 0, 10, "abcde", 1, 9);
    test(S("abcdefghijklmnopqrst"), 0, 10, "abcde", 2, 8);
    test(S("abcdefghijklmnopqrst"), 0, 10, "abcde", 4, 6);
    test(S("abcdefghijklmnopqrst"), 0, 10, "abcde", 5, 5);
    test(S("abcdefghijklmnopqrst"), 0, 10, "abcdefghij", 0, 10);
    test(S("abcdefghijklmnopqrst"), 0, 10, "abcdefghij", 1, 9);
    test(S("abcdefghijklmnopqrst"), 0, 10, "abcdefghij", 5, 5);
    test(S("abcdefghijklmnopqrst"), 0, 10, "abcdefghij", 9, 1);
    test(S("abcdefghijklmnopqrst"), 0, 10, "abcdefghij", 10, 0);
    test(S("abcdefghijklmnopqrst"), 0, 10, "abcdefghijklmnopqrst", 0, 10);
    test(S("abcdefghijklmnopqrst"), 0, 10, "abcdefghijklmnopqrst", 1, 9);
    test(S("abcdefghijklmnopqrst"), 0, 10, "abcdefghijklmnopqrst", 10, 0);
    test(S("abcdefghijklmnopqrst"), 0, 10, "abcdefghijklmnopqrst", 19, -9);
    test(S("abcdefghijklmnopqrst"), 0, 10, "abcdefghijklmnopqrst", 20, -10);
    test(S("abcdefghijklmnopqrst"), 0, 19, "", 0, 19);
    test(S("abcdefghijklmnopqrst"), 0, 19, "abcde", 0, 19);
    test(S("abcdefghijklmnopqrst"), 0, 19, "abcde", 1, 18);
    test(S("abcdefghijklmnopqrst"), 0, 19, "abcde", 2, 17);
    test(S("abcdefghijklmnopqrst"), 0, 19, "abcde", 4, 15);
    test(S("abcdefghijklmnopqrst"), 0, 19, "abcde", 5, 14);
    test(S("abcdefghijklmnopqrst"), 0, 19, "abcdefghij", 0, 19);
    test(S("abcdefghijklmnopqrst"), 0, 19, "abcdefghij", 1, 18);
    test(S("abcdefghijklmnopqrst"), 0, 19, "abcdefghij", 5, 14);
    test(S("abcdefghijklmnopqrst"), 0, 19, "abcdefghij", 9, 10);
    test(S("abcdefghijklmnopqrst"), 0, 19, "abcdefghij", 10, 9);
    test(S("abcdefghijklmnopqrst"), 0, 19, "abcdefghijklmnopqrst", 0, 19);
    test(S("abcdefghijklmnopqrst"), 0, 19, "abcdefghijklmnopqrst", 1, 18);
    test(S("abcdefghijklmnopqrst"), 0, 19, "abcdefghijklmnopqrst", 10, 9);
    test(S("abcdefghijklmnopqrst"), 0, 19, "abcdefghijklmnopqrst", 19, 0);
    test(S("abcdefghijklmnopqrst"), 0, 19, "abcdefghijklmnopqrst", 20, -1);
    test(S("abcdefghijklmnopqrst"), 0, 20, "", 0, 20);
    test(S("abcdefghijklmnopqrst"), 0, 20, "abcde", 0, 20);
    test(S("abcdefghijklmnopqrst"), 0, 20, "abcde", 1, 19);
    test(S("abcdefghijklmnopqrst"), 0, 20, "abcde", 2, 18);
    test(S("abcdefghijklmnopqrst"), 0, 20, "abcde", 4, 16);
    test(S("abcdefghijklmnopqrst"), 0, 20, "abcde", 5, 15);
    test(S("abcdefghijklmnopqrst"), 0, 20, "abcdefghij", 0, 20);
    test(S("abcdefghijklmnopqrst"), 0, 20, "abcdefghij", 1, 19);
    test(S("abcdefghijklmnopqrst"), 0, 20, "abcdefghij", 5, 15);
    test(S("abcdefghijklmnopqrst"), 0, 20, "abcdefghij", 9, 11);
    test(S("abcdefghijklmnopqrst"), 0, 20, "abcdefghij", 10, 10);
    test(S("abcdefghijklmnopqrst"), 0, 20, "abcdefghijklmnopqrst", 0, 20);
    test(S("abcdefghijklmnopqrst"), 0, 20, "abcdefghijklmnopqrst", 1, 19);
    test(S("abcdefghijklmnopqrst"), 0, 20, "abcdefghijklmnopqrst", 10, 10);
    test(S("abcdefghijklmnopqrst"), 0, 20, "abcdefghijklmnopqrst", 19, 1);
    test(S("abcdefghijklmnopqrst"), 0, 20, "abcdefghijklmnopqrst", 20, 0);
    test(S("abcdefghijklmnopqrst"), 0, 21, "", 0, 20);
    test(S("abcdefghijklmnopqrst"), 0, 21, "abcde", 0, 20);
    test(S("abcdefghijklmnopqrst"), 0, 21, "abcde", 1, 19);
    test(S("abcdefghijklmnopqrst"), 0, 21, "abcde", 2, 18);
    test(S("abcdefghijklmnopqrst"), 0, 21, "abcde", 4, 16);
    test(S("abcdefghijklmnopqrst"), 0, 21, "abcde", 5, 15);
    test(S("abcdefghijklmnopqrst"), 0, 21, "abcdefghij", 0, 20);
    test(S("abcdefghijklmnopqrst"), 0, 21, "abcdefghij", 1, 19);
    test(S("abcdefghijklmnopqrst"), 0, 21, "abcdefghij", 5, 15);
    test(S("abcdefghijklmnopqrst"), 0, 21, "abcdefghij", 9, 11);
    test(S("abcdefghijklmnopqrst"), 0, 21, "abcdefghij", 10, 10);
    test(S("abcdefghijklmnopqrst"), 0, 21, "abcdefghijklmnopqrst", 0, 20);
    test(S("abcdefghijklmnopqrst"), 0, 21, "abcdefghijklmnopqrst", 1, 19);
    test(S("abcdefghijklmnopqrst"), 0, 21, "abcdefghijklmnopqrst", 10, 10);
    test(S("abcdefghijklmnopqrst"), 0, 21, "abcdefghijklmnopqrst", 19, 1);
    test(S("abcdefghijklmnopqrst"), 0, 21, "abcdefghijklmnopqrst", 20, 0);
    test(S("abcdefghijklmnopqrst"), 1, 0, "", 0, 0);
    test(S("abcdefghijklmnopqrst"), 1, 0, "abcde", 0, 0);
    test(S("abcdefghijklmnopqrst"), 1, 0, "abcde", 1, -1);
    test(S("abcdefghijklmnopqrst"), 1, 0, "abcde", 2, -2);
}

template <class S>
void test9()
{
    test(S("abcdefghijklmnopqrst"), 1, 0, "abcde", 4, -4);
    test(S("abcdefghijklmnopqrst"), 1, 0, "abcde", 5, -5);
    test(S("abcdefghijklmnopqrst"), 1, 0, "abcdefghij", 0, 0);
    test(S("abcdefghijklmnopqrst"), 1, 0, "abcdefghij", 1, -1);
    test(S("abcdefghijklmnopqrst"), 1, 0, "abcdefghij", 5, -5);
    test(S("abcdefghijklmnopqrst"), 1, 0, "abcdefghij", 9, -9);
    test(S("abcdefghijklmnopqrst"), 1, 0, "abcdefghij", 10, -10);
    test(S("abcdefghijklmnopqrst"), 1, 0, "abcdefghijklmnopqrst", 0, 0);
    test(S("abcdefghijklmnopqrst"), 1, 0, "abcdefghijklmnopqrst", 1, -1);
    test(S("abcdefghijklmnopqrst"), 1, 0, "abcdefghijklmnopqrst", 10, -10);
    test(S("abcdefghijklmnopqrst"), 1, 0, "abcdefghijklmnopqrst", 19, -19);
    test(S("abcdefghijklmnopqrst"), 1, 0, "abcdefghijklmnopqrst", 20, -20);
    test(S("abcdefghijklmnopqrst"), 1, 1, "", 0, 1);
    test(S("abcdefghijklmnopqrst"), 1, 1, "abcde", 0, 1);
    test(S("abcdefghijklmnopqrst"), 1, 1, "abcde", 1, 1);
    test(S("abcdefghijklmnopqrst"), 1, 1, "abcde", 2, 1);
    test(S("abcdefghijklmnopqrst"), 1, 1, "abcde", 4, 1);
    test(S("abcdefghijklmnopqrst"), 1, 1, "abcde", 5, 1);
    test(S("abcdefghijklmnopqrst"), 1, 1, "abcdefghij", 0, 1);
    test(S("abcdefghijklmnopqrst"), 1, 1, "abcdefghij", 1, 1);
    test(S("abcdefghijklmnopqrst"), 1, 1, "abcdefghij", 5, 1);
    test(S("abcdefghijklmnopqrst"), 1, 1, "abcdefghij", 9, 1);
    test(S("abcdefghijklmnopqrst"), 1, 1, "abcdefghij", 10, 1);
    test(S("abcdefghijklmnopqrst"), 1, 1, "abcdefghijklmnopqrst", 0, 1);
    test(S("abcdefghijklmnopqrst"), 1, 1, "abcdefghijklmnopqrst", 1, 1);
    test(S("abcdefghijklmnopqrst"), 1, 1, "abcdefghijklmnopqrst", 10, 1);
    test(S("abcdefghijklmnopqrst"), 1, 1, "abcdefghijklmnopqrst", 19, 1);
    test(S("abcdefghijklmnopqrst"), 1, 1, "abcdefghijklmnopqrst", 20, 1);
    test(S("abcdefghijklmnopqrst"), 1, 9, "", 0, 9);
    test(S("abcdefghijklmnopqrst"), 1, 9, "abcde", 0, 9);
    test(S("abcdefghijklmnopqrst"), 1, 9, "abcde", 1, 1);
    test(S("abcdefghijklmnopqrst"), 1, 9, "abcde", 2, 1);
    test(S("abcdefghijklmnopqrst"), 1, 9, "abcde", 4, 1);
    test(S("abcdefghijklmnopqrst"), 1, 9, "abcde", 5, 1);
    test(S("abcdefghijklmnopqrst"), 1, 9, "abcdefghij", 0, 9);
    test(S("abcdefghijklmnopqrst"), 1, 9, "abcdefghij", 1, 1);
    test(S("abcdefghijklmnopqrst"), 1, 9, "abcdefghij", 5, 1);
    test(S("abcdefghijklmnopqrst"), 1, 9, "abcdefghij", 9, 1);
    test(S("abcdefghijklmnopqrst"), 1, 9, "abcdefghij", 10, 1);
    test(S("abcdefghijklmnopqrst"), 1, 9, "abcdefghijklmnopqrst", 0, 9);
    test(S("abcdefghijklmnopqrst"), 1, 9, "abcdefghijklmnopqrst", 1, 1);
    test(S("abcdefghijklmnopqrst"), 1, 9, "abcdefghijklmnopqrst", 10, 1);
    test(S("abcdefghijklmnopqrst"), 1, 9, "abcdefghijklmnopqrst", 19, 1);
    test(S("abcdefghijklmnopqrst"), 1, 9, "abcdefghijklmnopqrst", 20, 1);
    test(S("abcdefghijklmnopqrst"), 1, 18, "", 0, 18);
    test(S("abcdefghijklmnopqrst"), 1, 18, "abcde", 0, 18);
    test(S("abcdefghijklmnopqrst"), 1, 18, "abcde", 1, 1);
    test(S("abcdefghijklmnopqrst"), 1, 18, "abcde", 2, 1);
    test(S("abcdefghijklmnopqrst"), 1, 18, "abcde", 4, 1);
    test(S("abcdefghijklmnopqrst"), 1, 18, "abcde", 5, 1);
    test(S("abcdefghijklmnopqrst"), 1, 18, "abcdefghij", 0, 18);
    test(S("abcdefghijklmnopqrst"), 1, 18, "abcdefghij", 1, 1);
    test(S("abcdefghijklmnopqrst"), 1, 18, "abcdefghij", 5, 1);
    test(S("abcdefghijklmnopqrst"), 1, 18, "abcdefghij", 9, 1);
    test(S("abcdefghijklmnopqrst"), 1, 18, "abcdefghij", 10, 1);
    test(S("abcdefghijklmnopqrst"), 1, 18, "abcdefghijklmnopqrst", 0, 18);
    test(S("abcdefghijklmnopqrst"), 1, 18, "abcdefghijklmnopqrst", 1, 1);
    test(S("abcdefghijklmnopqrst"), 1, 18, "abcdefghijklmnopqrst", 10, 1);
    test(S("abcdefghijklmnopqrst"), 1, 18, "abcdefghijklmnopqrst", 19, 1);
    test(S("abcdefghijklmnopqrst"), 1, 18, "abcdefghijklmnopqrst", 20, 1);
    test(S("abcdefghijklmnopqrst"), 1, 19, "", 0, 19);
    test(S("abcdefghijklmnopqrst"), 1, 19, "abcde", 0, 19);
    test(S("abcdefghijklmnopqrst"), 1, 19, "abcde", 1, 1);
    test(S("abcdefghijklmnopqrst"), 1, 19, "abcde", 2, 1);
    test(S("abcdefghijklmnopqrst"), 1, 19, "abcde", 4, 1);
    test(S("abcdefghijklmnopqrst"), 1, 19, "abcde", 5, 1);
    test(S("abcdefghijklmnopqrst"), 1, 19, "abcdefghij", 0, 19);
    test(S("abcdefghijklmnopqrst"), 1, 19, "abcdefghij", 1, 1);
    test(S("abcdefghijklmnopqrst"), 1, 19, "abcdefghij", 5, 1);
    test(S("abcdefghijklmnopqrst"), 1, 19, "abcdefghij", 9, 1);
    test(S("abcdefghijklmnopqrst"), 1, 19, "abcdefghij", 10, 1);
    test(S("abcdefghijklmnopqrst"), 1, 19, "abcdefghijklmnopqrst", 0, 19);
    test(S("abcdefghijklmnopqrst"), 1, 19, "abcdefghijklmnopqrst", 1, 1);
    test(S("abcdefghijklmnopqrst"), 1, 19, "abcdefghijklmnopqrst", 10, 1);
    test(S("abcdefghijklmnopqrst"), 1, 19, "abcdefghijklmnopqrst", 19, 1);
    test(S("abcdefghijklmnopqrst"), 1, 19, "abcdefghijklmnopqrst", 20, 1);
    test(S("abcdefghijklmnopqrst"), 1, 20, "", 0, 19);
    test(S("abcdefghijklmnopqrst"), 1, 20, "abcde", 0, 19);
    test(S("abcdefghijklmnopqrst"), 1, 20, "abcde", 1, 1);
    test(S("abcdefghijklmnopqrst"), 1, 20, "abcde", 2, 1);
    test(S("abcdefghijklmnopqrst"), 1, 20, "abcde", 4, 1);
    test(S("abcdefghijklmnopqrst"), 1, 20, "abcde", 5, 1);
    test(S("abcdefghijklmnopqrst"), 1, 20, "abcdefghij", 0, 19);
    test(S("abcdefghijklmnopqrst"), 1, 20, "abcdefghij", 1, 1);
    test(S("abcdefghijklmnopqrst"), 1, 20, "abcdefghij", 5, 1);
    test(S("abcdefghijklmnopqrst"), 1, 20, "abcdefghij", 9, 1);
    test(S("abcdefghijklmnopqrst"), 1, 20, "abcdefghij", 10, 1);
    test(S("abcdefghijklmnopqrst"), 1, 20, "abcdefghijklmnopqrst", 0, 19);
    test(S("abcdefghijklmnopqrst"), 1, 20, "abcdefghijklmnopqrst", 1, 1);
    test(S("abcdefghijklmnopqrst"), 1, 20, "abcdefghijklmnopqrst", 10, 1);
    test(S("abcdefghijklmnopqrst"), 1, 20, "abcdefghijklmnopqrst", 19, 1);
    test(S("abcdefghijklmnopqrst"), 1, 20, "abcdefghijklmnopqrst", 20, 1);
    test(S("abcdefghijklmnopqrst"), 10, 0, "", 0, 0);
    test(S("abcdefghijklmnopqrst"), 10, 0, "abcde", 0, 0);
    test(S("abcdefghijklmnopqrst"), 10, 0, "abcde", 1, -1);
    test(S("abcdefghijklmnopqrst"), 10, 0, "abcde", 2, -2);
    test(S("abcdefghijklmnopqrst"), 10, 0, "abcde", 4, -4);
    test(S("abcdefghijklmnopqrst"), 10, 0, "abcde", 5, -5);
    test(S("abcdefghijklmnopqrst"), 10, 0, "abcdefghij", 0, 0);
    test(S("abcdefghijklmnopqrst"), 10, 0, "abcdefghij", 1, -1);
}

template <class S>
void test10()
{
    test(S("abcdefghijklmnopqrst"), 10, 0, "abcdefghij", 5, -5);
    test(S("abcdefghijklmnopqrst"), 10, 0, "abcdefghij", 9, -9);
    test(S("abcdefghijklmnopqrst"), 10, 0, "abcdefghij", 10, -10);
    test(S("abcdefghijklmnopqrst"), 10, 0, "abcdefghijklmnopqrst", 0, 0);
    test(S("abcdefghijklmnopqrst"), 10, 0, "abcdefghijklmnopqrst", 1, -1);
    test(S("abcdefghijklmnopqrst"), 10, 0, "abcdefghijklmnopqrst", 10, -10);
    test(S("abcdefghijklmnopqrst"), 10, 0, "abcdefghijklmnopqrst", 19, -19);
    test(S("abcdefghijklmnopqrst"), 10, 0, "abcdefghijklmnopqrst", 20, -20);
    test(S("abcdefghijklmnopqrst"), 10, 1, "", 0, 1);
    test(S("abcdefghijklmnopqrst"), 10, 1, "abcde", 0, 1);
    test(S("abcdefghijklmnopqrst"), 10, 1, "abcde", 1, 10);
    test(S("abcdefghijklmnopqrst"), 10, 1, "abcde", 2, 10);
    test(S("abcdefghijklmnopqrst"), 10, 1, "abcde", 4, 10);
    test(S("abcdefghijklmnopqrst"), 10, 1, "abcde", 5, 10);
    test(S("abcdefghijklmnopqrst"), 10, 1, "abcdefghij", 0, 1);
    test(S("abcdefghijklmnopqrst"), 10, 1, "abcdefghij", 1, 10);
    test(S("abcdefghijklmnopqrst"), 10, 1, "abcdefghij", 5, 10);
    test(S("abcdefghijklmnopqrst"), 10, 1, "abcdefghij", 9, 10);
    test(S("abcdefghijklmnopqrst"), 10, 1, "abcdefghij", 10, 10);
    test(S("abcdefghijklmnopqrst"), 10, 1, "abcdefghijklmnopqrst", 0, 1);
    test(S("abcdefghijklmnopqrst"), 10, 1, "abcdefghijklmnopqrst", 1, 10);
    test(S("abcdefghijklmnopqrst"), 10, 1, "abcdefghijklmnopqrst", 10, 10);
    test(S("abcdefghijklmnopqrst"), 10, 1, "abcdefghijklmnopqrst", 19, 10);
    test(S("abcdefghijklmnopqrst"), 10, 1, "abcdefghijklmnopqrst", 20, 10);
    test(S("abcdefghijklmnopqrst"), 10, 5, "", 0, 5);
    test(S("abcdefghijklmnopqrst"), 10, 5, "abcde", 0, 5);
    test(S("abcdefghijklmnopqrst"), 10, 5, "abcde", 1, 10);
    test(S("abcdefghijklmnopqrst"), 10, 5, "abcde", 2, 10);
    test(S("abcdefghijklmnopqrst"), 10, 5, "abcde", 4, 10);
    test(S("abcdefghijklmnopqrst"), 10, 5, "abcde", 5, 10);
    test(S("abcdefghijklmnopqrst"), 10, 5, "abcdefghij", 0, 5);
    test(S("abcdefghijklmnopqrst"), 10, 5, "abcdefghij", 1, 10);
    test(S("abcdefghijklmnopqrst"), 10, 5, "abcdefghij", 5, 10);
    test(S("abcdefghijklmnopqrst"), 10, 5, "abcdefghij", 9, 10);
    test(S("abcdefghijklmnopqrst"), 10, 5, "abcdefghij", 10, 10);
    test(S("abcdefghijklmnopqrst"), 10, 5, "abcdefghijklmnopqrst", 0, 5);
    test(S("abcdefghijklmnopqrst"), 10, 5, "abcdefghijklmnopqrst", 1, 10);
    test(S("abcdefghijklmnopqrst"), 10, 5, "abcdefghijklmnopqrst", 10, 10);
    test(S("abcdefghijklmnopqrst"), 10, 5, "abcdefghijklmnopqrst", 19, 10);
    test(S("abcdefghijklmnopqrst"), 10, 5, "abcdefghijklmnopqrst", 20, 10);
    test(S("abcdefghijklmnopqrst"), 10, 9, "", 0, 9);
    test(S("abcdefghijklmnopqrst"), 10, 9, "abcde", 0, 9);
    test(S("abcdefghijklmnopqrst"), 10, 9, "abcde", 1, 10);
    test(S("abcdefghijklmnopqrst"), 10, 9, "abcde", 2, 10);
    test(S("abcdefghijklmnopqrst"), 10, 9, "abcde", 4, 10);
    test(S("abcdefghijklmnopqrst"), 10, 9, "abcde", 5, 10);
    test(S("abcdefghijklmnopqrst"), 10, 9, "abcdefghij", 0, 9);
    test(S("abcdefghijklmnopqrst"), 10, 9, "abcdefghij", 1, 10);
    test(S("abcdefghijklmnopqrst"), 10, 9, "abcdefghij", 5, 10);
    test(S("abcdefghijklmnopqrst"), 10, 9, "abcdefghij", 9, 10);
    test(S("abcdefghijklmnopqrst"), 10, 9, "abcdefghij", 10, 10);
    test(S("abcdefghijklmnopqrst"), 10, 9, "abcdefghijklmnopqrst", 0, 9);
    test(S("abcdefghijklmnopqrst"), 10, 9, "abcdefghijklmnopqrst", 1, 10);
    test(S("abcdefghijklmnopqrst"), 10, 9, "abcdefghijklmnopqrst", 10, 10);
    test(S("abcdefghijklmnopqrst"), 10, 9, "abcdefghijklmnopqrst", 19, 10);
    test(S("abcdefghijklmnopqrst"), 10, 9, "abcdefghijklmnopqrst", 20, 10);
    test(S("abcdefghijklmnopqrst"), 10, 10, "", 0, 10);
    test(S("abcdefghijklmnopqrst"), 10, 10, "abcde", 0, 10);
    test(S("abcdefghijklmnopqrst"), 10, 10, "abcde", 1, 10);
    test(S("abcdefghijklmnopqrst"), 10, 10, "abcde", 2, 10);
    test(S("abcdefghijklmnopqrst"), 10, 10, "abcde", 4, 10);
    test(S("abcdefghijklmnopqrst"), 10, 10, "abcde", 5, 10);
    test(S("abcdefghijklmnopqrst"), 10, 10, "abcdefghij", 0, 10);
    test(S("abcdefghijklmnopqrst"), 10, 10, "abcdefghij", 1, 10);
    test(S("abcdefghijklmnopqrst"), 10, 10, "abcdefghij", 5, 10);
    test(S("abcdefghijklmnopqrst"), 10, 10, "abcdefghij", 9, 10);
    test(S("abcdefghijklmnopqrst"), 10, 10, "abcdefghij", 10, 10);
    test(S("abcdefghijklmnopqrst"), 10, 10, "abcdefghijklmnopqrst", 0, 10);
    test(S("abcdefghijklmnopqrst"), 10, 10, "abcdefghijklmnopqrst", 1, 10);
    test(S("abcdefghijklmnopqrst"), 10, 10, "abcdefghijklmnopqrst", 10, 10);
    test(S("abcdefghijklmnopqrst"), 10, 10, "abcdefghijklmnopqrst", 19, 10);
    test(S("abcdefghijklmnopqrst"), 10, 10, "abcdefghijklmnopqrst", 20, 10);
    test(S("abcdefghijklmnopqrst"), 10, 11, "", 0, 10);
    test(S("abcdefghijklmnopqrst"), 10, 11, "abcde", 0, 10);
    test(S("abcdefghijklmnopqrst"), 10, 11, "abcde", 1, 10);
    test(S("abcdefghijklmnopqrst"), 10, 11, "abcde", 2, 10);
    test(S("abcdefghijklmnopqrst"), 10, 11, "abcde", 4, 10);
    test(S("abcdefghijklmnopqrst"), 10, 11, "abcde", 5, 10);
    test(S("abcdefghijklmnopqrst"), 10, 11, "abcdefghij", 0, 10);
    test(S("abcdefghijklmnopqrst"), 10, 11, "abcdefghij", 1, 10);
    test(S("abcdefghijklmnopqrst"), 10, 11, "abcdefghij", 5, 10);
    test(S("abcdefghijklmnopqrst"), 10, 11, "abcdefghij", 9, 10);
    test(S("abcdefghijklmnopqrst"), 10, 11, "abcdefghij", 10, 10);
    test(S("abcdefghijklmnopqrst"), 10, 11, "abcdefghijklmnopqrst", 0, 10);
    test(S("abcdefghijklmnopqrst"), 10, 11, "abcdefghijklmnopqrst", 1, 10);
    test(S("abcdefghijklmnopqrst"), 10, 11, "abcdefghijklmnopqrst", 10, 10);
    test(S("abcdefghijklmnopqrst"), 10, 11, "abcdefghijklmnopqrst", 19, 10);
    test(S("abcdefghijklmnopqrst"), 10, 11, "abcdefghijklmnopqrst", 20, 10);
    test(S("abcdefghijklmnopqrst"), 19, 0, "", 0, 0);
    test(S("abcdefghijklmnopqrst"), 19, 0, "abcde", 0, 0);
    test(S("abcdefghijklmnopqrst"), 19, 0, "abcde", 1, -1);
    test(S("abcdefghijklmnopqrst"), 19, 0, "abcde", 2, -2);
    test(S("abcdefghijklmnopqrst"), 19, 0, "abcde", 4, -4);
    test(S("abcdefghijklmnopqrst"), 19, 0, "abcde", 5, -5);
    test(S("abcdefghijklmnopqrst"), 19, 0, "abcdefghij", 0, 0);
    test(S("abcdefghijklmnopqrst"), 19, 0, "abcdefghij", 1, -1);
    test(S("abcdefghijklmnopqrst"), 19, 0, "abcdefghij", 5, -5);
    test(S("abcdefghijklmnopqrst"), 19, 0, "abcdefghij", 9, -9);
    test(S("abcdefghijklmnopqrst"), 19, 0, "abcdefghij", 10, -10);
    test(S("abcdefghijklmnopqrst"), 19, 0, "abcdefghijklmnopqrst", 0, 0);
}

template <class S>
void test11()
{
    test(S("abcdefghijklmnopqrst"), 19, 0, "abcdefghijklmnopqrst", 1, -1);
    test(S("abcdefghijklmnopqrst"), 19, 0, "abcdefghijklmnopqrst", 10, -10);
    test(S("abcdefghijklmnopqrst"), 19, 0, "abcdefghijklmnopqrst", 19, -19);
    test(S("abcdefghijklmnopqrst"), 19, 0, "abcdefghijklmnopqrst", 20, -20);
    test(S("abcdefghijklmnopqrst"), 19, 1, "", 0, 1);
    test(S("abcdefghijklmnopqrst"), 19, 1, "abcde", 0, 1);
    test(S("abcdefghijklmnopqrst"), 19, 1, "abcde", 1, 19);
    test(S("abcdefghijklmnopqrst"), 19, 1, "abcde", 2, 19);
    test(S("abcdefghijklmnopqrst"), 19, 1, "abcde", 4, 19);
    test(S("abcdefghijklmnopqrst"), 19, 1, "abcde", 5, 19);
    test(S("abcdefghijklmnopqrst"), 19, 1, "abcdefghij", 0, 1);
    test(S("abcdefghijklmnopqrst"), 19, 1, "abcdefghij", 1, 19);
    test(S("abcdefghijklmnopqrst"), 19, 1, "abcdefghij", 5, 19);
    test(S("abcdefghijklmnopqrst"), 19, 1, "abcdefghij", 9, 19);
    test(S("abcdefghijklmnopqrst"), 19, 1, "abcdefghij", 10, 19);
    test(S("abcdefghijklmnopqrst"), 19, 1, "abcdefghijklmnopqrst", 0, 1);
    test(S("abcdefghijklmnopqrst"), 19, 1, "abcdefghijklmnopqrst", 1, 19);
    test(S("abcdefghijklmnopqrst"), 19, 1, "abcdefghijklmnopqrst", 10, 19);
    test(S("abcdefghijklmnopqrst"), 19, 1, "abcdefghijklmnopqrst", 19, 19);
    test(S("abcdefghijklmnopqrst"), 19, 1, "abcdefghijklmnopqrst", 20, 19);
    test(S("abcdefghijklmnopqrst"), 19, 2, "", 0, 1);
    test(S("abcdefghijklmnopqrst"), 19, 2, "abcde", 0, 1);
    test(S("abcdefghijklmnopqrst"), 19, 2, "abcde", 1, 19);
    test(S("abcdefghijklmnopqrst"), 19, 2, "abcde", 2, 19);
    test(S("abcdefghijklmnopqrst"), 19, 2, "abcde", 4, 19);
    test(S("abcdefghijklmnopqrst"), 19, 2, "abcde", 5, 19);
    test(S("abcdefghijklmnopqrst"), 19, 2, "abcdefghij", 0, 1);
    test(S("abcdefghijklmnopqrst"), 19, 2, "abcdefghij", 1, 19);
    test(S("abcdefghijklmnopqrst"), 19, 2, "abcdefghij", 5, 19);
    test(S("abcdefghijklmnopqrst"), 19, 2, "abcdefghij", 9, 19);
    test(S("abcdefghijklmnopqrst"), 19, 2, "abcdefghij", 10, 19);
    test(S("abcdefghijklmnopqrst"), 19, 2, "abcdefghijklmnopqrst", 0, 1);
    test(S("abcdefghijklmnopqrst"), 19, 2, "abcdefghijklmnopqrst", 1, 19);
    test(S("abcdefghijklmnopqrst"), 19, 2, "abcdefghijklmnopqrst", 10, 19);
    test(S("abcdefghijklmnopqrst"), 19, 2, "abcdefghijklmnopqrst", 19, 19);
    test(S("abcdefghijklmnopqrst"), 19, 2, "abcdefghijklmnopqrst", 20, 19);
    test(S("abcdefghijklmnopqrst"), 20, 0, "", 0, 0);
    test(S("abcdefghijklmnopqrst"), 20, 0, "abcde", 0, 0);
    test(S("abcdefghijklmnopqrst"), 20, 0, "abcde", 1, -1);
    test(S("abcdefghijklmnopqrst"), 20, 0, "abcde", 2, -2);
    test(S("abcdefghijklmnopqrst"), 20, 0, "abcde", 4, -4);
    test(S("abcdefghijklmnopqrst"), 20, 0, "abcde", 5, -5);
    test(S("abcdefghijklmnopqrst"), 20, 0, "abcdefghij", 0, 0);
    test(S("abcdefghijklmnopqrst"), 20, 0, "abcdefghij", 1, -1);
    test(S("abcdefghijklmnopqrst"), 20, 0, "abcdefghij", 5, -5);
    test(S("abcdefghijklmnopqrst"), 20, 0, "abcdefghij", 9, -9);
    test(S("abcdefghijklmnopqrst"), 20, 0, "abcdefghij", 10, -10);
    test(S("abcdefghijklmnopqrst"), 20, 0, "abcdefghijklmnopqrst", 0, 0);
    test(S("abcdefghijklmnopqrst"), 20, 0, "abcdefghijklmnopqrst", 1, -1);
    test(S("abcdefghijklmnopqrst"), 20, 0, "abcdefghijklmnopqrst", 10, -10);
    test(S("abcdefghijklmnopqrst"), 20, 0, "abcdefghijklmnopqrst", 19, -19);
    test(S("abcdefghijklmnopqrst"), 20, 0, "abcdefghijklmnopqrst", 20, -20);
    test(S("abcdefghijklmnopqrst"), 20, 1, "", 0, 0);
    test(S("abcdefghijklmnopqrst"), 20, 1, "abcde", 0, 0);
    test(S("abcdefghijklmnopqrst"), 20, 1, "abcde", 1, -1);
    test(S("abcdefghijklmnopqrst"), 20, 1, "abcde", 2, -2);
    test(S("abcdefghijklmnopqrst"), 20, 1, "abcde", 4, -4);
    test(S("abcdefghijklmnopqrst"), 20, 1, "abcde", 5, -5);
    test(S("abcdefghijklmnopqrst"), 20, 1, "abcdefghij", 0, 0);
    test(S("abcdefghijklmnopqrst"), 20, 1, "abcdefghij", 1, -1);
    test(S("abcdefghijklmnopqrst"), 20, 1, "abcdefghij", 5, -5);
    test(S("abcdefghijklmnopqrst"), 20, 1, "abcdefghij", 9, -9);
    test(S("abcdefghijklmnopqrst"), 20, 1, "abcdefghij", 10, -10);
    test(S("abcdefghijklmnopqrst"), 20, 1, "abcdefghijklmnopqrst", 0, 0);
    test(S("abcdefghijklmnopqrst"), 20, 1, "abcdefghijklmnopqrst", 1, -1);
    test(S("abcdefghijklmnopqrst"), 20, 1, "abcdefghijklmnopqrst", 10, -10);
    test(S("abcdefghijklmnopqrst"), 20, 1, "abcdefghijklmnopqrst", 19, -19);
    test(S("abcdefghijklmnopqrst"), 20, 1, "abcdefghijklmnopqrst", 20, -20);
    test(S("abcdefghijklmnopqrst"), 21, 0, "", 0, 0);
    test(S("abcdefghijklmnopqrst"), 21, 0, "abcde", 0, 0);
    test(S("abcdefghijklmnopqrst"), 21, 0, "abcde", 1, 0);
    test(S("abcdefghijklmnopqrst"), 21, 0, "abcde", 2, 0);
    test(S("abcdefghijklmnopqrst"), 21, 0, "abcde", 4, 0);
    test(S("abcdefghijklmnopqrst"), 21, 0, "abcde", 5, 0);
    test(S("abcdefghijklmnopqrst"), 21, 0, "abcdefghij", 0, 0);
    test(S("abcdefghijklmnopqrst"), 21, 0, "abcdefghij", 1, 0);
    test(S("abcdefghijklmnopqrst"), 21, 0, "abcdefghij", 5, 0);
    test(S("abcdefghijklmnopqrst"), 21, 0, "abcdefghij", 9, 0);
    test(S("abcdefghijklmnopqrst"), 21, 0, "abcdefghij", 10, 0);
    test(S("abcdefghijklmnopqrst"), 21, 0, "abcdefghijklmnopqrst", 0, 0);
    test(S("abcdefghijklmnopqrst"), 21, 0, "abcdefghijklmnopqrst", 1, 0);
    test(S("abcdefghijklmnopqrst"), 21, 0, "abcdefghijklmnopqrst", 10, 0);
    test(S("abcdefghijklmnopqrst"), 21, 0, "abcdefghijklmnopqrst", 19, 0);
    test(S("abcdefghijklmnopqrst"), 21, 0, "abcdefghijklmnopqrst", 20, 0);
}

int main(int, char**)
{
    {
    typedef std::string S;
    test0<S>();
    test1<S>();
    test2<S>();
    test3<S>();
    test4<S>();
    test5<S>();
    test6<S>();
    test7<S>();
    test8<S>();
    test9<S>();
    test10<S>();
    test11<S>();
    }
#if TEST_STD_VER >= 11
    {
    typedef std::basic_string<char, std::char_traits<char>, min_allocator<char>> S;
    test0<S>();
    test1<S>();
    test2<S>();
    test3<S>();
    test4<S>();
    test5<S>();
    test6<S>();
    test7<S>();
    test8<S>();
    test9<S>();
    test10<S>();
    test11<S>();
    }
#endif

  return 0;
}
