\name{laslett}
\alias{laslett}
\title{
  Laslett's Transform
}
\description{
  Apply Laslett's Transform to a spatial region,
  returning the original and transformed regions,
  and the original and transformed positions of the lower tangent points.
  This is a diagnostic for the Boolean model.
}
\usage{
laslett(X, \dots, verbose = FALSE, plotit = TRUE, discretise = FALSE,
        type=c("lower", "upper", "left", "right"))
}
\arguments{
  \item{X}{
    Spatial region to be transformed.
    A window (object of class \code{"owin"}) or a logical-valued pixel
    image (object of class \code{"im"}).
  }
  \item{\dots}{
    Graphics arguments to control the plot (passed to
    \code{\link{plot.laslett}} when \code{plotit=TRUE})
    or arguments determining the pixel resolution
    (passed to \code{\link[spatstat.geom]{as.mask}}).
  }
  \item{verbose}{
    Logical value indicating whether to print progress reports.
  }
  \item{plotit}{
    Logical value indicating whether to plot the result.
  }
  \item{discretise}{
    Logical value indicating whether polygonal windows should first be
    converted to pixel masks before the Laslett transform is
    computed. This should be set to \code{TRUE} for very complicated
    polygons. 
  }
  \item{type}{
    Type of tangent points to be detected.
    This also determines the direction of contraction in the
    set transformation.
    Default is \code{type="lower"}.
  }
}
\details{
  This function finds the lower tangent points of the spatial region \code{X},
  then applies Laslett's Transform to the space,
  and records the transformed positions of the lower tangent points.

  Laslett's transform is a diagnostic for the Boolean Model.
  A test of the Boolean model can be performed by applying a test of CSR
  to the transformed tangent points. See the Examples.

  The rationale is that, if the region \code{X} was generated by a
  Boolean model with convex grains, then the lower tangent points of
  \code{X}, when subjected to Laslett's transform,
  become a Poisson point process (Cressie, 1993, section 9.3.5;
  Molchanov, 1997; Barbour and Schmidt, 2001).

  Intuitively, Laslett's transform is a way to account for the fact that
  tangent points of \code{X} cannot occur \emph{inside} \code{X}.
  It treats the interior of \code{X} as empty space, and collapses
  this empty space so that only the \emph{exterior} of \code{X} remains.
  In this collapsed space, the tangent points are completely random.

  Formally, Laslett's transform is a random (i.e. data-dependent)
  spatial transformation which maps each spatial
  location \eqn{(x,y)} to a new location \eqn{(x',y)} at the same height
  \eqn{y}. The transformation is defined so that \eqn{x'}
  is the total \emph{uncovered} length of the line segment from \eqn{(0,y)} to
  \eqn{(x,y)}, that is, the total length of the parts of this segment that
  fall outside the region \code{X}.

  In more colourful terms, suppose we use an abacus to display a
  pixellated version of \code{X}. Each wire of the abacus represents one
  horizontal line in the pixel image. Each pixel lying \emph{outside}
  the region \code{X} is represented by a bead of the abacus; pixels
  \emph{inside} \code{X} are represented by the absence of a bead. Next
  we find any beads which are lower tangent points of \code{X}, and
  paint them green. Then Laslett's Transform is applied by pushing all
  beads to the left, as far as possible. The final locations of all the
  beads provide a new spatial region, inside which is the point pattern
  of tangent points (marked by the green-painted beads). 

  If \code{plotit=TRUE} (the default), a before-and-after plot is
  generated, showing the region \code{X} and the tangent points
  before and after the transformation. This plot can also be generated
  by calling \code{plot(a)} where \code{a} is the object returned by
  the function \code{laslett}.

  If the argument \code{type} is given, then this determines the
  type of tangents that will be detected, and also the direction of
  contraction in Laslett's transform. The computation is performed
  by first rotating \code{X}, applying Laslett's transform for lower
  tangent points, then rotating back.

  There are separate algorithms for polygonal windows and
  pixellated windows (binary masks). The polygonal algorithm may be slow
  for very complicated polygons. If this happens, setting
  \code{discretise=TRUE} will convert the polygonal window to a binary
  mask and invoke the pixel raster algorithm.
}
\value{
  A list, which also belongs to the class \code{"laslett"}
  so that it can immediately be printed and plotted.
  
  The list elements are:
  \describe{
    \item{oldX:}{the original dataset \code{X};}
    \item{TanOld:}{a point pattern, whose window is \code{Frame(X)},
      containing the lower tangent points of \code{X};}
    \item{TanNew:}{a point pattern, whose window is the Laslett transform
      of \code{Frame(X)}, and which contains the Laslett-transformed
      positions of the tangent points;}
    \item{Rect:}{a rectangular window, which is the largest rectangle
      lying inside the transformed set;}
    \item{df:}{a data frame giving the locations of the tangent points
      before and after transformation. }
    \item{type:}{character string specifying the type of tangents.}
  }
}
\references{
  Barbour, A.D. and Schmidt, V. (2001)
  On Laslett's Transform for the Boolean Model.
  \emph{Advances in Applied Probability} 
  \bold{33}(1), 1--5.

  Cressie, N.A.C. (1993)
  \emph{Statistics for spatial data}, second edition.
  John Wiley and Sons.

  Molchanov, I. (1997)
  \emph{Statistics of the Boolean Model for Practitioners and Mathematicians}.
  Wiley.
}
\author{
  Kassel Hingee and \adrian.
}
\seealso{
  \code{\link{plot.laslett}}
}
\examples{
a <- laslett(heather$coarse)
transformedHeather <- with(a, Window(TanNew))
plot(transformedHeather, invert=TRUE)

with(a, clarkevans.test(TanNew[Rect], correction="D", nsim=39))

X <- discs(runifrect(15) \%mark\% 0.2, npoly=16)
b <- laslett(X, type="left")
b
}
\keyword{spatial}
\keyword{manip}

