#!/bin/awk -f

###### Convert yaboot.conf into kboot.conf

### Function definitions

# Parse one global option
function parse_global_opt(line)
{
  if (line ~ /^delay=/) {
    print line
  }
  else if (sub(/^timeout=/, "", line)) {
    printf("timeout=%d\n", line / 10)
  }
  else if (sub(/^default=/, "", line)) {
    default_label = unquote_string(line)
  }
  else if (sub(/^root=/, "", line)) {
    default_root = unquote_string(line)
    print "root=" default_root
  }
  else if (sub(/^initrd=/, "", line)) {
    default_initrd = unquote_string(line)
  }
  else if (sub(/^append=/, "", line)) {
    default_append = unquote_string(line)
  }
}

# Parse one boot entry ("image=")
function parse_entry(num_params, params,
		     i, result, image, label, alias, root, initrd, append) # local
{
  if (!num_params) {
    return
  }

  label = ""
  alias = ""
  root = ""
  initrd = default_initrd
  append = default_append

  for (i = 1; i < num_params; i++) {
    if (sub(/^alias=/, "", params[i])) {
      alias = unquote_string(params[i])
    }
    else if (sub(/^label=/, "", params[i])) {
      label = unquote_string(params[i])
    }
    else if (sub(/^root=/, "", params[i])) {
      root = unquote_string(params[i])
    }
    else if (sub(/^initrd=/, "", params[i])) {
      initrd = unquote_string(params[i])
    }
    else if (sub(/^append=/, "", params[i])) {
      append = unquote_string(params[i])
    }
  }

  sub(/^image=/, "", params[0])
  image = unquote_string(params[0])

  if (!label) {
    i = match(image, /[^\/]+$/)
    label = substr(image, i)
  }

  result = image

  if (root) {
    result = result " root=" root
  }
  if (initrd) {
    result = result " initrd=" initrd
  }
  if (append) {
    result = result " " append
  }

  entries[label] = result

  if (alias) {
    aliases[alias] = label
  }

  if (!default_label) {
    default_label = label
  }
}

# Normalize boot entry names
function normalize_label(label,
			 result) # local
{
  result = label
  gsub(/[^[:alnum:]]/, "_", result)
  sub(/^_/, "image_", result)
  sub(/^[^[:alpha:]]/, "image_&", result)
  return result
}

# Unquote
function unquote_string(str,
			result) # local
{
  result = str
  if (sub(/^"/, "", result)) {
    sub(/"$/, "", result)
  }
  return result
}

### Main

# Setup
BEGIN {
  image_argc = 0
}

# Comment
/^#/ { next }

# Remove spaces
{
  sub(/^[[:space:]]*/, "")
  sub(/[[:space:]]*$/, "")
  sub(/[[:space:]]*=[[:space:]]*/, "=")
}

# Empty line
/^$/ { next }

# New image
/^image=/ {
  parse_entry(image_argc, image_argv)
  image_argc = 0
  image_argv[image_argc++] = $0
  next
}

# Parameter w/ argument
/^[^=][^=]*=/ {
  if (image_argc) {
    image_argv[image_argc++] = $0
  }
  else {
    parse_global_opt($0)
  }
  next
}

# Parameter w/o argument
{
  if (image_argc) {
    image_argv[image_argc++] = $0
  }
  else {
    parse_global_opt($0)
  }
  next
}

# Output
END {
  parse_entry(image_argc, image_argv)

  # default label
  if (default_label) {
    print "default=" normalize_label(default_label)
  }

  # each entry
  for (i in entries) {
    print normalize_label(i) "='" entries[i] "'"
  }

  # aliases
  for (i in aliases) {
    print normalize_label(i) "='" entries[aliases[i]] "'"
  }
}
