/* cdw
 * Copyright (C) 2002 Varkonyi Balazs
 * Copyright (C) 2007 - 2016 Kamil Ignacak
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */

#define _GNU_SOURCE /* strndup() */

#include <string.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <stdbool.h>
#include <dirent.h> /* PATH_MAX */
#include <errno.h>

#include "cdw_string.h"
#include "cdw_debug.h"
#include "cdw_file.h"
#include "gettext.h"
#include "cdw_fs.h"
#include "cdw_list_display.h"



/**
   \file cdw_file.c

   Data structure for storing information about files existing in
   native file system.

   This module provides two basic functions:
   \li cdw_file_new() - constructor
   \li cdw_file_delete() - destructor

   three utility functions:
   \li cdw_file_duplicate()
   \li cdw_file_equal() - a predicate
   \li cdw_file_equal_by_fullpath() - a predicate

   and two special functions:
   \li cdw_file_dealloc_files_from_list() - used to deallocate a group of files from given list
   \li cdw_file_display_file() - used by list display widget

   List of arguments of cdw_file_new() constructor reflects its main usage.
   The function is used to create file object when caller has a path to
   a directory, and a result of scandir() call, i.e. a list of file names in
   the directory. Thus constructor is called with dirpath and filename.

   As a result values of type cdw_file_t should (must?) be used only for
   real files existing in file system.
*/



static cdw_file_t *cdw_file_new_base(void);
static cdw_rv_t    cdw_file_new_stat(cdw_file_t *file);
static cdw_rv_t    cdw_file_new_fullpath(cdw_file_t *file, char const * dirpath, char const * name);


/**
   \brief Create new "file" variable

   Create new variable describing file specified by \p dirpath and
   \p name. Fields of the variable describe the file.
   If given file is a regular file (or link) then \p dirpath should describe
   full directory path to directory, in which the file is located, and
   \p name should be the name of the file in this directory.
   If given file is a directory, then \p name should be name of this
   concrete directory, and \p dirpath should be path to its parent
   directory.

   The fact that there are two arguments: \p dirpath and \p name, is a result
   of context where the function is used. It is used in
   cdw_fs_copy_dirent_to_list(list, dirpath, struct dirent **eps, ),
   where there is one dirpath, and set of filenames stored in
   "struct dirent **eps". cdw_fs_copy_dirent_to_list() calls cdw_file_new()
   for each item in "eps" to create new file. This is why there are two
   arguments to cdw_file_new(), instead of one: fullpath.

   \param dirpath - full path to parent directory in which the file is located
   \param name - name of the file in given directory

   \return pointer to new "file" variable on success
   \return NULL on failure
*/
cdw_file_t *cdw_file_new(char const * dirpath, char const * name)
{
	cdw_assert (dirpath, "ERROR: \"dirpath\" argument is NULL\n");

	cdw_file_t *file = cdw_file_new_base();
	if (!file) {
		cdw_vdm ("ERROR: failed to create new file with cdw_file_new_base()\n");
		return (cdw_file_t *) NULL;
	}

	cdw_rv_t crv = cdw_file_new_fullpath(file, dirpath, name);
	cdw_assert (crv == CDW_OK, "ERROR: call to cdw_file_new_fullpath() failed\n");

	crv = cdw_file_new_stat(file);
	cdw_assert (crv == CDW_OK, "ERROR: call to cdw_file_new_stat() failed\n");

	if (!strcmp(file->fullpath + file->name_start, "../")
	    || !strcmp(file->fullpath + file->name_start, "..")) {
		file->is_ref_to_parent_dir = true;
	} else {
		file->is_ref_to_parent_dir = false;
	}
#if 0
	/* 2014-02-24: I don't want a dirpath to end with a slash
	   anymore. I'm almost certain that there is a difference in
	   passing "/path/to/symlink/to/dir/" and
	   "/path/to/symlink/to/dir". If such a path is on list of
	   selected directories, lstat() will recognize first path as
	   a dir (not what I want), and the second path as symlink
	   (this is what I want). The difference is significant when
	   you think about state of "follow symlinks when calculating
	   space" checkbox in Configuration window.

	   TODO: perhaps the function should be rewritten so that it
	   makes sure that there is no trailing slash in its arg?
	*/
	if (file->type == CDW_FS_DIR) {
		crv = cdw_fs_correct_dir_path_ending(&(file->fullpath));
		if (crv != CDW_OK) {
			cdw_vdm ("ERROR: failed to assure correct ending of dirpath (\"%s\")\n", file->fullpath);
			cdw_file_delete(&file);
			return (cdw_file_t *) NULL;
		}
	}
#endif
	return file;
}





/**
   \brief Call lstat() and stat() for given file

   Call lstat() and stat() for given file to check its parameters, like
   file validity, file type (dir/file/link), file size.

   \return CDW_OK
*/
cdw_rv_t cdw_file_new_stat(cdw_file_t *file)
{
	/* 1: call lstat() to catch links: in case if fullpath is
	   link, the function examines links, not target of link  */

	struct stat stbuf;
	int rv = lstat(file->fullpath, &stbuf);
	if (rv == 0) {
		if (S_ISLNK(stbuf.st_mode)) {
			file->type |= CDW_FS_LINK;
			cdw_sdm ("INFO: is a link: \"%s\"\n", file->fullpath);
		}
		file->invalid = false;
	} else {
		file->invalid = true;
		/* may happen when we create new file object that is not
		   related to any file in file system, i.e. there is a
		   fullpath (created for some reason), but the file itself
		   doesn't exist */
		/* perror("-1 (1) because"); */
		cdw_vdm ("ERROR: lstat() returns -1 for fullpath \"%s\"\n", file->fullpath);
	}

	/* 2: stat() called for links examines targets */
	rv = stat(file->fullpath, &stbuf);
	int e = errno;
	if (rv == 0) {
		file->invalid = false;
	} else {
		file->invalid = true;
		cdw_vdm ("WARNING: stat() failed for fullpath \"%s\" (calling stat() for links examines target); error = \"%s\"\n", file->fullpath, strerror(e));
		/* TODO: check how caller of this function behaves when CDW_SYS_ERROR is returned */
		/* return CDW_SYS_ERROR; */
	}

	if (S_ISDIR(stbuf.st_mode)) {
		/* TODO: shouldn't we assign some non-zero size? */
		file->size = 0;
		file->type |= CDW_FS_DIR;
	} else if (S_ISREG(stbuf.st_mode)) {
		file->size = stbuf.st_size;;
		file->type |= CDW_FS_FILE;
	} else if (S_ISLNK(stbuf.st_mode)) {
		file->size = 0;
		file->type |= CDW_FS_LINK;
	} else {
		/* char device, block device, named pipe, socket */
		file->size = 0;
		file->type |= CDW_FS_OTHER;
	}

	return CDW_OK;
}





/**
   \brief Create correct fullpath for given file

   Function concatenates \p dirpath and \p name, making sure that there
   is exactly one slash between them. The function also sets correct value
   of "name_start" field and - if needed - of printable_fullpath field.

   Function does not check if ending slash is missing if result fullpath
   is a path to a dir, because the function can't tell if given file is
   a dir or not. So resulting fullpath is not always 100% correct.

   \param file - file in which a fullpath (and other fields) needs to be set
   \param dirpath - path to directory, first of two components of fullpath
   \param name - name of file in directory specified by dirpath; second of two components of fullpath

   \return CDW_ERROR on errors
   \return CDW_OK on success
*/
cdw_rv_t cdw_file_new_fullpath(cdw_file_t *file, char const * dirpath, char const * name)
{
	char *correct_dirpath = strdup(dirpath);
	cdw_rv_t crv = cdw_fs_correct_dir_path_ending(&correct_dirpath);
	cdw_assert (crv == CDW_OK, "ERROR: failed to assure correct ending of dirpath\n");
	file->fullpath = cdw_string_concat(correct_dirpath, name, (char *) NULL);
	free(correct_dirpath);
	correct_dirpath = (char *) NULL;
	if (!file->fullpath) {
		cdw_assert (0, "ERROR: failed to concatenate dirpath and name: \"%s\" + \"%s\"\n", dirpath, name);
		return CDW_ERROR;
	}

	file->name_start = cdw_fs_get_filename_start(file->fullpath);
	if (file->name_start < 0) {
		cdw_vdm ("ERROR: failed to get start of file name in fullpath \"%s\"\n", file->fullpath);
		free(file->fullpath);
		file->fullpath = (char *) NULL;
		cdw_assert (0, "ERROR: failed to get filename start in \"%s\"\n", file->fullpath);
		return CDW_ERROR;
	}

	file->printable_fullpath = cdw_string_get_printable_if_needed(file->fullpath);

#ifndef NDEBUG
	if (file->printable_fullpath) {
		ssize_t i = cdw_fs_get_filename_start(file->fullpath);
		ssize_t j = cdw_fs_get_filename_start(file->printable_fullpath);
		cdw_assert (i == j, "ERROR: wrong assumption about file name start in path and printable path\n");
	}
#endif

	return CDW_OK;
}





/**
   \brief Allocate memory for "file" variable, initialize it

   Function allocates memory for "file" variable and
   initializes some of its fields to some default values.

   The function does not allocate space for ->fullpath or ->printable_fullpath.

   \return pointer to newly allocated varible on success
   \return NULL on failure
*/
cdw_file_t *cdw_file_new_base(void)
{
	cdw_file_t *file = (cdw_file_t *) malloc(sizeof (cdw_file_t));
	if (!file) {
		cdw_vdm ("ERROR: failed to allocate memory for new file\n");
		return (cdw_file_t *) NULL;
	} else {
		file->fullpath = (char *) NULL;
		file->printable_fullpath = (char *) NULL;
		file->name_start = -1;
		file->size = 0;
		file->invalid = true;
		file->type = CDW_FS_NONE;
		file->is_ref_to_parent_dir = false;

		return file;
	}
}





/**
   \brief Duplicate "file" variable

   Function creates exact duplicate of given \file. Values of
   all fields are copied to new "file" variable, and pointer to
   this new variable is returned.

   No memory/pointer is shared between original file and its duplicate.

   The function doesn't check consistency or semantic validity of data
   in \p file. It only ensures that file->fullpath is not NULL.

   \return pointer to new variable being copy of argument - on success
   \return NULL on failure
*/
cdw_file_t *cdw_file_duplicate(cdw_file_t const * file)
{
	cdw_assert (file, "ERROR: \"file\" argument is NULL\n");
	cdw_assert (file->fullpath, "ERROR: file's fullpath is NULL\n");

	cdw_file_t *copy = (cdw_file_t *) malloc(sizeof(cdw_file_t));
	if (!copy) {
		cdw_vdm ("ERROR: failed to allocate memory for duplicate of file\n");
		return (cdw_file_t *) NULL;
	}

	if (file->printable_fullpath) {
		copy->printable_fullpath = strdup(file->printable_fullpath);
		if (!copy->printable_fullpath) {
			cdw_file_delete(&copy);

			cdw_vdm ("ERROR: failed to duplicate printable fullpath of file (\"%s\")\n", file->printable_fullpath);
			return (cdw_file_t *) NULL;
		}
	} else {
		copy->printable_fullpath = (char *) NULL;
	}

	copy->name_start = file->name_start;

	copy->fullpath = strdup(file->fullpath);
	if (!copy->fullpath) {
		cdw_file_delete(&copy);

		cdw_vdm ("ERROR: failed to duplicate fullpath of file (\"%s\")\n", file->fullpath);
		return (cdw_file_t *) NULL;
	}

	copy->type = file->type;
	copy->size = file->size;
	copy->invalid = file->invalid;
	copy->is_ref_to_parent_dir = file->is_ref_to_parent_dir;

	return copy;
}





/**
   \brief Deallocate all resources referenced by "file" variable, free the "file"

   Function frees() all memory referenced to by fields of \p file, and frees
   \p file itself. NULL is assigned to \p file before return, so that caller
   doesn't have to do this himself.

   \return CDW_ERROR if \p file is a pointer to NULL file
   \return CDW_OK on success
*/
cdw_rv_t cdw_file_delete(cdw_file_t **file)
{
	cdw_assert (file, "ERROR: \"file\" argument is NULL\n");

	if (*file == (cdw_file_t *) NULL) {
		/* TODO: shouldn't we issue only a warning and return CDW_OK? */
		cdw_vdm ("ERROR: \"file\" pointer points to NULL file\n");
		return CDW_ERROR;
	}

	if ((*file)->printable_fullpath) {
		free((*file)->printable_fullpath);
		(*file)->printable_fullpath = (char *) NULL;
	}

	if ((*file)->fullpath) {
		free((*file)->fullpath);
		(*file)->fullpath = (char *) NULL;
	} else {
		cdw_vdm ("WARNING: file's fullpath is NULL\n");
	}

	free(*file);
	*file = (cdw_file_t *) NULL;

	return CDW_OK;
}





/**
   \brief Compare all fields of two "file" variables

   Function compares values of all fields of given "file" variables.

   The function may be used as argument to cdw_dll_append()

   \return true if all fields of the variables have the same values
   \return false otherwise
*/
bool cdw_file_equal(void const * _file1, void const * _file2)
{
	cdw_assert (_file1, "ERROR: first argument is NULL\n");
	cdw_assert (_file2, "ERROR: second argument is NULL\n");

	cdw_file_t const * file1 = (cdw_file_t const *) _file1;
	cdw_file_t const * file2 = (cdw_file_t const *) _file2;

	cdw_assert (file1, "ERROR: file1 is NULL\n");
	cdw_assert (file2, "ERROR: file2 is NULL\n");
	cdw_assert (file1->name_start > 0, "ERROR: file1->name_start <= 0: %zd\n", file1->name_start);
	cdw_assert (file2->name_start > 0, "ERROR: file1->name_start <= 0: %zd\n", file2->name_start);
	cdw_assert (file1->fullpath, "ERROR: file1->fullpath is NULL\n");
	cdw_assert (file2->fullpath, "ERROR: file2->fullpath is NULL\n");

	if (!file1 || !file2) {
		cdw_vdm ("ERROR: one (or both) argument is NULL\n");
		return false;
	}

	if (!file1->fullpath || !file2->fullpath) {
		cdw_vdm ("ERROR: one (or both) fullpath is NULL\n");
		return false;
	} else {
		if (0 != strcmp(file1->fullpath, file2->fullpath)) {
			cdw_sdm ("INFO: files have different fullpaths\n    \"%s\"\n    \"%s\"\n", file1->fullpath, file2->fullpath);
			return false;
		}
	}

	if (!file1->printable_fullpath && !file2->printable_fullpath) {
		/* NULL "printable_fullpath" strings are allowed. */
	} else if (file1->printable_fullpath && file2->printable_fullpath) {
		if (0 != strcmp(file1->printable_fullpath, file2->printable_fullpath)) {
			cdw_sdm ("INFO: files have different printable fullpaths\n");
			return false;
		}
	} else {
		cdw_sdm ("INFO: one printable fullpath is NULL, the other is non-NULL\n");
		return false;
	}

	if (file1->name_start != file2->name_start) {
		cdw_vdm ("ERROR: different file name starts: %zd != %zd\n", file1->name_start, file2->name_start);
		return false;
	}
	if (file1->is_ref_to_parent_dir != file2->is_ref_to_parent_dir) {
		cdw_sdm ("INFO: files differ in \"is ref to parent dir\" values\n");
		return false;
	}
	if (file1->type != file2->type) {
		cdw_sdm ("INFO: files differ in file types\n");
		return false;
	}
	if (file1->size != file2->size) {
		cdw_sdm ("INFO: files differ in file sizes\n");
		return false;
	}
	if (file1->invalid != file2->invalid) {
		cdw_sdm ("INFO: files differ in \"invalid\" values\n");
		return false;
	}
	return true;
}





/**
   \brief Compare a given fullpath with fullpath of given file

   The function may be used as argument to cdw_dll_append()

   \return true if given fullpath and given file->fullpath are equal
   \return false otherwise
*/
bool cdw_file_equal_by_fullpath(void const * _fullpath, void const * _file)
{
	cdw_assert (_fullpath, "ERROR: first argument is NULL\n");
	cdw_assert (_file, "ERROR: second argument is NULL\n");

	char const * fullpath = (char const *) _fullpath;
	cdw_file_t const * file = (cdw_file_t const *) _file;

	if (0 == strcmp(fullpath, file->fullpath)) {
		return true;
	} else {
		return false;
	}
}





/**
   \brief Display in given display one file and its attributes

   Display in given \p row of given \p display one file. Set font/background
   of the file properly (based on value of \p isreverse), display additional
   file attributes in the same \p row.

   \p row is a number of row, in which given file should be displayed

   \p isreverse controls attributes of displayed item:
   \li if set to true then item is displayed with foreground and background
   attributes reversed compared to whole file selector window
   \li if set to false then item is displayed with the same foreground and
   background attributes as whole file selector window

   \p row is relative to beginning of display's window. If a window
   has only 15 rows, \p row is always in range 0-14.

   \param display - display to be used to show the file
   \param row - number of window row (zero-based), in which to display file
   \param data - cdw_file_t entry (file or dir) to be displayed
   \param isreverse - controls attributes of displayed item
*/
void cdw_file_display_file(void *display, void *data, size_t row, size_t h_offset, bool isreverse)
{
	cdw_assert (display, "ERROR: \"display\" argument is NULL\n");
	CDW_LIST_DISPLAY *displ = (CDW_LIST_DISPLAY *) display;
	cdw_assert (displ->subwindow, "ERROR: subwindow of display is NULL\n");

	int n_cols = getmaxx(displ->subwindow);
	int n_rows = getmaxy(displ->subwindow);
	cdw_assert (n_cols > 0, "ERROR: we have subwindow with 0 columns, or NULL subwindow\n");
	cdw_assert (n_rows > (int) row, "ERROR: row number larger than number of rows in subwindow\n");
	cdw_assert (data, "ERROR: you try to display NULL file\n");

	cdw_file_t *file = (cdw_file_t *) data;
	cdw_assert (file->fullpath, "ERROR: file has no fullpath\n");

	/* this fills space between item name on the left side of the
	   window and item attributes on the right side of the window */
	mvwhline(displ->subwindow, (int) row, 0, ' ', n_cols);

	char *string = file->printable_fullpath
		? file->printable_fullpath : file->fullpath;

	if (displ->display_format == CDW_LIST_DISPLAY_FORMAT_SHORT) {
		/* Display just a file/dir name, no path. */
		string += file->name_start;

	} else if (displ->display_format == CDW_LIST_DISPLAY_FORMAT_LONG) {
		/* string = string - display full path. */

	} else {
		cdw_assert (0, "ERROR: no code to handle display format %d\n", displ->display_format);
	}

	long unsigned int attr = COLOR_PAIR(displ->colors);
	if (file->type & CDW_FS_DIR) {
		attr |= A_BOLD;
	}
	if (isreverse) {
		attr |= A_REVERSE;
	} else {
		attr |= A_NORMAL;
	}
	(void) wattrset(displ->subwindow, attr);

	char format[10];
	if (file->invalid) {
		strcpy(format, "!%.*s");
	} else {
		if (file->type & CDW_FS_LINK) {
			strcpy(format, "~%.*s");
		} else {
			strcpy(format, "%.*s");
		}
	}
	if (strlen(string) >= h_offset) {
		mvwprintw(displ->subwindow, (int) row, 0, format, n_cols - 7, string + h_offset);
	} else {
		mvwprintw(displ->subwindow, (int) row, 0, " ");
	}

	/* testing code */
#if 0
	if (string[0] == 'S') { /* I have created local files with names with special chars, that start with 'S' */
		wchar_t dest[1000];
		size_t conv_len = mbstowcs(dest, string, 50);
		size_t len = strlen(string);
		fprintf(stderr, "INFO: analyzing \"%s\"\n", string);
		for (size_t i = 0; i < len; i++) {
			fprintf(stderr, "INFO: char: %u - > '%c'\n", (unsigned char) string[i], string[i]);
		}
		fprintf(stderr, "INFO: char: strlen(string) = %zd\n", strlen(string));
		for (size_t i = 0; i < len; i++) {
			fprintf(stderr, "INFO: wchar: %u - > '%c'\n", (unsigned char) dest[i], dest[i]);
			//perror("d");
		}
		fprintf(stderr, "INFO: wchar: wcslen(dest) = %zd, mbstowcs(): %zd\n\n\n", wcslen(dest), conv_len);
	}
#endif

	if (file->invalid) {
		/* 2TRANS: "E" stands for "Error" - this label marks invalid
		   item; please keep as short as possible */
		mvwprintw(displ->subwindow, (int) row, n_cols - 5, _("E"));
	} else {
		if (file->type & CDW_FS_DIR) {
			mvwprintw(displ->subwindow, (int) row, n_cols - 5,
				  /* 2TRANS: this is label indicating that labeled item
				     is a directory; please keep as short as possible */
				  _("DIR"));

		} else if (file->type & CDW_FS_FILE) {
			float size = (float) (file->size >= CDW_1_MEGA ?
					      (float) file->size / (1.0 * CDW_1_MEGA) :
					      (float) file->size / 1024.0);
			mvwprintw(displ->subwindow, (int) row, n_cols - 8, "%6.1f%s", size,
				  (file->size >= CDW_1_MEGA ?
				   /* 2TRANS: suffix for megabytes,
				      please keep as short as possible */
				   _("M") :
				   /* 2TRANS: suffix for kilobytes,
				      please keep as short as possible */
				   _("K")));
		} else { /* currently unsupported file type */
			/* 2TRANS: "n/a" means here that there is no label suitable
			   for description of given item; keep as short as possible */
			mvwprintw(displ->subwindow, (int) row, n_cols - 5, _("n/a"));
		}
	}
	(void) wattrset(displ->subwindow, A_NORMAL | COLOR_PAIR(displ->colors));

	return;
}





/**
   \brief Deallocate all files that are on given list

   Function frees all memory associated with all 'file' variables
   (including the "file" variables themselves) stored on given
   list. The list itself is not affected, only payload of the list is
   free()d.

   \param list - list of files, from which you want to dealloc all files

*/
void cdw_file_dealloc_files_from_list(cdw_dll_item_t * list)
{
	if (!list) {
		cdw_vdm ("WARNING: passed NULL list to the function\n");
		return;
	}

	for (cdw_dll_item_t * item = list; item; item = item->next) {
		cdw_file_delete((cdw_file_t **) &(item->data));
	}

	return;
}





/**
   \brief Remove last part of a path

   Create a new fullpath by removing last part (file name or directory
   name) from \p file's fullpath.

   In contrast to all other places, dirpath returned by this function
   does not end with a slash. This is a special case needed in special
   place.

   Returned pointer is owned by caller.

   \param file - file with initial value of path

   \return new, shortened path on success
   \return NULL on failure
*/
char *cdw_file_shorten_fullpath(cdw_file_t const * file)
{
	ssize_t pos = cdw_fs_get_filename_start(file->fullpath);
	if (pos == -1) {
		return (char *) NULL;
	}


	/* This removes trailing slash from directory name.  The name
	   of a directory shall not contain an ending slash.

	   'pos > 1' condition is met for all directories except for
	   root directory. */
	if (pos > 1 && file->fullpath[pos - 1] == '/') {
		pos--;
	}

	char *new_fullpath = strndup(file->fullpath, (size_t) pos);
	cdw_vdm ("INFO: original fullpath  = \"%s\"\n", file->fullpath);
	cdw_vdm ("INFO: shortened fullpath = \"%s\"\n", new_fullpath);
	return new_fullpath;
}





#ifdef CDW_UNIT_TEST_CODE





static void     test_cdw_file_equal(void);
static void     test_cdw_file_new_base(void);
static void     test_cdw_file_new_file_delete(void);
static void     test_cdw_file_duplicate(void);
static cdw_rv_t test_cdw_file_new_helper(char **cwd, char **dirpath, char **filename);
static void     test_cdw_file_new_fullpath(void);





void cdw_file_run_tests(void)
{
	fprintf(stderr, "testing cdw_file.c\n");

	test_cdw_file_equal();
	test_cdw_file_new_base();
	test_cdw_file_new_file_delete();
	test_cdw_file_duplicate();
	test_cdw_file_new_fullpath();

	fprintf(stderr, "done\n\n");

	return;
}





void test_cdw_file_new_base(void)
{
	fprintf(stderr, "\ttesting cdw_file_new_base()... ");

	cdw_file_t *file = cdw_file_new_base();

	cdw_assert_test (file, "ERROR: function cdw_file_new_base() failed to allocate file\n");
	cdw_assert_test (!file->fullpath, "ERROR: ->fullpath not set to NULL\n");
	cdw_assert_test (file->name_start == -1, "ERROR: ->name_start not set to -1\n");

	cdw_assert_test (file->size == 0, "ERROR: ->size not set to zero\n");
	cdw_assert_test (file->invalid == true, "ERROR: ->invalid not set to true\n");
	cdw_assert_test (file->type == CDW_FS_NONE, "ERROR: ->type not set to CDW_FS_NONE\n");
	cdw_assert_test (file->is_ref_to_parent_dir == false, "ERROR: ->is_ref_to_parent_dir not set to false\n");

	cdw_file_delete(&file);

	fprintf(stderr, "OK\n");
	return;
}





void test_cdw_file_equal(void)
{
	fprintf(stderr, "\ttesting cdw_file_equal()... ");

	cdw_file_t file1, file2;
	bool test;
	cdw_rv_t crv;


	/* Prepare. */
	file1.fullpath = strdup("/home/acerion/");
	file2.fullpath = strdup("/home/acerion/");
	file1.printable_fullpath = (char *) NULL;
	file2.printable_fullpath = (char *) NULL;
	file1.name_start = 6;
	file2.name_start = 6;
	file1.type = CDW_FS_FILE;
	file2.type = CDW_FS_FILE;
	file1.size = 123456;
	file2.size = 123456;
	file1.invalid = true;
	file2.invalid = true;
	file1.is_ref_to_parent_dir = true;
	file2.is_ref_to_parent_dir = true;


	/* Test. */

	/* at the beginning the files are equal */
	test = cdw_file_equal((void const *) &file1, (void const *) &file2);
	cdw_assert_test (test == true, "ERROR: equal files recognized as not equal\n");


	/* testing with different "fullpath" field values */
	crv = cdw_string_set(&(file1.fullpath), "/my/test/path/1/");
	cdw_assert_test (crv == CDW_OK, "ERROR: can't set new file fullpath\n");

	test = cdw_file_equal((void const *) &file1, (void const *) &file2);
	cdw_assert_test (test == false, "ERROR: files with different fullpaths are recognized as equal\n");

	crv = cdw_string_set(&(file1.fullpath), "/home/acerion/");
	cdw_assert_test (crv == CDW_OK, "ERROR: can't restore old file fullpath\n");

	test = cdw_file_equal((void const *) &file1, (void const *) &file2);
	cdw_assert_test (test == true, "ERROR: test failed after restoring original fullpath in file1\n");


	/* testing with different "printable_fullpath" field values */
	crv = cdw_string_set(&(file1.printable_fullpath), "my test string");
	cdw_assert_test (crv == CDW_OK, "ERROR: can't set file1 printable fullpath\n");
	test = cdw_file_equal((void const *) &file1, (void const *) &file2);
	cdw_assert_test (test == false, "ERROR: files with different printable fullpaths are recognized as equal (1)\n");

	crv = cdw_string_set(&(file2.printable_fullpath), "let there be chars");
	cdw_assert_test (crv == CDW_OK, "ERROR: can't set file2 printable fullpath (1)\n");
	test = cdw_file_equal((void const *) &file1, (void const *) &file2);
	cdw_assert_test (test == false, "ERROR: files with different printable fullpaths are recognized as equal (2)\n");

	crv = cdw_string_set(&(file2.printable_fullpath), "my test string");
	cdw_assert_test (crv == CDW_OK, "ERROR: can't set file2 printable fullpath (2)\n");
	test = cdw_file_equal((void const *) &file1, (void const *) &file2);
	cdw_assert_test (test == true, "ERROR: files with equal printable fullpaths are recognized as non-equal\n");


	/* testing with different "type" field values */
	file1.type = CDW_FS_DIR;
	file2.type = CDW_FS_FILE;
	test = cdw_file_equal((void const *) &file1, (void const *) &file2);
	cdw_assert_test (test == false, "ERROR: files with different types aren't recognized as different (1)\n");

	file1.type = CDW_FS_FILE | CDW_FS_LINK;   /* A link to a file. */
	file2.type = CDW_FS_FILE;
	test = cdw_file_equal((void const *) &file1, (void const *) &file2);
	cdw_assert_test (test == false, "ERROR: files with different types aren't recognized as different (2)\n");

	file1.type = CDW_FS_DIR;
	file2.type = CDW_FS_DIR;
	test = cdw_file_equal((void const *) &file1, (void const *) &file2);
	cdw_assert_test (test == true, "ERROR: files with the same types aren't recognized as equal (1)\n");

	file1.type = CDW_FS_LINK;
	file2.type = CDW_FS_LINK;
	test = cdw_file_equal((void const *) &file1, (void const *) &file2);
	cdw_assert_test (test == true, "ERROR: files with the same types aren't recognized as equal (2)\n");

	file1.type = CDW_FS_LINK;
	file2.type = CDW_FS_DIR;
	test = cdw_file_equal((void const *) &file1, (void const *) &file2);
	cdw_assert_test (test == false, "ERROR: files with different types aren't recognized as different (2)\n");

	file1.type = CDW_FS_FILE;
	file2.type = CDW_FS_FILE;
	test = cdw_file_equal((void const *) &file1, (void const *) &file2);
	cdw_assert_test (test == true, "ERROR: files with restored the same types aren't recognized as the same\n");


	/* testing with different "size" field values */
	file1.size = 123456;
	file2.size = 123456;
	test = cdw_file_equal((void const *) &file1, (void const *) &file2);
	cdw_assert_test (test == true, "ERROR: files with the same sizes are recognized as different (1)\n");

	file1.size = 0;
	file2.size = 0;
	test = cdw_file_equal((void const *) &file1, (void const *) &file2);
	cdw_assert_test (test == true, "ERROR: files with the same sizes are recognized as different (2)\n");

	file1.size = 654321;
	file2.size = 123456;
	test = cdw_file_equal((void const *) &file1, (void const *) &file2);
	cdw_assert_test (test == false, "ERROR: files with different sizes are recognized as the same (1)\n");

	file1.size = 0;
	file2.size = 123456;
	test = cdw_file_equal((void const *) &file1, (void const *) &file2);
	cdw_assert_test (test == false, "ERROR: files with different sizes are recognized as the same (2)\n");

	file1.size = 123456;
	file2.size = 123456;
	test = cdw_file_equal((void const *) &file1, (void const *) &file2);
	cdw_assert_test (test == true, "ERROR: files with restored the same sizes are recognized as different\n");


	/* testing with different "invalid" field values */
	file1.invalid = true;
	file2.invalid = true;
	test = cdw_file_equal((void const *) &file1, (void const *) &file2);
	cdw_assert_test (test == true, "ERROR: files with the same 'invalid' fields are recognized as different\n");

	file1.invalid = false;
	file2.invalid = true;
	test = cdw_file_equal((void const *) &file1, (void const *) &file2);
	cdw_assert_test (test == false, "ERROR: files with different 'invalid' fields are recognized as the same\n");

	file1.invalid = false;
	file2.invalid = false;
	test = cdw_file_equal((void const *) &file1, (void const *) &file2);
	cdw_assert_test (test == true, "ERROR: files with restored the same 'invalid' fields are recognized as different\n");


	/* testing with different "is_ref_to_parent_dir" field values */
	file1.is_ref_to_parent_dir = true;
	file2.is_ref_to_parent_dir = true;
	test = cdw_file_equal((void const *) &file1, (void const *) &file2);
	cdw_assert_test (test == true, "ERROR: files with the same 'is_ref_to_parent_dir' fields are recognized as different\n");

	file1.is_ref_to_parent_dir = false;
	file2.is_ref_to_parent_dir = true;
	test = cdw_file_equal((void const *) &file1, (void const *) &file2);
	cdw_assert_test (test == false, "ERROR: files with different 'is_ref_to_parent_dir' fields are recognized as the same\n");

	file1.is_ref_to_parent_dir = false;
	file2.is_ref_to_parent_dir = false;
	test = cdw_file_equal((void const *) &file1, (void const *) &file2);
	cdw_assert_test (test == true, "ERROR: files with restored the same 'is_ref_to_parent_dir' fields are recognized as different\n");


	/* Clean up. */
	CDW_STRING_DELETE(file1.fullpath);
	CDW_STRING_DELETE(file2.fullpath);

	CDW_STRING_DELETE(file1.printable_fullpath);
	CDW_STRING_DELETE(file2.printable_fullpath);


	fprintf(stderr, "OK\n");
	return;
}





cdw_rv_t test_cdw_file_new_helper(char **cwd, char **dirpath, char **filename)
{
	cdw_assert_test (!(*cwd), "ERROR: cwd should be pointer to NULL");
	cdw_assert_test (!(*dirpath), "ERROR: dirpath should be pointer to NULL");
	cdw_assert_test (!(*filename), "ERROR: filename should be pointer to NULL");

	/* first get some existing dir path */
	*cwd = (char *) malloc(PATH_MAX); /* PATH_MAX already includes ending NULL */
	cdw_assert_test (*cwd, "ERROR: failed to allocate memory for cwd\n");

	char *res = getcwd(*cwd, PATH_MAX); /* Returns up to N chars (including terminating NULL) via *cwd. */
	cdw_assert_test (res, "ERROR: failed to get current working directory (1)\n");
	cdw_assert_test (res == *cwd, "ERROR: failed to get current working directory (2)\n");
	cdw_rv_t crv = cdw_fs_correct_dir_path_ending(cwd);
	cdw_assert_test (crv == CDW_OK, "ERROR: failed to ensure proper ending of cwd\n");


	ssize_t start = cdw_fs_get_filename_start(*cwd);
	cdw_assert_test (start >= 0, "ERROR: failed to get start of file name in path \"%s\"\n", *cwd);
	// cdw_vdm ("INFO: cwd = \"%s\", start = %d\n", *cwd, start);
	*filename = strdup(*cwd + start);
	*dirpath = strndup(*cwd, (size_t) start);
	cdw_assert_test (*filename && *dirpath, "ERROR: failed to strdup() filename and dirpath\n");
	// cdw_vdm ("INFO: dirtpath = \"%s\"\n", *dirpath);
	// cdw_vdm ("INFO: filename = \"%s\"\n", *filename);

	return CDW_OK;
}





void test_cdw_file_new_file_delete(void)
{
	fprintf(stderr, "\ttesting cdw_file_new() and cdw_file_delete()... ");


	/* Prepare. */
	char *cwd = (char *) NULL;
	char *dirpath = (char *) NULL;
	char *filename = (char *) NULL;
	test_cdw_file_new_helper(&cwd, &dirpath, &filename);


	/* Test.  */
	cdw_file_t *file = cdw_file_new(dirpath, filename);
	cdw_assert_test (file, "ERROR: function cdw_file_new_base() failed to allocate file\n");

	cdw_assert_test (file->fullpath, "ERROR: fullpath in new file base is NULL\n");
	cdw_assert_test (!strcmp(file->fullpath, cwd), "ERROR: incorrect fullpath: should be \"%s\", is: \"%s\"\n", cwd, file->fullpath);
	cdw_assert_test (!strcmp(file->fullpath + file->name_start, filename), "ERROR: incorrect name: should be \"%s\", is: \"%s\"\n", filename, file->fullpath + file->name_start);
	cdw_assert_test (file->size == 0, "ERROR: size of new file is not 0 for dir\n");
	cdw_assert_test (file->invalid == false, "ERROR: existing file is invalid\n");
	cdw_assert_test (file->type == CDW_FS_DIR, "ERROR: dir is not recognized as dir\n");
	cdw_assert_test (file->is_ref_to_parent_dir == false, "ERROR: current dir should not be reference to parent dir\n");

	/* now test cdw_file_delete(), but there isn't much to test */

	cdw_file_delete(&file);
	cdw_assert_test (!file, "ERROR: failed to correctly delete file, file is not NULL\n");


	/* Clean up. */
	CDW_STRING_DELETE(cwd);
	CDW_STRING_DELETE(filename);
	CDW_STRING_DELETE(dirpath);

	fprintf(stderr, "OK\n");
	return;
}





void test_cdw_file_duplicate(void)
{
	fprintf(stderr, "\ttesting cdw_file_duplicate()... ");


	/* Prepare. */
	char *cwd = (char *) NULL;
	char *dirpath = (char *) NULL;
	char *filename = (char *) NULL;
	test_cdw_file_new_helper(&cwd, &dirpath, &filename);


	/* Test. */
	/* We tested creating file before, but just in case... */
	cdw_file_t *file = cdw_file_new(dirpath, filename);
	cdw_assert_test (file, "ERROR: function cdw_file_new_base() failed to allocate file\n");

	cdw_assert_test (file->fullpath, "ERROR: fullpath in new file base is NULL\n");
	cdw_assert_test (!strcmp(file->fullpath, cwd), "ERROR: incorrect fullpath: should be \"%s\", is: \"%s\"\n", cwd, file->fullpath);
	cdw_assert_test (!strcmp(file->fullpath + file->name_start, filename), "ERROR: incorrect name: should be \"%s\", is: \"%s\"\n", filename, file->fullpath + file->name_start);
	cdw_assert_test (file->size == 0, "ERROR: size of new file is not 0 for dir\n");
	cdw_assert_test (file->invalid == false, "ERROR: existing file is invalid\n");
	cdw_assert_test (file->type == CDW_FS_DIR, "ERROR: dir is not recognized as dir\n");
	cdw_assert_test (file->is_ref_to_parent_dir == false, "ERROR: current dir should not be reference to parent dir\n");

	/* Small manual change to improve testing of duplicating
	   file. I don't want to test duplicating file with non-zero
	   size, this would be not too interesting ('file' is a
	   directory in this testcase, and 'size' would be zero). */
	file->size = 17345;

	/* now do the real testing */
	cdw_file_t *duplicate = cdw_file_duplicate(file);

	cdw_assert_test (duplicate, "ERROR: failed to create duplicate of a file\n");
	cdw_assert_test (!strcmp(file->fullpath, duplicate->fullpath),
			 "ERROR: file and its duplicate have different fullpaths:\nfile->fullpath      = \"%s\"\nduplicate->fullpath = \"%s\"\n",
			 file->fullpath, duplicate->fullpath);
	cdw_assert_test (file->name_start == duplicate->name_start,
			 "ERROR: file and its duplicate have different name starts:\nfile->name_start      = \"%zd\"\nduplicate->name_start = \"%zd\"\n",
			 file->name_start, duplicate->name_start);
	cdw_assert_test (file->size == duplicate->size,
			 "ERROR: file and its duplicate have different file sizes:\nfile->size      = %lld\nduplicate->size = %lld\n",
			 file->size, duplicate->size);
	cdw_assert_test (file->invalid == duplicate->invalid,
			 "ERROR: file and its duplicate have different \"invalid\" field values:\nfile->invalid      = %s\nduplicate->invalid = %s\n",
			 file->invalid ? "true" : "false", duplicate->invalid ? "true" : "false");
	cdw_assert_test (file->type == duplicate->type,
			 "ERROR: file and its duplicate have different \"type\" field values:\nfile->type      = %d\nduplicate->type = %d\n",
			 file->type, duplicate->type);
	cdw_assert_test (file->is_ref_to_parent_dir == duplicate->is_ref_to_parent_dir,
			 "ERROR: file and its duplicate have different \"is reference to parent dir\" field values\n");


	/* Clean up. */
	cdw_file_delete(&file);
	cdw_assert_test (!file, "ERROR: failed to correctly delete file, file is not NULL\n");
	cdw_file_delete(&duplicate);
	cdw_assert_test (!duplicate, "ERROR: failed to correctly delete file duplicate, it is not NULL\n");

	CDW_STRING_DELETE(cwd);
	CDW_STRING_DELETE(filename);
	CDW_STRING_DELETE(dirpath);


	fprintf(stderr, "OK\n");
	return;
}





void test_cdw_file_new_fullpath(void)
{
	fprintf(stderr, "\ttesting cdw_file_new_fullpath()... ");


	/* Prepare. */
	/* Set only fields that will be needed to test function */
	cdw_file_t file;
	file.printable_fullpath = (char *) NULL;
	file.name_start = -1;
	file.type = CDW_FS_DIR;
	char *dirpath = strdup("/this/is/a/test/");
	/* this assignment causes compiler warning, but it is intentional:
	   this char will be non-printable, but it is necessary if you want
	   to test creating printable fullpath */
	dirpath[12] = 0xFF;
	char *name = strdup("name");


	/* Test. */
	cdw_rv_t crv = cdw_file_new_fullpath(&file, dirpath, name);
	cdw_assert_test (crv == CDW_OK, "ERROR: call to cdw_file_new_fullpath() failed\n");
	cdw_assert_test (!strcmp(file.printable_fullpath, "/this/is/a/t?st/name"),
			 "ERROR: printable fullpath is incorrect: \"%s\"\n", file.printable_fullpath);
	cdw_assert_test (file.name_start != -1, "ERROR: name_start is -1\n");
	cdw_assert_test (file.name_start == (ssize_t) strlen(dirpath),
			 "ERROR: incorrect value of name_start: %zd\n", file.name_start);


	/* Clean up.*/
	CDW_STRING_DELETE(dirpath);
	CDW_STRING_DELETE(name);
	CDW_STRING_DELETE(file.fullpath);
	CDW_STRING_DELETE(file.printable_fullpath);


	fprintf(stderr, "OK\n");
	return;
}





#endif /* CDW_UNIT_TEST_CODE */
