/**
 *****************************************************************************
 *
 *
 *   SPDX-License-Identifier: BSD-3-Clause
 *   Copyright(c) 2007-2026 Intel Corporation
 * 
 *   These contents may have been developed with support from one or more
 *   Intel-operated generative artificial intelligence solutions.
 *
 *
 ***************************************************************************/

/**
 *****************************************************************************
 * @file cpa_sample_code_ec_montedwds_perf.c
 *
 * @defgroup ecMontEdwdsThreads
 *
 * @ingroup ecMontEdwdsThreads
 *
 * @description
 *      This file contains vectors used in ECED performance test
 *      and function getEcEdTestVectors to select vectors for test
 *
 *****************************************************************************/

#include "cpa.h"
#include "cpa_types.h"
#include "cpa_sample_code_ec_montedwds_vectors.h"

#if CY_API_VERSION_AT_LEAST(2, 3)
/******************
 * Edwards 448 Kg *
 ******************/

static Cpa8U edwards_448_kg_x[64] = {
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x4f, 0x19, 0x70,
    0xc6, 0x6b, 0xed, 0x0d, 0xed, 0x22, 0x1d, 0x15, 0xa6, 0x22, 0xbf,
    0x36, 0xda, 0x9e, 0x14, 0x65, 0x70, 0x47, 0x0f, 0x17, 0x67, 0xea,
    0x6d, 0xe3, 0x24, 0xa3, 0xd3, 0xa4, 0x64, 0x12, 0xae, 0x1a, 0xf7,
    0x2a, 0xb6, 0x65, 0x11, 0x43, 0x3b, 0x80, 0xe1, 0x8b, 0x00, 0x93,
    0x8e, 0x26, 0x26, 0xa8, 0x2b, 0xc7, 0x0c, 0xc0, 0x5e};

static Cpa8U edwards_448_kg_y[64] = {
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x69, 0x3f, 0x46,
    0x71, 0x6e, 0xb6, 0xbc, 0x24, 0x88, 0x76, 0x20, 0x37, 0x56, 0xc9,
    0xc7, 0x62, 0x4b, 0xea, 0x73, 0x73, 0x6c, 0xa3, 0x98, 0x40, 0x87,
    0x78, 0x9c, 0x1e, 0x05, 0xa0, 0xc2, 0xd7, 0x3a, 0xd3, 0xff, 0x1c,
    0xe6, 0x7c, 0x39, 0xc4, 0xfd, 0xbd, 0x13, 0x2c, 0x4e, 0xd7, 0xc8,
    0xad, 0x98, 0x08, 0x79, 0x5b, 0xf2, 0x30, 0xfa, 0x14};

static Cpa8U edwards_448_kg_k[8][64] = {
    {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xb7, 0xbb, 0xc0,
     0x1f, 0xa7, 0x01, 0x05, 0xa7, 0x4f, 0xee, 0xce, 0x15, 0x66, 0xf5,
     0xf9, 0x83, 0x74, 0xd1, 0xee, 0x1e, 0xd8, 0x36, 0xc0, 0x05, 0xb9,
     0x9c, 0x51, 0x38, 0x1d, 0x5e, 0x02, 0x75, 0xee, 0xf3, 0xa4, 0x5b,
     0x54, 0xf0, 0x11, 0xb4, 0x88, 0xa5, 0x72, 0xf4, 0x67, 0x66, 0xed,
     0xc7, 0x8e, 0x80, 0xa0, 0xce, 0xa0, 0x30, 0x39, 0xe8},
    {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xf2, 0xfe, 0x3a,
     0xd2, 0x8f, 0xad, 0x21, 0x35, 0x8f, 0xf9, 0xc3, 0x69, 0xc2, 0x4b,
     0x14, 0xdc, 0x01, 0x0e, 0x8e, 0x04, 0x16, 0x03, 0xde, 0xaf, 0x51,
     0x51, 0x95, 0xaa, 0xc6, 0xdc, 0x63, 0xf7, 0x45, 0xec, 0xfe, 0x4b,
     0x76, 0xe0, 0x77, 0x15, 0xc6, 0xc0, 0xba, 0x82, 0x2c, 0x7c, 0x79,
     0xc3, 0x23, 0x4f, 0x70, 0x35, 0x90, 0x5e, 0xa9, 0x88},
    {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xed, 0xf5, 0xd7,
     0xa9, 0x38, 0x54, 0xb0, 0xe9, 0x78, 0xd7, 0x1a, 0xd6, 0xd0, 0xc3,
     0xec, 0xb9, 0x36, 0x00, 0x73, 0xbc, 0xf1, 0x8e, 0xf1, 0x62, 0xbb,
     0x07, 0x01, 0x5d, 0x5a, 0xa7, 0xc3, 0x55, 0x5c, 0x4a, 0xbc, 0x1d,
     0xf2, 0x7f, 0x22, 0x64, 0x69, 0x06, 0xfe, 0x78, 0xbd, 0x65, 0xe4,
     0x27, 0xfb, 0xd5, 0x0e, 0x85, 0x89, 0x65, 0x76, 0xa0},
    {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xf0, 0x5e, 0x3e,
     0x64, 0xc6, 0x39, 0x86, 0x0f, 0xe2, 0x8c, 0xe3, 0xa4, 0xb1, 0x10,
     0x50, 0xc7, 0x8d, 0xe3, 0x6f, 0x4c, 0x77, 0x50, 0xeb, 0x69, 0xc8,
     0x08, 0x84, 0xc7, 0xb7, 0x3d, 0xea, 0xf5, 0x7c, 0x27, 0x1b, 0x63,
     0xe7, 0x6f, 0xc0, 0xf9, 0xbf, 0xca, 0x11, 0xa2, 0x3f, 0xac, 0xdf,
     0x27, 0xbd, 0x69, 0x7d, 0x12, 0x97, 0x1f, 0x6d, 0xd0},
    {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x82, 0x24, 0x92,
     0x79, 0x7a, 0xe5, 0x96, 0x7f, 0xad, 0xf0, 0xbd, 0xcf, 0x3a, 0xe7,
     0x6b, 0x8d, 0xe3, 0x19, 0x09, 0xd0, 0x7e, 0x26, 0x6e, 0xce, 0x0c,
     0x2a, 0xdd, 0x25, 0x86, 0x4e, 0xf9, 0x7d, 0xd5, 0x3b, 0xae, 0x57,
     0xe7, 0xd4, 0xcc, 0x85, 0x55, 0xdd, 0xb7, 0xa5, 0xa9, 0xe6, 0xf7,
     0xe4, 0x1c, 0x2f, 0xc0, 0xb2, 0x52, 0x9b, 0xf7, 0x2c},
    {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xaf, 0x1d, 0x28,
     0x0a, 0xc6, 0x5e, 0x3c, 0x96, 0x49, 0x7f, 0xd3, 0x1a, 0x2b, 0xb5,
     0xc2, 0x37, 0xb2, 0xd3, 0x50, 0x81, 0x4a, 0x6b, 0x88, 0x8d, 0x5e,
     0x1f, 0x1a, 0x35, 0x78, 0x75, 0x4f, 0x99, 0xdf, 0xa6, 0xfc, 0x10,
     0x57, 0xc5, 0x5f, 0x32, 0x19, 0x75, 0xab, 0x1f, 0x3f, 0xc1, 0x38,
     0xfe, 0x22, 0x40, 0x2d, 0x0e, 0xd2, 0xec, 0x3c, 0x24},
    {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xdf, 0x0e, 0x31,
     0xe8, 0x79, 0xa9, 0xb2, 0x33, 0xbe, 0x19, 0x20, 0x92, 0xf6, 0xab,
     0xa2, 0x49, 0xd3, 0x6c, 0x5b, 0x45, 0xcf, 0xfd, 0x45, 0x71, 0xe5,
     0xf7, 0x3a, 0xea, 0x4e, 0x7a, 0x69, 0x08, 0xee, 0x67, 0x50, 0x90,
     0x94, 0x9e, 0xf7, 0x85, 0x88, 0xf5, 0x21, 0x91, 0xe6, 0x55, 0xc6,
     0x20, 0x43, 0x90, 0x14, 0xe7, 0x64, 0x1e, 0xbe, 0x80},
    {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xaa, 0x4a, 0x9a,
     0x57, 0x90, 0x44, 0x6b, 0xdf, 0xd5, 0x7d, 0xc7, 0x32, 0xa4, 0xee,
     0x89, 0x9b, 0x20, 0xfd, 0x1f, 0xbe, 0x00, 0xb0, 0xe9, 0xf7, 0x04,
     0x4e, 0x4b, 0x94, 0x50, 0xa9, 0x86, 0xba, 0x61, 0x87, 0xec, 0xaf,
     0x4a, 0x06, 0x75, 0x2a, 0xcb, 0x02, 0xa6, 0x02, 0x0e, 0x99, 0x70,
     0x91, 0x0c, 0x5c, 0x48, 0x98, 0x68, 0xc3, 0x01, 0x80}};

static Cpa8U edwards_448_kg_u[8][64] = {
    {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xb0, 0x2f, 0x7d,
     0x05, 0x80, 0xfd, 0x8e, 0x88, 0xf3, 0xfc, 0x8e, 0xcd, 0x47, 0xf4,
     0x34, 0x99, 0xb0, 0x00, 0x0f, 0xaf, 0x1e, 0x84, 0xd0, 0xc2, 0x28,
     0x37, 0x36, 0xc9, 0x91, 0xc4, 0xa6, 0x44, 0x47, 0xce, 0x4e, 0x8d,
     0x8a, 0x6c, 0x74, 0x01, 0x0b, 0xaf, 0x72, 0x6e, 0xf2, 0x00, 0x06,
     0xbc, 0xf1, 0xfa, 0x99, 0x0e, 0x7a, 0x82, 0x22, 0x87},
    {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xd2, 0xff, 0xd9,
     0xb5, 0x3d, 0x3b, 0xff, 0x14, 0x1f, 0x68, 0xcf, 0x02, 0x48, 0x02,
     0x0a, 0x14, 0xe2, 0xf1, 0xab, 0xa4, 0x33, 0x25, 0x8c, 0x33, 0xc2,
     0x90, 0x60, 0x7f, 0x78, 0xea, 0xb2, 0xc6, 0xb9, 0x69, 0xbd, 0xf2,
     0x26, 0x77, 0x5b, 0xa7, 0xdb, 0x74, 0xbc, 0xf2, 0x0f, 0xfd, 0xda,
     0xb9, 0xd3, 0xad, 0xa6, 0xb2, 0x3a, 0x0e, 0xe8, 0x57},
    {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x06, 0xce, 0xd0,
     0xba, 0x06, 0xc2, 0x93, 0xe6, 0x41, 0x46, 0x7b, 0x5f, 0xbd, 0xfd,
     0x5f, 0x4d, 0x53, 0x99, 0x06, 0x79, 0x60, 0xd5, 0x2f, 0x25, 0x9a,
     0x58, 0x1d, 0xd6, 0x81, 0x77, 0xb6, 0xb8, 0xe0, 0x7c, 0xdd, 0x04,
     0x4e, 0xa2, 0xbe, 0xca, 0x47, 0xbb, 0xf1, 0x94, 0x86, 0x42, 0x70,
     0x6a, 0x13, 0x52, 0x0f, 0xca, 0x34, 0x74, 0x14, 0xf4},
    {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc2, 0x4c, 0x1e,
     0x5e, 0x9a, 0x83, 0x47, 0x0f, 0xc7, 0x82, 0xed, 0x6d, 0xff, 0xf9,
     0x03, 0x6d, 0x9d, 0x27, 0xad, 0x2e, 0x57, 0xbf, 0x76, 0xb8, 0x78,
     0xe9, 0x7e, 0x11, 0xf4, 0x8a, 0x3a, 0xaf, 0x32, 0x32, 0xc3, 0x5b,
     0x64, 0x8a, 0xb6, 0x11, 0x01, 0xfe, 0x06, 0x33, 0xd4, 0x41, 0x5f,
     0xb1, 0x9e, 0x79, 0x0e, 0xc7, 0x19, 0xe9, 0xae, 0x89},
    {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xa1, 0x7b, 0x86,
     0x4b, 0x4c, 0xeb, 0x6e, 0xc3, 0xf3, 0x9a, 0x62, 0x01, 0x5a, 0xe4,
     0xdf, 0xba, 0xad, 0x5f, 0x7d, 0x27, 0x45, 0x56, 0xb1, 0x8e, 0x39,
     0x2f, 0x04, 0x03, 0x9c, 0x29, 0x5a, 0x28, 0xda, 0x2d, 0xe6, 0x2c,
     0xa4, 0x05, 0x71, 0x74, 0x35, 0x31, 0x23, 0x7f, 0xb5, 0xdd, 0x33,
     0x84, 0xfe, 0x56, 0x14, 0x1f, 0x26, 0xcb, 0x33, 0xa9},
    {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x6a, 0xda, 0xbb,
     0xa6, 0x5b, 0x22, 0x20, 0x7b, 0x5a, 0x3f, 0x12, 0xe4, 0xd1, 0x8b,
     0x9a, 0x71, 0x26, 0x90, 0x3e, 0x72, 0x5c, 0xe0, 0xc3, 0xdc, 0x3d,
     0x79, 0x63, 0x21, 0x17, 0xc9, 0xfb, 0x06, 0x74, 0xf1, 0xeb, 0xb8,
     0x1b, 0x09, 0x97, 0x26, 0x6a, 0x08, 0xb8, 0xbb, 0xb0, 0xa6, 0x24,
     0xaa, 0xe9, 0xe1, 0x22, 0x97, 0xfc, 0x59, 0x67, 0xa8},
    {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xd2, 0xc8, 0x23,
     0xe1, 0x8a, 0x36, 0x52, 0xca, 0x5c, 0xab, 0xb1, 0x45, 0xda, 0x4e,
     0xe3, 0xf7, 0xbe, 0x12, 0x6e, 0x6f, 0x72, 0x10, 0xdb, 0x91, 0x61,
     0xe1, 0xd4, 0x5c, 0xb9, 0x65, 0x59, 0x19, 0xa3, 0x38, 0x64, 0xc7,
     0xd2, 0x1e, 0x00, 0xf9, 0xb9, 0x45, 0x3a, 0x79, 0xf0, 0x77, 0x6e,
     0x8d, 0xec, 0x22, 0x5e, 0x68, 0xa8, 0xf6, 0x50, 0x12},
    {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x3e, 0x41, 0x13,
     0xbd, 0xb2, 0x2e, 0x7f, 0xfd, 0xe4, 0xd8, 0x3b, 0x2d, 0xcb, 0xeb,
     0x2a, 0x42, 0x0b, 0xc1, 0xba, 0xa4, 0xd2, 0xc0, 0x34, 0xdf, 0x3e,
     0x63, 0xac, 0xbd, 0x31, 0xc4, 0xaa, 0x42, 0xf8, 0x69, 0x28, 0x0f,
     0xec, 0xd4, 0xb2, 0x34, 0xc8, 0x48, 0xf8, 0x1e, 0x64, 0xac, 0xb8,
     0x63, 0x0e, 0x9e, 0x67, 0xfa, 0x01, 0xbe, 0xb8, 0xc2}};

static Cpa8U edwards_448_kg_v[8][64] = {
    {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x61, 0x25, 0xe8,
     0xaf, 0xbe, 0x1a, 0xfa, 0xd1, 0x6c, 0x0f, 0xe5, 0xf1, 0x3d, 0x78,
     0xd6, 0x1b, 0x06, 0xc7, 0x46, 0x9b, 0x76, 0x24, 0xf1, 0xed, 0x78,
     0x67, 0xe9, 0x80, 0x5d, 0xa7, 0x0e, 0x8a, 0x1f, 0x0e, 0xa7, 0x85,
     0x24, 0x34, 0xa1, 0x1d, 0x6a, 0xd4, 0x6a, 0x61, 0xec, 0x87, 0xe7,
     0x2c, 0xfd, 0x61, 0xb4, 0x59, 0x9b, 0x44, 0xd7, 0x5f},
    {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x94, 0x3a, 0x4c,
     0x7b, 0x62, 0x60, 0x51, 0x23, 0x9c, 0x16, 0x82, 0xcb, 0xa4, 0x8e,
     0x43, 0xb8, 0x02, 0x28, 0x74, 0x00, 0xeb, 0x01, 0xea, 0x6a, 0x86,
     0xc0, 0x98, 0x67, 0x6c, 0x0c, 0xfa, 0x2b, 0x37, 0xc0, 0x58, 0x93,
     0x5d, 0xa5, 0x34, 0xc8, 0x0a, 0xcd, 0x7e, 0x5f, 0x54, 0x31, 0xe5,
     0x6a, 0x45, 0xff, 0xcd, 0x30, 0xf4, 0x28, 0xba, 0x43},
    {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x14, 0x00, 0x47,
     0x64, 0x69, 0x00, 0x8b, 0x1d, 0x9f, 0x7e, 0x5f, 0xcb, 0xa4, 0x3e,
     0x54, 0x1a, 0x13, 0xe7, 0x1c, 0xd7, 0x61, 0xe1, 0xfc, 0x65, 0xe3,
     0xb1, 0x28, 0x93, 0x6e, 0xa3, 0x55, 0x9b, 0x10, 0xa0, 0x67, 0x4b,
     0xd8, 0x1c, 0xc0, 0xaa, 0x90, 0x6c, 0xb8, 0x10, 0x1b, 0x83, 0x9a,
     0x49, 0xf3, 0x1b, 0x5a, 0xf3, 0x78, 0x9e, 0xea, 0xdc},
    {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xf5, 0x17, 0x98,
     0xf0, 0x9e, 0x14, 0x5e, 0xd7, 0x6b, 0x72, 0x61, 0x53, 0x94, 0x9b,
     0xd1, 0x94, 0x6f, 0x6e, 0xe4, 0xb6, 0x68, 0xe9, 0xd8, 0x24, 0xdc,
     0x4a, 0xd4, 0x3f, 0xee, 0x10, 0x95, 0xcc, 0x63, 0x7c, 0x5d, 0x01,
     0xfc, 0xc5, 0x6b, 0x8d, 0x79, 0x5e, 0x6f, 0x75, 0x40, 0x77, 0x18,
     0x30, 0x1f, 0xcc, 0xf2, 0xc6, 0xa0, 0x6d, 0xa1, 0x3b},
    {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x8e, 0x57, 0xfa,
     0x5d, 0xe6, 0x4b, 0x17, 0x7b, 0xc0, 0xc6, 0x13, 0x8e, 0x93, 0xa0,
     0x1c, 0x38, 0x9f, 0x4d, 0x6b, 0x17, 0x9e, 0xc5, 0x5d, 0x5c, 0x81,
     0xb3, 0x7b, 0x4f, 0x29, 0xda, 0x61, 0x23, 0x53, 0x22, 0x3a, 0x9d,
     0x2d, 0x11, 0xa8, 0xe5, 0xbe, 0xae, 0x7b, 0x46, 0xf0, 0x29, 0x20,
     0x77, 0xa5, 0x93, 0xa4, 0x0a, 0x9b, 0x07, 0xda, 0xb3},
    {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x1f, 0x08, 0x8a,
     0xbd, 0x28, 0xcb, 0x06, 0x09, 0xb5, 0x3f, 0x3b, 0xb0, 0x9b, 0x09,
     0x1b, 0x26, 0xbd, 0xdf, 0xa1, 0xf4, 0x8b, 0x80, 0xd6, 0x39, 0x3a,
     0xc5, 0x8a, 0x6f, 0xa2, 0x83, 0x34, 0x16, 0xdd, 0xf1, 0xa9, 0x20,
     0x2c, 0x13, 0xc6, 0xb1, 0x0a, 0x56, 0xe5, 0xcf, 0x63, 0x83, 0x7f,
     0x2c, 0x80, 0xab, 0xdb, 0x8e, 0xf5, 0x05, 0x97, 0xdf},
    {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x3a, 0xed, 0xd4,
     0xf3, 0x2e, 0x86, 0x90, 0x76, 0x0e, 0xc8, 0x9e, 0x2e, 0xe2, 0x81,
     0x3d, 0x8a, 0x79, 0x65, 0xae, 0x08, 0x2b, 0x40, 0x12, 0xfe, 0x9b,
     0x6b, 0x93, 0xa9, 0x43, 0xff, 0x6b, 0x6f, 0x18, 0x25, 0x8f, 0xa0,
     0x86, 0x24, 0xef, 0xe2, 0x71, 0x41, 0xbe, 0x18, 0xe7, 0xd9, 0x5d,
     0x9f, 0x07, 0xe2, 0xcf, 0x4d, 0x01, 0x6f, 0x75, 0x79},
    {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xf4, 0x8c, 0x83,
     0x37, 0x87, 0xc3, 0x1c, 0xf1, 0xcd, 0xf8, 0xae, 0xec, 0x4e, 0xf6,
     0x56, 0x0c, 0x5c, 0x68, 0x5d, 0x1c, 0xb8, 0xff, 0x8e, 0xa0, 0x9d,
     0x79, 0x0e, 0xec, 0xe1, 0xec, 0x44, 0xad, 0x14, 0xb1, 0x8b, 0x57,
     0x58, 0xf2, 0x01, 0x08, 0xeb, 0x3f, 0xfc, 0xad, 0x9b, 0xcc, 0xdb,
     0xff, 0x94, 0xac, 0xa5, 0x70, 0x8a, 0x2e, 0x1b, 0xa8}};

/********************
 * Edwards 25519 Kg *
 ********************/

static Cpa8U edwards_25519_kg_x[32] = {
    0x21, 0x69, 0x36, 0xd3, 0xcd, 0x6e, 0x53, 0xfe, 0xc0, 0xa4, 0xe2,
    0x31, 0xfd, 0xd6, 0xdc, 0x5c, 0x69, 0x2c, 0xc7, 0x60, 0x95, 0x25,
    0xa7, 0xb2, 0xc9, 0x56, 0x2d, 0x60, 0x8f, 0x25, 0xd5, 0x1a};

static Cpa8U edwards_25519_kg_y[32] = {
    0x66, 0x66, 0x66, 0x66, 0x66, 0x66, 0x66, 0x66, 0x66, 0x66, 0x66,
    0x66, 0x66, 0x66, 0x66, 0x66, 0x66, 0x66, 0x66, 0x66, 0x66, 0x66,
    0x66, 0x66, 0x66, 0x66, 0x66, 0x66, 0x66, 0x66, 0x66, 0x58};

static Cpa8U edwards_25519_kg_k[8][32] = {
    {0x4f, 0xe9, 0x4d, 0x90, 0x06, 0xf0, 0x20, 0xa5, 0xa3, 0xc0, 0x80,
     0xd9, 0x68, 0x27, 0xff, 0xfd, 0x3c, 0x01, 0x0a, 0xc0, 0xf1, 0x2e,
     0x7a, 0x42, 0xcb, 0x33, 0x28, 0x4f, 0x86, 0x83, 0x7c, 0x30},
    {0x51, 0x2e, 0x50, 0x2e, 0xb0, 0x24, 0x9a, 0x25, 0x5e, 0x1c, 0x82,
     0x7f, 0x3b, 0x6b, 0x6c, 0x7f, 0x0a, 0x79, 0xf4, 0xca, 0x85, 0x75,
     0xa9, 0x15, 0x28, 0xd5, 0x82, 0x58, 0xd7, 0x9e, 0xbd, 0x68},
    {0x5c, 0xa9, 0x1e, 0x99, 0x81, 0xa1, 0x25, 0x13, 0x1b, 0xf5, 0xc2,
     0xc5, 0x4e, 0x7f, 0x4d, 0xba, 0x11, 0x3d, 0xc2, 0x15, 0x5b, 0xa5,
     0x23, 0x90, 0x84, 0x02, 0xd9, 0x5e, 0x75, 0x8b, 0x9a, 0x90},
    {0x7c, 0xc3, 0x94, 0xb7, 0x80, 0xe3, 0xcc, 0x85, 0x8a, 0x53, 0xc2,
     0xe4, 0x23, 0xa6, 0xdb, 0x1d, 0x78, 0x00, 0xb7, 0x4b, 0x31, 0x8e,
     0x55, 0xb2, 0x55, 0x3c, 0x13, 0x9d, 0xb9, 0xf1, 0xc9, 0xb0},
    {0x77, 0x76, 0x48, 0xb1, 0x87, 0xed, 0xb5, 0xee, 0xa5, 0x44, 0x67,
     0xd6, 0x61, 0x6e, 0xf1, 0x0c, 0x29, 0x79, 0x43, 0xbd, 0x9b, 0x22,
     0x5a, 0xf9, 0x48, 0x0a, 0x18, 0x94, 0x64, 0xbc, 0x77, 0xa8},
    {0x79, 0xf2, 0x90, 0x14, 0x7a, 0xbb, 0x17, 0x07, 0x38, 0x80, 0x1f,
     0x1e, 0x3e, 0x91, 0x8d, 0xbc, 0x01, 0x7a, 0x7e, 0xd1, 0x79, 0x85,
     0x66, 0x29, 0x73, 0x27, 0x30, 0x3e, 0x7f, 0xb9, 0x9c, 0x60},
    {0x78, 0xe6, 0x71, 0x17, 0xe6, 0xc8, 0x4b, 0x78, 0x35, 0xfc, 0xf4,
     0xff, 0x07, 0x15, 0x7f, 0x7a, 0xc2, 0x1c, 0xc0, 0xc7, 0xe5, 0xf2,
     0xbe, 0xfd, 0x85, 0x5b, 0x48, 0x76, 0x32, 0x81, 0xec, 0x10},
    {0x41, 0xb2, 0x55, 0xea, 0x7a, 0x35, 0x18, 0x8e, 0xb1, 0x72, 0x7f,
     0xaa, 0xed, 0x75, 0x6b, 0x04, 0x3e, 0x67, 0xcb, 0x25, 0xe9, 0x14,
     0xd4, 0xf8, 0xa4, 0xde, 0x16, 0x08, 0x0d, 0xd9, 0xd7, 0x70}};

static Cpa8U edwards_25519_kg_u[8][32] = {
    {0x55, 0xd0, 0xe0, 0x9a, 0x2b, 0x9d, 0x34, 0x29, 0x22, 0x97, 0xe0,
     0x8d, 0x60, 0xd0, 0xf6, 0x20, 0xc5, 0x13, 0xd4, 0x72, 0x53, 0x18,
     0x7c, 0x24, 0xb1, 0x27, 0x86, 0xbd, 0x77, 0x76, 0x45, 0xce},
    {0x74, 0xad, 0x28, 0x20, 0x5b, 0x4f, 0x38, 0x4b, 0xc0, 0x81, 0x3e,
     0x65, 0x85, 0x86, 0x4e, 0x52, 0x80, 0x85, 0xf9, 0x1f, 0xb6, 0xa5,
     0x09, 0x6f, 0x24, 0x4a, 0xe0, 0x1e, 0x57, 0xde, 0x43, 0xae},
    {0x61, 0x21, 0x3a, 0xa2, 0xdc, 0x9d, 0x68, 0x83, 0x3f, 0x65, 0xd1,
     0xb4, 0x8d, 0xcf, 0x85, 0x98, 0x18, 0x23, 0x6f, 0x17, 0x34, 0xe3,
     0xe9, 0xb9, 0x45, 0xa9, 0xff, 0x54, 0x86, 0xcd, 0xbd, 0x02},
    {0x4f, 0x30, 0xcc, 0x0d, 0xb3, 0xd9, 0x9a, 0xd8, 0xae, 0x1b, 0x3a,
     0x69, 0x69, 0x37, 0xfe, 0xdf, 0x45, 0x8c, 0x2c, 0x72, 0x32, 0xd9,
     0x12, 0xfc, 0x5f, 0x0f, 0xd8, 0x7f, 0x64, 0xb8, 0x88, 0x0c},
    {0x15, 0xda, 0xf1, 0x5d, 0x98, 0xe9, 0xa7, 0xce, 0xc9, 0xa0, 0xd9,
     0x49, 0xaa, 0x8a, 0x10, 0x10, 0x44, 0xc0, 0x71, 0xef, 0x22, 0x4e,
     0x23, 0xc3, 0x37, 0x6a, 0x6c, 0xfa, 0xf5, 0x88, 0xf8, 0xb1},
    {0x77, 0xac, 0x8b, 0xf2, 0x94, 0xc8, 0x4f, 0xf0, 0x87, 0x20, 0x60,
     0x83, 0x8b, 0x2c, 0x72, 0xc2, 0xd4, 0xcb, 0x75, 0x86, 0x4a, 0x07,
     0x0f, 0x3f, 0x3e, 0x57, 0xbb, 0x91, 0xd0, 0x85, 0x3a, 0x2c},
    {0x28, 0xea, 0x2a, 0xb7, 0xb7, 0x31, 0x69, 0xd1, 0xeb, 0x05, 0x37,
     0x70, 0xcc, 0x4c, 0x90, 0xcc, 0xfd, 0x10, 0x09, 0xbe, 0x19, 0x4b,
     0xa5, 0x2c, 0x4c, 0x3a, 0xf9, 0x95, 0xf4, 0x1e, 0xa6, 0xde},
    {0x3a, 0x9e, 0x95, 0xd1, 0x8a, 0xcb, 0xc3, 0xef, 0x5e, 0x28, 0xec,
     0x77, 0xb9, 0x63, 0xc4, 0x00, 0xdc, 0xdb, 0x07, 0xd9, 0x8b, 0x26,
     0xa2, 0x9d, 0x3f, 0xb7, 0xf4, 0xe3, 0xa0, 0xa1, 0xf0, 0x40}};

static Cpa8U edwards_25519_kg_v[8][32] = {
    {0x1a, 0x51, 0x07, 0xf7, 0x68, 0x1a, 0x02, 0xaf, 0x25, 0x23, 0xa6,
     0xda, 0xf3, 0x72, 0xe1, 0x0e, 0x3a, 0x07, 0x64, 0xc9, 0xd3, 0xfe,
     0x4b, 0xd5, 0xb7, 0x0a, 0xb1, 0x82, 0x01, 0x98, 0x5a, 0xd7},
    {0x0c, 0x66, 0xf4, 0x2a, 0xf1, 0x55, 0xcd, 0xc0, 0x8c, 0x96, 0xc4,
     0x2e, 0xcf, 0x2c, 0x98, 0x9c, 0xbc, 0x7e, 0x1b, 0x4d, 0xa7, 0x0a,
     0xb7, 0x92, 0x5a, 0x89, 0x43, 0xe8, 0xc3, 0x17, 0x40, 0x3d},
    {0x25, 0x80, 0x90, 0x48, 0x15, 0x91, 0xeb, 0x5d, 0xac, 0x03, 0x33,
     0xba, 0x13, 0xed, 0x16, 0x08, 0x58, 0xf0, 0x30, 0x02, 0xd0, 0x7e,
     0xa4, 0x8d, 0xa3, 0xa1, 0x18, 0x62, 0x8e, 0xcd, 0x51, 0xfc},
    {0x57, 0x70, 0xa5, 0x1a, 0xdc, 0xf2, 0xf5, 0xbc, 0x40, 0xd4, 0x4d,
     0x11, 0x76, 0x5d, 0x24, 0x3b, 0x5d, 0x94, 0xce, 0x63, 0x97, 0xb7,
     0x7c, 0x6c, 0x3a, 0x61, 0xf2, 0xce, 0x5b, 0x18, 0x1a, 0xe6},
    {0x27, 0x33, 0x02, 0xd9, 0x92, 0x1f, 0x00, 0x8e, 0xeb, 0x2a, 0x15,
     0x8b, 0x87, 0xdb, 0xab, 0xea, 0x53, 0x23, 0x47, 0x28, 0x31, 0xc4,
     0x5d, 0xe2, 0x86, 0x31, 0x53, 0xc4, 0x02, 0xc1, 0xda, 0xc0},
    {0x01, 0x5f, 0xfb, 0x9b, 0x89, 0x7f, 0xbc, 0x0d, 0xf5, 0x5b, 0x30,
     0xb8, 0xaa, 0xbb, 0x86, 0x48, 0x53, 0x5b, 0x76, 0xe9, 0x5b, 0x59,
     0xb1, 0x9b, 0x86, 0x4b, 0x80, 0x66, 0x07, 0xaf, 0x53, 0xe2},
    {0x3d, 0x64, 0xe3, 0x3e, 0xb8, 0x87, 0x50, 0x2b, 0x0b, 0x64, 0xe1,
     0x60, 0x52, 0x61, 0x6d, 0xe1, 0x54, 0xcc, 0x85, 0xd1, 0x33, 0x4a,
     0x44, 0xbe, 0xf2, 0xd7, 0x05, 0x05, 0xa4, 0xbf, 0xcf, 0xfb},
    {0x63, 0x8a, 0x17, 0x90, 0x8b, 0x6d, 0x35, 0xab, 0xbf, 0x02, 0x39,
     0x3b, 0x01, 0x74, 0xe7, 0x63, 0xd9, 0x1a, 0xde, 0x93, 0xf0, 0x8b,
     0x88, 0x89, 0xba, 0xaa, 0x67, 0x6e, 0xa3, 0xe3, 0xa5, 0x98}};

/******************
 * Edwards 448 Kp *
 ******************/

static Cpa8U edwards_448_kp_x[8][64] = {
    {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xb0, 0x2f, 0x7d,
     0x05, 0x80, 0xfd, 0x8e, 0x88, 0xf3, 0xfc, 0x8e, 0xcd, 0x47, 0xf4,
     0x34, 0x99, 0xb0, 0x00, 0x0f, 0xaf, 0x1e, 0x84, 0xd0, 0xc2, 0x28,
     0x37, 0x36, 0xc9, 0x91, 0xc4, 0xa6, 0x44, 0x47, 0xce, 0x4e, 0x8d,
     0x8a, 0x6c, 0x74, 0x01, 0x0b, 0xaf, 0x72, 0x6e, 0xf2, 0x00, 0x06,
     0xbc, 0xf1, 0xfa, 0x99, 0x0e, 0x7a, 0x82, 0x22, 0x87},
    {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xd2, 0xff, 0xd9,
     0xb5, 0x3d, 0x3b, 0xff, 0x14, 0x1f, 0x68, 0xcf, 0x02, 0x48, 0x02,
     0x0a, 0x14, 0xe2, 0xf1, 0xab, 0xa4, 0x33, 0x25, 0x8c, 0x33, 0xc2,
     0x90, 0x60, 0x7f, 0x78, 0xea, 0xb2, 0xc6, 0xb9, 0x69, 0xbd, 0xf2,
     0x26, 0x77, 0x5b, 0xa7, 0xdb, 0x74, 0xbc, 0xf2, 0x0f, 0xfd, 0xda,
     0xb9, 0xd3, 0xad, 0xa6, 0xb2, 0x3a, 0x0e, 0xe8, 0x57},
    {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x06, 0xce, 0xd0,
     0xba, 0x06, 0xc2, 0x93, 0xe6, 0x41, 0x46, 0x7b, 0x5f, 0xbd, 0xfd,
     0x5f, 0x4d, 0x53, 0x99, 0x06, 0x79, 0x60, 0xd5, 0x2f, 0x25, 0x9a,
     0x58, 0x1d, 0xd6, 0x81, 0x77, 0xb6, 0xb8, 0xe0, 0x7c, 0xdd, 0x04,
     0x4e, 0xa2, 0xbe, 0xca, 0x47, 0xbb, 0xf1, 0x94, 0x86, 0x42, 0x70,
     0x6a, 0x13, 0x52, 0x0f, 0xca, 0x34, 0x74, 0x14, 0xf4},
    {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc2, 0x4c, 0x1e,
     0x5e, 0x9a, 0x83, 0x47, 0x0f, 0xc7, 0x82, 0xed, 0x6d, 0xff, 0xf9,
     0x03, 0x6d, 0x9d, 0x27, 0xad, 0x2e, 0x57, 0xbf, 0x76, 0xb8, 0x78,
     0xe9, 0x7e, 0x11, 0xf4, 0x8a, 0x3a, 0xaf, 0x32, 0x32, 0xc3, 0x5b,
     0x64, 0x8a, 0xb6, 0x11, 0x01, 0xfe, 0x06, 0x33, 0xd4, 0x41, 0x5f,
     0xb1, 0x9e, 0x79, 0x0e, 0xc7, 0x19, 0xe9, 0xae, 0x89},
    {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xa1, 0x7b, 0x86,
     0x4b, 0x4c, 0xeb, 0x6e, 0xc3, 0xf3, 0x9a, 0x62, 0x01, 0x5a, 0xe4,
     0xdf, 0xba, 0xad, 0x5f, 0x7d, 0x27, 0x45, 0x56, 0xb1, 0x8e, 0x39,
     0x2f, 0x04, 0x03, 0x9c, 0x29, 0x5a, 0x28, 0xda, 0x2d, 0xe6, 0x2c,
     0xa4, 0x05, 0x71, 0x74, 0x35, 0x31, 0x23, 0x7f, 0xb5, 0xdd, 0x33,
     0x84, 0xfe, 0x56, 0x14, 0x1f, 0x26, 0xcb, 0x33, 0xa9},
    {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x6a, 0xda, 0xbb,
     0xa6, 0x5b, 0x22, 0x20, 0x7b, 0x5a, 0x3f, 0x12, 0xe4, 0xd1, 0x8b,
     0x9a, 0x71, 0x26, 0x90, 0x3e, 0x72, 0x5c, 0xe0, 0xc3, 0xdc, 0x3d,
     0x79, 0x63, 0x21, 0x17, 0xc9, 0xfb, 0x06, 0x74, 0xf1, 0xeb, 0xb8,
     0x1b, 0x09, 0x97, 0x26, 0x6a, 0x08, 0xb8, 0xbb, 0xb0, 0xa6, 0x24,
     0xaa, 0xe9, 0xe1, 0x22, 0x97, 0xfc, 0x59, 0x67, 0xa8},
    {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xd2, 0xc8, 0x23,
     0xe1, 0x8a, 0x36, 0x52, 0xca, 0x5c, 0xab, 0xb1, 0x45, 0xda, 0x4e,
     0xe3, 0xf7, 0xbe, 0x12, 0x6e, 0x6f, 0x72, 0x10, 0xdb, 0x91, 0x61,
     0xe1, 0xd4, 0x5c, 0xb9, 0x65, 0x59, 0x19, 0xa3, 0x38, 0x64, 0xc7,
     0xd2, 0x1e, 0x00, 0xf9, 0xb9, 0x45, 0x3a, 0x79, 0xf0, 0x77, 0x6e,
     0x8d, 0xec, 0x22, 0x5e, 0x68, 0xa8, 0xf6, 0x50, 0x12},
    {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x3e, 0x41, 0x13,
     0xbd, 0xb2, 0x2e, 0x7f, 0xfd, 0xe4, 0xd8, 0x3b, 0x2d, 0xcb, 0xeb,
     0x2a, 0x42, 0x0b, 0xc1, 0xba, 0xa4, 0xd2, 0xc0, 0x34, 0xdf, 0x3e,
     0x63, 0xac, 0xbd, 0x31, 0xc4, 0xaa, 0x42, 0xf8, 0x69, 0x28, 0x0f,
     0xec, 0xd4, 0xb2, 0x34, 0xc8, 0x48, 0xf8, 0x1e, 0x64, 0xac, 0xb8,
     0x63, 0x0e, 0x9e, 0x67, 0xfa, 0x01, 0xbe, 0xb8, 0xc2}};

static Cpa8U edwards_448_kp_y[8][64] = {
    {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x61, 0x25, 0xe8,
     0xaf, 0xbe, 0x1a, 0xfa, 0xd1, 0x6c, 0x0f, 0xe5, 0xf1, 0x3d, 0x78,
     0xd6, 0x1b, 0x06, 0xc7, 0x46, 0x9b, 0x76, 0x24, 0xf1, 0xed, 0x78,
     0x67, 0xe9, 0x80, 0x5d, 0xa7, 0x0e, 0x8a, 0x1f, 0x0e, 0xa7, 0x85,
     0x24, 0x34, 0xa1, 0x1d, 0x6a, 0xd4, 0x6a, 0x61, 0xec, 0x87, 0xe7,
     0x2c, 0xfd, 0x61, 0xb4, 0x59, 0x9b, 0x44, 0xd7, 0x5f},
    {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x94, 0x3a, 0x4c,
     0x7b, 0x62, 0x60, 0x51, 0x23, 0x9c, 0x16, 0x82, 0xcb, 0xa4, 0x8e,
     0x43, 0xb8, 0x02, 0x28, 0x74, 0x00, 0xeb, 0x01, 0xea, 0x6a, 0x86,
     0xc0, 0x98, 0x67, 0x6c, 0x0c, 0xfa, 0x2b, 0x37, 0xc0, 0x58, 0x93,
     0x5d, 0xa5, 0x34, 0xc8, 0x0a, 0xcd, 0x7e, 0x5f, 0x54, 0x31, 0xe5,
     0x6a, 0x45, 0xff, 0xcd, 0x30, 0xf4, 0x28, 0xba, 0x43},
    {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x14, 0x00, 0x47,
     0x64, 0x69, 0x00, 0x8b, 0x1d, 0x9f, 0x7e, 0x5f, 0xcb, 0xa4, 0x3e,
     0x54, 0x1a, 0x13, 0xe7, 0x1c, 0xd7, 0x61, 0xe1, 0xfc, 0x65, 0xe3,
     0xb1, 0x28, 0x93, 0x6e, 0xa3, 0x55, 0x9b, 0x10, 0xa0, 0x67, 0x4b,
     0xd8, 0x1c, 0xc0, 0xaa, 0x90, 0x6c, 0xb8, 0x10, 0x1b, 0x83, 0x9a,
     0x49, 0xf3, 0x1b, 0x5a, 0xf3, 0x78, 0x9e, 0xea, 0xdc},
    {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xf5, 0x17, 0x98,
     0xf0, 0x9e, 0x14, 0x5e, 0xd7, 0x6b, 0x72, 0x61, 0x53, 0x94, 0x9b,
     0xd1, 0x94, 0x6f, 0x6e, 0xe4, 0xb6, 0x68, 0xe9, 0xd8, 0x24, 0xdc,
     0x4a, 0xd4, 0x3f, 0xee, 0x10, 0x95, 0xcc, 0x63, 0x7c, 0x5d, 0x01,
     0xfc, 0xc5, 0x6b, 0x8d, 0x79, 0x5e, 0x6f, 0x75, 0x40, 0x77, 0x18,
     0x30, 0x1f, 0xcc, 0xf2, 0xc6, 0xa0, 0x6d, 0xa1, 0x3b},
    {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x8e, 0x57, 0xfa,
     0x5d, 0xe6, 0x4b, 0x17, 0x7b, 0xc0, 0xc6, 0x13, 0x8e, 0x93, 0xa0,
     0x1c, 0x38, 0x9f, 0x4d, 0x6b, 0x17, 0x9e, 0xc5, 0x5d, 0x5c, 0x81,
     0xb3, 0x7b, 0x4f, 0x29, 0xda, 0x61, 0x23, 0x53, 0x22, 0x3a, 0x9d,
     0x2d, 0x11, 0xa8, 0xe5, 0xbe, 0xae, 0x7b, 0x46, 0xf0, 0x29, 0x20,
     0x77, 0xa5, 0x93, 0xa4, 0x0a, 0x9b, 0x07, 0xda, 0xb3},
    {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x1f, 0x08, 0x8a,
     0xbd, 0x28, 0xcb, 0x06, 0x09, 0xb5, 0x3f, 0x3b, 0xb0, 0x9b, 0x09,
     0x1b, 0x26, 0xbd, 0xdf, 0xa1, 0xf4, 0x8b, 0x80, 0xd6, 0x39, 0x3a,
     0xc5, 0x8a, 0x6f, 0xa2, 0x83, 0x34, 0x16, 0xdd, 0xf1, 0xa9, 0x20,
     0x2c, 0x13, 0xc6, 0xb1, 0x0a, 0x56, 0xe5, 0xcf, 0x63, 0x83, 0x7f,
     0x2c, 0x80, 0xab, 0xdb, 0x8e, 0xf5, 0x05, 0x97, 0xdf},
    {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x3a, 0xed, 0xd4,
     0xf3, 0x2e, 0x86, 0x90, 0x76, 0x0e, 0xc8, 0x9e, 0x2e, 0xe2, 0x81,
     0x3d, 0x8a, 0x79, 0x65, 0xae, 0x08, 0x2b, 0x40, 0x12, 0xfe, 0x9b,
     0x6b, 0x93, 0xa9, 0x43, 0xff, 0x6b, 0x6f, 0x18, 0x25, 0x8f, 0xa0,
     0x86, 0x24, 0xef, 0xe2, 0x71, 0x41, 0xbe, 0x18, 0xe7, 0xd9, 0x5d,
     0x9f, 0x07, 0xe2, 0xcf, 0x4d, 0x01, 0x6f, 0x75, 0x79},
    {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xf4, 0x8c, 0x83,
     0x37, 0x87, 0xc3, 0x1c, 0xf1, 0xcd, 0xf8, 0xae, 0xec, 0x4e, 0xf6,
     0x56, 0x0c, 0x5c, 0x68, 0x5d, 0x1c, 0xb8, 0xff, 0x8e, 0xa0, 0x9d,
     0x79, 0x0e, 0xec, 0xe1, 0xec, 0x44, 0xad, 0x14, 0xb1, 0x8b, 0x57,
     0x58, 0xf2, 0x01, 0x08, 0xeb, 0x3f, 0xfc, 0xad, 0x9b, 0xcc, 0xdb,
     0xff, 0x94, 0xac, 0xa5, 0x70, 0x8a, 0x2e, 0x1b, 0xa8}};

static Cpa8U edwards_448_kp_k[8][64] = {
    {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x28, 0x03, 0x79,
     0xde, 0x16, 0x3f, 0xfe, 0x49, 0x98, 0x66, 0x1c, 0x2d, 0x46, 0xd9,
     0xdf, 0x0f, 0xff, 0xa8, 0x6e, 0x5f, 0xe7, 0x18, 0xef, 0xc5, 0xa6,
     0xca, 0x51, 0xe7, 0xe9, 0x58, 0xba, 0x02, 0x87, 0x18, 0x61, 0x23,
     0xf9, 0xf0, 0xd3, 0x95, 0x7a, 0x43, 0x5c, 0x13, 0xb3, 0x65, 0xa5,
     0xa1, 0x1f, 0x4c, 0x76, 0x3a, 0xc0, 0x71, 0x26, 0xae},
    {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x23, 0xa7, 0x37,
     0xef, 0x5f, 0x95, 0xae, 0xaf, 0xb2, 0xf3, 0x27, 0xb8, 0x45, 0x3e,
     0x18, 0xe3, 0xac, 0x96, 0x09, 0x1f, 0x3c, 0x13, 0xdd, 0x13, 0xf0,
     0x2e, 0xbd, 0xb0, 0x10, 0x92, 0x91, 0x01, 0x08, 0x49, 0x04, 0xa9,
     0x63, 0xe3, 0x8c, 0x32, 0xf7, 0x30, 0x4d, 0x3f, 0x3e, 0x33, 0x7f,
     0xb9, 0x58, 0x8b, 0xca, 0x22, 0xc6, 0x17, 0x45, 0xec},
    {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x29, 0x55, 0xc6,
     0x46, 0x35, 0x30, 0x9a, 0xf7, 0x19, 0x6a, 0x40, 0x2c, 0x8b, 0x30,
     0x0b, 0x87, 0x8d, 0x41, 0xf5, 0x33, 0xa4, 0x09, 0xb6, 0xf2, 0x7c,
     0x61, 0xe4, 0x09, 0xc4, 0x45, 0x3a, 0xf4, 0x0c, 0x60, 0x25, 0xc7,
     0x17, 0x0a, 0x02, 0x5d, 0xb1, 0xcd, 0x6f, 0x40, 0x85, 0x6d, 0xb6,
     0x92, 0x7a, 0x98, 0xc1, 0x84, 0x95, 0xef, 0x5b, 0xb5},
    {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x2e, 0x09, 0x0a,
     0x6a, 0xb7, 0xe7, 0x0b, 0x99, 0x3b, 0xce, 0xb5, 0x61, 0x52, 0x77,
     0xf9, 0x75, 0x92, 0x88, 0x19, 0x85, 0xb7, 0x4e, 0xb9, 0x4d, 0x9e,
     0x63, 0x21, 0xf6, 0xa5, 0x8e, 0x5b, 0x75, 0x94, 0xe6, 0x8a, 0x54,
     0x52, 0xe6, 0xd4, 0xa8, 0x98, 0x75, 0x16, 0x77, 0x8d, 0x2e, 0x3b,
     0x55, 0x7c, 0xfc, 0x76, 0x3d, 0x67, 0xaa, 0xe7, 0xc9},
    {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x31, 0x46, 0x4f,
     0x1a, 0x10, 0xa9, 0xa7, 0x59, 0xe7, 0x04, 0x70, 0xd1, 0x54, 0x4d,
     0x3e, 0xd7, 0xf6, 0x1c, 0xd0, 0xe8, 0x80, 0x71, 0xc6, 0x92, 0x28,
     0xc5, 0x5f, 0xc3, 0xe8, 0xe6, 0x98, 0xab, 0x71, 0x4f, 0xb8, 0xbb,
     0xe6, 0x0c, 0x72, 0xee, 0xba, 0x57, 0xee, 0x23, 0x67, 0x1d, 0xc3,
     0x7a, 0x10, 0x7d, 0x25, 0x15, 0x27, 0xe9, 0x3b, 0xa6},
    {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x24, 0x26, 0xaf,
     0x77, 0xfb, 0x26, 0xcc, 0x01, 0x77, 0x5d, 0xb9, 0x47, 0x97, 0x5a,
     0x5e, 0x5f, 0xe5, 0xf3, 0x4c, 0xbb, 0xd9, 0xd7, 0x5f, 0x4d, 0x30,
     0x0c, 0x0f, 0xc9, 0x79, 0xb4, 0x18, 0xf9, 0x4a, 0xb1, 0x86, 0x40,
     0x40, 0x59, 0x44, 0x68, 0x6d, 0x92, 0x94, 0x17, 0xf0, 0x75, 0xfa,
     0x9c, 0x2c, 0x50, 0xa3, 0x16, 0xba, 0x9f, 0xbe, 0x35},
    {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x16, 0x0e, 0x6b,
     0x23, 0x05, 0x08, 0x04, 0xb8, 0x46, 0x10, 0x81, 0x37, 0xf5, 0x8f,
     0x59, 0x6c, 0x66, 0x2c, 0x7d, 0x4d, 0x26, 0xc5, 0x93, 0x9a, 0x13,
     0x00, 0xab, 0xb5, 0x25, 0x46, 0xc7, 0xde, 0x5c, 0xd4, 0x26, 0x53,
     0x49, 0x2b, 0x80, 0x32, 0xd5, 0x25, 0x5d, 0x36, 0x2e, 0x1b, 0xdb,
     0xad, 0x3d, 0x9f, 0x15, 0x33, 0x43, 0x8e, 0x8b, 0x2b},
    {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x28, 0xcf, 0x04,
     0x7f, 0x8b, 0xde, 0xe7, 0x6a, 0x68, 0x11, 0x61, 0xb8, 0xde, 0xfa,
     0xe4, 0x9c, 0x03, 0x81, 0xad, 0x19, 0x9e, 0xd8, 0xb5, 0x0a, 0x2a,
     0x5d, 0xc3, 0x1f, 0x80, 0x90, 0x32, 0x21, 0x70, 0xae, 0x9b, 0xc3,
     0x25, 0xf9, 0x39, 0xec, 0xd0, 0x1c, 0x92, 0x94, 0x35, 0x9e, 0xd6,
     0x69, 0x3a, 0x42, 0xc7, 0xc6, 0xb8, 0xbc, 0x57, 0xb0}};

static Cpa8U edwards_448_kp_u[8][64] = {
    {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xdb, 0xb1, 0xfe,
     0xe3, 0x1d, 0x2a, 0x0e, 0x7c, 0x50, 0x08, 0x48, 0xbf, 0x8f, 0x55,
     0x03, 0xc0, 0xe7, 0x5d, 0x62, 0x85, 0xdd, 0x25, 0x83, 0xf2, 0xde,
     0x69, 0xba, 0xa5, 0x34, 0x4a, 0x79, 0x38, 0x77, 0x42, 0xab, 0x20,
     0x80, 0xf5, 0xe0, 0x48, 0xac, 0xd4, 0x59, 0x95, 0xa0, 0x54, 0x53,
     0x61, 0x7f, 0xf4, 0x33, 0x9b, 0xa4, 0xbb, 0x4f, 0x33},
    {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xd8, 0x19, 0x12,
     0x3d, 0x62, 0xae, 0xed, 0x39, 0xdd, 0x78, 0x36, 0x95, 0x59, 0x68,
     0x58, 0x75, 0xa1, 0x24, 0x77, 0x39, 0xfa, 0x4a, 0xe4, 0x28, 0x40,
     0x14, 0x65, 0xfe, 0xab, 0xb0, 0xbc, 0x11, 0x66, 0x65, 0x7f, 0x0d,
     0xa9, 0x3d, 0x64, 0xf9, 0xaf, 0x28, 0xe5, 0x6e, 0x78, 0xec, 0x67,
     0x4e, 0x62, 0xf6, 0x84, 0x98, 0x0b, 0x37, 0xab, 0x04},
    {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x42, 0x47, 0x88,
     0xb6, 0x24, 0x68, 0x71, 0x50, 0x66, 0x62, 0x8f, 0xb6, 0xea, 0x52,
     0x2b, 0xcb, 0x5f, 0xa6, 0x97, 0x0c, 0x33, 0x59, 0x28, 0x52, 0x72,
     0xec, 0xca, 0xff, 0xfd, 0x48, 0xee, 0x6c, 0xe7, 0xb4, 0x6b, 0x69,
     0x2c, 0x41, 0x23, 0xb9, 0x34, 0x33, 0xf9, 0x86, 0x36, 0x80, 0x4c,
     0x56, 0x3c, 0x1b, 0x83, 0xa6, 0xa9, 0x55, 0x16, 0x76},
    {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x6b, 0xf6, 0x31,
     0xdd, 0x2f, 0x80, 0x29, 0x8d, 0x6e, 0x7b, 0xb0, 0x7d, 0x64, 0x05,
     0x80, 0x96, 0x02, 0x5a, 0xa0, 0x68, 0x72, 0x6f, 0xf3, 0xfc, 0x14,
     0xe8, 0x47, 0x16, 0xec, 0xa7, 0x61, 0x8e, 0xd5, 0x39, 0xef, 0x28,
     0x3d, 0x01, 0x96, 0x62, 0x60, 0xc1, 0x2a, 0x25, 0xef, 0x26, 0x8b,
     0x52, 0x9e, 0xd6, 0x48, 0x3d, 0xbb, 0x00, 0x0e, 0x68},
    {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xa1, 0xae, 0xf6,
     0xf9, 0x3b, 0x35, 0x07, 0x5e, 0xf8, 0xba, 0x46, 0xd2, 0x6e, 0x93,
     0xde, 0x3a, 0xf4, 0x8e, 0x77, 0x29, 0xbd, 0x73, 0x4e, 0x33, 0x66,
     0x6f, 0x92, 0xee, 0xaf, 0x0a, 0x1a, 0x58, 0x26, 0xe9, 0x7f, 0xa5,
     0x42, 0xec, 0x46, 0x08, 0x5a, 0xf5, 0x32, 0x65, 0xc1, 0x94, 0x4a,
     0x0c, 0x44, 0x38, 0xf4, 0x41, 0xd9, 0x39, 0x0e, 0x05},
    {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x17, 0x3a, 0x32,
     0x55, 0x9b, 0x17, 0x0a, 0x5b, 0xf4, 0x04, 0x03, 0x78, 0x27, 0x3f,
     0x14, 0x03, 0x68, 0x08, 0x2b, 0xc8, 0x9d, 0xd6, 0xfa, 0x9a, 0x50,
     0x33, 0x02, 0xe5, 0x93, 0x60, 0x84, 0xff, 0x8d, 0xd2, 0x0f, 0x7c,
     0x56, 0xf9, 0xd3, 0x1f, 0xa5, 0xeb, 0x35, 0x24, 0xe2, 0xef, 0x3b,
     0xa0, 0x58, 0x12, 0x02, 0xec, 0x03, 0x6c, 0x93, 0xf5},
    {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x79, 0x5f, 0x05,
     0x57, 0x8d, 0xfd, 0x11, 0xb6, 0x70, 0xef, 0x3f, 0x8a, 0x7f, 0xa6,
     0xa8, 0xfa, 0x95, 0x0e, 0x80, 0x95, 0xaf, 0xdf, 0x0c, 0x1a, 0xfe,
     0x85, 0x33, 0x87, 0xee, 0x42, 0x96, 0x15, 0xfc, 0xeb, 0x87, 0xbf,
     0xee, 0x92, 0x6d, 0x6c, 0x5f, 0xed, 0x6a, 0x1b, 0x04, 0xb4, 0xde,
     0x3b, 0x28, 0x3e, 0x8d, 0xb0, 0xd5, 0x2f, 0x07, 0xcc},
    {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x63, 0x83, 0x13,
     0x65, 0x92, 0xe4, 0x1c, 0x0d, 0x0a, 0xd2, 0x4b, 0x5f, 0x2c, 0xd7,
     0x26, 0x83, 0xad, 0x31, 0x2b, 0xd2, 0x6b, 0x3e, 0xb7, 0xfb, 0x73,
     0x7b, 0xac, 0xcd, 0xce, 0xd1, 0x54, 0x17, 0xc6, 0x3d, 0x0e, 0x90,
     0xdb, 0xf0, 0x5f, 0xdd, 0x0b, 0x52, 0x4a, 0x49, 0xa2, 0xb1, 0x46,
     0x59, 0x3c, 0xff, 0xe9, 0x79, 0xd4, 0x14, 0xa7, 0x49}};

static Cpa8U edwards_448_kp_v[8][64] = {
    {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xa6, 0xc5, 0x7b,
     0x02, 0x90, 0x55, 0x48, 0x1a, 0x04, 0xf5, 0x37, 0x78, 0x22, 0x54,
     0xc5, 0xe9, 0x38, 0x10, 0x80, 0xca, 0x28, 0x56, 0xfe, 0x51, 0xfa,
     0x91, 0x41, 0x44, 0x26, 0xec, 0x8e, 0xa3, 0xe3, 0xc7, 0x91, 0xf7,
     0x75, 0x28, 0x7e, 0xa3, 0xd4, 0x63, 0xa7, 0x7c, 0x54, 0xd9, 0xbe,
     0xff, 0x74, 0xb1, 0xf3, 0x96, 0xe6, 0x2e, 0x2d, 0x8b},
    {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x15, 0x5c, 0xf7,
     0x2d, 0xfb, 0x25, 0x0e, 0xcf, 0x6c, 0xfd, 0x15, 0x84, 0x9a, 0x49,
     0xec, 0xa3, 0xec, 0x64, 0xb2, 0x17, 0xc4, 0x93, 0xa5, 0x44, 0x6d,
     0xf8, 0xbc, 0x43, 0xa5, 0x00, 0xc8, 0x59, 0x81, 0xf2, 0x4f, 0x4e,
     0xcb, 0xc1, 0x1b, 0xcb, 0x23, 0x36, 0xe3, 0xa8, 0xdc, 0x15, 0xef,
     0xe7, 0x67, 0xab, 0x78, 0xea, 0x5d, 0x1e, 0xe2, 0x53},
    {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x89, 0xdc, 0xad,
     0x56, 0x11, 0xd2, 0x24, 0xf4, 0x5e, 0x68, 0xaa, 0x54, 0x95, 0xdb,
     0x37, 0xfe, 0x0b, 0xc9, 0xf3, 0x9d, 0xf3, 0x4e, 0x06, 0xb4, 0x58,
     0x6b, 0x95, 0xbf, 0xae, 0x2d, 0x63, 0x80, 0x0c, 0xdf, 0xf2, 0xe0,
     0xd3, 0x6f, 0xc4, 0x1c, 0x7a, 0x6c, 0x11, 0x36, 0x94, 0x74, 0x07,
     0x17, 0x27, 0x33, 0xcf, 0x53, 0x8a, 0x31, 0x90, 0xab},
    {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x95, 0xff, 0xc6,
     0xd4, 0x40, 0xde, 0x9f, 0x99, 0xb4, 0x11, 0x20, 0xe7, 0xa3, 0x5a,
     0xdc, 0x2e, 0xfb, 0x19, 0xce, 0x23, 0x44, 0x1e, 0x06, 0x4c, 0xf7,
     0x65, 0xaa, 0xd4, 0x87, 0x6c, 0x0d, 0x17, 0x61, 0x87, 0xe8, 0x20,
     0x60, 0xdb, 0xd5, 0xc6, 0xc8, 0x2b, 0x39, 0x67, 0x6c, 0x91, 0xa4,
     0x36, 0x4d, 0x59, 0x6d, 0xeb, 0x65, 0x5f, 0x2a, 0x7e},
    {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x68, 0xc5, 0x58,
     0xb3, 0xad, 0xfb, 0x4e, 0x54, 0xa8, 0x03, 0xc8, 0x82, 0x12, 0xe9,
     0xee, 0x10, 0x4f, 0xb3, 0x1e, 0x1e, 0x4b, 0xdb, 0x3a, 0xda, 0x78,
     0xad, 0x75, 0xc7, 0x22, 0xbd, 0x1b, 0xbb, 0x15, 0xc6, 0xf3, 0x5b,
     0x0a, 0x65, 0x1c, 0xd8, 0x73, 0xc5, 0x18, 0x27, 0xdc, 0x8c, 0x8a,
     0xe7, 0x74, 0x57, 0x69, 0x14, 0xd0, 0x5e, 0x27, 0x52},
    {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xbd, 0xb9, 0xf4,
     0xe6, 0x98, 0xec, 0x6f, 0xe9, 0xf1, 0x86, 0x1e, 0xe2, 0xea, 0xd9,
     0xb3, 0x0d, 0x47, 0x5c, 0x76, 0x3a, 0x1d, 0x74, 0xda, 0xf2, 0x54,
     0xcc, 0x0b, 0x85, 0x57, 0x96, 0x19, 0x0e, 0xf2, 0x88, 0x18, 0x3f,
     0xcc, 0xad, 0xed, 0x42, 0xef, 0x52, 0x19, 0x9d, 0x39, 0x6d, 0x2e,
     0x64, 0x5d, 0x4a, 0x18, 0xda, 0x84, 0xb0, 0x85, 0x04},
    {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xcb, 0x91, 0xb0,
     0x81, 0x67, 0x00, 0x3c, 0xa9, 0xa1, 0x0b, 0x6d, 0xf2, 0x0d, 0x63,
     0x84, 0x92, 0x86, 0x53, 0x0b, 0xde, 0x07, 0x2b, 0x07, 0xe2, 0x9b,
     0x8a, 0x47, 0xd0, 0x26, 0xff, 0xa5, 0x07, 0x77, 0x85, 0x6f, 0x7a,
     0x34, 0xbb, 0xb7, 0x82, 0x7c, 0x1a, 0x42, 0x5d, 0x95, 0xb7, 0x86,
     0x8f, 0xcf, 0x09, 0x75, 0x0f, 0xc0, 0x73, 0xea, 0x61},
    {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x9a, 0x3c, 0x6b,
     0xa0, 0xbb, 0xd7, 0xc9, 0x7a, 0xc4, 0xf2, 0xfa, 0xdd, 0x38, 0x19,
     0xb1, 0x41, 0xc1, 0x08, 0x66, 0x1a, 0x40, 0xc1, 0x75, 0x6f, 0xc9,
     0x8f, 0xb4, 0xd4, 0x1a, 0x84, 0x69, 0xff, 0x4c, 0x1c, 0x32, 0xaf,
     0xb7, 0xeb, 0xdb, 0xcb, 0xdc, 0x0e, 0x1a, 0xc3, 0xd1, 0xda, 0xda,
     0x7c, 0x68, 0xce, 0x52, 0x7e, 0x78, 0xcf, 0xf1, 0x21}};

/********************
 * Edwards 25519 Kp *
 ********************/

static Cpa8U edwards_25519_kp_x[7][32] = {
    {0x55, 0xd0, 0xe0, 0x9a, 0x2b, 0x9d, 0x34, 0x29, 0x22, 0x97, 0xe0,
     0x8d, 0x60, 0xd0, 0xf6, 0x20, 0xc5, 0x13, 0xd4, 0x72, 0x53, 0x18,
     0x7c, 0x24, 0xb1, 0x27, 0x86, 0xbd, 0x77, 0x76, 0x45, 0xce},
    {0x74, 0xad, 0x28, 0x20, 0x5b, 0x4f, 0x38, 0x4b, 0xc0, 0x81, 0x3e,
     0x65, 0x85, 0x86, 0x4e, 0x52, 0x80, 0x85, 0xf9, 0x1f, 0xb6, 0xa5,
     0x09, 0x6f, 0x24, 0x4a, 0xe0, 0x1e, 0x57, 0xde, 0x43, 0xae},
    {0x61, 0x21, 0x3a, 0xa2, 0xdc, 0x9d, 0x68, 0x83, 0x3f, 0x65, 0xd1,
     0xb4, 0x8d, 0xcf, 0x85, 0x98, 0x18, 0x23, 0x6f, 0x17, 0x34, 0xe3,
     0xe9, 0xb9, 0x45, 0xa9, 0xff, 0x54, 0x86, 0xcd, 0xbd, 0x02},
    {0x4f, 0x30, 0xcc, 0x0d, 0xb3, 0xd9, 0x9a, 0xd8, 0xae, 0x1b, 0x3a,
     0x69, 0x69, 0x37, 0xfe, 0xdf, 0x45, 0x8c, 0x2c, 0x72, 0x32, 0xd9,
     0x12, 0xfc, 0x5f, 0x0f, 0xd8, 0x7f, 0x64, 0xb8, 0x88, 0x0c},
    {0x15, 0xda, 0xf1, 0x5d, 0x98, 0xe9, 0xa7, 0xce, 0xc9, 0xa0, 0xd9,
     0x49, 0xaa, 0x8a, 0x10, 0x10, 0x44, 0xc0, 0x71, 0xef, 0x22, 0x4e,
     0x23, 0xc3, 0x37, 0x6a, 0x6c, 0xfa, 0xf5, 0x88, 0xf8, 0xb1},
    {0x77, 0xac, 0x8b, 0xf2, 0x94, 0xc8, 0x4f, 0xf0, 0x87, 0x20, 0x60,
     0x83, 0x8b, 0x2c, 0x72, 0xc2, 0xd4, 0xcb, 0x75, 0x86, 0x4a, 0x07,
     0x0f, 0x3f, 0x3e, 0x57, 0xbb, 0x91, 0xd0, 0x85, 0x3a, 0x2c},
    {0x3a, 0x9e, 0x95, 0xd1, 0x8a, 0xcb, 0xc3, 0xef, 0x5e, 0x28, 0xec,
     0x77, 0xb9, 0x63, 0xc4, 0x00, 0xdc, 0xdb, 0x07, 0xd9, 0x8b, 0x26,
     0xa2, 0x9d, 0x3f, 0xb7, 0xf4, 0xe3, 0xa0, 0xa1, 0xf0, 0x40}};

static Cpa8U edwards_25519_kp_y[7][32] = {
    {0x1a, 0x51, 0x07, 0xf7, 0x68, 0x1a, 0x02, 0xaf, 0x25, 0x23, 0xa6,
     0xda, 0xf3, 0x72, 0xe1, 0x0e, 0x3a, 0x07, 0x64, 0xc9, 0xd3, 0xfe,
     0x4b, 0xd5, 0xb7, 0x0a, 0xb1, 0x82, 0x01, 0x98, 0x5a, 0xd7},
    {0x0c, 0x66, 0xf4, 0x2a, 0xf1, 0x55, 0xcd, 0xc0, 0x8c, 0x96, 0xc4,
     0x2e, 0xcf, 0x2c, 0x98, 0x9c, 0xbc, 0x7e, 0x1b, 0x4d, 0xa7, 0x0a,
     0xb7, 0x92, 0x5a, 0x89, 0x43, 0xe8, 0xc3, 0x17, 0x40, 0x3d},
    {0x25, 0x80, 0x90, 0x48, 0x15, 0x91, 0xeb, 0x5d, 0xac, 0x03, 0x33,
     0xba, 0x13, 0xed, 0x16, 0x08, 0x58, 0xf0, 0x30, 0x02, 0xd0, 0x7e,
     0xa4, 0x8d, 0xa3, 0xa1, 0x18, 0x62, 0x8e, 0xcd, 0x51, 0xfc},
    {0x57, 0x70, 0xa5, 0x1a, 0xdc, 0xf2, 0xf5, 0xbc, 0x40, 0xd4, 0x4d,
     0x11, 0x76, 0x5d, 0x24, 0x3b, 0x5d, 0x94, 0xce, 0x63, 0x97, 0xb7,
     0x7c, 0x6c, 0x3a, 0x61, 0xf2, 0xce, 0x5b, 0x18, 0x1a, 0xe6},
    {0x27, 0x33, 0x02, 0xd9, 0x92, 0x1f, 0x00, 0x8e, 0xeb, 0x2a, 0x15,
     0x8b, 0x87, 0xdb, 0xab, 0xea, 0x53, 0x23, 0x47, 0x28, 0x31, 0xc4,
     0x5d, 0xe2, 0x86, 0x31, 0x53, 0xc4, 0x02, 0xc1, 0xda, 0xc0},
    {0x01, 0x5f, 0xfb, 0x9b, 0x89, 0x7f, 0xbc, 0x0d, 0xf5, 0x5b, 0x30,
     0xb8, 0xaa, 0xbb, 0x86, 0x48, 0x53, 0x5b, 0x76, 0xe9, 0x5b, 0x59,
     0xb1, 0x9b, 0x86, 0x4b, 0x80, 0x66, 0x07, 0xaf, 0x53, 0xe2},
    {0x63, 0x8a, 0x17, 0x90, 0x8b, 0x6d, 0x35, 0xab, 0xbf, 0x02, 0x39,
     0x3b, 0x01, 0x74, 0xe7, 0x63, 0xd9, 0x1a, 0xde, 0x93, 0xf0, 0x8b,
     0x88, 0x89, 0xba, 0xaa, 0x67, 0x6e, 0xa3, 0xe3, 0xa5, 0x98}};

static Cpa8U edwards_25519_kp_k[7][32] = {
    {0x04, 0x54, 0x52, 0x2e, 0x16, 0x7e, 0x3e, 0x8a, 0x13, 0x2c, 0xec,
     0x31, 0x61, 0x25, 0xd8, 0xf8, 0x6c, 0xdf, 0x00, 0xc6, 0xe7, 0x04,
     0x05, 0x29, 0x3d, 0x19, 0x96, 0x4c, 0x8e, 0xbc, 0xea, 0x86},
    {0x03, 0x5c, 0xe3, 0x07, 0xf6, 0x52, 0x45, 0x10, 0x11, 0x0b, 0x4e,
     0xa1, 0xc8, 0xaf, 0x0e, 0x81, 0xfb, 0x70, 0x51, 0x18, 0xeb, 0xcf,
     0x88, 0x69, 0x12, 0xf8, 0xd2, 0xd8, 0x7b, 0x57, 0x76, 0xb3},
    {0x06, 0x0a, 0xb5, 0x1a, 0x60, 0xe3, 0xf1, 0xce, 0xb6, 0x05, 0x49,
     0x47, 0x9b, 0x15, 0x2a, 0xe2, 0xf4, 0xa4, 0x1d, 0x9d, 0xd8, 0xda,
     0x0f, 0x6c, 0x3e, 0xf2, 0x89, 0x2d, 0x51, 0x11, 0x8e, 0x95},
    {0x08, 0x03, 0xd9, 0x31, 0x3e, 0xa9, 0x31, 0x38, 0x4c, 0x19, 0xb0,
     0x14, 0x0e, 0x96, 0x54, 0xe5, 0xe9, 0xa8, 0xc2, 0x4d, 0xa9, 0x9f,
     0xf0, 0xb1, 0x2b, 0x78, 0x68, 0x9e, 0xb1, 0xab, 0xa2, 0x65},
    {0x03, 0x9d, 0x0a, 0x5c, 0x18, 0xd7, 0xa4, 0x52, 0xfe, 0x09, 0xad,
     0x56, 0xe6, 0x78, 0x6e, 0xf0, 0x40, 0xde, 0xce, 0x96, 0xb6, 0xea,
     0x3c, 0xd4, 0x98, 0x9b, 0x3e, 0x44, 0x01, 0x4e, 0x66, 0xcc},
    {0x00, 0xc5, 0xda, 0x12, 0xaa, 0x3b, 0x14, 0x47, 0x32, 0x3f, 0x3e,
     0x07, 0x6d, 0x63, 0xf7, 0x18, 0xa8, 0x26, 0xdb, 0x16, 0x3c, 0xfb,
     0xe7, 0x61, 0x53, 0x65, 0xb9, 0x93, 0x90, 0x13, 0xf7, 0xab},
    {0x00, 0x98, 0xdb, 0x9d, 0x1c, 0x24, 0x2d, 0xec, 0x82, 0x48, 0x46,
     0xc7, 0x8a, 0x94, 0x83, 0x61, 0x26, 0x15, 0xc9, 0x2f, 0x67, 0x4e,
     0x6f, 0x2c, 0xf3, 0x15, 0x16, 0x4b, 0x32, 0xca, 0xbb, 0xb1}};

static Cpa8U edwards_25519_kp_u[7][32] = {
    {0x7f, 0xa3, 0xaa, 0xef, 0x0a, 0x60, 0x4a, 0x9a, 0x4c, 0x2a, 0x9b,
     0x83, 0xfa, 0xe0, 0x2f, 0xb0, 0x6a, 0x02, 0xa1, 0x9c, 0x94, 0x3e,
     0x3a, 0xf8, 0x94, 0xce, 0x58, 0x56, 0xc8, 0x93, 0x6f, 0x22},
    {0x0d, 0x96, 0x8d, 0xd4, 0x6d, 0xe0, 0xff, 0x98, 0xf4, 0xa6, 0x91,
     0x6e, 0x60, 0xf8, 0x4c, 0x80, 0x68, 0x44, 0x4d, 0xbc, 0x2d, 0x93,
     0xf5, 0xd3, 0xb9, 0xcf, 0x06, 0xda, 0xde, 0x04, 0xa9, 0x94},
    {0x56, 0xdd, 0xbc, 0x2f, 0x15, 0x1c, 0xa7, 0x46, 0x19, 0xe8, 0x47,
     0x26, 0xf5, 0x22, 0xc6, 0x7c, 0x47, 0x42, 0x4b, 0x89, 0xdb, 0x11,
     0x10, 0xaa, 0xf4, 0xb1, 0xf9, 0xdf, 0xbd, 0x78, 0xd2, 0xbc},
    {0x7b, 0x9e, 0xaa, 0xf6, 0x4c, 0x3f, 0x5c, 0x59, 0x89, 0x43, 0x3a,
     0x7d, 0xcb, 0x78, 0xed, 0xca, 0xb3, 0xf2, 0x28, 0xd6, 0x3d, 0x49,
     0x93, 0x85, 0x6b, 0x8e, 0x03, 0x97, 0x19, 0xd8, 0x35, 0x77},
    {0x30, 0x60, 0xc6, 0xac, 0x57, 0xa8, 0x4c, 0xa8, 0x52, 0x46, 0xd5,
     0x78, 0x44, 0xfe, 0xe7, 0x30, 0x8f, 0xdb, 0x6f, 0xc1, 0x52, 0x15,
     0xeb, 0xe7, 0x52, 0xfb, 0xe1, 0x73, 0xa6, 0xde, 0x01, 0xf3},
    {0x45, 0xdb, 0x2d, 0x41, 0x3f, 0x62, 0x2d, 0x40, 0x40, 0x08, 0x3e,
     0xe1, 0xc7, 0xec, 0xd0, 0x04, 0xd7, 0x9f, 0x47, 0x40, 0x69, 0x1d,
     0x45, 0xd2, 0x95, 0x74, 0x17, 0xd7, 0x16, 0x5c, 0xaa, 0xef},
    {0x7b, 0xc3, 0x1f, 0x42, 0x61, 0x77, 0x64, 0xa7, 0x23, 0xef, 0x5f,
     0x65, 0x25, 0x78, 0x39, 0xda, 0x1f, 0xe7, 0x07, 0xec, 0x6d, 0xb0,
     0xed, 0x32, 0xbf, 0xd2, 0x13, 0xf2, 0x93, 0xaf, 0x8b, 0x00}};

static Cpa8U edwards_25519_kp_v[7][32] = {
    {0x0d, 0xce, 0xde, 0x2a, 0x33, 0xba, 0xb5, 0xc4, 0x4f, 0x1e, 0xe9,
     0x65, 0x99, 0x5e, 0x21, 0xdf, 0x5c, 0xfc, 0x67, 0x84, 0x2f, 0x1b,
     0x90, 0xe2, 0xc6, 0x9f, 0x9e, 0xa6, 0x59, 0x0a, 0x89, 0x9d},
    {0x3b, 0xa1, 0x6a, 0x01, 0x5e, 0x1d, 0xd4, 0x2b, 0x3d, 0x08, 0x8c,
     0x7a, 0x68, 0xc3, 0x44, 0xec, 0x47, 0xaa, 0xba, 0x46, 0x3f, 0x67,
     0xf4, 0xe9, 0x09, 0x9c, 0x63, 0x4f, 0x64, 0x78, 0x1e, 0x00},
    {0x43, 0x82, 0x9f, 0xf4, 0x28, 0x01, 0x33, 0xff, 0x74, 0xd7, 0xd6,
     0xcd, 0xa0, 0xef, 0x9a, 0x64, 0x58, 0x28, 0xee, 0xa3, 0x5f, 0xd9,
     0x83, 0x5b, 0xff, 0x8b, 0xba, 0xa0, 0x40, 0xc8, 0xe9, 0x65},
    {0x04, 0x31, 0xfa, 0xe6, 0x90, 0x32, 0xa8, 0x14, 0xed, 0xc9, 0x50,
     0x9f, 0x96, 0xdb, 0xd4, 0xa9, 0x3a, 0x72, 0x08, 0xfb, 0xab, 0x47,
     0xc0, 0x5a, 0x9e, 0x79, 0x26, 0x25, 0xbd, 0xe2, 0x60, 0x90},
    {0x69, 0x3b, 0xd6, 0x36, 0x1a, 0xad, 0x3a, 0x2d, 0x65, 0x81, 0x9f,
     0xbf, 0xf5, 0xac, 0x7e, 0x09, 0x6c, 0x0c, 0xf4, 0xeb, 0x96, 0xb9,
     0x26, 0xb5, 0x3c, 0x68, 0xdd, 0xa0, 0xba, 0xb8, 0x5b, 0xc3},
    {0x4c, 0xd0, 0x42, 0x86, 0xf7, 0x94, 0x77, 0x7d, 0x66, 0xa1, 0x75,
     0x46, 0x5d, 0xcd, 0xb6, 0x72, 0x3c, 0x33, 0xdb, 0xcd, 0xab, 0xf1,
     0x64, 0xba, 0x92, 0x10, 0x9d, 0x00, 0x56, 0xe6, 0x3a, 0xa4},
    {0x43, 0xd2, 0x04, 0xba, 0xaf, 0x6e, 0x50, 0x5f, 0xe2, 0xdb, 0xfa,
     0xd3, 0xf9, 0x5b, 0xb5, 0x39, 0x88, 0x53, 0x99, 0x51, 0xbd, 0xd1,
     0x1b, 0x22, 0x4a, 0x08, 0x91, 0x11, 0x2b, 0x7d, 0x8b, 0x94}};

/*********************
 * Montgomery 448 Kg *
 *********************/

static Cpa8U montgomery_448_kg_k[9][64] = {
    {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
     0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
     0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
     0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
     0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
     0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
    {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xf1, 0x90, 0x92,
     0x36, 0xd3, 0xaf, 0x35, 0x96, 0x00, 0xd5, 0x41, 0xe6, 0x05, 0xd8,
     0xb7, 0x98, 0x03, 0x35, 0xd1, 0x1a, 0xb4, 0xdc, 0xeb, 0xce, 0x27,
     0x30, 0x5d, 0xef, 0x75, 0x01, 0xda, 0x8c, 0xed, 0x72, 0xca, 0xfc,
     0xa3, 0x20, 0x59, 0x98, 0x41, 0xde, 0x0c, 0x70, 0x73, 0x9f, 0xd8,
     0xad, 0x72, 0xa3, 0xe7, 0x4a, 0x48, 0xb8, 0x09, 0xf2},
    {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc9, 0xa5, 0x95,
     0xcc, 0x30, 0x9e, 0x74, 0x6b, 0xdb, 0xf2, 0x19, 0x34, 0xfa, 0x61,
     0xd0, 0x4e, 0x31, 0x51, 0xa5, 0x2a, 0xf0, 0x43, 0xa5, 0xf6, 0xdb,
     0xe6, 0x28, 0x04, 0xf5, 0x4d, 0x05, 0x6c, 0xe1, 0x04, 0x29, 0x34,
     0xc0, 0xdb, 0xf5, 0x77, 0xbf, 0x89, 0x4a, 0x6c, 0xf2, 0x9e, 0xfa,
     0xa8, 0x8c, 0x9f, 0xdd, 0x17, 0x64, 0xf3, 0xc8, 0x75},
    {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xe4, 0xce, 0xe4,
     0xdc, 0x79, 0x74, 0x97, 0xf8, 0x58, 0xe5, 0xe1, 0xfd, 0x19, 0x61,
     0x0b, 0x93, 0x92, 0x98, 0x9d, 0xe9, 0x09, 0x4c, 0xd8, 0x93, 0x5a,
     0xab, 0xce, 0xdb, 0x16, 0xe7, 0x0e, 0x6d, 0xec, 0x3a, 0x1f, 0xa2,
     0x84, 0x7b, 0xe2, 0x04, 0x51, 0xfe, 0x60, 0x40, 0xd2, 0xe9, 0x97,
     0x18, 0xa6, 0x30, 0x27, 0xe9, 0xe8, 0x22, 0xb0, 0xb4},
    {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xae, 0x85, 0xfd,
     0x2f, 0x9c, 0xa9, 0x9a, 0xe1, 0xbe, 0xc0, 0xeb, 0x1d, 0x01, 0x48,
     0x11, 0x68, 0x87, 0xa4, 0xcf, 0xa5, 0x50, 0xa7, 0x60, 0x08, 0x80,
     0xaf, 0x63, 0x8e, 0x3b, 0x5d, 0x6c, 0xf1, 0x58, 0xc8, 0x21, 0x0c,
     0x88, 0xfd, 0x1e, 0x3f, 0x16, 0xa1, 0x81, 0xd2, 0xac, 0x35, 0xd3,
     0xd7, 0xe4, 0xfa, 0x79, 0x1b, 0x65, 0x39, 0x35, 0xc8},
    {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xdd, 0x41, 0x8d,
     0x9e, 0x49, 0xd2, 0x35, 0x10, 0xc1, 0x53, 0x01, 0x70, 0x9a, 0x86,
     0x93, 0x2f, 0xfb, 0xf8, 0xa5, 0xee, 0x13, 0x6e, 0x06, 0x87, 0x5f,
     0x1b, 0xea, 0x57, 0x59, 0x23, 0x73, 0x99, 0x35, 0x8e, 0x1e, 0xdc,
     0x16, 0x02, 0x0b, 0x54, 0xeb, 0x7d, 0xf6, 0x84, 0xe5, 0x5c, 0x4d,
     0x8f, 0x76, 0xf1, 0xf5, 0x7b, 0xcb, 0x0d, 0x78, 0x8e},
    {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xb7, 0x73, 0x3b,
     0xbf, 0x9d, 0xdd, 0x12, 0x40, 0x99, 0x7f, 0xab, 0x0f, 0xb1, 0x74,
     0x94, 0xf5, 0x13, 0x8a, 0x22, 0xa6, 0xd4, 0xef, 0xb6, 0x27, 0xde,
     0x27, 0x70, 0xc6, 0x3c, 0xc0, 0xe2, 0xc5, 0x4e, 0xff, 0x31, 0xcb,
     0x49, 0x7d, 0x17, 0x69, 0xa0, 0xa3, 0x4a, 0xf4, 0xae, 0x9f, 0x54,
     0x49, 0x87, 0x6a, 0xb0, 0x57, 0x00, 0xad, 0x74, 0x26},
    {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xd3, 0xef, 0xa7,
     0x50, 0xf2, 0x10, 0x33, 0x04, 0xfa, 0xc2, 0xc9, 0x3d, 0x7a, 0x9d,
     0xc7, 0x75, 0xfb, 0xe6, 0xa0, 0x50, 0x0c, 0xbd, 0x55, 0x6e, 0x14,
     0xcb, 0xed, 0xcf, 0x98, 0xa9, 0xf3, 0xf1, 0x37, 0x8b, 0x7a, 0x2f,
     0xe3, 0x52, 0x94, 0xa8, 0xb2, 0x1e, 0xfd, 0xe4, 0x3f, 0x1a, 0x69,
     0xfa, 0xb7, 0xbf, 0x81, 0x81, 0x92, 0x84, 0x66, 0xcc},
    {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x3f, 0xff, 0xff,
     0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
     0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
     0xff, 0xff, 0xff, 0x7c, 0xca, 0x23, 0xe9, 0xc4, 0x4e, 0xdb, 0x49,
     0xae, 0xd6, 0x36, 0x90, 0x21, 0x6c, 0xc2, 0x72, 0x8d, 0xc5, 0x8f,
     0x55, 0x23, 0x78, 0xc2, 0x92, 0xab, 0x58, 0x44, 0xf3}};

static Cpa8U montgomery_448_kg_u[9][64] = {
    {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x5d, 0x6d,
     0xdd, 0xdf, 0xe4, 0x40, 0xcc, 0x63, 0xd8, 0xc7, 0x35, 0xc6, 0x8f,
     0x70, 0xdd, 0xba, 0x89, 0xc2, 0x88, 0xfd, 0x03, 0xeb, 0x48, 0x7c,
     0xc3, 0x8c, 0x85, 0x46, 0xde, 0x9a, 0x9a, 0xc6, 0xa1, 0xa7, 0x78,
     0x09, 0xb5, 0x39, 0xe2, 0x9e, 0xb0, 0x62, 0x4b, 0x21, 0xe7, 0xbe,
     0x69, 0xc5, 0x3b, 0xba, 0x4d, 0xa4, 0x20, 0xb8, 0xe9},
    {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xb2, 0x61, 0xb9,
     0x75, 0x6f, 0xa4, 0x84, 0x07, 0x52, 0x25, 0x6f, 0xee, 0xa1, 0xcd,
     0xdc, 0x40, 0xc0, 0x09, 0xe5, 0xfc, 0x3f, 0xad, 0x0d, 0x45, 0xfe,
     0x44, 0xdd, 0x26, 0x4e, 0xd0, 0x8d, 0xe7, 0xdb, 0x4a, 0xdb, 0x66,
     0xef, 0x65, 0x78, 0x82, 0xa6, 0xd3, 0xcd, 0x74, 0xb7, 0x29, 0xc0,
     0x4c, 0x66, 0x9d, 0xdf, 0x77, 0x57, 0x0b, 0xec, 0x6e},
    {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x1d, 0xd7, 0x90,
     0xaa, 0x4e, 0x47, 0x75, 0x28, 0xa7, 0x8d, 0xe5, 0x23, 0x8b, 0xaa,
     0xe9, 0xd4, 0xcb, 0x51, 0x9f, 0x9d, 0x14, 0x34, 0x76, 0x2f, 0xa2,
     0x4b, 0xb1, 0x7f, 0xa3, 0x37, 0x78, 0x74, 0x0b, 0x6d, 0x8f, 0xd9,
     0xdb, 0xcc, 0x58, 0x45, 0x9a, 0xfe, 0x40, 0xf0, 0x15, 0x2f, 0x39,
     0x6a, 0xde, 0xae, 0x63, 0x6c, 0x35, 0x2f, 0x14, 0xca},
    {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xf5, 0x8e, 0x9b,
     0x0e, 0xb3, 0xac, 0xa0, 0xc8, 0xdd, 0x17, 0xa7, 0x34, 0xe6, 0xad,
     0x6a, 0x8c, 0xdb, 0xd5, 0x18, 0xbe, 0x3d, 0x30, 0x7c, 0xaf, 0x83,
     0xda, 0x76, 0x9a, 0xe1, 0x6b, 0x46, 0x51, 0xfa, 0x40, 0xdd, 0x08,
     0xe6, 0x74, 0x99, 0xfb, 0x0b, 0x4a, 0x02, 0x33, 0x9e, 0x51, 0x1e,
     0xc1, 0x28, 0x67, 0xe2, 0x16, 0xe2, 0x06, 0xe6, 0xfc},
    {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x1f, 0xf7, 0xbb,
     0x41, 0x22, 0x6a, 0xc3, 0x6b, 0x4f, 0xda, 0x8f, 0xb8, 0x59, 0x6b,
     0x04, 0xef, 0x06, 0x50, 0xa7, 0x7f, 0xf7, 0xf5, 0xad, 0xd5, 0xd0,
     0x73, 0x54, 0xd2, 0xd2, 0x09, 0xa8, 0x38, 0x30, 0xb8, 0x31, 0x4f,
     0xb4, 0x67, 0x7d, 0xcc, 0xb2, 0x61, 0xfa, 0x5b, 0xbf, 0xc8, 0x65,
     0xf4, 0xb4, 0x8f, 0x85, 0x4a, 0x87, 0x9b, 0x21, 0xaf},
    {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x11, 0x04, 0x1c,
     0x2c, 0xdd, 0xd0, 0xef, 0xd7, 0x85, 0x1a, 0x69, 0x2e, 0xd2, 0xb0,
     0x8d, 0x1f, 0xab, 0x89, 0x79, 0x69, 0x21, 0xee, 0x2d, 0xb1, 0x15,
     0xeb, 0xa7, 0x8b, 0x63, 0xb4, 0x00, 0xa1, 0xc1, 0x28, 0xe3, 0xa9,
     0x59, 0x2b, 0x3e, 0xc6, 0xc0, 0x1c, 0x04, 0xee, 0x2c, 0xe7, 0x62,
     0x4f, 0x5a, 0xf1, 0xe5, 0x6e, 0x7d, 0xba, 0x16, 0x3f},
    {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xb8, 0xde, 0x8b,
     0xc6, 0x4c, 0xd8, 0xaa, 0x53, 0xd8, 0x94, 0xd2, 0xd6, 0x23, 0x70,
     0xe4, 0xb0, 0x0a, 0x2b, 0x61, 0xb0, 0x94, 0xc4, 0x3b, 0xc1, 0xec,
     0x43, 0xd4, 0x92, 0x95, 0x97, 0xed, 0x82, 0x63, 0x45, 0xf7, 0x2c,
     0x68, 0x1d, 0x20, 0x23, 0x52, 0x41, 0x34, 0x11, 0x89, 0xc1, 0xf9,
     0xe2, 0x60, 0x3a, 0xa9, 0xcc, 0x60, 0xd3, 0x23, 0x90},
    {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0x34, 0xbf,
     0xa7, 0x1a, 0xd1, 0xba, 0xc2, 0x91, 0x94, 0x94, 0x33, 0xee, 0x6e,
     0x7e, 0xb2, 0xd8, 0x71, 0x9b, 0xcf, 0xb3, 0x5c, 0xa6, 0x2c, 0x05,
     0x16, 0xd1, 0x82, 0x3f, 0x85, 0xb0, 0x2a, 0x71, 0x99, 0x3d, 0x58,
     0x63, 0xb6, 0x79, 0xe0, 0xb5, 0x73, 0xa8, 0x42, 0x95, 0x81, 0x7b,
     0x9c, 0xbd, 0xbe, 0xc8, 0x7f, 0xa1, 0xfd, 0x5b, 0x08},
    {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xf1, 0x34, 0x86,
     0x79, 0x0f, 0x19, 0xe6, 0x93, 0x12, 0x81, 0xd5, 0x25, 0x86, 0x6d,
     0xe5, 0x4f, 0x3e, 0x80, 0xcb, 0x96, 0xe6, 0x1f, 0x4f, 0x1f, 0x48,
     0xa3, 0xc2, 0x54, 0xd8, 0x28, 0xa4, 0x0f, 0xc8, 0xbd, 0xa5, 0x08,
     0xf9, 0x79, 0x40, 0x04, 0x56, 0xbb, 0x6d, 0xfa, 0xbc, 0x06, 0x22,
     0x16, 0xc2, 0x1b, 0x79, 0x3a, 0x4a, 0x99, 0x8d, 0xa1}};

/***********************
 * Montgomery 25519 Kg *
 ***********************/

static Cpa8U montgomery_25519_kg_k[9][32] = {
    {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
     0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
     0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
    {0x97, 0x1a, 0xa0, 0x18, 0xaf, 0x39, 0x44, 0xe9, 0x32, 0x62, 0x4c,
     0x47, 0xef, 0x47, 0xb3, 0x46, 0x16, 0xd6, 0xee, 0xbb, 0x42, 0x73,
     0x83, 0xa6, 0x8f, 0xb4, 0xbd, 0x2f, 0xd7, 0x5f, 0x25, 0x03},
    {0x38, 0xab, 0xf0, 0x73, 0xea, 0x9a, 0xc5, 0x45, 0x95, 0x93, 0x60,
     0x56, 0x99, 0x64, 0x8c, 0x5f, 0x27, 0x01, 0x21, 0xb4, 0xfb, 0xbd,
     0x77, 0xd0, 0xe6, 0xfb, 0xe2, 0xb2, 0x94, 0x75, 0x16, 0xee},
    {0xfb, 0x3e, 0x6b, 0x4d, 0x41, 0x03, 0x2b, 0xd8, 0xdf, 0xed, 0x32,
     0xb5, 0xc5, 0xb3, 0x5f, 0xea, 0xd4, 0xe2, 0x27, 0x8d, 0x39, 0x58,
     0xcf, 0x6f, 0x20, 0xfb, 0x69, 0x6e, 0xfb, 0x75, 0x0d, 0xa1},
    {0xa2, 0xbe, 0xd7, 0x4e, 0xfd, 0x78, 0x1f, 0x87, 0xee, 0xe0, 0xa1,
     0x19, 0x2c, 0xa2, 0xde, 0xa9, 0x23, 0xe5, 0x5a, 0xcb, 0xcf, 0xb3,
     0xbe, 0x15, 0x49, 0x85, 0x85, 0xbf, 0x8a, 0x19, 0x04, 0x15},
    {0xb3, 0x48, 0xee, 0xe7, 0x10, 0x98, 0x0b, 0x0d, 0xbb, 0x9c, 0xe9,
     0x30, 0x9b, 0x3c, 0x44, 0xb2, 0x4f, 0x28, 0x9a, 0x7d, 0xc4, 0xa4,
     0xbd, 0xea, 0xed, 0xea, 0x09, 0x3d, 0x30, 0x8a, 0x49, 0x80},
    {0xa3, 0x65, 0xbd, 0xf7, 0xb8, 0xc0, 0xa8, 0x11, 0xdc, 0x48, 0x02,
     0x24, 0x8b, 0xc9, 0x5b, 0xab, 0x77, 0x23, 0x9f, 0xa9, 0x38, 0x77,
     0xe9, 0xdc, 0xb1, 0x36, 0xad, 0xb5, 0x9f, 0x33, 0x78, 0x76},
    {0x23, 0x92, 0x44, 0xab, 0x13, 0x58, 0xb8, 0x23, 0xad, 0x93, 0xec,
     0x93, 0xf6, 0x7a, 0x28, 0xd9, 0xb8, 0x09, 0x21, 0xd5, 0xd5, 0xb4,
     0x30, 0x9d, 0xbc, 0x59, 0x48, 0xe1, 0xe6, 0xfe, 0x87, 0x94},
    {0xe9, 0xd7, 0x8f, 0xd4, 0x15, 0xd5, 0xdd, 0xf2, 0x57, 0x50, 0x74,
     0x19, 0x9c, 0x79, 0xcf, 0xc8, 0xb8, 0x56, 0x10, 0x8b, 0xcf, 0x27,
     0xbe, 0xd1, 0xd6, 0x60, 0x8a, 0x36, 0x6d, 0xc4, 0x63, 0x34}};

static Cpa8U montgomery_25519_kg_u[9][32] = {
    {0x74, 0x3b, 0xcb, 0x58, 0x5f, 0x99, 0x90, 0xed, 0xc2, 0xcf, 0xc4,
     0xaf, 0x84, 0xf6, 0xff, 0x30, 0x07, 0x29, 0xbb, 0x5f, 0xac, 0xda,
     0x28, 0x15, 0x43, 0x62, 0xcd, 0x47, 0xa3, 0x7d, 0xe5, 0x2f},
    {0x25, 0x87, 0x67, 0x85, 0x42, 0x0b, 0x87, 0x42, 0x1c, 0xe3, 0x37,
     0x80, 0xbc, 0x9a, 0x6f, 0xfa, 0xf5, 0xef, 0x38, 0xbb, 0xea, 0x02,
     0x58, 0x43, 0xa6, 0x64, 0xb9, 0x10, 0x81, 0x38, 0xcb, 0x04},
    {0x37, 0x2e, 0xfb, 0xd7, 0x27, 0x8a, 0xb5, 0xca, 0x0a, 0x43, 0x4c,
     0x5d, 0x82, 0x20, 0x4d, 0x9b, 0x71, 0x60, 0x35, 0xe4, 0xd0, 0xea,
     0x8d, 0x30, 0xea, 0x18, 0x59, 0xc2, 0x08, 0x23, 0x8f, 0x34},
    {0x03, 0x2b, 0xca, 0xa1, 0x53, 0xd3, 0x55, 0x02, 0x16, 0x54, 0xa9,
     0xe7, 0xc1, 0x79, 0x33, 0x65, 0xde, 0x0e, 0xf9, 0x5d, 0x51, 0x2d,
     0x6c, 0x44, 0x35, 0xbb, 0x39, 0xc9, 0x2e, 0xdb, 0xb8, 0x1a},
    {0x18, 0xf6, 0x26, 0x0c, 0x18, 0x89, 0x9e, 0x57, 0x5f, 0xb9, 0x25,
     0x35, 0xd0, 0x96, 0xd2, 0x3b, 0x87, 0x4f, 0x2e, 0x3e, 0xbb, 0xc5,
     0x3c, 0x61, 0x48, 0x92, 0xe3, 0xa3, 0x10, 0x7f, 0xf1, 0x6d},
    {0x7e, 0x26, 0xe4, 0xe8, 0x22, 0x47, 0xc0, 0xad, 0x79, 0x59, 0xf7,
     0xf3, 0x93, 0x57, 0xd8, 0x54, 0xce, 0x50, 0xfa, 0x88, 0xab, 0xda,
     0xd3, 0xc4, 0x78, 0x9b, 0xdd, 0x75, 0xca, 0xd9, 0x3a, 0x74},
    {0x25, 0x09, 0xa1, 0x5a, 0x65, 0x1f, 0xd2, 0x4f, 0x01, 0xc1, 0xbf,
     0x4b, 0xf5, 0xda, 0x59, 0xe1, 0x51, 0xeb, 0xb3, 0x77, 0x1e, 0x98,
     0x7a, 0xbb, 0xa6, 0x11, 0x1d, 0x0c, 0xe0, 0xc5, 0x80, 0x17},
    {0x20, 0x0c, 0x77, 0xda, 0x28, 0xe9, 0xc7, 0xc6, 0x4a, 0x24, 0x31,
     0x4b, 0xa5, 0x4b, 0xff, 0x10, 0x5b, 0x74, 0x49, 0x94, 0x2a, 0x7e,
     0x5d, 0xd8, 0x99, 0xaf, 0xe0, 0x24, 0x1e, 0x07, 0x72, 0x2b},
    {0x2f, 0x18, 0x45, 0x26, 0x45, 0x3c, 0xfd, 0xcb, 0x1f, 0x16, 0xbf,
     0x58, 0x29, 0x9b, 0xac, 0x8a, 0x2b, 0x84, 0xce, 0x7a, 0xeb, 0x9c,
     0x4b, 0x61, 0xf7, 0x94, 0x43, 0xf6, 0xe8, 0xb8, 0x90, 0x21}};

/*********************
 * Montgomery 448 Kp *
 *********************/

static Cpa8U montgomery_448_kp_x[10][64] = {
    {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x86, 0xA0, 0xF8,
     0x4E, 0xFB, 0xA7, 0xA7, 0x8A, 0xA1, 0xAD, 0x94, 0xDB, 0x29, 0x54,
     0xFA, 0x83, 0x25, 0xDA, 0xC6, 0x19, 0x8C, 0xC3, 0xBD, 0xDD, 0x31,
     0xC0, 0x4D, 0x81, 0xF9, 0x08, 0x0F, 0x02, 0x7F, 0x43, 0x07, 0xBD,
     0x4C, 0x33, 0x88, 0xAD, 0x8A, 0x3F, 0x26, 0xD5, 0xF2, 0x6C, 0x5F,
     0xDA, 0xBF, 0x87, 0x34, 0xFA, 0x40, 0xE6, 0xFC, 0x06},
    {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xDB, 0x57, 0xD1,
     0xE8, 0x1C, 0xE7, 0xBD, 0xDF, 0x1C, 0xB9, 0x78, 0x8A, 0xE2, 0x05,
     0xE2, 0x2F, 0xE5, 0xBE, 0x70, 0x35, 0x4D, 0x6C, 0xE5, 0x94, 0x58,
     0x01, 0x5D, 0x16, 0x1B, 0x20, 0xB6, 0xE9, 0xA1, 0xE9, 0xF8, 0x52,
     0xBB, 0x5D, 0xFB, 0xA8, 0xC1, 0xD4, 0x55, 0x9E, 0x7D, 0x0B, 0x5B,
     0x30, 0xD3, 0x56, 0xCD, 0x93, 0xF9, 0xC2, 0xBC, 0x0F},
    {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0e, 0x32, 0x37,
     0x76, 0xab, 0x6d, 0x78, 0x01, 0x51, 0x3a, 0xb2, 0xa4, 0x6e, 0x4f,
     0x5a, 0xe7, 0x80, 0x78, 0x9f, 0x78, 0x4d, 0x7b, 0x29, 0x90, 0x25,
     0x40, 0x61, 0x4c, 0x32, 0x91, 0x07, 0xbf, 0x2d, 0x3a, 0xea, 0x86,
     0x6b, 0x30, 0x84, 0x99, 0x22, 0xff, 0x3f, 0x39, 0x3f, 0xab, 0x17,
     0x1b, 0x5c, 0x8f, 0x96, 0xa1, 0x72, 0x0f, 0x69, 0x10},
    {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x27, 0x94, 0x4e,
     0x0c, 0x8c, 0x3f, 0xf8, 0x20, 0x4c, 0xdd, 0x97, 0x02, 0x98, 0x86,
     0x9d, 0xdf, 0x5a, 0x14, 0x27, 0xd9, 0xce, 0x12, 0xf2, 0xbc, 0xaf,
     0xde, 0x9f, 0xb9, 0x0b, 0xba, 0xbd, 0x6e, 0xdf, 0x61, 0xb0, 0x16,
     0x76, 0xc2, 0x17, 0x54, 0x6d, 0xf5, 0x1f, 0x79, 0xf8, 0xfa, 0x7a,
     0x89, 0x78, 0x12, 0x26, 0xa8, 0x5b, 0xdb, 0xcc, 0xac},
    {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x47, 0xfc, 0xac,
     0xf6, 0xdd, 0xd2, 0x22, 0x1e, 0xad, 0xaf, 0x7f, 0x57, 0x04, 0xec,
     0x35, 0x95, 0xe3, 0x84, 0xc1, 0x35, 0x74, 0xec, 0xc9, 0xb2, 0x6c,
     0xa8, 0x2b, 0xae, 0x85, 0x2e, 0x24, 0x72, 0x78, 0x1b, 0x52, 0xf3,
     0x9a, 0x8c, 0x8b, 0x1a, 0xa0, 0x50, 0x3b, 0x23, 0x8c, 0xd0, 0xc6,
     0xb7, 0xcc, 0x19, 0x76, 0x65, 0xa4, 0x4a, 0x0b, 0xd6},
    {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc8, 0x03, 0x11,
     0xfe, 0xe1, 0x27, 0x3e, 0x18, 0x0c, 0xe2, 0x01, 0x96, 0x8f, 0x8f,
     0xd7, 0x85, 0x6e, 0x50, 0x9f, 0xb9, 0xd4, 0x42, 0xd2, 0x27, 0x11,
     0x49, 0xd4, 0xec, 0xa0, 0x86, 0xd9, 0x6d, 0x98, 0xec, 0x74, 0x23,
     0x47, 0x22, 0xdc, 0x79, 0x01, 0x4f, 0x8b, 0x7a, 0x61, 0xc5, 0xf4,
     0xb9, 0x55, 0xa7, 0x05, 0x4d, 0x1b, 0x22, 0x00, 0x37},
    {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x28, 0xae, 0xb9,
     0x22, 0xe2, 0xae, 0xc4, 0xcc, 0x39, 0xc8, 0x9f, 0x30, 0xb2, 0xa5,
     0x96, 0x44, 0xa7, 0x98, 0x27, 0x98, 0x33, 0x8c, 0x47, 0xbc, 0x2a,
     0xe1, 0xe7, 0x59, 0x91, 0xb9, 0xe8, 0x56, 0xd0, 0xb7, 0x97, 0xae,
     0x3b, 0xab, 0xaa, 0x81, 0x4d, 0xd0, 0x18, 0xe2, 0x20, 0xbf, 0xfb,
     0xd2, 0x94, 0xa7, 0x27, 0xa4, 0x2a, 0x97, 0x86, 0xad},
    {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xe4, 0xfa, 0x66,
     0xee, 0xe2, 0xa7, 0x83, 0x69, 0xb6, 0x64, 0x17, 0x7a, 0x6e, 0xbd,
     0xe4, 0xe2, 0xcf, 0x9b, 0x02, 0xa0, 0x78, 0x94, 0x25, 0x91, 0x48,
     0x06, 0x28, 0xac, 0x5d, 0x9c, 0x06, 0xa7, 0xa4, 0x2e, 0xd7, 0x28,
     0x41, 0x6d, 0xba, 0x57, 0x9c, 0xf6, 0xdd, 0x18, 0x94, 0x67, 0x40,
     0xe6, 0x34, 0x89, 0xd2, 0xa9, 0x13, 0x22, 0x43, 0x2b},
    {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xe5, 0x8c, 0x5a,
     0xb4, 0x60, 0x5a, 0xc7, 0x0f, 0xd3, 0x06, 0x59, 0x14, 0xac, 0x92,
     0xdd, 0x30, 0x03, 0x69, 0x68, 0xf3, 0x79, 0x3e, 0x7f, 0x54, 0x3b,
     0xc3, 0xcb, 0x87, 0xfd, 0xbb, 0xe4, 0xe9, 0xc5, 0x54, 0x7f, 0x66,
     0x47, 0x55, 0x19, 0x35, 0x0d, 0x78, 0x12, 0x15, 0x1e, 0x91, 0x97,
     0xd3, 0x27, 0x89, 0x14, 0xdb, 0xca, 0xfc, 0xef, 0xce},
    {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x5e, 0xb6, 0x69,
     0x24, 0xb3, 0x01, 0xe7, 0xea, 0x41, 0x1e, 0x2a, 0x97, 0xab, 0x04,
     0x2b, 0x33, 0x01, 0x37, 0xf3, 0x09, 0xb7, 0x02, 0xaf, 0xcc, 0x5a,
     0x57, 0xd9, 0x23, 0x81, 0x24, 0x99, 0xe9, 0x46, 0xe7, 0x4f, 0x49,
     0x16, 0xdc, 0xa5, 0x9c, 0x56, 0x27, 0x93, 0xd5, 0xe4, 0x4b, 0x41,
     0xf3, 0x9e, 0x5b, 0x58, 0xc8, 0xa3, 0xc6, 0x70, 0xd6}};

static Cpa8U montgomery_448_kp_k[10][64] = {
    {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xD3, 0x0A, 0x60,
     0x1C, 0x4F, 0x9A, 0x25, 0x29, 0x4B, 0xF5, 0x68, 0xA3, 0xEB, 0x43,
     0x49, 0xF4, 0xBF, 0x8F, 0xD7, 0xCD, 0xF8, 0x24, 0x4C, 0x98, 0x9C,
     0x77, 0x0A, 0x70, 0x21, 0xE1, 0xAA, 0xD1, 0xD0, 0x04, 0x51, 0x04,
     0xEF, 0xAC, 0x82, 0x88, 0xD2, 0x34, 0x9A, 0xA1, 0xFE, 0x66, 0x52,
     0x49, 0x88, 0x8E, 0xEC, 0xF9, 0xDD, 0x2F, 0x26, 0x3D},
    {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x5F, 0x09, 0xE3,
     0x5B, 0x8D, 0x2C, 0xDD, 0x82, 0x12, 0x37, 0xB4, 0xA5, 0xE0, 0x44,
     0x5C, 0x31, 0xD3, 0x46, 0x5F, 0xE2, 0x06, 0x48, 0x3E, 0x7C, 0xD7,
     0x5D, 0x34, 0x38, 0xC5, 0xF8, 0x21, 0xB0, 0x1C, 0x46, 0x0D, 0x8E,
     0x90, 0x00, 0xF6, 0xFE, 0xE8, 0x9D, 0x2F, 0xA4, 0xDC, 0x5D, 0x66,
     0x52, 0x93, 0x39, 0xB8, 0x28, 0x44, 0x49, 0x3D, 0x20},
    {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x27, 0x94, 0x4e,
     0x0c, 0x8c, 0x3f, 0xf8, 0x20, 0x4c, 0xdd, 0x97, 0x02, 0x98, 0x86,
     0x9d, 0xdf, 0x5a, 0x14, 0x27, 0xd9, 0xce, 0x12, 0xf2, 0xbc, 0xaf,
     0xde, 0x9f, 0xb9, 0x0b, 0xba, 0xbd, 0x6e, 0xdf, 0x61, 0xb0, 0x16,
     0x76, 0xc2, 0x17, 0x54, 0x6d, 0xf5, 0x1f, 0x79, 0xf8, 0xfa, 0x7a,
     0x89, 0x78, 0x12, 0x26, 0xa8, 0x5b, 0xdb, 0xcc, 0xac},
    {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x47, 0xfc, 0xac,
     0xf6, 0xdd, 0xd2, 0x22, 0x1e, 0xad, 0xaf, 0x7f, 0x57, 0x04, 0xec,
     0x35, 0x95, 0xe3, 0x84, 0xc1, 0x35, 0x74, 0xec, 0xc9, 0xb2, 0x6c,
     0xa8, 0x2b, 0xae, 0x85, 0x2e, 0x24, 0x72, 0x78, 0x1b, 0x52, 0xf3,
     0x9a, 0x8c, 0x8b, 0x1a, 0xa0, 0x50, 0x3b, 0x23, 0x8c, 0xd0, 0xc6,
     0xb7, 0xcc, 0x19, 0x76, 0x65, 0xa4, 0x4a, 0x0b, 0xd6},
    {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc8, 0x03, 0x11,
     0xfe, 0xe1, 0x27, 0x3e, 0x18, 0x0c, 0xe2, 0x01, 0x96, 0x8f, 0x8f,
     0xd7, 0x85, 0x6e, 0x50, 0x9f, 0xb9, 0xd4, 0x42, 0xd2, 0x27, 0x11,
     0x49, 0xd4, 0xec, 0xa0, 0x86, 0xd9, 0x6d, 0x98, 0xec, 0x74, 0x23,
     0x47, 0x22, 0xdc, 0x79, 0x01, 0x4f, 0x8b, 0x7a, 0x61, 0xc5, 0xf4,
     0xb9, 0x55, 0xa7, 0x05, 0x4d, 0x1b, 0x22, 0x00, 0x37},
    {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x28, 0xae, 0xb9,
     0x22, 0xe2, 0xae, 0xc4, 0xcc, 0x39, 0xc8, 0x9f, 0x30, 0xb2, 0xa5,
     0x96, 0x44, 0xa7, 0x98, 0x27, 0x98, 0x33, 0x8c, 0x47, 0xbc, 0x2a,
     0xe1, 0xe7, 0x59, 0x91, 0xb9, 0xe8, 0x56, 0xd0, 0xb7, 0x97, 0xae,
     0x3b, 0xab, 0xaa, 0x81, 0x4d, 0xd0, 0x18, 0xe2, 0x20, 0xbf, 0xfb,
     0xd2, 0x94, 0xa7, 0x27, 0xa4, 0x2a, 0x97, 0x86, 0xad},
    {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xe4, 0xfa, 0x66,
     0xee, 0xe2, 0xa7, 0x83, 0x69, 0xb6, 0x64, 0x17, 0x7a, 0x6e, 0xbd,
     0xe4, 0xe2, 0xcf, 0x9b, 0x02, 0xa0, 0x78, 0x94, 0x25, 0x91, 0x48,
     0x06, 0x28, 0xac, 0x5d, 0x9c, 0x06, 0xa7, 0xa4, 0x2e, 0xd7, 0x28,
     0x41, 0x6d, 0xba, 0x57, 0x9c, 0xf6, 0xdd, 0x18, 0x94, 0x67, 0x40,
     0xe6, 0x34, 0x89, 0xd2, 0xa9, 0x13, 0x22, 0x43, 0x2b},
    {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xe5, 0x8c, 0x5a,
     0xb4, 0x60, 0x5a, 0xc7, 0x0f, 0xd3, 0x06, 0x59, 0x14, 0xac, 0x92,
     0xdd, 0x30, 0x03, 0x69, 0x68, 0xf3, 0x79, 0x3e, 0x7f, 0x54, 0x3b,
     0xc3, 0xcb, 0x87, 0xfd, 0xbb, 0xe4, 0xe9, 0xc5, 0x54, 0x7f, 0x66,
     0x47, 0x55, 0x19, 0x35, 0x0d, 0x78, 0x12, 0x15, 0x1e, 0x91, 0x97,
     0xd3, 0x27, 0x89, 0x14, 0xdb, 0xca, 0xfc, 0xef, 0xce},
    {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x5e, 0xb6, 0x69,
     0x24, 0xb3, 0x01, 0xe7, 0xea, 0x41, 0x1e, 0x2a, 0x97, 0xab, 0x04,
     0x2b, 0x33, 0x01, 0x37, 0xf3, 0x09, 0xb7, 0x02, 0xaf, 0xcc, 0x5a,
     0x57, 0xd9, 0x23, 0x81, 0x24, 0x99, 0xe9, 0x46, 0xe7, 0x4f, 0x49,
     0x16, 0xdc, 0xa5, 0x9c, 0x56, 0x27, 0x93, 0xd5, 0xe4, 0x4b, 0x41,
     0xf3, 0x9e, 0x5b, 0x58, 0xc8, 0xa3, 0xc6, 0x70, 0xd6},
    {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x84, 0x29, 0x2b,
     0x24, 0xa4, 0xb6, 0xe1, 0x0f, 0x08, 0x77, 0x00, 0xbe, 0x8f, 0x76,
     0x38, 0x75, 0xea, 0xf6, 0x53, 0xfa, 0x4a, 0x3d, 0x1e, 0x2d, 0x4f,
     0xef, 0x3d, 0xde, 0x1a, 0xc4, 0x9e, 0x7a, 0x2c, 0x0c, 0x99, 0x1a,
     0xc9, 0xcb, 0x88, 0x91, 0xb3, 0x97, 0x26, 0x88, 0xc6, 0xab, 0xf5,
     0xf7, 0xb8, 0x34, 0xe4, 0xb3, 0x6f, 0x32, 0xf1, 0x11}};

static Cpa8U montgomery_448_kp_u[10][64] = {
    {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x6F, 0x6B, 0xD9,
     0x3D, 0xF7, 0x82, 0x62, 0x76, 0x21, 0x1E, 0x11, 0x61, 0x39, 0x22,
     0x98, 0x9D, 0x77, 0xB0, 0x01, 0x6A, 0xC6, 0x5F, 0x44, 0xEB, 0xAD,
     0xBA, 0x4F, 0xE1, 0x9F, 0x23, 0x5F, 0x6D, 0x54, 0xD7, 0x12, 0x24,
     0x0A, 0xB5, 0x79, 0xDF, 0xFB, 0x6A, 0x5E, 0xD8, 0xB1, 0x1D, 0xDA,
     0x97, 0x66, 0xDC, 0x60, 0x5A, 0xF9, 0x4F, 0x3E, 0xCE},
    {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x9D, 0x17, 0x7C,
     0xDA, 0x99, 0x4E, 0x51, 0x54, 0xC9, 0xC1, 0x75, 0xC5, 0x33, 0x36,
     0xE6, 0x77, 0x20, 0xD6, 0x21, 0x43, 0xF3, 0x0D, 0x70, 0xA5, 0xE3,
     0x3E, 0x1B, 0xAD, 0xA7, 0xC4, 0x63, 0xFE, 0x30, 0x1E, 0x8E, 0x56,
     0x13, 0xAC, 0x47, 0x70, 0xF3, 0x9F, 0x6A, 0xDB, 0xB2, 0x63, 0x2F,
     0x2F, 0x7A, 0xFF, 0x39, 0x62, 0x57, 0x02, 0x4A, 0x88},
    {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x47, 0xfc, 0xac,
     0xf6, 0xdd, 0xd2, 0x22, 0x1e, 0xad, 0xaf, 0x7f, 0x57, 0x04, 0xec,
     0x35, 0x95, 0xe3, 0x84, 0xc1, 0x35, 0x74, 0xec, 0xc9, 0xb2, 0x6c,
     0xa8, 0x2b, 0xae, 0x85, 0x2e, 0x24, 0x72, 0x78, 0x1b, 0x52, 0xf3,
     0x9a, 0x8c, 0x8b, 0x1a, 0xa0, 0x50, 0x3b, 0x23, 0x8c, 0xd0, 0xc6,
     0xb7, 0xcc, 0x19, 0x76, 0x65, 0xa4, 0x4a, 0x0b, 0xd6},
    {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc8, 0x03, 0x11,
     0xfe, 0xe1, 0x27, 0x3e, 0x18, 0x0c, 0xe2, 0x01, 0x96, 0x8f, 0x8f,
     0xd7, 0x85, 0x6e, 0x50, 0x9f, 0xb9, 0xd4, 0x42, 0xd2, 0x27, 0x11,
     0x49, 0xd4, 0xec, 0xa0, 0x86, 0xd9, 0x6d, 0x98, 0xec, 0x74, 0x23,
     0x47, 0x22, 0xdc, 0x79, 0x01, 0x4f, 0x8b, 0x7a, 0x61, 0xc5, 0xf4,
     0xb9, 0x55, 0xa7, 0x05, 0x4d, 0x1b, 0x22, 0x00, 0x37},
    {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x28, 0xae, 0xb9,
     0x22, 0xe2, 0xae, 0xc4, 0xcc, 0x39, 0xc8, 0x9f, 0x30, 0xb2, 0xa5,
     0x96, 0x44, 0xa7, 0x98, 0x27, 0x98, 0x33, 0x8c, 0x47, 0xbc, 0x2a,
     0xe1, 0xe7, 0x59, 0x91, 0xb9, 0xe8, 0x56, 0xd0, 0xb7, 0x97, 0xae,
     0x3b, 0xab, 0xaa, 0x81, 0x4d, 0xd0, 0x18, 0xe2, 0x20, 0xbf, 0xfb,
     0xd2, 0x94, 0xa7, 0x27, 0xa4, 0x2a, 0x97, 0x86, 0xad},
    {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xe4, 0xfa, 0x66,
     0xee, 0xe2, 0xa7, 0x83, 0x69, 0xb6, 0x64, 0x17, 0x7a, 0x6e, 0xbd,
     0xe4, 0xe2, 0xcf, 0x9b, 0x02, 0xa0, 0x78, 0x94, 0x25, 0x91, 0x48,
     0x06, 0x28, 0xac, 0x5d, 0x9c, 0x06, 0xa7, 0xa4, 0x2e, 0xd7, 0x28,
     0x41, 0x6d, 0xba, 0x57, 0x9c, 0xf6, 0xdd, 0x18, 0x94, 0x67, 0x40,
     0xe6, 0x34, 0x89, 0xd2, 0xa9, 0x13, 0x22, 0x43, 0x2b},
    {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xe5, 0x8c, 0x5a,
     0xb4, 0x60, 0x5a, 0xc7, 0x0f, 0xd3, 0x06, 0x59, 0x14, 0xac, 0x92,
     0xdd, 0x30, 0x03, 0x69, 0x68, 0xf3, 0x79, 0x3e, 0x7f, 0x54, 0x3b,
     0xc3, 0xcb, 0x87, 0xfd, 0xbb, 0xe4, 0xe9, 0xc5, 0x54, 0x7f, 0x66,
     0x47, 0x55, 0x19, 0x35, 0x0d, 0x78, 0x12, 0x15, 0x1e, 0x91, 0x97,
     0xd3, 0x27, 0x89, 0x14, 0xdb, 0xca, 0xfc, 0xef, 0xce},
    {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x5e, 0xb6, 0x69,
     0x24, 0xb3, 0x01, 0xe7, 0xea, 0x41, 0x1e, 0x2a, 0x97, 0xab, 0x04,
     0x2b, 0x33, 0x01, 0x37, 0xf3, 0x09, 0xb7, 0x02, 0xaf, 0xcc, 0x5a,
     0x57, 0xd9, 0x23, 0x81, 0x24, 0x99, 0xe9, 0x46, 0xe7, 0x4f, 0x49,
     0x16, 0xdc, 0xa5, 0x9c, 0x56, 0x27, 0x93, 0xd5, 0xe4, 0x4b, 0x41,
     0xf3, 0x9e, 0x5b, 0x58, 0xc8, 0xa3, 0xc6, 0x70, 0xd6},
    {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x84, 0x29, 0x2b,
     0x24, 0xa4, 0xb6, 0xe1, 0x0f, 0x08, 0x77, 0x00, 0xbe, 0x8f, 0x76,
     0x38, 0x75, 0xea, 0xf6, 0x53, 0xfa, 0x4a, 0x3d, 0x1e, 0x2d, 0x4f,
     0xef, 0x3d, 0xde, 0x1a, 0xc4, 0x9e, 0x7a, 0x2c, 0x0c, 0x99, 0x1a,
     0xc9, 0xcb, 0x88, 0x91, 0xb3, 0x97, 0x26, 0x88, 0xc6, 0xab, 0xf5,
     0xf7, 0xb8, 0x34, 0xe4, 0xb3, 0x6f, 0x32, 0xf1, 0x11},
    {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x38, 0x9f, 0x21,
     0x6e, 0xa2, 0xfa, 0x54, 0xef, 0x53, 0xc3, 0x3e, 0xea, 0xce, 0x9f,
     0xb7, 0xe2, 0x86, 0x82, 0xb8, 0x2d, 0x20, 0x87, 0xd0, 0x10, 0xcf,
     0x67, 0x6c, 0xaf, 0xd4, 0x09, 0x5e, 0x97, 0x7b, 0xc1, 0xd5, 0xbd,
     0xeb, 0xe3, 0x4c, 0x27, 0x67, 0xc4, 0x26, 0x88, 0x28, 0x00, 0x5b,
     0x1e, 0xaf, 0x9d, 0x5b, 0xd5, 0x49, 0x47, 0x3b, 0xaa}};

/***********************
 * Montgomery 25519 Kp *
 ***********************/

static Cpa8U montgomery_25519_kp_x[9][32] = {
    {0x4C, 0x1C, 0xAB, 0xD0, 0xA6, 0x03, 0xA9, 0x10, 0x3b, 0x35, 0xB3,
     0x26, 0xEC, 0x24, 0x66, 0x72, 0x7C, 0x5F, 0xB1, 0x24, 0xA4, 0xC1,
     0x94, 0x35, 0xDB, 0x30, 0x30, 0x58, 0x67, 0x68, 0xDB, 0xE6},
    {0x93, 0xA4, 0x15, 0xC7, 0x49, 0xD5, 0x4C, 0xFC, 0x3E, 0x3C, 0xC0,
     0x6F, 0x10, 0xE7, 0xDB, 0x31, 0x2C, 0xAE, 0x38, 0x05, 0x9D, 0x95,
     0xB7, 0xF4, 0xD3, 0x11, 0x68, 0x78, 0x12, 0x0F, 0x21, 0xE5},
    {0x62, 0x9d, 0x47, 0x54, 0x8a, 0x31, 0x49, 0xad, 0xa9, 0x6f, 0x66,
     0x8e, 0x67, 0x49, 0x3a, 0x2a, 0x9c, 0x1e, 0xe4, 0xf5, 0x90, 0xb1,
     0x18, 0xf9, 0xdf, 0x98, 0x01, 0x38, 0x4b, 0x47, 0xe4, 0x05},
    {0x36, 0xd3, 0xe1, 0xe0, 0xde, 0x0c, 0x2e, 0x89, 0xaf, 0x97, 0x57,
     0x05, 0x78, 0xaf, 0x81, 0xdb, 0x9e, 0xa9, 0x9b, 0xc4, 0xb1, 0xb1,
     0x86, 0xfe, 0xa5, 0x7e, 0x16, 0x85, 0x86, 0xe4, 0x45, 0x88},
    {0x1a, 0x0f, 0x77, 0x9c, 0xf1, 0xc8, 0x4a, 0x4b, 0x46, 0x23, 0x01,
     0x28, 0x04, 0x5f, 0x5e, 0x41, 0x84, 0x29, 0xa7, 0x76, 0xba, 0xd8,
     0xfb, 0x84, 0x37, 0x36, 0x4e, 0xc5, 0x18, 0x5d, 0x60, 0xe8},
    {0x5e, 0xf9, 0x5f, 0xe7, 0xe6, 0x18, 0xe3, 0x2a, 0x12, 0x4b, 0xb2,
     0x4f, 0xee, 0x74, 0xfb, 0x20, 0xf5, 0xbe, 0x3f, 0xbe, 0x98, 0xcf,
     0xf2, 0x05, 0x39, 0xff, 0x46, 0xd1, 0x11, 0x68, 0xf3, 0x1d},
    {0x3c, 0xb6, 0xce, 0xa6, 0xd3, 0x0d, 0xa9, 0x02, 0x0d, 0x03, 0x31,
     0x16, 0x12, 0xfe, 0x79, 0x19, 0x07, 0xa2, 0x2e, 0x5e, 0x7b, 0xd1,
     0x46, 0x07, 0x08, 0xe7, 0xd5, 0x70, 0x72, 0x89, 0xd4, 0x49},
    {0x27, 0xd2, 0xc1, 0x87, 0xf7, 0xf8, 0x5e, 0x92, 0xde, 0x57, 0x10,
     0x51, 0x4c, 0xf9, 0x3b, 0xdb, 0x3e, 0xdd, 0xf9, 0x42, 0x2f, 0x4c,
     0xda, 0x54, 0x16, 0xd3, 0x49, 0x86, 0x89, 0xb4, 0x1d, 0x68},
    {0x42, 0x68, 0xe3, 0x90, 0xc0, 0xeb, 0x16, 0x5c, 0x07, 0x0b, 0x64,
     0x78, 0x8e, 0x8c, 0x31, 0xc3, 0x64, 0x0f, 0xf3, 0xbf, 0xac, 0x96,
     0xa6, 0xf0, 0x31, 0xe1, 0xc8, 0x7c, 0x9e, 0x19, 0xb5, 0xae}};

static Cpa8U montgomery_25519_kp_k[9][32] = {
    {0xC4, 0x9A, 0x44, 0xBA, 0x44, 0x22, 0x6A, 0x50, 0x18, 0x5A, 0xFC,
     0xC1, 0x0A, 0x4C, 0x14, 0x62, 0xDD, 0x5E, 0x46, 0x82, 0x4B, 0x15,
     0x16, 0x3B, 0x9D, 0x7C, 0x52, 0xF0, 0x6B, 0xE3, 0x46, 0xA5},
    {0x0D, 0xBA, 0x18, 0x79, 0x9E, 0x16, 0xA4, 0x2C, 0xD4, 0x01, 0xEA,
     0xE0, 0x21, 0x64, 0x1B, 0xC1, 0xF5, 0x6A, 0x7D, 0x95, 0x91, 0x26,
     0xD2, 0x5A, 0x3C, 0x67, 0xB4, 0xD1, 0xD4, 0xE9, 0x66, 0x4B},
    {0x36, 0xd3, 0xe1, 0xe0, 0xde, 0x0c, 0x2e, 0x89, 0xaf, 0x97, 0x57,
     0x05, 0x78, 0xaf, 0x81, 0xdb, 0x9e, 0xa9, 0x9b, 0xc4, 0xb1, 0xb1,
     0x86, 0xfe, 0xa5, 0x7e, 0x16, 0x85, 0x86, 0xe4, 0x45, 0x88},
    {0x1a, 0x0f, 0x77, 0x9c, 0xf1, 0xc8, 0x4a, 0x4b, 0x46, 0x23, 0x01,
     0x28, 0x04, 0x5f, 0x5e, 0x41, 0x84, 0x29, 0xa7, 0x76, 0xba, 0xd8,
     0xfb, 0x84, 0x37, 0x36, 0x4e, 0xc5, 0x18, 0x5d, 0x60, 0xe8},
    {0x5e, 0xf9, 0x5f, 0xe7, 0xe6, 0x18, 0xe3, 0x2a, 0x12, 0x4b, 0xb2,
     0x4f, 0xee, 0x74, 0xfb, 0x20, 0xf5, 0xbe, 0x3f, 0xbe, 0x98, 0xcf,
     0xf2, 0x05, 0x39, 0xff, 0x46, 0xd1, 0x11, 0x68, 0xf3, 0x1d},
    {0x3c, 0xb6, 0xce, 0xa6, 0xd3, 0x0d, 0xa9, 0x02, 0x0d, 0x03, 0x31,
     0x16, 0x12, 0xfe, 0x79, 0x19, 0x07, 0xa2, 0x2e, 0x5e, 0x7b, 0xd1,
     0x46, 0x07, 0x08, 0xe7, 0xd5, 0x70, 0x72, 0x89, 0xd4, 0x49},
    {0x27, 0xd2, 0xc1, 0x87, 0xf7, 0xf8, 0x5e, 0x92, 0xde, 0x57, 0x10,
     0x51, 0x4c, 0xf9, 0x3b, 0xdb, 0x3e, 0xdd, 0xf9, 0x42, 0x2f, 0x4c,
     0xda, 0x54, 0x16, 0xd3, 0x49, 0x86, 0x89, 0xb4, 0x1d, 0x68},
    {0x42, 0x68, 0xe3, 0x90, 0xc0, 0xeb, 0x16, 0x5c, 0x07, 0x0b, 0x64,
     0x78, 0x8e, 0x8c, 0x31, 0xc3, 0x64, 0x0f, 0xf3, 0xbf, 0xac, 0x96,
     0xa6, 0xf0, 0x31, 0xe1, 0xc8, 0x7c, 0x9e, 0x19, 0xb5, 0xae},
    {0x3b, 0x00, 0x4a, 0xc9, 0xfe, 0x39, 0x73, 0x85, 0x15, 0x1c, 0xa3,
     0xf6, 0x7e, 0x5e, 0x1e, 0x89, 0x69, 0xee, 0xc7, 0x81, 0x66, 0x77,
     0xf6, 0x64, 0xe2, 0xa9, 0xb6, 0x60, 0x60, 0xeb, 0x5b, 0x0a}};

static Cpa8U montgomery_25519_kp_u[9][32] = {
    {0x52, 0x85, 0xA2, 0x77, 0x55, 0x07, 0xB4, 0x54, 0xF7, 0x71, 0x1C,
     0x49, 0x03, 0xCF, 0xEC, 0x32, 0x4F, 0x08, 0x8D, 0xF2, 0x4D, 0xEA,
     0x94, 0x8E, 0x90, 0xC6, 0xE9, 0x9D, 0x37, 0x55, 0xDA, 0xC3},
    {0x57, 0x79, 0xAC, 0x7A, 0x64, 0xF7, 0xF8, 0xE6, 0x52, 0xA1, 0x9F,
     0x79, 0x68, 0x5A, 0x59, 0x8B, 0xF8, 0x73, 0xB8, 0xB4, 0x5C, 0xE4,
     0xAD, 0x7A, 0x7D, 0x90, 0xE8, 0x76, 0x94, 0xDE, 0xCB, 0x95},
    {0x1a, 0x0f, 0x77, 0x9c, 0xf1, 0xc8, 0x4a, 0x4b, 0x46, 0x23, 0x01,
     0x28, 0x04, 0x5f, 0x5e, 0x41, 0x84, 0x29, 0xa7, 0x76, 0xba, 0xd8,
     0xfb, 0x84, 0x37, 0x36, 0x4e, 0xc5, 0x18, 0x5d, 0x60, 0xe8},
    {0x5e, 0xf9, 0x5f, 0xe7, 0xe6, 0x18, 0xe3, 0x2a, 0x12, 0x4b, 0xb2,
     0x4f, 0xee, 0x74, 0xfb, 0x20, 0xf5, 0xbe, 0x3f, 0xbe, 0x98, 0xcf,
     0xf2, 0x05, 0x39, 0xff, 0x46, 0xd1, 0x11, 0x68, 0xf3, 0x1d},
    {0x3c, 0xb6, 0xce, 0xa6, 0xd3, 0x0d, 0xa9, 0x02, 0x0d, 0x03, 0x31,
     0x16, 0x12, 0xfe, 0x79, 0x19, 0x07, 0xa2, 0x2e, 0x5e, 0x7b, 0xd1,
     0x46, 0x07, 0x08, 0xe7, 0xd5, 0x70, 0x72, 0x89, 0xd4, 0x49},
    {0x27, 0xd2, 0xc1, 0x87, 0xf7, 0xf8, 0x5e, 0x92, 0xde, 0x57, 0x10,
     0x51, 0x4c, 0xf9, 0x3b, 0xdb, 0x3e, 0xdd, 0xf9, 0x42, 0x2f, 0x4c,
     0xda, 0x54, 0x16, 0xd3, 0x49, 0x86, 0x89, 0xb4, 0x1d, 0x68},
    {0x42, 0x68, 0xe3, 0x90, 0xc0, 0xeb, 0x16, 0x5c, 0x07, 0x0b, 0x64,
     0x78, 0x8e, 0x8c, 0x31, 0xc3, 0x64, 0x0f, 0xf3, 0xbf, 0xac, 0x96,
     0xa6, 0xf0, 0x31, 0xe1, 0xc8, 0x7c, 0x9e, 0x19, 0xb5, 0xae},
    {0x3b, 0x00, 0x4a, 0xc9, 0xfe, 0x39, 0x73, 0x85, 0x15, 0x1c, 0xa3,
     0xf6, 0x7e, 0x5e, 0x1e, 0x89, 0x69, 0xee, 0xc7, 0x81, 0x66, 0x77,
     0xf6, 0x64, 0xe2, 0xa9, 0xb6, 0x60, 0x60, 0xeb, 0x5b, 0x0a},
    {0x18, 0x9a, 0xf6, 0x04, 0x4d, 0xc5, 0x6c, 0xda, 0x6d, 0x08, 0x86,
     0xa2, 0xc6, 0x71, 0xdd, 0x05, 0xd4, 0x69, 0x80, 0xe1, 0xe1, 0x88,
     0x7d, 0x39, 0x4d, 0x57, 0xe7, 0x30, 0xf2, 0x7d, 0xba, 0xe4}};

/*****************************************************************************
 * @ingroup ecMontEdwdsThreads
 *
 * @description
 *      This function selects vectors used in ECED performance test
 *
 *****************************************************************************/
CpaStatus getEcMontEdwdsTestVectors(CpaBoolean generator,
                                    CpaCyEcMontEdwdsCurveType curveType,
                                    Cpa32U vector,
                                    sample_ec_montedwds_vectors_t *testVectors)
{
    CpaStatus status = CPA_STATUS_SUCCESS;

    if (generator == CPA_TRUE)
    { /* KG */

        switch (curveType)
        {
            case CPA_CY_EC_MONTEDWDS_CURVE25519_TYPE: /* Montgomery 25519 Kg */
                testVectors->x = NULL;
                testVectors->xSize = 0;
                testVectors->y = NULL;
                testVectors->ySize = 0;
                testVectors->k = montgomery_25519_kg_k[vector];
                testVectors->kSize = sizeof(montgomery_25519_kg_k[vector]);
                testVectors->u = montgomery_25519_kg_u[vector];
                testVectors->uSize = sizeof(montgomery_25519_kg_u[vector]);
                testVectors->v = NULL;
                testVectors->vSize = 0;
                testVectors->vectorsNum = sizeof(montgomery_25519_kg_k) /
                                          sizeof(montgomery_25519_kg_k[0]);
                break;

            case CPA_CY_EC_MONTEDWDS_ED25519_TYPE: /* Edwards 25519 Kg */
                testVectors->x = edwards_25519_kg_x;
                testVectors->xSize = sizeof(edwards_25519_kg_x);
                testVectors->y = edwards_25519_kg_y;
                testVectors->ySize = sizeof(edwards_25519_kg_y);
                testVectors->k = edwards_25519_kg_k[vector];
                testVectors->kSize = sizeof(edwards_25519_kg_k[vector]);
                testVectors->u = edwards_25519_kg_u[vector];
                testVectors->uSize = sizeof(edwards_25519_kg_u[vector]);
                testVectors->v = edwards_25519_kg_v[vector];
                testVectors->vSize = sizeof(edwards_25519_kg_v[vector]);
                testVectors->vectorsNum =
                    sizeof(edwards_25519_kg_k) / sizeof(edwards_25519_kg_k[0]);
                break;

            case CPA_CY_EC_MONTEDWDS_CURVE448_TYPE: /* Montgomery 448 Kg */
                testVectors->x = NULL;
                testVectors->xSize = 0;
                testVectors->y = NULL;
                testVectors->ySize = 0;
                testVectors->k = montgomery_448_kg_k[vector];
                testVectors->kSize = sizeof(montgomery_448_kg_k[vector]);
                testVectors->u = montgomery_448_kg_u[vector];
                testVectors->uSize = sizeof(montgomery_448_kg_u[vector]);
                testVectors->v = NULL;
                testVectors->vSize = 0;
                testVectors->vectorsNum = sizeof(montgomery_448_kg_k) /
                                          sizeof(montgomery_448_kg_k[0]);
                break;

            case CPA_CY_EC_MONTEDWDS_ED448_TYPE: /* Edwards 448 Kg */
                testVectors->x = edwards_448_kg_x;
                testVectors->xSize = sizeof(edwards_448_kg_x);
                testVectors->y = edwards_448_kg_y;
                testVectors->ySize = sizeof(edwards_448_kg_y);
                testVectors->k = edwards_448_kg_k[vector];
                testVectors->kSize = sizeof(edwards_448_kg_k[vector]);
                testVectors->u = edwards_448_kg_u[vector];
                testVectors->uSize = sizeof(edwards_448_kg_u[vector]);
                testVectors->v = edwards_448_kg_v[vector];
                testVectors->vSize = sizeof(edwards_448_kg_v[vector]);
                testVectors->vectorsNum =
                    sizeof(edwards_448_kg_k) / sizeof(edwards_448_kg_k[0]);
                break;

            default:
                PRINT("Unknown ECED curveType %d\n", curveType);
                status = CPA_STATUS_FAIL;
                break;
        }
    }
    else
    { /* KP */

        switch (curveType)
        {
            case CPA_CY_EC_MONTEDWDS_CURVE25519_TYPE: /* Montgomery 25519 Kp */
                testVectors->x = montgomery_25519_kp_x[vector];
                testVectors->xSize = sizeof(montgomery_25519_kp_x[vector]);
                testVectors->y = NULL;
                testVectors->ySize = 0;
                testVectors->k = montgomery_25519_kp_k[vector];
                testVectors->kSize = sizeof(montgomery_25519_kp_k[vector]);
                testVectors->u = montgomery_25519_kp_u[vector];
                testVectors->uSize = sizeof(montgomery_25519_kp_u[vector]);
                testVectors->v = NULL;
                testVectors->vSize = 0;
                testVectors->vectorsNum = sizeof(montgomery_25519_kp_k) /
                                          sizeof(montgomery_25519_kp_k[0]);
                break;

            case CPA_CY_EC_MONTEDWDS_ED25519_TYPE: /* Edwards 25519 Kp */
                testVectors->x = edwards_25519_kp_x[vector];
                testVectors->xSize = sizeof(edwards_25519_kp_x[vector]);
                testVectors->y = edwards_25519_kp_y[vector];
                testVectors->ySize = sizeof(edwards_25519_kp_y[vector]);
                testVectors->k = edwards_25519_kp_k[vector];
                testVectors->kSize = sizeof(edwards_25519_kp_k[vector]);
                testVectors->u = edwards_25519_kp_u[vector];
                testVectors->uSize = sizeof(edwards_25519_kp_u[vector]);
                testVectors->v = edwards_25519_kp_v[vector];
                testVectors->vSize = sizeof(edwards_25519_kp_v[vector]);
                testVectors->vectorsNum =
                    sizeof(edwards_25519_kp_k) / sizeof(edwards_25519_kp_k[0]);
                break;

            case CPA_CY_EC_MONTEDWDS_CURVE448_TYPE: /* Montgomery 448 Kp */
                testVectors->x = montgomery_448_kp_x[vector];
                testVectors->xSize = sizeof(montgomery_448_kp_x[vector]);
                testVectors->y = NULL;
                testVectors->ySize = 0;
                testVectors->k = montgomery_448_kp_k[vector];
                testVectors->kSize = sizeof(montgomery_448_kp_k[vector]);
                testVectors->u = montgomery_448_kp_u[vector];
                testVectors->uSize = sizeof(montgomery_448_kp_u[vector]);
                testVectors->v = NULL;
                testVectors->vSize = 0;
                testVectors->vectorsNum = sizeof(montgomery_448_kp_k) /
                                          sizeof(montgomery_448_kp_k[0]);
                break;

            case CPA_CY_EC_MONTEDWDS_ED448_TYPE: /* Edwards 448 Kp */
                testVectors->x = edwards_448_kp_x[vector];
                testVectors->xSize = sizeof(edwards_448_kp_x[vector]);
                testVectors->y = edwards_448_kp_y[vector];
                testVectors->ySize = sizeof(edwards_448_kp_y[vector]);
                testVectors->k = edwards_448_kp_k[vector];
                testVectors->kSize = sizeof(edwards_448_kp_k[vector]);
                testVectors->u = edwards_448_kp_u[vector];
                testVectors->uSize = sizeof(edwards_448_kp_u[vector]);
                testVectors->v = edwards_448_kp_v[vector];
                testVectors->vSize = sizeof(edwards_448_kp_v[vector]);
                testVectors->vectorsNum =
                    sizeof(edwards_448_kp_k) / sizeof(edwards_448_kp_k[0]);
                break;

            default:
                PRINT("Unknown ECED curveType %d\n", curveType);
                status = CPA_STATUS_FAIL;
                break;
        }
    }
    return status;
}
EXPORT_SYMBOL(getEcMontEdwdsTestVectors);

#endif /* CY_API_VERSION_AT_LEAST(2, 3) */
