package tempentries

import (
	"testing"

	"github.com/stretchr/testify/require"
	"github.com/ubuntu/authd/internal/testutils/golden"
	"github.com/ubuntu/authd/internal/users/idgenerator"
	"github.com/ubuntu/authd/internal/users/types"
)

func TestRegisterUser(t *testing.T) {
	t.Parallel()

	uidToGenerate := uint32(12345)
	userName := "authd-temp-users-test"

	tests := map[string]struct {
		userName                string
		uidsToGenerate          []uint32
		userAlreadyRemoved      bool
		replacesPreAuthUser     bool
		preAuthUIDAlreadyExists bool

		wantErr bool
	}{
		"Successfully_register_a_new_user": {},
		"Successfully_register_a_user_if_the_first_generated_UID_is_already_in_use": {
			uidsToGenerate: []uint32{0, uidToGenerate}, // UID 0 (root) always exists
		},
		"Successfully_register_a_user_if_the_pre-auth_user_already_exists": {
			replacesPreAuthUser: true,
			uidsToGenerate:      []uint32{}, // No UID generation needed
		},

		"Error_when_name_is_already_in_use": {userName: "root", wantErr: true},
		"Error_when_pre-auth_user_already_exists_and_name_is_not_unique": {
			replacesPreAuthUser: true,
			userName:            "root",
			wantErr:             true,
		},
		"Error_when_pre-auth_UID_is_not_unique": {
			replacesPreAuthUser:     true,
			preAuthUIDAlreadyExists: true,
			wantErr:                 true,
		},
	}

	for name, tc := range tests {
		t.Run(name, func(t *testing.T) {
			t.Parallel()

			if tc.userName == "" {
				tc.userName = userName
			}

			if tc.uidsToGenerate == nil {
				tc.uidsToGenerate = []uint32{uidToGenerate}
			}

			idGeneratorMock := &idgenerator.IDGeneratorMock{UIDsToGenerate: tc.uidsToGenerate}
			records := NewTemporaryRecords(idGeneratorMock)

			var preAuthUID uint32
			if tc.replacesPreAuthUser {
				preAuthUID = uidToGenerate
				if tc.preAuthUIDAlreadyExists {
					preAuthUID = 0 // UID 0 (root) always exists
				}
				_, _, err := records.preAuthUserRecords.addPreAuthUser(preAuthUID, tc.userName)
				require.NoError(t, err, "addPreAuthUser should not return an error, but did")
			}

			uid, cleanup, err := records.RegisterUser(tc.userName)
			if tc.wantErr {
				require.Error(t, err, "RegisterUser should return an error, but did not")
				return
			}
			require.NoError(t, err, "RegisterUser should not return an error, but did")
			require.Equal(t, uidToGenerate, uid, "UID should be the one generated by the IDGenerator")

			if tc.replacesPreAuthUser {
				// Check that the pre-auth user was removed
				_, err = records.preAuthUserRecords.userByID(preAuthUID)
				require.Error(t, err, "userByID should return an error, but did not")
			}

			user, err := records.UserByID(uid)
			require.NoError(t, err, "UserByID should not return an error, but did")
			checkUser(t, user)

			cleanup()
		})
	}
}

func TestUserByIDAndName(t *testing.T) {
	t.Parallel()

	userName := "authd-temp-users-test"
	uidToGenerate := uint32(12345)

	tests := map[string]struct {
		registerUser       bool
		userAlreadyRemoved bool
		byName             bool

		wantErr bool
	}{
		"Successfully_get_a_user_by_ID":   {registerUser: true},
		"Successfully_get_a_user_by_name": {registerUser: true, byName: true},

		"Error_when_user_is_not_registered_-_UserByID":   {wantErr: true},
		"Error_when_user_is_not_registered_-_UserByName": {byName: true, wantErr: true},
		"Error_when_user_is_already_removed_-_UserByID": {
			registerUser:       true,
			userAlreadyRemoved: true,
			wantErr:            true,
		},
		"Error_when_user_is_already_removed_-_UserByName": {
			registerUser:       true,
			userAlreadyRemoved: true,
			byName:             true,
			wantErr:            true,
		},
	}

	for name, tc := range tests {
		t.Run(name, func(t *testing.T) {
			t.Parallel()

			idGeneratorMock := &idgenerator.IDGeneratorMock{UIDsToGenerate: []uint32{uidToGenerate}}
			records := NewTemporaryRecords(idGeneratorMock)
			userRecords := records.temporaryUserRecords

			if tc.registerUser {
				uid, cleanup, err := records.RegisterUser(userName)
				require.NoError(t, err, "RegisterUser should not return an error, but did")
				require.Equal(t, uidToGenerate, uid, "UID should be the one generated by the IDGenerator")

				if tc.userAlreadyRemoved {
					cleanup()
				} else {
					defer cleanup()
				}
			}

			var user types.UserEntry
			var err error
			if tc.byName {
				user, err = userRecords.userByName(userName)
			} else {
				user, err = userRecords.userByID(uidToGenerate)
			}

			if tc.wantErr {
				require.Error(t, err, "UserByID should return an error, but did not")
				return
			}
			require.NoError(t, err, "UserByID should not return an error, but did")
			checkUser(t, user)
		})
	}
}

func checkUser(t *testing.T, user types.UserEntry) {
	t.Helper()

	// The gecos field is randomly generated, so unset it before comparing the user with the golden file.
	require.NotEmpty(t, user.Gecos, "Gecos should not be empty")
	user.Gecos = ""

	golden.CheckOrUpdateYAML(t, user)
}
