//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      GUI/Support/Util/ActionFactory.h
//! @brief     Defines class ActionFactory
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2021
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

#ifndef BORNAGAIN_GUI_SUPPORT_UTIL_ACTIONFACTORY_H
#define BORNAGAIN_GUI_SUPPORT_UTIL_ACTIONFACTORY_H

#include <functional>

class QAction;
class QObject;
class QString;
class QWidget;

//! Factory to create commonly used actions
class ActionFactory {
public:
    // delete copy/move constructor/assignment:
    ActionFactory(const ActionFactory&) = delete;
    ActionFactory(ActionFactory&&) = delete;
    ActionFactory& operator=(const ActionFactory&) = delete;
    ActionFactory& operator=(ActionFactory&&) = delete;

    //! Create "remove" action.
    //!
    //! The "what" text will be used in the tooltip, appended to "Remove ".
    //! If a slot is given, it will be connected to the "triggered" signal.
    static QAction* createRemoveAction(QObject* parent, const QString& what,
                                       std::function<void()> slot = nullptr);
    //! Create "duplicate" action.
    //!
    //! The "what" text will be used in the tooltip, appended to "Duplicate ".
    //! If a slot is given, it will be connected to the "triggered" signal.
    static QAction* createDuplicateAction(QObject* parent, const QString& what,
                                          std::function<void()> slot = nullptr);

    //! Create "show in Realspace" action.
    //!
    //! The "what" text will be used in the tooltip.
    //! If a slot is given, it will be connected to the "triggered" signal.
    static QAction* createShowInRealspaceAction(QObject* parent, const QString& what,
                                                std::function<void()> slot = nullptr);

    //! Create "toggle properties panel" action.
    //!
    //! If toggledWidget is given, it will be shown/hidden when the actions changes its state.
    static QAction* createTogglePropertiesPanelAction(QObject* parent,
                                                      QWidget* toggledWidget = nullptr);

private:
    ActionFactory() = default;
};

#endif // BORNAGAIN_GUI_SUPPORT_UTIL_ACTIONFACTORY_H
