//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      GUI/View/Mask/LineViews.cpp
//! @brief     Implements classes VerticalLineView and HorizontalLineView
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2018
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

#include "GUI/View/Mask/LineViews.h"
#include "GUI/Model/Data/MaskItems.h"
#include "GUI/View/Mask/ColorMapSceneAdaptor.h"
#include <QCursor>
#include <QPainter>
#include <QStyleOptionGraphicsItem>

namespace {

const double mask_width = 8.0;
const double mask_visible_width = 3.0;

} // namespace

VerticalLineView::VerticalLineView(VerticalLineItem* item)
    : IShape2DView(item)
    , m_item(item)
{
    setFlag(QGraphicsItem::ItemIsSelectable);
    setFlag(QGraphicsItem::ItemIsMovable);
    setFlag(QGraphicsItem::ItemSendsGeometryChanges);
    setCursor(Qt::SizeHorCursor);
}

QPainterPath VerticalLineView::shape() const
{
    QPainterPath p;
    p.addRect(QRectF(-mask_visible_width / 2., 0.0, mask_visible_width, m_bounding_rect.height()));
    return p;
}

MaskItemObject* VerticalLineView::parameterizedItem() const
{
    return m_item;
}

void VerticalLineView::onChangedX()
{
    setBlockOnProperty(true);
    m_item->setPosX(fromSceneX(this->x()));
    setBlockOnProperty(false);
}

void VerticalLineView::onPropertyChange()
{
    setX(toSceneX(m_item->posX()));
}

void VerticalLineView::update_view()
{
    QRectF plot_scene_rectangle = m_adaptor->viewportRectangle();

    setX(toSceneX(m_item->posX()));
    setY(plot_scene_rectangle.top());

    m_bounding_rect = QRectF(-mask_width / 2., 0.0, mask_width, plot_scene_rectangle.height());
    update();
}

void VerticalLineView::paint(QPainter* painter, const QStyleOptionGraphicsItem*, QWidget*)
{
    bool mask_value = static_cast<VerticalLineItem*>(m_item)->maskValue();
    painter->setBrush(MaskEditorHelper::getMaskBrush(mask_value));
    painter->setPen(MaskEditorHelper::getMaskPen(mask_value));
    painter->drawRect(
        QRectF(-mask_visible_width / 2., 0.0, mask_visible_width, m_bounding_rect.height()));

    if (isSelected()) {
        QPen pen;
        pen.setStyle(Qt::DashLine);
        painter->setPen(pen);
        painter->setBrush(Qt::NoBrush);
        painter->drawRect(
            QRectF(-mask_visible_width / 2., 0.0, mask_visible_width, m_bounding_rect.height()));
    }
}

//! Allows item movement along x, prevent movement along y
QVariant VerticalLineView::itemChange(QGraphicsItem::GraphicsItemChange change,
                                      const QVariant& value)
{
    if (isSelected() && change == ItemPositionChange && scene()) {
        QPointF newPos = value.toPointF();
        newPos.setY(y());
        return newPos;
    }
    return QGraphicsItem::itemChange(change, value);
}

// --------------------------------------------------------------------------------------------- //

HorizontalLineView::HorizontalLineView(HorizontalLineItem* item)
    : IShape2DView(item)
    , m_item(item)
{
    setFlag(QGraphicsItem::ItemIsSelectable);
    setFlag(QGraphicsItem::ItemIsMovable);
    setFlag(QGraphicsItem::ItemSendsGeometryChanges);
    setCursor(Qt::SizeVerCursor);
}

QPainterPath HorizontalLineView::shape() const
{
    QPainterPath path;
    path.addRect(
        QRectF(0.0, -mask_visible_width / 2., m_bounding_rect.width(), mask_visible_width));
    return path;
}

MaskItemObject* HorizontalLineView::parameterizedItem() const
{
    return m_item;
}

void HorizontalLineView::onChangedY()
{
    setBlockOnProperty(true);
    m_item->setPosY(fromSceneY(this->y()));
    setBlockOnProperty(false);
}

void HorizontalLineView::onPropertyChange()
{
    setY(toSceneY(m_item->posY()));
}

void HorizontalLineView::update_view()
{
    QRectF plot_scene_rectangle = m_adaptor->viewportRectangle();

    setX(plot_scene_rectangle.left());
    setY(toSceneY(m_item->posY()));

    m_bounding_rect = QRectF(0.0, -mask_width / 2., plot_scene_rectangle.width(), mask_width);
    update();
}

void HorizontalLineView::paint(QPainter* painter, const QStyleOptionGraphicsItem*, QWidget*)
{
    bool mask_value = static_cast<HorizontalLineItem*>(m_item)->maskValue();
    painter->setBrush(MaskEditorHelper::getMaskBrush(mask_value));
    painter->setPen(MaskEditorHelper::getMaskPen(mask_value));
    painter->drawRect(
        QRectF(0.0, -mask_visible_width / 2., m_bounding_rect.width(), mask_visible_width));

    if (isSelected()) {
        QPen pen;
        pen.setStyle(Qt::DashLine);
        painter->setPen(pen);
        painter->setBrush(Qt::NoBrush);
        painter->drawRect(
            QRectF(0.0, -mask_visible_width / 2., m_bounding_rect.width(), mask_visible_width));
    }
}

//! Allows item movement along y, prevent movement along x
QVariant HorizontalLineView::itemChange(QGraphicsItem::GraphicsItemChange change,
                                        const QVariant& value)
{
    if (isSelected() && change == ItemPositionChange && scene()) {
        QPointF newPos = value.toPointF();
        newPos.setX(x());
        return newPos;
    }
    return QGraphicsItem::itemChange(change, value);
}
