// -*- C++ -*-
//
// MPISampler.cc is a part of Herwig++ - A multi-purpose Monte Carlo event generator
// Copyright (C) 2002-2011 The Herwig Collaboration
//
// Herwig++ is licenced under version 2 of the GPL, see COPYING for details.
// Please respect the MCnet academic guidelines, see GUIDELINES for details.
//
//
// This is the implementation of the non-inlined, non-templated member
// functions of the MPISampler class.
//

#include "MPISampler.h"
#include "ThePEG/Interface/ClassDocumentation.h"
#include "ThePEG/Interface/Parameter.h"
#include "ThePEG/Repository/EventGenerator.h"

#include "ThePEG/Persistency/PersistentOStream.h"
#include "ThePEG/Persistency/PersistentIStream.h"



using namespace Herwig;

MPISampler::~MPISampler() {}

void MPISampler::initialize() {
  theSampler.setRnd(0);
  theSampler.eps(theEps);
  theSampler.margin(theMargin);
  theSampler.nTry(2);
  theSampler.maxTry(theProcessHandler->maxLoop());

  bool nozero = false;
  for ( int i = 0, N = theProcessHandler->nBins(); i < N; ++i )
    if ( theSampler.addFunction(theProcessHandler->nDim(i), theProcessHandler) )
      nozero = true;
  if ( !nozero ) throw EventInitNoXSec()
    << "The UnderlyingEvent handler '" << theProcessHandler->name()
    << "' cannot be initialized because the cross-section for the selected "
    << "sub-processes was zero." << Exception::maybeabort;

  //clear the sampler because this was just a test run (nTry=2)
  theSampler.clear(); 
}

double MPISampler::generate() {
  if ( !theSampler.generate() ) throw EventLoopException()
    << "The maximum number of attempts per event (" << theProcessHandler->maxLoop()
    << ") in multiple interaction handler '" << theProcessHandler->name() << "' was exceeded."
    << Exception::eventerror;
  lastPoint() = theSampler.lastPoint();
  return 1.0;
}

void MPISampler::rejectLast() {
  theSampler.reject();
}

CrossSection MPISampler::integratedXSec() const {
  return theSampler.integral()*nanobarn;
}

CrossSection MPISampler::integratedXSecErr() const {
  return theSampler.integralErr()*nanobarn;
}

CrossSection MPISampler::maxXSec() const {
  return theSampler.maxInt()*nanobarn;
}

int MPISampler::lastBin() const {
  return theSampler.last() - 1;
}

double MPISampler::sumWeights() const {
  return theSampler.n();
}

void MPISampler::dofinish() {
  if ( theSampler.n() <= 0 &&
       theProcessHandler && theProcessHandler->statLevel() > 1 ) {
    generator()->log()
      << "No events generated by the MPIsampler '" << name() << "'" << endl;
  }
  else if ( theProcessHandler && theProcessHandler->statLevel() > 1 )
    generator()->log()
      << "Statistics for the MPI sampler '" << name() << "':" << endl
      << "Number of samplers:" << setw(10) << theSampler.size() << endl
      << "Number of bins:    " << setw(10) << theSampler.nBins() << endl
      << "Depth of bins:     " << setw(10) << theSampler.depth() << endl
      << "efficiency:        " << setw(10) << theSampler.efficiency() << endl;
  if ( theSampler.compensating() )
    generator()->logWarning(
      ACDCStillCompensating()
      << "The run was ended while the MPISampler '" << name()
      << "' was still trying to compensate for weights larger than 1. "
      << "The cross section estimates may therefore be statistically "
      << "inaccurate. At least " << theSampler.compleft() << " additional "
      << "samplings are needed to get out of compensation mode. "
      << "This may be avoided if you increase the value of the "
      << "Ntry parameter determining how many points are presampled before "
      << "the run." << Exception::warning);
  
    
    SamplerBase::dofinish();
}

void MPISampler::doinitrun() {
  SamplerBase::doinitrun();

  theSampler.setRnd(0);
  theSampler.eps(theEps);
  theSampler.margin(theMargin);
  theSampler.nTry(theNTry);
  theSampler.maxTry(theProcessHandler->maxLoop());
  bool nozero = false;

  for ( int i = 0, N = theProcessHandler->nBins(); i < N; ++i )
    if ( theSampler.addFunction(theProcessHandler->nDim(i), theProcessHandler) )
      nozero = true;
  if ( !nozero ) throw EventInitNoXSec()
    << "The UnderlyingEvent handler '" << theProcessHandler->name()
    << "' cannot be initialized because the cross-section for the selected "
    << "sub-processes was zero." << Exception::maybeabort;

}

void MPISampler::persistentOutput(PersistentOStream & os) const {
  os << theEps << theMargin << theNTry << theProcessHandler;
  theSampler.output(os);
}

void MPISampler::persistentInput(PersistentIStream & is, int) {
  is >> theEps >> theMargin >> theNTry >> theProcessHandler;
  theSampler.input(is);
  if ( generator() ) theSampler.setRnd(0);
}

ClassDescription<MPISampler> MPISampler::initMPISampler;
// Definition of the static class description member.

void MPISampler::Init() {

  static ClassDocumentation<MPISampler> documentation
    ("This class inherits from ThePEG::SampleBase and implements "
     "the Auto Compensating Divide-and-Conquer phase space generator, "
     "ACDCGenerator::ACDCGen.");

  /*
   * Object will be created outside of *.in files 
   *

  static Parameter<MPISampler,double> interfaceMargin
    ("Margin",
     "The factor controlling the loss of efficiency when compensating "
     "for a previously underestimated phase space point. If close to one, "
     "the efficiency is increased at the expence of increased number "
     "of cells.",
     &MPISampler::theMargin, 1.1, 1.0, 2.0, true, false, true);

  static Parameter<MPISampler,double> interfaceEps
    ("Epsilon",
     "The smallest possible cell division allowed.",
     &MPISampler::theEps, 100.0*Constants::epsilon, Constants::epsilon,
     1.0e-6, true, false, true);

  static Parameter<MPISampler,int> interfaceNTry
    ("Ntry",
     "The number of phase space points tried in the initialization.",
     &MPISampler::theNTry, 1000, 2, 1000000, true, false, true);

  interfaceNTry.rank(10);
  interfaceEps.rank(9);

  */
}
