/*
 *  setoutdircommand.cpp
 *  Mothur
 *
 *  Created by westcott on 1/21/10.
 *  Copyright 2010 Schloss Lab. All rights reserved.
 *
 */

#include "setdircommand.h"

//**********************************************************************************************************************
vector<string> SetDirectoryCommand::setParameters(){	
	try {
		CommandParameter ptempdefault("tempdefault", "String", "", "", "", "", "","",false,false); parameters.push_back(ptempdefault);
        CommandParameter pdebug("debug", "Boolean", "", "F", "", "", "","",false,false); parameters.push_back(pdebug);
        CommandParameter pmodnames("modifynames", "Boolean", "", "T", "", "", "","",false,false); parameters.push_back(pmodnames);
		CommandParameter pinput("input", "String", "", "", "", "", "","",false,false,true); parameters.push_back(pinput);
		CommandParameter poutput("output", "String", "", "", "", "", "","",false,false,true); parameters.push_back(poutput);
		CommandParameter pinputdir("inputdir", "String", "", "", "", "", "","",false,false); parameters.push_back(pinputdir);
		CommandParameter poutputdir("outputdir", "String", "", "", "", "", "","",false,false); parameters.push_back(poutputdir);
		
		vector<string> myArray;
		for (int i = 0; i < parameters.size(); i++) {	myArray.push_back(parameters[i].name);		}
		return myArray;
	}
	catch(exception& e) {
		m->errorOut(e, "SetDirectoryCommand", "setParameters");
		exit(1);
	}
}
//**********************************************************************************************************************
string SetDirectoryCommand::getHelpString(){	
	try {
		string helpString = "";
		helpString += "The set.dir command can be used to direct the output files generated by mothur to a specific place, the directory must exist.\n";
		helpString += "The set.dir command can also be used to specify the directory where your input files are located, the directory must exist.\n";
		helpString += "The set.dir command can also be used to override or set the default location mothur will look for files if it is unable to find them, the directory must exist.\n";
        helpString += "The set.dir command can also be used to run mothur in debug mode.\n";
        helpString += "The set.dir command can also be used to set the modifynames parameter. Default=t, meaning if your sequence names contain ':' change them to '_' to avoid issues while making trees.  modifynames=F will leave sequence names as they are.\n";
		helpString += "The set.dir command parameters are input, output, tempdefault and debug and one is required.\n";
        helpString += "To run mothur in debug mode set debug=true. Default debug=false.\n";
		helpString += "To return the output to the same directory as the input files you may enter: output=clear.\n";
		helpString += "To return the input to the current working directory you may enter: input=clear.\n";
		helpString += "To set the output to the directory where mothur.exe is located you may enter: output=default.\n";
		helpString += "To set the input to the directory where mothur.exe is located you may enter: input=default.\n";
		helpString += "To return the tempdefault to the default you provided at compile time you may enter: tempdefault=clear.\n";
		helpString += "To set the tempdefault to the directory where mothur.exe is located you may enter: tempdefault=default.\n";
		helpString += "The set.dir command should be in the following format: set.dir(output=yourOutputDirectory, input=yourInputDirectory, tempdefault=yourTempDefault).\n";
		helpString += "Example set.outdir(output=/Users/lab/desktop/outputs, input=/Users/lab/desktop/inputs).\n";
		helpString += "Note: No spaces between parameter labels (i.e. output), '=' and parameters (i.e.yourOutputDirectory).\n";
		return helpString;
	}
	catch(exception& e) {
		m->errorOut(e, "SetDirectoryCommand", "getHelpString");
		exit(1);
	}
}
//**********************************************************************************************************************

SetDirectoryCommand::SetDirectoryCommand(string option)  {
	try {
		abort = false; calledHelp = false;   
		
		//allow user to run help
		if(option == "help") { help(); abort = true; calledHelp = true; }
		else if(option == "citation") { citation(); abort = true; calledHelp = true;}
		
		else {
			vector<string> myArray = setParameters();
			
			OptionParser parser(option);
			map<string, string> parameters = parser.getParameters();
			
			ValidParameters validParameter;
			//check to make sure all parameters are valid for command
			for (map<string,string>::iterator it = parameters.begin(); it != parameters.end(); it++) { 
				if (validParameter.isValidParameter(it->first, myArray, it->second) != true) {  abort = true;  }
			}
		
			output = validParameter.validFile(parameters, "output", false);			
			if (output == "not found") {  output = "";  } 
			
			input = validParameter.validFile(parameters, "input", false);			
			if (input == "not found") {  input = "";  }
			
			tempdefault = validParameter.validFile(parameters, "tempdefault", false);			
			if (tempdefault == "not found") {  tempdefault = "";  }
            
            bool debug = false;
            bool nodebug = false;
            debugOnly = false;
            string temp = validParameter.validFile(parameters, "debug", false);			
			if (temp == "not found") {  debug = false;  nodebug=true; }
            else {  debug = m->isTrue(temp); }
            m->debug = debug;
            
            bool nomod = false;
            temp = validParameter.validFile(parameters, "modifynames", false);
			if (temp == "not found") {  modifyNames = true;  nomod=true; }
            else {  modifyNames = m->isTrue(temp); }
            m->modifyNames = modifyNames;
            
            if (debug) { m->mothurOut("Setting [DEBUG] flag.\n"); }
            
				
			if ((input == "") && (output == "") && (tempdefault == "") && nodebug && nomod) {
				m->mothurOut("You must provide either an input, output, tempdefault, debug or modifynames for the set.outdir command."); m->mothurOutEndLine(); abort = true;
			}else if((input == "") && (output == "") && (tempdefault == "")) { debugOnly = true; }
		}
	}
	catch(exception& e) {
		m->errorOut(e, "SetDirectoryCommand", "SetDirectoryCommand");
		exit(1);
	}
}
//**********************************************************************************************************************

int SetDirectoryCommand::execute(){
	try {
		
		if (abort == true) { if (calledHelp) { return 0; }  return 2;	}
		
        if (debugOnly) { return 0; }
        
		commandFactory = CommandFactory::getInstance();
		
		m->mothurOut("Mothur's directories:"); m->mothurOutEndLine();
		
		//redirect output
		if ((output == "clear") || (output == "")) {  output = "";  commandFactory->setOutputDirectory(output);  }
		else if (output == "default") { 
			string exepath = m->argv;
			output = exepath.substr(0, (exepath.find_last_of('m')));
			
			m->mothurOut("outputDir=" + output); m->mothurOutEndLine();  
			commandFactory->setOutputDirectory(output);
		}else {
            if (m->dirCheck(output)) {
                m->mothurOut("outputDir=" + output); m->mothurOutEndLine();  
				commandFactory->setOutputDirectory(output);
            }
		}
		
		//redirect input
		if ((input == "clear") || (input == "")) {  input = "";  commandFactory->setInputDirectory(input);  }
		else if (input == "default") { 
			string exepath = m->argv;
			input = exepath.substr(0, (exepath.find_last_of('m')));
			
			m->mothurOut("inputDir=" + input); m->mothurOutEndLine();  
			commandFactory->setInputDirectory(input);
		}else {
            if (m->dirCheck(input)) {
                m->mothurOut("inputDir=" + input); m->mothurOutEndLine();  
				commandFactory->setInputDirectory(input); 
            }
        }
		
		//set default
		if (tempdefault == "clear") {  
			#ifdef MOTHUR_FILES
				string temp = MOTHUR_FILES; 
				m->mothurOut("tempDefault=" + temp); m->mothurOutEndLine();  
				m->setDefaultPath(temp);
			#else
				string temp = ""; 
				m->mothurOut("No default directory defined at compile time."); m->mothurOutEndLine();  
				m->setDefaultPath(temp);
			#endif
		}else if (tempdefault == "") {  //do nothing
		}else if (tempdefault == "default") { 
			string exepath = m->argv;
			tempdefault = exepath.substr(0, (exepath.find_last_of('m')));
			
			m->mothurOut("tempDefault=" + tempdefault); m->mothurOutEndLine();  
			m->setDefaultPath(tempdefault);
		}else {
            if (m->dirCheck(tempdefault)) {
                m->mothurOut("tempDefault=" + tempdefault); m->mothurOutEndLine();  
				m->setDefaultPath(tempdefault); 
            }
        }

		return 0;
	}
	catch(exception& e) {
		m->errorOut(e, "SetDirectoryCommand", "execute");
		exit(1);
	}
}
//**********************************************************************************************************************/
