/* -*- Mode: C; c-basic-offset: 2; indent-tabs-mode: nil -*-
 *
 * Software image processing filters
 *
 * Copyright © 2006, 2007, 2008 Fluendo Embedded S.L.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Author: Loïc Molinari <loic@fluendo.com>
 */

/**
 * SECTION:pgmimaging
 * @short_description: A library to process pixbufs through software filters.
 *
 * #PgmImaging is a library providing several software image processing filters.
 * The functions can be used as CPU (Central Processing Unit) software fallback
 * when effects can't be applied by Pigment plugins through the GPU (Graphics
 * Processing Unit).
 *
 * Last reviewed on 2008-07-29 (0.3.7)
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */

#include "pgmimaging.h"
#include <cairo.h>

/* Stride alignment in bytes */
#define ARGB32_STRIDE_FOR_WIDTH(w) (((32 * (w) + 7) / 8 + 3) & ~3)

/* Private functions */

/* Convert a premultiplied ARGB32 buffer to a GdkPixbuf */
static gboolean
pixbuf_from_premultiplied_argb32 (const guint32 *argb32,
                                  guint width,
                                  guint height,
                                  guint stride,
                                  GdkPixbuf **pixbuf)
{
  gint dst_stride, stride_uint, dst_stride_uint;
  GdkPixbuf *dst_pixbuf;
  guint32 *dst;
  guint i, j;

  dst_pixbuf = gdk_pixbuf_new (GDK_COLORSPACE_RGB, TRUE, 8, width, height);
  if (dst_pixbuf == NULL)
    return FALSE;

  dst_stride = gdk_pixbuf_get_rowstride (dst_pixbuf);
  dst = (guint32*) gdk_pixbuf_get_pixels (dst_pixbuf);

  dst_stride_uint = dst_stride / 4;
  stride_uint = stride / 4;

  for (i = 0; i < height; i++)
    {
      for (j = 0; j < width; j++)
        {
          guint32 pixel = argb32[i * stride_uint + j];
          guint8 a, r, g, b;

#if G_BYTE_ORDER == G_LITTLE_ENDIAN
          a = (pixel & 0xff000000) >> 24;
          if (a == 0)
            dst[i * dst_stride_uint + j] = 0;
          else
            {
              r = (((pixel & 0x00ff0000) >> 16) * 255 + a / 2) / a;
              g = (((pixel & 0x0000ff00) >> 8) * 255 + a / 2) / a;
              b = ((pixel & 0x000000ff) * 255 + a / 2) / a;
              dst[i * dst_stride_uint + j] =
                (a << 24) | (b << 16) | (g << 8) | r;
            }
#else
          a = pixel & 0x000000ff;
          if (a == 0)
            dst[i * dst_stride_uint + j] = 0;
          else
            {
              r = (((pixel & 0x0000ff00) >> 8) * 255 + a / 2) / a;
              g = (((pixel & 0x00ff0000) >> 16) * 255 + a / 2) / a;
              b = (((pixel & 0xff000000) >> 24) * 255 + a / 2) / a;
              dst[i * dst_stride_uint + j] =
                a | (b << 8) | (g << 16) | (r << 24);
            }
#endif
        }
    }

  *pixbuf = dst_pixbuf;

  return TRUE;
}

/* Convert a GdkPixbuf to a premultiplied ARGB32 buffer */
static gboolean
premultiplied_argb32_from_pixbuf (const GdkPixbuf *pixbuf,
                                  guint32 **argb32,
                                  guint *width,
                                  guint *height,
                                  guint *stride)
{
  guint src_stride, dst_stride, src_width, src_height;
  guint src_stride_uint, dst_stride_uint;
  guint32 *src, *dst;
  guint i, j;

  src = (guint32*) gdk_pixbuf_get_pixels (pixbuf);
  src_stride = gdk_pixbuf_get_rowstride (pixbuf);
  src_width = gdk_pixbuf_get_width (pixbuf);
  src_height = gdk_pixbuf_get_height (pixbuf);

  dst_stride = ARGB32_STRIDE_FOR_WIDTH (src_width);
  dst = (guint32*) g_try_malloc (dst_stride * src_height);
  if (dst == NULL)
    return FALSE;

  src_stride_uint = src_stride / 4;
  dst_stride_uint = dst_stride / 4;

  if (gdk_pixbuf_get_has_alpha (pixbuf))
    {
      /* from RGBA, need to pre-multiply by alpha */
      for (i = 0; i < src_height; i++)
        {
          for (j = 0; j < src_width; j++)
            {
              guint32 pixel = src[i * src_stride_uint + j];
              guint8 a, b, g, r;

#if G_BYTE_ORDER == G_LITTLE_ENDIAN
              a = (pixel & 0xff000000) >> 24;
              if (a == 0)
                dst[i * dst_stride_uint + j] = 0;
              else
                {
                  b = (((pixel & 0x00ff0000) >> 16) * a) / 255;
                  g = (((pixel & 0x0000ff00) >> 8) * a) / 255;
                  r = ((pixel & 0x000000ff) * a) / 255;
                  dst[i * dst_stride_uint + j] =
                    (a << 24) | (r << 16) | (g << 8) | b;
                }
#else
              a = pixel & 0x000000ff;
              if (a == 0)
                dst[i * dst_stride_uint + j] = 0;
              else
                {
                  b = (((pixel & 0x0000ff00) >> 8) * a) / 255;
                  g = (((pixel & 0x00ff0000) >> 16) * a) / 255;
                  r = (((pixel & 0xff000000) >> 24) * a) / 255;
                  dst[i * dst_stride_uint + j] =
                    a | (r << 8) | (g << 16) | (b << 24);
                }
#endif
            }
        }
    }
  else
    {
      /* From RGB, no need to pre-multiply, just set A to 255 */
      for (i = 0; i < src_height; i++)
        {
          guchar *s = ((guchar*) src) + i * src_stride;

          for (j = 0; j < src_width; j++)
            {
              guint32 pixel;

#if G_BYTE_ORDER == G_LITTLE_ENDIAN
              pixel = 0xff000000;
              pixel |= *s++ << 16;
              pixel |= *s++ << 8;
              pixel |= *s++;
#else
              pixel = 0x000000ff;
              pixel |= *s++ << 8;
              pixel |= *s++ << 16;
              pixel |= *s++ << 24;
#endif

              dst[i * dst_stride_uint + j] = pixel;
            }
        }
    }

  *argb32 = dst;
  *stride = dst_stride;
  *width = src_width;
  *height = src_height;

  return TRUE;
}

/* Public functions */

/**
 * pgm_imaging_linear_alpha_gradient:
 * @pixbuf: A #GdkPixbuf.
 * @start_x: x coordinate of the start point, in the range [0.0, 1.0].
 * @start_y: y coordinate of the start point, in the range [0.0, 1.0].
 * @start_alpha: Alpha component at the start point, in the range [0.0, 1.0].
 * @end_x: x coordinate of the end point, in the range [0.0, 1.0].
 * @end_y: y coordinate of the end point, in the range [0.0, 1.0].
 * @end_alpha: Alpha component at the end point, in the range [0.0, 1.0].
 *
 * Creates a new #GdkPixbuf by compositing the alpha channel of @pixbuf with
 * the linear alpha gradient mask given as parameters. Note that the resulting
 * pixbuf always contains an alpha channel even if the original @pixbuf does
 * not. In that case, the alpha channel is automatically added with a default
 * value of 1.0 corresponding to opaque pixels.
 *
 * Returns: A newly-created #GdkPixbuf with a reference count of 1, or %NULL if
 * not enough memory could be allocated.
 */
GdkPixbuf*
pgm_imaging_linear_alpha_gradient (const GdkPixbuf *pixbuf,
                                   gfloat start_x,
                                   gfloat start_y,
                                   gfloat start_alpha,
                                   gfloat end_x,
                                   gfloat end_y,
                                   gfloat end_alpha)
{
  GdkPixbuf *processed_pixbuf = NULL;
  cairo_surface_t *image_surface, *dest_surface;
  guint32 *image_buffer, *dest_buffer;
  guint width, height, stride;
  cairo_pattern_t *gradient;
  cairo_t *cr;
  gint ret;

  g_return_val_if_fail (pixbuf != NULL, NULL);

  /* GdkPixbuf only supports RGB color space at the time this is written and the
   * function assumes the given pixbufs are in that color space, let's prevent
   * addition of other spaces in GdkPixbuf. */
  g_return_val_if_fail (gdk_pixbuf_get_colorspace (pixbuf)
                        == GDK_COLORSPACE_RGB, NULL);

  /* Create the image surface from the content of the given pixbuf */
  ret = premultiplied_argb32_from_pixbuf (pixbuf, &image_buffer,
                                          &width, &height, &stride);
  if (!ret)
    goto image_creation_error;
  image_surface = cairo_image_surface_create_for_data ((guchar*) image_buffer,
                                                       CAIRO_FORMAT_ARGB32,
                                                       width, height, stride);

  /* Create the destination surface */
  dest_buffer = g_try_malloc0 (stride * height);
  if (!dest_buffer)
    goto dest_creation_error;
  dest_surface = cairo_image_surface_create_for_data ((guchar*) dest_buffer,
                                                      CAIRO_FORMAT_ARGB32,
                                                      width, height, stride);

  /* Define the linear gradient */
  gradient = cairo_pattern_create_linear ((gdouble) start_x, (gdouble) start_y,
                                          (gdouble) end_x, (gdouble) end_y);
  cairo_pattern_add_color_stop_rgba (gradient, 0.0, 0.0, 0.0, 0.0,
                                     (gdouble) start_alpha);
  cairo_pattern_add_color_stop_rgba (gradient, 1.0, 0.0, 0.0, 0.0,
                                     (gdouble) end_alpha);

  /* Composite the final image */
  cr = cairo_create (dest_surface);
  cairo_set_source_surface (cr, image_surface, 0.0, 0.0);
  cairo_scale (cr, width, height);
  cairo_mask (cr, gradient);
  cairo_surface_flush (dest_surface);

  /* Build the new processed pixbuf */
  pixbuf_from_premultiplied_argb32 (dest_buffer, width, height, stride,
                                    &processed_pixbuf);

  /* Clean up */
  cairo_destroy (cr);
  cairo_pattern_destroy (gradient);
  cairo_surface_destroy (dest_surface);
 dest_creation_error:
  cairo_surface_destroy (image_surface);
 image_creation_error:

  return processed_pixbuf;
}
