use core:lang;
use lang:bs;
use lang:bs:macro;

/**
 * A block corresponding to one layout entry.
 */
class LayoutRoot on Compiler {
	// Root block.
	protected ExprBlock root;

	// Create.
	init(ExprBlock root) {
		init() {
			root = root;
		}
	}

	// Get the block.
	ExprBlock block() {
		root;
	}

	// Add a block.
	void add(LayoutRoot sub) {
		root.add(sub.root);
	}

	// To string.
	void toS(StrBuf to) {
		root.toS(to);
	}
}

// Create a new root block.
LayoutRoot createRoot(Block parent) {
	LayoutRoot(ExprBlock(SrcPos(), parent));
}

// Sane default for maybe actuals.
// Idea: provide syntax: "foo or Actuals()" equivalent to this?
private Actuals sane(Actuals? params) {
	if (params)
		params;
	else
		Actuals();
}

// Apply a property to the object denoted in Expr. Attempts to call a function first, and then
// attempts an assignment if possible.
private Expr? applyProperty(Block block, Expr expr, SStr name, Actuals params) {
	Expr e = namedExpr(block, name, expr, params);
	// Did we find it already?
	unless (e as UnresolvedName) {
		return e;
	}

	// If only one parameter, attempt assignment.
	if (params.expressions.count != 1)
		return null;

	Expr l = namedExpr(block, name, expr, Actuals());
	if (l as UnresolvedName) {
		return null;
	}

	AssignOpInfo assign(SStr("="), 10, true);
	return assign.meaning(block, l, params.expressions[0]);
}

/**
 * A block that is an inner node in the layout tree.
 */
class LayoutBlock extends LayoutRoot {
	// Block used to put data from child items in so that we are still able to get something
	// positioned 'last' in the parent block.
	private ExprBlock sub;

	// Variable storing the resulting component.
	private Expr result;

	// If set, stores the created instance before it was passed to the 'component' function so that
	// we may access properties in this object as well!
	private Expr? preWrap;

	// Properties that should be applied in the parent later on.
	private value ParentExpr {
		SStr name;
		Actuals params;

		init(SStr name, Actuals params) {
			init() { name = name; params = params; }
		}
	}

	// Properties to be applied in the parent.
	private ParentExpr[] parent;

	// Create.
	init(LayoutRoot parent, SrcName name, Actuals? params) {
		ExprBlock block(name.pos, parent.root);
		ExprBlock sub(name.pos, block);

		// TODO: Allow 'this' as a first parameter to constructors?
		Expr src = namedExpr(block, name, sane(params));
		Expr? preWrap;
		if (!Value(named{Component}).canStore(src.result.type)) {
			// Store 'src' in a variable, before calling 'component', so that we may access it later!
			Var raw(block, SStr("@ raw"), src);
			block.add(raw);
			LocalVarAccess rawAcc(name.pos, raw.var);
			preWrap = rawAcc;

			src = namedExpr(block, name.pos, "component", rawAcc);
		}

		Var result(block, SStr("@ result"), src);
		block.add(result);
		block.add(sub);

		LocalVarAccess resultAcc(name.pos, result.var);
		block.add(resultAcc);

		init(block) {
			sub = sub;
			result = resultAcc;
			preWrap = preWrap;
		}
	}

	// More basic creation for subclasses.
	init(ExprBlock root, ExprBlock sub, Expr result) {
		init(root) {
			sub = sub;
			result = result;
		}
	}

	init(ExprBlock root, ExprBlock sub, Expr result, Expr? preWrap) {
		init(root) {
			sub = sub;
			result = result;
			preWrap = preWrap;
		}
	}


	// Add a property.
	void add(SStr name, Actuals? params) {
		// Look in this class.
		if (expr = applyProperty(sub, result, name, sane(params))) {
			sub.add(expr);
			return;
		}

		// Failed... See if we want to look inside the raw element.
		if (preWrap) {
			if (expr = applyProperty(sub, preWrap, name, sane(params))) {
				sub.add(expr);
				return;
			}
		}

		// Nothing here. See if the parent has something useful to do.
		parent << ParentExpr(name, sane(params));
	}

	// Add a child block.
	void add(LayoutBlock e) {
		Expr toAdd = namedExpr(sub, sub.pos, "toAdd", e.root, Actuals());
		Expr add = namedExpr(sub, sub.pos, "add", result, Actuals(toAdd));
		if (e.parent.empty) {
			sub.add(add);
			return;
		}

		// Add properties from the child.
		ExprBlock child(sub.pos, sub);
		sub.add(child);

		Var info(child, SStr("@ info"), add);
		child.add(info);
		LocalVarAccess access(sub.pos, info.var);

		for (x in e.parent) {
			if (expr = applyProperty(child, access, x.name, x.params)) {
				child.add(expr);
			} else {
				throw SyntaxError(x.name.pos, "Unable to find a suitable property: ${x.name.v}");
			}
		}
	}

	// Use the sub block if required anywhere.
	ExprBlock block() {
		sub;
	}

	// Output a string representation.
	void toS(StrBuf to) {
		root.toS(to);
	}
}
